/**
 * Copyright (c) 2016 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.server.rest;

import java.net.URI;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.google.gson.annotations.SerializedName;

final class StatusReponse {

    @SerializedName("situation")
    private ProblemSituation situation;

    /**
     * Inline-level HTML
     */
    @SerializedName("message")
    private String message;

    @SerializedName("links")
    private List<Link> links;

    @SerializedName("auxiliaryInformationRequests")
    private Set<String> auxiliaryInformationRequests;

    public ProblemSituation getSituation() {
        return situation;
    }

    public String getMessage() {
        return message;
    }

    public List<Link> getLinks() {
        return links;
    }

    public Set<String> getAuxiliaryInformationRequests() {
        return auxiliaryInformationRequests;
    }
}

final class Link {

    @SerializedName("href")
    private URI href;

    @SerializedName("rel")
    private String rel;

    @SerializedName("title")
    private String title;

    public URI getHref() {
        return href;
    }

    public String getRel() {
        return rel;
    }

    public String getTitle() {
        return title;
    }

    @Override
    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (other == null) {
            return false;
        }
        if (getClass() != other.getClass()) {
            return false;
        }
        Link that = (Link) other;
        return Objects.equals(this.href, that.href) && Objects.equals(this.rel, that.rel) && Objects.equals(this.title, that.title);
    }

    @Override
    public int hashCode() {
        return Objects.hash(href, rel, title);
    }
}

enum ProblemSituation {

    /**
     * The problem with the software has not been resolved yet; it neither was {@linkplain #FIXED fixed} already nor {@linkplain #WONTFIX
     * won't it be fixed}.
     *
     * This situation carries <strong>no</strong> information about the progress made towards resolving the problem; all this situation
     * signals is that the problem still awaits resolution. In particular, an open problem may or may not be already known to the
     * developers.
     */
    OPEN("Open"),

    /**
     * The problem with the software has been fixed by the developers.
     */
    FIXED("Fixed"),

    /**
     * The problem with the software won't be fixed by the developers, e.g., because the developers don't consider their software to be at
     * fault or because the developers only consider it to be a very minor issue.
     */
    WONTFIX("Won't Fix"),

    /**
     * The problem is not consider to be a real issue with the software itself by the developers, e.g., because it is merely a log message
     * meant to alert the user to a configuration issue.
     */
    IGNORE("Ignored"),

    FAILURE("Failure");

    private final String label;

    ProblemSituation(String label) {
        this.label = label;
    }

    @Override
    public String toString() {
        return label;
    }
}
