(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define(["require", "exports", "vscode-languageserver-protocol/lib/protocol.sematicTokens.proposed", "dockerfile-ast", "vscode-languageserver-types", "./docker"], factory);
    }
})(function (require, exports) {
    /* --------------------------------------------------------------------------------------------
     * Copyright (c) Remy Suen. All rights reserved.
     * Licensed under the MIT License. See License.txt in the project root for license information.
     * ------------------------------------------------------------------------------------------ */
    'use strict';
    Object.defineProperty(exports, "__esModule", { value: true });
    var protocol_sematicTokens_proposed_1 = require("vscode-languageserver-protocol/lib/protocol.sematicTokens.proposed");
    var dockerfile_ast_1 = require("dockerfile-ast");
    var vscode_languageserver_types_1 = require("vscode-languageserver-types");
    var docker_1 = require("./docker");
    var TokensLegend = /** @class */ (function () {
        function TokensLegend() {
        }
        TokensLegend.init = function () {
            this.tokenTypes[protocol_sematicTokens_proposed_1.SemanticTokenTypes.keyword] = 0;
            this.tokenTypes[protocol_sematicTokens_proposed_1.SemanticTokenTypes.comment] = 1;
            this.tokenTypes[protocol_sematicTokens_proposed_1.SemanticTokenTypes.parameter] = 2;
            this.tokenTypes[protocol_sematicTokens_proposed_1.SemanticTokenTypes.property] = 3;
            this.tokenTypes[protocol_sematicTokens_proposed_1.SemanticTokenTypes.label] = 4;
            this.tokenTypes[protocol_sematicTokens_proposed_1.SemanticTokenTypes.class] = 5;
            this.tokenTypes[protocol_sematicTokens_proposed_1.SemanticTokenTypes.macro] = 6;
            this.tokenTypes[protocol_sematicTokens_proposed_1.SemanticTokenTypes.string] = 7;
            this.tokenTypes[protocol_sematicTokens_proposed_1.SemanticTokenTypes.variable] = 8;
            this.tokenTypes[protocol_sematicTokens_proposed_1.SemanticTokenTypes.operator] = 9;
            this.tokenModifiers[protocol_sematicTokens_proposed_1.SemanticTokenModifiers.declaration] = 1;
            this.tokenModifiers[protocol_sematicTokens_proposed_1.SemanticTokenModifiers.definition] = 2;
            this.tokenModifiers[protocol_sematicTokens_proposed_1.SemanticTokenModifiers.deprecated] = 4;
            this.tokenModifiers[protocol_sematicTokens_proposed_1.SemanticTokenModifiers.reference] = 8;
        };
        TokensLegend.getTokenType = function (type) {
            var tokenType = this.tokenTypes[type];
            return tokenType;
        };
        TokensLegend.getTokenModifiers = function (modifiers) {
            var bit = 0;
            for (var _i = 0, modifiers_1 = modifiers; _i < modifiers_1.length; _i++) {
                var modifier = modifiers_1[_i];
                bit |= this.tokenModifiers[modifier];
            }
            return bit;
        };
        TokensLegend.tokenTypes = {};
        TokensLegend.tokenModifiers = {};
        return TokensLegend;
    }());
    exports.TokensLegend = TokensLegend;
    TokensLegend.init();
    var DockerSemanticTokens = /** @class */ (function () {
        function DockerSemanticTokens(content) {
            this.currentRange = null;
            this.tokens = [];
            this.quote = null;
            this.escapedQuote = null;
            this.content = content;
            this.document = vscode_languageserver_types_1.TextDocument.create("", "", 0, content);
            this.dockerfile = dockerfile_ast_1.DockerfileParser.parse(content);
            this.escapeCharacter = this.dockerfile.getEscapeCharacter();
        }
        DockerSemanticTokens.prototype.computeSemanticTokens = function () {
            var lines = this.dockerfile.getComments();
            var instructions = this.dockerfile.getInstructions();
            for (var _i = 0, instructions_1 = instructions; _i < instructions_1.length; _i++) {
                var instruction = instructions_1[_i];
                var range = instruction.getRange();
                if (range.start.line !== range.end.line) {
                    for (var i = 0; i < lines.length; i++) {
                        var commentRange = lines[i].getRange();
                        if (range.start.line < commentRange.start.line && commentRange.start.line < range.end.line) {
                            // this is an embedded comment, remove it
                            lines.splice(i, 1);
                            i--;
                        }
                    }
                }
            }
            lines = lines.concat(this.dockerfile.getInstructions());
            lines.sort(function (a, b) {
                return a.getRange().start.line - b.getRange().start.line;
            });
            for (var _a = 0, _b = this.dockerfile.getDirectives(); _a < _b.length; _a++) {
                var directive = _b[_a];
                var range = directive.getRange();
                var nameRange = directive.getNameRange();
                var prefixRange = { start: range.start, end: nameRange.start };
                this.createToken(null, prefixRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.comment, [], false);
                this.createToken(null, nameRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.property, [], false);
                var valueRange = directive.getValueRange();
                var operatorRange = {
                    start: { character: valueRange.start.character - 1, line: valueRange.start.line },
                    end: { character: valueRange.start.character, line: valueRange.start.line },
                };
                this.createToken(null, operatorRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.operator, [], false);
                if (valueRange.start.character !== valueRange.end.character) {
                    this.createToken(null, valueRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.parameter, [], false);
                }
            }
            for (var i = 0; i < lines.length; i++) {
                if (lines[i] instanceof dockerfile_ast_1.Comment) {
                    var range = lines[i].getRange();
                    this.createToken(null, range, protocol_sematicTokens_proposed_1.SemanticTokenTypes.comment, [], false);
                }
                else {
                    // trailing open quotes should not cause subsequent argument parameters to be flagged as strings
                    this.quote = null;
                    this.escapedQuote = null;
                    this.createTokensForInstruction(lines[i]);
                }
            }
            return {
                data: this.tokens
            };
        };
        DockerSemanticTokens.prototype.createTokensForInstruction = function (instruction) {
            var instructionRange = instruction.getInstructionRange();
            var modifiers = [];
            if (instruction.getKeyword() === dockerfile_ast_1.Keyword.MAINTAINER) {
                modifiers = [protocol_sematicTokens_proposed_1.SemanticTokenModifiers.deprecated];
            }
            this.createToken(instruction, instructionRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.keyword, modifiers);
            if (instruction instanceof dockerfile_ast_1.ModifiableInstruction) {
                for (var _i = 0, _a = instruction.getFlags(); _i < _a.length; _i++) {
                    var flag = _a[_i];
                    var flagRange = flag.getRange();
                    var nameRange = flag.getNameRange();
                    var mergedRange = {
                        start: flagRange.start,
                        end: nameRange.end
                    };
                    this.createToken(instruction, mergedRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.parameter);
                    var flagValue = flag.getValue();
                    if (flagValue !== null) {
                        if (flag.hasOptions()) {
                            var operatorRange = {
                                start: mergedRange.end,
                                end: {
                                    line: mergedRange.end.line,
                                    character: mergedRange.end.character + 1
                                }
                            };
                            this.createToken(instruction, operatorRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.operator, [], false, false);
                            for (var _b = 0, _c = flag.getOptions(); _b < _c.length; _b++) {
                                var option = _c[_b];
                                nameRange = option.getNameRange();
                                this.createToken(instruction, nameRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.parameter);
                                var valueRange = option.getValueRange();
                                if (valueRange !== null) {
                                    var operatorRange_1 = {
                                        start: nameRange.end,
                                        end: valueRange.start
                                    };
                                    this.createToken(instruction, operatorRange_1, protocol_sematicTokens_proposed_1.SemanticTokenTypes.operator, [], false, false);
                                    if (option.getValue() !== "") {
                                        this.createToken(instruction, valueRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.property);
                                    }
                                }
                            }
                        }
                        else {
                            var valueRange = flag.getValueRange();
                            var operatorRange = {
                                start: mergedRange.end,
                                end: valueRange.start
                            };
                            this.createToken(instruction, operatorRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.operator, [], false, false);
                            if (flagValue !== "") {
                                this.createToken(instruction, valueRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.property);
                            }
                        }
                    }
                }
            }
            switch (instruction.getKeyword()) {
                case dockerfile_ast_1.Keyword.ARG:
                case dockerfile_ast_1.Keyword.ENV:
                    var propertyInstruction = instruction;
                    for (var _d = 0, _e = propertyInstruction.getProperties(); _d < _e.length; _d++) {
                        var property = _e[_d];
                        var nameRange_1 = property.getNameRange();
                        this.createToken(instruction, nameRange_1, protocol_sematicTokens_proposed_1.SemanticTokenTypes.variable, [protocol_sematicTokens_proposed_1.SemanticTokenModifiers.declaration], false);
                        var valueRange = property.getValueRange();
                        if (valueRange !== null) {
                            var operatorRange = {
                                start: nameRange_1.end,
                                end: valueRange.start
                            };
                            this.createToken(instruction, operatorRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.operator, [], false, false);
                            this.createToken(instruction, valueRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.parameter, [], true, true);
                        }
                    }
                    return;
                case dockerfile_ast_1.Keyword.FROM:
                    var from = instruction;
                    var nameRange = from.getImageNameRange();
                    if (nameRange !== null) {
                        this.createToken(instruction, nameRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.class);
                    }
                    var tagRange = from.getImageTagRange();
                    if (tagRange !== null) {
                        this.createToken(instruction, tagRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.label);
                    }
                    var digestRange = from.getImageDigestRange();
                    if (digestRange !== null) {
                        this.createToken(instruction, digestRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.label);
                    }
                    var fromArgs = instruction.getArguments();
                    if (fromArgs.length > 1) {
                        if (fromArgs[1].getValue().toUpperCase() === "AS") {
                            var range_1 = fromArgs[1].getRange();
                            this.createToken(instruction, range_1, protocol_sematicTokens_proposed_1.SemanticTokenTypes.keyword);
                            if (fromArgs.length > 2) {
                                this.createToken(instruction, fromArgs[2].getRange(), protocol_sematicTokens_proposed_1.SemanticTokenTypes.label);
                                if (fromArgs.length > 3) {
                                    this.createArgumentTokens(instruction, fromArgs.slice(3));
                                }
                            }
                        }
                        else {
                            this.createArgumentTokens(instruction, fromArgs.slice(1));
                        }
                    }
                    return;
                case dockerfile_ast_1.Keyword.HEALTHCHECK:
                    var healthcheck = instruction;
                    var subcommand = healthcheck.getSubcommand();
                    if (subcommand !== null) {
                        var range_2 = subcommand.getRange();
                        this.createToken(instruction, range_2, protocol_sematicTokens_proposed_1.SemanticTokenTypes.keyword);
                        var args = instruction.getArguments();
                        if (args.length > 1) {
                            this.createArgumentTokens(instruction, args.slice(1));
                        }
                    }
                    return;
                case dockerfile_ast_1.Keyword.ONBUILD:
                    var onbuild = instruction;
                    var range = onbuild.getTriggerRange();
                    if (range !== null) {
                        this.createTokensForInstruction(onbuild.getTriggerInstruction());
                    }
                    return;
            }
            this.createArgumentTokens(instruction, instruction.getArguments());
        };
        DockerSemanticTokens.prototype.createArgumentTokens = function (instruction, args) {
            for (var i = 0; i < args.length; i++) {
                this.createToken(instruction, args[i].getRange(), protocol_sematicTokens_proposed_1.SemanticTokenTypes.parameter, [], true, true);
            }
        };
        DockerSemanticTokens.prototype.handleLineChange = function (instruction, next, previous) {
            var comment = -1;
            for (var i = this.document.offsetAt(previous); i < this.document.offsetAt(next); i++) {
                switch (this.content.charAt(i)) {
                    case this.escapeCharacter:
                        // mark the escape character if it's not in a comment 
                        if (comment === -1) {
                            this.createEscapeToken(instruction, i);
                        }
                        break;
                    case '\r':
                    case '\n':
                        if (comment !== -1) {
                            var commentRange = {
                                start: this.document.positionAt(comment),
                                end: this.document.positionAt(i)
                            };
                            this.createToken(null, commentRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.comment, [], false);
                            comment = -1;
                        }
                        break;
                    case '#':
                        if (comment === -1) {
                            comment = i;
                        }
                        break;
                }
            }
        };
        DockerSemanticTokens.prototype.createEscapeToken = function (instruction, offset) {
            var escapeRange = {
                start: this.document.positionAt(offset),
                end: this.document.positionAt(offset + 1),
            };
            this.createToken(instruction, escapeRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.macro, [], false, false, false);
        };
        DockerSemanticTokens.prototype.createToken = function (instruction, range, tokenType, tokenModifiers, checkVariables, checkStrings, checkNewline) {
            if (tokenModifiers === void 0) { tokenModifiers = []; }
            if (checkVariables === void 0) { checkVariables = true; }
            if (checkStrings === void 0) { checkStrings = false; }
            if (checkNewline === void 0) { checkNewline = true; }
            if (checkNewline && this.currentRange !== null && this.currentRange.end.line !== range.start.line) {
                // this implies that there's been a line change between one arg and the next
                this.handleLineChange(instruction, range.start, this.currentRange.end);
            }
            if (checkStrings) {
                var startOffset = this.document.offsetAt(range.start);
                var quoteStart = startOffset;
                var newOffset = -1;
                var endOffset = this.document.offsetAt(range.end);
                for (var i = startOffset; i < endOffset; i++) {
                    var ch = this.content.charAt(i);
                    switch (ch) {
                        case '\\':
                            var next = this.content.charAt(i + 1);
                            if (next === '\'' || next === '"') {
                                if (this.quote === null) {
                                    if (this.escapedQuote === null) {
                                        quoteStart = i;
                                        this.escapedQuote = next;
                                        if (startOffset !== quoteStart) {
                                            var intermediateRange = {
                                                start: this.document.positionAt(startOffset),
                                                end: this.document.positionAt(quoteStart),
                                            };
                                            this.createToken(instruction, intermediateRange, tokenType, tokenModifiers);
                                        }
                                    }
                                    else {
                                        var quoteRange = {
                                            start: this.document.positionAt(quoteStart),
                                            end: this.document.positionAt(i + 2),
                                        };
                                        this.createToken(instruction, quoteRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.string, [], true, false);
                                        newOffset = i + 2;
                                        this.escapedQuote = null;
                                    }
                                }
                                i++;
                            }
                            break;
                        case '\'':
                        case '"':
                            if (this.quote === null) {
                                if (this.escapedQuote === null) {
                                    this.quote = ch;
                                    quoteStart = i;
                                    if (startOffset !== quoteStart) {
                                        var intermediateRange = {
                                            start: this.document.positionAt(startOffset),
                                            end: this.document.positionAt(quoteStart),
                                        };
                                        this.createToken(instruction, intermediateRange, tokenType, tokenModifiers);
                                    }
                                }
                            }
                            else if (this.quote === ch) {
                                var quoteRange = {
                                    start: this.document.positionAt(quoteStart),
                                    end: this.document.positionAt(i + 1),
                                };
                                this.createToken(instruction, quoteRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.string, [], true, false);
                                newOffset = i + 1;
                                this.quote = null;
                            }
                            break;
                    }
                }
                if (this.quote !== null) {
                    var quoteRange = {
                        start: this.document.positionAt(quoteStart),
                        end: this.document.positionAt(endOffset),
                    };
                    this.createToken(instruction, quoteRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.string, [], true, false);
                    return;
                }
                else if (newOffset !== -1) {
                    if (newOffset !== endOffset) {
                        var intermediateRange = {
                            start: this.document.positionAt(newOffset),
                            end: this.document.positionAt(endOffset),
                        };
                        this.createToken(instruction, intermediateRange, tokenType, tokenModifiers);
                    }
                    return;
                }
                else if (this.quote !== null || this.escapedQuote !== null) {
                    // there is now an open string, change the token to a string
                    tokenType = protocol_sematicTokens_proposed_1.SemanticTokenTypes.string;
                    // reset the range to the start of the string
                    range = {
                        start: this.document.positionAt(quoteStart),
                        end: range.end
                    };
                }
            }
            if (range.start.line !== range.end.line) {
                var startOffset = this.document.offsetAt(range.start);
                var endOffset = this.document.offsetAt(range.end);
                var intermediateAdded = false;
                var handleNewlines = true;
                var escaping = false;
                for (var i = startOffset; i < endOffset; i++) {
                    var ch = this.content.charAt(i);
                    switch (ch) {
                        case '#':
                            if (escaping) {
                                commentCheck: for (var j = i + 1; j < endOffset; j++) {
                                    switch (this.content.charAt(j)) {
                                        case ' ':
                                        case '\t':
                                            break;
                                        case '\r':
                                            var crComment = {
                                                start: this.document.positionAt(i),
                                                end: this.document.positionAt(j)
                                            };
                                            this.createToken(null, crComment, protocol_sematicTokens_proposed_1.SemanticTokenTypes.comment, [], false);
                                            i = j + 1;
                                            startOffset = -1;
                                            break commentCheck;
                                        case '\n':
                                            var lfComment = {
                                                start: this.document.positionAt(i),
                                                end: this.document.positionAt(j)
                                            };
                                            this.createToken(null, lfComment, protocol_sematicTokens_proposed_1.SemanticTokenTypes.comment, [], false);
                                            i = j;
                                            startOffset = -1;
                                            break commentCheck;
                                    }
                                }
                            }
                            break;
                        case this.escapeCharacter:
                            // note whether the intermediate token has been added or not
                            var added = false;
                            escapeCheck: for (var j = i + 1; j < endOffset; j++) {
                                switch (this.content.charAt(j)) {
                                    case ' ':
                                    case '\t':
                                    case '\r':
                                        break;
                                    case '\n':
                                        if (!added) {
                                            if (!intermediateAdded) {
                                                var intermediateRange_1 = {
                                                    start: this.document.positionAt(startOffset),
                                                    end: this.document.positionAt(i),
                                                };
                                                this.createToken(instruction, intermediateRange_1, tokenType, tokenModifiers);
                                                intermediateAdded = true;
                                            }
                                            this.createEscapeToken(instruction, i);
                                        }
                                        // escaped newlines have are being handled here already
                                        handleNewlines = false;
                                        escaping = true;
                                        added = true;
                                        i = j;
                                        startOffset = -1;
                                        break;
                                    default:
                                        if (startOffset === -1) {
                                            startOffset = j;
                                        }
                                        break escapeCheck;
                                }
                            }
                            break;
                        default:
                            if (startOffset === -1) {
                                intermediateAdded = false;
                                escaping = false;
                                startOffset = i;
                            }
                            break;
                    }
                }
                var intermediateRange = {
                    start: this.document.positionAt(startOffset),
                    end: this.document.positionAt(endOffset),
                };
                this.createToken(instruction, intermediateRange, tokenType, tokenModifiers, checkVariables, checkStrings, handleNewlines);
                return;
            }
            if (checkVariables) {
                var startPosition = range.start;
                var lastVariableRange = null;
                for (var _i = 0, _a = instruction.getVariables(); _i < _a.length; _i++) {
                    var variable = _a[_i];
                    var variableRange = variable.getRange();
                    if (docker_1.Util.isInsideRange(variableRange.start, range)) {
                        if (docker_1.Util.positionBefore(startPosition, variableRange.start)) {
                            // create a parameter token for the characters
                            // before the variable
                            this.createToken(instruction, {
                                start: startPosition,
                                end: variableRange.start
                            }, tokenType, tokenModifiers, false);
                        }
                        this.createToken(instruction, variableRange, protocol_sematicTokens_proposed_1.SemanticTokenTypes.variable, [protocol_sematicTokens_proposed_1.SemanticTokenModifiers.reference], false);
                        lastVariableRange = variableRange;
                        if (docker_1.Util.positionEquals(range.end, variableRange.end)) {
                            return;
                        }
                        startPosition = variableRange.end;
                    }
                }
                if (lastVariableRange !== null) {
                    // alter the range so it is the characters that comes
                    // after the last matched variable
                    range = { start: lastVariableRange.end, end: range.end };
                }
            }
            if (this.currentRange === null) {
                this.tokens = this.tokens.concat([
                    range.start.line,
                    range.start.character,
                    range.end.character - range.start.character,
                    TokensLegend.getTokenType(tokenType),
                    TokensLegend.getTokenModifiers(tokenModifiers)
                ]);
            }
            else if (this.currentRange.end.line !== range.start.line) {
                this.tokens = this.tokens.concat([
                    range.start.line - this.currentRange.end.line,
                    range.start.character,
                    range.end.character - range.start.character,
                    TokensLegend.getTokenType(tokenType),
                    TokensLegend.getTokenModifiers(tokenModifiers)
                ]);
            }
            else {
                this.tokens = this.tokens.concat([
                    range.start.line - this.currentRange.start.line,
                    range.start.character - this.currentRange.start.character,
                    range.end.character - range.start.character,
                    TokensLegend.getTokenType(tokenType),
                    TokensLegend.getTokenModifiers(tokenModifiers)
                ]);
            }
            this.currentRange = range;
        };
        return DockerSemanticTokens;
    }());
    exports.DockerSemanticTokens = DockerSemanticTokens;
});
