"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const YAML = require("../src");
const failsafe = require("../src/schema/failsafe");
const fs = require("fs");
const path = require("path");
const json = require("../src/schema/json");
const core = require("../src/schema/core");
suite('YAML Tag Tests', () => {
    test('Failsafe YAML schema should validate all properties successfully', function () {
        const failSafeSchema = fs.readFileSync(path.join('test', 'testFixtures', 'failsafe_schema.yaml'));
        const document = YAML.load(failSafeSchema.toString(), {
            schema: failsafe
        });
        chai_1.assert.equal(document.errors.length, 0);
        chai_1.assert.equal(document.mappings[0].key.value, "Block-map");
        chai_1.assert.equal(document.mappings[0].value.mappings.length, 3);
        chai_1.assert.equal(document.mappings[0].value.mappings[0].key.value, "Clark");
        chai_1.assert.equal(document.mappings[0].value.mappings[0].value.value, "Evans");
        chai_1.assert.equal(document.mappings[0].value.mappings[1].key.value, "Ingy");
        chai_1.assert.equal(document.mappings[0].value.mappings[1].value.value, "döt Net");
        chai_1.assert.equal(document.mappings[0].value.mappings[2].key.value, "Oren");
        chai_1.assert.equal(document.mappings[0].value.mappings[2].value.value, "Ben-Kiki");
        chai_1.assert.equal(document.mappings[1].key.value, "Flow-map");
        chai_1.assert.equal(document.mappings[1].value.mappings.length, 3);
        chai_1.assert.equal(document.mappings[1].value.mappings[0].key.value, "Clark");
        chai_1.assert.equal(document.mappings[1].value.mappings[0].value.value, "Evans");
        chai_1.assert.equal(document.mappings[1].value.mappings[1].key.value, "Ingy");
        chai_1.assert.equal(document.mappings[1].value.mappings[1].value.value, "döt Net");
        chai_1.assert.equal(document.mappings[1].value.mappings[2].key.value, "Oren");
        chai_1.assert.equal(document.mappings[1].value.mappings[2].value.value, "Ben-Kiki");
        chai_1.assert.equal(document.mappings[2].key.value, "Block-seq");
        chai_1.assert.equal(document.mappings[2].value.items.length, 3);
        chai_1.assert.equal(document.mappings[2].value.items[0].value, "Clark Evans");
        chai_1.assert.equal(document.mappings[2].value.items[1].value, "Ingy döt Net");
        chai_1.assert.equal(document.mappings[2].value.items[2].value, "Oren Ben-Kiki");
        chai_1.assert.equal(document.mappings[3].key.value, "Flow-seq");
        chai_1.assert.equal(document.mappings[3].value.items.length, 3);
        chai_1.assert.equal(document.mappings[3].value.items[0].value, "Clark Evans");
        chai_1.assert.equal(document.mappings[3].value.items[1].value, "Ingy döt Net");
        chai_1.assert.equal(document.mappings[3].value.items[2].value, "Oren Ben-Kiki");
        chai_1.assert.equal(document.mappings[4].key.value, "Block-style");
        chai_1.assert.equal(document.mappings[4].value.value, "String: just a theory.");
        chai_1.assert.equal(document.mappings[5].key.value, "Flow-style");
        chai_1.assert.equal(document.mappings[5].value.value, "String: just a theory.");
    });
    test('JSON YAML schema should validate all properties successfully', function () {
        const jsonSchema = fs.readFileSync(path.join('test', 'testFixtures', 'json_schema.yaml'));
        const document = YAML.load(jsonSchema.toString(), {
            schema: json
        });
        chai_1.assert.equal(document.errors.length, 0);
        chai_1.assert.equal(document.mappings[0].key.value, "YAML");
        chai_1.assert.equal(document.mappings[0].value.value, "true");
        chai_1.assert.equal(document.mappings[1].key.value, "Pluto");
        chai_1.assert.equal(document.mappings[1].value.value, "false");
        chai_1.assert.equal(document.mappings[2].key.value, "negativeInt");
        chai_1.assert.equal(document.mappings[2].value.value, "-12");
        chai_1.assert.equal(document.mappings[3].key.value, "zeroInt");
        chai_1.assert.equal(document.mappings[3].value.value, "0");
        chai_1.assert.equal(document.mappings[4].key.value, "positiveInt");
        chai_1.assert.equal(document.mappings[4].value.value, "34");
        chai_1.assert.equal(document.mappings[5].key.value, "negativeFloat");
        chai_1.assert.equal(document.mappings[5].value.value, "-1");
        chai_1.assert.equal(document.mappings[6].key.value, "zeroFloat");
        chai_1.assert.equal(document.mappings[6].value.value, "0");
        chai_1.assert.equal(document.mappings[7].key.value, "positiveFloat");
        chai_1.assert.equal(document.mappings[7].value.value, "2.3e4");
        chai_1.assert.equal(document.mappings[8].key.value, "infinityFloat");
        chai_1.assert.equal(document.mappings[8].value.value, ".inf");
        chai_1.assert.equal(document.mappings[9].key.value, "notFloat");
        chai_1.assert.equal(document.mappings[9].value.value, ".nan");
        chai_1.assert.equal(document.mappings[10].key.value, "null");
        chai_1.assert.equal(document.mappings[10].value.value, "value for null key");
        chai_1.assert.equal(document.mappings[11].key.value, "key with null value");
        chai_1.assert.equal(document.mappings[11].value.value, "null");
    });
    test('Core YAML schema should validate all properties successfully', function () {
        const jsonSchema = fs.readFileSync(path.join('test', 'testFixtures', 'core_schema.yaml'));
        const document = YAML.load(jsonSchema.toString(), {
            schema: core
        });
        chai_1.assert.equal(document.errors.length, 0);
        chai_1.assert.equal(document.mappings[0].key.value, "A null");
        chai_1.assert.equal(document.mappings[0].value.value, "null");
        chai_1.assert.equal(document.mappings[1].key.value, "Also a null");
        chai_1.assert.equal(document.mappings[1].value, null);
        chai_1.assert.equal(document.mappings[2].key.value, "Not a null");
        chai_1.assert.equal(document.mappings[2].value.value, "");
        chai_1.assert.equal(document.mappings[3].key.value, "Booleans");
        chai_1.assert.equal(document.mappings[3].value.items.length, 4);
        chai_1.assert.equal(document.mappings[3].value.items[0].value, 'true');
        chai_1.assert.equal(document.mappings[3].value.items[1].value, 'True');
        chai_1.assert.equal(document.mappings[3].value.items[2].value, 'false');
        chai_1.assert.equal(document.mappings[3].value.items[3].value, 'FALSE');
        chai_1.assert.equal(document.mappings[4].key.value, "Integers");
        chai_1.assert.equal(document.mappings[4].value.items.length, 4);
        chai_1.assert.equal(document.mappings[4].value.items[0].value, '0');
        chai_1.assert.equal(document.mappings[4].value.items[1].value, '0o7');
        chai_1.assert.equal(document.mappings[4].value.items[2].value, '0x3A');
        chai_1.assert.equal(document.mappings[4].value.items[3].value, '-19');
        chai_1.assert.equal(document.mappings[5].key.value, "Floats");
        chai_1.assert.equal(document.mappings[5].value.items.length, 5);
        chai_1.assert.equal(document.mappings[5].value.items[0].value, '0.');
        chai_1.assert.equal(document.mappings[5].value.items[1].value, '-0.0');
        chai_1.assert.equal(document.mappings[5].value.items[2].value, '.5');
        chai_1.assert.equal(document.mappings[5].value.items[3].value, '+12e03');
        chai_1.assert.equal(document.mappings[5].value.items[4].value, '-2E+05');
        chai_1.assert.equal(document.mappings[6].key.value, "Also floats");
        chai_1.assert.equal(document.mappings[6].value.items.length, 4);
        chai_1.assert.equal(document.mappings[6].value.items[0].value, '.inf');
        chai_1.assert.equal(document.mappings[6].value.items[1].value, '-.Inf');
        chai_1.assert.equal(document.mappings[6].value.items[2].value, '+.INF');
        chai_1.assert.equal(document.mappings[6].value.items[3].value, '.NAN');
    });
    test('omap tag should validate correctly', function () {
        const omap = fs.readFileSync(path.join('test', 'testFixtures', 'omap.yaml'));
        const document = YAML.load(omap.toString());
        chai_1.assert.equal(document.errors.length, 0);
        chai_1.assert.equal(document.mappings[0].key.value, "Bestiary");
        chai_1.assert.equal(document.mappings[0].value.items.length, 3);
        chai_1.assert.equal(document.mappings[0].value.items[0].mappings[0].key.value, "aardvark");
        chai_1.assert.equal(document.mappings[0].value.items[0].mappings[0].value.value, "African pig-like ant eater. Ugly.");
        chai_1.assert.equal(document.mappings[0].value.items[1].mappings[0].key.value, "anteater");
        chai_1.assert.equal(document.mappings[0].value.items[1].mappings[0].value.value, "South-American ant eater. Two species.");
        chai_1.assert.equal(document.mappings[0].value.items[2].mappings[0].key.value, "anaconda");
        chai_1.assert.equal(document.mappings[0].value.items[2].mappings[0].value.value, "South-American constrictor snake. Scaly.");
        chai_1.assert.equal(document.mappings[1].key.value, "Numbers");
        chai_1.assert.equal(document.mappings[1].value.items.length, 3);
        chai_1.assert.equal(document.mappings[1].value.items[0].mappings[0].key.value, "one");
        chai_1.assert.equal(document.mappings[1].value.items[0].mappings[0].value.value, "1");
        chai_1.assert.equal(document.mappings[1].value.items[1].mappings[0].key.value, "two");
        chai_1.assert.equal(document.mappings[1].value.items[1].mappings[0].value.value, "2");
        chai_1.assert.equal(document.mappings[1].value.items[2].mappings[0].key.value, "three");
        chai_1.assert.equal(document.mappings[1].value.items[2].mappings[0].value.value, "3");
    });
    test('set tag should validate correctly', function () {
        const set = fs.readFileSync(path.join('test', 'testFixtures', 'set.yaml'));
        const document = YAML.load(set.toString());
        chai_1.assert.equal(document.errors.length, 0);
        chai_1.assert.equal(document.mappings.length, 3);
        chai_1.assert.equal(document.mappings[0].key.value, "Mark McGwire");
        chai_1.assert.equal(document.mappings[0].value, null);
        chai_1.assert.equal(document.mappings[1].key.value, "Sammy Sosa");
        chai_1.assert.equal(document.mappings[1].value, null);
        chai_1.assert.equal(document.mappings[2].key.value, "Ken Griff");
        chai_1.assert.equal(document.mappings[2].value, null);
    });
    test('timestamp tag should validate correctly', function () {
        const timestamp = fs.readFileSync(path.join('test', 'testFixtures', 'timestamp.yaml'));
        const document = YAML.load(timestamp.toString());
        chai_1.assert.equal(document.errors.length, 0);
        chai_1.assert.equal(document.mappings.length, 4);
        chai_1.assert.equal(document.mappings[0].key.value, "canonical");
        chai_1.assert.equal(document.mappings[0].value.value, "2001-12-15T02:59:43.1Z");
        chai_1.assert.equal(document.mappings[1].key.value, "iso8601");
        chai_1.assert.equal(document.mappings[1].value.value, "2001-12-14t21:59:43.10-05:00");
        chai_1.assert.equal(document.mappings[2].key.value, "spaced");
        chai_1.assert.equal(document.mappings[2].value.value, "2001-12-14 21:59:43.10 -5");
        chai_1.assert.equal(document.mappings[3].key.value, "date");
        chai_1.assert.equal(document.mappings[3].value.value, "2002-12-14");
    });
});
//# sourceMappingURL=tags.test.js.map