/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.search;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Predicate;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexReaderContext;
import org.apache.lucene.index.LeafReaderContext;
import org.apache.lucene.index.ReaderUtil;
import org.apache.lucene.search.BoostQuery;
import org.apache.lucene.search.BulkScorer;
import org.apache.lucene.search.ConstantScoreQuery;
import org.apache.lucene.search.ConstantScoreScorer;
import org.apache.lucene.search.ConstantScoreWeight;
import org.apache.lucene.search.DocIdSet;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.search.LeafCollector;
import org.apache.lucene.search.Matches;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.QueryCache;
import org.apache.lucene.search.QueryCachingPolicy;
import org.apache.lucene.search.Scorable;
import org.apache.lucene.search.ScoreMode;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.search.ScorerSupplier;
import org.apache.lucene.search.Weight;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.Accountables;
import org.apache.lucene.util.BitDocIdSet;
import org.apache.lucene.util.FixedBitSet;
import org.apache.lucene.util.RamUsageEstimator;
import org.apache.lucene.util.RoaringDocIdSet;

public class LRUQueryCache
implements QueryCache,
Accountable {
    private final int maxSize;
    private final long maxRamBytesUsed;
    private final Predicate<LeafReaderContext> leavesToCache;
    private final Map<Query, Query> uniqueQueries;
    private final Set<Query> mostRecentlyUsedQueries;
    private final Map<IndexReader.CacheKey, LeafCache> cache;
    private final ReentrantLock lock;
    private final float skipCacheFactor;
    private volatile long ramBytesUsed;
    private volatile long hitCount;
    private volatile long missCount;
    private volatile long cacheCount;
    private volatile long cacheSize;

    public LRUQueryCache(int maxSize, long maxRamBytesUsed, Predicate<LeafReaderContext> leavesToCache, float skipCacheFactor) {
        this.maxSize = maxSize;
        this.maxRamBytesUsed = maxRamBytesUsed;
        this.leavesToCache = leavesToCache;
        if (!(skipCacheFactor >= 1.0f)) {
            throw new IllegalArgumentException("skipCacheFactor must be no less than 1, get " + skipCacheFactor);
        }
        this.skipCacheFactor = skipCacheFactor;
        this.uniqueQueries = new LinkedHashMap<Query, Query>(16, 0.75f, true);
        this.mostRecentlyUsedQueries = this.uniqueQueries.keySet();
        this.cache = new IdentityHashMap<IndexReader.CacheKey, LeafCache>();
        this.lock = new ReentrantLock();
        this.ramBytesUsed = 0L;
    }

    public LRUQueryCache(int maxSize, long maxRamBytesUsed) {
        this(maxSize, maxRamBytesUsed, new MinSegmentSizePredicate(10000), 10.0f);
    }

    protected void onHit(Object readerCoreKey, Query query) {
        assert (this.lock.isHeldByCurrentThread());
        ++this.hitCount;
    }

    protected void onMiss(Object readerCoreKey, Query query) {
        assert (this.lock.isHeldByCurrentThread());
        assert (query != null);
        ++this.missCount;
    }

    protected void onQueryCache(Query query, long ramBytesUsed) {
        assert (this.lock.isHeldByCurrentThread());
        this.ramBytesUsed += ramBytesUsed;
    }

    protected void onQueryEviction(Query query, long ramBytesUsed) {
        assert (this.lock.isHeldByCurrentThread());
        this.ramBytesUsed -= ramBytesUsed;
    }

    protected void onDocIdSetCache(Object readerCoreKey, long ramBytesUsed) {
        assert (this.lock.isHeldByCurrentThread());
        ++this.cacheSize;
        ++this.cacheCount;
        this.ramBytesUsed += ramBytesUsed;
    }

    protected void onDocIdSetEviction(Object readerCoreKey, int numEntries, long sumRamBytesUsed) {
        assert (this.lock.isHeldByCurrentThread());
        this.ramBytesUsed -= sumRamBytesUsed;
        this.cacheSize -= (long)numEntries;
    }

    protected void onClear() {
        assert (this.lock.isHeldByCurrentThread());
        this.ramBytesUsed = 0L;
        this.cacheSize = 0L;
    }

    boolean requiresEviction() {
        assert (this.lock.isHeldByCurrentThread());
        int size = this.mostRecentlyUsedQueries.size();
        if (size == 0) {
            return false;
        }
        return size > this.maxSize || this.ramBytesUsed() > this.maxRamBytesUsed;
    }

    CacheAndCount get(Query key, IndexReader.CacheHelper cacheHelper) {
        assert (this.lock.isHeldByCurrentThread());
        assert (!(key instanceof BoostQuery));
        assert (!(key instanceof ConstantScoreQuery));
        IndexReader.CacheKey readerKey = cacheHelper.getKey();
        LeafCache leafCache = this.cache.get(readerKey);
        if (leafCache == null) {
            this.onMiss(readerKey, key);
            return null;
        }
        Query singleton = this.uniqueQueries.get(key);
        if (singleton == null) {
            this.onMiss(readerKey, key);
            return null;
        }
        CacheAndCount cached = leafCache.get(singleton);
        if (cached == null) {
            this.onMiss(readerKey, singleton);
        } else {
            this.onHit(readerKey, singleton);
        }
        return cached;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void putIfAbsent(Query query, CacheAndCount cached, IndexReader.CacheHelper cacheHelper) {
        assert (!(query instanceof BoostQuery));
        assert (!(query instanceof ConstantScoreQuery));
        this.lock.lock();
        try {
            Query singleton = this.uniqueQueries.putIfAbsent(query, query);
            if (singleton == null) {
                this.onQueryCache(query, LRUQueryCache.getRamBytesUsed(query));
            } else {
                query = singleton;
            }
            IndexReader.CacheKey key = cacheHelper.getKey();
            LeafCache leafCache = this.cache.get(key);
            if (leafCache == null) {
                leafCache = new LeafCache(key);
                LeafCache previous = this.cache.put(key, leafCache);
                this.ramBytesUsed += RamUsageEstimator.HASHTABLE_RAM_BYTES_PER_ENTRY;
                assert (previous == null);
                cacheHelper.addClosedListener(this::clearCoreCacheKey);
            }
            leafCache.putIfAbsent(query, cached);
            this.evictIfNecessary();
        }
        finally {
            this.lock.unlock();
        }
    }

    private void evictIfNecessary() {
        assert (this.lock.isHeldByCurrentThread());
        if (this.requiresEviction()) {
            Iterator<Query> iterator = this.mostRecentlyUsedQueries.iterator();
            do {
                Query query = iterator.next();
                int size = this.mostRecentlyUsedQueries.size();
                iterator.remove();
                if (size == this.mostRecentlyUsedQueries.size()) {
                    throw new ConcurrentModificationException("Removal from the cache failed! This is probably due to a query which has been modified after having been put into  the cache or a badly implemented clone(). Query class: [" + query.getClass() + "], query: [" + query + "]");
                }
                this.onEviction(query);
            } while (iterator.hasNext() && this.requiresEviction());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void clearCoreCacheKey(Object coreKey) {
        this.lock.lock();
        try {
            LeafCache leafCache = this.cache.remove(coreKey);
            if (leafCache != null) {
                this.ramBytesUsed -= RamUsageEstimator.HASHTABLE_RAM_BYTES_PER_ENTRY;
                int numEntries = leafCache.cache.size();
                if (numEntries > 0) {
                    this.onDocIdSetEviction(coreKey, numEntries, leafCache.ramBytesUsed);
                } else {
                    assert (numEntries == 0);
                    assert (leafCache.ramBytesUsed == 0L);
                }
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public void clearQuery(Query query) {
        this.lock.lock();
        try {
            Query singleton = this.uniqueQueries.remove(query);
            if (singleton != null) {
                this.onEviction(singleton);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void onEviction(Query singleton) {
        assert (this.lock.isHeldByCurrentThread());
        this.onQueryEviction(singleton, LRUQueryCache.getRamBytesUsed(singleton));
        for (LeafCache leafCache : this.cache.values()) {
            leafCache.remove(singleton);
        }
    }

    public void clear() {
        this.lock.lock();
        try {
            this.cache.clear();
            this.mostRecentlyUsedQueries.clear();
            this.onClear();
        }
        finally {
            this.lock.unlock();
        }
    }

    private static long getRamBytesUsed(Query query) {
        if (query instanceof Accountable) {
            return RamUsageEstimator.LINKED_HASHTABLE_RAM_BYTES_PER_ENTRY + ((Accountable)((Object)query)).ramBytesUsed();
        }
        return RamUsageEstimator.LINKED_HASHTABLE_RAM_BYTES_PER_ENTRY + 1024L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void assertConsistent() {
        this.lock.lock();
        try {
            if (this.requiresEviction()) {
                throw new AssertionError((Object)("requires evictions: size=" + this.mostRecentlyUsedQueries.size() + ", maxSize=" + this.maxSize + ", ramBytesUsed=" + this.ramBytesUsed() + ", maxRamBytesUsed=" + this.maxRamBytesUsed));
            }
            for (LeafCache leafCache : this.cache.values()) {
                Iterator<LeafCache> keys = Collections.newSetFromMap(new IdentityHashMap());
                keys.addAll(leafCache.cache.keySet());
                keys.removeAll(this.mostRecentlyUsedQueries);
                if (!keys.isEmpty()) {
                    throw new AssertionError((Object)("One leaf cache contains more keys than the top-level cache: " + keys));
                }
            }
            long recomputedRamBytesUsed = RamUsageEstimator.HASHTABLE_RAM_BYTES_PER_ENTRY * (long)this.cache.size();
            for (Query query : this.mostRecentlyUsedQueries) {
                recomputedRamBytesUsed += LRUQueryCache.getRamBytesUsed(query);
            }
            for (LeafCache leafCache : this.cache.values()) {
                recomputedRamBytesUsed += RamUsageEstimator.HASHTABLE_RAM_BYTES_PER_ENTRY * (long)leafCache.cache.size();
                for (CacheAndCount cached : leafCache.cache.values()) {
                    recomputedRamBytesUsed += cached.ramBytesUsed();
                }
            }
            if (recomputedRamBytesUsed != this.ramBytesUsed) {
                throw new AssertionError((Object)("ramBytesUsed mismatch : " + this.ramBytesUsed + " != " + recomputedRamBytesUsed));
            }
            long recomputedCacheSize = 0L;
            for (LeafCache leafCache : this.cache.values()) {
                recomputedCacheSize += (long)leafCache.cache.size();
            }
            if (recomputedCacheSize != this.getCacheSize()) {
                throw new AssertionError((Object)("cacheSize mismatch : " + this.getCacheSize() + " != " + recomputedCacheSize));
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    List<Query> cachedQueries() {
        this.lock.lock();
        try {
            ArrayList<Query> arrayList = new ArrayList<Query>(this.mostRecentlyUsedQueries);
            return arrayList;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public Weight doCache(Weight weight, QueryCachingPolicy policy) {
        while (weight instanceof CachingWrapperWeight) {
            weight = ((CachingWrapperWeight)weight).in;
        }
        return new CachingWrapperWeight(weight, policy);
    }

    @Override
    public long ramBytesUsed() {
        return this.ramBytesUsed;
    }

    @Override
    public Collection<Accountable> getChildResources() {
        this.lock.lock();
        try {
            Collection<Accountable> collection = Accountables.namedAccountables("segment", this.cache);
            return collection;
        }
        finally {
            this.lock.unlock();
        }
    }

    protected CacheAndCount cacheImpl(BulkScorer scorer, int maxDoc) throws IOException {
        if (scorer.cost() * 100L >= (long)maxDoc) {
            return LRUQueryCache.cacheIntoBitSet(scorer, maxDoc);
        }
        return LRUQueryCache.cacheIntoRoaringDocIdSet(scorer, maxDoc);
    }

    private static CacheAndCount cacheIntoBitSet(BulkScorer scorer, int maxDoc) throws IOException {
        final FixedBitSet bitSet = new FixedBitSet(maxDoc);
        final int[] count = new int[1];
        scorer.score(new LeafCollector(){

            @Override
            public void setScorer(Scorable scorer) throws IOException {
            }

            @Override
            public void collect(int doc) throws IOException {
                count[0] = count[0] + 1;
                bitSet.set(doc);
            }
        }, null);
        return new CacheAndCount(new BitDocIdSet(bitSet, count[0]), count[0]);
    }

    private static CacheAndCount cacheIntoRoaringDocIdSet(BulkScorer scorer, int maxDoc) throws IOException {
        final RoaringDocIdSet.Builder builder = new RoaringDocIdSet.Builder(maxDoc);
        scorer.score(new LeafCollector(){

            @Override
            public void setScorer(Scorable scorer) throws IOException {
            }

            @Override
            public void collect(int doc) throws IOException {
                builder.add(doc);
            }
        }, null);
        RoaringDocIdSet cache = builder.build();
        return new CacheAndCount(cache, cache.cardinality());
    }

    public final long getTotalCount() {
        return this.getHitCount() + this.getMissCount();
    }

    public final long getHitCount() {
        return this.hitCount;
    }

    public final long getMissCount() {
        return this.missCount;
    }

    public final long getCacheSize() {
        return this.cacheSize;
    }

    public final long getCacheCount() {
        return this.cacheCount;
    }

    public final long getEvictionCount() {
        return this.getCacheCount() - this.getCacheSize();
    }

    protected static class CacheAndCount
    implements Accountable {
        protected static final CacheAndCount EMPTY = new CacheAndCount(DocIdSet.EMPTY, 0);
        private static final long BASE_RAM_BYTES_USED = RamUsageEstimator.shallowSizeOfInstance(CacheAndCount.class);
        private final DocIdSet cache;
        private final int count;

        public CacheAndCount(DocIdSet cache, int count) {
            this.cache = cache;
            this.count = count;
        }

        public DocIdSetIterator iterator() throws IOException {
            return this.cache.iterator();
        }

        public int count() {
            return this.count;
        }

        @Override
        public long ramBytesUsed() {
            return BASE_RAM_BYTES_USED + this.cache.ramBytesUsed();
        }
    }

    private class CachingWrapperWeight
    extends ConstantScoreWeight {
        private final Weight in;
        private final QueryCachingPolicy policy;
        private final AtomicBoolean used;

        CachingWrapperWeight(Weight in, QueryCachingPolicy policy) {
            super(in.getQuery(), 1.0f);
            this.in = in;
            this.policy = policy;
            this.used = new AtomicBoolean(false);
        }

        @Override
        public Matches matches(LeafReaderContext context, int doc) throws IOException {
            return this.in.matches(context, doc);
        }

        private boolean cacheEntryHasReasonableWorstCaseSize(int maxDoc) {
            long worstCaseRamUsage = maxDoc / 8;
            long totalRamAvailable = LRUQueryCache.this.maxRamBytesUsed;
            return worstCaseRamUsage * 5L < totalRamAvailable;
        }

        private CacheAndCount cache(LeafReaderContext context) throws IOException {
            BulkScorer scorer = this.in.bulkScorer(context);
            if (scorer == null) {
                return CacheAndCount.EMPTY;
            }
            return LRUQueryCache.this.cacheImpl(scorer, context.reader().maxDoc());
        }

        private boolean shouldCache(LeafReaderContext context) throws IOException {
            return this.cacheEntryHasReasonableWorstCaseSize(ReaderUtil.getTopLevelContext(context).reader().maxDoc()) && LRUQueryCache.this.leavesToCache.test(context);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public ScorerSupplier scorerSupplier(final LeafReaderContext context) throws IOException {
            CacheAndCount cached;
            if (this.used.compareAndSet(false, true)) {
                this.policy.onUse(this.getQuery());
            }
            if (!this.in.isCacheable(context)) {
                return this.in.scorerSupplier(context);
            }
            if (!this.shouldCache(context)) {
                return this.in.scorerSupplier(context);
            }
            final IndexReader.CacheHelper cacheHelper = context.reader().getCoreCacheHelper();
            if (cacheHelper == null) {
                return this.in.scorerSupplier(context);
            }
            if (!LRUQueryCache.this.lock.tryLock()) {
                return this.in.scorerSupplier(context);
            }
            try {
                cached = LRUQueryCache.this.get(this.in.getQuery(), cacheHelper);
            }
            finally {
                LRUQueryCache.this.lock.unlock();
            }
            if (cached == null) {
                if (this.policy.shouldCache(this.in.getQuery())) {
                    final ScorerSupplier supplier = this.in.scorerSupplier(context);
                    if (supplier == null) {
                        LRUQueryCache.this.putIfAbsent(this.in.getQuery(), CacheAndCount.EMPTY, cacheHelper);
                        return null;
                    }
                    final long cost = supplier.cost();
                    return new ScorerSupplier(){

                        @Override
                        public Scorer get(long leadCost) throws IOException {
                            if ((float)cost / LRUQueryCache.this.skipCacheFactor > (float)leadCost) {
                                return supplier.get(leadCost);
                            }
                            Scorer scorer = supplier.get(Long.MAX_VALUE);
                            CacheAndCount cached = LRUQueryCache.this.cacheImpl(new Weight.DefaultBulkScorer(scorer), context.reader().maxDoc());
                            LRUQueryCache.this.putIfAbsent(CachingWrapperWeight.this.in.getQuery(), cached, cacheHelper);
                            DocIdSetIterator disi = cached.iterator();
                            if (disi == null) {
                                disi = DocIdSetIterator.empty();
                            }
                            return new ConstantScoreScorer((Weight)CachingWrapperWeight.this, 0.0f, ScoreMode.COMPLETE_NO_SCORES, disi);
                        }

                        @Override
                        public long cost() {
                            return cost;
                        }
                    };
                }
                return this.in.scorerSupplier(context);
            }
            assert (cached != null);
            if (cached == CacheAndCount.EMPTY) {
                return null;
            }
            final DocIdSetIterator disi = cached.iterator();
            if (disi == null) {
                return null;
            }
            return new ScorerSupplier(){

                @Override
                public Scorer get(long LeadCost) throws IOException {
                    return new ConstantScoreScorer((Weight)CachingWrapperWeight.this, 0.0f, ScoreMode.COMPLETE_NO_SCORES, disi);
                }

                @Override
                public long cost() {
                    return disi.cost();
                }
            };
        }

        @Override
        public Scorer scorer(LeafReaderContext context) throws IOException {
            ScorerSupplier scorerSupplier = this.scorerSupplier(context);
            if (scorerSupplier == null) {
                return null;
            }
            return scorerSupplier.get(Long.MAX_VALUE);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public int count(LeafReaderContext context) throws IOException {
            CacheAndCount cached;
            int innerCount = this.in.count(context);
            if (innerCount != -1) {
                return innerCount;
            }
            if (context.reader().hasDeletions()) {
                return -1;
            }
            if (this.used.compareAndSet(false, true)) {
                this.policy.onUse(this.getQuery());
            }
            if (!this.in.isCacheable(context)) {
                return -1;
            }
            if (!this.shouldCache(context)) {
                return -1;
            }
            IndexReader.CacheHelper cacheHelper = context.reader().getCoreCacheHelper();
            if (cacheHelper == null) {
                return -1;
            }
            if (!LRUQueryCache.this.lock.tryLock()) {
                return -1;
            }
            try {
                cached = LRUQueryCache.this.get(this.in.getQuery(), cacheHelper);
            }
            finally {
                LRUQueryCache.this.lock.unlock();
            }
            if (cached == null) {
                return -1;
            }
            return cached.count();
        }

        @Override
        public boolean isCacheable(LeafReaderContext ctx) {
            return this.in.isCacheable(ctx);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public BulkScorer bulkScorer(LeafReaderContext context) throws IOException {
            CacheAndCount cached;
            if (this.used.compareAndSet(false, true)) {
                this.policy.onUse(this.getQuery());
            }
            if (!this.in.isCacheable(context)) {
                return this.in.bulkScorer(context);
            }
            if (!this.shouldCache(context)) {
                return this.in.bulkScorer(context);
            }
            IndexReader.CacheHelper cacheHelper = context.reader().getCoreCacheHelper();
            if (cacheHelper == null) {
                return this.in.bulkScorer(context);
            }
            if (!LRUQueryCache.this.lock.tryLock()) {
                return this.in.bulkScorer(context);
            }
            try {
                cached = LRUQueryCache.this.get(this.in.getQuery(), cacheHelper);
            }
            finally {
                LRUQueryCache.this.lock.unlock();
            }
            if (cached == null) {
                if (this.policy.shouldCache(this.in.getQuery())) {
                    cached = this.cache(context);
                    LRUQueryCache.this.putIfAbsent(this.in.getQuery(), cached, cacheHelper);
                } else {
                    return this.in.bulkScorer(context);
                }
            }
            assert (cached != null);
            if (cached == CacheAndCount.EMPTY) {
                return null;
            }
            DocIdSetIterator disi = cached.iterator();
            if (disi == null) {
                return null;
            }
            return new Weight.DefaultBulkScorer(new ConstantScoreScorer((Weight)this, 0.0f, ScoreMode.COMPLETE_NO_SCORES, disi));
        }
    }

    private class LeafCache
    implements Accountable {
        private final Object key;
        private final Map<Query, CacheAndCount> cache;
        private volatile long ramBytesUsed;

        LeafCache(Object key) {
            this.key = key;
            this.cache = new IdentityHashMap<Query, CacheAndCount>();
            this.ramBytesUsed = 0L;
        }

        private void onDocIdSetCache(long ramBytesUsed) {
            this.ramBytesUsed += ramBytesUsed;
            LRUQueryCache.this.onDocIdSetCache(this.key, ramBytesUsed);
        }

        private void onDocIdSetEviction(long ramBytesUsed) {
            this.ramBytesUsed -= ramBytesUsed;
            LRUQueryCache.this.onDocIdSetEviction(this.key, 1, ramBytesUsed);
        }

        CacheAndCount get(Query query) {
            assert (!(query instanceof BoostQuery));
            assert (!(query instanceof ConstantScoreQuery));
            return this.cache.get(query);
        }

        void putIfAbsent(Query query, CacheAndCount cached) {
            assert (!(query instanceof BoostQuery));
            assert (!(query instanceof ConstantScoreQuery));
            if (this.cache.putIfAbsent(query, cached) == null) {
                this.onDocIdSetCache(RamUsageEstimator.HASHTABLE_RAM_BYTES_PER_ENTRY + cached.ramBytesUsed());
            }
        }

        void remove(Query query) {
            assert (!(query instanceof BoostQuery));
            assert (!(query instanceof ConstantScoreQuery));
            CacheAndCount removed = this.cache.remove(query);
            if (removed != null) {
                this.onDocIdSetEviction(RamUsageEstimator.HASHTABLE_RAM_BYTES_PER_ENTRY + removed.ramBytesUsed());
            }
        }

        @Override
        public long ramBytesUsed() {
            return this.ramBytesUsed;
        }
    }

    static class MinSegmentSizePredicate
    implements Predicate<LeafReaderContext> {
        private final int minSize;

        MinSegmentSizePredicate(int minSize) {
            this.minSize = minSize;
        }

        @Override
        public boolean test(LeafReaderContext context) {
            int maxDoc = context.reader().maxDoc();
            if (maxDoc < this.minSize) {
                return false;
            }
            IndexReaderContext topLevelContext = ReaderUtil.getTopLevelContext(context);
            int averageTotalDocs = topLevelContext.reader().maxDoc() / topLevelContext.leaves().size();
            return maxDoc * 2 > averageTotalDocs;
        }
    }
}

