"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.create = void 0;
const path = require("path-browserify");
const semver = require("semver");
const tsWithImportCache = require("typescript-auto-import-cache");
const getFormatCodeSettings_1 = require("../configs/getFormatCodeSettings");
const getUserPreferences_1 = require("../configs/getUserPreferences");
const shared_1 = require("../shared");
const lspConverters_1 = require("../utils/lspConverters");
const snippetForFunctionCall_1 = require("../utils/snippetForFunctionCall");
const codeActions = require("../semanticFeatures/codeAction");
const codeActionResolve = require("../semanticFeatures/codeActionResolve");
const semanticTokens = require("../semanticFeatures/semanticTokens");
const documentRegistries = [];
function getDocumentRegistry(ts, useCaseSensitiveFileNames, currentDirectory) {
    let documentRegistry = documentRegistries.find(item => item[0] === useCaseSensitiveFileNames && item[1] === currentDirectory)?.[2];
    if (!documentRegistry) {
        documentRegistry = ts.createDocumentRegistry(useCaseSensitiveFileNames, currentDirectory);
        documentRegistries.push([useCaseSensitiveFileNames, currentDirectory, documentRegistry]);
    }
    return documentRegistry;
}
function create(ts, { isValidationEnabled = async (document, context) => {
    return await context.env.getConfiguration?.((0, shared_1.getConfigTitle)(document) + '.validate.enable') ?? true;
}, isSuggestionsEnabled = async (document, context) => {
    return await context.env.getConfiguration?.((0, shared_1.getConfigTitle)(document) + '.suggest.enabled') ?? true;
}, } = {}) {
    return {
        name: 'typescript-semantic',
        triggerCharacters: getBasicTriggerCharacters(ts.version),
        signatureHelpTriggerCharacters: ['(', ',', '<'],
        signatureHelpRetriggerCharacters: [')'],
        create(context) {
            if (!context.language.typescript) {
                return {};
            }
            const { projectHost, languageServiceHost } = context.language.typescript;
            const sys = projectHost;
            const created = tsWithImportCache.createLanguageService(ts, sys, languageServiceHost, proxiedHost => ts.createLanguageService(proxiedHost, getDocumentRegistry(ts, sys.useCaseSensitiveFileNames, languageServiceHost.getCurrentDirectory())));
            const { languageService } = created;
            if (created.setPreferences && context.env.getConfiguration) {
                updatePreferences();
                context.env.onDidChangeConfiguration?.(updatePreferences);
                async function updatePreferences() {
                    const preferences = await context.env.getConfiguration?.('typescript.preferences');
                    if (preferences) {
                        created.setPreferences?.(preferences);
                    }
                }
            }
            if (created.projectUpdated) {
                const sourceScriptNames = new Set();
                const normalizeFileName = sys.useCaseSensitiveFileNames
                    ? (id) => id
                    : (id) => id.toLowerCase();
                updateSourceScriptFileNames();
                context.env.onDidChangeWatchedFiles?.(params => {
                    const someFileCreateOrDeiete = params.changes.some(change => change.type !== 2);
                    if (someFileCreateOrDeiete) {
                        updateSourceScriptFileNames();
                    }
                    for (const change of params.changes) {
                        const fileName = context.env.typescript.uriToFileName(change.uri);
                        if (sourceScriptNames.has(normalizeFileName(fileName))) {
                            created.projectUpdated?.(languageServiceHost.getCurrentDirectory());
                        }
                    }
                });
                function updateSourceScriptFileNames() {
                    sourceScriptNames.clear();
                    for (const fileName of languageServiceHost.getScriptFileNames()) {
                        const uri = context.env.typescript.fileNameToUri(fileName);
                        const sourceScript = context.language.scripts.get(uri);
                        if (sourceScript?.generated) {
                            const tsCode = sourceScript.generated.languagePlugin.typescript?.getServiceScript(sourceScript.generated.root);
                            if (tsCode) {
                                sourceScriptNames.add(normalizeFileName(fileName));
                            }
                        }
                        else if (sourceScript) {
                            sourceScriptNames.add(normalizeFileName(fileName));
                        }
                    }
                }
            }
            const ctx = {
                ...context,
                languageServiceHost,
                languageService,
                uriToFileName(uri) {
                    const virtualScript = getVirtualScriptByUri(uri);
                    if (virtualScript) {
                        return virtualScript.fileName;
                    }
                    return context.env.typescript.uriToFileName(uri);
                },
                fileNameToUri(fileName) {
                    const uri = context.env.typescript.fileNameToUri(fileName);
                    const sourceScript = context.language.scripts.get(uri);
                    const extraServiceScript = context.language.typescript.getExtraServiceScript(fileName);
                    let virtualCode = extraServiceScript?.code;
                    if (!virtualCode && sourceScript?.generated?.languagePlugin.typescript) {
                        const serviceScript = sourceScript.generated.languagePlugin.typescript.getServiceScript(sourceScript.generated.root);
                        if (serviceScript) {
                            virtualCode = serviceScript.code;
                        }
                    }
                    if (sourceScript && virtualCode) {
                        return context.encodeEmbeddedDocumentUri(sourceScript.id, virtualCode.id);
                    }
                    return uri;
                },
                getTextDocument(uri) {
                    const decoded = context.decodeEmbeddedDocumentUri(uri);
                    if (decoded) {
                        const sourceScript = context.language.scripts.get(decoded[0]);
                        const virtualCode = sourceScript?.generated?.embeddedCodes.get(decoded[1]);
                        if (virtualCode) {
                            return context.documents.get(uri, virtualCode.languageId, virtualCode.snapshot);
                        }
                    }
                    else {
                        const sourceFile = context.language.scripts.get(uri);
                        if (sourceFile) {
                            return context.documents.get(uri, sourceFile.languageId, sourceFile.snapshot);
                        }
                    }
                },
            };
            const getCodeActions = codeActions.register(ctx);
            const doCodeActionResolve = codeActionResolve.register(ctx);
            const getDocumentSemanticTokens = semanticTokens.register(ts, ctx);
            /* typescript-language-features is hardcode true */
            const renameInfoOptions = { allowRenameOfImportPath: true };
            let formattingOptions;
            return {
                provide: {
                    'typescript/languageService': () => languageService,
                    'typescript/languageServiceHost': () => languageServiceHost,
                    'typescript/documentFileName': uri => ctx.uriToFileName(uri),
                    'typescript/documentUri': fileName => ctx.fileNameToUri(fileName),
                },
                dispose() {
                    languageService.dispose();
                },
                provideDocumentFormattingEdits(_document, _range, options) {
                    formattingOptions = options;
                    return undefined;
                },
                provideOnTypeFormattingEdits(_document, _position, _key, options) {
                    formattingOptions = options;
                    return undefined;
                },
                async provideCompletionItems(document, position, completeContext, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    if (!await isSuggestionsEnabled(document, context)) {
                        return;
                    }
                    return await worker(token, async () => {
                        const preferences = await (0, getUserPreferences_1.getUserPreferences)(ctx, document);
                        const fileName = ctx.uriToFileName(document.uri);
                        const offset = document.offsetAt(position);
                        const info = (0, shared_1.safeCall)(() => ctx.languageService.getCompletionsAtPosition(fileName, offset, {
                            ...preferences,
                            triggerCharacter: completeContext.triggerCharacter,
                            triggerKind: completeContext.triggerKind,
                        }));
                        if (info) {
                            return (0, lspConverters_1.convertCompletionInfo)(ts, info, document, position, tsEntry => ({
                                uri: document.uri,
                                fileName,
                                offset,
                                originalItem: {
                                    name: tsEntry.name,
                                    source: tsEntry.source,
                                    data: tsEntry.data,
                                    labelDetails: tsEntry.labelDetails,
                                },
                            }));
                        }
                    });
                },
                async resolveCompletionItem(item, token) {
                    return await worker(token, async () => {
                        const data = item.data;
                        if (!data) {
                            return item;
                        }
                        const { fileName, offset } = data;
                        const document = ctx.getTextDocument(data.uri);
                        const [formatOptions, preferences] = await Promise.all([
                            (0, getFormatCodeSettings_1.getFormatCodeSettings)(ctx, document, formattingOptions),
                            (0, getUserPreferences_1.getUserPreferences)(ctx, document),
                        ]);
                        const details = (0, shared_1.safeCall)(() => ctx.languageService.getCompletionEntryDetails(fileName, offset, data.originalItem.name, formatOptions, data.originalItem.source, preferences, data.originalItem.data));
                        if (!details) {
                            return item;
                        }
                        if (data.originalItem.labelDetails) {
                            item.labelDetails ??= {};
                            Object.assign(item.labelDetails, data.originalItem.labelDetails);
                        }
                        (0, lspConverters_1.applyCompletionEntryDetails)(ts, item, details, document, ctx.fileNameToUri, ctx.getTextDocument);
                        const useCodeSnippetsOnMethodSuggest = await ctx.env.getConfiguration?.((0, shared_1.getConfigTitle)(document) + '.suggest.completeFunctionCalls') ?? false;
                        const useCodeSnippet = useCodeSnippetsOnMethodSuggest
                            && (item.kind === 3
                                || item.kind === 2);
                        if (useCodeSnippet) {
                            const shouldCompleteFunction = isValidFunctionCompletionContext(ctx.languageService, fileName, offset, document);
                            if (shouldCompleteFunction) {
                                const { snippet, parameterCount } = (0, snippetForFunctionCall_1.snippetForFunctionCall)({
                                    insertText: item.insertText ?? item.textEdit?.newText, // insertText is dropped by LSP in some case: https://github.com/microsoft/vscode-languageserver-node/blob/9b742021fb04ad081aa3676a9eecf4fa612084b4/client/src/common/codeConverter.ts#L659-L664
                                    label: item.label,
                                }, details.displayParts);
                                if (item.textEdit) {
                                    item.textEdit.newText = snippet;
                                }
                                if (item.insertText) {
                                    item.insertText = snippet;
                                }
                                item.insertTextFormat = 2;
                                if (parameterCount > 0) {
                                    //Fix for https://github.com/microsoft/vscode/issues/104059
                                    //Don't show parameter hints if "editor.parameterHints.enabled": false
                                    // if (await getConfiguration('editor.parameterHints.enabled', document.uri)) {
                                    // 	item.command = {
                                    // 		title: 'triggerParameterHints',
                                    // 		command: 'editor.action.triggerParameterHints',
                                    // 	};
                                    // }
                                }
                            }
                        }
                        return item;
                        function isValidFunctionCompletionContext(client, filepath, offset, document) {
                            // Workaround for https://github.com/microsoft/TypeScript/issues/12677
                            // Don't complete function calls inside of destructive assignments or imports
                            try {
                                const response = client.getQuickInfoAtPosition(filepath, offset);
                                if (response) {
                                    switch (response.kind) {
                                        case 'var':
                                        case 'let':
                                        case 'const':
                                        case 'alias':
                                            return false;
                                    }
                                }
                            }
                            catch {
                                // Noop
                            }
                            // Don't complete function call if there is already something that looks like a function call
                            // https://github.com/microsoft/vscode/issues/18131
                            const position = document.positionAt(offset);
                            const after = (0, lspConverters_1.getLineText)(document, position.line).slice(position.character);
                            return after.match(/^[a-z_$0-9]*\s*\(/gi) === null;
                        }
                    }) ?? item;
                },
                provideRenameRange(document, position, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    return worker(token, () => {
                        const fileName = ctx.uriToFileName(document.uri);
                        const offset = document.offsetAt(position);
                        const renameInfo = (0, shared_1.safeCall)(() => ctx.languageService.getRenameInfo(fileName, offset, renameInfoOptions));
                        if (!renameInfo) {
                            return;
                        }
                        if (!renameInfo.canRename) {
                            return { message: renameInfo.localizedErrorMessage };
                        }
                        return (0, lspConverters_1.convertTextSpan)(renameInfo.triggerSpan, document);
                    });
                },
                provideRenameEdits(document, position, newName, token) {
                    if (!isSemanticDocument(document, true)) {
                        return;
                    }
                    return worker(token, async () => {
                        const fileName = ctx.uriToFileName(document.uri);
                        const offset = document.offsetAt(position);
                        const renameInfo = (0, shared_1.safeCall)(() => ctx.languageService.getRenameInfo(fileName, offset, renameInfoOptions));
                        if (!renameInfo?.canRename) {
                            return;
                        }
                        if (renameInfo.fileToRename) {
                            const [formatOptions, preferences] = await Promise.all([
                                (0, getFormatCodeSettings_1.getFormatCodeSettings)(ctx, document, formattingOptions),
                                (0, getUserPreferences_1.getUserPreferences)(ctx, document),
                            ]);
                            return renameFile(renameInfo.fileToRename, newName, formatOptions, preferences);
                        }
                        const { providePrefixAndSuffixTextForRename } = await (0, getUserPreferences_1.getUserPreferences)(ctx, document);
                        const entries = ctx.languageService.findRenameLocations(fileName, offset, false, false, providePrefixAndSuffixTextForRename);
                        if (!entries) {
                            return;
                        }
                        return (0, lspConverters_1.convertRenameLocations)(newName, entries, ctx.fileNameToUri, ctx.getTextDocument);
                        function renameFile(fileToRename, newName, formatOptions, preferences) {
                            // Make sure we preserve file extension if none provided
                            if (!path.extname(newName)) {
                                newName += path.extname(fileToRename);
                            }
                            const dirname = path.dirname(fileToRename);
                            const newFilePath = path.join(dirname, newName);
                            const response = (0, shared_1.safeCall)(() => ctx.languageService.getEditsForFileRename(fileToRename, newFilePath, formatOptions, preferences));
                            if (!response) {
                                return;
                            }
                            const edits = (0, lspConverters_1.convertFileTextChanges)(response, ctx.fileNameToUri, ctx.getTextDocument);
                            if (!edits.documentChanges) {
                                edits.documentChanges = [];
                            }
                            edits.documentChanges.push({
                                kind: 'rename',
                                oldUri: ctx.fileNameToUri(fileToRename),
                                newUri: ctx.fileNameToUri(newFilePath),
                            });
                            return edits;
                        }
                    });
                },
                provideCodeActions(document, range, context, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    return worker(token, () => {
                        return getCodeActions(document, range, context, formattingOptions);
                    });
                },
                async resolveCodeAction(codeAction, token) {
                    return await worker(token, () => {
                        return doCodeActionResolve(codeAction, formattingOptions);
                    }) ?? codeAction;
                },
                provideInlayHints(document, range, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    return worker(token, async () => {
                        const preferences = await (0, getUserPreferences_1.getUserPreferences)(ctx, document);
                        const fileName = ctx.uriToFileName(document.uri);
                        const start = document.offsetAt(range.start);
                        const end = document.offsetAt(range.end);
                        const inlayHints = (0, shared_1.safeCall)(() => 'provideInlayHints' in ctx.languageService
                            ? ctx.languageService.provideInlayHints(fileName, { start, length: end - start }, preferences)
                            : []);
                        if (!inlayHints) {
                            return [];
                        }
                        return inlayHints.map(hint => (0, lspConverters_1.convertInlayHint)(hint, document));
                    });
                },
                provideCallHierarchyItems(document, position, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    return worker(token, () => {
                        const fileName = ctx.uriToFileName(document.uri);
                        const offset = document.offsetAt(position);
                        const calls = (0, shared_1.safeCall)(() => ctx.languageService.prepareCallHierarchy(fileName, offset));
                        if (!calls) {
                            return [];
                        }
                        const items = Array.isArray(calls) ? calls : [calls];
                        return items.map(item => (0, lspConverters_1.convertCallHierarchyItem)(item, ctx));
                    });
                },
                async provideCallHierarchyIncomingCalls(item, token) {
                    return await worker(token, () => {
                        const document = ctx.getTextDocument(item.uri);
                        const fileName = ctx.uriToFileName(item.uri);
                        const offset = document.offsetAt(item.selectionRange.start);
                        const calls = (0, shared_1.safeCall)(() => ctx.languageService.provideCallHierarchyIncomingCalls(fileName, offset));
                        if (!calls) {
                            return [];
                        }
                        const items = Array.isArray(calls) ? calls : [calls];
                        return items.map(item => (0, lspConverters_1.convertCallHierarchyIncomingCall)(item, ctx));
                    }) ?? [];
                },
                async provideCallHierarchyOutgoingCalls(item, token) {
                    return await worker(token, () => {
                        const document = ctx.getTextDocument(item.uri);
                        const fileName = ctx.uriToFileName(item.uri);
                        const offset = document.offsetAt(item.selectionRange.start);
                        const calls = (0, shared_1.safeCall)(() => ctx.languageService.provideCallHierarchyOutgoingCalls(fileName, offset));
                        if (!calls) {
                            return [];
                        }
                        const items = Array.isArray(calls) ? calls : [calls];
                        return items.map(item => (0, lspConverters_1.convertCallHierarchyOutgoingCall)(item, document, ctx));
                    }) ?? [];
                },
                provideDefinition(document, position, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    return worker(token, () => {
                        const fileName = ctx.uriToFileName(document.uri);
                        const offset = document.offsetAt(position);
                        const info = (0, shared_1.safeCall)(() => ctx.languageService.getDefinitionAndBoundSpan(fileName, offset));
                        if (!info) {
                            return [];
                        }
                        return (0, lspConverters_1.convertDefinitionInfoAndBoundSpan)(info, document, ctx);
                    });
                },
                provideTypeDefinition(document, position, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    return worker(token, () => {
                        const fileName = ctx.uriToFileName(document.uri);
                        const offset = document.offsetAt(position);
                        const entries = (0, shared_1.safeCall)(() => ctx.languageService.getTypeDefinitionAtPosition(fileName, offset));
                        if (!entries) {
                            return [];
                        }
                        return entries.map(entry => (0, lspConverters_1.convertDocumentSpantoLocationLink)(entry, ctx));
                    });
                },
                async provideDiagnostics(document, token) {
                    return provideDiagnosticsWorker(document, token, 'syntactic');
                },
                async provideSemanticDiagnostics(document, token) {
                    return provideDiagnosticsWorker(document, token, 'semantic');
                },
                provideHover(document, position, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    return worker(token, () => {
                        const fileName = ctx.uriToFileName(document.uri);
                        const offset = document.offsetAt(position);
                        const info = (0, shared_1.safeCall)(() => ctx.languageService.getQuickInfoAtPosition(fileName, offset));
                        if (!info) {
                            return;
                        }
                        return (0, lspConverters_1.convertQuickInfo)(ts, info, document, ctx.fileNameToUri, ctx.getTextDocument);
                    });
                },
                provideImplementation(document, position, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    return worker(token, () => {
                        const fileName = ctx.uriToFileName(document.uri);
                        const offset = document.offsetAt(position);
                        const entries = (0, shared_1.safeCall)(() => ctx.languageService.getImplementationAtPosition(fileName, offset));
                        if (!entries) {
                            return [];
                        }
                        return entries.map(entry => (0, lspConverters_1.convertDocumentSpantoLocationLink)(entry, ctx));
                    });
                },
                provideReferences(document, position, referenceContext, token) {
                    if (!isSemanticDocument(document, true)) {
                        return;
                    }
                    return worker(token, () => {
                        const fileName = ctx.uriToFileName(document.uri);
                        const offset = document.offsetAt(position);
                        const references = (0, shared_1.safeCall)(() => ctx.languageService.findReferences(fileName, offset));
                        if (!references) {
                            return [];
                        }
                        const result = [];
                        for (const reference of references) {
                            if (referenceContext.includeDeclaration) {
                                const definition = (0, lspConverters_1.convertDocumentSpanToLocation)(reference.definition, ctx);
                                if (definition) {
                                    result.push(definition);
                                }
                            }
                            for (const referenceEntry of reference.references) {
                                const reference = (0, lspConverters_1.convertDocumentSpanToLocation)(referenceEntry, ctx);
                                if (reference) {
                                    result.push(reference);
                                }
                            }
                        }
                        return result;
                    });
                },
                provideFileReferences(document, token) {
                    if (!isSemanticDocument(document, true)) {
                        return;
                    }
                    return worker(token, () => {
                        const fileName = ctx.uriToFileName(document.uri);
                        const entries = (0, shared_1.safeCall)(() => ctx.languageService.getFileReferences(fileName));
                        if (!entries) {
                            return [];
                        }
                        return entries.map(entry => (0, lspConverters_1.convertDocumentSpanToLocation)(entry, ctx));
                    });
                },
                provideDocumentHighlights(document, position, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    return worker(token, () => {
                        const fileName = ctx.uriToFileName(document.uri);
                        const offset = document.offsetAt(position);
                        const highlights = (0, shared_1.safeCall)(() => ctx.languageService.getDocumentHighlights(fileName, offset, [fileName]));
                        if (!highlights) {
                            return [];
                        }
                        const results = [];
                        for (const highlight of highlights) {
                            for (const span of highlight.highlightSpans) {
                                results.push((0, lspConverters_1.convertHighlightSpan)(span, document));
                            }
                        }
                        return results;
                    });
                },
                provideDocumentSemanticTokens(document, range, legend, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    return worker(token, () => {
                        return getDocumentSemanticTokens(document, range, legend);
                    });
                },
                provideWorkspaceSymbols(query, token) {
                    return worker(token, () => {
                        const items = (0, shared_1.safeCall)(() => ctx.languageService.getNavigateToItems(query));
                        if (!items) {
                            return [];
                        }
                        return items
                            .filter(item => item.containerName || item.kind !== 'alias')
                            .map(item => (0, lspConverters_1.convertNavigateToItem)(item, ctx.getTextDocument(ctx.fileNameToUri(item.fileName))))
                            .filter(shared_1.notEmpty);
                    });
                },
                provideFileRenameEdits(oldUri, newUri, token) {
                    return worker(token, async () => {
                        const document = ctx.getTextDocument(oldUri);
                        const [formatOptions, preferences] = await Promise.all([
                            (0, getFormatCodeSettings_1.getFormatCodeSettings)(ctx, document, formattingOptions),
                            (0, getUserPreferences_1.getUserPreferences)(ctx, document),
                        ]);
                        const fileToRename = ctx.uriToFileName(oldUri);
                        const newFilePath = ctx.uriToFileName(newUri);
                        const response = (0, shared_1.safeCall)(() => ctx.languageService.getEditsForFileRename(fileToRename, newFilePath, formatOptions, preferences));
                        if (!response?.length) {
                            return;
                        }
                        return (0, lspConverters_1.convertFileTextChanges)(response, ctx.fileNameToUri, ctx.getTextDocument);
                    });
                },
                provideSelectionRanges(document, positions, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    return worker(token, () => {
                        return positions
                            .map(position => {
                            const fileName = ctx.uriToFileName(document.uri);
                            const offset = document.offsetAt(position);
                            const range = (0, shared_1.safeCall)(() => ctx.languageService.getSmartSelectionRange(fileName, offset));
                            if (!range) {
                                return;
                            }
                            return (0, lspConverters_1.convertSelectionRange)(range, document);
                        })
                            .filter(shared_1.notEmpty);
                    });
                },
                provideSignatureHelp(document, position, context, token) {
                    if (!isSemanticDocument(document)) {
                        return;
                    }
                    return worker(token, () => {
                        const options = {};
                        if (context?.triggerKind === 1) {
                            options.triggerReason = {
                                kind: 'invoked'
                            };
                        }
                        else if (context?.triggerKind === 2) {
                            options.triggerReason = {
                                kind: 'characterTyped',
                                triggerCharacter: context.triggerCharacter,
                            };
                        }
                        else if (context?.triggerKind === 3) {
                            options.triggerReason = {
                                kind: 'retrigger',
                                triggerCharacter: context.triggerCharacter,
                            };
                        }
                        const fileName = ctx.uriToFileName(document.uri);
                        const offset = document.offsetAt(position);
                        const helpItems = (0, shared_1.safeCall)(() => ctx.languageService.getSignatureHelpItems(fileName, offset, options));
                        if (!helpItems) {
                            return;
                        }
                        return {
                            activeSignature: helpItems.selectedItemIndex,
                            activeParameter: helpItems.argumentIndex,
                            signatures: helpItems.items.map(item => {
                                const signature = {
                                    label: '',
                                    documentation: undefined,
                                    parameters: []
                                };
                                signature.label += ts.displayPartsToString(item.prefixDisplayParts);
                                item.parameters.forEach((p, i, a) => {
                                    const label = ts.displayPartsToString(p.displayParts);
                                    const parameter = {
                                        label,
                                        documentation: ts.displayPartsToString(p.documentation)
                                    };
                                    signature.label += label;
                                    signature.parameters.push(parameter);
                                    if (i < a.length - 1) {
                                        signature.label += ts.displayPartsToString(item.separatorDisplayParts);
                                    }
                                });
                                signature.label += ts.displayPartsToString(item.suffixDisplayParts);
                                return signature;
                            }),
                        };
                    });
                },
            };
            async function provideDiagnosticsWorker(document, token, mode) {
                if (!isSemanticDocument(document)) {
                    return;
                }
                if (!await isValidationEnabled(document, context)) {
                    return;
                }
                return await worker(token, () => {
                    const fileName = ctx.uriToFileName(document.uri);
                    const program = ctx.languageService.getProgram();
                    const sourceFile = program?.getSourceFile(fileName);
                    if (!program || !sourceFile) {
                        return [];
                    }
                    const token = {
                        isCancellationRequested() {
                            return ctx.language.typescript?.languageServiceHost.getCancellationToken?.().isCancellationRequested() ?? false;
                        },
                        throwIfCancellationRequested() { },
                    };
                    if (mode === 'syntactic') {
                        const syntacticDiagnostics = (0, shared_1.safeCall)(() => program.getSyntacticDiagnostics(sourceFile, token)) ?? [];
                        const suggestionDiagnostics = (0, shared_1.safeCall)(() => ctx.languageService.getSuggestionDiagnostics(fileName)) ?? [];
                        return [...syntacticDiagnostics, ...suggestionDiagnostics]
                            .map(diagnostic => (0, lspConverters_1.convertDiagnostic)(diagnostic, document, ctx.fileNameToUri, ctx.getTextDocument))
                            .filter(shared_1.notEmpty);
                    }
                    else if (mode === 'semantic') {
                        const semanticDiagnostics = (0, shared_1.safeCall)(() => program.getSemanticDiagnostics(sourceFile, token)) ?? [];
                        const declarationDiagnostics = getEmitDeclarations(program.getCompilerOptions())
                            ? (0, shared_1.safeCall)(() => program.getDeclarationDiagnostics(sourceFile, token)) ?? []
                            : [];
                        return [...semanticDiagnostics, ...declarationDiagnostics]
                            .map(diagnostic => (0, lspConverters_1.convertDiagnostic)(diagnostic, document, ctx.fileNameToUri, ctx.getTextDocument))
                            .filter(shared_1.notEmpty);
                    }
                });
            }
            function getEmitDeclarations(compilerOptions) {
                return !!(compilerOptions.declaration || compilerOptions.composite);
            }
            function isSemanticDocument(document, withJson = false) {
                const virtualScript = getVirtualScriptByUri(document.uri);
                if (virtualScript) {
                    return true;
                }
                if (withJson && (0, shared_1.isJsonDocument)(document)) {
                    return true;
                }
                return (0, shared_1.isTsDocument)(document);
            }
            async function worker(token, fn) {
                let result;
                let oldSysVersion;
                let newSysVersion = await projectHost.syncSystem?.();
                do {
                    oldSysVersion = newSysVersion;
                    try {
                        result = await fn();
                    }
                    catch (err) {
                        console.warn(err);
                        break;
                    }
                    newSysVersion = await projectHost.syncSystem?.();
                } while (newSysVersion !== oldSysVersion && !token.isCancellationRequested);
                return result;
            }
            function getVirtualScriptByUri(uri) {
                const decoded = context.decodeEmbeddedDocumentUri(uri);
                const sourceScript = decoded && context.language.scripts.get(decoded[0]);
                const virtualCode = decoded && sourceScript?.generated?.embeddedCodes.get(decoded[1]);
                if (virtualCode && sourceScript?.generated?.languagePlugin.typescript) {
                    const { getServiceScript, getExtraServiceScripts } = sourceScript.generated?.languagePlugin.typescript;
                    const sourceFileName = context.env.typescript.uriToFileName(sourceScript.id);
                    if (getServiceScript(sourceScript.generated.root)?.code === virtualCode) {
                        return {
                            fileName: sourceFileName,
                            code: virtualCode,
                        };
                    }
                    for (const extraScript of getExtraServiceScripts?.(sourceFileName, sourceScript.generated.root) ?? []) {
                        if (extraScript.code === virtualCode) {
                            return extraScript;
                        }
                    }
                }
            }
        },
    };
}
exports.create = create;
function getBasicTriggerCharacters(tsVersion) {
    const triggerCharacters = ['.', '"', '\'', '`', '/', '<'];
    // https://github.com/microsoft/vscode/blob/8e65ae28d5fb8b3c931135da1a41edb9c80ae46f/extensions/typescript-language-features/src/languageFeatures/completions.ts#L811-L833
    if (semver.lt(tsVersion, '3.1.0') || semver.gte(tsVersion, '3.2.0')) {
        triggerCharacters.push('@');
    }
    if (semver.gte(tsVersion, '3.8.1')) {
        triggerCharacters.push('#');
    }
    if (semver.gte(tsVersion, '4.3.0')) {
        triggerCharacters.push(' ');
    }
    return triggerCharacters;
}
//# sourceMappingURL=semantic.js.map