"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdtDebugClient = void 0;
/*********************************************************************
 * Copyright (c) 2018 Ericsson and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
const cp = require("child_process");
const debugadapter_testsupport_1 = require("@vscode/debugadapter-testsupport");
const chai_1 = require("chai");
const chai = require("chai");
const deepEqualInAnyOrder = require("deep-equal-in-any-order");
chai.use(deepEqualInAnyOrder);
/**
 * Extend the DebugClient to support Reverse Requests:
 * https://microsoft.github.io/debug-adapter-protocol/specification#Reverse_Requests_RunInTerminal
 */
class CdtDebugClient extends debugadapter_testsupport_1.DebugClient {
    constructor() {
        super(...arguments);
        /**
         * Reverse Request Handlers:
         */
        this.reverseRequestHandlers = {
            runInTerminal: (args) => __awaiter(this, void 0, void 0, function* () {
                const process = yield new Promise((resolve, reject) => {
                    const child = cp.spawn(args.args[0], args.args.slice(1), {
                        cwd: args.cwd,
                        env: sanitizeEnv(args.env),
                    });
                    if (typeof child.pid !== 'undefined') {
                        return resolve(child);
                    }
                    child.once('error', (error) => {
                        reject(error);
                    });
                });
                return {
                    processId: process.pid,
                };
            }),
        };
    }
    /**
     * Notify the Debug Adapter by default that this client supports `runInTerminal`.
     */
    initializeRequest(args) {
        if (!args) {
            args = {
                supportsRunInTerminalRequest: true,
                adapterID: this['_debugType'],
                linesStartAt1: true,
                columnsStartAt1: true,
                pathFormat: 'path',
            };
        }
        return super.initializeRequest(args);
    }
    /**
     * Send a continueRequest and wait for target to stop
     */
    continue(args, reason, expected) {
        return __awaiter(this, void 0, void 0, function* () {
            const waitForStopped = this.assertStoppedLocation(reason, expected);
            const continueResp = this.continueRequest(args);
            yield Promise.all([waitForStopped, continueResp]);
            return waitForStopped;
        });
    }
    /**
     * Send a nextRequest and wait for target to stop
     */
    next(args, expected) {
        return __awaiter(this, void 0, void 0, function* () {
            const waitForStopped = this.assertStoppedLocation('step', expected);
            const next = this.nextRequest(args);
            yield Promise.all([waitForStopped, next]);
            return waitForStopped;
        });
    }
    /**
     * Send a stepInRequest and wait for target to stop
     */
    stepIn(args, expected) {
        return __awaiter(this, void 0, void 0, function* () {
            const waitForStopped = this.assertStoppedLocation('step', expected);
            const next = this.stepInRequest(args);
            yield Promise.all([waitForStopped, next]);
            return waitForStopped;
        });
    }
    /**
     * Send a stepOutRequest and wait for target to stop
     */
    stepOut(args, expected) {
        return __awaiter(this, void 0, void 0, function* () {
            const waitForStopped = this.assertStoppedLocation('step', expected);
            const next = this.stepOutRequest(args);
            yield Promise.all([waitForStopped, next]);
            return waitForStopped;
        });
    }
    /**
     * Send a stepBackRequest and wait for target to stop
     */
    stepBack(args, expected) {
        return __awaiter(this, void 0, void 0, function* () {
            const waitForStopped = this.assertStoppedLocation('step', expected);
            const next = this.stepBackRequest(args);
            yield Promise.all([waitForStopped, next]);
            return waitForStopped;
        });
    }
    /*
     * Returns a promise that will resolve if an output event with a specific category was received.
     * The promise will be rejected if a timeout occurs.
     */
    waitForOutputEvent(category, timeout = this.defaultTimeout) {
        return __awaiter(this, void 0, void 0, function* () {
            const isOutputEvent = (event) => {
                return (!!event.body &&
                    !!event.body.output);
            };
            const timer = setTimeout(() => {
                throw new Error(`no output event with category '${category}' received after ${timeout} ms`);
            }, timeout);
            for (;;) {
                const event = yield new Promise((resolve) => this.once('output', (e) => resolve(e)));
                if (!isOutputEvent(event)) {
                    continue;
                }
                if (event.body.category === category) {
                    clearTimeout(timer);
                    return event;
                }
            }
        });
    }
    /**
     * Send a response following a Debug Adapter Reverse Request.
     * @param request original request to respond to.
     * @param handler processes the request and returns the response body.
     */
    doRespond(request) {
        return __awaiter(this, void 0, void 0, function* () {
            const { command } = request;
            const handler = this['reverseRequestHandlers'][command];
            const response = {
                type: 'response',
                request_seq: request.seq,
                command,
                success: true,
            };
            if (!handler) {
                response.success = false;
                response.message = `Unknown command: ${command}`;
            }
            else {
                try {
                    response.body = yield handler(request.arguments);
                }
                catch (error) {
                    response.success = false;
                    response.message =
                        error instanceof Error ? error.message : String(error);
                }
            }
            const json = JSON.stringify(response);
            this['outputStream'].write(`Content-Length: ${Buffer.byteLength(json, 'utf-8')}\r\n\r\n${json}`);
        });
    }
    /*
     * Like assertStoppedLocation, but support multiple locations for multiple threads
     */
    assertStoppedLocations(reason, expected) {
        return Promise.all(expected.map((_l) => this.waitForEvent('stopped')))
            .then((events) => {
            events.forEach((event) => {
                (0, chai_1.expect)(event.body.reason).equals(reason);
            });
            return this.threadsRequest();
        })
            .then((threads) => Promise.all(threads.body.threads.map((thread) => {
            return this.stackTraceRequest({
                threadId: thread.id,
            });
        })).then((frames) => {
            const locations = frames.map((frame) => {
                var _a;
                return {
                    path: (_a = frame.body.stackFrames[0].source) === null || _a === void 0 ? void 0 : _a.path,
                    line: frame.body.stackFrames[0].line,
                };
            });
            (0, chai_1.expect)(locations).to.deep.equalInAnyOrder(expected);
        }));
    }
    /**
     * List hitBreakpoint, but supports multiple breakpoints that all have to be hit.
     *
     * XXX/TODO: Limitation each location needs to be a different path
     */
    hitBreakpoints(launchArgs, locations) {
        return Promise.all([
            this.waitForEvent('initialized')
                .then((_event) => {
                return Promise.all(locations.map((l) => this.setBreakpointsRequest({
                    source: { path: l.path },
                    breakpoints: [
                        {
                            line: l.line,
                        },
                    ],
                })));
            })
                .then((responses) => {
                responses.forEach((response, locationIndex) => {
                    response.body.breakpoints.forEach((bp) => {
                        (0, chai_1.expect)(bp.verified).to.be.true;
                        (0, chai_1.expect)(bp.line).equal(locations[locationIndex].line);
                    });
                });
                return this.configurationDoneRequest();
            }),
            this.launch(launchArgs),
            this.assertStoppedLocations('breakpoint', locations),
        ]);
    }
}
exports.CdtDebugClient = CdtDebugClient;
/**
 * DebugProtocol.dispatch is private, overriding manually.
 */
CdtDebugClient.prototype['dispatch'] = function dispatch(raw) {
    const message = JSON.parse(raw);
    if (isRequest(message)) {
        this['doRespond'](message);
    }
    else {
        debugadapter_testsupport_1.DebugClient.prototype['dispatch'].apply(this, [raw]);
    }
};
function isRequest(message) {
    return message.type === 'request';
}
function sanitizeEnv(env) {
    if (!env) {
        return {};
    }
    const sanitized = {};
    for (const key of Object.keys(env)) {
        if (typeof env[key] === 'string') {
            sanitized[key] = env[key];
        }
    }
    return sanitized;
}
//# sourceMappingURL=debugClient.js.map