"use strict";
/*********************************************************************
 * Copyright (c) 2019 Kichwa Coders and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.GDBTargetDebugSession = void 0;
const GDBDebugSession_1 = require("./GDBDebugSession");
const debugadapter_1 = require("@vscode/debugadapter");
const mi = require("./mi");
const os = require("os");
const child_process_1 = require("child_process");
const serialport_1 = require("serialport");
const net_1 = require("net");
const util_1 = require("./util");
class GDBTargetDebugSession extends GDBDebugSession_1.GDBDebugSession {
    constructor() {
        super(...arguments);
        this.killGdbServer = true;
    }
    attachOrLaunchRequest(response, request, args) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            this.setupCommonLoggerAndHandlers(args);
            if (request === 'launch') {
                const launchArgs = args;
                if (((_a = launchArgs.target) === null || _a === void 0 ? void 0 : _a.serverParameters) === undefined &&
                    !launchArgs.program) {
                    this.sendErrorResponse(response, 1, 'The program must be specified in the launch request arguments');
                    return;
                }
                yield this.startGDBServer(launchArgs);
            }
            yield this.startGDBAndAttachToTarget(response, args);
        });
    }
    launchRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const [request, resolvedArgs] = this.applyRequestArguments('launch', args);
                yield this.attachOrLaunchRequest(response, request, resolvedArgs);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    attachRequest(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const [request, resolvedArgs] = this.applyRequestArguments('attach', args);
                yield this.attachOrLaunchRequest(response, request, resolvedArgs);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    setupCommonLoggerAndHandlers(args) {
        debugadapter_1.logger.setup(args.verbose ? debugadapter_1.Logger.LogLevel.Verbose : debugadapter_1.Logger.LogLevel.Warn, args.logFile || false);
        this.gdb.on('consoleStreamOutput', (output, category) => {
            this.sendEvent(new debugadapter_1.OutputEvent(output, category));
        });
        this.gdb.on('execAsync', (resultClass, resultData) => this.handleGDBAsync(resultClass, resultData));
        this.gdb.on('notifyAsync', (resultClass, resultData) => this.handleGDBNotify(resultClass, resultData));
    }
    startGDBServer(args) {
        return __awaiter(this, void 0, void 0, function* () {
            if (args.target === undefined) {
                args.target = {};
            }
            const target = args.target;
            const serverExe = target.server !== undefined ? target.server : 'gdbserver';
            const serverCwd = target.cwd !== undefined ? target.cwd : (0, util_1.getGdbCwd)(args);
            const serverParams = target.serverParameters !== undefined
                ? target.serverParameters
                : ['--once', ':0', args.program];
            this.killGdbServer = target.automaticallyKillServer !== false;
            const gdbEnvironment = args.environment
                ? (0, util_1.createEnvValues)(process.env, args.environment)
                : process.env;
            const serverEnvironment = target.environment
                ? (0, util_1.createEnvValues)(gdbEnvironment, target.environment)
                : gdbEnvironment;
            // Wait until gdbserver is started and ready to receive connections.
            yield new Promise((resolve, reject) => {
                this.gdbserver = (0, child_process_1.spawn)(serverExe, serverParams, {
                    cwd: serverCwd,
                    env: serverEnvironment,
                });
                let gdbserverStartupResolved = false;
                let accumulatedStdout = '';
                let accumulatedStderr = '';
                let checkTargetPort = (_data) => {
                    // do nothing by default
                };
                if (target.port && target.serverParameters) {
                    setTimeout(() => {
                        gdbserverStartupResolved = true;
                        resolve();
                    }, target.serverStartupDelay !== undefined
                        ? target.serverStartupDelay
                        : 0);
                }
                else {
                    checkTargetPort = (data) => {
                        const regex = new RegExp(target.serverPortRegExp
                            ? target.serverPortRegExp
                            : 'Listening on port ([0-9]+)\r?\n');
                        const m = regex.exec(data);
                        if (m !== null) {
                            target.port = m[1];
                            checkTargetPort = (_data) => {
                                // do nothing now that we have our port
                            };
                            setTimeout(() => {
                                gdbserverStartupResolved = true;
                                resolve();
                            }, target.serverStartupDelay !== undefined
                                ? target.serverStartupDelay
                                : 0);
                        }
                    };
                }
                if (this.gdbserver.stdout) {
                    this.gdbserver.stdout.on('data', (data) => {
                        const out = data.toString();
                        if (!gdbserverStartupResolved) {
                            accumulatedStdout += out;
                        }
                        this.sendEvent(new debugadapter_1.OutputEvent(out, 'server'));
                        checkTargetPort(accumulatedStdout);
                    });
                }
                else {
                    throw new Error('Missing stdout in spawned gdbserver');
                }
                if (this.gdbserver.stderr) {
                    this.gdbserver.stderr.on('data', (data) => {
                        const err = data.toString();
                        if (!gdbserverStartupResolved) {
                            accumulatedStderr += err;
                        }
                        this.sendEvent(new debugadapter_1.OutputEvent(err, 'server'));
                        checkTargetPort(accumulatedStderr);
                    });
                }
                else {
                    throw new Error('Missing stderr in spawned gdbserver');
                }
                this.gdbserver.on('exit', (code, signal) => {
                    let exitmsg;
                    if (code === null) {
                        exitmsg = `${serverExe} is killed by signal ${signal}`;
                    }
                    else {
                        exitmsg = `${serverExe} has exited with code ${code}`;
                    }
                    this.sendEvent(new debugadapter_1.OutputEvent(exitmsg, 'server'));
                    if (!gdbserverStartupResolved) {
                        gdbserverStartupResolved = true;
                        reject(new Error(exitmsg + '\n' + accumulatedStderr));
                    }
                });
                this.gdbserver.on('error', (err) => {
                    const errmsg = `${serverExe} has hit error ${err}`;
                    this.sendEvent(new debugadapter_1.OutputEvent(errmsg, 'server'));
                    if (!gdbserverStartupResolved) {
                        gdbserverStartupResolved = true;
                        reject(new Error(errmsg + '\n' + accumulatedStderr));
                    }
                });
            });
        });
    }
    initializeUARTConnection(uart, host) {
        var _a, _b, _c, _d;
        if (uart.serialPort !== undefined) {
            // Set the path to the serial port
            this.serialPort = new serialport_1.SerialPort({
                path: uart.serialPort,
                // If the serial port path is defined, then so will the baud rate.
                baudRate: (_a = uart.baudRate) !== null && _a !== void 0 ? _a : 115200,
                // If the serial port path is deifned, then so will the number of data bits.
                dataBits: (_b = uart.characterSize) !== null && _b !== void 0 ? _b : 8,
                // If the serial port path is defined, then so will the number of stop bits.
                stopBits: (_c = uart.stopBits) !== null && _c !== void 0 ? _c : 1,
                // If the serial port path is defined, then so will the parity check type.
                parity: (_d = uart.parity) !== null && _d !== void 0 ? _d : 'none',
                // If the serial port path is defined, then so will the type of handshaking method.
                rtscts: uart.handshakingMethod === 'RTS/CTS' ? true : false,
                xon: uart.handshakingMethod === 'XON/XOFF' ? true : false,
                xoff: uart.handshakingMethod === 'XON/XOFF' ? true : false,
                autoOpen: false,
            });
            this.serialPort.on('open', () => {
                var _a;
                this.sendEvent(new debugadapter_1.OutputEvent(`listening on serial port ${(_a = this.serialPort) === null || _a === void 0 ? void 0 : _a.path}${os.EOL}`, 'Serial Port'));
            });
            const SerialUartParser = new serialport_1.ReadlineParser({
                delimiter: uart.eolCharacter === 'CRLF' ? '\r\n' : '\n',
                encoding: 'utf8',
            });
            this.serialPort
                .pipe(SerialUartParser)
                .on('data', (line) => {
                this.sendEvent(new debugadapter_1.OutputEvent(line + os.EOL, 'Serial Port'));
            });
            this.serialPort.on('close', () => {
                this.sendEvent(new debugadapter_1.OutputEvent(`closing serial port connection${os.EOL}`, 'Serial Port'));
            });
            this.serialPort.on('error', (err) => {
                this.sendEvent(new debugadapter_1.OutputEvent(`error on serial port connection${os.EOL} - ${err}`, 'Serial Port'));
            });
            this.serialPort.open();
        }
        else if (uart.socketPort !== undefined) {
            this.socket = new net_1.Socket();
            this.socket.setEncoding('utf-8');
            let tcpUartData = '';
            this.socket.on('data', (data) => {
                for (const char of data) {
                    if (char === '\n') {
                        this.sendEvent(new debugadapter_1.OutputEvent(tcpUartData + '\n', 'Socket'));
                        tcpUartData = '';
                    }
                    else {
                        tcpUartData += char;
                    }
                }
            });
            this.socket.on('close', () => {
                this.sendEvent(new debugadapter_1.OutputEvent(tcpUartData + os.EOL, 'Socket'));
                this.sendEvent(new debugadapter_1.OutputEvent(`closing socket connection${os.EOL}`, 'Socket'));
            });
            this.socket.on('error', (err) => {
                this.sendEvent(new debugadapter_1.OutputEvent(`error on socket connection${os.EOL} - ${err}`, 'Socket'));
            });
            this.socket.connect(
            // Putting a + (unary plus operator) infront of the string converts it to a number.
            +uart.socketPort, 
            // Default to localhost if target.host is undefined.
            host !== null && host !== void 0 ? host : 'localhost', () => {
                this.sendEvent(new debugadapter_1.OutputEvent(`listening on tcp port ${uart === null || uart === void 0 ? void 0 : uart.socketPort}${os.EOL}`, 'Socket'));
            });
        }
    }
    startGDBAndAttachToTarget(response, args) {
        return __awaiter(this, void 0, void 0, function* () {
            if (args.target === undefined) {
                args.target = {};
            }
            const target = args.target;
            try {
                this.isAttach = true;
                yield this.spawn(args);
                yield this.gdb.sendFileExecAndSymbols(args.program);
                yield this.gdb.sendEnablePrettyPrint();
                if (args.imageAndSymbols) {
                    if (args.imageAndSymbols.symbolFileName) {
                        if (args.imageAndSymbols.symbolOffset) {
                            yield this.gdb.sendAddSymbolFile(args.imageAndSymbols.symbolFileName, args.imageAndSymbols.symbolOffset);
                        }
                        else {
                            yield this.gdb.sendFileSymbolFile(args.imageAndSymbols.symbolFileName);
                        }
                    }
                }
                if (target.connectCommands === undefined) {
                    this.targetType =
                        target.type !== undefined ? target.type : 'remote';
                    let defaultTarget;
                    if (target.port !== undefined) {
                        defaultTarget = [
                            target.host !== undefined
                                ? `${target.host}:${target.port}`
                                : `localhost:${target.port}`,
                        ];
                    }
                    else {
                        defaultTarget = [];
                    }
                    const targetParameters = target.parameters !== undefined
                        ? target.parameters
                        : defaultTarget;
                    yield mi.sendTargetSelectRequest(this.gdb, {
                        type: this.targetType,
                        parameters: targetParameters,
                    });
                    this.sendEvent(new debugadapter_1.OutputEvent(`connected to ${this.targetType} target ${targetParameters.join(' ')}`));
                }
                else {
                    yield this.gdb.sendCommands(target.connectCommands);
                    this.sendEvent(new debugadapter_1.OutputEvent('connected to target using provided connectCommands'));
                }
                yield this.gdb.sendCommands(args.initCommands);
                if (target.uart !== undefined) {
                    this.initializeUARTConnection(target.uart, target.host);
                }
                if (args.imageAndSymbols) {
                    if (args.imageAndSymbols.imageFileName) {
                        yield this.gdb.sendLoad(args.imageAndSymbols.imageFileName, args.imageAndSymbols.imageOffset);
                    }
                }
                yield this.gdb.sendCommands(args.preRunCommands);
                this.sendEvent(new debugadapter_1.InitializedEvent());
                this.sendResponse(response);
                this.isInitialized = true;
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
    stopGDBServer() {
        return __awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => {
                var _a;
                if (!this.gdbserver || this.gdbserver.exitCode !== null) {
                    resolve();
                }
                else {
                    this.gdbserver.on('exit', () => {
                        resolve();
                    });
                    (_a = this.gdbserver) === null || _a === void 0 ? void 0 : _a.kill();
                }
                setTimeout(() => {
                    reject();
                }, 1000);
            });
        });
    }
    disconnectRequest(response, _args) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                if (this.serialPort !== undefined && this.serialPort.isOpen)
                    this.serialPort.close();
                if (this.targetType === 'remote') {
                    if (this.gdb.getAsyncMode() && this.isRunning) {
                        // See #295 - this use of "then" is to try to slightly delay the
                        // call to disconnect. A proper solution that waits for the
                        // interrupt to be successful is needed to avoid future
                        // "Cannot execute this command while the target is running"
                        // errors
                        this.gdb
                            .sendCommand('interrupt')
                            .then(() => this.gdb.sendCommand('disconnect'));
                    }
                    else {
                        yield this.gdb.sendCommand('disconnect');
                    }
                }
                yield this.gdb.sendGDBExit();
                if (this.killGdbServer) {
                    yield this.stopGDBServer();
                    this.sendEvent(new debugadapter_1.OutputEvent('gdbserver stopped', 'server'));
                }
                this.sendResponse(response);
            }
            catch (err) {
                this.sendErrorResponse(response, 1, err instanceof Error ? err.message : String(err));
            }
        });
    }
}
exports.GDBTargetDebugSession = GDBTargetDebugSession;
//# sourceMappingURL=GDBTargetDebugSession.js.map