"use strict";
/********************************************************************************
 * Copyright (c) 2021 STMicroelectronics and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 *******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.forAllProjects = exports.listProjects = exports.listContexts = exports.selectContext = exports.unsetCompileFlags = exports.setCompileFlags = exports.getContext = exports.setContext = void 0;
const path = require("path");
const clangd_config_1 = require("./clangd-config");
const file_util_1 = require("./util/file-util");
const maybe_array_1 = require("./util/maybe-array");
/**
 * Set the context selected in a project. The `.clangd` file will be updated to
 * refer to the given context's compilation database.
 *
 * @param configurationInput the clangd configuration to update, either the `.clangd` file's path or its content
 * @param contextDirectory the directory that contains the compilation database file
 * @throws {@link ClangdConfigException} on error in saving the updated configuration file
 */
function setContext(configurationInput, contextDirectory) {
    const configuration = typeof configurationInput === 'string' ? clangd_config_1.loadOrCreate(configurationInput) : configurationInput;
    configuration.CompileFlags.CompilationDatabase = contextDirectory;
    clangd_config_1.save(configuration);
}
exports.setContext = setContext;
/**
 * Retrieve the active context of the given `.clangd` configuration file.
 *
 * @param configPath the clangd context file to read
 * @return the context-relevant content of the file, or `undefined` if the file does not exist
 * @throws {@link ClangdConfigException} on error in loading an existing configuration file
 */
function getContext(configPath) {
    const config = clangd_config_1.load(configPath);
    if (config && config.CompileFlags.CompilationDatabase) {
        return {
            name: path.relative(path.dirname(config.filePath), config.CompileFlags.CompilationDatabase),
            compilationDatabase: config.CompileFlags.CompilationDatabase,
        };
    }
    return undefined;
}
exports.getContext = getContext;
/**
 * In the given `.clangd` file, specify flags to add to and/or remove from invocations of the `clang` compiler
 * in clangd's analysis.
 *
 * @param configurationInput the clangd configuration to update, either the `.clangd` file path or its content
 * @param addFlags an optional array of flags to add to the list of flags added to the `clang` command-line
 * @param removeFlags an optional array of flags to add to the list of flags removed from the `clang` command-line
 * @throws {@link ClangdConfigException} on error in saving the updated configuration file
 */
function setCompileFlags(configurationInput, addFlags, removeFlags) {
    const configuration = typeof configurationInput === 'string' ? clangd_config_1.loadOrCreate(configurationInput) : configurationInput;
    if (addFlags) {
        configuration.CompileFlags.Add = includeFlags(addFlags, configuration.CompileFlags.Add);
    }
    if (removeFlags) {
        configuration.CompileFlags.Remove = includeFlags(removeFlags, configuration.CompileFlags.Remove);
    }
    clangd_config_1.save(configuration);
}
exports.setCompileFlags = setCompileFlags;
/**
 * Remove from the given `.clangd` file some flags that are currently added or removed from invocations of the `clang` compiler
 * in clangd's analysis. In the `.clangd` file, any given flag will generally be either in the `Add` list or the `Remove` list.
 * This function ensures that the flags specified will not appear in either list in the updated `.clangd` file.
 *
 * @param configurationInput the clangd configuration to update, either the `.clangd` file path or its content
 * @param flags an optional array of flags to remove from the lists of flags added to and removed from the `clang` command-line
 * @throws {@link ClangdConfigException} on error in saving the updated configuration file
 */
function unsetCompileFlags(configurationInput, flags) {
    const configuration = typeof configurationInput === 'string' ? clangd_config_1.loadOrCreate(configurationInput) : configurationInput;
    if (flags && configuration.CompileFlags.Add) {
        configuration.CompileFlags.Add = excludeFlags(flags, configuration.CompileFlags.Add);
        if (configuration.CompileFlags.Add.length === 0) {
            delete configuration.CompileFlags.Add;
        }
    }
    if (flags && configuration.CompileFlags.Remove) {
        configuration.CompileFlags.Remove = excludeFlags(flags, configuration.CompileFlags.Remove);
        if (configuration.CompileFlags.Remove.length === 0) {
            delete configuration.CompileFlags.Remove;
        }
    }
    clangd_config_1.save(configuration);
}
exports.unsetCompileFlags = unsetCompileFlags;
/**
 * Ensure that an array of flags includes some required flags.
 *
 * @param newFlags flags to ensure are included along with the `currentFlags`
 * @param currentFlags an existing single flag or array of flags
 * @returns a new array based on the `currentFlags` that includes the `newFlags`
 */
function includeFlags(newFlags, currentFlags) {
    const base = maybe_array_1.asArray(currentFlags !== null && currentFlags !== void 0 ? currentFlags : []);
    return Array.from(new Set(base.concat(newFlags)));
}
/**
 * Ensure that an array of flags does not include some unwanted flags.
 *
 * @param flagsToRemove flags to ensure are excluded from the array of `currentFlags`
 * @param currentFlags an existing single flag or array of flags
 * @returns a new array based on the `currentFlags` that does not include any of the `flagsToRemove`
 */
function excludeFlags(flagsToRemove, currentFlags) {
    const result = new Set(maybe_array_1.asArray(currentFlags !== null && currentFlags !== void 0 ? currentFlags : []));
    flagsToRemove.forEach(flag => result.delete(flag));
    return Array.from(result);
}
/**
 * Activate a named context in all of the projects that have a context of that name.
 * Any project that does not have the context is left unchanged.
 *
 * @param contextsConfig the projects from the `.clangd-contexts` file
 * @param contextName the context to active in the projects
 */
function selectContext(contextsConfig, contextName) {
    contextsConfig.projects.forEach(proj => {
        const configDir = contextsConfig.toContextDir(proj, contextName);
        // Not necessarily every project actually has this context defined
        if (file_util_1.isDirectory(configDir)) {
            setContext(contextsConfig.getClangdConfig(proj), configDir);
        }
    });
}
exports.selectContext = selectContext;
/**
 * List the distinct context names across all projects.
 *
 * @param contextsConfig the clangd workspace configuration from the `.clangd-contexts` file
 */
function listContexts(contextsConfig) {
    contextsConfig.getClangdContexts().forEach(context => console.log(context));
}
exports.listContexts = listContexts;
/**
 * List the projects in the workspace.
 *
 * @param contextsConfig the clangd workspace configuration from the `.clangd-contexts` file
 * @param baseDir a directory relative to which the output directory paths are deresolved. For example, in a
 *   CLI tool this might be the current working directory. If omitted, absolute paths are output
 */
function listProjects(contextsConfig, baseDir) {
    const deresolve = baseDir ? p => {
        const outputPath = path.relative(baseDir, p);
        const normalizedOutputPath = outputPath === '' ? '.' : outputPath;
        return normalizedOutputPath + path.sep;
    } : p => path.resolve(contextsConfig.path, p);
    forAllProjects(contextsConfig, proj => {
        const projectPath = typeof proj === 'string' ? proj : path.dirname(proj.filePath);
        const outputPath = deresolve(projectPath);
        console.log(outputPath);
    });
}
exports.listProjects = listProjects;
/**
 * Perform some action on all valid project directories.
 *
 * @param contextsConfig the clangd workspace configuration from the `.clangd-contexts` file
 * @param action an action to perform on each project directory
 */
function forAllProjects(contextsConfig, action) {
    contextsConfig.getClangdProjects().forEach(project => {
        if (file_util_1.isDirectory(project)) {
            action(project);
        }
    });
}
exports.forAllProjects = forAllProjects;
//# sourceMappingURL=clangd-context.js.map