# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import gettext
import optparse
import os
import sys

_ = gettext.gettext

PYGTK_REQUIRED = (2, 6, 0)

def open_project(app, filename, profile):
    if not profile:
        return app.open_project(filename)

    print 'profiling'
    import hotshot
    from hotshot import stats
    prof = hotshot.Profile("gazpacho.prof")
    retval = prof.runcall(app.open_project, filename)
    prof.close()

    s = stats.load("gazpacho.prof")
    s.strip_dirs()
    s.sort_stats('time', 'calls')
    s.print_stats(25)

def run_batch(app, command):
    ns = dict(new=lambda w: app.create(w))
    eval(command, {}, ns)

def show_debug():
    print 'Loading PyGTK...'
    import gobject
    import gtk
    if (os.path.dirname(os.path.dirname(gtk.__file__)) !=
        os.path.dirname(gobject.__file__)):
        print ('WARNING: Gtk and GObject modules are not loaded from '
               'the same prefix')
        
    print 'Python:\t', '.'.join(map(str, sys.version_info[:3]))
    print 'GTK:\t', '.'.join(map(str, gtk.gtk_version))
    print 'PyGTK:\t', '.'.join(map(str, gtk.pygtk_version))
    
def launch(filenames=[], debug=False, profile=False,
           batch=None, update=None):
    if debug:
        show_debug()

    import gtk
    if gtk.pygtk_version < PYGTK_REQUIRED:
        raise SystemExit("PyGTK 2.6.0 or higher required to run Gazpacho")

    # Ignore deprecation warnings when using 2.7.0
    if gtk.pygtk_version >= (2, 7, 0):
        if gtk.pygtk_version in ((2, 7, 0), (2, 7, 1)):
            raise SystemExit("PyGtk 2.7.0 and 2.7.1 are buggy "
                             "Upgrade to 2.7.2 or downgrade to 2.6.x")
        
        if debug:
            print 'Using PyGTK 2.7.x, ignoring deprecation warnings'
        import warnings
        warnings.filterwarnings('ignore', category=DeprecationWarning)

    # Delay imports, so command line parsing is not slowed down
    from gazpacho.application import Application
    from gazpacho.debugwindow import DebugWindow, show
    
    gazpacho = Application()

    if not debug:
        DebugWindow.application = gazpacho
        sys.excepthook = show

    for filename in filenames:
        if not os.path.exists(filename):
            raise SystemExit('%s: no such a file or directory' % filename)
        
        if not os.access(filename, os.R_OK):
            raise SystemExit('Could not open file %s: Permission denied.' %
                             filename)
        open_project(gazpacho, filename, profile)

    if update:
        for project in gazpacho.get_projects():
            project.save(project.path)

        return
    
    # If no filenames were specified, open up an empty project
    if not filenames:
        gazpacho.new_project()
    
    if batch:
        run_batch(gazpacho, batch)
        return

    gazpacho.show_all()

    gtk.main()

def main(args=[]):
    from gazpacho.environ import environ
    gettext.bindtextdomain('gazpacho', environ.get_languages_dir())
    
    parser = optparse.OptionParser()
    parser.add_option('', '--profile',
                      action="store_true", 
                      dest="profile",
                      help=_("Turn on profiling support"))
    parser.add_option('', '--debug',
                      action="store_true", 
                      dest="debug",
                      help=_("Turn on pdb debugging support"))
    parser.add_option('', '--batch',
                      action="store", 
                      dest="batch",
                      help=_("Batch command"))
    parser.add_option('-u', '--update',
                      action="store_true", 
                      dest="update",
                      help=_("Load glade file and save it"))
    
    options, args = parser.parse_args(args)

    if options.batch:
        options.debug = True
        
    filenames = []
    if len(args) >= 2:
        filenames = [os.path.abspath(name) for name in args[1:]]

    launch(filenames, options.debug, options.profile,
           options.batch, options.update)
