# -*- coding: utf-8 -*-

# Copyright (C) 2004-2006 Johan Svedberg <johan@svedberg.com>

# This file is part of ontv.

# ontv is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# ontv is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with ontv; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os.path
import sys
import thread
from gettext import gettext as _

import gtk
import gobject
import xmltv

from Listings import Listings
from GUI import ErrorDialog
from Channel import Channel
from Program import Program

class XMLTVFile(gobject.GObject):
    __gproperties__ = {"path": (str, "XMLTV file path",
                       "The path to the XMLTV file", "",
                       gobject.PARAM_READWRITE)}

    __gsignals__ = {"loaded": (gobject.SIGNAL_RUN_LAST, None, (object,)),
                    "loading": (gobject.SIGNAL_RUN_LAST, None, ())}

    def __init__(self, config):
        gobject.GObject.__init__(self)
        self.config = config
        self.listings = Listings(config)

        self.__channel_names = {}

    def do_get_property(self, property):
        if property.name == "path":
            return self.path

    def do_set_property(self, property, value):
        if property.name == "path":
            self.path = value
            self.load()

    def load(self, uicomponent=None, verb=None):
        thread.start_new_thread(self.__load_in_thread, (verb,))

    def __load_in_thread(self, force_reload):
        if os.path.exists(self.listings.file) and not force_reload:
            self.listings = self.listings.load()
        if self.has_changed() or force_reload:
            gtk.threads_enter()
            self.emit("loading")
            gtk.threads_leave()

            title = error_msg = ""
            try:
                self.__load_channels()
                self.__load_programs()
                self.listings.mtime = os.path.getmtime(self.path)
                if self.config.debug:
                    print("Loaded listings from XMLTV file: %s" % self.path)
            except IOError, ioe:
                title = _("Failed to load ") + self.path
                if ioe.errno == 2:
                    error_msg = _("File not found")
                elif ioe.errno == 13:
                    error_msg = _("Access denied")
                else:
                    error_msg = _("Unknown error")
            except Exception, e:
                if e.__module__ == "xml.parsers.expat":
                    title = _("Error while parsing ") + self.path
                    error_msg = _("Not well formed at line %s, column %s") % \
                                (e.lineno, e.offset)

            if title and error_msg:
                gtk.threads_enter()
                ed = ErrorDialog(title, error_msg)
                ed.run()
                ed.destroy()
                gtk.threads_leave()
                self.emit("loaded", (None))
                return

            for channel in self.listings.channels.values():
                channel.programs.sort()

            self.listings.save()

        gtk.threads_enter()
        self.emit("loaded", (self.listings))
        gtk.threads_leave()

    def has_changed(self):
        if os.path.exists(self.path):
            return os.path.getmtime(self.path) != self.listings.mtime
        return True

    def __load_channels(self):
        for channel in xmltv.read_channels(open(self.path)):
            c = Channel(channel)
            self.__channel_names[c.id] = c.name
            self.listings.add_channel(c)
            if self.config.debug:
                print("Added channel \"%s\" to Listings." % c.name)

    def __load_programs(self):
        for program in xmltv.read_programmes(open(self.path)):
            error_msg = ""
            if not program.has_key("title"):
                continue
            elif not program.has_key("start"):
                error_msg = "missing start time"
            elif not program.has_key("stop"):
                error_msg = "missing stop time"

            if error_msg:
                print >> sys.stderr, \
                      "Unable to add program \"%s\" to Listings: %s" % \
                      (program["title"][0][0], error_msg)
                continue

            channel_name = self.__get_channel_name(program)
            p = Program(program)
            self.listings.add_program(p, channel_name)
            if self.config.debug:
                print("Added program \"%s\" to Listings." % p.title)

    def __get_channel_name(self, program):
        return self.__channel_names[program["channel"]]

# vim: set sw=4 et sts=4 tw=79 fo+=l:
