/* export.c -  encrypt functions
 *	Copyright (C) 2000 Werner Koch (dd9jn), g10 Code GmbH
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "util.h"
#include "context.h"
#include "ops.h"


static void
export_status_handler (gpgme_ctx_t ctx, gpg_status_code_t code, char *args)
{
    DEBUG2 ("export_status: code=%d args=`%s'\n", code, args );
    /* FIXME: Need to do more */
}


static gpgme_error_t
export_start (gpgme_ctx_t ctx, gpgme_recipients_t recp, gpgme_data_t keydata)
{
    struct user_id_s *u;
    gpgme_error_t rc = 0;
    size_t nkeys = 0;

    fail_on_pending_request( ctx );
    ctx->pending = 1;

    /* create a process object */
    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        goto leave;

    if( ctx->use_logging )
	_gpgme_gpg_set_logging_handler( ctx->gpg, ctx );
    _gpgme_gpg_set_status_handler ( ctx->gpg, export_status_handler, ctx );

    /* build the commandline */
    _gpgme_add_comment (ctx);
    if (ctx->use_armor)
        _gpgme_gpg_add_arg( ctx->gpg, "--armor" );
    _gpgme_gpg_add_arg( ctx->gpg, ctx->with_secret_key? "--export-secret-keys"
						      : "--export" );
    if( !keydata || gpgme_data_get_type( keydata ) != GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( Invalid_Value );
        goto leave;
    }
    _gpgme_data_set_mode( keydata, GPGME_DATA_MODE_IN );
    _gpgme_gpg_add_data( ctx->gpg, keydata, 1 );
    _gpgme_gpg_add_arg( ctx->gpg, "--" );

    for( u = recp->list; u; u = u->next ) {
	_gpgme_gpg_add_arg( ctx->gpg, u->name );
	nkeys++;
    }
    if( nkeys > 1 && ctx->with_secret_key ) {
    /* If we export the secret key, it's not a good idea to do this
       when we also export other keys. In this mode, it's only allowed
       to export one public and one private key. */
	rc = mk_error( Invalid_Mode );
	goto leave;
    }
    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );

 leave:
    if( rc ) {
        ctx->pending = 0; 
        _gpgme_gpg_release( &ctx->gpg );
    }
    return rc;
}


static gpgme_error_t
file_export_start( gpgme_ctx_t ctx, gpgme_recipients_t recp, const char *output )
{
    struct user_id_s *u;
    gpgme_error_t rc;
      
    if( !output )
	return mk_error( Invalid_Value );

    fail_on_pending_request( ctx );
    ctx->pending = 1;
    
    _gpgme_gpg_release( &ctx->gpg );
    rc =_gpgme_gpg_new( &ctx->gpg );
    if( rc )
        goto leave;
    
    _gpgme_gpg_add_arg (ctx->gpg, "--yes");
    _gpgme_gpg_add_arg (ctx->gpg, "--export");
    if (ctx->use_armor)
        _gpgme_gpg_add_arg( ctx->gpg, "--armor" );
    _gpgme_add_comment (ctx);
    _gpgme_gpg_add_arg (ctx->gpg, "--output");
    _gpgme_gpg_add_arg (ctx->gpg, output);
    for (u = recp->list; u; u = u->next)
	_gpgme_gpg_add_arg (ctx->gpg, u->name);
    
    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    
leave:
    if( rc ) {
        ctx->pending = 0; 
        _gpgme_gpg_release( &ctx->gpg );
    }
    return rc;
} /* file_export_start */


/**
 * gpgme_op_export:
 * @c: the context
 * @recp: a list of recipients or NULL
 * @keydata: Returns the keys
 * 
 * This function can be used to extract public keys from the GnuPG key
 * database either in armored (by using gpgme_set_armor()) or in plain
 * binary form.  The function expects a list of user IDs in @recp for
 * whom the public keys are to be exportedkinit
 *
 * 
 * Return value: 0 for success or an error code
 **/
gpgme_error_t
gpgme_op_export( gpgme_ctx_t ctx, gpgme_recipients_t recp, gpgme_data_t keydata )
{
    gpgme_error_t err;
    
    err = export_start( ctx, recp, keydata );
    if( !err ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
	if( gpgme_data_get_type( keydata ) == GPGME_DATA_TYPE_NONE )
	    err = mk_error( General_Error );
	else if( gpgme_get_process_rc( ctx ) )
	    err = mk_error( Internal_GPG_Problem );
    }
    return err;
} /* gpgme_op_export */


gpgme_error_t
gpgme_op_file_export( gpgme_ctx_t ctx, gpgme_recipients_t recp, const char *output )
{	
    gpgme_error_t err;
        
    err = file_export_start( ctx, recp, output );
    if( !err ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
	if( gpgme_get_process_rc( ctx ) )
	    err = mk_error( Internal_GPG_Problem );
    }
    
    return err;
} /* gpgme_op_file_export */


gpgme_error_t
gpgme_op_clip_export (gpgme_recipients_t rset)
{
    gpgme_error_t err = 0;
    gpgme_ctx_t ctx = NULL;
    gpgme_data_t keydata = NULL;
    
    err = gpgme_new (&ctx);
    if (!err)
	gpgme_control( ctx, GPGME_CTRL_ARMOR, 1 );
    if (!err)
	err = gpgme_data_new (&keydata);
    if( !err )        
	err = gpgme_op_export (ctx, rset, keydata);
    if (!err) {
	gpgme_data_change_version (&keydata);
	gpgme_data_release_and_set_clipboard (keydata);
    }
    gpgme_release (ctx);
    return err;
} /* gpgme_op_clip_export */