/* wptSecmem.cpp - W32 ``Secure`` memory handling
 *	Copyright (C) 2001, 2002 Timo Schulz <twoaday@freakmail.de>
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 *
 * This is the notice from the original author:
 *
 * win32vmSafe - a 'no write to disk' volatile buffer
 * implemented via standard win32 api calls.
 *
 * Copyright (C) 2001 John J. Kane <jkane89@softhome.net>
 * Brief notice: This code is released under the terms of
 * the GNU Lesser General Public License (LGPL) Version 2.1;
 * refer to www.gnu.org if you're clueless.
 *
 * WARNING:
 * This is win32vmSafe ALPHA v0.2.1. No security analysis has
 * been done on this version; do not use in production code.
 * It demonstrates basic functionality only.
 * This version should be considered deprecated after 2001-09-01.
 * Contact John Kane for information on the current status
 * of this package.
 *
 * Heavily modified for the use with WinPT by Timo Schulz.
 */

#if 0
#include <windows.h>

/*  CONTEXT:
  Under Windows, a user process cannot lock pages in memory.
Normally, critical data will be flushed to the system VMM swap
file at unpredictable intervals.  Only a kernel-level VXD/DLL
can lock physical pages, and we do not have a GPL page-locker.
*HOWEVER:
Please note the following distinction:
  (1) A page can be locked IN memory to guarantee that it is
always accessible and never triggers a page fault.
  (2) A page can be protected to ensure that its contents
are never copied OUT of memory into permanent storage.
  It turns out that there is a way to implement (2) without
the work required to ensure (1).  Win32 memory-mapped files
allow you to map the first N bytes of any file into a page.
The remaining 4096-N bytes of that page become a VOLATILE
BUFFER area which is not written back to the mapped file even
if the process is swapped out.  The data is DISCARDED without
being written; when the process is swapped back in, the former
contents of the unmapped area are not preserved.
  A 'magic number', sentinel, or any other integrity check
scheme can be used to detect the loss of the contents of
the volatile buffer.
  The application can respond by aborting the interrupted
action with an error message, or by retrying.
*SUMMARY:
  This approach DOES satisfy a security constraint of not
allowing critical key data to be written to disk.  It does
not attempt to prevent the process from being swapped, and
does not preserve the volatile buffer across swap-outs.
This behavior (although safe) is non-intuitive; it may be
best to make it a non-default option if implemented.
*COMPLEXITY IMPLICATIONS FOR NON-WINDOWS PORTS:
  None.
*COMPLEXITY IMPLICATIONS FOR WINDOWS PORTS:
  The code footprint is small.  No VXD; it's compiled
right into the main executable.  No end-user installation
step.  Data in excess of 4k requires multiple buffers.
May allow the elimination of certain previously-implemented
win32 security workarounds.  The parent routine must detect
and handle the (rare) loss-of-data event.
*PORTABILITY:
  There are reasons why this method is likely to be portable
across all flavors of win32 systems, even in environments
where the user lacks NT Administrator rights.  It relies
only on core win32 primitives which are supported on
everything since Win95 (possibly Win3.1).
*/

struct secmem_s {
    HANDLE fh;
    HANDLE mh;
    char *ptr;
    int mapsize;
};

static secmem_s mem;
static char fname[128];
static int fname_set = 0;

int
vm_safe_init(int mapsize)
{
    if ( !fname_set )
        return 1;
    
    mem.fh = CreateFile(
        fname,
        GENERIC_READ | GENERIC_WRITE,
        FILE_SHARE_READ | FILE_SHARE_WRITE,
        NULL,
        OPEN_ALWAYS,
        FILE_ATTRIBUTE_TEMPORARY, 
        NULL );
    if ( mem.fh == INVALID_HANDLE_VALUE )
        return 1;
    
    mem.mh = CreateFileMapping( mem.fh, NULL, PAGE_READWRITE, 0, mapsize, 
                                NULL );
    mem.ptr = (char *)MapViewOfFile( mem.mh, FILE_MAP_READ | FILE_MAP_WRITE,
                                     0, 0, 0 );
    if ( mem.ptr == NULL ) {
        CloseHandle( mem.fh );
        CloseHandle( mem.mh );
        return 1;
    }
    
    VirtualLock( mem.ptr, mapsize );
    mem.mapsize = mapsize;
    
    return 0;
} /* vm_safe_init */

int
vm_safe_init_noswap(void)
{
    int rc;
    
    rc = vm_safe_init( 512 );
    if ( rc )
        return 1;
    mem.ptr += 512;
    return 0;
} /* vm_safe_init_noswap */

void
vm_safe_free( void )
{
    memset( mem.ptr, 0, mem.mapsize );
    mem.ptr = NULL;
    mem.mapsize = 0;
    
    CloseHandle( mem.mh );
    mem.mh = NULL;
    CloseHandle( mem.fh );
    mem.fh = INVALID_HANDLE_VALUE;
} /* vm_safe_init */

void*
vm_safe_get( void )
{
	return mem.ptr;
} /* vm_safe_get */

void
vm_safe_set_swapfile( const char *swapname )
{
    if ( swapname ) {
        strcpy( fname, swapname );
        fname_set = 1;
    }
} /* vm_safe_set_swapfile */

int
vm_safe_create_swapfile( const char *swapname, int size )
{
    HANDLE fd;
    char *buf = NULL;
    DWORD nwritten;
    
    if ( !size )
        return 1;
    
    fd = CreateFile( swapname, GENERIC_WRITE, FILE_SHARE_WRITE, NULL, 
                     CREATE_ALWAYS,	0, NULL );
    if ( fd == INVALID_HANDLE_VALUE )
        return 1;
    
    buf = new char[size+1];
    if( !buf )
        BUG( NULL );
    memset( buf, 0, size );
    WriteFile( fd, buf, size, &nwritten, NULL );
    
    CloseHandle( fd );
    free_if_alloc( buf );
    return 0;
} /* vm_safe_create_swapfile */
#endif
