/*
   Copyright 2012 Sebastian Trueg <trueg@kde.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License or (at your option) version 3 or any later version
   accepted by the membership of KDE e.V. (or its successor approved
   by the membership of KDE e.V.), which shall act as a proxy
   defined in Section 14 of version 3 of the license.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "kio_music.h"

#include <Nepomuk/ResourceManager>
#include <Nepomuk/Vocabulary/NMM>
#include <Nepomuk/Vocabulary/NIE>
#include <Nepomuk/Vocabulary/NFO>
#include <Nepomuk/Query/FileQuery>
#include <Nepomuk/Query/ComparisonTerm>
#include <Nepomuk/Query/ResourceTypeTerm>
#include <Nepomuk/Query/LiteralTerm>
#include <Nepomuk/Query/AndTerm>

#include <KUrl>
#include <kio/global.h>
#include <klocale.h>
#include <kio/job.h>
#include <KUser>
#include <KDebug>
#include <KLocale>
#include <KComponentData>

#include <Soprano/Vocabulary/NAO>
#include <Soprano/QueryResultIterator>
#include <Soprano/Model>
#include <Soprano/Node>
#include <Soprano/LiteralValue>

#include <QtCore/QDate>
#include <QtCore/QCoreApplication>
#include <QStringList>

using namespace KIO;
using namespace Nepomuk::Vocabulary;
using namespace Soprano::Vocabulary;


namespace {
    KIO::UDSEntry createFolderUDSEntry( const QString& name, const QString& displayName, const KUrl& nepomukUri = KUrl() )
    {
        KIO::UDSEntry uds;
        uds.insert( KIO::UDSEntry::UDS_NAME, name );
        uds.insert( KIO::UDSEntry::UDS_DISPLAY_NAME, displayName );
        uds.insert( KIO::UDSEntry::UDS_FILE_TYPE, S_IFDIR );
        uds.insert( KIO::UDSEntry::UDS_MIME_TYPE, QString::fromLatin1( "inode/directory" ) );
        uds.insert( KIO::UDSEntry::UDS_ACCESS, 0700 );
        uds.insert( KIO::UDSEntry::UDS_USER, KUser().loginName() );
        if(!nepomukUri.isEmpty()) {
            uds.insert( KIO::UDSEntry::UDS_NEPOMUK_URI, nepomukUri.url() );
        }
        return uds;
    }

    KIO::UDSEntry createMusicPieceUDSEntry( Soprano::QueryResultIterator& it )
    {
        QString name = it["p"].toString();
        if(!name.isEmpty())
            name.append(QLatin1String(" - "));
        name.append(it["t"].toString());
        if(it["tn"].isLiteral())
            name.prepend(QString::fromLatin1("%1 ").arg(it["tn"].literal().toInt(), 2, 10, QLatin1Char('0')));

        UDSEntry uds;
        uds.insert( KIO::UDSEntry::UDS_NAME, name );
        uds.insert( KIO::UDSEntry::UDS_FILE_TYPE, S_IFREG );
        uds.insert( KIO::UDSEntry::UDS_MIME_TYPE, it["mt"].toString() );
        uds.insert( KIO::UDSEntry::UDS_DISPLAY_TYPE, i18n("Music Track") );
        uds.insert( KIO::UDSEntry::UDS_ACCESS, 0700 );
        uds.insert( KIO::UDSEntry::UDS_USER, KUser().loginName() );
        uds.insert( KIO::UDSEntry::UDS_URL, KUrl(it["url"].uri()).url() );
        return uds;
    }

    QString prepareUrlToken(const QString& name)
    {
        return name.toLower().replace('/', '_').replace('&', '_');
    }

    QString prepareUrlToken(const QUrl& uri)
    {
        return QString::fromAscii(QUrl::toPercentEncoding(QString::fromAscii(uri.toEncoded())));
    }

    QUrl recoverUriFromUrlToken(const QString& token)
    {
        return QUrl::fromEncoded(QUrl::fromPercentEncoding(token.toAscii()).toAscii());
    }
}


Nepomuk::MusicProtocol::MusicProtocol( const QByteArray& poolSocket, const QByteArray& appSocket )
    : KIO::SlaveBase( "music", poolSocket, appSocket )
{
}


Nepomuk::MusicProtocol::~MusicProtocol()
{
}


void Nepomuk::MusicProtocol::listDir( const KUrl& url )
{
    // root folder
    if(url.path().length() <= 1) {
        // show folders for browsing by artist or genre
        listEntry(createFolderUDSEntry(QLatin1String("artists"), i18n("Browse by Artist")), false);
        listEntry(createFolderUDSEntry(QLatin1String("genres"), i18n("Browse by Genre")), false);
        listEntry(UDSEntry(), true);
        finished();
    }
    else {
        const QStringList pathTokens = url.path().split('/', QString::SkipEmptyParts);

        if(pathTokens.count() == 1) {
            //
            // music:/artists
            //
            if(pathTokens.first() == QLatin1String("artists")) {
                // query all artists
                Soprano::QueryResultIterator it
                        = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select distinct ?p ?pn where { ?r nmm:performer ?p . ?p nco:fullname ?pn . }"),
                                                                                          Soprano::Query::QueryLanguageSparql);
                while(it.next()) {
                    const QUrl artist = it["p"].uri();
                    const QString artistName = it["pn"].toString();
                    listEntry(createFolderUDSEntry(prepareUrlToken(artist), artistName, artist), false);
                }
                listEntry(UDSEntry(), true);
                finished();
            }

            //
            // music:/genres
            //
            else if(pathTokens.first() == QLatin1String("genres")) {
                // query all genres
                Soprano::QueryResultIterator it
                        = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select distinct ?g where { ?r nmm:genre ?g . }"),
                                                                                          Soprano::Query::QueryLanguageSparql);
                while(it.next()) {
                    const QString genreName = it["g"].toString();
                    listEntry(createFolderUDSEntry(prepareUrlToken(genreName), genreName), false);
                }
                listEntry(UDSEntry(), true);
                finished();
            }

            else {
                error( KIO::ERR_CANNOT_ENTER_DIRECTORY, url.prettyUrl() );
            }
        }

        else if(pathTokens.count() == 2) {
            //
            // music:/artists/<ARTIST>
            //
            if(pathTokens.first() == QLatin1String("artists")) {
                listEntry(createFolderUDSEntry(QLatin1String("albums"), i18n("Browse Albums")), false);
                listEntry(createFolderUDSEntry(QLatin1String("tracks"), i18n("Browse all Tracks")), false);
                listEntry(UDSEntry(), true);
                finished();
            }
            //
            // music:/genres/<GENRE>
            //
            else if(pathTokens.first() == QLatin1String("genres")) {
                listEntry(createFolderUDSEntry(QLatin1String("artists"), i18n("Browse Artists")), false);
                listEntry(createFolderUDSEntry(QLatin1String("albums"), i18n("Browse Albums")), false);
                listEntry(createFolderUDSEntry(QLatin1String("tracks"), i18n("Browse all Tracks")), false);
                listEntry(UDSEntry(), true);
                finished();
            }

            else {
                error( KIO::ERR_CANNOT_ENTER_DIRECTORY, url.prettyUrl() );
            }
        }

        else if(pathTokens.count() == 3) {
            //
            // music:/artists/<ARTIST>
            //
            if(pathTokens.first() == QLatin1String("artists")) {
                //
                // music:/artists/<ARTIST>/albums
                //
                if(pathTokens[2] == QLatin1String("albums")) {
                    // query all albums by that artist
                    const QUrl artist = recoverUriFromUrlToken(pathTokens[1]);
                    Soprano::QueryResultIterator it
                            = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select distinct ?a ?at where { "
                                                                                                                  "?a a nmm:MusicAlbum ; "
                                                                                                                  "nie:title ?at . "
                                                                                                                  "?r nmm:musicAlbum ?a ; "
                                                                                                                  "nmm:performer %1 . "
                                                                                                                  "}")
                                                                                              .arg(Soprano::Node::resourceToN3(artist)),
                                                                                              Soprano::Query::QueryLanguageSparql);
                    while(it.next()) {
                        const QUrl album = it["a"].uri();
                        const QString albumName = it["at"].toString();
                        listEntry(createFolderUDSEntry(prepareUrlToken(album), albumName, album), false);
                    }
                    listEntry(UDSEntry(), true);
                    finished();
                }

                //
                // music:/artists/<ARTIST>/tracks
                //
                else if(pathTokens[2] == QLatin1String("tracks")) {
                    // query all tracks by that artist
                    const QUrl artist = recoverUriFromUrlToken(pathTokens[1]);
                    Soprano::QueryResultIterator it
                            = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select distinct * where { "
                                                                                                                  "?r a nmm:MusicPiece ; "
                                                                                                                  "nie:title ?t ; "
                                                                                                                  "nie:url ?url ; "
                                                                                                                  "nie:mimeType ?mt ; "
                                                                                                                  "nmm:performer %1 . "
                                                                                                                  "}")
                                                                                              .arg(Soprano::Node::resourceToN3(artist)),
                                                                                              Soprano::Query::QueryLanguageSparql);
                    while(it.next()) {
                        listEntry(createMusicPieceUDSEntry(it), false);
                    }
                    listEntry(UDSEntry(), true);
                    finished();
                }

                else {
                    error( KIO::ERR_CANNOT_ENTER_DIRECTORY, url.prettyUrl() );
                }
            }

            //
            // music:/genres/<GENRE>
            //
            else if(pathTokens.first() == QLatin1String("genres")) {
                //
                // music:/genres/<GENRE>/artists
                //
                if(pathTokens[2] == QLatin1String("artists")) {
                    // query all artists in that genre
                    Soprano::QueryResultIterator it
                            = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select distinct ?p ?pn where { "
                                                                                                                  "?r a nmm:MusicPiece ; "
                                                                                                                  "nmm:genre ?g ; "
                                                                                                                  "nmm:performer ?p . "
                                                                                                                  "?p nco:fullname ?pn . "
                                                                                                                  "FILTER(bif:replace(bif:replace(bif:lower(?g),'/','_'),'&','_')='%1') . "
                                                                                                                  "}")
                                                                                              .arg(pathTokens[1]),
                                                                                              Soprano::Query::QueryLanguageSparql);
                    while(it.next()) {
                        const QUrl artist = it["p"].uri();
                        const QString artistName = it["pn"].toString();
                        listEntry(createFolderUDSEntry(prepareUrlToken(artist), artistName, artist), false);
                    }
                    listEntry(UDSEntry(), true);
                    finished();
                }

                //
                // music:/genres/<GENRE>/albums
                //
                else if(pathTokens[2] == QLatin1String("albums")) {
                    const QString genre = pathTokens[1];
                    Soprano::QueryResultIterator it
                            = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select distinct ?a ?at where { "
                                                                                                                  "?a a nmm:MusicAlbum ; "
                                                                                                                  "nie:title ?at . "
                                                                                                                  "?r nmm:musicAlbum ?a ; "
                                                                                                                  "nmm:genre ?g . "
                                                                                                                  "FILTER(bif:replace(bif:replace(bif:lower(?g),'/','_'),'&','_')='%1') . "
                                                                                                                  "}")
                                                                                              .arg(genre),
                                                                                              Soprano::Query::QueryLanguageSparql);
                    while(it.next()) {
                        const QUrl album = it["a"].uri();
                        const QString albumName = it["at"].toString();
                        listEntry(createFolderUDSEntry(prepareUrlToken(album), albumName, album), false);
                    }
                    listEntry(UDSEntry(), true);
                    finished();
                }

                //
                // music:/genres/<GENRE>/tracks
                //
                else if(pathTokens[2] == QLatin1String("tracks")) {
                    // query all tracks from that genre
                    const QString genre = pathTokens[1];
                    Soprano::QueryResultIterator it
                            = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select distinct * where { "
                                                                                                                  "?r a nmm:MusicPiece ; "
                                                                                                                  "nie:url ?url ;"
                                                                                                                  "nie:title ?t ; "
                                                                                                                  "nie:mimeType ?mt ; "
                                                                                                                  "nmm:performer [ nco:fullname ?p ] ; "
                                                                                                                  "nmm:genre ?g . "
                                                                                                                  "FILTER(bif:replace(bif:replace(bif:lower(?g),'/','_'),'&','_')='%1') . "
                                                                                                                  "}")
                                                                                              .arg(genre),
                                                                                              Soprano::Query::QueryLanguageSparql);
                    while(it.next()) {
                        listEntry(createMusicPieceUDSEntry(it), false);
                    }
                    listEntry(UDSEntry(), true);
                    finished();
                }

                else {
                    error( KIO::ERR_CANNOT_ENTER_DIRECTORY, url.prettyUrl() );
                }
            }

            else {
                error( KIO::ERR_CANNOT_ENTER_DIRECTORY, url.prettyUrl() );
            }
        }

        else if(pathTokens.count() == 4) {
            //
            // music:/artists/<ARTIST>/albums/<ALBUM>
            //
            if(pathTokens[0] == QLatin1String("artists")) {
                if(pathTokens[2] == QLatin1String("albums")) {
                    // query all tracks from that album
                    const QUrl album = recoverUriFromUrlToken(pathTokens[3]);
                    Soprano::QueryResultIterator it
                            = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select distinct * where { "
                                                                                                                  "?r a nmm:MusicPiece ; "
                                                                                                                  "nie:title ?t ; "
                                                                                                                  "nie:mimeType ?mt ; "
                                                                                                                  "nmm:trackNumber ?tn ; "
                                                                                                                  "nie:url ?url . "
                                                                                                                  "?r nmm:performer [ nco:fullname ?p ] . "
                                                                                                                  "?r nmm:musicAlbum %1 . "
                                                                                                                  "}")
                                                                                              .arg(Soprano::Node::resourceToN3(album)),
                                                                                              Soprano::Query::QueryLanguageSparql);
                    while(it.next()) {
                        listEntry(createMusicPieceUDSEntry(it), false);
                    }
                    listEntry(UDSEntry(), true);
                    finished();
                }

                else {
                    error( KIO::ERR_CANNOT_ENTER_DIRECTORY, url.prettyUrl() );
                }
            }

            else if(pathTokens[0] == QLatin1String("genres")) {
                //
                // music:/genres/<GENRE>/artists/<ARTIST>
                //
                if(pathTokens[2] == QLatin1String("artists")) {
                    listEntry(createFolderUDSEntry(QLatin1String("albums"), i18n("Browse Albums")), false);
                    listEntry(createFolderUDSEntry(QLatin1String("tracks"), i18n("Browse all Tracks")), false);
                    listEntry(UDSEntry(), true);
                    finished();
                }

                //
                // music:/genres/<GENRE>/albums/<ALBUM>
                //
                else if(pathTokens[2] == QLatin1String("albums")) {
                    // query all tracks from that album
                    const QUrl album = recoverUriFromUrlToken(pathTokens[3]);
                    Soprano::QueryResultIterator it
                            = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select distinct * where { "
                                                                                                                  "?r a nmm:MusicPiece ; "
                                                                                                                  "nie:title ?t ; "
                                                                                                                  "nie:mimeType ?mt ; "
                                                                                                                  "nmm:trackNumber ?tn ; "
                                                                                                                  "nie:url ?url . "
                                                                                                                  "?r nmm:performer [ nco:fullname ?p ] . "
                                                                                                                  "?r nmm:musicAlbum %1 . "
                                                                                                                  "}")
                                                                                              .arg(Soprano::Node::resourceToN3(album)),
                                                                                              Soprano::Query::QueryLanguageSparql);
                    while(it.next()) {
                        listEntry(createMusicPieceUDSEntry(it), false);
                    }
                    listEntry(UDSEntry(), true);
                    finished();
                }

                else {
                    error( KIO::ERR_CANNOT_ENTER_DIRECTORY, url.prettyUrl() );
                }
            }

            else {
                error( KIO::ERR_CANNOT_ENTER_DIRECTORY, url.prettyUrl() );
            }
        }

        else if(pathTokens.count() == 5) {
            //
            // music:/genres/<GENRE>/artists/<ARTIST>/tracks
            //
            if(pathTokens[2] == QLatin1String("artists")) {
                // query all tracks by that artist
                const QString genre = pathTokens[1];
                const QUrl artist = recoverUriFromUrlToken(pathTokens[3]);
                Soprano::QueryResultIterator it
                        = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select distinct * where { "
                                                                                                              "?r a nmm:MusicPiece ; "
                                                                                                              "nie:title ?t ; "
                                                                                                              "nie:url ?url ; "
                                                                                                              "nie:mimeType ?mt ; "
                                                                                                              "nmm:performer %1 ; "
                                                                                                              "nmm:genre ?g . "
                                                                                                              "FILTER(bif:replace(bif:replace(bif:lower(?g),'/','_'),'&','_')='%2') . "
                                                                                                              "}")
                                                                                          .arg(Soprano::Node::resourceToN3(artist),
                                                                                               genre),
                                                                                          Soprano::Query::QueryLanguageSparql);
                while(it.next()) {
                    listEntry(createMusicPieceUDSEntry(it), false);
                }
                listEntry(UDSEntry(), true);
                finished();
            }

            else {
                error( KIO::ERR_CANNOT_ENTER_DIRECTORY, url.prettyUrl() );
            }
        }
        else {
            error( KIO::ERR_CANNOT_ENTER_DIRECTORY, url.prettyUrl() );
        }
    }
}

void Nepomuk::MusicProtocol::mkdir( const KUrl &url, int permissions )
{
    Q_UNUSED(permissions);
    error( ERR_UNSUPPORTED_ACTION, url.prettyUrl() );
}


void Nepomuk::MusicProtocol::get( const KUrl& url )
{
    error( ERR_UNSUPPORTED_ACTION, url.prettyUrl() );
}


void Nepomuk::MusicProtocol::put( const KUrl& url, int permissions, KIO::JobFlags flags )
{
    Q_UNUSED(permissions);
    Q_UNUSED(flags);

    error( KIO::ERR_UNSUPPORTED_ACTION, url.prettyUrl() );
}


void Nepomuk::MusicProtocol::copy( const KUrl& src, const KUrl& dest, int permissions, KIO::JobFlags flags )
{
    Q_UNUSED(src);
    Q_UNUSED(dest);
    Q_UNUSED(permissions);
    Q_UNUSED(flags);

    error( ERR_UNSUPPORTED_ACTION, src.prettyUrl() );
}


void Nepomuk::MusicProtocol::rename( const KUrl& src, const KUrl& dest, KIO::JobFlags flags )
{
    Q_UNUSED(src);
    Q_UNUSED(dest);
    Q_UNUSED(flags);

    error( ERR_UNSUPPORTED_ACTION, src.prettyUrl() );
}


void Nepomuk::MusicProtocol::del( const KUrl& url, bool isfile )
{
    Q_UNUSED(isfile);
    error( ERR_UNSUPPORTED_ACTION, url.prettyUrl() );
}


void Nepomuk::MusicProtocol::mimetype( const KUrl& url )
{
    // FIXME
    error( ERR_UNSUPPORTED_ACTION, url.prettyUrl() );
}


void Nepomuk::MusicProtocol::stat( const KUrl& url )
{
    const QStringList pathTokens = url.path().split('/', QString::SkipEmptyParts);

    if(pathTokens.isEmpty()) {
        statEntry(createFolderUDSEntry(QLatin1String("music"), i18n("Browse Music")));
        finished();
    }

    // a folder which contains artists
    else if(pathTokens.last() == QLatin1String("artists")) {
        statEntry(createFolderUDSEntry(pathTokens.last(), i18n("Browse by Artist")));
        finished();
    }

    // a folder which contains genres (there is only the one)
    else if(pathTokens.last() == QLatin1String("genres")) {
        statEntry(createFolderUDSEntry(pathTokens.last(), i18n("Browse by Genre")));
        finished();
    }

    // any folder containing tracks
    else if(pathTokens.last() == QLatin1String("tracks")) {
        statEntry(createFolderUDSEntry(QLatin1String("tracks"), i18n("Browse all Tracks")));
        finished();
    }

    // any folder listing albums
    else if(pathTokens.last() == QLatin1String("albums")) {
        statEntry(createFolderUDSEntry(QLatin1String("tracks"), i18n("Browse Albums")));
        finished();
    }

    // any folder listing an album
    else if(pathTokens.contains(QLatin1String("albums"))) {
        // stat an album
        const QUrl album = recoverUriFromUrlToken(pathTokens.last());
        Soprano::QueryResultIterator it
                = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select ?t where { "
                                                                                                      "%1 nie:title ?t } LIMIT 1")
                                                                                  .arg(Soprano::Node::resourceToN3(album)),
                                                                                  Soprano::Query::QueryLanguageSparql);
        if(it.next()) {
            statEntry(createFolderUDSEntry(prepareUrlToken(album), it["t"].toString(), album));
            finished();
        }
        else {
            error(KIO::ERR_DOES_NOT_EXIST, url.prettyUrl());
        }
    }

    // a genre folder
    else if(pathTokens.count() == 2 &&
            pathTokens[0] == QLatin1String("genres")) {
        const QString encodedGenre = pathTokens[1];
        Soprano::QueryResultIterator it
                = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select ?g where { "
                                                                                                      "?r a nmm:MusicPiece ; "
                                                                                                      "nmm:genre ?g . "
                                                                                                      "FILTER(bif:replace(bif:replace(bif:lower(?g),'/','_'),'&','_')='%1') . "
                                                                                                      "} LIMIT 1")
                                                                                  .arg(encodedGenre),
                                                                                  Soprano::Query::QueryLanguageSparql);
        if(it.next()) {
            statEntry(createFolderUDSEntry(encodedGenre, it["g"].toString()));
            finished();
        }
        else {
            error(KIO::ERR_DOES_NOT_EXIST, url.prettyUrl());
        }
    }

    // an artist
    else if(pathTokens.contains(QLatin1String("artists"))) {
        const QUrl uri = recoverUriFromUrlToken(pathTokens.last());
        Soprano::QueryResultIterator it
                = Nepomuk::ResourceManager::instance()->mainModel()->executeQuery(QString::fromLatin1("select ?n where { "
                                                                                                      "%1 nco:fullname ?n } LIMIT 1")
                                                                                  .arg(Soprano::Node::resourceToN3(uri)),
                                                                                  Soprano::Query::QueryLanguageSparql);
        if(it.next()) {
            statEntry(createFolderUDSEntry(pathTokens.last(), it["n"].toString(), uri));
            finished();
        }
        else {
            error(KIO::ERR_DOES_NOT_EXIST, url.prettyUrl());
        }
    }

    else {
        error( ERR_DOES_NOT_EXIST, url.prettyUrl() );
    }
}


extern "C"
{
    KDE_EXPORT int kdemain( int argc, char **argv )
    {
        // necessary to use other kio slaves
        KComponentData( "kio_music" );
        QCoreApplication app( argc, argv );

        kDebug(7102) << "Starting music slave " << getpid();

        if (argc != 4) {
            kError() << "Usage: kio_music protocol domain-socket1 domain-socket2";
            exit(-1);
        }

        Nepomuk::MusicProtocol slave(argv[2], argv[3]);
        slave.dispatchLoop();

        kDebug(7102) << "Music slave Done";

        return 0;
    }
}
