# frozen_string_literal: true

require 'time'

module GitlabQuality
  module TestTooling
    module TestMetricsExporter
      class TestMetrics
        def initialize(example, timestamp)
          @example = example
          @timestamp = timestamp
        end

        # Test data hash
        #
        # @return [Hash]
        def data
          {
            timestamp: timestamp,
            **rspec_metrics,
            **ci_metrics,
            **custom_metrics
          }.compact
        end

        private

        attr_reader :example, :timestamp

        # Exporter configuration
        #
        # @return [Config]
        def config
          Config.configuration
        end

        # Rspec related metrics
        #
        # @return [Hash]
        def rspec_metrics # rubocop:disable Metrics/AbcSize
          {
            id: without_relative_path(example.id),
            name: example.full_description,
            hash: OpenSSL::Digest.hexdigest("SHA256", "#{file_path}#{example.full_description}")[..40],
            file_path: file_path,
            status: example.execution_result.status,
            run_time: (example.execution_result.run_time * 1000).round,
            location: example_location,
            # TODO: remove exception_class once migration to exception_classes is fully complete on clickhouse side
            exception_class: example.execution_result.exception&.class&.to_s,
            exception_classes: exception_classes.map { |e| e.class.to_s }.uniq,
            failure_exception: failure_exception,
            quarantined: quarantined?,
            feature_category: example.metadata[:feature_category] || "",
            test_retried: config.test_retried_proc.call(example),
            run_type: run_type,
            spec_file_path_prefix: config.spec_file_path_prefix
          }
        end

        # CI related metrics
        #
        # @return [Hash]
        def ci_metrics
          {
            ci_project_id: env("CI_PROJECT_ID")&.to_i,
            ci_project_path: env("CI_PROJECT_PATH"),
            ci_job_name: ci_job_name,
            ci_job_id: env('CI_JOB_ID')&.to_i,
            ci_pipeline_id: env('CI_PIPELINE_ID')&.to_i,
            ci_merge_request_iid: (env('CI_MERGE_REQUEST_IID') || env('TOP_UPSTREAM_MERGE_REQUEST_IID'))&.to_i,
            ci_branch: env("CI_COMMIT_REF_NAME"),
            ci_target_branch: env("CI_MERGE_REQUEST_TARGET_BRANCH_NAME"),
            ci_server_url: env("CI_SERVER_URL")
          }
        end

        # Additional custom metrics
        #
        # @return [Hash]
        def custom_metrics
          metrics = example.metadata
            .slice(*config.extra_rspec_metadata_keys)
            .merge(config.custom_metrics_proc.call(example))

          metrics.each_with_object({}) do |(k, value), custom_metrics|
            custom_metrics[k.to_sym] = metrics_value(value)
          end
        end

        # Checks if spec is quarantined
        #
        # @return [String]
        def quarantined?
          return false unless example.metadata.key?(:quarantine)

          # if quarantine key is present and status is pending, consider it quarantined
          example.execution_result.status == :pending
        end

        # Base ci job name
        #
        # @return [String]
        def ci_job_name
          env("CI_JOB_NAME")&.gsub(%r{ \d{1,2}/\d{1,2}}, '')
        end

        # Example location
        #
        # @return [String]
        def example_location
          return @example_location if @example_location

          # ensures that location will be correct even in case of shared examples
          file = example
                 .metadata
                 .fetch(:shared_group_inclusion_backtrace)
                 .last
                 &.formatted_inclusion_location

          return without_relative_path(example.location) unless file

          @example_location = without_relative_path(file)
        end

        # File path based on actual test location, not shared example location
        #
        # @return [String]
        def file_path
          @file_path ||= example_location.gsub(/:\d+$/, "")
        end

        # Failure exception classes
        #
        # @return [Array<Exception>]
        def exception_classes
          exception = example.execution_result.exception
          return [] unless exception
          return [exception] unless exception.respond_to?(:all_exceptions)

          exception.all_exceptions.flatten
        end

        # Truncated exception stacktrace
        #
        # @return [String]
        def failure_exception
          exception = example.execution_result.exception
          return unless exception

          exception.to_s.tr("\n", " ").slice(0, 1000)
        end

        # Test run type | suite name
        #
        # @return [String]
        def run_type
          config.run_type || ci_job_name || "unknown"
        end

        # Return non empty environment variable value
        #
        # @param [String] name
        # @return [String, nil]
        def env(name)
          return unless ENV[name] && !ENV[name].empty?

          ENV.fetch(name)
        end

        # Metrics value cast to a valid type
        #
        # @param value [Object]
        # @return [Object]
        def metrics_value(value)
          return value if value.is_a?(Numeric) || value.is_a?(String) || bool?(value) || value.nil?

          value.to_s
        end

        # Value is a true or false
        #
        # @param val [Object]
        # @return [Boolean]
        def bool?(val)
          [true, false].include?(val)
        end

        # Path without leading ./
        #
        # @param path [String]
        # @return [String]
        def without_relative_path(path)
          path.gsub(%r{^\./}, "")
        end
      end
    end
  end
end
