/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.controller.ControllerResult;
import org.apache.kafka.controller.PeriodicTask;
import org.apache.kafka.controller.PeriodicTaskFlag;
import org.apache.kafka.controller.errors.PeriodicControlTaskException;
import org.slf4j.Logger;

class PeriodicTaskControlManager {
    private final Logger log;
    private final Time time;
    private final QueueAccessor queueAccessor;
    private boolean active;
    private final Map<String, PeriodicTask> tasks;

    private PeriodicTaskControlManager(LogContext logContext, Time time, QueueAccessor queueAccessor) {
        this.log = logContext.logger(PeriodicTaskControlManager.class);
        this.time = time;
        this.queueAccessor = queueAccessor;
        this.active = false;
        this.tasks = new HashMap<String, PeriodicTask>();
    }

    boolean active() {
        return this.active;
    }

    void registerTask(PeriodicTask task) {
        if (this.tasks.containsKey(task.name())) {
            this.log.debug("Periodic task {} is already registered.", (Object)task.name());
            return;
        }
        this.tasks.put(task.name(), task);
        this.log.info("Registering periodic task {} to run every {} ms", (Object)task.name(), (Object)TimeUnit.NANOSECONDS.toMillis(task.periodNs()));
        this.reschedule(task, false, false);
    }

    void unregisterTask(String taskName) {
        PeriodicTask task = this.tasks.remove(taskName);
        if (task == null) {
            this.log.debug("Periodic task {} is already unregistered.", (Object)taskName);
            return;
        }
        this.log.info("Unregistering periodic task {}", (Object)taskName);
        this.reschedule(task, false, false);
    }

    private long nextDelayTimeNs(PeriodicTask task, boolean immediate, boolean error) {
        if (immediate) {
            return task.immediatePeriodNs();
        }
        if (error) {
            return TimeUnit.MINUTES.toNanos(5L);
        }
        return task.periodNs();
    }

    private void reschedule(PeriodicTask task, boolean immediate, boolean error) {
        if (!this.active) {
            this.log.trace("cancelling {} because we are inactive.", (Object)task.name());
            this.queueAccessor.cancelDeferred(task.name());
        } else if (this.tasks.containsKey(task.name())) {
            long nextDelayTimeNs = this.nextDelayTimeNs(task, immediate, error);
            long nextRunTimeNs = this.time.nanoseconds() + nextDelayTimeNs;
            this.log.trace("rescheduling {} in {} ns (immediate = {}, error = {})", new Object[]{task.name(), nextDelayTimeNs, immediate, error});
            this.queueAccessor.scheduleDeferred(task.name(), nextRunTimeNs, new PeriodicTaskOperation(task));
        } else {
            this.log.trace("cancelling {} because it does not appear in the task map.", (Object)task.name());
            this.queueAccessor.cancelDeferred(task.name());
        }
    }

    void activate() {
        if (this.active) {
            throw new RuntimeException("Can't activate already active PeriodicTaskControlManager.");
        }
        this.active = true;
        for (PeriodicTask task : this.tasks.values()) {
            this.reschedule(task, false, false);
        }
        Object[] taskNames = this.tasks.keySet().toArray(new String[0]);
        Arrays.sort(taskNames);
        this.log.info("Activated periodic tasks: {}", (Object)String.join((CharSequence)", ", (CharSequence[])taskNames));
    }

    void deactivate() {
        if (!this.active) {
            return;
        }
        this.active = false;
        for (PeriodicTask task : this.tasks.values()) {
            this.reschedule(task, false, false);
        }
        Object[] taskNames = this.tasks.keySet().toArray(new String[0]);
        Arrays.sort(taskNames);
        this.log.info("Deactivated periodic tasks: {}", (Object)String.join((CharSequence)", ", (CharSequence[])taskNames));
    }

    static interface QueueAccessor {
        public void scheduleDeferred(String var1, long var2, Supplier<ControllerResult<Void>> var4);

        public void cancelDeferred(String var1);
    }

    class PeriodicTaskOperation
    implements Supplier<ControllerResult<Void>> {
        private final PeriodicTask task;

        PeriodicTaskOperation(PeriodicTask task) {
            this.task = task;
        }

        @Override
        public ControllerResult<Void> get() {
            ControllerResult<Boolean> result;
            long startNs = 0L;
            if (PeriodicTaskControlManager.this.log.isDebugEnabled() || this.task.flags().contains((Object)PeriodicTaskFlag.VERBOSE)) {
                startNs = PeriodicTaskControlManager.this.time.nanoseconds();
            }
            try {
                result = this.task.op().get();
            }
            catch (Exception e) {
                PeriodicTaskControlManager.this.reschedule(this.task, false, true);
                throw new PeriodicControlTaskException(this.task.name() + ": periodic task failed: " + e.getMessage(), e);
            }
            if (PeriodicTaskControlManager.this.log.isDebugEnabled() || this.task.flags().contains((Object)PeriodicTaskFlag.VERBOSE)) {
                long endNs = PeriodicTaskControlManager.this.time.nanoseconds();
                long durationUs = TimeUnit.NANOSECONDS.toMicros(endNs - startNs);
                if (this.task.flags().contains((Object)PeriodicTaskFlag.VERBOSE)) {
                    PeriodicTaskControlManager.this.log.info("Periodic task {} generated {} records in {} microseconds.", new Object[]{this.task.name(), result.records().size(), durationUs});
                } else if (PeriodicTaskControlManager.this.log.isDebugEnabled()) {
                    PeriodicTaskControlManager.this.log.debug("Periodic task {} generated {} records in {} microseconds.", new Object[]{this.task.name(), result.records().size(), durationUs});
                }
            }
            PeriodicTaskControlManager.this.reschedule(this.task, result.response(), false);
            if (result.isAtomic()) {
                return ControllerResult.atomicOf(result.records(), null);
            }
            return ControllerResult.of(result.records(), null);
        }
    }

    static class Builder {
        private LogContext logContext = null;
        private Time time = Time.SYSTEM;
        private QueueAccessor queueAccessor = null;

        Builder() {
        }

        Builder setLogContext(LogContext logContext) {
            this.logContext = logContext;
            return this;
        }

        Builder setTime(Time time) {
            this.time = time;
            return this;
        }

        Builder setQueueAccessor(QueueAccessor queueAccessor) {
            this.queueAccessor = queueAccessor;
            return this;
        }

        PeriodicTaskControlManager build() {
            if (this.logContext == null) {
                this.logContext = new LogContext();
            }
            if (this.queueAccessor == null) {
                throw new RuntimeException("You must set queueAccessor");
            }
            return new PeriodicTaskControlManager(this.logContext, this.time, this.queueAccessor);
        }
    }
}

