// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import BN from 'bn.js';
import { ContentType, SequenceIdentifier, Maximum } from './identifiers';
import { needs } from '@aws-crypto/material-management';
/*
 * This public interface for reading the BodyHeader format is provided for
 * the use of the Encryption SDK for JavaScript only.  It can be used
 * as a reference but is not intended to be use by any packages other
 * than the Encryption SDK for JavaScript.
 *
 * See:
 * https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#body-framing
 * https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#body-no-framing
 */
/**
 * decodeBodyHeader
 *
 * I need to be able to parse the BodyHeader, but since the data may be streamed
 * I may not have all the data yet.  The caller is expected to maintain and append
 * to the buffer and call this function with the same readPos until the function
 * returns a BodyHeader.
 *
 * @param buffer Uint8Array
 * @param headerInfo HeaderInfo
 * @param readPos number
 * @returns BodyHeader|false
 */
export function decodeBodyHeader(buffer, headerInfo, readPos) {
    /* Precondition: The contentType must be a supported format. */
    needs(ContentType[headerInfo.messageHeader.contentType], 'Unknown contentType');
    switch (headerInfo.messageHeader.contentType) {
        case ContentType.FRAMED_DATA:
            return decodeFrameBodyHeader(buffer, headerInfo, readPos);
        case ContentType.NO_FRAMING:
            return decodeNonFrameBodyHeader(buffer, headerInfo, readPos);
    }
    return false;
}
/**
 *  Exported for testing.  Used by decodeBodyHeader to compose a complete solution.
 * @param buffer Uint8Array
 * @param headerInfo HeaderInfo
 * @param readPos number
 */
export function decodeFrameBodyHeader(buffer, headerInfo, readPos) {
    /* Precondition: The contentType must be FRAMED_DATA. */
    needs(ContentType.FRAMED_DATA === headerInfo.messageHeader.contentType, 'Unknown contentType');
    const { frameLength } = headerInfo.messageHeader;
    const { ivLength, tagLength } = headerInfo.algorithmSuite;
    /* Uint8Array is a view on top of the underlying ArrayBuffer.
     * This means that raw underlying memory stored in the ArrayBuffer
     * may be larger than the Uint8Array.  This is especially true of
     * the Node.js Buffer object.  The offset and length *must* be
     * passed to the DataView otherwise I will get unexpected results.
     */
    const dataView = new DataView(buffer.buffer, buffer.byteOffset, buffer.byteLength);
    /* Precondition: decodeFrameBodyHeader readPos must be within the byte length of the buffer given. */
    needs(dataView.byteLength >= readPos && readPos >= 0, 'readPos out of bounds.');
    /* Check for early return (Postcondition): There must be enough data to decodeFrameBodyHeader.
     * The format expressed here is
     * SequenceIdentifier: Uint32
     * IVLength: Uint8
     * There is a special case where the SequenceIdentifier is the Final Frame.
     */
    if (4 + ivLength + readPos > dataView.byteLength)
        return false;
    const sequenceNumber = dataView.getUint32(readPos);
    /* Postcondition: decodeFrameBodyHeader sequenceNumber must be greater than 0. */
    needs(sequenceNumber > 0, 'Malformed sequenceNumber.');
    if (sequenceNumber === SequenceIdentifier.SEQUENCE_NUMBER_END) {
        return decodeFinalFrameBodyHeader(buffer, headerInfo, readPos);
    }
    const iv = buffer.slice((readPos += 4), (readPos += ivLength));
    return {
        sequenceNumber,
        iv,
        contentLength: frameLength,
        readPos,
        tagLength,
        isFinalFrame: false,
        contentType: ContentType.FRAMED_DATA,
    };
}
/**
 *  Exported for testing.  Used by decodeBodyHeader to compose a complete solution.
 * @param buffer Uint8Array
 * @param headerInfo HeaderInfo
 * @param readPos number
 */
export function decodeFinalFrameBodyHeader(buffer, headerInfo, readPos) {
    /* Precondition: The contentType must be FRAMED_DATA to be a Final Frame. */
    needs(ContentType.FRAMED_DATA === headerInfo.messageHeader.contentType, 'Unknown contentType');
    const { ivLength, tagLength } = headerInfo.algorithmSuite;
    /* Uint8Array is a view on top of the underlying ArrayBuffer.
     * This means that raw underlying memory stored in the ArrayBuffer
     * may be larger than the Uint8Array.  This is especially true of
     * the Node.js Buffer object.  The offset and length *must* be
     * passed to the DataView otherwise I will get unexpected results.
     */
    const dataView = new DataView(buffer.buffer, buffer.byteOffset, buffer.byteLength);
    /* Precondition: decodeFinalFrameBodyHeader readPos must be within the byte length of the buffer given. */
    needs(dataView.byteLength >= readPos && readPos >= 0, 'readPos out of bounds.');
    /* Check for early return (Postcondition): There must be enough data to decodeFinalFrameBodyHeader.
     * The format expressed here is
     * SEQUENCE_NUMBER_END: Uint32(FFFF)
     * SequenceIdentifier: Uint32
     * IVLength: Uint8
     * Reserved: Uint32
     * ContentLength: Uint32
     */
    if (4 + 4 + ivLength + 4 + readPos > dataView.byteLength)
        return false;
    /* The precondition SEQUENCE_NUMBER_END: Uint32(FFFF) is handled above. */
    const sequenceEnd = dataView.getUint32(readPos, false); // big endian
    /* Postcondition: sequenceEnd must be SEQUENCE_NUMBER_END. */
    needs(sequenceEnd === SequenceIdentifier.SEQUENCE_NUMBER_END, 'Malformed final frame: Invalid sequence number end value');
    const sequenceNumber = dataView.getUint32((readPos += 4), false); // big endian
    /* Postcondition: decodeFinalFrameBodyHeader sequenceNumber must be greater than 0. */
    needs(sequenceNumber > 0, 'Malformed sequenceNumber.');
    const iv = buffer.slice((readPos += 4), (readPos += ivLength));
    const contentLength = dataView.getUint32(readPos);
    /* Postcondition: The final frame MUST NOT exceed the frameLength. */
    needs(headerInfo.messageHeader.frameLength >= contentLength, 'Final frame length exceeds frame length.');
    return {
        sequenceNumber,
        iv,
        contentLength,
        readPos: readPos + 4,
        tagLength,
        isFinalFrame: true,
        contentType: ContentType.FRAMED_DATA,
    };
}
/**
 * Exported for testing.  Used by decodeBodyHeader to compose a complete solution.
 * @param buffer Uint8Array
 * @param headerInfo HeaderInfo
 * @param readPos number
 */
export function decodeNonFrameBodyHeader(buffer, headerInfo, readPos) {
    /* Precondition: The contentType must be NO_FRAMING. */
    needs(ContentType.NO_FRAMING === headerInfo.messageHeader.contentType, 'Unknown contentType');
    const { ivLength, tagLength } = headerInfo.algorithmSuite;
    /* Uint8Array is a view on top of the underlying ArrayBuffer.
     * This means that raw underlying memory stored in the ArrayBuffer
     * may be larger than the Uint8Array.  This is especially true of
     * the Node.js Buffer object.  The offset and length *must* be
     * passed to the DataView otherwise I will get unexpected results.
     */
    const dataView = new DataView(buffer.buffer, buffer.byteOffset, buffer.byteLength);
    /* Precondition: decodeNonFrameBodyHeader readPos must be within the byte length of the buffer given. */
    needs(dataView.byteLength >= readPos && readPos >= 0, 'readPos out of bounds.');
    /* Check for early return (Postcondition): There must be enough data to decodeNonFrameBodyHeader.
     * The format expressed here is
     * IVLength: Uint8
     * ContentLength: Uint64
     */
    if (ivLength + 8 + readPos > dataView.byteLength)
        return false;
    const iv = buffer.slice(readPos, (readPos += ivLength));
    const contentLengthBuff = buffer.slice(readPos, (readPos += 8));
    const contentLengthBN = new BN([...contentLengthBuff], 16, 'be');
    // This will throw if the number is larger than Number.MAX_SAFE_INTEGER.
    // i.e. a 53 bit number
    const contentLength = contentLengthBN.toNumber();
    /* Postcondition: Non-framed content length MUST NOT exceed AES-GCM safe limits.
     * https://github.com/awslabs/aws-encryption-sdk-specification/blob/master/data-format/message-body.md#encrypted-content-length
     */
    needs(Maximum.BYTES_PER_AES_GCM_NONCE > contentLength, 'Content length out of bounds.');
    return {
        sequenceNumber: 1,
        iv,
        contentLength,
        readPos,
        tagLength,
        isFinalFrame: true,
        contentType: ContentType.NO_FRAMING,
    };
}
//# sourceMappingURL=data:application/json;base64,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