#ifndef KEEPAWAKE_H
#define KEEPAWAKE_H


#define KEEPAWAKE_MAJOR 0
#define KEEPAWAKE_MINOR 5
#define KEEPAWAKE_PATCH 1


#include <stdarg.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdlib.h>

typedef struct KeepAwakeBuilder KeepAwakeBuilder;

/**
 * Keeps the machine or display awake (as configured), until dropped. Create using [struct@Builder].
 */
typedef struct KeepAwake KeepAwake;

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

/**
 * Create a new [`KeepAwakeBuilder`].
 */
KeepAwakeBuilder *keepawake_new(void);

/**
 * Prevent the display from turning off.
 */
void keepawake_display(KeepAwakeBuilder *builder, bool value);

/**
 * Prevent the system from sleeping due to idleness.
 */
void keepawake_idle(KeepAwakeBuilder *builder, bool value);

/**
 * Prevent the system from sleeping. Only works under certain, OS dependant, conditions.
 */
void keepawake_sleep(KeepAwakeBuilder *builder, bool value);

/**
 * Reason the consumer is keeping the system awake. Defaults to `"User requested"`. (Used on Linux & macOS)
 */
void keepawake_reason(KeepAwakeBuilder *builder,
                      const char *value);

/**
 * Name of the program keeping the system awake. Defaults to `"keepawake-rs"`. (Used on Linux)
 */
void keepawake_app_name(KeepAwakeBuilder *builder, const char *value);

/**
 * Reverse domain name of the program keeping the system awake. Defaults to `"io.github.segevfiner.keepawake-rs"`. (Used on Linux)
 */
void keepawake_app_reverse_domain(KeepAwakeBuilder *builder,
                                  const char *value);

/**
 * Create the [`KeepAwake`]. Optionally destroying the builder.
 */
struct KeepAwake *keepawake_create(KeepAwakeBuilder *builder, bool free_builder);

/**
 * Destroy the [`KeepAwakeBuilder`].
 */
void keepawake_builder_destroy(KeepAwakeBuilder *builder);

/**
 * Destroy the [`KeepAwake`].
 */
void keepawake_destroy(struct KeepAwake *awake);

#ifdef __cplusplus
} // extern "C"
#endif // __cplusplus

#endif /* KEEPAWAKE_H */
