%feature("docstring") OT::LinearModelAnalysis
"Analyse a linear model.

Parameters
----------
linearModelResult : :class:`~openturns.LinearModelResult`
    A linear model result.

See Also
--------
LinearModelResult

Notes
-----
This class relies on a linear model result structure and analyses the results.

By default, on graphs, labels of the 3 most significant points are displayed.
This number can be changed by modifying the ResourceMap key
(``LinearModelAnalysis-Identifiers``).

The class has a pretty-print method which is triggered by
the `print()` function.
This prints the following results, where we focus on the properties
of a satisfactory regression model.

- Each row of the table of coefficients tests if one single coefficient is zero.
  For a single coefficient, if the p-value of the T-test is close to zero,
  we can reject the hypothesis that this coefficient is zero.
- The R2 score measures how the predicted output values are close to the
  observed values.
  If the R2 is close to 1 (e.g. larger than 0.95), then the predictions are
  accurate on average.
  Furthermore, the adjusted R2 value takes into account the data set
  size and the number of hyperparameters.
- The F-test tests if all the coefficients are simultaneously zero.
  If the p-value is close to zero, then we can reject this hypothesis:
  there is at least one nonzero coefficient.
- The normality test checks that the residuals have a normal distribution.
  The normality assumption can be accepted (or, more precisely, cannot be
  rejected) if the p-value is larger than a threshold (e.g. 0.05).

The essentials of regression theory are presented in :ref:`regression_analysis`.
The goodness of fit tests for normality are presented in :ref:`graphical_fitting_test`, 
:ref:`chi2_fitting_test`, :ref:`kolmogorov_smirnov_test`, :ref:`cramer_vonmises_test`  and
:ref:`anderson_darling_test`.

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.Normal()
>>> Xsample = distribution.getSample(30)
>>> func = ot.SymbolicFunction(['x'], ['2 * x + 1'])
>>> Ysample = func(Xsample) + ot.Normal().getSample(30)
>>> algo = ot.LinearModelAlgorithm(Ysample, Xsample)
>>> result = algo.getResult()
>>> analysis = ot.LinearModelAnalysis(result)
>>> # print(analysis)  # Pretty-print"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getLinearModelResult
"Accessor to the linear model result.

Returns
-------
linearModelResult : :class:`~openturns.LinearModelResult`
    The  linear model result which had been passed to the constructor."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getCoefficientsTScores
"Accessor to the coefficients of linear expansion over their standard error.

Returns
-------
tScores : :class:`~openturns.Point`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getCoefficientsPValues
"Accessor to the coefficients of the p values.

Returns
-------
pValues : :class:`~openturns.Point`
"
// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getCoefficientsConfidenceInterval
"Accessor to the confidence interval of level :math:`\\alpha` for the coefficients
of the linear expansion

Returns
-------
confidenceInterval : :class:`~openturns.Interval`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getFisherScore
"Accessor to the Fisher statistics.

Returns
-------
fisherScore : float
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getFisherPValue
"Accessor to the Fisher p value.

Returns
-------
fisherPValue : float
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getResidualsStandardError
"Accessor to the standard error of the residuals.

Returns
-------
stdError : float
    This is the unbiased estimator :math:`\\hat{\\sigma}` of the standard deviation
    of the Gaussian observation error of the observed output.

Notes
-----
The standard error of the residuals is the *root mean squared error* (also
called *standard error of regression*), estimated as:

.. math::
    \\widehat{\\sigma} = \\sqrt{\\frac{1}{\\operatorname{dof}} SSE}

where :math:`\\operatorname{dof}` is the number of degrees of freedom
and :math:`SSE` is the sum of squared errors:

.. math::
    SSE = \\sum_{i = 1}^n (y_i - \\widehat{y}_i)^2

where :math:`\\{y_i\\}_{i = 1, ..., n}` are the observations and
:math:`\\{\\widehat{y}_i\\}_{i = 1, ..., n}` are the predictions from the linear model.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getNormalityTestResultChiSquared
"Performs Chi-Square test.
The statistical test checks the Gaussian assumption of the residuals.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.

Notes
-----
The Chi-Square test is a goodness of fit test which objective is to check the
normality assumption (null hypothesis) of residuals (and thus the model).

Usually, Chi-Square test applies for discrete distributions. Here we rely on
the :meth:`~openturns.FittingTest.ChiSquared` to check the normality.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getNormalityTestResultKolmogorovSmirnov
"Performs Kolmogorov test.

Performs Kolmogorov test to check Gaussian assumption of the residuals.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.


Notes
-----
We check that the residual is Gaussian thanks to :meth:`~openturns.FittingTest.Kolmogorov`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getNormalityTestResultAndersonDarling
"Performs Anderson-Darling test.
The statistical test checks the Gaussian assumption of the residuals.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.


Notes
-----
We check that the residual is Gaussian thanks to :meth:`~openturns.NormalityTest.AndersonDarlingNormal`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::getNormalityTestCramerVonMises
"Performs Cramer-Von Mises test.

The statistical test checks the Gaussian assumption of the model (null hypothesis).

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.


Notes
-----
We check that the residual is Gaussian thanks to :meth:`~openturns.NormalityTest.CramerVonMisesNormal`."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawModelVsFitted
"Accessor to plot of model versus fitted values.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------
%feature("docstring") OT::LinearModelAnalysis::drawResidualsVsFitted
"Accessor to plot of residuals versus fitted values.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawScaleLocation
"Accessor to a Scale-Location plot of sqrt(abs(residuals)) versus fitted values.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawQQplot
"Accessor to plot a Normal quantiles-quantiles plot of standardized residuals.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawCookDistance
"Accessor to plot of Cook's distances versus row labels.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawResidualsVsLeverages
"Accessor to plot of residuals versus leverages that adds bands corresponding to Cook's distances of 0.5 and 1.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAnalysis::drawCookVsLeverages
"Accessor to plot of Cook's distances versus leverage/(1-leverage). 

Returns
-------
graph : :class:`~openturns.Graph`
"

