import os
import time

from conan.api.output import ConanOutput, TimedOutput
from conans.client.rest import response_to_str
from conan.internal.errors import InternalErrorException, RequestErrorException, AuthenticationException, \
    ForbiddenException, NotFoundException
from conan.errors import ConanException
from conans.util.files import sha1sum


class FileUploader(object):

    def __init__(self, requester, verify, config, source_credentials=None):
        self._requester = requester
        self._config = config
        self._verify_ssl = verify
        self._source_credentials = source_credentials

    @staticmethod
    def _handle_400_response(response, auth):
        if response.status_code == 400:
            raise RequestErrorException(response_to_str(response))

        if response.status_code == 401:
            raise AuthenticationException(response_to_str(response))

        if response.status_code == 403:
            if auth is None or auth.bearer is None:
                raise AuthenticationException(response_to_str(response))
            raise ForbiddenException(response_to_str(response))

    def _dedup(self, url, headers, auth):
        """ send the headers to see if it is possible to skip uploading the file, because it
        is already in the server. Artifactory support file deduplication
        """
        dedup_headers = {"X-Checksum-Deploy": "true"}
        if headers:
            dedup_headers.update(headers)
        response = self._requester.put(url, data="", verify=self._verify_ssl, headers=dedup_headers,
                                       auth=auth, source_credentials=self._source_credentials)
        if response.status_code == 500:
            raise InternalErrorException(response_to_str(response))

        self._handle_400_response(response, auth)

        if response.status_code == 201:  # Artifactory returns 201 if the file is there
            return response

    def exists(self, url, auth):
        response = self._requester.head(url, verify=self._verify_ssl, auth=auth,
                                        source_credentials=self._source_credentials)
        return bool(response.ok)

    def upload(self, url, abs_path, auth=None, dedup=False, retry=None, retry_wait=None,
               ref=None):
        retry = retry if retry is not None else self._config.get("core.upload:retry", default=1,
                                                                 check_type=int)
        retry_wait = retry_wait if retry_wait is not None else \
            self._config.get("core.upload:retry_wait", default=5, check_type=int)

        # Send always the header with the Sha1
        headers = {"X-Checksum-Sha1": sha1sum(abs_path)}
        if dedup:
            response = self._dedup(url, headers, auth)
            if response:
                return response

        for counter in range(retry + 1):
            try:
                return self._upload_file(url, abs_path, headers, auth, ref)
            except (NotFoundException, ForbiddenException, AuthenticationException,
                    RequestErrorException):
                raise
            except ConanException as exc:
                if counter == retry:
                    raise
                else:
                    ConanOutput().warning(exc, warn_tag="network")
                    ConanOutput().info("Waiting %d seconds to retry..." % retry_wait)
                    time.sleep(retry_wait)

    def _upload_file(self, url, abs_path, headers, auth, ref):
        class FileProgress:  # Wrapper just to provide an upload progress every 10 seconds
            def __init__(self, f, total_size):
                self._f = f
                self._total = total_size
                self._name = os.path.basename(f.name)
                self._t = TimedOutput(interval=10)
                self._read = 0

            def __getattr__(self, item):
                return getattr(self._f, item)

            def read(self, n=-1):
                read_bytes = self._f.read(n)
                self._read += len(read_bytes)
                self._t.info(f"{ref}: Uploading {self._name}: {int(self._read*100/self._total)}%")
                return read_bytes

        filesize = os.path.getsize(abs_path)
        with open(abs_path, mode='rb') as file_handler:
            big_file = filesize > 100000000  # 100 MB
            file_handler = FileProgress(file_handler, filesize) if big_file else file_handler
            try:
                response = self._requester.put(url, data=file_handler, verify=self._verify_ssl,
                                               headers=headers, auth=auth,
                                               source_credentials=self._source_credentials)
                self._handle_400_response(response, auth)
                response.raise_for_status()  # Raise HTTPError for bad http response status
                return response
            except ConanException:
                raise
            except Exception as exc:
                raise ConanException(exc)
