/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

#pragma once

#include <glib.h>

/**
 * SECTION: version
 * @section_id: version-macros
 * @title: Version related macros
 * @include: gandiva-glib/gandiva-glib.h
 *
 * Gandiva GLib provides macros that can be used by C pre-processor.
 * They are useful to check version related things at compile time.
 */

/**
 * GGANDIVA_VERSION_MAJOR:
 *
 * The major version.
 *
 * Since: 1.0.0
 */
#define GGANDIVA_VERSION_MAJOR (15)

/**
 * GGANDIVA_VERSION_MINOR:
 *
 * The minor version.
 *
 * Since: 1.0.0
 */
#define GGANDIVA_VERSION_MINOR (0)

/**
 * GGANDIVA_VERSION_MICRO:
 *
 * The micro version.
 *
 * Since: 1.0.0
 */
#define GGANDIVA_VERSION_MICRO (1)

/**
 * GGANDIVA_VERSION_TAG:
 *
 * The version tag. Normally, it's an empty string. It's "SNAPSHOT"
 * for snapshot version.
 *
 * Since: 1.0.0
 */
#define GGANDIVA_VERSION_TAG   ""

/**
 * GGANDIVA_VERSION_CHECK:
 * @major: A major version to check for.
 * @minor: A minor version to check for.
 * @micro: A micro version to check for.
 *
 * You can use this macro in C pre-processor.
 *
 * Returns: %TRUE if the compile time Gandiva GLib version is the
 *   same as or newer than the passed version, %FALSE otherwise.
 *
 * Since: 1.0.0
 */
#define GGANDIVA_VERSION_CHECK(major, minor, micro)     \
  (GGANDIVA_MAJOR_VERSION > (major) ||                  \
   (GGANDIVA_MAJOR_VERSION == (major) &&                \
    GGANDIVA_MINOR_VERSION > (minor)) ||                \
   (GGANDIVA_MAJOR_VERSION == (major) &&                \
    GGANDIVA_MINOR_VERSION == (minor) &&                \
    GGANDIVA_MICRO_VERSION >= (micro)))

/**
 * GGANDIVA_DISABLE_DEPRECATION_WARNINGS:
 *
 * If this macro is defined, no deprecated warnings are produced.
 *
 * You must define this macro before including the
 * gandiva-glib/gandiva-glib.h header.
 *
 * Since: 1.0.0
 */

#ifdef GGANDIVA_DISABLE_DEPRECATION_WARNINGS
#  define GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_FOR(function)
#  define GGANDIVA_UNAVAILABLE(major, minor)
#else
#  define GGANDIVA_DEPRECATED G_DEPRECATED
#  define GGANDIVA_DEPRECATED_FOR(function) G_DEPRECATED_FOR(function)
#  define GGANDIVA_UNAVAILABLE(major, minor) G_UNAVAILABLE(major, minor)
#endif

/**
 * GGANDIVA_VERSION_1_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.0.0
 */
#define GGANDIVA_VERSION_1_0 G_ENCODE_VERSION(1, 0)

/**
 * GGANDIVA_VERSION_4_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 4.0.0
 */
#define GGANDIVA_VERSION_4_0 G_ENCODE_VERSION(4, 0)

/**
 * GGANDIVA_VERSION_MIN_REQUIRED:
 *
 * You can use this macro for compile time API version check.
 *
 * This macro value must be one of the predefined version macros such
 * as %GGANDIVA_VERSION_1_0.
 *
 * If you use any functions that is defined by newer version than
 * %GGANDIVA_VERSION_MIN_REQUIRED, deprecated warnings are produced at
 * compile time.
 *
 * You must define this macro before including the
 * gandiva-glib/gandiva-glib.h header.
 *
 * Since: 1.0.0
 */
#ifndef GGANDIVA_VERSION_MIN_REQUIRED
#  define GGANDIVA_VERSION_MIN_REQUIRED                                 \
  G_ENCODE_VERSION(GGANDIVA_VERSION_MAJOR, GGANDIVA_VERSION_MINOR)
#endif

/**
 * GGANDIVA_VERSION_MAX_ALLOWED:
 *
 * You can use this macro for compile time API version check.
 *
 * This macro value must be one of the predefined version macros such
 * as %GGANDIVA_VERSION_1_0.
 *
 * If you use any functions that is defined by newer version than
 * %GGANDIVA_VERSION_MAX_ALLOWED, deprecated warnings are produced at
 * compile time.
 *
 * You must define this macro before including the
 * gandiva-glib/gandiva-glib.h header.
 *
 * Since: 1.0.0
 */
#ifndef GGANDIVA_VERSION_MAX_ALLOWED
#  define GGANDIVA_VERSION_MAX_ALLOWED                                  \
  G_ENCODE_VERSION(GGANDIVA_VERSION_MAJOR, GGANDIVA_VERSION_MINOR)
#endif


#define GGANDIVA_AVAILABLE_IN_ALL

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_4_0
#  define GGANDIVA_DEPRECATED_IN_4_0                GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_4_0_FOR(function)  GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_4_0
#  define GGANDIVA_DEPRECATED_IN_4_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_4_0
#  define GGANDIVA_AVAILABLE_IN_4_0 GGANDIVA_UNAVAILABLE(4, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_4_0
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_1_0
#  define GGANDIVA_DEPRECATED_IN_1_0                GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_1_0_FOR(function)  GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_1_0
#  define GGANDIVA_DEPRECATED_IN_1_0_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_1_0
#  define GGANDIVA_AVAILABLE_IN_1_0 GGANDIVA_UNAVAILABLE(1, 0)
#else
#  define GGANDIVA_AVAILABLE_IN_1_0
#endif

#if GGANDIVA_VERSION_MIN_REQUIRED >= GGANDIVA_VERSION_0_17
#  define GGANDIVA_DEPRECATED_IN_0_17                GGANDIVA_DEPRECATED
#  define GGANDIVA_DEPRECATED_IN_0_17_FOR(function)  GGANDIVA_DEPRECATED_FOR(function)
#else
#  define GGANDIVA_DEPRECATED_IN_0_17
#  define GGANDIVA_DEPRECATED_IN_0_17_FOR(function)
#endif

#if GGANDIVA_VERSION_MAX_ALLOWED < GGANDIVA_VERSION_0_17
#  define GGANDIVA_AVAILABLE_IN_0_17 GGANDIVA_UNAVAILABLE(0, 17)
#else
#  define GGANDIVA_AVAILABLE_IN_0_17
#endif
