--
--  Copyright (C) 2014-2022, AdaCore
--  SPDX-License-Identifier: Apache-2.0
--

with Ada.Containers;                  use Ada.Containers;
with Ada.Containers.Hashed_Maps;
with Ada.Containers.Vectors;
with Ada.Directories;
with Ada.Exceptions;
with Ada.Finalization;
with Ada.Strings.Unbounded;           use Ada.Strings.Unbounded;
with Ada.Strings.Wide_Wide_Unbounded; use Ada.Strings.Wide_Wide_Unbounded;
pragma Warnings (Off, "internal");
with Ada.Strings.Wide_Wide_Unbounded.Aux;
use Ada.Strings.Wide_Wide_Unbounded.Aux;
pragma Warnings (On, "internal");
with Ada.Text_IO;                     use Ada.Text_IO;
with Ada.Unchecked_Conversion;
with Ada.Unchecked_Deallocation;
with System;
with GNAT.Task_Lock;
with GNAT.Traceback.Symbolic;
with GNATCOLL.Traces;
with Langkit_Support.Adalog.Debug;
with Langkit_Support.Generic_API.Analysis;
with Langkit_Support.Generic_API.Introspection;
with Langkit_Support.Hashes; use Langkit_Support.Hashes;
with Langkit_Support.Images; use Langkit_Support.Images;
with Langkit_Support.Names;  use Langkit_Support.Names;
with Langkit_Support.Relative_Get;
with Libadalang.Private_Converters;
use Libadalang.Private_Converters;
pragma Warnings (Off, "referenced");
          with Ada.Containers.Hashed_Sets;
          with Libadalang.Env_Hooks;
            use Libadalang.Env_Hooks;
          with Libadalang.Implementation.Extensions;
          with Libadalang.Implementation.Extensions;
            use Libadalang.Implementation.Extensions;
          with Libadalang.Internal_Default_Provider;
          with Libadalang.Sources;
pragma Warnings (On, "referenced");
package body Libadalang.Implementation is
   use Precomputed_Symbols;
   pragma Warnings (Off, "has no effect");
   use Solver;
   pragma Warnings (On, "has no effect");
   package Context_Vectors is new Ada.Containers.Vectors
     (Index_Type   => Positive,
      Element_Type => Internal_Context);
   type Contexts_Destructor is limited
      new Ada.Finalization.Limited_Controlled with null record;
   overriding procedure Finalize (CD : in out Contexts_Destructor);
   --  Helper to destroy all contexts when terminating the process
   package Context_Pool is
      procedure Acquire (Context : out Internal_Context)
         with Post => Context /= null;
      --  If a context is free for reuse, increment its serial number and
      --  return it. Otherwise, allocate a new one. In any case, this does not
      --  initialize it, except for the Serial_Number field.
      procedure Release (Context : in out Internal_Context)
         with Pre  => Context /= null,
              Post => Context = null;
      --  Tag Context as free for reuse and set it to null
      procedure Free;
      --  Free all contexts in this pool. Intended to be called only when the
      --  process is terminating, to avoid reported memory leaks.
   private
      Available : Context_Vectors.Vector;
      --  List of allocated contexts that can be re-used right now
      CD : Contexts_Destructor with Unreferenced;
      --  Singleton whose only purpose is to free all contexts in Available
      --  when finalized.
   end Context_Pool;
   procedure Register_Destroyable_Helper
     (Unit    : Internal_Unit;
      Object  : System.Address;
      Destroy : Destroy_Procedure);
   --  Common underlying implementation for Register_Destroyable_Gen
   pragma Warnings (Off, "referenced");
   function Construct_Entity_Array
     (V : AST_Envs.Entity_Vectors.Vector) return Internal_Entity_Array_Access;
   pragma Warnings (On, "referenced");
   procedure Reset_Envs_Caches (Unit : Internal_Unit);
   --  Reset the env caches of all lexical environments created for ``Unit``
   procedure Destroy (Env : in out Lexical_Env_Access);
   function Snaps_At_Start (Self : Bare_Ada_Node) return Boolean;
   function Snaps_At_End (Self : Bare_Ada_Node) return Boolean;
   --  Those maps are used to give unique ids to lexical envs while pretty
   --  printing them.
   package Address_To_Id_Maps is new Ada.Containers.Hashed_Maps
     (Lexical_Env, Integer, Hash, "=");
   type Dump_Lexical_Env_State is record
      Env_Ids : Address_To_Id_Maps.Map;
      --  Mapping: Lexical_Env -> Integer, used to remember which unique Ids we
      --  assigned to the lexical environments we found.
      Next_Id : Positive := 1;
      --  Id to assign to the next unknown lexical environment
      Root_Env : Lexical_Env;
      --  Lexical environment we consider a root (this is the Root_Scope from
      --  the current analysis context), or null if unknown.
   end record;
   --  Holder for the state of lexical environment dumpers
   function Get_Env_Id
     (E : Lexical_Env; State : in out Dump_Lexical_Env_State) return String;
   --  If E is known, return its unique Id from State. Otherwise, assign it a
   --  new unique Id and return it.
   procedure Print
     (Node        : Langkit_Support.Generic_API.Analysis.Lk_Node;
      Show_Slocs  : Boolean;
      Line_Prefix : String := "");
   --  Helper for the public overload, but working on the generic API node type
   ------------------------
   -- Precomputed_Symbol --
   ------------------------
   pragma Warnings (Off, "referenced");
   function Precomputed_Symbol
     (Index : Precomputed_Symbol_Index) return Text_Type is
   pragma Warnings (On, "referenced");
   begin
         declare
            Raw_Text : constant Text_Type := (case Index is
               when Precomputed_Sym => """&""",
               when Precomputed_Sym_1 => """&""""+""",
               when Precomputed_Sym_10 => """=""",
               when Precomputed_Sym_11 => """>""",
               when Precomputed_Sym_12 => """>=""",
               when Precomputed_Sym_2 => """*""",
               when Precomputed_Sym_20 => "<<>>",
               when Precomputed_Sym_3 => """**""",
               when Precomputed_Sym_4 => """+""",
               when Precomputed_Sym_5 => """-""",
               when Precomputed_Sym_6 => """/""",
               when Precomputed_Sym_7 => """/=""",
               when Precomputed_Sym_8 => """<""",
               when Precomputed_Sym_9 => """<=""",
               when Precomputed_Sym_Abort => "abort",
               when Precomputed_Sym_Abort_Signal => "Abort_Signal",
               when Precomputed_Sym_Abort_Signal_315 => "abort_signal_",
               when Precomputed_Sym_Abs => """abs""",
               when Precomputed_Sym_Abs_316 => "abs",
               when Precomputed_Sym_Abstract => "abstract",
               when Precomputed_Sym_Abstract_State => "Abstract_State",
               when Precomputed_Sym_Accept => "accept",
               when Precomputed_Sym_Access => "Access",
               when Precomputed_Sym_Access_319 => "access",
               when Precomputed_Sym_Ada => "Ada",
               when Precomputed_Sym_Ada_320 => "ada",
               when Precomputed_Sym_Add_Named => "Add_Named",
               when Precomputed_Sym_Add_Unnamed => "Add_Unnamed",
               when Precomputed_Sym_Address => "Address",
               when Precomputed_Sym_Address_Size => "Address_Size",
               when Precomputed_Sym_Adjacent => "Adjacent",
               when Precomputed_Sym_Adjacent_321 => "adjacent",
               when Precomputed_Sym_Aft => "Aft",
               when Precomputed_Sym_Aggregate => "Aggregate",
               when Precomputed_Sym_Aliased => "aliased",
               when Precomputed_Sym_Alignment => "Alignment",
               when Precomputed_Sym_All => "all",
               when Precomputed_Sym_And => """and""",
               when Precomputed_Sym_And_324 => "and",
               when Precomputed_Sym_Annotate => "Annotate",
               when Precomputed_Sym_Array => "array",
               when Precomputed_Sym_Asm_Input => "Asm_Input",
               when Precomputed_Sym_Asm_Input_326 => "asm_input",
               when Precomputed_Sym_Asm_Input_Operand => "Asm_Input_Operand",
               when Precomputed_Sym_Asm_Output => "Asm_Output",
               when Precomputed_Sym_Asm_Output_327 => "asm_output",
               when Precomputed_Sym_Asm_Output_Operand => "Asm_Output_Operand",
               when Precomputed_Sym_Assert => "Assert",
               when Precomputed_Sym_Assert_And_Cut => "Assert_And_Cut",
               when Precomputed_Sym_Assume => "Assume",
               when Precomputed_Sym_Asynchronous => "Asynchronous",
               when Precomputed_Sym_At => "at",
               when Precomputed_Sym_Atomic => "Atomic",
               when Precomputed_Sym_Atomic_Components => "Atomic_Components",
               when Precomputed_Sym_Attach_Handler => "Attach_Handler",
               when Precomputed_Sym_Aux_Dec => "Aux_DEC",
               when Precomputed_Sym_Base => "Base",
               when Precomputed_Sym_Begin => "begin",
               when Precomputed_Sym_Big_Integer => "Big_Integer",
               when Precomputed_Sym_Big_Integers => "Big_Integers",
               when Precomputed_Sym_Big_Numbers => "Big_Numbers",
               when Precomputed_Sym_Bit_Order => "Bit_Order",
               when Precomputed_Sym_Body => "body",
               when Precomputed_Sym_Boolean => "Boolean",
               when Precomputed_Sym_Buffer => "Buffer",
               when Precomputed_Sym_By_Entry => "By_Entry",
               when Precomputed_Sym_Callable => "Callable",
               when Precomputed_Sym_Caller => "Caller",
               when Precomputed_Sym_Case => "case",
               when Precomputed_Sym_Ceiling => "Ceiling",
               when Precomputed_Sym_Ceiling_332 => "ceiling",
               when Precomputed_Sym_Character => "Character",
               when Precomputed_Sym_Class => "Class",
               when Precomputed_Sym_Code_Address => "Code_Address",
               when Precomputed_Sym_Compile_Time_Error => "Compile_Time_Error",
               when Precomputed_Sym_Compile_Time_Warning => "Compile_Time_Warning",
               when Precomputed_Sym_Component_Size => "Component_Size",
               when Precomputed_Sym_Compose => "Compose",
               when Precomputed_Sym_Compose_333 => "compose",
               when Precomputed_Sym_Constant => "constant",
               when Precomputed_Sym_Constant_Indexing => "Constant_Indexing",
               when Precomputed_Sym_Constrained => "Constrained",
               when Precomputed_Sym_Contract_Cases => "Contract_Cases",
               when Precomputed_Sym_Convention => "Convention",
               when Precomputed_Sym_Copy_Sign => "Copy_Sign",
               when Precomputed_Sym_Copy_Sign_335 => "copy_sign",
               when Precomputed_Sym_Count => "Count",
               when Precomputed_Sym_Cursor => "Cursor",
               when Precomputed_Sym_Debug => "Debug",
               when Precomputed_Sym_Declare => "declare",
               when Precomputed_Sym_Decreases => "Decreases",
               when Precomputed_Sym_Default_Bit_Order => "Default_Bit_Order",
               when Precomputed_Sym_Default_Initial_Condition => "Default_Initial_Condition",
               when Precomputed_Sym_Default_Iterator => "Default_Iterator",
               when Precomputed_Sym_Default_Scalar_Storage_Order => "Default_Scalar_Storage_Order",
               when Precomputed_Sym_Definite => "Definite",
               when Precomputed_Sym_Delay => "delay",
               when Precomputed_Sym_Delta => "Delta",
               when Precomputed_Sym_Delta_338 => "delta",
               when Precomputed_Sym_Denorm => "Denorm",
               when Precomputed_Sym_Depends => "Depends",
               when Precomputed_Sym_Deref => "Deref",
               when Precomputed_Sym_Descriptor_Size => "Descriptor_Size",
               when Precomputed_Sym_Digits => "Digits",
               when Precomputed_Sym_Digits_339 => "digits",
               when Precomputed_Sym_Discard_Names => "Discard_Names",
               when Precomputed_Sym_Do => "do",
               when Precomputed_Sym_Dummy => "__dummy",
               when Precomputed_Sym_Duration => "Duration",
               when Precomputed_Sym_Dynamic_Predicate => "Dynamic_Predicate",
               when Precomputed_Sym_Elaborate_Body => "Elaborate_Body",
               when Precomputed_Sym_Element => "Element",
               when Precomputed_Sym_Else => "else",
               when Precomputed_Sym_Elsif => "elsif",
               when Precomputed_Sym_Enabled => "Enabled",
               when Precomputed_Sym_End => "end",
               when Precomputed_Sym_Ensures => "Ensures",
               when Precomputed_Sym_Entity => "Entity",
               when Precomputed_Sym_Entry => "entry",
               when Precomputed_Sym_Enum_Rep => "Enum_Rep",
               when Precomputed_Sym_Enum_Rep_345 => "enum_rep",
               when Precomputed_Sym_Enum_Val => "Enum_Val",
               when Precomputed_Sym_Enum_Val_346 => "enum_val",
               when Precomputed_Sym_Epsilon => "Epsilon",
               when Precomputed_Sym_Exception => "exception",
               when Precomputed_Sym_Exception_Id => "Exception_Id",
               when Precomputed_Sym_Exception_Occurrence => "Exception_Occurrence",
               when Precomputed_Sym_Exceptions => "Exceptions",
               when Precomputed_Sym_Exit => "exit",
               when Precomputed_Sym_Exponent => "Exponent",
               when Precomputed_Sym_Exponent_349 => "exponent",
               when Precomputed_Sym_Export => "Export",
               when Precomputed_Sym_External_Tag => "External_Tag",
               when Precomputed_Sym_Fast_Math => "Fast_Math",
               when Precomputed_Sym_Finalization_Size => "Finalization_Size",
               when Precomputed_Sym_First => "First",
               when Precomputed_Sym_First_Bit => "First_Bit",
               when Precomputed_Sym_First_Valid => "First_Valid",
               when Precomputed_Sym_Fixed_Value => "Fixed_Value",
               when Precomputed_Sym_Fixed_Value_350 => "fixed_value",
               when Precomputed_Sym_Floor => "Floor",
               when Precomputed_Sym_Floor_351 => "floor",
               when Precomputed_Sym_For => "for",
               when Precomputed_Sym_Fore => "Fore",
               when Precomputed_Sym_Fraction => "Fraction",
               when Precomputed_Sym_Fraction_353 => "fraction",
               when Precomputed_Sym_Function => "function",
               when Precomputed_Sym_Generic => "generic",
               when Precomputed_Sym_Ghost => "Ghost",
               when Precomputed_Sym_Global => "Global",
               when Precomputed_Sym_Gnat => "GNAT",
               when Precomputed_Sym_Gnatprove => "GNATprove",
               when Precomputed_Sym_Goto => "goto",
               when Precomputed_Sym_Has_Access_Values => "Has_Access_Values",
               when Precomputed_Sym_Has_Discriminants => "Has_Discriminants",
               when Precomputed_Sym_Has_Same_Storage => "Has_Same_Storage",
               when Precomputed_Sym_Has_Tagged_Values => "Has_Tagged_Values",
               when Precomputed_Sym_Identity => "Identity",
               when Precomputed_Sym_If => "if",
               when Precomputed_Sym_Image => "Image",
               when Precomputed_Sym_Image_358 => "image",
               when Precomputed_Sym_Img => "Img",
               when Precomputed_Sym_Implicit_Dereference => "Implicit_Dereference",
               when Precomputed_Sym_Import => "Import",
               when Precomputed_Sym_Import_Function => "Import_Function",
               when Precomputed_Sym_Import_Procedure => "Import_Procedure",
               when Precomputed_Sym_In => "in",
               when Precomputed_Sym_Increases => "Increases",
               when Precomputed_Sym_Independent => "Independent",
               when Precomputed_Sym_Independent_Components => "Independent_Components",
               when Precomputed_Sym_Index => "Index",
               when Precomputed_Sym_Initial_Condition => "Initial_Condition",
               when Precomputed_Sym_Initialized => "Initialized",
               when Precomputed_Sym_Inline => "Inline",
               when Precomputed_Sym_Input => "Input",
               when Precomputed_Sym_Input_360 => "input",
               when Precomputed_Sym_Integer => "Integer",
               when Precomputed_Sym_Integer_Literal => "Integer_Literal",
               when Precomputed_Sym_Integer_Value => "Integer_Value",
               when Precomputed_Sym_Integer_Value_361 => "integer_value",
               when Precomputed_Sym_Interface => "Interface",
               when Precomputed_Sym_Interface_362 => "interface",
               when Precomputed_Sym_Interrupt_Handler => "Interrupt_Handler",
               when Precomputed_Sym_Invalid_Value => "Invalid_Value",
               when Precomputed_Sym_Invariant => "Invariant",
               when Precomputed_Sym_Invariant_Class => "Invariant'Class",
               when Precomputed_Sym_Is => "is",
               when Precomputed_Sym_Iterable => "Iterable",
               when Precomputed_Sym_Iterator_Element => "Iterator_Element",
               when Precomputed_Sym_Iterator_Interfaces => "Iterator_Interfaces",
               when Precomputed_Sym_Large => "Large",
               when Precomputed_Sym_Last => "Last",
               when Precomputed_Sym_Last_Bit => "Last_Bit",
               when Precomputed_Sym_Last_Valid => "Last_Valid",
               when Precomputed_Sym_Leading_Part => "Leading_Part",
               when Precomputed_Sym_Leading_Part_364 => "leading_part",
               when Precomputed_Sym_Left => "left",
               when Precomputed_Sym_Length => "Length",
               when Precomputed_Sym_Library_Level => "Library_Level",
               when Precomputed_Sym_Limited => "limited",
               when Precomputed_Sym_Loop => "loop",
               when Precomputed_Sym_Loop_Entry => "Loop_Entry",
               when Precomputed_Sym_Loop_Invariant => "Loop_Invariant",
               when Precomputed_Sym_Machine => "Machine",
               when Precomputed_Sym_Machine_368 => "machine",
               when Precomputed_Sym_Machine_Code => "Machine_Code",
               when Precomputed_Sym_Machine_Emax => "Machine_Emax",
               when Precomputed_Sym_Machine_Emin => "Machine_Emin",
               when Precomputed_Sym_Machine_Mantissa => "Machine_Mantissa",
               when Precomputed_Sym_Machine_Overflows => "Machine_Overflows",
               when Precomputed_Sym_Machine_Radix => "Machine_Radix",
               when Precomputed_Sym_Machine_Rounding => "Machine_Rounding",
               when Precomputed_Sym_Machine_Rounding_369 => "machine_rounding",
               when Precomputed_Sym_Machine_Rounds => "Machine_Rounds",
               when Precomputed_Sym_Mantissa => "Mantissa",
               when Precomputed_Sym_Max => "Max",
               when Precomputed_Sym_Max_370 => "max",
               when Precomputed_Sym_Max_Alignment_For_Allocation => "Max_Alignment_For_Allocation",
               when Precomputed_Sym_Max_Integer_Size => "Max_Integer_Size",
               when Precomputed_Sym_Max_Size_In_Storage_Elements => "Max_Size_In_Storage_Elements",
               when Precomputed_Sym_Maximum_Alignment => "Maximum_Alignment",
               when Precomputed_Sym_Mechanism_Code => "Mechanism_Code",
               when Precomputed_Sym_Min => "Min",
               when Precomputed_Sym_Min_371 => "min",
               when Precomputed_Sym_Mod => """mod""",
               when Precomputed_Sym_Mod_170 => "Mod",
               when Precomputed_Sym_Mod_372 => "mod",
               when Precomputed_Sym_Model => "Model",
               when Precomputed_Sym_Model_373 => "model",
               when Precomputed_Sym_Model_Emin => "Model_Emin",
               when Precomputed_Sym_Model_Epsilon => "Model_Epsilon",
               when Precomputed_Sym_Model_Mantissa => "Model_Mantissa",
               when Precomputed_Sym_Model_Of => "Model_Of",
               when Precomputed_Sym_Model_Small => "Model_Small",
               when Precomputed_Sym_Modulus => "Modulus",
               when Precomputed_Sym_New => "new",
               when Precomputed_Sym_Nextpart => "__nextpart",
               when Precomputed_Sym_No_Elaboration_Code => "No_Elaboration_Code",
               when Precomputed_Sym_No_Elaboration_Code_All => "No_Elaboration_Code_All",
               when Precomputed_Sym_No_Return => "No_Return",
               when Precomputed_Sym_Not => """not""",
               when Precomputed_Sym_Not_375 => "not",
               when Precomputed_Sym_Null => "null",
               when Precomputed_Sym_Numerics => "Numerics",
               when Precomputed_Sym_Object_Size => "Object_Size",
               when Precomputed_Sym_Obsolescent => "Obsolescent",
               when Precomputed_Sym_Of => "of",
               when Precomputed_Sym_Off => "Off",
               when Precomputed_Sym_Old => "Old",
               when Precomputed_Sym_On => "On",
               when Precomputed_Sym_Or => """or""",
               when Precomputed_Sym_Or_378 => "or",
               when Precomputed_Sym_Others => "others",
               when Precomputed_Sym_Out => "out",
               when Precomputed_Sym_Output => "Output",
               when Precomputed_Sym_Output_381 => "output",
               when Precomputed_Sym_Overlaps_Storage => "Overlaps_Storage",
               when Precomputed_Sym_Overriding => "overriding",
               when Precomputed_Sym_Pack => "Pack",
               when Precomputed_Sym_Package => "package",
               when Precomputed_Sym_Passed_By_Reference => "Passed_By_Reference",
               when Precomputed_Sym_Pos => "Pos",
               when Precomputed_Sym_Pos_384 => "pos",
               when Precomputed_Sym_Position => "Position",
               when Precomputed_Sym_Post => "Post",
               when Precomputed_Sym_Post_Class => "Post'Class",
               when Precomputed_Sym_Postcondition => "Postcondition",
               when Precomputed_Sym_Postcondition_Class => "Postcondition'Class",
               when Precomputed_Sym_Pragma => "pragma",
               when Precomputed_Sym_Pre => "Pre",
               when Precomputed_Sym_Pre_Class => "Pre'Class",
               when Precomputed_Sym_Precondition => "Precondition",
               when Precomputed_Sym_Precondition_Class => "Precondition'Class",
               when Precomputed_Sym_Pred => "Pred",
               when Precomputed_Sym_Pred_386 => "pred",
               when Precomputed_Sym_Predicate => "Predicate",
               when Precomputed_Sym_Predicate_Failure => "Predicate_Failure",
               when Precomputed_Sym_Preelaborable_Initialization => "Preelaborable_Initialization",
               when Precomputed_Sym_Preelaborate => "Preelaborate",
               when Precomputed_Sym_Private => "private",
               when Precomputed_Sym_Privatepart => "__privatepart",
               when Precomputed_Sym_Procedure => "procedure",
               when Precomputed_Sym_Protected => "protected",
               when Precomputed_Sym_Pure => "Pure",
               when Precomputed_Sym_Put_Image => "Put_Image",
               when Precomputed_Sym_Put_Image_390 => "put_image",
               when Precomputed_Sym_Raise => "raise",
               when Precomputed_Sym_Range => "Range",
               when Precomputed_Sym_Range_392 => "range",
               when Precomputed_Sym_Range_Length => "Range_Length",
               when Precomputed_Sym_Read => "Read",
               when Precomputed_Sym_Read_393 => "read",
               when Precomputed_Sym_Real_Literal => "Real_Literal",
               when Precomputed_Sym_Record => "record",
               when Precomputed_Sym_Reduce => "Reduce",
               when Precomputed_Sym_Refined_Depends => "Refined_Depends",
               when Precomputed_Sym_Refined_Global => "Refined_Global",
               when Precomputed_Sym_Refined_Post => "Refined_Post",
               when Precomputed_Sym_Refined_State => "Refined_State",
               when Precomputed_Sym_Rem => """rem""",
               when Precomputed_Sym_Rem_395 => "rem",
               when Precomputed_Sym_Remainder => "Remainder",
               when Precomputed_Sym_Remainder_396 => "remainder",
               when Precomputed_Sym_Remote_Call_Interface => "Remote_Call_Interface",
               when Precomputed_Sym_Remote_Types => "Remote_Types",
               when Precomputed_Sym_Renames => "renames",
               when Precomputed_Sym_Requeue => "requeue",
               when Precomputed_Sym_Requires => "Requires",
               when Precomputed_Sym_Restrictions => "Restrictions",
               when Precomputed_Sym_Result => "Result",
               when Precomputed_Sym_Return => "return",
               when Precomputed_Sym_Reverse => "reverse",
               when Precomputed_Sym_Right => "right",
               when Precomputed_Sym_Root_Buffer_Type => "Root_Buffer_Type",
               when Precomputed_Sym_Root_Integer => "root_integer",
               when Precomputed_Sym_Root_Real => "root_real",
               when Precomputed_Sym_Root_Storage_Pool => "Root_Storage_Pool",
               when Precomputed_Sym_Root_Stream_Type => "Root_Stream_Type",
               when Precomputed_Sym_Root_Types => "root_types_",
               when Precomputed_Sym_Round => "Round",
               when Precomputed_Sym_Round_405 => "round",
               when Precomputed_Sym_Rounding => "Rounding",
               when Precomputed_Sym_Rounding_406 => "rounding",
               when Precomputed_Sym_S => "S",
               when Precomputed_Sym_Safe_First => "Safe_First",
               when Precomputed_Sym_Safe_Large => "Safe_Large",
               when Precomputed_Sym_Safe_Last => "Safe_Last",
               when Precomputed_Sym_Safe_Small => "Safe_Small",
               when Precomputed_Sym_Scalar_Storage_Order => "Scalar_Storage_Order",
               when Precomputed_Sym_Scale => "Scale",
               when Precomputed_Sym_Scaling => "Scaling",
               when Precomputed_Sym_Scaling_407 => "scaling",
               when Precomputed_Sym_Select => "select",
               when Precomputed_Sym_Separate => "separate",
               when Precomputed_Sym_Shared_Passive => "Shared_Passive",
               when Precomputed_Sym_Signed_Zeros => "Signed_Zeros",
               when Precomputed_Sym_Size => "Size",
               when Precomputed_Sym_Skip_Flow_And_Proof => "Skip_Flow_And_Proof",
               when Precomputed_Sym_Skip_Proof => "Skip_Proof",
               when Precomputed_Sym_Small => "Small",
               when Precomputed_Sym_Small_Denominator => "Small_Denominator",
               when Precomputed_Sym_Small_Numerator => "Small_Numerator",
               when Precomputed_Sym_Some => "some",
               when Precomputed_Sym_Spark_Mode => "SPARK_Mode",
               when Precomputed_Sym_Stable_Properties => "Stable_Properties",
               when Precomputed_Sym_Standard => "Standard",
               when Precomputed_Sym_Standard_411 => "standard",
               when Precomputed_Sym_Static_Predicate => "Static_Predicate",
               when Precomputed_Sym_Storage_Elements => "Storage_Elements",
               when Precomputed_Sym_Storage_Pool => "Storage_Pool",
               when Precomputed_Sym_Storage_Pools => "Storage_Pools",
               when Precomputed_Sym_Storage_Size => "Storage_Size",
               when Precomputed_Sym_Storage_Unit => "Storage_Unit",
               when Precomputed_Sym_Stream_Size => "Stream_Size",
               when Precomputed_Sym_Streams => "Streams",
               when Precomputed_Sym_String => "String",
               when Precomputed_Sym_String_Literal => "String_Literal",
               when Precomputed_Sym_Strings => "Strings",
               when Precomputed_Sym_Structural => "Structural",
               when Precomputed_Sym_Style_Checks => "Style_Checks",
               when Precomputed_Sym_Subprogram_Variant => "Subprogram_Variant",
               when Precomputed_Sym_Subtype => "subtype",
               when Precomputed_Sym_Succ => "Succ",
               when Precomputed_Sym_Succ_413 => "succ",
               when Precomputed_Sym_Synchronization => "Synchronization",
               when Precomputed_Sym_Synchronized => "synchronized",
               when Precomputed_Sym_System => "System",
               when Precomputed_Sym_System_Allocator_Alignment => "System_Allocator_Alignment",
               when Precomputed_Sym_Tag => "Tag",
               when Precomputed_Sym_Tagged => "tagged",
               when Precomputed_Sym_Tags => "Tags",
               when Precomputed_Sym_Target_Name => "Target_Name",
               when Precomputed_Sym_Task => "task",
               when Precomputed_Sym_Task_Id => "Task_Id",
               when Precomputed_Sym_Task_Identification => "Task_Identification",
               when Precomputed_Sym_Terminate => "terminate",
               when Precomputed_Sym_Terminated => "Terminated",
               when Precomputed_Sym_Test_Case => "Test_Case",
               when Precomputed_Sym_Text_Buffers => "Text_Buffers",
               when Precomputed_Sym_Text_Io => "text_io",
               when Precomputed_Sym_Then => "then",
               when Precomputed_Sym_To_Address => "To_Address",
               when Precomputed_Sym_Truncation => "Truncation",
               when Precomputed_Sym_Truncation_420 => "truncation",
               when Precomputed_Sym_Type => "type",
               when Precomputed_Sym_Type_Class => "Type_Class",
               when Precomputed_Sym_Type_Invariant => "Type_Invariant",
               when Precomputed_Sym_Type_Invariant_Class => "Type_Invariant'Class",
               when Precomputed_Sym_Type_Key => "Type_Key",
               when Precomputed_Sym_Unbiased_Rounding => "Unbiased_Rounding",
               when Precomputed_Sym_Unbiased_Rounding_422 => "unbiased_rounding",
               when Precomputed_Sym_Unchecked_Access => "Unchecked_Access",
               when Precomputed_Sym_Unchecked_Union => "Unchecked_Union",
               when Precomputed_Sym_Unconstrained_Array => "Unconstrained_Array",
               when Precomputed_Sym_Universal_Fixed_Type => "Universal_Fixed_Type_",
               when Precomputed_Sym_Universal_Int_Type => "Universal_Int_Type_",
               when Precomputed_Sym_Universal_Real_Type => "Universal_Real_Type_",
               when Precomputed_Sym_Unreferenced => "Unreferenced",
               when Precomputed_Sym_Unrestricted_Access => "Unrestricted_Access",
               when Precomputed_Sym_Until => "until",
               when Precomputed_Sym_Update => "Update",
               when Precomputed_Sym_Use => "use",
               when Precomputed_Sym_Vads_Size => "VADS_Size",
               when Precomputed_Sym_Val => "Val",
               when Precomputed_Sym_Val_425 => "val",
               when Precomputed_Sym_Valid => "Valid",
               when Precomputed_Sym_Valid_Scalars => "Valid_Scalars",
               when Precomputed_Sym_Value => "Value",
               when Precomputed_Sym_Value_426 => "value",
               when Precomputed_Sym_Value_Size => "Value_Size",
               when Precomputed_Sym_Variable_Indexing => "Variable_Indexing",
               when Precomputed_Sym_Volatile => "Volatile",
               when Precomputed_Sym_Volatile_Components => "Volatile_Components",
               when Precomputed_Sym_Warnings => "Warnings",
               when Precomputed_Sym_Wchar_T_Size => "Wchar_T_Size",
               when Precomputed_Sym_When => "when",
               when Precomputed_Sym_While => "while",
               when Precomputed_Sym_Wide_Character => "Wide_Character",
               when Precomputed_Sym_Wide_Image => "Wide_Image",
               when Precomputed_Sym_Wide_Image_429 => "wide_image",
               when Precomputed_Sym_Wide_String => "Wide_String",
               when Precomputed_Sym_Wide_Text_Io => "wide_text_io",
               when Precomputed_Sym_Wide_Value => "Wide_Value",
               when Precomputed_Sym_Wide_Value_431 => "wide_value",
               when Precomputed_Sym_Wide_Wide_Character => "Wide_Wide_Character",
               when Precomputed_Sym_Wide_Wide_Image => "Wide_Wide_Image",
               when Precomputed_Sym_Wide_Wide_Image_432 => "wide_wide_image",
               when Precomputed_Sym_Wide_Wide_String => "Wide_Wide_String",
               when Precomputed_Sym_Wide_Wide_Text_Io => "wide_wide_text_io",
               when Precomputed_Sym_Wide_Wide_Value => "Wide_Wide_Value",
               when Precomputed_Sym_Wide_Wide_Value_434 => "wide_wide_value",
               when Precomputed_Sym_Width => "Width",
               when Precomputed_Sym_With => "with",
               when Precomputed_Sym_Word_Size => "Word_Size",
               when Precomputed_Sym_Write => "Write",
               when Precomputed_Sym_Write_436 => "write",
               when Precomputed_Sym_Xor => """xor""",
               when Precomputed_Sym_Xor_437 => "xor"
            );
            Symbol : constant Symbolization_Result :=
                  Libadalang.Sources.Canonicalize (Raw_Text)
            ;
         begin
            if Symbol.Success then
               return Symbol.Symbol;
            else
               raise Program_Error with
                 "Cannot canonicalize symbol literal: " & Image (Raw_Text);
            end if;
         end;
   end Precomputed_Symbol;
   ----------------------------
   -- Construct_Entity_Array --
   ----------------------------
   function Construct_Entity_Array
     (V : AST_Envs.Entity_Vectors.Vector) return Internal_Entity_Array_Access
   is
      Ret : Internal_Entity_Array_Access :=
        Create_Internal_Entity_Array (V.Length);
   begin
      for J in V.First_Index .. V.Last_Index loop
         Ret.Items (J) := V.Get (J);
      end loop;
      declare
         Tmp : AST_Envs.Entity_Vectors.Vector := V;
      begin
         Tmp.Destroy;
      end;
      return Ret;
   end Construct_Entity_Array;
   -----------
   -- Image --
   -----------
   function Image (Self : Symbol_Type) return String_Type is
   begin
      return Create_String (Image (Self));
   end Image;
   ------------------
   -- Context_Pool --
   ------------------
   package body Context_Pool is
      -------------
      -- Acquire --
      -------------
      procedure Acquire (Context : out Internal_Context) is
      begin
         GNAT.Task_Lock.Lock;
         if Available.Is_Empty then
            Context := new Analysis_Context_Type;
            Context.Serial_Number := 1;
         else
            Context := Available.Last_Element;
            Available.Delete_Last;
         end if;
         GNAT.Task_Lock.Unlock;
         Context.Initialized := False;
         Context.Ref_Count := 1;
      exception
         when others =>
            GNAT.Task_Lock.Unlock;
            raise;
      end Acquire;
      -------------
      -- Release --
      -------------
      procedure Release (Context : in out Internal_Context) is
      begin
         GNAT.Task_Lock.Lock;
         Available.Append (Context);
         Context.Serial_Number := Context.Serial_Number + 1;
         Context := null;
         GNAT.Task_Lock.Unlock;
      exception
         when others =>
            GNAT.Task_Lock.Unlock;
            raise;
      end Release;
      ----------
      -- Free --
      ----------
      procedure Free is
      begin
         GNAT.Task_Lock.Lock;
         for C of Available loop
            Free (C);
         end loop;
         GNAT.Task_Lock.Unlock;
      exception
         when others =>
            GNAT.Task_Lock.Unlock;
            raise;
      end Free;
   end Context_Pool;
   --------------
   -- Finalize --
   --------------
   overriding procedure Finalize (CD : in out Contexts_Destructor) is
      pragma Unreferenced (CD);
   begin
      Context_Pool.Free;
   end Finalize;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (File_Reader : in out Internal_File_Reader_Access) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Internal_File_Reader'Class, Internal_File_Reader_Access);
   begin
      if File_Reader /= null and then File_Reader.all.Dec_Ref then
         Destroy (File_Reader);
      end if;
   end Dec_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (Self : in out Internal_Event_Handler_Access) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Internal_Event_Handler'Class, Internal_Event_Handler_Access);
   begin
      if Self /= null and then Self.all.Dec_Ref then
         Destroy (Self);
      end if;
   end Dec_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (Provider : in out Internal_Unit_Provider_Access) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Internal_Unit_Provider'Class, Internal_Unit_Provider_Access);
   begin
      if Provider /= null and then Provider.all.Dec_Ref then
         Destroy (Provider);
      end if;
   end Dec_Ref;
   ----------------
   -- Get_Env_Id --
   ----------------
   function Get_Env_Id
     (E : Lexical_Env; State : in out Dump_Lexical_Env_State) return String
   is
      C        : Address_To_Id_Maps.Cursor;
      Inserted : Boolean;
   begin
      if E = Null_Lexical_Env then
         return "$null";
      elsif E = State.Root_Env then
         --  Insert root env with a special Id so that we only print it once
         State.Env_Ids.Insert (E, -1, C, Inserted);
         return "$root";
      end if;
      State.Env_Ids.Insert (E, State.Next_Id, C, Inserted);
      if Inserted then
         State.Next_Id := State.Next_Id + 1;
      end if;
      return '@' & Stripped_Image (Address_To_Id_Maps.Element (C));
   end Get_Env_Id;
   pragma Warnings (Off, "referenced");
   function To_Lookup_Kind_Type (K : Lookup_Kind) return Lookup_Kind_Type
   is
     (Lookup_Kind_Type'Val (Lookup_Kind'Pos (K)));
   pragma Warnings (On, "referenced");
   ----------------------
   -- Allocate_Context --
   ----------------------
   function Allocate_Context return Internal_Context is
   begin
      return Context : Internal_Context do
         Context_Pool.Acquire (Context);
      end return;
   end Allocate_Context;
   ------------------------
   -- Initialize_Context --
   ------------------------
   procedure Initialize_Context
     (Context        : Internal_Context;
      Charset        : String;
      File_Reader    : Internal_File_Reader_Access;
      Unit_Provider  : Internal_Unit_Provider_Access;
      Event_Handler  : Internal_Event_Handler_Access;
      With_Trivia    : Boolean;
      Tab_Stop       : Positive)
   is
      Actual_Charset : constant String :=
        (if Charset = "" then Default_Charset else Charset);
      Symbols        : constant Precomputed_Symbol_Table
        := Create_Symbol_Table;
   begin
      Context.Initialized := True;
      Context.Symbols := Symbol_Table (Symbols);
      Context.Charset := To_Unbounded_String (Actual_Charset);
      Context.Tab_Stop := Tab_Stop;
      Context.With_Trivia := With_Trivia;
      Context.Root_Scope := Create_Static_Lexical_Env
        (Parent    => Null_Lexical_Env,
         Node      => null,
         Sym_Table => Context.Symbols);
      --  Create a new ownership share for Event_Handler so that it lives at
      --  least as long as this analysis context.
      Context.Event_Handler := Event_Handler;
      if Context.Event_Handler /= null then
         Context.Event_Handler.Inc_Ref;
      end if;
      --  Create a new ownership share for File_Reader so that it lives at
      --  least as long as this analysis context.
      Context.File_Reader := File_Reader;
      if Context.File_Reader /= null then
         Context.File_Reader.Inc_Ref;
      end if;
      --  Create a new ownership share for Unit_Provider so that it lives at
      --  least as long as this analysis context.
      Context.Unit_Provider := Unit_Provider;
      if Context.Unit_Provider /= null then
         Context.Unit_Provider.Inc_Ref;
      end if;
         if Context.Unit_Provider = null then
            Context.Unit_Provider := Libadalang.Internal_Default_Provider.Create;
         end if;
      Initialize (Context.Parser);
      Context.Discard_Errors_In_Populate_Lexical_Env := True;
      Context.Logic_Resolution_Timeout :=
        Langkit_Support.Adalog.Default_Timeout_Ticks_Number;
      Context.In_Populate_Lexical_Env := False;
      Context.Cache_Version := 0;
      Context.Reparse_Cache_Version := 0;
      Context.Rewriting_Handle := No_Rewriting_Handle_Pointer;
      Context.Templates_Unit := No_Analysis_Unit;
      Context.Available_Rebindings := Env_Rebindings_Vectors.Empty_Vector;
Fetch_Standard (Context);
   end Initialize_Context;
   -----------------
   -- Create_Unit --
   -----------------
   function Create_Unit
     (Context             : Internal_Context;
      Normalized_Filename : Virtual_File;
      Charset             : String;
      Rule                : Grammar_Rule) return Internal_Unit
   is
      use Units_Maps;
      Unit : Internal_Unit;
   begin
      Unit := Create_Special_Unit
        (Context, Normalized_Filename, Charset, Rule);
      Context.Units.Insert (Normalized_Filename, Unit);
      return Unit;
   end Create_Unit;
   --------------
   -- Get_Unit --
   --------------
   function Get_Unit
     (Context           : Internal_Context;
      Filename, Charset : String;
      Reparse           : Boolean;
      Input             : Internal_Lexer_Input;
      Rule              : Grammar_Rule;
      Is_Internal       : Boolean := False) return Internal_Unit
   is
      use Units_Maps;
      Normalized_Filename : constant GNATCOLL.VFS.Virtual_File :=
         Normalized_Unit_Filename (Context, Filename);
      Cur     : constant Cursor :=
         Context.Units.Find (Normalized_Filename);
      Created : constant Boolean := Cur = No_Element;
      Unit    : Internal_Unit;
      Actual_Charset : Unbounded_String;
      Refined_Input  : Internal_Lexer_Input := Input;
   begin
      --  Determine which encoding to use. Use the Charset parameter (if
      --  provided), otherwise use the context-wide default.
      Actual_Charset := (if Charset'Length /= 0
                         then To_Unbounded_String (Charset)
                         else Context.Charset);
      if Refined_Input.Kind = File then
         Refined_Input.Filename := Normalized_Filename;
      end if;
      if Refined_Input.Kind in File | Bytes_Buffer then
         Refined_Input.Charset := Actual_Charset;
         --  Unless the caller requested a specific charset for this unit,
         --  allow the lexer to automatically discover the source file encoding
         --  before defaulting to the context-specific one. We do this trying
         --  to match a byte order mark.
         Refined_Input.Read_BOM := Charset'Length = 0;
      end if;
      --  Create the Internal_Unit if needed
      Unit :=
        (if Created
         then Create_Unit (Context, Normalized_Filename,
                           To_String (Actual_Charset), Rule)
         else Element (Cur));
      --  If an internal unit is requested, set the corresponding flag.
      --  Otherwise, make sure that the unit we return isn't internal.
      if Is_Internal then
         Unit.Is_Internal := True;
      end if;
      --  (Re)parse it if needed
      if Created or else Reparse then
         --  It is illegal to reparse an internal unit for public API users.
         --  Since public APIs do not allow to pass True to Is_Internal, we can
         --  check here that only the implementation can ask to reparse an
         --  internal unit.
         if Unit.Is_Internal and then not Is_Internal then
            raise Precondition_Failure with "cannot reparse an internal unit";
         end if;
         declare
            Reparsed : Reparsed_Unit;
         begin
            Do_Parsing (Unit, Refined_Input, Reparsed);
            Update_After_Reparse (Unit, Reparsed);
         end;
         --  Now that we have removed reparsed the unit, update its current
         --  charset.
         Unit.Charset := Actual_Charset;
      end if;
      if Context.Event_Handler /= null then
         Context.Event_Handler.Unit_Parsed_Callback
           (Context  => Context,
            Unit     => Unit,
            Reparsed => not Created and then Reparse);
      end if;
      return Unit;
   end Get_Unit;
   --------------
   -- Has_Unit --
   --------------
   function Has_Unit
     (Context : Internal_Context; Unit_Filename : String) return Boolean is
   begin
      return Context.Units.Contains
        (Normalized_Unit_Filename (Context, Unit_Filename));
   end Has_Unit;
   -------------------
   -- Get_From_File --
   -------------------
   function Get_From_File
     (Context  : Internal_Context;
      Filename : String;
      Charset  : String;
      Reparse  : Boolean;
      Rule     : Grammar_Rule) return Internal_Unit
   is
      Input : constant Internal_Lexer_Input :=
        (Kind     => File,
         Charset  => <>,
         Read_BOM => False,
         Filename => <>);
   begin
      if Reparse and then Has_Rewriting_Handle (Context) then
         raise Precondition_Failure with
            "cannot reparse during tree rewriting";
      end if;
      return Get_Unit (Context, Filename, Charset, Reparse, Input, Rule);
   end Get_From_File;
   ---------------------
   -- Get_From_Buffer --
   ---------------------
   function Get_From_Buffer
     (Context  : Internal_Context;
      Filename : String;
      Charset  : String;
      Buffer   : String;
      Rule     : Grammar_Rule) return Internal_Unit
   is
      Input : constant Internal_Lexer_Input :=
        (Kind        => Bytes_Buffer,
         Charset     => <>,
         Read_BOM    => False,
         Bytes       => Buffer'Address,
         Bytes_Count => Buffer'Length);
   begin
      if Has_Rewriting_Handle (Context) then
         raise Precondition_Failure with
            "cannot parse from buffer during tree rewriting";
      elsif Context.File_Reader /= null then
         raise Precondition_Failure with
            "cannot parse from buffer with a file reader";
      end if;
      return Get_Unit (Context, Filename, Charset, True, Input, Rule);
   end Get_From_Buffer;
   --------------------
   -- Get_With_Error --
   --------------------
   function Get_With_Error
     (Context  : Internal_Context;
      Filename : String;
      Error    : Text_Type;
      Charset  : String;
      Rule     : Grammar_Rule) return Internal_Unit
   is
      use Units_Maps;
      Normalized_Filename : constant Virtual_File :=
         Normalized_Unit_Filename (Context, Filename);
      Cur                 : constant Cursor :=
         Context.Units.Find (Normalized_Filename);
   begin
      if Cur = No_Element then
         declare
            Unit : constant Internal_Unit := Create_Unit
              (Context, Normalized_Filename, Charset, Rule);
         begin
            Append (Unit.Diagnostics, No_Source_Location_Range, Error);
            return Unit;
         end;
      else
         return Element (Cur);
      end if;
   end Get_With_Error;
   -----------------------
   -- Get_From_Provider --
   -----------------------
   function Get_From_Provider
     (Context : Internal_Context;
      Name    : Text_Type;
      Kind    : Analysis_Unit_Kind;
      Charset : String;
      Reparse : Boolean) return Internal_Unit is
   begin
      if Reparse and then Has_Rewriting_Handle (Context) then
         raise Precondition_Failure with
            "cannot reparse during tree rewriting";
      end if;
      declare
         Result      : Internal_Unit;
         Dummy_Index : Positive;
      begin
         Context.Unit_Provider.Get_Unit_And_PLE_Root
           (Context, Name, Kind, Charset, Reparse, Result, Dummy_Index);
         return Result;
      exception
         when Precondition_Failure | Property_Error =>
            raise Invalid_Unit_Name_Error with
               "Invalid unit name: " & Image (Name, With_Quotes => True)
               & " (" & Analysis_Unit_Kind'Image (Kind) & ")";
      end;
   end Get_From_Provider;
   -------------------
   -- Unit_Provider --
   -------------------
   function Unit_Provider
     (Context : Internal_Context) return Internal_Unit_Provider_Access
   is (Context.Unit_Provider);
   ------------------
   -- Resolve_Unit --
   ------------------
   procedure Resolve_Unit
     (Context : Internal_Context;
      Name    : Text_Type;
      Kind    : Analysis_Unit_Kind;
      Unit    : out Resolved_Unit)
   is
      --  Look for the cache entry corresponding to Unit; create one if needed
      Dummy    : Resolved_Unit_Array;
      Key      : constant Symbol_Type := Find (Context.Symbols, Name);
      Pos      : Unit_Provider_Cache_Maps.Cursor;
      Inserted : Boolean;
   begin
      Context.Unit_Provider_Cache.Insert (Key, Dummy, Pos, Inserted);
      declare
         Units : Resolved_Unit_Array renames
           Context.Unit_Provider_Cache.Reference (Pos);
         U     : Resolved_Unit renames Units (Kind);
      begin
         --  If the cache entry is not populated for the requested kind, run
         --  the query and save the result for later requests.
         if U.Filename = null then
            declare
               Provider : Internal_Unit_Provider'Class renames
                 Context.Unit_Provider.all;
               Filename : Unbounded_String;
            begin
               Provider.Get_Unit_Location
                 (Name           => Name,
                  Kind           => Kind,
                  Filename       => Filename,
                  PLE_Root_Index => U.PLE_Root_Index);
               Provider.Get_Unit_And_PLE_Root
                 (Context        => Context,
                  Name           => Name,
                  Kind           => Kind,
                  Unit           => U.Unit,
                  PLE_Root_Index => U.PLE_Root_Index);
               U.Filename := new String'(To_String (Filename));
            end;
         end if;
         Unit := U;
      end;
   end Resolve_Unit;
   -----------------------
   -- Get_Unit_Location --
   -----------------------
   procedure Get_Unit_Location
     (Context        : Internal_Context;
      Name           : Text_Type;
      Kind           : Analysis_Unit_Kind;
      Filename       : out String_Access;
      PLE_Root_Index : out Positive)
   is
      U : Resolved_Unit;
   begin
      Resolve_Unit (Context, Name, Kind, U);
      Filename := U.Filename;
      PLE_Root_Index := U.PLE_Root_Index;
   end Get_Unit_Location;
   ---------------------------
   -- Get_Unit_And_PLE_Root --
   ---------------------------
   procedure Get_Unit_And_PLE_Root
     (Context        : Internal_Context;
      Name           : Text_Type;
      Kind           : Analysis_Unit_Kind;
      Unit           : out Internal_Unit;
      PLE_Root_Index : out Positive)
   is
      U : Resolved_Unit;
   begin
      Resolve_Unit (Context, Name, Kind, U);
      Unit := U.Unit;
      PLE_Root_Index := U.PLE_Root_Index;
   end Get_Unit_And_PLE_Root;
   ----------
   -- Hash --
   ----------
   function Hash (Context : Internal_Context) return Hash_Type is
      function H is new Hash_Access (Analysis_Context_Type, Internal_Context);
   begin
      return H (Context);
   end Hash;
   ---------------------
   -- Has_With_Trivia --
   ---------------------
   function Has_With_Trivia (Context : Internal_Context) return Boolean is
   begin
      return Context.With_Trivia;
   end Has_With_Trivia;
   --------------------------------------------
   -- Discard_Errors_In_Populate_Lexical_Env --
   --------------------------------------------
   procedure Discard_Errors_In_Populate_Lexical_Env
     (Context : Internal_Context; Discard : Boolean) is
   begin
      Context.Discard_Errors_In_Populate_Lexical_Env := Discard;
   end Discard_Errors_In_Populate_Lexical_Env;
   ----------------------------------
   -- Set_Logic_Resolution_Timeout --
   ----------------------------------
   procedure Set_Logic_Resolution_Timeout
     (Context : Internal_Context; Timeout : Natural) is
   begin
      Context.Logic_Resolution_Timeout := Timeout;
   end Set_Logic_Resolution_Timeout;
   --------------------------
   -- Has_Rewriting_Handle --
   --------------------------
   function Has_Rewriting_Handle (Context : Internal_Context) return Boolean is
   begin
      return Context.Rewriting_Handle /= No_Rewriting_Handle_Pointer;
   end Has_Rewriting_Handle;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (Context : Internal_Context) is
   begin
      if Context /= null then
         Context.Ref_Count := Context.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (Context : in out Internal_Context) is
   begin
      if Context /= null then
         Context.Ref_Count := Context.Ref_Count - 1;
         if Context.Ref_Count = 0 then
            --  If this context was not completely initialized, just release
            --  the allocated object. Do the full destruction otherwise.
            if Context.Initialized then
               Destroy (Context);
            end if;
            Context_Pool.Release (Context);
         end if;
      end if;
   end Dec_Ref;
   -------------
   -- Destroy --
   -------------
   procedure Destroy (Context : Internal_Context) is
   begin
      --  Destroy all named environment data structures
      for Desc of Context.Named_Envs loop
         for V of Desc.Foreign_Nodes loop
            V.Destroy;
         end loop;
         Destroy (Desc);
      end loop;
      Context.Named_Envs.Clear;
      --  If we are asked to free this context, it means that no one else have
      --  references to its analysis units, so it's safe to destroy these.
      for Unit of Context.Units loop
         Destroy (Unit);
      end loop;
      Context.Units := Units_Maps.Empty_Map;
      Context.Filenames := Virtual_File_Maps.Empty_Map;
      declare
         procedure Destroy is new Ada.Unchecked_Deallocation
           (Env_Rebindings_Type, Env_Rebindings);
         AR : Env_Rebindings_Vectors.Vector renames
            Context.Available_Rebindings;
         R  : Env_Rebindings;
      begin
         for I in AR.First_Index .. AR.Last_Index loop
            R := AR.Get (I);
            Destroy (R);
         end loop;
         AR.Destroy;
      end;
      for Pos in Context.Unit_Provider_Cache.Iterate loop
         declare
            Units : Resolved_Unit_Array renames
              Context.Unit_Provider_Cache.Reference (Pos);
         begin
            for U of Units loop
               Free (U.Filename);
            end loop;
         end;
      end loop;
      Context.Unit_Provider_Cache.Clear;
      Destroy (Context.Templates_Unit);
      AST_Envs.Destroy (Context.Root_Scope);
      Destroy (Context.Symbols);
      Destroy (Context.Parser);
      Dec_Ref (Context.File_Reader);
      Dec_Ref (Context.Unit_Provider);
      Dec_Ref (Context.Event_Handler);
   end Destroy;
   -------------
   -- Context --
   -------------
   function Context (Unit : Internal_Unit) return Internal_Context is
   begin
      return Unit.Context;
   end Context;
   ----------
   -- Hash --
   ----------
   function Hash (Unit : Internal_Unit) return Hash_Type is
      function H is new Hash_Access (Analysis_Unit_Type, Internal_Unit);
   begin
      return H (Unit);
   end Hash;
   -------------
   -- Reparse --
   -------------
   procedure Reparse (Unit : Internal_Unit; Charset : String) is
      Dummy : constant Internal_Unit := Get_From_File
        (Unit.Context, +Unit.Filename.Full_Name, Charset,
         Reparse => True,
         Rule    => Unit.Rule);
   begin
      null;
   end Reparse;
   -------------
   -- Reparse --
   -------------
   procedure Reparse (Unit : Internal_Unit; Charset : String; Buffer : String)
   is
      Dummy : constant Internal_Unit := Get_From_Buffer
        (Unit.Context, +Unit.Filename.Full_Name, Charset, Buffer, Unit.Rule);
   begin
      null;
   end Reparse;
   -----------------------
   -- Reset_Envs_Caches --
   -----------------------
   procedure Reset_Envs_Caches (Unit : Internal_Unit) is
      procedure Internal (Node : Bare_Ada_Node);
      --  Reset env caches in ``Node`` and then in its children recursively
      Generic_Unit : constant Generic_Unit_Ptr := Convert_Unit (Unit);
      --------------
      -- Internal --
      --------------
      procedure Internal (Node : Bare_Ada_Node) is
      begin
         if Node = null then
            return;
         end if;
         --  Make sure to only reset caches of envs belonging to this unit
         if Node.Self_Env.Owner = Generic_Unit then
            Reset_Caches (Node.Self_Env);
         end if;
         for I in 1 .. Children_Count (Node) loop
            Internal (Child (Node, I));
         end loop;
      end Internal;
   begin
      Internal (Unit.Ast_Root);
   end Reset_Envs_Caches;
   --------------------------
   -- Populate_Lexical_Env --
   --------------------------
   procedure Populate_Lexical_Env
     (Unit           : Internal_Unit;
      PLE_Root_Index : Positive
   ) is
      Context : constant Internal_Context := Unit.Context;
      Saved_In_Populate_Lexical_Env : constant Boolean :=
        Context.In_Populate_Lexical_Env;
      Has_Errors : Boolean := False;
      --  Whether at least one Property_Error occurred during this PLE pass
   begin
      --  TODO??? Handle env invalidation when reparsing a unit and when a
      --  previous call raised a Property_Error.
      --  If we have already run PLE on this root, there is nothing to do.
      --  Otherwise, keep track of the fact that PLE was requested for it,
      --  possibly extending the vector if needed.
      if Unit.Env_Populated_Roots.Last_Index >= PLE_Root_Index
         and then Unit.Env_Populated_Roots.Get (PLE_Root_Index)
      then
         return;
      end if;
      for Dummy in Unit.Env_Populated_Roots.Last_Index + 1 .. PLE_Root_Index
      loop
         Unit.Env_Populated_Roots.Append (False);
      end loop;
      Unit.Env_Populated_Roots.Set (PLE_Root_Index, True);
      --  Create context for the PLE run: all exit points must call the Cleanup
      --  procedure above first to clean this context.
      Context.In_Populate_Lexical_Env := True;
      if Main_Trace.Active then
         Main_Trace.Trace
           ("Populating lexical envs for"
            & " root" & PLE_Root_Index'Image & " of"
            & " unit: " & Basename (Unit));
         Main_Trace.Increase_Indent;
      end if;
      --  Fetch the node on which to run PLE: it's the unit root node, or one
      --  of its children if PLE roots are enabled and the unit has a list of
      --  PLE roots. Then run PLE itself.
      declare
         PLE_Root : Bare_Ada_Node := Unit.Ast_Root;
      begin
            if Unit.Ast_Root /= null
               and then Unit.Ast_Root.Kind
                        = Ada_Compilation_Unit_List
            then
               PLE_Root := Child (Unit.Ast_Root, PLE_Root_Index);
            end if;
         if PLE_Root /= null then
            Has_Errors := Populate_Lexical_Env (PLE_Root);
         end if;
      end;
      --  Restore the context for PLE run (undo what was done above)
      Context.In_Populate_Lexical_Env := Saved_In_Populate_Lexical_Env;
      if Main_Trace.Active then
         Main_Trace.Decrease_Indent;
         Main_Trace.Trace
           ("Finished populating lexical envs for"
            & " root" & PLE_Root_Index'Image & " of"
            & " unit: " & Basename (Unit));
      end if;
      Reset_Envs_Caches (Unit);
      if Has_Errors and then not Context.Discard_Errors_In_Populate_Lexical_Env
      then
         raise Property_Error with
            "errors occurred in Populate_Lexical_Env";
      end if;
   end Populate_Lexical_Env;
   -----------------------------------
   -- Populate_Lexical_Env_For_Unit --
   -----------------------------------
   procedure Populate_Lexical_Env_For_Unit (Node : Bare_Ada_Node) is
      Root  : Bare_Ada_Node;
      Index : Natural;
   begin
      Lookup_PLE_Root (Node, Root, Index);
      if Index = 0 then
         Index := 1;
      end if;
      Populate_Lexical_Env (Node.Unit, Index);
   end Populate_Lexical_Env_For_Unit;
   ------------------
   -- Get_Filename --
   ------------------
   function Get_Filename (Unit : Internal_Unit) return String is
     (+Unit.Filename.Full_Name);
   -----------------
   -- Get_Charset --
   -----------------
   function Get_Charset (Unit : Internal_Unit) return String is
   begin
      return To_String (Unit.Charset);
   end Get_Charset;
   ---------------------
   -- Has_Diagnostics --
   ---------------------
   function Has_Diagnostics (Unit : Internal_Unit) return Boolean is
   begin
      return not Unit.Diagnostics.Is_Empty;
   end Has_Diagnostics;
   -----------------
   -- Diagnostics --
   -----------------
   function Diagnostics (Unit : Internal_Unit) return Diagnostics_Array is
      Result : Diagnostics_Array (1 .. Natural (Unit.Diagnostics.Length));
      I      : Natural := 1;
   begin
      for D of Unit.Diagnostics loop
         Result (I) := D;
         I := I + 1;
      end loop;
      return Result;
   end Diagnostics;
   ---------------------------
   -- Format_GNU_Diagnostic --
   ---------------------------
   function Format_GNU_Diagnostic
     (Unit : Internal_Unit; D : Diagnostic) return String
   is
      Filename : constant String := Basename (Unit);
      Sloc     : constant Source_Location := Start_Sloc (D.Sloc_Range);
      Msg      : constant String :=
         Image
           (Ada.Strings.Wide_Wide_Unbounded.To_Wide_Wide_String (D.Message));
   begin
      return (Filename
              & (if Sloc = No_Source_Location then "" else ":" & Image (Sloc))
              & ": " & Msg);
   end Format_GNU_Diagnostic;
   ----------
   -- Root --
   ----------
   function Root (Unit : Internal_Unit) return Bare_Ada_Node is
     (Unit.Ast_Root);
   -----------------
   -- First_Token --
   -----------------
   function First_Token (Unit : Internal_Unit) return Token_Reference is
     (Wrap_Token_Reference (Unit.Context,
                            Unit.TDH'Access,
                            First_Token_Or_Trivia (Unit.TDH)));
   ----------------
   -- Last_Token --
   ----------------
   function Last_Token (Unit : Internal_Unit) return Token_Reference is
     (Wrap_Token_Reference (Unit.Context,
                            Unit.TDH'Access,
                            Last_Token_Or_Trivia (Unit.TDH)));
   -----------------
   -- Token_Count --
   -----------------
   function Token_Count (Unit : Internal_Unit) return Natural is
     (Unit.TDH.Tokens.Length);
   ------------------
   -- Trivia_Count --
   ------------------
   function Trivia_Count (Unit : Internal_Unit) return Natural is
     (Unit.TDH.Trivias.Length);
   ----------
   -- Text --
   ----------
   function Text (Unit : Internal_Unit) return Text_Type is
   begin
      return Text (First_Token (Unit), Last_Token (Unit));
   end Text;
   ------------------
   -- Lookup_Token --
   ------------------
   function Lookup_Token
     (Unit : Internal_Unit; Sloc : Source_Location) return Token_Reference
   is
      Result : constant Token_Or_Trivia_Index := Lookup_Token (Unit.TDH, Sloc);
   begin
      return Wrap_Token_Reference (Unit.Context, Unit.TDH'Access, Result);
   end Lookup_Token;
   ---------------------
   -- Lookup_PLE_Root --
   ---------------------
   procedure Lookup_PLE_Root
     (Node  : Bare_Ada_Node;
      Root  : out Bare_Ada_Node;
      Index : out Natural)
   is
      Unit : constant Internal_Unit := Node.Unit;
   begin
      --  If this unit does not contain a list of PLE roots, just return the
      --  unit root node.
      if Unit.PLE_Roots_Starting_Token.Is_Empty then
         Root := Unit.Ast_Root;
         Index := 0;
         return;
      end if;
      --  Otherwise, look for the last PLE root whose first token (in
      --  Unit.PLE_Roots_Starting_Token) appears before Node's (T). This vector
      --  is sorted by construction, so we can perform a binary search.
      declare
         T      : constant Token_Index := Node.Token_Start_Index;
         Tokens : Token_Index_Vectors.Vector renames
           Unit.PLE_Roots_Starting_Token;
         First : Positive := Tokens.First_Index;
         Last  : Positive := Tokens.Last_Index;
         I     : Positive;
      begin
         while First < Last loop
            --  Because we look for the "floor" (last element that is <= T), we
            --  need to look at the value in Last when there are only two
            --  elements left to look at. If we did not do that, then we would
            --  go into an infinite loop when Tokens[First] < T.
            I := (if First + 1 = Last
                  then Last
                  else (First + Last) / 2);
            declare
               I_T : constant Token_Index := Tokens.Get (I);
            begin
               if I_T <= T then
                  First := I;
               else
                  Last := I - 1;
               end if;
            end;
         end loop;
         Root := Child (Unit.Ast_Root, First);
         Index := First;
      end;
   end Lookup_PLE_Root;
   --------------
   -- Ple_Root --
   --------------
   function Ple_Root
     (Node : Bare_Ada_Node) return Bare_Ada_Node
   is
      Root        : Bare_Ada_Node;
      Dummy_Index : Natural;
   begin
      if Node = null then
         raise Property_Error with "null node dereference";
      end if;
      Lookup_PLE_Root (Node, Root, Dummy_Index);
      return Root;
   end Ple_Root;
   ----------------------
   -- Dump_Lexical_Env --
   ----------------------
   procedure Dump_Lexical_Env (Unit : Internal_Unit) is
      Node     : constant Bare_Ada_Node := Unit.Ast_Root;
      Root_Env : constant Lexical_Env := Unit.Context.Root_Scope;
      State    : Dump_Lexical_Env_State := (Root_Env => Root_Env, others => <>);
      function Get_Parent (Env : Lexical_Env) return Lexical_Env
      is (Unwrap (Env).Parent);
      --------------------------
      -- Explore_Parent_Chain --
      --------------------------
      procedure Explore_Parent_Chain (Env : Lexical_Env) is
         P : Lexical_Env;
      begin
         if Env /= Null_Lexical_Env then
            P := Get_Parent (Env);
            Dump_One_Lexical_Env
              (Env, Get_Env_Id (Env, State), Get_Env_Id (P, State));
            Explore_Parent_Chain (P);
         end if;
      end Explore_Parent_Chain;
      --------------
      -- Internal --
      --------------
      procedure Internal (Current : Bare_Ada_Node) is
         Explore_Parent : Boolean := False;
         Env, Parent    : Lexical_Env;
      begin
         if Current = null then
            return;
         end if;
         --  We only dump environments that we haven't dumped before. This way
         --  we'll only dump environments at the site of their creation, and
         --  not in any subsequent link. We use the Env_Ids map to check which
         --  envs we have already seen or not.
         if not State.Env_Ids.Contains (Current.Self_Env) then
            Env := Current.Self_Env;
            Parent := Get_Parent (Env);
            Explore_Parent := not State.Env_Ids.Contains (Parent);
            Dump_One_Lexical_Env
              (Env, Get_Env_Id (Env, State), Get_Env_Id (Parent, State));
            if Explore_Parent then
               Explore_Parent_Chain (Parent);
            end if;
         end if;
         for Child of Internal_Bare_Ada_Node_Array'(Children (Current))
         loop
            Internal (Child);
         end loop;
      end Internal;
      --  This procedure implements the main recursive logic of dumping the
      --  environments.
   begin
      Internal (Bare_Ada_Node (Node));
   end Dump_Lexical_Env;
   --------------
   -- Get_Line --
   --------------
   function Get_Line
     (Unit : Internal_Unit; Line_Number : Positive) return Text_Type
   is
   begin
      return Get_Line (Unit.TDH, Line_Number);
   end Get_Line;
   -----------
   -- Print --
   -----------
   procedure Print (Unit : Internal_Unit; Show_Slocs : Boolean) is
   begin
      if Unit.Ast_Root = null then
         Put_Line ("<empty analysis unit>");
      else
         Print (Unit.Ast_Root, Show_Slocs);
      end if;
   end Print;
   ---------------
   -- PP_Trivia --
   ---------------
   procedure PP_Trivia (Unit : Internal_Unit) is
      procedure Process (Trivia : Token_Index) is
         Data : constant Stored_Token_Data :=
            Unit.TDH.Trivias.Get (Natural (Trivia)).T;
      begin
         Put_Line (Image (Text (Unit.TDH, Data)));
      end Process;
      Last_Token : constant Token_Index :=
         Token_Index (Token_Vectors.Last_Index (Unit.TDH.Tokens) - 1);
      --  Index for the last token in Unit excluding the Termination token
      --  (hence the -1).
   begin
      for Tok of Get_Leading_Trivias (Unit.TDH) loop
         Process (Tok);
      end loop;
      PP_Trivia (Unit.Ast_Root);
      for Tok of Get_Trivias (Unit.TDH, Last_Token) loop
         Process (Tok);
      end loop;
   end PP_Trivia;
   -------------
   -- Destroy --
   -------------
   procedure Destroy (Unit : in out Internal_Unit) is
   begin
      if Unit = No_Analysis_Unit then
         return;
      end if;
      --  Clear the env caches while the unit is still fully alive to make sure
      --  that what is accessed in ``Lexical_Env_Cache_Updated`` is still
      --  valid, as it will be called back by lexical envs that are being
      --  destroyed.
      Reset_Envs_Caches (Unit);
      Unit.PLE_Roots_Starting_Token.Destroy;
      Unit.Env_Populated_Roots.Destroy;
      Unit.Exiled_Entries.Destroy;
      Unit.Foreign_Nodes.Destroy;
      Unit.Exiled_Entries_In_NED.Destroy;
      Unit.Exiled_Envs.Destroy;
      Unit.Named_Envs.Destroy;
      Analysis_Unit_Sets.Destroy (Unit.Referenced_Units);
         Destroy (Unit.Memoization_Map);
      Destroy_Rebindings (Unit.Rebindings'Access);
      Unit.Rebindings.Destroy;
      if Unit.Ast_Root /= null then
         Destroy (Unit.Ast_Root);
      end if;
      Free (Unit.TDH);
      Free (Unit.Ast_Mem_Pool);
      Destroy_Unit_Destroyables (Unit);
      Destroyable_Vectors.Destroy (Unit.Destroyables);
for Cur in Unit.Nodes_Nameres.Iterate loop
   declare
      V : Resolution_Val renames Unit.Nodes_Nameres.Reference (Cur);
   begin
      Free_Memoized_Error (V.Exc_Id, V.Exc_Msg);
      Dec_Ref (V.Return_Value);
   end;
end loop;
      Free (Unit);
   end Destroy;
   -------------------
   -- Is_Token_Node --
   -------------------
   function Is_Token_Node (Node : Bare_Ada_Node) return Boolean is
   begin
      return Is_Token_Node (Node.Kind);
   end Is_Token_Node;
   ------------------
   -- Is_Synthetic --
   ------------------
   function Is_Synthetic (Node : Bare_Ada_Node) return Boolean is
   begin
      return Node.Kind in Synthetic_Nodes;
   end Is_Synthetic;
   ------------------------------
   -- Raise_Property_Exception --
   ------------------------------
   procedure Raise_Property_Exception
     (Node    : Bare_Ada_Node;
      Exc     : Ada.Exceptions.Exception_Id;
      Message : String)
   is
      Sloc_Prefix : constant String :=
        (if Node = null
         then ""
         else Ada.Directories.Simple_Name (Get_Filename (Unit (Node)))
              & ":" & Image (Sloc_Range (Node)) & ": ");
   begin
      Ada.Exceptions.Raise_Exception (Exc, Sloc_Prefix & Message);
   end Raise_Property_Exception;
   ------------------------------
   -- Register_Destroyable_Gen --
   ------------------------------
   procedure Register_Destroyable_Gen
     (Unit : Internal_Unit; Object : T_Access)
   is
      function Convert is new Ada.Unchecked_Conversion
        (System.Address, Destroy_Procedure);
      procedure Destroy_Procedure (Object : in out T_Access) renames Destroy;
   begin
      Register_Destroyable_Helper
        (Unit,
         Object.all'Address,
         Convert (Destroy_Procedure'Address));
   end Register_Destroyable_Gen;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Ada_Node_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Ada_Node
   is
      function Absolute_Get
        (T : Bare_Ada_Node_Array_Access; Index : Integer)
         return Bare_Ada_Node
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Ada_Node,
         Sequence_Type => Bare_Ada_Node_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Ada_Node;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Ada_Node_Array_Access) return Bare_Ada_Node_Array_Access is
      Ret : Bare_Ada_Node_Array_Access := Create_Bare_Ada_Node_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Ada_Node_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Ada_Node_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Ada_Node_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Ada_Node_Array (Items_Count : Natural) return Bare_Ada_Node_Array_Access
   is (if Items_Count = 0
       then No_Bare_Ada_Node_Array_Type
       else new Bare_Ada_Node_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Ada_Node_Array
     (Items : Internal_Bare_Ada_Node_Array) return Bare_Ada_Node_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Ada_Node_Array_Type;
      end if;
      return new Bare_Ada_Node_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Ada_Node_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Ada_Node_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Bare_Anonymous_Expr_Decl_Vectors is new Langkit_Support.Vectors (Bare_Anonymous_Expr_Decl);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Anonymous_Expr_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Anonymous_Expr_Decl
   is
      function Absolute_Get
        (T : Bare_Anonymous_Expr_Decl_Array_Access; Index : Integer)
         return Bare_Anonymous_Expr_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Anonymous_Expr_Decl,
         Sequence_Type => Bare_Anonymous_Expr_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Anonymous_Expr_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Anonymous_Expr_Decl_Array_Access) return Bare_Anonymous_Expr_Decl_Array_Access is
      Ret : Bare_Anonymous_Expr_Decl_Array_Access := Create_Bare_Anonymous_Expr_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Anonymous_Expr_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Anonymous_Expr_Decl_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Anonymous_Expr_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Anonymous_Expr_Decl_Array (Items_Count : Natural) return Bare_Anonymous_Expr_Decl_Array_Access
   is (if Items_Count = 0
       then No_Bare_Anonymous_Expr_Decl_Array_Type
       else new Bare_Anonymous_Expr_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Anonymous_Expr_Decl_Array
     (Items : Internal_Bare_Anonymous_Expr_Decl_Array) return Bare_Anonymous_Expr_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Anonymous_Expr_Decl_Array_Type;
      end if;
      return new Bare_Anonymous_Expr_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Anonymous_Expr_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Anonymous_Expr_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Bare_Compilation_Unit_Vectors is new Langkit_Support.Vectors (Bare_Compilation_Unit);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Compilation_Unit_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Compilation_Unit
   is
      function Absolute_Get
        (T : Bare_Compilation_Unit_Array_Access; Index : Integer)
         return Bare_Compilation_Unit
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Compilation_Unit,
         Sequence_Type => Bare_Compilation_Unit_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Compilation_Unit;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Compilation_Unit_Array_Access) return Bare_Compilation_Unit_Array_Access is
      Ret : Bare_Compilation_Unit_Array_Access := Create_Bare_Compilation_Unit_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Compilation_Unit_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Compilation_Unit_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Compilation_Unit_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Compilation_Unit_Array (Items_Count : Natural) return Bare_Compilation_Unit_Array_Access
   is (if Items_Count = 0
       then No_Bare_Compilation_Unit_Array_Type
       else new Bare_Compilation_Unit_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Compilation_Unit_Array
     (Items : Internal_Bare_Compilation_Unit_Array) return Bare_Compilation_Unit_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Compilation_Unit_Array_Type;
      end if;
      return new Bare_Compilation_Unit_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Compilation_Unit_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Compilation_Unit_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Bare_Name_Vectors is new Langkit_Support.Vectors (Bare_Name);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Name_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Name
   is
      function Absolute_Get
        (T : Bare_Name_Array_Access; Index : Integer)
         return Bare_Name
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Name,
         Sequence_Type => Bare_Name_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Name;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Name_Array_Access) return Bare_Name_Array_Access is
      Ret : Bare_Name_Array_Access := Create_Bare_Name_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Name_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Name_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Name_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Name_Array (Items_Count : Natural) return Bare_Name_Array_Access
   is (if Items_Count = 0
       then No_Bare_Name_Array_Type
       else new Bare_Name_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Name_Array
     (Items : Internal_Bare_Name_Array) return Bare_Name_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Name_Array_Type;
      end if;
      return new Bare_Name_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Name_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Name_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Bare_Pragma_Node_Vectors is new Langkit_Support.Vectors (Bare_Pragma_Node);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Pragma_Node_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Pragma_Node
   is
      function Absolute_Get
        (T : Bare_Pragma_Node_Array_Access; Index : Integer)
         return Bare_Pragma_Node
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Pragma_Node,
         Sequence_Type => Bare_Pragma_Node_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Pragma_Node;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Pragma_Node_Array_Access) return Bare_Pragma_Node_Array_Access is
      Ret : Bare_Pragma_Node_Array_Access := Create_Bare_Pragma_Node_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Pragma_Node_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Pragma_Node_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Pragma_Node_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Pragma_Node_Array (Items_Count : Natural) return Bare_Pragma_Node_Array_Access
   is (if Items_Count = 0
       then No_Bare_Pragma_Node_Array_Type
       else new Bare_Pragma_Node_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Pragma_Node_Array
     (Items : Internal_Bare_Pragma_Node_Array) return Bare_Pragma_Node_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Pragma_Node_Array_Type;
      end if;
      return new Bare_Pragma_Node_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Pragma_Node_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Pragma_Node_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Single_Tok_Node_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Single_Tok_Node
   is
      function Absolute_Get
        (T : Bare_Single_Tok_Node_Array_Access; Index : Integer)
         return Bare_Single_Tok_Node
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Single_Tok_Node,
         Sequence_Type => Bare_Single_Tok_Node_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Single_Tok_Node;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Single_Tok_Node_Array_Access) return Bare_Single_Tok_Node_Array_Access is
      Ret : Bare_Single_Tok_Node_Array_Access := Create_Bare_Single_Tok_Node_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Single_Tok_Node_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Single_Tok_Node_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Single_Tok_Node_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Single_Tok_Node_Array (Items_Count : Natural) return Bare_Single_Tok_Node_Array_Access
   is (if Items_Count = 0
       then No_Bare_Single_Tok_Node_Array_Type
       else new Bare_Single_Tok_Node_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Single_Tok_Node_Array
     (Items : Internal_Bare_Single_Tok_Node_Array) return Bare_Single_Tok_Node_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Single_Tok_Node_Array_Type;
      end if;
      return new Bare_Single_Tok_Node_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Single_Tok_Node_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Single_Tok_Node_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Bare_Use_Clause_Vectors is new Langkit_Support.Vectors (Bare_Use_Clause);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Bare_Use_Clause_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Use_Clause
   is
      function Absolute_Get
        (T : Bare_Use_Clause_Array_Access; Index : Integer)
         return Bare_Use_Clause
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Use_Clause,
         Sequence_Type => Bare_Use_Clause_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Use_Clause;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Ada_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Bare_Use_Clause_Array_Access) return Bare_Use_Clause_Array_Access is
      Ret : Bare_Use_Clause_Array_Access := Create_Bare_Use_Clause_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Use_Clause_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Bare_Use_Clause_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Use_Clause_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Bare_Use_Clause_Array (Items_Count : Natural) return Bare_Use_Clause_Array_Access
   is (if Items_Count = 0
       then No_Bare_Use_Clause_Array_Type
       else new Bare_Use_Clause_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Bare_Use_Clause_Array
     (Items : Internal_Bare_Use_Clause_Array) return Bare_Use_Clause_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Use_Clause_Array_Type;
      end if;
      return new Bare_Use_Clause_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Bare_Use_Clause_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Use_Clause_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Aspect_Vectors is new Langkit_Support.Vectors (Internal_Aspect);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Aspect_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Aspect
   is
      function Absolute_Get
        (T : Internal_Aspect_Array_Access; Index : Integer)
         return Internal_Aspect
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Aspect,
         Sequence_Type => Internal_Aspect_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Aspect;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Aspect;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Aspect_Array_Access) return Internal_Aspect_Array_Access is
      Ret : Internal_Aspect_Array_Access := Create_Internal_Aspect_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Aspect_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Aspect_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Aspect_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Aspect_Array (Items_Count : Natural) return Internal_Aspect_Array_Access
   is (if Items_Count = 0
       then No_Internal_Aspect_Array_Type
       else new Internal_Aspect_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Aspect_Array
     (Items : Internal_Internal_Aspect_Array) return Internal_Aspect_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Aspect_Array_Type;
      end if;
      return new Internal_Aspect_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Aspect_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Aspect_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      function Make_Unique (A : Internal_Aspect_Array_Access) return Internal_Aspect_Array_Access is
         --  To keep the order of the result deterministic, this function works
         --  in two steps. First, determine for each item in A whether it
         --  duplicate a previous item. Only then, go through all items in A
         --  and add only non-duplicated ones to the result.
         Duplicates : array (A.Items'Range) of Boolean := (others => False);
         --  For each item in A, indicate whether the value is a duplicate of a
         --  previous one in the array.
         Unique_Elements : Natural;
         --  Number of unique elements in A
      begin
         --  Step 1: compute Duplicates
         declare
            package Sets is new Ada.Containers.Hashed_Sets
              (Element_Type        => Internal_Aspect,
               "="                 => "=",
               Equivalent_Elements => "=",
               Hash                => Hash);
            Set : Sets.Set;
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Aspect renames A.Items (I);
               begin
                  if Set.Contains (Item) then
                     Duplicates (I) := True;
                  else
                     Set.Insert (Item);
                  end if;
               end;
            end loop;
            Unique_Elements := Natural (Set.Length);
         end;
         --  Step 2: create the result
         declare
            Next   : Positive := 1;
            Result : constant Internal_Aspect_Array_Access :=
               Create_Internal_Aspect_Array (Unique_Elements);
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Aspect renames A.Items (I);
               begin
                  if not Duplicates (I) then
                     Result.Items (Next) := Item;
                     Next := Next + 1;
                  end if;
               end;
            end loop;
            pragma Assert (Next = Result.N + 1);
            return Result;
         end;
      end Make_Unique;
      package Internal_Completion_Item_Vectors is new Langkit_Support.Vectors (Internal_Completion_Item);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Completion_Item_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Completion_Item
   is
      function Absolute_Get
        (T : Internal_Completion_Item_Array_Access; Index : Integer)
         return Internal_Completion_Item
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Completion_Item,
         Sequence_Type => Internal_Completion_Item_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Completion_Item;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Completion_Item;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Completion_Item_Array_Access) return Internal_Completion_Item_Array_Access is
      Ret : Internal_Completion_Item_Array_Access := Create_Internal_Completion_Item_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Completion_Item_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Completion_Item_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Completion_Item_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Completion_Item_Array (Items_Count : Natural) return Internal_Completion_Item_Array_Access
   is (if Items_Count = 0
       then No_Internal_Completion_Item_Array_Type
       else new Internal_Completion_Item_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Completion_Item_Array
     (Items : Internal_Internal_Completion_Item_Array) return Internal_Completion_Item_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Completion_Item_Array_Type;
      end if;
      return new Internal_Completion_Item_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Completion_Item_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- To_Iterator --
      -----------------
      function To_Iterator
        (Self    : Internal_Completion_Item_Array_Access;
         Context : Internal_Context) return Internal_Completion_Item_Iterator_Access
      is
      begin
         Inc_Ref (Self);
         return new Internal_Internal_Completion_Item_Iterator'
           (Ref_Count  => 1,
            Safety_Net => Create_Safety_Net (Context),
            Elements   => Self,
            Index      => 1);
      end To_Iterator;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Completion_Item_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Discriminant_Values_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Discriminant_Values
   is
      function Absolute_Get
        (T : Internal_Discriminant_Values_Array_Access; Index : Integer)
         return Internal_Discriminant_Values
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Discriminant_Values,
         Sequence_Type => Internal_Discriminant_Values_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Discriminant_Values;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Discriminant_Values;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Discriminant_Values_Array_Access) return Internal_Discriminant_Values_Array_Access is
      Ret : Internal_Discriminant_Values_Array_Access := Create_Internal_Discriminant_Values_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Discriminant_Values_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Discriminant_Values_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Discriminant_Values_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Discriminant_Values_Array (Items_Count : Natural) return Internal_Discriminant_Values_Array_Access
   is (if Items_Count = 0
       then No_Internal_Discriminant_Values_Array_Type
       else new Internal_Discriminant_Values_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Discriminant_Values_Array
     (Items : Internal_Internal_Discriminant_Values_Array) return Internal_Discriminant_Values_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Discriminant_Values_Array_Type;
      end if;
      return new Internal_Discriminant_Values_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Discriminant_Values_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Discriminant_Values_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Doc_Annotation_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Doc_Annotation
   is
      function Absolute_Get
        (T : Internal_Doc_Annotation_Array_Access; Index : Integer)
         return Internal_Doc_Annotation
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Doc_Annotation,
         Sequence_Type => Internal_Doc_Annotation_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Doc_Annotation;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Doc_Annotation;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Doc_Annotation_Array_Access) return Internal_Doc_Annotation_Array_Access is
      Ret : Internal_Doc_Annotation_Array_Access := Create_Internal_Doc_Annotation_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Doc_Annotation_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Doc_Annotation_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Doc_Annotation_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Doc_Annotation_Array (Items_Count : Natural) return Internal_Doc_Annotation_Array_Access
   is (if Items_Count = 0
       then No_Internal_Doc_Annotation_Array_Type
       else new Internal_Doc_Annotation_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Doc_Annotation_Array
     (Items : Internal_Internal_Doc_Annotation_Array) return Internal_Doc_Annotation_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Doc_Annotation_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Doc_Annotation_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Doc_Annotation_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Doc_Annotation_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Accept_Stmt_Vectors is new Langkit_Support.Vectors (Internal_Entity_Accept_Stmt);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Accept_Stmt_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Accept_Stmt
   is
      function Absolute_Get
        (T : Internal_Entity_Accept_Stmt_Array_Access; Index : Integer)
         return Internal_Entity_Accept_Stmt
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Accept_Stmt,
         Sequence_Type => Internal_Entity_Accept_Stmt_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Accept_Stmt;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Accept_Stmt;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Accept_Stmt_Array_Access) return Internal_Entity_Accept_Stmt_Array_Access is
      Ret : Internal_Entity_Accept_Stmt_Array_Access := Create_Internal_Entity_Accept_Stmt_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Accept_Stmt_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Accept_Stmt_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Accept_Stmt_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Accept_Stmt_Array (Items_Count : Natural) return Internal_Entity_Accept_Stmt_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Accept_Stmt_Array_Type
       else new Internal_Entity_Accept_Stmt_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Accept_Stmt_Array
     (Items : Internal_Internal_Entity_Accept_Stmt_Array) return Internal_Entity_Accept_Stmt_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Accept_Stmt_Array_Type;
      end if;
      return new Internal_Entity_Accept_Stmt_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Accept_Stmt_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Accept_Stmt_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Ada_Node_List_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Ada_Node_List
   is
      function Absolute_Get
        (T : Internal_Entity_Ada_Node_List_Array_Access; Index : Integer)
         return Internal_Entity_Ada_Node_List
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Ada_Node_List,
         Sequence_Type => Internal_Entity_Ada_Node_List_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Ada_Node_List;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Ada_Node_List;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Ada_Node_List_Array_Access) return Internal_Entity_Ada_Node_List_Array_Access is
      Ret : Internal_Entity_Ada_Node_List_Array_Access := Create_Internal_Entity_Ada_Node_List_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Ada_Node_List_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Ada_Node_List_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Ada_Node_List_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Ada_Node_List_Array (Items_Count : Natural) return Internal_Entity_Ada_Node_List_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Ada_Node_List_Array_Type
       else new Internal_Entity_Ada_Node_List_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Ada_Node_List_Array
     (Items : Internal_Internal_Entity_Ada_Node_List_Array) return Internal_Entity_Ada_Node_List_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Ada_Node_List_Array_Type;
      end if;
      return new Internal_Entity_Ada_Node_List_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Ada_Node_List_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Ada_Node_List_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Vectors is new Langkit_Support.Vectors (Internal_Entity);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity
   is
      function Absolute_Get
        (T : Internal_Entity_Array_Access; Index : Integer)
         return Internal_Entity
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity,
         Sequence_Type => Internal_Entity_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Array_Access) return Internal_Entity_Array_Access is
      Ret : Internal_Entity_Array_Access := Create_Internal_Entity_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Array (Items_Count : Natural) return Internal_Entity_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Array_Type
       else new Internal_Entity_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Array
     (Items : AST_Envs.Entity_Array) return Internal_Entity_Array_Access
   is (if Items'Length = 0
       then No_Internal_Entity_Array_Type
       else new Internal_Entity_Array_Record'
         (N         => Items'Length,
          Items     => Implementation.Internal_Internal_Entity_Array (Items),
          Ref_Count => 1));
   function Create_Internal_Entity_Array
     (Items : Internal_Internal_Entity_Array) return Internal_Entity_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Array_Type;
      end if;
      return new Internal_Entity_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Base_Assoc_Vectors is new Langkit_Support.Vectors (Internal_Entity_Base_Assoc);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Base_Assoc_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Base_Assoc
   is
      function Absolute_Get
        (T : Internal_Entity_Base_Assoc_Array_Access; Index : Integer)
         return Internal_Entity_Base_Assoc
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Base_Assoc,
         Sequence_Type => Internal_Entity_Base_Assoc_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Base_Assoc;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Base_Assoc;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Base_Assoc_Array_Access) return Internal_Entity_Base_Assoc_Array_Access is
      Ret : Internal_Entity_Base_Assoc_Array_Access := Create_Internal_Entity_Base_Assoc_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Base_Assoc_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Base_Assoc_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Base_Assoc_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Base_Assoc_Array (Items_Count : Natural) return Internal_Entity_Base_Assoc_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Base_Assoc_Array_Type
       else new Internal_Entity_Base_Assoc_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Base_Assoc_Array
     (Items : Internal_Internal_Entity_Base_Assoc_Array) return Internal_Entity_Base_Assoc_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Base_Assoc_Array_Type;
      end if;
      return new Internal_Entity_Base_Assoc_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Base_Assoc_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Base_Assoc_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Base_Formal_Param_Decl_Vectors is new Langkit_Support.Vectors (Internal_Entity_Base_Formal_Param_Decl);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Base_Formal_Param_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Base_Formal_Param_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Base_Formal_Param_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Base_Formal_Param_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Base_Formal_Param_Decl,
         Sequence_Type => Internal_Entity_Base_Formal_Param_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Base_Formal_Param_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Base_Formal_Param_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Base_Formal_Param_Decl_Array_Access) return Internal_Entity_Base_Formal_Param_Decl_Array_Access is
      Ret : Internal_Entity_Base_Formal_Param_Decl_Array_Access := Create_Internal_Entity_Base_Formal_Param_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Base_Formal_Param_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Base_Formal_Param_Decl_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Base_Formal_Param_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Base_Formal_Param_Decl_Array (Items_Count : Natural) return Internal_Entity_Base_Formal_Param_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Base_Formal_Param_Decl_Array_Type
       else new Internal_Entity_Base_Formal_Param_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Base_Formal_Param_Decl_Array
     (Items : Internal_Internal_Entity_Base_Formal_Param_Decl_Array) return Internal_Entity_Base_Formal_Param_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Base_Formal_Param_Decl_Array_Type;
      end if;
      return new Internal_Entity_Base_Formal_Param_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Base_Formal_Param_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Base_Formal_Param_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Base_Type_Decl_Vectors is new Langkit_Support.Vectors (Internal_Entity_Base_Type_Decl);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Base_Type_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Base_Type_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Base_Type_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Base_Type_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Base_Type_Decl,
         Sequence_Type => Internal_Entity_Base_Type_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Base_Type_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Base_Type_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Base_Type_Decl_Array_Access) return Internal_Entity_Base_Type_Decl_Array_Access is
      Ret : Internal_Entity_Base_Type_Decl_Array_Access := Create_Internal_Entity_Base_Type_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Base_Type_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Base_Type_Decl_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Base_Type_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Base_Type_Decl_Array (Items_Count : Natural) return Internal_Entity_Base_Type_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Base_Type_Decl_Array_Type
       else new Internal_Entity_Base_Type_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Base_Type_Decl_Array
     (Items : Internal_Internal_Entity_Base_Type_Decl_Array) return Internal_Entity_Base_Type_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Base_Type_Decl_Array_Type;
      end if;
      return new Internal_Entity_Base_Type_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Base_Type_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Base_Type_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      function Make_Unique (A : Internal_Entity_Base_Type_Decl_Array_Access) return Internal_Entity_Base_Type_Decl_Array_Access is
         --  To keep the order of the result deterministic, this function works
         --  in two steps. First, determine for each item in A whether it
         --  duplicate a previous item. Only then, go through all items in A
         --  and add only non-duplicated ones to the result.
         Duplicates : array (A.Items'Range) of Boolean := (others => False);
         --  For each item in A, indicate whether the value is a duplicate of a
         --  previous one in the array.
         Unique_Elements : Natural;
         --  Number of unique elements in A
      begin
         --  Step 1: compute Duplicates
         declare
            package Sets is new Ada.Containers.Hashed_Sets
              (Element_Type        => Internal_Entity_Base_Type_Decl,
               "="                 => "=",
               Equivalent_Elements => "=",
               Hash                => Hash);
            Set : Sets.Set;
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Entity_Base_Type_Decl renames A.Items (I);
               begin
                  if Set.Contains (Item) then
                     Duplicates (I) := True;
                  else
                     Set.Insert (Item);
                  end if;
               end;
            end loop;
            Unique_Elements := Natural (Set.Length);
         end;
         --  Step 2: create the result
         declare
            Next   : Positive := 1;
            Result : constant Internal_Entity_Base_Type_Decl_Array_Access :=
               Create_Internal_Entity_Base_Type_Decl_Array (Unique_Elements);
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Entity_Base_Type_Decl renames A.Items (I);
               begin
                  if not Duplicates (I) then
                     Result.Items (Next) := Item;
                     Next := Next + 1;
                  end if;
               end;
            end loop;
            pragma Assert (Next = Result.N + 1);
            return Result;
         end;
      end Make_Unique;
      ----------
      -- Hash --
      ----------
      function Hash (R : Internal_Entity_Base_Type_Decl_Array_Access) return Hash_Type is
         Result : Hash_Type := Initial_Hash;
      begin
         for I in R.Items'Range loop
            Result := Combine (Result, Hash (R.Items (I)));
         end loop;
         return Result;
      end Hash;
      package Internal_Entity_Basic_Decl_Vectors is new Langkit_Support.Vectors (Internal_Entity_Basic_Decl);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Basic_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Basic_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Basic_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Basic_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Basic_Decl,
         Sequence_Type => Internal_Entity_Basic_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Basic_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Basic_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Basic_Decl_Array_Access) return Internal_Entity_Basic_Decl_Array_Access is
      Ret : Internal_Entity_Basic_Decl_Array_Access := Create_Internal_Entity_Basic_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Basic_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Basic_Decl_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Basic_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Basic_Decl_Array (Items_Count : Natural) return Internal_Entity_Basic_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Basic_Decl_Array_Type
       else new Internal_Entity_Basic_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Basic_Decl_Array
     (Items : Internal_Internal_Entity_Basic_Decl_Array) return Internal_Entity_Basic_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Basic_Decl_Array_Type;
      end if;
      return new Internal_Entity_Basic_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Basic_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Basic_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      function Make_Unique (A : Internal_Entity_Basic_Decl_Array_Access) return Internal_Entity_Basic_Decl_Array_Access is
         --  To keep the order of the result deterministic, this function works
         --  in two steps. First, determine for each item in A whether it
         --  duplicate a previous item. Only then, go through all items in A
         --  and add only non-duplicated ones to the result.
         Duplicates : array (A.Items'Range) of Boolean := (others => False);
         --  For each item in A, indicate whether the value is a duplicate of a
         --  previous one in the array.
         Unique_Elements : Natural;
         --  Number of unique elements in A
      begin
         --  Step 1: compute Duplicates
         declare
            package Sets is new Ada.Containers.Hashed_Sets
              (Element_Type        => Internal_Entity_Basic_Decl,
               "="                 => "=",
               Equivalent_Elements => "=",
               Hash                => Hash);
            Set : Sets.Set;
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Entity_Basic_Decl renames A.Items (I);
               begin
                  if Set.Contains (Item) then
                     Duplicates (I) := True;
                  else
                     Set.Insert (Item);
                  end if;
               end;
            end loop;
            Unique_Elements := Natural (Set.Length);
         end;
         --  Step 2: create the result
         declare
            Next   : Positive := 1;
            Result : constant Internal_Entity_Basic_Decl_Array_Access :=
               Create_Internal_Entity_Basic_Decl_Array (Unique_Elements);
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Entity_Basic_Decl renames A.Items (I);
               begin
                  if not Duplicates (I) then
                     Result.Items (Next) := Item;
                     Next := Next + 1;
                  end if;
               end;
            end loop;
            pragma Assert (Next = Result.N + 1);
            return Result;
         end;
      end Make_Unique;
      package Internal_Entity_Compilation_Unit_Vectors is new Langkit_Support.Vectors (Internal_Entity_Compilation_Unit);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Compilation_Unit_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Compilation_Unit
   is
      function Absolute_Get
        (T : Internal_Entity_Compilation_Unit_Array_Access; Index : Integer)
         return Internal_Entity_Compilation_Unit
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Compilation_Unit,
         Sequence_Type => Internal_Entity_Compilation_Unit_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Compilation_Unit;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Compilation_Unit;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Compilation_Unit_Array_Access) return Internal_Entity_Compilation_Unit_Array_Access is
      Ret : Internal_Entity_Compilation_Unit_Array_Access := Create_Internal_Entity_Compilation_Unit_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Compilation_Unit_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Compilation_Unit_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Compilation_Unit_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Compilation_Unit_Array (Items_Count : Natural) return Internal_Entity_Compilation_Unit_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Compilation_Unit_Array_Type
       else new Internal_Entity_Compilation_Unit_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Compilation_Unit_Array
     (Items : Internal_Internal_Entity_Compilation_Unit_Array) return Internal_Entity_Compilation_Unit_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Compilation_Unit_Array_Type;
      end if;
      return new Internal_Entity_Compilation_Unit_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Compilation_Unit_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Compilation_Unit_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      function Make_Unique (A : Internal_Entity_Compilation_Unit_Array_Access) return Internal_Entity_Compilation_Unit_Array_Access is
         --  To keep the order of the result deterministic, this function works
         --  in two steps. First, determine for each item in A whether it
         --  duplicate a previous item. Only then, go through all items in A
         --  and add only non-duplicated ones to the result.
         Duplicates : array (A.Items'Range) of Boolean := (others => False);
         --  For each item in A, indicate whether the value is a duplicate of a
         --  previous one in the array.
         Unique_Elements : Natural;
         --  Number of unique elements in A
      begin
         --  Step 1: compute Duplicates
         declare
            package Sets is new Ada.Containers.Hashed_Sets
              (Element_Type        => Internal_Entity_Compilation_Unit,
               "="                 => "=",
               Equivalent_Elements => "=",
               Hash                => Hash);
            Set : Sets.Set;
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Entity_Compilation_Unit renames A.Items (I);
               begin
                  if Set.Contains (Item) then
                     Duplicates (I) := True;
                  else
                     Set.Insert (Item);
                  end if;
               end;
            end loop;
            Unique_Elements := Natural (Set.Length);
         end;
         --  Step 2: create the result
         declare
            Next   : Positive := 1;
            Result : constant Internal_Entity_Compilation_Unit_Array_Access :=
               Create_Internal_Entity_Compilation_Unit_Array (Unique_Elements);
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Entity_Compilation_Unit renames A.Items (I);
               begin
                  if not Duplicates (I) then
                     Result.Items (Next) := Item;
                     Next := Next + 1;
                  end if;
               end;
            end loop;
            pragma Assert (Next = Result.N + 1);
            return Result;
         end;
      end Make_Unique;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Declarative_Part_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Declarative_Part
   is
      function Absolute_Get
        (T : Internal_Entity_Declarative_Part_Array_Access; Index : Integer)
         return Internal_Entity_Declarative_Part
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Declarative_Part,
         Sequence_Type => Internal_Entity_Declarative_Part_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Declarative_Part;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Declarative_Part;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Declarative_Part_Array_Access) return Internal_Entity_Declarative_Part_Array_Access is
      Ret : Internal_Entity_Declarative_Part_Array_Access := Create_Internal_Entity_Declarative_Part_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Declarative_Part_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Declarative_Part_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Declarative_Part_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Declarative_Part_Array (Items_Count : Natural) return Internal_Entity_Declarative_Part_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Declarative_Part_Array_Type
       else new Internal_Entity_Declarative_Part_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Declarative_Part_Array
     (Items : Internal_Internal_Entity_Declarative_Part_Array) return Internal_Entity_Declarative_Part_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Declarative_Part_Array_Type;
      end if;
      return new Internal_Entity_Declarative_Part_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Declarative_Part_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Declarative_Part_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Defining_Name_Vectors is new Langkit_Support.Vectors (Internal_Entity_Defining_Name);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Defining_Name_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Defining_Name
   is
      function Absolute_Get
        (T : Internal_Entity_Defining_Name_Array_Access; Index : Integer)
         return Internal_Entity_Defining_Name
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Defining_Name,
         Sequence_Type => Internal_Entity_Defining_Name_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Defining_Name;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Defining_Name;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Defining_Name_Array_Access) return Internal_Entity_Defining_Name_Array_Access is
      Ret : Internal_Entity_Defining_Name_Array_Access := Create_Internal_Entity_Defining_Name_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Defining_Name_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Defining_Name_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Defining_Name_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Defining_Name_Array (Items_Count : Natural) return Internal_Entity_Defining_Name_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Defining_Name_Array_Type
       else new Internal_Entity_Defining_Name_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Defining_Name_Array
     (Items : Internal_Internal_Entity_Defining_Name_Array) return Internal_Entity_Defining_Name_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Defining_Name_Array_Type;
      end if;
      return new Internal_Entity_Defining_Name_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Defining_Name_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Defining_Name_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Expr_Vectors is new Langkit_Support.Vectors (Internal_Entity_Expr);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Expr_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Expr
   is
      function Absolute_Get
        (T : Internal_Entity_Expr_Array_Access; Index : Integer)
         return Internal_Entity_Expr
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Expr,
         Sequence_Type => Internal_Entity_Expr_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Expr;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Expr;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Expr_Array_Access) return Internal_Entity_Expr_Array_Access is
      Ret : Internal_Entity_Expr_Array_Access := Create_Internal_Entity_Expr_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Expr_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Expr_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Expr_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Expr_Array (Items_Count : Natural) return Internal_Entity_Expr_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Expr_Array_Type
       else new Internal_Entity_Expr_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Expr_Array
     (Items : Internal_Internal_Entity_Expr_Array) return Internal_Entity_Expr_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Expr_Array_Type;
      end if;
      return new Internal_Entity_Expr_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Expr_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Expr_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Generic_Instantiation_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Generic_Instantiation
   is
      function Absolute_Get
        (T : Internal_Entity_Generic_Instantiation_Array_Access; Index : Integer)
         return Internal_Entity_Generic_Instantiation
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Generic_Instantiation,
         Sequence_Type => Internal_Entity_Generic_Instantiation_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Generic_Instantiation;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Generic_Instantiation;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Generic_Instantiation_Array_Access) return Internal_Entity_Generic_Instantiation_Array_Access is
      Ret : Internal_Entity_Generic_Instantiation_Array_Access := Create_Internal_Entity_Generic_Instantiation_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Generic_Instantiation_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Generic_Instantiation_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Generic_Instantiation_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Generic_Instantiation_Array (Items_Count : Natural) return Internal_Entity_Generic_Instantiation_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Generic_Instantiation_Array_Type
       else new Internal_Entity_Generic_Instantiation_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Generic_Instantiation_Array
     (Items : Internal_Internal_Entity_Generic_Instantiation_Array) return Internal_Entity_Generic_Instantiation_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Generic_Instantiation_Array_Type;
      end if;
      return new Internal_Entity_Generic_Instantiation_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Generic_Instantiation_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Generic_Instantiation_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Identifier_Vectors is new Langkit_Support.Vectors (Internal_Entity_Identifier);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Identifier_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Identifier
   is
      function Absolute_Get
        (T : Internal_Entity_Identifier_Array_Access; Index : Integer)
         return Internal_Entity_Identifier
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Identifier,
         Sequence_Type => Internal_Entity_Identifier_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Identifier;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Identifier;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Identifier_Array_Access) return Internal_Entity_Identifier_Array_Access is
      Ret : Internal_Entity_Identifier_Array_Access := Create_Internal_Entity_Identifier_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Identifier_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Identifier_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Identifier_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Identifier_Array (Items_Count : Natural) return Internal_Entity_Identifier_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Identifier_Array_Type
       else new Internal_Entity_Identifier_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Identifier_Array
     (Items : Internal_Internal_Entity_Identifier_Array) return Internal_Entity_Identifier_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Identifier_Array_Type;
      end if;
      return new Internal_Entity_Identifier_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Identifier_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Identifier_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Mode_Vectors is new Langkit_Support.Vectors (Internal_Entity_Mode);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Mode_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Mode
   is
      function Absolute_Get
        (T : Internal_Entity_Mode_Array_Access; Index : Integer)
         return Internal_Entity_Mode
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Mode,
         Sequence_Type => Internal_Entity_Mode_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Mode;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Mode;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Mode_Array_Access) return Internal_Entity_Mode_Array_Access is
      Ret : Internal_Entity_Mode_Array_Access := Create_Internal_Entity_Mode_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Mode_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Mode_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Mode_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Mode_Array (Items_Count : Natural) return Internal_Entity_Mode_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Mode_Array_Type
       else new Internal_Entity_Mode_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Mode_Array
     (Items : Internal_Internal_Entity_Mode_Array) return Internal_Entity_Mode_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Mode_Array_Type;
      end if;
      return new Internal_Entity_Mode_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Mode_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Mode_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Name_Vectors is new Langkit_Support.Vectors (Internal_Entity_Name);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Name_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Name
   is
      function Absolute_Get
        (T : Internal_Entity_Name_Array_Access; Index : Integer)
         return Internal_Entity_Name
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Name,
         Sequence_Type => Internal_Entity_Name_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Name;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Name;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Name_Array_Access) return Internal_Entity_Name_Array_Access is
      Ret : Internal_Entity_Name_Array_Access := Create_Internal_Entity_Name_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Name_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Name_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Name_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Name_Array (Items_Count : Natural) return Internal_Entity_Name_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Name_Array_Type
       else new Internal_Entity_Name_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Name_Array
     (Items : Internal_Internal_Entity_Name_Array) return Internal_Entity_Name_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Name_Array_Type;
      end if;
      return new Internal_Entity_Name_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Name_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Name_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Param_Spec_Vectors is new Langkit_Support.Vectors (Internal_Entity_Param_Spec);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Param_Spec_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Param_Spec
   is
      function Absolute_Get
        (T : Internal_Entity_Param_Spec_Array_Access; Index : Integer)
         return Internal_Entity_Param_Spec
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Param_Spec,
         Sequence_Type => Internal_Entity_Param_Spec_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Param_Spec;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Param_Spec;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Param_Spec_Array_Access) return Internal_Entity_Param_Spec_Array_Access is
      Ret : Internal_Entity_Param_Spec_Array_Access := Create_Internal_Entity_Param_Spec_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Param_Spec_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Param_Spec_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Param_Spec_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Param_Spec_Array (Items_Count : Natural) return Internal_Entity_Param_Spec_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Param_Spec_Array_Type
       else new Internal_Entity_Param_Spec_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Param_Spec_Array
     (Items : Internal_Internal_Entity_Param_Spec_Array) return Internal_Entity_Param_Spec_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Param_Spec_Array_Type;
      end if;
      return new Internal_Entity_Param_Spec_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Param_Spec_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Param_Spec_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Pragma_Node_Vectors is new Langkit_Support.Vectors (Internal_Entity_Pragma_Node);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Pragma_Node_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Pragma_Node
   is
      function Absolute_Get
        (T : Internal_Entity_Pragma_Node_Array_Access; Index : Integer)
         return Internal_Entity_Pragma_Node
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Pragma_Node,
         Sequence_Type => Internal_Entity_Pragma_Node_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Pragma_Node;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Pragma_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Pragma_Node_Array_Access) return Internal_Entity_Pragma_Node_Array_Access is
      Ret : Internal_Entity_Pragma_Node_Array_Access := Create_Internal_Entity_Pragma_Node_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Pragma_Node_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Pragma_Node_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Pragma_Node_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Pragma_Node_Array (Items_Count : Natural) return Internal_Entity_Pragma_Node_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Pragma_Node_Array_Type
       else new Internal_Entity_Pragma_Node_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Pragma_Node_Array
     (Items : Internal_Internal_Entity_Pragma_Node_Array) return Internal_Entity_Pragma_Node_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Pragma_Node_Array_Type;
      end if;
      return new Internal_Entity_Pragma_Node_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Pragma_Node_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Pragma_Node_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Type_Decl_Vectors is new Langkit_Support.Vectors (Internal_Entity_Type_Decl);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Type_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Type_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Type_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Type_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Type_Decl,
         Sequence_Type => Internal_Entity_Type_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Type_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Type_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Type_Decl_Array_Access) return Internal_Entity_Type_Decl_Array_Access is
      Ret : Internal_Entity_Type_Decl_Array_Access := Create_Internal_Entity_Type_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Type_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Type_Decl_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Type_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Type_Decl_Array (Items_Count : Natural) return Internal_Entity_Type_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Type_Decl_Array_Type
       else new Internal_Entity_Type_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Type_Decl_Array
     (Items : Internal_Internal_Entity_Type_Decl_Array) return Internal_Entity_Type_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Type_Decl_Array_Type;
      end if;
      return new Internal_Entity_Type_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Type_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Type_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Entity_Type_Expr_Vectors is new Langkit_Support.Vectors (Internal_Entity_Type_Expr);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Entity_Type_Expr_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Type_Expr
   is
      function Absolute_Get
        (T : Internal_Entity_Type_Expr_Array_Access; Index : Integer)
         return Internal_Entity_Type_Expr
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Entity_Type_Expr,
         Sequence_Type => Internal_Entity_Type_Expr_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Entity_Type_Expr;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Type_Expr;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Entity_Type_Expr_Array_Access) return Internal_Entity_Type_Expr_Array_Access is
      Ret : Internal_Entity_Type_Expr_Array_Access := Create_Internal_Entity_Type_Expr_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Type_Expr_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Entity_Type_Expr_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Type_Expr_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Entity_Type_Expr_Array (Items_Count : Natural) return Internal_Entity_Type_Expr_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Type_Expr_Array_Type
       else new Internal_Entity_Type_Expr_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Entity_Type_Expr_Array
     (Items : Internal_Internal_Entity_Type_Expr_Array) return Internal_Entity_Type_Expr_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Type_Expr_Array_Type;
      end if;
      return new Internal_Entity_Type_Expr_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Entity_Type_Expr_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Type_Expr_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Env_Assoc_Vectors is new Langkit_Support.Vectors (Internal_Env_Assoc);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Env_Assoc_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Env_Assoc
   is
      function Absolute_Get
        (T : Internal_Env_Assoc_Array_Access; Index : Integer)
         return Internal_Env_Assoc
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Env_Assoc,
         Sequence_Type => Internal_Env_Assoc_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Env_Assoc;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Env_Assoc;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Env_Assoc_Array_Access) return Internal_Env_Assoc_Array_Access is
      Ret : Internal_Env_Assoc_Array_Access := Create_Internal_Env_Assoc_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Env_Assoc_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Env_Assoc_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Env_Assoc_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Env_Assoc_Array (Items_Count : Natural) return Internal_Env_Assoc_Array_Access
   is (if Items_Count = 0
       then No_Internal_Env_Assoc_Array_Type
       else new Internal_Env_Assoc_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Env_Assoc_Array
     (Items : Internal_Internal_Env_Assoc_Array) return Internal_Env_Assoc_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Env_Assoc_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Env_Assoc_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Env_Assoc_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Env_Assoc_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Expected_Type_For_Expr_Vectors is new Langkit_Support.Vectors (Internal_Expected_Type_For_Expr);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Expected_Type_For_Expr_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Expected_Type_For_Expr
   is
      function Absolute_Get
        (T : Internal_Expected_Type_For_Expr_Array_Access; Index : Integer)
         return Internal_Expected_Type_For_Expr
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Expected_Type_For_Expr,
         Sequence_Type => Internal_Expected_Type_For_Expr_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Expected_Type_For_Expr;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Expected_Type_For_Expr;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Expected_Type_For_Expr_Array_Access) return Internal_Expected_Type_For_Expr_Array_Access is
      Ret : Internal_Expected_Type_For_Expr_Array_Access := Create_Internal_Expected_Type_For_Expr_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Expected_Type_For_Expr_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Expected_Type_For_Expr_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Expected_Type_For_Expr_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Expected_Type_For_Expr_Array (Items_Count : Natural) return Internal_Expected_Type_For_Expr_Array_Access
   is (if Items_Count = 0
       then No_Internal_Expected_Type_For_Expr_Array_Type
       else new Internal_Expected_Type_For_Expr_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Expected_Type_For_Expr_Array
     (Items : Internal_Internal_Expected_Type_For_Expr_Array) return Internal_Expected_Type_For_Expr_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Expected_Type_For_Expr_Array_Type;
      end if;
      return new Internal_Expected_Type_For_Expr_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Expected_Type_For_Expr_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Expected_Type_For_Expr_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Inner_Env_Assoc_Vectors is new Langkit_Support.Vectors (Internal_Inner_Env_Assoc);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Inner_Env_Assoc_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Inner_Env_Assoc
   is
      function Absolute_Get
        (T : Internal_Inner_Env_Assoc_Array_Access; Index : Integer)
         return Internal_Inner_Env_Assoc
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Inner_Env_Assoc,
         Sequence_Type => Internal_Inner_Env_Assoc_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Inner_Env_Assoc;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Inner_Env_Assoc;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Inner_Env_Assoc_Array_Access) return Internal_Inner_Env_Assoc_Array_Access is
      Ret : Internal_Inner_Env_Assoc_Array_Access := Create_Internal_Inner_Env_Assoc_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Inner_Env_Assoc_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Inner_Env_Assoc_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Inner_Env_Assoc_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Inner_Env_Assoc_Array (Items_Count : Natural) return Internal_Inner_Env_Assoc_Array_Access
   is (if Items_Count = 0
       then No_Internal_Inner_Env_Assoc_Array_Type
       else new Internal_Inner_Env_Assoc_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Inner_Env_Assoc_Array
     (Items : Internal_Internal_Inner_Env_Assoc_Array) return Internal_Inner_Env_Assoc_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Inner_Env_Assoc_Array_Type;
      end if;
      return new Internal_Inner_Env_Assoc_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Inner_Env_Assoc_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Inner_Env_Assoc_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Logic_Context_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Logic_Context
   is
      function Absolute_Get
        (T : Internal_Logic_Context_Array_Access; Index : Integer)
         return Internal_Logic_Context
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Logic_Context,
         Sequence_Type => Internal_Logic_Context_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Logic_Context;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Logic_Context;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Logic_Context_Array_Access) return Internal_Logic_Context_Array_Access is
      Ret : Internal_Logic_Context_Array_Access := Create_Internal_Logic_Context_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Logic_Context_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Logic_Context_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Logic_Context_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Logic_Context_Array (Items_Count : Natural) return Internal_Logic_Context_Array_Access
   is (if Items_Count = 0
       then No_Internal_Logic_Context_Array_Type
       else new Internal_Logic_Context_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Logic_Context_Array
     (Items : Internal_Internal_Logic_Context_Array) return Internal_Logic_Context_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Logic_Context_Array_Type;
      end if;
      return new Internal_Logic_Context_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Logic_Context_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Logic_Context_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Param_Actual_Vectors is new Langkit_Support.Vectors (Internal_Param_Actual);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Param_Actual_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Param_Actual
   is
      function Absolute_Get
        (T : Internal_Param_Actual_Array_Access; Index : Integer)
         return Internal_Param_Actual
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Param_Actual,
         Sequence_Type => Internal_Param_Actual_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Param_Actual;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Param_Actual;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Param_Actual_Array_Access) return Internal_Param_Actual_Array_Access is
      Ret : Internal_Param_Actual_Array_Access := Create_Internal_Param_Actual_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Param_Actual_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Param_Actual_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Param_Actual_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Param_Actual_Array (Items_Count : Natural) return Internal_Param_Actual_Array_Access
   is (if Items_Count = 0
       then No_Internal_Param_Actual_Array_Type
       else new Internal_Param_Actual_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Param_Actual_Array
     (Items : Internal_Internal_Param_Actual_Array) return Internal_Param_Actual_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Param_Actual_Array_Type;
      end if;
      return new Internal_Param_Actual_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Param_Actual_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Param_Actual_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Param_Match_Vectors is new Langkit_Support.Vectors (Internal_Param_Match);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Param_Match_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Param_Match
   is
      function Absolute_Get
        (T : Internal_Param_Match_Array_Access; Index : Integer)
         return Internal_Param_Match
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Param_Match,
         Sequence_Type => Internal_Param_Match_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Param_Match;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Param_Match;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Param_Match_Array_Access) return Internal_Param_Match_Array_Access is
      Ret : Internal_Param_Match_Array_Access := Create_Internal_Param_Match_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Param_Match_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Param_Match_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Param_Match_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Param_Match_Array (Items_Count : Natural) return Internal_Param_Match_Array_Access
   is (if Items_Count = 0
       then No_Internal_Param_Match_Array_Type
       else new Internal_Param_Match_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Param_Match_Array
     (Items : Internal_Internal_Param_Match_Array) return Internal_Param_Match_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Param_Match_Array_Type;
      end if;
      return new Internal_Param_Match_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Param_Match_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Param_Match_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Ref_Result_Vectors is new Langkit_Support.Vectors (Internal_Ref_Result);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Ref_Result_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Ref_Result
   is
      function Absolute_Get
        (T : Internal_Ref_Result_Array_Access; Index : Integer)
         return Internal_Ref_Result
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Ref_Result,
         Sequence_Type => Internal_Ref_Result_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Ref_Result;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Ref_Result;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Ref_Result_Array_Access) return Internal_Ref_Result_Array_Access is
      Ret : Internal_Ref_Result_Array_Access := Create_Internal_Ref_Result_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Ref_Result_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Ref_Result_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Ref_Result_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Ref_Result_Array (Items_Count : Natural) return Internal_Ref_Result_Array_Access
   is (if Items_Count = 0
       then No_Internal_Ref_Result_Array_Type
       else new Internal_Ref_Result_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Ref_Result_Array
     (Items : Internal_Internal_Ref_Result_Array) return Internal_Ref_Result_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Ref_Result_Array_Type;
      end if;
      return new Internal_Ref_Result_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Ref_Result_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Ref_Result_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Shape_Vectors is new Langkit_Support.Vectors (Internal_Shape);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Shape_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Shape
   is
      function Absolute_Get
        (T : Internal_Shape_Array_Access; Index : Integer)
         return Internal_Shape
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Shape,
         Sequence_Type => Internal_Shape_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Shape;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Shape;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Shape_Array_Access) return Internal_Shape_Array_Access is
      Ret : Internal_Shape_Array_Access := Create_Internal_Shape_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Shape_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Shape_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Shape_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Shape_Array (Items_Count : Natural) return Internal_Shape_Array_Access
   is (if Items_Count = 0
       then No_Internal_Shape_Array_Type
       else new Internal_Shape_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Shape_Array
     (Items : Internal_Internal_Shape_Array) return Internal_Shape_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Shape_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Shape_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Shape_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Shape_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Single_Actual_Vectors is new Langkit_Support.Vectors (Internal_Single_Actual);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Single_Actual_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Single_Actual
   is
      function Absolute_Get
        (T : Internal_Single_Actual_Array_Access; Index : Integer)
         return Internal_Single_Actual
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Single_Actual,
         Sequence_Type => Internal_Single_Actual_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Single_Actual;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Single_Actual;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Single_Actual_Array_Access) return Internal_Single_Actual_Array_Access is
      Ret : Internal_Single_Actual_Array_Access := Create_Internal_Single_Actual_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Single_Actual_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Single_Actual_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Single_Actual_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Single_Actual_Array (Items_Count : Natural) return Internal_Single_Actual_Array_Access
   is (if Items_Count = 0
       then No_Internal_Single_Actual_Array_Type
       else new Internal_Single_Actual_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Single_Actual_Array
     (Items : Internal_Internal_Single_Actual_Array) return Internal_Single_Actual_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Single_Actual_Array_Type;
      end if;
      return new Internal_Single_Actual_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Single_Actual_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Single_Actual_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Solver_Diagnostic_Vectors is new Langkit_Support.Vectors (Internal_Solver_Diagnostic);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Solver_Diagnostic_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Solver_Diagnostic
   is
      function Absolute_Get
        (T : Internal_Solver_Diagnostic_Array_Access; Index : Integer)
         return Internal_Solver_Diagnostic
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Solver_Diagnostic,
         Sequence_Type => Internal_Solver_Diagnostic_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Solver_Diagnostic;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Solver_Diagnostic;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Solver_Diagnostic_Array_Access) return Internal_Solver_Diagnostic_Array_Access is
      Ret : Internal_Solver_Diagnostic_Array_Access := Create_Internal_Solver_Diagnostic_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Solver_Diagnostic_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Solver_Diagnostic_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Solver_Diagnostic_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Solver_Diagnostic_Array (Items_Count : Natural) return Internal_Solver_Diagnostic_Array_Access
   is (if Items_Count = 0
       then No_Internal_Solver_Diagnostic_Array_Type
       else new Internal_Solver_Diagnostic_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Solver_Diagnostic_Array
     (Items : Internal_Internal_Solver_Diagnostic_Array) return Internal_Solver_Diagnostic_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Solver_Diagnostic_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Solver_Diagnostic_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Solver_Diagnostic_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Solver_Diagnostic_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Substitution_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Substitution
   is
      function Absolute_Get
        (T : Internal_Substitution_Array_Access; Index : Integer)
         return Internal_Substitution
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Substitution,
         Sequence_Type => Internal_Substitution_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Substitution;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Substitution;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Substitution_Array_Access) return Internal_Substitution_Array_Access is
      Ret : Internal_Substitution_Array_Access := Create_Internal_Substitution_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Substitution_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Substitution_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Substitution_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Substitution_Array (Items_Count : Natural) return Internal_Substitution_Array_Access
   is (if Items_Count = 0
       then No_Internal_Substitution_Array_Type
       else new Internal_Substitution_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Substitution_Array
     (Items : Internal_Internal_Substitution_Array) return Internal_Substitution_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Substitution_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Substitution_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Substitution_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Substitution_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Internal_Unit_Vectors is new Langkit_Support.Vectors (Internal_Unit);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_Unit_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Unit
   is
      function Absolute_Get
        (T : Internal_Unit_Array_Access; Index : Integer)
         return Internal_Unit
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_Unit,
         Sequence_Type => Internal_Unit_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_Unit;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return null;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_Unit_Array_Access) return Internal_Unit_Array_Access is
      Ret : Internal_Unit_Array_Access := Create_Internal_Unit_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Unit_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_Unit_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Unit_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_Unit_Array (Items_Count : Natural) return Internal_Unit_Array_Access
   is (if Items_Count = 0
       then No_Internal_Unit_Array_Type
       else new Internal_Unit_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_Unit_Array
     (Items : Internal_Internal_Unit_Array) return Internal_Unit_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Unit_Array_Type;
      end if;
      return new Internal_Unit_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_Unit_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Unit_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      function Make_Unique (A : Internal_Unit_Array_Access) return Internal_Unit_Array_Access is
         --  To keep the order of the result deterministic, this function works
         --  in two steps. First, determine for each item in A whether it
         --  duplicate a previous item. Only then, go through all items in A
         --  and add only non-duplicated ones to the result.
         Duplicates : array (A.Items'Range) of Boolean := (others => False);
         --  For each item in A, indicate whether the value is a duplicate of a
         --  previous one in the array.
         Unique_Elements : Natural;
         --  Number of unique elements in A
      begin
         --  Step 1: compute Duplicates
         declare
            package Sets is new Ada.Containers.Hashed_Sets
              (Element_Type        => Internal_Unit,
               "="                 => "=",
               Equivalent_Elements => "=",
               Hash                => Hash);
            Set : Sets.Set;
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Unit renames A.Items (I);
               begin
                  if Set.Contains (Item) then
                     Duplicates (I) := True;
                  else
                     Set.Insert (Item);
                  end if;
               end;
            end loop;
            Unique_Elements := Natural (Set.Length);
         end;
         --  Step 2: create the result
         declare
            Next   : Positive := 1;
            Result : constant Internal_Unit_Array_Access :=
               Create_Internal_Unit_Array (Unique_Elements);
         begin
            for I in A.Items'Range loop
               declare
                  Item : Internal_Unit renames A.Items (I);
               begin
                  if not Duplicates (I) then
                     Result.Items (Next) := Item;
                     Next := Next + 1;
                  end if;
               end;
            end loop;
            pragma Assert (Next = Result.N + 1);
            return Result;
         end;
      end Make_Unique;
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Internal_User_Defined_Function_Subp_Spec_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_User_Defined_Function_Subp_Spec
   is
      function Absolute_Get
        (T : Internal_User_Defined_Function_Subp_Spec_Array_Access; Index : Integer)
         return Internal_User_Defined_Function_Subp_Spec
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Internal_User_Defined_Function_Subp_Spec,
         Sequence_Type => Internal_User_Defined_Function_Subp_Spec_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Internal_User_Defined_Function_Subp_Spec;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_User_Defined_Function_Subp_Spec;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Internal_User_Defined_Function_Subp_Spec_Array_Access) return Internal_User_Defined_Function_Subp_Spec_Array_Access is
      Ret : Internal_User_Defined_Function_Subp_Spec_Array_Access := Create_Internal_User_Defined_Function_Subp_Spec_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_User_Defined_Function_Subp_Spec_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Internal_User_Defined_Function_Subp_Spec_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_User_Defined_Function_Subp_Spec_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Internal_User_Defined_Function_Subp_Spec_Array (Items_Count : Natural) return Internal_User_Defined_Function_Subp_Spec_Array_Access
   is (if Items_Count = 0
       then No_Internal_User_Defined_Function_Subp_Spec_Array_Type
       else new Internal_User_Defined_Function_Subp_Spec_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Internal_User_Defined_Function_Subp_Spec_Array
     (Items : Internal_Internal_User_Defined_Function_Subp_Spec_Array) return Internal_User_Defined_Function_Subp_Spec_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_User_Defined_Function_Subp_Spec_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_User_Defined_Function_Subp_Spec_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Internal_User_Defined_Function_Subp_Spec_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_User_Defined_Function_Subp_Spec_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Lexical_Env_Vectors is new Langkit_Support.Vectors (Lexical_Env);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Lexical_Env_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Lexical_Env
   is
      function Absolute_Get
        (T : Lexical_Env_Array_Access; Index : Integer)
         return Lexical_Env
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Lexical_Env,
         Sequence_Type => Lexical_Env_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Lexical_Env;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return Empty_Env;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Lexical_Env_Array_Access) return Lexical_Env_Array_Access is
      Ret : Lexical_Env_Array_Access := Create_Lexical_Env_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Lexical_Env_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Lexical_Env_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Lexical_Env_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Lexical_Env_Array (Items_Count : Natural) return Lexical_Env_Array_Access
   is (if Items_Count = 0
       then No_Lexical_Env_Array_Type
       else new Lexical_Env_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Lexical_Env_Array
     (Items : Internal_Lexical_Env_Array) return Lexical_Env_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Lexical_Env_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Lexical_Env_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Lexical_Env_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Lexical_Env_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Logic_Equation_Vectors is new Langkit_Support.Vectors (Logic_Equation);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Logic_Equation_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Logic_Equation
   is
      function Absolute_Get
        (T : Logic_Equation_Array_Access; Index : Integer)
         return Logic_Equation
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Logic_Equation,
         Sequence_Type => Logic_Equation_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Logic_Equation;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return Null_Logic_Equation;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Logic_Equation_Array_Access) return Logic_Equation_Array_Access is
      Ret : Logic_Equation_Array_Access := Create_Logic_Equation_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Logic_Equation_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Logic_Equation_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Logic_Equation_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Logic_Equation_Array (Items_Count : Natural) return Logic_Equation_Array_Access
   is (if Items_Count = 0
       then No_Logic_Equation_Array_Type
       else new Logic_Equation_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Logic_Equation_Array
     (Items : Internal_Logic_Equation_Array) return Logic_Equation_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Logic_Equation_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Logic_Equation_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Logic_Equation_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Logic_Equation_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package String_Type_Vectors is new Langkit_Support.Vectors (String_Type);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : String_Type_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return String_Type
   is
      function Absolute_Get
        (T : String_Type_Array_Access; Index : Integer)
         return String_Type
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => String_Type,
         Sequence_Type => String_Type_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : String_Type;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return Empty_String;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : String_Type_Array_Access) return String_Type_Array_Access is
      Ret : String_Type_Array_Access := Create_String_Type_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;
      ------------------
      -- Join_Strings --
      ------------------
      function Join_Strings
        (Separator : String_Type;
         Strings   : String_Type_Array_Access) return String_Type
      is
         Separator_Length : constant Natural := Separator.Length;
         Length           : Natural := 0;
         First            : Boolean;
      begin
         --  First, compute the length of the result: the sum of all string
         --  lengths in Strings.
         First := True;
         for S of Strings.Items loop
            if First then
               First := False;
            else
               Length := Length + Separator_Length;
            end if;
            Length := Length + S.Length;
         end loop;
         --  Create the result string with the correct length. Do not use our
         --  constructor and initialize the content in-place, to avoid extra
         --  copies.
         return Result : constant String_Type :=
            new String_Record (Length)
         do
            Result.Ref_Count := 1;
            --  Now copy the content of all strings into the result
            declare
               Last : Natural := 0;
            begin
               First := True;
               for S of Strings.Items loop
                  if First then
                     First := False;
                  else
                     Result.Content (Last + 1 .. Last + Separator_Length) :=
                        Separator.Content;
                     Last := Last + Separator_Length;
                  end if;
                  Result.Content (Last + 1 .. Last + S.Length) := S.Content;
                  Last := Last + S.Length;
               end loop;
            end;
         end return;
      end Join_Strings;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : String_Type_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : String_Type_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out String_Type_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_String_Type_Array (Items_Count : Natural) return String_Type_Array_Access
   is (if Items_Count = 0
       then No_String_Type_Array_Type
       else new String_Type_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_String_Type_Array
     (Items : Internal_String_Type_Array) return String_Type_Array_Access is
   begin
      if Items'Length = 0 then
         return No_String_Type_Array_Type;
      end if;
         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new String_Type_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : String_Type_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : String_Type_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      package Symbol_Type_Vectors is new Langkit_Support.Vectors (Symbol_Type);
   ---------
   -- Get --
   ---------
   function Get
     (Node    : Bare_Ada_Node;
      T       : Symbol_Type_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Symbol_Type
   is
      function Absolute_Get
        (T : Symbol_Type_Array_Access; Index : Integer)
         return Symbol_Type
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Symbol_Type,
         Sequence_Type => Symbol_Type_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Symbol_Type;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Symbol;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;
   ------------
   -- Concat --
   ------------
   function Concat (L, R : Symbol_Type_Array_Access) return Symbol_Type_Array_Access is
      Ret : Symbol_Type_Array_Access := Create_Symbol_Type_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Symbol_Type_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   ------------
   -- Length --
   ------------
   function Length (T : Symbol_Type_Array_Access) return Natural is (T.N);
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Symbol_Type_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
   function Create_Symbol_Type_Array (Items_Count : Natural) return Symbol_Type_Array_Access
   is (if Items_Count = 0
       then No_Symbol_Type_Array_Type
       else new Symbol_Type_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));
   function Create_Symbol_Type_Array
     (Items : Internal_Symbol_Type_Array) return Symbol_Type_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Symbol_Type_Array_Type;
      end if;
      return new Symbol_Type_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (L, R : Symbol_Type_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;
      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;
      return True;
   end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Symbol_Type_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;
      function Make_Unique (A : Symbol_Type_Array_Access) return Symbol_Type_Array_Access is
         --  To keep the order of the result deterministic, this function works
         --  in two steps. First, determine for each item in A whether it
         --  duplicate a previous item. Only then, go through all items in A
         --  and add only non-duplicated ones to the result.
         Duplicates : array (A.Items'Range) of Boolean := (others => False);
         --  For each item in A, indicate whether the value is a duplicate of a
         --  previous one in the array.
         Unique_Elements : Natural;
         --  Number of unique elements in A
      begin
         --  Step 1: compute Duplicates
         declare
            package Sets is new Ada.Containers.Hashed_Sets
              (Element_Type        => Symbol_Type,
               "="                 => "=",
               Equivalent_Elements => "=",
               Hash                => Hash);
            Set : Sets.Set;
         begin
            for I in A.Items'Range loop
               declare
                  Item : Symbol_Type renames A.Items (I);
               begin
                  if Set.Contains (Item) then
                     Duplicates (I) := True;
                  else
                     Set.Insert (Item);
                  end if;
               end;
            end loop;
            Unique_Elements := Natural (Set.Length);
         end;
         --  Step 2: create the result
         declare
            Next   : Positive := 1;
            Result : constant Symbol_Type_Array_Access :=
               Create_Symbol_Type_Array (Unique_Elements);
         begin
            for I in A.Items'Range loop
               declare
                  Item : Symbol_Type renames A.Items (I);
               begin
                  if not Duplicates (I) then
                     Result.Items (Next) := Item;
                     Next := Next + 1;
                  end if;
               end;
            end loop;
            pragma Assert (Next = Result.N + 1);
            return Result;
         end;
      end Make_Unique;
   ----------
   -- Next --
   ----------
   function Next
     (T       : Bare_Ada_Node_Iterator_Access;
      Element : out Bare_Ada_Node) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);
      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Bare_Ada_Node_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Bare_Ada_Node_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Bare_Ada_Node_Iterator_Access) return String is
      begin
         return "<Iterator of AdaNode, index="
                & A.Index'Image & ">";
      end Trace_Image;
   ----------
   -- Next --
   ----------
   function Next
     (T       : Internal_Completion_Item_Iterator_Access;
      Element : out Internal_Completion_Item) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);
      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Completion_Item_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Completion_Item_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Completion_Item_Iterator_Access) return String is
      begin
         return "<Iterator of CompletionItem, index="
                & A.Index'Image & ">";
      end Trace_Image;
   ----------
   -- Next --
   ----------
   function Next
     (T       : Internal_Entity_Iterator_Access;
      Element : out Internal_Entity) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);
      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Entity_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Entity_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Entity_Iterator_Access) return String is
      begin
         return "<Iterator of AdaNode.entity, index="
                & A.Index'Image & ">";
      end Trace_Image;
   ----------
   -- Next --
   ----------
   function Next
     (T       : Internal_Inner_Env_Assoc_Iterator_Access;
      Element : out Internal_Inner_Env_Assoc) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);
      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (T : Internal_Inner_Env_Assoc_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (T : in out Internal_Inner_Env_Assoc_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;
      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (A : Internal_Inner_Env_Assoc_Iterator_Access) return String is
      begin
         return "<Iterator of InnerEnvAssoc, index="
                & A.Index'Image & ">";
      end Trace_Image;
   ---------
   -- "<" --
   ---------
   function "<" (Left, Right : Internal_Unit) return Boolean is
   begin
      return Left.Filename < Right.Filename;
   end "<";
function Hash (Key : Mmz_Key_Item) return Hash_Type;
function Equivalent (L, R : Mmz_Key_Item) return Boolean;
procedure Destroy (Key : in out Mmz_Key_Array_Access);
----------------
-- Equivalent --
----------------
function Equivalent (L, R : Mmz_Key_Item) return Boolean is
begin
   if L.Kind /= R.Kind then
      return False;
   end if;
   case L.Kind is
         when Mmz_Bare_Access_Def =>
               return L.As_Bare_Access_Def = R.As_Bare_Access_Def;
         when Mmz_Bare_Ada_Node =>
               return L.As_Bare_Ada_Node = R.As_Bare_Ada_Node;
         when Mmz_Bare_Anonymous_Expr_Decl =>
               return L.As_Bare_Anonymous_Expr_Decl = R.As_Bare_Anonymous_Expr_Decl;
         when Mmz_Bare_Array_Subcomponent_Choice_Name =>
               return L.As_Bare_Array_Subcomponent_Choice_Name = R.As_Bare_Array_Subcomponent_Choice_Name;
         when Mmz_Bare_Array_Type_Def =>
               return L.As_Bare_Array_Type_Def = R.As_Bare_Array_Type_Def;
         when Mmz_Bare_Assoc_List =>
               return L.As_Bare_Assoc_List = R.As_Bare_Assoc_List;
         when Mmz_Bare_Base_Aggregate =>
               return L.As_Bare_Base_Aggregate = R.As_Bare_Base_Aggregate;
         when Mmz_Bare_Base_Id =>
               return L.As_Bare_Base_Id = R.As_Bare_Base_Id;
         when Mmz_Bare_Base_Subp_Spec =>
               return L.As_Bare_Base_Subp_Spec = R.As_Bare_Base_Subp_Spec;
         when Mmz_Bare_Base_Type_Decl =>
               return L.As_Bare_Base_Type_Decl = R.As_Bare_Base_Type_Decl;
         when Mmz_Bare_Basic_Decl =>
               return L.As_Bare_Basic_Decl = R.As_Bare_Basic_Decl;
         when Mmz_Bare_Body_Node =>
               return L.As_Bare_Body_Node = R.As_Bare_Body_Node;
         when Mmz_Bare_Compilation_Unit =>
               return L.As_Bare_Compilation_Unit = R.As_Bare_Compilation_Unit;
         when Mmz_Bare_Decimal_Fixed_Point_Def =>
               return L.As_Bare_Decimal_Fixed_Point_Def = R.As_Bare_Decimal_Fixed_Point_Def;
         when Mmz_Bare_Defining_Name =>
               return L.As_Bare_Defining_Name = R.As_Bare_Defining_Name;
         when Mmz_Bare_End_Name =>
               return L.As_Bare_End_Name = R.As_Bare_End_Name;
         when Mmz_Bare_Enum_Literal_Decl =>
               return L.As_Bare_Enum_Literal_Decl = R.As_Bare_Enum_Literal_Decl;
         when Mmz_Bare_Enum_Type_Def =>
               return L.As_Bare_Enum_Type_Def = R.As_Bare_Enum_Type_Def;
         when Mmz_Bare_Expr =>
               return L.As_Bare_Expr = R.As_Bare_Expr;
         when Mmz_Bare_For_Loop_Var_Decl =>
               return L.As_Bare_For_Loop_Var_Decl = R.As_Bare_For_Loop_Var_Decl;
         when Mmz_Bare_Formal_Discrete_Type_Def =>
               return L.As_Bare_Formal_Discrete_Type_Def = R.As_Bare_Formal_Discrete_Type_Def;
         when Mmz_Bare_Formal_Subp_Decl =>
               return L.As_Bare_Formal_Subp_Decl = R.As_Bare_Formal_Subp_Decl;
         when Mmz_Bare_Generic_Instantiation =>
               return L.As_Bare_Generic_Instantiation = R.As_Bare_Generic_Instantiation;
         when Mmz_Bare_Mod_Int_Type_Def =>
               return L.As_Bare_Mod_Int_Type_Def = R.As_Bare_Mod_Int_Type_Def;
         when Mmz_Bare_Name =>
               return L.As_Bare_Name = R.As_Bare_Name;
         when Mmz_Bare_Ordinary_Fixed_Point_Def =>
               return L.As_Bare_Ordinary_Fixed_Point_Def = R.As_Bare_Ordinary_Fixed_Point_Def;
         when Mmz_Bare_Private_Type_Def =>
               return L.As_Bare_Private_Type_Def = R.As_Bare_Private_Type_Def;
         when Mmz_Bare_Real_Type_Def =>
               return L.As_Bare_Real_Type_Def = R.As_Bare_Real_Type_Def;
         when Mmz_Bare_Record_Type_Def =>
               return L.As_Bare_Record_Type_Def = R.As_Bare_Record_Type_Def;
         when Mmz_Bare_Signed_Int_Type_Def =>
               return L.As_Bare_Signed_Int_Type_Def = R.As_Bare_Signed_Int_Type_Def;
         when Mmz_Bare_Subtype_Decl =>
               return L.As_Bare_Subtype_Decl = R.As_Bare_Subtype_Decl;
         when Mmz_Bare_Subunit =>
               return L.As_Bare_Subunit = R.As_Bare_Subunit;
         when Mmz_Bare_Task_Type_Decl =>
               return L.As_Bare_Task_Type_Decl = R.As_Bare_Task_Type_Decl;
         when Mmz_Bare_Type_Decl =>
               return L.As_Bare_Type_Decl = R.As_Bare_Type_Decl;
         when Mmz_Bare_Type_Expr =>
               return L.As_Bare_Type_Expr = R.As_Bare_Type_Expr;
         when Mmz_Boolean =>
               return L.As_Boolean = R.As_Boolean;
         when Mmz_Env_Rebindings =>
               return L.As_Env_Rebindings = R.As_Env_Rebindings;
         when Mmz_Integer =>
               return L.As_Integer = R.As_Integer;
         when Mmz_Internal_Entity_Base_Type_Decl_Array_Access =>
               return Equivalent (L.As_Internal_Entity_Base_Type_Decl_Array_Access, R.As_Internal_Entity_Base_Type_Decl_Array_Access);
         when Mmz_Internal_Entity_Expr =>
               return L.As_Internal_Entity_Expr = R.As_Internal_Entity_Expr;
         when Mmz_Internal_Entity_Info =>
               return L.As_Internal_Entity_Info = R.As_Internal_Entity_Info;
         when Mmz_Internal_Metadata =>
               return L.As_Internal_Metadata = R.As_Internal_Metadata;
         when Mmz_Lexical_Env =>
               return Equivalent (L.As_Lexical_Env, R.As_Lexical_Env);
         when Mmz_Symbol_Type =>
               return L.As_Symbol_Type = R.As_Symbol_Type;
   end case;
end Equivalent;
----------
-- Hash --
----------
function Hash (Key : Mmz_Key_Item) return Hash_Type is
begin
   case Key.Kind is
         when Mmz_Bare_Access_Def =>
            return Hash (Key.As_Bare_Access_Def);
         when Mmz_Bare_Ada_Node =>
            return Hash (Key.As_Bare_Ada_Node);
         when Mmz_Bare_Anonymous_Expr_Decl =>
            return Hash (Key.As_Bare_Anonymous_Expr_Decl);
         when Mmz_Bare_Array_Subcomponent_Choice_Name =>
            return Hash (Key.As_Bare_Array_Subcomponent_Choice_Name);
         when Mmz_Bare_Array_Type_Def =>
            return Hash (Key.As_Bare_Array_Type_Def);
         when Mmz_Bare_Assoc_List =>
            return Hash (Key.As_Bare_Assoc_List);
         when Mmz_Bare_Base_Aggregate =>
            return Hash (Key.As_Bare_Base_Aggregate);
         when Mmz_Bare_Base_Id =>
            return Hash (Key.As_Bare_Base_Id);
         when Mmz_Bare_Base_Subp_Spec =>
            return Hash (Key.As_Bare_Base_Subp_Spec);
         when Mmz_Bare_Base_Type_Decl =>
            return Hash (Key.As_Bare_Base_Type_Decl);
         when Mmz_Bare_Basic_Decl =>
            return Hash (Key.As_Bare_Basic_Decl);
         when Mmz_Bare_Body_Node =>
            return Hash (Key.As_Bare_Body_Node);
         when Mmz_Bare_Compilation_Unit =>
            return Hash (Key.As_Bare_Compilation_Unit);
         when Mmz_Bare_Decimal_Fixed_Point_Def =>
            return Hash (Key.As_Bare_Decimal_Fixed_Point_Def);
         when Mmz_Bare_Defining_Name =>
            return Hash (Key.As_Bare_Defining_Name);
         when Mmz_Bare_End_Name =>
            return Hash (Key.As_Bare_End_Name);
         when Mmz_Bare_Enum_Literal_Decl =>
            return Hash (Key.As_Bare_Enum_Literal_Decl);
         when Mmz_Bare_Enum_Type_Def =>
            return Hash (Key.As_Bare_Enum_Type_Def);
         when Mmz_Bare_Expr =>
            return Hash (Key.As_Bare_Expr);
         when Mmz_Bare_For_Loop_Var_Decl =>
            return Hash (Key.As_Bare_For_Loop_Var_Decl);
         when Mmz_Bare_Formal_Discrete_Type_Def =>
            return Hash (Key.As_Bare_Formal_Discrete_Type_Def);
         when Mmz_Bare_Formal_Subp_Decl =>
            return Hash (Key.As_Bare_Formal_Subp_Decl);
         when Mmz_Bare_Generic_Instantiation =>
            return Hash (Key.As_Bare_Generic_Instantiation);
         when Mmz_Bare_Mod_Int_Type_Def =>
            return Hash (Key.As_Bare_Mod_Int_Type_Def);
         when Mmz_Bare_Name =>
            return Hash (Key.As_Bare_Name);
         when Mmz_Bare_Ordinary_Fixed_Point_Def =>
            return Hash (Key.As_Bare_Ordinary_Fixed_Point_Def);
         when Mmz_Bare_Private_Type_Def =>
            return Hash (Key.As_Bare_Private_Type_Def);
         when Mmz_Bare_Real_Type_Def =>
            return Hash (Key.As_Bare_Real_Type_Def);
         when Mmz_Bare_Record_Type_Def =>
            return Hash (Key.As_Bare_Record_Type_Def);
         when Mmz_Bare_Signed_Int_Type_Def =>
            return Hash (Key.As_Bare_Signed_Int_Type_Def);
         when Mmz_Bare_Subtype_Decl =>
            return Hash (Key.As_Bare_Subtype_Decl);
         when Mmz_Bare_Subunit =>
            return Hash (Key.As_Bare_Subunit);
         when Mmz_Bare_Task_Type_Decl =>
            return Hash (Key.As_Bare_Task_Type_Decl);
         when Mmz_Bare_Type_Decl =>
            return Hash (Key.As_Bare_Type_Decl);
         when Mmz_Bare_Type_Expr =>
            return Hash (Key.As_Bare_Type_Expr);
         when Mmz_Boolean =>
            return Hash (Key.As_Boolean);
         when Mmz_Env_Rebindings =>
            return Hash (Key.As_Env_Rebindings);
         when Mmz_Integer =>
            return Hash (Key.As_Integer);
         when Mmz_Internal_Entity_Base_Type_Decl_Array_Access =>
            return Hash (Key.As_Internal_Entity_Base_Type_Decl_Array_Access);
         when Mmz_Internal_Entity_Expr =>
            return Hash (Key.As_Internal_Entity_Expr);
         when Mmz_Internal_Entity_Info =>
            return Hash (Key.As_Internal_Entity_Info);
         when Mmz_Internal_Metadata =>
            return Hash (Key.As_Internal_Metadata);
         when Mmz_Lexical_Env =>
            return Hash (Key.As_Lexical_Env);
         when Mmz_Symbol_Type =>
            return Hash (Key.As_Symbol_Type);
   end case;
end Hash;
----------
-- Hash --
----------
function Hash (Key : Mmz_Key) return Hash_Type is
   Result : Hash_Type := Mmz_Property'Pos (Key.Property);
begin
   for K of Key.Items.all loop
      Result := Combine (Result, Hash (K));
   end loop;
   return Result;
end Hash;
----------------
-- Equivalent --
----------------
function Equivalent (L, R : Mmz_Key) return Boolean is
   L_Items : Mmz_Key_Array renames L.Items.all;
   R_Items : Mmz_Key_Array renames R.Items.all;
begin
   if L.Property /= R.Property or else L_Items'Length /= R_Items'Length then
      return False;
   end if;
   for I in L_Items'Range loop
      if not Equivalent (L_Items (I), R_Items (I)) then
         return False;
      end if;
   end loop;
   return True;
end Equivalent;
-------------
-- Destroy --
-------------
procedure Destroy (Map : in out Memoization_Maps.Map) is
   use Memoization_Maps;
   --  We need keys and values to be valid when clearing the memoization map,
   --  but on the other hand we need to free keys and values as well. To
   --  achieve both goals, we first copy key and values into arrays, then we
   --  clear the map, and then we free keys/values in the arrays. Allocate both
   --  arrays on the heap to avoid stack overflow, as they can be quite big.
   Length : constant Natural := Natural (Map.Length);
   type Key_Array is array (1 .. Length) of Mmz_Key_Array_Access;
   type Key_Array_Access is access Key_Array;
   procedure Free is new Ada.Unchecked_Deallocation
     (Key_Array, Key_Array_Access);
   type Value_Array is array (1 .. Length) of Mmz_Value;
   type Value_Array_Access is access Value_Array;
   procedure Free is new Ada.Unchecked_Deallocation
     (Value_Array, Value_Array_Access);
   Keys   : Key_Array_Access := new Key_Array;
   Values : Value_Array_Access := new Value_Array;
   I      : Positive := 1;
begin
   for Cur in Map.Iterate loop
      Keys (I) := Key (Cur).Items;
      Values (I) := Element (Cur);
      I := I + 1;
   end loop;
   Map.Clear;
   for K_Array of Keys.all loop
      Destroy (K_Array);
   end loop;
   for V of Values.all loop
      case V.Kind is
         when Mmz_Error =>
            Free_Memoized_Error (V.Exc_Id, V.Exc_Msg);
            when Mmz_Internal_Entity_Array_Access =>
               Dec_Ref (V.As_Internal_Entity_Array_Access);
            when Mmz_Internal_Entity_Base_Formal_Param_Decl_Array_Access =>
               Dec_Ref (V.As_Internal_Entity_Base_Formal_Param_Decl_Array_Access);
            when Mmz_Internal_Entity_Base_Type_Decl_Array_Access =>
               Dec_Ref (V.As_Internal_Entity_Base_Type_Decl_Array_Access);
            when Mmz_Internal_Entity_Basic_Decl_Array_Access =>
               Dec_Ref (V.As_Internal_Entity_Basic_Decl_Array_Access);
            when Mmz_Internal_Entity_Compilation_Unit_Array_Access =>
               Dec_Ref (V.As_Internal_Entity_Compilation_Unit_Array_Access);
            when Mmz_Internal_Entity_Expr_Array_Access =>
               Dec_Ref (V.As_Internal_Entity_Expr_Array_Access);
            when Mmz_Internal_Env_Assoc =>
               Dec_Ref (V.As_Internal_Env_Assoc);
            when Mmz_Internal_Env_Assoc_Array_Access =>
               Dec_Ref (V.As_Internal_Env_Assoc_Array_Access);
            when Mmz_Internal_Inner_Env_Assoc_Array_Access =>
               Dec_Ref (V.As_Internal_Inner_Env_Assoc_Array_Access);
            when Mmz_Internal_Param_Match_Array_Access =>
               Dec_Ref (V.As_Internal_Param_Match_Array_Access);
            when Mmz_Internal_Single_Actual_Array_Access =>
               Dec_Ref (V.As_Internal_Single_Actual_Array_Access);
            when Mmz_Internal_Unit_Array_Access =>
               Dec_Ref (V.As_Internal_Unit_Array_Access);
            when Mmz_Lexical_Env =>
               Dec_Ref (V.As_Lexical_Env);
         when others => null;
      end case;
   end loop;
   Free (Keys);
   Free (Values);
end Destroy;
-------------
-- Destroy --
-------------
procedure Destroy (Key : in out Mmz_Key_Array_Access) is
   procedure Free is new Ada.Unchecked_Deallocation
     (Mmz_Key_Array, Mmz_Key_Array_Access);
begin
      for K of Key.all loop
         case K.Kind is
               when Mmz_Internal_Entity_Base_Type_Decl_Array_Access =>
                  Dec_Ref (K.As_Internal_Entity_Base_Type_Decl_Array_Access);
               when Mmz_Lexical_Env =>
                  Dec_Ref (K.As_Lexical_Env);
            when others => null;
         end case;
      end loop;
   Free (Key);
end Destroy;
-------------------------
-- Find_Memoized_Value --
-------------------------
function Find_Memoized_Value
  (Unit       : Internal_Unit;
   Handle     : out Memoization_Handle;
   Value      : out Mmz_Value;
   Create_Key : access function return Mmz_Key) return Boolean
is
   Inserted : Boolean;
begin
   --  Make sure that we don't lookup stale caches
   Reset_Caches (Unit);
   --  Initialize handle: create the key and create a cursor pointing to an
   --  existing entry.
   Handle.Key := Create_Key.all;
   Handle.Cache_Version := Unit.Cache_Version;
   Value := (Kind => Mmz_Evaluating);
   Unit.Memoization_Map.Insert (Handle.Key, Value, Handle.Cur, Inserted);
   --  No existing entry yet? The above just created one. Otherwise, destroy
   --  our key and reuse the existing entry's.
   if not Inserted then
      Destroy (Handle.Key.Items);
      Handle.Key := Memoization_Maps.Key (Handle.Cur);
      Value := Memoization_Maps.Element (Handle.Cur);
   end if;
   return not Inserted;
end Find_Memoized_Value;
------------------------
-- Add_Memoized_Value --
------------------------
procedure Add_Memoized_Value
  (Unit   : Internal_Unit;
   Handle : in out Memoization_Handle;
   Value  : Mmz_Value;
   Stored : out Boolean) is
begin
   --  If Handle was created using a memoization map that has been since then
   --  reset, do nothing: the result can be partly stale due to the event that
   --  triggered the memoization tables reset.
   Stored := Unit.Cache_Version <= Handle.Cache_Version;
   if Stored then
      Unit.Memoization_Map.Replace_Element (Handle.Cur, Value);
   end if;
end Add_Memoized_Value;
------------------------
-- Add_Memoized_Error --
------------------------
procedure Add_Memoized_Error
  (Unit   : Internal_Unit;
   Handle : in out Memoization_Handle;
   Exc    : Ada.Exceptions.Exception_Occurrence;
   Stored : out Boolean)
is
   Value : Mmz_Value (Kind => Mmz_Error);
begin
   Store_Memoized_Error (Exc, Value.Exc_Id, Value.Exc_Msg);
   Add_Memoized_Value (Unit, Handle, Value, Stored);
   if not Stored then
      Free_Memoized_Error (Value.Exc_Id, Value.Exc_Msg);
   end if;
end Add_Memoized_Error;
----------------------------
-- Reraise_Memoized_Error --
----------------------------
procedure Reraise_Memoized_Error (Value : Mmz_Value) is
begin
   Reraise_Memoized_Error (Value.Exc_Id, Value.Exc_Msg);
end Reraise_Memoized_Error;
--------------------------
-- Store_Memoized_Error --
--------------------------
procedure Store_Memoized_Error
  (Exc     : Ada.Exceptions.Exception_Occurrence;
   Exc_Id  : out Ada.Exceptions.Exception_Id;
   Exc_Msg : out String_Access) is
begin
   Exc_Id := Ada.Exceptions.Exception_Identity (Exc);
   Exc_Msg := new String'(Ada.Exceptions.Exception_Message (Exc));
end Store_Memoized_Error;
-------------------------
-- Free_Memoized_Error --
-------------------------
procedure Free_Memoized_Error
  (Exc_Id  : in out Ada.Exceptions.Exception_Id;
   Exc_Msg : in out String_Access)
is
   pragma Unreferenced (Exc_Id);
begin
   Free (Exc_Msg);
end Free_Memoized_Error;
----------------------------
-- Reraise_Memoized_Error --
----------------------------
procedure Reraise_Memoized_Error
  (Exc_Id  : Ada.Exceptions.Exception_Id;
   Exc_Msg : String_Access) is
begin
   Ada.Exceptions.Raise_Exception (Exc_Id, Exc_Msg.all & " (memoized)");
end Reraise_Memoized_Error;
   package Solver_Diagnostic_Vectors is new Langkit_Support.Vectors
     (Internal_Solver_Diagnostic);
   ----------------------------
   -- Allocate_Logic_Context --
   ----------------------------
   function Allocate_Logic_Context
     (Ctx : Internal_Logic_Context) return Internal_Logic_Context_Access
   is (new Internal_Logic_Context'(Ctx));
   -------------------------
   -- Trace_Logic_Context --
   -------------------------
   function Trace_Logic_Context
     (Ctx : Internal_Logic_Context_Access) return String
   is (Trace_Image (Ctx.all));
   -----------------
   -- Deep_Equals --
   -----------------
   function Deep_Equals
     (X, Y : Internal_Logic_Context_Access) return Boolean
   is (X.all = Y.all);
   ------------------------
   -- Free_Logic_Context --
   ------------------------
   procedure Free_Logic_Context
     (Ctx : in out Internal_Logic_Context_Access)
   is
      procedure Free is new Ada.Unchecked_Deallocation
        (Internal_Logic_Context, Internal_Logic_Context_Access);
   begin
      Free (Ctx);
   end Free_Logic_Context;
   -------------------
   -- Solve_Wrapper --
   -------------------
   function Solve_Wrapper
     (R            : Solver.Relation;
      Context_Node : Bare_Ada_Node) return Boolean is
   begin
      if Context_Node /= null and then Langkit_Support.Adalog.Debug.Debug then
         Assign_Names_To_Logic_Vars (Context_Node);
      end if;
      begin
         return Solver.Solve_First
           (R, Timeout => Context_Node.Unit.Context.Logic_Resolution_Timeout);
      exception
         when Langkit_Support.Adalog.Early_Binding_Error =>
            Raise_Property_Exception
              (Context_Node,
               Property_Error'Identity,
               "invalid equation for logic resolution");
         when Langkit_Support.Adalog.Timeout_Error =>
            Raise_Property_Exception
              (Context_Node,
               Property_Error'Identity,
               "logic resolution timed out");
      end;
   end Solve_Wrapper;
   ----------------------------
   -- Solve_With_Diagnostics --
   ----------------------------
   function Solve_With_Diagnostics
     (R            : Solver.Relation;
      Context_Node : Bare_Ada_Node) return Internal_Solver_Result
   is
      Ret : Internal_Solver_Result :=
        (True, No_Internal_Solver_Diagnostic_Array_Type);
      Acc : Solver_Diagnostic_Vectors.Vector;
      --  Vectors that will accumulate diagnostic emitted during resolution
      procedure Emit_Diagnostic (Diag : Internal_Solver_Diagnostic) is
      begin
         Acc.Append (Diag);
      end Emit_Diagnostic;
   begin
      Ret.Success := Solve_Wrapper (R, Context_Node);
      if not Ret.Success then
         Ret.Success := Solver.Solve_First
           (R,
            Solve_Options => (Report_Errors => True),
            Diag_Emitter => Emit_Diagnostic'Unrestricted_Access,
            Timeout => Context_Node.Unit.Context.Logic_Resolution_Timeout);
         Ret.Diagnostics := Create_Internal_Solver_Diagnostic_Array
           (Acc.Length);
         for I in 1 .. Acc.Length loop
            Ret.Diagnostics.Items (I) := Acc.Get (I);
         end loop;
         Acc.Destroy;
      end if;
      return Ret;
   end Solve_With_Diagnostics;
   -------------
   -- Destroy --
   -------------
   procedure Destroy (Env : in out Lexical_Env_Access) is
      Mutable_Env : Lexical_Env :=
        (Wrap (Env), 0, Env.Kind, No_Generic_Unit, 0);
   begin
      Destroy (Mutable_Env);
      Env := null;
   end Destroy;
   ----------------
   -- Initialize --
   ----------------
   procedure Initialize
     (Self              : Bare_Ada_Node;
      Kind              : Ada_Node_Kind_Type;
      Unit              : Internal_Unit;
      Token_Start_Index : Token_Index;
      Token_End_Index   : Token_Index;
      Parent            : Bare_Ada_Node := null;
      Self_Env          : Lexical_Env := AST_Envs.Empty_Env) is
   begin
      pragma Unreferenced (Kind);
      Self.Parent := Parent;
      Self.Unit := Unit;
      Self.Token_Start_Index := Token_Start_Index;
      Self.Token_End_Index := Token_End_Index;
      Self.Self_Env := Self_Env;
      Self.Last_Attempted_Child := -1;
   end Initialize;
   --------------------
   -- Use_Direct_Env --
   --------------------
   procedure Use_Direct_Env (State : in out PLE_Node_State; Env : Lexical_Env)
   is
   begin
      State.Current_Env := Env;
      State.Current_NED := null;
   end Use_Direct_Env;
   -------------------
   -- Use_Named_Env --
   -------------------
   procedure Use_Named_Env
     (State   : in out PLE_Node_State;
      Context : Internal_Context;
      Name    : Symbol_Type) is
   begin
      State.Current_NED := Get_Named_Env_Descriptor (Context, Name);
      State.Current_Env := State.Current_NED.Env_With_Precedence;
   end Use_Named_Env;
   ---------------------
   -- Set_Initial_Env --
   ---------------------
   procedure Set_Initial_Env
     (Self         : Bare_Ada_Node;
      State        : in out PLE_Node_State;
      Env          : Internal_Designated_Env;
      DSL_Location : String) is
   begin
      case Env.Kind is
         when None =>
            Use_Direct_Env (State, Empty_Env);
         when Current_Env =>
            null;
         when Named_Env =>
            Use_Named_Env (State, Self.Unit.Context, Env.Env_Name);
         when Direct_Env =>
            --  Sanitize this environment value: make sure it's a non-foreign
            --  and primary environment.
            if Env.Direct_Env.Kind /= Static_Primary then
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Cannot set an env that is not static-primary as the"
                  & " initial env");
            elsif Is_Foreign_Strict (Env.Direct_Env, Self) then
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "unsound foreign environment in SetInitialEnv ("
                  & DSL_Location & ")");
            end if;
            Use_Direct_Env (State, Env.Direct_Env);
      end case;
   end Set_Initial_Env;
   ----------------
   -- Add_To_Env --
   ----------------
   procedure Add_To_Env
     (Self         : Bare_Ada_Node;
      State        : PLE_Node_State;
      Key          : Symbol_Type;
      Value        : Bare_Ada_Node;
      Md           : Internal_Metadata;
      Resolver     : Entity_Resolver;
      Dest_Env     : Internal_Designated_Env;
      DSL_Location : String)
   is
      Context    : constant Internal_Context := Self.Unit.Context;
      Root_Scope : Lexical_Env renames Context.Root_Scope;
      --  Shortcuts
      Actual_Dest_Env : Lexical_Env;
      Dest_NED        : Named_Env_Descriptor_Access;
      --  Description for the destination environment
   begin
      --  Skip the env addition if explicitly requested
      if Key = No_Symbol
         or else Value = null
         or else (case Dest_Env.Kind is
                  when None        => True,
                  when Current_Env => False,
                  when Named_Env   => Dest_Env.Env_Name = No_Symbol,
                  when Direct_Env  => Dest_Env.Direct_Env = Empty_Env)
      then
         return;
      end if;
      if Value.Unit /= Self.Unit then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "Cannot add_to_env an AST node that comes from another analysis"
            & " unit");
      end if;
         --  Make sure metadata does not contain any foreign node
         if (Md.Primitive /= null and then Md.Primitive.Unit /= Self.Unit) or else (Md.Primitive_Real_Type /= null and then Md.Primitive_Real_Type.Unit /= Self.Unit)
         then
            Raise_Property_Exception
              (Self,
               Property_Error'Identity,
               "Cannot add metadata that contains foreign nodes");
         end if;
      --  Then determine the destination environment
      case Dest_Env.Kind is
         when None =>
            raise Program_Error with "unreachable code";
         when Current_Env =>
            --  Just use the current environment
            Dest_NED := State.Current_NED;
            Actual_Dest_Env := State.Current_Env;
         when Named_Env =>
            --  There is an environment name: just lookup the corresponding
            --  NED/env.
            Dest_NED := Get_Named_Env_Descriptor (Context, Dest_Env.Env_Name);
            Actual_Dest_Env := Dest_NED.Env_With_Precedence;
         when Direct_Env =>
            --  There is an explicit destination environment
            Dest_NED := null;
            Actual_Dest_Env := Dest_Env.Direct_Env;
      end case;
      --  Sanitize it
      if Actual_Dest_Env.Kind /= Static_Primary then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "Cannot add elements to a lexical env that is not static-primary");
      elsif
         --  Since lexical envs need to sort the foreign nodes they contain,
         --  and that the total order on nodes is not defined for synthetic
         --  nodes, it is not possible to add a synthetic node to a foreign
         --  lexical environment.
         --
         --  This reasoning applies to environments that belong to foreign
         --  units, but also to the root environment.
         Is_Foreign (Actual_Dest_Env, Self) and then Is_Synthetic (Value)
      then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "Cannot add a synthetic node to a lexical env from another"
            & " analysis unit");
      elsif
         --  Reject direct references to foreign destination environments.
         --
         --  This is an attempt at identifying uses of the unsound relocation
         --  mechanism (as opposed to named environments), so this applies to
         --  all foreign environments (root scope included).
         DSL_Location'Length > 0
         and then Dest_Env.Kind = Direct_Env
         and then Is_Foreign_Strict (Actual_Dest_Env, Self)
      then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "unsound foreign environment in AddToEnv (" & DSL_Location & ")");
      end if;
      --  Now that everything is sanitized, we can proceed with the actual
      --  key/value pair addition. Note that this does nothing if
      --  Actual_Dest_Env ended up empty.
      Add (Actual_Dest_Env, Thin (Key), Value, Md, Resolver);
      --  If we're adding the element to an environment by env name, we must
      --  register this association in two places: in the target named env
      --  entry, and in Value's unit.
      if Dest_NED /= null then
         declare
            use NED_Assoc_Maps;
            FN    : Map renames Dest_NED.Foreign_Nodes;
            Dummy : Boolean;
            Cur   : Cursor;
         begin
            FN.Insert (Key      => Key,
                       New_Item => Internal_Map_Node_Vectors.Empty_Vector,
                       Position => Cur,
                       Inserted => Dummy);
            declare
               V : Internal_Map_Node_Vectors.Vector renames
                  FN.Reference (Cur);
            begin
               V.Append ((Value, null, Md, Resolver));
               --  Note that the rebindings field is unused by the relocation
               --  mechanism (since we don't even allow adding env entries
               --  with custom rebindings), hence we simply leave it to null.
            end;
         end;
         Value.Unit.Exiled_Entries_In_NED.Append ((Dest_NED, Key, Value));
      --  Otherwise, if we're adding the element to an environment that belongs
      --  to a different unit, or to the root scope, then...
      elsif Is_Foreign_Not_Empty (Actual_Dest_Env, Self) then
         --  Add the Key/Value association to the list of entries contained in
         --  other units, so we can remove them when reparsing Value's unit.
         Value.Unit.Exiled_Entries.Append ((Actual_Dest_Env, Key, Value));
         if Actual_Dest_Env /= Root_Scope then
            --  Add Val to the list of foreign nodes that Actual_Dest_Env's
            --  unit contains, so that when that unit is reparsed, we can call
            --  Add_To_Env again on those nodes.
            Convert_Unit (Actual_Dest_Env.Owner).Foreign_Nodes.Append
              ((Value, Self.Unit));
         end if;
      end if;
   end Add_To_Env;
   -------------
   -- Ref_Env --
   -------------
   procedure Ref_Env
     (Self                : Bare_Ada_Node;
      Dest_Env            : Lexical_Env;
      Ref_Env_Nodes       : in out Bare_Ada_Node_Array_Access;
      Resolver            : Lexical_Env_Resolver;
      Kind                : Ref_Kind;
      Cats                : Ref_Categories;
      Shed_Rebindings     : Boolean) is
   begin
      for N of Ref_Env_Nodes.Items loop
         if N /= null then
            if N.Unit /= Self.Unit then
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "attempt to add a referenced environment to a foreign unit");
            end if;
            Reference (Dest_Env, N, Resolver, Kind, Cats, Shed_Rebindings);
         end if;
      end loop;
      Dec_Ref (Ref_Env_Nodes);
   end Ref_Env;
   -------------
   -- Add_Env --
   -------------
   procedure Add_Env
     (Self              : Bare_Ada_Node;
      State             : in out PLE_Node_State;
      No_Parent         : Boolean;
      Transitive_Parent : Boolean;
      Names             : in out Symbol_Type_Array_Access)
   is
      Parent_From_Name : constant Boolean := State.Current_NED /= null;
      --  Does the parent environment comes from a named environment lookup?
      --  Determine the parent of this new environment:
      --
      --  (1) no parent if requested;
      --  (2) the current environment as the static parent if it comes from a
      --      named env lookup or if it is not foreign (or is the empty/root
      --      environment).
      Parent : constant Lexical_Env :=
        (if No_Parent
         then Null_Lexical_Env
         else State.Current_Env);
   begin
      --  Create the environment itself
      Self.Self_Env := Create_Static_Lexical_Env
        (Parent            => Parent,
         Node              => Self,
         Transitive_Parent => Transitive_Parent,
         Sym_Table         => Self.Unit.Context.Symbols);
      --  If the parent of this new environment comes from a named environment
      --  lookup, register this new environment so that its parent is updated
      --  when the precence for this named environment changes.
      if Parent_From_Name then
         declare
            NED : constant Named_Env_Descriptor_Access := State.Current_NED;
         begin
            Self.Unit.Exiled_Envs.Append ((NED, Self.Self_Env));
            NED.Foreign_Envs.Insert (Self, Self.Self_Env);
         end;
      end if;
      --  From now on, the current environment is Self.Self_Env, with a direct
      --  access to it. It does not go through the env naming scheme, since
      --  only this node and its children (i.e. non-foreign nodes) will access
      --  it as a "current" environment during PLE.
      Use_Direct_Env (State, Self.Self_Env);
      --  Register the environment we just created on all the requested names
      if Names /= null then
         declare
            Context   : constant Internal_Context := Self.Unit.Context;
            Env       : constant Lexical_Env := Self.Self_Env;
            NENU      : NED_Maps.Map renames
               State.Unit_State.Named_Envs_Needing_Update;
         begin
            for N of Names.Items loop
               Register_Named_Env (Context, N, Env, NENU);
            end loop;
            Dec_Ref (Names);
         end;
      end if;
   end Add_Env;
   ---------------------
   -- Pre_Env_Actions --
   ---------------------
   procedure Pre_Env_Actions
     (Self            : Bare_Ada_Node;
      State           : in out PLE_Node_State;
      Add_To_Env_Only : Boolean := False) is
   begin
   case Self.Kind is
            when Ada_With_Clause =>
            With_Clause_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Use_Type_Clause =>
            Use_Type_Clause_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Use_Package_Clause =>
            Use_Package_Clause_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Access_To_Subp_Def =>
            Access_To_Subp_Def_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Named_Stmt =>
            Named_Stmt_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Extended_Return_Stmt =>
            Extended_Return_Stmt_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Begin_Block .. Ada_Decl_Block =>
            Block_Stmt_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_For_Loop_Stmt =>
            For_Loop_Stmt_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Accept_Stmt .. Ada_Accept_Stmt_With_Stmts =>
            Accept_Stmt_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Pragma_Node =>
            Pragma_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Quantified_Expr =>
            Quantified_Expr_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Decl_Expr =>
            Decl_Expr_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Private_Part =>
            Private_Part_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Single_Task_Decl =>
            Single_Task_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Single_Protected_Decl =>
            Single_Protected_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Package_Renaming_Decl =>
            Package_Renaming_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Object_Decl .. Ada_No_Type_Object_Renaming_Decl =>
            Object_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Number_Decl =>
            Number_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Label_Decl =>
            Label_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Subp_Renaming_Decl =>
            Generic_Subp_Renaming_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Package_Renaming_Decl =>
            Generic_Package_Renaming_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Package_Instantiation .. Ada_Generic_Subp_Instantiation =>
            Generic_Instantiation_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Subp_Decl =>
            Generic_Subp_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Package_Decl =>
            Generic_Package_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_For_Loop_Var_Decl =>
            For_Loop_Var_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Exception_Handler =>
            Exception_Handler_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Exception_Decl =>
            Exception_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Entry_Index_Spec =>
            Entry_Index_Spec_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Task_Body =>
            Task_Body_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Protected_Body =>
            Protected_Body_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Package_Body =>
            Package_Body_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Entry_Body =>
            Entry_Body_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Task_Body_Stub =>
            Task_Body_Stub_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Subp_Body_Stub =>
            Subp_Body_Stub_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Protected_Body_Stub =>
            Protected_Body_Stub_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Package_Body_Stub =>
            Package_Body_Stub_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Expr_Function .. Ada_Subp_Renaming_Decl =>
            Base_Subp_Body_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Subp_Internal =>
            Generic_Subp_Internal_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Enum_Literal_Decl .. Ada_Synthetic_Char_Enum_Lit =>
            Enum_Literal_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Entry_Decl =>
            Entry_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Abstract_Subp_Decl .. Ada_Subp_Decl | Ada_Synthetic_Subp_Decl =>
            Basic_Subp_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Anonymous_Type_Decl .. Ada_Synth_Anonymous_Type_Decl =>
            null;
            when Ada_Concrete_Type_Decl .. Ada_Formal_Type_Decl =>
            Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Single_Task_Type_Decl =>
            Single_Task_Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Task_Type_Decl =>
            Task_Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Protected_Type_Decl =>
            Protected_Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Incomplete_Type_Decl .. Ada_Incomplete_Tagged_Type_Decl =>
            Incomplete_Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Classwide_Type_Decl =>
            null;
            when Ada_Subtype_Decl =>
            Subtype_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Discrete_Base_Subtype_Decl =>
            Base_Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Package_Decl =>
            Package_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Generic_Package_Internal =>
            Generic_Package_Internal_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Param_Spec =>
            Param_Spec_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Discriminant_Spec =>
            Discriminant_Spec_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Component_Decl =>
            Component_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
            when Ada_Abstract_State_Decl =>
            Abstract_State_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      when others =>  null; 
   end case;
   end Pre_Env_Actions;
   ----------------------
   -- Post_Env_Actions --
   ----------------------
   procedure Post_Env_Actions
     (Self : Bare_Ada_Node; State : in out PLE_Node_State) is
   begin
   case Self.Kind is
            when Ada_With_Clause =>
            null;
            when Ada_Use_Type_Clause =>
            Use_Type_Clause_Post_Env_Actions (Self, State);
            when Ada_Use_Package_Clause =>
            Use_Package_Clause_Post_Env_Actions (Self, State);
            when Ada_Access_To_Subp_Def =>
            null;
            when Ada_Named_Stmt =>
            null;
            when Ada_Extended_Return_Stmt =>
            null;
            when Ada_Begin_Block .. Ada_Decl_Block =>
            null;
            when Ada_For_Loop_Stmt =>
            null;
            when Ada_Accept_Stmt .. Ada_Accept_Stmt_With_Stmts =>
            null;
            when Ada_Pragma_Node =>
            null;
            when Ada_Quantified_Expr =>
            null;
            when Ada_Decl_Expr =>
            null;
            when Ada_Private_Part =>
            null;
            when Ada_Single_Task_Decl =>
            null;
            when Ada_Single_Protected_Decl =>
            null;
            when Ada_Package_Renaming_Decl =>
            null;
            when Ada_Object_Decl .. Ada_No_Type_Object_Renaming_Decl =>
            null;
            when Ada_Number_Decl =>
            null;
            when Ada_Label_Decl =>
            null;
            when Ada_Generic_Subp_Renaming_Decl =>
            null;
            when Ada_Generic_Package_Renaming_Decl =>
            null;
            when Ada_Generic_Package_Instantiation .. Ada_Generic_Subp_Instantiation =>
            null;
            when Ada_Generic_Subp_Decl =>
            Generic_Subp_Decl_Post_Env_Actions (Self, State);
            when Ada_Generic_Package_Decl =>
            null;
            when Ada_For_Loop_Var_Decl =>
            null;
            when Ada_Exception_Handler =>
            null;
            when Ada_Exception_Decl =>
            null;
            when Ada_Entry_Index_Spec =>
            null;
            when Ada_Task_Body =>
            null;
            when Ada_Protected_Body =>
            null;
            when Ada_Package_Body =>
            null;
            when Ada_Entry_Body =>
            null;
            when Ada_Task_Body_Stub =>
            null;
            when Ada_Subp_Body_Stub =>
            null;
            when Ada_Protected_Body_Stub =>
            null;
            when Ada_Package_Body_Stub =>
            null;
            when Ada_Expr_Function .. Ada_Subp_Renaming_Decl =>
            null;
            when Ada_Generic_Subp_Internal =>
            null;
            when Ada_Enum_Literal_Decl .. Ada_Synthetic_Char_Enum_Lit =>
            null;
            when Ada_Entry_Decl =>
            null;
            when Ada_Abstract_Subp_Decl .. Ada_Subp_Decl | Ada_Synthetic_Subp_Decl =>
            Basic_Subp_Decl_Post_Env_Actions (Self, State);
            when Ada_Anonymous_Type_Decl .. Ada_Synth_Anonymous_Type_Decl =>
            null;
            when Ada_Concrete_Type_Decl .. Ada_Formal_Type_Decl =>
            Type_Decl_Post_Env_Actions (Self, State);
            when Ada_Single_Task_Type_Decl =>
            null;
            when Ada_Task_Type_Decl =>
            null;
            when Ada_Protected_Type_Decl =>
            null;
            when Ada_Incomplete_Type_Decl .. Ada_Incomplete_Tagged_Type_Decl =>
            null;
            when Ada_Classwide_Type_Decl =>
            null;
            when Ada_Subtype_Decl =>
            null;
            when Ada_Discrete_Base_Subtype_Decl =>
            null;
            when Ada_Package_Decl =>
            null;
            when Ada_Generic_Package_Internal =>
            null;
            when Ada_Param_Spec =>
            null;
            when Ada_Discriminant_Spec =>
            null;
            when Ada_Component_Decl =>
            null;
            when Ada_Abstract_State_Decl =>
            null;
      when others =>  null; 
   end case;
   end Post_Env_Actions;
   ----------------
   -- Get_Symbol --
   ----------------
   function Get_Symbol
     (Node : Bare_Ada_Node) return Symbol_Type is
   begin
      if Node = null then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "cannot get the symbol of a null node");
      end if;
      return Get_Symbol (Token (Node, Node.Token_Start_Index));
   end Get_Symbol;
   ----------
   -- Text --
   ----------
   function Text
     (Node : Bare_Ada_Node) return Text_Type
   is
   begin
      if Node = null then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "cannot get the text of a null node");
      end if;
      declare
         Start_T : constant Token_Reference :=
            Token (Node, Node.Token_Start_Index);
         End_T   : constant Token_Reference :=
            Token (Node, Node.Token_End_Index);
      begin
         --  No text is associated to synthetic and ghost nodes
         if Is_Synthetic (Node) then
            return "";
         end if;
         if Is_Ghost (Node) then
            return "";
         end if;
         return Text (Start_T, End_T);
      end;
   end Text;
   ---------------------
   -- Is_Visible_From --
   ---------------------
   function Is_Visible_From
     (Self                     : Bare_Ada_Node;
      Referenced_Env, Base_Env : Lexical_Env) return Boolean
   is
      Referenced_Unit : constant Internal_Unit :=
         Convert_Unit (Referenced_Env.Owner);
      Base_Unit       : constant Internal_Unit :=
         Convert_Unit (Base_Env.Owner);
   begin
      if Referenced_Unit = null then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "referenced environment does not belong to any analysis unit");
      elsif Base_Unit = null then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "base environment does not belong to any analysis unit");
      end if;
      return Is_Referenced_From (Referenced_Unit, Base_Unit);
   end Is_Visible_From;
   ----------
   -- Unit --
   ----------
   function Unit (Node : Bare_Ada_Node) return Internal_Unit is
   begin
      return Node.Unit;
   end Unit;
   function Lookup_Internal
     (Node : Bare_Ada_Node;
      Sloc : Source_Location) return Bare_Ada_Node;
   procedure Lookup_Relative
     (Node       : Bare_Ada_Node;
      Sloc       : Source_Location;
      Position   : out Relative_Position;
      Node_Found : out Bare_Ada_Node);
   --  Implementation helpers for the looking up process
   -----------------
   -- Set_Parents --
   -----------------
   procedure Set_Parents
     (Node, Parent : Bare_Ada_Node)
   is
   begin
      if Node = null then
         return;
      end if;
      Node.Parent := Bare_Ada_Node (Parent);
      for I in 1 .. Children_Count (Node) loop
         Set_Parents (Child (Node, I), Node);
      end loop;
   end Set_Parents;
   -------------
   -- Destroy --
   -------------
   procedure Destroy (Node : Bare_Ada_Node) is
   begin
      if Node = null then
         return;
      end if;
      Free_User_Fields (Node);
      for I in 1 .. Children_Count (Node) loop
         Destroy (Child (Node, I));
      end loop;
   end Destroy;
   -----------
   -- Child --
   -----------
   function Child (Node  : Bare_Ada_Node;
                   Index : Positive) return Bare_Ada_Node
   is
      Result          : Bare_Ada_Node;
      Index_In_Bounds : Boolean;
   begin
      Get_Child (Node, Index, Index_In_Bounds, Result);
      return Result;
   end Child;
   --------------
   -- Traverse --
   --------------
   function Traverse
     (Node  : Bare_Ada_Node;
      Visit : access function (Node : Bare_Ada_Node)
              return Visit_Status)
     return Visit_Status
   is
      Status : Visit_Status := Into;
   begin
      if Node /= null then
         Status := Visit (Node);
         --  Skip processing the child nodes if the returned status is Over
         --  or Stop. In the former case the previous call to Visit has taken
         --  care of processing the needed childs, and in the latter case we
         --  must immediately stop processing the tree.
         if Status = Into then
            for I in 1 .. Children_Count (Node) loop
               declare
                  Cur_Child : constant Bare_Ada_Node :=
                     Child (Node, I);
               begin
                  if Cur_Child /= null then
                     Status := Traverse (Cur_Child, Visit);
                     exit when Status /= Into;
                  end if;
               end;
            end loop;
         end if;
      end if;
      if Status = Stop then
         return Stop;
      --  At this stage the Over status has no sense and we just continue
      --  processing the tree.
      else
         return Into;
      end if;
   end Traverse;
   --------------
   -- Traverse --
   --------------
   procedure Traverse
     (Node  : Bare_Ada_Node;
      Visit : access function (Node : Bare_Ada_Node)
                               return Visit_Status)
   is
      Result_Status : Visit_Status;
      pragma Unreferenced (Result_Status);
   begin
      Result_Status := Traverse (Node, Visit);
   end Traverse;
   ------------------------
   -- Traverse_With_Data --
   ------------------------
   function Traverse_With_Data
     (Node  : Bare_Ada_Node;
      Visit : access function (Node : Bare_Ada_Node;
                               Data : in out Data_Type)
                               return Visit_Status;
      Data  : in out Data_Type)
      return Visit_Status
   is
      function Helper (Node : Bare_Ada_Node) return Visit_Status;
      ------------
      -- Helper --
      ------------
      function Helper (Node : Bare_Ada_Node) return Visit_Status is
      begin
         return Visit (Node, Data);
      end Helper;
      Saved_Data : Data_Type;
      Result     : Visit_Status;
   begin
      if Reset_After_Traversal then
         Saved_Data := Data;
      end if;
      Result := Traverse (Node, Helper'Access);
      if Reset_After_Traversal then
         Data := Saved_Data;
      end if;
      return Result;
   end Traverse_With_Data;
   ----------------
   -- Sloc_Range --
   ----------------
   function Sloc_Range
     (Node : Bare_Ada_Node) return Source_Location_Range
   is
      type Token_Anchor is (T_Start, T_End);
      type Token_Pos is record
         Pos    : Token_Index;
         Anchor : Token_Anchor;
      end record;
      TDH                    : Token_Data_Handler renames Node.Unit.TDH;
      Token_Start, Token_End : Token_Pos;
      function Get (Index : Token_Index) return Stored_Token_Data is
        (Get_Token (TDH, Index));
      function Sloc (T : Token_Pos) return Source_Location is
        (if T.Anchor = T_Start
         then Sloc_Start (TDH, Get (T.Pos))
         else Sloc_End (TDH, Get (T.Pos)));
   begin
      if Is_Synthetic (Node) then
         return (if Node.Parent = null
                 then No_Source_Location_Range
                 else Sloc_Range (Node.Parent));
      end if;
      if Is_Ghost (Node) then
         Token_Start := (if Node.Token_Start_Index = 1
                         then (1, T_Start)
                         else (Node.Token_Start_Index - 1, T_End));
         Token_End := Token_Start;
      else
         Token_Start := (Node.Token_Start_Index, T_Start);
         Token_End := (Node.Token_End_Index, T_End);
      end if;
      if Snaps_At_Start (Node)
         and then not Is_Ghost (Node)
         and then Token_Start.Pos /= 1
      then
         Token_Start := (Token_Start.Pos - 1, T_End);
      end if;
      if Snaps_At_End (Node) and then Token_End.Pos /= Last_Token (TDH) then
         Token_End := (Token_End.Pos + 1, T_Start);
      end if;
      return Make_Range (Sloc (Token_Start), Sloc (Token_End));
   end Sloc_Range;
   ------------
   -- Lookup --
   ------------
   function Lookup
     (Node : Bare_Ada_Node;
      Sloc : Source_Location) return Bare_Ada_Node
   is
      Position : Relative_Position;
      Result   : Bare_Ada_Node;
   begin
      if Sloc = No_Source_Location then
         return null;
      end if;
      Lookup_Relative
        (Bare_Ada_Node (Node), Sloc, Position, Result);
      return Result;
   end Lookup;
   ---------------------
   -- Lookup_Internal --
   ---------------------
   function Lookup_Internal
     (Node : Bare_Ada_Node;
      Sloc : Source_Location) return Bare_Ada_Node
   is
      --  For this implementation helper (i.e. internal primitive), we can
      --  assume that all lookups fall into this node's sloc range.
      pragma Assert (Compare (Sloc_Range (Node), Sloc) = Inside);
      Children : constant Internal_Bare_Ada_Node_Array :=
         Implementation.Children (Node);
      Pos      : Relative_Position;
      Result   : Bare_Ada_Node;
   begin
      --  Look for a child node that contains Sloc (i.e. return the most
      --  precise result).
      for Child of Children loop
         --  Note that we assume here that child nodes are ordered so that the
         --  first one has a sloc range that is before the sloc range of the
         --  second child node, etc.
         if Child /= null then
            Lookup_Relative (Child, Sloc, Pos, Result);
            case Pos is
               when Before =>
                   --  If this is the first node, Sloc is before it, so we can
                   --  stop here.  Otherwise, Sloc is between the previous
                   --  child node and the next one...  so we can stop here,
                   --  too.
                   return Node;
               when Inside =>
                   return Result;
               when After =>
                   --  Sloc is after the current child node, so see with the
                   --  next one.
                   null;
            end case;
         end if;
      end loop;
      --  If we reach this point, we found no children that covers Sloc, but
      --  Node still covers it (see the assertion).
      return Node;
   end Lookup_Internal;
   -------------
   -- Compare --
   -------------
   function Compare
     (Node : Bare_Ada_Node;
      Sloc : Source_Location) return Relative_Position is
   begin
      return Compare (Sloc_Range (Node), Sloc);
   end Compare;
   ---------------------
   -- Lookup_Relative --
   ---------------------
   procedure Lookup_Relative
     (Node       : Bare_Ada_Node;
      Sloc       : Source_Location;
      Position   : out Relative_Position;
      Node_Found : out Bare_Ada_Node)
   is
      Result : constant Relative_Position :=
        Compare (Node, Sloc);
   begin
      Position := Result;
      Node_Found := (if Result = Inside
                     then Lookup_Internal (Node, Sloc)
                     else null);
   end Lookup_Relative;
   -------------
   -- Compare --
   -------------
   function Compare
     (Self, Left, Right : Bare_Ada_Node;
      Relation          : Comparison_Relation) return Boolean
   is
      LS, RS : Source_Location;
   begin
      if Left = null or else Right = null or else Left.Unit /= Right.Unit then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "invalid node comparison");
      end if;
      LS := Start_Sloc (Sloc_Range (Left));
      RS := Start_Sloc (Sloc_Range (Right));
      return (case Relation is
              when Langkit_Support.Types.Less_Than        => LS < RS,
              when Langkit_Support.Types.Less_Or_Equal    => LS <= RS,
              when Langkit_Support.Types.Greater_Than     => LS > RS,
              when Langkit_Support.Types.Greater_Or_Equal => LS >= RS);
   end Compare;
   --------------
   -- Children --
   --------------
   function Children
     (Node : Bare_Ada_Node) return Internal_Bare_Ada_Node_Array
   is
      First : constant Integer := Bare_Ada_Node_Vectors.Index_Type'First;
      Last  : constant Integer := First + Children_Count (Node) - 1;
   begin
      return A : Internal_Bare_Ada_Node_Array (First .. Last)
      do
         for I in First .. Last loop
            A (I) := Child (Node, I);
         end loop;
      end return;
   end Children;
   function Children
     (Node : Bare_Ada_Node) return Bare_Ada_Node_Array_Access
   is
      C : Internal_Bare_Ada_Node_Array := Children (Node);
   begin
      return Ret : Bare_Ada_Node_Array_Access :=
         Create_Bare_Ada_Node_Array (C'Length)
      do
         Ret.Items := C;
      end return;
   end Children;
   ---------
   -- Get --
   ---------
   function Get
     (Self    : Bare_Ada_Node;
      Node    : Bare_Ada_List;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Ada_Node
   is
      function Length (Node : Bare_Ada_List) return Natural
      is (Node.Count);
      --  Wrapper around the Length primitive to get the compiler happy for the
      --  the package instantiation below.
      function Absolute_Get
        (L     : Bare_Ada_List;
         Index : Integer) return Bare_Ada_Node
      is (L.Nodes.all (Index + 1));
      --  L.Nodes is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Bare_Ada_Node,
         Sequence_Type => Bare_Ada_List,
         Length        => Length,
         Get           => Absolute_Get);
      Result : Bare_Ada_Node;
   begin
      if Node = null and then Or_Null then
         return null;
      elsif Relative_Get (Node, Index, Result) then
         return Result;
      elsif Or_Null then
         return null;
      else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "out-of-bounds AST list access");
      end if;
   end Get;
   ---------------
   -- PP_Trivia --
   ---------------
   procedure PP_Trivia
     (Node        : Bare_Ada_Node;
      Line_Prefix : String := "")
   is
      Children_Prefix : constant String := Line_Prefix & "|  ";
   begin
      if Node = null then
         Put_Line (Line_Prefix & "None");
         return;
      end if;
      Put_Line (Line_Prefix & Kind_Name (Node));
      for C of Children_And_Trivia (Node) loop
         case C.Kind is
            when Trivia =>
               Put_Line (Children_Prefix & Image (Text (C.Trivia)));
            when Child =>
               PP_Trivia (C.Node, Children_Prefix);
         end case;
      end loop;
   end PP_Trivia;
   --------------------------
   -- Populate_Lexical_Env --
   --------------------------
   function Populate_Lexical_Env (Node : Bare_Ada_Node) return Boolean is
      Context    : constant Internal_Context := Node.Unit.Context;
      Unit_State : aliased PLE_Unit_State := (Named_Envs_Needing_Update => <>);
      Root_State : constant PLE_Node_State :=
        (Unit_State  => Unit_State'Unchecked_Access,
         Current_Env => Context.Root_Scope,
         Current_NED => null);
      function Populate_Internal
        (Node         : Bare_Ada_Node;
         Parent_State : PLE_Node_State) return Boolean;
      --  Do the lexical env population on Node and recurse on its children
      procedure Register_Foreign_Env
        (Node : Bare_Ada_Node; State : PLE_Node_State);
      --  Given a node and its PLE state, register Node.Self_Env as being
      --  initialized through the named environment mechanism, if that's indeed
      --  the case. Do nothing otherwise.
      -----------------------
      -- Populate_Internal --
      -----------------------
      function Populate_Internal
        (Node         : Bare_Ada_Node;
         Parent_State : PLE_Node_State) return Boolean
      is
         Result : Boolean := False;
         State  : PLE_Node_State := Parent_State;
      begin
         if Node = null then
            return Result;
         end if;
         --  By default (i.e. unless env actions add a new env), the
         --  environment we store in Node is the current one.
         Node.Self_Env := State.Current_Env;
         --  Run pre/post actions, and run PLE on children in between. Make
         --  sure we register the potential foreign Node.Self_Env environment
         --  at the end, even when an exception interrupts PLE to keep the
         --  state consistent.
         begin
            Pre_Env_Actions (Node, State);
            if State.Current_Env /= Null_Lexical_Env then
               Node.Self_Env := State.Current_Env;
               Register_Foreign_Env (Node, State);
            end if;
            --  Call recursively on children
            for I in First_Child_Index (Node) .. Last_Child_Index (Node) loop
               Result := Populate_Internal
                 (Child (Node, I), State) or else Result;
            end loop;
            Post_Env_Actions (Node, State);
         exception
            when Exc : Precondition_Failure | Property_Error =>
               if PLE_Errors_Trace.Is_Active then
                   GNATCOLL.Traces.Trace
                     (PLE_Errors_Trace,
                      "Exception raised during PLE "
                      & Ada.Exceptions.Exception_Name (Exc) & " : "
                      & Ada.Exceptions.Exception_Message (Exc));
                   GNATCOLL.Traces.Trace
                     (PLE_Errors_Trace,
                      GNAT.Traceback.Symbolic.Symbolic_Traceback (Exc));
               end if;
               Register_Foreign_Env (Node, State);
               return True;
         end;
         return Result;
      end Populate_Internal;
      --------------------------
      -- Register_Foreign_Env --
      --------------------------
      procedure Register_Foreign_Env
        (Node : Bare_Ada_Node; State : PLE_Node_State) is
      begin
         if State.Current_NED /= null then
            State.Current_NED.Nodes_With_Foreign_Env.Insert (Node);
            Node.Unit.Nodes_With_Foreign_Env.Insert (Node, State.Current_NED);
         end if;
      end Register_Foreign_Env;
   begin
      --  This is intended to be called on the root node only (when there is no
      --  PLE root) or on a PLE root (child of the root node with a specific
      --  kind).
      if
         Node.Parent /= null
            and then Node.Kind /= Ada_Compilation_Unit
      then
         raise Program_Error;
      end if;
      return Result : constant Boolean :=
         Populate_Internal (Node, Root_State)
      do
         Update_Named_Envs (Context, Unit_State.Named_Envs_Needing_Update);
      end return;
   end Populate_Lexical_Env;
   ------------------------------
   -- AST_Envs_Node_Text_Image --
   ------------------------------
   function AST_Envs_Node_Text_Image
     (Node  : Bare_Ada_Node;
      Short : Boolean := True) return Text_Type is
   begin
      if Short then
         if Node = null then
            return "null";
         end if;
         return To_Text (Basename (Node.Unit))
           & ":" & To_Text (Image (Start_Sloc (Sloc_Range (Node))));
      else
         return Short_Text_Image (Node);
      end if;
   end AST_Envs_Node_Text_Image;
   -------------------
   -- Is_Rebindable --
   -------------------
   function Is_Rebindable (Node : Bare_Ada_Node) return Boolean is
   begin
         return Node.Kind in Ada_Generic_Package_Decl .. Ada_Generic_Subp_Decl;
   end Is_Rebindable;
   -----------------------
   -- Acquire_Rebinding --
   -----------------------
   function Acquire_Rebinding
     (Node             : Bare_Ada_Node;
      Parent           : Env_Rebindings;
      Old_Env, New_Env : Lexical_Env) return Env_Rebindings
   is
      Result    : Env_Rebindings;
      Available : Env_Rebindings_Vectors.Vector renames
         Node.Unit.Context.Available_Rebindings;
   begin
      --  Use an existing and available Env_Rebindings_Type record for Node's
      --  Context, otherwise allocate a new rebinding.
      Result := (if Available.Is_Empty
                 then new Env_Rebindings_Type'(Version => 0, others => <>)
                 else Available.Pop);
      Result.Parent := Parent;
      Result.Old_Env := Old_Env;
      Result.New_Env := New_Env;
      Result.Children := Env_Rebindings_Vectors.Empty_Vector;
      return Result;
   end Acquire_Rebinding;
   -----------------------
   -- Release_Rebinding --
   -----------------------
   procedure Release_Rebinding (Self : in out Env_Rebindings) is
      Available : Env_Rebindings_Vectors.Vector renames
         Unwrap (Self.Old_Env).Node.Unit.Context.Available_Rebindings;
   begin
      --  Bumping the version number, to invalidate existing references to
      --  Self.
      Self.Version := Self.Version + 1;
      Self.Children.Destroy;
      Available.Append (Self);
      Self := null;
   end Release_Rebinding;
   ------------------------
   -- Register_Rebinding --
   ------------------------
   procedure Register_Rebinding
     (Node : Bare_Ada_Node; Rebinding : Env_Rebindings) is
   begin
      Node.Unit.Rebindings.Append (Rebinding);
   end Register_Rebinding;
      -------------------------------
      -- Lexical_Env_Cache_Updated --
      -------------------------------
      procedure Lexical_Env_Cache_Updated
        (Node         : Bare_Ada_Node;
         Delta_Amount : Long_Long_Integer)
      is
         Ctx : constant Internal_Context := Node.Unit.Context;
         Ctx_Stats : Context_Env_Caches_Stats renames Ctx.Env_Caches_Stats;
         All_Env_Caches_Entry_Count : constant Long_Long_Integer :=
           Ctx_Stats.Entry_Count;
         --  Snapshot of the total number of entries before any collection is
         --  attempted. This is needed because if a collection happens,
         --  ``Ctx.Env_Caches_Stats.Entry_Count`` will be updated on the go.
      begin
         Ctx_Stats.Entry_Count :=
           All_Env_Caches_Entry_Count + Delta_Amount;
         Node.Unit.Env_Caches_Stats.Entry_Count :=
           Node.Unit.Env_Caches_Stats.Entry_Count + Delta_Amount;
         --  If the number of entries exceeds the threshold we had set, attempt
         --  to invalidate caches. Don't do anything if this notification was
         --  for removed entries (i.e. if ``Delta_Amount`` is negative) as it
         --  could mean that we are already in the middle of a collection.
         if Delta_Amount > 0 and then
            Ctx_Stats.Entry_Count > Ctx.Env_Caches_Collection_Threshold
         then
            if Cache_Invalidation_Trace.Is_Active then
               Cache_Invalidation_Trace.Trace
                 ("Attempting cache collection because number of entries"
                  & " reached" & All_Env_Caches_Entry_Count'Image);
               Cache_Invalidation_Trace.Increase_Indent;
            end if;
            for Unit of Ctx.Units loop
               if Libadalang.Implementation.Extensions.Should_Collect_Env_Caches
                 (Ctx, Unit, All_Env_Caches_Entry_Count)
               then
                  --  Clear all caches and set counters that are meant to
                  --  track events since the unit's last collection.
                  Reset_Envs_Caches (Unit);
                  Unit.Env_Caches_Stats.Lookup_Count := 0;
                  Unit.Env_Caches_Stats.Hit_Count := 0;
                  Unit.Env_Caches_Stats.Last_Overall_Lookup_Count :=
                    Ctx_Stats.Lookup_Count;
               end if;
               Unit.Env_Caches_Stats.Previous_Lookup_Count :=
                 Unit.Env_Caches_Stats.Lookup_Count;
            end loop;
            Ctx_Stats.Previous_Lookup_Count := Ctx_Stats.Lookup_Count;
            --  Setup the threshold so that the next collection is triggered
            --  when we reach the current number of entries (after this
            --  collection) plus the increment that is defined for this
            --  language.
            Ctx.Env_Caches_Collection_Threshold :=
              Ctx_Stats.Entry_Count
              + 100000;
            if Cache_Invalidation_Trace.Is_Active then
               Cache_Invalidation_Trace.Trace
                 ("New collection threshold :"
                  & Ctx.Env_Caches_Collection_Threshold'Image);
               Cache_Invalidation_Trace.Decrease_Indent;
            end if;
         end if;
      end Lexical_Env_Cache_Updated;
      ---------------------------------
      -- Lexical_Env_Cache_Looked_Up --
      ---------------------------------
      procedure Lexical_Env_Cache_Looked_Up (Node : Bare_Ada_Node) is
         Unit : constant Internal_Unit := Node.Unit;
         Ctx  : constant Internal_Context := Unit.Context;
      begin
         Unit.Env_Caches_Stats.Lookup_Count :=
            Unit.Env_Caches_Stats.Lookup_Count + 1;
         Ctx.Env_Caches_Stats.Lookup_Count :=
            Ctx.Env_Caches_Stats.Lookup_Count + 1;
      end Lexical_Env_Cache_Looked_Up;
      ---------------------------
      -- Lexical_Env_Cache_Hit --
      ---------------------------
      procedure Lexical_Env_Cache_Hit (Node : Bare_Ada_Node) is
         Unit : constant Internal_Unit := Node.Unit;
      begin
         Unit.Env_Caches_Stats.Hit_Count :=
            Unit.Env_Caches_Stats.Hit_Count + 1;
      end Lexical_Env_Cache_Hit;
   --------------------
   -- Element_Parent --
   --------------------
   function Element_Parent
     (Node : Bare_Ada_Node) return Bare_Ada_Node
   is (Node.Parent);
   ---------------
   -- Node_Unit --
   ---------------
   function Node_Unit (Node : Bare_Ada_Node) return Generic_Unit_Ptr is
   begin
      return Convert_Unit (Node.Unit);
   end Node_Unit;
   ----------
   -- Hash --
   ----------
   function Hash (Node : Bare_Ada_Node) return Hash_Type
   is
      function H is new Hash_Access
        (Root_Node_Record, Bare_Ada_Node);
   begin
      return H (Node);
   end Hash;
      function Hash (B : Boolean) return Hash_Type is (Boolean'Pos (B));
      function Hash (I : Integer) return Hash_Type is (Hash_Type'Mod (I));
   ------------------------
   -- Named environments --
   ------------------------
   ---------
   -- Add --
   ---------
   procedure Add
     (Self : in out NED_Assoc_Maps.Map;
      Key  : Symbol_Type;
      Node : AST_Envs.Internal_Map_Node)
   is
      use NED_Assoc_Maps;
      Pos   : Cursor;
      Dummy : Boolean;
   begin
      --  Make sure there is a vector entry for Key
      Self.Insert (Key, Internal_Map_Node_Vectors.Empty_Vector, Pos, Dummy);
      --  Append Node to that vector
      declare
         V : Internal_Map_Node_Vectors.Vector renames Self.Reference (Pos);
      begin
         V.Append (Node);
      end;
   end Add;
   ------------
   -- Remove --
   ------------
   procedure Remove
     (Self : in out NED_Assoc_Maps.Map;
      Key  : Symbol_Type;
      Node : Bare_Ada_Node)
   is
      use NED_Assoc_Maps;
      V : Internal_Map_Node_Vectors.Vector renames Self.Reference (Key);
   begin
      --  Remove the (assumed unique) entry in V whose node is Node. The order
      --  of items in V is not significant, so we can use Pop for efficient
      --  removal. Do the traversal in reverse order for correctness.
      for I in reverse 1 .. V.Length loop
         if V.Get_Access (I).Node = Node then
            V.Pop (I);
            exit;
         end if;
      end loop;
   end Remove;
   ------------------------------
   -- Get_Named_Env_Descriptor --
   ------------------------------
   function Get_Named_Env_Descriptor
     (Context : Internal_Context;
      Name    : Symbol_Type) return Named_Env_Descriptor_Access
   is
      use NED_Maps;
      --  Look for an existing entry for Name
      Pos : constant Cursor := Context.Named_Envs.Find (Name);
   begin
      if Has_Element (Pos) then
         return Element (Pos);
      end if;
      --  There is no such entry: create one
      return Result : constant Named_Env_Descriptor_Access :=
         new Named_Env_Descriptor'
           (Name                   => Name,
            Envs                   => <>,
            Env_With_Precedence    => Empty_Env,
            Foreign_Nodes          => <>,
            Foreign_Envs           => <>,
            Nodes_With_Foreign_Env => <>)
      do
         Context.Named_Envs.Insert (Name, Result);
      end return;
   end Get_Named_Env_Descriptor;
   ------------------------
   -- Register_Named_Env --
   ------------------------
   procedure Register_Named_Env
     (Context                   : Internal_Context;
      Name                      : Symbol_Type;
      Env                       : Lexical_Env;
      Named_Envs_Needing_Update : in out NED_Maps.Map)
   is
      NED_Access : constant Named_Env_Descriptor_Access :=
         Get_Named_Env_Descriptor (Context, Name);
      NED        : Named_Env_Descriptor renames NED_Access.all;
      Node       : constant Bare_Ada_Node := Env_Node (Env);
   begin
      NED.Envs.Insert (Node, Env);
      Node.Unit.Named_Envs.Append ((Name, Env));
      --  If that insertion must change the env that has precedence, signal
      --  that NED requires an update.
      if NED.Envs.First_Element /= NED.Env_With_Precedence then
         Named_Envs_Needing_Update.Include (Name, NED_Access);
      end if;
   end Register_Named_Env;
   ----------------------
   -- Update_Named_Env --
   ----------------------
   procedure Update_Named_Envs
     (Context : Internal_Context; Named_Envs : NED_Maps.Map)
   is
      Require_Cache_Reset : Boolean := False;
   begin
      for Cur in Named_Envs.Iterate loop
         declare
            NE      : Named_Env_Descriptor renames NED_Maps.Element (Cur).all;
            New_Env : constant Lexical_Env :=
              (if NE.Envs.Is_Empty
               then Empty_Env
               else NE.Envs.First_Element);
         begin
            --  If there was an environment with precedence, remove its foreign
            --  nodes.
            if NE.Env_With_Precedence /= Empty_Env then
               for Cur in NE.Foreign_Nodes.Iterate loop
                  declare
                     Key   : constant Symbol_Type :=
                        NED_Assoc_Maps.Key (Cur);
                     Nodes : Internal_Map_Node_Vectors.Vector renames
                        NE.Foreign_Nodes.Reference (Cur);
                  begin
                     for N of Nodes loop
                        Remove (NE.Env_With_Precedence, Thin (Key), N.Node);
                     end loop;
                  end;
               end loop;
            end if;
            --  Now, set the new environment that has precedence
            NE.Env_With_Precedence := New_Env;
            --  Add the foreign nodes to the new environment with precedence,
            --  if any.
            for Cur in NE.Foreign_Nodes.Iterate loop
               declare
                  Key   : constant Symbol_Type :=
                     NED_Assoc_Maps.Key (Cur);
                  Nodes : Internal_Map_Node_Vectors.Vector renames
                     NE.Foreign_Nodes.Reference (Cur);
               begin
                  for N of Nodes loop
                     Add (New_Env, Thin (Key), N.Node, N.Md, N.Resolver);
                  end loop;
               end;
            end loop;
            --  Set the parent environment of all foreign environments
            for Cur in NE.Foreign_Envs.Iterate loop
               declare
                  Env : Lexical_Env_Record renames
                     Unwrap (Sorted_Env_Maps.Element (Cur)).all;
               begin
                  Env.Parent := New_Env;
                  --  We have updated the lexical env hierarchy (e.g. an env
                  --  which had no parent may have one now), so the cached
                  --  entries for queries that traveresed the old env hierarchy
                  --  need to be invalidated.
                  Require_Cache_Reset := True;
               end;
            end loop;
            --  Update nodes whose environment was the old env with precedence
            for N of NE.Nodes_With_Foreign_Env loop
               N.Self_Env := New_Env;
            end loop;
         end;
      end loop;
      if Require_Cache_Reset then
         Invalidate_Caches (Context, Invalidate_Envs => True);
      end if;
   end Update_Named_Envs;
   --------------------------
   -- Big integers wrapper --
   --------------------------
   ------------------------
   -- Create_Big_Integer --
   ------------------------
   function Create_Big_Integer
     (Image : String; Base : Integer := 10) return Big_Integer_Type
   is
      use GNATCOLL.GMP;
      use GNATCOLL.GMP.Integers;
   begin
      return new Big_Integer_Record'(Value     => Make (Image, Int (Base)),
                                     Ref_Count => 1);
   end Create_Big_Integer;
   ------------------------
   -- Create_Big_Integer --
   ------------------------
   function Create_Big_Integer
     (Big_Int : GNATCOLL.GMP.Integers.Big_Integer) return Big_Integer_Type
   is
      Result : constant Big_Integer_Type :=
         new Big_Integer_Record'(Value     => <>,
                                 Ref_Count => 1);
   begin
      Result.Value.Set (Big_Int);
      return Result;
   end Create_Big_Integer;
   ------------------------
   -- Create_Big_Integer --
   ------------------------
   function Create_Big_Integer (Int : Integer) return Big_Integer_Type is
      Result : constant Big_Integer_Type :=
         new Big_Integer_Record'(Value     => <>,
                                 Ref_Count => 1);
   begin
      Result.Value.Set (GNATCOLL.GMP.Long (Int));
      return Result;
   end Create_Big_Integer;
   -------------------------------
   -- Create_Public_Big_Integer --
   -------------------------------
   function Create_Public_Big_Integer
     (Big_Int : Big_Integer_Type) return GNATCOLL.GMP.Integers.Big_Integer is
   begin
      return Result : GNATCOLL.GMP.Integers.Big_Integer do
         Result.Set (Big_Int.Value);
      end return;
   end Create_Public_Big_Integer;
   -----------------
   -- Trace_Image --
   -----------------
   function Trace_Image (I : Big_Integer_Type) return String is
   begin
      return GNATCOLL.GMP.Integers.Image (I.Value);
   end Trace_Image;
   ----------------
   -- To_Integer --
   ----------------
   function To_Integer
     (Self    : Bare_Ada_Node;
      Big_Int : Big_Integer_Type) return Integer
   is
      Image : constant String := Big_Int.Value.Image;
   begin
      return Integer'Value (Image);
   exception
      when Constraint_Error =>
         Raise_Property_Exception
           (Self, Property_Error'Identity, "out of range big integer");
   end To_Integer;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (Big_Int : Big_Integer_Type) is
   begin
      if Big_Int.Ref_Count /= -1 then
         Big_Int.Ref_Count := Big_Int.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (Big_Int : in out Big_Integer_Type) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Big_Integer_Record, Big_Integer_Type);
   begin
      if Big_Int = null or else Big_Int.Ref_Count = -1 then
         return;
      end if;
      Big_Int.Ref_Count := Big_Int.Ref_Count - 1;
      if Big_Int.Ref_Count = 0 then
         Destroy (Big_Int);
      end if;
   end Dec_Ref;
   ----------------
   -- Equivalent --
   ----------------
   function Equivalent (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value = Right.Value;
   end Equivalent;
   ---------
   -- "<" --
   ---------
   function "<" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value < Right.Value;
   end "<";
   ----------
   -- "<=" --
   ----------
   function "<=" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value <= Right.Value;
   end "<=";
   ---------
   -- ">" --
   ---------
   function ">" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value > Right.Value;
   end ">";
   ----------
   -- ">=" --
   ----------
   function ">=" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value >= Right.Value;
   end ">=";
   ---------
   -- "+" --
   ---------
   function "+" (Left, Right : Big_Integer_Type) return Big_Integer_Type is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Create_Big_Integer (Left.Value + Right.Value);
   end "+";
   ---------
   -- "-" --
   ---------
   function "-" (Left, Right : Big_Integer_Type) return Big_Integer_Type is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Create_Big_Integer (Left.Value - Right.Value);
   end "-";
   ---------
   -- "-" --
   ---------
   function "-" (Value : Big_Integer_Type) return Big_Integer_Type is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Create_Big_Integer (-Value.Value);
   end "-";
   ------------------
   -- Unit_Version --
   ------------------
   function Unit_Version (Unit : Generic_Unit_Ptr) return Version_Number is
   begin
      return Convert_Unit (Unit).Unit_Version;
   end Unit_Version;
   -------------------------
   -- Get_Context_Version --
   -------------------------
   function Get_Context_Version
     (Node : Bare_Ada_Node) return Version_Number is
   begin
      return Node.Unit.Context.Cache_Version;
   end Get_Context_Version;
   ---------------
   --  Self_Env --
   ---------------
   function Self_Env (Node : Bare_Ada_Node) return Lexical_Env is
   begin
      return Node.Self_Env;
   end Self_Env;
   --------------------------
   -- Properties_May_Raise --
   --------------------------
   function Properties_May_Raise
     (Exc : Ada.Exceptions.Exception_Occurrence) return Boolean is
   begin
      return Ada.Exceptions.Exception_Identity (Exc) in
            Precondition_Failure'Identity
               |
            Property_Error'Identity
      ;
   end Properties_May_Raise;
   ----------------------
   -- Short_Text_Image --
   ----------------------
   function Short_Text_Image (Self : Bare_Ada_Node) return Text_Type
   is
   begin
      if Self = null then
         return "None";
      end if;
   case Self.Kind is
            when Ada_Char_Literal .. Ada_String_Literal =>
         return Base_Id_Short_Image (Self);
            when Ada_Defining_Name .. Ada_Synthetic_Defining_Name =>
         return Defining_Name_Short_Image (Self);
            when Ada_Abstract_State_Decl .. Ada_Synthetic_Object_Decl =>
         return Basic_Decl_Short_Image (Self);
      when others => 
         return "<" & To_Text (Kind_Name (Self))
                & " "
                & To_Text
                  (Ada.Directories.Simple_Name
                     (Get_Filename (Unit (Self))))
                & ":" & To_Text (Image (Sloc_Range (Self))) & ">";
   end case;
   end Short_Text_Image;
   --------------------
   -- Snaps_At_Start --
   --------------------
   function Snaps_At_Start (Self : Bare_Ada_Node) return Boolean is
   begin
   case Self.Kind is
            when Ada_Handled_Stmts =>
         return True;
            when Ada_Declarative_Part .. Ada_Public_Part =>
         return True;
      when others => 
         return False;
   end case;
   end Snaps_At_Start;
   ------------------
   -- Snaps_At_End --
   ------------------
   function Snaps_At_End (Self : Bare_Ada_Node) return Boolean is
   begin
   case Self.Kind is
            when Ada_Handled_Stmts =>
         return True;
            when Ada_Declarative_Part .. Ada_Public_Part =>
         return True;
      when others => 
         return Is_Incomplete (Self);
   end case;
   end Snaps_At_End;
   -------------
   -- Parents --
   -------------
   function Parents
     (Node      : Bare_Ada_Node;
      With_Self : Boolean := True)
      return Bare_Ada_Node_Array_Access
   is
      Count : Natural := 0;
      Start : Bare_Ada_Node :=
        (if With_Self then Node else Node.Parent);
      Cur   : Bare_Ada_Node := Start;
   begin
      while Cur /= null loop
         Count := Count + 1;
         Cur := Cur.Parent;
      end loop;
      declare
         Result : constant Bare_Ada_Node_Array_Access :=
            Create_Bare_Ada_Node_Array (Count);
      begin
         Cur := Start;
         for I in Result.Items'Range loop
            Result.Items (I) := Cur;
            Cur := Cur.Parent;
         end loop;
         return Result;
      end;
   end Parents;
   -----------------------
   -- First_Child_Index --
   -----------------------
   function First_Child_Index (Node : Bare_Ada_Node) return Natural
   is (1);
   ----------------------
   -- Last_Child_Index --
   ----------------------
   function Last_Child_Index (Node : Bare_Ada_Node) return Natural
   is (Children_Count (Node));
   ---------------
   -- Get_Child --
   ---------------
   procedure Get_Child
     (Node            : Bare_Ada_Node;
      Index           : Positive;
      Index_In_Bounds : out Boolean;
      Result          : out Bare_Ada_Node)
   is
      K : constant Ada_Node_Kind_Type := Node.Kind;
   begin
      Index_In_Bounds := True;
      Result := null;
      case Ada_Ada_Node (K) is
when Ada_Ada_List =>
declare
N_Bare_Ada_List : constant Bare_Ada_List := Node;
begin
                    if Index > N_Bare_Ada_List.Count then
                        Index_In_Bounds := False;
                    else
                        Result := N_Bare_Ada_List.Nodes (Index);
                    end if;
                    return;
end;
when Ada_Constrained_Array_Indices_Range =>
declare
N_Bare_Constrained_Array_Indices : constant Bare_Constrained_Array_Indices := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Constrained_Array_Indices.Constrained_Array_Indices_F_List;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Unconstrained_Array_Indices_Range =>
declare
N_Bare_Unconstrained_Array_Indices : constant Bare_Unconstrained_Array_Indices := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Unconstrained_Array_Indices.Unconstrained_Array_Indices_F_Types;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Aspect_Assoc_Range =>
declare
N_Bare_Aspect_Assoc : constant Bare_Aspect_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Aspect_Assoc.Aspect_Assoc_F_Id;
                            return;
                        when 2 =>
                            Result := N_Bare_Aspect_Assoc.Aspect_Assoc_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_At_Clause_Range =>
declare
N_Bare_At_Clause : constant Bare_At_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_At_Clause.At_Clause_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_At_Clause.At_Clause_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Attribute_Def_Clause_Range =>
declare
N_Bare_Attribute_Def_Clause : constant Bare_Attribute_Def_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Attribute_Def_Clause.Attribute_Def_Clause_F_Attribute_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Attribute_Def_Clause.Attribute_Def_Clause_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Enum_Rep_Clause_Range =>
declare
N_Bare_Enum_Rep_Clause : constant Bare_Enum_Rep_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Enum_Rep_Clause.Enum_Rep_Clause_F_Type_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Enum_Rep_Clause.Enum_Rep_Clause_F_Aggregate;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Record_Rep_Clause_Range =>
declare
N_Bare_Record_Rep_Clause : constant Bare_Record_Rep_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Record_Rep_Clause.Record_Rep_Clause_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Record_Rep_Clause.Record_Rep_Clause_F_At_Expr;
                            return;
                        when 3 =>
                            Result := N_Bare_Record_Rep_Clause.Record_Rep_Clause_F_Components;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Aspect_Spec_Range =>
declare
N_Bare_Aspect_Spec : constant Bare_Aspect_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Aspect_Spec.Aspect_Spec_F_Aspect_Assocs;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Contract_Case_Assoc_Range =>
declare
N_Bare_Contract_Case_Assoc : constant Bare_Contract_Case_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Contract_Case_Assoc.Contract_Case_Assoc_F_Guard;
                            return;
                        when 2 =>
                            Result := N_Bare_Contract_Case_Assoc.Contract_Case_Assoc_F_Consequence;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Pragma_Argument_Assoc_Range =>
declare
N_Bare_Pragma_Argument_Assoc : constant Bare_Pragma_Argument_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Pragma_Argument_Assoc.Pragma_Argument_Assoc_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Pragma_Argument_Assoc.Pragma_Argument_Assoc_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Entry_Spec_Range =>
declare
N_Bare_Entry_Spec : constant Bare_Entry_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Entry_Spec.Entry_Spec_F_Entry_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Entry_Spec.Entry_Spec_F_Family_Type;
                            return;
                        when 3 =>
                            Result := N_Bare_Entry_Spec.Entry_Spec_F_Entry_Params;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subp_Spec_Range =>
declare
N_Bare_Subp_Spec : constant Bare_Subp_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Subp_Spec.Subp_Spec_F_Subp_Kind;
                            return;
                        when 2 =>
                            Result := N_Bare_Subp_Spec.Subp_Spec_F_Subp_Name;
                            return;
                        when 3 =>
                            Result := N_Bare_Subp_Spec.Subp_Spec_F_Subp_Params;
                            return;
                        when 4 =>
                            Result := N_Bare_Subp_Spec.Subp_Spec_F_Subp_Returns;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Synthetic_Binary_Spec_Range =>
declare
N_Bare_Synthetic_Binary_Spec : constant Bare_Synthetic_Binary_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Synthetic_Binary_Spec.Synthetic_Binary_Spec_F_Left_Param;
                            return;
                        when 2 =>
                            Result := N_Bare_Synthetic_Binary_Spec.Synthetic_Binary_Spec_F_Right_Param;
                            return;
                        when 3 =>
                            Result := N_Bare_Synthetic_Binary_Spec.Synthetic_Binary_Spec_F_Return_Type_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Synthetic_Unary_Spec_Range =>
declare
N_Bare_Synthetic_Unary_Spec : constant Bare_Synthetic_Unary_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Synthetic_Unary_Spec.Synthetic_Unary_Spec_F_Right_Param;
                            return;
                        when 2 =>
                            Result := N_Bare_Synthetic_Unary_Spec.Synthetic_Unary_Spec_F_Return_Type_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Component_List_Range =>
declare
N_Bare_Component_List : constant Bare_Component_List := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Component_List.Component_List_F_Components;
                            return;
                        when 2 =>
                            Result := N_Bare_Component_List.Component_List_F_Variant_Part;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Known_Discriminant_Part_Range =>
declare
N_Bare_Known_Discriminant_Part : constant Bare_Known_Discriminant_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Known_Discriminant_Part.Known_Discriminant_Part_F_Discr_Specs;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Entry_Completion_Formal_Params_Range =>
declare
N_Bare_Entry_Completion_Formal_Params : constant Bare_Entry_Completion_Formal_Params := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Entry_Completion_Formal_Params.Entry_Completion_Formal_Params_F_Params;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Formal_Part_Range =>
declare
N_Bare_Generic_Formal_Part : constant Bare_Generic_Formal_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Formal_Part.Generic_Formal_Part_F_Decls;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Base_Record_Def =>
declare
N_Bare_Base_Record_Def : constant Bare_Base_Record_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Base_Record_Def.Base_Record_Def_F_Components;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Aggregate_Assoc_Range =>
declare
N_Bare_Aggregate_Assoc : constant Bare_Aggregate_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Aggregate_Assoc.Aggregate_Assoc_F_Designators;
                            return;
                        when 2 =>
                            Result := N_Bare_Aggregate_Assoc.Aggregate_Assoc_F_R_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Composite_Constraint_Assoc_Range =>
declare
N_Bare_Composite_Constraint_Assoc : constant Bare_Composite_Constraint_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Composite_Constraint_Assoc.Composite_Constraint_Assoc_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Composite_Constraint_Assoc.Composite_Constraint_Assoc_F_Constraint_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Iterated_Assoc_Range =>
declare
N_Bare_Iterated_Assoc : constant Bare_Iterated_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Iterated_Assoc.Iterated_Assoc_F_Spec;
                            return;
                        when 2 =>
                            Result := N_Bare_Iterated_Assoc.Iterated_Assoc_F_Key_Expr;
                            return;
                        when 3 =>
                            Result := N_Bare_Iterated_Assoc.Iterated_Assoc_F_R_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Param_Assoc_Range =>
declare
N_Bare_Param_Assoc : constant Bare_Param_Assoc := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Param_Assoc.Param_Assoc_F_Designator;
                            return;
                        when 2 =>
                            Result := N_Bare_Param_Assoc.Param_Assoc_F_R_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Abstract_State_Decl_Range =>
declare
N_Bare_Abstract_State_Decl : constant Bare_Abstract_State_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Abstract_State_Decl.Abstract_State_Decl_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Abstract_State_Decl.Abstract_State_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Anonymous_Expr_Decl_Range =>
declare
N_Bare_Anonymous_Expr_Decl : constant Bare_Anonymous_Expr_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Anonymous_Expr_Decl.Anonymous_Expr_Decl_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Component_Decl_Range =>
declare
N_Bare_Component_Decl : constant Bare_Component_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Component_Decl.Component_Decl_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Component_Decl.Component_Decl_F_Component_Def;
                            return;
                        when 3 =>
                            Result := N_Bare_Component_Decl.Component_Decl_F_Default_Expr;
                            return;
                        when 4 =>
                            Result := N_Bare_Component_Decl.Component_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Discriminant_Spec_Range =>
declare
N_Bare_Discriminant_Spec : constant Bare_Discriminant_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Discriminant_Spec.Discriminant_Spec_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Discriminant_Spec.Discriminant_Spec_F_Type_Expr;
                            return;
                        when 3 =>
                            Result := N_Bare_Discriminant_Spec.Discriminant_Spec_F_Default_Expr;
                            return;
                        when 4 =>
                            Result := N_Bare_Discriminant_Spec.Discriminant_Spec_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Formal =>
declare
N_Bare_Generic_Formal : constant Bare_Generic_Formal := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Formal.Generic_Formal_F_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Param_Spec_Range =>
declare
N_Bare_Param_Spec : constant Bare_Param_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Param_Spec.Param_Spec_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Param_Spec.Param_Spec_F_Has_Aliased;
                            return;
                        when 3 =>
                            Result := N_Bare_Param_Spec.Param_Spec_F_Mode;
                            return;
                        when 4 =>
                            Result := N_Bare_Param_Spec.Param_Spec_F_Type_Expr;
                            return;
                        when 5 =>
                            Result := N_Bare_Param_Spec.Param_Spec_F_Default_Expr;
                            return;
                        when 6 =>
                            Result := N_Bare_Param_Spec.Param_Spec_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Synthetic_Formal_Param_Decl_Range =>
declare
N_Bare_Synthetic_Formal_Param_Decl : constant Bare_Synthetic_Formal_Param_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Synthetic_Formal_Param_Decl.Synthetic_Formal_Param_Decl_F_Param_Type;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Base_Package_Decl =>
declare
N_Bare_Base_Package_Decl : constant Bare_Base_Package_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Base_Package_Decl.Base_Package_Decl_F_Package_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Base_Package_Decl.Base_Package_Decl_F_Aspects;
                            return;
                        when 3 =>
                            Result := N_Bare_Base_Package_Decl.Base_Package_Decl_F_Public_Part;
                            return;
                        when 4 =>
                            Result := N_Bare_Base_Package_Decl.Base_Package_Decl_F_Private_Part;
                            return;
                        when 5 =>
                            Result := N_Bare_Base_Package_Decl.Base_Package_Decl_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Base_Type_Decl =>
declare
N_Bare_Base_Type_Decl : constant Bare_Base_Type_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Base_Type_Decl.Base_Type_Decl_F_Name;
                            return;
                        when others => null;
                    end case;
case Ada_Base_Type_Decl (K) is
when Ada_Subtype_Decl_Range =>
declare
N_Bare_Subtype_Decl : constant Bare_Subtype_Decl := N_Bare_Base_Type_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Subtype_Decl.Subtype_Decl_F_Subtype;
                            return;
                        when 3 =>
                            Result := N_Bare_Subtype_Decl.Subtype_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Incomplete_Type_Decl_Range =>
declare
N_Bare_Incomplete_Type_Decl : constant Bare_Incomplete_Type_Decl := N_Bare_Base_Type_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Incomplete_Type_Decl.Incomplete_Type_Decl_F_Discriminants;
                            return;
                        when others => null;
                    end case;
case Ada_Incomplete_Type_Decl_Range (K) is
when Ada_Incomplete_Formal_Type_Decl_Range =>
declare
N_Bare_Incomplete_Formal_Type_Decl : constant Bare_Incomplete_Formal_Type_Decl := N_Bare_Incomplete_Type_Decl;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Incomplete_Formal_Type_Decl.Incomplete_Formal_Type_Decl_F_Is_Tagged;
                            return;
                        when 4 =>
                            Result := N_Bare_Incomplete_Formal_Type_Decl.Incomplete_Formal_Type_Decl_F_Default_Type;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Incomplete_Tagged_Type_Decl_Range =>
declare
N_Bare_Incomplete_Tagged_Type_Decl : constant Bare_Incomplete_Tagged_Type_Decl := N_Bare_Incomplete_Type_Decl;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Incomplete_Tagged_Type_Decl.Incomplete_Tagged_Type_Decl_F_Has_Abstract;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when Ada_Protected_Type_Decl_Range =>
declare
N_Bare_Protected_Type_Decl : constant Bare_Protected_Type_Decl := N_Bare_Base_Type_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Protected_Type_Decl.Protected_Type_Decl_F_Discriminants;
                            return;
                        when 3 =>
                            Result := N_Bare_Protected_Type_Decl.Protected_Type_Decl_F_Aspects;
                            return;
                        when 4 =>
                            Result := N_Bare_Protected_Type_Decl.Protected_Type_Decl_F_Interfaces;
                            return;
                        when 5 =>
                            Result := N_Bare_Protected_Type_Decl.Protected_Type_Decl_F_Definition;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Task_Type_Decl_Range =>
declare
N_Bare_Task_Type_Decl : constant Bare_Task_Type_Decl := N_Bare_Base_Type_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Task_Type_Decl.Task_Type_Decl_F_Discriminants;
                            return;
                        when 3 =>
                            Result := N_Bare_Task_Type_Decl.Task_Type_Decl_F_Aspects;
                            return;
                        when 4 =>
                            Result := N_Bare_Task_Type_Decl.Task_Type_Decl_F_Definition;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Type_Decl =>
declare
N_Bare_Type_Decl : constant Bare_Type_Decl := N_Bare_Base_Type_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Type_Decl.Type_Decl_F_Discriminants;
                            return;
                        when 3 =>
                            Result := N_Bare_Type_Decl.Type_Decl_F_Type_Def;
                            return;
                        when others => null;
                    end case;
case Ada_Type_Decl (K) is
when Ada_Concrete_Type_Decl_Range =>
declare
N_Bare_Concrete_Type_Decl : constant Bare_Concrete_Type_Decl := N_Bare_Type_Decl;
begin
case Index is
                        when 4 =>
                            Result := N_Bare_Concrete_Type_Decl.Concrete_Type_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Formal_Type_Decl_Range =>
declare
N_Bare_Formal_Type_Decl : constant Bare_Formal_Type_Decl := N_Bare_Type_Decl;
begin
case Index is
                        when 4 =>
                            Result := N_Bare_Formal_Type_Decl.Formal_Type_Decl_F_Default_Type;
                            return;
                        when 5 =>
                            Result := N_Bare_Formal_Type_Decl.Formal_Type_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when others => null;
end case;
end;
when Ada_Classic_Subp_Decl =>
declare
N_Bare_Classic_Subp_Decl : constant Bare_Classic_Subp_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Classic_Subp_Decl.Classic_Subp_Decl_F_Overriding;
                            return;
                        when 2 =>
                            Result := N_Bare_Classic_Subp_Decl.Classic_Subp_Decl_F_Subp_Spec;
                            return;
                        when others => null;
                    end case;
case Ada_Classic_Subp_Decl (K) is
when Ada_Abstract_Subp_Decl_Range =>
declare
N_Bare_Abstract_Subp_Decl : constant Bare_Abstract_Subp_Decl := N_Bare_Classic_Subp_Decl;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Abstract_Subp_Decl.Abstract_Subp_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Formal_Subp_Decl =>
declare
N_Bare_Formal_Subp_Decl : constant Bare_Formal_Subp_Decl := N_Bare_Classic_Subp_Decl;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Formal_Subp_Decl.Formal_Subp_Decl_F_Default_Expr;
                            return;
                        when 4 =>
                            Result := N_Bare_Formal_Subp_Decl.Formal_Subp_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subp_Decl_Range =>
declare
N_Bare_Subp_Decl : constant Bare_Subp_Decl := N_Bare_Classic_Subp_Decl;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Subp_Decl.Subp_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when Ada_Entry_Decl_Range =>
declare
N_Bare_Entry_Decl : constant Bare_Entry_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Entry_Decl.Entry_Decl_F_Overriding;
                            return;
                        when 2 =>
                            Result := N_Bare_Entry_Decl.Entry_Decl_F_Spec;
                            return;
                        when 3 =>
                            Result := N_Bare_Entry_Decl.Entry_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Enum_Literal_Decl_Range =>
declare
N_Bare_Enum_Literal_Decl : constant Bare_Enum_Literal_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Enum_Literal_Decl.Enum_Literal_Decl_F_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Subp_Internal_Range =>
declare
N_Bare_Generic_Subp_Internal : constant Bare_Generic_Subp_Internal := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Subp_Internal.Generic_Subp_Internal_F_Subp_Spec;
                            return;
                        when 2 =>
                            Result := N_Bare_Generic_Subp_Internal.Generic_Subp_Internal_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Synthetic_Subp_Decl_Range =>
declare
N_Bare_Synthetic_Subp_Decl : constant Bare_Synthetic_Subp_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Synthetic_Subp_Decl.Synthetic_Subp_Decl_F_Spec;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Accept_Stmt_Body_Range =>
declare
N_Bare_Accept_Stmt_Body : constant Bare_Accept_Stmt_Body := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Accept_Stmt_Body.Accept_Stmt_Body_F_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Base_Subp_Body =>
declare
N_Bare_Base_Subp_Body : constant Bare_Base_Subp_Body := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Base_Subp_Body.Base_Subp_Body_F_Overriding;
                            return;
                        when 2 =>
                            Result := N_Bare_Base_Subp_Body.Base_Subp_Body_F_Subp_Spec;
                            return;
                        when others => null;
                    end case;
case Ada_Base_Subp_Body (K) is
when Ada_Expr_Function_Range =>
declare
N_Bare_Expr_Function : constant Bare_Expr_Function := N_Bare_Base_Subp_Body;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Expr_Function.Expr_Function_F_Expr;
                            return;
                        when 4 =>
                            Result := N_Bare_Expr_Function.Expr_Function_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Null_Subp_Decl_Range =>
declare
N_Bare_Null_Subp_Decl : constant Bare_Null_Subp_Decl := N_Bare_Base_Subp_Body;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Null_Subp_Decl.Null_Subp_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subp_Body_Range =>
declare
N_Bare_Subp_Body : constant Bare_Subp_Body := N_Bare_Base_Subp_Body;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Subp_Body.Subp_Body_F_Aspects;
                            return;
                        when 4 =>
                            Result := N_Bare_Subp_Body.Subp_Body_F_Decls;
                            return;
                        when 5 =>
                            Result := N_Bare_Subp_Body.Subp_Body_F_Stmts;
                            return;
                        when 6 =>
                            Result := N_Bare_Subp_Body.Subp_Body_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subp_Renaming_Decl_Range =>
declare
N_Bare_Subp_Renaming_Decl : constant Bare_Subp_Renaming_Decl := N_Bare_Base_Subp_Body;
begin
case Index is
                        when 3 =>
                            Result := N_Bare_Subp_Renaming_Decl.Subp_Renaming_Decl_F_Renames;
                            return;
                        when 4 =>
                            Result := N_Bare_Subp_Renaming_Decl.Subp_Renaming_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when Ada_Package_Body_Stub_Range =>
declare
N_Bare_Package_Body_Stub : constant Bare_Package_Body_Stub := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Package_Body_Stub.Package_Body_Stub_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Package_Body_Stub.Package_Body_Stub_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Protected_Body_Stub_Range =>
declare
N_Bare_Protected_Body_Stub : constant Bare_Protected_Body_Stub := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Protected_Body_Stub.Protected_Body_Stub_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Protected_Body_Stub.Protected_Body_Stub_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subp_Body_Stub_Range =>
declare
N_Bare_Subp_Body_Stub : constant Bare_Subp_Body_Stub := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Subp_Body_Stub.Subp_Body_Stub_F_Overriding;
                            return;
                        when 2 =>
                            Result := N_Bare_Subp_Body_Stub.Subp_Body_Stub_F_Subp_Spec;
                            return;
                        when 3 =>
                            Result := N_Bare_Subp_Body_Stub.Subp_Body_Stub_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Task_Body_Stub_Range =>
declare
N_Bare_Task_Body_Stub : constant Bare_Task_Body_Stub := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Task_Body_Stub.Task_Body_Stub_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Task_Body_Stub.Task_Body_Stub_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Entry_Body_Range =>
declare
N_Bare_Entry_Body : constant Bare_Entry_Body := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Entry_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Index_Spec;
                            return;
                        when 3 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Params;
                            return;
                        when 4 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Aspects;
                            return;
                        when 5 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Barrier;
                            return;
                        when 6 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Decls;
                            return;
                        when 7 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_Stmts;
                            return;
                        when 8 =>
                            Result := N_Bare_Entry_Body.Entry_Body_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Package_Body_Range =>
declare
N_Bare_Package_Body : constant Bare_Package_Body := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Package_Body.Package_Body_F_Package_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Package_Body.Package_Body_F_Aspects;
                            return;
                        when 3 =>
                            Result := N_Bare_Package_Body.Package_Body_F_Decls;
                            return;
                        when 4 =>
                            Result := N_Bare_Package_Body.Package_Body_F_Stmts;
                            return;
                        when 5 =>
                            Result := N_Bare_Package_Body.Package_Body_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Protected_Body_Range =>
declare
N_Bare_Protected_Body : constant Bare_Protected_Body := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Protected_Body.Protected_Body_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Protected_Body.Protected_Body_F_Aspects;
                            return;
                        when 3 =>
                            Result := N_Bare_Protected_Body.Protected_Body_F_Decls;
                            return;
                        when 4 =>
                            Result := N_Bare_Protected_Body.Protected_Body_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Task_Body_Range =>
declare
N_Bare_Task_Body : constant Bare_Task_Body := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Task_Body.Task_Body_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Task_Body.Task_Body_F_Aspects;
                            return;
                        when 3 =>
                            Result := N_Bare_Task_Body.Task_Body_F_Decls;
                            return;
                        when 4 =>
                            Result := N_Bare_Task_Body.Task_Body_F_Stmts;
                            return;
                        when 5 =>
                            Result := N_Bare_Task_Body.Task_Body_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Entry_Index_Spec_Range =>
declare
N_Bare_Entry_Index_Spec : constant Bare_Entry_Index_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Entry_Index_Spec.Entry_Index_Spec_F_Id;
                            return;
                        when 2 =>
                            Result := N_Bare_Entry_Index_Spec.Entry_Index_Spec_F_Subtype;
                            return;
                        when 3 =>
                            Result := N_Bare_Entry_Index_Spec.Entry_Index_Spec_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Exception_Decl_Range =>
declare
N_Bare_Exception_Decl : constant Bare_Exception_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Exception_Decl.Exception_Decl_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Exception_Decl.Exception_Decl_F_Renames;
                            return;
                        when 3 =>
                            Result := N_Bare_Exception_Decl.Exception_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Exception_Handler_Range =>
declare
N_Bare_Exception_Handler : constant Bare_Exception_Handler := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Exception_Handler.Exception_Handler_F_Exception_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Exception_Handler.Exception_Handler_F_Handled_Exceptions;
                            return;
                        when 3 =>
                            Result := N_Bare_Exception_Handler.Exception_Handler_F_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_For_Loop_Var_Decl_Range =>
declare
N_Bare_For_Loop_Var_Decl : constant Bare_For_Loop_Var_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_For_Loop_Var_Decl.For_Loop_Var_Decl_F_Id;
                            return;
                        when 2 =>
                            Result := N_Bare_For_Loop_Var_Decl.For_Loop_Var_Decl_F_Id_Type;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Decl =>
declare
N_Bare_Generic_Decl : constant Bare_Generic_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Decl.Generic_Decl_F_Formal_Part;
                            return;
                        when others => null;
                    end case;
case Ada_Generic_Decl (K) is
when Ada_Generic_Package_Decl_Range =>
declare
N_Bare_Generic_Package_Decl : constant Bare_Generic_Package_Decl := N_Bare_Generic_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Generic_Package_Decl.Generic_Package_Decl_F_Package_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Subp_Decl_Range =>
declare
N_Bare_Generic_Subp_Decl : constant Bare_Generic_Subp_Decl := N_Bare_Generic_Decl;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Generic_Subp_Decl.Generic_Subp_Decl_F_Subp_Decl;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when Ada_Generic_Package_Instantiation_Range =>
declare
N_Bare_Generic_Package_Instantiation : constant Bare_Generic_Package_Instantiation := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Package_Instantiation.Generic_Package_Instantiation_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Generic_Package_Instantiation.Generic_Package_Instantiation_F_Generic_Pkg_Name;
                            return;
                        when 3 =>
                            Result := N_Bare_Generic_Package_Instantiation.Generic_Package_Instantiation_F_Params;
                            return;
                        when 4 =>
                            Result := N_Bare_Generic_Package_Instantiation.Generic_Package_Instantiation_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Subp_Instantiation_Range =>
declare
N_Bare_Generic_Subp_Instantiation : constant Bare_Generic_Subp_Instantiation := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Subp_Instantiation.Generic_Subp_Instantiation_F_Overriding;
                            return;
                        when 2 =>
                            Result := N_Bare_Generic_Subp_Instantiation.Generic_Subp_Instantiation_F_Kind;
                            return;
                        when 3 =>
                            Result := N_Bare_Generic_Subp_Instantiation.Generic_Subp_Instantiation_F_Subp_Name;
                            return;
                        when 4 =>
                            Result := N_Bare_Generic_Subp_Instantiation.Generic_Subp_Instantiation_F_Generic_Subp_Name;
                            return;
                        when 5 =>
                            Result := N_Bare_Generic_Subp_Instantiation.Generic_Subp_Instantiation_F_Params;
                            return;
                        when 6 =>
                            Result := N_Bare_Generic_Subp_Instantiation.Generic_Subp_Instantiation_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Package_Renaming_Decl_Range =>
declare
N_Bare_Generic_Package_Renaming_Decl : constant Bare_Generic_Package_Renaming_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Package_Renaming_Decl.Generic_Package_Renaming_Decl_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Generic_Package_Renaming_Decl.Generic_Package_Renaming_Decl_F_Renames;
                            return;
                        when 3 =>
                            Result := N_Bare_Generic_Package_Renaming_Decl.Generic_Package_Renaming_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Generic_Subp_Renaming_Decl_Range =>
declare
N_Bare_Generic_Subp_Renaming_Decl : constant Bare_Generic_Subp_Renaming_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Generic_Subp_Renaming_Decl.Generic_Subp_Renaming_Decl_F_Kind;
                            return;
                        when 2 =>
                            Result := N_Bare_Generic_Subp_Renaming_Decl.Generic_Subp_Renaming_Decl_F_Name;
                            return;
                        when 3 =>
                            Result := N_Bare_Generic_Subp_Renaming_Decl.Generic_Subp_Renaming_Decl_F_Renames;
                            return;
                        when 4 =>
                            Result := N_Bare_Generic_Subp_Renaming_Decl.Generic_Subp_Renaming_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Label_Decl_Range =>
declare
N_Bare_Label_Decl : constant Bare_Label_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Label_Decl.Label_Decl_F_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Named_Stmt_Decl_Range =>
declare
N_Bare_Named_Stmt_Decl : constant Bare_Named_Stmt_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Named_Stmt_Decl.Named_Stmt_Decl_F_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Number_Decl_Range =>
declare
N_Bare_Number_Decl : constant Bare_Number_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Number_Decl.Number_Decl_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Number_Decl.Number_Decl_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Object_Decl_Range =>
declare
N_Bare_Object_Decl : constant Bare_Object_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Ids;
                            return;
                        when 2 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Has_Aliased;
                            return;
                        when 3 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Has_Constant;
                            return;
                        when 4 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Mode;
                            return;
                        when 5 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Type_Expr;
                            return;
                        when 6 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Default_Expr;
                            return;
                        when 7 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Renaming_Clause;
                            return;
                        when 8 =>
                            Result := N_Bare_Object_Decl.Object_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Package_Renaming_Decl_Range =>
declare
N_Bare_Package_Renaming_Decl : constant Bare_Package_Renaming_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Package_Renaming_Decl.Package_Renaming_Decl_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Package_Renaming_Decl.Package_Renaming_Decl_F_Renames;
                            return;
                        when 3 =>
                            Result := N_Bare_Package_Renaming_Decl.Package_Renaming_Decl_F_Aspects;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Single_Protected_Decl_Range =>
declare
N_Bare_Single_Protected_Decl : constant Bare_Single_Protected_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Single_Protected_Decl.Single_Protected_Decl_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Single_Protected_Decl.Single_Protected_Decl_F_Aspects;
                            return;
                        when 3 =>
                            Result := N_Bare_Single_Protected_Decl.Single_Protected_Decl_F_Interfaces;
                            return;
                        when 4 =>
                            Result := N_Bare_Single_Protected_Decl.Single_Protected_Decl_F_Definition;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Single_Task_Decl_Range =>
declare
N_Bare_Single_Task_Decl : constant Bare_Single_Task_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Single_Task_Decl.Single_Task_Decl_F_Task_Type;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Case_Stmt_Alternative_Range =>
declare
N_Bare_Case_Stmt_Alternative : constant Bare_Case_Stmt_Alternative := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Case_Stmt_Alternative.Case_Stmt_Alternative_F_Choices;
                            return;
                        when 2 =>
                            Result := N_Bare_Case_Stmt_Alternative.Case_Stmt_Alternative_F_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Compilation_Unit_Range =>
declare
N_Bare_Compilation_Unit : constant Bare_Compilation_Unit := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Compilation_Unit.Compilation_Unit_F_Prelude;
                            return;
                        when 2 =>
                            Result := N_Bare_Compilation_Unit.Compilation_Unit_F_Body;
                            return;
                        when 3 =>
                            Result := N_Bare_Compilation_Unit.Compilation_Unit_F_Pragmas;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Component_Clause_Range =>
declare
N_Bare_Component_Clause : constant Bare_Component_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Component_Clause.Component_Clause_F_Id;
                            return;
                        when 2 =>
                            Result := N_Bare_Component_Clause.Component_Clause_F_Position;
                            return;
                        when 3 =>
                            Result := N_Bare_Component_Clause.Component_Clause_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Component_Def_Range =>
declare
N_Bare_Component_Def : constant Bare_Component_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Component_Def.Component_Def_F_Has_Aliased;
                            return;
                        when 2 =>
                            Result := N_Bare_Component_Def.Component_Def_F_Has_Constant;
                            return;
                        when 3 =>
                            Result := N_Bare_Component_Def.Component_Def_F_Type_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Composite_Constraint_Range =>
declare
N_Bare_Composite_Constraint : constant Bare_Composite_Constraint := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Composite_Constraint.Composite_Constraint_F_Constraints;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Delta_Constraint_Range =>
declare
N_Bare_Delta_Constraint : constant Bare_Delta_Constraint := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Delta_Constraint.Delta_Constraint_F_Delta;
                            return;
                        when 2 =>
                            Result := N_Bare_Delta_Constraint.Delta_Constraint_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Digits_Constraint_Range =>
declare
N_Bare_Digits_Constraint : constant Bare_Digits_Constraint := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Digits_Constraint.Digits_Constraint_F_Digits;
                            return;
                        when 2 =>
                            Result := N_Bare_Digits_Constraint.Digits_Constraint_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Range_Constraint_Range =>
declare
N_Bare_Range_Constraint : constant Bare_Range_Constraint := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Range_Constraint.Range_Constraint_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Declarative_Part_Range =>
declare
N_Bare_Declarative_Part : constant Bare_Declarative_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Declarative_Part.Declarative_Part_F_Decls;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Elsif_Expr_Part_Range =>
declare
N_Bare_Elsif_Expr_Part : constant Bare_Elsif_Expr_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Elsif_Expr_Part.Elsif_Expr_Part_F_Cond_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Elsif_Expr_Part.Elsif_Expr_Part_F_Then_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Elsif_Stmt_Part_Range =>
declare
N_Bare_Elsif_Stmt_Part : constant Bare_Elsif_Stmt_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Elsif_Stmt_Part.Elsif_Stmt_Part_F_Cond_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Elsif_Stmt_Part.Elsif_Stmt_Part_F_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Abstract_State_Decl_Expr_Range =>
declare
N_Bare_Abstract_State_Decl_Expr : constant Bare_Abstract_State_Decl_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Abstract_State_Decl_Expr.Abstract_State_Decl_Expr_F_State_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Allocator_Range =>
declare
N_Bare_Allocator : constant Bare_Allocator := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Allocator.Allocator_F_Subpool;
                            return;
                        when 2 =>
                            Result := N_Bare_Allocator.Allocator_F_Type_Or_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Base_Aggregate =>
declare
N_Bare_Base_Aggregate : constant Bare_Base_Aggregate := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Base_Aggregate.Base_Aggregate_F_Ancestor_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Base_Aggregate.Base_Aggregate_F_Assocs;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Bin_Op_Range =>
declare
N_Bare_Bin_Op : constant Bare_Bin_Op := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Bin_Op.Bin_Op_F_Left;
                            return;
                        when 2 =>
                            Result := N_Bare_Bin_Op.Bin_Op_F_Op;
                            return;
                        when 3 =>
                            Result := N_Bare_Bin_Op.Bin_Op_F_Right;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Case_Expr_Alternative_Range =>
declare
N_Bare_Case_Expr_Alternative : constant Bare_Case_Expr_Alternative := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Case_Expr_Alternative.Case_Expr_Alternative_F_Choices;
                            return;
                        when 2 =>
                            Result := N_Bare_Case_Expr_Alternative.Case_Expr_Alternative_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Concat_Op_Range =>
declare
N_Bare_Concat_Op : constant Bare_Concat_Op := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Concat_Op.Concat_Op_F_First_Operand;
                            return;
                        when 2 =>
                            Result := N_Bare_Concat_Op.Concat_Op_F_Other_Operands;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Concat_Operand_Range =>
declare
N_Bare_Concat_Operand : constant Bare_Concat_Operand := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Concat_Operand.Concat_Operand_F_Operator;
                            return;
                        when 2 =>
                            Result := N_Bare_Concat_Operand.Concat_Operand_F_Operand;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Case_Expr_Range =>
declare
N_Bare_Case_Expr : constant Bare_Case_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Case_Expr.Case_Expr_F_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Case_Expr.Case_Expr_F_Cases;
                            return;
                        when others => null;
                    end case;
end;
when Ada_If_Expr_Range =>
declare
N_Bare_If_Expr : constant Bare_If_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Cond_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Then_Expr;
                            return;
                        when 3 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Alternatives;
                            return;
                        when 4 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Else_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Contract_Cases_Range =>
declare
N_Bare_Contract_Cases : constant Bare_Contract_Cases := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Contract_Cases.Contract_Cases_F_Contract_Cases;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Decl_Expr_Range =>
declare
N_Bare_Decl_Expr : constant Bare_Decl_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Decl_Expr.Decl_Expr_F_Decls;
                            return;
                        when 2 =>
                            Result := N_Bare_Decl_Expr.Decl_Expr_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Format_String_Literal_Range =>
declare
N_Bare_Format_String_Literal : constant Bare_Format_String_Literal := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Format_String_Literal.Format_String_Literal_F_Opening_Chunk;
                            return;
                        when 2 =>
                            Result := N_Bare_Format_String_Literal.Format_String_Literal_F_Mid_Exprs;
                            return;
                        when 3 =>
                            Result := N_Bare_Format_String_Literal.Format_String_Literal_F_Trailing_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Membership_Expr_Range =>
declare
N_Bare_Membership_Expr : constant Bare_Membership_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Membership_Expr.Membership_Expr_F_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Membership_Expr.Membership_Expr_F_Op;
                            return;
                        when 3 =>
                            Result := N_Bare_Membership_Expr.Membership_Expr_F_Membership_Exprs;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Array_Subcomponent_Choice_Name_Range =>
declare
N_Bare_Array_Subcomponent_Choice_Name : constant Bare_Array_Subcomponent_Choice_Name := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Array_Subcomponent_Choice_Name.Array_Subcomponent_Choice_Name_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Array_Subcomponent_Choice_Name.Array_Subcomponent_Choice_Name_F_Suffix;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Attribute_Ref_Range =>
declare
N_Bare_Attribute_Ref : constant Bare_Attribute_Ref := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Attribute_Ref.Attribute_Ref_F_Prefix;
                            return;
                        when 2 =>
                            Result := N_Bare_Attribute_Ref.Attribute_Ref_F_Attribute;
                            return;
                        when 3 =>
                            Result := N_Bare_Attribute_Ref.Attribute_Ref_F_Args;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Call_Expr_Range =>
declare
N_Bare_Call_Expr : constant Bare_Call_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Call_Expr.Call_Expr_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Call_Expr.Call_Expr_F_Suffix;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Defining_Name_Range =>
declare
N_Bare_Defining_Name : constant Bare_Defining_Name := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Defining_Name.Defining_Name_F_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Discrete_Subtype_Name_Range =>
declare
N_Bare_Discrete_Subtype_Name : constant Bare_Discrete_Subtype_Name := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Discrete_Subtype_Name.Discrete_Subtype_Name_F_Subtype;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Dotted_Name_Range =>
declare
N_Bare_Dotted_Name : constant Bare_Dotted_Name := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Dotted_Name.Dotted_Name_F_Prefix;
                            return;
                        when 2 =>
                            Result := N_Bare_Dotted_Name.Dotted_Name_F_Suffix;
                            return;
                        when others => null;
                    end case;
end;
when Ada_End_Name_Range =>
declare
N_Bare_End_Name : constant Bare_End_Name := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_End_Name.End_Name_F_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Explicit_Deref_Range =>
declare
N_Bare_Explicit_Deref : constant Bare_Explicit_Deref := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Explicit_Deref.Explicit_Deref_F_Prefix;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Qual_Expr_Range =>
declare
N_Bare_Qual_Expr : constant Bare_Qual_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Qual_Expr.Qual_Expr_F_Prefix;
                            return;
                        when 2 =>
                            Result := N_Bare_Qual_Expr.Qual_Expr_F_Suffix;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Reduce_Attribute_Ref_Range =>
declare
N_Bare_Reduce_Attribute_Ref : constant Bare_Reduce_Attribute_Ref := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Reduce_Attribute_Ref.Reduce_Attribute_Ref_F_Prefix;
                            return;
                        when 2 =>
                            Result := N_Bare_Reduce_Attribute_Ref.Reduce_Attribute_Ref_F_Attribute;
                            return;
                        when 3 =>
                            Result := N_Bare_Reduce_Attribute_Ref.Reduce_Attribute_Ref_F_Args;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Update_Attribute_Ref_Range =>
declare
N_Bare_Update_Attribute_Ref : constant Bare_Update_Attribute_Ref := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Update_Attribute_Ref.Update_Attribute_Ref_F_Prefix;
                            return;
                        when 2 =>
                            Result := N_Bare_Update_Attribute_Ref.Update_Attribute_Ref_F_Attribute;
                            return;
                        when 3 =>
                            Result := N_Bare_Update_Attribute_Ref.Update_Attribute_Ref_F_Values;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Paren_Expr_Range =>
declare
N_Bare_Paren_Expr : constant Bare_Paren_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Paren_Expr.Paren_Expr_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Quantified_Expr_Range =>
declare
N_Bare_Quantified_Expr : constant Bare_Quantified_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Quantified_Expr.Quantified_Expr_F_Quantifier;
                            return;
                        when 2 =>
                            Result := N_Bare_Quantified_Expr.Quantified_Expr_F_Loop_Spec;
                            return;
                        when 3 =>
                            Result := N_Bare_Quantified_Expr.Quantified_Expr_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Raise_Expr_Range =>
declare
N_Bare_Raise_Expr : constant Bare_Raise_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Raise_Expr.Raise_Expr_F_Exception_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Raise_Expr.Raise_Expr_F_Error_Message;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Un_Op_Range =>
declare
N_Bare_Un_Op : constant Bare_Un_Op := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Un_Op.Un_Op_F_Op;
                            return;
                        when 2 =>
                            Result := N_Bare_Un_Op.Un_Op_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_For_Loop_Iter_Filter_Range =>
declare
N_Bare_For_Loop_Iter_Filter : constant Bare_For_Loop_Iter_Filter := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_For_Loop_Iter_Filter.For_Loop_Iter_Filter_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Format_String_Chunk_Range =>
declare
N_Bare_Format_String_Chunk : constant Bare_Format_String_Chunk := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Format_String_Chunk.Format_String_Chunk_F_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Format_String_Chunk.Format_String_Chunk_F_String_Tok;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Handled_Stmts_Range =>
declare
N_Bare_Handled_Stmts : constant Bare_Handled_Stmts := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Handled_Stmts.Handled_Stmts_F_Stmts;
                            return;
                        when 2 =>
                            Result := N_Bare_Handled_Stmts.Handled_Stmts_F_Exceptions;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Library_Item_Range =>
declare
N_Bare_Library_Item : constant Bare_Library_Item := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Library_Item.Library_Item_F_Has_Private;
                            return;
                        when 2 =>
                            Result := N_Bare_Library_Item.Library_Item_F_Item;
                            return;
                        when others => null;
                    end case;
end;
when Ada_For_Loop_Spec_Range =>
declare
N_Bare_For_Loop_Spec : constant Bare_For_Loop_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_For_Loop_Spec.For_Loop_Spec_F_Var_Decl;
                            return;
                        when 2 =>
                            Result := N_Bare_For_Loop_Spec.For_Loop_Spec_F_Loop_Type;
                            return;
                        when 3 =>
                            Result := N_Bare_For_Loop_Spec.For_Loop_Spec_F_Has_Reverse;
                            return;
                        when 4 =>
                            Result := N_Bare_For_Loop_Spec.For_Loop_Spec_F_Iter_Expr;
                            return;
                        when 5 =>
                            Result := N_Bare_For_Loop_Spec.For_Loop_Spec_F_Iter_Filter;
                            return;
                        when others => null;
                    end case;
end;
when Ada_While_Loop_Spec_Range =>
declare
N_Bare_While_Loop_Spec : constant Bare_While_Loop_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_While_Loop_Spec.While_Loop_Spec_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Multi_Abstract_State_Decl_Range =>
declare
N_Bare_Multi_Abstract_State_Decl : constant Bare_Multi_Abstract_State_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Multi_Abstract_State_Decl.Multi_Abstract_State_Decl_F_Decls;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Params_Range =>
declare
N_Bare_Params : constant Bare_Params := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Params.Params_F_Params;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Paren_Abstract_State_Decl_Range =>
declare
N_Bare_Paren_Abstract_State_Decl : constant Bare_Paren_Abstract_State_Decl := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Paren_Abstract_State_Decl.Paren_Abstract_State_Decl_F_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Pp_Elsif_Directive_Range =>
declare
N_Bare_Pp_Elsif_Directive : constant Bare_Pp_Elsif_Directive := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Pp_Elsif_Directive.Pp_Elsif_Directive_F_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Pp_Elsif_Directive.Pp_Elsif_Directive_F_Then_Kw;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Pp_If_Directive_Range =>
declare
N_Bare_Pp_If_Directive : constant Bare_Pp_If_Directive := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Pp_If_Directive.Pp_If_Directive_F_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Pp_If_Directive.Pp_If_Directive_F_Then_Kw;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Pragma_Node_Range =>
declare
N_Bare_Pragma_Node : constant Bare_Pragma_Node := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Pragma_Node.Pragma_Node_F_Id;
                            return;
                        when 2 =>
                            Result := N_Bare_Pragma_Node.Pragma_Node_F_Args;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Protected_Def_Range =>
declare
N_Bare_Protected_Def : constant Bare_Protected_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Protected_Def.Protected_Def_F_Public_Part;
                            return;
                        when 2 =>
                            Result := N_Bare_Protected_Def.Protected_Def_F_Private_Part;
                            return;
                        when 3 =>
                            Result := N_Bare_Protected_Def.Protected_Def_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Range_Spec_Range =>
declare
N_Bare_Range_Spec : constant Bare_Range_Spec := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Range_Spec.Range_Spec_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Renaming_Clause_Range =>
declare
N_Bare_Renaming_Clause : constant Bare_Renaming_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Renaming_Clause.Renaming_Clause_F_Renamed_Object;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Select_When_Part_Range =>
declare
N_Bare_Select_When_Part : constant Bare_Select_When_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Select_When_Part.Select_When_Part_F_Cond_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Select_When_Part.Select_When_Part_F_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Accept_Stmt_Range =>
declare
N_Bare_Accept_Stmt : constant Bare_Accept_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Accept_Stmt.Accept_Stmt_F_Body_Decl;
                            return;
                        when 2 =>
                            Result := N_Bare_Accept_Stmt.Accept_Stmt_F_Entry_Index_Expr;
                            return;
                        when 3 =>
                            Result := N_Bare_Accept_Stmt.Accept_Stmt_F_Params;
                            return;
                        when others => null;
                    end case;
case Ada_Accept_Stmt_Range (K) is
when Ada_Accept_Stmt_With_Stmts_Range =>
declare
N_Bare_Accept_Stmt_With_Stmts : constant Bare_Accept_Stmt_With_Stmts := N_Bare_Accept_Stmt;
begin
case Index is
                        when 4 =>
                            Result := N_Bare_Accept_Stmt_With_Stmts.Accept_Stmt_With_Stmts_F_Stmts;
                            return;
                        when 5 =>
                            Result := N_Bare_Accept_Stmt_With_Stmts.Accept_Stmt_With_Stmts_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when Ada_Base_Loop_Stmt =>
declare
N_Bare_Base_Loop_Stmt : constant Bare_Base_Loop_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Base_Loop_Stmt.Base_Loop_Stmt_F_Spec;
                            return;
                        when 2 =>
                            Result := N_Bare_Base_Loop_Stmt.Base_Loop_Stmt_F_Stmts;
                            return;
                        when 3 =>
                            Result := N_Bare_Base_Loop_Stmt.Base_Loop_Stmt_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Begin_Block_Range =>
declare
N_Bare_Begin_Block : constant Bare_Begin_Block := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Begin_Block.Begin_Block_F_Stmts;
                            return;
                        when 2 =>
                            Result := N_Bare_Begin_Block.Begin_Block_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Decl_Block_Range =>
declare
N_Bare_Decl_Block : constant Bare_Decl_Block := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Decl_Block.Decl_Block_F_Decls;
                            return;
                        when 2 =>
                            Result := N_Bare_Decl_Block.Decl_Block_F_Stmts;
                            return;
                        when 3 =>
                            Result := N_Bare_Decl_Block.Decl_Block_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Case_Stmt_Range =>
declare
N_Bare_Case_Stmt : constant Bare_Case_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Case_Stmt.Case_Stmt_F_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_Case_Stmt.Case_Stmt_F_Pragmas;
                            return;
                        when 3 =>
                            Result := N_Bare_Case_Stmt.Case_Stmt_F_Alternatives;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Extended_Return_Stmt_Range =>
declare
N_Bare_Extended_Return_Stmt : constant Bare_Extended_Return_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Extended_Return_Stmt.Extended_Return_Stmt_F_Decl;
                            return;
                        when 2 =>
                            Result := N_Bare_Extended_Return_Stmt.Extended_Return_Stmt_F_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_If_Stmt_Range =>
declare
N_Bare_If_Stmt : constant Bare_If_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_If_Stmt.If_Stmt_F_Cond_Expr;
                            return;
                        when 2 =>
                            Result := N_Bare_If_Stmt.If_Stmt_F_Then_Stmts;
                            return;
                        when 3 =>
                            Result := N_Bare_If_Stmt.If_Stmt_F_Alternatives;
                            return;
                        when 4 =>
                            Result := N_Bare_If_Stmt.If_Stmt_F_Else_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Named_Stmt_Range =>
declare
N_Bare_Named_Stmt : constant Bare_Named_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Named_Stmt.Named_Stmt_F_Decl;
                            return;
                        when 2 =>
                            Result := N_Bare_Named_Stmt.Named_Stmt_F_Stmt;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Select_Stmt_Range =>
declare
N_Bare_Select_Stmt : constant Bare_Select_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Select_Stmt.Select_Stmt_F_Guards;
                            return;
                        when 2 =>
                            Result := N_Bare_Select_Stmt.Select_Stmt_F_Else_Stmts;
                            return;
                        when 3 =>
                            Result := N_Bare_Select_Stmt.Select_Stmt_F_Abort_Stmts;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Abort_Stmt_Range =>
declare
N_Bare_Abort_Stmt : constant Bare_Abort_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Abort_Stmt.Abort_Stmt_F_Names;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Assign_Stmt_Range =>
declare
N_Bare_Assign_Stmt : constant Bare_Assign_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Assign_Stmt.Assign_Stmt_F_Dest;
                            return;
                        when 2 =>
                            Result := N_Bare_Assign_Stmt.Assign_Stmt_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Call_Stmt_Range =>
declare
N_Bare_Call_Stmt : constant Bare_Call_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Call_Stmt.Call_Stmt_F_Call;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Delay_Stmt_Range =>
declare
N_Bare_Delay_Stmt : constant Bare_Delay_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Delay_Stmt.Delay_Stmt_F_Has_Until;
                            return;
                        when 2 =>
                            Result := N_Bare_Delay_Stmt.Delay_Stmt_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Exit_Stmt_Range =>
declare
N_Bare_Exit_Stmt : constant Bare_Exit_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Exit_Stmt.Exit_Stmt_F_Loop_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Exit_Stmt.Exit_Stmt_F_Cond_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Goto_Stmt_Range =>
declare
N_Bare_Goto_Stmt : constant Bare_Goto_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Goto_Stmt.Goto_Stmt_F_Label_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Label_Range =>
declare
N_Bare_Label : constant Bare_Label := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Label.Label_F_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Raise_Stmt_Range =>
declare
N_Bare_Raise_Stmt : constant Bare_Raise_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Raise_Stmt.Raise_Stmt_F_Exception_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Raise_Stmt.Raise_Stmt_F_Error_Message;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Requeue_Stmt_Range =>
declare
N_Bare_Requeue_Stmt : constant Bare_Requeue_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Requeue_Stmt.Requeue_Stmt_F_Call_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Requeue_Stmt.Requeue_Stmt_F_Has_Abort;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Return_Stmt_Range =>
declare
N_Bare_Return_Stmt : constant Bare_Return_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Return_Stmt.Return_Stmt_F_Return_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Simple_Decl_Stmt_Range =>
declare
N_Bare_Simple_Decl_Stmt : constant Bare_Simple_Decl_Stmt := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Simple_Decl_Stmt.Simple_Decl_Stmt_F_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subunit_Range =>
declare
N_Bare_Subunit : constant Bare_Subunit := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Subunit.Subunit_F_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Subunit.Subunit_F_Body;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Task_Def_Range =>
declare
N_Bare_Task_Def : constant Bare_Task_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Task_Def.Task_Def_F_Interfaces;
                            return;
                        when 2 =>
                            Result := N_Bare_Task_Def.Task_Def_F_Public_Part;
                            return;
                        when 3 =>
                            Result := N_Bare_Task_Def.Task_Def_F_Private_Part;
                            return;
                        when 4 =>
                            Result := N_Bare_Task_Def.Task_Def_F_End_Name;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Access_Def =>
declare
N_Bare_Access_Def : constant Bare_Access_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Access_Def.Access_Def_F_Has_Not_Null;
                            return;
                        when others => null;
                    end case;
case Ada_Access_Def (K) is
when Ada_Access_To_Subp_Def_Range =>
declare
N_Bare_Access_To_Subp_Def : constant Bare_Access_To_Subp_Def := N_Bare_Access_Def;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Access_To_Subp_Def.Access_To_Subp_Def_F_Has_Protected;
                            return;
                        when 3 =>
                            Result := N_Bare_Access_To_Subp_Def.Access_To_Subp_Def_F_Subp_Spec;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Anonymous_Type_Access_Def_Range =>
declare
N_Bare_Anonymous_Type_Access_Def : constant Bare_Anonymous_Type_Access_Def := N_Bare_Access_Def;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Anonymous_Type_Access_Def.Anonymous_Type_Access_Def_F_Type_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Type_Access_Def_Range =>
declare
N_Bare_Type_Access_Def : constant Bare_Type_Access_Def := N_Bare_Access_Def;
begin
case Index is
                        when 2 =>
                            Result := N_Bare_Type_Access_Def.Type_Access_Def_F_Has_All;
                            return;
                        when 3 =>
                            Result := N_Bare_Type_Access_Def.Type_Access_Def_F_Has_Constant;
                            return;
                        when 4 =>
                            Result := N_Bare_Type_Access_Def.Type_Access_Def_F_Subtype_Indication;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
end;
when Ada_Array_Type_Def_Range =>
declare
N_Bare_Array_Type_Def : constant Bare_Array_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Array_Type_Def.Array_Type_Def_F_Indices;
                            return;
                        when 2 =>
                            Result := N_Bare_Array_Type_Def.Array_Type_Def_F_Component_Type;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Derived_Type_Def_Range =>
declare
N_Bare_Derived_Type_Def : constant Bare_Derived_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Has_Abstract;
                            return;
                        when 2 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Has_Limited;
                            return;
                        when 3 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Has_Synchronized;
                            return;
                        when 4 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Subtype_Indication;
                            return;
                        when 5 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Interfaces;
                            return;
                        when 6 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Record_Extension;
                            return;
                        when 7 =>
                            Result := N_Bare_Derived_Type_Def.Derived_Type_Def_F_Has_With_Private;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Enum_Type_Def_Range =>
declare
N_Bare_Enum_Type_Def : constant Bare_Enum_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Enum_Type_Def.Enum_Type_Def_F_Enum_Literals;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Interface_Type_Def_Range =>
declare
N_Bare_Interface_Type_Def : constant Bare_Interface_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Interface_Type_Def.Interface_Type_Def_F_Interface_Kind;
                            return;
                        when 2 =>
                            Result := N_Bare_Interface_Type_Def.Interface_Type_Def_F_Interfaces;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Mod_Int_Type_Def_Range =>
declare
N_Bare_Mod_Int_Type_Def : constant Bare_Mod_Int_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Mod_Int_Type_Def.Mod_Int_Type_Def_F_Expr;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Private_Type_Def_Range =>
declare
N_Bare_Private_Type_Def : constant Bare_Private_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Private_Type_Def.Private_Type_Def_F_Has_Abstract;
                            return;
                        when 2 =>
                            Result := N_Bare_Private_Type_Def.Private_Type_Def_F_Has_Tagged;
                            return;
                        when 3 =>
                            Result := N_Bare_Private_Type_Def.Private_Type_Def_F_Has_Limited;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Decimal_Fixed_Point_Def_Range =>
declare
N_Bare_Decimal_Fixed_Point_Def : constant Bare_Decimal_Fixed_Point_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Decimal_Fixed_Point_Def.Decimal_Fixed_Point_Def_F_Delta;
                            return;
                        when 2 =>
                            Result := N_Bare_Decimal_Fixed_Point_Def.Decimal_Fixed_Point_Def_F_Digits;
                            return;
                        when 3 =>
                            Result := N_Bare_Decimal_Fixed_Point_Def.Decimal_Fixed_Point_Def_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Floating_Point_Def_Range =>
declare
N_Bare_Floating_Point_Def : constant Bare_Floating_Point_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Floating_Point_Def.Floating_Point_Def_F_Num_Digits;
                            return;
                        when 2 =>
                            Result := N_Bare_Floating_Point_Def.Floating_Point_Def_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Ordinary_Fixed_Point_Def_Range =>
declare
N_Bare_Ordinary_Fixed_Point_Def : constant Bare_Ordinary_Fixed_Point_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Ordinary_Fixed_Point_Def.Ordinary_Fixed_Point_Def_F_Delta;
                            return;
                        when 2 =>
                            Result := N_Bare_Ordinary_Fixed_Point_Def.Ordinary_Fixed_Point_Def_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Record_Type_Def_Range =>
declare
N_Bare_Record_Type_Def : constant Bare_Record_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Record_Type_Def.Record_Type_Def_F_Has_Abstract;
                            return;
                        when 2 =>
                            Result := N_Bare_Record_Type_Def.Record_Type_Def_F_Has_Tagged;
                            return;
                        when 3 =>
                            Result := N_Bare_Record_Type_Def.Record_Type_Def_F_Has_Limited;
                            return;
                        when 4 =>
                            Result := N_Bare_Record_Type_Def.Record_Type_Def_F_Record_Def;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Signed_Int_Type_Def_Range =>
declare
N_Bare_Signed_Int_Type_Def : constant Bare_Signed_Int_Type_Def := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Signed_Int_Type_Def.Signed_Int_Type_Def_F_Range;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Anonymous_Type_Range =>
declare
N_Bare_Anonymous_Type : constant Bare_Anonymous_Type := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Anonymous_Type.Anonymous_Type_F_Type_Decl;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Subtype_Indication_Range =>
declare
N_Bare_Subtype_Indication : constant Bare_Subtype_Indication := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Subtype_Indication.Subtype_Indication_F_Has_Not_Null;
                            return;
                        when 2 =>
                            Result := N_Bare_Subtype_Indication.Subtype_Indication_F_Name;
                            return;
                        when 3 =>
                            Result := N_Bare_Subtype_Indication.Subtype_Indication_F_Constraint;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Synthetic_Type_Expr_Range =>
declare
N_Bare_Synthetic_Type_Expr : constant Bare_Synthetic_Type_Expr := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Synthetic_Type_Expr.Synthetic_Type_Expr_F_Target_Type;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Unconstrained_Array_Index_Range =>
declare
N_Bare_Unconstrained_Array_Index : constant Bare_Unconstrained_Array_Index := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Unconstrained_Array_Index.Unconstrained_Array_Index_F_Subtype_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Unconstrained_Array_Index.Unconstrained_Array_Index_F_Lower_Bound;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Use_Package_Clause_Range =>
declare
N_Bare_Use_Package_Clause : constant Bare_Use_Package_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Use_Package_Clause.Use_Package_Clause_F_Packages;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Use_Type_Clause_Range =>
declare
N_Bare_Use_Type_Clause : constant Bare_Use_Type_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Use_Type_Clause.Use_Type_Clause_F_Has_All;
                            return;
                        when 2 =>
                            Result := N_Bare_Use_Type_Clause.Use_Type_Clause_F_Types;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Value_Sequence_Range =>
declare
N_Bare_Value_Sequence : constant Bare_Value_Sequence := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Value_Sequence.Value_Sequence_F_Iter_Assoc;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Variant_Range =>
declare
N_Bare_Variant : constant Bare_Variant := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Variant.Variant_F_Choices;
                            return;
                        when 2 =>
                            Result := N_Bare_Variant.Variant_F_Components;
                            return;
                        when others => null;
                    end case;
end;
when Ada_Variant_Part_Range =>
declare
N_Bare_Variant_Part : constant Bare_Variant_Part := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_Variant_Part.Variant_Part_F_Discr_Name;
                            return;
                        when 2 =>
                            Result := N_Bare_Variant_Part.Variant_Part_F_Variant;
                            return;
                        when others => null;
                    end case;
end;
when Ada_With_Clause_Range =>
declare
N_Bare_With_Clause : constant Bare_With_Clause := Node;
begin
case Index is
                        when 1 =>
                            Result := N_Bare_With_Clause.With_Clause_F_Has_Limited;
                            return;
                        when 2 =>
                            Result := N_Bare_With_Clause.With_Clause_F_Has_Private;
                            return;
                        when 3 =>
                            Result := N_Bare_With_Clause.With_Clause_F_Packages;
                            return;
                        when others => null;
                    end case;
end;
when others => null;
end case;
      --  Execution should reach this point iff nothing matched this index, so
      --  we must be out of bounds.
      Index_In_Bounds := False;
   end Get_Child;
   -----------
   -- Print --
   -----------
   procedure Print
     (Node        : Langkit_Support.Generic_API.Analysis.Lk_Node;
      Show_Slocs  : Boolean;
      Line_Prefix : String := "")
   is
      use Langkit_Support.Generic_API.Analysis;
      use Langkit_Support.Generic_API.Introspection;
      T : Type_Ref;
   begin
      if Node.Is_Null then
         Put_Line ("None");
         return;
      end if;
      T := Type_Of (Node);
      Put (Line_Prefix & Image (Node_Type_Repr_Name (T)));
      if Show_Slocs then
         Put ("[" & Image (Node.Sloc_Range) & "]");
      end if;
      if Node.Is_Incomplete then
         Put (" <<INCOMPLETE>>");
      end if;
      if Node.Is_Token_Node then
         Put_Line (": " & Image (Node.Text));
      elsif Is_List_Node (Node) then
         --  List nodes are displayed in a special way (they have no field)
         declare
            Count : constant Natural := Node.Children_Count;
            Child : Lk_Node;
         begin
            if Count = 0 then
               Put_Line (": <empty list>");
               return;
            end if;
            New_Line;
            for I in 1 .. Count loop
               Child := Node.Child (I);
               if not Child.Is_Null then
                  Print (Child, Show_Slocs, Line_Prefix & "|  ");
               end if;
            end loop;
         end;
      else
         --  This is for regular nodes: display each syntax field (i.e.
         --  non-property member).
         declare
            Attr_Prefix     : constant String := Line_Prefix & "|";
            Children_Prefix : constant String := Line_Prefix & "|  ";
            M_List          : constant Struct_Member_Ref_Array := Members (T);
            Child           : Lk_Node;
         begin
            New_Line;
            for M of M_List loop
               if not Is_Property (M) and then not Is_Null_For (M, T) then
                  Child := As_Node (Eval_Node_Member (Node, M));
                  Put (Attr_Prefix
                       & Image (Format_Name (Member_Name (M), Lower)) & ":");
                  if Child.Is_Null then
                     Put_Line (" <null>");
                  else
                     New_Line;
                     Print (Child, Show_Slocs, Children_Prefix);
                  end if;
               end if;
            end loop;
         end;
      end if;
   end Print;
   -----------
   -- Print --
   -----------
   procedure Print
     (Node        : Bare_Ada_Node;
      Show_Slocs  : Boolean;
      Line_Prefix : String := "")
   is
      Entity : constant Internal_Entity := (Node, No_Entity_Info);
   begin
      Print (To_Generic_Node (Entity), Show_Slocs, Line_Prefix);
   end Print;
   ------------
   -- Parent --
   ------------
   function Parent (Node : Bare_Ada_Node) return Bare_Ada_Node is
   begin
      return Node.Parent;
   end Parent;
   ------------------
   -- Stored_Token --
   ------------------
   function Stored_Token
     (Node  : Bare_Ada_Node;
      Token : Token_Reference) return Token_Index
   is
      Index : constant Token_Or_Trivia_Index := Get_Token_Index (Token);
   begin
      if Node.Unit.TDH'Access /= Get_Token_TDH (Token) then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "Cannot associate a token and a node from different analysis"
            & " units");
      elsif Index.Trivia /= No_Token_Index then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "A node cannot hold trivia");
      end if;
      return Index.Token;
   end Stored_Token;
   -------------------------
   -- Children_And_Trivia --
   -------------------------
   function Children_And_Trivia
     (Node : Bare_Ada_Node) return Bare_Children_Vector
   is
      Ret_Vec : Bare_Children_Vector;
      Ctx     : Internal_Context renames Node.Unit.Context;
      TDH     : Token_Data_Handler renames Node.Unit.TDH;
      procedure Append_Trivias (First, Last : Token_Index);
      --  Append all the trivias of tokens between indices First and Last to
      --  the returned vector.
      function Filter_Children
        (Parent : Bare_Ada_Node)
         return Internal_Bare_Ada_Node_Array;
      --  Return an array for all children in Parent that are not null
      --------------------
      -- Append_Trivias --
      --------------------
      procedure Append_Trivias (First, Last : Token_Index) is
      begin
         for I in First .. Last loop
            for D of Get_Trivias (TDH, I) loop
               Ret_Vec.Append
                 (Bare_Child_Record'
                    (Kind   => Trivia,
                     Trivia => Wrap_Token_Reference
                                 (Ctx, TDH'Access, (I, D))));
            end loop;
         end loop;
      end Append_Trivias;
      ---------------------
      -- Filter_Children --
      ---------------------
      function Filter_Children
        (Parent : Bare_Ada_Node)
         return Internal_Bare_Ada_Node_Array
      is
         Children : constant Internal_Bare_Ada_Node_Array :=
            Implementation.Children (Parent);
         Result   : Internal_Bare_Ada_Node_Array (Children'Range);
         Next     : Integer := Result'First;
      begin
         for I in Children'Range loop
            if Children (I) /= null then
               Result (Next) := Children (I);
               Next := Next + 1;
            end if;
         end loop;
         return Result (Result'First .. Next - 1);
      end Filter_Children;
      First_Child : constant Positive := 1;
      N_Children  : constant Internal_Bare_Ada_Node_Array :=
         Filter_Children (Node);
   begin
      if N_Children'Length > 0
        and then (Node.Token_Start_Index
                    /= N_Children (First_Child).Token_Start_Index)
      then
         Append_Trivias (Node.Token_Start_Index,
                         N_Children (First_Child).Token_Start_Index - 1);
      end if;
      --  Append each node to Ret_Vec, and append trivia that follow after each
      --  non-ghost nodes.
      for I in N_Children'Range loop
         Ret_Vec.Append (Bare_Child_Record'(Child, N_Children (I)));
         if not Is_Ghost (N_Children (I)) then
            Append_Trivias (N_Children (I).Token_End_Index,
                            (if I = N_Children'Last
                             then Node.Token_End_Index - 1
                             else N_Children (I + 1).Token_Start_Index - 1));
         end if;
      end loop;
      return Ret_Vec;
   end Children_And_Trivia;
   --------------
   -- Is_Ghost --
   --------------
   function Is_Ghost (Node : Bare_Ada_Node) return Boolean
   is (Node.Token_End_Index = No_Token_Index);
   -------------------
   -- Is_Incomplete --
   -------------------
   function Is_Incomplete (Node : Bare_Ada_Node) return Boolean
   is
      LGC : Bare_Ada_Node;
   begin
     if Is_List_Node (Node.Kind) then
        LGC := (if Last_Child_Index (Node) /= 0
                then Child (Node, Last_Child_Index (Node))
                else null);
        return LGC /= null and then Is_Incomplete (LGC);
      else
         return Node.Last_Attempted_Child > -1;
      end if;
   end;
   -----------------
   -- Token_Start --
   -----------------
   function Token_Start (Node : Bare_Ada_Node) return Token_Reference
   is (Token (Node, Node.Token_Start_Index));
   ---------------
   -- Token_End --
   ---------------
   function Token_End (Node : Bare_Ada_Node) return Token_Reference
   is
     (if Node.Token_End_Index = No_Token_Index
      then Token_Start (Node)
      else Token (Node, Node.Token_End_Index));
   -----------
   -- Token --
   -----------
   function Token
     (Node  : Bare_Ada_Node;
      Index : Token_Index) return Token_Reference
   is
      Unit    : constant Internal_Unit := Node.Unit;
      Context : constant Internal_Context := Unit.Context;
   begin
      return Wrap_Token_Reference
        (Context, Token_Data (Unit), (Index, No_Token_Index));
   end Token;
   ---------
   -- "<" --
   ---------
   function "<" (Left, Right : Bare_Ada_Node) return Boolean is
   begin
      --  Reject invalid inputs
      if Left /= null and Is_Synthetic (Left) then
         raise Property_Error with "left node is synthetic";
      elsif Right /= null and Is_Synthetic (Right) then
         raise Property_Error with "right node is synthetic";
      end if;
      --  Null nodes come first
      if Left = null then
         return Right /= null;
      elsif Right = null then
         return False;
      end if;
      --  So we have two non-null nodes. Sort by unit filename
      if Left.Unit < Right.Unit then
         return True;
      elsif Left.Unit /= Right.Unit then
         return False;
      end if;
      --  Both nodes come from the same unit: compare their token indexes
      if Left.Token_Start_Index < Right.Token_Start_Index then
         return True;
      elsif Left.Token_Start_Index > Right.Token_Start_Index then
         return False;
      else
         return Left.Token_End_Index < Right.Token_End_Index;
      end if;
   end "<";
   -------------
   -- Is_Null --
   -------------
   function Is_Null (Node : Bare_Ada_Node) return Boolean
   is (Node = null);
   ----------
   -- Kind --
   ----------
   function Kind (Node : Bare_Ada_Node) return Ada_Node_Kind_Type
   is (Node.Kind);
   -----------------
   -- Child_Index --
   -----------------
   function Child_Index (Node : Bare_Ada_Node) return Integer
   is
      N : Bare_Ada_Node := null;
   begin
      if Node.Parent = null then
         raise Property_Error with
            "Trying to get the child index of a root node";
      end if;
      for I in First_Child_Index (Node.Parent)
            .. Last_Child_Index (Node.Parent)
      loop
         N := Child (Node.Parent, I);
         if N = Node then
            return I - 1;
         end if;
      end loop;
      --  If we reach this point, then Node isn't a Child of Node.Parent. This
      --  is not supposed to happen.
      raise Program_Error;
   end Child_Index;
   -------------------
   -- Fetch_Sibling --
   -------------------
   function Fetch_Sibling
     (Node   : Bare_Ada_Node;
      Offset : Integer) return Bare_Ada_Node is
   begin
      --  Root nodes have no sibling: handle them now to avoid invalid requests
      --  in the code below.
      if Node.Parent = null then
         return null;
      end if;
      declare
         Node_Index : constant Positive := Child_Index (Node) + 1;
         --  Child_Index is 0-based, but the Child primitive expects a 1-based
         --  index.
         Sibling_Index : constant Integer := Node_Index + Offset;
      begin
         --  Child returns null for out-of-bound indexes
         return (if Sibling_Index >= 1
                 then Child (Node.Parent, Sibling_Index)
                 else null);
      end;
   end Fetch_Sibling;
   -------------------
   -- Fetch_Sibling --
   -------------------
   function Fetch_Sibling
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info;
      Offset : Integer) return Internal_Entity
   is
      Sibling : constant Bare_Ada_Node := Fetch_Sibling (Node, Offset);
   begin
      --  Don't forget to clear entity info if the result is nul
      return (if Sibling = null
              then No_Entity
              else (Sibling, E_Info));
   end Fetch_Sibling;
   ----------------------
   -- Previous_Sibling --
   ----------------------
   function Previous_Sibling
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity is
   begin
      return Fetch_Sibling (Node, E_Info, -1);
   end Previous_Sibling;
   ------------------
   -- Next_Sibling --
   ------------------
   function Next_Sibling
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity is
   begin
      return Fetch_Sibling (Node, E_Info, 1);
   end Next_Sibling;
   -------------
   -- Inc_Ref --
   -------------
   procedure Inc_Ref (Self : Node_Builder_Type) is
   begin
      if Self.Ref_Count > 0 then
         Self.Ref_Count := Self.Ref_Count + 1;
      end if;
   end Inc_Ref;
   -------------
   -- Dec_Ref --
   -------------
   procedure Dec_Ref (Self : in out Node_Builder_Type) is
   begin
      if Self = null or else Self.Ref_Count < 0 then
         return;
      elsif Self.Ref_Count = 1 then
         Self.Release;
         Free (Self);
      else
         Self.Ref_Count := Self.Ref_Count - 1;
      end if;
   end Dec_Ref;
   ------------------------------
   -- Create_Copy_Node_Builder --
   ------------------------------
   function Create_Copy_Node_Builder
     (Value : Bare_Ada_Node) return Node_Builder_Type is
   begin
      --  No need to allocate a new builder if in practice it cannot be
      --  distinguished from the "null" builder.
      if Value = null then
         return Null_Node_Builder;
      else
         return new Copy_Node_Builder_Record'(Ref_Count => 1, Value => Value);
      end if;
   end Create_Copy_Node_Builder;
         type Bare_Synthetic_Unary_Spec_Node_Builder_Record is new Node_Builder_Record with
            record
               Synthetic_Unary_Spec_Subp_Symbol : Symbol_Type;
               Synthetic_Unary_Spec_F_Right_Param : Bare_Synthetic_Formal_Param_Decl_Node_Builder;
               Synthetic_Unary_Spec_F_Return_Type_Expr : Bare_Synthetic_Type_Expr_Node_Builder;
            end record;
         type Bare_Synthetic_Unary_Spec_Node_Builder_Access is access all Bare_Synthetic_Unary_Spec_Node_Builder_Record;
         overriding function Build
           (Self              : Bare_Synthetic_Unary_Spec_Node_Builder_Record;
            Parent, Self_Node : Bare_Ada_Node)
            return Bare_Ada_Node;
            overriding procedure Release (Self : in out Bare_Synthetic_Unary_Spec_Node_Builder_Record);
         -----------
         -- Build --
         -----------
         overriding function Build
           (Self              : Bare_Synthetic_Unary_Spec_Node_Builder_Record;
            Parent, Self_Node : Bare_Ada_Node)
            return Bare_Ada_Node
         is
            Result : Bare_Ada_Node;
            Unit   : constant Internal_Unit := Self_Node.Unit;
            Env    : constant Lexical_Env :=
              (if Parent = null
               then Empty_Env
               else Parent.Self_Env);
         begin
            if Parent /= null and then Parent.Unit /= Unit then
               Raise_Property_Exception
                 (Self_Node,
                  Property_Error'Identity,
                  "synthetic node parents must belong to the same unit as the"
                  & " nodes that trigger node synthetization");
            end if;
            if Is_Foreign_Strict (Env, Parent) then
               Raise_Property_Exception
                 (Self_Node,
                  Property_Error'Identity,
                  "synthetic nodes cannot have foreign lexical envs");
            end if;
            Result := new Root_Node_Record
              (Ada_Synthetic_Unary_Spec);
            Initialize
              (Self => Result,
               Kind => Ada_Synthetic_Unary_Spec,
               Unit => Unit,
               Token_Start_Index => No_Token_Index,
               Token_End_Index   => No_Token_Index,
               Parent   => Parent,
               Self_Env => Env);
            Register_Destroyable (Unit, Result);
               declare
                  Children : array (1 .. 2)
                               of Bare_Ada_Node;
               begin
                     Children (1) :=
                       Self.Synthetic_Unary_Spec_F_Right_Param.Build (Result, Self_Node);
                        if Children (1) = null then
                           Raise_Property_Exception
                             (Self_Node,
                              Property_Error'Identity,
                              "SyntheticUnarySpec.right_param cannot be null in synthetic"
                              & " nodes; add a nullable annotation to this"
                              & " field to allow it");
                        end if;
                     Children (2) :=
                       Self.Synthetic_Unary_Spec_F_Return_Type_Expr.Build (Result, Self_Node);
                        if Children (2) = null then
                           Raise_Property_Exception
                             (Self_Node,
                              Property_Error'Identity,
                              "SyntheticUnarySpec.return_type_expr cannot be null in synthetic"
                              & " nodes; add a nullable annotation to this"
                              & " field to allow it");
                        end if;
                  Initialize_Fields_For_Synthetic_Unary_Spec
                    (Self => Result,
                     Synthetic_Unary_Spec_F_Right_Param => Children (1),
                     Synthetic_Unary_Spec_F_Return_Type_Expr => Children (2));
               end;
               Result.Synthetic_Unary_Spec_Subp_Symbol :=
                  Self.Synthetic_Unary_Spec_Subp_Symbol;
            return Result;
         end Build;
            -------------
            -- Release --
            -------------
            overriding procedure Release (Self : in out Bare_Synthetic_Unary_Spec_Node_Builder_Record) is
            begin
                     Dec_Ref (Self.Synthetic_Unary_Spec_F_Right_Param);
                     Dec_Ref (Self.Synthetic_Unary_Spec_F_Return_Type_Expr);
            end Release;
         function Create_Bare_Synthetic_Unary_Spec_Node_Builder
             (Synthetic_Unary_Spec_Subp_Symbol : Symbol_Type;
               Synthetic_Unary_Spec_F_Right_Param : Bare_Synthetic_Formal_Param_Decl_Node_Builder;
               Synthetic_Unary_Spec_F_Return_Type_Expr : Bare_Synthetic_Type_Expr_Node_Builder)
           return Bare_Synthetic_Unary_Spec_Node_Builder
         is
            Builder : constant Bare_Synthetic_Unary_Spec_Node_Builder_Access := new Bare_Synthetic_Unary_Spec_Node_Builder_Record;
         begin
            Builder.Ref_Count := 1;
               Builder.Synthetic_Unary_Spec_Subp_Symbol := Synthetic_Unary_Spec_Subp_Symbol;
               Builder.Synthetic_Unary_Spec_F_Right_Param := Synthetic_Unary_Spec_F_Right_Param;
                  Inc_Ref (Builder.Synthetic_Unary_Spec_F_Right_Param);
               Builder.Synthetic_Unary_Spec_F_Return_Type_Expr := Synthetic_Unary_Spec_F_Return_Type_Expr;
                  Inc_Ref (Builder.Synthetic_Unary_Spec_F_Return_Type_Expr);
            return Node_Builder_Type (Builder);
         end Create_Bare_Synthetic_Unary_Spec_Node_Builder;
         type Bare_Synthetic_Binary_Spec_Node_Builder_Record is new Node_Builder_Record with
            record
               Synthetic_Binary_Spec_Subp_Symbol : Symbol_Type;
               Synthetic_Binary_Spec_F_Left_Param : Bare_Synthetic_Formal_Param_Decl_Node_Builder;
               Synthetic_Binary_Spec_F_Right_Param : Bare_Synthetic_Formal_Param_Decl_Node_Builder;
               Synthetic_Binary_Spec_F_Return_Type_Expr : Bare_Type_Expr_Node_Builder;
            end record;
         type Bare_Synthetic_Binary_Spec_Node_Builder_Access is access all Bare_Synthetic_Binary_Spec_Node_Builder_Record;
         overriding function Build
           (Self              : Bare_Synthetic_Binary_Spec_Node_Builder_Record;
            Parent, Self_Node : Bare_Ada_Node)
            return Bare_Ada_Node;
            overriding procedure Release (Self : in out Bare_Synthetic_Binary_Spec_Node_Builder_Record);
         -----------
         -- Build --
         -----------
         overriding function Build
           (Self              : Bare_Synthetic_Binary_Spec_Node_Builder_Record;
            Parent, Self_Node : Bare_Ada_Node)
            return Bare_Ada_Node
         is
            Result : Bare_Ada_Node;
            Unit   : constant Internal_Unit := Self_Node.Unit;
            Env    : constant Lexical_Env :=
              (if Parent = null
               then Empty_Env
               else Parent.Self_Env);
         begin
            if Parent /= null and then Parent.Unit /= Unit then
               Raise_Property_Exception
                 (Self_Node,
                  Property_Error'Identity,
                  "synthetic node parents must belong to the same unit as the"
                  & " nodes that trigger node synthetization");
            end if;
            if Is_Foreign_Strict (Env, Parent) then
               Raise_Property_Exception
                 (Self_Node,
                  Property_Error'Identity,
                  "synthetic nodes cannot have foreign lexical envs");
            end if;
            Result := new Root_Node_Record
              (Ada_Synthetic_Binary_Spec);
            Initialize
              (Self => Result,
               Kind => Ada_Synthetic_Binary_Spec,
               Unit => Unit,
               Token_Start_Index => No_Token_Index,
               Token_End_Index   => No_Token_Index,
               Parent   => Parent,
               Self_Env => Env);
            Register_Destroyable (Unit, Result);
               declare
                  Children : array (1 .. 3)
                               of Bare_Ada_Node;
               begin
                     Children (1) :=
                       Self.Synthetic_Binary_Spec_F_Left_Param.Build (Result, Self_Node);
                        if Children (1) = null then
                           Raise_Property_Exception
                             (Self_Node,
                              Property_Error'Identity,
                              "SyntheticBinarySpec.left_param cannot be null in synthetic"
                              & " nodes; add a nullable annotation to this"
                              & " field to allow it");
                        end if;
                     Children (2) :=
                       Self.Synthetic_Binary_Spec_F_Right_Param.Build (Result, Self_Node);
                        if Children (2) = null then
                           Raise_Property_Exception
                             (Self_Node,
                              Property_Error'Identity,
                              "SyntheticBinarySpec.right_param cannot be null in synthetic"
                              & " nodes; add a nullable annotation to this"
                              & " field to allow it");
                        end if;
                     Children (3) :=
                       Self.Synthetic_Binary_Spec_F_Return_Type_Expr.Build (Result, Self_Node);
                  Initialize_Fields_For_Synthetic_Binary_Spec
                    (Self => Result,
                     Synthetic_Binary_Spec_F_Left_Param => Children (1),
                     Synthetic_Binary_Spec_F_Right_Param => Children (2),
                     Synthetic_Binary_Spec_F_Return_Type_Expr => Children (3));
               end;
               Result.Synthetic_Binary_Spec_Subp_Symbol :=
                  Self.Synthetic_Binary_Spec_Subp_Symbol;
            return Result;
         end Build;
            -------------
            -- Release --
            -------------
            overriding procedure Release (Self : in out Bare_Synthetic_Binary_Spec_Node_Builder_Record) is
            begin
                     Dec_Ref (Self.Synthetic_Binary_Spec_F_Left_Param);
                     Dec_Ref (Self.Synthetic_Binary_Spec_F_Right_Param);
                     Dec_Ref (Self.Synthetic_Binary_Spec_F_Return_Type_Expr);
            end Release;
         function Create_Bare_Synthetic_Binary_Spec_Node_Builder
             (Synthetic_Binary_Spec_Subp_Symbol : Symbol_Type;
               Synthetic_Binary_Spec_F_Left_Param : Bare_Synthetic_Formal_Param_Decl_Node_Builder;
               Synthetic_Binary_Spec_F_Right_Param : Bare_Synthetic_Formal_Param_Decl_Node_Builder;
               Synthetic_Binary_Spec_F_Return_Type_Expr : Bare_Type_Expr_Node_Builder)
           return Bare_Synthetic_Binary_Spec_Node_Builder
         is
            Builder : constant Bare_Synthetic_Binary_Spec_Node_Builder_Access := new Bare_Synthetic_Binary_Spec_Node_Builder_Record;
         begin
            Builder.Ref_Count := 1;
               Builder.Synthetic_Binary_Spec_Subp_Symbol := Synthetic_Binary_Spec_Subp_Symbol;
               Builder.Synthetic_Binary_Spec_F_Left_Param := Synthetic_Binary_Spec_F_Left_Param;
                  Inc_Ref (Builder.Synthetic_Binary_Spec_F_Left_Param);
               Builder.Synthetic_Binary_Spec_F_Right_Param := Synthetic_Binary_Spec_F_Right_Param;
                  Inc_Ref (Builder.Synthetic_Binary_Spec_F_Right_Param);
               Builder.Synthetic_Binary_Spec_F_Return_Type_Expr := Synthetic_Binary_Spec_F_Return_Type_Expr;
                  Inc_Ref (Builder.Synthetic_Binary_Spec_F_Return_Type_Expr);
            return Node_Builder_Type (Builder);
         end Create_Bare_Synthetic_Binary_Spec_Node_Builder;
         type Bare_Synthetic_Type_Expr_Node_Builder_Record is new Node_Builder_Record with
            record
               Synthetic_Type_Expr_F_Target_Type : Bare_Base_Type_Decl_Node_Builder;
            end record;
         type Bare_Synthetic_Type_Expr_Node_Builder_Access is access all Bare_Synthetic_Type_Expr_Node_Builder_Record;
         overriding function Build
           (Self              : Bare_Synthetic_Type_Expr_Node_Builder_Record;
            Parent, Self_Node : Bare_Ada_Node)
            return Bare_Ada_Node;
            overriding procedure Release (Self : in out Bare_Synthetic_Type_Expr_Node_Builder_Record);
         -----------
         -- Build --
         -----------
         overriding function Build
           (Self              : Bare_Synthetic_Type_Expr_Node_Builder_Record;
            Parent, Self_Node : Bare_Ada_Node)
            return Bare_Ada_Node
         is
            Result : Bare_Ada_Node;
            Unit   : constant Internal_Unit := Self_Node.Unit;
            Env    : constant Lexical_Env :=
              (if Parent = null
               then Empty_Env
               else Parent.Self_Env);
         begin
            if Parent /= null and then Parent.Unit /= Unit then
               Raise_Property_Exception
                 (Self_Node,
                  Property_Error'Identity,
                  "synthetic node parents must belong to the same unit as the"
                  & " nodes that trigger node synthetization");
            end if;
            if Is_Foreign_Strict (Env, Parent) then
               Raise_Property_Exception
                 (Self_Node,
                  Property_Error'Identity,
                  "synthetic nodes cannot have foreign lexical envs");
            end if;
            Result := new Root_Node_Record
              (Ada_Synthetic_Type_Expr);
            Initialize
              (Self => Result,
               Kind => Ada_Synthetic_Type_Expr,
               Unit => Unit,
               Token_Start_Index => No_Token_Index,
               Token_End_Index   => No_Token_Index,
               Parent   => Parent,
               Self_Env => Env);
            Register_Destroyable (Unit, Result);
               declare
                  Children : array (1 .. 1)
                               of Bare_Ada_Node;
               begin
                     Children (1) :=
                       Self.Synthetic_Type_Expr_F_Target_Type.Build (Result, Self_Node);
                        if Children (1) = null then
                           Raise_Property_Exception
                             (Self_Node,
                              Property_Error'Identity,
                              "SyntheticTypeExpr.target_type cannot be null in synthetic"
                              & " nodes; add a nullable annotation to this"
                              & " field to allow it");
                        end if;
                  Initialize_Fields_For_Synthetic_Type_Expr
                    (Self => Result,
                     Synthetic_Type_Expr_F_Target_Type => Children (1));
               end;
            return Result;
         end Build;
            -------------
            -- Release --
            -------------
            overriding procedure Release (Self : in out Bare_Synthetic_Type_Expr_Node_Builder_Record) is
            begin
                     Dec_Ref (Self.Synthetic_Type_Expr_F_Target_Type);
            end Release;
         function Create_Bare_Synthetic_Type_Expr_Node_Builder
             (Synthetic_Type_Expr_F_Target_Type : Bare_Base_Type_Decl_Node_Builder)
           return Bare_Synthetic_Type_Expr_Node_Builder
         is
            Builder : constant Bare_Synthetic_Type_Expr_Node_Builder_Access := new Bare_Synthetic_Type_Expr_Node_Builder_Record;
         begin
            Builder.Ref_Count := 1;
               Builder.Synthetic_Type_Expr_F_Target_Type := Synthetic_Type_Expr_F_Target_Type;
                  Inc_Ref (Builder.Synthetic_Type_Expr_F_Target_Type);
            return Node_Builder_Type (Builder);
         end Create_Bare_Synthetic_Type_Expr_Node_Builder;
         type Bare_Synthetic_Formal_Param_Decl_Node_Builder_Record is new Node_Builder_Record with
            record
               Synthetic_Formal_Param_Decl_Param_Name : Symbol_Type;
               Synthetic_Formal_Param_Decl_F_Param_Type : Bare_Type_Expr_Node_Builder;
            end record;
         type Bare_Synthetic_Formal_Param_Decl_Node_Builder_Access is access all Bare_Synthetic_Formal_Param_Decl_Node_Builder_Record;
         overriding function Build
           (Self              : Bare_Synthetic_Formal_Param_Decl_Node_Builder_Record;
            Parent, Self_Node : Bare_Ada_Node)
            return Bare_Ada_Node;
            overriding procedure Release (Self : in out Bare_Synthetic_Formal_Param_Decl_Node_Builder_Record);
         -----------
         -- Build --
         -----------
         overriding function Build
           (Self              : Bare_Synthetic_Formal_Param_Decl_Node_Builder_Record;
            Parent, Self_Node : Bare_Ada_Node)
            return Bare_Ada_Node
         is
            Result : Bare_Ada_Node;
            Unit   : constant Internal_Unit := Self_Node.Unit;
            Env    : constant Lexical_Env :=
              (if Parent = null
               then Empty_Env
               else Parent.Self_Env);
         begin
            if Parent /= null and then Parent.Unit /= Unit then
               Raise_Property_Exception
                 (Self_Node,
                  Property_Error'Identity,
                  "synthetic node parents must belong to the same unit as the"
                  & " nodes that trigger node synthetization");
            end if;
            if Is_Foreign_Strict (Env, Parent) then
               Raise_Property_Exception
                 (Self_Node,
                  Property_Error'Identity,
                  "synthetic nodes cannot have foreign lexical envs");
            end if;
            Result := new Root_Node_Record
              (Ada_Synthetic_Formal_Param_Decl);
            Initialize
              (Self => Result,
               Kind => Ada_Synthetic_Formal_Param_Decl,
               Unit => Unit,
               Token_Start_Index => No_Token_Index,
               Token_End_Index   => No_Token_Index,
               Parent   => Parent,
               Self_Env => Env);
            Register_Destroyable (Unit, Result);
               declare
                  Children : array (1 .. 1)
                               of Bare_Ada_Node;
               begin
                     Children (1) :=
                       Self.Synthetic_Formal_Param_Decl_F_Param_Type.Build (Result, Self_Node);
                        if Children (1) = null then
                           Raise_Property_Exception
                             (Self_Node,
                              Property_Error'Identity,
                              "SyntheticFormalParamDecl.param_type cannot be null in synthetic"
                              & " nodes; add a nullable annotation to this"
                              & " field to allow it");
                        end if;
                  Initialize_Fields_For_Synthetic_Formal_Param_Decl
                    (Self => Result,
                     Synthetic_Formal_Param_Decl_F_Param_Type => Children (1));
               end;
               Result.Synthetic_Formal_Param_Decl_Param_Name :=
                  Self.Synthetic_Formal_Param_Decl_Param_Name;
            return Result;
         end Build;
            -------------
            -- Release --
            -------------
            overriding procedure Release (Self : in out Bare_Synthetic_Formal_Param_Decl_Node_Builder_Record) is
            begin
                     Dec_Ref (Self.Synthetic_Formal_Param_Decl_F_Param_Type);
            end Release;
         function Create_Bare_Synthetic_Formal_Param_Decl_Node_Builder
             (Synthetic_Formal_Param_Decl_Param_Name : Symbol_Type;
               Synthetic_Formal_Param_Decl_F_Param_Type : Bare_Type_Expr_Node_Builder)
           return Bare_Synthetic_Formal_Param_Decl_Node_Builder
         is
            Builder : constant Bare_Synthetic_Formal_Param_Decl_Node_Builder_Access := new Bare_Synthetic_Formal_Param_Decl_Node_Builder_Record;
         begin
            Builder.Ref_Count := 1;
               Builder.Synthetic_Formal_Param_Decl_Param_Name := Synthetic_Formal_Param_Decl_Param_Name;
               Builder.Synthetic_Formal_Param_Decl_F_Param_Type := Synthetic_Formal_Param_Decl_F_Param_Type;
                  Inc_Ref (Builder.Synthetic_Formal_Param_Decl_F_Param_Type);
            return Node_Builder_Type (Builder);
         end Create_Bare_Synthetic_Formal_Param_Decl_Node_Builder;
         type Bare_Synthetic_Subp_Decl_Node_Builder_Record is new Node_Builder_Record with
            record
               Synthetic_Subp_Decl_F_Spec : Bare_Base_Subp_Spec_Node_Builder;
            end record;
         type Bare_Synthetic_Subp_Decl_Node_Builder_Access is access all Bare_Synthetic_Subp_Decl_Node_Builder_Record;
         overriding function Build
           (Self              : Bare_Synthetic_Subp_Decl_Node_Builder_Record;
            Parent, Self_Node : Bare_Ada_Node)
            return Bare_Ada_Node;
            overriding procedure Release (Self : in out Bare_Synthetic_Subp_Decl_Node_Builder_Record);
         -----------
         -- Build --
         -----------
         overriding function Build
           (Self              : Bare_Synthetic_Subp_Decl_Node_Builder_Record;
            Parent, Self_Node : Bare_Ada_Node)
            return Bare_Ada_Node
         is
            Result : Bare_Ada_Node;
            Unit   : constant Internal_Unit := Self_Node.Unit;
            Env    : constant Lexical_Env :=
              (if Parent = null
               then Empty_Env
               else Parent.Self_Env);
         begin
            if Parent /= null and then Parent.Unit /= Unit then
               Raise_Property_Exception
                 (Self_Node,
                  Property_Error'Identity,
                  "synthetic node parents must belong to the same unit as the"
                  & " nodes that trigger node synthetization");
            end if;
            if Is_Foreign_Strict (Env, Parent) then
               Raise_Property_Exception
                 (Self_Node,
                  Property_Error'Identity,
                  "synthetic nodes cannot have foreign lexical envs");
            end if;
            Result := new Root_Node_Record
              (Ada_Synthetic_Subp_Decl);
            Initialize
              (Self => Result,
               Kind => Ada_Synthetic_Subp_Decl,
               Unit => Unit,
               Token_Start_Index => No_Token_Index,
               Token_End_Index   => No_Token_Index,
               Parent   => Parent,
               Self_Env => Env);
            Register_Destroyable (Unit, Result);
               declare
                  Children : array (1 .. 1)
                               of Bare_Ada_Node;
               begin
                     Children (1) :=
                       Self.Synthetic_Subp_Decl_F_Spec.Build (Result, Self_Node);
                        if Children (1) = null then
                           Raise_Property_Exception
                             (Self_Node,
                              Property_Error'Identity,
                              "SyntheticSubpDecl.spec cannot be null in synthetic"
                              & " nodes; add a nullable annotation to this"
                              & " field to allow it");
                        end if;
                  Initialize_Fields_For_Synthetic_Subp_Decl
                    (Self => Result,
                     Synthetic_Subp_Decl_F_Spec => Children (1));
               end;
            return Result;
         end Build;
            -------------
            -- Release --
            -------------
            overriding procedure Release (Self : in out Bare_Synthetic_Subp_Decl_Node_Builder_Record) is
            begin
                     Dec_Ref (Self.Synthetic_Subp_Decl_F_Spec);
            end Release;
         function Create_Bare_Synthetic_Subp_Decl_Node_Builder
             (Synthetic_Subp_Decl_F_Spec : Bare_Base_Subp_Spec_Node_Builder)
           return Bare_Synthetic_Subp_Decl_Node_Builder
         is
            Builder : constant Bare_Synthetic_Subp_Decl_Node_Builder_Access := new Bare_Synthetic_Subp_Decl_Node_Builder_Record;
         begin
            Builder.Ref_Count := 1;
               Builder.Synthetic_Subp_Decl_F_Spec := Synthetic_Subp_Decl_F_Spec;
                  Inc_Ref (Builder.Synthetic_Subp_Decl_F_Spec);
            return Node_Builder_Type (Builder);
         end Create_Bare_Synthetic_Subp_Decl_Node_Builder;
   ----------------------
   -- Compare_Metadata --
   ----------------------
   --  Deactivate "not referenced" warnings because if the metadata struct has
   --  no fields, formals and temporaries won't be referenced in the two
   --  following functions.
   pragma Warnings (Off, "referenced");
   function Compare_Metadata (L, R : Internal_Metadata) return Boolean is
   begin
            if L.Primitive /= R.Primitive then
               return False;
            end if;
            if L.Primitive_Real_Type /= R.Primitive_Real_Type then
               return False;
            end if;
      return True;
   end Compare_Metadata;
   ----------
   -- Hash --
   ----------
   function Hash (Self : Internal_Metadata) return Hash_Type is
      Ret : Hash_Type := Langkit_Support.Hashes.Initial_Hash;
   begin
               Ret := Combine (Ret, Hash (Self.Primitive));
               Ret := Combine (Ret, Hash (Self.Primitive_Real_Type));
      return Ret;
   end Hash;
   pragma Warnings (On, "referenced");
   -------------
   -- Combine --
   -------------
   function Combine
     (L, R : Internal_Metadata) return Internal_Metadata
   is
      Ret : Internal_Metadata := No_Metadata;
   begin
         Ret.Dottable_Subp := L.Dottable_Subp or R.Dottable_Subp;
         if L.Primitive = No_Bare_Ada_Node then
            Ret.Primitive := R.Primitive;
         elsif R.Primitive = No_Bare_Ada_Node then
            Ret.Primitive := L.Primitive;
         else
            raise Property_Error with "Wrong combine for metadata field";
         end if;
         if L.Primitive_Real_Type = No_Bare_Ada_Node then
            Ret.Primitive_Real_Type := R.Primitive_Real_Type;
         elsif R.Primitive_Real_Type = No_Bare_Ada_Node then
            Ret.Primitive_Real_Type := L.Primitive_Real_Type;
         else
            raise Property_Error with "Wrong combine for metadata field";
         end if;
      return Ret;
   end Combine;
   -------------------------------
   -- Create_Static_Lexical_Env --
   -------------------------------
   function Create_Static_Lexical_Env
     (Parent            : Lexical_Env;
      Node              : Bare_Ada_Node;
      Sym_Table         : Symbol_Table;
      Transitive_Parent : Boolean := False) return Lexical_Env
   is
      Unit : constant Internal_Unit :=
        (if Node = null then null else Node.Unit);
   begin
      return Result : Lexical_Env := Create_Lexical_Env
        (Parent, Node, Transitive_Parent, Sym_Table, Convert_Unit (Unit))
      do
         if Unit /= null then
            Register_Destroyable (Unit, Unwrap (Result.Env));
         end if;
      end return;
   end Create_Static_Lexical_Env;
   ---------
   -- Get --
   ---------
   function Get
     (Self  : Bare_Ada_Node;
      A     : AST_Envs.Entity_Array;
      Index : Integer) return Internal_Entity
   is
      function Length (A : AST_Envs.Entity_Array) return Natural
      is (A'Length);
      function Get
        (A     : AST_Envs.Entity_Array;
         Index : Integer) return Internal_Entity
      is (A (Index + 1)); --  A is 1-based but Index is 0-based
      function Relative_Get is new Langkit_Support.Relative_Get
        (Item_Type     => Entity,
         Sequence_Type => AST_Envs.Entity_Array,
         Length        => Length,
         Get           => Get);
      Result : Internal_Entity;
   begin
      if Relative_Get (A, Index, Result) then
         return Result;
      else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "out-of-bounds array access");
      end if;
   end Get;
   -----------
   -- Group --
   -----------
   function Group
     (Envs   : Lexical_Env_Array_Access;
      Env_Md : Internal_Metadata := No_Metadata) return Lexical_Env
   is (Group (Lexical_Env_Array (Envs.Items), Env_Md));
      --
      --  Untyped wrappers for Bare_Ada_Node
      --
function Ada_Node_P_Resolve_Generic_Actual
  (E : Entity
  ) return Internal_Entity;
function Ada_Node_P_Use_Clauses_In_Spec_Of_Subp_Body
  (E : Entity
  ) return Lexical_Env;
function Ada_Node_P_Use_Clauses_In_Generic_Formal_Part
  (E : Entity
  ) return Lexical_Env;
function Ada_Node_P_Nested_Generic_Formal_Part
  (E : Entity
  ) return Lexical_Env;
      --
      --  Untyped wrappers for Bare_Base_Type_Decl
      --
function Base_Type_Decl_F_Direct_Primitive_Subps
  (E : Entity
  ) return Internal_Inner_Env_Assoc_Array_Access;
function Base_Type_Decl_P_Dottable_Subps
  (E : Entity
  ) return Internal_Inner_Env_Assoc_Array_Access;
      --
      --  Untyped wrappers for Bare_Type_Decl
      --
function Type_Decl_P_Refined_Parent_Primitives_Env
  (E : Entity
  ) return Lexical_Env;
      --
      --  Untyped wrappers for Bare_Body_Node
      --
function Body_Node_P_Subunit_Decl_Env
  (E : Entity
  ) return Lexical_Env;
function Body_Node_P_Body_Decl_Scope
  (E : Entity
  ) return Lexical_Env;
      --
      --  Untyped wrappers for Bare_Package_Body
      --
function Package_Body_P_Package_Decl_Use_Clauses_Envs
  (E : Entity
  ) return Lexical_Env;
      --
      --  Untyped wrappers for Bare_Task_Body
      --
function Task_Body_P_Task_Type_Decl_Scope
  (E : Entity
  ) return Lexical_Env;
      --
      --  Untyped wrappers for Bare_Generic_Instantiation
      --
function Generic_Instantiation_P_Instantiation_Bindings
  (E : Entity
  ) return Internal_Inner_Env_Assoc_Array_Access;
function Generic_Instantiation_P_Parent_Instantiation_Env
  (E : Entity
  ) return Lexical_Env;
      --
      --  Untyped wrappers for Bare_Name
      --
function Name_P_Name_Designated_Type_Env
  (E : Entity
  ) return Lexical_Env;
function Name_P_Use_Package_Name_Designated_Env
  (E : Entity
  ) return Lexical_Env;
   ------------------
   -- Children_Env --
   ------------------
   function Children_Env
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Lexical_Env
   is (Rebind_Env (Node.Self_Env, E_Info));
   --------------
   -- Node_Env --
   --------------
   function Node_Env
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Lexical_Env
   is
      function Get_Base_Env return Lexical_Env;
      --  Return the environment that we need to rebind before returning
      ------------------
      -- Get_Base_Env --
      ------------------
      function Get_Base_Env return Lexical_Env is
         pragma Warnings (Off, "referenced");
         function Get_Parent_Env return Lexical_Env;
         pragma Warnings (On, "referenced");
         --------------------
         -- Get_Parent_Env --
         --------------------
         function Get_Parent_Env return Lexical_Env is
            Parent : constant Lexical_Env := AST_Envs.Parent (Node.Self_Env);
         begin
            --  If Node is the root scope or the empty environment, Parent can
            --  be a wrapper around the null node. Turn this into the
            --  Empty_Env, as null envs are erroneous values in properties.
            return (if Unwrap (Parent) = null
                    then Empty_Env
                    else Parent);
         end Get_Parent_Env;
      begin
         return
           (if Node.Kind in Ada_Generic_Package_Internal | Ada_Package_Decl | Ada_Subtype_Decl | Ada_Incomplete_Type_Decl | Ada_Incomplete_Formal_Type_Decl | Ada_Incomplete_Tagged_Type_Decl | Ada_Protected_Type_Decl | Ada_Task_Type_Decl | Ada_Single_Task_Type_Decl | Ada_Concrete_Type_Decl | Ada_Formal_Type_Decl | Ada_Abstract_Subp_Decl | Ada_Abstract_Formal_Subp_Decl | Ada_Concrete_Formal_Subp_Decl | Ada_Subp_Decl | Ada_Entry_Decl | Ada_Enum_Literal_Decl | Ada_Synthetic_Char_Enum_Lit | Ada_Generic_Subp_Internal | Ada_Synthetic_Subp_Decl | Ada_Expr_Function | Ada_Null_Subp_Decl | Ada_Subp_Body | Ada_Subp_Renaming_Decl | Ada_Package_Body_Stub | Ada_Protected_Body_Stub | Ada_Subp_Body_Stub | Ada_Task_Body_Stub | Ada_Entry_Body | Ada_Package_Body | Ada_Protected_Body | Ada_Task_Body | Ada_Exception_Handler | Ada_Generic_Package_Decl | Ada_Generic_Subp_Decl | Ada_Generic_Package_Instantiation | Ada_Generic_Subp_Instantiation | Ada_Generic_Package_Renaming_Decl | Ada_Generic_Subp_Renaming_Decl | Ada_Package_Renaming_Decl | Ada_Single_Protected_Decl | Ada_Single_Task_Decl | Ada_Private_Part | Ada_Decl_Expr | Ada_Quantified_Expr | Ada_Accept_Stmt | Ada_Accept_Stmt_With_Stmts | Ada_For_Loop_Stmt | Ada_Begin_Block | Ada_Decl_Block | Ada_Extended_Return_Stmt | Ada_Named_Stmt | Ada_Access_To_Subp_Def
            then Get_Parent_Env
            else Node.Self_Env);
      end Get_Base_Env;
      Base_Env : Lexical_Env := Get_Base_Env;
      Result   : constant Lexical_Env := Rebind_Env (Base_Env, E_Info);
   begin
      Dec_Ref (Base_Env);
      return Result;
   end Node_Env;
   ------------
   -- Parent --
   ------------
   function Parent
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity is
   begin
      --  TODO: shed entity information as appropriate
      return (Node.Parent, E_Info);
   end Parent;
   -------------
   -- Parents --
   -------------
   function Parents
     (Node      : Bare_Ada_Node;
      With_Self : Boolean := True;
      E_Info    : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity_Array_Access
   is
      Bare_Parents : Bare_Ada_Node_Array_Access := Parents (Node, With_Self);
      Result       : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (Bare_Parents.N);
   begin
      --  TODO: shed entity information as appropriate
      for I in Bare_Parents.Items'Range loop
         Result.Items (I) := (Bare_Parents.Items (I), E_Info);
      end loop;
      Dec_Ref (Bare_Parents);
      return Result;
   end Parents;
   --------------
   -- Children --
   --------------
   function Children
     (Node   : Bare_Ada_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity_Array_Access
   is
      Bare_Children : Bare_Ada_Node_Array_Access := Children (Node);
      Result        : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (Bare_Children.N);
   begin
      --  TODO: shed entity information as appropriate
      for I in Bare_Children.Items'Range loop
         Result.Items (I) := (Bare_Children.Items (I), E_Info);
      end loop;
      Dec_Ref (Bare_Children);
      return Result;
   end Children;
   ---------------------
   -- New_Unit_String --
   ---------------------
   function New_Unit_String
     (Unit : Internal_Unit; Str : String) return String_Access
   is
      procedure Register_Destroyable_String is new Register_Destroyable_Gen
        (String, String_Access, Free);
   begin
      return Ret : String_Access := new String'(Str) do
         Register_Destroyable_String (Unit, Ret);
      end return;
   end New_Unit_String;
   --------------------------------
   -- Assign_Names_To_Logic_Vars --
   --------------------------------
   procedure Assign_Names_To_Logic_Vars (Node : Bare_Ada_Node) is
      pragma Warnings (Off, "referenced");
      procedure Assign
        (Node  : Bare_Ada_Node;
         LV    : in out Logic_Var_Record;
         Field : String);
      --  Assign a name to the LV logic variable. Node must be the node that
      --  owns LV, and Field must be the name of the field in Node that holds
      --  LV.
      ------------
      -- Assign --
      ------------
      procedure Assign
        (Node  : Bare_Ada_Node;
         LV    : in out Logic_Var_Record;
         Field : String) is
      begin
         LV.Dbg_Name :=
           new String'(Image (Short_Text_Image (Node)) & "." & Field);
      end Assign;
      K : constant Ada_Node_Kind_Type := Node.Kind;
      pragma Warnings (On, "referenced");
   begin
      case Ada_Ada_Node (K) is
when Ada_Attribute_Ref_Range =>
declare
N_Bare_Attribute_Ref : constant Bare_Attribute_Ref := Node;
begin
Assign (N_Bare_Attribute_Ref,        N_Bare_Attribute_Ref.Attribute_Ref_R_Ref_Var,        "r_ref_var");
end;
when Ada_Call_Expr_Range =>
declare
N_Bare_Call_Expr : constant Bare_Call_Expr := Node;
begin
Assign (N_Bare_Call_Expr,        N_Bare_Call_Expr.Call_Expr_R_Called_Spec,        "r_called_spec");
end;
when Ada_Explicit_Deref_Range =>
declare
N_Bare_Explicit_Deref : constant Bare_Explicit_Deref := Node;
begin
Assign (N_Bare_Explicit_Deref,        N_Bare_Explicit_Deref.Explicit_Deref_R_Called_Spec,        "r_called_spec");
end;
when Ada_Reduce_Attribute_Ref_Range =>
declare
N_Bare_Reduce_Attribute_Ref : constant Bare_Reduce_Attribute_Ref := Node;
begin
Assign (N_Bare_Reduce_Attribute_Ref,        N_Bare_Reduce_Attribute_Ref.Reduce_Attribute_Ref_R_Ref_Var,        "r_ref_var");
end;
when Ada_Target_Name_Range =>
declare
N_Bare_Target_Name : constant Bare_Target_Name := Node;
begin
Assign (N_Bare_Target_Name,        N_Bare_Target_Name.Target_Name_R_Ref_Var,        "r_ref_var");
end;
when Ada_Update_Attribute_Ref_Range =>
declare
N_Bare_Update_Attribute_Ref : constant Bare_Update_Attribute_Ref := Node;
begin
Assign (N_Bare_Update_Attribute_Ref,        N_Bare_Update_Attribute_Ref.Update_Attribute_Ref_R_Ref_Var,        "r_ref_var");
end;
when others => null;
end case;
      for Child of Internal_Bare_Ada_Node_Array'(Children (Node)) loop
         if Child /= null then
            Assign_Names_To_Logic_Vars (Child);
         end if;
      end loop;
   end Assign_Names_To_Logic_Vars;
   --------------------------
   -- Initialization_Error --
   --------------------------
   function Initialization_Error
     (Exc : Ada.Exceptions.Exception_Occurrence)
      return Error_Initialization_State
   is
      use Ada.Exceptions;
   begin
      if
            Exception_Identity (Exc) = Precondition_Failure'Identity
         then
            return Raised_Precondition_Failure;
         elsif
            Exception_Identity (Exc) = Property_Error'Identity
         then
            return Raised_Property_Error;
      else
         raise Program_Error;
      end if;
   end Initialization_Error;
   ----------------------------------
   -- Reraise_Initialization_Error --
   ----------------------------------
   procedure Reraise_Initialization_Error
     (Node    : Bare_Ada_Node;
      State   : Error_Initialization_State;
      Message : String)
   is
      Exc : Ada.Exceptions.Exception_Id;
   begin
      case State is
            when Raised_Precondition_Failure =>
               Exc := Precondition_Failure'Identity;
            when Raised_Property_Error =>
               Exc := Property_Error'Identity;
      end case;
      Raise_Property_Exception (Node, Exc, Message);
   end Reraise_Initialization_Error;
   ----------------
   -- Text_Image --
   ----------------
   function Text_Image (Ent : Internal_Entity) return Text_Type is
   begin
      if Ent.Node /= null then
         declare
            Node_Image : constant Text_Type := Short_Text_Image (Ent.Node);
         begin
            return
            (if Ent.Info.Rebindings /= null
             then "<| "
             & Node_Image (Node_Image'First + 1 .. Node_Image'Last - 1) & " "
             & AST_Envs.Text_Image (Ent.Info.Rebindings) & " |>"
             else Node_Image);
         end;
      else
         return "None";
      end if;
   end Text_Image;
   ---------------------
   -- Full_Sloc_Image --
   ---------------------
   function Full_Sloc_Image (Node : Bare_Ada_Node) return String_Type
   is
      Res      : constant Text_Type :=
        To_Text
          (Ada.Directories.Simple_Name
             (Get_Filename (Unit (Node))))
           & ":" & To_Text (Image (Start_Sloc (Sloc_Range (Node)))) & ": ";
   begin
      return Create_String (Res);
   end Full_Sloc_Image;
   -----------
   -- Image --
   -----------
   function Image (Ent : Internal_Entity) return String is
      Result : constant Text_Type := Text_Image (Ent);
   begin
      return Image (Result);
   end Image;
   ---------------
   -- Can_Reach --
   ---------------
   function Can_Reach (El, From : Bare_Ada_Node) return Boolean is
   begin
      return Ada_Node_P_Can_Reach (El, From);
   end Can_Reach;
   -----------------
   -- Hash_Entity --
   -----------------
   function Hash_Entity (Self : Internal_Entity) return Hash_Type is
   begin
      return Combine
        ((Hash (Self.Node), Hash (Self.Info.Rebindings), Hash (Self.Info.Md)));
   end Hash_Entity;
   --------------------
   -- Compare_Entity --
   --------------------
   function Compare_Entity (Left, Right : Internal_Entity) return Boolean
   is
   begin
      return Left.Node = Right.Node
             and then Left.Info.Rebindings = Right.Info.Rebindings
             and then Compare_Metadata (Left.Info.Md, Right.Info.Md);
   end Compare_Entity;
   --------------------------------
   -- Create_Dynamic_Lexical_Env --
   --------------------------------
   function Create_Dynamic_Lexical_Env
     (Self              : Bare_Ada_Node;
      Assocs_Getter     : Inner_Env_Assocs_Resolver;
      Assoc_Resolver    : Entity_Resolver;
      Transitive_Parent : Boolean;
      Sym_Table         : Symbol_Table) return Lexical_Env
   is
      Unit : constant Internal_Unit := Self.Unit;
   begin
      --  This restriction is necessary to avoid relocation issues when
      --  Self.Self_Env is terminated.
      if Is_Foreign_Strict (Self.Self_Env, Self) then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "cannot create a dynamic lexical env when Self.Self_Env is"
            & " foreign");
      end if;
      return Result : constant Lexical_Env := Create_Dynamic_Lexical_Env
        (Parent            => Null_Lexical_Env,
         Node              => Self,
         Transitive_Parent => Transitive_Parent,
         Owner             => Convert_Unit (Unit),
         Assocs_Getter     => Assocs_Getter,
         Assoc_Resolver    => Assoc_Resolver,
         Sym_Table         => Sym_Table)
      do
         --  Since dynamic lexical environments can only be created in lazy
         --  field initializers, it is fine to tie Result's lifetime to the
         --  its owning unit's lifetime.
         Register_Destroyable (Unit, Unwrap (Result));
      end return;
   end Create_Dynamic_Lexical_Env;
   procedure Destroy_Synthetic_Node (Node : in out Bare_Ada_Node);
   --  Helper for the Register_Destroyable above
   ------------
   -- Length --
   ------------
   function Length (Node : Bare_Ada_List) return Natural
   is (if Node = null then 0 else Children_Count (Node));
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (B : Boolean) return String is
      begin
         return (if B then "True" else "False");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (I : Integer) return String is
      begin
         return Integer'Image (I);
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (S : Symbol_Type) return String is
      begin
         return (if S = No_Symbol
                 then "None"
                 else Image (S, With_Quotes => True));
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (C : Character_Type) return String is
         C_Str : constant Text_Type := (1 => C);
      begin
         return "'" & Image (C_Str) & "'";
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (S : String_Type) return String is
      begin
         return Image (S.Content, With_Quotes => True);
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (Env : Lexical_Env) return String is
      begin
         case Env.Kind is
         when Static_Primary =>
            return "<LexicalEnv static-primary for "
                   & Trace_Image (Env_Node (Env)) & ">";
         when others =>
            return "<LexicalEnv synthetic>";
         end case;
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (R : Env_Rebindings) return String is
      begin
         return Image (Text_Image (R));
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (Unit : Internal_Unit) return String is
      begin
         return "Internal_Unit (""" & Basename (Unit) & """)";
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (Eq : Logic_Equation) return String is
         pragma Unreferenced (Eq);
      begin
         return "<LogicEquation>";
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (Var : Logic_Var) return String is
         pragma Unreferenced (Var);
      begin
         return "<LogicVariable>";
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (K : Analysis_Unit_Kind) return String is
      begin
         return Analysis_Unit_Kind'Image (K);
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      function Trace_Image (Self : Ref_Categories) return String is
         Result : Unbounded_String;
         First  : Boolean := True;
      begin
         Append (Result, "RefCategories(");
         for C in Ref_Category loop
            if Self (C) then
               if First then
                  First := False;
               else
                  Append (Result, ", ");
               end if;
               Append (Result, C'Image);
            end if;
         end loop;
         Append (Result, ")");
         return To_String (Result);
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Metadata) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Dottable_Subp => " & Trace_Image (R.Dottable_Subp)
                        & ", "
                     & "Primitive => " & Trace_Image (R.Primitive)
                        & ", "
                     & "Primitive_Real_Type => " & Trace_Image (R.Primitive_Real_Type)
               & ")");
      end Trace_Image;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Info) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine ((Hash (R.Md), Hash (R.Rebindings), Hash (R.From_Rebound)));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Info) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Md => " & Trace_Image (R.Md)
                        & ", "
                     & "Rebindings => " & Trace_Image (R.Rebindings)
                        & ", "
                     & "From_Rebound => " & Trace_Image (R.From_Rebound)
               & ")");
      end Trace_Image;
      function Create_Internal_Entity
        (Node : Bare_Ada_Node; Info : Internal_Entity_Info)
         return Internal_Entity is
      begin
         if Node = null then
            return No_Entity;
         end if;
         return (Node => Node, Info => Info);
      end;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Expr
        (Node : Bare_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Expr is
      begin
         if Node = null then
            return No_Entity_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Expr) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Aspect) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine ((Hash (R.Exists), Hash (R.Node), Hash (R.Value), Hash (R.Inherited)));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Aspect) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Exists => " & Trace_Image (R.Exists)
                        & ", "
                     & "Node => " & Trace_Image (R.Node)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
                        & ", "
                     & "Inherited => " & Trace_Image (R.Inherited)
               & ")");
      end Trace_Image;
      function Create_Internal_Entity_Basic_Decl
        (Node : Bare_Basic_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Basic_Decl is
      begin
         if Node = null then
            return No_Entity_Basic_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Basic_Decl) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Basic_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Completion_Item) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Decl => " & Trace_Image (R.Decl)
                        & ", "
                     & "Is_Dot_Call => " & Trace_Image (R.Is_Dot_Call)
                        & ", "
                     & "Is_Visible => " & Trace_Image (R.Is_Visible)
                        & ", "
                     & "Weight => " & Trace_Image (R.Weight)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Designated_Env) is
      begin
               Inc_Ref (R.Direct_Env);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Designated_Env) is
      begin
               Dec_Ref (R.Direct_Env);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Designated_Env) return Boolean is
      begin
         return L.Kind = R.Kind and then L.Env_Name = R.Env_Name and then Equivalent (L.Direct_Env, R.Direct_Env);
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Designated_Env) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Kind => " & Trace_Image (R.Kind)
                        & ", "
                     & "Env_Name => " & Trace_Image (R.Env_Name)
                        & ", "
                     & "Direct_Env => " & Trace_Image (R.Direct_Env)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Discrete_Range) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Low_Bound => " & Trace_Image (R.Low_Bound)
                        & ", "
                     & "High_Bound => " & Trace_Image (R.High_Bound)
               & ")");
      end Trace_Image;
      function Create_Internal_Entity_Ada_List
        (Node : Bare_Ada_List; Info : Internal_Entity_Info)
         return Internal_Entity_Ada_List is
      begin
         if Node = null then
            return No_Entity_Ada_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Ada_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Ada_Node_List
        (Node : Bare_Ada_Node_List; Info : Internal_Entity_Info)
         return Internal_Entity_Ada_Node_List is
      begin
         if Node = null then
            return No_Entity_Ada_Node_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Ada_Node_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Alternatives_List
        (Node : Bare_Alternatives_List; Info : Internal_Entity_Info)
         return Internal_Entity_Alternatives_List is
      begin
         if Node = null then
            return No_Entity_Alternatives_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Alternatives_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Name
        (Node : Bare_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Name is
      begin
         if Node = null then
            return No_Entity_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Single_Tok_Node
        (Node : Bare_Single_Tok_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Single_Tok_Node is
      begin
         if Node = null then
            return No_Entity_Single_Tok_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Single_Tok_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Id
        (Node : Bare_Base_Id; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Id is
      begin
         if Node = null then
            return No_Entity_Base_Id;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Id) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Identifier
        (Node : Bare_Identifier; Info : Internal_Entity_Info)
         return Internal_Entity_Identifier is
      begin
         if Node = null then
            return No_Entity_Identifier;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Identifier) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Discriminant_Values) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Discriminant => " & Trace_Image (R.Discriminant)
                        & ", "
                     & "Values => " & Trace_Image (R.Values)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Doc_Annotation) is
      begin
               Inc_Ref (R.Key);
               Inc_Ref (R.Value);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Doc_Annotation) is
      begin
               Dec_Ref (R.Key);
               Dec_Ref (R.Value);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Doc_Annotation) return Boolean is
      begin
         return Equivalent (L.Key, R.Key) and then Equivalent (L.Value, R.Value);
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Doc_Annotation) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Key => " & Trace_Image (R.Key)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
               & ")");
      end Trace_Image;
      function Create_Internal_Entity_Abort_Node
        (Node : Bare_Abort_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Abort_Node is
      begin
         if Node = null then
            return No_Entity_Abort_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abort_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abort_Absent
        (Node : Bare_Abort_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Abort_Absent is
      begin
         if Node = null then
            return No_Entity_Abort_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abort_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abort_Present
        (Node : Bare_Abort_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Abort_Present is
      begin
         if Node = null then
            return No_Entity_Abort_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abort_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Stmt
        (Node : Bare_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Stmt is
      begin
         if Node = null then
            return No_Entity_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Simple_Stmt
        (Node : Bare_Simple_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Simple_Stmt is
      begin
         if Node = null then
            return No_Entity_Simple_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Simple_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abort_Stmt
        (Node : Bare_Abort_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Abort_Stmt is
      begin
         if Node = null then
            return No_Entity_Abort_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abort_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_Node
        (Node : Bare_Abstract_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_Node is
      begin
         if Node = null then
            return No_Entity_Abstract_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_Absent
        (Node : Bare_Abstract_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_Absent is
      begin
         if Node = null then
            return No_Entity_Abstract_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Basic_Subp_Decl
        (Node : Bare_Basic_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Basic_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Basic_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Basic_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Classic_Subp_Decl
        (Node : Bare_Classic_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Classic_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Classic_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Classic_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Formal_Subp_Decl
        (Node : Bare_Formal_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Formal_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Formal_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Formal_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_Formal_Subp_Decl
        (Node : Bare_Abstract_Formal_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_Formal_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Abstract_Formal_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_Formal_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_Present
        (Node : Bare_Abstract_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_Present is
      begin
         if Node = null then
            return No_Entity_Abstract_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_State_Decl
        (Node : Bare_Abstract_State_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_State_Decl is
      begin
         if Node = null then
            return No_Entity_Abstract_State_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_State_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_State_Decl_Expr
        (Node : Bare_Abstract_State_Decl_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_State_Decl_Expr is
      begin
         if Node = null then
            return No_Entity_Abstract_State_Decl_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_State_Decl_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_State_Decl_List
        (Node : Bare_Abstract_State_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_State_Decl_List is
      begin
         if Node = null then
            return No_Entity_Abstract_State_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_State_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Abstract_Subp_Decl
        (Node : Bare_Abstract_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Abstract_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Abstract_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Abstract_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Composite_Stmt
        (Node : Bare_Composite_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Composite_Stmt is
      begin
         if Node = null then
            return No_Entity_Composite_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Composite_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Accept_Stmt
        (Node : Bare_Accept_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Accept_Stmt is
      begin
         if Node = null then
            return No_Entity_Accept_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Accept_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Body_Node
        (Node : Bare_Body_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Body_Node is
      begin
         if Node = null then
            return No_Entity_Body_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Body_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Accept_Stmt_Body
        (Node : Bare_Accept_Stmt_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Accept_Stmt_Body is
      begin
         if Node = null then
            return No_Entity_Accept_Stmt_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Accept_Stmt_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Accept_Stmt_With_Stmts
        (Node : Bare_Accept_Stmt_With_Stmts; Info : Internal_Entity_Info)
         return Internal_Entity_Accept_Stmt_With_Stmts is
      begin
         if Node = null then
            return No_Entity_Accept_Stmt_With_Stmts;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Accept_Stmt_With_Stmts) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Type_Def
        (Node : Bare_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Def is
      begin
         if Node = null then
            return No_Entity_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Access_Def
        (Node : Bare_Access_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Access_Def is
      begin
         if Node = null then
            return No_Entity_Access_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Access_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Access_To_Subp_Def
        (Node : Bare_Access_To_Subp_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Access_To_Subp_Def is
      begin
         if Node = null then
            return No_Entity_Access_To_Subp_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Access_To_Subp_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Aggregate
        (Node : Bare_Base_Aggregate; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Aggregate is
      begin
         if Node = null then
            return No_Entity_Base_Aggregate;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Aggregate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aggregate
        (Node : Bare_Aggregate; Info : Internal_Entity_Info)
         return Internal_Entity_Aggregate is
      begin
         if Node = null then
            return No_Entity_Aggregate;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aggregate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Basic_Assoc
        (Node : Bare_Basic_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Basic_Assoc is
      begin
         if Node = null then
            return No_Entity_Basic_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Basic_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aggregate_Assoc
        (Node : Bare_Aggregate_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Aggregate_Assoc is
      begin
         if Node = null then
            return No_Entity_Aggregate_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aggregate_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aliased_Node
        (Node : Bare_Aliased_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Aliased_Node is
      begin
         if Node = null then
            return No_Entity_Aliased_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aliased_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aliased_Absent
        (Node : Bare_Aliased_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Aliased_Absent is
      begin
         if Node = null then
            return No_Entity_Aliased_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aliased_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aliased_Present
        (Node : Bare_Aliased_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Aliased_Present is
      begin
         if Node = null then
            return No_Entity_Aliased_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aliased_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_All_Node
        (Node : Bare_All_Node; Info : Internal_Entity_Info)
         return Internal_Entity_All_Node is
      begin
         if Node = null then
            return No_Entity_All_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_All_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_All_Absent
        (Node : Bare_All_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_All_Absent is
      begin
         if Node = null then
            return No_Entity_All_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_All_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_All_Present
        (Node : Bare_All_Present; Info : Internal_Entity_Info)
         return Internal_Entity_All_Present is
      begin
         if Node = null then
            return No_Entity_All_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_All_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Allocator
        (Node : Bare_Allocator; Info : Internal_Entity_Info)
         return Internal_Entity_Allocator is
      begin
         if Node = null then
            return No_Entity_Allocator;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Allocator) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Anonymous_Expr_Decl
        (Node : Bare_Anonymous_Expr_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Anonymous_Expr_Decl is
      begin
         if Node = null then
            return No_Entity_Anonymous_Expr_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Anonymous_Expr_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Type_Expr
        (Node : Bare_Type_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Expr is
      begin
         if Node = null then
            return No_Entity_Type_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Anonymous_Type
        (Node : Bare_Anonymous_Type; Info : Internal_Entity_Info)
         return Internal_Entity_Anonymous_Type is
      begin
         if Node = null then
            return No_Entity_Anonymous_Type;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Anonymous_Type) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Type_Access_Def
        (Node : Bare_Base_Type_Access_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Type_Access_Def is
      begin
         if Node = null then
            return No_Entity_Base_Type_Access_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Type_Access_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Anonymous_Type_Access_Def
        (Node : Bare_Anonymous_Type_Access_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Anonymous_Type_Access_Def is
      begin
         if Node = null then
            return No_Entity_Anonymous_Type_Access_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Anonymous_Type_Access_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Type_Decl
        (Node : Bare_Base_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Base_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Base_Type_Decl) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Type_Decl
        (Node : Bare_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Anonymous_Type_Decl
        (Node : Bare_Anonymous_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Anonymous_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Anonymous_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Anonymous_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Array_Indices
        (Node : Bare_Array_Indices; Info : Internal_Entity_Info)
         return Internal_Entity_Array_Indices is
      begin
         if Node = null then
            return No_Entity_Array_Indices;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Array_Indices) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Array_Subcomponent_Choice_Name
        (Node : Bare_Array_Subcomponent_Choice_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Array_Subcomponent_Choice_Name is
      begin
         if Node = null then
            return No_Entity_Array_Subcomponent_Choice_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Array_Subcomponent_Choice_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Array_Type_Def
        (Node : Bare_Array_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Array_Type_Def is
      begin
         if Node = null then
            return No_Entity_Array_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Array_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aspect_Assoc
        (Node : Bare_Aspect_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Aspect_Assoc is
      begin
         if Node = null then
            return No_Entity_Aspect_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aspect_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aspect_Assoc_List
        (Node : Bare_Aspect_Assoc_List; Info : Internal_Entity_Info)
         return Internal_Entity_Aspect_Assoc_List is
      begin
         if Node = null then
            return No_Entity_Aspect_Assoc_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aspect_Assoc_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aspect_Clause
        (Node : Bare_Aspect_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Aspect_Clause is
      begin
         if Node = null then
            return No_Entity_Aspect_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aspect_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Aspect_Spec
        (Node : Bare_Aspect_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Aspect_Spec is
      begin
         if Node = null then
            return No_Entity_Aspect_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Aspect_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Assign_Stmt
        (Node : Bare_Assign_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Assign_Stmt is
      begin
         if Node = null then
            return No_Entity_Assign_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Assign_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Basic_Assoc_List
        (Node : Bare_Basic_Assoc_List; Info : Internal_Entity_Info)
         return Internal_Entity_Basic_Assoc_List is
      begin
         if Node = null then
            return No_Entity_Basic_Assoc_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Basic_Assoc_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Assoc_List
        (Node : Bare_Assoc_List; Info : Internal_Entity_Info)
         return Internal_Entity_Assoc_List is
      begin
         if Node = null then
            return No_Entity_Assoc_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Assoc_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_At_Clause
        (Node : Bare_At_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_At_Clause is
      begin
         if Node = null then
            return No_Entity_At_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_At_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Attribute_Def_Clause
        (Node : Bare_Attribute_Def_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Attribute_Def_Clause is
      begin
         if Node = null then
            return No_Entity_Attribute_Def_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Attribute_Def_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Attribute_Ref
        (Node : Bare_Attribute_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Attribute_Ref is
      begin
         if Node = null then
            return No_Entity_Attribute_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Attribute_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Assoc
        (Node : Bare_Base_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Assoc is
      begin
         if Node = null then
            return No_Entity_Base_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Assoc_List
        (Node : Bare_Base_Assoc_List; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Assoc_List is
      begin
         if Node = null then
            return No_Entity_Base_Assoc_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Assoc_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Formal_Param_Decl
        (Node : Bare_Base_Formal_Param_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Formal_Param_Decl is
      begin
         if Node = null then
            return No_Entity_Base_Formal_Param_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Formal_Param_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Formal_Param_Holder
        (Node : Bare_Base_Formal_Param_Holder; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Formal_Param_Holder is
      begin
         if Node = null then
            return No_Entity_Base_Formal_Param_Holder;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Formal_Param_Holder) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Loop_Stmt
        (Node : Bare_Base_Loop_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Loop_Stmt is
      begin
         if Node = null then
            return No_Entity_Base_Loop_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Loop_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Package_Decl
        (Node : Bare_Base_Package_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Package_Decl is
      begin
         if Node = null then
            return No_Entity_Base_Package_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Package_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Record_Def
        (Node : Bare_Base_Record_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Record_Def is
      begin
         if Node = null then
            return No_Entity_Base_Record_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Record_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Subp_Body
        (Node : Bare_Base_Subp_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Subp_Body is
      begin
         if Node = null then
            return No_Entity_Base_Subp_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Subp_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Subp_Spec
        (Node : Bare_Base_Subp_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Subp_Spec is
      begin
         if Node = null then
            return No_Entity_Base_Subp_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Subp_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Base_Subtype_Decl
        (Node : Bare_Base_Subtype_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Subtype_Decl is
      begin
         if Node = null then
            return No_Entity_Base_Subtype_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Subtype_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Block_Stmt
        (Node : Bare_Block_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Block_Stmt is
      begin
         if Node = null then
            return No_Entity_Block_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Block_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Begin_Block
        (Node : Bare_Begin_Block; Info : Internal_Entity_Info)
         return Internal_Entity_Begin_Block is
      begin
         if Node = null then
            return No_Entity_Begin_Block;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Begin_Block) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Bin_Op
        (Node : Bare_Bin_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Bin_Op is
      begin
         if Node = null then
            return No_Entity_Bin_Op;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Bin_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Body_Stub
        (Node : Bare_Body_Stub; Info : Internal_Entity_Info)
         return Internal_Entity_Body_Stub is
      begin
         if Node = null then
            return No_Entity_Body_Stub;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Body_Stub) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Box_Expr
        (Node : Bare_Box_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Box_Expr is
      begin
         if Node = null then
            return No_Entity_Box_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Box_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Bracket_Aggregate
        (Node : Bare_Bracket_Aggregate; Info : Internal_Entity_Info)
         return Internal_Entity_Bracket_Aggregate is
      begin
         if Node = null then
            return No_Entity_Bracket_Aggregate;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Bracket_Aggregate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Delta_Aggregate
        (Node : Bare_Delta_Aggregate; Info : Internal_Entity_Info)
         return Internal_Entity_Delta_Aggregate is
      begin
         if Node = null then
            return No_Entity_Delta_Aggregate;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Delta_Aggregate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Bracket_Delta_Aggregate
        (Node : Bare_Bracket_Delta_Aggregate; Info : Internal_Entity_Info)
         return Internal_Entity_Bracket_Delta_Aggregate is
      begin
         if Node = null then
            return No_Entity_Bracket_Delta_Aggregate;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Bracket_Delta_Aggregate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Call_Expr
        (Node : Bare_Call_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Call_Expr is
      begin
         if Node = null then
            return No_Entity_Call_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Call_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Call_Stmt
        (Node : Bare_Call_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Call_Stmt is
      begin
         if Node = null then
            return No_Entity_Call_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Call_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Cond_Expr
        (Node : Bare_Cond_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Cond_Expr is
      begin
         if Node = null then
            return No_Entity_Cond_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Cond_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Case_Expr
        (Node : Bare_Case_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Case_Expr is
      begin
         if Node = null then
            return No_Entity_Case_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Case_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Case_Expr_Alternative
        (Node : Bare_Case_Expr_Alternative; Info : Internal_Entity_Info)
         return Internal_Entity_Case_Expr_Alternative is
      begin
         if Node = null then
            return No_Entity_Case_Expr_Alternative;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Case_Expr_Alternative) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Case_Expr_Alternative_List
        (Node : Bare_Case_Expr_Alternative_List; Info : Internal_Entity_Info)
         return Internal_Entity_Case_Expr_Alternative_List is
      begin
         if Node = null then
            return No_Entity_Case_Expr_Alternative_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Case_Expr_Alternative_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Case_Stmt
        (Node : Bare_Case_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Case_Stmt is
      begin
         if Node = null then
            return No_Entity_Case_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Case_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Case_Stmt_Alternative
        (Node : Bare_Case_Stmt_Alternative; Info : Internal_Entity_Info)
         return Internal_Entity_Case_Stmt_Alternative is
      begin
         if Node = null then
            return No_Entity_Case_Stmt_Alternative;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Case_Stmt_Alternative) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Case_Stmt_Alternative_List
        (Node : Bare_Case_Stmt_Alternative_List; Info : Internal_Entity_Info)
         return Internal_Entity_Case_Stmt_Alternative_List is
      begin
         if Node = null then
            return No_Entity_Case_Stmt_Alternative_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Case_Stmt_Alternative_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Char_Literal
        (Node : Bare_Char_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Char_Literal is
      begin
         if Node = null then
            return No_Entity_Char_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Char_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Classwide_Type_Decl
        (Node : Bare_Classwide_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Classwide_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Classwide_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Classwide_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Compilation_Unit
        (Node : Bare_Compilation_Unit; Info : Internal_Entity_Info)
         return Internal_Entity_Compilation_Unit is
      begin
         if Node = null then
            return No_Entity_Compilation_Unit;
         end if;
         return (Node => Node, Info => Info);
      end;
      ----------
      -- Hash --
      ----------
      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Compilation_Unit) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Compilation_Unit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Compilation_Unit_List
        (Node : Bare_Compilation_Unit_List; Info : Internal_Entity_Info)
         return Internal_Entity_Compilation_Unit_List is
      begin
         if Node = null then
            return No_Entity_Compilation_Unit_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Compilation_Unit_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Component_Clause
        (Node : Bare_Component_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Component_Clause is
      begin
         if Node = null then
            return No_Entity_Component_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Component_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Component_Decl
        (Node : Bare_Component_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Component_Decl is
      begin
         if Node = null then
            return No_Entity_Component_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Component_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Component_Def
        (Node : Bare_Component_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Component_Def is
      begin
         if Node = null then
            return No_Entity_Component_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Component_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Component_List
        (Node : Bare_Component_List; Info : Internal_Entity_Info)
         return Internal_Entity_Component_List is
      begin
         if Node = null then
            return No_Entity_Component_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Component_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constraint
        (Node : Bare_Constraint; Info : Internal_Entity_Info)
         return Internal_Entity_Constraint is
      begin
         if Node = null then
            return No_Entity_Constraint;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constraint) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Composite_Constraint
        (Node : Bare_Composite_Constraint; Info : Internal_Entity_Info)
         return Internal_Entity_Composite_Constraint is
      begin
         if Node = null then
            return No_Entity_Composite_Constraint;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Composite_Constraint) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Composite_Constraint_Assoc
        (Node : Bare_Composite_Constraint_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Composite_Constraint_Assoc is
      begin
         if Node = null then
            return No_Entity_Composite_Constraint_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Composite_Constraint_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Concat_Op
        (Node : Bare_Concat_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Concat_Op is
      begin
         if Node = null then
            return No_Entity_Concat_Op;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Concat_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Concat_Operand
        (Node : Bare_Concat_Operand; Info : Internal_Entity_Info)
         return Internal_Entity_Concat_Operand is
      begin
         if Node = null then
            return No_Entity_Concat_Operand;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Concat_Operand) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Concat_Operand_List
        (Node : Bare_Concat_Operand_List; Info : Internal_Entity_Info)
         return Internal_Entity_Concat_Operand_List is
      begin
         if Node = null then
            return No_Entity_Concat_Operand_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Concat_Operand_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Concrete_Formal_Subp_Decl
        (Node : Bare_Concrete_Formal_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Concrete_Formal_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Concrete_Formal_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Concrete_Formal_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Concrete_Type_Decl
        (Node : Bare_Concrete_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Concrete_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Concrete_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Concrete_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constant_Node
        (Node : Bare_Constant_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Constant_Node is
      begin
         if Node = null then
            return No_Entity_Constant_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constant_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constant_Absent
        (Node : Bare_Constant_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Constant_Absent is
      begin
         if Node = null then
            return No_Entity_Constant_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constant_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constant_Present
        (Node : Bare_Constant_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Constant_Present is
      begin
         if Node = null then
            return No_Entity_Constant_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constant_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constrained_Array_Indices
        (Node : Bare_Constrained_Array_Indices; Info : Internal_Entity_Info)
         return Internal_Entity_Constrained_Array_Indices is
      begin
         if Node = null then
            return No_Entity_Constrained_Array_Indices;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constrained_Array_Indices) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subtype_Indication
        (Node : Bare_Subtype_Indication; Info : Internal_Entity_Info)
         return Internal_Entity_Subtype_Indication is
      begin
         if Node = null then
            return No_Entity_Subtype_Indication;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subtype_Indication) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constrained_Subtype_Indication
        (Node : Bare_Constrained_Subtype_Indication; Info : Internal_Entity_Info)
         return Internal_Entity_Constrained_Subtype_Indication is
      begin
         if Node = null then
            return No_Entity_Constrained_Subtype_Indication;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constrained_Subtype_Indication) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Constraint_List
        (Node : Bare_Constraint_List; Info : Internal_Entity_Info)
         return Internal_Entity_Constraint_List is
      begin
         if Node = null then
            return No_Entity_Constraint_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Constraint_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Contract_Case_Assoc
        (Node : Bare_Contract_Case_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Contract_Case_Assoc is
      begin
         if Node = null then
            return No_Entity_Contract_Case_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Contract_Case_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Contract_Case_Assoc_List
        (Node : Bare_Contract_Case_Assoc_List; Info : Internal_Entity_Info)
         return Internal_Entity_Contract_Case_Assoc_List is
      begin
         if Node = null then
            return No_Entity_Contract_Case_Assoc_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Contract_Case_Assoc_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Contract_Cases
        (Node : Bare_Contract_Cases; Info : Internal_Entity_Info)
         return Internal_Entity_Contract_Cases is
      begin
         if Node = null then
            return No_Entity_Contract_Cases;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Contract_Cases) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Real_Type_Def
        (Node : Bare_Real_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Real_Type_Def is
      begin
         if Node = null then
            return No_Entity_Real_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Real_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Decimal_Fixed_Point_Def
        (Node : Bare_Decimal_Fixed_Point_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Decimal_Fixed_Point_Def is
      begin
         if Node = null then
            return No_Entity_Decimal_Fixed_Point_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decimal_Fixed_Point_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Decl_Block
        (Node : Bare_Decl_Block; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_Block is
      begin
         if Node = null then
            return No_Entity_Decl_Block;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_Block) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Decl_Expr
        (Node : Bare_Decl_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_Expr is
      begin
         if Node = null then
            return No_Entity_Decl_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Decl_List
        (Node : Bare_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_List is
      begin
         if Node = null then
            return No_Entity_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Declarative_Part
        (Node : Bare_Declarative_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Declarative_Part is
      begin
         if Node = null then
            return No_Entity_Declarative_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Declarative_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Defining_Name
        (Node : Bare_Defining_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Defining_Name is
      begin
         if Node = null then
            return No_Entity_Defining_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Defining_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Defining_Name_List
        (Node : Bare_Defining_Name_List; Info : Internal_Entity_Info)
         return Internal_Entity_Defining_Name_List is
      begin
         if Node = null then
            return No_Entity_Defining_Name_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Defining_Name_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Delay_Stmt
        (Node : Bare_Delay_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Delay_Stmt is
      begin
         if Node = null then
            return No_Entity_Delay_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Delay_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Delta_Constraint
        (Node : Bare_Delta_Constraint; Info : Internal_Entity_Info)
         return Internal_Entity_Delta_Constraint is
      begin
         if Node = null then
            return No_Entity_Delta_Constraint;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Delta_Constraint) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Derived_Type_Def
        (Node : Bare_Derived_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Derived_Type_Def is
      begin
         if Node = null then
            return No_Entity_Derived_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Derived_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Digits_Constraint
        (Node : Bare_Digits_Constraint; Info : Internal_Entity_Info)
         return Internal_Entity_Digits_Constraint is
      begin
         if Node = null then
            return No_Entity_Digits_Constraint;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Digits_Constraint) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discrete_Base_Subtype_Decl
        (Node : Bare_Discrete_Base_Subtype_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Discrete_Base_Subtype_Decl is
      begin
         if Node = null then
            return No_Entity_Discrete_Base_Subtype_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discrete_Base_Subtype_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discrete_Subtype_Indication
        (Node : Bare_Discrete_Subtype_Indication; Info : Internal_Entity_Info)
         return Internal_Entity_Discrete_Subtype_Indication is
      begin
         if Node = null then
            return No_Entity_Discrete_Subtype_Indication;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discrete_Subtype_Indication) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discrete_Subtype_Name
        (Node : Bare_Discrete_Subtype_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Discrete_Subtype_Name is
      begin
         if Node = null then
            return No_Entity_Discrete_Subtype_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discrete_Subtype_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Identifier_List
        (Node : Bare_Identifier_List; Info : Internal_Entity_Info)
         return Internal_Entity_Identifier_List is
      begin
         if Node = null then
            return No_Entity_Identifier_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Identifier_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discriminant_Choice_List
        (Node : Bare_Discriminant_Choice_List; Info : Internal_Entity_Info)
         return Internal_Entity_Discriminant_Choice_List is
      begin
         if Node = null then
            return No_Entity_Discriminant_Choice_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discriminant_Choice_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discriminant_Part
        (Node : Bare_Discriminant_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Discriminant_Part is
      begin
         if Node = null then
            return No_Entity_Discriminant_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discriminant_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discriminant_Spec
        (Node : Bare_Discriminant_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Discriminant_Spec is
      begin
         if Node = null then
            return No_Entity_Discriminant_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discriminant_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Discriminant_Spec_List
        (Node : Bare_Discriminant_Spec_List; Info : Internal_Entity_Info)
         return Internal_Entity_Discriminant_Spec_List is
      begin
         if Node = null then
            return No_Entity_Discriminant_Spec_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Discriminant_Spec_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Dotted_Name
        (Node : Bare_Dotted_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Dotted_Name is
      begin
         if Node = null then
            return No_Entity_Dotted_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Dotted_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Elsif_Expr_Part
        (Node : Bare_Elsif_Expr_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Elsif_Expr_Part is
      begin
         if Node = null then
            return No_Entity_Elsif_Expr_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Elsif_Expr_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Elsif_Expr_Part_List
        (Node : Bare_Elsif_Expr_Part_List; Info : Internal_Entity_Info)
         return Internal_Entity_Elsif_Expr_Part_List is
      begin
         if Node = null then
            return No_Entity_Elsif_Expr_Part_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Elsif_Expr_Part_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Elsif_Stmt_Part
        (Node : Bare_Elsif_Stmt_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Elsif_Stmt_Part is
      begin
         if Node = null then
            return No_Entity_Elsif_Stmt_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Elsif_Stmt_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Elsif_Stmt_Part_List
        (Node : Bare_Elsif_Stmt_Part_List; Info : Internal_Entity_Info)
         return Internal_Entity_Elsif_Stmt_Part_List is
      begin
         if Node = null then
            return No_Entity_Elsif_Stmt_Part_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Elsif_Stmt_Part_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_End_Name
        (Node : Bare_End_Name; Info : Internal_Entity_Info)
         return Internal_Entity_End_Name is
      begin
         if Node = null then
            return No_Entity_End_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_End_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Entry_Body
        (Node : Bare_Entry_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Entry_Body is
      begin
         if Node = null then
            return No_Entity_Entry_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Entry_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Entry_Completion_Formal_Params
        (Node : Bare_Entry_Completion_Formal_Params; Info : Internal_Entity_Info)
         return Internal_Entity_Entry_Completion_Formal_Params is
      begin
         if Node = null then
            return No_Entity_Entry_Completion_Formal_Params;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Entry_Completion_Formal_Params) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Entry_Decl
        (Node : Bare_Entry_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Entry_Decl is
      begin
         if Node = null then
            return No_Entity_Entry_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Entry_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Entry_Index_Spec
        (Node : Bare_Entry_Index_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Entry_Index_Spec is
      begin
         if Node = null then
            return No_Entity_Entry_Index_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Entry_Index_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Entry_Spec
        (Node : Bare_Entry_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Entry_Spec is
      begin
         if Node = null then
            return No_Entity_Entry_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Entry_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Enum_Lit_Synth_Type_Expr
        (Node : Bare_Enum_Lit_Synth_Type_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Lit_Synth_Type_Expr is
      begin
         if Node = null then
            return No_Entity_Enum_Lit_Synth_Type_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Lit_Synth_Type_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Enum_Literal_Decl
        (Node : Bare_Enum_Literal_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Literal_Decl is
      begin
         if Node = null then
            return No_Entity_Enum_Literal_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Literal_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Enum_Literal_Decl_List
        (Node : Bare_Enum_Literal_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Literal_Decl_List is
      begin
         if Node = null then
            return No_Entity_Enum_Literal_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Literal_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Enum_Rep_Clause
        (Node : Bare_Enum_Rep_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Rep_Clause is
      begin
         if Node = null then
            return No_Entity_Enum_Rep_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Rep_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Enum_Subp_Spec
        (Node : Bare_Enum_Subp_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Subp_Spec is
      begin
         if Node = null then
            return No_Entity_Enum_Subp_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Subp_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Enum_Type_Def
        (Node : Bare_Enum_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Type_Def is
      begin
         if Node = null then
            return No_Entity_Enum_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Error_Decl
        (Node : Bare_Error_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Error_Decl is
      begin
         if Node = null then
            return No_Entity_Error_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Error_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Error_Stmt
        (Node : Bare_Error_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Error_Stmt is
      begin
         if Node = null then
            return No_Entity_Error_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Error_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Exception_Decl
        (Node : Bare_Exception_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Exception_Decl is
      begin
         if Node = null then
            return No_Entity_Exception_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Exception_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Exception_Handler
        (Node : Bare_Exception_Handler; Info : Internal_Entity_Info)
         return Internal_Entity_Exception_Handler is
      begin
         if Node = null then
            return No_Entity_Exception_Handler;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Exception_Handler) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Exit_Stmt
        (Node : Bare_Exit_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Exit_Stmt is
      begin
         if Node = null then
            return No_Entity_Exit_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Exit_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Explicit_Deref
        (Node : Bare_Explicit_Deref; Info : Internal_Entity_Info)
         return Internal_Entity_Explicit_Deref is
      begin
         if Node = null then
            return No_Entity_Explicit_Deref;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Explicit_Deref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Expr_List
        (Node : Bare_Expr_List; Info : Internal_Entity_Info)
         return Internal_Entity_Expr_List is
      begin
         if Node = null then
            return No_Entity_Expr_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Expr_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Expr_Alternatives_List
        (Node : Bare_Expr_Alternatives_List; Info : Internal_Entity_Info)
         return Internal_Entity_Expr_Alternatives_List is
      begin
         if Node = null then
            return No_Entity_Expr_Alternatives_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Expr_Alternatives_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Expr_Function
        (Node : Bare_Expr_Function; Info : Internal_Entity_Info)
         return Internal_Entity_Expr_Function is
      begin
         if Node = null then
            return No_Entity_Expr_Function;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Expr_Function) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Extended_Return_Stmt
        (Node : Bare_Extended_Return_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Extended_Return_Stmt is
      begin
         if Node = null then
            return No_Entity_Extended_Return_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Extended_Return_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Object_Decl
        (Node : Bare_Object_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Object_Decl is
      begin
         if Node = null then
            return No_Entity_Object_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Object_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Extended_Return_Stmt_Object_Decl
        (Node : Bare_Extended_Return_Stmt_Object_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Extended_Return_Stmt_Object_Decl is
      begin
         if Node = null then
            return No_Entity_Extended_Return_Stmt_Object_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Extended_Return_Stmt_Object_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Floating_Point_Def
        (Node : Bare_Floating_Point_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Floating_Point_Def is
      begin
         if Node = null then
            return No_Entity_Floating_Point_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Floating_Point_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_For_Loop_Iter_Filter
        (Node : Bare_For_Loop_Iter_Filter; Info : Internal_Entity_Info)
         return Internal_Entity_For_Loop_Iter_Filter is
      begin
         if Node = null then
            return No_Entity_For_Loop_Iter_Filter;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_For_Loop_Iter_Filter) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Loop_Spec
        (Node : Bare_Loop_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Loop_Spec is
      begin
         if Node = null then
            return No_Entity_Loop_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Loop_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_For_Loop_Spec
        (Node : Bare_For_Loop_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_For_Loop_Spec is
      begin
         if Node = null then
            return No_Entity_For_Loop_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_For_Loop_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_For_Loop_Stmt
        (Node : Bare_For_Loop_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_For_Loop_Stmt is
      begin
         if Node = null then
            return No_Entity_For_Loop_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_For_Loop_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_For_Loop_Var_Decl
        (Node : Bare_For_Loop_Var_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_For_Loop_Var_Decl is
      begin
         if Node = null then
            return No_Entity_For_Loop_Var_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_For_Loop_Var_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Formal_Discrete_Type_Def
        (Node : Bare_Formal_Discrete_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Formal_Discrete_Type_Def is
      begin
         if Node = null then
            return No_Entity_Formal_Discrete_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Formal_Discrete_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Formal_Type_Decl
        (Node : Bare_Formal_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Formal_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Formal_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Formal_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Format_String_Chunk
        (Node : Bare_Format_String_Chunk; Info : Internal_Entity_Info)
         return Internal_Entity_Format_String_Chunk is
      begin
         if Node = null then
            return No_Entity_Format_String_Chunk;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Format_String_Chunk) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Format_String_Chunk_List
        (Node : Bare_Format_String_Chunk_List; Info : Internal_Entity_Info)
         return Internal_Entity_Format_String_Chunk_List is
      begin
         if Node = null then
            return No_Entity_Format_String_Chunk_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Format_String_Chunk_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Format_String_Literal
        (Node : Bare_Format_String_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Format_String_Literal is
      begin
         if Node = null then
            return No_Entity_Format_String_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Format_String_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Format_String_Tok_Node
        (Node : Bare_Format_String_Tok_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Format_String_Tok_Node is
      begin
         if Node = null then
            return No_Entity_Format_String_Tok_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Format_String_Tok_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Format_String_Tok_End
        (Node : Bare_Format_String_Tok_End; Info : Internal_Entity_Info)
         return Internal_Entity_Format_String_Tok_End is
      begin
         if Node = null then
            return No_Entity_Format_String_Tok_End;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Format_String_Tok_End) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Format_String_Tok_Mid
        (Node : Bare_Format_String_Tok_Mid; Info : Internal_Entity_Info)
         return Internal_Entity_Format_String_Tok_Mid is
      begin
         if Node = null then
            return No_Entity_Format_String_Tok_Mid;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Format_String_Tok_Mid) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Format_String_Tok_Start
        (Node : Bare_Format_String_Tok_Start; Info : Internal_Entity_Info)
         return Internal_Entity_Format_String_Tok_Start is
      begin
         if Node = null then
            return No_Entity_Format_String_Tok_Start;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Format_String_Tok_Start) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Format_String_Tok_String
        (Node : Bare_Format_String_Tok_String; Info : Internal_Entity_Info)
         return Internal_Entity_Format_String_Tok_String is
      begin
         if Node = null then
            return No_Entity_Format_String_Tok_String;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Format_String_Tok_String) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Decl
        (Node : Bare_Generic_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Formal
        (Node : Bare_Generic_Formal; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal is
      begin
         if Node = null then
            return No_Entity_Generic_Formal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Formal_Obj_Decl
        (Node : Bare_Generic_Formal_Obj_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal_Obj_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Formal_Obj_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal_Obj_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Formal_Package
        (Node : Bare_Generic_Formal_Package; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal_Package is
      begin
         if Node = null then
            return No_Entity_Generic_Formal_Package;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal_Package) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Formal_Part
        (Node : Bare_Generic_Formal_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal_Part is
      begin
         if Node = null then
            return No_Entity_Generic_Formal_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Formal_Subp_Decl
        (Node : Bare_Generic_Formal_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Formal_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Formal_Type_Decl
        (Node : Bare_Generic_Formal_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Formal_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Instantiation
        (Node : Bare_Generic_Instantiation; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Instantiation is
      begin
         if Node = null then
            return No_Entity_Generic_Instantiation;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Instantiation) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Package_Decl
        (Node : Bare_Generic_Package_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Package_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Package_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Package_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Package_Instantiation
        (Node : Bare_Generic_Package_Instantiation; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Package_Instantiation is
      begin
         if Node = null then
            return No_Entity_Generic_Package_Instantiation;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Package_Instantiation) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Package_Internal
        (Node : Bare_Generic_Package_Internal; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Package_Internal is
      begin
         if Node = null then
            return No_Entity_Generic_Package_Internal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Package_Internal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Renaming_Decl
        (Node : Bare_Generic_Renaming_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Renaming_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Renaming_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Renaming_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Package_Renaming_Decl
        (Node : Bare_Generic_Package_Renaming_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Package_Renaming_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Package_Renaming_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Package_Renaming_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Subp_Decl
        (Node : Bare_Generic_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Subp_Instantiation
        (Node : Bare_Generic_Subp_Instantiation; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Subp_Instantiation is
      begin
         if Node = null then
            return No_Entity_Generic_Subp_Instantiation;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Subp_Instantiation) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Subp_Internal
        (Node : Bare_Generic_Subp_Internal; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Subp_Internal is
      begin
         if Node = null then
            return No_Entity_Generic_Subp_Internal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Subp_Internal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Generic_Subp_Renaming_Decl
        (Node : Bare_Generic_Subp_Renaming_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Subp_Renaming_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Subp_Renaming_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Subp_Renaming_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Goto_Stmt
        (Node : Bare_Goto_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Goto_Stmt is
      begin
         if Node = null then
            return No_Entity_Goto_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Goto_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Handled_Stmts
        (Node : Bare_Handled_Stmts; Info : Internal_Entity_Info)
         return Internal_Entity_Handled_Stmts is
      begin
         if Node = null then
            return No_Entity_Handled_Stmts;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Handled_Stmts) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_If_Expr
        (Node : Bare_If_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_If_Expr is
      begin
         if Node = null then
            return No_Entity_If_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_If_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_If_Stmt
        (Node : Bare_If_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_If_Stmt is
      begin
         if Node = null then
            return No_Entity_If_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_If_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Incomplete_Type_Decl
        (Node : Bare_Incomplete_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Incomplete_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Incomplete_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Incomplete_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Incomplete_Formal_Type_Decl
        (Node : Bare_Incomplete_Formal_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Incomplete_Formal_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Incomplete_Formal_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Incomplete_Formal_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Incomplete_Tagged_Type_Decl
        (Node : Bare_Incomplete_Tagged_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Incomplete_Tagged_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Incomplete_Tagged_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Incomplete_Tagged_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Num_Literal
        (Node : Bare_Num_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Num_Literal is
      begin
         if Node = null then
            return No_Entity_Num_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Num_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Int_Literal
        (Node : Bare_Int_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Int_Literal is
      begin
         if Node = null then
            return No_Entity_Int_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Int_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Interface_Kind
        (Node : Bare_Interface_Kind; Info : Internal_Entity_Info)
         return Internal_Entity_Interface_Kind is
      begin
         if Node = null then
            return No_Entity_Interface_Kind;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Interface_Kind) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Interface_Kind_Limited
        (Node : Bare_Interface_Kind_Limited; Info : Internal_Entity_Info)
         return Internal_Entity_Interface_Kind_Limited is
      begin
         if Node = null then
            return No_Entity_Interface_Kind_Limited;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Interface_Kind_Limited) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Interface_Kind_Protected
        (Node : Bare_Interface_Kind_Protected; Info : Internal_Entity_Info)
         return Internal_Entity_Interface_Kind_Protected is
      begin
         if Node = null then
            return No_Entity_Interface_Kind_Protected;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Interface_Kind_Protected) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Interface_Kind_Synchronized
        (Node : Bare_Interface_Kind_Synchronized; Info : Internal_Entity_Info)
         return Internal_Entity_Interface_Kind_Synchronized is
      begin
         if Node = null then
            return No_Entity_Interface_Kind_Synchronized;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Interface_Kind_Synchronized) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Interface_Kind_Task
        (Node : Bare_Interface_Kind_Task; Info : Internal_Entity_Info)
         return Internal_Entity_Interface_Kind_Task is
      begin
         if Node = null then
            return No_Entity_Interface_Kind_Task;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Interface_Kind_Task) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Interface_Type_Def
        (Node : Bare_Interface_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Interface_Type_Def is
      begin
         if Node = null then
            return No_Entity_Interface_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Interface_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Iter_Type
        (Node : Bare_Iter_Type; Info : Internal_Entity_Info)
         return Internal_Entity_Iter_Type is
      begin
         if Node = null then
            return No_Entity_Iter_Type;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Iter_Type) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Iter_Type_In
        (Node : Bare_Iter_Type_In; Info : Internal_Entity_Info)
         return Internal_Entity_Iter_Type_In is
      begin
         if Node = null then
            return No_Entity_Iter_Type_In;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Iter_Type_In) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Iter_Type_Of
        (Node : Bare_Iter_Type_Of; Info : Internal_Entity_Info)
         return Internal_Entity_Iter_Type_Of is
      begin
         if Node = null then
            return No_Entity_Iter_Type_Of;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Iter_Type_Of) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Iterated_Assoc
        (Node : Bare_Iterated_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Iterated_Assoc is
      begin
         if Node = null then
            return No_Entity_Iterated_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Iterated_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Known_Discriminant_Part
        (Node : Bare_Known_Discriminant_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Known_Discriminant_Part is
      begin
         if Node = null then
            return No_Entity_Known_Discriminant_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Known_Discriminant_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Label
        (Node : Bare_Label; Info : Internal_Entity_Info)
         return Internal_Entity_Label is
      begin
         if Node = null then
            return No_Entity_Label;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Label) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Label_Decl
        (Node : Bare_Label_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Label_Decl is
      begin
         if Node = null then
            return No_Entity_Label_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Label_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Library_Item
        (Node : Bare_Library_Item; Info : Internal_Entity_Info)
         return Internal_Entity_Library_Item is
      begin
         if Node = null then
            return No_Entity_Library_Item;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Library_Item) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Limited_Node
        (Node : Bare_Limited_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Limited_Node is
      begin
         if Node = null then
            return No_Entity_Limited_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Limited_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Limited_Absent
        (Node : Bare_Limited_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Limited_Absent is
      begin
         if Node = null then
            return No_Entity_Limited_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Limited_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Limited_Present
        (Node : Bare_Limited_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Limited_Present is
      begin
         if Node = null then
            return No_Entity_Limited_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Limited_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Loop_Stmt
        (Node : Bare_Loop_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Loop_Stmt is
      begin
         if Node = null then
            return No_Entity_Loop_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Loop_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Membership_Expr
        (Node : Bare_Membership_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Membership_Expr is
      begin
         if Node = null then
            return No_Entity_Membership_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Membership_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Mod_Int_Type_Def
        (Node : Bare_Mod_Int_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Mod_Int_Type_Def is
      begin
         if Node = null then
            return No_Entity_Mod_Int_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Mod_Int_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Mode
        (Node : Bare_Mode; Info : Internal_Entity_Info)
         return Internal_Entity_Mode is
      begin
         if Node = null then
            return No_Entity_Mode;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Mode) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Mode_Default
        (Node : Bare_Mode_Default; Info : Internal_Entity_Info)
         return Internal_Entity_Mode_Default is
      begin
         if Node = null then
            return No_Entity_Mode_Default;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Mode_Default) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Mode_In
        (Node : Bare_Mode_In; Info : Internal_Entity_Info)
         return Internal_Entity_Mode_In is
      begin
         if Node = null then
            return No_Entity_Mode_In;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Mode_In) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Mode_In_Out
        (Node : Bare_Mode_In_Out; Info : Internal_Entity_Info)
         return Internal_Entity_Mode_In_Out is
      begin
         if Node = null then
            return No_Entity_Mode_In_Out;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Mode_In_Out) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Mode_Out
        (Node : Bare_Mode_Out; Info : Internal_Entity_Info)
         return Internal_Entity_Mode_Out is
      begin
         if Node = null then
            return No_Entity_Mode_Out;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Mode_Out) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Multi_Abstract_State_Decl
        (Node : Bare_Multi_Abstract_State_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Multi_Abstract_State_Decl is
      begin
         if Node = null then
            return No_Entity_Multi_Abstract_State_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Multi_Abstract_State_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Multi_Dim_Array_Assoc
        (Node : Bare_Multi_Dim_Array_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Multi_Dim_Array_Assoc is
      begin
         if Node = null then
            return No_Entity_Multi_Dim_Array_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Multi_Dim_Array_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Name_List
        (Node : Bare_Name_List; Info : Internal_Entity_Info)
         return Internal_Entity_Name_List is
      begin
         if Node = null then
            return No_Entity_Name_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Name_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Named_Stmt
        (Node : Bare_Named_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Named_Stmt is
      begin
         if Node = null then
            return No_Entity_Named_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Named_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Named_Stmt_Decl
        (Node : Bare_Named_Stmt_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Named_Stmt_Decl is
      begin
         if Node = null then
            return No_Entity_Named_Stmt_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Named_Stmt_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_No_Type_Object_Renaming_Decl
        (Node : Bare_No_Type_Object_Renaming_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_No_Type_Object_Renaming_Decl is
      begin
         if Node = null then
            return No_Entity_No_Type_Object_Renaming_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_No_Type_Object_Renaming_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Not_Null
        (Node : Bare_Not_Null; Info : Internal_Entity_Info)
         return Internal_Entity_Not_Null is
      begin
         if Node = null then
            return No_Entity_Not_Null;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Not_Null) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Not_Null_Absent
        (Node : Bare_Not_Null_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Not_Null_Absent is
      begin
         if Node = null then
            return No_Entity_Not_Null_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Not_Null_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Not_Null_Present
        (Node : Bare_Not_Null_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Not_Null_Present is
      begin
         if Node = null then
            return No_Entity_Not_Null_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Not_Null_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Null_Component_Decl
        (Node : Bare_Null_Component_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Component_Decl is
      begin
         if Node = null then
            return No_Entity_Null_Component_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Component_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Null_Literal
        (Node : Bare_Null_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Literal is
      begin
         if Node = null then
            return No_Entity_Null_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Null_Record_Aggregate
        (Node : Bare_Null_Record_Aggregate; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Record_Aggregate is
      begin
         if Node = null then
            return No_Entity_Null_Record_Aggregate;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Record_Aggregate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Null_Record_Def
        (Node : Bare_Null_Record_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Record_Def is
      begin
         if Node = null then
            return No_Entity_Null_Record_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Record_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Null_Stmt
        (Node : Bare_Null_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Stmt is
      begin
         if Node = null then
            return No_Entity_Null_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Null_Subp_Decl
        (Node : Bare_Null_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Null_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Number_Decl
        (Node : Bare_Number_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Number_Decl is
      begin
         if Node = null then
            return No_Entity_Number_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Number_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op
        (Node : Bare_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Op is
      begin
         if Node = null then
            return No_Entity_Op;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Abs
        (Node : Bare_Op_Abs; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Abs is
      begin
         if Node = null then
            return No_Entity_Op_Abs;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Abs) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_And
        (Node : Bare_Op_And; Info : Internal_Entity_Info)
         return Internal_Entity_Op_And is
      begin
         if Node = null then
            return No_Entity_Op_And;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_And) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_And_Then
        (Node : Bare_Op_And_Then; Info : Internal_Entity_Info)
         return Internal_Entity_Op_And_Then is
      begin
         if Node = null then
            return No_Entity_Op_And_Then;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_And_Then) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Concat
        (Node : Bare_Op_Concat; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Concat is
      begin
         if Node = null then
            return No_Entity_Op_Concat;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Concat) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Div
        (Node : Bare_Op_Div; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Div is
      begin
         if Node = null then
            return No_Entity_Op_Div;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Div) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Double_Dot
        (Node : Bare_Op_Double_Dot; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Double_Dot is
      begin
         if Node = null then
            return No_Entity_Op_Double_Dot;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Double_Dot) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Eq
        (Node : Bare_Op_Eq; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Eq is
      begin
         if Node = null then
            return No_Entity_Op_Eq;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Eq) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Gt
        (Node : Bare_Op_Gt; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Gt is
      begin
         if Node = null then
            return No_Entity_Op_Gt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Gt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Gte
        (Node : Bare_Op_Gte; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Gte is
      begin
         if Node = null then
            return No_Entity_Op_Gte;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Gte) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_In
        (Node : Bare_Op_In; Info : Internal_Entity_Info)
         return Internal_Entity_Op_In is
      begin
         if Node = null then
            return No_Entity_Op_In;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_In) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Lt
        (Node : Bare_Op_Lt; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Lt is
      begin
         if Node = null then
            return No_Entity_Op_Lt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Lt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Lte
        (Node : Bare_Op_Lte; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Lte is
      begin
         if Node = null then
            return No_Entity_Op_Lte;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Lte) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Minus
        (Node : Bare_Op_Minus; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Minus is
      begin
         if Node = null then
            return No_Entity_Op_Minus;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Minus) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Mod
        (Node : Bare_Op_Mod; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Mod is
      begin
         if Node = null then
            return No_Entity_Op_Mod;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Mod) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Mult
        (Node : Bare_Op_Mult; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Mult is
      begin
         if Node = null then
            return No_Entity_Op_Mult;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Mult) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Neq
        (Node : Bare_Op_Neq; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Neq is
      begin
         if Node = null then
            return No_Entity_Op_Neq;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Neq) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Not
        (Node : Bare_Op_Not; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Not is
      begin
         if Node = null then
            return No_Entity_Op_Not;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Not) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Not_In
        (Node : Bare_Op_Not_In; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Not_In is
      begin
         if Node = null then
            return No_Entity_Op_Not_In;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Not_In) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Or
        (Node : Bare_Op_Or; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Or is
      begin
         if Node = null then
            return No_Entity_Op_Or;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Or) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Or_Else
        (Node : Bare_Op_Or_Else; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Or_Else is
      begin
         if Node = null then
            return No_Entity_Op_Or_Else;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Or_Else) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Plus
        (Node : Bare_Op_Plus; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Plus is
      begin
         if Node = null then
            return No_Entity_Op_Plus;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Plus) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Pow
        (Node : Bare_Op_Pow; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Pow is
      begin
         if Node = null then
            return No_Entity_Op_Pow;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Pow) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Rem
        (Node : Bare_Op_Rem; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Rem is
      begin
         if Node = null then
            return No_Entity_Op_Rem;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Rem) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Op_Xor
        (Node : Bare_Op_Xor; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Xor is
      begin
         if Node = null then
            return No_Entity_Op_Xor;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Xor) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Ordinary_Fixed_Point_Def
        (Node : Bare_Ordinary_Fixed_Point_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Ordinary_Fixed_Point_Def is
      begin
         if Node = null then
            return No_Entity_Ordinary_Fixed_Point_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Ordinary_Fixed_Point_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Others_Designator
        (Node : Bare_Others_Designator; Info : Internal_Entity_Info)
         return Internal_Entity_Others_Designator is
      begin
         if Node = null then
            return No_Entity_Others_Designator;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Others_Designator) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Overriding_Node
        (Node : Bare_Overriding_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Overriding_Node is
      begin
         if Node = null then
            return No_Entity_Overriding_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Overriding_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Overriding_Not_Overriding
        (Node : Bare_Overriding_Not_Overriding; Info : Internal_Entity_Info)
         return Internal_Entity_Overriding_Not_Overriding is
      begin
         if Node = null then
            return No_Entity_Overriding_Not_Overriding;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Overriding_Not_Overriding) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Overriding_Overriding
        (Node : Bare_Overriding_Overriding; Info : Internal_Entity_Info)
         return Internal_Entity_Overriding_Overriding is
      begin
         if Node = null then
            return No_Entity_Overriding_Overriding;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Overriding_Overriding) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Overriding_Unspecified
        (Node : Bare_Overriding_Unspecified; Info : Internal_Entity_Info)
         return Internal_Entity_Overriding_Unspecified is
      begin
         if Node = null then
            return No_Entity_Overriding_Unspecified;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Overriding_Unspecified) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Package_Body
        (Node : Bare_Package_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Package_Body is
      begin
         if Node = null then
            return No_Entity_Package_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Package_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Package_Body_Stub
        (Node : Bare_Package_Body_Stub; Info : Internal_Entity_Info)
         return Internal_Entity_Package_Body_Stub is
      begin
         if Node = null then
            return No_Entity_Package_Body_Stub;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Package_Body_Stub) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Package_Decl
        (Node : Bare_Package_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Package_Decl is
      begin
         if Node = null then
            return No_Entity_Package_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Package_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Package_Renaming_Decl
        (Node : Bare_Package_Renaming_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Package_Renaming_Decl is
      begin
         if Node = null then
            return No_Entity_Package_Renaming_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Package_Renaming_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Param_Assoc
        (Node : Bare_Param_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Param_Assoc is
      begin
         if Node = null then
            return No_Entity_Param_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Param_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Param_Spec
        (Node : Bare_Param_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Param_Spec is
      begin
         if Node = null then
            return No_Entity_Param_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Param_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Param_Spec_List
        (Node : Bare_Param_Spec_List; Info : Internal_Entity_Info)
         return Internal_Entity_Param_Spec_List is
      begin
         if Node = null then
            return No_Entity_Param_Spec_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Param_Spec_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Params
        (Node : Bare_Params; Info : Internal_Entity_Info)
         return Internal_Entity_Params is
      begin
         if Node = null then
            return No_Entity_Params;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Params) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Paren_Abstract_State_Decl
        (Node : Bare_Paren_Abstract_State_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Paren_Abstract_State_Decl is
      begin
         if Node = null then
            return No_Entity_Paren_Abstract_State_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Paren_Abstract_State_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Paren_Expr
        (Node : Bare_Paren_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Paren_Expr is
      begin
         if Node = null then
            return No_Entity_Paren_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Paren_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Parent_List
        (Node : Bare_Parent_List; Info : Internal_Entity_Info)
         return Internal_Entity_Parent_List is
      begin
         if Node = null then
            return No_Entity_Parent_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Parent_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pp_Directive
        (Node : Bare_Pp_Directive; Info : Internal_Entity_Info)
         return Internal_Entity_Pp_Directive is
      begin
         if Node = null then
            return No_Entity_Pp_Directive;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pp_Directive) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pp_Else_Directive
        (Node : Bare_Pp_Else_Directive; Info : Internal_Entity_Info)
         return Internal_Entity_Pp_Else_Directive is
      begin
         if Node = null then
            return No_Entity_Pp_Else_Directive;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pp_Else_Directive) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pp_Elsif_Directive
        (Node : Bare_Pp_Elsif_Directive; Info : Internal_Entity_Info)
         return Internal_Entity_Pp_Elsif_Directive is
      begin
         if Node = null then
            return No_Entity_Pp_Elsif_Directive;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pp_Elsif_Directive) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pp_End_If_Directive
        (Node : Bare_Pp_End_If_Directive; Info : Internal_Entity_Info)
         return Internal_Entity_Pp_End_If_Directive is
      begin
         if Node = null then
            return No_Entity_Pp_End_If_Directive;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pp_End_If_Directive) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pp_If_Directive
        (Node : Bare_Pp_If_Directive; Info : Internal_Entity_Info)
         return Internal_Entity_Pp_If_Directive is
      begin
         if Node = null then
            return No_Entity_Pp_If_Directive;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pp_If_Directive) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pp_Then_Kw
        (Node : Bare_Pp_Then_Kw; Info : Internal_Entity_Info)
         return Internal_Entity_Pp_Then_Kw is
      begin
         if Node = null then
            return No_Entity_Pp_Then_Kw;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pp_Then_Kw) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pragma_Argument_Assoc
        (Node : Bare_Pragma_Argument_Assoc; Info : Internal_Entity_Info)
         return Internal_Entity_Pragma_Argument_Assoc is
      begin
         if Node = null then
            return No_Entity_Pragma_Argument_Assoc;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pragma_Argument_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pragma_Node
        (Node : Bare_Pragma_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Pragma_Node is
      begin
         if Node = null then
            return No_Entity_Pragma_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pragma_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Pragma_Node_List
        (Node : Bare_Pragma_Node_List; Info : Internal_Entity_Info)
         return Internal_Entity_Pragma_Node_List is
      begin
         if Node = null then
            return No_Entity_Pragma_Node_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pragma_Node_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Private_Node
        (Node : Bare_Private_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Private_Node is
      begin
         if Node = null then
            return No_Entity_Private_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Private_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Private_Absent
        (Node : Bare_Private_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Private_Absent is
      begin
         if Node = null then
            return No_Entity_Private_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Private_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Private_Part
        (Node : Bare_Private_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Private_Part is
      begin
         if Node = null then
            return No_Entity_Private_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Private_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Private_Present
        (Node : Bare_Private_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Private_Present is
      begin
         if Node = null then
            return No_Entity_Private_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Private_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Private_Type_Def
        (Node : Bare_Private_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Private_Type_Def is
      begin
         if Node = null then
            return No_Entity_Private_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Private_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Node
        (Node : Bare_Protected_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Node is
      begin
         if Node = null then
            return No_Entity_Protected_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Absent
        (Node : Bare_Protected_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Absent is
      begin
         if Node = null then
            return No_Entity_Protected_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Body
        (Node : Bare_Protected_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Body is
      begin
         if Node = null then
            return No_Entity_Protected_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Body_Stub
        (Node : Bare_Protected_Body_Stub; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Body_Stub is
      begin
         if Node = null then
            return No_Entity_Protected_Body_Stub;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Body_Stub) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Def
        (Node : Bare_Protected_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Def is
      begin
         if Node = null then
            return No_Entity_Protected_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Present
        (Node : Bare_Protected_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Present is
      begin
         if Node = null then
            return No_Entity_Protected_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Protected_Type_Decl
        (Node : Bare_Protected_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Protected_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Protected_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Protected_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Public_Part
        (Node : Bare_Public_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Public_Part is
      begin
         if Node = null then
            return No_Entity_Public_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Public_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Qual_Expr
        (Node : Bare_Qual_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Qual_Expr is
      begin
         if Node = null then
            return No_Entity_Qual_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Qual_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Quantified_Expr
        (Node : Bare_Quantified_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Quantified_Expr is
      begin
         if Node = null then
            return No_Entity_Quantified_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Quantified_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Quantifier
        (Node : Bare_Quantifier; Info : Internal_Entity_Info)
         return Internal_Entity_Quantifier is
      begin
         if Node = null then
            return No_Entity_Quantifier;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Quantifier) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Quantifier_All
        (Node : Bare_Quantifier_All; Info : Internal_Entity_Info)
         return Internal_Entity_Quantifier_All is
      begin
         if Node = null then
            return No_Entity_Quantifier_All;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Quantifier_All) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Quantifier_Some
        (Node : Bare_Quantifier_Some; Info : Internal_Entity_Info)
         return Internal_Entity_Quantifier_Some is
      begin
         if Node = null then
            return No_Entity_Quantifier_Some;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Quantifier_Some) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Raise_Expr
        (Node : Bare_Raise_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Raise_Expr is
      begin
         if Node = null then
            return No_Entity_Raise_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Raise_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Raise_Stmt
        (Node : Bare_Raise_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Raise_Stmt is
      begin
         if Node = null then
            return No_Entity_Raise_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Raise_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Range_Constraint
        (Node : Bare_Range_Constraint; Info : Internal_Entity_Info)
         return Internal_Entity_Range_Constraint is
      begin
         if Node = null then
            return No_Entity_Range_Constraint;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Range_Constraint) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Range_Spec
        (Node : Bare_Range_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Range_Spec is
      begin
         if Node = null then
            return No_Entity_Range_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Range_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Real_Literal
        (Node : Bare_Real_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Real_Literal is
      begin
         if Node = null then
            return No_Entity_Real_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Real_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Record_Def
        (Node : Bare_Record_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Record_Def is
      begin
         if Node = null then
            return No_Entity_Record_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Record_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Record_Rep_Clause
        (Node : Bare_Record_Rep_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Record_Rep_Clause is
      begin
         if Node = null then
            return No_Entity_Record_Rep_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Record_Rep_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Record_Type_Def
        (Node : Bare_Record_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Record_Type_Def is
      begin
         if Node = null then
            return No_Entity_Record_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Record_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Reduce_Attribute_Ref
        (Node : Bare_Reduce_Attribute_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Reduce_Attribute_Ref is
      begin
         if Node = null then
            return No_Entity_Reduce_Attribute_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Reduce_Attribute_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Relation_Op
        (Node : Bare_Relation_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Relation_Op is
      begin
         if Node = null then
            return No_Entity_Relation_Op;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Relation_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Renaming_Clause
        (Node : Bare_Renaming_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Renaming_Clause is
      begin
         if Node = null then
            return No_Entity_Renaming_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Renaming_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Requeue_Stmt
        (Node : Bare_Requeue_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Requeue_Stmt is
      begin
         if Node = null then
            return No_Entity_Requeue_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Requeue_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Return_Stmt
        (Node : Bare_Return_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Return_Stmt is
      begin
         if Node = null then
            return No_Entity_Return_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Return_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Reverse_Node
        (Node : Bare_Reverse_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Reverse_Node is
      begin
         if Node = null then
            return No_Entity_Reverse_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Reverse_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Reverse_Absent
        (Node : Bare_Reverse_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Reverse_Absent is
      begin
         if Node = null then
            return No_Entity_Reverse_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Reverse_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Reverse_Present
        (Node : Bare_Reverse_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Reverse_Present is
      begin
         if Node = null then
            return No_Entity_Reverse_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Reverse_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Select_Stmt
        (Node : Bare_Select_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Select_Stmt is
      begin
         if Node = null then
            return No_Entity_Select_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Select_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Select_When_Part
        (Node : Bare_Select_When_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Select_When_Part is
      begin
         if Node = null then
            return No_Entity_Select_When_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Select_When_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Select_When_Part_List
        (Node : Bare_Select_When_Part_List; Info : Internal_Entity_Info)
         return Internal_Entity_Select_When_Part_List is
      begin
         if Node = null then
            return No_Entity_Select_When_Part_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Select_When_Part_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Signed_Int_Type_Def
        (Node : Bare_Signed_Int_Type_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Signed_Int_Type_Def is
      begin
         if Node = null then
            return No_Entity_Signed_Int_Type_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Signed_Int_Type_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Simple_Decl_Stmt
        (Node : Bare_Simple_Decl_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_Simple_Decl_Stmt is
      begin
         if Node = null then
            return No_Entity_Simple_Decl_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Simple_Decl_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Single_Protected_Decl
        (Node : Bare_Single_Protected_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Single_Protected_Decl is
      begin
         if Node = null then
            return No_Entity_Single_Protected_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Single_Protected_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Single_Task_Decl
        (Node : Bare_Single_Task_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Single_Task_Decl is
      begin
         if Node = null then
            return No_Entity_Single_Task_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Single_Task_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Task_Type_Decl
        (Node : Bare_Task_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Task_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Task_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Task_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Single_Task_Type_Decl
        (Node : Bare_Single_Task_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Single_Task_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Single_Task_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Single_Task_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Stmt_List
        (Node : Bare_Stmt_List; Info : Internal_Entity_Info)
         return Internal_Entity_Stmt_List is
      begin
         if Node = null then
            return No_Entity_Stmt_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Stmt_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_String_Literal
        (Node : Bare_String_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_String_Literal is
      begin
         if Node = null then
            return No_Entity_String_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_String_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Body
        (Node : Bare_Subp_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Body is
      begin
         if Node = null then
            return No_Entity_Subp_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Body_Stub
        (Node : Bare_Subp_Body_Stub; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Body_Stub is
      begin
         if Node = null then
            return No_Entity_Subp_Body_Stub;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Body_Stub) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Decl
        (Node : Bare_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Kind
        (Node : Bare_Subp_Kind; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Kind is
      begin
         if Node = null then
            return No_Entity_Subp_Kind;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Kind) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Kind_Function
        (Node : Bare_Subp_Kind_Function; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Kind_Function is
      begin
         if Node = null then
            return No_Entity_Subp_Kind_Function;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Kind_Function) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Kind_Procedure
        (Node : Bare_Subp_Kind_Procedure; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Kind_Procedure is
      begin
         if Node = null then
            return No_Entity_Subp_Kind_Procedure;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Kind_Procedure) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Renaming_Decl
        (Node : Bare_Subp_Renaming_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Renaming_Decl is
      begin
         if Node = null then
            return No_Entity_Subp_Renaming_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Renaming_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subp_Spec
        (Node : Bare_Subp_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Subp_Spec is
      begin
         if Node = null then
            return No_Entity_Subp_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subp_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subtype_Decl
        (Node : Bare_Subtype_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Subtype_Decl is
      begin
         if Node = null then
            return No_Entity_Subtype_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subtype_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Subunit
        (Node : Bare_Subunit; Info : Internal_Entity_Info)
         return Internal_Entity_Subunit is
      begin
         if Node = null then
            return No_Entity_Subunit;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subunit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synchronized_Node
        (Node : Bare_Synchronized_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Synchronized_Node is
      begin
         if Node = null then
            return No_Entity_Synchronized_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synchronized_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synchronized_Absent
        (Node : Bare_Synchronized_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Synchronized_Absent is
      begin
         if Node = null then
            return No_Entity_Synchronized_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synchronized_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synchronized_Present
        (Node : Bare_Synchronized_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Synchronized_Present is
      begin
         if Node = null then
            return No_Entity_Synchronized_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synchronized_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synth_Anonymous_Type_Decl
        (Node : Bare_Synth_Anonymous_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Synth_Anonymous_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Synth_Anonymous_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synth_Anonymous_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Binary_Spec
        (Node : Bare_Synthetic_Binary_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Binary_Spec is
      begin
         if Node = null then
            return No_Entity_Synthetic_Binary_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Binary_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Char_Enum_Lit
        (Node : Bare_Synthetic_Char_Enum_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Char_Enum_Lit is
      begin
         if Node = null then
            return No_Entity_Synthetic_Char_Enum_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Char_Enum_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Defining_Name
        (Node : Bare_Synthetic_Defining_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Defining_Name is
      begin
         if Node = null then
            return No_Entity_Synthetic_Defining_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Defining_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Formal_Param_Decl
        (Node : Bare_Synthetic_Formal_Param_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Formal_Param_Decl is
      begin
         if Node = null then
            return No_Entity_Synthetic_Formal_Param_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Formal_Param_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Identifier
        (Node : Bare_Synthetic_Identifier; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Identifier is
      begin
         if Node = null then
            return No_Entity_Synthetic_Identifier;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Identifier) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Object_Decl
        (Node : Bare_Synthetic_Object_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Object_Decl is
      begin
         if Node = null then
            return No_Entity_Synthetic_Object_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Object_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Renaming_Clause
        (Node : Bare_Synthetic_Renaming_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Renaming_Clause is
      begin
         if Node = null then
            return No_Entity_Synthetic_Renaming_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Renaming_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Subp_Decl
        (Node : Bare_Synthetic_Subp_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Subp_Decl is
      begin
         if Node = null then
            return No_Entity_Synthetic_Subp_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Subp_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Type_Expr
        (Node : Bare_Synthetic_Type_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Type_Expr is
      begin
         if Node = null then
            return No_Entity_Synthetic_Type_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Type_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Synthetic_Unary_Spec
        (Node : Bare_Synthetic_Unary_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Unary_Spec is
      begin
         if Node = null then
            return No_Entity_Synthetic_Unary_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Unary_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Tagged_Node
        (Node : Bare_Tagged_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Tagged_Node is
      begin
         if Node = null then
            return No_Entity_Tagged_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Tagged_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Tagged_Absent
        (Node : Bare_Tagged_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Tagged_Absent is
      begin
         if Node = null then
            return No_Entity_Tagged_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Tagged_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Tagged_Present
        (Node : Bare_Tagged_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Tagged_Present is
      begin
         if Node = null then
            return No_Entity_Tagged_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Tagged_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Target_Name
        (Node : Bare_Target_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Target_Name is
      begin
         if Node = null then
            return No_Entity_Target_Name;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Target_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Task_Body
        (Node : Bare_Task_Body; Info : Internal_Entity_Info)
         return Internal_Entity_Task_Body is
      begin
         if Node = null then
            return No_Entity_Task_Body;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Task_Body) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Task_Body_Stub
        (Node : Bare_Task_Body_Stub; Info : Internal_Entity_Info)
         return Internal_Entity_Task_Body_Stub is
      begin
         if Node = null then
            return No_Entity_Task_Body_Stub;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Task_Body_Stub) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Task_Def
        (Node : Bare_Task_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Task_Def is
      begin
         if Node = null then
            return No_Entity_Task_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Task_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Terminate_Alternative
        (Node : Bare_Terminate_Alternative; Info : Internal_Entity_Info)
         return Internal_Entity_Terminate_Alternative is
      begin
         if Node = null then
            return No_Entity_Terminate_Alternative;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Terminate_Alternative) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Type_Access_Def
        (Node : Bare_Type_Access_Def; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Access_Def is
      begin
         if Node = null then
            return No_Entity_Type_Access_Def;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Access_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Type_Attributes_Repository
        (Node : Bare_Type_Attributes_Repository; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Attributes_Repository is
      begin
         if Node = null then
            return No_Entity_Type_Attributes_Repository;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Attributes_Repository) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Un_Op
        (Node : Bare_Un_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Un_Op is
      begin
         if Node = null then
            return No_Entity_Un_Op;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Un_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Unconstrained_Array_Index
        (Node : Bare_Unconstrained_Array_Index; Info : Internal_Entity_Info)
         return Internal_Entity_Unconstrained_Array_Index is
      begin
         if Node = null then
            return No_Entity_Unconstrained_Array_Index;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Unconstrained_Array_Index) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Unconstrained_Array_Index_List
        (Node : Bare_Unconstrained_Array_Index_List; Info : Internal_Entity_Info)
         return Internal_Entity_Unconstrained_Array_Index_List is
      begin
         if Node = null then
            return No_Entity_Unconstrained_Array_Index_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Unconstrained_Array_Index_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Unconstrained_Array_Indices
        (Node : Bare_Unconstrained_Array_Indices; Info : Internal_Entity_Info)
         return Internal_Entity_Unconstrained_Array_Indices is
      begin
         if Node = null then
            return No_Entity_Unconstrained_Array_Indices;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Unconstrained_Array_Indices) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Unknown_Discriminant_Part
        (Node : Bare_Unknown_Discriminant_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Unknown_Discriminant_Part is
      begin
         if Node = null then
            return No_Entity_Unknown_Discriminant_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Unknown_Discriminant_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Until_Node
        (Node : Bare_Until_Node; Info : Internal_Entity_Info)
         return Internal_Entity_Until_Node is
      begin
         if Node = null then
            return No_Entity_Until_Node;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Until_Node) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Until_Absent
        (Node : Bare_Until_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Until_Absent is
      begin
         if Node = null then
            return No_Entity_Until_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Until_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Until_Present
        (Node : Bare_Until_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Until_Present is
      begin
         if Node = null then
            return No_Entity_Until_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Until_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Update_Attribute_Ref
        (Node : Bare_Update_Attribute_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Update_Attribute_Ref is
      begin
         if Node = null then
            return No_Entity_Update_Attribute_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Update_Attribute_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Use_Clause
        (Node : Bare_Use_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Use_Clause is
      begin
         if Node = null then
            return No_Entity_Use_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Use_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Use_Package_Clause
        (Node : Bare_Use_Package_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Use_Package_Clause is
      begin
         if Node = null then
            return No_Entity_Use_Package_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Use_Package_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Use_Type_Clause
        (Node : Bare_Use_Type_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_Use_Type_Clause is
      begin
         if Node = null then
            return No_Entity_Use_Type_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Use_Type_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Value_Sequence
        (Node : Bare_Value_Sequence; Info : Internal_Entity_Info)
         return Internal_Entity_Value_Sequence is
      begin
         if Node = null then
            return No_Entity_Value_Sequence;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Value_Sequence) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Variant
        (Node : Bare_Variant; Info : Internal_Entity_Info)
         return Internal_Entity_Variant is
      begin
         if Node = null then
            return No_Entity_Variant;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Variant) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Variant_List
        (Node : Bare_Variant_List; Info : Internal_Entity_Info)
         return Internal_Entity_Variant_List is
      begin
         if Node = null then
            return No_Entity_Variant_List;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Variant_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_Variant_Part
        (Node : Bare_Variant_Part; Info : Internal_Entity_Info)
         return Internal_Entity_Variant_Part is
      begin
         if Node = null then
            return No_Entity_Variant_Part;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Variant_Part) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_While_Loop_Spec
        (Node : Bare_While_Loop_Spec; Info : Internal_Entity_Info)
         return Internal_Entity_While_Loop_Spec is
      begin
         if Node = null then
            return No_Entity_While_Loop_Spec;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_While_Loop_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_While_Loop_Stmt
        (Node : Bare_While_Loop_Stmt; Info : Internal_Entity_Info)
         return Internal_Entity_While_Loop_Stmt is
      begin
         if Node = null then
            return No_Entity_While_Loop_Stmt;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_While_Loop_Stmt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_With_Clause
        (Node : Bare_With_Clause; Info : Internal_Entity_Info)
         return Internal_Entity_With_Clause is
      begin
         if Node = null then
            return No_Entity_With_Clause;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_With_Clause) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_With_Private
        (Node : Bare_With_Private; Info : Internal_Entity_Info)
         return Internal_Entity_With_Private is
      begin
         if Node = null then
            return No_Entity_With_Private;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_With_Private) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_With_Private_Absent
        (Node : Bare_With_Private_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_With_Private_Absent is
      begin
         if Node = null then
            return No_Entity_With_Private_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_With_Private_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      function Create_Internal_Entity_With_Private_Present
        (Node : Bare_With_Private_Present; Info : Internal_Entity_Info)
         return Internal_Entity_With_Private_Present is
      begin
         if Node = null then
            return No_Entity_With_Private_Present;
         end if;
         return (Node => Node, Info => Info);
      end;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_With_Private_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Env_Assoc) is
      begin
               Inc_Ref (R.Dest_Env);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Env_Assoc) is
      begin
               Dec_Ref (R.Dest_Env);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Env_Assoc) return Boolean is
      begin
         return L.Key = R.Key and then L.Value = R.Value and then Equivalent (L.Dest_Env, R.Dest_Env) and then L.Metadata = R.Metadata;
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Env_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Key => " & Trace_Image (R.Key)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
                        & ", "
                     & "Dest_Env => " & Trace_Image (R.Dest_Env)
                        & ", "
                     & "Metadata => " & Trace_Image (R.Metadata)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Eval_Discrete_Range) is
      begin
               Inc_Ref (R.Low_Bound);
               Inc_Ref (R.High_Bound);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Eval_Discrete_Range) is
      begin
               Dec_Ref (R.Low_Bound);
               Dec_Ref (R.High_Bound);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Eval_Discrete_Range) return Boolean is
      begin
         return Equivalent (L.Low_Bound, R.Low_Bound) and then Equivalent (L.High_Bound, R.High_Bound);
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Eval_Discrete_Range) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Low_Bound => " & Trace_Image (R.Low_Bound)
                        & ", "
                     & "High_Bound => " & Trace_Image (R.High_Bound)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Expected_Type_For_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Expected_Type => " & Trace_Image (R.Expected_Type)
                        & ", "
                     & "Expr => " & Trace_Image (R.Expr)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Inner_Env_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Key => " & Trace_Image (R.Key)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
                        & ", "
                     & "Rebindings => " & Trace_Image (R.Rebindings)
                        & ", "
                     & "Metadata => " & Trace_Image (R.Metadata)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Logic_Context) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Ref_Node => " & Trace_Image (R.Ref_Node)
                        & ", "
                     & "Decl_Node => " & Trace_Image (R.Decl_Node)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Logic_Val_Result) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Success => " & Trace_Image (R.Success)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Multidim_Aggregate_Info) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Agg => " & Trace_Image (R.Agg)
                        & ", "
                     & "Typ => " & Trace_Image (R.Typ)
                        & ", "
                     & "Rank => " & Trace_Image (R.Rank)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Param_Actual) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Param => " & Trace_Image (R.Param)
                        & ", "
                     & "Actual => " & Trace_Image (R.Actual)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Single_Actual) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Name => " & Trace_Image (R.Name)
                        & ", "
                     & "Assoc => " & Trace_Image (R.Assoc)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Param_Match) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Has_Matched => " & Trace_Image (R.Has_Matched)
                        & ", "
                     & "Actual => " & Trace_Image (R.Actual)
                        & ", "
                     & "Formal => " & Trace_Image (R.Formal)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Ref_Result) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Ref => " & Trace_Image (R.Ref)
                        & ", "
                     & "Kind => " & Trace_Image (R.Kind)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Refd_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Decl => " & Trace_Image (R.Decl)
                        & ", "
                     & "Kind => " & Trace_Image (R.Kind)
               & ")");
      end Trace_Image;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Refd_Def) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Def_Name => " & Trace_Image (R.Def_Name)
                        & ", "
                     & "Kind => " & Trace_Image (R.Kind)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Shape) is
      begin
               Inc_Ref (R.Components);
               Inc_Ref (R.Discriminants_Values);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Shape) is
      begin
               Dec_Ref (R.Components);
               Dec_Ref (R.Discriminants_Values);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Shape) return Boolean is
      begin
         return Equivalent (L.Components, R.Components) and then Equivalent (L.Discriminants_Values, R.Discriminants_Values);
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Shape) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Components => " & Trace_Image (R.Components)
                        & ", "
                     & "Discriminants_Values => " & Trace_Image (R.Discriminants_Values)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Solver_Diagnostic) is
      begin
               Inc_Ref (R.Message_Template);
               Inc_Ref (R.Args);
               Inc_Ref (R.Contexts);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Solver_Diagnostic) is
      begin
               Dec_Ref (R.Message_Template);
               Dec_Ref (R.Args);
               Dec_Ref (R.Contexts);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Solver_Diagnostic) return Boolean is
      begin
         return Equivalent (L.Message_Template, R.Message_Template) and then Equivalent (L.Args, R.Args) and then L.Location = R.Location and then Equivalent (L.Contexts, R.Contexts) and then L.Round = R.Round;
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Solver_Diagnostic) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Message_Template => " & Trace_Image (R.Message_Template)
                        & ", "
                     & "Args => " & Trace_Image (R.Args)
                        & ", "
                     & "Location => " & Trace_Image (R.Location)
                        & ", "
                     & "Contexts => " & Trace_Image (R.Contexts)
                        & ", "
                     & "Round => " & Trace_Image (R.Round)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Solver_Result) is
      begin
               Inc_Ref (R.Diagnostics);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Solver_Result) is
      begin
               Dec_Ref (R.Diagnostics);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Solver_Result) return Boolean is
      begin
         return L.Success = R.Success and then Equivalent (L.Diagnostics, R.Diagnostics);
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Solver_Result) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Success => " & Trace_Image (R.Success)
                        & ", "
                     & "Diagnostics => " & Trace_Image (R.Diagnostics)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_Substitution) is
      begin
               Inc_Ref (R.To_Value);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_Substitution) is
      begin
               Dec_Ref (R.To_Value);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_Substitution) return Boolean is
      begin
         return L.From_Decl = R.From_Decl and then Equivalent (L.To_Value, R.To_Value) and then L.Value_Type = R.Value_Type;
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Substitution) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "From_Decl => " & Trace_Image (R.From_Decl)
                        & ", "
                     & "To_Value => " & Trace_Image (R.To_Value)
                        & ", "
                     & "Value_Type => " & Trace_Image (R.Value_Type)
               & ")");
      end Trace_Image;
      -------------
      -- Inc_Ref --
      -------------
      procedure Inc_Ref (R : Internal_User_Defined_Function_Subp_Spec) is
      begin
               Inc_Ref (R.Subp_Params_Types);
      end Inc_Ref;
      -------------
      -- Dec_Ref --
      -------------
      procedure Dec_Ref (R : in out Internal_User_Defined_Function_Subp_Spec) is
      begin
               Dec_Ref (R.Subp_Params_Types);
      end Dec_Ref;
      ----------------
      -- Equivalent --
      ----------------
      function Equivalent (L, R : Internal_User_Defined_Function_Subp_Spec) return Boolean is
      begin
         return Equivalent (L.Subp_Params_Types, R.Subp_Params_Types) and then L.Subp_Return_Type = R.Subp_Return_Type;
      end Equivalent;
      -----------------
      -- Trace_Image --
      -----------------
      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_User_Defined_Function_Subp_Spec) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Subp_Params_Types => " & Trace_Image (R.Subp_Params_Types)
                        & ", "
                     & "Subp_Return_Type => " & Trace_Image (R.Subp_Return_Type)
               & ")");
      end Trace_Image;
   type Ada_Node_P_Is_Not_Null_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Ada_Node_P_Is_Not_Null_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Ada_Node_P_Is_Not_Null_0_Predicate) return String;
   function Create_Ada_Node_P_Is_Not_Null_0_Predicate
      return Ada_Node_P_Is_Not_Null_0_Predicate is
   begin
      return Ada_Node_P_Is_Not_Null_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Ada_Node_P_Is_Not_Null_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : Bare_Ada_Node;
   begin
      Node := Entity.Node;
      return Ada_Node_P_Is_Not_Null (Node, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Ada_Node_P_Is_Not_Null_0_Predicate) return String is
   begin
      return "AdaNode.is_not_null";
   end Image;
   type Ada_Node_P_Missing_Entity_Error_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Ada_Node_P_Missing_Entity_Error_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
      overriding procedure Failed
        (Self : Ada_Node_P_Missing_Entity_Error_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;
   overriding function Image (Self : Ada_Node_P_Missing_Entity_Error_0_Predicate) return String;
   function Create_Ada_Node_P_Missing_Entity_Error_0_Predicate
   (
         Error_Location : Bare_Ada_Node
   )
      return Ada_Node_P_Missing_Entity_Error_0_Predicate is
   begin
      return Ada_Node_P_Missing_Entity_Error_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Ada_Node_P_Missing_Entity_Error_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : Bare_Ada_Node;
   begin
      Node := Entity.Node;
      return Ada_Node_P_Missing_Entity_Error (Node);
   end Call;
   ------------
   -- Failed --
   ------------
      overriding procedure Failed
        (Self : Ada_Node_P_Missing_Entity_Error_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   is
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (0);
      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);
      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("no such entity"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Ada_Node_P_Missing_Entity_Error_0_Predicate) return String is
   begin
      return "AdaNode.missing_entity_error";
   end Image;
   type Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Internal_Entity_Basic_Decl;
         Field_1 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate) return String;
   function Create_Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate
   (
         Field_0 : Internal_Entity_Basic_Decl;
         Field_1 : Bare_Ada_Node
   )
      return Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate is
   begin
      return Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Field_1 => Field_1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Basic_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Basic_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Basic_Decl_P_Subp_Decl_Match_Signature (Node, Other => Self.Field_0, Origin => Self.Field_1, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Basic_Decl_P_Subp_Decl_Match_Signature_0_Predicate) return String is
   begin
      return "BasicDecl.subp_decl_match_signature";
   end Image;
   type Basic_Decl_P_Access_To_Subp_Decl_Match_Signature_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Internal_Entity_Basic_Decl;
         Field_1 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Basic_Decl_P_Access_To_Subp_Decl_Match_Signature_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Basic_Decl_P_Access_To_Subp_Decl_Match_Signature_0_Predicate) return String;
   function Create_Basic_Decl_P_Access_To_Subp_Decl_Match_Signature_0_Predicate
   (
         Field_0 : Internal_Entity_Basic_Decl;
         Field_1 : Bare_Ada_Node
   )
      return Basic_Decl_P_Access_To_Subp_Decl_Match_Signature_0_Predicate is
   begin
      return Basic_Decl_P_Access_To_Subp_Decl_Match_Signature_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Field_1 => Field_1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Basic_Decl_P_Access_To_Subp_Decl_Match_Signature_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Basic_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Basic_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Basic_Decl_P_Access_To_Subp_Decl_Match_Signature (Node, Other => Self.Field_0, Origin => Self.Field_1, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Basic_Decl_P_Access_To_Subp_Decl_Match_Signature_0_Predicate) return String is
   begin
      return "BasicDecl.access_to_subp_decl_match_signature";
   end Image;
   type Basic_Decl_P_Subp_Renaming_Decl_Match_Signature_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Internal_Entity_Basic_Decl;
         Field_1 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Basic_Decl_P_Subp_Renaming_Decl_Match_Signature_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
   overriding function Image (Self : Basic_Decl_P_Subp_Renaming_Decl_Match_Signature_0_Predicate) return String;
   function Create_Basic_Decl_P_Subp_Renaming_Decl_Match_Signature_0_Predicate
   (
         Field_0 : Internal_Entity_Basic_Decl;
         Field_1 : Bare_Ada_Node
   )
      return Basic_Decl_P_Subp_Renaming_Decl_Match_Signature_0_Predicate is
   begin
      return Basic_Decl_P_Subp_Renaming_Decl_Match_Signature_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Field_1 => Field_1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Basic_Decl_P_Subp_Renaming_Decl_Match_Signature_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : Bare_Basic_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Basic_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
               if Entities (2).Node /= null
                  and then Entities (2).Node.Kind
                           not in Ada_Basic_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Basic_Decl_P_Subp_Renaming_Decl_Match_Signature (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Other => Self.Field_0, Origin => Self.Field_1, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Basic_Decl_P_Subp_Renaming_Decl_Match_Signature_0_Predicate) return String is
   begin
      return "BasicDecl.subp_renaming_decl_match_signature";
   end Image;
   type Basic_Decl_P_Is_Subprogram_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Basic_Decl_P_Is_Subprogram_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Basic_Decl_P_Is_Subprogram_0_Predicate) return String;
   function Create_Basic_Decl_P_Is_Subprogram_0_Predicate
      return Basic_Decl_P_Is_Subprogram_0_Predicate is
   begin
      return Basic_Decl_P_Is_Subprogram_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Basic_Decl_P_Is_Subprogram_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : Bare_Basic_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Basic_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Basic_Decl_P_Is_Subprogram (Node);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Basic_Decl_P_Is_Subprogram_0_Predicate) return String is
   begin
      return "BasicDecl.is_subprogram";
   end Image;
   type Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Internal_Entity_Base_Type_Decl;
         Field_1 : Boolean;
   end record;
      overriding function Call
        (Self : Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate) return String;
   function Create_Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate
   (
         Field_0 : Internal_Entity_Base_Type_Decl;
         Field_1 : Boolean
   )
      return Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate is
   begin
      return Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Field_1 => Field_1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Basic_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Basic_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Basic_Decl_P_Is_Stream_Subprogram_For_Type (Node, Typ => Self.Field_0, Return_Obj => Self.Field_1, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Basic_Decl_P_Is_Stream_Subprogram_For_Type_0_Predicate) return String is
   begin
      return "BasicDecl.is_stream_subprogram_for_type";
   end Image;
   type Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Internal_Entity_Base_Type_Decl;
   end record;
      overriding function Call
        (Self : Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate) return String;
   function Create_Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate
   (
         Field_0 : Internal_Entity_Base_Type_Decl
   )
      return Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate is
   begin
      return Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Basic_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Basic_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Basic_Decl_P_Is_Put_Image_Subprogram_For_Type (Node, Typ => Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Basic_Decl_P_Is_Put_Image_Subprogram_For_Type_0_Predicate) return String is
   begin
      return "BasicDecl.is_put_image_subprogram_for_type";
   end Image;
   type Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Is_Array_Or_Rec (Node, Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Array_Or_Rec_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_array_or_rec";
   end Image;
   type Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Is_Non_Null_Char_Type (Node, Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Non_Null_Char_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_non_null_char_type";
   end Image;
   type Base_Type_Decl_P_Is_Universal_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Universal_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Universal_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Universal_Type_0_Predicate
      return Base_Type_Decl_P_Is_Universal_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Universal_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Universal_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Is_Universal_Type (Node, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Universal_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_universal_type";
   end Image;
   type Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate
      return Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Is_Not_Universal_Type (Node, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Not_Universal_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_not_universal_type";
   end Image;
   type Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Is_Access_Type_Predicate (Node, Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Access_Type_Predicate_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_access_type_predicate";
   end Image;
   type Base_Type_Decl_P_Is_Iterator_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Iterator_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Iterator_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Iterator_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Iterator_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Iterator_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Iterator_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Is_Iterator_Type (Node, Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Iterator_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_iterator_type";
   end Image;
   type Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate
      return Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Is_Not_Root_Int_Type (Node, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Not_Root_Int_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_not_root_int_type";
   end Image;
   type Base_Type_Decl_P_Is_Not_Any_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Error_Location : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Not_Any_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Is_Not_Any_Type_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Not_Any_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Not_Any_Type_0_Predicate
   (
         Error_Location : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Not_Any_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Not_Any_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Error_Location => Error_Location);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Not_Any_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Is_Not_Any_Type (Node);
   end Call;
   ------------
   -- Failed --
   ------------
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Is_Not_Any_Type_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   is
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (0);
      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);
      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("No inferable type for expression"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Not_Any_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_not_any_type";
   end Image;
   type Base_Type_Decl_P_Allows_String_Literal_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
         Error_Location : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Allows_String_Literal_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Allows_String_Literal_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;
   overriding function Image (Self : Base_Type_Decl_P_Allows_String_Literal_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Allows_String_Literal_0_Predicate
   (
         Field_0 : Bare_Ada_Node;
         Error_Location : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Allows_String_Literal_0_Predicate is
   begin
      return Base_Type_Decl_P_Allows_String_Literal_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Error_Location => Error_Location);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Allows_String_Literal_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Allows_String_Literal (Node, Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   ------------
   -- Failed --
   ------------
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Allows_String_Literal_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   is
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);
      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);
      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("{} does not allow string literals"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Allows_String_Literal_0_Predicate) return String is
   begin
      return "BaseTypeDecl.allows_string_literal";
   end Image;
   type Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
               if Entities (2).Node /= null
                  and then Entities (2).Node.Kind
                           not in Ada_Basic_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Is_Subp_Access_Of (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Subp_Access_Of_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_subp_access_of";
   end Image;
   type Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
         Error_Location : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate
   (
         Field_0 : Bare_Ada_Node;
         Error_Location : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Error_Location => Error_Location);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Is_Tagged_Type_With_Deref (Node, Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   ------------
   -- Failed --
   ------------
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   is
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);
      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);
      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("expected tagged type, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Tagged_Type_With_Deref_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_tagged_type_with_deref";
   end Image;
   type Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Is_Array_Def_With_Deref (Node, Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_array_def_with_deref";
   end Image;
   type Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null (Node, Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Array_Def_With_Deref_Or_Null_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_array_def_with_deref_or_null";
   end Image;
   type Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
               if Entities (2).Node /= null
                  and then Entities (2).Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Matching_Prefix_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Matching_Prefix_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.matching_prefix_type";
   end Image;
   type Base_Type_Decl_P_Matching_Formal_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
         Error_Location : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Formal_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Matching_Formal_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;
   overriding function Image (Self : Base_Type_Decl_P_Matching_Formal_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Matching_Formal_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node;
         Error_Location : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Matching_Formal_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Matching_Formal_Type_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Error_Location => Error_Location);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Formal_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
               if Entities (2).Node /= null
                  and then Entities (2).Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Matching_Formal_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   ------------
   -- Failed --
   ------------
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Matching_Formal_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   is
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (2);
      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);
      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("expected {}, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entities (2);
      Args.Items (2) := Entities (1);
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Matching_Formal_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.matching_formal_type";
   end Image;
   type Base_Type_Decl_P_Matching_Membership_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Membership_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Matching_Membership_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Matching_Membership_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Matching_Membership_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Matching_Membership_Type_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Membership_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
               if Entities (2).Node /= null
                  and then Entities (2).Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Matching_Membership_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Matching_Membership_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.matching_membership_type";
   end Image;
   type Base_Type_Decl_P_Matching_Assign_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
         Error_Location : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Assign_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Matching_Assign_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;
   overriding function Image (Self : Base_Type_Decl_P_Matching_Assign_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Matching_Assign_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node;
         Error_Location : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Matching_Assign_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Matching_Assign_Type_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Error_Location => Error_Location);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Assign_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
               if Entities (2).Node /= null
                  and then Entities (2).Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Matching_Assign_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   ------------
   -- Failed --
   ------------
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Matching_Assign_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   is
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (2);
      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);
      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("expected {}, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entities (2);
      Args.Items (2) := Entities (1);
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Matching_Assign_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.matching_assign_type";
   end Image;
   type Base_Type_Decl_P_Matching_Type_0_Predicate is
   new Solver_Ifc.N_Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
         Error_Location : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   ;
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;
   overriding function Image (Self : Base_Type_Decl_P_Matching_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Matching_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node;
         Error_Location : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Matching_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Matching_Type_0_Predicate'(N => 2, Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Error_Location => Error_Location);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array
        ) return Boolean
   is
         Entity : Internal_Entity := Entities (1);
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
               if Entities (2).Node /= null
                  and then Entities (2).Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Matching_Type (Node, (Node => Entities (2).Node, Info => Entities (2).Info), Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   ------------
   -- Failed --
   ------------
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Matching_Type_0_Predicate;
            Entities : Entity_Vars.Value_Array;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   is
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (2);
      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);
      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("expected {}, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entities (2);
      Args.Items (2) := Entities (1);
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Matching_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.matching_type";
   end Image;
   type Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Internal_Entity_Base_Type_Decl;
         Field_1 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate
   (
         Field_0 : Internal_Entity_Base_Type_Decl;
         Field_1 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Field_1 => Field_1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Matching_Allocator_Type (Node, Allocated_Type => Self.Field_0, Origin => Self.Field_1, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Matching_Allocator_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.matching_allocator_type";
   end Image;
   type Base_Type_Decl_P_Derives_From_Std_Bool_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
         Error_Location : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Derives_From_Std_Bool_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Derives_From_Std_Bool_Type_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   ;
   overriding function Image (Self : Base_Type_Decl_P_Derives_From_Std_Bool_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Derives_From_Std_Bool_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node;
         Error_Location : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Derives_From_Std_Bool_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Derives_From_Std_Bool_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0, Error_Location => Error_Location);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Derives_From_Std_Bool_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Base_Type_Decl_P_Derives_From_Std_Bool_Type (Node, Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   ------------
   -- Failed --
   ------------
      overriding procedure Failed
        (Self : Base_Type_Decl_P_Derives_From_Std_Bool_Type_0_Predicate;
            Entity : Internal_Entity;
         Ctxs    : Solver_Ifc.Logic_Context_Array;
         Round   : Natural;
         Emitter : Solver_Ifc.Diagnostic_Emitter
        )
   is
      Args : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (1);
      Contexts : Internal_Logic_Context_Array_Access :=
         Create_Internal_Logic_Context_Array (Ctxs'Length);
      Diag : constant Internal_Solver_Diagnostic :=
        (Message_Template => Create_String ("expected boolean type, got {}"),
         Args             => Args,
         Contexts         => Contexts,
         Location         => Self.Error_Location,
         Round            => Round);
   begin
      Args.Items (1) := Entity;
      for I in Ctxs'Range loop
         Contexts.Items (I) := Ctxs (I).all;
      end loop;
      Emitter (Diag);
   end Failed;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Derives_From_Std_Bool_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.derives_from_std_bool_type";
   end Image;
   type Base_Type_Decl_P_Is_Task_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         null;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Task_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Task_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Task_Type_0_Predicate
      return Base_Type_Decl_P_Is_Task_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Task_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Task_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
         pragma Unreferenced (Self);
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Dispatcher_Base_Type_Decl_P_Is_Task_Type (Node, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Task_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_task_type";
   end Image;
   type Base_Type_Decl_P_Is_Discrete_Type_0_Predicate is
   new Solver_Ifc.Predicate_Type
   with record
         Field_0 : Bare_Ada_Node;
   end record;
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Discrete_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   ;
   overriding function Image (Self : Base_Type_Decl_P_Is_Discrete_Type_0_Predicate) return String;
   function Create_Base_Type_Decl_P_Is_Discrete_Type_0_Predicate
   (
         Field_0 : Bare_Ada_Node
   )
      return Base_Type_Decl_P_Is_Discrete_Type_0_Predicate is
   begin
      return Base_Type_Decl_P_Is_Discrete_Type_0_Predicate'(Cache_Set => False, Cache_Key => <>, Cache_Value => <>, Ref_Count => 1, Field_0 => Field_0);
   end;
   ----------
   -- Call --
   ----------
      overriding function Call
        (Self : Base_Type_Decl_P_Is_Discrete_Type_0_Predicate;
            Entity : Internal_Entity
        ) return Boolean
   is
      Node : Bare_Base_Type_Decl;
   begin
               if Entity.Node /= null
                  and then Entity.Node.Kind
                           not in Ada_Base_Type_Decl
               then
                  Raise_Property_Exception
                    (Node, Property_Error'Identity, "mismatching node type");
               end if;
      Node := Entity.Node;
      return Dispatcher_Base_Type_Decl_P_Is_Discrete_Type (Node, Origin => Self.Field_0, E_Info => Entity.Info);
   end Call;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Base_Type_Decl_P_Is_Discrete_Type_0_Predicate) return String is
   begin
      return "BaseTypeDecl.is_discrete_type";
   end Image;
   type Logic_Functor_885 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_885; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_885) return String;
   overriding function Convert (Self : Logic_Functor_885; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Accessed_Type_No_Call (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_885) return String is
   begin
      return ("BaseTypeDecl.accessed_type_no_call");
   end Image;
   type Logic_Functor_843 is new Solver_Ifc.Converter_Type with
      null record;
   overriding function Convert (Self : Logic_Functor_843; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_843) return String;
   overriding function Convert (Self : Logic_Functor_843; From : Internal_Entity) return Internal_Entity is
         pragma Unreferenced (Self);
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Anonymous_Access_Type_Or_Null (Node => From.Node, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_843) return String is
   begin
      return ("BaseTypeDecl.anonymous_access_type_or_null");
   end Image;
   type Logic_Functor_867 is new Solver_Ifc.Combiner_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Combine (Self : Logic_Functor_867; Vals : Entity_Vars.Value_Array) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_867) return String;
   overriding function Combine (Self : Logic_Functor_867; Vals : Entity_Vars.Value_Array) return Internal_Entity is
         From : constant Internal_Entity := Vals (1);
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
            if Vals (2).Node /= null
               and then Vals (2).Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
      Ret := Base_Type_Decl_P_Array_Concat_Result_Type (Node => From.Node, Other => (Node => Vals (2).Node,  Info => Vals (2).Info), Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Combine;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_867) return String is
   begin
      return ("BaseTypeDecl.array_concat_result_type");
   end Image;
   type Logic_Functor_841 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_841; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_841) return String;
   overriding function Convert (Self : Logic_Functor_841; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Base_Subtype (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_841) return String is
   begin
      return ("BaseTypeDecl.base_subtype");
   end Image;
   type Logic_Functor_895 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_895; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_895) return String;
   overriding function Convert (Self : Logic_Functor_895; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Comp_Type (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_895) return String is
   begin
      return ("BaseTypeDecl.comp_type");
   end Image;
   type Logic_Functor_919 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_919; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_919) return String;
   overriding function Convert (Self : Logic_Functor_919; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Basic_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Corresponding_Char_Literal (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_919) return String is
   begin
      return ("BaseTypeDecl.corresponding_char_literal");
   end Image;
   type Logic_Functor_878 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_878; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_878) return String;
   overriding function Convert (Self : Logic_Functor_878; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Cursor_Type (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_878) return String is
   begin
      return ("BaseTypeDecl.cursor_type");
   end Image;
   type Logic_Functor_866 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_866; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_866) return String;
   overriding function Convert (Self : Logic_Functor_866; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Derefed_Base_Subtype (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_866) return String is
   begin
      return ("BaseTypeDecl.derefed_base_subtype");
   end Image;
   type Logic_Functor_865 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_865; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_865) return String;
   overriding function Convert (Self : Logic_Functor_865; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Derefed_Type (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_865) return String is
   begin
      return ("BaseTypeDecl.derefed_type");
   end Image;
   type Logic_Functor_868 is new Solver_Ifc.Combiner_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Combine (Self : Logic_Functor_868; Vals : Entity_Vars.Value_Array) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_868) return String;
   overriding function Combine (Self : Logic_Functor_868; Vals : Entity_Vars.Value_Array) return Internal_Entity is
         From : constant Internal_Entity := Vals (1);
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
            if Vals (2).Node /= null
               and then Vals (2).Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for comb_prop");
            end if;
      Ret := Base_Type_Decl_P_Expected_Array_Concat_Operand_Type (Node => From.Node, Operand_Type => (Node => Vals (2).Node,  Info => Vals (2).Info), Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Combine;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_868) return String is
   begin
      return ("BaseTypeDecl.expected_array_concat_operand_type");
   end Image;
   type Logic_Functor_900 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_900; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_900) return String;
   overriding function Convert (Self : Logic_Functor_900; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Iterable_Comp_Type_Or_Null (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_900) return String is
   begin
      return ("BaseTypeDecl.iterable_comp_type_or_null");
   end Image;
   type Logic_Functor_870 is new Solver_Ifc.Converter_Type with
      null record;
   overriding function Convert (Self : Logic_Functor_870; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_870) return String;
   overriding function Convert (Self : Logic_Functor_870; From : Internal_Entity) return Internal_Entity is
         pragma Unreferenced (Self);
      Ret : Internal_Entity_Discrete_Base_Subtype_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Base_Type_Decl_P_Scalar_Base_Type (Node => From.Node, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_870) return String is
   begin
      return ("BaseTypeDecl.scalar_base_type");
   end Image;
   type Logic_Functor_681 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_681; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_681) return String;
   overriding function Convert (Self : Logic_Functor_681; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Basic_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Basic_Decl_P_Identity_Type (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_681) return String is
   begin
      return ("BasicDecl.identity_type");
   end Image;
   type Logic_Functor_945 is new Solver_Ifc.Converter_Type with
      null record;
   overriding function Convert (Self : Logic_Functor_945; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_945) return String;
   overriding function Convert (Self : Logic_Functor_945; From : Internal_Entity) return Internal_Entity is
         pragma Unreferenced (Self);
      Ret : Internal_Entity_Classwide_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Base_Type_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Dispatcher_Base_Type_Decl_P_Classwide_Type (Node => From.Node, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_945) return String is
   begin
      return ("BaseTypeDecl.classwide_type");
   end Image;
   type Logic_Functor_720 is new Solver_Ifc.Converter_Type with
      record
            Origin : Bare_Ada_Node;
      end record;
   overriding function Convert (Self : Logic_Functor_720; From : Internal_Entity) return Internal_Entity with Inline;
   overriding function Image (Self : Logic_Functor_720) return String;
   overriding function Convert (Self : Logic_Functor_720; From : Internal_Entity) return Internal_Entity is
      Ret : Internal_Entity_Base_Type_Decl;
   begin
            if From.Node /= null
               and then From.Node.Kind not in
                 Ada_Basic_Decl
            then
               Raise_Property_Exception
                 (From.Node,
                  Property_Error'Identity,
                  "mismatching node type for conv_prop");
            end if;
      Ret := Dispatcher_Basic_Decl_P_Expr_Type (Node => From.Node, Origin => Self.Origin, E_Info => From.Info);
      return (Node => Ret.Node, Info => Ret.Info);
   end Convert;
   -----------
   -- Image --
   -----------
   overriding function Image (Self : Logic_Functor_720) return String is
   begin
      return ("BasicDecl.expr_type");
   end Image;
   --
   --  Primitives for Bare_Ada_Node
   --
--# property-start AdaNode.declarative_scope 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:47'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Declarative_Scope
  (Node : Bare_Ada_Node
  )
   return Bare_Declarative_Part
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   Property_Result : Bare_Declarative_Part;
            procedure Finalizer_Scope_5 with Inline_Always;
      Node_Parents : Bare_Ada_Node_Array_Access;
Is_A : Boolean;
Find_Result : Bare_Ada_Node;
Cast_Result : Bare_Declarative_Part;
            procedure Finalizer_Scope_5 is
            begin
                     Dec_Ref (Node_Parents);
            end Finalizer_Scope_5;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
         --# expr-start 4 '<Cast to DeclarativePart at nodes.lkt:48:9>' Cast_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:48'
--# expr-start 3 '<Find at nodes.lkt:48:9>' Find_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:48'
   --# expr-start 1 '<FieldAccess for parents at nodes.lkt:48:9>' Node_Parents 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:48'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Node_Parents := Parents (Self, True); 
--# expr-done 1
      Find_Result := No_Bare_Ada_Node;
      declare
         Collection : constant Bare_Ada_Node_Array_Access := Node_Parents;
      begin
         for Item_0 of
               Collection.Items
         loop
   --# scope-start
               --# bind p Item_0
      --# expr-start 2 '<IsA DeclarativePart at nodes.lkt:48:36>' Is_A 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:48'
Is_A := Item_0 /= null 
and then Item_0.Kind in Ada_Declarative_Part_Range; 
--# expr-done 2
      if Is_A then
         Find_Result := Item_0; 
      end if;
   --# end
               exit when Is_A;
         end loop;
      end;
--# expr-done 3
   if Find_Result = null
      or else Find_Result.Kind in Ada_Declarative_Part_Range
   then
      Cast_Result := Find_Result;
   else
         Cast_Result := No_Bare_Ada_Node;
   end if;
--# expr-done 4
         Property_Result := Cast_Result;
   --# end
      Finalizer_Scope_5;
      exception
         when Exc : Precondition_Failure | Property_Error =>
                     Finalizer_Scope_5;
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Declarative_Scope;
--# end
--# property-start AdaNode.owning_unit_kind 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:51'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Owning_Unit_Kind
  (Node : Bare_Ada_Node
  )
   return Analysis_Unit_Kind
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   Property_Result : Analysis_Unit_Kind;
      Fld : Internal_Unit;
Fld_1 : Bare_Ada_Node;
Cast_Result : Bare_Compilation_Unit;
Fld_2 : Analysis_Unit_Kind;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
         --# expr-start 8 '<FieldAccess for unit_kind at nodes.lkt:52:9>' Fld_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:52'
--# expr-start 7 '<Cast to CompilationUnit at nodes.lkt:52:9>' Cast_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:52'
--# expr-start 6 '<FieldAccess for root at nodes.lkt:52:9>' Fld_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:52'
--# expr-start 5 '<FieldAccess for unit at nodes.lkt:52:9>' Fld 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:52'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Libadalang.Implementation.Unit (Node => Self);
--# expr-done 5
      if Fld = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld_1 := Fld.Ast_Root;
--# expr-done 6
   if Fld_1 = null
      or else Fld_1.Kind in Ada_Compilation_Unit_Range
   then
      Cast_Result := Fld_1;
   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;
--# expr-done 7
      if Cast_Result = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start CompilationUnit.unit_kind
Fld_2 := Libadalang.Implementation.Compilation_Unit_P_Unit_Kind (Node => Cast_Result);
--# end
--# expr-done 8
         Property_Result := Fld_2;
   --# end
      exception
         when Exc : Precondition_Failure | Property_Error =>
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Owning_Unit_Kind;
--# end
--# property-start AdaNode.withed_unit_helper 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:60'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Withed_Unit_Helper
  (Node : Bare_Ada_Node
      ; Unit_Name : Bare_Name
  )
   return Bare_Compilation_Unit
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   --# bind unit_name Unit_Name
   Property_Result : Bare_Compilation_Unit;
            procedure Finalizer_Scope_7 with Inline_Always;
      Local_Unit_Name_Array : Symbol_Type_Array_Access;
Local_Spec : Bare_Compilation_Unit;
Local_Body : Bare_Compilation_Unit;
Ignored : Bare_Compilation_Unit;
Var_Expr_0 : Bare_Compilation_Unit;
Var_Expr : Bare_Compilation_Unit;
Fld : Symbol_Type_Array_Access;
Fld_1 : Bare_Compilation_Unit;
Fld_2 : Bare_Compilation_Unit;
Is_Null : Boolean;
If_Result : Bare_Compilation_Unit;
Fld_3 : Bare_Compilation_Unit;
Fld_4 : Bare_Basic_Decl;
Is_A : Boolean;
Scope_Result : Boolean;
Result_Var : Boolean;
Is_Null_1 : Boolean;
If_Result_1 : Boolean;
Is_Null_2 : Boolean;
Is_Null_3 : Boolean;
If_Result_2 : Boolean;
If_Result_3 : Boolean;
If_Result_4 : Bare_Compilation_Unit;
Scope_Result_1 : Bare_Compilation_Unit;
Result_Var_1 : Bare_Compilation_Unit;
Let_Result : Bare_Compilation_Unit;
Let_Result_1 : Bare_Compilation_Unit;
Let_Result_2 : Bare_Compilation_Unit;
Let_Result_3 : Bare_Compilation_Unit;
            procedure Finalizer_Scope_7 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Local_Unit_Name_Array);
            end Finalizer_Scope_7;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
      begin
   --# scope-start
         --# expr-start 24 '<Let at nodes.lkt:63:9>' Let_Result_3 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:63'
--# scope-start
--# expr-start 9 '<FieldAccess for as_symbol_array at nodes.lkt:63:31>' Fld 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:63'
      if Unit_Name = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]Name.as_symbol_array'
Fld := Libadalang.Implementation.Dispatcher_Name_P_As_Symbol_Array (Node => Unit_Name);
--# end
--# expr-done 9
Local_Unit_Name_Array := Fld; Inc_Ref (Local_Unit_Name_Array);
--# bind unit_name_array Local_Unit_Name_Array
--# expr-start 23 '<Let at nodes.lkt:64:9>' Let_Result_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:64'
--# scope-start
--# expr-start 10 '<FieldAccess for designated_compilation_unit at nodes.lkt:64:20>' Fld_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:64'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.designated_compilation_unit
Fld_1 := Libadalang.Implementation.Ada_Node_P_Designated_Compilation_Unit (Node => Self, Name => Local_Unit_Name_Array, Kind => Unit_Specification, Not_Found_Is_Error => False);
--# end
--# expr-done 10
Local_Spec := Fld_1; 
--# bind spec Local_Spec
--# expr-start 22 '<Let at nodes.lkt:67:9>' Let_Result_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:67'
--# scope-start
--# expr-start 13 '<If at nodes.lkt:67:20>' If_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:67'
--# expr-start 11 '<IsNull at nodes.lkt:67:23>' Is_Null 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:67'
Is_Null := Local_Spec = null; 
--# expr-done 11
if Is_Null then
   --# expr-start 12 '<FieldAccess for designated_compilation_unit at nodes.lkt:67:41>' Fld_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:67'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.designated_compilation_unit
Fld_2 := Libadalang.Implementation.Ada_Node_P_Designated_Compilation_Unit (Node => Self, Name => Local_Unit_Name_Array, Kind => Unit_Body, Not_Found_Is_Error => False);
--# end
--# expr-done 12
   If_Result := Fld_2;
else
   If_Result := No_Bare_Ada_Node;
end if;
--# expr-done 13
Local_Body := If_Result; 
--# bind body Local_Body
--# expr-start 21 '<Let at nodes.lkt:73:9>' Let_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:73'
--# scope-start
--# expr-start 20 '<If at nodes.lkt:73:17>' If_Result_4 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:73'
Var_Expr_0 := Local_Body;
if Var_Expr_0 /= No_Bare_Ada_Node then
   --# scope-start
   --# bind v1 Var_Expr_0
--# expr-start 15 '<IsA PackageBody at nodes.lkt:74:29>' Is_A 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:74'
--# expr-start 14 '<FieldAccess for decl at nodes.lkt:74:29>' Fld_4 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:74'
      if Var_Expr_0 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start CompilationUnit.decl
Fld_4 := Libadalang.Implementation.Compilation_Unit_P_Decl (Node => Var_Expr_0);
--# end
--# expr-done 14
Is_A := Fld_4 /= null 
and then Fld_4.Kind in Ada_Package_Body_Range; 
--# expr-done 15
Scope_Result := Is_A;
   --# end
   Result_Var := Scope_Result;
else
   Result_Var := False;
end if;
if Result_Var then
   --# expr-start 16 '<IsNull at nodes.lkt:74:59>' Is_Null_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:74'
Is_Null_1 := Local_Spec = null; 
--# expr-done 16
   If_Result_1 := Is_Null_1;
else
   If_Result_1 := False;
end if;
if If_Result_1 then
   If_Result_3 := True;
else
--# expr-start 17 '<IsNull at nodes.lkt:75:15>' Is_Null_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:75'
Is_Null_2 := Local_Spec = null; 
--# expr-done 17
if Is_Null_2 then
   --# expr-start 18 '<IsNull at nodes.lkt:75:32>' Is_Null_3 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:75'
Is_Null_3 := Local_Body = null; 
--# expr-done 18
   If_Result_2 := Is_Null_3;
else
   If_Result_2 := False;
end if;
   If_Result_3 := If_Result_2;
end if;
if If_Result_3 then
   --# expr-start 19 '<FieldAccess for designated_compilation_unit at nodes.lkt:75:51>' Fld_3 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:75'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.designated_compilation_unit
Fld_3 := Libadalang.Implementation.Ada_Node_P_Designated_Compilation_Unit (Node => Self, Name => Local_Unit_Name_Array, Kind => Unit_Specification, Not_Found_Is_Error => True);
--# end
--# expr-done 19
   If_Result_4 := Fld_3;
else
   If_Result_4 := No_Bare_Ada_Node;
end if;
--# expr-done 20
Ignored := If_Result_4; 
--# bind _ Ignored
Var_Expr := Local_Spec;
if Var_Expr /= No_Bare_Ada_Node then
   --# scope-start
   --# bind e Var_Expr
Scope_Result_1 := Var_Expr;
   --# end
   Result_Var_1 := Scope_Result_1;
else
   Result_Var_1 := Local_Body;
end if;
Let_Result := Result_Var_1; 
--# end
--# expr-done 21
Let_Result_1 := Let_Result; 
--# end
--# expr-done 22
Let_Result_2 := Let_Result_1; 
--# end
--# expr-done 23
Let_Result_3 := Let_Result_2; 
--# end
--# expr-done 24
         Property_Result := Let_Result_3;
   --# end
      Finalizer_Scope_7;
      exception
         when Exc : Precondition_Failure | Property_Error =>
                     Finalizer_Scope_7;
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Withed_Unit_Helper;
--# end
--# property-start AdaNode.is_contract_aspect 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:83'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Is_Contract_Aspect
  (Node : Bare_Ada_Node
      ; Name : Symbol_Type
  )
   return Boolean
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   --# bind name Name
   Property_Result : Boolean;
      Sym : Symbol_Type;
Sym_1 : Symbol_Type;
Sym_2 : Symbol_Type;
Sym_3 : Symbol_Type;
Sym_4 : Symbol_Type;
Sym_5 : Symbol_Type;
Sym_6 : Symbol_Type;
Sym_7 : Symbol_Type;
Sym_8 : Symbol_Type;
Sym_9 : Symbol_Type;
Sym_10 : Symbol_Type;
Sym_11 : Symbol_Type;
Sym_12 : Symbol_Type;
Sym_13 : Symbol_Type;
Sym_14 : Symbol_Type;
Sym_15 : Symbol_Type;
Sym_16 : Symbol_Type;
Sym_17 : Symbol_Type;
Sym_18 : Symbol_Type;
Sym_19 : Symbol_Type;
Sym_20 : Symbol_Type;
Sym_21 : Symbol_Type;
Sym_22 : Symbol_Type;
Sym_23 : Symbol_Type;
Sym_24 : Symbol_Type;
Sym_25 : Symbol_Type;
Sym_26 : Symbol_Type;
Sym_27 : Symbol_Type;
Is_Equal : Boolean;
Is_Equal_1 : Boolean;
If_Result : Boolean;
Is_Equal_2 : Boolean;
If_Result_1 : Boolean;
Is_Equal_3 : Boolean;
If_Result_2 : Boolean;
Is_Equal_4 : Boolean;
If_Result_3 : Boolean;
Is_Equal_5 : Boolean;
If_Result_4 : Boolean;
Is_Equal_6 : Boolean;
If_Result_5 : Boolean;
Is_Equal_7 : Boolean;
If_Result_6 : Boolean;
Is_Equal_8 : Boolean;
If_Result_7 : Boolean;
Is_Equal_9 : Boolean;
If_Result_8 : Boolean;
Is_Equal_10 : Boolean;
If_Result_9 : Boolean;
Is_Equal_11 : Boolean;
If_Result_10 : Boolean;
Is_Equal_12 : Boolean;
If_Result_11 : Boolean;
Is_Equal_13 : Boolean;
If_Result_12 : Boolean;
Is_Equal_14 : Boolean;
If_Result_13 : Boolean;
Is_Equal_15 : Boolean;
If_Result_14 : Boolean;
Is_Equal_16 : Boolean;
If_Result_15 : Boolean;
Is_Equal_17 : Boolean;
If_Result_16 : Boolean;
Is_Equal_18 : Boolean;
If_Result_17 : Boolean;
Is_Equal_19 : Boolean;
If_Result_18 : Boolean;
Is_Equal_20 : Boolean;
If_Result_19 : Boolean;
Is_Equal_21 : Boolean;
If_Result_20 : Boolean;
Is_Equal_22 : Boolean;
If_Result_21 : Boolean;
Is_Equal_23 : Boolean;
If_Result_22 : Boolean;
Is_Equal_24 : Boolean;
If_Result_23 : Boolean;
Is_Equal_25 : Boolean;
If_Result_24 : Boolean;
Is_Equal_26 : Boolean;
If_Result_25 : Boolean;
Is_Equal_27 : Boolean;
If_Result_26 : Boolean;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
--# expr-start 25 '<Symbol Pre at nodes.lkt:84:17>' Sym 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Pre); 
--# expr-done 25
Is_Equal_27 := Name = Sym; 
if Is_Equal_27 then
   If_Result_26 := True;
else
--# expr-start 26 '<Symbol Pre'"'"'Class at nodes.lkt:84:26>' Sym_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_1 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Pre_Class); 
--# expr-done 26
Is_Equal_26 := Name = Sym_1; 
if Is_Equal_26 then
   If_Result_25 := True;
else
--# expr-start 27 '<Symbol Post at nodes.lkt:84:41>' Sym_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_2 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Post); 
--# expr-done 27
Is_Equal_25 := Name = Sym_2; 
if Is_Equal_25 then
   If_Result_24 := True;
else
--# expr-start 28 '<Symbol Post'"'"'Class at nodes.lkt:84:51>' Sym_3 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_3 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Post_Class); 
--# expr-done 28
Is_Equal_24 := Name = Sym_3; 
if Is_Equal_24 then
   If_Result_23 := True;
else
--# expr-start 29 '<Symbol Refined_Post at nodes.lkt:84:67>' Sym_4 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_4 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Refined_Post); 
--# expr-done 29
Is_Equal_23 := Name = Sym_4; 
if Is_Equal_23 then
   If_Result_22 := True;
else
--# expr-start 30 '<Symbol Precondition at nodes.lkt:84:85>' Sym_5 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_5 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Precondition); 
--# expr-done 30
Is_Equal_22 := Name = Sym_5; 
if Is_Equal_22 then
   If_Result_21 := True;
else
--# expr-start 31 '<Symbol Postcondition at nodes.lkt:84:103>' Sym_6 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_6 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Postcondition); 
--# expr-done 31
Is_Equal_21 := Name = Sym_6; 
if Is_Equal_21 then
   If_Result_20 := True;
else
--# expr-start 32 '<Symbol Precondition'"'"'Class at nodes.lkt:84:122>' Sym_7 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_7 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Precondition_Class); 
--# expr-done 32
Is_Equal_20 := Name = Sym_7; 
if Is_Equal_20 then
   If_Result_19 := True;
else
--# expr-start 33 '<Symbol Postcondition'"'"'Class at nodes.lkt:84:146>' Sym_8 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_8 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Postcondition_Class); 
--# expr-done 33
Is_Equal_19 := Name = Sym_8; 
if Is_Equal_19 then
   If_Result_18 := True;
else
--# expr-start 34 '<Symbol Invariant at nodes.lkt:84:171>' Sym_9 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_9 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Invariant); 
--# expr-done 34
Is_Equal_18 := Name = Sym_9; 
if Is_Equal_18 then
   If_Result_17 := True;
else
--# expr-start 35 '<Symbol Invariant'"'"'Class at nodes.lkt:84:186>' Sym_10 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_10 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Invariant_Class); 
--# expr-done 35
Is_Equal_17 := Name = Sym_10; 
if Is_Equal_17 then
   If_Result_16 := True;
else
--# expr-start 36 '<Symbol Type_Invariant at nodes.lkt:84:207>' Sym_11 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_11 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Type_Invariant); 
--# expr-done 36
Is_Equal_16 := Name = Sym_11; 
if Is_Equal_16 then
   If_Result_15 := True;
else
--# expr-start 37 '<Symbol Type_Invariant'"'"'Class at nodes.lkt:84:227>' Sym_12 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_12 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Type_Invariant_Class); 
--# expr-done 37
Is_Equal_15 := Name = Sym_12; 
if Is_Equal_15 then
   If_Result_14 := True;
else
--# expr-start 38 '<Symbol Predicate at nodes.lkt:84:253>' Sym_13 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_13 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Predicate); 
--# expr-done 38
Is_Equal_14 := Name = Sym_13; 
if Is_Equal_14 then
   If_Result_13 := True;
else
--# expr-start 39 '<Symbol Static_Predicate at nodes.lkt:84:268>' Sym_14 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_14 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Static_Predicate); 
--# expr-done 39
Is_Equal_13 := Name = Sym_14; 
if Is_Equal_13 then
   If_Result_12 := True;
else
--# expr-start 40 '<Symbol Dynamic_Predicate at nodes.lkt:84:290>' Sym_15 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_15 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Dynamic_Predicate); 
--# expr-done 40
Is_Equal_12 := Name = Sym_15; 
if Is_Equal_12 then
   If_Result_11 := True;
else
--# expr-start 41 '<Symbol Default_Initial_Condition at nodes.lkt:84:313>' Sym_16 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_16 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Default_Initial_Condition); 
--# expr-done 41
Is_Equal_11 := Name = Sym_16; 
if Is_Equal_11 then
   If_Result_10 := True;
else
--# expr-start 42 '<Symbol Initial_Condition at nodes.lkt:84:344>' Sym_17 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_17 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Initial_Condition); 
--# expr-done 42
Is_Equal_10 := Name = Sym_17; 
if Is_Equal_10 then
   If_Result_9 := True;
else
--# expr-start 43 '<Symbol Contract_Cases at nodes.lkt:84:367>' Sym_18 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_18 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Contract_Cases); 
--# expr-done 43
Is_Equal_9 := Name = Sym_18; 
if Is_Equal_9 then
   If_Result_8 := True;
else
--# expr-start 44 '<Symbol Test_Case at nodes.lkt:84:387>' Sym_19 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_19 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Test_Case); 
--# expr-done 44
Is_Equal_8 := Name = Sym_19; 
if Is_Equal_8 then
   If_Result_7 := True;
else
--# expr-start 45 '<Symbol Global at nodes.lkt:84:402>' Sym_20 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_20 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Global); 
--# expr-done 45
Is_Equal_7 := Name = Sym_20; 
if Is_Equal_7 then
   If_Result_6 := True;
else
--# expr-start 46 '<Symbol Refined_Global at nodes.lkt:84:414>' Sym_21 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_21 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Refined_Global); 
--# expr-done 46
Is_Equal_6 := Name = Sym_21; 
if Is_Equal_6 then
   If_Result_5 := True;
else
--# expr-start 47 '<Symbol Refined_State at nodes.lkt:84:434>' Sym_22 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_22 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Refined_State); 
--# expr-done 47
Is_Equal_5 := Name = Sym_22; 
if Is_Equal_5 then
   If_Result_4 := True;
else
--# expr-start 48 '<Symbol Stable_Properties at nodes.lkt:84:453>' Sym_23 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_23 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Stable_Properties); 
--# expr-done 48
Is_Equal_4 := Name = Sym_23; 
if Is_Equal_4 then
   If_Result_3 := True;
else
--# expr-start 49 '<Symbol Depends at nodes.lkt:84:476>' Sym_24 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_24 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Depends); 
--# expr-done 49
Is_Equal_3 := Name = Sym_24; 
if Is_Equal_3 then
   If_Result_2 := True;
else
--# expr-start 50 '<Symbol Refined_Depends at nodes.lkt:84:489>' Sym_25 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_25 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Refined_Depends); 
--# expr-done 50
Is_Equal_2 := Name = Sym_25; 
if Is_Equal_2 then
   If_Result_1 := True;
else
--# expr-start 51 '<Symbol Predicate_Failure at nodes.lkt:84:510>' Sym_26 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_26 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Predicate_Failure); 
--# expr-done 51
Is_Equal_1 := Name = Sym_26; 
if Is_Equal_1 then
   If_Result := True;
else
--# expr-start 52 '<Symbol SPARK_Mode at nodes.lkt:84:533>' Sym_27 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:84'
Sym_27 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Spark_Mode); 
--# expr-done 52
Is_Equal := Name = Sym_27; 
   If_Result := Is_Equal;
end if;
   If_Result_1 := If_Result;
end if;
   If_Result_2 := If_Result_1;
end if;
   If_Result_3 := If_Result_2;
end if;
   If_Result_4 := If_Result_3;
end if;
   If_Result_5 := If_Result_4;
end if;
   If_Result_6 := If_Result_5;
end if;
   If_Result_7 := If_Result_6;
end if;
   If_Result_8 := If_Result_7;
end if;
   If_Result_9 := If_Result_8;
end if;
   If_Result_10 := If_Result_9;
end if;
   If_Result_11 := If_Result_10;
end if;
   If_Result_12 := If_Result_11;
end if;
   If_Result_13 := If_Result_12;
end if;
   If_Result_14 := If_Result_13;
end if;
   If_Result_15 := If_Result_14;
end if;
   If_Result_16 := If_Result_15;
end if;
   If_Result_17 := If_Result_16;
end if;
   If_Result_18 := If_Result_17;
end if;
   If_Result_19 := If_Result_18;
end if;
   If_Result_20 := If_Result_19;
end if;
   If_Result_21 := If_Result_20;
end if;
   If_Result_22 := If_Result_21;
end if;
   If_Result_23 := If_Result_22;
end if;
   If_Result_24 := If_Result_23;
end if;
   If_Result_25 := If_Result_24;
end if;
   If_Result_26 := If_Result_25;
end if;
         Property_Result := If_Result_26;
   --# end
      exception
         when Exc : Precondition_Failure | Property_Error =>
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Is_Contract_Aspect;
--# end
--# property-start AdaNode.aspect_has_forward_visibility 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:89'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Aspect_Has_Forward_Visibility
  (Node : Bare_Ada_Node
      ; Name : Symbol_Type
  )
   return Boolean
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   --# bind name Name
   Property_Result : Boolean;
      Fld : Boolean;
Sym : Symbol_Type;
Sym_1 : Symbol_Type;
Is_Equal : Boolean;
Is_Equal_1 : Boolean;
If_Result : Boolean;
If_Result_1 : Boolean;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
--# expr-start 53 '<FieldAccess for is_contract_aspect at nodes.lkt:90:9>' Fld 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:90'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.is_contract_aspect
Fld := Libadalang.Implementation.Ada_Node_P_Is_Contract_Aspect (Node => Self, Name => Name);
--# end
--# expr-done 53
if Fld then
   If_Result_1 := True;
else
--# expr-start 54 '<Symbol Iterator_Element at nodes.lkt:90:50>' Sym 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:90'
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Iterator_Element); 
--# expr-done 54
Is_Equal_1 := Name = Sym; 
if Is_Equal_1 then
   If_Result := True;
else
--# expr-start 55 '<Symbol Default_Iterator at nodes.lkt:90:72>' Sym_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:90'
Sym_1 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Default_Iterator); 
--# expr-done 55
Is_Equal := Name = Sym_1; 
   If_Result := Is_Equal;
end if;
   If_Result_1 := If_Result;
end if;
         Property_Result := If_Result_1;
   --# end
      exception
         when Exc : Precondition_Failure | Property_Error =>
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Aspect_Has_Forward_Visibility;
--# end
--# property-start AdaNode.in_aspect_with_forward_visibility 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:92'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_In_Aspect_With_Forward_Visibility
  (Node : Bare_Ada_Node
  )
   return Boolean
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   Property_Result : Boolean;
            procedure Finalizer_Scope_10 with Inline_Always;
      Var_Expr_1 : Bare_Aspect_Assoc;
Var_Expr_2 : Bare_Pragma_Node;
Var_Expr : Boolean;
Node_Parents : Bare_Ada_Node_Array_Access;
Cast_Expr : Bare_Ada_Node;
Cast_Result : Bare_Aspect_Assoc;
Fld : Bare_Name;
As_Entity : Internal_Entity_Name;
Is_Null : Boolean;
If_Result : Internal_Entity_Name;
Fld_1 : Symbol_Type;
Fld_2 : Boolean;
Scope_Result : Boolean;
Result_Var : Boolean;
Scope_Result_1 : Boolean;
Cast_Expr_1 : Bare_Ada_Node;
Cast_Result_1 : Bare_Pragma_Node;
Fld_3 : Bare_Identifier;
As_Entity_1 : Internal_Entity_Identifier;
Is_Null_1 : Boolean;
If_Result_1 : Internal_Entity_Identifier;
Fld_4 : Symbol_Type;
Fld_5 : Boolean;
Scope_Result_2 : Boolean;
Result_Var_1 : Boolean;
Result_Var_2 : Boolean;
Find_Result : Bare_Ada_Node;
Is_Null_2 : Boolean;
Not_Val : Boolean;
            procedure Finalizer_Scope_10 is
            begin
                     Dec_Ref (Node_Parents);
            end Finalizer_Scope_10;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
         --# expr-start 69 '<Not at nodes.lkt:92:53>' Not_Val 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:92'
--# expr-start 68 '<IsNull at nodes.lkt:92:57>' Is_Null_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:92'
--# expr-start 67 '<Find at nodes.lkt:92:57>' Find_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:92'
   --# expr-start 56 '<FieldAccess for parents at nodes.lkt:92:57>' Node_Parents 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:92'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Node_Parents := Parents (Self, True); 
--# expr-done 56
      Find_Result := No_Bare_Ada_Node;
      declare
         Collection : constant Bare_Ada_Node_Array_Access := Node_Parents;
      begin
         for Item_0 of
               Collection.Items
         loop
   --# scope-start
               --# bind p Item_0
--# expr-start 57 '<Cast to AspectAssoc at nodes.lkt:93:16>' Cast_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:93'
Cast_Expr := Item_0; 
   if Cast_Expr = null
      or else Cast_Expr.Kind in Ada_Aspect_Assoc_Range
   then
      Cast_Result := Cast_Expr;
   else
         Cast_Result := No_Bare_Ada_Node;
   end if;
--# expr-done 57
Var_Expr_1 := Cast_Result;
if Var_Expr_1 /= No_Bare_Ada_Node then
   --# scope-start
   --# bind a Var_Expr_1
--# expr-start 61 '<FieldAccess for aspect_has_forward_visibility at nodes.lkt:94:20>' Fld_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:94'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# expr-start 60 '<FieldAccess for name_symbol at nodes.lkt:94:55>' Fld_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:94'
--# expr-start 59 '<as_bare_entity at nodes.lkt:94:55>' If_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:94'
--# expr-start 58 '<FieldAccess for id at nodes.lkt:94:55>' Fld 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:94'
      if Var_Expr_1 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Var_Expr_1.Aspect_Assoc_F_Id;
--# expr-done 58
Is_Null := Fld = null; 
if Is_Null then
   If_Result := No_Entity_Name;
else
As_Entity := (Info => No_Entity_Info, Node => Fld); 
   If_Result := As_Entity;
end if;
--# expr-done 59
      if If_Result.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]Name.name_symbol'
Fld_1 := Libadalang.Implementation.Dispatcher_Name_P_Name_Symbol (Node => If_Result.Node);
--# end
--# expr-done 60
--# property-call-start AdaNode.aspect_has_forward_visibility
Fld_2 := Libadalang.Implementation.Ada_Node_P_Aspect_Has_Forward_Visibility (Node => Self, Name => Fld_1);
--# end
--# expr-done 61
Scope_Result := Fld_2;
   --# end
   Result_Var := Scope_Result;
else
   Result_Var := False;
end if;
Var_Expr := Result_Var;
if Var_Expr /= False then
   --# scope-start
   --# bind e Var_Expr
Scope_Result_1 := Var_Expr;
   --# end
   Result_Var_2 := Scope_Result_1;
else
--# expr-start 62 '<Cast to Pragma at nodes.lkt:95:15>' Cast_Result_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:95'
Cast_Expr_1 := Item_0; 
   if Cast_Expr_1 = null
      or else Cast_Expr_1.Kind in Ada_Pragma_Node_Range
   then
      Cast_Result_1 := Cast_Expr_1;
   else
         Cast_Result_1 := No_Bare_Ada_Node;
   end if;
--# expr-done 62
Var_Expr_2 := Cast_Result_1;
if Var_Expr_2 /= No_Bare_Ada_Node then
   --# scope-start
   --# bind p Var_Expr_2
--# expr-start 66 '<FieldAccess for aspect_has_forward_visibility at nodes.lkt:96:20>' Fld_5 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:96'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# expr-start 65 '<FieldAccess for name_symbol at nodes.lkt:96:55>' Fld_4 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:96'
--# expr-start 64 '<as_bare_entity at nodes.lkt:96:55>' If_Result_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:96'
--# expr-start 63 '<FieldAccess for id at nodes.lkt:96:55>' Fld_3 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:96'
      if Var_Expr_2 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld_3 := Var_Expr_2.Pragma_Node_F_Id;
--# expr-done 63
Is_Null_1 := Fld_3 = null; 
if Is_Null_1 then
   If_Result_1 := No_Entity_Identifier;
else
As_Entity_1 := (Info => No_Entity_Info, Node => Fld_3); 
   If_Result_1 := As_Entity_1;
end if;
--# expr-done 64
      if If_Result_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]Name.name_symbol'
Fld_4 := Libadalang.Implementation.Dispatcher_Name_P_Name_Symbol (Node => If_Result_1.Node);
--# end
--# expr-done 65
--# property-call-start AdaNode.aspect_has_forward_visibility
Fld_5 := Libadalang.Implementation.Ada_Node_P_Aspect_Has_Forward_Visibility (Node => Self, Name => Fld_4);
--# end
--# expr-done 66
Scope_Result_2 := Fld_5;
   --# end
   Result_Var_1 := Scope_Result_2;
else
   Result_Var_1 := False;
end if;
   Result_Var_2 := Result_Var_1;
end if;
      if Result_Var_2 then
         Find_Result := Item_0; 
      end if;
   --# end
               exit when Result_Var_2;
         end loop;
      end;
--# expr-done 67
Is_Null_2 := Find_Result = null; 
--# expr-done 68
Not_Val := not (Is_Null_2); 
--# expr-done 69
         Property_Result := Not_Val;
   --# end
      Finalizer_Scope_10;
      exception
         when Exc : Precondition_Failure | Property_Error =>
                     Finalizer_Scope_10;
            raise;
      end;
   return Property_Result;
end Ada_Node_P_In_Aspect_With_Forward_Visibility;
--# end
--# property-start AdaNode.in_aspect 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:101'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_In_Aspect
  (Node : Bare_Ada_Node
      ; Name : Symbol_Type
  )
   return Boolean
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   --# bind name Name
   Property_Result : Boolean;
            procedure Finalizer_Scope_11 with Inline_Always;
      Var_Expr_1 : Bare_Aspect_Assoc;
Node_Parents : Bare_Ada_Node_Array_Access;
Cast_Expr : Bare_Ada_Node;
Cast_Result : Bare_Aspect_Assoc;
Fld : Bare_Name;
Fld_1 : Boolean;
Scope_Result : Boolean;
Result_Var : Boolean;
Quantifier_Result : Boolean;
            procedure Finalizer_Scope_11 is
            begin
                     Dec_Ref (Node_Parents);
            end Finalizer_Scope_11;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
         --# expr-start 74 '<AnyQuantifier at nodes.lkt:101:41>' Quantifier_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:101'
   --# expr-start 70 '<FieldAccess for parents at nodes.lkt:101:41>' Node_Parents 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:101'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Node_Parents := Parents (Self, True); 
--# expr-done 70
      Quantifier_Result := False;
      declare
         Collection : constant Bare_Ada_Node_Array_Access := Node_Parents;
      begin
         for Item_0 of
               Collection.Items
         loop
   --# scope-start
               --# bind p Item_0
--# expr-start 71 '<Cast to AspectAssoc at nodes.lkt:102:16>' Cast_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:102'
Cast_Expr := Item_0; 
   if Cast_Expr = null
      or else Cast_Expr.Kind in Ada_Aspect_Assoc_Range
   then
      Cast_Result := Cast_Expr;
   else
         Cast_Result := No_Bare_Ada_Node;
   end if;
--# expr-done 71
Var_Expr_1 := Cast_Result;
if Var_Expr_1 /= No_Bare_Ada_Node then
   --# scope-start
   --# bind a Var_Expr_1
--# expr-start 73 '<FieldAccess for name_is at nodes.lkt:102:44>' Fld_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:102'
--# expr-start 72 '<FieldAccess for id at nodes.lkt:102:44>' Fld 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:102'
      if Var_Expr_1 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Var_Expr_1.Aspect_Assoc_F_Id;
--# expr-done 72
      if Fld = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start Name.name_is
Fld_1 := Libadalang.Implementation.Name_P_Name_Is (Node => Fld, Sym => Name);
--# end
--# expr-done 73
Scope_Result := Fld_1;
   --# end
   Result_Var := Scope_Result;
else
   Result_Var := False;
end if;
      Quantifier_Result := Result_Var;
   --# end
               exit when Quantifier_Result;
         end loop;
      end;
--# expr-done 74
         Property_Result := Quantifier_Result;
   --# end
      Finalizer_Scope_11;
      exception
         when Exc : Precondition_Failure | Property_Error =>
                     Finalizer_Scope_11;
            raise;
      end;
   return Property_Result;
end Ada_Node_P_In_Aspect;
--# end
--# property-start AdaNode.empty_env 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:105'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Empty_Env
  (Node : Bare_Ada_Node
  )
   return Lexical_Env
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   Property_Result : Lexical_Env;
            procedure Finalizer_Scope_12 with Inline_Always;
      Node_Parents : Bare_Ada_Node_Array_Access;
Is_A : Boolean;
Find_Result : Bare_Ada_Node;
Cast_Result : Bare_Compilation_Unit;
Fld : Lexical_Env := Empty_Env;
            procedure Finalizer_Scope_12 is
            begin
                     Dec_Ref (Node_Parents);
                     Dec_Ref (Fld);
            end Finalizer_Scope_12;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
      begin
   --# scope-start
         --# expr-start 79 '<FieldAccess for get_empty_env at nodes.lkt:106:9>' Fld 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:106'
--# expr-start 78 '<Cast to CompilationUnit at nodes.lkt:106:9>' Cast_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:106'
--# expr-start 77 '<Find at nodes.lkt:106:9>' Find_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:106'
   --# expr-start 75 '<FieldAccess for parents at nodes.lkt:106:9>' Node_Parents 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:106'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Node_Parents := Parents (Self, True); 
--# expr-done 75
      Find_Result := No_Bare_Ada_Node;
      declare
         Collection : constant Bare_Ada_Node_Array_Access := Node_Parents;
      begin
         for Item_0 of
               Collection.Items
         loop
   --# scope-start
               --# bind p Item_0
      --# expr-start 76 '<IsA CompilationUnit at nodes.lkt:106:36>' Is_A 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:106'
Is_A := Item_0 /= null 
and then Item_0.Kind in Ada_Compilation_Unit_Range; 
--# expr-done 76
      if Is_A then
         Find_Result := Item_0; 
      end if;
   --# end
               exit when Is_A;
         end loop;
      end;
--# expr-done 77
   if Find_Result = null
      or else Find_Result.Kind in Ada_Compilation_Unit_Range
   then
      Cast_Result := Find_Result;
   else
         Cast_Result := No_Bare_Ada_Node;
   end if;
--# expr-done 78
      if Cast_Result = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Fld := Libadalang.Implementation.Extensions.Compilation_Unit_P_Get_Empty_Env (Node => Cast_Result);
--# expr-done 79
         Property_Result := Fld;
            Inc_Ref (Property_Result);
   --# end
      Finalizer_Scope_12;
      exception
         when Exc : Precondition_Failure | Property_Error =>
                     Finalizer_Scope_12;
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Empty_Env;
--# end
--# property-start AdaNode.is_not_null 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:109'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Is_Not_Null
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Boolean
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind entity Ent
   Property_Result : Boolean;
      Is_Null : Boolean;
Not_Val : Boolean;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
         --# expr-start 81 '<Not at nodes.lkt:114:9>' Not_Val 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:114'
--# expr-start 80 '<IsNull at nodes.lkt:114:13>' Is_Null 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:114'
Is_Null := Ent.Node = null; 
--# expr-done 80
Not_Val := not (Is_Null); 
--# expr-done 81
         Property_Result := Not_Val;
   --# end
      exception
         when Exc : Precondition_Failure | Property_Error =>
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Is_Not_Null;
--# end
--# property-start AdaNode.eval_discrete_range 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:117'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Eval_Discrete_Range
  (Node : Bare_Ada_Node
      ; Dr : Internal_Discrete_Range
  )
   return Internal_Eval_Discrete_Range
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   --# bind dr Dr
   Property_Result : Internal_Eval_Discrete_Range;
            procedure Finalizer_Scope_14 with Inline_Always;
            procedure Finalizer_Scope_1908 with Inline_Always;
      Var_Expr_0 : Internal_Entity_Expr;
Fld : Internal_Entity_Expr;
Fld_1 : Big_Integer_Type;
Scope_Result : Big_Integer_Type;
Big_Int : Big_Integer_Type;
Result_Var : Big_Integer_Type;
Fld_2 : Internal_Entity_Expr;
Fld_3 : Big_Integer_Type;
New_Struct : Internal_Eval_Discrete_Range := No_Eval_Discrete_Range;
Is_Equal : Boolean;
If_Result : Internal_Eval_Discrete_Range := No_Eval_Discrete_Range;
            procedure Finalizer_Scope_14 is
            begin
                     Dec_Ref (Scope_Result);
                     Dec_Ref (Big_Int);
                     Dec_Ref (Result_Var);
                     Dec_Ref (Fld_3);
                     Dec_Ref (New_Struct);
                     Dec_Ref (If_Result);
            end Finalizer_Scope_14;
            procedure Finalizer_Scope_1908 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_1908;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
         --# expr-start 89 '<If at nodes.lkt:118:9>' If_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:118'
--# expr-start 82 '<Eq at nodes.lkt:118:12>' Is_Equal 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:118'
Is_Equal := Dr = No_Discrete_Range; 
--# expr-done 82
if Is_Equal then
   If_Result := raise Precondition_Failure with "Attempting to evaluate a null discrete range";
else
   --# expr-start 88 '<New EvalDiscreteRange at nodes.lkt:118:141>' New_Struct 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:118'
--# expr-start 84 '<FieldAccess for eval_as_int at nodes.lkt:119:93>' Fld_3 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:119'
--# expr-start 83 '<FieldAccess for high_bound at nodes.lkt:119:93>' Fld_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:119'
Fld_2 := Dr.High_Bound;
--# expr-done 83
      if Fld_2.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start Expr.eval_as_int
Fld_3 := Libadalang.Implementation.Expr_P_Eval_As_Int (Node => Fld_2.Node, E_Info => Fld_2.Info);
--# end
--# expr-done 84
--# expr-start 85 '<FieldAccess for low_bound at nodes.lkt:119:23>' Fld 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:119'
Fld := Dr.Low_Bound;
--# expr-done 85
Var_Expr_0 := Fld;
if Var_Expr_0 /= No_Entity_Expr then
   --# scope-start
   --# bind lb Var_Expr_0
--# expr-start 86 '<FieldAccess for eval_as_int at nodes.lkt:119:47>' Fld_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:119'
      if Var_Expr_0.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start Expr.eval_as_int
Fld_1 := Libadalang.Implementation.Expr_P_Eval_As_Int (Node => Var_Expr_0.Node, E_Info => Var_Expr_0.Info);
--# end
--# expr-done 86
Scope_Result := Fld_1;
   Inc_Ref (Scope_Result);
   --# end
      Finalizer_Scope_1908;
   Result_Var := Scope_Result;
else
   --# expr-start 87 '<BigInteger 0 at nodes.lkt:119:77>' Big_Int 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:119'
Big_Int := Create_Big_Integer ("0"); 
--# expr-done 87
   Result_Var := Big_Int;
end if;
      Inc_Ref (Result_Var);
Inc_Ref (Fld_3);
Inc_Ref (Result_Var);
New_Struct := (High_Bound => Fld_3, Low_Bound => Result_Var); 
--# expr-done 88
   If_Result := New_Struct;
end if;
      Inc_Ref (If_Result);
--# expr-done 89
         Property_Result := If_Result;
            Inc_Ref (Property_Result);
   --# end
      Finalizer_Scope_14;
      exception
         when Exc : Precondition_Failure | Property_Error =>
                     Finalizer_Scope_14;
                     Finalizer_Scope_1908;
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Eval_Discrete_Range;
--# end
--# property-start AdaNode.sym_join 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:123'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Sym_Join
  (Node : Bare_Ada_Node
      ; Syms : Symbol_Type_Array_Access
      ; Sep : String_Type
  )
   return String_Type
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   --# bind syms Syms
   --# bind sep Sep
   Property_Result : String_Type;
            procedure Finalizer_Scope_15 with Inline_Always;
            procedure Finalizer_Scope_1909 with Inline_Always;
      Fld : String_Type;
Map_Result : String_Type_Array_Access;
Join_Result : String_Type;
            procedure Finalizer_Scope_15 is
            begin
                     Dec_Ref (Map_Result);
                     Dec_Ref (Join_Result);
            end Finalizer_Scope_15;
            procedure Finalizer_Scope_1909 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_1909;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
      begin
   --# scope-start
         --# expr-start 92 '<Join at nodes.lkt:124:9>' Join_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:124'
--# expr-start 91 '<Map at nodes.lkt:124:18>' Map_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:124'
      declare
         Map_Result_Vec : String_Type_Vectors.Vector;
      begin
      declare
         Collection : constant Symbol_Type_Array_Access := Syms;
      begin
         for Item_0 of
               Collection.Items
         loop
   --# scope-start
               --# bind s Item_0
      --# expr-start 90 '<FieldAccess for image at nodes.lkt:124:34>' Fld 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:124'
Fld := Libadalang.Implementation.Image (Self => Item_0);
--# expr-done 90
         declare
            Item_To_Append : constant String_Type := Fld;
         begin
               Inc_Ref (Item_To_Append);
            String_Type_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   --# end
      Finalizer_Scope_1909;
         end loop;
      end;
         Map_Result := Create_String_Type_Array
           (Items_Count => Natural (String_Type_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := String_Type_Vectors.Get
              (Map_Result_Vec,
               I + String_Type_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         String_Type_Vectors.Destroy (Map_Result_Vec);
      end;
--# expr-done 91
Join_Result := Join_Strings (Sep, Map_Result); 
--# expr-done 92
         Property_Result := Join_Result;
            Inc_Ref (Property_Result);
   --# end
      Finalizer_Scope_15;
      exception
         when Exc : Precondition_Failure | Property_Error =>
                     Finalizer_Scope_15;
                     Finalizer_Scope_1909;
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Sym_Join;
--# end
--# property-start AdaNode.enclosing_compilation_unit 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:132'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Enclosing_Compilation_Unit
  (Node : Bare_Ada_Node
  )
   return Bare_Compilation_Unit
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   Property_Result : Bare_Compilation_Unit;
            procedure Finalizer_Scope_16 with Inline_Always;
      Node_Parents : Bare_Ada_Node_Array_Access;
Is_A : Boolean;
Find_Result : Bare_Ada_Node;
Cast_Result : Bare_Compilation_Unit;
            procedure Finalizer_Scope_16 is
            begin
                     Dec_Ref (Node_Parents);
            end Finalizer_Scope_16;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
         --# expr-start 96 '<Cast to CompilationUnit at nodes.lkt:133:9>' Cast_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:133'
--# expr-start 95 '<Find at nodes.lkt:133:9>' Find_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:133'
   --# expr-start 93 '<FieldAccess for parents at nodes.lkt:133:9>' Node_Parents 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:133'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
Node_Parents := Parents (Self, True); 
--# expr-done 93
      Find_Result := No_Bare_Ada_Node;
      declare
         Collection : constant Bare_Ada_Node_Array_Access := Node_Parents;
      begin
         for Item_0 of
               Collection.Items
         loop
   --# scope-start
               --# bind n Item_0
      --# expr-start 94 '<IsA CompilationUnit at nodes.lkt:133:36>' Is_A 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:133'
Is_A := Item_0 /= null 
and then Item_0.Kind in Ada_Compilation_Unit_Range; 
--# expr-done 94
      if Is_A then
         Find_Result := Item_0; 
      end if;
   --# end
               exit when Is_A;
         end loop;
      end;
--# expr-done 95
   if Find_Result = null
      or else Find_Result.Kind in Ada_Compilation_Unit_Range
   then
      Cast_Result := Find_Result;
   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;
--# expr-done 96
         Property_Result := Cast_Result;
   --# end
      Finalizer_Scope_16;
      exception
         when Exc : Precondition_Failure | Property_Error =>
                     Finalizer_Scope_16;
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Enclosing_Compilation_Unit;
--# end
--# property-start AdaNode.is_children_env 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:137'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Is_Children_Env
  (Node : Bare_Ada_Node
      ; Parent : Lexical_Env
      ; Current_Env : Lexical_Env
  )
   return Boolean
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   --# bind parent Parent
   --# bind current_env Current_Env
   Property_Result : Boolean;
            procedure Finalizer_Scope_17 with Inline_Always;
      Env_Parent : Lexical_Env := Empty_Env;
Fld : Boolean;
Is_Equal : Boolean;
If_Result : Boolean;
Is_Equal_1 : Boolean;
If_Result_1 : Boolean;
Is_Equal_2 : Boolean;
If_Result_2 : Boolean;
            procedure Finalizer_Scope_17 is
            begin
                     Dec_Ref (Env_Parent);
            end Finalizer_Scope_17;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
         --# expr-start 104 '<If at nodes.lkt:137:78>' If_Result_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:137'
--# expr-start 97 '<Eq at nodes.lkt:137:81>' Is_Equal_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:137'
Is_Equal_2 := Equivalent (Parent, Empty_Env); 
--# expr-done 97
if Is_Equal_2 then
   If_Result_2 := False;
else
   --# expr-start 103 '<If at nodes.lkt:137:78>' If_Result_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:137'
--# expr-start 98 '<Eq at nodes.lkt:138:10>' Is_Equal_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:138'
Is_Equal_1 := Equivalent (Current_Env, Parent); 
--# expr-done 98
if Is_Equal_1 then
   If_Result_1 := True;
else
   --# expr-start 102 '<If at nodes.lkt:137:78>' If_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:137'
--# expr-start 99 '<IsNull at nodes.lkt:139:10>' Is_Equal 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:139'
Is_Equal := Equivalent (Current_Env, Empty_Env); 
--# expr-done 99
if Is_Equal then
   If_Result := False;
else
   --# expr-start 101 '<FieldAccess for is_children_env at nodes.lkt:140:10>' Fld 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:140'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# expr-start 100 '<env_parent at nodes.lkt:140:39>' Env_Parent 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:140'
Env_Parent := AST_Envs.Parent (Current_Env); 
--# expr-done 100
--# property-call-start AdaNode.is_children_env
Fld := Libadalang.Implementation.Ada_Node_P_Is_Children_Env (Node => Self, Parent => Parent, Current_Env => Env_Parent);
--# end
--# expr-done 101
   If_Result := Fld;
end if;
--# expr-done 102
   If_Result_1 := If_Result;
end if;
--# expr-done 103
   If_Result_2 := If_Result_1;
end if;
--# expr-done 104
         Property_Result := If_Result_2;
   --# end
      Finalizer_Scope_17;
      exception
         when Exc : Precondition_Failure | Property_Error =>
                     Finalizer_Scope_17;
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Is_Children_Env;
--# end
--# property-start AdaNode.without_md 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:143'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Without_Md
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Internal_Entity
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind entity Ent
   Property_Result : Internal_Entity;
      Fld : Bare_Ada_Node;
Fld_1 : Internal_Entity_Info;
Fld_2 : Env_Rebindings;
Fld_3 : Internal_Entity_Info;
Fld_4 : Boolean;
New_Struct : Internal_Entity_Info;
New_Struct_1 : Internal_Entity;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
         --# expr-start 111 '<New AdaNode.entity at nodes.lkt:143:41>' New_Struct_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:143'
--# expr-start 109 '<New EntityInfo at nodes.lkt:144:30>' New_Struct 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:144'
--# expr-start 106 '<FieldAccess for from_rebound at nodes.lkt:145:78>' Fld_4 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:145'
--# expr-start 105 '<FieldAccess for info at nodes.lkt:145:78>' Fld_3 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:145'
Fld_3 := Ent.Info;
--# expr-done 105
Fld_4 := Fld_3.From_Rebound;
--# expr-done 106
--# expr-start 108 '<FieldAccess for rebindings at nodes.lkt:145:43>' Fld_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:145'
--# expr-start 107 '<FieldAccess for info at nodes.lkt:145:43>' Fld_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:145'
Fld_1 := Ent.Info;
--# expr-done 107
Fld_2 := Fld_1.Rebindings;
--# expr-done 108
New_Struct := (From_Rebound => Fld_4, Md => No_Metadata, Rebindings => Fld_2); 
--# expr-done 109
--# expr-start 110 '<FieldAccess for node at nodes.lkt:144:14>' Fld 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:144'
Fld := Ent.Node;
--# expr-done 110
New_Struct_1 := (Info => New_Struct, Node => Fld); 
--# expr-done 111
         Property_Result := New_Struct_1;
   --# end
      exception
         when Exc : Precondition_Failure | Property_Error =>
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Without_Md;
--# end
--# property-start AdaNode.get_uninstantiated_node 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:153'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Get_Uninstantiated_Node
  (Node : Bare_Ada_Node
  )
   return Internal_Entity
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   Property_Result : Internal_Entity;
      As_Entity : Internal_Entity;
Is_Null : Boolean;
If_Result : Internal_Entity;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
         --# expr-start 112 '<as_bare_entity at nodes.lkt:153:54>' If_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:153'
Is_Null := Self = null; 
if Is_Null then
   If_Result := No_Entity;
else
As_Entity := (Info => No_Entity_Info, Node => Self); 
   If_Result := As_Entity;
end if;
--# expr-done 112
         Property_Result := If_Result;
   --# end
      exception
         when Exc : Precondition_Failure | Property_Error =>
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Get_Uninstantiated_Node;
--# end
--# property-start AdaNode.complete 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:157'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Complete
  (Node : Bare_Ada_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )
   return Internal_Completion_Item_Iterator_Access
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind entity Ent
   Property_Result : Internal_Completion_Item_Iterator_Access;
            procedure Finalizer_Scope_20 with Inline_Always;
      Fld : Bare_Ada_Node;
Bound_Origin : Bare_Ada_Node;
Fld_1 : Internal_Completion_Item_Array_Access;
Fld_2 : Internal_Entity_Basic_Decl;
Is_A : Boolean;
Not_Val : Boolean;
Map_Result : Internal_Completion_Item_Array_Access;
Iter : Internal_Completion_Item_Iterator_Access;
Dyn_Var_Bind_Result : Internal_Completion_Item_Iterator_Access;
            procedure Finalizer_Scope_20 is
            begin
                     Dec_Ref (Fld_1);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Iter);
                     Dec_Ref (Dyn_Var_Bind_Result);
            end Finalizer_Scope_20;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      if Self /= null then
         Reset_Caches (Self.Unit);
            Populate_Lexical_Env_For_Unit (Self);
      end if;
      begin
   --# scope-start
         --# expr-start 120 '<bind at nodes.lkt:158:9>' Dyn_Var_Bind_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:158'
--# expr-start 113 '<FieldAccess for origin_node at nodes.lkt:158:23>' Fld 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:158'
      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start AdaNode.origin_node
Fld := Libadalang.Implementation.Ada_Node_P_Origin_Node (Node => Self);
--# end
--# expr-done 113
Bound_Origin := Fld; 
--# expr-start 119 '<FieldAccess for to_iterator at nodes.lkt:160:9>' Iter 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:160'
--# expr-start 118 '<Map at nodes.lkt:160:9>' Map_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:160'
   --# expr-start 114 '<FieldAccess for complete_items at nodes.lkt:160:9>' Fld_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:160'
      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]AdaNode.complete_items'
Fld_1 := Libadalang.Implementation.Dispatcher_Ada_Node_P_Complete_Items (Node => Ent.Node, Origin => Bound_Origin, E_Info => Ent.Info);
--# end
--# expr-done 114
      declare
         Map_Result_Vec : Internal_Completion_Item_Vectors.Vector;
      begin
      declare
         Collection : constant Internal_Completion_Item_Array_Access := Fld_1;
      begin
         for Item_0 of
               Collection.Items
         loop
   --# scope-start
               --# bind n Item_0
         --# expr-start 117 '<Not at nodes.lkt:167:20>' Not_Val 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:167'
--# expr-start 116 '<IsA SyntheticSubpDecl, SyntheticObjectDecl at nodes.lkt:167:24>' Is_A 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:167'
--# expr-start 115 '<FieldAccess for decl at nodes.lkt:167:24>' Fld_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:167'
Fld_2 := Item_0.Decl;
--# expr-done 115
Is_A := Fld_2.Node /= null 
and then Fld_2.Node.Kind in Ada_Synthetic_Subp_Decl_Range | Ada_Synthetic_Object_Decl_Range; 
--# expr-done 116
Not_Val := not (Is_A); 
--# expr-done 117
         if Not_Val then
         declare
            Item_To_Append : constant Internal_Completion_Item := Item_0;
         begin
            Internal_Completion_Item_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
         end if;
   --# end
         end loop;
      end;
         Map_Result := Create_Internal_Completion_Item_Array
           (Items_Count => Natural (Internal_Completion_Item_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Completion_Item_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Completion_Item_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Completion_Item_Vectors.Destroy (Map_Result_Vec);
      end;
--# expr-done 118
Iter := To_Iterator (Map_Result, Self.Unit.Context); 
--# expr-done 119
Dyn_Var_Bind_Result := Iter; Inc_Ref (Dyn_Var_Bind_Result);
--# expr-done 120
         Property_Result := Dyn_Var_Bind_Result;
            Inc_Ref (Property_Result);
   --# end
      Finalizer_Scope_20;
      exception
         when Exc : Precondition_Failure | Property_Error =>
                     Finalizer_Scope_20;
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Complete;
--# end
--# property-start AdaNode.complete_item_weight_matching_type 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:176'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Complete_Item_Weight_Matching_Type
  (Node : Bare_Ada_Node
      ; Item : Internal_Entity_Basic_Decl
      ; Name : Internal_Entity_Name
      ; Origin : Bare_Ada_Node
  )
   return Integer
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   --# bind item Item
   --# bind name Name
   --# bind origin Origin
   Property_Result : Integer;
      Local_Te_Not_Null : Boolean;
Local_Td : Internal_Entity_Base_Type_Decl;
Var_Expr_0 : Internal_Entity_Base_Type_Decl;
Var_Expr_1 : Internal_Entity_Type_Expr;
Var_Expr_2 : Internal_Entity_Basic_Decl;
Var_Expr_3 : Internal_Entity_Type_Expr;
Var_Expr_4 : Internal_Entity_Basic_Decl;
Var_Expr_5 : Internal_Entity_Type_Expr;
Fld : Internal_Entity_Type_Expr;
Is_Null : Boolean;
Not_Val : Boolean;
Fld_1 : Internal_Entity_Type_Expr;
Fld_2 : Internal_Entity_Base_Type_Decl;
Scope_Result : Internal_Entity_Base_Type_Decl;
Result_Var : Internal_Entity_Base_Type_Decl;
Fld_3 : Internal_Entity_Base_Type_Decl;
Bound_Imprecise_Fallback : Boolean;
Fld_4 : Internal_Entity_Basic_Decl;
Fld_5 : Internal_Entity_Type_Expr;
Fld_6 : Internal_Entity_Base_Type_Decl;
Fld_7 : Boolean;
Scope_Result_1 : Boolean;
Result_Var_1 : Boolean;
Scope_Result_2 : Boolean;
Result_Var_2 : Boolean;
If_Result : Boolean;
Dyn_Var_Bind_Result : Boolean;
If_Result_1 : Integer;
Fld_8 : Internal_Entity_Basic_Decl;
Fld_9 : Internal_Entity_Type_Expr;
Fld_10 : Internal_Entity_Base_Type_Decl;
Fld_11 : Boolean;
Scope_Result_3 : Boolean;
Result_Var_3 : Boolean;
Scope_Result_4 : Boolean;
Result_Var_4 : Boolean;
If_Result_2 : Boolean;
If_Result_3 : Integer;
Scope_Result_5 : Integer;
Result_Var_5 : Integer;
Let_Result : Integer;
Let_Result_1 : Integer;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      if Self /= null then
         Reset_Caches (Self.Unit);
      end if;
      begin
   --# scope-start
         --# expr-start 139 '<Let at nodes.lkt:177:9>' Let_Result_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:177'
--# scope-start
--# expr-start 123 '<Not at nodes.lkt:177:27>' Not_Val 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:177'
--# expr-start 122 '<IsNull at nodes.lkt:177:31>' Is_Null 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:177'
--# expr-start 121 '<FieldAccess for type_expression at nodes.lkt:177:31>' Fld 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:177'
      if Item.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]BasicDecl.type_expression'
Fld := Libadalang.Implementation.Dispatcher_Basic_Decl_P_Type_Expression (Node => Item.Node, E_Info => Item.Info);
--# end
--# expr-done 121
Is_Null := Fld.Node = null; 
--# expr-done 122
Not_Val := not (Is_Null); 
--# expr-done 123
Local_Te_Not_Null := Not_Val; 
--# bind te_not_null Local_Te_Not_Null
--# expr-start 138 '<Let at nodes.lkt:178:9>' Let_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:178'
--# scope-start
--# expr-start 124 '<FieldAccess for type_expression at nodes.lkt:178:18>' Fld_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:178'
      if Item.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]BasicDecl.type_expression'
Fld_1 := Libadalang.Implementation.Dispatcher_Basic_Decl_P_Type_Expression (Node => Item.Node, E_Info => Item.Info);
--# end
--# expr-done 124
Var_Expr_1 := Fld_1;
if Var_Expr_1 /= No_Entity_Type_Expr then
   --# scope-start
--# expr-start 125 '<FieldAccess for designated_type_decl at nodes.lkt:178:18>' Fld_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:178'
      if Var_Expr_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start TypeExpr.designated_type_decl
Fld_2 := Libadalang.Implementation.Type_Expr_P_Designated_Type_Decl (Node => Var_Expr_1.Node, E_Info => Var_Expr_1.Info);
--# end
--# expr-done 125
Scope_Result := Fld_2;
   --# end
   Result_Var := Scope_Result;
else
   Result_Var := No_Entity_Base_Type_Decl;
end if;
Local_Td := Result_Var; 
--# bind td Local_Td
--# expr-start 126 '<FieldAccess for expr_type at nodes.lkt:181:9>' Fld_3 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:181'
      if Item.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]BasicDecl.expr_type'
Fld_3 := Libadalang.Implementation.Dispatcher_Basic_Decl_P_Expr_Type (Node => Item.Node, Origin => Origin, E_Info => Item.Info);
--# end
--# expr-done 126
Var_Expr_0 := Fld_3;
if Var_Expr_0 /= No_Entity_Base_Type_Decl then
   --# scope-start
   --# bind _ Var_Expr_0
--# expr-start 137 '<If at nodes.lkt:184:13>' If_Result_3 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:184'
--# expr-start 127 '<FieldAccess for referenced_decl at nodes.lkt:184:17>' Fld_8 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:184'
      if Name.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start Name.referenced_decl
Fld_8 := Libadalang.Implementation.Name_P_Referenced_Decl (Node => Name.Node, Imprecise_Fallback => False, E_Info => Name.Info);
--# end
--# expr-done 127
Var_Expr_2 := Fld_8;
if Var_Expr_2 /= No_Entity_Basic_Decl then
   --# scope-start
--# expr-start 128 '<FieldAccess for type_expression at nodes.lkt:184:17>' Fld_9 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:184'
      if Var_Expr_2.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]BasicDecl.type_expression'
Fld_9 := Libadalang.Implementation.Dispatcher_Basic_Decl_P_Type_Expression (Node => Var_Expr_2.Node, E_Info => Var_Expr_2.Info);
--# end
--# expr-done 128
Var_Expr_3 := Fld_9;
if Var_Expr_3 /= No_Entity_Type_Expr then
   --# scope-start
--# expr-start 130 '<FieldAccess for matching_assign_type at nodes.lkt:184:17>' Fld_11 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:184'
--# expr-start 129 '<FieldAccess for designated_type_decl at nodes.lkt:184:17>' Fld_10 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:184'
      if Var_Expr_3.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start TypeExpr.designated_type_decl
Fld_10 := Libadalang.Implementation.Type_Expr_P_Designated_Type_Decl (Node => Var_Expr_3.Node, E_Info => Var_Expr_3.Info);
--# end
--# expr-done 129
      if Fld_10.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start BaseTypeDecl.matching_assign_type
Fld_11 := Libadalang.Implementation.Base_Type_Decl_P_Matching_Assign_Type (Node => Fld_10.Node, Expected_Type => Local_Td, Origin => Origin, E_Info => Fld_10.Info);
--# end
--# expr-done 130
Scope_Result_3 := Fld_11;
   --# end
   Result_Var_3 := Scope_Result_3;
else
   Result_Var_3 := False;
end if;
Scope_Result_4 := Result_Var_3;
   --# end
   Result_Var_4 := Scope_Result_4;
else
   Result_Var_4 := False;
end if;
if Result_Var_4 then
   If_Result_2 := Local_Te_Not_Null;
else
   If_Result_2 := False;
end if;
if If_Result_2 then
   If_Result_3 := 100;
else
   --# expr-start 136 '<If at nodes.lkt:184:13>' If_Result_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:184'
--# expr-start 135 '<bind at nodes.lkt:187:17>' Dyn_Var_Bind_Result 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:187'
Bound_Imprecise_Fallback := True; 
--# expr-start 131 '<FieldAccess for referenced_decl at nodes.lkt:189:17>' Fld_4 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:189'
      if Name.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start Name.referenced_decl
Fld_4 := Libadalang.Implementation.Name_P_Referenced_Decl (Node => Name.Node, Imprecise_Fallback => Bound_Imprecise_Fallback, E_Info => Name.Info);
--# end
--# expr-done 131
Var_Expr_4 := Fld_4;
if Var_Expr_4 /= No_Entity_Basic_Decl then
   --# scope-start
--# expr-start 132 '<FieldAccess for type_expression at nodes.lkt:189:17>' Fld_5 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:189'
      if Var_Expr_4.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start '[dispatcher]BasicDecl.type_expression'
Fld_5 := Libadalang.Implementation.Dispatcher_Basic_Decl_P_Type_Expression (Node => Var_Expr_4.Node, E_Info => Var_Expr_4.Info);
--# end
--# expr-done 132
Var_Expr_5 := Fld_5;
if Var_Expr_5 /= No_Entity_Type_Expr then
   --# scope-start
--# expr-start 134 '<FieldAccess for matching_assign_type at nodes.lkt:189:17>' Fld_7 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:189'
--# expr-start 133 '<FieldAccess for designated_type_decl at nodes.lkt:189:17>' Fld_6 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:189'
      if Var_Expr_5.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start TypeExpr.designated_type_decl
Fld_6 := Libadalang.Implementation.Type_Expr_P_Designated_Type_Decl (Node => Var_Expr_5.Node, E_Info => Var_Expr_5.Info);
--# end
--# expr-done 133
      if Fld_6.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;
--# property-call-start BaseTypeDecl.matching_assign_type
Fld_7 := Libadalang.Implementation.Base_Type_Decl_P_Matching_Assign_Type (Node => Fld_6.Node, Expected_Type => Local_Td, Origin => Origin, E_Info => Fld_6.Info);
--# end
--# expr-done 134
Scope_Result_1 := Fld_7;
   --# end
   Result_Var_1 := Scope_Result_1;
else
   Result_Var_1 := False;
end if;
Scope_Result_2 := Result_Var_1;
   --# end
   Result_Var_2 := Scope_Result_2;
else
   Result_Var_2 := False;
end if;
if Result_Var_2 then
   If_Result := Local_Te_Not_Null;
else
   If_Result := False;
end if;
Dyn_Var_Bind_Result := If_Result; 
--# expr-done 135
if Dyn_Var_Bind_Result then
   If_Result_1 := 70;
else
   If_Result_1 := 50;
end if;
--# expr-done 136
   If_Result_3 := If_Result_1;
end if;
--# expr-done 137
Scope_Result_5 := If_Result_3;
   --# end
   Result_Var_5 := Scope_Result_5;
else
   Result_Var_5 := 0;
end if;
Let_Result := Result_Var_5; 
--# end
--# expr-done 138
Let_Result_1 := Let_Result; 
--# end
--# expr-done 139
         Property_Result := Let_Result_1;
   --# end
      exception
         when Exc : Precondition_Failure | Property_Error =>
            raise;
      end;
   return Property_Result;
end Ada_Node_P_Complete_Item_Weight_Matching_Type;
--# end
--# property-start AdaNode.valid_keywords 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:202'
pragma Warnings (Off, "is not referenced");
 function Ada_Node_P_Valid_Keywords
  (Node : Bare_Ada_Node
  )
   return Symbol_Type_Array_Access
is
   Self : Bare_Ada_Node  := Bare_Ada_Node (Node);
      --# bind self Self
   Property_Result : Symbol_Type_Array_Access;
            procedure Finalizer_Scope_22 with Inline_Always;
      Sym : Symbol_Type;
Sym_1 : Symbol_Type;
Sym_2 : Symbol_Type;
Sym_3 : Symbol_Type;
Sym_4 : Symbol_Type;
Sym_5 : Symbol_Type;
Sym_6 : Symbol_Type;
Sym_7 : Symbol_Type;
Sym_8 : Symbol_Type;
Sym_9 : Symbol_Type;
Sym_10 : Symbol_Type;
Sym_11 : Symbol_Type;
Sym_12 : Symbol_Type;
Sym_13 : Symbol_Type;
Sym_14 : Symbol_Type;
Sym_15 : Symbol_Type;
Sym_16 : Symbol_Type;
Sym_17 : Symbol_Type;
Sym_18 : Symbol_Type;
Sym_19 : Symbol_Type;
Sym_20 : Symbol_Type;
Sym_21 : Symbol_Type;
Sym_22 : Symbol_Type;
Sym_23 : Symbol_Type;
Sym_24 : Symbol_Type;
Sym_25 : Symbol_Type;
Sym_26 : Symbol_Type;
Sym_27 : Symbol_Type;
Sym_28 : Symbol_Type;
Sym_29 : Symbol_Type;
Sym_30 : Symbol_Type;
Sym_31 : Symbol_Type;
Sym_32 : Symbol_Type;
Sym_33 : Symbol_Type;
Sym_34 : Symbol_Type;
Sym_35 : Symbol_Type;
Sym_36 : Symbol_Type;
Sym_37 : Symbol_Type;
Sym_38 : Symbol_Type;
Sym_39 : Symbol_Type;
Sym_40 : Symbol_Type;
Sym_41 : Symbol_Type;
Sym_42 : Symbol_Type;
Sym_43 : Symbol_Type;
Sym_44 : Symbol_Type;
Sym_45 : Symbol_Type;
Sym_46 : Symbol_Type;
Sym_47 : Symbol_Type;
Sym_48 : Symbol_Type;
Sym_49 : Symbol_Type;
Sym_50 : Symbol_Type;
Sym_51 : Symbol_Type;
Sym_52 : Symbol_Type;
Sym_53 : Symbol_Type;
Sym_54 : Symbol_Type;
Sym_55 : Symbol_Type;
Sym_56 : Symbol_Type;
Sym_57 : Symbol_Type;
Sym_58 : Symbol_Type;
Sym_59 : Symbol_Type;
Sym_60 : Symbol_Type;
Sym_61 : Symbol_Type;
Sym_62 : Symbol_Type;
Sym_63 : Symbol_Type;
Sym_64 : Symbol_Type;
Sym_65 : Symbol_Type;
Sym_66 : Symbol_Type;
Sym_67 : Symbol_Type;
Sym_68 : Symbol_Type;
Sym_69 : Symbol_Type;
Sym_70 : Symbol_Type;
Sym_71 : Symbol_Type;
Sym_72 : Symbol_Type;
Array_Lit : Symbol_Type_Array_Access;
            procedure Finalizer_Scope_22 is
            begin
                     Dec_Ref (Array_Lit);
            end Finalizer_Scope_22;
begin
   --# property-body-start
   pragma Assert (Self = Node);
      begin
   --# scope-start
         --# expr-start 213 '<ArrayLiteral at nodes.lkt:203:9>' Array_Lit 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
--# expr-start 140 '<Symbol abort at nodes.lkt:203:10>' Sym 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Abort); 
--# expr-done 140
--# expr-start 141 '<Symbol abs at nodes.lkt:203:20>' Sym_1 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_1 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Abs_316); 
--# expr-done 141
--# expr-start 142 '<Symbol abstract at nodes.lkt:203:28>' Sym_2 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_2 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Abstract); 
--# expr-done 142
--# expr-start 143 '<Symbol accept at nodes.lkt:203:41>' Sym_3 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_3 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Accept); 
--# expr-done 143
--# expr-start 144 '<Symbol access at nodes.lkt:203:52>' Sym_4 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_4 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Access_319); 
--# expr-done 144
--# expr-start 145 '<Symbol aliased at nodes.lkt:203:63>' Sym_5 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_5 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Aliased); 
--# expr-done 145
--# expr-start 146 '<Symbol all at nodes.lkt:203:75>' Sym_6 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_6 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_All); 
--# expr-done 146
--# expr-start 147 '<Symbol and at nodes.lkt:203:83>' Sym_7 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_7 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_And_324); 
--# expr-done 147
--# expr-start 148 '<Symbol array at nodes.lkt:203:91>' Sym_8 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_8 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Array); 
--# expr-done 148
--# expr-start 149 '<Symbol at at nodes.lkt:203:101>' Sym_9 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_9 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_At); 
--# expr-done 149
--# expr-start 150 '<Symbol begin at nodes.lkt:203:108>' Sym_10 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_10 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Begin); 
--# expr-done 150
--# expr-start 151 '<Symbol body at nodes.lkt:203:118>' Sym_11 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_11 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Body); 
--# expr-done 151
--# expr-start 152 '<Symbol case at nodes.lkt:203:127>' Sym_12 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_12 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Case); 
--# expr-done 152
--# expr-start 153 '<Symbol constant at nodes.lkt:203:136>' Sym_13 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_13 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Constant); 
--# expr-done 153
--# expr-start 154 '<Symbol declare at nodes.lkt:203:149>' Sym_14 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_14 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Declare); 
--# expr-done 154
--# expr-start 155 '<Symbol delay at nodes.lkt:203:161>' Sym_15 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_15 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Delay); 
--# expr-done 155
--# expr-start 156 '<Symbol delta at nodes.lkt:203:171>' Sym_16 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_16 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Delta_338); 
--# expr-done 156
--# expr-start 157 '<Symbol digits at nodes.lkt:203:181>' Sym_17 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_17 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Digits_339); 
--# expr-done 157
--# expr-start 158 '<Symbol do at nodes.lkt:203:192>' Sym_18 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_18 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Do); 
--# expr-done 158
--# expr-start 159 '<Symbol else at nodes.lkt:203:199>' Sym_19 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_19 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Else); 
--# expr-done 159
--# expr-start 160 '<Symbol elsif at nodes.lkt:203:208>' Sym_20 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_20 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Elsif); 
--# expr-done 160
--# expr-start 161 '<Symbol end at nodes.lkt:203:218>' Sym_21 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_21 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_End); 
--# expr-done 161
--# expr-start 162 '<Symbol entry at nodes.lkt:203:226>' Sym_22 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_22 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Entry); 
--# expr-done 162
--# expr-start 163 '<Symbol exception at nodes.lkt:203:236>' Sym_23 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_23 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Exception); 
--# expr-done 163
--# expr-start 164 '<Symbol exit at nodes.lkt:203:250>' Sym_24 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_24 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Exit); 
--# expr-done 164
--# expr-start 165 '<Symbol for at nodes.lkt:203:259>' Sym_25 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_25 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_For); 
--# expr-done 165
--# expr-start 166 '<Symbol function at nodes.lkt:203:267>' Sym_26 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_26 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Function); 
--# expr-done 166
--# expr-start 167 '<Symbol generic at nodes.lkt:203:280>' Sym_27 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_27 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Generic); 
--# expr-done 167
--# expr-start 168 '<Symbol goto at nodes.lkt:203:292>' Sym_28 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_28 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Goto); 
--# expr-done 168
--# expr-start 169 '<Symbol if at nodes.lkt:203:301>' Sym_29 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_29 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_If); 
--# expr-done 169
--# expr-start 170 '<Symbol in at nodes.lkt:203:308>' Sym_30 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_30 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_In); 
--# expr-done 170
--# expr-start 171 '<Symbol interface at nodes.lkt:203:315>' Sym_31 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_31 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Interface_362); 
--# expr-done 171
--# expr-start 172 '<Symbol is at nodes.lkt:203:329>' Sym_32 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_32 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Is); 
--# expr-done 172
--# expr-start 173 '<Symbol limited at nodes.lkt:203:336>' Sym_33 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_33 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Limited); 
--# expr-done 173
--# expr-start 174 '<Symbol loop at nodes.lkt:203:348>' Sym_34 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_34 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Loop); 
--# expr-done 174
--# expr-start 175 '<Symbol mod at nodes.lkt:203:357>' Sym_35 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_35 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Mod_372); 
--# expr-done 175
--# expr-start 176 '<Symbol new at nodes.lkt:203:365>' Sym_36 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_36 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_New); 
--# expr-done 176
--# expr-start 177 '<Symbol not at nodes.lkt:203:373>' Sym_37 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_37 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Not_375); 
--# expr-done 177
--# expr-start 178 '<Symbol null at nodes.lkt:203:381>' Sym_38 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_38 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Null); 
--# expr-done 178
--# expr-start 179 '<Symbol others at nodes.lkt:203:390>' Sym_39 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_39 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Others); 
--# expr-done 179
--# expr-start 180 '<Symbol out at nodes.lkt:203:401>' Sym_40 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_40 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Out); 
--# expr-done 180
--# expr-start 181 '<Symbol of at nodes.lkt:203:409>' Sym_41 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_41 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Of); 
--# expr-done 181
--# expr-start 182 '<Symbol or at nodes.lkt:203:416>' Sym_42 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_42 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Or_378); 
--# expr-done 182
--# expr-start 183 '<Symbol overriding at nodes.lkt:203:423>' Sym_43 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_43 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Overriding); 
--# expr-done 183
--# expr-start 184 '<Symbol package at nodes.lkt:203:438>' Sym_44 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_44 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Package); 
--# expr-done 184
--# expr-start 185 '<Symbol pragma at nodes.lkt:203:450>' Sym_45 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_45 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Pragma); 
--# expr-done 185
--# expr-start 186 '<Symbol private at nodes.lkt:203:461>' Sym_46 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_46 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Private); 
--# expr-done 186
--# expr-start 187 '<Symbol procedure at nodes.lkt:203:473>' Sym_47 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_47 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Procedure); 
--# expr-done 187
--# expr-start 188 '<Symbol protected at nodes.lkt:203:487>' Sym_48 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_48 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Protected); 
--# expr-done 188
--# expr-start 189 '<Symbol raise at nodes.lkt:203:501>' Sym_49 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_49 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Raise); 
--# expr-done 189
--# expr-start 190 '<Symbol range at nodes.lkt:203:511>' Sym_50 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_50 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Range_392); 
--# expr-done 190
--# expr-start 191 '<Symbol record at nodes.lkt:203:521>' Sym_51 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_51 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Record); 
--# expr-done 191
--# expr-start 192 '<Symbol rem at nodes.lkt:203:532>' Sym_52 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_52 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Rem_395); 
--# expr-done 192
--# expr-start 193 '<Symbol renames at nodes.lkt:203:540>' Sym_53 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_53 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Renames); 
--# expr-done 193
--# expr-start 194 '<Symbol requeue at nodes.lkt:203:552>' Sym_54 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_54 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Requeue); 
--# expr-done 194
--# expr-start 195 '<Symbol return at nodes.lkt:203:564>' Sym_55 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_55 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Return); 
--# expr-done 195
--# expr-start 196 '<Symbol reverse at nodes.lkt:203:575>' Sym_56 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_56 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Reverse); 
--# expr-done 196
--# expr-start 197 '<Symbol select at nodes.lkt:203:587>' Sym_57 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_57 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Select); 
--# expr-done 197
--# expr-start 198 '<Symbol separate at nodes.lkt:203:598>' Sym_58 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_58 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Separate); 
--# expr-done 198
--# expr-start 199 '<Symbol some at nodes.lkt:203:611>' Sym_59 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_59 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Some); 
--# expr-done 199
--# expr-start 200 '<Symbol subtype at nodes.lkt:203:620>' Sym_60 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_60 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Subtype); 
--# expr-done 200
--# expr-start 201 '<Symbol synchronized at nodes.lkt:203:632>' Sym_61 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_61 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Synchronized); 
--# expr-done 201
--# expr-start 202 '<Symbol tagged at nodes.lkt:203:649>' Sym_62 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_62 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Tagged); 
--# expr-done 202
--# expr-start 203 '<Symbol task at nodes.lkt:203:660>' Sym_63 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_63 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Task); 
--# expr-done 203
--# expr-start 204 '<Symbol terminate at nodes.lkt:203:669>' Sym_64 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_64 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Terminate); 
--# expr-done 204
--# expr-start 205 '<Symbol then at nodes.lkt:203:683>' Sym_65 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_65 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Then); 
--# expr-done 205
--# expr-start 206 '<Symbol type at nodes.lkt:203:692>' Sym_66 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_66 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Type); 
--# expr-done 206
--# expr-start 207 '<Symbol until at nodes.lkt:203:701>' Sym_67 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_67 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Until); 
--# expr-done 207
--# expr-start 208 '<Symbol use at nodes.lkt:203:711>' Sym_68 'D:\M\B\src\build-MINGW64\ada\nodes.lkt:203'
Sym_68 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols)