










package com.adacore.liblktlang;

import java.util.Arrays;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.Objects;
import java.util.Optional;

import java.lang.StringBuilder;
import java.lang.Iterable;
import java.lang.reflect.Method;

import java.math.BigInteger;

import java.io.File;
import java.nio.ByteOrder;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;

import com.oracle.truffle.api.CompilerDirectives;
import com.oracle.truffle.api.strings.TruffleString;

import org.graalvm.nativeimage.CurrentIsolate;
import org.graalvm.nativeimage.ImageInfo;
import org.graalvm.nativeimage.IsolateThread;
import org.graalvm.nativeimage.StackValue;
import org.graalvm.nativeimage.UnmanagedMemory;
import org.graalvm.nativeimage.c.CContext;
import org.graalvm.nativeimage.c.function.CEntryPoint;
import org.graalvm.nativeimage.c.function.CEntryPointLiteral;
import org.graalvm.nativeimage.c.function.CFunction;
import org.graalvm.nativeimage.c.function.CFunctionPointer;
import org.graalvm.nativeimage.c.function.InvokeCFunctionPointer;
import org.graalvm.nativeimage.c.struct.*;
import org.graalvm.nativeimage.c.type.*;
import org.graalvm.word.PointerBase;
import org.graalvm.word.Pointer;
import org.graalvm.word.UnsignedWord;
import org.graalvm.word.WordFactory;


/*
====================

This is the Java bindings of the liblktlang API.
You can use all functionalities of the library in a Java environment.
Those bindings call the native library using JNI and Native-Image C API.

====================
*/
public final class Liblktlang {

    // ==========
    // Native entry points
    // ==========

    /**
     * This method is the only valid callback to pass to a native event
     * handler for unit requests events.
     * This method will dispatch the execution according to the passed
     * analysis context.
     */
    @CEntryPoint
    static void unitRequested(
        final IsolateThread thread,
        final AnalysisContextNative contextNative,
        final TextNative nameNative,
        final AnalysisUnitNative fromNative,
        final byte foundNative,
        final byte isNotFoundErrorNative
    ) {
        try(
            final AnalysisContext context = AnalysisContext.wrap(
                contextNative
            );
            final Text text = Text.wrap(nameNative)
        ) {
            // Get the callback from the context event handler
            final EventHandler.UnitRequestedCallback callback = context
                .getEventHandler()
                .getUnitRequestCallback();

            // Call the callback
            if(callback != null) {
                callback.invoke(
                    context,
                    text.getContent(),
                    AnalysisUnit.wrap(fromNative),
                    foundNative != 0,
                    isNotFoundErrorNative != 0
                );
            }
        }
    }

    /**
     * This method is the only valid callback to pass to a native event
     * handler for unit parsing events.
     * This method will dispatch the execution according to the passed
     * analysis context.
     */
    @CEntryPoint
    static void unitParsed(
        final IsolateThread thread,
        final AnalysisContextNative contextNative,
        final AnalysisUnitNative unitNative,
        final byte reparsedNative
    ) {
        try(
            final AnalysisContext context = AnalysisContext.wrap(
                contextNative
            )
        ) {
            // Get the callback from the context event handler
            final EventHandler.UnitParsedCallback callback = context
                .getEventHandler()
                .getUnitParsedCallback();

            // Call the callback
            if(callback != null) {
                callback.invoke(
                    context,
                    AnalysisUnit.wrap(unitNative),
                    reparsedNative != 0
                );
            }
        }
    }

    // ==========
    // Static values
    // ==========

    /** The default grammar rule to parse the inputs. */
    private static final GrammarRule DEFAULT_GRAMMAR_RULE =
        GrammarRule.MAIN_RULE_RULE;

    /** The os name in lower case. */
    private static final String OS =
            System.getProperty("os.name").toLowerCase();

    /** The byte order of the system. */
    private static final ByteOrder BYTE_ORDER = ByteOrder.nativeOrder();

    /** The node to convert a Java string to a truffle string */
    private static final TruffleString.FromJavaStringNode fromJavaStringNode =
        TruffleString.FromJavaStringNode.create();

    /** The node to convert a truffle string to a Java string. */
    private static final TruffleString.ToJavaStringNode toJavaStringNode =
        TruffleString.ToJavaStringNode.create();

    /** The node to convert a byte array to a truffle string. */
    private static final TruffleString.FromByteArrayNode fromByteArrayNode =
        TruffleString.FromByteArrayNode.create();

    /** The node to convert a truffle string to a byte array. */
    private static final TruffleString.CopyToByteArrayNode toByteArrayNode =
        TruffleString.CopyToByteArrayNode.create();

    /** A map to store node descriptions associated to their camel name. */
    public static final Map<String, Reflection.Node>
        NODE_DESCRIPTION_MAP = new HashMap<>();

    static {
        // Fill the node description map and set the node kind descriptions
        NODE_DESCRIPTION_MAP.put(
            "LktNode",
            LktNode.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BaseLexerCaseRuleAlt",
            BaseLexerCaseRuleAlt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LexerCaseRuleCondAlt",
            LexerCaseRuleCondAlt.description
        );
        NodeKind.LEXER_CASE_RULE_COND_ALT.setDescription(
            LexerCaseRuleCondAlt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LexerCaseRuleDefaultAlt",
            LexerCaseRuleDefaultAlt.description
        );
        NodeKind.LEXER_CASE_RULE_DEFAULT_ALT.setDescription(
            LexerCaseRuleDefaultAlt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BlockStringLine",
            BlockStringLine.description
        );
        NodeKind.BLOCK_STRING_LINE.setDescription(
            BlockStringLine.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ClassQualifier",
            ClassQualifier.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ClassQualifierAbsent",
            ClassQualifierAbsent.description
        );
        NodeKind.CLASS_QUALIFIER_ABSENT.setDescription(
            ClassQualifierAbsent.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ClassQualifierPresent",
            ClassQualifierPresent.description
        );
        NodeKind.CLASS_QUALIFIER_PRESENT.setDescription(
            ClassQualifierPresent.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Decl",
            Decl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BaseGrammarRuleDecl",
            BaseGrammarRuleDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarRuleDecl",
            GrammarRuleDecl.description
        );
        NodeKind.GRAMMAR_RULE_DECL.setDescription(
            GrammarRuleDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SyntheticLexerDecl",
            SyntheticLexerDecl.description
        );
        NodeKind.SYNTHETIC_LEXER_DECL.setDescription(
            SyntheticLexerDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BaseValDecl",
            BaseValDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NodeDecl",
            NodeDecl.description
        );
        NodeKind.NODE_DECL.setDescription(
            NodeDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SelfDecl",
            SelfDecl.description
        );
        NodeKind.SELF_DECL.setDescription(
            SelfDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "UserValDecl",
            UserValDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumLitDecl",
            EnumLitDecl.description
        );
        NodeKind.ENUM_LIT_DECL.setDescription(
            EnumLitDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ExplicitlyTypedDecl",
            ExplicitlyTypedDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ComponentDecl",
            ComponentDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FieldDecl",
            FieldDecl.description
        );
        NodeKind.FIELD_DECL.setDescription(
            FieldDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FunArgDecl",
            FunArgDecl.description
        );
        NodeKind.FUN_ARG_DECL.setDescription(
            FunArgDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LambdaArgDecl",
            LambdaArgDecl.description
        );
        NodeKind.LAMBDA_ARG_DECL.setDescription(
            LambdaArgDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DynVarDecl",
            DynVarDecl.description
        );
        NodeKind.DYN_VAR_DECL.setDescription(
            DynVarDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "MatchValDecl",
            MatchValDecl.description
        );
        NodeKind.MATCH_VAL_DECL.setDescription(
            MatchValDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ValDecl",
            ValDecl.description
        );
        NodeKind.VAL_DECL.setDescription(
            ValDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FunDecl",
            FunDecl.description
        );
        NodeKind.FUN_DECL.setDescription(
            FunDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnvSpecDecl",
            EnvSpecDecl.description
        );
        NodeKind.ENV_SPEC_DECL.setDescription(
            EnvSpecDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GenericDecl",
            GenericDecl.description
        );
        NodeKind.GENERIC_DECL.setDescription(
            GenericDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarDecl",
            GrammarDecl.description
        );
        NodeKind.GRAMMAR_DECL.setDescription(
            GrammarDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LexerDecl",
            LexerDecl.description
        );
        NodeKind.LEXER_DECL.setDescription(
            LexerDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LexerFamilyDecl",
            LexerFamilyDecl.description
        );
        NodeKind.LEXER_FAMILY_DECL.setDescription(
            LexerFamilyDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TypeDecl",
            TypeDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumClassAltDecl",
            EnumClassAltDecl.description
        );
        NodeKind.ENUM_CLASS_ALT_DECL.setDescription(
            EnumClassAltDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FunctionType",
            FunctionType.description
        );
        NodeKind.FUNCTION_TYPE.setDescription(
            FunctionType.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GenericFormalTypeDecl",
            GenericFormalTypeDecl.description
        );
        NodeKind.GENERIC_FORMAL_TYPE_DECL.setDescription(
            GenericFormalTypeDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "InstantiatedGenericType",
            InstantiatedGenericType.description
        );
        NodeKind.INSTANTIATED_GENERIC_TYPE.setDescription(
            InstantiatedGenericType.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NamedTypeDecl",
            NamedTypeDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BasicClassDecl",
            BasicClassDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ClassDecl",
            ClassDecl.description
        );
        NodeKind.CLASS_DECL.setDescription(
            ClassDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumClassDecl",
            EnumClassDecl.description
        );
        NodeKind.ENUM_CLASS_DECL.setDescription(
            EnumClassDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumTypeDecl",
            EnumTypeDecl.description
        );
        NodeKind.ENUM_TYPE_DECL.setDescription(
            EnumTypeDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "StructDecl",
            StructDecl.description
        );
        NodeKind.STRUCT_DECL.setDescription(
            StructDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TraitDecl",
            TraitDecl.description
        );
        NodeKind.TRAIT_DECL.setDescription(
            TraitDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DeclAnnotation",
            DeclAnnotation.description
        );
        NodeKind.DECL_ANNOTATION.setDescription(
            DeclAnnotation.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DeclAnnotationParams",
            DeclAnnotationParams.description
        );
        NodeKind.DECL_ANNOTATION_PARAMS.setDescription(
            DeclAnnotationParams.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ElsifBranch",
            ElsifBranch.description
        );
        NodeKind.ELSIF_BRANCH.setDescription(
            ElsifBranch.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumClassCase",
            EnumClassCase.description
        );
        NodeKind.ENUM_CLASS_CASE.setDescription(
            EnumClassCase.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ExcludesNull",
            ExcludesNull.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ExcludesNullAbsent",
            ExcludesNullAbsent.description
        );
        NodeKind.EXCLUDES_NULL_ABSENT.setDescription(
            ExcludesNullAbsent.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ExcludesNullPresent",
            ExcludesNullPresent.description
        );
        NodeKind.EXCLUDES_NULL_PRESENT.setDescription(
            ExcludesNullPresent.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Expr",
            Expr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "AnyOf",
            AnyOf.description
        );
        NodeKind.ANY_OF.setDescription(
            AnyOf.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ArrayLiteral",
            ArrayLiteral.description
        );
        NodeKind.ARRAY_LITERAL.setDescription(
            ArrayLiteral.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BaseDotExpr",
            BaseDotExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DotExpr",
            DotExpr.description
        );
        NodeKind.DOT_EXPR.setDescription(
            DotExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NullCondDottedName",
            NullCondDottedName.description
        );
        NodeKind.NULL_COND_DOTTED_NAME.setDescription(
            NullCondDottedName.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BinOp",
            BinOp.description
        );
        NodeKind.BIN_OP.setDescription(
            BinOp.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BlockExpr",
            BlockExpr.description
        );
        NodeKind.BLOCK_EXPR.setDescription(
            BlockExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "CallExpr",
            CallExpr.description
        );
        NodeKind.CALL_EXPR.setDescription(
            CallExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "CastExpr",
            CastExpr.description
        );
        NodeKind.CAST_EXPR.setDescription(
            CastExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ErrorOnNull",
            ErrorOnNull.description
        );
        NodeKind.ERROR_ON_NULL.setDescription(
            ErrorOnNull.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GenericInstantiation",
            GenericInstantiation.description
        );
        NodeKind.GENERIC_INSTANTIATION.setDescription(
            GenericInstantiation.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarExpr",
            GrammarExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarCut",
            GrammarCut.description
        );
        NodeKind.GRAMMAR_CUT.setDescription(
            GrammarCut.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarDiscard",
            GrammarDiscard.description
        );
        NodeKind.GRAMMAR_DISCARD.setDescription(
            GrammarDiscard.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarDontSkip",
            GrammarDontSkip.description
        );
        NodeKind.GRAMMAR_DONT_SKIP.setDescription(
            GrammarDontSkip.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarList",
            GrammarList.description
        );
        NodeKind.GRAMMAR_LIST.setDescription(
            GrammarList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarNull",
            GrammarNull.description
        );
        NodeKind.GRAMMAR_NULL.setDescription(
            GrammarNull.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarOpt",
            GrammarOpt.description
        );
        NodeKind.GRAMMAR_OPT.setDescription(
            GrammarOpt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarOptError",
            GrammarOptError.description
        );
        NodeKind.GRAMMAR_OPT_ERROR.setDescription(
            GrammarOptError.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarOptErrorGroup",
            GrammarOptErrorGroup.description
        );
        NodeKind.GRAMMAR_OPT_ERROR_GROUP.setDescription(
            GrammarOptErrorGroup.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarOptGroup",
            GrammarOptGroup.description
        );
        NodeKind.GRAMMAR_OPT_GROUP.setDescription(
            GrammarOptGroup.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarOrExpr",
            GrammarOrExpr.description
        );
        NodeKind.GRAMMAR_OR_EXPR.setDescription(
            GrammarOrExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarPick",
            GrammarPick.description
        );
        NodeKind.GRAMMAR_PICK.setDescription(
            GrammarPick.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarImplicitPick",
            GrammarImplicitPick.description
        );
        NodeKind.GRAMMAR_IMPLICIT_PICK.setDescription(
            GrammarImplicitPick.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarPredicate",
            GrammarPredicate.description
        );
        NodeKind.GRAMMAR_PREDICATE.setDescription(
            GrammarPredicate.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarRuleRef",
            GrammarRuleRef.description
        );
        NodeKind.GRAMMAR_RULE_REF.setDescription(
            GrammarRuleRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarSkip",
            GrammarSkip.description
        );
        NodeKind.GRAMMAR_SKIP.setDescription(
            GrammarSkip.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarStopCut",
            GrammarStopCut.description
        );
        NodeKind.GRAMMAR_STOP_CUT.setDescription(
            GrammarStopCut.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ParseNodeExpr",
            ParseNodeExpr.description
        );
        NodeKind.PARSE_NODE_EXPR.setDescription(
            ParseNodeExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TokenLit",
            TokenLit.description
        );
        NodeKind.TOKEN_LIT.setDescription(
            TokenLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TokenNoCaseLit",
            TokenNoCaseLit.description
        );
        NodeKind.TOKEN_NO_CASE_LIT.setDescription(
            TokenNoCaseLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TokenPatternLit",
            TokenPatternLit.description
        );
        NodeKind.TOKEN_PATTERN_LIT.setDescription(
            TokenPatternLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TokenRef",
            TokenRef.description
        );
        NodeKind.TOKEN_REF.setDescription(
            TokenRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Id",
            Id.description
        );
        NodeKind.ID.setDescription(
            Id.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DefId",
            DefId.description
        );
        NodeKind.DEF_ID.setDescription(
            DefId.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ModuleRefId",
            ModuleRefId.description
        );
        NodeKind.MODULE_REF_ID.setDescription(
            ModuleRefId.description
        );
        NODE_DESCRIPTION_MAP.put(
            "RefId",
            RefId.description
        );
        NodeKind.REF_ID.setDescription(
            RefId.description
        );
        NODE_DESCRIPTION_MAP.put(
            "IfExpr",
            IfExpr.description
        );
        NodeKind.IF_EXPR.setDescription(
            IfExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Isa",
            Isa.description
        );
        NodeKind.ISA.setDescription(
            Isa.description
        );
        NODE_DESCRIPTION_MAP.put(
            "KeepExpr",
            KeepExpr.description
        );
        NodeKind.KEEP_EXPR.setDescription(
            KeepExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LambdaExpr",
            LambdaExpr.description
        );
        NodeKind.LAMBDA_EXPR.setDescription(
            LambdaExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Lit",
            Lit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BigNumLit",
            BigNumLit.description
        );
        NodeKind.BIG_NUM_LIT.setDescription(
            BigNumLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "CharLit",
            CharLit.description
        );
        NodeKind.CHAR_LIT.setDescription(
            CharLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NullLit",
            NullLit.description
        );
        NodeKind.NULL_LIT.setDescription(
            NullLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NumLit",
            NumLit.description
        );
        NodeKind.NUM_LIT.setDescription(
            NumLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "StringLit",
            StringLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BlockStringLit",
            BlockStringLit.description
        );
        NodeKind.BLOCK_STRING_LIT.setDescription(
            BlockStringLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SingleLineStringLit",
            SingleLineStringLit.description
        );
        NodeKind.SINGLE_LINE_STRING_LIT.setDescription(
            SingleLineStringLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "PatternSingleLineStringLit",
            PatternSingleLineStringLit.description
        );
        NodeKind.PATTERN_SINGLE_LINE_STRING_LIT.setDescription(
            PatternSingleLineStringLit.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LogicExpr",
            LogicExpr.description
        );
        NodeKind.LOGIC_EXPR.setDescription(
            LogicExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "MatchExpr",
            MatchExpr.description
        );
        NodeKind.MATCH_EXPR.setDescription(
            MatchExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NotExpr",
            NotExpr.description
        );
        NodeKind.NOT_EXPR.setDescription(
            NotExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ParenExpr",
            ParenExpr.description
        );
        NodeKind.PAREN_EXPR.setDescription(
            ParenExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "RaiseExpr",
            RaiseExpr.description
        );
        NodeKind.RAISE_EXPR.setDescription(
            RaiseExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SubscriptExpr",
            SubscriptExpr.description
        );
        NodeKind.SUBSCRIPT_EXPR.setDescription(
            SubscriptExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "NullCondSubscriptExpr",
            NullCondSubscriptExpr.description
        );
        NodeKind.NULL_COND_SUBSCRIPT_EXPR.setDescription(
            NullCondSubscriptExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TryExpr",
            TryExpr.description
        );
        NodeKind.TRY_EXPR.setDescription(
            TryExpr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "UnOp",
            UnOp.description
        );
        NodeKind.UN_OP.setDescription(
            UnOp.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FullDecl",
            FullDecl.description
        );
        NodeKind.FULL_DECL.setDescription(
            FullDecl.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarListSep",
            GrammarListSep.description
        );
        NodeKind.GRAMMAR_LIST_SEP.setDescription(
            GrammarListSep.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Import",
            Import.description
        );
        NodeKind.IMPORT.setDescription(
            Import.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LangkitRoot",
            LangkitRoot.description
        );
        NodeKind.LANGKIT_ROOT.setDescription(
            LangkitRoot.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LexerCaseRule",
            LexerCaseRule.description
        );
        NodeKind.LEXER_CASE_RULE.setDescription(
            LexerCaseRule.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LexerCaseRuleSend",
            LexerCaseRuleSend.description
        );
        NodeKind.LEXER_CASE_RULE_SEND.setDescription(
            LexerCaseRuleSend.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ListKind",
            ListKind.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ListKindOne",
            ListKindOne.description
        );
        NodeKind.LIST_KIND_ONE.setDescription(
            ListKindOne.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ListKindZero",
            ListKindZero.description
        );
        NodeKind.LIST_KIND_ZERO.setDescription(
            ListKindZero.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LktNodeBaseList",
            LktNodeBaseList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BaseLexerCaseRuleAltList",
            BaseLexerCaseRuleAltList.description
        );
        NodeKind.BASE_LEXER_CASE_RULE_ALT_LIST.setDescription(
            BaseLexerCaseRuleAltList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BlockStringLineList",
            BlockStringLineList.description
        );
        NodeKind.BLOCK_STRING_LINE_LIST.setDescription(
            BlockStringLineList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "CallExprList",
            CallExprList.description
        );
        NodeKind.CALL_EXPR_LIST.setDescription(
            CallExprList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DeclAnnotationList",
            DeclAnnotationList.description
        );
        NodeKind.DECL_ANNOTATION_LIST.setDescription(
            DeclAnnotationList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ElsifBranchList",
            ElsifBranchList.description
        );
        NodeKind.ELSIF_BRANCH_LIST.setDescription(
            ElsifBranchList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumClassAltDeclList",
            EnumClassAltDeclList.description
        );
        NodeKind.ENUM_CLASS_ALT_DECL_LIST.setDescription(
            EnumClassAltDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumClassCaseList",
            EnumClassCaseList.description
        );
        NodeKind.ENUM_CLASS_CASE_LIST.setDescription(
            EnumClassCaseList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "EnumLitDeclList",
            EnumLitDeclList.description
        );
        NodeKind.ENUM_LIT_DECL_LIST.setDescription(
            EnumLitDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ExprList",
            ExprList.description
        );
        NodeKind.EXPR_LIST.setDescription(
            ExprList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "AnyOfList",
            AnyOfList.description
        );
        NodeKind.ANY_OF_LIST.setDescription(
            AnyOfList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FullDeclList",
            FullDeclList.description
        );
        NodeKind.FULL_DECL_LIST.setDescription(
            FullDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DeclBlock",
            DeclBlock.description
        );
        NodeKind.DECL_BLOCK.setDescription(
            DeclBlock.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GenericFormalDeclList",
            GenericFormalDeclList.description
        );
        NodeKind.GENERIC_FORMAL_DECL_LIST.setDescription(
            GenericFormalDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FunArgDeclList",
            FunArgDeclList.description
        );
        NodeKind.FUN_ARG_DECL_LIST.setDescription(
            FunArgDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarExprList",
            GrammarExprList.description
        );
        NodeKind.GRAMMAR_EXPR_LIST.setDescription(
            GrammarExprList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GrammarExprListList",
            GrammarExprListList.description
        );
        NodeKind.GRAMMAR_EXPR_LIST_LIST.setDescription(
            GrammarExprListList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ImportList",
            ImportList.description
        );
        NodeKind.IMPORT_LIST.setDescription(
            ImportList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LambdaArgDeclList",
            LambdaArgDeclList.description
        );
        NodeKind.LAMBDA_ARG_DECL_LIST.setDescription(
            LambdaArgDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "LktNodeList",
            LktNodeList.description
        );
        NodeKind.LKT_NODE_LIST.setDescription(
            LktNodeList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "BlockDeclList",
            BlockDeclList.description
        );
        NodeKind.BLOCK_DECL_LIST.setDescription(
            BlockDeclList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "MatchBranchList",
            MatchBranchList.description
        );
        NodeKind.MATCH_BRANCH_LIST.setDescription(
            MatchBranchList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "ParamList",
            ParamList.description
        );
        NodeKind.PARAM_LIST.setDescription(
            ParamList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "RefIdList",
            RefIdList.description
        );
        NodeKind.REF_ID_LIST.setDescription(
            RefIdList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TypeRefList",
            TypeRefList.description
        );
        NodeKind.TYPE_REF_LIST.setDescription(
            TypeRefList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "IsaList",
            IsaList.description
        );
        NodeKind.ISA_LIST.setDescription(
            IsaList.description
        );
        NODE_DESCRIPTION_MAP.put(
            "MatchBranch",
            MatchBranch.description
        );
        NodeKind.MATCH_BRANCH.setDescription(
            MatchBranch.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Op",
            Op.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpAmp",
            OpAmp.description
        );
        NodeKind.OP_AMP.setDescription(
            OpAmp.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpAnd",
            OpAnd.description
        );
        NodeKind.OP_AND.setDescription(
            OpAnd.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpDiv",
            OpDiv.description
        );
        NodeKind.OP_DIV.setDescription(
            OpDiv.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpEq",
            OpEq.description
        );
        NodeKind.OP_EQ.setDescription(
            OpEq.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpGt",
            OpGt.description
        );
        NodeKind.OP_GT.setDescription(
            OpGt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpGte",
            OpGte.description
        );
        NodeKind.OP_GTE.setDescription(
            OpGte.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpLt",
            OpLt.description
        );
        NodeKind.OP_LT.setDescription(
            OpLt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpLte",
            OpLte.description
        );
        NodeKind.OP_LTE.setDescription(
            OpLte.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpMinus",
            OpMinus.description
        );
        NodeKind.OP_MINUS.setDescription(
            OpMinus.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpMult",
            OpMult.description
        );
        NodeKind.OP_MULT.setDescription(
            OpMult.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpNe",
            OpNe.description
        );
        NodeKind.OP_NE.setDescription(
            OpNe.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpOr",
            OpOr.description
        );
        NodeKind.OP_OR.setDescription(
            OpOr.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpOrInt",
            OpOrInt.description
        );
        NodeKind.OP_OR_INT.setDescription(
            OpOrInt.description
        );
        NODE_DESCRIPTION_MAP.put(
            "OpPlus",
            OpPlus.description
        );
        NodeKind.OP_PLUS.setDescription(
            OpPlus.description
        );
        NODE_DESCRIPTION_MAP.put(
            "Param",
            Param.description
        );
        NodeKind.PARAM.setDescription(
            Param.description
        );
        NODE_DESCRIPTION_MAP.put(
            "TypeRef",
            TypeRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "DefaultListTypeRef",
            DefaultListTypeRef.description
        );
        NodeKind.DEFAULT_LIST_TYPE_REF.setDescription(
            DefaultListTypeRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "FunctionTypeRef",
            FunctionTypeRef.description
        );
        NodeKind.FUNCTION_TYPE_REF.setDescription(
            FunctionTypeRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "GenericTypeRef",
            GenericTypeRef.description
        );
        NodeKind.GENERIC_TYPE_REF.setDescription(
            GenericTypeRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "SimpleTypeRef",
            SimpleTypeRef.description
        );
        NodeKind.SIMPLE_TYPE_REF.setDescription(
            SimpleTypeRef.description
        );
        NODE_DESCRIPTION_MAP.put(
            "VarBind",
            VarBind.description
        );
        NodeKind.VAR_BIND.setDescription(
            VarBind.description
        );
    }

    // ==========
    // Util functions
    // ==========

    /**
     * Get the string representing the memory.
     *
     * @param pointer The pointer to start displaying the memory from.
     * @param count The number of bytes to display from the pointer.
     * @return The string representing the memory as hex bytes.
     */
    private static String dumpMemory(
        final Pointer pointer,
        final long count
    ) {
        final StringBuilder res = new StringBuilder();
        for(int i = 0 ; i < count ; i++) {
            final byte toDump = pointer.readByte(i);
            res.append(String.format("%02x ", toDump));
        }
        return res.toString();
    }

    /**
     * Convert a Java string to a C string by allocating memory.
     *
     * @param jString The Java string to convert.
     * @return The native C char pointer. This pointer MUST be freed.
     */
    @CompilerDirectives.TruffleBoundary
    private static CCharPointer toCString(
        final String jString
    ) {
        final UnsignedWord size = WordFactory.unsigned(jString.length() + 1);
        final CCharPointer res = UnmanagedMemory.calloc(size);
        if(jString.length() > 0) {
            CTypeConversion.toCString(
                jString,
                StandardCharsets.UTF_8,
                res,
                size
            );
        }

        return res;
    }

    /**
     * Convert a Java byte array in a native byte array.
     *
     * @param bytes The Java bytes.
     * @return The native C char pointer. This pointer MUST be freed.
     */
    private static CCharPointer toCBytes(
        final byte[] bytes
    ) {
        final UnsignedWord size = WordFactory.unsigned(bytes.length);
        final CCharPointer res = UnmanagedMemory.malloc(size);
        for(int i = 0 ; i < bytes.length ; i++) {
            res.write(i, bytes[i]);
        }
        return res;
    }

    /**
     * Convert a native-image C string to a Java string.
     *
     * @param pointer The char pointer to convert to a Java string.
     * @return The Java string.
     */
    @CompilerDirectives.TruffleBoundary
    private static String toJString(
        final CCharPointer pointer
    ) {
        return CTypeConversion.toJavaString(pointer);
    }

    /**
     * This function decode a utf 32 int array in a
     * string without calling Java charset.
     *
     * @param chars The int array to decode.
     * @return The resulting string.
     */
    @CompilerDirectives.TruffleBoundary
    private static String decodeUTF32(
        final byte[] toDecode
    ) {
        return toJavaStringNode.execute(
            fromByteArrayNode.execute(toDecode, TruffleString.Encoding.UTF_32)
        );
    }

    /**
     * This function encode a given string to a int array
     * according to the utf 32 standard.
     *
     * @param toEncode The string to encode.
     * @return The encoded string in an int array.
     */
    @CompilerDirectives.TruffleBoundary
    private static byte[] encodeUTF32(
        final String toEncode
    ) {
        return toByteArrayNode.execute(
            fromJavaStringNode.execute(
                toEncode,
                TruffleString.Encoding.UTF_32
            ),
            TruffleString.Encoding.UTF_32
        );
    }

    /**
     * Get the string representation of the given string.
     * This function escaped needed chars and format the string.
     *
     * @param source The source string to get the representation for.
     * @return The representation of the string.
     */
    private static String stringRepresentation(
        final String source
    ) {
        return source
            .replace("\"", "\\\"")
            .replace("\n", "\\x0a");
    }

    /**
     * Convert a C Langkit exception to the LangkitException class.
     */
    private static LangkitException wrapException(
        final LangkitExceptionNative exc
    ) {
        return new LangkitException(
            exc.get_kind(),
            toJString(exc.get_information())
        );
    }

    /**
      * Return the exception raised by the last C API call, or null if the last
      * call was successful.
      */
    private static LangkitException getLastException() {
        LangkitException result = null;

        if(ImageInfo.inImageCode()) {
            final LangkitExceptionNative exceptionNative =
                NI_LIB.lkt_get_last_exception();
            if(exceptionNative.isNonNull()) {
                result = wrapException(exceptionNative);
            }
        } else {
            result = JNI_LIB.lkt_get_last_exception();
        }
        return result;
    }

    /**
     * Check the last exception raised by langkit and throw it.
     *
     * @throws The last langkit exception if there is one.
     */
    @CompilerDirectives.TruffleBoundary
    private static void checkException() throws LangkitException {
        LangkitException exc = getLastException();
        if(exc != null)
            throw exc;
    }

    /**
     * Create a native array from the given rewriting nodes array.
     * The returned pointer must be freed using UnmanagedMemory#free methods.
     */
    private static WordPointer rewritingNodesToNative(
        final RewritingNode[] rewritingNodes
    ) {
        final WordPointer res = UnmanagedMemory.malloc(
            rewritingNodes.length * SizeOf.get(WordPointer.class)
        );
        for(int i = 0; i < rewritingNodes.length; i++) {
            res.write(i, rewritingNodes[i].unwrap());
        }
        return res;
    }

    // ==========
    // Util interfaces
    // ==========

    /**
     * Interface to visit the parse tree.
     */
    public static interface BasicVisitor<T> {
        T visit(LktNode node);
        T visit(LexerCaseRuleCondAlt node);
        T visit(LexerCaseRuleDefaultAlt node);
        T visit(BlockStringLine node);
        T visit(ClassQualifierAbsent node);
        T visit(ClassQualifierPresent node);
        T visit(GrammarRuleDecl node);
        T visit(SyntheticLexerDecl node);
        T visit(NodeDecl node);
        T visit(SelfDecl node);
        T visit(EnumLitDecl node);
        T visit(FieldDecl node);
        T visit(FunArgDecl node);
        T visit(LambdaArgDecl node);
        T visit(DynVarDecl node);
        T visit(MatchValDecl node);
        T visit(ValDecl node);
        T visit(FunDecl node);
        T visit(EnvSpecDecl node);
        T visit(GenericDecl node);
        T visit(GrammarDecl node);
        T visit(LexerDecl node);
        T visit(LexerFamilyDecl node);
        T visit(EnumClassAltDecl node);
        T visit(FunctionType node);
        T visit(GenericFormalTypeDecl node);
        T visit(InstantiatedGenericType node);
        T visit(ClassDecl node);
        T visit(EnumClassDecl node);
        T visit(EnumTypeDecl node);
        T visit(StructDecl node);
        T visit(TraitDecl node);
        T visit(DeclAnnotation node);
        T visit(DeclAnnotationParams node);
        T visit(ElsifBranch node);
        T visit(EnumClassCase node);
        T visit(ExcludesNullAbsent node);
        T visit(ExcludesNullPresent node);
        T visit(AnyOf node);
        T visit(ArrayLiteral node);
        T visit(DotExpr node);
        T visit(NullCondDottedName node);
        T visit(BinOp node);
        T visit(BlockExpr node);
        T visit(CallExpr node);
        T visit(CastExpr node);
        T visit(ErrorOnNull node);
        T visit(GenericInstantiation node);
        T visit(GrammarCut node);
        T visit(GrammarDiscard node);
        T visit(GrammarDontSkip node);
        T visit(GrammarList node);
        T visit(GrammarNull node);
        T visit(GrammarOpt node);
        T visit(GrammarOptError node);
        T visit(GrammarOptErrorGroup node);
        T visit(GrammarOptGroup node);
        T visit(GrammarOrExpr node);
        T visit(GrammarPick node);
        T visit(GrammarImplicitPick node);
        T visit(GrammarPredicate node);
        T visit(GrammarRuleRef node);
        T visit(GrammarSkip node);
        T visit(GrammarStopCut node);
        T visit(ParseNodeExpr node);
        T visit(TokenLit node);
        T visit(TokenNoCaseLit node);
        T visit(TokenPatternLit node);
        T visit(TokenRef node);
        T visit(Id node);
        T visit(DefId node);
        T visit(ModuleRefId node);
        T visit(RefId node);
        T visit(IfExpr node);
        T visit(Isa node);
        T visit(KeepExpr node);
        T visit(LambdaExpr node);
        T visit(BigNumLit node);
        T visit(CharLit node);
        T visit(NullLit node);
        T visit(NumLit node);
        T visit(BlockStringLit node);
        T visit(SingleLineStringLit node);
        T visit(PatternSingleLineStringLit node);
        T visit(LogicExpr node);
        T visit(MatchExpr node);
        T visit(NotExpr node);
        T visit(ParenExpr node);
        T visit(RaiseExpr node);
        T visit(SubscriptExpr node);
        T visit(NullCondSubscriptExpr node);
        T visit(TryExpr node);
        T visit(UnOp node);
        T visit(FullDecl node);
        T visit(GrammarListSep node);
        T visit(Import node);
        T visit(LangkitRoot node);
        T visit(LexerCaseRule node);
        T visit(LexerCaseRuleSend node);
        T visit(ListKindOne node);
        T visit(ListKindZero node);
        T visit(BaseLexerCaseRuleAltList node);
        T visit(BlockStringLineList node);
        T visit(CallExprList node);
        T visit(DeclAnnotationList node);
        T visit(ElsifBranchList node);
        T visit(EnumClassAltDeclList node);
        T visit(EnumClassCaseList node);
        T visit(EnumLitDeclList node);
        T visit(ExprList node);
        T visit(AnyOfList node);
        T visit(FullDeclList node);
        T visit(DeclBlock node);
        T visit(GenericFormalDeclList node);
        T visit(FunArgDeclList node);
        T visit(GrammarExprList node);
        T visit(GrammarExprListList node);
        T visit(ImportList node);
        T visit(LambdaArgDeclList node);
        T visit(LktNodeList node);
        T visit(BlockDeclList node);
        T visit(MatchBranchList node);
        T visit(ParamList node);
        T visit(RefIdList node);
        T visit(TypeRefList node);
        T visit(IsaList node);
        T visit(MatchBranch node);
        T visit(OpAmp node);
        T visit(OpAnd node);
        T visit(OpDiv node);
        T visit(OpEq node);
        T visit(OpGt node);
        T visit(OpGte node);
        T visit(OpLt node);
        T visit(OpLte node);
        T visit(OpMinus node);
        T visit(OpMult node);
        T visit(OpNe node);
        T visit(OpOr node);
        T visit(OpOrInt node);
        T visit(OpPlus node);
        T visit(Param node);
        T visit(DefaultListTypeRef node);
        T visit(FunctionTypeRef node);
        T visit(GenericTypeRef node);
        T visit(SimpleTypeRef node);
        T visit(VarBind node);
    }

    /**
     * Interface to visit the parse tree with a parameter.
     */
    public static interface ParamVisitor<T, P> {
        T visit(LktNode node, P param);
        T visit(LexerCaseRuleCondAlt node, P param);
        T visit(LexerCaseRuleDefaultAlt node, P param);
        T visit(BlockStringLine node, P param);
        T visit(ClassQualifierAbsent node, P param);
        T visit(ClassQualifierPresent node, P param);
        T visit(GrammarRuleDecl node, P param);
        T visit(SyntheticLexerDecl node, P param);
        T visit(NodeDecl node, P param);
        T visit(SelfDecl node, P param);
        T visit(EnumLitDecl node, P param);
        T visit(FieldDecl node, P param);
        T visit(FunArgDecl node, P param);
        T visit(LambdaArgDecl node, P param);
        T visit(DynVarDecl node, P param);
        T visit(MatchValDecl node, P param);
        T visit(ValDecl node, P param);
        T visit(FunDecl node, P param);
        T visit(EnvSpecDecl node, P param);
        T visit(GenericDecl node, P param);
        T visit(GrammarDecl node, P param);
        T visit(LexerDecl node, P param);
        T visit(LexerFamilyDecl node, P param);
        T visit(EnumClassAltDecl node, P param);
        T visit(FunctionType node, P param);
        T visit(GenericFormalTypeDecl node, P param);
        T visit(InstantiatedGenericType node, P param);
        T visit(ClassDecl node, P param);
        T visit(EnumClassDecl node, P param);
        T visit(EnumTypeDecl node, P param);
        T visit(StructDecl node, P param);
        T visit(TraitDecl node, P param);
        T visit(DeclAnnotation node, P param);
        T visit(DeclAnnotationParams node, P param);
        T visit(ElsifBranch node, P param);
        T visit(EnumClassCase node, P param);
        T visit(ExcludesNullAbsent node, P param);
        T visit(ExcludesNullPresent node, P param);
        T visit(AnyOf node, P param);
        T visit(ArrayLiteral node, P param);
        T visit(DotExpr node, P param);
        T visit(NullCondDottedName node, P param);
        T visit(BinOp node, P param);
        T visit(BlockExpr node, P param);
        T visit(CallExpr node, P param);
        T visit(CastExpr node, P param);
        T visit(ErrorOnNull node, P param);
        T visit(GenericInstantiation node, P param);
        T visit(GrammarCut node, P param);
        T visit(GrammarDiscard node, P param);
        T visit(GrammarDontSkip node, P param);
        T visit(GrammarList node, P param);
        T visit(GrammarNull node, P param);
        T visit(GrammarOpt node, P param);
        T visit(GrammarOptError node, P param);
        T visit(GrammarOptErrorGroup node, P param);
        T visit(GrammarOptGroup node, P param);
        T visit(GrammarOrExpr node, P param);
        T visit(GrammarPick node, P param);
        T visit(GrammarImplicitPick node, P param);
        T visit(GrammarPredicate node, P param);
        T visit(GrammarRuleRef node, P param);
        T visit(GrammarSkip node, P param);
        T visit(GrammarStopCut node, P param);
        T visit(ParseNodeExpr node, P param);
        T visit(TokenLit node, P param);
        T visit(TokenNoCaseLit node, P param);
        T visit(TokenPatternLit node, P param);
        T visit(TokenRef node, P param);
        T visit(Id node, P param);
        T visit(DefId node, P param);
        T visit(ModuleRefId node, P param);
        T visit(RefId node, P param);
        T visit(IfExpr node, P param);
        T visit(Isa node, P param);
        T visit(KeepExpr node, P param);
        T visit(LambdaExpr node, P param);
        T visit(BigNumLit node, P param);
        T visit(CharLit node, P param);
        T visit(NullLit node, P param);
        T visit(NumLit node, P param);
        T visit(BlockStringLit node, P param);
        T visit(SingleLineStringLit node, P param);
        T visit(PatternSingleLineStringLit node, P param);
        T visit(LogicExpr node, P param);
        T visit(MatchExpr node, P param);
        T visit(NotExpr node, P param);
        T visit(ParenExpr node, P param);
        T visit(RaiseExpr node, P param);
        T visit(SubscriptExpr node, P param);
        T visit(NullCondSubscriptExpr node, P param);
        T visit(TryExpr node, P param);
        T visit(UnOp node, P param);
        T visit(FullDecl node, P param);
        T visit(GrammarListSep node, P param);
        T visit(Import node, P param);
        T visit(LangkitRoot node, P param);
        T visit(LexerCaseRule node, P param);
        T visit(LexerCaseRuleSend node, P param);
        T visit(ListKindOne node, P param);
        T visit(ListKindZero node, P param);
        T visit(BaseLexerCaseRuleAltList node, P param);
        T visit(BlockStringLineList node, P param);
        T visit(CallExprList node, P param);
        T visit(DeclAnnotationList node, P param);
        T visit(ElsifBranchList node, P param);
        T visit(EnumClassAltDeclList node, P param);
        T visit(EnumClassCaseList node, P param);
        T visit(EnumLitDeclList node, P param);
        T visit(ExprList node, P param);
        T visit(AnyOfList node, P param);
        T visit(FullDeclList node, P param);
        T visit(DeclBlock node, P param);
        T visit(GenericFormalDeclList node, P param);
        T visit(FunArgDeclList node, P param);
        T visit(GrammarExprList node, P param);
        T visit(GrammarExprListList node, P param);
        T visit(ImportList node, P param);
        T visit(LambdaArgDeclList node, P param);
        T visit(LktNodeList node, P param);
        T visit(BlockDeclList node, P param);
        T visit(MatchBranchList node, P param);
        T visit(ParamList node, P param);
        T visit(RefIdList node, P param);
        T visit(TypeRefList node, P param);
        T visit(IsaList node, P param);
        T visit(MatchBranch node, P param);
        T visit(OpAmp node, P param);
        T visit(OpAnd node, P param);
        T visit(OpDiv node, P param);
        T visit(OpEq node, P param);
        T visit(OpGt node, P param);
        T visit(OpGte node, P param);
        T visit(OpLt node, P param);
        T visit(OpLte node, P param);
        T visit(OpMinus node, P param);
        T visit(OpMult node, P param);
        T visit(OpNe node, P param);
        T visit(OpOr node, P param);
        T visit(OpOrInt node, P param);
        T visit(OpPlus node, P param);
        T visit(Param node, P param);
        T visit(DefaultListTypeRef node, P param);
        T visit(FunctionTypeRef node, P param);
        T visit(GenericTypeRef node, P param);
        T visit(SimpleTypeRef node, P param);
        T visit(VarBind node, P param);
    }

    // ==========
    // Util classes
    // ==========

    /**
     * This class represents a pointer and can hold NI and JNI addresses.
     */
    public static final class PointerWrapper {

        // ----- Instance attributes -----

        /** The pointer NI value. */
        private PointerBase ni;

        /** The pointer JNI value. */
        private final long jni;

        // ----- Constructors -----

        /**
         * Create a new custom pointer from a NI pointer based value.
         *
         * @param niPointer The pointer based value.
         */
        PointerWrapper(
            final PointerBase niPointer
        ) {
            this.ni = niPointer;
            this.jni = -1;
        }

        /**
         * Create a new custom pointer from a long value.
         *
         * @param jniPointer The pointer in a long value.
         */
        PointerWrapper(
            final long jniPointer
        ) {
            this.jni = jniPointer;
        }

        /**
         * Wrap the given NI pointer in the Java class.
         *
         * @param niPointer The NI pointer to wrap.
         * @return The wrapped pointer.
         */
        static PointerWrapper wrap(
            final PointerBase niPointer
        ) {
            return new PointerWrapper(niPointer);
        }

        /**
         * Get the null pointer according to the execution mode.
         *
         * @return The null custom pointer.
         */
        public static PointerWrapper nullPointer() {

            if(ImageInfo.inImageCode()) {
                return new PointerWrapper(WordFactory.nullPointer());
            } else {
                return new PointerWrapper(0L);
            }

        }

        // ----- Instance methods -----

        /**
         * Get the pointer as an NI pointer based value.
         *
         * @return The pointer based value for NI.
         */
        public <T extends PointerBase> T ni() {
            return (T) this.ni;
        }

        /**
         * Get the pointer as a long Java value.
         *
         * @return The pointer as a long value for JNI.
         */
        public long jni() {
            return this.jni;
        }

        /**
         * Get if the pointer is null.
         *
         * @return True if the pointer is null, false else.
         */
        public boolean isNull() {

            if(ImageInfo.inImageCode()) {
                return this.ni.isNull();
            } else {
                return this.jni == 0;
            }

        }

        // ----- Override methods -----

        @Override
        public String toString() {

            if(ImageInfo.inImageCode()) {
                return "PointerWrapper{"
                    + this.ni.rawValue()
                    + "}";
            } else {
                return "PointerWrapper{"
                    + this.jni
                    + "}";
            }

        }

        @Override
        public boolean equals(Object o) {
            if(o == this) return true;
            if(!(o instanceof PointerWrapper)) return false;
            final PointerWrapper other = (PointerWrapper) o;
            if(ImageInfo.inImageCode()) {
                return this.ni.equal(other.ni);
            } else {
                return this.jni == other.jni;
            }
        }

        @Override
        public int hashCode() {

            if(ImageInfo.inImageCode()) {
                return (int) this.ni.rawValue();
            } else {
                return (int) this.jni;
            }

        }

    }

    // ==========
    // Reflection utils
    // ==========

    public static final class Reflection {
        /**
        * This class represents the description of a node.
        */
        public static final class Node {

            // ----- Instance attributes -----

            /** Kind of the node. This kind is null if the node is abstract */
            public final NodeKind kind;

            /** Whether the node is a token node */
            public final boolean isTokenNode;

            /** Whether the node is a list node */
            public final boolean isListNode;

            /** Java class of the node */
            public final Class<? extends LktNode> clazz;

            /** Simple name of the Java class of the node */
            public final String className;

            /** Fields of the node, sorted by parsing order */
            public final String[] fields;

            /** Map containing description for all fields of the node */
            public final Map<String, Field> fieldDescriptions;

            // ----- Constructors -----

            /** Create a new node description with its kind and class */
            public Node (
                NodeKind kind,
                final boolean isTokenNode,
                final boolean isListNode,
                final Class<? extends LktNode> clazz,
                final String className,
                final String[] fields,
                final Map<String, Field> fieldDescriptions
            ) {
                this.kind = kind;
                this.isTokenNode = isTokenNode;
                this.isListNode = isListNode;
                this.clazz = clazz;
                this.className = className;
                this.fieldDescriptions = fieldDescriptions;
                this.fields = fields;
            }

        }

        /**
        * This class represents the description of a node field.
        */
        public static final class Field {

            // ----- Instance attributes -----

            /** The Java method for the field */
            public final Method javaMethod;

            /** The parameters of the method */
            public final List<Param> params;

            /** The generic member reference of this field */
            public final MemberReference memberRef;

            // ----- Constructors -----

            /**
            * Create a new field description.
            *
            * @param method The Java method to access the field.
            * @param params The parameters of the field call.
            */
            public Field(
                final Method javaMethod,
                final List<Param> params,
                final MemberReference memberRef
            ) {
                this.javaMethod = javaMethod;
                this.params = params;
                this.memberRef = memberRef;
            }

        }

        /**
        * This class represents a parameter description.
        */
        public static class Param {

            // ----- Instance attributes -----

            /** The type of the argument */
            public final Class<?> type;

            /** The name of the parameter */
            public final String name;

            /** The optional default value of the parameter */
            public final Optional<Object> defaultValue;

            // ----- Constructors -----

            /**
            * Create a new langkit parameter.
            *
            * @param type The type of the parameter.
            * @param name The name of the parameter.
            */
            public Param(
                final Class<?> type,
                final String name
            ) {
                this.type = type;
                this.name = name;
                this.defaultValue = Optional.empty();
            }

            /** Create a new parameter description with a default value. */
            public Param(
                final Class<?> type,
                final String name,
                final Object defaultValue
            ) {
                this.type = type;
                this.name = name;
                this.defaultValue = Optional.ofNullable(defaultValue);
            }

        }
    }

    // ==========
    // Language specific extensions
    // ==========

    


    // ==========
    // Defining the JNI bindings library
    // ==========

    
    
    
    
    
    
    

    /** This class contains all native function definitions for JNI */
    public static final class JNI_LIB {

        // ----- Static initializer -----

        static {
            if(!ImageInfo.inImageCode()) {
                // Load the needed libraries
                if(OS.indexOf("win") < 0) {
                    System.loadLibrary("langkit_sigsegv_handler");
                }
                System.loadLibrary("lktlang_jni");

                // Initialize the JNI library
                lkt_initialize();

                // Register the library finalizer
                Runtime.getRuntime().addShutdownHook(
                    new Thread(JNI_LIB::lkt_finalize)
                );
            }
        }

        // ----- Language specific functions -----

        


        // ----- Lifecycle functions ------

        /** Function to initialize the JNI library */
        public static native void lkt_initialize();

        /** Function to finalize the JNI library */
        public static native void lkt_finalize();

        // ----- Exception functions ------

        /** Get the last langkit exception */
        @CompilerDirectives.TruffleBoundary
        public static native LangkitException lkt_get_last_exception();

        // ----- Text functions -----

        /** Create a new text from its content */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_create_text(
            byte[] utf32Content
        );

        /** Destroy the given text */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_destroy_text(
            Text text
        );

        // ----- Rewriting apply result functions -----

        /** Get the diagnostics from the rewriting apply result */
        @CompilerDirectives.TruffleBoundary
        public static native Diagnostic[]
        lkt_rewriting_get_result_diagnostics(
            int diagnosticsCount,
            long diagnosticsReference
        );

        /** Free the rewriting apply result structure */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_free_apply_result(
            RewritingApplyResult applyResult
        );

        // ----- File reader functions -----

        /** Decrease the reference counter of the given file reader */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_dec_ref_file_reader(
            FileReader fileReader
        );

        // ----- Unit provider functions -----

        /** Decrease the ref counter of the unit provider */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_dec_ref_unit_provider(
            UnitProvider unitProvider
        );

        // ----- Event handler functions -----

        /** Create a new event handler */
        @CompilerDirectives.TruffleBoundary
        public static native PointerWrapper lkt_create_event_handler(
            EventHandler.UnitRequestedCallback unitRequestedCallback,
            EventHandler.UnitParsedCallback unitParsedCallback
        );

        /** Decrease the ref counter of the event handler */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_dec_ref_event_handler(
            EventHandler eventHandler
        );

        // ----- Token functions -----

        /** Get the next token */
        @CompilerDirectives.TruffleBoundary
        public static native Token lkt_token_next(
            Token token
        );

        /** Get the previous token */
        @CompilerDirectives.TruffleBoundary
        public static native Token lkt_token_previous(
            Token token
        );

        /** Get if the tokens are equivalent */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_token_is_equivalent(
            Token left,
            Token right
        );

        /** Get text between the two tokens */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_token_range_text(
            Token start,
            Token end
        );

        // ----- Analysis context functions -----

        /** Create a new analysis context */
        @CompilerDirectives.TruffleBoundary
        public static native PointerWrapper lkt_create_analysis_context(
            String charset,
            FileReader fileReader,
            UnitProvider unitProvider,
            EventHandler eventHandler,
            boolean withTrivia,
            int tabstop
        );

        /** Increase the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_context_incref(
            long context
        );

        /** Decrease the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_context_decref(
            long context
        );

        // ----- Analysis unit functions -----

        /** Get the analysis unit from a file */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit
        lkt_get_analysis_unit_from_file(
            AnalysisContext context,
            String fileName,
            String charset,
            boolean reparse,
            int grammarRule
        );

        /** Get the analysis unit from a buffer */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit
        lkt_get_analysis_unit_from_buffer(
            AnalysisContext context,
            String fileName,
            String charset,
            String buffer,
            long bufferSize,
            int grammarRule
        );

        /** Get the analysis unit from the unit provider. */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit
        lkt_get_analysis_unit_from_provider(
            AnalysisContext context,
            Text name,
            int kind,
            String charset,
            boolean reparse
        );

        /** Get the root of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_unit_root(
            AnalysisUnit unit
        );

        /** Get the file name of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_unit_filename(
            AnalysisUnit unit
        );

        /** Get the token count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_unit_token_count(
            AnalysisUnit unit
        );

        /** Get the trivia count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_unit_trivia_count(
            AnalysisUnit unit
        );

        /** Get the first token of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Token lkt_unit_first_token(
            AnalysisUnit unit
        );

        /** Get the last token of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Token lkt_unit_last_token(
            AnalysisUnit unit
        );

        /** Get the context of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisContext lkt_unit_context(
            AnalysisUnit unit
        );

        /** Get the number of diagnostic in the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_unit_diagnostic_count(
            AnalysisUnit unit
        );

        /** Get the nth diagnostic of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Diagnostic lkt_unit_diagnostic(
            AnalysisUnit unit,
            int n
        );

        // ----- Rewriting context functions -----

        /** Start a rewriting session and return the new context */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingContext
        lkt_rewriting_start_rewriting(
            AnalysisContext analysisContext
        );

        /** Get the analysis context from the given rewriting context */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisContext
        lkt_rewriting_handle_to_context(
            RewritingContext rewritingContext
        );

        /** Get a pointer to the rewriting units owned by the context */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingUnit[] lkt_rewriting_unit_handles(
            RewritingContext rewritingContext
        );

        /** Create a node in the rewriting context and return it */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_create_node(
            RewritingContext rewritingContext,
            int nodeKind
        );

        /** Create a node in the rewriting context with the given children */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode
        lkt_rewriting_create_regular_node(
            RewritingContext rewritingContext,
            int nodeKind,
            RewritingNode[] children
        );

        /** Create a token node in the rewriting context and return it */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode
        lkt_rewriting_create_token_node(
            RewritingContext rewritingContext,
            int nodeKind,
            Text nodeText
        );

        /** Create a new node tree from the given template */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode
        lkt_rewriting_create_from_template(
            RewritingContext rewriting_context,
            Text template_text,
            RewritingNode[] arguments,
            int rule
        );

        /** Apply the rewriting session and close it fi success */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingApplyResult lkt_rewriting_apply(
            RewritingContext rewritingContext
        );

        /** Abort the rewriting session */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_abort_rewriting(
            RewritingContext rewritingContext
        );

        // ----- Rewriting unit functions -----

        /** Get the rewriting unit associated */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingUnit lkt_rewriting_unit_to_handle(
            AnalysisUnit analysisUnit
        );

        /** Get the analysis unit corresponding to the given rewriting unit */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit lkt_rewriting_handle_to_unit(
            RewritingUnit rewritingUnit
        );

        /** Get the root of the given rewriting unit */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_unit_root(
            RewritingUnit rewritingUnit
        );

        /** Set the root of the rewriting unit to the rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_unit_set_root(
            RewritingUnit rewritingUnit,
            RewritingNode rewritingNode
        );

        /** Unparse the given rewriting unit and return its textual value */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_rewriting_unit_unparse(
            RewritingUnit rewritingUnit
        );

        // ----- Rewriting node functions -----

        /** Get the rewriting node from the given parsed node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_node_to_handle(
            Entity entity
        );

        /** Get the parsed node from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_rewriting_handle_to_node(
            RewritingNode rewritingNode
        );

        /** Get the rewriting context from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingContext
        lkt_rewriting_node_to_context(
            RewritingNode rewriting_node
        );

        /** Clone the given rewriting node and return the copy */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_clone(
            RewritingNode toClone
        );

        /** Unparse the given rewriting node in the given text */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_rewriting_node_unparse(
            RewritingNode rewritingNode
        );

        /** Get the kind of the rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_rewriting_kind(
            RewritingNode rewritingNode
        );

        /** Get the rewriting node image */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_rewriting_node_image(
            RewritingNode rewritingNode
        );

        /** Return whether the node is tied to a rewriting unit */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_rewriting_tied(
            RewritingNode rewritingNode
        );

        /** Return the parent of the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_parent(
            RewritingNode rewritingNode
        );

        /** Get the rewriting node children */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode[] lkt_rewriting_children(
            RewritingNode rewritingNode
        );

        /** Get the child at the given member reference */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_child(
            RewritingNode parent,
            int childMemberReference
        );

        /** Set the given child at the given member reference */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_set_child(
            RewritingNode parent,
            int childMemberReference,
            RewritingNode newChild
        );

        /** Replace the rewriting node by the new one */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_replace(
            RewritingNode rewritingNode,
            RewritingNode newNode
        );

        /** Get the first child of the rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_first_child(
            RewritingNode parent
        );

        /** Get the last child of the rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_last_child(
            RewritingNode parent
        );

        /** Get the next child from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_next_child(
            RewritingNode rewritingNode
        );

        /** Get the previous child from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        public static native RewritingNode lkt_rewriting_previous_child(
            RewritingNode rewritingNode
        );

        /** Insert the provided rewriting node before the other node */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_insert_before(
            RewritingNode rewritingNode,
            RewritingNode toInsert
        );

        /** Insert the provided rewriting node after the other node */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_insert_after(
            RewritingNode rewritingNode,
            RewritingNode toInsert
        );

        /**
         * Insert the provided rewriting node at the beginning of the
         * children
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_insert_first(
            RewritingNode rewritingNode,
            RewritingNode toInsert
        );

        /** Insert the provided rewriting node at the end of the children */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_insert_last(
            RewritingNode rewritingNode,
            RewritingNode toInsert
        );

        /** Remove the given rewriting node from its list parent */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_remove_child(
            RewritingNode toRemove
        );

        /** Get the text of the rewriting token node */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_rewriting_text(
            RewritingNode rewritingNode
        );

        /** Set the text of the rewriting token node */
        @CompilerDirectives.TruffleBoundary
        public static native void lkt_rewriting_set_text(
            RewritingNode rewritingNode,
            Text text
        );

        // ----- Node functions -----

        /** Return whether the two given entities are equal */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_node_is_equivalent(
            Entity entity_left,
            Entity entity_right
        );

        /** Get the hash of a node */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_node_hash(
            Entity entity
        );

        /** Get the node kind */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_node_kind(
            Entity entity
        );

        /** Get the node text */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_node_text(
            Entity entity
        );

        /** Get the node source location range */
        @CompilerDirectives.TruffleBoundary
        public static native SourceLocationRange lkt_node_sloc_range(
            Entity entity
        );

        /** Get the node children count */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_node_children_count(
            Entity entity
        );

        /** Get the node nth child */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_node_child(
            Entity entity,
            int n
        );

        /** Get if the node is a token node */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_node_is_token_node(
            Entity entity
        );

        /** Get the unit of the node */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit lkt_node_unit(
            Entity entity
        );

        /** Get the entity image of the node */
        @CompilerDirectives.TruffleBoundary
        public static native Text lkt_node_image(
            Entity entity
        );

        // ----- Node fields accessors and properties -----

        
    

            

        /** Isomethod of lkt_lkt_node_p_node_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_node_gen_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_node_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_node_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_token_node_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_token_node_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_error_node_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_error_node_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_char_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_char_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_int_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_int_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_bool_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_bool_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_bigint_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_bigint_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_string_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_string_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_symbol_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_symbol_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_property_error_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_property_error_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_regexp_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_regexp_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_array_gen_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_array_gen_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_array_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_array_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_astlist_gen_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_astlist_gen_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_astlist_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_astlist_type(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_iterator_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_iterator_gen_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_iterator_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_iterator_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_analysis_unit_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_analysis_unit_gen_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_analysis_unit_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_analysis_unit_trait(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_p_topmost_invalid_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_p_topmost_invalid_decl(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_parent langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_parent(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_parents langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native LktNode[] lkt_lkt_node_parents(
            boolean
            with_self,
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_children langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native LktNode[] lkt_lkt_node_children(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_token_start langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Token lkt_lkt_node_token_start(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_token_end langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Token lkt_lkt_node_token_end(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_child_index langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native int lkt_lkt_node_child_index(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_previous_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_previous_sibling(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_next_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lkt_node_next_sibling(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit lkt_lkt_node_unit(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_is_ghost langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_lkt_node_is_ghost(
            Entity node
        );
            

        /** Isomethod of lkt_lkt_node_full_sloc_image langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_lkt_node_full_sloc_image(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_base_lexer_case_rule_alt_f_send langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_base_lexer_case_rule_alt_f_send(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_lexer_case_rule_cond_alt_f_cond_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_case_rule_cond_alt_f_cond_exprs(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_class_qualifier_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_class_qualifier_p_as_bool(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_decl_f_syn_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_f_syn_name(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Symbol lkt_decl_p_name(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_full_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_decl_p_full_name(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_decl_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String lkt_decl_p_decl_type_name(
            Entity node
        );
            

        /** Isomethod of lkt_decl_p_as_bare_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_p_as_bare_decl(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_base_grammar_rule_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_base_grammar_rule_decl_f_expr(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_base_val_decl_p_get_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_base_val_decl_p_get_type(
            boolean
            no_inference,
            Entity node
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_explicitly_typed_decl_f_decl_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_explicitly_typed_decl_f_decl_type(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_component_decl_f_default_val langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_component_decl_f_default_val(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_fun_arg_decl_f_decl_annotations langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_fun_arg_decl_f_decl_annotations(
            Entity node
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_val_decl_f_val langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_val_decl_f_val(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_fun_decl_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_fun_decl_f_args(
            Entity node
        );
            

        /** Isomethod of lkt_fun_decl_f_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_fun_decl_f_return_type(
            Entity node
        );
            

        /** Isomethod of lkt_fun_decl_f_body langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_fun_decl_f_body(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_env_spec_decl_f_actions langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_env_spec_decl_f_actions(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_generic_decl_f_generic_formal_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_decl_f_generic_formal_decls(
            Entity node
        );
            

        /** Isomethod of lkt_generic_decl_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_decl_f_decl(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_decl_f_rules langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_decl_f_rules(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_decl_p_lexer langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_decl_p_lexer(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_lexer_decl_f_rules langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_decl_f_rules(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_lexer_family_decl_f_rules langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_family_decl_f_rules(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_type_decl_f_traits langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_type_decl_f_traits(
            Entity node
        );
            

        /** Isomethod of lkt_type_decl_p_is_class langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_type_decl_p_is_class(
            Entity node
        );
            

        /** Isomethod of lkt_type_decl_p_implemented_traits langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native TypeDecl[] lkt_type_decl_p_implemented_traits(
            Entity node
        );
            

        /** Isomethod of lkt_type_decl_f_syn_base_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_type_decl_f_syn_base_type(
            Entity node
        );
            

        /** Isomethod of lkt_type_decl_p_base_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_type_decl_p_base_type(
            Entity node
        );
            

        /** Isomethod of lkt_type_decl_p_base_types langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native TypeDecl[] lkt_type_decl_p_base_types(
            Entity node
        );
            

        /** Isomethod of lkt_type_decl_p_root_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_type_decl_p_root_type(
            Entity node
        );
            

        /** Isomethod of lkt_type_decl_p_is_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_type_decl_p_is_subtype(
            Entity
            potential_base,
            Entity node
        );
            

        /** Isomethod of lkt_type_decl_p_is_generic langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_type_decl_p_is_generic(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_generic_formal_type_decl_f_has_class langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_formal_type_decl_f_has_class(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_instantiated_generic_type_p_get_inner_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_instantiated_generic_type_p_get_inner_type(
            Entity node
        );
            

        /** Isomethod of lkt_instantiated_generic_type_p_get_actuals langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native TypeDecl[] lkt_instantiated_generic_type_p_get_actuals(
            Entity node
        );
            

        /** Isomethod of lkt_instantiated_generic_type_p_get_instantiated_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_instantiated_generic_type_p_get_instantiated_type(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_named_type_decl_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_named_type_decl_f_decls(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_enum_class_decl_f_branches langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_enum_class_decl_f_branches(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_enum_type_decl_f_literals langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_enum_type_decl_f_literals(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_decl_annotation_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_annotation_f_name(
            Entity node
        );
            

        /** Isomethod of lkt_decl_annotation_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_annotation_f_params(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_decl_annotation_params_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_decl_annotation_params_f_params(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_elsif_branch_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_elsif_branch_f_cond_expr(
            Entity node
        );
            

        /** Isomethod of lkt_elsif_branch_f_then_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_elsif_branch_f_then_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_enum_class_case_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_enum_class_case_f_decls(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_excludes_null_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_excludes_null_p_as_bool(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_expr_p_in_type_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_expr_p_in_type_ref(
            Entity node
        );
            

        /** Isomethod of lkt_expr_p_is_regular_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_expr_p_is_regular_expr(
            Entity node
        );
            

        /** Isomethod of lkt_expr_p_expr_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native SemanticResult lkt_expr_p_expr_type(
            Entity node
        );
            

        /** Isomethod of lkt_expr_p_check_expr_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_expr_p_check_expr_type(
            Entity node
        );
            

        /** Isomethod of lkt_expr_p_expr_context_free_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_expr_p_expr_context_free_type(
            Entity node
        );
            

        /** Isomethod of lkt_expr_p_referenced_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native SemanticResult lkt_expr_p_referenced_decl(
            Entity node
        );
            

        /** Isomethod of lkt_expr_p_check_referenced_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_expr_p_check_referenced_decl(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_any_of_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_any_of_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_any_of_f_values langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_any_of_f_values(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_array_literal_f_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_array_literal_f_exprs(
            Entity node
        );
            

        /** Isomethod of lkt_array_literal_f_element_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_array_literal_f_element_type(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_base_dot_expr_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_base_dot_expr_f_prefix(
            Entity node
        );
            

        /** Isomethod of lkt_base_dot_expr_f_suffix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_base_dot_expr_f_suffix(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_bin_op_f_left langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_bin_op_f_left(
            Entity node
        );
            

        /** Isomethod of lkt_bin_op_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_bin_op_f_op(
            Entity node
        );
            

        /** Isomethod of lkt_bin_op_f_right langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_bin_op_f_right(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_block_expr_f_val_defs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_block_expr_f_val_defs(
            Entity node
        );
            

        /** Isomethod of lkt_block_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_block_expr_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_call_expr_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_call_expr_f_name(
            Entity node
        );
            

        /** Isomethod of lkt_call_expr_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_call_expr_f_args(
            Entity node
        );
            

        /** Isomethod of lkt_call_expr_p_called_object_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_call_expr_p_called_object_type(
            Entity node
        );
            

        /** Isomethod of lkt_call_expr_p_called_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_call_expr_p_called_decl(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_cast_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_cast_expr_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_cast_expr_f_excludes_null langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_cast_expr_f_excludes_null(
            Entity node
        );
            

        /** Isomethod of lkt_cast_expr_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_cast_expr_f_dest_type(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_error_on_null_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_error_on_null_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_generic_instantiation_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_instantiation_f_name(
            Entity node
        );
            

        /** Isomethod of lkt_generic_instantiation_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_instantiation_f_args(
            Entity node
        );
            

        /** Isomethod of lkt_generic_instantiation_p_instantiated_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_instantiation_p_instantiated_decl(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_grammar_discard_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_discard_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_dont_skip_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_dont_skip_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_dont_skip_f_dont_skip langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_dont_skip_f_dont_skip(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_list_f_list_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_list_f_list_type(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_list_f_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_list_f_kind(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_list_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_list_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_list_f_sep langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_list_f_sep(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_null_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_null_f_name(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_opt_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_error_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_opt_error_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_error_group_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_opt_error_group_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_group_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_opt_group_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_or_expr_f_sub_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_or_expr_f_sub_exprs(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_pick_f_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_pick_f_exprs(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_grammar_predicate_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_predicate_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_predicate_f_prop_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_predicate_f_prop_ref(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_rule_ref_f_node_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_rule_ref_f_node_name(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_skip_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_skip_f_name(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_stop_cut_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_stop_cut_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_parse_node_expr_f_node_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_parse_node_expr_f_node_name(
            Entity node
        );
            

        /** Isomethod of lkt_parse_node_expr_f_sub_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_parse_node_expr_f_sub_exprs(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_token_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native DecodedStringValue lkt_token_lit_p_denoted_value(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_token_no_case_lit_f_lit langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_token_no_case_lit_f_lit(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_token_pattern_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native DecodedStringValue lkt_token_pattern_lit_p_denoted_value(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_token_ref_f_token_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_token_ref_f_token_name(
            Entity node
        );
            

        /** Isomethod of lkt_token_ref_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_token_ref_f_expr(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_if_expr_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_if_expr_f_cond_expr(
            Entity node
        );
            

        /** Isomethod of lkt_if_expr_f_then_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_if_expr_f_then_expr(
            Entity node
        );
            

        /** Isomethod of lkt_if_expr_f_alternatives langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_if_expr_f_alternatives(
            Entity node
        );
            

        /** Isomethod of lkt_if_expr_f_else_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_if_expr_f_else_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_isa_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_isa_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_isa_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_isa_f_dest_type(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_keep_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_keep_expr_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_keep_expr_f_keep_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_keep_expr_f_keep_type(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_lambda_expr_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lambda_expr_f_params(
            Entity node
        );
            

        /** Isomethod of lkt_lambda_expr_f_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lambda_expr_f_return_type(
            Entity node
        );
            

        /** Isomethod of lkt_lambda_expr_f_body langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lambda_expr_f_body(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_char_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native DecodedCharValue lkt_char_lit_p_denoted_value(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_null_lit_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_null_lit_f_dest_type(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_string_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native DecodedStringValue lkt_string_lit_p_denoted_value(
            Entity node
        );
            

        /** Isomethod of lkt_string_lit_p_is_prefixed_string langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_string_lit_p_is_prefixed_string(
            Entity node
        );
            

        /** Isomethod of lkt_string_lit_p_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Char lkt_string_lit_p_prefix(
            Entity node
        );
            

        /** Isomethod of lkt_string_lit_p_is_regexp_literal langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_string_lit_p_is_regexp_literal(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_block_string_lit_f_lines langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_block_string_lit_f_lines(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_logic_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_logic_expr_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_match_expr_f_match_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_match_expr_f_match_expr(
            Entity node
        );
            

        /** Isomethod of lkt_match_expr_f_branches langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_match_expr_f_branches(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_not_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_not_expr_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_paren_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_paren_expr_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_raise_expr_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_raise_expr_f_dest_type(
            Entity node
        );
            

        /** Isomethod of lkt_raise_expr_f_except_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_raise_expr_f_except_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_subscript_expr_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_subscript_expr_f_prefix(
            Entity node
        );
            

        /** Isomethod of lkt_subscript_expr_f_index langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_subscript_expr_f_index(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_try_expr_f_try_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_try_expr_f_try_expr(
            Entity node
        );
            

        /** Isomethod of lkt_try_expr_f_or_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_try_expr_f_or_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_un_op_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_un_op_f_op(
            Entity node
        );
            

        /** Isomethod of lkt_un_op_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_un_op_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_full_decl_f_doc langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_full_decl_f_doc(
            Entity node
        );
            

        /** Isomethod of lkt_full_decl_f_decl_annotations langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_full_decl_f_decl_annotations(
            Entity node
        );
            

        /** Isomethod of lkt_full_decl_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_full_decl_f_decl(
            Entity node
        );
            

        /** Isomethod of lkt_full_decl_p_has_annotation langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean lkt_full_decl_p_has_annotation(
            Symbol
            name,
            Entity node
        );
            

        /** Isomethod of lkt_full_decl_p_check_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native TreeSemanticResult lkt_full_decl_p_check_decl(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_grammar_list_sep_f_token langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_list_sep_f_token(
            Entity node
        );
            

        /** Isomethod of lkt_grammar_list_sep_f_extra langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_grammar_list_sep_f_extra(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_import_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_import_f_name(
            Entity node
        );
            

        /** Isomethod of lkt_import_p_referenced_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit lkt_import_p_referenced_unit(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_langkit_root_f_imports langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_langkit_root_f_imports(
            Entity node
        );
            

        /** Isomethod of lkt_langkit_root_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_langkit_root_f_decls(
            Entity node
        );
            

        /** Isomethod of lkt_langkit_root_p_check_semantic langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native TreeSemanticResult lkt_langkit_root_p_check_semantic(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_lexer_case_rule_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_case_rule_f_expr(
            Entity node
        );
            

        /** Isomethod of lkt_lexer_case_rule_f_alts langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_case_rule_f_alts(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_lexer_case_rule_send_f_sent langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_case_rule_send_f_sent(
            Entity node
        );
            

        /** Isomethod of lkt_lexer_case_rule_send_f_match_size langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_lexer_case_rule_send_f_match_size(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_match_branch_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_match_branch_f_decl(
            Entity node
        );
            

        /** Isomethod of lkt_match_branch_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_match_branch_f_expr(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_param_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_param_f_name(
            Entity node
        );
            

        /** Isomethod of lkt_param_f_value langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_param_f_value(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_type_ref_p_designated_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_type_ref_p_designated_type(
            Entity node
        );

        
    


        
    

            

        /** Isomethod of lkt_function_type_ref_f_args_types langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_function_type_ref_f_args_types(
            Entity node
        );
            

        /** Isomethod of lkt_function_type_ref_f_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_function_type_ref_f_return_type(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_generic_type_ref_f_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_type_ref_f_type_name(
            Entity node
        );
            

        /** Isomethod of lkt_generic_type_ref_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_generic_type_ref_f_params(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_simple_type_ref_f_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_simple_type_ref_f_type_name(
            Entity node
        );

        
    

            

        /** Isomethod of lkt_var_bind_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_var_bind_f_name(
            Entity node
        );
            

        /** Isomethod of lkt_var_bind_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity lkt_var_bind_f_expr(
            Entity node
        );


    }


    // ==========
    // Defining the Native-Image bindings library
    // ==========

    
    
    
    
    
    
    

    /** This class contains the directives for the shared lib loading */
    public static final class LibDirectives implements CContext.Directives {
        @Override
        public List<String> getHeaderFiles() {
            List<String> res = new ArrayList<>();
            res.add("<liblktlang.h>");
            res.add("<stdlib.h>");
            return res;
        }

        @Override
        public List<String> getLibraries() {
            List<String> res = new ArrayList<>();
            res.add("lktlang");
            return res;
        }
    }

    // ===== Language specific structures =====

    


    // ===== Constant structures =====

    /** The structure for the langkit exceptions */
    @CContext(LibDirectives.class)
    @CStruct("lkt_exception")
    public interface LangkitExceptionNative extends PointerBase {
        @CField("kind") public int get_kind();
        @CField("kind") public void set_kind(
            int kind
        );

        @CField("information") public CCharPointer get_information();
        @CField("information") public void set_information(
            CCharPointer information
        );
    }

    /** The big integers are just pointers */
    public interface BigIntegerNative extends Pointer {}

    /** The structure for the symbols */
    @CContext(LibDirectives.class)
    @CStruct("lkt_symbol_type")
    public interface SymbolNative extends PointerBase {
        @CField("thin_sym") public int get_thin_sym();
        @CField("thin_sym") public void set_thin_sym(
            int data
        );

        @CField("table") public VoidPointer get_table();
        @CField("table") public void set_table(
            VoidPointer bounds
        );
    }

    /** The string wrappers are just pointers */
    public interface StringNative extends Pointer {}

    /** The structure for the text */
    @CContext(LibDirectives.class)
    @CStruct("lkt_text")
    public interface TextNative extends PointerBase {
        @CField("chars") public CIntPointer get_chars();
        @CField("chars") public void set_chars(
            CIntPointer chars
        );

        @CField("length") public long get_length();
        @CField("length") public void set_length(
            long length
        );

        @CField("is_allocated") public int get_is_allocated();
        @CField("is_allocated") public void set_is_allocated(
            int is_allocated
        );
    }

    /** The structure for the source locations */
    @CContext(LibDirectives.class)
    @CStruct("lkt_source_location")
    public interface SourceLocationNative extends PointerBase {
        @CField("line") public int get_line();
        @CField("line") public void set_line(
            int line
        );

        @CField("column") public short get_column();
        @CField("column") public void set_column(
            short column
        );
    }

    /** The structure for the source location ranges */
    @CContext(LibDirectives.class)
    @CStruct("lkt_source_location_range")
    public interface SourceLocationRangeNative extends PointerBase {
        @CField("start.line") public int get_start_line();
        @CField("start.line") public void set_start_line(
            int start_line
        );

        @CField("start.column") public short get_start_column();
        @CField("start.column") public void set_start_column(
            short start_column
        );

        @CField("end.line") public int get_end_line();
        @CField("end.line") public void set_end_line(
            int end_line
        );

        @CField("end.column") public short get_end_column();
        @CField("end.column") public void set_end_column(
            short end_column
        );
    }

    /** The structure for the diagnostic */
    @CContext(LibDirectives.class)
    @CStruct("lkt_diagnostic")
    public interface DiagnosticNative extends PointerBase {
        @CField("sloc_range.start.line") public int get_start_line();
        @CField("sloc_range.start.line") public void set_start_line(
            int start_line
        );

        @CField("sloc_range.start.column") public short get_start_column();
        @CField("sloc_range.start.column") public void set_start_column(
            short start_column
        );

        @CField("sloc_range.end.line") public int get_end_line();
        @CField("sloc_range.end.line") public void set_end_line(
            int end_line
        );

        @CField("sloc_range.end.column") public short get_end_column();
        @CField("sloc_range.end.column") public void set_end_column(
            short end_column
        );

        @CField("message.chars") public CIntPointer get_message_chars();
        @CField("message.chars") public void set_message_chars(
            CIntPointer chars
        );

        @CField("message.length") public long get_message_length();
        @CField("message.length") public void set_message_length(
            long length
        );

        @CField("message.is_allocated") public int get_message_is_allocated();
        @CField("message.is_allocated") public void set_message_is_allocated(
            int is_allocated
        );
    }

    /** The file reader is just a pointer */
    public interface FileReaderNative extends Pointer {}

    /** The unit provider is just a pointer */
    public interface UnitProviderNative extends Pointer {}

    /** The event handler is just a pointer */
    public interface EventHandlerNative extends Pointer {}

    /** The event handler unit requested callback type */
    public interface UnitRequestedFunctionPointer extends CFunctionPointer {
        @InvokeCFunctionPointer
        void invoke(
            VoidPointer data,
            AnalysisContextNative context,
            TextNative name,
            AnalysisUnitNative from,
            boolean found,
            boolean is_not_found_error
        );
    }

    /** The event handler unit parsed callback type */
    public interface UnitParsedFunctionPointer extends CFunctionPointer {
        @InvokeCFunctionPointer
        void invoke(
            VoidPointer data,
            AnalysisContextNative context,
            AnalysisUnitNative unit,
            boolean reparsed
        );
    }

    /** Anonymous structure for the token data handler */
    @RawStructure
    public interface TokenDataHandlerNative extends PointerBase {
        @RawField public long version();
    }

    /** The structure representing a token */
    @CContext(LibDirectives.class)
    @CStruct("lkt_token")
    public interface TokenNative extends PointerBase {
        @CField("context") public AnalysisContextNative get_context();
        @CField("context") public void set_context(
            AnalysisContextNative context
        );

        @CField("token_data") public TokenDataHandlerNative get_data();
        @CField("token_data") public void set_data(
            TokenDataHandlerNative data
        );

        @CField("token_index") public int get_token_index();
        @CField("token_index") public void set_token_index(
            int token_index
        );

        @CField("trivia_index") public int get_trivia_index();
        @CField("trivia_index") public void set_trivia_index(
            int trivia_index
        );
    }

    /** Anonymous strucutre for analysis context */
    @RawStructure
    public interface AnalysisContextNative extends PointerBase {
        @RawField public long serial_number();
    }

    /** Anonymous strucutre for analysis unit */
    @RawStructure
    public interface AnalysisUnitNative extends PointerBase {
        @RawField public long version_number();
    }

    /** The structure for reswriting apply results */
    @CContext(LibDirectives.class)
    @CStruct("lkt_rewriting_apply_result")
    public interface RewritingApplyResultNative extends PointerBase {
        @CField("success") public int get_success();
        @CField("success") public void set_success(int success);

        @CField("unit") public AnalysisUnitNative get_unit();
        @CField("unit") public void set_unit(AnalysisUnitNative unit);

        @CField("diagnostics_count") public int get_diagnostics_count();
        @CField("diagnostics_count") public void set_diagnostics_count(
            int diagnostics_count
        );

        @CField("diagnostics") public DiagnosticNative get_diagnostics();
        @CField("diagnostics") public void set_diagnostics(
            DiagnosticNative diagnostics
        );
    }

    /** The rewriting context type is just a pointer */
    public interface RewritingContextNative extends Pointer {}

    /** The rewriting unit native type is just a pointer */
    public interface RewritingUnitNative extends Pointer {}

    /** The rewriting node native type is just a pointer */
    public interface RewritingNodeNative extends Pointer {}

    // ===== Generated structures =====

        
    
    

    /** The structure for the langkit lkt_internal_decoded_char_value */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_decoded_char_value")
    public interface DecodedCharValueNative extends PointerBase {
        @CField("value")
        public int
        get_value();

        @CField("value")
        public void
        set_value(
            int val
        );

        @CFieldAddress("value")
        public <T extends PointerBase> T address_value();

        @CField("has_error")
        public byte
        get_has_error();

        @CField("has_error")
        public void
        set_has_error(
            byte val
        );

        @CFieldAddress("has_error")
        public <T extends PointerBase> T address_has_error();

        @CField("error_sloc")
        public SourceLocationNative
        get_error_sloc();

        @CField("error_sloc")
        public void
        set_error_sloc(
            SourceLocationNative val
        );

        @CFieldAddress("error_sloc")
        public <T extends PointerBase> T address_error_sloc();

        @CField("error_message")
        public StringNative
        get_error_message();

        @CField("error_message")
        public void
        set_error_message(
            StringNative val
        );

        @CFieldAddress("error_message")
        public <T extends PointerBase> T address_error_message();


    }

        
    
    

    /** The structure for the langkit lkt_internal_decoded_string_value */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_decoded_string_value")
    public interface DecodedStringValueNative extends PointerBase {
        @CField("value")
        public StringNative
        get_value();

        @CField("value")
        public void
        set_value(
            StringNative val
        );

        @CFieldAddress("value")
        public <T extends PointerBase> T address_value();

        @CField("has_error")
        public byte
        get_has_error();

        @CField("has_error")
        public void
        set_has_error(
            byte val
        );

        @CFieldAddress("has_error")
        public <T extends PointerBase> T address_has_error();

        @CField("error_sloc")
        public SourceLocationNative
        get_error_sloc();

        @CField("error_sloc")
        public void
        set_error_sloc(
            SourceLocationNative val
        );

        @CFieldAddress("error_sloc")
        public <T extends PointerBase> T address_error_sloc();

        @CField("error_message")
        public StringNative
        get_error_message();

        @CField("error_message")
        public void
        set_error_message(
            StringNative val
        );

        @CFieldAddress("error_message")
        public <T extends PointerBase> T address_error_message();


    }

        
        
    
    

    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_metadata")
    public interface MetadataNative extends PointerBase {
        @CField("dummy") public byte get_dummy();
        @CField("dummy") public void set_dummy(byte dummy);
    }

        
    
    

    /** The structure for the langkit lkt_internal_entity_info */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_entity_info")
    public interface EntityInfoNative extends PointerBase {
        @CField("md.dummy")
        public byte
        get_md_dummy();

        @CField("md.dummy")
        public void
        set_md_dummy(
            byte val
        );

        @CFieldAddress("md.dummy")
        public <T extends PointerBase> T address_md_dummy();

        @CField("rebindings")
        public Pointer
        get_rebindings();

        @CField("rebindings")
        public void
        set_rebindings(
            Pointer val
        );

        @CFieldAddress("rebindings")
        public <T extends PointerBase> T address_rebindings();

        @CField("from_rebound")
        public byte
        get_from_rebound();

        @CField("from_rebound")
        public void
        set_from_rebound(
            byte val
        );

        @CFieldAddress("from_rebound")
        public <T extends PointerBase> T address_from_rebound();


            
        @CFieldAddress("md")
        public MetadataNative
        address_md();

    }

    
    

    /** The structure for the langkit lkt_node */
    @CContext(LibDirectives.class)
    @CStruct("lkt_node")
    public interface EntityNative extends PointerBase {
        @CField("node")
        public Pointer
        get_node();

        @CField("node")
        public void
        set_node(
            Pointer val
        );

        @CFieldAddress("node")
        public <T extends PointerBase> T address_node();

        @CField("info.md.dummy")
        public byte
        get_info_md_dummy();

        @CField("info.md.dummy")
        public void
        set_info_md_dummy(
            byte val
        );

        @CFieldAddress("info.md.dummy")
        public <T extends PointerBase> T address_info_md_dummy();

        @CField("info.rebindings")
        public Pointer
        get_info_rebindings();

        @CField("info.rebindings")
        public void
        set_info_rebindings(
            Pointer val
        );

        @CFieldAddress("info.rebindings")
        public <T extends PointerBase> T address_info_rebindings();

        @CField("info.from_rebound")
        public byte
        get_info_from_rebound();

        @CField("info.from_rebound")
        public void
        set_info_from_rebound(
            byte val
        );

        @CFieldAddress("info.from_rebound")
        public <T extends PointerBase> T address_info_from_rebound();


            
        @CFieldAddress("info.md")
        public EntityInfoNative
        address_info();

    }

        
        
        
        
        
    
    

    /** The structure for the langkit lkt_internal_semantic_result */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_semantic_result")
    public interface SemanticResultNative extends PointerBase {
        @CField("node")
        public Pointer
        get_node();

        @CField("node")
        public void
        set_node(
            Pointer val
        );

        @CFieldAddress("node")
        public <T extends PointerBase> T address_node();

        @CField("result_type.node")
        public Pointer
        get_result_type_node();

        @CField("result_type.node")
        public void
        set_result_type_node(
            Pointer val
        );

        @CFieldAddress("result_type.node")
        public <T extends PointerBase> T address_result_type_node();

        @CField("result_type.info.md.dummy")
        public byte
        get_result_type_info_md_dummy();

        @CField("result_type.info.md.dummy")
        public void
        set_result_type_info_md_dummy(
            byte val
        );

        @CFieldAddress("result_type.info.md.dummy")
        public <T extends PointerBase> T address_result_type_info_md_dummy();

        @CField("result_type.info.rebindings")
        public Pointer
        get_result_type_info_rebindings();

        @CField("result_type.info.rebindings")
        public void
        set_result_type_info_rebindings(
            Pointer val
        );

        @CFieldAddress("result_type.info.rebindings")
        public <T extends PointerBase> T address_result_type_info_rebindings();

        @CField("result_type.info.from_rebound")
        public byte
        get_result_type_info_from_rebound();

        @CField("result_type.info.from_rebound")
        public void
        set_result_type_info_from_rebound(
            byte val
        );

        @CFieldAddress("result_type.info.from_rebound")
        public <T extends PointerBase> T address_result_type_info_from_rebound();

        @CField("result_decl.node")
        public Pointer
        get_result_decl_node();

        @CField("result_decl.node")
        public void
        set_result_decl_node(
            Pointer val
        );

        @CFieldAddress("result_decl.node")
        public <T extends PointerBase> T address_result_decl_node();

        @CField("result_decl.info.md.dummy")
        public byte
        get_result_decl_info_md_dummy();

        @CField("result_decl.info.md.dummy")
        public void
        set_result_decl_info_md_dummy(
            byte val
        );

        @CFieldAddress("result_decl.info.md.dummy")
        public <T extends PointerBase> T address_result_decl_info_md_dummy();

        @CField("result_decl.info.rebindings")
        public Pointer
        get_result_decl_info_rebindings();

        @CField("result_decl.info.rebindings")
        public void
        set_result_decl_info_rebindings(
            Pointer val
        );

        @CFieldAddress("result_decl.info.rebindings")
        public <T extends PointerBase> T address_result_decl_info_rebindings();

        @CField("result_decl.info.from_rebound")
        public byte
        get_result_decl_info_from_rebound();

        @CField("result_decl.info.from_rebound")
        public void
        set_result_decl_info_from_rebound(
            byte val
        );

        @CFieldAddress("result_decl.info.from_rebound")
        public <T extends PointerBase> T address_result_decl_info_from_rebound();

        @CField("has_error")
        public byte
        get_has_error();

        @CField("has_error")
        public void
        set_has_error(
            byte val
        );

        @CFieldAddress("has_error")
        public <T extends PointerBase> T address_has_error();

        @CField("error_message")
        public StringNative
        get_error_message();

        @CField("error_message")
        public void
        set_error_message(
            StringNative val
        );

        @CFieldAddress("error_message")
        public <T extends PointerBase> T address_error_message();

        @CField("exempt_analysis")
        public byte
        get_exempt_analysis();

        @CField("exempt_analysis")
        public void
        set_exempt_analysis(
            byte val
        );

        @CFieldAddress("exempt_analysis")
        public <T extends PointerBase> T address_exempt_analysis();


            
        @CFieldAddress("result_type.node")
        public EntityNative
        address_result_type();

            
        @CFieldAddress("result_decl.node")
        public EntityNative
        address_result_decl();

    }

        
        
        
        
        
        
        
    
    

    /** The structure for the langkit lkt_internal_tree_semantic_result */
    @CContext(LibDirectives.class)
    @CStruct("lkt_internal_tree_semantic_result")
    public interface TreeSemanticResultNative extends PointerBase {
        @CField("results")
        public SemanticResultArrayNative
        get_results();

        @CField("results")
        public void
        set_results(
            SemanticResultArrayNative val
        );

        @CFieldAddress("results")
        public <T extends PointerBase> T address_results();

        @CField("has_error")
        public byte
        get_has_error();

        @CField("has_error")
        public void
        set_has_error(
            byte val
        );

        @CFieldAddress("has_error")
        public <T extends PointerBase> T address_has_error();


    }


    // ===== Generated arrays =====

    
    

    /**
     * The native structure of the lkt_node_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "lkt_node_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface LktNodeArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }

    
    

    /**
     * The native structure of the lkt_node_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "lkt_node_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface TypeDeclArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }

    
    

    /**
     * The native structure of the lkt_internal_semantic_result_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "lkt_internal_semantic_result_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface SemanticResultArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }


    // ===== Generated iterators =====


    // ===== Native function definitions =====

    /** This class contains all native function definitions for NI */
    @CContext(LibDirectives.class)
    public static final class NI_LIB {

        // ----- Language specific functions -----

        


        // ----- Entry point literals -----

        /**
         * This entry point literal provide a pointer to the unit requested
         * callback.
         */
        public static final CEntryPointLiteral<UnitRequestedFunctionPointer>
            unitRequestedFunction = CEntryPointLiteral.create(
                Liblktlang.class,
                "unitRequested",
                IsolateThread.class,
                AnalysisContextNative.class,
                TextNative.class,
                AnalysisUnitNative.class,
                byte.class,
                byte.class
            );

        /**
         * This entry point literal provide a pointer to the unit parsed
         * callback.
         */
        public static final CEntryPointLiteral<UnitParsedFunctionPointer>
            unitParsedFunction = CEntryPointLiteral.create(
                Liblktlang.class,
                "unitParsed",
                IsolateThread.class,
                AnalysisContextNative.class,
                AnalysisUnitNative.class,
                byte.class
            );

        // ----- Util functions -----

        /** Util function to free langkit side allocated memory */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_free(
            PointerBase pointer
        );

        // ----- Exception functions -----

        /** Get the last exception raised by langkit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native LangkitExceptionNative
        lkt_get_last_exception();

        // ----- Big integer functions -----

        /** Create a big integer from a text */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native BigIntegerNative lkt_create_big_integer(
            TextNative text
        );

        /** Get the text representation of a big integer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_big_integer_text(
            BigIntegerNative big_integer,
            TextNative text
        );

        /** Decrease the reference counter of the big integer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_big_integer_decref(
            BigIntegerNative big_integer
        );

        // ----- Symbol functions -----

        /** Create a new symbol in the given context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_context_symbol(
            AnalysisContextNative context,
            TextNative text,
            SymbolNative res
        );

        /** Get the text of a given symbol */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_symbol_text(
            SymbolNative symbol,
            TextNative text
        );

        // ----- String functions -----

        /** Create a new string wrapper in langkit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native StringNative lkt_create_string(
            CIntPointer content,
            int length
        );

        /** Decrease the reference counter of a string */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_string_dec_ref(
            StringNative string
        );

        // ----- Text functions -----

        /** Destroy a text in the memory */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_destroy_text(
            TextNative text
        );

        // ----- Rewriting result functions -----

        /** Free a rewriting apply result */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_free_apply_result(
            RewritingApplyResultNative apply_result
        );

        // ----- File reader functions -----

        /** Decrease the reference counter of the given file reader */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_dec_ref_file_reader(
            FileReaderNative fileReader
        );

        // ----- Unit provider functions -----

        /** Decrease the ref counter of the unit provider */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_dec_ref_unit_provider(
            UnitProviderNative unitProvider
        );

        // ----- Event handler functions -----

        /** Create a new event handler */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native EventHandlerNative lkt_create_event_handler(
            VoidPointer data,
            VoidPointer destroy_callback,
            UnitRequestedFunctionPointer unit_requested_func,
            UnitParsedFunctionPointer unit_parsed_func
        );

        /** Decrease the ref counter of the event handler */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_dec_ref_event_handler(
            EventHandlerNative eventHandler
        );

        // ----- Token functions -----

        /**
         * Kind for this token.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_get_kind(
            TokenNative token
        );

        /**
         * Return the source location range of the given token.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_token_sloc_range(
            TokenNative token,
            SourceLocationRangeNative result
        );

        /** Get the next token */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_token_next(
            TokenNative token,
            TokenNative res
        );

        /** Get the previous token */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_token_previous(
            TokenNative token,
            TokenNative res
        );

        /** Get if two tokens are equivalent */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native byte lkt_token_is_equivalent(
            TokenNative left,
            TokenNative right
        );

        /** Get the text in a token range */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_token_range_text(
            TokenNative start,
            TokenNative end,
            TextNative res
        );

        // ----- Analysis context functions -----

        /** Allocate a new analysis context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisContextNative
        lkt_allocate_analysis_context();

        /** Create a new analysis context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void
        lkt_initialize_analysis_context(
            AnalysisContextNative context,
            CCharPointer charset,
            FileReaderNative file_reader,
            UnitProviderNative unit_provider,
            EventHandlerNative event_handler,
            int with_trivia,
            int tab_stop
        );

        /** Increase the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_context_incref(
            AnalysisContextNative context
        );

        /** Decrease the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_context_decref(
            AnalysisContextNative context
        );

        // ----- Analysis unit functions -----

        /** Get a unit from a file */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        lkt_get_analysis_unit_from_file(
            AnalysisContextNative context,
            CCharPointer file_name,
            CCharPointer charset,
            int reparse,
            int rule
        );

        /** Get a unit from a buffer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        lkt_get_analysis_unit_from_buffer(
            AnalysisContextNative context,
            CCharPointer file_name,
            CCharPointer charset,
            CCharPointer buffer,
            long buffer_size,
            int rule
        );

        /** Get a unit from the unit provider. */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        lkt_get_analysis_unit_from_provider(
            AnalysisContextNative context,
            TextNative name,
            int kind,
            CCharPointer charset,
            int reparse
        );

        /** Get the root of an analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_unit_root(
            AnalysisUnitNative unit,
            EntityNative res
        );

        /** Get the file name for a given unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native CCharPointer lkt_unit_filename(
            AnalysisUnitNative unit
        );

        /** Get the token count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_unit_token_count(
            AnalysisUnitNative unit
        );

        /** Get the trivia count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_unit_trivia_count(
            AnalysisUnitNative unit
        );

        /** Get the first token of an analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_unit_first_token(
            AnalysisUnitNative unit,
            TokenNative res
        );

        /** Get the last token of an analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_unit_last_token (
            AnalysisUnitNative unit,
            TokenNative res
        );

        /** Get the context for a given unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisContextNative lkt_unit_context(
            AnalysisUnitNative unit
        );

        /** Get the diagnostic count of the unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_unit_diagnostic_count(
            AnalysisUnitNative unit
        );

        /** Get the nth diagnostic for the unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_unit_diagnostic(
            AnalysisUnitNative unit,
            int n,
            DiagnosticNative diagnostic
        );

        // ----- Rewriting context functions -----

        /** Start a new rewriting session on the given analysis context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingContextNative
        lkt_rewriting_start_rewriting(
            AnalysisContextNative analysis_context
        );

        /** Get the analysis context from the given rewriting context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisContextNative
        lkt_rewriting_handle_to_context(
            RewritingContextNative rewriting_context
        );

        /** Get a pointer to the rewriting units owned by the context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native WordPointer lkt_rewriting_unit_handles(
            RewritingContextNative rewriting_context
        );

        /** Create a node in the rewriting context and return it */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_create_node(
            RewritingContextNative rewriting_context,
            int node_kind
        );

        /** Create a node in the rewriting context with the given children */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_create_regular_node(
            RewritingContextNative rewriting_context,
            int node_kind,
            WordPointer children,
            int count
        );

        /** Create a token node in the rewriting context and return it */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_create_token_node(
            RewritingContextNative rewriting_context,
            int node_kind,
            TextNative node_text
        );

        /** Create a new node tree from the given template */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_create_from_template(
            RewritingContextNative rewriting_context,
            TextNative template_text,
            WordPointer arguments,
            int count,
            int rule
        );

        /** Apply the rewriting session and close it if success */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_apply(
            RewritingContextNative rewriting_context,
            RewritingApplyResultNative apply_result
        );

        /** Abort the rewriting session */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_abort_rewriting(
            RewritingContextNative rewriting_context
        );

        // ----- Rewriting unit functions -----

        /** Get the rewriting unit corresponding to the given analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingUnitNative
        lkt_rewriting_unit_to_handle(
            AnalysisUnitNative unit
        );

        /** Get the analysis unit corresponding to the given rewriting unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        lkt_rewriting_handle_to_unit(
            RewritingUnitNative rewriting_unit
        );

        /** Get the root of the given rewriting unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative lkt_rewriting_unit_root(
            RewritingUnitNative rewriting_unit
        );

        /** Set the root of the rewriting unit to the rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_unit_set_root(
            RewritingUnitNative rewriting_unit,
            RewritingNodeNative rewriting_node
        );

        /** Unparse the rewriting unit to get its textual content */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_unit_unparse(
            RewritingUnitNative rewriting_unit,
            TextNative result
        );

        // ----- Rewriting node functions -----

        /** Get the rewriting node from the given parsed node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_node_to_handle(
            Pointer node
        );

        /** Get the parsed node from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native Pointer lkt_rewriting_handle_to_node(
            RewritingNodeNative rewriting_node
        );

        /** Get the rewriting context from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingContextNative
        lkt_rewriting_node_to_context(
            RewritingNodeNative rewriting_node
        );

        /** Clone the given rewriting node and return the copy */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative lkt_rewriting_clone(
            RewritingNodeNative to_clone
        );

        /** Unparse the given rewriting node in the given text */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_node_unparse(
            RewritingNodeNative rewriting_node,
            TextNative result
        );

        /** Get the kind of the rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_rewriting_kind(
            RewritingNodeNative rewriting_node
        );

        /** Get the rewriting node image */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_node_image(
            RewritingNodeNative rewriting_node,
            TextNative result
        );

        /** Return whether the node is tied to a rewriting unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_rewriting_tied(
            RewritingNodeNative rewriting_node
        );

        /** Return the parent of the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative lkt_rewriting_parent(
            RewritingNodeNative rewriting_node
        );

        /** Get the rewriting node children */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_children(
            RewritingNodeNative rewriting_node,
            WordPointer result_reference,
            CIntPointer result_count
        );

        /** Get the child at the given member reference */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative lkt_rewriting_child(
            RewritingNodeNative parent,
            int child_member_reference
        );

        /** Set the given child at the given member reference */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_set_child(
            RewritingNodeNative parent,
            int child_member_reference,
            RewritingNodeNative new_child
        );

        /** Replace the rewriting node by the new one */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_replace(
            RewritingNodeNative rewriting_node,
            RewritingNodeNative new_node
        );

        /** Get the first child of the rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_first_child(
            RewritingNodeNative parent
        );

        /** Get the last child of the rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_last_child(
            RewritingNodeNative parent
        );

        /** Get the next child from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_next_child(
            RewritingNodeNative rewriting_node
        );

        /** Get the previous child from the given rewriting node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native RewritingNodeNative
        lkt_rewriting_previous_child(
            RewritingNodeNative rewriting_node
        );

        /** Insert the provided rewriting node before the other node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_insert_before(
            RewritingNodeNative rewriting_node,
            RewritingNodeNative to_insert
        );

        /** Insert the provided rewriting node after the other node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_insert_after(
            RewritingNodeNative rewriting_node,
            RewritingNodeNative to_insert
        );

        /**
         * Insert the provided rewriting node at the beginning of the
         * children
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_insert_first(
            RewritingNodeNative rewriting_node,
            RewritingNodeNative to_insert
        );

        /** Insert the provided rewriting node at the end of the children */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_insert_last(
            RewritingNodeNative rewriting_node,
            RewritingNodeNative to_insert
        );

        /** Remove the given rewriting node from its list parent */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_remove_child(
            RewritingNodeNative to_remove
        );

        /** Get the text of the rewriting token node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_text(
            RewritingNodeNative rewriting_node,
            TextNative result
        );

        /** Set the text of the rewriting token node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_rewriting_set_text(
            RewritingNodeNative rewriting_node,
            TextNative text
        );

        // ----- Array functions -----

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native LktNodeArrayNative lkt_node_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_node_array_dec_ref(LktNodeArrayNative array);

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native SemanticResultArrayNative lkt_internal_semantic_result_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_semantic_result_array_dec_ref(SemanticResultArrayNative array);


        // ----- Structure functions -----

            
        
    

        /**
         * Decreate the reference counter of the given struct.
         *
         * @param structNative The structure to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_decoded_char_value_dec_ref(
            DecodedCharValueNative structNative
        );

            
        
    

        /**
         * Decreate the reference counter of the given struct.
         *
         * @param structNative The structure to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_decoded_string_value_dec_ref(
            DecodedStringValueNative structNative
        );

            
            
        
    


            
        
    


        
    


            
            
            
            
            
        
    

        /**
         * Decreate the reference counter of the given struct.
         *
         * @param structNative The structure to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_semantic_result_dec_ref(
            SemanticResultNative structNative
        );

            
            
            
            
            
            
            
        
    

        /**
         * Decreate the reference counter of the given struct.
         *
         * @param structNative The structure to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_internal_tree_semantic_result_dec_ref(
            TreeSemanticResultNative structNative
        );


        // ----- Node functions -----

        /** Create a bare entity from a node pointer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_create_bare_entity(
            Pointer node,
            EntityNative result
        );

        /** Return whether the two given entities are equal */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_node_is_equivalent(
            EntityNative entity_left,
            EntityNative entity_right
        );

        /** Get the hash of a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_node_hash(
            EntityNative entity
        );

        /** Get the type of a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_node_kind(
            EntityNative entity
        );

        /** Get the text from a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_node_text(
            EntityNative entity,
            TextNative text
        );

        /** Get the source location range for a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_node_sloc_range(
            EntityNative entity,
            SourceLocationRangeNative slocr
        );

        /** Get the number of children for a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_node_children_count(
            EntityNative entity
        );

        /** Get the nth child for the node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_node_child(
            EntityNative entity,
            int n,
            EntityNative res
        );

        /** Get if the node is a token node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_node_is_token_node(
            EntityNative entity
        );

        /** Get the unit of the node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative lkt_node_unit(
            EntityNative entity
        );

        /** Get the image of a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void lkt_node_image(
            EntityNative entity,
            TextNative text
        );

        // ----- Node fields accessors and properties -----

        
    

            

        /** Isomethod of lkt_lkt_node_p_node_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_node_gen_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_node_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_node_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_token_node_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_token_node_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_error_node_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_error_node_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_char_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_char_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_int_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_int_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_bool_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_bool_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_bigint_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_bigint_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_string_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_string_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_symbol_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_symbol_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_property_error_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_property_error_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_regexp_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_regexp_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_array_gen_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_array_gen_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_array_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_array_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_astlist_gen_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_astlist_gen_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_astlist_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_astlist_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_iterator_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_iterator_gen_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_iterator_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_iterator_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_analysis_unit_gen_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_analysis_unit_gen_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_analysis_unit_trait langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_analysis_unit_trait(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_p_topmost_invalid_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_p_topmost_invalid_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_parent langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_parent(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_parents langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_parents(
            EntityNative node,
            byte with_self,
            WordPointer result
        );
            

        /** Isomethod of lkt_lkt_node_children langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_children(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_lkt_node_token_start langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_token_start(
            EntityNative node,
            TokenNative result
        );
            

        /** Isomethod of lkt_lkt_node_token_end langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_token_end(
            EntityNative node,
            TokenNative result
        );
            

        /** Isomethod of lkt_lkt_node_child_index langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_child_index(
            EntityNative node,
            CIntPointer result
        );
            

        /** Isomethod of lkt_lkt_node_previous_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_previous_sibling(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_next_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_next_sibling(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lkt_node_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_unit(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_lkt_node_is_ghost langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_is_ghost(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of lkt_lkt_node_full_sloc_image langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lkt_node_full_sloc_image(
            EntityNative node,
            WordPointer result
        );

        
    

            

        /** Isomethod of lkt_base_lexer_case_rule_alt_f_send langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_base_lexer_case_rule_alt_f_send(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_lexer_case_rule_cond_alt_f_cond_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_case_rule_cond_alt_f_cond_exprs(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_class_qualifier_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_class_qualifier_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_decl_f_syn_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_f_syn_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_decl_p_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_name(
            EntityNative node,
            SymbolNative result
        );
            

        /** Isomethod of lkt_decl_p_full_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_full_name(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_decl_p_decl_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_decl_type_name(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_decl_p_as_bare_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_p_as_bare_decl(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_base_grammar_rule_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_base_grammar_rule_decl_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_base_val_decl_p_get_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_base_val_decl_p_get_type(
            EntityNative node,
            byte no_inference,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_explicitly_typed_decl_f_decl_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_explicitly_typed_decl_f_decl_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_component_decl_f_default_val langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_component_decl_f_default_val(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_fun_arg_decl_f_decl_annotations langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_fun_arg_decl_f_decl_annotations(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_val_decl_f_val langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_val_decl_f_val(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_fun_decl_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_fun_decl_f_args(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_fun_decl_f_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_fun_decl_f_return_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_fun_decl_f_body langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_fun_decl_f_body(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_env_spec_decl_f_actions langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_env_spec_decl_f_actions(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_generic_decl_f_generic_formal_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_decl_f_generic_formal_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_generic_decl_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_decl_f_decl(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_decl_f_rules langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_decl_f_rules(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_decl_p_lexer langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_decl_p_lexer(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_lexer_decl_f_rules langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_decl_f_rules(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_lexer_family_decl_f_rules langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_family_decl_f_rules(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_type_decl_f_traits langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_f_traits(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_type_decl_p_is_class langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_p_is_class(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of lkt_type_decl_p_implemented_traits langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_p_implemented_traits(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_type_decl_f_syn_base_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_f_syn_base_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_type_decl_p_base_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_p_base_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_type_decl_p_base_types langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_p_base_types(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_type_decl_p_root_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_p_root_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_type_decl_p_is_subtype langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_p_is_subtype(
            EntityNative node,
            EntityNative potential_base,
            CCharPointer result
        );
            

        /** Isomethod of lkt_type_decl_p_is_generic langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_decl_p_is_generic(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_generic_formal_type_decl_f_has_class langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_formal_type_decl_f_has_class(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_instantiated_generic_type_p_get_inner_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_instantiated_generic_type_p_get_inner_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_instantiated_generic_type_p_get_actuals langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_instantiated_generic_type_p_get_actuals(
            EntityNative node,
            WordPointer result
        );
            

        /** Isomethod of lkt_instantiated_generic_type_p_get_instantiated_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_instantiated_generic_type_p_get_instantiated_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_named_type_decl_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_named_type_decl_f_decls(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_enum_class_decl_f_branches langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_enum_class_decl_f_branches(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_enum_type_decl_f_literals langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_enum_type_decl_f_literals(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_decl_annotation_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_annotation_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_decl_annotation_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_annotation_f_params(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_decl_annotation_params_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_decl_annotation_params_f_params(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_elsif_branch_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_elsif_branch_f_cond_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_elsif_branch_f_then_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_elsif_branch_f_then_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_enum_class_case_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_enum_class_case_f_decls(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_excludes_null_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_excludes_null_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_expr_p_in_type_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_expr_p_in_type_ref(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of lkt_expr_p_is_regular_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_expr_p_is_regular_expr(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of lkt_expr_p_expr_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_expr_p_expr_type(
            EntityNative node,
            SemanticResultNative result
        );
            

        /** Isomethod of lkt_expr_p_check_expr_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_expr_p_check_expr_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_expr_p_expr_context_free_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_expr_p_expr_context_free_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_expr_p_referenced_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_expr_p_referenced_decl(
            EntityNative node,
            SemanticResultNative result
        );
            

        /** Isomethod of lkt_expr_p_check_referenced_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_expr_p_check_referenced_decl(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_any_of_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_any_of_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_any_of_f_values langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_any_of_f_values(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_array_literal_f_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_array_literal_f_exprs(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_array_literal_f_element_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_array_literal_f_element_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_base_dot_expr_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_base_dot_expr_f_prefix(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_base_dot_expr_f_suffix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_base_dot_expr_f_suffix(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_bin_op_f_left langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_bin_op_f_left(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_bin_op_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_bin_op_f_op(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_bin_op_f_right langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_bin_op_f_right(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_block_expr_f_val_defs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_block_expr_f_val_defs(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_block_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_block_expr_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_call_expr_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_call_expr_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_call_expr_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_call_expr_f_args(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_call_expr_p_called_object_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_call_expr_p_called_object_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_call_expr_p_called_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_call_expr_p_called_decl(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_cast_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_cast_expr_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_cast_expr_f_excludes_null langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_cast_expr_f_excludes_null(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_cast_expr_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_cast_expr_f_dest_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_error_on_null_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_error_on_null_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_generic_instantiation_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_instantiation_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_generic_instantiation_f_args langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_instantiation_f_args(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_generic_instantiation_p_instantiated_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_instantiation_p_instantiated_decl(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_grammar_discard_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_discard_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_dont_skip_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_dont_skip_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_dont_skip_f_dont_skip langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_dont_skip_f_dont_skip(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_list_f_list_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_list_f_list_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_list_f_kind langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_list_f_kind(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_list_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_list_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_list_f_sep langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_list_f_sep(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_null_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_null_f_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_opt_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_error_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_opt_error_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_error_group_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_opt_error_group_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_opt_group_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_opt_group_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_or_expr_f_sub_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_or_expr_f_sub_exprs(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_pick_f_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_pick_f_exprs(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_grammar_predicate_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_predicate_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_predicate_f_prop_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_predicate_f_prop_ref(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_rule_ref_f_node_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_rule_ref_f_node_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_skip_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_skip_f_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_stop_cut_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_stop_cut_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_parse_node_expr_f_node_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_parse_node_expr_f_node_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_parse_node_expr_f_sub_exprs langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_parse_node_expr_f_sub_exprs(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_token_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_lit_p_denoted_value(
            EntityNative node,
            DecodedStringValueNative result
        );

        
    

            

        /** Isomethod of lkt_token_no_case_lit_f_lit langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_no_case_lit_f_lit(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_token_pattern_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_pattern_lit_p_denoted_value(
            EntityNative node,
            DecodedStringValueNative result
        );

        
    

            

        /** Isomethod of lkt_token_ref_f_token_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_ref_f_token_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_token_ref_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_token_ref_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_if_expr_f_cond_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_if_expr_f_cond_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_if_expr_f_then_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_if_expr_f_then_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_if_expr_f_alternatives langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_if_expr_f_alternatives(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_if_expr_f_else_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_if_expr_f_else_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_isa_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_isa_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_isa_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_isa_f_dest_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_keep_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_keep_expr_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_keep_expr_f_keep_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_keep_expr_f_keep_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_lambda_expr_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lambda_expr_f_params(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lambda_expr_f_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lambda_expr_f_return_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lambda_expr_f_body langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lambda_expr_f_body(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_char_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_char_lit_p_denoted_value(
            EntityNative node,
            DecodedCharValueNative result
        );

        
    

            

        /** Isomethod of lkt_null_lit_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_null_lit_f_dest_type(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_string_lit_p_denoted_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_string_lit_p_denoted_value(
            EntityNative node,
            DecodedStringValueNative result
        );
            

        /** Isomethod of lkt_string_lit_p_is_prefixed_string langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_string_lit_p_is_prefixed_string(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of lkt_string_lit_p_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_string_lit_p_prefix(
            EntityNative node,
            CIntPointer result
        );
            

        /** Isomethod of lkt_string_lit_p_is_regexp_literal langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_string_lit_p_is_regexp_literal(
            EntityNative node,
            CCharPointer result
        );

        
    

            

        /** Isomethod of lkt_block_string_lit_f_lines langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_block_string_lit_f_lines(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of lkt_logic_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_logic_expr_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_match_expr_f_match_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_match_expr_f_match_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_match_expr_f_branches langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_match_expr_f_branches(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_not_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_not_expr_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_paren_expr_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_paren_expr_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_raise_expr_f_dest_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_raise_expr_f_dest_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_raise_expr_f_except_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_raise_expr_f_except_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_subscript_expr_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_subscript_expr_f_prefix(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_subscript_expr_f_index langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_subscript_expr_f_index(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_try_expr_f_try_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_try_expr_f_try_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_try_expr_f_or_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_try_expr_f_or_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_un_op_f_op langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_un_op_f_op(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_un_op_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_un_op_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_full_decl_f_doc langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_full_decl_f_doc(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_full_decl_f_decl_annotations langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_full_decl_f_decl_annotations(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_full_decl_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_full_decl_f_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_full_decl_p_has_annotation langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_full_decl_p_has_annotation(
            EntityNative node,
            SymbolNative name,
            CCharPointer result
        );
            

        /** Isomethod of lkt_full_decl_p_check_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_full_decl_p_check_decl(
            EntityNative node,
            TreeSemanticResultNative result
        );

        
    

            

        /** Isomethod of lkt_grammar_list_sep_f_token langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_list_sep_f_token(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_grammar_list_sep_f_extra langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_grammar_list_sep_f_extra(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_import_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_import_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_import_p_referenced_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_import_p_referenced_unit(
            EntityNative node,
            WordPointer result
        );

        
    

            

        /** Isomethod of lkt_langkit_root_f_imports langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_langkit_root_f_imports(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_langkit_root_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_langkit_root_f_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_langkit_root_p_check_semantic langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_langkit_root_p_check_semantic(
            EntityNative node,
            TreeSemanticResultNative result
        );

        
    

            

        /** Isomethod of lkt_lexer_case_rule_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_case_rule_f_expr(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lexer_case_rule_f_alts langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_case_rule_f_alts(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_lexer_case_rule_send_f_sent langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_case_rule_send_f_sent(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_lexer_case_rule_send_f_match_size langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_lexer_case_rule_send_f_match_size(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_match_branch_f_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_match_branch_f_decl(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_match_branch_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_match_branch_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of lkt_param_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_param_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_param_f_value langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_param_f_value(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_type_ref_p_designated_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_type_ref_p_designated_type(
            EntityNative node,
            EntityNative result
        );

        
    


        
    

            

        /** Isomethod of lkt_function_type_ref_f_args_types langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_function_type_ref_f_args_types(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_function_type_ref_f_return_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_function_type_ref_f_return_type(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_generic_type_ref_f_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_type_ref_f_type_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_generic_type_ref_f_params langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_generic_type_ref_f_params(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_simple_type_ref_f_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_simple_type_ref_f_type_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of lkt_var_bind_f_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_var_bind_f_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of lkt_var_bind_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int lkt_var_bind_f_expr(
            EntityNative node,
            EntityNative result
        );


    }


    // ==========
    // Exceptions
    // ==========

    /**
     * This class represents exception during symbol manipulation.
     */
    public static final class SymbolException extends RuntimeException {
        public SymbolException(
            final String symbol
        ) {
            super("Invalid symbol : '" + symbol + "'");
        }
    }

    /**
     * This class reprsents exception during enum manipulation.
     */
    public static final class EnumException extends RuntimeException {
        public EnumException(
            final String msg
        ) {
            super(msg);
        }
    }

    /**
     * This class represents an exception in the references manipulation.
     */
    public static final class ReferenceException extends RuntimeException {
        public ReferenceException(
            final String msg
        ) {
            super(msg);
        }
    }

    /**
     * This class wraps the exceptions from the langkit native library.
     */
    public static class LangkitException extends RuntimeException {

        // ----- Instance attributes -----

        /** The kind of the langkit exception. */
        public final ExceptionKind kind;

        // ----- Constructors -----

        /**
         * Create a new langkit exception.
         *
         * @param kind The kind of the exception represented by an integer
         * which will be mapped to an enum value.
         * @param message The message of the exception.
         */
        public LangkitException(
            final int kind,
            final String message
        ) {
            super(message);
            this.kind = ExceptionKind.fromC(kind);
        }

    }

    // ==========
    // Enum definitions
    // ==========

    // ===== Constants enumeration =====

    /**
     * Kind for this token.
     */
    public enum TokenKind {

        // ----- Enum values -----

        NO_TOKEN(-1, "No_Token"),
        LKT_AMP(0, "Amp"),
        LKT_AND_KW(1, "And_Kw"),
        LKT_AT(2, "At"),
        LKT_BIG_NUMBER(3, "Big_Number"),
        LKT_BIND_KW(4, "Bind_Kw"),
        LKT_BLOCK_STRING_LINE(5, "Block_String_Line"),
        LKT_CASE_KW(6, "Case_Kw"),
        LKT_CHAR(7, "Char"),
        LKT_CLASS_KW(8, "Class_Kw"),
        LKT_COLON(9, "Colon"),
        LKT_COMB(10, "Comb"),
        LKT_COMMA(11, "Comma"),
        LKT_COMMENT(12, "Comment"),
        LKT_DISCARD_KW(13, "Discard_Kw"),
        LKT_DIV(14, "Div"),
        LKT_DOC_COMMENT(15, "Doc_Comment"),
        LKT_DOT(16, "Dot"),
        LKT_DYN_VAR_KW(17, "Dyn_Var_Kw"),
        LKT_E_Q(18, "E_Q"),
        LKT_ELIF_KW(19, "Elif_Kw"),
        LKT_ELSE_KW(20, "Else_Kw"),
        LKT_ENUM_KW(21, "Enum_Kw"),
        LKT_EQUAL(22, "Equal"),
        LKT_EXCL_MARK(23, "Excl_Mark"),
        LKT_FAT_RIGHT_ARROW(24, "Fat_Right_Arrow"),
        LKT_FUN_KW(25, "Fun_Kw"),
        LKT_G_T(26, "G_T"),
        LKT_G_T_E(27, "G_T_E"),
        LKT_GENERIC_KW(28, "Generic_Kw"),
        LKT_GRAMMAR_KW(29, "Grammar_Kw"),
        LKT_IDENTIFIER(30, "Identifier"),
        LKT_IF_KW(31, "If_Kw"),
        LKT_IMPLEMENTS_KW(32, "Implements_Kw"),
        LKT_IMPORT_KW(33, "Import_Kw"),
        LKT_IN_KW(34, "In_Kw"),
        LKT_INT_MARK(35, "Int_Mark"),
        LKT_IS_KW(36, "Is_Kw"),
        LKT_L_BRACE(37, "L_Brace"),
        LKT_L_BRACK(38, "L_Brack"),
        LKT_L_PAR(39, "L_Par"),
        LKT_L_T(40, "L_T"),
        LKT_L_T_E(41, "L_T_E"),
        LKT_LEFT_ARROW(42, "Left_Arrow"),
        LKT_LEXER_KW(43, "Lexer_Kw"),
        LKT_LEXING_FAILURE(44, "Lexing_Failure"),
        LKT_MATCH_KW(45, "Match_Kw"),
        LKT_MINUS(46, "Minus"),
        LKT_N_E(47, "N_E"),
        LKT_NOT_KW(48, "Not_Kw"),
        LKT_NULL_KW(49, "Null_Kw"),
        LKT_NUMBER(50, "Number"),
        LKT_OR_KW(51, "Or_Kw"),
        LKT_P_STRING(52, "P_String"),
        LKT_PERCENT(53, "Percent"),
        LKT_PIPE(54, "Pipe"),
        LKT_PLUS(55, "Plus"),
        LKT_PRIVATE_KW(56, "Private_Kw"),
        LKT_PUBLIC_KW(57, "Public_Kw"),
        LKT_R_BRACE(58, "R_Brace"),
        LKT_R_BRACK(59, "R_Brack"),
        LKT_R_PAR(60, "R_Par"),
        LKT_RAISE_KW(61, "Raise_Kw"),
        LKT_RIGHT_ARROW(62, "Right_Arrow"),
        LKT_SEMICOLON(63, "Semicolon"),
        LKT_STRING(64, "String"),
        LKT_STRUCT_KW(65, "Struct_Kw"),
        LKT_TERMINATION(66, "Termination"),
        LKT_THEN_KW(67, "Then_Kw"),
        LKT_TIMES(68, "Times"),
        LKT_TRAIT_KW(69, "Trait_Kw"),
        LKT_TRY_KW(70, "Try_Kw"),
        LKT_VAL_KW(71, "Val_Kw"),
        LKT_WHITESPACE(72, "Whitespace"),
        ;

        // ----- Class attributes -----

        /** Singleton that represents the none token kind. */
        public static final TokenKind NONE = NO_TOKEN;

        /** The map from int to enum values. */
        private static final Map<Integer, TokenKind> map = new HashMap<>();

        // ----- Instance attributes -----

        /** The value of the enum instance. */
        private final int value;

        /** The name of the enum instance in the Langkit DSL. */
        public final String name;

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(TokenKind elem : TokenKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private TokenKind(
            final int value,
            final String name
        ) {
            this.value = value;
            this.name = name;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to a token kind
         * value.
         */
        static TokenKind fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static TokenKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get TokenKind from " + cValue
                );
            return (TokenKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    /**
     * Enumerated type describing all possible exceptions that need to be
     * handled in the C bindings.
     */
    public enum ExceptionKind {

        // ----- Enum values -----

        FILE_READ_ERROR(0),
        BAD_TYPE_ERROR(1),
        OUT_OF_BOUNDS_ERROR(2),
        INVALID_INPUT(3),
        INVALID_SYMBOL_ERROR(4),
        INVALID_UNIT_NAME_ERROR(5),
        NATIVE_EXCEPTION(6),
        PRECONDITION_FAILURE(7),
        PROPERTY_ERROR(8),
        TEMPLATE_ARGS_ERROR(9),
        TEMPLATE_FORMAT_ERROR(10),
        TEMPLATE_INSTANTIATION_ERROR(11),
        STALE_REFERENCE_ERROR(12),
        SYNTAX_ERROR(13),
        UNKNOWN_CHARSET(14),
        MALFORMED_TREE_ERROR(15),
        ;

        // ----- Class attributes -----

        /** Singleton that represents the none expcetion kind. */
        public static final ExceptionKind NONE =
            FILE_READ_ERROR;

        /** The map from int to enum values. */
        private static final Map<Integer, ExceptionKind> map =
            new HashMap<>();

        // ----- Instance ttributes -----

        /** The value of the enum instance. */
        private final int value;

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(ExceptionKind elem : ExceptionKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private ExceptionKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to a exception
         * kind value.
         */
        static ExceptionKind fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static ExceptionKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get ExceptionKind from " + cValue
                );
            return (ExceptionKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    /**
     * Kind of AST nodes in parse trees.
     */
    public enum NodeKind {

        // ----- Enum values -----

            
        // LKT_NODE is abstract
            
        // BASE_LEXER_CASE_RULE_ALT is abstract
            
        /**
         * Alternative of a case rule which sends the token only if the kind of
         * the previous token is among a given set.
         *
         * This node type has no derivation.
         */
        LEXER_CASE_RULE_COND_ALT(1),
            
        /**
         * Default alternative of a case rule which sends the token if all the
         * previous alternatives failed.
         *
         * This node type has no derivation.
         */
        LEXER_CASE_RULE_DEFAULT_ALT(2),
            
        /**
         * A single line in a block string literal.
         *
         * This node type has no derivation.
         */
        BLOCK_STRING_LINE(3),
            
        // CLASS_QUALIFIER is abstract
            
        /**
         * This node type has no derivation.
         */
        CLASS_QUALIFIER_ABSENT(4),
            
        /**
         * This node type has no derivation.
         */
        CLASS_QUALIFIER_PRESENT(5),
            
        // DECL is abstract
            
        // BASE_GRAMMAR_RULE_DECL is abstract
            
        /**
         * Declaration of a grammar rule inside of a grammar.
         *
         * This node type has no derivation.
         */
        GRAMMAR_RULE_DECL(6),
            
        /**
         * This node type has no derivation.
         */
        SYNTHETIC_LEXER_DECL(7),
            
        // BASE_VAL_DECL is abstract
            
        /**
         * Synthetic declaration for the implicit "node" variable available in
         * properties.
         *
         * This node type has no derivation.
         */
        NODE_DECL(8),
            
        /**
         * Synthetic declaration for the implicit "self" variable available in
         * properties.
         *
         * This node type has no derivation.
         */
        SELF_DECL(9),
            
        // USER_VAL_DECL is abstract
            
        /**
         * Enum literal declaration.
         *
         * This node type has no derivation.
         */
        ENUM_LIT_DECL(10),
            
        // EXPLICITLY_TYPED_DECL is abstract
            
        // COMPONENT_DECL is abstract
            
        /**
         * Field declaration.
         *
         * This node type has no derivation.
         */
        FIELD_DECL(11),
            
        /**
         * Function argument declaration.
         *
         * This node type has no derivation.
         */
        FUN_ARG_DECL(12),
            
        /**
         * Function argument declaration.
         *
         * This node type has no derivation.
         */
        LAMBDA_ARG_DECL(13),
            
        /**
         * Dynamic variable declaration.
         *
         * This node type has no derivation.
         */
        DYN_VAR_DECL(14),
            
        /**
         * Value declaration in a match branch.
         *
         * This node type has no derivation.
         */
        MATCH_VAL_DECL(15),
            
        /**
         * Value declaration.
         *
         * This node type has no derivation.
         */
        VAL_DECL(16),
            
        /**
         * Function declaration.
         *
         * This node type has no derivation.
         */
        FUN_DECL(17),
            
        /**
         * Env spec declaration.
         *
         * Each node type can have one or no env spec. Env specs contains only
         * a list of env actions.
         *
         * This node type has no derivation.
         */
        ENV_SPEC_DECL(18),
            
        /**
         * Generic entity declaration.
         *
         * This node type has no derivation.
         */
        GENERIC_DECL(19),
            
        /**
         * Declaration of a language's grammar.
         *
         * This node type has no derivation.
         */
        GRAMMAR_DECL(20),
            
        /**
         * Declaration of a language's lexer.
         *
         * This node type has no derivation.
         */
        LEXER_DECL(21),
            
        /**
         * Declaration of a token family.
         *
         * This node type has no derivation.
         */
        LEXER_FAMILY_DECL(22),
            
        // TYPE_DECL is abstract
            
        /**
         * Alternative for an enum class decl.
         *
         * This node type has no derivation.
         */
        ENUM_CLASS_ALT_DECL(23),
            
        /**
         * Function type.
         *
         * This node type has no derivation.
         */
        FUNCTION_TYPE(24),
            
        /**
         * Declaration of a generic formal type in a generic declaration.
         *
         * This node type has no derivation.
         */
        GENERIC_FORMAL_TYPE_DECL(25),
            
        /**
         * Instantiated generic type.
         *
         * This node type has no derivation.
         */
        INSTANTIATED_GENERIC_TYPE(26),
            
        // NAMED_TYPE_DECL is abstract
            
        // BASIC_CLASS_DECL is abstract
            
        /**
         * Declaration for a LK class. This only cover node classes for the
         * moment, but might be extended to support regular classes in the
         * future.
         *
         * This node type has no derivation.
         */
        CLASS_DECL(27),
            
        /**
         * Declaration for a LK class. This only cover node classes for the
         * moment, but might be extended to support regular classes in the
         * future.
         *
         * This node type has no derivation.
         */
        ENUM_CLASS_DECL(28),
            
        /**
         * Enum type declaration.
         *
         * This node type has no derivation.
         */
        ENUM_TYPE_DECL(29),
            
        /**
         * Declaration for a LK struct.
         *
         * This node type has no derivation.
         */
        STRUCT_DECL(30),
            
        /**
         * Trait declaration. For the moment, a Trait can just be used to group
         * behavior for built-in types. It's not usable as a type-bound since
         * we don't have generics, and you cannot implement one either.
         *
         * The reason they're added is to lay down the basics of what we want
         * the Lkt type system to be.
         *
         * TODO: Traits are *not* types. They're treated as such in the grammar
         * for convenience for now, but it's probably not a good idea. Migrate
         * away from this.
         *
         * This node type has no derivation.
         */
        TRAIT_DECL(31),
            
        /**
         * Compile time annotation attached to a declaration.
         *
         * This node type has no derivation.
         */
        DECL_ANNOTATION(32),
            
        /**
         * List of arguments for an annotation with a call syntax. This
         * intermediate node is necessary in order to determine after parsing
         * whether there is no param list, or if the list is empty.
         *
         * This node type has no derivation.
         */
        DECL_ANNOTATION_PARAMS(33),
            
        /**
         * Elsif branch of an if expression.
         *
         * This node type has no derivation.
         */
        ELSIF_BRANCH(34),
            
        /**
         * Case branch for an enum class declaration.
         *
         * This node type has no derivation.
         */
        ENUM_CLASS_CASE(35),
            
        // EXCLUDES_NULL is abstract
            
        /**
         * This node type has no derivation.
         */
        EXCLUDES_NULL_ABSENT(36),
            
        /**
         * This node type has no derivation.
         */
        EXCLUDES_NULL_PRESENT(37),
            
        // EXPR is abstract
            
        /**
         * "Any of" expression.
         *
         * This node type has no derivation.
         */
        ANY_OF(38),
            
        /**
         * Literal for an array value.
         *
         * This node type has no derivation.
         */
        ARRAY_LITERAL(39),
            
        // BASE_DOT_EXPR is abstract
            
        /**
         * Dotted expression.
         *
         * This node type has no derivation.
         */
        DOT_EXPR(40),
            
        /**
         * Null conditional dotted expression.
         *
         * This node type has no derivation.
         */
        NULL_COND_DOTTED_NAME(41),
            
        /**
         * Binary operator expression.
         *
         * This node type has no derivation.
         */
        BIN_OP(42),
            
        /**
         * Block expression.
         *
         * This node type has no derivation.
         */
        BLOCK_EXPR(43),
            
        /**
         * Call expression.
         *
         * This node type has no derivation.
         */
        CALL_EXPR(44),
            
        /**
         * Cast expression.
         *
         * This node type has no derivation.
         */
        CAST_EXPR(45),
            
        /**
         * Expression that throws an error if LHS is null.
         *
         * This node type has no derivation.
         */
        ERROR_ON_NULL(46),
            
        /**
         * Generic instantiation.
         *
         * This node type has no derivation.
         */
        GENERIC_INSTANTIATION(47),
            
        // GRAMMAR_EXPR is abstract
            
        /**
         * Grammar expression for a cut.
         *
         * This node type has no derivation.
         */
        GRAMMAR_CUT(48),
            
        /**
         * Grammar expression to discard the match.
         *
         * This node type has no derivation.
         */
        GRAMMAR_DISCARD(49),
            
        /**
         * Grammar expression (error recovery) to ensure that any nested skip
         * parser calls won't skip certain parse results.
         *
         * This node type has no derivation.
         */
        GRAMMAR_DONT_SKIP(50),
            
        /**
         * Grammar expression to parse lists of results. Results can be
         * separated by a separator. List can be empty ('*') or not ('+').
         *
         * This node type has no derivation.
         */
        GRAMMAR_LIST(51),
            
        /**
         * Grammar expression to parse a null node.
         *
         * This node type has no derivation.
         */
        GRAMMAR_NULL(52),
            
        /**
         * Grammar expression for an optional parsing result.
         *
         * This node type has no derivation.
         */
        GRAMMAR_OPT(53),
            
        /**
         * Grammar expression for an optional parsing result. Missing result
         * creates an error, but parsing continues.
         *
         * This node type has no derivation.
         */
        GRAMMAR_OPT_ERROR(54),
            
        /**
         * Grammar expression for a group of optional parsing results. Failure
         * to parse an optional result creates an error, but parsing continues.
         *
         * This node type has no derivation.
         */
        GRAMMAR_OPT_ERROR_GROUP(55),
            
        /**
         * Grammar expression for a group of optional parsing results.
         *
         * This node type has no derivation.
         */
        GRAMMAR_OPT_GROUP(56),
            
        /**
         * Grammar ``Or`` expression (disjunctive choice between several
         * grammar options).
         *
         * This node type has no derivation.
         */
        GRAMMAR_OR_EXPR(57),
            
        /**
         * Grammar expression to pick the significant parse out of a list of
         * parses (will automatically discard token results).
         *
         * Derived nodes: ``GrammarImplicitPick``
         */
        GRAMMAR_PICK(58),
            
        /**
         * Implicit pick operation.
         *
         * This node type has no derivation.
         */
        GRAMMAR_IMPLICIT_PICK(59),
            
        /**
         * Grammar expression for a predicate: Only parse something if the
         * predicate (that is a reference to a node property) returns True.
         *
         * This node type has no derivation.
         */
        GRAMMAR_PREDICATE(60),
            
        /**
         * Grammar expression for a reference to another grammar rule.
         *
         * This node type has no derivation.
         */
        GRAMMAR_RULE_REF(61),
            
        /**
         * Grammar expression (error recovery) to skip a parsing result.
         *
         * This node type has no derivation.
         */
        GRAMMAR_SKIP(62),
            
        /**
         * Grammar expression for a StopCut.
         *
         * This node type has no derivation.
         */
        GRAMMAR_STOP_CUT(63),
            
        /**
         * Expression for the parsing of a Node.
         *
         * This node type has no derivation.
         */
        PARSE_NODE_EXPR(64),
            
        /**
         * Grammar expression for a token literal.
         *
         * This node type has no derivation.
         */
        TOKEN_LIT(65),
            
        /**
         * Grammar expression for a case insensitive token literal.
         *
         * This node type has no derivation.
         */
        TOKEN_NO_CASE_LIT(66),
            
        /**
         * Grammar expression for a pattern literal.
         *
         * This node type has no derivation.
         */
        TOKEN_PATTERN_LIT(67),
            
        /**
         * Grammar expression for a token reference.
         *
         * This node type has no derivation.
         */
        TOKEN_REF(68),
            
        /**
         * Identifier.
         *
         * Derived nodes: ``DefId``, ``ModuleRefId``, ``RefId``
         */
        ID(69),
            
        /**
         * Defining identifier.
         *
         * This node type has no derivation.
         */
        DEF_ID(70),
            
        /**
         * Id referencing a langkit module.
         *
         * This node type has no derivation.
         */
        MODULE_REF_ID(71),
            
        /**
         * Reference identifier.
         *
         * This node type has no derivation.
         */
        REF_ID(72),
            
        /**
         * If expression.
         *
         * This node type has no derivation.
         */
        IF_EXPR(73),
            
        /**
         * Isa expression.
         *
         * This node type has no derivation.
         */
        ISA(74),
            
        /**
         * Keep expression.
         *
         * This node type has no derivation.
         */
        KEEP_EXPR(75),
            
        /**
         * Lambda expression.
         *
         * This node type has no derivation.
         */
        LAMBDA_EXPR(76),
            
        // LIT is abstract
            
        /**
         * Big number literal expression.
         *
         * This node type has no derivation.
         */
        BIG_NUM_LIT(77),
            
        /**
         * Character literal expression.
         *
         * This node type has no derivation.
         */
        CHAR_LIT(78),
            
        /**
         * Null literal expression.
         *
         * This node type has no derivation.
         */
        NULL_LIT(79),
            
        /**
         * Number literal expression.
         *
         * This node type has no derivation.
         */
        NUM_LIT(80),
            
        // STRING_LIT is abstract
            
        /**
         * String literal expression, made of multiple line strings.
         *
         * The denoted string value is the concatenation of all line string
         * items. Each line string item must be either:
         *
         * * The empty string designator (``|"``), to denote an empty line
         *   (``\n``).
         *
         * * ``|" <content>``, to designate a non-empty line. The space before
         *   ``<content>`` is mandatory, and is not included in the denoted
         *   string value. ``<content>`` can be anything that appear in a
         *   regular string literal: escape sequences are interpreted the same
         *   way.
         *
         * This node type has no derivation.
         */
        BLOCK_STRING_LIT(81),
            
        /**
         * Single line string literal expression.
         *
         * Note that in order to reduce the size of the node type hierarchy, we
         * define only one node (StringLit) for all our string literals (only
         * regular strings and pattern string literals at the moment). This
         * will also make it easy to add new string prefixes in the future.
         *
         * Derived nodes: ``PatternSingleLineStringLit``
         */
        SINGLE_LINE_STRING_LIT(82),
            
        /**
         * Pattern single line string literal expression.
         *
         * This node type has no derivation.
         */
        PATTERN_SINGLE_LINE_STRING_LIT(83),
            
        /**
         * Class for logic expressions (any ``basic_expr`` starting with %).
         *
         * This node type has no derivation.
         */
        LOGIC_EXPR(84),
            
        /**
         * Binary operator expression.
         *
         * This node type has no derivation.
         */
        MATCH_EXPR(85),
            
        /**
         * Boolean negation expression.
         *
         * This node type has no derivation.
         */
        NOT_EXPR(86),
            
        /**
         * Parenthesized expression.
         *
         * This node type has no derivation.
         */
        PAREN_EXPR(87),
            
        /**
         * Raise expression.
         *
         * This node type has no derivation.
         */
        RAISE_EXPR(88),
            
        /**
         * Array subscript expression.
         *
         * Derived nodes: ``NullCondSubscriptExpr``
         */
        SUBSCRIPT_EXPR(89),
            
        /**
         * Null conditional subscript expression.
         *
         * This node type has no derivation.
         */
        NULL_COND_SUBSCRIPT_EXPR(90),
            
        /**
         * Try expression.
         *
         * This node type has no derivation.
         */
        TRY_EXPR(91),
            
        /**
         * Unary operator expression.
         *
         * This node type has no derivation.
         */
        UN_OP(92),
            
        /**
         * Container for an lkt declaration. Contains the decl node plus the
         * documentation and annotations.
         *
         * This node type has no derivation.
         */
        FULL_DECL(93),
            
        /**
         * Specification for the separator of a list parser.
         *
         * This node type has no derivation.
         */
        GRAMMAR_LIST_SEP(94),
            
        /**
         * Statement to import another source file.
         *
         * This node type has no derivation.
         */
        IMPORT(95),
            
        /**
         * For the moment, root node of a lkt compilation unit.
         *
         * This node type has no derivation.
         */
        LANGKIT_ROOT(96),
            
        /**
         * Lexer construct to introduce a conditional lexing action.
         *
         * This node type has no derivation.
         */
        LEXER_CASE_RULE(97),
            
        /**
         * Lexer construction used by case alternatives to represent the token
         * to send if that alternative was chosen.
         *
         * This node type has no derivation.
         */
        LEXER_CASE_RULE_SEND(98),
            
        // LIST_KIND is abstract
            
        /**
         * This node type has no derivation.
         */
        LIST_KIND_ONE(99),
            
        /**
         * This node type has no derivation.
         */
        LIST_KIND_ZERO(100),
            
        // LKT_NODE_BASE_LIST is abstract
            
        /**
         * List of BaseLexerCaseRuleAlt.
         *
         * This node type has no derivation.
         */
        BASE_LEXER_CASE_RULE_ALT_LIST(101),
            
        /**
         * List of BlockStringLine.
         *
         * This node type has no derivation.
         */
        BLOCK_STRING_LINE_LIST(102),
            
        /**
         * List of CallExpr.
         *
         * This node type has no derivation.
         */
        CALL_EXPR_LIST(103),
            
        /**
         * List of DeclAnnotation.
         *
         * This node type has no derivation.
         */
        DECL_ANNOTATION_LIST(104),
            
        /**
         * List of ElsifBranch.
         *
         * This node type has no derivation.
         */
        ELSIF_BRANCH_LIST(105),
            
        /**
         * List of EnumClassAltDecl.
         *
         * This node type has no derivation.
         */
        ENUM_CLASS_ALT_DECL_LIST(106),
            
        /**
         * List of EnumClassCase.
         *
         * This node type has no derivation.
         */
        ENUM_CLASS_CASE_LIST(107),
            
        /**
         * List of EnumLitDecl.
         *
         * This node type has no derivation.
         */
        ENUM_LIT_DECL_LIST(108),
            
        /**
         * List of Expr.
         *
         * This list node can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BaseDotExpr``, ``BinOp``, ``BlockExpr``,
         * ``CallExpr``, ``CastExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``, ``NotExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``, ``UnOp``
         *
         * Derived nodes: ``AnyOfList``
         */
        EXPR_LIST(109),
            
        /**
         * Pipe-separated list of expressions.
         *
         * This is used to represent the "values" operand of an ``AnyOf``
         * expression.
         *
         * This list node can contain one of the following nodes:
         * ``ArrayLiteral``, ``BaseDotExpr``, ``BlockExpr``, ``CallExpr``,
         * ``CastExpr``, ``ErrorOnNull``, ``GenericInstantiation``, ``IfExpr``,
         * ``KeepExpr``, ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``
         *
         * This node type has no derivation.
         */
        ANY_OF_LIST(110),
            
        /**
         * List of FullDecl.
         *
         * Derived nodes: ``DeclBlock``, ``GenericFormalDeclList``
         */
        FULL_DECL_LIST(111),
            
        /**
         * List of declarations that also introduces a containing lexical
         * scope.
         *
         * This node type has no derivation.
         */
        DECL_BLOCK(112),
            
        /**
         * Comma-separated list of generic formal types.
         *
         * This node type has no derivation.
         */
        GENERIC_FORMAL_DECL_LIST(113),
            
        /**
         * List of FunArgDecl.
         *
         * This node type has no derivation.
         */
        FUN_ARG_DECL_LIST(114),
            
        /**
         * List of GrammarExpr.
         *
         * This node type has no derivation.
         */
        GRAMMAR_EXPR_LIST(115),
            
        /**
         * List of GrammarExpr.list.
         *
         * This node type has no derivation.
         */
        GRAMMAR_EXPR_LIST_LIST(116),
            
        /**
         * List of Import.
         *
         * This node type has no derivation.
         */
        IMPORT_LIST(117),
            
        /**
         * List of LambdaArgDecl.
         *
         * This node type has no derivation.
         */
        LAMBDA_ARG_DECL_LIST(118),
            
        /**
         * List of LktNode.
         *
         * This list node can contain one of the following nodes: ``FullDecl``,
         * ``LexerCaseRule``, ``ValDecl``, ``VarBind``
         *
         * Derived nodes: ``BlockDeclList``
         */
        LKT_NODE_LIST(119),
            
        /**
         * Semicolon-separated list of declarations.
         *
         * This is used to represent declarations in a block expression.
         *
         * This list node can contain one of the following nodes: ``ValDecl``,
         * ``VarBind``
         *
         * This node type has no derivation.
         */
        BLOCK_DECL_LIST(120),
            
        /**
         * List of MatchBranch.
         *
         * This node type has no derivation.
         */
        MATCH_BRANCH_LIST(121),
            
        /**
         * List of Param.
         *
         * This node type has no derivation.
         */
        PARAM_LIST(122),
            
        /**
         * List of RefId.
         *
         * This node type has no derivation.
         */
        REF_ID_LIST(123),
            
        /**
         * List of TypeRef.
         *
         * This list node can contain one of the following nodes:
         * ``FunctionTypeRef``, ``GenericTypeRef``, ``SimpleTypeRef``
         *
         * Derived nodes: ``IsaList``
         */
        TYPE_REF_LIST(124),
            
        /**
         * Pipe-separated list of type references.
         *
         * This is used to represent the accepted types in an ``Isa``
         * expression.
         *
         * This list node can contain one of the following nodes:
         * ``FunctionTypeRef``, ``GenericTypeRef``, ``SimpleTypeRef``
         *
         * This node type has no derivation.
         */
        ISA_LIST(125),
            
        /**
         * Branch inside a match expression.
         *
         * This node type has no derivation.
         */
        MATCH_BRANCH(126),
            
        // OP is abstract
            
        /**
         * This node type has no derivation.
         */
        OP_AMP(127),
            
        /**
         * This node type has no derivation.
         */
        OP_AND(128),
            
        /**
         * This node type has no derivation.
         */
        OP_DIV(129),
            
        /**
         * This node type has no derivation.
         */
        OP_EQ(130),
            
        /**
         * This node type has no derivation.
         */
        OP_GT(131),
            
        /**
         * This node type has no derivation.
         */
        OP_GTE(132),
            
        /**
         * This node type has no derivation.
         */
        OP_LT(133),
            
        /**
         * This node type has no derivation.
         */
        OP_LTE(134),
            
        /**
         * This node type has no derivation.
         */
        OP_MINUS(135),
            
        /**
         * This node type has no derivation.
         */
        OP_MULT(136),
            
        /**
         * This node type has no derivation.
         */
        OP_NE(137),
            
        /**
         * This node type has no derivation.
         */
        OP_OR(138),
            
        /**
         * This node type has no derivation.
         */
        OP_OR_INT(139),
            
        /**
         * This node type has no derivation.
         */
        OP_PLUS(140),
            
        /**
         * Parameter for function calls or for annotations.
         *
         * This node type has no derivation.
         */
        PARAM(141),
            
        // TYPE_REF is abstract
            
        /**
         * "list" type reference in parsers.
         *
         * This node type has no derivation.
         */
        DEFAULT_LIST_TYPE_REF(142),
            
        /**
         * Reference to a function type.
         *
         * This node type has no derivation.
         */
        FUNCTION_TYPE_REF(143),
            
        /**
         * Reference to a generic type.
         *
         * This node type has no derivation.
         */
        GENERIC_TYPE_REF(144),
            
        /**
         * Simple reference to a type.
         *
         * This node type has no derivation.
         */
        SIMPLE_TYPE_REF(145),
            
        /**
         * Dynamic var bind expression.
         *
         * This node type has no derivation.
         */
        VAR_BIND(146),
        ;

        // ----- Class attributes -----

        /** Map containing relation from node kind value and enum instance. */
        private static final Map<Integer, NodeKind> map = new HashMap<>();

        // ----- Instance attributes -----

        /** Integer value of the node kind. */
        public final int value;

        /** Description associated to the node kind. */
        private Reflection.Node description;

        // ----- Constructors -----

        static {
            // Initialize the lookup map
            for(NodeKind elem : NodeKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private NodeKind(
            final int value
        ) {
            this.value = value;

            // The "description" field is intiialized with null to avoid
            // static code execution order issues.
            this.description = null;
        }

        // ----- Class methods -----

        /**
         * Get the enum instance from the given C integer value.
         *
         * @throws EnumException If the given C value is not a valid enum
         *                       value.
         */
        public static NodeKind fromC(
            final int cValue
        ) {
            if(!map.containsKey(cValue))
                throw new EnumException("Cannot get NodeKind from " + cValue);
            return (NodeKind) map.get(cValue);
        }

        // ----- Instance methods -----

        /** Get the C integer value of the enum instance. */
        public int toC() {
            return this.value;
        }

        public Reflection.Node getDescription() {
            return this.description;
        }

        void setDescription(Reflection.Node description) {
            this.description = description;
        }

    }

    /** This enum contains all language nodes' member references */
    public enum MemberReference {

        // ----- Enum values -----

            
        LKT_DECODED_CHAR_VALUE_VALUE(1),
            
        LKT_DECODED_CHAR_VALUE_HAS_ERROR(2),
            
        LKT_DECODED_CHAR_VALUE_ERROR_SLOC(3),
            
        LKT_DECODED_CHAR_VALUE_ERROR_MESSAGE(4),
            
        LKT_DECODED_STRING_VALUE_VALUE(5),
            
        LKT_DECODED_STRING_VALUE_HAS_ERROR(6),
            
        LKT_DECODED_STRING_VALUE_ERROR_SLOC(7),
            
        LKT_DECODED_STRING_VALUE_ERROR_MESSAGE(8),
            
        LKT_SEMANTIC_RESULT_NODE(9),
            
        LKT_SEMANTIC_RESULT_RESULT_TYPE(10),
            
        LKT_SEMANTIC_RESULT_RESULT_DECL(11),
            
        LKT_SEMANTIC_RESULT_HAS_ERROR(12),
            
        LKT_SEMANTIC_RESULT_ERROR_MESSAGE(13),
            
        LKT_SEMANTIC_RESULT_EXEMPT_ANALYSIS(14),
            
        LKT_TREE_SEMANTIC_RESULT_RESULTS(15),
            
        LKT_TREE_SEMANTIC_RESULT_HAS_ERROR(16),
            
        LKT_BASE_LEXER_CASE_RULE_ALT_F_SEND(17),
            
        LKT_LEXER_CASE_RULE_COND_ALT_F_COND_EXPRS(18),
            
        LKT_DECL_F_SYN_NAME(19),
            
        LKT_BASE_GRAMMAR_RULE_DECL_F_EXPR(20),
            
        LKT_EXPLICITLY_TYPED_DECL_F_DECL_TYPE(21),
            
        LKT_COMPONENT_DECL_F_DEFAULT_VAL(22),
            
        LKT_FUN_ARG_DECL_F_DECL_ANNOTATIONS(23),
            
        LKT_VAL_DECL_F_VAL(24),
            
        LKT_FUN_DECL_F_ARGS(25),
            
        LKT_FUN_DECL_F_RETURN_TYPE(26),
            
        LKT_FUN_DECL_F_BODY(27),
            
        LKT_ENV_SPEC_DECL_F_ACTIONS(28),
            
        LKT_GENERIC_DECL_F_GENERIC_FORMAL_DECLS(29),
            
        LKT_GENERIC_DECL_F_DECL(30),
            
        LKT_GRAMMAR_DECL_F_RULES(31),
            
        LKT_LEXER_DECL_F_RULES(32),
            
        LKT_LEXER_FAMILY_DECL_F_RULES(33),
            
        LKT_TYPE_DECL_F_TRAITS(34),
            
        LKT_TYPE_DECL_F_SYN_BASE_TYPE(35),
            
        LKT_GENERIC_FORMAL_TYPE_DECL_F_HAS_CLASS(36),
            
        LKT_NAMED_TYPE_DECL_F_DECLS(37),
            
        LKT_ENUM_CLASS_DECL_F_BRANCHES(38),
            
        LKT_ENUM_TYPE_DECL_F_LITERALS(39),
            
        LKT_DECL_ANNOTATION_F_NAME(40),
            
        LKT_DECL_ANNOTATION_F_PARAMS(41),
            
        LKT_DECL_ANNOTATION_PARAMS_F_PARAMS(42),
            
        LKT_ELSIF_BRANCH_F_COND_EXPR(43),
            
        LKT_ELSIF_BRANCH_F_THEN_EXPR(44),
            
        LKT_ENUM_CLASS_CASE_F_DECLS(45),
            
        LKT_ANY_OF_F_EXPR(46),
            
        LKT_ANY_OF_F_VALUES(47),
            
        LKT_ARRAY_LITERAL_F_EXPRS(48),
            
        LKT_ARRAY_LITERAL_F_ELEMENT_TYPE(49),
            
        LKT_BASE_DOT_EXPR_F_PREFIX(50),
            
        LKT_BASE_DOT_EXPR_F_SUFFIX(51),
            
        LKT_BIN_OP_F_LEFT(52),
            
        LKT_BIN_OP_F_OP(53),
            
        LKT_BIN_OP_F_RIGHT(54),
            
        LKT_BLOCK_EXPR_F_VAL_DEFS(55),
            
        LKT_BLOCK_EXPR_F_EXPR(56),
            
        LKT_CALL_EXPR_F_NAME(57),
            
        LKT_CALL_EXPR_F_ARGS(58),
            
        LKT_CAST_EXPR_F_EXPR(59),
            
        LKT_CAST_EXPR_F_EXCLUDES_NULL(60),
            
        LKT_CAST_EXPR_F_DEST_TYPE(61),
            
        LKT_ERROR_ON_NULL_F_EXPR(62),
            
        LKT_GENERIC_INSTANTIATION_F_NAME(63),
            
        LKT_GENERIC_INSTANTIATION_F_ARGS(64),
            
        LKT_GRAMMAR_DISCARD_F_EXPR(65),
            
        LKT_GRAMMAR_DONT_SKIP_F_EXPR(66),
            
        LKT_GRAMMAR_DONT_SKIP_F_DONT_SKIP(67),
            
        LKT_GRAMMAR_LIST_F_LIST_TYPE(68),
            
        LKT_GRAMMAR_LIST_F_KIND(69),
            
        LKT_GRAMMAR_LIST_F_EXPR(70),
            
        LKT_GRAMMAR_LIST_F_SEP(71),
            
        LKT_GRAMMAR_NULL_F_NAME(72),
            
        LKT_GRAMMAR_OPT_F_EXPR(73),
            
        LKT_GRAMMAR_OPT_ERROR_F_EXPR(74),
            
        LKT_GRAMMAR_OPT_ERROR_GROUP_F_EXPR(75),
            
        LKT_GRAMMAR_OPT_GROUP_F_EXPR(76),
            
        LKT_GRAMMAR_OR_EXPR_F_SUB_EXPRS(77),
            
        LKT_GRAMMAR_PICK_F_EXPRS(78),
            
        LKT_GRAMMAR_PREDICATE_F_EXPR(79),
            
        LKT_GRAMMAR_PREDICATE_F_PROP_REF(80),
            
        LKT_GRAMMAR_RULE_REF_F_NODE_NAME(81),
            
        LKT_GRAMMAR_SKIP_F_NAME(82),
            
        LKT_GRAMMAR_STOP_CUT_F_EXPR(83),
            
        LKT_PARSE_NODE_EXPR_F_NODE_NAME(84),
            
        LKT_PARSE_NODE_EXPR_F_SUB_EXPRS(85),
            
        LKT_TOKEN_NO_CASE_LIT_F_LIT(86),
            
        LKT_TOKEN_REF_F_TOKEN_NAME(87),
            
        LKT_TOKEN_REF_F_EXPR(88),
            
        LKT_IF_EXPR_F_COND_EXPR(89),
            
        LKT_IF_EXPR_F_THEN_EXPR(90),
            
        LKT_IF_EXPR_F_ALTERNATIVES(91),
            
        LKT_IF_EXPR_F_ELSE_EXPR(92),
            
        LKT_ISA_F_EXPR(93),
            
        LKT_ISA_F_DEST_TYPE(94),
            
        LKT_KEEP_EXPR_F_EXPR(95),
            
        LKT_KEEP_EXPR_F_KEEP_TYPE(96),
            
        LKT_LAMBDA_EXPR_F_PARAMS(97),
            
        LKT_LAMBDA_EXPR_F_RETURN_TYPE(98),
            
        LKT_LAMBDA_EXPR_F_BODY(99),
            
        LKT_NULL_LIT_F_DEST_TYPE(100),
            
        LKT_BLOCK_STRING_LIT_F_LINES(101),
            
        LKT_LOGIC_EXPR_F_EXPR(102),
            
        LKT_MATCH_EXPR_F_MATCH_EXPR(103),
            
        LKT_MATCH_EXPR_F_BRANCHES(104),
            
        LKT_NOT_EXPR_F_EXPR(105),
            
        LKT_PAREN_EXPR_F_EXPR(106),
            
        LKT_RAISE_EXPR_F_DEST_TYPE(107),
            
        LKT_RAISE_EXPR_F_EXCEPT_EXPR(108),
            
        LKT_SUBSCRIPT_EXPR_F_PREFIX(109),
            
        LKT_SUBSCRIPT_EXPR_F_INDEX(110),
            
        LKT_TRY_EXPR_F_TRY_EXPR(111),
            
        LKT_TRY_EXPR_F_OR_EXPR(112),
            
        LKT_UN_OP_F_OP(113),
            
        LKT_UN_OP_F_EXPR(114),
            
        LKT_FULL_DECL_F_DOC(115),
            
        LKT_FULL_DECL_F_DECL_ANNOTATIONS(116),
            
        LKT_FULL_DECL_F_DECL(117),
            
        LKT_GRAMMAR_LIST_SEP_F_TOKEN(118),
            
        LKT_GRAMMAR_LIST_SEP_F_EXTRA(119),
            
        LKT_IMPORT_F_NAME(120),
            
        LKT_LANGKIT_ROOT_F_IMPORTS(121),
            
        LKT_LANGKIT_ROOT_F_DECLS(122),
            
        LKT_LEXER_CASE_RULE_F_EXPR(123),
            
        LKT_LEXER_CASE_RULE_F_ALTS(124),
            
        LKT_LEXER_CASE_RULE_SEND_F_SENT(125),
            
        LKT_LEXER_CASE_RULE_SEND_F_MATCH_SIZE(126),
            
        LKT_MATCH_BRANCH_F_DECL(127),
            
        LKT_MATCH_BRANCH_F_EXPR(128),
            
        LKT_PARAM_F_NAME(129),
            
        LKT_PARAM_F_VALUE(130),
            
        LKT_FUNCTION_TYPE_REF_F_ARGS_TYPES(131),
            
        LKT_FUNCTION_TYPE_REF_F_RETURN_TYPE(132),
            
        LKT_GENERIC_TYPE_REF_F_TYPE_NAME(133),
            
        LKT_GENERIC_TYPE_REF_F_PARAMS(134),
            
        LKT_SIMPLE_TYPE_REF_F_TYPE_NAME(135),
            
        LKT_VAR_BIND_F_NAME(136),
            
        LKT_VAR_BIND_F_EXPR(137),
            
        LKT_LKT_NODE_P_NODE_GEN_TRAIT(138),
            
        LKT_LKT_NODE_P_NODE_TRAIT(139),
            
        LKT_LKT_NODE_P_TOKEN_NODE_TRAIT(140),
            
        LKT_LKT_NODE_P_ERROR_NODE_TRAIT(141),
            
        LKT_LKT_NODE_P_CHAR_TYPE(142),
            
        LKT_LKT_NODE_P_INT_TYPE(143),
            
        LKT_LKT_NODE_P_BOOL_TYPE(144),
            
        LKT_LKT_NODE_P_BIGINT_TYPE(145),
            
        LKT_LKT_NODE_P_STRING_TYPE(146),
            
        LKT_LKT_NODE_P_SYMBOL_TYPE(147),
            
        LKT_LKT_NODE_P_PROPERTY_ERROR_TYPE(148),
            
        LKT_LKT_NODE_P_REGEXP_TYPE(149),
            
        LKT_LKT_NODE_P_ARRAY_GEN_TYPE(150),
            
        LKT_LKT_NODE_P_ARRAY_TYPE(151),
            
        LKT_LKT_NODE_P_ASTLIST_GEN_TYPE(152),
            
        LKT_LKT_NODE_P_ASTLIST_TYPE(153),
            
        LKT_LKT_NODE_P_ITERATOR_GEN_TRAIT(154),
            
        LKT_LKT_NODE_P_ITERATOR_TRAIT(155),
            
        LKT_LKT_NODE_P_ANALYSIS_UNIT_GEN_TRAIT(156),
            
        LKT_LKT_NODE_P_ANALYSIS_UNIT_TRAIT(157),
            
        LKT_LKT_NODE_P_TOPMOST_INVALID_DECL(158),
            
        LKT_PARENT(159),
            
        LKT_PARENTS(160),
            
        LKT_CHILDREN(161),
            
        LKT_TOKEN_START(162),
            
        LKT_TOKEN_END(163),
            
        LKT_CHILD_INDEX(164),
            
        LKT_PREVIOUS_SIBLING(165),
            
        LKT_NEXT_SIBLING(166),
            
        LKT_UNIT(167),
            
        LKT_IS_GHOST(168),
            
        LKT_FULL_SLOC_IMAGE(169),
            
        LKT_CLASS_QUALIFIER_P_AS_BOOL(170),
            
        LKT_DECL_P_NAME(171),
            
        LKT_DECL_P_FULL_NAME(172),
            
        LKT_DECL_P_DECL_TYPE_NAME(173),
            
        LKT_DECL_P_AS_BARE_DECL(174),
            
        LKT_BASE_VAL_DECL_P_GET_TYPE(175),
            
        LKT_GRAMMAR_DECL_P_LEXER(176),
            
        LKT_TYPE_DECL_P_IS_CLASS(177),
            
        LKT_TYPE_DECL_P_IMPLEMENTED_TRAITS(178),
            
        LKT_TYPE_DECL_P_BASE_TYPE(179),
            
        LKT_TYPE_DECL_P_BASE_TYPES(180),
            
        LKT_TYPE_DECL_P_ROOT_TYPE(181),
            
        LKT_TYPE_DECL_P_IS_SUBTYPE(182),
            
        LKT_TYPE_DECL_P_IS_GENERIC(183),
            
        LKT_INSTANTIATED_GENERIC_TYPE_P_GET_INNER_TYPE(184),
            
        LKT_INSTANTIATED_GENERIC_TYPE_P_GET_ACTUALS(185),
            
        LKT_INSTANTIATED_GENERIC_TYPE_P_GET_INSTANTIATED_TYPE(186),
            
        LKT_EXCLUDES_NULL_P_AS_BOOL(187),
            
        LKT_EXPR_P_IN_TYPE_REF(188),
            
        LKT_EXPR_P_IS_REGULAR_EXPR(189),
            
        LKT_EXPR_P_EXPR_TYPE(190),
            
        LKT_EXPR_P_CHECK_EXPR_TYPE(191),
            
        LKT_EXPR_P_EXPR_CONTEXT_FREE_TYPE(192),
            
        LKT_EXPR_P_REFERENCED_DECL(193),
            
        LKT_EXPR_P_CHECK_REFERENCED_DECL(194),
            
        LKT_CALL_EXPR_P_CALLED_OBJECT_TYPE(195),
            
        LKT_CALL_EXPR_P_CALLED_DECL(196),
            
        LKT_GENERIC_INSTANTIATION_P_INSTANTIATED_DECL(197),
            
        LKT_TOKEN_LIT_P_DENOTED_VALUE(198),
            
        LKT_TOKEN_PATTERN_LIT_P_DENOTED_VALUE(199),
            
        LKT_CHAR_LIT_P_DENOTED_VALUE(200),
            
        LKT_STRING_LIT_P_DENOTED_VALUE(201),
            
        LKT_STRING_LIT_P_IS_PREFIXED_STRING(202),
            
        LKT_STRING_LIT_P_PREFIX(203),
            
        LKT_STRING_LIT_P_IS_REGEXP_LITERAL(204),
            
        LKT_FULL_DECL_P_HAS_ANNOTATION(205),
            
        LKT_FULL_DECL_P_CHECK_DECL(206),
            
        LKT_IMPORT_P_REFERENCED_UNIT(207),
            
        LKT_LANGKIT_ROOT_P_CHECK_SEMANTIC(208),
            
        LKT_TYPE_REF_P_DESIGNATED_TYPE(209),
        ;

        // ----- Class attributes -----

        /** Map containing relation from native value to Jave instance. */
        private static Map<Integer, MemberReference> map = new HashMap<>();

        // ----- Instance attribtues -----

        /** Native value if the member reference. */
        private final int value;

        // ----- Constructors -----

        /** Private constructor. */
        private MemberReference (
            final int value
        ) {
            this.value = value;
        }

        static {
            // Initialize the lookup map
            for(MemberReference elem : MemberReference.values()) {
                map.put(elem.value, elem);
            }
        }

        // ----- Class methods -----

        /**
         * Get the enum instance from the given C integer value.
         *
         * @throws EnumException If the given C value is not a valid enum
         *                       value.
         */
        public static MemberReference fromC(
            final int cValue
        ) {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get MemberReference from " + cValue
                );
            return (MemberReference) map.get(cValue);
        }

        // ----- Instance methods -----

        /** Get the native value of the enum instance. */
        public int toC() {
            return this.value;
        }

    }

    // ===== Generated enums =====

    
    

    /**
     * Specify a kind of analysis unit. Specification units provide an
     * interface to the outer world while body units provide an implementation
     * for the corresponding interface.
     */
    public enum AnalysisUnitKind {

        // ----- Enum values -----

        UNIT_SPECIFICATION(0),
        UNIT_BODY(1),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final AnalysisUnitKind NONE =
            UNIT_SPECIFICATION;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, AnalysisUnitKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(AnalysisUnitKind elem : AnalysisUnitKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private AnalysisUnitKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to an enum value.
         */
        static AnalysisUnitKind fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static AnalysisUnitKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get AnalysisUnitKind from " + cValue
                );
            return (AnalysisUnitKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**

     */
    public enum LookupKind {

        // ----- Enum values -----

        RECURSIVE(0),
        FLAT(1),
        MINIMAL(2),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final LookupKind NONE =
            RECURSIVE;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, LookupKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(LookupKind elem : LookupKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private LookupKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to an enum value.
         */
        static LookupKind fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static LookupKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get LookupKind from " + cValue
                );
            return (LookupKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**
     * Discriminant for DesignatedEnv structures.
     */
    public enum DesignatedEnvKind {

        // ----- Enum values -----

        NONE_ENUM(0),
        CURRENT_ENV(1),
        NAMED_ENV(2),
        DIRECT_ENV(3),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final DesignatedEnvKind NONE =
            NONE_ENUM;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, DesignatedEnvKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(DesignatedEnvKind elem : DesignatedEnvKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private DesignatedEnvKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to an enum value.
         */
        static DesignatedEnvKind fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static DesignatedEnvKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get DesignatedEnvKind from " + cValue
                );
            return (DesignatedEnvKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**
     * Gramar rule to use for parsing.
     */
    public enum GrammarRule {

        // ----- Enum values -----

        MAIN_RULE_RULE(0),
        ID_RULE(1),
        REF_ID_RULE(2),
        TYPE_REF_ID_RULE(3),
        DEF_ID_RULE(4),
        DOC_RULE(5),
        IMPORT_STMT_RULE(6),
        IMPORTS_RULE(7),
        LEXER_DECL_RULE(8),
        GRAMMAR_DECL_RULE(9),
        GRAMMAR_RULE_RULE(10),
        LEXER_RULE_RULE(11),
        LEXER_FAMILY_DECL_RULE(12),
        LEXER_CASE_RULE_RULE(13),
        LEXER_CASE_ALT_RULE(14),
        LEXER_CASE_SEND_RULE(15),
        GRAMMAR_PRIMARY_RULE(16),
        GRAMMAR_EXPR_RULE(17),
        GRAMMAR_PICK_RULE(18),
        GRAMMAR_IMPLICIT_PICK_RULE(19),
        GRAMMAR_OPT_RULE(20),
        GRAMMAR_OPT_ERROR_RULE(21),
        GRAMMAR_CUT_RULE(22),
        GRAMMAR_STOPCUT_RULE(23),
        GRAMMAR_OR_EXPR_RULE(24),
        GRAMMAR_DISCARD_EXPR_RULE(25),
        TOKEN_LITERAL_RULE(26),
        TOKEN_NO_CASE_LITERAL_RULE(27),
        TOKEN_PATTERN_LITERAL_RULE(28),
        PARSE_NODE_EXPR_RULE(29),
        GRAMMAR_RULE_REF_RULE(30),
        GRAMMAR_LIST_EXPR_RULE(31),
        GRAMMAR_LIST_SEP_RULE(32),
        GRAMMAR_SKIP_RULE(33),
        GRAMMAR_NULL_RULE(34),
        GRAMMAR_TOKEN_RULE(35),
        TYPE_DECL_RULE(36),
        GENERIC_DECL_RULE(37),
        GENERIC_FORMAL_TYPE_RULE(38),
        ENUM_LIT_DECL_RULE(39),
        FUN_DECL_RULE(40),
        LAMBDA_ARG_DECL_RULE(41),
        FUN_ARG_DECL_RULE(42),
        FUN_ARG_LIST_RULE(43),
        LAMBDA_ARG_LIST_RULE(44),
        FIELD_DECL_RULE(45),
        BARE_DECL_RULE(46),
        DECL_RULE(47),
        TYPE_EXPR_RULE(48),
        TYPE_REF_RULE(49),
        TYPE_LIST_RULE(50),
        DECLS_RULE(51),
        DECL_BLOCK_RULE(52),
        VAL_DECL_RULE(53),
        DYNVAR_DECL_RULE(54),
        VAR_BIND_RULE(55),
        ENV_SPEC_ACTION_RULE(56),
        ENV_SPEC_DECL_RULE(57),
        BLOCK_RULE(58),
        EXPR_RULE(59),
        REL_RULE(60),
        EQ_RULE(61),
        ARITH_1_RULE(62),
        ARITH_2_RULE(63),
        ARITH_3_RULE(64),
        ISA_OR_PRIMARY_RULE(65),
        PRIMARY_RULE(66),
        MATCH_EXPR_RULE(67),
        NUM_LIT_RULE(68),
        BIG_NUM_LIT_RULE(69),
        STRING_LIT_RULE(70),
        BLOCK_STRING_LIT_RULE(71),
        CHAR_LIT_RULE(72),
        IF_EXPR_RULE(73),
        RAISE_EXPR_RULE(74),
        TRY_EXPR_RULE(75),
        ARRAY_LITERAL_RULE(76),
        BASIC_EXPR_RULE(77),
        TERM_RULE(78),
        BASIC_NAME_RULE(79),
        LAMBDA_EXPR_RULE(80),
        NULL_LIT_RULE(81),
        PARAM_RULE(82),
        PARAMS_RULE(83),
        DECL_ANNOTATION_PARAMS_RULE(84),
        DECL_ANNOTATION_RULE(85),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final GrammarRule NONE =
            MAIN_RULE_RULE;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, GrammarRule> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(GrammarRule elem : GrammarRule.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private GrammarRule(
            final int value
        ) {
            this.value = value;
        }

        // ----- Graal C API methods -----

        /**
         * Internal method to wrap a Native Image C pointer to an enum value.
         */
        static GrammarRule fromC(
            final CIntPointer pointer
        ) throws EnumException {
            return fromC(pointer.read());
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static GrammarRule fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get GrammarRule from " + cValue
                );
            return (GrammarRule) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }


    // ==========
    // Java wrapping classes
    // ==========

    // ===== Constant structure wrapping classes =====

    /**
     * This class provides static methods to help wrapping and unwrapping
     * native boolean values.
     */
    public static final class BooleanWrapper {

        // ----- Graal C API methods -----

        static boolean wrap(
            final CCharPointer pointer
        ) {
            return pointer.read() != 0;
        }

        static boolean wrap(
            final byte nativeValue
        ) {
            return nativeValue != 0;
        }

    }

    /**
     * This class provides static methods to help wrapping and unwrapping
     * native integer values.
     */
    public static final class IntegerWrapper {

        // ----- Graal C API methods -----

        static int wrap(
            final CIntPointer pointer
        ) {
            return pointer.read();
        }

        static int wrap(
            final int nativeValue
        ) {
            return nativeValue;
        }

    }

    /**
     * This class wraps the langkit characters which are 32 bit wide.
     */
    public static final class Char {

        // ----- Class attributes -----

        /** Singleton that represents the none char. */
        public static final Char NONE = new Char(0);

        // ----- Instance attributes -----

        /** The value of the character. */
        public final int value;

        // ----- Constructors -----

        /**
         * Create a new character from its value. In langkit characters are
         * 32 bit wide so represented by Java integer.
         *
         * @param value The value of the character.
         */
        Char(
            final int value
        ) {
            this.value = value;
        }

        /**
         * Create a character from its integer value.
         *
         * @param value The character value.
         * @return The newly created character.
         */
        public static Char create(
            final int value
        ) {
            return new Char(value);
        }

        /**
         * Create a character from a Java character.
         *
         * @param value The source value of the character.
         * @return The newly created character.
         */
        public static Char create(
            final char value
        ) {
            return new Char((int) value);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given NI pointer in a Java class.
         *
         * @param pointer The NI pointer to wrap.
         * @return The wrapped character.
         */
        static Char wrap(
            final CIntPointer pointer
        ) {
            return wrap(pointer.read());
        }

        /**
         * Wrap an integer to a character.
         *
         * @param value The value of the character in an integer.
         * @return The newly created character.
         */
        static Char wrap(
            final int value
        ) {
            return new Char(value);
        }

        /**
         * Unwrap the character in the given int pointer.
         *
         * @param pointer The pointer to unwrap the character in.
         */
        void unwrap(
            final CIntPointer pointer
        ) {
            pointer.write(this.value);
        }

        /**
         * Unwrap the character in a Java integer.
         *
         * @return The character value in a Java integer.
         */
        int unwrap() {
            return this.value;
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            final ByteBuffer buffer = ByteBuffer.allocate(4);
            buffer.order(BYTE_ORDER);
            buffer.putInt(this.value);
            return decodeUTF32(buffer.array());
        }

        @Override
        public boolean equals(Object o) {
            if(o == this) return true;
            if(!(o instanceof Char)) return false;
            Char other = (Char) o;
            return this.value == other.value;
        }

    }

    /**
     * Arbitrarily large integer.
     */
    static final class BigIntegerWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none big integer. */
        public static final BigInteger NONE = BigInteger.ZERO;

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer which points to a native big integer.
         *
         * @param pointer The pointer to the native big integer.
         * @return The Java big integer.
         */
        static BigInteger wrap(
            final WordPointer pointer
        ) {
            return wrap((BigIntegerNative) pointer.read());
        }

        /**
         * Wrap the given native big integer in a Java big integer.
         *
         * @param bigIntegerNative The big integer native value.
         * @return The Java big integer.
         */
        static BigInteger wrap(
            final BigIntegerNative bigIntegerNative
        ) {
            final String representation = getRepresentation(bigIntegerNative);
            return new BigInteger(representation);
        }

        /**
         * Unwrap the given Java big integer in the given word pointer.
         *
         * @param bigInteger The big integer to unwrap.
         * @param pointer The word pointer to place the big integer in.
         */
        static void unwrap(
            final BigInteger bigInteger,
            final WordPointer pointer
        ) {
            pointer.write(unwrap(bigInteger));
        }

        /**
         * Unwrap the given big integer.
         *
         * @param bigInteger The big integer to unwrap.
         * @return The native big integer newly allocated.
         */
        static BigIntegerNative unwrap(
            final BigInteger bigInteger
        ) {
            // Create the representation of the big integer
            final String representation = bigInteger.toString();
            try(final Text bigIntegerText = Text.create(representation)) {
                TextNative bigIntegerTextNative = StackValue.get(
                    TextNative.class
                );
                bigIntegerText.unwrap(bigIntegerTextNative);

                // Create the big intger by calling the native function
                return NI_LIB.lkt_create_big_integer(
                    bigIntegerTextNative
                );
            }
        }

        /**
         * Release the big integer pointed by the given pointer.
         *
         * @param pointer The pointer to the big integer to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((BigIntegerNative) pointer.read());
        }

        /**
         * Release the given native big integer.
         *
         * @param bigIntegerNative The native big integer to release.
         */
        static void release(
            final BigIntegerNative bigIntegerNative
        ) {
            NI_LIB.lkt_big_integer_decref(bigIntegerNative);
        }

        // ----- Class methods -----

        /**
         * Get the string representation of the given native big integer.
         *
         * @param bigIntegerNative The native big integer to get the
         * representation from.
         */
        private static String getRepresentation(
            final BigIntegerNative bigIntegerNative
        ) {
            // Allocate the stack value for the text
            final TextNative bigIntegerTextNative = StackValue.get(
                TextNative.class
            );
            Text.NONE.unwrap(bigIntegerTextNative);

            // Call the native function
            NI_LIB.lkt_big_integer_text(
                bigIntegerNative,
                bigIntegerTextNative
            );

            // Wrap the text and return the result
            try(final Text bigIntegerText = Text.wrap(bigIntegerTextNative)) {
                return bigIntegerText.getContent();
            }
        }

    }

    /**
     * Reference to a symbol. Symbols are owned by analysis contexts, so they
     * must not outlive them. This type exists only in the C API, and roughly
     * wraps the corresponding Ada type (an array fat pointer).
     */
    public static final class Symbol {

        // ----- Class attributes -----

        /** Singleton that represents the none symbol. */
        public static final Symbol NONE = new Symbol("");

        // ----- Instance attributes

        /** The text of the symbol. */
        public final String text;

        // ----- Constructors -----

        /**
         * Create a new symbol from its text.
         *
         * @param text The symbol text.
         */
        Symbol(
            final String text
        ) {
            this.text = text;
        }

        /**
         * Public access to the symbol creation.
         *
         * @param text The text of the symbol.
         */
        public static Symbol create(
            final String text
        ) {
            return new Symbol(text);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to a native symbol.
         *
         * @param pointer The pointer to the native symbol.
         * @return The wrapped symbol.
         */
        static Symbol wrap(
            final WordPointer pointer
        ) {
            return wrap((SymbolNative) pointer.read());
        }

        /**
         * Wrap the given symbol native value in a Java value.
         *
         * @param symbolNative The symbol native value.
         * @return The wrapped symbol.
         */
        static Symbol wrap(
            final SymbolNative symbolNative
        ) {
            // Get the symbol text
            final TextNative textNative = StackValue.get(TextNative.class);
            Text.NONE.unwrap(textNative);
            NI_LIB.lkt_symbol_text(
                symbolNative,
                textNative
            );

            // Return the new symbol
            try(final Text text = Text.wrap(textNative)) {
                return new Symbol(text.getContent());
            }
        }

        /**
         * Unwrap the symbol in the given native structure.
         *
         * @param symbolNative The native structure to unwrap in.
         */
        void unwrap(
            final SymbolNative symbolNative,
            final AnalysisContext context
        ) {
            // Unwrap the symbol text
            try(Text text = Text.create(this.text)) {
                final TextNative textNative = StackValue.get(
                    TextNative.class
                );
                text.unwrap(textNative);

                // Call the symbol creation
                final int resCode = NI_LIB.lkt_context_symbol(
                    context.reference.ni(),
                    textNative,
                    symbolNative
                );

                // Check the symbol creation success
                if(resCode == 0) {
                    throw new SymbolException(this.text);
                }
            }
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.text;
        }

    }

    /**
     * Type to contain Unicode text data.
     */
    static final class StringWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none string. */
        public static final String NONE = "";

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer that points to a native string.
         *
         * @param pointer The pointer to the native string.
         * @return The Java string wrapped from the native string.
         */
        static String wrap(
            final WordPointer pointer
        ) {
            return wrap((StringNative) pointer.read());
        }

        /**
         * Wrap a native string wrapper in a Java string.
         *
         * @param stringNative The native string to wrap.
         * @return The Java string created from the native one.
         */
        static String wrap(
            final StringNative stringNative
        ) {
            return getNativeStringContent(stringNative);
        }

        /**
         * Unwrap the string in the given word pointer.
         *
         * @param string The string to unwrap.
         * @param pointer The word pointer to place the native string in.
         */
        static void unwrap(
            final String string,
            final WordPointer pointer
        ) {
            pointer.write(unwrap(string));
        }

        /**
         * Unwrap the given string in a native one.
         *
         * @param string The string to unwrap.
         * @return The native string unwrapped.
         */
        static StringNative unwrap(
            final String string
        ) {
            return nativeStringFromContent(string);
        }

        /**
         * Release the native string at the given pointer.
         *
         * @param pointer The pointer to the native string.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((StringNative) pointer.read());
        }

        /**
         * Release the given native string.
         *
         * @param stringNative The native string to release.
         */
        static void release(
            final StringNative stringNative
        ) {
            NI_LIB.lkt_string_dec_ref(stringNative);
        }

        // ----- Class methods -----

        /**
         * Get the content of the given native string in a Java one.
         *
         * @param stringNative The native string to get the content from.
         * @return The Java string.
         */
        private static String getNativeStringContent(
            final StringNative stringNative
        ) {
            // Prepare the working variable
            final Pointer pointer = (Pointer) stringNative;
            final int length = pointer.readInt(0);
            final byte[] contentArray = new byte[length * 4];

            // Get the content from the native string
            for(int i = 0 ; i < contentArray.length ; i++) {
                contentArray[i] = pointer.readByte(i + 8);
            }

            // Return the decoded string
            return decodeUTF32(contentArray);
        }

        /**
         * Create a native string from a Java string.
         *
         * @param string The Java string to create the native string from.
         * @return The native string.
         */
        private static StringNative nativeStringFromContent(
            final String string
        ) {
            // Encode the string in UTF-32
            final byte[] contentArray = encodeUTF32(string);
            final int length = string.length();

            // Create the native array
            final Pointer contentArrayNative = UnmanagedMemory.malloc(
                contentArray.length
            );
            for(int i = 0 ; i < contentArray.length ; i++) {
                contentArrayNative.writeByte(i, contentArray[i]);
            }

            // Create the native string
            final StringNative res = NI_LIB.lkt_create_string(
                (CIntPointer) contentArrayNative,
                length
            );

            // Free the memory
            UnmanagedMemory.free(contentArrayNative);

            // Return the result
            return res;
        }

    }

    /**
     * String encoded in UTF-32 (native endianness).
     */
    public static final class Text implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none text. */
        public static final Text NONE = new Text(
            PointerWrapper.nullPointer(),
            0,
            false
        );

        // ----- Instance attributes -----

        /** The pointer to the characters. */
        private final PointerWrapper charPointer;

        /** The size of the text. */
        private final long length;

        /** If the text is allocated. */
        private final boolean isAllocated;

        /** If the text object is the owner of its buffer. */
        private final boolean isOwner;

        /** The content of the text in a Java string. */
        private String content;

        // ----- Constructors -----

        /**
         * Create a new text from its content.
         *
         * @param charPointer The pointer to the characters of the text.
         * @param length The length of the text in character.
         * @param isAllocated If the text is allocated in the memory.
         */
        Text(
            final PointerWrapper charPointer,
            final long length,
            final boolean isAllocated
        ) {
            this(
                charPointer,
                length,
                isAllocated,
                false,
                null
            );
        }

        /**
         * Create a new text from its content and buffer.
         *
         * @param charPointer The pointer to the characters of the text.
         * @param length The length of the text in character.
         * @param isAllocated If the text is allocated in the memory.
         * @param contentArray The characters of the text (for JNI)
         */
        Text(
            final PointerWrapper charPointer,
            final long length,
            final boolean isAllocated,
            final byte[] contentArray
        ) {
            this(
                charPointer,
                length,
                isAllocated,
                false,
                contentArray
            );
        }

        /**
         * Create a new text from its content and buffer.
         *
         * @param charPointer The pointer to the characters of the text.
         * @param length The length of the text in character.
         * @param isAllocated If the text is allocated in the memory.
         * @param isOwner If the Java object owns the text buffer.
         * @param contentArray The characters of the text
         * (as strings, this is only used in JNI mode).
         */
        private Text(
            final PointerWrapper charPointer,
            final long length,
            final boolean isAllocated,
            final boolean isOwner,
            final byte[] contentArray
        ) {
            this.charPointer = charPointer;
            this.length = length;
            this.isAllocated = isAllocated;
            this.isOwner = isOwner;

            if(contentArray != null) {
                this.content = decodeUTF32(contentArray);
            } else {
                this.content = null;
            }
        }

        /**
         * Create a new langkit text from its string content.
         *
         * @param content The content of the text in a Java string.
         * @return The newly created text.
         */
        public static Text create(
            final String content
        ) {
            final byte[] contentArray = encodeUTF32(content);

            if(ImageInfo.inImageCode()) {
                final PointerWrapper charPointer = new PointerWrapper(
                    toCBytes(contentArray)
                );
                return new Text(
                    charPointer,
                    (long) content.length(),
                    false,
                    true,
                    contentArray
                );
            } else {
                return JNI_LIB.lkt_create_text(contentArray);
            }
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the native text pointed by the given pointer.
         *
         * @param pointer The pointer to the native text to wrap.
         * @return The wrapped text.
         */
        static Text wrap(
            final WordPointer pointer
        ) {
            return wrap((TextNative) pointer.read());
        }

        /**
         * Wrap a text native value in the Java class.
         *
         * @param textNative The text native NI value.
         * @return The newly wrapped text.
         */
        static Text wrap(
            final TextNative textNative
        ) {
            return new Text(
                new PointerWrapper(textNative.get_chars()),
                textNative.get_length(),
                textNative.get_is_allocated() != 0
            );
        }

        /**
         * Unwrap the text in the given NI pointer.
         *
         * @param textNative The NI pointer to the native text structure.
         */
        void unwrap(
            final TextNative textNative
        ) {
            textNative.set_chars(this.charPointer.ni());
            textNative.set_length(this.length);
            textNative.set_is_allocated(this.isAllocated ? 1 : 0);
        }

        // ----- Instance methods -----

        /**
         * Get the content of the text in a Java string.
         *
         * @return the content of the text.
         */
        public String getContent() {
            // Content is null only when using the Graal C API.
            if(this.content == null) {
                final byte[] contentArray = new byte[(int) this.length * 4];
                for(int i = 0 ; i < contentArray.length ; i++) {
                    contentArray[i] = (
                        (CCharPointer) this.charPointer.ni()
                    ).read(i);
                }
                this.content = decodeUTF32(contentArray);
            }

            return this.content;
        }

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                if(this.isOwner) {
                    UnmanagedMemory.free(this.charPointer.ni());
                } else {
                    final TextNative textNative = StackValue.get(
                        TextNative.class
                    );
                    this.unwrap(textNative);
                    NI_LIB.lkt_destroy_text(textNative);
                }
            } else {
                JNI_LIB.lkt_destroy_text(this);
            }
            checkException();

        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.getContent();
        }

    }

    /**
     * Location in a source file. Line and column numbers are one-based.
     */
    public static final class SourceLocation {

        // ----- Class attributes -----

        /** Singleton that represents the none source location. */
        public static final SourceLocation NONE = new SourceLocation(
            0,
            (short) 0
        );

        // ----- Instance attributes -----

        /** The line of the source location. */
        public final int line;

        /** The column of the source location. */
        public final short column;

        // ----- Constructors -----

        /**
         * Create a new source location from a line and a column.
         *
         * @param line The line of the source location.
         * @param column The column of the source location.
         */
        SourceLocation(
            final int line,
            final short column
        ) {
            this.line = line;
            this.column = column;
        }

        /**
         * Create a source location from its line and column.
         *
         * @param line The line.
         * @param column The column.
         * @return The newly create source location.
         */
        public static SourceLocation create(
            final int line,
            final short column
        ) {
            return new SourceLocation(
                line,
                column
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given source location in the Java class.
         *
         * @param pointer Pointer to the native source location.
         * @return The newly wrapper source location.
         */
        static SourceLocation wrap(
            final WordPointer pointer
        ) {
            return wrap((SourceLocationNative) pointer.read());
        }

        /**
         * Wrap a source location native value in the Java class.
         *
         * @param sourceLocationNative The source location NI native value.
         * @return The newly wrapped source location.
         */
        static SourceLocation wrap(
            final SourceLocationNative sourceLocationNative
        ) {
            return new SourceLocation(
                sourceLocationNative.get_line(),
                sourceLocationNative.get_column()
            );
        }

        /**
         * Uwrap the source location in the given NI pointer.
         *
         * @param sourceLocationNative The NI pointer to the native structure
         *  to fill.
         */
        public void unwrap(
            final SourceLocationNative sourceLocationNative
        ) {
            sourceLocationNative.set_line(this.line);
            sourceLocationNative.set_column(this.column);
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.line + ":" + this.column;
        }

    }

    /**
     * Location of a span of text in a source file.
     */
    public static final class SourceLocationRange {

        // ----- Class attributes -----

        /** Singleton that represents the none source location range. */
        public static final SourceLocationRange NONE =
            new SourceLocationRange(
                SourceLocation.NONE,
                SourceLocation.NONE
            );

        // ----- Instance attributes -----

        /** The start of the range. */
        public final SourceLocation start;

        /** The end of the range. */
        public final SourceLocation end;

        // ----- Constructors -----

        /**
         * Create a source location range from its bounds.
         *
         * @param start The start of the range.
         * @param end The end of the range.
         */
        SourceLocationRange(
            final SourceLocation start,
            final SourceLocation end
        ) {
            this.start = start;
            this.end = end;
        }

        /**
         * Create a new source location range from its bounds.
         *
         * @param start The starting bound.
         * @param end The ending bound.
         * @return The newly created source location range.
         */
        public static SourceLocationRange create(
            final SourceLocation start,
            final SourceLocation end
        ) {
            return new SourceLocationRange(
                start,
                end
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given source location range in the Java class.*
         *
         * @param pointer The pointer to the native source location range.
         * @return The newly wrapped source location range.
         */
        static SourceLocationRange wrap(
            final WordPointer pointer
        ) {
            return wrap((SourceLocationRangeNative) pointer.read());
        }

        /**
         * Wrap a source location range native value in the Java class.
         *
         * @param sourceLocationRangeNative The source location range NI
         * native value.
         * @return The newly wrapped source location range.
         */
        static SourceLocationRange wrap(
            final SourceLocationRangeNative sourceLocationRangeNative
        ) {
            return new SourceLocationRange(
                new SourceLocation(
                    sourceLocationRangeNative.get_start_line(),
                    sourceLocationRangeNative.get_start_column()
                ),
                new SourceLocation(
                    sourceLocationRangeNative.get_end_line(),
                    sourceLocationRangeNative.get_end_column()
                )
            );
        }

        /**
         * Uwrap the source location range in the given NI pointer.
         *
         * @param sourceLocationRangeNative The NI pointer to the native
         * structure to fill.
         */
        void unwrap(
            final SourceLocationRangeNative sourceLocationRangeNative
        ) {
            sourceLocationRangeNative.set_start_line(this.start.line);
            sourceLocationRangeNative.set_start_column(this.start.column);
            sourceLocationRangeNative.set_end_line(this.end.line);
            sourceLocationRangeNative.set_end_column(this.end.column);
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.start.toString() + "-" + this.end.toString();
        }

    }

    /**
     * Diagnostic for an analysis unit: cannot open the source file, parsing
     * error, ...
     */
    public static final class Diagnostic {

        // ----- Class attributes -----

        /** Singleton that represents the none diagnostic. */
        public static final Diagnostic NONE = new Diagnostic(
            SourceLocationRange.NONE,
            Text.NONE
        );

        // ----- Instance attributes -----

        /** The source location range of the diagnostic. */
        public final SourceLocationRange sourceLocationRange;

        /** The message of the diagnostic. */
        public final Text message;

        // ----- Constructors -----

        /**
         * Create a diagnostic from its content.
         *
         * @param sourceLocationRange The range of the diagnostic.
         * @param message The message of the diagnostic.
         */
        Diagnostic(
            final SourceLocationRange sourceLocationRange,
            final Text message
        ) {
            this.sourceLocationRange = sourceLocationRange;
            this.message = message;
        }

        /**
         * Create a new diagnostic from its content.
         *
         * @param sourceLocationRange The source location range concerned by
         * this diagnostic.
         * @param message The message of the diagnostic.
         * @return The newly created diagnostic
         */
        public static Diagnostic create(
            final SourceLocationRange sourceLocationRange,
            final Text message
        ) {
            return new Diagnostic(
                sourceLocationRange,
                message
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to a native diagnostic.
         *
         * @param pointer The pointer to the native diagnostic.
         * @return The wrapped diagnostic.
         */
        static Diagnostic wrap(
            final WordPointer pointer
        ) {
            return wrap((DiagnosticNative) pointer.read());
        }

        /**
         * Wrap a diagnostic native value in the Java class.
         *
         * @param diagnosticNative The diagnostic NI native value.
         * @return The newly wrapped diagnostic.
         */
        static Diagnostic wrap(
            final DiagnosticNative diagnosticNative
        ) {
            return new Diagnostic(
                new SourceLocationRange(
                    new SourceLocation(
                        diagnosticNative.get_start_line(),
                        diagnosticNative.get_start_column()
                    ),
                    new SourceLocation(
                        diagnosticNative.get_end_line(),
                        diagnosticNative.get_end_column()
                    )
                ),
                new Text(
                    new PointerWrapper(diagnosticNative.get_message_chars()),
                    diagnosticNative.get_message_length(),
                    diagnosticNative.get_message_is_allocated() != 0
                )
            );
        }

        /**
         * Unwrap the diagnostic in the given NI pointer.
         *
         * @param diagnosticNative The pointer to the native structure to
         * fill.
         */
        public void unwrap(
            final DiagnosticNative diagnosticNative
        ) {
            diagnosticNative.set_start_line(
                this.sourceLocationRange.start.line
            );
            diagnosticNative.set_start_column(
                this.sourceLocationRange.start.column
            );
            diagnosticNative.set_end_line(
                this.sourceLocationRange.end.line
            );
            diagnosticNative.set_end_column(
                this.sourceLocationRange.end.column
            );
            diagnosticNative.set_message_chars(
                this.message.charPointer.ni()
            );
            diagnosticNative.set_message_length(
                this.message.length
            );
            diagnosticNative.set_message_is_allocated(
                this.message.isAllocated ? 1 : 0
            );
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.message.toString()
                + " at <"
                + this.sourceLocationRange.toString()
                + ">";
        }

    }

    /**
     * Result of applying a rewriting session.
     *
     * On success, ``Success`` is true.
     *
     * On failure, ``Success`` is false, ``Unit`` is set to the unit on which
     * rewriting failed, and ``Diagnostics`` is set to related rewriting
     * errors.
     */
    public static final class RewritingApplyResult implements AutoCloseable {

        // ----- Instance attributes -----

        /** Whether the rewriting application was successful. */
        public final boolean success;

        /**
         * If the rewriting failed, this is the unit on which the rewriting
         * failed. Otherwise it is the None unit.
         */
        public final AnalysisUnit unit;

        /**
         * Number of diagnostics in the result. 0 if the apply has been a
         * success.
         */
        private final int diagnosticsCount;

        /** Pointer to the diagnostic array, null if successful. */
        private final PointerWrapper diagnosticsReference;

        /** Cache for the unwrapped diagnostics. */
        private Diagnostic[] diagnostics;

        // ----- Constructors ------

        /** Create a new rewriting result with its content. */
        RewritingApplyResult(
            final boolean success,
            final AnalysisUnit unit,
            final int diagnosticsCount,
            final PointerWrapper diagnosticsReference
        ) {
            this.success = success;
            this.unit = unit;
            this.diagnosticsCount = diagnosticsCount;
            this.diagnosticsReference = diagnosticsReference;
            this.diagnostics = null;
        }

        /** Shortcut creation method for success. */
        static RewritingApplyResult success() {
            return new RewritingApplyResult(
                true,
                AnalysisUnit.NONE,
                0,
                PointerWrapper.nullPointer()
            );
        }

        // ----- Graal C API methods -----

        /** Wrap a pointer to a native rewriting apply result. */
        static RewritingApplyResult wrap(
            final Pointer pointer
        ) {
            return wrap((RewritingApplyResultNative) pointer.readWord(0));
        }

        /** Wrap a native rewriting apply result to the Java class. */
        static RewritingApplyResult wrap(
            final RewritingApplyResultNative rewritingApplyResultNative
        ) {
            if(rewritingApplyResultNative.get_success() > 0) {
                return RewritingApplyResult.success();
            }
            return new RewritingApplyResult(
                false,
                AnalysisUnit.wrap(rewritingApplyResultNative.get_unit()),
                rewritingApplyResultNative.get_diagnostics_count(),
                new PointerWrapper(
                    rewritingApplyResultNative.get_diagnostics()
                )
            );
        }

        /** Unwrap the rewriting apply result to the provided native value. */
        void unwrap(
            final RewritingApplyResultNative rewritingApplyResultNative
        ) {
            rewritingApplyResultNative.set_success(this.success ? 1 : 0);
            rewritingApplyResultNative.set_unit(this.unit.unwrap());
            rewritingApplyResultNative.set_diagnostics_count(
                this.diagnosticsCount
            );
            rewritingApplyResultNative.set_diagnostics(
                (DiagnosticNative) this.diagnosticsReference.ni()
            );
        }

        // ----- Instance methods -----

        /** Unwrap all diagnostics, cache the result and return it */
        public Diagnostic[] getDiagnostics() {
            if(this.diagnostics == null) {
                if(this.success) {
                    this.diagnostics = new Diagnostic[0];
                } else {

                    if(ImageInfo.inImageCode()) {
                        this.diagnostics =
                            new Diagnostic[this.diagnosticsCount];
                        final Pointer diagnostics =
                            (Pointer) this.diagnosticsReference.ni();
                        DiagnosticNative diagnosticNative;
                        final int diagnosticNativeSize = SizeOf.get(
                            DiagnosticNative.class
                        );
                        for(int i = 0; i < this.diagnosticsCount; i++) {
                            diagnosticNative = WordFactory.unsigned(
                                diagnostics.add(i * diagnosticNativeSize)
                                        .rawValue()
                            );
                            this.diagnostics[i] = Diagnostic.wrap(
                                diagnosticNative
                            );
                        }
                    } else {
                        this.diagnostics =
                            JNI_LIB
                                .lkt_rewriting_get_result_diagnostics(
                                    this.diagnosticsCount,
                                    this.diagnosticsReference.jni()
                                );
                    }

                }
            }
            return this.diagnostics;
        }

        /**
         * Free the result of the ``Apply`` operation.
         */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                final RewritingApplyResultNative resultNative =
                    StackValue.get(RewritingApplyResultNative.class);
                unwrap(resultNative);
                NI_LIB.lkt_rewriting_free_apply_result(resultNative);
            } else {
                JNI_LIB.lkt_rewriting_free_apply_result(this);
            }

        }

    }

    /**
     * Interface to override how source files are fetched and decoded.
     */
    public static final class FileReader implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none file reader. */
        public static final FileReader NONE = new FileReader(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the file reader */
        private final PointerWrapper reference;

        // ----- Constructors -----

        /**
         * Create a new file reader from its native reference.
         *
         * @param reference The reference to the native file reader.
         */
        FileReader(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to a native file reader.
         *
         * @param pointer The pointer to the native file reader.
         * @return The newly wrapped file reader.
         */
        static FileReader wrap(
            final WordPointer pointer
        ) {
            return wrap((FileReaderNative) pointer.read());
        }

        /**
         * Wrap the given file reader in the Java class.
         *
         * @param fileReaderNative The native file reader to wrap.
         * @return The wrapped file reader.
         */
        static FileReader wrap(
            final FileReaderNative fileReaderNative
        ) {
            return new FileReader(
                new PointerWrapper(fileReaderNative)
            );
        }

        /**
         * Unwrap the file reader in the given pointer.
         *
         * @param pointer The pointer to unwrap the file reader in.
         */
        void unwrap(
            final WordPointer pointer
        ) {
            pointer.write(0, this.unwrap());
        }

        /**
         * Get the unwrapped file reader.
         *
         * @return The unwrapped native file reader.
         */
        FileReaderNative unwrap() {
            return (FileReaderNative) this.reference.ni();
        }

        // ----- Instance methods -----

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_dec_ref_file_reader(
                    this.reference.ni()
                );
            } else {
                JNI_LIB.lkt_dec_ref_file_reader(this);
            }
            checkException();

        }

    }

    /**
     * Interface to fetch analysis units from a name and a unit kind.
     *
     * The unit provider mechanism provides an abstraction which assumes that
     * to any couple (unit name, unit kind) we can associate at most one source
     * file. This means that several couples can be associated to the same
     * source file, but on the other hand, only one one source file can be
     * associated to a couple.
     *
     * This is used to make the semantic analysis able to switch from one
     * analysis units to another.
     *
     * See the documentation of each unit provider for the exact semantics of
     * the unit name/kind information.
     */
    public static final class UnitProvider implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none unit provider. */
        public static final UnitProvider NONE = new UnitProvider(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the unit provider */
        private final PointerWrapper reference;

        // ----- Constructors -----

        /**
         * Create a new unit provider with the reference to the native one.
         *
         * @param reference The reference to the native unit provider.
         */
        UnitProvider(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to a native unit provider.
         *
         * @param pointer The pointer to the native unit provider.
         * @return The wrapped unit provider.
         */
        static UnitProvider wrap(
            final WordPointer pointer
        ) {
            return wrap((UnitProviderNative) pointer.read());
        }

        /**
         * Wrap the given native unit provider.
         *
         * @param unitProviderNative The native unit provider.
         * @return The wrapped unit provider.
         */
        static UnitProvider wrap(
            final UnitProviderNative unitProviderNative
        ) {
            return new UnitProvider(
                new PointerWrapper(unitProviderNative)
            );
        }

        /**
         * Unwrap the unit provider in the given native pointer.
         *
         * @param pointer The pointer to place the native unit provider in.
         */
        void unwrap(
            final WordPointer pointer
        ) {
            pointer.write(this.unwrap());
        }

        /**
         * Get the native unit provider.
         *
         * @return The native unit provider.
         */
        UnitProviderNative unwrap() {
            return (UnitProviderNative) this.reference.ni();
        }

        // ----- Instance methods -----

        /** @see java.lang.AutoCloseable#close() */
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_dec_ref_unit_provider(this.reference.ni());
            } else {
                JNI_LIB.lkt_dec_ref_unit_provider(this);
            }
            checkException();

        }

    }

    /**
     * Interface to handle events sent by the analysis context.
     */
    public static final class EventHandler implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none event handler. */
        public static final EventHandler NONE = new EventHandler(
            PointerWrapper.nullPointer(),
            null,
            null
        );

        /** This map contains all created event handlers. */
        private static final Map<PointerWrapper, EventHandler>
            eventHandlerCache = new ConcurrentHashMap<>();

        // ----- Instance attributes -----

        /** The reference to the native event handler. */
        private final PointerWrapper reference;

        /**
         * The Java callback when an analysis unit is requested in the
         * associated context.
         */
        private final UnitRequestedCallback unitRequestedCallback;

        /**
         * The Java callback when an analysis unit is parsed in the
         * associated context.
         */
        private final UnitParsedCallback unitParsedCallback;

        // ----- Constructors -----

        /**
         * Create a new event handler from its native reference.
         *
         * @param reference The reference to the native event handler.
         * @param unitRequestedCallback The callback for unit requests.
         * @param unitParsedCallback The callback for unit parsing.
         */
        EventHandler(
            final PointerWrapper reference,
            final UnitRequestedCallback unitRequestedCallback,
            final UnitParsedCallback unitParsedCallback
        ) {
            this.reference = reference;
            this.unitRequestedCallback = unitRequestedCallback;
            this.unitParsedCallback = unitParsedCallback;
        }

        /**
         * Create a new even handler with its callbacks. Callbacks can be null.
         *
         * @param unitRequestedCallback The callback for analysis unit requests.
         * @param unitParsedCallback The callback for analysis unit parsing.
         */
        public static EventHandler create(
            final UnitRequestedCallback unitRequestedCallback,
            final UnitParsedCallback unitParsedCallback
        ) {
            // Prepare the reference to the native event handler
            final PointerWrapper reference;

            if(ImageInfo.inImageCode()) {
                // Get the current thread
                final IsolateThread thread = CurrentIsolate.getCurrentThread();

                // Create the native event handler
                final EventHandlerNative resNative =
                    NI_LIB.lkt_create_event_handler(
                        (VoidPointer) thread,
                        WordFactory.nullPointer(),
                        NI_LIB.unitRequestedFunction.getFunctionPointer(),
                        NI_LIB.unitParsedFunction.getFunctionPointer()
                    );

                // Set the result to the created event handler
                reference = new PointerWrapper(resNative);
            } else {
                reference = JNI_LIB.lkt_create_event_handler(
                    unitRequestedCallback,
                    unitParsedCallback
                );
            }

            // Return the new event handler wrapped object
            return new EventHandler(
                reference,
                unitRequestedCallback,
                unitParsedCallback
            );
        }

        /**
         * Get event handler Java object from its native pointer.
         *
         * @param reference The pointer to the native event handler.
         * @return The associated Java event handler.
         */
        static EventHandler fromReference(
            final PointerWrapper reference
        ) {
            if(eventHandlerCache.containsKey(reference)) {
                return eventHandlerCache.get(reference);
            } else {
                throw new ReferenceException(
                    "Cannot get event handler from this reference: " +
                    reference.toString()
                );
            }
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to an event handler.
         *
         * @param pointer The pointer to the native event handler.
         * @return The wrapped event handler.
         */
        EventHandler wrap(
            final WordPointer pointer
        ) {
            return wrap((EventHandlerNative) pointer.read());
        }

        /**
         * Wrap the given native event handler.
         *
         * @param eventHandlerNative The native value of the event handler.
         * @return The wrapped event handler.
         */
        EventHandler wrap(
            final EventHandlerNative eventHandlerNative
        ) {
            return fromReference(new PointerWrapper(eventHandlerNative));
        }

        /**
         * Unwrap the event handler in the given native pointer.
         *
         * @param pointer The pointer to place the native event handler.
         */
        void unwrap(
            final WordPointer pointer
        ) {
            pointer.write(this.unwrap());
        }

        /**
         * Unwrap the event handler to a native value.
         *
         * @return The native value of the event handler.
         */
        EventHandlerNative unwrap() {
            return (EventHandlerNative) this.reference.ni();
        }

        // ----- Getters -----

        public UnitRequestedCallback getUnitRequestCallback() {
            return this.unitRequestedCallback;
        }

        public UnitParsedCallback getUnitParsedCallback() {
            return this.unitParsedCallback;
        }

        // ----- Instance methods -----

        /** @see java.lang.AutoCloseable#close() */
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_dec_ref_event_handler(this.reference.ni());
            } else {
                JNI_LIB.lkt_dec_ref_event_handler(this);
            }
            checkException();

        }

        // ----- Inner classes -----

        /**
         * Callback that will be called when a unit is requested from the
         * context ``Context``.
         *
         * ``Name`` is the name of the requested unit.
         *
         * ``From`` is the unit from which the unit was requested.
         *
         * ``Found`` indicates whether the requested unit was found or not.
         *
         * ``Is_Not_Found_Error`` indicates whether the fact that the unit was
         * not found is an error or not.
         *
         * .. warning:: The interface of this callback is probably subject to
         *    change, so should be treated as experimental.
         */
        @FunctionalInterface
        public interface UnitRequestedCallback {
            void invoke(
                AnalysisContext context,
                String name,
                AnalysisUnit from,
                boolean found,
                boolean isNotFoundError
            );
        }

        /**
         * Callback that will be called when any unit is parsed from the
         * context ``Context``.
         *
         * ``Unit`` is the resulting unit.
         *
         * ``Reparsed`` indicates whether the unit was reparsed, or whether it
         * was the first parse.
         */
        @FunctionalInterface
        public interface UnitParsedCallback {
            void invoke(
                AnalysisContext context,
                AnalysisUnit unit,
                boolean reparsed
            );
        }

    }

    /**
     * Reference to a token in an analysis unit.
     */
    public static class Token {

        // ----- Instance attributes -----

        /**
         * We only store the reference to the context to avoid ref-count
         * problems. To access the token context go throught the
         * analysis unit.
         */
        protected final PointerWrapper contextRef;

        /** The unit of the token. */
        public final AnalysisUnit unit;

        /** The pointer to the token data handler. */
        protected final PointerWrapper tokenDataHandler;

        /** The index of the token. */
        public final int tokenIndex;

        /** The trivia index. */
        public final int triviaIndex;

        /** The kind of the token. */
        public final TokenKind kind;

        /** The text of the token. */
        protected final String text;

        /** The source location range of the token. */
        public final SourceLocationRange sourceLocationRange;

        // ----- Constructors -----

        /**
         * Create a new token from its content.
         *
         * @param contextRef The context of the token.
         * @param unit The unit which owns the token.
         * @param tokenDataHandler The pointer to the token data.
         * @param tokenIndex The index of the token.
         * @param triviaIndex The trivia index of the token.
         * @param kind The kind of the token in an integer.
         * @param text The text of the token.
         * @param sourceLocationRange The location of the token.
         */
        Token(
            final PointerWrapper contextRef,
            final AnalysisUnit unit,
            final PointerWrapper tokenDataHandler,
            final int tokenIndex,
            final int triviaIndex,
            final TokenKind kind,
            final String text,
            final SourceLocationRange sourceLocationRange
        ) {
            this.contextRef = contextRef;
            this.unit = unit;
            this.tokenDataHandler = tokenDataHandler;
            this.tokenIndex = tokenIndex;
            this.triviaIndex = triviaIndex;
            this.kind = kind;
            this.text = text;
            this.sourceLocationRange = sourceLocationRange;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the pointer to a native token.
         *
         * @param pointer The pointer to the native token.
         * @param unit The analysis unit which owns the token.
         * @return The wrapped token or a none value if the token is a none
         * one.
         */
        static Token wrap(
            final WordPointer pointer,
            final AnalysisUnit unit
        ) {
            return wrap(
                (TokenNative) pointer.read(),
                unit
            );
        }

        /**
         * Wrap a native token value in the Java class.
         *
         * @param tokenNative The native NI token value.
         * @param unit The analysis unit that owns the token.
         * @return The wrapped token or a none value if the token data
         * handler is null.
         */
        static Token wrap(
            final TokenNative tokenNative,
            final AnalysisUnit unit
        ) {
            if(tokenNative.get_data().isNull())
                return NONE(unit);

            // Fetch the token kind, source location range and text from the
            // tokenNative reference.
            final TokenKind kind = TokenKind.fromC(
                NI_LIB.lkt_token_get_kind(tokenNative)
            );

            final SourceLocationRangeNative slocRangeNative =
                StackValue.get(SourceLocationRangeNative.class);
            NI_LIB.lkt_token_sloc_range(
                tokenNative,
                slocRangeNative
            );
            final SourceLocationRange slocRange =
                SourceLocationRange.wrap(slocRangeNative);

            final TextNative textNative = StackValue.get(TextNative.class);
            NI_LIB.lkt_token_range_text(
                tokenNative,
                tokenNative,
                textNative
            );
            final String text = Text.wrap(textNative).getContent();
            NI_LIB.lkt_destroy_text(textNative);

            // Wrap them in a high-level Token instance
            return new Token(
                new PointerWrapper(tokenNative.get_context()),
                unit,
                new PointerWrapper(tokenNative.get_data()),
                tokenNative.get_token_index(),
                tokenNative.get_trivia_index(),
                kind,
                text,
                slocRange
            );
        }

        /**
         * Unwrap the token in the given NI pointer.
         *
         * @param tokenNative The NI pointer to the native structure to
         * fill.
         */
        public void unwrap(
            final TokenNative tokenNative
        ) {
            tokenNative.set_context(this.contextRef.ni());
            tokenNative.set_data(this.tokenDataHandler.ni());
            tokenNative.set_token_index(this.tokenIndex);
            tokenNative.set_trivia_index(this.triviaIndex);
        }

        // ----- Getters -----

        public String getText() {
            return this.text;
        }

        public boolean isTrivia() {
            return this.triviaIndex != 0;
        }

        public boolean isNone() {
            return false;
        }

        // ----- Class methods -----

        /**
         * Get a none token for the given unit.
         *
         * @param unit The unit to get a none token for.
         * @return The none token for the given unit.
         */
        public static Token NONE(
            final AnalysisUnit unit
        ) {
            return NoToken.getInstance(unit);
        }

        /**
         * Get the text from the start token to the end token.
         *
         * @param start The start token.
         * @param end The end token.
         * @return The text between the two tokens.
         */
        @CompilerDirectives.TruffleBoundary
        public static String textRange(
            final Token start,
            final Token end
        ) {

            if(ImageInfo.inImageCode()) {
                final TokenNative startNative = StackValue.get(
                    TokenNative.class
                );
                start.unwrap(startNative);

                final TokenNative endNative = StackValue.get(
                    TokenNative.class
                );
                end.unwrap(endNative);

                final TextNative textNative = StackValue.get(
                    TextNative.class
                );
                Text.NONE.unwrap(textNative);
                NI_LIB.lkt_token_range_text(
                    startNative,
                    endNative,
                    textNative
                );
                try(final Text resText = Text.wrap(textNative)) {
                    return resText.getContent();
                }
            } else {
                try(
                    final Text resText = JNI_LIB.lkt_token_range_text(
                        start,
                        end
                    )
                ) {
                    return resText.getContent();
                }
            }

        }

        // ----- Instance methods -----

        /**
         * Get the next token.
         *
         * @return The next token in the source.
         */
        public Token next() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                this.unwrap(tokenNative);

                final TokenNative nextNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.lkt_token_next(
                    tokenNative,
                    nextNative
                );
                return wrap(nextNative, this.unit);
            } else {
                return JNI_LIB.lkt_token_next(this);
            }

        }

        /**
         * Get the previous token.
         *
         * @return The previous token in the source.
         */
        public Token previous() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                this.unwrap(tokenNative);

                final TokenNative previousNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.lkt_token_previous(
                    tokenNative,
                    previousNative
                );
                return wrap(previousNative, this.unit);
            } else {
                return JNI_LIB.lkt_token_previous(this);
            }

        }

        /**
         * Check of the token is equivalent to the other one.
         *
         * @param other The other token to compare with.
         */
        public boolean isEquivalent(
            final Token other
        ) {

            if(ImageInfo.inImageCode()) {
                final TokenNative leftNative = StackValue.get(
                    TokenNative.class
                );
                this.unwrap(leftNative);

                final TokenNative rightNative = StackValue.get(
                    TokenNative.class
                );
                other.unwrap(rightNative);

                return NI_LIB.lkt_token_is_equivalent(
                    leftNative,
                    rightNative
                ) != 0;
            } else {
                return JNI_LIB.lkt_token_is_equivalent(this, other);
            }

        }

        // ----- Override methods -----

        @Override
        @CompilerDirectives.TruffleBoundary
        public String toString() {
            return "<Token Kind="
                + this.kind.name
                + " Text=\""
                + stringRepresentation(this.getText())
                + "\">";
        }

        @Override
        public boolean equals(
            Object o
        ) {
            if(o == this) return true;
            if(!(o instanceof Token)) return false;
            final Token other = (Token) o;
            return this.tokenDataHandler.equals(other.tokenDataHandler) &&
                    this.tokenIndex == other.tokenIndex &&
                    this.triviaIndex == other.triviaIndex;
        }

        // ----- Inner classes -----

        /**
        * This class represents the absence of token.
        */
        private static final class NoToken extends Token {

            // ----- Class attributes -----

            /** The map of the no token instances */
            private static final Map<AnalysisUnit, NoToken> instances
                = new HashMap<>();

            // ----- Constructors -----

            /**
            * Create a new no token for the given analysis unit.
            * The constructor is private to avoid too many instances.
            *
            * @param unit The analysis unit to create the no token for.
            */
            private NoToken(
                final PointerWrapper contextRef,
                final AnalysisUnit unit
            ) {
                super(
                    contextRef,
                    unit,
                    PointerWrapper.nullPointer(),
                    0,
                    0,
                    TokenKind.fromC(-1),
                    null,
                    SourceLocationRange.NONE
                );
            }

            /**
            * Get the no token instance for the given analysis unit.
            *
            * @param unit The unit to get the no token instance for.
            * @return The no token instance.
            */
            static NoToken getInstance(
                final AnalysisUnit unit
            ) {
                if(!instances.containsKey(unit)) {
                    try(AnalysisContext context = unit.getContext()) {
                        instances.put(
                            unit,
                            new NoToken(context.reference, unit)
                        );
                    }
                }
                return instances.get(unit);
            }

            // ----- Getters -----

            @Override
            public String getText() {
                return "";
            }

            // ----- Instance methods -----

            @Override
            public Token next() {
                return this;
            }

            @Override
            public Token previous() {
                return this;
            }

            @Override
            public boolean isEquivalent(
                final Token other
            ) {
                return other instanceof NoToken;
            }

            @Override
            public boolean isNone() {
                return true;
            }

            @Override
            public void unwrap(
                final TokenNative tokenNative
            ) {
                tokenNative.set_context(this.contextRef.ni());
                tokenNative.set_data(this.tokenDataHandler.ni());
                tokenNative.set_token_index(this.tokenIndex);
                tokenNative.set_trivia_index(this.triviaIndex);
            }

            // ----- Override methods -----

            @Override
            @CompilerDirectives.TruffleBoundary
            public String toString() {
                return "<Token Kind="
                    + this.kind.name
                    + " Text=\"\">";
            }

            @Override
            public boolean equals(
                Object o
            ) {
                return o == this;
            }

        }

    }

    /**
     * This type represents a context for all source analysis. This is the
     * first type you need to create to use Liblktlang. It will contain the
     * results of all analysis, and is the main holder for all the data.
     *
     * You can create several analysis contexts if you need to, which enables
     * you, for example to:
     *
     * * analyze several different projects at the same time;
     *
     * * analyze different parts of the same projects in parallel.
     *
     * In the current design, contexts always keep all of their analysis units
     * allocated. If you need to get this memory released, the only option at
     * your disposal is to destroy your analysis context instance.
     */
    public static final class AnalysisContext implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none analysis context. */
        public static final AnalysisContext NONE = new AnalysisContext(
            PointerWrapper.nullPointer(),
            null
        );

        /** This map contains all created analysis contexts. */
        private static final Map<PointerWrapper, AnalysisContext> contextCache
            = new ConcurrentHashMap<>();

        // ----- Instance attributes -----

        /** The reference to the native analysis context. */
        private final PointerWrapper reference;

        /** The event handler associated with the context. */
        private EventHandler eventHandler;

        /**
         * The rewriting context associated with this analysis context.
         * It can be the none value.
         */
        RewritingContext rewritingContext = RewritingContext.NONE;

        // ----- Constructors -----

        /**
         * Create a new analysis unit from its reference.
         *
         * @param reference The native analysis context.
         * @param eventHandler The associated event handler.
         */
        private AnalysisContext(
            final PointerWrapper reference,
            final EventHandler eventHandler
        ) {
            this.reference = reference;
            this.eventHandler = eventHandler;

            increaseRefCounter(this);
        }

        /**
         * Create a new analysis context from scratch with its configuration.
         *
         * @param charset The charset for the analysis context, it can be null.
         * @param fileReader The file reader for the analysis context, it
         * can be null.
         * @param unitProvider The unit provider for the analysis context,
         * it can be null.
         * @param eventHandler The event handler for the analysis context,
         * it can be null.
         * @param withTrivia If the analysis context should include trivias.
         * @param tabStop The effect of the tabulations on the column number.
         */
        private AnalysisContext(
            final String charset,
            final FileReader fileReader,
            final UnitProvider unitProvider,
            final EventHandler eventHandler,
            final boolean withTrivia,
            final int tabStop
        ) {
            // Call the function to allocate the native analysis context
            final PointerWrapper reference;
            if(ImageInfo.inImageCode()) {
                reference = new PointerWrapper(
                    NI_LIB.lkt_allocate_analysis_context()
                );
            } else {
                reference = JNI_LIB.lkt_create_analysis_context(
                    charset,
                    (fileReader == null ?
                        FileReader.NONE :
                        fileReader),
                    (unitProvider == null ?
                        UnitProvider.NONE :
                        unitProvider),
                    (eventHandler == null ?
                        EventHandler.NONE :
                        eventHandler),
                    withTrivia,
                    tabStop
                );
            }

            // Place the context in the cache for potention callbacks during
            // context initialization.
            this.reference = reference;
            this.eventHandler = eventHandler;
            contextCache.put(this.reference, this);

            // Perform the context initialization
            if(ImageInfo.inImageCode()) {
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);

                NI_LIB.lkt_initialize_analysis_context(
                    (AnalysisContextNative) reference.ni(),
                    charsetNative,
                    (fileReader == null ?
                        WordFactory.nullPointer() :
                        fileReader.reference.ni()),
                    (unitProvider == null ?
                        WordFactory.nullPointer() :
                        unitProvider.reference.ni()),
                    (eventHandler == null ?
                        WordFactory.nullPointer() :
                        eventHandler.reference.ni()),
                    (withTrivia ? 1 : 0),
                    tabStop
                );
                if(charset != null) UnmanagedMemory.free(charsetNative);
            }
        }

        /**
         * Create a new analysis context with the default parameters.
         *
         * @return The newly create analysis unit.
         */
        public static AnalysisContext create() {
            return new AnalysisContext(
                null,
                null,
                null,
                null,
                true,
                8
            );
        }

        /**
         * Create an analysis context with its parameters.
         *
         * @param charset The charset for the analysis context, it can be null.
         * @param fileReader The file reader for the analysis context, it
         * can be null.
         * @param unitProvider The unit provider for the analysis context,
         * it can be null.
         * @param eventHandler The event handler for the analysis context,
         * it can be null.
         * @param withTrivia If the analysis context should include trivias.
         * @param tabStop The effect of the tabulations on the column number.
         * @return The newly create analysis unit.
         */
        public static AnalysisContext create(
            final String charset,
            final FileReader fileReader,
            final UnitProvider unitProvider,
            final EventHandler eventHandler,
            final boolean withTrivia,
            final int tabStop
        ) {
            return new AnalysisContext(
                charset,
                fileReader,
                unitProvider,
                eventHandler,
                withTrivia,
                tabStop
            );
        }

        /**
         * Get the analysis context object from its reference.
         *
         * @param reference The native reference to the analysis context.
         * @param eventHandler The corresponding event handler.
         * @param setEventHandler Whether to set the result's eventHandler
         * field to eventHandler when there is already a cached
         * AnalysisContext.
         * @return The Java analysis unit associated with the reference.
         */
        static AnalysisContext fromReference(
            final PointerWrapper reference,
            final EventHandler eventHandler,
            final boolean setEventHandler
        ) {
            if(contextCache.containsKey(reference)) {
                final AnalysisContext res = contextCache.get(reference);
                increaseRefCounter(res);
                if(setEventHandler)
                    res.eventHandler = eventHandler;
                return res;
            } else {
                final AnalysisContext result =
                    new AnalysisContext(reference, eventHandler);
                contextCache.put(reference, result);
                return result;
            }
        }

        /**
         * Get the analysis context object from its reference.
         *
         * @param reference The native reference to the analysis context.
         * @return The Java analysis unit associated with the reference.
         */
        static AnalysisContext fromReference(
            final PointerWrapper reference
        ) {
            return fromReference(reference, null, false);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a native pointer to the native analysis context in the
         * Java class.
         *
         * @param pointer The pointer to the NI analysis context
         * native value.
         * @return The newly wrapped analysis context.
         */
        static AnalysisContext wrap(
            final WordPointer pointer
        ) {
            return wrap((AnalysisContextNative) pointer.read());
        }

        /**
         * Wrap an analysis context native value in the Java class.
         *
         * @param analysisContextNative The NI analysis context native value.
         * @return The newly wrapped analysis context.
         */
        static AnalysisContext wrap(
            final AnalysisContextNative analysisContextNative
        ) {
            return fromReference(new PointerWrapper(analysisContextNative));
        }

        /**
         * Unwrap the analysis context in the given native pointer.
         *
         * @param pointer The pointer to place the native analysis unit.
         */
        void unwrap(
            final WordPointer pointer
        ) {
            pointer.write(this.unwrap());
        }

        /**
         * Get the native value of the analysis context.
         *
         * @return The native analysis context.
         */
        AnalysisContextNative unwrap() {
            return (AnalysisContextNative) this.reference.ni();
        }

        // ----- Getters -----

        public EventHandler getEventHandler() {
            return this.eventHandler;
        }

        /**
         * Get the currently open rewriting context associated to this
         * analysis context. The None rewriting context is returned if the
         * current context hasn't started a rewriting session.
         * @see AnalysisContext#startRewriting()
         */
        public RewritingContext getRewritingContext() {
            return this.rewritingContext;
        }

        // ----- Class methods -----

        /**
         * Increase the reference counter of the given context.
         *
         * @param context The context to increase the reference counter of.
         */
        private static void increaseRefCounter(
            final AnalysisContext context
        ) {
            // Increase the context reference counter of the context if not null
            if(!context.reference.isNull()) {
                if(ImageInfo.inImageCode()) {
                    NI_LIB.lkt_context_incref(context.reference.ni());
                } else {
                    JNI_LIB.lkt_context_incref(context.reference.jni());
                }
            }
        }

        // ----- Instance methods -----

        /**
         * Get an analysis unit from the given file in the current context.
         *
         * @param fileName The file to get the analysis unit from.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromFile(
            final String fileName
        ) {
            return this.getUnitFromFile(
                fileName,
                null,
                false,
                DEFAULT_GRAMMAR_RULE
            );
        }

        /**
         * Get an analysis unit from the given file in the current context
         * with additional parameters.
         *
         * @param fileName The file to get the analysis unit from.
         * @param charset The charset of the given file.
         * @param reparse If the file should be reparsed.
         * @param rule The grammar rule to parse the source with.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromFile(
            final String fileName,
            final String charset,
            final boolean reparse,
            final GrammarRule rule
        ) {

            if(ImageInfo.inImageCode()) {
                final CCharPointer fileNameNative = toCString(fileName);
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);

                final AnalysisUnitNative resNative =
                    NI_LIB.lkt_get_analysis_unit_from_file(
                    this.reference.ni(),
                    fileNameNative,
                    charsetNative,
                    (reparse ? 1 : 0),
                    rule.toC()
                );
                UnmanagedMemory.free(fileNameNative);
                if(charset != null) UnmanagedMemory.free(charsetNative);
                return AnalysisUnit.wrap(resNative);
            } else {
                return JNI_LIB.lkt_get_analysis_unit_from_file(
                    this,
                    fileName,
                    charset,
                    reparse,
                    rule.toC()
                );
            }

        }

        /**
         * Get an analysis unit from the given buffer in the current context.
         *
         * @param buffer The buffer to parse.
         * @param name The name of the buffer.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromBuffer(
            final String buffer,
            final String name
        ) {
            return this.getUnitFromBuffer(
                buffer,
                name,
                null,
                DEFAULT_GRAMMAR_RULE
            );
        }

        /**
         * Get an analysis unit from the given buffer in the current context
         * with additional parameters.
         *
         * @param buffer The buffer to parse.
         * @param name The name of the buffer.
         * @param charset The charset of the buffer.
         * @param rule The rule to parse the buffer with.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromBuffer(
            final String buffer,
            final String name,
            final String charset,
            final GrammarRule rule
        ) {

            if(ImageInfo.inImageCode()) {
                final CCharPointer bufferNative = toCString(buffer);
                final CCharPointer nameNative = toCString(name);
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);

                final AnalysisUnitNative resNative =
                    NI_LIB.lkt_get_analysis_unit_from_buffer(
                    this.reference.ni(),
                    nameNative,
                    charsetNative,
                    bufferNative,
                    buffer.length(),
                    rule.toC()
                );
                UnmanagedMemory.free(bufferNative);
                UnmanagedMemory.free(nameNative);
                if(charset != null) UnmanagedMemory.free(charsetNative);
                return AnalysisUnit.wrap(resNative);
            } else {
                return JNI_LIB.lkt_get_analysis_unit_from_buffer(
                    this,
                    name,
                    charset,
                    buffer,
                    buffer.length(),
                    rule.toC()
                );
            }

        }

        /**
         * Get an analysis unit from the given unit name and unit kind in the
         * current context with additional parameters.
         *
         * @param name Name of the unit.
         * @param kind Kind of the unit.
         * @param charset The charset of the buffer.
         * @param rule The rule to parse the buffer with.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromProvider(
            final Text name,
            final AnalysisUnitKind kind,
            final String charset,
            final boolean reparse
        ) {
            if(ImageInfo.inImageCode()) {
                TextNative nameNative = StackValue.get(TextNative.class);
                name.unwrap(nameNative);
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);
                final AnalysisUnitNative resNative =
                    NI_LIB.lkt_get_analysis_unit_from_provider(
                    this.reference.ni(),
                    nameNative,
                    kind.toC(),
                    charsetNative,
                    (reparse ? 1 : 0)
                );
                if(charset != null) UnmanagedMemory.free(charsetNative);
                return AnalysisUnit.wrap(resNative);
            } else {
                return JNI_LIB.lkt_get_analysis_unit_from_provider(
                    this,
                    name,
                    kind.toC(),
                    charset,
                    reparse
                );
            }
        }

        public AnalysisUnit getUnitFromProvider(
            final Text name,
            final AnalysisUnitKind kind
        ) {
            return this.getUnitFromProvider(name, kind, "", false);
        }

        /**
         * Start a rewriting session for Context.
         *
         * This handle will keep track of all changes to do on Context's
         * analysis units. Once the set of changes is complete, call the Apply
         * procedure to actually update Context. This makes it possible to
         * inspect the "old" Context state while creating the list of changes.
         *
         * There can be only one rewriting session per analysis context, so
         * this will raise an Existing_Rewriting_Handle_Error exception if
         * Context already has a living rewriting session.
         */
        public RewritingContext startRewriting() {
            final RewritingContext res;

            if(ImageInfo.inImageCode()) {
                final RewritingContextNative resNative =
                    NI_LIB.lkt_rewriting_start_rewriting(
                        this.reference.ni()
                    );
                res = RewritingContext.wrap(resNative);
            } else {
                res = JNI_LIB.lkt_rewriting_start_rewriting(this);
            }

            checkException();
            this.rewritingContext = res;
            return this.rewritingContext;
        }

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_context_decref(this.reference.ni());
            } else {
                JNI_LIB.lkt_context_decref(this.reference.jni());
            }
            checkException();

        }

        


    }

    /**
     * This type represents the analysis of a single file.
     *
     * This type has strong-reference semantics and is ref-counted.
     * Furthermore, a reference to a unit contains an implicit reference to the
     * context that owns it. This means that keeping a reference to a unit will
     * keep the context and all the unit it contains allocated.
     */
    public static final class AnalysisUnit {

        // ----- Class attributes -----

        /** Singleton that represents the none analysis unit. */
        public static final AnalysisUnit NONE = new AnalysisUnit(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the native analysis unit. */
        private final PointerWrapper reference;

        /** The cache for the unit root. */
        private LktNode root;

        // ----- Constructors -----

        /**
         * Create a new analysis unit from its value.
         *
         * @param reference The native analysis unit native value in
         * a pointer wrapper.
         */
        AnalysisUnit(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native analysis unit in the Java class.
         *
         * @param pointer The pointer the native analysis unit value.
         * @return The newly wrapped analysis unit.
         */
        static AnalysisUnit wrap(
            final WordPointer pointer
        ) {
            return wrap((AnalysisUnitNative) pointer.read());
        }

        /**
         * Wrap a NI analysis unit native value in the Java class.
         *
         * @param analysisUnitNative The NI analysis unit native value.
         * @return The newly wrapped analysis unit.
         */
        static AnalysisUnit wrap(
            final AnalysisUnitNative analysisUnitNative
        ) {
            return new AnalysisUnit(new PointerWrapper(analysisUnitNative));
        }

        /**
         * Unwrap the analysis unit in the given pointer.
         *
         * @param pointer The pointer to place the native analysis unit in.
         */
        void unwrap(
            final WordPointer pointer
        ) {
            pointer.write(this.unwrap());
        }

        /**
         * Unwrap the analysis unit as a native value.
         *
         * @return The native analysis unit.
         */
        AnalysisUnitNative unwrap() {
            return (AnalysisUnitNative) this.reference.ni();
        }

        // ----- Instance methods -----

        /**
         * Get the root node of the analysis unit.
         *
         * @return The root node.
         */
        public LktNode getRoot() {
            if(this.root == null) {

                if(ImageInfo.inImageCode()) {
                    final EntityNative entityNative = StackValue.get(
                        EntityNative.class
                    );
                    NI_LIB.lkt_unit_root(
                        this.reference.ni(),
                        entityNative
                    );
                    this.root = LktNode.fromEntity(
                        Entity.wrap(entityNative)
                    );
                } else {
                    this.root = LktNode.fromEntity(
                        JNI_LIB.lkt_unit_root(this)
                    );
                }

            }
            return this.root;
        }

        /**
         * Get the analysis unit file name with its full path.
         *
         * @return The unit file name.
         */
        public String getFileName() {
            return this.getFileName(true);
        }

        /**
         * Get the unit's file name.
         *
         * @param fullPath If the method should return the
         * file full absolute path.
         * @return The file name.
         */
        @CompilerDirectives.TruffleBoundary
        public String getFileName(
            final boolean fullPath
        ) {
            final String absoluteFile;

            if(ImageInfo.inImageCode()) {
                final CCharPointer resNative = NI_LIB.lkt_unit_filename(
                    this.reference.ni()
                );
                absoluteFile = toJString(resNative);
                NI_LIB.lkt_free(resNative);
            } else {
                absoluteFile = JNI_LIB.lkt_unit_filename(this);
            }

            if(fullPath) {
                return absoluteFile;
            } else {
                return new File(absoluteFile).getName();
            }
        }

        /**
         * Get the number of tokens in the analysis unit.
         *
         * @return The number of token.
         */
        public int getTokenCount() {

            if(ImageInfo.inImageCode()) {
                return NI_LIB.lkt_unit_token_count(this.reference.ni());
            } else {
                return JNI_LIB.lkt_unit_token_count(this);
            }

        }

        /**
         * Get the number of trivia in the analysis unit.
         *
         * @return The number of trivia.
         */
        public int getTriviaCount() {

            if(ImageInfo.inImageCode()) {
                return NI_LIB.lkt_unit_trivia_count(
                    this.reference.ni()
                );
            } else {
                return JNI_LIB.lkt_unit_trivia_count(this);
            }

        }

        /**
         * Return the first token of the analysis unit.
         *
         * @return The first token.
         */
        public Token getFirstToken() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.lkt_unit_first_token(
                    this.reference.ni(),
                    tokenNative
                );
                return Token.wrap(tokenNative, this);
            } else {
                return JNI_LIB.lkt_unit_first_token(this);
            }

        }

        /**
         * Return the last token of the analysis unit.
         *
         * @return The last token.
         */
        public Token getLastToken() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.lkt_unit_last_token(
                    this.reference.ni(),
                    tokenNative
                );
                return Token.wrap(tokenNative, this);
            } else {
                return JNI_LIB.lkt_unit_last_token(this);
            }

        }

        /**
         * Get the text of the analysis unit in a string.
         *
         * @return The text of the analysis unit source.
         */
        public String getText() {
            return Token.textRange(
                this.getFirstToken(),
                this.getLastToken()
            );
        }

        /**
         * Get the analysis context that owns the unit.
         *
         * @return The owning analysis context.
         */
        public AnalysisContext getContext() {

            if(ImageInfo.inImageCode()) {
                final AnalysisContextNative contextNative =
                    NI_LIB.lkt_unit_context(this.reference.ni());
                return AnalysisContext.wrap(contextNative);
            } else {
                return JNI_LIB.lkt_unit_context(this);
            }

        }

        /**
         * Get the list of associated diagnostics. Those are parsing errors.
         *
         * @return The diagnostics of the unit.
         */
        public Diagnostic[] getDiagnostics() {
            final int diagnosticCount;

            if(ImageInfo.inImageCode()) {
                diagnosticCount = NI_LIB.lkt_unit_diagnostic_count(
                    this.reference.ni()
                );
            } else {
                diagnosticCount = JNI_LIB.lkt_unit_diagnostic_count(
                    this
                );
            }

            Diagnostic[] res = new Diagnostic[diagnosticCount];

            if(ImageInfo.inImageCode()) {
                final DiagnosticNative diagnosticNative = StackValue.get(
                    DiagnosticNative.class
                );
                for(int i = 0 ; i < diagnosticCount ; i++) {
                    NI_LIB.lkt_unit_diagnostic(
                        this.reference.ni(),
                        i,
                        diagnosticNative
                    );
                    res[i] = Diagnostic.wrap(diagnosticNative);
                }
            } else {
                for(int i = 0 ; i < diagnosticCount ; i++) {
                    res[i] = JNI_LIB.lkt_unit_diagnostic(this, i);
                }
            }

            return res;
        }

        /**
         * Return the rewriting handle corresponding to Unit
         */
        public RewritingUnit getRewritingUnit() {
            final RewritingUnit res;

            if(ImageInfo.inImageCode()) {
                RewritingUnitNative rewritingUnitNative =
                    NI_LIB.lkt_rewriting_unit_to_handle(
                        this.unwrap()
                    );
                res = RewritingUnit.wrap(rewritingUnitNative);
            } else {
                res = JNI_LIB.lkt_rewriting_unit_to_handle(this);
            }

            checkException();
            return res;
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return "<AnalysisUnit \"" + this.getFileName(false) + "\">";
        }

        @Override
        public boolean equals(Object o) {
            if(this == o) return true;
            if(!(o instanceof AnalysisUnit)) return false;
            final AnalysisUnit other = (AnalysisUnit) o;
            return this.reference.equals(other.reference);
        }

    }

    /**
     * Handle for an analysis context rewriting session
     */
    public static final class RewritingContext implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton representing the none rewriting context */
        public static final RewritingContext NONE = new RewritingContext(
            PointerWrapper.nullPointer()
        );

        /**
         * This map contains all wrapped rewriting context associated to their
         * address.
         */
        private static final Map<PointerWrapper, RewritingContext> contextCache
            = new ConcurrentHashMap<>();

        // ----- Instance attributes -----

        /** The reference to the native rewriting context handle. */
        private final PointerWrapper reference;

        /**
         * Cache for the analysis context associated with this rewriting
         * context.
         */
        private AnalysisContext analysisContext;

        /** Whether the rewriting context has already been closed. */
        private boolean closed;

        // ----- Constructors -----

        /** Create a new rewriting context with its native reference. */
        private RewritingContext(
            final PointerWrapper reference
        ) {
            this.reference = reference;
            this.closed = false;
        }

        /**
         * From the given native reference, get the associate rewriting
         * context Java object. A native reference can only have one
         * associated Java instance.
         */
        @CompilerDirectives.TruffleBoundary
        static RewritingContext fromReference(
            final PointerWrapper reference
        ) {
            if(!contextCache.containsKey(reference)) {
                contextCache.put(
                    reference,
                    new RewritingContext(reference)
                );
            }
            final RewritingContext res = contextCache.get(reference);
            res.closed = false;
            return res;
        }

        // ----- Graal C API methods -----

        /** Wrap a native pointer to a native rewriting context. */
        static RewritingContext wrap(
            final Pointer pointer
        ) {
            return wrap((RewritingContextNative) pointer.readWord(0));
        }

        /** Wrap a native rewriting context. */
        static RewritingContext wrap(
            final RewritingContextNative rewritingContextNative
        ) {
            return fromReference(new PointerWrapper(rewritingContextNative));
        }

        /** Unwrap the analysis context in the given native pointer. */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /** Unwrap the rewriting context as a native value. */
        RewritingContextNative unwrap() {
            return (RewritingContextNative) this.reference.ni();
        }

        // ----- Getters -----

        public boolean isClosed() {
            return this.closed;
        }


        /**
         * Return the analysis context associated to Handle
         */
        public AnalysisContext getAnalysisContext() {
            if(this.analysisContext == null) {

                if(ImageInfo.inImageCode()) {
                    this.analysisContext = AnalysisContext.wrap(
                        NI_LIB.lkt_rewriting_handle_to_context(
                            this.unwrap()
                        )
                    );
                } else {
                    this.analysisContext =
                        JNI_LIB.lkt_rewriting_handle_to_context(this);
                }

            }
            return this.analysisContext;
        }

        // ----- Instance methods -----

        /**
         * Return the list of unit rewriting handles in the given context
         * handle for units that the Apply primitive will modify.
         */
        public RewritingUnit[] rewritingUnits() {

            if(ImageInfo.inImageCode()) {
                // Get the native array
                final WordPointer unitArrayNative =
                    NI_LIB.lkt_rewriting_unit_handles(this.unwrap());

                // Fill the Java result list
                final List<RewritingUnit> resList = new ArrayList<>();
                int cursor = 0;
                while(((Pointer) unitArrayNative.read(cursor)).isNonNull()) {
                    resList.add(RewritingUnit.wrap(
                        (RewritingUnitNative) unitArrayNative.read(cursor)
                    ));
                    cursor++;
                }

                // Free the native array
                NI_LIB.lkt_free(unitArrayNative);

                // Return the Java list as an array
                return resList.toArray(new RewritingUnit[0]);
            } else {
                return JNI_LIB.lkt_rewriting_unit_handles(this);
            }

        }

        /**
         * Create a new node of the given Kind, with empty text (for token
         * nodes) or children (for regular nodes).
         */
        public RewritingNode createNode(
            final NodeKind kind
        ) {
            final RewritingNode res;

            if(ImageInfo.inImageCode()) {
                res = RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_create_node(
                        this.unwrap(),
                        kind.toC()
                    )
                );
            } else {
                res = JNI_LIB.lkt_rewriting_create_node(
                    this,
                    kind.toC()
                );
            }

            checkException();
            return res;
        }

        /**
         * Create a new regular node of the given Kind and assign it the given
         * Children.
         *
         * Except for lists, which can have any number of children, the size of
         * Children must match the number of children associated to the given
         * Kind. Besides, all given children must not be tied.
         */
        public RewritingNode createNode(
            final NodeKind kind,
            final RewritingNode... children
        ) {
            final RewritingNode res;

            if(ImageInfo.inImageCode()) {
                final WordPointer childrenNative =
                    rewritingNodesToNative(children);
                res = RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_create_regular_node(
                        this.unwrap(),
                        kind.toC(),
                        childrenNative,
                        children.length
                    )
                );
                UnmanagedMemory.free(childrenNative);
            } else {
                res = JNI_LIB.lkt_rewriting_create_regular_node(
                    this,
                    kind.toC(),
                    children
                );
            }

            checkException();
            return res;
        }

        /**
         * Create a new token node with the given Kind and Text
         */
        public RewritingNode createTokenNode(
            final NodeKind kind,
            final String text
        ) {
            try (
                final Text nodeText = Text.create(text);
            ) {
                final RewritingNode res;

                if(ImageInfo.inImageCode()) {
                    final TextNative nodeTextNative = StackValue.get(
                        TextNative.class
                    );
                    nodeText.unwrap(nodeTextNative);
                    res = RewritingNode.wrap(
                        NI_LIB.lkt_rewriting_create_token_node(
                            this.unwrap(),
                            kind.toC(),
                            nodeTextNative
                        )
                    );
                } else {
                    res = JNI_LIB.lkt_rewriting_create_token_node(
                        this,
                        kind.toC(),
                        nodeText
                    );
                }

                checkException();
                return res;
            }
        }

        /**
         * Create a tree of new nodes from the given Template string, replacing
         * placeholders with nodes in Arguments and parsed according to the
         * given grammar Rule.
         */
        public RewritingNode createFromTemplate(
            final String template,
            final GrammarRule rule,
            final RewritingNode... arguments
        ) {
            try (
                final Text templateText = Text.create(template);
            ) {
                final RewritingNode res;

                if(ImageInfo.inImageCode()) {
                    final WordPointer argumentsNative =
                        rewritingNodesToNative(arguments);
                    TextNative templateTextNative =
                        StackValue.get(TextNative.class);
                    templateText.unwrap(templateTextNative);
                    res = RewritingNode.wrap(
                        NI_LIB.lkt_rewriting_create_from_template(
                            this.unwrap(),
                            templateTextNative,
                            argumentsNative,
                            arguments.length,
                            rule.toC()
                        )
                    );
                    UnmanagedMemory.free(argumentsNative);
                } else {
                    res = JNI_LIB.lkt_rewriting_create_from_template(
                        this,
                        templateText,
                        arguments,
                        rule.toC()
                    );
                }

                checkException();
                return res;
            }
        }

        /**
         * Apply all modifications to Handle's analysis context. If that
         * worked, close Handle and return (Success => True). Otherwise,
         * reparsing did not work, so keep Handle and its Context unchanged and
         * return details about the error that happened.
         *
         * Note that on success, this invalidates all related unit/node
         * handles.
         */
        public RewritingApplyResult apply() {
            final RewritingApplyResult res;

            if(ImageInfo.inImageCode()) {
                RewritingApplyResultNative resNative = StackValue.get(
                    RewritingApplyResultNative.class
                );
                NI_LIB.lkt_rewriting_apply(
                    this.unwrap(),
                    resNative
                );
                res = RewritingApplyResult.wrap(resNative);
            } else {
                res = JNI_LIB.lkt_rewriting_apply(this);
            }

            this.closed = res.success;
            return res;
        }

        /**
         * Discard all modifications registered in Handle and close Handle.
         * This invalidates all related unit/node handles.
         */
        public void close() {
            if(!this.closed) {
                if(this.analysisContext != null) {
                    this.analysisContext.close();
                    this.analysisContext = null;
                }

                if(ImageInfo.inImageCode()) {
                    NI_LIB.lkt_rewriting_abort_rewriting(this.unwrap());
                } else {
                    JNI_LIB.lkt_rewriting_abort_rewriting(this);
                }

                checkException();
                this.closed = true;
            }
        }

    }

    /**
     * Handle for the process of rewriting an analysis unit. Such handles are
     * owned by a Rewriting_Handle instance.
     */
    public static final class RewritingUnit {

        // ----- Class methods -----

        /** Singleton representing the none value for the rewriting unit. */
        public static final RewritingUnit NONE = new RewritingUnit(
            PointerWrapper.nullPointer()
        );

        // ----- Instance methods -----

        /** Reference to the native rewriting unit. */
        private final PointerWrapper reference;

        /** Cache for the analysis unit associated to the rewriting unit. */
        private AnalysisUnit analysisUnit;

        /** Cache for the unit root rewriting node. */
        private RewritingNode root;

        // ----- Constructors -----

        /** Create a new rewriting unit with its native reference. */
        RewritingUnit(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /** Wrap the given pointer to the native rewriting unit. */
        static RewritingUnit wrap(
            final Pointer pointer
        ) {
            return wrap((RewritingUnitNative) pointer.readWord(0));
        }

        /** Wrap the native rewriting unit. */
        static RewritingUnit wrap(
            final RewritingUnitNative rewritingUnitNative
        ) {
            return new RewritingUnit(new PointerWrapper(rewritingUnitNative));
        }

        /** Unwrap the rewriting unit in the given pointer. */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /** Unwrap the rewriting unit and return its native value. */
        RewritingUnitNative unwrap() {
            return (RewritingUnitNative) this.reference.ni();
        }

        // ----- Getters -----

        /**
         * Return the unit corresponding to Handle
         */
        public AnalysisUnit getAnalysisUnit() {
            if(this.analysisUnit == null) {

                if(ImageInfo.inImageCode()) {
                    this.analysisUnit = AnalysisUnit.wrap(
                        NI_LIB.lkt_rewriting_handle_to_unit(
                            this.unwrap()
                        )
                    );
                } else {
                    this.analysisUnit =
                        JNI_LIB.lkt_rewriting_handle_to_unit(this);
                }

            }
            return this.analysisUnit;
        }

        /**
         * Return the node handle corresponding to the root of the unit which
         * Handle designates.
         */
        public RewritingNode getRoot() {
            if(this.root == null) {

                if(ImageInfo.inImageCode()) {
                    this.root = RewritingNode.wrap(
                        NI_LIB.lkt_rewriting_unit_root(this.unwrap())
                    );
                } else {
                    this.root = JNI_LIB.lkt_rewriting_unit_root(this);
                }

            }
            return this.root;
        }

        // ----- Setters -----

        /**
         * Set the root node for the unit Handle to Root. This unties the
         * previous root handle. If Root is not No_Node_Rewriting_Handle, this
         * also ties Root to Handle.
         *
         * Root must not already be tied to another analysis unit handle.
         */
        public void setRoot(
            final RewritingNode root
        ) {
            this.root = root;

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_unit_set_root(
                    this.unwrap(),
                    root.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_unit_set_root(this, root);
            }
        }

        // ----- Instance methods -----

        /**
         * Return the text associated to the given unit.
         */
        public String unparse() {
            final Text unparseText;

            if(ImageInfo.inImageCode()) {
                final TextNative textNative = StackValue.get(
                    TextNative.class
                );
                NI_LIB.lkt_rewriting_unit_unparse(
                    this.unwrap(),
                    textNative
                );
                unparseText = Text.wrap(textNative);
            } else {
                unparseText = JNI_LIB.lkt_rewriting_unit_unparse(this);
            }

            String res = unparseText.getContent();
            unparseText.close();
            return res;
        }

        // ----- Override methods -----

        @Override
        public boolean equals(Object o) {
            if(this == o) return true;
            if(!(o instanceof RewritingUnit)) return false;
            final RewritingUnit other = (RewritingUnit) o;
            return this.reference.equals(other.reference);
        }

    }

    /**
     * Handle for the process of rewriting an AST node. Such handles are owned
     * by a Rewriting_Handle instance.
     */
    public static final class RewritingNode {

        // ----- Class attributes -----

        /** Singleton representing the none value for rewriting node. */
        public static final RewritingNode NONE = new RewritingNode(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the native rewriting node. */
        private final PointerWrapper reference;

        /** Kind of the rewriting node. */
        private NodeKind kind;

        /** Cache for the associated parsed node. */
        private LktNode parsedNode;

        /** Cache for the rewriting context containing this node. */
        private RewritingContext rewritingContext;

        // ----- Constructors -----

        /** Create a new rewriting node with its native reference. */
        RewritingNode(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API -----

        /** Wrap a pointer to a native rewriting node. */
        static RewritingNode wrap(
            final Pointer pointer
        ) {
            return wrap((RewritingNodeNative) pointer.readWord(0));
        }

        /** Wrap the native rewriting node. */
        static RewritingNode wrap(
            final RewritingNodeNative rewritingNodeNative
        ) {
            return new RewritingNode(new PointerWrapper(rewritingNodeNative));
        }

        /** Unwrap the rewriting node in the given pointer. */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /** Unwrap the rewriting node and return its native value. */
        RewritingNodeNative unwrap() {
            return (RewritingNodeNative) this.reference.ni();
        }

        // ----- Getters -----

        /**
         * Return the kind corresponding to Handle's node
         */
        public NodeKind getKind() {
            if(this.kind == null) {
                final int kindNative;

                if(ImageInfo.inImageCode()) {
                    kindNative = NI_LIB.lkt_rewriting_kind(
                        this.unwrap()
                    );
                } else {
                    kindNative = JNI_LIB.lkt_rewriting_kind(this);
                }

                this.kind = NodeKind.fromC(kindNative);
            }
            return this.kind;
        }

        /**
         * Return the node which the given rewriting Handle relates to. This
         * can be the null entity if this handle designates a new node.
         */
        public LktNode getParsedNode() {
            if(this.parsedNode == null) {
                final Entity nodeEntity;

                if(ImageInfo.inImageCode()) {
                    final Pointer nodeNative =
                        NI_LIB.lkt_rewriting_handle_to_node(
                            this.unwrap()
                        );
                    final EntityNative entityNative = StackValue.get(
                        EntityNative.class
                    );
                    NI_LIB.lkt_create_bare_entity(
                        nodeNative,
                        entityNative
                    );
                    nodeEntity = Entity.wrap(entityNative);
                } else {
                    nodeEntity = JNI_LIB.lkt_rewriting_handle_to_node(
                        this
                    );
                }

                this.parsedNode = LktNode.fromEntity(nodeEntity);
            }
            return this.parsedNode;
        }

        /**
         * Return a handle for the rewriting context to which Handle belongs
         */
        public RewritingContext getRewritingContext() {
            if(this.rewritingContext == null) {

                if(ImageInfo.inImageCode()) {
                    this.rewritingContext = RewritingContext.wrap(
                        NI_LIB.lkt_rewriting_node_to_context(
                            this.unwrap()
                        )
                    );
                } else {
                    this.rewritingContext =
                        JNI_LIB.lkt_rewriting_node_to_context(this);
                }

            }
            return this.rewritingContext;
        }

        /** Get whether the rewriting node is a none node. */
        public boolean isNone() {
            return this.reference.isNull();
        }

        // ----- Instance methods -----

        /**
         * Create a clone of the Handle node tree. The result is not tied to
         * any analysis unit tree.
         */
        public RewritingNode clone() {

            if(ImageInfo.inImageCode()) {
                return RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_clone(this.unwrap())
                );
            } else {
                return JNI_LIB.lkt_rewriting_clone(this);
            }

        }

        /**
         * Turn the given rewritten node Handles designates into text. This is
         * the text that is used in Apply in order to re-create an analysis
         * unit.
         */
        public String unparse() {
            final Text text;

            if(ImageInfo.inImageCode()) {
                TextNative textNative = StackValue.get(TextNative.class);
                NI_LIB.lkt_rewriting_node_unparse(
                    this.unwrap(),
                    textNative
                );
                text = Text.wrap(textNative);
            } else {
                text = JNI_LIB.lkt_rewriting_node_unparse(this);
            }

            final String res = text.getContent();
            text.close();
            return res;
        }

        /**
         * Return a representation of ``Handle`` as a string.
         */
        public String image() {
            final Text text;

            if(ImageInfo.inImageCode()) {
                TextNative textNative = StackValue.get(TextNative.class);
                NI_LIB.lkt_rewriting_node_image(
                    this.unwrap(),
                    textNative
                );
                text = Text.wrap(textNative);
            } else {
                text = JNI_LIB.lkt_rewriting_node_image(this);
            }

            final String res = text.getContent();
            text.close();
            return res;
        }

        /**
         * Return whether this node handle is tied to an analysis unit. If it
         * is not, it can be passed as the Child parameter to Set_Child.
         */
        public boolean isTied() {

            if(ImageInfo.inImageCode()) {
                return NI_LIB.lkt_rewriting_tied(
                    this.unwrap()
                ) > 0;
            } else {
                return JNI_LIB.lkt_rewriting_tied(this);
            }

        }

        /**
         * Return a handle for the node that is the parent of Handle's node.
         * This is ``No_Rewriting_Handle`` for a node that is not tied to any
         * tree yet.
         */
        public RewritingNode parent() {

            if(ImageInfo.inImageCode()) {
                return RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_parent(
                        this.unwrap()
                    )
                );
            } else {
                return JNI_LIB.lkt_rewriting_parent(this);
            }

        }

        /**
         * Return the list of children for ``Handle``.
         */
        public RewritingNode[] children() {

            if(ImageInfo.inImageCode()) {
                // Call the native function
                final WordPointer childrenPointer =
                    StackValue.get(WordPointer.class);
                final CIntPointer countPointer =
                    StackValue.get(CIntPointer.class);
                NI_LIB.lkt_rewriting_children(
                    this.unwrap(),
                    childrenPointer,
                    countPointer
                );
                final WordPointer children = childrenPointer.read();
                final int count = countPointer.read();

                // Create the Java array and fill it
                final RewritingNode[] res = new RewritingNode[count];
                for(int i = 0; i < res.length; i++) {
                    res[i] = RewritingNode.wrap(
                        (RewritingNodeNative) children.read(i)
                    );
                }

                // Free the native children
                NI_LIB.lkt_free(children);

                // Return the result
                return res;
            } else {
                return JNI_LIB.lkt_rewriting_children(this);
            }

        }

        /**
         * Return the node that is in the syntax ``Field`` for ``Handle``
         */
        public RewritingNode getChild(
            final MemberReference childMember
        ) {
            final RewritingNode res;

            if(ImageInfo.inImageCode()) {
                res = RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_child(
                        this.unwrap(),
                        childMember.toC()
                    )
                );
            } else {
                res = JNI_LIB.lkt_rewriting_child(
                    this,
                    childMember.toC()
                );
            }

            checkException();
            return res;
        }

        /**
         * If ``Child`` is ``No_Rewriting_Node``, untie the syntax field in
         * ``Handle`` corresponding to ``Field``, so it can be attached to
         * another one. Otherwise, ``Child`` must have no parent as it will be
         * tied to ``Handle``'s tree.
         */
        public void setChild(
            MemberReference childMember,
            RewritingNode child
        ) {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_set_child(
                    this.unwrap(),
                    childMember.toC(),
                    child.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_set_child(
                    this,
                    childMember.toC(),
                    child
                );
            }
            checkException();

        }

        /**
         * If Handle is the root of an analysis unit, untie it and set New_Node
         * as its new root. Otherwise, replace Handle with New_Node in Handle's
         * parent node.
         *
         * Note that: * Handle must be tied to an existing analysis unit
         * handle. * New_Node must not already be tied to another analysis unit
         * handle.
         */
        public void replace(
            final RewritingNode newNode
        ) {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_replace(
                    this.unwrap(),
                    newNode.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_replace(this, newNode);
            }
            checkException();

        }

        /**
         * Assuming ``Handle`` refers to a list node, return a handle to its
         * first child, or ``No_Node_Rewriting_Handle``` if it has no child
         * node.
         */
        public RewritingNode firstChild() {

            if(ImageInfo.inImageCode()) {
                return RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_first_child(
                        this.unwrap()
                    )
                );
            } else {
                return JNI_LIB.lkt_rewriting_first_child(this);
            }

        }

        /**
         * Assuming ``Handle`` refers to a list node, return a handle to its
         * last child, or ``No_Node_Rewriting_Handle``` if it has no child
         * node.
         */
        public RewritingNode lastChild() {

            if(ImageInfo.inImageCode()) {
                return RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_last_child(
                        this.unwrap()
                    )
                );
            } else {
                return JNI_LIB.lkt_rewriting_last_child(this);
            }

        }

        /**
         * Assuming ``Handle`` refers to the child of a list node, return a
         * handle to its next sibling, or ``No_Node_Rewriting_Handle``` if it
         * is the last sibling.
         */
        public RewritingNode nextChild() {

            if(ImageInfo.inImageCode()) {
                return RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_next_child(
                        this.unwrap()
                    )
                );
            } else {
                return JNI_LIB.lkt_rewriting_next_child(this);
            }

        }

        /**
         * Assuming ``Handle`` refers to the child of a list node, return a
         * handle to its previous sibling, or ``No_Node_Rewriting_Handle``` if
         * it is the first sibling.
         */
        public RewritingNode previousChild() {

            if(ImageInfo.inImageCode()) {
                return RewritingNode.wrap(
                    NI_LIB.lkt_rewriting_previous_child(
                        this.unwrap()
                    )
                );
            } else {
                return JNI_LIB.lkt_rewriting_previous_child(this);
            }

        }

        /**
         * Assuming ``Handle`` refers to the child of a list node, insert
         * ``New_Sibling`` as a new child in this list, right before
         * ``Handle``.
         */
        public void insertBefore(
            final RewritingNode toInsert
        ) {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_insert_before(
                    this.unwrap(),
                    toInsert.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_insert_before(this, toInsert);
            }
            checkException();

        }

        /**
         * Assuming ``Handle`` refers to the child of a list node, insert
         * ``New_Sibling`` as a new child in this list, right before
         * ``Handle``.
         */
        public void insertAfter(
            final RewritingNode toInsert
        ) {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_insert_after(
                    this.unwrap(),
                    toInsert.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_insert_after(this, toInsert);
            }
            checkException();

        }

        /**
         * Assuming ``Handle`` refers to a list node, insert ``New_Child`` to
         * be the first child in this list.
         */
        public void insertFirst(
            final RewritingNode toInsert
        ) {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_insert_first(
                    this.unwrap(),
                    toInsert.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_insert_first(this, toInsert);
            }
            checkException();

        }

        /**
         * Assuming ``Handle`` refers to a list node, insert ``New_Child`` to
         * be the last child in this list.
         */
        public void insertLast(
            final RewritingNode toInsert
        ) {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_insert_last(
                    this.unwrap(),
                    toInsert.unwrap()
                );
            } else {
                JNI_LIB.lkt_rewriting_insert_last(this, toInsert);
            }
            checkException();

        }

        /**
         * Assuming Handle refers to the child of a list node, remove it from
         * that list.
         */
        public void removeFromParent() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.lkt_rewriting_remove_child(this.unwrap());
            } else {
                JNI_LIB.lkt_rewriting_remove_child(this);
            }
            checkException();

        }

        /** Get the text of the token node. */
        public String getText() {
            final Text resultText;

            if(ImageInfo.inImageCode()) {
                final TextNative textNative = StackValue.get(
                    TextNative.class
                );
                NI_LIB.lkt_rewriting_text(
                    this.unwrap(),
                    textNative
                );
                resultText = Text.wrap(textNative);
            } else {
                resultText = JNI_LIB.lkt_rewriting_text(this);
            }

            final String res = resultText.getContent();
            resultText.close();
            return res;
        }

        /**
         * Override text associated to the given token node.
         */
        public void setText(
            final String text
        ) {
            try (
                final Text nodeText = Text.create(text);
            ) {

                if(ImageInfo.inImageCode()) {
                    final TextNative textNative = StackValue.get(
                        TextNative.class
                    );
                    nodeText.unwrap(textNative);
                    NI_LIB.lkt_rewriting_set_text(
                        this.unwrap(),
                        textNative
                    );
                } else {
                    JNI_LIB.lkt_rewriting_set_text(this, nodeText);
                }
                checkException();

            }
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.image();
        }

        @Override
        public boolean equals(Object o) {
            if(o == this) return true;
            if(!(o instanceof RewritingNode)) return false;
            final RewritingNode other = (RewritingNode) o;
            return this.reference.equals(other.reference);
        }

    }

    // ===== Generated structure wrapping classes =====

        
    
    

    /**
     * Result for ``CharLit.p_denoted_value``.
     *
     * If that property is successful, set ``has_error`` to false and ``value``
     * to the decoded character value. Otherwise, set ``has_error`` to true and
     * ``error_sloc`` and ``error_message`` to give information about the
     * decoding failure.
     */
    public static final class DecodedCharValue {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final DecodedCharValue NONE = new DecodedCharValue(
            Char.NONE,false,SourceLocation.NONE,StringWrapper.NONE
        );

        // ----- Instance attributes -----

        public final
        Char
        value;
        public final
        boolean
        hasError;
        public final
        SourceLocation
        errorSloc;
        public final
        String
        errorMessage;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        DecodedCharValue(
            final Char value,final boolean hasError,final SourceLocation errorSloc,final String errorMessage
        ) {
            this.value = value;
            this.hasError = hasError;
            this.errorSloc = errorSloc;
            this.errorMessage = errorMessage;
        }

        /**
         * Create a new structure with the field values.
         */
        public static DecodedCharValue create(
            final Char value,final boolean hasError,final SourceLocation errorSloc,final String errorMessage
        ) {
            return new DecodedCharValue(
                value,hasError,errorSloc,errorMessage
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static DecodedCharValue wrap(
            final WordPointer pointer
        ) {
            return wrap((DecodedCharValueNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static DecodedCharValue wrap(
            final DecodedCharValueNative structNative
        ) {
            return new DecodedCharValue(
                Char.wrap(structNative.get_value()),BooleanWrapper.wrap(structNative.get_has_error()),SourceLocation.wrap(structNative.get_error_sloc()),StringWrapper.wrap(structNative.get_error_message())
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final DecodedCharValueNative structNative
        ) {
            CIntPointer valueNative = structNative.address_value();valueNative.write(this.value.value);
            CCharPointer hasErrorNative = structNative.address_has_error();hasErrorNative.write(this.hasError ? (byte) 1 : (byte) 0);
            SourceLocationNative errorSlocNative = structNative.address_error_sloc();this.errorSloc.unwrap(errorSlocNative);
            WordPointer errorMessageNative = structNative.address_error_message();StringWrapper.unwrap(this.errorMessage, errorMessageNative);
        }

        /**
         * Release the structure.
         *
         * @param structNative The native structure to release.
         */
        static void release(DecodedCharValueNative structNative) {
            NI_LIB.lkt_internal_decoded_char_value_dec_ref(structNative);
        }

    }

        
    
    

    /**
     * Result for ``StringLit.p_denoted_value``.
     *
     * If that property is successful, set ``has_error`` to false and ``value``
     * to the decoded string value. Otherwise, set ``has_error`` to true and
     * ``error_sloc`` and ``error_message`` to give information about the
     * decoding failure.
     */
    public static final class DecodedStringValue {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final DecodedStringValue NONE = new DecodedStringValue(
            StringWrapper.NONE,false,SourceLocation.NONE,StringWrapper.NONE
        );

        // ----- Instance attributes -----

        public final
        String
        value;
        public final
        boolean
        hasError;
        public final
        SourceLocation
        errorSloc;
        public final
        String
        errorMessage;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        DecodedStringValue(
            final String value,final boolean hasError,final SourceLocation errorSloc,final String errorMessage
        ) {
            this.value = value;
            this.hasError = hasError;
            this.errorSloc = errorSloc;
            this.errorMessage = errorMessage;
        }

        /**
         * Create a new structure with the field values.
         */
        public static DecodedStringValue create(
            final String value,final boolean hasError,final SourceLocation errorSloc,final String errorMessage
        ) {
            return new DecodedStringValue(
                value,hasError,errorSloc,errorMessage
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static DecodedStringValue wrap(
            final WordPointer pointer
        ) {
            return wrap((DecodedStringValueNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static DecodedStringValue wrap(
            final DecodedStringValueNative structNative
        ) {
            return new DecodedStringValue(
                StringWrapper.wrap(structNative.get_value()),BooleanWrapper.wrap(structNative.get_has_error()),SourceLocation.wrap(structNative.get_error_sloc()),StringWrapper.wrap(structNative.get_error_message())
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final DecodedStringValueNative structNative
        ) {
            WordPointer valueNative = structNative.address_value();StringWrapper.unwrap(this.value, valueNative);
            CCharPointer hasErrorNative = structNative.address_has_error();hasErrorNative.write(this.hasError ? (byte) 1 : (byte) 0);
            SourceLocationNative errorSlocNative = structNative.address_error_sloc();this.errorSloc.unwrap(errorSlocNative);
            WordPointer errorMessageNative = structNative.address_error_message();StringWrapper.unwrap(this.errorMessage, errorMessageNative);
        }

        /**
         * Release the structure.
         *
         * @param structNative The native structure to release.
         */
        static void release(DecodedStringValueNative structNative) {
            NI_LIB.lkt_internal_decoded_string_value_dec_ref(structNative);
        }

    }

        
        
    
    

    public static final class Metadata {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final Metadata NONE = new Metadata();

        // ----- Instance attributes ------

        /** The dummy field is always false (0) */
        final boolean dummy = false;

        // ----- Constructors -----

        /**
         * An empty constructor because the structure is empty
         */
        Metadata() {}

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The None instance because the structure is empty.
         */
        static Metadata wrap(
            final WordPointer pointer
        ) {
            return Metadata.NONE;
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final MetadataNative structNative
        ) {
            // Do nothing because the dummy field is useless
        }


    }

        
    
    

    /**

     */
    public static final class EntityInfo {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final EntityInfo NONE = new EntityInfo(
            Metadata.NONE,PointerWrapper.nullPointer(),false
        );

        // ----- Instance attributes -----

        public final
        Metadata
        md;
        public final
        PointerWrapper
        rebindings;
        public final
        boolean
        fromRebound;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        EntityInfo(
            final Metadata md,final PointerWrapper rebindings,final boolean fromRebound
        ) {
            this.md = md;
            this.rebindings = rebindings;
            this.fromRebound = fromRebound;
        }

        /**
         * Create a new structure with the field values.
         */
        public static EntityInfo create(
            final Metadata md,final PointerWrapper rebindings,final boolean fromRebound
        ) {
            return new EntityInfo(
                md,rebindings,fromRebound
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static EntityInfo wrap(
            final WordPointer pointer
        ) {
            return wrap((EntityInfoNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static EntityInfo wrap(
            final EntityInfoNative structNative
        ) {
            return new EntityInfo(
                Metadata.NONE,PointerWrapper.wrap(structNative.get_rebindings()),BooleanWrapper.wrap(structNative.get_from_rebound())
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final EntityInfoNative structNative
        ) {
            MetadataNative mdNative = structNative.address_md();this.md.unwrap(mdNative);
            Pointer rebindingsNative = structNative.address_rebindings();rebindingsNative.writeWord(0, this.rebindings.ni());
            CCharPointer fromReboundNative = structNative.address_from_rebound();fromReboundNative.write(this.fromRebound ? (byte) 1 : (byte) 0);
        }


    }

    
    

    /**

     */
    public static final class Entity {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final Entity NONE = new Entity(
            PointerWrapper.nullPointer(),EntityInfo.NONE
        );

        // ----- Instance attributes -----

        public final
        PointerWrapper
        node;
        public final
        EntityInfo
        info;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        Entity(
            final PointerWrapper node,final EntityInfo info
        ) {
            this.node = node;
            this.info = info;
        }

        /**
         * Create a new structure with the field values.
         */
        public static Entity create(
            final PointerWrapper node,final EntityInfo info
        ) {
            return new Entity(
                node,info
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static Entity wrap(
            final WordPointer pointer
        ) {
            return wrap((EntityNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static Entity wrap(
            final EntityNative structNative
        ) {
            return new Entity(
                PointerWrapper.wrap(structNative.get_node()),EntityInfo.wrap(structNative.address_info())
            );
        }

        /**
         * Special wrapping method to construct a new entity structure
         * from a native bare node pointer.
         */
        static Entity wrapBareNode(
            final Pointer bareNode
        ) {
            return new Entity(
                PointerWrapper.wrap(bareNode),EntityInfo.NONE
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final EntityNative structNative
        ) {
            Pointer nodeNative = structNative.address_node();nodeNative.writeWord(0, this.node.ni());
            EntityInfoNative infoNative = structNative.address_info();this.info.unwrap(infoNative);
        }


    }

        
        
        
        
        
    
    

    /**
     * Result for a call to a semantic property that can return an error.
     *
     * In every case, the node field will be populated with the node upon which
     * the request was made. Then, the result can be either:
     *
     * * ``result_type`` if the property was a type returning property.
     *
     * * ``result_decl`` if the property was a decl returning property
     *
     * * ``error_message`` if an error was found as part of the resolution
     *   process.
     *
     * Only one of those fields can have a value.
     *
     * If all fields are null, it means that ``expr_type`` has been called on a
     * non regular expression.
     *
     * TODO: Turn this into a real variant record when we have variants.
     */
    public static final class SemanticResult {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final SemanticResult NONE = new SemanticResult(
            LktNode.NONE,TypeDecl.NONE,Decl.NONE,false,StringWrapper.NONE,false
        );

        // ----- Instance attributes -----

        public final
        LktNode
        node;
        public final
        TypeDecl
        resultType;
        public final
        Decl
        resultDecl;
        public final
        boolean
        hasError;
        public final
        String
        errorMessage;
        public final
        boolean
        exemptAnalysis;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        SemanticResult(
            final LktNode node,final TypeDecl resultType,final Decl resultDecl,final boolean hasError,final String errorMessage,final boolean exemptAnalysis
        ) {
            this.node = node;
            this.resultType = resultType;
            this.resultDecl = resultDecl;
            this.hasError = hasError;
            this.errorMessage = errorMessage;
            this.exemptAnalysis = exemptAnalysis;
        }

        /**
         * Create a new structure with the field values.
         */
        public static SemanticResult create(
            final LktNode node,final TypeDecl resultType,final Decl resultDecl,final boolean hasError,final String errorMessage,final boolean exemptAnalysis
        ) {
            return new SemanticResult(
                node,resultType,resultDecl,hasError,errorMessage,exemptAnalysis
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static SemanticResult wrap(
            final WordPointer pointer
        ) {
            return wrap((SemanticResultNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static SemanticResult wrap(
            final SemanticResultNative structNative
        ) {
            return new SemanticResult(
                LktNode.fromEntity(Entity.wrapBareNode(structNative.get_node())),TypeDecl.fromEntity(Entity.wrap(structNative.address_result_type())),Decl.fromEntity(Entity.wrap(structNative.address_result_decl())),BooleanWrapper.wrap(structNative.get_has_error()),StringWrapper.wrap(structNative.get_error_message()),BooleanWrapper.wrap(structNative.get_exempt_analysis())
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final SemanticResultNative structNative
        ) {
            Pointer nodeNative = structNative.address_node();nodeNative.writeWord(0, this.node.entity.node.ni());
            EntityNative resultTypeNative = structNative.address_result_type();this.resultType.unwrap(resultTypeNative);
            EntityNative resultDeclNative = structNative.address_result_decl();this.resultDecl.unwrap(resultDeclNative);
            CCharPointer hasErrorNative = structNative.address_has_error();hasErrorNative.write(this.hasError ? (byte) 1 : (byte) 0);
            WordPointer errorMessageNative = structNative.address_error_message();StringWrapper.unwrap(this.errorMessage, errorMessageNative);
            CCharPointer exemptAnalysisNative = structNative.address_exempt_analysis();exemptAnalysisNative.write(this.exemptAnalysis ? (byte) 1 : (byte) 0);
        }

        /**
         * Release the structure.
         *
         * @param structNative The native structure to release.
         */
        static void release(SemanticResultNative structNative) {
            NI_LIB.lkt_internal_semantic_result_dec_ref(structNative);
        }

    }

        
        
        
        
        
        
        
    
    

    /**
     * Collection of semantic results for a subtree. This will carry both:
     *
     * * An array of semantic results for all of the subtree's nodes.
     *
     * * A flag indicating whether the subtree contained errors or not.
     */
    public static final class TreeSemanticResult {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final TreeSemanticResult NONE = new TreeSemanticResult(
            SemanticResultArrayWrapper.NONE,false
        );

        // ----- Instance attributes -----

        public final
        SemanticResult[]
        results;
        public final
        boolean
        hasError;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        TreeSemanticResult(
            final SemanticResult[] results,final boolean hasError
        ) {
            this.results = results;
            this.hasError = hasError;
        }

        /**
         * Create a new structure with the field values.
         */
        public static TreeSemanticResult create(
            final SemanticResult[] results,final boolean hasError
        ) {
            return new TreeSemanticResult(
                results,hasError
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static TreeSemanticResult wrap(
            final WordPointer pointer
        ) {
            return wrap((TreeSemanticResultNative) pointer.read());
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static TreeSemanticResult wrap(
            final TreeSemanticResultNative structNative
        ) {
            return new TreeSemanticResult(
                SemanticResultArrayWrapper.wrap(structNative.get_results()),BooleanWrapper.wrap(structNative.get_has_error())
            );
        }


        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final TreeSemanticResultNative structNative
        ) {
            WordPointer resultsNative = structNative.address_results();SemanticResultArrayWrapper.unwrap(this.results, resultsNative);
            CCharPointer hasErrorNative = structNative.address_has_error();hasErrorNative.write(this.hasError ? (byte) 1 : (byte) 0);
        }

        /**
         * Release the structure.
         *
         * @param structNative The native structure to release.
         */
        static void release(TreeSemanticResultNative structNative) {
            NI_LIB.lkt_internal_tree_semantic_result_dec_ref(structNative);
        }

    }


    // ===== Generated array utility classes =====

    
    

    /**
     * This class represents the lkt_node_array Java wrapping class
     */
    public static final class LktNodeArrayWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final LktNode[] NONE = new LktNode[0];

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static LktNode[] wrap(
            final WordPointer pointer
        ) {
            return wrap((LktNodeArrayNative) pointer.read());
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static LktNode[] wrap(
            final LktNodeArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final LktNode[] content = new LktNode[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            EntityNative toRead;

            // Iterate over all array elements
            final int elemSize = SizeOf.get(EntityNative.class);
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = LktNode.fromEntity(Entity.wrap(toRead));
            }

            return content;
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        static void unwrap(
            LktNode[] source,
            final WordPointer pointer
            
        ) {
            // Create a new native array with the size
            final LktNodeArrayNative resNative = unwrap(
                source
                
            );

            // Place the result in the pointer
            pointer.write(resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        static LktNodeArrayNative unwrap(
            LktNode[] source
            
        ) {
            // Create a new native array with the size
            final LktNodeArrayNative res = NI_LIB.lkt_node_array_create(
                source.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            EntityNative toWrite;

            // Place all elements in the native array
            final int elemSize = SizeOf.get(EntityNative.class);
            for(int i = 0 ; i < source.length ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toWrite = WordFactory.unsigned(nativeItem.rawValue());
                source[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((LktNodeArrayNative) pointer.read());
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final LktNodeArrayNative arrayNative
        ) {
            NI_LIB.lkt_node_array_dec_ref(arrayNative);
        }

        // ----- JNI methods -----

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static LktNode[] jniWrap(
            final Entity[] jniContent
        ) {
            final LktNode[] content =
                new LktNode[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    LktNode.fromEntity(jniContent[i]);
            }
            return content;
        }

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private static Entity[] jniUnwrap(LktNode[] content) {
            final Entity[] res =
                new Entity[content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = (content[i] != null ? content[i].entity : null);
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the lkt_node_array Java wrapping class
     */
    public static final class TypeDeclArrayWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final TypeDecl[] NONE = new TypeDecl[0];

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static TypeDecl[] wrap(
            final WordPointer pointer
        ) {
            return wrap((LktNodeArrayNative) pointer.read());
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static TypeDecl[] wrap(
            final LktNodeArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final TypeDecl[] content = new TypeDecl[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            EntityNative toRead;

            // Iterate over all array elements
            final int elemSize = SizeOf.get(EntityNative.class);
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = TypeDecl.fromEntity(Entity.wrap(toRead));
            }

            return content;
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        static void unwrap(
            TypeDecl[] source,
            final WordPointer pointer
            
        ) {
            // Create a new native array with the size
            final LktNodeArrayNative resNative = unwrap(
                source
                
            );

            // Place the result in the pointer
            pointer.write(resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        static LktNodeArrayNative unwrap(
            TypeDecl[] source
            
        ) {
            // Create a new native array with the size
            final LktNodeArrayNative res = NI_LIB.lkt_node_array_create(
                source.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            EntityNative toWrite;

            // Place all elements in the native array
            final int elemSize = SizeOf.get(EntityNative.class);
            for(int i = 0 ; i < source.length ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toWrite = WordFactory.unsigned(nativeItem.rawValue());
                source[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((LktNodeArrayNative) pointer.read());
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final LktNodeArrayNative arrayNative
        ) {
            NI_LIB.lkt_node_array_dec_ref(arrayNative);
        }

        // ----- JNI methods -----

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static TypeDecl[] jniWrap(
            final Entity[] jniContent
        ) {
            final TypeDecl[] content =
                new TypeDecl[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    TypeDecl.fromEntity(jniContent[i]);
            }
            return content;
        }

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private static Entity[] jniUnwrap(TypeDecl[] content) {
            final Entity[] res =
                new Entity[content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = (content[i] != null ? content[i].entity : null);
            }
            return res;
        }

    }

    
    

    /**
     * This class represents the lkt_internal_semantic_result_array Java wrapping class
     */
    public static final class SemanticResultArrayWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final SemanticResult[] NONE = new SemanticResult[0];

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static SemanticResult[] wrap(
            final WordPointer pointer
        ) {
            return wrap((SemanticResultArrayNative) pointer.read());
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static SemanticResult[] wrap(
            final SemanticResultArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final SemanticResult[] content = new SemanticResult[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            SemanticResultNative toRead;

            // Iterate over all array elements
            final int elemSize = SizeOf.get(SemanticResultNative.class);
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = SemanticResult.wrap(toRead);
            }

            return content;
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        static void unwrap(
            SemanticResult[] source,
            final WordPointer pointer
            
        ) {
            // Create a new native array with the size
            final SemanticResultArrayNative resNative = unwrap(
                source
                
            );

            // Place the result in the pointer
            pointer.write(resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        static SemanticResultArrayNative unwrap(
            SemanticResult[] source
            
        ) {
            // Create a new native array with the size
            final SemanticResultArrayNative res = NI_LIB.lkt_internal_semantic_result_array_create(
                source.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            SemanticResultNative toWrite;

            // Place all elements in the native array
            final int elemSize = SizeOf.get(SemanticResultNative.class);
            for(int i = 0 ; i < source.length ; i++) {
                nativeItem = nativeItems.add(i * elemSize);
                toWrite = WordFactory.unsigned(nativeItem.rawValue());
                source[i].unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final WordPointer pointer
        ) {
            release((SemanticResultArrayNative) pointer.read());
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final SemanticResultArrayNative arrayNative
        ) {
            NI_LIB.lkt_internal_semantic_result_array_dec_ref(arrayNative);
        }

        // ----- JNI methods -----

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static SemanticResult[] jniWrap(
            final SemanticResult[] jniContent
        ) {
            final SemanticResult[] content =
                new SemanticResult[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    jniContent[i];
            }
            return content;
        }

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private static SemanticResult[] jniUnwrap(SemanticResult[] content) {
            final SemanticResult[] res =
                new SemanticResult[content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = content[i];
            }
            return res;
        }

    }


    // ===== Generated iterator wrapping classes =====


    // ===== Node classes =====

    /**
     * Data type for all nodes. Nodes are assembled to make up a tree.  See the
     * node primitives below to inspect such trees.
     *
     * Unlike for contexts and units, this type has weak-reference semantics:
     * keeping a reference to a node has no effect on the decision to keep the
     * unit that it owns allocated. This means that once all references to the
     * context and units related to a node are dropped, the context and its
     * units are deallocated and the node becomes a stale reference: most
     * operations on it will raise a ``Stale_Reference_Error``.
     *
     * Note that since reparsing an analysis unit deallocates all the nodes it
     * contains, this operation makes all reference to these nodes stale as
     * well.
     */
    public static abstract class LktNode {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                LktNode.class,
                "LktNode",
                new String[] {
                    
                },
                new HashMap<>(
                    
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pNodeGenTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_node_gen_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_NODE_GEN_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pNodeTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_node_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_NODE_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pTokenNodeTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_token_node_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_TOKEN_NODE_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pErrorNodeTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_error_node_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ERROR_NODE_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pCharType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_char_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_CHAR_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pIntType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_int_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_INT_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pBoolType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_bool_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_BOOL_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pBigintType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_bigint_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_BIGINT_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pStringType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_string_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_STRING_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pSymbolType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_symbol_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_SYMBOL_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pPropertyErrorType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_property_error_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_PROPERTY_ERROR_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pRegexpType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_regexp_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_REGEXP_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pArrayGenType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_array_gen_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ARRAY_GEN_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pArrayType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_array_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ARRAY_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pAstlistGenType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_astlist_gen_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ASTLIST_GEN_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pAstlistType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_astlist_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ASTLIST_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pIteratorGenTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_iterator_gen_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ITERATOR_GEN_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pIteratorTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_iterator_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ITERATOR_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pAnalysisUnitGenTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_analysis_unit_gen_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ANALYSIS_UNIT_GEN_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pAnalysisUnitTrait",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_analysis_unit_trait",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_ANALYSIS_UNIT_TRAIT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "pTopmostInvalidDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_topmost_invalid_decl",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LKT_NODE_P_TOPMOST_INVALID_DECL
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "parent",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "parent",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_PARENT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "parents",
                        new Class[]{boolean.class}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();
                    parameters.add(new Reflection.Param(
                        boolean.class,
                        "withSelf",
                        true
                    ));

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "parents",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_PARENTS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "children",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "children",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CHILDREN
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "tokenStart",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "token_start",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TOKEN_START
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "tokenEnd",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "token_end",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TOKEN_END
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "childIndex",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "child_index",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CHILD_INDEX
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "previousSibling",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "previous_sibling",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_PREVIOUS_SIBLING
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "nextSibling",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "next_sibling",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_NEXT_SIBLING
                        )
                    );
                }
                
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "isGhost",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "is_ghost",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_IS_GHOST
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = LktNode.class.getMethod(
                        "fullSlocImage",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "full_sloc_image",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FULL_SLOC_IMAGE
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LktNode NONE = new NoneNode();

        /** The entity of the node. */
        public final Entity entity;

        /** Cache for the associated rewriting node. */
        protected RewritingNode rewritingNode;

        /** The analysis unit that owns the node. */
        protected AnalysisUnit unit;

        /** The cache for the image of the node. */
        protected String image;

        // ----- Constructors -----

        /**
         * Create a new node with its entity.
         *
         * @param entity The node's entity.
         */
        protected LktNode(
            final Entity entity
        ) {
            this.entity = entity;
            this.unit = null;
            this.image = null;
        }

        /**
         * Get a node from the given entity.
         *
         * @param entity The entity to get the node from.
         * @return The newly created node.
         */
        public static LktNode fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LktNode.NONE :
                dispatchNodeCreation(entity);
        }

        /**
         * Dispatch the node creation to return the valid Java object
         * according to the node kind.
         *
         * @param entity The entity to create the node from.
         * @return The wrapped node in the correct class.
         */
        protected static LktNode dispatchNodeCreation(
            final Entity entity
        ) {
            int nodeKind = -1;

            if(ImageInfo.inImageCode()) {
                EntityNative entityNative = StackValue.get(
                    EntityNative.class
                );
                entity.unwrap(entityNative);
                nodeKind = NI_LIB.lkt_node_kind(entityNative);
            } else {
                nodeKind = JNI_LIB.lkt_node_kind(entity);
            }

            switch(nodeKind) {
                case 1:
                    return entity.node.isNull() ?
                        LexerCaseRuleCondAlt.NONE :
                        new LexerCaseRuleCondAlt(entity);
                case 2:
                    return entity.node.isNull() ?
                        LexerCaseRuleDefaultAlt.NONE :
                        new LexerCaseRuleDefaultAlt(entity);
                case 3:
                    return entity.node.isNull() ?
                        BlockStringLine.NONE :
                        new BlockStringLine(entity);
                case 4:
                    return entity.node.isNull() ?
                        ClassQualifierAbsent.NONE :
                        new ClassQualifierAbsent(entity);
                case 5:
                    return entity.node.isNull() ?
                        ClassQualifierPresent.NONE :
                        new ClassQualifierPresent(entity);
                case 6:
                    return entity.node.isNull() ?
                        GrammarRuleDecl.NONE :
                        new GrammarRuleDecl(entity);
                case 7:
                    return entity.node.isNull() ?
                        SyntheticLexerDecl.NONE :
                        new SyntheticLexerDecl(entity);
                case 8:
                    return entity.node.isNull() ?
                        NodeDecl.NONE :
                        new NodeDecl(entity);
                case 9:
                    return entity.node.isNull() ?
                        SelfDecl.NONE :
                        new SelfDecl(entity);
                case 10:
                    return entity.node.isNull() ?
                        EnumLitDecl.NONE :
                        new EnumLitDecl(entity);
                case 11:
                    return entity.node.isNull() ?
                        FieldDecl.NONE :
                        new FieldDecl(entity);
                case 12:
                    return entity.node.isNull() ?
                        FunArgDecl.NONE :
                        new FunArgDecl(entity);
                case 13:
                    return entity.node.isNull() ?
                        LambdaArgDecl.NONE :
                        new LambdaArgDecl(entity);
                case 14:
                    return entity.node.isNull() ?
                        DynVarDecl.NONE :
                        new DynVarDecl(entity);
                case 15:
                    return entity.node.isNull() ?
                        MatchValDecl.NONE :
                        new MatchValDecl(entity);
                case 16:
                    return entity.node.isNull() ?
                        ValDecl.NONE :
                        new ValDecl(entity);
                case 17:
                    return entity.node.isNull() ?
                        FunDecl.NONE :
                        new FunDecl(entity);
                case 18:
                    return entity.node.isNull() ?
                        EnvSpecDecl.NONE :
                        new EnvSpecDecl(entity);
                case 19:
                    return entity.node.isNull() ?
                        GenericDecl.NONE :
                        new GenericDecl(entity);
                case 20:
                    return entity.node.isNull() ?
                        GrammarDecl.NONE :
                        new GrammarDecl(entity);
                case 21:
                    return entity.node.isNull() ?
                        LexerDecl.NONE :
                        new LexerDecl(entity);
                case 22:
                    return entity.node.isNull() ?
                        LexerFamilyDecl.NONE :
                        new LexerFamilyDecl(entity);
                case 23:
                    return entity.node.isNull() ?
                        EnumClassAltDecl.NONE :
                        new EnumClassAltDecl(entity);
                case 24:
                    return entity.node.isNull() ?
                        FunctionType.NONE :
                        new FunctionType(entity);
                case 25:
                    return entity.node.isNull() ?
                        GenericFormalTypeDecl.NONE :
                        new GenericFormalTypeDecl(entity);
                case 26:
                    return entity.node.isNull() ?
                        InstantiatedGenericType.NONE :
                        new InstantiatedGenericType(entity);
                case 27:
                    return entity.node.isNull() ?
                        ClassDecl.NONE :
                        new ClassDecl(entity);
                case 28:
                    return entity.node.isNull() ?
                        EnumClassDecl.NONE :
                        new EnumClassDecl(entity);
                case 29:
                    return entity.node.isNull() ?
                        EnumTypeDecl.NONE :
                        new EnumTypeDecl(entity);
                case 30:
                    return entity.node.isNull() ?
                        StructDecl.NONE :
                        new StructDecl(entity);
                case 31:
                    return entity.node.isNull() ?
                        TraitDecl.NONE :
                        new TraitDecl(entity);
                case 32:
                    return entity.node.isNull() ?
                        DeclAnnotation.NONE :
                        new DeclAnnotation(entity);
                case 33:
                    return entity.node.isNull() ?
                        DeclAnnotationParams.NONE :
                        new DeclAnnotationParams(entity);
                case 34:
                    return entity.node.isNull() ?
                        ElsifBranch.NONE :
                        new ElsifBranch(entity);
                case 35:
                    return entity.node.isNull() ?
                        EnumClassCase.NONE :
                        new EnumClassCase(entity);
                case 36:
                    return entity.node.isNull() ?
                        ExcludesNullAbsent.NONE :
                        new ExcludesNullAbsent(entity);
                case 37:
                    return entity.node.isNull() ?
                        ExcludesNullPresent.NONE :
                        new ExcludesNullPresent(entity);
                case 38:
                    return entity.node.isNull() ?
                        AnyOf.NONE :
                        new AnyOf(entity);
                case 39:
                    return entity.node.isNull() ?
                        ArrayLiteral.NONE :
                        new ArrayLiteral(entity);
                case 40:
                    return entity.node.isNull() ?
                        DotExpr.NONE :
                        new DotExpr(entity);
                case 41:
                    return entity.node.isNull() ?
                        NullCondDottedName.NONE :
                        new NullCondDottedName(entity);
                case 42:
                    return entity.node.isNull() ?
                        BinOp.NONE :
                        new BinOp(entity);
                case 43:
                    return entity.node.isNull() ?
                        BlockExpr.NONE :
                        new BlockExpr(entity);
                case 44:
                    return entity.node.isNull() ?
                        CallExpr.NONE :
                        new CallExpr(entity);
                case 45:
                    return entity.node.isNull() ?
                        CastExpr.NONE :
                        new CastExpr(entity);
                case 46:
                    return entity.node.isNull() ?
                        ErrorOnNull.NONE :
                        new ErrorOnNull(entity);
                case 47:
                    return entity.node.isNull() ?
                        GenericInstantiation.NONE :
                        new GenericInstantiation(entity);
                case 48:
                    return entity.node.isNull() ?
                        GrammarCut.NONE :
                        new GrammarCut(entity);
                case 49:
                    return entity.node.isNull() ?
                        GrammarDiscard.NONE :
                        new GrammarDiscard(entity);
                case 50:
                    return entity.node.isNull() ?
                        GrammarDontSkip.NONE :
                        new GrammarDontSkip(entity);
                case 51:
                    return entity.node.isNull() ?
                        GrammarList.NONE :
                        new GrammarList(entity);
                case 52:
                    return entity.node.isNull() ?
                        GrammarNull.NONE :
                        new GrammarNull(entity);
                case 53:
                    return entity.node.isNull() ?
                        GrammarOpt.NONE :
                        new GrammarOpt(entity);
                case 54:
                    return entity.node.isNull() ?
                        GrammarOptError.NONE :
                        new GrammarOptError(entity);
                case 55:
                    return entity.node.isNull() ?
                        GrammarOptErrorGroup.NONE :
                        new GrammarOptErrorGroup(entity);
                case 56:
                    return entity.node.isNull() ?
                        GrammarOptGroup.NONE :
                        new GrammarOptGroup(entity);
                case 57:
                    return entity.node.isNull() ?
                        GrammarOrExpr.NONE :
                        new GrammarOrExpr(entity);
                case 58:
                    return entity.node.isNull() ?
                        GrammarPick.NONE :
                        new GrammarPick(entity);
                case 59:
                    return entity.node.isNull() ?
                        GrammarImplicitPick.NONE :
                        new GrammarImplicitPick(entity);
                case 60:
                    return entity.node.isNull() ?
                        GrammarPredicate.NONE :
                        new GrammarPredicate(entity);
                case 61:
                    return entity.node.isNull() ?
                        GrammarRuleRef.NONE :
                        new GrammarRuleRef(entity);
                case 62:
                    return entity.node.isNull() ?
                        GrammarSkip.NONE :
                        new GrammarSkip(entity);
                case 63:
                    return entity.node.isNull() ?
                        GrammarStopCut.NONE :
                        new GrammarStopCut(entity);
                case 64:
                    return entity.node.isNull() ?
                        ParseNodeExpr.NONE :
                        new ParseNodeExpr(entity);
                case 65:
                    return entity.node.isNull() ?
                        TokenLit.NONE :
                        new TokenLit(entity);
                case 66:
                    return entity.node.isNull() ?
                        TokenNoCaseLit.NONE :
                        new TokenNoCaseLit(entity);
                case 67:
                    return entity.node.isNull() ?
                        TokenPatternLit.NONE :
                        new TokenPatternLit(entity);
                case 68:
                    return entity.node.isNull() ?
                        TokenRef.NONE :
                        new TokenRef(entity);
                case 69:
                    return entity.node.isNull() ?
                        Id.NONE :
                        new Id(entity);
                case 70:
                    return entity.node.isNull() ?
                        DefId.NONE :
                        new DefId(entity);
                case 71:
                    return entity.node.isNull() ?
                        ModuleRefId.NONE :
                        new ModuleRefId(entity);
                case 72:
                    return entity.node.isNull() ?
                        RefId.NONE :
                        new RefId(entity);
                case 73:
                    return entity.node.isNull() ?
                        IfExpr.NONE :
                        new IfExpr(entity);
                case 74:
                    return entity.node.isNull() ?
                        Isa.NONE :
                        new Isa(entity);
                case 75:
                    return entity.node.isNull() ?
                        KeepExpr.NONE :
                        new KeepExpr(entity);
                case 76:
                    return entity.node.isNull() ?
                        LambdaExpr.NONE :
                        new LambdaExpr(entity);
                case 77:
                    return entity.node.isNull() ?
                        BigNumLit.NONE :
                        new BigNumLit(entity);
                case 78:
                    return entity.node.isNull() ?
                        CharLit.NONE :
                        new CharLit(entity);
                case 79:
                    return entity.node.isNull() ?
                        NullLit.NONE :
                        new NullLit(entity);
                case 80:
                    return entity.node.isNull() ?
                        NumLit.NONE :
                        new NumLit(entity);
                case 81:
                    return entity.node.isNull() ?
                        BlockStringLit.NONE :
                        new BlockStringLit(entity);
                case 82:
                    return entity.node.isNull() ?
                        SingleLineStringLit.NONE :
                        new SingleLineStringLit(entity);
                case 83:
                    return entity.node.isNull() ?
                        PatternSingleLineStringLit.NONE :
                        new PatternSingleLineStringLit(entity);
                case 84:
                    return entity.node.isNull() ?
                        LogicExpr.NONE :
                        new LogicExpr(entity);
                case 85:
                    return entity.node.isNull() ?
                        MatchExpr.NONE :
                        new MatchExpr(entity);
                case 86:
                    return entity.node.isNull() ?
                        NotExpr.NONE :
                        new NotExpr(entity);
                case 87:
                    return entity.node.isNull() ?
                        ParenExpr.NONE :
                        new ParenExpr(entity);
                case 88:
                    return entity.node.isNull() ?
                        RaiseExpr.NONE :
                        new RaiseExpr(entity);
                case 89:
                    return entity.node.isNull() ?
                        SubscriptExpr.NONE :
                        new SubscriptExpr(entity);
                case 90:
                    return entity.node.isNull() ?
                        NullCondSubscriptExpr.NONE :
                        new NullCondSubscriptExpr(entity);
                case 91:
                    return entity.node.isNull() ?
                        TryExpr.NONE :
                        new TryExpr(entity);
                case 92:
                    return entity.node.isNull() ?
                        UnOp.NONE :
                        new UnOp(entity);
                case 93:
                    return entity.node.isNull() ?
                        FullDecl.NONE :
                        new FullDecl(entity);
                case 94:
                    return entity.node.isNull() ?
                        GrammarListSep.NONE :
                        new GrammarListSep(entity);
                case 95:
                    return entity.node.isNull() ?
                        Import.NONE :
                        new Import(entity);
                case 96:
                    return entity.node.isNull() ?
                        LangkitRoot.NONE :
                        new LangkitRoot(entity);
                case 97:
                    return entity.node.isNull() ?
                        LexerCaseRule.NONE :
                        new LexerCaseRule(entity);
                case 98:
                    return entity.node.isNull() ?
                        LexerCaseRuleSend.NONE :
                        new LexerCaseRuleSend(entity);
                case 99:
                    return entity.node.isNull() ?
                        ListKindOne.NONE :
                        new ListKindOne(entity);
                case 100:
                    return entity.node.isNull() ?
                        ListKindZero.NONE :
                        new ListKindZero(entity);
                case 101:
                    return entity.node.isNull() ?
                        BaseLexerCaseRuleAltList.NONE :
                        new BaseLexerCaseRuleAltList(entity);
                case 102:
                    return entity.node.isNull() ?
                        BlockStringLineList.NONE :
                        new BlockStringLineList(entity);
                case 103:
                    return entity.node.isNull() ?
                        CallExprList.NONE :
                        new CallExprList(entity);
                case 104:
                    return entity.node.isNull() ?
                        DeclAnnotationList.NONE :
                        new DeclAnnotationList(entity);
                case 105:
                    return entity.node.isNull() ?
                        ElsifBranchList.NONE :
                        new ElsifBranchList(entity);
                case 106:
                    return entity.node.isNull() ?
                        EnumClassAltDeclList.NONE :
                        new EnumClassAltDeclList(entity);
                case 107:
                    return entity.node.isNull() ?
                        EnumClassCaseList.NONE :
                        new EnumClassCaseList(entity);
                case 108:
                    return entity.node.isNull() ?
                        EnumLitDeclList.NONE :
                        new EnumLitDeclList(entity);
                case 109:
                    return entity.node.isNull() ?
                        ExprList.NONE :
                        new ExprList(entity);
                case 110:
                    return entity.node.isNull() ?
                        AnyOfList.NONE :
                        new AnyOfList(entity);
                case 111:
                    return entity.node.isNull() ?
                        FullDeclList.NONE :
                        new FullDeclList(entity);
                case 112:
                    return entity.node.isNull() ?
                        DeclBlock.NONE :
                        new DeclBlock(entity);
                case 113:
                    return entity.node.isNull() ?
                        GenericFormalDeclList.NONE :
                        new GenericFormalDeclList(entity);
                case 114:
                    return entity.node.isNull() ?
                        FunArgDeclList.NONE :
                        new FunArgDeclList(entity);
                case 115:
                    return entity.node.isNull() ?
                        GrammarExprList.NONE :
                        new GrammarExprList(entity);
                case 116:
                    return entity.node.isNull() ?
                        GrammarExprListList.NONE :
                        new GrammarExprListList(entity);
                case 117:
                    return entity.node.isNull() ?
                        ImportList.NONE :
                        new ImportList(entity);
                case 118:
                    return entity.node.isNull() ?
                        LambdaArgDeclList.NONE :
                        new LambdaArgDeclList(entity);
                case 119:
                    return entity.node.isNull() ?
                        LktNodeList.NONE :
                        new LktNodeList(entity);
                case 120:
                    return entity.node.isNull() ?
                        BlockDeclList.NONE :
                        new BlockDeclList(entity);
                case 121:
                    return entity.node.isNull() ?
                        MatchBranchList.NONE :
                        new MatchBranchList(entity);
                case 122:
                    return entity.node.isNull() ?
                        ParamList.NONE :
                        new ParamList(entity);
                case 123:
                    return entity.node.isNull() ?
                        RefIdList.NONE :
                        new RefIdList(entity);
                case 124:
                    return entity.node.isNull() ?
                        TypeRefList.NONE :
                        new TypeRefList(entity);
                case 125:
                    return entity.node.isNull() ?
                        IsaList.NONE :
                        new IsaList(entity);
                case 126:
                    return entity.node.isNull() ?
                        MatchBranch.NONE :
                        new MatchBranch(entity);
                case 127:
                    return entity.node.isNull() ?
                        OpAmp.NONE :
                        new OpAmp(entity);
                case 128:
                    return entity.node.isNull() ?
                        OpAnd.NONE :
                        new OpAnd(entity);
                case 129:
                    return entity.node.isNull() ?
                        OpDiv.NONE :
                        new OpDiv(entity);
                case 130:
                    return entity.node.isNull() ?
                        OpEq.NONE :
                        new OpEq(entity);
                case 131:
                    return entity.node.isNull() ?
                        OpGt.NONE :
                        new OpGt(entity);
                case 132:
                    return entity.node.isNull() ?
                        OpGte.NONE :
                        new OpGte(entity);
                case 133:
                    return entity.node.isNull() ?
                        OpLt.NONE :
                        new OpLt(entity);
                case 134:
                    return entity.node.isNull() ?
                        OpLte.NONE :
                        new OpLte(entity);
                case 135:
                    return entity.node.isNull() ?
                        OpMinus.NONE :
                        new OpMinus(entity);
                case 136:
                    return entity.node.isNull() ?
                        OpMult.NONE :
                        new OpMult(entity);
                case 137:
                    return entity.node.isNull() ?
                        OpNe.NONE :
                        new OpNe(entity);
                case 138:
                    return entity.node.isNull() ?
                        OpOr.NONE :
                        new OpOr(entity);
                case 139:
                    return entity.node.isNull() ?
                        OpOrInt.NONE :
                        new OpOrInt(entity);
                case 140:
                    return entity.node.isNull() ?
                        OpPlus.NONE :
                        new OpPlus(entity);
                case 141:
                    return entity.node.isNull() ?
                        Param.NONE :
                        new Param(entity);
                case 142:
                    return entity.node.isNull() ?
                        DefaultListTypeRef.NONE :
                        new DefaultListTypeRef(entity);
                case 143:
                    return entity.node.isNull() ?
                        FunctionTypeRef.NONE :
                        new FunctionTypeRef(entity);
                case 144:
                    return entity.node.isNull() ?
                        GenericTypeRef.NONE :
                        new GenericTypeRef(entity);
                case 145:
                    return entity.node.isNull() ?
                        SimpleTypeRef.NONE :
                        new SimpleTypeRef(entity);
                case 146:
                    return entity.node.isNull() ?
                        VarBind.NONE :
                        new VarBind(entity);
                default:
                    throw new EnumException(
                        "Cannot find the node type from " + nodeKind
                    );
            }
        }

        // ----- Graal C API methods -----

        /**
         * Util internal method to unwrap a node to a native entity struct
         */
        void unwrap(
            final EntityNative entityNative
        ) {
            this.entity.unwrap(entityNative);
        }

        // ----- Getters -----

        public Reflection.Node getDescription() {
            return LktNode.description;
        }

        public NodeKind getKind() {
            return this.getDescription().kind;
        }

        public String getClassName() {
            return this.getDescription().className;
        }

        public boolean isTokenNode() {
            return this.getDescription().isTokenNode;
        }

        public boolean isListNode() {
            return this.getDescription().isListNode;
        }

        public String[] getFieldNames() {
            return this.getDescription().fields;
        }

        public Map<String, Reflection.Field> getFieldDescriptions()
        {
            return this.getDescription().fieldDescriptions;
        }

        @CompilerDirectives.TruffleBoundary
        public Reflection.Field getFieldDescription(
            final String name
        ) {
            return this.getDescription()
                .fieldDescriptions
                .get(name);
        }

        public boolean isNone() {
            return this.entity.node.isNull();
        }

        // ----- Instance methods -----

        /**
         * Get the analysis unit of the node.
         *
         * @return The unit which owns the node.
         */
        public AnalysisUnit getUnit() {
            if(this.unit == null) {

                if(ImageInfo.inImageCode()) {
                    final EntityNative entityNative = StackValue.get(
                        EntityNative.class
                    );
                    this.entity.unwrap(entityNative);

                    final AnalysisUnitNative unitNative =
                        NI_LIB.lkt_node_unit(entityNative);
                    this.unit = AnalysisUnit.wrap(unitNative);
                } else {
                    this.unit = JNI_LIB.lkt_node_unit(this.entity);
                }

            }
            return this.unit;
        }

        /**
         * Get the children count of the node.
         *
         * @return The children count.
         */
        public int getChildrenCount() {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                return NI_LIB.lkt_node_children_count(thisNative);
            } else {
                return JNI_LIB.lkt_node_children_count(this.entity);
            }

        }

        /**
         * Get the child at the given position.
         *
         * @param n The index of the child to get.
         * @return The child at the given index.
         */
        public LktNode getChild(
            final int n
        ) {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final EntityNative resNative = StackValue.get(
                    EntityNative.class
                );
                NI_LIB.lkt_node_child(
                    thisNative,
                    n,
                    resNative
                );

                return fromEntity(Entity.wrap(resNative));
            } else {
                return fromEntity(JNI_LIB.lkt_node_child(
                    this.entity,
                    n
                ));
            }

        }

        /**
         * Get the text of the node.
         *
         * @return The text of the node.
         */
        public String getText() {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final TextNative resNative = StackValue.get(TextNative.class);
                Text.NONE.unwrap(resNative);
                NI_LIB.lkt_node_text(
                    thisNative,
                    resNative
                );

                try(final Text resText = Text.wrap(resNative)) {
                    return resText.getContent();
                }
            } else {
                try(
                    final Text resText = JNI_LIB.lkt_node_text(
                        this.entity
                    );
                ) {
                    return resText.getContent();
                }
            }

        }

        /**
         * Get the image of the node.
         *
         * @return The node's image.
         */
        public String getImage() {
            if(this.image == null) {

                if(ImageInfo.inImageCode()) {
                    final EntityNative thisNative = StackValue.get(
                        EntityNative.class
                    );
                    this.entity.unwrap(thisNative);

                    final TextNative resNative = StackValue.get(
                        TextNative.class
                    );
                    Text.NONE.unwrap(resNative);
                    NI_LIB.lkt_node_image(thisNative, resNative);

                    try(final Text resText = Text.wrap(resNative)) {
                        this.image = resText.getContent();
                    }
                } else {
                    try(
                        final Text resText = JNI_LIB.lkt_node_image(
                            this.entity
                        )
                    ) {
                        this.image = resText.getContent();
                    }
                }

            }

            return this.image;
        }

        /**
         * Get the source location range of the node.
         *
         * @return The source location range of the node.
         */
        public SourceLocationRange getSourceLocationRange() {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final SourceLocationRangeNative resNative = StackValue.get(
                    SourceLocationRangeNative.class
                );
                NI_LIB.lkt_node_sloc_range(
                    thisNative,
                    resNative
                );

                return SourceLocationRange.wrap(resNative);
            } else {
                return JNI_LIB.lkt_node_sloc_range(this.entity);
            }

        }

        /**
         * Return the rewriting handle corresponding to Node.
         *
         * The owning unit of Node must be free of diagnostics.
         */
        public RewritingNode getRewritingNode() {
            if(this.rewritingNode == null) {
                final RewritingNode tmp;

                if(ImageInfo.inImageCode()) {
                    tmp = RewritingNode.wrap(
                        NI_LIB.lkt_rewriting_node_to_handle(
                            (Pointer) this.entity.node.ni()
                        )
                    );
                } else {
                    tmp = JNI_LIB.lkt_rewriting_node_to_handle(
                        this.entity
                    );
                }
                checkException();
                this.rewritingNode = tmp;

            }
            return this.rewritingNode;
        }

        // ----- Dumping methods -----

        /**
         * Return the parsing tree in a string.
         *
         * @return The string containing the representation of the parsing tree
         * from the node.
         */
        @CompilerDirectives.TruffleBoundary
        public String dumpTree() {
            final StringBuilder builder = new StringBuilder();
            this.dumpTree(builder);
            return builder.toString();
        }

        /**
         * Dump the parse tree in the given string builder.
         *
         * @param builder The builder to dump the parse tree in.
         */
        @CompilerDirectives.TruffleBoundary
        public void dumpTree(
            final StringBuilder builder
        ) {
            this.dumpTree(builder, "");
        }

        /**
         * Dump a node field in the given string builder.
         *
         * @param builder The string builder to put the file in.
         * @param indent The current indentation string.
         * @param name The name of the field.
         * @param value The value of the field.
         */
        protected static void dumpField(
            final StringBuilder builder,
            final String indent,
            final String name,
            final LktNode value
        ) {
            builder.append(indent)
                .append(name)
                .append(":\n");
            value.dumpTree(builder, indent + "  ");
        }

        /**
         * Dump the parse tree in the given string builder with the indent
         * level.
         *
         * @param builder The builder to dump the tree in.
         * @param indent The starting indent level.
         */
        @CompilerDirectives.TruffleBoundary
        protected void dumpTree(
            final StringBuilder builder,
            String indent
        ) {
            // Prepare the working variables
            String image = this.getImage();
            image = image.substring(1, image.length());
            final int childrenCount = this.getChildrenCount();

            // Print the node
            builder.append(indent)
                .append(image);
            if(this.isTokenNode()) {
                builder.append(": ")
                    .append(this.getText());
            }
            builder.append('\n');

            // Print the field of the node
            indent = indent + "|";
            if(this.isListNode()) {
                for(int i = 0 ; i < childrenCount ; i++) {
                    final LktNode child = this.getChild(i);
                    dumpField(builder, indent, "item_" + i, child);
                }
            } else {
                for(int i = 0 ; i < childrenCount ; i++) {
                    final LktNode child = this.getChild(i);
                    final String name = this.getFieldNames()[i];
                    dumpField(builder, indent, name, child);
                }
            }
        }

        // ----- Visitor methods -----

        /**
         * Accept the given visitor.
         *
         * @param visitor The visitor to accept.
         * @return The result of the visit.
         */
        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        /**
         * Accept the given visitor.
         *
         * @param visitor The visitor to accept.
         * @param param The parameter of the visit.
         * @return The result of the visit.
         */
        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * Unit method. Return the ``Node`` builtin generic trait.
         */
        public GenericDecl pNodeGenTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_node_gen_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_node_gen_trait(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the ``Node`` builtin trait.
         */
        public TraitDecl pNodeTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_node_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TraitDecl res = TraitDecl.NONE;
                if(propException == null) {
                    res = TraitDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_node_trait(
                    this.entity
                );

                // Wrap and return the result
                return TraitDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the ``TokenNode`` builtin trait.
         */
        public NamedTypeDecl pTokenNodeTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_token_node_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_token_node_trait(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the ``ErrorNode`` builtin trait.
         */
        public NamedTypeDecl pErrorNodeTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_error_node_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_error_node_trait(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the character builtin type.
         */
        public NamedTypeDecl pCharType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_char_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_char_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the integer builtin type.
         */
        public NamedTypeDecl pIntType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_int_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_int_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the boolean builtin type.
         */
        public NamedTypeDecl pBoolType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_bool_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_bool_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the big integer builtin type.
         */
        public NamedTypeDecl pBigintType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_bigint_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_bigint_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the string builtin type.
         */
        public NamedTypeDecl pStringType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_string_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_string_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the string builtin type.
         */
        public NamedTypeDecl pSymbolType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_symbol_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_symbol_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the property error builtin type.
         */
        public NamedTypeDecl pPropertyErrorType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_property_error_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_property_error_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the regexp builtin type.
         */
        public NamedTypeDecl pRegexpType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_regexp_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_regexp_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the array builtin generic type.
         */
        public GenericDecl pArrayGenType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_array_gen_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_array_gen_type(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the array builtin type.
         */
        public NamedTypeDecl pArrayType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_array_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_array_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the ASTList builtin generic type.
         */
        public GenericDecl pAstlistGenType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_astlist_gen_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_astlist_gen_type(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the ASTList builtin type.
         */
        public NamedTypeDecl pAstlistType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_astlist_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NamedTypeDecl res = NamedTypeDecl.NONE;
                if(propException == null) {
                    res = NamedTypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_astlist_type(
                    this.entity
                );

                // Wrap and return the result
                return NamedTypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the Iterator builtin generic trait.
         */
        public GenericDecl pIteratorGenTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_iterator_gen_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_iterator_gen_trait(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the Iterator builtin trait.
         */
        public TraitDecl pIteratorTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_iterator_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TraitDecl res = TraitDecl.NONE;
                if(propException == null) {
                    res = TraitDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_iterator_trait(
                    this.entity
                );

                // Wrap and return the result
                return TraitDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the ``AnalysisUnit`` builtin generic trait.
         */
        public GenericDecl pAnalysisUnitGenTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_analysis_unit_gen_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericDecl res = GenericDecl.NONE;
                if(propException == null) {
                    res = GenericDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_analysis_unit_gen_trait(
                    this.entity
                );

                // Wrap and return the result
                return GenericDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Unit method. Return the ``AnalysisUnit`` builtin trait.
         */
        public TraitDecl pAnalysisUnitTrait(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_analysis_unit_trait(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TraitDecl res = TraitDecl.NONE;
                if(propException == null) {
                    res = TraitDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_analysis_unit_trait(
                    this.entity
                );

                // Wrap and return the result
                return TraitDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Return the topmost (from ``Self`` to the root node) FullDecl
         * annotated with ``@invalid``, null otherwise.
         */
        public LktNode pTopmostInvalidDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_p_topmost_invalid_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode res = LktNode.NONE;
                if(propException == null) {
                    res = LktNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_p_topmost_invalid_decl(
                    this.entity
                );

                // Wrap and return the result
                return LktNode.fromEntity(res);
            }

        }

        
    

        /**
         * Return the syntactic parent for this node. Return null for the root
         * node.
         */
        public LktNode parent(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_parent(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode res = LktNode.NONE;
                if(propException == null) {
                    res = LktNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_parent(
                    this.entity
                );

                // Wrap and return the result
                return LktNode.fromEntity(res);
            }

        }

        
    

        /**
         * Return an array that contains the lexical parents, this node
         * included iff ``with_self`` is True. Nearer parents are first in the
         * list.
         */
        public LktNode[] parents(
            final boolean withSelf
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                byte withSelfNative = (withSelf ? (byte) 1 : (byte) 0);

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_parents(
                    thisNative,
                    withSelfNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode[] res = LktNodeArrayWrapper.NONE;
                if(propException == null) {
                    res = LktNodeArrayWrapper.wrap(resNative);

                    LktNodeArrayWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final LktNode[] res = JNI_LIB.lkt_lkt_node_parents(
                    withSelf,
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return an array that contains the direct lexical children.
         *
         * .. warning:: This constructs a whole array every-time you call it,
         *    and as such is less efficient than calling the ``Child`` built-
         *    in.
         */
        public LktNode[] children(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_children(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode[] res = LktNodeArrayWrapper.NONE;
                if(propException == null) {
                    res = LktNodeArrayWrapper.wrap(resNative);

                    LktNodeArrayWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final LktNode[] res = JNI_LIB.lkt_lkt_node_children(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the first token used to parse this node.
         */
        public Token tokenStart(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                // Get the node unit
                final AnalysisUnit currentUnit = this.getUnit();


                // Unwrap the arguments

                // Create the result native
                final TokenNative resNative =
                    StackValue.get(TokenNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_token_start(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Token res = Token.NONE(currentUnit);
                if(propException == null) {
                    res = Token.wrap(resNative, currentUnit);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Token res = JNI_LIB.lkt_lkt_node_token_start(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the last token used to parse this node.
         */
        public Token tokenEnd(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                // Get the node unit
                final AnalysisUnit currentUnit = this.getUnit();


                // Unwrap the arguments

                // Create the result native
                final TokenNative resNative =
                    StackValue.get(TokenNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_token_end(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Token res = Token.NONE(currentUnit);
                if(propException == null) {
                    res = Token.wrap(resNative, currentUnit);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Token res = JNI_LIB.lkt_lkt_node_token_end(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the 0-based index for Node in its parent's children.
         */
        public int childIndex(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CIntPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_child_index(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                int res = 0;
                if(propException == null) {
                    res = IntegerWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final int res = JNI_LIB.lkt_lkt_node_child_index(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the node's previous sibling, or null if there is no such
         * sibling.
         */
        public LktNode previousSibling(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_previous_sibling(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode res = LktNode.NONE;
                if(propException == null) {
                    res = LktNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_previous_sibling(
                    this.entity
                );

                // Wrap and return the result
                return LktNode.fromEntity(res);
            }

        }

        
    

        /**
         * Return the node's next sibling, or null if there is no such sibling.
         */
        public LktNode nextSibling(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lkt_node_next_sibling(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNode res = LktNode.NONE;
                if(propException == null) {
                    res = LktNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lkt_node_next_sibling(
                    this.entity
                );

                // Wrap and return the result
                return LktNode.fromEntity(res);
            }

        }

        
    


        
    

        /**
         * Return whether the node is a ghost.
         *
         * Unlike regular nodes, ghost nodes cover no token in the input
         * source: they are logically located instead between two tokens. Both
         * the ``token_start`` and the ``token_end`` of all ghost nodes is the
         * token right after this logical position.
         */
        public boolean isGhost(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_is_ghost(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_lkt_node_is_ghost(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return a string containing the filename + the sloc in GNU conformant
         * format. Useful to create diagnostics from a node.
         */
        public String fullSlocImage(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_lkt_node_full_sloc_image(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                String res = StringWrapper.NONE;
                if(propException == null) {
                    res = StringWrapper.wrap(resNative);

                    StringWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final String res = JNI_LIB.lkt_lkt_node_full_sloc_image(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Override methods -----

        @Override
        public String toString() {
            return this.getImage();
        }

        @Override
        public boolean equals(Object o) {
            if(this == o) return true;
            if(!(o instanceof LktNode)) return false;
            final LktNode other = (LktNode) o;

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final EntityNative otherNative = StackValue.get(
                    EntityNative.class
                );
                other.entity.unwrap(otherNative);

                return NI_LIB.lkt_node_is_equivalent(
                    thisNative,
                    otherNative
                ) != 0;
            } else {
                return JNI_LIB.lkt_node_is_equivalent(
                    this.entity, other.entity
                ) != 0;
            }
        }

        @Override
        public int hashCode() {
            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                return NI_LIB.lkt_node_hash(thisNative);
            } else {
                return JNI_LIB.lkt_node_hash(this.entity);
            }

        }

        // ----- Inner classes -----

        /**
         * This class represents the none node without any concrete type.
         */
        private static final class NoneNode extends LktNode {
            NoneNode() {super(Entity.NONE);}
        }

    }

    // ===== Generated node wrapping classes =====

    
    

    /**
     * Base class for the different kind of alternatives allowed in a case
     * rule.
     *
     * Derived nodes: ``LexerCaseRuleCondAlt``, ``LexerCaseRuleDefaultAlt``
     */
    public static abstract
    class BaseLexerCaseRuleAlt extends LktNode {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                BaseLexerCaseRuleAlt.class,
                "BaseLexerCaseRuleAlt",
                new String[] {
                    
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = BaseLexerCaseRuleAlt.class.getMethod(
                        "fSend",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_send",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BASE_LEXER_CASE_RULE_ALT_F_SEND
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BaseLexerCaseRuleAlt NONE =
            new BaseLexerCaseRuleAltNone();

        // ----- Constructors -----

        protected BaseLexerCaseRuleAlt(
            final Entity entity
        ) {
            super(entity);
        }

        public static BaseLexerCaseRuleAlt fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BaseLexerCaseRuleAlt.NONE :

                (BaseLexerCaseRuleAlt) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BaseLexerCaseRuleAlt.description;
        }


        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public LexerCaseRuleSend fSend(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_base_lexer_case_rule_alt_f_send(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LexerCaseRuleSend res = LexerCaseRuleSend.NONE;
                if(propException == null) {
                    res = LexerCaseRuleSend.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_base_lexer_case_rule_alt_f_send(
                    this.entity
                );

                // Wrap and return the result
                return LexerCaseRuleSend.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * BaseLexerCaseRuleAlt.
         */
        private static final class BaseLexerCaseRuleAltNone extends BaseLexerCaseRuleAlt {
            BaseLexerCaseRuleAltNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * Alternative of a case rule which sends the token only if the kind of the
     * previous token is among a given set.
     *
     * This node type has no derivation.
     */
    public static 
    class LexerCaseRuleCondAlt extends BaseLexerCaseRuleAlt {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.LEXER_CASE_RULE_COND_ALT,
                false,
                false,
                LexerCaseRuleCondAlt.class,
                "LexerCaseRuleCondAlt",
                new String[] {
                    "f_cond_exprs","f_send"
                },
                new HashMap<>(
                    BaseLexerCaseRuleAlt.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = LexerCaseRuleCondAlt.class.getMethod(
                        "fCondExprs",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_cond_exprs",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LEXER_CASE_RULE_COND_ALT_F_COND_EXPRS
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LexerCaseRuleCondAlt NONE =
            new LexerCaseRuleCondAlt(
                Entity.NONE
            );

        // ----- Constructors -----

        protected LexerCaseRuleCondAlt(
            final Entity entity
        ) {
            super(entity);
        }

        public static LexerCaseRuleCondAlt fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LexerCaseRuleCondAlt.NONE :

                new LexerCaseRuleCondAlt(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return LexerCaseRuleCondAlt.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public RefIdList fCondExprs(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lexer_case_rule_cond_alt_f_cond_exprs(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                RefIdList res = RefIdList.NONE;
                if(propException == null) {
                    res = RefIdList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lexer_case_rule_cond_alt_f_cond_exprs(
                    this.entity
                );

                // Wrap and return the result
                return RefIdList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Default alternative of a case rule which sends the token if all the
     * previous alternatives failed.
     *
     * This node type has no derivation.
     */
    public static 
    class LexerCaseRuleDefaultAlt extends BaseLexerCaseRuleAlt {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.LEXER_CASE_RULE_DEFAULT_ALT,
                false,
                false,
                LexerCaseRuleDefaultAlt.class,
                "LexerCaseRuleDefaultAlt",
                new String[] {
                    "f_send"
                },
                new HashMap<>(
                    BaseLexerCaseRuleAlt.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LexerCaseRuleDefaultAlt NONE =
            new LexerCaseRuleDefaultAlt(
                Entity.NONE
            );

        // ----- Constructors -----

        protected LexerCaseRuleDefaultAlt(
            final Entity entity
        ) {
            super(entity);
        }

        public static LexerCaseRuleDefaultAlt fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LexerCaseRuleDefaultAlt.NONE :

                new LexerCaseRuleDefaultAlt(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return LexerCaseRuleDefaultAlt.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * A single line in a block string literal.
     *
     * This node type has no derivation.
     */
    public static 
    class BlockStringLine extends LktNode {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.BLOCK_STRING_LINE,
                true,
                false,
                BlockStringLine.class,
                "BlockStringLine",
                new String[] {
                    
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BlockStringLine NONE =
            new BlockStringLine(
                Entity.NONE
            );

        // ----- Constructors -----

        protected BlockStringLine(
            final Entity entity
        ) {
            super(entity);
        }

        public static BlockStringLine fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BlockStringLine.NONE :

                new BlockStringLine(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BlockStringLine.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Whether this generic formal type must be a class.
     *
     * Derived nodes: ``ClassQualifierAbsent``, ``ClassQualifierPresent``
     */
    public static abstract
    class ClassQualifier extends LktNode {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                ClassQualifier.class,
                "ClassQualifier",
                new String[] {
                    
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ClassQualifier.class.getMethod(
                        "pAsBool",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_as_bool",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CLASS_QUALIFIER_P_AS_BOOL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ClassQualifier NONE =
            new ClassQualifierNone();

        // ----- Constructors -----

        protected ClassQualifier(
            final Entity entity
        ) {
            super(entity);
        }

        public static ClassQualifier fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ClassQualifier.NONE :

                (ClassQualifier) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ClassQualifier.description;
        }


        // ----- Field accessors -----

        
    

        /**
         * Return whether this is an instance of ClassQualifierPresent
         */
        public boolean pAsBool(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_class_qualifier_p_as_bool(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_class_qualifier_p_as_bool(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * ClassQualifier.
         */
        private static final class ClassQualifierNone extends ClassQualifier {
            ClassQualifierNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * This node type has no derivation.
     */
    public static 
    class ClassQualifierAbsent extends ClassQualifier {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.CLASS_QUALIFIER_ABSENT,
                false,
                false,
                ClassQualifierAbsent.class,
                "ClassQualifierAbsent",
                new String[] {
                    
                },
                new HashMap<>(
                    ClassQualifier.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ClassQualifierAbsent NONE =
            new ClassQualifierAbsent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ClassQualifierAbsent(
            final Entity entity
        ) {
            super(entity);
        }

        public static ClassQualifierAbsent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ClassQualifierAbsent.NONE :

                new ClassQualifierAbsent(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ClassQualifierAbsent.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * This node type has no derivation.
     */
    public static 
    class ClassQualifierPresent extends ClassQualifier {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.CLASS_QUALIFIER_PRESENT,
                false,
                false,
                ClassQualifierPresent.class,
                "ClassQualifierPresent",
                new String[] {
                    
                },
                new HashMap<>(
                    ClassQualifier.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ClassQualifierPresent NONE =
            new ClassQualifierPresent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ClassQualifierPresent(
            final Entity entity
        ) {
            super(entity);
        }

        public static ClassQualifierPresent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ClassQualifierPresent.NONE :

                new ClassQualifierPresent(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ClassQualifierPresent.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Base class for declarations. Encompasses regular declarations as well as
     * special declarations such as grammars, grammar rules, etc.
     *
     * Derived nodes: ``BaseGrammarRuleDecl``, ``BaseValDecl``,
     * ``EnvSpecDecl``, ``GenericDecl``, ``GrammarDecl``, ``LexerDecl``,
     * ``LexerFamilyDecl``, ``TypeDecl``
     */
    public static abstract
    class Decl extends LktNode {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                Decl.class,
                "Decl",
                new String[] {
                    
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "fSynName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_syn_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_F_SYN_NAME
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_NAME
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pFullName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_full_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_FULL_NAME
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pDeclTypeName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_decl_type_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_DECL_TYPE_NAME
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Decl.class.getMethod(
                        "pAsBareDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_as_bare_decl",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_P_AS_BARE_DECL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final Decl NONE =
            new DeclNone();

        // ----- Constructors -----

        protected Decl(
            final Entity entity
        ) {
            super(entity);
        }

        public static Decl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                Decl.NONE :

                (Decl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return Decl.description;
        }


        // ----- Field accessors -----

        
    

        /**
         * This field may be null even when there are no parsing errors.
         */
        public DefId fSynName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_decl_f_syn_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                DefId res = DefId.NONE;
                if(propException == null) {
                    res = DefId.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_decl_f_syn_name(
                    this.entity
                );

                // Wrap and return the result
                return DefId.fromEntity(res);
            }

        }

        
    

        /**
         * Return the symbol corresponding to the name of this declaration.
         */
        public Symbol pName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final SymbolNative resNative =
                    StackValue.get(SymbolNative.class);

                // Call the native function
                NI_LIB.lkt_decl_p_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Symbol res = Symbol.NONE;
                if(propException == null) {
                    res = Symbol.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Symbol res = JNI_LIB.lkt_decl_p_name(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the full name of this decl, as it should be seen by
         * users/shown in diagnostics.
         */
        public String pFullName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_decl_p_full_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                String res = StringWrapper.NONE;
                if(propException == null) {
                    res = StringWrapper.wrap(resNative);

                    StringWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final String res = JNI_LIB.lkt_decl_p_full_name(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the name of the declaration type, as it should be seen by
         * users/shown in diagnostics.
         */
        public String pDeclTypeName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_decl_p_decl_type_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                String res = StringWrapper.NONE;
                if(propException == null) {
                    res = StringWrapper.wrap(resNative);

                    StringWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final String res = JNI_LIB.lkt_decl_p_decl_type_name(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Get this declaration without rebindings information.
         */
        public Decl pAsBareDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_decl_p_as_bare_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Decl res = Decl.NONE;
                if(propException == null) {
                    res = Decl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_decl_p_as_bare_decl(
                    this.entity
                );

                // Wrap and return the result
                return Decl.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * Decl.
         */
        private static final class DeclNone extends Decl {
            DeclNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * Base class for grammar rules inside of grammars/lexers.
     *
     * Derived nodes: ``GrammarRuleDecl``, ``SyntheticLexerDecl``
     */
    public static abstract
    class BaseGrammarRuleDecl extends Decl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                BaseGrammarRuleDecl.class,
                "BaseGrammarRuleDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = BaseGrammarRuleDecl.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BASE_GRAMMAR_RULE_DECL_F_EXPR
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BaseGrammarRuleDecl NONE =
            new BaseGrammarRuleDeclNone();

        // ----- Constructors -----

        protected BaseGrammarRuleDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static BaseGrammarRuleDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BaseGrammarRuleDecl.NONE :

                (BaseGrammarRuleDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BaseGrammarRuleDecl.description;
        }


        // ----- Field accessors -----

        
    

        /**
         * This field may be null even when there are no parsing errors.
         */
        public GrammarExpr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_base_grammar_rule_decl_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GrammarExpr res = GrammarExpr.NONE;
                if(propException == null) {
                    res = GrammarExpr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_base_grammar_rule_decl_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return GrammarExpr.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * BaseGrammarRuleDecl.
         */
        private static final class BaseGrammarRuleDeclNone extends BaseGrammarRuleDecl {
            BaseGrammarRuleDeclNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * Declaration of a grammar rule inside of a grammar.
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarRuleDecl extends BaseGrammarRuleDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_RULE_DECL,
                false,
                false,
                GrammarRuleDecl.class,
                "GrammarRuleDecl",
                new String[] {
                    "f_syn_name","f_expr"
                },
                new HashMap<>(
                    BaseGrammarRuleDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarRuleDecl NONE =
            new GrammarRuleDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GrammarRuleDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarRuleDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarRuleDecl.NONE :

                new GrammarRuleDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarRuleDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * This node type has no derivation.
     */
    public static 
    class SyntheticLexerDecl extends BaseGrammarRuleDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.SYNTHETIC_LEXER_DECL,
                false,
                false,
                SyntheticLexerDecl.class,
                "SyntheticLexerDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    BaseGrammarRuleDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final SyntheticLexerDecl NONE =
            new SyntheticLexerDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected SyntheticLexerDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static SyntheticLexerDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                SyntheticLexerDecl.NONE :

                new SyntheticLexerDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return SyntheticLexerDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Abstract class for named values declarations, such as arguments, local
     * value bindings, fields, etc.
     *
     * Derived nodes: ``NodeDecl``, ``SelfDecl``, ``UserValDecl``
     */
    public static abstract
    class BaseValDecl extends Decl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                BaseValDecl.class,
                "BaseValDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = BaseValDecl.class.getMethod(
                        "pGetType",
                        new Class[]{boolean.class}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();
                    parameters.add(new Reflection.Param(
                        boolean.class,
                        "noInference",
                        false
                    ));

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_get_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BASE_VAL_DECL_P_GET_TYPE
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BaseValDecl NONE =
            new BaseValDeclNone();

        // ----- Constructors -----

        protected BaseValDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static BaseValDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BaseValDecl.NONE :

                (BaseValDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BaseValDecl.description;
        }


        // ----- Field accessors -----

        
    

        /**
         * Get the type of this value declaration.
         */
        public TypeDecl pGetType(
            final boolean noInference
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                byte noInferenceNative = (noInference ? (byte) 1 : (byte) 0);

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_base_val_decl_p_get_type(
                    thisNative,
                    noInferenceNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl res = TypeDecl.NONE;
                if(propException == null) {
                    res = TypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_base_val_decl_p_get_type(
                    noInference,
                    this.entity
                );

                // Wrap and return the result
                return TypeDecl.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * BaseValDecl.
         */
        private static final class BaseValDeclNone extends BaseValDecl {
            BaseValDeclNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * Synthetic declaration for the implicit "node" variable available in
     * properties.
     *
     * This node type has no derivation.
     */
    public static 
    class NodeDecl extends BaseValDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.NODE_DECL,
                false,
                false,
                NodeDecl.class,
                "NodeDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    BaseValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final NodeDecl NONE =
            new NodeDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected NodeDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static NodeDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                NodeDecl.NONE :

                new NodeDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return NodeDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Synthetic declaration for the implicit "self" variable available in
     * properties.
     *
     * This node type has no derivation.
     */
    public static 
    class SelfDecl extends BaseValDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.SELF_DECL,
                false,
                false,
                SelfDecl.class,
                "SelfDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    BaseValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final SelfDecl NONE =
            new SelfDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected SelfDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static SelfDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                SelfDecl.NONE :

                new SelfDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return SelfDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Class for user declared val declarations (not synthetic).
     *
     * Derived nodes: ``EnumLitDecl``, ``ExplicitlyTypedDecl``, ``FunDecl``
     */
    public static abstract
    class UserValDecl extends BaseValDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                UserValDecl.class,
                "UserValDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    BaseValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final UserValDecl NONE =
            new UserValDeclNone();

        // ----- Constructors -----

        protected UserValDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static UserValDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                UserValDecl.NONE :

                (UserValDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return UserValDecl.description;
        }


        // ----- Field accessors -----


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * UserValDecl.
         */
        private static final class UserValDeclNone extends UserValDecl {
            UserValDeclNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * Enum literal declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class EnumLitDecl extends UserValDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ENUM_LIT_DECL,
                false,
                false,
                EnumLitDecl.class,
                "EnumLitDecl",
                new String[] {
                    "f_syn_name"
                },
                new HashMap<>(
                    UserValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final EnumLitDecl NONE =
            new EnumLitDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected EnumLitDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static EnumLitDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                EnumLitDecl.NONE :

                new EnumLitDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return EnumLitDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Subset of user declared value declarations for values that have a type
     * that can be syntactically annotated by the user.
     *
     * Derived nodes: ``ComponentDecl``, ``DynVarDecl``, ``MatchValDecl``,
     * ``ValDecl``
     */
    public static abstract
    class ExplicitlyTypedDecl extends UserValDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                ExplicitlyTypedDecl.class,
                "ExplicitlyTypedDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    UserValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ExplicitlyTypedDecl.class.getMethod(
                        "fDeclType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_decl_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_EXPLICITLY_TYPED_DECL_F_DECL_TYPE
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ExplicitlyTypedDecl NONE =
            new ExplicitlyTypedDeclNone();

        // ----- Constructors -----

        protected ExplicitlyTypedDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static ExplicitlyTypedDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ExplicitlyTypedDecl.NONE :

                (ExplicitlyTypedDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ExplicitlyTypedDecl.description;
        }


        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes:
         * ``FunctionTypeRef``, ``GenericTypeRef``, ``SimpleTypeRef``
         *
         * This field may be null even when there are no parsing errors.
         */
        public TypeRef fDeclType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_explicitly_typed_decl_f_decl_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeRef res = TypeRef.NONE;
                if(propException == null) {
                    res = TypeRef.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_explicitly_typed_decl_f_decl_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeRef.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * ExplicitlyTypedDecl.
         */
        private static final class ExplicitlyTypedDeclNone extends ExplicitlyTypedDecl {
            ExplicitlyTypedDeclNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * Subset of explicitly typed declarations for value declarations that:
     *
     * 1. Have an optional default value.
     *
     * 2. Are part of a bigger declaration that can be referred to via a call
     *    expression (either a type or a function).
     *
     * Derived nodes: ``FieldDecl``, ``FunArgDecl``, ``LambdaArgDecl``
     */
    public static abstract
    class ComponentDecl extends ExplicitlyTypedDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                ComponentDecl.class,
                "ComponentDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    ExplicitlyTypedDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ComponentDecl.class.getMethod(
                        "fDefaultVal",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_default_val",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_COMPONENT_DECL_F_DEFAULT_VAL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ComponentDecl NONE =
            new ComponentDeclNone();

        // ----- Constructors -----

        protected ComponentDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static ComponentDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ComponentDecl.NONE :

                (ComponentDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ComponentDecl.description;
        }


        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BaseDotExpr``, ``BinOp``, ``BlockExpr``,
         * ``CallExpr``, ``CastExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``, ``NotExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``, ``UnOp``
         *
         * This field may be null even when there are no parsing errors.
         */
        public Expr fDefaultVal(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_component_decl_f_default_val(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_component_decl_f_default_val(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * ComponentDecl.
         */
        private static final class ComponentDeclNone extends ComponentDecl {
            ComponentDeclNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * Field declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class FieldDecl extends ComponentDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.FIELD_DECL,
                false,
                false,
                FieldDecl.class,
                "FieldDecl",
                new String[] {
                    "f_syn_name","f_decl_type","f_default_val"
                },
                new HashMap<>(
                    ComponentDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final FieldDecl NONE =
            new FieldDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected FieldDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static FieldDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                FieldDecl.NONE :

                new FieldDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return FieldDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Function argument declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class FunArgDecl extends ComponentDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.FUN_ARG_DECL,
                false,
                false,
                FunArgDecl.class,
                "FunArgDecl",
                new String[] {
                    "f_decl_annotations","f_syn_name","f_decl_type","f_default_val"
                },
                new HashMap<>(
                    ComponentDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = FunArgDecl.class.getMethod(
                        "fDeclAnnotations",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_decl_annotations",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FUN_ARG_DECL_F_DECL_ANNOTATIONS
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final FunArgDecl NONE =
            new FunArgDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected FunArgDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static FunArgDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                FunArgDecl.NONE :

                new FunArgDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return FunArgDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public DeclAnnotationList fDeclAnnotations(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_fun_arg_decl_f_decl_annotations(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                DeclAnnotationList res = DeclAnnotationList.NONE;
                if(propException == null) {
                    res = DeclAnnotationList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_fun_arg_decl_f_decl_annotations(
                    this.entity
                );

                // Wrap and return the result
                return DeclAnnotationList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Function argument declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class LambdaArgDecl extends ComponentDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.LAMBDA_ARG_DECL,
                false,
                false,
                LambdaArgDecl.class,
                "LambdaArgDecl",
                new String[] {
                    "f_syn_name","f_decl_type","f_default_val"
                },
                new HashMap<>(
                    ComponentDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LambdaArgDecl NONE =
            new LambdaArgDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected LambdaArgDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static LambdaArgDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LambdaArgDecl.NONE :

                new LambdaArgDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return LambdaArgDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Dynamic variable declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class DynVarDecl extends ExplicitlyTypedDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.DYN_VAR_DECL,
                false,
                false,
                DynVarDecl.class,
                "DynVarDecl",
                new String[] {
                    "f_syn_name","f_decl_type"
                },
                new HashMap<>(
                    ExplicitlyTypedDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final DynVarDecl NONE =
            new DynVarDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected DynVarDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static DynVarDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                DynVarDecl.NONE :

                new DynVarDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return DynVarDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Value declaration in a match branch.
     *
     * This node type has no derivation.
     */
    public static 
    class MatchValDecl extends ExplicitlyTypedDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.MATCH_VAL_DECL,
                false,
                false,
                MatchValDecl.class,
                "MatchValDecl",
                new String[] {
                    "f_syn_name","f_decl_type"
                },
                new HashMap<>(
                    ExplicitlyTypedDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final MatchValDecl NONE =
            new MatchValDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected MatchValDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static MatchValDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                MatchValDecl.NONE :

                new MatchValDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return MatchValDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Value declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class ValDecl extends ExplicitlyTypedDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.VAL_DECL,
                false,
                false,
                ValDecl.class,
                "ValDecl",
                new String[] {
                    "f_syn_name","f_decl_type","f_val"
                },
                new HashMap<>(
                    ExplicitlyTypedDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ValDecl.class.getMethod(
                        "fVal",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_val",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_VAL_DECL_F_VAL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ValDecl NONE =
            new ValDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ValDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static ValDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ValDecl.NONE :

                new ValDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ValDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BaseDotExpr``, ``BinOp``, ``BlockExpr``,
         * ``CallExpr``, ``CastExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``, ``NotExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``, ``UnOp``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fVal(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_val_decl_f_val(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_val_decl_f_val(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Function declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class FunDecl extends UserValDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.FUN_DECL,
                false,
                false,
                FunDecl.class,
                "FunDecl",
                new String[] {
                    "f_syn_name","f_args","f_return_type","f_body"
                },
                new HashMap<>(
                    UserValDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = FunDecl.class.getMethod(
                        "fArgs",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_args",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FUN_DECL_F_ARGS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = FunDecl.class.getMethod(
                        "fReturnType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_return_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FUN_DECL_F_RETURN_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = FunDecl.class.getMethod(
                        "fBody",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_body",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_FUN_DECL_F_BODY
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final FunDecl NONE =
            new FunDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected FunDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static FunDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                FunDecl.NONE :

                new FunDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return FunDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public FunArgDeclList fArgs(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_fun_decl_f_args(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                FunArgDeclList res = FunArgDeclList.NONE;
                if(propException == null) {
                    res = FunArgDeclList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_fun_decl_f_args(
                    this.entity
                );

                // Wrap and return the result
                return FunArgDeclList.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes:
         * ``FunctionTypeRef``, ``GenericTypeRef``, ``SimpleTypeRef``
         *
         * When there are no parsing errors, this field is never null.
         */
        public TypeRef fReturnType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_fun_decl_f_return_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeRef res = TypeRef.NONE;
                if(propException == null) {
                    res = TypeRef.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_fun_decl_f_return_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeRef.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BaseDotExpr``, ``BinOp``, ``BlockExpr``,
         * ``CallExpr``, ``CastExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``, ``NotExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``, ``UnOp``
         *
         * This field may be null even when there are no parsing errors.
         */
        public Expr fBody(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_fun_decl_f_body(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_fun_decl_f_body(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Env spec declaration.
     *
     * Each node type can have one or no env spec. Env specs contains only a
     * list of env actions.
     *
     * This node type has no derivation.
     */
    public static 
    class EnvSpecDecl extends Decl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ENV_SPEC_DECL,
                false,
                false,
                EnvSpecDecl.class,
                "EnvSpecDecl",
                new String[] {
                    "f_syn_name","f_actions"
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = EnvSpecDecl.class.getMethod(
                        "fActions",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_actions",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ENV_SPEC_DECL_F_ACTIONS
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final EnvSpecDecl NONE =
            new EnvSpecDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected EnvSpecDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static EnvSpecDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                EnvSpecDecl.NONE :

                new EnvSpecDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return EnvSpecDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public CallExprList fActions(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_env_spec_decl_f_actions(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                CallExprList res = CallExprList.NONE;
                if(propException == null) {
                    res = CallExprList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_env_spec_decl_f_actions(
                    this.entity
                );

                // Wrap and return the result
                return CallExprList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Generic entity declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class GenericDecl extends Decl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GENERIC_DECL,
                false,
                false,
                GenericDecl.class,
                "GenericDecl",
                new String[] {
                    "f_generic_formal_decls","f_decl"
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GenericDecl.class.getMethod(
                        "fGenericFormalDecls",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_generic_formal_decls",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GENERIC_DECL_F_GENERIC_FORMAL_DECLS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GenericDecl.class.getMethod(
                        "fDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_decl",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GENERIC_DECL_F_DECL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GenericDecl NONE =
            new GenericDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GenericDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static GenericDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GenericDecl.NONE :

                new GenericDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GenericDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public GenericFormalDeclList fGenericFormalDecls(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_generic_decl_f_generic_formal_decls(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GenericFormalDeclList res = GenericFormalDeclList.NONE;
                if(propException == null) {
                    res = GenericFormalDeclList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_generic_decl_f_generic_formal_decls(
                    this.entity
                );

                // Wrap and return the result
                return GenericFormalDeclList.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes: ``DynVarDecl``,
         * ``EnvSpecDecl``, ``FieldDecl``, ``FunDecl``, ``GenericDecl``,
         * ``GrammarDecl``, ``GrammarRuleDecl``, ``LexerDecl``,
         * ``NamedTypeDecl``, ``ValDecl``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Decl fDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_generic_decl_f_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Decl res = Decl.NONE;
                if(propException == null) {
                    res = Decl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_generic_decl_f_decl(
                    this.entity
                );

                // Wrap and return the result
                return Decl.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Declaration of a language's grammar.
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarDecl extends Decl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_DECL,
                false,
                false,
                GrammarDecl.class,
                "GrammarDecl",
                new String[] {
                    "f_syn_name","f_rules"
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GrammarDecl.class.getMethod(
                        "fRules",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_rules",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_DECL_F_RULES
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GrammarDecl.class.getMethod(
                        "pLexer",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_lexer",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_DECL_P_LEXER
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarDecl NONE =
            new GrammarDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GrammarDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarDecl.NONE :

                new GrammarDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public FullDeclList fRules(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_decl_f_rules(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                FullDeclList res = FullDeclList.NONE;
                if(propException == null) {
                    res = FullDeclList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_decl_f_rules(
                    this.entity
                );

                // Wrap and return the result
                return FullDeclList.fromEntity(res);
            }

        }

        
    

        /**
         * Return the lexer that is associated to this grammar.
         */
        public Decl pLexer(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_decl_p_lexer(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Decl res = Decl.NONE;
                if(propException == null) {
                    res = Decl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_decl_p_lexer(
                    this.entity
                );

                // Wrap and return the result
                return Decl.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Declaration of a language's lexer.
     *
     * This node type has no derivation.
     */
    public static 
    class LexerDecl extends Decl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.LEXER_DECL,
                false,
                false,
                LexerDecl.class,
                "LexerDecl",
                new String[] {
                    "f_syn_name","f_rules"
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = LexerDecl.class.getMethod(
                        "fRules",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_rules",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LEXER_DECL_F_RULES
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LexerDecl NONE =
            new LexerDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected LexerDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static LexerDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LexerDecl.NONE :

                new LexerDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return LexerDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field contains a list that itself contains one of the following
         * nodes: ``FullDecl``, ``LexerCaseRule``
         *
         * When there are no parsing errors, this field is never null.
         */
        public LktNodeList fRules(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lexer_decl_f_rules(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LktNodeList res = LktNodeList.NONE;
                if(propException == null) {
                    res = LktNodeList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lexer_decl_f_rules(
                    this.entity
                );

                // Wrap and return the result
                return LktNodeList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Declaration of a token family.
     *
     * This node type has no derivation.
     */
    public static 
    class LexerFamilyDecl extends Decl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.LEXER_FAMILY_DECL,
                false,
                false,
                LexerFamilyDecl.class,
                "LexerFamilyDecl",
                new String[] {
                    "f_syn_name","f_rules"
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = LexerFamilyDecl.class.getMethod(
                        "fRules",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_rules",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_LEXER_FAMILY_DECL_F_RULES
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LexerFamilyDecl NONE =
            new LexerFamilyDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected LexerFamilyDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static LexerFamilyDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LexerFamilyDecl.NONE :

                new LexerFamilyDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return LexerFamilyDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public FullDeclList fRules(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_lexer_family_decl_f_rules(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                FullDeclList res = FullDeclList.NONE;
                if(propException == null) {
                    res = FullDeclList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_lexer_family_decl_f_rules(
                    this.entity
                );

                // Wrap and return the result
                return FullDeclList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Abstract base class for type declarations.
     *
     * Derived nodes: ``EnumClassAltDecl``, ``FunctionType``,
     * ``GenericFormalTypeDecl``, ``InstantiatedGenericType``,
     * ``NamedTypeDecl``
     */
    public static abstract
    class TypeDecl extends Decl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                TypeDecl.class,
                "TypeDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    Decl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "fTraits",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_traits",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_F_TRAITS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "pIsClass",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_is_class",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_P_IS_CLASS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "pImplementedTraits",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_implemented_traits",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_P_IMPLEMENTED_TRAITS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "fSynBaseType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_syn_base_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_F_SYN_BASE_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "pBaseType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_base_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_P_BASE_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "pBaseTypes",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_base_types",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_P_BASE_TYPES
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "pRootType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_root_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_P_ROOT_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "pIsSubtype",
                        new Class[]{TypeDecl.class}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();
                    parameters.add(new Reflection.Param(
                        TypeDecl.class,
                        "potentialBase"
                    ));

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_is_subtype",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_P_IS_SUBTYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypeDecl.class.getMethod(
                        "pIsGeneric",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_is_generic",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_TYPE_DECL_P_IS_GENERIC
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final TypeDecl NONE =
            new TypeDeclNone();

        // ----- Constructors -----

        protected TypeDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static TypeDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                TypeDecl.NONE :

                (TypeDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return TypeDecl.description;
        }


        // ----- Field accessors -----

        
    

        /**
         * Traits for this type
         *
         * This field may be null even when there are no parsing errors.
         */
        public TypeRefList fTraits(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_type_decl_f_traits(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeRefList res = TypeRefList.NONE;
                if(propException == null) {
                    res = TypeRefList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_type_decl_f_traits(
                    this.entity
                );

                // Wrap and return the result
                return TypeRefList.fromEntity(res);
            }

        }

        
    

        /**
         * Return whether this type declaration is a class.
         */
        public boolean pIsClass(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_type_decl_p_is_class(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_type_decl_p_is_class(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Traits implemented by this type.
         */
        public TypeDecl[] pImplementedTraits(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_type_decl_p_implemented_traits(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl[] res = TypeDeclArrayWrapper.NONE;
                if(propException == null) {
                    res = TypeDeclArrayWrapper.wrap(resNative);

                    TypeDeclArrayWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final TypeDecl[] res = JNI_LIB.lkt_type_decl_p_implemented_traits(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * This field may be null even when there are no parsing errors.
         */
        public TypeRef fSynBaseType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_type_decl_f_syn_base_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeRef res = TypeRef.NONE;
                if(propException == null) {
                    res = TypeRef.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_type_decl_f_syn_base_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeRef.fromEntity(res);
            }

        }

        
    

        /**
         * Return the base type for this node, if any.
         */
        public TypeRef pBaseType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_type_decl_p_base_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeRef res = TypeRef.NONE;
                if(propException == null) {
                    res = TypeRef.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_type_decl_p_base_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeRef.fromEntity(res);
            }

        }

        
    

        /**
         * Return the chain of base types for this type, if applicable, else
         * null.
         */
        public TypeDecl[] pBaseTypes(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_type_decl_p_base_types(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl[] res = TypeDeclArrayWrapper.NONE;
                if(propException == null) {
                    res = TypeDeclArrayWrapper.wrap(resNative);

                    TypeDeclArrayWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final TypeDecl[] res = JNI_LIB.lkt_type_decl_p_base_types(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the root type of this type, if applicable, else null.
         */
        public TypeDecl pRootType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_type_decl_p_root_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl res = TypeDecl.NONE;
                if(propException == null) {
                    res = TypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_type_decl_p_root_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Return whether ``self`` is a subtype of ``potential_base``.
         */
        public boolean pIsSubtype(
            final TypeDecl potentialBase
        ) {

            // Verify that arguments are not null
            if(potentialBase == null) throw new IllegalArgumentException(
                "Argument 'potentialBase' of type " +
                "TypeDecl cannot be null"
            );

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                EntityNative potentialBaseNative = StackValue.get(EntityNative.class);potentialBase.entity.unwrap(potentialBaseNative);

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_type_decl_p_is_subtype(
                    thisNative,
                    potentialBaseNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_type_decl_p_is_subtype(
                    (potentialBase != null ? potentialBase.entity : null),
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Returns whether this type is an instantiated generic type.
         */
        public boolean pIsGeneric(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_type_decl_p_is_generic(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_type_decl_p_is_generic(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * TypeDecl.
         */
        private static final class TypeDeclNone extends TypeDecl {
            TypeDeclNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * Alternative for an enum class decl.
     *
     * This node type has no derivation.
     */
    public static 
    class EnumClassAltDecl extends TypeDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ENUM_CLASS_ALT_DECL,
                false,
                false,
                EnumClassAltDecl.class,
                "EnumClassAltDecl",
                new String[] {
                    "f_syn_name"
                },
                new HashMap<>(
                    TypeDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final EnumClassAltDecl NONE =
            new EnumClassAltDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected EnumClassAltDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static EnumClassAltDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                EnumClassAltDecl.NONE :

                new EnumClassAltDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return EnumClassAltDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Function type.
     *
     * This node type has no derivation.
     */
    public static 
    class FunctionType extends TypeDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.FUNCTION_TYPE,
                false,
                false,
                FunctionType.class,
                "FunctionType",
                new String[] {
                    
                },
                new HashMap<>(
                    TypeDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final FunctionType NONE =
            new FunctionType(
                Entity.NONE
            );

        // ----- Constructors -----

        protected FunctionType(
            final Entity entity
        ) {
            super(entity);
        }

        public static FunctionType fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                FunctionType.NONE :

                new FunctionType(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return FunctionType.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Declaration of a generic formal type in a generic declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class GenericFormalTypeDecl extends TypeDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GENERIC_FORMAL_TYPE_DECL,
                false,
                false,
                GenericFormalTypeDecl.class,
                "GenericFormalTypeDecl",
                new String[] {
                    "f_has_class","f_syn_name"
                },
                new HashMap<>(
                    TypeDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GenericFormalTypeDecl.class.getMethod(
                        "fHasClass",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_has_class",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GENERIC_FORMAL_TYPE_DECL_F_HAS_CLASS
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GenericFormalTypeDecl NONE =
            new GenericFormalTypeDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GenericFormalTypeDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static GenericFormalTypeDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GenericFormalTypeDecl.NONE :

                new GenericFormalTypeDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GenericFormalTypeDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public ClassQualifier fHasClass(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_generic_formal_type_decl_f_has_class(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                ClassQualifier res = ClassQualifier.NONE;
                if(propException == null) {
                    res = ClassQualifier.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_generic_formal_type_decl_f_has_class(
                    this.entity
                );

                // Wrap and return the result
                return ClassQualifier.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Instantiated generic type.
     *
     * This node type has no derivation.
     */
    public static 
    class InstantiatedGenericType extends TypeDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.INSTANTIATED_GENERIC_TYPE,
                false,
                false,
                InstantiatedGenericType.class,
                "InstantiatedGenericType",
                new String[] {
                    
                },
                new HashMap<>(
                    TypeDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = InstantiatedGenericType.class.getMethod(
                        "pGetInnerType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_get_inner_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_INSTANTIATED_GENERIC_TYPE_P_GET_INNER_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = InstantiatedGenericType.class.getMethod(
                        "pGetActuals",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_get_actuals",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_INSTANTIATED_GENERIC_TYPE_P_GET_ACTUALS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = InstantiatedGenericType.class.getMethod(
                        "pGetInstantiatedType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_get_instantiated_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_INSTANTIATED_GENERIC_TYPE_P_GET_INSTANTIATED_TYPE
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final InstantiatedGenericType NONE =
            new InstantiatedGenericType(
                Entity.NONE
            );

        // ----- Constructors -----

        protected InstantiatedGenericType(
            final Entity entity
        ) {
            super(entity);
        }

        public static InstantiatedGenericType fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                InstantiatedGenericType.NONE :

                new InstantiatedGenericType(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return InstantiatedGenericType.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * Return the generic type that ``self`` instantiates.
         */
        public TypeDecl pGetInnerType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_instantiated_generic_type_p_get_inner_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl res = TypeDecl.NONE;
                if(propException == null) {
                    res = TypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_instantiated_generic_type_p_get_inner_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Return the declaration of types that were passed as generic actuals
         * to create ``self``.
         */
        public TypeDecl[] pGetActuals(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final WordPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_instantiated_generic_type_p_get_actuals(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl[] res = TypeDeclArrayWrapper.NONE;
                if(propException == null) {
                    res = TypeDeclArrayWrapper.wrap(resNative);

                    TypeDeclArrayWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final TypeDecl[] res = JNI_LIB.lkt_instantiated_generic_type_p_get_actuals(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the instantiated type decl with proper rebindings, that can
         * be used for typing and code generation.
         */
        public TypeDecl pGetInstantiatedType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_instantiated_generic_type_p_get_instantiated_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl res = TypeDecl.NONE;
                if(propException == null) {
                    res = TypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_instantiated_generic_type_p_get_instantiated_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeDecl.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Explicit named type declaration.
     *
     * Derived nodes: ``BasicClassDecl``, ``EnumTypeDecl``, ``StructDecl``,
     * ``TraitDecl``
     */
    public static abstract
    class NamedTypeDecl extends TypeDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                NamedTypeDecl.class,
                "NamedTypeDecl",
                new String[] {
                    
                },
                new HashMap<>(
                    TypeDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = NamedTypeDecl.class.getMethod(
                        "fDecls",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_decls",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_NAMED_TYPE_DECL_F_DECLS
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final NamedTypeDecl NONE =
            new NamedTypeDeclNone();

        // ----- Constructors -----

        protected NamedTypeDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static NamedTypeDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                NamedTypeDecl.NONE :

                (NamedTypeDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return NamedTypeDecl.description;
        }


        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public DeclBlock fDecls(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_named_type_decl_f_decls(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                DeclBlock res = DeclBlock.NONE;
                if(propException == null) {
                    res = DeclBlock.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_named_type_decl_f_decls(
                    this.entity
                );

                // Wrap and return the result
                return DeclBlock.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * NamedTypeDecl.
         */
        private static final class NamedTypeDeclNone extends NamedTypeDecl {
            NamedTypeDeclNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * Common ancestor for declarations of regular classes and enum classes.
     *
     * Derived nodes: ``ClassDecl``, ``EnumClassDecl``
     */
    public static abstract
    class BasicClassDecl extends NamedTypeDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                BasicClassDecl.class,
                "BasicClassDecl",
                new String[] {
                    "f_syn_name","f_syn_base_type","f_traits"
                },
                new HashMap<>(
                    NamedTypeDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BasicClassDecl NONE =
            new BasicClassDeclNone();

        // ----- Constructors -----

        protected BasicClassDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static BasicClassDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BasicClassDecl.NONE :

                (BasicClassDecl) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BasicClassDecl.description;
        }


        // ----- Field accessors -----


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * BasicClassDecl.
         */
        private static final class BasicClassDeclNone extends BasicClassDecl {
            BasicClassDeclNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * Declaration for a LK class. This only cover node classes for the moment,
     * but might be extended to support regular classes in the future.
     *
     * This node type has no derivation.
     */
    public static 
    class ClassDecl extends BasicClassDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.CLASS_DECL,
                false,
                false,
                ClassDecl.class,
                "ClassDecl",
                new String[] {
                    "f_syn_name","f_syn_base_type","f_traits","f_decls"
                },
                new HashMap<>(
                    BasicClassDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ClassDecl NONE =
            new ClassDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ClassDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static ClassDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ClassDecl.NONE :

                new ClassDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ClassDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Declaration for a LK class. This only cover node classes for the moment,
     * but might be extended to support regular classes in the future.
     *
     * This node type has no derivation.
     */
    public static 
    class EnumClassDecl extends BasicClassDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ENUM_CLASS_DECL,
                false,
                false,
                EnumClassDecl.class,
                "EnumClassDecl",
                new String[] {
                    "f_syn_name","f_syn_base_type","f_traits","f_branches","f_decls"
                },
                new HashMap<>(
                    BasicClassDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = EnumClassDecl.class.getMethod(
                        "fBranches",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_branches",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ENUM_CLASS_DECL_F_BRANCHES
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final EnumClassDecl NONE =
            new EnumClassDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected EnumClassDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static EnumClassDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                EnumClassDecl.NONE :

                new EnumClassDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return EnumClassDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public EnumClassCaseList fBranches(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_enum_class_decl_f_branches(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                EnumClassCaseList res = EnumClassCaseList.NONE;
                if(propException == null) {
                    res = EnumClassCaseList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_enum_class_decl_f_branches(
                    this.entity
                );

                // Wrap and return the result
                return EnumClassCaseList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Enum type declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class EnumTypeDecl extends NamedTypeDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ENUM_TYPE_DECL,
                false,
                false,
                EnumTypeDecl.class,
                "EnumTypeDecl",
                new String[] {
                    "f_syn_name","f_traits","f_literals","f_decls"
                },
                new HashMap<>(
                    NamedTypeDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = EnumTypeDecl.class.getMethod(
                        "fLiterals",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_literals",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ENUM_TYPE_DECL_F_LITERALS
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final EnumTypeDecl NONE =
            new EnumTypeDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected EnumTypeDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static EnumTypeDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                EnumTypeDecl.NONE :

                new EnumTypeDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return EnumTypeDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public EnumLitDeclList fLiterals(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_enum_type_decl_f_literals(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                EnumLitDeclList res = EnumLitDeclList.NONE;
                if(propException == null) {
                    res = EnumLitDeclList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_enum_type_decl_f_literals(
                    this.entity
                );

                // Wrap and return the result
                return EnumLitDeclList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Declaration for a LK struct.
     *
     * This node type has no derivation.
     */
    public static 
    class StructDecl extends NamedTypeDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.STRUCT_DECL,
                false,
                false,
                StructDecl.class,
                "StructDecl",
                new String[] {
                    "f_syn_name","f_traits","f_decls"
                },
                new HashMap<>(
                    NamedTypeDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final StructDecl NONE =
            new StructDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected StructDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static StructDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                StructDecl.NONE :

                new StructDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return StructDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Trait declaration. For the moment, a Trait can just be used to group
     * behavior for built-in types. It's not usable as a type-bound since we
     * don't have generics, and you cannot implement one either.
     *
     * The reason they're added is to lay down the basics of what we want the
     * Lkt type system to be.
     *
     * TODO: Traits are *not* types. They're treated as such in the grammar for
     * convenience for now, but it's probably not a good idea. Migrate away
     * from this.
     *
     * This node type has no derivation.
     */
    public static 
    class TraitDecl extends NamedTypeDecl {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.TRAIT_DECL,
                false,
                false,
                TraitDecl.class,
                "TraitDecl",
                new String[] {
                    "f_syn_name","f_decls"
                },
                new HashMap<>(
                    NamedTypeDecl.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final TraitDecl NONE =
            new TraitDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected TraitDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static TraitDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                TraitDecl.NONE :

                new TraitDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return TraitDecl.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Compile time annotation attached to a declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class DeclAnnotation extends LktNode {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.DECL_ANNOTATION,
                false,
                false,
                DeclAnnotation.class,
                "DeclAnnotation",
                new String[] {
                    "f_name","f_params"
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = DeclAnnotation.class.getMethod(
                        "fName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_ANNOTATION_F_NAME
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = DeclAnnotation.class.getMethod(
                        "fParams",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_params",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_ANNOTATION_F_PARAMS
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final DeclAnnotation NONE =
            new DeclAnnotation(
                Entity.NONE
            );

        // ----- Constructors -----

        protected DeclAnnotation(
            final Entity entity
        ) {
            super(entity);
        }

        public static DeclAnnotation fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                DeclAnnotation.NONE :

                new DeclAnnotation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return DeclAnnotation.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public Id fName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_decl_annotation_f_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Id res = Id.NONE;
                if(propException == null) {
                    res = Id.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_decl_annotation_f_name(
                    this.entity
                );

                // Wrap and return the result
                return Id.fromEntity(res);
            }

        }

        
    

        /**
         * This field may be null even when there are no parsing errors.
         */
        public DeclAnnotationParams fParams(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_decl_annotation_f_params(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                DeclAnnotationParams res = DeclAnnotationParams.NONE;
                if(propException == null) {
                    res = DeclAnnotationParams.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_decl_annotation_f_params(
                    this.entity
                );

                // Wrap and return the result
                return DeclAnnotationParams.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * List of arguments for an annotation with a call syntax. This
     * intermediate node is necessary in order to determine after parsing
     * whether there is no param list, or if the list is empty.
     *
     * This node type has no derivation.
     */
    public static 
    class DeclAnnotationParams extends LktNode {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.DECL_ANNOTATION_PARAMS,
                false,
                false,
                DeclAnnotationParams.class,
                "DeclAnnotationParams",
                new String[] {
                    "f_params"
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = DeclAnnotationParams.class.getMethod(
                        "fParams",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_params",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_DECL_ANNOTATION_PARAMS_F_PARAMS
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final DeclAnnotationParams NONE =
            new DeclAnnotationParams(
                Entity.NONE
            );

        // ----- Constructors -----

        protected DeclAnnotationParams(
            final Entity entity
        ) {
            super(entity);
        }

        public static DeclAnnotationParams fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                DeclAnnotationParams.NONE :

                new DeclAnnotationParams(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return DeclAnnotationParams.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public ParamList fParams(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_decl_annotation_params_f_params(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                ParamList res = ParamList.NONE;
                if(propException == null) {
                    res = ParamList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_decl_annotation_params_f_params(
                    this.entity
                );

                // Wrap and return the result
                return ParamList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Elsif branch of an if expression.
     *
     * This node type has no derivation.
     */
    public static 
    class ElsifBranch extends LktNode {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ELSIF_BRANCH,
                false,
                false,
                ElsifBranch.class,
                "ElsifBranch",
                new String[] {
                    "f_cond_expr","f_then_expr"
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ElsifBranch.class.getMethod(
                        "fCondExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_cond_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ELSIF_BRANCH_F_COND_EXPR
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = ElsifBranch.class.getMethod(
                        "fThenExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_then_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ELSIF_BRANCH_F_THEN_EXPR
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ElsifBranch NONE =
            new ElsifBranch(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ElsifBranch(
            final Entity entity
        ) {
            super(entity);
        }

        public static ElsifBranch fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ElsifBranch.NONE :

                new ElsifBranch(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ElsifBranch.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BaseDotExpr``, ``BinOp``, ``BlockExpr``,
         * ``CallExpr``, ``CastExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``, ``NotExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``, ``UnOp``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fCondExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_elsif_branch_f_cond_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_elsif_branch_f_cond_expr(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BaseDotExpr``, ``BinOp``, ``BlockExpr``,
         * ``CallExpr``, ``CastExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``, ``NotExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``, ``UnOp``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fThenExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_elsif_branch_f_then_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_elsif_branch_f_then_expr(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Case branch for an enum class declaration.
     *
     * This node type has no derivation.
     */
    public static 
    class EnumClassCase extends LktNode {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ENUM_CLASS_CASE,
                false,
                false,
                EnumClassCase.class,
                "EnumClassCase",
                new String[] {
                    "f_decls"
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = EnumClassCase.class.getMethod(
                        "fDecls",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_decls",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ENUM_CLASS_CASE_F_DECLS
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final EnumClassCase NONE =
            new EnumClassCase(
                Entity.NONE
            );

        // ----- Constructors -----

        protected EnumClassCase(
            final Entity entity
        ) {
            super(entity);
        }

        public static EnumClassCase fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                EnumClassCase.NONE :

                new EnumClassCase(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return EnumClassCase.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public EnumClassAltDeclList fDecls(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_enum_class_case_f_decls(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                EnumClassAltDeclList res = EnumClassAltDeclList.NONE;
                if(propException == null) {
                    res = EnumClassAltDeclList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_enum_class_case_f_decls(
                    this.entity
                );

                // Wrap and return the result
                return EnumClassAltDeclList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Whether the containing cast expression will raise on null cast result or
     * not.
     *
     * Derived nodes: ``ExcludesNullAbsent``, ``ExcludesNullPresent``
     */
    public static abstract
    class ExcludesNull extends LktNode {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                ExcludesNull.class,
                "ExcludesNull",
                new String[] {
                    
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ExcludesNull.class.getMethod(
                        "pAsBool",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_as_bool",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_EXCLUDES_NULL_P_AS_BOOL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ExcludesNull NONE =
            new ExcludesNullNone();

        // ----- Constructors -----

        protected ExcludesNull(
            final Entity entity
        ) {
            super(entity);
        }

        public static ExcludesNull fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ExcludesNull.NONE :

                (ExcludesNull) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ExcludesNull.description;
        }


        // ----- Field accessors -----

        
    

        /**
         * Return whether this is an instance of ExcludesNullPresent
         */
        public boolean pAsBool(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_excludes_null_p_as_bool(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_excludes_null_p_as_bool(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * ExcludesNull.
         */
        private static final class ExcludesNullNone extends ExcludesNull {
            ExcludesNullNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * This node type has no derivation.
     */
    public static 
    class ExcludesNullAbsent extends ExcludesNull {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.EXCLUDES_NULL_ABSENT,
                false,
                false,
                ExcludesNullAbsent.class,
                "ExcludesNullAbsent",
                new String[] {
                    
                },
                new HashMap<>(
                    ExcludesNull.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ExcludesNullAbsent NONE =
            new ExcludesNullAbsent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ExcludesNullAbsent(
            final Entity entity
        ) {
            super(entity);
        }

        public static ExcludesNullAbsent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ExcludesNullAbsent.NONE :

                new ExcludesNullAbsent(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ExcludesNullAbsent.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * This node type has no derivation.
     */
    public static 
    class ExcludesNullPresent extends ExcludesNull {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.EXCLUDES_NULL_PRESENT,
                false,
                false,
                ExcludesNullPresent.class,
                "ExcludesNullPresent",
                new String[] {
                    
                },
                new HashMap<>(
                    ExcludesNull.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ExcludesNullPresent NONE =
            new ExcludesNullPresent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ExcludesNullPresent(
            final Entity entity
        ) {
            super(entity);
        }

        public static ExcludesNullPresent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ExcludesNullPresent.NONE :

                new ExcludesNullPresent(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ExcludesNullPresent.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Base class for expressions. Encompasses regular expressions as well as
     * special expressions (grammar expressions, etc).
     *
     * Derived nodes: ``AnyOf``, ``ArrayLiteral``, ``BaseDotExpr``, ``BinOp``,
     * ``BlockExpr``, ``CallExpr``, ``CastExpr``, ``ErrorOnNull``,
     * ``GenericInstantiation``, ``GrammarExpr``, ``Id``, ``IfExpr``, ``Isa``,
     * ``KeepExpr``, ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``,
     * ``NotExpr``, ``ParenExpr``, ``RaiseExpr``, ``SubscriptExpr``,
     * ``TryExpr``, ``UnOp``
     */
    public static abstract
    class Expr extends LktNode {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                Expr.class,
                "Expr",
                new String[] {
                    
                },
                new HashMap<>(
                    LktNode.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = Expr.class.getMethod(
                        "pInTypeRef",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_in_type_ref",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_EXPR_P_IN_TYPE_REF
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Expr.class.getMethod(
                        "pIsRegularExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_is_regular_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_EXPR_P_IS_REGULAR_EXPR
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Expr.class.getMethod(
                        "pExprType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_expr_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_EXPR_P_EXPR_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Expr.class.getMethod(
                        "pCheckExprType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_check_expr_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_EXPR_P_CHECK_EXPR_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Expr.class.getMethod(
                        "pExprContextFreeType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_expr_context_free_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_EXPR_P_EXPR_CONTEXT_FREE_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Expr.class.getMethod(
                        "pReferencedDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_referenced_decl",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_EXPR_P_REFERENCED_DECL
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Expr.class.getMethod(
                        "pCheckReferencedDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_check_referenced_decl",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_EXPR_P_CHECK_REFERENCED_DECL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final Expr NONE =
            new ExprNone();

        // ----- Constructors -----

        protected Expr(
            final Entity entity
        ) {
            super(entity);
        }

        public static Expr fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                Expr.NONE :

                (Expr) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return Expr.description;
        }


        // ----- Field accessors -----

        
    

        /**
         * Return whether this expression is part of a type reference.
         */
        public boolean pInTypeRef(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_expr_p_in_type_ref(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_expr_p_in_type_ref(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return whether this expression is a regular expression that can be
         * evaluated at runtime, in particular:
         *
         * * Not part of a type reference.
         *
         * * Not part of a declaration annotation.
         *
         * * Not a defining identifier.
         *
         * .. TODO: List to be expanded probably to take into account grammar
         *    expressions.
         */
        public boolean pIsRegularExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(WordPointer.class));

                // Call the native function
                NI_LIB.lkt_expr_p_is_regular_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = BooleanWrapper.wrap(resNative);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.lkt_expr_p_is_regular_expr(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the type of this expression, if it is a regular expression
         * (see ``is_regular_expr``), null otherwise.
         */
        public SemanticResult pExprType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final SemanticResultNative resNative =
                    StackValue.get(SemanticResultNative.class);

                // Call the native function
                NI_LIB.lkt_expr_p_expr_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                SemanticResult res = SemanticResult.NONE;
                if(propException == null) {
                    res = SemanticResult.wrap(resNative);

                    SemanticResult.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final SemanticResult res = JNI_LIB.lkt_expr_p_expr_type(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the type of this expression. Assumes that this is a regular
         * and valid expression. If this is called on a non regular or non
         * valid expression, it will raise an error.
         */
        public TypeDecl pCheckExprType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_expr_p_check_expr_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl res = TypeDecl.NONE;
                if(propException == null) {
                    res = TypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_expr_p_check_expr_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * If the type of this expression can be determined with no bottom up
         * context, return it. This will be used by ``expr_type_impl``'s
         * default implementation.
         */
        public TypeDecl pExprContextFreeType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_expr_p_expr_context_free_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl res = TypeDecl.NONE;
                if(propException == null) {
                    res = TypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_expr_p_expr_context_free_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Return the declaration referenced by this expression, if applicable,
         * null otherwise.
         */
        public SemanticResult pReferencedDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final SemanticResultNative resNative =
                    StackValue.get(SemanticResultNative.class);

                // Call the native function
                NI_LIB.lkt_expr_p_referenced_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                SemanticResult res = SemanticResult.NONE;
                if(propException == null) {
                    res = SemanticResult.wrap(resNative);

                    SemanticResult.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final SemanticResult res = JNI_LIB.lkt_expr_p_referenced_decl(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the referenced decl of this expr, raise otherwise.
         */
        public Decl pCheckReferencedDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_expr_p_check_referenced_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Decl res = Decl.NONE;
                if(propException == null) {
                    res = Decl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_expr_p_check_referenced_decl(
                    this.entity
                );

                // Wrap and return the result
                return Decl.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * Expr.
         */
        private static final class ExprNone extends Expr {
            ExprNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * "Any of" expression.
     *
     * This node type has no derivation.
     */
    public static 
    class AnyOf extends Expr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ANY_OF,
                false,
                false,
                AnyOf.class,
                "AnyOf",
                new String[] {
                    "f_expr","f_values"
                },
                new HashMap<>(
                    Expr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = AnyOf.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ANY_OF_F_EXPR
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AnyOf.class.getMethod(
                        "fValues",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_values",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ANY_OF_F_VALUES
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AnyOf NONE =
            new AnyOf(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AnyOf(
            final Entity entity
        ) {
            super(entity);
        }

        public static AnyOf fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AnyOf.NONE :

                new AnyOf(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return AnyOf.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes: ``ArrayLiteral``,
         * ``BaseDotExpr``, ``BlockExpr``, ``CallExpr``, ``CastExpr``,
         * ``ErrorOnNull``, ``GenericInstantiation``, ``IfExpr``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_any_of_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_any_of_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }

        
    

        /**
         * This field contains a list that itself contains one of the following
         * nodes: ``ArrayLiteral``, ``BaseDotExpr``, ``BlockExpr``,
         * ``CallExpr``, ``CastExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``KeepExpr``, ``LambdaExpr``,
         * ``Lit``, ``LogicExpr``, ``MatchExpr``, ``ParenExpr``, ``RaiseExpr``,
         * ``RefId``, ``SubscriptExpr``, ``TryExpr``
         *
         * When there are no parsing errors, this field is never null.
         */
        public AnyOfList fValues(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_any_of_f_values(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AnyOfList res = AnyOfList.NONE;
                if(propException == null) {
                    res = AnyOfList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_any_of_f_values(
                    this.entity
                );

                // Wrap and return the result
                return AnyOfList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Literal for an array value.
     *
     * This node type has no derivation.
     */
    public static 
    class ArrayLiteral extends Expr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ARRAY_LITERAL,
                false,
                false,
                ArrayLiteral.class,
                "ArrayLiteral",
                new String[] {
                    "f_exprs","f_element_type"
                },
                new HashMap<>(
                    Expr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ArrayLiteral.class.getMethod(
                        "fExprs",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_exprs",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ARRAY_LITERAL_F_EXPRS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = ArrayLiteral.class.getMethod(
                        "fElementType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_element_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ARRAY_LITERAL_F_ELEMENT_TYPE
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ArrayLiteral NONE =
            new ArrayLiteral(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ArrayLiteral(
            final Entity entity
        ) {
            super(entity);
        }

        public static ArrayLiteral fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ArrayLiteral.NONE :

                new ArrayLiteral(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ArrayLiteral.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field contains a list that itself contains one of the following
         * nodes: ``AnyOf``, ``ArrayLiteral``, ``BaseDotExpr``, ``BinOp``,
         * ``BlockExpr``, ``CallExpr``, ``CastExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``, ``NotExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``, ``UnOp``
         *
         * When there are no parsing errors, this field is never null.
         */
        public ExprList fExprs(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_array_literal_f_exprs(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                ExprList res = ExprList.NONE;
                if(propException == null) {
                    res = ExprList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_array_literal_f_exprs(
                    this.entity
                );

                // Wrap and return the result
                return ExprList.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes:
         * ``FunctionTypeRef``, ``GenericTypeRef``, ``SimpleTypeRef``
         *
         * This field may be null even when there are no parsing errors.
         */
        public TypeRef fElementType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_array_literal_f_element_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeRef res = TypeRef.NONE;
                if(propException == null) {
                    res = TypeRef.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_array_literal_f_element_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeRef.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Base class for regular dotted expressions and null-conditional ones.
     *
     * Derived nodes: ``DotExpr``, ``NullCondDottedName``
     */
    public static abstract
    class BaseDotExpr extends Expr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                BaseDotExpr.class,
                "BaseDotExpr",
                new String[] {
                    "f_prefix","f_suffix"
                },
                new HashMap<>(
                    Expr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = BaseDotExpr.class.getMethod(
                        "fPrefix",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_prefix",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BASE_DOT_EXPR_F_PREFIX
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = BaseDotExpr.class.getMethod(
                        "fSuffix",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_suffix",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BASE_DOT_EXPR_F_SUFFIX
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BaseDotExpr NONE =
            new BaseDotExprNone();

        // ----- Constructors -----

        protected BaseDotExpr(
            final Entity entity
        ) {
            super(entity);
        }

        public static BaseDotExpr fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BaseDotExpr.NONE :

                (BaseDotExpr) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BaseDotExpr.description;
        }


        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes: ``ArrayLiteral``,
         * ``BaseDotExpr``, ``BlockExpr``, ``CallExpr``, ``CastExpr``,
         * ``ErrorOnNull``, ``GenericInstantiation``, ``KeepExpr``, ``Lit``,
         * ``LogicExpr``, ``MatchExpr``, ``ParenExpr``, ``RefId``,
         * ``SubscriptExpr``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fPrefix(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_base_dot_expr_f_prefix(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_base_dot_expr_f_prefix(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public RefId fSuffix(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_base_dot_expr_f_suffix(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                RefId res = RefId.NONE;
                if(propException == null) {
                    res = RefId.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_base_dot_expr_f_suffix(
                    this.entity
                );

                // Wrap and return the result
                return RefId.fromEntity(res);
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * BaseDotExpr.
         */
        private static final class BaseDotExprNone extends BaseDotExpr {
            BaseDotExprNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * Dotted expression.
     *
     * This node type has no derivation.
     */
    public static 
    class DotExpr extends BaseDotExpr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.DOT_EXPR,
                false,
                false,
                DotExpr.class,
                "DotExpr",
                new String[] {
                    "f_prefix","f_suffix"
                },
                new HashMap<>(
                    BaseDotExpr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final DotExpr NONE =
            new DotExpr(
                Entity.NONE
            );

        // ----- Constructors -----

        protected DotExpr(
            final Entity entity
        ) {
            super(entity);
        }

        public static DotExpr fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                DotExpr.NONE :

                new DotExpr(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return DotExpr.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Null conditional dotted expression.
     *
     * This node type has no derivation.
     */
    public static 
    class NullCondDottedName extends BaseDotExpr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.NULL_COND_DOTTED_NAME,
                false,
                false,
                NullCondDottedName.class,
                "NullCondDottedName",
                new String[] {
                    "f_prefix","f_suffix"
                },
                new HashMap<>(
                    BaseDotExpr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final NullCondDottedName NONE =
            new NullCondDottedName(
                Entity.NONE
            );

        // ----- Constructors -----

        protected NullCondDottedName(
            final Entity entity
        ) {
            super(entity);
        }

        public static NullCondDottedName fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                NullCondDottedName.NONE :

                new NullCondDottedName(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return NullCondDottedName.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Binary operator expression.
     *
     * This node type has no derivation.
     */
    public static 
    class BinOp extends Expr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.BIN_OP,
                false,
                false,
                BinOp.class,
                "BinOp",
                new String[] {
                    "f_left","f_op","f_right"
                },
                new HashMap<>(
                    Expr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = BinOp.class.getMethod(
                        "fLeft",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_left",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BIN_OP_F_LEFT
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = BinOp.class.getMethod(
                        "fOp",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_op",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BIN_OP_F_OP
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = BinOp.class.getMethod(
                        "fRight",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_right",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BIN_OP_F_RIGHT
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BinOp NONE =
            new BinOp(
                Entity.NONE
            );

        // ----- Constructors -----

        protected BinOp(
            final Entity entity
        ) {
            super(entity);
        }

        public static BinOp fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BinOp.NONE :

                new BinOp(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BinOp.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BaseDotExpr``, ``BinOp``, ``BlockExpr``,
         * ``CallExpr``, ``CastExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``, ``NotExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``, ``UnOp``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fLeft(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_bin_op_f_left(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_bin_op_f_left(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public Op fOp(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_bin_op_f_op(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Op res = Op.NONE;
                if(propException == null) {
                    res = Op.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_bin_op_f_op(
                    this.entity
                );

                // Wrap and return the result
                return Op.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BaseDotExpr``, ``BinOp``, ``BlockExpr``,
         * ``CallExpr``, ``CastExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``, ``NotExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``, ``UnOp``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fRight(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_bin_op_f_right(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_bin_op_f_right(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Block expression.
     *
     * This node type has no derivation.
     */
    public static 
    class BlockExpr extends Expr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.BLOCK_EXPR,
                false,
                false,
                BlockExpr.class,
                "BlockExpr",
                new String[] {
                    "f_val_defs","f_expr"
                },
                new HashMap<>(
                    Expr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = BlockExpr.class.getMethod(
                        "fValDefs",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_val_defs",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BLOCK_EXPR_F_VAL_DEFS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = BlockExpr.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_BLOCK_EXPR_F_EXPR
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BlockExpr NONE =
            new BlockExpr(
                Entity.NONE
            );

        // ----- Constructors -----

        protected BlockExpr(
            final Entity entity
        ) {
            super(entity);
        }

        public static BlockExpr fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BlockExpr.NONE :

                new BlockExpr(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return BlockExpr.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field contains a list that itself contains one of the following
         * nodes: ``ValDecl``, ``VarBind``
         *
         * When there are no parsing errors, this field is never null.
         */
        public BlockDeclList fValDefs(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_block_expr_f_val_defs(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                BlockDeclList res = BlockDeclList.NONE;
                if(propException == null) {
                    res = BlockDeclList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_block_expr_f_val_defs(
                    this.entity
                );

                // Wrap and return the result
                return BlockDeclList.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes: ``AnyOf``,
         * ``ArrayLiteral``, ``BaseDotExpr``, ``BinOp``, ``BlockExpr``,
         * ``CallExpr``, ``CastExpr``, ``ErrorOnNull``,
         * ``GenericInstantiation``, ``IfExpr``, ``Isa``, ``KeepExpr``,
         * ``LambdaExpr``, ``Lit``, ``LogicExpr``, ``MatchExpr``, ``NotExpr``,
         * ``ParenExpr``, ``RaiseExpr``, ``RefId``, ``SubscriptExpr``,
         * ``TryExpr``, ``UnOp``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_block_expr_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_block_expr_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Call expression.
     *
     * This node type has no derivation.
     */
    public static 
    class CallExpr extends Expr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.CALL_EXPR,
                false,
                false,
                CallExpr.class,
                "CallExpr",
                new String[] {
                    "f_name","f_args"
                },
                new HashMap<>(
                    Expr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = CallExpr.class.getMethod(
                        "fName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CALL_EXPR_F_NAME
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = CallExpr.class.getMethod(
                        "fArgs",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_args",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CALL_EXPR_F_ARGS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = CallExpr.class.getMethod(
                        "pCalledObjectType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_called_object_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CALL_EXPR_P_CALLED_OBJECT_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = CallExpr.class.getMethod(
                        "pCalledDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_called_decl",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CALL_EXPR_P_CALLED_DECL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final CallExpr NONE =
            new CallExpr(
                Entity.NONE
            );

        // ----- Constructors -----

        protected CallExpr(
            final Entity entity
        ) {
            super(entity);
        }

        public static CallExpr fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                CallExpr.NONE :

                new CallExpr(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return CallExpr.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes: ``ArrayLiteral``,
         * ``BaseDotExpr``, ``BlockExpr``, ``CallExpr``, ``CastExpr``,
         * ``ErrorOnNull``, ``GenericInstantiation``, ``KeepExpr``, ``Lit``,
         * ``LogicExpr``, ``MatchExpr``, ``ParenExpr``, ``RefId``,
         * ``SubscriptExpr``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_call_expr_f_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_call_expr_f_name(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public ParamList fArgs(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_call_expr_f_args(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                ParamList res = ParamList.NONE;
                if(propException == null) {
                    res = ParamList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_call_expr_f_args(
                    this.entity
                );

                // Wrap and return the result
                return ParamList.fromEntity(res);
            }

        }

        
    

        /**
         * Return the type of the called object.
         */
        public TypeDecl pCalledObjectType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_call_expr_p_called_object_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeDecl res = TypeDecl.NONE;
                if(propException == null) {
                    res = TypeDecl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_call_expr_p_called_object_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeDecl.fromEntity(res);
            }

        }

        
    

        /**
         * Return the declaration that is called by this call expression, if
         * there is one that can be statically determined.
         */
        public Decl pCalledDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_call_expr_p_called_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Decl res = Decl.NONE;
                if(propException == null) {
                    res = Decl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_call_expr_p_called_decl(
                    this.entity
                );

                // Wrap and return the result
                return Decl.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Cast expression.
     *
     * This node type has no derivation.
     */
    public static 
    class CastExpr extends Expr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.CAST_EXPR,
                false,
                false,
                CastExpr.class,
                "CastExpr",
                new String[] {
                    "f_expr","f_excludes_null","f_dest_type"
                },
                new HashMap<>(
                    Expr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = CastExpr.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CAST_EXPR_F_EXPR
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = CastExpr.class.getMethod(
                        "fExcludesNull",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_excludes_null",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CAST_EXPR_F_EXCLUDES_NULL
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = CastExpr.class.getMethod(
                        "fDestType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_dest_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_CAST_EXPR_F_DEST_TYPE
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final CastExpr NONE =
            new CastExpr(
                Entity.NONE
            );

        // ----- Constructors -----

        protected CastExpr(
            final Entity entity
        ) {
            super(entity);
        }

        public static CastExpr fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                CastExpr.NONE :

                new CastExpr(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return CastExpr.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes: ``ArrayLiteral``,
         * ``BaseDotExpr``, ``BlockExpr``, ``CallExpr``, ``CastExpr``,
         * ``ErrorOnNull``, ``GenericInstantiation``, ``KeepExpr``, ``Lit``,
         * ``LogicExpr``, ``MatchExpr``, ``ParenExpr``, ``RefId``,
         * ``SubscriptExpr``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_cast_expr_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_cast_expr_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public ExcludesNull fExcludesNull(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_cast_expr_f_excludes_null(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                ExcludesNull res = ExcludesNull.NONE;
                if(propException == null) {
                    res = ExcludesNull.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_cast_expr_f_excludes_null(
                    this.entity
                );

                // Wrap and return the result
                return ExcludesNull.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes:
         * ``FunctionTypeRef``, ``GenericTypeRef``, ``SimpleTypeRef``
         *
         * When there are no parsing errors, this field is never null.
         */
        public TypeRef fDestType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_cast_expr_f_dest_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeRef res = TypeRef.NONE;
                if(propException == null) {
                    res = TypeRef.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_cast_expr_f_dest_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeRef.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Expression that throws an error if LHS is null.
     *
     * This node type has no derivation.
     */
    public static 
    class ErrorOnNull extends Expr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.ERROR_ON_NULL,
                false,
                false,
                ErrorOnNull.class,
                "ErrorOnNull",
                new String[] {
                    "f_expr"
                },
                new HashMap<>(
                    Expr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ErrorOnNull.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_ERROR_ON_NULL_F_EXPR
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ErrorOnNull NONE =
            new ErrorOnNull(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ErrorOnNull(
            final Entity entity
        ) {
            super(entity);
        }

        public static ErrorOnNull fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ErrorOnNull.NONE :

                new ErrorOnNull(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return ErrorOnNull.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes: ``ArrayLiteral``,
         * ``BaseDotExpr``, ``BlockExpr``, ``CallExpr``, ``CastExpr``,
         * ``ErrorOnNull``, ``GenericInstantiation``, ``KeepExpr``, ``Lit``,
         * ``LogicExpr``, ``MatchExpr``, ``ParenExpr``, ``RefId``,
         * ``SubscriptExpr``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_error_on_null_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_error_on_null_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Generic instantiation.
     *
     * This node type has no derivation.
     */
    public static 
    class GenericInstantiation extends Expr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GENERIC_INSTANTIATION,
                false,
                false,
                GenericInstantiation.class,
                "GenericInstantiation",
                new String[] {
                    "f_name","f_args"
                },
                new HashMap<>(
                    Expr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GenericInstantiation.class.getMethod(
                        "fName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GENERIC_INSTANTIATION_F_NAME
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GenericInstantiation.class.getMethod(
                        "fArgs",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_args",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GENERIC_INSTANTIATION_F_ARGS
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GenericInstantiation.class.getMethod(
                        "pInstantiatedDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "p_instantiated_decl",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GENERIC_INSTANTIATION_P_INSTANTIATED_DECL
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GenericInstantiation NONE =
            new GenericInstantiation(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GenericInstantiation(
            final Entity entity
        ) {
            super(entity);
        }

        public static GenericInstantiation fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GenericInstantiation.NONE :

                new GenericInstantiation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GenericInstantiation.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes: ``ArrayLiteral``,
         * ``BaseDotExpr``, ``BlockExpr``, ``CallExpr``, ``CastExpr``,
         * ``ErrorOnNull``, ``GenericInstantiation``, ``KeepExpr``, ``Lit``,
         * ``LogicExpr``, ``MatchExpr``, ``ParenExpr``, ``RefId``,
         * ``SubscriptExpr``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_generic_instantiation_f_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_generic_instantiation_f_name(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }

        
    

        /**
         * This field contains a list that itself contains one of the following
         * nodes: ``FunctionTypeRef``, ``GenericTypeRef``, ``SimpleTypeRef``
         *
         * When there are no parsing errors, this field is never null.
         */
        public TypeRefList fArgs(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_generic_instantiation_f_args(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeRefList res = TypeRefList.NONE;
                if(propException == null) {
                    res = TypeRefList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_generic_instantiation_f_args(
                    this.entity
                );

                // Wrap and return the result
                return TypeRefList.fromEntity(res);
            }

        }

        
    

        /**
         * Get the type designated by this instantiation.
         */
        public Decl pInstantiatedDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_generic_instantiation_p_instantiated_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Decl res = Decl.NONE;
                if(propException == null) {
                    res = Decl.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_generic_instantiation_p_instantiated_decl(
                    this.entity
                );

                // Wrap and return the result
                return Decl.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Base class for expressions related to grammars.
     *
     * Derived nodes: ``GrammarCut``, ``GrammarDiscard``, ``GrammarDontSkip``,
     * ``GrammarList``, ``GrammarNull``, ``GrammarOptErrorGroup``,
     * ``GrammarOptError``, ``GrammarOptGroup``, ``GrammarOpt``,
     * ``GrammarOrExpr``, ``GrammarPick``, ``GrammarPredicate``,
     * ``GrammarRuleRef``, ``GrammarSkip``, ``GrammarStopCut``,
     * ``ParseNodeExpr``, ``TokenLit``, ``TokenNoCaseLit``,
     * ``TokenPatternLit``, ``TokenRef``
     */
    public static abstract
    class GrammarExpr extends Expr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                null,
                false,
                false,
                GrammarExpr.class,
                "GrammarExpr",
                new String[] {
                    
                },
                new HashMap<>(
                    Expr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarExpr NONE =
            new GrammarExprNone();

        // ----- Constructors -----

        protected GrammarExpr(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarExpr fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarExpr.NONE :

                (GrammarExpr) LktNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarExpr.description;
        }


        // ----- Field accessors -----


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * GrammarExpr.
         */
        private static final class GrammarExprNone extends GrammarExpr {
            GrammarExprNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * Grammar expression for a cut.
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarCut extends GrammarExpr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_CUT,
                false,
                false,
                GrammarCut.class,
                "GrammarCut",
                new String[] {
                    
                },
                new HashMap<>(
                    GrammarExpr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarCut NONE =
            new GrammarCut(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GrammarCut(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarCut fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarCut.NONE :

                new GrammarCut(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarCut.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * Grammar expression to discard the match.
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarDiscard extends GrammarExpr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_DISCARD,
                false,
                false,
                GrammarDiscard.class,
                "GrammarDiscard",
                new String[] {
                    "f_expr"
                },
                new HashMap<>(
                    GrammarExpr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GrammarDiscard.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_DISCARD_F_EXPR
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarDiscard NONE =
            new GrammarDiscard(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GrammarDiscard(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarDiscard fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarDiscard.NONE :

                new GrammarDiscard(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarDiscard.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public GrammarExpr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_discard_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GrammarExpr res = GrammarExpr.NONE;
                if(propException == null) {
                    res = GrammarExpr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_discard_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return GrammarExpr.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Grammar expression (error recovery) to ensure that any nested skip
     * parser calls won't skip certain parse results.
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarDontSkip extends GrammarExpr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_DONT_SKIP,
                false,
                false,
                GrammarDontSkip.class,
                "GrammarDontSkip",
                new String[] {
                    "f_expr","f_dont_skip"
                },
                new HashMap<>(
                    GrammarExpr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GrammarDontSkip.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_DONT_SKIP_F_EXPR
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GrammarDontSkip.class.getMethod(
                        "fDontSkip",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_dont_skip",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_DONT_SKIP_F_DONT_SKIP
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarDontSkip NONE =
            new GrammarDontSkip(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GrammarDontSkip(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarDontSkip fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarDontSkip.NONE :

                new GrammarDontSkip(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarDontSkip.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public GrammarExpr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_dont_skip_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GrammarExpr res = GrammarExpr.NONE;
                if(propException == null) {
                    res = GrammarExpr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_dont_skip_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return GrammarExpr.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public GrammarExpr fDontSkip(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_dont_skip_f_dont_skip(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GrammarExpr res = GrammarExpr.NONE;
                if(propException == null) {
                    res = GrammarExpr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_dont_skip_f_dont_skip(
                    this.entity
                );

                // Wrap and return the result
                return GrammarExpr.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Grammar expression to parse lists of results. Results can be separated
     * by a separator. List can be empty ('*') or not ('+').
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarList extends GrammarExpr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_LIST,
                false,
                false,
                GrammarList.class,
                "GrammarList",
                new String[] {
                    "f_list_type","f_kind","f_expr","f_sep"
                },
                new HashMap<>(
                    GrammarExpr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GrammarList.class.getMethod(
                        "fListType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_list_type",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_LIST_F_LIST_TYPE
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GrammarList.class.getMethod(
                        "fKind",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_kind",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_LIST_F_KIND
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GrammarList.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_LIST_F_EXPR
                        )
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GrammarList.class.getMethod(
                        "fSep",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_sep",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_LIST_F_SEP
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarList NONE =
            new GrammarList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GrammarList(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarList.NONE :

                new GrammarList(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarList.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public TypeRef fListType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_list_f_list_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeRef res = TypeRef.NONE;
                if(propException == null) {
                    res = TypeRef.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_list_f_list_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeRef.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public ListKind fKind(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_list_f_kind(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                ListKind res = ListKind.NONE;
                if(propException == null) {
                    res = ListKind.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_list_f_kind(
                    this.entity
                );

                // Wrap and return the result
                return ListKind.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public GrammarExpr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_list_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GrammarExpr res = GrammarExpr.NONE;
                if(propException == null) {
                    res = GrammarExpr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_list_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return GrammarExpr.fromEntity(res);
            }

        }

        
    

        /**
         * This field may be null even when there are no parsing errors.
         */
        public GrammarListSep fSep(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_list_f_sep(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GrammarListSep res = GrammarListSep.NONE;
                if(propException == null) {
                    res = GrammarListSep.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_list_f_sep(
                    this.entity
                );

                // Wrap and return the result
                return GrammarListSep.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Grammar expression to parse a null node.
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarNull extends GrammarExpr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_NULL,
                false,
                false,
                GrammarNull.class,
                "GrammarNull",
                new String[] {
                    "f_name"
                },
                new HashMap<>(
                    GrammarExpr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GrammarNull.class.getMethod(
                        "fName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_name",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_NULL_F_NAME
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarNull NONE =
            new GrammarNull(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GrammarNull(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarNull fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarNull.NONE :

                new GrammarNull(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarNull.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes:
         * ``FunctionTypeRef``, ``GenericTypeRef``, ``SimpleTypeRef``
         *
         * When there are no parsing errors, this field is never null.
         */
        public TypeRef fName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_null_f_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeRef res = TypeRef.NONE;
                if(propException == null) {
                    res = TypeRef.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_null_f_name(
                    this.entity
                );

                // Wrap and return the result
                return TypeRef.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Grammar expression for an optional parsing result.
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarOpt extends GrammarExpr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_OPT,
                false,
                false,
                GrammarOpt.class,
                "GrammarOpt",
                new String[] {
                    "f_expr"
                },
                new HashMap<>(
                    GrammarExpr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GrammarOpt.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_OPT_F_EXPR
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarOpt NONE =
            new GrammarOpt(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GrammarOpt(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarOpt fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarOpt.NONE :

                new GrammarOpt(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarOpt.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public GrammarExpr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_opt_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GrammarExpr res = GrammarExpr.NONE;
                if(propException == null) {
                    res = GrammarExpr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_opt_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return GrammarExpr.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Grammar expression for an optional parsing result. Missing result
     * creates an error, but parsing continues.
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarOptError extends GrammarExpr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_OPT_ERROR,
                false,
                false,
                GrammarOptError.class,
                "GrammarOptError",
                new String[] {
                    "f_expr"
                },
                new HashMap<>(
                    GrammarExpr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GrammarOptError.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_OPT_ERROR_F_EXPR
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarOptError NONE =
            new GrammarOptError(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GrammarOptError(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarOptError fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarOptError.NONE :

                new GrammarOptError(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarOptError.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public GrammarExpr fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_opt_error_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GrammarExpr res = GrammarExpr.NONE;
                if(propException == null) {
                    res = GrammarExpr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_opt_error_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return GrammarExpr.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Grammar expression for a group of optional parsing results. Failure to
     * parse an optional result creates an error, but parsing continues.
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarOptErrorGroup extends GrammarExpr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_OPT_ERROR_GROUP,
                false,
                false,
                GrammarOptErrorGroup.class,
                "GrammarOptErrorGroup",
                new String[] {
                    "f_expr"
                },
                new HashMap<>(
                    GrammarExpr.description.fieldDescriptions
                )
            );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GrammarOptErrorGroup.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Reflection.Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    description.fieldDescriptions.put(
                        "f_expr",
                        new Reflection.Field(
                            method,
                            parameters,
                            MemberReference.LKT_GRAMMAR_OPT_ERROR_GROUP_F_EXPR
                        )
                    );
                }
            } catch (Exception e) {
                // This catch block handles exceptions from the Java reflection
                // API. Since calls to this API are generated, those exceptions
                // cannot be raised unless the Java bindings are erroneous.
                System.err.println(
                    "ERROR DURING LIBLKTLANG STATIC INITIALISATION"
                );
                e.printStackTrace();
                System.exit(1);
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GrammarOptErrorGroup NONE =
            new GrammarOptErrorGroup(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GrammarOptErrorGroup(
            final Entity entity
        ) {
            super(entity);
        }

        public static GrammarOptErrorGroup fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GrammarOptErrorGroup.NONE :

                new GrammarOptErrorGroup(entity);
        }

        // ----- Instance methods -----

        @Override
        public Reflection.Node getDescription() {
            return GrammarOptErrorGroup.description;
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public GrammarExprList fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.lkt_grammar_opt_error_group_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GrammarExprList res = GrammarExprList.NONE;
                if(propException == null) {
                    res = GrammarExprList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.lkt_grammar_opt_error_group_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return GrammarExprList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**
     * Grammar expression for a group of optional parsing results.
     *
     * This node type has no derivation.
     */
    public static 
    class GrammarOptGroup extends GrammarExpr {

        // ----- Static -----

        
    

        /** Full description of the node (kind, fields, class...) */
        public static final Reflection.Node description =
            new Reflection.Node(
                NodeKind.GRAMMAR_OPT_GROUP,
                false,
                false,
                GrammarOptGroup.class,
                "GrammarOptGroup",
                new String[] {
                    "f_ex