!!****f* ABINIT/vdw_dftd3_data
!!
!! NAME
!! vdw_dftd3_data
!!
!! FUNCTION
!! Retrieve data (tabulated coefficients) for DFT-D3 corrections (see
!! vdw_dftd3 routine for more information). It retrieves the tabulated
!! dispersion coefficients C6, the supporting points of the interpolation
!! and the cut-off radii (R0) for the dispersion correction
!! COPYRIGHT
!! Copyright (C) 2015-2024 ABINIT group (BVT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt.
!! INPUTS
!! OUTPUT
!!  vdw_dftd3_r0: R0ij cut-off radii used for the damping function to avoid
!!                the divergence
!!  vdw_dftd3_c6: C6ij tabulated dispersion coefficients used for the interpolation
!!                of the coordination number ordered by reference number* (
!!                Grimme, J. Chem. Phys. 132, 154104 (2010) [[cite:Grimme2010]]) and then by
!!                pair of chemical elements (H-H,He-H,He-He,etc.;
!!                only i-j coefficient are given as j-i is identical by symmetry)
!!  index_c6: table containing the index of non-zero values of C6 coefficients
!!  vdw_dftd3_cni: CN referenced coordination number used as supporting points of the
!!                 interpolation, ordered by reference number and by pair of chemical elements
!!  index_cni: table containing the index of non-zero values of CNi coefficients
!!  vdw_dftd3_cnj: referenced coordination number used as supporting points of the
!!                 interpolation, ordered by reference number and by pair of chemical element
!!  index_cnj: table containing the index of non-zero values of CNj coefficients
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

#include "abi_common.h"

module m_vdw_dftd3_data

 use defs_basis

 implicit none

 public :: vdw_dftd3_data

contains

subroutine vdw_dftd3_data(vdw_dftd3_r0,vdw_dftd3_c6,index_c6,vdw_dftd3_cni,index_cni,&
&vdw_dftd3_cnj,index_cnj)

#ifdef FC_INTEL
!DEC$ NOOPTIMIZE
#endif
 real(dp),intent(out) :: vdw_dftd3_r0(4465)
 real(dp),intent(out):: vdw_dftd3_c6(32385)
 integer,intent(out):: index_c6(254)
 real(dp),intent(out):: vdw_dftd3_cni(27884)
 integer,intent(out):: index_cni(27884)
 real(dp),intent(out):: vdw_dftd3_cnj(13171)
 integer,intent(out):: index_cnj(13171)

! *************************************************************************

 vdw_dftd3_r0(1:70)=& ! 1:4465 = 4465 values
&(/ 2.1823,  1.8547,  1.7347,  2.9086,  2.5732,  3.4956,  2.3550 &
& ,  2.5095,  2.9802,  3.0982,  2.5141,  2.3917,  2.9977,  2.9484 &
& ,  3.2160,  2.4492,  2.2527,  3.1933,  3.0214,  2.9531,  2.9103 &
& ,  2.3667,  2.1328,  2.8784,  2.7660,  2.7776,  2.7063,  2.6225 &
& ,  2.1768,  2.0625,  2.6395,  2.6648,  2.6482,  2.5697,  2.4846 &
& ,  2.4817,  2.0646,  1.9891,  2.5086,  2.6908,  2.6233,  2.4770 &
& ,  2.3885,  2.3511,  2.2996,  1.9892,  1.9251,  2.4190,  2.5473 &
& ,  2.4994,  2.4091,  2.3176,  2.2571,  2.1946,  2.1374,  2.9898 &
& ,  2.6397,  3.6031,  3.1219,  3.7620,  3.2485,  2.9357,  2.7093 &
& ,  2.5781,  2.4839,  3.7082,  2.5129,  2.7321,  3.1052,  3.2962 /)
 vdw_dftd3_r0(71:140)=&
& (/ 3.1331,  3.2000,  2.9586,  3.0822,  2.8582,  2.7120,  3.2570 &
& ,  3.4839,  2.8766,  2.7427,  3.2776,  3.2363,  3.5929,  3.2826 &
& ,  3.0911,  2.9369,  2.9030,  2.7789,  3.3921,  3.3970,  4.0106 &
& ,  2.8884,  2.6605,  3.7513,  3.1613,  3.3605,  3.3325,  3.0991 &
& ,  2.9297,  2.8674,  2.7571,  3.8129,  3.3266,  3.7105,  3.7917 &
& ,  2.8304,  2.5538,  3.3932,  3.1193,  3.1866,  3.1245,  3.0465 &
& ,  2.8727,  2.7664,  2.6926,  3.4608,  3.2984,  3.5142,  3.5418 &
& ,  3.5017,  2.6190,  2.4797,  3.1331,  3.0540,  3.0651,  2.9879 &
& ,  2.9054,  2.8805,  2.7330,  2.6331,  3.2096,  3.5668,  3.3684 &
& ,  3.3686,  3.3180,  3.3107,  2.4757,  2.4019,  2.9789,  3.1468 /)
 vdw_dftd3_r0(141:210)=&
& (/ 2.9768,  2.8848,  2.7952,  2.7457,  2.6881,  2.5728,  3.0574 &
& ,  3.3264,  3.3562,  3.2529,  3.1916,  3.1523,  3.1046,  2.3725 &
& ,  2.3289,  2.8760,  2.9804,  2.9093,  2.8040,  2.7071,  2.6386 &
& ,  2.5720,  2.5139,  2.9517,  3.1606,  3.2085,  3.1692,  3.0982 &
& ,  3.0352,  2.9730,  2.9148,  3.2147,  2.8315,  3.8724,  3.4621 &
& ,  3.8823,  3.3760,  3.0746,  2.8817,  2.7552,  2.6605,  3.9740 &
& ,  3.6192,  3.6569,  3.9586,  3.6188,  3.3917,  3.2479,  3.1434 &
& ,  4.2411,  2.7597,  3.0588,  3.3474,  3.6214,  3.4353,  3.4729 &
& ,  3.2487,  3.3200,  3.0914,  2.9403,  3.4972,  3.7993,  3.6773 &
& ,  3.8678,  3.5808,  3.8243,  3.5826,  3.4156,  3.8765,  4.1035 /)
 vdw_dftd3_r0(211:280)=&
& (/ 2.7361,  2.9765,  3.2475,  3.5004,  3.4185,  3.4378,  3.2084 &
& ,  3.2787,  3.0604,  2.9187,  3.4037,  3.6759,  3.6586,  3.8327 &
& ,  3.5372,  3.7665,  3.5310,  3.3700,  3.7788,  3.9804,  3.8903 &
& ,  2.6832,  2.9060,  3.2613,  3.4359,  3.3538,  3.3860,  3.1550 &
& ,  3.2300,  3.0133,  2.8736,  3.4024,  3.6142,  3.5979,  3.5295 &
& ,  3.4834,  3.7140,  3.4782,  3.3170,  3.7434,  3.9623,  3.8181 &
& ,  3.7642,  2.6379,  2.8494,  3.1840,  3.4225,  3.2771,  3.3401 &
& ,  3.1072,  3.1885,  2.9714,  2.8319,  3.3315,  3.5979,  3.5256 &
& ,  3.4980,  3.4376,  3.6714,  3.4346,  3.2723,  3.6859,  3.8985 &
& ,  3.7918,  3.7372,  3.7211,  2.9230,  2.6223,  3.4161,  2.8999 /)
 vdw_dftd3_r0(281:350)=&
& (/ 3.0557,  3.3308,  3.0555,  2.8508,  2.7385,  2.6640,  3.5263 &
& ,  3.0277,  3.2990,  3.7721,  3.5017,  3.2751,  3.1368,  3.0435 &
& ,  3.7873,  3.2858,  3.2140,  3.1727,  3.2178,  3.4414,  2.5490 &
& ,  2.7623,  3.0991,  3.3252,  3.1836,  3.2428,  3.0259,  3.1225 &
& ,  2.9032,  2.7621,  3.2490,  3.5110,  3.4429,  3.3845,  3.3574 &
& ,  3.6045,  3.3658,  3.2013,  3.6110,  3.8241,  3.7090,  3.6496 &
& ,  3.6333,  3.0896,  3.5462,  2.4926,  2.7136,  3.0693,  3.2699 &
& ,  3.1272,  3.1893,  2.9658,  3.0972,  2.8778,  2.7358,  3.2206 &
& ,  3.4566,  3.3896,  3.3257,  3.2946,  3.5693,  3.3312,  3.1670 &
& ,  3.5805,  3.7711,  3.6536,  3.5927,  3.5775,  3.0411,  3.4885 /)
 vdw_dftd3_r0(351:420)=&
& (/ 3.4421,  2.4667,  2.6709,  3.0575,  3.2357,  3.0908,  3.1537 &
& ,  2.9235,  3.0669,  2.8476,  2.7054,  3.2064,  3.4519,  3.3593 &
& ,  3.2921,  3.2577,  3.2161,  3.2982,  3.1339,  3.5606,  3.7582 &
& ,  3.6432,  3.5833,  3.5691,  3.0161,  3.4812,  3.4339,  3.4327 &
& ,  2.4515,  2.6338,  3.0511,  3.2229,  3.0630,  3.1265,  2.8909 &
& ,  3.0253,  2.8184,  2.6764,  3.1968,  3.4114,  3.3492,  3.2691 &
& ,  3.2320,  3.1786,  3.2680,  3.1036,  3.5453,  3.7259,  3.6090 &
& ,  3.5473,  3.5327,  3.0018,  3.4413,  3.3907,  3.3593,  3.3462 &
& ,  2.4413,  2.6006,  3.0540,  3.1987,  3.0490,  3.1058,  2.8643 &
& ,  2.9948,  2.7908,  2.6491,  3.1950,  3.3922,  3.3316,  3.2585 /)
 vdw_dftd3_r0(421:490)=&
& (/ 3.2136,  3.1516,  3.2364,  3.0752,  3.5368,  3.7117,  3.5941 &
& ,  3.5313,  3.5164,  2.9962,  3.4225,  3.3699,  3.3370,  3.3234 &
& ,  3.3008,  2.4318,  2.5729,  3.0416,  3.1639,  3.0196,  3.0843 &
& ,  2.8413,  2.7436,  2.7608,  2.6271,  3.1811,  3.3591,  3.3045 &
& ,  3.2349,  3.1942,  3.1291,  3.2111,  3.0534,  3.5189,  3.6809 &
& ,  3.5635,  3.5001,  3.4854,  2.9857,  3.3897,  3.3363,  3.3027 &
& ,  3.2890,  3.2655,  3.2309,  2.8502,  2.6934,  3.2467,  3.1921 &
& ,  3.5663,  3.2541,  3.0571,  2.9048,  2.8657,  2.7438,  3.3547 &
& ,  3.3510,  3.9837,  3.6871,  3.4862,  3.3389,  3.2413,  3.1708 &
& ,  3.6096,  3.6280,  3.6860,  3.5568,  3.4836,  3.2868,  3.3994 /)
 vdw_dftd3_r0(491:560)=&
& (/ 3.3476,  3.3170,  3.2950,  3.2874,  3.2606,  3.9579,  2.9226 &
& ,  2.6838,  3.7867,  3.1732,  3.3872,  3.3643,  3.1267,  2.9541 &
& ,  2.8505,  2.7781,  3.8475,  3.3336,  3.7359,  3.8266,  3.5733 &
& ,  3.3959,  3.2775,  3.1915,  3.9878,  3.8816,  3.5810,  3.5364 &
& ,  3.5060,  3.8097,  3.3925,  3.3348,  3.3019,  3.2796,  3.2662 &
& ,  3.2464,  3.7136,  3.8619,  2.9140,  2.6271,  3.4771,  3.1774 &
& ,  3.2560,  3.1970,  3.1207,  2.9406,  2.8322,  2.7571,  3.5455 &
& ,  3.3514,  3.5837,  3.6177,  3.5816,  3.3902,  3.2604,  3.1652 &
& ,  3.7037,  3.6283,  3.5858,  3.5330,  3.4884,  3.5789,  3.4094 &
& ,  3.3473,  3.3118,  3.2876,  3.2707,  3.2521,  3.5570,  3.6496 /)
 vdw_dftd3_r0(561:630)=&
& (/ 3.6625,  2.7300,  2.5870,  3.2471,  3.1487,  3.1667,  3.0914 &
& ,  3.0107,  2.9812,  2.8300,  2.7284,  3.3259,  3.3182,  3.4707 &
& ,  3.4748,  3.4279,  3.4182,  3.2547,  3.1353,  3.5116,  3.9432 &
& ,  3.8828,  3.8303,  3.7880,  3.3760,  3.7218,  3.3408,  3.3059 &
& ,  3.2698,  3.2446,  3.2229,  3.4422,  3.5023,  3.5009,  3.5268 &
& ,  2.6026,  2.5355,  3.1129,  3.2863,  3.1029,  3.0108,  2.9227 &
& ,  2.8694,  2.8109,  2.6929,  3.1958,  3.4670,  3.4018,  3.3805 &
& ,  3.3218,  3.2815,  3.2346,  3.0994,  3.3937,  3.7266,  3.6697 &
& ,  3.6164,  3.5730,  3.2522,  3.5051,  3.4686,  3.4355,  3.4084 &
& ,  3.3748,  3.3496,  3.3692,  3.4052,  3.3910,  3.3849,  3.3662 /)
 vdw_dftd3_r0(631:700)=&
& (/ 2.5087,  2.4814,  3.0239,  3.1312,  3.0535,  2.9457,  2.8496 &
& ,  2.7780,  2.7828,  2.6532,  3.1063,  3.3143,  3.3549,  3.3120 &
& ,  3.2421,  3.1787,  3.1176,  3.0613,  3.3082,  3.5755,  3.5222 &
& ,  3.4678,  3.4231,  3.1684,  3.3528,  3.3162,  3.2827,  3.2527 &
& ,  3.2308,  3.2029,  3.3173,  3.3343,  3.3092,  3.2795,  3.2452 &
& ,  3.2096,  3.2893,  2.8991,  4.0388,  3.6100,  3.9388,  3.4475 &
& ,  3.1590,  2.9812,  2.8586,  2.7683,  4.1428,  3.7911,  3.8225 &
& ,  4.0372,  3.7059,  3.4935,  3.3529,  3.2492,  4.4352,  4.0826 &
& ,  3.9733,  3.9254,  3.8646,  3.9315,  3.7837,  3.7465,  3.7211 &
& ,  3.7012,  3.6893,  3.6676,  3.7736,  4.0660,  3.7926,  3.6158 /)
 vdw_dftd3_r0(701:770)=&
& (/ 3.5017,  3.4166,  4.6176,  2.8786,  3.1658,  3.5823,  3.7689 &
& ,  3.5762,  3.5789,  3.3552,  3.4004,  3.1722,  3.0212,  3.7241 &
& ,  3.9604,  3.8500,  3.9844,  3.7035,  3.9161,  3.6751,  3.5075 &
& ,  4.1151,  4.2877,  4.1579,  4.1247,  4.0617,  3.4874,  3.9848 &
& ,  3.9280,  3.9079,  3.8751,  3.8604,  3.8277,  3.8002,  3.9981 &
& ,  3.7544,  4.0371,  3.8225,  3.6718,  4.3092,  4.4764,  2.8997 &
& ,  3.0953,  3.4524,  3.6107,  3.6062,  3.5783,  3.3463,  3.3855 &
& ,  3.1746,  3.0381,  3.6019,  3.7938,  3.8697,  3.9781,  3.6877 &
& ,  3.8736,  3.6451,  3.4890,  3.9858,  4.1179,  4.0430,  3.9563 &
& ,  3.9182,  3.4002,  3.8310,  3.7716,  3.7543,  3.7203,  3.7053 /)
 vdw_dftd3_r0(771:840)=&
& (/ 3.6742,  3.8318,  3.7631,  3.7392,  3.9892,  3.7832,  3.6406 &
& ,  4.1701,  4.3016,  4.2196,  2.8535,  3.0167,  3.3978,  3.5363 &
& ,  3.5393,  3.5301,  3.2960,  3.3352,  3.1287,  2.9967,  3.6659 &
& ,  3.7239,  3.8070,  3.7165,  3.6368,  3.8162,  3.5885,  3.4336 &
& ,  3.9829,  4.0529,  3.9584,  3.9025,  3.8607,  3.3673,  3.7658 &
& ,  3.7035,  3.6866,  3.6504,  3.6339,  3.6024,  3.7708,  3.7283 &
& ,  3.6896,  3.9315,  3.7250,  3.5819,  4.1457,  4.2280,  4.1130 &
& ,  4.0597,  3.0905,  2.7998,  3.6448,  3.0739,  3.2996,  3.5262 &
& ,  3.2559,  3.0518,  2.9394,  2.8658,  3.7514,  3.2295,  3.5643 &
& ,  3.7808,  3.6931,  3.4723,  3.3357,  3.2429,  4.0280,  3.5589 /)
 vdw_dftd3_r0(841:910)=&
& (/ 3.4636,  3.4994,  3.4309,  3.6177,  3.2946,  3.2376,  3.2050 &
& ,  3.1847,  3.1715,  3.1599,  3.5555,  3.8111,  3.7693,  3.5718 &
& ,  3.4498,  3.3662,  4.1608,  3.7417,  3.6536,  3.6154,  3.8596 &
& ,  3.0301,  2.7312,  3.5821,  3.0473,  3.2137,  3.4679,  3.1975 &
& ,  2.9969,  2.8847,  2.8110,  3.6931,  3.2076,  3.4943,  3.5956 &
& ,  3.6379,  3.4190,  3.2808,  3.1860,  3.9850,  3.5105,  3.4330 &
& ,  3.3797,  3.4155,  3.6033,  3.2737,  3.2145,  3.1807,  3.1596 &
& ,  3.1461,  3.1337,  3.4812,  3.6251,  3.7152,  3.5201,  3.3966 &
& ,  3.3107,  4.1128,  3.6899,  3.6082,  3.5604,  3.7834,  3.7543 &
& ,  2.9189,  2.6777,  3.4925,  2.9648,  3.1216,  3.2940,  3.0975 /)
 vdw_dftd3_r0(911:980)=&
& (/ 2.9757,  2.8493,  2.7638,  3.6085,  3.1214,  3.4006,  3.4793 &
& ,  3.5147,  3.3806,  3.2356,  3.1335,  3.9144,  3.4183,  3.3369 &
& ,  3.2803,  3.2679,  3.4871,  3.1714,  3.1521,  3.1101,  3.0843 &
& ,  3.0670,  3.0539,  3.3890,  3.5086,  3.5895,  3.4783,  3.3484 &
& ,  3.2559,  4.0422,  3.5967,  3.5113,  3.4576,  3.6594,  3.6313 &
& ,  3.5690,  2.8578,  2.6334,  3.4673,  2.9245,  3.0732,  3.2435 &
& ,  3.0338,  2.9462,  2.8143,  2.7240,  3.5832,  3.0789,  3.3617 &
& ,  3.4246,  3.4505,  3.3443,  3.1964,  3.0913,  3.8921,  3.3713 &
& ,  3.2873,  3.2281,  3.2165,  3.4386,  3.1164,  3.1220,  3.0761 &
& ,  3.0480,  3.0295,  3.0155,  3.3495,  3.4543,  3.5260,  3.4413 /)
 vdw_dftd3_r0(981:1050)=&
& (/ 3.3085,  3.2134,  4.0170,  3.5464,  3.4587,  3.4006,  3.6027 &
& ,  3.5730,  3.4945,  3.4623,  2.8240,  2.5960,  3.4635,  2.9032 &
& ,  3.0431,  3.2115,  2.9892,  2.9148,  2.7801,  2.6873,  3.5776 &
& ,  3.0568,  3.3433,  3.3949,  3.4132,  3.3116,  3.1616,  3.0548 &
& ,  3.8859,  3.3719,  3.2917,  3.2345,  3.2274,  3.4171,  3.1293 &
& ,  3.0567,  3.0565,  3.0274,  3.0087,  2.9939,  3.3293,  3.4249 &
& ,  3.4902,  3.4091,  3.2744,  3.1776,  4.0078,  3.5374,  3.4537 &
& ,  3.3956,  3.5747,  3.5430,  3.4522,  3.4160,  3.3975,  2.8004 &
& ,  2.5621,  3.4617,  2.9154,  3.0203,  3.1875,  2.9548,  2.8038 &
& ,  2.7472,  2.6530,  3.5736,  3.0584,  3.3304,  3.3748,  3.3871 /)
 vdw_dftd3_r0(1051:1120)=&
& (/ 3.2028,  3.1296,  3.0214,  3.8796,  3.3337,  3.2492,  3.1883 &
& ,  3.1802,  3.4050,  3.0756,  3.0478,  3.0322,  3.0323,  3.0163 &
& ,  3.0019,  3.3145,  3.4050,  3.4656,  3.3021,  3.2433,  3.1453 &
& ,  3.9991,  3.5017,  3.4141,  3.3520,  3.5583,  3.5251,  3.4243 &
& ,  3.3851,  3.3662,  3.3525,  2.7846,  2.5324,  3.4652,  2.8759 &
& ,  3.0051,  3.1692,  2.9273,  2.7615,  2.7164,  2.6212,  3.5744 &
& ,  3.0275,  3.3249,  3.3627,  3.3686,  3.1669,  3.0584,  2.9915 &
& ,  3.8773,  3.3099,  3.2231,  3.1600,  3.1520,  3.4023,  3.0426 &
& ,  3.0099,  2.9920,  2.9809,  2.9800,  2.9646,  3.3068,  3.3930 &
& ,  3.4486,  3.2682,  3.1729,  3.1168,  3.9952,  3.4796,  3.3901 /)
 vdw_dftd3_r0(1121:1190)=&
& (/ 3.3255,  3.5530,  3.5183,  3.4097,  3.3683,  3.3492,  3.3360 &
& ,  3.3308,  2.5424,  2.6601,  3.2555,  3.2807,  3.1384,  3.1737 &
& ,  2.9397,  2.8429,  2.8492,  2.7225,  3.3875,  3.4910,  3.4520 &
& ,  3.3608,  3.3036,  3.2345,  3.2999,  3.1487,  3.7409,  3.8392 &
& ,  3.7148,  3.6439,  3.6182,  3.1753,  3.5210,  3.4639,  3.4265 &
& ,  3.4075,  3.3828,  3.3474,  3.4071,  3.3754,  3.3646,  3.3308 &
& ,  3.4393,  3.2993,  3.8768,  3.9891,  3.8310,  3.7483,  3.3417 &
& ,  3.3019,  3.2250,  3.1832,  3.1578,  3.1564,  3.1224,  3.4620 &
& ,  2.9743,  2.8058,  3.4830,  3.3474,  3.6863,  3.3617,  3.1608 &
& ,  3.0069,  2.9640,  2.8427,  3.5885,  3.5219,  4.1314,  3.8120 /)
 vdw_dftd3_r0(1191:1260)=&
& (/ 3.6015,  3.4502,  3.3498,  3.2777,  3.8635,  3.8232,  3.8486 &
& ,  3.7215,  3.6487,  3.4724,  3.5627,  3.5087,  3.4757,  3.4517 &
& ,  3.4423,  3.4139,  4.1028,  3.8388,  3.6745,  3.5562,  3.4806 &
& ,  3.4272,  4.0182,  3.9991,  4.0007,  3.9282,  3.7238,  3.6498 &
& ,  3.5605,  3.5211,  3.5009,  3.4859,  3.4785,  3.5621,  4.2623 &
& ,  3.0775,  2.8275,  4.0181,  3.3385,  3.5379,  3.5036,  3.2589 &
& ,  3.0804,  3.0094,  2.9003,  4.0869,  3.5088,  3.9105,  3.9833 &
& ,  3.7176,  3.5323,  3.4102,  3.3227,  4.2702,  4.0888,  3.7560 &
& ,  3.7687,  3.6681,  3.6405,  3.5569,  3.4990,  3.4659,  3.4433 &
& ,  3.4330,  3.4092,  3.8867,  4.0190,  3.7961,  3.6412,  3.5405 /)
 vdw_dftd3_r0(1261:1330)=&
& (/ 3.4681,  4.3538,  4.2136,  3.9381,  3.8912,  3.9681,  3.7909 &
& ,  3.6774,  3.6262,  3.5999,  3.5823,  3.5727,  3.5419,  4.0245 &
& ,  4.1874,  3.0893,  2.7917,  3.7262,  3.3518,  3.4241,  3.5433 &
& ,  3.2773,  3.0890,  2.9775,  2.9010,  3.8048,  3.5362,  3.7746 &
& ,  3.7911,  3.7511,  3.5495,  3.4149,  3.3177,  4.0129,  3.8370 &
& ,  3.7739,  3.7125,  3.7152,  3.7701,  3.5813,  3.5187,  3.4835 &
& ,  3.4595,  3.4439,  3.4242,  3.7476,  3.8239,  3.8346,  3.6627 &
& ,  3.5479,  3.4639,  4.1026,  3.9733,  3.9292,  3.8667,  3.9513 &
& ,  3.8959,  3.7698,  3.7089,  3.6765,  3.6548,  3.6409,  3.5398 &
& ,  3.8759,  3.9804,  4.0150,  2.9091,  2.7638,  3.5066,  3.3377 /)
 vdw_dftd3_r0(1331:1400)=&
& (/ 3.3481,  3.2633,  3.1810,  3.1428,  2.9872,  2.8837,  3.5929 &
& ,  3.5183,  3.6729,  3.6596,  3.6082,  3.5927,  3.4224,  3.2997 &
& ,  3.8190,  4.1865,  4.1114,  4.0540,  3.6325,  3.5697,  3.5561 &
& ,  3.5259,  3.4901,  3.4552,  3.4315,  3.4091,  3.6438,  3.6879 &
& ,  3.6832,  3.7043,  3.5557,  3.4466,  3.9203,  4.2919,  4.2196 &
& ,  4.1542,  3.7573,  3.7039,  3.6546,  3.6151,  3.5293,  3.4849 &
& ,  3.4552,  3.5192,  3.7673,  3.8359,  3.8525,  3.8901,  2.7806 &
& ,  2.7209,  3.3812,  3.4958,  3.2913,  3.1888,  3.0990,  3.0394 &
& ,  2.9789,  2.8582,  3.4716,  3.6883,  3.6105,  3.5704,  3.5059 &
& ,  3.4619,  3.4138,  3.2742,  3.7080,  3.9773,  3.9010,  3.8409 /)
 vdw_dftd3_r0(1401:1470)=&
& (/ 3.7944,  3.4465,  3.7235,  3.6808,  3.6453,  3.6168,  3.5844 &
& ,  3.5576,  3.5772,  3.5959,  3.5768,  3.5678,  3.5486,  3.4228 &
& ,  3.8107,  4.0866,  4.0169,  3.9476,  3.6358,  3.5800,  3.5260 &
& ,  3.4838,  3.4501,  3.4204,  3.3553,  3.6487,  3.6973,  3.7398 &
& ,  3.7405,  3.7459,  3.7380,  2.6848,  2.6740,  3.2925,  3.3386 &
& ,  3.2473,  3.1284,  3.0301,  2.9531,  2.9602,  2.8272,  3.3830 &
& ,  3.5358,  3.5672,  3.5049,  3.4284,  3.3621,  3.3001,  3.2451 &
& ,  3.6209,  3.8299,  3.7543,  3.6920,  3.6436,  3.3598,  3.5701 &
& ,  3.5266,  3.4904,  3.4590,  3.4364,  3.4077,  3.5287,  3.5280 &
& ,  3.4969,  3.4650,  3.4304,  3.3963,  3.7229,  3.9402,  3.8753 /)
 vdw_dftd3_r0(1471:1540)=&
& (/ 3.8035,  3.5499,  3.4913,  3.4319,  3.3873,  3.3520,  3.3209 &
& ,  3.2948,  3.5052,  3.6465,  3.6696,  3.6577,  3.6388,  3.6142 &
& ,  3.5889,  3.3968,  3.0122,  4.2241,  3.7887,  4.0049,  3.5384 &
& ,  3.2698,  3.1083,  2.9917,  2.9057,  4.3340,  3.9900,  4.6588 &
& ,  4.1278,  3.8125,  3.6189,  3.4851,  3.3859,  4.6531,  4.3134 &
& ,  4.2258,  4.1309,  4.0692,  4.0944,  3.9850,  3.9416,  3.9112 &
& ,  3.8873,  3.8736,  3.8473,  4.6027,  4.1538,  3.8994,  3.7419 &
& ,  3.6356,  3.5548,  4.8353,  4.5413,  4.3891,  4.3416,  4.3243 &
& ,  4.2753,  4.2053,  4.1790,  4.1685,  4.1585,  4.1536,  4.0579 &
& ,  4.1980,  4.4564,  4.2192,  4.0528,  3.9489,  3.8642,  5.0567 /)
 vdw_dftd3_r0(1541:1610)=&
& (/ 3.0630,  3.3271,  4.0432,  4.0046,  4.1555,  3.7426,  3.5130 &
& ,  3.5174,  3.2884,  3.1378,  4.1894,  4.2321,  4.1725,  4.1833 &
& ,  3.8929,  4.0544,  3.8118,  3.6414,  4.6373,  4.6268,  4.4750 &
& ,  4.4134,  4.3458,  3.8582,  4.2583,  4.1898,  4.1562,  4.1191 &
& ,  4.1069,  4.0639,  4.1257,  4.1974,  3.9532,  4.1794,  3.9660 &
& ,  3.8130,  4.8160,  4.8272,  4.6294,  4.5840,  4.0770,  4.0088 &
& ,  3.9103,  3.8536,  3.8324,  3.7995,  3.7826,  4.2294,  4.3380 &
& ,  4.4352,  4.1933,  4.4580,  4.2554,  4.1072,  5.0454,  5.1814 &
& ,  3.0632,  3.2662,  3.6432,  3.8088,  3.7910,  3.7381,  3.5093 &
& ,  3.5155,  3.3047,  3.1681,  3.7871,  3.9924,  4.0637,  4.1382 /)
 vdw_dftd3_r0(1611:1680)=&
& (/ 3.8591,  4.0164,  3.7878,  3.6316,  4.1741,  4.3166,  4.2395 &
& ,  4.1831,  4.1107,  3.5857,  4.0270,  3.9676,  3.9463,  3.9150 &
& ,  3.9021,  3.8708,  4.0240,  4.1551,  3.9108,  4.1337,  3.9289 &
& ,  3.7873,  4.3666,  4.5080,  4.4232,  4.3155,  3.8461,  3.8007 &
& ,  3.6991,  3.6447,  3.6308,  3.5959,  3.5749,  4.0359,  4.3124 &
& ,  4.3539,  4.1122,  4.3772,  4.1785,  4.0386,  4.7004,  4.8604 &
& ,  4.6261,  2.9455,  3.2470,  3.6108,  3.8522,  3.6625,  3.6598 &
& ,  3.4411,  3.4660,  3.2415,  3.0944,  3.7514,  4.0397,  3.9231 &
& ,  4.0561,  3.7860,  3.9845,  3.7454,  3.5802,  4.1366,  4.3581 &
& ,  4.2351,  4.2011,  4.1402,  3.5381,  4.0653,  4.0093,  3.9883 /)
 vdw_dftd3_r0(1681:1750)=&
& (/ 3.9570,  3.9429,  3.9112,  3.8728,  4.0682,  3.8351,  4.1054 &
& ,  3.8928,  3.7445,  4.3415,  4.5497,  4.3833,  4.3122,  3.8051 &
& ,  3.7583,  3.6622,  3.6108,  3.5971,  3.5628,  3.5408,  4.0780 &
& ,  4.0727,  4.2836,  4.0553,  4.3647,  4.1622,  4.0178,  4.5802 &
& ,  4.9125,  4.5861,  4.6201,  2.9244,  3.2241,  3.5848,  3.8293 &
& ,  3.6395,  3.6400,  3.4204,  3.4499,  3.2253,  3.0779,  3.7257 &
& ,  4.0170,  3.9003,  4.0372,  3.7653,  3.9672,  3.7283,  3.5630 &
& ,  4.1092,  4.3347,  4.2117,  4.1793,  4.1179,  3.5139,  4.0426 &
& ,  3.9867,  3.9661,  3.9345,  3.9200,  3.8883,  3.8498,  4.0496 &
& ,  3.8145,  4.0881,  3.8756,  3.7271,  4.3128,  4.5242,  4.3578 /)
 vdw_dftd3_r0(1751:1820)=&
& (/ 4.2870,  3.7796,  3.7318,  3.6364,  3.5854,  3.5726,  3.5378 &
& ,  3.5155,  4.0527,  4.0478,  4.2630,  4.0322,  4.3449,  4.1421 &
& ,  3.9975,  4.5499,  4.8825,  4.5601,  4.5950,  4.5702,  2.9046 &
& ,  3.2044,  3.5621,  3.8078,  3.6185,  3.6220,  3.4019,  3.4359 &
& ,  3.2110,  3.0635,  3.7037,  3.9958,  3.8792,  4.0194,  3.7460 &
& ,  3.9517,  3.7128,  3.5474,  4.0872,  4.3138,  4.1906,  4.1593 &
& ,  4.0973,  3.4919,  4.0216,  3.9657,  3.9454,  3.9134,  3.8986 &
& ,  3.8669,  3.8289,  4.0323,  3.7954,  4.0725,  3.8598,  3.7113 &
& ,  4.2896,  4.5021,  4.3325,  4.2645,  3.7571,  3.7083,  3.6136 &
& ,  3.5628,  3.5507,  3.5155,  3.4929,  4.0297,  4.0234,  4.2442 /)
 vdw_dftd3_r0(1821:1890)=&
& (/ 4.0112,  4.3274,  4.1240,  3.9793,  4.5257,  4.8568,  4.5353 &
& ,  4.5733,  4.5485,  4.5271,  2.8878,  3.1890,  3.5412,  3.7908 &
& ,  3.5974,  3.6078,  3.3871,  3.4243,  3.1992,  3.0513,  3.6831 &
& ,  3.9784,  3.8579,  4.0049,  3.7304,  3.9392,  3.7002,  3.5347 &
& ,  4.0657,  4.2955,  4.1705,  4.1424,  4.0800,  3.4717,  4.0043 &
& ,  3.9485,  3.9286,  3.8965,  3.8815,  3.8500,  3.8073,  4.0180 &
& ,  3.7796,  4.0598,  3.8470,  3.6983,  4.2678,  4.4830,  4.3132 &
& ,  4.2444,  3.7370,  3.6876,  3.5935,  3.5428,  3.5314,  3.4958 &
& ,  3.4730,  4.0117,  4.0043,  4.2287,  3.9939,  4.3134,  4.1096 &
& ,  3.9646,  4.5032,  4.8356,  4.5156,  4.5544,  4.5297,  4.5083 /)
 vdw_dftd3_r0(1891:1960)=&
& (/ 4.4896,  2.8709,  3.1737,  3.5199,  3.7734,  3.5802,  3.5934 &
& ,  3.3724,  3.4128,  3.1877,  3.0396,  3.6624,  3.9608,  3.8397 &
& ,  3.9893,  3.7145,  3.9266,  3.6877,  3.5222,  4.0448,  4.2771 &
& ,  4.1523,  4.1247,  4.0626,  3.4530,  3.9866,  3.9310,  3.9115 &
& ,  3.8792,  3.8641,  3.8326,  3.7892,  4.0025,  3.7636,  4.0471 &
& ,  3.8343,  3.6854,  4.2464,  4.4635,  4.2939,  4.2252,  3.7169 &
& ,  3.6675,  3.5739,  3.5235,  3.5126,  3.4768,  3.4537,  3.9932 &
& ,  3.9854,  4.2123,  3.9765,  4.2992,  4.0951,  3.9500,  4.4811 &
& ,  4.8135,  4.4959,  4.5351,  4.5105,  4.4891,  4.4705,  4.4515 &
& ,  2.8568,  3.1608,  3.5050,  3.7598,  3.5665,  3.5803,  3.3601 /)
 vdw_dftd3_r0(1961:2030)=&
& (/ 3.4031,  3.1779,  3.0296,  3.6479,  3.9471,  3.8262,  3.9773 &
& ,  3.7015,  3.9162,  3.6771,  3.5115,  4.0306,  4.2634,  4.1385 &
& ,  4.1116,  4.0489,  3.4366,  3.9732,  3.9176,  3.8983,  3.8659 &
& ,  3.8507,  3.8191,  3.7757,  3.9907,  3.7506,  4.0365,  3.8235 &
& ,  3.6745,  4.2314,  4.4490,  4.2792,  4.2105,  3.7003,  3.6510 &
& ,  3.5578,  3.5075,  3.4971,  3.4609,  3.4377,  3.9788,  3.9712 &
& ,  4.1997,  3.9624,  4.2877,  4.0831,  3.9378,  4.4655,  4.7974 &
& ,  4.4813,  4.5209,  4.4964,  4.4750,  4.4565,  4.4375,  4.4234 &
& ,  2.6798,  3.0151,  3.2586,  3.5292,  3.5391,  3.4902,  3.2887 &
& ,  3.3322,  3.1228,  2.9888,  3.4012,  3.7145,  3.7830,  3.6665 /)
 vdw_dftd3_r0(2031:2100)=&
& (/ 3.5898,  3.8077,  3.5810,  3.4265,  3.7726,  4.0307,  3.9763 &
& ,  3.8890,  3.8489,  3.2706,  3.7595,  3.6984,  3.6772,  3.6428 &
& ,  3.6243,  3.5951,  3.7497,  3.6775,  3.6364,  3.9203,  3.7157 &
& ,  3.5746,  3.9494,  4.2076,  4.1563,  4.0508,  3.5329,  3.4780 &
& ,  3.3731,  3.3126,  3.2846,  3.2426,  3.2135,  3.7491,  3.9006 &
& ,  3.8332,  3.8029,  4.1436,  3.9407,  3.7998,  4.1663,  4.5309 &
& ,  4.3481,  4.2911,  4.2671,  4.2415,  4.2230,  4.2047,  4.1908 &
& ,  4.1243,  2.5189,  2.9703,  3.3063,  3.6235,  3.4517,  3.3989 &
& ,  3.2107,  3.2434,  3.0094,  2.8580,  3.4253,  3.8157,  3.7258 &
& ,  3.6132,  3.5297,  3.7566,  3.5095,  3.3368,  3.7890,  4.1298 /)
 vdw_dftd3_r0(2101:2170)=&
& (/ 4.0190,  3.9573,  3.9237,  3.2677,  3.8480,  3.8157,  3.7656 &
& ,  3.7317,  3.7126,  3.6814,  3.6793,  3.6218,  3.5788,  3.8763 &
& ,  3.6572,  3.5022,  3.9737,  4.3255,  4.1828,  4.1158,  3.5078 &
& ,  3.4595,  3.3600,  3.3088,  3.2575,  3.2164,  3.1856,  3.8522 &
& ,  3.8665,  3.8075,  3.7772,  4.1391,  3.9296,  3.7772,  4.2134 &
& ,  4.7308,  4.3787,  4.3894,  4.3649,  4.3441,  4.3257,  4.3073 &
& ,  4.2941,  4.1252,  4.2427,  3.0481,  2.9584,  3.6919,  3.5990 &
& ,  3.8881,  3.4209,  3.1606,  3.1938,  2.9975,  2.8646,  3.8138 &
& ,  3.7935,  3.7081,  3.9155,  3.5910,  3.4808,  3.4886,  3.3397 &
& ,  4.1336,  4.1122,  3.9888,  3.9543,  3.8917,  3.5894,  3.8131 /)
 vdw_dftd3_r0(2171:2240)=&
& (/ 3.7635,  3.7419,  3.7071,  3.6880,  3.6574,  3.6546,  3.9375 &
& ,  3.6579,  3.5870,  3.6361,  3.5039,  4.3149,  4.2978,  4.1321 &
& ,  4.1298,  3.8164,  3.7680,  3.7154,  3.6858,  3.6709,  3.6666 &
& ,  3.6517,  3.8174,  3.8608,  4.1805,  3.9102,  3.8394,  3.8968 &
& ,  3.7673,  4.5274,  4.6682,  4.3344,  4.3639,  4.3384,  4.3162 &
& ,  4.2972,  4.2779,  4.2636,  4.0253,  4.1168,  4.1541,  2.8136 &
& ,  3.0951,  3.4635,  3.6875,  3.4987,  3.5183,  3.2937,  3.3580 &
& ,  3.1325,  2.9832,  3.6078,  3.8757,  3.7616,  3.9222,  3.6370 &
& ,  3.8647,  3.6256,  3.4595,  3.9874,  4.1938,  4.0679,  4.0430 &
& ,  3.9781,  3.3886,  3.9008,  3.8463,  3.8288,  3.7950,  3.7790 /)
 vdw_dftd3_r0(2241:2310)=&
& (/ 3.7472,  3.7117,  3.9371,  3.6873,  3.9846,  3.7709,  3.6210 &
& ,  4.1812,  4.3750,  4.2044,  4.1340,  3.6459,  3.5929,  3.5036 &
& ,  3.4577,  3.4528,  3.4146,  3.3904,  3.9014,  3.9031,  4.1443 &
& ,  3.8961,  4.2295,  4.0227,  3.8763,  4.4086,  4.7097,  4.4064 &
& ,  4.4488,  4.4243,  4.4029,  4.3842,  4.3655,  4.3514,  4.1162 &
& ,  4.2205,  4.1953,  4.2794,  2.8032,  3.0805,  3.4519,  3.6700 &
& ,  3.4827,  3.5050,  3.2799,  3.3482,  3.1233,  2.9747,  3.5971 &
& ,  3.8586,  3.7461,  3.9100,  3.6228,  3.8535,  3.6147,  3.4490 &
& ,  3.9764,  4.1773,  4.0511,  4.0270,  3.9614,  3.3754,  3.8836 &
& ,  3.8291,  3.8121,  3.7780,  3.7619,  3.7300,  3.6965,  3.9253 /)
 vdw_dftd3_r0(2311:2380)=&
& (/ 3.6734,  3.9733,  3.7597,  3.6099,  4.1683,  4.3572,  4.1862 &
& ,  4.1153,  3.6312,  3.5772,  3.4881,  3.4429,  3.4395,  3.4009 &
& ,  3.3766,  3.8827,  3.8868,  4.1316,  3.8807,  4.2164,  4.0092 &
& ,  3.8627,  4.3936,  4.6871,  4.3882,  4.4316,  4.4073,  4.3858 &
& ,  4.3672,  4.3485,  4.3344,  4.0984,  4.2036,  4.1791,  4.2622 &
& ,  4.2450,  2.7967,  3.0689,  3.4445,  3.6581,  3.4717,  3.4951 &
& ,  3.2694,  3.3397,  3.1147,  2.9661,  3.5898,  3.8468,  3.7358 &
& ,  3.9014,  3.6129,  3.8443,  3.6054,  3.4396,  3.9683,  4.1656 &
& ,  4.0394,  4.0158,  3.9498,  3.3677,  3.8718,  3.8164,  3.8005 &
& ,  3.7662,  3.7500,  3.7181,  3.6863,  3.9170,  3.6637,  3.9641 /)
 vdw_dftd3_r0(2381:2450)=&
& (/ 3.7503,  3.6004,  4.1590,  4.3448,  4.1739,  4.1029,  3.6224 &
& ,  3.5677,  3.4785,  3.4314,  3.4313,  3.3923,  3.3680,  3.8698 &
& ,  3.8758,  4.1229,  3.8704,  4.2063,  3.9987,  3.8519,  4.3832 &
& ,  4.6728,  4.3759,  4.4195,  4.3952,  4.3737,  4.3551,  4.3364 &
& ,  4.3223,  4.0861,  4.1911,  4.1676,  4.2501,  4.2329,  4.2208 &
& ,  2.7897,  3.0636,  3.4344,  3.6480,  3.4626,  3.4892,  3.2626 &
& ,  3.3344,  3.1088,  2.9597,  3.5804,  3.8359,  3.7251,  3.8940 &
& ,  3.6047,  3.8375,  3.5990,  3.4329,  3.9597,  4.1542,  4.0278 &
& ,  4.0048,  3.9390,  3.3571,  3.8608,  3.8056,  3.7899,  3.7560 &
& ,  3.7400,  3.7081,  3.6758,  3.9095,  3.6552,  3.9572,  3.7436 /)
 vdw_dftd3_r0(2451:2520)=&
& (/ 3.5933,  4.1508,  4.3337,  4.1624,  4.0916,  3.6126,  3.5582 &
& ,  3.4684,  3.4212,  3.4207,  3.3829,  3.3586,  3.8604,  3.8658 &
& ,  4.1156,  3.8620,  4.1994,  3.9917,  3.8446,  4.3750,  4.6617 &
& ,  4.3644,  4.4083,  4.3840,  4.3625,  4.3439,  4.3253,  4.3112 &
& ,  4.0745,  4.1807,  4.1578,  4.2390,  4.2218,  4.2097,  4.1986 &
& ,  2.8395,  3.0081,  3.3171,  3.4878,  3.5360,  3.5145,  3.2809 &
& ,  3.3307,  3.1260,  2.9940,  3.4741,  3.6675,  3.7832,  3.6787 &
& ,  3.6156,  3.8041,  3.5813,  3.4301,  3.8480,  3.9849,  3.9314 &
& ,  3.8405,  3.8029,  3.2962,  3.7104,  3.6515,  3.6378,  3.6020 &
& ,  3.5849,  3.5550,  3.7494,  3.6893,  3.6666,  3.9170,  3.7150 /)
 vdw_dftd3_r0(2521:2590)=&
& (/ 3.5760,  4.0268,  4.1596,  4.1107,  3.9995,  3.5574,  3.5103 &
& ,  3.4163,  3.3655,  3.3677,  3.3243,  3.2975,  3.7071,  3.9047 &
& ,  3.8514,  3.8422,  3.8022,  3.9323,  3.7932,  4.2343,  4.4583 &
& ,  4.3115,  4.2457,  4.2213,  4.1945,  4.1756,  4.1569,  4.1424 &
& ,  4.0620,  4.0494,  3.9953,  4.0694,  4.0516,  4.0396,  4.0280 &
& ,  4.0130,  2.9007,  2.9674,  3.8174,  3.5856,  3.6486,  3.5339 &
& ,  3.2832,  3.3154,  3.1144,  2.9866,  3.9618,  3.8430,  3.9980 &
& ,  3.8134,  3.6652,  3.7985,  3.5756,  3.4207,  4.4061,  4.2817 &
& ,  4.1477,  4.0616,  3.9979,  3.6492,  3.8833,  3.8027,  3.7660 &
& ,  3.7183,  3.6954,  3.6525,  3.9669,  3.8371,  3.7325,  3.9160 /)
 vdw_dftd3_r0(2591:2660)=&
& (/ 3.7156,  3.5714,  4.6036,  4.4620,  4.3092,  4.2122,  3.8478 &
& ,  3.7572,  3.6597,  3.5969,  3.5575,  3.5386,  3.5153,  3.7818 &
& ,  4.1335,  4.0153,  3.9177,  3.8603,  3.9365,  3.7906,  4.7936 &
& ,  4.7410,  4.5461,  4.5662,  4.5340,  4.5059,  4.4832,  4.4604 &
& ,  4.4429,  4.2346,  4.4204,  4.3119,  4.3450,  4.3193,  4.3035 &
& ,  4.2933,  4.1582,  4.2450,  2.8559,  2.9050,  3.8325,  3.5442 &
& ,  3.5077,  3.4905,  3.2396,  3.2720,  3.0726,  2.9467,  3.9644 &
& ,  3.8050,  3.8981,  3.7762,  3.6216,  3.7531,  3.5297,  3.3742 &
& ,  4.3814,  4.2818,  4.1026,  4.0294,  3.9640,  3.6208,  3.8464 &
& ,  3.7648,  3.7281,  3.6790,  3.6542,  3.6117,  3.8650,  3.8010 /)
 vdw_dftd3_r0(2661:2730)=&
& (/ 3.6894,  3.8713,  3.6699,  3.5244,  4.5151,  4.4517,  4.2538 &
& ,  4.1483,  3.8641,  3.7244,  3.6243,  3.5589,  3.5172,  3.4973 &
& ,  3.4715,  3.7340,  4.0316,  3.9958,  3.8687,  3.8115,  3.8862 &
& ,  3.7379,  4.7091,  4.7156,  4.5199,  4.5542,  4.5230,  4.4959 &
& ,  4.4750,  4.4529,  4.4361,  4.1774,  4.3774,  4.2963,  4.3406 &
& ,  4.3159,  4.3006,  4.2910,  4.1008,  4.1568,  4.0980,  2.8110 &
& ,  2.8520,  3.7480,  3.5105,  3.4346,  3.3461,  3.1971,  3.2326 &
& ,  3.0329,  2.9070,  3.8823,  3.7928,  3.8264,  3.7006,  3.5797 &
& ,  3.7141,  3.4894,  3.3326,  4.3048,  4.2217,  4.0786,  3.9900 &
& ,  3.9357,  3.6331,  3.8333,  3.7317,  3.6957,  3.6460,  3.6197 /)
 vdw_dftd3_r0(2731:2800)=&
& (/ 3.5779,  3.7909,  3.7257,  3.6476,  3.5729,  3.6304,  3.4834 &
& ,  4.4368,  4.3921,  4.2207,  4.1133,  3.8067,  3.7421,  3.6140 &
& ,  3.5491,  3.5077,  3.4887,  3.4623,  3.6956,  3.9568,  3.8976 &
& ,  3.8240,  3.7684,  3.8451,  3.6949,  4.6318,  4.6559,  4.4533 &
& ,  4.4956,  4.4641,  4.4366,  4.4155,  4.3936,  4.3764,  4.1302 &
& ,  4.3398,  4.2283,  4.2796,  4.2547,  4.2391,  4.2296,  4.0699 &
& ,  4.1083,  4.0319,  3.9855,  2.7676,  2.8078,  3.6725,  3.4804 &
& ,  3.3775,  3.2411,  3.1581,  3.1983,  2.9973,  2.8705,  3.8070 &
& ,  3.7392,  3.7668,  3.6263,  3.5402,  3.6807,  3.4545,  3.2962 &
& ,  4.2283,  4.1698,  4.0240,  3.9341,  3.8711,  3.5489,  3.7798 /)
 vdw_dftd3_r0(2801:2870)=&
& (/ 3.7000,  3.6654,  3.6154,  3.5882,  3.5472,  3.7289,  3.6510 &
& ,  3.6078,  3.5355,  3.5963,  3.4480,  4.3587,  4.3390,  4.1635 &
& ,  4.0536,  3.7193,  3.6529,  3.5512,  3.4837,  3.4400,  3.4191 &
& ,  3.3891,  3.6622,  3.8934,  3.8235,  3.7823,  3.7292,  3.8106 &
& ,  3.6589,  4.5535,  4.6013,  4.3961,  4.4423,  4.4109,  4.3835 &
& ,  4.3625,  4.3407,  4.3237,  4.0863,  4.2835,  4.1675,  4.2272 &
& ,  4.2025,  4.1869,  4.1774,  4.0126,  4.0460,  3.9815,  3.9340 &
& ,  3.8955,  2.6912,  2.7604,  3.6037,  3.4194,  3.3094,  3.1710 &
& ,  3.0862,  3.1789,  2.9738,  2.8427,  3.7378,  3.6742,  3.6928 &
& ,  3.5512,  3.4614,  3.4087,  3.4201,  3.2607,  4.1527,  4.0977 /)
 vdw_dftd3_r0(2871:2940)=&
& (/ 3.9523,  3.8628,  3.8002,  3.4759,  3.7102,  3.6466,  3.6106 &
& ,  3.5580,  3.5282,  3.4878,  3.6547,  3.5763,  3.5289,  3.5086 &
& ,  3.5593,  3.4099,  4.2788,  4.2624,  4.0873,  3.9770,  3.6407 &
& ,  3.5743,  3.5178,  3.4753,  3.3931,  3.3694,  3.3339,  3.6002 &
& ,  3.8164,  3.7478,  3.7028,  3.6952,  3.7669,  3.6137,  4.4698 &
& ,  4.5488,  4.3168,  4.3646,  4.3338,  4.3067,  4.2860,  4.2645 &
& ,  4.2478,  4.0067,  4.2349,  4.0958,  4.1543,  4.1302,  4.1141 &
& ,  4.1048,  3.9410,  3.9595,  3.8941,  3.8465,  3.8089,  3.7490 &
& ,  2.7895,  2.5849,  3.6484,  3.0162,  3.1267,  3.2125,  3.0043 &
& ,  2.9572,  2.8197,  2.7261,  3.7701,  3.2446,  3.5239,  3.4696 /)
 vdw_dftd3_r0(2941:3010)=&
& (/ 3.4261,  3.3508,  3.1968,  3.0848,  4.1496,  3.6598,  3.5111 &
& ,  3.4199,  3.3809,  3.5382,  3.2572,  3.2100,  3.1917,  3.1519 &
& ,  3.1198,  3.1005,  3.5071,  3.5086,  3.5073,  3.4509,  3.3120 &
& ,  3.2082,  4.2611,  3.8117,  3.6988,  3.5646,  3.6925,  3.6295 &
& ,  3.5383,  3.4910,  3.4625,  3.4233,  3.4007,  3.2329,  3.6723 &
& ,  3.6845,  3.6876,  3.6197,  3.4799,  3.3737,  4.4341,  4.0525 &
& ,  3.9011,  3.8945,  3.8635,  3.8368,  3.8153,  3.7936,  3.7758 &
& ,  3.4944,  3.4873,  3.9040,  3.7110,  3.6922,  3.6799,  3.6724 &
& ,  3.5622,  3.6081,  3.5426,  3.4922,  3.4498,  3.3984,  3.4456 &
& ,  2.7522,  2.5524,  3.5742,  2.9508,  3.0751,  3.0158,  2.9644 /)
 vdw_dftd3_r0(3011:3080)=&
& (/ 2.8338,  2.7891,  2.6933,  3.6926,  3.1814,  3.4528,  3.4186 &
& ,  3.3836,  3.2213,  3.1626,  3.0507,  4.0548,  3.5312,  3.4244 &
& ,  3.3409,  3.2810,  3.4782,  3.1905,  3.1494,  3.1221,  3.1128 &
& ,  3.0853,  3.0384,  3.4366,  3.4562,  3.4638,  3.3211,  3.2762 &
& ,  3.1730,  4.1632,  3.6825,  3.5822,  3.4870,  3.6325,  3.5740 &
& ,  3.4733,  3.4247,  3.3969,  3.3764,  3.3525,  3.1984,  3.5989 &
& ,  3.6299,  3.6433,  3.4937,  3.4417,  3.3365,  4.3304,  3.9242 &
& ,  3.7793,  3.7623,  3.7327,  3.7071,  3.6860,  3.6650,  3.6476 &
& ,  3.3849,  3.3534,  3.8216,  3.5870,  3.5695,  3.5584,  3.5508 &
& ,  3.4856,  3.5523,  3.4934,  3.4464,  3.4055,  3.3551,  3.3888 /)
 vdw_dftd3_r0(3081:3150)=&
& (/ 3.3525,  2.7202,  2.5183,  3.4947,  2.8731,  3.0198,  3.1457 &
& ,  2.9276,  2.7826,  2.7574,  2.6606,  3.6090,  3.0581,  3.3747 &
& ,  3.3677,  3.3450,  3.1651,  3.1259,  3.0147,  3.9498,  3.3857 &
& ,  3.2917,  3.2154,  3.1604,  3.4174,  3.0735,  3.0342,  3.0096 &
& ,  3.0136,  2.9855,  2.9680,  3.3604,  3.4037,  3.4243,  3.2633 &
& ,  3.1810,  3.1351,  4.0557,  3.5368,  3.4526,  3.3699,  3.5707 &
& ,  3.5184,  3.4085,  3.3595,  3.3333,  3.3143,  3.3041,  3.1094 &
& ,  3.5193,  3.5745,  3.6025,  3.4338,  3.3448,  3.2952,  4.2158 &
& ,  3.7802,  3.6431,  3.6129,  3.5853,  3.5610,  3.5406,  3.5204 &
& ,  3.5036,  3.2679,  3.2162,  3.7068,  3.4483,  3.4323,  3.4221 /)
 vdw_dftd3_r0(3151:3220)=&
& (/ 3.4138,  3.3652,  3.4576,  3.4053,  3.3618,  3.3224,  3.2711 &
& ,  3.3326,  3.2950,  3.2564,  2.5315,  2.6104,  3.2734,  3.2299 &
& ,  3.1090,  2.9942,  2.9159,  2.8324,  2.8350,  2.7216,  3.3994 &
& ,  3.4475,  3.4354,  3.3438,  3.2807,  3.2169,  3.2677,  3.1296 &
& ,  3.7493,  3.8075,  3.6846,  3.6104,  3.5577,  3.2052,  3.4803 &
& ,  3.4236,  3.3845,  3.3640,  3.3365,  3.3010,  3.3938,  3.3624 &
& ,  3.3440,  3.3132,  3.4035,  3.2754,  3.8701,  3.9523,  3.8018 &
& ,  3.7149,  3.3673,  3.3199,  3.2483,  3.2069,  3.1793,  3.1558 &
& ,  3.1395,  3.4097,  3.5410,  3.5228,  3.5116,  3.4921,  3.4781 &
& ,  3.4690,  4.0420,  4.1759,  4.0078,  4.0450,  4.0189,  3.9952 /)
 vdw_dftd3_r0(3221:3290)=&
& (/ 3.9770,  3.9583,  3.9434,  3.7217,  3.8228,  3.7826,  3.8640 &
& ,  3.8446,  3.8314,  3.8225,  3.6817,  3.7068,  3.6555,  3.6159 &
& ,  3.5831,  3.5257,  3.2133,  3.1689,  3.1196,  3.3599,  2.9852 &
& ,  2.7881,  3.5284,  3.3493,  3.6958,  3.3642,  3.1568,  3.0055 &
& ,  2.9558,  2.8393,  3.6287,  3.5283,  4.1511,  3.8259,  3.6066 &
& ,  3.4527,  3.3480,  3.2713,  3.9037,  3.8361,  3.8579,  3.7311 &
& ,  3.6575,  3.5176,  3.5693,  3.5157,  3.4814,  3.4559,  3.4445 &
& ,  3.4160,  4.1231,  3.8543,  3.6816,  3.5602,  3.4798,  3.4208 &
& ,  4.0542,  4.0139,  4.0165,  3.9412,  3.7698,  3.6915,  3.6043 &
& ,  3.5639,  3.5416,  3.5247,  3.5153,  3.5654,  4.2862,  4.0437 /)
 vdw_dftd3_r0(3291:3360)=&
& (/ 3.8871,  3.7741,  3.6985,  3.6413,  4.2345,  4.3663,  4.3257 &
& ,  4.0869,  4.0612,  4.0364,  4.0170,  3.9978,  3.9834,  3.9137 &
& ,  3.8825,  3.8758,  3.9143,  3.8976,  3.8864,  3.8768,  3.9190 &
& ,  4.1613,  4.0566,  3.9784,  3.9116,  3.8326,  3.7122,  3.6378 &
& ,  3.5576,  3.5457,  4.3127,  3.1160,  2.8482,  4.0739,  3.3599 &
& ,  3.5698,  3.5366,  3.2854,  3.1039,  2.9953,  2.9192,  4.1432 &
& ,  3.5320,  3.9478,  4.0231,  3.7509,  3.5604,  3.4340,  3.3426 &
& ,  4.3328,  3.8288,  3.7822,  3.7909,  3.6907,  3.6864,  3.5793 &
& ,  3.5221,  3.4883,  3.4649,  3.4514,  3.4301,  3.9256,  4.0596 &
& ,  3.8307,  3.6702,  3.5651,  3.4884,  4.4182,  4.2516,  3.9687 /)
 vdw_dftd3_r0(3361:3430)=&
& (/ 3.9186,  3.9485,  3.8370,  3.7255,  3.6744,  3.6476,  3.6295 &
& ,  3.6193,  3.5659,  4.0663,  4.2309,  4.0183,  3.8680,  3.7672 &
& ,  3.6923,  4.5240,  4.4834,  4.1570,  4.3204,  4.2993,  4.2804 &
& ,  4.2647,  4.2481,  4.2354,  3.8626,  3.8448,  4.2267,  4.1799 &
& ,  4.1670,  3.8738,  3.8643,  3.8796,  4.0575,  4.0354,  3.9365 &
& ,  3.8611,  3.7847,  3.7388,  3.6826,  3.6251,  3.5492,  4.0889 &
& ,  4.2764,  3.1416,  2.8325,  3.7735,  3.3787,  3.4632,  3.5923 &
& ,  3.3214,  3.1285,  3.0147,  2.9366,  3.8527,  3.5602,  3.8131 &
& ,  3.8349,  3.7995,  3.5919,  3.4539,  3.3540,  4.0654,  3.8603 &
& ,  3.7972,  3.7358,  3.7392,  3.8157,  3.6055,  3.5438,  3.5089 /)
 vdw_dftd3_r0(3431:3500)=&
& (/ 3.4853,  3.4698,  3.4508,  3.7882,  3.8682,  3.8837,  3.7055 &
& ,  3.5870,  3.5000,  4.1573,  4.0005,  3.9568,  3.8936,  3.9990 &
& ,  3.9433,  3.8172,  3.7566,  3.7246,  3.7033,  3.6900,  3.5697 &
& ,  3.9183,  4.0262,  4.0659,  3.8969,  3.7809,  3.6949,  4.2765 &
& ,  4.2312,  4.1401,  4.0815,  4.0580,  4.0369,  4.0194,  4.0017 &
& ,  3.9874,  3.8312,  3.8120,  3.9454,  3.9210,  3.9055,  3.8951 &
& ,  3.8866,  3.8689,  3.9603,  3.9109,  3.9122,  3.8233,  3.7438 &
& ,  3.7436,  3.6981,  3.6555,  3.5452,  3.9327,  4.0658,  4.1175 &
& ,  2.9664,  2.8209,  3.5547,  3.3796,  3.3985,  3.3164,  3.2364 &
& ,  3.1956,  3.0370,  2.9313,  3.6425,  3.5565,  3.7209,  3.7108 /)
 vdw_dftd3_r0(3501:3570)=&
& (/ 3.6639,  3.6484,  3.4745,  3.3492,  3.8755,  4.2457,  3.7758 &
& ,  3.7161,  3.6693,  3.6155,  3.5941,  3.5643,  3.5292,  3.4950 &
& ,  3.4720,  3.4503,  3.6936,  3.7392,  3.7388,  3.7602,  3.6078 &
& ,  3.4960,  3.9800,  4.3518,  4.2802,  3.8580,  3.8056,  3.7527 &
& ,  3.7019,  3.6615,  3.5768,  3.5330,  3.5038,  3.5639,  3.8192 &
& ,  3.8883,  3.9092,  3.9478,  3.7995,  3.6896,  4.1165,  4.5232 &
& ,  4.4357,  4.4226,  4.4031,  4.3860,  4.3721,  4.3580,  4.3466 &
& ,  4.2036,  4.2037,  3.8867,  4.2895,  4.2766,  4.2662,  4.2598 &
& ,  3.8408,  3.9169,  3.8681,  3.8250,  3.7855,  3.7501,  3.6753 &
& ,  3.5499,  3.4872,  3.5401,  3.8288,  3.9217,  3.9538,  4.0054 /)
 vdw_dftd3_r0(3571:3640)=&
& (/ 2.8388,  2.7890,  3.4329,  3.5593,  3.3488,  3.2486,  3.1615 &
& ,  3.1000,  3.0394,  2.9165,  3.5267,  3.7479,  3.6650,  3.6263 &
& ,  3.5658,  3.5224,  3.4762,  3.3342,  3.7738,  4.0333,  3.9568 &
& ,  3.8975,  3.8521,  3.4929,  3.7830,  3.7409,  3.7062,  3.6786 &
& ,  3.6471,  3.6208,  3.6337,  3.6519,  3.6363,  3.6278,  3.6110 &
& ,  3.4825,  3.8795,  4.1448,  4.0736,  4.0045,  3.6843,  3.6291 &
& ,  3.5741,  3.5312,  3.4974,  3.4472,  3.4034,  3.7131,  3.7557 &
& ,  3.7966,  3.8005,  3.8068,  3.8015,  3.6747,  4.0222,  4.3207 &
& ,  4.2347,  4.2191,  4.1990,  4.1811,  4.1666,  4.1521,  4.1401 &
& ,  3.9970,  3.9943,  3.9592,  4.0800,  4.0664,  4.0559,  4.0488 /)
 vdw_dftd3_r0(3641:3710)=&
& (/ 3.9882,  4.0035,  3.9539,  3.9138,  3.8798,  3.8355,  3.5359 &
& ,  3.4954,  3.3962,  3.5339,  3.7595,  3.8250,  3.8408,  3.8600 &
& ,  3.8644,  2.7412,  2.7489,  3.3374,  3.3950,  3.3076,  3.1910 &
& ,  3.0961,  3.0175,  3.0280,  2.8929,  3.4328,  3.5883,  3.6227 &
& ,  3.5616,  3.4894,  3.4241,  3.3641,  3.3120,  3.6815,  3.8789 &
& ,  3.8031,  3.7413,  3.6939,  3.4010,  3.6225,  3.5797,  3.5443 &
& ,  3.5139,  3.4923,  3.4642,  3.5860,  3.5849,  3.5570,  3.5257 &
& ,  3.4936,  3.4628,  3.7874,  3.9916,  3.9249,  3.8530,  3.5932 &
& ,  3.5355,  3.4757,  3.4306,  3.3953,  3.3646,  3.3390,  3.5637 &
& ,  3.7053,  3.7266,  3.7177,  3.6996,  3.6775,  3.6558,  3.9331 /)
 vdw_dftd3_r0(3711:3780)=&
& (/ 4.1655,  4.0879,  4.0681,  4.0479,  4.0299,  4.0152,  4.0006 &
& ,  3.9883,  3.8500,  3.8359,  3.8249,  3.9269,  3.9133,  3.9025 &
& ,  3.8948,  3.8422,  3.8509,  3.7990,  3.7570,  3.7219,  3.6762 &
& ,  3.4260,  3.3866,  3.3425,  3.5294,  3.7022,  3.7497,  3.7542 &
& ,  3.7494,  3.7370,  3.7216,  3.4155,  3.0522,  4.2541,  3.8218 &
& ,  4.0438,  3.5875,  3.3286,  3.1682,  3.0566,  2.9746,  4.3627 &
& ,  4.0249,  4.6947,  4.1718,  3.8639,  3.6735,  3.5435,  3.4479 &
& ,  4.6806,  4.3485,  4.2668,  4.1690,  4.1061,  4.1245,  4.0206 &
& ,  3.9765,  3.9458,  3.9217,  3.9075,  3.8813,  3.9947,  4.1989 &
& ,  3.9507,  3.7960,  3.6925,  3.6150,  4.8535,  4.5642,  4.4134 /)
 vdw_dftd3_r0(3781:3850)=&
& (/ 4.3688,  4.3396,  4.2879,  4.2166,  4.1888,  4.1768,  4.1660 &
& ,  4.1608,  4.0745,  4.2289,  4.4863,  4.2513,  4.0897,  3.9876 &
& ,  3.9061,  5.0690,  5.0446,  4.6186,  4.6078,  4.5780,  4.5538 &
& ,  4.5319,  4.5101,  4.4945,  4.1912,  4.2315,  4.5534,  4.4373 &
& ,  4.4224,  4.4120,  4.4040,  4.2634,  4.7770,  4.6890,  4.6107 &
& ,  4.5331,  4.4496,  4.4082,  4.3095,  4.2023,  4.0501,  4.2595 &
& ,  4.5497,  4.3056,  4.1506,  4.0574,  3.9725,  5.0796,  3.0548 &
& ,  3.3206,  3.8132,  3.9720,  3.7675,  3.7351,  3.5167,  3.5274 &
& ,  3.3085,  3.1653,  3.9500,  4.1730,  4.0613,  4.1493,  3.8823 &
& ,  4.0537,  3.8200,  3.6582,  4.3422,  4.5111,  4.3795,  4.3362 /)
 vdw_dftd3_r0(3851:3920)=&
& (/ 4.2751,  3.7103,  4.1973,  4.1385,  4.1129,  4.0800,  4.0647 &
& ,  4.0308,  4.0096,  4.1619,  3.9360,  4.1766,  3.9705,  3.8262 &
& ,  4.5348,  4.7025,  4.5268,  4.5076,  3.9562,  3.9065,  3.8119 &
& ,  3.7605,  3.7447,  3.7119,  3.6916,  4.1950,  4.2110,  4.3843 &
& ,  4.1631,  4.4427,  4.2463,  4.1054,  4.7693,  5.0649,  4.7365 &
& ,  4.7761,  4.7498,  4.7272,  4.7076,  4.6877,  4.6730,  4.4274 &
& ,  4.5473,  4.5169,  4.5975,  4.5793,  4.5667,  4.5559,  4.3804 &
& ,  4.6920,  4.6731,  4.6142,  4.5600,  4.4801,  4.0149,  3.8856 &
& ,  3.7407,  4.1545,  4.2253,  4.4229,  4.1923,  4.5022,  4.3059 &
& ,  4.1591,  4.7883,  4.9294,  3.3850,  3.4208,  3.7004,  3.8800 /)
 vdw_dftd3_r0(3921:3990)=&
& (/ 3.9886,  3.9040,  3.6719,  3.6547,  3.4625,  3.3370,  3.8394 &
& ,  4.0335,  4.2373,  4.3023,  4.0306,  4.1408,  3.9297,  3.7857 &
& ,  4.1907,  4.3230,  4.2664,  4.2173,  4.1482,  3.6823,  4.0711 &
& ,  4.0180,  4.0017,  3.9747,  3.9634,  3.9383,  4.1993,  4.3205 &
& ,  4.0821,  4.2547,  4.0659,  3.9359,  4.3952,  4.5176,  4.3888 &
& ,  4.3607,  3.9583,  3.9280,  3.8390,  3.7971,  3.7955,  3.7674 &
& ,  3.7521,  4.1062,  4.3633,  4.2991,  4.2767,  4.4857,  4.3039 &
& ,  4.1762,  4.6197,  4.8654,  4.6633,  4.5878,  4.5640,  4.5422 &
& ,  4.5231,  4.5042,  4.4901,  4.3282,  4.3978,  4.3483,  4.4202 &
& ,  4.4039,  4.3926,  4.3807,  4.2649,  4.6135,  4.5605,  4.5232 /)
 vdw_dftd3_r0(3991:4060)=&
& (/ 4.4676,  4.3948,  4.0989,  3.9864,  3.8596,  4.0942,  4.2720 &
& ,  4.3270,  4.3022,  4.5410,  4.3576,  4.2235,  4.6545,  4.7447 &
& ,  4.7043,  3.0942,  3.2075,  3.5152,  3.6659,  3.8289,  3.7459 &
& ,  3.5156,  3.5197,  3.3290,  3.2069,  3.6702,  3.8448,  4.0340 &
& ,  3.9509,  3.8585,  3.9894,  3.7787,  3.6365,  4.1425,  4.1618 &
& ,  4.0940,  4.0466,  3.9941,  3.5426,  3.8952,  3.8327,  3.8126 &
& ,  3.7796,  3.7635,  3.7356,  4.0047,  3.9655,  3.9116,  4.1010 &
& ,  3.9102,  3.7800,  4.2964,  4.3330,  4.2622,  4.2254,  3.8195 &
& ,  3.7560,  3.6513,  3.5941,  3.5810,  3.5420,  3.5178,  3.8861 &
& ,  4.1459,  4.1147,  4.0772,  4.3120,  4.1207,  3.9900,  4.4733 /)
 vdw_dftd3_r0(4061:4130)=&
& (/ 4.6157,  4.4580,  4.4194,  4.3954,  4.3739,  4.3531,  4.3343 &
& ,  4.3196,  4.2140,  4.2339,  4.1738,  4.2458,  4.2278,  4.2158 &
& ,  4.2039,  4.1658,  4.3595,  4.2857,  4.2444,  4.1855,  4.1122 &
& ,  3.7839,  3.6879,  3.5816,  3.8633,  4.1585,  4.1402,  4.1036 &
& ,  4.3694,  4.1735,  4.0368,  4.5095,  4.5538,  4.5240,  4.4252 &
& ,  3.0187,  3.1918,  3.5127,  3.6875,  3.7404,  3.6943,  3.4702 &
& ,  3.4888,  3.2914,  3.1643,  3.6669,  3.8724,  3.9940,  4.0816 &
& ,  3.8054,  3.9661,  3.7492,  3.6024,  4.0428,  4.1951,  4.1466 &
& ,  4.0515,  4.0075,  3.5020,  3.9158,  3.8546,  3.8342,  3.8008 &
& ,  3.7845,  3.7549,  3.9602,  3.8872,  3.8564,  4.0793,  3.8835 /)
 vdw_dftd3_r0(4131:4200)=&
& (/ 3.7495,  4.2213,  4.3704,  4.3300,  4.2121,  3.7643,  3.7130 &
& ,  3.6144,  3.5599,  3.5474,  3.5093,  3.4853,  3.9075,  4.1115 &
& ,  4.0473,  4.0318,  4.2999,  4.1050,  3.9710,  4.4320,  4.6706 &
& ,  4.5273,  4.4581,  4.4332,  4.4064,  4.3873,  4.3684,  4.3537 &
& ,  4.2728,  4.2549,  4.2032,  4.2794,  4.2613,  4.2491,  4.2375 &
& ,  4.2322,  4.3665,  4.3061,  4.2714,  4.2155,  4.1416,  3.7660 &
& ,  3.6628,  3.5476,  3.8790,  4.1233,  4.0738,  4.0575,  4.3575 &
& ,  4.1586,  4.0183,  4.4593,  4.5927,  4.4865,  4.3813,  4.4594 &
& ,  2.9875,  3.1674,  3.4971,  3.6715,  3.7114,  3.6692,  3.4446 &
& ,  3.4676,  3.2685,  3.1405,  3.6546,  3.8579,  3.9637,  4.0581 /)
 vdw_dftd3_r0(4201:4270)=&
& (/ 3.7796,  3.9463,  3.7275,  3.5792,  4.0295,  4.1824,  4.1247 &
& ,  4.0357,  3.9926,  3.4827,  3.9007,  3.8392,  3.8191,  3.7851 &
& ,  3.7687,  3.7387,  3.9290,  3.8606,  3.8306,  4.0601,  3.8625 &
& ,  3.7269,  4.2062,  4.3566,  4.3022,  4.1929,  3.7401,  3.6888 &
& ,  3.5900,  3.5350,  3.5226,  3.4838,  3.4594,  3.8888,  4.0813 &
& ,  4.0209,  4.0059,  4.2810,  4.0843,  3.9486,  4.4162,  4.6542 &
& ,  4.5005,  4.4444,  4.4196,  4.3933,  4.3741,  4.3552,  4.3406 &
& ,  4.2484,  4.2413,  4.1907,  4.2656,  4.2474,  4.2352,  4.2236 &
& ,  4.2068,  4.3410,  4.2817,  4.2479,  4.1921,  4.1182,  3.7346 &
& ,  3.6314,  3.5168,  3.8582,  4.0927,  4.0469,  4.0313,  4.3391 /)
 vdw_dftd3_r0(4271:4340)=&
& (/ 4.1381,  3.9962,  4.4429,  4.5787,  4.4731,  4.3588,  4.4270 &
& ,  4.3957,  2.9659,  3.1442,  3.4795,  3.6503,  3.6814,  3.6476 &
& ,  3.4222,  3.4491,  3.2494,  3.1209,  3.6324,  3.8375,  3.9397 &
& ,  3.8311,  3.7581,  3.9274,  3.7085,  3.5598,  4.0080,  4.1641 &
& ,  4.1057,  4.0158,  3.9726,  3.4667,  3.8802,  3.8188,  3.7989 &
& ,  3.7644,  3.7474,  3.7173,  3.9049,  3.8424,  3.8095,  4.0412 &
& ,  3.8436,  3.7077,  4.1837,  4.3366,  4.2816,  4.1686,  3.7293 &
& ,  3.6709,  3.5700,  3.5153,  3.5039,  3.4684,  3.4437,  3.8663 &
& ,  4.0575,  4.0020,  3.9842,  4.2612,  4.0643,  3.9285,  4.3928 &
& ,  4.6308,  4.4799,  4.4244,  4.3996,  4.3737,  4.3547,  4.3358 /)
 vdw_dftd3_r0(4341:4410)=&
& (/ 4.3212,  4.2275,  4.2216,  4.1676,  4.2465,  4.2283,  4.2161 &
& ,  4.2045,  4.1841,  4.3135,  4.2562,  4.2226,  4.1667,  4.0932 &
& ,  3.7134,  3.6109,  3.4962,  3.8352,  4.0688,  4.0281,  4.0099 &
& ,  4.3199,  4.1188,  3.9768,  4.4192,  4.5577,  4.4516,  4.3365 &
& ,  4.4058,  4.3745,  4.3539,  2.8763,  3.1294,  3.5598,  3.7465 &
& ,  3.5659,  3.5816,  3.3599,  3.4024,  3.1877,  3.0484,  3.7009 &
& ,  3.9451,  3.8465,  3.9873,  3.7079,  3.9083,  3.6756,  3.5150 &
& ,  4.0829,  4.2780,  4.1511,  4.1260,  4.0571,  3.4865,  3.9744 &
& ,  3.9150,  3.8930,  3.8578,  3.8402,  3.8073,  3.7977,  4.0036 &
& ,  3.7604,  4.0288,  3.8210,  3.6757,  4.2646,  4.4558,  4.2862 /)
 vdw_dftd3_r0(4411:4465)=&
& (/ 4.2122,  3.7088,  3.6729,  3.5800,  3.5276,  3.5165,  3.4783 &
& ,  3.4539,  3.9553,  3.9818,  4.2040,  3.9604,  4.2718,  4.0689 &
& ,  3.9253,  4.4869,  4.7792,  4.4918,  4.5342,  4.5090,  4.4868 &
& ,  4.4680,  4.4486,  4.4341,  4.2023,  4.3122,  4.2710,  4.3587 &
& ,  4.3407,  4.3281,  4.3174,  4.1499,  4.3940,  4.3895,  4.3260 &
& ,  4.2725,  4.1961,  3.7361,  3.6193,  3.4916,  3.9115,  3.9914 &
& ,  3.9809,  3.9866,  4.3329,  4.1276,  3.9782,  4.5097,  4.6769 &
& ,  4.5158,  4.3291,  4.3609,  4.3462,  4.3265,  4.4341 /)

 index_c6(1:100)=[&
& 1,2,3,4,5,6,7,8,9,10,11, &
& 12,13,14,15,16,17,18,19,20,21,22, &
& 23,24,25,26,27,28,29,30,31,32,33, &
& 34,35,36,37,38,39,40,41,42,43,44, &
& 45,46,47,48,49,50,51,52,53,54,55, &
& 56,57,58,59,60,61,62,63,64,65,66, &
& 67,68,69,70,71,72,73,74,75,76,77, &
& 78,79,80,81,82,83,84,85,86,87,88, &
& 89,90,91,92,93,94,101,103,104,105,106, &
& 107]
 index_c6(101:200)=[&
& 108,109,111,112,113,114,115,116,117,119,120, &
& 121,122,123,124,125,126,127,128,129,130,131, &
& 132,133,134,135,137,138,139,140,141,142,143, &
& 144,145,146,147,148,149,150,151,152,153,155, &
& 156,157,159,160,161,162,163,164,165,166,167, &
& 168,169,170,171,172,173,174,175,176,177,178, &
& 179,180,181,182,183,184,185,187,188,189,190, &
& 191,192,193,194,204,205,206,207,208,212,213, &
& 214,215,216,220,221,222,223,224,225,226,227, &
& 228]
 index_c6(201:254)=[&
& 231,232,233,234,238,239,240,241,242,243,244, &
& 245,246,249,250,251,252,256,257,272,273,274, &
& 275,276,277,278,281,282,283,284,288,305,306, &
& 307,313,314,315,327,328,331,332,333,349,350, &
& 351,381,382,383,405,406,414,432,450,482]

 vdw_dftd3_c6(1:100)=[&
& 3.0267000e+00,2.0835000e+00,1.5583000e+00,3.8944800e+01,2.2150800e+01, &
& 1.1634454e+03,2.4441500e+01,1.4824600e+01,4.9461900e+02,2.5748630e+02, &
& 1.7314300e+01,1.1097500e+01,2.8373080e+02,1.6159710e+02,1.0717770e+02, &
& 1.2140200e+01,8.1841000e+00,1.6990300e+02,1.0295600e+02,7.1279400e+01, &
& 4.9113000e+01,8.7171000e+00,6.1380000e+00,1.0848540e+02,6.8645800e+01, &
& 4.9113200e+01,3.4814600e+01,2.5268500e+01,6.7180000e+00,4.8949000e+00, &
& 7.6961300e+01,5.0125200e+01,3.6724700e+01,2.6592900e+01,1.9654600e+01, &
& 1.5505900e+01,5.1616000e+00,3.8825000e+00,5.5093300e+01,3.6745300e+01, &
& 2.7482100e+01,2.0282700e+01,1.5241800e+01,1.2183400e+01,9.6916000e+00, &
& 4.0112000e+00,3.1025000e+00,4.0473100e+01,2.7486700e+01,2.0902200e+01, &
& 1.5674000e+01,1.1947900e+01,9.6606000e+00,7.7691000e+00,6.2896000e+00, &
& 4.6823200e+01,2.6862800e+01,1.3673272e+03,5.8745630e+02,3.3872120e+02, &
& 2.0376310e+02,1.3065630e+02,9.3026300e+01,6.6842300e+01,4.9279900e+01, &
& 1.6080286e+03,3.8353100e+01,2.3032000e+01,8.3081560e+02,4.1821640e+02, &
& 2.5813030e+02,1.6260820e+02,1.0761500e+02,7.8225000e+01,5.7160500e+01, &
& 4.2677100e+01,9.8516970e+02,6.8337580e+02,3.6290900e+01,2.2322400e+01, &
& 7.0582540e+02,3.7263020e+02,2.3647800e+02,1.5209340e+02,1.0220000e+02, &
& 7.5075500e+01,5.5341200e+01,4.1596600e+01,8.3896480e+02,6.0346890e+02, &
& 5.4054060e+02,2.9594700e+01,1.8850000e+01,4.9534490e+02,2.7978630e+02, &
& 1.8451110e+02,1.2213870e+02,8.3849800e+01,6.2534900e+01,4.6693600e+01]
 vdw_dftd3_c6(101:200)=[&
& 3.5455000e+01,5.9104580e+02,4.4764230e+02,4.0896060e+02,3.1785740e+02, &
& 2.3760400e+01,1.5668900e+01,3.5080300e+02,2.0873310e+02,1.4234810e+02, &
& 9.6750300e+01,6.7787300e+01,5.1309800e+01,3.8808400e+01,2.9776700e+01, &
& 4.2000640e+02,3.3078010e+02,3.0729650e+02,2.4435460e+02,1.9168870e+02, &
& 2.0094800e+01,1.3610800e+01,2.7378670e+02,1.6795130e+02,1.1711210e+02, &
& 8.1091900e+01,5.7673400e+01,4.4147000e+01,3.3726400e+01,2.6094000e+01, &
& 3.2859900e+02,2.6466650e+02,2.4850080e+02,2.0053740e+02,1.5948980e+02, &
& 1.3400660e+02,1.6705200e+01,1.1630200e+01,2.1066260e+02,1.3298080e+02, &
& 9.4761200e+01,6.6840700e+01,4.8262400e+01,3.7368800e+01,2.8844500e+01, &
& 2.2512100e+01,2.5351360e+02,2.0849780e+02,1.9775940e+02,1.6186860e+02, &
& 1.3047250e+02,1.1070060e+02,9.2346000e+01,1.3870000e+01,9.9130000e+00, &
& 1.6354970e+02,1.0572290e+02,7.6794900e+01,5.5089800e+01,4.0343500e+01, &
& 3.1578300e+01,2.4617800e+01,1.9377400e+01,1.9734400e+02,1.6510060e+02, &
& 1.5795950e+02,1.3089270e+02,1.0676980e+02,9.1401400e+01,7.6938300e+01, &
& 6.4646200e+01,7.6237600e+01,4.4041100e+01,2.3871574e+03,9.7231970e+02, &
& 5.5419800e+02,3.3226940e+02,2.1320010e+02,1.5214200e+02,1.0964390e+02, &
& 8.1086100e+01,2.7986124e+03,1.6420587e+03,1.3879361e+03,9.6762830e+02, &
& 6.8449680e+02,5.3523840e+02,4.1315120e+02,3.2211550e+02,4.9835009e+03, &
& 6.5818000e+01,3.9070100e+01,1.6144719e+03,7.5790700e+02,4.5484390e+02, &
& 2.8170350e+02,1.8454980e+02,1.3338470e+02,9.7082400e+01,7.2307200e+01]
 vdw_dftd3_c6(201:300)=[&
& 1.9077081e+03,1.2525932e+03,1.0888420e+03,7.9063470e+02,5.7546270e+02, &
& 4.5684430e+02,3.5747370e+02,2.8168230e+02,3.2404393e+03,2.3526862e+03, &
& 5.4966900e+01,3.2899500e+01,1.2781183e+03,6.1733580e+02,3.7504180e+02, &
& 2.3419500e+02,1.5429470e+02,1.1193590e+02,8.1724200e+01,6.1015400e+01, &
& 1.5125338e+03,1.0155454e+03,8.8860250e+02,6.5120400e+02,4.7727970e+02, &
& 3.8041070e+02,2.9876860e+02,2.3614170e+02,2.5499412e+03,1.8887902e+03, &
& 1.5224676e+03,5.3687500e+01,3.2531600e+01,1.1929128e+03,5.8748980e+02, &
& 3.6108650e+02,2.2757230e+02,1.5101990e+02,1.1014520e+02,8.0797500e+01, &
& 6.0559400e+01,1.4131570e+03,9.6298310e+02,8.4746150e+02,6.2623530e+02, &
& 4.6232470e+02,3.7026310e+02,2.9218530e+02,2.3192080e+02,2.3746690e+03, &
& 1.7795162e+03,1.4382841e+03,1.3619185e+03,4.9481900e+01,3.0185100e+01, &
& 1.0690426e+03,5.3334990e+02,3.3010230e+02,2.0914500e+02,1.3935120e+02, &
& 1.0193250e+02,7.4967100e+01,5.6310800e+01,1.2673139e+03,8.7225790e+02, &
& 7.7034320e+02,5.7211290e+02,4.2414700e+02,3.4060200e+02,2.6948630e+02, &
& 2.1440020e+02,2.1241985e+03,1.6049178e+03,1.2995197e+03,1.2323235e+03, &
& 1.1160984e+03,3.9122100e+01,2.4146300e+01,8.4409360e+02,4.1816350e+02, &
& 2.5930210e+02,1.6498820e+02,1.1048610e+02,8.1200000e+01,6.0009900e+01, &
& 4.5282000e+01,1.0004652e+03,6.8440150e+02,6.0440710e+02,4.4922160e+02, &
& 3.3383800e+02,2.6880290e+02,2.1336680e+02,1.7034650e+02,1.6836701e+03, &
& 1.2630809e+03,1.0216871e+03,9.6885650e+02,8.7738550e+02,6.9074250e+02]
 vdw_dftd3_c6(301:400)=[&
& 4.3002800e+01,2.6497800e+01,8.9197560e+02,4.5336860e+02,2.8347440e+02, &
& 1.8100350e+02,1.2132330e+02,8.9134600e+01,6.5811200e+01,4.9595900e+01, &
& 1.0585103e+03,7.3904110e+02,6.5607380e+02,4.9081560e+02,3.6612040e+02, &
& 2.9517370e+02,2.3445330e+02,1.8717280e+02,1.7680322e+03,1.3514785e+03, &
& 1.0971539e+03,1.0426625e+03,9.4560720e+02,7.4331790e+02,8.0274840e+02, &
& 3.3911000e+01,2.1175000e+01,6.9845390e+02,3.5353580e+02,2.2175470e+02, &
& 1.4234670e+02,9.5978200e+01,7.0894200e+01,5.2631000e+01,3.9865900e+01, &
& 8.2889360e+02,5.7651210e+02,5.1208230e+02,3.8374450e+02,2.8716060e+02, &
& 2.3226250e+02,1.8518230e+02,1.4843190e+02,1.3889522e+03,1.0565193e+03, &
& 8.5717300e+02,8.1484020e+02,7.3905420e+02,5.8181140e+02,6.2754030e+02, &
& 4.9133490e+02,3.6323400e+01,2.2723300e+01,7.0153240e+02,3.6919280e+02, &
& 2.3486930e+02,1.5187030e+02,1.0275030e+02,7.5991200e+01,5.6430700e+01, &
& 4.2727000e+01,8.3418260e+02,5.9833020e+02,5.3597060e+02,4.0599070e+02, &
& 3.0592560e+02,2.4821160e+02,1.9835700e+02,1.5919570e+02,1.3829771e+03, &
& 1.0817690e+03,8.8245030e+02,8.4183350e+02,7.6531780e+02,6.0141780e+02, &
& 6.5196880e+02,5.0977640e+02,5.3277940e+02,3.7159600e+01,2.3079200e+01, &
& 7.3885800e+02,3.8391400e+02,2.4241740e+02,1.5585070e+02,1.0498340e+02, &
& 7.7401000e+01,5.7324800e+01,4.3312900e+01,8.7793540e+02,6.2363050e+02, &
& 5.5653890e+02,4.1935930e+02,3.1456980e+02,2.5446830e+02,2.0277050e+02, &
& 1.6232640e+02,1.4584501e+03,1.1321074e+03,9.2189900e+02,8.7809390e+02]
 vdw_dftd3_c6(401:500)=[&
& 7.9751950e+02,6.2664720e+02,6.7845310e+02,5.3032210e+02,5.5308340e+02, &
& 5.7474360e+02,2.8594000e+01,1.8021600e+01,5.6945260e+02,2.9279550e+02, &
& 1.8517410e+02,1.1963500e+02,8.1085500e+01,6.0133500e+01,4.4809800e+01, &
& 3.4054400e+01,6.7647730e+02,4.7624260e+02,4.2477230e+02,3.2019130e+02, &
& 2.4079740e+02,1.9540750e+02,1.5631610e+02,1.2567760e+02,1.1299518e+03, &
& 8.6836740e+02,7.0607200e+02,6.7241280e+02,6.1057000e+02,4.8069380e+02, &
& 5.1930780e+02,4.0672680e+02,4.2308770e+02,4.3963650e+02,3.3718080e+02, &
& 2.9868900e+01,1.8969500e+01,5.4091600e+02,2.9331060e+02,1.8962930e+02, &
& 1.2410320e+02,8.4731100e+01,6.3077400e+01,4.7112200e+01,3.5842700e+01, &
& 6.4434080e+02,4.7291430e+02,4.2716100e+02,3.2728100e+02,2.4897430e+02, &
& 2.0324270e+02,1.6338480e+02,1.3181190e+02,1.0625928e+03,8.4685270e+02, &
& 6.9367870e+02,6.6410090e+02,6.0506230e+02,4.7555940e+02,5.1709100e+02, &
& 4.0455580e+02,4.2487900e+02,4.4009000e+02,3.3664930e+02,3.4052130e+02, &
& 3.5169700e+01,2.2045800e+01,6.5674410e+02,3.5111410e+02,2.2539330e+02, &
& 1.4655440e+02,9.9440700e+01,7.3625300e+01,5.4678200e+01,4.1368100e+01, &
& 7.8135240e+02,5.6716380e+02,5.1058540e+02,3.8926560e+02,2.9479250e+02, &
& 2.3981830e+02,1.9205780e+02,1.5434230e+02,1.2928121e+03,1.0198835e+03, &
& 8.3374240e+02,7.9686170e+02,7.2525140e+02,5.6976450e+02,6.1883870e+02, &
& 4.8378150e+02,5.0713160e+02,5.2581620e+02,4.0196130e+02,4.0545650e+02, &
& 4.8375160e+02,3.1817000e+01,2.0431800e+01,5.2792680e+02,2.9842200e+02]
 vdw_dftd3_c6(501:600)=[&
& 1.9734480e+02,1.3109540e+02,9.0332800e+01,6.7593200e+01,5.0641200e+01, &
& 3.8574600e+01,6.3010410e+02,4.7737440e+02,4.3650600e+02,3.3983280e+02, &
& 2.6182350e+02,2.1531010e+02,1.7419690e+02,1.4120630e+02,1.0326651e+03, &
& 8.4336440e+02,6.9470240e+02,6.6837860e+02,6.1076390e+02,4.7996390e+02, &
& 5.2418480e+02,4.1021060e+02,4.3385080e+02,4.4798680e+02,3.4242610e+02, &
& 3.4998850e+02,4.1595300e+02,3.6354740e+02,2.7788400e+01,1.8328400e+01, &
& 4.1515320e+02,2.4548310e+02,1.6690600e+02,1.1326160e+02,7.9308600e+01, &
& 6.0028900e+01,4.5418000e+01,3.4868600e+01,4.9693380e+02,3.8949270e+02, &
& 3.6119130e+02,2.8659090e+02,2.2447740e+02,1.8663480e+02,1.5260350e+02, &
& 1.2485610e+02,8.1065540e+02,6.7900440e+02,5.6269470e+02,5.4467970e+02, &
& 4.9949700e+02,3.9319220e+02,4.3091460e+02,3.3800960e+02,3.5971500e+02, &
& 3.7003590e+02,2.8333230e+02,2.9250300e+02,3.4635080e+02,3.0710100e+02, &
& 2.6294980e+02,2.5309800e+01,1.7024300e+01,3.5454290e+02,2.1502380e+02, &
& 1.4880080e+02,1.0243460e+02,7.2540300e+01,5.5362700e+01,4.2192300e+01, &
& 3.2584300e+01,4.2518640e+02,3.3957420e+02,3.1761820e+02,2.5500990e+02, &
& 2.0190190e+02,1.6913080e+02,1.3931630e+02,1.1474490e+02,6.9291160e+02, &
& 5.8796470e+02,4.8887150e+02,4.7503480e+02,4.3657180e+02,3.4429560e+02, &
& 3.7782760e+02,2.9703960e+02,3.1701740e+02,3.2534190e+02,2.4963850e+02, &
& 2.5904590e+02,3.0595010e+02,2.7365300e+02,2.3634730e+02,2.1367380e+02, &
& 2.2483400e+01,1.5455300e+01,2.9478970e+02,1.8337420e+02,1.2924950e+02]
 vdw_dftd3_c6(601:700)=[&
& 9.0342400e+01,6.4763200e+01,4.9879500e+01,3.8322500e+01,2.9795800e+01, &
& 3.5428690e+02,2.8827360e+02,2.7200980e+02,2.2105370e+02,1.7699300e+02, &
& 1.4945630e+02,1.2409090e+02,1.0294710e+02,5.7723180e+02,4.9597540e+02, &
& 4.1374680e+02,4.0365760e+02,3.7180710e+02,2.9391850e+02,3.2284330e+02, &
& 2.5453060e+02,2.7230530e+02,2.7877050e+02,2.1450590e+02,2.2363480e+02, &
& 2.6334580e+02,2.3762950e+02,2.0707720e+02,1.8836050e+02,1.6712970e+02, &
& 1.9818200e+01,1.3921100e+01,2.4455540e+02,1.5556160e+02,1.1153710e+02, &
& 7.9108000e+01,5.7389500e+01,4.4600300e+01,3.4545200e+01,2.7043000e+01, &
& 2.9455740e+02,2.4359650e+02,2.3169040e+02,1.9039340e+02,1.5405750e+02, &
& 1.3109280e+02,1.0968390e+02,9.1643000e+01,4.8024720e+02,4.1697770e+02, &
& 3.4884740e+02,3.4162730e+02,3.1532510e+02,2.4994130e+02,2.7464190e+02, &
& 2.1720710e+02,2.3275940e+02,2.3774470e+02,1.8353470e+02,1.9204840e+02, &
& 2.2545370e+02,2.0505980e+02,1.8018180e+02,1.6484940e+02,1.4718300e+02, &
& 1.3040170e+02,8.5949900e+01,5.0049100e+01,2.6473331e+03,1.0827052e+03, &
& 6.2008780e+02,3.7353840e+02,2.4069400e+02,1.7233980e+02,1.2458790e+02, &
& 9.2382700e+01,3.1041016e+03,1.8264648e+03,1.5469229e+03,1.0820563e+03, &
& 7.6813350e+02,6.0222210e+02,4.6615650e+02,3.6440670e+02,5.5302806e+03, &
& 3.5994349e+03,2.8340920e+03,2.6412435e+03,2.3637175e+03,1.8740326e+03, &
& 1.9687631e+03,1.5471982e+03,1.5418997e+03,1.6251034e+03,1.2594427e+03, &
& 1.1862631e+03,1.4423807e+03,1.1551526e+03,9.0942570e+02,7.7892040e+02]
 vdw_dftd3_c6(701:800)=[&
& 6.5036060e+02,5.4231600e+02,6.1387755e+03,7.8390100e+01,4.6665800e+01, &
& 1.9442260e+03,9.0392650e+02,5.4157290e+02,3.3545970e+02,2.1996080e+02, &
& 1.5914600e+02,1.1596470e+02,8.6462700e+01,2.2961906e+03,1.4958141e+03, &
& 1.2985356e+03,9.4141210e+02,6.8496320e+02,5.4396270e+02,4.2589970e+02, &
& 3.3586900e+02,3.9153529e+03,2.8194935e+03,2.2604217e+03,2.1283951e+03, &
& 1.9186941e+03,1.5113043e+03,1.6147190e+03,1.2632327e+03,1.2908718e+03, &
& 1.3513702e+03,1.0377284e+03,1.0096809e+03,1.2165138e+03,1.0044824e+03, &
& 8.0833700e+02,7.0005320e+02,5.9073260e+02,4.9691200e+02,4.3492577e+03, &
& 3.3813672e+03,7.0057000e+01,4.2419500e+01,1.5853950e+03,7.7148440e+02, &
& 4.7243640e+02,2.9721710e+02,1.9706110e+02,1.4366020e+02,1.0534470e+02, &
& 7.8931600e+01,1.8768634e+03,1.2667602e+03,1.1122980e+03,8.1954180e+02, &
& 6.0399160e+02,4.8337970e+02,3.8124620e+02,3.0251430e+02,3.1660625e+03, &
& 2.3508018e+03,1.8967809e+03,1.7943180e+03,1.6224689e+03,1.2764352e+03, &
& 1.3714492e+03,1.0723265e+03,1.1053016e+03,1.1535737e+03,8.8416480e+02, &
& 8.7069350e+02,1.0455505e+03,8.7478360e+02,7.1175060e+02,6.2033340e+02, &
& 5.2682990e+02,4.4570130e+02,3.5209466e+03,2.8137007e+03,2.3658925e+03, &
& 6.3780100e+01,3.9108800e+01,1.3632842e+03,6.8152830e+02,4.2323020e+02, &
& 2.6904900e+02,1.7977770e+02,1.3178520e+02,9.7097700e+01,7.3031500e+01, &
& 1.6162863e+03,1.1138339e+03,9.8504940e+02,7.3318070e+02,5.4488820e+02, &
& 4.3838900e+02,3.4753340e+02,2.7699080e+02,2.7111542e+03,2.0485251e+03]
 vdw_dftd3_c6(801:900)=[&
& 1.6591156e+03,1.5741401e+03,1.4260687e+03,1.1215489e+03,1.2087451e+03, &
& 9.4517130e+02,9.7897670e+02,1.0197589e+03,7.8111330e+02,7.7458710e+02, &
& 9.2819820e+02,7.8294420e+02,6.4155640e+02,5.6154740e+02,4.7900400e+02, &
& 4.0687270e+02,3.0176856e+03,2.4494723e+03,2.0726170e+03,1.8227181e+03, &
& 5.8676800e+01,3.6341800e+01,1.2090304e+03,6.1398500e+02,3.8490720e+02, &
& 2.4652410e+02,1.6569430e+02,1.2198680e+02,9.0223400e+01,6.8076900e+01, &
& 1.4346641e+03,1.0005663e+03,8.8903020e+02,6.6615070e+02,4.9797030e+02, &
& 4.0219740e+02,3.2006760e+02,2.5597690e+02,2.4006918e+03,1.8308523e+03, &
& 1.4860620e+03,1.4126818e+03,1.2813349e+03,1.0078571e+03,1.0879837e+03, &
& 8.5106340e+02,8.8388900e+02,9.1953540e+02,7.0437890e+02,7.0135490e+02, &
& 8.3928280e+02,7.1168230e+02,5.8600840e+02,5.1450770e+02,4.4029990e+02, &
& 3.7513390e+02,2.6739344e+03,2.1882864e+03,1.8586377e+03,1.6384938e+03, &
& 1.4752500e+03,4.6040600e+01,2.9161700e+01,8.9051260e+02,4.6330650e+02, &
& 2.9535920e+02,1.9195280e+02,1.3060450e+02,9.7067500e+01,7.2418600e+01, &
& 5.5048700e+01,1.0583899e+03,7.5166280e+02,6.7311930e+02,5.1023520e+02, &
& 3.8558950e+02,3.1386570e+02,2.5177230e+02,2.0285720e+02,1.7658946e+03, &
& 1.3654837e+03,1.1120098e+03,1.0606255e+03,9.6394340e+02,7.5899550e+02, &
& 8.2093130e+02,6.4312470e+02,6.7034900e+02,6.9583480e+02,5.3365350e+02, &
& 5.3452070e+02,6.3792330e+02,5.4578450e+02,4.5341860e+02,4.0049980e+02, &
& 3.4498260e+02,2.9579940e+02,1.9694796e+03,1.6315424e+03,1.3940976e+03]
 vdw_dftd3_c6(901:1000)=[&
& 1.2339095e+03,1.1140924e+03,8.4589720e+02,5.1052700e+01,3.2058800e+01, &
& 1.0116099e+03,5.2142490e+02,3.3032570e+02,2.1347990e+02,1.4456970e+02, &
& 1.0705880e+02,7.9612100e+01,6.0352100e+01,1.2015234e+03,8.4734970e+02, &
& 7.5661950e+02,5.7105010e+02,4.2976770e+02,3.4877440e+02,2.7891230e+02, &
& 2.2407940e+02,2.0076137e+03,1.5434258e+03,1.2553446e+03,1.1958434e+03, &
& 1.0860231e+03,8.5474740e+02,9.2387430e+02,7.2332020e+02,7.5296780e+02, &
& 7.8224830e+02,5.9961920e+02,5.9931410e+02,7.1598110e+02,6.1053270e+02, &
& 5.0550910e+02,4.4547780e+02,3.8275910e+02,3.2738270e+02,2.2379665e+03, &
& 1.8443774e+03,1.5723759e+03,1.3895979e+03,1.2533465e+03,9.4964900e+02, &
& 1.0670169e+03,3.9574500e+01,2.5462200e+01,7.3055330e+02,3.8726810e+02, &
& 2.4993820e+02,1.6412790e+02,1.1263780e+02,8.4272500e+01,6.3259900e+01, &
& 4.8342900e+01,8.6937740e+02,6.2621040e+02,5.6407160e+02,4.3121320e+02, &
& 3.2840720e+02,2.6878240e+02,2.1680700e+02,1.7558760e+02,1.4473048e+03, &
& 1.1311769e+03,9.2358110e+02,8.8314920e+02,8.0388070e+02,6.3343940e+02, &
& 6.8617720e+02,5.3814350e+02,5.6248000e+02,5.8290140e+02,4.4742590e+02, &
& 4.5015430e+02,5.3610930e+02,4.6167950e+02,3.8598830e+02,3.4238140e+02, &
& 2.9626680e+02,2.5515160e+02,1.6158387e+03,1.3511988e+03,1.1599071e+03, &
& 1.0297665e+03,9.3174450e+02,7.1028370e+02,7.9619510e+02,5.9819880e+02, &
& 4.3166100e+01,2.7574500e+01,7.9420060e+02,4.2441200e+02,2.7371950e+02, &
& 1.7927080e+02,1.2264260e+02,9.1495900e+01,6.8488800e+01,5.2207200e+01]
 vdw_dftd3_c6(1001:1100)=[&
& 9.4542210e+02,6.8566280e+02,6.1792440e+02,4.7237380e+02,3.5925470e+02, &
& 2.9352140e+02,2.3627500e+02,1.9093520e+02,1.5674954e+03,1.2346013e+03, &
& 1.0091933e+03,9.6523080e+02,8.7880270e+02,6.9163140e+02,7.5032510e+02, &
& 5.8775440e+02,6.1543640e+02,6.3776340e+02,4.8875010e+02,4.9263190e+02, &
& 5.8677320e+02,5.0544520e+02,4.2219710e+02,3.7406320e+02,3.2319410e+02, &
& 2.7786800e+02,1.7497615e+03,1.4735487e+03,1.2668436e+03,1.1253403e+03, &
& 1.0182915e+03,7.7583530e+02,8.6989620e+02,6.5317750e+02,7.1394270e+02, &
& 4.0237200e+01,2.5871600e+01,7.2242940e+02,3.9043100e+02,2.5334250e+02, &
& 1.6671110e+02,1.1447630e+02,8.5642700e+01,6.4270700e+01,4.9099000e+01, &
& 8.6062090e+02,6.2959120e+02,5.6914100e+02,4.3694630e+02,3.3352190e+02, &
& 2.7315680e+02,2.2040990e+02,1.7850210e+02,1.4240271e+03,1.1296670e+03, &
& 9.2485560e+02,8.8576570e+02,8.0712580e+02,6.3532570e+02,6.8996470e+02, &
& 5.4065990e+02,5.6710440e+02,5.8718190e+02,4.5005590e+02,4.5480310e+02, &
& 5.4115010e+02,4.6770420e+02,3.9185280e+02,3.4783930e+02,3.0113900e+02, &
& 2.5939640e+02,1.5904415e+03,1.3479007e+03,1.1619289e+03,1.0338735e+03, &
& 9.3655280e+02,7.1494950e+02,8.0104420e+02,6.0279560e+02,6.5887530e+02, &
& 6.0850410e+02,3.3541300e+01,2.1789600e+01,6.1414770e+02,3.2557750e+02, &
& 2.1068110e+02,1.3887580e+02,9.5713700e+01,7.1892600e+01,5.4190300e+01, &
& 4.1578000e+01,7.3108320e+02,5.2647760e+02,4.7457130e+02,3.6335850e+02, &
& 2.7735770e+02,2.2750240e+02,1.8398760e+02,1.4942690e+02,1.2184699e+03]
 vdw_dftd3_c6(1101:1200)=[&
& 9.5149350e+02,7.7689140e+02,7.4320410e+02,6.7665530e+02,5.3368780e+02, &
& 5.7781270e+02,4.5363270e+02,4.7392440e+02,4.9097830e+02,3.7735310e+02, &
& 3.7956550e+02,4.5158870e+02,3.8930750e+02,3.2603710e+02,2.8964690e+02, &
& 2.5111890e+02,2.1673360e+02,1.3607517e+03,1.1369099e+03,9.7620250e+02, &
& 8.6700560e+02,7.8481880e+02,5.9905300e+02,6.7117170e+02,5.0502690e+02, &
& 5.5108020e+02,5.0877420e+02,4.2674500e+02,3.5687900e+01,2.3156300e+01, &
& 6.2105360e+02,3.4041220e+02,2.2262730e+02,1.4741380e+02,1.0173770e+02, &
& 7.6409200e+01,5.7550000e+01,4.4106200e+01,7.4057870e+02,5.4767210e+02, &
& 4.9703370e+02,3.8367900e+02,2.9424650e+02,2.4176140e+02,1.9570440e+02, &
& 1.5896500e+02,1.2225553e+03,9.7842510e+02,8.0260200e+02,7.7003030e+02, &
& 7.0242130e+02,5.5309190e+02,6.0140530e+02,4.7153290e+02,4.9563130e+02, &
& 5.1262200e+02,3.9305100e+02,3.9845720e+02,4.7342030e+02,4.1090580e+02, &
& 3.4560850e+02,3.0755390e+02,2.6697000e+02,2.3054930e+02,1.3663938e+03, &
& 1.1670471e+03,1.0094418e+03,9.0011540e+02,8.1654950e+02,6.2495060e+02, &
& 6.9953820e+02,5.2793870e+02,5.7701070e+02,5.3341340e+02,4.4587010e+02, &
& 4.6819000e+02,4.4041700e+01,2.7958600e+01,8.2896730e+02,4.3833960e+02, &
& 2.8117370e+02,1.8333680e+02,1.2497980e+02,9.2996800e+01,6.9455800e+01, &
& 5.2851700e+01,9.8609990e+02,7.0935650e+02,6.3758320e+02,4.8553920e+02, &
& 3.6802750e+02,2.9998530e+02,2.4091040e+02,1.9426460e+02,1.6388153e+03, &
& 1.2812612e+03,1.0458401e+03,9.9909710e+02,9.0897530e+02,7.1527990e+02]
 vdw_dftd3_c6(1201:1300)=[&
& 7.7527360e+02,6.0709640e+02,6.3472630e+02,6.5825260e+02,5.0438020e+02, &
& 5.0724520e+02,6.0471830e+02,5.1935010e+02,4.3261670e+02,3.8259680e+02, &
& 3.2992500e+02,2.8312690e+02,1.8285696e+03,1.5296862e+03,1.3118922e+03, &
& 1.1635963e+03,1.0518917e+03,7.9999940e+02,8.9764510e+02,6.7265160e+02, &
& 7.3525250e+02,6.7809960e+02,5.6733030e+02,5.9335740e+02,7.5773970e+02, &
& 4.1533400e+01,2.6786200e+01,7.0845970e+02,3.9379910e+02,2.5871460e+02, &
& 1.7143670e+02,1.1815090e+02,8.8540800e+01,6.6499600e+01,5.0811000e+01, &
& 8.4513480e+02,6.3194340e+02,5.7540180e+02,4.4576230e+02,3.4241690e+02, &
& 2.8132550e+02,2.2756350e+02,1.8458450e+02,1.3902912e+03,1.1230616e+03, &
& 9.2301820e+02,8.8663180e+02,8.0941340e+02,6.3665610e+02,6.9374420e+02, &
& 5.4336290e+02,5.7281690e+02,5.9204640e+02,4.5320890e+02,4.6119710e+02, &
& 5.4805490e+02,4.7710340e+02,4.0194150e+02,3.5778710e+02,3.1050930e+02, &
& 2.6795320e+02,1.5543698e+03,1.3385665e+03,1.1613674e+03,1.0372812e+03, &
& 9.4177920e+02,7.2129060e+02,8.0723580e+02,6.0961520e+02,6.6689540e+02, &
& 6.1675220e+02,5.1444770e+02,5.4154900e+02,6.8561380e+02,6.2756770e+02, &
& 3.7768100e+01,2.4858300e+01,5.8668880e+02,3.4030300e+02,2.2904210e+02, &
& 1.5448510e+02,1.0784360e+02,8.1535800e+01,6.1691200e+01,4.7407300e+01, &
& 7.0169030e+02,5.4198260e+02,4.9972920e+02,3.9368990e+02,3.0669120e+02, &
& 2.5424180e+02,2.0740430e+02,1.6945330e+02,1.1476700e+03,9.5065430e+02, &
& 7.8583230e+02,7.5894060e+02,6.9506830e+02,5.4719810e+02,5.9851680e+02]
 vdw_dftd3_c6(1301:1400)=[&
& 4.6945690e+02,4.9804720e+02,5.1305850e+02,3.9302490e+02,4.0386210e+02, &
& 4.7839930e+02,4.2186160e+02,3.5955640e+02,3.2236080e+02,2.8180500e+02, &
& 2.4479020e+02,1.2861750e+03,1.1321713e+03,9.9240010e+02,8.9208770e+02, &
& 8.1343450e+02,6.2768490e+02,7.0051630e+02,5.3338040e+02,5.8333460e+02, &
& 5.4091250e+02,4.5064270e+02,4.7657830e+02,5.9828910e+02,5.5335920e+02, &
& 4.9293790e+02,3.5484100e+01,2.3691100e+01,5.2125110e+02,3.0954390e+02, &
& 2.1148980e+02,1.4430320e+02,1.0161240e+02,7.7297000e+01,5.8788100e+01, &
& 4.5361300e+01,6.2440760e+02,4.9087600e+02,4.5604000e+02,3.6295180e+02, &
& 2.8527790e+02,2.3790180e+02,1.9518610e+02,1.6026740e+02,1.0193121e+03, &
& 8.5518820e+02,7.0913030e+02,6.8712350e+02,6.3049010e+02,4.9691430e+02, &
& 5.4441560e+02,4.2764390e+02,4.5508360e+02,4.6784930e+02,3.5881670e+02, &
& 3.7059360e+02,4.3811020e+02,3.8929970e+02,3.3422340e+02,3.0105450e+02, &
& 2.6445160e+02,2.3074260e+02,1.1441913e+03,1.0183413e+03,8.9780430e+02, &
& 8.1010280e+02,7.4062110e+02,5.7429320e+02,6.3974650e+02,4.8969560e+02, &
& 5.3521890e+02,4.9709590e+02,4.1415120e+02,4.3888610e+02,5.4812670e+02, &
& 5.0996980e+02,4.5710570e+02,4.2553550e+02,3.2517100e+01,2.2081300e+01, &
& 4.4985120e+02,2.7373980e+02,1.9010940e+02,1.3140510e+02,9.3456200e+01, &
& 7.1605900e+01,5.4799000e+01,4.2495400e+01,5.3985360e+02,4.3217920e+02, &
& 4.0476310e+02,3.2569120e+02,2.5851210e+02,2.1703370e+02,1.7923060e+02, &
& 1.4802160e+02,8.8017410e+02,7.4793060e+02,6.2218410e+02,6.0505080e+02]
 vdw_dftd3_c6(1401:1500)=[&
& 5.5631660e+02,4.3916600e+02,4.8180770e+02,3.7922120e+02,4.0469290e+02, &
& 4.1513030e+02,3.1896420e+02,3.3106510e+02,3.9045420e+02,3.4977600e+02, &
& 3.0267430e+02,2.7406300e+02,2.4205560e+02,2.1228200e+02,9.8989210e+02, &
& 8.9067230e+02,7.9002570e+02,7.1573180e+02,6.5622760e+02,5.1167720e+02, &
& 5.6879090e+02,4.3799920e+02,4.7824810e+02,4.4496620e+02,3.7093440e+02, &
& 3.9376550e+02,4.8897090e+02,4.5771560e+02,4.1298520e+02,3.8609970e+02, &
& 3.5196670e+02,2.9605500e+01,2.0459400e+01,3.8699690e+02,2.4076890e+02, &
& 1.6985780e+02,1.1891780e+02,8.5429300e+01,6.5940300e+01,5.0790000e+01, &
& 3.9595100e+01,4.6528170e+02,3.7861340e+02,3.5729600e+02,2.9049850e+02, &
& 2.3277860e+02,1.9673740e+02,1.6353680e+02,1.3585920e+02,7.5832530e+02, &
& 6.5153920e+02,5.4356310e+02,5.3043120e+02,4.8865220e+02,3.8650670e+02, &
& 4.2441250e+02,3.3482590e+02,3.5809540e+02,3.6655970e+02,2.8228880e+02, &
& 2.9421380e+02,3.4613450e+02,3.1242650e+02,2.7241100e+02,2.4792750e+02, &
& 2.2015810e+02,1.9406990e+02,8.5452230e+02,7.7608590e+02,6.9223860e+02, &
& 6.2952300e+02,5.7878760e+02,4.5379980e+02,5.0338110e+02,3.8995320e+02, &
& 4.2526840e+02,3.9634680e+02,3.3076460e+02,3.5152850e+02,4.3409320e+02, &
& 4.0859520e+02,3.7095720e+02,3.4822390e+02,3.1889050e+02,2.9022230e+02, &
& 1.0500960e+02,6.1318400e+01,3.2422404e+03,1.3162189e+03,7.5492430e+02, &
& 4.5569600e+02,2.9414830e+02,2.1086720e+02,1.5256720e+02,1.1317640e+02, &
& 3.7989439e+03,2.2209652e+03,1.8817164e+03,1.3170056e+03,9.3630690e+02]
 vdw_dftd3_c6(1501:1600)=[&
& 7.3500610e+02,5.6967010e+02,4.4581710e+02,6.8083900e+03,4.3871669e+03, &
& 3.4508369e+03,3.2158544e+03,2.8775279e+03,2.2830674e+03,2.3963445e+03, &
& 1.8842824e+03,1.8757209e+03,1.9768609e+03,1.5334063e+03,1.4430004e+03, &
& 1.7555105e+03,1.4062206e+03,1.1083145e+03,9.5019750e+02,7.9421840e+02, &
& 6.6295560e+02,7.5604361e+03,5.3051627e+03,4.2896097e+03,3.6748556e+03, &
& 3.2563633e+03,2.3987520e+03,2.7260486e+03,1.9683066e+03,2.1299275e+03, &
& 1.9358392e+03,1.6577829e+03,1.6630447e+03,2.2264867e+03,1.8919207e+03, &
& 1.5665042e+03,1.3945709e+03,1.2075672e+03,1.0433955e+03,9.3307294e+03, &
& 9.9457900e+01,5.9114100e+01,2.5515260e+03,1.1604419e+03,6.9068520e+02, &
& 4.2636670e+02,2.7907420e+02,2.0174110e+02,1.4691740e+02,1.0950000e+02, &
& 3.0097233e+03,1.9263504e+03,1.6657662e+03,1.2012021e+03,8.7116470e+02, &
& 6.9086510e+02,5.4031600e+02,4.2579090e+02,5.1712978e+03,3.6591655e+03, &
& 2.9244395e+03,2.7489797e+03,2.4751427e+03,1.9521126e+03,2.0794950e+03, &
& 1.6284040e+03,1.6569384e+03,1.7363987e+03,1.3357319e+03,1.2926630e+03, &
& 1.5597012e+03,1.2819152e+03,1.0285424e+03,8.8961470e+02,7.4984750e+02, &
& 6.3024360e+02,5.7435673e+03,4.3943607e+03,3.6394735e+03,3.1601776e+03, &
& 2.8195147e+03,2.0984813e+03,2.3739131e+03,1.7354827e+03,1.8901371e+03, &
& 1.7272738e+03,1.4605858e+03,1.4937369e+03,1.9640913e+03,1.7108832e+03, &
& 1.4425135e+03,1.2956354e+03,1.1317221e+03,9.8512660e+02,7.0164917e+03, &
& 5.7269887e+03,8.9101800e+01,5.3798200e+01,2.0850661e+03,9.9403160e+02]
 vdw_dftd3_c6(1601:1700)=[&
& 6.0449150e+02,3.7879280e+02,2.5055770e+02,1.8240750e+02,1.3361230e+02, &
& 1.0002770e+02,2.4655965e+03,1.6371586e+03,1.4317496e+03,1.0491934e+03, &
& 7.7049240e+02,6.1555590e+02,4.8476910e+02,3.8423060e+02,4.1858968e+03, &
& 3.0598941e+03,2.4617619e+03,2.3247454e+03,2.0995853e+03,1.6534435e+03, &
& 1.7717653e+03,1.3863132e+03,1.4234023e+03,1.4871454e+03,1.1414078e+03, &
& 1.1183973e+03,1.3447968e+03,1.1200021e+03,9.0836700e+02,7.9048550e+02, &
& 6.7038430e+02,5.6651010e+02,4.6537454e+03,3.6666556e+03,3.0693037e+03, &
& 2.6820879e+03,2.4019326e+03,1.7981970e+03,2.0296163e+03,1.4938895e+03, &
& 1.6299126e+03,1.4934855e+03,1.2574297e+03,1.2959364e+03,1.6894085e+03, &
& 1.4890440e+03,1.2682622e+03,1.1455595e+03,1.0065104e+03,8.8082020e+02, &
& 5.6757434e+03,4.7544826e+03,3.9906172e+03,4.4510400e+01,2.9308100e+01, &
& 7.0433450e+02,4.0374270e+02,2.7061870e+02,1.8221640e+02,1.2713550e+02, &
& 9.6118900e+01,7.2735900e+01,5.5904800e+01,8.4189860e+02,6.4422200e+02, &
& 5.9240470e+02,4.6526310e+02,3.6180150e+02,2.9974470e+02,2.4444680e+02, &
& 1.9971380e+02,1.3810183e+03,1.1346748e+03,9.3640680e+02,9.0333060e+02, &
& 8.2670470e+02,6.5114400e+02,7.1114230e+02,5.5801770e+02,5.9073870e+02, &
& 6.0892420e+02,4.6681100e+02,4.7833500e+02,5.6687830e+02,4.9862840e+02, &
& 4.2429060e+02,3.8016060e+02,3.3218840e+02,2.8850100e+02,1.5470780e+03, &
& 1.3520803e+03,1.1820124e+03,1.0609721e+03,9.6662010e+02,7.4511760e+02, &
& 8.3183600e+02,6.3266390e+02,6.9156460e+02,6.4096110e+02,5.3459930e+02]
 vdw_dftd3_c6(1701:1800)=[&
& 5.6439100e+02,7.0950090e+02,6.5467230e+02,5.8216180e+02,5.3941670e+02, &
& 4.8703330e+02,4.3728340e+02,1.8844554e+03,1.7247415e+03,1.5122504e+03, &
& 6.8803530e+02,8.8687700e+01,5.3152300e+01,2.2103168e+03,1.0189540e+03, &
& 6.1076150e+02,3.7910380e+02,2.4926330e+02,1.8083830e+02,1.3215840e+02, &
& 9.8822000e+01,2.6090440e+03,1.6875419e+03,1.4644446e+03,1.0615213e+03, &
& 7.7317210e+02,6.1486520e+02,4.8225090e+02,3.8105030e+02,4.4737784e+03, &
& 3.1902086e+03,2.5547353e+03,2.4050561e+03,2.1676325e+03,1.7091137e+03, &
& 1.8237858e+03,1.4281039e+03,1.4570358e+03,1.5254141e+03,1.1729549e+03, &
& 1.1394352e+03,1.3728649e+03,1.1331627e+03,9.1252630e+02,7.9102140e+02, &
& 6.6833280e+02,5.6301560e+02,4.9710778e+03,3.8289626e+03,3.1814405e+03, &
& 2.7679894e+03,2.4726905e+03,1.8440715e+03,2.0845975e+03,1.5276056e+03, &
& 1.6644640e+03,1.5224806e+03,1.2860441e+03,1.3182791e+03,1.7282162e+03, &
& 1.5110459e+03,1.2783406e+03,1.1504200e+03,1.0070198e+03,8.7837980e+02, &
& 6.0738935e+03,4.9833126e+03,4.1503203e+03,1.5271461e+03,4.3422386e+03, &
& 8.5408000e+01,5.1295200e+01,2.0915068e+03,9.7461670e+02,5.8618960e+02, &
& 3.6465190e+02,2.4011390e+02,1.7436780e+02,1.2753080e+02,9.5420200e+01, &
& 2.4706020e+03,1.6117831e+03,1.4013191e+03,1.0185098e+03,7.4324070e+02, &
& 5.9167520e+02,4.6450620e+02,3.6731630e+02,4.2163998e+03,3.0363662e+03, &
& 2.4350889e+03,2.2942857e+03,2.0689809e+03,1.6306018e+03,1.7421942e+03, &
& 1.3638456e+03,1.3940822e+03,1.4587399e+03,1.1209931e+03,1.0915722e+03]
 vdw_dftd3_c6(1801:1900)=[&
& 1.3142306e+03,1.0872533e+03,8.7700660e+02,7.6089490e+02,6.4343050e+02, &
& 5.4243830e+02,4.6850846e+03,3.6420492e+03,3.0329183e+03,2.6420526e+03, &
& 2.3616974e+03,1.7631139e+03,1.9921534e+03,1.4616458e+03,1.5933696e+03, &
& 1.4581671e+03,1.2304969e+03,1.2633482e+03,1.6535039e+03,1.4488711e+03, &
& 1.2277455e+03,1.1058143e+03,9.6879470e+02,8.4567030e+02,5.7167999e+03, &
& 4.7334185e+03,3.9519956e+03,1.4659695e+03,4.1259023e+03,3.9244211e+03, &
& 8.3331000e+01,5.0074800e+01,2.0312740e+03,9.4912550e+02,5.7141130e+02, &
& 3.5567120e+02,2.3429130e+02,1.7018170e+02,1.2449460e+02,9.3163500e+01, &
& 2.3998207e+03,1.5689885e+03,1.3648494e+03,9.9275070e+02,7.2481970e+02, &
& 5.7716960e+02,4.5323130e+02,3.5847210e+02,4.0920575e+03,2.9529794e+03, &
& 2.3691244e+03,2.2326503e+03,2.0137194e+03,1.5868512e+03,1.6960441e+03, &
& 1.3276064e+03,1.3577395e+03,1.4205045e+03,1.0914214e+03,1.0634898e+03, &
& 1.2801762e+03,1.0597536e+03,8.5521770e+02,7.4216490e+02,6.2773420e+02, &
& 5.2930700e+02,4.5470720e+03,3.5414930e+03,2.9509303e+03,2.5714964e+03, &
& 2.2990481e+03,1.7168038e+03,1.9396215e+03,1.4235531e+03,1.5520522e+03, &
& 1.4205469e+03,1.1984194e+03,1.2309598e+03,1.6104142e+03,1.4119588e+03, &
& 1.1970157e+03,1.0783854e+03,9.4498140e+02,8.2504730e+02,5.5474900e+03, &
& 4.6012171e+03,3.8440367e+03,1.4290690e+03,4.0113635e+03,3.8161118e+03, &
& 3.7109375e+03,8.1412300e+01,4.8940600e+01,1.9766955e+03,9.2581580e+02, &
& 5.5783800e+02,3.4739210e+02,2.2890650e+02,1.6630040e+02,1.2167270e+02]
 vdw_dftd3_c6(1901:2000)=[&
& 9.1061400e+01,2.3356474e+03,1.5299120e+03,1.3314786e+03,9.6910260e+02, &
& 7.0786040e+02,5.6378900e+02,4.4280950e+02,3.5028120e+02,3.9796309e+03, &
& 2.8770680e+03,2.3089962e+03,2.1764184e+03,1.9632723e+03,1.5469195e+03, &
& 1.6538773e+03,1.2944951e+03,1.3244770e+03,1.3855313e+03,1.0643798e+03, &
& 1.0377490e+03,1.2489915e+03,1.0345024e+03,8.3516220e+02,7.2489870e+02, &
& 6.1324050e+02,5.1716170e+02,4.4222696e+03,3.4499906e+03,2.8761716e+03, &
& 2.5070805e+03,2.2418085e+03,1.6744360e+03,1.8915883e+03,1.3886668e+03, &
& 1.5142016e+03,1.3860622e+03,1.1690360e+03,1.2012473e+03,1.5709643e+03, &
& 1.3780830e+03,1.1687524e+03,1.0531255e+03,9.2301940e+02,8.0600020e+02, &
& 5.3944846e+03,4.4810449e+03,3.7456921e+03,1.3951471e+03,3.9071923e+03, &
& 3.7175407e+03,3.6152067e+03,3.5220508e+03,7.9713800e+01,4.7936500e+01, &
& 1.9280810e+03,9.0511010e+02,5.4580150e+02,3.4005800e+02,2.2413830e+02, &
& 1.6286380e+02,1.1917340e+02,8.9198800e+01,2.2784898e+03,1.4951792e+03, &
& 1.3018438e+03,9.4812880e+02,6.9283380e+02,5.5193900e+02,4.3358310e+02, &
& 3.4303070e+02,3.8794729e+03,2.8095233e+03,2.2555157e+03,2.1264190e+03, &
& 1.9184254e+03,1.5114156e+03,1.6164023e+03,1.2650643e+03,1.2949313e+03, &
& 1.3544595e+03,1.0403490e+03,1.0148959e+03,1.2213030e+03,1.0121053e+03, &
& 8.1738880e+02,7.0960390e+02,6.0040640e+02,5.0640980e+02,4.3110961e+03, &
& 3.3685646e+03,2.8096885e+03,2.4498188e+03,2.1909387e+03,1.6367956e+03, &
& 1.8489097e+03,1.3576810e+03,1.4805878e+03,1.3554425e+03,1.1429352e+03]
 vdw_dftd3_c6(2001:2100)=[&
& 1.1748699e+03,1.5359254e+03,1.3480216e+03,1.1436919e+03,1.0307380e+03, &
& 9.0356300e+02,7.8913200e+02,5.2582068e+03,4.3740826e+03,3.6582120e+03, &
& 1.3650615e+03,3.8144921e+03,3.6298268e+03,3.5300228e+03,3.4391601e+03, &
& 3.3583122e+03,6.4188000e+01,3.9578600e+01,1.4077335e+03,6.8992870e+02, &
& 4.2648970e+02,2.7091830e+02,1.8125810e+02,1.3314380e+02,9.8357300e+01, &
& 7.4191200e+01,1.6673895e+03,1.1308198e+03,9.9679240e+02,7.3902000e+02, &
& 5.4836330e+02,4.4123030e+02,3.5003690e+02,2.7935150e+02,2.8176333e+03, &
& 2.0945326e+03,1.6917052e+03,1.6028927e+03,1.4507174e+03,1.1427015e+03, &
& 1.2280426e+03,9.6156000e+02,9.9206730e+02,1.0341932e+03,7.9387630e+02, &
& 7.8351140e+02,9.3936520e+02,7.8963690e+02,6.4598490e+02,5.6529910e+02, &
& 4.8231720e+02,4.0998030e+02,3.1359242e+03,2.5077589e+03,2.1132076e+03, &
& 1.8544837e+03,1.6654372e+03,1.2530730e+03,1.4116759e+03,1.0450617e+03, &
& 1.1404525e+03,1.0471279e+03,8.8053130e+02,9.1107960e+02,1.1799720e+03, &
& 1.0480976e+03,8.9951620e+02,8.1630470e+02,7.2098470e+02,6.3421630e+02, &
& 3.8238426e+03,3.2436561e+03,2.7405478e+03,1.0709597e+03,2.8381357e+03, &
& 2.7056826e+03,2.6326657e+03,2.5660607e+03,2.5068352e+03,1.8916719e+03, &
& 7.0179400e+01,4.2367900e+01,1.8023852e+03,8.0848660e+02,4.8279770e+02, &
& 2.9984690e+02,1.9762340e+02,1.4377540e+02,1.0539700e+02,7.9048400e+01, &
& 2.1240831e+03,1.3433651e+03,1.1621560e+03,8.3917760e+02,6.1083180e+02, &
& 4.8623190e+02,3.8195840e+02,3.0241490e+02,3.6876305e+03,2.5641443e+03]
 vdw_dftd3_c6(2101:2200)=[&
& 2.0454927e+03,1.9228302e+03,1.7309890e+03,1.3680049e+03,1.4541614e+03, &
& 1.1408713e+03,1.1579011e+03,1.2132014e+03,9.3575440e+02,9.0359910e+02, &
& 1.0900939e+03,8.9646510e+02,7.2120420e+02,6.2544800e+02,5.2893810e+02, &
& 4.4620840e+02,4.0985678e+03,3.0839899e+03,2.5485911e+03,2.2114907e+03, &
& 1.9735648e+03,1.4705265e+03,1.6631347e+03,1.2173511e+03,1.3235223e+03, &
& 1.2096847e+03,1.0256835e+03,1.0465789e+03,1.3756067e+03,1.1975597e+03, &
& 1.0112975e+03,9.0991270e+02,7.9669950e+02,6.9543500e+02,5.0226967e+03, &
& 4.0300910e+03,3.3357435e+03,1.2096287e+03,3.5071716e+03,3.3250797e+03, &
& 3.2313241e+03,3.1461740e+03,3.0703853e+03,2.2762532e+03,2.8516677e+03, &
& 6.7958000e+01,4.1162200e+01,1.7207353e+03,7.7577180e+02,4.6524390e+02, &
& 2.8984550e+02,1.9145500e+02,1.3949370e+02,1.0238110e+02,7.6855700e+01, &
& 2.0278568e+03,1.2872471e+03,1.1160341e+03,8.0833220e+02,5.8989410e+02, &
& 4.7029770e+02,3.6998330e+02,2.9329050e+02,3.5269705e+03,2.4517734e+03, &
& 1.9574581e+03,1.8415883e+03,1.6587019e+03,1.3107538e+03,1.3944853e+03, &
& 1.0940016e+03,1.1118174e+03,1.1642755e+03,8.9783730e+02,8.6875770e+02, &
& 1.0475640e+03,8.6358600e+02,6.9627600e+02,6.0460480e+02,5.1197160e+02, &
& 4.3238930e+02,3.9216962e+03,2.9485688e+03,2.4401762e+03,2.1194908e+03, &
& 1.8927858e+03,1.4117363e+03,1.5962074e+03,1.1696785e+03,1.2718937e+03, &
& 1.1630374e+03,9.8556970e+02,1.0068382e+03,1.3215257e+03,1.1527975e+03, &
& 9.7544350e+02,8.7867530e+02,7.7029080e+02,6.7313940e+02,4.8123323e+03]
 vdw_dftd3_c6(2201:2300)=[&
& 3.8522949e+03,3.1924039e+03,1.1661633e+03,3.3558765e+03,3.1799629e+03, &
& 3.0904072e+03,3.0090778e+03,2.9367065e+03,2.1810675e+03,2.7299739e+03, &
& 2.6173310e+03,7.2220300e+01,4.3571100e+01,1.7049679e+03,8.1136880e+02, &
& 4.9188750e+02,3.0751980e+02,2.0315540e+02,1.4783550e+02,1.0830760e+02, &
& 8.1141600e+01,2.0163392e+03,1.3374783e+03,1.1679358e+03,8.5407400e+02, &
& 6.2593940e+02,4.9945760e+02,3.9293850e+02,3.1124870e+02,3.4195160e+03, &
& 2.5014164e+03,2.0120352e+03,1.8992044e+03,1.7148635e+03,1.3502932e+03, &
& 1.4466020e+03,1.1317710e+03,1.1614905e+03,1.2139348e+03,9.3167930e+02, &
& 9.1200820e+02,1.0964450e+03,9.1172230e+02,7.3821290e+02,6.4174520e+02, &
& 5.4371620e+02,4.5912050e+02,3.8008304e+03,2.9970212e+03,2.5073572e+03, &
& 2.1900272e+03,1.9605194e+03,1.4668017e+03,1.6559466e+03,1.2180651e+03, &
& 1.3291426e+03,1.2176582e+03,1.0253902e+03,1.0563718e+03,1.3778901e+03, &
& 1.2131043e+03,1.0318071e+03,9.3112390e+02,8.1732200e+02,7.1464770e+02, &
& 4.6328428e+03,3.8855093e+03,3.2599132e+03,1.2305989e+03,3.3915411e+03, &
& 3.2298168e+03,3.1416098e+03,3.0612537e+03,2.9897660e+03,2.2379026e+03, &
& 2.7243602e+03,2.6065136e+03,2.6641668e+03,7.0715400e+01,4.2665800e+01, &
& 1.6647700e+03,7.9368770e+02,4.8143850e+02,3.0106920e+02,1.9891850e+02, &
& 1.4475870e+02,1.0605520e+02,7.9453900e+01,1.9689928e+03,1.3079814e+03, &
& 1.1425656e+03,8.3589690e+02,6.1278090e+02,4.8901070e+02,3.8474960e+02, &
& 3.0477480e+02,3.3372227e+03,2.4446758e+03,1.9669050e+03,1.8568743e+03]
 vdw_dftd3_c6(2301:2400)=[&
& 1.6768129e+03,1.3201912e+03,1.4147048e+03,1.1067246e+03,1.1361897e+03, &
& 1.1873873e+03,9.1117070e+02,8.9233320e+02,1.0726768e+03,8.9230170e+02, &
& 7.2266540e+02,6.2829340e+02,5.3236410e+02,4.4955920e+02,3.7094297e+03, &
& 2.9287164e+03,2.4511775e+03,2.1414202e+03,1.9172203e+03,1.4346130e+03, &
& 1.6195186e+03,1.1914716e+03,1.3002637e+03,1.1912962e+03,1.0029797e+03, &
& 1.0336009e+03,1.3478497e+03,1.1871040e+03,1.0099583e+03,9.1151370e+02, &
& 8.0019180e+02,6.9972310e+02,4.5209645e+03,3.7960924e+03,3.1862327e+03, &
& 1.2044185e+03,3.3138879e+03,3.1561992e+03,3.0700842e+03,2.9916275e+03, &
& 2.9218310e+03,2.1877803e+03,2.6611461e+03,2.5461249e+03,2.6039653e+03, &
& 2.5451713e+03,6.9379100e+01,4.1872700e+01,1.6270951e+03,7.7746020e+02, &
& 4.7198600e+02,2.9530140e+02,1.9516300e+02,1.4204780e+02,1.0408000e+02, &
& 7.7978800e+01,1.9246593e+03,1.2807911e+03,1.1193391e+03,8.1942800e+02, &
& 6.0097010e+02,4.7969090e+02,3.7748730e+02,2.9906220e+02,3.2599368e+03, &
& 2.3919954e+03,1.9251263e+03,1.8177881e+03,1.6417354e+03,1.2924384e+03, &
& 1.3853712e+03,1.0836957e+03,1.1130259e+03,1.1630369e+03,8.9235160e+02, &
& 8.7439610e+02,1.0509705e+03,8.7471320e+02,7.0869500e+02,6.1626580e+02, &
& 5.2226550e+02,4.4109320e+02,3.6236511e+03,2.8652613e+03,2.3992487e+03, &
& 2.0966400e+03,1.8774174e+03,1.4051352e+03,1.5861100e+03,1.1671878e+03, &
& 1.2739026e+03,1.1672707e+03,9.8252100e+02,1.0128904e+03,1.3203887e+03, &
& 1.1634988e+03,9.9025880e+02,8.9390800e+02,7.8488410e+02,6.8644520e+02]
 vdw_dftd3_c6(2401:2500)=[&
& 4.4159694e+03,3.7128812e+03,3.1180001e+03,1.1807776e+03,3.2417266e+03, &
& 3.0878489e+03,3.0036922e+03,2.9270118e+03,2.8587977e+03,2.1415264e+03, &
& 2.6023122e+03,2.4899643e+03,2.5481802e+03,2.4906989e+03,2.4374539e+03, &
& 6.8494900e+01,4.1286600e+01,1.6127581e+03,7.6939830e+02,4.6660990e+02, &
& 2.9168170e+02,1.9263240e+02,1.4013000e+02,1.0262510e+02,7.6858100e+01, &
& 1.9075332e+03,1.2678866e+03,1.1075215e+03,8.1018450e+02,5.9379110e+02, &
& 4.7373900e+02,3.7262670e+02,2.9508360e+02,3.2317117e+03,2.3690958e+03, &
& 1.9062802e+03,1.7996377e+03,1.6251430e+03,1.2793493e+03,1.3711198e+03, &
& 1.0724959e+03,1.1012182e+03,1.1508553e+03,8.8299190e+02,8.6485460e+02, &
& 1.0396601e+03,8.6479960e+02,7.0027180e+02,6.0871740e+02,5.1566490e+02, &
& 4.3535350e+02,3.5920465e+03,2.8379423e+03,2.3754679e+03,2.0753417e+03, &
& 1.8580353e+03,1.3902059e+03,1.5694426e+03,1.1545209e+03,1.2600764e+03, &
& 1.1544639e+03,9.7180930e+02,1.0016214e+03,1.3062023e+03,1.1504571e+03, &
& 9.7868710e+02,8.8319190e+02,7.7520910e+02,6.7775440e+02,4.3774766e+03, &
& 3.6779606e+03,3.0875154e+03,1.1670799e+03,3.2108489e+03,3.0582277e+03, &
& 2.9748238e+03,2.8988349e+03,2.8312344e+03,2.1199794e+03,2.5778005e+03, &
& 2.4663583e+03,2.5233630e+03,2.4664143e+03,2.4136514e+03,2.3901227e+03, &
& 5.8651900e+01,3.5951600e+01,1.2947583e+03,6.3458810e+02,3.9146520e+02, &
& 2.4795570e+02,1.6541840e+02,1.2121480e+02,8.9340500e+01,6.7257500e+01, &
& 1.5332675e+03,1.0401825e+03,9.1638240e+02,6.7855410e+02,5.0255170e+02]
 vdw_dftd3_c6(2501:2600)=[&
& 4.0367180e+02,3.1962860e+02,2.5459790e+02,2.5921299e+03,1.9255453e+03, &
& 1.5553939e+03,1.4734091e+03,1.3334027e+03,1.0496142e+03,1.1285309e+03, &
& 8.8302170e+02,9.1141500e+02,9.5032780e+02,7.2889260e+02,7.1952410e+02, &
& 8.6291390e+02,7.2475130e+02,5.9205870e+02,5.1745760e+02,4.4084280e+02, &
& 3.7414020e+02,2.8847585e+03,2.3048558e+03,1.9421387e+03,1.7040440e+03, &
& 1.5299721e+03,1.1501604e+03,1.2962701e+03,9.5869920e+02,1.0467597e+03, &
& 9.6091280e+02,8.0744060e+02,8.3583100e+02,1.0833008e+03,9.6184020e+02, &
& 8.2469640e+02,7.4775790e+02,6.5970490e+02,5.7959110e+02,3.5195236e+03, &
& 2.9806436e+03,2.5182192e+03,9.8175310e+02,2.6089193e+03,2.4861931e+03, &
& 2.4191164e+03,2.3579439e+03,2.3035540e+03,1.7376565e+03,2.0919380e+03, &
& 2.0059243e+03,2.0565310e+03,2.0105203e+03,1.9680474e+03,1.9483102e+03, &
& 1.5974796e+03,5.8850400e+01,3.6574900e+01,1.1772268e+03,6.0758590e+02, &
& 3.8350840e+02,2.4668260e+02,1.6624720e+02,1.2259130e+02,9.0778400e+01, &
& 6.8550500e+01,1.3981254e+03,9.8752390e+02,8.8079020e+02,6.6333140e+02, &
& 4.9770150e+02,4.0279470e+02,3.2111450e+02,2.5716510e+02,2.3300562e+03, &
& 1.7971024e+03,1.4619589e+03,1.3919848e+03,1.2638660e+03,9.9362820e+02, &
& 1.0747287e+03,8.4045370e+02,8.7543360e+02,9.0984070e+02,6.9642830e+02, &
& 6.9621350e+02,8.3232490e+02,7.0869710e+02,5.8542500e+02,5.1487840e+02, &
& 4.4133770e+02,3.7652870e+02,2.5964033e+03,2.1464040e+03,1.8297515e+03, &
& 1.6164737e+03,1.4572492e+03,1.1025669e+03,1.2395201e+03,9.2341980e+02]
 vdw_dftd3_c6(2601:2700)=[&
& 1.0097527e+03,9.2946760e+02,7.7779950e+02,8.1120400e+02,1.0422993e+03, &
& 9.3663610e+02,8.1148690e+02,7.4006690e+02,6.5681710e+02,5.8013770e+02, &
& 3.1606446e+03,2.7609094e+03,2.3609186e+03,9.6347730e+02,2.4240491e+03, &
& 2.3170514e+03,2.2560584e+03,2.2002906e+03,2.1507417e+03,1.6406567e+03, &
& 1.9309949e+03,1.8529581e+03,1.9265925e+03,1.8843048e+03,1.8455050e+03, &
& 1.8262149e+03,1.5072441e+03,1.4412394e+03,5.4401500e+01,3.4283400e+01, &
& 1.0273577e+03,5.4443550e+02,3.4862850e+02,2.2671590e+02,1.5406960e+02, &
& 1.1429670e+02,8.5081400e+01,6.4523800e+01,1.2220335e+03,8.8084690e+02, &
& 7.9141350e+02,6.0213470e+02,4.5569700e+02,3.7087230e+02,2.9728390e+02, &
& 2.3922910e+02,2.0266698e+03,1.5894883e+03,1.2977826e+03,1.2395033e+03, &
& 1.1275812e+03,8.8658800e+02,9.6151940e+02,7.5231040e+02,7.8703270e+02, &
& 8.1634550e+02,6.2484080e+02,6.2867410e+02,7.4997760e+02,6.4373010e+02, &
& 5.3560150e+02,4.7316850e+02,4.0746910e+02,3.4913050e+02,2.2608400e+03, &
& 1.8969799e+03,1.6272670e+03,1.4431973e+03,1.3043213e+03,9.9116570e+02, &
& 1.1124511e+03,8.3281910e+02,9.1084260e+02,8.3983510e+02,7.0192840e+02, &
& 7.3456590e+02,9.3877020e+02,8.4925790e+02,7.4051010e+02,6.7794300e+02, &
& 6.0419180e+02,5.3577730e+02,2.7516509e+03,2.4344207e+03,2.0948531e+03, &
& 8.7810980e+02,2.1417667e+03,2.0496125e+03,1.9962879e+03,1.9474669e+03, &
& 1.9041085e+03,1.4623131e+03,1.7026705e+03,1.6355752e+03,1.7084912e+03, &
& 1.6713110e+03,1.6373225e+03,1.6197710e+03,1.3428697e+03,1.2925836e+03]
 vdw_dftd3_c6(2701:2800)=[&
& 1.1638241e+03,4.6481300e+01,2.9820600e+01,8.4241230e+02,4.5198610e+02, &
& 2.9290150e+02,1.9259840e+02,1.3214100e+02,9.8766600e+01,7.4030900e+01, &
& 5.6476700e+01,1.0029508e+03,7.2933520e+02,6.5874710e+02,5.0518900e+02, &
& 3.8541090e+02,3.1556320e+02,2.5452870e+02,2.0602860e+02,1.6644321e+03, &
& 1.3117641e+03,1.0728050e+03,1.0269214e+03,9.3538590e+02,7.3650630e+02, &
& 7.9917580e+02,6.2633800e+02,6.5622600e+02,6.7963140e+02,5.2108220e+02, &
& 5.2588940e+02,6.2622030e+02,5.4069840e+02,4.5278370e+02,4.0184290e+02, &
& 3.4779630e+02,2.9948800e+02,1.8587683e+03,1.5659413e+03,1.3478163e+03, &
& 1.1983202e+03,1.0850878e+03,8.2786250e+02,9.2774070e+02,6.9763460e+02, &
& 7.6226600e+02,7.0376190e+02,5.8870290e+02,6.1664100e+02,7.8470960e+02, &
& 7.1294940e+02,6.2490100e+02,5.7415840e+02,5.1383150e+02,4.5758530e+02, &
& 2.2639013e+03,2.0087751e+03,1.7339129e+03,7.4068630e+02,1.7694139e+03, &
& 1.6938988e+03,1.6500004e+03,1.6097798e+03,1.5740661e+03,1.2144448e+03, &
& 1.4078390e+03,1.3533433e+03,1.4132821e+03,1.3825761e+03,1.3545779e+03, &
& 1.3397711e+03,1.1143056e+03,1.0763807e+03,9.7195240e+02,8.1436220e+02, &
& 4.7481400e+01,3.0549200e+01,8.4030290e+02,4.5686870e+02,2.9769360e+02, &
& 1.9640730e+02,1.3504060e+02,1.0106690e+02,7.5833700e+01,5.7897000e+01, &
& 1.0011833e+03,7.3565530e+02,6.6653080e+02,5.1321450e+02,3.9266040e+02, &
& 3.2200220e+02,2.6008050e+02,2.1074970e+02,1.6563829e+03,1.3172961e+03, &
& 1.0793168e+03,1.0345489e+03,9.4314230e+02,7.4238390e+02,8.0678450e+02]
 vdw_dftd3_c6(2801:2900)=[&
& 6.3220880e+02,6.6388710e+02,6.8701430e+02,5.2648590e+02,5.3299620e+02, &
& 6.3416620e+02,5.4932810e+02,4.6115310e+02,4.0981080e+02,3.5513780e+02, &
& 3.0613060e+02,1.8506131e+03,1.5716863e+03,1.3568533e+03,1.2084737e+03, &
& 1.0954282e+03,8.3707080e+02,9.3753010e+02,7.0623580e+02,7.7195670e+02, &
& 7.1318460e+02,5.9599800e+02,6.2541990e+02,7.9423010e+02,7.2367260e+02, &
& 6.3581610e+02,5.8492700e+02,5.2412750e+02,4.6726420e+02,2.2535037e+03, &
& 2.0134740e+03,1.7433791e+03,7.5313690e+02,1.7753357e+03,1.7005706e+03, &
& 1.6567768e+03,1.6166266e+03,1.5809835e+03,1.2234224e+03,1.4105904e+03, &
& 1.3567115e+03,1.4207081e+03,1.3899934e+03,1.3620334e+03,1.3469951e+03, &
& 1.1225706e+03,1.0876931e+03,9.8375910e+02,8.2496150e+02,8.3633100e+02, &
& 4.3502700e+01,2.8354600e+01,7.3192340e+02,4.0716460e+02,2.6873040e+02, &
& 1.7905630e+02,1.2405100e+02,9.3359700e+01,7.0395400e+01,5.3964200e+01, &
& 8.7335890e+02,6.5307500e+02,5.9556280e+02,4.6268400e+02,3.5670880e+02, &
& 2.9400210e+02,2.3864150e+02,1.9423140e+02,1.4396620e+03,1.1612618e+03, &
& 9.5445330e+02,9.1746900e+02,8.3784550e+02,6.5982040e+02,7.1850590e+02, &
& 5.6350590e+02,5.9375330e+02,6.1335130e+02,4.7025530e+02,4.7853750e+02, &
& 5.6822510e+02,4.9561460e+02,4.1868920e+02,3.7355900e+02,3.2506940e+02, &
& 2.8130190e+02,1.6103754e+03,1.3848254e+03,1.2021044e+03,1.0743491e+03, &
& 9.7608420e+02,7.4894180e+02,8.3752750e+02,6.3378860e+02,6.9267540e+02, &
& 6.4089680e+02,5.3528690e+02,5.6312050e+02,7.1169730e+02,6.5211350e+02]
 vdw_dftd3_c6(2901:3000)=[&
& 5.7610750e+02,5.3178230e+02,4.7825570e+02,4.2787240e+02,1.9610192e+03, &
& 1.7707984e+03,1.5416747e+03,6.8178680e+02,1.5643082e+03,1.4998732e+03, &
& 1.4616315e+03,1.4265281e+03,1.3953763e+03,1.0864242e+03,1.2413205e+03, &
& 1.1950783e+03,1.2556728e+03,1.2287111e+03,1.2042523e+03,1.1906528e+03, &
& 9.9636440e+02,9.7080130e+02,8.8109380e+02,7.4095590e+02,7.5215710e+02, &
& 6.7852780e+02,4.0192800e+01,2.6493600e+01,6.4922250e+02,3.6768190e+02, &
& 2.4523810e+02,1.6475860e+02,1.1488740e+02,8.6879300e+01,6.5790400e+01, &
& 5.0615900e+01,7.7563370e+02,5.8794870e+02,5.3899920e+02,4.2178390e+02, &
& 3.2723040e+02,2.7085510e+02,2.2077630e+02,1.8037110e+02,1.2755198e+03, &
& 1.0398993e+03,8.5678860e+02,8.2551250e+02,7.5491930e+02,5.9487280e+02, &
& 6.4870850e+02,5.0921810e+02,5.3789460e+02,5.5484790e+02,4.2567880e+02, &
& 4.3487920e+02,5.1550510e+02,4.5212560e+02,3.8392650e+02,3.4368330e+02, &
& 3.0011780e+02,2.6056720e+02,1.4282493e+03,1.2397351e+03,1.0808340e+03, &
& 9.6864880e+02,8.8171010e+02,6.7887180e+02,7.5818340e+02,5.7594520e+02, &
& 6.2929090e+02,5.8296220e+02,4.8681310e+02,5.1303760e+02,6.4586290e+02, &
& 5.9440490e+02,5.2746810e+02,4.8823190e+02,4.4042470e+02,3.9518890e+02, &
& 1.7394992e+03,1.5831333e+03,1.3842225e+03,6.2381140e+02,1.4006813e+03, &
& 1.3439514e+03,1.3099454e+03,1.2786986e+03,1.2509767e+03,9.7887040e+02, &
& 1.1106969e+03,1.0701912e+03,1.1269388e+03,1.1028614e+03,1.0810811e+03, &
& 1.0686460e+03,8.9729150e+02,8.7805750e+02,7.9916860e+02,6.7370750e+02]
 vdw_dftd3_c6(3001:3100)=[&
& 6.8458550e+02,6.1911480e+02,5.6606600e+02,3.3737400e+01,2.2655000e+01, &
& 5.2329220e+02,3.0016240e+02,2.0254080e+02,1.3757620e+02,9.6865800e+01, &
& 7.3818300e+01,5.6305300e+01,4.3593000e+01,6.2596250e+02,4.7885350e+02, &
& 4.4120140e+02,3.4788170e+02,2.7198720e+02,2.2647100e+02,1.8574640e+02, &
& 1.5265670e+02,1.0299899e+03,8.4443870e+02,6.9692720e+02,6.7305270e+02, &
& 6.1630660e+02,4.8652160e+02,5.3065270e+02,4.1743140e+02,4.4140210e+02, &
& 4.5462950e+02,3.4958030e+02,3.5801260e+02,4.2342920e+02,3.7342980e+02, &
& 3.1904090e+02,2.8687030e+02,2.5174820e+02,2.1965520e+02,1.1547638e+03, &
& 1.0070623e+03,8.8099930e+02,7.9155440e+02,7.2193400e+02,5.5823890e+02, &
& 6.2242220e+02,4.7507150e+02,5.1845950e+02,4.8093890e+02,4.0219150e+02, &
& 4.2403900e+02,5.3144420e+02,4.9101090e+02,4.3782090e+02,4.0661850e+02, &
& 3.6825920e+02,3.3179040e+02,1.4071724e+03,1.2854322e+03,1.1275157e+03, &
& 5.1768570e+02,1.1388994e+03,1.0932462e+03,1.0657050e+03,1.0403742e+03, &
& 1.0179032e+03,8.0035340e+02,9.0395940e+02,8.7158450e+02,9.1760410e+02, &
& 8.9802720e+02,8.8036430e+02,8.7003670e+02,7.3291850e+02,7.1957520e+02, &
& 6.5682120e+02,5.5562970e+02,5.6502640e+02,5.1242690e+02,4.6966640e+02, &
& 3.9114480e+02,3.1643100e+01,2.1405700e+01,4.8271200e+02,2.7848740e+02, &
& 1.8878590e+02,1.2878430e+02,9.1021000e+01,6.9577200e+01,5.3225800e+01, &
& 4.1316600e+01,5.7775500e+02,4.4384910e+02,4.0978640e+02,3.2409250e+02, &
& 2.5414960e+02,2.1210390e+02,1.7438080e+02,1.4364860e+02,9.5062180e+02]
 vdw_dftd3_c6(3101:3200)=[&
& 7.8160440e+02,6.4557500e+02,6.2406930e+02,5.7177460e+02,4.5167990e+02, &
& 4.9272680e+02,3.8791960e+02,4.1040680e+02,4.2244780e+02,3.2511980e+02, &
& 3.3331770e+02,3.9382270e+02,3.4809220e+02,2.9810130e+02,2.6850000e+02, &
& 2.3607650e+02,2.0637560e+02,1.0663289e+03,9.3220550e+02,8.1674650e+02, &
& 7.3461520e+02,6.7055170e+02,5.1941000e+02,5.7874610e+02,4.4259210e+02, &
& 4.8280890e+02,4.4812160e+02,3.7494860e+02,3.9541490e+02,4.9465360e+02, &
& 4.5775290e+02,4.0894270e+02,3.8029110e+02,3.4494080e+02,3.1127000e+02, &
& 1.2995975e+03,1.1895405e+03,1.0448996e+03,4.8348310e+02,1.0546339e+03, &
& 1.0125735e+03,9.8712160e+02,9.6370210e+02,9.4292740e+02,7.4291920e+02, &
& 8.3729210e+02,8.0754210e+02,8.5029540e+02,8.3217180e+02,8.1583760e+02, &
& 8.0619030e+02,6.8007310e+02,6.6862000e+02,6.1103860e+02,5.1760830e+02, &
& 5.2653700e+02,4.7806720e+02,4.3860830e+02,3.6582470e+02,3.4235260e+02, &
& 3.2647500e+01,2.2083700e+01,4.8740200e+02,2.8506170e+02,1.9416150e+02, &
& 1.3273170e+02,9.3883600e+01,7.1776000e+01,5.4902300e+01,4.2608500e+01, &
& 5.8379600e+02,4.5335960e+02,4.1983840e+02,3.3321670e+02,2.6185840e+02, &
& 2.1871850e+02,1.7991240e+02,1.4823250e+02,9.5736170e+02,7.9465850e+02, &
& 6.5759170e+02,6.3652100e+02,5.8366570e+02,4.6085090e+02,5.0355400e+02, &
& 3.9629990e+02,4.2025520e+02,4.3227410e+02,3.3243400e+02,3.4187100e+02, &
& 4.0370590e+02,3.5784550e+02,3.0703710e+02,2.7676970e+02,2.4349600e+02, &
& 2.1293560e+02,1.0744128e+03,9.4720110e+02,8.3242220e+02,7.4998240e+02]
 vdw_dftd3_c6(3201:3300)=[&
& 6.8523850e+02,5.3145620e+02,5.9192160e+02,4.5328020e+02,4.9471860e+02, &
& 4.5942820e+02,3.8395180e+02,4.0566070e+02,5.0664510e+02,4.7008500e+02, &
& 4.2079320e+02,3.9167400e+02,3.5554970e+02,3.2102510e+02,1.3092921e+03, &
& 1.2070341e+03,1.0636276e+03,4.9715300e+02,1.0712093e+03,1.0290835e+03, &
& 1.0033846e+03,9.7972320e+02,9.5873970e+02,7.5756310e+02,8.4922660e+02, &
& 8.1952610e+02,8.6529630e+02,8.4694850e+02,8.3044190e+02,8.2053560e+02, &
& 6.9354790e+02,6.8391950e+02,6.2592010e+02,5.3051690e+02,5.4005040e+02, &
& 4.9085310e+02,4.5068210e+02,3.7600960e+02,3.5193500e+02,3.6207550e+02, &
& 4.5565900e+01,2.9383200e+01,8.4156200e+02,4.4509900e+02,2.8743460e+02, &
& 1.8891370e+02,1.2977590e+02,9.7188700e+01,7.3035300e+01,5.5878200e+01, &
& 1.0013499e+03,7.1984280e+02,6.4850370e+02,4.9588920e+02,3.7786060e+02, &
& 3.0940570e+02,2.4971520e+02,2.0236390e+02,1.6699732e+03,1.3012325e+03, &
& 1.0621257e+03,1.0156676e+03,9.2450540e+02,7.2875260e+02,7.8917040e+02, &
& 6.1912140e+02,6.4688450e+02,6.7034360e+02,5.1478030e+02,5.1777130e+02, &
& 6.1653920e+02,5.3101480e+02,4.4413160e+02,3.9408930e+02,3.4115420e+02, &
& 2.9394640e+02,1.8646164e+03,1.5547036e+03,1.3341432e+03,1.1843558e+03, &
& 1.0716858e+03,8.1713880e+02,9.1594930e+02,6.8833610e+02,7.5141170e+02, &
& 6.9349420e+02,5.8127400e+02,6.0745600e+02,7.7388330e+02,7.0133540e+02, &
& 6.1379650e+02,5.6366330e+02,5.0431850e+02,4.4915730e+02,2.2724078e+03, &
& 1.9977665e+03,1.7187930e+03,7.2802070e+02,1.7584731e+03,1.6821281e+03]
 vdw_dftd3_c6(3301:3400)=[&
& 1.6382173e+03,1.5980083e+03,1.5622939e+03,1.2025229e+03,1.4026627e+03, &
& 1.3479037e+03,1.4013832e+03,1.3707449e+03,1.3427654e+03,1.3281907e+03, &
& 1.1030932e+03,1.0619960e+03,9.5783300e+02,8.0262850e+02,8.1249710e+02, &
& 7.2924020e+02,6.6279090e+02,5.4691720e+02,5.0961460e+02,5.2190910e+02, &
& 7.9223780e+02,4.5178600e+01,2.9396300e+01,7.7049850e+02,4.2598770e+02, &
& 2.8017140e+02,1.8622350e+02,1.2882430e+02,9.6880100e+01,7.3028500e+01, &
& 5.5990900e+01,9.1912570e+02,6.8410110e+02,6.2271340e+02,4.8258810e+02, &
& 3.7129390e+02,3.0563890e+02,2.4781410e+02,2.0153300e+02,1.5164193e+03, &
& 1.2187009e+03,1.0008602e+03,9.6137180e+02,8.7756550e+02,6.9108340e+02, &
& 7.5211560e+02,5.8980850e+02,6.2087440e+02,6.4167640e+02,4.9199760e+02, &
& 4.9993220e+02,5.9373850e+02,5.1689320e+02,4.3592810e+02,3.8853730e+02, &
& 3.3776830e+02,2.9204720e+02,1.6956909e+03,1.4534745e+03,1.2598691e+03, &
& 1.1249617e+03,1.0214733e+03,7.8299060e+02,8.7595820e+02,6.6216750e+02, &
& 7.2370460e+02,6.6938820e+02,5.5927100e+02,5.8793680e+02,7.4387970e+02, &
& 6.8057520e+02,6.0037140e+02,5.5368260e+02,4.9748720e+02,4.4470070e+02, &
& 2.0646687e+03,1.8594085e+03,1.6164677e+03,7.1062890e+02,1.6419885e+03, &
& 1.5739763e+03,1.5337469e+03,1.4968291e+03,1.4640624e+03,1.1380979e+03, &
& 1.3034316e+03,1.2545335e+03,1.3170177e+03,1.2886907e+03,1.2629662e+03, &
& 1.2487891e+03,1.0439034e+03,1.0154887e+03,9.2083050e+02,7.7385260e+02, &
& 7.8529450e+02,7.0790090e+02,6.4554700e+02,5.3402070e+02,4.9813010e+02]
 vdw_dftd3_c6(3401:3500)=[&
& 5.1131450e+02,7.6196480e+02,7.3881560e+02,4.2241900e+01,2.7932900e+01, &
& 6.6152780e+02,3.8103070e+02,2.5605370e+02,1.7278310e+02,1.2079930e+02, &
& 9.1493400e+01,6.9368900e+01,5.3420400e+01,7.9108680e+02,6.0760080e+02, &
& 5.5943360e+02,4.4013430e+02,3.4277880e+02,2.8429480e+02,2.3212190e+02, &
& 1.8987560e+02,1.2966253e+03,1.0686331e+03,8.8251530e+02,8.5188550e+02, &
& 7.7993360e+02,6.1445090e+02,6.7130550e+02,5.2693210e+02,5.5817400e+02, &
& 5.7514870e+02,4.4105510e+02,4.5236780e+02,5.3570130e+02,4.7183760e+02, &
& 4.0199200e+02,3.6047350e+02,3.1527740e+02,2.7407550e+02,1.4529418e+03, &
& 1.2732370e+03,1.1143977e+03,1.0010315e+03,9.1248200e+02,7.0407130e+02, &
& 7.8575170e+02,5.9827920e+02,6.5393370e+02,6.0631280e+02,5.0575740e+02, &
& 5.3417140e+02,6.7073540e+02,6.1958270e+02,5.5156050e+02,5.1139760e+02, &
& 4.6208340e+02,4.1520110e+02,1.7696390e+03,1.6234740e+03,1.4251223e+03, &
& 6.5171260e+02,1.4382615e+03,1.3809451e+03,1.3462674e+03,1.3143784e+03, &
& 1.2860963e+03,1.0103860e+03,1.1390166e+03,1.0983110e+03,1.1597774e+03, &
& 1.1351424e+03,1.1129109e+03,1.0999410e+03,9.2612910e+02,9.0981540e+02, &
& 8.2982850e+02,7.0043950e+02,7.1241700e+02,6.4537650e+02,5.9085050e+02, &
& 4.9072340e+02,4.5847260e+02,4.7152310e+02,6.8856550e+02,6.7269040e+02, &
& 6.1752960e+02,4.0710600e+01,2.7211800e+01,6.0731510e+02,3.5746030e+02, &
& 2.4330570e+02,1.6573270e+02,1.1666150e+02,8.8773600e+01,6.7567200e+01, &
& 5.2188100e+01,7.2726310e+02,5.6780720e+02,5.2626000e+02,4.1768440e+02]
 vdw_dftd3_c6(3501:3600)=[&
& 3.2771360e+02,2.7309950e+02,2.2398500e+02,1.8392230e+02,1.1891902e+03, &
& 9.9221250e+02,8.2178250e+02,7.9553840e+02,7.2956570e+02,5.7517310e+02, &
& 6.2948100e+02,4.9459140e+02,5.2549790e+02,5.4053270e+02,4.1478690e+02, &
& 4.2746160e+02,5.0536000e+02,4.4808220e+02,3.8408530e+02,3.4572660e+02, &
& 3.0354480e+02,2.6479780e+02,1.3343692e+03,1.1818604e+03,1.0398622e+03, &
& 9.3720360e+02,8.5623980e+02,6.6336080e+02,7.3919570e+02,5.6531410e+02, &
& 6.1769980e+02,5.7350990e+02,4.7823360e+02,5.0616860e+02,6.3278330e+02, &
& 5.8760970e+02,5.2587280e+02,4.8916400e+02,4.4353000e+02,3.9983350e+02, &
& 1.6260419e+03,1.5047607e+03,1.3277397e+03,6.2085480e+02,1.3354309e+03, &
& 1.2832336e+03,1.2512879e+03,1.2218775e+03,1.1958046e+03,9.4515700e+02, &
& 1.0569864e+03,1.0202790e+03,1.0796889e+03,1.0568831e+03,1.0363788e+03, &
& 1.0240338e+03,8.6580300e+02,8.5504390e+02,7.8248020e+02,6.6238200e+02, &
& 6.7449800e+02,6.1277720e+02,5.6231000e+02,4.6825250e+02,4.3792710e+02, &
& 4.5080740e+02,6.5074290e+02,6.3822510e+02,5.8861880e+02,5.6260110e+02, &
& 3.8046800e+01,2.5807100e+01,5.3565210e+02,3.2315400e+02,2.2339710e+02, &
& 1.5398830e+02,1.0936180e+02,8.3743700e+01,6.4079700e+01,4.9704300e+01, &
& 6.4255990e+02,5.1104720e+02,4.7738160e+02,3.8289140e+02,3.0317170e+02, &
& 2.5418990e+02,2.0969510e+02,1.7306680e+02,1.0486754e+03,8.8679380e+02, &
& 7.3687680e+02,7.1582420e+02,6.5777000e+02,5.1922800e+02,5.6918500e+02, &
& 4.4794120e+02,4.7740460e+02,4.9002960e+02,3.7654500e+02,3.9005450e+02]
 vdw_dftd3_c6(3601:3700)=[&
& 4.6013230e+02,4.1119140e+02,3.5508990e+02,3.2115720e+02,2.8336260e+02, &
& 2.4831690e+02,1.1787843e+03,1.0561781e+03,9.3492800e+02,8.4595890e+02, &
& 7.7500770e+02,6.0351860e+02,6.7120920e+02,5.1618070e+02,5.6361340e+02, &
& 5.2417160e+02,4.3716280e+02,4.6363160e+02,5.7648690e+02,5.3857130e+02, &
& 4.8503660e+02,4.5297260e+02,4.1249290e+02,3.7339600e+02,1.4375237e+03, &
& 1.3427557e+03,1.1918004e+03,5.7219880e+02,1.1941552e+03,1.1484879e+03, &
& 1.1201674e+03,1.0940580e+03,1.0709219e+03,8.5263060e+02,9.4504500e+02, &
& 9.1333710e+02,9.6825870e+02,9.4792020e+02,9.2971630e+02,9.1834420e+02, &
& 7.8031510e+02,7.7523370e+02,7.1228880e+02,6.0524130e+02,6.1710680e+02, &
& 5.6258800e+02,5.1772850e+02,4.3264880e+02,4.0518020e+02,4.1747370e+02, &
& 5.9434970e+02,5.8543140e+02,5.4287440e+02,5.2058720e+02,4.8365360e+02, &
& 3.5275000e+01,2.4288600e+01,4.7129940e+02,2.9040790e+02,2.0366210e+02, &
& 1.4199460e+02,1.0173400e+02,7.8399300e+01,6.0320800e+01,4.6997000e+01, &
& 5.6630470e+02,4.5752070e+02,4.3040700e+02,3.4853240e+02,2.7834410e+02, &
& 2.3475630e+02,1.9477780e+02,1.6157780e+02,9.2348960e+02,7.8946510e+02, &
& 6.5782090e+02,6.4106890e+02,5.9013650e+02,4.6657860e+02,5.1200690e+02, &
& 4.0371620e+02,4.3125430e+02,4.4180520e+02,3.4010290e+02,3.5375690e+02, &
& 4.1640720e+02,3.7473780e+02,3.2584680e+02,2.9605910e+02,2.6246780e+02, &
& 2.3103920e+02,1.0398788e+03,9.4037660e+02,8.3682440e+02,7.5986860e+02, &
& 6.9790750e+02,5.4617620e+02,6.0628810e+02,4.6875270e+02,5.1133700e+02]
 vdw_dftd3_c6(3701:3800)=[&
& 4.7629130e+02,3.9751230e+02,4.2213930e+02,5.2225240e+02,4.9045740e+02, &
& 4.4423510e+02,4.1640380e+02,3.8074630e+02,3.4603400e+02,1.2691514e+03, &
& 1.1942553e+03,1.0653886e+03,5.2379990e+02,1.0641113e+03,1.0241601e+03, &
& 9.9910040e+02,9.7596710e+02,9.5547530e+02,7.6577540e+02,8.4248080e+02, &
& 8.1508100e+02,8.6486000e+02,8.4676410e+02,8.3063370e+02,8.2022230e+02, &
& 7.0009390e+02,6.9908500e+02,6.4467130e+02,5.4983920e+02,5.6121370e+02, &
& 5.1327610e+02,4.7360870e+02,3.9719060e+02,3.7248250e+02,3.8402410e+02, &
& 5.3990550e+02,5.3369840e+02,4.9729720e+02,4.7831130e+02,4.4604470e+02, &
& 4.1282750e+02,1.0074940e+02,5.9659400e+01,2.8945966e+03,1.2189348e+03, &
& 7.1065850e+02,4.3419670e+02,2.8277630e+02,2.0397230e+02,1.4835840e+02, &
& 1.1051200e+02,3.3986461e+03,2.0446296e+03,1.7465418e+03,1.2378269e+03, &
& 8.8882250e+02,7.0197610e+02,5.4727120e+02,4.3044680e+02,6.0209123e+03, &
& 3.9885535e+03,3.1533933e+03,2.9483214e+03,2.6441348e+03,2.0951563e+03, &
& 2.2092516e+03,1.7359767e+03,1.7406013e+03,1.8302286e+03,1.4168737e+03, &
& 1.3464975e+03,1.6330869e+03,1.3219669e+03,1.0509064e+03,9.0547240e+02, &
& 7.6070860e+02,6.3792100e+02,6.6879551e+03,4.8140613e+03,3.9238630e+03, &
& 3.3777348e+03,3.0012009e+03,2.2208404e+03,2.5189457e+03,1.8286037e+03, &
& 1.9819822e+03,1.8052792e+03,1.5403020e+03,1.5550964e+03,2.0671731e+03, &
& 1.7732620e+03,1.4801255e+03,1.3236329e+03,1.1516386e+03,9.9949930e+02, &
& 8.2315280e+03,6.3397235e+03,5.1715628e+03,1.7765919e+03,5.4969601e+03]
 vdw_dftd3_c6(3801:3900)=[&
& 5.1886826e+03,5.0376830e+03,4.9009739e+03,4.7792385e+03,3.5004650e+03, &
& 4.5180145e+03,4.3256882e+03,4.2216216e+03,4.1211013e+03,4.0270614e+03, &
& 3.9908778e+03,3.2187704e+03,2.9218119e+03,2.5563295e+03,2.1076991e+03, &
& 2.1032454e+03,1.8383882e+03,1.6363884e+03,1.3271181e+03,1.2270914e+03, &
& 1.2394986e+03,2.1094753e+03,1.9333025e+03,1.6699965e+03,1.5408317e+03, &
& 1.3687363e+03,1.2134959e+03,7.3147398e+03,9.9413100e+01,5.9674000e+01, &
& 2.4340697e+03,1.1320344e+03,6.8137890e+02,4.2416130e+02,2.7936020e+02, &
& 2.0282600e+02,1.4825160e+02,1.1081260e+02,2.8745717e+03,1.8719600e+03, &
& 1.6279493e+03,1.1837103e+03,8.6434290e+02,6.8837560e+02,5.4058680e+02, &
& 4.2751470e+02,4.9128581e+03,3.5289590e+03,2.8292273e+03,2.6655950e+03, &
& 2.4036878e+03,1.8947707e+03,2.0238919e+03,1.5845838e+03,1.6192221e+03, &
& 1.6942469e+03,1.3022136e+03,1.2677966e+03,1.5270149e+03,1.2635730e+03, &
& 1.0197251e+03,8.8504330e+02,7.4865020e+02,6.3127570e+02,5.4594039e+03, &
& 4.2339736e+03,3.5246180e+03,3.0700206e+03,2.7442675e+03,2.0488350e+03, &
& 2.3149349e+03,1.6984672e+03,1.8511077e+03,1.6939362e+03,1.4296950e+03, &
& 1.4674624e+03,1.9210215e+03,1.6832537e+03,1.4267831e+03,1.2854608e+03, &
& 1.1265319e+03,9.8363370e+02,6.6646049e+03,5.5052582e+03,4.5943095e+03, &
& 1.7037104e+03,4.7976061e+03,4.5623865e+03,4.4362592e+03,4.3214914e+03, &
& 4.2193690e+03,3.1447184e+03,3.8693475e+03,3.7006978e+03,3.7536534e+03, &
& 3.6679746e+03,3.5884351e+03,3.5540067e+03,2.8892983e+03,2.6921229e+03]
 vdw_dftd3_c6(3901:4000)=[&
& 2.3813318e+03,1.9684730e+03,1.9760672e+03,1.7428522e+03,1.5616887e+03, &
& 1.2704258e+03,1.1766222e+03,1.1957573e+03,1.9547292e+03,1.8286515e+03, &
& 1.6046349e+03,1.4914089e+03,1.3351632e+03,1.1909286e+03,6.0457168e+03, &
& 5.3054399e+03,8.9406000e+01,5.4461300e+01,2.0082591e+03,9.7721480e+02, &
& 6.0005330e+02,3.7866720e+02,2.5180860e+02,1.8403180e+02,1.3527920e+02, &
& 1.0158570e+02,2.3774617e+03,1.6040300e+03,1.4098711e+03,1.0405681e+03, &
& 7.6851320e+02,6.1613190e+02,4.8688350e+02,3.8707200e+02,4.0174782e+03, &
& 2.9771478e+03,2.4021540e+03,2.2732739e+03,2.0559589e+03,1.6183634e+03, &
& 1.7384495e+03,1.3600663e+03,1.4017622e+03,1.4625357e+03,1.1217466e+03, &
& 1.1049523e+03,1.3263601e+03,1.1110958e+03,9.0553420e+02,7.9026450e+02, &
& 6.7216580e+02,5.6954790e+02,4.4690053e+03,3.5643510e+03,2.9977722e+03, &
& 2.6270446e+03,2.3567083e+03,1.7692538e+03,1.9948729e+03,1.4730457e+03, &
& 1.6081722e+03,1.4753924e+03,1.2402274e+03,1.2822835e+03,1.6650502e+03, &
& 1.4751121e+03,1.2620917e+03,1.1429090e+03,1.0069141e+03,8.8341400e+02, &
& 5.4478929e+03,4.6120247e+03,3.8897052e+03,1.5031569e+03,4.0323992e+03, &
& 3.8433536e+03,3.7393646e+03,3.6445469e+03,3.5602278e+03,2.6796802e+03, &
& 3.2333681e+03,3.0968078e+03,3.1769406e+03,3.1056847e+03,3.0398461e+03, &
& 3.0095999e+03,2.4623774e+03,2.3202277e+03,2.0645670e+03,1.7115570e+03, &
& 1.7232231e+03,1.5276535e+03,1.3743640e+03,1.1213924e+03,1.0400238e+03, &
& 1.0600607e+03,1.6946605e+03,1.6008616e+03,1.4174346e+03,1.3236360e+03]
 vdw_dftd3_c6(4001:4100)=[&
& 1.1913272e+03,1.0675065e+03,4.9814961e+03,4.4669584e+03,3.7996565e+03, &
& 8.2083600e+01,5.1110300e+01,1.6724768e+03,8.5252490e+02,5.3608580e+02, &
& 3.4435760e+02,2.3207230e+02,1.7124180e+02,1.2693930e+02,9.5984500e+01, &
& 1.9852103e+03,1.3883756e+03,1.2352681e+03,9.2751070e+02,6.9479830e+02, &
& 5.6206360e+02,4.4804860e+02,3.5892730e+02,3.3208171e+03,2.5377308e+03, &
& 2.0608901e+03,1.9602827e+03,1.7786546e+03,1.3994778e+03,1.5110816e+03, &
& 1.1825093e+03,1.2287380e+03,1.2777896e+03,9.7919840e+02,9.7588560e+02, &
& 1.1670583e+03,9.9120150e+02,8.1755520e+02,7.1866790e+02,6.1585050e+02, &
& 5.2542400e+02,3.6996637e+03,3.0331273e+03,2.5787074e+03,2.2748325e+03, &
& 2.0492299e+03,1.5491780e+03,1.7421202e+03,1.2966551e+03,1.4168423e+03, &
& 1.3036060e+03,1.0925939e+03,1.1371745e+03,1.4631580e+03,1.3115880e+03, &
& 1.1344385e+03,1.0338769e+03,9.1709200e+02,8.0980180e+02,4.5055681e+03, &
& 3.9071262e+03,3.3315473e+03,1.3478223e+03,3.4277894e+03,3.2744792e+03, &
& 3.1877428e+03,3.1084769e+03,3.0380345e+03,2.3122086e+03,2.7358113e+03, &
& 2.6241922e+03,2.7191319e+03,2.6591268e+03,2.6040009e+03,2.5769767e+03, &
& 2.1236819e+03,2.0248239e+03,1.8137564e+03,1.5101361e+03,1.5249297e+03, &
& 1.3598576e+03,1.2292164e+03,1.0074225e+03,9.3609580e+02,9.5680470e+02, &
& 1.4915698e+03,1.4229431e+03,1.2726601e+03,1.1951492e+03,1.0828719e+03, &
& 9.7614170e+02,4.1555932e+03,3.8047750e+03,3.2704258e+03,2.8472704e+03, &
& 8.0726000e+01,5.0005200e+01,1.7306038e+03,8.5763790e+02,5.3308180e+02]
 vdw_dftd3_c6(4101:4200)=[&
& 3.3998630e+02,2.2813330e+02,1.6791870e+02,1.2426420e+02,9.3866300e+01, &
& 2.0511151e+03,1.4030924e+03,1.2403071e+03,9.2324960e+02,6.8729140e+02, &
& 5.5412740e+02,4.4044180e+02,3.5207430e+02,3.4570434e+03,2.5892888e+03, &
& 2.0945389e+03,1.9869265e+03,1.7996577e+03,1.4173421e+03,1.5251022e+03, &
& 1.1941778e+03,1.2344914e+03,1.2859300e+03,9.8684670e+02,9.7668650e+02, &
& 1.1699494e+03,9.8664000e+02,8.0937540e+02,7.0943800e+02,6.0630260e+02, &
& 5.1614250e+02,3.8488310e+03,3.0987868e+03,2.6179361e+03,2.3009943e+03, &
& 2.0684111e+03,1.5587481e+03,1.7549706e+03,1.3015598e+03,1.4207211e+03, &
& 1.3053320e+03,1.0968175e+03,1.1367042e+03,1.4690715e+03,1.3085146e+03, &
& 1.1258498e+03,1.0231942e+03,9.0511410e+02,7.9733130e+02,4.6916964e+03, &
& 4.0037089e+03,3.3915606e+03,1.3396248e+03,3.5059336e+03,3.3442162e+03, &
& 3.2544268e+03,3.1724783e+03,3.0996207e+03,2.3450320e+03,2.8084101e+03, &
& 2.6917985e+03,2.7690943e+03,2.7073188e+03,2.6503867e+03,2.6234579e+03, &
& 2.1538516e+03,2.0393959e+03,1.8205331e+03,1.5133881e+03,1.5256409e+03, &
& 1.3566458e+03,1.2236676e+03,1.0014754e+03,9.2999430e+02,9.4896980e+02, &
& 1.4976018e+03,1.4206996e+03,1.2642423e+03,1.1841627e+03,1.0698651e+03, &
& 9.6217680e+02,4.3039645e+03,3.8865716e+03,3.3200764e+03,2.8725136e+03, &
& 2.9089206e+03,7.8203400e+01,4.8498000e+01,1.6716156e+03,8.2971320e+02, &
& 5.1601950e+02,3.2928020e+02,2.2106590e+02,1.6279260e+02,1.2052900e+02, &
& 9.1086900e+01,1.9814675e+03,1.3571628e+03,1.2000275e+03,8.9364440e+02]
 vdw_dftd3_c6(4201:4300)=[&
& 6.6549650e+02,5.3670890e+02,4.2673540e+02,3.4123400e+02,3.3376961e+03, &
& 2.5033576e+03,2.0254740e+03,1.9216819e+03,1.7407294e+03,1.3709264e+03, &
& 1.4753669e+03,1.1552737e+03,1.1945389e+03,1.2442037e+03,9.5483090e+02, &
& 9.4527750e+02,1.1321107e+03,9.5506070e+02,7.8370690e+02,6.8708390e+02, &
& 5.8734500e+02,5.0013680e+02,3.7160678e+03,2.9957186e+03,2.5317373e+03, &
& 2.2256807e+03,2.0009439e+03,1.5082494e+03,1.6979484e+03,1.2596142e+03, &
& 1.3749665e+03,1.2634136e+03,1.0615457e+03,1.1003399e+03,1.4216099e+03, &
& 1.2666329e+03,1.0901053e+03,9.9086960e+02,8.7669020e+02,7.7245190e+02, &
& 4.5291727e+03,3.8697611e+03,3.2793360e+03,1.2970461e+03,3.3890395e+03, &
& 3.2330975e+03,3.1463690e+03,3.0672053e+03,2.9968237e+03,2.2680138e+03, &
& 2.7140296e+03,2.6013226e+03,2.6775672e+03,2.6178718e+03,2.5628645e+03, &
& 2.5367915e+03,2.0830385e+03,1.9730821e+03,1.7616844e+03,1.4646529e+03, &
& 1.4766337e+03,1.3133066e+03,1.1847530e+03,9.6980220e+02,9.0065620e+02, &
& 9.1908240e+02,1.4493061e+03,1.3752752e+03,1.2241322e+03,1.1467509e+03, &
& 1.0362496e+03,9.3211210e+02,4.1565121e+03,3.7572430e+03,3.2107560e+03, &
& 2.7789432e+03,2.8136269e+03,2.7215209e+03,7.9439100e+01,4.8724100e+01, &
& 1.8005066e+03,8.6956410e+02,5.3289880e+02,3.3646270e+02,2.2420940e+02, &
& 1.6429590e+02,1.2116670e+02,9.1309000e+01,2.1313007e+03,1.4294308e+03, &
& 1.2543604e+03,9.2419700e+02,6.8227500e+02,5.4731890e+02,4.3301050e+02, &
& 3.4483050e+02,3.6085602e+03,2.6611178e+03,2.1449232e+03,2.0287846e+03]
 vdw_dftd3_c6(4301:4400)=[&
& 1.8341800e+03,1.4450682e+03,1.5501899e+03,1.2138708e+03,1.2487909e+03, &
& 1.3033334e+03,1.0009549e+03,9.8372150e+02,1.1804499e+03,9.8737510e+02, &
& 8.0425620e+02,7.0203110e+02,5.9750350e+02,5.0682160e+02,4.0138973e+03, &
& 3.1875734e+03,2.6766472e+03,2.3437586e+03,2.1018417e+03,1.5778736e+03, &
& 1.7790323e+03,1.3137158e+03,1.4331563e+03,1.3146561e+03,1.1068617e+03, &
& 1.1425196e+03,1.4840150e+03,1.3126653e+03,1.1221440e+03,1.0160025e+03, &
& 8.9523210e+02,7.8580380e+02,4.8930436e+03,4.1279330e+03,3.4757124e+03, &
& 1.3372850e+03,3.6078825e+03,3.4378544e+03,3.3445279e+03,3.2594431e+03, &
& 3.1837627e+03,2.3935584e+03,2.8964080e+03,2.7730351e+03,2.8396469e+03, &
& 2.7757424e+03,2.7166522e+03,2.6897033e+03,2.1986632e+03,2.0680029e+03, &
& 1.8390464e+03,1.5251115e+03,1.5347972e+03,1.3602378e+03,1.2236448e+03, &
& 9.9920960e+02,9.2702850e+02,9.4428620e+02,1.5116714e+03,1.4258266e+03, &
& 1.2612415e+03,1.1774038e+03,1.0596377e+03,9.4975540e+02,4.4697774e+03, &
& 3.9952606e+03,3.3932601e+03,2.9170992e+03,2.9646668e+03,2.8671025e+03, &
& 3.0329760e+03,7.7102100e+01,4.7379800e+01,1.7278140e+03,8.3944780e+02, &
& 5.1579770e+02,3.2625000e+02,2.1767650e+02,1.5964350e+02,1.1781930e+02, &
& 8.8836500e+01,2.0459284e+03,1.3785876e+03,1.2114443e+03,8.9432550e+02, &
& 6.6120920e+02,5.3088280e+02,4.2034980e+02,3.3497590e+02,3.4586883e+03, &
& 2.5612750e+03,2.0661768e+03,1.9554505e+03,1.7685649e+03,1.3931385e+03, &
& 1.4955657e+03,1.1710013e+03,1.2060158e+03,1.2582170e+03,9.6605990e+02]
 vdw_dftd3_c6(4401:4500)=[&
& 9.5086030e+02,1.1405197e+03,9.5550390e+02,7.7929760e+02,6.8073550e+02, &
& 5.7979560e+02,4.9211280e+02,3.8477324e+03,3.0671463e+03,2.5790022e+03, &
& 2.2600639e+03,2.0277440e+03,1.5233742e+03,1.7171041e+03,1.2690647e+03, &
& 1.3847315e+03,1.2706578e+03,1.0692791e+03,1.1047467e+03,1.4334370e+03, &
& 1.2697368e+03,1.0867671e+03,9.8462910e+02,8.6818680e+02,7.6254070e+02, &
& 4.6894909e+03,3.9694488e+03,3.3469278e+03,1.2946997e+03,3.4708103e+03, &
& 3.3082782e+03,3.2187253e+03,3.1370576e+03,3.0644224e+03,2.3068366e+03, &
& 2.7842586e+03,2.6660816e+03,2.7342974e+03,2.6729042e+03,2.6161724e+03, &
& 2.5900957e+03,2.1189811e+03,1.9960266e+03,1.7764187e+03,1.4737544e+03, &
& 1.4836747e+03,1.3158080e+03,1.1842955e+03,9.6746430e+02,8.9773860e+02, &
& 9.1479120e+02,1.4602247e+03,1.3790263e+03,1.2212647e+03,1.1407827e+03, &
& 1.0273948e+03,9.2140470e+02,4.2885858e+03,3.8444325e+03,3.2695288e+03, &
& 2.8146033e+03,2.8581875e+03,2.7642585e+03,2.9218223e+03,2.8152366e+03, &
& 4.7379000e+00,3.1287000e+00,6.8939100e+01,4.1307800e+01,2.8276700e+01, &
& 1.9265300e+01,1.3516400e+01,1.0237100e+01,7.7441000e+00,5.9403000e+00, &
& 8.2564100e+01,6.5370300e+01,6.0855800e+01,4.8518200e+01,3.8139700e+01, &
& 3.1771300e+01,2.6017000e+01,2.1305500e+01,1.3444410e+02,1.1348570e+02, &
& 9.4208300e+01,9.1333400e+01,8.3831300e+01,6.5983700e+01,7.2411100e+01, &
& 5.6799300e+01,6.0574600e+01,6.2253200e+01,4.7649900e+01,4.9346900e+01, &
& 5.8420700e+01,5.1990500e+01,4.4651200e+01,4.0201000e+01,3.5274000e+01]
 vdw_dftd3_c6(4501:4600)=[&
& 3.0726100e+01,1.5093400e+02,1.3506640e+02,1.1929610e+02,1.0773060e+02, &
& 9.8518300e+01,7.6366100e+01,8.5080000e+01,6.5086300e+01,7.1198500e+01, &
& 6.6121100e+01,5.4965800e+01,5.8356800e+01,7.2907600e+01,6.7943000e+01, &
& 6.0947000e+01,5.6742700e+01,5.1465600e+01,4.6379400e+01,1.8402230e+02, &
& 1.7170330e+02,1.5211060e+02,7.1883000e+01,1.5245020e+02,1.4658510e+02, &
& 1.4296200e+02,1.3962540e+02,1.3666950e+02,1.0836620e+02,1.2040560e+02, &
& 1.1631780e+02,1.2352200e+02,1.2093020e+02,1.1860700e+02,1.1718110e+02, &
& 9.9298400e+01,9.8513600e+01,9.0285800e+01,7.6415000e+01,7.7879300e+01, &
& 7.0802400e+01,6.4989000e+01,5.4045100e+01,5.0508600e+01,5.2054700e+01, &
& 7.4881700e+01,7.3669200e+01,6.8103100e+01,6.5160500e+01,6.0333800e+01, &
& 5.5434500e+01,1.7492120e+02,1.7050820e+02,1.5181390e+02,1.3747240e+02, &
& 1.3587010e+02,1.3156440e+02,1.3473040e+02,1.3059460e+02,7.5916000e+00, &
& 1.4316500e+01,8.7773000e+00,2.8221060e+02,1.4813230e+02,9.3672900e+01, &
& 6.0086000e+01,4.0294400e+01,2.9557500e+01,2.1760500e+01,1.6338800e+01, &
& 3.3534500e+02,2.4016270e+02,2.1472650e+02,1.6205080e+02,1.2150670e+02, &
& 9.8124900e+01,7.7986500e+01,6.2220500e+01,5.5515160e+02,4.3415760e+02, &
& 3.5403170e+02,3.3738960e+02,3.0654820e+02,2.4050330e+02,2.6090190e+02, &
& 2.0361490e+02,2.1289780e+02,2.2117390e+02,1.6880750e+02,1.6949490e+02, &
& 2.0269000e+02,1.7294250e+02,1.4284710e+02,1.2547680e+02,1.0733800e+02, &
& 9.1333300e+01,6.1856810e+02,5.1783930e+02,4.4294480e+02,3.9191790e+02]
 vdw_dftd3_c6(4601:4700)=[&
& 3.5350080e+02,2.6737610e+02,3.0065330e+02,2.2388710e+02,2.4525610e+02, &
& 2.2580420e+02,1.8833740e+02,1.9709400e+02,2.5313700e+02,2.2808140e+02, &
& 1.9776980e+02,1.8030560e+02,1.5986700e+02,1.4098180e+02,7.5234180e+02, &
& 6.6457190e+02,5.7042660e+02,2.3452710e+02,5.8397970e+02,5.5869720e+02, &
& 5.4412320e+02,5.3079090e+02,5.1894980e+02,3.9672230e+02,4.6354420e+02, &
& 4.4501010e+02,4.6541000e+02,4.5528270e+02,4.4600300e+02,4.4132200e+02, &
& 3.6474670e+02,3.5006170e+02,3.1424830e+02,2.6138330e+02,2.6436970e+02, &
& 2.3602580e+02,2.1346440e+02,1.7461320e+02,1.6213420e+02,1.6604890e+02, &
& 2.5738230e+02,2.4683550e+02,2.2143250e+02,2.0812690e+02,1.8860730e+02, &
& 1.6989430e+02,6.9764490e+02,6.4903250e+02,5.6135190e+02,4.9107770e+02, &
& 4.9346510e+02,4.7742540e+02,4.9952020e+02,4.8234660e+02,2.4057300e+01, &
& 8.5319700e+01,1.8465600e+01,1.1465500e+01,3.5221600e+02,1.8634530e+02, &
& 1.1908150e+02,7.7095100e+01,5.2085200e+01,3.8413200e+01,2.8409800e+01, &
& 2.1406700e+01,4.1864880e+02,3.0137770e+02,2.7074610e+02,2.0575440e+02, &
& 1.5536440e+02,1.2609930e+02,1.0072120e+02,8.0717400e+01,6.9485960e+02, &
& 5.4358420e+02,4.4371900e+02,4.2362170e+02,3.8527540e+02,3.0260610e+02, &
& 3.2840120e+02,2.5660430e+02,2.6862840e+02,2.7871160e+02,2.1297970e+02, &
& 2.1442560e+02,2.5620490e+02,2.1973750e+02,1.8253380e+02,1.6096870e+02, &
& 1.3827470e+02,1.1812400e+02,7.7496560e+02,6.4864220e+02,5.5615030e+02, &
& 4.9301000e+02,4.4537360e+02,3.3790350e+02,3.7952540e+02,2.8356950e+02]
 vdw_dftd3_c6(4701:4800)=[&
& 3.1037810e+02,2.8603950e+02,2.3869620e+02,2.4999150e+02,3.2009120e+02, &
& 2.8947130e+02,2.5216490e+02,2.3063090e+02,2.0522370e+02,1.8162600e+02, &
& 9.4385470e+02,8.3262510e+02,7.1607580e+02,2.9887820e+02,7.3216420e+02, &
& 7.0044420e+02,6.8219010e+02,6.6548680e+02,6.5065570e+02,4.9918370e+02, &
& 5.8212240e+02,5.5929170e+02,5.8370670e+02,5.7100390e+02,5.5938980e+02, &
& 5.5342660e+02,4.5865820e+02,4.4137890e+02,3.9713120e+02,3.3121240e+02, &
& 3.3522610e+02,2.9995650e+02,2.7181190e+02,2.2291570e+02,2.0718310e+02, &
& 2.1231320e+02,3.2610680e+02,3.1346750e+02,2.8229350e+02,2.6602000e+02, &
& 2.4186880e+02,2.1856870e+02,8.7585630e+02,8.1413080e+02,7.0540110e+02, &
& 6.1906050e+02,6.2133990e+02,6.0114960e+02,6.2763850e+02,6.0621390e+02, &
& 3.0786600e+01,1.0750830e+02,1.3584450e+02,1.4723700e+01,9.5836000e+00, &
& 2.3142090e+02,1.3372650e+02,8.9742900e+01,6.0299600e+01,4.1901500e+01, &
& 3.1535400e+01,2.3736900e+01,1.8142400e+01,2.7656710e+02,2.1299790e+02, &
& 1.9620140e+02,1.5429010e+02,1.1993530e+02,9.9215900e+01,8.0727200e+01, &
& 6.5758000e+01,4.5251930e+02,3.7388150e+02,3.0887210e+02,2.9808710e+02, &
& 2.7287470e+02,2.1463600e+02,2.3480000e+02,1.8397370e+02,1.9517310e+02, &
& 2.0113510e+02,1.5388240e+02,1.5808480e+02,1.8757970e+02,1.6519300e+02, &
& 1.4056250e+02,1.2584370e+02,1.0980600e+02,9.5176300e+01,5.0694210e+02, &
& 4.4526080e+02,3.8988230e+02,3.5019700e+02,3.1911510e+02,2.4585270e+02, &
& 2.7453340e+02,2.0863620e+02,2.2828220e+02,2.1155590e+02,1.7606030e+02]
 vdw_dftd3_c6(4801:4900)=[&
& 1.8621990e+02,2.3421710e+02,2.1642760e+02,1.9255420e+02,1.7839190e+02, &
& 1.6096790e+02,1.4436970e+02,6.1760580e+02,5.6751360e+02,4.9845340e+02, &
& 2.2741890e+02,5.0249670e+02,4.8250770e+02,4.7040290e+02,4.5927530e+02, &
& 4.4941020e+02,3.5290870e+02,3.9744990e+02,3.8329520e+02,4.0532850e+02, &
& 3.9673770e+02,3.8898900e+02,3.8447010e+02,3.2364500e+02,3.1828720e+02, &
& 2.9018010e+02,2.4456750e+02,2.4877750e+02,2.2518190e+02,2.0597020e+02, &
& 1.7065560e+02,1.5926350e+02,1.6388200e+02,2.4000000e+02,2.3460480e+02, &
& 2.1530270e+02,2.0513340e+02,1.8899910e+02,1.7288540e+02,5.8298150e+02, &
& 5.6100860e+02,4.9563890e+02,4.4481020e+02,4.4155780e+02,4.2748070e+02, &
& 4.4027100e+02,4.2632550e+02,2.3841200e+01,7.7663300e+01,9.8993800e+01, &
& 7.5368600e+01,1.1393200e+01,7.7065000e+00,1.6045900e+02,9.6543100e+01, &
& 6.6798600e+01,4.6068100e+01,3.2700900e+01,2.5009700e+01,1.9097400e+01, &
& 1.4771600e+01,1.9237290e+02,1.5261300e+02,1.4260940e+02,1.1444050e+02, &
& 9.0676700e+01,7.6055200e+01,6.2746500e+01,5.1768400e+01,3.1458000e+02, &
& 2.6512060e+02,2.2018540e+02,2.1386820e+02,1.9648850e+02,1.5511710e+02, &
& 1.6997960e+02,1.3377000e+02,1.4251400e+02,1.4627110e+02,1.1239140e+02, &
& 1.1639870e+02,1.3747230e+02,1.2288050e+02,1.0616860e+02,9.6059900e+01, &
& 8.4772300e+01,7.4286500e+01,3.5363020e+02,3.1589230e+02,2.7947270e+02, &
& 2.5282130e+02,2.3159760e+02,1.8033900e+02,2.0054640e+02,1.5419380e+02, &
& 1.6831150e+02,1.5650060e+02,1.3051440e+02,1.3836720e+02,1.7212290e+02]
 vdw_dftd3_c6(4901:5000)=[&
& 1.6079100e+02,1.4484620e+02,1.3531220e+02,1.2325380e+02,1.1159230e+02, &
& 4.3151720e+02,4.0187740e+02,3.5646100e+02,1.7092620e+02,3.5713990e+02, &
& 3.4341530e+02,3.3492450e+02,3.2709910e+02,3.2016570e+02,2.5481220e+02, &
& 2.8285470e+02,2.7333750e+02,2.8939200e+02,2.8330010e+02,2.7784930e+02, &
& 2.7445000e+02,2.3313940e+02,2.3163340e+02,2.1280410e+02,1.8084620e+02, &
& 1.8435860e+02,1.6804770e+02,1.5462480e+02,1.2918350e+02,1.2094850e+02, &
& 1.2459510e+02,1.7748530e+02,1.7476750e+02,1.6207400e+02,1.5545060e+02, &
& 1.4445320e+02,1.3324320e+02,4.1058600e+02,3.9952250e+02,3.5611270e+02, &
& 3.2349590e+02,3.1967610e+02,3.0960820e+02,3.1659670e+02,3.0693030e+02, &
& 1.8057500e+01,5.6346800e+01,7.2316200e+01,5.6529500e+01,4.3245200e+01, &
& 8.1417000e+00,5.7601000e+00,1.0127010e+02,6.3858000e+01,4.5731900e+01, &
& 3.2484800e+01,2.3629500e+01,1.8412800e+01,1.4301500e+01,1.1224300e+01, &
& 1.2196010e+02,1.0014430e+02,9.5091600e+01,7.8046600e+01,6.3166900e+01, &
& 5.3810700e+01,4.5095200e+01,3.7752200e+01,1.9944050e+02,1.7204550e+02, &
& 1.4376580e+02,1.4071150e+02,1.2982970e+02,1.0301890e+02,1.1302480e+02, &
& 8.9486800e+01,9.5705100e+01,9.7776100e+01,7.5598700e+01,7.8918300e+01, &
& 9.2602500e+01,8.4127600e+01,7.3914200e+01,6.7663600e+01,6.0472800e+01, &
& 5.3649600e+01,2.2519190e+02,2.0515740e+02,1.8368510e+02,1.6755010e+02, &
& 1.5443560e+02,1.2180660e+02,1.3478870e+02,1.0509050e+02,1.1434960e+02, &
& 1.0674380e+02,8.9341900e+01,9.4877000e+01,1.1649590e+02,1.1009510e+02]
 vdw_dftd3_c6(5001:5100)=[&
& 1.0051820e+02,9.4750800e+01,8.7202800e+01,7.9777300e+01,2.7525670e+02, &
& 2.6044810e+02,2.3366190e+02,1.1854700e+02,2.3257830e+02,2.2400400e+02, &
& 2.1856110e+02,2.1352770e+02,2.0907000e+02,1.6905560e+02,1.8459040e+02, &
& 1.7880740e+02,1.8945310e+02,1.8549310e+02,1.8198230e+02,1.7962020e+02, &
& 1.5422920e+02,1.5491020e+02,1.4357630e+02,1.2323020e+02,1.2591210e+02, &
& 1.1569380e+02,1.0717750e+02,9.0445600e+01,8.5014600e+01,8.7649000e+01, &
& 1.2108000e+02,1.2012680e+02,1.1264850e+02,1.0881340e+02,1.0205240e+02, &
& 9.4992800e+01,2.6444300e+02,2.6065290e+02,2.3476520e+02,2.1630740e+02, &
& 2.1274560e+02,2.0616310e+02,2.0915240e+02,2.0304180e+02,1.2598000e+01, &
& 3.7493900e+01,4.8488500e+01,3.9043300e+01,3.0541000e+01,2.2124100e+01, &
& 6.0575000e+00,4.4593000e+00,6.7931200e+01,4.4496800e+01,3.2813600e+01, &
& 2.3912000e+01,1.7769800e+01,1.4076400e+01,1.1099400e+01,8.8252000e+00, &
& 8.2182000e+01,6.9367500e+01,6.6745100e+01,5.5817500e+01,4.5993700e+01, &
& 3.9709100e+01,3.3731300e+01,2.8600000e+01,1.3465080e+02,1.1824110e+02, &
& 9.9293900e+01,9.7834500e+01,9.0600700e+01,7.2287200e+01,7.9304700e+01, &
& 6.3186500e+01,6.7711700e+01,6.8908600e+01,5.3642000e+01,5.6288400e+01, &
& 6.5621400e+01,6.0404100e+01,5.3815300e+01,4.9755200e+01,4.4951000e+01, &
& 4.0304300e+01,1.5267570e+02,1.4115920e+02,1.2762820e+02,1.1723280e+02, &
& 1.0863780e+02,8.6680300e+01,9.5496600e+01,7.5391000e+01,8.1764000e+01, &
& 7.6587100e+01,6.4388000e+01,6.8394300e+01,8.3027800e+01,7.9170400e+01]
 vdw_dftd3_c6(5101:5200)=[&
& 7.3071800e+01,6.9390100e+01,6.4414100e+01,5.9446800e+01,1.8687910e+02, &
& 1.7895490e+02,1.6203710e+02,8.6166500e+01,1.6052330e+02,1.5480670e+02, &
& 1.5109700e+02,1.4765520e+02,1.4460750e+02,1.1853880e+02,1.2778280e+02, &
& 1.2401580e+02,1.3130040e+02,1.2856660e+02,1.2616140e+02,1.2444020e+02, &
& 1.0783510e+02,1.0920420e+02,1.0197340e+02,8.8315600e+01,9.0391800e+01, &
& 8.3622700e+01,7.7919400e+01,6.6359100e+01,6.2600800e+01,6.4555200e+01, &
& 8.6955200e+01,8.6739200e+01,8.2055900e+01,7.9713300e+01,7.5326800e+01, &
& 7.0648000e+01,1.8100550e+02,1.8009880e+02,1.6360420e+02,1.5256500e+02, &
& 1.4955510e+02,1.4501130e+02,1.4619280e+02,1.4207750e+02,9.1812000e+00, &
& 2.6270800e+01,3.4198300e+01,2.8232700e+01,2.2517800e+01,1.6677500e+01, &
& 1.2816100e+01,4.2672000e+00,3.3077000e+00,4.2062700e+01,2.8861500e+01, &
& 2.2093100e+01,1.6637300e+01,1.2710700e+01,1.0286700e+01,8.2739000e+00, &
& 6.6950000e+00,5.1249800e+01,4.4714100e+01,4.3736400e+01,3.7442100e+01, &
& 3.1557700e+01,2.7713700e+01,2.3950500e+01,2.0639400e+01,8.4368800e+01, &
& 7.5585000e+01,6.3856400e+01,6.3474800e+01,5.9066500e+01,4.7531800e+01, &
& 5.2078000e+01,4.1894200e+01,4.4940700e+01,4.5515500e+01,3.5810000e+01, &
& 3.7755900e+01,4.3568500e+01,4.0750000e+01,3.6938900e+01,3.4579200e+01, &
& 3.1669900e+01,2.8780000e+01,9.6223500e+01,9.0408100e+01,8.2733700e+01, &
& 7.6671200e+01,7.1549300e+01,5.7978400e+01,6.3507900e+01,5.0977000e+01, &
& 5.5024000e+01,5.1774000e+01,4.3848000e+01,4.6534000e+01,5.5658900e+01]
 vdw_dftd3_c6(5201:5300)=[&
& 5.3624300e+01,5.0147300e+01,4.8053500e+01,4.5084800e+01,4.2063400e+01, &
& 1.1797180e+02,1.1446760e+02,1.0480950e+02,5.9137900e+01,1.0336160e+02, &
& 9.9833100e+01,9.7478900e+01,9.5284600e+01,9.3341000e+01,7.7874200e+01, &
& 8.2715400e+01,8.0460700e+01,8.4947800e+01,8.3182700e+01,8.1643700e+01, &
& 8.0457500e+01,7.0552200e+01,7.2088700e+01,6.7959000e+01,5.9575800e+01, &
& 6.1094900e+01,5.7011100e+01,5.3521400e+01,4.6144200e+01,4.3746800e+01, &
& 4.5120500e+01,5.8912900e+01,5.9109700e+01,5.6500700e+01,5.5261400e+01, &
& 5.2703300e+01,4.9892700e+01,1.1543210e+02,1.1600620e+02,1.0653030e+02, &
& 1.0090290e+02,9.8576800e+01,9.5662500e+01,9.5732200e+01,9.3161400e+01, &
& 6.3000000e+00,1.7173300e+01,2.2541300e+01,1.9202900e+01,1.5685700e+01, &
& 1.1947000e+01,9.4048000e+00,7.1341000e+00,2.0756700e+01,1.2728700e+01, &
& 4.2571610e+02,2.1865720e+02,1.3695360e+02,8.7395600e+01,5.8485300e+01, &
& 4.2891100e+01,3.1606000e+01,2.3774500e+01,5.0543630e+02,3.5591800e+02, &
& 3.1645880e+02,2.3713260e+02,1.7692050e+02,1.4254300e+02,1.1310250e+02, &
& 9.0173500e+01,8.4082990e+02,6.4831050e+02,5.2709170e+02,5.0123930e+02, &
& 4.5480300e+02,3.5715080e+02,3.8634260e+02,3.0174010e+02,3.1417180e+02, &
& 3.2682710e+02,2.4981970e+02,2.4939250e+02,2.9840070e+02,2.5314330e+02, &
& 2.0818020e+02,1.8247440e+02,1.5582020e+02,1.3243480e+02,9.3630060e+02, &
& 7.7398540e+02,6.5879630e+02,5.8124800e+02,5.2340580e+02,3.9500580e+02, &
& 4.4452690e+02,3.3022730e+02,3.6142870e+02,3.3243920e+02,2.7794280e+02]
 vdw_dftd3_c6(5301:5400)=[&
& 2.8984740e+02,3.7337940e+02,3.3470230e+02,2.8898880e+02,2.6288460e+02, &
& 2.3260530e+02,2.0479860e+02,1.1389256e+03,9.9536600e+02,8.5007680e+02, &
& 3.4313820e+02,8.7351340e+02,8.3492630e+02,8.1293500e+02,7.9283360e+02, &
& 7.7497190e+02,5.8968210e+02,6.9502110e+02,6.6665150e+02,6.9407420e+02, &
& 6.7884860e+02,6.6485800e+02,6.5799220e+02,5.4208720e+02,5.1742950e+02, &
& 4.6327240e+02,3.8490910e+02,3.8879710e+02,3.4639970e+02,3.1281320e+02, &
& 2.5572890e+02,2.3740440e+02,2.4279190e+02,3.7972090e+02,3.6256490e+02, &
& 3.2393500e+02,3.0383030e+02,2.7473150e+02,2.4706350e+02,1.0521318e+03, &
& 9.6974210e+02,8.3486500e+02,7.2693870e+02,7.3265540e+02,7.0879360e+02, &
& 7.4391900e+02,7.1791000e+02,3.4995200e+01,1.2582450e+02,1.5832680e+02, &
& 1.1340160e+02,8.2014400e+01,5.4445400e+01,3.8122700e+01,2.4956000e+01, &
& 1.8610520e+02,3.1991300e+01,1.9464800e+01,6.7003300e+02,3.4026020e+02, &
& 2.1221800e+02,1.3492270e+02,8.9964600e+01,6.5760900e+01,4.8286500e+01, &
& 3.6191100e+01,7.9475970e+02,5.5459670e+02,4.9208290e+02,3.6758110e+02, &
& 2.7352680e+02,2.1994890e+02,1.7413990e+02,1.3851900e+02,1.3273176e+03, &
& 1.0137695e+03,8.2289570e+02,7.8169930e+02,7.0876840e+02,5.5661270e+02, &
& 6.0144990e+02,4.6964370e+02,4.8817150e+02,5.0815850e+02,3.8842090e+02, &
& 3.8688840e+02,4.6356910e+02,3.9225330e+02,3.2187320e+02,2.8171730e+02, &
& 2.4015650e+02,2.0374730e+02,1.4776556e+03,1.2109593e+03,1.0281365e+03, &
& 9.0577500e+02,8.1492890e+02,6.1402650e+02,6.9144970e+02,5.1267590e+02]
 vdw_dftd3_c6(5401:5500)=[&
& 5.6100100e+02,5.1564570e+02,4.3128070e+02,4.4915670e+02,5.7992340e+02, &
& 5.1865180e+02,4.4692300e+02,4.0607280e+02,3.5880520e+02,3.1545810e+02, &
& 1.7990180e+03,1.5595050e+03,1.3282510e+03,5.3086500e+02,1.3672714e+03, &
& 1.3059000e+03,1.2712883e+03,1.2396749e+03,1.2115825e+03,9.1963290e+02, &
& 1.0898214e+03,1.0451961e+03,1.0842313e+03,1.0603386e+03,1.0383601e+03, &
& 1.0277356e+03,8.4553860e+02,8.0500140e+02,7.1971520e+02,5.9743440e+02, &
& 6.0309070e+02,5.3662210e+02,4.8407160e+02,3.9524010e+02,3.6670170e+02, &
& 3.7485520e+02,5.8953950e+02,5.6172980e+02,5.0092230e+02,4.6935230e+02, &
& 4.2385000e+02,3.8067930e+02,1.6575070e+03,1.5173457e+03,1.3028376e+03, &
& 1.1313898e+03,1.1418668e+03,1.1045083e+03,1.1608135e+03,1.1198466e+03, &
& 5.4102100e+01,1.9570570e+02,2.4622270e+02,1.7556460e+02,1.2666570e+02, &
& 8.3770300e+01,5.8430300e+01,3.7988400e+01,2.8960820e+02,4.5127080e+02, &
& 3.3104800e+01,2.0515800e+01,6.2753090e+02,3.3473350e+02,2.1388480e+02, &
& 1.3832540e+02,9.3354200e+01,6.8798300e+01,5.0857700e+01,3.8315400e+01, &
& 7.4634190e+02,5.4102850e+02,4.8623650e+02,3.6962320e+02,2.7892720e+02, &
& 2.2621560e+02,1.8054300e+02,1.4457980e+02,1.2332237e+03,9.7304930e+02, &
& 7.9522060e+02,7.5945320e+02,6.9091560e+02,5.4224590e+02,5.8914750e+02, &
& 4.6004960e+02,4.8229900e+02,5.0035250e+02,3.8199230e+02,3.8514490e+02, &
& 4.5995440e+02,3.9466500e+02,3.2772290e+02,2.8884850e+02,2.4796920e+02, &
& 2.1169490e+02,1.3752558e+03,1.1602539e+03,9.9637570e+02,8.8385410e+02]
 vdw_dftd3_c6(5501:5600)=[&
& 7.9860940e+02,6.0591350e+02,6.8054270e+02,5.0852890e+02,5.5700430e+02, &
& 5.1341630e+02,4.2799620e+02,4.4880710e+02,5.7434200e+02,5.1984240e+02, &
& 4.5284940e+02,4.1405440e+02,3.6827920e+02,3.2576250e+02,1.6731922e+03, &
& 1.4872964e+03,1.2815229e+03,5.3656260e+02,1.3086041e+03,1.2526928e+03, &
& 1.2202246e+03,1.1905003e+03,1.1641089e+03,8.9395150e+02,1.0381047e+03, &
& 9.9738370e+02,1.0450038e+03,1.0223674e+03,1.0016784e+03,9.9098080e+02, &
& 8.2160050e+02,7.9184560e+02,7.1274220e+02,5.9416160e+02,6.0157800e+02, &
& 5.3838550e+02,4.8789840e+02,3.9997680e+02,3.7172100e+02,3.8105900e+02, &
& 5.8469720e+02,5.6268750e+02,5.0686630e+02,4.7758620e+02,4.3409690e+02, &
& 3.9211690e+02,1.5558848e+03,1.4554241e+03,1.2633724e+03,1.1100628e+03, &
& 1.1130738e+03,1.0770005e+03,1.1238008e+03,1.0856846e+03,5.5263600e+01, &
& 1.9307070e+02,2.4376120e+02,1.7772540e+02,1.2969220e+02,8.6872800e+01, &
& 6.1217500e+01,4.0322000e+01,2.8422510e+02,4.4173220e+02,4.3775750e+02, &
& 2.7498900e+01,1.7635600e+01,4.5095830e+02,2.5658810e+02,1.7018910e+02, &
& 1.1320460e+02,7.8019000e+01,5.8355700e+01,4.3685500e+01,3.3241600e+01, &
& 5.3834120e+02,4.0991740e+02,3.7552330e+02,2.9299940e+02,2.2606570e+02, &
& 1.8600510e+02,1.5052310e+02,1.2200120e+02,8.8126240e+02,7.2251110e+02, &
& 5.9567250e+02,5.7349880e+02,5.2428080e+02,4.1188790e+02,4.5022340e+02, &
& 3.5224080e+02,3.7301250e+02,3.8500480e+02,2.9414060e+02,3.0117090e+02, &
& 3.5794110e+02,3.1339110e+02,2.6507110e+02,2.3633230e+02,2.0529760e+02]
 vdw_dftd3_c6(5601:5700)=[&
& 1.7718110e+02,9.8604380e+02,8.6034210e+02,7.5038170e+02,6.7218340e+02, &
& 6.1131470e+02,4.6910920e+02,5.2464790e+02,3.9698620e+02,4.3472510e+02, &
& 4.0236620e+02,3.3466730e+02,3.5359760e+02,4.4659370e+02,4.1089600e+02, &
& 3.6378460e+02,3.3592400e+02,3.0198520e+02,2.6983720e+02,1.2005803e+03, &
& 1.0973980e+03,9.6023320e+02,4.2980740e+02,9.7039570e+02,9.3129170e+02, &
& 9.0779700e+02,8.8621930e+02,8.6708440e+02,6.7748060e+02,7.6730550e+02, &
& 7.3939750e+02,7.8137480e+02,7.6476680e+02,7.4974130e+02,7.4120300e+02, &
& 6.2181690e+02,6.0906080e+02,5.5365420e+02,4.6519420e+02,4.7279230e+02, &
& 4.2679800e+02,3.8949880e+02,3.2171830e+02,2.9988580e+02,3.0842630e+02, &
& 4.5658720e+02,4.4500360e+02,4.0670170e+02,3.8646750e+02,3.5486660e+02, &
& 3.2354110e+02,1.1298915e+03,1.0825018e+03,9.5314260e+02,8.5147580e+02, &
& 8.4679590e+02,8.1968560e+02,8.4655430e+02,8.1936980e+02,4.4903000e+01, &
& 1.4874780e+02,1.8910900e+02,1.4251230e+02,1.0608060e+02,7.2637300e+01, &
& 5.2131100e+01,3.5129000e+01,2.1750490e+02,3.3702500e+02,3.3965160e+02, &
& 2.7026500e+02,2.3090800e+01,1.5261900e+01,3.3966930e+02,2.0222820e+02, &
& 1.3807740e+02,9.3966300e+01,6.5912700e+01,4.9937400e+01,3.7803500e+01, &
& 2.9028000e+01,4.0671500e+02,3.2041760e+02,2.9780760e+02,2.3698640e+02, &
& 1.8606840e+02,1.5492290e+02,1.2683120e+02,1.0386530e+02,6.6311840e+02, &
& 5.5742550e+02,4.6235110e+02,4.4795900e+02,4.1101210e+02,3.2359140e+02, &
& 3.5484320e+02,2.7840200e+02,2.9657890e+02,3.0491450e+02,2.3348980e+02]
 vdw_dftd3_c6(5701:5800)=[&
& 2.4143800e+02,2.8580720e+02,2.5397440e+02,2.1789200e+02,1.9608060e+02, &
& 1.7199100e+02,1.4979330e+02,7.4424750e+02,6.6356560e+02,5.8524670e+02, &
& 5.2808410e+02,4.8270360e+02,3.7394270e+02,4.1670800e+02,3.1859120e+02, &
& 3.4844490e+02,3.2352760e+02,2.6912200e+02,2.8548020e+02,3.5690690e+02, &
& 3.3217220e+02,2.9765980e+02,2.7698090e+02,2.5110810e+02,2.2622110e+02, &
& 9.0729930e+02,8.4401200e+02,7.4659930e+02,3.5115850e+02,7.4914380e+02, &
& 7.2014500e+02,7.0229590e+02,6.8586200e+02,6.7130090e+02,5.3154240e+02, &
& 5.9201350e+02,5.7174810e+02,6.0649400e+02,5.9374030e+02,5.8229560e+02, &
& 5.7532620e+02,4.8706190e+02,4.8246160e+02,4.4186160e+02,3.7386540e+02, &
& 3.8091200e+02,3.4613260e+02,3.1760850e+02,2.6410910e+02,2.4683080e+02, &
& 2.5431280e+02,3.6658420e+02,3.6026310e+02,3.3271860e+02,3.1818410e+02, &
& 2.9447100e+02,2.7046770e+02,8.6138240e+02,8.3751280e+02,7.4476320e+02, &
& 6.7356880e+02,6.6625820e+02,6.4514050e+02,6.6128050e+02,6.4087610e+02, &
& 3.7023600e+01,1.1774720e+02,1.5060680e+02,1.1638180e+02,8.8076900e+01, &
& 6.1428600e+01,4.4777000e+01,3.0768800e+01,1.7144690e+02,2.6503780e+02, &
& 2.7035800e+02,2.1928630e+02,1.8062380e+02,1.9866700e+01,1.3459800e+01, &
& 2.7130110e+02,1.6618070e+02,1.1581680e+02,8.0180400e+01,5.7025500e+01, &
& 4.3654800e+01,3.3355000e+01,2.5810700e+01,3.2559690e+02,2.6194370e+02, &
& 2.4585680e+02,1.9832660e+02,1.5769710e+02,1.3249220e+02,1.0944860e+02, &
& 9.0371600e+01,5.3051920e+02,4.5237450e+02,3.7661420e+02,3.6651510e+02]
 vdw_dftd3_c6(5801:5900)=[&
& 3.3712500e+02,2.6607620e+02,2.9212570e+02,2.2987760e+02,2.4559880e+02, &
& 2.5180970e+02,1.9338650e+02,2.0107040e+02,2.3725630e+02,2.1294370e+02, &
& 1.8454670e+02,1.6722640e+02,1.4776790e+02,1.2961230e+02,5.9687630e+02, &
& 5.3867240e+02,4.7852040e+02,4.3390360e+02,3.9804190e+02,3.1058610e+02, &
& 3.4514340e+02,2.6595280e+02,2.9041460e+02,2.7025240e+02,2.2511810e+02, &
& 2.3917840e+02,2.9682140e+02,2.7827870e+02,2.5143440e+02,2.3525190e+02, &
& 2.1460020e+02,1.9452370e+02,7.2846530e+02,6.8423530e+02,6.0944200e+02, &
& 2.9645810e+02,6.0891190e+02,5.8591340e+02,5.7153970e+02,5.5828160e+02, &
& 5.4653950e+02,4.3677850e+02,4.8158940e+02,4.6577580e+02,4.9452850e+02, &
& 4.8417910e+02,4.7494430e+02,4.6905670e+02,3.9959520e+02,3.9859150e+02, &
& 3.6694770e+02,3.1220810e+02,3.1855560e+02,2.9082630e+02,2.6791140e+02, &
& 2.2401000e+02,2.0979980e+02,2.1632530e+02,3.0615090e+02,3.0235050e+02, &
& 2.8117730e+02,2.7007790e+02,2.5135500e+02,2.3212580e+02,6.9553530e+02, &
& 6.8165580e+02,6.0993410e+02,5.5624920e+02,5.4849710e+02,5.3125540e+02, &
& 5.4187030e+02,5.2557770e+02,3.1412700e+01,9.7084900e+01,1.2475120e+02, &
& 9.8113900e+01,7.5203900e+01,5.3208600e+01,3.9268300e+01,2.7411700e+01, &
& 1.4105990e+02,2.1766290e+02,2.2379420e+02,1.8394830e+02,1.5318230e+02, &
& 1.3099650e+02,1.6527300e+01,1.1509200e+01,2.0877430e+02,1.3164980e+02, &
& 9.3774300e+01,6.6135400e+01,4.7754000e+01,3.6978300e+01,2.8546800e+01, &
& 2.2283200e+01,2.5123260e+02,2.0645210e+02,1.9576470e+02,1.6018790e+02]
 vdw_dftd3_c6(5901:6000)=[&
& 1.2909560e+02,1.0952700e+02,9.1366700e+01,7.6125600e+01,4.0951050e+02, &
& 3.5409350e+02,2.9590280e+02,2.8935120e+02,2.6685570e+02,2.1129280e+02, &
& 2.3214470e+02,1.8336610e+02,1.9637550e+02,2.0075710e+02,1.5477560e+02, &
& 1.6173370e+02,1.9011390e+02,1.7239330e+02,1.5099990e+02,1.3784430e+02, &
& 1.2277630e+02,1.0852260e+02,4.6202520e+02,4.2188670e+02,3.7756420e+02, &
& 3.4413180e+02,3.1691100e+02,2.4926710e+02,2.7614550e+02,2.1463800e+02, &
& 2.3390310e+02,2.1819030e+02,1.8215470e+02,1.9372730e+02,2.3849880e+02, &
& 2.2523810e+02,2.0525120e+02,1.9314490e+02,1.7734580e+02,1.6181190e+02, &
& 5.6460060e+02,5.3526780e+02,4.8012440e+02,2.4192230e+02,4.7771970e+02, &
& 4.6012500e+02,4.4895170e+02,4.3862430e+02,4.2948100e+02,3.4664610e+02, &
& 3.7839840e+02,3.6651660e+02,3.8919850e+02,3.8108470e+02,3.7388780e+02, &
& 3.6907690e+02,3.1653270e+02,3.1792390e+02,2.9429820e+02,2.5196160e+02, &
& 2.5744470e+02,2.3620650e+02,2.1851340e+02,1.8384350e+02,1.7259550e+02, &
& 1.7806050e+02,2.4721900e+02,2.4529500e+02,2.2972670e+02,2.2165670e+02, &
& 2.0750080e+02,1.9273160e+02,5.4229530e+02,5.3547060e+02,4.8217720e+02, &
& 4.4362620e+02,4.3615260e+02,4.2258040e+02,4.2886190e+02,4.1631150e+02, &
& 2.5741400e+01,7.7202300e+01,9.9699500e+01,7.9883300e+01,6.2086400e+01, &
& 4.4621700e+01,3.3380100e+01,2.3705900e+01,1.1198290e+02,1.7241480e+02, &
& 1.7871270e+02,1.4895590e+02,1.2549300e+02,1.0828930e+02,9.0398500e+01, &
& 2.9300000e+01,1.8665500e+01,5.6029540e+02,2.9222240e+02,1.8695660e+02]
 vdw_dftd3_c6(6001:6100)=[&
& 1.2192920e+02,8.3215300e+01,6.1994800e+01,4.6352000e+01,3.5296900e+01, &
& 6.6603570e+02,4.7376610e+02,4.2489260e+02,3.2282330e+02,2.4457290e+02, &
& 1.9946720e+02,1.6033040e+02,1.2942830e+02,1.1122277e+03,8.6017710e+02, &
& 7.0072810e+02,6.6875460e+02,6.0799510e+02,4.7897440e+02,5.1805830e+02, &
& 4.0608760e+02,4.2338090e+02,4.3928450e+02,3.3711170e+02,3.3789860e+02, &
& 4.0307450e+02,3.4544340e+02,2.8755350e+02,2.5436700e+02,2.1946470e+02, &
& 1.8848190e+02,1.2408631e+03,1.0279671e+03,8.7903460e+02,7.7852380e+02, &
& 7.0330600e+02,5.3463030e+02,5.9993020e+02,4.4930300e+02,4.9058070e+02, &
& 4.5224640e+02,3.7908360e+02,3.9551350e+02,5.0568380e+02,4.5646130e+02, &
& 3.9784160e+02,3.6440900e+02,3.2510680e+02,2.8872750e+02,1.5118898e+03, &
& 1.3222619e+03,1.1337749e+03,4.7224550e+02,1.1622835e+03,1.1112638e+03, &
& 1.0820865e+03,1.0553891e+03,1.0316716e+03,7.9078540e+02,9.2736720e+02, &
& 8.9050560e+02,9.2461970e+02,9.0432570e+02,8.8575380e+02,8.7628990e+02, &
& 7.2564410e+02,6.9618310e+02,6.2634420e+02,5.2368920e+02,5.2962490e+02, &
& 4.7425390e+02,4.3019640e+02,3.5414110e+02,3.2964710e+02,3.3733350e+02, &
& 5.1695330e+02,4.9569810e+02,4.4630330e+02,4.2086840e+02,3.8335330e+02, &
& 3.4734370e+02,1.4001686e+03,1.2915073e+03,1.1159288e+03,9.7819280e+02, &
& 9.8391310e+02,9.5206270e+02,9.9535270e+02,9.6105430e+02,4.8449900e+01, &
& 1.6874120e+02,2.1344440e+02,1.5577810e+02,1.1457230e+02,7.7631700e+01, &
& 5.5403200e+01,3.7198200e+01,2.4922000e+02,3.8733490e+02,3.8264220e+02]
 vdw_dftd3_c6(6101:6200)=[&
& 2.9694090e+02,2.3721910e+02,1.9738060e+02,1.5873300e+02,3.3802070e+02, &
& 5.2987100e+01,3.1879600e+01,1.3100314e+03,6.0256610e+02,3.6268620e+02, &
& 2.2596290e+02,1.4897930e+02,1.0827180e+02,7.9221800e+01,5.9277900e+01, &
& 1.5457064e+03,9.9724860e+02,8.6695170e+02,6.3004560e+02,4.6026350e+02, &
& 3.6677110e+02,2.8821820e+02,2.2808920e+02,2.6610830e+03,1.8864221e+03, &
& 1.5101427e+03,1.4223244e+03,1.2821610e+03,1.0115955e+03,1.0791532e+03, &
& 8.4546860e+02,8.6250360e+02,9.0260140e+02,6.9450580e+02,6.7502990e+02, &
& 8.1349430e+02,6.7269170e+02,5.4301230e+02,4.7148220e+02,3.9902260e+02, &
& 3.3664830e+02,2.9580410e+03,2.2654155e+03,1.8821338e+03,1.6379667e+03, &
& 1.4638890e+03,1.0926733e+03,1.2349316e+03,9.0575150e+02,9.8637270e+02, &
& 9.0244050e+02,7.6261740e+02,7.8166000e+02,1.0241614e+03,8.9641550e+02, &
& 7.5979880e+02,6.8470730e+02,6.0026980e+02,5.2435470e+02,3.6192660e+03, &
& 2.9513362e+03,2.4567601e+03,9.0740120e+02,2.5715319e+03,2.4420495e+03, &
& 2.3740445e+03,2.3122147e+03,2.2571993e+03,1.6806338e+03,2.0816173e+03, &
& 1.9928728e+03,2.0062958e+03,1.9602391e+03,1.9174640e+03,1.8991306e+03, &
& 1.5446852e+03,1.4351503e+03,1.2687518e+03,1.0493296e+03,1.0529922e+03, &
& 9.2845210e+02,8.3187520e+02,6.7699020e+02,6.2708530e+02,6.3711120e+02, &
& 1.0430556e+03,9.7428450e+02,8.5466400e+02,7.9444910e+02,7.1141430e+02, &
& 6.3479210e+02,3.2712563e+03,2.8409447e+03,2.3867637e+03,2.0296245e+03, &
& 2.0761172e+03,2.0066915e+03,2.1352269e+03,2.0540131e+03,9.0792700e+01]
 vdw_dftd3_c6(6201:6300)=[&
& 3.4561200e+02,4.3402840e+02,2.9872640e+02,2.1289570e+02,1.3902190e+02, &
& 9.6158400e+01,6.2058000e+01,5.1669170e+02,8.0921820e+02,7.7508320e+02, &
& 5.7629620e+02,4.4601700e+02,3.6319510e+02,2.8549050e+02,6.8899110e+02, &
& 1.5251891e+03,2.9995300e+01,1.9384100e+01,5.0482680e+02,2.8244200e+02, &
& 1.8614190e+02,1.2363790e+02,8.5340400e+01,6.4007200e+01,4.8093600e+01, &
& 3.6746800e+01,6.0249290e+02,4.5276000e+02,4.1290580e+02,3.2058960e+02, &
& 2.4673580e+02,2.0296720e+02,1.6436550e+02,1.3344040e+02,9.8858810e+02, &
& 8.0323800e+02,6.6067800e+02,6.3504570e+02,5.7995840e+02,4.5620230e+02, &
& 4.9734740e+02,3.8959740e+02,4.1105720e+02,4.2466930e+02,3.2508500e+02, &
& 3.3123100e+02,3.9353000e+02,3.4316790e+02,2.8955820e+02,2.5800840e+02, &
& 2.2413750e+02,1.9358680e+02,1.1054608e+03,9.5722710e+02,8.3166790e+02, &
& 7.4343180e+02,6.7533140e+02,5.1773400e+02,5.7916230e+02,4.3784320e+02, &
& 4.7895980e+02,4.4308160e+02,3.6949330e+02,3.8918590e+02,4.9219040e+02, &
& 4.5113400e+02,3.9831520e+02,3.6739080e+02,3.3004180e+02,2.9487060e+02, &
& 1.3445274e+03,1.2227426e+03,1.0658140e+03,4.7118540e+02,1.0799647e+03, &
& 1.0362055e+03,1.0098893e+03,9.8572640e+02,9.6428550e+02,7.5078480e+02, &
& 8.5514710e+02,8.2284200e+02,8.6811040e+02,8.4953940e+02,8.3269610e+02, &
& 8.2330890e+02,6.8862420e+02,6.7192580e+02,6.0975150e+02,5.1221050e+02, &
& 5.2005780e+02,4.6895650e+02,4.2768450e+02,3.5348450e+02,3.2959850e+02, &
& 3.3854780e+02,5.0363930e+02,4.8925410e+02,4.4592310e+02,4.2320630e+02]
 vdw_dftd3_c6(6301:6400)=[&
& 3.8821220e+02,3.5380050e+02,1.2630191e+03,1.2039859e+03,1.0563541e+03, &
& 9.4062410e+02,9.3765790e+02,9.0769560e+02,9.3986910e+02,9.0928950e+02, &
& 4.8975300e+01,1.6364460e+02,2.0781070e+02,1.5582280e+02,1.1597260e+02, &
& 7.9538300e+01,5.7261100e+01,3.8809700e+01,2.3997140e+02,3.7177680e+02, &
& 3.7318450e+02,2.9561010e+02,2.3937010e+02,2.0076220e+02,1.6268060e+02, &
& 3.2770920e+02,6.4070760e+02,3.2450680e+02,2.9000900e+01,1.8808400e+01, &
& 4.8829310e+02,2.7202010e+02,1.7950770e+02,1.1943760e+02,8.2583400e+01, &
& 6.2030900e+01,4.6676500e+01,3.5711700e+01,5.8252700e+02,4.3610910e+02, &
& 3.9786180e+02,3.0910810e+02,2.3816440e+02,1.9611280e+02,1.5898920e+02, &
& 1.2921800e+02,9.6033760e+02,7.7468630e+02,6.3687060e+02,6.1222250e+02, &
& 5.5911570e+02,4.4007620e+02,4.7949690e+02,3.7582420e+02,3.9627480e+02, &
& 4.0935550e+02,3.1360260e+02,3.1938340e+02,3.7941090e+02,3.3097100e+02, &
& 2.7949900e+02,2.4922580e+02,2.1669210e+02,1.8732310e+02,1.0741973e+03, &
& 9.2365570e+02,8.0203240e+02,7.1686090e+02,6.5128470e+02,4.9950110e+02, &
& 5.5872110e+02,4.2257080e+02,4.6203140e+02,4.2745770e+02,3.5672450e+02, &
& 3.7552610e+02,4.7481120e+02,4.3519480e+02,3.8444870e+02,3.5477930e+02, &
& 3.1891710e+02,2.8513250e+02,1.3086418e+03,1.1809424e+03,1.0284169e+03, &
& 4.5482000e+02,1.0433187e+03,1.0002155e+03,9.7472270e+02,9.5132570e+02, &
& 9.3056540e+02,7.2458860e+02,8.2779700e+02,7.9723640e+02,8.3746180e+02, &
& 8.1949730e+02,8.0320320e+02,7.9414040e+02,6.6477270e+02,6.4787460e+02]
 vdw_dftd3_c6(6401:6500)=[&
& 5.8798130e+02,4.9423570e+02,5.0177330e+02,4.5257260e+02,4.1285580e+02, &
& 3.4145400e+02,3.1846190e+02,3.2707700e+02,4.8621160e+02,4.7215520e+02, &
& 4.3048050e+02,4.0869510e+02,3.7510290e+02,3.4205750e+02,1.2263970e+03, &
& 1.1623660e+03,1.0191422e+03,9.0723670e+02,9.0483310e+02,8.7587480e+02, &
& 9.0696160e+02,8.7736450e+02,4.7275500e+01,1.5766670e+02,2.0036690e+02, &
& 1.5035200e+02,1.1205230e+02,7.6983400e+01,5.5516000e+01,3.7722200e+01, &
& 2.3126120e+02,3.5836020e+02,3.5964970e+02,2.8509570e+02,2.3107730e+02, &
& 1.9398450e+02,1.5736040e+02,3.1625180e+02,6.1943010e+02,3.1292020e+02, &
& 3.0199880e+02,2.8406700e+01,1.8435900e+01,4.7914020e+02,2.6670500e+02, &
& 1.7589090e+02,1.1700980e+02,8.0914400e+01,6.0793300e+01,4.5763000e+01, &
& 3.5029000e+01,5.7164550e+02,4.2771560e+02,3.9004320e+02,3.0289850e+02, &
& 2.3331760e+02,1.9211130e+02,1.5575070e+02,1.2660270e+02,9.4198370e+02, &
& 7.6007180e+02,6.2478010e+02,6.0052210e+02,5.4839080e+02,4.3166830e+02, &
& 4.7025290e+02,3.6861360e+02,3.8857290e+02,4.0143290e+02,3.0757630e+02, &
& 3.1312860e+02,3.7194270e+02,3.2434350e+02,2.7383520e+02,2.4415540e+02, &
& 2.1227920e+02,1.8351710e+02,1.0535834e+03,9.0623640e+02,7.8672550e+02, &
& 7.0307840e+02,6.3870140e+02,4.8981600e+02,5.4789060e+02,4.1435750e+02, &
& 4.5302620e+02,4.1911340e+02,3.4982880e+02,3.6818710e+02,4.6556470e+02, &
& 4.2659330e+02,3.7675440e+02,3.4763740e+02,3.1247140e+02,2.7936120e+02, &
& 1.2831193e+03,1.1586662e+03,1.0088336e+03,4.4575700e+02,1.0235395e+03]
 vdw_dftd3_c6(6501:6600)=[&
& 9.8134570e+02,9.5633210e+02,9.3337380e+02,9.1300140e+02,7.1072370e+02, &
& 8.1199050e+02,7.8182720e+02,8.2162740e+02,8.0399890e+02,7.8800640e+02, &
& 7.7912190e+02,6.5197940e+02,6.3529960e+02,5.7649460e+02,4.8455830e+02, &
& 4.9191200e+02,4.4364130e+02,4.0468640e+02,3.3471530e+02,3.1218730e+02, &
& 3.2060120e+02,4.7675960e+02,4.6287410e+02,4.2191810e+02,4.0051720e+02, &
& 3.6755880e+02,3.3516100e+02,1.2027024e+03,1.1403430e+03,9.9965520e+02, &
& 8.8972590e+02,8.8748800e+02,8.5909970e+02,8.8977190e+02,8.6071780e+02, &
& 4.6308900e+01,1.5457620e+02,1.9640270e+02,1.4731530e+02,1.0977970e+02, &
& 7.5430300e+01,5.4410900e+01,3.6996200e+01,2.2678860e+02,3.5139280e+02, &
& 3.5255750e+02,2.7935630e+02,2.2637670e+02,1.9002990e+02,1.5415760e+02, &
& 3.1011080e+02,6.0755710e+02,3.0673890e+02,2.9600290e+02,2.9014360e+02, &
& 2.8056200e+01,1.8175400e+01,4.8059440e+02,2.6540880e+02,1.7437190e+02, &
& 1.1571820e+02,7.9902400e+01,5.9982900e+01,4.5129000e+01,3.4535000e+01, &
& 5.7314360e+02,4.2622760e+02,3.8786950e+02,3.0039280e+02,2.3090900e+02, &
& 1.8990520e+02,1.5380570e+02,1.2492650e+02,9.4587000e+02,7.5936840e+02, &
& 6.2353320e+02,5.9879810e+02,5.4652440e+02,4.3023530e+02,4.6829610e+02, &
& 3.6707850e+02,3.8644860e+02,3.9945300e+02,3.0611810e+02,3.1106030e+02, &
& 3.6963020e+02,3.2164040e+02,2.7107700e+02,2.4145850e+02,2.0973930e+02, &
& 1.8118190e+02,1.0575718e+03,9.0561870e+02,7.8474890e+02,7.0054320e+02, &
& 6.3596390e+02,4.8719110e+02,5.4517240e+02,4.1181870e+02,4.5019430e+02]
 vdw_dftd3_c6(6601:6700)=[&
& 4.1632300e+02,3.4767860e+02,3.6555310e+02,4.6283040e+02,4.2332080e+02, &
& 3.7325910e+02,3.4409820e+02,3.0900630e+02,2.7604060e+02,1.2879616e+03, &
& 1.1586672e+03,1.0069636e+03,4.4178090e+02,1.0229773e+03,9.8048290e+02, &
& 9.5540420e+02,9.3239460e+02,9.1197350e+02,7.0857690e+02,8.1204950e+02, &
& 7.8162600e+02,8.2030710e+02,8.0266090e+02,7.8663360e+02,7.7782320e+02, &
& 6.5005720e+02,6.3220990e+02,5.7309450e+02,4.8138140e+02,4.8847190e+02, &
& 4.4016550e+02,4.0125220e+02,3.3169650e+02,3.0930750e+02,3.1751490e+02, &
& 4.7384960e+02,4.5937430e+02,4.1810440e+02,3.9657360e+02,3.6360680e+02, &
& 3.3130140e+02,1.2054436e+03,1.1392761e+03,9.9702490e+02,8.8576610e+02, &
& 8.8442800e+02,8.5610390e+02,8.8770660e+02,8.5853020e+02,4.5817300e+01, &
& 1.5375390e+02,1.9523170e+02,1.4594770e+02,1.0858200e+02,7.4489000e+01, &
& 5.3673300e+01,3.6459400e+01,2.2578370e+02,3.4993280e+02,3.5042960e+02, &
& 2.7695120e+02,2.2403030e+02,1.8785700e+02,1.5223830e+02,3.0839010e+02, &
& 6.0707000e+02,3.0430940e+02,2.9365980e+02,2.8786570e+02,2.8569050e+02, &
& 2.6181000e+01,1.7152600e+01,4.1608360e+02,2.3896060e+02,1.5984920e+02, &
& 1.0733820e+02,7.4699800e+01,5.6362200e+01,4.2576400e+01,3.2680600e+01, &
& 4.9735820e+02,3.8129680e+02,3.5045220e+02,2.7492570e+02,2.1341010e+02, &
& 1.7651650e+02,1.4370010e+02,1.1720570e+02,8.1424680e+02,6.7083370e+02, &
& 5.5379310e+02,5.3412660e+02,4.8879470e+02,3.8469310e+02,4.2042070e+02, &
& 3.2962730e+02,3.4919410e+02,3.6001720e+02,2.7572060e+02,2.8266670e+02]
 vdw_dftd3_c6(6701:6800)=[&
& 3.3506150e+02,2.9451840e+02,2.5028310e+02,2.2398570e+02,1.9545360e+02, &
& 1.6950970e+02,9.1194330e+02,7.9901250e+02,6.9867600e+02,6.2707900e+02, &
& 5.7118620e+02,4.3993260e+02,4.9132100e+02,3.7334420e+02,4.0836770e+02, &
& 3.7842950e+02,3.1534820e+02,3.3315180e+02,4.1907070e+02,3.8661840e+02, &
& 3.4352050e+02,3.1804850e+02,2.8687200e+02,2.5728030e+02,1.1103588e+03, &
& 1.0186688e+03,8.9350560e+02,4.0587370e+02,9.0204820e+02,8.6604420e+02, &
& 8.4428560e+02,8.2428300e+02,8.0654320e+02,6.3257430e+02,7.1375520e+02, &
& 6.8811250e+02,7.2723780e+02,7.1180010e+02,6.9785650e+02,6.8978470e+02, &
& 5.8012870e+02,5.6940560e+02,5.1879050e+02,4.3720880e+02,4.4459970e+02, &
& 4.0230970e+02,3.6794540e+02,3.0505010e+02,2.8480180e+02,2.9291770e+02, &
& 4.2959510e+02,4.1941490e+02,3.8444960e+02,3.6605470e+02,3.3709280e+02, &
& 3.0829470e+02,1.0472098e+03,1.0062859e+03,8.8823000e+02,7.9631050e+02, &
& 7.9125770e+02,7.6609840e+02,7.8992350e+02,7.6480610e+02,4.2392700e+01, &
& 1.3874910e+02,1.7668170e+02,1.3421360e+02,1.0064630e+02,6.9621500e+01, &
& 5.0485800e+01,3.4548100e+01,2.0296760e+02,3.1401190e+02,3.1732870e+02, &
& 2.5388000e+02,2.0709950e+02,1.7457750e+02,1.4221510e+02,2.7872590e+02, &
& 5.3581490e+02,2.7819650e+02,2.6846570e+02,2.6311270e+02,2.6077730e+02, &
& 2.3955540e+02,2.4345200e+01,1.6056000e+01,3.7921170e+02,2.1947750e+02, &
& 1.4762980e+02,9.9588000e+01,6.9563800e+01,5.2635100e+01,3.9862800e+01, &
& 3.0664800e+01,4.5354440e+02,3.4971160e+02,3.2227810e+02,2.5376380e+02]
 vdw_dftd3_c6(6801:6900)=[&
& 1.9765690e+02,1.6387940e+02,1.3373240e+02,1.0931550e+02,7.4225200e+02, &
& 6.1392330e+02,5.0734010e+02,4.8989960e+02,4.4862620e+02,3.5327090e+02, &
& 3.8626020e+02,3.0304800e+02,3.2134480e+02,3.3106140e+02,2.5370380e+02, &
& 2.6053410e+02,3.0855290e+02,2.7197110e+02,2.3176180e+02,2.0779480e+02, &
& 1.8168360e+02,1.5786610e+02,8.3179600e+02,7.3123300e+02,6.4066410e+02, &
& 5.7577520e+02,5.2495880e+02,4.0508170e+02,4.5208310e+02,3.4423220e+02, &
& 3.7640450e+02,3.4902580e+02,2.9090830e+02,3.0752090e+02,3.8606280e+02, &
& 3.5691590e+02,3.1786240e+02,2.9473480e+02,2.6629100e+02,2.3921900e+02, &
& 1.0130459e+03,9.3186100e+02,8.1889940e+02,3.7545680e+02,8.2577350e+02, &
& 7.9302390e+02,7.7315750e+02,7.5488610e+02,7.3868360e+02,5.8077730e+02, &
& 6.5347520e+02,6.3025330e+02,6.6633780e+02,6.5221550e+02,6.3947820e+02, &
& 6.3201230e+02,5.3244100e+02,5.2361220e+02,4.7773970e+02,4.0318600e+02, &
& 4.1018490e+02,3.7164480e+02,3.4026720e+02,2.8250560e+02,2.6390350e+02, &
& 2.7150750e+02,3.9615750e+02,3.8734040e+02,3.5575280e+02,3.3914780e+02, &
& 3.1279910e+02,2.8650170e+02,9.5682050e+02,9.2149130e+02,8.1480800e+02, &
& 7.3215470e+02,7.2684330e+02,7.0377870e+02,7.2468840e+02,7.0180630e+02, &
& 3.9274200e+01,1.2754920e+02,1.6261080e+02,1.2411700e+02,9.3384800e+01, &
& 6.4846800e+01,4.7181600e+01,3.2432100e+01,1.8646690e+02,2.8833710e+02, &
& 2.9202470e+02,2.3448430e+02,1.9184290e+02,1.6207500e+02,1.3234800e+02, &
& 2.5675790e+02,4.9074810e+02,2.5688890e+02,2.4796360e+02,2.4301480e+02]
 vdw_dftd3_c6(6901:7000)=[&
& 2.4078340e+02,2.2153400e+02,2.0499250e+02,2.3231000e+01,1.5354500e+01, &
& 3.6237460e+02,2.0922270e+02,1.4074280e+02,9.5005100e+01,6.6421800e+01, &
& 5.0301600e+01,3.8131300e+01,2.9359800e+01,4.3337360e+02,3.3347670e+02, &
& 3.0725400e+02,2.4191590e+02,1.8848520e+02,1.5634230e+02,1.2765130e+02, &
& 1.0440910e+02,7.1018180e+02,5.8595780e+02,4.8407230e+02,4.6739840e+02, &
& 4.2799360e+02,3.3714360e+02,3.6847190e+02,2.8919370e+02,3.0649680e+02, &
& 3.1577130e+02,2.4210300e+02,2.4848550e+02,2.9424080e+02,2.5932240e+02, &
& 2.2102330e+02,1.9822130e+02,1.7337930e+02,1.5071900e+02,7.9588590e+02, &
& 6.9807620e+02,6.1134030e+02,5.4932940e+02,5.0083550e+02,3.8652450e+02, &
& 4.3134780e+02,3.2850150e+02,3.5910570e+02,3.3298960e+02,2.7768230e+02, &
& 2.9340740e+02,3.6832240e+02,3.4042650e+02,3.0318560e+02,2.8116330e+02, &
& 2.5408770e+02,2.2832560e+02,9.6954620e+02,8.8993460e+02,7.8163500e+02, &
& 3.5816770e+02,7.8858090e+02,7.5717110e+02,7.3817480e+02,7.2070520e+02, &
& 7.0521290e+02,5.5434620e+02,6.2444810e+02,6.0226720e+02,6.3603570e+02, &
& 6.2253720e+02,6.1036000e+02,6.0323520e+02,5.0816310e+02,4.9947010e+02, &
& 4.5568660e+02,3.8467230e+02,3.9131100e+02,3.5455820e+02,3.2464910e+02, &
& 2.6963380e+02,2.5191510e+02,2.5913820e+02,3.7809130e+02,3.6955020e+02, &
& 3.3939390e+02,3.2357100e+02,2.9847990e+02,2.7345010e+02,9.1516100e+02, &
& 8.7983390e+02,7.7761650e+02,6.9856190e+02,6.9371430e+02,6.7170070e+02, &
& 6.9179850e+02,6.6991580e+02,3.7449600e+01,1.2160370e+02,1.5505540e+02]
 vdw_dftd3_c6(7001:7100)=[&
& 1.1835300e+02,8.9099200e+01,6.1925600e+01,4.5100200e+01,3.1051600e+01, &
& 1.7783550e+02,2.7498400e+02,2.7841730e+02,2.2355480e+02,1.8295110e+02, &
& 1.5462430e+02,1.2633190e+02,2.4501880e+02,4.6868990e+02,2.4500350e+02, &
& 2.3653250e+02,2.3181800e+02,2.2970220e+02,2.1130250e+02,1.9554300e+02, &
& 1.8654620e+02,2.2682200e+01,1.4983300e+01,3.5596060e+02,2.0490860e+02, &
& 1.3762540e+02,9.2810700e+01,6.4851900e+01,4.9099600e+01,3.7215800e+01, &
& 2.8655400e+01,4.2564550e+02,3.2678640e+02,3.0082710e+02,2.3659520e+02, &
& 1.8418470e+02,1.5270360e+02,1.2463170e+02,1.0191160e+02,6.9776500e+02, &
& 5.7475220e+02,4.7462540e+02,4.5811360e+02,4.1940370e+02,3.3038080e+02, &
& 3.6096920e+02,2.8330110e+02,3.0010550e+02,3.0925430e+02,2.3712000e+02, &
& 2.4319600e+02,2.8800730e+02,2.5361340e+02,2.1600430e+02,1.9364320e+02, &
& 1.6931330e+02,1.4714170e+02,7.8184410e+02,6.8477150e+02,5.9926520e+02, &
& 5.3824800e+02,4.9059730e+02,3.7845940e+02,4.2241440e+02,3.2155060e+02, &
& 3.5149860e+02,3.2588580e+02,2.7180880e+02,2.8709580e+02,3.6057310e+02, &
& 3.3303010e+02,2.9640640e+02,2.7477510e+02,2.4822370e+02,2.2298560e+02, &
& 9.5234010e+02,8.7316220e+02,7.6636770e+02,3.5020540e+02,7.7355490e+02, &
& 7.4267570e+02,7.2402070e+02,7.0686710e+02,6.9165380e+02,5.4327420e+02, &
& 6.1263290e+02,5.9076940e+02,6.2369960e+02,6.1045110e+02,5.9849350e+02, &
& 5.9152550e+02,4.9802530e+02,4.8915250e+02,4.4609160e+02,3.7646650e+02, &
& 3.8290120e+02,3.4682510e+02,3.1748840e+02,2.6363220e+02,2.4629020e+02]
 vdw_dftd3_c6(7101:7200)=[&
& 2.5331490e+02,3.7009330e+02,3.6153570e+02,3.3183930e+02,3.1626530e+02, &
& 2.9163450e+02,2.6709740e+02,8.9847060e+02,8.6294020e+02,7.6220730e+02, &
& 6.8424210e+02,6.7974920e+02,6.5817270e+02,6.7818560e+02,6.5668000e+02, &
& 3.6590100e+01,1.1907250e+02,1.5177980e+02,1.1570000e+02,8.7043200e+01, &
& 6.0460800e+01,4.4017300e+01,3.0300400e+01,1.7420010e+02,2.6937790e+02, &
& 2.7254090e+02,2.1860660e+02,1.7877350e+02,1.5102860e+02,1.2334570e+02, &
& 2.3988490e+02,4.5968120e+02,2.3965630e+02,2.3136310e+02,2.2676010e+02, &
& 2.2471760e+02,2.0661260e+02,1.9117980e+02,1.8238760e+02,1.7833110e+02, &
& 2.2155000e+01,1.4514700e+01,3.6743080e+02,2.0593130e+02,1.3637050e+02, &
& 9.1102500e+01,6.3270900e+01,4.7723900e+01,3.6074900e+01,2.7728200e+01, &
& 4.3871980e+02,3.3000180e+02,3.0146500e+02,2.3475480e+02,1.8132670e+02, &
& 1.4964210e+02,1.2163290e+02,9.9139800e+01,7.2200020e+02,5.8537040e+02, &
& 4.8164720e+02,4.6339240e+02,4.2341930e+02,3.3350140e+02,3.6341710e+02, &
& 2.8510180e+02,3.0073560e+02,3.1051330e+02,2.3812070e+02,2.4268930e+02, &
& 2.8785200e+02,2.5153860e+02,2.1283460e+02,1.9007710e+02,1.6558600e+02, &
& 1.4345250e+02,8.0787860e+02,6.9786230e+02,6.0685850e+02,5.4294990e+02, &
& 4.9363630e+02,3.7922990e+02,4.2391540e+02,3.2125020e+02,3.5112220e+02, &
& 3.2504660e+02,2.7146340e+02,2.8581800e+02,3.6066260e+02,3.3100630e+02, &
& 2.9285890e+02,2.7055260e+02,2.4354750e+02,2.1809290e+02,9.8359160e+02, &
& 8.9168850e+02,7.7770700e+02,3.4643740e+02,7.8837610e+02,7.5615770e+02]
 vdw_dftd3_c6(7201:7300)=[&
& 7.3695090e+02,7.1931180e+02,7.0365920e+02,5.4891190e+02,6.2523960e+02, &
& 6.0214160e+02,6.3352600e+02,6.1995970e+02,6.0766330e+02,6.0075720e+02, &
& 5.0338410e+02,4.9122120e+02,4.4629550e+02,3.7560840e+02,3.8145840e+02, &
& 3.4444290e+02,3.1453250e+02,2.6057850e+02,2.4322040e+02,2.4982310e+02, &
& 3.6971340e+02,3.5938220e+02,3.2809610e+02,3.1175430e+02,2.8648030e+02, &
& 2.6159230e+02,9.2344480e+02,8.7838880e+02,7.7137100e+02,6.8798510e+02, &
& 6.8576140e+02,6.6390850e+02,6.8694090e+02,6.6465810e+02,3.5987100e+01, &
& 1.1943970e+02,1.5184260e+02,1.1434660e+02,8.5473100e+01,5.8990300e+01, &
& 4.2745800e+01,2.9280800e+01,1.7525310e+02,2.7127550e+02,2.7263840e+02, &
& 2.1661540e+02,1.7596280e+02,1.4802440e+02,1.2039270e+02,2.4017910e+02, &
& 4.6791560e+02,2.3801280e+02,2.2972260e+02,2.2520200e+02,2.2340090e+02, &
& 2.0444800e+02,1.8894450e+02,1.8027190e+02,1.7633580e+02,1.7499690e+02, &
& 2.1213900e+01,1.4118500e+01,3.1788260e+02,1.8730120e+02,1.2724880e+02, &
& 8.6470900e+01,6.0733100e+01,4.6135500e+01,3.5062900e+01,2.7052400e+01, &
& 3.8065680e+02,2.9754400e+02,2.7562990e+02,2.1852820e+02,1.7118890e+02, &
& 1.4245910e+02,1.1666370e+02,9.5658600e+01,6.2148410e+02,5.1951820e+02, &
& 4.3036750e+02,4.1653280e+02,3.8196290e+02,3.0093230e+02,3.2951960e+02, &
& 2.5872900e+02,2.7503810e+02,2.8296560e+02,2.1695700e+02,2.2366220e+02, &
& 2.6447240e+02,2.3434070e+02,2.0064080e+02,1.8041890e+02,1.5822130e+02, &
& 1.3785900e+02,6.9720120e+02,6.1859280e+02,5.4431400e+02,4.9050970e+02]
 vdw_dftd3_c6(7301:7400)=[&
& 4.4803280e+02,3.4683870e+02,3.8662560e+02,2.9543070e+02,3.2298310e+02, &
& 2.9983170e+02,2.4983480e+02,2.6457160e+02,3.3095850e+02,3.0726350e+02, &
& 2.7478160e+02,2.5542890e+02,2.3140060e+02,2.0840550e+02,8.4934460e+02, &
& 7.8727710e+02,6.9479590e+02,3.2433650e+02,6.9872430e+02,6.7147320e+02, &
& 6.5477760e+02,6.3940750e+02,6.2578200e+02,4.9442430e+02,5.5258030e+02, &
& 5.3339380e+02,5.6508580e+02,5.5316870e+02,5.4245160e+02,5.3600770e+02, &
& 4.5307620e+02,4.4744320e+02,4.0933690e+02,3.4623600e+02,3.5258880e+02, &
& 3.2019590e+02,2.9371130e+02,2.4437210e+02,2.2847850e+02,2.3524690e+02, &
& 3.4006610e+02,3.3355770e+02,3.0748710e+02,2.9376220e+02,2.7163520e+02, &
& 2.4938090e+02,8.0499020e+02,7.8026750e+02,6.9268280e+02,6.2529330e+02, &
& 6.1943840e+02,5.9985590e+02,6.1596070e+02,5.9681690e+02,3.4031200e+01, &
& 1.0901270e+02,1.3924200e+02,1.0720260e+02,8.1068600e+01,5.6614000e+01, &
& 4.1386200e+01,2.8629100e+01,1.5912280e+02,2.4580540e+02,2.5007300e+02, &
& 2.0213060e+02,1.6618870e+02,1.4088040e+02,1.1545010e+02,2.1998520e+02, &
& 4.1556380e+02,2.2124670e+02,2.1361910e+02,2.0934280e+02,2.0729260e+02, &
& 1.9131250e+02,1.7720340e+02,1.6904610e+02,1.6523610e+02,1.6292360e+02, &
& 1.5345280e+02,3.3251500e+01,2.0987400e+01,6.0281040e+02,3.2666720e+02, &
& 2.1127560e+02,1.3814970e+02,9.4132900e+01,6.9903900e+01,5.2047600e+01, &
& 3.9458700e+01,7.1775310e+02,5.2642360e+02,4.7573180e+02,3.6461100e+02, &
& 2.7735230e+02,2.2627960e+02,1.8171780e+02,1.4638550e+02,1.1848327e+03]
 vdw_dftd3_c6(7401:7500)=[&
& 9.4258280e+02,7.7197630e+02,7.3902810e+02,6.7328230e+02,5.2900060e+02, &
& 5.7532290e+02,4.4990400e+02,4.7264010e+02,4.8954990e+02,3.7425370e+02, &
& 3.7873290e+02,4.5140030e+02,3.8973700e+02,3.2572680e+02,2.8839340e+02, &
& 2.4882110e+02,2.1348210e+02,1.3227322e+03,1.1239167e+03,9.6907770e+02, &
& 8.6203630e+02,7.8048050e+02,5.9457650e+02,6.6676010e+02,5.0051620e+02, &
& 5.4783460e+02,5.0566980e+02,4.2174200e+02,4.4286390e+02,5.6415640e+02, &
& 5.1303150e+02,4.4930060e+02,4.1227930e+02,3.6821830e+02,3.2708380e+02, &
& 1.6098951e+03,1.4393610e+03,1.2450382e+03,5.3206830e+02,1.2682581e+03, &
& 1.2147867e+03,1.1834931e+03,1.1548172e+03,1.1293623e+03,8.7169660e+02, &
& 1.0061273e+03,9.6741930e+02,1.0147443e+03,9.9283980e+02,9.7287530e+02, &
& 9.6226810e+02,8.0056070e+02,7.7479490e+02,6.9954820e+02,5.8500690e+02, &
& 5.9290660e+02,5.3219500e+02,4.8350900e+02,3.9773910e+02,3.7015170e+02, &
& 3.7970920e+02,5.7562810e+02,5.5583070e+02,5.0298850e+02,4.7531340e+02, &
& 4.3366250e+02,3.9318500e+02,1.5014541e+03,1.4115217e+03,1.2297307e+03, &
& 1.0857394e+03,1.0865380e+03,1.0514942e+03,1.0941031e+03,1.0575095e+03, &
& 5.4991900e+01,1.8876220e+02,2.3890680e+02,1.7609760e+02,1.2956700e+02, &
& 8.7664500e+01,6.2352200e+01,4.1590900e+01,2.7751430e+02,4.3082970e+02, &
& 4.2895210e+02,3.3552060e+02,2.6893700e+02,2.2384530e+02,1.7986830e+02, &
& 3.7584950e+02,7.5181440e+02,3.6854170e+02,3.5534980e+02,3.4832930e+02, &
& 3.4597960e+02,3.1442440e+02,2.8975880e+02,2.7631300e+02,2.7040230e+02]
 vdw_dftd3_c6(7501:7600)=[&
& 2.6973440e+02,2.4870030e+02,4.2164970e+02,2.9952700e+01,1.9376100e+01, &
& 4.8401650e+02,2.7657170e+02,1.8420550e+02,1.2305740e+02,8.5162100e+01, &
& 6.3924500e+01,4.8023300e+01,3.6661500e+01,5.7808820e+02,4.4152390e+02, &
& 4.0516110e+02,3.1697000e+02,2.4526830e+02,2.0228630e+02,1.6412660e+02, &
& 1.3337940e+02,9.4657560e+02,7.7753610e+02,6.4140640e+02,6.1804010e+02, &
& 5.6526390e+02,4.4441960e+02,4.8576480e+02,3.8038530e+02,4.0291460e+02, &
& 4.1564390e+02,3.1786240e+02,3.2569280e+02,3.8671290e+02,3.3924040e+02, &
& 2.8758470e+02,2.5685170e+02,2.2357440e+02,1.9336340e+02,1.0596044e+03, &
& 9.2600250e+02,8.0859910e+02,7.2498050e+02,6.5980390e+02,5.0715030e+02, &
& 5.6682940e+02,4.2969960e+02,4.7030780e+02,4.3552680e+02,3.6250760e+02, &
& 3.8302040e+02,4.8291150e+02,4.4490480e+02,3.9458220e+02,3.6482430e+02, &
& 3.2847500e+02,2.9399660e+02,1.2903249e+03,1.1809729e+03,1.0344900e+03, &
& 4.6618800e+02,1.0448624e+03,1.0029138e+03,9.7765230e+02,9.5444310e+02, &
& 9.3386150e+02,7.3092270e+02,8.2651040e+02,7.9662990e+02,8.4175260e+02, &
& 8.2386870e+02,8.0770310e+02,7.9843800e+02,6.7060970e+02,6.5755650e+02, &
& 5.9836780e+02,5.0344310e+02,5.1179750e+02,4.6250420e+02,4.2248760e+02, &
& 3.4951600e+02,3.2600720e+02,3.3530450e+02,4.9429340e+02,4.8215480e+02, &
& 4.4128680e+02,4.1974070e+02,3.8594190e+02,3.5237490e+02,1.2154454e+03, &
& 1.1657107e+03,1.0274859e+03,9.1935170e+02,9.1390230e+02,8.8471530e+02, &
& 9.1297050e+02,8.8377800e+02,4.8724700e+01,1.6045020e+02,2.0416900e+02]
 vdw_dftd3_c6(7601:7700)=[&
& 1.5444090e+02,1.1534880e+02,7.9327800e+01,5.7175200e+01,3.8758300e+01, &
& 2.3460480e+02,3.6333200e+02,3.6665610e+02,2.9253350e+02,2.3795800e+02, &
& 2.0005260e+02,1.6242060e+02,3.2115740e+02,6.2067350e+02,3.2013420e+02, &
& 3.0883040e+02,3.0262720e+02,2.9997060e+02,2.7526070e+02,2.5437350e+02, &
& 2.4255720e+02,2.3717370e+02,2.3483270e+02,2.1945080e+02,3.6268320e+02, &
& 3.1685210e+02,2.7357400e+01,1.8081600e+01,4.0634270e+02,2.4075590e+02, &
& 1.6396550e+02,1.1142570e+02,7.8113800e+01,5.9176400e+01,4.4807900e+01, &
& 3.4422600e+01,4.8646380e+02,3.8183960e+02,3.5437060e+02,2.8148930e+02, &
& 2.2071460e+02,1.8364620e+02,1.5027420e+02,1.2303580e+02,7.9363060e+02, &
& 6.6532210e+02,5.5149990e+02,5.3402660e+02,4.8982100e+02,3.8565770e+02, &
& 4.2268740e+02,3.3163890e+02,3.5300740e+02,3.6305660e+02,2.7805780e+02, &
& 2.8717780e+02,3.3996250e+02,3.0168000e+02,2.5852630e+02,2.3250720e+02, &
& 2.0384000e+02,1.7747160e+02,8.9049310e+02,7.9208160e+02,6.9780370e+02, &
& 6.2921900e+02,5.7490090e+02,4.4507990e+02,4.9610230e+02,3.7904270e+02, &
& 4.1454110e+02,3.8481520e+02,3.2022310e+02,3.3948140e+02,4.2470080e+02, &
& 3.9481650e+02,3.5342450e+02,3.2867560e+02,2.9780640e+02,2.6817000e+02, &
& 1.0853677e+03,1.0077913e+03,8.9047640e+02,4.1703500e+02,8.9428150e+02, &
& 8.5952480e+02,8.3818370e+02,8.1853810e+02,8.0112860e+02,6.3358090e+02, &
& 7.0687070e+02,6.8251260e+02,7.2360130e+02,7.0836530e+02,6.9468070e+02, &
& 6.8640110e+02,5.8061170e+02,5.7440900e+02,5.2573750e+02,4.4464860e+02]
 vdw_dftd3_c6(7701:7800)=[&
& 4.5291770e+02,4.1136690e+02,3.7733280e+02,3.1370400e+02,2.9316560e+02, &
& 3.0197230e+02,4.3615840e+02,4.2826390e+02,3.9514950e+02,3.7768730e+02, &
& 3.4934070e+02,3.2071970e+02,1.0295399e+03,9.9943950e+02,8.8791120e+02, &
& 8.0215530e+02,7.9394730e+02,7.6878210e+02,7.8864020e+02,7.6421100e+02, &
& 4.3906700e+01,1.4013530e+02,1.7913900e+02,1.3814460e+02,1.0445320e+02, &
& 7.2807300e+01,5.3064500e+01,3.6472800e+01,2.0419370e+02,3.1566880e+02, &
& 3.2160430e+02,2.6040440e+02,2.1425140e+02,1.8159050e+02,1.4869430e+02, &
& 2.8230700e+02,5.3224610e+02,2.8445160e+02,2.7459290e+02,2.6902700e+02, &
& 2.6629390e+02,2.4597130e+02,2.2781130e+02,2.1726350e+02,2.1232210e+02, &
& 2.0913220e+02,1.9727710e+02,3.1978710e+02,2.8257170e+02,2.5419240e+02, &
& 2.5314900e+01,1.7028000e+01,3.5504500e+02,2.1516710e+02,1.4885900e+02, &
& 1.0246200e+02,7.2556700e+01,5.5375200e+01,4.2202700e+01,3.2593500e+01, &
& 4.2577330e+02,3.3984270e+02,3.1781330e+02,2.5511530e+02,2.0196030e+02, &
& 1.6917120e+02,1.3934530e+02,1.1476770e+02,6.9398180e+02,5.8857630e+02, &
& 4.8933050e+02,4.7544530e+02,4.3692930e+02,3.4458580e+02,3.7811330e+02, &
& 2.9726970e+02,3.1722330e+02,3.2556670e+02,2.4982100e+02,2.5919160e+02, &
& 3.0612750e+02,2.7376800e+02,2.3642060e+02,2.1372980e+02,1.8840310e+02, &
& 1.6488330e+02,7.8009860e+02,7.0080150e+02,6.2089180e+02,5.6200010e+02, &
& 5.1489450e+02,4.0077220e+02,4.4579160e+02,3.4259700e+02,3.7429000e+02, &
& 3.4804020e+02,2.8983280e+02,3.0772140e+02,3.8283810e+02,3.5796090e+02]
 vdw_dftd3_c6(7801:7900)=[&
& 3.2248120e+02,3.0115060e+02,2.7413770e+02,2.4798670e+02,9.5163050e+02, &
& 8.9062640e+02,7.9124710e+02,3.8031600e+02,7.9188170e+02,7.6169980e+02, &
& 7.4294350e+02,7.2565360e+02,7.1033760e+02,5.6579140e+02,6.2617050e+02, &
& 6.0528280e+02,6.4237950e+02,6.2891080e+02,6.1686660e+02,6.0931460e+02, &
& 5.1790690e+02,5.1523880e+02,4.7346190e+02,4.0207780e+02,4.1003520e+02, &
& 3.7374220e+02,3.4383760e+02,2.8699450e+02,2.6861450e+02,2.7687780e+02, &
& 3.9433850e+02,3.8873570e+02,3.6061850e+02,3.4584750e+02,3.2125280e+02, &
& 2.9613640e+02,9.0670650e+02,8.8597380e+02,7.9097170e+02,7.1920180e+02, &
& 7.1002990e+02,6.8765590e+02,7.0268800e+02,6.8135890e+02,4.0211700e+01, &
& 1.2555640e+02,1.6106370e+02,1.2588880e+02,9.6087200e+01,6.7679600e+01, &
& 4.9766500e+01,3.4588200e+01,1.8260620e+02,2.8192490e+02,2.8901630e+02, &
& 2.3642530e+02,1.9613740e+02,1.6726720e+02,1.3787350e+02,2.5453760e+02, &
& 4.7198210e+02,2.5812950e+02,2.4934370e+02,2.4427270e+02,2.4158030e+02, &
& 2.2407570e+02,2.0787480e+02,1.9829930e+02,1.9372130e+02,1.9016850e+02, &
& 1.8048120e+02,2.8854930e+02,2.5695230e+02,2.3257840e+02,2.1378620e+02, &
& 2.2623000e+01,1.5538700e+01,2.9768350e+02,1.8488320e+02,1.3018810e+02, &
& 9.0934300e+01,6.5154300e+01,5.0163200e+01,3.8529400e+01,2.9950100e+01, &
& 3.5772460e+02,2.9072760e+02,2.7418940e+02,2.2268120e+02,1.7819840e+02, &
& 1.5042030e+02,1.2484950e+02,1.0354670e+02,5.8289140e+02,5.0041520e+02, &
& 4.1736480e+02,4.0709700e+02,3.7492830e+02,2.9636070e+02,3.2549450e+02]
 vdw_dftd3_c6(7901:8000)=[&
& 2.5659390e+02,2.7446200e+02,2.8101540e+02,2.1621400e+02,2.2534570e+02, &
& 2.6539390e+02,2.3936360e+02,2.0849570e+02,1.8959740e+02,1.6817910e+02, &
& 1.4806890e+02,6.5665740e+02,5.9602140e+02,5.3134260e+02,4.8298730e+02, &
& 4.4388480e+02,3.4768000e+02,3.8579860e+02,2.9851740e+02,3.2566500e+02, &
& 3.0341070e+02,2.5300950e+02,2.6894910e+02,3.3247910e+02,3.1279940e+02, &
& 2.8377610e+02,2.6623860e+02,2.4362960e+02,2.2153670e+02,8.0186140e+02, &
& 7.5662710e+02,6.7619300e+02,3.3452620e+02,6.7428760e+02,6.4912820e+02, &
& 6.3328350e+02,6.1865400e+02,6.0569940e+02,4.8634440e+02,5.3364850e+02, &
& 5.1649050e+02,5.4846290e+02,5.3700790e+02,5.2681470e+02,5.2016740e+02, &
& 4.4455160e+02,4.4488510e+02,4.1064300e+02,3.5042340e+02,3.5779320e+02, &
& 3.2743170e+02,3.0224750e+02,2.5348160e+02,2.3768320e+02,2.4514110e+02, &
& 3.4374040e+02,3.4024220e+02,3.1748380e+02,3.0560970e+02,2.8522290e+02, &
& 2.6413340e+02,7.6779060e+02,7.5526170e+02,6.7787670e+02,6.2082600e+02, &
& 6.1130750e+02,5.9218710e+02,6.0258350e+02,5.8469980e+02,3.5512300e+01, &
& 1.0820480e+02,1.3936170e+02,1.1058020e+02,8.5326500e+01,6.0835900e+01, &
& 4.5201900e+01,3.1831300e+01,1.5710050e+02,2.4214760e+02,2.4992480e+02, &
& 2.0678800e+02,1.7315820e+02,1.4872190e+02,1.2352740e+02,2.2116420e+02, &
& 4.0253800e+02,2.2577840e+02,2.1827110e+02,2.1382760e+02,2.1128090e+02, &
& 1.9683950e+02,1.8295500e+02,1.7459100e+02,1.7050040e+02,1.6678160e+02, &
& 1.5930490e+02,2.5072470e+02,2.2517920e+02,2.0523050e+02,1.8964080e+02]
 vdw_dftd3_c6(8001:8100)=[&
& 1.6923750e+02,3.5196300e+01,2.2623100e+01,6.5740760e+02,3.4543780e+02, &
& 2.2250810e+02,1.4599740e+02,1.0014180e+02,7.4886800e+01,5.6177900e+01, &
& 4.2895700e+01,7.8185270e+02,5.5913370e+02,5.0299330e+02,3.8390840e+02, &
& 2.9216040e+02,2.3905720e+02,1.9278770e+02,1.5610100e+02,1.3058840e+03, &
& 1.0130706e+03,8.2610410e+02,7.8941170e+02,7.1821250e+02,5.6618560e+02, &
& 6.1264440e+02,4.8062360e+02,5.0159400e+02,5.1998260e+02,3.9935340e+02, &
& 4.0105480e+02,4.7797370e+02,4.1103890e+02,3.4339780e+02,3.0453120e+02, &
& 2.6345920e+02,2.2685900e+02,1.4577817e+03,1.2108108e+03,1.0374434e+03, &
& 9.2013580e+02,8.3213680e+02,6.3394710e+02,7.1077790e+02,5.3361350e+02, &
& 5.8237420e+02,5.3725650e+02,4.5047140e+02,4.7031190e+02,5.9989870e+02, &
& 5.4287770e+02,4.7456410e+02,4.3555550e+02,3.8946840e+02,3.4667910e+02, &
& 1.7768880e+03,1.5569958e+03,1.3375010e+03,5.6314800e+02,1.3695245e+03, &
& 1.3097014e+03,1.2753968e+03,1.2439950e+03,1.2161018e+03,9.3460820e+02, &
& 1.0930934e+03,1.0500925e+03,1.0903500e+03,1.0664469e+03,1.0446045e+03, &
& 1.0333196e+03,8.5724780e+02,8.2420500e+02,7.4273970e+02,6.2209770e+02, &
& 6.2947280e+02,5.6455160e+02,5.1279660e+02,4.2290190e+02,3.9392880e+02, &
& 4.0326450e+02,6.1404690e+02,5.8982750e+02,5.3239070e+02,5.0286730e+02, &
& 4.5900050e+02,4.1673210e+02,1.6476456e+03,1.5223656e+03,1.3176426e+03, &
& 1.1578649e+03,1.1634976e+03,1.1259076e+03,1.1753584e+03,1.1351209e+03, &
& 5.7903900e+01,1.9967710e+02,2.5295740e+02,1.8572060e+02,1.3721250e+02]
 vdw_dftd3_c6(8101:8200)=[&
& 9.3458400e+01,6.7004500e+01,4.5248900e+01,2.9467950e+02,4.5778310e+02, &
& 4.5336360e+02,3.5341420e+02,2.8343680e+02,2.3654670e+02,1.9086020e+02, &
& 4.0103710e+02,8.1234890e+02,3.8990570e+02,3.7639190e+02,3.6906490e+02, &
& 3.6686860e+02,3.3222150e+02,3.0627200e+02,2.9229920e+02,2.8612410e+02, &
& 2.8601310e+02,2.6273320e+02,4.4607980e+02,3.8250120e+02,3.3721600e+02, &
& 3.0472790e+02,2.6546470e+02,4.7625910e+02,6.4406500e+01,3.8833300e+01, &
& 1.6182465e+03,7.3479260e+02,4.4112610e+02,2.7470310e+02,1.8119730e+02, &
& 1.3178460e+02,9.6508000e+01,7.2270300e+01,1.9079592e+03,1.2181274e+03, &
& 1.0570218e+03,7.6638470e+02,5.5938210e+02,4.4578600e+02,3.5042730e+02, &
& 2.7747590e+02,3.3022504e+03,2.3148420e+03,1.8497319e+03,1.7407410e+03, &
& 1.5682263e+03,1.2385036e+03,1.3188096e+03,1.0340602e+03,1.0522336e+03, &
& 1.1016666e+03,8.4879520e+02,8.2251670e+02,9.9191230e+02,8.1848870e+02, &
& 6.6009280e+02,5.7308500e+02,4.8506500e+02,4.0937610e+02,3.6709261e+03, &
& 2.7824577e+03,2.3056086e+03,2.0037829e+03,1.7897638e+03,1.3350700e+03, &
& 1.5092981e+03,1.1061259e+03,1.2034301e+03,1.1005370e+03,9.3160220e+02, &
& 9.5275330e+02,1.2501173e+03,1.0916636e+03,9.2413690e+02,8.3250600e+02, &
& 7.2971760e+02,6.3746600e+02,4.4969846e+03,3.6314540e+03,3.0140781e+03, &
& 1.1044325e+03,3.1621198e+03,2.9999283e+03,2.9157949e+03,2.8393501e+03, &
& 2.7713234e+03,2.0594692e+03,2.5668729e+03,2.4578402e+03,2.4609470e+03, &
& 2.4041111e+03,2.3512803e+03,2.3289369e+03,1.8927447e+03,1.7529565e+03]
 vdw_dftd3_c6(8201:8300)=[&
& 1.5479897e+03,1.2804929e+03,1.2840434e+03,1.1312913e+03,1.0131095e+03, &
& 8.2466090e+02,7.6388530e+02,7.7550460e+02,1.2742283e+03,1.1873359e+03, &
& 1.0400523e+03,9.6630350e+02,8.6500840e+02,7.7178770e+02,4.0533897e+03, &
& 3.4910837e+03,2.9248174e+03,2.4812627e+03,2.5425532e+03,2.4572393e+03, &
& 2.6180036e+03,2.5175053e+03,1.1032750e+02,4.2146100e+02,5.2954940e+02, &
& 3.6336070e+02,2.5898850e+02,1.6917720e+02,1.1710180e+02,7.5675800e+01, &
& 6.3083040e+02,9.8878570e+02,9.4475350e+02,7.0100090e+02,5.4212620e+02, &
& 4.4147190e+02,3.4712550e+02,8.4199830e+02,1.8769417e+03,7.7994880e+02, &
& 7.5467530e+02,7.4017630e+02,7.3983150e+02,6.5184050e+02,5.9697670e+02, &
& 5.7030240e+02,5.5938830e+02,5.6991510e+02,5.0526080e+02,9.1623120e+02, &
& 7.5505840e+02,6.4702200e+02,5.7371350e+02,4.8934610e+02,9.9279700e+02, &
& 2.3126602e+03,5.1235600e+01,3.1988900e+01,1.0745875e+03,5.3251100e+02, &
& 3.3413510e+02,2.1477600e+02,1.4492070e+02,1.0704380e+02,7.9407600e+01, &
& 6.0062800e+01,1.2727028e+03,8.6937390e+02,7.7186060e+02,5.7805500e+02, &
& 4.3301710e+02,3.5057750e+02,2.7973550e+02,2.2431260e+02,2.1637031e+03, &
& 1.6036362e+03,1.2972964e+03,1.2323433e+03,1.1169531e+03,8.8060690e+02, &
& 9.4758340e+02,7.4266410e+02,7.6821070e+02,7.9937100e+02,6.1409830e+02, &
& 6.0904990e+02,7.2958130e+02,6.1799850e+02,5.0955030e+02,4.4813810e+02, &
& 3.8429010e+02,3.2813160e+02,2.4114577e+03,1.9209295e+03,1.6243636e+03, &
& 1.4293576e+03,1.2864770e+03,9.7159590e+02,1.0932424e+03,8.1262860e+02]
 vdw_dftd3_c6(8301:8400)=[&
& 8.8627500e+02,8.1484020e+02,6.8497960e+02,7.1021720e+02,9.1615920e+02, &
& 8.1841300e+02,7.0709330e+02,6.4444940e+02,5.7184320e+02,5.0522620e+02, &
& 2.9502475e+03,2.4856310e+03,2.1057491e+03,8.4084100e+02,2.1783229e+03, &
& 2.0748807e+03,2.0189224e+03,1.9678827e+03,1.9225226e+03,1.4581969e+03, &
& 1.7521144e+03,1.6830549e+03,1.7170123e+03,1.6785781e+03,1.6432139e+03, &
& 1.6263339e+03,1.3398149e+03,1.2689492e+03,1.1346407e+03,9.4540990e+02, &
& 9.5356340e+02,8.4938170e+02,7.6730030e+02,6.2916050e+02,5.8463530e+02, &
& 5.9694450e+02,9.3562990e+02,8.8887560e+02,7.9367010e+02,7.4514710e+02, &
& 6.7518420e+02,6.0886250e+02,2.6980970e+03,2.4134225e+03,2.0622685e+03, &
& 1.7871863e+03,1.8092861e+03,1.7497379e+03,1.8406901e+03,1.7746391e+03, &
& 8.5681500e+01,3.0688620e+02,3.8765640e+02,2.7740500e+02,2.0197440e+02, &
& 1.3518390e+02,9.5445800e+01,6.3188900e+01,4.5499280e+02,7.0969610e+02, &
& 6.9347320e+02,5.3081800e+02,4.1987180e+02,3.4697100e+02,2.7697730e+02, &
& 6.1384630e+02,1.2935159e+03,5.8631350e+02,5.6686700e+02,5.5572150e+02, &
& 5.5347640e+02,4.9633310e+02,4.5638780e+02,4.3567010e+02,4.2674050e+02, &
& 4.2937430e+02,3.8958110e+02,6.7825900e+02,5.7321390e+02,5.0000710e+02, &
& 4.4848720e+02,3.8738760e+02,7.2686130e+02,1.5865402e+03,1.1322983e+03, &
& 3.9666900e+01,2.5703700e+01,6.7848100e+02,3.7401350e+02,2.4601470e+02, &
& 1.6345480e+02,1.1293390e+02,8.4787600e+01,6.3768000e+01,4.8760000e+01, &
& 8.0899730e+02,6.0055110e+02,5.4671760e+02,4.2369920e+02,3.2600870e+02]
 vdw_dftd3_c6(8401:8500)=[&
& 2.6830770e+02,2.1743220e+02,1.7666900e+02,1.3359899e+03,1.0709104e+03, &
& 8.7901650e+02,8.4415650e+02,7.7041850e+02,6.0670400e+02,6.6010050e+02, &
& 5.1758120e+02,5.4466430e+02,5.6293440e+02,4.3155440e+02,4.3840800e+02, &
& 5.2118020e+02,4.5369050e+02,3.8264580e+02,3.4103380e+02,2.9639440e+02, &
& 2.5615010e+02,1.4937993e+03,1.2775813e+03,1.1066630e+03,9.8782570e+02, &
& 8.9679520e+02,6.8718480e+02,7.6884280e+02,5.8090070e+02,6.3480240e+02, &
& 5.8702020e+02,4.9038170e+02,5.1538610e+02,6.5256540e+02,5.9689190e+02, &
& 5.2654260e+02,4.8561760e+02,4.3630460e+02,3.8993850e+02,1.8190406e+03, &
& 1.6353622e+03,1.4206192e+03,6.2328410e+02,1.4429680e+03,1.3832190e+03, &
& 1.3477804e+03,1.3152670e+03,1.2864106e+03,9.9936150e+02,1.1461276e+03, &
& 1.1025614e+03,1.1568729e+03,1.1319421e+03,1.1093035e+03,1.0968681e+03, &
& 9.1629250e+02,8.9141000e+02,8.0810660e+02,6.7904580e+02,6.8897190e+02, &
& 6.2088940e+02,5.6604620e+02,4.6802460e+02,4.3643220e+02,4.4796630e+02, &
& 6.6836720e+02,6.4781650e+02,5.8976480e+02,5.5956430e+02,5.1325740e+02, &
& 4.6784580e+02,1.7028592e+03,1.6078178e+03,1.4061219e+03,1.2490250e+03, &
& 1.2473212e+03,1.2073349e+03,1.2519154e+03,1.2106981e+03,6.4706800e+01, &
& 2.1671700e+02,2.7536440e+02,2.0600140e+02,1.5342290e+02,1.0531510e+02, &
& 7.5899400e+01,5.1515800e+01,3.1816780e+02,4.9330570e+02,4.9402450e+02, &
& 3.9071920e+02,3.1631690e+02,2.6540890e+02,2.1521010e+02,4.3506490e+02, &
& 8.5671310e+02,4.2930190e+02,4.1420300e+02,4.0602230e+02,4.0292000e+02]
 vdw_dftd3_c6(8501:8600)=[&
& 3.6779370e+02,3.3962780e+02,3.2399210e+02,3.1694170e+02,3.1499080e+02, &
& 2.9240050e+02,4.8787590e+02,4.2320760e+02,3.7591790e+02,3.4120400e+02, &
& 2.9856340e+02,5.1769610e+02,1.0443401e+03,7.8082120e+02,5.6867890e+02, &
& 4.0030400e+01,2.5953400e+01,6.8410400e+02,3.7796930e+02,2.4842660e+02, &
& 1.6499800e+02,1.1399890e+02,8.5602400e+01,6.4401000e+01,4.9263500e+01, &
& 8.1603170e+02,6.0700730e+02,5.5235500e+02,4.2788370e+02,3.2909470e+02, &
& 2.7080590e+02,2.1944620e+02,1.7831830e+02,1.3433714e+03,1.0819930e+03, &
& 8.8833560e+02,8.5304640e+02,7.7852660e+02,6.1302240e+02,6.6703050e+02, &
& 5.2300200e+02,5.5040920e+02,5.6890880e+02,4.3610420e+02,4.4298860e+02, &
& 5.2649640e+02,4.5819200e+02,3.8630910e+02,3.4424250e+02,2.9915350e+02, &
& 2.5853040e+02,1.5017789e+03,1.2904826e+03,1.1181402e+03,9.9812240e+02, &
& 9.0608010e+02,6.9429830e+02,7.7673680e+02,5.8689450e+02,6.4141640e+02, &
& 5.9314140e+02,4.9548000e+02,5.2076290e+02,6.5928650e+02,6.0296920e+02, &
& 5.3174240e+02,4.9032060e+02,4.4045990e+02,3.9361210e+02,1.8264461e+03, &
& 1.6509334e+03,1.4348956e+03,6.2949770e+02,1.4565113e+03,1.3970298e+03, &
& 1.3613178e+03,1.3285415e+03,1.2994488e+03,1.0094419e+03,1.1553947e+03, &
& 1.1106913e+03,1.1688393e+03,1.1436857e+03,1.1208436e+03,1.1082791e+03, &
& 9.2522660e+02,9.0068080e+02,8.1649740e+02,6.8598560e+02,6.9600400e+02, &
& 6.2720650e+02,5.7177930e+02,4.7276430e+02,4.4085970e+02,4.5247610e+02, &
& 6.7515180e+02,6.5444510e+02,5.9566400e+02,5.6506480e+02,5.1821240e+02]
 vdw_dftd3_c6(8601:8700)=[&
& 4.7230580e+02,1.7127050e+03,1.6235376e+03,1.4203983e+03,1.2619867e+03, &
& 1.2600147e+03,1.2197064e+03,1.2648378e+03,1.2232638e+03,6.5312900e+01, &
& 2.1897060e+02,2.7808810e+02,2.0799430e+02,1.5487640e+02,1.0631410e+02, &
& 7.6634700e+01,5.2041200e+01,3.2153780e+02,4.9834860e+02,4.9907610e+02, &
& 3.9454030e+02,3.1930990e+02,2.6788580e+02,2.1720780e+02,4.3952280e+02, &
& 8.6414110e+02,4.3378650e+02,4.1830680e+02,4.1010330e+02,4.0698940e+02, &
& 3.7146910e+02,3.4300120e+02,3.2719950e+02,3.2009600e+02,3.1821360e+02, &
& 2.9527620e+02,4.9283160e+02,4.2736220e+02,3.7950970e+02,3.4441690e+02, &
& 3.0134630e+02,5.2294490e+02,1.0528629e+03,7.8749500e+02,5.7445910e+02, &
& 5.8058240e+02,3.7547800e+01,2.4553000e+01,6.2818170e+02,3.4918970e+02, &
& 2.3104490e+02,1.5432880e+02,1.0713040e+02,8.0736000e+01,6.0941400e+01, &
& 4.6749800e+01,7.4949740e+02,5.5988560e+02,5.1108100e+02,3.9766020e+02, &
& 3.0713840e+02,2.5349850e+02,2.0604590e+02,1.6790140e+02,1.2379643e+03, &
& 9.9601570e+02,8.1851610e+02,7.8705650e+02,7.1884690e+02,5.6643000e+02, &
& 6.1659930e+02,4.8385370e+02,5.0969990e+02,5.2637950e+02,4.0384000e+02, &
& 4.1098660e+02,4.8798580e+02,4.2607030e+02,3.6044960e+02,3.2194150e+02, &
& 2.8047020e+02,2.4296840e+02,1.3851228e+03,1.1881982e+03,1.0314745e+03, &
& 9.2206600e+02,8.3798490e+02,6.4347210e+02,7.1936590e+02,5.4480750e+02, &
& 5.9515030e+02,5.5075570e+02,4.6023670e+02,4.8402860e+02,6.1139110e+02, &
& 5.6051040e+02,4.9569480e+02,4.5793020e+02,4.1222750e+02,3.6915310e+02]
 vdw_dftd3_c6(8701:8800)=[&
& 1.6877490e+03,1.5201317e+03,1.3232583e+03,5.8664300e+02,1.3428116e+03, &
& 1.2872763e+03,1.2544039e+03,1.2242314e+03,1.1974564e+03,9.3283320e+02, &
& 1.0667077e+03,1.0271026e+03,1.0774205e+03,1.0542484e+03,1.0332364e+03, &
& 1.0215311e+03,8.5524570e+02,8.3345230e+02,7.5675320e+02,6.3690770e+02, &
& 6.4654950e+02,5.8352570e+02,5.3265790e+02,4.4119610e+02,4.1171360e+02, &
& 4.2271910e+02,6.2693260e+02,6.0866200e+02,5.5532550e+02,5.2759970e+02, &
& 4.8478900e+02,4.4266740e+02,1.5815449e+03,1.4962077e+03,1.3112088e+03, &
& 1.1676309e+03,1.1649288e+03,1.1276927e+03,1.1676383e+03,1.1294830e+03, &
& 6.0977300e+01,2.0251730e+02,2.5760880e+02,1.9375570e+02,1.4487380e+02, &
& 9.9938100e+01,7.2352000e+01,4.9416600e+01,2.9717130e+02,4.6042340e+02, &
& 4.6216780e+02,3.6695490e+02,2.9806610e+02,2.5075730e+02,2.0393970e+02, &
& 4.0759060e+02,7.9747730e+02,4.0315400e+02,3.8917870e+02,3.8148040e+02, &
& 3.7844740e+02,3.4604230e+02,3.1976370e+02,3.0508870e+02,2.9841290e+02, &
& 2.9619890e+02,2.7560550e+02,4.5716410e+02,3.9775920e+02,3.5418680e+02, &
& 3.2209780e+02,2.8249330e+02,4.8541290e+02,9.7212330e+02,7.3014460e+02, &
& 5.3401080e+02,5.3935320e+02,5.0200730e+02,4.1338500e+01,2.6621700e+01, &
& 7.6378140e+02,4.0277270e+02,2.6037960e+02,1.7125810e+02,1.1764790e+02, &
& 8.8060900e+01,6.6109300e+01,5.0508700e+01,9.0840250e+02,6.5125640e+02, &
& 5.8698620e+02,4.4911910e+02,3.4248830e+02,2.8056130e+02,2.2648330e+02, &
& 1.8352050e+02,1.5188755e+03,1.1783302e+03,9.6136570e+02,9.1930990e+02]
 vdw_dftd3_c6(8801:8900)=[&
& 8.3673410e+02,6.5972010e+02,7.1418300e+02,5.6036480e+02,5.8529350e+02, &
& 6.0647400e+02,4.6581470e+02,4.6845290e+02,5.5815220e+02,4.8089260e+02, &
& 4.0245400e+02,3.5725110e+02,3.0935300e+02,2.6657770e+02,1.6961214e+03, &
& 1.4083895e+03,1.2079868e+03,1.0721935e+03,9.7020700e+02,7.3979150e+02, &
& 8.2924080e+02,6.2313860e+02,6.8005240e+02,6.2758100e+02,5.2609770e+02, &
& 5.4963380e+02,7.0042290e+02,6.3478310e+02,5.5577940e+02,5.1056820e+02, &
& 4.5697070e+02,4.0709700e+02,2.0688557e+03,1.8110626e+03,1.5570674e+03, &
& 6.5923480e+02,1.5936331e+03,1.5238714e+03,1.4839817e+03,1.4474665e+03, &
& 1.4150358e+03,1.0889839e+03,1.2727235e+03,1.2232739e+03,1.2689174e+03, &
& 1.2411151e+03,1.2157298e+03,1.2025314e+03,9.9886140e+02,9.6131920e+02, &
& 8.6700540e+02,7.2670660e+02,7.3557240e+02,6.6018700e+02,6.0002740e+02, &
& 4.9511760e+02,4.6130260e+02,4.7242330e+02,7.1722290e+02,6.8964350e+02, &
& 6.2335390e+02,5.8927250e+02,5.3837470e+02,4.8918620e+02,1.9184029e+03, &
& 1.7715639e+03,1.5346792e+03,1.3502400e+03,1.3560504e+03,1.3122121e+03, &
& 1.3686610e+03,1.3219594e+03,6.7892400e+01,2.3295390e+02,2.9538160e+02, &
& 2.1747620e+02,1.6092230e+02,1.0977550e+02,7.8792000e+01,5.3288000e+01, &
& 3.4355920e+02,5.3370850e+02,5.2930430e+02,4.1359910e+02,3.3227830e+02, &
& 2.7760020e+02,2.2420950e+02,4.6809770e+02,9.4599430e+02,4.5591090e+02, &
& 4.4025610e+02,4.3163910e+02,4.2896620e+02,3.8890660e+02,3.5865620e+02, &
& 3.4229790e+02,3.3502870e+02,3.3456150e+02,3.0788150e+02,5.2114410e+02]
 vdw_dftd3_c6(8901:9000)=[&
& 4.4771310e+02,3.9524720e+02,3.5747370e+02,3.1169010e+02,5.5612550e+02, &
& 1.1562604e+03,8.4837270e+02,6.0533890e+02,6.1127640e+02,5.6776700e+02, &
& 6.4971430e+02,3.3187500e+01,2.2061600e+01,5.1733300e+02,2.9751860e+02, &
& 2.0032990e+02,1.3555170e+02,9.5019000e+01,7.2117300e+01,5.4778100e+01, &
& 4.2243200e+01,6.1866690e+02,4.7441800e+02,4.3702090e+02,3.4421130e+02, &
& 2.6854160e+02,2.2308830e+02,1.8247210e+02,1.4952220e+02,1.0157997e+03, &
& 8.3526920e+02,6.8960880e+02,6.6581430e+02,6.0961590e+02,4.8063820e+02, &
& 5.2477480e+02,4.1224680e+02,4.3640540e+02,4.4958060e+02,3.4510430e+02, &
& 3.5378800e+02,4.1885730e+02,3.6917340e+02,3.1492530e+02,2.8273020e+02, &
& 2.4762030e+02,2.1556830e+02,1.1385265e+03,9.9561160e+02,8.7126000e+02, &
& 7.8271660e+02,7.1364770e+02,5.5112400e+02,6.1481850e+02,4.6856950e+02, &
& 5.1183150e+02,4.7463690e+02,3.9627420e+02,4.1825790e+02,5.2482360e+02, &
& 4.8488550e+02,4.3198650e+02,4.0083190e+02,3.6253740e+02,3.2611570e+02, &
& 1.3871242e+03,1.2701002e+03,1.1145962e+03,5.1058640e+02,1.1251127e+03, &
& 1.0801628e+03,1.0529974e+03,1.0280158e+03,1.0058586e+03,7.9051760e+02, &
& 8.9187640e+02,8.5997290e+02,9.0692140e+02,8.8762020e+02,8.7020630e+02, &
& 8.6003450e+02,7.2429040e+02,7.1147790e+02,6.4914710e+02,5.4842120e+02, &
& 5.5775610e+02,5.0549330e+02,4.6298700e+02,3.8491460e+02,3.5974270e+02, &
& 3.6989520e+02,5.3929820e+02,5.2677380e+02,4.8381430e+02,4.6139360e+02, &
& 4.2587210e+02,3.9047110e+02,1.3086904e+03,1.2553593e+03,1.1084934e+03]
 vdw_dftd3_c6(9001:9100)=[&
& 9.9548230e+02,9.8913640e+02,9.5777810e+02,9.8673520e+02,9.5542490e+02, &
& 5.3364800e+01,1.7297080e+02,2.2067460e+02,1.6859440e+02,1.2721250e+02, &
& 8.8661300e+01,6.4744800e+01,4.4708700e+01,2.5309000e+02,3.9136060e+02, &
& 3.9606340e+02,3.1817950e+02,2.6070120e+02,2.2064770e+02,1.8059160e+02, &
& 3.4947530e+02,6.6876460e+02,3.4907020e+02,3.3705530e+02,3.3035670e+02, &
& 3.2735530e+02,3.0103970e+02,2.7865260e+02,2.6587520e+02,2.5994720e+02, &
& 2.5694800e+02,2.4091240e+02,3.9335710e+02,3.4538720e+02,3.0960350e+02, &
& 2.8284600e+02,2.4934040e+02,4.1708310e+02,8.1419750e+02,6.2130580e+02, &
& 4.6191410e+02,4.6653580e+02,4.3510620e+02,4.8832640e+02,3.7926550e+02, &
& 3.1102400e+01,2.0777600e+01,4.8243420e+02,2.7741740e+02,1.8713270e+02, &
& 1.2690850e+02,8.9165900e+01,6.7811400e+01,5.1611700e+01,3.9876200e+01, &
& 5.7701810e+02,4.4233810e+02,4.0769990e+02,3.2145940e+02,2.5114770e+02, &
& 2.0890670e+02,1.7111870e+02,1.4042730e+02,9.4851400e+02,7.7905470e+02, &
& 6.4318930e+02,6.2118070e+02,5.6883480e+02,4.4875270e+02,4.8979330e+02, &
& 3.8501530e+02,4.0745740e+02,4.1967090e+02,3.2239880e+02,3.3047130e+02, &
& 3.9105080e+02,3.4491000e+02,2.9454210e+02,2.6467170e+02,2.3205800e+02, &
& 2.0225670e+02,1.0633477e+03,9.2882550e+02,8.1293380e+02,7.3049700e+02, &
& 6.6622590e+02,5.1491280e+02,5.7424560e+02,4.3804260e+02,4.7828480e+02, &
& 4.4362800e+02,3.7063780e+02,3.9106750e+02,4.9033250e+02,4.5317940e+02, &
& 4.0403840e+02,3.7513030e+02,3.3956200e+02,3.0572010e+02,1.2958507e+03]
 vdw_dftd3_c6(9101:9200)=[&
& 1.1851747e+03,1.0401066e+03,4.7759450e+02,1.0500763e+03,1.0080521e+03, &
& 9.8268750e+02,9.5935980e+02,9.3866880e+02,7.3813170e+02,8.3296700e+02, &
& 8.0325110e+02,8.4631030e+02,8.2828260e+02,8.1202140e+02,8.0250260e+02, &
& 6.7613200e+02,6.6419530e+02,6.0624630e+02,5.1256560e+02,5.2130610e+02, &
& 4.7268090e+02,4.3312920e+02,3.6041730e+02,3.3697140e+02,3.4645260e+02, &
& 5.0424040e+02,4.9256890e+02,4.5264510e+02,4.3185880e+02,3.9887680e+02, &
& 3.6599140e+02,1.2224485e+03,1.1715378e+03,1.0345633e+03,9.2953190e+02, &
& 9.2366730e+02,8.9441350e+02,9.2128530e+02,8.9206410e+02,4.9909000e+01, &
& 1.6134990e+02,2.0595300e+02,1.5759490e+02,1.1912650e+02,8.3222700e+01, &
& 6.0915900e+01,4.2208400e+01,2.3614440e+02,3.6507050e+02,3.6958140e+02, &
& 2.9723390e+02,2.4384390e+02,2.0662580e+02,1.6935840e+02,3.2659470e+02, &
& 6.2428500e+02,3.2625030e+02,3.1509820e+02,3.0884820e+02,3.0603400e+02, &
& 2.8151340e+02,2.6065670e+02,2.4873880e+02,2.4319090e+02,2.4032600e+02, &
& 2.2543980e+02,3.6731970e+02,3.2278160e+02,2.8959340e+02,2.6478150e+02, &
& 2.3366080e+02,3.8991830e+02,7.6024370e+02,5.8041450e+02,4.3182880e+02, &
& 4.3614400e+02,4.0693380e+02,4.5656180e+02,3.5492820e+02,3.3224250e+02, &
& 2.8966100e+01,1.9468900e+01,4.4391060e+02,2.5623270e+02,1.7343600e+02, &
& 1.1801370e+02,8.3171000e+01,6.3412500e+01,4.8382400e+01,3.7464700e+01, &
& 5.3116760e+02,4.0830220e+02,3.7687760e+02,2.9781740e+02,2.3321060e+02, &
& 1.9433730e+02,1.5949140e+02,1.3113380e+02,8.7329840e+02,7.1849830e+02]
 vdw_dftd3_c6(9201:9300)=[&
& 5.9349940e+02,5.7359720e+02,5.2547320e+02,4.1479180e+02,4.5273530e+02, &
& 3.5613440e+02,3.7699280e+02,3.8812000e+02,2.9838940e+02,3.0606330e+02, &
& 3.6187800e+02,3.1969460e+02,2.7350100e+02,2.4609210e+02,2.1609450e+02, &
& 1.8863390e+02,9.7940320e+02,8.5671690e+02,7.5059330e+02,6.7499290e+02, &
& 6.1597590e+02,4.7670900e+02,5.3137000e+02,4.0593980e+02,4.4306590e+02, &
& 4.1113610e+02,3.4367070e+02,3.6264320e+02,4.5405760e+02,4.2012680e+02, &
& 3.7509950e+02,3.4860740e+02,3.1592880e+02,2.8479640e+02,1.1937040e+03, &
& 1.0930044e+03,9.6013460e+02,4.4336290e+02,9.6887580e+02,9.3022500e+02, &
& 9.0685110e+02,8.8534770e+02,8.6627480e+02,6.8220730e+02,7.6879040e+02, &
& 7.4151490e+02,7.8120400e+02,7.6457050e+02,7.4957790e+02,7.4073920e+02, &
& 6.2471570e+02,6.1425310e+02,5.6115170e+02,4.7494930e+02,4.8315690e+02, &
& 4.3846770e+02,4.0208340e+02,3.3498540e+02,3.1334830e+02,3.2218840e+02, &
& 4.6735710e+02,4.5686740e+02,4.2032740e+02,4.0133490e+02,3.7107280e+02, &
& 3.4084580e+02,1.1269549e+03,1.0810405e+03,9.5553590e+02,8.5969500e+02, &
& 8.5393730e+02,8.2694450e+02,8.5119160e+02,8.2429470e+02,4.6348900e+01, &
& 1.4912090e+02,1.9048260e+02,1.4620200e+02,1.1079460e+02,7.7647900e+01, &
& 5.7004900e+01,3.9663600e+01,2.1823330e+02,3.3722560e+02,3.4179460e+02, &
& 2.7549250e+02,2.2646050e+02,1.9221720e+02,1.5785240e+02,3.0246350e+02, &
& 5.7613740e+02,3.0248980e+02,2.9221350e+02,2.8642780e+02,2.8377850e+02, &
& 2.6126350e+02,2.4201470e+02,2.3097880e+02,2.2581650e+02,2.2301540e+02]
 vdw_dftd3_c6(9301:9400)=[&
& 2.0945460e+02,3.4006750e+02,2.9932800e+02,2.6894270e+02,2.4619370e+02, &
& 2.1757380e+02,3.6130470e+02,7.0165500e+02,5.3685660e+02,4.0042090e+02, &
& 4.0443000e+02,3.7755350e+02,4.2312790e+02,3.2968260e+02,3.0870590e+02, &
& 2.8695190e+02,2.7951800e+01,1.8773100e+01,4.3260780e+02,2.4865730e+02, &
& 1.6783500e+02,1.1399930e+02,8.0266300e+01,6.1174800e+01,4.6672600e+01, &
& 3.6148600e+01,5.1756800e+02,3.9661250e+02,3.6552690e+02,2.8828630e+02, &
& 2.2539450e+02,1.8765970e+02,1.5390320e+02,1.2648250e+02,8.5100880e+02, &
& 6.9886180e+02,5.7696360e+02,5.5728800e+02,5.1036440e+02,4.0283830e+02, &
& 4.3950990e+02,3.4569950e+02,3.6569250e+02,3.7662810e+02,2.8955800e+02, &
& 2.9667280e+02,3.5080100e+02,3.0945430e+02,2.6439570e+02,2.3772170e+02, &
& 2.0860440e+02,1.8200160e+02,9.5413200e+02,8.3331680e+02,7.2933160e+02, &
& 6.5543290e+02,5.9785280e+02,4.6233830e+02,5.1549530e+02,3.9350880e+02, &
& 4.2951430e+02,3.9846620e+02,3.3315830e+02,3.5137120e+02,4.4027330e+02, &
& 4.0690290e+02,3.6287400e+02,3.3701310e+02,3.0521220e+02,2.7497440e+02, &
& 1.1625520e+03,1.0633544e+03,9.3316800e+02,4.2899800e+02,9.4235470e+02, &
& 9.0466880e+02,8.8190710e+02,8.6096980e+02,8.4239660e+02,6.6260290e+02, &
& 7.4772470e+02,7.2101030e+02,7.5950840e+02,7.4332210e+02,7.2872030e+02, &
& 7.2016540e+02,6.0683130e+02,5.9598520e+02,5.4409850e+02,4.6026700e+02, &
& 4.6810840e+02,4.2458230e+02,3.8918700e+02,3.2412400e+02,3.0315550e+02, &
& 3.1163540e+02,4.5304550e+02,4.4250950e+02,4.0670840e+02,3.8810010e+02]
 vdw_dftd3_c6(9401:9500)=[&
& 3.5859540e+02,3.2919900e+02,1.0968369e+03,1.0511515e+03,9.2833020e+02, &
& 8.3431790e+02,8.2919230e+02,8.0298090e+02,8.2716130e+02,8.0093320e+02, &
& 4.4783500e+01,1.4465630e+02,1.8465300e+02,1.4140600e+02,1.0702860e+02, &
& 7.4935400e+01,5.4985600e+01,3.8253900e+01,2.1184190e+02,3.2735090e+02, &
& 3.3138320e+02,2.6660080e+02,2.1886170e+02,1.8561970e+02,1.5232700e+02, &
& 2.9329110e+02,5.6012980e+02,2.9290630e+02,2.8293320e+02,2.7735250e+02, &
& 2.7484520e+02,2.5282980e+02,2.3415110e+02,2.2348050e+02,2.1850750e+02, &
& 2.1595810e+02,2.0256850e+02,3.2953870e+02,2.8964110e+02,2.5997060e+02, &
& 2.3782380e+02,2.1004110e+02,3.5022950e+02,6.8218170e+02,5.2087240e+02, &
& 3.8773360e+02,3.9166370e+02,3.6552610e+02,4.1006670e+02,3.1896650e+02, &
& 2.9866780e+02,2.7760230e+02,2.6861300e+02,2.7928100e+01,1.8860700e+01, &
& 4.0844290e+02,2.4209500e+02,1.6565870e+02,1.1343690e+02,8.0249200e+01, &
& 6.1323400e+01,4.6869200e+01,3.6340100e+01,4.8951290e+02,3.8417280e+02, &
& 3.5686540e+02,2.8421390e+02,2.2377830e+02,1.8701630e+02,1.5385920e+02, &
& 1.2673560e+02,8.0025930e+02,6.7026330e+02,5.5566790e+02,5.3855020e+02, &
& 4.9422370e+02,3.8999530e+02,4.2685790e+02,3.3576420e+02,3.5691720e+02, &
& 3.6686680e+02,2.8186750e+02,2.9078950e+02,3.4329430e+02,3.0514680e+02, &
& 2.6228400e+02,2.3657790e+02,2.0821130e+02,1.8208670e+02,8.9853420e+02, &
& 7.9845280e+02,7.0379920e+02,6.3513700e+02,5.8084120e+02,4.5097760e+02, &
& 5.0211680e+02,3.8494360e+02,4.2037460e+02,3.9057700e+02,3.2596620e+02]
 vdw_dftd3_c6(9501:9600)=[&
& 3.4505990e+02,4.3036070e+02,4.0034740e+02,3.5905550e+02,3.3449170e+02, &
& 3.0384100e+02,2.7444070e+02,1.0949943e+03,1.0161802e+03,8.9820920e+02, &
& 4.2389100e+02,9.0262940e+02,8.6759380e+02,8.4606160e+02,8.2622610e+02, &
& 8.0864130e+02,6.4073930e+02,7.1462060e+02,6.9004230e+02,7.3042860e+02, &
& 7.1501980e+02,7.0118290e+02,6.9275040e+02,5.8668300e+02,5.8027960e+02, &
& 5.3179240e+02,4.5093600e+02,4.5935930e+02,4.1790820e+02,3.8396000e+02, &
& 3.2036400e+02,2.9986260e+02,3.0870590e+02,4.4321180e+02,4.3519640e+02, &
& 4.0209000e+02,3.8476220e+02,3.5659780e+02,3.2818400e+02,1.0391920e+03, &
& 1.0081792e+03,8.9630090e+02,8.1114820e+02,8.0315200e+02,7.7787630e+02, &
& 7.9775990e+02,7.7311930e+02,4.4500600e+01,1.4109010e+02,1.8052070e+02, &
& 1.3987930e+02,1.0643320e+02,7.4892500e+01,5.5139200e+01,3.8490600e+01, &
& 2.0596470e+02,3.1790020e+02,3.2408300e+02,2.6314380e+02,2.1731990e+02, &
& 1.8495220e+02,1.5226460e+02,2.8627350e+02,5.3708890e+02,2.8838750e+02, &
& 2.7858690e+02,2.7303310e+02,2.7028900e+02,2.4982010e+02,2.3161700e+02, &
& 2.2102480e+02,2.1602110e+02,2.1274680e+02,2.0082240e+02,3.2310800e+02, &
& 2.8605510e+02,2.5796890e+02,2.3666040e+02,2.0960830e+02,3.4234100e+02, &
& 6.5327820e+02,5.0567470e+02,3.8134050e+02,3.8512060e+02,3.5990590e+02, &
& 4.0123820e+02,3.1533650e+02,2.9530360e+02,2.7461540e+02,2.6554500e+02, &
& 2.6345080e+02,4.2282700e+01,2.7023600e+01,7.6925140e+02,4.1349650e+02, &
& 2.6747500e+02,1.7545850e+02,1.2012790e+02,8.9652000e+01,6.7123200e+01]
 vdw_dftd3_c6(9601:9700)=[&
& 5.1174700e+01,9.1594130e+02,6.6728440e+02,6.0241430e+02,4.6150930e+02, &
& 3.5151570e+02,2.8739220e+02,2.3145540e+02,1.8709160e+02,1.5173687e+03, &
& 1.1989287e+03,9.8084640e+02,9.3876410e+02,8.5507440e+02,6.7285260e+02, &
& 7.3052210e+02,5.7217290e+02,5.9982550e+02,6.2133050e+02,4.7601650e+02, &
& 4.8059720e+02,5.7226580e+02,4.9379840e+02,4.1301260e+02,3.6615190e+02, &
& 3.1652310e+02,2.7222880e+02,1.6942091e+03,1.4306640e+03,1.2316979e+03, &
& 1.0950419e+03,9.9140130e+02,7.5589180e+02,8.4735570e+02,6.3674800e+02, &
& 6.9613210e+02,6.4264560e+02,5.3720580e+02,5.6303120e+02,7.1679570e+02, &
& 6.5114030e+02,5.7030150e+02,5.2361310e+02,4.6816800e+02,4.1650530e+02, &
& 2.0626273e+03,1.8342595e+03,1.5838792e+03,6.7579110e+02,1.6159882e+03, &
& 1.5472492e+03,1.5072236e+03,1.4705492e+03,1.4379844e+03,1.1092250e+03, &
& 1.2845648e+03,1.2348582e+03,1.2913513e+03,1.2633503e+03,1.2378116e+03, &
& 1.2243143e+03,1.0181407e+03,9.8356470e+02,8.8792510e+02,7.4340350e+02, &
& 7.5315120e+02,6.7623400e+02,6.1465820e+02,5.0654420e+02,4.7177480e+02, &
& 4.8363130e+02,7.3258080e+02,7.0644460e+02,6.3913150e+02,6.0410820e+02, &
& 5.5158440e+02,5.0067860e+02,1.9210063e+03,1.7975569e+03,1.5637265e+03, &
& 1.3796361e+03,1.3822903e+03,1.3377839e+03,1.3931101e+03,1.3462697e+03, &
& 6.9675400e+01,2.3904190e+02,3.0266780e+02,2.2316290e+02,1.6468120e+02, &
& 1.1197070e+02,8.0098600e+01,5.3936400e+01,3.5200740e+02,5.4627620e+02, &
& 5.4321880e+02,4.2482930e+02,3.4093960e+02,2.8433500e+02,2.2912290e+02]
 vdw_dftd3_c6(9701:9800)=[&
& 4.7801410e+02,9.5798970e+02,4.6767000e+02,4.5116580e+02,4.4234200e+02, &
& 4.3947280e+02,3.9910790e+02,3.6795920e+02,3.5103000e+02,3.4356200e+02, &
& 3.4289310e+02,3.1591660e+02,5.3451270e+02,4.5962850e+02,4.0554530e+02, &
& 3.6636520e+02,3.1893050e+02,5.6760020e+02,1.1685036e+03,8.6288330e+02, &
& 6.1965370e+02,6.2602540e+02,5.8105490e+02,6.6303800e+02,5.0016280e+02, &
& 4.6737710e+02,4.3299730e+02,4.1968680e+02,4.1115760e+02,6.7899200e+02, &
& 3.9050000e+01,2.5410500e+01,6.4241900e+02,3.6274340e+02,2.4059340e+02, &
& 1.6058760e+02,1.1127680e+02,8.3712000e+01,6.3080600e+01,4.8325800e+01, &
& 7.6709550e+02,5.8045000e+02,5.3107940e+02,4.1414030e+02,3.1993070e+02, &
& 2.6382770e+02,2.1417230e+02,1.7426390e+02,1.2595964e+03,1.0266097e+03, &
& 8.4554470e+02,8.1389640e+02,7.4392530e+02,5.8540690e+02,6.3875870e+02, &
& 5.0063040e+02,5.2899650e+02,5.4604730e+02,4.1817300e+02,4.2710250e+02, &
& 5.0688380e+02,4.4348920e+02,3.7536940e+02,3.3511820e+02,2.9171790e+02, &
& 2.5244080e+02,1.4095386e+03,1.2233073e+03,1.0654342e+03,9.5393220e+02, &
& 8.6754750e+02,6.6641970e+02,7.4499930e+02,5.6445890e+02,6.1737910e+02, &
& 5.7156020e+02,4.7659660e+02,5.0255130e+02,6.3411950e+02,5.8276930e+02, &
& 5.1594670e+02,4.7668110e+02,4.2899630e+02,3.8394570e+02,1.7161526e+03, &
& 1.5618289e+03,1.3644290e+03,6.1000420e+02,1.3811677e+03,1.3251151e+03, &
& 1.2915707e+03,1.2607608e+03,1.2334280e+03,9.6322340e+02,1.0940703e+03, &
& 1.0539748e+03,1.1110005e+03,1.0872901e+03,1.0658196e+03,1.0536773e+03]
 vdw_dftd3_c6(9801:9900)=[&
& 8.8359210e+02,8.6380750e+02,7.8517710e+02,6.6055390e+02,6.7110130e+02, &
& 6.0606460e+02,5.5342010e+02,4.5807960e+02,4.2739480e+02,4.3923750e+02, &
& 6.4947520e+02,6.3216060e+02,5.7757500e+02,5.4893280e+02,5.0441400e+02, &
& 4.6043530e+02,1.6132050e+03,1.5396549e+03,1.3539743e+03,1.2088936e+03, &
& 1.2037043e+03,1.1652987e+03,1.2045686e+03,1.1657062e+03,6.3518900e+01, &
& 2.1038150e+02,2.6750360e+02,2.0167580e+02,1.5060400e+02,1.0370700e+02, &
& 7.4928900e+01,5.1049200e+01,3.0824700e+02,4.7728480e+02,4.8037350e+02, &
& 3.8212550e+02,3.1042380e+02,2.6094970e+02,2.1197090e+02,4.2200550e+02, &
& 8.1999260e+02,4.1914320e+02,4.0443780e+02,3.9640790e+02,3.9312150e+02, &
& 3.6011140e+02,3.3275950e+02,3.1739780e+02,3.1042370e+02,3.0781940e+02, &
& 2.8694640e+02,4.7510140e+02,4.1406130e+02,3.6881960e+02,3.3526670e+02, &
& 2.9382390e+02,5.0249860e+02,9.9819190e+02,7.5411460e+02,5.5441070e+02, &
& 5.5998500e+02,5.2115630e+02,5.8793550e+02,4.5216820e+02,4.2274380e+02, &
& 3.9214710e+02,3.7963350e+02,3.7434300e+02,6.0309650e+02,5.4204390e+02, &
& 3.7118600e+01,2.4506800e+01,5.6950200e+02,3.3206650e+02,2.2424380e+02, &
& 1.5163340e+02,1.0605380e+02,8.0289900e+01,6.0817400e+01,4.6777500e+01, &
& 6.8136500e+02,5.2835450e+02,4.8798470e+02,3.8531020e+02,3.0075580e+02, &
& 2.4964690e+02,2.0391030e+02,1.6677960e+02,1.1139369e+03,9.2532660e+02, &
& 7.6543030e+02,7.3977510e+02,6.7780140e+02,5.3372530e+02,5.8400790e+02, &
& 4.5821350e+02,4.8646450e+02,5.0090110e+02,3.8380100e+02,3.9484340e+02]
 vdw_dftd3_c6(9901:10000)=[&
& 4.6751190e+02,4.1296610e+02,3.5254270e+02,3.1639840e+02,2.7688560e+02, &
& 2.4075150e+02,1.2488109e+03,1.1019644e+03,9.6716720e+02,8.7013340e+02, &
& 7.9387710e+02,6.1324700e+02,6.8412920e+02,5.2150710e+02,5.7027900e+02, &
& 5.2899540e+02,4.4070410e+02,4.6629410e+02,5.8471100e+02,5.4152090e+02, &
& 4.8305960e+02,4.4833330e+02,4.0544000e+02,3.6450640e+02,1.5212386e+03, &
& 1.4035743e+03,1.2355723e+03,5.7038460e+02,1.2443690e+03,1.1953394e+03, &
& 1.1654811e+03,1.1380116e+03,1.1136574e+03,8.7723370e+02,9.8436690e+02, &
& 9.4971600e+02,1.0050059e+03,9.8375110e+02,9.6460520e+02,9.5326960e+02, &
& 8.0412310e+02,7.9227200e+02,7.2359370e+02,6.1110290e+02,6.2195650e+02, &
& 5.6397010e+02,5.1667170e+02,4.2917250e+02,4.0097810e+02,4.1268870e+02, &
& 6.0016760e+02,5.8760720e+02,5.4048320e+02,5.1568470e+02,4.7605910e+02, &
& 4.3636860e+02,1.4387730e+03,1.3892118e+03,1.2302614e+03,1.1073931e+03, &
& 1.0983173e+03,1.0634833e+03,1.0937836e+03,1.0594592e+03,5.9779600e+01, &
& 1.9308010e+02,2.4635800e+02,1.8865770e+02,1.4217370e+02,9.8858300e+01, &
& 7.1981600e+01,4.9497000e+01,2.8199230e+02,4.3599230e+02,4.4239460e+02, &
& 3.5616810e+02,2.9192250e+02,2.4688440e+02,2.0178980e+02,3.8878440e+02, &
& 7.3983800e+02,3.8986450e+02,3.7633360e+02,3.6879080e+02,3.6529400e+02, &
& 3.3651780e+02,3.1148590e+02,2.9711100e+02,2.9044330e+02,2.8674360e+02, &
& 2.6942080e+02,4.3926920e+02,3.8642600e+02,3.4656560e+02,3.1651250e+02, &
& 2.7880760e+02,4.6396320e+02,8.9977960e+02,6.9024350e+02,5.1535410e+02]
 vdw_dftd3_c6(10001:10100)=[&
& 5.2041820e+02,4.8533740e+02,5.4348220e+02,4.2337510e+02,3.9603590e+02, &
& 3.6774930e+02,3.5570800e+02,3.5221320e+02,5.5762720e+02,5.0519000e+02, &
& 4.7347030e+02,3.5486100e+01,2.3710000e+01,5.1945180e+02,3.0894260e+02, &
& 2.1127990e+02,1.4426000e+02,1.0163150e+02,7.7336500e+01,5.8833200e+01, &
& 4.5404400e+01,6.2230990e+02,4.8978200e+02,4.5524770e+02,3.6255610e+02, &
& 2.8512390e+02,2.3785710e+02,1.9521310e+02,1.6033310e+02,1.0157738e+03, &
& 8.5290360e+02,7.0737690e+02,6.8556870e+02,6.2913900e+02,4.9588010e+02, &
& 5.4334380e+02,4.2683610e+02,4.5431700e+02,4.6700090e+02,3.5818700e+02, &
& 3.7006740e+02,4.3744170e+02,3.8889460e+02,3.3402630e+02,3.0096170e+02, &
& 2.6444460e+02,2.3079470e+02,1.1403397e+03,1.0156122e+03,8.9573040e+02, &
& 8.0842520e+02,7.3920970e+02,5.7337000e+02,6.3864620e+02,4.8901000e+02, &
& 5.3445110e+02,4.9643130e+02,4.1359130e+02,4.3835340e+02,5.4729130e+02, &
& 5.0938490e+02,4.5675890e+02,4.2531520e+02,3.8599880e+02,3.4821640e+02, &
& 1.3899562e+03,1.2920505e+03,1.1428009e+03,5.3897800e+02,1.1473718e+03, &
& 1.1029411e+03,1.0755989e+03,1.0504168e+03,1.0280990e+03,8.1457180e+02, &
& 9.0749290e+02,8.7640540e+02,9.2881650e+02,9.0926160e+02,8.9171060e+02, &
& 8.8100370e+02,7.4613080e+02,7.3873540e+02,6.7688650e+02,5.7338870e+02, &
& 5.8418960e+02,5.3122090e+02,4.8779780e+02,4.0633480e+02,3.8005290e+02, &
& 3.9145330e+02,5.6288150e+02,5.5306450e+02,5.1099880e+02,4.8888150e+02, &
& 4.5282060e+02,4.1635580e+02,1.3196282e+03,1.2821564e+03,1.1403381e+03]
 vdw_dftd3_c6(10101:10200)=[&
& 1.0319646e+03,1.0211167e+03,9.8886570e+02,1.0136901e+03,9.8242950e+02, &
& 5.6715600e+01,1.7998120e+02,2.3026410e+02,1.7825090e+02,1.3527200e+02, &
& 9.4770500e+01,6.9431700e+01,4.8103700e+01,2.6236970e+02,4.0525290e+02, &
& 4.1338560e+02,3.3559140e+02,2.7683720e+02,2.3520590e+02,1.9317040e+02, &
& 3.6384820e+02,6.8296210e+02,3.6699000e+02,3.5440290e+02,3.4726480e+02, &
& 3.4370870e+02,3.1777500e+02,2.9450790e+02,2.8094820e+02,2.7455800e+02, &
& 2.7027830e+02,2.5527130e+02,4.1170180e+02,3.6448820e+02,3.2849090e+02, &
& 3.0105670e+02,2.6622780e+02,4.3493910e+02,8.3036610e+02,6.4326970e+02, &
& 4.8508560e+02,4.8977500e+02,4.5747170e+02,5.0987810e+02,4.0053650e+02, &
& 3.7486560e+02,3.4838020e+02,3.3677710e+02,3.3432410e+02,5.2288630e+02, &
& 4.7620870e+02,4.4801730e+02,4.2510150e+02,3.2815100e+01,2.2271700e+01, &
& 4.5439540e+02,2.7643090e+02,1.9192730e+02,1.3262510e+02,9.4299100e+01, &
& 7.2235800e+01,5.5268900e+01,4.2851200e+01,5.4528550e+02,4.3644520e+02, &
& 4.0871490e+02,3.2881570e+02,2.6094460e+02,2.1904300e+02,1.8086040e+02, &
& 1.4934300e+02,8.8900260e+02,7.5534780e+02,6.2833170e+02,6.1099490e+02, &
& 5.6176430e+02,4.4344010e+02,4.8650250e+02,3.8289000e+02,4.0860640e+02, &
& 4.1915920e+02,3.2203460e+02,3.3424140e+02,3.9422890e+02,3.5311560e+02, &
& 3.0552140e+02,2.7661090e+02,2.4427520e+02,2.1420040e+02,9.9978710e+02, &
& 8.9949300e+02,7.9779010e+02,7.2272400e+02,6.6260680e+02,5.1659320e+02, &
& 5.7428050e+02,4.4217100e+02,4.8282170e+02,4.4920610e+02,3.7444670e+02]
 vdw_dftd3_c6(10201:10300)=[&
& 3.9749790e+02,4.9366250e+02,4.6207160e+02,4.1687150e+02,3.8970290e+02, &
& 3.5521750e+02,3.2180290e+02,1.2196272e+03,1.1429378e+03,1.0164155e+03, &
& 4.9161500e+02,1.0169588e+03,9.7834710e+02,9.5429670e+02,9.3211660e+02, &
& 9.1246710e+02,7.2800530e+02,8.0453430e+02,7.7785480e+02,8.2536440e+02, &
& 8.0806540e+02,7.9260610e+02,7.8284050e+02,6.6614790e+02,6.6319200e+02, &
& 6.1001430e+02,5.1873530e+02,5.2912210e+02,4.8278010e+02,4.4456400e+02, &
& 3.7168100e+02,3.4813080e+02,3.5883830e+02,5.0911500e+02,5.0219660e+02, &
& 4.6641830e+02,4.4766490e+02,4.1630430e+02,3.8423110e+02,1.1630702e+03, &
& 1.1376470e+03,1.0167803e+03,9.2598120e+02,9.1390750e+02,8.8520210e+02, &
& 9.0396880e+02,8.7665190e+02,5.1949600e+01,1.6143030e+02,2.0721850e+02, &
& 1.6249390e+02,1.2439580e+02,8.7986900e+01,6.4976400e+01,4.5461800e+01, &
& 2.3487760e+02,3.6232580e+02,3.7186150e+02,3.0487150e+02,2.5346770e+02, &
& 2.1658660e+02,1.7895810e+02,3.2822070e+02,6.0618590e+02,3.3318010e+02, &
& 3.2194330e+02,3.1543490e+02,3.1193950e+02,2.8957840e+02,2.6879340e+02, &
& 2.5647210e+02,2.5055360e+02,2.4584350e+02,2.3356320e+02,3.7176550e+02, &
& 3.3159870e+02,3.0060430e+02,2.7668620e+02,2.4586470e+02,3.9318090e+02, &
& 7.3690210e+02,5.7738050e+02,4.4044740e+02,4.4464020e+02,4.1612070e+02, &
& 4.6132230e+02,3.6592820e+02,3.4272790e+02,3.1886290e+02,3.0804750e+02, &
& 3.0665490e+02,4.7264410e+02,4.3305890e+02,4.0924850e+02,3.8959960e+02, &
& 3.5849950e+02,4.2305800e+01,2.7469500e+01,7.3291200e+02,3.9940200e+02]
 vdw_dftd3_c6(10301:10400)=[&
& 2.6223870e+02,1.7427180e+02,1.2049820e+02,9.0530300e+01,6.8123900e+01, &
& 5.2104100e+01,8.7326650e+02,6.4218460e+02,5.8368500e+02,4.5161710e+02, &
& 3.4740300e+02,2.8604180e+02,2.3194830e+02,1.8859500e+02,1.4495860e+03, &
& 1.1498295e+03,9.4227450e+02,9.0422250e+02,8.2477800e+02,6.5006270e+02, &
& 7.0614120e+02,5.5409250e+02,5.8184580e+02,6.0158080e+02,4.6170190e+02, &
& 4.6787310e+02,5.5650160e+02,4.8371170e+02,4.0779580e+02,3.6352860e+02, &
& 3.1607000e+02,2.7329990e+02,1.6207604e+03,1.3728515e+03,1.1863885e+03, &
& 1.0577660e+03,9.5979080e+02,7.3514980e+02,8.2258290e+02,6.2119680e+02, &
& 6.7827670e+02,6.2700570e+02,5.2450750e+02,5.5025650e+02,6.9737450e+02, &
& 6.3676120e+02,5.6125330e+02,5.1758070e+02,4.6507030e+02,4.1576900e+02, &
& 1.9759738e+03,1.7600602e+03,1.5249399e+03,6.6483460e+02,1.5520845e+03, &
& 1.4864897e+03,1.4481464e+03,1.4129908e+03,1.3817863e+03,1.0715717e+03, &
& 1.2355701e+03,1.1888012e+03,1.2416267e+03,1.2147181e+03,1.1902611e+03, &
& 1.1769746e+03,9.8243040e+02,9.5327210e+02,8.6345810e+02,7.2569910e+02, &
& 7.3587000e+02,6.6279760e+02,6.0405240e+02,4.9959380e+02,4.6588050e+02, &
& 4.7788590e+02,7.1477590e+02,6.9152340e+02,6.2890510e+02,5.9653870e+02, &
& 5.4712200e+02,4.9878710e+02,1.8443405e+03,1.7284117e+03,1.5078114e+03, &
& 1.3366572e+03,1.3367673e+03,1.2938174e+03,1.3431115e+03,1.2984675e+03, &
& 6.8953700e+01,2.3141650e+02,2.9414590e+02,2.1963870e+02,1.6367150e+02, &
& 1.1242780e+02,8.1084500e+01,5.5078600e+01,3.4006240e+02,5.2756300e+02]
 vdw_dftd3_c6(10401:10500)=[&
& 5.2732820e+02,4.1648610e+02,3.3711350e+02,2.8296690e+02,2.2958370e+02, &
& 4.6551820e+02,9.2179340e+02,4.5790120e+02,4.4205470e+02,4.3331390e+02, &
& 4.3010400e+02,3.9208310e+02,3.6204550e+02,3.4544700e+02,3.3794540e+02, &
& 3.3608210e+02,3.1158000e+02,5.2074900e+02,4.5117170e+02,4.0064390e+02, &
& 3.6372020e+02,3.1838270e+02,5.5397800e+02,1.1249345e+03,8.3761350e+02, &
& 6.0719510e+02,6.1320100e+02,5.7026210e+02,6.4766860e+02,4.9283010e+02, &
& 4.6081940e+02,4.2732470e+02,4.1377960e+02,4.0654590e+02,6.6184230e+02, &
& 5.9127640e+02,5.4930440e+02,5.1701090e+02,4.6948180e+02,6.4909240e+02, &
& 8.3014100e+01,4.9923200e+01,2.1804015e+03,9.6110490e+02,5.7230020e+02, &
& 3.5484660e+02,2.3350060e+02,1.6960300e+02,1.2408020e+02,9.2849400e+01, &
& 2.5659398e+03,1.5996001e+03,1.3814389e+03,9.9489260e+02,7.2325040e+02, &
& 5.7533170e+02,4.5158250e+02,3.5719350e+02,4.4958369e+03,3.0704624e+03, &
& 2.4434277e+03,2.2946062e+03,2.0640309e+03,1.6327022e+03,1.7320375e+03, &
& 1.3596156e+03,1.3759674e+03,1.4425133e+03,1.1138541e+03,1.0720643e+03, &
& 1.2954563e+03,1.0627315e+03,8.5391570e+02,7.4014960e+02,6.2555040e+02, &
& 5.2735210e+02,4.9979230e+03,3.6976569e+03,3.0451620e+03,2.6376782e+03, &
& 2.3521278e+03,1.7503829e+03,1.9809508e+03,1.4476221e+03,1.5723023e+03, &
& 1.4360400e+03,1.2194904e+03,1.2412980e+03,1.6356158e+03,1.4200022e+03, &
& 1.1973790e+03,1.0767503e+03,9.4225240e+02,8.2205680e+02,6.1419784e+03, &
& 4.8451776e+03,3.9942273e+03,1.4330469e+03,4.2139225e+03,3.9873600e+03]
 vdw_dftd3_c6(10501:10600)=[&
& 3.8737284e+03,3.7706523e+03,3.6789146e+03,2.7207653e+03,3.4419852e+03, &
& 3.2989422e+03,3.2598698e+03,3.1836011e+03,3.1125536e+03,3.0835002e+03, &
& 2.5019262e+03,2.2987008e+03,2.0237991e+03,1.6732640e+03,1.6751136e+03, &
& 1.4722933e+03,1.3162117e+03,1.0706911e+03,9.9139190e+02,1.0048014e+03, &
& 1.6688986e+03,1.5461030e+03,1.3487488e+03,1.2508513e+03,1.1176702e+03, &
& 9.9586790e+02,5.4979393e+03,4.6433162e+03,3.8656041e+03,3.2597257e+03, &
& 3.3538826e+03,3.2402913e+03,3.4630576e+03,3.3272478e+03,1.4256800e+02, &
& 5.5107920e+02,6.9284150e+02,4.7105030e+02,3.3492070e+02,2.1815960e+02, &
& 1.5070560e+02,9.7192600e+01,8.2695650e+02,1.2989425e+03,1.2335817e+03, &
& 9.0965670e+02,7.0099830e+02,5.6985090e+02,4.4736800e+02,1.1041068e+03, &
& 2.5056537e+03,1.0130298e+03,9.8236380e+02,9.6327250e+02,9.6365340e+02, &
& 8.4521190e+02,7.7366980e+02,7.3950900e+02,7.2552030e+02,7.4098720e+02, &
& 6.5373360e+02,1.1948773e+03,9.7957420e+02,8.3693640e+02,7.4102200e+02, &
& 6.3113970e+02,1.3013999e+03,3.0957594e+03,2.1010372e+03,1.3604000e+03, &
& 1.3693825e+03,1.2660803e+03,1.5162134e+03,1.0564422e+03,9.8676430e+02, &
& 9.1052680e+02,8.8540390e+02,8.4515800e+02,1.5259278e+03,1.2966408e+03, &
& 1.1653603e+03,1.0738698e+03,9.5154100e+02,1.4690705e+03,4.1717842e+03, &
& 6.4690600e+01,4.0332900e+01,1.3841283e+03,6.7673740e+02,4.2308150e+02, &
& 2.7142480e+02,1.8293910e+02,1.3503220e+02,1.0010900e+02,7.5679100e+01, &
& 1.6377878e+03,1.1067221e+03,9.8043760e+02,7.3212110e+02,5.4746530e+02]
 vdw_dftd3_c6(10601:10700)=[&
& 4.4287990e+02,3.5314670e+02,2.8303510e+02,2.8009569e+03,2.0504253e+03, &
& 1.6557011e+03,1.5712363e+03,1.4231173e+03,1.1226576e+03,1.2061393e+03, &
& 9.4568080e+02,9.7598360e+02,1.0161745e+03,7.8128810e+02,7.7266500e+02, &
& 9.2637230e+02,7.8274010e+02,6.4435600e+02,5.6629200e+02,4.8529550e+02, &
& 4.1416640e+02,3.1215559e+03,2.4580845e+03,2.0727978e+03,1.8212145e+03, &
& 1.6379243e+03,1.2356226e+03,1.3909989e+03,1.0325864e+03,1.1254492e+03, &
& 1.0341628e+03,8.7041020e+02,9.0076730e+02,1.1640480e+03,1.0373050e+03, &
& 8.9470260e+02,8.1481140e+02,7.2249320e+02,6.3795900e+02,3.8250881e+03, &
& 3.1862921e+03,2.6909591e+03,1.0645933e+03,2.7909891e+03,2.6551171e+03, &
& 2.5829722e+03,2.5172239e+03,2.4587886e+03,1.8607370e+03,2.2509654e+03, &
& 2.1634599e+03,2.1938636e+03,2.1444704e+03,2.0989702e+03,2.0775774e+03, &
& 1.7103545e+03,1.6140907e+03,1.4413394e+03,1.2005662e+03,1.2100818e+03, &
& 1.0767583e+03,9.7197370e+02,7.9670010e+02,7.4016750e+02,7.5524870e+02, &
& 1.1891662e+03,1.1270513e+03,1.0045839e+03,9.4243520e+02,8.5327010e+02, &
& 7.6899820e+02,3.4855386e+03,3.0892071e+03,2.6322561e+03,2.2749680e+03, &
& 2.3071665e+03,2.2309267e+03,2.3502253e+03,2.2650122e+03,1.0830590e+02, &
& 3.8990700e+02,4.9258240e+02,3.5111660e+02,2.5535340e+02,1.7069380e+02, &
& 1.2040040e+02,7.9615100e+01,5.7870070e+02,9.0345630e+02,8.8049000e+02, &
& 6.7215910e+02,5.3085730e+02,4.3834960e+02,3.4967640e+02,7.8068060e+02, &
& 1.6583459e+03,7.4263980e+02,7.1867820e+02,7.0447310e+02,7.0187680e+02]
 vdw_dftd3_c6(10701:10800)=[&
& 6.2823400e+02,5.7752630e+02,5.5142260e+02,5.4016790e+02,5.4406610e+02, &
& 4.9263420e+02,8.6066320e+02,7.2574740e+02,6.3226020e+02,5.6675130e+02, &
& 4.8922910e+02,9.2422030e+02,2.0365362e+03,1.4462394e+03,9.9010590e+02, &
& 9.9792220e+02,9.2583590e+02,1.0787908e+03,7.8658860e+02,7.3489180e+02, &
& 6.7965720e+02,6.5947100e+02,6.3938320e+02,1.0954590e+03,9.5522580e+02, &
& 8.7322870e+02,8.1328280e+02,7.2949490e+02,1.0633274e+03,2.7055816e+03, &
& 1.8500003e+03,4.3220200e+01,2.8506000e+01,6.8024790e+02,3.9079830e+02, &
& 2.6231920e+02,1.7683390e+02,1.2349550e+02,9.3432700e+01,7.0748100e+01, &
& 5.4406100e+01,8.1323530e+02,6.2332230e+02,5.7359020e+02,4.5092530e+02, &
& 3.5095780e+02,2.9093690e+02,2.3740720e+02,1.9406980e+02,1.3337958e+03, &
& 1.0971659e+03,9.0572410e+02,8.7400660e+02,8.0001520e+02,6.3019610e+02, &
& 6.8836930e+02,5.4023110e+02,5.7207380e+02,5.8956950e+02,4.5203420e+02, &
& 4.6341480e+02,5.4906920e+02,4.8331740e+02,4.1155330e+02,3.6892060e+02, &
& 3.2252720e+02,2.8024380e+02,1.4944074e+03,1.3073665e+03,1.1435524e+03, &
& 1.0268226e+03,9.3574670e+02,7.2166730e+02,8.0551110e+02,6.1296820e+02, &
& 6.6999170e+02,6.2106870e+02,5.1802070e+02,5.4699380e+02,6.8726670e+02, &
& 6.3451560e+02,5.6457530e+02,5.2332050e+02,4.7270040e+02,4.2459150e+02, &
& 1.8204201e+03,1.6674685e+03,1.4628142e+03,6.6720180e+02,1.4767389e+03, &
& 1.4176855e+03,1.3820320e+03,1.3492524e+03,1.3201809e+03,1.0364403e+03, &
& 1.1696897e+03,1.1277999e+03,1.1902918e+03,1.1649825e+03,1.1421370e+03]
 vdw_dftd3_c6(10801:10900)=[&
& 1.1288548e+03,9.5004210e+02,9.3285060e+02,8.5051620e+02,7.1766720e+02, &
& 7.2982300e+02,6.6090480e+02,6.0487530e+02,5.0215100e+02,4.6904220e+02, &
& 4.8234510e+02,7.0537820e+02,6.8881010e+02,6.3203130e+02,6.0229480e+02, &
& 5.5531140e+02,5.0853060e+02,1.7169108e+03,1.6476121e+03,1.4543881e+03, &
& 1.3048970e+03,1.2966238e+03,1.2554359e+03,1.2939094e+03,1.2527870e+03, &
& 6.9734100e+01,2.2705790e+02,2.8944340e+02,2.2051700e+02,1.6588050e+02, &
& 1.1516050e+02,8.3776500e+01,5.7559900e+01,3.3214950e+02,5.1379560e+02, &
& 5.1961650e+02,4.1662720e+02,3.4064700e+02,2.8774490e+02,2.3495440e+02, &
& 4.5743380e+02,8.7755280e+02,4.5670870e+02,4.4087260e+02,4.3208390e+02, &
& 4.2819360e+02,3.9355560e+02,3.6411700e+02,3.4735660e+02,3.3962320e+02, &
& 3.3586060e+02,3.1461930e+02,5.1545120e+02,4.5195710e+02,4.0453400e+02, &
& 3.6906970e+02,3.2478940e+02,5.4559170e+02,1.0680684e+03,8.1406370e+02, &
& 6.0412500e+02,6.1013660e+02,5.6871480e+02,6.3873760e+02,4.9520560e+02, &
& 4.6324120e+02,4.3008630e+02,4.1612810e+02,4.1129410e+02,6.5474970e+02, &
& 5.9136710e+02,5.5320050e+02,5.2290670e+02,4.7714300e+02,6.4438580e+02, &
& 1.3856402e+03,1.0306163e+03,6.4702500e+02,4.2334300e+01,2.7932400e+01, &
& 6.6553290e+02,3.8257600e+02,2.5686460e+02,1.7319330e+02,1.2097590e+02, &
& 9.1540700e+01,6.9326100e+01,5.3320100e+01,7.9568730e+02,6.1016250e+02, &
& 5.6154980e+02,4.4153690e+02,3.4370190e+02,2.8495290e+02,2.3255130e+02, &
& 1.9012240e+02,1.3047563e+03,1.0738235e+03,8.8652660e+02,8.5553540e+02]
 vdw_dftd3_c6(10901:11000)=[&
& 7.8313860e+02,6.1691000e+02,6.7388620e+02,5.2887800e+02,5.6009170e+02, &
& 5.7719940e+02,4.4255790e+02,4.5374620e+02,5.3757900e+02,4.7326730e+02, &
& 4.0304410e+02,3.6132220e+02,3.1591320e+02,2.7452240e+02,1.4619055e+03, &
& 1.2795273e+03,1.1193527e+03,1.0051742e+03,9.1606480e+02,7.0655730e+02, &
& 7.8861510e+02,6.0017770e+02,6.5600920e+02,6.0812890e+02,5.0722690e+02, &
& 5.3562280e+02,6.7289960e+02,6.2132160e+02,5.5289340e+02,5.1252450e+02, &
& 4.6298250e+02,4.1589380e+02,1.7807379e+03,1.6318559e+03,1.4317796e+03, &
& 6.5338750e+02,1.4452611e+03,1.3875284e+03,1.3526449e+03,1.3205717e+03, &
& 1.2921268e+03,1.0145593e+03,1.1446770e+03,1.1036788e+03,1.1650450e+03, &
& 1.1402779e+03,1.1179234e+03,1.1049164e+03,9.2996320e+02,9.1326380e+02, &
& 8.3272290e+02,7.0269660e+02,7.1461940e+02,6.4718140e+02,5.9234880e+02, &
& 4.9178730e+02,4.5937630e+02,4.7241230e+02,6.9065930e+02,6.7450000e+02, &
& 6.1895970e+02,5.8986940e+02,5.4389180e+02,4.9810570e+02,1.6797710e+03, &
& 1.6125437e+03,1.4236214e+03,1.2774745e+03,1.2692909e+03,1.2289809e+03, &
& 1.2665506e+03,1.2263194e+03,6.8293100e+01,2.2228830e+02,2.8337150e+02, &
& 2.1594380e+02,1.6246650e+02,1.1281340e+02,8.2084900e+01,5.6411700e+01, &
& 3.2516360e+02,5.0296550e+02,5.0872500e+02,4.0796290e+02,3.3360650e+02, &
& 2.8182650e+02,2.3014880e+02,4.4786360e+02,8.5883830e+02,4.4722200e+02, &
& 4.3170970e+02,4.2310550e+02,4.1928980e+02,3.8540430e+02,3.5658500e+02, &
& 3.4017190e+02,3.3259710e+02,3.2889540e+02,3.0812510e+02,5.0468300e+02]
 vdw_dftd3_c6(11001:11100)=[&
& 4.4257280e+02,3.9617290e+02,3.6146790e+02,3.1812780e+02,5.3419470e+02, &
& 1.0452541e+03,7.9688440e+02,5.9156300e+02,5.9746030e+02,5.5690660e+02, &
& 6.2539660e+02,4.8497040e+02,4.5367340e+02,4.2121350e+02,4.0754200e+02, &
& 4.0282810e+02,6.4108970e+02,5.7909510e+02,5.4176140e+02,5.1212090e+02, &
& 4.6733270e+02,6.3096990e+02,1.3558874e+03,1.0088176e+03,6.3363330e+02, &
& 6.2052060e+02,4.1035600e+01,2.7115000e+01,6.4218080e+02,3.6981420e+02, &
& 2.4860710e+02,1.6779680e+02,1.1730260e+02,8.8815900e+01,6.7299900e+01, &
& 5.1786000e+01,7.6786760e+02,5.8961410e+02,5.4296830e+02,4.2728580e+02, &
& 3.3286240e+02,2.7611290e+02,2.2545620e+02,1.8441100e+02,1.2590280e+03, &
& 1.0371340e+03,8.5644340e+02,8.2672570e+02,7.5688430e+02,5.9629710e+02, &
& 6.5144380e+02,5.1133840e+02,5.4164010e+02,5.5809040e+02,4.2796300e+02, &
& 4.3895460e+02,5.1995360e+02,4.5803850e+02,3.9031610e+02,3.5005540e+02, &
& 3.0619590e+02,2.6618930e+02,1.4108565e+03,1.2358086e+03,1.0815963e+03, &
& 9.7156440e+02,8.8562820e+02,6.8336810e+02,7.6261340e+02,5.8065630e+02, &
& 6.3462930e+02,5.8839190e+02,4.9078330e+02,5.1833560e+02,6.5088900e+02, &
& 6.0128620e+02,5.3534260e+02,4.9642060e+02,4.4860200e+02,4.0312330e+02, &
& 1.7186565e+03,1.5759396e+03,1.3833184e+03,6.3260690e+02,1.3959648e+03, &
& 1.3402848e+03,1.3066123e+03,1.2756493e+03,1.2481896e+03,9.8061140e+02, &
& 1.1056577e+03,1.0661553e+03,1.1255430e+03,1.1016251e+03,1.0800441e+03, &
& 1.0674513e+03,8.9877880e+02,8.8303380e+02,8.0541670e+02,6.7986960e+02]
 vdw_dftd3_c6(11101:11200)=[&
& 6.9147700e+02,6.2640460e+02,5.7347180e+02,4.7626530e+02,4.4493320e+02, &
& 4.5759140e+02,6.6821650e+02,6.5280350e+02,5.9931800e+02,5.7130760e+02, &
& 5.2695800e+02,4.8275550e+02,1.6217509e+03,1.5576603e+03,1.3757224e+03, &
& 1.2351334e+03,1.2269612e+03,1.1880136e+03,1.2239539e+03,1.1851389e+03, &
& 6.6143900e+01,2.1491570e+02,2.7404530e+02,2.0906270e+02,1.5740780e+02, &
& 1.0939440e+02,7.9656600e+01,5.4794700e+01,3.1433160e+02,4.8615820e+02, &
& 4.9197000e+02,3.9485010e+02,3.2309660e+02,2.7308190e+02,2.2312610e+02, &
& 4.3320740e+02,8.2964080e+02,4.3282760e+02,4.1783600e+02,4.0950580e+02, &
& 4.0578350e+02,3.7312230e+02,3.4526760e+02,3.2938120e+02,3.2203750e+02, &
& 3.1836420e+02,2.9841060e+02,4.8821620e+02,4.2840160e+02,3.8367750e+02, &
& 3.5019520e+02,3.0833610e+02,5.1680260e+02,1.0097020e+03,7.7048950e+02, &
& 5.7252260e+02,5.7822130e+02,5.3906400e+02,6.0508290e+02,4.6961340e+02, &
& 4.3933610e+02,4.0794200e+02,3.9468140e+02,3.9021360e+02,6.2022960e+02, &
& 5.6054060e+02,5.2460280e+02,4.9603960e+02,4.5281350e+02,6.1065170e+02, &
& 1.3096071e+03,9.7534290e+02,6.1350220e+02,6.0081010e+02,5.8174350e+02, &
& 4.1118600e+01,2.7091000e+01,6.5371250e+02,3.7335380e+02,2.5006900e+02, &
& 1.6835070e+02,1.1747400e+02,8.8833200e+01,6.7241500e+01,5.1697700e+01, &
& 7.8120500e+02,5.9600220e+02,5.4777420e+02,4.2994940e+02,3.3425170e+02, &
& 2.7691160e+02,2.2583700e+02,1.8453320e+02,1.2836315e+03,1.0509848e+03, &
& 8.6692280e+02,8.3609940e+02,7.6505120e+02,6.0271130e+02,6.5796350e+02]
 vdw_dftd3_c6(11201:11300)=[&
& 5.1637700e+02,5.4633630e+02,5.6322550e+02,4.3190330e+02,4.4225450e+02, &
& 5.2416760e+02,4.6081790e+02,3.9201390e+02,3.5121820e+02,3.0689590e+02, &
& 2.6654940e+02,1.4379300e+03,1.2526342e+03,1.0942695e+03,9.8185530e+02, &
& 8.9439170e+02,6.8930910e+02,7.6959900e+02,5.8520650e+02,6.3957050e+02, &
& 5.9271510e+02,4.9454430e+02,5.2185710e+02,6.5623190e+02,6.0519030e+02, &
& 5.3798990e+02,4.9842910e+02,4.4999200e+02,4.0401740e+02,1.7521410e+03, &
& 1.5986396e+03,1.4005068e+03,6.3591570e+02,1.4152709e+03,1.3581904e+03, &
& 1.3239329e+03,1.2924475e+03,1.2645213e+03,9.9151760e+02,1.1218549e+03, &
& 1.0816001e+03,1.1396825e+03,1.1153978e+03,1.0934602e+03,1.0807974e+03, &
& 9.0896190e+02,8.9127890e+02,8.1208250e+02,6.8499030e+02,6.9640000e+02, &
& 6.3031370e+02,5.7665350e+02,4.7857310e+02,4.4696260e+02,4.5952800e+02, &
& 6.7346620e+02,6.5702270e+02,6.0234550e+02,5.7374780e+02,5.2872700e+02, &
& 4.8398270e+02,1.6501361e+03,1.5785123e+03,1.3916728e+03,1.2470771e+03, &
& 1.2400445e+03,1.2006028e+03,1.2383187e+03,1.1987720e+03,6.6405700e+01, &
& 2.1686950e+02,2.7638280e+02,2.1014480e+02,1.5794060e+02,1.0955230e+02, &
& 7.9647700e+01,5.4685200e+01,3.1740960e+02,4.9113850e+02,4.9609590e+02, &
& 3.9717520e+02,3.2442460e+02,2.7388040e+02,2.2350790e+02,4.3688540e+02, &
& 8.4105190e+02,4.3551720e+02,4.2046710e+02,4.1208740e+02,4.0844600e+02, &
& 3.7510510e+02,3.4698680e+02,3.3102820e+02,3.2367720e+02,3.2026840e+02, &
& 2.9970920e+02,4.9191570e+02,4.3081190e+02,3.8530990e+02,3.5136440e+02]
 vdw_dftd3_c6(11301:11400)=[&
& 3.0905750e+02,5.2097030e+02,1.0240101e+03,7.7856140e+02,5.7624530e+02, &
& 5.8193050e+02,5.4238030e+02,6.0986550e+02,4.7194410e+02,4.4147690e+02, &
& 4.0984440e+02,3.9658040e+02,3.9174910e+02,6.2494310e+02,5.6383860e+02, &
& 5.2709190e+02,4.9801930e+02,4.5422360e+02,6.1477890e+02,1.3297509e+03, &
& 9.8604390e+02,6.1665720e+02,6.0388680e+02,5.8467880e+02,5.8778910e+02, &
& 3.9256400e+01,2.5691300e+01,6.5368460e+02,3.6511120e+02,2.4163430e+02, &
& 1.6139330e+02,1.1205120e+02,8.4473100e+01,6.3799600e+01,4.8980900e+01, &
& 7.8026740e+02,5.8523670e+02,5.3437570e+02,4.1591500e+02,3.2121200e+02, &
& 2.6508690e+02,2.1546030e+02,1.7559080e+02,1.2856965e+03,1.0394662e+03, &
& 8.5482160e+02,8.2218100e+02,7.5108930e+02,5.9166600e+02,6.4444210e+02, &
& 5.0560640e+02,5.3300400e+02,5.5039520e+02,4.2213760e+02,4.2993040e+02, &
& 5.1021570e+02,4.4563750e+02,3.7699910e+02,3.3669100e+02,2.9330360e+02, &
& 2.5408590e+02,1.4385445e+03,1.2395504e+03,1.0770858e+03,9.6327150e+02, &
& 8.7559310e+02,6.7249140e+02,7.5175670e+02,5.6950600e+02,6.2232140e+02, &
& 5.7599650e+02,4.8115240e+02,5.0633030e+02,6.3922390e+02,5.8633170e+02, &
& 5.1859750e+02,4.7906490e+02,4.3122670e+02,3.8614600e+02,1.7518319e+03, &
& 1.5845929e+03,1.3809239e+03,6.1365370e+02,1.4003880e+03,1.3429247e+03, &
& 1.3087424e+03,1.2773568e+03,1.2495054e+03,9.7406900e+02,1.1111877e+03, &
& 1.0699921e+03,1.1246771e+03,1.1005512e+03,1.0786794e+03,1.0664387e+03, &
& 8.9316870e+02,8.7110070e+02,7.9118140e+02,6.6583170e+02,6.7605760e+02]
 vdw_dftd3_c6(11401:11500)=[&
& 6.1029470e+02,5.5718050e+02,4.6153520e+02,4.3073110e+02,4.4232070e+02, &
& 6.5532760e+02,6.3664300e+02,5.8099830e+02,5.5199550e+02,5.0719420e+02, &
& 4.6310600e+02,1.6436058e+03,1.5604236e+03,1.3690464e+03,1.2201277e+03, &
& 1.2166738e+03,1.1778612e+03,1.2191450e+03,1.1794690e+03,6.3762900e+01, &
& 2.1174000e+02,2.6923050e+02,2.0261480e+02,1.5147130e+02,1.0450970e+02, &
& 7.5690800e+01,5.1755700e+01,3.1068640e+02,4.8112160e+02,4.8324150e+02, &
& 3.8377640e+02,3.1171580e+02,2.6222190e+02,2.1326140e+02,4.2593700e+02, &
& 8.3127500e+02,4.2169690e+02,4.0700940e+02,3.9898160e+02,3.9579870e+02, &
& 3.6204930e+02,3.3456150e+02,3.1919930e+02,3.1222210e+02,3.0989590e+02, &
& 2.8840040e+02,4.7803960e+02,4.1601900e+02,3.7044180e+02,3.3685390e+02, &
& 2.9542100e+02,5.0722850e+02,1.0128078e+03,7.6200330e+02,5.5829510e+02, &
& 5.6398160e+02,5.2490850e+02,5.9323860e+02,4.5513890e+02,4.2567220e+02, &
& 3.9496580e+02,3.8242790e+02,3.7661900e+02,6.0761490e+02,5.4518900e+02, &
& 5.0775680e+02,4.7858010e+02,4.3529630e+02,5.9594990e+02,1.3176067e+03, &
& 9.6582370e+02,5.9490700e+02,5.8256410e+02,5.6390150e+02,5.6731710e+02, &
& 5.4902810e+02,4.1327500e+01,2.7159500e+01,6.5909240e+02,3.7646100e+02, &
& 2.5184240e+02,1.6931590e+02,1.1799770e+02,8.9135900e+01,6.7403000e+01, &
& 5.1775600e+01,7.8761870e+02,6.0103840e+02,5.5215450e+02,4.3306620e+02, &
& 3.3637000e+02,2.7845510e+02,2.2691090e+02,1.8526210e+02,1.2927733e+03, &
& 1.0597192e+03,8.7413140e+02,8.4288400e+02,7.7118400e+02,6.0734870e+02]
 vdw_dftd3_c6(11501:11600)=[&
& 6.6313130e+02,5.2025690e+02,5.5050450e+02,5.6760260e+02,4.3508290e+02, &
& 4.4549820e+02,5.2813050e+02,4.6406710e+02,3.9450330e+02,3.5325210e+02, &
& 3.0847630e+02,2.6774640e+02,1.4479360e+03,1.2628491e+03,1.1030742e+03, &
& 9.8959400e+02,9.0127470e+02,6.9429580e+02,7.7529730e+02,5.8924130e+02, &
& 6.4413130e+02,5.9686540e+02,4.9784100e+02,5.2541210e+02,6.6098150e+02, &
& 6.0940140e+02,5.4146000e+02,5.0144410e+02,4.5249160e+02,4.0604790e+02, &
& 1.7636917e+03,1.6113765e+03,1.4116458e+03,6.3999700e+02,1.4263405e+03, &
& 1.3689945e+03,1.3344823e+03,1.3027626e+03,1.2746283e+03,9.9906170e+02, &
& 1.1300310e+03,1.0892614e+03,1.1488304e+03,1.1243657e+03,1.1022614e+03, &
& 1.0895200e+03,9.1593800e+02,8.9812800e+02,8.1811890e+02,6.8977370e+02, &
& 7.0124130e+02,6.3451700e+02,5.8034520e+02,4.8140120e+02,4.4951810e+02, &
& 4.6216400e+02,6.7804940e+02,6.6143650e+02,6.0616090e+02,5.7721150e+02, &
& 5.3169500e+02,4.8648120e+02,1.6615027e+03,1.5910020e+03,1.4026190e+03, &
& 1.2565351e+03,1.2494197e+03,1.2096721e+03,1.2478750e+03,1.2080168e+03, &
& 6.6823500e+01,2.1861950e+02,2.7850800e+02,2.1154850e+02,1.5883010e+02, &
& 1.1002650e+02,7.9894900e+01,5.4760000e+01,3.1995720e+02,4.9511470e+02, &
& 4.9998200e+02,3.9998160e+02,3.2645920e+02,2.7540540e+02,2.2456980e+02, &
& 4.3997650e+02,8.4744860e+02,4.3855500e+02,4.2330690e+02,4.1487410e+02, &
& 4.1122300e+02,3.7757470e+02,3.4920950e+02,3.3312400e+02,3.2573280e+02, &
& 3.2237280e+02,3.0155830e+02,4.9556680e+02,4.3376750e+02,3.8773360e+02]
 vdw_dftd3_c6(11601:11700)=[&
& 3.5339890e+02,3.1065690e+02,5.2453740e+02,1.0315808e+03,7.8404790e+02, &
& 5.8016290e+02,5.8594010e+02,5.4592550e+02,6.1398170e+02,4.7485900e+02, &
& 4.4414170e+02,4.1225320e+02,3.9892850e+02,3.9401610e+02,6.2938940e+02, &
& 5.6763680e+02,5.3044640e+02,5.0102050e+02,4.5675320e+02,6.1885080e+02, &
& 1.3390290e+03,9.9282930e+02,6.2058810e+02,6.0773400e+02,5.8838130e+02, &
& 5.9153680e+02,5.7104500e+02,5.9536280e+02,3.8915000e+01,2.5669500e+01, &
& 6.1281660e+02,3.5207530e+02,2.3626590e+02,1.5924220e+02,1.1120430e+02, &
& 8.4136300e+01,6.3715200e+01,4.9005000e+01,7.3264570e+02,5.6159910e+02, &
& 5.1672630e+02,4.0615220e+02,3.1605550e+02,2.6197720e+02,2.1376170e+02, &
& 1.7473800e+02,1.2013307e+03,9.8850950e+02,8.1603910e+02,7.8743780e+02, &
& 7.2076790e+02,5.6775360e+02,6.2017110e+02,4.8669710e+02,5.1538390e+02, &
& 5.3115950e+02,4.0724140e+02,4.1747630e+02,4.9461700e+02,4.3533390e+02, &
& 3.7064210e+02,3.3221730e+02,2.9041790e+02,2.5233250e+02,1.3459552e+03, &
& 1.1778523e+03,1.0302577e+03,9.2507300e+02,8.4300140e+02,6.5010950e+02, &
& 7.2565600e+02,5.5217930e+02,6.0356690e+02,5.5949210e+02,4.6665970e+02, &
& 4.9276250e+02,6.1913660e+02,5.7157510e+02,5.0851710e+02,4.7131940e+02, &
& 4.2569420e+02,3.8234290e+02,1.6393995e+03,1.5021922e+03,1.3178387e+03, &
& 6.0095180e+02,1.3303810e+03,1.2772241e+03,1.2451109e+03,1.2155853e+03, &
& 1.1893991e+03,9.3372280e+02,1.0536587e+03,1.0158840e+03,1.0723978e+03, &
& 1.0495999e+03,1.0290199e+03,1.0170572e+03,8.5590310e+02,8.4038840e+02]
 vdw_dftd3_c6(11701:11800)=[&
& 7.6618830e+02,6.4647490e+02,6.5742490e+02,5.9532710e+02,5.4484580e+02, &
& 4.5230750e+02,4.2248760e+02,4.3446390e+02,6.3542770e+02,6.2048770e+02, &
& 5.6929560e+02,5.4247560e+02,5.0012070e+02,4.5795880e+02,1.5463042e+03, &
& 1.4842932e+03,1.3102595e+03,1.1755520e+03,1.1681093e+03,1.1310127e+03, &
& 1.1657289e+03,1.1286850e+03,6.2797100e+01,2.0455140e+02,2.6072450e+02, &
& 1.9860250e+02,1.4938110e+02,1.0370340e+02,7.5446500e+01,5.1841400e+01, &
& 2.9924680e+02,4.6288270e+02,4.6808680e+02,3.7524590e+02,3.0676780e+02, &
& 2.5910430e+02,2.1155470e+02,4.1206550e+02,7.9052270e+02,4.1139690e+02, &
& 3.9711900e+02,3.8920800e+02,3.8571070e+02,3.5449400e+02,3.2797050e+02, &
& 3.1287350e+02,3.0591120e+02,3.0254470e+02,2.8337650e+02,4.6432020e+02, &
& 4.0707180e+02,3.6431690e+02,3.3235190e+02,2.9245640e+02,4.9146210e+02, &
& 9.6209720e+02,7.3326220e+02,5.4416190e+02,5.4959500e+02,5.1226070e+02, &
& 5.7535150e+02,4.4604100e+02,4.1725200e+02,3.8739090e+02,3.7483220e+02, &
& 3.7045920e+02,5.8983000e+02,5.3268550e+02,4.9826300e+02,4.7094350e+02, &
& 4.2969450e+02,5.8038410e+02,1.2480182e+03,9.2827050e+02,5.8277730e+02, &
& 5.7071750e+02,5.5258430e+02,5.5542950e+02,5.3587600e+02,5.5897620e+02, &
& 5.2491950e+02,3.9396700e+01,2.5865800e+01,6.3848500e+02,3.6118220e+02, &
& 2.4078360e+02,1.6156910e+02,1.1248360e+02,8.4928800e+01,6.4205600e+01, &
& 4.9316600e+01,7.6254390e+02,5.7748050e+02,5.2942090e+02,4.1417170e+02, &
& 3.2113630e+02,2.6561240e+02,2.1629400e+02,1.7651170e+02,1.2553434e+03]
 vdw_dftd3_c6(11801:11900)=[&
& 1.0213342e+03,8.4137840e+02,8.1057120e+02,7.4119950e+02,5.8388160e+02, &
& 6.3684280e+02,4.9970290e+02,5.2794350e+02,5.4462440e+02,4.1763490e+02, &
& 4.2675460e+02,5.0613150e+02,4.4382260e+02,3.7672490e+02,3.3707580e+02, &
& 2.9415100e+02,2.5518090e+02,1.4056043e+03,1.2176072e+03,1.0613052e+03, &
& 9.5099220e+02,8.6553240e+02,6.6608950e+02,7.4408720e+02,5.6489840e+02, &
& 6.1735350e+02,5.7182020e+02,4.7729120e+02,5.0312050e+02,6.3374930e+02, &
& 5.8321300e+02,5.1742880e+02,4.7882440e+02,4.3176670e+02,3.8721870e+02, &
& 1.7128830e+03,1.5552120e+03,1.3593802e+03,6.1183610e+02,1.3758472e+03, &
& 1.3197747e+03,1.2863433e+03,1.2556329e+03,1.2283897e+03,9.6089950e+02, &
& 1.0913776e+03,1.0518609e+03,1.1064740e+03,1.0828259e+03,1.0614329e+03, &
& 1.0492440e+03,8.8106130e+02,8.6194390e+02,7.8433470e+02,6.6095780e+02, &
& 6.7162740e+02,6.0723190e+02,5.5505840e+02,4.6025170e+02,4.2970270e+02, &
& 4.4159570e+02,6.5009130e+02,6.3314710e+02,5.7941850e+02,5.5135470e+02, &
& 5.0749380e+02,4.6406150e+02,1.6099709e+03,1.5338391e+03,1.3494846e+03, &
& 1.2065102e+03,1.2011049e+03,1.1628261e+03,1.2010327e+03,1.1623643e+03, &
& 6.3781600e+01,2.0966850e+02,2.6699220e+02,2.0215950e+02,1.5159630e+02, &
& 1.0489910e+02,7.6120900e+01,5.2145400e+01,3.0714640e+02,4.7548270e+02, &
& 4.7919230e+02,3.8242860e+02,3.1166960e+02,2.6271630e+02,2.1407020e+02, &
& 4.2205350e+02,8.1742610e+02,4.1957740e+02,4.0507580e+02,3.9701800e+02, &
& 3.9363860e+02,3.6094370e+02,3.3374440e+02,3.1840100e+02,3.1136890e+02]
 vdw_dftd3_c6(11901:12000)=[&
& 3.0844670e+02,2.8804090e+02,4.7467810e+02,4.1468730e+02,3.7023680e+02, &
& 3.3722390e+02,2.9624340e+02,5.0300950e+02,9.9563320e+02,7.5368720e+02, &
& 5.5531660e+02,5.6079120e+02,5.2244060e+02,5.8868540e+02,4.5392960e+02, &
& 4.2457780e+02,3.9405630e+02,3.8138420e+02,3.7631210e+02,6.0308050e+02, &
& 5.4294040e+02,5.0681590e+02,4.7839610e+02,4.3583490e+02,5.9259730e+02, &
& 1.2943501e+03,9.5498290e+02,5.9324060e+02,5.8094220e+02,5.6241030e+02, &
& 5.6558410e+02,5.4641960e+02,5.6923180e+02,5.3434800e+02,5.4443560e+02, &
& 3.7029500e+01,2.4480600e+01,5.7574990e+02,3.3286460e+02,2.2408560e+02, &
& 1.5136070e+02,1.0585820e+02,8.0170400e+01,6.0759800e+01,4.6759500e+01, &
& 6.8859100e+02,5.3038700e+02,4.8884720e+02,3.8509520e+02,3.0019740e+02, &
& 2.4909840e+02,2.0345290e+02,1.6644660e+02,1.1279480e+03,9.3174430e+02, &
& 7.6983360e+02,7.4340690e+02,6.8076970e+02,5.3627310e+02,5.8613150e+02, &
& 4.6004150e+02,4.8762020e+02,5.0232060e+02,3.8513270e+02,3.9536540e+02, &
& 4.6822770e+02,4.1281620e+02,3.5198750e+02,3.1577010e+02,2.7627680e+02, &
& 2.4022750e+02,1.2641518e+03,1.1100442e+03,9.7238640e+02,8.7389940e+02, &
& 7.9683270e+02,6.1510520e+02,6.8633810e+02,5.2281640e+02,5.7148280e+02, &
& 5.2993880e+02,4.4189760e+02,4.6694330e+02,5.8603950e+02,5.4179100e+02, &
& 4.8265820e+02,4.4769600e+02,4.0468130e+02,3.6373600e+02,1.5398453e+03, &
& 1.4149943e+03,1.2431883e+03,5.7024510e+02,1.2537691e+03,1.2039727e+03, &
& 1.1737842e+03,1.1460194e+03,1.1213978e+03,8.8175490e+02,9.9262670e+02]
 vdw_dftd3_c6(12001:12100)=[&
& 9.5731530e+02,1.0114677e+03,9.9000700e+02,9.7065280e+02,9.5930580e+02, &
& 8.0819030e+02,7.9471630e+02,7.2517730e+02,6.1226400e+02,6.2284620e+02, &
& 5.6442200e+02,5.1685750e+02,4.2931700e+02,4.0110390e+02,4.1259070e+02, &
& 6.0165160e+02,5.8816150e+02,5.4028410e+02,5.1517620e+02,4.7532260e+02, &
& 4.3554890e+02,1.4541544e+03,1.3992062e+03,1.2368406e+03,1.1113608e+03, &
& 1.1034821e+03,1.0684783e+03,1.1002364e+03,1.0654625e+03,5.9658600e+01, &
& 1.9347380e+02,2.4674640e+02,1.8847650e+02,1.4197970e+02,9.8721200e+01, &
& 7.1910400e+01,4.9479300e+01,2.8286510e+02,4.3742550e+02,4.4300780e+02, &
& 3.5589820e+02,2.9139070e+02,2.4635930e+02,2.0134780e+02,3.8995180e+02, &
& 7.4517810e+02,3.9002770e+02,3.7650890e+02,3.6899410e+02,3.6559720e+02, &
& 3.3635820e+02,3.1127980e+02,2.9694730e+02,2.9031390e+02,2.8689270e+02, &
& 2.6909650e+02,4.3973990e+02,3.8616340e+02,3.4600850e+02,3.1589290e+02, &
& 2.7820320e+02,4.6526060e+02,9.0671060e+02,6.9306070e+02,5.1580860e+02, &
& 5.2093800e+02,4.8572140e+02,5.4479060e+02,4.2334180e+02,3.9604750e+02, &
& 3.6776460e+02,3.5578930e+02,3.5190760e+02,5.5857850e+02,5.0519270e+02, &
& 4.7300760e+02,4.4736060e+02,4.0847930e+02,5.5005480e+02,1.1754332e+03, &
& 8.7714780e+02,5.5304140e+02,5.4160480e+02,5.2443100e+02,5.2702430e+02, &
& 5.0812970e+02,5.3036160e+02,4.9813190e+02,5.0688950e+02,4.7279500e+02, &
& 3.6008400e+01,2.3828900e+01,5.5724540e+02,3.2288400e+02,2.1762790e+02, &
& 1.4712440e+02,1.0295880e+02,7.8007900e+01,5.9142300e+01,4.5527900e+01]
 vdw_dftd3_c6(12101:12200)=[&
& 6.6655340e+02,5.1428720e+02,4.7430950e+02,3.7395400e+02,2.9171000e+02, &
& 2.4215880e+02,1.9786420e+02,1.6193000e+02,1.0915091e+03,9.0284450e+02, &
& 7.4618100e+02,7.2076670e+02,6.6014630e+02,5.2004920e+02,5.6851110e+02, &
& 4.4624320e+02,4.7314860e+02,4.8733120e+02,3.7365280e+02,3.8376960e+02, &
& 4.5441890e+02,4.0089710e+02,3.4201700e+02,3.0693000e+02,2.6863470e+02, &
& 2.3365510e+02,1.2234668e+03,1.0755659e+03,9.4268540e+02,8.4748690e+02, &
& 7.7291870e+02,5.9687070e+02,6.6589970e+02,5.0745680e+02,5.5468820e+02, &
& 5.1443460e+02,4.2894240e+02,4.5336080e+02,5.6875080e+02,5.2608060e+02, &
& 4.6889680e+02,4.3506030e+02,3.9338290e+02,3.5368410e+02,1.4903329e+03, &
& 1.3708120e+03,1.2050073e+03,5.5393560e+02,1.2148410e+03,1.1666903e+03, &
& 1.1374645e+03,1.1105819e+03,1.0867437e+03,8.5501380e+02,9.6171480e+02, &
& 9.2760080e+02,9.8034030e+02,9.5955360e+02,9.4081400e+02,9.2979290e+02, &
& 7.8364610e+02,7.7098740e+02,7.0375210e+02,5.9432800e+02,6.0467390e+02, &
& 5.4810470e+02,5.0202620e+02,4.1709650e+02,3.8972370e+02,4.0092380e+02, &
& 5.8398610e+02,5.7111950e+02,5.2486410e+02,5.0060050e+02,4.6201380e+02, &
& 4.2346810e+02,1.4079944e+03,1.3558908e+03,1.1991383e+03,1.0780836e+03, &
& 1.0701540e+03,1.0362222e+03,1.0666560e+03,1.0330090e+03,5.7976000e+01, &
& 1.8770500e+02,2.3944310e+02,1.8308850e+02,1.3800410e+02,9.6019300e+01, &
& 6.9979100e+01,4.8180500e+01,2.7437320e+02,4.2424930e+02,4.2989820e+02, &
& 3.4564250e+02,2.8315870e+02,2.3949360e+02,1.9581550e+02,3.7842050e+02]
 vdw_dftd3_c6(12201:12300)=[&
& 7.2211110e+02,3.7874080e+02,3.6562310e+02,3.5832130e+02,3.5499450e+02, &
& 3.2672790e+02,3.0240170e+02,2.8847830e+02,2.8202540e+02,2.7862110e+02, &
& 2.6147450e+02,4.2684110e+02,3.7506910e+02,3.3621830e+02,3.0704780e+02, &
& 2.7050300e+02,4.5156840e+02,8.7858370e+02,6.7226710e+02,5.0085680e+02, &
& 5.0582970e+02,4.7170110e+02,5.2880060e+02,4.1127310e+02,3.8477140e+02, &
& 3.5731920e+02,3.4566580e+02,3.4198940e+02,5.4219990e+02,4.9064160e+02, &
& 4.5955210e+02,4.3474240e+02,3.9707280e+02,5.3408180e+02,1.1387329e+03, &
& 8.5075470e+02,5.3723990e+02,5.2613310e+02,5.0946310e+02,5.1194390e+02, &
& 4.9346840e+02,5.1517330e+02,4.8389820e+02,4.9233800e+02,4.5931550e+02, &
& 4.4623140e+02,3.5748000e+01,2.3629700e+01,5.5525980e+02,3.2123930e+02, &
& 2.1630990e+02,1.4611730e+02,1.0218880e+02,7.7387400e+01,5.8646500e+01, &
& 4.5129800e+01,6.6409940e+02,5.1180090e+02,4.7179450e+02,3.7172710e+02, &
& 2.8980090e+02,2.4047400e+02,1.9640620e+02,1.6067610e+02,1.0876970e+03, &
& 8.9886340e+02,7.4273900e+02,7.1728970e+02,6.5688060e+02,5.1743220e+02, &
& 5.6559590e+02,4.4390510e+02,4.7058200e+02,4.8475110e+02,3.7163760e+02, &
& 3.8158090e+02,4.5190020e+02,3.9847870e+02,3.3979060e+02,3.0483420e+02, &
& 2.6670930e+02,2.3190470e+02,1.2190706e+03,1.0708351e+03,9.3818730e+02, &
& 8.4323610e+02,7.6891010e+02,5.9357780e+02,6.6231090e+02,5.0453730e+02, &
& 5.5152390e+02,5.1144340e+02,4.2643750e+02,4.5065890e+02,5.6556470e+02, &
& 5.2293560e+02,4.6590590e+02,4.3217320e+02,3.9065890e+02,3.5113440e+02]
 vdw_dftd3_c6(12301:12400)=[&
& 1.4849661e+03,1.3649240e+03,1.1993874e+03,5.5042640e+02,1.2094663e+03, &
& 1.1614494e+03,1.1323365e+03,1.1055604e+03,1.0818160e+03,8.5075460e+02, &
& 9.5749480e+02,9.2347850e+02,9.7580850e+02,9.5511040e+02,9.3644570e+02, &
& 9.2549440e+02,7.7979590e+02,7.6690850e+02,6.9985110e+02,5.9088900e+02, &
& 6.0112600e+02,5.4476460e+02,4.9887240e+02,4.1437420e+02,3.8714220e+02, &
& 3.9824650e+02,5.8061670e+02,5.6766880e+02,5.2151170e+02,4.9729730e+02, &
& 4.5884250e+02,4.2045390e+02,1.4024613e+03,1.3497931e+03,1.1933412e+03, &
& 1.0724182e+03,1.0647199e+03,1.0309456e+03,1.0614860e+03,1.0279565e+03, &
& 5.7593700e+01,1.8672100e+02,2.3814200e+02,1.8193830e+02,1.3705810e+02, &
& 9.5297900e+01,6.9413700e+01,4.7755100e+01,2.7296680e+02,4.2211770e+02, &
& 4.2756550e+02,3.4354750e+02,2.8129780e+02,2.3782830e+02,1.9437320e+02, &
& 3.7630470e+02,7.1887190e+02,3.7645520e+02,3.6340760e+02,3.5615030e+02, &
& 3.5286320e+02,3.2467610e+02,3.0047150e+02,2.8663320e+02,2.8022740e+02, &
& 2.7690380e+02,2.5976130e+02,4.2442030e+02,3.7275950e+02,3.3401900e+02, &
& 3.0495200e+02,2.6856840e+02,4.4898400e+02,8.7467070e+02,6.6876560e+02, &
& 4.9783950e+02,5.0278080e+02,4.6880410e+02,5.2574810e+02,4.0862370e+02, &
& 3.8227390e+02,3.5497280e+02,3.4340860e+02,3.3969190e+02,5.3908680e+02, &
& 4.8762850e+02,4.5659480e+02,4.3185070e+02,3.9432570e+02,5.3087060e+02, &
& 1.1338329e+03,8.4637940e+02,5.3382250e+02,5.2278360e+02,5.0620810e+02, &
& 5.0870480e+02,4.9043340e+02,5.1192810e+02,4.8082180e+02,4.8926080e+02]
 vdw_dftd3_c6(12401:12500)=[&
& 4.5637280e+02,4.4336390e+02,4.4052280e+02,3.7398600e+01,2.4543500e+01, &
& 5.9236510e+02,3.4023850e+02,2.2788020e+02,1.5321460e+02,1.0673400e+02, &
& 8.0586300e+01,6.0903100e+01,4.6756900e+01,7.0806290e+02,5.4277630e+02, &
& 4.9910710e+02,3.9184510e+02,3.0444600e+02,2.5199750e+02,2.0529690e+02, &
& 1.6755260e+02,1.1600767e+03,9.5507870e+02,7.8843250e+02,7.6056390e+02, &
& 6.9606330e+02,5.4795880e+02,5.9876220e+02,4.6957880e+02,4.9740630e+02, &
& 5.1274730e+02,3.9280830e+02,4.0272690e+02,4.7737550e+02,4.1982970e+02, &
& 3.5702170e+02,3.1968520e+02,2.7912740e+02,2.4221470e+02,1.2994482e+03, &
& 1.1377525e+03,9.9498690e+02,8.9315950e+02,8.1367980e+02,6.2696020e+02, &
& 7.0006760e+02,5.3219550e+02,5.8198660e+02,5.3936520e+02,4.4956060e+02, &
& 4.7487730e+02,5.9714660e+02,5.5105660e+02,4.8986860e+02,4.5372430e+02, &
& 4.0944070e+02,3.6738410e+02,1.5825500e+03,1.4507794e+03,1.2725822e+03, &
& 5.7883400e+02,1.2846682e+03,1.2333365e+03,1.2023371e+03,1.1738392e+03, &
& 1.1485655e+03,9.0112100e+02,1.0169049e+03,9.8043260e+02,1.0356000e+03, &
& 1.0136038e+03,9.9374150e+02,9.8222660e+02,8.2628510e+02,8.1115710e+02, &
& 7.3922210e+02,6.2322500e+02,6.3376790e+02,5.7362030e+02,5.2473400e+02, &
& 4.3520090e+02,4.0635950e+02,4.1792020e+02,6.1237220e+02,5.9791090e+02, &
& 5.4826140e+02,5.2217940e+02,4.8105700e+02,4.4014200e+02,1.4924670e+03, &
& 1.4332640e+03,1.2650809e+03,1.1344175e+03,1.1272049e+03,1.0913656e+03, &
& 1.1251094e+03,1.0893370e+03,6.0487100e+01,1.9759460e+02,2.5172230e+02]
 vdw_dftd3_c6(12501:12600)=[&
& 1.9141090e+02,1.4369300e+02,9.9505900e+01,7.2218200e+01,4.9450200e+01, &
& 2.8901230e+02,4.4716520e+02,4.5200550e+02,3.6191400e+02,2.9546110e+02, &
& 2.4922870e+02,2.0317400e+02,3.9728560e+02,7.6326450e+02,3.9658450e+02, &
& 3.8275050e+02,3.7510830e+02,3.7175280e+02,3.4156320e+02,3.1590920e+02, &
& 3.0132730e+02,2.9462620e+02,2.9146390e+02,2.7284960e+02,4.4803280e+02, &
& 3.9245010e+02,3.5089060e+02,3.1981290e+02,2.8109900e+02,4.7364740e+02, &
& 9.2871230e+02,7.0739100e+02,5.2442450e+02,5.2963760e+02,4.9348140e+02, &
& 5.5447820e+02,4.2941670e+02,4.0159420e+02,3.7273910e+02,3.6066920e+02, &
& 3.5642060e+02,5.6876740e+02,5.1338040e+02,4.7992240e+02,4.5334590e+02, &
& 4.1330020e+02,5.5918680e+02,1.2045119e+03,8.9547410e+02,5.6128730e+02, &
& 5.4966430e+02,5.3216530e+02,5.3496580e+02,5.1624120e+02,5.3845600e+02, &
& 5.0556900e+02,5.1473030e+02,4.7973200e+02,4.6600690e+02,4.6306960e+02, &
& 4.8707530e+02,3.4601200e+01,2.2892600e+01,5.3345810e+02,3.0982450e+02, &
& 2.0900590e+02,1.4134580e+02,9.8922300e+01,7.4945100e+01,5.6812400e+01, &
& 4.3726300e+01,6.3816340e+02,4.9328830e+02,4.5519600e+02,3.5911410e+02, &
& 2.8024060e+02,2.3266870e+02,1.9012190e+02,1.5559160e+02,1.0444701e+03, &
& 8.6528290e+02,7.1536520e+02,6.9115730e+02,6.3311630e+02,4.9870530e+02, &
& 5.4533910e+02,4.2801770e+02,4.5401590e+02,4.6756420e+02,3.5843930e+02, &
& 3.6835200e+02,4.3614550e+02,3.8497400e+02,3.2854540e+02,2.9488120e+02, &
& 2.5811290e+02,2.2451100e+02,1.1708400e+03,1.0307165e+03,9.0385210e+02]
 vdw_dftd3_c6(12601:12700)=[&
& 8.1281250e+02,7.4141830e+02,5.7266250e+02,6.3884880e+02,4.8694650e+02, &
& 5.3231850e+02,4.9373180e+02,4.1158220e+02,4.3515940e+02,5.4577490e+02, &
& 5.0506710e+02,4.5032940e+02,4.1790250e+02,3.7792070e+02,3.3981250e+02, &
& 1.4262370e+03,1.3133618e+03,1.1551299e+03,5.3193250e+02,1.1640997e+03, &
& 1.1180647e+03,1.0900872e+03,1.0643505e+03,1.0415296e+03,8.1984680e+02, &
& 9.2133410e+02,8.8874530e+02,9.3968880e+02,9.1978180e+02,9.0184110e+02, &
& 8.9126080e+02,7.5143080e+02,7.3969160e+02,6.7535320e+02,5.7039690e+02, &
& 5.8039830e+02,5.2619340e+02,4.8201740e+02,4.0048290e+02,3.7420390e+02, &
& 3.8500720e+02,5.6037600e+02,5.4825190e+02,5.0402540e+02,4.8080590e+02, &
& 4.4381570e+02,4.0683040e+02,1.3480198e+03,1.2994057e+03,1.1497484e+03, &
& 1.0341669e+03,1.0262543e+03,9.9371920e+02,1.0225738e+03,9.9038000e+02, &
& 5.5700800e+01,1.8012920e+02,2.2980950e+02,1.7585020e+02,1.3257550e+02, &
& 9.2251200e+01,6.7230800e+01,4.6277200e+01,2.6322190e+02,4.0699260e+02, &
& 4.1261700e+02,3.3194510e+02,2.7202390e+02,2.3010400e+02,1.8815090e+02, &
& 3.6307990e+02,6.9200830e+02,3.6363190e+02,3.5103090e+02,3.4401120e+02, &
& 3.4078980e+02,3.1376160e+02,2.9041450e+02,2.7703380e+02,2.7082780e+02, &
& 2.6748920e+02,2.5114070e+02,4.0972790e+02,3.6020340e+02,3.2297840e+02, &
& 2.9499200e+02,2.5990550e+02,4.3329230e+02,8.4185430e+02,6.4480130e+02, &
& 4.8082360e+02,4.8558420e+02,4.5284900e+02,5.0743830e+02,3.9493050e+02, &
& 3.6947180e+02,3.4311040e+02,3.3190160e+02,3.2846280e+02,5.2037630e+02]
 vdw_dftd3_c6(12701:12800)=[&
& 4.7110690e+02,4.4137250e+02,4.1760220e+02,3.8146630e+02,5.1266580e+02, &
& 1.0908422e+03,8.1591040e+02,5.1590750e+02,5.0524280e+02,4.8924070e+02, &
& 4.9159590e+02,4.7374820e+02,4.9470080e+02,4.6468280e+02,4.7273450e+02, &
& 4.4109990e+02,4.2854250e+02,4.2578520e+02,4.4751580e+02,4.1156650e+02, &
& 4.5346600e+01,2.8998600e+01,8.3344750e+02,4.4256360e+02,2.8628310e+02, &
& 1.8798820e+02,1.2879820e+02,9.6142400e+01,7.1959700e+01,5.4816700e+01, &
& 9.9134020e+02,7.1475720e+02,6.4492820e+02,4.9384800e+02,3.7640650e+02, &
& 3.0798130e+02,2.4821220e+02,2.0073430e+02,1.6539476e+03,1.2894650e+03, &
& 1.0530789e+03,1.0073778e+03,9.1714520e+02,7.2238490e+02,7.8308020e+02, &
& 6.1378620e+02,6.4219290e+02,6.6532650e+02,5.1028620e+02,5.1419020e+02, &
& 6.1287020e+02,5.2847210e+02,4.4218700e+02,3.9224560e+02,3.3928530e+02, &
& 2.9196450e+02,1.8469699e+03,1.5403424e+03,1.3230311e+03,1.1750345e+03, &
& 1.0634756e+03,8.1065520e+02,9.0884210e+02,6.8268070e+02,7.4565500e+02, &
& 6.8814480e+02,5.7595500e+02,6.0264510e+02,7.6801030e+02,6.9681050e+02, &
& 6.1025270e+02,5.6048120e+02,5.0136210e+02,4.4626410e+02,2.2528586e+03, &
& 1.9789909e+03,1.7040270e+03,7.2344560e+02,1.7420729e+03,1.6661615e+03, &
& 1.6226978e+03,1.5829071e+03,1.5475750e+03,1.1920226e+03,1.3894684e+03, &
& 1.3360613e+03,1.3884173e+03,1.3581065e+03,1.3304473e+03,1.3159859e+03, &
& 1.0939351e+03,1.0543108e+03,9.5118510e+02,7.9676620e+02,8.0679990e+02, &
& 7.2412960e+02,6.5806160e+02,5.4245470e+02,5.0520370e+02,5.1767100e+02]
 vdw_dftd3_c6(12801:12900)=[&
& 7.8562040e+02,7.5633770e+02,6.8397230e+02,6.4657460e+02,5.9053860e+02, &
& 5.3625670e+02,2.0908542e+03,1.9370166e+03,1.6804201e+03,1.4797956e+03, &
& 1.4847146e+03,1.4366817e+03,1.4973302e+03,1.4464901e+03,7.4629200e+01, &
& 2.5592080e+02,3.2443230e+02,2.3897670e+02,1.7654080e+02,1.2010680e+02, &
& 8.5937000e+01,5.7827200e+01,3.7699540e+02,5.8570530e+02,5.8157180e+02, &
& 4.5471270e+02,3.6512350e+02,3.0470560e+02,2.4570480e+02,5.1281530e+02, &
& 1.0341456e+03,5.0045280e+02,4.8316910e+02,4.7364140e+02,4.7060760e+02, &
& 4.2699780e+02,3.9370530e+02,3.7565220e+02,3.6764140e+02,3.6693480e+02, &
& 3.3796290e+02,5.7232270e+02,4.9197380e+02,4.3422870e+02,3.9247750e+02, &
& 3.4185570e+02,6.0910480e+02,1.2632459e+03,9.2911650e+02,6.6400370e+02, &
& 6.7040190e+02,6.2262030e+02,7.1174240e+02,5.3554020e+02,5.0050420e+02, &
& 4.6368240e+02,4.4930820e+02,4.3997740e+02,7.2721640e+02,6.4540420e+02, &
& 5.9673110e+02,5.5972270e+02,5.0615680e+02,7.1003790e+02,1.6551217e+03, &
& 1.1811130e+03,7.0092550e+02,6.8627630e+02,6.6395860e+02,6.6917280e+02, &
& 6.5059990e+02,6.7380220e+02,6.3135970e+02,6.4587040e+02,5.9787350e+02, &
& 5.8033350e+02,5.7700390e+02,6.0871420e+02,5.5695720e+02,7.8038720e+02, &
& 4.2599100e+01,2.7620800e+01,7.3880890e+02,4.0322390e+02,2.6458080e+02, &
& 1.7562680e+02,1.2132320e+02,9.1102500e+01,6.8542500e+01,5.2435800e+01, &
& 8.8035320e+02,6.4829810e+02,5.8923010e+02,4.5575650e+02,3.5031540e+02, &
& 2.8821970e+02,2.3353600e+02,1.8976410e+02,1.4604707e+03,1.1597504e+03]
 vdw_dftd3_c6(12901:13000)=[&
& 9.5068620e+02,9.1232910e+02,8.3223310e+02,6.5570910e+02,7.1259280e+02, &
& 5.5895410e+02,5.8724740e+02,6.0719280e+02,4.6580690e+02,4.7225470e+02, &
& 5.6160790e+02,4.8807900e+02,4.1125650e+02,3.6640670e+02,3.1837520e+02, &
& 2.7512740e+02,1.6328526e+03,1.3843496e+03,1.1967064e+03,1.0670698e+03, &
& 9.6823440e+02,7.4142020e+02,8.2976170e+02,6.2645270e+02,6.8426530e+02, &
& 6.3255740e+02,5.2892880e+02,5.5516740e+02,7.0365600e+02,6.4257250e+02, &
& 5.6623950e+02,5.2199830e+02,4.6882570e+02,4.1891390e+02,1.9906514e+03, &
& 1.7742433e+03,1.5377374e+03,6.7054450e+02,1.5650227e+03,1.4988909e+03, &
& 1.4602740e+03,1.4248668e+03,1.3934393e+03,1.0807361e+03,1.2454654e+03, &
& 1.1985562e+03,1.2522798e+03,1.2251780e+03,1.2005428e+03,1.1871547e+03, &
& 9.9115750e+02,9.6178800e+02,8.7115170e+02,7.3191440e+02,7.4227510e+02, &
& 6.6851880e+02,6.0921920e+02,5.0370150e+02,4.6968660e+02,4.8188500e+02, &
& 7.2091450e+02,6.9765140e+02,6.3443450e+02,6.0166450e+02,5.5163790e+02, &
& 5.0269970e+02,1.8580508e+03,1.7424506e+03,1.5208103e+03,1.3484147e+03, &
& 1.3482588e+03,1.3049404e+03,1.3545615e+03,1.3096099e+03,6.9517200e+01, &
& 2.3360750e+02,2.9681730e+02,2.2150850e+02,1.6489220e+02,1.1316000e+02, &
& 8.1561000e+01,5.5390500e+01,3.4328010e+02,5.3247660e+02,5.3228640e+02, &
& 4.2023260e+02,3.3991220e+02,2.8511970e+02,2.3115710e+02,4.6941050e+02, &
& 9.2943790e+02,4.6193760e+02,4.4595790e+02,4.3713970e+02,4.3391220e+02, &
& 3.9561160e+02,3.6527420e+02,3.4851570e+02,3.4095800e+02,3.3910500e+02]
 vdw_dftd3_c6(13001:13100)=[&
& 3.1436170e+02,5.2550400e+02,4.5518330e+02,4.0401890e+02,3.6659830e+02, &
& 3.2071300e+02,5.5849190e+02,1.1339963e+03,8.4472340e+02,6.1231920e+02, &
& 6.1833280e+02,5.7504470e+02,6.5304060e+02,4.9697040e+02,4.6466380e+02, &
& 4.3087120e+02,4.1726470e+02,4.1002790e+02,6.6780870e+02,5.9661500e+02, &
& 5.5414990e+02,5.2141070e+02,4.7327510e+02,6.5424880e+02,1.4806088e+03, &
& 1.0722142e+03,6.4990970e+02,6.3638010e+02,6.1587840e+02,6.2006220e+02, &
& 6.0108450e+02,6.2420110e+02,5.8539210e+02,5.9771440e+02,5.5479820e+02, &
& 5.3868740e+02,5.3545990e+02,5.6410720e+02,5.1709820e+02,7.1609920e+02, &
& 6.5975010e+02,3.9353000e+01,2.5891600e+01,6.4222390e+02,3.6075580e+02, &
& 2.4029820e+02,1.6132870e+02,1.1241380e+02,8.4942800e+01,6.4260400e+01, &
& 4.9383000e+01,7.6674770e+02,5.7730020e+02,5.2875590e+02,4.1330440e+02, &
& 3.2049320e+02,2.6521130e+02,2.1610340e+02,1.7647630e+02,1.2652105e+03, &
& 1.0237513e+03,8.4249870e+02,8.1128040e+02,7.4159550e+02,5.8457670e+02, &
& 6.3689720e+02,5.0004450e+02,5.2755830e+02,5.4433150e+02,4.1776790e+02, &
& 4.2620470e+02,5.0560850e+02,4.4299990e+02,3.7599840e+02,3.3652400e+02, &
& 2.9379580e+02,2.5500680e+02,1.4165256e+03,1.2211434e+03,1.0628100e+03, &
& 9.5167080e+02,8.6588950e+02,6.6629380e+02,7.4429510e+02,5.6498080e+02, &
& 6.1707470e+02,5.7145890e+02,4.7746470e+02,5.0269480e+02,6.3350610e+02, &
& 5.8238420e+02,5.1649490e+02,4.7798250e+02,4.3109430e+02,3.8674400e+02, &
& 1.7264733e+03,1.5612112e+03,1.3624276e+03,6.1099070e+02,1.3802768e+03]
 vdw_dftd3_c6(13101:13200)=[&
& 1.3236943e+03,1.2900298e+03,1.2591135e+03,1.2316839e+03,9.6239750e+02, &
& 1.0962341e+03,1.0560748e+03,1.1088832e+03,1.0850974e+03,1.0635675e+03, &
& 1.0513822e+03,8.8206880e+02,8.6187340e+02,7.8390750e+02,6.6076360e+02, &
& 6.7117640e+02,6.0666990e+02,5.5447220e+02,4.5992100e+02,4.2942760e+02, &
& 4.4112500e+02,6.5021480e+02,6.3255270e+02,5.7855330e+02,5.5048090e+02, &
& 5.0671990e+02,4.6345510e+02,1.6209287e+03,1.5387012e+03,1.3515716e+03, &
& 1.2069993e+03,1.2026346e+03,1.1642737e+03,1.2034096e+03,1.1644393e+03, &
& 6.3655000e+01,2.0943010e+02,2.6676230e+02,2.0180580e+02,1.5143990e+02, &
& 1.0487960e+02,7.6174700e+01,5.2233500e+01,3.0698940e+02,4.7538840e+02, &
& 4.7855460e+02,3.8166000e+02,3.1107370e+02,2.6232930e+02,2.1388550e+02, &
& 4.2224100e+02,8.2019820e+02,4.1903830e+02,4.0459180e+02,3.9656360e+02, &
& 3.9324210e+02,3.6030430e+02,3.3316370e+02,3.1788470e+02,3.1087480e+02, &
& 3.0807170e+02,2.8748760e+02,4.7409680e+02,4.1391720e+02,3.6953950e+02, &
& 3.3667770e+02,2.9588320e+02,5.0328860e+02,9.9960180e+02,7.5469660e+02, &
& 5.5497370e+02,5.6047340e+02,5.2208970e+02,5.8893300e+02,4.5342630e+02, &
& 4.2417090e+02,3.9371240e+02,3.8104950e+02,3.7575480e+02,6.0266490e+02, &
& 5.4210550e+02,5.0590030e+02,4.7755630e+02,4.3515150e+02,5.9256180e+02, &
& 1.3007753e+03,9.5659970e+02,5.9241900e+02,5.8013530e+02,5.6163190e+02, &
& 5.6485240e+02,5.4591020e+02,5.6845330e+02,5.3359460e+02,5.4379510e+02, &
& 5.0613240e+02,4.9159050e+02,4.8851540e+02,5.1389840e+02,4.7198590e+02]
 vdw_dftd3_c6(13201:13300)=[&
& 6.4582280e+02,5.9747460e+02,5.4340360e+02,3.9390200e+01,2.5661400e+01, &
& 6.7790210e+02,3.7113480e+02,2.4394890e+02,1.6225790e+02,1.1232650e+02, &
& 8.4509400e+01,6.3709900e+01,4.8832900e+01,8.0810000e+02,5.9653470e+02, &
& 5.4254110e+02,4.2013030e+02,3.2333970e+02,2.6632530e+02,2.1607690e+02, &
& 1.7582140e+02,1.3393807e+03,1.0664078e+03,8.7454390e+02,8.3959150e+02, &
& 7.6607010e+02,6.0376280e+02,6.5618380e+02,5.1491990e+02,5.4109810e+02, &
& 5.5933980e+02,4.2928690e+02,4.3539440e+02,5.1743050e+02,4.5007850e+02, &
& 3.7961010e+02,3.3848080e+02,2.9439730e+02,2.5467830e+02,1.4977174e+03, &
& 1.2728662e+03,1.1011436e+03,9.8233900e+02,8.9165750e+02,6.8333990e+02, &
& 7.6450150e+02,5.7773560e+02,6.3093880e+02,5.8342570e+02,4.8800010e+02, &
& 5.1225330e+02,6.4862950e+02,5.9271140e+02,5.2269700e+02,4.8212230e+02, &
& 4.3331830e+02,3.8749410e+02,1.8253618e+03,1.6308213e+03,1.4145400e+03, &
& 6.1899030e+02,1.4389986e+03,1.3785166e+03,1.3430628e+03,1.3105457e+03, &
& 1.2816825e+03,9.9495010e+02,1.1448282e+03,1.1017061e+03,1.1520955e+03, &
& 1.1271846e+03,1.1045493e+03,1.0921865e+03,9.1229470e+02,8.8587570e+02, &
& 8.0282330e+02,6.7490990e+02,6.8456550e+02,6.1687960e+02,5.6243030e+02, &
& 4.6538980e+02,4.3411300e+02,4.4539470e+02,6.6487270e+02,6.4374060e+02, &
& 5.8578780e+02,5.5576230e+02,5.0986150e+02,4.6493840e+02,1.7053318e+03, &
& 1.6022866e+03,1.3995346e+03,1.2420205e+03,1.2414957e+03,1.2016927e+03, &
& 1.2468949e+03,1.2056353e+03,6.4164300e+01,2.1507820e+02,2.7332730e+02]
 vdw_dftd3_c6(13301:13400)=[&
& 2.0434870e+02,1.5235770e+02,1.0479180e+02,7.5702000e+01,5.1588100e+01, &
& 3.1607860e+02,4.9005630e+02,4.9022120e+02,3.8746930e+02,3.1376610e+02, &
& 2.6346580e+02,2.1388020e+02,4.3270980e+02,8.5443090e+02,4.2613390e+02, &
& 4.1139560e+02,4.0328980e+02,4.0028790e+02,3.6513690e+02,3.3722470e+02, &
& 3.2177920e+02,3.1479940e+02,3.1300420e+02,2.9033450e+02,4.8428840e+02, &
& 4.1985020e+02,3.7295910e+02,3.3865860e+02,2.9654870e+02,5.1497470e+02, &
& 1.0423341e+03,7.7762370e+02,5.6481820e+02,5.7045940e+02,5.3064150e+02, &
& 6.0213570e+02,4.5894910e+02,4.2920470e+02,3.9810440e+02,3.8554200e+02, &
& 3.7894250e+02,6.1575540e+02,5.5049820e+02,5.1159560e+02,4.8159190e+02, &
& 4.3741870e+02,6.0346930e+02,1.3601123e+03,9.8679870e+02,5.9998280e+02, &
& 5.8750600e+02,5.6861140e+02,5.7238360e+02,5.5475530e+02,5.7615700e+02, &
& 5.4043300e+02,5.5166610e+02,5.1223480e+02,4.9737970e+02,4.9437500e+02, &
& 5.2066780e+02,4.7743820e+02,6.6008460e+02,6.0852880e+02,5.5146290e+02, &
& 5.6143380e+02,3.5464600e+01,2.3812500e+01,5.3066030e+02,3.1052240e+02, &
& 2.1141050e+02,1.4424940e+02,1.0174200e+02,7.7553600e+01,5.9120400e+01, &
& 4.5720500e+01,6.3530720e+02,4.9355920e+02,4.5721240e+02,3.6283630e+02, &
& 2.8491700e+02,2.3770450e+02,1.9521900e+02,1.6053010e+02,1.0420654e+03, &
& 8.6452710e+02,7.1543320e+02,6.9243390e+02,6.3488380e+02,5.0093670e+02, &
& 5.4765220e+02,4.3064570e+02,4.5696500e+02,4.7005570e+02,3.6110090e+02, &
& 3.7162530e+02,4.3932630e+02,3.8941390e+02,3.3396020e+02,3.0083180e+02]
 vdw_dftd3_c6(13401:13500)=[&
& 2.6438450e+02,2.3089140e+02,1.1693798e+03,1.0303644e+03,9.0555540e+02, &
& 8.1580510e+02,7.4526460e+02,5.7758390e+02,6.4348640e+02,4.9230820e+02, &
& 5.3754740e+02,4.9907810e+02,4.1664440e+02,4.4047370e+02,5.5060050e+02, &
& 5.1094940e+02,4.5728590e+02,4.2551160e+02,3.8604100e+02,3.4827200e+02, &
& 1.4257528e+03,1.3130562e+03,1.1570873e+03,5.4014830e+02,1.1649532e+03, &
& 1.1189840e+03,1.0910275e+03,1.0652955e+03,1.0424806e+03,8.2351360e+02, &
& 9.2339310e+02,8.9129600e+02,9.4085810e+02,9.2091700e+02,9.0298170e+02, &
& 8.9222740e+02,7.5415480e+02,7.4392130e+02,6.8068180e+02,5.7655350e+02, &
& 5.8693470e+02,5.3325550e+02,4.8940580e+02,4.0785710e+02,3.8154050e+02, &
& 3.9259340e+02,5.6672940e+02,5.5534130e+02,5.1207880e+02,4.8950850e+02, &
& 4.5312570e+02,4.1656370e+02,1.3491716e+03,1.3007391e+03,1.1529664e+03, &
& 1.0402935e+03,1.0315219e+03,9.9893380e+02,1.0261425e+03,9.9407460e+02, &
& 5.6656700e+01,1.8083220e+02,2.3124090e+02,1.7834690e+02,1.3537830e+02, &
& 9.4961200e+01,6.9708000e+01,4.8438300e+01,2.6414790e+02,4.0811310e+02, &
& 4.1494650e+02,3.3578860e+02,2.7667050e+02,2.3508600e+02,1.9319560e+02, &
& 3.6655110e+02,6.9317420e+02,3.6802850e+02,3.5552300e+02,3.4841720e+02, &
& 3.4501650e+02,3.1830430e+02,2.9495420e+02,2.8145410e+02,2.7510220e+02, &
& 2.7123140e+02,2.5547700e+02,4.1318390e+02,3.6483360e+02,3.2842530e+02, &
& 3.0094220e+02,2.6617570e+02,4.3809620e+02,8.4371770e+02,6.4942160e+02, &
& 4.8694320e+02,4.9170440e+02,4.5926500e+02,5.1329930e+02,4.0162410e+02]
 vdw_dftd3_c6(13501:13600)=[&
& 3.7601550e+02,3.4951790e+02,3.3798580e+02,3.3491410e+02,5.2556660e+02, &
& 4.7735010e+02,4.4845320e+02,4.2527250e+02,3.8963030e+02,5.1948750e+02, &
& 1.0935492e+03,8.2179670e+02,5.2402380e+02,5.1321590e+02,4.9708550e+02, &
& 4.9923160e+02,4.8040280e+02,5.0216590e+02,4.7197170e+02,4.7976830e+02, &
& 4.4820990e+02,4.3552800e+02,4.3264200e+02,4.5417000e+02,4.1827710e+02, &
& 5.6287830e+02,5.2375380e+02,4.7921590e+02,4.8386730e+02,4.2619030e+02, &
& 3.3511700e+01,2.2677900e+01,4.9147210e+02,2.8982980e+02,1.9838070e+02, &
& 1.3600690e+02,9.6324300e+01,7.3663400e+01,5.6326800e+01,4.3677300e+01, &
& 5.8883180e+02,4.6010610e+02,4.2726470e+02,3.4027370e+02,2.6810830e+02, &
& 2.2424930e+02,1.8465170e+02,1.5222030e+02,9.6506260e+02,8.0438060e+02, &
& 6.6634240e+02,6.4567380e+02,5.9240830e+02,4.6775860e+02,5.1152360e+02, &
& 4.0259270e+02,4.2750640e+02,4.3943760e+02,3.3788310e+02,3.4820520e+02, &
& 4.1118360e+02,3.6542010e+02,3.1422920e+02,2.8359750e+02,2.4976170e+02, &
& 2.1857530e+02,1.0836129e+03,9.5871100e+02,8.4420740e+02,7.6153710e+02, &
& 6.9635620e+02,5.4076150e+02,6.0199360e+02,4.6158650e+02,5.0378310e+02, &
& 4.6803160e+02,3.9092010e+02,4.1343530e+02,5.1570310e+02,4.7948370e+02, &
& 4.3006080e+02,4.0076390e+02,3.6420700e+02,3.2914390e+02,1.3211273e+03, &
& 1.2211564e+03,1.0781412e+03,5.0792020e+02,1.0842087e+03,1.0418248e+03, &
& 1.0158802e+03,9.9198570e+02,9.7080130e+02,7.6877680e+02,8.5938870e+02, &
& 8.2970230e+02,8.7655810e+02,8.5800850e+02,8.4134750e+02,8.3123230e+02]
 vdw_dftd3_c6(13601:13700)=[&
& 7.0368220e+02,6.9544610e+02,6.3722330e+02,5.4056670e+02,5.5051890e+02, &
& 5.0082420e+02,4.6015330e+02,3.8410330e+02,3.5955540e+02,3.7003820e+02, &
& 5.3143810e+02,5.2144650e+02,4.8170840e+02,4.6101200e+02,4.2739750e+02, &
& 3.9350690e+02,1.2524472e+03,1.2109999e+03,1.0753003e+03,9.7245520e+02, &
& 9.6345570e+02,9.3311220e+02,9.5733480e+02,9.2763110e+02,5.3325500e+01, &
& 1.6893310e+02,2.1625690e+02,1.6758690e+02,1.2766650e+02,8.9935700e+01, &
& 6.6274300e+01,4.6290900e+01,2.4667920e+02,3.8086140e+02,3.8803830e+02, &
& 3.1510570e+02,2.6039720e+02,2.2177880e+02,1.8273850e+02,3.4334310e+02, &
& 6.4530690e+02,3.4546620e+02,3.3378450e+02,3.2712730e+02,3.2385110e+02, &
& 2.9919940e+02,2.7742410e+02,2.6476150e+02,2.5876330e+02,2.5485680e+02, &
& 2.4052350e+02,3.8698910e+02,3.4259940e+02,3.0908100e+02,2.8369920e+02, &
& 2.5143160e+02,4.1068500e+02,7.8537910e+02,6.0690870e+02,4.5710620e+02, &
& 4.6161660e+02,4.3145100e+02,4.8128810e+02,3.7795500e+02,3.5398970e+02, &
& 3.2921750e+02,3.1831050e+02,3.1569280e+02,4.9262420e+02,4.4836200e+02, &
& 4.2188850e+02,4.0057160e+02,3.6757650e+02,4.8764860e+02,1.0171183e+03, &
& 7.6771820e+02,4.9283570e+02,4.8268880e+02,4.6758270e+02,4.6943970e+02, &
& 4.5134990e+02,4.7210950e+02,4.4388260e+02,4.5095330e+02,4.2164500e+02, &
& 4.0975870e+02,4.0699850e+02,4.2696980e+02,3.9353520e+02,5.2755310e+02, &
& 4.9159800e+02,4.5048000e+02,4.5433550e+02,4.0148890e+02,3.7849390e+02, &
& 3.2091700e+01,2.1833800e+01,4.6520110e+02,2.7545630e+02,1.8914850e+02]
 vdw_dftd3_c6(13701:13800)=[&
& 1.3007010e+02,9.2371000e+01,7.0797500e+01,5.4251600e+01,4.2149900e+01, &
& 5.5760480e+02,4.3700990e+02,4.0638720e+02,3.2432370e+02,2.5607430e+02, &
& 2.1453080e+02,1.7695160e+02,1.4611660e+02,9.1385330e+02,7.6327990e+02, &
& 6.3264330e+02,6.1344560e+02,5.6306260e+02,4.4482350e+02,4.8647550e+02, &
& 3.8312130e+02,4.0695490e+02,4.1813480e+02,3.2172230e+02,3.3177490e+02, &
& 3.9148440e+02,3.4844170e+02,3.0012110e+02,2.7118850e+02,2.3915510e+02, &
& 2.0957880e+02,1.0265038e+03,9.0978800e+02,8.0197930e+02,7.2399270e+02, &
& 6.6240900e+02,5.1504420e+02,5.7309040e+02,4.4003780e+02,4.8010810e+02, &
& 4.4621570e+02,3.7286390e+02,3.9438550e+02,4.9129160e+02,4.5727960e+02, &
& 4.1067770e+02,3.8304320e+02,3.4847210e+02,3.1527170e+02,1.2516079e+03, &
& 1.1586040e+03,1.0239496e+03,4.8500190e+02,1.0291720e+03,9.8909200e+02, &
& 9.6450040e+02,9.4184450e+02,9.2175840e+02,7.3100210e+02,8.1592560e+02, &
& 7.8790030e+02,8.3246930e+02,8.1486390e+02,7.9906290e+02,7.8940200e+02, &
& 6.6892220e+02,6.6171440e+02,6.0682340e+02,5.1528520e+02,5.2488810e+02, &
& 4.7789070e+02,4.3938870e+02,3.6716870e+02,3.4385620e+02,3.5390700e+02, &
& 5.0669330e+02,4.9751750e+02,4.6009860e+02,4.4063800e+02,4.0889250e+02, &
& 3.7682880e+02,1.1875608e+03,1.1496435e+03,1.0218132e+03,9.2532050e+02, &
& 9.1638160e+02,8.8757730e+02,9.0998500e+02,8.8186200e+02,5.0936600e+01, &
& 1.6064720e+02,2.0578590e+02,1.5992840e+02,1.2211080e+02,8.6265500e+01, &
& 6.3736400e+01,4.4679200e+01,2.3456040e+02,3.6198670e+02,3.6923330e+02]
 vdw_dftd3_c6(13801:13900)=[&
& 3.0045310e+02,2.4874070e+02,2.1216920e+02,1.7511990e+02,3.2711390e+02, &
& 6.1266150e+02,3.2950500e+02,3.1842170e+02,3.1208150e+02,3.0891510e+02, &
& 2.8562920e+02,2.6494780e+02,2.5288280e+02,2.4714320e+02,2.4327100e+02, &
& 2.2984390e+02,3.6860280e+02,3.2682470e+02,2.9524020e+02,2.7128510e+02, &
& 2.4074110e+02,3.9146850e+02,7.4566370e+02,5.7749450e+02,4.3601580e+02, &
& 4.4033040e+02,4.1176280e+02,4.5883330e+02,3.6109350e+02,3.3829080e+02, &
& 3.1473130e+02,3.0428650e+02,3.0192540e+02,4.6950870e+02,4.2783570e+02, &
& 4.0295210e+02,3.8287770e+02,3.5168470e+02,4.6516200e+02,9.6540540e+02, &
& 7.3040540e+02,4.7064590e+02,4.6096660e+02,4.4658000e+02,4.4826620e+02, &
& 4.3078610e+02,4.5075260e+02,4.2390040e+02,4.3051250e+02,4.0272530e+02, &
& 3.9139780e+02,3.8873440e+02,4.0763660e+02,3.7590100e+02,5.0277570e+02, &
& 4.6891240e+02,4.3008890e+02,4.3348520e+02,3.8382310e+02,3.6201210e+02, &
& 3.4636100e+02,3.0675700e+01,2.0941000e+01,4.4619080e+02,2.6322300e+02, &
& 1.8065820e+02,1.2431330e+02,8.8391500e+01,6.7837200e+01,5.2061000e+01, &
& 4.0509200e+01,5.3480760e+02,4.1788500e+02,3.8834460e+02,3.0976480e+02, &
& 2.4460550e+02,2.0502050e+02,1.6922710e+02,1.3986320e+02,8.7757830e+02, &
& 7.3094660e+02,6.0554530e+02,5.8704740e+02,5.3875790e+02,4.2583430e+02, &
& 4.6540030e+02,3.6671160e+02,3.8919570e+02,3.9992890e+02,3.0793520e+02, &
& 3.1723870e+02,3.7421180e+02,3.3290660e+02,2.8673520e+02,2.5915840e+02, &
& 2.2864850e+02,2.0049290e+02,9.8572350e+02,8.7147550e+02,7.6763160e+02]
 vdw_dftd3_c6(13901:14000)=[&
& 6.9275660e+02,6.3375930e+02,4.9282900e+02,5.4833760e+02,4.2110840e+02, &
& 4.5928620e+02,4.2686340e+02,3.5697430e+02,3.7730490e+02,4.6998930e+02, &
& 4.3719460e+02,3.9255510e+02,3.6615260e+02,3.3316950e+02,3.0152720e+02, &
& 1.2018793e+03,1.1102792e+03,9.8044160e+02,4.6371250e+02,9.8614070e+02, &
& 9.4759800e+02,9.2399680e+02,9.0225470e+02,8.8297590e+02,6.9990580e+02, &
& 7.8232470e+02,7.5533300e+02,7.9726220e+02,7.8037200e+02,7.6520560e+02, &
& 7.5596270e+02,6.4035930e+02,6.3292020e+02,5.8031360e+02,4.9290480e+02, &
& 5.0200340e+02,4.5704880e+02,4.2025190e+02,3.5135070e+02,3.2911380e+02, &
& 3.3864570e+02,4.8495430e+02,4.7588520e+02,4.3996460e+02,4.2133170e+02, &
& 3.9100970e+02,3.6042890e+02,1.1396407e+03,1.1012980e+03,9.7817100e+02, &
& 8.8538660e+02,8.7728480e+02,8.4972830e+02,8.7157240e+02,8.4456390e+02, &
& 4.8647800e+01,1.5352370e+02,1.9665700e+02,1.5278290e+02,1.1673350e+02, &
& 8.2569800e+01,6.1097100e+01,4.2933900e+01,2.2431640e+02,3.4612620e+02, &
& 3.5282440e+02,2.8698060e+02,2.3761790e+02,2.0277280e+02,1.6748190e+02, &
& 3.1303240e+02,5.8698040e+02,3.1500110e+02,3.0444790e+02,2.9841150e+02, &
& 2.9542220e+02,2.7304550e+02,2.5329860e+02,2.4179700e+02,2.3632330e+02, &
& 2.3269810e+02,2.1974110e+02,3.5230670e+02,3.1225540e+02,2.8208180e+02, &
& 2.5925510e+02,2.3016390e+02,3.7464460e+02,7.1461700e+02,5.5272720e+02, &
& 4.1693630e+02,4.2109580e+02,3.9381950e+02,4.3907200e+02,3.4535640e+02, &
& 3.2361650e+02,3.0113910e+02,2.9118100e+02,2.8880380e+02,4.4908360e+02]
 vdw_dftd3_c6(14001:14100)=[&
& 4.0902650e+02,3.8517040e+02,3.6599200e+02,3.3623370e+02,4.4489470e+02, &
& 9.2564640e+02,6.9919040e+02,4.4999540e+02,4.4074490e+02,4.2699740e+02, &
& 4.2862430e+02,4.1203450e+02,4.3096490e+02,4.0531130e+02,4.1167450e+02, &
& 3.8504690e+02,3.7421380e+02,3.7166190e+02,3.8968240e+02,3.5937430e+02, &
& 4.8090340e+02,4.4847840e+02,4.1133730e+02,4.1467050e+02,3.6709540e+02, &
& 3.4630750e+02,3.3139460e+02,3.1715610e+02,2.9920300e+01,2.0585200e+01, &
& 4.1310610e+02,2.5030210e+02,1.7402940e+02,1.2076050e+02,8.6338200e+01, &
& 6.6493700e+01,5.1170200e+01,3.9897400e+01,4.9600030e+02,3.9561740e+02, &
& 3.7029810e+02,2.9804830e+02,2.3698310e+02,1.9943500e+02,1.6520990e+02, &
& 1.3693730e+02,8.1050400e+02,6.8633430e+02,5.7063520e+02,5.5496780e+02, &
& 5.1028000e+02,4.0344500e+02,4.4199080e+02,3.4847560e+02,3.7126550e+02, &
& 3.8080390e+02,2.9324220e+02,3.0381440e+02,3.5775570e+02,3.2047320e+02, &
& 2.7762260e+02,2.5175200e+02,2.2282360e+02,1.9592110e+02,9.1174980e+02, &
& 8.1778560e+02,7.2486420e+02,6.5664200e+02,6.0218630e+02,4.7017490e+02, &
& 5.2236590e+02,4.0291110e+02,4.3946430e+02,4.0902800e+02,3.4173120e+02, &
& 3.6219820e+02,4.4915100e+02,4.2020430e+02,3.7928220e+02,3.5482940e+02, &
& 3.2385100e+02,2.9389350e+02,1.1120309e+03,1.0396919e+03,9.2389520e+02, &
& 4.4756220e+02,9.2541710e+02,8.9016940e+02,8.6825330e+02,8.4803530e+02, &
& 8.3011710e+02,6.6249390e+02,7.3314800e+02,7.0872400e+02,7.5071540e+02, &
& 7.3493810e+02,7.2083100e+02,7.1192400e+02,6.0586800e+02,6.0248380e+02]
 vdw_dftd3_c6(14101:14200)=[&
& 5.5437230e+02,4.7211020e+02,4.8147760e+02,4.3962380e+02,4.0514540e+02, &
& 3.3947380e+02,3.1826990e+02,3.2785490e+02,4.6406660e+02,4.5741320e+02, &
& 4.2489250e+02,4.0796160e+02,3.7973280e+02,3.5093490e+02,1.0599693e+03, &
& 1.0346573e+03,9.2429380e+02,8.4192530e+02,8.3161820e+02,8.0561970e+02, &
& 8.2308290e+02,7.9816520e+02,4.7166200e+01,1.4625750e+02,1.8779790e+02, &
& 1.4752400e+02,1.1336050e+02,8.0647100e+01,5.9931300e+01,4.2319500e+01, &
& 2.1316680e+02,3.2853430e+02,3.3698220e+02,2.7645860e+02,2.3025850e+02, &
& 1.9722520e+02,1.6349210e+02,2.9887190e+02,5.5142200e+02,3.0295130e+02, &
& 2.9286460e+02,2.8701690e+02,2.8389530e+02,2.6347450e+02,2.4469520e+02, &
& 2.3357610e+02,2.2821280e+02,2.2401780e+02,2.1272240e+02,3.3741410e+02, &
& 3.0105020e+02,2.7320270e+02,2.5183060e+02,2.2426200e+02,3.5823720e+02, &
& 6.7072660e+02,5.2511790e+02,4.0071700e+02,4.0464840e+02,3.7897070e+02, &
& 4.2021610e+02,3.3359820e+02,3.1269220e+02,2.9116220e+02,2.8137530e+02, &
& 2.7990990e+02,4.3003950e+02,3.9395270e+02,3.7239970e+02,3.5473990e+02, &
& 3.2681510e+02,4.2730670e+02,8.6655170e+02,6.6355950e+02,4.3445080e+02, &
& 4.2554240e+02,4.1237370e+02,4.1361800e+02,3.9655650e+02,4.1578790e+02, &
& 3.9129420e+02,3.9685720e+02,3.7198460e+02,3.6161110e+02,3.5907550e+02, &
& 3.7609030e+02,3.4734250e+02,4.6039360e+02,4.3075880e+02,3.9639370e+02, &
& 3.9843040e+02,3.5538470e+02,3.3559480e+02,3.2133060e+02,3.0747100e+02, &
& 2.9886160e+02,4.5657800e+01,2.9528900e+01,8.1893710e+02,4.4031470e+02]
 vdw_dftd3_c6(14201:14300)=[&
& 2.8625250e+02,1.8889680e+02,1.3007910e+02,9.7551000e+01,7.3378400e+01, &
& 5.6174600e+01,9.7532750e+02,7.1024040e+02,6.4228990e+02,4.9356290e+02, &
& 3.7740890e+02,3.0961610e+02,2.5028720e+02,2.0307020e+02,1.6199398e+03, &
& 1.2768842e+03,1.0446215e+03,1.0005590e+03,9.1170000e+02,7.1835580e+02, &
& 7.7938520e+02,6.1131150e+02,6.4053000e+02,6.6311020e+02,5.0888790e+02, &
& 5.1381200e+02,6.1125030e+02,5.2855090e+02,4.4341250e+02,3.9408000e+02, &
& 3.4165520e+02,2.9473510e+02,1.8096860e+03,1.5245549e+03,1.3131378e+03, &
& 1.1682143e+03,1.0584151e+03,8.0855780e+02,9.0569280e+02,6.8207110e+02, &
& 7.4493760e+02,6.8807030e+02,5.7599280e+02,6.0330050e+02,7.6665730e+02, &
& 6.9724530e+02,6.1199840e+02,5.6288000e+02,5.0438410e+02,4.4980100e+02, &
& 2.2046359e+03,1.9557490e+03,1.6891285e+03,7.2531910e+02,1.7235094e+03, &
& 1.6499797e+03,1.6072448e+03,1.5680817e+03,1.5333057e+03,1.1844744e+03, &
& 1.3721977e+03,1.3193709e+03,1.3768441e+03,1.3469267e+03,1.3196608e+03, &
& 1.3051597e+03,1.0865229e+03,1.0501050e+03,9.4898330e+02,7.9601210e+02, &
& 8.0654170e+02,7.2504930e+02,6.5978220e+02,5.4489940e+02,5.0792100e+02, &
& 5.2062730e+02,7.8494960e+02,7.5725660e+02,6.8621680e+02,6.4945010e+02, &
& 5.9409560e+02,5.4036210e+02,2.0530994e+03,1.9172106e+03,1.6681370e+03, &
& 1.4735523e+03,1.4764021e+03,1.4289400e+03,1.4871204e+03,1.4371768e+03, &
& 7.4826500e+01,2.5479910e+02,3.2309330e+02,2.3925450e+02,1.7739330e+02, &
& 1.2133560e+02,8.7298300e+01,5.9255000e+01,3.7528460e+02,5.8220680e+02]
 vdw_dftd3_c6(14301:14400)=[&
& 5.7956000e+02,4.5468720e+02,3.6616070e+02,3.0633030e+02,2.4776880e+02, &
& 5.1164600e+02,1.0224025e+03,5.0090730e+02,4.8349150e+02,4.7405210e+02, &
& 4.7089950e+02,4.2804350e+02,3.9493390e+02,3.7686900e+02,3.6882120e+02, &
& 3.6776620e+02,3.3940980e+02,5.7125400e+02,4.9237270e+02,4.3551000e+02, &
& 3.9431080e+02,3.4421330e+02,6.0811700e+02,1.2478398e+03,9.2281660e+02, &
& 6.6415540e+02,6.7092560e+02,6.2334490e+02,7.1055610e+02,5.3741150e+02, &
& 5.0247720e+02,4.6583490e+02,4.5142980e+02,4.4250690e+02,7.2660300e+02, &
& 6.4640140e+02,5.9861930e+02,5.6215510e+02,5.0917420e+02,7.0976410e+02, &
& 1.6307553e+03,1.1718156e+03,7.0286450e+02,6.8822190e+02,6.6593910e+02, &
& 6.7082470e+02,6.5173940e+02,6.7537850e+02,6.3315820e+02,6.4713510e+02, &
& 5.9975110e+02,5.8222680e+02,5.7881040e+02,6.1017640e+02,5.5877490e+02, &
& 7.7870390e+02,7.1593350e+02,6.4697310e+02,6.6039260e+02,5.6540350e+02, &
& 5.3044050e+02,5.0586340e+02,4.8403950e+02,4.6395530e+02,7.7861880e+02, &
& 4.2434000e+01,2.7907300e+01,6.8807560e+02,3.8943570e+02,2.5945980e+02, &
& 1.7407890e+02,1.2124080e+02,9.1603500e+01,6.9320700e+01,5.3310500e+01, &
& 8.2197960e+02,6.2284710e+02,5.7078670e+02,4.4634760e+02,3.4597620e+02, &
& 2.8614180e+02,2.3303850e+02,1.9023650e+02,1.3514635e+03,1.1015257e+03, &
& 9.0751920e+02,8.7424690e+02,7.9942710e+02,6.2978200e+02,6.8687860e+02, &
& 5.3901900e+02,5.6943390e+02,5.8746310e+02,4.5054860e+02,4.6028610e+02, &
& 5.4568040e+02,4.7836640e+02,4.0593740e+02,3.6317430e+02,3.1692710e+02]
 vdw_dftd3_c6(14401:14500)=[&
& 2.7497470e+02,1.5131000e+03,1.3130553e+03,1.1445456e+03,1.0255674e+03, &
& 9.3337410e+02,7.1832310e+02,8.0242420e+02,6.0924720e+02,6.6583460e+02, &
& 6.1675520e+02,5.1489300e+02,5.4271640e+02,6.8352170e+02,6.2890890e+02, &
& 5.5784160e+02,5.1614750e+02,4.6537870e+02,4.1735550e+02,1.8427711e+03, &
& 1.6767003e+03,1.4657704e+03,6.5962730e+02,1.4834234e+03,1.4232913e+03, &
& 1.3872742e+03,1.3541823e+03,1.3248237e+03,1.0362745e+03,1.1761635e+03, &
& 1.1332571e+03,1.1934379e+03,1.1679468e+03,1.1448817e+03,1.1317397e+03, &
& 9.5008080e+02,9.2948800e+02,8.4577140e+02,7.1270800e+02,7.2421000e+02, &
& 6.5478150e+02,5.9854060e+02,4.9639040e+02,4.6349930e+02,4.7629930e+02, &
& 7.0116210e+02,6.8285450e+02,6.2480060e+02,5.9446280e+02,5.4711560e+02, &
& 5.0027110e+02,1.7331333e+03,1.6537357e+03,1.4552479e+03,1.3011870e+03, &
& 1.2953664e+03,1.2541422e+03,1.2955258e+03,1.2538462e+03,6.8700900e+01, &
& 2.2605860e+02,2.8775870e+02,2.1782360e+02,1.6333240e+02,1.1306740e+02, &
& 8.2111500e+01,5.6344500e+01,3.3128460e+02,5.1267930e+02,5.1659790e+02, &
& 4.1211230e+02,3.3577950e+02,2.8302980e+02,2.3064950e+02,4.5512070e+02, &
& 8.8099550e+02,4.5242230e+02,4.3671780e+02,4.2807670e+02,4.2446770e+02, &
& 3.8917610e+02,3.5985440e+02,3.4332520e+02,3.3576340e+02,3.3269860e+02, &
& 3.1058260e+02,5.1174530e+02,4.4693270e+02,3.9894250e+02,3.6333700e+02, &
& 3.1918450e+02,5.4238170e+02,1.0728240e+03,8.1213110e+02,5.9868520e+02, &
& 6.0472750e+02,5.6326320e+02,6.3469440e+02,4.8944530e+02,4.5783760e+02]
 vdw_dftd3_c6(14501:14600)=[&
& 4.2497290e+02,4.1137260e+02,4.0583020e+02,6.5042200e+02,5.8543790e+02, &
& 5.4638190e+02,5.1567690e+02,4.6975780e+02,6.3871850e+02,1.3938082e+03, &
& 1.0287291e+03,6.3957980e+02,6.2632970e+02,6.0635080e+02,6.0975710e+02, &
& 5.8924030e+02,6.1369780e+02,5.7611550e+02,5.8697030e+02,5.4649440e+02, &
& 5.3080340e+02,5.2748330e+02,5.5493190e+02,5.0965210e+02,6.9621110e+02, &
& 6.4434370e+02,5.8625070e+02,5.9480320e+02,5.1723940e+02,4.8623500e+02, &
& 4.6424430e+02,4.4400340e+02,4.2798590e+02,6.9795700e+02,6.3299910e+02, &
& 4.1596400e+01,2.7609300e+01,6.3913910e+02,3.7140510e+02,2.5081810e+02, &
& 1.6984210e+02,1.1903470e+02,9.0305200e+01,6.8559300e+01,5.2850100e+01, &
& 7.6472640e+02,5.9132420e+02,5.4586550e+02,4.3092720e+02,3.3654820e+02, &
& 2.7961500e+02,2.2867090e+02,1.8730610e+02,1.2519864e+03,1.0371898e+03, &
& 8.5756950e+02,8.2873520e+02,7.5924150e+02,5.9825830e+02,6.5412160e+02, &
& 5.1359410e+02,5.4475080e+02,5.6093530e+02,4.3021600e+02,4.4212850e+02, &
& 5.2324780e+02,4.6207160e+02,3.9458520e+02,3.5432910e+02,3.1033860e+02, &
& 2.7012000e+02,1.4036411e+03,1.2355778e+03,1.0837333e+03,9.7477890e+02, &
& 8.8933630e+02,6.8726090e+02,7.6655860e+02,5.8463360e+02,6.3898360e+02, &
& 5.9276900e+02,4.9433450e+02,5.2259600e+02,6.5509190e+02,6.0640770e+02, &
& 5.4093950e+02,5.0216500e+02,4.5432770e+02,4.0872120e+02,1.7098764e+03, &
& 1.5744445e+03,1.3849862e+03,6.3893210e+02,1.3958040e+03,1.3406235e+03, &
& 1.3070810e+03,1.2762220e+03,1.2488582e+03,9.8347160e+02,1.1050192e+03]
 vdw_dftd3_c6(14601:14700)=[&
& 1.0659790e+03,1.1267767e+03,1.1029019e+03,1.0813880e+03,1.0686785e+03, &
& 9.0128220e+02,8.8729660e+02,8.1034560e+02,6.8471570e+02,6.9676070e+02, &
& 6.3189030e+02,5.7901680e+02,4.8134860e+02,4.4988030e+02,4.6287080e+02, &
& 6.7290150e+02,6.5843910e+02,6.0554990e+02,5.7780740e+02,5.3356470e+02, &
& 4.8931160e+02,1.6162977e+03,1.5578929e+03,1.3787984e+03,1.2407328e+03, &
& 1.2312018e+03,1.1922019e+03,1.2266358e+03,1.1880528e+03,6.6885700e+01, &
& 2.1599080e+02,2.7562110e+02,2.1110780e+02,1.5930450e+02,1.1100780e+02, &
& 8.1023800e+01,5.5921600e+01,3.1569470e+02,4.8798840e+02,4.9488070e+02, &
& 3.9838950e+02,3.2670150e+02,2.7653670e+02,2.2630380e+02,4.3578520e+02, &
& 8.2978660e+02,4.3655630e+02,4.2148580e+02,4.1307940e+02,4.0921130e+02, &
& 3.7685240e+02,3.4888250e+02,3.3284050e+02,3.2538910e+02,3.2133350e+02, &
& 3.0179750e+02,4.9162320e+02,4.3241530e+02,3.8792100e+02,3.5446370e+02, &
& 3.1248750e+02,5.2015670e+02,1.0095291e+03,7.7362030e+02,5.7727760e+02, &
& 5.8300940e+02,5.4383670e+02,6.0923200e+02,4.7449870e+02,4.4399260e+02, &
& 4.1240560e+02,3.9894980e+02,3.9486670e+02,6.2469100e+02,5.6575460e+02, &
& 5.3022150e+02,5.0181220e+02,4.5858080e+02,6.1548530e+02,1.3081372e+03, &
& 9.7887630e+02,6.1971040e+02,6.0690590e+02,5.8770740e+02,5.9049670e+02, &
& 5.6898830e+02,5.9417810e+02,5.5818530e+02,5.6779730e+02,5.2988060e+02, &
& 5.1480880e+02,5.1148010e+02,5.3747970e+02,4.9440600e+02,6.6852110e+02, &
& 6.2087010e+02,5.6690460e+02,5.7334500e+02,5.0267320e+02,4.7305980e+02]
 vdw_dftd3_c6(14701:14800)=[&
& 4.5195060e+02,4.3215130e+02,4.1777890e+02,6.7100110e+02,6.1223660e+02, &
& 5.9404570e+02,4.0784000e+01,2.7292700e+01,6.0378280e+02,3.5667730e+02, &
& 2.4325530e+02,1.6591800e+02,1.1689210e+02,8.8996000e+01,6.7762500e+01, &
& 5.2352000e+01,7.2318320e+02,5.6618910e+02,5.2532650e+02,4.1751820e+02, &
& 3.2794200e+02,2.7346760e+02,2.2441640e+02,1.8436010e+02,1.1819863e+03, &
& 9.8827090e+02,8.1891600e+02,7.9312700e+02,7.2754910e+02,5.7361570e+02, &
& 6.2798280e+02,4.9346120e+02,5.2458090e+02,5.3944100e+02,4.1395970e+02, &
& 4.2696110e+02,5.0466270e+02,4.4793420e+02,3.8430870e+02,3.4611240e+02, &
& 3.0404010e+02,2.6534650e+02,1.3265708e+03,1.1770902e+03,1.0365660e+03, &
& 9.3473590e+02,8.5428840e+02,6.6224370e+02,7.3778930e+02,5.6459860e+02, &
& 6.1691080e+02,5.7289550e+02,4.7765720e+02,5.0575670e+02,6.3186020e+02, &
& 5.8725260e+02,5.2598230e+02,4.8950040e+02,4.4405350e+02,4.0048270e+02, &
& 1.6166889e+03,1.4983060e+03,1.3231799e+03,6.2088910e+02,1.3300657e+03, &
& 1.2782459e+03,1.2464709e+03,1.2172125e+03,1.1912765e+03,9.4248920e+02, &
& 1.0526025e+03,1.0162272e+03,1.0758235e+03,1.0531221e+03,1.0327245e+03, &
& 1.0203815e+03,8.6329250e+02,8.5331590e+02,7.8130510e+02,6.6165650e+02, &
& 6.7388930e+02,6.1248620e+02,5.6223340e+02,4.6834590e+02,4.3807020e+02, &
& 4.5102990e+02,6.4993220e+02,6.3784100e+02,5.8869220e+02,5.6290490e+02, &
& 5.2111770e+02,4.7899830e+02,1.5330137e+03,1.4856837e+03,1.3195792e+03, &
& 1.1925537e+03,1.1810674e+03,1.1437723e+03,1.1736700e+03,1.1372796e+03]
 vdw_dftd3_c6(14801:14900)=[&
& 6.5214600e+01,2.0772960e+02,2.6561960e+02,2.0516820e+02,1.5561880e+02, &
& 1.0902870e+02,7.9923000e+01,5.5445000e+01,3.0313050e+02,4.6821060e+02, &
& 4.7685900e+02,3.8638850e+02,3.1841620e+02,2.7043650e+02,2.2208000e+02, &
& 4.2021160e+02,7.9141640e+02,4.2299690e+02,4.0851000e+02,4.0032420e+02, &
& 3.9632850e+02,3.6605400e+02,3.3920620e+02,3.2362470e+02,3.1630020e+02, &
& 3.1163550e+02,2.9390630e+02,4.7478450e+02,4.1970260e+02,3.7792620e+02, &
& 3.4623010e+02,3.0609970e+02,5.0220180e+02,9.6252050e+02,7.4349910e+02, &
& 5.5925120e+02,5.6472600e+02,5.2739750e+02,5.8857680e+02,4.6147450e+02, &
& 4.3195260e+02,4.0145890e+02,3.8817970e+02,3.8501900e+02,6.0340750e+02, &
& 5.4877650e+02,5.1584790e+02,4.8923280e+02,4.4819060e+02,5.9616690e+02, &
& 1.2455740e+03,9.4022950e+02,6.0235500e+02,5.8993220e+02,5.7139080e+02, &
& 5.7376380e+02,5.5177910e+02,5.7720750e+02,5.4252310e+02,5.5128420e+02, &
& 5.1527560e+02,5.0071660e+02,4.9739780e+02,5.2220000e+02,4.8093310e+02, &
& 6.4584430e+02,6.0127470e+02,5.5039340e+02,5.5542350e+02,4.8976750e+02, &
& 4.6132380e+02,4.4097230e+02,4.2163540e+02,4.0842210e+02,6.4879600e+02, &
& 5.9436400e+02,5.7797360e+02,5.6324630e+02,3.8512100e+01,2.6111100e+01, &
& 5.4178230e+02,3.2704600e+02,2.2612850e+02,1.5586820e+02,1.1068200e+02, &
& 8.4740100e+01,6.4828700e+01,5.0274100e+01,6.4991410e+02,5.1713720e+02, &
& 4.8314350e+02,3.8756890e+02,3.0689040e+02,2.5730000e+02,2.1224570e+02, &
& 1.7515290e+02,1.0605365e+03,8.9715640e+02,7.4554500e+02,7.2428050e+02]
 vdw_dftd3_c6(14901:15000)=[&
& 6.6556000e+02,5.2534660e+02,5.7594770e+02,4.5323530e+02,4.8311010e+02, &
& 4.9587210e+02,3.8099950e+02,3.9473600e+02,4.6567730e+02,4.1619670e+02, &
& 3.5943220e+02,3.2508320e+02,2.8681650e+02,2.5132750e+02,1.1921368e+03, &
& 1.0684888e+03,9.4594700e+02,8.5598480e+02,7.8421710e+02,6.1069650e+02, &
& 6.7919130e+02,5.2232000e+02,5.7034140e+02,5.3043260e+02,4.4233610e+02, &
& 4.6916940e+02,5.8336540e+02,5.4506230e+02,4.9091940e+02,4.5847770e+02, &
& 4.1750700e+02,3.7792660e+02,1.4538408e+03,1.3583391e+03,1.2057906e+03, &
& 5.7911420e+02,1.2080329e+03,1.1618576e+03,1.1332140e+03,1.1068065e+03, &
& 1.0834069e+03,8.6266010e+02,9.5596420e+02,9.2391610e+02,9.7957880e+02, &
& 9.5900710e+02,9.4059640e+02,9.2908800e+02,7.8950440e+02,7.8448200e+02, &
& 7.2081970e+02,6.1248290e+02,6.2450880e+02,5.6934740e+02,5.2395200e+02, &
& 4.3782610e+02,4.1001810e+02,4.2247690e+02,6.0141130e+02,5.9244910e+02, &
& 5.4942480e+02,5.2688570e+02,4.8951330e+02,4.5144550e+02,1.3844068e+03, &
& 1.3507426e+03,1.2053587e+03,1.0957307e+03,1.0824781e+03,1.0484614e+03, &
& 1.0720311e+03,1.0394256e+03,6.1075700e+01,1.9088070e+02,2.4479400e+02, &
& 1.9130870e+02,1.4621180e+02,1.0328040e+02,7.6218600e+01,5.3309200e+01, &
& 2.7800940e+02,4.2891920e+02,4.3934570e+02,3.5920390e+02,2.9808060e+02, &
& 2.5442840e+02,2.1002330e+02,3.8791110e+02,7.1971410e+02,3.9289950e+02, &
& 3.7962690e+02,3.7198630e+02,3.6797730e+02,3.4117250e+02,3.1658390e+02, &
& 3.0208580e+02,2.9515410e+02,2.8991570e+02,2.7492600e+02,4.3890000e+02]
 vdw_dftd3_c6(15001:15100)=[&
& 3.9064520e+02,3.5361220e+02,3.2517910e+02,2.8869910e+02,4.6445980e+02, &
& 8.7507120e+02,6.8319770e+02,5.1943980e+02,5.2444490e+02,4.9061730e+02, &
& 5.4479300e+02,4.3100110e+02,4.0366930e+02,3.7552180e+02,3.6288510e+02, &
& 3.6089310e+02,5.5818460e+02,5.1050570e+02,4.8183620e+02,4.5832530e+02, &
& 4.2136600e+02,5.5369970e+02,1.1306124e+03,8.6338050e+02,5.6202230e+02, &
& 5.5046390e+02,5.3332520e+02,5.3511070e+02,5.1328320e+02,5.3811960e+02, &
& 5.0616170e+02,5.1361300e+02,4.8106970e+02,4.6760130e+02,4.6439200e+02, &
& 4.8688020e+02,4.4918810e+02,5.9761830e+02,5.5826740e+02,5.1282040e+02, &
& 5.1597410e+02,4.5859230e+02,4.3254320e+02,4.1380590e+02,3.9569230e+02, &
& 3.8429790e+02,6.0117960e+02,5.5369590e+02,5.4001160e+02,5.2742780e+02, &
& 4.9544740e+02,4.7701600e+01,3.0982700e+01,8.2128980e+02,4.4907760e+02, &
& 2.9529920e+02,1.9640820e+02,1.3586190e+02,1.0209130e+02,7.6828100e+01, &
& 5.8759900e+01,9.7872720e+02,7.2162860e+02,6.5646580e+02,5.0848770e+02, &
& 3.9145280e+02,3.2243460e+02,2.6153630e+02,2.1269100e+02,1.6235241e+03, &
& 1.2906007e+03,1.0581233e+03,1.0157559e+03,9.2671700e+02,7.3034690e+02, &
& 7.9366630e+02,6.2273900e+02,6.5432150e+02,6.7636930e+02,5.1902340e+02, &
& 5.2639910e+02,6.2602660e+02,5.4461780e+02,4.5945110e+02,4.0971550e+02, &
& 3.5633240e+02,3.0818050e+02,1.8154672e+03,1.5407355e+03,1.3325023e+03, &
& 1.1885781e+03,1.0787813e+03,8.2661910e+02,9.2479920e+02,6.9868860e+02, &
& 7.6296140e+02,7.0540290e+02,5.8992230e+02,6.1917780e+02,7.8433640e+02]
 vdw_dftd3_c6(15101:15200)=[&
& 7.1671230e+02,6.3213270e+02,5.8314470e+02,5.2415300e+02,4.6871230e+02, &
& 2.2133513e+03,1.9746939e+03,1.7122469e+03,7.4865870e+02,1.7417523e+03, &
& 1.6683774e+03,1.6254063e+03,1.5860020e+03,1.5510288e+03,1.2037458e+03, &
& 1.3861364e+03,1.3338527e+03,1.3939911e+03,1.3638159e+03,1.3364029e+03, &
& 1.3214462e+03,1.1036008e+03,1.0717256e+03,9.7115790e+02,8.1639790e+02, &
& 8.2799830e+02,7.4602310e+02,6.8007030e+02,5.6255190e+02,5.2462170e+02, &
& 5.3824540e+02,8.0393350e+02,7.7827560e+02,7.0823270e+02,6.7199770e+02, &
& 6.1653950e+02,5.6222180e+02,2.0671766e+03,1.9399392e+03,1.6935702e+03, &
& 1.5024640e+03,1.5019252e+03,1.4536878e+03,1.5083332e+03,1.4583342e+03, &
& 7.7705500e+01,2.6024430e+02,3.3086670e+02,2.4738310e+02,1.8444310e+02, &
& 1.2675340e+02,9.1438800e+01,6.2121900e+01,3.8227080e+02,5.9297950e+02, &
& 5.9319050e+02,4.6899060e+02,3.7986120e+02,3.1896080e+02,2.5886520e+02, &
& 5.2346900e+02,1.0345424e+03,5.1544090e+02,4.9759520e+02,4.8774150e+02, &
& 4.8406800e+02,4.4152590e+02,4.0774620e+02,3.8903940e+02,3.8057220e+02, &
& 3.7831310e+02,3.5099820e+02,5.8593320e+02,5.0806740e+02,4.5140640e+02, &
& 4.0992700e+02,3.5893240e+02,6.2302850e+02,1.2623248e+03,9.4131820e+02, &
& 6.8339070e+02,6.9013390e+02,6.4188840e+02,7.2847950e+02,5.5498020e+02, &
& 5.1892730e+02,4.8122640e+02,4.6593230e+02,4.5799090e+02,7.4456260e+02, &
& 6.6567970e+02,6.1872370e+02,5.8251630e+02,5.2912450e+02,7.3044860e+02, &
& 1.6478674e+03,1.1947861e+03,7.2565350e+02,7.1054890e+02,6.8768570e+02]
 vdw_dftd3_c6(15201:15300)=[&
& 6.9226220e+02,6.7081490e+02,6.9684430e+02,6.5357440e+02,6.6719620e+02, &
& 6.1947600e+02,6.0150620e+02,5.9787770e+02,6.2970880e+02,5.7740830e+02, &
& 7.9873450e+02,7.3626140e+02,6.6710040e+02,6.7912190e+02,5.8515020e+02, &
& 5.4932980e+02,5.2401710e+02,5.0114870e+02,4.8150990e+02,7.9848790e+02, &
& 7.1909940e+02,6.9321150e+02,6.7161670e+02,6.2396020e+02,8.2204820e+02, &
& 8.3609000e+01,5.0796500e+01,2.0637836e+03,9.4042740e+02,5.6781070e+02, &
& 3.5548600e+02,2.3551600e+02,1.7184720e+02,1.2619530e+02,9.4703500e+01, &
& 2.4334034e+03,1.5570501e+03,1.3544541e+03,9.8580450e+02,7.2244190e+02, &
& 5.7742680e+02,4.5525930e+02,3.6144930e+02,4.2189528e+03,2.9555781e+03, &
& 2.3628855e+03,2.2256317e+03,2.0060593e+03,1.5850984e+03,1.6883164e+03, &
& 1.3245409e+03,1.3487863e+03,1.4111951e+03,1.0878878e+03,1.0558450e+03, &
& 1.2726682e+03,1.0532146e+03,8.5217950e+02,7.4155110e+02,6.2920990e+02, &
& 5.3228900e+02,4.6920594e+03,3.5533992e+03,2.9477908e+03,2.5643290e+03, &
& 2.2922709e+03,1.7126002e+03,1.9349917e+03,1.4205909e+03,1.5449801e+03, &
& 1.4136504e+03,1.1968365e+03,1.2247085e+03,1.6042050e+03,1.4037977e+03, &
& 1.1915335e+03,1.0753755e+03,9.4459510e+02,8.2692720e+02,5.7528877e+03, &
& 4.6383134e+03,3.8532093e+03,1.4235132e+03,4.0407923e+03,3.8327972e+03, &
& 3.7253298e+03,3.6276787e+03,3.5407934e+03,2.6359841e+03,3.2830964e+03, &
& 3.1457261e+03,3.1446702e+03,3.0720386e+03,3.0045907e+03,2.9758018e+03, &
& 2.4223257e+03,2.2460574e+03,1.9858562e+03,1.6449966e+03,1.6502454e+03]
 vdw_dftd3_c6(15301:15400)=[&
& 1.4557663e+03,1.3051372e+03,1.0638806e+03,9.8600810e+02,1.0014312e+03, &
& 1.6367789e+03,1.5272655e+03,1.3408219e+03,1.2476338e+03,1.1190260e+03, &
& 1.0003262e+03,5.1851636e+03,4.4613947e+03,3.7411085e+03,3.1788167e+03, &
& 3.2553913e+03,3.1461615e+03,3.3483818e+03,3.2202388e+03,1.4255890e+02, &
& 5.3986610e+02,6.7929310e+02,4.6840690e+02,3.3516710e+02,2.1993730e+02, &
& 1.5284480e+02,9.9280200e+01,8.0746090e+02,1.2655466e+03,1.2114303e+03, &
& 9.0234430e+02,7.0028070e+02,5.7180250e+02,4.5094310e+02,1.0806042e+03, &
& 2.4004019e+03,1.0031278e+03,9.7117090e+02,9.5238070e+02,9.5158400e+02, &
& 8.3982780e+02,7.6965310e+02,7.3532570e+02,7.2111550e+02,7.3354280e+02, &
& 6.5214040e+02,1.1763896e+03,9.7240690e+02,8.3549110e+02,7.4234170e+02, &
& 6.3468070e+02,1.2751297e+03,2.9585401e+03,2.0352098e+03,1.3434836e+03, &
& 1.3539176e+03,1.2515281e+03,1.4858293e+03,1.0500362e+03,9.8072260e+02, &
& 9.0551290e+02,8.8003940e+02,8.4398030e+02,1.5005632e+03,1.2849412e+03, &
& 1.1605961e+03,1.0727423e+03,9.5386310e+02,1.4475205e+03,3.9627043e+03, &
& 2.6130493e+03,1.3768762e+03,1.3474829e+03,1.3018447e+03,1.3198844e+03, &
& 1.3036758e+03,1.3293323e+03,1.2401840e+03,1.2827144e+03,1.1691884e+03, &
& 1.1330697e+03,1.1279011e+03,1.1968168e+03,1.0857933e+03,1.6232598e+03, &
& 1.4589496e+03,1.2878783e+03,1.3411577e+03,1.0896139e+03,1.0148760e+03, &
& 9.6392390e+02,9.2374840e+02,8.6831320e+02,1.6036402e+03,1.3818325e+03, &
& 1.3022054e+03,1.2430479e+03,1.1320728e+03,1.6245332e+03,3.7875943e+03]
 vdw_dftd3_c6(15401:15500)=[&
& 5.1107300e+01,3.3639800e+01,8.0602430e+02,4.6302300e+02,3.1058590e+02, &
& 2.0918580e+02,1.4595370e+02,1.1033330e+02,8.3476100e+01,6.4144300e+01, &
& 9.6353500e+02,7.3854190e+02,6.7947260e+02,5.3394560e+02,4.1534530e+02, &
& 3.4413770e+02,2.8065670e+02,2.2928490e+02,1.5796611e+03,1.2998428e+03, &
& 1.0730233e+03,1.0353252e+03,9.4761900e+02,7.4630200e+02,8.1529190e+02, &
& 6.3968180e+02,6.7745550e+02,6.9823130e+02,5.3518630e+02,5.4868120e+02, &
& 6.5023470e+02,5.7221090e+02,4.8704720e+02,4.3643770e+02,3.8138700e+02, &
& 3.3123010e+02,1.7697228e+03,1.5487458e+03,1.3545806e+03,1.2161834e+03, &
& 1.1081864e+03,8.5438760e+02,9.5376820e+02,7.2552720e+02,7.9314780e+02, &
& 7.3516550e+02,6.1302710e+02,6.4739340e+02,8.1366740e+02,7.5110850e+02, &
& 6.6812780e+02,6.1915920e+02,5.5909410e+02,5.0201500e+02,2.1555972e+03, &
& 1.9751996e+03,1.7327032e+03,7.8954030e+02,1.7491123e+03,1.6792118e+03, &
& 1.6369858e+03,1.5981651e+03,1.5637364e+03,1.2273570e+03,1.3851029e+03, &
& 1.3354249e+03,1.4098902e+03,1.3799203e+03,1.3528654e+03,1.3371510e+03, &
& 1.1251350e+03,1.1047547e+03,1.0070882e+03,8.4953530e+02,8.6391120e+02, &
& 7.8218420e+02,7.1574490e+02,5.9397880e+02,5.5473410e+02,5.7048480e+02, &
& 8.3486440e+02,8.1522210e+02,7.4786620e+02,7.1256000e+02,6.5680530e+02, &
& 6.0129860e+02,2.0330823e+03,1.9515778e+03,1.7226039e+03,1.5452383e+03, &
& 1.5354151e+03,1.4866193e+03,1.5322963e+03,1.4835856e+03,8.2527300e+01, &
& 2.6898140e+02,3.4282380e+02,2.6102340e+02,1.9621380e+02,1.3608950e+02]
 vdw_dftd3_c6(15501:15600)=[&
& 9.8908000e+01,6.7858600e+01,3.9343830e+02,6.0867090e+02,6.1547850e+02, &
& 4.9327920e+02,4.0312400e+02,3.4035860e+02,2.7775450e+02,5.4150520e+02, &
& 1.0393542e+03,5.4062540e+02,5.2182910e+02,5.1141880e+02,5.0681950e+02, &
& 4.6576900e+02,4.3087900e+02,4.1102320e+02,4.0187340e+02,3.9745630e+02, &
& 3.7225130e+02,6.1037340e+02,5.3502330e+02,4.7872210e+02,4.3661290e+02, &
& 3.8406800e+02,6.4577530e+02,1.2648795e+03,9.6384860e+02,7.1506360e+02, &
& 7.2218160e+02,6.7303520e+02,7.5600910e+02,5.8589660e+02,5.4801980e+02, &
& 5.0873460e+02,4.9222520e+02,4.8648840e+02,7.7511410e+02,6.9994150e+02, &
& 6.5463420e+02,6.1866240e+02,5.6435520e+02,7.6265600e+02,1.6407829e+03, &
& 1.2201975e+03,7.6564160e+02,7.4979030e+02,7.2595040e+02,7.2971000e+02, &
& 7.0402290e+02,7.3440270e+02,6.8961450e+02,7.0203220e+02,6.5440840e+02, &
& 6.3570180e+02,6.3167050e+02,6.6425610e+02,6.1046730e+02,8.2974450e+02, &
& 7.6921560e+02,7.0102290e+02,7.1006080e+02,6.1988720e+02,5.8290540e+02, &
& 5.5659850e+02,5.3213080e+02,5.1369080e+02,8.3188530e+02,7.5684580e+02, &
& 7.3324790e+02,7.1262070e+02,6.6475080e+02,8.5884400e+02,1.6304211e+03, &
& 9.0604480e+02,5.8042000e+01,3.7717100e+01,1.0088719e+03,5.4842030e+02, &
& 3.6005530e+02,2.3921200e+02,1.6541170e+02,1.2432800e+02,9.3640800e+01, &
& 7.1718500e+01,1.2017562e+03,8.8196280e+02,8.0168460e+02,6.2020400e+02, &
& 4.7697090e+02,3.9262180e+02,3.1831370e+02,2.5881200e+02,2.0007417e+03, &
& 1.5795049e+03,1.2941588e+03,1.2419423e+03,1.1328667e+03,8.9296930e+02]
 vdw_dftd3_c6(15601:15700)=[&
& 9.6999930e+02,7.6115260e+02,7.9925150e+02,8.2638550e+02,6.3431240e+02, &
& 6.4280130e+02,7.6433880e+02,6.6430070e+02,5.5996260e+02,4.9907290e+02, &
& 4.3384090e+02,3.7508740e+02,2.2372776e+03,1.8860887e+03,1.6294376e+03, &
& 1.4526448e+03,1.3181488e+03,1.0095160e+03,1.1298359e+03,8.5313360e+02, &
& 9.3156900e+02,8.6120120e+02,7.2051370e+02,7.5592110e+02,9.5811210e+02, &
& 8.7479670e+02,7.7106630e+02,7.1100220e+02,6.3878920e+02,5.7099520e+02, &
& 2.7306836e+03,2.4191244e+03,2.0947841e+03,9.1310340e+02,2.1341663e+03, &
& 2.0426932e+03,1.9899080e+03,1.9415270e+03,1.8985844e+03,1.4723209e+03, &
& 1.7011137e+03,1.6379975e+03,1.7057192e+03,1.6687205e+03,1.6350831e+03, &
& 1.6168530e+03,1.3505554e+03,1.3091260e+03,1.1857100e+03,9.9656680e+02, &
& 1.0106039e+03,9.1025670e+02,8.2962710e+02,6.8620440e+02,6.3997490e+02, &
& 6.5657070e+02,9.8206910e+02,9.5002070e+02,8.6404130e+02,8.1955770e+02, &
& 7.5162660e+02,6.8516600e+02,2.5440695e+03,2.3748028e+03,2.0713967e+03, &
& 1.8359296e+03,1.8365610e+03,1.7774636e+03,1.8453202e+03,1.7839174e+03, &
& 9.4641800e+01,3.1781250e+02,4.0399690e+02,3.0152640e+02,2.2458760e+02, &
& 1.5426790e+02,1.1130120e+02,7.5746100e+01,4.6715670e+02,7.2473020e+02, &
& 7.2426630e+02,5.7194340e+02,4.6283220e+02,3.8840280e+02,3.1507380e+02, &
& 6.3924470e+02,1.2682110e+03,6.2863700e+02,6.0726950e+02,5.9522520e+02, &
& 5.9085660e+02,5.3866860e+02,4.9744160e+02,4.7468740e+02,4.6439490e+02, &
& 4.6183290e+02,4.2819990e+02,7.1520850e+02,6.1960740e+02,5.5013410e+02]
 vdw_dftd3_c6(15701:15800)=[&
& 4.9933630e+02,4.3701980e+02,7.6067450e+02,1.5482877e+03,1.1522222e+03, &
& 8.3361520e+02,8.4145560e+02,7.8311230e+02,8.8973240e+02,6.7685180e+02, &
& 6.3296580e+02,5.8702360e+02,5.6847740e+02,5.5862930e+02,9.0926210e+02, &
& 8.1233800e+02,7.5464760e+02,7.1021200e+02,6.4484300e+02,8.9097610e+02, &
& 2.0249316e+03,1.4635069e+03,8.8503310e+02,8.6659780e+02,8.3870440e+02, &
& 8.4447740e+02,8.1850700e+02,8.4998270e+02,7.9715610e+02,8.1406210e+02, &
& 7.5550250e+02,7.3357320e+02,7.2916890e+02,7.6808740e+02,7.0415860e+02, &
& 9.7550990e+02,8.9859560e+02,8.1374230e+02,8.2881920e+02,7.1350780e+02, &
& 6.6976280e+02,6.3893520e+02,6.1116560e+02,5.8708670e+02,9.7511530e+02, &
& 8.7752310e+02,8.4561490e+02,8.1904220e+02,7.6065040e+02,1.0026461e+03, &
& 1.9926647e+03,1.0474302e+03,1.2245933e+03,5.2264500e+01,3.4332900e+01, &
& 8.9070150e+02,4.8502270e+02,3.2084970e+02,2.1465050e+02,1.4930470e+02, &
& 1.1273380e+02,8.5262100e+01,6.5533100e+01,1.0608040e+03,7.7888060e+02, &
& 7.1034880e+02,5.5219130e+02,4.2680750e+02,3.5264770e+02,2.8699980e+02, &
& 2.3418470e+02,1.7789468e+03,1.3939683e+03,1.1425935e+03,1.0979705e+03, &
& 1.0022471e+03,7.9096650e+02,8.5910440e+02,6.7498040e+02,7.0900470e+02, &
& 7.3240820e+02,5.6299640e+02,5.7131500e+02,6.7870160e+02,5.9190790e+02, &
& 5.0094070e+02,4.4774380e+02,3.9043140e+02,3.3858330e+02,1.9913447e+03, &
& 1.6656547e+03,1.4407783e+03,1.2860297e+03,1.1683462e+03,8.9688390e+02, &
& 1.0030323e+03,7.5931940e+02,8.2841280e+02,7.6640860e+02,6.4182000e+02]
 vdw_dftd3_c6(15801:15900)=[&
& 6.7342870e+02,8.5158740e+02,7.7929980e+02,6.8907930e+02,6.3681840e+02, &
& 5.7361210e+02,5.1407080e+02,2.4381558e+03,2.1382977e+03,1.8526946e+03, &
& 8.1584440e+02,1.8889925e+03,1.8057716e+03,1.7590005e+03,1.7161451e+03, &
& 1.6781174e+03,1.3048651e+03,1.5104078e+03,1.4576760e+03,1.5075770e+03, &
& 1.4748115e+03,1.4450628e+03,1.4287766e+03,1.1975147e+03,1.1607498e+03, &
& 1.0529859e+03,8.8698680e+02,8.9986720e+02,8.1184020e+02,7.4102110e+02, &
& 6.1428030e+02,5.7339400e+02,5.8842050e+02,8.7453780e+02,8.4698840e+02, &
& 7.7232730e+02,7.3386420e+02,6.7459880e+02,6.1636260e+02,2.2655221e+03, &
& 2.1000716e+03,1.8334406e+03,1.6280585e+03,1.6281996e+03,1.5757649e+03, &
& 1.6336303e+03,1.5794150e+03,8.4723800e+01,2.8145990e+02,3.5859140e+02, &
& 2.6924380e+02,2.0157990e+02,1.3931210e+02,1.0105790e+02,6.9270700e+01, &
& 4.1351280e+02,6.4146750e+02,6.4236160e+02,5.0971630e+02,4.1427030e+02, &
& 3.4884970e+02,2.8407250e+02,5.6838310e+02,1.1243632e+03,5.5973470e+02, &
& 5.4164230e+02,5.3075760e+02,5.2667650e+02,4.8098950e+02,4.4458610e+02, &
& 4.2438350e+02,4.1509660e+02,4.1210000e+02,3.8322820e+02,6.3562870e+02, &
& 5.5267960e+02,4.9229290e+02,4.4797180e+02,3.9323670e+02,6.7715660e+02, &
& 1.3743437e+03,1.0260397e+03,7.4279230e+02,7.4895660e+02,6.9892350e+02, &
& 7.9275350e+02,6.0535870e+02,5.6644570e+02,5.2569600e+02,5.0889220e+02, &
& 5.0075690e+02,8.0885400e+02,7.2462440e+02,6.7473780e+02,6.3619670e+02, &
& 5.7901430e+02,7.9465520e+02,1.8031914e+03,1.3050880e+03,7.9095680e+02]
 vdw_dftd3_c6(15901:16000)=[&
& 7.7448190e+02,7.4970660e+02,7.5473340e+02,7.3034120e+02,7.5925810e+02, &
& 7.1236590e+02,7.2723060e+02,6.7539950e+02,6.5589880e+02,6.5187600e+02, &
& 6.8605020e+02,6.2962830e+02,8.6895920e+02,8.0143340e+02,7.2695200e+02, &
& 7.3934260e+02,6.3931090e+02,6.0057540e+02,5.7328530e+02,5.4847560e+02, &
& 5.2767310e+02,8.6865890e+02,7.8363670e+02,7.5634500e+02,7.3358480e+02, &
& 6.8269100e+02,8.9435100e+02,1.7716049e+03,9.3586320e+02,1.0934914e+03, &
& 9.8037580e+02,4.6755900e+01,3.1074600e+01,7.3198050e+02,4.1975640e+02, &
& 2.8243120e+02,1.9102230e+02,1.3387230e+02,1.0159960e+02,7.7176400e+01, &
& 5.9527100e+01,8.7518610e+02,6.6958510e+02,6.1654260e+02,4.8533720e+02, &
& 3.7850250e+02,3.1437000e+02,2.5708750e+02,2.1063660e+02,1.4387994e+03, &
& 1.1798248e+03,9.7372410e+02,9.3993750e+02,8.6049210e+02,6.7850520e+02, &
& 7.4061370e+02,5.8183780e+02,6.1569900e+02,6.3436640e+02,4.8701500e+02, &
& 4.9903190e+02,5.9084610e+02,5.2052680e+02,4.4390020e+02,3.9844930e+02, &
& 3.4891220e+02,3.0370810e+02,1.6125430e+03,1.4064978e+03,1.2301136e+03, &
& 1.1047666e+03,1.0071282e+03,7.7757670e+02,8.6755080e+02,6.6100390e+02, &
& 7.2198170e+02,6.6945830e+02,5.5904400e+02,5.8989000e+02,7.4043240e+02, &
& 6.8380180e+02,6.0902530e+02,5.6501830e+02,5.1096120e+02,4.5956760e+02, &
& 1.9652428e+03,1.7949059e+03,1.5740959e+03,7.1985720e+02,1.5898595e+03, &
& 1.5259822e+03,1.4875408e+03,1.4521965e+03,1.4208475e+03,1.1161138e+03, &
& 1.2609766e+03,1.2159398e+03,1.2808410e+03,1.2535493e+03,1.2289183e+03]
 vdw_dftd3_c6(16001:16100)=[&
& 1.2145773e+03,1.0226803e+03,1.0038975e+03,9.1571760e+02,7.7355940e+02, &
& 7.8664370e+02,7.1280370e+02,6.5278320e+02,5.4266550e+02,5.0716770e+02, &
& 5.2145390e+02,7.6086200e+02,7.4290150e+02,6.8212270e+02,6.5042490e+02, &
& 6.0026280e+02,5.5029950e+02,1.8525059e+03,1.7734791e+03,1.5651457e+03, &
& 1.4048813e+03,1.3963790e+03,1.3520713e+03,1.3933621e+03,1.3490495e+03, &
& 7.5206600e+01,2.4403010e+02,3.1132040e+02,2.3766340e+02,1.7925610e+02, &
& 1.2489310e+02,9.1186300e+01,6.2986700e+01,3.5715820e+02,5.5232750e+02, &
& 5.5871280e+02,4.4861130e+02,3.6744990e+02,3.1093340e+02,2.5443970e+02, &
& 4.9305900e+02,9.4512240e+02,4.9217970e+02,4.7531010e+02,4.6586480e+02, &
& 4.6166860e+02,4.2443750e+02,3.9286320e+02,3.7486590e+02,3.6651980e+02, &
& 3.6235940e+02,3.3963850e+02,5.5481070e+02,4.8695430e+02,4.3639410e+02, &
& 3.9861480e+02,3.5133960e+02,5.8839800e+02,1.1509048e+03,8.7728230e+02, &
& 6.5137000e+02,6.5782330e+02,6.1355160e+02,6.8895110e+02,5.3468200e+02, &
& 5.0038440e+02,4.6479060e+02,4.4970340e+02,4.4450890e+02,7.0553710e+02, &
& 6.3759540e+02,5.9686560e+02,5.6459640e+02,5.1574020e+02,6.9501880e+02, &
& 1.4942771e+03,1.1109199e+03,6.9816140e+02,6.8372500e+02,6.6206630e+02, &
& 6.6540380e+02,6.4179300e+02,6.6949760e+02,6.2884090e+02,6.4005220e+02, &
& 5.9680940e+02,5.7978670e+02,5.7605760e+02,6.0540230e+02,5.5673610e+02, &
& 7.5552630e+02,7.0091760e+02,6.3934100e+02,6.4727080e+02,5.6614580e+02, &
& 5.3275200e+02,5.0897860e+02,4.8681770e+02,4.7017820e+02,7.5808150e+02]
 vdw_dftd3_c6(16101:16200)=[&
& 6.9014330e+02,6.6895140e+02,6.5052430e+02,6.0748900e+02,7.8264720e+02, &
& 1.4842630e+03,8.2601520e+02,9.5479790e+02,8.5410020e+02,7.5386170e+02, &
& 4.5123600e+01,3.0097800e+01,6.9501280e+02,4.0190350e+02,2.7145390e+02, &
& 1.8410950e+02,1.2930040e+02,9.8281200e+01,7.4758400e+01,5.7729000e+01, &
& 8.3149520e+02,6.4031040e+02,5.9077440e+02,4.6630230e+02,3.6444330e+02, &
& 3.0311500e+02,2.4821860e+02,2.0361560e+02,1.3643503e+03,1.1254292e+03, &
& 9.2988130e+02,8.9845340e+02,8.2298300e+02,6.4899350e+02,7.0890860e+02, &
& 5.5705720e+02,5.9015540e+02,6.0771500e+02,4.6659660e+02,4.7890520e+02, &
& 5.6662400e+02,5.0022560e+02,4.2735080e+02,3.8401570e+02,3.3665520e+02, &
& 2.9334880e+02,1.5296978e+03,1.3413275e+03,1.1753725e+03,1.0568235e+03, &
& 9.6412660e+02,7.4533540e+02,8.3117090e+02,6.3418960e+02,6.9270260e+02, &
& 6.4261150e+02,5.3650370e+02,5.6657800e+02,7.1009360e+02,6.5692100e+02, &
& 5.8601680e+02,5.4418120e+02,4.9261320e+02,4.4348890e+02,1.8638860e+03, &
& 1.7104153e+03,1.5029787e+03,6.9246950e+02,1.5160160e+03,1.4557245e+03, &
& 1.4191992e+03,1.3855997e+03,1.3558015e+03,1.0671944e+03,1.2015530e+03, &
& 1.1589243e+03,1.2228454e+03,1.1968607e+03,1.1734369e+03,1.1596457e+03, &
& 9.7768160e+02,9.6156840e+02,8.7807320e+02,7.4236820e+02,7.5524290e+02, &
& 6.8498790e+02,6.2777710e+02,5.2230280e+02,4.8830230e+02,5.0221500e+02, &
& 7.3000580e+02,7.1376740e+02,6.5632120e+02,6.2632590e+02,5.7857790e+02, &
& 5.3088320e+02,1.7602462e+03,1.6917254e+03,1.4957351e+03,1.3452345e+03]
 vdw_dftd3_c6(16201:16300)=[&
& 1.3357938e+03,1.2935031e+03,1.3314737e+03,1.2894340e+03,7.2427200e+01, &
& 2.3377270e+02,2.9840760e+02,2.2859970e+02,1.7275920e+02,1.2063570e+02, &
& 8.8243900e+01,6.1099300e+01,3.4193020e+02,5.2851140e+02,5.3561950e+02, &
& 4.3117410e+02,3.5382780e+02,2.9979380e+02,2.4565840e+02,4.7272210e+02, &
& 9.0129920e+02,4.7293970e+02,4.5672220e+02,4.4764490e+02,4.4350180e+02, &
& 4.0825230e+02,3.7802020e+02,3.6070350e+02,3.5264190e+02,3.4833050e+02, &
& 3.2702340e+02,5.3235460e+02,4.6817890e+02,4.2016770e+02,3.8416910e+02, &
& 3.3897840e+02,5.6438650e+02,1.0971255e+03,8.3943700e+02,6.2573340e+02, &
& 6.3197490e+02,5.8965940e+02,6.6095050e+02,5.1451210e+02,4.8157200e+02, &
& 4.4743080e+02,4.3285140e+02,4.2822300e+02,6.7704980e+02,6.1292050e+02, &
& 5.7443630e+02,5.4380490e+02,4.9721050e+02,6.6750050e+02,1.4228735e+03, &
& 1.0625153e+03,6.7166460e+02,6.5779210e+02,6.3700760e+02,6.4004320e+02, &
& 6.1687680e+02,6.4393890e+02,6.0497690e+02,6.1545730e+02,5.7428150e+02, &
& 5.5794580e+02,5.5432010e+02,5.8234520e+02,5.3579050e+02,7.2482070e+02, &
& 6.7316650e+02,6.1471110e+02,6.2176580e+02,5.4516120e+02,5.1320580e+02, &
& 4.9042130e+02,4.6907220e+02,4.5340690e+02,7.2771340e+02,6.6365960e+02, &
& 6.4384950e+02,6.2649490e+02,5.8554410e+02,7.5173280e+02,1.4153925e+03, &
& 7.9462310e+02,9.1695930e+02,8.2052080e+02,7.2537300e+02,6.9816440e+02, &
& 5.2391900e+01,3.3861500e+01,9.7720270e+02,5.1120280e+02,3.3002550e+02, &
& 2.1709250e+02,1.4929170e+02,1.1190540e+02,8.4162800e+01,6.4433300e+01]
 vdw_dftd3_c6(16301:16400)=[&
& 1.1617208e+03,8.2748750e+02,7.4501090e+02,5.6932030e+02,4.3398740e+02, &
& 3.5559410e+02,2.8720610e+02,2.3291960e+02,1.9509820e+03,1.5010394e+03, &
& 1.2234261e+03,1.1694190e+03,1.0640685e+03,8.3949690e+02,9.0788740e+02, &
& 7.1275770e+02,7.4344550e+02,7.7054070e+02,5.9237050e+02,5.9475930e+02, &
& 7.0853630e+02,6.0979120e+02,5.1011170e+02,4.5283130e+02,3.9222240e+02, &
& 3.3815470e+02,2.1787996e+03,1.7950359e+03,1.5372794e+03,1.3634850e+03, &
& 1.2334617e+03,9.4031770e+02,1.0541841e+03,7.9200470e+02,8.6390680e+02, &
& 7.9715370e+02,6.6898050e+02,6.9811240e+02,8.9002560e+02,8.0567160e+02, &
& 7.0498790e+02,6.4753750e+02,5.7956520e+02,5.1640770e+02,2.6607475e+03, &
& 2.3107445e+03,1.9831026e+03,8.3646730e+02,2.0333939e+03,1.9427158e+03, &
& 1.8916327e+03,1.8448932e+03,1.8033780e+03,1.3864919e+03,1.6271471e+03, &
& 1.5648029e+03,1.6162873e+03,1.5807484e+03,1.5482741e+03,1.5315185e+03, &
& 1.2721137e+03,1.2215252e+03,1.1011039e+03,9.2310550e+02,9.3407850e+02, &
& 8.3812920e+02,7.6168140e+02,6.2877650e+02,5.8595420e+02,5.9987110e+02, &
& 9.1189380e+02,8.7576580e+02,7.9106510e+02,7.4764610e+02,6.8300400e+02, &
& 6.2065460e+02,2.4608767e+03,2.2584201e+03,1.9536369e+03,1.7167267e+03, &
& 1.7259338e+03,1.6700620e+03,1.7432351e+03,1.6834151e+03,8.6005200e+01, &
& 2.9568750e+02,3.7496860e+02,2.7566970e+02,2.0403100e+02,1.3932300e+02, &
& 1.0015200e+02,6.7951400e+01,4.3651000e+02,6.7820670e+02,6.7169720e+02, &
& 5.2429270e+02,4.2107980e+02,3.5186230e+02,2.8433720e+02,5.9504560e+02]
 vdw_dftd3_c6(16401:16500)=[&
& 1.2074896e+03,5.7829290e+02,5.5886520e+02,5.4792340e+02,5.4465600e+02, &
& 4.9339110e+02,4.5504230e+02,4.3439050e+02,4.2520030e+02,4.2483420e+02, &
& 3.9059480e+02,6.6138520e+02,5.6767290e+02,5.0098750e+02,4.5312420e+02, &
& 3.9518880e+02,7.0695870e+02,1.4771099e+03,1.0809253e+03,7.6828830e+02, &
& 7.7551800e+02,7.2086910e+02,8.2604480e+02,6.1978620e+02,5.7963490e+02, &
& 5.3729800e+02,5.2079250e+02,5.0925590e+02,8.4221840e+02,7.4608910e+02, &
& 6.8936250e+02,6.4664990e+02,5.8507090e+02,8.2245460e+02,1.9411744e+03, &
& 1.3757929e+03,8.1046870e+02,7.9353070e+02,7.6776280e+02,7.7400610e+02, &
& 7.5318700e+02,7.7907900e+02,7.3004770e+02,7.4727740e+02,6.9119800e+02, &
& 6.7089470e+02,6.6702610e+02,7.0343090e+02,6.4373580e+02,9.0452750e+02, &
& 8.2941860e+02,7.4766610e+02,7.6481820e+02,6.5146000e+02,6.1088630e+02, &
& 5.8248590e+02,5.5757750e+02,5.3344460e+02,9.0297300e+02,8.0572640e+02, &
& 7.7305280e+02,7.4671410e+02,6.9111280e+02,9.2497250e+02,1.8985076e+03, &
& 9.5915900e+02,1.1307742e+03,1.0088729e+03,8.7461490e+02,8.3895690e+02, &
& 1.0512468e+03,1.2593100e+01,8.1628000e+00,1.9470360e+02,1.1396770e+02, &
& 7.6723000e+01,5.1562800e+01,3.5790200e+01,2.6895400e+01,2.0208300e+01, &
& 1.5417600e+01,2.3280600e+02,1.8116530e+02,1.6729290e+02,1.3189200e+02, &
& 1.0261760e+02,8.4869100e+01,6.9006900e+01,5.6151700e+01,3.7942250e+02, &
& 3.1651020e+02,2.6194430e+02,2.5305430e+02,2.3180650e+02,1.8215580e+02, &
& 1.9964260e+02,1.5628450e+02,1.6621660e+02,1.7120430e+02,1.3080120e+02]
 vdw_dftd3_c6(16501:16600)=[&
& 1.3479330e+02,1.5992730e+02,1.4114890e+02,1.2022240e+02,1.0763710e+02, &
& 9.3891100e+01,8.1330600e+01,4.2517040e+02,3.7664680e+02,3.3071900e+02, &
& 2.9747500e+02,2.7126060e+02,2.0909590e+02,2.3346060e+02,1.7751740e+02, &
& 1.9439840e+02,1.8021780e+02,1.4971980e+02,1.5869370e+02,1.9941110e+02, &
& 1.8468490e+02,1.6453480e+02,1.5249560e+02,1.3761920e+02,1.2340700e+02, &
& 5.1787260e+02,4.7936610e+02,4.2227540e+02,1.9417280e+02,4.2478830e+02, &
& 4.0812030e+02,3.9794790e+02,3.8859250e+02,3.8030110e+02,2.9932330e+02, &
& 3.3536650e+02,3.2359410e+02,3.4328500e+02,3.3605070e+02,3.2953550e+02, &
& 3.2568440e+02,2.7459690e+02,2.7080850e+02,2.4715510e+02,2.0829050e+02, &
& 2.1201830e+02,1.9203170e+02,1.7571480e+02,1.4552190e+02,1.3578630e+02, &
& 1.3983510e+02,2.0418140e+02,2.0002350e+02,1.8383790e+02,1.7525160e+02, &
& 1.6152400e+02,1.4775410e+02,4.9003150e+02,4.7451000e+02,4.2036790e+02, &
& 3.7811690e+02,3.7472790e+02,3.6278690e+02,3.7304730e+02,3.6135530e+02, &
& 2.0403900e+01,6.6200900e+01,8.4400000e+01,6.4430100e+01,4.8307200e+01, &
& 3.3327400e+01,2.4059200e+01,1.6318500e+01,9.6512500e+01,1.4938320e+02, &
& 1.5159310e+02,1.2183330e+02,9.9565600e+01,8.3918100e+01,6.8280600e+01, &
& 1.3247510e+02,2.5257780e+02,1.3298680e+02,1.2828620e+02,1.2567780e+02, &
& 1.2446450e+02,1.1464650e+02,1.0602860e+02,1.0108010e+02,9.8800400e+01, &
& 9.7534000e+01,9.1622700e+01,1.5021480e+02,1.3199340e+02,1.1815200e+02, &
& 1.0767130e+02,9.4551500e+01,1.5794790e+02,3.0694330e+02,2.3552320e+02]
 vdw_dftd3_c6(16601:16700)=[&
& 1.7565640e+02,1.7733570e+02,1.6520660e+02,1.8502440e+02,1.4388510e+02, &
& 1.3445760e+02,1.2471510e+02,1.2059750e+02,1.1946230e+02,1.9013130e+02, &
& 1.7217670e+02,1.6122340e+02,1.5238060e+02,1.3892730e+02,1.8713180e+02, &
& 3.9721940e+02,2.9790180e+02,1.8828490e+02,1.8438050e+02,1.7850850e+02, &
& 1.7939000e+02,1.7278660e+02,1.8061150e+02,1.6957170e+02,1.7251880e+02, &
& 1.6096160e+02,1.5637090e+02,1.5538670e+02,1.6348550e+02,1.5021190e+02, &
& 2.0349990e+02,1.8878180e+02,1.7212060e+02,1.7412660e+02,1.5226480e+02, &
& 1.4305840e+02,1.3650000e+02,1.3034490e+02,1.2597100e+02,2.0371920e+02, &
& 1.8586380e+02,1.8030690e+02,1.7530470e+02,1.6350870e+02,2.1081250e+02, &
& 3.9575600e+02,2.2289780e+02,2.5694260e+02,2.2939980e+02,2.0281970e+02, &
& 1.9512320e+02,2.3441020e+02,5.5136400e+01,1.2495200e+01,8.2586000e+00, &
& 1.8876110e+02,1.1041850e+02,7.4986500e+01,5.0909000e+01,3.5677800e+01, &
& 2.7024700e+01,2.0459200e+01,1.5712700e+01,2.2578430e+02,1.7539140e+02, &
& 1.6244730e+02,1.2874590e+02,1.0084320e+02,8.3884700e+01,6.8633100e+01, &
& 5.6192600e+01,3.7015350e+02,3.0690780e+02,2.5395340e+02,2.4566470e+02, &
& 2.2517830e+02,1.7740000e+02,1.9413780e+02,1.5238870e+02,1.6187200e+02, &
& 1.6656140e+02,1.2766860e+02,1.3152520e+02,1.5580800e+02,1.3799560e+02, &
& 1.1813540e+02,1.0621400e+02,9.3100300e+01,8.1051800e+01,4.1522010e+02, &
& 3.6565710e+02,3.2126520e+02,2.8928690e+02,2.6412850e+02,2.0430920e+02, &
& 2.2779590e+02,1.7387670e+02,1.9004450e+02,1.7633810e+02,1.4690170e+02]
 vdw_dftd3_c6(16701:16800)=[&
& 1.5547780e+02,1.9476490e+02,1.8069960e+02,1.6155700e+02,1.5017540e+02, &
& 1.3602390e+02,1.2246340e+02,5.0650540e+02,4.6597660e+02,4.1051230e+02, &
& 1.9074630e+02,4.1312090e+02,3.9675810e+02,3.8683630e+02,3.7770920e+02, &
& 3.6961950e+02,2.9164910e+02,3.2723330e+02,3.1588750e+02,3.3355970e+02, &
& 3.2649630e+02,3.2014370e+02,3.1635220e+02,2.6724090e+02,2.6368480e+02, &
& 2.4107990e+02,2.0385670e+02,2.0752370e+02,1.8834080e+02,1.7266610e+02, &
& 1.4353170e+02,1.3411890e+02,1.3806770e+02,2.0008980e+02,1.9608950e+02, &
& 1.8068820e+02,1.7261360e+02,1.5958520e+02,1.4647210e+02,4.7896760e+02, &
& 4.6147150e+02,4.0883210e+02,3.6846880e+02,3.6524790e+02,3.5364010e+02, &
& 3.6328030e+02,3.5190410e+02,2.0059700e+01,6.4255300e+01,8.2143000e+01, &
& 6.3168500e+01,4.7746200e+01,3.3267700e+01,2.4241600e+01,1.6653100e+01, &
& 9.3715900e+01,1.4497830e+02,1.4737660e+02,1.1908890e+02,9.7894700e+01, &
& 8.2949700e+01,6.7913200e+01,1.2959630e+02,2.4594980e+02,1.3017250e+02, &
& 1.2570170e+02,1.2315290e+02,1.2193400e+02,1.1246480e+02,1.0414370e+02, &
& 9.9334300e+01,9.7081600e+01,9.5700700e+01,9.0135100e+01,1.4648900e+02, &
& 1.2922270e+02,1.1615350e+02,1.0624910e+02,9.3737400e+01,1.5478270e+02, &
& 2.9929140e+02,2.3013840e+02,1.7218100e+02,1.7379460e+02,1.6219830e+02, &
& 1.8138850e+02,1.4162010e+02,1.3247760e+02,1.2302270e+02,1.1892360e+02, &
& 1.1788430e+02,1.8585760e+02,1.6872480e+02,1.5840370e+02,1.5008700e+02, &
& 1.3730200e+02,1.8365680e+02,3.8802680e+02,2.9127440e+02,1.8501940e+02]
 vdw_dftd3_c6(16801:16900)=[&
& 1.8119130e+02,1.7546970e+02,1.7626590e+02,1.6958240e+02,1.7736130e+02, &
& 1.6662020e+02,1.6942280e+02,1.5821700e+02,1.5373020e+02,1.5273000e+02, &
& 1.6046200e+02,1.4766360e+02,1.9919290e+02,1.8514520e+02,1.6918110e+02, &
& 1.7089180e+02,1.5017550e+02,1.4130850e+02,1.3497080e+02,1.2897340e+02, &
& 1.2481680e+02,1.9964430e+02,1.8252700e+02,1.7734000e+02,1.7270710e+02, &
& 1.6153370e+02,2.0689790e+02,3.8645800e+02,2.1894300e+02,2.5214850e+02, &
& 2.2571000e+02,1.9962770e+02,1.9215040e+02,2.3000030e+02,5.3991900e+01, &
& 5.3112800e+01,9.4203000e+00,6.5027000e+00,1.2377070e+02,7.6755200e+01, &
& 5.4085400e+01,3.7841900e+01,2.7170400e+01,2.0959700e+01,1.6131200e+01, &
& 1.2562900e+01,1.4876180e+02,1.2073860e+02,1.1385600e+02,9.2493100e+01, &
& 7.4078400e+01,6.2595300e+01,5.2020800e+01,4.3206400e+01,2.4263660e+02, &
& 2.0802230e+02,1.7346300e+02,1.6920640e+02,1.5583870e+02,1.2325910e+02, &
& 1.3529940e+02,1.0673260e+02,1.1409130e+02,1.1680720e+02,8.9951500e+01, &
& 9.3687000e+01,1.1027700e+02,9.9469800e+01,8.6689500e+01,7.8883800e+01, &
& 7.0035200e+01,6.1725400e+01,2.7337570e+02,2.4782850e+02,2.2088300e+02, &
& 2.0078170e+02,1.8454770e+02,1.4463550e+02,1.6045200e+02,1.2423890e+02, &
& 1.3547600e+02,1.2623620e+02,1.0535720e+02,1.1192520e+02,1.3828310e+02, &
& 1.3007660e+02,1.1803370e+02,1.1077470e+02,1.0142160e+02,9.2287600e+01, &
& 3.3381300e+02,3.1468370e+02,2.8115040e+02,1.3918110e+02,2.8046490e+02, &
& 2.6998650e+02,2.6339220e+02,2.5730290e+02,2.5190990e+02,2.0229780e+02]
 vdw_dftd3_c6(16901:17000)=[&
& 2.2209340e+02,2.1493990e+02,2.2808570e+02,2.2331660e+02,2.1907220e+02, &
& 2.1630440e+02,1.8487050e+02,1.8493620e+02,1.7072850e+02,1.4577740e+02, &
& 1.4883210e+02,1.3624100e+02,1.2580040e+02,1.0559200e+02,9.9045400e+01, &
& 1.0212660e+02,1.4307200e+02,1.4157460e+02,1.3211550e+02,1.2719520e+02, &
& 1.1875500e+02,1.1003140e+02,3.1957380e+02,3.1409480e+02,2.8185090e+02, &
& 2.5814870e+02,2.5426850e+02,2.4632970e+02,2.5069380e+02,2.4324780e+02, &
& 1.4762300e+01,4.4932000e+01,5.7880500e+01,4.5963200e+01,3.5521900e+01, &
& 2.5382700e+01,1.8903400e+01,1.3352900e+01,6.5273900e+01,1.0058170e+02, &
& 1.0379070e+02,8.5906800e+01,7.1991400e+01,6.1891600e+01,5.1472100e+01, &
& 9.2024000e+01,1.6741450e+02,9.3891100e+01,9.0783800e+01,8.8942900e+01, &
& 8.7888900e+01,8.1873100e+01,7.6113200e+01,7.2644300e+01,7.0944400e+01, &
& 6.9405800e+01,6.6283800e+01,1.0418810e+02,9.3589300e+01,8.5337900e+01, &
& 7.8902900e+01,7.0474400e+01,1.1048560e+02,2.0357060e+02,1.6111150e+02, &
& 1.2419200e+02,1.2536250e+02,1.1755240e+02,1.2970790e+02,1.0379560e+02, &
& 9.7296100e+01,9.0624900e+01,8.7495100e+01,8.7291500e+01,1.3265050e+02, &
& 1.2220300e+02,1.1597420e+02,1.1077110e+02,1.0234890e+02,1.3246620e+02, &
& 2.6261810e+02,2.0348290e+02,1.3513800e+02,1.3236910e+02,1.2830130e+02, &
& 1.2859630e+02,1.2294040e+02,1.2924610e+02,1.2168990e+02,1.2326190e+02, &
& 1.1576120e+02,1.1255880e+02,1.1174980e+02,1.1693130e+02,1.0814300e+02, &
& 1.4218370e+02,1.3342480e+02,1.2313620e+02,1.2340630e+02,1.1082750e+02]
 vdw_dftd3_c6(17001:17100)=[&
& 1.0472640e+02,1.0030120e+02,9.5922700e+01,9.3462800e+01,1.4324960e+02, &
& 1.3282830e+02,1.3004040e+02,1.2739760e+02,1.2019670e+02,1.4932790e+02, &
& 2.6405250e+02,1.5978300e+02,1.8182240e+02,1.6366940e+02,1.4625060e+02, &
& 1.4111910e+02,1.6450530e+02,3.9281400e+01,3.8983400e+01,2.9360200e+01, &
& 7.6610000e+00,5.4559000e+00,9.4006000e+01,5.9598300e+01,4.2834800e+01, &
& 3.0530500e+01,2.2279400e+01,1.7409300e+01,1.3561300e+01,1.0674000e+01, &
& 1.1330310e+02,9.3413000e+01,8.8840700e+01,7.3085800e+01,5.9285900e+01, &
& 5.0594600e+01,4.2481600e+01,3.5633400e+01,1.8521810e+02,1.6027500e+02, &
& 1.3403140e+02,1.3130150e+02,1.2121190e+02,9.6254700e+01,1.0560800e+02, &
& 8.3691600e+01,8.9534500e+01,9.1429500e+01,7.0763900e+01,7.3921000e+01, &
& 8.6619400e+01,7.8823500e+01,6.9377000e+01,6.3591900e+01,5.6918700e+01, &
& 5.0574600e+01,2.0923780e+02,1.9112680e+02,1.7135740e+02,1.5645400e+02, &
& 1.4431220e+02,1.1400040e+02,1.2608010e+02,9.8472800e+01,1.0710840e+02, &
& 1.0003700e+02,8.3787500e+01,8.8985800e+01,1.0908250e+02,1.0321130e+02, &
& 9.4365400e+01,8.9035900e+01,8.2037400e+01,7.5142700e+01,2.5574990e+02, &
& 2.4255280e+02,2.1789480e+02,1.1128270e+02,2.1678200e+02,2.0883570e+02, &
& 2.0377290e+02,1.9908850e+02,1.9493970e+02,1.5792210e+02,1.7208570e+02, &
& 1.6673560e+02,1.7670260e+02,1.7301200e+02,1.6974290e+02,1.6752470e+02, &
& 1.4402080e+02,1.4479850e+02,1.3433990e+02,1.1544060e+02,1.1798450e+02, &
& 1.0851360e+02,1.0061040e+02,8.5019700e+01,7.9962500e+01,8.2454000e+01]
 vdw_dftd3_c6(17101:17200)=[&
& 1.1348820e+02,1.1268290e+02,1.0579110e+02,1.0226460e+02,9.6007100e+01, &
& 8.9458900e+01,2.4600480e+02,2.4292720e+02,2.1910640e+02,2.0223940e+02, &
& 1.9881690e+02,1.9268400e+02,1.9532220e+02,1.8964690e+02,1.1821400e+01, &
& 3.5019400e+01,4.5316400e+01,3.6606300e+01,2.8693800e+01,2.0850100e+01, &
& 1.5763100e+01,1.1357100e+01,5.0867200e+01,7.8176000e+01,8.1202600e+01, &
& 6.8050400e+01,5.7663200e+01,5.0029800e+01,4.2036700e+01,7.2681800e+01, &
& 1.2958410e+02,7.4558700e+01,7.2182800e+01,7.0734700e+01,6.9847100e+01, &
& 6.5341500e+01,6.0892000e+01,5.8161100e+01,5.6786800e+01,5.5378900e+01, &
& 5.3207000e+01,8.2040200e+01,7.4362700e+01,6.8347700e+01,6.3607400e+01, &
& 5.7257800e+01,8.7544700e+01,1.5768360e+02,1.2634700e+02,9.8716500e+01, &
& 9.9662500e+01,9.3743400e+01,1.0285060e+02,8.3274400e+01,7.8197600e+01, &
& 7.2996500e+01,7.0450900e+01,7.0442000e+01,1.0489720e+02,9.7282600e+01, &
& 9.2826200e+01,8.9058200e+01,8.2771300e+01,1.0537260e+02,2.0323960e+02, &
& 1.5949680e+02,1.0811590e+02,1.0591500e+02,1.0271500e+02,1.0283980e+02, &
& 9.8062800e+01,1.0326480e+02,9.7359100e+01,9.8444500e+01,9.2693900e+01, &
& 9.0162700e+01,8.9477400e+01,9.3380800e+01,8.6620200e+01,1.1247170e+02, &
& 1.0607460e+02,9.8419900e+01,9.8272000e+01,8.9250800e+01,8.4579000e+01, &
& 8.1163500e+01,7.7711500e+01,7.5948200e+01,1.1375630e+02,1.0614180e+02, &
& 1.0427680e+02,1.0247400e+02,9.7157900e+01,1.1880190e+02,2.0507880e+02, &
& 1.2774120e+02,1.4466290e+02,1.3073320e+02,1.1732470e+02,1.1335810e+02]
 vdw_dftd3_c6(17201:17300)=[&
& 1.3059750e+02,3.1240700e+01,3.1222100e+01,2.3896500e+01,1.9676800e+01, &
& 5.3717000e+00,4.0179000e+00,5.8670300e+01,3.8702700e+01,2.8771300e+01, &
& 2.1142800e+01,1.5836400e+01,1.2627700e+01,1.0022200e+01,8.0175000e+00, &
& 7.1099900e+01,6.0298300e+01,5.8195100e+01,4.8906600e+01,4.0516700e+01, &
& 3.5137700e+01,2.9991800e+01,2.5551400e+01,1.1670690e+02,1.0271260e+02, &
& 8.6336500e+01,8.5221900e+01,7.9000300e+01,6.3186700e+01,6.9259800e+01, &
& 5.5335600e+01,5.9268800e+01,6.0257500e+01,4.7058500e+01,4.9389700e+01, &
& 5.7401600e+01,5.3011000e+01,4.7420200e+01,4.3981900e+01,3.9882100e+01, &
& 3.5896100e+01,1.3248790e+02,1.2269280e+02,1.1116060e+02,1.0228360e+02, &
& 9.4928100e+01,7.6023900e+01,8.3640900e+01,6.6302500e+01,7.1807100e+01, &
& 6.7335400e+01,5.6751100e+01,6.0233400e+01,7.2857100e+01,6.9604600e+01, &
& 6.4429100e+01,6.1315900e+01,5.7073800e+01,5.2826900e+01,1.6220010e+02, &
& 1.5554370e+02,1.4109330e+02,7.5991000e+01,1.3974720e+02,1.3480190e+02, &
& 1.3157850e+02,1.2858460e+02,1.2593290e+02,1.0360070e+02,1.1142770e+02, &
& 1.0818200e+02,1.1438200e+02,1.1199780e+02,1.0990310e+02,1.0838260e+02, &
& 9.4143100e+01,9.5450600e+01,8.9314200e+01,7.7589500e+01,7.9439800e+01, &
& 7.3642700e+01,6.8748100e+01,5.8749700e+01,5.5502000e+01,5.7232800e+01, &
& 7.6523500e+01,7.6401300e+01,7.2435200e+01,7.0476900e+01,6.6750400e+01, &
& 6.2757800e+01,1.5736950e+02,1.5673170e+02,1.4266360e+02,1.3347620e+02, &
& 1.3079990e+02,1.2685500e+02,1.2772950e+02,1.2416310e+02,8.0848000e+00]
 vdw_dftd3_c6(17301:17400)=[&
& 2.2893000e+01,2.9852300e+01,2.4818500e+01,1.9909000e+01,1.4860000e+01, &
& 1.1500900e+01,8.5397000e+00,3.3263600e+01,5.0866900e+01,5.3436500e+01, &
& 4.5727300e+01,3.9461700e+01,3.4751000e+01,2.9682200e+01,4.8638000e+01, &
& 8.3727900e+01,5.0337300e+01,4.8839600e+01,4.7881500e+01,4.7229700e+01, &
& 4.4488800e+01,4.1627000e+01,3.9813100e+01,3.8859700e+01,3.7708300e+01, &
& 3.6576400e+01,5.4596400e+01,5.0234300e+01,4.6775300e+01,4.3993100e+01, &
& 4.0100200e+01,5.8901500e+01,1.0201160e+02,8.3510800e+01,6.6749800e+01, &
& 6.7412800e+01,6.3740200e+01,6.9283800e+01,5.7192800e+01,5.3867000e+01, &
& 5.0470900e+01,4.8691200e+01,4.8849700e+01,7.0351100e+01,6.5963500e+01, &
& 6.3497200e+01,6.1358500e+01,5.7565500e+01,7.1326900e+01,1.3126330e+02, &
& 1.0532360e+02,7.3903200e+01,7.2415600e+01,7.0289800e+01,7.0251100e+01, &
& 6.6716600e+01,7.0433100e+01,6.6555900e+01,6.7106000e+01,6.3452700e+01, &
& 6.1756800e+01,6.1245400e+01,6.3639600e+01,5.9322400e+01,7.5441200e+01, &
& 7.1752400e+01,6.7168700e+01,6.6668200e+01,6.1667600e+01,5.8716200e+01, &
& 5.6528500e+01,5.4235200e+01,5.3255600e+01,7.6842400e+01,7.2435300e+01, &
& 7.1560300e+01,7.0670800e+01,6.7531900e+01,8.0432600e+01,1.3328520e+02, &
& 8.7210800e+01,9.7994500e+01,8.9142100e+01,8.0572400e+01,7.8019000e+01, &
& 8.8176800e+01,2.1128200e+01,2.1364800e+01,1.6785500e+01,1.4080700e+01, &
& 1.0370800e+01,2.1837000e+01,1.3849600e+01,3.7081900e+02,2.0867690e+02, &
& 1.3699320e+02,9.0330500e+01,6.1837200e+01,4.6034700e+01,3.4331400e+01]
 vdw_dftd3_c6(17401:17500)=[&
& 2.6053300e+01,4.4240190e+02,3.3424930e+02,3.0473370e+02,2.3613880e+02, &
& 1.8098430e+02,1.4821300e+02,1.1939110e+02,9.6381300e+01,7.2363460e+02, &
& 5.9087370e+02,4.8646530e+02,4.6745060e+02,4.2688150e+02,3.3501180e+02, &
& 3.6600240e+02,2.8599690e+02,3.0245180e+02,3.1258620e+02,2.3853700e+02, &
& 2.4356630e+02,2.8974250e+02,2.5239640e+02,2.1233920e+02,1.8862140e+02, &
& 1.6322210e+02,1.4036050e+02,8.0885370e+02,7.0339060e+02,6.1169610e+02, &
& 5.4679350e+02,4.9648070e+02,3.7976260e+02,4.2526560e+02,3.2066780e+02, &
& 3.5143020e+02,3.2495510e+02,2.7016330e+02,2.8521610e+02,3.6138350e+02, &
& 3.3128280e+02,2.9202460e+02,2.6885950e+02,2.4089430e+02,2.1454790e+02, &
& 9.8395840e+02,8.9740290e+02,7.8313020e+02,3.4513820e+02,7.9290630e+02, &
& 7.6073520e+02,7.4148810e+02,7.2382090e+02,7.0814950e+02,5.5111930e+02, &
& 6.2646370e+02,6.0329280e+02,6.3781980e+02,6.2424880e+02,6.1193920e+02, &
& 6.0508430e+02,5.0623650e+02,4.9422900e+02,4.4819920e+02,3.7559750e+02, &
& 3.8147930e+02,3.4361070e+02,3.1299940e+02,2.5789210e+02,2.4017810e+02, &
& 2.4690090e+02,3.6875940e+02,3.5855060e+02,3.2650440e+02,3.0951770e+02, &
& 2.8334810e+02,2.5758240e+02,9.2425730e+02,8.8381110e+02,7.7640740e+02, &
& 6.9112710e+02,6.8830150e+02,6.6622690e+02,6.8965180e+02,6.6729490e+02, &
& 3.5919000e+01,1.2078080e+02,1.5315750e+02,1.1441900e+02,8.4617800e+01, &
& 5.7539800e+01,4.1060500e+01,2.7484100e+01,1.7686830e+02,2.7415900e+02, &
& 2.7526090e+02,2.1754710e+02,1.7550060e+02,1.4658490e+02,1.1815670e+02]
 vdw_dftd3_c6(17501:17600)=[&
& 2.4017250e+02,4.7030590e+02,2.3815400e+02,2.2957540e+02,2.2498220e+02, &
& 2.2318550e+02,2.0404460e+02,1.8825250e+02,1.7945530e+02,1.7553430e+02, &
& 1.7436190e+02,1.6200320e+02,2.7124200e+02,2.3526700e+02,2.0852360e+02, &
& 1.8870250e+02,1.6443830e+02,2.8542190e+02,5.7194480e+02,4.3039400e+02, &
& 3.1464020e+02,3.1779230e+02,2.9518160e+02,3.3381660e+02,2.5522890e+02, &
& 2.3831870e+02,2.2073750e+02,2.1374770e+02,2.1054980e+02,3.4330980e+02, &
& 3.0754750e+02,2.8572510e+02,2.6854180e+02,2.4320970e+02,3.3524300e+02, &
& 7.4236730e+02,5.4507520e+02,3.3446520e+02,3.2749910e+02,3.1689690e+02, &
& 3.1894850e+02,3.0882190e+02,3.2130770e+02,3.0127990e+02,3.0734770e+02, &
& 2.8559620e+02,2.7731030e+02,2.7568110e+02,2.9074100e+02,2.6629260e+02, &
& 3.6710790e+02,3.3846150e+02,3.0662140e+02,3.1198680e+02,2.6878310e+02, &
& 2.5196650e+02,2.4010400e+02,2.2934010e+02,2.2047400e+02,3.6691360e+02, &
& 3.3131350e+02,3.1955040e+02,3.0934230e+02,2.8680210e+02,3.7742160e+02, &
& 7.3519780e+02,3.9606860e+02,4.6056730e+02,4.0974230e+02,3.5990110e+02, &
& 3.4571200e+02,4.2322520e+02,9.7810100e+01,9.5357500e+01,6.8298400e+01, &
& 5.3880300e+01,3.5966500e+01,1.7556160e+02,2.6352400e+01,1.6703100e+01, &
& 4.5636130e+02,2.5306490e+02,1.6559380e+02,1.0905410e+02,7.4611700e+01, &
& 5.5525000e+01,4.1391400e+01,3.1394400e+01,5.4393780e+02,4.0606870e+02, &
& 3.6938290e+02,2.8548250e+02,2.1853020e+02,1.7889260e+02,1.4406690e+02, &
& 1.1628170e+02,8.9462280e+02,7.2140890e+02,5.9270500e+02,5.6890770e+02]
 vdw_dftd3_c6(17601:17700)=[&
& 5.1913280e+02,4.0772880e+02,4.4462830e+02,3.4763940e+02,3.6672360e+02, &
& 3.7922800e+02,2.8968240e+02,2.9490170e+02,3.5111920e+02,3.0516430e+02, &
& 2.5642310e+02,2.2769360e+02,1.9697430e+02,1.6935470e+02,9.9974000e+02, &
& 8.5956780e+02,7.4518010e+02,6.6503860e+02,6.0335750e+02,4.6105860e+02, &
& 5.1647660e+02,3.8899560e+02,4.2598370e+02,3.9367760e+02,3.2774720e+02, &
& 3.4529970e+02,4.3824740e+02,4.0078940e+02,3.5278490e+02,3.2462780e+02, &
& 2.9073780e+02,2.5886870e+02,1.2173141e+03,1.0987485e+03,9.5554880e+02, &
& 4.1722080e+02,9.6973660e+02,9.2960420e+02,9.0588600e+02,8.8413220e+02, &
& 8.6483220e+02,6.7129000e+02,7.6813130e+02,7.3945380e+02,7.7813230e+02, &
& 7.6146230e+02,7.4632100e+02,7.3802330e+02,6.1647150e+02,6.0003080e+02, &
& 5.4344620e+02,4.5533130e+02,4.6211940e+02,4.1585260e+02,3.7855240e+02, &
& 3.1184500e+02,2.9037770e+02,2.9830060e+02,4.4739490e+02,4.3398370e+02, &
& 3.9457020e+02,3.7381510e+02,3.4202110e+02,3.1081620e+02,1.1397020e+03, &
& 1.0803973e+03,9.4598830e+02,8.3976740e+02,8.3781840e+02,8.1085380e+02, &
& 8.4069890e+02,8.1310950e+02,4.3366100e+01,1.4644260e+02,1.8573100e+02, &
& 1.3828380e+02,1.0220950e+02,6.9450100e+01,4.9531900e+01,3.3126500e+01, &
& 2.1467960e+02,3.3305050e+02,3.3353410e+02,2.6297700e+02,2.1192050e+02, &
& 1.7693650e+02,1.4258060e+02,2.9160280e+02,5.7547210e+02,2.8808340e+02, &
& 2.7780740e+02,2.7224960e+02,2.7016230e+02,2.4655880e+02,2.2743270e+02, &
& 2.1683930e+02,2.1211940e+02,2.1090190e+02,1.9559730e+02,3.2851000e+02]
 vdw_dftd3_c6(17701:17800)=[&
& 2.8437390e+02,2.5181270e+02,2.2779820e+02,1.9844730e+02,3.4649590e+02, &
& 7.0067460e+02,5.2422670e+02,3.8103090e+02,3.8476780e+02,3.5735760e+02, &
& 4.0520110e+02,3.0854590e+02,2.8812910e+02,2.6684840e+02,2.5840650e+02, &
& 2.5422360e+02,4.1600330e+02,3.7187790e+02,3.4513030e+02,3.2423490e+02, &
& 2.9353370e+02,4.0637240e+02,9.1183210e+02,6.6459040e+02,4.0429100e+02, &
& 3.9585770e+02,3.8302380e+02,3.8563610e+02,3.7369250e+02,3.8845580e+02, &
& 3.6415710e+02,3.7174820e+02,3.4511820e+02,3.3507830e+02,3.3312390e+02, &
& 3.5137350e+02,3.2173170e+02,4.4538150e+02,4.1012060e+02,3.7108910e+02, &
& 3.7797370e+02,3.2478730e+02,3.0440310e+02,2.9003910e+02,2.7707890e+02, &
& 2.6611650e+02,4.4475260e+02,4.0065540e+02,3.8605640e+02,3.7356510e+02, &
& 3.4618570e+02,4.5742940e+02,9.0069660e+02,4.7874300e+02,5.5821910e+02, &
& 4.9675560e+02,4.3513510e+02,4.1782680e+02,5.1393660e+02,1.1812330e+02, &
& 1.1524800e+02,8.2430300e+01,6.5016400e+01,4.3378800e+01,2.1219710e+02, &
& 2.5674970e+02,2.5702100e+01,1.6580600e+01,4.1243230e+02,2.3689910e+02, &
& 1.5801650e+02,1.0557130e+02,7.3007300e+01,5.4744300e+01,4.1072800e+01, &
& 3.1310700e+01,4.9265620e+02,3.7783600e+02,3.4711540e+02,2.7188440e+02, &
& 2.1048400e+02,1.7357980e+02,1.4078240e+02,1.1433480e+02,8.0549240e+02, &
& 6.6409770e+02,5.4821150e+02,5.2845930e+02,4.8345990e+02,3.7993850e+02, &
& 4.1561120e+02,3.2531170e+02,3.4494790e+02,3.5576410e+02,2.7189110e+02, &
& 2.7896960e+02,3.3128490e+02,2.9090920e+02,2.4673790e+02,2.2038060e+02]
 vdw_dftd3_c6(17801:17900)=[&
& 1.9179920e+02,1.6582300e+02,9.0177390e+02,7.9067650e+02,6.9120290e+02, &
& 6.2007920e+02,5.6449010e+02,4.3396780e+02,4.8500980e+02,3.6773060e+02, &
& 4.0262390e+02,3.7288980e+02,3.1011860e+02,3.2796220e+02,4.1337540e+02, &
& 3.8122300e+02,3.3831890e+02,3.1287500e+02,2.8172560e+02,2.5213370e+02, &
& 1.0981239e+03,1.0078503e+03,8.8388150e+02,3.9958530e+02,8.9190420e+02, &
& 8.5628170e+02,8.3476410e+02,8.1499180e+02,7.9746090e+02,6.2473210e+02, &
& 7.0500060e+02,6.7965440e+02,7.1903230e+02,7.0378810e+02,6.9001840e+02, &
& 6.8208270e+02,5.7324330e+02,5.6278660e+02,5.1235040e+02,4.3105180e+02, &
& 4.3832130e+02,3.9619890e+02,3.6196280e+02,2.9935640e+02,2.7918270e+02, &
& 2.8724760e+02,4.2296870e+02,4.1296020e+02,3.7820780e+02,3.5984120e+02, &
& 3.3092490e+02,3.0214620e+02,1.0353860e+03,9.9537320e+02,8.7825500e+02, &
& 7.8653170e+02,7.8131330e+02,7.5635270e+02,7.7996360e+02,7.5512290e+02, &
& 4.1822200e+01,1.3744580e+02,1.7493150e+02,1.3248310e+02,9.8927800e+01, &
& 6.7982000e+01,4.8939100e+01,3.3107500e+01,2.0080380e+02,3.1098970e+02, &
& 3.1418090e+02,2.5093760e+02,2.0419980e+02,1.7165370e+02,1.3931250e+02, &
& 2.7480560e+02,5.2989460e+02,2.7434930e+02,2.6463020e+02,2.5929250e+02, &
& 2.5696820e+02,2.3596760e+02,2.1806460e+02,2.0790840e+02,2.0327770e+02, &
& 2.0115730e+02,1.8816130e+02,3.1077040e+02,2.7174380e+02,2.4243590e+02, &
& 2.2046240e+02,1.9317380e+02,3.2730800e+02,6.4440170e+02,4.9021540e+02, &
& 3.6256720e+02,3.6609840e+02,3.4073580e+02,3.8317340e+02,2.9596310e+02]
 vdw_dftd3_c6(17901:18000)=[&
& 2.7654270e+02,2.5641330e+02,2.4807500e+02,2.4516140e+02,3.9356170e+02, &
& 3.5486870e+02,3.3134490e+02,3.1259390e+02,2.8440960e+02,3.8643270e+02, &
& 8.3548010e+02,6.2051600e+02,3.8738880e+02,3.7934290e+02,3.6719680e+02, &
& 3.6922530e+02,3.5636770e+02,3.7178190e+02,3.4890330e+02,3.5535340e+02, &
& 3.3101430e+02,3.2151310e+02,3.1953220e+02,3.3642390e+02,3.0879300e+02, &
& 4.2124260e+02,3.8987130e+02,3.5462710e+02,3.5955640e+02,3.1268360e+02, &
& 2.9359280e+02,2.8003950e+02,2.6748530e+02,2.5798760e+02,4.2147770e+02, &
& 3.8292490e+02,3.7065420e+02,3.5983420e+02,3.3497230e+02,4.3520780e+02, &
& 8.3000970e+02,4.5862090e+02,5.3065480e+02,4.7329240e+02,4.1725880e+02, &
& 4.0119370e+02,4.8568000e+02,1.1328200e+02,1.1084470e+02,8.0261900e+01, &
& 6.3715100e+01,4.2967900e+01,2.0177580e+02,2.4384070e+02,2.3312540e+02, &
& 2.2742900e+01,1.5024500e+01,3.3746620e+02,1.9995760e+02,1.3624020e+02, &
& 9.2608000e+01,6.4920900e+01,4.9172900e+01,3.7220500e+01,2.8580500e+01, &
& 4.0398020e+02,3.1708240e+02,2.9434030e+02,2.3387520e+02,1.8343220e+02, &
& 1.5264700e+02,1.2491620e+02,1.0227010e+02,6.5923940e+02,5.5248890e+02, &
& 4.5796080e+02,4.4347230e+02,4.0676720e+02,3.2027340e+02,3.5102240e+02, &
& 2.7541380e+02,2.9316610e+02,3.0149740e+02,2.3090910e+02,2.3850630e+02, &
& 2.8238710e+02,2.5064010e+02,2.1483630e+02,1.9324100e+02,1.6943130e+02, &
& 1.4751930e+02,7.3972690e+02,6.5778770e+02,5.7951050e+02,5.2257210e+02, &
& 4.7747720e+02,3.6967940e+02,4.1204130e+02,3.1482970e+02,3.4429870e+02]
 vdw_dftd3_c6(18001:18100)=[&
& 3.1960770e+02,2.6595030e+02,2.8194690e+02,3.5272470e+02,3.2794600e+02, &
& 2.9361680e+02,2.7309200e+02,2.4747320e+02,2.2286600e+02,9.0172930e+02, &
& 8.3699700e+02,7.3956740e+02,3.4646750e+02,7.4267020e+02,7.1379130e+02, &
& 6.9606280e+02,6.7974330e+02,6.6528190e+02,5.2618270e+02,5.8709720e+02, &
& 5.6687340e+02,6.0088550e+02,5.8822970e+02,5.7686430e+02,5.6998510e+02, &
& 4.8216270e+02,4.7707590e+02,4.3667570e+02,3.6935120e+02,3.7621900e+02, &
& 3.4171560e+02,3.1344980e+02,2.6059070e+02,2.4352110e+02,2.5084270e+02, &
& 3.6225560e+02,3.5571630e+02,3.2825370e+02,3.1378020e+02,2.9026290e+02, &
& 2.6650630e+02,8.5532140e+02,8.3008340e+02,7.3740950e+02,6.6621410e+02, &
& 6.5937380e+02,6.3846460e+02,6.5489730e+02,6.3460740e+02,3.6493600e+01, &
& 1.1639410e+02,1.4881940e+02,1.1479680e+02,8.6812800e+01,6.0508500e+01, &
& 4.4090400e+01,3.0289400e+01,1.6956980e+02,2.6216960e+02,2.6714040e+02, &
& 2.1636930e+02,1.7806220e+02,1.5093640e+02,1.2360110e+02,2.3449210e+02, &
& 4.4207020e+02,2.3629650e+02,2.2810870e+02,2.2347840e+02,2.2119860e+02, &
& 2.0433190e+02,1.8924970e+02,1.8048460e+02,1.7637500e+02,1.7370060e+02, &
& 1.6388840e+02,2.6564070e+02,2.3477850e+02,2.1123800e+02,1.9329960e+02, &
& 1.7058580e+02,2.8011830e+02,5.3743520e+02,4.1535780e+02,3.1230500e+02, &
& 3.1527630e+02,2.9424360e+02,3.2833170e+02,2.5720590e+02,2.4057510e+02, &
& 2.2341250e+02,2.1594190e+02,2.1429840e+02,3.3682850e+02,3.0637120e+02, &
& 2.8792660e+02,2.7294170e+02,2.4979920e+02,3.3288760e+02,6.9526390e+02]
 vdw_dftd3_c6(18101:18200)=[&
& 5.2525150e+02,3.3608240e+02,3.2913430e+02,3.1875390e+02,3.2010570e+02, &
& 3.0771680e+02,3.2211770e+02,3.0266100e+02,3.0757360e+02,2.8745570e+02, &
& 2.7932320e+02,2.7749490e+02,2.9150550e+02,2.6832640e+02,3.6072920e+02, &
& 3.3564790e+02,3.0701360e+02,3.0982750e+02,2.7286500e+02,2.5678780e+02, &
& 2.4528060e+02,2.3433320e+02,2.2697090e+02,3.6172460e+02,3.3138450e+02, &
& 3.2226070e+02,3.1398590e+02,2.9381590e+02,3.7507170e+02,6.9405520e+02, &
& 3.9772110e+02,4.5702730e+02,4.0899300e+02,3.6253380e+02,3.4905070e+02, &
& 4.1613190e+02,9.8185900e+01,9.6532400e+01,7.0926900e+01,5.6798100e+01, &
& 3.8857800e+01,1.7322270e+02,2.0920610e+02,2.0144650e+02,1.7555340e+02, &
& 1.9472900e+01,1.3209000e+01,2.6554640e+02,1.6268750e+02,1.1343240e+02, &
& 7.8572900e+01,5.5913600e+01,4.2824500e+01,3.2736500e+01,2.5343500e+01, &
& 3.1871200e+02,2.5643150e+02,2.4071650e+02,1.9423140e+02,1.5449350e+02, &
& 1.2984130e+02,1.0729640e+02,8.8626800e+01,5.1938020e+02,4.4287350e+02, &
& 3.6871310e+02,3.5885540e+02,3.3009380e+02,2.6056470e+02,2.8605270e+02, &
& 2.2513570e+02,2.4051750e+02,2.4658640e+02,1.8941220e+02,1.9693310e+02, &
& 2.3234090e+02,2.0856850e+02,1.8080070e+02,1.6386810e+02,1.4483870e+02, &
& 1.2707910e+02,5.8437840e+02,5.2738380e+02,4.6852690e+02,4.2487480e+02, &
& 3.8978890e+02,3.0421090e+02,3.3802960e+02,2.6053350e+02,2.8446820e+02, &
& 2.6473450e+02,2.2055850e+02,2.3431540e+02,2.9072550e+02,2.7258810e+02, &
& 2.4633500e+02,2.3051370e+02,2.1031760e+02,1.9068220e+02,7.1322290e+02]
 vdw_dftd3_c6(18201:18300)=[&
& 6.6991060e+02,5.9671860e+02,2.9045550e+02,5.9619860e+02,5.7368400e+02, &
& 5.5961080e+02,5.4662920e+02,5.3513180e+02,4.2773380e+02,4.7158580e+02, &
& 4.5610620e+02,4.8421010e+02,4.7407530e+02,4.6503260e+02,4.5926340e+02, &
& 3.9129390e+02,3.9032880e+02,3.5937970e+02,3.0582610e+02,3.1204680e+02, &
& 2.8491830e+02,2.6249830e+02,2.1953350e+02,2.0562600e+02,2.1201590e+02, &
& 2.9991980e+02,2.9620520e+02,2.7549690e+02,2.6464810e+02,2.4633950e+02, &
& 2.2753380e+02,6.8102410e+02,6.6741950e+02,5.9723200e+02,5.4474490e+02, &
& 5.3715360e+02,5.2027520e+02,5.3064210e+02,5.1469140e+02,3.0774700e+01, &
& 9.5053100e+01,1.2215340e+02,9.6110000e+01,7.3700600e+01,5.2175300e+01, &
& 3.8527600e+01,2.6916300e+01,1.3811660e+02,2.1310360e+02,2.1912850e+02, &
& 1.8016220e+02,1.5007480e+02,1.2837640e+02,1.0616060e+02,1.9334120e+02, &
& 3.5561610e+02,1.9666010e+02,1.9002900e+02,1.8615760e+02,1.8402800e+02, &
& 1.7103060e+02,1.5879330e+02,1.5149830e+02,1.4797530e+02,1.4502500e+02, &
& 1.3803960e+02,2.1921940e+02,1.9595580e+02,1.7790880e+02,1.6390830e+02, &
& 1.4577240e+02,2.3172650e+02,4.3227830e+02,3.3980820e+02,2.5999910e+02, &
& 2.6242950e+02,2.4567190e+02,2.7194340e+02,2.1620660e+02,2.0248030e+02, &
& 1.8837520e+02,1.8191000e+02,1.8125820e+02,2.7850870e+02,2.5563360e+02, &
& 2.4188440e+02,2.3047020e+02,2.1226280e+02,2.7721210e+02,5.5799180e+02, &
& 4.2930150e+02,2.8192380e+02,2.7612630e+02,2.6756270e+02,2.6833820e+02, &
& 2.5690680e+02,2.6982370e+02,2.5386320e+02,2.5739270e+02,2.4138000e+02]
 vdw_dftd3_c6(18301:18400)=[&
& 2.3465510e+02,2.3302060e+02,2.4417040e+02,2.2545330e+02,2.9846420e+02, &
& 2.7931410e+02,2.5702350e+02,2.5811720e+02,2.3037700e+02,2.1735740e+02, &
& 2.0795360e+02,1.9875510e+02,1.9332900e+02,3.0009690e+02,2.7730190e+02, &
& 2.7095850e+02,2.6500160e+02,2.4934960e+02,3.1247270e+02,5.5992810e+02, &
& 3.3346380e+02,3.8050360e+02,3.4179510e+02,3.0467440e+02,2.9376970e+02, &
& 3.4471170e+02,8.2197300e+01,8.1269300e+01,6.0669000e+01,4.9062900e+01, &
& 3.4104200e+01,1.4355230e+02,1.7327790e+02,1.6813020e+02,1.4787500e+02, &
& 1.2581090e+02,3.0678300e+01,1.9724700e+01,5.2531960e+02,2.9170210e+02, &
& 1.9141350e+02,1.2668360e+02,8.7190600e+01,6.5252500e+01,4.8933100e+01, &
& 3.7325800e+01,6.2655260e+02,4.6815270e+02,4.2605680e+02,3.2981710e+02, &
& 2.5315190e+02,2.0785310e+02,1.6800650e+02,1.3616380e+02,1.0305394e+03, &
& 8.3223790e+02,6.8387860e+02,6.5674740e+02,5.9945210e+02,4.7138320e+02, &
& 5.1365160e+02,4.0218720e+02,4.2395230e+02,4.3824820e+02,3.3535920e+02, &
& 3.4119230e+02,4.0564880e+02,3.5293360e+02,2.9714260e+02,2.6437900e+02, &
& 2.2931470e+02,1.9776450e+02,1.1520197e+03,9.9191430e+02,8.6032290e+02, &
& 7.6820000e+02,6.9731480e+02,5.3379780e+02,5.9749150e+02,4.5095970e+02, &
& 4.9339260e+02,4.5620630e+02,3.8042830e+02,4.0045350e+02,5.0726050e+02, &
& 4.6411930e+02,4.0901730e+02,3.7680930e+02,3.3804940e+02,3.0162730e+02, &
& 1.4021290e+03,1.2678490e+03,1.1031556e+03,4.8397150e+02,1.1194842e+03, &
& 1.0733952e+03,1.0460388e+03,1.0209351e+03,9.9865850e+02,7.7606950e+02]
 vdw_dftd3_c6(18401:18500)=[&
& 8.8704210e+02,8.5381990e+02,8.9865150e+02,8.7938870e+02,8.6189750e+02, &
& 8.5225470e+02,7.1226080e+02,6.9344710e+02,6.2855660e+02,5.2745270e+02, &
& 5.3533140e+02,4.8222940e+02,4.3941270e+02,3.6278720e+02,3.3812750e+02, &
& 3.4721000e+02,5.1871070e+02,5.0321870e+02,4.5790780e+02,4.3414870e+02, &
& 3.9775460e+02,3.6206750e+02,1.3139837e+03,1.2471871e+03,1.0925678e+03, &
& 9.7093620e+02,9.6877080e+02,9.3774220e+02,9.7203520e+02,9.4020330e+02, &
& 5.0236800e+01,1.6890090e+02,2.1431710e+02,1.6007340e+02,1.1882550e+02, &
& 8.1252500e+01,5.8342300e+01,3.9404800e+01,2.4782230e+02,3.8413080e+02, &
& 3.8485550e+02,3.0397250e+02,2.4556540e+02,2.0560060e+02,1.6628780e+02, &
& 3.3777950e+02,6.6406200e+02,3.3367800e+02,3.2184410e+02,3.1546860e+02, &
& 3.1305380e+02,2.8580680e+02,2.6379300e+02,2.5158410e+02,2.4611560e+02, &
& 2.4467160e+02,2.2701180e+02,3.7966500e+02,3.2905540e+02,2.9186010e+02, &
& 2.6450720e+02,2.3101070e+02,4.0165720e+02,8.0869090e+02,6.0584530e+02, &
& 4.4149510e+02,4.4598950e+02,4.1448110e+02,4.6957520e+02,3.5837320e+02, &
& 3.3488270e+02,3.1039090e+02,3.0060940e+02,2.9569410e+02,4.8167290e+02, &
& 4.3088530e+02,4.0023370e+02,3.7637060e+02,3.4127320e+02,4.7105090e+02, &
& 1.0519461e+03,7.6794120e+02,4.6904980e+02,4.5929190e+02,4.4446290e+02, &
& 4.4738260e+02,4.3352520e+02,4.5052800e+02,4.2251560e+02,4.3117490e+02, &
& 4.0047540e+02,3.8885070e+02,3.8653820e+02,4.0739760e+02,3.7332130e+02, &
& 5.1554770e+02,4.7526240e+02,4.3058420e+02,4.3832070e+02,3.7753700e+02]
 vdw_dftd3_c6(18501:18600)=[&
& 3.5420050e+02,3.3773330e+02,3.2285240e+02,3.1020900e+02,5.1564400e+02, &
& 4.6487620e+02,4.4812230e+02,4.3387110e+02,4.0255590e+02,5.3018890e+02, &
& 1.0398007e+03,5.5526920e+02,6.4689110e+02,5.7609270e+02,5.0534480e+02, &
& 4.8542450e+02,5.9584710e+02,1.3659980e+02,1.3359950e+02,9.6055300e+01, &
& 7.6136500e+01,5.1242500e+01,2.4509830e+02,2.9655720e+02,2.8198140e+02, &
& 2.4244540e+02,2.0138950e+02,3.4333430e+02,2.9861500e+01,1.9430400e+01, &
& 4.8195860e+02,2.7500610e+02,1.8324130e+02,1.2261150e+02,8.5046400e+01, &
& 6.3983900e+01,4.8189200e+01,3.6880000e+01,5.7574000e+02,4.3920300e+02, &
& 4.0293570e+02,3.1526550e+02,2.4412340e+02,2.0154660e+02,1.6374820e+02, &
& 1.3328650e+02,9.4339620e+02,7.7413680e+02,6.3848690e+02,6.1524940e+02, &
& 5.6271800e+02,4.4267360e+02,4.8359930e+02,3.7893660e+02,4.0112950e+02, &
& 4.1378410e+02,3.1671020e+02,3.2428910e+02,3.8481720e+02,3.3757610e+02, &
& 2.8630450e+02,2.5587020e+02,2.2292560e+02,1.9302240e+02,1.0561444e+03, &
& 9.2214300e+02,8.0504650e+02,7.2178370e+02,6.5695460e+02,5.0523460e+02, &
& 5.6455600e+02,4.2826290e+02,4.6853580e+02,4.3394840e+02,3.6150570e+02, &
& 3.8172980e+02,4.8100660e+02,4.4306050e+02,3.9301150e+02,3.6347770e+02, &
& 3.2743400e+02,2.9327410e+02,1.2859957e+03,1.1762602e+03,1.0300975e+03, &
& 4.6446000e+02,1.0408154e+03,9.9899710e+02,9.7382280e+02,9.5069090e+02, &
& 9.3017500e+02,7.2811220e+02,8.2368220e+02,7.9385410e+02,8.3837100e+02, &
& 8.2054230e+02,8.0442380e+02,7.9518560e+02,6.6789240e+02,6.5462000e+02]
 vdw_dftd3_c6(18601:18700)=[&
& 5.9577000e+02,5.0153070e+02,5.0981600e+02,4.6083870e+02,4.2109420e+02, &
& 3.4866710e+02,3.2533860e+02,3.3453100e+02,4.9268870e+02,4.8044970e+02, &
& 4.3974760e+02,4.1833640e+02,3.8479250e+02,3.5151420e+02,1.2112384e+03, &
& 1.1609837e+03,1.0231422e+03,9.1552140e+02,9.1036450e+02,8.8134280e+02, &
& 9.0964690e+02,8.8054330e+02,4.8492900e+01,1.5957150e+02,2.0306780e+02, &
& 1.5370500e+02,1.1497100e+02,7.9258600e+01,5.7280500e+01,3.8989900e+01, &
& 2.3346740e+02,3.6145020e+02,3.6466680e+02,2.9100370e+02,2.3687400e+02, &
& 1.9933220e+02,1.6205420e+02,3.2000430e+02,6.1816760e+02,3.1880040e+02, &
& 3.0759130e+02,3.0144200e+02,2.9881900e+02,2.7417030e+02,2.5341680e+02, &
& 2.4168330e+02,2.3632960e+02,2.3403740e+02,2.1866030e+02,3.6091960e+02, &
& 3.1533670e+02,2.8133310e+02,2.5597460e+02,2.2452090e+02,3.8121510e+02, &
& 7.5216410e+02,5.7086180e+02,4.2153880e+02,4.2572990e+02,3.9634700e+02, &
& 4.4614920e+02,3.4429480e+02,3.2185830e+02,2.9856920e+02,2.8894080e+02, &
& 2.8528140e+02,4.5781840e+02,4.1239390e+02,3.8490810e+02,3.6314460e+02, &
& 3.3053650e+02,4.4947740e+02,9.7596650e+02,7.2280680e+02,4.5030760e+02, &
& 4.4096740e+02,4.2686830e+02,4.2925270e+02,4.1458930e+02,4.3214410e+02, &
& 4.0560400e+02,4.1317950e+02,3.8477440e+02,3.7372510e+02,3.7141000e+02, &
& 3.9092160e+02,3.5888950e+02,4.8998500e+02,4.5345540e+02,4.1248260e+02, &
& 4.1838190e+02,3.6375220e+02,3.4171650e+02,3.2608130e+02,3.1164840e+02, &
& 3.0046490e+02,4.9070860e+02,4.4539670e+02,4.3091620e+02,4.1829040e+02]
 vdw_dftd3_c6(18701:18800)=[&
& 3.8946080e+02,5.0612520e+02,9.6871930e+02,5.3300150e+02,6.1730540e+02, &
& 5.5082530e+02,4.8540710e+02,4.6674110e+02,5.6587940e+02,1.3129800e+02, &
& 1.2867160e+02,9.3360300e+01,7.4329700e+01,5.0392000e+01,2.3400680e+02, &
& 2.8286280e+02,2.7035480e+02,2.3372820e+02,1.9526710e+02,3.2764580e+02, &
& 3.1398460e+02,3.0259300e+01,1.9720500e+01,4.8620980e+02,2.7793710e+02, &
& 1.8540740e+02,1.2418490e+02,8.6210900e+01,6.4903700e+01,4.8913100e+01, &
& 3.7455400e+01,5.8090670e+02,4.4375570e+02,4.0733310e+02,3.1895500e+02, &
& 2.4716030e+02,2.0416200e+02,1.6596320e+02,1.3515860e+02,9.5162680e+02, &
& 7.8176790e+02,6.4493880e+02,6.2162280e+02,5.6863160e+02,4.4737630e+02, &
& 4.8878880e+02,3.8306060e+02,4.0558090e+02,4.1831130e+02,3.2021980e+02, &
& 3.2800010e+02,3.8913250e+02,3.4156270e+02,2.8985600e+02,2.5914890e+02, &
& 2.2588180e+02,1.9566660e+02,1.0654841e+03,9.3121720e+02,8.1333020e+02, &
& 7.2942430e+02,6.6404520e+02,5.1089670e+02,5.7079220e+02,4.3319140e+02, &
& 4.7389730e+02,4.3897520e+02,3.6571240e+02,3.8622410e+02,4.8644970e+02, &
& 4.4827700e+02,3.9783330e+02,3.6805570e+02,3.3168000e+02,2.9718710e+02, &
& 1.2973700e+03,1.1876789e+03,1.0405570e+03,4.7013410e+02,1.0511001e+03, &
& 1.0089483e+03,9.8354270e+02,9.6019540e+02,9.3948900e+02,7.3579720e+02, &
& 8.3177260e+02,8.0171120e+02,8.4685680e+02,8.2885550e+02,8.1258600e+02, &
& 8.0323470e+02,6.7488900e+02,6.6176650e+02,6.0245920e+02,5.0731550e+02, &
& 5.1574640e+02,4.6633110e+02,4.2621440e+02,3.5302080e+02,3.2944450e+02]
 vdw_dftd3_c6(18801:18900)=[&
& 3.3877410e+02,4.9837090e+02,4.8615110e+02,4.4515370e+02,4.2359010e+02, &
& 3.8975590e+02,3.5616530e+02,1.2224279e+03,1.1725376e+03,1.0337520e+03, &
& 9.2548980e+02,9.2008400e+02,8.9076990e+02,9.1911090e+02,8.8975190e+02, &
& 4.9098300e+01,1.6130130e+02,2.0531340e+02,1.5556610e+02,1.1644710e+02, &
& 8.0346400e+01,5.8112700e+01,3.9601500e+01,2.3597180e+02,3.6527360e+02, &
& 3.6870140e+02,2.9444790e+02,2.3982940e+02,2.0191810e+02,1.6424590e+02, &
& 3.2361900e+02,6.2430360e+02,3.2257050e+02,3.1124080e+02,3.0502010e+02, &
& 3.0234740e+02,2.7750050e+02,2.5652910e+02,2.4465710e+02,2.3923230e+02, &
& 2.3685320e+02,2.2139430e+02,3.6502390e+02,3.1910990e+02,2.8483450e+02, &
& 2.5925390e+02,2.2749370e+02,3.8558300e+02,7.5959780e+02,5.7702860e+02, &
& 4.2651570e+02,4.3076130e+02,4.0109190e+02,4.5128660e+02,3.4854900e+02, &
& 3.2585820e+02,3.0231120e+02,2.9255140e+02,2.8891110e+02,4.6307770e+02, &
& 4.1733250e+02,3.8965780e+02,3.6772520e+02,3.3481970e+02,4.5477990e+02, &
& 9.8542930e+02,7.3055840e+02,4.5582340e+02,4.4637170e+02,4.3211200e+02, &
& 4.3449190e+02,4.1956220e+02,4.3740200e+02,4.1057030e+02,4.1818290e+02, &
& 3.8950840e+02,3.7833210e+02,3.7597960e+02,3.9567790e+02,3.6331600e+02, &
& 4.9559850e+02,4.5879570e+02,4.1747930e+02,4.2334010e+02,3.6833170e+02, &
& 3.4606910e+02,3.3026610e+02,3.1565750e+02,3.0440290e+02,4.9642620e+02, &
& 4.5079950e+02,4.3625560e+02,4.2355850e+02,3.9448330e+02,5.1210760e+02, &
& 9.7841480e+02,5.3951560e+02,6.2459130e+02,5.5742270e+02,4.9140260e+02]
 vdw_dftd3_c6(18901:19000)=[&
& 4.7254630e+02,5.7240170e+02,1.3288810e+02,1.3026620e+02,9.4601300e+01, &
& 7.5360900e+01,5.1141000e+01,2.3672460e+02,2.8613170e+02,2.7358870e+02, &
& 2.3663740e+02,1.9780370e+02,3.3148510e+02,3.1775890e+02,3.2158900e+02, &
& 2.7688000e+01,1.8112700e+01,4.4829230e+02,2.5437670e+02,1.6952210e+02, &
& 1.1361290e+02,7.8976100e+01,5.9544500e+01,4.4949300e+01,3.4478900e+01, &
& 5.3544830e+02,4.0656420e+02,3.7280190e+02,2.9163180e+02,2.2598490e+02, &
& 1.8676650e+02,1.5194160e+02,1.2386350e+02,8.7974450e+02,7.1816300e+02, &
& 5.9186950e+02,5.7022820e+02,5.2146020e+02,4.1056610e+02,4.4806980e+02, &
& 3.5139390e+02,3.7151490e+02,3.8325570e+02,2.9368160e+02,3.0031300e+02, &
& 3.5624690e+02,3.1241300e+02,2.6507780e+02,2.3705770e+02,2.0672770e+02, &
& 1.7919650e+02,9.8495040e+02,8.5590050e+02,7.4643350e+02,6.6896670e+02, &
& 6.0884420e+02,4.6840590e+02,5.2332230e+02,3.9715580e+02,4.3421760e+02, &
& 4.0217500e+02,3.3544360e+02,3.5382270e+02,4.4577660e+02,4.1035180e+02, &
& 3.6401890e+02,3.3677190e+02,3.0354590e+02,2.7207950e+02,1.1997581e+03, &
& 1.0926689e+03,9.5571690e+02,4.3033410e+02,9.6666190e+02,9.2752540e+02, &
& 9.0407530e+02,8.8253130e+02,8.6342100e+02,6.7551500e+02,7.6607760e+02, &
& 7.3824890e+02,7.7789910e+02,7.6130400e+02,7.4629430e+02,7.3772690e+02, &
& 6.1945150e+02,6.0644790e+02,5.5185300e+02,4.6484330e+02,4.7240710e+02, &
& 4.2706310e+02,3.9030550e+02,3.2344940e+02,3.0191160e+02,3.1033530e+02, &
& 4.5700230e+02,4.4528620e+02,4.0750360e+02,3.8771530e+02,3.5676420e+02]
 vdw_dftd3_c6(19001:19100)=[&
& 3.2609510e+02,1.1286750e+03,1.0779350e+03,9.4892310e+02,8.4862350e+02, &
& 8.4446350e+02,8.1754620e+02,8.4421690e+02,8.1709710e+02,4.4884200e+01, &
& 1.4764100e+02,1.8795310e+02,1.4227280e+02,1.0657110e+02,7.3627500e+01, &
& 5.3340000e+01,3.6451300e+01,2.1618670e+02,3.3467830e+02,3.3741280e+02, &
& 2.6924130e+02,2.1930330e+02,1.8472410e+02,1.5037570e+02,2.9674310e+02, &
& 5.7423510e+02,2.9523740e+02,2.8494970e+02,2.7927550e+02,2.7688190e+02, &
& 2.5393700e+02,2.3476650e+02,2.2394370e+02,2.1899430e+02,2.1692030e+02, &
& 2.0259800e+02,3.3410780e+02,2.9187130e+02,2.6049700e+02,2.3715920e+02, &
& 2.0820220e+02,3.5359290e+02,6.9913020e+02,5.2968240e+02,3.9060560e+02, &
& 3.9448540e+02,3.6737500e+02,4.1382300e+02,3.1915840e+02,2.9845770e+02, &
& 2.7694720e+02,2.6803530e+02,2.6453140e+02,4.2422720e+02,3.8198180e+02, &
& 3.5653490e+02,3.3646710e+02,3.0641230e+02,4.1667550e+02,9.0813400e+02, &
& 6.7092080e+02,4.1723950e+02,4.0858800e+02,3.9554120e+02,3.9776470e+02, &
& 3.8425760e+02,4.0037930e+02,3.7581950e+02,3.8288660e+02,3.5650840e+02, &
& 3.4627180e+02,3.4411550e+02,3.6209870e+02,3.3249970e+02,4.5419310e+02, &
& 4.2031920e+02,3.8236280e+02,3.8788720e+02,3.3726140e+02,3.1693610e+02, &
& 3.0251590e+02,2.8922160e+02,2.7881840e+02,4.5500920e+02,4.1279290e+02, &
& 3.9932210e+02,3.8766700e+02,3.6107620e+02,4.6915130e+02,9.0055920e+02, &
& 4.9379900e+02,5.7234950e+02,5.1099770e+02,4.5000790e+02,4.3270840e+02, &
& 5.2512640e+02,1.2146010e+02,1.1917710e+02,8.6606000e+01,6.9080900e+01]
 vdw_dftd3_c6(19101:19200)=[&
& 4.6987500e+01,2.1643950e+02,2.6172830e+02,2.5015910e+02,2.1641010e+02, &
& 1.8097100e+02,3.0339700e+02,2.9073090e+02,2.9424170e+02,2.6933490e+02, &
& 2.1313400e+01,1.4405100e+01,3.1428070e+02,1.8490790e+02,1.2637360e+02, &
& 8.6546400e+01,6.1247100e+01,4.6813800e+01,3.5781100e+01,2.7737200e+01, &
& 3.7649060e+02,2.9367770e+02,2.7250150e+02,2.1680010e+02,1.7067670e+02, &
& 1.4267950e+02,1.1742580e+02,9.6759300e+01,6.1705660e+02,5.1377680e+02, &
& 4.2547450e+02,4.1213870e+02,3.7806600e+02,2.9848940e+02,3.2635580e+02, &
& 2.5682780e+02,2.7263010e+02,2.8029770e+02,2.1550230e+02,2.2196560e+02, &
& 2.6215300e+02,2.3279920e+02,2.0004860e+02,1.8047050e+02,1.5887040e+02, &
& 1.3897880e+02,6.9273590e+02,6.1235490e+02,5.3890290e+02,4.8594470e+02, &
& 4.4423280e+02,3.4481050e+02,3.8391860e+02,2.9422580e+02,3.2113840e+02, &
& 2.9830140e+02,2.4916130e+02,2.6345110e+02,3.2878360e+02,3.0551080e+02, &
& 2.7385520e+02,2.5510770e+02,2.3174840e+02,2.0936220e+02,8.4446350e+02, &
& 7.8008380e+02,6.8834330e+02,3.2346600e+02,6.9246670e+02,6.6535710e+02, &
& 6.4877270e+02,6.3350050e+02,6.1995990e+02,4.9059980e+02,5.4884660e+02, &
& 5.2980690e+02,5.5970160e+02,5.4785030e+02,5.3720110e+02,5.3075810e+02, &
& 4.4908170e+02,4.4357620e+02,4.0628410e+02,3.4453870e+02,3.5083400e+02, &
& 3.1905950e+02,2.9307050e+02,2.4455970e+02,2.2890260e+02,2.3555590e+02, &
& 3.3874020e+02,3.3222990e+02,3.0674920e+02,2.9347990e+02,2.7198150e+02, &
& 2.5033250e+02,8.0027390e+02,7.7336630e+02,6.8635790e+02,6.2032700e+02]
 vdw_dftd3_c6(19201:19300)=[&
& 6.1475560e+02,5.9538630e+02,6.1108390e+02,5.9208300e+02,3.3942700e+01, &
& 1.0775270e+02,1.3789190e+02,1.0672440e+02,8.1233500e+01,5.7176600e+01, &
& 4.2104600e+01,2.9392300e+01,1.5738300e+02,2.4300490e+02,2.4743720e+02, &
& 2.0073440e+02,1.6576280e+02,1.4110920e+02,1.1621000e+02,2.1890440e+02, &
& 4.1206100e+02,2.2010130e+02,2.1264500e+02,2.0840940e+02,2.0634260e+02, &
& 1.9055280e+02,1.7666070e+02,1.6859690e+02,1.6478470e+02,1.6235450e+02, &
& 1.5312950e+02,2.4667960e+02,2.1822070e+02,1.9676310e+02,1.8053650e+02, &
& 1.5993620e+02,2.6178690e+02,5.0151140e+02,3.8709820e+02,2.9122970e+02, &
& 2.9411860e+02,2.7484220e+02,3.0675850e+02,2.4066050e+02,2.2538880e+02, &
& 2.0959760e+02,2.0266840e+02,2.0093810e+02,3.1400260e+02,2.8561070e+02, &
& 2.6862910e+02,2.5497950e+02,2.3389400e+02,3.1070160e+02,6.4956240e+02, &
& 4.8969530e+02,3.1384690e+02,3.0738260e+02,2.9775220e+02,2.9895830e+02, &
& 2.8752360e+02,3.0066880e+02,2.8267060e+02,2.8721540e+02,2.6848670e+02, &
& 2.6090960e+02,2.5915780e+02,2.7191140e+02,2.5057300e+02,3.3624450e+02, &
& 3.1321870e+02,2.8691560e+02,2.8946620e+02,2.5557530e+02,2.4090600e+02, &
& 2.3039700e+02,2.2040320e+02,2.1352200e+02,3.3805130e+02,3.0969410e+02, &
& 3.0120530e+02,2.9366450e+02,2.7525520e+02,3.4998930e+02,6.4793370e+02, &
& 3.7121200e+02,4.2672840e+02,3.8254780e+02,3.3923950e+02,3.2676380e+02, &
& 3.8936500e+02,9.1098500e+01,8.9961000e+01,6.6613100e+01,5.3772600e+01, &
& 3.7302400e+01,1.6055540e+02,1.9398150e+02,1.8700360e+02,1.6347330e+02]
 vdw_dftd3_c6(19301:19400)=[&
& 1.3829440e+02,2.2568810e+02,2.1764940e+02,2.2041700e+02,2.0186410e+02, &
& 1.5334350e+02,1.8469400e+01,1.2698500e+01,2.5656890e+02,1.5476870e+02, &
& 1.0748660e+02,7.4551100e+01,5.3282800e+01,4.1023700e+01,3.1558200e+01, &
& 2.4595700e+01,3.0795570e+02,2.4475170e+02,2.2891250e+02,1.8409450e+02, &
& 1.4631750e+02,1.2311700e+02,1.0197460e+02,8.4510700e+01,5.0386210e+02, &
& 4.2523820e+02,3.5332830e+02,3.4349160e+02,3.1574920e+02,2.4968190e+02, &
& 2.7339370e+02,2.1557380e+02,2.2950770e+02,2.3544990e+02,1.8134820e+02, &
& 1.8772140e+02,2.2112170e+02,1.9793940e+02,1.7140770e+02,1.5541540e+02, &
& 1.3754060e+02,1.2092190e+02,5.6671690e+02,5.0680360e+02,4.4877680e+02, &
& 4.0632670e+02,3.7252150e+02,2.9074940e+02,3.2305520e+02,2.4907370e+02, &
& 2.7161880e+02,2.5275950e+02,2.1123900e+02,2.2376440e+02,2.7763500e+02, &
& 2.5955840e+02,2.3417500e+02,2.1904270e+02,1.9989240e+02,1.8138370e+02, &
& 6.9130860e+02,6.4466000e+02,5.7226670e+02,2.7639550e+02,5.7357140e+02, &
& 5.5161300e+02,5.3799770e+02,5.2544060e+02,5.1431120e+02,4.1008630e+02, &
& 4.5463510e+02,4.3940520e+02,4.6497300e+02,4.5518120e+02,4.4642190e+02, &
& 4.4091820e+02,3.7499750e+02,3.7259650e+02,3.4269950e+02,2.9180770e+02, &
& 2.9752890e+02,2.7157970e+02,2.5022110e+02,2.0963300e+02,1.9652040e+02, &
& 2.0240200e+02,2.8686700e+02,2.8256650e+02,2.6234870e+02,2.5184810e+02, &
& 2.3438100e+02,2.1658230e+02,6.5835450e+02,6.4123120e+02,5.7224570e+02, &
& 5.2079400e+02,5.1467310e+02,4.9856630e+02,5.0962770e+02,4.9413580e+02]
 vdw_dftd3_c6(19401:19500)=[&
& 2.9120800e+01,9.0423100e+01,1.1610320e+02,9.1109100e+01,6.9988300e+01, &
& 4.9769800e+01,3.6969200e+01,2.6092600e+01,1.3182570e+02,2.0321730e+02, &
& 2.0829020e+02,1.7075150e+02,1.4216740e+02,1.2175410e+02,1.0091430e+02, &
& 1.8481700e+02,3.4176740e+02,1.8713640e+02,1.8091050e+02,1.7729930e+02, &
& 1.7538670e+02,1.6268440e+02,1.5107680e+02,1.4421590e+02,1.4090800e+02, &
& 1.3835720e+02,1.3131000e+02,2.0851100e+02,1.8592460e+02,1.6867800e+02, &
& 1.5546520e+02,1.3842960e+02,2.2151020e+02,4.1583400e+02,3.2497510e+02, &
& 2.4759810e+02,2.5002660e+02,2.3413150e+02,2.5980510e+02,2.0600200e+02, &
& 1.9308940e+02,1.7978240e+02,1.7373810e+02,1.7276980e+02,2.6575750e+02, &
& 2.4329840e+02,2.2991580e+02,2.1898580e+02,2.0172290e+02,2.6411800e+02, &
& 5.3758040e+02,4.1076200e+02,2.6829020e+02,2.6278540e+02,2.5464690e+02, &
& 2.5543700e+02,2.4496120e+02,2.5677460e+02,2.4162910e+02,2.4510920e+02, &
& 2.2968660e+02,2.2327460e+02,2.2171170e+02,2.3223100e+02,2.1445700e+02, &
& 2.8461080e+02,2.6619500e+02,2.4486880e+02,2.4620160e+02,2.1942180e+02, &
& 2.0718460e+02,1.9836510e+02,1.8980990e+02,1.8444130e+02,2.8672590e+02, &
& 2.6430990e+02,2.5793290e+02,2.5212510e+02,2.3719910e+02,2.9760870e+02, &
& 5.3830700e+02,3.1722510e+02,3.6280170e+02,3.2606070e+02,2.9035010e+02, &
& 2.7996200e+02,3.2981320e+02,7.7782300e+01,7.7081200e+01,5.7689300e+01, &
& 4.6868100e+01,3.2849500e+01,1.3616190e+02,1.6440450e+02,1.5932860e+02, &
& 1.4014430e+02,1.1934860e+02,1.9163220e+02,1.8555440e+02,1.8798380e+02]
 vdw_dftd3_c6(19501:19600)=[&
& 1.7219800e+02,1.3182880e+02,1.1384630e+02,1.8097700e+01,1.2427700e+01, &
& 2.5324320e+02,1.5229860e+02,1.0555810e+02,7.3107900e+01,5.2202800e+01, &
& 4.0170900e+01,3.0892500e+01,2.4074100e+01,3.0391240e+02,2.4099620e+02, &
& 2.2516630e+02,1.8083460e+02,1.4355850e+02,1.2070600e+02,9.9911900e+01, &
& 8.2759100e+01,4.9731050e+02,4.1906350e+02,3.4806420e+02,3.3822830e+02, &
& 3.1083750e+02,2.4576290e+02,2.6904940e+02,2.1211090e+02,2.2573460e+02, &
& 2.3164220e+02,1.7839240e+02,1.8454030e+02,2.1740400e+02,1.9441570e+02, &
& 1.6819670e+02,1.5241280e+02,1.3480510e+02,1.1845780e+02,5.5921770e+02, &
& 4.9943750e+02,4.4192580e+02,3.9992950e+02,3.6653290e+02,2.8589770e+02, &
& 3.1774320e+02,2.4481840e+02,2.6700480e+02,2.4842030e+02,2.0761810e+02, &
& 2.1987500e+02,2.7297690e+02,2.5501000e+02,2.2988770e+02,2.1492360e+02, &
& 1.9602880e+02,1.7779050e+02,6.8205790e+02,6.3538110e+02,5.6362450e+02, &
& 2.7135450e+02,5.6520430e+02,5.4351170e+02,5.3008260e+02,5.1769890e+02, &
& 5.0672260e+02,4.0367390e+02,4.4799700e+02,4.3292450e+02,4.5804090e+02, &
& 4.4838960e+02,4.3975060e+02,4.3434720e+02,3.6918400e+02,3.6653030e+02, &
& 3.3695440e+02,2.8678360e+02,2.9236350e+02,2.6675590e+02,2.4569670e+02, &
& 2.0576580e+02,1.9287140e+02,1.9862370e+02,2.8196900e+02,2.7759650e+02, &
& 2.5755980e+02,2.4714790e+02,2.2989230e+02,2.1233880e+02,6.4916640e+02, &
& 6.3174380e+02,5.6344000e+02,5.1237500e+02,5.0653490e+02,4.9067580e+02, &
& 5.0182620e+02,4.8653240e+02,2.8566500e+01,8.8953800e+01,1.1416130e+02]
 vdw_dftd3_c6(19601:19700)=[&
& 8.9435400e+01,6.8629300e+01,4.8755800e+01,3.6191900e+01,2.5531400e+01, &
& 1.2973320e+02,2.0000410e+02,2.0482820e+02,1.6769170e+02,1.3948020e+02, &
& 1.1937190e+02,9.8874700e+01,1.8170400e+02,3.3670380e+02,1.8382570e+02, &
& 1.7770070e+02,1.7415940e+02,1.7230360e+02,1.5974020e+02,1.4831710e+02, &
& 1.4158150e+02,1.3834240e+02,1.3590190e+02,1.2887600e+02,2.0495230e+02, &
& 1.8256910e+02,1.6550840e+02,1.5246290e+02,1.3568060e+02,2.1771950e+02, &
& 4.0967730e+02,3.1968600e+02,2.4320860e+02,2.4560370e+02,2.2993990e+02, &
& 2.5533300e+02,2.0221300e+02,1.8952850e+02,1.7645130e+02,1.7054030e+02, &
& 1.6952460e+02,2.6122680e+02,2.3895870e+02,2.2568120e+02,2.1486070e+02, &
& 1.9782510e+02,2.5941930e+02,5.2971550e+02,4.0410260e+02,2.6338510e+02, &
& 2.5797950e+02,2.4997990e+02,2.5078300e+02,2.4059220e+02,2.5210750e+02, &
& 2.3721640e+02,2.4067880e+02,2.2546970e+02,2.1916750e+02,2.1764010e+02, &
& 2.2800720e+02,2.1050650e+02,2.7971240e+02,2.6149250e+02,2.4043090e+02, &
& 2.4184390e+02,2.1530790e+02,2.0326950e+02,1.9460130e+02,1.8621520e+02, &
& 1.8088400e+02,2.8177560e+02,2.5955420e+02,2.5318480e+02,2.4740350e+02, &
& 2.3265410e+02,2.9230010e+02,5.3018710e+02,3.1143140e+02,3.5639700e+02, &
& 3.2021770e+02,2.8502300e+02,2.7479730e+02,3.2416540e+02,7.6349400e+01, &
& 7.5632700e+01,5.6542900e+01,4.5914000e+01,3.2158400e+01,1.3378530e+02, &
& 1.6153570e+02,1.5644370e+02,1.3750430e+02,1.1701210e+02,1.8826960e+02, &
& 1.8220620e+02,1.8458560e+02,1.6909000e+02,1.2934440e+02,1.1164910e+02]
 vdw_dftd3_c6(19701:19800)=[&
& 1.0950410e+02,2.0774100e+01,1.3880400e+01,3.1984480e+02,1.8489380e+02, &
& 1.2490870e+02,8.4755200e+01,5.9562800e+01,4.5306600e+01,3.4493500e+01, &
& 2.6662700e+01,3.8268210e+02,2.9460910e+02,2.7182260e+02,2.1457260e+02, &
& 1.6773560e+02,1.3954300e+02,1.1430910e+02,9.3808400e+01,6.2812380e+02, &
& 5.1787690e+02,4.2788660e+02,4.1344790e+02,3.7873270e+02,2.9871690e+02, &
& 3.2626030e+02,2.5642380e+02,2.7162880e+02,2.7970520e+02,2.1480930e+02, &
& 2.2045250e+02,2.6075480e+02,2.3021330e+02,1.9670460e+02,1.7678440e+02, &
& 1.5501690e+02,1.3511470e+02,7.0427300e+02,6.1724930e+02,5.4087060e+02, &
& 4.8633250e+02,4.4370070e+02,3.4307100e+02,3.8256460e+02,2.9196350e+02, &
& 3.1887320e+02,2.9583660e+02,2.4704870e+02,2.6087010e+02,3.2688520e+02, &
& 3.0241310e+02,2.6980100e+02,2.5056260e+02,2.2685100e+02,2.0426630e+02, &
& 8.5816140e+02,7.8713930e+02,6.9163680e+02,3.1880260e+02,6.9775630e+02, &
& 6.6998350e+02,6.5317050e+02,6.3770390e+02,6.2398650e+02,4.9120470e+02, &
& 5.5314370e+02,5.3353430e+02,5.6278640e+02,5.5082490e+02,5.4004150e+02, &
& 5.3369230e+02,4.4999220e+02,4.4251450e+02,4.0411740e+02,3.4172070e+02, &
& 3.4765120e+02,3.1534640e+02,2.8904310e+02,2.4054490e+02,2.2491750e+02, &
& 2.3132500e+02,3.3611600e+02,3.2863270e+02,3.0220800e+02,2.8841500e+02, &
& 2.6645950e+02,2.4453080e+02,8.1034850e+02,7.7851930e+02,6.8835160e+02, &
& 6.1915580e+02,6.1484900e+02,5.9539070e+02,6.1287940e+02,5.9353010e+02, &
& 3.3330700e+01,1.0755820e+02,1.3729980e+02,1.0520190e+02,7.9527300e+01]
 vdw_dftd3_c6(19801:19900)=[&
& 5.5568500e+01,4.0679900e+01,2.8212200e+01,1.5735780e+02,2.4318090e+02, &
& 2.4645390e+02,1.9841740e+02,1.6285410e+02,1.3801590e+02,1.1313180e+02, &
& 2.1759670e+02,4.1483380e+02,2.1769090e+02,2.1024640e+02,2.0607570e+02, &
& 2.0417500e+02,1.8795640e+02,1.7405360e+02,1.6609110e+02,1.6238310e+02, &
& 1.6040150e+02,1.5059460e+02,2.4499140e+02,2.1547520e+02,1.9340210e+02, &
& 1.7685580e+02,1.5608590e+02,2.5980390e+02,5.0499050e+02,3.8637760e+02, &
& 2.8802430e+02,2.9089880e+02,2.7145590e+02,3.0427490e+02,2.3690460e+02, &
& 2.2176150e+02,2.0606490e+02,1.9936290e+02,1.9723130e+02,3.1168130e+02, &
& 2.8217500e+02,2.6447480e+02,2.5038960e+02,2.2896540e+02,3.0722970e+02, &
& 6.5499420e+02,4.8905800e+02,3.0922990e+02,3.0284470e+02,2.9328050e+02, &
& 2.9467450e+02,2.8401110e+02,2.9645530e+02,2.7853190e+02,2.8335200e+02, &
& 2.6440130e+02,2.5688240e+02,2.5521000e+02,2.6809220e+02,2.4667760e+02, &
& 3.3363020e+02,3.0987820e+02,2.8300050e+02,2.8624380e+02,2.5103410e+02, &
& 2.3634850e+02,2.2587990e+02,2.1607200e+02,2.0886990e+02,3.3505330e+02, &
& 3.0558290e+02,2.9647100e+02,2.8849230e+02,2.6966300e+02,3.4599510e+02, &
& 6.5149430e+02,3.6582470e+02,4.2216210e+02,3.7782790e+02,3.3401640e+02, &
& 3.2150040e+02,3.8630110e+02,8.9789100e+01,8.8440500e+01,6.4985600e+01, &
& 5.2232200e+01,3.5989200e+01,1.5909800e+02,1.9227550e+02,1.8462850e+02, &
& 1.6065780e+02,1.3524560e+02,2.2342730e+02,2.1483990e+02,2.1751780e+02, &
& 1.9920380e+02,1.5048630e+02,1.2896280e+02,1.2658850e+02,1.4806570e+02]
 vdw_dftd3_c6(19901:20000)=[&
& 1.9464800e+01,1.3097900e+01,2.9104880e+02,1.7059360e+02,1.1609230e+02, &
& 7.9205600e+01,5.5895400e+01,4.2646200e+01,3.2555300e+01,2.5221000e+01, &
& 3.4857980e+02,2.7122230e+02,2.5118770e+02,1.9928550e+02,1.5644320e+02, &
& 1.3051020e+02,1.0719760e+02,8.8182900e+01,5.7086590e+02,4.7482250e+02, &
& 3.9304870e+02,3.8043770e+02,3.4885100e+02,2.7525510e+02,3.0096400e+02, &
& 2.3668420e+02,2.5118370e+02,2.5838880e+02,1.9852020e+02,2.0431190e+02, &
& 2.4136360e+02,2.1391340e+02,1.8341100e+02,1.6519610e+02,1.4518280e+02, &
& 1.2680940e+02,6.4057820e+02,5.6577900e+02,4.9739960e+02,4.4816000e+02, &
& 4.0942870e+02,3.1734720e+02,3.5355460e+02,2.7055020e+02,2.9544510e+02, &
& 2.7433660e+02,2.2905370e+02,2.4218240e+02,3.0263410e+02,2.8084550e+02, &
& 2.5131640e+02,2.3382360e+02,2.1211680e+02,1.9136210e+02,7.8055200e+02, &
& 7.2070440e+02,6.3535610e+02,2.9682360e+02,6.3962140e+02,6.1452410e+02, &
& 5.9919690e+02,5.8508560e+02,5.7257280e+02,4.5239680e+02,5.0669850e+02, &
& 4.8901870e+02,5.1684540e+02,5.0590420e+02,4.9606180e+02,4.9015300e+02, &
& 4.1429680e+02,4.0871690e+02,3.7400200e+02,3.1678470e+02,3.2251470e+02, &
& 2.9305060e+02,2.6898340e+02,2.2421600e+02,2.0978810e+02,2.1587920e+02, &
& 3.1149490e+02,3.0528260e+02,2.8148900e+02,2.6905740e+02,2.4904140e+02, &
& 2.2894100e+02,7.3916470e+02,7.1405300e+02,6.3328860e+02,5.7161590e+02, &
& 5.6672940e+02,5.4885680e+02,5.6381700e+02,5.4623160e+02,3.1098100e+01, &
& 9.9345400e+01,1.2698120e+02,9.7926800e+01,7.4314900e+01,5.2153200e+01]
 vdw_dftd3_c6(20001:20100)=[&
& 3.8319400e+01,2.6700000e+01,1.4518540e+02,2.2418120e+02,2.2795650e+02, &
& 1.8441750e+02,1.5191520e+02,1.2907630e+02,1.0609050e+02,2.0137990e+02, &
& 3.8042010e+02,2.0223920e+02,1.9535160e+02,1.9147150e+02,1.8962030e+02, &
& 1.7495290e+02,1.6212930e+02,1.5471980e+02,1.5124090e+02,1.4914790e+02, &
& 1.4045520e+02,2.2700010e+02,2.0040030e+02,1.8036950e+02,1.6525840e+02, &
& 1.4616810e+02,2.4066290e+02,4.6289020e+02,3.5647370e+02,2.6749730e+02, &
& 2.7016880e+02,2.5232810e+02,2.8197070e+02,2.2072030e+02,2.0667290e+02, &
& 1.9214160e+02,1.8584300e+02,1.8414330e+02,2.8888240e+02,2.6236960e+02, &
& 2.4645550e+02,2.3368760e+02,2.1408530e+02,2.8525530e+02,5.9951160e+02, &
& 4.5093480e+02,2.8796490e+02,2.8202970e+02,2.7316700e+02,2.7433460e+02, &
& 2.6403730e+02,2.7594550e+02,2.5937430e+02,2.6363770e+02,2.4631030e+02, &
& 2.3934130e+02,2.3775270e+02,2.4957470e+02,2.2985230e+02,3.0915100e+02, &
& 2.8770370e+02,2.6327960e+02,2.6585220e+02,2.3419810e+02,2.2066120e+02, &
& 2.1098510e+02,2.0183220e+02,1.9539650e+02,3.1077960e+02,2.8433510e+02, &
& 2.7630910e+02,2.6918960e+02,2.5203630e+02,3.2130350e+02,5.9764750e+02, &
& 3.4062710e+02,3.9198180e+02,3.5114350e+02,3.1117290e+02,2.9966590e+02, &
& 3.5795210e+02,8.3603000e+01,8.2447400e+01,6.0867800e+01,4.9051100e+01, &
& 3.3942700e+01,1.4769120e+02,1.7839840e+02,1.7172690e+02,1.4983280e+02, &
& 1.2649470e+02,2.0744400e+02,1.9985020e+02,2.0237460e+02,1.8532650e+02, &
& 1.4047480e+02,1.2062490e+02,1.1838000e+02,1.3802230e+02,1.2878200e+02]
 vdw_dftd3_c6(20101:20200)=[&
& 2.6956200e+01,1.7439200e+01,4.4148880e+02,2.5018890e+02,1.6615250e+02, &
& 1.1083840e+02,7.6664100e+01,5.7542500e+01,4.3238500e+01,3.3022800e+01, &
& 5.2706710e+02,3.9992630e+02,3.6633710e+02,2.8597610e+02,2.2097630e+02, &
& 1.8213800e+02,1.4771690e+02,1.2002340e+02,8.6510660e+02,7.0620410e+02, &
& 5.8191500e+02,5.6029450e+02,5.1220750e+02,4.0283090e+02,4.3988290e+02, &
& 3.4453520e+02,3.6443400e+02,3.7611120e+02,2.8776680e+02,2.9431290e+02, &
& 3.4953890e+02,3.0609340e+02,2.5916500e+02,2.3133530e+02,2.0127060e+02, &
& 1.7402300e+02,9.6816820e+02,8.4136100e+02,7.3335550e+02,6.5685920e+02, &
& 5.9747450e+02,4.5889060e+02,5.1304100e+02,3.8860430e+02,4.2520690e+02, &
& 3.9363700e+02,3.2788520e+02,3.4605680e+02,4.3674070e+02,4.0172510e+02, &
& 3.5584740e+02,3.2880760e+02,2.9588170e+02,2.6471260e+02,1.1792492e+03, &
& 1.0739458e+03,9.3892860e+02,4.2057330e+02,9.4969780e+02,9.1118510e+02, &
& 8.8814170e+02,8.6697870e+02,8.4820840e+02,6.6277010e+02,7.5204060e+02, &
& 7.2468160e+02,7.6414750e+02,7.4786110e+02,7.3312340e+02,7.2475960e+02, &
& 6.0808820e+02,5.9509760e+02,5.4106220e+02,4.5507210e+02,4.6243470e+02, &
& 4.1762900e+02,3.8132090e+02,3.1540110e+02,2.9416770e+02,3.0243400e+02, &
& 4.4706730e+02,4.3547920e+02,3.9809290e+02,3.7843240e+02,3.4775130e+02, &
& 3.1736560e+02,1.1088359e+03,1.0590662e+03,9.3187680e+02,8.3242240e+02, &
& 8.2834850e+02,8.0186390e+02,8.2836820e+02,8.0170370e+02,4.3888100e+01, &
& 1.4510430e+02,1.8457440e+02,1.3925460e+02,1.0391800e+02,7.1423400e+01]
 vdw_dftd3_c6(20201:20300)=[&
& 5.1470500e+01,3.4900800e+01,2.1236180e+02,3.2896970e+02,3.3140550e+02, &
& 2.6387400e+02,2.1439030e+02,1.8013610e+02,1.4618730e+02,2.9057390e+02, &
& 5.6411470e+02,2.8899240e+02,2.7882930e+02,2.7324400e+02,2.7091680e+02, &
& 2.4831530e+02,2.2943180e+02,2.1879640e+02,2.1396210e+02,2.1202750e+02, &
& 1.9784590e+02,3.2767720e+02,2.8580490e+02,2.5463590e+02,2.3143120e+02, &
& 2.0272250e+02,3.4599210e+02,6.8659850e+02,5.1944370e+02,3.8219520e+02, &
& 3.8593850e+02,3.5915800e+02,4.0491750e+02,3.1159910e+02,2.9122610e+02, &
& 2.7005910e+02,2.6136540e+02,2.5790070e+02,4.1546410e+02,3.7370470e+02, &
& 3.4844250e+02,3.2849050e+02,2.9869480e+02,4.0757240e+02,8.9181060e+02, &
& 6.5796510e+02,4.0771430e+02,3.9924480e+02,3.8644530e+02,3.8870550e+02, &
& 3.7562490e+02,3.9136320e+02,3.6723070e+02,3.7425920e+02,3.4830790e+02, &
& 3.3828140e+02,3.3621030e+02,3.5402460e+02,3.2484810e+02,4.4480230e+02, &
& 4.1120500e+02,3.7363290e+02,3.7928200e+02,3.2899190e+02,3.0890630e+02, &
& 2.9467460e+02,2.8159150e+02,2.7130180e+02,4.4509120e+02,4.0339300e+02, &
& 3.8999300e+02,3.7835750e+02,3.5197820e+02,4.5891490e+02,8.8404110e+02, &
& 4.8263570e+02,5.5982510e+02,4.9934950e+02,4.3935480e+02,4.2233120e+02, &
& 5.1359610e+02,1.1896190e+02,1.1648300e+02,8.4269100e+01,6.6955000e+01, &
& 4.5233600e+01,2.1230440e+02,2.5671210e+02,2.4505850e+02,2.1155710e+02, &
& 1.7644990e+02,2.9710840e+02,2.8448700e+02,2.8787970e+02,2.6338000e+02, &
& 1.9677040e+02,1.6756290e+02,1.6455530e+02,1.9438680e+02,1.8072890e+02]
 vdw_dftd3_c6(20301:20400)=[&
& 2.5787270e+02,2.8635900e+01,1.8618600e+01,4.5411250e+02,2.6160340e+02, &
& 1.7509380e+02,1.1741980e+02,8.1501900e+01,6.1309900e+01,4.6146800e+01, &
& 3.5284200e+01,5.4266570e+02,4.1703070e+02,3.8363050e+02,3.0113800e+02, &
& 2.3370440e+02,1.9313150e+02,1.5700310e+02,1.2781130e+02,8.8770750e+02, &
& 7.3264330e+02,6.0503430e+02,5.8362370e+02,5.3412560e+02,4.2006420e+02, &
& 4.5943140e+02,3.5991720e+02,3.8165880e+02,3.9345300e+02,3.0098740e+02, &
& 3.0895170e+02,3.6657420e+02,3.2239740e+02,2.7396550e+02,2.4506900e+02, &
& 2.1366510e+02,1.8507510e+02,9.9420460e+02,8.7244650e+02,7.6334100e+02, &
& 6.8527160e+02,6.2420610e+02,4.8055520e+02,5.3677950e+02,4.0763390e+02, &
& 4.4608760e+02,4.1332440e+02,3.4400650e+02,3.6375360e+02,4.5781190e+02, &
& 4.2263220e+02,3.7560470e+02,3.4772860e+02,3.1352800e+02,2.8100460e+02, &
& 1.2108497e+03,1.1120376e+03,9.7600660e+02,4.4363880e+02,9.8453560e+02, &
& 9.4530650e+02,9.2157530e+02,8.9976220e+02,8.8042150e+02,6.9068230e+02, &
& 7.7859640e+02,7.5073230e+02,7.9395920e+02,7.7712650e+02,7.6193210e+02, &
& 7.5311720e+02,6.3353080e+02,6.2243820e+02,5.6714200e+02,4.7771970e+02, &
& 4.8586480e+02,4.3956860e+02,4.0191180e+02,3.3286630e+02,3.1061480e+02, &
& 3.1958080e+02,4.6894260e+02,4.5811250e+02,4.2004290e+02,3.9996790e+02, &
& 3.6825120e+02,3.3664170e+02,1.1423947e+03,1.0988136e+03,9.7025760e+02, &
& 8.7002090e+02,8.6401750e+02,8.3647440e+02,8.6205870e+02,8.3468930e+02, &
& 4.6440900e+01,1.5187590e+02,1.9344570e+02,1.4696060e+02,1.1006510e+02]
 vdw_dftd3_c6(20401:20500)=[&
& 7.5928700e+01,5.4869300e+01,3.7320200e+01,2.2190310e+02,3.4351060e+02, &
& 3.4738790e+02,2.7806710e+02,2.2676600e+02,1.9099270e+02,1.5536720e+02, &
& 3.0442220e+02,5.8505150e+02,3.0419020e+02,2.9348990e+02,2.8758510e+02, &
& 2.8497290e+02,2.6188660e+02,2.4213490e+02,2.3089560e+02,2.2574350e+02, &
& 2.2325740e+02,2.0907070e+02,3.4402320e+02,3.0131020e+02,2.6923080e+02, &
& 2.4516040e+02,2.1518160e+02,3.6280200e+02,7.1158620e+02,5.4236550e+02, &
& 4.0208770e+02,4.0601940e+02,3.7812310e+02,4.2477910e+02,3.2883410e+02, &
& 3.0737400e+02,2.8513600e+02,2.7584840e+02,2.7272280e+02,4.3605730e+02, &
& 3.9365950e+02,3.6794630e+02,3.4743580e+02,3.1650110e+02,4.2861700e+02, &
& 9.2250460e+02,6.8647720e+02,4.3015130e+02,4.2122950e+02,4.0778710e+02, &
& 4.0995360e+02,3.9549150e+02,4.1271440e+02,3.8742350e+02,3.9445350e+02, &
& 3.6762060e+02,3.5709490e+02,3.5486500e+02,3.7342540e+02,3.4296160e+02, &
& 4.6674760e+02,4.3239170e+02,3.9371210e+02,3.9890670e+02,3.4767760e+02, &
& 3.2665250e+02,3.1170830e+02,2.9781700e+02,2.8742670e+02,4.6737550e+02, &
& 4.2511580e+02,4.1176820e+02,3.9999560e+02,3.7274130e+02,4.8272480e+02, &
& 9.1693150e+02,5.0916980e+02,5.8862060e+02,5.2540620e+02,4.6359030e+02, &
& 4.4586270e+02,5.3855800e+02,1.2562090e+02,1.2309690e+02,8.9444800e+01, &
& 7.1201700e+01,4.8247600e+01,2.2342490e+02,2.6999430e+02,2.5843380e+02, &
& 2.2370080e+02,1.8709060e+02,3.1254240e+02,2.9990140e+02,3.0352280e+02, &
& 2.7760900e+02,2.0803810e+02,1.7749730e+02,1.7426540e+02,2.0521220e+02]
 vdw_dftd3_c6(20501:20600)=[&
& 1.9097590e+02,2.7172720e+02,2.8665450e+02,2.6704200e+01,1.7705100e+01, &
& 3.9243650e+02,2.3351510e+02,1.5949690e+02,1.0864070e+02,7.6299200e+01, &
& 5.7878400e+01,4.3876200e+01,3.3739100e+01,4.6995960e+02,3.7006380e+02, &
& 3.4393430e+02,2.7373340e+02,2.1501030e+02,1.7911600e+02,1.4674050e+02, &
& 1.2026980e+02,7.6653110e+02,6.4403510e+02,5.3415750e+02,5.1756110e+02, &
& 4.7488980e+02,3.7400380e+02,4.1001990e+02,3.2180920e+02,3.4272150e+02, &
& 3.5234030e+02,2.6993300e+02,2.7903750e+02,3.3018810e+02,2.9343220e+02, &
& 2.5181550e+02,2.2668450e+02,1.9893080e+02,1.7335820e+02,8.6036720e+02, &
& 7.6674210e+02,6.7620360e+02,6.1017610e+02,5.5778500e+02,4.3224960e+02, &
& 4.8162290e+02,3.6836950e+02,4.0280170e+02,3.7403530e+02,3.1127990e+02, &
& 3.3010750e+02,4.1255410e+02,3.8394660e+02,3.4410360e+02,3.2025290e+02, &
& 2.9042110e+02,2.6173610e+02,1.0488144e+03,9.7532420e+02,8.6267800e+02, &
& 4.0598460e+02,8.6579390e+02,8.3226700e+02,8.1163560e+02,7.9263850e+02, &
& 7.7580510e+02,6.1436540e+02,6.8438050e+02,6.6094210e+02,7.0089270e+02, &
& 6.8614780e+02,6.7291500e+02,6.6485520e+02,5.6289600e+02,5.5747290e+02, &
& 5.1061380e+02,4.3217460e+02,4.4031270e+02,4.0018070e+02,3.6727240e+02, &
& 3.0555600e+02,2.8562940e+02,2.9425470e+02,4.2390230e+02,4.1654960e+02, &
& 3.8473230e+02,3.6796170e+02,3.4061170e+02,3.1293820e+02,9.9568830e+02, &
& 9.6779800e+02,8.6060590e+02,7.7842680e+02,7.7007950e+02,7.4569540e+02, &
& 7.6439730e+02,7.4080980e+02,4.2778900e+01,1.3598320e+02,1.7394150e+02]
 vdw_dftd3_c6(20601:20700)=[&
& 1.3447040e+02,1.0184690e+02,7.1123800e+01,5.1919900e+01,3.5758200e+01, &
& 1.9806920e+02,3.0612750e+02,3.1225010e+02,2.5331120e+02,2.0873090e+02, &
& 1.7710850e+02,1.4519610e+02,2.7423250e+02,5.1543230e+02,2.7666340e+02, &
& 2.6710550e+02,2.6168720e+02,2.5898560e+02,2.3941200e+02,2.2180340e+02, &
& 2.1154220e+02,2.0671720e+02,2.0348070e+02,1.9216860e+02,3.1071100e+02, &
& 2.7494970e+02,2.4761740e+02,2.2675190e+02,2.0027860e+02,3.2770320e+02, &
& 6.2656050e+02,4.8524610e+02,3.6563120e+02,3.6911250e+02,3.4461330e+02, &
& 3.8416440e+02,3.0149240e+02,2.8204640e+02,2.6198970e+02,2.5321800e+02, &
& 2.5140820e+02,3.9410730e+02,3.5883980e+02,3.3748280e+02,3.2008780e+02, &
& 2.9314450e+02,3.8968370e+02,8.1023570e+02,6.1351580e+02,3.9384480e+02, &
& 3.8570960e+02,3.7356920e+02,3.7509470e+02,3.6042550e+02,3.7742120e+02, &
& 3.5468510e+02,3.6034200e+02,3.3690980e+02,3.2739560e+02,3.2523710e+02, &
& 3.4156110e+02,3.1451130e+02,4.2201500e+02,3.9293770e+02,3.5967380e+02, &
& 3.6277430e+02,3.1999870e+02,3.0124260e+02,2.8780780e+02,2.7498950e+02, &
& 2.6648330e+02,4.2338910e+02,3.8827410e+02,3.7778300e+02,3.6822760e+02, &
& 3.4477670e+02,4.3908390e+02,8.0936840e+02,4.6604700e+02,5.3507930e+02, &
& 4.7904230e+02,4.2495090e+02,4.0922440e+02,4.8694030e+02,1.1501370e+02, &
& 1.1314120e+02,8.3286500e+01,6.6781800e+01,4.5789900e+01,2.0272440e+02, &
& 2.4478960e+02,2.3590120e+02,2.0577840e+02,1.7352340e+02,2.8380130e+02, &
& 2.7376840e+02,2.7719390e+02,2.5351690e+02,1.9175980e+02,1.6452480e+02]
 vdw_dftd3_c6(20701:20800)=[&
& 1.6141780e+02,1.8837020e+02,1.7573990e+02,2.4774210e+02,2.6203540e+02, &
& 2.4124940e+02,2.5126800e+01,1.6915700e+01,3.5181040e+02,2.1331490e+02, &
& 1.4765020e+02,1.0167860e+02,7.2032800e+01,5.4994300e+01,4.1926000e+01, &
& 3.2389200e+01,4.2191950e+02,3.3688690e+02,3.1511470e+02,2.5302890e+02, &
& 2.0037470e+02,1.6788680e+02,1.3832530e+02,1.1395770e+02,6.8773700e+02, &
& 5.8340090e+02,4.8505940e+02,4.7134280e+02,4.3318300e+02,3.4166370e+02, &
& 3.7490260e+02,2.9477780e+02,3.1457060e+02,3.2282380e+02,2.4774600e+02, &
& 2.5705830e+02,3.0357640e+02,2.7154740e+02,2.3456310e+02,2.1209150e+02, &
& 1.8699940e+02,1.6369060e+02,7.7312550e+02,6.9465560e+02,6.1553170e+02, &
& 5.5720710e+02,5.1054640e+02,3.9746440e+02,4.4207860e+02,3.3981570e+02, &
& 3.7122770e+02,3.4521240e+02,2.8750250e+02,3.0524540e+02,3.7968440e+02, &
& 3.5506520e+02,3.1993540e+02,2.9881560e+02,2.7205830e+02,2.4614950e+02, &
& 9.4314870e+02,8.8280690e+02,7.8440000e+02,3.7731490e+02,7.8497740e+02, &
& 7.5507130e+02,7.3648110e+02,7.1934360e+02,7.0416270e+02,5.6098910e+02, &
& 6.2074640e+02,6.0005550e+02,6.3681140e+02,6.2345960e+02,6.1152130e+02, &
& 6.0402840e+02,5.1348430e+02,5.1090340e+02,4.6953520e+02,3.9880630e+02, &
& 4.0670940e+02,3.7075460e+02,3.4112450e+02,2.8477860e+02,2.6655850e+02, &
& 2.7475930e+02,3.9114380e+02,3.8562000e+02,3.5778450e+02,3.4316660e+02, &
& 3.1880970e+02,2.9392920e+02,8.9872340e+02,8.7826560e+02,7.8418190e+02, &
& 7.1316020e+02,7.0403150e+02,6.8185240e+02,6.9668950e+02,6.7555290e+02]
 vdw_dftd3_c6(20801:20900)=[&
& 3.9896700e+01,1.2448680e+02,1.5970990e+02,1.2488450e+02,9.5355300e+01, &
& 6.7193600e+01,4.9429100e+01,3.4372900e+01,1.8104870e+02,2.7950350e+02, &
& 2.8658090e+02,2.3450660e+02,1.9460170e+02,1.6599750e+02,1.3686440e+02, &
& 2.5244980e+02,4.6788320e+02,2.5604740e+02,2.4734020e+02,2.4231110e+02, &
& 2.3963570e+02,2.2229610e+02,2.0623670e+02,1.9674030e+02,1.9219700e+02, &
& 1.8865600e+02,1.7907470e+02,2.8616240e+02,2.5488570e+02,2.3075580e+02, &
& 2.1214750e+02,1.8822630e+02,3.0225300e+02,5.6874160e+02,4.4473220e+02, &
& 3.3846100e+02,3.4164920e+02,3.1953400e+02,3.5457680e+02,2.8063740e+02, &
& 2.6272550e+02,2.4429540e+02,2.3598710e+02,2.3484700e+02,3.6336970e+02, &
& 3.3258250e+02,3.1402340e+02,2.9872470e+02,2.7458620e+02,3.6080530e+02, &
& 7.3458650e+02,5.6200060e+02,3.6616290e+02,3.5862210e+02,3.4744280e+02, &
& 3.4859290e+02,3.3417340e+02,3.5060320e+02,3.2973370e+02,3.3455220e+02, &
& 3.1341130e+02,3.0463840e+02,3.0255580e+02,3.1727860e+02,2.9267680e+02, &
& 3.8927450e+02,3.6365250e+02,3.3401770e+02,3.3595050e+02,2.9862210e+02, &
& 2.8153260e+02,2.6922650e+02,2.5729480e+02,2.4994700e+02,3.9112750e+02, &
& 3.6046170e+02,3.5169120e+02,3.4355060e+02,3.2270510e+02,4.0664340e+02, &
& 7.3595700e+02,4.3316650e+02,4.9533320e+02,4.4442440e+02,3.9550150e+02, &
& 3.8118140e+02,4.4946340e+02,1.0680860e+02,1.0541670e+02,7.8317700e+01, &
& 6.3154900e+01,4.3702200e+01,1.8715040e+02,2.2592450e+02,2.1868530e+02, &
& 1.9178480e+02,1.6266590e+02,2.6236130e+02,2.5392720e+02,2.5718410e+02]
 vdw_dftd3_c6(20901:21000)=[&
& 2.3527290e+02,1.7915590e+02,1.5430230e+02,1.5132020e+02,1.7548260e+02, &
& 1.6398640e+02,2.2956900e+02,2.4320020e+02,2.2498180e+02,2.1052310e+02, &
& 3.6860600e+01,2.3840100e+01,6.2823990e+02,3.4816900e+02,2.2895120e+02, &
& 1.5197450e+02,1.0490040e+02,7.8693300e+01,5.9141500e+01,4.5194100e+01, &
& 7.4930050e+02,5.5878190e+02,5.0880610e+02,3.9434550e+02,3.0325210e+02, &
& 2.4942670e+02,2.0199970e+02,1.6402750e+02,1.2349320e+03,9.9446760e+02, &
& 8.1692750e+02,7.8468860e+02,7.1627780e+02,5.6371140e+02,6.1384180e+02, &
& 4.8104470e+02,5.0672720e+02,5.2370020e+02,4.0117140e+02,4.0794990e+02, &
& 4.8488130e+02,4.2218130e+02,3.5593940e+02,3.1709380e+02,2.7544760e+02, &
& 2.3792190e+02,1.3808411e+03,1.1858018e+03,1.0282536e+03,9.1823780e+02, &
& 8.3371350e+02,6.3877680e+02,7.1471880e+02,5.3996390e+02,5.9037440e+02, &
& 5.4597680e+02,4.5572670e+02,4.7937970e+02,6.0681450e+02,5.5532330e+02, &
& 4.8982280e+02,4.5162950e+02,4.0561580e+02,3.6235560e+02,1.6813249e+03, &
& 1.5165369e+03,1.3190312e+03,5.7973570e+02,1.3389789e+03,1.2836521e+03, &
& 1.2508766e+03,1.2208006e+03,1.1941104e+03,9.2828660e+02,1.0621534e+03, &
& 1.0223708e+03,1.0743220e+03,1.0512437e+03,1.0302928e+03,1.0187310e+03, &
& 8.5160220e+02,8.2898030e+02,7.5166620e+02,6.3136390e+02,6.4074450e+02, &
& 5.7745940e+02,5.2643720e+02,4.3510780e+02,4.0568950e+02,4.1649740e+02, &
& 6.2114960e+02,6.0248990e+02,5.4855160e+02,5.2038560e+02,4.7718560e+02, &
& 4.3481090e+02,1.5749942e+03,1.4917535e+03,1.3062558e+03,1.1610723e+03]
 vdw_dftd3_c6(21001:21100)=[&
& 1.1587996e+03,1.1217009e+03,1.1626290e+03,1.1245057e+03,6.0189200e+01, &
& 2.0169000e+02,2.5612950e+02,1.9164290e+02,1.4261130e+02,9.7808500e+01, &
& 7.0428500e+01,4.7741100e+01,2.9600050e+02,4.5880010e+02,4.5973700e+02, &
& 3.6358720e+02,2.9421340e+02,2.4672930e+02,1.9993530e+02,4.0435370e+02, &
& 7.9454790e+02,3.9932770e+02,3.8527560e+02,3.7765080e+02,3.7474370e+02, &
& 3.4218210e+02,3.1593550e+02,3.0136140e+02,2.9480040e+02,2.9297830e+02, &
& 2.7198180e+02,4.5391820e+02,3.9376600e+02,3.4965980e+02,3.1724980e+02, &
& 2.7746890e+02,4.8106050e+02,9.6807870e+02,7.2518090e+02,5.2865740e+02, &
& 5.3402030e+02,4.9651280e+02,5.6243130e+02,4.2953830e+02,4.0150480e+02, &
& 3.7226120e+02,3.6048910e+02,3.5460960e+02,5.7628960e+02,5.1576060e+02, &
& 4.7938630e+02,4.5112580e+02,4.0947120e+02,5.6434540e+02,1.2601619e+03, &
& 9.1944900e+02,5.6190760e+02,5.5022410e+02,5.3249960e+02,5.3594970e+02, &
& 5.1922710e+02,5.3962270e+02,5.0614540e+02,5.1646990e+02,4.7978110e+02, &
& 4.6587230e+02,4.6307400e+02,4.8786280e+02,4.4725110e+02,6.1717070e+02, &
& 5.6919750e+02,5.1595990e+02,5.2505030e+02,4.5276340e+02,4.2495970e+02, &
& 4.0531970e+02,3.8754040e+02,3.7247490e+02,6.1741750e+02,5.5680680e+02, &
& 5.3692140e+02,5.2008520e+02,4.8293520e+02,6.3518450e+02,1.2452372e+03, &
& 6.6511670e+02,7.7484830e+02,6.9052460e+02,6.0568150e+02,5.8187050e+02, &
& 7.1381570e+02,1.6346210e+02,1.6010980e+02,1.1541070e+02,9.1672500e+01, &
& 6.1915900e+01,2.9291740e+02,3.5455260e+02,3.3737570e+02,2.9047610e+02]
 vdw_dftd3_c6(21101:21200)=[&
& 2.4169490e+02,4.1078570e+02,3.9219670e+02,3.9682160e+02,3.6331250e+02, &
& 2.7075430e+02,2.3012310e+02,2.2605370e+02,2.6782340e+02,2.4873790e+02, &
& 3.5556210e+02,3.7409750e+02,3.4006880e+02,3.1469330e+02,4.9174480e+02, &
& 3.6054300e+01,2.3646200e+01,5.6732020e+02,3.2691860e+02,2.1933470e+02, &
& 1.4760490e+02,1.0284790e+02,7.7633500e+01,5.8635900e+01,4.4975300e+01, &
& 6.7817260e+02,5.2114300e+02,4.7968840e+02,3.7706390e+02,2.9323240e+02, &
& 2.4282560e+02,1.9787960e+02,1.6150560e+02,1.1105426e+03,9.1608550e+02, &
& 7.5654430e+02,7.3005840e+02,6.6827290e+02,5.2603240e+02,5.7500600e+02, &
& 4.5091200e+02,4.7789770e+02,4.9251980e+02,3.7723750e+02,3.8708210e+02, &
& 4.5890330e+02,4.0396460e+02,3.4379750e+02,3.0797330e+02,2.6898800e+02, &
& 2.3345810e+02,1.2441578e+03,1.0912408e+03,9.5501530e+02,8.5765270e+02, &
& 7.8153770e+02,6.0242530e+02,6.7255530e+02,5.1147220e+02,5.5934530e+02, &
& 5.1843740e+02,4.3197050e+02,4.5648750e+02,5.7380210e+02,5.2991860e+02, &
& 4.7140640e+02,4.3680220e+02,3.9431850e+02,3.5391780e+02,1.5153755e+03, &
& 1.3911764e+03,1.2212152e+03,5.5696970e+02,1.2319890e+03,1.1829141e+03, &
& 1.1532166e+03,1.1259127e+03,1.1017008e+03,8.6500110e+02,9.7497080e+02, &
& 9.4011780e+02,9.9351430e+02,9.7243020e+02,9.5340360e+02,9.4232490e+02, &
& 7.9310590e+02,7.7930040e+02,7.1047900e+02,5.9913420e+02,6.0936150e+02, &
& 5.5168900e+02,5.0477160e+02,4.1866960e+02,3.9091310e+02,4.0209420e+02, &
& 5.8845790e+02,5.7489740e+02,5.2748310e+02,5.0256880e+02,4.6316440e+02]
 vdw_dftd3_c6(21201:21300)=[&
& 4.2389390e+02,1.4300247e+03,1.3749340e+03,1.2142860e+03,1.0895911e+03, &
& 1.0821909e+03,1.0477793e+03,1.0795998e+03,1.0453553e+03,5.8275500e+01, &
& 1.8988940e+02,2.4200800e+02,1.8428470e+02,1.3843530e+02,9.5889700e+01, &
& 6.9580600e+01,4.7591400e+01,2.7757010e+02,4.2950590e+02,4.3451020e+02, &
& 3.4831620e+02,2.8458190e+02,2.4014930e+02,1.9582670e+02,3.8177780e+02, &
& 7.3212430e+02,3.8147190e+02,3.6814810e+02,3.6077080e+02,3.5748370e+02, &
& 3.2862230e+02,3.0396440e+02,2.8991370e+02,2.8344240e+02,2.8025690e+02, &
& 2.6257350e+02,4.3079250e+02,3.7769300e+02,3.3790570e+02,3.0809200e+02, &
& 2.7088100e+02,4.5524600e+02,8.9074400e+02,6.7945910e+02,5.0443250e+02, &
& 5.0941810e+02,4.7468070e+02,5.3297190e+02,4.1319030e+02,3.8639150e+02, &
& 3.5861330e+02,3.4693600e+02,3.4299870e+02,5.4666900e+02,4.9381290e+02, &
& 4.6187660e+02,4.3645200e+02,3.9803820e+02,5.3790930e+02,1.1549423e+03, &
& 8.6004650e+02,5.4009920e+02,5.2891450e+02,5.1208670e+02,5.1473080e+02, &
& 4.9650390e+02,5.1809120e+02,4.8646500e+02,4.9519130e+02,4.6164810e+02, &
& 4.4845310e+02,4.4561860e+02,4.6867840e+02,4.3067940e+02,5.8521160e+02, &
& 5.4252680e+02,4.9440990e+02,5.0071270e+02,4.3713590e+02,4.1096170e+02, &
& 3.9233860e+02,3.7499560e+02,3.6203330e+02,5.8648270e+02,5.3375820e+02, &
& 5.1718370e+02,5.0262410e+02,4.6878500e+02,6.0578760e+02,1.1481563e+03, &
& 6.3919770e+02,7.3861040e+02,6.5977710e+02,5.8247220e+02,5.6032370e+02, &
& 6.7595320e+02,1.5742770e+02,1.5452800e+02,1.1266800e+02,8.9962600e+01]
 vdw_dftd3_c6(21301:21400)=[&
& 6.1279600e+01,2.7968340e+02,3.3801530e+02,3.2383050e+02,2.8074710e+02, &
& 2.3527170e+02,3.9183600e+02,3.7616230e+02,3.8074190e+02,3.4837280e+02, &
& 2.6170420e+02,2.2356370e+02,2.1947140e+02,2.5791900e+02,2.4013260e+02, &
& 3.4065350e+02,3.5942740e+02,3.2894310e+02,3.0565080e+02,4.6926390e+02, &
& 4.5108450e+02,3.6540500e+01,2.4088300e+01,5.6215830e+02,3.2721920e+02, &
& 2.2083610e+02,1.4926580e+02,1.0433800e+02,7.8933900e+01,5.9729900e+01, &
& 4.5882400e+01,6.7243030e+02,5.2068680e+02,4.8073130e+02,3.7942120e+02, &
& 2.9608060e+02,2.4572860e+02,2.0066650e+02,1.6407610e+02,1.0998393e+03, &
& 9.1252050e+02,7.5461820e+02,7.2916220e+02,6.6796950e+02,5.2595350e+02, &
& 5.7539200e+02,4.5141300e+02,4.7910890e+02,4.9336920e+02,3.7799640e+02, &
& 3.8872950e+02,4.6049590e+02,4.0661960e+02,3.4703710e+02,3.1142450e+02, &
& 2.7249100e+02,2.3688550e+02,1.2329252e+03,1.0868394e+03,9.5348210e+02, &
& 8.5760110e+02,7.8230880e+02,6.0414750e+02,6.7400680e+02,5.1361970e+02, &
& 5.6161690e+02,5.2087810e+02,4.3392780e+02,4.5901640e+02,5.7580200e+02, &
& 5.3307320e+02,4.7538210e+02,4.4115130e+02,3.9888950e+02,3.5856750e+02, &
& 1.5020119e+03,1.3846011e+03,1.2183604e+03,5.6144680e+02,1.2271618e+03, &
& 1.1787222e+03,1.1492501e+03,1.1221393e+03,1.0981031e+03,8.6458960e+02, &
& 9.7085510e+02,9.3659920e+02,9.9084400e+02,9.6987250e+02,9.5097990e+02, &
& 9.3981690e+02,7.9250890e+02,7.8062700e+02,7.1279020e+02,6.0188010e+02, &
& 6.1249070e+02,5.5526090e+02,5.0858990e+02,4.2235350e+02,3.9454210e+02]
 vdw_dftd3_c6(21401:21500)=[&
& 4.0600450e+02,5.9097830e+02,5.7841880e+02,5.3186650e+02,5.0739330e+02, &
& 4.6832900e+02,4.2922150e+02,1.4201434e+03,1.3701849e+03,1.2127510e+03, &
& 1.0910589e+03,1.0823309e+03,1.0479820e+03,1.0780658e+03,1.0441684e+03, &
& 5.8862500e+01,1.9022830e+02,2.4272640e+02,1.8577620e+02,1.3998770e+02, &
& 9.7288100e+01,7.0786500e+01,4.8576900e+01,2.7779360e+02,4.2964620e+02, &
& 4.3578140e+02,3.5070760e+02,2.8738300e+02,2.4300890e+02,1.9857750e+02, &
& 3.8302240e+02,7.2962110e+02,3.8386450e+02,3.7051580e+02,3.6307110e+02, &
& 3.5962670e+02,3.3118730e+02,3.0651240e+02,2.9235140e+02,2.8578230e+02, &
& 2.8217240e+02,2.6504020e+02,4.3263090e+02,3.8045750e+02,3.4114820e+02, &
& 3.1153680e+02,2.7438370e+02,4.5707170e+02,8.8748100e+02,6.8026830e+02, &
& 5.0752160e+02,5.1249950e+02,4.7788700e+02,5.3531530e+02,4.1671520e+02, &
& 3.8976730e+02,3.6187710e+02,3.4999750e+02,3.4647920e+02,5.4905720e+02, &
& 4.9724240e+02,4.6592890e+02,4.4083920e+02,4.0264130e+02,5.4112590e+02, &
& 1.1497163e+03,8.6075030e+02,5.4452030e+02,5.3325740e+02,5.1635730e+02, &
& 5.1883450e+02,4.9986370e+02,5.2215120e+02,4.9043320e+02,4.9890510e+02, &
& 4.6555960e+02,4.5230680e+02,4.4940350e+02,4.7239670e+02,4.3441480e+02, &
& 5.8778890e+02,5.4572750e+02,4.9809170e+02,5.0376490e+02,4.4135260e+02, &
& 4.1515170e+02,3.9646860e+02,3.7893340e+02,3.6628550e+02,5.8938990e+02, &
& 5.3773400e+02,5.2173760e+02,5.0754760e+02,4.7401940e+02,6.0950110e+02, &
& 1.1447357e+03,6.4438060e+02,7.4302780e+02,6.6426180e+02,5.8740530e+02]
 vdw_dftd3_c6(21501:21600)=[&
& 5.6528300e+02,6.7885160e+02,1.5874420e+02,1.5597060e+02,1.1413350e+02, &
& 9.1307200e+01,6.2389300e+01,2.8129500e+02,3.3986220e+02,3.2624310e+02, &
& 2.8345370e+02,2.3808430e+02,3.9411100e+02,3.7893950e+02,3.8359990e+02, &
& 3.5096590e+02,2.6433470e+02,2.2616840e+02,2.2198580e+02,2.6021080e+02, &
& 2.4244220e+02,3.4305200e+02,3.6224070e+02,3.3219680e+02,3.0908380e+02, &
& 4.7210810e+02,4.5473560e+02,4.5868960e+02,3.5315200e+01,2.3359200e+01, &
& 5.4961140e+02,3.1671620e+02,2.1337350e+02,1.4425580e+02,1.0094600e+02, &
& 7.6465800e+01,5.7946700e+01,4.4577600e+01,6.5711510e+02,5.0467530e+02, &
& 4.6522590e+02,3.6661690e+02,2.8599260e+02,2.3744480e+02,1.9403010e+02, &
& 1.5879150e+02,1.0788336e+03,8.8761670e+02,7.3301140e+02,7.0781110e+02, &
& 6.4811200e+02,5.1073850e+02,5.5795140e+02,4.3806030e+02,4.6406680e+02, &
& 4.7803720e+02,3.6666480e+02,3.7623990e+02,4.4572790e+02,3.9303660e+02, &
& 3.3529210e+02,3.0092580e+02,2.6340340e+02,2.2911910e+02,1.2092081e+03, &
& 1.0578748e+03,9.2614390e+02,8.3217430e+02,7.5877890e+02,5.8581260e+02, &
& 6.5360280e+02,4.9792860e+02,5.4408210e+02,5.0450370e+02,4.2085520e+02, &
& 4.4449320e+02,5.5793400e+02,5.1573740e+02,4.5957500e+02,4.2642020e+02, &
& 3.8559230e+02,3.4670630e+02,1.4737970e+03,1.3493970e+03,1.1846610e+03, &
& 5.4305060e+02,1.1952654e+03,1.1474757e+03,1.1186278e+03,1.0921025e+03, &
& 1.0685805e+03,8.4000980e+02,9.4732140e+02,9.1363480e+02,9.6355250e+02, &
& 9.4306260e+02,9.2458410e+02,9.1377400e+02,7.6978010e+02,7.5663130e+02]
 vdw_dftd3_c6(21601:21700)=[&
& 6.9039390e+02,5.8308880e+02,5.9308720e+02,5.3746300e+02,4.9219170e+02, &
& 4.0891560e+02,3.8205000e+02,3.9294050e+02,5.7302740e+02,5.5997550e+02, &
& 5.1444520e+02,4.9063940e+02,4.5281960e+02,4.1506050e+02,1.3904637e+03, &
& 1.3339556e+03,1.1782143e+03,1.0582704e+03,1.0511231e+03,1.0177328e+03, &
& 1.0481038e+03,1.0148828e+03,5.6846600e+01,1.8412220e+02,2.3495680e+02, &
& 1.7952910e+02,1.3534930e+02,9.4163100e+01,6.8608700e+01,4.7194100e+01, &
& 2.6917780e+02,4.1641830e+02,4.2164610e+02,3.3887850e+02,2.7761880e+02, &
& 2.3483210e+02,1.9201920e+02,3.7144330e+02,7.1075730e+02,3.7132440e+02, &
& 3.5852320e+02,3.5134710e+02,3.4810210e+02,3.2022560e+02,2.9637490e+02, &
& 2.8273960e+02,2.7641130e+02,2.7310650e+02,2.5621710e+02,4.1862090e+02, &
& 3.6771370e+02,3.2961390e+02,3.0104260e+02,2.6523400e+02,4.4326820e+02, &
& 8.6524620e+02,6.6073770e+02,4.9130810e+02,4.9611270e+02,4.6265460e+02, &
& 5.1908450e+02,4.0320320e+02,3.7722490e+02,3.5029100e+02,3.3883350e+02, &
& 3.3513120e+02,5.3177200e+02,4.8095640e+02,4.5041590e+02,4.2611180e+02, &
& 3.8921120e+02,5.2415620e+02,1.1227672e+03,8.3655040e+02,5.2667590e+02, &
& 5.1577970e+02,4.9943430e+02,5.0191700e+02,4.8385420e+02,5.0506200e+02, &
& 4.7436130e+02,4.8273800e+02,4.5024130e+02,4.3740790e+02,4.3460070e+02, &
& 4.5679920e+02,4.2006140e+02,5.6959350e+02,5.2852650e+02,4.8214800e+02, &
& 4.8793610e+02,4.2697730e+02,4.0168310e+02,3.8365960e+02,3.6680700e+02, &
& 3.5437800e+02,5.7113440e+02,5.2034420e+02,5.0456970e+02,4.9075210e+02]
 vdw_dftd3_c6(21701:21800)=[&
& 4.5830100e+02,5.9030800e+02,1.1160377e+03,6.2320270e+02,7.1978040e+02, &
& 6.4370980e+02,5.6841140e+02,5.4692870e+02,6.5859550e+02,1.5329480e+02, &
& 1.5076980e+02,1.1036390e+02,8.8394400e+01,6.0521700e+01,2.7178760e+02, &
& 3.2857270e+02,3.1520730e+02,2.7386490e+02,2.3008710e+02,3.8125920e+02, &
& 3.6637160e+02,3.7088000e+02,3.3949100e+02,2.5576580e+02,2.1884580e+02, &
& 2.1480530e+02,2.5178560e+02,2.3455580e+02,3.3167180e+02,3.5008010e+02, &
& 3.2096670e+02,2.9868080e+02,4.5688220e+02,4.3965480e+02,4.4341820e+02, &
& 4.2889620e+02,3.1668100e+01,2.1351600e+01,4.6157510e+02,2.7354460e+02, &
& 1.8746660e+02,1.2849780e+02,9.0904300e+01,6.9418100e+01,5.2984500e+01, &
& 4.1003500e+01,5.5303570e+02,4.3382780e+02,4.0329140e+02,3.2151630e+02, &
& 2.5341230e+02,2.1190690e+02,1.7439200e+02,1.4363890e+02,9.0508480e+02, &
& 7.5708580e+02,6.2755600e+02,6.0830000e+02,5.5823240e+02,4.4055160e+02, &
& 4.8213620e+02,3.7926880e+02,4.0314840e+02,4.1431560e+02,3.1832390e+02, &
& 3.2847070e+02,3.8804460e+02,3.4515780e+02,2.9691650e+02,2.6796230e+02, &
& 2.3592640e+02,2.0636370e+02,1.0163718e+03,9.0210970e+02,7.9517640e+02, &
& 7.1766320e+02,6.5637740e+02,5.0974240e+02,5.6744410e+02,4.3508880e+02, &
& 4.7502730e+02,4.4133190e+02,3.6828980e+02,3.8983000e+02,4.8621160e+02, &
& 4.5246160e+02,4.0603120e+02,3.7843120e+02,3.4390960e+02,3.1074420e+02, &
& 1.2391934e+03,1.1485106e+03,1.0151154e+03,4.7942470e+02,1.0198084e+03, &
& 9.8014620e+02,9.5578910e+02,9.3335430e+02,9.1346790e+02,7.2393420e+02]
 vdw_dftd3_c6(21801:21900)=[&
& 8.0773280e+02,7.7996820e+02,8.2502430e+02,8.0759920e+02,7.9196020e+02, &
& 7.8241790e+02,6.6269770e+02,6.5575410e+02,6.0105960e+02,5.0981570e+02, &
& 5.1931350e+02,4.7249570e+02,4.3413160e+02,3.6221680e+02,3.3898810e+02, &
& 3.4897520e+02,5.0083330e+02,4.9181440e+02,4.5457810e+02,4.3514050e+02, &
& 4.0344890e+02,3.7142770e+02,1.1758873e+03,1.1395556e+03,1.0127341e+03, &
& 9.1652630e+02,9.0743040e+02,8.7883040e+02,9.0104700e+02,8.7318690e+02, &
& 5.0417300e+01,1.5944040e+02,2.0415190e+02,1.5835630e+02,1.2059460e+02, &
& 8.4858200e+01,6.2432400e+01,4.3476000e+01,2.3259320e+02,3.5918880e+02, &
& 3.6631710e+02,2.9774360e+02,2.4610860e+02,2.0955990e+02,1.7257580e+02, &
& 3.2363810e+02,6.0709190e+02,3.2607870e+02,3.1499830e+02,3.0868010e+02, &
& 3.0552730e+02,2.8243340e+02,2.6186300e+02,2.4986950e+02,2.4418470e+02, &
& 2.4036730e+02,2.2704050e+02,3.6528020e+02,3.2362930e+02,2.9204750e+02, &
& 2.6805260e+02,2.3750190e+02,3.8712360e+02,7.3864010e+02,5.7185300e+02, &
& 4.3135120e+02,4.3556920e+02,4.0706590e+02,4.5371970e+02,3.5663380e+02, &
& 3.3393870e+02,3.1050050e+02,3.0014930e+02,2.9784780e+02,4.6456790e+02, &
& 4.2315700e+02,3.9833180e+02,3.7825790e+02,3.4710000e+02,4.6012630e+02, &
& 9.5602290e+02,7.2324370e+02,4.6518180e+02,4.5559950e+02,4.4133640e+02, &
& 4.4305470e+02,4.2578150e+02,4.4561250e+02,4.1895420e+02,4.2555500e+02, &
& 3.9799830e+02,3.8678540e+02,3.8418230e+02,4.0306860e+02,3.7150160e+02, &
& 4.9758720e+02,4.6378430e+02,4.2506300e+02,4.2853830e+02,3.7888850e+02]
 vdw_dftd3_c6(21901:22000)=[&
& 3.5711560e+02,3.4149670e+02,3.2657380e+02,3.1656330e+02,5.0005880e+02, &
& 4.5872390e+02,4.4646550e+02,4.3546460e+02,4.0832830e+02,5.1837750e+02, &
& 9.5463010e+02,5.5025080e+02,6.3174190e+02,5.6637730e+02,5.0265170e+02, &
& 4.8421620e+02,5.7561980e+02,1.3524280e+02,1.3350900e+02,9.8890300e+01, &
& 7.9767200e+01,5.5248400e+01,2.3801690e+02,2.8752250e+02,2.7743120e+02, &
& 2.4266760e+02,2.0536950e+02,3.3432060e+02,3.2266550e+02,3.2676700e+02, &
& 2.9914950e+02,2.2729470e+02,1.9544560e+02,1.9173330e+02,2.2295460e+02, &
& 2.0815700e+02,2.9172410e+02,3.0857810e+02,2.8464110e+02,2.6599860e+02, &
& 4.0105410e+02,3.8807180e+02,3.9206550e+02,3.7920750e+02,3.3710750e+02, &
& 2.8262700e+01,1.9355600e+01,3.9142550e+02,2.3684570e+02,1.6457770e+02, &
& 1.1407740e+02,8.1421700e+01,6.2588600e+01,4.8054000e+01,3.7372300e+01, &
& 4.6976960e+02,3.7426910e+02,3.5028040e+02,2.8184800e+02,2.2400340e+02, &
& 1.8840440e+02,1.5594050e+02,1.2911050e+02,7.6793390e+02,6.4950000e+02, &
& 5.3986620e+02,5.2491730e+02,4.8256220e+02,3.8140060e+02,4.1786140e+02, &
& 3.2931160e+02,3.5085640e+02,3.5990170e+02,2.7699700e+02,2.8699270e+02, &
& 3.3826270e+02,3.0293890e+02,2.6234800e+02,2.3782350e+02,2.1038120e+02, &
& 1.8484770e+02,8.6376550e+02,7.7394630e+02,6.8574890e+02,6.2104380e+02, &
& 5.6941590e+02,4.4434710e+02,4.9374300e+02,3.8057300e+02,4.1514630e+02, &
& 3.8629860e+02,3.2258210e+02,3.4192370e+02,4.2431450e+02,3.9689200e+02, &
& 3.5815190e+02,3.3500590e+02,3.0566670e+02,2.7727710e+02,1.0537818e+03]
 vdw_dftd3_c6(22001:22100)=[&
& 9.8418610e+02,8.7424490e+02,4.2266280e+02,8.7559270e+02,8.4216510e+02, &
& 8.2140430e+02,8.0225710e+02,7.8528980e+02,6.2638240e+02,6.9366130e+02, &
& 6.7051140e+02,7.1007710e+02,6.9514360e+02,6.8179200e+02,6.7337970e+02, &
& 5.7285580e+02,5.6966440e+02,5.2402130e+02,4.4608380e+02,4.5488600e+02, &
& 4.1519500e+02,3.8249740e+02,3.2027240e+02,3.0015390e+02,3.0919500e+02, &
& 4.3822480e+02,4.3187000e+02,4.0107000e+02,3.8504170e+02,3.5831060e+02, &
& 3.3103010e+02,1.0040757e+03,9.7925300e+02,8.7428430e+02,7.9590040e+02, &
& 7.8618960e+02,7.6155790e+02,7.7812490e+02,7.5451690e+02,4.4596300e+01, &
& 1.3836130e+02,1.7768370e+02,1.3947630e+02,1.0709060e+02,7.6053200e+01, &
& 5.6396200e+01,3.9666800e+01,2.0154030e+02,3.1080830e+02,3.1873880e+02, &
& 2.6141050e+02,2.1763470e+02,1.8630930e+02,1.5431180e+02,2.8243840e+02, &
& 5.2184580e+02,2.8622410e+02,2.7665670e+02,2.7110010e+02,2.6813250e+02, &
& 2.4878950e+02,2.3100810e+02,2.2048020e+02,2.1540300e+02,2.1142380e+02, &
& 2.0075620e+02,3.1899680e+02,2.8455050e+02,2.5816170e+02,2.3789530e+02, &
& 2.1174100e+02,3.3850140e+02,6.3481680e+02,4.9663990e+02,3.7865110e+02, &
& 3.8232250e+02,3.5796340e+02,3.9703670e+02,3.1491710e+02,2.9510120e+02, &
& 2.7469180e+02,2.6540330e+02,2.6402010e+02,4.0621170e+02,3.7204100e+02, &
& 3.5163900e+02,3.3492300e+02,3.0847430e+02,4.0390770e+02,8.2041740e+02, &
& 6.2770030e+02,4.1025930e+02,4.0183710e+02,3.8938420e+02,3.9058410e+02, &
& 3.7444730e+02,3.9266670e+02,3.6947670e+02,3.7477260e+02,3.5123330e+02]
 vdw_dftd3_c6(22101:22200)=[&
& 3.4142990e+02,3.3904640e+02,3.5518230e+02,3.2797090e+02,4.3515580e+02, &
& 4.0701010e+02,3.7438790e+02,3.7634630e+02,3.3543720e+02,3.1664800e+02, &
& 3.0309840e+02,2.8993290e+02,2.8176770e+02,4.3810480e+02,4.0400650e+02, &
& 3.9433950e+02,3.8548280e+02,3.6263740e+02,4.5515750e+02,8.2187390e+02, &
& 4.8513840e+02,5.5458830e+02,4.9832020e+02,4.4380410e+02,4.2790900e+02, &
& 5.0378650e+02,1.1912060e+02,1.1797580e+02,8.8215800e+01,7.1572300e+01, &
& 5.0040600e+01,2.0841310e+02,2.5164210e+02,2.4391880e+02,2.1451760e+02, &
& 1.8261700e+02,2.9311980e+02,2.8387840e+02,2.8758070e+02,2.6334090e+02, &
& 2.0147070e+02,1.7392920e+02,1.7055570e+02,1.9706800e+02,1.8430860e+02, &
& 2.5639330e+02,2.7164740e+02,2.5179730e+02,2.3610840e+02,3.5195640e+02, &
& 3.4203460e+02,3.4602660e+02,3.3471740e+02,2.9885080e+02,2.6586540e+02, &
& 2.6814200e+01,1.8486400e+01,3.6441020e+02,2.2214890e+02,1.5516280e+02, &
& 1.0802000e+02,7.7377200e+01,5.9646000e+01,4.5913300e+01,3.5788400e+01, &
& 4.3764910e+02,3.5061410e+02,3.2894340e+02,2.6558080e+02,2.1173900e+02, &
& 1.7849350e+02,1.4807630e+02,1.2286250e+02,7.1523760e+02,6.0729210e+02, &
& 5.0528540e+02,4.9186430e+02,4.5247550e+02,3.5786670e+02,3.9219220e+02, &
& 3.0933800e+02,3.2981210e+02,3.3808050e+02,2.6041990e+02,2.7017940e+02, &
& 3.1812240e+02,2.8561020e+02,2.4796070e+02,2.2516700e+02,1.9955490e+02, &
& 1.7565240e+02,8.0500650e+02,7.2369110e+02,6.4242790e+02,5.8255440e+02, &
& 5.3462660e+02,4.1798890e+02,4.6412390e+02,3.5848400e+02,3.9089370e+02]
 vdw_dftd3_c6(22201:22300)=[&
& 3.6395130e+02,3.0405500e+02,3.2240840e+02,3.9931090e+02,3.7418700e+02, &
& 3.3835410e+02,3.1691230e+02,2.8960010e+02,2.6310510e+02,9.8230160e+02, &
& 9.1991100e+02,8.1862810e+02,3.9923510e+02,8.1904250e+02,7.8798460e+02, &
& 7.6861590e+02,7.5074320e+02,7.3490660e+02,5.8761560e+02,6.4899040e+02, &
& 6.2756590e+02,6.6479550e+02,6.5083380e+02,6.3836820e+02,6.3042130e+02, &
& 5.3719250e+02,5.3511600e+02,4.9290160e+02,4.2019700e+02,4.2865240e+02, &
& 3.9172610e+02,3.6124810e+02,3.0292430e+02,2.8406330e+02,2.9267440e+02, &
& 4.1285870e+02,4.0737440e+02,3.7897400e+02,3.6421970e+02,3.3940270e+02, &
& 3.1398490e+02,9.3738470e+02,9.1624010e+02,8.1940930e+02,7.4758980e+02, &
& 7.3789240e+02,7.1483440e+02,7.2948730e+02,7.0750950e+02,4.2162700e+01, &
& 1.2989070e+02,1.6698720e+02,1.3166500e+02,1.0141840e+02,7.2295000e+01, &
& 5.3786300e+01,3.7996100e+01,1.8913000e+02,2.9149160e+02,2.9952560e+02, &
& 2.4646800e+02,2.0575330e+02,1.7650780e+02,1.4652980e+02,2.6580330e+02, &
& 4.8833520e+02,2.6990590e+02,2.6094910e+02,2.5571250e+02,2.5285150e+02, &
& 2.3492290e+02,2.1825880e+02,2.0833810e+02,2.0352280e+02,1.9956410e+02, &
& 1.8984580e+02,3.0020530e+02,2.6845380e+02,2.4404670e+02,2.2523300e+02, &
& 2.0082890e+02,3.1880410e+02,5.9403870e+02,4.6651060e+02,3.5708190e+02, &
& 3.6054300e+02,3.3781860e+02,3.7403450e+02,2.9766760e+02,2.7903330e+02, &
& 2.5985840e+02,2.5103670e+02,2.4993600e+02,3.8254570e+02,3.5105910e+02, &
& 3.3229950e+02,3.1685780e+02,2.9224830e+02,3.8090100e+02,7.6731680e+02]
 vdw_dftd3_c6(22301:22400)=[&
& 5.8947680e+02,3.8757590e+02,3.7963090e+02,3.6791330e+02,3.6893680e+02, &
& 3.5339960e+02,3.7083500e+02,3.4904790e+02,3.5387000e+02,3.3189420e+02, &
& 3.2266300e+02,3.2037880e+02,3.3542660e+02,3.0995020e+02,4.0980230e+02, &
& 3.8380190e+02,3.5352930e+02,3.5500360e+02,3.1737010e+02,2.9978350e+02, &
& 2.8707700e+02,2.7465620e+02,2.6716680e+02,4.1292460e+02,3.8150770e+02, &
& 3.7276100e+02,3.6468730e+02,3.4349450e+02,4.2926480e+02,7.6957810e+02, &
& 4.5825190e+02,5.2305080e+02,4.7040220e+02,4.1948400e+02,4.0459980e+02, &
& 4.7467630e+02,1.1243910e+02,1.1151080e+02,8.3694600e+01,6.8072000e+01, &
& 4.7784700e+01,1.9631020e+02,2.3698540e+02,2.3009880e+02,2.0278480e+02, &
& 1.7302410e+02,2.7628420e+02,2.6790880e+02,2.7143870e+02,2.4860210e+02, &
& 1.9071700e+02,1.6490150e+02,1.6168180e+02,1.8635590e+02,1.7440700e+02, &
& 2.4186460e+02,2.5639940e+02,2.3809660e+02,2.2355550e+02,3.3186870e+02, &
& 3.2301240e+02,3.2694770e+02,3.1629900e+02,2.8288320e+02,2.5200750e+02, &
& 2.3900710e+02,2.6980300e+01,1.8402600e+01,3.9027660e+02,2.3089960e+02, &
& 1.5871240e+02,1.0928410e+02,7.7707400e+01,5.9619400e+01,4.5728900e+01, &
& 3.5556300e+01,4.6780860e+02,3.6632450e+02,3.4074550e+02,2.7209130e+02, &
& 2.1501450e+02,1.8027000e+02,1.4881480e+02,1.2298220e+02,7.6742050e+02, &
& 6.4014310e+02,5.3051110e+02,5.1447440e+02,4.7224010e+02,3.7322320e+02, &
& 4.0804110e+02,3.2148500e+02,3.4137600e+02,3.5071710e+02,2.6998860e+02, &
& 2.7836200e+02,3.2839780e+02,2.9239200e+02,2.5200000e+02,2.2783110e+02]
 vdw_dftd3_c6(22401:22500)=[&
& 2.0104720e+02,1.7630020e+02,8.6212220e+02,7.6317560e+02,6.7268360e+02, &
& 6.0730800e+02,5.5572040e+02,4.3227820e+02,4.8090810e+02,3.6943280e+02, &
& 4.0294980e+02,3.7453940e+02,3.1311430e+02,3.3108150e+02,4.1228890e+02, &
& 3.8378510e+02,3.4480820e+02,3.2172250e+02,2.9282270e+02,2.6506110e+02, &
& 1.0513610e+03,9.7214520e+02,8.5902210e+02,4.0725580e+02,8.6353440e+02, &
& 8.2985260e+02,8.0920320e+02,7.9017910e+02,7.7331220e+02,6.1339590e+02, &
& 6.8497150e+02,6.6143730e+02,6.9834480e+02,6.8356120e+02,6.7029400e+02, &
& 6.6217850e+02,5.6118900e+02,5.5510630e+02,5.0914750e+02,4.3253910e+02, &
& 4.4058660e+02,4.0122800e+02,3.6898490e+02,3.0849150e+02,2.8895780e+02, &
& 2.9737620e+02,4.2541930e+02,4.1768300e+02,3.8636610e+02,3.7011470e+02, &
& 3.4358050e+02,3.1677380e+02,9.9738560e+02,9.6461400e+02,8.5721330e+02, &
& 7.7636620e+02,7.6896120e+02,7.4479890e+02,7.6357260e+02,7.3996210e+02, &
& 4.2771400e+01,1.3469310e+02,1.7260210e+02,1.3425050e+02,1.0261550e+02, &
& 7.2586300e+01,5.3693400e+01,3.7696700e+01,1.9669260e+02,3.0353120e+02, &
& 3.0963670e+02,2.5211010e+02,2.0887230e+02,1.7828770e+02,1.4727550e+02, &
& 2.7459280e+02,5.1413850e+02,2.7656690e+02,2.6729850e+02,2.6198070e+02, &
& 2.5931910e+02,2.3979050e+02,2.2246400e+02,2.1235000e+02,2.0752820e+02, &
& 2.0425110e+02,1.9302210e+02,3.0923520e+02,2.7429900e+02,2.4791670e+02, &
& 2.2791320e+02,2.0237580e+02,3.2868890e+02,6.2589490e+02,4.8472570e+02, &
& 3.6605390e+02,3.6967440e+02,3.4576000e+02,3.8525950e+02,3.0329340e+02]
 vdw_dftd3_c6(22501:22600)=[&
& 2.8418210e+02,2.6443210e+02,2.5564790e+02,2.5366630e+02,3.9403700e+02, &
& 3.5913800e+02,3.3834720e+02,3.2158990e+02,2.9551790e+02,3.9060480e+02, &
& 8.1058950e+02,6.1313650e+02,3.9521610e+02,3.8709050e+02,3.7502280e+02, &
& 3.7642310e+02,3.6171140e+02,3.7848060e+02,3.5595940e+02,3.6149530e+02, &
& 3.3818990e+02,3.2868340e+02,3.2643780e+02,3.4224650e+02,3.1566390e+02, &
& 4.2204710e+02,3.9370480e+02,3.6119710e+02,3.6399550e+02,3.2246250e+02, &
& 3.0419920e+02,2.9108740e+02,2.7853890e+02,2.7011300e+02,4.2470380e+02, &
& 3.8982000e+02,3.7955200e+02,3.7040460e+02,3.4770470e+02,4.4002170e+02, &
& 8.0924300e+02,4.6736610e+02,5.3649470e+02,4.8151730e+02,4.2750540e+02, &
& 4.1194110e+02,4.8914000e+02,1.1455870e+02,1.1335080e+02,8.4329800e+01, &
& 6.8302900e+01,4.7642200e+01,2.0139880e+02,2.4332100e+02,2.3501080e+02, &
& 2.0596780e+02,1.7475170e+02,2.8343990e+02,2.7371790e+02,2.7724060e+02, &
& 2.5398530e+02,1.9359700e+02,1.6675330e+02,1.6358130e+02,1.8973800e+02, &
& 1.7725080e+02,2.4732620e+02,2.6164130e+02,2.4169560e+02,2.2618930e+02, &
& 3.4023920e+02,3.2940570e+02,3.3291030e+02,3.2221010e+02,2.8693400e+02, &
& 2.5477490e+02,2.4134970e+02,2.4465860e+02,2.8063100e+01,1.9020800e+01, &
& 4.1221710e+02,2.4288250e+02,1.6615330e+02,1.1391260e+02,8.0711800e+01, &
& 6.1761900e+01,4.7264500e+01,3.6683700e+01,4.9392410e+02,3.8571180e+02, &
& 3.5802870e+02,2.8501380e+02,2.2452110e+02,1.8780070e+02,1.5466930e+02, &
& 1.2754830e+02,8.0942550e+02,6.7457400e+02,5.5875930e+02,5.4137410e+02]
 vdw_dftd3_c6(22601:22700)=[&
& 4.9669050e+02,3.9222970e+02,4.2885090e+02,3.3757700e+02,3.5837430e+02, &
& 3.6840320e+02,2.8332920e+02,2.9187250e+02,3.4456860e+02,3.0611920e+02, &
& 2.6318210e+02,2.3751920e+02,2.0919710e+02,1.8310950e+02,9.0881520e+02, &
& 8.0400360e+02,7.0783430e+02,6.3844630e+02,5.8376530e+02,4.5332840e+02, &
& 5.0466070e+02,3.8697280e+02,4.2232030e+02,3.9235480e+02,3.2780150e+02, &
& 3.4660550e+02,4.3231830e+02,4.0184330e+02,3.6034200e+02,3.3576050e+02, &
& 3.0512160e+02,2.7575770e+02,1.1077988e+03,1.0240901e+03,9.0399280e+02, &
& 4.2561430e+02,9.0929190e+02,8.7375470e+02,8.5199520e+02,8.3195410e+02, &
& 8.1418450e+02,6.4463620e+02,7.2073150e+02,6.9578030e+02,7.3512820e+02, &
& 7.1956930e+02,7.0559090e+02,6.9711310e+02,5.9004660e+02,5.8295090e+02, &
& 5.3409830e+02,4.5309060e+02,4.6141030e+02,4.1975000e+02,3.8566670e+02, &
& 3.2198390e+02,3.0143730e+02,3.1019540e+02,4.4555610e+02,4.3708990e+02, &
& 4.0369830e+02,3.8631140e+02,3.5811890e+02,3.2972180e+02,1.0501750e+03, &
& 1.0154760e+03,9.0160180e+02,8.1526350e+02,8.0784930e+02,7.8242750e+02, &
& 8.0290250e+02,7.7797910e+02,4.4649800e+01,1.4156390e+02,1.8118110e+02, &
& 1.4035990e+02,1.0693200e+02,7.5363600e+01,5.5574400e+01,3.8869700e+01, &
& 2.0679400e+02,3.1921760e+02,3.2513700e+02,2.6392340e+02,2.1806750e+02, &
& 1.8573680e+02,1.5307080e+02,2.8782600e+02,5.4107730e+02,2.8949650e+02, &
& 2.7971050e+02,2.7414900e+02,2.7142500e+02,2.5071870e+02,2.3247500e+02, &
& 2.2187670e+02,2.1686020e+02,2.1363790e+02,2.0155080e+02,3.2427260e+02]
 vdw_dftd3_c6(22701:22800)=[&
& 2.8698700e+02,2.5886990e+02,2.3760650e+02,2.1059700e+02,3.4426700e+02, &
& 6.5851960e+02,5.0868390e+02,3.8304250e+02,3.8685740e+02,3.6157810e+02, &
& 4.0341880e+02,3.1674990e+02,2.9669340e+02,2.7595750e+02,2.6684590e+02, &
& 2.6459410e+02,4.1294080e+02,3.7573450e+02,3.5348570e+02,3.3559490e+02, &
& 3.0794100e+02,4.0861880e+02,8.5276800e+02,6.4344430e+02,4.1297590e+02, &
& 4.0447640e+02,3.9181900e+02,3.9338030e+02,3.7830190e+02,3.9560900e+02, &
& 3.7196670e+02,3.7790420e+02,3.5332080e+02,3.4335820e+02,3.4104430e+02, &
& 3.5776850e+02,3.2975260e+02,4.4211340e+02,4.1196690e+02,3.7750500e+02, &
& 3.8078680e+02,3.3644710e+02,3.1720710e+02,3.0342150e+02,2.9030040e+02, &
& 2.8128840e+02,4.4468180e+02,4.0753420e+02,3.9643040e+02,3.8655960e+02, &
& 3.6242110e+02,4.6028590e+02,8.5084330e+02,4.8843100e+02,5.6129360e+02, &
& 5.0330170e+02,4.4649080e+02,4.3011690e+02,5.1213360e+02,1.1979600e+02, &
& 1.1834870e+02,8.7732000e+01,7.0890900e+01,4.9262900e+01,2.1108310e+02, &
& 2.5499480e+02,2.4589890e+02,2.1505640e+02,1.8203880e+02,2.9681460e+02, &
& 2.8630250e+02,2.8995300e+02,2.6557670e+02,2.0190070e+02,1.7364950e+02, &
& 1.7037740e+02,1.9810080e+02,1.8495540e+02,2.5878790e+02,2.7362340e+02, &
& 2.5230300e+02,2.3579430e+02,3.5611480e+02,3.4430530e+02,3.4779920e+02, &
& 3.3655350e+02,2.9922510e+02,2.6533520e+02,2.5122310e+02,2.5497130e+02, &
& 2.6587460e+02,3.4893100e+01,2.2907600e+01,5.6526540e+02,3.2028070e+02, &
& 2.1344840e+02,1.4316020e+02,9.9643500e+01,7.5235100e+01,5.6892200e+01]
 vdw_dftd3_c6(22801:22900)=[&
& 4.3721500e+01,6.7522680e+02,5.1209020e+02,4.6945400e+02,3.6720280e+02, &
& 2.8461460e+02,2.3533010e+02,1.9158160e+02,1.5631790e+02,1.1101907e+03, &
& 9.0511860e+02,7.4582820e+02,7.1855370e+02,6.5710020e+02,5.1754970e+02, &
& 5.6463620e+02,4.4298550e+02,4.6815800e+02,4.8295770e+02,3.7028130e+02, &
& 3.7846340e+02,4.4874480e+02,3.9348230e+02,3.3391140e+02,2.9869330e+02, &
& 2.6059250e+02,2.2602150e+02,1.2430036e+03,1.0788447e+03,9.4063630e+02, &
& 8.4295940e+02,7.6722430e+02,5.9040500e+02,6.5957470e+02,5.0072770e+02, &
& 5.4733000e+02,5.0698660e+02,4.2310210e+02,4.4611840e+02,5.6190060e+02, &
& 5.1714790e+02,4.5876760e+02,4.2447180e+02,3.8268090e+02,3.4312810e+02, &
& 1.5141166e+03,1.3775185e+03,1.2045072e+03,5.4238350e+02,1.2188063e+03, &
& 1.1693656e+03,1.1397842e+03,1.1126065e+03,1.0884965e+03,8.5158160e+02, &
& 9.6631820e+02,9.3121350e+02,9.8060600e+02,9.5967190e+02,9.4073330e+02, &
& 9.2993260e+02,7.8085310e+02,7.6411810e+02,6.9534460e+02,5.8588080e+02, &
& 5.9538900e+02,5.3831140e+02,4.9206190e+02,4.0798400e+02,3.8091340e+02, &
& 3.9149310e+02,5.7627840e+02,5.6137620e+02,5.1373240e+02,4.8880570e+02, &
& 4.4985640e+02,4.1128990e+02,1.4239693e+03,1.3587684e+03,1.1959642e+03, &
& 1.0695257e+03,1.0645406e+03,1.0306377e+03,1.0644434e+03,1.0302277e+03, &
& 5.6518900e+01,1.8591700e+02,2.3667710e+02,1.7917550e+02,1.3430140e+02, &
& 9.2908600e+01,6.7419300e+01,4.6208200e+01,2.7237350e+02,4.2155730e+02, &
& 4.2489670e+02,3.3903150e+02,2.7621540e+02,2.3276490e+02,1.8961390e+02]
 vdw_dftd3_c6(22901:23000)=[&
& 3.7405850e+02,7.2398140e+02,3.7201170e+02,3.5910570e+02,3.5198070e+02, &
& 3.4899300e+02,3.2004310e+02,2.9592090e+02,2.8231300e+02,2.7608710e+02, &
& 2.7351670e+02,2.5541010e+02,4.2085570e+02,3.6762820e+02,3.2815180e+02, &
& 2.9882440e+02,2.6244940e+02,4.4576050e+02,8.8157390e+02,6.6764710e+02, &
& 4.9222910e+02,4.9713820e+02,4.6306700e+02,5.2169780e+02,4.0238250e+02, &
& 3.7636250e+02,3.4931490e+02,3.3811650e+02,3.3364080e+02,5.3472480e+02, &
& 4.8141850e+02,4.4934480e+02,4.2408620e+02,3.8628580e+02,5.2507760e+02, &
& 1.1453931e+03,8.4571730e+02,5.2589560e+02,5.1499740e+02,4.9856690e+02, &
& 5.0136870e+02,4.8441020e+02,5.0462370e+02,4.7370770e+02,4.8262190e+02, &
& 4.4936380e+02,4.3646450e+02,4.3373820e+02,4.5634090e+02,4.1908700e+02, &
& 5.7239970e+02,5.2974800e+02,4.8197450e+02,4.8896420e+02,4.2523900e+02, &
& 3.9970400e+02,3.8159490e+02,3.6491310e+02,3.5178910e+02,5.7370310e+02, &
& 5.2042840e+02,5.0341800e+02,4.8873500e+02,4.5527610e+02,5.9117920e+02, &
& 1.1355611e+03,6.2234260e+02,7.2148760e+02,6.4433230e+02,5.6739040e+02, &
& 5.4560930e+02,6.6224690e+02,1.5292390e+02,1.5012670e+02,1.0920000e+02, &
& 8.7207600e+01,5.9451100e+01,2.7256770e+02,3.2959200e+02,3.1501770e+02, &
& 2.7258640e+02,2.2804730e+02,3.8227200e+02,3.6630170e+02,3.7073900e+02, &
& 3.3943990e+02,2.5457810e+02,2.1724670e+02,2.1333730e+02,2.5122060e+02, &
& 2.3374770e+02,3.3179130e+02,3.4968000e+02,3.1937190e+02,2.9645520e+02, &
& 4.5781020e+02,4.3894830e+02,4.4222600e+02,4.2786320e+02,3.7713750e+02]
 vdw_dftd3_c6(23001:23100)=[&
& 3.3210880e+02,3.1358880e+02,3.2040110e+02,3.3498740e+02,4.2791400e+02, &
& 3.7912200e+01,2.4783300e+01,6.1156120e+02,3.4839490e+02,2.3225130e+02, &
& 1.5560830e+02,1.0812990e+02,8.1505800e+01,6.1521400e+01,4.7194200e+01, &
& 7.3065870e+02,5.5662410e+02,5.1060870e+02,3.9957740e+02,3.0959920e+02, &
& 2.5580130e+02,2.0804020e+02,1.6954570e+02,1.1982993e+03,9.8182060e+02, &
& 8.0962470e+02,7.8020170e+02,7.1360600e+02,5.6166940e+02,6.1332300e+02, &
& 4.8085880e+02,5.0875980e+02,5.2479280e+02,4.0197520e+02,4.1137780e+02, &
& 4.8787860e+02,4.2801030e+02,3.6315760e+02,3.2471350e+02,2.8310510e+02, &
& 2.4534090e+02,1.3416152e+03,1.1697495e+03,1.0209627e+03,9.1534550e+02, &
& 8.3320500e+02,6.4106430e+02,7.1622390e+02,5.4361040e+02,5.9452770e+02, &
& 5.5071590e+02,4.5912190e+02,4.8457590e+02,6.1033820e+02,5.6212760e+02, &
& 4.9873950e+02,4.6138680e+02,4.1581730e+02,3.7264800e+02,1.6336348e+03, &
& 1.4924625e+03,1.3065898e+03,5.8945280e+02,1.3207683e+03,1.2676087e+03, &
& 1.2356389e+03,1.2062616e+03,1.1802039e+03,9.2388260e+02,1.0458286e+03, &
& 1.0079093e+03,1.0636161e+03,1.0409736e+03,1.0204995e+03,1.0087709e+03, &
& 8.4732020e+02,8.3014100e+02,7.5559170e+02,6.3637500e+02,6.4685600e+02, &
& 5.8485060e+02,5.3455500e+02,4.4293230e+02,4.1343530e+02,4.2506050e+02, &
& 6.2552300e+02,6.0984910e+02,5.5825050e+02,5.3115730e+02,4.8873370e+02, &
& 4.4666830e+02,1.5382573e+03,1.4729117e+03,1.2978097e+03,1.1613917e+03, &
& 1.1551516e+03,1.1183627e+03,1.1544374e+03,1.1174761e+03,6.1486800e+01]
 vdw_dftd3_c6(23101:23200)=[&
& 2.0220530e+02,2.5735510e+02,1.9488950e+02,1.4592740e+02,1.0078260e+02, &
& 7.2992300e+01,4.9877800e+01,2.9601550e+02,4.5814250e+02,4.6215210e+02, &
& 3.6887960e+02,3.0043180e+02,2.5300030e+02,2.0589540e+02,4.0608450e+02, &
& 7.8442190e+02,4.0440960e+02,3.9026550e+02,3.8249970e+02,3.7920210e+02, &
& 3.4791510e+02,3.2164840e+02,3.0680410e+02,3.0002320e+02,2.9714060e+02, &
& 2.7760830e+02,4.5760310e+02,3.9986460e+02,3.5687340e+02,3.2485020e+02, &
& 2.8512610e+02,4.8384610e+02,9.5464460e+02,7.2429340e+02,5.3482740e+02, &
& 5.4017920e+02,5.0302360e+02,5.6629020e+02,4.3711840e+02,4.0874340e+02, &
& 3.7927910e+02,3.6709070e+02,3.6239660e+02,5.8092600e+02,5.2329100e+02, &
& 4.8848390e+02,4.6096710e+02,4.1974500e+02,5.7029930e+02,1.2390351e+03, &
& 9.1712310e+02,5.7151660e+02,5.5966960e+02,5.4179760e+02,5.4480780e+02, &
& 5.2623920e+02,5.4841680e+02,5.1479430e+02,5.2439740e+02,4.8836000e+02, &
& 4.7434360e+02,4.7138980e+02,4.9604540e+02,4.5549010e+02,6.2165690e+02, &
& 5.7541630e+02,5.2355470e+02,5.3103290e+02,4.6190580e+02,4.3406330e+02, &
& 4.1431110e+02,3.9608640e+02,3.8190670e+02,6.2293810e+02,5.6542470e+02, &
& 5.4706010e+02,5.3109590e+02,4.9464220e+02,6.4214730e+02,1.2295628e+03, &
& 6.7640190e+02,7.8351290e+02,6.9937990e+02,6.1633520e+02,5.9268480e+02, &
& 7.1854240e+02,1.6642480e+02,1.6321840e+02,1.1859420e+02,9.4569600e+01, &
& 6.4302200e+01,2.9661470e+02,3.5855000e+02,3.4274070e+02,2.9645720e+02, &
& 2.4785630e+02,4.1557270e+02,3.9827950e+02,4.0309190e+02,3.6892090e+02]
 vdw_dftd3_c6(23201:23300)=[&
& 2.7646740e+02,2.3583660e+02,2.3159140e+02,2.7286680e+02,2.5387790e+02, &
& 3.6080170e+02,3.8032910e+02,3.4730490e+02,3.2226350e+02,4.9753160e+02, &
& 4.7720360e+02,4.8076680e+02,4.6495560e+02,4.0969630e+02,3.6063780e+02, &
& 3.4045080e+02,3.4781660e+02,3.6373910e+02,4.6496770e+02,5.0541480e+02, &
& 3.6227700e+01,2.4014100e+01,5.4685470e+02,3.2111700e+02,2.1779050e+02, &
& 1.4775190e+02,1.0359080e+02,7.8560000e+01,5.9593700e+01,4.5889400e+01, &
& 6.5457160e+02,5.1028240e+02,4.7233520e+02,3.7405660e+02,2.9271450e+02, &
& 2.4337850e+02,1.9910820e+02,1.6307830e+02,1.0694211e+03,8.9182320e+02, &
& 7.3841210e+02,7.1434970e+02,6.5487180e+02,5.1581520e+02,5.6471100e+02, &
& 4.4324120e+02,4.7101860e+02,4.8471110e+02,3.7150280e+02,3.8278190e+02, &
& 4.5296620e+02,4.0100980e+02,3.4304870e+02,3.0828460e+02,2.7015060e+02, &
& 2.3518820e+02,1.1994646e+03,1.0619947e+03,9.3369560e+02,8.4095330e+02, &
& 7.6784150e+02,5.9396240e+02,6.6226990e+02,5.0560890e+02,5.5281280e+02, &
& 5.1303450e+02,4.2738710e+02,4.5249980e+02,5.6656420e+02,5.2563100e+02, &
& 4.6972550e+02,4.3644300e+02,3.9516260e+02,3.5567420e+02,1.4614083e+03, &
& 1.3520527e+03,1.1922239e+03,5.5449760e+02,1.1993914e+03,1.1524260e+03, &
& 1.1237195e+03,1.0972997e+03,1.0738792e+03,8.4761650e+02,9.4867820e+02, &
& 9.1560320e+02,9.6949560e+02,9.4902750e+02,9.3061310e+02,9.1959590e+02, &
& 7.7679900e+02,7.6667010e+02,7.0099760e+02,5.9261750e+02,6.0337400e+02, &
& 5.4765300e+02,5.0212300e+02,4.1748050e+02,3.9019750e+02,4.0170990e+02]
 vdw_dftd3_c6(23301:23400)=[&
& 5.8191410e+02,5.7046250e+02,5.2553390e+02,5.0188930e+02,4.6384860e+02, &
& 4.2562050e+02,1.3840410e+03,1.3394353e+03,1.1880075e+03,1.0713664e+03, &
& 1.0616930e+03,1.0280688e+03,1.0561396e+03,1.0232018e+03,5.8196200e+01, &
& 1.8683530e+02,2.3860370e+02,1.8340000e+02,1.3853490e+02,9.6572400e+01, &
& 7.0463400e+01,4.8576600e+01,2.7268930e+02,4.2146260e+02,4.2844370e+02, &
& 3.4592360e+02,2.8414620e+02,2.4067810e+02,1.9703300e+02,3.7667550e+02, &
& 7.1337770e+02,3.7851240e+02,3.6542570e+02,3.5808830e+02,3.5459770e+02, &
& 3.2708070e+02,3.0288030e+02,2.8891150e+02,2.8239730e+02,2.7851920e+02, &
& 2.6216840e+02,4.2584750e+02,3.7544380e+02,3.3727630e+02,3.0839060e+02, &
& 2.7200760e+02,4.4976690e+02,8.6748280e+02,6.6777030e+02,5.0031680e+02, &
& 5.0520350e+02,4.7139900e+02,5.2700000e+02,4.1174300e+02,3.8521960e+02, &
& 3.5780550e+02,3.4602160e+02,3.4293620e+02,5.4066240e+02,4.9072030e+02, &
& 4.6051500e+02,4.3616550e+02,3.9886600e+02,5.3323930e+02,1.1229068e+03, &
& 8.4459250e+02,5.3785080e+02,5.2673890e+02,5.1010490e+02,5.1239170e+02, &
& 4.9317410e+02,5.1559660e+02,4.8442780e+02,4.9251520e+02,4.5997740e+02, &
& 4.4693120e+02,4.4402250e+02,4.6651200e+02,4.2927350e+02,5.7861310e+02, &
& 5.3790480e+02,4.9161910e+02,4.9667380e+02,4.3648690e+02,4.1080070e+02, &
& 3.9246080e+02,3.7514240e+02,3.6302260e+02,5.8068260e+02,5.3094590e+02, &
& 5.1573940e+02,5.0211810e+02,4.6948140e+02,6.0069020e+02,1.1195215e+03, &
& 6.3642920e+02,7.3257850e+02,6.5541880e+02,5.8044230e+02,5.5878270e+02]
 vdw_dftd3_c6(23401:23500)=[&
& 6.6842250e+02,1.5675410e+02,1.5413350e+02,1.1315480e+02,9.0704300e+01, &
& 6.2195400e+01,2.7727810e+02,3.3486500e+02,3.2195190e+02,2.8021780e+02, &
& 2.3581280e+02,3.8844680e+02,3.7399630e+02,3.7864590e+02,3.4645320e+02, &
& 2.6154130e+02,2.2411260e+02,2.1995380e+02,2.5727540e+02,2.3987350e+02, &
& 3.3847720e+02,3.5762210e+02,3.2850830e+02,3.0597730e+02,4.6536830e+02, &
& 4.4902050e+02,4.5315700e+02,4.3804100e+02,3.8788180e+02,3.4276350e+02, &
& 3.2403570e+02,3.2956900e+02,3.4418570e+02,4.3665200e+02,4.7467200e+02, &
& 4.4805730e+02,3.5246100e+01,2.3585100e+01,5.1271640e+02,3.0576290e+02, &
& 2.0945590e+02,1.4319470e+02,1.0097450e+02,7.6886300e+01,5.8522300e+01, &
& 4.5183600e+01,6.1435230e+02,4.8450380e+02,4.5072670e+02,3.5936400e+02, &
& 2.8289020e+02,2.3614560e+02,1.9392690e+02,1.5936080e+02,1.0025559e+03, &
& 8.4305400e+02,6.9945860e+02,6.7814850e+02,6.2246350e+02,4.9067860e+02, &
& 5.3774700e+02,4.2250680e+02,4.4986640e+02,4.6232080e+02,3.5464230e+02, &
& 3.6661620e+02,4.3326650e+02,3.8551090e+02,3.3138440e+02,2.9873310e+02, &
& 2.6262260e+02,2.2931350e+02,1.1257134e+03,1.0038665e+03,8.8595760e+02, &
& 7.9994710e+02,7.3167410e+02,5.6783410e+02,6.3235020e+02,4.8447400e+02, &
& 5.2945790e+02,4.9188120e+02,4.0979950e+02,4.3443580e+02,5.4208910e+02, &
& 5.0487630e+02,4.5302540e+02,4.2201870e+02,3.8318370e+02,3.4582830e+02, &
& 1.3722422e+03,1.2768973e+03,1.1301276e+03,5.3452440e+02,1.1341681e+03, &
& 1.0903536e+03,1.0633516e+03,1.0384792e+03,1.0164371e+03,8.0596520e+02]
 vdw_dftd3_c6(23501:23600)=[&
& 8.9702580e+02,8.6641260e+02,9.1841900e+02,8.9909510e+02,8.8176000e+02, &
& 8.7114240e+02,7.3817630e+02,7.3134130e+02,6.7040070e+02,5.6812010e+02, &
& 5.7890380e+02,5.2661010e+02,4.8371080e+02,4.0307740e+02,3.7705920e+02, &
& 3.8840990e+02,5.5767540e+02,5.4820960e+02,5.0681290e+02,4.8504920e+02, &
& 4.4946500e+02,4.1343530e+02,1.3034917e+03,1.2675667e+03,1.1280206e+03, &
& 1.0215531e+03,1.0104948e+03,9.7859690e+02,1.0027172e+03,9.7186890e+02, &
& 5.6276900e+01,1.7817450e+02,2.2803400e+02,1.7677650e+02,1.3427420e+02, &
& 9.4162200e+01,6.9039900e+01,4.7877100e+01,2.5966980e+02,4.0103040e+02, &
& 4.0936850e+02,3.3269600e+02,2.7467950e+02,2.3351090e+02,1.9189600e+02, &
& 3.6037990e+02,6.7520960e+02,3.6377550e+02,3.5131790e+02,3.4423720e+02, &
& 3.4067740e+02,3.1512290e+02,2.9209790e+02,2.7865290e+02,2.7230380e+02, &
& 2.6795690e+02,2.5325200e+02,4.0786260e+02,3.6139310e+02,3.2590920e+02, &
& 2.9882550e+02,2.6438680e+02,4.3088900e+02,8.2090550e+02,6.3678340e+02, &
& 4.8083080e+02,4.8546850e+02,4.5354170e+02,5.0518300e+02,3.9728740e+02, &
& 3.7184930e+02,3.4561400e+02,3.3407750e+02,3.3175670e+02,5.1804080e+02, &
& 4.7212430e+02,4.4439750e+02,4.2181780e+02,3.8675500e+02,5.1246920e+02, &
& 1.0614185e+03,8.0501180e+02,5.1860750e+02,5.0791390e+02,4.9198270e+02, &
& 4.9389760e+02,4.7446600e+02,4.9685260e+02,4.6706730e+02,4.7437630e+02, &
& 4.4371620e+02,4.3121480e+02,4.2833450e+02,4.4958170e+02,4.1422170e+02, &
& 5.5455640e+02,5.1681140e+02,4.7354520e+02,4.7737190e+02,4.2195370e+02]
 vdw_dftd3_c6(23601:23700)=[&
& 3.9750830e+02,3.7998620e+02,3.6322860e+02,3.5217630e+02,5.5705210e+02, &
& 5.1133530e+02,4.9776870e+02,4.8542330e+02,4.5493030e+02,5.7742140e+02, &
& 1.0607432e+03,6.1356050e+02,7.0396650e+02,6.3075620e+02,5.6000710e+02, &
& 5.3943990e+02,6.4068280e+02,1.5112750e+02,1.4889980e+02,1.1000870e+02, &
& 8.8495300e+01,6.1025800e+01,2.6613770e+02,3.2131290e+02,3.0994780e+02, &
& 2.7080050e+02,2.2881280e+02,3.7302450e+02,3.6007160e+02,3.6462620e+02, &
& 3.3363080e+02,2.5301950e+02,2.1739930e+02,2.1329230e+02,2.4838210e+02, &
& 2.3186030e+02,3.2567860e+02,3.4453460e+02,3.1759860e+02,2.9651530e+02, &
& 4.4715550e+02,4.3284780e+02,4.3727120e+02,4.2265520e+02,3.7537250e+02, &
& 3.3249710e+02,3.1460820e+02,3.1917350e+02,3.3302450e+02,4.2051550e+02, &
& 4.5707930e+02,4.3269410e+02,4.1857890e+02,4.4832100e+01,2.9186400e+01, &
& 7.5032270e+02,4.1786520e+02,2.7634300e+02,1.8434620e+02,1.2774190e+02, &
& 9.6095000e+01,7.2385200e+01,5.5409200e+01,8.9516260e+02,6.6976420e+02, &
& 6.1142610e+02,4.7565060e+02,3.6714490e+02,3.0279080e+02,2.4586350e+02, &
& 2.0010940e+02,1.4768694e+03,1.1905306e+03,9.7858840e+02,9.4093240e+02, &
& 8.5937300e+02,6.7681130e+02,7.3708980e+02,5.7808990e+02,6.0928650e+02, &
& 6.2924930e+02,4.8241490e+02,4.9120730e+02,5.8352030e+02,5.0945200e+02, &
& 4.3080210e+02,3.8458730e+02,3.3481150e+02,2.8979810e+02,1.6522832e+03, &
& 1.4199417e+03,1.2330006e+03,1.1022638e+03,1.0016660e+03,7.6883820e+02, &
& 8.5963920e+02,6.5071280e+02,7.1108920e+02,6.5796470e+02,5.4942990e+02]
 vdw_dftd3_c6(23701:23800)=[&
& 5.7810960e+02,7.3051140e+02,6.6979600e+02,5.9220770e+02,5.4693930e+02, &
& 4.9213770e+02,4.4046680e+02,2.0130282e+03,1.8160675e+03,1.5814671e+03, &
& 7.0080410e+02,1.6040906e+03,1.5379273e+03,1.4986941e+03,1.4626834e+03, &
& 1.4307306e+03,1.1145114e+03,1.2733916e+03,1.2261232e+03,1.2874672e+03, &
& 1.2598096e+03,1.2347344e+03,1.2207557e+03,1.0219725e+03,9.9637850e+02, &
& 9.0460210e+02,7.6098160e+02,7.7254200e+02,6.9708220e+02,6.3615080e+02, &
& 5.2655380e+02,4.9121220e+02,5.0441110e+02,7.4864790e+02,7.2700690e+02, &
& 6.6322010e+02,6.3000140e+02,5.7868480e+02,5.2816950e+02,1.8871061e+03, &
& 1.7877406e+03,1.5670784e+03,1.3954519e+03,1.3918292e+03,1.3473076e+03, &
& 1.3948404e+03,1.3493008e+03,7.2901300e+01,2.4228720e+02,3.0814100e+02, &
& 2.3165430e+02,1.7302490e+02,1.1914340e+02,8.6077300e+01,5.8583600e+01, &
& 3.5534390e+02,5.5067610e+02,5.5286340e+02,4.3885970e+02,3.5626970e+02, &
& 2.9950590e+02,2.4334210e+02,4.8693460e+02,9.5257710e+02,4.8182380e+02, &
& 4.6502110e+02,4.5578960e+02,4.5213520e+02,4.1343280e+02,3.8195910e+02, &
& 3.6437600e+02,3.5638920e+02,3.5372270e+02,3.2913430e+02,5.4664020e+02, &
& 4.7553820e+02,4.2329180e+02,3.8476970e+02,3.3723270e+02,5.7979800e+02, &
& 1.1609040e+03,8.7228820e+02,6.3807920e+02,6.4445150e+02,5.9966140e+02, &
& 6.7812320e+02,5.1957380e+02,4.8580720e+02,4.5060820e+02,4.3621390e+02, &
& 4.2955510e+02,6.9425600e+02,6.2268260e+02,5.7980000e+02,5.4638720e+02, &
& 4.9679900e+02,6.8134670e+02,1.5113430e+03,1.1059643e+03,6.7935610e+02]
 vdw_dftd3_c6(23801:23900)=[&
& 6.6524220e+02,6.4390080e+02,6.4787420e+02,6.2696840e+02,6.5218100e+02, &
& 6.1190030e+02,6.2406450e+02,5.8019150e+02,5.6343460e+02,5.5999100e+02, &
& 5.8959320e+02,5.4094020e+02,7.4394190e+02,6.8699120e+02,6.2358700e+02, &
& 6.3381630e+02,5.4831610e+02,5.1495440e+02,4.9133470e+02,4.6979990e+02, &
& 4.5204520e+02,7.4445260e+02,6.7269630e+02,6.4947180e+02,6.2976130e+02, &
& 5.8566430e+02,7.6695690e+02,1.4944404e+03,8.0404920e+02,9.3536030e+02, &
& 8.3446090e+02,7.3260640e+02,7.0402570e+02,8.6065850e+02,1.9758390e+02, &
& 1.9383810e+02,1.4029040e+02,1.1170570e+02,7.5741700e+01,3.5305310e+02, &
& 4.2739490e+02,4.0746200e+02,3.5168440e+02,2.9341430e+02,4.9541720e+02, &
& 4.7369840e+02,4.7934490e+02,4.3891900e+02,3.2804360e+02,2.7929530e+02, &
& 2.7428730e+02,3.2404980e+02,3.0116290e+02,4.2932210e+02,4.5201960e+02, &
& 4.1181840e+02,3.8169420e+02,5.9335720e+02,5.6724450e+02,5.7102950e+02, &
& 5.5267180e+02,4.8602980e+02,4.2718180e+02,4.0302800e+02,4.1253210e+02, &
& 4.3149070e+02,5.5308270e+02,6.0097790e+02,5.6310510e+02,5.4167760e+02, &
& 7.1654750e+02,4.6820200e+01,3.0588600e+01,7.4890520e+02,4.2804980e+02, &
& 2.8602900e+02,1.9193550e+02,1.3344100e+02,1.0055750e+02,7.5829800e+01, &
& 5.8079500e+01,8.9474480e+02,6.8323710e+02,6.2753480e+02,4.9189290e+02, &
& 3.8167570e+02,3.1561930e+02,2.5684550e+02,2.0937070e+02,1.4672405e+03, &
& 1.2041091e+03,9.9328050e+02,9.5757000e+02,8.7600500e+02,6.8947940e+02, &
& 7.5309320e+02,5.9043900e+02,6.2501030e+02,6.4450280e+02,4.9360530e+02]
 vdw_dftd3_c6(23901:24000)=[&
& 5.0558370e+02,5.9983010e+02,5.2686600e+02,4.4755460e+02,4.0047040e+02, &
& 3.4937100e+02,3.0289600e+02,1.6430879e+03,1.4346583e+03,1.2531317e+03, &
& 1.1240457e+03,1.0235000e+03,7.8790450e+02,8.8003140e+02,6.6828590e+02, &
& 7.3081350e+02,6.7702710e+02,5.6424900e+02,5.9573220e+02,7.4999110e+02, &
& 6.9136370e+02,6.1397930e+02,5.6835230e+02,5.1253890e+02,4.5957350e+02, &
& 2.0012557e+03,1.8302684e+03,1.6035452e+03,7.2568940e+02,1.6196684e+03, &
& 1.5546376e+03,1.5154611e+03,1.4794584e+03,1.4475288e+03,1.1341326e+03, &
& 1.2823779e+03,1.2360668e+03,1.3047191e+03,1.2769567e+03,1.2518739e+03, &
& 1.2374305e+03,1.0399691e+03,1.0199809e+03,9.2884540e+02,7.8260680e+02, &
& 7.9560020e+02,7.1959790e+02,6.5787990e+02,5.4519970e+02,5.0886690e+02, &
& 5.2322550e+02,7.6879970e+02,7.4998940e+02,6.8706370e+02,6.5405030e+02, &
& 6.0215770e+02,5.5060100e+02,1.8855867e+03,1.8071153e+03,1.5930087e+03, &
& 1.4265400e+03,1.4182359e+03,1.3730549e+03,1.4164457e+03,1.3711890e+03, &
& 7.5839400e+01,2.4848670e+02,3.1648150e+02,2.4013610e+02,1.8003260e+02, &
& 1.2441240e+02,9.0093200e+01,6.1449900e+01,3.6346120e+02,5.6268080e+02, &
& 5.6814570e+02,4.5421760e+02,3.7038990e+02,3.1215000e+02,2.5418510e+02, &
& 4.9920670e+02,9.6226620e+02,4.9762570e+02,4.8020280e+02,4.7059030e+02, &
& 4.6642410e+02,4.2818050e+02,3.9589450e+02,3.7758950e+02,3.6919520e+02, &
& 3.6539900e+02,3.4173160e+02,5.6276860e+02,4.9236040e+02,4.3983840e+02, &
& 4.0063200e+02,3.5185020e+02,5.9499260e+02,1.1711205e+03,8.8995010e+02]
 vdw_dftd3_c6(24001:24100)=[&
& 6.5820980e+02,6.6473460e+02,6.1909740e+02,6.9637520e+02,5.3817200e+02, &
& 5.0319100e+02,4.6688550e+02,4.5174170e+02,4.4617650e+02,7.1406170e+02, &
& 6.4381770e+02,6.0143340e+02,5.6786370e+02,5.1738510e+02,7.0211480e+02, &
& 1.5197933e+03,1.1269136e+03,7.0363390e+02,6.8904640e+02,6.6706290e+02, &
& 6.7068630e+02,6.4747300e+02,6.7511960e+02,6.3375280e+02,6.4544130e+02, &
& 6.0128460e+02,5.8404680e+02,5.8039470e+02,6.1065540e+02,5.6086530e+02, &
& 7.6459540e+02,7.0805680e+02,6.4452930e+02,6.5336840e+02,5.6893970e+02, &
& 5.3465980e+02,5.1029930e+02,4.8773540e+02,4.7044860e+02,7.6580950e+02, &
& 6.9566310e+02,6.7341930e+02,6.5403890e+02,6.0946360e+02,7.9062920e+02, &
& 1.5089609e+03,8.3278590e+02,9.6389140e+02,8.6055570e+02,7.5868030e+02, &
& 7.2961370e+02,8.8320780e+02,2.0508500e+02,2.0120830e+02,1.4633300e+02, &
& 1.1668030e+02,7.9287700e+01,3.6491620e+02,4.4120040e+02,4.2212020e+02, &
& 3.6544980e+02,3.0579960e+02,5.1130630e+02,4.9031810e+02,4.9624460e+02, &
& 4.5408320e+02,3.4051320e+02,2.9056100e+02,2.8526770e+02,3.3581530e+02, &
& 3.1248780e+02,4.4415730e+02,4.6838860e+02,4.2809060e+02,3.9744450e+02, &
& 6.1229940e+02,5.8772220e+02,5.9223960e+02,5.7268150e+02,5.0489560e+02, &
& 4.4457570e+02,4.1969160e+02,4.2842590e+02,4.4792670e+02,5.7208230e+02, &
& 6.2193900e+02,5.8452180e+02,5.6311340e+02,7.3991600e+02,7.6602020e+02, &
& 3.5219500e+01,2.3410600e+01,5.2531240e+02,3.0984670e+02,2.1084170e+02, &
& 1.4342000e+02,1.0074190e+02,7.6484200e+01,5.8056700e+01,4.4712000e+01]
 vdw_dftd3_c6(24101:24200)=[&
& 6.2893080e+02,4.9188460e+02,4.5602900e+02,3.6194550e+02,2.8382390e+02, &
& 2.3632710e+02,1.9360200e+02,1.5874630e+02,1.0274537e+03,8.5873510e+02, &
& 7.1140130e+02,6.8866520e+02,6.3154370e+02,4.9757540e+02,5.4486260e+02, &
& 4.2780560e+02,4.5484270e+02,4.6785600e+02,3.5868500e+02,3.6992010e+02, &
& 4.3768410e+02,3.8810850e+02,3.3255870e+02,2.9919670e+02,2.6248890e+02, &
& 2.2875780e+02,1.1528150e+03,1.0226698e+03,9.0010840e+02,8.1130330e+02, &
& 7.4116010e+02,5.7392370e+02,6.3963880e+02,4.8887360e+02,5.3438470e+02, &
& 4.9607390e+02,4.1327390e+02,4.3768030e+02,5.4743800e+02,5.0847990e+02, &
& 4.5499670e+02,4.2313470e+02,3.8349110e+02,3.4549990e+02,1.4048971e+03, &
& 1.3017488e+03,1.1490950e+03,5.3712000e+02,1.1550423e+03,1.1099829e+03, &
& 1.0823751e+03,1.0569604e+03,1.0344333e+03,8.1761230e+02,9.1364540e+02, &
& 8.8198390e+02,9.3409640e+02,9.1438960e+02,8.9667750e+02,8.8600250e+02, &
& 7.4911050e+02,7.4025210e+02,6.7737500e+02,5.7310850e+02,5.8363200e+02, &
& 5.3009450e+02,4.8629160e+02,4.0460200e+02,3.7824260e+02,3.8943520e+02, &
& 5.6259750e+02,5.5195500e+02,5.0903810e+02,4.8647930e+02,4.5000480e+02, &
& 4.1326650e+02,1.3317058e+03,1.2904127e+03,1.1454809e+03,1.0342317e+03, &
& 1.0243603e+03,9.9194790e+02,1.0182141e+03,9.8657360e+02,5.6451300e+01, &
& 1.8035670e+02,2.3052060e+02,1.7769180e+02,1.3451000e+02,9.3956900e+01, &
& 6.8655500e+01,4.7370200e+01,2.6307010e+02,4.0658200e+02,4.1382490e+02, &
& 3.3484900e+02,2.7554270e+02,2.3369990e+02,1.9157960e+02,3.6408150e+02]
 vdw_dftd3_c6(24201:24300)=[&
& 6.8723560e+02,3.6630740e+02,3.5366860e+02,3.4654410e+02,3.4308700e+02, &
& 3.1671570e+02,2.9336460e+02,2.7983390e+02,2.7349300e+02,2.6953150e+02, &
& 2.5402930e+02,4.1165060e+02,3.6351880e+02,3.2699810e+02,2.9929580e+02, &
& 2.6427820e+02,4.3494430e+02,8.3572700e+02,6.4479280e+02,4.8425820e+02, &
& 4.8896320e+02,4.5641490e+02,5.0966540e+02,3.9898040e+02,3.7331410e+02, &
& 3.4680080e+02,3.3529990e+02,3.3249420e+02,5.2263660e+02,4.7495740e+02, &
& 4.4616040e+02,4.2289020e+02,3.8708050e+02,5.1625270e+02,1.0815338e+03, &
& 8.1548330e+02,5.2103820e+02,5.1028040e+02,4.9420110e+02,4.9632490e+02, &
& 4.7741750e+02,4.9938720e+02,4.6926950e+02,4.7695560e+02,4.4565950e+02, &
& 4.3304490e+02,4.3020330e+02,4.5184050e+02,4.1595250e+02,5.5954550e+02, &
& 5.2058420e+02,4.7617010e+02,4.8071520e+02,4.2322750e+02,3.9843450e+02, &
& 3.8070180e+02,3.6388290e+02,3.5232230e+02,5.6157040e+02,5.1406470e+02, &
& 4.9968080e+02,4.8675350e+02,4.5547580e+02,5.8159320e+02,1.0790444e+03, &
& 6.1650660e+02,7.0890840e+02,6.3454270e+02,5.6236500e+02,5.4147420e+02, &
& 6.4630030e+02,1.5187100e+02,1.4945570e+02,1.0995340e+02,8.8233300e+01, &
& 6.0592700e+01,2.6815940e+02,3.2387400e+02,3.1174680e+02,2.7170710e+02, &
& 2.2898300e+02,3.7583520e+02,3.6214280e+02,3.6666280e+02,3.3547490e+02, &
& 2.5363880e+02,2.1751740e+02,2.1343980e+02,2.4927270e+02,2.3249050e+02, &
& 3.2767280e+02,3.4636060e+02,3.1855200e+02,2.9696280e+02,4.5040960e+02, &
& 4.3502220e+02,4.3916880e+02,4.2451400e+02,3.7628530e+02,3.3276620e+02]
 vdw_dftd3_c6(24301:24400)=[&
& 3.1465980e+02,3.1973310e+02,3.3378790e+02,4.2275520e+02,4.5955200e+02, &
& 4.3419590e+02,4.1956530e+02,5.4526090e+02,5.6623920e+02,4.2097660e+02, &
& 3.6595100e+01,2.4281400e+01,5.6610590e+02,3.2644690e+02,2.2041970e+02, &
& 1.4932900e+02,1.0467690e+02,7.9396800e+01,6.0239300e+01,4.6387800e+01, &
& 6.7687130e+02,5.1998460e+02,4.7979410e+02,3.7862690e+02,2.9580530e+02, &
& 2.4586750e+02,2.0113960e+02,1.6478050e+02,1.1126264e+03,9.1447030e+02, &
& 7.5525190e+02,7.2956490e+02,6.6815990e+02,5.2676740e+02,5.7538750e+02, &
& 4.5195440e+02,4.7878370e+02,4.9306650e+02,3.7838780e+02,3.8837940e+02, &
& 4.5998760e+02,4.0600970e+02,3.4676900e+02,3.1149230e+02,2.7290380e+02, &
& 2.3759510e+02,1.2473944e+03,1.0901054e+03,9.5467960e+02,8.5810820e+02, &
& 7.8268490e+02,6.0471660e+02,6.7451160e+02,5.1426690e+02,5.6175990e+02, &
& 5.2100450e+02,4.3478170e+02,4.5916630e+02,5.7597200e+02,5.3273890e+02, &
& 4.7516010e+02,4.4117360e+02,3.9923870e+02,3.5925420e+02,1.5209786e+03, &
& 1.3908237e+03,1.2212713e+03,5.6144200e+02,1.2321348e+03,1.1827756e+03, &
& 1.1530265e+03,1.1256719e+03,1.1014151e+03,8.6643530e+02,9.7722080e+02, &
& 9.4263090e+02,9.9314900e+02,9.7201510e+02,9.5296410e+02,9.4178690e+02, &
& 7.9383690e+02,7.8053540e+02,7.1253240e+02,6.0220240e+02,6.1258210e+02, &
& 5.5538920e+02,5.0882230e+02,4.2301850e+02,3.9532750e+02,4.0661330e+02, &
& 5.9190940e+02,5.7859300e+02,5.3193140e+02,5.0757870e+02,4.6877360e+02, &
& 4.2997720e+02,1.4348337e+03,1.3751465e+03,1.2148259e+03,1.0917991e+03]
 vdw_dftd3_c6(24401:24500)=[&
& 1.0843209e+03,1.0498792e+03,1.0808128e+03,1.0465858e+03,5.8803800e+01, &
& 1.8986110e+02,2.4244290e+02,1.8557490e+02,1.4012490e+02,9.7656800e+01, &
& 7.1264400e+01,4.9121900e+01,2.7753610e+02,4.2932400e+02,4.3497780e+02, &
& 3.5008490e+02,2.8716960e+02,2.4316080e+02,1.9905370e+02,3.8351900e+02, &
& 7.3299270e+02,3.8359320e+02,3.7045440e+02,3.6303280e+02,3.5964520e+02, &
& 3.3100010e+02,3.0643150e+02,2.9235680e+02,2.8579970e+02,2.8225140e+02, &
& 2.6501660e+02,4.3212490e+02,3.7997370e+02,3.4092760e+02,3.1161210e+02, &
& 2.7478910e+02,4.5784580e+02,8.9253360e+02,6.8212820e+02,5.0766120e+02, &
& 5.1256950e+02,4.7819280e+02,5.3626360e+02,4.1699040e+02,3.9019030e+02, &
& 3.6240560e+02,3.5051370e+02,3.4680430e+02,5.4909790e+02,4.9700410e+02, &
& 4.6575860e+02,4.4087250e+02,4.0297870e+02,5.4168560e+02,1.1586256e+03, &
& 8.6372770e+02,5.4454800e+02,5.3328550e+02,5.1641630e+02,5.1893410e+02, &
& 5.0005910e+02,5.2212860e+02,4.9044870e+02,4.9903390e+02,4.6555820e+02, &
& 4.5230800e+02,4.4938570e+02,4.7221050e+02,4.3437410e+02,5.8833640e+02, &
& 5.4616030e+02,4.9848100e+02,5.0425710e+02,4.4178790e+02,4.1572770e+02, &
& 3.9714540e+02,3.7972750e+02,3.6701200e+02,5.9000540e+02,5.3789130e+02, &
& 5.2181620e+02,5.0773170e+02,4.7444690e+02,6.1006800e+02,1.1516502e+03, &
& 6.4431070e+02,7.4392250e+02,6.6568850e+02,5.8786120e+02,5.6570590e+02, &
& 6.8049130e+02,1.5844380e+02,1.5595490e+02,1.1435140e+02,9.1693500e+01, &
& 6.2899100e+01,2.8061570e+02,3.3928390e+02,3.2570670e+02,2.8327190e+02]
 vdw_dftd3_c6(24501:24600)=[&
& 2.3825580e+02,3.9378620e+02,3.7862350e+02,3.8330360e+02,3.5090960e+02, &
& 2.6469190e+02,2.2664920e+02,2.2244630e+02,2.6044220e+02,2.4268500e+02, &
& 3.4272530e+02,3.6182680e+02,3.3202940e+02,3.0917670e+02,4.7200610e+02, &
& 4.5450240e+02,4.5850240e+02,4.4354250e+02,3.9244480e+02,3.4662740e+02, &
& 3.2763810e+02,3.3357090e+02,3.4831840e+02,4.4228570e+02,4.8056320e+02, &
& 4.5304590e+02,4.3732780e+02,5.7116140e+02,5.9193970e+02,4.3911410e+02, &
& 4.5876770e+02,3.4435200e+01,2.3230100e+01,4.9948070e+02,2.9673610e+02, &
& 2.0361200e+02,1.3967130e+02,9.8854100e+01,7.5508800e+01,5.7643800e+01, &
& 4.4614500e+01,5.9853480e+02,4.7040460e+02,4.3759830e+02,3.4916910e+02, &
& 2.7538760e+02,2.3036660e+02,1.8964180e+02,1.5623470e+02,9.7919910e+02, &
& 8.2027660e+02,6.8015970e+02,6.5948500e+02,6.0531050e+02,4.7771160e+02, &
& 5.2292780e+02,4.1137250e+02,4.3743830e+02,4.4947760e+02,3.4533400e+02, &
& 3.5654050e+02,4.2115000e+02,3.7485400e+02,3.2264010e+02,2.9126630e+02, &
& 2.5651700e+02,2.2442600e+02,1.0997481e+03,9.7735030e+02,8.6199720e+02, &
& 7.7824330e+02,7.1194490e+02,5.5309640e+02,6.1562580e+02,4.7221430e+02, &
& 5.1556800e+02,4.7905900e+02,3.9972620e+02,4.2322300e+02,5.2765140e+02, &
& 4.9129650e+02,4.4110510e+02,4.1123900e+02,3.7383140e+02,3.3786400e+02, &
& 1.3409193e+03,1.2440756e+03,1.1002138e+03,5.2077890e+02,1.1048675e+03, &
& 1.0619928e+03,1.0356282e+03,1.0113417e+03,9.8981510e+02,7.8493070e+02, &
& 8.7500190e+02,8.4502380e+02,8.9410530e+02,8.7523480e+02,8.5830520e+02]
 vdw_dftd3_c6(24601:24700)=[&
& 8.4794180e+02,7.1850660e+02,7.1139240e+02,6.5227240e+02,5.5338510e+02, &
& 5.6376710e+02,5.1307710e+02,4.7151500e+02,3.9347970e+02,3.6827210e+02, &
& 3.7916720e+02,5.4357710e+02,5.3401680e+02,4.9381190e+02,4.7281700e+02, &
& 4.3850630e+02,4.0379910e+02,1.2730029e+03,1.2347444e+03,1.0979052e+03, &
& 9.9418690e+02,9.8402510e+02,9.5302030e+02,9.7674380e+02,9.4660700e+02, &
& 5.4793400e+01,1.7298830e+02,2.2155260e+02,1.7203160e+02,1.3107440e+02, &
& 9.2276100e+01,6.7911900e+01,4.7308600e+01,2.5228990e+02,3.8957210e+02, &
& 3.9754110e+02,3.2338900e+02,2.6745480e+02,2.2781180e+02,1.8766470e+02, &
& 3.5118760e+02,6.5780950e+02,3.5408760e+02,3.4206260e+02,3.3519530e+02, &
& 3.3174290e+02,3.0679020e+02,2.8447550e+02,2.7144430e+02,2.6525960e+02, &
& 2.6103120e+02,2.4669600e+02,3.9651060e+02,3.5152430e+02,3.1735850e+02, &
& 2.9136250e+02,2.5822570e+02,4.2013560e+02,8.0028360e+02,6.2027440e+02, &
& 4.6837410e+02,4.7294180e+02,4.4204910e+02,4.9245940e+02,3.8741710e+02, &
& 3.6276960e+02,3.3732470e+02,3.2605860e+02,3.2365690e+02,5.0425970e+02, &
& 4.5956860e+02,4.3276910e+02,4.1105850e+02,3.7729830e+02,4.9958850e+02, &
& 1.0355821e+03,7.8441150e+02,5.0532050e+02,4.9491310e+02,4.7943010e+02, &
& 4.8126120e+02,4.6237370e+02,4.8403110e+02,4.5510020e+02,4.6220710e+02, &
& 4.3236450e+02,4.2019360e+02,4.1735840e+02,4.3783510e+02,4.0359860e+02, &
& 5.4010000e+02,5.0356000e+02,4.6165670e+02,4.6529970e+02,4.1168040e+02, &
& 3.8805470e+02,3.7109900e+02,3.5487160e+02,3.4408260e+02,5.4282280e+02]
 vdw_dftd3_c6(24701:24800)=[&
& 4.9822260e+02,4.8504830e+02,4.7318900e+02,4.4381290e+02,5.6285700e+02, &
& 1.0344519e+03,5.9772660e+02,6.8593330e+02,6.1504850e+02,5.4603460e+02, &
& 5.2604590e+02,6.2473940e+02,1.4693770e+02,1.4506890e+02,1.0751640e+02, &
& 8.6744300e+01,6.0101300e+01,2.5846390e+02,3.1219790e+02,3.0136080e+02, &
& 2.6370240e+02,2.2325700e+02,3.6300510e+02,3.5046740e+02,3.5493010e+02, &
& 3.2491900e+02,2.4698040e+02,2.1243170e+02,2.0838910e+02,2.4221550e+02, &
& 2.2617060e+02,3.1684610e+02,3.3520830e+02,3.0932650e+02,2.8913180e+02, &
& 4.3547330e+02,4.2156130e+02,4.2595070e+02,4.1195670e+02,3.6633030e+02, &
& 3.2483190e+02,3.0750010e+02,3.1180890e+02,3.2514160e+02,4.0961940e+02, &
& 4.4499610e+02,4.2145120e+02,4.0793570e+02,5.2780130e+02,5.4842160e+02, &
& 4.0887060e+02,4.2635470e+02,3.9809730e+02,3.1592800e+01,2.1621000e+01, &
& 4.3757960e+02,2.6483780e+02,1.8401940e+02,1.2752710e+02,9.0995800e+01, &
& 6.9928900e+01,5.3673800e+01,4.1731000e+01,5.2514830e+02,4.1847680e+02, &
& 3.9167000e+02,3.1514880e+02,2.5044750e+02,2.1061910e+02,1.7429820e+02, &
& 1.4428140e+02,8.5835650e+02,7.2611640e+02,6.0356880e+02,5.8685490e+02, &
& 5.3950230e+02,4.2636910e+02,4.6716530e+02,3.6813400e+02,3.9225490e+02, &
& 4.0236920e+02,3.0964490e+02,3.2085180e+02,3.7820130e+02,3.3870950e+02, &
& 2.9330960e+02,2.6587000e+02,2.3516470e+02,2.0659450e+02,9.6546080e+02, &
& 8.6521550e+02,7.6664930e+02,6.9431570e+02,6.3659010e+02,4.9673190e+02, &
& 5.5196750e+02,4.2541540e+02,4.6408940e+02,4.3183300e+02,3.6056290e+02]
 vdw_dftd3_c6(24801:24900)=[&
& 3.8221490e+02,4.7434870e+02,4.4370850e+02,4.0039260e+02,3.7450420e+02, &
& 3.4168440e+02,3.0992280e+02,1.1778622e+03,1.1002149e+03,9.7735910e+02, &
& 4.7249460e+02,9.7880490e+02,9.4144390e+02,9.1823780e+02,8.9683560e+02, &
& 8.7787040e+02,7.0022530e+02,7.7537110e+02,7.4950170e+02,7.9380070e+02, &
& 7.7710890e+02,7.6218580e+02,7.5278270e+02,6.4040610e+02,6.3688250e+02, &
& 5.8584610e+02,4.9867750e+02,5.0852330e+02,4.6413690e+02,4.2756890e+02, &
& 3.5797140e+02,3.3546850e+02,3.4558560e+02,4.8985300e+02,4.8277210e+02, &
& 4.4834220e+02,4.3041860e+02,4.0052020e+02,3.7000240e+02,1.1223371e+03, &
& 1.0947171e+03,9.7740400e+02,8.8977500e+02,8.7887760e+02,8.5133500e+02, &
& 8.6982820e+02,8.4344130e+02,4.9861600e+01,1.5470980e+02,1.9867640e+02, &
& 1.5594320e+02,1.1971100e+02,8.4990600e+01,6.3003200e+01,4.4293300e+01, &
& 2.2533300e+02,3.4751630e+02,3.5639850e+02,2.9229100e+02,2.4332340e+02, &
& 2.0827540e+02,1.7247660e+02,3.1572770e+02,5.8337200e+02,3.1998810e+02, &
& 3.0928520e+02,3.0306940e+02,2.9974820e+02,2.7813060e+02,2.5824560e+02, &
& 2.4647110e+02,2.4079380e+02,2.3633890e+02,2.2442270e+02,3.5665940e+02, &
& 3.1814490e+02,2.8862680e+02,2.6594970e+02,2.3668530e+02,3.7838750e+02, &
& 7.0963970e+02,5.5520870e+02,4.2330460e+02,4.2740240e+02,4.0015670e+02, &
& 4.4382790e+02,3.5202000e+02,3.2985620e+02,3.0703010e+02,2.9664300e+02, &
& 2.9510960e+02,4.5411220e+02,4.1592040e+02,3.9310920e+02,3.7441140e+02, &
& 3.4482440e+02,4.5152920e+02,9.1708910e+02,7.0171880e+02,4.5862530e+02]
 vdw_dftd3_c6(24901:25000)=[&
& 4.4920880e+02,4.3528510e+02,4.3662810e+02,4.1857700e+02,4.3896330e+02, &
& 4.1303190e+02,4.1895270e+02,3.9263770e+02,3.8167800e+02,3.7901540e+02, &
& 3.9706740e+02,3.6663560e+02,4.8647620e+02,4.5499830e+02,4.1851340e+02, &
& 4.2070230e+02,3.7494930e+02,3.5392940e+02,3.3877130e+02,3.2404200e+02, &
& 3.1491790e+02,4.8972910e+02,4.5162020e+02,4.4081790e+02,4.3091310e+02, &
& 4.0535970e+02,5.0882630e+02,9.1874100e+02,5.4234140e+02,6.1996910e+02, &
& 5.5704040e+02,4.9609310e+02,4.7831910e+02,5.6313100e+02,1.3319340e+02, &
& 1.3189570e+02,9.8601900e+01,7.9979000e+01,5.5894600e+01,2.3303610e+02, &
& 2.8137080e+02,2.7272890e+02,2.3983560e+02,2.0414550e+02,3.2770190e+02, &
& 3.1737170e+02,3.2150860e+02,2.9439580e+02,2.2519270e+02,1.9439360e+02, &
& 1.9062290e+02,2.2028000e+02,2.0601270e+02,2.8665620e+02,3.0371450e+02, &
& 2.8150840e+02,2.6395060e+02,3.9346440e+02,3.8237800e+02,3.8683830e+02, &
& 3.7417790e+02,3.3405520e+02,2.9716160e+02,2.8166080e+02,2.8475480e+02, &
& 2.9656750e+02,3.7125760e+02,4.0316950e+02,3.8318380e+02,3.7169830e+02, &
& 4.7755530e+02,4.9702050e+02,3.7200450e+02,3.8748710e+02,3.6309850e+02, &
& 3.3214480e+02,2.9959700e+01,2.0732100e+01,3.9946230e+02,2.4560490e+02, &
& 1.7242600e+02,1.2047520e+02,8.6515000e+01,6.6798300e+01,5.1483900e+01, &
& 4.0166600e+01,4.8002060e+02,3.8703270e+02,3.6408530e+02,2.9497480e+02, &
& 2.3585440e+02,1.9918590e+02,1.6551840e+02,1.3752330e+02,7.8398180e+02, &
& 6.6874580e+02,5.5703780e+02,5.4287970e+02,4.9973800e+02,3.9539650e+02]
 vdw_dftd3_c6(25001:25100)=[&
& 4.3357460e+02,3.4214260e+02,3.6517850e+02,3.7407150e+02,2.8825030e+02, &
& 2.9957880e+02,3.5252400e+02,3.1731180e+02,2.7613370e+02,2.5111640e+02, &
& 2.2287460e+02,1.9643010e+02,8.8292900e+02,7.9689500e+02,7.0887790e+02, &
& 6.4366450e+02,5.9124780e+02,4.6302150e+02,5.1380610e+02,3.9755570e+02, &
& 4.3340920e+02,4.0374870e+02,3.3729140e+02,3.5790130e+02,4.4252210e+02, &
& 4.1550540e+02,3.7647850e+02,3.5306060e+02,3.2305850e+02,2.9386160e+02, &
& 1.0777158e+03,1.0124643e+03,9.0281190e+02,4.4410240e+02,9.0205950e+02, &
& 8.6810970e+02,8.4683980e+02,8.2720360e+02,8.0980750e+02,6.4909210e+02, &
& 7.1473580e+02,6.9143160e+02,7.3288920e+02,7.1752640e+02,7.0383170e+02, &
& 6.9499350e+02,5.9320990e+02,5.9212750e+02,5.4613100e+02,4.6612960e+02, &
& 4.7570880e+02,4.3520440e+02,4.0169710e+02,3.3718850e+02,3.1631490e+02, &
& 3.2600310e+02,4.5788770e+02,4.5244550e+02,4.2163980e+02,4.0564640e+02, &
& 3.7847640e+02,3.5052560e+02,1.0301431e+03,1.0095485e+03,9.0447790e+02, &
& 8.2702680e+02,8.1550490e+02,7.9006160e+02,8.0513870e+02,7.8106340e+02, &
& 4.6979300e+01,1.4371930e+02,1.8497380e+02,1.4647120e+02,1.1311480e+02, &
& 8.0841400e+01,6.0262600e+01,4.2663600e+01,2.0909190e+02,3.2213980e+02, &
& 3.3174520e+02,2.7389260e+02,2.2921310e+02,1.9696310e+02,1.6378500e+02, &
& 2.9455150e+02,5.3811810e+02,2.9979870e+02,2.8989590e+02,2.8406410e+02, &
& 2.8079860e+02,2.6125710e+02,2.4283990e+02,2.3180890e+02,2.2642260e+02, &
& 2.2176420e+02,2.1139390e+02,3.3289310e+02,2.9843870e+02,2.7181450e+02]
 vdw_dftd3_c6(25101:25200)=[&
& 2.5118530e+02,2.2428140e+02,3.5352250e+02,6.5452330e+02,5.1611830e+02, &
& 3.9662190e+02,4.0043580e+02,3.7541960e+02,4.1489900e+02,3.3125900e+02, &
& 3.1057580e+02,2.8931700e+02,2.7942790e+02,2.7848020e+02,4.2425050e+02, &
& 3.9014410e+02,3.6984470e+02,3.5302740e+02,3.2600590e+02,4.2307320e+02, &
& 8.4493410e+02,6.5199750e+02,4.3118430e+02,4.2235310e+02,4.0935960e+02, &
& 4.1038090e+02,3.9271700e+02,4.1244200e+02,3.8830810e+02,3.9347340e+02, &
& 3.6931690e+02,3.5907870e+02,3.5650780e+02,3.7307840e+02,3.4495150e+02, &
& 4.5455470e+02,4.2623560e+02,3.9310340e+02,3.9431360e+02,3.5350380e+02, &
& 3.3405950e+02,3.1998180e+02,3.0613090e+02,2.9806110e+02,4.5819840e+02, &
& 4.2417020e+02,4.1489180e+02,4.0623070e+02,3.8304430e+02,4.7684810e+02, &
& 8.4851510e+02,5.0977850e+02,5.8092120e+02,5.2282120e+02,4.6679740e+02, &
& 4.5036480e+02,5.2648950e+02,1.2510340e+02,1.2418210e+02,9.3473800e+01, &
& 7.6136500e+01,5.3564100e+01,2.1793200e+02,2.6304720e+02,2.5582610e+02, &
& 2.2587090e+02,1.9308290e+02,3.0676340e+02,2.9784850e+02,3.0180320e+02, &
& 2.7640530e+02,2.1249880e+02,1.8396380e+02,1.8034190e+02,2.0743670e+02, &
& 1.9424620e+02,2.6882090e+02,2.8515730e+02,2.6525130e+02,2.4932490e+02, &
& 3.6858100e+02,3.5932890e+02,3.6388390e+02,3.5200850e+02,3.1525780e+02, &
& 2.8115270e+02,2.6675220e+02,2.6904190e+02,2.7993090e+02,3.4865850e+02, &
& 3.7851600e+02,3.6077560e+02,3.5056820e+02,4.4786270e+02,4.6673680e+02, &
& 3.5044320e+02,3.6470460e+02,3.4272390e+02,3.1423420e+02,2.9783380e+02]
 vdw_dftd3_c6(25201:25300)=[&
& 3.0229400e+01,2.0760800e+01,4.2636380e+02,2.5474750e+02,1.7635010e+02, &
& 1.2210880e+02,8.7190700e+01,6.7091300e+01,5.1585000e+01,4.0184600e+01, &
& 5.1143160e+02,4.0339630e+02,3.7653390e+02,3.0209560e+02,2.3974340e+02, &
& 2.0158550e+02,1.6687220e+02,1.3823710e+02,8.3884470e+02,7.0306100e+02, &
& 5.8340320e+02,5.6663000e+02,5.2055850e+02,4.1171570e+02,4.5035590e+02, &
& 3.5513960e+02,3.7753770e+02,3.8750230e+02,2.9855250e+02,3.0844330e+02, &
& 3.6354100e+02,3.2481010e+02,2.8090060e+02,2.5453380e+02,2.2513410e+02, &
& 1.9784740e+02,9.4312750e+02,8.3825760e+02,7.4070150e+02,6.6984280e+02, &
& 6.1369150e+02,4.7850440e+02,5.3185130e+02,4.0961050e+02,4.4656770e+02, &
& 4.1538870e+02,3.4736190e+02,3.6754320e+02,4.5659440e+02,4.2612630e+02, &
& 3.8393930e+02,3.5889370e+02,3.2731610e+02,2.9686250e+02,1.1506653e+03, &
& 1.0673048e+03,9.4534330e+02,4.5335410e+02,9.4886600e+02,9.1214940e+02, &
& 8.8952970e+02,8.6867840e+02,8.5019490e+02,6.7650560e+02,7.5286050e+02, &
& 7.2736460e+02,7.6817290e+02,7.5193870e+02,7.3739820e+02,7.2836400e+02, &
& 6.1861380e+02,6.1342840e+02,5.6363530e+02,4.7969460e+02,4.8887240e+02, &
& 4.4589270e+02,4.1058900e+02,3.4385250e+02,3.2228400e+02,3.3178240e+02, &
& 4.7174830e+02,4.6398210e+02,4.3022560e+02,4.1274680e+02,3.8386920e+02, &
& 3.5453980e+02,1.0936353e+03,1.0604655e+03,9.4439170e+02,8.5775700e+02, &
& 8.4861880e+02,8.2201600e+02,8.4127950e+02,8.1549360e+02,4.7710700e+01, &
& 1.4877410e+02,1.9095480e+02,1.4941700e+02,1.1467490e+02,8.1468600e+01]
 vdw_dftd3_c6(25301:25400)=[&
& 6.0474900e+01,4.2633000e+01,2.1706070e+02,3.3478560e+02,3.4247320e+02, &
& 2.8013350e+02,2.3294100e+02,1.9936180e+02,1.6514130e+02,3.0413400e+02, &
& 5.6534700e+02,3.0720500e+02,2.9699590e+02,2.9107410e+02,2.8800240e+02, &
& 2.6680930e+02,2.4770870e+02,2.3646980e+02,2.3106240e+02,2.2706590e+02, &
& 2.1517070e+02,3.4264070e+02,3.0498940e+02,2.7641200e+02,2.5462020e+02, &
& 2.2659700e+02,3.6441220e+02,6.8823470e+02,5.3570340e+02,4.0665130e+02, &
& 4.1063670e+02,3.8441940e+02,4.2730450e+02,3.3787470e+02,3.1668990e+02, &
& 2.9482600e+02,2.8494240e+02,2.8309650e+02,4.3680210e+02,3.9922460e+02, &
& 3.7689830e+02,3.5878610e+02,3.3031590e+02,4.3396820e+02,8.9082740e+02, &
& 6.7744550e+02,4.4003130e+02,4.3099700e+02,4.1762690e+02,4.1901960e+02, &
& 4.0212590e+02,4.2122030e+02,3.9630640e+02,4.0219400e+02,3.7665190e+02, &
& 3.6611320e+02,3.6356690e+02,3.8089510e+02,3.5163250e+02,4.6799190e+02, &
& 4.3730900e+02,4.0191040e+02,4.0442320e+02,3.5970450e+02,3.3957300e+02, &
& 3.2507870e+02,3.1108440e+02,3.0206450e+02,4.7125110e+02,4.3367490e+02, &
& 4.2287330e+02,4.1315870e+02,3.8847900e+02,4.8893950e+02,8.9067630e+02, &
& 5.2029480e+02,5.9599730e+02,5.3552370e+02,4.7621600e+02,4.5907020e+02, &
& 5.4251670e+02,1.2750670e+02,1.2636580e+02,9.4442200e+01,7.6693600e+01, &
& 5.3713500e+01,2.2345060e+02,2.6992960e+02,2.6131790e+02,2.2965740e+02, &
& 1.9542290e+02,3.1465860e+02,3.0440560e+02,3.0837100e+02,2.8252780e+02, &
& 2.1607310e+02,1.8647130e+02,1.8288010e+02,2.1145410e+02,1.9770250e+02]
 vdw_dftd3_c6(25401:25500)=[&
& 2.7493230e+02,2.9109370e+02,2.6957470e+02,2.5271240e+02,3.7790170e+02, &
& 3.6667450e+02,3.7083430e+02,3.5892720e+02,3.2031090e+02,2.8489430e+02, &
& 2.7005380e+02,2.7328420e+02,2.8460940e+02,3.5643230e+02,3.8687350e+02, &
& 3.6731080e+02,3.5616120e+02,4.5858100e+02,4.7669270e+02,3.5651190e+02, &
& 3.7170810e+02,3.4812240e+02,3.1841110e+02,3.0121160e+02,3.0552580e+02, &
& 3.1736800e+01,2.1706400e+01,4.4817370e+02,2.6864910e+02,1.8571950e+02, &
& 1.2832810e+02,9.1444100e+01,7.0247200e+01,5.3929300e+01,4.1957400e+01, &
& 5.3764280e+02,4.2531660e+02,3.9691230e+02,3.1822850e+02,2.5221940e+02, &
& 2.1181210e+02,1.7510170e+02,1.4486250e+02,8.7976340e+02,7.4019810e+02, &
& 6.1451280e+02,5.9680090e+02,5.4828530e+02,4.3331070e+02,4.7433180e+02, &
& 3.7375740e+02,3.9765430e+02,4.0819710e+02,3.1418780e+02,3.2482660e+02, &
& 3.8293500e+02,3.4202100e+02,2.9551180e+02,2.6753780e+02,2.3639130e+02, &
& 2.0751570e+02,9.8897460e+02,8.8212520e+02,7.7987150e+02,7.0532520e+02, &
& 6.4612000e+02,5.0347710e+02,5.5975880e+02,4.3081480e+02,4.6996810e+02, &
& 4.3711170e+02,3.6518920e+02,3.8669930e+02,4.8058480e+02,4.4855300e+02, &
& 4.0393700e+02,3.7737610e+02,3.4391650e+02,3.1165760e+02,1.2060907e+03, &
& 1.1224006e+03,9.9484110e+02,4.7685250e+02,9.9799760e+02,9.5958830e+02, &
& 9.3585100e+02,9.1396730e+02,8.9457000e+02,7.1180730e+02,7.9087860e+02, &
& 7.6413790e+02,8.0848440e+02,7.9144150e+02,7.7617700e+02,7.6668210e+02, &
& 6.5113390e+02,6.4597680e+02,5.9345200e+02,5.0469500e+02,5.1441860e+02]
 vdw_dftd3_c6(25501:25600)=[&
& 4.6906170e+02,4.3179110e+02,3.6131080e+02,3.3854780e+02,3.4860350e+02, &
& 4.9610630e+02,4.8812730e+02,4.5249450e+02,4.3395750e+02,4.0335980e+02, &
& 3.7228610e+02,1.1471809e+03,1.1154572e+03,9.9404090e+02,9.0293930e+02, &
& 8.9295040e+02,8.6495490e+02,8.8511780e+02,8.5805490e+02,5.0189900e+01, &
& 1.5683270e+02,2.0116870e+02,1.5724660e+02,1.2047330e+02,8.5411300e+01, &
& 6.3280300e+01,4.4500400e+01,2.2873580e+02,3.5278900e+02,3.6093540e+02, &
& 2.9500710e+02,2.4503090e+02,2.0947030e+02,1.7328290e+02,3.1991700e+02, &
& 5.9442400e+02,3.2334130e+02,3.1251320e+02,3.0627330e+02,3.0303740e+02, &
& 2.8075180e+02,2.6058650e+02,2.4872640e+02,2.4304070e+02,2.3886500e+02, &
& 2.2630430e+02,3.6087970e+02,3.2106640e+02,2.9076020e+02,2.6762610e+02, &
& 2.3793600e+02,3.8318340e+02,7.2324770e+02,5.6337460e+02,4.2778570e+02, &
& 4.3199420e+02,4.0427790e+02,4.4929790e+02,3.5522280e+02,3.3286530e+02, &
& 3.0980620e+02,2.9943630e+02,2.9752810e+02,4.5973620e+02,4.2013030e+02, &
& 3.9648380e+02,3.7725130e+02,3.4707680e+02,4.5631660e+02,9.3534640e+02, &
& 7.1222580e+02,4.6281270e+02,4.5330770e+02,4.3922170e+02,4.4070090e+02, &
& 4.2296520e+02,4.4307970e+02,4.1683270e+02,4.2302650e+02,3.9614680e+02, &
& 3.8505430e+02,3.8239220e+02,4.0073810e+02,3.6983970e+02,4.9231890e+02, &
& 4.5993990e+02,4.2259020e+02,4.2529100e+02,3.7803640e+02,3.5676720e+02, &
& 3.4146440e+02,3.2670140e+02,3.1719220e+02,4.9566950e+02,4.5614500e+02, &
& 4.4471840e+02,4.3437700e+02,4.0821270e+02,5.1413610e+02,9.3570360e+02]
 vdw_dftd3_c6(25601:25700)=[&
& 5.4728610e+02,6.2674370e+02,5.6283810e+02,5.0066010e+02,4.8260870e+02, &
& 5.7030020e+02,1.3424720e+02,1.3288680e+02,9.9142600e+01,8.0387800e+01, &
& 5.6164700e+01,2.3546090e+02,2.8433280e+02,2.7514210e+02,2.4157090e+02, &
& 2.0531920e+02,3.3124320e+02,3.2036790e+02,3.2452330e+02,2.9723410e+02, &
& 2.2702370e+02,1.9579190e+02,1.9203650e+02,2.2228680e+02,2.0779350e+02, &
& 2.8939310e+02,3.0638780e+02,2.8353080e+02,2.6560990e+02,3.9765110e+02, &
& 3.8577200e+02,3.9008630e+02,3.7742800e+02,3.3658580e+02,2.9917850e+02, &
& 2.8351650e+02,2.8700770e+02,2.9901090e+02,3.7493950e+02,4.0708260e+02, &
& 3.8634370e+02,3.7446930e+02,4.8236480e+02,5.0153880e+02,3.7491100e+02, &
& 3.9079260e+02,3.6580820e+02,3.3438680e+02,3.1617560e+02,3.2076260e+02, &
& 3.3687270e+02,3.7640700e+01,2.5087500e+01,5.8487960e+02,3.3672920e+02, &
& 2.2693930e+02,1.5370200e+02,1.0786020e+02,8.1954700e+01,6.2331700e+01, &
& 4.8136800e+01,6.9957010e+02,5.3690060e+02,4.9477230e+02,3.8992290e+02, &
& 3.0437830e+02,2.5298030e+02,2.0704280e+02,1.6977090e+02,1.1488307e+03, &
& 9.4494750e+02,7.8031190e+02,7.5356620e+02,6.9006660e+02,5.4417950e+02, &
& 5.9417250e+02,4.6687620e+02,4.9428940e+02,5.0915080e+02,3.9094350e+02, &
& 4.0086460e+02,4.7436550e+02,4.1828000e+02,3.5698080e+02,3.2059210e+02, &
& 2.8090020e+02,2.4465900e+02,1.2877826e+03,1.1263391e+03,9.8598710e+02, &
& 8.8600180e+02,8.0798210e+02,6.2423380e+02,6.9629870e+02,5.3092790e+02, &
& 5.7990290e+02,5.3785540e+02,4.4915460e+02,4.7410150e+02,5.9460280e+02]
 vdw_dftd3_c6(25701:25800)=[&
& 5.4954150e+02,4.8978390e+02,4.5457430e+02,4.1127480e+02,3.7008590e+02, &
& 1.5690757e+03,1.4367682e+03,1.2612174e+03,5.7883370e+02,1.2731210e+03, &
& 1.2222593e+03,1.1915365e+03,1.1632807e+03,1.1382189e+03,8.9497400e+02, &
& 1.0093816e+03,9.7340780e+02,1.0263420e+03,1.0045056e+03,9.8480680e+02, &
& 9.7327760e+02,8.1998780e+02,8.0559300e+02,7.3521850e+02,6.2134250e+02, &
& 6.3198300e+02,5.7293310e+02,5.2489900e+02,4.3657950e+02,4.0812070e+02, &
& 4.1966530e+02,6.1117330e+02,5.9712280e+02,5.4862180e+02,5.2330660e+02, &
& 4.8315870e+02,4.4313040e+02,1.4804889e+03,1.4202956e+03,1.2546279e+03, &
& 1.1272570e+03,1.1199568e+03,1.0844785e+03,1.1170554e+03,1.0816612e+03, &
& 6.0479200e+01,1.9581130e+02,2.4984400e+02,1.9103520e+02,1.4423770e+02, &
& 1.0063880e+02,7.3584100e+01,5.0930700e+01,2.8655560e+02,4.4298480e+02, &
& 4.4845700e+02,3.6047340e+02,2.9550270e+02,2.5021530e+02,2.0491230e+02, &
& 3.9586240e+02,7.5678810e+02,3.9555800e+02,3.8199770e+02,3.7441990e+02, &
& 3.7101600e+02,3.4129280e+02,3.1596540e+02,3.0149960e+02,2.9478150e+02, &
& 2.9134120e+02,2.7325060e+02,4.4554260e+02,3.9138140e+02,3.5096110e+02, &
& 3.2072380e+02,2.8284750e+02,4.7250610e+02,9.2136230e+02,7.0355580e+02, &
& 5.2340250e+02,5.2863200e+02,4.9314640e+02,5.5329240e+02,4.3004960e+02, &
& 4.0251430e+02,3.7395490e+02,3.6182100e+02,3.5776620e+02,5.6674250e+02, &
& 5.1255600e+02,4.8004840e+02,4.5424350e+02,4.1511430e+02,5.5834700e+02, &
& 1.1954806e+03,8.9067110e+02,5.6141900e+02,5.4982120e+02,5.3242950e+02]
 vdw_dftd3_c6(25801:25900)=[&
& 5.3504270e+02,5.1592290e+02,5.3831350e+02,5.0569480e+02,5.1458640e+02, &
& 4.7998260e+02,4.6631110e+02,4.6329610e+02,4.8680100e+02,4.4777810e+02, &
& 6.0670050e+02,5.6315480e+02,5.1397290e+02,5.2014190e+02,4.5549230e+02, &
& 4.2873640e+02,4.0967910e+02,3.9187930e+02,3.7862560e+02,6.0907890e+02, &
& 5.5494180e+02,5.3810030e+02,5.2340750e+02,4.8896760e+02,6.2876220e+02, &
& 1.1883406e+03,6.6419980e+02,7.6713680e+02,6.8633050e+02,6.0632050e+02, &
& 5.8350520e+02,7.0242340e+02,1.6303250e+02,1.6051470e+02,1.1775890e+02, &
& 9.4561900e+01,6.5054900e+01,2.8915800e+02,3.4950920e+02,3.3534010e+02, &
& 2.9153890e+02,2.4518670e+02,4.0606400e+02,3.9020820e+02,3.9504660e+02, &
& 3.6178060e+02,2.7298940e+02,2.3378110e+02,2.2948940e+02,2.6871310e+02, &
& 2.5040430e+02,3.5311120e+02,3.7266030e+02,3.4178700e+02,3.1822560e+02, &
& 4.8670590e+02,4.6832670e+02,4.7237120e+02,4.5708930e+02,4.0445180e+02, &
& 3.5727900e+02,3.3777820e+02,3.4411230e+02,3.5934930e+02,4.5622010e+02, &
& 4.9554180e+02,4.6689650e+02,4.5057090e+02,5.8874480e+02,6.0988230e+02, &
& 4.5235490e+02,4.7275440e+02,4.3936980e+02,3.9936360e+02,3.7591510e+02, &
& 3.8338850e+02,4.0305640e+02,4.8774180e+02,3.9792200e+01,2.6446700e+01, &
& 6.1488410e+02,3.5582050e+02,2.4002620e+02,1.6250360e+02,1.1392620e+02, &
& 8.6471200e+01,6.5688800e+01,5.0670700e+01,7.3557820e+02,5.6688720e+02, &
& 5.2287510e+02,4.1241080e+02,3.2196130e+02,2.6749610e+02,2.1879400e+02, &
& 1.7927010e+02,1.2058672e+03,9.9578930e+02,8.2287620e+02,7.9494560e+02]
 vdw_dftd3_c6(25901:26000)=[&
& 7.2813060e+02,5.7390570e+02,6.2713920e+02,4.9253590e+02,5.2201020e+02, &
& 5.3761470e+02,4.1250160e+02,4.2350910e+02,5.0121030e+02,4.4228050e+02, &
& 3.7753680e+02,3.3899500e+02,2.9691970e+02,2.5848000e+02,1.3518089e+03, &
& 1.1865298e+03,1.0397874e+03,9.3482320e+02,8.5268980e+02,6.5880460e+02, &
& 7.3487030e+02,5.6035040e+02,6.1229710e+02,5.6795070e+02,4.7388930e+02, &
& 5.0066290e+02,6.2779210e+02,5.8070530e+02,5.1776630e+02,4.8056780e+02, &
& 4.3474640e+02,3.9110730e+02,1.6469053e+03,1.5126316e+03,1.3293437e+03, &
& 6.1169810e+02,1.3407087e+03,1.2874426e+03,1.2551651e+03,1.2254747e+03, &
& 1.1991443e+03,9.4362380e+02,1.0620589e+03,1.0244069e+03,1.0816382e+03, &
& 1.0586802e+03,1.0379814e+03,1.0258078e+03,8.6471680e+02,8.5048450e+02, &
& 7.7645220e+02,6.5606950e+02,6.6747630e+02,6.0520260e+02,5.5449180e+02, &
& 4.6101730e+02,4.3090160e+02,4.4323910e+02,6.4499260e+02,6.3069680e+02, &
& 5.7974990e+02,5.5307510e+02,5.1064800e+02,4.6827350e+02,1.5554213e+03, &
& 1.4960560e+03,1.3229452e+03,1.1895930e+03,1.1810895e+03,1.1436704e+03, &
& 1.1773166e+03,1.1401626e+03,6.3979600e+01,2.0691270e+02,2.6400830e+02, &
& 2.0201910e+02,1.5244950e+02,1.0626350e+02,7.7602400e+01,5.3610900e+01, &
& 3.0258820e+02,4.6775760e+02,4.7397540e+02,3.8125380e+02,3.1255060e+02, &
& 2.6456000e+02,2.1653480e+02,4.1774830e+02,7.9700090e+02,4.1801280e+02, &
& 4.0362390e+02,3.9559060e+02,3.9193640e+02,3.6075390e+02,3.3396980e+02, &
& 3.1863900e+02,3.1152110e+02,3.0775700e+02,2.8885340e+02,4.7083440e+02]
 vdw_dftd3_c6(26001:26100)=[&
& 4.1385520e+02,3.7116030e+02,3.3912820e+02,2.9897830e+02,4.9860580e+02, &
& 9.6992320e+02,7.4205020e+02,5.5289600e+02,5.5839300e+02,5.2087780e+02, &
& 5.8393470e+02,4.5433520e+02,4.2516750e+02,3.9494400e+02,3.8209250e+02, &
& 3.7801500e+02,5.9852070e+02,5.4168720e+02,5.0748490e+02,4.8022270e+02, &
& 4.3881240e+02,5.8960800e+02,1.2575693e+03,9.3914200e+02,5.9329090e+02, &
& 5.8103250e+02,5.6264920e+02,5.6536800e+02,5.4495840e+02,5.6886990e+02, &
& 5.3439680e+02,5.4369520e+02,5.0726200e+02,4.9282300e+02,4.8963950e+02, &
& 5.1452780e+02,4.7326870e+02,6.4058910e+02,5.9475240e+02,5.4291370e+02, &
& 5.4925780e+02,4.8124530e+02,4.5291160e+02,4.3272460e+02,4.1382910e+02, &
& 3.9995160e+02,6.4299800e+02,5.8628560e+02,5.6868040e+02,5.5320800e+02, &
& 5.1680550e+02,6.6402220e+02,1.2510172e+03,7.0195930e+02,8.1010680e+02, &
& 7.2464740e+02,6.4054440e+02,6.1646960e+02,7.4111760e+02,1.7249040e+02, &
& 1.6970770e+02,1.2443980e+02,9.9829400e+01,6.8563200e+01,3.0583900e+02, &
& 3.6956280e+02,3.5468400e+02,3.0832680e+02,2.5923020e+02,4.2911330e+02, &
& 4.1249460e+02,4.1760390e+02,3.8232410e+02,2.8838020e+02,2.4693330e+02, &
& 2.4239630e+02,2.8387680e+02,2.6454680e+02,3.7331260e+02,3.9406900e+02, &
& 3.6145320e+02,3.3647870e+02,5.1420470e+02,4.9505130e+02,4.9936110e+02, &
& 4.8303860e+02,4.2736900e+02,3.7746330e+02,3.5681920e+02,3.6342680e+02, &
& 3.7957410e+02,4.8204630e+02,5.2374110e+02,4.9359530e+02,4.7634340e+02, &
& 6.2197270e+02,6.4464210e+02,4.7820960e+02,4.9956010e+02,4.6428480e+02]
 vdw_dftd3_c6(26101:26200)=[&
& 4.2194200e+02,3.9711980e+02,4.0488970e+02,4.2575640e+02,5.1525750e+02, &
& 5.4445630e+02,4.0764900e+01,2.7174600e+01,6.1318280e+02,3.5984820e+02, &
& 2.4435680e+02,1.6611890e+02,1.1675030e+02,8.8739700e+01,6.7475000e+01, &
& 5.2075900e+01,7.3412320e+02,5.7193110e+02,5.2949770e+02,4.1959890e+02, &
& 3.2872910e+02,2.7365740e+02,2.2421020e+02,1.8393680e+02,1.2004050e+03, &
& 1.0001815e+03,8.2806580e+02,8.0123380e+02,7.3459090e+02,5.7897210e+02, &
& 6.3356300e+02,4.9763260e+02,5.2856480e+02,5.4385410e+02,4.1719990e+02, &
& 4.2968510e+02,5.0815670e+02,4.5004640e+02,3.8531250e+02,3.4654960e+02, &
& 3.0400640e+02,2.6498470e+02,1.3466051e+03,1.1912973e+03,1.0473694e+03, &
& 9.4346900e+02,8.6162430e+02,6.6699900e+02,7.4348240e+02,5.6809890e+02, &
& 6.2085810e+02,5.7630050e+02,4.8048740e+02,5.0846720e+02,6.3617130e+02, &
& 5.9026500e+02,5.2774110e+02,4.9058420e+02,4.4449370e+02,4.0041400e+02, &
& 1.6407171e+03,1.5169526e+03,1.3375362e+03,6.2310240e+02,1.3458984e+03, &
& 1.2931656e+03,1.2609414e+03,1.2312801e+03,1.2049833e+03,9.5146290e+02, &
& 1.0651460e+03,1.0280001e+03,1.0878112e+03,1.0648244e+03,1.0441447e+03, &
& 1.0317560e+03,8.7174390e+02,8.6024520e+02,7.8678670e+02,6.6560880e+02, &
& 6.7767670e+02,6.1534220e+02,5.6441430e+02,4.6971680e+02,4.3919500e+02, &
& 4.5207010e+02,6.5391820e+02,6.4098700e+02,5.9069190e+02,5.6428990e+02, &
& 5.2180500e+02,4.7912360e+02,1.5538386e+03,1.5028627e+03,1.3329466e+03, &
& 1.2024650e+03,1.1918172e+03,1.1541338e+03,1.1856295e+03,1.1486554e+03]
 vdw_dftd3_c6(26201:26300)=[&
& 6.5353600e+01,2.0943720e+02,2.6754910e+02,2.0589850e+02,1.5580320e+02, &
& 1.0888470e+02,7.9657600e+01,5.5128300e+01,3.0582180e+02,4.7251580e+02, &
& 4.8037720e+02,3.8813130e+02,3.1914590e+02,2.7063230e+02,2.2188150e+02, &
& 4.2309090e+02,8.0050250e+02,4.2505990e+02,4.1044180e+02,4.0223240e+02, &
& 3.9832240e+02,3.6744750e+02,3.4035150e+02,3.2470530e+02,3.1739060e+02, &
& 3.1302130e+02,2.9468800e+02,4.7779970e+02,4.2144820e+02,3.7886260e+02, &
& 3.4667280e+02,3.0608650e+02,5.0534960e+02,9.7364740e+02,7.4962580e+02, &
& 5.6197920e+02,5.6751500e+02,5.2972340e+02,5.9210620e+02,4.6293930e+02, &
& 4.3325010e+02,4.0255460e+02,3.8932060e+02,3.8581270e+02,6.0715180e+02, &
& 5.5119600e+02,5.1744360e+02,4.9028150e+02,4.4864430e+02,5.9907190e+02, &
& 1.2605537e+03,9.4816680e+02,6.0443760e+02,5.9196230e+02,5.7330320e+02, &
& 5.7583010e+02,5.5423280e+02,5.7935270e+02,5.4441580e+02,5.5345320e+02, &
& 5.1696070e+02,5.0231200e+02,4.9901960e+02,5.2412520e+02,4.8244240e+02, &
& 6.4975910e+02,6.0428050e+02,5.5254180e+02,5.5811920e+02,4.9092230e+02, &
& 4.6222330e+02,4.4172420e+02,4.2235490e+02,4.0877310e+02,6.5249230e+02, &
& 5.9673810e+02,5.7973290e+02,5.6455350e+02,5.2811830e+02,6.7482210e+02, &
& 1.2567278e+03,7.1513070e+02,8.2307590e+02,7.3672690e+02,6.5261520e+02, &
& 6.2834540e+02,7.5123490e+02,1.7590820e+02,1.7315440e+02,1.2738320e+02, &
& 1.0231260e+02,7.0396100e+01,3.1101430e+02,3.7562900e+02,3.6130250e+02, &
& 3.1474790e+02,2.6518320e+02,4.3613820e+02,4.2000450e+02,4.2525490e+02]
 vdw_dftd3_c6(26301:26400)=[&
& 3.8922030e+02,2.9426590e+02,2.5234820e+02,2.4765980e+02,2.8933940e+02, &
& 2.6984110e+02,3.8000030e+02,4.0150980e+02,3.6905850e+02,3.4397610e+02, &
& 5.2267140e+02,5.0441040e+02,5.0913000e+02,4.9230130e+02,4.3628420e+02, &
& 3.8581740e+02,3.6486800e+02,3.7099700e+02,3.8733000e+02,4.9068810e+02, &
& 5.3323730e+02,5.0351030e+02,4.8640400e+02,6.3257940e+02,6.5653900e+02, &
& 4.8797410e+02,5.0923540e+02,4.7403980e+02,4.3128970e+02,4.0628510e+02, &
& 4.1359970e+02,4.3491300e+02,5.2503950e+02,5.5492310e+02,5.6604970e+02, &
& 4.0683500e+01,2.7269000e+01,5.9704830e+02,3.5412790e+02,2.4207220e+02, &
& 1.6537510e+02,1.1663660e+02,8.8864100e+01,6.7699200e+01,5.2323700e+01, &
& 7.1529180e+02,5.6173090e+02,5.2182740e+02,4.1539360e+02,3.2669480e+02, &
& 2.7264470e+02,2.2390350e+02,1.8404790e+02,1.1685636e+03,9.7928960e+02, &
& 8.1191130e+02,7.8675610e+02,7.2192550e+02,5.6923970e+02,6.2340350e+02, &
& 4.8993540e+02,5.2113280e+02,5.3572570e+02,4.1114170e+02,4.2443520e+02, &
& 5.0154410e+02,4.4570000e+02,3.8280030e+02,3.4497610e+02,3.0323440e+02, &
& 2.6479090e+02,1.3118379e+03,1.1663270e+03,1.0280879e+03,9.2765740e+02, &
& 8.4816650e+02,6.5796520e+02,7.3282990e+02,5.6123130e+02,6.1320640e+02, &
& 5.6959290e+02,4.7485530e+02,5.0299460e+02,6.2793100e+02,5.8416010e+02, &
& 5.2370500e+02,4.8765620e+02,4.4264170e+02,3.9942590e+02,1.5989091e+03, &
& 1.4842030e+03,1.3119805e+03,6.1810270e+02,1.3179599e+03,1.2667956e+03, &
& 1.2353557e+03,1.2064001e+03,1.1807346e+03,9.3518300e+02,1.0429069e+03]
 vdw_dftd3_c6(26401:26500)=[&
& 1.0070657e+03,1.0665471e+03,1.0440654e+03,1.0238793e+03,1.0115941e+03, &
& 8.5651120e+02,8.4744200e+02,7.7639170e+02,6.5782210e+02,6.7012730e+02, &
& 6.0937170e+02,5.5959870e+02,4.6635070e+02,4.3627520e+02,4.4926040e+02, &
& 6.4607450e+02,6.3450740e+02,5.8610180e+02,5.6069830e+02,5.1936900e+02, &
& 4.7763150e+02,1.5173140e+03,1.4724510e+03,1.3089597e+03,1.1841644e+03, &
& 1.1721895e+03,1.1351989e+03,1.1641193e+03,1.1281520e+03,6.4976000e+01, &
& 2.0631430e+02,2.6393510e+02,2.0426390e+02,1.5510600e+02,1.0879270e+02, &
& 7.9818800e+01,5.5426400e+01,3.0093900e+02,4.6475070e+02,4.7382140e+02, &
& 3.8451100e+02,3.1721920e+02,2.6961720e+02,2.2156890e+02,4.1756030e+02, &
& 7.8439170e+02,4.2082330e+02,4.0643450e+02,3.9827910e+02,3.9424420e+02, &
& 3.6438140e+02,3.3772700e+02,3.2221420e+02,3.1490220e+02,3.1008700e+02, &
& 2.9273330e+02,4.7199570e+02,4.1772680e+02,3.7646650e+02,3.4508980e+02, &
& 3.0527830e+02,4.9917560e+02,9.5388690e+02,7.3824840e+02,5.5634680e+02, &
& 5.6177040e+02,5.2477420e+02,5.8512270e+02,4.5947870e+02,4.3011170e+02, &
& 3.9979630e+02,3.8652800e+02,3.8357630e+02,5.9985960e+02,5.4609180e+02, &
& 5.1367970e+02,4.8740710e+02,4.4676180e+02,5.9303760e+02,1.2339967e+03, &
& 9.3345900e+02,5.9968400e+02,5.8732120e+02,5.6888850e+02,5.7117360e+02, &
& 5.4902970e+02,5.7457480e+02,5.4011010e+02,5.4869770e+02,5.1304540e+02, &
& 4.9857190e+02,4.9524940e+02,5.1984050e+02,4.7888930e+02,6.4207210e+02, &
& 5.9809800e+02,5.4779890e+02,5.5252070e+02,4.8785140e+02,4.5960480e+02]
 vdw_dftd3_c6(26501:26600)=[&
& 4.3937670e+02,4.2009830e+02,4.0712160e+02,6.4511470e+02,5.9155410e+02, &
& 5.7553810e+02,5.6108050e+02,5.2566240e+02,6.6813180e+02,1.2322448e+03, &
& 7.0944400e+02,8.1473780e+02,7.2995170e+02,6.4769500e+02,6.2385580e+02, &
& 7.4228920e+02,1.7455350e+02,1.7202380e+02,1.2705660e+02,1.0226240e+02, &
& 7.0592600e+01,3.0770750e+02,3.7155110e+02,3.5816390e+02,3.1278040e+02, &
& 2.6420220e+02,4.3156730e+02,4.1632140e+02,4.2158300e+02,3.8584400e+02, &
& 2.9255420e+02,2.5131440e+02,2.4658940e+02,2.8727800e+02,2.6812740e+02, &
& 3.7653350e+02,3.9818770e+02,3.6684470e+02,3.4242490e+02,5.1736980e+02, &
& 5.0039000e+02,5.0540540e+02,4.8864930e+02,4.3386770e+02,3.8425630e+02, &
& 3.6358980e+02,3.6907770e+02,3.8510740e+02,4.8643230e+02,5.2860140e+02, &
& 5.0008910e+02,4.8364170e+02,6.2662040e+02,6.5103510e+02,4.8484790e+02, &
& 5.0560090e+02,4.7147570e+02,4.2954230e+02,4.0508190e+02,4.1178430e+02, &
& 4.3290980e+02,5.2115810e+02,5.5085190e+02,5.6229090e+02,5.5897120e+02, &
& 4.9435900e+01,3.2220500e+01,8.2215000e+02,4.5899560e+02,3.0409170e+02, &
& 2.0312550e+02,1.4087730e+02,1.0603060e+02,7.9895700e+01,6.1167600e+01, &
& 9.8096360e+02,7.3528720e+02,6.7185720e+02,5.2330930e+02,4.0436880e+02, &
& 3.3371710e+02,2.7114080e+02,2.2078740e+02,1.6183210e+03,1.3060236e+03, &
& 1.0738645e+03,1.0329143e+03,9.4357380e+02,7.4319420e+02,8.0955010e+02, &
& 6.3499500e+02,6.6951560e+02,6.9128830e+02,5.3001490e+02,5.4001980e+02, &
& 6.4142920e+02,5.6053310e+02,4.7441890e+02,4.2376020e+02,3.6911450e+02]
 vdw_dftd3_c6(26601:26700)=[&
& 3.1963860e+02,1.8108441e+03,1.5576903e+03,1.3534487e+03,1.2104384e+03, &
& 1.1002859e+03,8.4497140e+02,9.4457550e+02,7.1540020e+02,7.8173110e+02, &
& 7.2344940e+02,6.0407200e+02,6.3577580e+02,8.0295170e+02,7.3674350e+02, &
& 6.5189600e+02,6.0235480e+02,5.4227570e+02,4.8556750e+02,2.2065294e+03, &
& 1.9920198e+03,1.7357045e+03,7.7135460e+02,1.7597955e+03,1.6873282e+03, &
& 1.6443172e+03,1.6048351e+03,1.5698044e+03,1.2237547e+03,1.3970237e+03, &
& 1.3453523e+03,1.4127942e+03,1.3824590e+03,1.3549696e+03,1.3395852e+03, &
& 1.1220389e+03,1.0946876e+03,9.9428480e+02,8.3675230e+02,8.4958700e+02, &
& 7.6688670e+02,7.0006410e+02,5.7964280e+02,5.4079780e+02,5.5540190e+02, &
& 8.2307860e+02,7.9969300e+02,7.3000700e+02,6.9372340e+02,6.3752300e+02, &
& 5.8212360e+02,2.0693475e+03,1.9615822e+03,1.7203415e+03,1.5329737e+03, &
& 1.5284967e+03,1.4796138e+03,1.5311137e+03,1.4812261e+03,8.0304800e+01, &
& 2.6620380e+02,3.3870130e+02,2.5501410e+02,1.9065070e+02,1.3139800e+02, &
& 9.4990900e+01,6.4689400e+01,3.9028390e+02,6.0479080e+02,6.0764500e+02, &
& 4.8292640e+02,3.9240610e+02,3.3009010e+02,2.6835530e+02,5.3522110e+02, &
& 1.0452655e+03,5.3005110e+02,5.1159590e+02,5.0142290e+02,4.9734050e+02, &
& 4.5500680e+02,4.2043750e+02,4.0108240e+02,3.9226860e+02,3.8915420e+02, &
& 3.6239480e+02,6.0102950e+02,5.2334040e+02,4.6617350e+02,4.2395730e+02, &
& 3.7177260e+02,6.3744040e+02,1.2738440e+03,9.5836130e+02,7.0195430e+02, &
& 7.0892730e+02,6.5979250e+02,7.4563430e+02,5.7194600e+02,5.3479690e+02]
 vdw_dftd3_c6(26701:26800)=[&
& 4.9608770e+02,4.8018350e+02,4.7304820e+02,7.6329290e+02,6.8512580e+02, &
& 6.3830590e+02,6.0176620e+02,5.4740910e+02,7.4957040e+02,1.6581771e+03, &
& 1.2150291e+03,7.4777800e+02,7.3224530e+02,7.0878020e+02,7.1308040e+02, &
& 6.8979990e+02,7.1779230e+02,6.7351320e+02,6.8677710e+02,6.3867240e+02, &
& 6.2024820e+02,6.1643990e+02,6.4892260e+02,5.9550270e+02,8.1802970e+02, &
& 7.5571890e+02,6.8626340e+02,6.9723760e+02,6.0379550e+02,5.6713440e+02, &
& 5.4115970e+02,5.1741800e+02,4.9805080e+02,8.1861890e+02,7.4024200e+02, &
& 7.1498450e+02,6.9350480e+02,6.4521980e+02,8.4379470e+02,1.6402094e+03, &
& 8.8501740e+02,1.0289540e+03,9.1819730e+02,8.0643310e+02,7.7504710e+02, &
& 9.4627710e+02,2.1752840e+02,2.1346830e+02,1.5465610e+02,1.2319970e+02, &
& 8.3591200e+01,3.8834390e+02,4.7010710e+02,4.4845450e+02,3.8732870e+02, &
& 3.2337880e+02,5.4493370e+02,5.2129830e+02,5.2752920e+02,4.8302220e+02, &
& 3.6126840e+02,3.0772170e+02,3.0218010e+02,3.5673210e+02,3.3160290e+02, &
& 4.7243170e+02,4.9753330e+02,4.5357900e+02,4.2057420e+02,6.5272580e+02, &
& 6.2438560e+02,6.2866890e+02,6.0842970e+02,5.3532960e+02,4.7069610e+02, &
& 4.4413890e+02,4.5438240e+02,4.7518820e+02,6.0862900e+02,6.6134830e+02, &
& 6.2001070e+02,5.9661110e+02,7.8841700e+02,8.1437400e+02,6.0044820e+02, &
& 6.2883490e+02,5.8136160e+02,5.2620370e+02,4.9362900e+02,5.0521810e+02, &
& 5.3138910e+02,6.4808000e+02,6.8468130e+02,6.9650710e+02,6.9009760e+02, &
& 8.6755840e+02,9.7472000e+00,6.6434000e+00,1.2998750e+02,8.0427600e+01]
 vdw_dftd3_c6(26801:26900)=[&
& 5.6411600e+01,3.9244900e+01,2.8012900e+01,2.1498900e+01,1.6460700e+01, &
& 1.2757700e+01,1.5611380e+02,1.2654250e+02,1.1915450e+02,9.6531300e+01, &
& 7.7042900e+01,6.4891800e+01,5.3734300e+01,4.4461300e+01,2.5422370e+02, &
& 2.1794570e+02,1.8168340e+02,1.7706600e+02,1.6299900e+02,1.2872330e+02, &
& 1.4140710e+02,1.1135690e+02,1.1910910e+02,1.2201440e+02,9.3766000e+01, &
& 9.7684500e+01,1.1517060e+02,1.0369470e+02,9.0137300e+01,8.1838100e+01, &
& 7.2461300e+01,6.3676800e+01,2.8624860e+02,2.5952790e+02,2.3111730e+02, &
& 2.0990660e+02,1.9277650e+02,1.5074470e+02,1.6737830e+02,1.2927260e+02, &
& 1.4111090e+02,1.3140190e+02,1.0947410e+02,1.1639220e+02,1.4412590e+02, &
& 1.3544390e+02,1.2268730e+02,1.1497520e+02,1.0506600e+02,9.5397300e+01, &
& 3.4948600e+02,3.2947840e+02,2.9416920e+02,1.4462380e+02,2.9344860e+02, &
& 2.8246410e+02,2.7556100e+02,2.6918990e+02,2.6354840e+02,2.1125650e+02, &
& 2.3210280e+02,2.2459400e+02,2.3859840e+02,2.3361550e+02,2.2917770e+02, &
& 2.2630560e+02,1.9318890e+02,1.9317020e+02,1.7812490e+02,1.5179460e+02, &
& 1.5495640e+02,1.4166610e+02,1.3065360e+02,1.0940400e+02,1.0252010e+02, &
& 1.0573840e+02,1.4882270e+02,1.4721780e+02,1.3720280e+02,1.3195960e+02, &
& 1.2301010e+02,1.1377160e+02,3.3435640e+02,3.2867970e+02,2.9472000e+02, &
& 2.6950210e+02,2.6545030e+02,2.5712540e+02,2.6182830e+02,2.5402630e+02, &
& 1.5353100e+01,4.7033800e+01,6.0523400e+01,4.7858000e+01,3.6816600e+01, &
& 2.6148100e+01,1.9357100e+01,1.3558100e+01,6.8275500e+01,1.0530540e+02]
 vdw_dftd3_c6(26901:27000)=[&
& 1.0855170e+02,8.9594800e+01,7.4849700e+01,6.4157300e+01,5.3163800e+01, &
& 9.5849700e+01,1.7514340e+02,9.7755000e+01,9.4475800e+01,9.2545100e+01, &
& 9.1452800e+01,8.5131200e+01,7.9083200e+01,7.5452500e+01,7.3686500e+01, &
& 7.2121600e+01,6.8807700e+01,1.0875080e+02,9.7495300e+01,8.8710800e+01, &
& 8.1856400e+01,7.2922100e+01,1.1497210e+02,2.1287590e+02,1.6813540e+02, &
& 1.2924090e+02,1.3043790e+02,1.2219410e+02,1.3497050e+02,1.0771280e+02, &
& 1.0089580e+02,9.3899700e+01,9.0650800e+01,9.0427700e+01,1.3818550e+02, &
& 1.2714090e+02,1.2050990e+02,1.1496400e+02,1.0603500e+02,1.3780050e+02, &
& 2.7459140e+02,2.1235690e+02,1.4039590e+02,1.3751220e+02,1.3326430e+02, &
& 1.3360610e+02,1.2777250e+02,1.3432640e+02,1.2641840e+02,1.2810090e+02, &
& 1.2023750e+02,1.1690070e+02,1.1607540e+02,1.2156170e+02,1.1232440e+02, &
& 1.4812250e+02,1.3881710e+02,1.2792520e+02,1.2830620e+02,1.1489450e+02, &
& 1.0845800e+02,1.0379770e+02,9.9205000e+01,9.6600200e+01,1.4899930e+02, &
& 1.3799420e+02,1.3500470e+02,1.3215970e+02,1.2451480e+02,1.5534870e+02, &
& 2.7595780e+02,1.6604890e+02,1.8911370e+02,1.7001950e+02,1.5177090e+02, &
& 1.4638890e+02,1.7106450e+02,4.0935800e+01,4.0519200e+01,3.0353600e+01, &
& 2.4591900e+01,1.7142400e+01,7.1302300e+01,8.6053200e+01,8.3658600e+01, &
& 7.3739700e+01,6.2878300e+01,1.0006100e+02,9.7163200e+01,9.8436300e+01, &
& 9.0057400e+01,6.8995800e+01,5.9631300e+01,5.8451500e+01,6.7393200e+01, &
& 6.3074000e+01,8.7768800e+01,9.3130100e+01,8.6548600e+01,8.1240000e+01]
 vdw_dftd3_c6(27001:27100)=[&
& 1.2013000e+02,1.1715700e+02,1.1862430e+02,1.1463280e+02,1.0248830e+02, &
& 9.1251800e+01,8.6498600e+01,8.7237600e+01,9.0828400e+01,1.1348260e+02, &
& 1.2333400e+02,1.1753380e+02,1.1415530e+02,1.4593300e+02,1.5222500e+02, &
& 1.1417940e+02,1.1873150e+02,1.1142620e+02,1.0200810e+02,9.6570600e+01, &
& 9.7623800e+01,1.0254510e+02,1.2215410e+02,1.2915240e+02,1.3219470e+02, &
& 1.3178890e+02,1.6086740e+02,3.1443600e+01,8.8210000e+00,6.1196000e+00, &
& 1.1594980e+02,7.1681800e+01,5.0530300e+01,3.5412900e+01,2.5479900e+01, &
& 1.9694300e+01,1.5188300e+01,1.1851300e+01,1.3937530e+02,1.1282100e+02, &
& 1.0635080e+02,8.6399900e+01,6.9251500e+01,5.8577600e+01,4.8744800e+01, &
& 4.0544100e+01,2.2766000e+02,1.9467190e+02,1.6226090e+02,1.5827050e+02, &
& 1.4575800e+02,1.1536970e+02,1.2654160e+02,9.9901700e+01,1.0669170e+02, &
& 1.0923050e+02,8.4202800e+01,8.7612400e+01,1.0307380e+02,9.2962100e+01, &
& 8.1056300e+01,7.3806400e+01,6.5586800e+01,5.7866500e+01,2.5651980e+02, &
& 2.3200320e+02,2.0665890e+02,1.8782030e+02,1.7263950e+02,1.3537160e+02, &
& 1.5013980e+02,1.1632470e+02,1.2677780e+02,1.1814270e+02,9.8703200e+01, &
& 1.0476920e+02,1.2938530e+02,1.2165980e+02,1.1040890e+02,1.0365030e+02, &
& 9.4948900e+01,8.6458100e+01,3.1323470e+02,2.9471680e+02,2.6313810e+02, &
& 1.3023410e+02,2.6266440e+02,2.5281970e+02,2.4663480e+02,2.4092330e+02, &
& 2.3586410e+02,1.8938240e+02,2.0816040e+02,2.0142940e+02,2.1351460e+02, &
& 2.0904160e+02,2.0505940e+02,2.0246690e+02,1.7301760e+02,1.7295930e+02]
 vdw_dftd3_c6(27101:27200)=[&
& 1.5967690e+02,1.3642100e+02,1.3925850e+02,1.2750310e+02,1.1776180e+02, &
& 9.8928700e+01,9.2827200e+01,9.5685300e+01,1.3397180e+02,1.3250170e+02, &
& 1.2364240e+02,1.1905310e+02,1.1119300e+02,1.0307880e+02,2.9972370e+02, &
& 2.9409590e+02,2.6375700e+02,2.4153050e+02,2.3801510e+02,2.3059460e+02, &
& 2.3475660e+02,2.2776830e+02,1.3799200e+01,4.1971300e+01,5.4077000e+01, &
& 4.2964600e+01,3.3254000e+01,2.3813600e+01,1.7775000e+01,1.2597000e+01, &
& 6.1017400e+01,9.3997800e+01,9.6956000e+01,8.0261800e+01,6.7308800e+01, &
& 5.7922100e+01,4.8232800e+01,8.6148100e+01,1.5677020e+02,8.7815300e+01, &
& 8.4925300e+01,8.3211100e+01,8.2233100e+01,7.6584600e+01,7.1211000e+01, &
& 6.7977400e+01,6.6389700e+01,6.4964100e+01,6.2022800e+01,9.7383800e+01, &
& 8.7478100e+01,7.9798600e+01,7.3825700e+01,6.5996700e+01,1.0345590e+02, &
& 1.9069760e+02,1.5078940e+02,1.1619730e+02,1.1730510e+02,1.1002550e+02, &
& 1.2143810e+02,9.7174400e+01,9.1117100e+01,8.4895400e+01,8.1971500e+01, &
& 8.1750000e+01,1.2410850e+02,1.1430720e+02,1.0848830e+02,1.0364770e+02, &
& 9.5813900e+01,1.2398030e+02,2.4612560e+02,1.9047850e+02,1.2645720e+02, &
& 1.2386820e+02,1.2006680e+02,1.2034080e+02,1.1507330e+02,1.2093240e+02, &
& 1.1387540e+02,1.1534910e+02,1.0832620e+02,1.0533020e+02,1.0456940e+02, &
& 1.0938790e+02,1.0119050e+02,1.3303630e+02,1.2486130e+02,1.1526060e+02, &
& 1.1551780e+02,1.0377670e+02,9.8098700e+01,9.3979300e+01,8.9905200e+01, &
& 8.7593000e+01,1.3410840e+02,1.2432080e+02,1.2170140e+02,1.1924010e+02]
 vdw_dftd3_c6(27201:27300)=[&
& 1.1253730e+02,1.3975100e+02,2.4737300e+02,1.4949980e+02,1.7017100e+02, &
& 1.5324030e+02,1.3692210e+02,1.3212700e+02,1.5407130e+02,3.6697700e+01, &
& 3.6460400e+01,2.7506300e+01,2.2427900e+01,1.5800900e+01,6.3794400e+01, &
& 7.7008700e+01,7.4996700e+01,6.6319400e+01,5.6782600e+01,8.9828500e+01, &
& 8.7304800e+01,8.8470300e+01,8.1022900e+01,6.2396300e+01,5.4067600e+01, &
& 5.2992800e+01,6.0851300e+01,5.7004600e+01,7.8781200e+01,8.3613100e+01, &
& 7.7887400e+01,7.3281400e+01,1.0796880e+02,1.0538580e+02,1.0676290e+02, &
& 1.0327750e+02,9.2598300e+01,8.2649600e+01,7.8436800e+01,7.9027900e+01, &
& 8.2193000e+01,1.0218770e+02,1.1093720e+02,1.0585840e+02,1.0293730e+02, &
& 1.3126760e+02,1.3686860e+02,1.0287240e+02,1.0702160e+02,1.0067270e+02, &
& 9.2374800e+01,8.7606400e+01,8.8522700e+01,9.2901000e+01,1.1026110e+02, &
& 1.1648110e+02,1.1921790e+02,1.1892080e+02,1.4470670e+02,2.8412100e+01, &
& 2.5780900e+01,6.7746000e+00,4.8836000e+00,8.1977000e+01,5.2023700e+01, &
& 3.7568500e+01,2.6935100e+01,1.9770700e+01,1.5524900e+01,1.2151100e+01, &
& 9.6050000e+00,9.8880600e+01,8.1535200e+01,7.7652000e+01,6.4057600e+01, &
& 5.2150900e+01,4.4652900e+01,3.7629500e+01,3.1680400e+01,1.6201600e+02, &
& 1.4002730e+02,1.1711120e+02,1.1482560e+02,1.0604890e+02,8.4363100e+01, &
& 9.2463600e+01,7.3418500e+01,7.8469600e+01,8.0085300e+01,6.2129900e+01, &
& 6.4863300e+01,7.5880000e+01,6.9171000e+01,6.1042500e+01,5.6081200e+01, &
& 5.0334400e+01,4.4854600e+01,1.8314860e+02,1.6708650e+02,1.4989060e+02]
 vdw_dftd3_c6(27301:27400)=[&
& 1.3695780e+02,1.2643070e+02,1.0010800e+02,1.1061190e+02,8.6616300e+01, &
& 9.4100700e+01,8.7943000e+01,7.3805700e+01,7.8301600e+01,9.5772700e+01, &
& 9.0687600e+01,8.3057700e+01,7.8481900e+01,7.2453100e+01,6.6508100e+01, &
& 2.2390760e+02,2.1212970e+02,1.9063700e+02,9.7988200e+01,1.8970840e+02, &
& 1.8275600e+02,1.7832410e+02,1.7422140e+02,1.7058700e+02,1.3842980e+02, &
& 1.5082040e+02,1.4614450e+02,1.5462870e+02,1.5139240e+02,1.4852730e+02, &
& 1.4657090e+02,1.2614450e+02,1.2685210e+02,1.1782080e+02,1.0145610e+02, &
& 1.0369670e+02,9.5491900e+01,8.8641300e+01,7.5088300e+01,7.0691000e+01, &
& 7.2867700e+01,9.9854500e+01,9.9155700e+01,9.3202900e+01,9.0185300e+01, &
& 8.4799100e+01,7.9155500e+01,2.1546340e+02,2.1255240e+02,1.9179540e+02, &
& 1.7728870e+02,1.7432070e+02,1.6896800e+02,1.7120630e+02,1.6624200e+02, &
& 1.0398700e+01,3.0602200e+01,3.9647800e+01,3.2164400e+01,2.5331800e+01, &
& 1.8518000e+01,1.4079300e+01,1.0221400e+01,4.4492400e+01,6.8314100e+01, &
& 7.1023100e+01,5.9688700e+01,5.0739800e+01,4.4158100e+01,3.7238100e+01, &
& 6.3868600e+01,1.1341610e+02,6.5520300e+01,6.3464500e+01,6.2202000e+01, &
& 6.1420200e+01,5.7491500e+01,5.3617100e+01,5.1231800e+01,5.0021800e+01, &
& 4.8763000e+01,4.6891100e+01,7.1907900e+01,6.5304100e+01,6.0151900e+01, &
& 5.6096200e+01,5.0629800e+01,7.7004000e+01,1.3809270e+02,1.1082380e+02, &
& 8.6803500e+01,8.7650400e+01,8.2525100e+01,9.0466000e+01,7.3425900e+01, &
& 6.9000500e+01,6.4465100e+01,6.2220900e+01,6.2213100e+01,9.2140300e+01]
 vdw_dftd3_c6(27401:27500)=[&
& 8.5547900e+01,8.1727900e+01,7.8506700e+01,7.3095400e+01,9.2705000e+01, &
& 1.7805360e+02,1.3991280e+02,9.5217800e+01,9.3284100e+01,9.0480700e+01, &
& 9.0568200e+01,8.6338600e+01,9.0910000e+01,8.5747500e+01,8.6674000e+01, &
& 8.1652000e+01,7.9429100e+01,7.8815000e+01,8.2180600e+01,7.6300400e+01, &
& 9.8802500e+01,9.3300800e+01,8.6690900e+01,8.6498300e+01,7.8773500e+01, &
& 7.4726900e+01,7.1762400e+01,6.8753800e+01,6.7229300e+01,1.0008700e+02, &
& 9.3482100e+01,9.1896500e+01,9.0376000e+01,8.5807500e+01,1.0451280e+02, &
& 1.7971610e+02,1.1246690e+02,1.2728190e+02,1.1517510e+02,1.0344890e+02, &
& 9.9987800e+01,1.1495380e+02,2.7422500e+01,2.7484200e+01,2.1148800e+01, &
& 1.7492800e+01,1.2607700e+01,4.7205900e+01,5.6971300e+01,5.5919700e+01, &
& 4.9981800e+01,4.3313000e+01,6.6870600e+01,6.5341500e+01,6.6260700e+01, &
& 6.0783400e+01,4.7503300e+01,4.1486200e+01,4.0637600e+01,4.6084100e+01, &
& 4.3309600e+01,5.8812600e+01,6.2560900e+01,5.8794500e+01,5.5703700e+01, &
& 8.0586800e+01,7.9155000e+01,8.0375800e+01,7.7868100e+01,7.0424400e+01, &
& 6.3308100e+01,6.0264100e+01,6.0414900e+01,6.2653200e+01,7.6779800e+01, &
& 8.3196800e+01,7.9891200e+01,7.8022400e+01,9.8267000e+01,1.0262510e+02, &
& 7.7737900e+01,8.0805200e+01,7.6585200e+01,7.0735300e+01,6.7427200e+01, &
& 6.7888500e+01,7.1112000e+01,8.3409500e+01,8.8005200e+01,9.0199000e+01, &
& 9.0200500e+01,1.0838490e+02,2.1720500e+01,1.9866900e+01,1.5581700e+01, &
& 2.1058400e+01,1.3698200e+01,3.2279250e+02,1.8966050e+02,1.2796450e+02]
 vdw_dftd3_c6(27501:27600)=[&
& 8.6160500e+01,5.9907400e+01,4.5087600e+01,3.3933400e+01,2.5933500e+01, &
& 3.8609800e+02,3.0133070e+02,2.7856390e+02,2.1994790e+02,1.7135570e+02, &
& 1.4185170e+02,1.1545520e+02,9.4042800e+01,6.2895370e+02,5.2583670e+02, &
& 4.3542050e+02,4.2087110e+02,3.8566110e+02,3.0312660e+02,3.3231560e+02, &
& 2.6022290e+02,2.7689480e+02,2.8511740e+02,2.1789580e+02,2.2471970e+02, &
& 2.6645340e+02,2.3543820e+02,2.0075140e+02,1.7986350e+02,1.5701910e+02, &
& 1.3612380e+02,7.0496090e+02,6.2569960e+02,5.4992420e+02,4.9495070e+02, &
& 4.5153080e+02,3.4833970e+02,3.8882260e+02,2.9592560e+02,3.2404080e+02, &
& 3.0049770e+02,2.4967590e+02,2.6472910e+02,3.3234100e+02,3.0808400e+02, &
& 2.7473330e+02,2.5478050e+02,2.3007980e+02,2.0645810e+02,8.5867990e+02, &
& 7.9610660e+02,7.0193980e+02,3.2415330e+02,7.0578730e+02,6.7819310e+02, &
& 6.6131780e+02,6.4579390e+02,6.3203590e+02,4.9802070e+02,5.5718410e+02, &
& 5.3772660e+02,5.7065080e+02,5.5863790e+02,5.4782520e+02,5.4139780e+02, &
& 4.5683250e+02,4.5089790e+02,4.1177370e+02,3.4723090e+02,3.5352640e+02, &
& 3.2038750e+02,2.9331160e+02,2.4307640e+02,2.2688500e+02,2.3369270e+02, &
& 3.4043160e+02,3.3373140e+02,3.0698840e+02,2.9279590e+02,2.7003130e+02, &
& 2.4716150e+02,8.1311660e+02,7.8841980e+02,6.9912600e+02,6.2955760e+02, &
& 6.2364620e+02,6.0379750e+02,6.2051110e+02,6.0113500e+02,3.4070000e+01, &
& 1.1021360e+02,1.4056320e+02,1.0751580e+02,8.0711100e+01,5.5779300e+01, &
& 4.0337400e+01,2.7443600e+01,1.6065940e+02,2.4856920e+02,2.5249530e+02]
 vdw_dftd3_c6(27601:27700)=[&
& 2.0322260e+02,1.6626980e+02,1.4026180e+02,1.1424050e+02,2.2072640e+02, &
& 4.1971290e+02,2.2183390e+02,2.1401930e+02,2.0967420e+02,2.0762990e+02, &
& 1.9138970e+02,1.7705420e+02,1.6880270e+02,1.6499240e+02,1.6280210e+02, &
& 1.5307760e+02,2.5034940e+02,2.2023170e+02,1.9730880e+02,1.7991980e+02, &
& 1.5811820e+02,2.6324390e+02,5.0999270e+02,3.9206600e+02,2.9297650e+02, &
& 2.9577860e+02,2.7564350e+02,3.0843590e+02,2.4026870e+02,2.2456570e+02, &
& 2.0834700e+02,2.0146610e+02,1.9966700e+02,3.1698520e+02,2.8733370e+02, &
& 2.6924000e+02,2.5459630e+02,2.3226170e+02,3.1205400e+02,6.5974760e+02, &
& 4.9580770e+02,3.1434490e+02,3.0783100e+02,2.9804520e+02,2.9947280e+02, &
& 2.8833130e+02,3.0148740e+02,2.8310610e+02,2.8795020e+02,2.6876200e+02, &
& 2.6111000e+02,2.5945460e+02,2.7290680e+02,2.5082830e+02,3.3919420e+02, &
& 3.1485670e+02,2.8725880e+02,2.9046490e+02,2.5437210e+02,2.3906730e+02, &
& 2.2815940e+02,2.1789610e+02,2.1069250e+02,3.3974310e+02,3.1027180e+02, &
& 3.0114890e+02,2.9290320e+02,2.7334560e+02,3.5155860e+02,6.5770110e+02, &
& 3.7210800e+02,4.2861200e+02,3.8281860e+02,3.3870230e+02,3.2590880e+02, &
& 3.9082830e+02,9.2012600e+01,9.0144000e+01,6.5696800e+01,5.2313600e+01, &
& 3.5460200e+01,1.6311000e+02,1.9694110e+02,1.8900340e+02,1.6395650e+02, &
& 1.3739070e+02,2.2781080e+02,2.1910140e+02,2.2177080e+02,2.0271680e+02, &
& 1.5223080e+02,1.3008090e+02,1.2768330e+02,1.4999510e+02,1.3971080e+02, &
& 1.9848130e+02,2.0964300e+02,1.9209160e+02,1.7848350e+02,2.7262520e+02]
 vdw_dftd3_c6(27701:27800)=[&
& 2.6276340e+02,2.6502430e+02,2.5593200e+02,2.2596360e+02,1.9915880e+02, &
& 1.8804460e+02,1.9149380e+02,2.0021140e+02,2.5528030e+02,2.7779270e+02, &
& 2.6182240e+02,2.5251890e+02,3.2959150e+02,3.4222420e+02,2.5364330e+02, &
& 2.6456000e+02,2.4551490e+02,2.2268310e+02,2.0925910e+02,2.1319310e+02, &
& 2.2444800e+02,2.7230380e+02,2.8809540e+02,2.9386510e+02,2.9167140e+02, &
& 3.6287330e+02,6.8433000e+01,6.1382900e+01,4.5939900e+01,1.5359450e+02, &
& 2.4983300e+01,1.6110400e+01,4.0808020e+02,2.3182510e+02,1.5403880e+02, &
& 1.0272020e+02,7.0976100e+01,5.3206100e+01,3.9917200e+01,3.0433800e+01, &
& 4.8717610e+02,3.7037780e+02,3.3945020e+02,2.6511890e+02,2.0487700e+02, &
& 1.6882250e+02,1.3684600e+02,1.1110440e+02,7.9883900e+02,6.5341740e+02, &
& 5.3857540e+02,5.1863330e+02,4.7415850e+02,3.7278380e+02,4.0724250e+02, &
& 3.1885890e+02,3.3746240e+02,3.4824770e+02,2.6631200e+02,2.7256060e+02, &
& 3.2381790e+02,2.8369090e+02,2.4023030e+02,2.1441210e+02,1.8649270e+02, &
& 1.6117030e+02,8.9400720e+02,7.7835390e+02,6.7875770e+02,6.0808480e+02, &
& 5.5314470e+02,4.2480720e+02,4.7494180e+02,3.5969290e+02,3.9366130e+02, &
& 3.6442410e+02,3.0337520e+02,3.2034290e+02,4.0432940e+02,3.7207850e+02, &
& 3.2966250e+02,3.0462850e+02,2.7410450e+02,2.4517980e+02,1.0888566e+03, &
& 9.9327630e+02,8.6886070e+02,3.8956690e+02,8.7832220e+02,8.4282680e+02, &
& 8.2153240e+02,8.0197510e+02,7.8463100e+02,6.1326270e+02,6.9516170e+02, &
& 6.6987120e+02,7.0696300e+02,6.9190950e+02,6.7829270e+02,6.7054790e+02]
 vdw_dftd3_c6(27801:27900)=[&
& 5.6267060e+02,5.5104570e+02,5.0107280e+02,4.2136670e+02,4.2822450e+02, &
& 3.8673130e+02,3.5308580e+02,2.9192880e+02,2.7221920e+02,2.7992390e+02, &
& 4.1374950e+02,4.0319540e+02,3.6867120e+02,3.5049670e+02,3.2207940e+02, &
& 2.9390180e+02,1.0244219e+03,9.7976180e+02,8.6241890e+02,7.7060950e+02, &
& 7.6655200e+02,7.4202530e+02,7.6630100e+02,7.4167020e+02,4.0697200e+01, &
& 1.3444900e+02,1.7103980e+02,1.2908980e+02,9.6287200e+01,6.6105200e+01, &
& 4.7566400e+01,3.2173000e+01,1.9665260e+02,3.0467970e+02,3.0710150e+02, &
& 2.4462880e+02,1.9876160e+02,1.6695970e+02,1.3542340e+02,2.6897780e+02, &
& 5.2175350e+02,2.6771200e+02,2.5825540e+02,2.5306610e+02,2.5088270e+02, &
& 2.3001820e+02,2.1251260e+02,2.0263880e+02,1.9815060e+02,1.9629960e+02, &
& 1.8325560e+02,3.0359070e+02,2.6489450e+02,2.3602870e+02,2.1449810e+02, &
& 1.8783780e+02,3.2026650e+02,6.3491250e+02,4.8074960e+02,3.5400180e+02, &
& 3.5746120e+02,3.3259800e+02,3.7482840e+02,2.8854060e+02,2.6962410e+02, &
& 2.4998080e+02,2.4189890e+02,2.3876910e+02,3.8467520e+02,3.4614310e+02, &
& 3.2280720e+02,3.0433880e+02,2.7671610e+02,3.7749200e+02,8.2435520e+02, &
& 6.0887560e+02,3.7764800e+02,3.6979850e+02,3.5793640e+02,3.6001510e+02, &
& 3.4780870e+02,3.6250210e+02,3.4013210e+02,3.4661200e+02,3.2261560e+02, &
& 3.1332960e+02,3.1141380e+02,3.2794410e+02,3.0090150e+02,4.1188700e+02, &
& 3.8080640e+02,3.4601610e+02,3.5118480e+02,3.0465320e+02,2.8600240e+02, &
& 2.7277910e+02,2.6060470e+02,2.5111540e+02,4.1199350e+02,3.7352430e+02]
 vdw_dftd3_c6(27901:28000)=[&
& 3.6118830e+02,3.5044170e+02,3.2600770e+02,4.2507230e+02,8.1753510e+02, &
& 4.4707850e+02,5.1836450e+02,4.6224170e+02,4.0682820e+02,3.9105750e+02, &
& 4.7525970e+02,1.1031340e+02,1.0796470e+02,7.8057400e+01,6.1953000e+01, &
& 4.1768700e+01,1.9678410e+02,2.3794340e+02,2.2718820e+02,1.9611960e+02, &
& 1.6353520e+02,2.7522420e+02,2.6358530e+02,2.6672430e+02,2.4396540e+02, &
& 1.8218540e+02,1.5510950e+02,1.5231640e+02,1.7996900e+02,1.6731730e+02, &
& 2.3896240e+02,2.5184820e+02,2.2963620e+02,2.1276790e+02,3.2933860e+02, &
& 3.1562100e+02,3.1785670e+02,3.0723810e+02,2.7018990e+02,2.3741460e+02, &
& 2.2392390e+02,2.2893290e+02,2.3956050e+02,3.0725150e+02,3.3420220e+02, &
& 3.1357940e+02,3.0173570e+02,3.9768030e+02,4.1154360e+02,3.0359840e+02, &
& 3.1746390e+02,2.9346710e+02,2.6545010e+02,2.4889430e+02,2.5448410e+02, &
& 2.6790110e+02,3.2692180e+02,3.4569090e+02,3.5194960e+02,3.4876390e+02, &
& 4.3764130e+02,8.1347800e+01,7.2953000e+01,5.4385700e+01,1.8402910e+02, &
& 2.2150460e+02,2.1250700e+01,1.4157700e+01,3.0477070e+02,1.8330560e+02, &
& 1.2601940e+02,8.6242700e+01,6.0766300e+01,4.6192700e+01,3.5073300e+01, &
& 2.6999000e+01,3.6521100e+02,2.8990900e+02,2.7035350e+02,2.1613030e+02, &
& 1.7040450e+02,1.4229820e+02,1.1683550e+02,9.5933600e+01,5.9510330e+02, &
& 5.0296540e+02,4.1774000e+02,4.0534620e+02,3.7223260e+02,2.9326800e+02, &
& 3.2176790e+02,2.5267400e+02,2.6948070e+02,2.7679850e+02,2.1213400e+02, &
& 2.1980160e+02,2.5991840e+02,2.3175640e+02,1.9950090e+02,1.7993740e+02]
 vdw_dftd3_c6(28001:28100)=[&
& 1.5821040e+02,1.3810760e+02,6.6844010e+02,5.9874980e+02,5.2942130e+02, &
& 4.7851960e+02,4.3792800e+02,3.4005920e+02,3.7860580e+02,2.9020770e+02, &
& 3.1726290e+02,2.9479860e+02,2.4529960e+02,2.6038920e+02,3.2473460e+02, &
& 3.0300330e+02,2.7228130e+02,2.5382690e+02,2.3058600e+02,2.0815070e+02, &
& 8.1515180e+02,7.6114330e+02,6.7494890e+02,3.2113780e+02,6.7620310e+02, &
& 6.5026390e+02,6.3420890e+02,6.1941760e+02,6.0631410e+02,4.8160310e+02, &
& 5.3441950e+02,5.1638640e+02,5.4808770e+02,5.3658580e+02,5.2628420e+02, &
& 5.1991030e+02,4.4109130e+02,4.3800800e+02,4.0185550e+02,3.4062380e+02, &
& 3.4722720e+02,3.1601980e+02,2.9035810e+02,2.4187400e+02,2.2620530e+02, &
& 2.3313640e+02,3.3397090e+02,3.2879450e+02,3.0437900e+02,2.9151300e+02, &
& 2.7029090e+02,2.4870320e+02,7.7546700e+02,7.5632330e+02,6.7406280e+02, &
& 6.1138240e+02,6.0405420e+02,5.8496010e+02,5.9856780e+02,5.8026720e+02, &
& 3.3919300e+01,1.0684660e+02,1.3686740e+02,1.0639230e+02,8.0849600e+01, &
& 5.6650800e+01,4.1460000e+01,2.8636200e+01,1.5544950e+02,2.4015820e+02, &
& 2.4565290e+02,2.0014220e+02,1.6545180e+02,1.4069580e+02,1.1560060e+02, &
& 2.1584670e+02,4.0282860e+02,2.1843190e+02,2.1092280e+02,2.0662740e+02, &
& 2.0440810e+02,1.8931020e+02,1.7549190e+02,1.6737650e+02,1.6353010e+02, &
& 1.6072250e+02,1.5220000e+02,2.4479640e+02,2.1733500e+02,1.9621220e+02, &
& 1.7998620e+02,1.5926660e+02,2.5815200e+02,4.8959430e+02,3.8114140e+02, &
& 2.8866370e+02,2.9138130e+02,2.7224210e+02,3.0274930e+02,2.3860140e+02]
 vdw_dftd3_c6(28101:28200)=[&
& 2.2325490e+02,2.0745050e+02,2.0043480e+02,1.9927140e+02,3.1050480e+02, &
& 2.8348980e+02,2.6714010e+02,2.5372280e+02,2.3274260e+02,3.0765740e+02, &
& 6.3261590e+02,4.8173950e+02,3.1158240e+02,3.0515280e+02,2.9558710e+02, &
& 2.9668160e+02,2.8470850e+02,2.9847730e+02,2.8058560e+02,2.8487000e+02, &
& 2.6661130e+02,2.5911400e+02,2.5737860e+02,2.7013590e+02,2.4893750e+02, &
& 3.3258120e+02,3.1015250e+02,2.8435150e+02,2.8639060e+02,2.5354990e+02, &
& 2.3881810e+02,2.2823770e+02,2.1805720e+02,2.1157770e+02,3.3379720e+02, &
& 3.0690290e+02,2.9903800e+02,2.9178570e+02,2.7360270e+02,3.4671520e+02, &
& 6.3297580e+02,3.6867540e+02,4.2237050e+02,3.7847190e+02,3.3627890e+02, &
& 3.2395480e+02,3.8367070e+02,9.1020600e+01,8.9639100e+01,6.6234300e+01, &
& 5.3207600e+01,3.6587500e+01,1.5995620e+02,1.9311320e+02,1.8650310e+02, &
& 1.6307640e+02,1.3785400e+02,2.2395660e+02,2.1640380e+02,2.1913910e+02, &
& 2.0040530e+02,1.5200100e+02,1.3062480e+02,1.2812710e+02,1.4911580e+02, &
& 1.3922010e+02,1.9576670e+02,2.0723900e+02,1.9122650e+02,1.7859070e+02, &
& 2.6845260e+02,2.6022540e+02,2.6296750e+02,2.5404600e+02,2.2570300e+02, &
& 1.9994180e+02,1.8915600e+02,1.9169620e+02,1.9999790e+02,2.5244560e+02, &
& 2.7452930e+02,2.6015780e+02,2.5179490e+02,3.2533280e+02,3.3853620e+02, &
& 2.5238610e+02,2.6287220e+02,2.4530820e+02,2.2353440e+02,2.1084220e+02, &
& 2.1397300e+02,2.2500270e+02,2.7049420e+02,2.8607870e+02,2.9230220e+02, &
& 2.9076120e+02,3.5840590e+02,6.8800700e+01,6.1943300e+01,4.6863500e+01]
 vdw_dftd3_c6(28201:28300)=[&
& 1.5204520e+02,1.8148400e+02,1.5168860e+02,1.8750000e+01,1.2752300e+01, &
& 2.7075980e+02,1.6088700e+02,1.1053650e+02,7.6007200e+01,5.3969500e+01, &
& 4.1362300e+01,3.1698500e+01,2.4634300e+01,3.2462410e+02,2.5515610e+02, &
& 2.3739610e+02,1.8954880e+02,1.4967090e+02,1.2537250e+02,1.0339310e+02, &
& 8.5361600e+01,5.3130440e+02,4.4506810e+02,3.6908350e+02,3.5797830e+02, &
& 3.2864070e+02,2.5957220e+02,2.8401670e+02,2.2363540e+02,2.3769940e+02, &
& 2.4420530e+02,1.8784400e+02,1.9385700e+02,2.2867850e+02,2.0362780e+02, &
& 1.7541210e+02,1.5849070e+02,1.3975410e+02,1.2245360e+02,5.9683060e+02, &
& 5.3035890e+02,4.6786120e+02,4.2252080e+02,3.8664910e+02,3.0067530e+02, &
& 3.3455830e+02,2.5692950e+02,2.8039530e+02,2.6063370e+02,2.1771120e+02, &
& 2.3040100e+02,2.8692990e+02,2.6720820e+02,2.4003960e+02,2.2389640e+02, &
& 2.0368550e+02,1.8426650e+02,7.2757110e+02,6.7509330e+02,5.9711540e+02, &
& 2.8341080e+02,5.9987830e+02,5.7662050e+02,5.6231220e+02,5.4912780e+02, &
& 5.3743950e+02,4.2647230e+02,4.7530650e+02,4.5903360e+02,4.8549610e+02, &
& 4.7524580e+02,4.6604870e+02,4.6040630e+02,3.9029370e+02,3.8633500e+02, &
& 3.5438040e+02,3.0091280e+02,3.0657700e+02,2.7917480e+02,2.5671260e+02, &
& 2.1450850e+02,2.0089400e+02,2.0681470e+02,2.9586670e+02,2.9066840e+02, &
& 2.6889430e+02,2.5754500e+02,2.3900100e+02,2.2025450e+02,6.9085290e+02, &
& 6.7010980e+02,5.9610010e+02,5.4016650e+02,5.3472260e+02,5.1792390e+02, &
& 5.3080430e+02,5.1445080e+02,2.9768000e+01,9.3835600e+01,1.2018800e+02]
 vdw_dftd3_c6(28301:28400)=[&
& 9.3454100e+01,7.1331800e+01,5.0381400e+01,3.7217800e+01,2.6103100e+01, &
& 1.3698410e+02,2.1133880e+02,2.1570660e+02,1.7559630e+02,1.4538010e+02, &
& 1.2399050e+02,1.0232230e+02,1.9095210e+02,3.5707210e+02,1.9252210e+02, &
& 1.8603690e+02,1.8233530e+02,1.8047740e+02,1.6694170e+02,1.5486080e+02, &
& 1.4780570e+02,1.4445180e+02,1.4216340e+02,1.3437060e+02,2.1533740e+02, &
& 1.9100140e+02,1.7255780e+02,1.5854630e+02,1.4068120e+02,2.2851060e+02, &
& 4.3445030e+02,3.3691730e+02,2.5466980e+02,2.5719610e+02,2.4051390e+02, &
& 2.6786660e+02,2.1097970e+02,1.9765180e+02,1.8389000e+02,1.7779340e+02, &
& 1.7646970e+02,2.7425370e+02,2.5002640e+02,2.3552760e+02,2.2379990e+02, &
& 2.0556330e+02,2.7160040e+02,5.6214810e+02,4.2601980e+02,2.7502300e+02, &
& 2.6936720e+02,2.6096100e+02,2.6192990e+02,2.5167010e+02,2.6338860e+02, &
& 2.4770800e+02,2.5153730e+02,2.3534260e+02,2.2872630e+02,2.2716850e+02, &
& 2.3821710e+02,2.1967420e+02,2.9356720e+02,2.7385420e+02,2.5123200e+02, &
& 2.5317100e+02,2.2427020e+02,2.1152860e+02,2.0238660e+02,1.9363610e+02, &
& 1.8779620e+02,2.9544290e+02,2.7127660e+02,2.6414600e+02,2.5774500e+02, &
& 2.4187370e+02,3.0597750e+02,5.6160150e+02,3.2525320e+02,3.7317930e+02, &
& 3.3481340e+02,2.9740820e+02,2.8658400e+02,3.4006000e+02,7.9784500e+01, &
& 7.8862400e+01,5.8605700e+01,4.7416800e+01,3.3021500e+01,1.4035180e+02, &
& 1.6949030e+02,1.6367210e+02,1.4335180e+02,1.2152560e+02,1.9732010e+02, &
& 1.9055180e+02,1.9300120e+02,1.7676920e+02,1.3462870e+02,1.1592230e+02]
 vdw_dftd3_c6(28401:28500)=[&
& 1.1372870e+02,1.3200850e+02,1.2331710e+02,1.7220040e+02,1.8217310e+02, &
& 1.6821430e+02,1.5734260e+02,2.3676080e+02,2.2925270e+02,2.3167830e+02, &
& 2.2415900e+02,1.9953250e+02,1.7710190e+02,1.6774480e+02,1.7007730e+02, &
& 1.7730100e+02,2.2299250e+02,2.4213510e+02,2.2940100e+02,2.2211080e+02, &
& 2.8697840e+02,2.9814430e+02,2.2247740e+02,2.3203320e+02,2.1683460e+02, &
& 1.9794790e+02,1.8697210e+02,1.8993130e+02,1.9953230e+02,2.3939990e+02, &
& 2.5289420e+02,2.5818280e+02,2.5681660e+02,3.1607620e+02,6.0653700e+01, &
& 5.4906600e+01,4.1919300e+01,1.3338200e+02,1.5940630e+02,1.3339560e+02, &
& 1.1828630e+02,2.0185300e+01,1.3430500e+01,3.2065850e+02,1.8269550e+02, &
& 1.2239150e+02,8.2593000e+01,5.7846900e+01,4.3916100e+01,3.3392400e+01, &
& 2.5794000e+01,3.8337730e+02,2.9192630e+02,2.6812070e+02,2.1041880e+02, &
& 1.6372990e+02,1.3584700e+02,1.1102480e+02,9.0955300e+01,6.3014120e+02, &
& 5.1554150e+02,4.2512770e+02,4.1001290e+02,3.7517450e+02,2.9585990e+02, &
& 3.2268450e+02,2.5353650e+02,2.6794820e+02,2.7623270e+02,2.1214340e+02, &
& 2.1694470e+02,2.5679600e+02,2.2570800e+02,1.9211030e+02,1.7227180e+02, &
& 1.5074260e+02,1.3116000e+02,7.0593280e+02,6.1461390e+02,5.3667450e+02, &
& 4.8150000e+02,4.3865020e+02,3.3835350e+02,3.7763020e+02,2.8746140e+02, &
& 3.1395770e+02,2.9103090e+02,2.4320090e+02,2.5636430e+02,3.2207710e+02, &
& 2.9688720e+02,2.6394960e+02,2.4463100e+02,2.2102500e+02,1.9865840e+02, &
& 8.5972520e+02,7.8452520e+02,6.8699020e+02,3.1211020e+02,6.9469690e+02]
 vdw_dftd3_c6(28501:28600)=[&
& 6.6673430e+02,6.4990940e+02,6.3444150e+02,6.2071840e+02,4.8671800e+02, &
& 5.5092420e+02,5.3097990e+02,5.5938190e+02,5.4744560e+02,5.3665810e+02, &
& 5.3043510e+02,4.4599990e+02,4.3700330e+02,3.9822740e+02,3.3618450e+02, &
& 3.4173640e+02,3.0943260e+02,2.8322570e+02,2.3539680e+02,2.1999670e+02, &
& 2.2609290e+02,3.3090130e+02,3.2264990e+02,2.9578510e+02,2.8178430e+02, &
& 2.5980700e+02,2.3801290e+02,8.0981600e+02,7.7454730e+02,6.8271560e+02, &
& 6.1185740e+02,6.0871940e+02,5.8942580e+02,6.0818860e+02,5.8874820e+02, &
& 3.2514900e+01,1.0615450e+02,1.3527020e+02,1.0292200e+02,7.7512400e+01, &
& 5.3969000e+01,3.9412500e+01,2.7270500e+01,1.5557860e+02,2.4053020e+02, &
& 2.4284190e+02,1.9441840e+02,1.5894530e+02,1.3437370e+02,1.0989000e+02, &
& 2.1449610e+02,4.1261420e+02,2.1361290e+02,2.0626510e+02,2.0220500e+02, &
& 2.0045980e+02,1.8404780e+02,1.7031170e+02,1.6252860e+02,1.5894050e+02, &
& 1.5733960e+02,1.4715970e+02,2.4100160e+02,2.1104400e+02,1.8884500e+02, &
& 1.7234930e+02,1.5180060e+02,2.5585120e+02,5.0246340e+02,3.8179030e+02, &
& 2.8270210e+02,2.8559450e+02,2.6625680e+02,2.9944610e+02,2.3182850e+02, &
& 2.1698060e+02,2.0155360e+02,1.9509170e+02,1.9260990e+02,3.0671200e+02, &
& 2.7663930e+02,2.5861940e+02,2.4442920e+02,2.2308960e+02,3.0164000e+02, &
& 6.5239960e+02,4.8348120e+02,3.0268200e+02,2.9642470e+02,2.8701630e+02, &
& 2.8852060e+02,2.7859570e+02,2.9030390e+02,2.7264920e+02,2.7761700e+02, &
& 2.5869870e+02,2.5129970e+02,2.4969380e+02,2.6246920e+02,2.4127880e+02]
 vdw_dftd3_c6(28601:28700)=[&
& 3.2823180e+02,3.0426470e+02,2.7731590e+02,2.8103260e+02,2.4528440e+02, &
& 2.3080460e+02,2.2051410e+02,2.1098620e+02,2.0359070e+02,3.2952070e+02, &
& 2.9946420e+02,2.8996580e+02,2.8177690e+02,2.6291680e+02,3.3961520e+02, &
& 6.4757740e+02,3.5809400e+02,4.1447150e+02,3.7053510e+02,3.2689030e+02, &
& 3.1449130e+02,3.8028110e+02,8.7785200e+01,8.6389500e+01,6.3209200e+01, &
& 5.0719800e+01,3.4859600e+01,1.5613270e+02,1.8876990e+02,1.8076730e+02, &
& 1.5685490e+02,1.3167410e+02,2.1936790e+02,2.1045590e+02,2.1305010e+02, &
& 1.9516290e+02,1.4698850e+02,1.2572050e+02,1.2344170e+02,1.4484240e+02, &
& 1.3489330e+02,1.9047990e+02,2.0085740e+02,1.8386230e+02,1.7100310e+02, &
& 2.6289820e+02,2.5242450e+02,2.5445670e+02,2.4630770e+02,2.1764570e+02, &
& 1.9205910e+02,1.8151960e+02,1.8521540e+02,1.9349000e+02,2.4615700e+02, &
& 2.6732250e+02,2.5143010e+02,2.4241490e+02,3.1782550e+02,3.2883880e+02, &
& 2.4349370e+02,2.5469760e+02,2.3640400e+02,2.1467590e+02,2.0191630e+02, &
& 2.0622720e+02,2.1681400e+02,2.6294210e+02,2.7771100e+02,2.8277180e+02, &
& 2.8049880e+02,3.4977880e+02,6.5561800e+01,5.9198000e+01,4.4744200e+01, &
& 1.4661210e+02,1.7631430e+02,1.4541330e+02,1.2886410e+02,1.4187630e+02, &
& 2.2539000e+01,1.4929600e+01,3.3234640e+02,1.9774540e+02,1.3488560e+02, &
& 9.1766100e+01,6.4397700e+01,4.8831000e+01,3.7012900e+01,2.8465400e+01, &
& 3.9801800e+02,3.1346580e+02,2.9117040e+02,2.3154320e+02,1.8169810e+02, &
& 1.5126110e+02,1.2384420e+02,1.0145720e+02,6.4868790e+02,5.4545480e+02]
 vdw_dftd3_c6(28701:28800)=[&
& 4.5241480e+02,4.3828120e+02,4.0212020e+02,3.1661340e+02,3.4715450e+02, &
& 2.7239980e+02,2.9012680e+02,2.9831410e+02,2.2848110e+02,2.3616610e+02, &
& 2.7943460e+02,2.4818660e+02,2.1282990e+02,1.9148490e+02,1.6795000e+02, &
& 1.4629100e+02,7.2799780e+02,6.4926400e+02,5.7254520e+02,5.1656560e+02, &
& 4.7214060e+02,3.6573900e+02,4.0759020e+02,3.1162540e+02,3.4083280e+02, &
& 3.1647210e+02,2.6332580e+02,2.7928860e+02,3.4913300e+02,3.2482290e+02, &
& 2.9096000e+02,2.7067740e+02,2.4534690e+02,2.2101200e+02,8.8720920e+02, &
& 8.2570960e+02,7.3032590e+02,3.4326010e+02,7.3303460e+02,7.0467870e+02, &
& 6.8722230e+02,6.7114820e+02,6.5690430e+02,5.2004840e+02,5.7921270e+02, &
& 5.5935900e+02,5.9350440e+02,5.8102860e+02,5.6982860e+02,5.6301640e+02, &
& 4.7657670e+02,4.7186840e+02,4.3210850e+02,3.6558740e+02,3.7247260e+02, &
& 3.3845240e+02,3.1056410e+02,2.5829950e+02,2.4143850e+02,2.4873340e+02, &
& 3.5861390e+02,3.5236140e+02,3.2532280e+02,3.1104340e+02,2.8780550e+02, &
& 2.6431470e+02,8.4229970e+02,8.1926690e+02,7.2858620e+02,6.5887100e+02, &
& 6.5182590e+02,6.3119280e+02,6.4715530e+02,6.2718910e+02,3.6144700e+01, &
& 1.1512840e+02,1.4719230e+02,1.1367480e+02,8.6017200e+01,6.0022900e+01, &
& 4.3796200e+01,3.0155700e+01,1.6772780e+02,2.5920610e+02,2.6429930e+02, &
& 2.1422700e+02,1.7638160e+02,1.4956740e+02,1.2254290e+02,2.3199890e+02, &
& 4.3628440e+02,2.3401650e+02,2.2591360e+02,2.2133820e+02,2.1907040e+02, &
& 2.0247400e+02,1.8755800e+02,1.7887740e+02,1.7480580e+02,1.7212150e+02]
 vdw_dftd3_c6(28801:28900)=[&
& 1.6247350e+02,2.6291340e+02,2.3250850e+02,2.0926950e+02,1.9154240e+02, &
& 1.6909250e+02,2.7717010e+02,5.3024480e+02,4.1052730e+02,3.0919640e+02, &
& 3.1215680e+02,2.9139440e+02,3.2490950e+02,2.5488120e+02,2.3843110e+02, &
& 2.2146490e+02,2.1407940e+02,2.1251480e+02,3.3348150e+02,3.0351790e+02, &
& 2.8533630e+02,2.7052970e+02,2.4764850e+02,3.2944100e+02,6.8550020e+02, &
& 5.1898320e+02,3.3298690e+02,3.2610980e+02,3.1583740e+02,3.1714450e+02, &
& 3.0481740e+02,3.1912880e+02,2.9989300e+02,3.0469800e+02,2.8484920e+02, &
& 2.7680030e+02,2.7498180e+02,2.8882640e+02,2.6590550e+02,3.5695090e+02, &
& 3.3227960e+02,3.0407910e+02,3.0677590e+02,2.7045010e+02,2.5457310e+02, &
& 2.4321020e+02,2.3238620e+02,2.2515610e+02,3.5816930e+02,3.2836650e+02, &
& 3.1941280e+02,3.1125260e+02,2.9132130e+02,3.7119590e+02,6.8477170e+02, &
& 3.9403910e+02,4.5247860e+02,4.0499240e+02,3.5926250e+02,3.4595630e+02, &
& 4.1186560e+02,9.7234600e+01,9.5603900e+01,7.0318400e+01,5.6361200e+01, &
& 3.8626300e+01,1.7153850e+02,2.0708710e+02,1.9947600e+02,1.7389310e+02, &
& 1.4653820e+02,2.4008000e+02,2.3151840e+02,2.3440930e+02,2.1438080e+02, &
& 1.6205470e+02,1.3899280e+02,1.3638180e+02,1.5925840e+02,1.4856420e+02, &
& 2.0951300e+02,2.2156120e+02,2.0387080e+02,1.9004420e+02,2.8761670e+02, &
& 2.7811250e+02,2.8082410e+02,2.7131370e+02,2.4051090e+02,2.1269370e+02, &
& 2.0110400e+02,2.0422770e+02,2.1323450e+02,2.7009880e+02,2.9372900e+02, &
& 2.7771860e+02,2.6841360e+02,3.4819780e+02,3.6193370e+02,2.6924010e+02]
 vdw_dftd3_c6(28901:29000)=[&
& 2.8063840e+02,2.6136250e+02,2.3778890e+02,2.2400480e+02,2.2773200e+02, &
& 2.3955770e+02,2.8897680e+02,3.0561160e+02,3.1199930e+02,3.1006550e+02, &
& 3.8347400e+02,7.3075400e+01,6.5758400e+01,4.9615900e+01,1.6240650e+02, &
& 1.9417780e+02,1.6155910e+02,1.4216340e+02,1.5548130e+02,1.7231530e+02, &
& 2.5925500e+01,1.7061400e+01,3.9241090e+02,2.3062870e+02,1.5627830e+02, &
& 1.0580040e+02,7.3967500e+01,5.5931400e+01,4.2287900e+01,3.2451500e+01, &
& 4.6954960e+02,3.6633190e+02,3.3910340e+02,2.6843700e+02,2.0985610e+02, &
& 1.7427160e+02,1.4234130e+02,1.1635890e+02,7.6657140e+02,6.3980000e+02, &
& 5.2977990e+02,5.1242740e+02,4.6971220e+02,3.6970490e+02,4.0496220e+02, &
& 3.1759430e+02,3.3769310e+02,3.4754450e+02,2.6609200e+02,2.7433390e+02, &
& 3.2494560e+02,2.8761030e+02,2.4587190e+02,2.2078460e+02,1.9325910e+02, &
& 1.6801940e+02,8.5965830e+02,7.6173640e+02,6.6974730e+02,6.0315190e+02, &
& 5.5060110e+02,4.2558490e+02,4.7466380e+02,3.6203660e+02,3.9601830e+02, &
& 3.6742870e+02,3.0577550e+02,3.2392920e+02,4.0593030e+02,3.7661080e+02, &
& 3.3642890e+02,3.1246310e+02,2.8271950e+02,2.5424580e+02,1.0475077e+03, &
& 9.6965590e+02,8.5512980e+02,3.9708240e+02,8.5990430e+02,8.2624710e+02, &
& 8.0566910e+02,7.8673400e+02,7.6995120e+02,6.0750730e+02,6.7977520e+02, &
& 6.5608890e+02,6.9513050e+02,6.8046640e+02,6.6727580e+02,6.5939100e+02, &
& 5.5687690e+02,5.4981880e+02,5.0258340e+02,4.2456930e+02,4.3228120e+02, &
& 3.9218940e+02,3.5941840e+02,2.9849470e+02,2.7884190e+02,2.8712700e+02]
 vdw_dftd3_c6(29001:29100)=[&
& 4.1657100e+02,4.0843950e+02,3.7618150e+02,3.5916750e+02,3.3177440e+02, &
& 3.0422370e+02,9.9208740e+02,9.6060620e+02,8.5194620e+02,7.6802670e+02, &
& 7.6090240e+02,7.3674860e+02,7.5679710e+02,7.3318910e+02,4.1732300e+01, &
& 1.3414080e+02,1.7128910e+02,1.3152340e+02,9.9176300e+01,6.8932300e+01, &
& 5.0126900e+01,3.4364200e+01,1.9562310e+02,3.0252130e+02,3.0755040e+02, &
& 2.4819650e+02,2.0368650e+02,1.7232720e+02,1.4084920e+02,2.6984030e+02, &
& 5.1153740e+02,2.7124750e+02,2.6180740e+02,2.5651490e+02,2.5399050e+02, &
& 2.3426800e+02,2.1686750e+02,2.0682100e+02,2.0214420e+02,1.9934300e+02, &
& 1.8764810e+02,3.0545790e+02,2.6922150e+02,2.4171350e+02,2.2085650e+02, &
& 1.9459280e+02,3.2210960e+02,6.2192630e+02,4.7871320e+02,3.5847720e+02, &
& 3.6192950e+02,3.3757750e+02,3.7742760e+02,2.9466340e+02,2.7557070e+02, &
& 2.5584410e+02,2.4737280e+02,2.4521190e+02,3.8734040e+02,3.5152010e+02, &
& 3.2980470e+02,3.1226190e+02,2.8538200e+02,3.8205620e+02,8.0495990e+02, &
& 6.0548420e+02,3.8513200e+02,3.7716470e+02,3.6522950e+02,3.6689290e+02, &
& 3.5308810e+02,3.6924660e+02,3.4685640e+02,3.5267810e+02,3.2934030e+02, &
& 3.1999040e+02,3.1792410e+02,3.3414390e+02,3.0737050e+02,4.1466360e+02, &
& 3.8533830e+02,3.5200450e+02,3.5566060e+02,3.1228140e+02,2.9375400e+02, &
& 2.8052440e+02,2.6802920e+02,2.5933470e+02,4.1573130e+02,3.8005970e+02, &
& 3.6914810e+02,3.5933580e+02,3.3582630e+02,4.3041160e+02,8.0257380e+02, &
& 4.5579310e+02,5.2467090e+02,4.6916950e+02,4.1536370e+02,3.9979830e+02]
 vdw_dftd3_c6(29101:29200)=[&
& 4.7845560e+02,1.1247080e+02,1.1046530e+02,8.0908400e+01,6.4694600e+01, &
& 4.4161100e+01,1.9895420e+02,2.4029350e+02,2.3095630e+02,2.0085430e+02, &
& 1.6882630e+02,2.7842050e+02,2.6802690e+02,2.7133540e+02,2.4816060e+02, &
& 1.8702850e+02,1.6012070e+02,1.5714100e+02,1.8402720e+02,1.7152460e+02, &
& 2.4266150e+02,2.5640090e+02,2.3540190e+02,2.1911610e+02,3.3346480e+02, &
& 3.2171900e+02,3.2464310e+02,3.1369090e+02,2.7753210e+02,2.4504710e+02, &
& 2.3155270e+02,2.3553230e+02,2.4604900e+02,3.1261830e+02,3.3998590e+02, &
& 3.2084600e+02,3.0976360e+02,4.0345600e+02,4.1889570e+02,3.1096060e+02, &
& 3.2438890e+02,3.0155580e+02,2.7396610e+02,2.5778980e+02,2.6244250e+02, &
& 2.7611510e+02,3.3401410e+02,3.5322530e+02,3.6036290e+02,3.5787940e+02, &
& 4.4425160e+02,8.4139800e+01,7.5651400e+01,5.6913400e+01,1.8779330e+02, &
& 2.2491460e+02,1.8642580e+02,1.6395940e+02,1.7981780e+02,1.9898010e+02, &
& 2.2994980e+02,2.5814400e+01,1.7190000e+01,3.7309010e+02,2.2360990e+02, &
& 1.5340550e+02,1.0484620e+02,7.3824600e+01,5.6105700e+01,4.2601300e+01, &
& 3.2802100e+01,4.4701790e+02,3.5391780e+02,3.2966240e+02,2.6315950e+02, &
& 2.0724190e+02,1.7294560e+02,1.4192450e+02,1.1649640e+02,7.2861950e+02, &
& 6.1467050e+02,5.1029120e+02,4.9492760e+02,4.5438220e+02,3.5797570e+02, &
& 3.9264080e+02,3.0830750e+02,3.2864030e+02,3.3766070e+02,2.5878300e+02, &
& 2.6791080e+02,3.1682010e+02,2.8218280e+02,2.4267360e+02,2.1875310e+02, &
& 1.9224210e+02,1.6775050e+02,8.1822470e+02,7.3174620e+02,6.4647740e+02]
 vdw_dftd3_c6(29201:29300)=[&
& 5.8401600e+02,5.3429150e+02,4.1465090e+02,4.6175770e+02,3.5373830e+02, &
& 3.8672900e+02,3.5928420e+02,2.9901370e+02,3.1728800e+02,3.9591570e+02, &
& 3.6910130e+02,3.3139460e+02,3.0877390e+02,2.8035770e+02,2.5296780e+02, &
& 9.9762710e+02,9.3038380e+02,8.2434880e+02,3.9090330e+02,8.2640440e+02, &
& 7.9461360e+02,7.7497180e+02,7.5687850e+02,7.4084790e+02,5.8791190e+02, &
& 6.5318150e+02,6.3103160e+02,6.6958290e+02,6.5552070e+02,6.4291690e+02, &
& 6.3515750e+02,5.3851890e+02,5.3425460e+02,4.8990920e+02,4.1509390e+02, &
& 4.2306850e+02,3.8489200e+02,3.5353050e+02,2.9442160e+02,2.7532990e+02, &
& 2.8371810e+02,4.0709580e+02,4.0053490e+02,3.7051780e+02,3.5469940e+02, &
& 3.2871350e+02,3.0233200e+02,9.4844440e+02,9.2407530e+02,8.2301470e+02, &
& 7.4585710e+02,7.3724570e+02,7.1393940e+02,7.3098970e+02,7.0857740e+02, &
& 4.1243400e+01,1.3030340e+02,1.6682950e+02,1.2946150e+02,9.8293000e+01, &
& 6.8826400e+01,5.0355500e+01,3.4778500e+01,1.8967700e+02,2.9304400e+02, &
& 2.9946280e+02,2.4364100e+02,2.0121190e+02,1.7100300e+02,1.4042820e+02, &
& 2.6315470e+02,4.9217040e+02,2.6602940e+02,2.5687630e+02,2.5165870e+02, &
& 2.4899550e+02,2.3046590e+02,2.1360980e+02,2.0373690e+02,1.9906950e+02, &
& 1.9575950e+02,1.8520600e+02,2.9830600e+02,2.6455660e+02,2.3866010e+02, &
& 2.1881490e+02,1.9353150e+02,3.1465180e+02,5.9821020e+02,4.6492540e+02, &
& 3.5156410e+02,3.5489700e+02,3.3152300e+02,3.6896070e+02,2.9041810e+02, &
& 2.7173870e+02,2.5249180e+02,2.4399260e+02,2.4246050e+02,3.7845390e+02]
 vdw_dftd3_c6(29301:29400)=[&
& 3.4521760e+02,3.2509990e+02,3.0863590e+02,2.8298000e+02,3.7467690e+02, &
& 7.7312930e+02,5.8768050e+02,3.7925720e+02,3.7143070e+02,3.5977520e+02, &
& 3.6114910e+02,3.4673650e+02,3.6334580e+02,3.4154120e+02,3.4682760e+02, &
& 3.2449690e+02,3.1536050e+02,3.1325730e+02,3.2883440e+02,3.0296350e+02, &
& 4.0527710e+02,3.7777130e+02,3.4618800e+02,3.4883500e+02,3.0849820e+02, &
& 2.9054470e+02,2.7766310e+02,2.6530450e+02,2.5731840e+02,4.0678760e+02, &
& 3.7370420e+02,3.6395410e+02,3.5500400e+02,3.3273420e+02,4.2221560e+02, &
& 7.7316810e+02,4.4875130e+02,5.1446500e+02,4.6088700e+02,4.0932390e+02, &
& 3.9428590e+02,4.6765480e+02,1.1074050e+02,1.0903310e+02,8.0490200e+01, &
& 6.4644600e+01,4.4441700e+01,1.9481980e+02,2.3520410e+02,2.2699490e+02, &
& 1.9834020e+02,1.6754910e+02,2.7279590e+02,2.6344960e+02,2.6677170e+02, &
& 2.4398790e+02,1.8493070e+02,1.5885780e+02,1.5583590e+02,1.8150050e+02, &
& 1.6942270e+02,2.3833640e+02,2.5222610e+02,2.3257440e+02,2.1711610e+02, &
& 3.2696240e+02,3.1671890e+02,3.1999160e+02,3.0916430e+02,2.7453760e+02, &
& 2.4311650e+02,2.2998060e+02,2.3320160e+02,2.4333990e+02,3.0738320e+02, &
& 3.3424830e+02,3.1655690e+02,3.0627350e+02,3.9613570e+02,4.1206100e+02, &
& 3.0703930e+02,3.1988160e+02,2.9837090e+02,2.7179960e+02,2.5630050e+02, &
& 2.6023930e+02,2.7366690e+02,3.2926130e+02,3.4820710e+02,3.5568970e+02, &
& 3.5372780e+02,4.3636490e+02,8.3603600e+01,7.5280800e+01,5.6939800e+01, &
& 1.8498860e+02,2.2091650e+02,1.8443640e+02,1.6228780e+02,1.7705570e+02]
 vdw_dftd3_c6(29401:29500)=[&
& 1.9651790e+02,2.2680190e+02,2.2428510e+02,3.0319300e+01,2.0340200e+01, &
& 4.4469780e+02,2.6404390e+02,1.8045680e+02,1.2327050e+02,8.6956900e+01, &
& 6.6274100e+01,5.0515600e+01,3.9067400e+01,5.3285380e+02,4.1885670e+02, &
& 3.8906980e+02,3.0967790e+02,2.4351060e+02,2.0320970e+02,1.6688670e+02, &
& 1.3720100e+02,8.6995980e+02,7.2995940e+02,6.0530470e+02,5.8657520e+02, &
& 5.3826720e+02,4.2440160e+02,4.6484110e+02,3.6531290e+02,3.8862700e+02, &
& 3.9951060e+02,3.0659770e+02,3.1653820e+02,3.7394790e+02,3.3229320e+02, &
& 2.8535990e+02,2.5714220e+02,2.2602240e+02,1.9737620e+02,9.7661940e+02, &
& 8.6927450e+02,7.6640590e+02,6.9159970e+02,6.3235800e+02,4.9057200e+02, &
& 5.4639300e+02,4.1848510e+02,4.5727800e+02,4.2477830e+02,3.5412210e+02, &
& 3.7514860e+02,4.6825210e+02,4.3562520e+02,3.9051070e+02,3.6359740e+02, &
& 3.3001060e+02,2.9777960e+02,1.1900927e+03,1.1059339e+03,9.7785820e+02, &
& 4.6088570e+02,9.8223880e+02,9.4418200e+02,9.2077280e+02,8.9921070e+02, &
& 8.8009790e+02,6.9717420e+02,7.7703650e+02,7.5035490e+02,7.9506940e+02, &
& 7.7832350e+02,7.6328690e+02,7.5412760e+02,6.3857600e+02,6.3185040e+02, &
& 5.7890360e+02,4.9047740e+02,4.9968040e+02,4.5440140e+02,4.1730580e+02, &
& 3.4779410e+02,3.2538710e+02,3.3507650e+02,4.8176440e+02,4.7318850e+02, &
& 4.3707980e+02,4.1810900e+02,3.8726470e+02,3.5612660e+02,1.1296762e+03, &
& 1.0972829e+03,9.7575910e+02,8.8288780e+02,8.7388960e+02,8.4633640e+02, &
& 8.6787930e+02,8.4109640e+02,4.8427000e+01,1.5382640e+02,1.9674540e+02]
 vdw_dftd3_c6(29501:29600)=[&
& 1.5226150e+02,1.1561300e+02,8.1111000e+01,5.9534400e+01,4.1374600e+01, &
& 2.2440970e+02,3.4649400e+02,3.5326540e+02,2.8664120e+02,2.3644610e+02, &
& 2.0095580e+02,1.6514950e+02,3.1131870e+02,5.8448340e+02,3.1379200e+02, &
& 3.0305660e+02,2.9698540e+02,2.9398400e+02,2.7172730e+02,2.5185120e+02, &
& 2.4028600e+02,2.3483900e+02,2.3127000e+02,2.1830270e+02,3.5191540e+02, &
& 3.1142600e+02,2.8063500e+02,2.5722750e+02,2.2754720e+02,3.7215270e+02, &
& 7.1067900e+02,5.5021640e+02,4.1478140e+02,4.1884820e+02,3.9127220e+02, &
& 4.3621610e+02,3.4263080e+02,3.2074540e+02,2.9815620e+02,2.8828650e+02, &
& 2.8607710e+02,4.4732690e+02,4.0722750e+02,3.8302800e+02,3.6340770e+02, &
& 3.3308010e+02,4.4205370e+02,9.1909280e+02,6.9561840e+02,4.4715510e+02, &
& 4.3794100e+02,4.2419840e+02,4.2589770e+02,4.0941900e+02,4.2843500e+02, &
& 4.0274990e+02,4.0914080e+02,3.8256860e+02,3.7177680e+02,3.6929910e+02, &
& 3.8763420e+02,3.5709720e+02,4.7865480e+02,4.4589750e+02,4.0842780e+02, &
& 4.1195430e+02,3.6377170e+02,3.4272900e+02,3.2766460e+02,3.1331210e+02, &
& 3.0363280e+02,4.8106690e+02,4.4115490e+02,4.2919400e+02,4.1838610e+02, &
& 3.9195070e+02,4.9802340e+02,9.1797160e+02,5.2898700e+02,6.0741450e+02, &
& 5.4418980e+02,4.8298360e+02,4.6522570e+02,5.5343860e+02,1.3011390e+02, &
& 1.2821910e+02,9.4715300e+01,7.6253000e+01,5.2669000e+01,2.2944250e+02, &
& 2.7699400e+02,2.6699740e+02,2.3314220e+02,1.9692250e+02,3.2180650e+02, &
& 3.1041930e+02,3.1434380e+02,2.8770260e+02,2.1815660e+02,1.8741200e+02]
 vdw_dftd3_c6(29601:29700)=[&
& 1.8389760e+02,2.1424530e+02,1.9997120e+02,2.8072610e+02,2.9685620e+02, &
& 2.7345920e+02,2.5524180e+02,3.8575940e+02,3.7308750e+02,3.7681730e+02, &
& 3.6432260e+02,3.2348820e+02,2.8650850e+02,2.7111330e+02,2.7523510e+02, &
& 2.8720260e+02,3.6275140e+02,3.9418070e+02,3.7288760e+02,3.6059670e+02, &
& 4.6716080e+02,4.8537390e+02,3.6148780e+02,3.7694960e+02,3.5152420e+02, &
& 3.2026850e+02,3.0203680e+02,3.0706620e+02,3.2283420e+02,3.8866590e+02, &
& 4.1079890e+02,4.1930730e+02,4.1680560e+02,5.1446020e+02,9.8223500e+01, &
& 8.8657200e+01,6.7270400e+01,2.1743180e+02,2.5998580e+02,2.1672230e+02, &
& 1.9153170e+02,2.0921040e+02,2.3116180e+02,2.6679440e+02,2.6367810e+02, &
& 3.1083150e+02,3.4550300e+01,2.2887500e+01,5.2718510e+02,3.0788140e+02, &
& 2.0823790e+02,1.4104070e+02,9.8802500e+01,7.4902900e+01,5.6815100e+01, &
& 4.3755600e+01,6.3086870e+02,4.8975320e+02,4.5261920e+02,3.5774280e+02, &
& 2.7953920e+02,2.3224340e+02,1.8988240e+02,1.5546270e+02,1.0315350e+03, &
& 8.5745370e+02,7.0944180e+02,6.8588510e+02,6.2854190e+02,4.9509360e+02, &
& 5.4171840e+02,4.2519040e+02,4.5143600e+02,4.6473810e+02,3.5624130e+02, &
& 3.6657850e+02,4.3386350e+02,3.8351890e+02,3.2767900e+02,2.9427420e+02, &
& 2.5771960e+02,2.2426550e+02,1.1566447e+03,1.0211992e+03,8.9668220e+02, &
& 8.0699680e+02,7.3648020e+02,5.6927660e+02,6.3492090e+02,4.8434860e+02, &
& 5.2953710e+02,4.9130410e+02,4.0943180e+02,4.3319920e+02,5.4284130e+02, &
& 5.0298910e+02,4.4897690e+02,4.1689450e+02,3.7722550e+02,3.3935050e+02]
 vdw_dftd3_c6(29701:29800)=[&
& 1.4090375e+03,1.3006349e+03,1.1454236e+03,5.3013690e+02,1.1533721e+03, &
& 1.1079903e+03,1.0803308e+03,1.0548803e+03,1.0323159e+03,8.1371020e+02, &
& 9.1255420e+02,8.8050980e+02,9.3168500e+02,9.1198360e+02,8.9424260e+02, &
& 8.8370520e+02,7.4578300e+02,7.3506570e+02,6.7161750e+02,5.6750780e+02, &
& 5.7764320e+02,5.2400420e+02,4.8023510e+02,3.9915430e+02,3.7302900e+02, &
& 3.8392690e+02,5.5745160e+02,5.4594690e+02,5.0242860e+02,4.7954670e+02, &
& 4.4291920e+02,4.0620580e+02,1.3331000e+03,1.2876457e+03,1.1407848e+03, &
& 1.0274883e+03,1.0189158e+03,9.8663370e+02,1.0144405e+03,9.8265960e+02, &
& 5.5564300e+01,1.7907060e+02,2.2855840e+02,1.7527450e+02,1.3225620e+02, &
& 9.2115400e+01,6.7181100e+01,4.6303000e+01,2.6154000e+02,4.0427630e+02, &
& 4.1041760e+02,3.3075310e+02,2.7134880e+02,2.2967560e+02,1.8790960e+02, &
& 3.6096750e+02,6.8580030e+02,3.6213940e+02,3.4961060e+02,3.4261200e+02, &
& 3.3934490e+02,3.1272950e+02,2.8953010e+02,2.7618840e+02,2.6998570e+02, &
& 2.6647580e+02,2.5050770e+02,4.0773530e+02,3.5895200e+02,3.2214500e+02, &
& 2.9438040e+02,2.5950110e+02,4.3088110e+02,8.3409790e+02,6.4047400e+02, &
& 4.7873550e+02,4.8344780e+02,4.5098400e+02,5.0476500e+02,3.9362700e+02, &
& 3.6827100e+02,3.4204040e+02,3.3083540e+02,3.2765190e+02,5.1780290e+02, &
& 4.6937510e+02,4.4010850e+02,4.1661070e+02,3.8076150e+02,5.1028570e+02, &
& 1.0802049e+03,8.1022120e+02,5.1419410e+02,5.0356860e+02,4.8764360e+02, &
& 4.8991040e+02,4.7183730e+02,4.9298820e+02,4.6313550e+02,4.7101170e+02]
 vdw_dftd3_c6(29801:29900)=[&
& 4.3969410e+02,4.2720060e+02,4.2443580e+02,4.4601630e+02,4.1029850e+02, &
& 5.5410220e+02,5.1478350e+02,4.7018380e+02,4.7532640e+02,4.1708260e+02, &
& 3.9248360e+02,3.7493960e+02,3.5843860e+02,3.4665580e+02,5.5606680e+02, &
& 5.0782080e+02,4.9295080e+02,4.7971720e+02,4.4828660e+02,5.7477940e+02, &
& 1.0760744e+03,6.0843480e+02,7.0109350e+02,6.2707340e+02,5.5492490e+02, &
& 5.3413870e+02,6.4033600e+02,1.4976800e+02,1.4723850e+02,1.0796360e+02, &
& 8.6517400e+01,5.9302900e+01,2.6525750e+02,3.2039240e+02,3.0775700e+02, &
& 2.6762990e+02,2.2503380e+02,3.7171230e+02,3.5761290e+02,3.6204400e+02, &
& 3.3130930e+02,2.4989390e+02,2.1401180e+02,2.1006230e+02,2.4593880e+02, &
& 2.2923870e+02,3.2367220e+02,3.4184070e+02,3.1373420e+02,2.9207370e+02, &
& 4.4529880e+02,4.2922790e+02,4.3306580e+02,4.1869180e+02,3.7051430e+02, &
& 3.2726370e+02,3.0934100e+02,3.1485220e+02,3.2886980e+02,4.1760930e+02, &
& 4.5391950e+02,4.2812220e+02,4.1326530e+02,5.3867350e+02,5.5884130e+02, &
& 4.1480660e+02,4.3299810e+02,4.0255440e+02,3.6584940e+02,3.4433760e+02, &
& 3.5080960e+02,3.6898910e+02,4.4636650e+02,4.7183450e+02,4.8114100e+02, &
& 4.7772640e+02,5.9304720e+02,1.1213310e+02,1.0101310e+02,7.6210000e+01, &
& 2.5014210e+02,2.9982180e+02,2.4837820e+02,2.1915410e+02,2.4046050e+02, &
& 2.6525500e+02,3.0652410e+02,3.0226660e+02,3.5623590e+02,4.0914530e+02, &
& 3.5093900e+01,2.3379200e+01,5.1864270e+02,3.0744210e+02,2.0967450e+02, &
& 1.4283480e+02,1.0045000e+02,7.6342400e+01,5.8017600e+01,4.4740800e+01]
 vdw_dftd3_c6(29901:30000)=[&
& 6.2119430e+02,4.8774180e+02,4.5276050e+02,3.5991590e+02,2.8255640e+02, &
& 2.3542680e+02,1.9298930e+02,1.5834200e+02,1.0138323e+03,8.5006180e+02, &
& 7.0472450e+02,6.8261700e+02,6.2623920e+02,4.9343100e+02,5.4059460e+02, &
& 4.2451240e+02,4.5168930e+02,4.6446650e+02,3.5610950e+02,3.6766370e+02, &
& 4.3474170e+02,3.8597850e+02,3.3106180e+02,2.9800720e+02,2.6158890e+02, &
& 2.2809040e+02,1.1378099e+03,1.0121518e+03,8.9191400e+02,8.0449370e+02, &
& 7.3528120e+02,5.6979780e+02,6.3489910e+02,4.8565920e+02,5.3091130e+02, &
& 4.9300840e+02,4.1066850e+02,4.3517770e+02,5.4381980e+02,5.0566970e+02, &
& 4.5291870e+02,4.2141870e+02,3.8213760e+02,3.4444580e+02,1.3865806e+03, &
& 1.2877738e+03,1.1381053e+03,5.3448010e+02,1.1432636e+03,1.0988818e+03, &
& 1.0716137e+03,1.0465046e+03,1.0242500e+03,8.1059520e+02,9.0407560e+02, &
& 8.7295200e+02,9.2519070e+02,9.0570520e+02,8.8820360e+02,8.7758750e+02, &
& 7.4266050e+02,7.3464070e+02,6.7268970e+02,5.6941570e+02,5.8004130e+02, &
& 5.2713410e+02,4.8380280e+02,4.0272830e+02,3.7658500e+02,3.8784150e+02, &
& 5.5900180e+02,5.4891370e+02,5.0669310e+02,4.8446800e+02,4.4838730e+02, &
& 4.1197350e+02,1.3155898e+03,1.2773025e+03,1.1352332e+03,1.0262943e+03, &
& 1.0158964e+03,9.8378580e+02,1.0091230e+03,9.7791270e+02,5.6194600e+01, &
& 1.7902740e+02,2.2888940e+02,1.7677430e+02,1.3392400e+02,9.3656200e+01, &
& 6.8514100e+01,4.7383500e+01,2.6106680e+02,4.0330920e+02,4.1097880e+02, &
& 3.3303840e+02,2.7432020e+02,2.3280570e+02,1.9097240e+02,3.6149280e+02]
 vdw_dftd3_c6(30001:30100)=[&
& 6.8031840e+02,3.6424560e+02,3.5170890e+02,3.4463160e+02,3.4115480e+02, &
& 3.1519770e+02,2.9203450e+02,2.7857600e+02,2.7225780e+02,2.6817480e+02, &
& 2.5301490e+02,4.0901750e+02,3.6162180e+02,3.2554390e+02,2.9810330e+02, &
& 2.6336490e+02,4.3194800e+02,8.2711060e+02,6.3960320e+02,4.8140710e+02, &
& 4.8607860e+02,4.5385780e+02,5.0629760e+02,3.9707420e+02,3.7157400e+02, &
& 3.4525330e+02,3.3380120e+02,3.3121190e+02,5.1938200e+02,4.7251890e+02, &
& 4.4417700e+02,4.2118710e+02,3.8570840e+02,5.1302910e+02,1.0697955e+03, &
& 8.0869310e+02,5.1850620e+02,5.0780630e+02,4.9182970e+02,4.9386870e+02, &
& 4.7482940e+02,4.9689070e+02,4.6699360e+02,4.7450580e+02,4.4355070e+02, &
& 4.3101790e+02,4.2817260e+02,4.4962020e+02,4.1401670e+02,5.5586120e+02, &
& 5.1747490e+02,4.7362850e+02,4.7790620e+02,4.2136930e+02,3.9678090e+02, &
& 3.7918850e+02,3.6245620e+02,3.5113700e+02,5.5815810e+02,5.1151250e+02, &
& 4.9747420e+02,4.8477400e+02,4.5383680e+02,5.7800010e+02,1.0681281e+03, &
& 6.1348930e+02,7.0481290e+02,6.3107130e+02,5.5973100e+02,5.3903440e+02, &
& 6.4211740e+02,1.5112150e+02,1.4873950e+02,1.0957100e+02,8.7995800e+01, &
& 6.0520900e+01,2.6667450e+02,3.2197480e+02,3.1012840e+02,2.7048000e+02, &
& 2.2811080e+02,3.7364230e+02,3.6026270e+02,3.6478380e+02,3.3375970e+02, &
& 2.5258170e+02,2.1675670e+02,2.1269690e+02,2.4819240e+02,2.3156140e+02, &
& 3.2593780e+02,3.4461890e+02,3.1716790e+02,2.9578510e+02,4.4775000e+02, &
& 4.3282190e+02,4.3705000e+02,4.2243200e+02,3.7466630e+02,3.3151190e+02]
 vdw_dftd3_c6(30101:30200)=[&
& 3.1355000e+02,3.1846020e+02,3.3242910e+02,4.2067140e+02,4.5728600e+02, &
& 4.3233570e+02,4.1789910e+02,5.4209840e+02,5.6322280e+02,4.1908730e+02, &
& 4.3700180e+02,4.0713540e+02,3.7060120e+02,3.4925650e+02,3.5516130e+02, &
& 3.7350160e+02,4.5030970e+02,4.7607850e+02,4.8592250e+02,4.8291440e+02, &
& 5.9698270e+02,1.1375420e+02,1.0251480e+02,7.7544000e+01,2.5247130e+02, &
& 3.0196690e+02,2.5132430e+02,2.2165120e+02,2.4237210e+02,2.6809680e+02, &
& 3.0954100e+02,3.0575380e+02,3.6007570e+02,4.1300240e+02,4.1737900e+02, &
& 3.4044000e+01,2.3069600e+01,4.8987970e+02,2.9240300e+02,2.0096210e+02, &
& 1.3805320e+02,9.7883500e+01,7.4909600e+01,5.7318900e+01,4.4478100e+01, &
& 5.8737390e+02,4.6339590e+02,4.3142990e+02,3.4462380e+02,2.7203200e+02, &
& 2.2771370e+02,1.8762390e+02,1.5474190e+02,9.5974420e+02,8.0685000e+02, &
& 6.6951760e+02,6.4952300e+02,5.9639240e+02,4.7075810e+02,5.1551420e+02, &
& 4.0565730e+02,4.3161180e+02,4.4337980e+02,3.4075230e+02,3.5207290e+02, &
& 4.1543610e+02,3.7009550e+02,3.1876960e+02,2.8790150e+02,2.5371080e+02, &
& 2.2213540e+02,1.0781354e+03,9.6112990e+02,8.4864220e+02,7.6669670e+02, &
& 7.0169050e+02,5.4556530e+02,6.0711050e+02,4.6612980e+02,5.0894760e+02, &
& 4.7308430e+02,3.9480510e+02,4.1818980e+02,5.2081600e+02,4.8532990e+02, &
& 4.3604710e+02,4.0666470e+02,3.6983570e+02,3.3441780e+02,1.3141925e+03, &
& 1.2227209e+03,1.0825707e+03,5.1466410e+02,1.0866848e+03,1.0447574e+03, &
& 1.0188963e+03,9.9506410e+02,9.7393920e+02,7.7328240e+02,8.6029910e+02]
 vdw_dftd3_c6(30201:30300)=[&
& 8.3099980e+02,8.8007510e+02,8.6153880e+02,8.4491510e+02,8.3467880e+02, &
& 7.0786620e+02,7.0135370e+02,6.4346570e+02,5.4618300e+02,5.5658840e+02, &
& 5.0684330e+02,4.6602790e+02,3.8918560e+02,3.6440110e+02,3.7525010e+02, &
& 5.3671560e+02,5.2765960e+02,4.8826870e+02,4.6765830e+02,4.3390630e+02, &
& 3.9973560e+02,1.2488786e+03,1.2142011e+03,1.0810636e+03,9.8016620e+02, &
& 9.6970180e+02,9.3922300e+02,9.6210980e+02,9.3257400e+02,5.4112100e+01, &
& 1.7052180e+02,2.1838760e+02,1.6984800e+02,1.2953340e+02,9.1360600e+01, &
& 6.7384900e+01,4.7131600e+01,2.4874970e+02,3.8382790e+02,3.9201850e+02, &
& 3.1922290e+02,2.6420730e+02,2.2519300e+02,1.8567430e+02,3.4642790e+02, &
& 6.4689750e+02,3.4969760e+02,3.3785900e+02,3.3110740e+02,3.2768960e+02, &
& 3.0324930e+02,2.8126900e+02,2.6841200e+02,2.6230550e+02,2.5806750e+02, &
& 2.4404160e+02,3.9123290e+02,3.4713060e+02,3.1356550e+02,2.8799690e+02, &
& 2.5539640e+02,4.1451570e+02,7.8676620e+02,6.1108840e+02,4.6240460e+02, &
& 4.6695430e+02,4.3660630e+02,4.8596370e+02,3.8300380e+02,3.5872510e+02, &
& 3.3367710e+02,3.2258500e+02,3.2032730e+02,4.9788730e+02,4.5413030e+02, &
& 4.2784290e+02,4.0648930e+02,3.7325390e+02,4.9296130e+02,1.0173501e+03, &
& 7.7250690e+02,4.9942220e+02,4.8915000e+02,4.7387610e+02,4.7562120e+02, &
& 4.5685930e+02,4.7832350e+02,4.4982470e+02,4.5672830e+02,4.2739350e+02, &
& 4.1538260e+02,4.1256270e+02,4.3270380e+02,3.9897500e+02,5.3287050e+02, &
& 4.9712170e+02,4.5606110e+02,4.5949090e+02,4.0710880e+02,3.8388990e+02]
 vdw_dftd3_c6(30301:30400)=[&
& 3.6723030e+02,3.5126140e+02,3.4073060e+02,5.3610400e+02,4.9251500e+02, &
& 4.7966590e+02,4.6803600e+02,4.3913860e+02,5.5539950e+02,1.0170070e+03, &
& 5.9069510e+02,6.7732330e+02,6.0753220e+02,5.3986600e+02,5.2021880e+02, &
& 6.1673560e+02,1.4507840e+02,1.4327430e+02,1.0636560e+02,8.5946800e+01, &
& 5.9722700e+01,2.5519400e+02,3.0808810e+02,2.9753400e+02,2.6049720e+02, &
& 2.2070250e+02,3.5843680e+02,3.4620560e+02,3.5064040e+02,3.2104220e+02, &
& 2.4431950e+02,2.1030300e+02,2.0632120e+02,2.3960260e+02,2.2381460e+02, &
& 3.1290410e+02,3.3107290e+02,3.0565610e+02,2.8580050e+02,4.2995510e+02, &
& 4.1647270e+02,4.2087950e+02,4.0706770e+02,3.6222600e+02,3.2139710e+02, &
& 3.0435760e+02,3.0856230e+02,3.2173420e+02,4.0490970e+02,4.3980180e+02, &
& 4.1670500e+02,4.0341630e+02,5.2107880e+02,5.4160900e+02,4.0413150e+02, &
& 4.2132440e+02,3.9364640e+02,3.5924180e+02,3.3923690e+02,3.4453850e+02, &
& 3.6204930e+02,4.3455430e+02,4.5916510e+02,4.6885050e+02,4.6635820e+02, &
& 5.7392320e+02,1.1015230e+02,9.9617500e+01,7.5930400e+01,2.4251390e+02, &
& 2.8970710e+02,2.4238610e+02,2.1463710e+02,2.3384620e+02,2.5833420e+02, &
& 2.9794670e+02,2.9487070e+02,3.4780090e+02,3.9805300e+02,4.0261790e+02, &
& 3.8960690e+02,3.9201800e+01,2.6154200e+01,5.9230750e+02,3.4659360e+02, &
& 2.3511450e+02,1.5978570e+02,1.1231150e+02,8.5390200e+01,6.4954800e+01, &
& 5.0154300e+01,7.0906070e+02,5.5115190e+02,5.0990250e+02,4.0375640e+02, &
& 3.1618250e+02,2.6318820e+02,2.1563940e+02,1.7693360e+02,1.1602124e+03]
 vdw_dftd3_c6(30401:30500)=[&
& 9.6482980e+02,7.9848900e+02,7.7240680e+02,7.0804470e+02,5.5813800e+02, &
& 6.1052950e+02,4.7961190e+02,5.0914700e+02,5.2395450e+02,4.0203690e+02, &
& 4.1376860e+02,4.8932470e+02,4.3309510e+02,3.7065200e+02,3.3332050e+02, &
& 2.9238780e+02,2.5487000e+02,1.3013911e+03,1.1493369e+03,1.0098313e+03, &
& 9.0933820e+02,8.3029650e+02,6.4261650e+02,7.1635260e+02,5.4725930e+02, &
& 5.9800230e+02,5.5503560e+02,4.6292020e+02,4.8966130e+02,6.1279550e+02, &
& 5.6824370e+02,5.0783300e+02,4.7198620e+02,4.2758360e+02,3.8515720e+02, &
& 1.5855821e+03,1.4639079e+03,1.2899128e+03,5.9969740e+02,1.2986324e+03, &
& 1.2476103e+03,1.2164825e+03,1.1878330e+03,1.1624311e+03,9.1733060e+02, &
& 1.0280579e+03,9.9208130e+02,1.0492180e+03,1.0270231e+03,1.0070473e+03, &
& 9.9511980e+02,8.4045040e+02,8.2879200e+02,7.5780230e+02,6.4104200e+02, &
& 6.5256880e+02,5.9243330e+02,5.4333580e+02,4.5219510e+02,4.2282310e+02, &
& 4.3514150e+02,6.2994840e+02,6.1718100e+02,5.6851380e+02,5.4299390e+02, &
& 5.0202460e+02,4.6091780e+02,1.5008218e+03,1.4498449e+03,1.2851719e+03, &
& 1.1587220e+03,1.1488988e+03,1.1125743e+03,1.1434030e+03,1.1076634e+03, &
& 6.2853000e+01,2.0170300e+02,2.5762390e+02,1.9809520e+02,1.4988210e+02, &
& 1.0475870e+02,7.6663400e+01,5.3088800e+01,2.9465140e+02,4.5526290e+02, &
& 4.6254200e+02,3.7345290e+02,3.0696940e+02,2.6028570e+02,2.1340440e+02, &
& 4.0761910e+02,7.7230000e+02,4.0917160e+02,3.9511390e+02,3.8722670e+02, &
& 3.8350230e+02,3.5362940e+02,3.2753820e+02,3.1249710e+02,3.0547070e+02]
 vdw_dftd3_c6(30501:30600)=[&
& 3.0136610e+02,2.8355320e+02,4.6002380e+02,4.0553300e+02,3.6444410e+02, &
& 3.3344240e+02,2.9439140e+02,4.8683630e+02,9.3949220e+02,7.2246140e+02, &
& 5.4104620e+02,5.4639880e+02,5.0999070e+02,5.7035510e+02,4.4558690e+02, &
& 4.1703620e+02,3.8750290e+02,3.7479630e+02,3.7128680e+02,5.8473500e+02, &
& 5.3055010e+02,4.9790060e+02,4.7168750e+02,4.3157390e+02,5.7682000e+02, &
& 1.2167162e+03,9.1391750e+02,5.8172760e+02,5.6972170e+02,5.5175870e+02, &
& 5.5422640e+02,5.3359620e+02,5.5760720e+02,5.2397000e+02,5.3273770e+02, &
& 4.9751580e+02,4.8340880e+02,4.8024390e+02,5.0441450e+02,4.6426970e+02, &
& 6.2577080e+02,5.8183450e+02,5.3190280e+02,5.3741510e+02,4.7245060e+02, &
& 4.4483850e+02,4.2512390e+02,4.0652810e+02,3.9335910e+02,6.2845220e+02, &
& 5.7444200e+02,5.5791660e+02,5.4322420e+02,5.0808860e+02,6.4972250e+02, &
& 1.2125054e+03,6.8824310e+02,7.9252420e+02,7.0936470e+02,6.2816270e+02, &
& 6.0477390e+02,7.2374190e+02,1.6920360e+02,1.6658210e+02,1.2253210e+02, &
& 9.8442100e+01,6.7767000e+01,2.9929840e+02,3.6151620e+02,3.4761240e+02, &
& 3.0275900e+02,2.5504960e+02,4.1985730e+02,4.0420110e+02,4.0925130e+02, &
& 3.7462180e+02,2.8320230e+02,2.4283500e+02,2.3833220e+02,2.7849510e+02, &
& 2.5970610e+02,3.6569310e+02,3.8632290e+02,3.5500350e+02,3.3085020e+02, &
& 5.0318910e+02,4.8540080e+02,4.8989620e+02,4.7377460e+02,4.1981260e+02, &
& 3.7122640e+02,3.5107190e+02,3.5706620e+02,3.7278680e+02,4.7232770e+02, &
& 5.1322190e+02,4.8446450e+02,4.6794670e+02,6.0895860e+02,6.3182890e+02]
 vdw_dftd3_c6(30601:30700)=[&
& 4.6949580e+02,4.9006850e+02,4.5612840e+02,4.1497090e+02,3.9089370e+02, &
& 3.9804290e+02,4.1852820e+02,5.0537700e+02,5.3408270e+02,5.4469760e+02, &
& 5.4102390e+02,6.7046940e+02,1.2713380e+02,1.1469320e+02,8.6806100e+01, &
& 2.8266470e+02,3.3866010e+02,2.8113750e+02,2.4847100e+02,2.7223020e+02, &
& 3.0012720e+02,3.4667580e+02,3.4212760e+02,4.0346440e+02,4.6298340e+02, &
& 4.6750670e+02,4.5116740e+02,5.2418400e+02,3.9743800e+01,2.6627800e+01, &
& 5.8321980e+02,3.4623830e+02,2.3662000e+02,1.6159220e+02,1.1393640e+02, &
& 8.6791400e+01,6.6113100e+01,5.1096100e+01,6.9876860e+02,5.4919660e+02, &
& 5.1016460e+02,4.0605930e+02,3.1927240e+02,2.6639090e+02,2.1872080e+02, &
& 1.7975600e+02,1.1409307e+03,9.5709360e+02,7.9362090e+02,7.6903960e+02, &
& 7.0568530e+02,5.5635380e+02,6.0939430e+02,4.7886060e+02,5.0944850e+02, &
& 5.2372160e+02,4.0186070e+02,4.1492100e+02,4.9028010e+02,4.3566600e+02, &
& 3.7411390e+02,3.3709110e+02,2.9625040e+02,2.5864890e+02,1.2807871e+03, &
& 1.1397642e+03,1.0048393e+03,9.0672570e+02,8.2902620e+02,6.4306490e+02, &
& 7.1626640e+02,5.4850410e+02,5.9937630e+02,5.5674870e+02,4.6407170e+02, &
& 4.9165550e+02,6.1377870e+02,5.7101580e+02,5.1187060e+02,4.7658080e+02, &
& 4.3252650e+02,3.9023930e+02,1.5608728e+03,1.4501356e+03,1.2821300e+03, &
& 6.0410230e+02,1.2878172e+03,1.2378955e+03,1.2071956e+03,1.1789199e+03, &
& 1.1538573e+03,9.1395230e+02,1.0187768e+03,9.8379020e+02,1.0423525e+03, &
& 1.0203958e+03,1.0006811e+03,9.8867650e+02,8.3714090e+02,8.2836360e+02]
 vdw_dftd3_c6(30701:30800)=[&
& 7.5891210e+02,6.4292980e+02,6.5498530e+02,5.9558850e+02,5.4692440e+02, &
& 4.5573730e+02,4.2633520e+02,4.3904410e+02,6.3141780e+02,6.2017920e+02, &
& 5.7284430e+02,5.4797510e+02,5.0752580e+02,4.6668040e+02,1.4815226e+03, &
& 1.4387553e+03,1.2792885e+03,1.1574185e+03,1.1456007e+03,1.1094582e+03, &
& 1.1376821e+03,1.1025605e+03,6.3497600e+01,2.0170420e+02,2.5799550e+02, &
& 1.9963690e+02,1.5154880e+02,1.0626960e+02,7.7952800e+01,5.4119700e+01, &
& 2.9420730e+02,4.5432800e+02,4.6321330e+02,3.7584780e+02,3.1000460e+02, &
& 2.6343270e+02,2.1644080e+02,4.0808240e+02,7.6638090e+02,4.1133690e+02, &
& 3.9725380e+02,3.8928370e+02,3.8534090e+02,3.5616420e+02,3.3009770e+02, &
& 3.1492830e+02,3.0778390e+02,3.0308940e+02,2.8611150e+02,4.6138820e+02, &
& 4.0830150e+02,3.6791630e+02,3.3720180e+02,2.9824950e+02,4.8781130e+02, &
& 9.3186330e+02,7.2138820e+02,5.4373140e+02,5.4904170e+02,5.1286200e+02, &
& 5.7179580e+02,4.4904810e+02,4.2033540e+02,3.9069990e+02,3.7774640e+02, &
& 3.7486870e+02,5.8634250e+02,5.3378120e+02,5.0206010e+02,4.7633350e+02, &
& 4.3655340e+02,5.7950930e+02,1.2052309e+03,9.1205970e+02,5.8609720e+02, &
& 5.7401590e+02,5.5599730e+02,5.5823140e+02,5.3660520e+02,5.6156960e+02, &
& 5.2788270e+02,5.3626980e+02,5.0142950e+02,4.8728280e+02,4.8403880e+02, &
& 5.0809720e+02,4.6804880e+02,6.2749410e+02,5.8451000e+02,5.3534350e+02, &
& 5.3996900e+02,4.7674360e+02,4.4912500e+02,4.2935080e+02,4.1050840e+02, &
& 3.9782250e+02,6.3050780e+02,5.7817950e+02,5.6250870e+02,5.4834310e+02]
 vdw_dftd3_c6(30801:30900)=[&
& 5.1367500e+02,6.5289240e+02,1.2037016e+03,6.9337750e+02,7.9621000e+02, &
& 7.1328770e+02,6.3298840e+02,6.0969380e+02,7.2537520e+02,1.7061220e+02, &
& 1.6810260e+02,1.2413050e+02,9.9890200e+01,6.8938900e+01,3.0082770e+02, &
& 3.6319850e+02,3.5008330e+02,3.0566690e+02,2.5814020e+02,4.2185200e+02, &
& 4.0692810e+02,4.1206730e+02,3.7711770e+02,2.8588310e+02,2.4556070e+02, &
& 2.4095060e+02,2.8076020e+02,2.6203990e+02,3.6803530e+02,3.8919220e+02, &
& 3.5850420e+02,3.3459570e+02,5.0567630e+02,4.8906840e+02,4.9395440e+02, &
& 4.7754800e+02,4.2396710e+02,3.7545330e+02,3.5525030e+02,3.6064340e+02, &
& 3.7633490e+02,4.7544030e+02,5.1667410e+02,4.8876370e+02,4.7264780e+02, &
& 6.1239630e+02,6.3629250e+02,4.7384040e+02,4.9409450e+02,4.6071470e+02, &
& 4.1970150e+02,3.9577380e+02,4.0234540e+02,4.2301710e+02,5.0934070e+02, &
& 5.3837690e+02,5.4954870e+02,5.4627450e+02,6.7441620e+02,1.2875890e+02, &
& 1.1617900e+02,8.8102700e+01,2.8508900e+02,3.4088270e+02,2.8413180e+02, &
& 2.5096550e+02,2.7414590e+02,3.0303570e+02,3.4976730e+02,3.4567500e+02, &
& 4.0735630e+02,4.6691700e+02,4.7196350e+02,4.5575440e+02,5.2876490e+02, &
& 5.3388100e+02,9.2092000e+00,6.3181000e+00,1.2066280e+02,7.5131100e+01, &
& 5.2960600e+01,3.7004200e+01,2.6508300e+01,2.0399800e+01,1.5657800e+01, &
& 1.2160800e+01,1.4500200e+02,1.1807480e+02,1.1143770e+02,9.0574000e+01, &
& 7.2514500e+01,6.1218400e+01,5.0810200e+01,4.2132400e+01,2.3618020e+02, &
& 2.0306790e+02,1.6941860e+02,1.6529120e+02,1.5225000e+02,1.2032780e+02]
 vdw_dftd3_c6(30901:31000)=[&
& 1.3219840e+02,1.0419900e+02,1.1150630e+02,1.1415060e+02,8.7805200e+01, &
& 9.1572800e+01,1.0787050e+02,9.7349000e+01,8.4829800e+01,7.7153300e+01, &
& 6.8441600e+01,6.0254400e+01,2.6610180e+02,2.4185040e+02,2.1572630e+02, &
& 1.9615430e+02,1.8030420e+02,1.4125220e+02,1.5672560e+02,1.2128760e+02, &
& 1.3232810e+02,1.2329130e+02,1.0277530e+02,1.0928870e+02,1.3508140e+02, &
& 1.2715210e+02,1.1540180e+02,1.0829220e+02,9.9110900e+01,9.0129600e+01, &
& 3.2498720e+02,3.0696860e+02,2.7449320e+02,1.3602870e+02,2.7357430e+02, &
& 2.6338860e+02,2.5696540e+02,2.5103460e+02,2.4578320e+02,1.9745530e+02, &
& 2.1647080e+02,2.0953600e+02,2.2258700e+02,2.1794160e+02,2.1381010e+02, &
& 2.1110760e+02,1.8048560e+02,1.8074340e+02,1.6687410e+02,1.4241400e+02, &
& 1.4542480e+02,1.3310380e+02,1.2287620e+02,1.0304210e+02,9.6613000e+01, &
& 9.9656100e+01,1.3964810e+02,1.3828410e+02,1.2908340e+02,1.2428020e+02, &
& 1.1600970e+02,1.0744210e+02,3.1132220e+02,3.0650670e+02,2.7521710e+02, &
& 2.5216490e+02,2.4821560e+02,2.4044920e+02,2.4457260e+02,2.3732770e+02, &
& 1.4454100e+01,4.3974600e+01,5.6652700e+01,4.4988800e+01,3.4722600e+01, &
& 2.4752500e+01,1.8383100e+01,1.2929100e+01,6.3812300e+01,9.8372400e+01, &
& 1.0158890e+02,8.4115800e+01,7.0462900e+01,6.0525500e+01,5.0270800e+01, &
& 8.9856900e+01,1.6335340e+02,9.1795100e+01,8.8738900e+01,8.6926400e+01, &
& 8.5880600e+01,8.0037500e+01,7.4393000e+01,7.0987100e+01,6.9319400e+01, &
& 6.7783500e+01,6.4779800e+01,1.0192230e+02,9.1589500e+01,8.3502300e+01]
 vdw_dftd3_c6(31001:31100)=[&
& 7.7170000e+01,6.8871400e+01,1.0786680e+02,1.9856600e+02,1.5735980e+02, &
& 1.2138420e+02,1.2250750e+02,1.1484620e+02,1.2665710e+02,1.0138250e+02, &
& 9.4998700e+01,8.8451500e+01,8.5377300e+01,8.5227400e+01,1.2959790e+02, &
& 1.1945590e+02,1.1338670e+02,1.0828980e+02,1.0002100e+02,1.2944610e+02, &
& 2.5605470e+02,1.9871970e+02,1.3207060e+02,1.2936140e+02,1.2538070e+02, &
& 1.2566750e+02,1.2008750e+02,1.2632050e+02,1.1891980e+02,1.2044740e+02, &
& 1.1313110e+02,1.1000160e+02,1.0921440e+02,1.1430830e+02,1.0569650e+02, &
& 1.3894310e+02,1.3037550e+02,1.2030200e+02,1.2054080e+02,1.0824460e+02, &
& 1.0224360e+02,9.7889800e+01,9.3574800e+01,9.1193000e+01,1.3986370e+02, &
& 1.2975270e+02,1.2706290e+02,1.2448510e+02,1.1742660e+02,1.4593990e+02, &
& 2.5758460e+02,1.5618080e+02,1.7763750e+02,1.5984520e+02,1.4284510e+02, &
& 1.3782260e+02,1.6054880e+02,3.8474500e+01,3.8139800e+01,2.8676300e+01, &
& 2.3290000e+01,1.6298700e+01,6.6869600e+01,8.0700500e+01,7.8583500e+01, &
& 6.9411500e+01,5.9323900e+01,9.3919400e+01,9.1307200e+01,9.2515400e+01, &
& 8.4657000e+01,6.5034900e+01,5.6292100e+01,5.5169800e+01,6.3454300e+01, &
& 5.9424700e+01,8.2444700e+01,8.7528200e+01,8.1489600e+01,7.6593900e+01, &
& 1.1280930e+02,1.1017370e+02,1.1160800e+02,1.0787110e+02,9.6603000e+01, &
& 8.6127600e+01,8.1685300e+01,8.2288700e+01,8.5628100e+01,1.0669460e+02, &
& 1.1592860e+02,1.1062760e+02,1.0754340e+02,1.3712500e+02,1.4310680e+02, &
& 1.0750530e+02,1.1175770e+02,1.0503500e+02,9.6275900e+01,9.1232500e+01]
 vdw_dftd3_c6(31101:31200)=[&
& 9.2145800e+01,9.6759100e+01,1.1499010e+02,1.2156000e+02,1.2447110e+02, &
& 1.2415670e+02,1.5117960e+02,2.9680400e+01,2.6850600e+01,2.0588600e+01, &
& 6.4333200e+01,7.6403800e+01,6.4812700e+01,5.7198700e+01,6.1697400e+01, &
& 6.8793500e+01,7.9165600e+01,7.8747800e+01,9.2535500e+01,1.0552950e+02, &
& 1.0712000e+02,1.0385430e+02,1.1970710e+02,1.2129570e+02,2.8031500e+01, &
& 7.3662000e+00,5.2567000e+00,8.7818300e+01,5.6466200e+01,4.0896200e+01, &
& 2.9283000e+01,2.1419900e+01,1.6754400e+01,1.3054600e+01,1.0271600e+01, &
& 1.0592480e+02,8.8262000e+01,8.4307300e+01,6.9722400e+01,5.6785200e+01, &
& 4.8568100e+01,4.0851600e+01,3.4305000e+01,1.7296850e+02,1.5082030e+02, &
& 1.2634910e+02,1.2400130e+02,1.1458700e+02,9.1023300e+01,9.9976800e+01, &
& 7.9263000e+01,8.4954700e+01,8.6660100e+01,6.7082300e+01,7.0282000e+01, &
& 8.2324800e+01,7.5207600e+01,6.6414700e+01,6.0991300e+01,5.4682400e+01, &
& 4.8650000e+01,1.9559960e+02,1.7983950e+02,1.6177380e+02,1.4800690e+02, &
& 1.3670500e+02,1.0823560e+02,1.1959950e+02,9.3626900e+01,1.0181940e+02, &
& 9.5161200e+01,7.9665700e+01,8.4713900e+01,1.0361980e+02,9.8343100e+01, &
& 9.0179700e+01,8.5234300e+01,7.8668600e+01,7.2161700e+01,2.3924420e+02, &
& 2.2805560e+02,2.0554070e+02,1.0630350e+02,2.0400390e+02,1.9661650e+02, &
& 1.9187330e+02,1.8748180e+02,1.8359400e+02,1.4929180e+02,1.6190910e+02, &
& 1.5698310e+02,1.6653800e+02,1.6307020e+02,1.6000710e+02,1.5788900e+02, &
& 1.3608820e+02,1.3729240e+02,1.2762070e+02,1.0983770e+02,1.1232740e+02]
 vdw_dftd3_c6(31201:31300)=[&
& 1.0346160e+02,9.6033300e+01,8.1236300e+01,7.6428200e+01,7.8850000e+01, &
& 1.0789760e+02,1.0736650e+02,1.0105710e+02,9.7833200e+01,9.1999000e+01, &
& 8.5843200e+01,2.3075540e+02,2.2882200e+02,2.0694660e+02,1.9164430e+02, &
& 1.8809650e+02,1.8230090e+02,1.8437780e+02,1.7908370e+02,1.1329900e+01, &
& 3.3215800e+01,4.3062000e+01,3.4999000e+01,2.7520600e+01,2.0046800e+01, &
& 1.5175100e+01,1.0939100e+01,4.8162200e+01,7.4002800e+01,7.7136200e+01, &
& 6.4967200e+01,5.5237500e+01,4.8022100e+01,4.0420900e+01,6.9043400e+01, &
& 1.2207070e+02,7.1085400e+01,6.8829700e+01,6.7438300e+01,6.6556800e+01, &
& 6.2393100e+01,5.8177600e+01,5.5565400e+01,5.4239700e+01,5.2798400e+01, &
& 5.0885600e+01,7.8050500e+01,7.1014300e+01,6.5441400e+01,6.1004100e+01, &
& 5.5002900e+01,8.3241800e+01,1.4851410e+02,1.1976540e+02,9.4117300e+01, &
& 9.5000900e+01,8.9422900e+01,9.7841700e+01,7.9575000e+01,7.4728700e+01, &
& 6.9773000e+01,6.7307000e+01,6.7402800e+01,9.9752200e+01,9.2802100e+01, &
& 8.8743400e+01,8.5265700e+01,7.9371600e+01,1.0047420e+02,1.9124940e+02, &
& 1.5114110e+02,1.0329330e+02,1.0119170e+02,9.8146700e+01,9.8227000e+01, &
& 9.3519300e+01,9.8621300e+01,9.3007000e+01,9.3977100e+01,8.8582200e+01, &
& 8.6174300e+01,8.5510700e+01,8.9191500e+01,8.2798600e+01,1.0701830e+02, &
& 1.0110190e+02,9.3960100e+01,9.3667200e+01,8.5392500e+01,8.0955200e+01, &
& 7.7699700e+01,7.4376300e+01,7.2783200e+01,1.0824170e+02,1.0128820e+02, &
& 9.9667900e+01,9.8057500e+01,9.3106900e+01,1.1330310e+02,1.9337770e+02]
 vdw_dftd3_c6(31301:31400)=[&
& 1.2204020e+02,1.3787680e+02,1.2470980e+02,1.1209800e+02,1.0834540e+02, &
& 1.2418240e+02,2.9881900e+01,2.9891600e+01,2.2951700e+01,1.8917200e+01, &
& 1.3552500e+01,5.1352500e+01,6.1961000e+01,6.0868700e+01,5.4394100e+01, &
& 4.7094300e+01,7.2552400e+01,7.0967000e+01,7.1959500e+01,6.5947300e+01, &
& 5.1461000e+01,4.4917900e+01,4.3990200e+01,4.9914800e+01,4.6907100e+01, &
& 6.3911700e+01,6.8034300e+01,6.3962100e+01,6.0572300e+01,8.7389100e+01, &
& 8.5966800e+01,8.7311600e+01,8.4504100e+01,7.6386900e+01,6.8623700e+01, &
& 6.5290500e+01,6.5394900e+01,6.7836300e+01,8.3227000e+01,9.0268300e+01, &
& 8.6757400e+01,8.4742500e+01,1.0658000e+02,1.1146330e+02,8.4445100e+01, &
& 8.7680900e+01,8.3080500e+01,7.6686700e+01,7.3066600e+01,7.3485300e+01, &
& 7.7012400e+01,9.0346700e+01,9.5397300e+01,9.7851700e+01,9.7881700e+01, &
& 1.1758210e+02,2.3621000e+01,2.1537700e+01,1.6816900e+01,5.0039700e+01, &
& 5.9158300e+01,5.1004800e+01,4.5389100e+01,4.8414900e+01,5.3961700e+01, &
& 6.1903800e+01,6.1943800e+01,7.2970300e+01,8.2717000e+01,8.4217700e+01, &
& 8.2282500e+01,9.4131500e+01,9.5601800e+01,2.2379700e+01,1.8206700e+01, &
& 2.1006400e+01,1.3854700e+01,3.0879530e+02,1.8425660e+02,1.2577640e+02, &
& 8.5522300e+01,5.9931900e+01,4.5368800e+01,3.4318700e+01,2.6335800e+01, &
& 3.6977170e+02,2.9187160e+02,2.7132020e+02,2.1589980e+02,1.6943560e+02, &
& 1.4099550e+02,1.1535420e+02,9.4402700e+01,6.0221160e+02,5.0727030e+02, &
& 4.2088950e+02,4.0781600e+02,3.7420690e+02,2.9450370e+02,3.2309480e+02]
 vdw_dftd3_c6(31401:31500)=[&
& 2.5339650e+02,2.7008810e+02,2.7767940e+02,2.1252810e+02,2.1988700e+02, &
& 2.6030910e+02,2.3133020e+02,1.9840790e+02,1.7847990e+02,1.5647810e+02, &
& 1.3621020e+02,6.7586350e+02,6.0371340e+02,5.3268760e+02,4.8073180e+02, &
& 4.3942810e+02,3.4034780e+02,3.7931480e+02,2.8993610e+02,3.1720640e+02, &
& 2.9452200e+02,2.4486850e+02,2.5988150e+02,3.2493450e+02,3.0248790e+02, &
& 2.7103870e+02,2.5216040e+02,2.2853700e+02,2.0580950e+02,8.2382010e+02, &
& 7.6760950e+02,6.7935020e+02,3.1968540e+02,6.8142510e+02,6.5512090e+02, &
& 6.3890630e+02,6.2397630e+02,6.1074860e+02,4.8368880e+02,5.3820120e+02, &
& 5.1982010e+02,5.5187870e+02,5.4029010e+02,5.2989220e+02,5.2355130e+02, &
& 4.4329500e+02,4.3927910e+02,4.0232530e+02,3.4031200e+02,3.4676670e+02, &
& 3.1508760e+02,2.8909700e+02,2.4031030e+02,2.2456090e+02,2.3141000e+02, &
& 3.3360810e+02,3.2796730e+02,3.0290300e+02,2.8964470e+02,2.6800240e+02, &
& 2.4608550e+02,7.8245740e+02,7.6183620e+02,6.7780430e+02,6.1315740e+02, &
& 6.0632200e+02,5.8710090e+02,6.0167790e+02,5.8314710e+02,3.3712600e+01, &
& 1.0727210e+02,1.3717830e+02,1.0598260e+02,8.0142700e+01,5.5839400e+01, &
& 4.0663500e+01,2.7908600e+01,1.5615450e+02,2.4139030e+02,2.4630130e+02, &
& 1.9975300e+02,1.6446690e+02,1.3940860e+02,1.1413580e+02,2.1587970e+02, &
& 4.0567650e+02,2.1796280e+02,2.1039040e+02,2.0610620e+02,2.0396330e+02, &
& 1.8858340e+02,1.7467580e+02,1.6656740e+02,1.6276320e+02,1.6019460e+02, &
& 1.5131410e+02,2.4494460e+02,2.1672460e+02,1.9508440e+02,1.7853020e+02]
 vdw_dftd3_c6(31501:31600)=[&
& 1.5754230e+02,2.5790200e+02,4.9295740e+02,3.8205370e+02,2.8794440e+02, &
& 2.9066410e+02,2.7129600e+02,3.0236440e+02,2.3727980e+02,2.2191000e+02, &
& 2.0606750e+02,1.9915530e+02,1.9779040e+02,3.1041220e+02,2.8266840e+02, &
& 2.6580530e+02,2.5202720e+02,2.3068720e+02,3.0678390e+02,6.3715160e+02, &
& 4.8296010e+02,3.1010960e+02,3.0369930e+02,2.9412610e+02,2.9533450e+02, &
& 2.8374430e+02,2.9720650e+02,2.7927050e+02,2.8372210e+02,2.6527330e+02, &
& 2.5777900e+02,2.5608900e+02,2.6901820e+02,2.4764990e+02,3.3234580e+02, &
& 3.0938790e+02,2.8312170e+02,2.8556720e+02,2.5178560e+02,2.3694270e+02, &
& 2.2631410e+02,2.1617050e+02,2.0948600e+02,3.3326920e+02,3.0567610e+02, &
& 2.9742190e+02,2.8985610e+02,2.7129160e+02,3.4569720e+02,6.3670530e+02, &
& 3.6700470e+02,4.2126010e+02,3.7698280e+02,3.3444420e+02,3.2204370e+02, &
& 3.8313380e+02,9.0695600e+01,8.9119000e+01,6.5487900e+01,5.2413600e+01, &
& 3.5824700e+01,1.5991150e+02,1.9305340e+02,1.8599960e+02,1.6212890e+02, &
& 1.3657580e+02,2.2361620e+02,2.1570270e+02,2.1838960e+02,1.9966610e+02, &
& 1.5083440e+02,1.2933240e+02,1.2689320e+02,1.4822590e+02,1.3826350e+02, &
& 1.9524610e+02,2.0652060e+02,1.9004830e+02,1.7712870e+02,2.6785250e+02, &
& 2.5910620e+02,2.6164690e+02,2.5270610e+02,2.2395860e+02,1.9799630e+02, &
& 1.8716590e+02,1.9002130e+02,1.9842220e+02,2.5147160e+02,2.7356280e+02, &
& 2.5871400e+02,2.5005790e+02,3.2429550e+02,3.3722250e+02,2.5084450e+02, &
& 2.6138080e+02,2.4338630e+02,2.2137160e+02,2.0849420e+02,2.1188810e+02]
 vdw_dftd3_c6(31601:31700)=[&
& 2.2292680e+02,2.6896090e+02,2.8451810e+02,2.9053760e+02,2.8876380e+02, &
& 3.5718170e+02,6.8110600e+01,6.1216900e+01,4.6103100e+01,1.5146030e+02, &
& 1.8102370e+02,1.5063080e+02,1.3228840e+02,1.4466440e+02,1.6060910e+02, &
& 1.8547740e+02,1.8319150e+02,2.1524240e+02,2.4705720e+02,2.4975520e+02, &
& 2.4045180e+02,2.7943950e+02,2.8220780e+02,6.4108300e+01,5.0205300e+01, &
& 1.4977340e+02,2.3178400e+01,1.5480100e+01,3.3056660e+02,1.9940340e+02, &
& 1.3726170e+02,9.4035700e+01,6.6329300e+01,5.0474100e+01,3.8369700e+01, &
& 2.9574000e+01,3.9624310e+02,3.1527200e+02,2.9419700e+02,2.3539440e+02, &
& 1.8572260e+02,1.5516910e+02,1.2747850e+02,1.0474120e+02,6.4529730e+02, &
& 5.4647840e+02,4.5407710e+02,4.4076620e+02,4.0485430e+02,3.1900750e+02, &
& 3.5008820e+02,2.7496140e+02,2.9335770e+02,3.0126690e+02,2.3092580e+02, &
& 2.3939590e+02,2.8294690e+02,2.5245910e+02,2.1744760e+02,1.9619620e+02, &
& 1.7258230e+02,1.5072600e+02,7.2493940e+02,6.5048490e+02,5.7557310e+02, &
& 5.2045920e+02,4.7644890e+02,3.7016770e+02,4.1205940e+02,3.1604420e+02, &
& 3.4550510e+02,3.2111170e+02,2.6721350e+02,2.8372370e+02,3.5360350e+02, &
& 3.3013010e+02,2.9681450e+02,2.7677860e+02,2.5152290e+02,2.2713240e+02, &
& 8.8397990e+02,8.2666440e+02,7.3357140e+02,3.5001970e+02,7.3468830e+02, &
& 7.0659670e+02,6.8917900e+02,6.7312870e+02,6.5890990e+02,5.2379680e+02, &
& 5.8056100e+02,5.6104910e+02,5.9575320e+02,5.8326490e+02,5.7208340e+02, &
& 5.6513830e+02,4.7972770e+02,4.7662680e+02,4.3746730e+02,3.7093970e+02]
 vdw_dftd3_c6(31701:31800)=[&
& 3.7819490e+02,3.4433520e+02,3.1647760e+02,2.6374820e+02,2.4671740e+02, &
& 2.5430200e+02,3.6374800e+02,3.5827790e+02,3.3183780e+02,3.1789300e+02, &
& 2.9484760e+02,2.7138550e+02,8.4145020e+02,8.2171730e+02,7.3289850e+02, &
& 6.6526380e+02,6.5709130e+02,6.3634590e+02,6.5090570e+02,6.3106470e+02, &
& 3.6966400e+01,1.1625770e+02,1.4894110e+02,1.1591450e+02,8.8153200e+01, &
& 6.1839900e+01,4.5314100e+01,3.1360300e+01,1.6914050e+02,2.6122720e+02, &
& 2.6736160e+02,2.1800740e+02,1.8033050e+02,1.5342250e+02,1.2613260e+02, &
& 2.3497420e+02,4.3767790e+02,2.3797060e+02,2.2980530e+02,2.2513260e+02, &
& 2.2270380e+02,2.0634830e+02,1.9131890e+02,1.8247980e+02,1.7828580e+02, &
& 1.7518440e+02,1.6597640e+02,2.6653500e+02,2.3678670e+02,2.1387070e+02, &
& 1.9624890e+02,1.7373160e+02,2.8107340e+02,5.3187620e+02,4.1462030e+02, &
& 3.1443980e+02,3.1740770e+02,2.9662500e+02,3.2966960e+02,2.6011760e+02, &
& 2.4341820e+02,2.2622740e+02,2.1858540e+02,2.1737600e+02,3.3818310e+02, &
& 3.0894110e+02,2.9123020e+02,2.7666850e+02,2.5387120e+02,3.3505240e+02, &
& 6.8699570e+02,5.2395750e+02,3.3962030e+02,3.3261730e+02,3.2220350e+02, &
& 3.2336760e+02,3.1025500e+02,3.2531030e+02,3.0584570e+02,3.1046310e+02, &
& 2.9063460e+02,2.8247120e+02,2.8057190e+02,2.9443330e+02,2.7137930e+02, &
& 3.6212170e+02,3.3783560e+02,3.0986550e+02,3.1199870e+02,2.7647840e+02, &
& 2.6047040e+02,2.4897200e+02,2.3789260e+02,2.3089260e+02,3.6361850e+02, &
& 3.3453020e+02,3.2604840e+02,3.1819910e+02,2.9845520e+02,3.7759560e+02]
 vdw_dftd3_c6(31801:31900)=[&
& 6.8770170e+02,4.0183270e+02,4.6011490e+02,4.1238990e+02,3.6661110e+02, &
& 3.5322130e+02,4.1784880e+02,9.9170000e+01,9.7688000e+01,7.2259500e+01, &
& 5.8097400e+01,4.0013200e+01,1.7423120e+02,2.1029720e+02,2.0317830e+02, &
& 1.7773730e+02,1.5032780e+02,2.4395900e+02,2.3581030e+02,2.3880070e+02, &
& 2.1840030e+02,1.6577460e+02,1.4252950e+02,1.3980680e+02,1.6260680e+02, &
& 1.5184970e+02,2.1328870e+02,2.2581390e+02,2.0845000e+02,1.9473060e+02, &
& 2.9242890e+02,2.8359490e+02,2.8662000e+02,2.7689930e+02,2.4611820e+02, &
& 2.1811560e+02,2.0639140e+02,2.0911520e+02,2.1815300e+02,2.7516580e+02, &
& 2.9921170e+02,2.8364510e+02,2.7457510e+02,3.5440230e+02,3.6886820e+02, &
& 2.7514660e+02,2.8653550e+02,2.6750210e+02,2.4384710e+02,2.3006870e+02, &
& 2.3344450e+02,2.4547400e+02,2.9494270e+02,3.1192660e+02,3.1874160e+02, &
& 3.1709360e+02,3.9042810e+02,7.5031600e+01,6.7585400e+01,5.1187400e+01, &
& 1.6569080e+02,1.9769460e+02,1.6535710e+02,1.4552910e+02,1.5855360e+02, &
& 1.7612940e+02,2.0319220e+02,2.0106830e+02,2.3637610e+02,2.7080520e+02, &
& 2.7404840e+02,2.6442240e+02,3.0657470e+02,3.0987480e+02,7.0691700e+01, &
& 5.5685700e+01,1.6418110e+02,1.8029120e+02,3.1602000e+01,2.1215400e+01, &
& 4.5591990e+02,2.7300460e+02,1.8738080e+02,1.2831130e+02,9.0618400e+01, &
& 6.9093900e+01,5.2666100e+01,4.0719500e+01,5.4650960e+02,4.3237650e+02, &
& 4.0262520e+02,3.2143630e+02,2.5331660e+02,2.1163180e+02,1.7395170e+02, &
& 1.4307940e+02,8.9120910e+02,7.5145760e+02,6.2382770e+02,6.0513210e+02]
 vdw_dftd3_c6(31901:32000)=[&
& 5.5562000e+02,4.3806280e+02,4.8022450e+02,3.7740640e+02,4.0204340e+02, &
& 4.1305650e+02,3.1692740e+02,3.2786560e+02,3.8724130e+02,3.4490650e+02, &
& 2.9675050e+02,2.6767240e+02,2.3547970e+02,2.0576180e+02,1.0009319e+03, &
& 8.9470560e+02,7.9039380e+02,7.1409530e+02,6.5341870e+02,5.0749190e+02, &
& 5.6500310e+02,4.3325440e+02,4.7345910e+02,4.3998550e+02,3.6659430e+02, &
& 3.8876210e+02,4.8465320e+02,4.5175780e+02,4.0568490e+02,3.7809630e+02, &
& 3.4349120e+02,3.1018060e+02,1.2200459e+03,1.1376146e+03,1.0078507e+03, &
& 4.7859810e+02,1.0109240e+03,9.7204260e+02,9.4802150e+02,9.2588830e+02, &
& 9.0627400e+02,7.1942080e+02,7.9940550e+02,7.7227150e+02,8.1910080e+02, &
& 8.0189050e+02,7.8645970e+02,7.7695490e+02,6.5887080e+02,6.5329340e+02, &
& 5.9921110e+02,5.0805680e+02,5.1781620e+02,4.7129410e+02,4.3309810e+02, &
& 3.6112210e+02,3.3790780e+02,3.4812330e+02,4.9876900e+02,4.9062310e+02, &
& 4.5390390e+02,4.3458510e+02,4.0290830e+02,3.7079470e+02,1.1599029e+03, &
& 1.1298719e+03,1.0064862e+03,9.1246220e+02,9.0219930e+02,8.7376550e+02, &
& 8.9481040e+02,8.6739140e+02,5.0394000e+01,1.5913760e+02,2.0372270e+02, &
& 1.5821370e+02,1.2031900e+02,8.4513800e+01,6.2066500e+01,4.3143200e+01, &
& 2.3190190e+02,3.5800890e+02,3.6577110e+02,2.9764030e+02,2.4597730e+02, &
& 2.0927130e+02,1.7213180e+02,3.2212730e+02,6.0186910e+02,3.2550230e+02, &
& 3.1438040e+02,3.0805200e+02,3.0483850e+02,2.8215030e+02,2.6159540e+02, &
& 2.4956670e+02,2.4387510e+02,2.3989280e+02,2.2689510e+02,3.6463660e+02]
 vdw_dftd3_c6(32001:32100)=[&
& 3.2340730e+02,2.9186440e+02,2.6775420e+02,2.3705490e+02,3.8525160e+02, &
& 7.3162570e+02,5.6861590e+02,4.3018180e+02,4.3434220e+02,4.0590280e+02, &
& 4.5173170e+02,3.5582860e+02,3.3309730e+02,3.0966650e+02,2.9933180e+02, &
& 2.9736580e+02,4.6329930e+02,4.2259500e+02,3.9800110e+02,3.7792500e+02, &
& 3.4668560e+02,4.5839400e+02,9.4552740e+02,7.1867700e+02,4.6437650e+02, &
& 4.5481020e+02,4.4056910e+02,4.4222540e+02,4.2469280e+02,4.4484260e+02, &
& 4.1824050e+02,4.2468210e+02,3.9737350e+02,3.8619540e+02,3.8360070e+02, &
& 4.0253610e+02,3.7098070e+02,4.9582350e+02,4.6234390e+02,4.2390050e+02, &
& 4.2713880e+02,3.7805750e+02,3.5625810e+02,3.4062570e+02,3.2564110e+02, &
& 3.1586290e+02,4.9832750e+02,4.5783400e+02,4.4587260e+02,4.3494050e+02, &
& 4.0780020e+02,5.1650850e+02,9.4554140e+02,5.4936590e+02,6.2982470e+02, &
& 5.6451290e+02,5.0156450e+02,4.8322840e+02,5.7296750e+02,1.3526050e+02, &
& 1.3332140e+02,9.8654000e+01,7.9452200e+01,5.4900500e+01,2.3805490e+02, &
& 2.8733370e+02,2.7734590e+02,2.4249400e+02,2.0506870e+02,3.3372060e+02, &
& 3.2229080e+02,3.2638510e+02,2.9865910e+02,2.2674830e+02,1.9495560e+02, &
& 1.9127030e+02,2.2252350e+02,2.0778550e+02,2.9143960e+02,3.0837570e+02, &
& 2.8445060e+02,2.6569090e+02,4.0006500e+02,3.8751810e+02,3.9155320e+02, &
& 3.7846930e+02,3.3635920e+02,2.9811220e+02,2.8214800e+02,2.8612510e+02, &
& 2.9849500e+02,3.7650510e+02,4.0920080e+02,3.8757520e+02,3.7504230e+02, &
& 4.8468920e+02,5.0402900e+02,3.7581480e+02,3.9163870e+02,3.6554820e+02]
 vdw_dftd3_c6(32101:32200)=[&
& 3.3325000e+02,3.1443610e+02,3.1935010e+02,3.3574880e+02,4.0362310e+02, &
& 4.2669190e+02,4.3576920e+02,4.3337030e+02,5.3385140e+02,1.0232470e+02, &
& 9.2326000e+01,7.0080000e+01,2.2605040e+02,2.6996790e+02,2.2554180e+02, &
& 1.9911820e+02,2.1713920e+02,2.4041780e+02,2.7737170e+02,2.7434930e+02, &
& 3.2315090e+02,3.7016740e+02,3.7439290e+02,3.6163550e+02,4.1924330e+02, &
& 4.2353200e+02,9.6417900e+01,7.6081000e+01,2.2392830e+02,2.4599220e+02, &
& 3.3610200e+02,3.6783600e+01,2.4792600e+01,5.3122450e+02,3.1732060e+02, &
& 2.1779730e+02,1.4929680e+02,1.0560300e+02,8.0644400e+01,6.1573700e+01, &
& 4.7684400e+01,6.3682150e+02,5.0280810e+02,4.6801840e+02,3.7358140e+02, &
& 2.9452850e+02,2.4623230e+02,2.0258030e+02,1.6681140e+02,1.0395015e+03, &
& 8.7485890e+02,7.2603180e+02,7.0421370e+02,6.4655260e+02,5.1001320e+02, &
& 5.5878150e+02,4.3938300e+02,4.6773930e+02,4.8056400e+02,3.6899340e+02, &
& 3.8142390e+02,4.5032390e+02,4.0100460e+02,3.4508980e+02,3.1140410e+02, &
& 2.7412300e+02,2.3971550e+02,1.1675181e+03,1.0418714e+03,9.1996850e+02, &
& 8.3102350e+02,7.6040220e+02,5.9076540e+02,6.5761940e+02,5.0446970e+02, &
& 5.5107490e+02,5.1214360e+02,4.2704150e+02,4.5257800e+02,5.6405260e+02, &
& 5.2557130e+02,4.7195910e+02,4.3993100e+02,3.9979760e+02,3.6119780e+02, &
& 1.4230570e+03,1.3251273e+03,1.1733669e+03,5.5692770e+02,1.1775400e+03, &
& 1.1321484e+03,1.1041399e+03,1.0783321e+03,1.0554583e+03,8.3768840e+02, &
& 9.3165810e+02,8.9994220e+02,9.5379600e+02,9.3372910e+02,9.1573150e+02]
 vdw_dftd3_c6(32201:32300)=[&
& 9.0466320e+02,7.6704550e+02,7.6010300e+02,6.9715900e+02,5.9132700e+02, &
& 6.0261270e+02,5.4853340e+02,5.0415770e+02,4.2063020e+02,3.9369250e+02, &
& 4.0549310e+02,5.8079990e+02,5.7106960e+02,5.2826530e+02,5.0580110e+02, &
& 4.6902930e+02,4.3179360e+02,1.3523973e+03,1.3158434e+03,1.1716435e+03, &
& 1.0619693e+03,1.0504296e+03,1.0173620e+03,1.0421717e+03,1.0101837e+03, &
& 5.8587500e+01,1.8499140e+02,2.3683180e+02,1.8395600e+02,1.4003800e+02, &
& 9.8524700e+01,7.2487200e+01,5.0522200e+01,2.6973770e+02,4.1633670e+02, &
& 4.2518440e+02,3.4595860e+02,2.8602000e+02,2.4349690e+02,2.0046870e+02, &
& 3.7503550e+02,7.0097800e+02,3.7867460e+02,3.6578460e+02,3.5844930e+02, &
& 3.5474250e+02,3.2825710e+02,3.0438240e+02,2.9042420e+02,2.8381310e+02, &
& 2.7924290e+02,2.6402220e+02,4.2402640e+02,3.7603190e+02,3.3942410e+02, &
& 3.1150380e+02,2.7595340e+02,4.4859350e+02,8.5231430e+02,6.6191080e+02, &
& 5.0057750e+02,5.0546450e+02,4.7245030e+02,5.2594030e+02,4.1423030e+02, &
& 3.8785700e+02,3.6065710e+02,3.4865520e+02,3.4625000e+02,5.3916270e+02, &
& 4.9165990e+02,4.6303140e+02,4.3973150e+02,4.0350580e+02,5.3352020e+02, &
& 1.1018490e+03,8.3668870e+02,5.4039720e+02,5.2927220e+02,5.1271470e+02, &
& 5.1464180e+02,4.9434850e+02,5.1763800e+02,4.8672230e+02,4.9423650e+02, &
& 4.6243040e+02,4.4942300e+02,4.4639320e+02,4.6834080e+02,4.3169210e+02, &
& 5.7700800e+02,5.3808640e+02,4.9341180e+02,4.9722060e+02,4.4014790e+02, &
& 4.1487760e+02,3.9675610e+02,3.7939980e+02,3.6797200e+02,5.8018030e+02]
 vdw_dftd3_c6(32301:32385)=[&
& 5.3289470e+02,5.1891070e+02,5.0620180e+02,4.7470440e+02,6.0111980e+02, &
& 1.1015277e+03,6.3923770e+02,7.3305960e+02,6.5721210e+02,5.8388650e+02, &
& 5.6256740e+02,6.6728070e+02,1.5719910e+02,1.5506850e+02,1.1488240e+02, &
& 9.2650800e+01,6.4174400e+01,2.7667550e+02,3.3398620e+02,3.2238950e+02, &
& 2.8199150e+02,2.3862230e+02,3.8820850e+02,3.7487440e+02,3.7965070e+02, &
& 3.4749740e+02,2.6405320e+02,2.2711690e+02,2.2282550e+02,2.5908200e+02, &
& 2.4194730e+02,3.3891650e+02,3.5857470e+02,3.3082030e+02,3.0911340e+02, &
& 4.6550130e+02,4.5081160e+02,4.5551320e+02,4.4042440e+02,3.9158340e+02, &
& 3.4719340e+02,3.2867310e+02,3.3331550e+02,3.4765890e+02,4.3817200e+02, &
& 4.7608450e+02,4.5091710e+02,4.3638380e+02,5.6401480e+02,5.8633110e+02, &
& 4.3725330e+02,4.5578380e+02,4.2555250e+02,3.8809680e+02,3.6629140e+02, &
& 3.7206660e+02,3.9108870e+02,4.6992250e+02,4.9666200e+02,5.0715130e+02, &
& 5.0436840e+02,6.2120410e+02,1.1907380e+02,1.0755120e+02,8.1781000e+01, &
& 2.6273830e+02,3.1386840e+02,2.6230480e+02,2.3191300e+02,2.5285690e+02, &
& 2.7961360e+02,3.2256300e+02,3.1909300e+02,3.7612340e+02,4.3071080e+02, &
& 4.3559950e+02,4.2108940e+02,4.8797400e+02,4.9291380e+02,1.1222540e+02, &
& 8.8703000e+01,2.6034680e+02,2.8611790e+02,3.9112500e+02,4.5528540e+02]

 index_cni(1:100)=[&
& 1,1654,1655,1656,1657,1658,1659,1660,1661,1662,1663, &
& 1664,1665,1666,1667,1668,1669,1670,1671,1672,1673,1674, &
& 1675,1676,1677,1678,1679,1680,1681,1682,1683,1684,1685, &
& 1686,1687,1688,1689,1690,1691,1692,1693,1694,1695,1696, &
& 1697,1698,1699,1700,1701,1702,1703,1704,1705,1706,1707, &
& 1708,1709,1710,1711,4561,4562,4563,4564,4565,4566,4567, &
& 4568,4569,4570,4571,4572,4573,4574,4575,4576,4577,4578, &
& 4579,4580,4581,4582,4583,4584,4585,4586,4587,4588,4589, &
& 4590,4591,4592,4593,4594,4595,4596,4597,4598,4599,4600, &
& 4601]
 index_cni(101:200)=[&
& 4602,4603,4604,4605,4606,4607,4608,4609,4610,4611,4612, &
& 4613,4614,4615,4616,4617,4618,4619,4620,4621,4622,4623, &
& 4624,4625,4626,4627,4628,4629,4630,4631,4632,4633,4634, &
& 4635,4636,4637,4638,4639,4640,4641,4642,4643,4644,4645, &
& 4646,4647,4648,4649,4650,4651,4652,4653,4654,4655,4656, &
& 4657,4658,4659,4660,4661,4662,4663,4664,4665,4666,4667, &
& 4668,4669,4670,4671,4672,4673,4674,4675,4676,4677,4678, &
& 4679,4680,4681,4682,4683,4684,4685,4686,4687,4688,4689, &
& 4690,4691,4692,4693,4694,4695,4696,4697,4698,4699,4700, &
& 4701]
 index_cni(201:300)=[&
& 4702,4703,4704,4705,4706,4707,4708,4709,4710,4711,4712, &
& 4713,4714,4715,4716,4717,4718,4719,4720,4721,4722,4723, &
& 4724,4725,4726,4727,4728,4729,4730,4731,4732,4733,4734, &
& 4735,4736,4737,4738,4739,4740,4741,4742,4743,4744,4745, &
& 4746,4747,4748,4749,4750,4751,4752,4753,4754,4755,4756, &
& 4757,4758,4759,4760,4761,4762,4763,4764,4765,4766,4767, &
& 4768,4769,4770,4771,4772,4773,4774,4775,4776,4777,4778, &
& 4779,4780,4781,4782,4783,4784,4785,4786,4787,4788,4789, &
& 4790,4791,4792,4793,4794,4795,4796,4797,4798,4799,4800, &
& 4801]
 index_cni(301:400)=[&
& 4802,4803,4804,4805,4806,4807,4808,4809,4810,4811,4812, &
& 4813,4814,4815,4816,4817,4818,4819,4820,4821,4822,4823, &
& 4824,4825,4826,4827,4828,4829,4830,4831,4832,4833,4834, &
& 4835,4836,4837,4838,4839,4840,4841,4842,4843,4844,4845, &
& 4846,4847,4848,4849,4850,4851,4852,4853,4854,4855,4856, &
& 4857,4858,4859,4860,4861,4862,4863,4864,4865,4866,4867, &
& 4868,4869,4870,4871,4872,4873,4874,4875,4876,4877,4878, &
& 4879,4880,4881,4882,4883,4884,4885,4886,4887,4888,4889, &
& 4890,4891,4892,4893,4894,4895,4896,4897,4898,4899,4900, &
& 4901]
 index_cni(401:500)=[&
& 4902,4903,4904,4905,4906,4907,4908,4909,4910,4911,4912, &
& 4913,4914,4915,4916,4917,4918,4919,4920,4921,4922,4923, &
& 4924,4925,4926,4927,4928,4929,4930,4931,4932,4933,4934, &
& 4935,4936,4937,4938,4939,4940,4941,4942,4943,4944,4945, &
& 4946,4947,4948,4949,4950,4951,4952,4953,4954,4955,4956, &
& 4957,4958,4959,4960,4961,4962,4963,4964,4965,4966,4967, &
& 4968,4969,4970,4971,4972,4973,4974,4975,4976,4977,4978, &
& 4979,4980,4981,4982,4983,4984,4985,4986,4987,4988,4989, &
& 4990,4991,4992,4993,4994,4995,4996,4997,4998,4999,5000, &
& 5001]
 index_cni(501:600)=[&
& 5002,5003,5004,5005,5006,5007,5008,5009,5010,5011,5012, &
& 5013,5014,5015,5016,5017,5018,5019,5020,5021,5022,5023, &
& 5024,5025,5026,5027,5028,5029,5030,5031,5032,5033,5034, &
& 5035,5036,5037,5038,5039,5040,5041,5042,5043,5044,5045, &
& 5046,5047,5048,5049,5050,5051,5052,5053,5054,5055,5056, &
& 5057,5058,5059,5060,5061,5062,5063,5064,5065,5066,5067, &
& 5068,5069,5070,5071,5072,5073,5074,5075,5076,5077,5078, &
& 5079,5080,5081,5082,5083,5084,5085,5086,5087,5088,5089, &
& 5090,5091,5092,5093,5094,5095,5096,5097,5098,5099,5100, &
& 5101]
 index_cni(601:700)=[&
& 5102,5103,5104,5105,5106,5107,5108,5109,5110,5111,5112, &
& 5113,5114,5115,5116,5117,5118,5119,5120,5121,5122,5123, &
& 5124,5125,5126,5127,5128,5129,5130,5131,5132,5133,5134, &
& 5135,5136,5137,5138,5139,5140,5141,5142,5143,5144,5145, &
& 5146,5147,5148,5149,5150,5151,5152,5153,5154,5155,5156, &
& 5157,5158,5159,5160,5161,5162,5163,5164,5165,5166,5167, &
& 5168,5169,5170,5171,5172,5173,5174,5175,5176,5177,5178, &
& 5179,5180,5181,5182,5183,5184,5185,5186,5187,5188,5189, &
& 5190,5191,5192,5193,5194,5195,5196,5197,5198,5199,5200, &
& 5201]
 index_cni(701:800)=[&
& 5202,5203,5204,5205,5206,5207,5208,5209,5210,5211,5212, &
& 5213,5214,5215,5216,5217,5218,5219,5220,5221,5222,5223, &
& 5224,5225,5226,5227,5228,5229,5230,5231,5232,5233,5234, &
& 5235,5236,5237,5238,5239,5240,5241,5242,5243,5244,5245, &
& 5246,5247,5248,5249,5250,5251,5252,5253,5254,5255,5256, &
& 5257,5258,5259,5260,5261,5262,5263,5264,5265,5266,5267, &
& 5268,5269,5270,5271,5272,5273,5274,5275,5276,5277,5278, &
& 5279,5280,5281,5282,5283,5284,5285,5286,5287,5288,5289, &
& 5290,5291,5292,5293,5294,5295,5296,5297,5298,5299,5300, &
& 5301]
 index_cni(801:900)=[&
& 5302,5303,5304,5305,5306,5307,5308,5309,5310,5311,5312, &
& 5313,5314,5315,5316,5317,5318,5319,5320,5321,5322,5323, &
& 5324,5325,5326,5327,5328,5329,5330,5331,5332,5333,5334, &
& 5335,5336,5337,5338,5339,5340,5341,5342,5343,5344,5345, &
& 5346,5347,5348,5349,5350,5351,5352,5353,5354,5355,5356, &
& 5357,5358,5359,5360,5361,5362,5363,5364,5365,5366,5367, &
& 5368,5369,5370,5371,5372,5373,5374,5375,5376,5377,5378, &
& 5379,5380,5381,5382,5383,5384,5385,5386,5387,5388,5389, &
& 5390,5391,5392,5393,5394,5395,5396,5397,5398,5399,5400, &
& 5401]
 index_cni(901:1000)=[&
& 5402,5403,5404,5405,5406,5407,5408,5409,5410,5411,5412, &
& 5413,5414,5415,5416,5417,5418,5419,5420,5421,5422,5423, &
& 5424,5425,5426,5427,5428,5429,5430,5431,5432,5433,5434, &
& 5435,5436,5437,5438,5439,5440,5441,5442,5443,5444,5445, &
& 5446,5447,5448,5449,5450,5451,5452,5453,5454,5455,5456, &
& 5457,5458,5459,5460,5461,5462,5463,5464,5465,5466,5467, &
& 5468,5469,5470,5471,5472,5473,5474,5475,5476,5477,5478, &
& 5479,5480,5481,5482,5483,5484,5485,5486,5487,5488,5489, &
& 5490,5491,5492,5493,5494,5495,5496,5497,5498,5499,5500, &
& 5501]
 index_cni(1001:1100)=[&
& 5502,5503,5504,5505,5506,5507,5508,5509,5510,5511,5512, &
& 5513,5514,5515,5516,5517,5518,5519,5520,5521,5522,5523, &
& 5524,5525,5526,5527,5528,5529,5530,5531,5532,5533,5534, &
& 5535,5536,5537,5538,5539,5540,5541,5542,5543,5544,5545, &
& 5546,5547,5548,5549,5550,5551,5552,5553,5554,5555,5556, &
& 5557,5558,5559,5560,5561,5562,5563,5564,5565,5566,5567, &
& 5568,5569,5570,5571,5572,5573,5574,5575,5576,5577,5578, &
& 5579,5580,5581,5582,5583,5584,5585,5586,5587,5588,5589, &
& 5590,5591,5592,5593,5594,5595,5596,5597,5598,5599,5600, &
& 5601]
 index_cni(1101:1200)=[&
& 5602,5603,5604,5605,5606,5607,5608,5609,5610,5611,5612, &
& 5613,5614,5615,5616,5617,5618,5619,5620,5621,5622,5623, &
& 5624,5625,5626,5627,5628,5629,5630,5631,5632,5633,5634, &
& 5635,5636,5637,5638,5639,5640,5641,5642,5643,5644,5645, &
& 5646,5647,5648,5649,5650,5651,5652,5653,5654,5655,5656, &
& 5657,5658,5659,5660,5661,5662,5663,5664,5665,5666,5667, &
& 5668,5669,5670,5671,5672,5673,5674,5675,5676,5677,5678, &
& 5679,5680,5681,5682,5683,5684,5685,5686,5687,5688,5689, &
& 5690,5691,5692,5693,5694,5695,5696,5697,5698,5699,5700, &
& 5701]
 index_cni(1201:1300)=[&
& 5702,5703,5704,5705,5706,5707,5708,5709,5710,5711,5712, &
& 5713,5714,5715,5716,5717,5718,5719,5720,5721,5722,5723, &
& 5724,5725,5726,5727,5728,5729,5730,5731,5732,5733,5734, &
& 5735,5736,5737,5738,5739,5740,5741,5742,5743,5744,5745, &
& 5746,5747,5748,5749,5750,5751,5752,5753,5754,5755,5756, &
& 5757,5758,5759,5760,5761,5762,5763,5764,5765,5766,5767, &
& 5768,5769,5770,5771,5772,5773,5774,5775,5776,5777,5778, &
& 5779,5780,5781,5782,5783,5784,5785,5786,5787,5788,5789, &
& 5790,5791,5792,5793,5794,5795,5796,5797,5798,5799,5800, &
& 5801]
 index_cni(1301:1400)=[&
& 5802,5803,5804,5805,5806,5807,5808,5809,5810,5811,5812, &
& 5813,5814,5815,5816,5817,5818,5819,5820,5821,5822,5823, &
& 5824,5825,5826,5827,5828,5829,5830,5831,5832,5833,5834, &
& 5835,5836,5837,5838,5839,5840,5841,5842,5843,5844,5845, &
& 5846,5847,5848,5849,5850,5851,5852,5853,5854,5855,5856, &
& 5857,5858,5859,5860,5861,5862,5863,5864,5865,5866,5867, &
& 5868,5869,5870,5871,5872,5873,5874,5875,5876,5877,5878, &
& 5879,5880,5881,5882,5883,5884,5885,5886,5887,5888,5889, &
& 5890,5891,5892,5893,5894,5895,5896,5897,5898,5899,5900, &
& 5901]
 index_cni(1401:1500)=[&
& 5902,5903,5904,5905,5906,5907,5908,5909,5910,5911,5912, &
& 5913,5914,5915,5916,5917,5918,5919,5920,5921,5922,5923, &
& 5924,5925,5926,5927,5928,5929,5930,5931,5932,5933,5934, &
& 5935,5936,5937,5938,5939,5940,5941,5942,5943,5944,5945, &
& 5946,5947,5948,5949,5950,5951,5952,5953,5954,5955,5956, &
& 5957,5958,5959,5960,5961,5962,5963,5964,5965,5966,5967, &
& 5968,5969,5970,5971,5972,5973,5974,5975,5976,5977,5978, &
& 5979,5980,5981,5982,5983,5984,5985,5986,5987,5988,5989, &
& 5990,5991,5992,5993,5994,5995,5996,5997,5998,5999,6000, &
& 6001]
 index_cni(1501:1600)=[&
& 6002,6003,6004,6005,6006,6007,6008,6009,6010,6011,6012, &
& 6013,6014,6015,6016,6017,6018,6019,6020,6021,6022,6023, &
& 6024,6025,6026,6027,6028,6029,6030,6031,6032,6033,6034, &
& 6035,6036,6037,6038,6039,6040,6041,6042,6043,6044,6045, &
& 6046,6047,6048,6049,6050,6051,6052,6053,6054,6055,6056, &
& 6057,6058,6059,6060,6061,6062,6063,6064,6065,6066,6067, &
& 6068,6069,6070,6071,6072,6073,6074,6075,6076,6077,6078, &
& 6079,6080,6081,6082,6083,6084,6085,6086,6087,6088,6089, &
& 6090,6091,6092,6093,6094,6095,6096,6097,6098,6099,6100, &
& 6101]
 index_cni(1601:1700)=[&
& 6102,6103,6104,6105,6106,6107,6108,6109,6110,6111,6112, &
& 6113,6114,6115,6116,6117,6118,6119,6120,6121,6122,6123, &
& 6124,6125,6126,6127,6128,6129,6130,6131,6132,6133,6134, &
& 6135,6136,6137,6138,6139,6140,6141,6142,6143,6144,6145, &
& 6146,6147,6148,6149,6150,6151,6152,6153,6154,6155,6156, &
& 6157,6158,6159,6160,6161,6162,6163,6164,6165,6166,6167, &
& 6168,6169,6170,6171,6172,6173,6174,6175,6176,6177,6178, &
& 6179,6180,6181,6182,6183,6184,6185,6186,6187,6188,6189, &
& 6190,6191,6192,6193,6194,6195,6196,6197,6198,6199,6200, &
& 6201]
 index_cni(1701:1800)=[&
& 6202,6203,6204,6205,6206,6207,6208,6209,6210,6211,6212, &
& 6213,6214,6215,6216,6217,6218,6219,6220,6221,6222,6223, &
& 6224,6225,6226,6227,6228,6229,6230,6231,6232,6233,6234, &
& 6235,6236,6237,6238,6239,6240,6241,6242,6243,6244,6245, &
& 6246,6247,6248,6249,6250,6251,6252,6253,6254,6255,6256, &
& 6257,6258,6259,6260,6261,6262,6263,6264,6265,6266,6267, &
& 6268,6269,6270,6271,6272,6273,6274,6275,6276,6277,6278, &
& 6279,6280,6281,6282,6283,6284,6285,6286,6287,6288,6289, &
& 6290,6291,6292,6293,6294,6295,6296,6297,6298,6299,6300, &
& 6301]
 index_cni(1801:1900)=[&
& 6302,6303,6304,6305,6306,6307,6308,6309,6310,6311,6312, &
& 6313,6314,6315,6316,6317,6318,6319,6320,6321,6322,6323, &
& 6324,6325,6326,6327,6328,6329,6330,6331,6332,6333,6334, &
& 6335,6336,6337,6338,6339,6340,6341,6342,6343,6344,6345, &
& 6346,6347,6348,6349,6350,6351,6352,6353,6354,6355,6356, &
& 6357,6358,6359,6360,6361,6362,6363,6364,6365,6366,6367, &
& 6368,6369,6370,6371,6372,6373,6374,6375,6376,6377,6378, &
& 6379,6380,6381,6382,6383,6384,6385,6386,6387,6388,6389, &
& 6390,6391,6392,6393,6394,6395,6396,6397,6398,6399,6400, &
& 6401]
 index_cni(1901:2000)=[&
& 6402,6403,6404,6405,6406,6407,6408,6409,6410,6411,6412, &
& 6413,6414,6415,6416,6417,6418,6419,6420,6421,6422,6423, &
& 6424,6425,6426,6427,6428,6429,6430,6431,6432,6433,6434, &
& 6435,6436,6437,6438,6439,6440,6441,6442,6443,6444,6445, &
& 6446,6447,6448,6449,6450,6451,6452,6453,6454,6455,6456, &
& 6457,6458,6459,6460,6461,6462,6463,6464,6465,6466,6467, &
& 6468,6469,6470,6471,6472,6473,6474,6475,6476,6477,6478, &
& 6479,6480,6481,6482,6483,6484,6485,6486,6487,6488,6489, &
& 6490,6491,6492,6493,6494,6495,6496,6497,6498,6499,6500, &
& 6501]
 index_cni(2001:2100)=[&
& 6502,6503,6504,6505,6506,6507,6508,6509,6510,6511,6512, &
& 6513,6514,6515,6516,6517,6518,6519,6520,6521,6522,6523, &
& 6524,6525,6526,6527,6528,6529,6530,6531,6532,6533,6534, &
& 6535,6536,6537,6538,6539,6540,6541,6542,6543,6544,6545, &
& 6546,6547,6548,6549,6550,6551,6552,6553,6554,6555,6556, &
& 6557,6558,6559,6560,6561,6562,6563,6564,6565,6566,6567, &
& 6568,6569,6570,6571,6572,6573,6574,6575,6576,6577,6578, &
& 6579,6580,6581,6582,6583,6584,6585,6586,6587,6588,6589, &
& 6590,6591,6592,6593,6594,6595,6596,6597,6598,6599,6600, &
& 6601]
 index_cni(2101:2200)=[&
& 6602,6603,6604,6605,6606,6607,6608,6609,6610,6611,6612, &
& 6613,6614,6615,6616,6617,6618,6619,6620,6621,6622,6623, &
& 6624,6625,6626,6627,6628,6629,6630,6631,6632,6633,6634, &
& 6635,6636,6637,6638,6639,6640,6641,6642,6643,6644,6645, &
& 6646,6647,6648,6649,6650,6651,6652,6653,6654,6655,6656, &
& 6657,6658,6659,6660,6661,6662,6663,6664,6665,6666,6667, &
& 6668,6669,6670,6671,6672,6673,6674,6675,6676,6677,6678, &
& 6679,6680,6681,6682,6683,6684,6685,6686,6687,6688,6689, &
& 6690,6691,6692,6693,6694,6695,6696,6697,6698,6699,6700, &
& 6701]
 index_cni(2201:2300)=[&
& 6702,6703,6704,6705,6706,6707,6708,6709,6710,6711,6712, &
& 6713,6714,6715,6716,6717,6718,6719,6720,6721,6722,6723, &
& 6724,6725,6726,6727,6728,6729,6730,6731,6732,6733,6734, &
& 6735,6736,6737,6738,6739,6740,6741,6742,6743,6744,6745, &
& 6746,6747,6748,6749,6750,6751,6752,6753,6754,6755,6756, &
& 6757,6758,6759,6760,6761,6762,6763,6764,6765,6766,6767, &
& 6768,6769,6770,6771,6772,6773,6774,6775,6776,6777,6778, &
& 6779,6780,6781,6782,6783,6784,6785,6786,6787,6788,6789, &
& 6790,6791,6792,6793,6794,6795,6796,6797,6798,6799,6800, &
& 6801]
 index_cni(2301:2400)=[&
& 6802,6803,6804,6805,6806,6807,6808,6809,6810,6811,6812, &
& 6813,6814,6815,6816,6817,6818,6819,6820,6821,6822,6823, &
& 6824,6825,6826,6827,6828,6829,6830,6831,6832,6833,6834, &
& 6835,6836,6837,6838,6839,6840,6841,6842,6843,6844,6845, &
& 6846,6847,6848,6849,6850,6851,6852,6853,6854,6855,6856, &
& 6857,6858,6859,6860,6861,6862,6863,6864,6865,6866,6867, &
& 6868,6869,6870,6871,6872,6873,6874,6875,6876,6877,6878, &
& 6879,6880,6881,6882,6883,6884,6885,6886,6887,6888,6889, &
& 6890,6891,6892,6893,6894,6895,6896,6897,6898,6899,6900, &
& 6901]
 index_cni(2401:2500)=[&
& 6902,6903,6904,6905,6906,6907,6908,6909,6910,6911,6912, &
& 6913,6914,6915,6916,6917,6918,6919,6920,6921,6922,6923, &
& 6924,6925,6926,6927,6928,6929,6930,6931,6932,6933,6934, &
& 6935,6936,6937,6938,6939,6940,6941,6942,6943,6944,6945, &
& 6946,6947,6948,6949,6950,6951,6952,6953,6954,6955,6956, &
& 6957,6958,6959,6960,6961,6962,6963,6964,6965,6966,6967, &
& 6968,6969,6970,6971,6972,6973,6974,6975,6976,6977,6978, &
& 6979,6980,6981,6982,6983,6984,6985,6986,6987,6988,6989, &
& 6990,6991,6992,6993,6994,6995,6996,6997,6998,6999,7000, &
& 7001]
 index_cni(2501:2600)=[&
& 7002,7003,7004,7005,7006,7007,7008,7009,7010,7011,7012, &
& 7013,7014,7015,7016,7017,7018,7019,7020,7021,7022,7023, &
& 7024,7025,7026,7027,7028,7029,7030,7031,7032,7033,7034, &
& 7035,7036,7037,7038,7039,7040,7041,7042,7043,7044,7045, &
& 7046,7047,7048,7049,7050,7051,7052,7053,7054,7055,7056, &
& 7057,7058,7059,7060,7061,7062,7063,7064,7065,7066,7067, &
& 7068,7069,7070,7071,7072,7073,7074,7075,7076,7077,7078, &
& 7079,7080,7081,7082,7083,7084,7085,7086,7087,7088,7089, &
& 7090,7091,7092,7093,7094,7095,7096,7097,7098,7099,7100, &
& 7101]
 index_cni(2601:2700)=[&
& 7102,7103,7104,7105,7106,7107,7108,7109,7110,7111,7112, &
& 7113,7114,7115,7116,7117,7118,7119,7120,7121,7122,7123, &
& 7124,7125,7126,7127,7128,7129,7130,7131,7132,7133,7134, &
& 7135,7136,7137,7138,7139,7140,7141,7142,7143,7144,7145, &
& 7146,7147,7148,7149,7150,7151,7152,7153,7154,7155,7156, &
& 7157,7158,7159,7160,7161,7162,7163,7164,7165,7166,7167, &
& 7168,7169,7170,7171,7172,7173,7174,7175,7176,7177,7178, &
& 7179,7180,7181,7182,7183,7184,7185,7186,7187,7188,7189, &
& 7190,7191,7192,7193,7194,7195,7196,7197,7198,7199,7200, &
& 7201]
 index_cni(2701:2800)=[&
& 7202,7203,7204,7205,7206,7207,7208,7209,7210,7211,7212, &
& 7213,7214,7215,7216,7217,7218,7219,7220,7221,7222,7223, &
& 7224,7225,7226,7227,7228,7229,7230,7231,7232,7233,7234, &
& 7235,7236,7237,7238,7239,7240,7241,7242,7243,7244,7245, &
& 7246,7247,7248,7249,7250,7251,7252,7253,7254,7255,7256, &
& 7257,7258,7259,7260,7261,7262,7263,7264,7265,7266,7267, &
& 7268,7269,7270,7271,7272,7273,7274,7275,7276,7277,7278, &
& 7279,7280,7281,7282,7283,7284,7285,7286,7287,7288,7289, &
& 7290,7291,7292,7293,7294,7295,7296,7297,7298,7299,7300, &
& 7301]
 index_cni(2801:2900)=[&
& 7302,7303,7304,7305,7306,7307,7308,7309,7310,7311,7312, &
& 7313,7314,7315,7316,7317,7318,7319,7320,7321,7322,7323, &
& 7324,7325,7326,7327,7328,7329,7330,7331,7332,7333,7334, &
& 7335,7336,7337,7338,7339,7340,7341,7342,7343,7344,7345, &
& 7346,7347,7348,7349,7350,7351,7352,7353,7354,7355,7356, &
& 7357,7358,7359,7360,7361,7362,7363,7364,7365,7366,7367, &
& 7368,7369,7370,7371,7372,7373,7374,7375,7376,7377,7378, &
& 7379,7380,7381,7382,7383,7384,7385,7386,7387,7388,7389, &
& 7390,7391,7392,7393,7394,7395,7396,7397,7398,7399,7400, &
& 7401]
 index_cni(2901:3000)=[&
& 7402,7403,7404,7405,7406,7407,7408,7409,7410,7411,7412, &
& 7413,7414,7415,7416,7417,7418,7419,7420,7421,7422,7423, &
& 7424,7425,7426,7427,7428,7429,7430,7431,7432,7433,7434, &
& 7435,7436,7437,7438,7439,7440,7441,7442,7443,7444,7445, &
& 7446,7447,7448,7449,7450,7451,7452,7453,7454,7455,7456, &
& 7457,7458,7459,7460,7461,7462,7463,7464,7465,7466,7467, &
& 7468,7469,7470,7471,7472,7473,7474,7475,7476,7477,7478, &
& 7479,7480,7481,7482,7483,7484,7485,7486,7487,7488,7489, &
& 7490,7491,7492,7493,7494,7495,7496,7497,7498,7499,7500, &
& 7501]
 index_cni(3001:3100)=[&
& 7502,7503,7504,7505,7506,7507,7508,7509,7510,7511,7512, &
& 7513,7514,7515,7516,7517,7518,7519,7520,7521,7522,7523, &
& 7524,7525,7526,7527,7528,7529,7530,7531,7532,7533,7534, &
& 7535,7536,7537,7538,7539,7540,7541,7542,7543,7544,7545, &
& 7546,7547,7548,7549,7550,7551,7552,7553,7554,7555,7556, &
& 7557,7558,7559,7560,7561,7562,7563,7564,7565,7566,7567, &
& 7568,7569,7570,7571,7572,7573,7574,7575,7576,7577,7578, &
& 7579,7580,7581,7582,7583,7584,7585,7586,7587,7588,7589, &
& 7590,7591,7592,7593,7594,7595,7596,7597,7598,7599,7600, &
& 7601]
 index_cni(3101:3200)=[&
& 7602,7603,7604,7605,7606,7607,7608,7609,7610,7611,7612, &
& 7613,7614,7615,7616,7617,7618,7619,7620,7621,7622,7623, &
& 7624,7625,7626,7627,7628,7629,7630,7631,7632,7633,7634, &
& 7635,7636,7637,7638,7639,7640,7641,7642,7643,7644,7645, &
& 7646,7647,7648,7649,7650,7651,7652,7653,7654,7655,7656, &
& 7657,7658,7659,7660,7661,7662,7663,7664,7665,7666,7667, &
& 7668,7669,7670,7671,7672,7673,7674,7675,7676,7677,7678, &
& 7679,7680,7681,7682,7683,7684,7685,7686,7687,7688,7689, &
& 7690,7691,7692,7693,7694,7695,7696,7697,7698,7699,7700, &
& 7701]
 index_cni(3201:3300)=[&
& 7702,7703,7704,7705,7706,7707,7708,7709,7710,7711,7712, &
& 7713,7714,7715,7716,7717,7718,7719,7720,7721,7722,7723, &
& 7724,7725,7726,7727,7728,7729,7730,7731,7732,7733,7734, &
& 7735,7736,7737,7738,7739,7740,7741,7742,7743,7744,7745, &
& 7746,7747,7748,7749,7750,7751,7752,7753,7754,7755,7756, &
& 7757,7758,7759,7760,7761,7762,7763,7764,7765,7766,7767, &
& 7768,7769,7770,7771,7772,7773,7774,7775,7776,7777,7778, &
& 7779,7780,7781,7782,7783,7784,7785,7786,7787,7788,7789, &
& 7790,7791,7792,7793,7794,7795,7796,7797,7798,7799,7800, &
& 7801]
 index_cni(3301:3400)=[&
& 7802,7803,7804,7805,7806,7807,7808,7809,7810,7811,7812, &
& 7813,7814,7815,7816,7817,7818,7819,7820,7821,7822,7823, &
& 7824,7825,7826,7827,7828,7829,7830,7831,7832,7833,7834, &
& 7835,7836,7837,7838,7839,7840,7841,7842,7843,7844,7845, &
& 7846,7847,7848,7849,7850,7851,7852,7853,7854,7855,7856, &
& 7857,7858,7859,7860,7861,7862,7863,7864,7865,7866,7867, &
& 7868,7869,7870,7871,7872,7873,7874,7875,7876,7877,7878, &
& 7879,7880,7881,7882,7883,7884,7885,7886,7887,7888,7889, &
& 7890,7891,7892,7893,7894,7895,7896,7897,7898,7899,7900, &
& 7901]
 index_cni(3401:3500)=[&
& 7902,7903,7904,7905,7906,7907,7908,7909,7910,7911,7912, &
& 7913,7914,7915,7916,7917,7918,7919,7920,7921,7922,7923, &
& 7924,7925,7926,7927,7928,7929,7930,7931,7932,7933,7934, &
& 7935,7936,7937,7938,7939,7940,7941,7942,7943,7944,7945, &
& 7946,7947,7948,7949,7950,7951,7952,7953,7954,7955,7956, &
& 7957,7958,7959,7960,7961,7962,7963,7964,7965,7966,7967, &
& 7968,7969,7970,7971,7972,7973,7974,7975,7976,7977,7978, &
& 7979,7980,7981,7982,7983,7984,7985,7986,7987,7988,7989, &
& 7990,7991,7992,7993,7994,7995,7996,7997,7998,7999,8000, &
& 8001]
 index_cni(3501:3600)=[&
& 8002,8003,8004,8005,8006,8007,8008,8009,8010,8011,8012, &
& 8013,8014,8015,8016,8017,8018,8019,8020,8021,8022,8023, &
& 8024,8025,8026,8027,8028,8029,8030,8031,8032,8033,8034, &
& 8035,8036,8037,8038,8039,8040,8041,8042,8043,8044,8045, &
& 8046,8047,8048,8049,8050,8051,8052,8053,8054,8055,8056, &
& 8057,8058,8059,8060,8061,8062,8063,8064,8065,8066,8067, &
& 8068,8069,8070,8071,8072,8073,8074,8075,8076,8077,8078, &
& 8079,8080,8081,8082,8083,8084,8085,8086,8087,8088,8089, &
& 8090,8091,8092,8093,8094,8095,8096,8097,8098,8099,8100, &
& 8101]
 index_cni(3601:3700)=[&
& 8102,8103,8104,8105,8106,8107,8108,8109,8110,8111,8112, &
& 8113,8114,8115,8116,8117,8118,8119,8120,8121,8122,8123, &
& 8124,8125,8126,8127,8128,8129,8130,8131,8132,8133,8134, &
& 8135,8136,8137,8138,8139,8140,8141,8142,8143,8144,8145, &
& 8146,8147,8148,8149,8150,8151,8152,8153,8154,8155,8156, &
& 8157,8158,8159,8160,8161,8162,8163,8164,8165,8166,8167, &
& 8168,8169,8170,8171,8172,8173,8174,8175,8176,8177,8178, &
& 8179,8180,8181,8182,8183,8184,8185,8186,8187,8188,8189, &
& 8190,8191,8192,8193,8194,8195,8196,8197,8198,8199,8200, &
& 8201]
 index_cni(3701:3800)=[&
& 8202,8203,8204,8205,8206,8207,8208,8209,8210,8211,8212, &
& 8213,8214,8215,8216,8217,8218,8219,8220,8221,8222,8223, &
& 8224,8225,8226,8227,8228,8229,8230,8231,8232,8233,8234, &
& 8235,8236,8237,8238,8239,8240,8241,8242,8243,8244,8245, &
& 8246,8247,8248,8249,8250,8251,8252,8253,8254,8255,8256, &
& 8257,8258,8259,8260,8261,8262,8263,8264,8265,8266,8267, &
& 8268,8269,8270,8271,8272,8273,8274,8275,8276,8277,8278, &
& 8279,8280,8281,8282,8283,8284,8285,8286,8287,8288,8289, &
& 8290,8291,8292,8293,8294,8295,8296,8297,8298,8299,8300, &
& 8301]
 index_cni(3801:3900)=[&
& 8302,8303,8304,8305,8306,8307,8308,8309,8310,8311,8312, &
& 8313,8314,8315,8316,8317,8318,8319,8320,8321,8322,8323, &
& 8324,8325,8326,8327,8328,8329,8330,8331,8332,8333,8334, &
& 8335,8336,8337,8338,8339,8340,8341,8342,8343,8344,8345, &
& 8346,8347,8348,8349,8350,8351,8352,8353,8354,8355,8356, &
& 8357,8358,8359,8360,8361,8362,8363,8364,8365,8366,8367, &
& 8368,8369,8370,8371,8372,8373,8374,8375,8376,8377,8378, &
& 8379,8380,8381,8382,8383,8384,8385,8386,8387,8388,8389, &
& 8390,8391,8392,8393,8394,8395,8396,8397,8398,8399,8400, &
& 8401]
 index_cni(3901:4000)=[&
& 8402,8403,8404,8405,8406,8407,8408,8409,8410,8411,8412, &
& 8413,8414,8415,8416,8417,8418,8419,8420,8421,8422,8423, &
& 8424,8425,8426,8427,8428,8429,8430,8431,8432,8433,8434, &
& 8435,8436,8437,8438,8439,8440,8441,8442,8443,8444,8445, &
& 8446,8447,8448,8449,8450,8451,8452,8453,8454,8455,8456, &
& 8457,8458,8459,8460,8461,8462,8463,8464,8465,8466,8467, &
& 8468,8469,8470,8471,8472,8473,8474,8475,8476,8477,8478, &
& 8479,8480,8481,8482,8483,8484,8485,8486,8487,8488,8489, &
& 8490,8491,8492,8493,8494,8495,8496,8497,8498,8499,8500, &
& 8501]
 index_cni(4001:4100)=[&
& 8502,8503,8504,8505,8506,8507,8508,8509,8510,8511,8512, &
& 8513,8514,8515,8516,8517,8518,8519,8520,8521,8522,8523, &
& 8524,8525,8526,8527,8528,8529,8530,8531,8532,8533,8534, &
& 8535,8536,8537,8538,8539,8540,8541,8542,8543,8544,8545, &
& 8546,8547,8548,8549,8550,8551,8552,8553,8554,8555,8556, &
& 8557,8558,8559,8560,8561,8562,8563,8564,8565,8566,8567, &
& 8568,8569,8570,8571,8572,8573,8574,8575,8576,8577,8578, &
& 8579,8580,8581,8582,8583,8584,8585,8586,8587,8588,8589, &
& 8590,8591,8592,8593,8594,8595,8596,8597,8598,8599,8600, &
& 8601]
 index_cni(4101:4200)=[&
& 8602,8603,8604,8605,8606,8607,8608,8609,8610,8611,8612, &
& 8613,8614,8615,8616,8617,8618,8619,8620,8621,8622,8623, &
& 8624,8625,8626,8627,8628,8629,8630,8631,8632,8633,8634, &
& 8635,8636,8637,8638,8639,8640,8641,8642,8643,8644,8645, &
& 8646,8647,8648,8649,8650,8651,8652,8653,8654,8655,8656, &
& 8657,8658,8659,8660,8661,8662,8663,8664,8665,8666,8667, &
& 8668,8669,8670,8671,8672,8673,8674,8675,8676,8677,8678, &
& 8679,8680,8681,8682,8683,8684,8685,8686,8687,8688,8689, &
& 8690,8691,8692,8693,8694,8695,8696,8697,8698,8699,8700, &
& 8701]
 index_cni(4201:4300)=[&
& 8702,8703,8704,8705,8706,8707,8708,8709,8710,8711,8712, &
& 8713,8714,8715,8716,8717,8718,8719,8720,8721,8722,8723, &
& 8724,8725,8726,8727,8728,8729,8730,8731,8732,8733,8734, &
& 8735,8736,8737,8738,8739,8740,8741,8742,8743,8744,8745, &
& 8746,8747,8748,8749,8750,8751,8752,8753,8754,8755,8756, &
& 8757,8758,8759,8760,8761,8762,8763,8764,8765,8766,8767, &
& 8768,8769,8770,8771,8772,8773,8774,8775,8776,8777,8778, &
& 8779,8780,8781,8782,8783,8784,8785,8786,8787,8788,8789, &
& 8790,8791,8792,8793,8794,8795,8796,8797,8798,8799,8800, &
& 8801]
 index_cni(4301:4400)=[&
& 8802,8803,8804,8805,8806,8807,8808,8809,8810,8811,8812, &
& 8813,8814,8815,8816,8817,8818,8819,8820,8821,8822,8823, &
& 8824,8825,8826,8827,8828,8829,8830,8831,8832,8833,8834, &
& 8835,8836,8837,8838,8839,8840,8841,8842,8843,8844,8845, &
& 8846,8847,8848,8849,8850,8851,8852,8853,8854,8855,8856, &
& 8857,8858,8859,8860,8861,8862,8863,8864,8865,8866,8867, &
& 8868,8869,8870,8871,8872,8873,8874,8875,8876,8877,8878, &
& 8879,8880,8881,8882,8883,8884,8885,8886,8887,8888,8889, &
& 8890,8891,8892,8893,8894,8895,8896,8897,8898,8899,8900, &
& 8901]
 index_cni(4401:4500)=[&
& 8902,8903,8904,8905,8906,8907,8908,8909,8910,8911,8912, &
& 8913,8914,8915,8916,8917,8918,8919,8920,8921,8922,8923, &
& 8924,8925,8926,8927,8928,8929,8930,8931,8932,8933,8934, &
& 8935,8936,8937,8938,8939,8940,8941,8942,8943,8944,8945, &
& 8946,8947,8948,8949,8950,8951,8952,8953,8954,8955,8956, &
& 8957,8958,8959,8960,8961,8962,8963,8964,8965,8966,8967, &
& 8968,8969,8970,8971,8972,8973,8974,8975,8976,8977,8978, &
& 8979,8980,8981,8982,8983,8984,8985,8986,8987,8988,8989, &
& 8990,8991,8992,8993,8994,8995,8996,8997,8998,8999,9000, &
& 9001]
 index_cni(4501:4600)=[&
& 9002,9003,9004,9005,9006,9007,9008,9009,9010,9011,9012, &
& 9013,9014,9015,9016,9017,9018,9019,9020,9021,9022,9023, &
& 9024,9025,9026,9027,9028,9029,9030,9031,9032,9033,9034, &
& 9035,9036,9037,9038,9039,9040,9041,9042,9043,9044,9045, &
& 9046,9047,9048,9049,9050,9051,9052,9053,9054,9055,9056, &
& 9057,9058,9059,9060,9061,9062,9063,9064,9065,9066,9067, &
& 9068,9069,9070,9071,9072,9073,9074,9075,9076,9077,9078, &
& 9079,9080,9081,9082,9083,9084,9085,9086,9087,9088,9089, &
& 9090,9091,9092,9093,9094,9095,9096,9097,9098,9099,9100, &
& 9101]
 index_cni(4601:4700)=[&
& 9102,9103,9104,9105,9106,9107,9108,9109,9110,9111,9112, &
& 9113,9114,9115,9116,9117,9118,9119,9120,9121,9122,9123, &
& 9124,9125,9126,9127,9128,9129,9130,9131,9132,9133,9134, &
& 9135,9136,9137,9138,9139,9140,9141,9142,9143,9144,9145, &
& 9146,9147,9148,9149,9150,9151,9152,9153,9154,9155,9156, &
& 9157,9158,9159,9160,9161,9162,9163,9164,9165,9166,9167, &
& 9168,9169,9170,9171,9172,9173,9174,9175,9176,9177,9178, &
& 9179,9180,9181,9182,9183,9184,9185,9186,9187,9188,9189, &
& 9190,9191,9192,9193,9194,9195,9196,9197,9198,9199,9200, &
& 9201]
 index_cni(4701:4800)=[&
& 9202,9203,9204,9205,9206,9207,9208,9209,9210,9211,9212, &
& 9213,9214,9215,9216,9217,9218,9219,9220,9221,9222,9223, &
& 9224,9225,9226,9227,9228,9229,9230,9231,9232,9233,9234, &
& 9235,9236,9237,9238,9239,9240,9241,9242,9243,9244,9245, &
& 9246,9247,9248,9249,9250,9251,9252,9253,9254,9255,9256, &
& 9257,9258,9259,9260,9261,9262,9263,9264,9265,9266,9267, &
& 9268,9269,9270,9271,9272,9273,9274,9275,9276,9277,9278, &
& 9279,9280,9281,9282,9283,9284,9285,9286,9287,9288,9289, &
& 9290,9291,9292,9293,9294,9295,9296,9297,9298,9299,9300, &
& 9301]
 index_cni(4801:4900)=[&
& 9302,9303,9304,9305,9306,9307,9308,9309,9310,9311,9312, &
& 9313,9314,9315,9316,9317,9318,9319,9320,9321,9322,9323, &
& 9324,9325,9326,9327,9328,9329,9330,9331,9332,9333,9334, &
& 9335,9336,9337,9338,9339,9340,9341,9342,9343,9344,9345, &
& 9346,9347,9348,9349,9350,9351,9352,9353,9354,9355,9356, &
& 9357,9358,9359,9360,9361,9362,9363,9364,9365,9366,9367, &
& 9368,9369,9370,9371,9372,9373,9374,9375,9376,9377,9378, &
& 9379,9380,9381,9382,9383,9384,9385,9386,9387,9388,9389, &
& 9390,9391,9392,9393,9394,9395,9396,9397,9398,9399,9400, &
& 9401]
 index_cni(4901:5000)=[&
& 9402,9403,9404,9405,9406,9407,9408,9409,9410,9411,9412, &
& 9413,9414,9415,9416,9417,9418,9419,9420,9421,9422,9423, &
& 9424,9425,9426,9427,9428,9429,9430,9431,9432,9433,9434, &
& 9435,9436,9437,9438,9439,9440,9441,9442,9443,9444,9445, &
& 9446,9447,9448,9449,9450,9451,9452,9453,9454,9455,9456, &
& 9457,9458,9459,9460,9461,9462,9463,9464,9465,9466,9467, &
& 9468,9469,9470,9471,9472,9473,9474,9475,9476,9477,9478, &
& 9479,9480,9481,9482,9483,9484,9485,9486,9487,9488,9489, &
& 9490,9491,9492,9493,9494,9495,9496,9497,9498,9499,9500, &
& 9501]
 index_cni(5001:5100)=[&
& 9502,9503,9504,9505,9506,9507,9508,9509,9510,9511,9512, &
& 9513,9514,9515,9516,9517,9518,9519,9520,9521,9522,9523, &
& 9524,9525,9526,9527,9528,9529,9530,9531,9532,9533,9534, &
& 9535,9536,9537,9538,9539,9540,9541,9542,9543,9544,9545, &
& 9546,9547,9548,9549,9550,9551,9552,9553,9554,9555,9556, &
& 9557,9558,9559,9560,9561,9562,9563,9564,9565,9566,9567, &
& 9568,9569,9570,9571,9572,9573,9574,9575,9576,9577,9578, &
& 9579,9580,9581,9582,9583,9584,9585,9586,9587,9588,9589, &
& 9590,9591,9592,9593,9594,9595,9596,9597,9598,9599,9600, &
& 9601]
 index_cni(5101:5200)=[&
& 9602,9603,9604,9605,9606,9607,9608,9609,9610,9611,9612, &
& 9613,9614,9615,9616,9617,9618,9619,9620,9621,9622,9623, &
& 9624,9625,9626,9627,9628,9629,9630,9631,9632,9633,9634, &
& 9635,9636,9637,9638,9639,9640,9641,9642,9643,9644,9645, &
& 9646,9647,9648,9649,9650,9651,9652,9653,9654,9655,9656, &
& 9657,9658,9659,9660,9661,9662,9663,9664,9665,9666,9667, &
& 9668,9669,9670,9671,9672,9673,9674,9675,9676,9677,9678, &
& 9679,9680,9681,9682,9683,9684,9685,9686,9687,9688,9689, &
& 9690,9691,9692,9693,9694,9695,9696,9697,9698,9699,9700, &
& 9701]
 index_cni(5201:5300)=[&
& 9702,9703,9704,9705,9706,9707,9708,9709,9710,9711,9712, &
& 9713,9714,9715,9716,9717,9718,9719,9720,9721,9722,9723, &
& 9724,9725,9726,9727,9728,9729,9730,9731,9732,9733,9734, &
& 9735,9736,9737,9738,9739,9740,9741,9742,9743,9744,9745, &
& 9746,9747,9748,9749,9750,9751,9752,9753,9754,9755,9756, &
& 9757,9758,9759,9760,9761,9762,9763,9764,9765,9766,9767, &
& 9768,9769,9770,9771,9772,9773,9774,9775,9776,9777,9778, &
& 9779,9780,9781,9782,9783,9784,9785,9786,9787,9788,9789, &
& 9790,9791,9792,9793,9794,9795,9796,9797,9798,9799,9800, &
& 9801]
 index_cni(5301:5400)=[&
& 9802,9803,9804,9805,9806,9807,9808,9809,9810,9811,9812, &
& 9813,9814,9815,9816,9817,9818,9819,9820,9821,9822,9823, &
& 9824,9825,9826,9827,9828,9829,9830,9831,9832,9833,9834, &
& 9835,9836,9837,9838,9839,9840,9841,9842,9843,9844,9845, &
& 9846,9847,9848,9849,9850,9851,9852,9853,9854,9855,9856, &
& 9857,9858,9859,9860,9861,9862,9863,9864,9865,9866,9867, &
& 9868,9869,9870,9871,9872,9873,9874,9875,9876,9877,9878, &
& 9879,9880,9881,9882,9883,9884,9885,9886,9887,9888,9889, &
& 9890,9891,9892,9893,9894,9895,9896,9897,9898,9899,9900, &
& 9901]
 index_cni(5401:5500)=[&
& 9902,9903,9904,9905,9906,9907,9908,9909,9910,9911,9912, &
& 9913,9914,9915,9916,9917,9918,9919,9920,9921,9922,9923, &
& 9924,9925,9926,9927,9928,9929,9930,9931,9932,9933,9934, &
& 9935,9936,9937,9938,9939,9940,9941,9942,9943,9944,9945, &
& 9946,9947,9948,9949,9950,9951,9952,9953,9954,9955,9956, &
& 9957,9958,9959,9960,9961,9962,9963,9964,9965,9966,9967, &
& 9968,9969,9970,9971,9972,9973,9974,9975,9976,9977,9978, &
& 9979,9980,9981,9982,9983,9984,9985,9986,9987,9988,9989, &
& 9990,9991,9992,9993,9994,9995,9996,9997,9998,9999,10000, &
& 10001]
 index_cni(5501:5600)=[&
& 10002,10003,10004,10005,10006,10007,10008,10009,10010,10011,10012, &
& 10013,10014,10015,10016,10017,10018,10019,10020,10021,10022,10023, &
& 10024,10025,10026,10027,10028,10029,10030,10031,10032,10033,10034, &
& 10035,10036,10037,10038,10039,10040,10041,10042,10043,10044,10045, &
& 10046,10047,10048,10049,10050,10051,10052,10053,10054,10055,10056, &
& 10057,10058,10059,10060,10061,10062,10063,10064,10065,10066,10067, &
& 10068,10069,10070,10071,10072,10073,10074,10075,10076,10077,10078, &
& 10079,10080,10081,10082,10083,10084,10085,10086,10087,10088,10089, &
& 10090,10091,10092,10093,10094,10095,10096,10097,10098,10099,10100, &
& 10101]
 index_cni(5601:5700)=[&
& 10102,10103,10104,10105,10106,10107,10108,10109,10110,10111,10112, &
& 10113,10114,10115,10116,10117,10118,10119,10120,10121,10122,10123, &
& 10124,10125,10126,10127,10128,10129,10130,10131,10132,10133,10134, &
& 10135,10136,10137,10138,10139,10140,10141,10142,10143,10144,10145, &
& 10146,10147,10148,10149,10150,10151,10152,10153,10154,10155,10156, &
& 10157,10158,10159,10160,10161,10162,10163,10164,10165,10166,10167, &
& 10168,10169,10170,10171,10172,10173,10174,10175,10176,10177,10178, &
& 10179,10180,10181,10182,10183,10184,10185,10186,10187,10188,10189, &
& 10190,10191,10192,10193,10194,10195,10196,10197,10198,10199,10200, &
& 10201]
 index_cni(5701:5800)=[&
& 10202,10203,10204,10205,10206,10207,10208,10209,10210,10211,10212, &
& 10213,10214,10215,10216,10217,10218,10219,10220,10221,10222,10223, &
& 10224,10225,10226,10227,10228,10229,10230,10231,10232,10233,10234, &
& 10235,10236,10237,10238,10239,10240,10241,10242,10243,10244,10245, &
& 10246,10247,10248,10249,10250,10251,10252,10253,10254,10255,10256, &
& 10257,10258,10259,10260,10261,10262,10263,10264,10265,10266,10267, &
& 10268,10269,10270,10271,10272,10273,10274,10275,10276,10277,10278, &
& 10279,10280,10281,10282,10283,10284,10285,10286,10287,10288,10289, &
& 10290,10291,10292,10293,10294,10295,10296,10297,10298,10299,10300, &
& 10301]
 index_cni(5801:5900)=[&
& 10302,10303,10304,10305,10306,10307,10308,10309,10310,10311,10312, &
& 10313,10314,10315,10316,10317,10318,10319,10320,10321,10322,10323, &
& 10324,10325,10326,10327,10328,10329,10330,10331,10332,10333,10334, &
& 10335,10336,10337,10338,10339,10340,10341,10342,10343,10344,10345, &
& 10346,10347,10348,10349,10350,10351,10352,10353,10354,10355,10356, &
& 10357,10358,10359,10360,10361,10362,10363,10364,10365,10366,10367, &
& 10368,10369,10370,10371,10372,10373,10374,10375,10376,10377,10378, &
& 10379,10380,10381,10382,10383,10384,10385,10386,10387,10388,10389, &
& 10390,10391,10392,10393,10394,10395,10396,10397,10398,10399,10400, &
& 10401]
 index_cni(5901:6000)=[&
& 10402,10403,10404,10405,10406,10407,10408,10409,10410,10411,10412, &
& 10413,10414,10415,10416,10417,10418,10419,10420,10421,10422,10423, &
& 10424,10425,10426,10427,10428,10429,10430,10431,10432,10433,10434, &
& 10435,10436,10437,10438,10439,10440,10441,10442,10443,10444,10445, &
& 10446,10447,10448,10449,10450,10451,10452,10453,10454,10455,10456, &
& 10457,10458,10459,10460,10461,10462,10463,10464,10465,10466,10467, &
& 10468,10469,10470,10471,10472,10473,10474,10475,10476,10477,10478, &
& 10479,10480,10481,10482,10483,10484,10485,10486,10487,10488,10489, &
& 10490,10491,10492,10493,10494,10495,10496,10497,10498,10499,10500, &
& 10501]
 index_cni(6001:6100)=[&
& 10502,10503,10504,10505,10506,10507,10508,10509,10510,10511,10512, &
& 10513,10514,10515,10516,10517,10518,10519,10520,10521,10522,10523, &
& 10524,10525,10526,10527,10528,10529,10530,10531,10532,10533,10534, &
& 10535,10536,10537,10538,10539,10540,10541,10542,10543,10544,10545, &
& 10546,10547,10548,10549,10550,10551,10552,10553,10554,10555,10556, &
& 10557,10558,10559,10560,10561,10562,10563,10564,10565,10566,10567, &
& 10568,10569,10570,10571,10572,10573,10574,10575,10576,10577,10578, &
& 10579,10580,10581,10582,10583,10584,10585,10586,10587,10588,10589, &
& 10590,10591,10592,10593,10594,10595,10596,10597,10598,10599,10600, &
& 10601]
 index_cni(6101:6200)=[&
& 10602,10603,10604,10605,10606,10607,10608,10609,10610,10611,10612, &
& 10613,10614,10615,10616,10617,10618,10619,10620,10621,10622,10623, &
& 10624,10625,10626,10627,10628,10629,10630,10631,10632,10633,10634, &
& 10635,10636,10637,10638,10639,10640,10641,10642,10643,10644,10645, &
& 10646,10647,10648,10649,10650,10651,10652,10653,10654,10655,10656, &
& 10657,10658,10659,10660,10661,10662,10663,10664,10665,10666,10667, &
& 10668,10669,10670,10671,10672,10673,10674,10675,10676,10677,10678, &
& 10679,10680,10681,10682,10683,10684,10685,10686,10687,10688,10689, &
& 10690,10691,10692,10693,10694,10695,10696,10697,10698,10699,10700, &
& 10701]
 index_cni(6201:6300)=[&
& 10702,10703,10704,10705,10706,10707,10708,10709,10710,10711,10712, &
& 10713,10714,10715,10716,10717,10718,10719,10720,10721,10722,10723, &
& 10724,10725,10726,10727,10728,10729,10730,10731,10732,10733,10734, &
& 10735,10736,10737,10738,10739,10740,10741,10742,10743,10744,10745, &
& 10746,10747,10748,10749,10750,10751,10752,10753,10754,10755,10756, &
& 10757,10758,10759,10760,10761,10762,10763,10764,10765,10766,10767, &
& 10768,10769,10770,10771,10772,10773,10774,10775,10776,10777,10778, &
& 10779,10780,10781,10782,10783,10784,10785,10786,10787,10788,10789, &
& 10790,10791,10792,10793,10794,10795,10796,10797,10798,10799,10800, &
& 10801]
 index_cni(6301:6400)=[&
& 10802,10803,10804,10805,10806,10807,10808,10809,10810,10811,10812, &
& 10813,10814,10815,10816,10817,10818,10819,10820,10821,10822,10823, &
& 10824,10825,10826,10827,10828,10829,10830,10831,10832,10833,10834, &
& 10835,10836,10837,10838,10839,10840,10841,10842,10843,10844,10845, &
& 10846,10847,10848,10849,10850,10851,10852,10853,10854,10855,10856, &
& 10857,10858,10859,10860,10861,10862,10863,10864,10865,10866,10867, &
& 10868,10869,10870,10871,10872,10873,10874,10875,10876,10877,10878, &
& 10879,10880,10881,10882,10883,10884,10885,10886,10887,10888,10889, &
& 10890,10891,10892,10893,10894,10895,10896,10897,10898,10899,10900, &
& 10901]
 index_cni(6401:6500)=[&
& 10902,10903,10904,10905,10906,10907,10908,10909,10910,10911,10912, &
& 10913,10914,10915,10916,10917,10918,10919,10920,10921,10922,10923, &
& 10924,10925,10926,10927,10928,10929,10930,10931,10932,10933,10934, &
& 10935,10936,10937,10938,10939,10940,10941,10942,10943,10944,10945, &
& 10946,10947,10948,10949,10950,10951,10952,10953,10954,10955,10956, &
& 10957,10958,10959,10960,10961,10962,10963,10964,10965,10966,10967, &
& 10968,10969,10970,10971,10972,10973,10974,10975,10976,10977,10978, &
& 10979,10980,10981,10982,10983,10984,10985,10986,10987,10988,10989, &
& 10990,10991,10992,10993,10994,10995,10996,10997,10998,10999,11000, &
& 11001]
 index_cni(6501:6600)=[&
& 11002,11003,11004,11005,11006,11007,11008,11009,11010,11011,11012, &
& 11013,11014,11015,11016,11017,11018,11019,11020,11021,11022,11023, &
& 11024,11025,11026,11027,11028,11029,11030,11031,11032,11033,11034, &
& 11035,11036,11037,11038,11039,11040,11041,11042,11043,11044,11045, &
& 11046,11047,11048,11049,11050,11051,11052,11053,11054,11055,11056, &
& 11057,11058,11059,11060,11061,11062,11063,11064,11065,11066,11067, &
& 11068,11069,11070,11071,11072,11073,11074,11075,11076,11077,11078, &
& 11079,11080,11081,11082,11083,11084,11085,11086,11087,11088,11089, &
& 11090,11091,11092,11093,11094,11095,11096,11097,11098,11099,11100, &
& 11101]
 index_cni(6601:6700)=[&
& 11102,11103,11104,11105,11106,11107,11108,11109,11110,11111,11112, &
& 11113,11114,11115,11116,11117,11118,11119,11120,11121,11122,11123, &
& 11124,11125,11126,11127,11128,11129,11130,11131,11132,11133,11134, &
& 11135,11136,11137,11138,11139,11140,11141,11142,11143,11144,11145, &
& 11146,11147,11148,11149,11150,11151,11152,11153,11154,11155,11156, &
& 11157,11158,11159,11160,11161,11162,11163,11164,11165,11166,11167, &
& 11168,11169,11170,11171,11172,11173,11174,11175,11176,11177,11178, &
& 11179,11180,11181,11182,11183,11184,11185,11186,11187,11188,11189, &
& 11190,11191,11192,11193,11194,11195,11196,11197,11198,11199,11200, &
& 11201]
 index_cni(6701:6800)=[&
& 11202,11203,11204,11205,11206,11207,11208,11209,11210,11211,11212, &
& 11213,11214,11215,11216,11217,11218,11219,11220,11221,11222,11223, &
& 11224,11225,11226,11227,11228,11229,11230,11231,11232,11233,11234, &
& 11235,11236,11237,11238,11239,11240,11241,11242,11243,11244,11245, &
& 11246,11247,11248,11249,11250,11251,11252,11253,11254,11255,11256, &
& 11257,11258,11259,11260,11261,11262,11263,11264,11265,11266,11267, &
& 11268,11269,11270,11271,11272,11273,11274,11275,11276,11277,11278, &
& 11279,11280,11281,11282,11283,11284,11285,11286,11287,11288,11289, &
& 11290,11291,11292,11293,11294,11295,11296,11297,11298,11299,11300, &
& 11301]
 index_cni(6801:6900)=[&
& 11302,11303,11304,11305,11306,11307,11308,11309,11310,11311,11312, &
& 11313,11314,11315,11316,11317,11318,11319,11320,11321,11322,11323, &
& 11324,11325,11326,11327,11328,11329,11330,11331,11332,11333,11334, &
& 11335,11336,11337,11338,11339,11340,11341,11342,11343,11344,11345, &
& 11346,11347,11348,11349,11350,11351,11352,11353,11354,11355,11356, &
& 11357,11358,11359,11360,11361,11362,11363,11364,11365,11366,11367, &
& 11368,11369,11370,11371,11372,11373,11374,11375,11376,11377,11378, &
& 11379,11380,11381,11382,11383,11384,11385,11386,11387,11388,11389, &
& 11390,11391,11392,11393,11394,11395,11396,11397,11398,11399,11400, &
& 11401]
 index_cni(6901:7000)=[&
& 11402,11403,11404,11405,11406,11407,11408,11409,11410,11411,11412, &
& 11413,11414,11415,11416,11417,11418,11419,11420,11421,11422,11423, &
& 11424,11425,11426,11427,11428,11429,11430,11431,11432,11433,11434, &
& 11435,11436,11437,11438,11439,11440,11441,11442,11443,11444,11445, &
& 11446,11447,11448,11449,11450,11451,11452,11453,11454,11455,11456, &
& 11457,11458,11459,11460,11461,11462,11463,11464,11465,11466,11467, &
& 11468,11469,11470,11471,11472,11473,11474,11475,11476,11477,11478, &
& 11479,11480,11481,11482,11483,11484,11485,11486,11487,11488,11489, &
& 11490,11491,11492,11493,11494,11495,11496,11497,11498,11499,11500, &
& 11501]
 index_cni(7001:7100)=[&
& 11502,11503,11504,11505,11506,11507,11508,11509,11510,11511,11512, &
& 11513,11514,11515,11516,11517,11518,11519,11520,11521,11522,11523, &
& 11524,11525,11526,11527,11528,11529,11530,11531,11532,11533,11534, &
& 11535,11536,11537,11538,11539,11540,11541,11542,11543,11544,11545, &
& 11546,11547,11548,11549,11550,11551,11552,11553,11554,11555,11556, &
& 11557,11558,11559,11560,11561,11562,11563,11564,11565,11566,11567, &
& 11568,11569,11570,11571,11572,11573,11574,11575,11576,11577,11578, &
& 11579,11580,11581,11582,11583,11584,11585,11586,11587,11588,11589, &
& 11590,11591,11592,11593,11594,11595,11596,11597,11598,11599,11600, &
& 11601]
 index_cni(7101:7200)=[&
& 11602,11603,11604,11605,11606,11607,11608,11609,11610,11611,11612, &
& 11613,11614,11615,11616,11617,11618,11619,11620,11621,11622,11623, &
& 11624,11625,11626,11627,11628,11629,11630,11631,11632,11633,11634, &
& 11635,11636,11637,11638,11639,11640,11641,11642,11643,11644,11645, &
& 11646,11647,11648,11649,11650,11651,11652,11653,11654,11655,11656, &
& 11657,11658,11659,11660,11661,11662,11663,11664,11665,11666,11667, &
& 11668,11669,11670,11671,11672,11673,11674,11675,11676,11677,11678, &
& 11679,11680,11681,11682,11683,11684,11685,11686,11687,11688,11689, &
& 11690,11691,11692,11693,11694,11695,11696,11697,11698,11699,11700, &
& 11701]
 index_cni(7201:7300)=[&
& 11702,11703,11704,11705,11706,11707,11708,11709,11710,11711,11712, &
& 11713,11714,11715,11716,11717,11718,11719,11720,11721,11722,11723, &
& 11724,11725,11726,11727,11728,11729,11730,11731,11732,11733,11734, &
& 11735,11736,11737,11738,11739,11740,11741,11742,11743,11744,11745, &
& 11746,11747,11748,11749,11750,11751,11752,11753,11754,11755,11756, &
& 11757,11758,11759,11760,11761,11762,11763,11764,11765,11766,11767, &
& 11768,11769,11770,11771,11772,11773,11774,11775,11776,11777,11778, &
& 11779,11780,11781,11782,11783,11784,11785,11786,11787,11788,11789, &
& 11790,11791,11792,11793,11794,11795,11796,11797,11798,11799,11800, &
& 11801]
 index_cni(7301:7400)=[&
& 11802,11803,11804,11805,11806,11807,11808,11809,11810,11811,11812, &
& 11813,11814,11815,11816,11817,11818,11819,11820,11821,11822,11823, &
& 11824,11825,11826,11827,11828,11829,11830,11831,11832,11833,11834, &
& 11835,11836,11837,11838,11839,11840,11841,11842,11843,11844,11845, &
& 11846,11847,11848,11849,11850,11851,11852,11853,11854,11855,11856, &
& 11857,11858,11859,11860,11861,11862,11863,11864,11865,11866,11867, &
& 11868,11869,11870,11871,11872,11873,11874,11875,11876,11877,11878, &
& 11879,11880,11881,11882,11883,11884,11885,11886,11887,11888,11889, &
& 11890,11891,11892,11893,11894,11895,11896,11897,11898,11899,11900, &
& 11901]
 index_cni(7401:7500)=[&
& 11902,11903,11904,11905,11906,11907,11908,11909,11910,11911,11912, &
& 11913,11914,11915,11916,11917,11918,11919,11920,11921,11922,11923, &
& 11924,11925,11926,11927,11928,11929,11930,11931,11932,11933,11934, &
& 11935,11936,11937,11938,11939,11940,11941,11942,11943,11944,11945, &
& 11946,11947,11948,11949,11950,11951,11952,11953,11954,11955,11956, &
& 11957,11958,11959,11960,11961,11962,11963,11964,11965,11966,11967, &
& 11968,11969,11970,11971,11972,11973,11974,11975,11976,11977,11978, &
& 11979,11980,11981,11982,11983,11984,11985,11986,11987,11988,11989, &
& 11990,11991,11992,11993,11994,11995,11996,11997,11998,11999,12000, &
& 12001]
 index_cni(7501:7600)=[&
& 12002,12003,12004,12005,12006,12007,12008,12009,12010,12011,12012, &
& 12013,12014,12015,12016,12017,12018,12019,12020,12021,12022,12023, &
& 12024,12025,12026,12027,12028,12029,12030,12031,12032,12033,12034, &
& 12035,12036,12037,12038,12039,12040,12041,12042,12043,12044,12045, &
& 12046,12047,12048,12049,12050,12051,12052,12053,12054,12055,12056, &
& 12057,12058,12059,12060,12061,12062,12063,12064,12065,12066,12067, &
& 12068,12069,12070,12071,12072,12073,12074,12075,12076,12077,12078, &
& 12079,12080,12081,12082,12083,12084,12085,12086,12087,12088,12089, &
& 12090,12091,12092,12093,12094,12095,12096,12097,12098,12099,12100, &
& 12101]
 index_cni(7601:7700)=[&
& 12102,12103,12104,12105,12106,12107,12108,12109,12110,12111,12112, &
& 12113,12114,12115,12116,12117,12118,12119,12120,12121,12122,12123, &
& 12124,12125,12126,12127,12128,12129,12130,12131,12132,12133,12134, &
& 12135,12136,12137,12138,12139,12140,12141,12142,12143,12144,12145, &
& 12146,12147,12148,12149,12150,12151,12152,12153,12154,12155,12156, &
& 12157,12158,12159,12160,12161,12162,12163,12164,12165,12166,12167, &
& 12168,12169,12170,12171,12172,12173,12174,12175,12176,12177,12178, &
& 12179,12180,12181,12182,12183,12184,12185,12186,12187,12188,12189, &
& 12190,12191,12192,12193,12194,12195,12196,12197,12198,12199,12200, &
& 12201]
 index_cni(7701:7800)=[&
& 12202,12203,12204,12205,12206,12207,12208,12209,12210,12211,12212, &
& 12213,12214,12215,12216,12217,12218,12219,12220,12221,12222,12223, &
& 12224,12225,12226,12227,12228,12229,12230,12231,12232,12233,12234, &
& 12235,12236,12237,12238,12239,12240,12241,12242,12243,12244,12245, &
& 12246,12247,12248,12249,12250,12251,12252,12253,12254,12255,12256, &
& 12257,12258,12259,12260,12261,12262,12263,12264,12265,12266,12267, &
& 12268,12269,12270,12271,12272,12273,12274,12275,12276,12277,12278, &
& 12279,12280,12281,12282,12283,12284,12285,12286,12287,12288,12289, &
& 12290,12291,12292,12293,12294,12295,12296,12297,12298,12299,12300, &
& 12301]
 index_cni(7801:7900)=[&
& 12302,12303,12304,12305,12306,12307,12308,12309,12310,12311,12312, &
& 12313,12314,12315,12316,12317,12318,12319,12320,12321,12322,12323, &
& 12324,12325,12326,12327,12328,12329,12330,12331,12332,12333,12334, &
& 12335,12336,12337,12338,12339,12340,12341,12342,12343,12344,12345, &
& 12346,12347,12348,12349,12350,12351,12352,12353,12354,12355,12356, &
& 12357,12358,12359,12360,12361,12362,12363,12364,12365,12366,12367, &
& 12368,12369,12370,12371,12372,12373,12374,12375,12376,12377,12378, &
& 12379,12380,12381,12382,12383,12384,12385,12386,12387,12388,12389, &
& 12390,12391,12392,12393,12394,12395,12396,12397,12398,12399,12400, &
& 12401]
 index_cni(7901:8000)=[&
& 12402,12403,12404,12405,12406,12407,12408,12409,12410,12411,12412, &
& 12413,12414,12415,12416,12417,12418,12419,12420,12421,12422,12423, &
& 12424,12425,12426,12427,12428,12429,12430,12431,12432,12433,12434, &
& 12435,12436,12437,12438,12439,12440,12441,12442,12443,12444,12445, &
& 12446,12447,12448,12449,12450,12451,12452,12453,12454,12455,12456, &
& 12457,12458,12459,12460,12461,12462,12463,12464,12465,12466,12467, &
& 12468,12469,12470,12471,12472,12473,12474,12475,12476,12477,12478, &
& 12479,12480,12481,12482,12483,12484,12485,12486,12487,12488,12489, &
& 12490,12491,12492,12493,12494,12495,12496,12497,12498,12499,12500, &
& 12501]
 index_cni(8001:8100)=[&
& 12502,12503,12504,12505,12506,12507,12508,12509,12510,12511,12512, &
& 12513,12514,12515,12516,12517,12518,12519,12520,12521,12522,12523, &
& 12524,12525,12526,12527,12528,12529,12530,12531,12532,12533,12534, &
& 12535,12536,12537,12538,12539,12540,12541,12542,12543,12544,12545, &
& 12546,12547,12548,12549,12550,12551,12552,12553,12554,12555,12556, &
& 12557,12558,12559,12560,12561,12562,12563,12564,12565,12566,12567, &
& 12568,12569,12570,12571,12572,12573,12574,12575,12576,12577,12578, &
& 12579,12580,12581,12582,12583,12584,12585,12586,12587,12588,12589, &
& 12590,12591,12592,12593,12594,12595,12596,12597,12598,12599,12600, &
& 12601]
 index_cni(8101:8200)=[&
& 12602,12603,12604,12605,12606,12607,12608,12609,12610,12611,12612, &
& 12613,12614,12615,12616,12617,12618,12619,12620,12621,12622,12623, &
& 12624,12625,12626,12627,12628,12629,12630,12631,12632,12633,12634, &
& 12635,12636,12637,12638,12639,12640,12641,12642,12643,12644,12645, &
& 12646,12647,12648,12649,12650,12651,12652,12653,12654,12655,12656, &
& 12657,12658,12659,12660,12661,12662,12663,12664,12665,12666,12667, &
& 12668,12669,12670,12671,12672,12673,12674,12675,12676,12677,12678, &
& 12679,12680,12681,12682,12683,12684,12685,12686,12687,12688,12689, &
& 12690,12691,12692,12693,12694,12695,12696,12697,12698,12699,12700, &
& 12701]
 index_cni(8201:8300)=[&
& 12702,12703,12704,12705,12706,12707,12708,12709,12710,12711,12712, &
& 12713,12714,12715,12716,12717,12718,12719,12720,12721,12722,12723, &
& 12724,12725,12726,12727,12728,12729,12730,12731,12732,12733,12734, &
& 12735,12736,12737,12738,12739,12740,12741,12742,12743,12744,12745, &
& 12746,12747,12748,12749,12750,12751,12752,12753,12754,12755,12756, &
& 12757,12758,12759,12760,12761,12762,12763,12764,12765,12766,12767, &
& 12768,12769,12770,12771,12772,12773,12774,12775,12776,12777,12778, &
& 12779,12780,12781,12782,12783,12784,12785,12786,12787,12788,12789, &
& 12790,12791,12792,12793,12794,12795,12796,12797,12798,12799,12800, &
& 12801]
 index_cni(8301:8400)=[&
& 12802,12803,12804,12805,12806,12807,12808,12809,12810,12811,12812, &
& 12813,12814,12815,12816,12817,12818,12819,12820,12821,12822,12823, &
& 12824,12825,12826,12827,12828,12829,12830,12831,12832,12833,12834, &
& 12835,12836,12837,12838,12839,12840,12841,12842,12843,12844,12845, &
& 12846,12847,12848,12849,12850,12851,12852,12853,12854,12855,12856, &
& 12857,12858,12859,12860,12861,12862,12863,12864,12865,12866,12867, &
& 12868,12869,12870,12871,12872,12873,12874,12875,12876,12877,12878, &
& 12879,12880,12881,12882,12883,12884,12885,12886,12887,12888,12889, &
& 12890,12891,12892,12893,12894,12895,12896,12897,12898,12899,12900, &
& 12901]
 index_cni(8401:8500)=[&
& 12902,12903,12904,12905,12906,12907,12908,12909,12910,12911,12912, &
& 12913,12914,12915,12916,12917,12918,12919,12920,12921,12922,12923, &
& 12924,12925,12926,12927,12928,12929,12930,12931,12932,12933,12934, &
& 12935,12936,12937,12938,12939,12940,12941,12942,12943,12944,12945, &
& 12946,12947,12948,12949,12950,12951,12952,12953,12954,12955,12956, &
& 12957,12958,12959,12960,12961,12962,12963,12964,12965,12966,12967, &
& 12968,12969,12970,12971,12972,12973,12974,12975,12976,12977,12978, &
& 12979,12980,12981,12982,12983,12984,12985,12986,12987,12988,12989, &
& 12990,12991,12992,12993,12994,12995,12996,12997,12998,12999,13000, &
& 13001]
 index_cni(8501:8600)=[&
& 13002,13003,13004,13005,13006,13007,13008,13009,13010,13011,13012, &
& 13013,13014,13015,13016,13017,13018,13019,13020,13021,13022,13023, &
& 13024,13025,13026,13027,13028,13029,13030,13031,13032,13033,13034, &
& 13035,13036,13037,13038,13039,13040,13041,13042,13043,13044,13045, &
& 13046,13047,13048,13049,13050,13051,13052,13053,13054,13055,13056, &
& 13057,13058,13059,13060,13061,13062,13063,13064,13065,13066,13067, &
& 13068,13069,13070,13071,13072,13073,13074,13075,13076,13077,13078, &
& 13079,13080,13081,13082,13083,13084,13085,13086,13087,13088,13089, &
& 13090,13091,13092,13093,13094,13095,13096,13097,13098,13099,13100, &
& 13101]
 index_cni(8601:8700)=[&
& 13102,13103,13104,13105,13106,13107,13108,13109,13110,13111,13112, &
& 13113,13114,13115,13116,13117,13118,13119,13120,13121,13122,13123, &
& 13124,13125,13126,13127,13128,13129,13130,13131,13132,13133,13134, &
& 13135,13136,13137,13138,13139,13140,13141,13142,13143,13144,13145, &
& 13146,13147,13148,13149,13150,13151,13152,13153,13154,13155,13156, &
& 13157,13158,13159,13160,13161,13162,13163,13164,13165,13166,13167, &
& 13168,13169,13170,13171,13172,13173,13174,13175,13176,13177,13178, &
& 13179,13180,13181,13182,13183,13184,13185,13186,13187,13188,13189, &
& 13190,13191,13192,13193,13194,13195,13196,13197,13198,13199,13200, &
& 13201]
 index_cni(8701:8800)=[&
& 13202,13203,13204,13205,13206,13207,13208,13209,13210,13211,13212, &
& 13213,13214,13215,13216,13217,13218,13219,13220,13221,13222,13223, &
& 13224,13225,13226,13227,13228,13229,13230,13231,13232,13233,13234, &
& 13235,13236,13237,13238,13239,13240,13241,13242,13243,13244,13245, &
& 13246,13247,13248,13249,13250,13251,13252,13253,13254,13255,13256, &
& 13257,13258,13259,13260,13261,13262,13263,13264,13265,13266,13267, &
& 13268,13269,13270,13271,13272,13273,13274,13275,13276,13277,13278, &
& 13279,13280,13281,13282,13283,13284,13285,13286,13287,13288,13289, &
& 13290,13291,13292,13293,13294,13295,13296,13297,13298,13299,13300, &
& 13301]
 index_cni(8801:8900)=[&
& 13302,13303,13304,13305,13306,13307,13308,13309,13310,13311,13312, &
& 13313,13314,13315,13316,13317,13318,13319,13320,13321,13322,13323, &
& 13324,13325,13326,13327,13328,13329,13330,13331,13332,13333,13334, &
& 13335,13336,13337,13338,13339,13340,13341,13342,13343,13344,13345, &
& 13346,13347,13348,13349,13350,13351,13352,13353,13354,13355,13356, &
& 13357,13358,13359,13360,13361,13362,13363,13364,13365,13366,13367, &
& 13368,13369,13370,13371,13372,13373,13374,13375,13376,13377,13378, &
& 13379,13380,13381,13382,13383,13384,13385,13386,13387,13388,13389, &
& 13390,13391,13392,13393,13394,13395,13396,13397,13398,13399,13400, &
& 13401]
 index_cni(8901:9000)=[&
& 13402,13403,13404,13405,13406,13407,13408,13409,13410,13411,13412, &
& 13413,13414,13415,13416,13417,13418,13419,13420,13421,13422,13423, &
& 13424,13425,13426,13427,13428,13429,13430,13431,13432,13433,13434, &
& 13435,13436,13437,13438,13439,13440,13441,13442,13443,13444,13445, &
& 13446,13447,13448,13449,13450,13451,13452,13453,13454,13455,13456, &
& 13457,13458,13459,13460,13461,13462,13463,13464,13465,13466,13467, &
& 13468,13469,13470,13471,13472,13473,13474,13475,13476,13477,13478, &
& 13479,13480,13481,13482,13483,13484,13485,13486,13487,13488,13489, &
& 13490,13491,13492,13493,13494,13495,13496,13497,13498,13499,13500, &
& 13501]
 index_cni(9001:9100)=[&
& 13502,13503,13504,13505,13506,13507,13508,13509,13510,13511,13512, &
& 13513,13514,13515,13516,13517,13518,13519,13520,13521,13522,13523, &
& 13524,13525,13526,13527,13528,13529,13530,13531,13532,13533,13534, &
& 13535,13536,13537,13538,13539,13540,13541,13542,13543,13544,13545, &
& 13546,13547,13548,13549,13550,13551,13552,13553,13554,13555,13556, &
& 13557,13558,13559,13560,13561,13562,13563,13564,13565,13566,13567, &
& 13568,13569,13570,13571,13572,13573,13574,13575,13576,13577,13578, &
& 13579,13580,13581,13582,13583,13584,13585,13586,13587,13588,13589, &
& 13590,13591,13592,13593,13594,13595,13596,13597,13598,13599,13600, &
& 13601]
 index_cni(9101:9200)=[&
& 13602,13603,13604,13605,13606,13607,13608,13609,13610,13611,13612, &
& 13613,13614,13615,13616,13617,13618,13619,13620,13621,13622,13623, &
& 13624,13625,13626,13627,13628,13629,13630,13631,13632,13633,13634, &
& 13635,13636,13637,13638,13639,13640,13641,13642,13643,13644,13645, &
& 13646,13647,13648,13649,13650,13651,13652,13653,13654,13655,13656, &
& 13657,13658,13659,13660,13661,13662,13663,13664,13665,13666,13667, &
& 13668,13669,13670,13671,13672,13673,13674,13675,13676,13677,13678, &
& 13679,13680,13681,13682,13683,13684,13685,13686,13687,13688,13689, &
& 13690,13691,13692,13693,13694,13695,13696,13697,13698,13699,13700, &
& 13701]
 index_cni(9201:9300)=[&
& 13702,13703,13704,13705,13706,13707,13708,13709,13710,13711,13712, &
& 13713,13714,13715,13716,13717,13718,13719,13720,13721,13722,13723, &
& 13724,13725,13726,13727,13728,13729,13730,13731,13732,13733,13734, &
& 13735,13736,13737,13738,13739,13740,13741,13742,13743,13744,13745, &
& 13746,13747,13748,13749,13750,13751,13752,13753,13754,13755,13756, &
& 13757,13758,13759,13760,13761,13762,13763,13764,13765,13766,13767, &
& 13768,13769,13770,13771,13772,13773,13774,13775,13776,13777,13778, &
& 13779,13780,13781,13782,13783,13784,13785,13786,13787,13788,13789, &
& 13790,13791,13792,13793,13794,13795,13796,13797,13798,13799,13800, &
& 13801]
 index_cni(9301:9400)=[&
& 13802,13803,13804,13805,13806,13807,13808,13809,13810,13811,13812, &
& 13813,13814,13815,13816,13817,13818,13819,13820,13821,13822,13823, &
& 13824,13825,13826,13827,13828,13829,13830,13831,13832,13833,13834, &
& 13835,13836,13837,13838,13839,13840,13841,13842,13843,13844,13845, &
& 13846,13847,13848,13849,13850,13851,13852,13853,13854,13855,13856, &
& 13857,13858,13859,13860,13861,13862,13863,13864,13865,13866,13867, &
& 13868,13869,13870,13871,13872,13873,13874,13875,13876,13877,13878, &
& 13879,13880,13881,13882,13883,13884,13885,13886,13887,13888,13889, &
& 13890,13891,13892,13893,13894,13895,13896,13897,13898,13899,13900, &
& 13901]
 index_cni(9401:9500)=[&
& 13902,13903,13904,13905,13906,13907,13908,13909,13910,13911,13912, &
& 13913,13914,13915,13916,13917,13918,13919,13920,13921,13922,13923, &
& 13924,13925,13926,13927,13928,13929,13930,13931,13932,13933,13934, &
& 13935,13936,13937,13938,13939,13940,13941,13942,13943,13944,13945, &
& 13946,13947,13948,13949,13950,13951,13952,13953,13954,13955,13956, &
& 13957,13958,13959,13960,13961,13962,13963,13964,13965,13966,13967, &
& 13968,13969,13970,13971,13972,13973,13974,13975,13976,13977,13978, &
& 13979,13980,13981,13982,13983,13984,13985,13986,13987,13988,13989, &
& 13990,13991,13992,13993,13994,13995,13996,13997,13998,13999,14000, &
& 14001]
 index_cni(9501:9600)=[&
& 14002,14003,14004,14005,14006,14007,14008,14009,14010,14011,14012, &
& 14013,14014,14015,14016,14017,14018,14019,14020,14021,14022,14023, &
& 14024,14025,14026,14027,14028,14029,14030,14031,14032,14033,14034, &
& 14035,14036,14037,14038,14039,14040,14041,14042,14043,14044,14045, &
& 14046,14047,14048,14049,14050,14051,14052,14053,14054,14055,14056, &
& 14057,14058,14059,14060,14061,14062,14063,14064,14065,14066,14067, &
& 14068,14069,14070,14071,14072,14073,14074,14075,14076,14077,14078, &
& 14079,14080,14081,14082,14083,14084,14085,14086,14087,14088,14089, &
& 14090,14091,14092,14093,14094,14095,14096,14097,14098,14099,14100, &
& 14101]
 index_cni(9601:9700)=[&
& 14102,14103,14104,14105,14106,14107,14108,14109,14110,14111,14112, &
& 14113,14114,14115,14116,14117,14118,14119,14120,14121,14122,14123, &
& 14124,14125,14126,14127,14128,14129,14130,14131,14132,14133,14134, &
& 14135,14136,14137,14138,14139,14140,14141,14142,14143,14144,14145, &
& 14146,14147,14148,14149,14150,14151,14152,14153,14154,14155,14156, &
& 14157,14158,14159,14160,14161,14162,14163,14164,14165,14166,14167, &
& 14168,14169,14170,14171,14172,14173,14174,14175,14176,14177,14178, &
& 14179,14180,14181,14182,14183,14184,14185,14186,14187,14188,14189, &
& 14190,14191,14192,14193,14194,14195,14196,14197,14198,14199,14200, &
& 14201]
 index_cni(9701:9800)=[&
& 14202,14203,14204,14205,14206,14207,14208,14209,14210,14211,14212, &
& 14213,14214,14215,14216,14217,14218,14219,14220,14221,14222,14223, &
& 14224,14225,14226,14227,14228,14229,14230,14231,14232,14233,14234, &
& 14235,14236,14237,14238,14239,14240,14241,14242,14243,14244,14245, &
& 14246,14247,14248,14249,14250,14251,14252,14253,14254,14255,14256, &
& 14257,14258,14259,14260,14261,14262,14263,14264,14265,14266,14267, &
& 14268,14269,14270,14271,14272,14273,14274,14275,14276,14277,14278, &
& 14279,14280,14281,14282,14283,14284,14285,14286,14287,14288,14289, &
& 14290,14291,14292,14293,14294,14295,14296,14297,14298,14299,14300, &
& 14301]
 index_cni(9801:9900)=[&
& 14302,14303,14304,14305,14306,14307,14308,14309,14310,14311,14312, &
& 14313,14314,14315,14316,14317,14318,14319,14320,14321,14322,14323, &
& 14324,14325,14326,14327,14328,14329,14330,14331,14332,14333,14334, &
& 14335,14336,14337,14338,14339,14340,14341,14342,14343,14344,14345, &
& 14346,14347,14348,14349,14350,14351,14352,14353,14354,14355,14356, &
& 14357,14358,14359,14360,14361,14362,14363,14364,14365,14366,14367, &
& 14368,14369,14370,14371,14372,14373,14374,14375,14376,14377,14378, &
& 14379,14380,14381,14382,14383,14384,14385,14386,14387,14388,14389, &
& 14390,14391,14392,14393,14394,14395,14396,14397,14398,14399,14400, &
& 14401]
 index_cni(9901:10000)=[&
& 14402,14403,14404,14405,14406,14407,14408,14409,14410,14411,14412, &
& 14413,14414,14415,14416,14417,14418,14419,14420,14421,14422,14423, &
& 14424,14425,14426,14427,14428,14429,14430,14431,14432,14433,14434, &
& 14435,14436,14437,14438,14439,14440,14441,14442,14443,14444,14445, &
& 14446,14447,14448,14449,14450,14451,14452,14453,14454,14455,14456, &
& 14457,14458,14459,14460,14461,14462,14463,14464,14465,14466,14467, &
& 14468,14469,14470,14471,14472,14473,14474,14475,14476,14477,14478, &
& 14479,14480,14481,14482,14483,14484,14485,14486,14487,14488,14489, &
& 14490,14491,14492,14493,14494,14495,14496,14497,14498,14499,14500, &
& 14501]
 index_cni(10001:10100)=[&
& 14502,14503,14504,14505,14506,14507,14508,14509,14510,14511,14512, &
& 14513,14514,14515,14516,14517,14518,14519,14520,14521,14522,14523, &
& 14524,14525,14526,14527,14528,14529,14530,14531,14532,14533,14534, &
& 14535,14536,14537,14538,14539,14540,14541,14542,14543,14544,14545, &
& 14546,14547,14548,14549,14550,14551,14552,14553,14554,14555,14556, &
& 14557,14558,14559,14560,14561,14562,14563,14564,14565,14566,14567, &
& 14568,14569,14570,14571,14572,14573,14574,14575,14576,14577,14578, &
& 14579,14580,14581,14582,14583,14584,14585,14586,14587,14588,14589, &
& 14590,14591,14592,14593,14594,14595,14596,14597,14598,14599,14600, &
& 14601]
 index_cni(10101:10200)=[&
& 14602,14603,14604,14605,14606,14607,14608,14609,14610,14611,14612, &
& 14613,14614,14615,14616,14617,14618,14619,14620,14621,14622,14623, &
& 14624,14625,14626,14627,14628,14629,14630,14631,14632,14633,14634, &
& 14635,14636,14637,14638,14639,14640,14641,14642,14643,14644,14645, &
& 14646,14647,14648,14649,14650,14651,14652,14653,14654,14655,14656, &
& 14657,14658,14659,14660,14661,14662,14663,14664,14665,14666,14667, &
& 14668,14669,14670,14671,14672,14673,14674,14675,14676,14677,14678, &
& 14679,14680,14681,14682,14683,14684,14685,14686,14687,14688,14689, &
& 14690,14691,14692,14693,14694,14695,14696,14697,14698,14699,14700, &
& 14701]
 index_cni(10201:10300)=[&
& 14702,14703,14704,14705,14706,14707,14708,14709,14710,14711,14712, &
& 14713,14714,14715,14716,14717,14718,14719,14720,14721,14722,14723, &
& 14724,14725,14726,14727,14728,14729,14730,14731,14732,14733,14734, &
& 14735,14736,14737,14738,14739,14740,14741,14742,14743,14744,14745, &
& 14746,14747,14748,14749,14750,14751,14752,14753,14754,14755,14756, &
& 14757,14758,14759,14760,14761,14762,14763,14764,14765,14766,14767, &
& 14768,14769,14770,14771,14772,14773,14774,14775,14776,14777,14778, &
& 14779,14780,14781,14782,14783,14784,14785,14786,14787,14788,14789, &
& 14790,14791,14792,14793,14794,14795,14796,14797,14798,14799,14800, &
& 14801]
 index_cni(10301:10400)=[&
& 14802,14803,14804,14805,14806,14807,14808,14809,14810,14811,14812, &
& 14813,14814,14815,14816,14817,14818,14819,14820,14821,14822,14823, &
& 14824,14825,14826,14827,14828,14829,14830,14831,14832,14833,14834, &
& 14835,14836,14837,14838,14839,14840,14841,14842,14843,14844,14845, &
& 14846,14847,14848,14849,14850,14851,14852,14853,14854,14855,14856, &
& 14857,14858,14859,14860,14861,14862,14863,14864,14865,14866,14867, &
& 14868,14869,14870,14871,14872,14873,14874,14875,14876,14877,14878, &
& 14879,14880,14881,14882,14883,14884,14885,14886,14887,14888,14889, &
& 14890,14891,14892,14893,14894,14895,14896,14897,14898,14899,14900, &
& 14901]
 index_cni(10401:10500)=[&
& 14902,14903,14904,14905,14906,14907,14908,14909,14910,14911,14912, &
& 14913,14914,14915,14916,14917,14918,14919,14920,14921,14922,14923, &
& 14924,14925,14926,14927,14928,14929,14930,14931,14932,14933,14934, &
& 14935,14936,14937,14938,14939,14940,14941,14942,14943,14944,14945, &
& 14946,14947,14948,14949,14950,14951,14952,14953,14954,14955,14956, &
& 14957,14958,14959,14960,14961,14962,14963,14964,14965,14966,14967, &
& 14968,14969,14970,14971,14972,14973,14974,14975,14976,14977,14978, &
& 14979,14980,14981,14982,14983,14984,14985,14986,14987,14988,14989, &
& 14990,14991,14992,14993,14994,14995,14996,14997,14998,14999,15000, &
& 15001]
 index_cni(10501:10600)=[&
& 15002,15003,15004,15005,15006,15007,15008,15009,15010,15011,15012, &
& 15013,15014,15015,15016,15017,15018,15019,15020,15021,15022,15023, &
& 15024,15025,15026,15027,15028,15029,15030,15031,15032,15033,15034, &
& 15035,15036,15037,15038,15039,15040,15041,15042,15043,15044,15045, &
& 15046,15047,15048,15049,15050,15051,15052,15053,15054,15055,15056, &
& 15057,15058,15059,15060,15061,15062,15063,15064,15065,15066,15067, &
& 15068,15069,15070,15071,15072,15073,15074,15075,15076,15077,15078, &
& 15079,15080,15081,15082,15083,15084,15085,15086,15087,15088,15089, &
& 15090,15091,15092,15093,15094,15095,15096,15097,15098,15099,15100, &
& 15101]
 index_cni(10601:10700)=[&
& 15102,15103,15104,15105,15106,15107,15108,15109,15110,15111,15112, &
& 15113,15114,15115,15116,15117,15118,15119,15120,15121,15122,15123, &
& 15124,15125,15126,15127,15128,15129,15130,15131,15132,15133,15134, &
& 15135,15136,15137,15138,15139,15140,15141,15142,15143,15144,15145, &
& 15146,15147,15148,15149,15150,15151,15152,15153,15154,15155,15156, &
& 15157,15158,15159,15160,15161,15162,15163,15164,15165,15166,15167, &
& 15168,15169,15170,15171,15172,15173,15174,15175,15176,15177,15178, &
& 15179,15180,15181,15182,15183,15184,15185,15186,15187,15188,15189, &
& 15190,15191,15192,15193,15194,15195,15196,15197,15198,15199,15200, &
& 15201]
 index_cni(10701:10800)=[&
& 15202,15203,15204,15205,15206,15207,15208,15209,15210,15211,15212, &
& 15213,15214,15215,15216,15217,15218,15219,15220,15221,15222,15223, &
& 15224,15225,15226,15227,15228,15229,15230,15231,15232,15233,15234, &
& 15235,15236,15237,15238,15239,15240,15241,15242,15243,15244,15245, &
& 15246,15247,15248,15249,15250,15251,15252,15253,15254,15255,15256, &
& 15257,15258,15259,15260,15261,15262,15263,15264,15265,15266,15267, &
& 15268,15269,15270,15271,15272,15273,15274,15275,15276,15277,15278, &
& 15279,15280,15281,15282,15283,15284,15285,15286,15287,15288,15289, &
& 15290,15291,15292,15293,15294,15295,15296,15297,15298,15299,15300, &
& 15301]
 index_cni(10801:10900)=[&
& 15302,15303,15304,15305,15306,15307,15308,15309,15310,15311,15312, &
& 15313,15314,15315,15316,15317,15318,15319,15320,15321,15322,15323, &
& 15324,15325,15326,15327,15328,15329,15330,15331,15332,15333,15334, &
& 15335,15336,15337,15338,15339,15340,15341,15342,15343,15344,15345, &
& 15346,15347,15348,15349,15350,15351,15352,15353,15354,15355,15356, &
& 15357,15358,15359,15360,15361,15362,15363,15364,15365,15366,15367, &
& 15368,15369,15370,15371,15372,15373,15374,15375,15376,15377,15378, &
& 15379,15380,15381,15382,15383,15384,15385,15386,15387,15388,15389, &
& 15390,15391,15392,15393,15394,15395,15396,15397,15398,15399,15400, &
& 15401]
 index_cni(10901:11000)=[&
& 15402,15403,15404,15405,15406,15407,15408,15409,15410,15411,15412, &
& 15413,15414,15415,15416,15417,15418,15419,15420,15421,15422,15423, &
& 15424,15425,15426,15427,15428,15429,15430,15431,15432,15433,15434, &
& 15435,15436,15437,15438,15439,15440,15441,15442,15443,15444,15445, &
& 15446,15447,15448,15449,15450,15451,15452,15453,15454,15455,15456, &
& 15457,15458,15459,15460,15461,15462,15463,15464,15465,15466,15467, &
& 15468,15469,15470,15471,15472,15473,15474,15475,15476,15477,15478, &
& 15479,15480,15481,15482,15483,15484,15485,15486,15487,15488,15489, &
& 15490,15491,15492,15493,15494,15495,15496,15497,15498,15499,15500, &
& 15501]
 index_cni(11001:11100)=[&
& 15502,15503,15504,15505,15506,15507,15508,15509,15510,15511,15512, &
& 15513,15514,15515,15516,15517,15518,15519,15520,15521,15522,15523, &
& 15524,15525,15526,15527,15528,15529,15530,15531,15532,15533,15534, &
& 15535,15536,15537,15538,15539,15540,15541,15542,15543,15544,15545, &
& 15546,15547,15548,15549,15550,15551,15552,15553,15554,15555,15556, &
& 15557,15558,15559,15560,15561,15562,15563,15564,15565,15566,15567, &
& 15568,15569,15570,15571,15572,15573,15574,15575,15576,15577,15578, &
& 15579,15580,15581,15582,15583,15584,15585,15586,15587,15588,15589, &
& 15590,15591,15592,15593,15594,15595,15596,15597,15598,15599,15600, &
& 15601]
 index_cni(11101:11200)=[&
& 15602,15603,15604,15605,15606,15607,15608,15609,15610,15611,15612, &
& 15613,15614,15615,15616,15617,15618,15619,15620,15621,15622,15623, &
& 15624,15625,15626,15627,15628,15629,15630,15631,15632,15633,15634, &
& 15635,15636,15637,15638,15639,15640,15641,15642,15643,15644,15645, &
& 15646,15647,15648,15649,15650,15651,15652,15653,15654,15655,15656, &
& 15657,15658,15659,15660,15661,15662,15663,15664,15665,15666,15667, &
& 15668,15669,15670,15671,15672,15673,15674,15675,15676,15677,15678, &
& 15679,15680,15681,15682,15683,15684,15685,15686,15687,15688,15689, &
& 15690,15691,15692,15693,15694,15695,15696,15697,15698,15699,15700, &
& 15701]
 index_cni(11201:11300)=[&
& 15702,15703,15704,15705,15706,15707,15708,15709,15710,15711,15712, &
& 15713,15714,15715,15716,15717,15718,15719,15720,15721,15722,15723, &
& 15724,15725,15726,15727,15728,15729,15730,15731,15732,15733,15734, &
& 15735,15736,15737,15738,15739,15740,15741,15742,15743,15744,15745, &
& 15746,15747,15748,15749,15750,15751,15752,15753,15754,15755,15756, &
& 15757,15758,15759,15760,15761,15762,15763,15764,15765,15766,15767, &
& 15768,15769,15770,15771,15772,15773,15774,15775,15776,15777,15778, &
& 15779,15780,15781,15782,15783,15784,15785,15786,15787,15788,15789, &
& 15790,15791,15792,15793,15794,15795,15796,15797,15798,15799,15800, &
& 15801]
 index_cni(11301:11400)=[&
& 15802,15803,15804,15805,15806,15807,15808,15809,15810,15811,15812, &
& 15813,15814,15815,15816,15817,15818,15819,15820,15821,15822,15823, &
& 15824,15825,15826,15827,15828,15829,15830,15831,15832,15833,15834, &
& 15835,15836,15837,15838,15839,15840,15841,15842,15843,15844,15845, &
& 15846,15847,15848,15849,15850,15851,15852,15853,15854,15855,15856, &
& 15857,15858,15859,15860,15861,15862,15863,15864,15865,15866,15867, &
& 15868,15869,15870,15871,15872,15873,15874,15875,15876,15877,15878, &
& 15879,15880,15881,15882,15883,15884,15885,15886,15887,15888,15889, &
& 15890,15891,15892,15893,15894,15895,15896,15897,15898,15899,15900, &
& 15901]
 index_cni(11401:11500)=[&
& 15902,15903,15904,15905,15906,15907,15908,15909,15910,15911,15912, &
& 15913,15914,15915,15916,15917,15918,15919,15920,15921,15922,15923, &
& 15924,15925,15926,15927,15928,15929,15930,15931,15932,15933,15934, &
& 15935,15936,15937,15938,15939,15940,15941,15942,15943,15944,15945, &
& 15946,15947,15948,15949,15950,15951,15952,15953,15954,15955,15956, &
& 15957,15958,15959,15960,15961,15962,15963,15964,15965,15966,15967, &
& 15968,15969,15970,15971,15972,15973,15974,15975,15976,15977,15978, &
& 15979,15980,15981,15982,15983,15984,15985,15986,15987,15988,15989, &
& 15990,15991,15992,15993,15994,15995,15996,15997,15998,15999,16000, &
& 16001]
 index_cni(11501:11600)=[&
& 16002,16003,16004,16005,16006,16007,16008,16009,16010,16011,16012, &
& 16013,16014,16015,16016,16017,16018,16019,16020,16021,16022,16023, &
& 16024,16025,16026,16027,16028,16029,16030,16031,16032,16033,16034, &
& 16035,16036,16037,16038,16039,16040,16041,16042,16043,16044,16045, &
& 16046,16047,16048,16049,16050,16051,16052,16053,16054,16055,16056, &
& 16057,16058,16059,16060,16061,16062,16063,16064,16065,16066,16067, &
& 16068,16069,16070,16071,16072,16073,16074,16075,16076,16077,16078, &
& 16079,16080,16081,16082,16083,16084,16085,16086,16087,16088,16089, &
& 16090,16091,16092,16093,16094,16095,16096,16097,16098,16099,16100, &
& 16101]
 index_cni(11601:11700)=[&
& 16102,16103,16104,16105,16106,16107,16108,16109,16110,16111,16112, &
& 16113,16114,16115,16116,16117,16118,16119,16120,16121,16122,16123, &
& 16124,16125,16126,16127,16128,16129,16130,16131,16132,16133,16134, &
& 16135,16136,16137,16138,16139,16140,16141,16142,16143,16144,16145, &
& 16146,16147,16148,16149,16150,16151,16152,16153,16154,16155,16156, &
& 16157,16158,16159,16160,16161,16162,16163,16164,16165,16166,16167, &
& 16168,16169,16170,16171,16172,16173,16174,16175,16176,16177,16178, &
& 16179,16180,16181,16182,16183,16184,16185,16186,16187,16188,16189, &
& 16190,16191,16192,16193,16194,16195,16196,16197,16198,16199,16200, &
& 16201]
 index_cni(11701:11800)=[&
& 16202,16203,16204,16205,16206,16207,16208,16209,16210,16211,16212, &
& 16213,16214,16215,16216,16217,16218,16219,16220,16221,16222,16223, &
& 16224,16225,16226,16227,16228,16229,16230,16231,16232,16233,16234, &
& 16235,16236,16237,16238,16239,16240,16241,16242,16243,16244,16245, &
& 16246,16247,16248,16249,16250,16251,16252,16253,16254,16255,16256, &
& 16257,16258,16259,16260,16261,16262,16263,16264,16265,16266,16267, &
& 16268,16269,16270,16271,16272,16273,16274,16275,16276,16277,16278, &
& 16279,16280,16281,16282,16283,16284,16285,16286,16287,16288,16289, &
& 16290,16291,16292,16293,16294,16295,16296,16297,16298,16299,16300, &
& 16301]
 index_cni(11801:11900)=[&
& 16302,16303,16304,16305,16306,16307,16308,16309,16310,16311,16312, &
& 16313,16314,16315,16316,16317,16318,16319,16320,16321,16322,16323, &
& 16324,16325,16326,16327,16328,16329,16330,16331,16332,16333,16334, &
& 16335,16336,16337,16338,16339,16340,16341,16342,16343,16344,16345, &
& 16346,16347,16348,16349,16350,16351,16352,16353,16354,16355,16356, &
& 16357,16358,16359,16360,16361,16362,16363,16364,16365,16366,16367, &
& 16368,16369,16370,16371,16372,16373,16374,16375,16376,16377,16378, &
& 16379,16380,16381,16382,16383,16384,16385,16386,16387,16388,16389, &
& 16390,16391,16392,16393,16394,16395,16396,16397,16398,16399,16400, &
& 16401]
 index_cni(11901:12000)=[&
& 16402,16403,16404,16405,16406,16407,16408,16409,16410,16411,16412, &
& 16413,16414,16415,16416,16417,16418,16419,16420,16421,16422,16423, &
& 16424,16425,16426,16427,16428,16429,16430,16431,16432,16433,16434, &
& 16435,16436,16437,16438,16439,16440,16441,16442,16443,16444,16445, &
& 16446,16447,16448,16449,16450,16451,16452,16453,16454,16455,16456, &
& 16457,16458,16459,16460,16461,16462,16463,16464,16465,16466,16467, &
& 16468,16469,16470,16471,16472,16473,16474,16475,16476,16477,16478, &
& 16479,16480,16481,16482,16483,16484,16485,16486,16487,16488,16489, &
& 16490,16491,16492,16493,16494,16495,16496,16497,16498,16499,16500, &
& 16501]
 index_cni(12001:12100)=[&
& 16502,16503,16504,16505,16506,16507,16508,16509,16510,16511,16512, &
& 16513,16514,16515,16516,16517,16518,16519,16520,16521,16522,16523, &
& 16524,16525,16526,16527,16528,16529,16530,16531,16532,16533,16534, &
& 16535,16536,16537,16538,16539,16540,16541,16542,16543,16544,16545, &
& 16546,16547,16548,16549,16550,16551,16552,16553,16554,16555,16556, &
& 16557,16558,16559,16560,16561,16562,16563,16564,16565,16566,16567, &
& 16568,16569,16570,16571,16572,16573,16574,16575,16576,16577,16578, &
& 16579,16580,16581,16582,16583,16584,16585,16586,16587,16588,16589, &
& 16590,16591,16592,16593,16594,16595,16596,16597,16598,16599,16600, &
& 16601]
 index_cni(12101:12200)=[&
& 16602,16603,16604,16605,16606,16607,16608,16609,16610,16611,16612, &
& 16613,16614,16615,16616,16617,16618,16619,16620,16621,16622,16623, &
& 16624,16625,16626,16627,16628,16629,16630,16631,16632,16633,16634, &
& 16635,16636,16637,16638,16639,16640,16641,16642,16643,16644,16645, &
& 16646,16647,16648,16649,16650,16651,16652,16653,16654,16655,16656, &
& 16657,16658,16659,16660,16661,16662,16663,16664,16665,16666,16667, &
& 16668,16669,16670,16671,16672,16673,16674,16675,16676,16677,16678, &
& 16679,16680,16681,16682,16683,16684,16685,16686,16687,16688,16689, &
& 16690,16691,16692,16693,16694,16695,16696,16697,16698,16699,16700, &
& 16701]
 index_cni(12201:12300)=[&
& 16702,16703,16704,16705,16706,16707,16708,16709,16710,16711,16712, &
& 16713,16714,16715,16716,16717,16718,16719,16720,16721,16722,16723, &
& 16724,16725,16726,16727,16728,16729,16730,16731,16732,16733,16734, &
& 16735,16736,16737,16738,16739,16740,16741,16742,16743,16744,16745, &
& 16746,16747,16748,16749,16750,16751,16752,16753,16754,16755,16756, &
& 16757,16758,16759,16760,16761,16762,16763,16764,16765,16766,16767, &
& 16768,16769,16770,16771,16772,16773,16774,16775,16776,16777,16778, &
& 16779,16780,16781,16782,16783,16784,16785,16786,16787,16788,16789, &
& 16790,16791,16792,16793,16794,16795,16796,16797,16798,16799,16800, &
& 16801]
 index_cni(12301:12400)=[&
& 16802,16803,16804,16805,16806,16807,16808,16809,16810,16811,16812, &
& 16813,16814,16815,16816,16817,16818,16819,16820,16821,16822,16823, &
& 16824,16825,16826,16827,16828,16829,16830,16831,16832,16833,16834, &
& 16835,16836,16837,16838,16839,16840,16841,16842,16843,16844,16845, &
& 16846,16847,16848,16849,16850,16851,16852,16853,16854,16855,16856, &
& 16857,16858,16859,16860,16861,16862,16863,16864,16865,16866,16867, &
& 16868,16869,16870,16871,16872,16873,16874,16875,16876,16877,16878, &
& 16879,16880,16881,16882,16883,16884,16885,16886,16887,16888,16889, &
& 16890,16891,16892,16893,16894,16895,16896,16897,16898,16899,16900, &
& 16901]
 index_cni(12401:12500)=[&
& 16902,16903,16904,16905,16906,16907,16908,16909,16910,16911,16912, &
& 16913,16914,16915,16916,16917,16918,16919,16920,16921,16922,16923, &
& 16924,16925,16926,16927,16928,16929,16930,16931,16932,16933,16934, &
& 16935,16936,16937,16938,16939,16940,16941,16942,16943,16944,16945, &
& 16946,16947,16948,16949,16950,16951,16952,16953,16954,16955,16956, &
& 16957,16958,16959,16960,16961,16962,16963,16964,16965,16966,16967, &
& 16968,16969,16970,16971,16972,16973,16974,16975,16976,16977,16978, &
& 16979,16980,16981,16982,16983,16984,16985,16986,16987,16988,16989, &
& 16990,16991,16992,16993,16994,16995,16996,16997,16998,16999,17000, &
& 17001]
 index_cni(12501:12600)=[&
& 17002,17003,17004,17005,17006,17007,17008,17009,17010,17011,17012, &
& 17013,17014,17015,17016,17017,17018,17019,17020,17021,17022,17023, &
& 17024,17025,17026,17027,17028,17029,17030,17031,17032,17033,17034, &
& 17035,17036,17037,17038,17039,17040,17041,17042,17043,17044,17045, &
& 17046,17047,17048,17049,17050,17051,17052,17053,17054,17055,17056, &
& 17057,17058,17059,17060,17061,17062,17063,17064,17065,17066,17067, &
& 17068,17069,17070,17071,17072,17073,17074,17075,17076,17077,17078, &
& 17079,17080,17081,17082,17083,17084,17085,17086,17087,17088,17089, &
& 17090,17091,17092,17093,17094,17095,17096,17097,17098,17099,17100, &
& 17101]
 index_cni(12601:12700)=[&
& 17102,17103,17104,17105,17106,17107,17108,17109,17110,17111,17112, &
& 17113,17114,17115,17116,17117,17118,17119,17120,17121,17122,17123, &
& 17124,17125,17126,17127,17128,17129,17130,17131,17132,17133,17134, &
& 17135,17136,17137,17138,17139,17140,17141,17142,17143,17144,17145, &
& 17146,17147,17148,17149,17150,17151,17152,17153,17154,17155,17156, &
& 17157,17158,17159,17160,17161,17162,17163,17164,17165,17166,17167, &
& 17168,17169,17170,17171,17172,17173,17174,17175,17176,17177,17178, &
& 17179,17180,17181,17182,17183,17184,17185,17186,17187,17188,17189, &
& 17190,17191,17192,17193,17194,17195,17196,17197,17198,17199,17200, &
& 17201]
 index_cni(12701:12800)=[&
& 17202,17203,17204,17205,17206,17207,17208,17209,17210,17211,17212, &
& 17213,17214,17215,17216,17217,17218,17219,17220,17221,17222,17223, &
& 17224,17225,17226,17227,17228,17229,17230,17231,17232,17233,17234, &
& 17235,17236,17237,17238,17239,17240,17241,17242,17243,17244,17245, &
& 17246,17247,17248,17249,17250,17251,17252,17253,17254,17255,17256, &
& 17257,17258,17259,17260,17261,17262,17263,17264,17265,17266,17267, &
& 17268,17269,17270,17271,17272,17273,17274,17275,17276,17277,17278, &
& 17279,17280,17281,17282,17283,17284,17285,17286,17287,17288,17289, &
& 17290,17291,17292,17293,17294,17295,17296,17297,17298,17299,17300, &
& 17301]
 index_cni(12801:12900)=[&
& 17302,17303,17304,17305,17306,17307,17308,17309,17310,17311,17312, &
& 17313,17314,17315,17316,17317,17318,17319,17320,17321,17322,17323, &
& 17324,17325,17326,17327,17328,17329,17330,17331,17332,17333,17334, &
& 17335,17336,17337,17338,17339,17340,17341,17342,17343,17344,17345, &
& 17346,17347,17348,17349,17350,17351,17352,17353,17354,17355,17356, &
& 17357,17358,17359,17360,17361,17362,17363,17364,17365,17366,17367, &
& 17368,17369,17370,17371,17372,17373,17374,17375,17376,17377,17378, &
& 17379,17380,17381,17382,17383,17384,17385,17386,17387,17388,17389, &
& 17390,17391,17392,17393,17394,17395,17396,17397,17398,17399,17400, &
& 17401]
 index_cni(12901:13000)=[&
& 17402,17403,17404,17405,17406,17407,17408,17409,17410,17411,17412, &
& 17413,17414,17415,17416,17417,17418,17419,17420,17421,17422,17423, &
& 17424,17425,17426,17427,17428,17429,17430,17431,17432,17433,17434, &
& 17435,17436,17437,17438,17439,17440,17441,17442,17443,17444,17445, &
& 17446,17447,17448,17449,17450,17451,17452,17453,17454,17455,17456, &
& 17457,17458,17459,17460,17461,17462,17463,17464,17465,17466,17467, &
& 17468,17469,17470,17471,17472,17473,17474,17475,17476,17477,17478, &
& 17479,17480,17481,17482,17483,17484,17485,17486,17487,17488,17489, &
& 17490,17491,17492,17493,17494,17495,17496,17497,17498,17499,17500, &
& 17501]
 index_cni(13001:13100)=[&
& 17502,17503,17504,17505,17506,17507,17508,17509,17510,17511,17512, &
& 17513,17514,17515,17516,17517,17518,17519,17520,17521,17522,17523, &
& 17524,17525,17526,17527,17528,17529,17530,17531,17532,17533,17534, &
& 17535,17536,17537,17538,17539,17540,17541,17542,17543,17544,17545, &
& 17546,17547,17548,17549,17550,17551,17552,17553,17554,17555,17556, &
& 17557,17558,17559,17560,17561,17562,17563,17564,17565,17566,17567, &
& 17568,17569,17570,17571,17572,17573,17574,17575,17576,17577,17578, &
& 17579,17580,17581,17582,17583,17584,17585,17586,17587,17588,17589, &
& 17590,17591,17592,17593,17594,17595,17596,17597,17598,17599,17600, &
& 17601]
 index_cni(13101:13200)=[&
& 17602,17603,17604,17605,17606,17607,17608,17609,17610,17611,17612, &
& 17613,17614,17615,17616,17617,17618,17619,17620,17621,17622,17623, &
& 17624,17625,17626,17627,17628,17629,17630,17631,17632,17633,17634, &
& 17635,17636,17637,17638,17639,17640,17641,17642,17643,17644,17645, &
& 17646,17647,17648,17649,17650,17651,17652,17653,17654,17655,17656, &
& 17657,17658,17659,17660,17661,17662,17663,17664,17665,17666,17667, &
& 17668,17669,17670,17671,17672,17673,17674,17675,17676,17677,17678, &
& 17679,17680,17681,17682,17683,17684,17685,17686,17687,17688,17689, &
& 17690,17691,17692,17693,17694,17695,17696,17697,17698,17699,17700, &
& 17701]
 index_cni(13201:13300)=[&
& 17702,17703,17704,17705,17706,17707,17708,17709,17710,17711,17712, &
& 17713,17714,17715,17716,17717,17718,17719,17720,17721,17722,17723, &
& 17724,17725,17726,17727,17728,17729,17730,17731,17732,17733,17734, &
& 17735,17736,17737,17738,17739,17740,17741,17742,17743,17744,17745, &
& 17746,17747,17748,17749,17750,17751,17752,17753,17754,17755,17756, &
& 17757,17758,17759,17760,17761,17762,17763,17764,17765,17766,17767, &
& 17768,17769,17770,17771,17772,17773,17774,17775,17776,17777,17778, &
& 17779,17780,17781,17782,17783,17784,17785,17786,17787,17788,17789, &
& 17790,17791,17792,17793,17794,17795,17796,17797,17798,17799,17800, &
& 17801]
 index_cni(13301:13400)=[&
& 17802,17803,17804,17805,17806,17807,17808,17809,17810,17811,17812, &
& 17813,17814,17815,17816,17817,17818,17819,17820,17821,17822,17823, &
& 17824,17825,17826,17827,17828,17829,17830,17831,17832,17833,17834, &
& 17835,17836,17837,17838,17839,17840,17841,17842,17843,17844,17845, &
& 17846,17847,17848,17849,17850,17851,17852,17853,17854,17855,17856, &
& 17857,17858,17859,17860,17861,17862,17863,17864,17865,17866,17867, &
& 17868,17869,17870,17871,17872,17873,17874,17875,17876,17877,17878, &
& 17879,17880,17881,17882,17883,17884,17885,17886,17887,17888,17889, &
& 17890,17891,17892,17893,17894,17895,17896,17897,17898,17899,17900, &
& 17901]
 index_cni(13401:13500)=[&
& 17902,17903,17904,17905,17906,17907,17908,17909,17910,17911,17912, &
& 17913,17914,17915,17916,17917,17918,17919,17920,17921,17922,17923, &
& 17924,17925,17926,17927,17928,17929,17930,17931,17932,17933,17934, &
& 17935,17936,17937,17938,17939,17940,17941,17942,17943,17944,17945, &
& 17946,17947,17948,17949,17950,17951,17952,17953,17954,17955,17956, &
& 17957,17958,17959,17960,17961,17962,17963,17964,17965,17966,17967, &
& 17968,17969,17970,17971,17972,17973,17974,17975,17976,17977,17978, &
& 17979,17980,17981,17982,17983,17984,17985,17986,17987,17988,17989, &
& 17990,17991,17992,17993,17994,17995,17996,17997,17998,17999,18000, &
& 18001]
 index_cni(13501:13600)=[&
& 18002,18003,18004,18005,18006,18007,18008,18009,18010,18011,18012, &
& 18013,18014,18015,18016,18017,18018,18019,18020,18021,18022,18023, &
& 18024,18025,18026,18027,18028,18029,18030,18031,18032,18033,18034, &
& 18035,18036,18037,18038,18039,18040,18041,18042,18043,18044,18045, &
& 18046,18047,18048,18049,18050,18051,18052,18053,18054,18055,18056, &
& 18057,18058,18059,18060,18061,18062,18063,18064,18065,18066,18067, &
& 18068,18069,18070,18071,18072,18073,18074,18075,18076,18077,18078, &
& 18079,18080,18081,18082,18083,18084,18085,18086,18087,18088,18089, &
& 18090,18091,18092,18093,18094,18095,18096,18097,18098,18099,18100, &
& 18101]
 index_cni(13601:13700)=[&
& 18102,18103,18104,18105,18106,18107,18108,18109,18110,18111,18112, &
& 18113,18114,18115,18116,18117,18118,18119,18120,18121,18122,18123, &
& 18124,18125,18126,18127,18128,18129,18130,18131,18132,18133,18134, &
& 18135,18136,18137,18138,18139,18140,18141,18142,18143,18144,18145, &
& 18146,18147,18148,18149,18150,18151,18152,18153,18154,18155,18156, &
& 18157,18158,18159,18160,18161,18162,18163,18164,18165,18166,18167, &
& 18168,18169,18170,18171,18172,18173,18174,18175,18176,18177,18178, &
& 18179,18180,18181,18182,18183,18184,18185,18186,18187,18188,18189, &
& 18190,18191,18192,18193,18194,18195,18196,18197,18198,18199,18200, &
& 18201]
 index_cni(13701:13800)=[&
& 18202,18203,18204,18205,18206,18207,18208,18209,18210,18211,18212, &
& 18213,18214,18215,18216,18217,18218,18219,18220,18221,18222,18223, &
& 18224,18225,18226,18227,18228,18229,18230,18231,18232,18233,18234, &
& 18235,18236,18237,18238,18239,18240,18241,18242,18243,18244,18245, &
& 18246,18247,18248,18249,18250,18251,18252,18253,18254,18255,18256, &
& 18257,18258,18259,18260,18261,18262,18263,18264,18265,18266,18267, &
& 18268,18269,18270,18271,18272,18273,18274,18275,18276,18277,18278, &
& 18279,18280,18281,18282,18283,18284,18285,18286,18287,18288,18289, &
& 18290,18291,18292,18293,18294,18295,18296,18297,18298,18299,18300, &
& 18301]
 index_cni(13801:13900)=[&
& 18302,18303,18304,18305,18306,18307,18308,18309,18310,18311,18312, &
& 18313,18314,18315,18316,18317,18318,18319,18320,18321,18322,18323, &
& 18324,18325,18326,18327,18328,18329,18330,18331,18332,18333,18334, &
& 18335,18336,18337,18338,18339,18340,18341,18342,18343,18344,18345, &
& 18346,18347,18348,18349,18350,18351,18352,18353,18354,18355,18356, &
& 18357,18358,18359,18360,18361,18362,18363,18364,18365,18366,18367, &
& 18368,18369,18370,18371,18372,18373,18374,18375,18376,18377,18378, &
& 18379,18380,18381,18382,18383,18384,18385,18386,18387,18388,18389, &
& 18390,18391,18392,18393,18394,18395,18396,18397,18398,18399,18400, &
& 18401]
 index_cni(13901:14000)=[&
& 18402,18403,18404,18405,18406,18407,18408,18409,18410,18411,18412, &
& 18413,18414,18415,18416,18417,18418,18419,18420,18421,18422,18423, &
& 18424,18425,18426,18427,18428,18429,18430,18431,18432,18433,18434, &
& 18435,18436,18437,18438,18439,18440,18441,18442,18443,18444,18445, &
& 18446,18447,18448,18449,18450,18451,18452,18453,18454,18455,18456, &
& 18457,18458,18459,18460,18461,18462,18463,18464,18465,18466,18467, &
& 18468,18469,18470,18471,18472,18473,18474,18475,18476,18477,18478, &
& 18479,18480,18481,18482,18483,18484,18485,18486,18487,18488,18489, &
& 18490,18491,18492,18493,18494,18495,18496,18497,18498,18499,18500, &
& 18501]
 index_cni(14001:14100)=[&
& 18502,18503,18504,18505,18506,18507,18508,18509,18510,18511,18512, &
& 18513,18514,18515,18516,18517,18518,18519,18520,18521,18522,18523, &
& 18524,18525,18526,18527,18528,18529,18530,18531,18532,18533,18534, &
& 18535,18536,18537,18538,18539,18540,18541,18542,18543,18544,18545, &
& 18546,18547,18548,18549,18550,18551,18552,18553,18554,18555,18556, &
& 18557,18558,18559,18560,18561,18562,18563,18564,18565,18566,18567, &
& 18568,18569,18570,18571,18572,18573,18574,18575,18576,18577,18578, &
& 18579,18580,18581,18582,18583,18584,18585,18586,18587,18588,18589, &
& 18590,18591,18592,18593,18594,18595,18596,18597,18598,18599,18600, &
& 18601]
 index_cni(14101:14200)=[&
& 18602,18603,18604,18605,18606,18607,18608,18609,18610,18611,18612, &
& 18613,18614,18615,18616,18617,18618,18619,18620,18621,18622,18623, &
& 18624,18625,18626,18627,18628,18629,18630,18631,18632,18633,18634, &
& 18635,18636,18637,18638,18639,18640,18641,18642,18643,18644,18645, &
& 18646,18647,18648,18649,18650,18651,18652,18653,18654,18655,18656, &
& 18657,18658,18659,18660,18661,18662,18663,18664,18665,18666,18667, &
& 18668,18669,18670,18671,18672,18673,18674,18675,18676,18677,18678, &
& 18679,18680,18681,18682,18683,18684,18685,18686,18687,18688,18689, &
& 18690,18691,18692,18693,18694,18695,18696,18697,18698,18699,18700, &
& 18701]
 index_cni(14201:14300)=[&
& 18702,18703,18704,18705,18706,18707,18708,18709,18710,18711,18712, &
& 18713,18714,18715,18716,18717,18718,18719,18720,18721,18722,18723, &
& 18724,18725,18726,18727,18728,18729,18730,18731,18732,18733,18734, &
& 18735,18736,18737,18738,18739,18740,18741,18742,18743,18744,18745, &
& 18746,18747,18748,18749,18750,18751,18752,18753,18754,18755,18756, &
& 18757,18758,18759,18760,18761,18762,18763,18764,18765,18766,18767, &
& 18768,18769,18770,18771,18772,18773,18774,18775,18776,18777,18778, &
& 18779,18780,18781,18782,18783,18784,18785,18786,18787,18788,18789, &
& 18790,18791,18792,18793,18794,18795,18796,18797,18798,18799,18800, &
& 18801]
 index_cni(14301:14400)=[&
& 18802,18803,18804,18805,18806,18807,18808,18809,18810,18811,18812, &
& 18813,18814,18815,18816,18817,18818,18819,18820,18821,18822,18823, &
& 18824,18825,18826,18827,18828,18829,18830,18831,18832,18833,18834, &
& 18835,18836,18837,18838,18839,18840,18841,18842,18843,18844,18845, &
& 18846,18847,18848,18849,18850,18851,18852,18853,18854,18855,18856, &
& 18857,18858,18859,18860,18861,18862,18863,18864,18865,18866,18867, &
& 18868,18869,18870,18871,18872,18873,18874,18875,18876,18877,18878, &
& 18879,18880,18881,18882,18883,18884,18885,18886,18887,18888,18889, &
& 18890,18891,18892,18893,18894,18895,18896,18897,18898,18899,18900, &
& 18901]
 index_cni(14401:14500)=[&
& 18902,18903,18904,18905,18906,18907,18908,18909,18910,18911,18912, &
& 18913,18914,18915,18916,18917,18918,18919,18920,18921,18922,18923, &
& 18924,18925,18926,18927,18928,18929,18930,18931,18932,18933,18934, &
& 18935,18936,18937,18938,18939,18940,18941,18942,18943,18944,18945, &
& 18946,18947,18948,18949,18950,18951,18952,18953,18954,18955,18956, &
& 18957,18958,18959,18960,18961,18962,18963,18964,18965,18966,18967, &
& 18968,18969,18970,18971,18972,18973,18974,18975,18976,18977,18978, &
& 18979,18980,18981,18982,18983,18984,18985,18986,18987,18988,18989, &
& 18990,18991,18992,18993,18994,18995,18996,18997,18998,18999,19000, &
& 19001]
 index_cni(14501:14600)=[&
& 19002,19003,19004,19005,19006,19007,19008,19009,19010,19011,19012, &
& 19013,19014,19015,19016,19017,19018,19019,19020,19021,19022,19023, &
& 19024,19025,19026,19027,19028,19029,19030,19031,19032,19033,19034, &
& 19035,19036,19037,19038,19039,19040,19041,19042,19043,19044,19045, &
& 19046,19047,19048,19049,19050,19051,19052,19053,19054,19055,19056, &
& 19057,19058,19059,19060,19061,19062,19063,19064,19065,19066,19067, &
& 19068,19069,19070,19071,19072,19073,19074,19075,19076,19077,19078, &
& 19079,19080,19081,19082,19083,19084,19085,19086,19087,19088,19089, &
& 19090,19091,19092,19093,19094,19095,19096,19097,19098,19099,19100, &
& 19101]
 index_cni(14601:14700)=[&
& 19102,19103,19104,19105,19106,19107,19108,19109,19110,19111,19112, &
& 19113,19114,19115,19116,19117,19118,19119,19120,19121,19122,19123, &
& 19124,19125,19126,19127,19128,19129,19130,19131,19132,19133,19134, &
& 19135,19136,19137,19138,19139,19140,19141,19142,19143,19144,19145, &
& 19146,19147,19148,19149,19150,19151,19152,19153,19154,19155,19156, &
& 19157,19158,19159,19160,19161,19162,19163,19164,19165,19166,19167, &
& 19168,19169,19170,19171,19172,19173,19174,19175,19176,19177,19178, &
& 19179,19180,19181,19182,19183,19184,19185,19186,19187,19188,19189, &
& 19190,19191,19192,19193,19194,19195,19196,19197,19198,19199,19200, &
& 19201]
 index_cni(14701:14800)=[&
& 19202,19203,19204,19205,19206,19207,19208,19209,19210,19211,19212, &
& 19213,19214,19215,19216,19217,19218,19219,19220,19221,19222,19223, &
& 19224,19225,19226,19227,19228,19229,19230,19231,19232,19233,19234, &
& 19235,19236,19237,19238,19239,19240,19241,19242,19243,19244,19245, &
& 19246,19247,19248,19249,19250,19251,19252,19253,19254,19255,19256, &
& 19257,19258,19259,19260,19261,19262,19263,19264,19265,19266,19267, &
& 19268,19269,19270,19271,19272,19273,19274,19275,19276,19277,19278, &
& 19279,19280,19281,19282,19283,19284,19285,19286,19287,19288,19289, &
& 19290,19291,19292,19293,19294,19295,19296,19297,19298,19299,19300, &
& 19301]
 index_cni(14801:14900)=[&
& 19302,19303,19304,19305,19306,19307,19308,19309,19310,19311,19312, &
& 19313,19314,19315,19316,19317,19318,19319,19320,19321,19322,19323, &
& 19324,19325,19326,19327,19328,19329,19330,19331,19332,19333,19334, &
& 19335,19336,19337,19338,19339,19340,19341,19342,19343,19344,19345, &
& 19346,19347,19348,19349,19350,19351,19352,19353,19354,19355,19356, &
& 19357,19358,19359,19360,19361,19362,19363,19364,19365,19366,19367, &
& 19368,19369,19370,19371,19372,19373,19374,19375,19376,19377,19378, &
& 19379,19380,19381,19382,19383,19384,19385,19386,19387,19388,19389, &
& 19390,19391,19392,19393,19394,19395,19396,19397,19398,19399,19400, &
& 19401]
 index_cni(14901:15000)=[&
& 19402,19403,19404,19405,19406,19407,19408,19409,19410,19411,19412, &
& 19413,19414,19415,19416,19417,19418,19419,19420,19421,19422,19423, &
& 19424,19425,19426,19427,19428,19429,19430,19431,19432,19433,19434, &
& 19435,19436,19437,19438,19439,19440,19441,19442,19443,19444,19445, &
& 19446,19447,19448,19449,19450,19451,19452,19453,19454,19455,19456, &
& 19457,19458,19459,19460,19461,19462,19463,19464,19465,19466,19467, &
& 19468,19469,19470,19471,19472,19473,19474,19475,19476,19477,19478, &
& 19479,19480,19481,19482,19483,19484,19485,19486,19487,19488,19489, &
& 19490,19491,19492,19493,19494,19495,19496,19497,19498,19499,19500, &
& 19501]
 index_cni(15001:15100)=[&
& 19502,19503,19504,19505,19506,19507,19508,19509,19510,19511,19512, &
& 19513,19514,19515,19516,19517,19518,19519,19520,19521,19522,19523, &
& 19524,19525,19526,19527,19528,19529,19530,19531,19532,19533,19534, &
& 19535,19536,19537,19538,19539,19540,19541,19542,19543,19544,19545, &
& 19546,19547,19548,19549,19550,19551,19552,19553,19554,19555,19556, &
& 19557,19558,19559,19560,19561,19562,19563,19564,19565,19566,19567, &
& 19568,19569,19570,19571,19572,19573,19574,19575,19576,19577,19578, &
& 19579,19580,19581,19582,19583,19584,19585,19586,19587,19588,19589, &
& 19590,19591,19592,19593,19594,19595,19596,19597,19598,19599,19600, &
& 19601]
 index_cni(15101:15200)=[&
& 19602,19603,19604,19605,19606,19607,19608,19609,19610,19611,19612, &
& 19613,19614,19615,19616,19617,19618,19619,19620,19621,19622,19623, &
& 19624,19625,19626,19627,19628,19629,19630,19631,19632,19633,19634, &
& 19635,19636,19637,19638,19639,19640,19641,19642,19643,19644,19645, &
& 19646,19647,19648,19649,19650,19651,19652,19653,19654,19655,19656, &
& 19657,19658,19659,19660,19661,19662,19663,19664,19665,19666,19667, &
& 19668,19669,19670,19671,19672,19673,19674,19675,19676,19677,19678, &
& 19679,19680,19681,19682,19683,19684,19685,19686,19687,19688,19689, &
& 19690,19691,19692,19693,19694,19695,19696,19697,19698,19699,19700, &
& 19701]
 index_cni(15201:15300)=[&
& 19702,19703,19704,19705,19706,19707,19708,19709,19710,19711,19712, &
& 19713,19714,19715,19716,19717,19718,19719,19720,19721,19722,19723, &
& 19724,19725,19726,19727,19728,19729,19730,19731,19732,19733,19734, &
& 19735,19736,19737,19738,19739,19740,19741,19742,19743,19744,19745, &
& 19746,19747,19748,19749,19750,19751,19752,19753,19754,19755,19756, &
& 19757,19758,19759,19760,19761,19762,19763,19764,19765,19766,19767, &
& 19768,19769,19770,19771,19772,19773,19774,19775,19776,19777,19778, &
& 19779,19780,19781,19782,19783,19784,19785,19786,19787,19788,19789, &
& 19790,19791,19792,19793,19794,19795,19796,19797,19798,19799,19800, &
& 19801]
 index_cni(15301:15400)=[&
& 19802,19803,19804,19805,19806,19807,19808,19809,19810,19811,19812, &
& 19813,19814,19815,19816,19817,19818,19819,19820,19821,19822,19823, &
& 19824,19825,19826,19827,19828,19829,19830,19831,19832,19833,19834, &
& 19835,19836,19837,19838,19839,19840,19841,19842,19843,19844,19845, &
& 19846,19847,19848,19849,19850,19851,19852,19853,19854,19855,19856, &
& 19857,19858,19859,19860,19861,19862,19863,19864,19865,19866,19867, &
& 19868,19869,19870,19871,19872,19873,19874,19875,19876,19877,19878, &
& 19879,19880,19881,19882,19883,19884,19885,19886,19887,19888,19889, &
& 19890,19891,19892,19893,19894,19895,19896,19897,19898,19899,19900, &
& 19901]
 index_cni(15401:15500)=[&
& 19902,19903,19904,19905,19906,19907,19908,19909,19910,19911,19912, &
& 19913,19914,19915,19916,19917,19918,19919,19920,19921,19922,19923, &
& 19924,19925,19926,19927,19928,19929,19930,19931,19932,19933,19934, &
& 19935,19936,19937,19938,19939,19940,19941,19942,19943,19944,19945, &
& 19946,19947,19948,19949,19950,19951,19952,19953,19954,19955,19956, &
& 19957,19958,19959,19960,19961,19962,19963,19964,19965,19966,19967, &
& 19968,19969,19970,19971,19972,19973,19974,19975,19976,19977,19978, &
& 19979,19980,19981,19982,19983,19984,19985,19986,19987,19988,19989, &
& 19990,19991,19992,19993,19994,19995,19996,19997,19998,19999,20000, &
& 20001]
 index_cni(15501:15600)=[&
& 20002,20003,20004,20005,20006,20007,20008,20009,20010,20011,20012, &
& 20013,20014,20015,20016,20017,20018,20019,20020,20021,20022,20023, &
& 20024,20025,20026,20027,20028,20029,20030,20031,20032,20033,20034, &
& 20035,20036,20037,20038,20039,20040,20041,20042,20043,20044,20045, &
& 20046,20047,20048,20049,20050,20051,20052,20053,20054,20055,20056, &
& 20057,20058,20059,20060,20061,20062,20063,20064,20065,20066,20067, &
& 20068,20069,20070,20071,20072,20073,20074,20075,20076,20077,20078, &
& 20079,20080,20081,20082,20083,20084,20085,20086,20087,20088,20089, &
& 20090,20091,20092,20093,20094,20095,20096,20097,20098,20099,20100, &
& 20101]
 index_cni(15601:15700)=[&
& 20102,20103,20104,20105,20106,20107,20108,20109,20110,20111,20112, &
& 20113,20114,20115,20116,20117,20118,20119,20120,20121,20122,20123, &
& 20124,20125,20126,20127,20128,20129,20130,20131,20132,20133,20134, &
& 20135,20136,20137,20138,20139,20140,20141,20142,20143,20144,20145, &
& 20146,20147,20148,20149,20150,20151,20152,20153,20154,20155,20156, &
& 20157,20158,20159,20160,20161,20162,20163,20164,20165,20166,20167, &
& 20168,20169,20170,20171,20172,20173,20174,20175,20176,20177,20178, &
& 20179,20180,20181,20182,20183,20184,20185,20186,20187,20188,20189, &
& 20190,20191,20192,20193,20194,20195,20196,20197,20198,20199,20200, &
& 20201]
 index_cni(15701:15800)=[&
& 20202,20203,20204,20205,20206,20207,20208,20209,20210,20211,20212, &
& 20213,20214,20215,20216,20217,20218,20219,20220,20221,20222,20223, &
& 20224,20225,20226,20227,20228,20229,20230,20231,20232,20233,20234, &
& 20235,20236,20237,20238,20239,20240,20241,20242,20243,20244,20245, &
& 20246,20247,20248,20249,20250,20251,20252,20253,20254,20255,20256, &
& 20257,20258,20259,20260,20261,20262,20263,20264,20265,20266,20267, &
& 20268,20269,20270,20271,20272,20273,20274,20275,20276,20277,20278, &
& 20279,20280,20281,20282,20283,20284,20285,20286,20287,20288,20289, &
& 20290,20291,20292,20293,20294,20295,20296,20297,20298,20299,20300, &
& 20301]
 index_cni(15801:15900)=[&
& 20302,20303,20304,20305,20306,20307,20308,20309,20310,20311,20312, &
& 20313,20314,20315,20316,20317,20318,20319,20320,20321,20322,20323, &
& 20324,20325,20326,20327,20328,20329,20330,20331,20332,20333,20334, &
& 20335,20336,20337,20338,20339,20340,20341,20342,20343,20344,20345, &
& 20346,20347,20348,20349,20350,20351,20352,20353,20354,20355,20356, &
& 20357,20358,20359,20360,20361,20362,20363,20364,20365,20366,20367, &
& 20368,20369,20370,20371,20372,20373,20374,20375,20376,20377,20378, &
& 20379,20380,20381,20382,20383,20384,20385,20386,20387,20388,20389, &
& 20390,20391,20392,20393,20394,20395,20396,20397,20398,20399,20400, &
& 20401]
 index_cni(15901:16000)=[&
& 20402,20403,20404,20405,20406,20407,20408,20409,20410,20411,20412, &
& 20413,20414,20415,20416,20417,20418,20419,20420,20421,20422,20423, &
& 20424,20425,20426,20427,20428,20429,20430,20431,20432,20433,20434, &
& 20435,20436,20437,20438,20439,20440,20441,20442,20443,20444,20445, &
& 20446,20447,20448,20449,20450,20451,20452,20453,20454,20455,20456, &
& 20457,20458,20459,20460,20461,20462,20463,20464,20465,20466,20467, &
& 20468,20469,20470,20471,20472,20473,20474,20475,20476,20477,20478, &
& 20479,20480,20481,20482,20483,20484,20485,20486,20487,20488,20489, &
& 20490,20491,20492,20493,20494,20495,20496,20497,20498,20499,20500, &
& 20501]
 index_cni(16001:16100)=[&
& 20502,20503,20504,20505,20506,20507,20508,20509,20510,20511,20512, &
& 20513,20514,20515,20516,20517,20518,20519,20520,20521,20522,20523, &
& 20524,20525,20526,20527,20528,20529,20530,20531,20532,20533,20534, &
& 20535,20536,20537,20538,20539,20540,20541,20542,20543,20544,20545, &
& 20546,20547,20548,20549,20550,20551,20552,20553,20554,20555,20556, &
& 20557,20558,20559,20560,20561,20562,20563,20564,20565,20566,20567, &
& 20568,20569,20570,20571,20572,20573,20574,20575,20576,20577,20578, &
& 20579,20580,20581,20582,20583,20584,20585,20586,20587,20588,20589, &
& 20590,20591,20592,20593,20594,20595,20596,20597,20598,20599,20600, &
& 20601]
 index_cni(16101:16200)=[&
& 20602,20603,20604,20605,20606,20607,20608,20609,20610,20611,20612, &
& 20613,20614,20615,20616,20617,20618,20619,20620,20621,20622,20623, &
& 20624,20625,20626,20627,20628,20629,20630,20631,20632,20633,20634, &
& 20635,20636,20637,20638,20639,20640,20641,20642,20643,20644,20645, &
& 20646,20647,20648,20649,20650,20651,20652,20653,20654,20655,20656, &
& 20657,20658,20659,20660,20661,20662,20663,20664,20665,20666,20667, &
& 20668,20669,20670,20671,20672,20673,20674,20675,20676,20677,20678, &
& 20679,20680,20681,20682,20683,20684,20685,20686,20687,20688,20689, &
& 20690,20691,20692,20693,20694,20695,20696,20697,20698,20699,20700, &
& 20701]
 index_cni(16201:16300)=[&
& 20702,20703,20704,20705,20706,20707,20708,20709,20710,20711,20712, &
& 20713,20714,20715,20716,20717,20718,20719,20720,20721,20722,20723, &
& 20724,20725,20726,20727,20728,20729,20730,20731,20732,20733,20734, &
& 20735,20736,20737,20738,20739,20740,20741,20742,20743,20744,20745, &
& 20746,20747,20748,20749,20750,20751,20752,20753,20754,20755,20756, &
& 20757,20758,20759,20760,20761,20762,20763,20764,20765,20766,20767, &
& 20768,20769,20770,20771,20772,20773,20774,20775,20776,20777,20778, &
& 20779,20780,20781,20782,20783,20784,20785,20786,20787,20788,20789, &
& 20790,20791,20792,20793,20794,20795,20796,20797,20798,20799,20800, &
& 20801]
 index_cni(16301:16400)=[&
& 20802,20803,20804,20805,20806,20807,20808,20809,20810,20811,20812, &
& 20813,20814,20815,20816,20817,20818,20819,20820,20821,20822,20823, &
& 20824,20825,20826,20827,20828,20829,20830,20831,20832,20833,20834, &
& 20835,20836,20837,20838,20839,20840,20841,20842,20843,20844,20845, &
& 20846,20847,20848,20849,20850,20851,20852,20853,20854,20855,20856, &
& 20857,20858,20859,20860,20861,20862,20863,20864,20865,20866,20867, &
& 20868,20869,20870,20871,20872,20873,20874,20875,20876,20877,20878, &
& 20879,20880,20881,20882,20883,20884,20885,20886,20887,20888,20889, &
& 20890,20891,20892,20893,20894,20895,20896,20897,20898,20899,20900, &
& 20901]
 index_cni(16401:16500)=[&
& 20902,20903,20904,20905,20906,20907,20908,20909,20910,20911,20912, &
& 20913,20914,20915,20916,20917,20918,20919,20920,20921,20922,20923, &
& 20924,20925,20926,20927,20928,20929,20930,20931,20932,20933,20934, &
& 20935,20936,20937,20938,20939,20940,20941,20942,20943,20944,20945, &
& 20946,20947,20948,20949,20950,20951,20952,20953,20954,20955,20956, &
& 20957,20958,20959,20960,20961,20962,20963,20964,20965,20966,20967, &
& 20968,20969,20970,20971,20972,20973,20974,20975,20976,20977,20978, &
& 20979,20980,20981,20982,20983,20984,20985,20986,20987,20988,20989, &
& 20990,20991,20992,20993,20994,20995,20996,20997,20998,20999,21000, &
& 21001]
 index_cni(16501:16600)=[&
& 21002,21003,21004,21005,21006,21007,21008,21009,21010,21011,21012, &
& 21013,21014,21015,21016,21017,21018,21019,21020,21021,21022,21023, &
& 21024,21025,21026,21027,21028,21029,21030,21031,21032,21033,21034, &
& 21035,21036,21037,21038,21039,21040,21041,21042,21043,21044,21045, &
& 21046,21047,21048,21049,21050,21051,21052,21053,21054,21055,21056, &
& 21057,21058,21059,21060,21061,21062,21063,21064,21065,21066,21067, &
& 21068,21069,21070,21071,21072,21073,21074,21075,21076,21077,21078, &
& 21079,21080,21081,21082,21083,21084,21085,21086,21087,21088,21089, &
& 21090,21091,21092,21093,21094,21095,21096,21097,21098,21099,21100, &
& 21101]
 index_cni(16601:16700)=[&
& 21102,21103,21104,21105,21106,21107,21108,21109,21110,21111,21112, &
& 21113,21114,21115,21116,21117,21118,21119,21120,21121,21122,21123, &
& 21124,21125,21126,21127,21128,21129,21130,21131,21132,21133,21134, &
& 21135,21136,21137,21138,21139,21140,21141,21142,21143,21144,21145, &
& 21146,21147,21148,21149,21150,21151,21152,21153,21154,21155,21156, &
& 21157,21158,21159,21160,21161,21162,21163,21164,21165,21166,21167, &
& 21168,21169,21170,21171,21172,21173,21174,21175,21176,21177,21178, &
& 21179,21180,21181,21182,21183,21184,21185,21186,21187,21188,21189, &
& 21190,21191,21192,21193,21194,21195,21196,21197,21198,21199,21200, &
& 21201]
 index_cni(16701:16800)=[&
& 21202,21203,21204,21205,21206,21207,21208,21209,21210,21211,21212, &
& 21213,21214,21215,21216,21217,21218,21219,21220,21221,21222,21223, &
& 21224,21225,21226,21227,21228,21229,21230,21231,21232,21233,21234, &
& 21235,21236,21237,21238,21239,21240,21241,21242,21243,21244,21245, &
& 21246,21247,21248,21249,21250,21251,21252,21253,21254,21255,21256, &
& 21257,21258,21259,21260,21261,21262,21263,21264,21265,21266,21267, &
& 21268,21269,21270,21271,21272,21273,21274,21275,21276,21277,21278, &
& 21279,21280,21281,21282,21283,21284,21285,21286,21287,21288,21289, &
& 21290,21291,21292,21293,21294,21295,21296,21297,21298,21299,21300, &
& 21301]
 index_cni(16801:16900)=[&
& 21302,21303,21304,21305,21306,21307,21308,21309,21310,21311,21312, &
& 21313,21314,21315,21316,21317,21318,21319,21320,21321,21322,21323, &
& 21324,21325,21326,21327,21328,21329,21330,21331,21332,21333,21334, &
& 21335,21336,21337,21338,21339,21340,21341,21342,21343,21344,21345, &
& 21346,21347,21348,21349,21350,21351,21352,21353,21354,21355,21356, &
& 21357,21358,21359,21360,21361,21362,21363,21364,21365,21366,21367, &
& 21368,21369,21370,21371,21372,21373,21374,21375,21376,21377,21378, &
& 21379,21380,21381,21382,21383,21384,21385,21386,21387,21388,21389, &
& 21390,21391,21392,21393,21394,21395,21396,21397,21398,21399,21400, &
& 21401]
 index_cni(16901:17000)=[&
& 21402,21403,21404,21405,21406,21407,21408,21409,21410,21411,21412, &
& 21413,21414,21415,21416,21417,21418,21419,21420,21421,21422,21423, &
& 21424,21425,21426,21427,21428,21429,21430,21431,21432,21433,21434, &
& 21435,21436,21437,21438,21439,21440,21441,21442,21443,21444,21445, &
& 21446,21447,21448,21449,21450,21451,21452,21453,21454,21455,21456, &
& 21457,21458,21459,21460,21461,21462,21463,21464,21465,21466,21467, &
& 21468,21469,21470,21471,21472,21473,21474,21475,21476,21477,21478, &
& 21479,21480,21481,21482,21483,21484,21485,21486,21487,21488,21489, &
& 21490,21491,21492,21493,21494,21495,21496,21497,21498,21499,21500, &
& 21501]
 index_cni(17001:17100)=[&
& 21502,21503,21504,21505,21506,21507,21508,21509,21510,21511,21512, &
& 21513,21514,21515,21516,21517,21518,21519,21520,21521,21522,21523, &
& 21524,21525,21526,21527,21528,21529,21530,21531,21532,21533,21534, &
& 21535,21536,21537,21538,21539,21540,21541,21542,21543,21544,21545, &
& 21546,21547,21548,21549,21550,21551,21552,21553,21554,21555,21556, &
& 21557,21558,21559,21560,21561,21562,21563,21564,21565,21566,21567, &
& 21568,21569,21570,21571,21572,21573,21574,21575,21576,21577,21578, &
& 21579,21580,21581,21582,21583,21584,21585,21586,21587,21588,21589, &
& 21590,21591,21592,21593,21594,21595,21596,21597,21598,21599,21600, &
& 21601]
 index_cni(17101:17200)=[&
& 21602,21603,21604,21605,21606,21607,21608,21609,21610,21611,21612, &
& 21613,21614,21615,21616,21617,21618,21619,21620,21621,21622,21623, &
& 21624,21625,21626,21627,21628,21629,21630,21631,21632,21633,21634, &
& 21635,21636,21637,21638,21639,21640,21641,21642,21643,21644,21645, &
& 21646,21647,21648,21649,21650,21651,21652,21653,21654,21655,21656, &
& 21657,21658,21659,21660,21661,21662,21663,21664,21665,21666,21667, &
& 21668,21669,21670,21671,21672,21673,21674,21675,21676,21677,21678, &
& 21679,21680,21681,21682,21683,21684,21685,21686,21687,21688,21689, &
& 21690,21691,21692,21693,21694,21695,21696,21697,21698,21699,21700, &
& 21701]
 index_cni(17201:17300)=[&
& 21702,21703,21704,21705,21706,21707,21708,21709,21710,21711,21712, &
& 21713,21714,21715,21716,21717,21718,21719,21720,21721,21722,21723, &
& 21724,21725,21726,21727,21728,21729,21730,21731,21732,21733,21734, &
& 21735,21736,21737,21738,21739,21740,21741,21742,21743,21744,21745, &
& 21746,21747,21748,21749,21750,21751,21752,21753,21754,21755,21756, &
& 21757,21758,21759,21760,21761,21762,21763,21764,21765,21766,21767, &
& 21768,21769,21770,21771,21772,21773,21774,21775,21776,21777,21778, &
& 21779,21780,21781,21782,21783,21784,21785,21786,21787,21788,21789, &
& 21790,21791,21792,21793,21794,21795,21796,21797,21798,21799,21800, &
& 21801]
 index_cni(17301:17400)=[&
& 21802,21803,21804,21805,21806,21807,21808,21809,21810,21811,21812, &
& 21813,21814,21815,21816,21817,21818,21819,21820,21821,21822,21823, &
& 21824,21825,21826,21827,21828,21829,21830,21831,21832,21833,21834, &
& 21835,21836,21837,21838,21839,21840,21841,21842,21843,21844,21845, &
& 21846,21847,21848,21849,21850,21851,21852,21853,21854,21855,21856, &
& 21857,21858,21859,21860,21861,21862,21863,21864,21865,21866,21867, &
& 21868,21869,21870,21871,21872,21873,21874,21875,21876,21877,21878, &
& 21879,21880,21881,21882,21883,21884,21885,21886,21887,21888,21889, &
& 21890,21891,21892,21893,21894,21895,21896,21897,21898,21899,21900, &
& 21901]
 index_cni(17401:17500)=[&
& 21902,21903,21904,21905,21906,21907,21908,21909,21910,21911,21912, &
& 21913,21914,21915,21916,21917,21918,21919,21920,21921,21922,21923, &
& 21924,21925,21926,21927,21928,21929,21930,21931,21932,21933,21934, &
& 21935,21936,21937,21938,21939,21940,21941,21942,21943,21944,21945, &
& 21946,21947,21948,21949,21950,21951,21952,21953,21954,21955,21956, &
& 21957,21958,21959,21960,21961,21962,21963,21964,21965,21966,21967, &
& 21968,21969,21970,21971,21972,21973,21974,21975,21976,21977,21978, &
& 21979,21980,21981,21982,21983,21984,21985,21986,21987,21988,21989, &
& 21990,21991,21992,21993,21994,21995,21996,21997,21998,21999,22000, &
& 22001]
 index_cni(17501:17600)=[&
& 22002,22003,22004,22005,22006,22007,22008,22009,22010,22011,22012, &
& 22013,22014,22015,22016,22017,22018,22019,22020,22021,22022,22023, &
& 22024,22025,22026,22027,22028,22029,22030,22031,22032,22033,22034, &
& 22035,22036,22037,22038,22039,22040,22041,22042,22043,22044,22045, &
& 22046,22047,22048,22049,22050,22051,22052,22053,22054,22055,22056, &
& 22057,22058,22059,22060,22061,22062,22063,22064,22065,22066,22067, &
& 22068,22069,22070,22071,22072,22073,22074,22075,22076,22077,22078, &
& 22079,22080,22081,22082,22083,22084,22085,22086,22087,22088,22089, &
& 22090,22091,22092,22093,22094,22095,22096,22097,22098,22099,22100, &
& 22101]
 index_cni(17601:17700)=[&
& 22102,22103,22104,22105,22106,22107,22108,22109,22110,22111,22112, &
& 22113,22114,22115,22116,22117,22118,22119,22120,22121,22122,22123, &
& 22124,22125,22126,22127,22128,22129,22130,22131,22132,22133,22134, &
& 22135,22136,22137,22138,22139,22140,22141,22142,22143,22144,22145, &
& 22146,22147,22148,22149,22150,22151,22152,22153,22154,22155,22156, &
& 22157,22158,22159,22160,22161,22162,22163,22164,22165,22166,22167, &
& 22168,22169,22170,22171,22172,22173,22174,22175,22176,22177,22178, &
& 22179,22180,22181,22182,22183,22184,22185,22186,22187,22188,22189, &
& 22190,22191,22192,22193,22194,22195,22196,22197,22198,22199,22200, &
& 22201]
 index_cni(17701:17800)=[&
& 22202,22203,22204,22205,22206,22207,22208,22209,22210,22211,22212, &
& 22213,22214,22215,22216,22217,22218,22219,22220,22221,22222,22223, &
& 22224,22225,22226,22227,22228,22229,22230,22231,22232,22233,22234, &
& 22235,22236,22237,22238,22239,22240,22241,22242,22243,22244,22245, &
& 22246,22247,22248,22249,22250,22251,22252,22253,22254,22255,22256, &
& 22257,22258,22259,22260,22261,22262,22263,22264,22265,22266,22267, &
& 22268,22269,22270,22271,22272,22273,22274,22275,22276,22277,22278, &
& 22279,22280,22281,22282,22283,22284,22285,22286,22287,22288,22289, &
& 22290,22291,22292,22293,22294,22295,22296,22297,22298,22299,22300, &
& 22301]
 index_cni(17801:17900)=[&
& 22302,22303,22304,22305,22306,22307,22308,22309,22310,22311,22312, &
& 22313,22314,22315,22316,22317,22318,22319,22320,22321,22322,22323, &
& 22324,22325,22326,22327,22328,22329,22330,22331,22332,22333,22334, &
& 22335,22336,22337,22338,22339,22340,22341,22342,22343,22344,22345, &
& 22346,22347,22348,22349,22350,22351,22352,22353,22354,22355,22356, &
& 22357,22358,22359,22360,22361,22362,22363,22364,22365,22366,22367, &
& 22368,22369,22370,22371,22372,22373,22374,22375,22376,22377,22378, &
& 22379,22380,22381,22382,22383,22384,22385,22386,22387,22388,22389, &
& 22390,22391,22392,22393,22394,22395,22396,22397,22398,22399,22400, &
& 22401]
 index_cni(17901:18000)=[&
& 22402,22403,22404,22405,22406,22407,22408,22409,22410,22411,22412, &
& 22413,22414,22415,22416,22417,22418,22419,22420,22421,22422,22423, &
& 22424,22425,22426,22427,22428,22429,22430,22431,22432,22433,22434, &
& 22435,22436,22437,22438,22439,22440,22441,22442,22443,22444,22445, &
& 22446,22447,22448,22449,22450,22451,22452,22453,22454,22455,22456, &
& 22457,22458,22459,22460,22461,22462,22463,22464,22465,22466,22467, &
& 22468,22469,22470,22471,22472,22473,22474,22475,22476,22477,22478, &
& 22479,22480,22481,22482,22483,22484,22485,22486,22487,22488,22489, &
& 22490,22491,22492,22493,22494,22495,22496,22497,22498,22499,22500, &
& 22501]
 index_cni(18001:18100)=[&
& 22502,22503,22504,22505,22506,22507,22508,22509,22510,22511,22512, &
& 22513,22514,22515,22516,22517,22518,22519,22520,22521,22522,22523, &
& 22524,22525,22526,22527,22528,22529,22530,22531,22532,22533,22534, &
& 22535,22536,22537,22538,22539,22540,22541,22542,22543,22544,22545, &
& 22546,22547,22548,22549,22550,22551,22552,22553,22554,22555,22556, &
& 22557,22558,22559,22560,22561,22562,22563,22564,22565,22566,22567, &
& 22568,22569,22570,22571,22572,22573,22574,22575,22576,22577,22578, &
& 22579,22580,22581,22582,22583,22584,22585,22586,22587,22588,22589, &
& 22590,22591,22592,22593,22594,22595,22596,22597,22598,22599,22600, &
& 22601]
 index_cni(18101:18200)=[&
& 22602,22603,22604,22605,22606,22607,22608,22609,22610,22611,22612, &
& 22613,22614,22615,22616,22617,22618,22619,22620,22621,22622,22623, &
& 22624,22625,22626,22627,22628,22629,22630,22631,22632,22633,22634, &
& 22635,22636,22637,22638,22639,22640,22641,22642,22643,22644,22645, &
& 22646,22647,22648,22649,22650,22651,22652,22653,22654,22655,22656, &
& 22657,22658,22659,22660,22661,22662,22663,22664,22665,22666,22667, &
& 22668,22669,22670,22671,22672,22673,22674,22675,22676,22677,22678, &
& 22679,22680,22681,22682,22683,22684,22685,22686,22687,22688,22689, &
& 22690,22691,22692,22693,22694,22695,22696,22697,22698,22699,22700, &
& 22701]
 index_cni(18201:18300)=[&
& 22702,22703,22704,22705,22706,22707,22708,22709,22710,22711,22712, &
& 22713,22714,22715,22716,22717,22718,22719,22720,22721,22722,22723, &
& 22724,22725,22726,22727,22728,22729,22730,22731,22732,22733,22734, &
& 22735,22736,22737,22738,22739,22740,22741,22742,22743,22744,22745, &
& 22746,22747,22748,22749,22750,22751,22752,22753,22754,22755,22756, &
& 22757,22758,22759,22760,22761,22762,22763,22764,22765,22766,22767, &
& 22768,22769,22770,22771,22772,22773,22774,22775,22776,22777,22778, &
& 22779,22780,22781,22782,22783,22784,22785,22786,22787,22788,22789, &
& 22790,22791,22792,22793,22794,22795,22796,22797,22798,22799,22800, &
& 22801]
 index_cni(18301:18400)=[&
& 22802,22803,22804,22805,22806,22807,22808,22809,22810,22811,22812, &
& 22813,22814,22815,22816,22817,22818,22819,22820,22821,22822,22823, &
& 22824,22825,22826,22827,22828,22829,22830,22831,22832,22833,22834, &
& 22835,22836,22837,22838,22839,22840,22841,22842,22843,22844,22845, &
& 22846,22847,22848,22849,22850,22851,22852,22853,22854,22855,22856, &
& 22857,22858,22859,22860,22861,22862,22863,22864,22865,22866,22867, &
& 22868,22869,22870,22871,22872,22873,22874,22875,22876,22877,22878, &
& 22879,22880,22881,22882,22883,22884,22885,22886,22887,22888,22889, &
& 22890,22891,22892,22893,22894,22895,22896,22897,22898,22899,22900, &
& 22901]
 index_cni(18401:18500)=[&
& 22902,22903,22904,22905,22906,22907,22908,22909,22910,22911,22912, &
& 22913,22914,22915,22916,22917,22918,22919,22920,22921,22922,22923, &
& 22924,22925,22926,22927,22928,22929,22930,22931,22932,22933,22934, &
& 22935,22936,22937,22938,22939,22940,22941,22942,22943,22944,22945, &
& 22946,22947,22948,22949,22950,22951,22952,22953,22954,22955,22956, &
& 22957,22958,22959,22960,22961,22962,22963,22964,22965,22966,22967, &
& 22968,22969,22970,22971,22972,22973,22974,22975,22976,22977,22978, &
& 22979,22980,22981,22982,22983,22984,22985,22986,22987,22988,22989, &
& 22990,22991,22992,22993,22994,22995,22996,22997,22998,22999,23000, &
& 23001]
 index_cni(18501:18600)=[&
& 23002,23003,23004,23005,23006,23007,23008,23009,23010,23011,23012, &
& 23013,23014,23015,23016,23017,23018,23019,23020,23021,23022,23023, &
& 23024,23025,23026,23027,23028,23029,23030,23031,23032,23033,23034, &
& 23035,23036,23037,23038,23039,23040,23041,23042,23043,23044,23045, &
& 23046,23047,23048,23049,23050,23051,23052,23053,23054,23055,23056, &
& 23057,23058,23059,23060,23061,23062,23063,23064,23065,23066,23067, &
& 23068,23069,23070,23071,23072,23073,23074,23075,23076,23077,23078, &
& 23079,23080,23081,23082,23083,23084,23085,23086,23087,23088,23089, &
& 23090,23091,23092,23093,23094,23095,23096,23097,23098,23099,23100, &
& 23101]
 index_cni(18601:18700)=[&
& 23102,23103,23104,23105,23106,23107,23108,23109,23110,23111,23112, &
& 23113,23114,23115,23116,23117,23118,23119,23120,23121,23122,23123, &
& 23124,23125,23126,23127,23128,23129,23130,23131,23132,23133,23134, &
& 23135,23136,23137,23138,23139,23140,23141,23142,23143,23144,23145, &
& 23146,23147,23148,23149,23150,23151,23152,23153,23154,23155,23156, &
& 23157,23158,23159,23160,23161,23162,23163,23164,23165,23166,23167, &
& 23168,23169,23170,23171,23172,23173,23174,23175,23176,23177,23178, &
& 23179,23180,23181,23182,23183,23184,23185,23186,23187,23188,23189, &
& 23190,23191,23192,23193,23194,23195,23196,23197,23198,23199,23200, &
& 23201]
 index_cni(18701:18800)=[&
& 23202,23203,23204,23205,23206,23207,23208,23209,23210,23211,23212, &
& 23213,23214,23215,23216,23217,23218,23219,23220,23221,23222,23223, &
& 23224,23225,23226,23227,23228,23229,23230,23231,23232,23233,23234, &
& 23235,23236,23237,23238,23239,23240,23241,23242,23243,23244,23245, &
& 23246,23247,23248,23249,23250,23251,23252,23253,23254,23255,23256, &
& 23257,23258,23259,23260,23261,23262,23263,23264,23265,23266,23267, &
& 23268,23269,23270,23271,23272,23273,23274,23275,23276,23277,23278, &
& 23279,23280,23281,23282,23283,23284,23285,23286,23287,23288,23289, &
& 23290,23291,23292,23293,23294,23295,23296,23297,23298,23299,23300, &
& 23301]
 index_cni(18801:18900)=[&
& 23302,23303,23304,23305,23306,23307,23308,23309,23310,23311,23312, &
& 23313,23314,23315,23316,23317,23318,23319,23320,23321,23322,23323, &
& 23324,23325,23326,23327,23328,23329,23330,23331,23332,23333,23334, &
& 23335,23336,23337,23338,23339,23340,23341,23342,23343,23344,23345, &
& 23346,23347,23348,23349,23350,23351,23352,23353,23354,23355,23356, &
& 23357,23358,23359,23360,23361,23362,23363,23364,23365,23366,23367, &
& 23368,23369,23370,23371,23372,23373,23374,23375,23376,23377,23378, &
& 23379,23380,23381,23382,23383,23384,23385,23386,23387,23388,23389, &
& 23390,23391,23392,23393,23394,23395,23396,23397,23398,23399,23400, &
& 23401]
 index_cni(18901:19000)=[&
& 23402,23403,23404,23405,23406,23407,23408,23409,23410,23411,23412, &
& 23413,23414,23415,23416,23417,23418,23419,23420,23421,23422,23423, &
& 23424,23425,23426,23427,23428,23429,23430,23431,23432,23433,23434, &
& 23435,23436,23437,23438,23439,23440,23441,23442,23443,23444,23445, &
& 23446,23447,23448,23449,23450,23451,23452,23453,23454,23455,23456, &
& 23457,23458,23459,23460,23461,23462,23463,23464,23465,23466,23467, &
& 23468,23469,23470,23471,23472,23473,23474,23475,23476,23477,23478, &
& 23479,23480,23481,23482,23483,23484,23485,23486,23487,23488,23489, &
& 23490,23491,23492,23493,23494,23495,23496,23497,23498,23499,23500, &
& 23501]
 index_cni(19001:19100)=[&
& 23502,23503,23504,23505,23506,23507,23508,23509,23510,23511,23512, &
& 23513,23514,23515,23516,23517,23518,23519,23520,23521,23522,23523, &
& 23524,23525,23526,23527,23528,23529,23530,23531,23532,23533,23534, &
& 23535,23536,23537,23538,23539,23540,23541,23542,23543,23544,23545, &
& 23546,23547,23548,23549,23550,23551,23552,23553,23554,23555,23556, &
& 23557,23558,23559,23560,23561,23562,23563,23564,23565,23566,23567, &
& 23568,23569,23570,23571,23572,23573,23574,23575,23576,23577,23578, &
& 23579,23580,23581,23582,23583,23584,23585,23586,23587,23588,23589, &
& 23590,23591,23592,23593,23594,23595,23596,23597,23598,23599,23600, &
& 23601]
 index_cni(19101:19200)=[&
& 23602,23603,23604,23605,23606,23607,23608,23609,23610,23611,23612, &
& 23613,23614,23615,23616,23617,23618,23619,23620,23621,23622,23623, &
& 23624,23625,23626,23627,23628,23629,23630,23631,23632,23633,23634, &
& 23635,23636,23637,23638,23639,23640,23641,23642,23643,23644,23645, &
& 23646,23647,23648,23649,23650,23651,23652,23653,23654,23655,23656, &
& 23657,23658,23659,23660,23661,23662,23663,23664,23665,23666,23667, &
& 23668,23669,23670,23671,23672,23673,23674,23675,23676,23677,23678, &
& 23679,23680,23681,23682,23683,23684,23685,23686,23687,23688,23689, &
& 23690,23691,23692,23693,23694,23695,23696,23697,23698,23699,23700, &
& 23701]
 index_cni(19201:19300)=[&
& 23702,23703,23704,23705,23706,23707,23708,23709,23710,23711,23712, &
& 23713,23714,23715,23716,23717,23718,23719,23720,23721,23722,23723, &
& 23724,23725,23726,23727,23728,23729,23730,23731,23732,23733,23734, &
& 23735,23736,23737,23738,23739,23740,23741,23742,23743,23744,23745, &
& 23746,23747,23748,23749,23750,23751,23752,23753,23754,23755,23756, &
& 23757,23758,23759,23760,23761,23762,23763,23764,23765,23766,23767, &
& 23768,23769,23770,23771,23772,23773,23774,23775,23776,23777,23778, &
& 23779,23780,23781,23782,23783,23784,23785,23786,23787,23788,23789, &
& 23790,23791,23792,23793,23794,23795,23796,23797,23798,23799,23800, &
& 23801]
 index_cni(19301:19400)=[&
& 23802,23803,23804,23805,23806,23807,23808,23809,23810,23811,23812, &
& 23813,23814,23815,23816,23817,23818,23819,23820,23821,23822,23823, &
& 23824,23825,23826,23827,23828,23829,23830,23831,23832,23833,23834, &
& 23835,23836,23837,23838,23839,23840,23841,23842,23843,23844,23845, &
& 23846,23847,23848,23849,23850,23851,23852,23853,23854,23855,23856, &
& 23857,23858,23859,23860,23861,23862,23863,23864,23865,23866,23867, &
& 23868,23869,23870,23871,23872,23873,23874,23875,23876,23877,23878, &
& 23879,23880,23881,23882,23883,23884,23885,23886,23887,23888,23889, &
& 23890,23891,23892,23893,23894,23895,23896,23897,23898,23899,23900, &
& 23901]
 index_cni(19401:19500)=[&
& 23902,23903,23904,23905,23906,23907,23908,23909,23910,23911,23912, &
& 23913,23914,23915,23916,23917,23918,23919,23920,23921,23922,23923, &
& 23924,23925,23926,23927,23928,23929,23930,23931,23932,23933,23934, &
& 23935,23936,23937,23938,23939,23940,23941,23942,23943,23944,23945, &
& 23946,23947,23948,23949,23950,23951,23952,23953,23954,23955,23956, &
& 23957,23958,23959,23960,23961,23962,23963,23964,23965,23966,23967, &
& 23968,23969,23970,23971,23972,23973,23974,23975,23976,23977,23978, &
& 23979,23980,23981,23982,23983,23984,23985,23986,23987,23988,23989, &
& 23990,23991,23992,23993,23994,23995,23996,23997,23998,23999,24000, &
& 24001]
 index_cni(19501:19600)=[&
& 24002,24003,24004,24005,24006,24007,24008,24009,24010,24011,24012, &
& 24013,24014,24015,24016,24017,24018,24019,24020,24021,24022,24023, &
& 24024,24025,24026,24027,24028,24029,24030,24031,24032,24033,24034, &
& 24035,24036,24037,24038,24039,24040,24041,24042,24043,24044,24045, &
& 24046,24047,24048,24049,24050,24051,24052,24053,24054,24055,24056, &
& 24057,24058,24059,24060,24061,24062,24063,24064,24065,24066,24067, &
& 24068,24069,24070,24071,24072,24073,24074,24075,24076,24077,24078, &
& 24079,24080,24081,24082,24083,24084,24085,24086,24087,24088,24089, &
& 24090,24091,24092,24093,24094,24095,24096,24097,24098,24099,24100, &
& 24101]
 index_cni(19601:19700)=[&
& 24102,24103,24104,24105,24106,24107,24108,24109,24110,24111,24112, &
& 24113,24114,24115,24116,24117,24118,24119,24120,24121,24122,24123, &
& 24124,24125,24126,24127,24128,24129,24130,24131,24132,24133,24134, &
& 24135,24136,24137,24138,24139,24140,24141,24142,24143,24144,24145, &
& 24146,24147,24148,24149,24150,24151,24152,24153,24154,24155,24156, &
& 24157,24158,24159,24160,24161,24162,24163,24164,24165,24166,24167, &
& 24168,24169,24170,24171,24172,24173,24174,24175,24176,24177,24178, &
& 24179,24180,24181,24182,24183,24184,24185,24186,24187,24188,24189, &
& 24190,24191,24192,24193,24194,24195,24196,24197,24198,24199,24200, &
& 24201]
 index_cni(19701:19800)=[&
& 24202,24203,24204,24205,24206,24207,24208,24209,24210,24211,24212, &
& 24213,24214,24215,24216,24217,24218,24219,24220,24221,24222,24223, &
& 24224,24225,24226,24227,24228,24229,24230,24231,24232,24233,24234, &
& 24235,24236,24237,24238,24239,24240,24241,24242,24243,24244,24245, &
& 24246,24247,24248,24249,24250,24251,24252,24253,24254,24255,24256, &
& 24257,24258,24259,24260,24261,24262,24263,24264,24265,24266,24267, &
& 24268,24269,24270,24271,24272,24273,24274,24275,24276,24277,24278, &
& 24279,24280,24281,24282,24283,24284,24285,24286,24287,24288,24289, &
& 24290,24291,24292,24293,24294,24295,24296,24297,24298,24299,24300, &
& 24301]
 index_cni(19801:19900)=[&
& 24302,24303,24304,24305,24306,24307,24308,24309,24310,24311,24312, &
& 24313,24314,24315,24316,24317,24318,24319,24320,24321,24322,24323, &
& 24324,24325,24326,24327,24328,24329,24330,24331,24332,24333,24334, &
& 24335,24336,24337,24338,24339,24340,24341,24342,24343,24344,24345, &
& 24346,24347,24348,24349,24350,24351,24352,24353,24354,24355,24356, &
& 24357,24358,24359,24360,24361,24362,24363,24364,24365,24366,24367, &
& 24368,24369,24370,24371,24372,24373,24374,24375,24376,24377,24378, &
& 24379,24380,24381,24382,24383,24384,24385,24386,24387,24388,24389, &
& 24390,24391,24392,24393,24394,24395,24396,24397,24398,24399,24400, &
& 24401]
 index_cni(19901:20000)=[&
& 24402,24403,24404,24405,24406,24407,24408,24409,24410,24411,24412, &
& 24413,24414,24415,24416,24417,24418,24419,24420,24421,24422,24423, &
& 24424,24425,24426,24427,24428,24429,24430,24431,24432,24433,24434, &
& 24435,24436,24437,24438,24439,24440,24441,24442,24443,24444,24445, &
& 24446,24447,24448,24449,24450,24451,24452,24453,24454,24455,24456, &
& 24457,24458,24459,24460,24461,24462,24463,24464,24465,24466,24467, &
& 24468,24469,24470,24471,24472,24473,24474,24475,24476,24477,24478, &
& 24479,24480,24481,24482,24483,24484,24485,24486,24487,24488,24489, &
& 24490,24491,24492,24493,24494,24495,24496,24497,24498,24499,24500, &
& 24501]
 index_cni(20001:20100)=[&
& 24502,24503,24504,24505,24506,24507,24508,24509,24510,24511,24512, &
& 24513,24514,24515,24516,24517,24518,24519,24520,24521,24522,24523, &
& 24524,24525,24526,24527,24528,24529,24530,24531,24532,24533,24534, &
& 24535,24536,24537,24538,24539,24540,24541,24542,24543,24544,24545, &
& 24546,24547,24548,24549,24550,24551,24552,24553,24554,24555,24556, &
& 24557,24558,24559,24560,24561,24562,24563,24564,24565,24566,24567, &
& 24568,24569,24570,24571,24572,24573,24574,24575,24576,24577,24578, &
& 24579,24580,24581,24582,24583,24584,24585,24586,24587,24588,24589, &
& 24590,24591,24592,24593,24594,24595,24596,24597,24598,24599,24600, &
& 24601]
 index_cni(20101:20200)=[&
& 24602,24603,24604,24605,24606,24607,24608,24609,24610,24611,24612, &
& 24613,24614,24615,24616,24617,24618,24619,24620,24621,24622,24623, &
& 24624,24625,24626,24627,24628,24629,24630,24631,24632,24633,24634, &
& 24635,24636,24637,24638,24639,24640,24641,24642,24643,24644,24645, &
& 24646,24647,24648,24649,24650,24651,24652,24653,24654,24655,24656, &
& 24657,24658,24659,24660,24661,24662,24663,24664,24665,24666,24667, &
& 24668,24669,24670,24671,24672,24673,24674,24675,24676,24677,24678, &
& 24679,24680,24681,24682,24683,24684,24685,24686,24687,24688,24689, &
& 24690,24691,24692,24693,24694,24695,24696,24697,24698,24699,24700, &
& 24701]
 index_cni(20201:20300)=[&
& 24702,24703,24704,24705,24706,24707,24708,24709,24710,24711,24712, &
& 24713,24714,24715,24716,24717,24718,24719,24720,24721,24722,24723, &
& 24724,24725,24726,24727,24728,24729,24730,24731,24732,24733,24734, &
& 24735,24736,24737,24738,24739,24740,24741,24742,24743,24744,24745, &
& 24746,24747,24748,24749,24750,24751,24752,24753,24754,24755,24756, &
& 24757,24758,24759,24760,24761,24762,24763,24764,24765,24766,24767, &
& 24768,24769,24770,24771,24772,24773,24774,24775,24776,24777,24778, &
& 24779,24780,24781,24782,24783,24784,24785,24786,24787,24788,24789, &
& 24790,24791,24792,24793,24794,24795,24796,24797,24798,24799,24800, &
& 24801]
 index_cni(20301:20400)=[&
& 24802,24803,24804,24805,24806,24807,24808,24809,24810,24811,24812, &
& 24813,24814,24815,24816,24817,24818,24819,24820,24821,24822,24823, &
& 24824,24825,24826,24827,24828,24829,24830,24831,24832,24833,24834, &
& 24835,24836,24837,24838,24839,24840,24841,24842,24843,24844,24845, &
& 24846,24847,24848,24849,24850,24851,24852,24853,24854,24855,24856, &
& 24857,24858,24859,24860,24861,24862,24863,24864,24865,24866,24867, &
& 24868,24869,24870,24871,24872,24873,24874,24875,24876,24877,24878, &
& 24879,24880,24881,24882,24883,24884,24885,24886,24887,24888,24889, &
& 24890,24891,24892,24893,24894,24895,24896,24897,24898,24899,24900, &
& 24901]
 index_cni(20401:20500)=[&
& 24902,24903,24904,24905,24906,24907,24908,24909,24910,24911,24912, &
& 24913,24914,24915,24916,24917,24918,24919,24920,24921,24922,24923, &
& 24924,24925,24926,24927,24928,24929,24930,24931,24932,24933,24934, &
& 24935,24936,24937,24938,24939,24940,24941,24942,24943,24944,24945, &
& 24946,24947,24948,24949,24950,24951,24952,24953,24954,24955,24956, &
& 24957,24958,24959,24960,24961,24962,24963,24964,24965,24966,24967, &
& 24968,24969,24970,24971,24972,24973,24974,24975,24976,24977,24978, &
& 24979,24980,24981,24982,24983,24984,24985,24986,24987,24988,24989, &
& 24990,24991,24992,24993,24994,24995,24996,24997,24998,24999,25000, &
& 25001]
 index_cni(20501:20600)=[&
& 25002,25003,25004,25005,25006,25007,25008,25009,25010,25011,25012, &
& 25013,25014,25015,25016,25017,25018,25019,25020,25021,25022,25023, &
& 25024,25025,25026,25027,25028,25029,25030,25031,25032,25033,25034, &
& 25035,25036,25037,25038,25039,25040,25041,25042,25043,25044,25045, &
& 25046,25047,25048,25049,25050,25051,25052,25053,25054,25055,25056, &
& 25057,25058,25059,25060,25061,25062,25063,25064,25065,25066,25067, &
& 25068,25069,25070,25071,25072,25073,25074,25075,25076,25077,25078, &
& 25079,25080,25081,25082,25083,25084,25085,25086,25087,25088,25089, &
& 25090,25091,25092,25093,25094,25095,25096,25097,25098,25099,25100, &
& 25101]
 index_cni(20601:20700)=[&
& 25102,25103,25104,25105,25106,25107,25108,25109,25110,25111,25112, &
& 25113,25114,25115,25116,25117,25118,25119,25120,25121,25122,25123, &
& 25124,25125,25126,25127,25128,25129,25130,25131,25132,25133,25134, &
& 25135,25136,25137,25138,25139,25140,25141,25142,25143,25144,25145, &
& 25146,25147,25148,25149,25150,25151,25152,25153,25154,25155,25156, &
& 25157,25158,25159,25160,25161,25162,25163,25164,25165,25166,25167, &
& 25168,25169,25170,25171,25172,25173,25174,25175,25176,25177,25178, &
& 25179,25180,25181,25182,25183,25184,25185,25186,25187,25188,25189, &
& 25190,25191,25192,25193,25194,25195,25196,25197,25198,25199,25200, &
& 25201]
 index_cni(20701:20800)=[&
& 25202,25203,25204,25205,25206,25207,25208,25209,25210,25211,25212, &
& 25213,25214,25215,25216,25217,25218,25219,25220,25221,25222,25223, &
& 25224,25225,25226,25227,25228,25229,25230,25231,25232,25233,25234, &
& 25235,25236,25237,25238,25239,25240,25241,25242,25243,25244,25245, &
& 25246,25247,25248,25249,25250,25251,25252,25253,25254,25255,25256, &
& 25257,25258,25259,25260,25261,25262,25263,25264,25265,25266,25267, &
& 25268,25269,25270,25271,25272,25273,25274,25275,25276,25277,25278, &
& 25279,25280,25281,25282,25283,25284,25285,25286,25287,25288,25289, &
& 25290,25291,25292,25293,25294,25295,25296,25297,25298,25299,25300, &
& 25301]
 index_cni(20801:20900)=[&
& 25302,25303,25304,25305,25306,25307,25308,25309,25310,25311,25312, &
& 25313,25314,25315,25316,25317,25318,25319,25320,25321,25322,25323, &
& 25324,25325,25326,25327,25328,25329,25330,25331,25332,25333,25334, &
& 25335,25336,25337,25338,25339,25340,25341,25342,25343,25344,25345, &
& 25346,25347,25348,25349,25350,25351,25352,25353,25354,25355,25356, &
& 25357,25358,25359,25360,25361,25362,25363,25364,25365,25366,25367, &
& 25368,25369,25370,25371,25372,25373,25374,25375,25376,25377,25378, &
& 25379,25380,25381,25382,25383,25384,25385,25386,25387,25388,25389, &
& 25390,25391,25392,25393,25394,25395,25396,25397,25398,25399,25400, &
& 25401]
 index_cni(20901:21000)=[&
& 25402,25403,25404,25405,25406,25407,25408,25409,25410,25411,25412, &
& 25413,25414,25415,25416,25417,25418,25419,25420,25421,25422,25423, &
& 25424,25425,25426,25427,25428,25429,25430,25431,25432,25433,25434, &
& 25435,25436,25437,25438,25439,25440,25441,25442,25443,25444,25445, &
& 25446,25447,25448,25449,25450,25451,25452,25453,25454,25455,25456, &
& 25457,25458,25459,25460,25461,25462,25463,25464,25465,25466,25467, &
& 25468,25469,25470,25471,25472,25473,25474,25475,25476,25477,25478, &
& 25479,25480,25481,25482,25483,25484,25485,25486,25487,25488,25489, &
& 25490,25491,25492,25493,25494,25495,25496,25497,25498,25499,25500, &
& 25501]
 index_cni(21001:21100)=[&
& 25502,25503,25504,25505,25506,25507,25508,25509,25510,25511,25512, &
& 25513,25514,25515,25516,25517,25518,25519,25520,25521,25522,25523, &
& 25524,25525,25526,25527,25528,25529,25530,25531,25532,25533,25534, &
& 25535,25536,25537,25538,25539,25540,25541,25542,25543,25544,25545, &
& 25546,25547,25548,25549,25550,25551,25552,25553,25554,25555,25556, &
& 25557,25558,25559,25560,25561,25562,25563,25564,25565,25566,25567, &
& 25568,25569,25570,25571,25572,25573,25574,25575,25576,25577,25578, &
& 25579,25580,25581,25582,25583,25584,25585,25586,25587,25588,25589, &
& 25590,25591,25592,25593,25594,25595,25596,25597,25598,25599,25600, &
& 25601]
 index_cni(21101:21200)=[&
& 25602,25603,25604,25605,25606,25607,25608,25609,25610,25611,25612, &
& 25613,25614,25615,25616,25617,25618,25619,25620,25621,25622,25623, &
& 25624,25625,25626,25627,25628,25629,25630,25631,25632,25633,25634, &
& 25635,25636,25637,25638,25639,25640,25641,25642,25643,25644,25645, &
& 25646,25647,25648,25649,25650,25651,25652,25653,25654,25655,25656, &
& 25657,25658,25659,25660,25661,25662,25663,25664,25665,25666,25667, &
& 25668,25669,25670,25671,25672,25673,25674,25675,25676,25677,25678, &
& 25679,25680,25681,25682,25683,25684,25685,25686,25687,25688,25689, &
& 25690,25691,25692,25693,25694,25695,25696,25697,25698,25699,25700, &
& 25701]
 index_cni(21201:21300)=[&
& 25702,25703,25704,25705,25706,25707,25708,25709,25710,25711,25712, &
& 25713,25714,25715,25716,25717,25718,25719,25720,25721,25722,25723, &
& 25724,25725,25726,25727,25728,25729,25730,25731,25732,25733,25734, &
& 25735,25736,25737,25738,25739,25740,25741,25742,25743,25744,25745, &
& 25746,25747,25748,25749,25750,25751,25752,25753,25754,25755,25756, &
& 25757,25758,25759,25760,25761,25762,25763,25764,25765,25766,25767, &
& 25768,25769,25770,25771,25772,25773,25774,25775,25776,25777,25778, &
& 25779,25780,25781,25782,25783,25784,25785,25786,25787,25788,25789, &
& 25790,25791,25792,25793,25794,25795,25796,25797,25798,25799,25800, &
& 25801]
 index_cni(21301:21400)=[&
& 25802,25803,25804,25805,25806,25807,25808,25809,25810,25811,25812, &
& 25813,25814,25815,25816,25817,25818,25819,25820,25821,25822,25823, &
& 25824,25825,25826,25827,25828,25829,25830,25831,25832,25833,25834, &
& 25835,25836,25837,25838,25839,25840,25841,25842,25843,25844,25845, &
& 25846,25847,25848,25849,25850,25851,25852,25853,25854,25855,25856, &
& 25857,25858,25859,25860,25861,25862,25863,25864,25865,25866,25867, &
& 25868,25869,25870,25871,25872,25873,25874,25875,25876,25877,25878, &
& 25879,25880,25881,25882,25883,25884,25885,25886,25887,25888,25889, &
& 25890,25891,25892,25893,25894,25895,25896,25897,25898,25899,25900, &
& 25901]
 index_cni(21401:21500)=[&
& 25902,25903,25904,25905,25906,25907,25908,25909,25910,25911,25912, &
& 25913,25914,25915,25916,25917,25918,25919,25920,25921,25922,25923, &
& 25924,25925,25926,25927,25928,25929,25930,25931,25932,25933,25934, &
& 25935,25936,25937,25938,25939,25940,25941,25942,25943,25944,25945, &
& 25946,25947,25948,25949,25950,25951,25952,25953,25954,25955,25956, &
& 25957,25958,25959,25960,25961,25962,25963,25964,25965,25966,25967, &
& 25968,25969,25970,25971,25972,25973,25974,25975,25976,25977,25978, &
& 25979,25980,25981,25982,25983,25984,25985,25986,25987,25988,25989, &
& 25990,25991,25992,25993,25994,25995,25996,25997,25998,25999,26000, &
& 26001]
 index_cni(21501:21600)=[&
& 26002,26003,26004,26005,26006,26007,26008,26009,26010,26011,26012, &
& 26013,26014,26015,26016,26017,26018,26019,26020,26021,26022,26023, &
& 26024,26025,26026,26027,26028,26029,26030,26031,26032,26033,26034, &
& 26035,26036,26037,26038,26039,26040,26041,26042,26043,26044,26045, &
& 26046,26047,26048,26049,26050,26051,26052,26053,26054,26055,26056, &
& 26057,26058,26059,26060,26061,26062,26063,26064,26065,26066,26067, &
& 26068,26069,26070,26071,26072,26073,26074,26075,26076,26077,26078, &
& 26079,26080,26081,26082,26083,26084,26085,26086,26087,26088,26089, &
& 26090,26091,26092,26093,26094,26095,26096,26097,26098,26099,26100, &
& 26101]
 index_cni(21601:21700)=[&
& 26102,26103,26104,26105,26106,26107,26108,26109,26110,26111,26112, &
& 26113,26114,26115,26116,26117,26118,26119,26120,26121,26122,26123, &
& 26124,26125,26126,26127,26128,26129,26130,26131,26132,26133,26134, &
& 26135,26136,26137,26138,26139,26140,26141,26142,26143,26144,26145, &
& 26146,26147,26148,26149,26150,26151,26152,26153,26154,26155,26156, &
& 26157,26158,26159,26160,26161,26162,26163,26164,26165,26166,26167, &
& 26168,26169,26170,26171,26172,26173,26174,26175,26176,26177,26178, &
& 26179,26180,26181,26182,26183,26184,26185,26186,26187,26188,26189, &
& 26190,26191,26192,26193,26194,26195,26196,26197,26198,26199,26200, &
& 26201]
 index_cni(21701:21800)=[&
& 26202,26203,26204,26205,26206,26207,26208,26209,26210,26211,26212, &
& 26213,26214,26215,26216,26217,26218,26219,26220,26221,26222,26223, &
& 26224,26225,26226,26227,26228,26229,26230,26231,26232,26233,26234, &
& 26235,26236,26237,26238,26239,26240,26241,26242,26243,26244,26245, &
& 26246,26247,26248,26249,26250,26251,26252,26253,26254,26255,26256, &
& 26257,26258,26259,26260,26261,26262,26263,26264,26265,26266,26267, &
& 26268,26269,26270,26271,26272,26273,26274,26275,26276,26277,26278, &
& 26279,26280,26281,26282,26283,26284,26285,26286,26287,26288,26289, &
& 26290,26291,26292,26293,26294,26295,26296,26297,26298,26299,26300, &
& 26301]
 index_cni(21801:21900)=[&
& 26302,26303,26304,26305,26306,26307,26308,26309,26310,26311,26312, &
& 26313,26314,26315,26316,26317,26318,26319,26320,26321,26322,26323, &
& 26324,26325,26326,26327,26328,26329,26330,26331,26332,26333,26334, &
& 26335,26336,26337,26338,26339,26340,26341,26342,26343,26344,26345, &
& 26346,26347,26348,26349,26350,26351,26352,26353,26354,26355,26356, &
& 26357,26358,26359,26360,26361,26362,26363,26364,26365,26366,26367, &
& 26368,26369,26370,26371,26372,26373,26374,26375,26376,26377,26378, &
& 26379,26380,26381,26382,26383,26384,26385,26386,26387,26388,26389, &
& 26390,26391,26392,26393,26394,26395,26396,26397,26398,26399,26400, &
& 26401]
 index_cni(21901:22000)=[&
& 26402,26403,26404,26405,26406,26407,26408,26409,26410,26411,26412, &
& 26413,26414,26415,26416,26417,26418,26419,26420,26421,26422,26423, &
& 26424,26425,26426,26427,26428,26429,26430,26431,26432,26433,26434, &
& 26435,26436,26437,26438,26439,26440,26441,26442,26443,26444,26445, &
& 26446,26447,26448,26449,26450,26451,26452,26453,26454,26455,26456, &
& 26457,26458,26459,26460,26461,26462,26463,26464,26465,26466,26467, &
& 26468,26469,26470,26471,26472,26473,26474,26475,26476,26477,26478, &
& 26479,26480,26481,26482,26483,26484,26485,26486,26487,26488,26489, &
& 26490,26491,26492,26493,26494,26495,26496,26497,26498,26499,26500, &
& 26501]
 index_cni(22001:22100)=[&
& 26502,26503,26504,26505,26506,26507,26508,26509,26510,26511,26512, &
& 26513,26514,26515,26516,26517,26518,26519,26520,26521,26522,26523, &
& 26524,26525,26526,26527,26528,26529,26530,26531,26532,26533,26534, &
& 26535,26536,26537,26538,26539,26540,26541,26542,26543,26544,26545, &
& 26546,26547,26548,26549,26550,26551,26552,26553,26554,26555,26556, &
& 26557,26558,26559,26560,26561,26562,26563,26564,26565,26566,26567, &
& 26568,26569,26570,26571,26572,26573,26574,26575,26576,26577,26578, &
& 26579,26580,26581,26582,26583,26584,26585,26586,26587,26588,26589, &
& 26590,26591,26592,26593,26594,26595,26596,26597,26598,26599,26600, &
& 26601]
 index_cni(22101:22200)=[&
& 26602,26603,26604,26605,26606,26607,26608,26609,26610,26611,26612, &
& 26613,26614,26615,26616,26617,26618,26619,26620,26621,26622,26623, &
& 26624,26625,26626,26627,26628,26629,26630,26631,26632,26633,26634, &
& 26635,26636,26637,26638,26639,26640,26641,26642,26643,26644,26645, &
& 26646,26647,26648,26649,26650,26651,26652,26653,26654,26655,26656, &
& 26657,26658,26659,26660,26661,26662,26663,26664,26665,26666,26667, &
& 26668,26669,26670,26671,26672,26673,26674,26675,26676,26677,26678, &
& 26679,26680,26681,26682,26683,26684,26685,26686,26687,26688,26689, &
& 26690,26691,26692,26693,26694,26695,26696,26697,26698,26699,26700, &
& 26701]
 index_cni(22201:22300)=[&
& 26702,26703,26704,26705,26706,26707,26708,26709,26710,26711,26712, &
& 26713,26714,26715,26716,26717,26718,26719,26720,26721,26722,26723, &
& 26724,26725,26726,26727,26728,26729,26730,26731,26732,26733,26734, &
& 26735,26736,26737,26738,26739,26740,26741,26742,26743,26744,26745, &
& 26746,26747,26748,26749,26750,26751,26752,26753,26754,26755,26756, &
& 26757,26758,26759,26760,26761,26762,26763,26764,26765,26766,26767, &
& 26768,26769,26770,26771,26772,26773,26774,26775,26776,26777,26778, &
& 26779,26780,26781,26782,26783,26784,26785,26786,26787,26788,26789, &
& 26790,26791,26792,26793,26794,26795,26796,26797,26798,26799,26800, &
& 26801]
 index_cni(22301:22400)=[&
& 26802,26803,26804,26805,26806,26807,26808,26809,26810,26811,26812, &
& 26813,26814,26815,26816,26817,26818,26819,26820,26821,26822,26823, &
& 26824,26825,26826,26827,26828,26829,26830,26831,26832,26833,26834, &
& 26835,26836,26837,26838,26839,26840,26841,26842,26843,26844,26845, &
& 26846,26847,26848,26849,26850,26851,26852,26853,26854,26855,26856, &
& 26857,26858,26859,26860,26861,26862,26863,26864,26865,26866,26867, &
& 26868,26869,26870,26871,26872,26873,26874,26875,26876,26877,26878, &
& 26879,26880,26881,26882,26883,26884,26885,26886,26887,26888,26889, &
& 26890,26891,26892,26893,26894,26895,26896,26897,26898,26899,26900, &
& 26901]
 index_cni(22401:22500)=[&
& 26902,26903,26904,26905,26906,26907,26908,26909,26910,26911,26912, &
& 26913,26914,26915,26916,26917,26918,26919,26920,26921,26922,26923, &
& 26924,26925,26926,26927,26928,26929,26930,26931,26932,26933,26934, &
& 26935,26936,26937,26938,26939,26940,26941,26942,26943,26944,26945, &
& 26946,26947,26948,26949,26950,26951,26952,26953,26954,26955,26956, &
& 26957,26958,26959,26960,26961,26962,26963,26964,26965,26966,26967, &
& 26968,26969,26970,26971,26972,26973,26974,26975,26976,26977,26978, &
& 26979,26980,26981,26982,26983,26984,26985,26986,26987,26988,26989, &
& 26990,26991,26992,26993,26994,26995,26996,26997,26998,26999,27000, &
& 27001]
 index_cni(22501:22600)=[&
& 27002,27003,27004,27005,27006,27007,27008,27009,27010,27011,27012, &
& 27013,27014,27015,27016,27017,27018,27019,27020,27021,27022,27023, &
& 27024,27025,27026,27027,27028,27029,27030,27031,27032,27033,27034, &
& 27035,27036,27037,27038,27039,27040,27041,27042,27043,27044,27045, &
& 27046,27047,27048,27049,27050,27051,27052,27053,27054,27055,27056, &
& 27057,27058,27059,27060,27061,27062,27063,27064,27065,27066,27067, &
& 27068,27069,27070,27071,27072,27073,27074,27075,27076,27077,27078, &
& 27079,27080,27081,27082,27083,27084,27085,27086,27087,27088,27089, &
& 27090,27091,27092,27093,27094,27095,27096,27097,27098,27099,27100, &
& 27101]
 index_cni(22601:22700)=[&
& 27102,27103,27104,27105,27106,27107,27108,27109,27110,27111,27112, &
& 27113,27114,27115,27116,27117,27118,27119,27120,27121,27122,27123, &
& 27124,27125,27126,27127,27128,27129,27130,27131,27132,27133,27134, &
& 27135,27136,27137,27138,27139,27140,27141,27142,27143,27144,27145, &
& 27146,27147,27148,27149,27150,27151,27152,27153,27154,27155,27156, &
& 27157,27158,27159,27160,27161,27162,27163,27164,27165,27166,27167, &
& 27168,27169,27170,27171,27172,27173,27174,27175,27176,27177,27178, &
& 27179,27180,27181,27182,27183,27184,27185,27186,27187,27188,27189, &
& 27190,27191,27192,27193,27194,27195,27196,27197,27198,27199,27200, &
& 27201]
 index_cni(22701:22800)=[&
& 27202,27203,27204,27205,27206,27207,27208,27209,27210,27211,27212, &
& 27213,27214,27215,27216,27217,27218,27219,27220,27221,27222,27223, &
& 27224,27225,27226,27227,27228,27229,27230,27231,27232,27233,27234, &
& 27235,27236,27237,27238,27239,27240,27241,27242,27243,27244,27245, &
& 27246,27247,27248,27249,27250,27251,27252,27253,27254,27255,27256, &
& 27257,27258,27259,27260,27261,27262,27263,27264,27265,27266,27267, &
& 27268,27269,27270,27271,27272,27273,27274,27275,27276,27277,27278, &
& 27279,27280,27281,27282,27283,27284,27285,27286,27287,27288,27289, &
& 27290,27291,27292,27293,27294,27295,27296,27297,27298,27299,27300, &
& 27301]
 index_cni(22801:22900)=[&
& 27302,27303,27304,27305,27306,27307,27308,27309,27310,27311,27312, &
& 27313,27314,27315,27316,27317,27318,27319,27320,27321,27322,27323, &
& 27324,27325,27326,27327,27328,27329,27330,27331,27332,27333,27334, &
& 27335,27336,27337,27338,27339,27340,27341,27342,27343,27344,27345, &
& 27346,27347,27348,27349,27350,27351,27352,27353,27354,27355,27356, &
& 27357,27358,27359,27360,27361,27362,27363,27364,27365,27366,27367, &
& 27368,27369,27370,27371,27372,27373,27374,27375,27376,27377,27378, &
& 27379,27380,27381,27382,27383,27384,27385,27386,27387,27388,27389, &
& 27390,27391,27392,27393,27394,27395,27396,27397,27398,27399,27400, &
& 27401]
 index_cni(22901:23000)=[&
& 27402,27403,27404,27405,27406,27407,27408,27409,27410,27411,27412, &
& 27413,27414,27415,27416,27417,27418,27419,27420,27421,27422,27423, &
& 27424,27425,27426,27427,27428,27429,27430,27431,27432,27433,27434, &
& 27435,27436,27437,27438,27439,27440,27441,27442,27443,27444,27445, &
& 27446,27447,27448,27449,27450,27451,27452,27453,27454,27455,27456, &
& 27457,27458,27459,27460,27461,27462,27463,27464,27465,27466,27467, &
& 27468,27469,27470,27471,27472,27473,27474,27475,27476,27477,27478, &
& 27479,27480,27481,27482,27483,27484,27485,27486,27487,27488,27489, &
& 27490,27491,27492,27493,27494,27495,27496,27497,27498,27499,27500, &
& 27501]
 index_cni(23001:23100)=[&
& 27502,27503,27504,27505,27506,27507,27508,27509,27510,27511,27512, &
& 27513,27514,27515,27516,27517,27518,27519,27520,27521,27522,27523, &
& 27524,27525,27526,27527,27528,27529,27530,27531,27532,27533,27534, &
& 27535,27536,27537,27538,27539,27540,27541,27542,27543,27544,27545, &
& 27546,27547,27548,27549,27550,27551,27552,27553,27554,27555,27556, &
& 27557,27558,27559,27560,27561,27562,27563,27564,27565,27566,27567, &
& 27568,27569,27570,27571,27572,27573,27574,27575,27576,27577,27578, &
& 27579,27580,27581,27582,27583,27584,27585,27586,27587,27588,27589, &
& 27590,27591,27592,27593,27594,27595,27596,27597,27598,27599,27600, &
& 27601]
 index_cni(23101:23200)=[&
& 27602,27603,27604,27605,27606,27607,27608,27609,27610,27611,27612, &
& 27613,27614,27615,27616,27617,27618,27619,27620,27621,27622,27623, &
& 27624,27625,27626,27627,27628,27629,27630,27631,27632,27633,27634, &
& 27635,27636,27637,27638,27639,27640,27641,27642,27643,27644,27645, &
& 27646,27647,27648,27649,27650,27651,27652,27653,27654,27655,27656, &
& 27657,27658,27659,27660,27661,27662,27663,27664,27665,27666,27667, &
& 27668,27669,27670,27671,27672,27673,27674,27675,27676,27677,27678, &
& 27679,27680,27681,27682,27683,27684,27685,27686,27687,27688,27689, &
& 27690,27691,27692,27693,27694,27695,27696,27697,27698,27699,27700, &
& 27701]
 index_cni(23201:23300)=[&
& 27702,27703,27704,27705,27706,27707,27708,27709,27710,27711,27712, &
& 27713,27714,27715,27716,27717,27718,27719,27720,27721,27722,27723, &
& 27724,27725,27726,27727,27728,27729,27730,27731,27732,27733,27734, &
& 27735,27736,27737,27738,27739,27740,27741,27742,27743,27744,27745, &
& 27746,27747,27748,27749,27750,27751,27752,27753,27754,27755,27756, &
& 27757,27758,27759,27760,27761,27762,27763,27764,27765,27766,27767, &
& 27768,27769,27770,27771,27772,27773,27774,27775,27776,27777,27778, &
& 27779,27780,27781,27782,27783,27784,27785,27786,27787,27788,27789, &
& 27790,27791,27792,27793,27794,27795,27796,27797,27798,27799,27800, &
& 27801]
 index_cni(23301:23400)=[&
& 27802,27803,27804,27805,27806,27807,27808,27809,27810,27811,27812, &
& 27813,27814,27815,27816,27817,27818,27819,27820,27821,27822,27823, &
& 27824,27825,27826,27827,27828,27829,27830,27831,27832,27833,27834, &
& 27835,27836,27837,27838,27839,27840,27841,27842,27843,27844,27845, &
& 27846,27847,27848,27849,27850,27851,27852,27853,27854,27855,27856, &
& 27857,27858,27859,27860,27861,27862,27863,27864,27865,27866,27867, &
& 27868,27869,27870,27871,27872,27873,27874,27875,27876,27877,27878, &
& 27879,27880,27881,27882,27883,27884,27885,27886,27887,27888,27889, &
& 27890,27891,27892,27893,27894,27895,27896,27897,27898,27899,27900, &
& 27901]
 index_cni(23401:23500)=[&
& 27902,27903,27904,27905,27906,27907,27908,27909,27910,27911,27912, &
& 27913,27914,27915,27916,27917,27918,27919,27920,27921,27922,27923, &
& 27924,27925,27926,27927,27928,27929,27930,27931,27932,27933,27934, &
& 27935,27936,27937,27938,27939,27940,27941,27942,27943,27944,27945, &
& 27946,27947,27948,27949,27950,27951,27952,27953,27954,27955,27956, &
& 27957,27958,27959,27960,27961,27962,27963,27964,27965,27966,27967, &
& 27968,27969,27970,27971,27972,27973,27974,27975,27976,27977,27978, &
& 27979,27980,27981,27982,27983,27984,27985,27986,27987,27988,27989, &
& 27990,27991,27992,27993,27994,27995,27996,27997,27998,27999,28000, &
& 28001]
 index_cni(23501:23600)=[&
& 28002,28003,28004,28005,28006,28007,28008,28009,28010,28011,28012, &
& 28013,28014,28015,28016,28017,28018,28019,28020,28021,28022,28023, &
& 28024,28025,28026,28027,28028,28029,28030,28031,28032,28033,28034, &
& 28035,28036,28037,28038,28039,28040,28041,28042,28043,28044,28045, &
& 28046,28047,28048,28049,28050,28051,28052,28053,28054,28055,28056, &
& 28057,28058,28059,28060,28061,28062,28063,28064,28065,28066,28067, &
& 28068,28069,28070,28071,28072,28073,28074,28075,28076,28077,28078, &
& 28079,28080,28081,28082,28083,28084,28085,28086,28087,28088,28089, &
& 28090,28091,28092,28093,28094,28095,28096,28097,28098,28099,28100, &
& 28101]
 index_cni(23601:23700)=[&
& 28102,28103,28104,28105,28106,28107,28108,28109,28110,28111,28112, &
& 28113,28114,28115,28116,28117,28118,28119,28120,28121,28122,28123, &
& 28124,28125,28126,28127,28128,28129,28130,28131,28132,28133,28134, &
& 28135,28136,28137,28138,28139,28140,28141,28142,28143,28144,28145, &
& 28146,28147,28148,28149,28150,28151,28152,28153,28154,28155,28156, &
& 28157,28158,28159,28160,28161,28162,28163,28164,28165,28166,28167, &
& 28168,28169,28170,28171,28172,28173,28174,28175,28176,28177,28178, &
& 28179,28180,28181,28182,28183,28184,28185,28186,28187,28188,28189, &
& 28190,28191,28192,28193,28194,28195,28196,28197,28198,28199,28200, &
& 28201]
 index_cni(23701:23800)=[&
& 28202,28203,28204,28205,28206,28207,28208,28209,28210,28211,28212, &
& 28213,28214,28215,28216,28217,28218,28219,28220,28221,28222,28223, &
& 28224,28225,28226,28227,28228,28229,28230,28231,28232,28233,28234, &
& 28235,28236,28237,28238,28239,28240,28241,28242,28243,28244,28245, &
& 28246,28247,28248,28249,28250,28251,28252,28253,28254,28255,28256, &
& 28257,28258,28259,28260,28261,28262,28263,28264,28265,28266,28267, &
& 28268,28269,28270,28271,28272,28273,28274,28275,28276,28277,28278, &
& 28279,28280,28281,28282,28283,28284,28285,28286,28287,28288,28289, &
& 28290,28291,28292,28293,28294,28295,28296,28297,28298,28299,28300, &
& 28301]
 index_cni(23801:23900)=[&
& 28302,28303,28304,28305,28306,28307,28308,28309,28310,28311,28312, &
& 28313,28314,28315,28316,28317,28318,28319,28320,28321,28322,28323, &
& 28324,28325,28326,28327,28328,28329,28330,28331,28332,28333,28334, &
& 28335,28336,28337,28338,28339,28340,28341,28342,28343,28344,28345, &
& 28346,28347,28348,28349,28350,28351,28352,28353,28354,28355,28356, &
& 28357,28358,28359,28360,28361,28362,28363,28364,28365,28366,28367, &
& 28368,28369,28370,28371,28372,28373,28374,28375,28376,28377,28378, &
& 28379,28380,28381,28382,28383,28384,28385,28386,28387,28388,28389, &
& 28390,28391,28392,28393,28394,28395,28396,28397,28398,28399,28400, &
& 28401]
 index_cni(23901:24000)=[&
& 28402,28403,28404,28405,28406,28407,28408,28409,28410,28411,28412, &
& 28413,28414,28415,28416,28417,28418,28419,28420,28421,28422,28423, &
& 28424,28425,28426,28427,28428,28429,28430,28431,28432,28433,28434, &
& 28435,28436,28437,28438,28439,28440,28441,28442,28443,28444,28445, &
& 28446,28447,28448,28449,28450,28451,28452,28453,28454,28455,28456, &
& 28457,28458,28459,28460,28461,28462,28463,28464,28465,28466,28467, &
& 28468,28469,28470,28471,28472,28473,28474,28475,28476,28477,28478, &
& 28479,28480,28481,28482,28483,28484,28485,28486,28487,28488,28489, &
& 28490,28491,28492,28493,28494,28495,28496,28497,28498,28499,28500, &
& 28501]
 index_cni(24001:24100)=[&
& 28502,28503,28504,28505,28506,28507,28508,28509,28510,28511,28512, &
& 28513,28514,28515,28516,28517,28518,28519,28520,28521,28522,28523, &
& 28524,28525,28526,28527,28528,28529,28530,28531,28532,28533,28534, &
& 28535,28536,28537,28538,28539,28540,28541,28542,28543,28544,28545, &
& 28546,28547,28548,28549,28550,28551,28552,28553,28554,28555,28556, &
& 28557,28558,28559,28560,28561,28562,28563,28564,28565,28566,28567, &
& 28568,28569,28570,28571,28572,28573,28574,28575,28576,28577,28578, &
& 28579,28580,28581,28582,28583,28584,28585,28586,28587,28588,28589, &
& 28590,28591,28592,28593,28594,28595,28596,28597,28598,28599,28600, &
& 28601]
 index_cni(24101:24200)=[&
& 28602,28603,28604,28605,28606,28607,28608,28609,28610,28611,28612, &
& 28613,28614,28615,28616,28617,28618,28619,28620,28621,28622,28623, &
& 28624,28625,28626,28627,28628,28629,28630,28631,28632,28633,28634, &
& 28635,28636,28637,28638,28639,28640,28641,28642,28643,28644,28645, &
& 28646,28647,28648,28649,28650,28651,28652,28653,28654,28655,28656, &
& 28657,28658,28659,28660,28661,28662,28663,28664,28665,28666,28667, &
& 28668,28669,28670,28671,28672,28673,28674,28675,28676,28677,28678, &
& 28679,28680,28681,28682,28683,28684,28685,28686,28687,28688,28689, &
& 28690,28691,28692,28693,28694,28695,28696,28697,28698,28699,28700, &
& 28701]
 index_cni(24201:24300)=[&
& 28702,28703,28704,28705,28706,28707,28708,28709,28710,28711,28712, &
& 28713,28714,28715,28716,28717,28718,28719,28720,28721,28722,28723, &
& 28724,28725,28726,28727,28728,28729,28730,28731,28732,28733,28734, &
& 28735,28736,28737,28738,28739,28740,28741,28742,28743,28744,28745, &
& 28746,28747,28748,28749,28750,28751,28752,28753,28754,28755,28756, &
& 28757,28758,28759,28760,28761,28762,28763,28764,28765,28766,28767, &
& 28768,28769,28770,28771,28772,28773,28774,28775,28776,28777,28778, &
& 28779,28780,28781,28782,28783,28784,28785,28786,28787,28788,28789, &
& 28790,28791,28792,28793,28794,28795,28796,28797,28798,28799,28800, &
& 28801]
 index_cni(24301:24400)=[&
& 28802,28803,28804,28805,28806,28807,28808,28809,28810,28811,28812, &
& 28813,28814,28815,28816,28817,28818,28819,28820,28821,28822,28823, &
& 28824,28825,28826,28827,28828,28829,28830,28831,28832,28833,28834, &
& 28835,28836,28837,28838,28839,28840,28841,28842,28843,28844,28845, &
& 28846,28847,28848,28849,28850,28851,28852,28853,28854,28855,28856, &
& 28857,28858,28859,28860,28861,28862,28863,28864,28865,28866,28867, &
& 28868,28869,28870,28871,28872,28873,28874,28875,28876,28877,28878, &
& 28879,28880,28881,28882,28883,28884,28885,28886,28887,28888,28889, &
& 28890,28891,28892,28893,28894,28895,28896,28897,28898,28899,28900, &
& 28901]
 index_cni(24401:24500)=[&
& 28902,28903,28904,28905,28906,28907,28908,28909,28910,28911,28912, &
& 28913,28914,28915,28916,28917,28918,28919,28920,28921,28922,28923, &
& 28924,28925,28926,28927,28928,28929,28930,28931,28932,28933,28934, &
& 28935,28936,28937,28938,28939,28940,28941,28942,28943,28944,28945, &
& 28946,28947,28948,28949,28950,28951,28952,28953,28954,28955,28956, &
& 28957,28958,28959,28960,28961,28962,28963,28964,28965,28966,28967, &
& 28968,28969,28970,28971,28972,28973,28974,28975,28976,28977,28978, &
& 28979,28980,28981,28982,28983,28984,28985,28986,28987,28988,28989, &
& 28990,28991,28992,28993,28994,28995,28996,28997,28998,28999,29000, &
& 29001]
 index_cni(24501:24600)=[&
& 29002,29003,29004,29005,29006,29007,29008,29009,29010,29011,29012, &
& 29013,29014,29015,29016,29017,29018,29019,29020,29021,29022,29023, &
& 29024,29025,29026,29027,29028,29029,29030,29031,29032,29033,29034, &
& 29035,29036,29037,29038,29039,29040,29041,29042,29043,29044,29045, &
& 29046,29047,29048,29049,29050,29051,29052,29053,29054,29055,29056, &
& 29057,29058,29059,29060,29061,29062,29063,29064,29065,29066,29067, &
& 29068,29069,29070,29071,29072,29073,29074,29075,29076,29077,29078, &
& 29079,29080,29081,29082,29083,29084,29085,29086,29087,29088,29089, &
& 29090,29091,29092,29093,29094,29095,29096,29097,29098,29099,29100, &
& 29101]
 index_cni(24601:24700)=[&
& 29102,29103,29104,29105,29106,29107,29108,29109,29110,29111,29112, &
& 29113,29114,29115,29116,29117,29118,29119,29120,29121,29122,29123, &
& 29124,29125,29126,29127,29128,29129,29130,29131,29132,29133,29134, &
& 29135,29136,29137,29138,29139,29140,29141,29142,29143,29144,29145, &
& 29146,29147,29148,29149,29150,29151,29152,29153,29154,29155,29156, &
& 29157,29158,29159,29160,29161,29162,29163,29164,29165,29166,29167, &
& 29168,29169,29170,29171,29172,29173,29174,29175,29176,29177,29178, &
& 29179,29180,29181,29182,29183,29184,29185,29186,29187,29188,29189, &
& 29190,29191,29192,29193,29194,29195,29196,29197,29198,29199,29200, &
& 29201]
 index_cni(24701:24800)=[&
& 29202,29203,29204,29205,29206,29207,29208,29209,29210,29211,29212, &
& 29213,29214,29215,29216,29217,29218,29219,29220,29221,29222,29223, &
& 29224,29225,29226,29227,29228,29229,29230,29231,29232,29233,29234, &
& 29235,29236,29237,29238,29239,29240,29241,29242,29243,29244,29245, &
& 29246,29247,29248,29249,29250,29251,29252,29253,29254,29255,29256, &
& 29257,29258,29259,29260,29261,29262,29263,29264,29265,29266,29267, &
& 29268,29269,29270,29271,29272,29273,29274,29275,29276,29277,29278, &
& 29279,29280,29281,29282,29283,29284,29285,29286,29287,29288,29289, &
& 29290,29291,29292,29293,29294,29295,29296,29297,29298,29299,29300, &
& 29301]
 index_cni(24801:24900)=[&
& 29302,29303,29304,29305,29306,29307,29308,29309,29310,29311,29312, &
& 29313,29314,29315,29316,29317,29318,29319,29320,29321,29322,29323, &
& 29324,29325,29326,29327,29328,29329,29330,29331,29332,29333,29334, &
& 29335,29336,29337,29338,29339,29340,29341,29342,29343,29344,29345, &
& 29346,29347,29348,29349,29350,29351,29352,29353,29354,29355,29356, &
& 29357,29358,29359,29360,29361,29362,29363,29364,29365,29366,29367, &
& 29368,29369,29370,29371,29372,29373,29374,29375,29376,29377,29378, &
& 29379,29380,29381,29382,29383,29384,29385,29386,29387,29388,29389, &
& 29390,29391,29392,29393,29394,29395,29396,29397,29398,29399,29400, &
& 29401]
 index_cni(24901:25000)=[&
& 29402,29403,29404,29405,29406,29407,29408,29409,29410,29411,29412, &
& 29413,29414,29415,29416,29417,29418,29419,29420,29421,29422,29423, &
& 29424,29425,29426,29427,29428,29429,29430,29431,29432,29433,29434, &
& 29435,29436,29437,29438,29439,29440,29441,29442,29443,29444,29445, &
& 29446,29447,29448,29449,29450,29451,29452,29453,29454,29455,29456, &
& 29457,29458,29459,29460,29461,29462,29463,29464,29465,29466,29467, &
& 29468,29469,29470,29471,29472,29473,29474,29475,29476,29477,29478, &
& 29479,29480,29481,29482,29483,29484,29485,29486,29487,29488,29489, &
& 29490,29491,29492,29493,29494,29495,29496,29497,29498,29499,29500, &
& 29501]
 index_cni(25001:25100)=[&
& 29502,29503,29504,29505,29506,29507,29508,29509,29510,29511,29512, &
& 29513,29514,29515,29516,29517,29518,29519,29520,29521,29522,29523, &
& 29524,29525,29526,29527,29528,29529,29530,29531,29532,29533,29534, &
& 29535,29536,29537,29538,29539,29540,29541,29542,29543,29544,29545, &
& 29546,29547,29548,29549,29550,29551,29552,29553,29554,29555,29556, &
& 29557,29558,29559,29560,29561,29562,29563,29564,29565,29566,29567, &
& 29568,29569,29570,29571,29572,29573,29574,29575,29576,29577,29578, &
& 29579,29580,29581,29582,29583,29584,29585,29586,29587,29588,29589, &
& 29590,29591,29592,29593,29594,29595,29596,29597,29598,29599,29600, &
& 29601]
 index_cni(25101:25200)=[&
& 29602,29603,29604,29605,29606,29607,29608,29609,29610,29611,29612, &
& 29613,29614,29615,29616,29617,29618,29619,29620,29621,29622,29623, &
& 29624,29625,29626,29627,29628,29629,29630,29631,29632,29633,29634, &
& 29635,29636,29637,29638,29639,29640,29641,29642,29643,29644,29645, &
& 29646,29647,29648,29649,29650,29651,29652,29653,29654,29655,29656, &
& 29657,29658,29659,29660,29661,29662,29663,29664,29665,29666,29667, &
& 29668,29669,29670,29671,29672,29673,29674,29675,29676,29677,29678, &
& 29679,29680,29681,29682,29683,29684,29685,29686,29687,29688,29689, &
& 29690,29691,29692,29693,29694,29695,29696,29697,29698,29699,29700, &
& 29701]
 index_cni(25201:25300)=[&
& 29702,29703,29704,29705,29706,29707,29708,29709,29710,29711,29712, &
& 29713,29714,29715,29716,29717,29718,29719,29720,29721,29722,29723, &
& 29724,29725,29726,29727,29728,29729,29730,29731,29732,29733,29734, &
& 29735,29736,29737,29738,29739,29740,29741,29742,29743,29744,29745, &
& 29746,29747,29748,29749,29750,29751,29752,29753,29754,29755,29756, &
& 29757,29758,29759,29760,29761,29762,29763,29764,29765,29766,29767, &
& 29768,29769,29770,29771,29772,29773,29774,29775,29776,29777,29778, &
& 29779,29780,29781,29782,29783,29784,29785,29786,29787,29788,29789, &
& 29790,29791,29792,29793,29794,29795,29796,29797,29798,29799,29800, &
& 29801]
 index_cni(25301:25400)=[&
& 29802,29803,29804,29805,29806,29807,29808,29809,29810,29811,29812, &
& 29813,29814,29815,29816,29817,29818,29819,29820,29821,29822,29823, &
& 29824,29825,29826,29827,29828,29829,29830,29831,29832,29833,29834, &
& 29835,29836,29837,29838,29839,29840,29841,29842,29843,29844,29845, &
& 29846,29847,29848,29849,29850,29851,29852,29853,29854,29855,29856, &
& 29857,29858,29859,29860,29861,29862,29863,29864,29865,29866,29867, &
& 29868,29869,29870,29871,29872,29873,29874,29875,29876,29877,29878, &
& 29879,29880,29881,29882,29883,29884,29885,29886,29887,29888,29889, &
& 29890,29891,29892,29893,29894,29895,29896,29897,29898,29899,29900, &
& 29901]
 index_cni(25401:25500)=[&
& 29902,29903,29904,29905,29906,29907,29908,29909,29910,29911,29912, &
& 29913,29914,29915,29916,29917,29918,29919,29920,29921,29922,29923, &
& 29924,29925,29926,29927,29928,29929,29930,29931,29932,29933,29934, &
& 29935,29936,29937,29938,29939,29940,29941,29942,29943,29944,29945, &
& 29946,29947,29948,29949,29950,29951,29952,29953,29954,29955,29956, &
& 29957,29958,29959,29960,29961,29962,29963,29964,29965,29966,29967, &
& 29968,29969,29970,29971,29972,29973,29974,29975,29976,29977,29978, &
& 29979,29980,29981,29982,29983,29984,29985,29986,29987,29988,29989, &
& 29990,29991,29992,29993,29994,29995,29996,29997,29998,29999,30000, &
& 30001]
 index_cni(25501:25600)=[&
& 30002,30003,30004,30005,30006,30007,30008,30009,30010,30011,30012, &
& 30013,30014,30015,30016,30017,30018,30019,30020,30021,30022,30023, &
& 30024,30025,30026,30027,30028,30029,30030,30031,30032,30033,30034, &
& 30035,30036,30037,30038,30039,30040,30041,30042,30043,30044,30045, &
& 30046,30047,30048,30049,30050,30051,30052,30053,30054,30055,30056, &
& 30057,30058,30059,30060,30061,30062,30063,30064,30065,30066,30067, &
& 30068,30069,30070,30071,30072,30073,30074,30075,30076,30077,30078, &
& 30079,30080,30081,30082,30083,30084,30085,30086,30087,30088,30089, &
& 30090,30091,30092,30093,30094,30095,30096,30097,30098,30099,30100, &
& 30101]
 index_cni(25601:25700)=[&
& 30102,30103,30104,30105,30106,30107,30108,30109,30110,30111,30112, &
& 30113,30114,30115,30116,30117,30118,30119,30120,30121,30122,30123, &
& 30124,30125,30126,30127,30128,30129,30130,30131,30132,30133,30134, &
& 30135,30136,30137,30138,30139,30140,30141,30142,30143,30144,30145, &
& 30146,30147,30148,30149,30150,30151,30152,30153,30154,30155,30156, &
& 30157,30158,30159,30160,30161,30162,30163,30164,30165,30166,30167, &
& 30168,30169,30170,30171,30172,30173,30174,30175,30176,30177,30178, &
& 30179,30180,30181,30182,30183,30184,30185,30186,30187,30188,30189, &
& 30190,30191,30192,30193,30194,30195,30196,30197,30198,30199,30200, &
& 30201]
 index_cni(25701:25800)=[&
& 30202,30203,30204,30205,30206,30207,30208,30209,30210,30211,30212, &
& 30213,30214,30215,30216,30217,30218,30219,30220,30221,30222,30223, &
& 30224,30225,30226,30227,30228,30229,30230,30231,30232,30233,30234, &
& 30235,30236,30237,30238,30239,30240,30241,30242,30243,30244,30245, &
& 30246,30247,30248,30249,30250,30251,30252,30253,30254,30255,30256, &
& 30257,30258,30259,30260,30261,30262,30263,30264,30265,30266,30267, &
& 30268,30269,30270,30271,30272,30273,30274,30275,30276,30277,30278, &
& 30279,30280,30281,30282,30283,30284,30285,30286,30287,30288,30289, &
& 30290,30291,30292,30293,30294,30295,30296,30297,30298,30299,30300, &
& 30301]
 index_cni(25801:25900)=[&
& 30302,30303,30304,30305,30306,30307,30308,30309,30310,30311,30312, &
& 30313,30314,30315,30316,30317,30318,30319,30320,30321,30322,30323, &
& 30324,30325,30326,30327,30328,30329,30330,30331,30332,30333,30334, &
& 30335,30336,30337,30338,30339,30340,30341,30342,30343,30344,30345, &
& 30346,30347,30348,30349,30350,30351,30352,30353,30354,30355,30356, &
& 30357,30358,30359,30360,30361,30362,30363,30364,30365,30366,30367, &
& 30368,30369,30370,30371,30372,30373,30374,30375,30376,30377,30378, &
& 30379,30380,30381,30382,30383,30384,30385,30386,30387,30388,30389, &
& 30390,30391,30392,30393,30394,30395,30396,30397,30398,30399,30400, &
& 30401]
 index_cni(25901:26000)=[&
& 30402,30403,30404,30405,30406,30407,30408,30409,30410,30411,30412, &
& 30413,30414,30415,30416,30417,30418,30419,30420,30421,30422,30423, &
& 30424,30425,30426,30427,30428,30429,30430,30431,30432,30433,30434, &
& 30435,30436,30437,30438,30439,30440,30441,30442,30443,30444,30445, &
& 30446,30447,30448,30449,30450,30451,30452,30453,30454,30455,30456, &
& 30457,30458,30459,30460,30461,30462,30463,30464,30465,30466,30467, &
& 30468,30469,30470,30471,30472,30473,30474,30475,30476,30477,30478, &
& 30479,30480,30481,30482,30483,30484,30485,30486,30487,30488,30489, &
& 30490,30491,30492,30493,30494,30495,30496,30497,30498,30499,30500, &
& 30501]
 index_cni(26001:26100)=[&
& 30502,30503,30504,30505,30506,30507,30508,30509,30510,30511,30512, &
& 30513,30514,30515,30516,30517,30518,30519,30520,30521,30522,30523, &
& 30524,30525,30526,30527,30528,30529,30530,30531,30532,30533,30534, &
& 30535,30536,30537,30538,30539,30540,30541,30542,30543,30544,30545, &
& 30546,30547,30548,30549,30550,30551,30552,30553,30554,30555,30556, &
& 30557,30558,30559,30560,30561,30562,30563,30564,30565,30566,30567, &
& 30568,30569,30570,30571,30572,30573,30574,30575,30576,30577,30578, &
& 30579,30580,30581,30582,30583,30584,30585,30586,30587,30588,30589, &
& 30590,30591,30592,30593,30594,30595,30596,30597,30598,30599,30600, &
& 30601]
 index_cni(26101:26200)=[&
& 30602,30603,30604,30605,30606,30607,30608,30609,30610,30611,30612, &
& 30613,30614,30615,30616,30617,30618,30619,30620,30621,30622,30623, &
& 30624,30625,30626,30627,30628,30629,30630,30631,30632,30633,30634, &
& 30635,30636,30637,30638,30639,30640,30641,30642,30643,30644,30645, &
& 30646,30647,30648,30649,30650,30651,30652,30653,30654,30655,30656, &
& 30657,30658,30659,30660,30661,30662,30663,30664,30665,30666,30667, &
& 30668,30669,30670,30671,30672,30673,30674,30675,30676,30677,30678, &
& 30679,30680,30681,30682,30683,30684,30685,30686,30687,30688,30689, &
& 30690,30691,30692,30693,30694,30695,30696,30697,30698,30699,30700, &
& 30701]
 index_cni(26201:26300)=[&
& 30702,30703,30704,30705,30706,30707,30708,30709,30710,30711,30712, &
& 30713,30714,30715,30716,30717,30718,30719,30720,30721,30722,30723, &
& 30724,30725,30726,30727,30728,30729,30730,30731,30732,30733,30734, &
& 30735,30736,30737,30738,30739,30740,30741,30742,30743,30744,30745, &
& 30746,30747,30748,30749,30750,30751,30752,30753,30754,30755,30756, &
& 30757,30758,30759,30760,30761,30762,30763,30764,30765,30766,30767, &
& 30768,30769,30770,30771,30772,30773,30774,30775,30776,30777,30778, &
& 30779,30780,30781,30782,30783,30784,30785,30786,30787,30788,30789, &
& 30790,30791,30792,30793,30794,30795,30796,30797,30798,30799,30800, &
& 30801]
 index_cni(26301:26400)=[&
& 30802,30803,30804,30805,30806,30807,30808,30809,30810,30811,30812, &
& 30813,30814,30815,30816,30817,30818,30819,30820,30821,30822,30823, &
& 30824,30825,30826,30827,30828,30829,30830,30831,30832,30833,30834, &
& 30835,30836,30837,30838,30839,30840,30841,30842,30843,30844,30845, &
& 30846,30847,30848,30849,30850,30851,30852,30853,30854,30855,30856, &
& 30857,30858,30859,30860,30861,30862,30863,30864,30865,30866,30867, &
& 30868,30869,30870,30871,30872,30873,30874,30875,30876,30877,30878, &
& 30879,30880,30881,30882,30883,30884,30885,30886,30887,30888,30889, &
& 30890,30891,30892,30893,30894,30895,30896,30897,30898,30899,30900, &
& 30901]
 index_cni(26401:26500)=[&
& 30902,30903,30904,30905,30906,30907,30908,30909,30910,30911,30912, &
& 30913,30914,30915,30916,30917,30918,30919,30920,30921,30922,30923, &
& 30924,30925,30926,30927,30928,30929,30930,30931,30932,30933,30934, &
& 30935,30936,30937,30938,30939,30940,30941,30942,30943,30944,30945, &
& 30946,30947,30948,30949,30950,30951,30952,30953,30954,30955,30956, &
& 30957,30958,30959,30960,30961,30962,30963,30964,30965,30966,30967, &
& 30968,30969,30970,30971,30972,30973,30974,30975,30976,30977,30978, &
& 30979,30980,30981,30982,30983,30984,30985,30986,30987,30988,30989, &
& 30990,30991,30992,30993,30994,30995,30996,30997,30998,30999,31000, &
& 31001]
 index_cni(26501:26600)=[&
& 31002,31003,31004,31005,31006,31007,31008,31009,31010,31011,31012, &
& 31013,31014,31015,31016,31017,31018,31019,31020,31021,31022,31023, &
& 31024,31025,31026,31027,31028,31029,31030,31031,31032,31033,31034, &
& 31035,31036,31037,31038,31039,31040,31041,31042,31043,31044,31045, &
& 31046,31047,31048,31049,31050,31051,31052,31053,31054,31055,31056, &
& 31057,31058,31059,31060,31061,31062,31063,31064,31065,31066,31067, &
& 31068,31069,31070,31071,31072,31073,31074,31075,31076,31077,31078, &
& 31079,31080,31081,31082,31083,31084,31085,31086,31087,31088,31089, &
& 31090,31091,31092,31093,31094,31095,31096,31097,31098,31099,31100, &
& 31101]
 index_cni(26601:26700)=[&
& 31102,31103,31104,31105,31106,31107,31108,31109,31110,31111,31112, &
& 31113,31114,31115,31116,31117,31118,31119,31120,31121,31122,31123, &
& 31124,31125,31126,31127,31128,31129,31130,31131,31132,31133,31134, &
& 31135,31136,31137,31138,31139,31140,31141,31142,31143,31144,31145, &
& 31146,31147,31148,31149,31150,31151,31152,31153,31154,31155,31156, &
& 31157,31158,31159,31160,31161,31162,31163,31164,31165,31166,31167, &
& 31168,31169,31170,31171,31172,31173,31174,31175,31176,31177,31178, &
& 31179,31180,31181,31182,31183,31184,31185,31186,31187,31188,31189, &
& 31190,31191,31192,31193,31194,31195,31196,31197,31198,31199,31200, &
& 31201]
 index_cni(26701:26800)=[&
& 31202,31203,31204,31205,31206,31207,31208,31209,31210,31211,31212, &
& 31213,31214,31215,31216,31217,31218,31219,31220,31221,31222,31223, &
& 31224,31225,31226,31227,31228,31229,31230,31231,31232,31233,31234, &
& 31235,31236,31237,31238,31239,31240,31241,31242,31243,31244,31245, &
& 31246,31247,31248,31249,31250,31251,31252,31253,31254,31255,31256, &
& 31257,31258,31259,31260,31261,31262,31263,31264,31265,31266,31267, &
& 31268,31269,31270,31271,31272,31273,31274,31275,31276,31277,31278, &
& 31279,31280,31281,31282,31283,31284,31285,31286,31287,31288,31289, &
& 31290,31291,31292,31293,31294,31295,31296,31297,31298,31299,31300, &
& 31301]
 index_cni(26801:26900)=[&
& 31302,31303,31304,31305,31306,31307,31308,31309,31310,31311,31312, &
& 31313,31314,31315,31316,31317,31318,31319,31320,31321,31322,31323, &
& 31324,31325,31326,31327,31328,31329,31330,31331,31332,31333,31334, &
& 31335,31336,31337,31338,31339,31340,31341,31342,31343,31344,31345, &
& 31346,31347,31348,31349,31350,31351,31352,31353,31354,31355,31356, &
& 31357,31358,31359,31360,31361,31362,31363,31364,31365,31366,31367, &
& 31368,31369,31370,31371,31372,31373,31374,31375,31376,31377,31378, &
& 31379,31380,31381,31382,31383,31384,31385,31386,31387,31388,31389, &
& 31390,31391,31392,31393,31394,31395,31396,31397,31398,31399,31400, &
& 31401]
 index_cni(26901:27000)=[&
& 31402,31403,31404,31405,31406,31407,31408,31409,31410,31411,31412, &
& 31413,31414,31415,31416,31417,31418,31419,31420,31421,31422,31423, &
& 31424,31425,31426,31427,31428,31429,31430,31431,31432,31433,31434, &
& 31435,31436,31437,31438,31439,31440,31441,31442,31443,31444,31445, &
& 31446,31447,31448,31449,31450,31451,31452,31453,31454,31455,31456, &
& 31457,31458,31459,31460,31461,31462,31463,31464,31465,31466,31467, &
& 31468,31469,31470,31471,31472,31473,31474,31475,31476,31477,31478, &
& 31479,31480,31481,31482,31483,31484,31485,31486,31487,31488,31489, &
& 31490,31491,31492,31493,31494,31495,31496,31497,31498,31499,31500, &
& 31501]
 index_cni(27001:27100)=[&
& 31502,31503,31504,31505,31506,31507,31508,31509,31510,31511,31512, &
& 31513,31514,31515,31516,31517,31518,31519,31520,31521,31522,31523, &
& 31524,31525,31526,31527,31528,31529,31530,31531,31532,31533,31534, &
& 31535,31536,31537,31538,31539,31540,31541,31542,31543,31544,31545, &
& 31546,31547,31548,31549,31550,31551,31552,31553,31554,31555,31556, &
& 31557,31558,31559,31560,31561,31562,31563,31564,31565,31566,31567, &
& 31568,31569,31570,31571,31572,31573,31574,31575,31576,31577,31578, &
& 31579,31580,31581,31582,31583,31584,31585,31586,31587,31588,31589, &
& 31590,31591,31592,31593,31594,31595,31596,31597,31598,31599,31600, &
& 31601]
 index_cni(27101:27200)=[&
& 31602,31603,31604,31605,31606,31607,31608,31609,31610,31611,31612, &
& 31613,31614,31615,31616,31617,31618,31619,31620,31621,31622,31623, &
& 31624,31625,31626,31627,31628,31629,31630,31631,31632,31633,31634, &
& 31635,31636,31637,31638,31639,31640,31641,31642,31643,31644,31645, &
& 31646,31647,31648,31649,31650,31651,31652,31653,31654,31655,31656, &
& 31657,31658,31659,31660,31661,31662,31663,31664,31665,31666,31667, &
& 31668,31669,31670,31671,31672,31673,31674,31675,31676,31677,31678, &
& 31679,31680,31681,31682,31683,31684,31685,31686,31687,31688,31689, &
& 31690,31691,31692,31693,31694,31695,31696,31697,31698,31699,31700, &
& 31701]
 index_cni(27201:27300)=[&
& 31702,31703,31704,31705,31706,31707,31708,31709,31710,31711,31712, &
& 31713,31714,31715,31716,31717,31718,31719,31720,31721,31722,31723, &
& 31724,31725,31726,31727,31728,31729,31730,31731,31732,31733,31734, &
& 31735,31736,31737,31738,31739,31740,31741,31742,31743,31744,31745, &
& 31746,31747,31748,31749,31750,31751,31752,31753,31754,31755,31756, &
& 31757,31758,31759,31760,31761,31762,31763,31764,31765,31766,31767, &
& 31768,31769,31770,31771,31772,31773,31774,31775,31776,31777,31778, &
& 31779,31780,31781,31782,31783,31784,31785,31786,31787,31788,31789, &
& 31790,31791,31792,31793,31794,31795,31796,31797,31798,31799,31800, &
& 31801]
 index_cni(27301:27400)=[&
& 31802,31803,31804,31805,31806,31807,31808,31809,31810,31811,31812, &
& 31813,31814,31815,31816,31817,31818,31819,31820,31821,31822,31823, &
& 31824,31825,31826,31827,31828,31829,31830,31831,31832,31833,31834, &
& 31835,31836,31837,31838,31839,31840,31841,31842,31843,31844,31845, &
& 31846,31847,31848,31849,31850,31851,31852,31853,31854,31855,31856, &
& 31857,31858,31859,31860,31861,31862,31863,31864,31865,31866,31867, &
& 31868,31869,31870,31871,31872,31873,31874,31875,31876,31877,31878, &
& 31879,31880,31881,31882,31883,31884,31885,31886,31887,31888,31889, &
& 31890,31891,31892,31893,31894,31895,31896,31897,31898,31899,31900, &
& 31901]
 index_cni(27401:27500)=[&
& 31902,31903,31904,31905,31906,31907,31908,31909,31910,31911,31912, &
& 31913,31914,31915,31916,31917,31918,31919,31920,31921,31922,31923, &
& 31924,31925,31926,31927,31928,31929,31930,31931,31932,31933,31934, &
& 31935,31936,31937,31938,31939,31940,31941,31942,31943,31944,31945, &
& 31946,31947,31948,31949,31950,31951,31952,31953,31954,31955,31956, &
& 31957,31958,31959,31960,31961,31962,31963,31964,31965,31966,31967, &
& 31968,31969,31970,31971,31972,31973,31974,31975,31976,31977,31978, &
& 31979,31980,31981,31982,31983,31984,31985,31986,31987,31988,31989, &
& 31990,31991,31992,31993,31994,31995,31996,31997,31998,31999,32000, &
& 32001]
 index_cni(27501:27600)=[&
& 32002,32003,32004,32005,32006,32007,32008,32009,32010,32011,32012, &
& 32013,32014,32015,32016,32017,32018,32019,32020,32021,32022,32023, &
& 32024,32025,32026,32027,32028,32029,32030,32031,32032,32033,32034, &
& 32035,32036,32037,32038,32039,32040,32041,32042,32043,32044,32045, &
& 32046,32047,32048,32049,32050,32051,32052,32053,32054,32055,32056, &
& 32057,32058,32059,32060,32061,32062,32063,32064,32065,32066,32067, &
& 32068,32069,32070,32071,32072,32073,32074,32075,32076,32077,32078, &
& 32079,32080,32081,32082,32083,32084,32085,32086,32087,32088,32089, &
& 32090,32091,32092,32093,32094,32095,32096,32097,32098,32099,32100, &
& 32101]
 index_cni(27601:27700)=[&
& 32102,32103,32104,32105,32106,32107,32108,32109,32110,32111,32112, &
& 32113,32114,32115,32116,32117,32118,32119,32120,32121,32122,32123, &
& 32124,32125,32126,32127,32128,32129,32130,32131,32132,32133,32134, &
& 32135,32136,32137,32138,32139,32140,32141,32142,32143,32144,32145, &
& 32146,32147,32148,32149,32150,32151,32152,32153,32154,32155,32156, &
& 32157,32158,32159,32160,32161,32162,32163,32164,32165,32166,32167, &
& 32168,32169,32170,32171,32172,32173,32174,32175,32176,32177,32178, &
& 32179,32180,32181,32182,32183,32184,32185,32186,32187,32188,32189, &
& 32190,32191,32192,32193,32194,32195,32196,32197,32198,32199,32200, &
& 32201]
 index_cni(27701:27800)=[&
& 32202,32203,32204,32205,32206,32207,32208,32209,32210,32211,32212, &
& 32213,32214,32215,32216,32217,32218,32219,32220,32221,32222,32223, &
& 32224,32225,32226,32227,32228,32229,32230,32231,32232,32233,32234, &
& 32235,32236,32237,32238,32239,32240,32241,32242,32243,32244,32245, &
& 32246,32247,32248,32249,32250,32251,32252,32253,32254,32255,32256, &
& 32257,32258,32259,32260,32261,32262,32263,32264,32265,32266,32267, &
& 32268,32269,32270,32271,32272,32273,32274,32275,32276,32277,32278, &
& 32279,32280,32281,32282,32283,32284,32285,32286,32287,32288,32289, &
& 32290,32291,32292,32293,32294,32295,32296,32297,32298,32299,32300, &
& 32301]
 index_cni(27801:27884)=[&
& 32302,32303,32304,32305,32306,32307,32308,32309,32310,32311,32312, &
& 32313,32314,32315,32316,32317,32318,32319,32320,32321,32322,32323, &
& 32324,32325,32326,32327,32328,32329,32330,32331,32332,32333,32334, &
& 32335,32336,32337,32338,32339,32340,32341,32342,32343,32344,32345, &
& 32346,32347,32348,32349,32350,32351,32352,32353,32354,32355,32356, &
& 32357,32358,32359,32360,32361,32362,32363,32364,32365,32366,32367, &
& 32368,32369,32370,32371,32372,32373,32374,32375,32376,32377,32378, &
& 32379,32380,32381,32382,32383,32384,32385]

 vdw_dftd3_cni(1:100)=[&
& 0.9118,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991, &
& 2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991, &
& 2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991, &
& 2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991, &
& 2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991, &
& 2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991,2.7991, &
& 2.7991,2.7991,2.7991,2.7991,2.7991,0.9865,0.9865,0.9865,0.9865, &
& 0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865, &
& 0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865, &
& 0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865, &
& 0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865, &
& 0.9865]
 vdw_dftd3_cni(101:200)=[&
& 0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865, &
& 0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865, &
& 0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865, &
& 0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865, &
& 0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865, &
& 0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865,0.9865, &
& 0.9865,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808, &
& 0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808, &
& 0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808, &
& 0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808, &
& 0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808, &
& 0.9808]
 vdw_dftd3_cni(201:300)=[&
& 0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808, &
& 0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808, &
& 0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808, &
& 0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808, &
& 0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808, &
& 0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9808,0.9706,0.9706, &
& 0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706, &
& 0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706, &
& 0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706, &
& 0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706, &
& 0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706, &
& 0.9706]
 vdw_dftd3_cni(301:400)=[&
& 0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706, &
& 0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706, &
& 0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706, &
& 0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706, &
& 0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706,0.9706, &
& 0.9706,0.9706,0.9706,0.9706,0.9706,0.9868,0.9868,0.9868,0.9868, &
& 0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868, &
& 0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868, &
& 0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868, &
& 0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868, &
& 0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868, &
& 0.9868]
 vdw_dftd3_cni(401:500)=[&
& 0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868, &
& 0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868, &
& 0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868, &
& 0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868, &
& 0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868,0.9868, &
& 0.9868,0.9868,0.9868,0.9868,0.9944,0.9944,0.9944,0.9944,0.9944, &
& 0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944, &
& 0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944, &
& 0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944, &
& 0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944, &
& 0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944, &
& 0.9944]
 vdw_dftd3_cni(501:600)=[&
& 0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944, &
& 0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944, &
& 0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944, &
& 0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944, &
& 0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944,0.9944, &
& 0.9944,0.9944,0.9944,0.9944,0.9925,0.9925,0.9925,0.9925,0.9925, &
& 0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925, &
& 0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925, &
& 0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925, &
& 0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925, &
& 0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925, &
& 0.9925]
 vdw_dftd3_cni(601:700)=[&
& 0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925, &
& 0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925, &
& 0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925, &
& 0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925, &
& 0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925,0.9925, &
& 0.9925,0.9925,0.9925,0.9925,0.9925,0.9982,0.9982,0.9982,0.9982, &
& 0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982, &
& 0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982, &
& 0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982, &
& 0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982, &
& 0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982, &
& 0.9982]
 vdw_dftd3_cni(701:800)=[&
& 0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982, &
& 0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982, &
& 0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982, &
& 0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982, &
& 0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982, &
& 0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9982,0.9684,0.9684, &
& 0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684, &
& 0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684, &
& 0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684, &
& 0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684, &
& 0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684, &
& 0.9684]
 vdw_dftd3_cni(801:900)=[&
& 0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684, &
& 0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684, &
& 0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684, &
& 0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684, &
& 0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684, &
& 0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684,0.9684, &
& 0.9684,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628, &
& 0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628, &
& 0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628, &
& 0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628, &
& 0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628, &
& 0.9628]
 vdw_dftd3_cni(901:1000)=[&
& 0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628, &
& 0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628, &
& 0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628, &
& 0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628, &
& 0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628, &
& 0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628,0.9628, &
& 0.9628,0.9628,0.9628,0.9628,0.9628,0.9648,0.9648,0.9648,0.9648, &
& 0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648, &
& 0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648, &
& 0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648, &
& 0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648, &
& 0.9648]
 vdw_dftd3_cni(1001:1100)=[&
& 0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648, &
& 0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648, &
& 0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648, &
& 0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648, &
& 0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648, &
& 0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648, &
& 0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648,0.9648, &
& 0.9648,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507, &
& 0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507, &
& 0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507, &
& 0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507, &
& 0.9507]
 vdw_dftd3_cni(1101:1200)=[&
& 0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507, &
& 0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507, &
& 0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507, &
& 0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507, &
& 0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507, &
& 0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507, &
& 0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507, &
& 0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9507,0.9947,0.9947, &
& 0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947, &
& 0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947, &
& 0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947, &
& 0.9947]
 vdw_dftd3_cni(1201:1300)=[&
& 0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947, &
& 0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947, &
& 0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947, &
& 0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947, &
& 0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947, &
& 0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947, &
& 0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947, &
& 0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947,0.9947, &
& 0.9947,0.9947,0.9947,0.9947,0.9947,0.9948,0.9948,0.9948,0.9948, &
& 0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948, &
& 0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948, &
& 0.9948]
 vdw_dftd3_cni(1301:1400)=[&
& 0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948, &
& 0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948, &
& 0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948, &
& 0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948, &
& 0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948, &
& 0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948, &
& 0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948, &
& 0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948, &
& 0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948,0.9948, &
& 0.9948,0.9948,0.9948,0.9948,0.9972,0.9972,0.9972,0.9972,0.9972, &
& 0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972, &
& 0.9972]
 vdw_dftd3_cni(1401:1500)=[&
& 0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972, &
& 0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972, &
& 0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972, &
& 0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972, &
& 0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972, &
& 0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972, &
& 0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972, &
& 0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972, &
& 0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972, &
& 0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972,0.9972, &
& 0.9972,0.9972,0.9972,0.9972,0.9767,0.9767,0.9767,0.9767,0.9767, &
& 0.9767]
 vdw_dftd3_cni(1501:1600)=[&
& 0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767, &
& 0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767, &
& 0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767, &
& 0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767, &
& 0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767, &
& 0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767, &
& 0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767, &
& 0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767, &
& 0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767, &
& 0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767, &
& 0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767,0.9767, &
& 0.9767]
 vdw_dftd3_cni(1601:1700)=[&
& 0.9767,0.9767,0.9767,0.9767,0.9831,0.9831,0.9831,0.9831,0.9831, &
& 0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831, &
& 0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831, &
& 0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831, &
& 0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831, &
& 0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831, &
& 0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831, &
& 0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831, &
& 0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831, &
& 0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831, &
& 0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831, &
& 0.9831]
 vdw_dftd3_cni(1701:1800)=[&
& 0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,0.9831, &
& 0.9831,0.9831,0.9831,0.9831,0.9831,0.9831,1.8627,1.8627,1.8627, &
& 1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627, &
& 1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627, &
& 1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627, &
& 1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627, &
& 1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627, &
& 1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627, &
& 1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627, &
& 1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627, &
& 1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627, &
& 1.8627]
 vdw_dftd3_cni(1801:1900)=[&
& 1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627, &
& 1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627, &
& 1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627,1.8627, &
& 1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299, &
& 1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299, &
& 1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299, &
& 1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299, &
& 1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299, &
& 1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299, &
& 1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299, &
& 1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299, &
& 1.8299]
 vdw_dftd3_cni(1901:2000)=[&
& 1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299, &
& 1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299, &
& 1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299, &
& 1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299,1.8299, &
& 1.8299,1.8299,1.8299,1.8299,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.9138]
 vdw_dftd3_cni(2001:2100)=[&
& 1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138,1.9138, &
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269, &
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269, &
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269, &
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269, &
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269, &
& 1.8269]
 vdw_dftd3_cni(2101:2200)=[&
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269, &
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269, &
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269, &
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269, &
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269, &
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269, &
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.8269, &
& 1.8269,1.8269,1.8269,1.8269,1.8269,1.8269,1.6406,1.6406,1.6406, &
& 1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406, &
& 1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406, &
& 1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406, &
& 1.6406]
 vdw_dftd3_cni(2201:2300)=[&
& 1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406, &
& 1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406, &
& 1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406, &
& 1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406, &
& 1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406, &
& 1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406, &
& 1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406, &
& 1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406, &
& 1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406,1.6406, &
& 1.6406,1.6406,1.6406,1.6406,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483]
 vdw_dftd3_cni(2301:2400)=[&
& 1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483,1.6483, &
& 1.6483]
 vdw_dftd3_cni(2401:2500)=[&
& 1.6483,1.6483,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149]
 vdw_dftd3_cni(2501:2600)=[&
& 1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149,1.7149, &
& 1.7149,1.7149,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937]
 vdw_dftd3_cni(2601:2700)=[&
& 1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937,1.7937, &
& 1.7937,1.7937,1.7937,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576]
 vdw_dftd3_cni(2701:2800)=[&
& 0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576,0.9576, &
& 0.9576,0.9576,0.9576,0.9576,0.9576,1.9419,1.9419,1.9419,1.9419, &
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419, &
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419, &
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419, &
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419, &
& 1.9419]
 vdw_dftd3_cni(2801:2900)=[&
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419, &
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419, &
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419, &
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419, &
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419, &
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419, &
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419, &
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419, &
& 1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,1.9419,0.9601, &
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601]
 vdw_dftd3_cni(2901:3000)=[&
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601,0.9601, &
& 0.9601]
 vdw_dftd3_cni(3001:3100)=[&
& 0.9601,0.9601,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434]
 vdw_dftd3_cni(3101:3200)=[&
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434, &
& 0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9434,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889]
 vdw_dftd3_cni(3201:3300)=[&
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889,0.9889, &
& 0.9889,0.9889,0.9889,0.9889,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901]
 vdw_dftd3_cni(3301:3400)=[&
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901,0.9901, &
& 0.9901,0.9901,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974]
 vdw_dftd3_cni(3401:3500)=[&
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974,0.9974, &
& 0.9974]
 vdw_dftd3_cni(3501:3600)=[&
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738]
 vdw_dftd3_cni(3601:3700)=[&
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738,0.9738, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801]
 vdw_dftd3_cni(3701:3800)=[&
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801,0.9801, &
& 0.9801,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153]
 vdw_dftd3_cni(3801:3900)=[&
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153,1.9153, &
& 1.9153,1.9153,1.9153,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355]
 vdw_dftd3_cni(3901:4000)=[&
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355]
 vdw_dftd3_cni(4001:4100)=[&
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355,1.9355, &
& 1.9355,1.9355,1.9355,1.9355,1.9355,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545]
 vdw_dftd3_cni(4101:4200)=[&
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545,1.9545, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420]
 vdw_dftd3_cni(4201:4300)=[&
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420,1.9420, &
& 1.9420,1.9420,1.9420,1.9420,1.9420,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682]
 vdw_dftd3_cni(4301:4400)=[&
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682]
 vdw_dftd3_cni(4401:4500)=[&
& 1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682,1.6682, &
& 1.6682,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584]
 vdw_dftd3_cni(4501:4600)=[&
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584, &
& 1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.8584,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003]
 vdw_dftd3_cni(4601:4700)=[&
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003, &
& 1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.9003,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630]
 vdw_dftd3_cni(4701:4800)=[&
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630]
 vdw_dftd3_cni(4801:4900)=[&
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,1.8630, &
& 1.8630,1.8630,1.8630,1.8630,1.8630,1.8630,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679]
 vdw_dftd3_cni(4901:5000)=[&
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679, &
& 0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,0.9679,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539]
 vdw_dftd3_cni(5001:5100)=[&
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539,1.9539, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633]
 vdw_dftd3_cni(5101:5200)=[&
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633]
 vdw_dftd3_cni(5201:5300)=[&
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633,0.9633, &
& 0.9633,0.9633,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514]
 vdw_dftd3_cni(5301:5400)=[&
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9514, &
& 0.9514,0.9514,0.9514,0.9514,0.9514,0.9514,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749]
 vdw_dftd3_cni(5401:5500)=[&
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749]
 vdw_dftd3_cni(5501:5600)=[&
& 0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749,0.9749, &
& 0.9749,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811]
 vdw_dftd3_cni(5601:5700)=[&
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811, &
& 0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9811,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968]
 vdw_dftd3_cni(5701:5800)=[&
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968,0.9968, &
& 0.9968,0.9968,0.9968,0.9968,0.9968,0.9909,0.9909,0.9909,0.9909, &
& 0.9909]
 vdw_dftd3_cni(5801:5900)=[&
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909]
 vdw_dftd3_cni(5901:6000)=[&
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909,0.9909, &
& 0.9909,0.9909,0.9909,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797]
 vdw_dftd3_cni(6001:6100)=[&
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797,0.9797, &
& 0.9797,0.9797,0.9797,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373]
 vdw_dftd3_cni(6101:6200)=[&
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373]
 vdw_dftd3_cni(6201:6300)=[&
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373,1.9373, &
& 1.9373,1.9373,1.9373,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425]
 vdw_dftd3_cni(6301:6400)=[&
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425,2.9425, &
& 2.9425,2.9425,2.9425,2.9425,2.9425,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455]
 vdw_dftd3_cni(6401:6500)=[&
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455]
 vdw_dftd3_cni(6501:6600)=[&
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455, &
& 2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9455,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413]
 vdw_dftd3_cni(6601:6700)=[&
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413,2.9413, &
& 2.9413,2.9413,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300]
 vdw_dftd3_cni(6701:6800)=[&
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300]
 vdw_dftd3_cni(6801:6900)=[&
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,2.9300, &
& 2.9300,2.9300,2.9300,2.9300,2.9300,2.9300,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286]
 vdw_dftd3_cni(6901:7000)=[&
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286,1.8286, &
& 1.8286,1.8286,1.8286,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732]
 vdw_dftd3_cni(7001:7100)=[&
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732]
 vdw_dftd3_cni(7101:7200)=[&
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732,2.8732, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086]
 vdw_dftd3_cni(7201:7300)=[&
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086, &
& 2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.9086,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965]
 vdw_dftd3_cni(7301:7400)=[&
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965]
 vdw_dftd3_cni(7401:7500)=[&
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965, &
& 2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.8965,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242]
 vdw_dftd3_cni(7501:7600)=[&
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242, &
& 2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9242,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282]
 vdw_dftd3_cni(7601:7700)=[&
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282]
 vdw_dftd3_cni(7701:7800)=[&
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282,2.9282, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246]
 vdw_dftd3_cni(7801:7900)=[&
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246,2.9246, &
& 2.9246]
 vdw_dftd3_cni(7901:8000)=[&
& 2.9246,2.9246,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482]
 vdw_dftd3_cni(8001:8100)=[&
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.8482, &
& 2.8482,2.8482,2.8482,2.8482,2.8482,2.8482,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219]
 vdw_dftd3_cni(8101:8200)=[&
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219]
 vdw_dftd3_cni(8201:8300)=[&
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219,2.9219, &
& 2.9219,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254]
 vdw_dftd3_cni(8301:8400)=[&
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254, &
& 1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9254,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459]
 vdw_dftd3_cni(8401:8500)=[&
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459]
 vdw_dftd3_cni(8501:8600)=[&
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459,1.9459, &
& 1.9459,1.9459,1.9459,1.9459,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292]
 vdw_dftd3_cni(8601:8700)=[&
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292,1.9292, &
& 1.9292]
 vdw_dftd3_cni(8701:8800)=[&
& 1.9292,1.9292,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104]
 vdw_dftd3_cni(8801:8900)=[&
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104,1.8104, &
& 1.8104,1.8104,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858]
 vdw_dftd3_cni(8901:9000)=[&
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858]
 vdw_dftd3_cni(9001:9100)=[&
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858,1.8858, &
& 1.8858,1.8858,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648]
 vdw_dftd3_cni(9101:9200)=[&
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648,1.8648, &
& 1.8648,1.8648,1.8648,1.8648,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188]
 vdw_dftd3_cni(9201:9300)=[&
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188]
 vdw_dftd3_cni(9301:9400)=[&
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,1.9188, &
& 1.9188,1.9188,1.9188,1.9188,1.9188,1.9188,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846]
 vdw_dftd3_cni(9401:9500)=[&
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846]
 vdw_dftd3_cni(9501:9600)=[&
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846,0.9846, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896]
 vdw_dftd3_cni(9601:9700)=[&
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896,1.9896, &
& 1.9896,1.9896,1.9896,1.9896,1.9896,0.9267,0.9267,0.9267,0.9267, &
& 0.9267]
 vdw_dftd3_cni(9701:9800)=[&
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267]
 vdw_dftd3_cni(9801:9900)=[&
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267,0.9267, &
& 0.9267,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383]
 vdw_dftd3_cni(9901:10000)=[&
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383]
 vdw_dftd3_cni(10001:10100)=[&
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383, &
& 0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9383,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820]
 vdw_dftd3_cni(10101:10200)=[&
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820,0.9820, &
& 0.9820]
 vdw_dftd3_cni(10201:10300)=[&
& 0.9820,0.9820,0.9820,0.9820,0.9820,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815]
 vdw_dftd3_cni(10301:10400)=[&
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815,0.9815, &
& 0.9815,0.9815,0.9815,0.9815,0.9815,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954]
 vdw_dftd3_cni(10401:10500)=[&
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954]
 vdw_dftd3_cni(10501:10600)=[&
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954,0.9954, &
& 0.9954,0.9954,0.9954,0.9954,0.9954,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705]
 vdw_dftd3_cni(10601:10700)=[&
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705]
 vdw_dftd3_cni(10701:10800)=[&
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9705, &
& 0.9705,0.9705,0.9705,0.9705,0.9705,0.9705,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662]
 vdw_dftd3_cni(10801:10900)=[&
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662,0.9662, &
& 2.9070]
 vdw_dftd3_cni(10901:11000)=[&
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070]
 vdw_dftd3_cni(11001:11100)=[&
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070,2.9070, &
& 2.9070,2.9070,2.9070,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844]
 vdw_dftd3_cni(11101:11200)=[&
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844]
 vdw_dftd3_cni(11201:11300)=[&
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844, &
& 2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8844,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738]
 vdw_dftd3_cni(11301:11400)=[&
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738]
 vdw_dftd3_cni(11401:11500)=[&
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738,2.8738, &
& 2.8738,2.8738,2.8738,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878]
 vdw_dftd3_cni(11501:11600)=[&
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.8878]
 vdw_dftd3_cni(11601:11700)=[&
& 2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878,2.8878, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095]
 vdw_dftd3_cni(11701:11800)=[&
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095, &
& 2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,2.9095,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209]
 vdw_dftd3_cni(11801:11900)=[&
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209]
 vdw_dftd3_cni(11901:12000)=[&
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209, &
& 1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9209,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697]
 vdw_dftd3_cni(12001:12100)=[&
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697]
 vdw_dftd3_cni(12101:12200)=[&
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697, &
& 1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9697,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441]
 vdw_dftd3_cni(12201:12300)=[&
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441]
 vdw_dftd3_cni(12301:12400)=[&
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441, &
& 1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9441,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985]
 vdw_dftd3_cni(12401:12500)=[&
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985]
 vdw_dftd3_cni(12501:12600)=[&
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985,1.9985, &
& 1.9985,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143]
 vdw_dftd3_cni(12601:12700)=[&
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143,2.0143, &
& 2.0143]
 vdw_dftd3_cni(12701:12800)=[&
& 2.0143,2.0143,2.0143,2.0143,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887]
 vdw_dftd3_cni(12801:12900)=[&
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887,1.9887, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496]
 vdw_dftd3_cni(12901:13000)=[&
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496]
 vdw_dftd3_cni(13001:13100)=[&
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496,1.9496, &
& 1.9496,1.9496,1.9496,1.9496,1.9496,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311]
 vdw_dftd3_cni(13101:13200)=[&
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311]
 vdw_dftd3_cni(13201:13300)=[&
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311,1.9311, &
& 1.9311,1.9311,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435]
 vdw_dftd3_cni(13301:13400)=[&
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435]
 vdw_dftd3_cni(13401:13500)=[&
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435,1.9435, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102]
 vdw_dftd3_cni(13501:13600)=[&
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102]
 vdw_dftd3_cni(13601:13700)=[&
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102, &
& 2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,2.0102,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903]
 vdw_dftd3_cni(13701:13800)=[&
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903]
 vdw_dftd3_cni(13801:13900)=[&
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903, &
& 1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9903,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349]
 vdw_dftd3_cni(13901:14000)=[&
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349]
 vdw_dftd3_cni(14001:14100)=[&
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349,1.9349, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999]
 vdw_dftd3_cni(14101:14200)=[&
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999]
 vdw_dftd3_cni(14201:14300)=[&
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999,2.8999, &
& 2.8999,2.8999,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675]
 vdw_dftd3_cni(14301:14400)=[&
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675]
 vdw_dftd3_cni(14401:14500)=[&
& 3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675,3.8675, &
& 3.8675,3.8675,3.8675,3.8675,3.8675,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110]
 vdw_dftd3_cni(14501:14600)=[&
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 2.9110]
 vdw_dftd3_cni(14601:14700)=[&
& 2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110,2.9110, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191]
 vdw_dftd3_cni(14701:14800)=[&
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191,10.6191, &
& 10.6191]
 vdw_dftd3_cni(14801:14900)=[&
& 10.6191,10.6191,10.6191,10.6191,10.6191,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849]
 vdw_dftd3_cni(14901:15000)=[&
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849,9.8849, &
& 9.8849]
 vdw_dftd3_cni(15001:15100)=[&
& 9.8849,9.8849,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376]
 vdw_dftd3_cni(15101:15200)=[&
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376,9.1376, &
& 9.1376]
 vdw_dftd3_cni(15201:15300)=[&
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263]
 vdw_dftd3_cni(15301:15400)=[&
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263,2.9263, &
& 6.5458]
 vdw_dftd3_cni(15401:15500)=[&
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458]
 vdw_dftd3_cni(15501:15600)=[&
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458,6.5458, &
& 1.9315]
 vdw_dftd3_cni(15601:15700)=[&
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315]
 vdw_dftd3_cni(15701:15800)=[&
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315,1.9315, &
& 1.9315]
 vdw_dftd3_cni(15801:15900)=[&
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447]
 vdw_dftd3_cni(15901:16000)=[&
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447,1.9447, &
& 1.9447]
 vdw_dftd3_cni(16001:16100)=[&
& 1.9447,1.9447,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793]
 vdw_dftd3_cni(16101:16200)=[&
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793,1.9793, &
& 1.9793]
 vdw_dftd3_cni(16201:16300)=[&
& 1.9793,1.9793,1.9793,1.9793,1.9793,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812]
 vdw_dftd3_cni(16301:16400)=[&
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9812]
 vdw_dftd3_cni(16401:16500)=[&
& 1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812,1.9812, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143]
 vdw_dftd3_cni(16501:16600)=[&
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143]
 vdw_dftd3_cni(16601:16700)=[&
& 1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143,1.9143, &
& 1.9143,1.9143,1.9143,1.9143,1.9143,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903]
 vdw_dftd3_cni(16701:16800)=[&
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903]
 vdw_dftd3_cni(16801:16900)=[&
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903,2.8903, &
& 2.8903,2.8903,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106]
 vdw_dftd3_cni(16901:17000)=[&
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106]
 vdw_dftd3_cni(17001:17100)=[&
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106,3.9106, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225]
 vdw_dftd3_cni(17101:17200)=[&
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225]
 vdw_dftd3_cni(17201:17300)=[&
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225, &
& 2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,2.9225,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556]
 vdw_dftd3_cni(17301:17400)=[&
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556]
 vdw_dftd3_cni(17401:17500)=[&
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556, &
& 11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,11.0556,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402]
 vdw_dftd3_cni(17501:17600)=[&
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402]
 vdw_dftd3_cni(17601:17700)=[&
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402,9.5402, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895]
 vdw_dftd3_cni(17701:17800)=[&
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895]
 vdw_dftd3_cni(17801:17900)=[&
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895,8.8895, &
& 8.8895,8.8895,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696]
 vdw_dftd3_cni(17901:18000)=[&
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696]
 vdw_dftd3_cni(18001:18100)=[&
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696,2.9696, &
& 2.9696,2.9696,2.9696,2.9696,2.9696,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095]
 vdw_dftd3_cni(18101:18200)=[&
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095]
 vdw_dftd3_cni(18201:18300)=[&
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095,5.7095, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378]
 vdw_dftd3_cni(18301:18400)=[&
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378]
 vdw_dftd3_cni(18401:18500)=[&
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378,1.9378, &
& 1.9378]
 vdw_dftd3_cni(18501:18600)=[&
& 1.9378,1.9378,1.9378,1.9378,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505]
 vdw_dftd3_cni(18601:18700)=[&
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505]
 vdw_dftd3_cni(18701:18800)=[&
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505,1.9505, &
& 1.9505,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523]
 vdw_dftd3_cni(18801:18900)=[&
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523]
 vdw_dftd3_cni(18901:19000)=[&
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523, &
& 1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9523,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639]
 vdw_dftd3_cni(19001:19100)=[&
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639]
 vdw_dftd3_cni(19101:19200)=[&
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467]
 vdw_dftd3_cni(19201:19300)=[&
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467]
 vdw_dftd3_cni(19301:19400)=[&
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467, &
& 1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,1.8467,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175]
 vdw_dftd3_cni(19401:19500)=[&
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175]
 vdw_dftd3_cni(19501:19600)=[&
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175, &
& 2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,2.9175,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840]
 vdw_dftd3_cni(19601:19700)=[&
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840]
 vdw_dftd3_cni(19701:19800)=[&
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 3.8840]
 vdw_dftd3_cni(19801:19900)=[&
& 3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840,3.8840, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988]
 vdw_dftd3_cni(19901:20000)=[&
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988]
 vdw_dftd3_cni(20001:20100)=[&
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988,2.8988, &
& 2.8988,2.8988,2.8988,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153]
 vdw_dftd3_cni(20101:20200)=[&
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153]
 vdw_dftd3_cni(20201:20300)=[&
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153, &
& 10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,10.9153,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054]
 vdw_dftd3_cni(20301:20400)=[&
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054]
 vdw_dftd3_cni(20401:20500)=[&
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054,9.8054, &
& 9.8054,9.8054,9.8054,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527]
 vdw_dftd3_cni(20501:20600)=[&
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527]
 vdw_dftd3_cni(20601:20700)=[&
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527,9.1527, &
& 2.9424]
 vdw_dftd3_cni(20701:20800)=[&
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424]
 vdw_dftd3_cni(20801:20900)=[&
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424]
 vdw_dftd3_cni(20901:21000)=[&
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,2.9424, &
& 2.9424,2.9424,2.9424,2.9424,2.9424,2.9424,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669]
 vdw_dftd3_cni(21001:21100)=[&
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669]
 vdw_dftd3_cni(21101:21200)=[&
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669,6.6669, &
& 6.6669,6.6669,6.6669,6.6669,6.6669,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302]
 vdw_dftd3_cni(21201:21300)=[&
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302]
 vdw_dftd3_cni(21301:21400)=[&
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302,1.9302, &
& 1.9302,1.9302,1.9302,1.9302,1.9302,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356]
 vdw_dftd3_cni(21401:21500)=[&
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356]
 vdw_dftd3_cni(21501:21600)=[&
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356,1.9356, &
& 1.9356]
 vdw_dftd3_cni(21601:21700)=[&
& 1.9356,1.9356,1.9356,1.9356,1.9356,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655]
 vdw_dftd3_cni(21701:21800)=[&
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655]
 vdw_dftd3_cni(21801:21900)=[&
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655, &
& 1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9655,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639]
 vdw_dftd3_cni(21901:22000)=[&
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639]
 vdw_dftd3_cni(22001:22100)=[&
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639,1.9639, &
& 1.9639,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075]
 vdw_dftd3_cni(22101:22200)=[&
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075]
 vdw_dftd3_cni(22201:22300)=[&
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075,1.8075, &
& 1.8075,1.8075,1.8075,1.8075,1.8075,2.9128,2.9128,2.9128,2.9128, &
& 2.9128]
 vdw_dftd3_cni(22301:22400)=[&
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128]
 vdw_dftd3_cni(22401:22500)=[&
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128]
 vdw_dftd3_cni(22501:22600)=[&
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128,2.9128, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987]
 vdw_dftd3_cni(22601:22700)=[&
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987]
 vdw_dftd3_cni(22701:22800)=[&
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9987, &
& 2.9987,2.9987,2.9987,2.9987,2.9987,2.9987,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903]
 vdw_dftd3_cni(22801:22900)=[&
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903]
 vdw_dftd3_cni(22901:23000)=[&
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903,2.9903, &
& 2.9903,2.9903,2.9903,2.9903,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146]
 vdw_dftd3_cni(23001:23100)=[&
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146]
 vdw_dftd3_cni(23101:23200)=[&
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146]
 vdw_dftd3_cni(23201:23300)=[&
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146,2.9146, &
& 2.9146,2.9146,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407]
 vdw_dftd3_cni(23301:23400)=[&
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407]
 vdw_dftd3_cni(23401:23500)=[&
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407,2.9407, &
& 2.9407,2.9407,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859]
 vdw_dftd3_cni(23501:23600)=[&
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859]
 vdw_dftd3_cni(23601:23700)=[&
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859,2.9859, &
& 2.9859]
 vdw_dftd3_cni(23701:23800)=[&
& 2.9859,2.9859,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785]
 vdw_dftd3_cni(23801:23900)=[&
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785]
 vdw_dftd3_cni(23901:24000)=[&
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785,7.7785, &
& 7.7785,7.7785,7.7785,7.7785,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918]
 vdw_dftd3_cni(24001:24100)=[&
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918]
 vdw_dftd3_cni(24101:24200)=[&
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918, &
& 6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,6.2918,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233]
 vdw_dftd3_cni(24201:24300)=[&
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233]
 vdw_dftd3_cni(24301:24400)=[&
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233]
 vdw_dftd3_cni(24401:24500)=[&
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233,2.9233, &
& 2.9233,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186]
 vdw_dftd3_cni(24501:24600)=[&
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186]
 vdw_dftd3_cni(24601:24700)=[&
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9186, &
& 2.9186,2.9186,2.9186,2.9186,2.9186,2.9186,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709]
 vdw_dftd3_cni(24701:24800)=[&
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709]
 vdw_dftd3_cni(24801:24900)=[&
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709,2.9709, &
& 2.9709]
 vdw_dftd3_cni(24901:25000)=[&
& 2.9709,2.9709,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353]
 vdw_dftd3_cni(25001:25100)=[&
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353]
 vdw_dftd3_cni(25101:25200)=[&
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353,2.9353, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259]
 vdw_dftd3_cni(25201:25300)=[&
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259]
 vdw_dftd3_cni(25301:25400)=[&
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259, &
& 2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9259,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315]
 vdw_dftd3_cni(25401:25500)=[&
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315]
 vdw_dftd3_cni(25501:25600)=[&
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315]
 vdw_dftd3_cni(25601:25700)=[&
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315, &
& 2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9315,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420]
 vdw_dftd3_cni(25701:25800)=[&
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420]
 vdw_dftd3_cni(25801:25900)=[&
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420, &
& 2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9420,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081]
 vdw_dftd3_cni(25901:26000)=[&
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081]
 vdw_dftd3_cni(26001:26100)=[&
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081]
 vdw_dftd3_cni(26101:26200)=[&
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081,2.9081, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500]
 vdw_dftd3_cni(26201:26300)=[&
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500]
 vdw_dftd3_cni(26301:26400)=[&
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500,2.9500, &
& 2.9500,2.9500,2.9500,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856]
 vdw_dftd3_cni(26401:26500)=[&
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856]
 vdw_dftd3_cni(26501:26600)=[&
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856]
 vdw_dftd3_cni(26601:26700)=[&
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,4.5856, &
& 4.5856,4.5856,4.5856,4.5856,4.5856,4.5856,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844]
 vdw_dftd3_cni(26701:26800)=[&
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844]
 vdw_dftd3_cni(26801:26900)=[&
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844,3.9844, &
& 3.9844,3.9844,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677]
 vdw_dftd3_cni(26901:27000)=[&
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677]
 vdw_dftd3_cni(27001:27100)=[&
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677]
 vdw_dftd3_cni(27101:27200)=[&
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677, &
& 3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8677,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972]
 vdw_dftd3_cni(27201:27300)=[&
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972]
 vdw_dftd3_cni(27301:27400)=[&
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972,3.8972, &
& 3.8972,3.8972,3.8972,3.8972,3.8972,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123]
 vdw_dftd3_cni(27401:27500)=[&
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123]
 vdw_dftd3_cni(27501:27600)=[&
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123]
 vdw_dftd3_cni(27601:27700)=[&
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123,3.9123, &
& 3.9123,3.9123,3.9123,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098]
 vdw_dftd3_cni(27701:27800)=[&
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098]
 vdw_dftd3_cni(27801:27884)=[&
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098,3.9098, &
& 3.9098,3.9098,3.9098]

 index_cnj(1:100)=[&
& 1,2,4,7,11,16,22,29,37,46,56, &
& 67,79,92,106,121,137,154,172,191,211,232, &
& 254,277,301,326,352,379,407,436,466,497,529, &
& 562,596,631,667,704,742,781,821,862,904,947, &
& 991,1036,1082,1129,1177,1226,1276,1327,1379,1432,1486, &
& 1541,1597,1654,1711,1712,1769,1771,1828,1831,1888,1892, &
& 1949,1954,2011,2017,2074,2081,2138,2146,2203,2212,2269, &
& 2279,2336,2347,2404,2416,2473,2486,2543,2557,2614,2629, &
& 2686,2702,2759,2776,2833,2851,2908,2927,2984,3004,3061, &
& 3082]
 index_cnj(101:200)=[&
& 3139,3161,3218,3241,3298,3322,3379,3404,3461,3487,3544, &
& 3571,3628,3656,3713,3742,3799,3829,3886,3917,3974,4006, &
& 4063,4096,4153,4187,4244,4279,4336,4372,4429,4466,4523, &
& 4561,4618,4656,4657,4714,4752,4753,4754,4811,4849,4850, &
& 4851,4852,4909,4947,4948,4949,4950,4951,5008,5046,5047, &
& 5048,5049,5050,5051,5108,5146,5147,5148,5149,5150,5151, &
& 5152,5209,5247,5248,5249,5250,5251,5252,5253,5254,5311, &
& 5349,5350,5351,5352,5353,5354,5355,5356,5357,5414,5452, &
& 5453,5454,5455,5456,5457,5458,5459,5460,5461,5518,5556, &
& 5557]
 index_cnj(201:300)=[&
& 5558,5559,5560,5561,5562,5563,5564,5565,5566,5623,5661, &
& 5662,5663,5664,5665,5666,5667,5668,5669,5670,5671,5672, &
& 5729,5767,5768,5769,5770,5771,5772,5773,5774,5775,5776, &
& 5777,5778,5779,5836,5874,5875,5876,5877,5878,5879,5880, &
& 5881,5882,5883,5884,5885,5886,5887,5944,5982,5983,5984, &
& 5985,5986,5987,5988,5989,5990,5991,5992,5993,5994,5995, &
& 5996,6053,6091,6092,6093,6094,6095,6096,6097,6098,6099, &
& 6100,6101,6102,6103,6104,6105,6106,6163,6201,6202,6203, &
& 6204,6205,6206,6207,6208,6209,6210,6211,6212,6213,6214, &
& 6215]
 index_cnj(301:400)=[&
& 6216,6217,6274,6312,6313,6314,6315,6316,6317,6318,6319, &
& 6320,6321,6322,6323,6324,6325,6326,6327,6328,6329,6386, &
& 6424,6425,6426,6427,6428,6429,6430,6431,6432,6433,6434, &
& 6435,6436,6437,6438,6439,6440,6441,6442,6499,6537,6538, &
& 6539,6540,6541,6542,6543,6544,6545,6546,6547,6548,6549, &
& 6550,6551,6552,6553,6554,6555,6556,6613,6651,6652,6653, &
& 6654,6655,6656,6657,6658,6659,6660,6661,6662,6663,6664, &
& 6665,6666,6667,6668,6669,6670,6671,6728,6766,6767,6768, &
& 6769,6770,6771,6772,6773,6774,6775,6776,6777,6778,6779, &
& 6780]
 index_cnj(401:500)=[&
& 6781,6782,6783,6784,6785,6786,6787,6844,6882,6883,6884, &
& 6885,6886,6887,6888,6889,6890,6891,6892,6893,6894,6895, &
& 6896,6897,6898,6899,6900,6901,6902,6903,6904,6961,6999, &
& 7000,7001,7002,7003,7004,7005,7006,7007,7008,7009,7010, &
& 7011,7012,7013,7014,7015,7016,7017,7018,7019,7020,7021, &
& 7022,7079,7117,7118,7119,7120,7121,7122,7123,7124,7125, &
& 7126,7127,7128,7129,7130,7131,7132,7133,7134,7135,7136, &
& 7137,7138,7139,7140,7141,7198,7236,7237,7238,7239,7240, &
& 7241,7242,7243,7244,7245,7246,7247,7248,7249,7250,7251, &
& 7252]
 index_cnj(501:600)=[&
& 7253,7254,7255,7256,7257,7258,7259,7260,7261,7318,7356, &
& 7357,7358,7359,7360,7361,7362,7363,7364,7365,7366,7367, &
& 7368,7369,7370,7371,7372,7373,7374,7375,7376,7377,7378, &
& 7379,7380,7381,7382,7439,7477,7478,7479,7480,7481,7482, &
& 7483,7484,7485,7486,7487,7488,7489,7490,7491,7492,7493, &
& 7494,7495,7496,7497,7498,7499,7500,7501,7502,7503,7504, &
& 7561,7599,7600,7601,7602,7603,7604,7605,7606,7607,7608, &
& 7609,7610,7611,7612,7613,7614,7615,7616,7617,7618,7619, &
& 7620,7621,7622,7623,7624,7625,7626,7627,7684,7722,7723, &
& 7724]
 index_cnj(601:700)=[&
& 7725,7726,7727,7728,7729,7730,7731,7732,7733,7734,7735, &
& 7736,7737,7738,7739,7740,7741,7742,7743,7744,7745,7746, &
& 7747,7748,7749,7750,7751,7808,7846,7847,7848,7849,7850, &
& 7851,7852,7853,7854,7855,7856,7857,7858,7859,7860,7861, &
& 7862,7863,7864,7865,7866,7867,7868,7869,7870,7871,7872, &
& 7873,7874,7875,7876,7933,7971,7972,7973,7974,7975,7976, &
& 7977,7978,7979,7980,7981,7982,7983,7984,7985,7986,7987, &
& 7988,7989,7990,7991,7992,7993,7994,7995,7996,7997,7998, &
& 7999,8000,8001,8002,8059,8097,8098,8099,8100,8101,8102, &
& 8103]
 index_cnj(701:800)=[&
& 8104,8105,8106,8107,8108,8109,8110,8111,8112,8113,8114, &
& 8115,8116,8117,8118,8119,8120,8121,8122,8123,8124,8125, &
& 8126,8127,8128,8129,8186,8224,8225,8226,8227,8228,8229, &
& 8230,8231,8232,8233,8234,8235,8236,8237,8238,8239,8240, &
& 8241,8242,8243,8244,8245,8246,8247,8248,8249,8250,8251, &
& 8252,8253,8254,8255,8256,8257,8314,8352,8353,8354,8355, &
& 8356,8357,8358,8359,8360,8361,8362,8363,8364,8365,8366, &
& 8367,8368,8369,8370,8371,8372,8373,8374,8375,8376,8377, &
& 8378,8379,8380,8381,8382,8383,8384,8385,8386,8443,8481, &
& 8482]
 index_cnj(801:900)=[&
& 8483,8484,8485,8486,8487,8488,8489,8490,8491,8492,8493, &
& 8494,8495,8496,8497,8498,8499,8500,8501,8502,8503,8504, &
& 8505,8506,8507,8508,8509,8510,8511,8512,8513,8514,8515, &
& 8516,8573,8611,8612,8613,8614,8615,8616,8617,8618,8619, &
& 8620,8621,8622,8623,8624,8625,8626,8627,8628,8629,8630, &
& 8631,8632,8633,8634,8635,8636,8637,8638,8639,8640,8641, &
& 8642,8643,8644,8645,8646,8647,8704,8742,8743,8744,8745, &
& 8746,8747,8748,8749,8750,8751,8752,8753,8754,8755,8756, &
& 8757,8758,8759,8760,8761,8762,8763,8764,8765,8766,8767, &
& 8768]
 index_cnj(901:1000)=[&
& 8769,8770,8771,8772,8773,8774,8775,8776,8777,8778,8779, &
& 8836,8874,8875,8876,8877,8878,8879,8880,8881,8882,8883, &
& 8884,8885,8886,8887,8888,8889,8890,8891,8892,8893,8894, &
& 8895,8896,8897,8898,8899,8900,8901,8902,8903,8904,8905, &
& 8906,8907,8908,8909,8910,8911,8912,8969,9007,9008,9009, &
& 9010,9011,9012,9013,9014,9015,9016,9017,9018,9019,9020, &
& 9021,9022,9023,9024,9025,9026,9027,9028,9029,9030,9031, &
& 9032,9033,9034,9035,9036,9037,9038,9039,9040,9041,9042, &
& 9043,9044,9045,9046,9103,9141,9142,9143,9144,9145,9146, &
& 9147]
 index_cnj(1001:1100)=[&
& 9148,9149,9150,9151,9152,9153,9154,9155,9156,9157,9158, &
& 9159,9160,9161,9162,9163,9164,9165,9166,9167,9168,9169, &
& 9170,9171,9172,9173,9174,9175,9176,9177,9178,9179,9180, &
& 9181,9238,9276,9277,9278,9279,9280,9281,9282,9283,9284, &
& 9285,9286,9287,9288,9289,9290,9291,9292,9293,9294,9295, &
& 9296,9297,9298,9299,9300,9301,9302,9303,9304,9305,9306, &
& 9307,9308,9309,9310,9311,9312,9313,9314,9315,9316,9317, &
& 9374,9412,9413,9414,9415,9416,9417,9418,9419,9420,9421, &
& 9422,9423,9424,9425,9426,9427,9428,9429,9430,9431,9432, &
& 9433]
 index_cnj(1101:1200)=[&
& 9434,9435,9436,9437,9438,9439,9440,9441,9442,9443,9444, &
& 9445,9446,9447,9448,9449,9450,9451,9452,9453,9454,9511, &
& 9549,9550,9551,9552,9553,9554,9555,9556,9557,9558,9559, &
& 9560,9561,9562,9563,9564,9565,9566,9567,9568,9569,9570, &
& 9571,9572,9573,9574,9575,9576,9577,9578,9579,9580,9581, &
& 9582,9583,9584,9585,9586,9587,9588,9589,9590,9591,9592, &
& 9649,9687,9688,9689,9690,9691,9692,9693,9694,9695,9696, &
& 9697,9698,9699,9700,9701,9702,9703,9704,9705,9706,9707, &
& 9708,9709,9710,9711,9712,9713,9714,9715,9716,9717,9718, &
& 9719]
 index_cnj(1201:1300)=[&
& 9720,9721,9722,9723,9724,9725,9726,9727,9728,9729,9730, &
& 9731,9788,9826,9827,9828,9829,9830,9831,9832,9833,9834, &
& 9835,9836,9837,9838,9839,9840,9841,9842,9843,9844,9845, &
& 9846,9847,9848,9849,9850,9851,9852,9853,9854,9855,9856, &
& 9857,9858,9859,9860,9861,9862,9863,9864,9865,9866,9867, &
& 9868,9869,9870,9871,9928,9966,9967,9968,9969,9970,9971, &
& 9972,9973,9974,9975,9976,9977,9978,9979,9980,9981,9982, &
& 9983,9984,9985,9986,9987,9988,9989,9990,9991,9992,9993, &
& 9994,9995,9996,9997,9998,9999,10000,10001,10002,10003,10004, &
& 10005]
 index_cnj(1301:1400)=[&
& 10006,10007,10008,10009,10010,10011,10012,10069,10107,10108,10109, &
& 10110,10111,10112,10113,10114,10115,10116,10117,10118,10119,10120, &
& 10121,10122,10123,10124,10125,10126,10127,10128,10129,10130,10131, &
& 10132,10133,10134,10135,10136,10137,10138,10139,10140,10141,10142, &
& 10143,10144,10145,10146,10147,10148,10149,10150,10151,10152,10153, &
& 10154,10211,10249,10250,10251,10252,10253,10254,10255,10256,10257, &
& 10258,10259,10260,10261,10262,10263,10264,10265,10266,10267,10268, &
& 10269,10270,10271,10272,10273,10274,10275,10276,10277,10278,10279, &
& 10280,10281,10282,10283,10284,10285,10286,10287,10288,10289,10290, &
& 10291]
 index_cnj(1401:1500)=[&
& 10292,10293,10294,10295,10296,10297,10354,10392,10393,10394,10395, &
& 10396,10397,10398,10399,10400,10401,10402,10403,10404,10405,10406, &
& 10407,10408,10409,10410,10411,10412,10413,10414,10415,10416,10417, &
& 10418,10419,10420,10421,10422,10423,10424,10425,10426,10427,10428, &
& 10429,10430,10431,10432,10433,10434,10435,10436,10437,10438,10439, &
& 10440,10441,10498,10536,10537,10538,10539,10540,10541,10542,10543, &
& 10544,10545,10546,10547,10548,10549,10550,10551,10552,10553,10554, &
& 10555,10556,10557,10558,10559,10560,10561,10562,10563,10564,10565, &
& 10566,10567,10568,10569,10570,10571,10572,10573,10574,10575,10576, &
& 10577]
 index_cnj(1501:1600)=[&
& 10578,10579,10580,10581,10582,10583,10584,10585,10586,10643,10681, &
& 10682,10683,10684,10685,10686,10687,10688,10689,10690,10691,10692, &
& 10693,10694,10695,10696,10697,10698,10699,10700,10701,10702,10703, &
& 10704,10705,10706,10707,10708,10709,10710,10711,10712,10713,10714, &
& 10715,10716,10717,10718,10719,10720,10721,10722,10723,10724,10725, &
& 10726,10727,10728,10729,10730,10731,10732,10789,10827,10828,10829, &
& 10830,10831,10832,10833,10834,10835,10836,10837,10838,10839,10840, &
& 10841,10842,10843,10844,10845,10846,10847,10848,10849,10850,10851, &
& 10852,10853,10854,10855,10856,10857,10858,10859,10860,10861,10862, &
& 10863]
 index_cnj(1601:1700)=[&
& 10864,10865,10866,10867,10868,10869,10870,10871,10872,10873,10874, &
& 10875,10876,10877,10878,10879,10936,10974,10975,10976,10977,10978, &
& 10979,10980,10981,10982,10983,10984,10985,10986,10987,10988,10989, &
& 10990,10991,10992,10993,10994,10995,10996,10997,10998,10999,11000, &
& 11001,11002,11003,11004,11005,11006,11007,11008,11009,11010,11011, &
& 11012,11013,11014,11015,11016,11017,11018,11019,11020,11021,11022, &
& 11023,11024,11025,11026,11027,11084,11122,11123,11124,11125,11126, &
& 11127,11128,11129,11130,11131,11132,11133,11134,11135,11136,11137, &
& 11138,11139,11140,11141,11142,11143,11144,11145,11146,11147,11148, &
& 11149]
 index_cnj(1701:1800)=[&
& 11150,11151,11152,11153,11154,11155,11156,11157,11158,11159,11160, &
& 11161,11162,11163,11164,11165,11166,11167,11168,11169,11170,11171, &
& 11172,11173,11174,11175,11176,11233,11271,11272,11273,11274,11275, &
& 11276,11277,11278,11279,11280,11281,11282,11283,11284,11285,11286, &
& 11287,11288,11289,11290,11291,11292,11293,11294,11295,11296,11297, &
& 11298,11299,11300,11301,11302,11303,11304,11305,11306,11307,11308, &
& 11309,11310,11311,11312,11313,11314,11315,11316,11317,11318,11319, &
& 11320,11321,11322,11323,11324,11325,11326,11383,11421,11422,11423, &
& 11424,11425,11426,11427,11428,11429,11430,11431,11432,11433,11434, &
& 11435]
 index_cnj(1801:1900)=[&
& 11436,11437,11438,11439,11440,11441,11442,11443,11444,11445,11446, &
& 11447,11448,11449,11450,11451,11452,11453,11454,11455,11456,11457, &
& 11458,11459,11460,11461,11462,11463,11464,11465,11466,11467,11468, &
& 11469,11470,11471,11472,11473,11474,11475,11476,11477,11534,11572, &
& 11573,11574,11575,11576,11577,11578,11579,11580,11581,11582,11583, &
& 11584,11585,11586,11587,11588,11589,11590,11591,11592,11593,11594, &
& 11595,11596,11597,11598,11599,11600,11601,11602,11603,11604,11605, &
& 11606,11607,11608,11609,11610,11611,11612,11613,11614,11615,11616, &
& 11617,11618,11619,11620,11621,11622,11623,11624,11625,11626,11627, &
& 11628]
 index_cnj(1901:2000)=[&
& 11629,11686,11724,11725,11726,11727,11728,11729,11730,11731,11732, &
& 11733,11734,11735,11736,11737,11738,11739,11740,11741,11742,11743, &
& 11744,11745,11746,11747,11748,11749,11750,11751,11752,11753,11754, &
& 11755,11756,11757,11758,11759,11760,11761,11762,11763,11764,11765, &
& 11766,11767,11768,11769,11770,11771,11772,11773,11774,11775,11776, &
& 11777,11778,11779,11780,11781,11782,11839,11877,11878,11879,11880, &
& 11881,11882,11883,11884,11885,11886,11887,11888,11889,11890,11891, &
& 11892,11893,11894,11895,11896,11897,11898,11899,11900,11901,11902, &
& 11903,11904,11905,11906,11907,11908,11909,11910,11911,11912,11913, &
& 11914]
 index_cnj(2001:2100)=[&
& 11915,11916,11917,11918,11919,11920,11921,11922,11923,11924,11925, &
& 11926,11927,11928,11929,11930,11931,11932,11933,11934,11935,11936, &
& 11993,12031,12032,12033,12034,12035,12036,12037,12038,12039,12040, &
& 12041,12042,12043,12044,12045,12046,12047,12048,12049,12050,12051, &
& 12052,12053,12054,12055,12056,12057,12058,12059,12060,12061,12062, &
& 12063,12064,12065,12066,12067,12068,12069,12070,12071,12072,12073, &
& 12074,12075,12076,12077,12078,12079,12080,12081,12082,12083,12084, &
& 12085,12086,12087,12088,12089,12090,12091,12148,12186,12187,12188, &
& 12189,12190,12191,12192,12193,12194,12195,12196,12197,12198,12199, &
& 12200]
 index_cnj(2101:2200)=[&
& 12201,12202,12203,12204,12205,12206,12207,12208,12209,12210,12211, &
& 12212,12213,12214,12215,12216,12217,12218,12219,12220,12221,12222, &
& 12223,12224,12225,12226,12227,12228,12229,12230,12231,12232,12233, &
& 12234,12235,12236,12237,12238,12239,12240,12241,12242,12243,12244, &
& 12245,12246,12247,12304,12342,12343,12344,12345,12346,12347,12348, &
& 12349,12350,12351,12352,12353,12354,12355,12356,12357,12358,12359, &
& 12360,12361,12362,12363,12364,12365,12366,12367,12368,12369,12370, &
& 12371,12372,12373,12374,12375,12376,12377,12378,12379,12380,12381, &
& 12382,12383,12384,12385,12386,12387,12388,12389,12390,12391,12392, &
& 12393]
 index_cnj(2201:2300)=[&
& 12394,12395,12396,12397,12398,12399,12400,12401,12402,12403,12404, &
& 12461,12499,12500,12501,12502,12503,12504,12505,12506,12507,12508, &
& 12509,12510,12511,12512,12513,12514,12515,12516,12517,12518,12519, &
& 12520,12521,12522,12523,12524,12525,12526,12527,12528,12529,12530, &
& 12531,12532,12533,12534,12535,12536,12537,12538,12539,12540,12541, &
& 12542,12543,12544,12545,12546,12547,12548,12549,12550,12551,12552, &
& 12553,12554,12555,12556,12557,12558,12559,12560,12561,12562,12619, &
& 12657,12658,12659,12660,12661,12662,12663,12664,12665,12666,12667, &
& 12668,12669,12670,12671,12672,12673,12674,12675,12676,12677,12678, &
& 12679]
 index_cnj(2301:2400)=[&
& 12680,12681,12682,12683,12684,12685,12686,12687,12688,12689,12690, &
& 12691,12692,12693,12694,12695,12696,12697,12698,12699,12700,12701, &
& 12702,12703,12704,12705,12706,12707,12708,12709,12710,12711,12712, &
& 12713,12714,12715,12716,12717,12718,12719,12720,12721,12778,12816, &
& 12817,12818,12819,12820,12821,12822,12823,12824,12825,12826,12827, &
& 12828,12829,12830,12831,12832,12833,12834,12835,12836,12837,12838, &
& 12839,12840,12841,12842,12843,12844,12845,12846,12847,12848,12849, &
& 12850,12851,12852,12853,12854,12855,12856,12857,12858,12859,12860, &
& 12861,12862,12863,12864,12865,12866,12867,12868,12869,12870,12871, &
& 12872]
 index_cnj(2401:2500)=[&
& 12873,12874,12875,12876,12877,12878,12879,12880,12881,12938,12976, &
& 12977,12978,12979,12980,12981,12982,12983,12984,12985,12986,12987, &
& 12988,12989,12990,12991,12992,12993,12994,12995,12996,12997,12998, &
& 12999,13000,13001,13002,13003,13004,13005,13006,13007,13008,13009, &
& 13010,13011,13012,13013,13014,13015,13016,13017,13018,13019,13020, &
& 13021,13022,13023,13024,13025,13026,13027,13028,13029,13030,13031, &
& 13032,13033,13034,13035,13036,13037,13038,13039,13040,13041,13042, &
& 13099,13137,13138,13139,13140,13141,13142,13143,13144,13145,13146, &
& 13147,13148,13149,13150,13151,13152,13153,13154,13155,13156,13157, &
& 13158]
 index_cnj(2501:2600)=[&
& 13159,13160,13161,13162,13163,13164,13165,13166,13167,13168,13169, &
& 13170,13171,13172,13173,13174,13175,13176,13177,13178,13179,13180, &
& 13181,13182,13183,13184,13185,13186,13187,13188,13189,13190,13191, &
& 13192,13193,13194,13195,13196,13197,13198,13199,13200,13201,13202, &
& 13203,13204,13261,13299,13300,13301,13302,13303,13304,13305,13306, &
& 13307,13308,13309,13310,13311,13312,13313,13314,13315,13316,13317, &
& 13318,13319,13320,13321,13322,13323,13324,13325,13326,13327,13328, &
& 13329,13330,13331,13332,13333,13334,13335,13336,13337,13338,13339, &
& 13340,13341,13342,13343,13344,13345,13346,13347,13348,13349,13350, &
& 13351]
 index_cnj(2601:2700)=[&
& 13352,13353,13354,13355,13356,13357,13358,13359,13360,13361,13362, &
& 13363,13364,13365,13366,13367,13424,13462,13463,13464,13465,13466, &
& 13467,13468,13469,13470,13471,13472,13473,13474,13475,13476,13477, &
& 13478,13479,13480,13481,13482,13483,13484,13485,13486,13487,13488, &
& 13489,13490,13491,13492,13493,13494,13495,13496,13497,13498,13499, &
& 13500,13501,13502,13503,13504,13505,13506,13507,13508,13509,13510, &
& 13511,13512,13513,13514,13515,13516,13517,13518,13519,13520,13521, &
& 13522,13523,13524,13525,13526,13527,13528,13529,13530,13531,13588, &
& 13626,13627,13628,13629,13630,13631,13632,13633,13634,13635,13636, &
& 13637]
 index_cnj(2701:2800)=[&
& 13638,13639,13640,13641,13642,13643,13644,13645,13646,13647,13648, &
& 13649,13650,13651,13652,13653,13654,13655,13656,13657,13658,13659, &
& 13660,13661,13662,13663,13664,13665,13666,13667,13668,13669,13670, &
& 13671,13672,13673,13674,13675,13676,13677,13678,13679,13680,13681, &
& 13682,13683,13684,13685,13686,13687,13688,13689,13690,13691,13692, &
& 13693,13694,13695,13696,13753,13791,13792,13793,13794,13795,13796, &
& 13797,13798,13799,13800,13801,13802,13803,13804,13805,13806,13807, &
& 13808,13809,13810,13811,13812,13813,13814,13815,13816,13817,13818, &
& 13819,13820,13821,13822,13823,13824,13825,13826,13827,13828,13829, &
& 13830]
 index_cnj(2801:2900)=[&
& 13831,13832,13833,13834,13835,13836,13837,13838,13839,13840,13841, &
& 13842,13843,13844,13845,13846,13847,13848,13849,13850,13851,13852, &
& 13853,13854,13855,13856,13857,13858,13859,13860,13861,13862,13919, &
& 13957,13958,13959,13960,13961,13962,13963,13964,13965,13966,13967, &
& 13968,13969,13970,13971,13972,13973,13974,13975,13976,13977,13978, &
& 13979,13980,13981,13982,13983,13984,13985,13986,13987,13988,13989, &
& 13990,13991,13992,13993,13994,13995,13996,13997,13998,13999,14000, &
& 14001,14002,14003,14004,14005,14006,14007,14008,14009,14010,14011, &
& 14012,14013,14014,14015,14016,14017,14018,14019,14020,14021,14022, &
& 14023]
 index_cnj(2901:3000)=[&
& 14024,14025,14026,14027,14028,14029,14086,14124,14125,14126,14127, &
& 14128,14129,14130,14131,14132,14133,14134,14135,14136,14137,14138, &
& 14139,14140,14141,14142,14143,14144,14145,14146,14147,14148,14149, &
& 14150,14151,14152,14153,14154,14155,14156,14157,14158,14159,14160, &
& 14161,14162,14163,14164,14165,14166,14167,14168,14169,14170,14171, &
& 14172,14173,14174,14175,14176,14177,14178,14179,14180,14181,14182, &
& 14183,14184,14185,14186,14187,14188,14189,14190,14191,14192,14193, &
& 14194,14195,14196,14197,14254,14292,14293,14294,14295,14296,14297, &
& 14298,14299,14300,14301,14302,14303,14304,14305,14306,14307,14308, &
& 14309]
 index_cnj(3001:3100)=[&
& 14310,14311,14312,14313,14314,14315,14316,14317,14318,14319,14320, &
& 14321,14322,14323,14324,14325,14326,14327,14328,14329,14330,14331, &
& 14332,14333,14334,14335,14336,14337,14338,14339,14340,14341,14342, &
& 14343,14344,14345,14346,14347,14348,14349,14350,14351,14352,14353, &
& 14354,14355,14356,14357,14358,14359,14360,14361,14362,14363,14364, &
& 14365,14366,14423,14461,14462,14463,14464,14465,14466,14467,14468, &
& 14469,14470,14471,14472,14473,14474,14475,14476,14477,14478,14479, &
& 14480,14481,14482,14483,14484,14485,14486,14487,14488,14489,14490, &
& 14491,14492,14493,14494,14495,14496,14497,14498,14499,14500,14501, &
& 14502]
 index_cnj(3101:3200)=[&
& 14503,14504,14505,14506,14507,14508,14509,14510,14511,14512,14513, &
& 14514,14515,14516,14517,14518,14519,14520,14521,14522,14523,14524, &
& 14525,14526,14527,14528,14529,14530,14531,14532,14533,14534,14535, &
& 14536,14593,14631,14632,14633,14634,14635,14636,14637,14638,14639, &
& 14640,14641,14642,14643,14644,14645,14646,14647,14648,14649,14650, &
& 14651,14652,14653,14654,14655,14656,14657,14658,14659,14660,14661, &
& 14662,14663,14664,14665,14666,14667,14668,14669,14670,14671,14672, &
& 14673,14674,14675,14676,14677,14678,14679,14680,14681,14682,14683, &
& 14684,14685,14686,14687,14688,14689,14690,14691,14692,14693,14694, &
& 14695]
 index_cnj(3201:3300)=[&
& 14696,14697,14698,14699,14700,14701,14702,14703,14704,14705,14706, &
& 14707,14764,14802,14803,14804,14805,14806,14807,14808,14809,14810, &
& 14811,14812,14813,14814,14815,14816,14817,14818,14819,14820,14821, &
& 14822,14823,14824,14825,14826,14827,14828,14829,14830,14831,14832, &
& 14833,14834,14835,14836,14837,14838,14839,14840,14841,14842,14843, &
& 14844,14845,14846,14847,14848,14849,14850,14851,14852,14853,14854, &
& 14855,14856,14857,14858,14859,14860,14861,14862,14863,14864,14865, &
& 14866,14867,14868,14869,14870,14871,14872,14873,14874,14875,14876, &
& 14877,14878,14879,14936,14974,14975,14976,14977,14978,14979,14980, &
& 14981]
 index_cnj(3301:3400)=[&
& 14982,14983,14984,14985,14986,14987,14988,14989,14990,14991,14992, &
& 14993,14994,14995,14996,14997,14998,14999,15000,15001,15002,15003, &
& 15004,15005,15006,15007,15008,15009,15010,15011,15012,15013,15014, &
& 15015,15016,15017,15018,15019,15020,15021,15022,15023,15024,15025, &
& 15026,15027,15028,15029,15030,15031,15032,15033,15034,15035,15036, &
& 15037,15038,15039,15040,15041,15042,15043,15044,15045,15046,15047, &
& 15048,15049,15050,15051,15052,15109,15147,15148,15149,15150,15151, &
& 15152,15153,15154,15155,15156,15157,15158,15159,15160,15161,15162, &
& 15163,15164,15165,15166,15167,15168,15169,15170,15171,15172,15173, &
& 15174]
 index_cnj(3401:3500)=[&
& 15175,15176,15177,15178,15179,15180,15181,15182,15183,15184,15185, &
& 15186,15187,15188,15189,15190,15191,15192,15193,15194,15195,15196, &
& 15197,15198,15199,15200,15201,15202,15203,15204,15205,15206,15207, &
& 15208,15209,15210,15211,15212,15213,15214,15215,15216,15217,15218, &
& 15219,15220,15221,15222,15223,15224,15225,15226,15283,15321,15322, &
& 15323,15324,15325,15326,15327,15328,15329,15330,15331,15332,15333, &
& 15334,15335,15336,15337,15338,15339,15340,15341,15342,15343,15344, &
& 15345,15346,15347,15348,15349,15350,15351,15352,15353,15354,15355, &
& 15356,15357,15358,15359,15360,15361,15362,15363,15364,15365,15366, &
& 15367]
 index_cnj(3501:3600)=[&
& 15368,15369,15370,15371,15372,15373,15374,15375,15376,15377,15378, &
& 15379,15380,15381,15382,15383,15384,15385,15386,15387,15388,15389, &
& 15390,15391,15392,15393,15394,15395,15396,15397,15398,15399,15400, &
& 15401,15458,15496,15497,15498,15499,15500,15501,15502,15503,15504, &
& 15505,15506,15507,15508,15509,15510,15511,15512,15513,15514,15515, &
& 15516,15517,15518,15519,15520,15521,15522,15523,15524,15525,15526, &
& 15527,15528,15529,15530,15531,15532,15533,15534,15535,15536,15537, &
& 15538,15539,15540,15541,15542,15543,15544,15545,15546,15547,15548, &
& 15549,15550,15551,15552,15553,15554,15555,15556,15557,15558,15559, &
& 15560]
 index_cnj(3601:3700)=[&
& 15561,15562,15563,15564,15565,15566,15567,15568,15569,15570,15571, &
& 15572,15573,15574,15575,15576,15577,15634,15672,15673,15674,15675, &
& 15676,15677,15678,15679,15680,15681,15682,15683,15684,15685,15686, &
& 15687,15688,15689,15690,15691,15692,15693,15694,15695,15696,15697, &
& 15698,15699,15700,15701,15702,15703,15704,15705,15706,15707,15708, &
& 15709,15710,15711,15712,15713,15714,15715,15716,15717,15718,15719, &
& 15720,15721,15722,15723,15724,15725,15726,15727,15728,15729,15730, &
& 15731,15732,15733,15734,15735,15736,15737,15738,15739,15740,15741, &
& 15742,15743,15744,15745,15746,15747,15748,15749,15750,15751,15752, &
& 15753]
 index_cnj(3701:3800)=[&
& 15754,15811,15849,15850,15851,15852,15853,15854,15855,15856,15857, &
& 15858,15859,15860,15861,15862,15863,15864,15865,15866,15867,15868, &
& 15869,15870,15871,15872,15873,15874,15875,15876,15877,15878,15879, &
& 15880,15881,15882,15883,15884,15885,15886,15887,15888,15889,15890, &
& 15891,15892,15893,15894,15895,15896,15897,15898,15899,15900,15901, &
& 15902,15903,15904,15905,15906,15907,15908,15909,15910,15911,15912, &
& 15913,15914,15915,15916,15917,15918,15919,15920,15921,15922,15923, &
& 15924,15925,15926,15927,15928,15929,15930,15931,15932,15989,16027, &
& 16028,16029,16030,16031,16032,16033,16034,16035,16036,16037,16038, &
& 16039]
 index_cnj(3801:3900)=[&
& 16040,16041,16042,16043,16044,16045,16046,16047,16048,16049,16050, &
& 16051,16052,16053,16054,16055,16056,16057,16058,16059,16060,16061, &
& 16062,16063,16064,16065,16066,16067,16068,16069,16070,16071,16072, &
& 16073,16074,16075,16076,16077,16078,16079,16080,16081,16082,16083, &
& 16084,16085,16086,16087,16088,16089,16090,16091,16092,16093,16094, &
& 16095,16096,16097,16098,16099,16100,16101,16102,16103,16104,16105, &
& 16106,16107,16108,16109,16110,16111,16168,16206,16207,16208,16209, &
& 16210,16211,16212,16213,16214,16215,16216,16217,16218,16219,16220, &
& 16221,16222,16223,16224,16225,16226,16227,16228,16229,16230,16231, &
& 16232]
 index_cnj(3901:4000)=[&
& 16233,16234,16235,16236,16237,16238,16239,16240,16241,16242,16243, &
& 16244,16245,16246,16247,16248,16249,16250,16251,16252,16253,16254, &
& 16255,16256,16257,16258,16259,16260,16261,16262,16263,16264,16265, &
& 16266,16267,16268,16269,16270,16271,16272,16273,16274,16275,16276, &
& 16277,16278,16279,16280,16281,16282,16283,16284,16285,16286,16287, &
& 16288,16289,16290,16291,16348,16386,16387,16388,16389,16390,16391, &
& 16392,16393,16394,16395,16396,16397,16398,16399,16400,16401,16402, &
& 16403,16404,16405,16406,16407,16408,16409,16410,16411,16412,16413, &
& 16414,16415,16416,16417,16418,16419,16420,16421,16422,16423,16424, &
& 16425]
 index_cnj(4001:4100)=[&
& 16426,16427,16428,16429,16430,16431,16432,16433,16434,16435,16436, &
& 16437,16438,16439,16440,16441,16442,16443,16444,16445,16446,16447, &
& 16448,16449,16450,16451,16452,16453,16454,16455,16456,16457,16458, &
& 16459,16460,16461,16462,16463,16464,16465,16466,16467,16468,16469, &
& 16470,16471,16472,16529,16567,16568,16569,16570,16571,16572,16573, &
& 16574,16575,16576,16577,16578,16579,16580,16581,16582,16583,16584, &
& 16585,16586,16587,16588,16589,16590,16591,16592,16593,16594,16595, &
& 16596,16597,16598,16599,16600,16601,16602,16603,16604,16605,16606, &
& 16607,16608,16609,16610,16611,16612,16613,16614,16615,16616,16617, &
& 16618]
 index_cnj(4101:4200)=[&
& 16619,16620,16621,16622,16623,16624,16625,16626,16627,16628,16629, &
& 16630,16631,16632,16633,16634,16635,16636,16637,16638,16639,16640, &
& 16641,16642,16643,16644,16645,16646,16647,16648,16649,16650,16651, &
& 16652,16653,16654,16711,16749,16750,16751,16752,16753,16754,16755, &
& 16756,16757,16758,16759,16760,16761,16762,16763,16764,16765,16766, &
& 16767,16768,16769,16770,16771,16772,16773,16774,16775,16776,16777, &
& 16778,16779,16780,16781,16782,16783,16784,16785,16786,16787,16788, &
& 16789,16790,16791,16792,16793,16794,16795,16796,16797,16798,16799, &
& 16800,16801,16802,16803,16804,16805,16806,16807,16808,16809,16810, &
& 16811]
 index_cnj(4201:4300)=[&
& 16812,16813,16814,16815,16816,16817,16818,16819,16820,16821,16822, &
& 16823,16824,16825,16826,16827,16828,16829,16830,16831,16832,16833, &
& 16834,16835,16836,16837,16894,16932,16933,16934,16935,16936,16937, &
& 16938,16939,16940,16941,16942,16943,16944,16945,16946,16947,16948, &
& 16949,16950,16951,16952,16953,16954,16955,16956,16957,16958,16959, &
& 16960,16961,16962,16963,16964,16965,16966,16967,16968,16969,16970, &
& 16971,16972,16973,16974,16975,16976,16977,16978,16979,16980,16981, &
& 16982,16983,16984,16985,16986,16987,16988,16989,16990,16991,16992, &
& 16993,16994,16995,16996,16997,16998,16999,17000,17001,17002,17003, &
& 17004]
 index_cnj(4301:4400)=[&
& 17005,17006,17007,17008,17009,17010,17011,17012,17013,17014,17015, &
& 17016,17017,17018,17019,17020,17021,17078,17116,17117,17118,17119, &
& 17120,17121,17122,17123,17124,17125,17126,17127,17128,17129,17130, &
& 17131,17132,17133,17134,17135,17136,17137,17138,17139,17140,17141, &
& 17142,17143,17144,17145,17146,17147,17148,17149,17150,17151,17152, &
& 17153,17154,17155,17156,17157,17158,17159,17160,17161,17162,17163, &
& 17164,17165,17166,17167,17168,17169,17170,17171,17172,17173,17174, &
& 17175,17176,17177,17178,17179,17180,17181,17182,17183,17184,17185, &
& 17186,17187,17188,17189,17190,17191,17192,17193,17194,17195,17196, &
& 17197]
 index_cnj(4401:4500)=[&
& 17198,17199,17200,17201,17202,17203,17204,17205,17206,17263,17301, &
& 17302,17303,17304,17305,17306,17307,17308,17309,17310,17311,17312, &
& 17313,17314,17315,17316,17317,17318,17319,17320,17321,17322,17323, &
& 17324,17325,17326,17327,17328,17329,17330,17331,17332,17333,17334, &
& 17335,17336,17337,17338,17339,17340,17341,17342,17343,17344,17345, &
& 17346,17347,17348,17349,17350,17351,17352,17353,17354,17355,17356, &
& 17357,17358,17359,17360,17361,17362,17363,17364,17365,17366,17367, &
& 17368,17369,17370,17371,17372,17373,17374,17375,17376,17377,17378, &
& 17379,17380,17381,17382,17383,17384,17385,17386,17387,17388,17389, &
& 17390]
 index_cnj(4501:4600)=[&
& 17391,17392,17449,17487,17488,17489,17490,17491,17492,17493,17494, &
& 17495,17496,17497,17498,17499,17500,17501,17502,17503,17504,17505, &
& 17506,17507,17508,17509,17510,17511,17512,17513,17514,17515,17516, &
& 17517,17518,17519,17520,17521,17522,17523,17524,17525,17526,17527, &
& 17528,17529,17530,17531,17532,17533,17534,17535,17536,17537,17538, &
& 17539,17540,17541,17542,17543,17544,17545,17546,17547,17548,17549, &
& 17550,17551,17552,17553,17554,17555,17556,17557,17558,17559,17560, &
& 17561,17562,17563,17564,17565,17566,17567,17568,17569,17570,17571, &
& 17572,17573,17574,17575,17576,17577,17578,17579,17636,17674,17675, &
& 17676]
 index_cnj(4601:4700)=[&
& 17677,17678,17679,17680,17681,17682,17683,17684,17685,17686,17687, &
& 17688,17689,17690,17691,17692,17693,17694,17695,17696,17697,17698, &
& 17699,17700,17701,17702,17703,17704,17705,17706,17707,17708,17709, &
& 17710,17711,17712,17713,17714,17715,17716,17717,17718,17719,17720, &
& 17721,17722,17723,17724,17725,17726,17727,17728,17729,17730,17731, &
& 17732,17733,17734,17735,17736,17737,17738,17739,17740,17741,17742, &
& 17743,17744,17745,17746,17747,17748,17749,17750,17751,17752,17753, &
& 17754,17755,17756,17757,17758,17759,17760,17761,17762,17763,17764, &
& 17765,17766,17767,17824,17862,17863,17864,17865,17866,17867,17868, &
& 17869]
 index_cnj(4701:4800)=[&
& 17870,17871,17872,17873,17874,17875,17876,17877,17878,17879,17880, &
& 17881,17882,17883,17884,17885,17886,17887,17888,17889,17890,17891, &
& 17892,17893,17894,17895,17896,17897,17898,17899,17900,17901,17902, &
& 17903,17904,17905,17906,17907,17908,17909,17910,17911,17912,17913, &
& 17914,17915,17916,17917,17918,17919,17920,17921,17922,17923,17924, &
& 17925,17926,17927,17928,17929,17930,17931,17932,17933,17934,17935, &
& 17936,17937,17938,17939,17940,17941,17942,17943,17944,17945,17946, &
& 17947,17948,17949,17950,17951,17952,17953,17954,17955,17956,18013, &
& 18051,18052,18053,18054,18055,18056,18057,18058,18059,18060,18061, &
& 18062]
 index_cnj(4801:4900)=[&
& 18063,18064,18065,18066,18067,18068,18069,18070,18071,18072,18073, &
& 18074,18075,18076,18077,18078,18079,18080,18081,18082,18083,18084, &
& 18085,18086,18087,18088,18089,18090,18091,18092,18093,18094,18095, &
& 18096,18097,18098,18099,18100,18101,18102,18103,18104,18105,18106, &
& 18107,18108,18109,18110,18111,18112,18113,18114,18115,18116,18117, &
& 18118,18119,18120,18121,18122,18123,18124,18125,18126,18127,18128, &
& 18129,18130,18131,18132,18133,18134,18135,18136,18137,18138,18139, &
& 18140,18141,18142,18143,18144,18145,18146,18203,18241,18242,18243, &
& 18244,18245,18246,18247,18248,18249,18250,18251,18252,18253,18254, &
& 18255]
 index_cnj(4901:5000)=[&
& 18256,18257,18258,18259,18260,18261,18262,18263,18264,18265,18266, &
& 18267,18268,18269,18270,18271,18272,18273,18274,18275,18276,18277, &
& 18278,18279,18280,18281,18282,18283,18284,18285,18286,18287,18288, &
& 18289,18290,18291,18292,18293,18294,18295,18296,18297,18298,18299, &
& 18300,18301,18302,18303,18304,18305,18306,18307,18308,18309,18310, &
& 18311,18312,18313,18314,18315,18316,18317,18318,18319,18320,18321, &
& 18322,18323,18324,18325,18326,18327,18328,18329,18330,18331,18332, &
& 18333,18334,18335,18336,18337,18394,18432,18433,18434,18435,18436, &
& 18437,18438,18439,18440,18441,18442,18443,18444,18445,18446,18447, &
& 18448]
 index_cnj(5001:5100)=[&
& 18449,18450,18451,18452,18453,18454,18455,18456,18457,18458,18459, &
& 18460,18461,18462,18463,18464,18465,18466,18467,18468,18469,18470, &
& 18471,18472,18473,18474,18475,18476,18477,18478,18479,18480,18481, &
& 18482,18483,18484,18485,18486,18487,18488,18489,18490,18491,18492, &
& 18493,18494,18495,18496,18497,18498,18499,18500,18501,18502,18503, &
& 18504,18505,18506,18507,18508,18509,18510,18511,18512,18513,18514, &
& 18515,18516,18517,18518,18519,18520,18521,18522,18523,18524,18525, &
& 18526,18527,18528,18529,18586,18624,18625,18626,18627,18628,18629, &
& 18630,18631,18632,18633,18634,18635,18636,18637,18638,18639,18640, &
& 18641]
 index_cnj(5101:5200)=[&
& 18642,18643,18644,18645,18646,18647,18648,18649,18650,18651,18652, &
& 18653,18654,18655,18656,18657,18658,18659,18660,18661,18662,18663, &
& 18664,18665,18666,18667,18668,18669,18670,18671,18672,18673,18674, &
& 18675,18676,18677,18678,18679,18680,18681,18682,18683,18684,18685, &
& 18686,18687,18688,18689,18690,18691,18692,18693,18694,18695,18696, &
& 18697,18698,18699,18700,18701,18702,18703,18704,18705,18706,18707, &
& 18708,18709,18710,18711,18712,18713,18714,18715,18716,18717,18718, &
& 18719,18720,18721,18722,18779,18817,18818,18819,18820,18821,18822, &
& 18823,18824,18825,18826,18827,18828,18829,18830,18831,18832,18833, &
& 18834]
 index_cnj(5201:5300)=[&
& 18835,18836,18837,18838,18839,18840,18841,18842,18843,18844,18845, &
& 18846,18847,18848,18849,18850,18851,18852,18853,18854,18855,18856, &
& 18857,18858,18859,18860,18861,18862,18863,18864,18865,18866,18867, &
& 18868,18869,18870,18871,18872,18873,18874,18875,18876,18877,18878, &
& 18879,18880,18881,18882,18883,18884,18885,18886,18887,18888,18889, &
& 18890,18891,18892,18893,18894,18895,18896,18897,18898,18899,18900, &
& 18901,18902,18903,18904,18905,18906,18907,18908,18909,18910,18911, &
& 18912,18913,18914,18915,18916,18973,19011,19012,19013,19014,19015, &
& 19016,19017,19018,19019,19020,19021,19022,19023,19024,19025,19026, &
& 19027]
 index_cnj(5301:5400)=[&
& 19028,19029,19030,19031,19032,19033,19034,19035,19036,19037,19038, &
& 19039,19040,19041,19042,19043,19044,19045,19046,19047,19048,19049, &
& 19050,19051,19052,19053,19054,19055,19056,19057,19058,19059,19060, &
& 19061,19062,19063,19064,19065,19066,19067,19068,19069,19070,19071, &
& 19072,19073,19074,19075,19076,19077,19078,19079,19080,19081,19082, &
& 19083,19084,19085,19086,19087,19088,19089,19090,19091,19092,19093, &
& 19094,19095,19096,19097,19098,19099,19100,19101,19102,19103,19104, &
& 19105,19106,19107,19108,19109,19110,19111,19168,19206,19207,19208, &
& 19209,19210,19211,19212,19213,19214,19215,19216,19217,19218,19219, &
& 19220]
 index_cnj(5401:5500)=[&
& 19221,19222,19223,19224,19225,19226,19227,19228,19229,19230,19231, &
& 19232,19233,19234,19235,19236,19237,19238,19239,19240,19241,19242, &
& 19243,19244,19245,19246,19247,19248,19249,19250,19251,19252,19253, &
& 19254,19255,19256,19257,19258,19259,19260,19261,19262,19263,19264, &
& 19265,19266,19267,19268,19269,19270,19271,19272,19273,19274,19275, &
& 19276,19277,19278,19279,19280,19281,19282,19283,19284,19285,19286, &
& 19287,19288,19289,19290,19291,19292,19293,19294,19295,19296,19297, &
& 19298,19299,19300,19301,19302,19303,19304,19305,19306,19307,19364, &
& 19402,19403,19404,19405,19406,19407,19408,19409,19410,19411,19412, &
& 19413]
 index_cnj(5501:5600)=[&
& 19414,19415,19416,19417,19418,19419,19420,19421,19422,19423,19424, &
& 19425,19426,19427,19428,19429,19430,19431,19432,19433,19434,19435, &
& 19436,19437,19438,19439,19440,19441,19442,19443,19444,19445,19446, &
& 19447,19448,19449,19450,19451,19452,19453,19454,19455,19456,19457, &
& 19458,19459,19460,19461,19462,19463,19464,19465,19466,19467,19468, &
& 19469,19470,19471,19472,19473,19474,19475,19476,19477,19478,19479, &
& 19480,19481,19482,19483,19484,19485,19486,19487,19488,19489,19490, &
& 19491,19492,19493,19494,19495,19496,19497,19498,19499,19500,19501, &
& 19502,19503,19504,19561,19599,19600,19601,19602,19603,19604,19605, &
& 19606]
 index_cnj(5601:5700)=[&
& 19607,19608,19609,19610,19611,19612,19613,19614,19615,19616,19617, &
& 19618,19619,19620,19621,19622,19623,19624,19625,19626,19627,19628, &
& 19629,19630,19631,19632,19633,19634,19635,19636,19637,19638,19639, &
& 19640,19641,19642,19643,19644,19645,19646,19647,19648,19649,19650, &
& 19651,19652,19653,19654,19655,19656,19657,19658,19659,19660,19661, &
& 19662,19663,19664,19665,19666,19667,19668,19669,19670,19671,19672, &
& 19673,19674,19675,19676,19677,19678,19679,19680,19681,19682,19683, &
& 19684,19685,19686,19687,19688,19689,19690,19691,19692,19693,19694, &
& 19695,19696,19697,19698,19699,19700,19701,19702,19759,19797,19798, &
& 19799]
 index_cnj(5701:5800)=[&
& 19800,19801,19802,19803,19804,19805,19806,19807,19808,19809,19810, &
& 19811,19812,19813,19814,19815,19816,19817,19818,19819,19820,19821, &
& 19822,19823,19824,19825,19826,19827,19828,19829,19830,19831,19832, &
& 19833,19834,19835,19836,19837,19838,19839,19840,19841,19842,19843, &
& 19844,19845,19846,19847,19848,19849,19850,19851,19852,19853,19854, &
& 19855,19856,19857,19858,19859,19860,19861,19862,19863,19864,19865, &
& 19866,19867,19868,19869,19870,19871,19872,19873,19874,19875,19876, &
& 19877,19878,19879,19880,19881,19882,19883,19884,19885,19886,19887, &
& 19888,19889,19890,19891,19892,19893,19894,19895,19896,19897,19898, &
& 19899]
 index_cnj(5801:5900)=[&
& 19900,19901,19958,19996,19997,19998,19999,20000,20001,20002,20003, &
& 20004,20005,20006,20007,20008,20009,20010,20011,20012,20013,20014, &
& 20015,20016,20017,20018,20019,20020,20021,20022,20023,20024,20025, &
& 20026,20027,20028,20029,20030,20031,20032,20033,20034,20035,20036, &
& 20037,20038,20039,20040,20041,20042,20043,20044,20045,20046,20047, &
& 20048,20049,20050,20051,20052,20053,20054,20055,20056,20057,20058, &
& 20059,20060,20061,20062,20063,20064,20065,20066,20067,20068,20069, &
& 20070,20071,20072,20073,20074,20075,20076,20077,20078,20079,20080, &
& 20081,20082,20083,20084,20085,20086,20087,20088,20089,20090,20091, &
& 20092]
 index_cnj(5901:6000)=[&
& 20093,20094,20095,20096,20097,20098,20099,20100,20101,20158,20196, &
& 20197,20198,20199,20200,20201,20202,20203,20204,20205,20206,20207, &
& 20208,20209,20210,20211,20212,20213,20214,20215,20216,20217,20218, &
& 20219,20220,20221,20222,20223,20224,20225,20226,20227,20228,20229, &
& 20230,20231,20232,20233,20234,20235,20236,20237,20238,20239,20240, &
& 20241,20242,20243,20244,20245,20246,20247,20248,20249,20250,20251, &
& 20252,20253,20254,20255,20256,20257,20258,20259,20260,20261,20262, &
& 20263,20264,20265,20266,20267,20268,20269,20270,20271,20272,20273, &
& 20274,20275,20276,20277,20278,20279,20280,20281,20282,20283,20284, &
& 20285]
 index_cnj(6001:6100)=[&
& 20286,20287,20288,20289,20290,20291,20292,20293,20294,20295,20296, &
& 20297,20298,20299,20300,20301,20302,20359,20397,20398,20399,20400, &
& 20401,20402,20403,20404,20405,20406,20407,20408,20409,20410,20411, &
& 20412,20413,20414,20415,20416,20417,20418,20419,20420,20421,20422, &
& 20423,20424,20425,20426,20427,20428,20429,20430,20431,20432,20433, &
& 20434,20435,20436,20437,20438,20439,20440,20441,20442,20443,20444, &
& 20445,20446,20447,20448,20449,20450,20451,20452,20453,20454,20455, &
& 20456,20457,20458,20459,20460,20461,20462,20463,20464,20465,20466, &
& 20467,20468,20469,20470,20471,20472,20473,20474,20475,20476,20477, &
& 20478]
 index_cnj(6101:6200)=[&
& 20479,20480,20481,20482,20483,20484,20485,20486,20487,20488,20489, &
& 20490,20491,20492,20493,20494,20495,20496,20497,20498,20499,20500, &
& 20501,20502,20503,20504,20561,20599,20600,20601,20602,20603,20604, &
& 20605,20606,20607,20608,20609,20610,20611,20612,20613,20614,20615, &
& 20616,20617,20618,20619,20620,20621,20622,20623,20624,20625,20626, &
& 20627,20628,20629,20630,20631,20632,20633,20634,20635,20636,20637, &
& 20638,20639,20640,20641,20642,20643,20644,20645,20646,20647,20648, &
& 20649,20650,20651,20652,20653,20654,20655,20656,20657,20658,20659, &
& 20660,20661,20662,20663,20664,20665,20666,20667,20668,20669,20670, &
& 20671]
 index_cnj(6201:6300)=[&
& 20672,20673,20674,20675,20676,20677,20678,20679,20680,20681,20682, &
& 20683,20684,20685,20686,20687,20688,20689,20690,20691,20692,20693, &
& 20694,20695,20696,20697,20698,20699,20700,20701,20702,20703,20704, &
& 20705,20706,20707,20764,20802,20803,20804,20805,20806,20807,20808, &
& 20809,20810,20811,20812,20813,20814,20815,20816,20817,20818,20819, &
& 20820,20821,20822,20823,20824,20825,20826,20827,20828,20829,20830, &
& 20831,20832,20833,20834,20835,20836,20837,20838,20839,20840,20841, &
& 20842,20843,20844,20845,20846,20847,20848,20849,20850,20851,20852, &
& 20853,20854,20855,20856,20857,20858,20859,20860,20861,20862,20863, &
& 20864]
 index_cnj(6301:6400)=[&
& 20865,20866,20867,20868,20869,20870,20871,20872,20873,20874,20875, &
& 20876,20877,20878,20879,20880,20881,20882,20883,20884,20885,20886, &
& 20887,20888,20889,20890,20891,20892,20893,20894,20895,20896,20897, &
& 20898,20899,20900,20901,20902,20903,20904,20905,20906,20907,20908, &
& 20909,20910,20911,20968,21006,21007,21008,21009,21010,21011,21012, &
& 21013,21014,21015,21016,21017,21018,21019,21020,21021,21022,21023, &
& 21024,21025,21026,21027,21028,21029,21030,21031,21032,21033,21034, &
& 21035,21036,21037,21038,21039,21040,21041,21042,21043,21044,21045, &
& 21046,21047,21048,21049,21050,21051,21052,21053,21054,21055,21056, &
& 21057]
 index_cnj(6401:6500)=[&
& 21058,21059,21060,21061,21062,21063,21064,21065,21066,21067,21068, &
& 21069,21070,21071,21072,21073,21074,21075,21076,21077,21078,21079, &
& 21080,21081,21082,21083,21084,21085,21086,21087,21088,21089,21090, &
& 21091,21092,21093,21094,21095,21096,21097,21098,21099,21100,21101, &
& 21102,21103,21104,21105,21106,21107,21108,21109,21110,21111,21112, &
& 21113,21114,21115,21116,21173,21211,21212,21213,21214,21215,21216, &
& 21217,21218,21219,21220,21221,21222,21223,21224,21225,21226,21227, &
& 21228,21229,21230,21231,21232,21233,21234,21235,21236,21237,21238, &
& 21239,21240,21241,21242,21243,21244,21245,21246,21247,21248,21249, &
& 21250]
 index_cnj(6501:6600)=[&
& 21251,21252,21253,21254,21255,21256,21257,21258,21259,21260,21261, &
& 21262,21263,21264,21265,21266,21267,21268,21269,21270,21271,21272, &
& 21273,21274,21275,21276,21277,21278,21279,21280,21281,21282,21283, &
& 21284,21285,21286,21287,21288,21289,21290,21291,21292,21293,21294, &
& 21295,21296,21297,21298,21299,21300,21301,21302,21303,21304,21305, &
& 21306,21307,21308,21309,21310,21311,21312,21313,21314,21315,21316, &
& 21317,21318,21319,21320,21321,21322,21379,21417,21418,21419,21420, &
& 21421,21422,21423,21424,21425,21426,21427,21428,21429,21430,21431, &
& 21432,21433,21434,21435,21436,21437,21438,21439,21440,21441,21442, &
& 21443]
 index_cnj(6601:6700)=[&
& 21444,21445,21446,21447,21448,21449,21450,21451,21452,21453,21454, &
& 21455,21456,21457,21458,21459,21460,21461,21462,21463,21464,21465, &
& 21466,21467,21468,21469,21470,21471,21472,21473,21474,21475,21476, &
& 21477,21478,21479,21480,21481,21482,21483,21484,21485,21486,21487, &
& 21488,21489,21490,21491,21492,21493,21494,21495,21496,21497,21498, &
& 21499,21500,21501,21502,21503,21504,21505,21506,21507,21508,21509, &
& 21510,21511,21512,21513,21514,21515,21516,21517,21518,21519,21520, &
& 21521,21522,21523,21524,21525,21526,21527,21528,21529,21586,21624, &
& 21625,21626,21627,21628,21629,21630,21631,21632,21633,21634,21635, &
& 21636]
 index_cnj(6701:6800)=[&
& 21637,21638,21639,21640,21641,21642,21643,21644,21645,21646,21647, &
& 21648,21649,21650,21651,21652,21653,21654,21655,21656,21657,21658, &
& 21659,21660,21661,21662,21663,21664,21665,21666,21667,21668,21669, &
& 21670,21671,21672,21673,21674,21675,21676,21677,21678,21679,21680, &
& 21681,21682,21683,21684,21685,21686,21687,21688,21689,21690,21691, &
& 21692,21693,21694,21695,21696,21697,21698,21699,21700,21701,21702, &
& 21703,21704,21705,21706,21707,21708,21709,21710,21711,21712,21713, &
& 21714,21715,21716,21717,21718,21719,21720,21721,21722,21723,21724, &
& 21725,21726,21727,21728,21729,21730,21731,21732,21733,21734,21735, &
& 21736]
 index_cnj(6801:6900)=[&
& 21737,21794,21832,21833,21834,21835,21836,21837,21838,21839,21840, &
& 21841,21842,21843,21844,21845,21846,21847,21848,21849,21850,21851, &
& 21852,21853,21854,21855,21856,21857,21858,21859,21860,21861,21862, &
& 21863,21864,21865,21866,21867,21868,21869,21870,21871,21872,21873, &
& 21874,21875,21876,21877,21878,21879,21880,21881,21882,21883,21884, &
& 21885,21886,21887,21888,21889,21890,21891,21892,21893,21894,21895, &
& 21896,21897,21898,21899,21900,21901,21902,21903,21904,21905,21906, &
& 21907,21908,21909,21910,21911,21912,21913,21914,21915,21916,21917, &
& 21918,21919,21920,21921,21922,21923,21924,21925,21926,21927,21928, &
& 21929]
 index_cnj(6901:7000)=[&
& 21930,21931,21932,21933,21934,21935,21936,21937,21938,21939,21940, &
& 21941,21942,21943,21944,21945,21946,22003,22041,22042,22043,22044, &
& 22045,22046,22047,22048,22049,22050,22051,22052,22053,22054,22055, &
& 22056,22057,22058,22059,22060,22061,22062,22063,22064,22065,22066, &
& 22067,22068,22069,22070,22071,22072,22073,22074,22075,22076,22077, &
& 22078,22079,22080,22081,22082,22083,22084,22085,22086,22087,22088, &
& 22089,22090,22091,22092,22093,22094,22095,22096,22097,22098,22099, &
& 22100,22101,22102,22103,22104,22105,22106,22107,22108,22109,22110, &
& 22111,22112,22113,22114,22115,22116,22117,22118,22119,22120,22121, &
& 22122]
 index_cnj(7001:7100)=[&
& 22123,22124,22125,22126,22127,22128,22129,22130,22131,22132,22133, &
& 22134,22135,22136,22137,22138,22139,22140,22141,22142,22143,22144, &
& 22145,22146,22147,22148,22149,22150,22151,22152,22153,22154,22155, &
& 22156,22213,22251,22252,22253,22254,22255,22256,22257,22258,22259, &
& 22260,22261,22262,22263,22264,22265,22266,22267,22268,22269,22270, &
& 22271,22272,22273,22274,22275,22276,22277,22278,22279,22280,22281, &
& 22282,22283,22284,22285,22286,22287,22288,22289,22290,22291,22292, &
& 22293,22294,22295,22296,22297,22298,22299,22300,22301,22302,22303, &
& 22304,22305,22306,22307,22308,22309,22310,22311,22312,22313,22314, &
& 22315]
 index_cnj(7101:7200)=[&
& 22316,22317,22318,22319,22320,22321,22322,22323,22324,22325,22326, &
& 22327,22328,22329,22330,22331,22332,22333,22334,22335,22336,22337, &
& 22338,22339,22340,22341,22342,22343,22344,22345,22346,22347,22348, &
& 22349,22350,22351,22352,22353,22354,22355,22356,22357,22358,22359, &
& 22360,22361,22362,22363,22364,22365,22366,22367,22424,22462,22463, &
& 22464,22465,22466,22467,22468,22469,22470,22471,22472,22473,22474, &
& 22475,22476,22477,22478,22479,22480,22481,22482,22483,22484,22485, &
& 22486,22487,22488,22489,22490,22491,22492,22493,22494,22495,22496, &
& 22497,22498,22499,22500,22501,22502,22503,22504,22505,22506,22507, &
& 22508]
 index_cnj(7201:7300)=[&
& 22509,22510,22511,22512,22513,22514,22515,22516,22517,22518,22519, &
& 22520,22521,22522,22523,22524,22525,22526,22527,22528,22529,22530, &
& 22531,22532,22533,22534,22535,22536,22537,22538,22539,22540,22541, &
& 22542,22543,22544,22545,22546,22547,22548,22549,22550,22551,22552, &
& 22553,22554,22555,22556,22557,22558,22559,22560,22561,22562,22563, &
& 22564,22565,22566,22567,22568,22569,22570,22571,22572,22573,22574, &
& 22575,22576,22577,22578,22579,22636,22674,22675,22676,22677,22678, &
& 22679,22680,22681,22682,22683,22684,22685,22686,22687,22688,22689, &
& 22690,22691,22692,22693,22694,22695,22696,22697,22698,22699,22700, &
& 22701]
 index_cnj(7301:7400)=[&
& 22702,22703,22704,22705,22706,22707,22708,22709,22710,22711,22712, &
& 22713,22714,22715,22716,22717,22718,22719,22720,22721,22722,22723, &
& 22724,22725,22726,22727,22728,22729,22730,22731,22732,22733,22734, &
& 22735,22736,22737,22738,22739,22740,22741,22742,22743,22744,22745, &
& 22746,22747,22748,22749,22750,22751,22752,22753,22754,22755,22756, &
& 22757,22758,22759,22760,22761,22762,22763,22764,22765,22766,22767, &
& 22768,22769,22770,22771,22772,22773,22774,22775,22776,22777,22778, &
& 22779,22780,22781,22782,22783,22784,22785,22786,22787,22788,22789, &
& 22790,22791,22792,22849,22887,22888,22889,22890,22891,22892,22893, &
& 22894]
 index_cnj(7401:7500)=[&
& 22895,22896,22897,22898,22899,22900,22901,22902,22903,22904,22905, &
& 22906,22907,22908,22909,22910,22911,22912,22913,22914,22915,22916, &
& 22917,22918,22919,22920,22921,22922,22923,22924,22925,22926,22927, &
& 22928,22929,22930,22931,22932,22933,22934,22935,22936,22937,22938, &
& 22939,22940,22941,22942,22943,22944,22945,22946,22947,22948,22949, &
& 22950,22951,22952,22953,22954,22955,22956,22957,22958,22959,22960, &
& 22961,22962,22963,22964,22965,22966,22967,22968,22969,22970,22971, &
& 22972,22973,22974,22975,22976,22977,22978,22979,22980,22981,22982, &
& 22983,22984,22985,22986,22987,22988,22989,22990,22991,22992,22993, &
& 22994]
 index_cnj(7501:7600)=[&
& 22995,22996,22997,22998,22999,23000,23001,23002,23003,23004,23005, &
& 23006,23063,23101,23102,23103,23104,23105,23106,23107,23108,23109, &
& 23110,23111,23112,23113,23114,23115,23116,23117,23118,23119,23120, &
& 23121,23122,23123,23124,23125,23126,23127,23128,23129,23130,23131, &
& 23132,23133,23134,23135,23136,23137,23138,23139,23140,23141,23142, &
& 23143,23144,23145,23146,23147,23148,23149,23150,23151,23152,23153, &
& 23154,23155,23156,23157,23158,23159,23160,23161,23162,23163,23164, &
& 23165,23166,23167,23168,23169,23170,23171,23172,23173,23174,23175, &
& 23176,23177,23178,23179,23180,23181,23182,23183,23184,23185,23186, &
& 23187]
 index_cnj(7601:7700)=[&
& 23188,23189,23190,23191,23192,23193,23194,23195,23196,23197,23198, &
& 23199,23200,23201,23202,23203,23204,23205,23206,23207,23208,23209, &
& 23210,23211,23212,23213,23214,23215,23216,23217,23218,23219,23220, &
& 23221,23278,23316,23317,23318,23319,23320,23321,23322,23323,23324, &
& 23325,23326,23327,23328,23329,23330,23331,23332,23333,23334,23335, &
& 23336,23337,23338,23339,23340,23341,23342,23343,23344,23345,23346, &
& 23347,23348,23349,23350,23351,23352,23353,23354,23355,23356,23357, &
& 23358,23359,23360,23361,23362,23363,23364,23365,23366,23367,23368, &
& 23369,23370,23371,23372,23373,23374,23375,23376,23377,23378,23379, &
& 23380]
 index_cnj(7701:7800)=[&
& 23381,23382,23383,23384,23385,23386,23387,23388,23389,23390,23391, &
& 23392,23393,23394,23395,23396,23397,23398,23399,23400,23401,23402, &
& 23403,23404,23405,23406,23407,23408,23409,23410,23411,23412,23413, &
& 23414,23415,23416,23417,23418,23419,23420,23421,23422,23423,23424, &
& 23425,23426,23427,23428,23429,23430,23431,23432,23433,23434,23435, &
& 23436,23437,23494,23532,23533,23534,23535,23536,23537,23538,23539, &
& 23540,23541,23542,23543,23544,23545,23546,23547,23548,23549,23550, &
& 23551,23552,23553,23554,23555,23556,23557,23558,23559,23560,23561, &
& 23562,23563,23564,23565,23566,23567,23568,23569,23570,23571,23572, &
& 23573]
 index_cnj(7801:7900)=[&
& 23574,23575,23576,23577,23578,23579,23580,23581,23582,23583,23584, &
& 23585,23586,23587,23588,23589,23590,23591,23592,23593,23594,23595, &
& 23596,23597,23598,23599,23600,23601,23602,23603,23604,23605,23606, &
& 23607,23608,23609,23610,23611,23612,23613,23614,23615,23616,23617, &
& 23618,23619,23620,23621,23622,23623,23624,23625,23626,23627,23628, &
& 23629,23630,23631,23632,23633,23634,23635,23636,23637,23638,23639, &
& 23640,23641,23642,23643,23644,23645,23646,23647,23648,23649,23650, &
& 23651,23652,23653,23654,23711,23749,23750,23751,23752,23753,23754, &
& 23755,23756,23757,23758,23759,23760,23761,23762,23763,23764,23765, &
& 23766]
 index_cnj(7901:8000)=[&
& 23767,23768,23769,23770,23771,23772,23773,23774,23775,23776,23777, &
& 23778,23779,23780,23781,23782,23783,23784,23785,23786,23787,23788, &
& 23789,23790,23791,23792,23793,23794,23795,23796,23797,23798,23799, &
& 23800,23801,23802,23803,23804,23805,23806,23807,23808,23809,23810, &
& 23811,23812,23813,23814,23815,23816,23817,23818,23819,23820,23821, &
& 23822,23823,23824,23825,23826,23827,23828,23829,23830,23831,23832, &
& 23833,23834,23835,23836,23837,23838,23839,23840,23841,23842,23843, &
& 23844,23845,23846,23847,23848,23849,23850,23851,23852,23853,23854, &
& 23855,23856,23857,23858,23859,23860,23861,23862,23863,23864,23865, &
& 23866]
 index_cnj(8001:8100)=[&
& 23867,23868,23869,23870,23871,23872,23929,23967,23968,23969,23970, &
& 23971,23972,23973,23974,23975,23976,23977,23978,23979,23980,23981, &
& 23982,23983,23984,23985,23986,23987,23988,23989,23990,23991,23992, &
& 23993,23994,23995,23996,23997,23998,23999,24000,24001,24002,24003, &
& 24004,24005,24006,24007,24008,24009,24010,24011,24012,24013,24014, &
& 24015,24016,24017,24018,24019,24020,24021,24022,24023,24024,24025, &
& 24026,24027,24028,24029,24030,24031,24032,24033,24034,24035,24036, &
& 24037,24038,24039,24040,24041,24042,24043,24044,24045,24046,24047, &
& 24048,24049,24050,24051,24052,24053,24054,24055,24056,24057,24058, &
& 24059]
 index_cnj(8101:8200)=[&
& 24060,24061,24062,24063,24064,24065,24066,24067,24068,24069,24070, &
& 24071,24072,24073,24074,24075,24076,24077,24078,24079,24080,24081, &
& 24082,24083,24084,24085,24086,24087,24088,24089,24090,24091,24148, &
& 24186,24187,24188,24189,24190,24191,24192,24193,24194,24195,24196, &
& 24197,24198,24199,24200,24201,24202,24203,24204,24205,24206,24207, &
& 24208,24209,24210,24211,24212,24213,24214,24215,24216,24217,24218, &
& 24219,24220,24221,24222,24223,24224,24225,24226,24227,24228,24229, &
& 24230,24231,24232,24233,24234,24235,24236,24237,24238,24239,24