///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoHavingDesc.cc
// -----------------
// Cego having clause structure class definition
//                                                 
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2019 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoHavingDesc
//
// Description: The CegoHavingDesc class is a container class for the query having clause representation
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoHavingDesc.h"
#include "CegoCondDesc.h"
#include "CegoSelect.h"
#include "CegoXMLdef.h"
#include "CegoDatabaseFormater.h"

CegoHavingDesc::CegoHavingDesc(CegoExpr* pAggExpr, CegoExpr* pExpr, const CegoComparison& comp)
{
    _pAggExpr = pAggExpr;    
    _pExpr = pExpr;
    _comp = comp;
}

CegoHavingDesc::CegoHavingDesc(char* buf, CegoDistManager *pGTM, int tabSetId)
{
    _pAggExpr = 0;
    _pExpr = 0;
    decode(buf, pGTM, tabSetId);
}

CegoHavingDesc::CegoHavingDesc(Element* pHE, CegoDistManager *pGTM)
{
    _pAggExpr = 0;
    _pExpr = 0;
    fromElement(pHE, pGTM);
}

CegoHavingDesc::~CegoHavingDesc()
{
    if ( _pAggExpr )
	delete _pAggExpr;
    if ( _pExpr )
	delete _pExpr;
}

CegoComparison CegoHavingDesc::getComparison()
{
    return _comp;
}

ListT<CegoAttrDesc*> CegoHavingDesc::getAttrRefList() const
{
    ListT<CegoAttrDesc*> attrList;

    attrList = _pExpr->getAttrRefList();
    attrList += _pAggExpr->getAttrRefList();
    
    return attrList;
}

void CegoHavingDesc::encode(char *buf)
{
    char* pBuf = (char*)buf;

    _pAggExpr->encode(pBuf);
    pBuf = pBuf + _pAggExpr->getEncodingLength();

    _pExpr->encode(pBuf);
    pBuf = pBuf + _pExpr->getEncodingLength();
}

void CegoHavingDesc::decode(char *buf, CegoDistManager *pGTM, int tabSetId)
{
    if ( _pAggExpr )
	delete _pAggExpr;
    if ( _pExpr )
	delete _pExpr;

    char* pBuf = (char*)buf;

    _pAggExpr = new CegoExpr(pBuf, pGTM, tabSetId);
    pBuf = pBuf + _pAggExpr->getEncodingLength();

    _pExpr = new CegoExpr(pBuf, pGTM, tabSetId);
    pBuf = pBuf + _pExpr->getEncodingLength();
}

int CegoHavingDesc::getEncodingLength() const
{
    return _pAggExpr->getEncodingLength() + _pExpr->getEncodingLength();
}

CegoExpr* CegoHavingDesc::getAggExpr()
{
    return _pAggExpr;
}

CegoExpr* CegoHavingDesc::getExpr()
{
    return _pExpr;
}

Element* CegoHavingDesc::toElement() const
{
    Element *pHavingElement = new Element( XML_HAVING_ELEMENT );
    
    pHavingElement->addContent( _pAggExpr->toElement() );
    pHavingElement->addContent( _pExpr->toElement() );
    
    switch (_comp)
    {
    case EQUAL:
	pHavingElement->setAttribute(XML_COMP_ATTR, XML_EQUAL_VALUE);
	break;
    case NOT_EQUAL:
	pHavingElement->setAttribute(XML_COMP_ATTR, XML_NOTEQUAL_VALUE);
	break;
    case LESS_THAN:
	pHavingElement->setAttribute(XML_COMP_ATTR, XML_LESSTHAN_VALUE);
	break;
    case MORE_THAN:
	pHavingElement->setAttribute(XML_COMP_ATTR, XML_MORETHAN_VALUE);
	break;
    case LESS_EQUAL_THAN:
	pHavingElement->setAttribute(XML_COMP_ATTR, XML_LESSEQUALTHAN_VALUE);
	break;
    case MORE_EQUAL_THAN:
	pHavingElement->setAttribute(XML_COMP_ATTR, XML_MOREEQUALTHAN_VALUE);
	break;
    }
    
    return pHavingElement;
}

void CegoHavingDesc::fromElement(Element *pHavingElement, CegoDistManager *pGTM)
{
    if ( _pAggExpr )
	delete _pAggExpr;
    if ( _pExpr )
	delete _pExpr;
   
    ListT<Element*> el = pHavingElement->getChildren(XML_EXPR_ELEMENT);
    Element **pEE = el.First();
    if ( pEE )
    {
	_pExpr = new CegoExpr(*pEE, pGTM);
    }
    ListT<Element*> al = pHavingElement->getChildren(XML_AGGREGATION_ELEMENT);
    Element **pAE = al.First();
    if ( pAE )
    {
	_pAggExpr = new CegoExpr(*pAE, pGTM);
    }

    Chain compString = pHavingElement->getAttributeValue( XML_COMP_ATTR );

    if ( compString == Chain(XML_EQUAL_VALUE) )
	_comp = EQUAL;
    else if ( compString == Chain(XML_NOTEQUAL_VALUE) )
	_comp = NOT_EQUAL;
    else if ( compString == Chain(XML_LESSTHAN_VALUE) )
	_comp = LESS_THAN;
    else if ( compString == Chain(XML_MORETHAN_VALUE) )
	_comp = MORE_THAN;
    else if ( compString == Chain(XML_LESSEQUALTHAN_VALUE) )
	_comp = LESS_EQUAL_THAN;
    else if ( compString == Chain(XML_MOREEQUALTHAN_VALUE) )
	_comp = MORE_EQUAL_THAN;
}

Chain CegoHavingDesc::getId() const
{
    Chain s;

    s += _pAggExpr->getId();
    
    switch (_comp)
    {
    case EQUAL:
	s += "=";
	break;
    case NOT_EQUAL:
	s += "!=";
	break;
    case LESS_THAN:
	s += "<";
	break;
    case MORE_THAN:
	s += ">";
	break;
    case LESS_EQUAL_THAN:
	s += "<=";
	break;
    case MORE_EQUAL_THAN:
	s += ">=";
	break;
    }
    
    s += _pExpr->getId();

    return s;
}

Chain CegoHavingDesc::toChain(const Chain& indent) const
{
    Chain s;

    s += indent + _pAggExpr->toChain();
    
    switch (_comp)
    {
    case EQUAL:
	s += " = ";
	break;
    case NOT_EQUAL:
	s += " != ";
	break;
    case LESS_THAN:
	s += " < ";
	break;
    case MORE_THAN:
	s += " > ";
	break;
    case LESS_EQUAL_THAN:
	s += " <= ";
	break;
    case MORE_EQUAL_THAN:
	s += " >= ";
	break;
    }
    
    s += _pExpr->toChain();

    return s;
}

Chain CegoHavingDesc::dbFormat(CegoDatabaseFormater *pForm) const
{
    return pForm->formatHaving(_pAggExpr, _pExpr, _comp);
}

CegoHavingDesc* CegoHavingDesc::clone(bool isAttrRef)
{
    return new CegoHavingDesc(_pAggExpr->clone(isAttrRef), _pExpr->clone(isAttrRef), _comp);
}

ostream& operator << (ostream& s, const CegoHavingDesc& p)
{
    s << p.toChain();
    return s;
}
