#pragma once

#include <map>
#include <set>
#include <string>
#include <tuple>
#include <vector>

#include <krpc/decoder.hpp>
#include <krpc/encoder.hpp>
#include <krpc/error.hpp>
#include <krpc/event.hpp>
#include <krpc/object.hpp>
#include <krpc/service.hpp>
#include <krpc/stream.hpp>

namespace krpc {
namespace services {

class SpaceCenter : public Service {
 public:
  explicit SpaceCenter(Client* client);

  // Class forward declarations
  class Alarm;
  class AlarmManager;
  class Antenna;
  class AutoPilot;
  class Camera;
  class CargoBay;
  class CelestialBody;
  class CommLink;
  class CommNode;
  class Comms;
  class Contract;
  class ContractManager;
  class ContractParameter;
  class Control;
  class ControlSurface;
  class CrewMember;
  class Decoupler;
  class DockingPort;
  class Engine;
  class Experiment;
  class Fairing;
  class Flight;
  class Force;
  class Intake;
  class LaunchClamp;
  class LaunchSite;
  class Leg;
  class Light;
  class Module;
  class Node;
  class Orbit;
  class Parachute;
  class Part;
  class Parts;
  class Propellant;
  class RCS;
  class Radiator;
  class ReactionWheel;
  class ReferenceFrame;
  class Resource;
  class ResourceConverter;
  class ResourceDrain;
  class ResourceHarvester;
  class ResourceTransfer;
  class Resources;
  class RoboticController;
  class RoboticHinge;
  class RoboticPiston;
  class RoboticRotation;
  class RoboticRotor;
  class ScienceData;
  class ScienceSubject;
  class Sensor;
  class SolarPanel;
  class Thruster;
  class Vessel;
  class Waypoint;
  class WaypointManager;
  class Wheel;

  /**
   * The state of an antenna. See SpaceCenter::Antenna::state.
   */
  enum struct AntennaState {
    /**
     * Antenna is fully deployed.
     */
    deployed = 0,
    /**
     * Antenna is fully retracted.
     */
    retracted = 1,
    /**
     * Antenna is being deployed.
     */
    deploying = 2,
    /**
     * Antenna is being retracted.
     */
    retracting = 3,
    /**
     * Antenna is broken.
     */
    broken = 4
  };

  /**
   * The state of an auto-strut. SpaceCenter::Part::auto_strut_mode
   */
  enum struct AutoStrutMode {
    /**
     * Off
     */
    off = 0,
    /**
     * Root
     */
    root = 1,
    /**
     * Heaviest
     */
    heaviest = 2,
    /**
     * Grandparent
     */
    grandparent = 3,
    /**
     * ForceRoot
     */
    force_root = 4,
    /**
     * ForceHeaviest
     */
    force_heaviest = 5,
    /**
     * ForceGrandparent
     */
    force_grandparent = 6
  };

  /**
   * See SpaceCenter::Camera::mode.
   */
  enum struct CameraMode {
    /**
     * The camera is showing the active vessel, in "auto" mode.
     */
    automatic = 0,
    /**
     * The camera is showing the active vessel, in "free" mode.
     */
    free = 1,
    /**
     * The camera is showing the active vessel, in "chase" mode.
     */
    chase = 2,
    /**
     * The camera is showing the active vessel, in "locked" mode.
     */
    locked = 3,
    /**
     * The camera is showing the active vessel, in "orbital" mode.
     */
    orbital = 4,
    /**
     * The Intra-Vehicular Activity view is being shown.
     */
    iva = 5,
    /**
     * The map view is being shown.
     */
    map = 6
  };

  /**
   * The state of a cargo bay. See SpaceCenter::CargoBay::state.
   */
  enum struct CargoBayState {
    /**
     * Cargo bay is fully open.
     */
    open = 0,
    /**
     * Cargo bay closed and locked.
     */
    closed = 1,
    /**
     * Cargo bay is opening.
     */
    opening = 2,
    /**
     * Cargo bay is closing.
     */
    closing = 3
  };

  /**
   * The type of a communication link.
   * See SpaceCenter::CommLink::type.
   */
  enum struct CommLinkType {
    /**
     * Link is to a base station on Kerbin.
     */
    home = 0,
    /**
     * Link is to a control source, for example a manned spacecraft.
     */
    control = 1,
    /**
     * Link is to a relay satellite.
     */
    relay = 2
  };

  /**
   * The state of a contract. See SpaceCenter::Contract::state.
   */
  enum struct ContractState {
    /**
     * The contract is active.
     */
    active = 0,
    /**
     * The contract has been canceled.
     */
    canceled = 1,
    /**
     * The contract has been completed.
     */
    completed = 2,
    /**
     * The deadline for the contract has expired.
     */
    deadline_expired = 3,
    /**
     * The contract has been declined.
     */
    declined = 4,
    /**
     * The contract has been failed.
     */
    failed = 5,
    /**
     * The contract has been generated.
     */
    generated = 6,
    /**
     * The contract has been offered to the player.
     */
    offered = 7,
    /**
     * The contract was offered to the player, but the offer expired.
     */
    offer_expired = 8,
    /**
     * The contract has been withdrawn.
     */
    withdrawn = 9
  };

  /**
   * See SpaceCenter::Control::input_mode.
   */
  enum struct ControlInputMode {
    /**
     * Control inputs are added to the vessels current control inputs.
     */
    additive = 0,
    /**
     * Control inputs (when they are non-zero) override the vessels current control inputs.
     */
    override = 1
  };

  /**
   * The control source of a vessel.
   * See SpaceCenter::Control::source.
   */
  enum struct ControlSource {
    /**
     * Vessel is controlled by a Kerbal.
     */
    kerbal = 0,
    /**
     * Vessel is controlled by a probe core.
     */
    probe = 1,
    /**
     * Vessel is not controlled.
     */
    none = 2
  };

  /**
   * The control state of a vessel.
   * See SpaceCenter::Control::state.
   */
  enum struct ControlState {
    /**
     * Full controllable.
     */
    full = 0,
    /**
     * Partially controllable.
     */
    partial = 1,
    /**
     * Not controllable.
     */
    none = 2
  };

  /**
   * A crew member's gender.
   * See SpaceCenter::CrewMember::gender.
   */
  enum struct CrewMemberGender {
    /**
     * Male.
     */
    male = 0,
    /**
     * Female.
     */
    female = 1
  };

  /**
   * The type of a crew member.
   * See SpaceCenter::CrewMember::type.
   */
  enum struct CrewMemberType {
    /**
     * An applicant for crew.
     */
    applicant = 0,
    /**
     * Rocket crew.
     */
    crew = 1,
    /**
     * A tourist.
     */
    tourist = 2,
    /**
     * An unowned crew member.
     */
    unowned = 3
  };

  /**
   * The state of a docking port. See SpaceCenter::DockingPort::state.
   */
  enum struct DockingPortState {
    /**
     * The docking port is ready to dock to another docking port.
     */
    ready = 0,
    /**
     * The docking port is docked to another docking port, or docked to
     * another part (from the VAB/SPH).
     */
    docked = 1,
    /**
     * The docking port is very close to another docking port,
     * but has not docked. It is using magnetic force to acquire a solid dock.
     */
    docking = 2,
    /**
     * The docking port has just been undocked from another docking port,
     * and is disabled until it moves away by a sufficient distance
     * (SpaceCenter::DockingPort::reengage_distance).
     */
    undocking = 3,
    /**
     * The docking port has a shield, and the shield is closed.
     */
    shielded = 4,
    /**
     * The docking ports shield is currently opening/closing.
     */
    moving = 5
  };

  /**
   * Resource drain mode.
   * See SpaceCenter::ResourceDrain::drain_mode.
   */
  enum struct DrainMode {
    /**
     * Drains from the parent part.
     */
    part = 0,
    /**
     * Drains from all available parts.
     */
    vessel = 1
  };

  /**
   * Editor facility.
   * See SpaceCenter::LaunchSite::editor_facility.
   */
  enum struct EditorFacility {
    /**
     * Vehicle Assembly Building.
     */
    vab = 1,
    /**
     * Space Plane Hanger.
     */
    sph = 2,
    /**
     * None.
     */
    none = 0
  };

  /**
   * The game mode.
   * Returned by SpaceCenter::GameMode
   */
  enum struct GameMode {
    /**
     * Sandbox mode.
     */
    sandbox = 0,
    /**
     * Career mode.
     */
    career = 1,
    /**
     * Science career mode.
     */
    science = 2,
    /**
     * Science sandbox mode.
     */
    science_sandbox = 3,
    /**
     * Mission mode.
     */
    mission = 4,
    /**
     * Mission builder mode.
     */
    mission_builder = 5,
    /**
     * Scenario mode.
     */
    scenario = 6,
    /**
     * Scenario mode that cannot be resumed.
     */
    scenario_non_resumable = 7
  };

  /**
   * The state of a landing leg. See SpaceCenter::Leg::state.
   */
  enum struct LegState {
    /**
     * Landing leg is fully deployed.
     */
    deployed = 0,
    /**
     * Landing leg is fully retracted.
     */
    retracted = 1,
    /**
     * Landing leg is being deployed.
     */
    deploying = 2,
    /**
     * Landing leg is being retracted.
     */
    retracting = 3,
    /**
     * Landing leg is broken.
     */
    broken = 4
  };

  /**
   * The set of things that are visible in map mode.
   * These may be combined with bitwise logic.
   */
  enum struct MapFilterType {
    /**
     * Everything.
     */
    all = -1,
    /**
     * Nothing.
     */
    none = 0,
    /**
     * Debris.
     */
    debris = 1,
    /**
     * Unknown.
     */
    unknown = 2,
    /**
     * SpaceObjects.
     */
    space_objects = 4,
    /**
     * Probes.
     */
    probes = 8,
    /**
     * Rovers.
     */
    rovers = 16,
    /**
     * Landers.
     */
    landers = 32,
    /**
     * Ships.
     */
    ships = 64,
    /**
     * Stations.
     */
    stations = 128,
    /**
     * Bases.
     */
    bases = 256,
    /**
     * EVAs.
     */
    ev_as = 512,
    /**
     * Flags.
     */
    flags = 1024,
    /**
     * Planes.
     */
    plane = 2048,
    /**
     * Relays.
     */
    relay = 4096,
    /**
     * Launch Sites.
     */
    site = 8192,
    /**
     * Deployed Science Controllers.
     */
    deployed_science_controller = 16384
  };

  /**
   * The state of the motor on a powered wheel. See SpaceCenter::Wheel::motor_state.
   */
  enum struct MotorState {
    /**
     * The motor is idle.
     */
    idle = 0,
    /**
     * The motor is running.
     */
    running = 1,
    /**
     * The motor is disabled.
     */
    disabled = 2,
    /**
     * The motor is inoperable.
     */
    inoperable = 3,
    /**
     * The motor does not have enough resources to run.
     */
    not_enough_resources = 4
  };

  /**
   * The state of a parachute. See SpaceCenter::Parachute::state.
   */
  enum struct ParachuteState {
    /**
     * The parachute is safely tucked away inside its housing.
     */
    stowed = 0,
    /**
     * The parachute is armed for deployment.
     */
    armed = 1,
    /**
     * The parachute has been deployed and is providing some drag,
     * but is not fully deployed yet. (Stock parachutes only)
     */
    semi_deployed = 2,
    /**
     * The parachute is fully deployed.
     */
    deployed = 3,
    /**
     * The parachute has been cut.
     */
    cut = 4
  };

  /**
   * The state of a radiator. SpaceCenter::Radiator::state
   */
  enum struct RadiatorState {
    /**
     * Radiator is fully extended.
     */
    extended = 0,
    /**
     * Radiator is fully retracted.
     */
    retracted = 1,
    /**
     * Radiator is being extended.
     */
    extending = 2,
    /**
     * Radiator is being retracted.
     */
    retracting = 3,
    /**
     * Radiator is broken.
     */
    broken = 4
  };

  /**
   * The state of a resource converter. See SpaceCenter::ResourceConverter::state.
   */
  enum struct ResourceConverterState {
    /**
     * Converter is running.
     */
    running = 0,
    /**
     * Converter is idle.
     */
    idle = 1,
    /**
     * Converter is missing a required resource.
     */
    missing_resource = 2,
    /**
     * No available storage for output resource.
     */
    storage_full = 3,
    /**
     * At preset resource capacity.
     */
    capacity = 4,
    /**
     * Unknown state. Possible with modified resource converters.
     * In this case, check SpaceCenter::ResourceConverter::status_info for more information.
     */
    unknown = 5
  };

  /**
   * The way in which a resource flows between parts. See SpaceCenter::Resources::flow_mode.
   */
  enum struct ResourceFlowMode {
    /**
     * The resource flows to any part in the vessel. For example, electric charge.
     */
    vessel = 0,
    /**
     * The resource flows from parts in the first stage, followed by the second,
     * and so on. For example, mono-propellant.
     */
    stage = 1,
    /**
     * The resource flows between adjacent parts within the vessel. For example,
     * liquid fuel or oxidizer.
     */
    adjacent = 2,
    /**
     * The resource does not flow. For example, solid fuel.
     */
    none = 3
  };

  /**
   * The state of a resource harvester. See SpaceCenter::ResourceHarvester::state.
   */
  enum struct ResourceHarvesterState {
    /**
     * The drill is deploying.
     */
    deploying = 0,
    /**
     * The drill is deployed and ready.
     */
    deployed = 1,
    /**
     * The drill is retracting.
     */
    retracting = 2,
    /**
     * The drill is retracted.
     */
    retracted = 3,
    /**
     * The drill is running.
     */
    active = 4
  };

  /**
   * A crew member's roster status.
   * See SpaceCenter::CrewMember::roster_status.
   */
  enum struct RosterStatus {
    /**
     * Available.
     */
    available = 0,
    /**
     * Assigned.
     */
    assigned = 1,
    /**
     * Dead.
     */
    dead = 2,
    /**
     * Missing.
     */
    missing = 3
  };

  /**
   * The behavior of the SAS auto-pilot. See SpaceCenter::AutoPilot::sas_mode.
   */
  enum struct SASMode {
    /**
     * Stability assist mode. Dampen out any rotation.
     */
    stability_assist = 0,
    /**
     * Point in the burn direction of the next maneuver node.
     */
    maneuver = 1,
    /**
     * Point in the prograde direction.
     */
    prograde = 2,
    /**
     * Point in the retrograde direction.
     */
    retrograde = 3,
    /**
     * Point in the orbit normal direction.
     */
    normal = 4,
    /**
     * Point in the orbit anti-normal direction.
     */
    anti_normal = 5,
    /**
     * Point in the orbit radial direction.
     */
    radial = 6,
    /**
     * Point in the orbit anti-radial direction.
     */
    anti_radial = 7,
    /**
     * Point in the direction of the current target.
     */
    target = 8,
    /**
     * Point away from the current target.
     */
    anti_target = 9
  };

  /**
   * The state of a solar panel. See SpaceCenter::SolarPanel::state.
   */
  enum struct SolarPanelState {
    /**
     * Solar panel is fully extended.
     */
    extended = 0,
    /**
     * Solar panel is fully retracted.
     */
    retracted = 1,
    /**
     * Solar panel is being extended.
     */
    extending = 2,
    /**
     * Solar panel is being retracted.
     */
    retracting = 3,
    /**
     * Solar panel is broken.
     */
    broken = 4
  };

  /**
   * The mode of the speed reported in the navball.
   * See SpaceCenter::Control::speed_mode.
   */
  enum struct SpeedMode {
    /**
     * Speed is relative to the vessel's orbit.
     */
    orbit = 0,
    /**
     * Speed is relative to the surface of the body being orbited.
     */
    surface = 1,
    /**
     * Speed is relative to the current target.
     */
    target = 2
  };

  /**
   * A crew member's suit type.
   * See SpaceCenter::CrewMember::suit_type.
   */
  enum struct SuitType {
    /**
     * Default.
     */
    default_ = 0,
    /**
     * Vintage.
     */
    vintage = 1,
    /**
     * Future.
     */
    future = 2,
    /**
     * Slim.
     */
    slim = 3
  };

  /**
   * The situation a vessel is in.
   * See SpaceCenter::Vessel::situation.
   */
  enum struct VesselSituation {
    /**
     * Vessel is awaiting launch.
     */
    pre_launch = 0,
    /**
     * Vessel is orbiting a body.
     */
    orbiting = 1,
    /**
     * Vessel is on a sub-orbital trajectory.
     */
    sub_orbital = 2,
    /**
     * Escaping.
     */
    escaping = 3,
    /**
     * Vessel is flying through an atmosphere.
     */
    flying = 4,
    /**
     * Vessel is landed on the surface of a body.
     */
    landed = 5,
    /**
     * Vessel has splashed down in an ocean.
     */
    splashed = 6,
    /**
     * Vessel is docked to another.
     */
    docked = 7
  };

  /**
   * The type of a vessel.
   * See SpaceCenter::Vessel::type.
   */
  enum struct VesselType {
    /**
     * Base.
     */
    base = 0,
    /**
     * Debris.
     */
    debris = 1,
    /**
     * Lander.
     */
    lander = 2,
    /**
     * Plane.
     */
    plane = 3,
    /**
     * Probe.
     */
    probe = 4,
    /**
     * Relay.
     */
    relay = 5,
    /**
     * Rover.
     */
    rover = 6,
    /**
     * Ship.
     */
    ship = 7,
    /**
     * Station.
     */
    station = 8,
    /**
     * SpaceObject.
     */
    space_object = 9,
    /**
     * Unknown.
     */
    unknown = 10,
    /**
     * EVA.
     */
    eva = 11,
    /**
     * Flag.
     */
    flag = 12,
    /**
     * DeployedScienceController.
     */
    deployed_science_controller = 13,
    /**
     * DeploedSciencePart.
     */
    deployed_science_part = 14,
    /**
     * DroppedPart.
     */
    dropped_part = 15,
    /**
     * DeployedGroundPart.
     */
    deployed_ground_part = 16
  };

  /**
   * The time warp mode.
   * Returned by SpaceCenter::WarpMode
   */
  enum struct WarpMode {
    /**
     * Time warp is active, and in regular "on-rails" mode.
     */
    rails = 0,
    /**
     * Time warp is active, and in physical time warp mode.
     */
    physics = 1,
    /**
     * Time warp is not active.
     */
    none = 2
  };

  /**
   * The state of a wheel. See SpaceCenter::Wheel::state.
   */
  enum struct WheelState {
    /**
     * Wheel is fully deployed.
     */
    deployed = 0,
    /**
     * Wheel is fully retracted.
     */
    retracted = 1,
    /**
     * Wheel is being deployed.
     */
    deploying = 2,
    /**
     * Wheel is being retracted.
     */
    retracting = 3,
    /**
     * Wheel is broken.
     */
    broken = 4
  };

  /**
   * Returns true if regular "on-rails" time warp can be used, at the specified warp
   * factor. The maximum time warp rate is limited by various things,
   * including how close the active vessel is to a planet. See
   * <a href="https://wiki.kerbalspaceprogram.com/wiki/Time_warp">the KSP wiki</a>
   * for details.
   * @param factor The warp factor to check.
   */
  bool can_rails_warp_at(int32_t factor);

  /**
   * Whether the current flight can be reverted to launch.
   */
  bool can_revert_to_launch();

  /**
   * Clears the current target.
   */
  void clear_target();

  /**
   * Creates a Kerbal.
   * @param name
   * @param job
   * @param male
   */
  void create_kerbal(std::string name, std::string job, bool male);

  /**
   * Find a Kerbal by name.
   * @param name
   * @return null if no Kerbal with the given name exists.
   */
  SpaceCenter::CrewMember get_kerbal(std::string name);

  /**
   * Launch a vessel.
   * @param craftDirectory Name of the directory in the current saves
   * "Ships" directory, that contains the craft file.
   * For example "VAB" or "SPH".
   * @param name Name of the vessel to launch. This is the name of the ".craft" file
   * in the save directory, without the ".craft" file extension.
   * @param launchSite Name of the launch site. For example "LaunchPad" or
   * "Runway".
   * @param recover If true and there is a vessel on the launch site,
   * recover it before launching.
   * @param crew If not null, a list of names of Kerbals to place in the craft. Otherwise the crew will use default assignments.
   * @param flagUrl If not null, the asset URL of the mission flag to use for the launch.
   *
   * Throws an exception if any of the games pre-flight checks fail.
   */
  void launch_vessel(std::string craft_directory, std::string name, std::string launch_site, bool recover, std::vector<std::string> crew, std::string flag_url);

  /**
   * Launch a new vessel from the SPH onto the runway.
   * @param name Name of the vessel to launch.
   * @param recover If true and there is a vessel on the runway,
   * recover it before launching.
   *
   * This is equivalent to calling SpaceCenter::launch_vessel with the craft directory
   * set to "SPH" and the launch site set to "Runway".
   * Throws an exception if any of the games pre-flight checks fail.
   */
  void launch_vessel_from_sph(std::string name, bool recover);

  /**
   * Launch a new vessel from the VAB onto the launchpad.
   * @param name Name of the vessel to launch.
   * @param recover If true and there is a vessel on the launch pad,
   * recover it before launching.
   *
   * This is equivalent to calling SpaceCenter::launch_vessel with the craft directory
   * set to "VAB" and the launch site set to "LaunchPad".
   * Throws an exception if any of the games pre-flight checks fail.
   */
  void launch_vessel_from_vab(std::string name, bool recover);

  /**
   * Returns a list of vessels from the given craftDirectory
   * that can be launched.
   * @param craftDirectory Name of the directory in the current saves
   * "Ships" directory. For example "VAB" or "SPH".
   */
  std::vector<std::string> launchable_vessels(std::string craft_directory);

  /**
   * Load the game with the given name.
   * This will create a load a save file called name.sfs from the folder of the
   * current save game.
   * @param name Name of the save.
   */
  void load(std::string name);

  /**
   * Switch to the space center view.
   */
  void load_space_center();

  /**
   * Load a quicksave.
   *
   * This is the same as calling SpaceCenter::load with the name "quicksave".
   */
  void quickload();

  /**
   * Save a quicksave.
   *
   * This is the same as calling SpaceCenter::save with the name "quicksave".
   */
  void quicksave();

  /**
   * Cast a ray from a given position in a given direction, and return the distance to the hit point.
   * If no hit occurs, returns infinity.
   * @param position Position, as a vector, of the origin of the ray.
   * @param direction Direction of the ray, as a unit vector.
   * @param referenceFrame The reference frame that the position and direction are in.
   * @return The distance to the hit, in meters, or infinity if there was no hit.
   */
  double raycast_distance(std::tuple<double, double, double> position, std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame reference_frame);

  /**
   * Cast a ray from a given position in a given direction, and return the part that it hits.
   * If no hit occurs, returns null.
   * @param position Position, as a vector, of the origin of the ray.
   * @param direction Direction of the ray, as a unit vector.
   * @param referenceFrame The reference frame that the position and direction are in.
   * @return The part that was hit or null if there was no hit.
   */
  SpaceCenter::Part raycast_part(std::tuple<double, double, double> position, std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame reference_frame);

  /**
   * Revert the current flight to launch.
   */
  void revert_to_launch();

  /**
   * Save the game with a given name.
   * This will create a save file called name.sfs in the folder of the
   * current save game.
   * @param name Name of the save.
   */
  void save(std::string name);

  /**
   * Saves a screenshot.
   * @param filePath The path of the file to save.
   * @param scale Resolution scaling factor
   */
  void screenshot(std::string file_path, int32_t scale);

  /**
   * Transfers a crew member to a different part.
   * @param crewMember The crew member to transfer.
   * @param targetPart The part to move them to.
   */
  void transfer_crew(SpaceCenter::CrewMember crew_member, SpaceCenter::Part target_part);

  /**
   * Converts a direction from one reference frame to another.
   * @param direction Direction, as a vector, in reference frame
   * from.
   * @param from The reference frame that the direction is in.
   * @param to The reference frame to covert the direction to.
   * @return The corresponding direction, as a vector, in reference frame
   * to.
   */
  std::tuple<double, double, double> transform_direction(std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to);

  /**
   * Converts a position from one reference frame to another.
   * @param position Position, as a vector, in reference frame
   * from.
   * @param from The reference frame that the position is in.
   * @param to The reference frame to covert the position to.
   * @return The corresponding position, as a vector, in reference frame
   * to.
   */
  std::tuple<double, double, double> transform_position(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to);

  /**
   * Converts a rotation from one reference frame to another.
   * @param rotation Rotation, as a quaternion of the form (x, y, z, w),
   * in reference frame from.
   * @param from The reference frame that the rotation is in.
   * @param to The reference frame to covert the rotation to.
   * @return The corresponding rotation, as a quaternion of the form
   * (x, y, z, w), in reference frame to.
   */
  std::tuple<double, double, double, double> transform_rotation(std::tuple<double, double, double, double> rotation, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to);

  /**
   * Converts a velocity (acting at the specified position) from one reference frame
   * to another. The position is required to take the relative angular velocity of the
   * reference frames into account.
   * @param position Position, as a vector, in reference frame
   * from.
   * @param velocity Velocity, as a vector that points in the direction of travel and
   * whose magnitude is the speed in meters per second, in reference frame
   * from.
   * @param from The reference frame that the position and velocity are in.
   * @param to The reference frame to covert the velocity to.
   * @return The corresponding velocity, as a vector, in reference frame
   * to.
   */
  std::tuple<double, double, double> transform_velocity(std::tuple<double, double, double> position, std::tuple<double, double, double> velocity, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to);

  /**
   * Uses time acceleration to warp forward to a time in the future, specified
   * by universal time ut. This call blocks until the desired
   * time is reached. Uses regular "on-rails" or physical time warp as appropriate.
   * For example, physical time warp is used when the active vessel is traveling
   * through an atmosphere. When using regular "on-rails" time warp, the warp
   * rate is limited by maxRailsRate, and when using physical
   * time warp, the warp rate is limited by maxPhysicsRate.
   * @param ut The universal time to warp to, in seconds.
   * @param maxRailsRate The maximum warp rate in regular "on-rails" time warp.
   * @param maxPhysicsRate The maximum warp rate in physical time warp.
   * @return When the time warp is complete.
   */
  void warp_to(double ut, float max_rails_rate, float max_physics_rate);

  /**
   * The currently active vessel.
   */
  SpaceCenter::Vessel active_vessel();

  /**
   * The currently active vessel.
   */
  void set_active_vessel(SpaceCenter::Vessel value);

  /**
   * The alarm manager.
   */
  SpaceCenter::AlarmManager alarm_manager();

  /**
   * A dictionary of all celestial bodies (planets, moons, etc.) in the game,
   * keyed by the name of the body.
   */
  std::map<std::string, SpaceCenter::CelestialBody> bodies();

  /**
   * An object that can be used to control the camera.
   */
  SpaceCenter::Camera camera();

  /**
   * The contract manager.
   */
  SpaceCenter::ContractManager contract_manager();

  /**
   * Whether <a href="https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/">Ferram Aerospace Research</a> is installed.
   */
  bool far_available();

  /**
   * The current amount of funds.
   */
  double funds();

  /**
   * The value of the <a href="https://en.wikipedia.org/wiki/Gravitational_constant">
   * gravitational constant</a> G in N(m/kg)^2.
   */
  double g();

  /**
   * The current mode the game is in.
   */
  SpaceCenter::GameMode game_mode();

  /**
   * A list of available launch sites.
   */
  std::vector<SpaceCenter::LaunchSite> launch_sites();

  /**
   * The visible objects in map mode.
   */
  SpaceCenter::MapFilterType map_filter();

  /**
   * The visible objects in map mode.
   */
  void set_map_filter(SpaceCenter::MapFilterType value);

  /**
   * The current maximum regular "on-rails" warp factor that can be set.
   * A value between 0 and 7 inclusive. See
   * <a href="https://wiki.kerbalspaceprogram.com/wiki/Time_warp">the KSP wiki</a>
   * for details.
   */
  int32_t maximum_rails_warp_factor();

  /**
   * Whether the navball is visible.
   */
  bool navball();

  /**
   * Whether the navball is visible.
   */
  void set_navball(bool value);

  /**
   * The physical time warp rate. A value between 0 and 3 inclusive. 0 means
   * no time warp. Returns 0 if regular "on-rails" time warp is active.
   */
  int32_t physics_warp_factor();

  /**
   * The physical time warp rate. A value between 0 and 3 inclusive. 0 means
   * no time warp. Returns 0 if regular "on-rails" time warp is active.
   */
  void set_physics_warp_factor(int32_t value);

  /**
   * The time warp rate, using regular "on-rails" time warp. A value between
   * 0 and 7 inclusive. 0 means no time warp. Returns 0 if physical time warp
   * is active.
   *
   * If requested time warp factor cannot be set, it will be set to the next
   * lowest possible value. For example, if the vessel is too close to a
   * planet. See <a href="https://wiki.kerbalspaceprogram.com/wiki/Time_warp">
   * the KSP wiki</a> for details.
   */
  int32_t rails_warp_factor();

  /**
   * The time warp rate, using regular "on-rails" time warp. A value between
   * 0 and 7 inclusive. 0 means no time warp. Returns 0 if physical time warp
   * is active.
   *
   * If requested time warp factor cannot be set, it will be set to the next
   * lowest possible value. For example, if the vessel is too close to a
   * planet. See <a href="https://wiki.kerbalspaceprogram.com/wiki/Time_warp">
   * the KSP wiki</a> for details.
   */
  void set_rails_warp_factor(int32_t value);

  /**
   * The current amount of reputation.
   */
  float reputation();

  /**
   * The current amount of science.
   */
  float science();

  /**
   * The currently targeted celestial body.
   */
  SpaceCenter::CelestialBody target_body();

  /**
   * The currently targeted celestial body.
   */
  void set_target_body(SpaceCenter::CelestialBody value);

  /**
   * The currently targeted docking port.
   */
  SpaceCenter::DockingPort target_docking_port();

  /**
   * The currently targeted docking port.
   */
  void set_target_docking_port(SpaceCenter::DockingPort value);

  /**
   * The currently targeted vessel.
   */
  SpaceCenter::Vessel target_vessel();

  /**
   * The currently targeted vessel.
   */
  void set_target_vessel(SpaceCenter::Vessel value);

  /**
   * Whether the UI is visible.
   */
  bool ui_visible();

  /**
   * Whether the UI is visible.
   */
  void set_ui_visible(bool value);

  /**
   * The current universal time in seconds.
   */
  double ut();

  /**
   * A list of all the vessels in the game.
   */
  std::vector<SpaceCenter::Vessel> vessels();

  /**
   * The current warp factor. This is the index of the rate at which time
   * is passing for either regular "on-rails" or physical time warp. Returns 0
   * if time warp is not active. When in on-rails time warp, this is equal to
   * SpaceCenter::rails_warp_factor, and in physics time warp, this is equal to
   * SpaceCenter::physics_warp_factor.
   */
  float warp_factor();

  /**
   * The current time warp mode. Returns SpaceCenter::WarpMode::none if time
   * warp is not active, SpaceCenter::WarpMode::rails if regular "on-rails" time warp
   * is active, or SpaceCenter::WarpMode::physics if physical time warp is active.
   */
  SpaceCenter::WarpMode warp_mode();

  /**
   * The current warp rate. This is the rate at which time is passing for
   * either on-rails or physical time warp. For example, a value of 10 means
   * time is passing 10x faster than normal. Returns 1 if time warp is not
   * active.
   */
  float warp_rate();

  /**
   * The waypoint manager.
   */
  SpaceCenter::WaypointManager waypoint_manager();

  ::krpc::Stream<bool> can_rails_warp_at_stream(int32_t factor);

  ::krpc::Stream<bool> can_revert_to_launch_stream();

  ::krpc::Stream<SpaceCenter::CrewMember> get_kerbal_stream(std::string name);

  ::krpc::Stream<std::vector<std::string>> launchable_vessels_stream(std::string craft_directory);

  ::krpc::Stream<double> raycast_distance_stream(std::tuple<double, double, double> position, std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame reference_frame);

  ::krpc::Stream<SpaceCenter::Part> raycast_part_stream(std::tuple<double, double, double> position, std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame reference_frame);

  ::krpc::Stream<std::tuple<double, double, double>> transform_direction_stream(std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to);

  ::krpc::Stream<std::tuple<double, double, double>> transform_position_stream(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to);

  ::krpc::Stream<std::tuple<double, double, double, double>> transform_rotation_stream(std::tuple<double, double, double, double> rotation, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to);

  ::krpc::Stream<std::tuple<double, double, double>> transform_velocity_stream(std::tuple<double, double, double> position, std::tuple<double, double, double> velocity, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to);

  ::krpc::Stream<SpaceCenter::Vessel> active_vessel_stream();

  ::krpc::Stream<SpaceCenter::AlarmManager> alarm_manager_stream();

  ::krpc::Stream<std::map<std::string, SpaceCenter::CelestialBody>> bodies_stream();

  ::krpc::Stream<SpaceCenter::Camera> camera_stream();

  ::krpc::Stream<SpaceCenter::ContractManager> contract_manager_stream();

  ::krpc::Stream<bool> far_available_stream();

  ::krpc::Stream<double> funds_stream();

  ::krpc::Stream<double> g_stream();

  ::krpc::Stream<SpaceCenter::GameMode> game_mode_stream();

  ::krpc::Stream<std::vector<SpaceCenter::LaunchSite>> launch_sites_stream();

  ::krpc::Stream<SpaceCenter::MapFilterType> map_filter_stream();

  ::krpc::Stream<int32_t> maximum_rails_warp_factor_stream();

  ::krpc::Stream<bool> navball_stream();

  ::krpc::Stream<int32_t> physics_warp_factor_stream();

  ::krpc::Stream<int32_t> rails_warp_factor_stream();

  ::krpc::Stream<float> reputation_stream();

  ::krpc::Stream<float> science_stream();

  ::krpc::Stream<SpaceCenter::CelestialBody> target_body_stream();

  ::krpc::Stream<SpaceCenter::DockingPort> target_docking_port_stream();

  ::krpc::Stream<SpaceCenter::Vessel> target_vessel_stream();

  ::krpc::Stream<bool> ui_visible_stream();

  ::krpc::Stream<double> ut_stream();

  ::krpc::Stream<std::vector<SpaceCenter::Vessel>> vessels_stream();

  ::krpc::Stream<float> warp_factor_stream();

  ::krpc::Stream<SpaceCenter::WarpMode> warp_mode_stream();

  ::krpc::Stream<float> warp_rate_stream();

  ::krpc::Stream<SpaceCenter::WaypointManager> waypoint_manager_stream();

  ::krpc::schema::ProcedureCall can_rails_warp_at_call(int32_t factor);

  ::krpc::schema::ProcedureCall can_revert_to_launch_call();

  ::krpc::schema::ProcedureCall clear_target_call();

  ::krpc::schema::ProcedureCall create_kerbal_call(std::string name, std::string job, bool male);

  ::krpc::schema::ProcedureCall get_kerbal_call(std::string name);

  ::krpc::schema::ProcedureCall launch_vessel_call(std::string craft_directory, std::string name, std::string launch_site, bool recover, std::vector<std::string> crew, std::string flag_url);

  ::krpc::schema::ProcedureCall launch_vessel_from_sph_call(std::string name, bool recover);

  ::krpc::schema::ProcedureCall launch_vessel_from_vab_call(std::string name, bool recover);

  ::krpc::schema::ProcedureCall launchable_vessels_call(std::string craft_directory);

  ::krpc::schema::ProcedureCall load_call(std::string name);

  ::krpc::schema::ProcedureCall load_space_center_call();

  ::krpc::schema::ProcedureCall quickload_call();

  ::krpc::schema::ProcedureCall quicksave_call();

  ::krpc::schema::ProcedureCall raycast_distance_call(std::tuple<double, double, double> position, std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame reference_frame);

  ::krpc::schema::ProcedureCall raycast_part_call(std::tuple<double, double, double> position, std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame reference_frame);

  ::krpc::schema::ProcedureCall revert_to_launch_call();

  ::krpc::schema::ProcedureCall save_call(std::string name);

  ::krpc::schema::ProcedureCall screenshot_call(std::string file_path, int32_t scale);

  ::krpc::schema::ProcedureCall transfer_crew_call(SpaceCenter::CrewMember crew_member, SpaceCenter::Part target_part);

  ::krpc::schema::ProcedureCall transform_direction_call(std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to);

  ::krpc::schema::ProcedureCall transform_position_call(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to);

  ::krpc::schema::ProcedureCall transform_rotation_call(std::tuple<double, double, double, double> rotation, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to);

  ::krpc::schema::ProcedureCall transform_velocity_call(std::tuple<double, double, double> position, std::tuple<double, double, double> velocity, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to);

  ::krpc::schema::ProcedureCall warp_to_call(double ut, float max_rails_rate, float max_physics_rate);

  ::krpc::schema::ProcedureCall active_vessel_call();

  ::krpc::schema::ProcedureCall set_active_vessel_call(SpaceCenter::Vessel value);

  ::krpc::schema::ProcedureCall alarm_manager_call();

  ::krpc::schema::ProcedureCall bodies_call();

  ::krpc::schema::ProcedureCall camera_call();

  ::krpc::schema::ProcedureCall contract_manager_call();

  ::krpc::schema::ProcedureCall far_available_call();

  ::krpc::schema::ProcedureCall funds_call();

  ::krpc::schema::ProcedureCall g_call();

  ::krpc::schema::ProcedureCall game_mode_call();

  ::krpc::schema::ProcedureCall launch_sites_call();

  ::krpc::schema::ProcedureCall map_filter_call();

  ::krpc::schema::ProcedureCall set_map_filter_call(SpaceCenter::MapFilterType value);

  ::krpc::schema::ProcedureCall maximum_rails_warp_factor_call();

  ::krpc::schema::ProcedureCall navball_call();

  ::krpc::schema::ProcedureCall set_navball_call(bool value);

  ::krpc::schema::ProcedureCall physics_warp_factor_call();

  ::krpc::schema::ProcedureCall set_physics_warp_factor_call(int32_t value);

  ::krpc::schema::ProcedureCall rails_warp_factor_call();

  ::krpc::schema::ProcedureCall set_rails_warp_factor_call(int32_t value);

  ::krpc::schema::ProcedureCall reputation_call();

  ::krpc::schema::ProcedureCall science_call();

  ::krpc::schema::ProcedureCall target_body_call();

  ::krpc::schema::ProcedureCall set_target_body_call(SpaceCenter::CelestialBody value);

  ::krpc::schema::ProcedureCall target_docking_port_call();

  ::krpc::schema::ProcedureCall set_target_docking_port_call(SpaceCenter::DockingPort value);

  ::krpc::schema::ProcedureCall target_vessel_call();

  ::krpc::schema::ProcedureCall set_target_vessel_call(SpaceCenter::Vessel value);

  ::krpc::schema::ProcedureCall ui_visible_call();

  ::krpc::schema::ProcedureCall set_ui_visible_call(bool value);

  ::krpc::schema::ProcedureCall ut_call();

  ::krpc::schema::ProcedureCall vessels_call();

  ::krpc::schema::ProcedureCall warp_factor_call();

  ::krpc::schema::ProcedureCall warp_mode_call();

  ::krpc::schema::ProcedureCall warp_rate_call();

  ::krpc::schema::ProcedureCall waypoint_manager_call();

  /**
   * An alarm. Can be accessed using SpaceCenter::alarm_manager.
   */
  class Alarm : public krpc::Object<Alarm> {
   public:
    explicit Alarm(Client* client = nullptr, uint64_t id = 0);

    /**
     * Description of the alarm.
     */
    std::string description();

    /**
     * Seconds between the alarm going off and the event it references.
     */
    double event_offset();

    /**
     * Unique identifier of the alarm.
     * KSP destroys and recreates an alarm when it is edited.
     * This id will remain constant between the old and new alarms.
     */
    uint32_t id();

    /**
     * Time the alarm will trigger.
     */
    double time();

    /**
     * Time until the alarm triggers.
     */
    double time_until();

    /**
     * Title of the alarm
     */
    std::string title();

    /**
     * Type of alarm
     */
    std::string type();

    /**
     * Vessel the alarm references. null if it does not reference a vessel.
     */
    SpaceCenter::Vessel vessel();

    ::krpc::Stream<std::string> description_stream();

    ::krpc::Stream<double> event_offset_stream();

    ::krpc::Stream<uint32_t> id_stream();

    ::krpc::Stream<double> time_stream();

    ::krpc::Stream<double> time_until_stream();

    ::krpc::Stream<std::string> title_stream();

    ::krpc::Stream<std::string> type_stream();

    ::krpc::Stream<SpaceCenter::Vessel> vessel_stream();

    ::krpc::schema::ProcedureCall description_call();

    ::krpc::schema::ProcedureCall event_offset_call();

    ::krpc::schema::ProcedureCall id_call();

    ::krpc::schema::ProcedureCall time_call();

    ::krpc::schema::ProcedureCall time_until_call();

    ::krpc::schema::ProcedureCall title_call();

    ::krpc::schema::ProcedureCall type_call();

    ::krpc::schema::ProcedureCall vessel_call();
  };

  /**
   * Alarm manager.
   * Obtained by calling SpaceCenter::alarm_manager.
   */
  class AlarmManager : public krpc::Object<AlarmManager> {
   public:
    explicit AlarmManager(Client* client = nullptr, uint64_t id = 0);

    /**
     * Create an alarm.
     * @param time Number of seconds from now that the alarm should trigger.
     * @param title Title for the alarm.
     * @param description Description for the alarm.
     */
    static SpaceCenter::Alarm add_alarm(Client& client, double time, std::string title, std::string description);

    /**
     * Create an alarm for the given vessel's next apoapsis.
     * @param vessel The vessel.
     * @param offset Time in seconds to offset the alarm by.
     * @param title Title for the alarm.
     * @param description Description for the alarm.
     */
    static SpaceCenter::Alarm add_apoapsis_alarm(Client& client, SpaceCenter::Vessel vessel, double offset, std::string title, std::string description);

    /**
     * Create an alarm for the given vessel and maneuver node.
     * @param vessel The vessel.
     * @param node The maneuver node.
     * @param offset Time in seconds to offset the alarm by.
     * @param addBurnTime Whether the node's burn time should be included in the alarm.
     * @param title Title for the alarm.
     * @param description Description for the alarm.
     */
    static SpaceCenter::Alarm add_maneuver_node_alarm(Client& client, SpaceCenter::Vessel vessel, SpaceCenter::Node node, double offset, bool add_burn_time, std::string title, std::string description);

    /**
     * Create an alarm for the given vessel's next periapsis.
     * @param vessel The vessel.
     * @param offset Time in seconds to offset the alarm by.
     * @param title Title for the alarm.
     * @param description Description for the alarm.
     */
    static SpaceCenter::Alarm add_periapsis_alarm(Client& client, SpaceCenter::Vessel vessel, double offset, std::string title, std::string description);

    /**
     * Create an alarm for the given vessel's next sphere of influence change.
     * @param vessel The vessel.
     * @param offset Time in seconds to offset the alarm by.
     * @param title Title for the alarm.
     * @param description Description for the alarm.
     */
    static SpaceCenter::Alarm add_soi_alarm(Client& client, SpaceCenter::Vessel vessel, double offset, std::string title, std::string description);

    /**
     * Create an alarm linked to a vessel.
     * @param time Number of seconds from now that the alarm should trigger.
     * @param vessel Vessel to link the alarm to.
     * @param title Title for the alarm.
     * @param description Description for the alarm.
     */
    static SpaceCenter::Alarm add_vessel_alarm(Client& client, double time, SpaceCenter::Vessel vessel, std::string title, std::string description);

    /**
     * A list of all alarms.
     */
    std::vector<SpaceCenter::Alarm> alarms();

    static ::krpc::Stream<SpaceCenter::Alarm> add_alarm_stream(Client& client, double time, std::string title, std::string description);

    static ::krpc::Stream<SpaceCenter::Alarm> add_apoapsis_alarm_stream(Client& client, SpaceCenter::Vessel vessel, double offset, std::string title, std::string description);

    static ::krpc::Stream<SpaceCenter::Alarm> add_maneuver_node_alarm_stream(Client& client, SpaceCenter::Vessel vessel, SpaceCenter::Node node, double offset, bool add_burn_time, std::string title, std::string description);

    static ::krpc::Stream<SpaceCenter::Alarm> add_periapsis_alarm_stream(Client& client, SpaceCenter::Vessel vessel, double offset, std::string title, std::string description);

    static ::krpc::Stream<SpaceCenter::Alarm> add_soi_alarm_stream(Client& client, SpaceCenter::Vessel vessel, double offset, std::string title, std::string description);

    static ::krpc::Stream<SpaceCenter::Alarm> add_vessel_alarm_stream(Client& client, double time, SpaceCenter::Vessel vessel, std::string title, std::string description);

    ::krpc::Stream<std::vector<SpaceCenter::Alarm>> alarms_stream();

    static ::krpc::schema::ProcedureCall add_alarm_call(Client& client, double time, std::string title, std::string description);

    static ::krpc::schema::ProcedureCall add_apoapsis_alarm_call(Client& client, SpaceCenter::Vessel vessel, double offset, std::string title, std::string description);

    static ::krpc::schema::ProcedureCall add_maneuver_node_alarm_call(Client& client, SpaceCenter::Vessel vessel, SpaceCenter::Node node, double offset, bool add_burn_time, std::string title, std::string description);

    static ::krpc::schema::ProcedureCall add_periapsis_alarm_call(Client& client, SpaceCenter::Vessel vessel, double offset, std::string title, std::string description);

    static ::krpc::schema::ProcedureCall add_soi_alarm_call(Client& client, SpaceCenter::Vessel vessel, double offset, std::string title, std::string description);

    static ::krpc::schema::ProcedureCall add_vessel_alarm_call(Client& client, double time, SpaceCenter::Vessel vessel, std::string title, std::string description);

    ::krpc::schema::ProcedureCall alarms_call();
  };

  /**
   * An antenna. Obtained by calling SpaceCenter::Part::antenna.
   */
  class Antenna : public krpc::Object<Antenna> {
   public:
    explicit Antenna(Client* client = nullptr, uint64_t id = 0);

    /**
     * Cancel current transmission of data.
     */
    void cancel();

    /**
     * Transmit data.
     */
    void transmit();

    /**
     * Whether partial data transmission is permitted.
     */
    bool allow_partial();

    /**
     * Whether partial data transmission is permitted.
     */
    void set_allow_partial(bool value);

    /**
     * Whether data can be transmitted by this antenna.
     */
    bool can_transmit();

    /**
     * Whether the antenna can be combined with other antennae on the vessel
     * to boost the power.
     */
    bool combinable();

    /**
     * Exponent used to calculate the combined power of multiple antennae on a vessel.
     */
    double combinable_exponent();

    /**
     * Whether the antenna is deployable.
     */
    bool deployable();

    /**
     * Whether the antenna is deployed.
     *
     * Fixed antennas are always deployed.
     * Returns an error if you try to deploy a fixed antenna.
     */
    bool deployed();

    /**
     * Whether the antenna is deployed.
     *
     * Fixed antennas are always deployed.
     * Returns an error if you try to deploy a fixed antenna.
     */
    void set_deployed(bool value);

    /**
     * Interval between sending packets in seconds.
     */
    float packet_interval();

    /**
     * Units of electric charge consumed per packet sent.
     */
    double packet_resource_cost();

    /**
     * Amount of data sent per packet in Mits.
     */
    float packet_size();

    /**
     * The part object for this antenna.
     */
    SpaceCenter::Part part();

    /**
     * The power of the antenna.
     */
    double power();

    /**
     * The current state of the antenna.
     */
    SpaceCenter::AntennaState state();

    ::krpc::Stream<bool> allow_partial_stream();

    ::krpc::Stream<bool> can_transmit_stream();

    ::krpc::Stream<bool> combinable_stream();

    ::krpc::Stream<double> combinable_exponent_stream();

    ::krpc::Stream<bool> deployable_stream();

    ::krpc::Stream<bool> deployed_stream();

    ::krpc::Stream<float> packet_interval_stream();

    ::krpc::Stream<double> packet_resource_cost_stream();

    ::krpc::Stream<float> packet_size_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<double> power_stream();

    ::krpc::Stream<SpaceCenter::AntennaState> state_stream();

    ::krpc::schema::ProcedureCall cancel_call();

    ::krpc::schema::ProcedureCall transmit_call();

    ::krpc::schema::ProcedureCall allow_partial_call();

    ::krpc::schema::ProcedureCall set_allow_partial_call(bool value);

    ::krpc::schema::ProcedureCall can_transmit_call();

    ::krpc::schema::ProcedureCall combinable_call();

    ::krpc::schema::ProcedureCall combinable_exponent_call();

    ::krpc::schema::ProcedureCall deployable_call();

    ::krpc::schema::ProcedureCall deployed_call();

    ::krpc::schema::ProcedureCall set_deployed_call(bool value);

    ::krpc::schema::ProcedureCall packet_interval_call();

    ::krpc::schema::ProcedureCall packet_resource_cost_call();

    ::krpc::schema::ProcedureCall packet_size_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall power_call();

    ::krpc::schema::ProcedureCall state_call();
  };

  /**
   * Provides basic auto-piloting utilities for a vessel.
   * Created by calling SpaceCenter::Vessel::auto_pilot.
   *
   * If a client engages the auto-pilot and then closes its connection to the server,
   * the auto-pilot will be disengaged and its target reference frame, direction and roll
   * reset to default.
   */
  class AutoPilot : public krpc::Object<AutoPilot> {
   public:
    explicit AutoPilot(Client* client = nullptr, uint64_t id = 0);

    /**
     * Disengage the auto-pilot.
     */
    void disengage();

    /**
     * Engage the auto-pilot.
     */
    void engage();

    /**
     * Set target pitch and heading angles.
     * @param pitch Target pitch angle, in degrees between -90° and +90°.
     * @param heading Target heading angle, in degrees between 0° and 360°.
     */
    void target_pitch_and_heading(float pitch, float heading);

    /**
     * Blocks until the vessel is pointing in the target direction and has
     * the target roll (if set). Throws an exception if the auto-pilot has not been engaged.
     */
    void wait();

    /**
     * The angle at which the autopilot considers the vessel to be pointing
     * close to the target.
     * This determines the midpoint of the target velocity attenuation function.
     * A vector of three angles, in degrees, one for each of the pitch, roll and yaw axes.
     * Defaults to 1° for each axis.
     */
    std::tuple<double, double, double> attenuation_angle();

    /**
     * The angle at which the autopilot considers the vessel to be pointing
     * close to the target.
     * This determines the midpoint of the target velocity attenuation function.
     * A vector of three angles, in degrees, one for each of the pitch, roll and yaw axes.
     * Defaults to 1° for each axis.
     */
    void set_attenuation_angle(std::tuple<double, double, double> value);

    /**
     * Whether the rotation rate controllers PID parameters should be automatically tuned
     * using the vessels moment of inertia and available torque. Defaults to true.
     * See SpaceCenter::AutoPilot::time_to_peak and SpaceCenter::AutoPilot::overshoot.
     */
    bool auto_tune();

    /**
     * Whether the rotation rate controllers PID parameters should be automatically tuned
     * using the vessels moment of inertia and available torque. Defaults to true.
     * See SpaceCenter::AutoPilot::time_to_peak and SpaceCenter::AutoPilot::overshoot.
     */
    void set_auto_tune(bool value);

    /**
     * The time the vessel should take to come to a stop pointing in the target direction.
     * This determines the angular acceleration used to decelerate the vessel.
     * A vector of three times, in seconds, one for each of the pitch, roll and yaw axes.
     * Defaults to 5 seconds for each axis.
     */
    std::tuple<double, double, double> deceleration_time();

    /**
     * The time the vessel should take to come to a stop pointing in the target direction.
     * This determines the angular acceleration used to decelerate the vessel.
     * A vector of three times, in seconds, one for each of the pitch, roll and yaw axes.
     * Defaults to 5 seconds for each axis.
     */
    void set_deceleration_time(std::tuple<double, double, double> value);

    /**
     * The error, in degrees, between the direction the ship has been asked
     * to point in and the direction it is pointing in. Throws an exception if the auto-pilot
     * has not been engaged and SAS is not enabled or is in stability assist mode.
     */
    float error();

    /**
     * The error, in degrees, between the vessels current and target heading.
     * Throws an exception if the auto-pilot has not been engaged.
     */
    float heading_error();

    /**
     * The target overshoot percentage used to autotune the PID controllers.
     * A vector of three values, between 0 and 1, for each of the pitch, roll and yaw axes.
     * Defaults to 0.01 for each axis.
     */
    std::tuple<double, double, double> overshoot();

    /**
     * The target overshoot percentage used to autotune the PID controllers.
     * A vector of three values, between 0 and 1, for each of the pitch, roll and yaw axes.
     * Defaults to 0.01 for each axis.
     */
    void set_overshoot(std::tuple<double, double, double> value);

    /**
     * The error, in degrees, between the vessels current and target pitch.
     * Throws an exception if the auto-pilot has not been engaged.
     */
    float pitch_error();

    /**
     * Gains for the pitch PID controller.
     *
     * When SpaceCenter::AutoPilot::auto_tune is true, these values are updated automatically,
     * which will overwrite any manual changes.
     */
    std::tuple<double, double, double> pitch_pid_gains();

    /**
     * Gains for the pitch PID controller.
     *
     * When SpaceCenter::AutoPilot::auto_tune is true, these values are updated automatically,
     * which will overwrite any manual changes.
     */
    void set_pitch_pid_gains(std::tuple<double, double, double> value);

    /**
     * The reference frame for the target direction (SpaceCenter::AutoPilot::target_direction).
     *
     * An error will be thrown if this property is set to a reference frame that rotates with
     * the vessel being controlled, as it is impossible to rotate the vessel in such a
     * reference frame.
     */
    SpaceCenter::ReferenceFrame reference_frame();

    /**
     * The reference frame for the target direction (SpaceCenter::AutoPilot::target_direction).
     *
     * An error will be thrown if this property is set to a reference frame that rotates with
     * the vessel being controlled, as it is impossible to rotate the vessel in such a
     * reference frame.
     */
    void set_reference_frame(SpaceCenter::ReferenceFrame value);

    /**
     * The error, in degrees, between the vessels current and target roll.
     * Throws an exception if the auto-pilot has not been engaged or no target roll is set.
     */
    float roll_error();

    /**
     * Gains for the roll PID controller.
     *
     * When SpaceCenter::AutoPilot::auto_tune is true, these values are updated automatically,
     * which will overwrite any manual changes.
     */
    std::tuple<double, double, double> roll_pid_gains();

    /**
     * Gains for the roll PID controller.
     *
     * When SpaceCenter::AutoPilot::auto_tune is true, these values are updated automatically,
     * which will overwrite any manual changes.
     */
    void set_roll_pid_gains(std::tuple<double, double, double> value);

    /**
     * The threshold at which the autopilot will try to match the target roll angle, if any.
     * Defaults to 5 degrees.
     */
    double roll_threshold();

    /**
     * The threshold at which the autopilot will try to match the target roll angle, if any.
     * Defaults to 5 degrees.
     */
    void set_roll_threshold(double value);

    /**
     * The state of SAS.
     *
     * Equivalent to SpaceCenter::Control::sas
     */
    bool sas();

    /**
     * The state of SAS.
     *
     * Equivalent to SpaceCenter::Control::sas
     */
    void set_sas(bool value);

    /**
     * The current SpaceCenter::SASMode.
     * These modes are equivalent to the mode buttons to the left of the navball that appear
     * when SAS is enabled.
     *
     * Equivalent to SpaceCenter::Control::sas_mode
     */
    SpaceCenter::SASMode sas_mode();

    /**
     * The current SpaceCenter::SASMode.
     * These modes are equivalent to the mode buttons to the left of the navball that appear
     * when SAS is enabled.
     *
     * Equivalent to SpaceCenter::Control::sas_mode
     */
    void set_sas_mode(SpaceCenter::SASMode value);

    /**
     * The maximum amount of time that the vessel should need to come to a complete stop.
     * This determines the maximum angular velocity of the vessel.
     * A vector of three stopping times, in seconds, one for each of the pitch, roll
     * and yaw axes. Defaults to 0.5 seconds for each axis.
     */
    std::tuple<double, double, double> stopping_time();

    /**
     * The maximum amount of time that the vessel should need to come to a complete stop.
     * This determines the maximum angular velocity of the vessel.
     * A vector of three stopping times, in seconds, one for each of the pitch, roll
     * and yaw axes. Defaults to 0.5 seconds for each axis.
     */
    void set_stopping_time(std::tuple<double, double, double> value);

    /**
     * Direction vector corresponding to the target pitch and heading.
     * This is in the reference frame specified by SpaceCenter::ReferenceFrame.
     */
    std::tuple<double, double, double> target_direction();

    /**
     * Direction vector corresponding to the target pitch and heading.
     * This is in the reference frame specified by SpaceCenter::ReferenceFrame.
     */
    void set_target_direction(std::tuple<double, double, double> value);

    /**
     * The target heading, in degrees, between 0° and 360°.
     */
    float target_heading();

    /**
     * The target heading, in degrees, between 0° and 360°.
     */
    void set_target_heading(float value);

    /**
     * The target pitch, in degrees, between -90° and +90°.
     */
    float target_pitch();

    /**
     * The target pitch, in degrees, between -90° and +90°.
     */
    void set_target_pitch(float value);

    /**
     * The target roll, in degrees. NaN if no target roll is set.
     */
    float target_roll();

    /**
     * The target roll, in degrees. NaN if no target roll is set.
     */
    void set_target_roll(float value);

    /**
     * The target time to peak used to autotune the PID controllers.
     * A vector of three times, in seconds, for each of the pitch, roll and yaw axes.
     * Defaults to 3 seconds for each axis.
     */
    std::tuple<double, double, double> time_to_peak();

    /**
     * The target time to peak used to autotune the PID controllers.
     * A vector of three times, in seconds, for each of the pitch, roll and yaw axes.
     * Defaults to 3 seconds for each axis.
     */
    void set_time_to_peak(std::tuple<double, double, double> value);

    /**
     * Gains for the yaw PID controller.
     *
     * When SpaceCenter::AutoPilot::auto_tune is true, these values are updated automatically,
     * which will overwrite any manual changes.
     */
    std::tuple<double, double, double> yaw_pid_gains();

    /**
     * Gains for the yaw PID controller.
     *
     * When SpaceCenter::AutoPilot::auto_tune is true, these values are updated automatically,
     * which will overwrite any manual changes.
     */
    void set_yaw_pid_gains(std::tuple<double, double, double> value);

    ::krpc::Stream<std::tuple<double, double, double>> attenuation_angle_stream();

    ::krpc::Stream<bool> auto_tune_stream();

    ::krpc::Stream<std::tuple<double, double, double>> deceleration_time_stream();

    ::krpc::Stream<float> error_stream();

    ::krpc::Stream<float> heading_error_stream();

    ::krpc::Stream<std::tuple<double, double, double>> overshoot_stream();

    ::krpc::Stream<float> pitch_error_stream();

    ::krpc::Stream<std::tuple<double, double, double>> pitch_pid_gains_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> reference_frame_stream();

    ::krpc::Stream<float> roll_error_stream();

    ::krpc::Stream<std::tuple<double, double, double>> roll_pid_gains_stream();

    ::krpc::Stream<double> roll_threshold_stream();

    ::krpc::Stream<bool> sas_stream();

    ::krpc::Stream<SpaceCenter::SASMode> sas_mode_stream();

    ::krpc::Stream<std::tuple<double, double, double>> stopping_time_stream();

    ::krpc::Stream<std::tuple<double, double, double>> target_direction_stream();

    ::krpc::Stream<float> target_heading_stream();

    ::krpc::Stream<float> target_pitch_stream();

    ::krpc::Stream<float> target_roll_stream();

    ::krpc::Stream<std::tuple<double, double, double>> time_to_peak_stream();

    ::krpc::Stream<std::tuple<double, double, double>> yaw_pid_gains_stream();

    ::krpc::schema::ProcedureCall disengage_call();

    ::krpc::schema::ProcedureCall engage_call();

    ::krpc::schema::ProcedureCall target_pitch_and_heading_call(float pitch, float heading);

    ::krpc::schema::ProcedureCall wait_call();

    ::krpc::schema::ProcedureCall attenuation_angle_call();

    ::krpc::schema::ProcedureCall set_attenuation_angle_call(std::tuple<double, double, double> value);

    ::krpc::schema::ProcedureCall auto_tune_call();

    ::krpc::schema::ProcedureCall set_auto_tune_call(bool value);

    ::krpc::schema::ProcedureCall deceleration_time_call();

    ::krpc::schema::ProcedureCall set_deceleration_time_call(std::tuple<double, double, double> value);

    ::krpc::schema::ProcedureCall error_call();

    ::krpc::schema::ProcedureCall heading_error_call();

    ::krpc::schema::ProcedureCall overshoot_call();

    ::krpc::schema::ProcedureCall set_overshoot_call(std::tuple<double, double, double> value);

    ::krpc::schema::ProcedureCall pitch_error_call();

    ::krpc::schema::ProcedureCall pitch_pid_gains_call();

    ::krpc::schema::ProcedureCall set_pitch_pid_gains_call(std::tuple<double, double, double> value);

    ::krpc::schema::ProcedureCall reference_frame_call();

    ::krpc::schema::ProcedureCall set_reference_frame_call(SpaceCenter::ReferenceFrame value);

    ::krpc::schema::ProcedureCall roll_error_call();

    ::krpc::schema::ProcedureCall roll_pid_gains_call();

    ::krpc::schema::ProcedureCall set_roll_pid_gains_call(std::tuple<double, double, double> value);

    ::krpc::schema::ProcedureCall roll_threshold_call();

    ::krpc::schema::ProcedureCall set_roll_threshold_call(double value);

    ::krpc::schema::ProcedureCall sas_call();

    ::krpc::schema::ProcedureCall set_sas_call(bool value);

    ::krpc::schema::ProcedureCall sas_mode_call();

    ::krpc::schema::ProcedureCall set_sas_mode_call(SpaceCenter::SASMode value);

    ::krpc::schema::ProcedureCall stopping_time_call();

    ::krpc::schema::ProcedureCall set_stopping_time_call(std::tuple<double, double, double> value);

    ::krpc::schema::ProcedureCall target_direction_call();

    ::krpc::schema::ProcedureCall set_target_direction_call(std::tuple<double, double, double> value);

    ::krpc::schema::ProcedureCall target_heading_call();

    ::krpc::schema::ProcedureCall set_target_heading_call(float value);

    ::krpc::schema::ProcedureCall target_pitch_call();

    ::krpc::schema::ProcedureCall set_target_pitch_call(float value);

    ::krpc::schema::ProcedureCall target_roll_call();

    ::krpc::schema::ProcedureCall set_target_roll_call(float value);

    ::krpc::schema::ProcedureCall time_to_peak_call();

    ::krpc::schema::ProcedureCall set_time_to_peak_call(std::tuple<double, double, double> value);

    ::krpc::schema::ProcedureCall yaw_pid_gains_call();

    ::krpc::schema::ProcedureCall set_yaw_pid_gains_call(std::tuple<double, double, double> value);
  };

  /**
   * Controls the game's camera.
   * Obtained by calling SpaceCenter::camera.
   */
  class Camera : public krpc::Object<Camera> {
   public:
    explicit Camera(Client* client = nullptr, uint64_t id = 0);

    /**
     * Default distance from the camera to the subject, in meters.
     */
    float default_distance();

    /**
     * The distance from the camera to the subject, in meters.
     * A value between SpaceCenter::Camera::min_distance and SpaceCenter::Camera::max_distance.
     */
    float distance();

    /**
     * The distance from the camera to the subject, in meters.
     * A value between SpaceCenter::Camera::min_distance and SpaceCenter::Camera::max_distance.
     */
    void set_distance(float value);

    /**
     * In map mode, the celestial body that the camera is focussed on.
     * Returns null if the camera is not focussed on a celestial body.
     * Returns an error is the camera is not in map mode.
     */
    SpaceCenter::CelestialBody focussed_body();

    /**
     * In map mode, the celestial body that the camera is focussed on.
     * Returns null if the camera is not focussed on a celestial body.
     * Returns an error is the camera is not in map mode.
     */
    void set_focussed_body(SpaceCenter::CelestialBody value);

    /**
     * In map mode, the maneuver node that the camera is focussed on.
     * Returns null if the camera is not focussed on a maneuver node.
     * Returns an error is the camera is not in map mode.
     */
    SpaceCenter::Node focussed_node();

    /**
     * In map mode, the maneuver node that the camera is focussed on.
     * Returns null if the camera is not focussed on a maneuver node.
     * Returns an error is the camera is not in map mode.
     */
    void set_focussed_node(SpaceCenter::Node value);

    /**
     * In map mode, the vessel that the camera is focussed on.
     * Returns null if the camera is not focussed on a vessel.
     * Returns an error is the camera is not in map mode.
     */
    SpaceCenter::Vessel focussed_vessel();

    /**
     * In map mode, the vessel that the camera is focussed on.
     * Returns null if the camera is not focussed on a vessel.
     * Returns an error is the camera is not in map mode.
     */
    void set_focussed_vessel(SpaceCenter::Vessel value);

    /**
     * The heading of the camera, in degrees.
     */
    float heading();

    /**
     * The heading of the camera, in degrees.
     */
    void set_heading(float value);

    /**
     * Maximum distance from the camera to the subject, in meters.
     */
    float max_distance();

    /**
     * The maximum pitch of the camera.
     */
    float max_pitch();

    /**
     * Minimum distance from the camera to the subject, in meters.
     */
    float min_distance();

    /**
     * The minimum pitch of the camera.
     */
    float min_pitch();

    /**
     * The current mode of the camera.
     */
    SpaceCenter::CameraMode mode();

    /**
     * The current mode of the camera.
     */
    void set_mode(SpaceCenter::CameraMode value);

    /**
     * The pitch of the camera, in degrees.
     * A value between SpaceCenter::Camera::min_pitch and SpaceCenter::Camera::max_pitch
     */
    float pitch();

    /**
     * The pitch of the camera, in degrees.
     * A value between SpaceCenter::Camera::min_pitch and SpaceCenter::Camera::max_pitch
     */
    void set_pitch(float value);

    ::krpc::Stream<float> default_distance_stream();

    ::krpc::Stream<float> distance_stream();

    ::krpc::Stream<SpaceCenter::CelestialBody> focussed_body_stream();

    ::krpc::Stream<SpaceCenter::Node> focussed_node_stream();

    ::krpc::Stream<SpaceCenter::Vessel> focussed_vessel_stream();

    ::krpc::Stream<float> heading_stream();

    ::krpc::Stream<float> max_distance_stream();

    ::krpc::Stream<float> max_pitch_stream();

    ::krpc::Stream<float> min_distance_stream();

    ::krpc::Stream<float> min_pitch_stream();

    ::krpc::Stream<SpaceCenter::CameraMode> mode_stream();

    ::krpc::Stream<float> pitch_stream();

    ::krpc::schema::ProcedureCall default_distance_call();

    ::krpc::schema::ProcedureCall distance_call();

    ::krpc::schema::ProcedureCall set_distance_call(float value);

    ::krpc::schema::ProcedureCall focussed_body_call();

    ::krpc::schema::ProcedureCall set_focussed_body_call(SpaceCenter::CelestialBody value);

    ::krpc::schema::ProcedureCall focussed_node_call();

    ::krpc::schema::ProcedureCall set_focussed_node_call(SpaceCenter::Node value);

    ::krpc::schema::ProcedureCall focussed_vessel_call();

    ::krpc::schema::ProcedureCall set_focussed_vessel_call(SpaceCenter::Vessel value);

    ::krpc::schema::ProcedureCall heading_call();

    ::krpc::schema::ProcedureCall set_heading_call(float value);

    ::krpc::schema::ProcedureCall max_distance_call();

    ::krpc::schema::ProcedureCall max_pitch_call();

    ::krpc::schema::ProcedureCall min_distance_call();

    ::krpc::schema::ProcedureCall min_pitch_call();

    ::krpc::schema::ProcedureCall mode_call();

    ::krpc::schema::ProcedureCall set_mode_call(SpaceCenter::CameraMode value);

    ::krpc::schema::ProcedureCall pitch_call();

    ::krpc::schema::ProcedureCall set_pitch_call(float value);
  };

  /**
   * A cargo bay. Obtained by calling SpaceCenter::Part::cargo_bay.
   */
  class CargoBay : public krpc::Object<CargoBay> {
   public:
    explicit CargoBay(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the cargo bay is open.
     */
    bool open();

    /**
     * Whether the cargo bay is open.
     */
    void set_open(bool value);

    /**
     * The part object for this cargo bay.
     */
    SpaceCenter::Part part();

    /**
     * The state of the cargo bay.
     */
    SpaceCenter::CargoBayState state();

    ::krpc::Stream<bool> open_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<SpaceCenter::CargoBayState> state_stream();

    ::krpc::schema::ProcedureCall open_call();

    ::krpc::schema::ProcedureCall set_open_call(bool value);

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall state_call();
  };

  /**
   * Represents a celestial body (such as a planet or moon).
   * See SpaceCenter::bodies.
   */
  class CelestialBody : public krpc::Object<CelestialBody> {
   public:
    explicit CelestialBody(Client* client = nullptr, uint64_t id = 0);

    /**
     * The altitude, in meters, of the given position in the given reference frame.
     * @param position Position as a vector.
     * @param referenceFrame Reference frame for the position vector.
     */
    double altitude_at_position(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The angular velocity of the body in the specified reference frame.
     * @return The angular velocity as a vector. The magnitude of the vector is the rotational
     * speed of the body, in radians per second. The direction of the vector indicates the axis
     * of rotation, using the right-hand rule.
     * @param referenceFrame The reference frame the returned
     * angular velocity is in.
     */
    std::tuple<double, double, double> angular_velocity(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The atmospheric density at the given position, in kg/m^3,
     * in the given reference frame.
     * @param position The position vector at which to measure the density.
     * @param referenceFrame Reference frame that the position vector is in.
     */
    double atmospheric_density_at_position(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The height of the surface relative to mean sea level, in meters,
     * at the given position. When over water, this is the height
     * of the sea-bed and is therefore  negative value.
     * @param latitude Latitude in degrees.
     * @param longitude Longitude in degrees.
     */
    double bedrock_height(double latitude, double longitude);

    /**
     * The position of the surface at the given latitude and longitude, in the given
     * reference frame. When over water, this is the position at the bottom of the sea-bed.
     * @return Position as a vector.
     * @param latitude Latitude in degrees.
     * @param longitude Longitude in degrees.
     * @param referenceFrame Reference frame for the returned position vector.
     */
    std::tuple<double, double, double> bedrock_position(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The biome at the given latitude and longitude, in degrees.
     */
    std::string biome_at(double latitude, double longitude);

    /**
     * Gets the air density, in kg/m^3, for the specified
     * altitude above sea level, in meters.
     *
     * This is an approximation, because actual calculations, taking sun exposure into account
     * to compute air temperature, require us to know the exact point on the body where the
     * density is to be computed (knowing the altitude is not enough).
     * However, the difference is small for high altitudes, so it makes very little difference
     * for trajectory prediction.
     */
    double density_at(double altitude);

    /**
     * The direction in which the north pole of the celestial body is pointing,
     * in the specified reference frame.
     * @return The direction as a unit vector.
     * @param referenceFrame The reference frame that the returned
     * direction is in.
     */
    std::tuple<double, double, double> direction(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The latitude of the given position, in the given reference frame.
     * @param position Position as a vector.
     * @param referenceFrame Reference frame for the position vector.
     */
    double latitude_at_position(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The longitude of the given position, in the given reference frame.
     * @param position Position as a vector.
     * @param referenceFrame Reference frame for the position vector.
     */
    double longitude_at_position(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The position at mean sea level at the given latitude and longitude,
     * in the given reference frame.
     * @return Position as a vector.
     * @param latitude Latitude in degrees.
     * @param longitude Longitude in degrees.
     * @param referenceFrame Reference frame for the returned position vector.
     */
    std::tuple<double, double, double> msl_position(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The position of the center of the body, in the specified reference frame.
     * @return The position as a vector.
     * @param referenceFrame The reference frame that the returned
     * position vector is in.
     */
    std::tuple<double, double, double> position(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The position at the given latitude, longitude and altitude, in the given reference frame.
     * @return Position as a vector.
     * @param latitude Latitude in degrees.
     * @param longitude Longitude in degrees.
     * @param altitude Altitude in meters above sea level.
     * @param referenceFrame Reference frame for the returned position vector.
     */
    std::tuple<double, double, double> position_at_altitude(double latitude, double longitude, double altitude, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * Gets the air pressure, in Pascals, for the specified
     * altitude above sea level, in meters.
     */
    double pressure_at(double altitude);

    /**
     * The rotation of the body, in the specified reference frame.
     * @return The rotation as a quaternion of the form (x, y, z, w).
     * @param referenceFrame The reference frame that the returned
     * rotation is in.
     */
    std::tuple<double, double, double, double> rotation(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The height of the surface relative to mean sea level, in meters,
     * at the given position. When over water this is equal to 0.
     * @param latitude Latitude in degrees.
     * @param longitude Longitude in degrees.
     */
    double surface_height(double latitude, double longitude);

    /**
     * The position of the surface at the given latitude and longitude, in the given
     * reference frame. When over water, this is the position of the surface of the water.
     * @return Position as a vector.
     * @param latitude Latitude in degrees.
     * @param longitude Longitude in degrees.
     * @param referenceFrame Reference frame for the returned position vector.
     */
    std::tuple<double, double, double> surface_position(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The temperature on the body at the given position, in the given reference frame.
     * @param position Position as a vector.
     * @param referenceFrame The reference frame that the position is in.
     *
     * This calculation is performed using the bodies current position, which means that
     * the value could be wrong if you want to know the temperature in the far future.
     */
    double temperature_at(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The linear velocity of the body, in the specified reference frame.
     * @return The velocity as a vector. The vector points in the direction of travel,
     * and its magnitude is the speed of the body in meters per second.
     * @param referenceFrame The reference frame that the returned
     * velocity vector is in.
     */
    std::tuple<double, double, double> velocity(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The depth of the atmosphere, in meters.
     */
    double atmosphere_depth();

    /**
     * The biomes present on this body.
     */
    std::set<std::string> biomes();

    /**
     * The equatorial radius of the body, in meters.
     */
    double equatorial_radius();

    /**
     * The altitude, in meters, above which a vessel is considered to be
     * flying "high" when doing science.
     */
    float flying_high_altitude_threshold();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Standard_gravitational_parameter">standard
     * gravitational parameter</a> of the body in m^3s^{-2}.
     */
    double gravitational_parameter();

    /**
     * true if the body has an atmosphere.
     */
    bool has_atmosphere();

    /**
     * true if there is oxygen in the atmosphere, required for air-breathing engines.
     */
    bool has_atmospheric_oxygen();

    /**
     * Whether or not the body has a solid surface.
     */
    bool has_solid_surface();

    /**
     * The initial rotation angle of the body (at UT 0), in radians.
     * A value between 0 and 2\pi
     */
    double initial_rotation();

    /**
     * Whether or not the body is a star.
     */
    bool is_star();

    /**
     * The mass of the body, in kilograms.
     */
    double mass();

    /**
     * The name of the body.
     */
    std::string name();

    /**
     * The reference frame that is fixed relative to this celestial body, and
     * orientated in a fixed direction (it does not rotate with the body).
     *
     * - The origin is at the center of the body.
     * - The axes do not rotate.
     * - The x-axis points in an arbitrary direction through the
     *   equator.
     * - The y-axis points from the center of the body towards
     *   the north pole.
     * - The z-axis points in an arbitrary direction through the
     *   equator.
     */
    SpaceCenter::ReferenceFrame non_rotating_reference_frame();

    /**
     * The orbit of the body.
     */
    SpaceCenter::Orbit orbit();

    /**
     * The reference frame that is fixed relative to this celestial body, but
     * orientated with the body's orbital prograde/normal/radial directions.
     *
     * - The origin is at the center of the body.
     * - The axes rotate with the orbital prograde/normal/radial
     *   directions.
     * - The x-axis points in the orbital anti-radial direction.
     * - The y-axis points in the orbital prograde direction.
     * - The z-axis points in the orbital normal direction.
     */
    SpaceCenter::ReferenceFrame orbital_reference_frame();

    /**
     * The reference frame that is fixed relative to the celestial body.
     *
     * - The origin is at the center of the body.
     * - The axes rotate with the body.
     * - The x-axis points from the center of the body
     *   towards the intersection of the prime meridian and equator (the
     *   position at 0° longitude, 0° latitude).
     * - The y-axis points from the center of the body
     *   towards the north pole.
     * - The z-axis points from the center of the body
     *   towards the equator at 90°E longitude.
     */
    SpaceCenter::ReferenceFrame reference_frame();

    /**
     * The current rotation angle of the body, in radians.
     * A value between 0 and 2\pi
     */
    double rotation_angle();

    /**
     * The sidereal rotational period of the body, in seconds.
     */
    double rotational_period();

    /**
     * The rotational speed of the body, in radians per second.
     */
    double rotational_speed();

    /**
     * A list of celestial bodies that are in orbit around this celestial body.
     */
    std::vector<SpaceCenter::CelestialBody> satellites();

    /**
     * The altitude, in meters, above which a vessel is considered to be
     * in "high" space when doing science.
     */
    float space_high_altitude_threshold();

    /**
     * The radius of the sphere of influence of the body, in meters.
     */
    double sphere_of_influence();

    /**
     * The acceleration due to gravity at sea level (mean altitude) on the body,
     * in m/s^2.
     */
    double surface_gravity();

    ::krpc::Stream<double> altitude_at_position_stream(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> angular_velocity_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<double> atmospheric_density_at_position_stream(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<double> bedrock_height_stream(double latitude, double longitude);

    ::krpc::Stream<std::tuple<double, double, double>> bedrock_position_stream(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::string> biome_at_stream(double latitude, double longitude);

    ::krpc::Stream<double> density_at_stream(double altitude);

    ::krpc::Stream<std::tuple<double, double, double>> direction_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<double> latitude_at_position_stream(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<double> longitude_at_position_stream(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> msl_position_stream(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> position_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> position_at_altitude_stream(double latitude, double longitude, double altitude, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<double> pressure_at_stream(double altitude);

    ::krpc::Stream<std::tuple<double, double, double, double>> rotation_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<double> surface_height_stream(double latitude, double longitude);

    ::krpc::Stream<std::tuple<double, double, double>> surface_position_stream(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<double> temperature_at_stream(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> velocity_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<double> atmosphere_depth_stream();

    ::krpc::Stream<std::set<std::string>> biomes_stream();

    ::krpc::Stream<double> equatorial_radius_stream();

    ::krpc::Stream<float> flying_high_altitude_threshold_stream();

    ::krpc::Stream<double> gravitational_parameter_stream();

    ::krpc::Stream<bool> has_atmosphere_stream();

    ::krpc::Stream<bool> has_atmospheric_oxygen_stream();

    ::krpc::Stream<bool> has_solid_surface_stream();

    ::krpc::Stream<double> initial_rotation_stream();

    ::krpc::Stream<bool> is_star_stream();

    ::krpc::Stream<double> mass_stream();

    ::krpc::Stream<std::string> name_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> non_rotating_reference_frame_stream();

    ::krpc::Stream<SpaceCenter::Orbit> orbit_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> orbital_reference_frame_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> reference_frame_stream();

    ::krpc::Stream<double> rotation_angle_stream();

    ::krpc::Stream<double> rotational_period_stream();

    ::krpc::Stream<double> rotational_speed_stream();

    ::krpc::Stream<std::vector<SpaceCenter::CelestialBody>> satellites_stream();

    ::krpc::Stream<float> space_high_altitude_threshold_stream();

    ::krpc::Stream<double> sphere_of_influence_stream();

    ::krpc::Stream<double> surface_gravity_stream();

    ::krpc::schema::ProcedureCall altitude_at_position_call(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall angular_velocity_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall atmospheric_density_at_position_call(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall bedrock_height_call(double latitude, double longitude);

    ::krpc::schema::ProcedureCall bedrock_position_call(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall biome_at_call(double latitude, double longitude);

    ::krpc::schema::ProcedureCall density_at_call(double altitude);

    ::krpc::schema::ProcedureCall direction_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall latitude_at_position_call(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall longitude_at_position_call(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall msl_position_call(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall position_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall position_at_altitude_call(double latitude, double longitude, double altitude, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall pressure_at_call(double altitude);

    ::krpc::schema::ProcedureCall rotation_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall surface_height_call(double latitude, double longitude);

    ::krpc::schema::ProcedureCall surface_position_call(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall temperature_at_call(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall velocity_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall atmosphere_depth_call();

    ::krpc::schema::ProcedureCall biomes_call();

    ::krpc::schema::ProcedureCall equatorial_radius_call();

    ::krpc::schema::ProcedureCall flying_high_altitude_threshold_call();

    ::krpc::schema::ProcedureCall gravitational_parameter_call();

    ::krpc::schema::ProcedureCall has_atmosphere_call();

    ::krpc::schema::ProcedureCall has_atmospheric_oxygen_call();

    ::krpc::schema::ProcedureCall has_solid_surface_call();

    ::krpc::schema::ProcedureCall initial_rotation_call();

    ::krpc::schema::ProcedureCall is_star_call();

    ::krpc::schema::ProcedureCall mass_call();

    ::krpc::schema::ProcedureCall name_call();

    ::krpc::schema::ProcedureCall non_rotating_reference_frame_call();

    ::krpc::schema::ProcedureCall orbit_call();

    ::krpc::schema::ProcedureCall orbital_reference_frame_call();

    ::krpc::schema::ProcedureCall reference_frame_call();

    ::krpc::schema::ProcedureCall rotation_angle_call();

    ::krpc::schema::ProcedureCall rotational_period_call();

    ::krpc::schema::ProcedureCall rotational_speed_call();

    ::krpc::schema::ProcedureCall satellites_call();

    ::krpc::schema::ProcedureCall space_high_altitude_threshold_call();

    ::krpc::schema::ProcedureCall sphere_of_influence_call();

    ::krpc::schema::ProcedureCall surface_gravity_call();
  };

  /**
   * Represents a communication node in the network. For example, a vessel or the KSC.
   */
  class CommLink : public krpc::Object<CommLink> {
   public:
    explicit CommLink(Client* client = nullptr, uint64_t id = 0);

    /**
     * Start point of the link.
     */
    SpaceCenter::CommNode end();

    /**
     * Signal strength of the link.
     */
    double signal_strength();

    /**
     * Start point of the link.
     */
    SpaceCenter::CommNode start();

    /**
     * The type of link.
     */
    SpaceCenter::CommLinkType type();

    ::krpc::Stream<SpaceCenter::CommNode> end_stream();

    ::krpc::Stream<double> signal_strength_stream();

    ::krpc::Stream<SpaceCenter::CommNode> start_stream();

    ::krpc::Stream<SpaceCenter::CommLinkType> type_stream();

    ::krpc::schema::ProcedureCall end_call();

    ::krpc::schema::ProcedureCall signal_strength_call();

    ::krpc::schema::ProcedureCall start_call();

    ::krpc::schema::ProcedureCall type_call();
  };

  /**
   * Represents a communication node in the network. For example, a vessel or the KSC.
   */
  class CommNode : public krpc::Object<CommNode> {
   public:
    explicit CommNode(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the communication node is a control point, for example a manned vessel.
     */
    bool is_control_point();

    /**
     * Whether the communication node is on Kerbin.
     */
    bool is_home();

    /**
     * Whether the communication node is a vessel.
     */
    bool is_vessel();

    /**
     * Name of the communication node.
     */
    std::string name();

    /**
     * The vessel for this communication node.
     */
    SpaceCenter::Vessel vessel();

    ::krpc::Stream<bool> is_control_point_stream();

    ::krpc::Stream<bool> is_home_stream();

    ::krpc::Stream<bool> is_vessel_stream();

    ::krpc::Stream<std::string> name_stream();

    ::krpc::Stream<SpaceCenter::Vessel> vessel_stream();

    ::krpc::schema::ProcedureCall is_control_point_call();

    ::krpc::schema::ProcedureCall is_home_call();

    ::krpc::schema::ProcedureCall is_vessel_call();

    ::krpc::schema::ProcedureCall name_call();

    ::krpc::schema::ProcedureCall vessel_call();
  };

  /**
   * Used to interact with CommNet for a given vessel.
   * Obtained by calling SpaceCenter::Vessel::comms.
   */
  class Comms : public krpc::Object<Comms> {
   public:
    explicit Comms(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the vessel can communicate with KSC.
     */
    bool can_communicate();

    /**
     * Whether the vessel can transmit science data to KSC.
     */
    bool can_transmit_science();

    /**
     * The communication path used to control the vessel.
     */
    std::vector<SpaceCenter::CommLink> control_path();

    /**
     * The combined power of all active antennae on the vessel.
     */
    double power();

    /**
     * Signal delay to KSC in seconds.
     */
    double signal_delay();

    /**
     * Signal strength to KSC.
     */
    double signal_strength();

    ::krpc::Stream<bool> can_communicate_stream();

    ::krpc::Stream<bool> can_transmit_science_stream();

    ::krpc::Stream<std::vector<SpaceCenter::CommLink>> control_path_stream();

    ::krpc::Stream<double> power_stream();

    ::krpc::Stream<double> signal_delay_stream();

    ::krpc::Stream<double> signal_strength_stream();

    ::krpc::schema::ProcedureCall can_communicate_call();

    ::krpc::schema::ProcedureCall can_transmit_science_call();

    ::krpc::schema::ProcedureCall control_path_call();

    ::krpc::schema::ProcedureCall power_call();

    ::krpc::schema::ProcedureCall signal_delay_call();

    ::krpc::schema::ProcedureCall signal_strength_call();
  };

  /**
   * A contract. Can be accessed using SpaceCenter::contract_manager.
   */
  class Contract : public krpc::Object<Contract> {
   public:
    explicit Contract(Client* client = nullptr, uint64_t id = 0);

    /**
     * Accept an offered contract.
     */
    void accept();

    /**
     * Cancel an active contract.
     */
    void cancel();

    /**
     * Decline an offered contract.
     */
    void decline();

    /**
     * Whether the contract is active.
     */
    bool active();

    /**
     * Whether the contract can be canceled.
     */
    bool can_be_canceled();

    /**
     * Whether the contract can be declined.
     */
    bool can_be_declined();

    /**
     * Whether the contract can be failed.
     */
    bool can_be_failed();

    /**
     * Description of the contract.
     */
    std::string description();

    /**
     * Whether the contract has been failed.
     */
    bool failed();

    /**
     * Funds received when accepting the contract.
     */
    double funds_advance();

    /**
     * Funds received on completion of the contract.
     */
    double funds_completion();

    /**
     * Funds lost if the contract is failed.
     */
    double funds_failure();

    /**
     * Keywords for the contract.
     */
    std::vector<std::string> keywords();

    /**
     * Notes for the contract.
     */
    std::string notes();

    /**
     * Parameters for the contract.
     */
    std::vector<SpaceCenter::ContractParameter> parameters();

    /**
     * Whether the contract has been read.
     */
    bool read();

    /**
     * Reputation gained on completion of the contract.
     */
    double reputation_completion();

    /**
     * Reputation lost if the contract is failed.
     */
    double reputation_failure();

    /**
     * Science gained on completion of the contract.
     */
    double science_completion();

    /**
     * Whether the contract has been seen.
     */
    bool seen();

    /**
     * State of the contract.
     */
    SpaceCenter::ContractState state();

    /**
     * Synopsis for the contract.
     */
    std::string synopsis();

    /**
     * Title of the contract.
     */
    std::string title();

    /**
     * Type of the contract.
     */
    std::string type();

    ::krpc::Stream<bool> active_stream();

    ::krpc::Stream<bool> can_be_canceled_stream();

    ::krpc::Stream<bool> can_be_declined_stream();

    ::krpc::Stream<bool> can_be_failed_stream();

    ::krpc::Stream<std::string> description_stream();

    ::krpc::Stream<bool> failed_stream();

    ::krpc::Stream<double> funds_advance_stream();

    ::krpc::Stream<double> funds_completion_stream();

    ::krpc::Stream<double> funds_failure_stream();

    ::krpc::Stream<std::vector<std::string>> keywords_stream();

    ::krpc::Stream<std::string> notes_stream();

    ::krpc::Stream<std::vector<SpaceCenter::ContractParameter>> parameters_stream();

    ::krpc::Stream<bool> read_stream();

    ::krpc::Stream<double> reputation_completion_stream();

    ::krpc::Stream<double> reputation_failure_stream();

    ::krpc::Stream<double> science_completion_stream();

    ::krpc::Stream<bool> seen_stream();

    ::krpc::Stream<SpaceCenter::ContractState> state_stream();

    ::krpc::Stream<std::string> synopsis_stream();

    ::krpc::Stream<std::string> title_stream();

    ::krpc::Stream<std::string> type_stream();

    ::krpc::schema::ProcedureCall accept_call();

    ::krpc::schema::ProcedureCall cancel_call();

    ::krpc::schema::ProcedureCall decline_call();

    ::krpc::schema::ProcedureCall active_call();

    ::krpc::schema::ProcedureCall can_be_canceled_call();

    ::krpc::schema::ProcedureCall can_be_declined_call();

    ::krpc::schema::ProcedureCall can_be_failed_call();

    ::krpc::schema::ProcedureCall description_call();

    ::krpc::schema::ProcedureCall failed_call();

    ::krpc::schema::ProcedureCall funds_advance_call();

    ::krpc::schema::ProcedureCall funds_completion_call();

    ::krpc::schema::ProcedureCall funds_failure_call();

    ::krpc::schema::ProcedureCall keywords_call();

    ::krpc::schema::ProcedureCall notes_call();

    ::krpc::schema::ProcedureCall parameters_call();

    ::krpc::schema::ProcedureCall read_call();

    ::krpc::schema::ProcedureCall reputation_completion_call();

    ::krpc::schema::ProcedureCall reputation_failure_call();

    ::krpc::schema::ProcedureCall science_completion_call();

    ::krpc::schema::ProcedureCall seen_call();

    ::krpc::schema::ProcedureCall state_call();

    ::krpc::schema::ProcedureCall synopsis_call();

    ::krpc::schema::ProcedureCall title_call();

    ::krpc::schema::ProcedureCall type_call();
  };

  /**
   * Contracts manager.
   * Obtained by calling SpaceCenter::contract_manager.
   */
  class ContractManager : public krpc::Object<ContractManager> {
   public:
    explicit ContractManager(Client* client = nullptr, uint64_t id = 0);

    /**
     * A list of all active contracts.
     */
    std::vector<SpaceCenter::Contract> active_contracts();

    /**
     * A list of all contracts.
     */
    std::vector<SpaceCenter::Contract> all_contracts();

    /**
     * A list of all completed contracts.
     */
    std::vector<SpaceCenter::Contract> completed_contracts();

    /**
     * A list of all failed contracts.
     */
    std::vector<SpaceCenter::Contract> failed_contracts();

    /**
     * A list of all offered, but unaccepted, contracts.
     */
    std::vector<SpaceCenter::Contract> offered_contracts();

    /**
     * A list of all contract types.
     */
    std::set<std::string> types();

    ::krpc::Stream<std::vector<SpaceCenter::Contract>> active_contracts_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Contract>> all_contracts_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Contract>> completed_contracts_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Contract>> failed_contracts_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Contract>> offered_contracts_stream();

    ::krpc::Stream<std::set<std::string>> types_stream();

    ::krpc::schema::ProcedureCall active_contracts_call();

    ::krpc::schema::ProcedureCall all_contracts_call();

    ::krpc::schema::ProcedureCall completed_contracts_call();

    ::krpc::schema::ProcedureCall failed_contracts_call();

    ::krpc::schema::ProcedureCall offered_contracts_call();

    ::krpc::schema::ProcedureCall types_call();
  };

  /**
   * A contract parameter. See SpaceCenter::Contract::parameters.
   */
  class ContractParameter : public krpc::Object<ContractParameter> {
   public:
    explicit ContractParameter(Client* client = nullptr, uint64_t id = 0);

    /**
     * Child contract parameters.
     */
    std::vector<SpaceCenter::ContractParameter> children();

    /**
     * Whether the parameter has been completed.
     */
    bool completed();

    /**
     * Whether the parameter has been failed.
     */
    bool failed();

    /**
     * Funds received on completion of the contract parameter.
     */
    double funds_completion();

    /**
     * Funds lost if the contract parameter is failed.
     */
    double funds_failure();

    /**
     * Notes for the parameter.
     */
    std::string notes();

    /**
     * Whether the contract parameter is optional.
     */
    bool optional();

    /**
     * Reputation gained on completion of the contract parameter.
     */
    double reputation_completion();

    /**
     * Reputation lost if the contract parameter is failed.
     */
    double reputation_failure();

    /**
     * Science gained on completion of the contract parameter.
     */
    double science_completion();

    /**
     * Title of the parameter.
     */
    std::string title();

    ::krpc::Stream<std::vector<SpaceCenter::ContractParameter>> children_stream();

    ::krpc::Stream<bool> completed_stream();

    ::krpc::Stream<bool> failed_stream();

    ::krpc::Stream<double> funds_completion_stream();

    ::krpc::Stream<double> funds_failure_stream();

    ::krpc::Stream<std::string> notes_stream();

    ::krpc::Stream<bool> optional_stream();

    ::krpc::Stream<double> reputation_completion_stream();

    ::krpc::Stream<double> reputation_failure_stream();

    ::krpc::Stream<double> science_completion_stream();

    ::krpc::Stream<std::string> title_stream();

    ::krpc::schema::ProcedureCall children_call();

    ::krpc::schema::ProcedureCall completed_call();

    ::krpc::schema::ProcedureCall failed_call();

    ::krpc::schema::ProcedureCall funds_completion_call();

    ::krpc::schema::ProcedureCall funds_failure_call();

    ::krpc::schema::ProcedureCall notes_call();

    ::krpc::schema::ProcedureCall optional_call();

    ::krpc::schema::ProcedureCall reputation_completion_call();

    ::krpc::schema::ProcedureCall reputation_failure_call();

    ::krpc::schema::ProcedureCall science_completion_call();

    ::krpc::schema::ProcedureCall title_call();
  };

  /**
   * Used to manipulate the controls of a vessel. This includes adjusting the
   * throttle, enabling/disabling systems such as SAS and RCS, or altering the
   * direction in which the vessel is pointing.
   * Obtained by calling SpaceCenter::Vessel::control.
   *
   * Control inputs (such as pitch, yaw and roll) are zeroed when all clients
   * that have set one or more of these inputs are no longer connected.
   */
  class Control : public krpc::Object<Control> {
   public:
    explicit Control(Client* client = nullptr, uint64_t id = 0);

    /**
     * Activates the next stage. Equivalent to pressing the space bar in-game.
     * @return A list of vessel objects that are jettisoned from the active vessel.
     *
     * When called, the active vessel may change. It is therefore possible that,
     * after calling this function, the object(s) returned by previous call(s) to
     * SpaceCenter::active_vessel no longer refer to the active vessel.
     * Throws an exception if staging is locked.
     */
    std::vector<SpaceCenter::Vessel> activate_next_stage();

    /**
     * Creates a maneuver node at the given universal time, and returns a
     * SpaceCenter::Node object that can be used to modify it.
     * Optionally sets the magnitude of the delta-v for the maneuver node
     * in the prograde, normal and radial directions.
     * @param ut Universal time of the maneuver node.
     * @param prograde Delta-v in the prograde direction.
     * @param normal Delta-v in the normal direction.
     * @param radial Delta-v in the radial direction.
     */
    SpaceCenter::Node add_node(double ut, float prograde, float normal, float radial);

    /**
     * Returns true if the given action group is enabled.
     * @param group A number between 0 and 9 inclusive,
     * or between 0 and 250 inclusive when the <a href="https://forum.kerbalspaceprogram.com/index.php?/topic/67235-122dec1016-action-groups-extended-250-action-groups-in-flight-editing-now-kosremotetech/">Extended Action Groups mod</a> is installed.
     */
    bool get_action_group(uint32_t group);

    /**
     * Remove all maneuver nodes.
     */
    void remove_nodes();

    /**
     * Sets the state of the given action group.
     * @param group A number between 0 and 9 inclusive,
     * or between 0 and 250 inclusive when the <a href="https://forum.kerbalspaceprogram.com/index.php?/topic/67235-122dec1016-action-groups-extended-250-action-groups-in-flight-editing-now-kosremotetech/">Extended Action Groups mod</a> is installed.
     * @param state
     */
    void set_action_group(uint32_t group, bool state);

    /**
     * Toggles the state of the given action group.
     * @param group A number between 0 and 9 inclusive,
     * or between 0 and 250 inclusive when the <a href="https://forum.kerbalspaceprogram.com/index.php?/topic/67235-122dec1016-action-groups-extended-250-action-groups-in-flight-editing-now-kosremotetech/">Extended Action Groups mod</a> is installed.
     */
    void toggle_action_group(uint32_t group);

    /**
     * The state of the abort action group.
     */
    bool abort();

    /**
     * The state of the abort action group.
     */
    void set_abort(bool value);

    /**
     * Returns whether all antennas on the vessel are deployed,
     * and sets the deployment state of all antennas.
     * See SpaceCenter::Antenna::deployed.
     */
    bool antennas();

    /**
     * Returns whether all antennas on the vessel are deployed,
     * and sets the deployment state of all antennas.
     * See SpaceCenter::Antenna::deployed.
     */
    void set_antennas(bool value);

    /**
     * The state of the wheel brakes.
     */
    bool brakes();

    /**
     * The state of the wheel brakes.
     */
    void set_brakes(bool value);

    /**
     * Returns whether any of the cargo bays on the vessel are open,
     * and sets the open state of all cargo bays.
     * See SpaceCenter::CargoBay::open.
     */
    bool cargo_bays();

    /**
     * Returns whether any of the cargo bays on the vessel are open,
     * and sets the open state of all cargo bays.
     * See SpaceCenter::CargoBay::open.
     */
    void set_cargo_bays(bool value);

    /**
     * The current stage of the vessel. Corresponds to the stage number in
     * the in-game UI.
     */
    int32_t current_stage();

    /**
     * The state of CustomAxis01.
     * A value between -1 and 1.
     */
    float custom_axis01();

    /**
     * The state of CustomAxis01.
     * A value between -1 and 1.
     */
    void set_custom_axis01(float value);

    /**
     * The state of CustomAxis02.
     * A value between -1 and 1.
     */
    float custom_axis02();

    /**
     * The state of CustomAxis02.
     * A value between -1 and 1.
     */
    void set_custom_axis02(float value);

    /**
     * The state of CustomAxis03.
     * A value between -1 and 1.
     */
    float custom_axis03();

    /**
     * The state of CustomAxis03.
     * A value between -1 and 1.
     */
    void set_custom_axis03(float value);

    /**
     * The state of CustomAxis04.
     * A value between -1 and 1.
     */
    float custom_axis04();

    /**
     * The state of CustomAxis04.
     * A value between -1 and 1.
     */
    void set_custom_axis04(float value);

    /**
     * The state of the forward translational control.
     * A value between -1 and 1.
     * Equivalent to the h and n keys.
     */
    float forward();

    /**
     * The state of the forward translational control.
     * A value between -1 and 1.
     * Equivalent to the h and n keys.
     */
    void set_forward(float value);

    /**
     * The state of the landing gear/legs.
     */
    bool gear();

    /**
     * The state of the landing gear/legs.
     */
    void set_gear(bool value);

    /**
     * Sets the behavior of the pitch, yaw, roll and translation control inputs.
     * When set to additive, these inputs are added to the vessels current inputs.
     * This mode is the default.
     * When set to override, these inputs (if non-zero) override the vessels inputs.
     * This mode prevents keyboard control, or SAS, from interfering with the controls when
     * they are set.
     */
    SpaceCenter::ControlInputMode input_mode();

    /**
     * Sets the behavior of the pitch, yaw, roll and translation control inputs.
     * When set to additive, these inputs are added to the vessels current inputs.
     * This mode is the default.
     * When set to override, these inputs (if non-zero) override the vessels inputs.
     * This mode prevents keyboard control, or SAS, from interfering with the controls when
     * they are set.
     */
    void set_input_mode(SpaceCenter::ControlInputMode value);

    /**
     * Returns whether all of the air intakes on the vessel are open,
     * and sets the open state of all air intakes.
     * See SpaceCenter::Intake::open.
     */
    bool intakes();

    /**
     * Returns whether all of the air intakes on the vessel are open,
     * and sets the open state of all air intakes.
     * See SpaceCenter::Intake::open.
     */
    void set_intakes(bool value);

    /**
     * Returns whether all landing legs on the vessel are deployed,
     * and sets the deployment state of all landing legs.
     * Does not include wheels (for example landing gear).
     * See SpaceCenter::Leg::deployed.
     */
    bool legs();

    /**
     * Returns whether all landing legs on the vessel are deployed,
     * and sets the deployment state of all landing legs.
     * Does not include wheels (for example landing gear).
     * See SpaceCenter::Leg::deployed.
     */
    void set_legs(bool value);

    /**
     * The state of the lights.
     */
    bool lights();

    /**
     * The state of the lights.
     */
    void set_lights(bool value);

    /**
     * Returns a list of all existing maneuver nodes, ordered by time from first to last.
     */
    std::vector<SpaceCenter::Node> nodes();

    /**
     * Returns whether all parachutes on the vessel are deployed,
     * and sets the deployment state of all parachutes.
     * Cannot be set to false.
     * See SpaceCenter::Parachute::deployed.
     */
    bool parachutes();

    /**
     * Returns whether all parachutes on the vessel are deployed,
     * and sets the deployment state of all parachutes.
     * Cannot be set to false.
     * See SpaceCenter::Parachute::deployed.
     */
    void set_parachutes(bool value);

    /**
     * The state of the pitch control.
     * A value between -1 and 1.
     * Equivalent to the w and s keys.
     */
    float pitch();

    /**
     * The state of the pitch control.
     * A value between -1 and 1.
     * Equivalent to the w and s keys.
     */
    void set_pitch(float value);

    /**
     * Returns whether all radiators on the vessel are deployed,
     * and sets the deployment state of all radiators.
     * See SpaceCenter::Radiator::deployed.
     */
    bool radiators();

    /**
     * Returns whether all radiators on the vessel are deployed,
     * and sets the deployment state of all radiators.
     * See SpaceCenter::Radiator::deployed.
     */
    void set_radiators(bool value);

    /**
     * The state of RCS.
     */
    bool rcs();

    /**
     * The state of RCS.
     */
    void set_rcs(bool value);

    /**
     * Returns whether all reactive wheels on the vessel are active,
     * and sets the active state of all reaction wheels.
     * See SpaceCenter::ReactionWheel::active.
     */
    bool reaction_wheels();

    /**
     * Returns whether all reactive wheels on the vessel are active,
     * and sets the active state of all reaction wheels.
     * See SpaceCenter::ReactionWheel::active.
     */
    void set_reaction_wheels(bool value);

    /**
     * Returns whether all of the resource harvesters on the vessel are deployed,
     * and sets the deployment state of all resource harvesters.
     * See SpaceCenter::ResourceHarvester::deployed.
     */
    bool resource_harvesters();

    /**
     * Returns whether all of the resource harvesters on the vessel are deployed,
     * and sets the deployment state of all resource harvesters.
     * See SpaceCenter::ResourceHarvester::deployed.
     */
    void set_resource_harvesters(bool value);

    /**
     * Returns whether any of the resource harvesters on the vessel are active,
     * and sets the active state of all resource harvesters.
     * See SpaceCenter::ResourceHarvester::active.
     */
    bool resource_harvesters_active();

    /**
     * Returns whether any of the resource harvesters on the vessel are active,
     * and sets the active state of all resource harvesters.
     * See SpaceCenter::ResourceHarvester::active.
     */
    void set_resource_harvesters_active(bool value);

    /**
     * The state of the right translational control.
     * A value between -1 and 1.
     * Equivalent to the j and l keys.
     */
    float right();

    /**
     * The state of the right translational control.
     * A value between -1 and 1.
     * Equivalent to the j and l keys.
     */
    void set_right(float value);

    /**
     * The state of the roll control.
     * A value between -1 and 1.
     * Equivalent to the q and e keys.
     */
    float roll();

    /**
     * The state of the roll control.
     * A value between -1 and 1.
     * Equivalent to the q and e keys.
     */
    void set_roll(float value);

    /**
     * The state of SAS.
     *
     * Equivalent to SpaceCenter::AutoPilot::sas
     */
    bool sas();

    /**
     * The state of SAS.
     *
     * Equivalent to SpaceCenter::AutoPilot::sas
     */
    void set_sas(bool value);

    /**
     * The current SpaceCenter::SASMode.
     * These modes are equivalent to the mode buttons to
     * the left of the navball that appear when SAS is enabled.
     *
     * Equivalent to SpaceCenter::AutoPilot::sas_mode
     */
    SpaceCenter::SASMode sas_mode();

    /**
     * The current SpaceCenter::SASMode.
     * These modes are equivalent to the mode buttons to
     * the left of the navball that appear when SAS is enabled.
     *
     * Equivalent to SpaceCenter::AutoPilot::sas_mode
     */
    void set_sas_mode(SpaceCenter::SASMode value);

    /**
     * Returns whether all solar panels on the vessel are deployed,
     * and sets the deployment state of all solar panels.
     * See SpaceCenter::SolarPanel::deployed.
     */
    bool solar_panels();

    /**
     * Returns whether all solar panels on the vessel are deployed,
     * and sets the deployment state of all solar panels.
     * See SpaceCenter::SolarPanel::deployed.
     */
    void set_solar_panels(bool value);

    /**
     * The source of the vessels control, for example by a kerbal or a probe core.
     */
    SpaceCenter::ControlSource source();

    /**
     * The current SpaceCenter::SpeedMode of the navball.
     * This is the mode displayed next to the speed at the top of the navball.
     */
    SpaceCenter::SpeedMode speed_mode();

    /**
     * The current SpaceCenter::SpeedMode of the navball.
     * This is the mode displayed next to the speed at the top of the navball.
     */
    void set_speed_mode(SpaceCenter::SpeedMode value);

    /**
     * Whether staging is locked on the vessel.
     *
     * This is equivalent to locking the staging using Alt+L
     */
    bool stage_lock();

    /**
     * Whether staging is locked on the vessel.
     *
     * This is equivalent to locking the staging using Alt+L
     */
    void set_stage_lock(bool value);

    /**
     * The control state of the vessel.
     */
    SpaceCenter::ControlState state();

    /**
     * The state of the throttle. A value between 0 and 1.
     */
    float throttle();

    /**
     * The state of the throttle. A value between 0 and 1.
     */
    void set_throttle(float value);

    /**
     * The state of the up translational control.
     * A value between -1 and 1.
     * Equivalent to the i and k keys.
     */
    float up();

    /**
     * The state of the up translational control.
     * A value between -1 and 1.
     * Equivalent to the i and k keys.
     */
    void set_up(float value);

    /**
     * The state of the wheel steering.
     * A value between -1 and 1.
     * A value of 1 steers to the left, and a value of -1 steers to the right.
     */
    float wheel_steering();

    /**
     * The state of the wheel steering.
     * A value between -1 and 1.
     * A value of 1 steers to the left, and a value of -1 steers to the right.
     */
    void set_wheel_steering(float value);

    /**
     * The state of the wheel throttle.
     * A value between -1 and 1.
     * A value of 1 rotates the wheels forwards, a value of -1 rotates
     * the wheels backwards.
     */
    float wheel_throttle();

    /**
     * The state of the wheel throttle.
     * A value between -1 and 1.
     * A value of 1 rotates the wheels forwards, a value of -1 rotates
     * the wheels backwards.
     */
    void set_wheel_throttle(float value);

    /**
     * Returns whether all wheels on the vessel are deployed,
     * and sets the deployment state of all wheels.
     * Does not include landing legs.
     * See SpaceCenter::Wheel::deployed.
     */
    bool wheels();

    /**
     * Returns whether all wheels on the vessel are deployed,
     * and sets the deployment state of all wheels.
     * Does not include landing legs.
     * See SpaceCenter::Wheel::deployed.
     */
    void set_wheels(bool value);

    /**
     * The state of the yaw control.
     * A value between -1 and 1.
     * Equivalent to the a and d keys.
     */
    float yaw();

    /**
     * The state of the yaw control.
     * A value between -1 and 1.
     * Equivalent to the a and d keys.
     */
    void set_yaw(float value);

    ::krpc::Stream<std::vector<SpaceCenter::Vessel>> activate_next_stage_stream();

    ::krpc::Stream<SpaceCenter::Node> add_node_stream(double ut, float prograde, float normal, float radial);

    ::krpc::Stream<bool> get_action_group_stream(uint32_t group);

    ::krpc::Stream<bool> abort_stream();

    ::krpc::Stream<bool> antennas_stream();

    ::krpc::Stream<bool> brakes_stream();

    ::krpc::Stream<bool> cargo_bays_stream();

    ::krpc::Stream<int32_t> current_stage_stream();

    ::krpc::Stream<float> custom_axis01_stream();

    ::krpc::Stream<float> custom_axis02_stream();

    ::krpc::Stream<float> custom_axis03_stream();

    ::krpc::Stream<float> custom_axis04_stream();

    ::krpc::Stream<float> forward_stream();

    ::krpc::Stream<bool> gear_stream();

    ::krpc::Stream<SpaceCenter::ControlInputMode> input_mode_stream();

    ::krpc::Stream<bool> intakes_stream();

    ::krpc::Stream<bool> legs_stream();

    ::krpc::Stream<bool> lights_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Node>> nodes_stream();

    ::krpc::Stream<bool> parachutes_stream();

    ::krpc::Stream<float> pitch_stream();

    ::krpc::Stream<bool> radiators_stream();

    ::krpc::Stream<bool> rcs_stream();

    ::krpc::Stream<bool> reaction_wheels_stream();

    ::krpc::Stream<bool> resource_harvesters_stream();

    ::krpc::Stream<bool> resource_harvesters_active_stream();

    ::krpc::Stream<float> right_stream();

    ::krpc::Stream<float> roll_stream();

    ::krpc::Stream<bool> sas_stream();

    ::krpc::Stream<SpaceCenter::SASMode> sas_mode_stream();

    ::krpc::Stream<bool> solar_panels_stream();

    ::krpc::Stream<SpaceCenter::ControlSource> source_stream();

    ::krpc::Stream<SpaceCenter::SpeedMode> speed_mode_stream();

    ::krpc::Stream<bool> stage_lock_stream();

    ::krpc::Stream<SpaceCenter::ControlState> state_stream();

    ::krpc::Stream<float> throttle_stream();

    ::krpc::Stream<float> up_stream();

    ::krpc::Stream<float> wheel_steering_stream();

    ::krpc::Stream<float> wheel_throttle_stream();

    ::krpc::Stream<bool> wheels_stream();

    ::krpc::Stream<float> yaw_stream();

    ::krpc::schema::ProcedureCall activate_next_stage_call();

    ::krpc::schema::ProcedureCall add_node_call(double ut, float prograde, float normal, float radial);

    ::krpc::schema::ProcedureCall get_action_group_call(uint32_t group);

    ::krpc::schema::ProcedureCall remove_nodes_call();

    ::krpc::schema::ProcedureCall set_action_group_call(uint32_t group, bool state);

    ::krpc::schema::ProcedureCall toggle_action_group_call(uint32_t group);

    ::krpc::schema::ProcedureCall abort_call();

    ::krpc::schema::ProcedureCall set_abort_call(bool value);

    ::krpc::schema::ProcedureCall antennas_call();

    ::krpc::schema::ProcedureCall set_antennas_call(bool value);

    ::krpc::schema::ProcedureCall brakes_call();

    ::krpc::schema::ProcedureCall set_brakes_call(bool value);

    ::krpc::schema::ProcedureCall cargo_bays_call();

    ::krpc::schema::ProcedureCall set_cargo_bays_call(bool value);

    ::krpc::schema::ProcedureCall current_stage_call();

    ::krpc::schema::ProcedureCall custom_axis01_call();

    ::krpc::schema::ProcedureCall set_custom_axis01_call(float value);

    ::krpc::schema::ProcedureCall custom_axis02_call();

    ::krpc::schema::ProcedureCall set_custom_axis02_call(float value);

    ::krpc::schema::ProcedureCall custom_axis03_call();

    ::krpc::schema::ProcedureCall set_custom_axis03_call(float value);

    ::krpc::schema::ProcedureCall custom_axis04_call();

    ::krpc::schema::ProcedureCall set_custom_axis04_call(float value);

    ::krpc::schema::ProcedureCall forward_call();

    ::krpc::schema::ProcedureCall set_forward_call(float value);

    ::krpc::schema::ProcedureCall gear_call();

    ::krpc::schema::ProcedureCall set_gear_call(bool value);

    ::krpc::schema::ProcedureCall input_mode_call();

    ::krpc::schema::ProcedureCall set_input_mode_call(SpaceCenter::ControlInputMode value);

    ::krpc::schema::ProcedureCall intakes_call();

    ::krpc::schema::ProcedureCall set_intakes_call(bool value);

    ::krpc::schema::ProcedureCall legs_call();

    ::krpc::schema::ProcedureCall set_legs_call(bool value);

    ::krpc::schema::ProcedureCall lights_call();

    ::krpc::schema::ProcedureCall set_lights_call(bool value);

    ::krpc::schema::ProcedureCall nodes_call();

    ::krpc::schema::ProcedureCall parachutes_call();

    ::krpc::schema::ProcedureCall set_parachutes_call(bool value);

    ::krpc::schema::ProcedureCall pitch_call();

    ::krpc::schema::ProcedureCall set_pitch_call(float value);

    ::krpc::schema::ProcedureCall radiators_call();

    ::krpc::schema::ProcedureCall set_radiators_call(bool value);

    ::krpc::schema::ProcedureCall rcs_call();

    ::krpc::schema::ProcedureCall set_rcs_call(bool value);

    ::krpc::schema::ProcedureCall reaction_wheels_call();

    ::krpc::schema::ProcedureCall set_reaction_wheels_call(bool value);

    ::krpc::schema::ProcedureCall resource_harvesters_call();

    ::krpc::schema::ProcedureCall set_resource_harvesters_call(bool value);

    ::krpc::schema::ProcedureCall resource_harvesters_active_call();

    ::krpc::schema::ProcedureCall set_resource_harvesters_active_call(bool value);

    ::krpc::schema::ProcedureCall right_call();

    ::krpc::schema::ProcedureCall set_right_call(float value);

    ::krpc::schema::ProcedureCall roll_call();

    ::krpc::schema::ProcedureCall set_roll_call(float value);

    ::krpc::schema::ProcedureCall sas_call();

    ::krpc::schema::ProcedureCall set_sas_call(bool value);

    ::krpc::schema::ProcedureCall sas_mode_call();

    ::krpc::schema::ProcedureCall set_sas_mode_call(SpaceCenter::SASMode value);

    ::krpc::schema::ProcedureCall solar_panels_call();

    ::krpc::schema::ProcedureCall set_solar_panels_call(bool value);

    ::krpc::schema::ProcedureCall source_call();

    ::krpc::schema::ProcedureCall speed_mode_call();

    ::krpc::schema::ProcedureCall set_speed_mode_call(SpaceCenter::SpeedMode value);

    ::krpc::schema::ProcedureCall stage_lock_call();

    ::krpc::schema::ProcedureCall set_stage_lock_call(bool value);

    ::krpc::schema::ProcedureCall state_call();

    ::krpc::schema::ProcedureCall throttle_call();

    ::krpc::schema::ProcedureCall set_throttle_call(float value);

    ::krpc::schema::ProcedureCall up_call();

    ::krpc::schema::ProcedureCall set_up_call(float value);

    ::krpc::schema::ProcedureCall wheel_steering_call();

    ::krpc::schema::ProcedureCall set_wheel_steering_call(float value);

    ::krpc::schema::ProcedureCall wheel_throttle_call();

    ::krpc::schema::ProcedureCall set_wheel_throttle_call(float value);

    ::krpc::schema::ProcedureCall wheels_call();

    ::krpc::schema::ProcedureCall set_wheels_call(bool value);

    ::krpc::schema::ProcedureCall yaw_call();

    ::krpc::schema::ProcedureCall set_yaw_call(float value);
  };

  /**
   * An aerodynamic control surface. Obtained by calling SpaceCenter::Part::control_surface.
   */
  class ControlSurface : public krpc::Object<ControlSurface> {
   public:
    explicit ControlSurface(Client* client = nullptr, uint64_t id = 0);

    /**
     * The authority limiter for the control surface, which controls how far the
     * control surface will move.
     */
    float authority_limiter();

    /**
     * The authority limiter for the control surface, which controls how far the
     * control surface will move.
     */
    void set_authority_limiter(float value);

    /**
     * The available torque, in Newton meters, that can be produced by this control surface,
     * in the positive and negative pitch, roll and yaw axes of the vessel. These axes
     * correspond to the coordinate axes of the SpaceCenter::Vessel::reference_frame.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > available_torque();

    /**
     * Whether the control surface has been fully deployed.
     */
    bool deployed();

    /**
     * Whether the control surface has been fully deployed.
     */
    void set_deployed(bool value);

    /**
     * Whether the control surface movement is inverted.
     */
    bool inverted();

    /**
     * Whether the control surface movement is inverted.
     */
    void set_inverted(bool value);

    /**
     * The part object for this control surface.
     */
    SpaceCenter::Part part();

    /**
     * Whether the control surface has pitch control enabled.
     */
    bool pitch_enabled();

    /**
     * Whether the control surface has pitch control enabled.
     */
    void set_pitch_enabled(bool value);

    /**
     * Whether the control surface has roll control enabled.
     */
    bool roll_enabled();

    /**
     * Whether the control surface has roll control enabled.
     */
    void set_roll_enabled(bool value);

    /**
     * Surface area of the control surface in m^2.
     */
    float surface_area();

    /**
     * Whether the control surface has yaw control enabled.
     */
    bool yaw_enabled();

    /**
     * Whether the control surface has yaw control enabled.
     */
    void set_yaw_enabled(bool value);

    ::krpc::Stream<float> authority_limiter_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> available_torque_stream();

    ::krpc::Stream<bool> deployed_stream();

    ::krpc::Stream<bool> inverted_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<bool> pitch_enabled_stream();

    ::krpc::Stream<bool> roll_enabled_stream();

    ::krpc::Stream<float> surface_area_stream();

    ::krpc::Stream<bool> yaw_enabled_stream();

    ::krpc::schema::ProcedureCall authority_limiter_call();

    ::krpc::schema::ProcedureCall set_authority_limiter_call(float value);

    ::krpc::schema::ProcedureCall available_torque_call();

    ::krpc::schema::ProcedureCall deployed_call();

    ::krpc::schema::ProcedureCall set_deployed_call(bool value);

    ::krpc::schema::ProcedureCall inverted_call();

    ::krpc::schema::ProcedureCall set_inverted_call(bool value);

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall pitch_enabled_call();

    ::krpc::schema::ProcedureCall set_pitch_enabled_call(bool value);

    ::krpc::schema::ProcedureCall roll_enabled_call();

    ::krpc::schema::ProcedureCall set_roll_enabled_call(bool value);

    ::krpc::schema::ProcedureCall surface_area_call();

    ::krpc::schema::ProcedureCall yaw_enabled_call();

    ::krpc::schema::ProcedureCall set_yaw_enabled_call(bool value);
  };

  /**
   * Represents crew in a vessel. Can be obtained using SpaceCenter::Vessel::crew.
   */
  class CrewMember : public krpc::Object<CrewMember> {
   public:
    explicit CrewMember(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the crew member is a badass.
     */
    bool badass();

    /**
     * Whether the crew member is a badass.
     */
    void set_badass(bool value);

    /**
     * The flight IDs for each entry in the career flight log.
     */
    std::vector<int32_t> career_log_flights();

    /**
     * The body name for each entry in the career flight log.
     */
    std::vector<std::string> career_log_targets();

    /**
     * The type for each entry in the career flight log.
     */
    std::vector<std::string> career_log_types();

    /**
     * The crew members courage.
     */
    float courage();

    /**
     * The crew members courage.
     */
    void set_courage(float value);

    /**
     * The crew members experience.
     */
    float experience();

    /**
     * The crew members experience.
     */
    void set_experience(float value);

    /**
     * The crew member's gender.
     */
    SpaceCenter::CrewMemberGender gender();

    /**
     * The crew members name.
     */
    std::string name();

    /**
     * The crew members name.
     */
    void set_name(std::string value);

    /**
     * Whether the crew member is on a mission.
     */
    bool on_mission();

    /**
     * The crew member's current roster status.
     */
    SpaceCenter::RosterStatus roster_status();

    /**
     * The crew members stupidity.
     */
    float stupidity();

    /**
     * The crew members stupidity.
     */
    void set_stupidity(float value);

    /**
     * The crew member's suit type.
     */
    SpaceCenter::SuitType suit_type();

    /**
     * The crew member's suit type.
     */
    void set_suit_type(SpaceCenter::SuitType value);

    /**
     * The crew member's job.
     */
    std::string trait();

    /**
     * The type of crew member.
     */
    SpaceCenter::CrewMemberType type();

    /**
     * Whether the crew member is a veteran.
     */
    bool veteran();

    /**
     * Whether the crew member is a veteran.
     */
    void set_veteran(bool value);

    ::krpc::Stream<bool> badass_stream();

    ::krpc::Stream<std::vector<int32_t>> career_log_flights_stream();

    ::krpc::Stream<std::vector<std::string>> career_log_targets_stream();

    ::krpc::Stream<std::vector<std::string>> career_log_types_stream();

    ::krpc::Stream<float> courage_stream();

    ::krpc::Stream<float> experience_stream();

    ::krpc::Stream<SpaceCenter::CrewMemberGender> gender_stream();

    ::krpc::Stream<std::string> name_stream();

    ::krpc::Stream<bool> on_mission_stream();

    ::krpc::Stream<SpaceCenter::RosterStatus> roster_status_stream();

    ::krpc::Stream<float> stupidity_stream();

    ::krpc::Stream<SpaceCenter::SuitType> suit_type_stream();

    ::krpc::Stream<std::string> trait_stream();

    ::krpc::Stream<SpaceCenter::CrewMemberType> type_stream();

    ::krpc::Stream<bool> veteran_stream();

    ::krpc::schema::ProcedureCall badass_call();

    ::krpc::schema::ProcedureCall set_badass_call(bool value);

    ::krpc::schema::ProcedureCall career_log_flights_call();

    ::krpc::schema::ProcedureCall career_log_targets_call();

    ::krpc::schema::ProcedureCall career_log_types_call();

    ::krpc::schema::ProcedureCall courage_call();

    ::krpc::schema::ProcedureCall set_courage_call(float value);

    ::krpc::schema::ProcedureCall experience_call();

    ::krpc::schema::ProcedureCall set_experience_call(float value);

    ::krpc::schema::ProcedureCall gender_call();

    ::krpc::schema::ProcedureCall name_call();

    ::krpc::schema::ProcedureCall set_name_call(std::string value);

    ::krpc::schema::ProcedureCall on_mission_call();

    ::krpc::schema::ProcedureCall roster_status_call();

    ::krpc::schema::ProcedureCall stupidity_call();

    ::krpc::schema::ProcedureCall set_stupidity_call(float value);

    ::krpc::schema::ProcedureCall suit_type_call();

    ::krpc::schema::ProcedureCall set_suit_type_call(SpaceCenter::SuitType value);

    ::krpc::schema::ProcedureCall trait_call();

    ::krpc::schema::ProcedureCall type_call();

    ::krpc::schema::ProcedureCall veteran_call();

    ::krpc::schema::ProcedureCall set_veteran_call(bool value);
  };

  /**
   * A decoupler. Obtained by calling SpaceCenter::Part::decoupler
   */
  class Decoupler : public krpc::Object<Decoupler> {
   public:
    explicit Decoupler(Client* client = nullptr, uint64_t id = 0);

    /**
     * Fires the decoupler. Returns the new vessel created when the decoupler fires.
     * Throws an exception if the decoupler has already fired.
     *
     * When called, the active vessel may change. It is therefore possible that,
     * after calling this function, the object(s) returned by previous call(s) to
     * SpaceCenter::active_vessel no longer refer to the active vessel.
     */
    SpaceCenter::Vessel decouple();

    /**
     * The part attached to this decoupler's explosive node.
     */
    SpaceCenter::Part attached_part();

    /**
     * Whether the decoupler has fired.
     */
    bool decoupled();

    /**
     * The impulse that the decoupler imparts when it is fired, in Newton seconds.
     */
    float impulse();

    /**
     * Whether the decoupler is an omni-decoupler (e.g. stack separator)
     */
    bool is_omni_decoupler();

    /**
     * The part object for this decoupler.
     */
    SpaceCenter::Part part();

    /**
     * Whether the decoupler is enabled in the staging sequence.
     */
    bool staged();

    ::krpc::Stream<SpaceCenter::Vessel> decouple_stream();

    ::krpc::Stream<SpaceCenter::Part> attached_part_stream();

    ::krpc::Stream<bool> decoupled_stream();

    ::krpc::Stream<float> impulse_stream();

    ::krpc::Stream<bool> is_omni_decoupler_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<bool> staged_stream();

    ::krpc::schema::ProcedureCall decouple_call();

    ::krpc::schema::ProcedureCall attached_part_call();

    ::krpc::schema::ProcedureCall decoupled_call();

    ::krpc::schema::ProcedureCall impulse_call();

    ::krpc::schema::ProcedureCall is_omni_decoupler_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall staged_call();
  };

  /**
   * A docking port. Obtained by calling SpaceCenter::Part::docking_port
   */
  class DockingPort : public krpc::Object<DockingPort> {
   public:
    explicit DockingPort(Client* client = nullptr, uint64_t id = 0);

    /**
     * The direction that docking port points in, in the given reference frame.
     * @return The direction as a unit vector.
     * @param referenceFrame The reference frame that the returned
     * direction is in.
     */
    std::tuple<double, double, double> direction(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The position of the docking port, in the given reference frame.
     * @return The position as a vector.
     * @param referenceFrame The reference frame that the returned
     * position vector is in.
     */
    std::tuple<double, double, double> position(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The rotation of the docking port, in the given reference frame.
     * @return The rotation as a quaternion of the form (x, y, z, w).
     * @param referenceFrame The reference frame that the returned
     * rotation is in.
     */
    std::tuple<double, double, double, double> rotation(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * Undocks the docking port and returns the new SpaceCenter::Vessel that is created.
     * This method can be called for either docking port in a docked pair.
     * Throws an exception if the docking port is not docked to anything.
     *
     * When called, the active vessel may change. It is therefore possible that,
     * after calling this function, the object(s) returned by previous call(s) to
     * SpaceCenter::active_vessel no longer refer to the active vessel.
     */
    SpaceCenter::Vessel undock();

    /**
     * Whether the docking port can be commanded to rotate while docked.
     */
    bool can_rotate();

    /**
     * The part that this docking port is docked to. Returns null if this
     * docking port is not docked to anything.
     */
    SpaceCenter::Part docked_part();

    /**
     * Whether the docking port has a shield.
     */
    bool has_shield();

    /**
     * Maximum rotation angle in degrees.
     */
    float maximum_rotation();

    /**
     * Minimum rotation angle in degrees.
     */
    float minimum_rotation();

    /**
     * The part object for this docking port.
     */
    SpaceCenter::Part part();

    /**
     * The distance a docking port must move away when it undocks before it
     * becomes ready to dock with another port, in meters.
     */
    float reengage_distance();

    /**
     * The reference frame that is fixed relative to this docking port, and
     * oriented with the port.
     *
     * - The origin is at the position of the docking port.
     * - The axes rotate with the docking port.
     * - The x-axis points out to the right side of the docking port.
     * - The y-axis points in the direction the docking port is facing.
     * - The z-axis points out of the bottom off the docking port.
     *
     * This reference frame is not necessarily equivalent to the reference frame
     * for the part, returned by SpaceCenter::Part::reference_frame.
     */
    SpaceCenter::ReferenceFrame reference_frame();

    /**
     * Lock rotation. When locked, allows auto-strut to work across the joint.
     */
    bool rotation_locked();

    /**
     * Lock rotation. When locked, allows auto-strut to work across the joint.
     */
    void set_rotation_locked(bool value);

    /**
     * Rotation target angle in degrees.
     */
    float rotation_target();

    /**
     * Rotation target angle in degrees.
     */
    void set_rotation_target(float value);

    /**
     * The state of the docking ports shield, if it has one.
     *
     * Returns true if the docking port has a shield, and the shield is
     * closed. Otherwise returns false. When set to true, the shield is
     * closed, and when set to false the shield is opened. If the docking
     * port does not have a shield, setting this attribute has no effect.
     */
    bool shielded();

    /**
     * The state of the docking ports shield, if it has one.
     *
     * Returns true if the docking port has a shield, and the shield is
     * closed. Otherwise returns false. When set to true, the shield is
     * closed, and when set to false the shield is opened. If the docking
     * port does not have a shield, setting this attribute has no effect.
     */
    void set_shielded(bool value);

    /**
     * The current state of the docking port.
     */
    SpaceCenter::DockingPortState state();

    ::krpc::Stream<std::tuple<double, double, double>> direction_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> position_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double, double>> rotation_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<SpaceCenter::Vessel> undock_stream();

    ::krpc::Stream<bool> can_rotate_stream();

    ::krpc::Stream<SpaceCenter::Part> docked_part_stream();

    ::krpc::Stream<bool> has_shield_stream();

    ::krpc::Stream<float> maximum_rotation_stream();

    ::krpc::Stream<float> minimum_rotation_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<float> reengage_distance_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> reference_frame_stream();

    ::krpc::Stream<bool> rotation_locked_stream();

    ::krpc::Stream<float> rotation_target_stream();

    ::krpc::Stream<bool> shielded_stream();

    ::krpc::Stream<SpaceCenter::DockingPortState> state_stream();

    ::krpc::schema::ProcedureCall direction_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall position_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall rotation_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall undock_call();

    ::krpc::schema::ProcedureCall can_rotate_call();

    ::krpc::schema::ProcedureCall docked_part_call();

    ::krpc::schema::ProcedureCall has_shield_call();

    ::krpc::schema::ProcedureCall maximum_rotation_call();

    ::krpc::schema::ProcedureCall minimum_rotation_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall reengage_distance_call();

    ::krpc::schema::ProcedureCall reference_frame_call();

    ::krpc::schema::ProcedureCall rotation_locked_call();

    ::krpc::schema::ProcedureCall set_rotation_locked_call(bool value);

    ::krpc::schema::ProcedureCall rotation_target_call();

    ::krpc::schema::ProcedureCall set_rotation_target_call(float value);

    ::krpc::schema::ProcedureCall shielded_call();

    ::krpc::schema::ProcedureCall set_shielded_call(bool value);

    ::krpc::schema::ProcedureCall state_call();
  };

  /**
   * An engine, including ones of various types.
   * For example liquid fuelled gimballed engines, solid rocket boosters and jet engines.
   * Obtained by calling SpaceCenter::Part::engine.
   *
   * For RCS thrusters SpaceCenter::Part::rcs.
   */
  class Engine : public krpc::Object<Engine> {
   public:
    explicit Engine(Client* client = nullptr, uint64_t id = 0);

    /**
     * The amount of thrust, in Newtons, that would be produced by the engine
     * when activated and with its throttle set to 100%.
     * Returns zero if the engine does not have any fuel.
     * Takes the given pressure into account.
     * @param pressure Atmospheric pressure in atmospheres
     */
    float available_thrust_at(double pressure);

    /**
     * The amount of thrust, in Newtons, that would be produced by the engine
     * when activated and fueled, with its throttle and throttle limiter set to 100%.
     * Takes the given pressure into account.
     * @param pressure Atmospheric pressure in atmospheres
     */
    float max_thrust_at(double pressure);

    /**
     * The specific impulse of the engine under the given pressure, in seconds. Returns zero
     * if the engine is not active.
     * @param pressure Atmospheric pressure in atmospheres
     */
    float specific_impulse_at(double pressure);

    /**
     * Toggle the current engine mode.
     */
    void toggle_mode();

    /**
     * Whether the engine is active. Setting this attribute may have no effect,
     * depending on SpaceCenter::Engine::can_shutdown and SpaceCenter::Engine::can_restart.
     */
    bool active();

    /**
     * Whether the engine is active. Setting this attribute may have no effect,
     * depending on SpaceCenter::Engine::can_shutdown and SpaceCenter::Engine::can_restart.
     */
    void set_active(bool value);

    /**
     * Whether the engine will automatically switch modes.
     */
    bool auto_mode_switch();

    /**
     * Whether the engine will automatically switch modes.
     */
    void set_auto_mode_switch(bool value);

    /**
     * The amount of thrust, in Newtons, that would be produced by the engine
     * when activated and with its throttle set to 100%.
     * Returns zero if the engine does not have any fuel.
     * Takes the engine's current SpaceCenter::Engine::thrust_limit and atmospheric conditions
     * into account.
     */
    float available_thrust();

    /**
     * The available torque, in Newton meters, that can be produced by this engine,
     * in the positive and negative pitch, roll and yaw axes of the vessel. These axes
     * correspond to the coordinate axes of the SpaceCenter::Vessel::reference_frame.
     * Returns zero if the engine is inactive, or not gimballed.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > available_torque();

    /**
     * Whether the engine can be restarted once shutdown. If the engine cannot be shutdown,
     * returns false. For example, this is true for liquid fueled rockets
     * and false for solid rocket boosters.
     */
    bool can_restart();

    /**
     * Whether the engine can be shutdown once activated. For example, this is
     * true for liquid fueled rockets and false for solid rocket boosters.
     */
    bool can_shutdown();

    /**
     * The gimbal limiter of the engine. A value between 0 and 1.
     * Returns 0 if the gimbal is locked.
     */
    float gimbal_limit();

    /**
     * The gimbal limiter of the engine. A value between 0 and 1.
     * Returns 0 if the gimbal is locked.
     */
    void set_gimbal_limit(float value);

    /**
     * Whether the engines gimbal is locked in place. Setting this attribute has
     * no effect if the engine is not gimballed.
     */
    bool gimbal_locked();

    /**
     * Whether the engines gimbal is locked in place. Setting this attribute has
     * no effect if the engine is not gimballed.
     */
    void set_gimbal_locked(bool value);

    /**
     * The range over which the gimbal can move, in degrees.
     * Returns 0 if the engine is not gimballed.
     */
    float gimbal_range();

    /**
     * Whether the engine is gimballed.
     */
    bool gimballed();

    /**
     * Whether the engine has any fuel available.
     */
    bool has_fuel();

    /**
     * Whether the engine has multiple modes of operation.
     */
    bool has_modes();

    /**
     * Whether the independent throttle is enabled for the engine.
     */
    bool independent_throttle();

    /**
     * Whether the independent throttle is enabled for the engine.
     */
    void set_independent_throttle(bool value);

    /**
     * The specific impulse of the engine at sea level on Kerbin, in seconds.
     */
    float kerbin_sea_level_specific_impulse();

    /**
     * The amount of thrust, in Newtons, that would be produced by the engine
     * when activated and fueled, with its throttle and throttle limiter set to 100%.
     */
    float max_thrust();

    /**
     * The maximum amount of thrust that can be produced by the engine in a
     * vacuum, in Newtons. This is the amount of thrust produced by the engine
     * when activated, SpaceCenter::Engine::thrust_limit is set to 100%, the main
     * vessel's throttle is set to 100% and the engine is in a vacuum.
     */
    float max_vacuum_thrust();

    /**
     * The name of the current engine mode.
     */
    std::string mode();

    /**
     * The name of the current engine mode.
     */
    void set_mode(std::string value);

    /**
     * The available modes for the engine.
     * A dictionary mapping mode names to SpaceCenter::Engine objects.
     */
    std::map<std::string, SpaceCenter::Engine> modes();

    /**
     * The part object for this engine.
     */
    SpaceCenter::Part part();

    /**
     * The names of the propellants that the engine consumes.
     */
    std::vector<std::string> propellant_names();

    /**
     * The ratio of resources that the engine consumes. A dictionary mapping resource names
     * to the ratio at which they are consumed by the engine.
     *
     * For example, if the ratios are 0.6 for LiquidFuel and 0.4 for Oxidizer, then for every
     * 0.6 units of LiquidFuel that the engine burns, it will burn 0.4 units of Oxidizer.
     */
    std::map<std::string, float> propellant_ratios();

    /**
     * The propellants that the engine consumes.
     */
    std::vector<SpaceCenter::Propellant> propellants();

    /**
     * The current specific impulse of the engine, in seconds. Returns zero
     * if the engine is not active.
     */
    float specific_impulse();

    /**
     * The current throttle setting for the engine. A value between 0 and 1.
     * This is not necessarily the same as the vessel's main throttle
     * setting, as some engines take time to adjust their throttle
     * (such as jet engines), or independent throttle may be enabled.
     *
     * When the engine's independent throttle is enabled
     * (see SpaceCenter::Engine::independent_throttle), can be used to set the throttle percentage.
     */
    float throttle();

    /**
     * The current throttle setting for the engine. A value between 0 and 1.
     * This is not necessarily the same as the vessel's main throttle
     * setting, as some engines take time to adjust their throttle
     * (such as jet engines), or independent throttle may be enabled.
     *
     * When the engine's independent throttle is enabled
     * (see SpaceCenter::Engine::independent_throttle), can be used to set the throttle percentage.
     */
    void set_throttle(float value);

    /**
     * Whether the SpaceCenter::Control::throttle affects the engine. For example,
     * this is true for liquid fueled rockets, and false for solid rocket
     * boosters.
     */
    bool throttle_locked();

    /**
     * The current amount of thrust being produced by the engine, in Newtons.
     */
    float thrust();

    /**
     * The thrust limiter of the engine. A value between 0 and 1. Setting this
     * attribute may have no effect, for example the thrust limit for a solid
     * rocket booster cannot be changed in flight.
     */
    float thrust_limit();

    /**
     * The thrust limiter of the engine. A value between 0 and 1. Setting this
     * attribute may have no effect, for example the thrust limit for a solid
     * rocket booster cannot be changed in flight.
     */
    void set_thrust_limit(float value);

    /**
     * The components of the engine that generate thrust.
     *
     * For example, this corresponds to the rocket nozzel on a solid rocket booster,
     * or the individual nozzels on a RAPIER engine.
     * The overall thrust produced by the engine, as reported by SpaceCenter::Engine::available_thrust,
     * SpaceCenter::Engine::max_thrust and others, is the sum of the thrust generated by each thruster.
     */
    std::vector<SpaceCenter::Thruster> thrusters();

    /**
     * The vacuum specific impulse of the engine, in seconds.
     */
    float vacuum_specific_impulse();

    ::krpc::Stream<float> available_thrust_at_stream(double pressure);

    ::krpc::Stream<float> max_thrust_at_stream(double pressure);

    ::krpc::Stream<float> specific_impulse_at_stream(double pressure);

    ::krpc::Stream<bool> active_stream();

    ::krpc::Stream<bool> auto_mode_switch_stream();

    ::krpc::Stream<float> available_thrust_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> available_torque_stream();

    ::krpc::Stream<bool> can_restart_stream();

    ::krpc::Stream<bool> can_shutdown_stream();

    ::krpc::Stream<float> gimbal_limit_stream();

    ::krpc::Stream<bool> gimbal_locked_stream();

    ::krpc::Stream<float> gimbal_range_stream();

    ::krpc::Stream<bool> gimballed_stream();

    ::krpc::Stream<bool> has_fuel_stream();

    ::krpc::Stream<bool> has_modes_stream();

    ::krpc::Stream<bool> independent_throttle_stream();

    ::krpc::Stream<float> kerbin_sea_level_specific_impulse_stream();

    ::krpc::Stream<float> max_thrust_stream();

    ::krpc::Stream<float> max_vacuum_thrust_stream();

    ::krpc::Stream<std::string> mode_stream();

    ::krpc::Stream<std::map<std::string, SpaceCenter::Engine>> modes_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<std::vector<std::string>> propellant_names_stream();

    ::krpc::Stream<std::map<std::string, float>> propellant_ratios_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Propellant>> propellants_stream();

    ::krpc::Stream<float> specific_impulse_stream();

    ::krpc::Stream<float> throttle_stream();

    ::krpc::Stream<bool> throttle_locked_stream();

    ::krpc::Stream<float> thrust_stream();

    ::krpc::Stream<float> thrust_limit_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Thruster>> thrusters_stream();

    ::krpc::Stream<float> vacuum_specific_impulse_stream();

    ::krpc::schema::ProcedureCall available_thrust_at_call(double pressure);

    ::krpc::schema::ProcedureCall max_thrust_at_call(double pressure);

    ::krpc::schema::ProcedureCall specific_impulse_at_call(double pressure);

    ::krpc::schema::ProcedureCall toggle_mode_call();

    ::krpc::schema::ProcedureCall active_call();

    ::krpc::schema::ProcedureCall set_active_call(bool value);

    ::krpc::schema::ProcedureCall auto_mode_switch_call();

    ::krpc::schema::ProcedureCall set_auto_mode_switch_call(bool value);

    ::krpc::schema::ProcedureCall available_thrust_call();

    ::krpc::schema::ProcedureCall available_torque_call();

    ::krpc::schema::ProcedureCall can_restart_call();

    ::krpc::schema::ProcedureCall can_shutdown_call();

    ::krpc::schema::ProcedureCall gimbal_limit_call();

    ::krpc::schema::ProcedureCall set_gimbal_limit_call(float value);

    ::krpc::schema::ProcedureCall gimbal_locked_call();

    ::krpc::schema::ProcedureCall set_gimbal_locked_call(bool value);

    ::krpc::schema::ProcedureCall gimbal_range_call();

    ::krpc::schema::ProcedureCall gimballed_call();

    ::krpc::schema::ProcedureCall has_fuel_call();

    ::krpc::schema::ProcedureCall has_modes_call();

    ::krpc::schema::ProcedureCall independent_throttle_call();

    ::krpc::schema::ProcedureCall set_independent_throttle_call(bool value);

    ::krpc::schema::ProcedureCall kerbin_sea_level_specific_impulse_call();

    ::krpc::schema::ProcedureCall max_thrust_call();

    ::krpc::schema::ProcedureCall max_vacuum_thrust_call();

    ::krpc::schema::ProcedureCall mode_call();

    ::krpc::schema::ProcedureCall set_mode_call(std::string value);

    ::krpc::schema::ProcedureCall modes_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall propellant_names_call();

    ::krpc::schema::ProcedureCall propellant_ratios_call();

    ::krpc::schema::ProcedureCall propellants_call();

    ::krpc::schema::ProcedureCall specific_impulse_call();

    ::krpc::schema::ProcedureCall throttle_call();

    ::krpc::schema::ProcedureCall set_throttle_call(float value);

    ::krpc::schema::ProcedureCall throttle_locked_call();

    ::krpc::schema::ProcedureCall thrust_call();

    ::krpc::schema::ProcedureCall thrust_limit_call();

    ::krpc::schema::ProcedureCall set_thrust_limit_call(float value);

    ::krpc::schema::ProcedureCall thrusters_call();

    ::krpc::schema::ProcedureCall vacuum_specific_impulse_call();
  };

  /**
   * Obtained by calling SpaceCenter::Part::experiment.
   */
  class Experiment : public krpc::Object<Experiment> {
   public:
    explicit Experiment(Client* client = nullptr, uint64_t id = 0);

    /**
     * Dump the experimental data contained by the experiment.
     */
    void dump();

    /**
     * Reset the experiment.
     */
    void reset();

    /**
     * Run the experiment.
     */
    void run();

    /**
     * Transmit all experimental data contained by this part.
     */
    void transmit();

    /**
     * Determines if the experiment is available given the current conditions.
     */
    bool available();

    /**
     * The name of the biome the experiment is currently in.
     */
    std::string biome();

    /**
     * The data contained in this experiment.
     */
    std::vector<SpaceCenter::ScienceData> data();

    /**
     * Whether the experiment has been deployed.
     */
    bool deployed();

    /**
     * Whether the experiment contains data.
     */
    bool has_data();

    /**
     * Whether the experiment is inoperable.
     */
    bool inoperable();

    /**
     * Internal name of the experiment, as used in
     * <a href="https://wiki.kerbalspaceprogram.com/wiki/CFG_File_Documentation">part cfg files</a>.
     */
    std::string name();

    /**
     * The part object for this experiment.
     */
    SpaceCenter::Part part();

    /**
     * Whether the experiment can be re-run.
     */
    bool rerunnable();

    /**
     * Containing information on the corresponding specific science result for the current
     * conditions. Returns null if the experiment is unavailable.
     */
    SpaceCenter::ScienceSubject science_subject();

    /**
     * Title of the experiment, as shown on the in-game UI.
     */
    std::string title();

    ::krpc::Stream<bool> available_stream();

    ::krpc::Stream<std::string> biome_stream();

    ::krpc::Stream<std::vector<SpaceCenter::ScienceData>> data_stream();

    ::krpc::Stream<bool> deployed_stream();

    ::krpc::Stream<bool> has_data_stream();

    ::krpc::Stream<bool> inoperable_stream();

    ::krpc::Stream<std::string> name_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<bool> rerunnable_stream();

    ::krpc::Stream<SpaceCenter::ScienceSubject> science_subject_stream();

    ::krpc::Stream<std::string> title_stream();

    ::krpc::schema::ProcedureCall dump_call();

    ::krpc::schema::ProcedureCall reset_call();

    ::krpc::schema::ProcedureCall run_call();

    ::krpc::schema::ProcedureCall transmit_call();

    ::krpc::schema::ProcedureCall available_call();

    ::krpc::schema::ProcedureCall biome_call();

    ::krpc::schema::ProcedureCall data_call();

    ::krpc::schema::ProcedureCall deployed_call();

    ::krpc::schema::ProcedureCall has_data_call();

    ::krpc::schema::ProcedureCall inoperable_call();

    ::krpc::schema::ProcedureCall name_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall rerunnable_call();

    ::krpc::schema::ProcedureCall science_subject_call();

    ::krpc::schema::ProcedureCall title_call();
  };

  /**
   * A fairing. Obtained by calling SpaceCenter::Part::fairing.
   * Supports both stock fairings, and those from the ProceduralFairings mod.
   */
  class Fairing : public krpc::Object<Fairing> {
   public:
    explicit Fairing(Client* client = nullptr, uint64_t id = 0);

    /**
     * Jettison the fairing. Has no effect if it has already been jettisoned.
     */
    void jettison();

    /**
     * Whether the fairing has been jettisoned.
     */
    bool jettisoned();

    /**
     * The part object for this fairing.
     */
    SpaceCenter::Part part();

    ::krpc::Stream<bool> jettisoned_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::schema::ProcedureCall jettison_call();

    ::krpc::schema::ProcedureCall jettisoned_call();

    ::krpc::schema::ProcedureCall part_call();
  };

  /**
   * Used to get flight telemetry for a vessel, by calling SpaceCenter::Vessel::flight.
   * All of the information returned by this class is given in the reference frame
   * passed to that method.
   * Obtained by calling SpaceCenter::Vessel::flight.
   *
   * To get orbital information, such as the apoapsis or inclination, see SpaceCenter::Orbit.
   */
  class Flight : public krpc::Object<Flight> {
   public:
    explicit Flight(Client* client = nullptr, uint64_t id = 0);

    /**
     * Simulate and return the total aerodynamic forces acting on the vessel,
     * if it where to be traveling with the given velocity at the given position in the
     * atmosphere of the given celestial body.
     * @return A vector pointing in the direction that the force acts,
     * with its magnitude equal to the strength of the force in Newtons.
     */
    std::tuple<double, double, double> simulate_aerodynamic_force_at(SpaceCenter::CelestialBody body, std::tuple<double, double, double> position, std::tuple<double, double, double> velocity);

    /**
     * The total aerodynamic forces acting on the vessel,
     * in reference frame SpaceCenter::ReferenceFrame.
     * @return A vector pointing in the direction that the force acts,
     * with its magnitude equal to the strength of the force in Newtons.
     */
    std::tuple<double, double, double> aerodynamic_force();

    /**
     * The pitch angle between the orientation of the vessel and its velocity vector,
     * in degrees.
     */
    float angle_of_attack();

    /**
     * The direction opposite to the normal of the vessels orbit,
     * in the reference frame SpaceCenter::ReferenceFrame.
     * @return The direction as a unit vector.
     */
    std::tuple<double, double, double> anti_normal();

    /**
     * The direction opposite to the radial direction of the vessels orbit,
     * in the reference frame SpaceCenter::ReferenceFrame.
     * @return The direction as a unit vector.
     */
    std::tuple<double, double, double> anti_radial();

    /**
     * The current density of the atmosphere around the vessel, in kg/m^3.
     */
    float atmosphere_density();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Ballistic_coefficient">ballistic coefficient</a>.
     *
     * Requires <a href="https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/">Ferram Aerospace Research</a>.
     */
    float ballistic_coefficient();

    /**
     * The altitude above the surface of the body, in meters. When over water, this is the altitude above the sea floor.
     * Measured from the center of mass of the vessel.
     */
    double bedrock_altitude();

    /**
     * The position of the center of mass of the vessel,
     * in the reference frame SpaceCenter::ReferenceFrame
     * @return The position as a vector.
     */
    std::tuple<double, double, double> center_of_mass();

    /**
     * The direction that the vessel is pointing in,
     * in the reference frame SpaceCenter::ReferenceFrame.
     * @return The direction as a unit vector.
     */
    std::tuple<double, double, double> direction();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Aerodynamic_force">aerodynamic drag</a> currently acting on the vessel.
     * @return A vector pointing in the direction of the force, with its magnitude
     * equal to the strength of the force in Newtons.
     */
    std::tuple<double, double, double> drag();

    /**
     * The coefficient of drag. This is the amount of drag produced by the vessel.
     * It depends on air speed, air density and wing area.
     *
     * Requires <a href="https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/">Ferram Aerospace Research</a>.
     */
    float drag_coefficient();

    /**
     * The dynamic pressure acting on the vessel, in Pascals. This is a measure of the
     * strength of the aerodynamic forces. It is equal to
     * \frac{1}{2} . \mbox{air density} . \mbox{velocity}^2.
     * It is commonly denoted Q.
     */
    float dynamic_pressure();

    /**
     * The elevation of the terrain under the vessel, in meters. This is the height of the terrain above sea level,
     * and is negative when the vessel is over the sea.
     */
    double elevation();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Equivalent_airspeed">equivalent air speed</a>
     * of the vessel, in meters per second.
     */
    float equivalent_air_speed();

    /**
     * The current G force acting on the vessel in g.
     */
    float g_force();

    /**
     * The heading of the vessel (its angle relative to north), in degrees.
     * A value between 0° and 360°.
     */
    float heading();

    /**
     * The horizontal speed of the vessel in meters per second,
     * in the reference frame SpaceCenter::ReferenceFrame.
     */
    double horizontal_speed();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Latitude">latitude</a> of the vessel for the body being orbited, in degrees.
     */
    double latitude();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Aerodynamic_force">aerodynamic lift</a>
     * currently acting on the vessel.
     * @return A vector pointing in the direction that the force acts,
     * with its magnitude equal to the strength of the force in Newtons.
     */
    std::tuple<double, double, double> lift();

    /**
     * The coefficient of lift. This is the amount of lift produced by the vessel, and
     * depends on air speed, air density and wing area.
     *
     * Requires <a href="https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/">Ferram Aerospace Research</a>.
     */
    float lift_coefficient();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Longitude">longitude</a> of the vessel for the body being orbited, in degrees.
     */
    double longitude();

    /**
     * The speed of the vessel, in multiples of the speed of sound.
     */
    float mach();

    /**
     * The altitude above sea level, in meters.
     * Measured from the center of mass of the vessel.
     */
    double mean_altitude();

    /**
     * The direction normal to the vessels orbit,
     * in the reference frame SpaceCenter::ReferenceFrame.
     * @return The direction as a unit vector.
     */
    std::tuple<double, double, double> normal();

    /**
     * The pitch of the vessel relative to the horizon, in degrees.
     * A value between -90° and +90°.
     */
    float pitch();

    /**
     * The prograde direction of the vessels orbit,
     * in the reference frame SpaceCenter::ReferenceFrame.
     * @return The direction as a unit vector.
     */
    std::tuple<double, double, double> prograde();

    /**
     * The radial direction of the vessels orbit,
     * in the reference frame SpaceCenter::ReferenceFrame.
     * @return The direction as a unit vector.
     */
    std::tuple<double, double, double> radial();

    /**
     * The retrograde direction of the vessels orbit,
     * in the reference frame SpaceCenter::ReferenceFrame.
     * @return The direction as a unit vector.
     */
    std::tuple<double, double, double> retrograde();

    /**
     * The vessels Reynolds number.
     *
     * Requires <a href="https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/">Ferram Aerospace Research</a>.
     */
    float reynolds_number();

    /**
     * The roll of the vessel relative to the horizon, in degrees.
     * A value between -180° and +180°.
     */
    float roll();

    /**
     * The rotation of the vessel, in the reference frame SpaceCenter::ReferenceFrame
     * @return The rotation as a quaternion of the form (x, y, z, w).
     */
    std::tuple<double, double, double, double> rotation();

    /**
     * The yaw angle between the orientation of the vessel and its velocity vector, in degrees.
     */
    float sideslip_angle();

    /**
     * The speed of the vessel in meters per second,
     * in the reference frame SpaceCenter::ReferenceFrame.
     */
    double speed();

    /**
     * The speed of sound, in the atmosphere around the vessel, in m/s.
     */
    float speed_of_sound();

    /**
     * The current amount of stall, between 0 and 1. A value greater than 0.005 indicates
     * a minor stall and a value greater than 0.5 indicates a large-scale stall.
     *
     * Requires <a href="https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/">Ferram Aerospace Research</a>.
     */
    float stall_fraction();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Total_air_temperature">static (ambient)
     * temperature</a> of the atmosphere around the vessel, in Kelvin.
     */
    float static_air_temperature();

    /**
     * The static atmospheric pressure acting on the vessel, in Pascals.
     */
    float static_pressure();

    /**
     * The static atmospheric pressure at mean sea level, in Pascals.
     */
    float static_pressure_at_msl();

    /**
     * The altitude above the surface of the body or sea level, whichever is closer, in meters.
     * Measured from the center of mass of the vessel.
     */
    double surface_altitude();

    /**
     * An estimate of the current terminal velocity of the vessel, in meters per second.
     * This is the speed at which the drag forces cancel out the force of gravity.
     */
    float terminal_velocity();

    /**
     * The thrust specific fuel consumption for the jet engines on the vessel. This is a
     * measure of the efficiency of the engines, with a lower value indicating a more
     * efficient vessel. This value is the number of Newtons of fuel that are burned,
     * per hour, to produce one newton of thrust.
     *
     * Requires <a href="https://forum.kerbalspaceprogram.com/index.php?/topic/19321-130-ferram-aerospace-research-v0159-liebe-82117/">Ferram Aerospace Research</a>.
     */
    float thrust_specific_fuel_consumption();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Total_air_temperature">total air temperature</a>
     * of the atmosphere around the vessel, in Kelvin.
     * This includes the SpaceCenter::Flight::static_air_temperature and the vessel's kinetic energy.
     */
    float total_air_temperature();

    /**
     * The <a href="https://en.wikipedia.org/wiki/True_airspeed">true air speed</a>
     * of the vessel, in meters per second.
     */
    float true_air_speed();

    /**
     * The velocity of the vessel, in the reference frame SpaceCenter::ReferenceFrame.
     * @return The velocity as a vector. The vector points in the direction of travel,
     * and its magnitude is the speed of the vessel in meters per second.
     */
    std::tuple<double, double, double> velocity();

    /**
     * The vertical speed of the vessel in meters per second,
     * in the reference frame SpaceCenter::ReferenceFrame.
     */
    double vertical_speed();

    ::krpc::Stream<std::tuple<double, double, double>> simulate_aerodynamic_force_at_stream(SpaceCenter::CelestialBody body, std::tuple<double, double, double> position, std::tuple<double, double, double> velocity);

    ::krpc::Stream<std::tuple<double, double, double>> aerodynamic_force_stream();

    ::krpc::Stream<float> angle_of_attack_stream();

    ::krpc::Stream<std::tuple<double, double, double>> anti_normal_stream();

    ::krpc::Stream<std::tuple<double, double, double>> anti_radial_stream();

    ::krpc::Stream<float> atmosphere_density_stream();

    ::krpc::Stream<float> ballistic_coefficient_stream();

    ::krpc::Stream<double> bedrock_altitude_stream();

    ::krpc::Stream<std::tuple<double, double, double>> center_of_mass_stream();

    ::krpc::Stream<std::tuple<double, double, double>> direction_stream();

    ::krpc::Stream<std::tuple<double, double, double>> drag_stream();

    ::krpc::Stream<float> drag_coefficient_stream();

    ::krpc::Stream<float> dynamic_pressure_stream();

    ::krpc::Stream<double> elevation_stream();

    ::krpc::Stream<float> equivalent_air_speed_stream();

    ::krpc::Stream<float> g_force_stream();

    ::krpc::Stream<float> heading_stream();

    ::krpc::Stream<double> horizontal_speed_stream();

    ::krpc::Stream<double> latitude_stream();

    ::krpc::Stream<std::tuple<double, double, double>> lift_stream();

    ::krpc::Stream<float> lift_coefficient_stream();

    ::krpc::Stream<double> longitude_stream();

    ::krpc::Stream<float> mach_stream();

    ::krpc::Stream<double> mean_altitude_stream();

    ::krpc::Stream<std::tuple<double, double, double>> normal_stream();

    ::krpc::Stream<float> pitch_stream();

    ::krpc::Stream<std::tuple<double, double, double>> prograde_stream();

    ::krpc::Stream<std::tuple<double, double, double>> radial_stream();

    ::krpc::Stream<std::tuple<double, double, double>> retrograde_stream();

    ::krpc::Stream<float> reynolds_number_stream();

    ::krpc::Stream<float> roll_stream();

    ::krpc::Stream<std::tuple<double, double, double, double>> rotation_stream();

    ::krpc::Stream<float> sideslip_angle_stream();

    ::krpc::Stream<double> speed_stream();

    ::krpc::Stream<float> speed_of_sound_stream();

    ::krpc::Stream<float> stall_fraction_stream();

    ::krpc::Stream<float> static_air_temperature_stream();

    ::krpc::Stream<float> static_pressure_stream();

    ::krpc::Stream<float> static_pressure_at_msl_stream();

    ::krpc::Stream<double> surface_altitude_stream();

    ::krpc::Stream<float> terminal_velocity_stream();

    ::krpc::Stream<float> thrust_specific_fuel_consumption_stream();

    ::krpc::Stream<float> total_air_temperature_stream();

    ::krpc::Stream<float> true_air_speed_stream();

    ::krpc::Stream<std::tuple<double, double, double>> velocity_stream();

    ::krpc::Stream<double> vertical_speed_stream();

    ::krpc::schema::ProcedureCall simulate_aerodynamic_force_at_call(SpaceCenter::CelestialBody body, std::tuple<double, double, double> position, std::tuple<double, double, double> velocity);

    ::krpc::schema::ProcedureCall aerodynamic_force_call();

    ::krpc::schema::ProcedureCall angle_of_attack_call();

    ::krpc::schema::ProcedureCall anti_normal_call();

    ::krpc::schema::ProcedureCall anti_radial_call();

    ::krpc::schema::ProcedureCall atmosphere_density_call();

    ::krpc::schema::ProcedureCall ballistic_coefficient_call();

    ::krpc::schema::ProcedureCall bedrock_altitude_call();

    ::krpc::schema::ProcedureCall center_of_mass_call();

    ::krpc::schema::ProcedureCall direction_call();

    ::krpc::schema::ProcedureCall drag_call();

    ::krpc::schema::ProcedureCall drag_coefficient_call();

    ::krpc::schema::ProcedureCall dynamic_pressure_call();

    ::krpc::schema::ProcedureCall elevation_call();

    ::krpc::schema::ProcedureCall equivalent_air_speed_call();

    ::krpc::schema::ProcedureCall g_force_call();

    ::krpc::schema::ProcedureCall heading_call();

    ::krpc::schema::ProcedureCall horizontal_speed_call();

    ::krpc::schema::ProcedureCall latitude_call();

    ::krpc::schema::ProcedureCall lift_call();

    ::krpc::schema::ProcedureCall lift_coefficient_call();

    ::krpc::schema::ProcedureCall longitude_call();

    ::krpc::schema::ProcedureCall mach_call();

    ::krpc::schema::ProcedureCall mean_altitude_call();

    ::krpc::schema::ProcedureCall normal_call();

    ::krpc::schema::ProcedureCall pitch_call();

    ::krpc::schema::ProcedureCall prograde_call();

    ::krpc::schema::ProcedureCall radial_call();

    ::krpc::schema::ProcedureCall retrograde_call();

    ::krpc::schema::ProcedureCall reynolds_number_call();

    ::krpc::schema::ProcedureCall roll_call();

    ::krpc::schema::ProcedureCall rotation_call();

    ::krpc::schema::ProcedureCall sideslip_angle_call();

    ::krpc::schema::ProcedureCall speed_call();

    ::krpc::schema::ProcedureCall speed_of_sound_call();

    ::krpc::schema::ProcedureCall stall_fraction_call();

    ::krpc::schema::ProcedureCall static_air_temperature_call();

    ::krpc::schema::ProcedureCall static_pressure_call();

    ::krpc::schema::ProcedureCall static_pressure_at_msl_call();

    ::krpc::schema::ProcedureCall surface_altitude_call();

    ::krpc::schema::ProcedureCall terminal_velocity_call();

    ::krpc::schema::ProcedureCall thrust_specific_fuel_consumption_call();

    ::krpc::schema::ProcedureCall total_air_temperature_call();

    ::krpc::schema::ProcedureCall true_air_speed_call();

    ::krpc::schema::ProcedureCall velocity_call();

    ::krpc::schema::ProcedureCall vertical_speed_call();
  };

  /**
   * Obtained by calling SpaceCenter::Part::add_force.
   */
  class Force : public krpc::Object<Force> {
   public:
    explicit Force(Client* client = nullptr, uint64_t id = 0);

    /**
     * Remove the force.
     */
    void remove();

    /**
     * The force vector, in Newtons.
     * @return A vector pointing in the direction that the force acts,
     * with its magnitude equal to the strength of the force in Newtons.
     */
    std::tuple<double, double, double> force_vector();

    /**
     * The force vector, in Newtons.
     * @return A vector pointing in the direction that the force acts,
     * with its magnitude equal to the strength of the force in Newtons.
     */
    void set_force_vector(std::tuple<double, double, double> value);

    /**
     * The part that this force is applied to.
     */
    SpaceCenter::Part part();

    /**
     * The position at which the force acts, in reference frame SpaceCenter::ReferenceFrame.
     * @return The position as a vector.
     */
    std::tuple<double, double, double> position();

    /**
     * The position at which the force acts, in reference frame SpaceCenter::ReferenceFrame.
     * @return The position as a vector.
     */
    void set_position(std::tuple<double, double, double> value);

    /**
     * The reference frame of the force vector and position.
     */
    SpaceCenter::ReferenceFrame reference_frame();

    /**
     * The reference frame of the force vector and position.
     */
    void set_reference_frame(SpaceCenter::ReferenceFrame value);

    ::krpc::Stream<std::tuple<double, double, double>> force_vector_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<std::tuple<double, double, double>> position_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> reference_frame_stream();

    ::krpc::schema::ProcedureCall remove_call();

    ::krpc::schema::ProcedureCall force_vector_call();

    ::krpc::schema::ProcedureCall set_force_vector_call(std::tuple<double, double, double> value);

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall position_call();

    ::krpc::schema::ProcedureCall set_position_call(std::tuple<double, double, double> value);

    ::krpc::schema::ProcedureCall reference_frame_call();

    ::krpc::schema::ProcedureCall set_reference_frame_call(SpaceCenter::ReferenceFrame value);
  };

  /**
   * An air intake. Obtained by calling SpaceCenter::Part::intake.
   */
  class Intake : public krpc::Object<Intake> {
   public:
    explicit Intake(Client* client = nullptr, uint64_t id = 0);

    /**
     * The area of the intake's opening, in square meters.
     */
    float area();

    /**
     * The rate of flow into the intake, in units of resource per second.
     */
    float flow();

    /**
     * Whether the intake is open.
     */
    bool open();

    /**
     * Whether the intake is open.
     */
    void set_open(bool value);

    /**
     * The part object for this intake.
     */
    SpaceCenter::Part part();

    /**
     * Speed of the flow into the intake, in m/s.
     */
    float speed();

    ::krpc::Stream<float> area_stream();

    ::krpc::Stream<float> flow_stream();

    ::krpc::Stream<bool> open_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<float> speed_stream();

    ::krpc::schema::ProcedureCall area_call();

    ::krpc::schema::ProcedureCall flow_call();

    ::krpc::schema::ProcedureCall open_call();

    ::krpc::schema::ProcedureCall set_open_call(bool value);

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall speed_call();
  };

  /**
   * A launch clamp. Obtained by calling SpaceCenter::Part::launch_clamp.
   */
  class LaunchClamp : public krpc::Object<LaunchClamp> {
   public:
    explicit LaunchClamp(Client* client = nullptr, uint64_t id = 0);

    /**
     * Releases the docking clamp. Has no effect if the clamp has already been released.
     */
    void release();

    /**
     * The part object for this launch clamp.
     */
    SpaceCenter::Part part();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::schema::ProcedureCall release_call();

    ::krpc::schema::ProcedureCall part_call();
  };

  /**
   * A place where craft can be launched from.
   * More of these can be added with mods like Kerbal Konstructs.
   */
  class LaunchSite : public krpc::Object<LaunchSite> {
   public:
    explicit LaunchSite(Client* client = nullptr, uint64_t id = 0);

    /**
     * The celestial body the launch site is on.
     */
    SpaceCenter::CelestialBody body();

    /**
     * Which editor is normally used for this launch site.
     */
    SpaceCenter::EditorFacility editor_facility();

    /**
     * The name of the launch site.
     */
    std::string name();

    ::krpc::Stream<SpaceCenter::CelestialBody> body_stream();

    ::krpc::Stream<SpaceCenter::EditorFacility> editor_facility_stream();

    ::krpc::Stream<std::string> name_stream();

    ::krpc::schema::ProcedureCall body_call();

    ::krpc::schema::ProcedureCall editor_facility_call();

    ::krpc::schema::ProcedureCall name_call();
  };

  /**
   * A landing leg. Obtained by calling SpaceCenter::Part::leg.
   */
  class Leg : public krpc::Object<Leg> {
   public:
    explicit Leg(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the leg is deployable.
     */
    bool deployable();

    /**
     * Whether the landing leg is deployed.
     *
     * Fixed landing legs are always deployed.
     * Returns an error if you try to deploy fixed landing gear.
     */
    bool deployed();

    /**
     * Whether the landing leg is deployed.
     *
     * Fixed landing legs are always deployed.
     * Returns an error if you try to deploy fixed landing gear.
     */
    void set_deployed(bool value);

    /**
     * Returns whether the leg is touching the ground.
     */
    bool is_grounded();

    /**
     * The part object for this landing leg.
     */
    SpaceCenter::Part part();

    /**
     * The current state of the landing leg.
     */
    SpaceCenter::LegState state();

    ::krpc::Stream<bool> deployable_stream();

    ::krpc::Stream<bool> deployed_stream();

    ::krpc::Stream<bool> is_grounded_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<SpaceCenter::LegState> state_stream();

    ::krpc::schema::ProcedureCall deployable_call();

    ::krpc::schema::ProcedureCall deployed_call();

    ::krpc::schema::ProcedureCall set_deployed_call(bool value);

    ::krpc::schema::ProcedureCall is_grounded_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall state_call();
  };

  /**
   * A light. Obtained by calling SpaceCenter::Part::light.
   */
  class Light : public krpc::Object<Light> {
   public:
    explicit Light(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the light is switched on.
     */
    bool active();

    /**
     * Whether the light is switched on.
     */
    void set_active(bool value);

    /**
     * Whether blinking is enabled.
     */
    bool blink();

    /**
     * Whether blinking is enabled.
     */
    void set_blink(bool value);

    /**
     * The blink rate of the light.
     */
    float blink_rate();

    /**
     * The blink rate of the light.
     */
    void set_blink_rate(float value);

    /**
     * The color of the light, as an RGB triple.
     */
    std::tuple<float, float, float> color();

    /**
     * The color of the light, as an RGB triple.
     */
    void set_color(std::tuple<float, float, float> value);

    /**
     * The part object for this light.
     */
    SpaceCenter::Part part();

    /**
     * The current power usage, in units of charge per second.
     */
    float power_usage();

    ::krpc::Stream<bool> active_stream();

    ::krpc::Stream<bool> blink_stream();

    ::krpc::Stream<float> blink_rate_stream();

    ::krpc::Stream<std::tuple<float, float, float>> color_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<float> power_usage_stream();

    ::krpc::schema::ProcedureCall active_call();

    ::krpc::schema::ProcedureCall set_active_call(bool value);

    ::krpc::schema::ProcedureCall blink_call();

    ::krpc::schema::ProcedureCall set_blink_call(bool value);

    ::krpc::schema::ProcedureCall blink_rate_call();

    ::krpc::schema::ProcedureCall set_blink_rate_call(float value);

    ::krpc::schema::ProcedureCall color_call();

    ::krpc::schema::ProcedureCall set_color_call(std::tuple<float, float, float> value);

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall power_usage_call();
  };

  /**
   * This can be used to interact with a specific part module. This includes part modules in
   * stock KSP, and those added by mods.
   *
   * In KSP, each part has zero or more
   * <a href="https://wiki.kerbalspaceprogram.com/wiki/CFG_File_Documentation#MODULES">PartModules</a>
   * associated with it. Each one contains some of the functionality of the part.
   * For example, an engine has a "ModuleEngines" part module that contains all the
   * functionality of an engine.
   */
  class Module : public krpc::Object<Module> {
   public:
    explicit Module(Client* client = nullptr, uint64_t id = 0);

    /**
     * Returns the value of a field with the given name.
     * @param name Name of the field.
     */
    std::string get_field(std::string name);

    /**
     * Returns the value of a field with the given identifier.
     * @param id Identifier of the field.
     */
    std::string get_field_by_id(std::string id);

    /**
     * true if the part has an action with the given name.
     * @param name
     */
    bool has_action(std::string name);

    /**
     * true if the part has an action with the given identifier.
     * @param id
     */
    bool has_action_with_id(std::string id);

    /**
     * true if the module has an event with the given name.
     * @param name
     */
    bool has_event(std::string name);

    /**
     * true if the module has an event with the given identifier.
     * @param id
     */
    bool has_event_with_id(std::string id);

    /**
     * Returns true if the module has a field with the given name.
     * @param name Name of the field.
     */
    bool has_field(std::string name);

    /**
     * Returns true if the module has a field with the given identifier.
     * @param id Identifier of the field.
     */
    bool has_field_with_id(std::string id);

    /**
     * Set the value of a field to its original value.
     * @param name Name of the field.
     */
    void reset_field(std::string name);

    /**
     * Set the value of a field to its original value.
     * @param id Identifier of the field.
     */
    void reset_field_by_id(std::string id);

    /**
     * Set the value of an action with the given name.
     * @param name
     * @param value
     */
    void set_action(std::string name, bool value);

    /**
     * Set the value of an action with the given identifier.
     * @param id
     * @param value
     */
    void set_action_by_id(std::string id, bool value);

    /**
     * Set the value of a field to true or false.
     * @param name Name of the field.
     * @param value Value to set.
     */
    void set_field_bool(std::string name, bool value);

    /**
     * Set the value of a field to true or false.
     * @param id Identifier of the field.
     * @param value Value to set.
     */
    void set_field_bool_by_id(std::string id, bool value);

    /**
     * Set the value of a field to the given floating point number.
     * @param name Name of the field.
     * @param value Value to set.
     */
    void set_field_float(std::string name, float value);

    /**
     * Set the value of a field to the given floating point number.
     * @param id Identifier of the field.
     * @param value Value to set.
     */
    void set_field_float_by_id(std::string id, float value);

    /**
     * Set the value of a field to the given integer number.
     * @param name Name of the field.
     * @param value Value to set.
     */
    void set_field_int(std::string name, int32_t value);

    /**
     * Set the value of a field to the given integer number.
     * @param id Identifier of the field.
     * @param value Value to set.
     */
    void set_field_int_by_id(std::string id, int32_t value);

    /**
     * Set the value of a field to the given string.
     * @param name Name of the field.
     * @param value Value to set.
     */
    void set_field_string(std::string name, std::string value);

    /**
     * Set the value of a field to the given string.
     * @param id Identifier of the field.
     * @param value Value to set.
     */
    void set_field_string_by_id(std::string id, std::string value);

    /**
     * Trigger the named event. Equivalent to clicking the button in the right-click menu
     * of the part.
     * @param name
     */
    void trigger_event(std::string name);

    /**
     * Trigger the event with the given identifier.
     * Equivalent to clicking the button in the right-click menu of the part.
     * @param id
     */
    void trigger_event_by_id(std::string id);

    /**
     * A list of all the names of the modules actions. These are the parts actions that can
     * be assigned to action groups in the in-game editor.
     */
    std::vector<std::string> actions();

    /**
     * A list of all the identifiers of the modules actions. These are the parts actions
     * that can be assigned to action groups in the in-game editor.
     */
    std::vector<std::string> actions_by_id();

    /**
     * A list of the names of all of the modules events. Events are the clickable buttons
     * visible in the right-click menu of the part.
     */
    std::vector<std::string> events();

    /**
     * A list of the identifiers of all of the modules events. Events are the clickable buttons
     * visible in the right-click menu of the part.
     */
    std::vector<std::string> events_by_id();

    /**
     * The modules field names and their associated values, as a dictionary.
     * These are the values visible in the right-click menu of the part.
     *
     * Throws an exception if there is more than one field with the same name.
     * In that case, use SpaceCenter::Module::fields_by_id to get the fields by identifier.
     */
    std::map<std::string, std::string> fields();

    /**
     * The modules field identifiers and their associated values, as a dictionary.
     * These are the values visible in the right-click menu of the part.
     */
    std::map<std::string, std::string> fields_by_id();

    /**
     * Name of the PartModule. For example, "ModuleEngines".
     */
    std::string name();

    /**
     * The part that contains this module.
     */
    SpaceCenter::Part part();

    ::krpc::Stream<std::string> get_field_stream(std::string name);

    ::krpc::Stream<std::string> get_field_by_id_stream(std::string id);

    ::krpc::Stream<bool> has_action_stream(std::string name);

    ::krpc::Stream<bool> has_action_with_id_stream(std::string id);

    ::krpc::Stream<bool> has_event_stream(std::string name);

    ::krpc::Stream<bool> has_event_with_id_stream(std::string id);

    ::krpc::Stream<bool> has_field_stream(std::string name);

    ::krpc::Stream<bool> has_field_with_id_stream(std::string id);

    ::krpc::Stream<std::vector<std::string>> actions_stream();

    ::krpc::Stream<std::vector<std::string>> actions_by_id_stream();

    ::krpc::Stream<std::vector<std::string>> events_stream();

    ::krpc::Stream<std::vector<std::string>> events_by_id_stream();

    ::krpc::Stream<std::map<std::string, std::string>> fields_stream();

    ::krpc::Stream<std::map<std::string, std::string>> fields_by_id_stream();

    ::krpc::Stream<std::string> name_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::schema::ProcedureCall get_field_call(std::string name);

    ::krpc::schema::ProcedureCall get_field_by_id_call(std::string id);

    ::krpc::schema::ProcedureCall has_action_call(std::string name);

    ::krpc::schema::ProcedureCall has_action_with_id_call(std::string id);

    ::krpc::schema::ProcedureCall has_event_call(std::string name);

    ::krpc::schema::ProcedureCall has_event_with_id_call(std::string id);

    ::krpc::schema::ProcedureCall has_field_call(std::string name);

    ::krpc::schema::ProcedureCall has_field_with_id_call(std::string id);

    ::krpc::schema::ProcedureCall reset_field_call(std::string name);

    ::krpc::schema::ProcedureCall reset_field_by_id_call(std::string id);

    ::krpc::schema::ProcedureCall set_action_call(std::string name, bool value);

    ::krpc::schema::ProcedureCall set_action_by_id_call(std::string id, bool value);

    ::krpc::schema::ProcedureCall set_field_bool_call(std::string name, bool value);

    ::krpc::schema::ProcedureCall set_field_bool_by_id_call(std::string id, bool value);

    ::krpc::schema::ProcedureCall set_field_float_call(std::string name, float value);

    ::krpc::schema::ProcedureCall set_field_float_by_id_call(std::string id, float value);

    ::krpc::schema::ProcedureCall set_field_int_call(std::string name, int32_t value);

    ::krpc::schema::ProcedureCall set_field_int_by_id_call(std::string id, int32_t value);

    ::krpc::schema::ProcedureCall set_field_string_call(std::string name, std::string value);

    ::krpc::schema::ProcedureCall set_field_string_by_id_call(std::string id, std::string value);

    ::krpc::schema::ProcedureCall trigger_event_call(std::string name);

    ::krpc::schema::ProcedureCall trigger_event_by_id_call(std::string id);

    ::krpc::schema::ProcedureCall actions_call();

    ::krpc::schema::ProcedureCall actions_by_id_call();

    ::krpc::schema::ProcedureCall events_call();

    ::krpc::schema::ProcedureCall events_by_id_call();

    ::krpc::schema::ProcedureCall fields_call();

    ::krpc::schema::ProcedureCall fields_by_id_call();

    ::krpc::schema::ProcedureCall name_call();

    ::krpc::schema::ProcedureCall part_call();
  };

  /**
   * Represents a maneuver node. Can be created using SpaceCenter::Control::add_node.
   */
  class Node : public krpc::Object<Node> {
   public:
    explicit Node(Client* client = nullptr, uint64_t id = 0);

    /**
     * Returns the burn vector for the maneuver node.
     * @param referenceFrame The reference frame that the returned vector is in.
     * Defaults to SpaceCenter::Vessel::orbital_reference_frame.
     * @return A vector whose direction is the direction of the maneuver node burn, and
     * magnitude is the delta-v of the burn in meters per second.
     *
     * Does not change when executing the maneuver node. See SpaceCenter::Node::remaining_burn_vector.
     */
    std::tuple<double, double, double> burn_vector(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The direction of the maneuver nodes burn.
     * @return The direction as a unit vector.
     * @param referenceFrame The reference frame that the returned
     * direction is in.
     */
    std::tuple<double, double, double> direction(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The position vector of the maneuver node in the given reference frame.
     * @return The position as a vector.
     * @param referenceFrame The reference frame that the returned
     * position vector is in.
     */
    std::tuple<double, double, double> position(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * Returns the remaining burn vector for the maneuver node.
     * @param referenceFrame The reference frame that the returned vector is in.
     * Defaults to SpaceCenter::Vessel::orbital_reference_frame.
     * @return A vector whose direction is the direction of the maneuver node burn, and
     * magnitude is the delta-v of the burn in meters per second.
     *
     * Changes as the maneuver node is executed. See SpaceCenter::Node::burn_vector.
     */
    std::tuple<double, double, double> remaining_burn_vector(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * Removes the maneuver node.
     */
    void remove();

    /**
     * The delta-v of the maneuver node, in meters per second.
     *
     * Does not change when executing the maneuver node. See SpaceCenter::Node::remaining_delta_v.
     */
    double delta_v();

    /**
     * The delta-v of the maneuver node, in meters per second.
     *
     * Does not change when executing the maneuver node. See SpaceCenter::Node::remaining_delta_v.
     */
    void set_delta_v(double value);

    /**
     * The magnitude of the maneuver nodes delta-v in the normal direction,
     * in meters per second.
     */
    double normal();

    /**
     * The magnitude of the maneuver nodes delta-v in the normal direction,
     * in meters per second.
     */
    void set_normal(double value);

    /**
     * The orbit that results from executing the maneuver node.
     */
    SpaceCenter::Orbit orbit();

    /**
     * The reference frame that is fixed relative to the maneuver node, and
     * orientated with the orbital prograde/normal/radial directions of the
     * original orbit at the maneuver node's position.
     *
     * - The origin is at the position of the maneuver node.
     * - The x-axis points in the orbital anti-radial direction of the original
     *   orbit, at the position of the maneuver node.
     * - The y-axis points in the orbital prograde direction of the original
     *   orbit, at the position of the maneuver node.
     * - The z-axis points in the orbital normal direction of the original orbit,
     *   at the position of the maneuver node.
     */
    SpaceCenter::ReferenceFrame orbital_reference_frame();

    /**
     * The magnitude of the maneuver nodes delta-v in the prograde direction,
     * in meters per second.
     */
    double prograde();

    /**
     * The magnitude of the maneuver nodes delta-v in the prograde direction,
     * in meters per second.
     */
    void set_prograde(double value);

    /**
     * The magnitude of the maneuver nodes delta-v in the radial direction,
     * in meters per second.
     */
    double radial();

    /**
     * The magnitude of the maneuver nodes delta-v in the radial direction,
     * in meters per second.
     */
    void set_radial(double value);

    /**
     * The reference frame that is fixed relative to the maneuver node's burn.
     *
     * - The origin is at the position of the maneuver node.
     * - The y-axis points in the direction of the burn.
     * - The x-axis and z-axis point in arbitrary but fixed directions.
     */
    SpaceCenter::ReferenceFrame reference_frame();

    /**
     * Gets the remaining delta-v of the maneuver node, in meters per second. Changes as the
     * node is executed. This is equivalent to the delta-v reported in-game.
     */
    double remaining_delta_v();

    /**
     * The time until the maneuver node will be encountered, in seconds.
     */
    double time_to();

    /**
     * The universal time at which the maneuver will occur, in seconds.
     */
    double ut();

    /**
     * The universal time at which the maneuver will occur, in seconds.
     */
    void set_ut(double value);

    ::krpc::Stream<std::tuple<double, double, double>> burn_vector_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> direction_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> position_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> remaining_burn_vector_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<double> delta_v_stream();

    ::krpc::Stream<double> normal_stream();

    ::krpc::Stream<SpaceCenter::Orbit> orbit_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> orbital_reference_frame_stream();

    ::krpc::Stream<double> prograde_stream();

    ::krpc::Stream<double> radial_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> reference_frame_stream();

    ::krpc::Stream<double> remaining_delta_v_stream();

    ::krpc::Stream<double> time_to_stream();

    ::krpc::Stream<double> ut_stream();

    ::krpc::schema::ProcedureCall burn_vector_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall direction_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall position_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall remaining_burn_vector_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall remove_call();

    ::krpc::schema::ProcedureCall delta_v_call();

    ::krpc::schema::ProcedureCall set_delta_v_call(double value);

    ::krpc::schema::ProcedureCall normal_call();

    ::krpc::schema::ProcedureCall set_normal_call(double value);

    ::krpc::schema::ProcedureCall orbit_call();

    ::krpc::schema::ProcedureCall orbital_reference_frame_call();

    ::krpc::schema::ProcedureCall prograde_call();

    ::krpc::schema::ProcedureCall set_prograde_call(double value);

    ::krpc::schema::ProcedureCall radial_call();

    ::krpc::schema::ProcedureCall set_radial_call(double value);

    ::krpc::schema::ProcedureCall reference_frame_call();

    ::krpc::schema::ProcedureCall remaining_delta_v_call();

    ::krpc::schema::ProcedureCall time_to_call();

    ::krpc::schema::ProcedureCall ut_call();

    ::krpc::schema::ProcedureCall set_ut_call(double value);
  };

  /**
   * Describes an orbit. For example, the orbit of a vessel, obtained by calling
   * SpaceCenter::Vessel::orbit, or a celestial body, obtained by calling
   * SpaceCenter::CelestialBody::orbit.
   */
  class Orbit : public krpc::Object<Orbit> {
   public:
    explicit Orbit(Client* client = nullptr, uint64_t id = 0);

    /**
     * Estimates and returns the distance at closest approach to a target orbit, in meters.
     * @param target Target orbit.
     */
    double distance_at_closest_approach(SpaceCenter::Orbit target);

    /**
     * The eccentric anomaly at the given universal time.
     * @param ut The universal time, in seconds.
     */
    double eccentric_anomaly_at_ut(double ut);

    /**
     * Returns the times at closest approach and corresponding distances, to a target orbit.
     * @return A list of two lists.
     * The first is a list of times at closest approach, as universal times in seconds.
     * The second is a list of corresponding distances at closest approach, in meters.
     * @param target Target orbit.
     * @param orbits The number of future orbits to search.
     */
    std::vector<std::vector<double> > list_closest_approaches(SpaceCenter::Orbit target, int32_t orbits);

    /**
     * The mean anomaly at the given time.
     * @param ut The universal time in seconds.
     */
    double mean_anomaly_at_ut(double ut);

    /**
     * The orbital speed at the given time, in meters per second.
     * @param time Time from now, in seconds.
     */
    double orbital_speed_at(double time);

    /**
     * The position at a given time, in the specified reference frame.
     * @return The position as a vector.
     * @param ut The universal time to measure the position at.
     * @param referenceFrame The reference frame that the returned
     * position vector is in.
     */
    std::tuple<double, double, double> position_at(double ut, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The orbital radius at the given time, in meters.
     * @param ut The universal time to measure the radius at.
     */
    double radius_at(double ut);

    /**
     * The orbital radius at the point in the orbit given by the true anomaly.
     * @param trueAnomaly The true anomaly.
     */
    double radius_at_true_anomaly(double true_anomaly);

    /**
     * Relative inclination of this orbit and the target orbit, in radians.
     * @param target Target orbit.
     */
    double relative_inclination(SpaceCenter::Orbit target);

    /**
     * Estimates and returns the time at closest approach to a target orbit.
     * @return The universal time at closest approach, in seconds.
     * @param target Target orbit.
     */
    double time_of_closest_approach(SpaceCenter::Orbit target);

    /**
     * The true anomaly of the ascending node with the given target orbit.
     * @param target Target orbit.
     */
    double true_anomaly_at_an(SpaceCenter::Orbit target);

    /**
     * The true anomaly of the descending node with the given target orbit.
     * @param target Target orbit.
     */
    double true_anomaly_at_dn(SpaceCenter::Orbit target);

    /**
     * The true anomaly at the given orbital radius.
     * @param radius The orbital radius in meters.
     */
    double true_anomaly_at_radius(double radius);

    /**
     * The true anomaly at the given time.
     * @param ut The universal time in seconds.
     */
    double true_anomaly_at_ut(double ut);

    /**
     * The universal time, in seconds, corresponding to the given true anomaly.
     * @param trueAnomaly True anomaly.
     */
    double ut_at_true_anomaly(double true_anomaly);

    /**
     * The direction from which the orbits longitude of ascending node is measured,
     * in the given reference frame.
     * @return The direction as a unit vector.
     * @param referenceFrame The reference frame that the returned
     * direction is in.
     */
    static std::tuple<double, double, double> reference_plane_direction(Client& client, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The direction that is normal to the orbits reference plane,
     * in the given reference frame.
     * The reference plane is the plane from which the orbits inclination is measured.
     * @return The direction as a unit vector.
     * @param referenceFrame The reference frame that the returned
     * direction is in.
     */
    static std::tuple<double, double, double> reference_plane_normal(Client& client, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * Gets the apoapsis of the orbit, in meters, from the center of mass
     * of the body being orbited.
     *
     * For the apoapsis altitude reported on the in-game map view,
     * use SpaceCenter::Orbit::apoapsis_altitude.
     */
    double apoapsis();

    /**
     * The apoapsis of the orbit, in meters, above the sea level of the body being orbited.
     *
     * This is equal to SpaceCenter::Orbit::apoapsis minus the equatorial radius of the body.
     */
    double apoapsis_altitude();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Argument_of_periapsis">argument of
     * periapsis</a>, in radians.
     */
    double argument_of_periapsis();

    /**
     * The celestial body (e.g. planet or moon) around which the object is orbiting.
     */
    SpaceCenter::CelestialBody body();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Eccentric_anomaly">eccentric anomaly</a>.
     */
    double eccentric_anomaly();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Orbital_eccentricity">eccentricity</a>
     * of the orbit.
     */
    double eccentricity();

    /**
     * The time since the epoch (the point at which the
     * <a href="https://en.wikipedia.org/wiki/Mean_anomaly">mean anomaly at epoch</a>
     * was measured, in seconds.
     */
    double epoch();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Orbital_inclination">inclination</a>
     * of the orbit,
     * in radians.
     */
    double inclination();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Longitude_of_the_ascending_node">longitude of
     * the ascending node</a>, in radians.
     */
    double longitude_of_ascending_node();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Mean_anomaly">mean anomaly</a>.
     */
    double mean_anomaly();

    /**
     * The <a href="https://en.wikipedia.org/wiki/Mean_anomaly">mean anomaly at epoch</a>.
     */
    double mean_anomaly_at_epoch();

    /**
     * If the object is going to change sphere of influence in the future, returns the new
     * orbit after the change. Otherwise returns null.
     */
    SpaceCenter::Orbit next_orbit();

    /**
     * The current orbital speed in meters per second.
     */
    double orbital_speed();

    /**
     * The periapsis of the orbit, in meters, from the center of mass
     * of the body being orbited.
     *
     * For the periapsis altitude reported on the in-game map view,
     * use SpaceCenter::Orbit::periapsis_altitude.
     */
    double periapsis();

    /**
     * The periapsis of the orbit, in meters, above the sea level of the body being orbited.
     *
     * This is equal to SpaceCenter::Orbit::periapsis minus the equatorial radius of the body.
     */
    double periapsis_altitude();

    /**
     * The orbital period, in seconds.
     */
    double period();

    /**
     * The current radius of the orbit, in meters. This is the distance between the center
     * of mass of the object in orbit, and the center of mass of the body around which it
     * is orbiting.
     *
     * This value will change over time if the orbit is elliptical.
     */
    double radius();

    /**
     * The semi-major axis of the orbit, in meters.
     */
    double semi_major_axis();

    /**
     * The semi-minor axis of the orbit, in meters.
     */
    double semi_minor_axis();

    /**
     * The current orbital speed of the object in meters per second.
     *
     * This value will change over time if the orbit is elliptical.
     */
    double speed();

    /**
     * The time until the object reaches apoapsis, in seconds.
     */
    double time_to_apoapsis();

    /**
     * The time until the object reaches periapsis, in seconds.
     */
    double time_to_periapsis();

    /**
     * The time until the object changes sphere of influence, in seconds. Returns NaN
     * if the object is not going to change sphere of influence.
     */
    double time_to_soi_change();

    /**
     * The <a href="https://en.wikipedia.org/wiki/True_anomaly">true anomaly</a>.
     */
    double true_anomaly();

    ::krpc::Stream<double> distance_at_closest_approach_stream(SpaceCenter::Orbit target);

    ::krpc::Stream<double> eccentric_anomaly_at_ut_stream(double ut);

    ::krpc::Stream<std::vector<std::vector<double> >> list_closest_approaches_stream(SpaceCenter::Orbit target, int32_t orbits);

    ::krpc::Stream<double> mean_anomaly_at_ut_stream(double ut);

    ::krpc::Stream<double> orbital_speed_at_stream(double time);

    ::krpc::Stream<std::tuple<double, double, double>> position_at_stream(double ut, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<double> radius_at_stream(double ut);

    ::krpc::Stream<double> radius_at_true_anomaly_stream(double true_anomaly);

    ::krpc::Stream<double> relative_inclination_stream(SpaceCenter::Orbit target);

    ::krpc::Stream<double> time_of_closest_approach_stream(SpaceCenter::Orbit target);

    ::krpc::Stream<double> true_anomaly_at_an_stream(SpaceCenter::Orbit target);

    ::krpc::Stream<double> true_anomaly_at_dn_stream(SpaceCenter::Orbit target);

    ::krpc::Stream<double> true_anomaly_at_radius_stream(double radius);

    ::krpc::Stream<double> true_anomaly_at_ut_stream(double ut);

    ::krpc::Stream<double> ut_at_true_anomaly_stream(double true_anomaly);

    static ::krpc::Stream<std::tuple<double, double, double>> reference_plane_direction_stream(Client& client, SpaceCenter::ReferenceFrame reference_frame);

    static ::krpc::Stream<std::tuple<double, double, double>> reference_plane_normal_stream(Client& client, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<double> apoapsis_stream();

    ::krpc::Stream<double> apoapsis_altitude_stream();

    ::krpc::Stream<double> argument_of_periapsis_stream();

    ::krpc::Stream<SpaceCenter::CelestialBody> body_stream();

    ::krpc::Stream<double> eccentric_anomaly_stream();

    ::krpc::Stream<double> eccentricity_stream();

    ::krpc::Stream<double> epoch_stream();

    ::krpc::Stream<double> inclination_stream();

    ::krpc::Stream<double> longitude_of_ascending_node_stream();

    ::krpc::Stream<double> mean_anomaly_stream();

    ::krpc::Stream<double> mean_anomaly_at_epoch_stream();

    ::krpc::Stream<SpaceCenter::Orbit> next_orbit_stream();

    ::krpc::Stream<double> orbital_speed_stream();

    ::krpc::Stream<double> periapsis_stream();

    ::krpc::Stream<double> periapsis_altitude_stream();

    ::krpc::Stream<double> period_stream();

    ::krpc::Stream<double> radius_stream();

    ::krpc::Stream<double> semi_major_axis_stream();

    ::krpc::Stream<double> semi_minor_axis_stream();

    ::krpc::Stream<double> speed_stream();

    ::krpc::Stream<double> time_to_apoapsis_stream();

    ::krpc::Stream<double> time_to_periapsis_stream();

    ::krpc::Stream<double> time_to_soi_change_stream();

    ::krpc::Stream<double> true_anomaly_stream();

    ::krpc::schema::ProcedureCall distance_at_closest_approach_call(SpaceCenter::Orbit target);

    ::krpc::schema::ProcedureCall eccentric_anomaly_at_ut_call(double ut);

    ::krpc::schema::ProcedureCall list_closest_approaches_call(SpaceCenter::Orbit target, int32_t orbits);

    ::krpc::schema::ProcedureCall mean_anomaly_at_ut_call(double ut);

    ::krpc::schema::ProcedureCall orbital_speed_at_call(double time);

    ::krpc::schema::ProcedureCall position_at_call(double ut, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall radius_at_call(double ut);

    ::krpc::schema::ProcedureCall radius_at_true_anomaly_call(double true_anomaly);

    ::krpc::schema::ProcedureCall relative_inclination_call(SpaceCenter::Orbit target);

    ::krpc::schema::ProcedureCall time_of_closest_approach_call(SpaceCenter::Orbit target);

    ::krpc::schema::ProcedureCall true_anomaly_at_an_call(SpaceCenter::Orbit target);

    ::krpc::schema::ProcedureCall true_anomaly_at_dn_call(SpaceCenter::Orbit target);

    ::krpc::schema::ProcedureCall true_anomaly_at_radius_call(double radius);

    ::krpc::schema::ProcedureCall true_anomaly_at_ut_call(double ut);

    ::krpc::schema::ProcedureCall ut_at_true_anomaly_call(double true_anomaly);

    static ::krpc::schema::ProcedureCall reference_plane_direction_call(Client& client, SpaceCenter::ReferenceFrame reference_frame);

    static ::krpc::schema::ProcedureCall reference_plane_normal_call(Client& client, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall apoapsis_call();

    ::krpc::schema::ProcedureCall apoapsis_altitude_call();

    ::krpc::schema::ProcedureCall argument_of_periapsis_call();

    ::krpc::schema::ProcedureCall body_call();

    ::krpc::schema::ProcedureCall eccentric_anomaly_call();

    ::krpc::schema::ProcedureCall eccentricity_call();

    ::krpc::schema::ProcedureCall epoch_call();

    ::krpc::schema::ProcedureCall inclination_call();

    ::krpc::schema::ProcedureCall longitude_of_ascending_node_call();

    ::krpc::schema::ProcedureCall mean_anomaly_call();

    ::krpc::schema::ProcedureCall mean_anomaly_at_epoch_call();

    ::krpc::schema::ProcedureCall next_orbit_call();

    ::krpc::schema::ProcedureCall orbital_speed_call();

    ::krpc::schema::ProcedureCall periapsis_call();

    ::krpc::schema::ProcedureCall periapsis_altitude_call();

    ::krpc::schema::ProcedureCall period_call();

    ::krpc::schema::ProcedureCall radius_call();

    ::krpc::schema::ProcedureCall semi_major_axis_call();

    ::krpc::schema::ProcedureCall semi_minor_axis_call();

    ::krpc::schema::ProcedureCall speed_call();

    ::krpc::schema::ProcedureCall time_to_apoapsis_call();

    ::krpc::schema::ProcedureCall time_to_periapsis_call();

    ::krpc::schema::ProcedureCall time_to_soi_change_call();

    ::krpc::schema::ProcedureCall true_anomaly_call();
  };

  /**
   * A parachute. Obtained by calling SpaceCenter::Part::parachute.
   */
  class Parachute : public krpc::Object<Parachute> {
   public:
    explicit Parachute(Client* client = nullptr, uint64_t id = 0);

    /**
     * Deploys the parachute. This has no effect if the parachute has already
     * been armed or deployed.
     */
    void arm();

    /**
     * Cuts the parachute.
     */
    void cut();

    /**
     * Deploys the parachute. This has no effect if the parachute has already
     * been deployed.
     */
    void deploy();

    /**
     * Whether the parachute has been armed or deployed.
     */
    bool armed();

    /**
     * The altitude at which the parachute will full deploy, in meters.
     * Only applicable to stock parachutes.
     */
    float deploy_altitude();

    /**
     * The altitude at which the parachute will full deploy, in meters.
     * Only applicable to stock parachutes.
     */
    void set_deploy_altitude(float value);

    /**
     * The minimum pressure at which the parachute will semi-deploy, in atmospheres.
     * Only applicable to stock parachutes.
     */
    float deploy_min_pressure();

    /**
     * The minimum pressure at which the parachute will semi-deploy, in atmospheres.
     * Only applicable to stock parachutes.
     */
    void set_deploy_min_pressure(float value);

    /**
     * Whether the parachute has been deployed.
     */
    bool deployed();

    /**
     * The part object for this parachute.
     */
    SpaceCenter::Part part();

    /**
     * The current state of the parachute.
     */
    SpaceCenter::ParachuteState state();

    ::krpc::Stream<bool> armed_stream();

    ::krpc::Stream<float> deploy_altitude_stream();

    ::krpc::Stream<float> deploy_min_pressure_stream();

    ::krpc::Stream<bool> deployed_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<SpaceCenter::ParachuteState> state_stream();

    ::krpc::schema::ProcedureCall arm_call();

    ::krpc::schema::ProcedureCall cut_call();

    ::krpc::schema::ProcedureCall deploy_call();

    ::krpc::schema::ProcedureCall armed_call();

    ::krpc::schema::ProcedureCall deploy_altitude_call();

    ::krpc::schema::ProcedureCall set_deploy_altitude_call(float value);

    ::krpc::schema::ProcedureCall deploy_min_pressure_call();

    ::krpc::schema::ProcedureCall set_deploy_min_pressure_call(float value);

    ::krpc::schema::ProcedureCall deployed_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall state_call();
  };

  /**
   * Represents an individual part. Vessels are made up of multiple parts.
   * Instances of this class can be obtained by several methods in SpaceCenter::Parts.
   */
  class Part : public krpc::Object<Part> {
   public:
    explicit Part(Client* client = nullptr, uint64_t id = 0);

    /**
     * Exert a constant force on the part, acting at the given position.
     * @return An object that can be used to remove or modify the force.
     * @param force A vector pointing in the direction that the force acts,
     * with its magnitude equal to the strength of the force in Newtons.
     * @param position The position at which the force acts, as a vector.
     * @param referenceFrame The reference frame that the
     * force and position are in.
     */
    SpaceCenter::Force add_force(std::tuple<double, double, double> force, std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The axis-aligned bounding box of the part in the given reference frame.
     * @return The positions of the minimum and maximum vertices of the box,
     * as position vectors.
     * @param referenceFrame The reference frame that the returned
     * position vectors are in.
     *
     * This is computed from the collision mesh of the part.
     * If the part is not collidable, the box has zero volume and is centered on
     * the SpaceCenter::Part::position of the part.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > bounding_box(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The position of the parts center of mass in the given reference frame.
     * If the part is physicsless, this is equivalent to SpaceCenter::Part::position.
     * @return The position as a vector.
     * @param referenceFrame The reference frame that the returned
     * position vector is in.
     */
    std::tuple<double, double, double> center_of_mass(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The direction the part points in, in the given reference frame.
     * @return The direction as a unit vector.
     * @param referenceFrame The reference frame that the returned
     * direction is in.
     */
    std::tuple<double, double, double> direction(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * Exert an instantaneous force on the part, acting at the given position.
     * @param force A vector pointing in the direction that the force acts,
     * with its magnitude equal to the strength of the force in Newtons.
     * @param position The position at which the force acts, as a vector.
     * @param referenceFrame The reference frame that the
     * force and position are in.
     *
     * The force is applied instantaneously in a single physics update.
     */
    void instantaneous_force(std::tuple<double, double, double> force, std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The position of the part in the given reference frame.
     * @return The position as a vector.
     * @param referenceFrame The reference frame that the returned
     * position vector is in.
     *
     * This is a fixed position in the part, defined by the parts model.
     * It s not necessarily the same as the parts center of mass.
     * Use SpaceCenter::Part::center_of_mass to get the parts center of mass.
     */
    std::tuple<double, double, double> position(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The rotation of the part, in the given reference frame.
     * @return The rotation as a quaternion of the form (x, y, z, w).
     * @param referenceFrame The reference frame that the returned
     * rotation is in.
     */
    std::tuple<double, double, double, double> rotation(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The linear velocity of the part in the given reference frame.
     * @return The velocity as a vector. The vector points in the direction of travel,
     * and its magnitude is the speed of the body in meters per second.
     * @param referenceFrame The reference frame that the returned
     * velocity vector is in.
     */
    std::tuple<double, double, double> velocity(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * An SpaceCenter::Antenna if the part is an antenna, otherwise null.
     *
     * If RemoteTech is installed, this will always return null.
     * To interact with RemoteTech antennas, use the RemoteTech service APIs.
     */
    SpaceCenter::Antenna antenna();

    /**
     * Auto-strut mode.
     */
    SpaceCenter::AutoStrutMode auto_strut_mode();

    /**
     * How many open seats the part has.
     */
    uint32_t available_seats();

    /**
     * Whether the part is axially attached to its parent, i.e. on the top
     * or bottom of its parent. If the part has no parent, returns false.
     */
    bool axially_attached();

    /**
     * A SpaceCenter::CargoBay if the part is a cargo bay, otherwise null.
     */
    SpaceCenter::CargoBay cargo_bay();

    /**
     * The reference frame that is fixed relative to this part, and centered on its
     * center of mass.
     *
     * - The origin is at the center of mass of the part, as returned by
     *   SpaceCenter::Part::center_of_mass.
     * - The axes rotate with the part.
     * - The x, y and z axis directions depend on the design of the part.
     *
     * For docking port parts, this reference frame is not necessarily equivalent to the
     * reference frame for the docking port, returned by
     * SpaceCenter::DockingPort::reference_frame.
     */
    SpaceCenter::ReferenceFrame center_of_mass_reference_frame();

    /**
     * The parts children. Returns an empty list if the part has no children.
     * This, in combination with SpaceCenter::Part::parent, can be used to traverse the vessels
     * parts tree.
     */
    std::vector<SpaceCenter::Part> children();

    /**
     * A SpaceCenter::ControlSurface if the part is an aerodynamic control surface,
     * otherwise null.
     */
    SpaceCenter::ControlSurface control_surface();

    /**
     * The cost of the part, in units of funds.
     */
    double cost();

    /**
     * Whether this part is crossfeed capable.
     */
    bool crossfeed();

    /**
     * The stage in which this part will be decoupled. Returns -1 if the part is never
     * decoupled from the vessel.
     */
    int32_t decouple_stage();

    /**
     * A SpaceCenter::Decoupler if the part is a decoupler, otherwise null.
     */
    SpaceCenter::Decoupler decoupler();

    /**
     * A SpaceCenter::DockingPort if the part is a docking port, otherwise null.
     */
    SpaceCenter::DockingPort docking_port();

    /**
     * The mass of the part, not including any resources it contains, in kilograms.
     * Returns zero if the part is massless.
     */
    double dry_mass();

    /**
     * The dynamic pressure acting on the part, in Pascals.
     */
    float dynamic_pressure();

    /**
     * An SpaceCenter::Engine if the part is an engine, otherwise null.
     */
    SpaceCenter::Engine engine();

    /**
     * An SpaceCenter::Experiment if the part contains a
     * single science experiment, otherwise null.
     *
     * Throws an exception if the part contains more than one experiment.
     * In that case, use SpaceCenter::Part::experiments to get the list of experiments in the part.
     */
    SpaceCenter::Experiment experiment();

    /**
     * A list of SpaceCenter::Experiment objects that the part contains.
     */
    std::vector<SpaceCenter::Experiment> experiments();

    /**
     * A SpaceCenter::Fairing if the part is a fairing, otherwise null.
     */
    SpaceCenter::Fairing fairing();

    /**
     * The asset URL for the part's flag.
     */
    std::string flag_url();

    /**
     * The asset URL for the part's flag.
     */
    void set_flag_url(std::string value);

    /**
     * The parts that are connected to this part via fuel lines, where the direction of the
     * fuel line is into this part.
     */
    std::vector<SpaceCenter::Part> fuel_lines_from();

    /**
     * The parts that are connected to this part via fuel lines, where the direction of the
     * fuel line is out of this part.
     */
    std::vector<SpaceCenter::Part> fuel_lines_to();

    /**
     * Whether the part is glowing.
     */
    void set_glow(bool value);

    /**
     * The color used to highlight the part, as an RGB triple.
     */
    std::tuple<double, double, double> highlight_color();

    /**
     * The color used to highlight the part, as an RGB triple.
     */
    void set_highlight_color(std::tuple<double, double, double> value);

    /**
     * Whether the part is highlighted.
     */
    bool highlighted();

    /**
     * Whether the part is highlighted.
     */
    void set_highlighted(bool value);

    /**
     * The impact tolerance of the part, in meters per second.
     */
    double impact_tolerance();

    /**
     * The inertia tensor of the part in the parts reference frame
     * (SpaceCenter::ReferenceFrame).
     * Returns the 3x3 matrix as a list of elements, in row-major order.
     */
    std::vector<double> inertia_tensor();

    /**
     * An SpaceCenter::Intake if the part is an intake, otherwise null.
     *
     * This includes any part that generates thrust. This covers many different types
     * of engine, including liquid fuel rockets, solid rocket boosters and jet engines.
     * For RCS thrusters see SpaceCenter::RCS.
     */
    SpaceCenter::Intake intake();

    /**
     * Whether this part is a fuel line.
     */
    bool is_fuel_line();

    /**
     * A SpaceCenter::LaunchClamp if the part is a launch clamp, otherwise null.
     */
    SpaceCenter::LaunchClamp launch_clamp();

    /**
     * A SpaceCenter::Leg if the part is a landing leg, otherwise null.
     */
    SpaceCenter::Leg leg();

    /**
     * A SpaceCenter::Light if the part is a light, otherwise null.
     */
    SpaceCenter::Light light();

    /**
     * The current mass of the part, including resources it contains, in kilograms.
     * Returns zero if the part is massless.
     */
    double mass();

    /**
     * Whether the part is
     * <a href="https://wiki.kerbalspaceprogram.com/wiki/Massless_part">massless</a>.
     */
    bool massless();

    /**
     * Maximum temperature that the skin of the part can survive, in Kelvin.
     */
    double max_skin_temperature();

    /**
     * Maximum temperature that the part can survive, in Kelvin.
     */
    double max_temperature();

    /**
     * The modules for this part.
     */
    std::vector<SpaceCenter::Module> modules();

    /**
     * The moment of inertia of the part in kg.m^2 around its center of mass
     * in the parts reference frame (SpaceCenter::ReferenceFrame).
     */
    std::tuple<double, double, double> moment_of_inertia();

    /**
     * Internal name of the part, as used in
     * <a href="https://wiki.kerbalspaceprogram.com/wiki/CFG_File_Documentation">part cfg files</a>.
     * For example "Mark1-2Pod".
     */
    std::string name();

    /**
     * A SpaceCenter::Parachute if the part is a parachute, otherwise null.
     */
    SpaceCenter::Parachute parachute();

    /**
     * The parts parent. Returns null if the part does not have a parent.
     * This, in combination with SpaceCenter::Part::children, can be used to traverse the vessels
     * parts tree.
     */
    SpaceCenter::Part parent();

    /**
     * Whether the part is radially attached to its parent, i.e. on the side of its parent.
     * If the part has no parent, returns false.
     */
    bool radially_attached();

    /**
     * A SpaceCenter::Radiator if the part is a radiator, otherwise null.
     */
    SpaceCenter::Radiator radiator();

    /**
     * A SpaceCenter::RCS if the part is an RCS block/thruster, otherwise null.
     */
    SpaceCenter::RCS rcs();

    /**
     * A SpaceCenter::ReactionWheel if the part is a reaction wheel, otherwise null.
     */
    SpaceCenter::ReactionWheel reaction_wheel();

    /**
     * The reference frame that is fixed relative to this part, and centered on a fixed
     * position within the part, defined by the parts model.
     *
     * - The origin is at the position of the part, as returned by
     *   SpaceCenter::Part::position.
     * - The axes rotate with the part.
     * - The x, y and z axis directions depend on the design of the part.
     *
     * For docking port parts, this reference frame is not necessarily equivalent to the
     * reference frame for the docking port, returned by
     * SpaceCenter::DockingPort::reference_frame.
     */
    SpaceCenter::ReferenceFrame reference_frame();

    /**
     * A SpaceCenter::ResourceConverter if the part is a resource converter,
     * otherwise null.
     */
    SpaceCenter::ResourceConverter resource_converter();

    /**
     * A SpaceCenter::ResourceDrain if the part is a resource drain, otherwise null.
     */
    SpaceCenter::ResourceDrain resource_drain();

    /**
     * A SpaceCenter::ResourceHarvester if the part is a resource harvester,
     * otherwise null.
     */
    SpaceCenter::ResourceHarvester resource_harvester();

    /**
     * A SpaceCenter::Resources object for the part.
     */
    SpaceCenter::Resources resources();

    /**
     * A SpaceCenter::RoboticController if the part is a robotic controller,
     * otherwise null.
     */
    SpaceCenter::RoboticController robotic_controller();

    /**
     * A SpaceCenter::RoboticHinge if the part is a robotic hinge, otherwise null.
     */
    SpaceCenter::RoboticHinge robotic_hinge();

    /**
     * A SpaceCenter::RoboticPiston if the part is a robotic piston, otherwise null.
     */
    SpaceCenter::RoboticPiston robotic_piston();

    /**
     * A SpaceCenter::RoboticRotation if the part is a robotic rotation servo, otherwise null.
     */
    SpaceCenter::RoboticRotation robotic_rotation();

    /**
     * A SpaceCenter::RoboticRotor if the part is a robotic rotor, otherwise null.
     */
    SpaceCenter::RoboticRotor robotic_rotor();

    /**
     * A SpaceCenter::Sensor if the part is a sensor, otherwise null.
     */
    SpaceCenter::Sensor sensor();

    /**
     * Whether the part is shielded from the exterior of the vessel, for example by a fairing.
     */
    bool shielded();

    /**
     * Temperature of the skin of the part, in Kelvin.
     */
    double skin_temperature();

    /**
     * A SpaceCenter::SolarPanel if the part is a solar panel, otherwise null.
     */
    SpaceCenter::SolarPanel solar_panel();

    /**
     * The stage in which this part will be activated. Returns -1 if the part is not
     * activated by staging.
     */
    int32_t stage();

    /**
     * The name tag for the part. Can be set to a custom string using the
     * in-game user interface.
     *
     * This string is shared with
     * <a href="https://forum.kerbalspaceprogram.com/index.php?/topic/61827-/">kOS</a>
     * if it is installed.
     */
    std::string tag();

    /**
     * The name tag for the part. Can be set to a custom string using the
     * in-game user interface.
     *
     * This string is shared with
     * <a href="https://forum.kerbalspaceprogram.com/index.php?/topic/61827-/">kOS</a>
     * if it is installed.
     */
    void set_tag(std::string value);

    /**
     * Temperature of the part, in Kelvin.
     */
    double temperature();

    /**
     * The rate at which heat energy is conducting into or out of the part via contact with
     * other parts. Measured in energy per unit time, or power, in Watts.
     * A positive value means the part is gaining heat energy, and negative means it is
     * losing heat energy.
     */
    float thermal_conduction_flux();

    /**
     * The rate at which heat energy is convecting into or out of the part from the
     * surrounding atmosphere. Measured in energy per unit time, or power, in Watts.
     * A positive value means the part is gaining heat energy, and negative means it is
     * losing heat energy.
     */
    float thermal_convection_flux();

    /**
     * The rate at which heat energy is begin generated by the part.
     * For example, some engines generate heat by combusting fuel.
     * Measured in energy per unit time, or power, in Watts.
     * A positive value means the part is gaining heat energy, and negative means it is losing
     * heat energy.
     */
    float thermal_internal_flux();

    /**
     * A measure of how much energy it takes to increase the internal temperature of the part,
     * in Joules per Kelvin.
     */
    float thermal_mass();

    /**
     * The rate at which heat energy is radiating into or out of the part from the surrounding
     * environment. Measured in energy per unit time, or power, in Watts.
     * A positive value means the part is gaining heat energy, and negative means it is
     * losing heat energy.
     */
    float thermal_radiation_flux();

    /**
     * A measure of how much energy it takes to increase the temperature of the resources
     * contained in the part, in Joules per Kelvin.
     */
    float thermal_resource_mass();

    /**
     * A measure of how much energy it takes to increase the skin temperature of the part,
     * in Joules per Kelvin.
     */
    float thermal_skin_mass();

    /**
     * The rate at which heat energy is transferring between the part's skin and its internals.
     * Measured in energy per unit time, or power, in Watts.
     * A positive value means the part's internals are gaining heat energy,
     * and negative means its skin is gaining heat energy.
     */
    float thermal_skin_to_internal_flux();

    /**
     * Title of the part, as shown when the part is right clicked in-game. For example "Mk1-2 Command Pod".
     */
    std::string title();

    /**
     * The vessel that contains this part.
     */
    SpaceCenter::Vessel vessel();

    /**
     * A SpaceCenter::Wheel if the part is a wheel, otherwise null.
     */
    SpaceCenter::Wheel wheel();

    ::krpc::Stream<SpaceCenter::Force> add_force_stream(std::tuple<double, double, double> force, std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> bounding_box_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> center_of_mass_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> direction_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> position_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double, double>> rotation_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> velocity_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<SpaceCenter::Antenna> antenna_stream();

    ::krpc::Stream<SpaceCenter::AutoStrutMode> auto_strut_mode_stream();

    ::krpc::Stream<uint32_t> available_seats_stream();

    ::krpc::Stream<bool> axially_attached_stream();

    ::krpc::Stream<SpaceCenter::CargoBay> cargo_bay_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> center_of_mass_reference_frame_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Part>> children_stream();

    ::krpc::Stream<SpaceCenter::ControlSurface> control_surface_stream();

    ::krpc::Stream<double> cost_stream();

    ::krpc::Stream<bool> crossfeed_stream();

    ::krpc::Stream<int32_t> decouple_stage_stream();

    ::krpc::Stream<SpaceCenter::Decoupler> decoupler_stream();

    ::krpc::Stream<SpaceCenter::DockingPort> docking_port_stream();

    ::krpc::Stream<double> dry_mass_stream();

    ::krpc::Stream<float> dynamic_pressure_stream();

    ::krpc::Stream<SpaceCenter::Engine> engine_stream();

    ::krpc::Stream<SpaceCenter::Experiment> experiment_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Experiment>> experiments_stream();

    ::krpc::Stream<SpaceCenter::Fairing> fairing_stream();

    ::krpc::Stream<std::string> flag_url_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Part>> fuel_lines_from_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Part>> fuel_lines_to_stream();

    ::krpc::Stream<std::tuple<double, double, double>> highlight_color_stream();

    ::krpc::Stream<bool> highlighted_stream();

    ::krpc::Stream<double> impact_tolerance_stream();

    ::krpc::Stream<std::vector<double>> inertia_tensor_stream();

    ::krpc::Stream<SpaceCenter::Intake> intake_stream();

    ::krpc::Stream<bool> is_fuel_line_stream();

    ::krpc::Stream<SpaceCenter::LaunchClamp> launch_clamp_stream();

    ::krpc::Stream<SpaceCenter::Leg> leg_stream();

    ::krpc::Stream<SpaceCenter::Light> light_stream();

    ::krpc::Stream<double> mass_stream();

    ::krpc::Stream<bool> massless_stream();

    ::krpc::Stream<double> max_skin_temperature_stream();

    ::krpc::Stream<double> max_temperature_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Module>> modules_stream();

    ::krpc::Stream<std::tuple<double, double, double>> moment_of_inertia_stream();

    ::krpc::Stream<std::string> name_stream();

    ::krpc::Stream<SpaceCenter::Parachute> parachute_stream();

    ::krpc::Stream<SpaceCenter::Part> parent_stream();

    ::krpc::Stream<bool> radially_attached_stream();

    ::krpc::Stream<SpaceCenter::Radiator> radiator_stream();

    ::krpc::Stream<SpaceCenter::RCS> rcs_stream();

    ::krpc::Stream<SpaceCenter::ReactionWheel> reaction_wheel_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> reference_frame_stream();

    ::krpc::Stream<SpaceCenter::ResourceConverter> resource_converter_stream();

    ::krpc::Stream<SpaceCenter::ResourceDrain> resource_drain_stream();

    ::krpc::Stream<SpaceCenter::ResourceHarvester> resource_harvester_stream();

    ::krpc::Stream<SpaceCenter::Resources> resources_stream();

    ::krpc::Stream<SpaceCenter::RoboticController> robotic_controller_stream();

    ::krpc::Stream<SpaceCenter::RoboticHinge> robotic_hinge_stream();

    ::krpc::Stream<SpaceCenter::RoboticPiston> robotic_piston_stream();

    ::krpc::Stream<SpaceCenter::RoboticRotation> robotic_rotation_stream();

    ::krpc::Stream<SpaceCenter::RoboticRotor> robotic_rotor_stream();

    ::krpc::Stream<SpaceCenter::Sensor> sensor_stream();

    ::krpc::Stream<bool> shielded_stream();

    ::krpc::Stream<double> skin_temperature_stream();

    ::krpc::Stream<SpaceCenter::SolarPanel> solar_panel_stream();

    ::krpc::Stream<int32_t> stage_stream();

    ::krpc::Stream<std::string> tag_stream();

    ::krpc::Stream<double> temperature_stream();

    ::krpc::Stream<float> thermal_conduction_flux_stream();

    ::krpc::Stream<float> thermal_convection_flux_stream();

    ::krpc::Stream<float> thermal_internal_flux_stream();

    ::krpc::Stream<float> thermal_mass_stream();

    ::krpc::Stream<float> thermal_radiation_flux_stream();

    ::krpc::Stream<float> thermal_resource_mass_stream();

    ::krpc::Stream<float> thermal_skin_mass_stream();

    ::krpc::Stream<float> thermal_skin_to_internal_flux_stream();

    ::krpc::Stream<std::string> title_stream();

    ::krpc::Stream<SpaceCenter::Vessel> vessel_stream();

    ::krpc::Stream<SpaceCenter::Wheel> wheel_stream();

    ::krpc::schema::ProcedureCall add_force_call(std::tuple<double, double, double> force, std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall bounding_box_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall center_of_mass_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall direction_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall instantaneous_force_call(std::tuple<double, double, double> force, std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall position_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall rotation_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall velocity_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall antenna_call();

    ::krpc::schema::ProcedureCall auto_strut_mode_call();

    ::krpc::schema::ProcedureCall available_seats_call();

    ::krpc::schema::ProcedureCall axially_attached_call();

    ::krpc::schema::ProcedureCall cargo_bay_call();

    ::krpc::schema::ProcedureCall center_of_mass_reference_frame_call();

    ::krpc::schema::ProcedureCall children_call();

    ::krpc::schema::ProcedureCall control_surface_call();

    ::krpc::schema::ProcedureCall cost_call();

    ::krpc::schema::ProcedureCall crossfeed_call();

    ::krpc::schema::ProcedureCall decouple_stage_call();

    ::krpc::schema::ProcedureCall decoupler_call();

    ::krpc::schema::ProcedureCall docking_port_call();

    ::krpc::schema::ProcedureCall dry_mass_call();

    ::krpc::schema::ProcedureCall dynamic_pressure_call();

    ::krpc::schema::ProcedureCall engine_call();

    ::krpc::schema::ProcedureCall experiment_call();

    ::krpc::schema::ProcedureCall experiments_call();

    ::krpc::schema::ProcedureCall fairing_call();

    ::krpc::schema::ProcedureCall flag_url_call();

    ::krpc::schema::ProcedureCall set_flag_url_call(std::string value);

    ::krpc::schema::ProcedureCall fuel_lines_from_call();

    ::krpc::schema::ProcedureCall fuel_lines_to_call();

    ::krpc::schema::ProcedureCall set_glow_call(bool value);

    ::krpc::schema::ProcedureCall highlight_color_call();

    ::krpc::schema::ProcedureCall set_highlight_color_call(std::tuple<double, double, double> value);

    ::krpc::schema::ProcedureCall highlighted_call();

    ::krpc::schema::ProcedureCall set_highlighted_call(bool value);

    ::krpc::schema::ProcedureCall impact_tolerance_call();

    ::krpc::schema::ProcedureCall inertia_tensor_call();

    ::krpc::schema::ProcedureCall intake_call();

    ::krpc::schema::ProcedureCall is_fuel_line_call();

    ::krpc::schema::ProcedureCall launch_clamp_call();

    ::krpc::schema::ProcedureCall leg_call();

    ::krpc::schema::ProcedureCall light_call();

    ::krpc::schema::ProcedureCall mass_call();

    ::krpc::schema::ProcedureCall massless_call();

    ::krpc::schema::ProcedureCall max_skin_temperature_call();

    ::krpc::schema::ProcedureCall max_temperature_call();

    ::krpc::schema::ProcedureCall modules_call();

    ::krpc::schema::ProcedureCall moment_of_inertia_call();

    ::krpc::schema::ProcedureCall name_call();

    ::krpc::schema::ProcedureCall parachute_call();

    ::krpc::schema::ProcedureCall parent_call();

    ::krpc::schema::ProcedureCall radially_attached_call();

    ::krpc::schema::ProcedureCall radiator_call();

    ::krpc::schema::ProcedureCall rcs_call();

    ::krpc::schema::ProcedureCall reaction_wheel_call();

    ::krpc::schema::ProcedureCall reference_frame_call();

    ::krpc::schema::ProcedureCall resource_converter_call();

    ::krpc::schema::ProcedureCall resource_drain_call();

    ::krpc::schema::ProcedureCall resource_harvester_call();

    ::krpc::schema::ProcedureCall resources_call();

    ::krpc::schema::ProcedureCall robotic_controller_call();

    ::krpc::schema::ProcedureCall robotic_hinge_call();

    ::krpc::schema::ProcedureCall robotic_piston_call();

    ::krpc::schema::ProcedureCall robotic_rotation_call();

    ::krpc::schema::ProcedureCall robotic_rotor_call();

    ::krpc::schema::ProcedureCall sensor_call();

    ::krpc::schema::ProcedureCall shielded_call();

    ::krpc::schema::ProcedureCall skin_temperature_call();

    ::krpc::schema::ProcedureCall solar_panel_call();

    ::krpc::schema::ProcedureCall stage_call();

    ::krpc::schema::ProcedureCall tag_call();

    ::krpc::schema::ProcedureCall set_tag_call(std::string value);

    ::krpc::schema::ProcedureCall temperature_call();

    ::krpc::schema::ProcedureCall thermal_conduction_flux_call();

    ::krpc::schema::ProcedureCall thermal_convection_flux_call();

    ::krpc::schema::ProcedureCall thermal_internal_flux_call();

    ::krpc::schema::ProcedureCall thermal_mass_call();

    ::krpc::schema::ProcedureCall thermal_radiation_flux_call();

    ::krpc::schema::ProcedureCall thermal_resource_mass_call();

    ::krpc::schema::ProcedureCall thermal_skin_mass_call();

    ::krpc::schema::ProcedureCall thermal_skin_to_internal_flux_call();

    ::krpc::schema::ProcedureCall title_call();

    ::krpc::schema::ProcedureCall vessel_call();

    ::krpc::schema::ProcedureCall wheel_call();
  };

  /**
   * Instances of this class are used to interact with the parts of a vessel.
   * An instance can be obtained by calling SpaceCenter::Vessel::parts.
   */
  class Parts : public krpc::Object<Parts> {
   public:
    explicit Parts(Client* client = nullptr, uint64_t id = 0);

    /**
     * A list of all parts that are decoupled in the given stage.
     * @param stage
     */
    std::vector<SpaceCenter::Part> in_decouple_stage(int32_t stage);

    /**
     * A list of all parts that are activated in the given stage.
     * @param stage
     */
    std::vector<SpaceCenter::Part> in_stage(int32_t stage);

    /**
     * A list of modules (combined across all parts in the vessel) whose
     * SpaceCenter::Module::name is moduleName.
     * @param moduleName
     */
    std::vector<SpaceCenter::Module> modules_with_name(std::string module_name);

    /**
     * A list of all parts that contain a SpaceCenter::Module whose
     * SpaceCenter::Module::name is moduleName.
     * @param moduleName
     */
    std::vector<SpaceCenter::Part> with_module(std::string module_name);

    /**
     * A list of parts whose SpaceCenter::Part::name is name.
     * @param name
     */
    std::vector<SpaceCenter::Part> with_name(std::string name);

    /**
     * A list of all parts whose SpaceCenter::Part::tag is tag.
     * @param tag
     */
    std::vector<SpaceCenter::Part> with_tag(std::string tag);

    /**
     * A list of all parts whose SpaceCenter::Part::title is title.
     * @param title
     */
    std::vector<SpaceCenter::Part> with_title(std::string title);

    /**
     * A list of all of the vessels parts.
     */
    std::vector<SpaceCenter::Part> all();

    /**
     * A list of all antennas in the vessel.
     *
     * If RemoteTech is installed, this will always return an empty list.
     * To interact with RemoteTech antennas, use the RemoteTech service APIs.
     */
    std::vector<SpaceCenter::Antenna> antennas();

    /**
     * A list of all cargo bays in the vessel.
     */
    std::vector<SpaceCenter::CargoBay> cargo_bays();

    /**
     * A list of all control surfaces in the vessel.
     */
    std::vector<SpaceCenter::ControlSurface> control_surfaces();

    /**
     * The part from which the vessel is controlled.
     */
    SpaceCenter::Part controlling();

    /**
     * The part from which the vessel is controlled.
     */
    void set_controlling(SpaceCenter::Part value);

    /**
     * A list of all decouplers in the vessel.
     */
    std::vector<SpaceCenter::Decoupler> decouplers();

    /**
     * A list of all docking ports in the vessel.
     */
    std::vector<SpaceCenter::DockingPort> docking_ports();

    /**
     * A list of all engines in the vessel.
     *
     * This includes any part that generates thrust. This covers many different types
     * of engine, including liquid fuel rockets, solid rocket boosters, jet engines and
     * RCS thrusters.
     */
    std::vector<SpaceCenter::Engine> engines();

    /**
     * A list of all science experiments in the vessel.
     */
    std::vector<SpaceCenter::Experiment> experiments();

    /**
     * A list of all fairings in the vessel.
     */
    std::vector<SpaceCenter::Fairing> fairings();

    /**
     * A list of all intakes in the vessel.
     */
    std::vector<SpaceCenter::Intake> intakes();

    /**
     * A list of all launch clamps attached to the vessel.
     */
    std::vector<SpaceCenter::LaunchClamp> launch_clamps();

    /**
     * A list of all landing legs attached to the vessel.
     */
    std::vector<SpaceCenter::Leg> legs();

    /**
     * A list of all lights in the vessel.
     */
    std::vector<SpaceCenter::Light> lights();

    /**
     * A list of all parachutes in the vessel.
     */
    std::vector<SpaceCenter::Parachute> parachutes();

    /**
     * A list of all radiators in the vessel.
     */
    std::vector<SpaceCenter::Radiator> radiators();

    /**
     * A list of all RCS blocks/thrusters in the vessel.
     */
    std::vector<SpaceCenter::RCS> rcs();

    /**
     * A list of all reaction wheels in the vessel.
     */
    std::vector<SpaceCenter::ReactionWheel> reaction_wheels();

    /**
     * A list of all resource converters in the vessel.
     */
    std::vector<SpaceCenter::ResourceConverter> resource_converters();

    /**
     * A list of all resource drains in the vessel.
     */
    std::vector<SpaceCenter::ResourceDrain> resource_drains();

    /**
     * A list of all resource harvesters in the vessel.
     */
    std::vector<SpaceCenter::ResourceHarvester> resource_harvesters();

    /**
     * A list of all robotic hinges in the vessel.
     */
    std::vector<SpaceCenter::RoboticHinge> robotic_hinges();

    /**
     * A list of all robotic pistons in the vessel.
     */
    std::vector<SpaceCenter::RoboticPiston> robotic_pistons();

    /**
     * A list of all robotic rotations in the vessel.
     */
    std::vector<SpaceCenter::RoboticRotation> robotic_rotations();

    /**
     * A list of all robotic rotors in the vessel.
     */
    std::vector<SpaceCenter::RoboticRotor> robotic_rotors();

    /**
     * The vessels root part.
     */
    SpaceCenter::Part root();

    /**
     * A list of all sensors in the vessel.
     */
    std::vector<SpaceCenter::Sensor> sensors();

    /**
     * A list of all solar panels in the vessel.
     */
    std::vector<SpaceCenter::SolarPanel> solar_panels();

    /**
     * A list of all wheels in the vessel.
     */
    std::vector<SpaceCenter::Wheel> wheels();

    ::krpc::Stream<std::vector<SpaceCenter::Part>> in_decouple_stage_stream(int32_t stage);

    ::krpc::Stream<std::vector<SpaceCenter::Part>> in_stage_stream(int32_t stage);

    ::krpc::Stream<std::vector<SpaceCenter::Module>> modules_with_name_stream(std::string module_name);

    ::krpc::Stream<std::vector<SpaceCenter::Part>> with_module_stream(std::string module_name);

    ::krpc::Stream<std::vector<SpaceCenter::Part>> with_name_stream(std::string name);

    ::krpc::Stream<std::vector<SpaceCenter::Part>> with_tag_stream(std::string tag);

    ::krpc::Stream<std::vector<SpaceCenter::Part>> with_title_stream(std::string title);

    ::krpc::Stream<std::vector<SpaceCenter::Part>> all_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Antenna>> antennas_stream();

    ::krpc::Stream<std::vector<SpaceCenter::CargoBay>> cargo_bays_stream();

    ::krpc::Stream<std::vector<SpaceCenter::ControlSurface>> control_surfaces_stream();

    ::krpc::Stream<SpaceCenter::Part> controlling_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Decoupler>> decouplers_stream();

    ::krpc::Stream<std::vector<SpaceCenter::DockingPort>> docking_ports_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Engine>> engines_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Experiment>> experiments_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Fairing>> fairings_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Intake>> intakes_stream();

    ::krpc::Stream<std::vector<SpaceCenter::LaunchClamp>> launch_clamps_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Leg>> legs_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Light>> lights_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Parachute>> parachutes_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Radiator>> radiators_stream();

    ::krpc::Stream<std::vector<SpaceCenter::RCS>> rcs_stream();

    ::krpc::Stream<std::vector<SpaceCenter::ReactionWheel>> reaction_wheels_stream();

    ::krpc::Stream<std::vector<SpaceCenter::ResourceConverter>> resource_converters_stream();

    ::krpc::Stream<std::vector<SpaceCenter::ResourceDrain>> resource_drains_stream();

    ::krpc::Stream<std::vector<SpaceCenter::ResourceHarvester>> resource_harvesters_stream();

    ::krpc::Stream<std::vector<SpaceCenter::RoboticHinge>> robotic_hinges_stream();

    ::krpc::Stream<std::vector<SpaceCenter::RoboticPiston>> robotic_pistons_stream();

    ::krpc::Stream<std::vector<SpaceCenter::RoboticRotation>> robotic_rotations_stream();

    ::krpc::Stream<std::vector<SpaceCenter::RoboticRotor>> robotic_rotors_stream();

    ::krpc::Stream<SpaceCenter::Part> root_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Sensor>> sensors_stream();

    ::krpc::Stream<std::vector<SpaceCenter::SolarPanel>> solar_panels_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Wheel>> wheels_stream();

    ::krpc::schema::ProcedureCall in_decouple_stage_call(int32_t stage);

    ::krpc::schema::ProcedureCall in_stage_call(int32_t stage);

    ::krpc::schema::ProcedureCall modules_with_name_call(std::string module_name);

    ::krpc::schema::ProcedureCall with_module_call(std::string module_name);

    ::krpc::schema::ProcedureCall with_name_call(std::string name);

    ::krpc::schema::ProcedureCall with_tag_call(std::string tag);

    ::krpc::schema::ProcedureCall with_title_call(std::string title);

    ::krpc::schema::ProcedureCall all_call();

    ::krpc::schema::ProcedureCall antennas_call();

    ::krpc::schema::ProcedureCall cargo_bays_call();

    ::krpc::schema::ProcedureCall control_surfaces_call();

    ::krpc::schema::ProcedureCall controlling_call();

    ::krpc::schema::ProcedureCall set_controlling_call(SpaceCenter::Part value);

    ::krpc::schema::ProcedureCall decouplers_call();

    ::krpc::schema::ProcedureCall docking_ports_call();

    ::krpc::schema::ProcedureCall engines_call();

    ::krpc::schema::ProcedureCall experiments_call();

    ::krpc::schema::ProcedureCall fairings_call();

    ::krpc::schema::ProcedureCall intakes_call();

    ::krpc::schema::ProcedureCall launch_clamps_call();

    ::krpc::schema::ProcedureCall legs_call();

    ::krpc::schema::ProcedureCall lights_call();

    ::krpc::schema::ProcedureCall parachutes_call();

    ::krpc::schema::ProcedureCall radiators_call();

    ::krpc::schema::ProcedureCall rcs_call();

    ::krpc::schema::ProcedureCall reaction_wheels_call();

    ::krpc::schema::ProcedureCall resource_converters_call();

    ::krpc::schema::ProcedureCall resource_drains_call();

    ::krpc::schema::ProcedureCall resource_harvesters_call();

    ::krpc::schema::ProcedureCall robotic_hinges_call();

    ::krpc::schema::ProcedureCall robotic_pistons_call();

    ::krpc::schema::ProcedureCall robotic_rotations_call();

    ::krpc::schema::ProcedureCall robotic_rotors_call();

    ::krpc::schema::ProcedureCall root_call();

    ::krpc::schema::ProcedureCall sensors_call();

    ::krpc::schema::ProcedureCall solar_panels_call();

    ::krpc::schema::ProcedureCall wheels_call();
  };

  /**
   * A propellant for an engine. Obtains by calling SpaceCenter::Engine::propellants.
   */
  class Propellant : public krpc::Object<Propellant> {
   public:
    explicit Propellant(Client* client = nullptr, uint64_t id = 0);

    /**
     * The current amount of propellant.
     */
    double current_amount();

    /**
     * The required amount of propellant.
     */
    double current_requirement();

    /**
     * If this propellant has a stack gauge or not.
     */
    bool draw_stack_gauge();

    /**
     * If this propellant should be ignored when calculating required mass flow
     * given specific impulse.
     */
    bool ignore_for_isp();

    /**
     * If this propellant should be ignored for thrust curve calculations.
     */
    bool ignore_for_thrust_curve();

    /**
     * If this propellant is deprived.
     */
    bool is_deprived();

    /**
     * The name of the propellant.
     */
    std::string name();

    /**
     * The propellant ratio.
     */
    float ratio();

    /**
     * The total amount of the underlying resource currently reachable given
     * resource flow rules.
     */
    double total_resource_available();

    /**
     * The total vehicle capacity for the underlying propellant resource,
     * restricted by resource flow rules.
     */
    double total_resource_capacity();

    ::krpc::Stream<double> current_amount_stream();

    ::krpc::Stream<double> current_requirement_stream();

    ::krpc::Stream<bool> draw_stack_gauge_stream();

    ::krpc::Stream<bool> ignore_for_isp_stream();

    ::krpc::Stream<bool> ignore_for_thrust_curve_stream();

    ::krpc::Stream<bool> is_deprived_stream();

    ::krpc::Stream<std::string> name_stream();

    ::krpc::Stream<float> ratio_stream();

    ::krpc::Stream<double> total_resource_available_stream();

    ::krpc::Stream<double> total_resource_capacity_stream();

    ::krpc::schema::ProcedureCall current_amount_call();

    ::krpc::schema::ProcedureCall current_requirement_call();

    ::krpc::schema::ProcedureCall draw_stack_gauge_call();

    ::krpc::schema::ProcedureCall ignore_for_isp_call();

    ::krpc::schema::ProcedureCall ignore_for_thrust_curve_call();

    ::krpc::schema::ProcedureCall is_deprived_call();

    ::krpc::schema::ProcedureCall name_call();

    ::krpc::schema::ProcedureCall ratio_call();

    ::krpc::schema::ProcedureCall total_resource_available_call();

    ::krpc::schema::ProcedureCall total_resource_capacity_call();
  };

  /**
   * An RCS block or thruster. Obtained by calling SpaceCenter::Part::rcs.
   */
  class RCS : public krpc::Object<RCS> {
   public:
    explicit RCS(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the RCS thrusters are active.
     * An RCS thruster is inactive if the RCS action group is disabled
     * (SpaceCenter::Control::rcs), the RCS thruster itself is not enabled
     * (SpaceCenter::RCS::enabled) or it is covered by a fairing (SpaceCenter::Part::shielded).
     */
    bool active();

    /**
     * The available force, in Newtons, that can be produced by this RCS,
     * in the positive and negative x, y and z axes of the vessel. These axes
     * correspond to the coordinate axes of the SpaceCenter::Vessel::reference_frame.
     * Returns zero if RCS is disabled.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > available_force();

    /**
     * The amount of thrust, in Newtons, that would be produced by the thruster when activated.
     * Returns zero if the thruster does not have any fuel.
     * Takes the thrusters current SpaceCenter::RCS::thrust_limit and atmospheric conditions
     * into account.
     */
    float available_thrust();

    /**
     * The available torque, in Newton meters, that can be produced by this RCS,
     * in the positive and negative pitch, roll and yaw axes of the vessel. These axes
     * correspond to the coordinate axes of the SpaceCenter::Vessel::reference_frame.
     * Returns zero if RCS is disable.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > available_torque();

    /**
     * Whether the RCS thrusters are enabled.
     */
    bool enabled();

    /**
     * Whether the RCS thrusters are enabled.
     */
    void set_enabled(bool value);

    /**
     * Whether the RCS thruster will fire when pitch control input is given.
     */
    bool forward_enabled();

    /**
     * Whether the RCS thruster will fire when pitch control input is given.
     */
    void set_forward_enabled(bool value);

    /**
     * Whether the RCS has fuel available.
     */
    bool has_fuel();

    /**
     * The specific impulse of the RCS at sea level on Kerbin, in seconds.
     */
    float kerbin_sea_level_specific_impulse();

    /**
     * The maximum amount of thrust that can be produced by the RCS thrusters when active,
     * in Newtons.
     * Takes the thrusters current SpaceCenter::RCS::thrust_limit and atmospheric conditions
     * into account.
     */
    float max_thrust();

    /**
     * The maximum amount of thrust that can be produced by the RCS thrusters when active
     * in a vacuum, in Newtons.
     */
    float max_vacuum_thrust();

    /**
     * The part object for this RCS.
     */
    SpaceCenter::Part part();

    /**
     * Whether the RCS thruster will fire when pitch control input is given.
     */
    bool pitch_enabled();

    /**
     * Whether the RCS thruster will fire when pitch control input is given.
     */
    void set_pitch_enabled(bool value);

    /**
     * The ratios of resources that the RCS consumes. A dictionary mapping resource names
     * to the ratios at which they are consumed by the RCS.
     */
    std::map<std::string, float> propellant_ratios();

    /**
     * The names of resources that the RCS consumes.
     */
    std::vector<std::string> propellants();

    /**
     * Whether the RCS thruster will fire when roll control input is given.
     */
    bool right_enabled();

    /**
     * Whether the RCS thruster will fire when roll control input is given.
     */
    void set_right_enabled(bool value);

    /**
     * Whether the RCS thruster will fire when roll control input is given.
     */
    bool roll_enabled();

    /**
     * Whether the RCS thruster will fire when roll control input is given.
     */
    void set_roll_enabled(bool value);

    /**
     * The current specific impulse of the RCS, in seconds. Returns zero
     * if the RCS is not active.
     */
    float specific_impulse();

    /**
     * The thrust limiter of the thruster. A value between 0 and 1.
     */
    float thrust_limit();

    /**
     * The thrust limiter of the thruster. A value between 0 and 1.
     */
    void set_thrust_limit(float value);

    /**
     * A list of thrusters, one of each nozzel in the RCS part.
     */
    std::vector<SpaceCenter::Thruster> thrusters();

    /**
     * Whether the RCS thruster will fire when yaw control input is given.
     */
    bool up_enabled();

    /**
     * Whether the RCS thruster will fire when yaw control input is given.
     */
    void set_up_enabled(bool value);

    /**
     * The vacuum specific impulse of the RCS, in seconds.
     */
    float vacuum_specific_impulse();

    /**
     * Whether the RCS thruster will fire when yaw control input is given.
     */
    bool yaw_enabled();

    /**
     * Whether the RCS thruster will fire when yaw control input is given.
     */
    void set_yaw_enabled(bool value);

    ::krpc::Stream<bool> active_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> available_force_stream();

    ::krpc::Stream<float> available_thrust_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> available_torque_stream();

    ::krpc::Stream<bool> enabled_stream();

    ::krpc::Stream<bool> forward_enabled_stream();

    ::krpc::Stream<bool> has_fuel_stream();

    ::krpc::Stream<float> kerbin_sea_level_specific_impulse_stream();

    ::krpc::Stream<float> max_thrust_stream();

    ::krpc::Stream<float> max_vacuum_thrust_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<bool> pitch_enabled_stream();

    ::krpc::Stream<std::map<std::string, float>> propellant_ratios_stream();

    ::krpc::Stream<std::vector<std::string>> propellants_stream();

    ::krpc::Stream<bool> right_enabled_stream();

    ::krpc::Stream<bool> roll_enabled_stream();

    ::krpc::Stream<float> specific_impulse_stream();

    ::krpc::Stream<float> thrust_limit_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Thruster>> thrusters_stream();

    ::krpc::Stream<bool> up_enabled_stream();

    ::krpc::Stream<float> vacuum_specific_impulse_stream();

    ::krpc::Stream<bool> yaw_enabled_stream();

    ::krpc::schema::ProcedureCall active_call();

    ::krpc::schema::ProcedureCall available_force_call();

    ::krpc::schema::ProcedureCall available_thrust_call();

    ::krpc::schema::ProcedureCall available_torque_call();

    ::krpc::schema::ProcedureCall enabled_call();

    ::krpc::schema::ProcedureCall set_enabled_call(bool value);

    ::krpc::schema::ProcedureCall forward_enabled_call();

    ::krpc::schema::ProcedureCall set_forward_enabled_call(bool value);

    ::krpc::schema::ProcedureCall has_fuel_call();

    ::krpc::schema::ProcedureCall kerbin_sea_level_specific_impulse_call();

    ::krpc::schema::ProcedureCall max_thrust_call();

    ::krpc::schema::ProcedureCall max_vacuum_thrust_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall pitch_enabled_call();

    ::krpc::schema::ProcedureCall set_pitch_enabled_call(bool value);

    ::krpc::schema::ProcedureCall propellant_ratios_call();

    ::krpc::schema::ProcedureCall propellants_call();

    ::krpc::schema::ProcedureCall right_enabled_call();

    ::krpc::schema::ProcedureCall set_right_enabled_call(bool value);

    ::krpc::schema::ProcedureCall roll_enabled_call();

    ::krpc::schema::ProcedureCall set_roll_enabled_call(bool value);

    ::krpc::schema::ProcedureCall specific_impulse_call();

    ::krpc::schema::ProcedureCall thrust_limit_call();

    ::krpc::schema::ProcedureCall set_thrust_limit_call(float value);

    ::krpc::schema::ProcedureCall thrusters_call();

    ::krpc::schema::ProcedureCall up_enabled_call();

    ::krpc::schema::ProcedureCall set_up_enabled_call(bool value);

    ::krpc::schema::ProcedureCall vacuum_specific_impulse_call();

    ::krpc::schema::ProcedureCall yaw_enabled_call();

    ::krpc::schema::ProcedureCall set_yaw_enabled_call(bool value);
  };

  /**
   * A radiator. Obtained by calling SpaceCenter::Part::radiator.
   */
  class Radiator : public krpc::Object<Radiator> {
   public:
    explicit Radiator(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the radiator is deployable.
     */
    bool deployable();

    /**
     * For a deployable radiator, true if the radiator is extended.
     * If the radiator is not deployable, this is always true.
     */
    bool deployed();

    /**
     * For a deployable radiator, true if the radiator is extended.
     * If the radiator is not deployable, this is always true.
     */
    void set_deployed(bool value);

    /**
     * The part object for this radiator.
     */
    SpaceCenter::Part part();

    /**
     * The current state of the radiator.
     *
     * A fixed radiator is always SpaceCenter::RadiatorState::extended.
     */
    SpaceCenter::RadiatorState state();

    ::krpc::Stream<bool> deployable_stream();

    ::krpc::Stream<bool> deployed_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<SpaceCenter::RadiatorState> state_stream();

    ::krpc::schema::ProcedureCall deployable_call();

    ::krpc::schema::ProcedureCall deployed_call();

    ::krpc::schema::ProcedureCall set_deployed_call(bool value);

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall state_call();
  };

  /**
   * A reaction wheel. Obtained by calling SpaceCenter::Part::reaction_wheel.
   */
  class ReactionWheel : public krpc::Object<ReactionWheel> {
   public:
    explicit ReactionWheel(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the reaction wheel is active.
     */
    bool active();

    /**
     * Whether the reaction wheel is active.
     */
    void set_active(bool value);

    /**
     * The available torque, in Newton meters, that can be produced by this reaction wheel,
     * in the positive and negative pitch, roll and yaw axes of the vessel. These axes
     * correspond to the coordinate axes of the SpaceCenter::Vessel::reference_frame.
     * Returns zero if the reaction wheel is inactive or broken.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > available_torque();

    /**
     * Whether the reaction wheel is broken.
     */
    bool broken();

    /**
     * The maximum torque, in Newton meters, that can be produced by this reaction wheel,
     * when it is active, in the positive and negative pitch, roll and yaw axes of the vessel.
     * These axes correspond to the coordinate axes of the SpaceCenter::Vessel::reference_frame.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > max_torque();

    /**
     * The part object for this reaction wheel.
     */
    SpaceCenter::Part part();

    ::krpc::Stream<bool> active_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> available_torque_stream();

    ::krpc::Stream<bool> broken_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> max_torque_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::schema::ProcedureCall active_call();

    ::krpc::schema::ProcedureCall set_active_call(bool value);

    ::krpc::schema::ProcedureCall available_torque_call();

    ::krpc::schema::ProcedureCall broken_call();

    ::krpc::schema::ProcedureCall max_torque_call();

    ::krpc::schema::ProcedureCall part_call();
  };

  /**
   * Represents a reference frame for positions, rotations and
   * velocities. Contains:
   *
   * - The position of the origin.
   * - The directions of the x, y and z axes.
   * - The linear velocity of the frame.
   * - The angular velocity of the frame.
   *
   * This class does not contain any properties or methods. It is only
   * used as a parameter to other functions.
   */
  class ReferenceFrame : public krpc::Object<ReferenceFrame> {
   public:
    explicit ReferenceFrame(Client* client = nullptr, uint64_t id = 0);

    /**
     * Create a hybrid reference frame. This is a custom reference frame
     * whose components inherited from other reference frames.
     * @param position The reference frame providing the position of the origin.
     * @param rotation The reference frame providing the rotation of the frame.
     * @param velocity The reference frame providing the linear velocity of the frame.
     * @param angularVelocity The reference frame providing the angular velocity
     * of the frame.
     *
     * The position reference frame is required but all other
     * reference frames are optional. If omitted, they are set to the
     * position reference frame.
     */
    static SpaceCenter::ReferenceFrame create_hybrid(Client& client, SpaceCenter::ReferenceFrame position, SpaceCenter::ReferenceFrame rotation, SpaceCenter::ReferenceFrame velocity, SpaceCenter::ReferenceFrame angular_velocity);

    /**
     * Create a relative reference frame. This is a custom reference frame
     * whose components offset the components of a parent reference frame.
     * @param referenceFrame The parent reference frame on which to
     * base this reference frame.
     * @param position The offset of the position of the origin,
     * as a position vector. Defaults to (0, 0, 0)
     * @param rotation The rotation to apply to the parent frames rotation,
     * as a quaternion of the form (x, y, z, w).
     * Defaults to (0, 0, 0, 1) (i.e. no rotation)
     * @param velocity The linear velocity to offset the parent frame by,
     * as a vector pointing in the direction of travel, whose magnitude is the speed in
     * meters per second. Defaults to (0, 0, 0).
     * @param angularVelocity The angular velocity to offset the parent frame by,
     * as a vector. This vector points in the direction of the axis of rotation,
     * and its magnitude is the speed of the rotation in radians per second.
     * Defaults to (0, 0, 0).
     */
    static SpaceCenter::ReferenceFrame create_relative(Client& client, SpaceCenter::ReferenceFrame reference_frame, std::tuple<double, double, double> position, std::tuple<double, double, double, double> rotation, std::tuple<double, double, double> velocity, std::tuple<double, double, double> angular_velocity);

    static ::krpc::Stream<SpaceCenter::ReferenceFrame> create_hybrid_stream(Client& client, SpaceCenter::ReferenceFrame position, SpaceCenter::ReferenceFrame rotation, SpaceCenter::ReferenceFrame velocity, SpaceCenter::ReferenceFrame angular_velocity);

    static ::krpc::Stream<SpaceCenter::ReferenceFrame> create_relative_stream(Client& client, SpaceCenter::ReferenceFrame reference_frame, std::tuple<double, double, double> position, std::tuple<double, double, double, double> rotation, std::tuple<double, double, double> velocity, std::tuple<double, double, double> angular_velocity);

    static ::krpc::schema::ProcedureCall create_hybrid_call(Client& client, SpaceCenter::ReferenceFrame position, SpaceCenter::ReferenceFrame rotation, SpaceCenter::ReferenceFrame velocity, SpaceCenter::ReferenceFrame angular_velocity);

    static ::krpc::schema::ProcedureCall create_relative_call(Client& client, SpaceCenter::ReferenceFrame reference_frame, std::tuple<double, double, double> position, std::tuple<double, double, double, double> rotation, std::tuple<double, double, double> velocity, std::tuple<double, double, double> angular_velocity);
  };

  /**
   * An individual resource stored within a part.
   * Created using methods in the SpaceCenter::Resources class.
   */
  class Resource : public krpc::Object<Resource> {
   public:
    explicit Resource(Client* client = nullptr, uint64_t id = 0);

    /**
     * The amount of the resource that is currently stored in the part.
     */
    float amount();

    /**
     * The density of the resource, in kg/l.
     */
    float density();

    /**
     * Whether use of this resource is enabled.
     */
    bool enabled();

    /**
     * Whether use of this resource is enabled.
     */
    void set_enabled(bool value);

    /**
     * The flow mode of the resource.
     */
    SpaceCenter::ResourceFlowMode flow_mode();

    /**
     * The total amount of the resource that can be stored in the part.
     */
    float max();

    /**
     * The name of the resource.
     */
    std::string name();

    /**
     * The part containing the resource.
     */
    SpaceCenter::Part part();

    ::krpc::Stream<float> amount_stream();

    ::krpc::Stream<float> density_stream();

    ::krpc::Stream<bool> enabled_stream();

    ::krpc::Stream<SpaceCenter::ResourceFlowMode> flow_mode_stream();

    ::krpc::Stream<float> max_stream();

    ::krpc::Stream<std::string> name_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::schema::ProcedureCall amount_call();

    ::krpc::schema::ProcedureCall density_call();

    ::krpc::schema::ProcedureCall enabled_call();

    ::krpc::schema::ProcedureCall set_enabled_call(bool value);

    ::krpc::schema::ProcedureCall flow_mode_call();

    ::krpc::schema::ProcedureCall max_call();

    ::krpc::schema::ProcedureCall name_call();

    ::krpc::schema::ProcedureCall part_call();
  };

  /**
   * A resource converter. Obtained by calling SpaceCenter::Part::resource_converter.
   */
  class ResourceConverter : public krpc::Object<ResourceConverter> {
   public:
    explicit ResourceConverter(Client* client = nullptr, uint64_t id = 0);

    /**
     * True if the specified converter is active.
     * @param index Index of the converter.
     */
    bool active(int32_t index);

    /**
     * List of the names of resources consumed by the specified converter.
     * @param index Index of the converter.
     */
    std::vector<std::string> inputs(int32_t index);

    /**
     * The name of the specified converter.
     * @param index Index of the converter.
     */
    std::string name(int32_t index);

    /**
     * List of the names of resources produced by the specified converter.
     * @param index Index of the converter.
     */
    std::vector<std::string> outputs(int32_t index);

    /**
     * Start the specified converter.
     * @param index Index of the converter.
     */
    void start(int32_t index);

    /**
     * The state of the specified converter.
     * @param index Index of the converter.
     */
    SpaceCenter::ResourceConverterState state(int32_t index);

    /**
     * Status information for the specified converter.
     * This is the full status message shown in the in-game UI.
     * @param index Index of the converter.
     */
    std::string status_info(int32_t index);

    /**
     * Stop the specified converter.
     * @param index Index of the converter.
     */
    void stop(int32_t index);

    /**
     * The core temperature of the converter, in Kelvin.
     */
    float core_temperature();

    /**
     * The number of converters in the part.
     */
    int32_t count();

    /**
     * The core temperature at which the converter will operate with peak efficiency, in Kelvin.
     */
    float optimum_core_temperature();

    /**
     * The part object for this converter.
     */
    SpaceCenter::Part part();

    /**
     * The thermal efficiency of the converter, as a percentage of its maximum.
     */
    float thermal_efficiency();

    ::krpc::Stream<bool> active_stream(int32_t index);

    ::krpc::Stream<std::vector<std::string>> inputs_stream(int32_t index);

    ::krpc::Stream<std::string> name_stream(int32_t index);

    ::krpc::Stream<std::vector<std::string>> outputs_stream(int32_t index);

    ::krpc::Stream<SpaceCenter::ResourceConverterState> state_stream(int32_t index);

    ::krpc::Stream<std::string> status_info_stream(int32_t index);

    ::krpc::Stream<float> core_temperature_stream();

    ::krpc::Stream<int32_t> count_stream();

    ::krpc::Stream<float> optimum_core_temperature_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<float> thermal_efficiency_stream();

    ::krpc::schema::ProcedureCall active_call(int32_t index);

    ::krpc::schema::ProcedureCall inputs_call(int32_t index);

    ::krpc::schema::ProcedureCall name_call(int32_t index);

    ::krpc::schema::ProcedureCall outputs_call(int32_t index);

    ::krpc::schema::ProcedureCall start_call(int32_t index);

    ::krpc::schema::ProcedureCall state_call(int32_t index);

    ::krpc::schema::ProcedureCall status_info_call(int32_t index);

    ::krpc::schema::ProcedureCall stop_call(int32_t index);

    ::krpc::schema::ProcedureCall core_temperature_call();

    ::krpc::schema::ProcedureCall count_call();

    ::krpc::schema::ProcedureCall optimum_core_temperature_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall thermal_efficiency_call();
  };

  /**
   * A resource drain. Obtained by calling SpaceCenter::Part::resource_drain.
   */
  class ResourceDrain : public krpc::Object<ResourceDrain> {
   public:
    explicit ResourceDrain(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the provided resource is enabled for draining.
     */
    bool check_resource(SpaceCenter::Resource resource);

    /**
     * Whether the given resource should be drained.
     */
    void set_resource(SpaceCenter::Resource resource, bool enabled);

    /**
     * Activates resource draining for all enabled parts.
     */
    void start();

    /**
     * Turns off resource draining.
     */
    void stop();

    /**
     * List of available resources.
     */
    std::vector<SpaceCenter::Resource> available_resources();

    /**
     * The drain mode.
     */
    SpaceCenter::DrainMode drain_mode();

    /**
     * The drain mode.
     */
    void set_drain_mode(SpaceCenter::DrainMode value);

    /**
     * Maximum possible drain rate.
     */
    float max_rate();

    /**
     * Minimum possible drain rate
     */
    float min_rate();

    /**
     * The part object for this resource drain.
     */
    SpaceCenter::Part part();

    /**
     * Current drain rate.
     */
    float rate();

    /**
     * Current drain rate.
     */
    void set_rate(float value);

    ::krpc::Stream<bool> check_resource_stream(SpaceCenter::Resource resource);

    ::krpc::Stream<std::vector<SpaceCenter::Resource>> available_resources_stream();

    ::krpc::Stream<SpaceCenter::DrainMode> drain_mode_stream();

    ::krpc::Stream<float> max_rate_stream();

    ::krpc::Stream<float> min_rate_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<float> rate_stream();

    ::krpc::schema::ProcedureCall check_resource_call(SpaceCenter::Resource resource);

    ::krpc::schema::ProcedureCall set_resource_call(SpaceCenter::Resource resource, bool enabled);

    ::krpc::schema::ProcedureCall start_call();

    ::krpc::schema::ProcedureCall stop_call();

    ::krpc::schema::ProcedureCall available_resources_call();

    ::krpc::schema::ProcedureCall drain_mode_call();

    ::krpc::schema::ProcedureCall set_drain_mode_call(SpaceCenter::DrainMode value);

    ::krpc::schema::ProcedureCall max_rate_call();

    ::krpc::schema::ProcedureCall min_rate_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall rate_call();

    ::krpc::schema::ProcedureCall set_rate_call(float value);
  };

  /**
   * A resource harvester (drill). Obtained by calling SpaceCenter::Part::resource_harvester.
   */
  class ResourceHarvester : public krpc::Object<ResourceHarvester> {
   public:
    explicit ResourceHarvester(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the harvester is actively drilling.
     */
    bool active();

    /**
     * Whether the harvester is actively drilling.
     */
    void set_active(bool value);

    /**
     * The core temperature of the drill, in Kelvin.
     */
    float core_temperature();

    /**
     * Whether the harvester is deployed.
     */
    bool deployed();

    /**
     * Whether the harvester is deployed.
     */
    void set_deployed(bool value);

    /**
     * The rate at which the drill is extracting ore, in units per second.
     */
    float extraction_rate();

    /**
     * The core temperature at which the drill will operate with peak efficiency, in Kelvin.
     */
    float optimum_core_temperature();

    /**
     * The part object for this harvester.
     */
    SpaceCenter::Part part();

    /**
     * The state of the harvester.
     */
    SpaceCenter::ResourceHarvesterState state();

    /**
     * The thermal efficiency of the drill, as a percentage of its maximum.
     */
    float thermal_efficiency();

    ::krpc::Stream<bool> active_stream();

    ::krpc::Stream<float> core_temperature_stream();

    ::krpc::Stream<bool> deployed_stream();

    ::krpc::Stream<float> extraction_rate_stream();

    ::krpc::Stream<float> optimum_core_temperature_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<SpaceCenter::ResourceHarvesterState> state_stream();

    ::krpc::Stream<float> thermal_efficiency_stream();

    ::krpc::schema::ProcedureCall active_call();

    ::krpc::schema::ProcedureCall set_active_call(bool value);

    ::krpc::schema::ProcedureCall core_temperature_call();

    ::krpc::schema::ProcedureCall deployed_call();

    ::krpc::schema::ProcedureCall set_deployed_call(bool value);

    ::krpc::schema::ProcedureCall extraction_rate_call();

    ::krpc::schema::ProcedureCall optimum_core_temperature_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall state_call();

    ::krpc::schema::ProcedureCall thermal_efficiency_call();
  };

  /**
   * Transfer resources between parts.
   */
  class ResourceTransfer : public krpc::Object<ResourceTransfer> {
   public:
    explicit ResourceTransfer(Client* client = nullptr, uint64_t id = 0);

    /**
     * Start transferring a resource transfer between a pair of parts. The transfer will move
     * at most maxAmount units of the resource, depending on how much of
     * the resource is available in the source part and how much storage is available in the
     * destination part.
     * Use SpaceCenter::ResourceTransfer::complete to check if the transfer is complete.
     * Use SpaceCenter::ResourceTransfer::amount to see how much of the resource has been transferred.
     * @param fromPart The part to transfer to.
     * @param toPart The part to transfer from.
     * @param resource The name of the resource to transfer.
     * @param maxAmount The maximum amount of resource to transfer.
     */
    static SpaceCenter::ResourceTransfer start(Client& client, SpaceCenter::Part from_part, SpaceCenter::Part to_part, std::string resource, float max_amount);

    /**
     * The amount of the resource that has been transferred.
     */
    float amount();

    /**
     * Whether the transfer has completed.
     */
    bool complete();

    static ::krpc::Stream<SpaceCenter::ResourceTransfer> start_stream(Client& client, SpaceCenter::Part from_part, SpaceCenter::Part to_part, std::string resource, float max_amount);

    ::krpc::Stream<float> amount_stream();

    ::krpc::Stream<bool> complete_stream();

    static ::krpc::schema::ProcedureCall start_call(Client& client, SpaceCenter::Part from_part, SpaceCenter::Part to_part, std::string resource, float max_amount);

    ::krpc::schema::ProcedureCall amount_call();

    ::krpc::schema::ProcedureCall complete_call();
  };

  /**
   * Represents the collection of resources stored in a vessel, stage or part.
   * Created by calling SpaceCenter::Vessel::resources,
   * SpaceCenter::Vessel::resources_in_decouple_stage or
   * SpaceCenter::Part::resources.
   */
  class Resources : public krpc::Object<Resources> {
   public:
    explicit Resources(Client* client = nullptr, uint64_t id = 0);

    /**
     * Returns the amount of a resource that is currently stored.
     * @param name The name of the resource.
     */
    float amount(std::string name);

    /**
     * Check whether the named resource can be stored.
     * @param name The name of the resource.
     */
    bool has_resource(std::string name);

    /**
     * Returns the amount of a resource that can be stored.
     * @param name The name of the resource.
     */
    float max(std::string name);

    /**
     * All the individual resources with the given name that can be stored.
     */
    std::vector<SpaceCenter::Resource> with_resource(std::string name);

    /**
     * Returns the density of a resource, in kg/l.
     * @param name The name of the resource.
     */
    static float density(Client& client, std::string name);

    /**
     * Returns the flow mode of a resource.
     * @param name The name of the resource.
     */
    static SpaceCenter::ResourceFlowMode flow_mode(Client& client, std::string name);

    /**
     * All the individual resources that can be stored.
     */
    std::vector<SpaceCenter::Resource> all();

    /**
     * Whether use of all the resources are enabled.
     *
     * This is true if all of the resources are enabled.
     * If any of the resources are not enabled, this is false.
     */
    bool enabled();

    /**
     * Whether use of all the resources are enabled.
     *
     * This is true if all of the resources are enabled.
     * If any of the resources are not enabled, this is false.
     */
    void set_enabled(bool value);

    /**
     * A list of resource names that can be stored.
     */
    std::vector<std::string> names();

    ::krpc::Stream<float> amount_stream(std::string name);

    ::krpc::Stream<bool> has_resource_stream(std::string name);

    ::krpc::Stream<float> max_stream(std::string name);

    ::krpc::Stream<std::vector<SpaceCenter::Resource>> with_resource_stream(std::string name);

    static ::krpc::Stream<float> density_stream(Client& client, std::string name);

    static ::krpc::Stream<SpaceCenter::ResourceFlowMode> flow_mode_stream(Client& client, std::string name);

    ::krpc::Stream<std::vector<SpaceCenter::Resource>> all_stream();

    ::krpc::Stream<bool> enabled_stream();

    ::krpc::Stream<std::vector<std::string>> names_stream();

    ::krpc::schema::ProcedureCall amount_call(std::string name);

    ::krpc::schema::ProcedureCall has_resource_call(std::string name);

    ::krpc::schema::ProcedureCall max_call(std::string name);

    ::krpc::schema::ProcedureCall with_resource_call(std::string name);

    static ::krpc::schema::ProcedureCall density_call(Client& client, std::string name);

    static ::krpc::schema::ProcedureCall flow_mode_call(Client& client, std::string name);

    ::krpc::schema::ProcedureCall all_call();

    ::krpc::schema::ProcedureCall enabled_call();

    ::krpc::schema::ProcedureCall set_enabled_call(bool value);

    ::krpc::schema::ProcedureCall names_call();
  };

  /**
   * A robotic controller. Obtained by calling SpaceCenter::Part::robotic_controller.
   */
  class RoboticController : public krpc::Object<RoboticController> {
   public:
    explicit RoboticController(Client* client = nullptr, uint64_t id = 0);

    /**
     * Add an axis to the controller.
     * @return Returns true if the axis is added successfully.
     */
    bool add_axis(SpaceCenter::Module module, std::string field_name);

    /**
     * Add key frame value for controller axis.
     * @return Returns true if the key frame is added successfully.
     */
    bool add_key_frame(SpaceCenter::Module module, std::string field_name, float time, float value);

    /**
     * The axes for the controller.
     */
    std::vector<std::vector<std::string> > axes();

    /**
     * Clear axis.
     * @return Returns true if the axis is cleared successfully.
     */
    bool clear_axis(SpaceCenter::Module module, std::string field_name);

    /**
     * Whether the controller has a part.
     */
    bool has_part(SpaceCenter::Part part);

    /**
     * The part object for this controller.
     */
    SpaceCenter::Part part();

    ::krpc::Stream<bool> add_axis_stream(SpaceCenter::Module module, std::string field_name);

    ::krpc::Stream<bool> add_key_frame_stream(SpaceCenter::Module module, std::string field_name, float time, float value);

    ::krpc::Stream<std::vector<std::vector<std::string> >> axes_stream();

    ::krpc::Stream<bool> clear_axis_stream(SpaceCenter::Module module, std::string field_name);

    ::krpc::Stream<bool> has_part_stream(SpaceCenter::Part part);

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::schema::ProcedureCall add_axis_call(SpaceCenter::Module module, std::string field_name);

    ::krpc::schema::ProcedureCall add_key_frame_call(SpaceCenter::Module module, std::string field_name, float time, float value);

    ::krpc::schema::ProcedureCall axes_call();

    ::krpc::schema::ProcedureCall clear_axis_call(SpaceCenter::Module module, std::string field_name);

    ::krpc::schema::ProcedureCall has_part_call(SpaceCenter::Part part);

    ::krpc::schema::ProcedureCall part_call();
  };

  /**
   * A robotic hinge. Obtained by calling SpaceCenter::Part::robotic_hinge.
   */
  class RoboticHinge : public krpc::Object<RoboticHinge> {
   public:
    explicit RoboticHinge(Client* client = nullptr, uint64_t id = 0);

    /**
     * Move hinge to it's built position.
     */
    void move_home();

    /**
     * Current angle.
     */
    float current_angle();

    /**
     * Damping percentage.
     */
    float damping();

    /**
     * Damping percentage.
     */
    void set_damping(float value);

    /**
     * Lock movement.
     */
    bool locked();

    /**
     * Lock movement.
     */
    void set_locked(bool value);

    /**
     * Whether the motor is engaged.
     */
    bool motor_engaged();

    /**
     * Whether the motor is engaged.
     */
    void set_motor_engaged(bool value);

    /**
     * The part object for this robotic hinge.
     */
    SpaceCenter::Part part();

    /**
     * Target movement rate in degrees per second.
     */
    float rate();

    /**
     * Target movement rate in degrees per second.
     */
    void set_rate(float value);

    /**
     * Target angle.
     */
    float target_angle();

    /**
     * Target angle.
     */
    void set_target_angle(float value);

    ::krpc::Stream<float> current_angle_stream();

    ::krpc::Stream<float> damping_stream();

    ::krpc::Stream<bool> locked_stream();

    ::krpc::Stream<bool> motor_engaged_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<float> rate_stream();

    ::krpc::Stream<float> target_angle_stream();

    ::krpc::schema::ProcedureCall move_home_call();

    ::krpc::schema::ProcedureCall current_angle_call();

    ::krpc::schema::ProcedureCall damping_call();

    ::krpc::schema::ProcedureCall set_damping_call(float value);

    ::krpc::schema::ProcedureCall locked_call();

    ::krpc::schema::ProcedureCall set_locked_call(bool value);

    ::krpc::schema::ProcedureCall motor_engaged_call();

    ::krpc::schema::ProcedureCall set_motor_engaged_call(bool value);

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall rate_call();

    ::krpc::schema::ProcedureCall set_rate_call(float value);

    ::krpc::schema::ProcedureCall target_angle_call();

    ::krpc::schema::ProcedureCall set_target_angle_call(float value);
  };

  /**
   * A robotic piston part. Obtained by calling SpaceCenter::Part::robotic_piston.
   */
  class RoboticPiston : public krpc::Object<RoboticPiston> {
   public:
    explicit RoboticPiston(Client* client = nullptr, uint64_t id = 0);

    /**
     * Move piston to it's built position.
     */
    void move_home();

    /**
     * Current extension of the piston.
     */
    float current_extension();

    /**
     * Damping percentage.
     */
    float damping();

    /**
     * Damping percentage.
     */
    void set_damping(float value);

    /**
     * Lock movement.
     */
    bool locked();

    /**
     * Lock movement.
     */
    void set_locked(bool value);

    /**
     * Whether the motor is engaged.
     */
    bool motor_engaged();

    /**
     * Whether the motor is engaged.
     */
    void set_motor_engaged(bool value);

    /**
     * The part object for this robotic piston.
     */
    SpaceCenter::Part part();

    /**
     * Target movement rate in degrees per second.
     */
    float rate();

    /**
     * Target movement rate in degrees per second.
     */
    void set_rate(float value);

    /**
     * Target extension of the piston.
     */
    float target_extension();

    /**
     * Target extension of the piston.
     */
    void set_target_extension(float value);

    ::krpc::Stream<float> current_extension_stream();

    ::krpc::Stream<float> damping_stream();

    ::krpc::Stream<bool> locked_stream();

    ::krpc::Stream<bool> motor_engaged_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<float> rate_stream();

    ::krpc::Stream<float> target_extension_stream();

    ::krpc::schema::ProcedureCall move_home_call();

    ::krpc::schema::ProcedureCall current_extension_call();

    ::krpc::schema::ProcedureCall damping_call();

    ::krpc::schema::ProcedureCall set_damping_call(float value);

    ::krpc::schema::ProcedureCall locked_call();

    ::krpc::schema::ProcedureCall set_locked_call(bool value);

    ::krpc::schema::ProcedureCall motor_engaged_call();

    ::krpc::schema::ProcedureCall set_motor_engaged_call(bool value);

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall rate_call();

    ::krpc::schema::ProcedureCall set_rate_call(float value);

    ::krpc::schema::ProcedureCall target_extension_call();

    ::krpc::schema::ProcedureCall set_target_extension_call(float value);
  };

  /**
   * A robotic rotation servo. Obtained by calling SpaceCenter::Part::robotic_rotation.
   */
  class RoboticRotation : public krpc::Object<RoboticRotation> {
   public:
    explicit RoboticRotation(Client* client = nullptr, uint64_t id = 0);

    /**
     * Move rotation servo to it's built position.
     */
    void move_home();

    /**
     * Current angle.
     */
    float current_angle();

    /**
     * Damping percentage.
     */
    float damping();

    /**
     * Damping percentage.
     */
    void set_damping(float value);

    /**
     * Lock Movement
     */
    bool locked();

    /**
     * Lock Movement
     */
    void set_locked(bool value);

    /**
     * Whether the motor is engaged.
     */
    bool motor_engaged();

    /**
     * Whether the motor is engaged.
     */
    void set_motor_engaged(bool value);

    /**
     * The part object for this robotic rotation servo.
     */
    SpaceCenter::Part part();

    /**
     * Target movement rate in degrees per second.
     */
    float rate();

    /**
     * Target movement rate in degrees per second.
     */
    void set_rate(float value);

    /**
     * Target angle.
     */
    float target_angle();

    /**
     * Target angle.
     */
    void set_target_angle(float value);

    ::krpc::Stream<float> current_angle_stream();

    ::krpc::Stream<float> damping_stream();

    ::krpc::Stream<bool> locked_stream();

    ::krpc::Stream<bool> motor_engaged_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<float> rate_stream();

    ::krpc::Stream<float> target_angle_stream();

    ::krpc::schema::ProcedureCall move_home_call();

    ::krpc::schema::ProcedureCall current_angle_call();

    ::krpc::schema::ProcedureCall damping_call();

    ::krpc::schema::ProcedureCall set_damping_call(float value);

    ::krpc::schema::ProcedureCall locked_call();

    ::krpc::schema::ProcedureCall set_locked_call(bool value);

    ::krpc::schema::ProcedureCall motor_engaged_call();

    ::krpc::schema::ProcedureCall set_motor_engaged_call(bool value);

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall rate_call();

    ::krpc::schema::ProcedureCall set_rate_call(float value);

    ::krpc::schema::ProcedureCall target_angle_call();

    ::krpc::schema::ProcedureCall set_target_angle_call(float value);
  };

  /**
   * A robotic rotor. Obtained by calling SpaceCenter::Part::robotic_rotor.
   */
  class RoboticRotor : public krpc::Object<RoboticRotor> {
   public:
    explicit RoboticRotor(Client* client = nullptr, uint64_t id = 0);

    /**
     * Current RPM.
     */
    float current_rpm();

    /**
     * Whether the rotor direction is inverted.
     */
    bool inverted();

    /**
     * Whether the rotor direction is inverted.
     */
    void set_inverted(bool value);

    /**
     * Lock movement.
     */
    bool locked();

    /**
     * Lock movement.
     */
    void set_locked(bool value);

    /**
     * Whether the motor is engaged.
     */
    bool motor_engaged();

    /**
     * Whether the motor is engaged.
     */
    void set_motor_engaged(bool value);

    /**
     * The part object for this robotic rotor.
     */
    SpaceCenter::Part part();

    /**
     * Target RPM.
     */
    float target_rpm();

    /**
     * Target RPM.
     */
    void set_target_rpm(float value);

    /**
     * Torque limit percentage.
     */
    float torque_limit();

    /**
     * Torque limit percentage.
     */
    void set_torque_limit(float value);

    ::krpc::Stream<float> current_rpm_stream();

    ::krpc::Stream<bool> inverted_stream();

    ::krpc::Stream<bool> locked_stream();

    ::krpc::Stream<bool> motor_engaged_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<float> target_rpm_stream();

    ::krpc::Stream<float> torque_limit_stream();

    ::krpc::schema::ProcedureCall current_rpm_call();

    ::krpc::schema::ProcedureCall inverted_call();

    ::krpc::schema::ProcedureCall set_inverted_call(bool value);

    ::krpc::schema::ProcedureCall locked_call();

    ::krpc::schema::ProcedureCall set_locked_call(bool value);

    ::krpc::schema::ProcedureCall motor_engaged_call();

    ::krpc::schema::ProcedureCall set_motor_engaged_call(bool value);

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall target_rpm_call();

    ::krpc::schema::ProcedureCall set_target_rpm_call(float value);

    ::krpc::schema::ProcedureCall torque_limit_call();

    ::krpc::schema::ProcedureCall set_torque_limit_call(float value);
  };

  /**
   * Obtained by calling SpaceCenter::Experiment::data.
   */
  class ScienceData : public krpc::Object<ScienceData> {
   public:
    explicit ScienceData(Client* client = nullptr, uint64_t id = 0);

    /**
     * Data amount.
     */
    float data_amount();

    /**
     * Science value.
     */
    float science_value();

    /**
     * Transmit value.
     */
    float transmit_value();

    ::krpc::Stream<float> data_amount_stream();

    ::krpc::Stream<float> science_value_stream();

    ::krpc::Stream<float> transmit_value_stream();

    ::krpc::schema::ProcedureCall data_amount_call();

    ::krpc::schema::ProcedureCall science_value_call();

    ::krpc::schema::ProcedureCall transmit_value_call();
  };

  /**
   * Obtained by calling SpaceCenter::Experiment::science_subject.
   */
  class ScienceSubject : public krpc::Object<ScienceSubject> {
   public:
    explicit ScienceSubject(Client* client = nullptr, uint64_t id = 0);

    /**
     * Multiply science value by this to determine data amount in mits.
     */
    float data_scale();

    /**
     * Whether the experiment has been completed.
     */
    bool is_complete();

    /**
     * Amount of science already earned from this subject, not updated until after
     * transmission/recovery.
     */
    float science();

    /**
     * Total science allowable for this subject.
     */
    float science_cap();

    /**
     * Diminishing value multiplier for decreasing the science value returned from repeated
     * experiments.
     */
    float scientific_value();

    /**
     * Multiplier for specific Celestial Body/Experiment Situation combination.
     */
    float subject_value();

    /**
     * Title of science subject, displayed in science archives
     */
    std::string title();

    ::krpc::Stream<float> data_scale_stream();

    ::krpc::Stream<bool> is_complete_stream();

    ::krpc::Stream<float> science_stream();

    ::krpc::Stream<float> science_cap_stream();

    ::krpc::Stream<float> scientific_value_stream();

    ::krpc::Stream<float> subject_value_stream();

    ::krpc::Stream<std::string> title_stream();

    ::krpc::schema::ProcedureCall data_scale_call();

    ::krpc::schema::ProcedureCall is_complete_call();

    ::krpc::schema::ProcedureCall science_call();

    ::krpc::schema::ProcedureCall science_cap_call();

    ::krpc::schema::ProcedureCall scientific_value_call();

    ::krpc::schema::ProcedureCall subject_value_call();

    ::krpc::schema::ProcedureCall title_call();
  };

  /**
   * A sensor, such as a thermometer. Obtained by calling SpaceCenter::Part::sensor.
   */
  class Sensor : public krpc::Object<Sensor> {
   public:
    explicit Sensor(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the sensor is active.
     */
    bool active();

    /**
     * Whether the sensor is active.
     */
    void set_active(bool value);

    /**
     * The part object for this sensor.
     */
    SpaceCenter::Part part();

    /**
     * The current value of the sensor.
     */
    std::string value();

    ::krpc::Stream<bool> active_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<std::string> value_stream();

    ::krpc::schema::ProcedureCall active_call();

    ::krpc::schema::ProcedureCall set_active_call(bool value);

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall value_call();
  };

  /**
   * A solar panel. Obtained by calling SpaceCenter::Part::solar_panel.
   */
  class SolarPanel : public krpc::Object<SolarPanel> {
   public:
    explicit SolarPanel(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether the solar panel is deployable.
     */
    bool deployable();

    /**
     * Whether the solar panel is extended.
     */
    bool deployed();

    /**
     * Whether the solar panel is extended.
     */
    void set_deployed(bool value);

    /**
     * The current amount of energy being generated by the solar panel, in
     * units of charge per second.
     */
    float energy_flow();

    /**
     * The part object for this solar panel.
     */
    SpaceCenter::Part part();

    /**
     * The current state of the solar panel.
     */
    SpaceCenter::SolarPanelState state();

    /**
     * The current amount of sunlight that is incident on the solar panel,
     * as a percentage. A value between 0 and 1.
     */
    float sun_exposure();

    ::krpc::Stream<bool> deployable_stream();

    ::krpc::Stream<bool> deployed_stream();

    ::krpc::Stream<float> energy_flow_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<SpaceCenter::SolarPanelState> state_stream();

    ::krpc::Stream<float> sun_exposure_stream();

    ::krpc::schema::ProcedureCall deployable_call();

    ::krpc::schema::ProcedureCall deployed_call();

    ::krpc::schema::ProcedureCall set_deployed_call(bool value);

    ::krpc::schema::ProcedureCall energy_flow_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall state_call();

    ::krpc::schema::ProcedureCall sun_exposure_call();
  };

  /**
   * The component of an SpaceCenter::Engine or SpaceCenter::RCS part that generates thrust.
   * Can obtained by calling SpaceCenter::Engine::thrusters or SpaceCenter::RCS::thrusters.
   *
   * Engines can consist of multiple thrusters.
   * For example, the S3 KS-25x4 "Mammoth" has four rocket nozzels, and so consists of
   * four thrusters.
   */
  class Thruster : public krpc::Object<Thruster> {
   public:
    explicit Thruster(Client* client = nullptr, uint64_t id = 0);

    /**
     * Position around which the gimbal pivots.
     * @return The position as a vector.
     * @param referenceFrame The reference frame that the returned
     * position vector is in.
     */
    std::tuple<double, double, double> gimbal_position(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The direction of the force generated by the thruster, when the engine is in its
     * initial position (no gimballing), in the given reference frame.
     * This is opposite to the direction in which the thruster expels propellant.
     * @return The direction as a unit vector.
     * @param referenceFrame The reference frame that the returned
     * direction is in.
     */
    std::tuple<double, double, double> initial_thrust_direction(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The position at which the thruster generates thrust, when the engine is in its
     * initial position (no gimballing), in the given reference frame.
     * @return The position as a vector.
     * @param referenceFrame The reference frame that the returned
     * position vector is in.
     *
     * This position can move when the gimbal rotates. This is because the thrust position and
     * gimbal position are not necessarily the same.
     */
    std::tuple<double, double, double> initial_thrust_position(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The direction of the force generated by the thruster, in the given reference frame.
     * This is opposite to the direction in which the thruster expels propellant.
     * For gimballed engines, this takes into account the current rotation of the gimbal.
     * @return The direction as a unit vector.
     * @param referenceFrame The reference frame that the returned
     * direction is in.
     */
    std::tuple<double, double, double> thrust_direction(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The position at which the thruster generates thrust, in the given reference frame.
     * For gimballed engines, this takes into account the current rotation of the gimbal.
     * @return The position as a vector.
     * @param referenceFrame The reference frame that the returned
     * position vector is in.
     */
    std::tuple<double, double, double> thrust_position(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The current gimbal angle in the pitch, roll and yaw axes, in degrees.
     */
    std::tuple<double, double, double> gimbal_angle();

    /**
     * Whether the thruster is gimballed.
     */
    bool gimballed();

    /**
     * The SpaceCenter::Part that contains this thruster.
     */
    SpaceCenter::Part part();

    /**
     * A reference frame that is fixed relative to the thruster and orientated with
     * its thrust direction (SpaceCenter::Thruster::thrust_direction).
     * For gimballed engines, this takes into account the current rotation of the gimbal.
     *
     * - The origin is at the position of thrust for this thruster
     *   (SpaceCenter::Thruster::thrust_position).
     * - The axes rotate with the thrust direction.
     *   This is the direction in which the thruster expels propellant, including any gimballing.
     * - The y-axis points along the thrust direction.
     * - The x-axis and z-axis are perpendicular to the thrust direction.
     */
    SpaceCenter::ReferenceFrame thrust_reference_frame();

    ::krpc::Stream<std::tuple<double, double, double>> gimbal_position_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> initial_thrust_direction_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> initial_thrust_position_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> thrust_direction_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> thrust_position_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> gimbal_angle_stream();

    ::krpc::Stream<bool> gimballed_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> thrust_reference_frame_stream();

    ::krpc::schema::ProcedureCall gimbal_position_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall initial_thrust_direction_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall initial_thrust_position_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall thrust_direction_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall thrust_position_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall gimbal_angle_call();

    ::krpc::schema::ProcedureCall gimballed_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall thrust_reference_frame_call();
  };

  /**
   * These objects are used to interact with vessels in KSP. This includes getting
   * orbital and flight data, manipulating control inputs and managing resources.
   * Created using SpaceCenter::active_vessel or SpaceCenter::vessels.
   */
  class Vessel : public krpc::Object<Vessel> {
   public:
    explicit Vessel(Client* client = nullptr, uint64_t id = 0);

    /**
     * The angular velocity of the vessel, in the given reference frame.
     * @return The angular velocity as a vector. The magnitude of the vector is the rotational
     * speed of the vessel, in radians per second. The direction of the vector indicates the
     * axis of rotation, using the right-hand rule.
     * @param referenceFrame The reference frame the returned
     * angular velocity is in.
     */
    std::tuple<double, double, double> angular_velocity(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * Gets the total available thrust that can be produced by the vessel's
     * active engines, in Newtons. This is computed by summing
     * SpaceCenter::Engine::available_thrust_at for every active engine in the vessel.
     * Takes the given pressure into account.
     * @param pressure Atmospheric pressure in atmospheres
     */
    float available_thrust_at(double pressure);

    /**
     * The axis-aligned bounding box of the vessel in the given reference frame.
     * @return The positions of the minimum and maximum vertices of the box,
     * as position vectors.
     * @param referenceFrame The reference frame that the returned
     * position vectors are in.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > bounding_box(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The direction in which the vessel is pointing, in the given reference frame.
     * @return The direction as a unit vector.
     * @param referenceFrame The reference frame that the returned
     * direction is in.
     */
    std::tuple<double, double, double> direction(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * Returns a SpaceCenter::Flight object that can be used to get flight
     * telemetry for the vessel, in the specified reference frame.
     * @param referenceFrame Reference frame. Defaults to the vessel's surface reference frame
     * (SpaceCenter::Vessel::surface_reference_frame).
     */
    SpaceCenter::Flight flight(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The total maximum thrust that can be produced by the vessel's active
     * engines, in Newtons. This is computed by summing
     * SpaceCenter::Engine::max_thrust_at for every active engine.
     * Takes the given pressure into account.
     * @param pressure Atmospheric pressure in atmospheres
     */
    float max_thrust_at(double pressure);

    /**
     * The position of the center of mass of the vessel, in the given reference frame.
     * @return The position as a vector.
     * @param referenceFrame The reference frame that the returned
     * position vector is in.
     */
    std::tuple<double, double, double> position(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * Recover the vessel.
     */
    void recover();

    /**
     * Returns a SpaceCenter::Resources object, that can used to get
     * information about resources stored in a given stage.
     * @param stage Get resources for parts that are decoupled in this stage.
     * @param cumulative When false, returns the resources for parts
     * decoupled in just the given stage. When true returns the resources decoupled in
     * the given stage and all subsequent stages combined.
     */
    SpaceCenter::Resources resources_in_decouple_stage(int32_t stage, bool cumulative);

    /**
     * The rotation of the vessel, in the given reference frame.
     * @return The rotation as a quaternion of the form (x, y, z, w).
     * @param referenceFrame The reference frame that the returned
     * rotation is in.
     */
    std::tuple<double, double, double, double> rotation(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * The combined specific impulse of all active engines, in seconds. This is computed using the formula
     * <a href="https://wiki.kerbalspaceprogram.com/wiki/Specific_impulse#Multiple_engines">described here</a>.
     * Takes the given pressure into account.
     * @param pressure Atmospheric pressure in atmospheres
     */
    float specific_impulse_at(double pressure);

    /**
     * The velocity of the center of mass of the vessel, in the given reference frame.
     * @return The velocity as a vector. The vector points in the direction of travel,
     * and its magnitude is the speed of the body in meters per second.
     * @param referenceFrame The reference frame that the returned
     * velocity vector is in.
     */
    std::tuple<double, double, double> velocity(SpaceCenter::ReferenceFrame reference_frame);

    /**
     * An SpaceCenter::AutoPilot object, that can be used to perform
     * simple auto-piloting of the vessel.
     */
    SpaceCenter::AutoPilot auto_pilot();

    /**
     * The maximum torque that the aerodynamic control surfaces can generate.
     * Returns the torques in N.m around each of the coordinate axes of the
     * vessels reference frame (SpaceCenter::ReferenceFrame).
     * These axes are equivalent to the pitch, roll and yaw axes of the vessel.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > available_control_surface_torque();

    /**
     * The maximum torque that the currently active and gimballed engines can generate.
     * Returns the torques in N.m around each of the coordinate axes of the
     * vessels reference frame (SpaceCenter::ReferenceFrame).
     * These axes are equivalent to the pitch, roll and yaw axes of the vessel.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > available_engine_torque();

    /**
     * The maximum torque that parts (excluding reaction wheels, gimballed engines,
     * RCS and control surfaces) can generate.
     * Returns the torques in N.m around each of the coordinate axes of the
     * vessels reference frame (SpaceCenter::ReferenceFrame).
     * These axes are equivalent to the pitch, roll and yaw axes of the vessel.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > available_other_torque();

    /**
     * The maximum force that the currently active RCS thrusters can generate.
     * Returns the forces in N along each of the coordinate axes of the
     * vessels reference frame (SpaceCenter::ReferenceFrame).
     * These axes are equivalent to the right, forward and bottom directions of the vessel.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > available_rcs_force();

    /**
     * The maximum torque that the currently active RCS thrusters can generate.
     * Returns the torques in N.m around each of the coordinate axes of the
     * vessels reference frame (SpaceCenter::ReferenceFrame).
     * These axes are equivalent to the pitch, roll and yaw axes of the vessel.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > available_rcs_torque();

    /**
     * The maximum torque that the currently active and powered reaction wheels can generate.
     * Returns the torques in N.m around each of the coordinate axes of the
     * vessels reference frame (SpaceCenter::ReferenceFrame).
     * These axes are equivalent to the pitch, roll and yaw axes of the vessel.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > available_reaction_wheel_torque();

    /**
     * Gets the total available thrust that can be produced by the vessel's
     * active engines, in Newtons. This is computed by summing
     * SpaceCenter::Engine::available_thrust for every active engine in the vessel.
     */
    float available_thrust();

    /**
     * The maximum torque that the vessel generates. Includes contributions from
     * reaction wheels, RCS, gimballed engines and aerodynamic control surfaces.
     * Returns the torques in N.m around each of the coordinate axes of the
     * vessels reference frame (SpaceCenter::ReferenceFrame).
     * These axes are equivalent to the pitch, roll and yaw axes of the vessel.
     */
    std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > available_torque();

    /**
     * The name of the biome the vessel is currently in.
     */
    std::string biome();

    /**
     * Returns a SpaceCenter::Comms object that can be used to interact
     * with CommNet for this vessel.
     */
    SpaceCenter::Comms comms();

    /**
     * Returns a SpaceCenter::Control object that can be used to manipulate
     * the vessel's control inputs. For example, its pitch/yaw/roll controls,
     * RCS and thrust.
     */
    SpaceCenter::Control control();

    /**
     * The crew in the vessel.
     */
    std::vector<SpaceCenter::CrewMember> crew();

    /**
     * The number of crew that can occupy the vessel.
     */
    int32_t crew_capacity();

    /**
     * The number of crew that are occupying the vessel.
     */
    int32_t crew_count();

    /**
     * The total mass of the vessel, excluding resources, in kg.
     */
    float dry_mass();

    /**
     * The inertia tensor of the vessel around its center of mass,
     * in the vessels reference frame (SpaceCenter::ReferenceFrame).
     * Returns the 3x3 matrix as a list of elements, in row-major order.
     */
    std::vector<double> inertia_tensor();

    /**
     * The combined specific impulse of all active engines at sea level on Kerbin, in seconds.
     * This is computed using the formula
     * <a href="https://wiki.kerbalspaceprogram.com/wiki/Specific_impulse#Multiple_engines">described here</a>.
     */
    float kerbin_sea_level_specific_impulse();

    /**
     * The total mass of the vessel, including resources, in kg.
     */
    float mass();

    /**
     * The total maximum thrust that can be produced by the vessel's active
     * engines, in Newtons. This is computed by summing
     * SpaceCenter::Engine::max_thrust for every active engine.
     */
    float max_thrust();

    /**
     * The total maximum thrust that can be produced by the vessel's active
     * engines when the vessel is in a vacuum, in Newtons. This is computed by
     * summing SpaceCenter::Engine::max_vacuum_thrust for every active engine.
     */
    float max_vacuum_thrust();

    /**
     * The mission elapsed time in seconds.
     */
    double met();

    /**
     * The moment of inertia of the vessel around its center of mass in kg.m^2.
     * The inertia values in the returned 3-tuple are around the
     * pitch, roll and yaw directions respectively.
     * This corresponds to the vessels reference frame (SpaceCenter::ReferenceFrame).
     */
    std::tuple<double, double, double> moment_of_inertia();

    /**
     * The name of the vessel.
     */
    std::string name();

    /**
     * The name of the vessel.
     */
    void set_name(std::string value);

    /**
     * The current orbit of the vessel.
     */
    SpaceCenter::Orbit orbit();

    /**
     * The reference frame that is fixed relative to the vessel,
     * and orientated with the vessels orbital prograde/normal/radial directions.
     *
     * - The origin is at the center of mass of the vessel.
     * - The axes rotate with the orbital prograde/normal/radial directions.
     * - The x-axis points in the orbital anti-radial direction.
     * - The y-axis points in the orbital prograde direction.
     * - The z-axis points in the orbital normal direction.
     *
     * Be careful not to confuse this with 'orbit' mode on the navball.
     */
    SpaceCenter::ReferenceFrame orbital_reference_frame();

    /**
     * A SpaceCenter::Parts object, that can used to interact with the parts that make up this vessel.
     */
    SpaceCenter::Parts parts();

    /**
     * Whether the vessel is recoverable.
     */
    bool recoverable();

    /**
     * The reference frame that is fixed relative to the vessel,
     * and orientated with the vessel.
     *
     * - The origin is at the center of mass of the vessel.
     * - The axes rotate with the vessel.
     * - The x-axis points out to the right of the vessel.
     * - The y-axis points in the forward direction of the vessel.
     * - The z-axis points out of the bottom off the vessel.
     */
    SpaceCenter::ReferenceFrame reference_frame();

    /**
     * A SpaceCenter::Resources object, that can used to get information
     * about resources stored in the vessel.
     */
    SpaceCenter::Resources resources();

    /**
     * The situation the vessel is in.
     */
    SpaceCenter::VesselSituation situation();

    /**
     * The combined specific impulse of all active engines, in seconds. This is computed using the formula
     * <a href="https://wiki.kerbalspaceprogram.com/wiki/Specific_impulse#Multiple_engines">described here</a>.
     */
    float specific_impulse();

    /**
     * The reference frame that is fixed relative to the vessel,
     * and orientated with the surface of the body being orbited.
     *
     * - The origin is at the center of mass of the vessel.
     * - The axes rotate with the north and up directions on the surface of the body.
     * - The x-axis points in the <a href="https://en.wikipedia.org/wiki/Zenith">zenith</a>
     *   direction (upwards, normal to the body being orbited, from the center of the body towards the center of
     *   mass of the vessel).
     * - The y-axis points northwards towards the
     *   <a href="https://en.wikipedia.org/wiki/Horizon">astronomical horizon</a> (north, and tangential to the
     *   surface of the body -- the direction in which a compass would point when on the surface).
     * - The z-axis points eastwards towards the
     *   <a href="https://en.wikipedia.org/wiki/Horizon">astronomical horizon</a> (east, and tangential to the
     *   surface of the body -- east on a compass when on the surface).
     *
     * Be careful not to confuse this with 'surface' mode on the navball.
     */
    SpaceCenter::ReferenceFrame surface_reference_frame();

    /**
     * The reference frame that is fixed relative to the vessel,
     * and orientated with the velocity vector of the vessel relative
     * to the surface of the body being orbited.
     *
     * - The origin is at the center of mass of the vessel.
     * - The axes rotate with the vessel's velocity vector.
     * - The y-axis points in the direction of the vessel's velocity vector,
     *   relative to the surface of the body being orbited.
     * - The z-axis is in the plane of the
     *   <a href="https://en.wikipedia.org/wiki/Horizon">astronomical horizon</a>.
     * - The x-axis is orthogonal to the other two axes.
     */
    SpaceCenter::ReferenceFrame surface_velocity_reference_frame();

    /**
     * The total thrust currently being produced by the vessel's engines, in
     * Newtons. This is computed by summing SpaceCenter::Engine::thrust for
     * every engine in the vessel.
     */
    float thrust();

    /**
     * The type of the vessel.
     */
    SpaceCenter::VesselType type();

    /**
     * The type of the vessel.
     */
    void set_type(SpaceCenter::VesselType value);

    /**
     * The combined vacuum specific impulse of all active engines, in seconds. This is computed using the formula
     * <a href="https://wiki.kerbalspaceprogram.com/wiki/Specific_impulse#Multiple_engines">described here</a>.
     */
    float vacuum_specific_impulse();

    ::krpc::Stream<std::tuple<double, double, double>> angular_velocity_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<float> available_thrust_at_stream(double pressure);

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> bounding_box_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<std::tuple<double, double, double>> direction_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<SpaceCenter::Flight> flight_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<float> max_thrust_at_stream(double pressure);

    ::krpc::Stream<std::tuple<double, double, double>> position_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<SpaceCenter::Resources> resources_in_decouple_stage_stream(int32_t stage, bool cumulative);

    ::krpc::Stream<std::tuple<double, double, double, double>> rotation_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<float> specific_impulse_at_stream(double pressure);

    ::krpc::Stream<std::tuple<double, double, double>> velocity_stream(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::Stream<SpaceCenter::AutoPilot> auto_pilot_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> available_control_surface_torque_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> available_engine_torque_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> available_other_torque_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> available_rcs_force_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> available_rcs_torque_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> available_reaction_wheel_torque_stream();

    ::krpc::Stream<float> available_thrust_stream();

    ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> available_torque_stream();

    ::krpc::Stream<std::string> biome_stream();

    ::krpc::Stream<SpaceCenter::Comms> comms_stream();

    ::krpc::Stream<SpaceCenter::Control> control_stream();

    ::krpc::Stream<std::vector<SpaceCenter::CrewMember>> crew_stream();

    ::krpc::Stream<int32_t> crew_capacity_stream();

    ::krpc::Stream<int32_t> crew_count_stream();

    ::krpc::Stream<float> dry_mass_stream();

    ::krpc::Stream<std::vector<double>> inertia_tensor_stream();

    ::krpc::Stream<float> kerbin_sea_level_specific_impulse_stream();

    ::krpc::Stream<float> mass_stream();

    ::krpc::Stream<float> max_thrust_stream();

    ::krpc::Stream<float> max_vacuum_thrust_stream();

    ::krpc::Stream<double> met_stream();

    ::krpc::Stream<std::tuple<double, double, double>> moment_of_inertia_stream();

    ::krpc::Stream<std::string> name_stream();

    ::krpc::Stream<SpaceCenter::Orbit> orbit_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> orbital_reference_frame_stream();

    ::krpc::Stream<SpaceCenter::Parts> parts_stream();

    ::krpc::Stream<bool> recoverable_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> reference_frame_stream();

    ::krpc::Stream<SpaceCenter::Resources> resources_stream();

    ::krpc::Stream<SpaceCenter::VesselSituation> situation_stream();

    ::krpc::Stream<float> specific_impulse_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> surface_reference_frame_stream();

    ::krpc::Stream<SpaceCenter::ReferenceFrame> surface_velocity_reference_frame_stream();

    ::krpc::Stream<float> thrust_stream();

    ::krpc::Stream<SpaceCenter::VesselType> type_stream();

    ::krpc::Stream<float> vacuum_specific_impulse_stream();

    ::krpc::schema::ProcedureCall angular_velocity_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall available_thrust_at_call(double pressure);

    ::krpc::schema::ProcedureCall bounding_box_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall direction_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall flight_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall max_thrust_at_call(double pressure);

    ::krpc::schema::ProcedureCall position_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall recover_call();

    ::krpc::schema::ProcedureCall resources_in_decouple_stage_call(int32_t stage, bool cumulative);

    ::krpc::schema::ProcedureCall rotation_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall specific_impulse_at_call(double pressure);

    ::krpc::schema::ProcedureCall velocity_call(SpaceCenter::ReferenceFrame reference_frame);

    ::krpc::schema::ProcedureCall auto_pilot_call();

    ::krpc::schema::ProcedureCall available_control_surface_torque_call();

    ::krpc::schema::ProcedureCall available_engine_torque_call();

    ::krpc::schema::ProcedureCall available_other_torque_call();

    ::krpc::schema::ProcedureCall available_rcs_force_call();

    ::krpc::schema::ProcedureCall available_rcs_torque_call();

    ::krpc::schema::ProcedureCall available_reaction_wheel_torque_call();

    ::krpc::schema::ProcedureCall available_thrust_call();

    ::krpc::schema::ProcedureCall available_torque_call();

    ::krpc::schema::ProcedureCall biome_call();

    ::krpc::schema::ProcedureCall comms_call();

    ::krpc::schema::ProcedureCall control_call();

    ::krpc::schema::ProcedureCall crew_call();

    ::krpc::schema::ProcedureCall crew_capacity_call();

    ::krpc::schema::ProcedureCall crew_count_call();

    ::krpc::schema::ProcedureCall dry_mass_call();

    ::krpc::schema::ProcedureCall inertia_tensor_call();

    ::krpc::schema::ProcedureCall kerbin_sea_level_specific_impulse_call();

    ::krpc::schema::ProcedureCall mass_call();

    ::krpc::schema::ProcedureCall max_thrust_call();

    ::krpc::schema::ProcedureCall max_vacuum_thrust_call();

    ::krpc::schema::ProcedureCall met_call();

    ::krpc::schema::ProcedureCall moment_of_inertia_call();

    ::krpc::schema::ProcedureCall name_call();

    ::krpc::schema::ProcedureCall set_name_call(std::string value);

    ::krpc::schema::ProcedureCall orbit_call();

    ::krpc::schema::ProcedureCall orbital_reference_frame_call();

    ::krpc::schema::ProcedureCall parts_call();

    ::krpc::schema::ProcedureCall recoverable_call();

    ::krpc::schema::ProcedureCall reference_frame_call();

    ::krpc::schema::ProcedureCall resources_call();

    ::krpc::schema::ProcedureCall situation_call();

    ::krpc::schema::ProcedureCall specific_impulse_call();

    ::krpc::schema::ProcedureCall surface_reference_frame_call();

    ::krpc::schema::ProcedureCall surface_velocity_reference_frame_call();

    ::krpc::schema::ProcedureCall thrust_call();

    ::krpc::schema::ProcedureCall type_call();

    ::krpc::schema::ProcedureCall set_type_call(SpaceCenter::VesselType value);

    ::krpc::schema::ProcedureCall vacuum_specific_impulse_call();
  };

  /**
   * Represents a waypoint. Can be created using SpaceCenter::WaypointManager::add_waypoint.
   */
  class Waypoint : public krpc::Object<Waypoint> {
   public:
    explicit Waypoint(Client* client = nullptr, uint64_t id = 0);

    /**
     * Removes the waypoint.
     */
    void remove();

    /**
     * The altitude of the waypoint above the surface of the body, in meters.
     * When over water, this is the altitude above the sea floor.
     */
    double bedrock_altitude();

    /**
     * The altitude of the waypoint above the surface of the body, in meters.
     * When over water, this is the altitude above the sea floor.
     */
    void set_bedrock_altitude(double value);

    /**
     * The celestial body the waypoint is attached to.
     */
    SpaceCenter::CelestialBody body();

    /**
     * The celestial body the waypoint is attached to.
     */
    void set_body(SpaceCenter::CelestialBody value);

    /**
     * true if this waypoint is part of a set of clustered waypoints with greek letter
     * names appended (Alpha, Beta, Gamma, etc).
     * If true, there is a one-to-one correspondence with the greek letter name and
     * the SpaceCenter::Waypoint::index.
     */
    bool clustered();

    /**
     * The seed of the icon color. See SpaceCenter::WaypointManager::colors for example colors.
     */
    int32_t color();

    /**
     * The seed of the icon color. See SpaceCenter::WaypointManager::colors for example colors.
     */
    void set_color(int32_t value);

    /**
     * The associated contract.
     */
    SpaceCenter::Contract contract();

    /**
     * true if the waypoint is attached to the ground.
     */
    bool grounded();

    /**
     * Whether the waypoint belongs to a contract.
     */
    bool has_contract();

    /**
     * The icon of the waypoint.
     */
    std::string icon();

    /**
     * The icon of the waypoint.
     */
    void set_icon(std::string value);

    /**
     * The integer index of this waypoint within its cluster of sibling waypoints.
     * In other words, when you have a cluster of waypoints called "Somewhere Alpha",
     * "Somewhere Beta" and "Somewhere Gamma", the alpha site has index 0, the beta
     * site has index 1 and the gamma site has index 2.
     * When SpaceCenter::Waypoint::clustered is false, this is zero.
     */
    int32_t index();

    /**
     * The latitude of the waypoint.
     */
    double latitude();

    /**
     * The latitude of the waypoint.
     */
    void set_latitude(double value);

    /**
     * The longitude of the waypoint.
     */
    double longitude();

    /**
     * The longitude of the waypoint.
     */
    void set_longitude(double value);

    /**
     * The altitude of the waypoint above sea level, in meters.
     */
    double mean_altitude();

    /**
     * The altitude of the waypoint above sea level, in meters.
     */
    void set_mean_altitude(double value);

    /**
     * The name of the waypoint as it appears on the map and the contract.
     */
    std::string name();

    /**
     * The name of the waypoint as it appears on the map and the contract.
     */
    void set_name(std::string value);

    /**
     * true if the waypoint is near to the surface of a body.
     */
    bool near_surface();

    /**
     * The altitude of the waypoint above the surface of the body or sea level,
     * whichever is closer, in meters.
     */
    double surface_altitude();

    /**
     * The altitude of the waypoint above the surface of the body or sea level,
     * whichever is closer, in meters.
     */
    void set_surface_altitude(double value);

    ::krpc::Stream<double> bedrock_altitude_stream();

    ::krpc::Stream<SpaceCenter::CelestialBody> body_stream();

    ::krpc::Stream<bool> clustered_stream();

    ::krpc::Stream<int32_t> color_stream();

    ::krpc::Stream<SpaceCenter::Contract> contract_stream();

    ::krpc::Stream<bool> grounded_stream();

    ::krpc::Stream<bool> has_contract_stream();

    ::krpc::Stream<std::string> icon_stream();

    ::krpc::Stream<int32_t> index_stream();

    ::krpc::Stream<double> latitude_stream();

    ::krpc::Stream<double> longitude_stream();

    ::krpc::Stream<double> mean_altitude_stream();

    ::krpc::Stream<std::string> name_stream();

    ::krpc::Stream<bool> near_surface_stream();

    ::krpc::Stream<double> surface_altitude_stream();

    ::krpc::schema::ProcedureCall remove_call();

    ::krpc::schema::ProcedureCall bedrock_altitude_call();

    ::krpc::schema::ProcedureCall set_bedrock_altitude_call(double value);

    ::krpc::schema::ProcedureCall body_call();

    ::krpc::schema::ProcedureCall set_body_call(SpaceCenter::CelestialBody value);

    ::krpc::schema::ProcedureCall clustered_call();

    ::krpc::schema::ProcedureCall color_call();

    ::krpc::schema::ProcedureCall set_color_call(int32_t value);

    ::krpc::schema::ProcedureCall contract_call();

    ::krpc::schema::ProcedureCall grounded_call();

    ::krpc::schema::ProcedureCall has_contract_call();

    ::krpc::schema::ProcedureCall icon_call();

    ::krpc::schema::ProcedureCall set_icon_call(std::string value);

    ::krpc::schema::ProcedureCall index_call();

    ::krpc::schema::ProcedureCall latitude_call();

    ::krpc::schema::ProcedureCall set_latitude_call(double value);

    ::krpc::schema::ProcedureCall longitude_call();

    ::krpc::schema::ProcedureCall set_longitude_call(double value);

    ::krpc::schema::ProcedureCall mean_altitude_call();

    ::krpc::schema::ProcedureCall set_mean_altitude_call(double value);

    ::krpc::schema::ProcedureCall name_call();

    ::krpc::schema::ProcedureCall set_name_call(std::string value);

    ::krpc::schema::ProcedureCall near_surface_call();

    ::krpc::schema::ProcedureCall surface_altitude_call();

    ::krpc::schema::ProcedureCall set_surface_altitude_call(double value);
  };

  /**
   * Waypoints are the location markers you can see on the map view showing you where contracts are targeted for.
   * With this structure, you can obtain coordinate data for the locations of these waypoints.
   * Obtained by calling SpaceCenter::waypoint_manager.
   */
  class WaypointManager : public krpc::Object<WaypointManager> {
   public:
    explicit WaypointManager(Client* client = nullptr, uint64_t id = 0);

    /**
     * Creates a waypoint at the given position at ground level, and returns a
     * SpaceCenter::Waypoint object that can be used to modify it.
     * @param latitude Latitude of the waypoint.
     * @param longitude Longitude of the waypoint.
     * @param body Celestial body the waypoint is attached to.
     * @param name Name of the waypoint.
     * @return
     */
    SpaceCenter::Waypoint add_waypoint(double latitude, double longitude, SpaceCenter::CelestialBody body, std::string name);

    /**
     * Creates a waypoint at the given position and altitude, and returns a
     * SpaceCenter::Waypoint object that can be used to modify it.
     * @param latitude Latitude of the waypoint.
     * @param longitude Longitude of the waypoint.
     * @param altitude Altitude (above sea level) of the waypoint.
     * @param body Celestial body the waypoint is attached to.
     * @param name Name of the waypoint.
     * @return
     */
    SpaceCenter::Waypoint add_waypoint_at_altitude(double latitude, double longitude, double altitude, SpaceCenter::CelestialBody body, std::string name);

    /**
     * An example map of known color - seed pairs.
     * Any other integers may be used as seed.
     */
    std::map<std::string, int32_t> colors();

    /**
     * Returns all available icons (from "GameData/Squad/Contracts/Icons/").
     */
    std::vector<std::string> icons();

    /**
     * A list of all existing waypoints.
     */
    std::vector<SpaceCenter::Waypoint> waypoints();

    ::krpc::Stream<SpaceCenter::Waypoint> add_waypoint_stream(double latitude, double longitude, SpaceCenter::CelestialBody body, std::string name);

    ::krpc::Stream<SpaceCenter::Waypoint> add_waypoint_at_altitude_stream(double latitude, double longitude, double altitude, SpaceCenter::CelestialBody body, std::string name);

    ::krpc::Stream<std::map<std::string, int32_t>> colors_stream();

    ::krpc::Stream<std::vector<std::string>> icons_stream();

    ::krpc::Stream<std::vector<SpaceCenter::Waypoint>> waypoints_stream();

    ::krpc::schema::ProcedureCall add_waypoint_call(double latitude, double longitude, SpaceCenter::CelestialBody body, std::string name);

    ::krpc::schema::ProcedureCall add_waypoint_at_altitude_call(double latitude, double longitude, double altitude, SpaceCenter::CelestialBody body, std::string name);

    ::krpc::schema::ProcedureCall colors_call();

    ::krpc::schema::ProcedureCall icons_call();

    ::krpc::schema::ProcedureCall waypoints_call();
  };

  /**
   * A wheel. Includes landing gear and rover wheels.
   * Obtained by calling SpaceCenter::Part::wheel.
   * Can be used to control the motors, steering and deployment of wheels, among other things.
   */
  class Wheel : public krpc::Object<Wheel> {
   public:
    explicit Wheel(Client* client = nullptr, uint64_t id = 0);

    /**
     * Whether automatic friction control is enabled.
     */
    bool auto_friction_control();

    /**
     * Whether automatic friction control is enabled.
     */
    void set_auto_friction_control(bool value);

    /**
     * The braking force, as a percentage of maximum, when the brakes are applied.
     */
    float brakes();

    /**
     * The braking force, as a percentage of maximum, when the brakes are applied.
     */
    void set_brakes(float value);

    /**
     * Whether the wheel is broken.
     */
    bool broken();

    /**
     * Current deflection of the wheel.
     */
    float deflection();

    /**
     * Whether the wheel is deployable.
     */
    bool deployable();

    /**
     * Whether the wheel is deployed.
     */
    bool deployed();

    /**
     * Whether the wheel is deployed.
     */
    void set_deployed(bool value);

    /**
     * Manual setting for the motor limiter.
     * Only takes effect if the wheel has automatic traction control disabled.
     * A value between 0 and 100 inclusive.
     */
    float drive_limiter();

    /**
     * Manual setting for the motor limiter.
     * Only takes effect if the wheel has automatic traction control disabled.
     * A value between 0 and 100 inclusive.
     */
    void set_drive_limiter(float value);

    /**
     * Whether the wheel is touching the ground.
     */
    bool grounded();

    /**
     * Whether the wheel has brakes.
     */
    bool has_brakes();

    /**
     * Whether the wheel has suspension.
     */
    bool has_suspension();

    /**
     * Manual friction control value. Only has an effect if automatic friction control is disabled.
     * A value between 0 and 5 inclusive.
     */
    float manual_friction_control();

    /**
     * Manual friction control value. Only has an effect if automatic friction control is disabled.
     * A value between 0 and 5 inclusive.
     */
    void set_manual_friction_control(float value);

    /**
     * Whether the motor is enabled.
     */
    bool motor_enabled();

    /**
     * Whether the motor is enabled.
     */
    void set_motor_enabled(bool value);

    /**
     * Whether the direction of the motor is inverted.
     */
    bool motor_inverted();

    /**
     * Whether the direction of the motor is inverted.
     */
    void set_motor_inverted(bool value);

    /**
     * The output of the motor. This is the torque currently being generated, in Newton meters.
     */
    float motor_output();

    /**
     * Whether the direction of the motor is inverted.
     */
    SpaceCenter::MotorState motor_state();

    /**
     * The part object for this wheel.
     */
    SpaceCenter::Part part();

    /**
     * Whether the wheel is powered by a motor.
     */
    bool powered();

    /**
     * Radius of the wheel, in meters.
     */
    float radius();

    /**
     * Whether the wheel is repairable.
     */
    bool repairable();

    /**
     * Current slip of the wheel.
     */
    float slip();

    /**
     * The current state of the wheel.
     */
    SpaceCenter::WheelState state();

    /**
     * Whether the wheel has steering.
     */
    bool steerable();

    /**
     * The steering angle limit.
     */
    float steering_angle_limit();

    /**
     * The steering angle limit.
     */
    void set_steering_angle_limit(float value);

    /**
     * Whether the wheel steering is enabled.
     */
    bool steering_enabled();

    /**
     * Whether the wheel steering is enabled.
     */
    void set_steering_enabled(bool value);

    /**
     * Whether the wheel steering is inverted.
     */
    bool steering_inverted();

    /**
     * Whether the wheel steering is inverted.
     */
    void set_steering_inverted(bool value);

    /**
     * Steering response time.
     */
    float steering_response_time();

    /**
     * Steering response time.
     */
    void set_steering_response_time(float value);

    /**
     * Current stress on the wheel.
     */
    float stress();

    /**
     * Current stress on the wheel as a percentage of its stress tolerance.
     */
    float stress_percentage();

    /**
     * Stress tolerance of the wheel.
     */
    float stress_tolerance();

    /**
     * Suspension damper strength, as set in the editor.
     */
    float suspension_damper_strength();

    /**
     * Suspension spring strength, as set in the editor.
     */
    float suspension_spring_strength();

    /**
     * Setting for the traction control.
     * Only takes effect if the wheel has automatic traction control enabled.
     * A value between 0 and 5 inclusive.
     */
    float traction_control();

    /**
     * Setting for the traction control.
     * Only takes effect if the wheel has automatic traction control enabled.
     * A value between 0 and 5 inclusive.
     */
    void set_traction_control(float value);

    /**
     * Whether automatic traction control is enabled.
     * A wheel only has traction control if it is powered.
     */
    bool traction_control_enabled();

    /**
     * Whether automatic traction control is enabled.
     * A wheel only has traction control if it is powered.
     */
    void set_traction_control_enabled(bool value);

    ::krpc::Stream<bool> auto_friction_control_stream();

    ::krpc::Stream<float> brakes_stream();

    ::krpc::Stream<bool> broken_stream();

    ::krpc::Stream<float> deflection_stream();

    ::krpc::Stream<bool> deployable_stream();

    ::krpc::Stream<bool> deployed_stream();

    ::krpc::Stream<float> drive_limiter_stream();

    ::krpc::Stream<bool> grounded_stream();

    ::krpc::Stream<bool> has_brakes_stream();

    ::krpc::Stream<bool> has_suspension_stream();

    ::krpc::Stream<float> manual_friction_control_stream();

    ::krpc::Stream<bool> motor_enabled_stream();

    ::krpc::Stream<bool> motor_inverted_stream();

    ::krpc::Stream<float> motor_output_stream();

    ::krpc::Stream<SpaceCenter::MotorState> motor_state_stream();

    ::krpc::Stream<SpaceCenter::Part> part_stream();

    ::krpc::Stream<bool> powered_stream();

    ::krpc::Stream<float> radius_stream();

    ::krpc::Stream<bool> repairable_stream();

    ::krpc::Stream<float> slip_stream();

    ::krpc::Stream<SpaceCenter::WheelState> state_stream();

    ::krpc::Stream<bool> steerable_stream();

    ::krpc::Stream<float> steering_angle_limit_stream();

    ::krpc::Stream<bool> steering_enabled_stream();

    ::krpc::Stream<bool> steering_inverted_stream();

    ::krpc::Stream<float> steering_response_time_stream();

    ::krpc::Stream<float> stress_stream();

    ::krpc::Stream<float> stress_percentage_stream();

    ::krpc::Stream<float> stress_tolerance_stream();

    ::krpc::Stream<float> suspension_damper_strength_stream();

    ::krpc::Stream<float> suspension_spring_strength_stream();

    ::krpc::Stream<float> traction_control_stream();

    ::krpc::Stream<bool> traction_control_enabled_stream();

    ::krpc::schema::ProcedureCall auto_friction_control_call();

    ::krpc::schema::ProcedureCall set_auto_friction_control_call(bool value);

    ::krpc::schema::ProcedureCall brakes_call();

    ::krpc::schema::ProcedureCall set_brakes_call(float value);

    ::krpc::schema::ProcedureCall broken_call();

    ::krpc::schema::ProcedureCall deflection_call();

    ::krpc::schema::ProcedureCall deployable_call();

    ::krpc::schema::ProcedureCall deployed_call();

    ::krpc::schema::ProcedureCall set_deployed_call(bool value);

    ::krpc::schema::ProcedureCall drive_limiter_call();

    ::krpc::schema::ProcedureCall set_drive_limiter_call(float value);

    ::krpc::schema::ProcedureCall grounded_call();

    ::krpc::schema::ProcedureCall has_brakes_call();

    ::krpc::schema::ProcedureCall has_suspension_call();

    ::krpc::schema::ProcedureCall manual_friction_control_call();

    ::krpc::schema::ProcedureCall set_manual_friction_control_call(float value);

    ::krpc::schema::ProcedureCall motor_enabled_call();

    ::krpc::schema::ProcedureCall set_motor_enabled_call(bool value);

    ::krpc::schema::ProcedureCall motor_inverted_call();

    ::krpc::schema::ProcedureCall set_motor_inverted_call(bool value);

    ::krpc::schema::ProcedureCall motor_output_call();

    ::krpc::schema::ProcedureCall motor_state_call();

    ::krpc::schema::ProcedureCall part_call();

    ::krpc::schema::ProcedureCall powered_call();

    ::krpc::schema::ProcedureCall radius_call();

    ::krpc::schema::ProcedureCall repairable_call();

    ::krpc::schema::ProcedureCall slip_call();

    ::krpc::schema::ProcedureCall state_call();

    ::krpc::schema::ProcedureCall steerable_call();

    ::krpc::schema::ProcedureCall steering_angle_limit_call();

    ::krpc::schema::ProcedureCall set_steering_angle_limit_call(float value);

    ::krpc::schema::ProcedureCall steering_enabled_call();

    ::krpc::schema::ProcedureCall set_steering_enabled_call(bool value);

    ::krpc::schema::ProcedureCall steering_inverted_call();

    ::krpc::schema::ProcedureCall set_steering_inverted_call(bool value);

    ::krpc::schema::ProcedureCall steering_response_time_call();

    ::krpc::schema::ProcedureCall set_steering_response_time_call(float value);

    ::krpc::schema::ProcedureCall stress_call();

    ::krpc::schema::ProcedureCall stress_percentage_call();

    ::krpc::schema::ProcedureCall stress_tolerance_call();

    ::krpc::schema::ProcedureCall suspension_damper_strength_call();

    ::krpc::schema::ProcedureCall suspension_spring_strength_call();

    ::krpc::schema::ProcedureCall traction_control_call();

    ::krpc::schema::ProcedureCall set_traction_control_call(float value);

    ::krpc::schema::ProcedureCall traction_control_enabled_call();

    ::krpc::schema::ProcedureCall set_traction_control_enabled_call(bool value);
  };
};

}  // namespace services

namespace encoder {

  inline std::string encode(const services::SpaceCenter::AntennaState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::AutoStrutMode& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::CameraMode& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::CargoBayState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::CommLinkType& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::ContractState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::ControlInputMode& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::ControlSource& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::ControlState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::CrewMemberGender& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::CrewMemberType& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::DockingPortState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::DrainMode& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::EditorFacility& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::GameMode& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::LegState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::MapFilterType& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::MotorState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::ParachuteState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::RadiatorState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::ResourceConverterState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::ResourceFlowMode& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::ResourceHarvesterState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::RosterStatus& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::SASMode& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::SolarPanelState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::SpeedMode& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::SuitType& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::VesselSituation& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::VesselType& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::WarpMode& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

  inline std::string encode(const services::SpaceCenter::WheelState& value) {
    return krpc::encoder::encode(static_cast<int32_t>(value));
  }

}  // namespace encoder

namespace decoder {

  inline void decode(services::SpaceCenter::AntennaState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::AntennaState>(x);
  }

  inline void decode(services::SpaceCenter::AutoStrutMode& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::AutoStrutMode>(x);
  }

  inline void decode(services::SpaceCenter::CameraMode& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::CameraMode>(x);
  }

  inline void decode(services::SpaceCenter::CargoBayState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::CargoBayState>(x);
  }

  inline void decode(services::SpaceCenter::CommLinkType& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::CommLinkType>(x);
  }

  inline void decode(services::SpaceCenter::ContractState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::ContractState>(x);
  }

  inline void decode(services::SpaceCenter::ControlInputMode& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::ControlInputMode>(x);
  }

  inline void decode(services::SpaceCenter::ControlSource& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::ControlSource>(x);
  }

  inline void decode(services::SpaceCenter::ControlState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::ControlState>(x);
  }

  inline void decode(services::SpaceCenter::CrewMemberGender& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::CrewMemberGender>(x);
  }

  inline void decode(services::SpaceCenter::CrewMemberType& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::CrewMemberType>(x);
  }

  inline void decode(services::SpaceCenter::DockingPortState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::DockingPortState>(x);
  }

  inline void decode(services::SpaceCenter::DrainMode& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::DrainMode>(x);
  }

  inline void decode(services::SpaceCenter::EditorFacility& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::EditorFacility>(x);
  }

  inline void decode(services::SpaceCenter::GameMode& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::GameMode>(x);
  }

  inline void decode(services::SpaceCenter::LegState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::LegState>(x);
  }

  inline void decode(services::SpaceCenter::MapFilterType& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::MapFilterType>(x);
  }

  inline void decode(services::SpaceCenter::MotorState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::MotorState>(x);
  }

  inline void decode(services::SpaceCenter::ParachuteState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::ParachuteState>(x);
  }

  inline void decode(services::SpaceCenter::RadiatorState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::RadiatorState>(x);
  }

  inline void decode(services::SpaceCenter::ResourceConverterState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::ResourceConverterState>(x);
  }

  inline void decode(services::SpaceCenter::ResourceFlowMode& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::ResourceFlowMode>(x);
  }

  inline void decode(services::SpaceCenter::ResourceHarvesterState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::ResourceHarvesterState>(x);
  }

  inline void decode(services::SpaceCenter::RosterStatus& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::RosterStatus>(x);
  }

  inline void decode(services::SpaceCenter::SASMode& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::SASMode>(x);
  }

  inline void decode(services::SpaceCenter::SolarPanelState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::SolarPanelState>(x);
  }

  inline void decode(services::SpaceCenter::SpeedMode& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::SpeedMode>(x);
  }

  inline void decode(services::SpaceCenter::SuitType& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::SuitType>(x);
  }

  inline void decode(services::SpaceCenter::VesselSituation& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::VesselSituation>(x);
  }

  inline void decode(services::SpaceCenter::VesselType& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::VesselType>(x);
  }

  inline void decode(services::SpaceCenter::WarpMode& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::WarpMode>(x);
  }

  inline void decode(services::SpaceCenter::WheelState& value, const std::string& data, Client* client) {
    int32_t x;
    decode(x, data, client);
    value = static_cast<services::SpaceCenter::WheelState>(x);
  }

}  // namespace decoder

namespace services {

inline SpaceCenter::SpaceCenter(Client* client):
  Service(client) {
}

inline bool SpaceCenter::can_rails_warp_at(int32_t factor = 1) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(factor));
  std::string _data = this->_client->invoke("SpaceCenter", "CanRailsWarpAt", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::can_revert_to_launch() {
  std::string _data = this->_client->invoke("SpaceCenter", "CanRevertToLaunch");
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::clear_target() {
  this->_client->invoke("SpaceCenter", "ClearTarget");
}

inline void SpaceCenter::create_kerbal(std::string name, std::string job, bool male) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(job));
  _args.push_back(encoder::encode(male));
  this->_client->invoke("SpaceCenter", "CreateKerbal", _args);
}

inline SpaceCenter::CrewMember SpaceCenter::get_kerbal(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  std::string _data = this->_client->invoke("SpaceCenter", "GetKerbal", _args);
  SpaceCenter::CrewMember _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::launch_vessel(std::string craft_directory, std::string name, std::string launch_site, bool recover = true, std::vector<std::string> crew = std::vector<std::string>(), std::string flag_url = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(craft_directory));
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(launch_site));
  _args.push_back(encoder::encode(recover));
  _args.push_back(encoder::encode(crew));
  _args.push_back(encoder::encode(flag_url));
  this->_client->invoke("SpaceCenter", "LaunchVessel", _args);
}

inline void SpaceCenter::launch_vessel_from_sph(std::string name, bool recover = true) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(recover));
  this->_client->invoke("SpaceCenter", "LaunchVesselFromSPH", _args);
}

inline void SpaceCenter::launch_vessel_from_vab(std::string name, bool recover = true) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(recover));
  this->_client->invoke("SpaceCenter", "LaunchVesselFromVAB", _args);
}

inline std::vector<std::string> SpaceCenter::launchable_vessels(std::string craft_directory) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(craft_directory));
  std::string _data = this->_client->invoke("SpaceCenter", "LaunchableVessels", _args);
  std::vector<std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::load(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  this->_client->invoke("SpaceCenter", "Load", _args);
}

inline void SpaceCenter::load_space_center() {
  this->_client->invoke("SpaceCenter", "LoadSpaceCenter");
}

inline void SpaceCenter::quickload() {
  this->_client->invoke("SpaceCenter", "Quickload");
}

inline void SpaceCenter::quicksave() {
  this->_client->invoke("SpaceCenter", "Quicksave");
}

inline double SpaceCenter::raycast_distance(std::tuple<double, double, double> position, std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(direction));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "RaycastDistance", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::raycast_part(std::tuple<double, double, double> position, std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(direction));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "RaycastPart", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::revert_to_launch() {
  this->_client->invoke("SpaceCenter", "RevertToLaunch");
}

inline void SpaceCenter::save(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  this->_client->invoke("SpaceCenter", "Save", _args);
}

inline void SpaceCenter::screenshot(std::string file_path, int32_t scale = 1) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(file_path));
  _args.push_back(encoder::encode(scale));
  this->_client->invoke("SpaceCenter", "Screenshot", _args);
}

inline void SpaceCenter::transfer_crew(SpaceCenter::CrewMember crew_member, SpaceCenter::Part target_part) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(crew_member));
  _args.push_back(encoder::encode(target_part));
  this->_client->invoke("SpaceCenter", "TransferCrew", _args);
}

inline std::tuple<double, double, double> SpaceCenter::transform_direction(std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(direction));
  _args.push_back(encoder::encode(from));
  _args.push_back(encoder::encode(to));
  std::string _data = this->_client->invoke("SpaceCenter", "TransformDirection", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::transform_position(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(from));
  _args.push_back(encoder::encode(to));
  std::string _data = this->_client->invoke("SpaceCenter", "TransformPosition", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double, double> SpaceCenter::transform_rotation(std::tuple<double, double, double, double> rotation, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(rotation));
  _args.push_back(encoder::encode(from));
  _args.push_back(encoder::encode(to));
  std::string _data = this->_client->invoke("SpaceCenter", "TransformRotation", _args);
  std::tuple<double, double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::transform_velocity(std::tuple<double, double, double> position, std::tuple<double, double, double> velocity, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(velocity));
  _args.push_back(encoder::encode(from));
  _args.push_back(encoder::encode(to));
  std::string _data = this->_client->invoke("SpaceCenter", "TransformVelocity", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::warp_to(double ut, float max_rails_rate = 100000.0, float max_physics_rate = 2.0) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(ut));
  _args.push_back(encoder::encode(max_rails_rate));
  _args.push_back(encoder::encode(max_physics_rate));
  this->_client->invoke("SpaceCenter", "WarpTo", _args);
}

inline SpaceCenter::Vessel SpaceCenter::active_vessel() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_ActiveVessel");
  SpaceCenter::Vessel _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::set_active_vessel(SpaceCenter::Vessel value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "set_ActiveVessel", _args);
}

inline SpaceCenter::AlarmManager SpaceCenter::alarm_manager() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_AlarmManager");
  SpaceCenter::AlarmManager _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::map<std::string, SpaceCenter::CelestialBody> SpaceCenter::bodies() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_Bodies");
  std::map<std::string, SpaceCenter::CelestialBody> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Camera SpaceCenter::camera() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_Camera");
  SpaceCenter::Camera _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ContractManager SpaceCenter::contract_manager() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_ContractManager");
  SpaceCenter::ContractManager _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::far_available() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_FARAvailable");
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::funds() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_Funds");
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::g() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_G");
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::GameMode SpaceCenter::game_mode() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_GameMode");
  SpaceCenter::GameMode _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::LaunchSite> SpaceCenter::launch_sites() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_LaunchSites");
  std::vector<SpaceCenter::LaunchSite> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::MapFilterType SpaceCenter::map_filter() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_MapFilter");
  SpaceCenter::MapFilterType _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::set_map_filter(SpaceCenter::MapFilterType value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "set_MapFilter", _args);
}

inline int32_t SpaceCenter::maximum_rails_warp_factor() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_MaximumRailsWarpFactor");
  int32_t _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::navball() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_Navball");
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::set_navball(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "set_Navball", _args);
}

inline int32_t SpaceCenter::physics_warp_factor() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_PhysicsWarpFactor");
  int32_t _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::set_physics_warp_factor(int32_t value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "set_PhysicsWarpFactor", _args);
}

inline int32_t SpaceCenter::rails_warp_factor() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_RailsWarpFactor");
  int32_t _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::set_rails_warp_factor(int32_t value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "set_RailsWarpFactor", _args);
}

inline float SpaceCenter::reputation() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_Reputation");
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::science() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_Science");
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::CelestialBody SpaceCenter::target_body() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_TargetBody");
  SpaceCenter::CelestialBody _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::set_target_body(SpaceCenter::CelestialBody value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "set_TargetBody", _args);
}

inline SpaceCenter::DockingPort SpaceCenter::target_docking_port() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_TargetDockingPort");
  SpaceCenter::DockingPort _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::set_target_docking_port(SpaceCenter::DockingPort value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "set_TargetDockingPort", _args);
}

inline SpaceCenter::Vessel SpaceCenter::target_vessel() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_TargetVessel");
  SpaceCenter::Vessel _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::set_target_vessel(SpaceCenter::Vessel value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "set_TargetVessel", _args);
}

inline bool SpaceCenter::ui_visible() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_UIVisible");
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::set_ui_visible(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "set_UIVisible", _args);
}

inline double SpaceCenter::ut() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_UT");
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Vessel> SpaceCenter::vessels() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_Vessels");
  std::vector<SpaceCenter::Vessel> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::warp_factor() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_WarpFactor");
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::WarpMode SpaceCenter::warp_mode() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_WarpMode");
  SpaceCenter::WarpMode _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::warp_rate() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_WarpRate");
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::WaypointManager SpaceCenter::waypoint_manager() {
  std::string _data = this->_client->invoke("SpaceCenter", "get_WaypointManager");
  SpaceCenter::WaypointManager _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<bool> SpaceCenter::can_rails_warp_at_stream(int32_t factor = 1) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(factor));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CanRailsWarpAt", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::can_revert_to_launch_stream() {
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CanRevertToLaunch"));
}

inline ::krpc::Stream<SpaceCenter::CrewMember> SpaceCenter::get_kerbal_stream(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  return ::krpc::Stream<SpaceCenter::CrewMember>(this->_client, this->_client->build_call("SpaceCenter", "GetKerbal", _args));
}

inline ::krpc::Stream<std::vector<std::string>> SpaceCenter::launchable_vessels_stream(std::string craft_directory) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(craft_directory));
  return ::krpc::Stream<std::vector<std::string>>(this->_client, this->_client->build_call("SpaceCenter", "LaunchableVessels", _args));
}

inline ::krpc::Stream<double> SpaceCenter::raycast_distance_stream(std::tuple<double, double, double> position, std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(direction));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "RaycastDistance", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::raycast_part_stream(std::tuple<double, double, double> position, std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(direction));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "RaycastPart", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::transform_direction_stream(std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(direction));
  _args.push_back(encoder::encode(from));
  _args.push_back(encoder::encode(to));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "TransformDirection", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::transform_position_stream(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(from));
  _args.push_back(encoder::encode(to));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "TransformPosition", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double, double>> SpaceCenter::transform_rotation_stream(std::tuple<double, double, double, double> rotation, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(rotation));
  _args.push_back(encoder::encode(from));
  _args.push_back(encoder::encode(to));
  return ::krpc::Stream<std::tuple<double, double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "TransformRotation", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::transform_velocity_stream(std::tuple<double, double, double> position, std::tuple<double, double, double> velocity, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(velocity));
  _args.push_back(encoder::encode(from));
  _args.push_back(encoder::encode(to));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "TransformVelocity", _args));
}

inline ::krpc::Stream<SpaceCenter::Vessel> SpaceCenter::active_vessel_stream() {
  return ::krpc::Stream<SpaceCenter::Vessel>(this->_client, this->_client->build_call("SpaceCenter", "get_ActiveVessel"));
}

inline ::krpc::Stream<SpaceCenter::AlarmManager> SpaceCenter::alarm_manager_stream() {
  return ::krpc::Stream<SpaceCenter::AlarmManager>(this->_client, this->_client->build_call("SpaceCenter", "get_AlarmManager"));
}

inline ::krpc::Stream<std::map<std::string, SpaceCenter::CelestialBody>> SpaceCenter::bodies_stream() {
  return ::krpc::Stream<std::map<std::string, SpaceCenter::CelestialBody>>(this->_client, this->_client->build_call("SpaceCenter", "get_Bodies"));
}

inline ::krpc::Stream<SpaceCenter::Camera> SpaceCenter::camera_stream() {
  return ::krpc::Stream<SpaceCenter::Camera>(this->_client, this->_client->build_call("SpaceCenter", "get_Camera"));
}

inline ::krpc::Stream<SpaceCenter::ContractManager> SpaceCenter::contract_manager_stream() {
  return ::krpc::Stream<SpaceCenter::ContractManager>(this->_client, this->_client->build_call("SpaceCenter", "get_ContractManager"));
}

inline ::krpc::Stream<bool> SpaceCenter::far_available_stream() {
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "get_FARAvailable"));
}

inline ::krpc::Stream<double> SpaceCenter::funds_stream() {
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "get_Funds"));
}

inline ::krpc::Stream<double> SpaceCenter::g_stream() {
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "get_G"));
}

inline ::krpc::Stream<SpaceCenter::GameMode> SpaceCenter::game_mode_stream() {
  return ::krpc::Stream<SpaceCenter::GameMode>(this->_client, this->_client->build_call("SpaceCenter", "get_GameMode"));
}

inline ::krpc::Stream<std::vector<SpaceCenter::LaunchSite>> SpaceCenter::launch_sites_stream() {
  return ::krpc::Stream<std::vector<SpaceCenter::LaunchSite>>(this->_client, this->_client->build_call("SpaceCenter", "get_LaunchSites"));
}

inline ::krpc::Stream<SpaceCenter::MapFilterType> SpaceCenter::map_filter_stream() {
  return ::krpc::Stream<SpaceCenter::MapFilterType>(this->_client, this->_client->build_call("SpaceCenter", "get_MapFilter"));
}

inline ::krpc::Stream<int32_t> SpaceCenter::maximum_rails_warp_factor_stream() {
  return ::krpc::Stream<int32_t>(this->_client, this->_client->build_call("SpaceCenter", "get_MaximumRailsWarpFactor"));
}

inline ::krpc::Stream<bool> SpaceCenter::navball_stream() {
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "get_Navball"));
}

inline ::krpc::Stream<int32_t> SpaceCenter::physics_warp_factor_stream() {
  return ::krpc::Stream<int32_t>(this->_client, this->_client->build_call("SpaceCenter", "get_PhysicsWarpFactor"));
}

inline ::krpc::Stream<int32_t> SpaceCenter::rails_warp_factor_stream() {
  return ::krpc::Stream<int32_t>(this->_client, this->_client->build_call("SpaceCenter", "get_RailsWarpFactor"));
}

inline ::krpc::Stream<float> SpaceCenter::reputation_stream() {
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "get_Reputation"));
}

inline ::krpc::Stream<float> SpaceCenter::science_stream() {
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "get_Science"));
}

inline ::krpc::Stream<SpaceCenter::CelestialBody> SpaceCenter::target_body_stream() {
  return ::krpc::Stream<SpaceCenter::CelestialBody>(this->_client, this->_client->build_call("SpaceCenter", "get_TargetBody"));
}

inline ::krpc::Stream<SpaceCenter::DockingPort> SpaceCenter::target_docking_port_stream() {
  return ::krpc::Stream<SpaceCenter::DockingPort>(this->_client, this->_client->build_call("SpaceCenter", "get_TargetDockingPort"));
}

inline ::krpc::Stream<SpaceCenter::Vessel> SpaceCenter::target_vessel_stream() {
  return ::krpc::Stream<SpaceCenter::Vessel>(this->_client, this->_client->build_call("SpaceCenter", "get_TargetVessel"));
}

inline ::krpc::Stream<bool> SpaceCenter::ui_visible_stream() {
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "get_UIVisible"));
}

inline ::krpc::Stream<double> SpaceCenter::ut_stream() {
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "get_UT"));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Vessel>> SpaceCenter::vessels_stream() {
  return ::krpc::Stream<std::vector<SpaceCenter::Vessel>>(this->_client, this->_client->build_call("SpaceCenter", "get_Vessels"));
}

inline ::krpc::Stream<float> SpaceCenter::warp_factor_stream() {
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "get_WarpFactor"));
}

inline ::krpc::Stream<SpaceCenter::WarpMode> SpaceCenter::warp_mode_stream() {
  return ::krpc::Stream<SpaceCenter::WarpMode>(this->_client, this->_client->build_call("SpaceCenter", "get_WarpMode"));
}

inline ::krpc::Stream<float> SpaceCenter::warp_rate_stream() {
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "get_WarpRate"));
}

inline ::krpc::Stream<SpaceCenter::WaypointManager> SpaceCenter::waypoint_manager_stream() {
  return ::krpc::Stream<SpaceCenter::WaypointManager>(this->_client, this->_client->build_call("SpaceCenter", "get_WaypointManager"));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::can_rails_warp_at_call(int32_t factor = 1) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(factor));
  return this->_client->build_call("SpaceCenter", "CanRailsWarpAt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::can_revert_to_launch_call() {
  return this->_client->build_call("SpaceCenter", "CanRevertToLaunch");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::clear_target_call() {
  return this->_client->build_call("SpaceCenter", "ClearTarget");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::create_kerbal_call(std::string name, std::string job, bool male) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(job));
  _args.push_back(encoder::encode(male));
  return this->_client->build_call("SpaceCenter", "CreateKerbal", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::get_kerbal_call(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  return this->_client->build_call("SpaceCenter", "GetKerbal", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::launch_vessel_call(std::string craft_directory, std::string name, std::string launch_site, bool recover = true, std::vector<std::string> crew = std::vector<std::string>(), std::string flag_url = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(craft_directory));
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(launch_site));
  _args.push_back(encoder::encode(recover));
  _args.push_back(encoder::encode(crew));
  _args.push_back(encoder::encode(flag_url));
  return this->_client->build_call("SpaceCenter", "LaunchVessel", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::launch_vessel_from_sph_call(std::string name, bool recover = true) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(recover));
  return this->_client->build_call("SpaceCenter", "LaunchVesselFromSPH", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::launch_vessel_from_vab_call(std::string name, bool recover = true) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(recover));
  return this->_client->build_call("SpaceCenter", "LaunchVesselFromVAB", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::launchable_vessels_call(std::string craft_directory) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(craft_directory));
  return this->_client->build_call("SpaceCenter", "LaunchableVessels", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::load_call(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  return this->_client->build_call("SpaceCenter", "Load", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::load_space_center_call() {
  return this->_client->build_call("SpaceCenter", "LoadSpaceCenter");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::quickload_call() {
  return this->_client->build_call("SpaceCenter", "Quickload");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::quicksave_call() {
  return this->_client->build_call("SpaceCenter", "Quicksave");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::raycast_distance_call(std::tuple<double, double, double> position, std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(direction));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "RaycastDistance", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::raycast_part_call(std::tuple<double, double, double> position, std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(direction));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "RaycastPart", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::revert_to_launch_call() {
  return this->_client->build_call("SpaceCenter", "RevertToLaunch");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::save_call(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(name));
  return this->_client->build_call("SpaceCenter", "Save", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::screenshot_call(std::string file_path, int32_t scale = 1) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(file_path));
  _args.push_back(encoder::encode(scale));
  return this->_client->build_call("SpaceCenter", "Screenshot", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::transfer_crew_call(SpaceCenter::CrewMember crew_member, SpaceCenter::Part target_part) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(crew_member));
  _args.push_back(encoder::encode(target_part));
  return this->_client->build_call("SpaceCenter", "TransferCrew", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::transform_direction_call(std::tuple<double, double, double> direction, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(direction));
  _args.push_back(encoder::encode(from));
  _args.push_back(encoder::encode(to));
  return this->_client->build_call("SpaceCenter", "TransformDirection", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::transform_position_call(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(from));
  _args.push_back(encoder::encode(to));
  return this->_client->build_call("SpaceCenter", "TransformPosition", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::transform_rotation_call(std::tuple<double, double, double, double> rotation, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(rotation));
  _args.push_back(encoder::encode(from));
  _args.push_back(encoder::encode(to));
  return this->_client->build_call("SpaceCenter", "TransformRotation", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::transform_velocity_call(std::tuple<double, double, double> position, std::tuple<double, double, double> velocity, SpaceCenter::ReferenceFrame from, SpaceCenter::ReferenceFrame to) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(velocity));
  _args.push_back(encoder::encode(from));
  _args.push_back(encoder::encode(to));
  return this->_client->build_call("SpaceCenter", "TransformVelocity", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::warp_to_call(double ut, float max_rails_rate = 100000.0, float max_physics_rate = 2.0) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(ut));
  _args.push_back(encoder::encode(max_rails_rate));
  _args.push_back(encoder::encode(max_physics_rate));
  return this->_client->build_call("SpaceCenter", "WarpTo", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::active_vessel_call() {
  return this->_client->build_call("SpaceCenter", "get_ActiveVessel");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::set_active_vessel_call(SpaceCenter::Vessel value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "set_ActiveVessel", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::alarm_manager_call() {
  return this->_client->build_call("SpaceCenter", "get_AlarmManager");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::bodies_call() {
  return this->_client->build_call("SpaceCenter", "get_Bodies");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::camera_call() {
  return this->_client->build_call("SpaceCenter", "get_Camera");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::contract_manager_call() {
  return this->_client->build_call("SpaceCenter", "get_ContractManager");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::far_available_call() {
  return this->_client->build_call("SpaceCenter", "get_FARAvailable");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::funds_call() {
  return this->_client->build_call("SpaceCenter", "get_Funds");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::g_call() {
  return this->_client->build_call("SpaceCenter", "get_G");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::game_mode_call() {
  return this->_client->build_call("SpaceCenter", "get_GameMode");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::launch_sites_call() {
  return this->_client->build_call("SpaceCenter", "get_LaunchSites");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::map_filter_call() {
  return this->_client->build_call("SpaceCenter", "get_MapFilter");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::set_map_filter_call(SpaceCenter::MapFilterType value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "set_MapFilter", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::maximum_rails_warp_factor_call() {
  return this->_client->build_call("SpaceCenter", "get_MaximumRailsWarpFactor");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::navball_call() {
  return this->_client->build_call("SpaceCenter", "get_Navball");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::set_navball_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "set_Navball", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::physics_warp_factor_call() {
  return this->_client->build_call("SpaceCenter", "get_PhysicsWarpFactor");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::set_physics_warp_factor_call(int32_t value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "set_PhysicsWarpFactor", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::rails_warp_factor_call() {
  return this->_client->build_call("SpaceCenter", "get_RailsWarpFactor");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::set_rails_warp_factor_call(int32_t value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "set_RailsWarpFactor", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::reputation_call() {
  return this->_client->build_call("SpaceCenter", "get_Reputation");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::science_call() {
  return this->_client->build_call("SpaceCenter", "get_Science");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::target_body_call() {
  return this->_client->build_call("SpaceCenter", "get_TargetBody");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::set_target_body_call(SpaceCenter::CelestialBody value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "set_TargetBody", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::target_docking_port_call() {
  return this->_client->build_call("SpaceCenter", "get_TargetDockingPort");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::set_target_docking_port_call(SpaceCenter::DockingPort value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "set_TargetDockingPort", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::target_vessel_call() {
  return this->_client->build_call("SpaceCenter", "get_TargetVessel");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::set_target_vessel_call(SpaceCenter::Vessel value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "set_TargetVessel", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ui_visible_call() {
  return this->_client->build_call("SpaceCenter", "get_UIVisible");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::set_ui_visible_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "set_UIVisible", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ut_call() {
  return this->_client->build_call("SpaceCenter", "get_UT");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::vessels_call() {
  return this->_client->build_call("SpaceCenter", "get_Vessels");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::warp_factor_call() {
  return this->_client->build_call("SpaceCenter", "get_WarpFactor");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::warp_mode_call() {
  return this->_client->build_call("SpaceCenter", "get_WarpMode");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::warp_rate_call() {
  return this->_client->build_call("SpaceCenter", "get_WarpRate");
}

inline ::krpc::schema::ProcedureCall SpaceCenter::waypoint_manager_call() {
  return this->_client->build_call("SpaceCenter", "get_WaypointManager");
}

inline SpaceCenter::Alarm::Alarm(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Alarm", id) {}

inline std::string SpaceCenter::Alarm::description() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Alarm_get_Description", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Alarm::event_offset() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Alarm_get_EventOffset", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline uint32_t SpaceCenter::Alarm::id() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Alarm_get_ID", _args);
  uint32_t _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Alarm::time() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Alarm_get_Time", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Alarm::time_until() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Alarm_get_TimeUntil", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Alarm::title() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Alarm_get_Title", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Alarm::type() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Alarm_get_Type", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Vessel SpaceCenter::Alarm::vessel() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Alarm_get_Vessel", _args);
  SpaceCenter::Vessel _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<std::string> SpaceCenter::Alarm::description_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Alarm_get_Description", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Alarm::event_offset_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Alarm_get_EventOffset", _args));
}

inline ::krpc::Stream<uint32_t> SpaceCenter::Alarm::id_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<uint32_t>(this->_client, this->_client->build_call("SpaceCenter", "Alarm_get_ID", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Alarm::time_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Alarm_get_Time", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Alarm::time_until_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Alarm_get_TimeUntil", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Alarm::title_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Alarm_get_Title", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Alarm::type_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Alarm_get_Type", _args));
}

inline ::krpc::Stream<SpaceCenter::Vessel> SpaceCenter::Alarm::vessel_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Vessel>(this->_client, this->_client->build_call("SpaceCenter", "Alarm_get_Vessel", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Alarm::description_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Alarm_get_Description", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Alarm::event_offset_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Alarm_get_EventOffset", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Alarm::id_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Alarm_get_ID", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Alarm::time_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Alarm_get_Time", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Alarm::time_until_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Alarm_get_TimeUntil", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Alarm::title_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Alarm_get_Title", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Alarm::type_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Alarm_get_Type", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Alarm::vessel_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Alarm_get_Vessel", _args);
}

inline SpaceCenter::AlarmManager::AlarmManager(Client* client, uint64_t id):
  Object(client, "SpaceCenter::AlarmManager", id) {}

inline std::vector<SpaceCenter::Alarm> SpaceCenter::AlarmManager::alarms() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AlarmManager_get_Alarms", _args);
  std::vector<SpaceCenter::Alarm> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Alarm SpaceCenter::AlarmManager::add_alarm(Client& _client, double time, std::string title = "Alarm", std::string description = "") {
    std::vector<std::string> _args;
  _args.push_back(encoder::encode(time));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  std::string _data = _client.invoke("SpaceCenter", "AlarmManager_static_AddAlarm", _args);
    SpaceCenter::Alarm _result;
  decoder::decode(_result, _data, &_client);
  return _result;
}

inline SpaceCenter::Alarm SpaceCenter::AlarmManager::add_apoapsis_alarm(Client& _client, SpaceCenter::Vessel vessel, double offset = 60.0, std::string title = "Apoapsis Alarm", std::string description = "") {
    std::vector<std::string> _args;
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(offset));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  std::string _data = _client.invoke("SpaceCenter", "AlarmManager_static_AddApoapsisAlarm", _args);
    SpaceCenter::Alarm _result;
  decoder::decode(_result, _data, &_client);
  return _result;
}

inline SpaceCenter::Alarm SpaceCenter::AlarmManager::add_maneuver_node_alarm(Client& _client, SpaceCenter::Vessel vessel, SpaceCenter::Node node, double offset = 60.0, bool add_burn_time = true, std::string title = "Maneuver Node Alarm", std::string description = "") {
    std::vector<std::string> _args;
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(node));
  _args.push_back(encoder::encode(offset));
  _args.push_back(encoder::encode(add_burn_time));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  std::string _data = _client.invoke("SpaceCenter", "AlarmManager_static_AddManeuverNodeAlarm", _args);
    SpaceCenter::Alarm _result;
  decoder::decode(_result, _data, &_client);
  return _result;
}

inline SpaceCenter::Alarm SpaceCenter::AlarmManager::add_periapsis_alarm(Client& _client, SpaceCenter::Vessel vessel, double offset = 60.0, std::string title = "Periapsis Alarm", std::string description = "") {
    std::vector<std::string> _args;
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(offset));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  std::string _data = _client.invoke("SpaceCenter", "AlarmManager_static_AddPeriapsisAlarm", _args);
    SpaceCenter::Alarm _result;
  decoder::decode(_result, _data, &_client);
  return _result;
}

inline SpaceCenter::Alarm SpaceCenter::AlarmManager::add_soi_alarm(Client& _client, SpaceCenter::Vessel vessel, double offset = 60.0, std::string title = "SOI Change Alarm", std::string description = "") {
    std::vector<std::string> _args;
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(offset));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  std::string _data = _client.invoke("SpaceCenter", "AlarmManager_static_AddSOIAlarm", _args);
    SpaceCenter::Alarm _result;
  decoder::decode(_result, _data, &_client);
  return _result;
}

inline SpaceCenter::Alarm SpaceCenter::AlarmManager::add_vessel_alarm(Client& _client, double time, SpaceCenter::Vessel vessel, std::string title = "Vessel Alarm", std::string description = "") {
    std::vector<std::string> _args;
  _args.push_back(encoder::encode(time));
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  std::string _data = _client.invoke("SpaceCenter", "AlarmManager_static_AddVesselAlarm", _args);
    SpaceCenter::Alarm _result;
  decoder::decode(_result, _data, &_client);
  return _result;
}

inline ::krpc::Stream<std::vector<SpaceCenter::Alarm>> SpaceCenter::AlarmManager::alarms_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Alarm>>(this->_client, this->_client->build_call("SpaceCenter", "AlarmManager_get_Alarms", _args));
}

inline ::krpc::Stream<SpaceCenter::Alarm> SpaceCenter::AlarmManager::add_alarm_stream(Client& _client, double time, std::string title = "Alarm", std::string description = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(time));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  return ::krpc::Stream<SpaceCenter::Alarm>(&_client, _client.build_call("SpaceCenter", "AlarmManager_static_AddAlarm", _args));
}

inline ::krpc::Stream<SpaceCenter::Alarm> SpaceCenter::AlarmManager::add_apoapsis_alarm_stream(Client& _client, SpaceCenter::Vessel vessel, double offset = 60.0, std::string title = "Apoapsis Alarm", std::string description = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(offset));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  return ::krpc::Stream<SpaceCenter::Alarm>(&_client, _client.build_call("SpaceCenter", "AlarmManager_static_AddApoapsisAlarm", _args));
}

inline ::krpc::Stream<SpaceCenter::Alarm> SpaceCenter::AlarmManager::add_maneuver_node_alarm_stream(Client& _client, SpaceCenter::Vessel vessel, SpaceCenter::Node node, double offset = 60.0, bool add_burn_time = true, std::string title = "Maneuver Node Alarm", std::string description = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(node));
  _args.push_back(encoder::encode(offset));
  _args.push_back(encoder::encode(add_burn_time));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  return ::krpc::Stream<SpaceCenter::Alarm>(&_client, _client.build_call("SpaceCenter", "AlarmManager_static_AddManeuverNodeAlarm", _args));
}

inline ::krpc::Stream<SpaceCenter::Alarm> SpaceCenter::AlarmManager::add_periapsis_alarm_stream(Client& _client, SpaceCenter::Vessel vessel, double offset = 60.0, std::string title = "Periapsis Alarm", std::string description = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(offset));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  return ::krpc::Stream<SpaceCenter::Alarm>(&_client, _client.build_call("SpaceCenter", "AlarmManager_static_AddPeriapsisAlarm", _args));
}

inline ::krpc::Stream<SpaceCenter::Alarm> SpaceCenter::AlarmManager::add_soi_alarm_stream(Client& _client, SpaceCenter::Vessel vessel, double offset = 60.0, std::string title = "SOI Change Alarm", std::string description = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(offset));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  return ::krpc::Stream<SpaceCenter::Alarm>(&_client, _client.build_call("SpaceCenter", "AlarmManager_static_AddSOIAlarm", _args));
}

inline ::krpc::Stream<SpaceCenter::Alarm> SpaceCenter::AlarmManager::add_vessel_alarm_stream(Client& _client, double time, SpaceCenter::Vessel vessel, std::string title = "Vessel Alarm", std::string description = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(time));
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  return ::krpc::Stream<SpaceCenter::Alarm>(&_client, _client.build_call("SpaceCenter", "AlarmManager_static_AddVesselAlarm", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AlarmManager::alarms_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AlarmManager_get_Alarms", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AlarmManager::add_alarm_call(Client& _client, double time, std::string title = "Alarm", std::string description = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(time));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  return _client.build_call("SpaceCenter", "AlarmManager_static_AddAlarm", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AlarmManager::add_apoapsis_alarm_call(Client& _client, SpaceCenter::Vessel vessel, double offset = 60.0, std::string title = "Apoapsis Alarm", std::string description = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(offset));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  return _client.build_call("SpaceCenter", "AlarmManager_static_AddApoapsisAlarm", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AlarmManager::add_maneuver_node_alarm_call(Client& _client, SpaceCenter::Vessel vessel, SpaceCenter::Node node, double offset = 60.0, bool add_burn_time = true, std::string title = "Maneuver Node Alarm", std::string description = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(node));
  _args.push_back(encoder::encode(offset));
  _args.push_back(encoder::encode(add_burn_time));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  return _client.build_call("SpaceCenter", "AlarmManager_static_AddManeuverNodeAlarm", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AlarmManager::add_periapsis_alarm_call(Client& _client, SpaceCenter::Vessel vessel, double offset = 60.0, std::string title = "Periapsis Alarm", std::string description = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(offset));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  return _client.build_call("SpaceCenter", "AlarmManager_static_AddPeriapsisAlarm", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AlarmManager::add_soi_alarm_call(Client& _client, SpaceCenter::Vessel vessel, double offset = 60.0, std::string title = "SOI Change Alarm", std::string description = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(offset));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  return _client.build_call("SpaceCenter", "AlarmManager_static_AddSOIAlarm", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AlarmManager::add_vessel_alarm_call(Client& _client, double time, SpaceCenter::Vessel vessel, std::string title = "Vessel Alarm", std::string description = "") {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(time));
  _args.push_back(encoder::encode(vessel));
  _args.push_back(encoder::encode(title));
  _args.push_back(encoder::encode(description));
  return _client.build_call("SpaceCenter", "AlarmManager_static_AddVesselAlarm", _args);
}

inline SpaceCenter::Antenna::Antenna(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Antenna", id) {}

inline void SpaceCenter::Antenna::cancel() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Antenna_Cancel", _args);
}

inline void SpaceCenter::Antenna::transmit() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Antenna_Transmit", _args);
}

inline bool SpaceCenter::Antenna::allow_partial() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Antenna_get_AllowPartial", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Antenna::set_allow_partial(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Antenna_set_AllowPartial", _args);
}

inline bool SpaceCenter::Antenna::can_transmit() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Antenna_get_CanTransmit", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Antenna::combinable() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Antenna_get_Combinable", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Antenna::combinable_exponent() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Antenna_get_CombinableExponent", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Antenna::deployable() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Antenna_get_Deployable", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Antenna::deployed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Antenna_get_Deployed", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Antenna::set_deployed(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Antenna_set_Deployed", _args);
}

inline float SpaceCenter::Antenna::packet_interval() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Antenna_get_PacketInterval", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Antenna::packet_resource_cost() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Antenna_get_PacketResourceCost", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Antenna::packet_size() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Antenna_get_PacketSize", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::Antenna::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Antenna_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Antenna::power() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Antenna_get_Power", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::AntennaState SpaceCenter::Antenna::state() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Antenna_get_State", _args);
  SpaceCenter::AntennaState _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<bool> SpaceCenter::Antenna::allow_partial_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Antenna_get_AllowPartial", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Antenna::can_transmit_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Antenna_get_CanTransmit", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Antenna::combinable_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Antenna_get_Combinable", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Antenna::combinable_exponent_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Antenna_get_CombinableExponent", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Antenna::deployable_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Antenna_get_Deployable", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Antenna::deployed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Antenna_get_Deployed", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Antenna::packet_interval_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Antenna_get_PacketInterval", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Antenna::packet_resource_cost_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Antenna_get_PacketResourceCost", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Antenna::packet_size_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Antenna_get_PacketSize", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Antenna::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Antenna_get_Part", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Antenna::power_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Antenna_get_Power", _args));
}

inline ::krpc::Stream<SpaceCenter::AntennaState> SpaceCenter::Antenna::state_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::AntennaState>(this->_client, this->_client->build_call("SpaceCenter", "Antenna_get_State", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::cancel_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_Cancel", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::transmit_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_Transmit", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::allow_partial_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_get_AllowPartial", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::set_allow_partial_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Antenna_set_AllowPartial", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::can_transmit_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_get_CanTransmit", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::combinable_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_get_Combinable", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::combinable_exponent_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_get_CombinableExponent", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::deployable_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_get_Deployable", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::deployed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_get_Deployed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::set_deployed_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Antenna_set_Deployed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::packet_interval_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_get_PacketInterval", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::packet_resource_cost_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_get_PacketResourceCost", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::packet_size_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_get_PacketSize", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_get_Part", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::power_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_get_Power", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Antenna::state_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Antenna_get_State", _args);
}

inline SpaceCenter::AutoPilot::AutoPilot(Client* client, uint64_t id):
  Object(client, "SpaceCenter::AutoPilot", id) {}

inline void SpaceCenter::AutoPilot::disengage() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "AutoPilot_Disengage", _args);
}

inline void SpaceCenter::AutoPilot::engage() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "AutoPilot_Engage", _args);
}

inline void SpaceCenter::AutoPilot::target_pitch_and_heading(float pitch, float heading) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(pitch));
  _args.push_back(encoder::encode(heading));
  this->_client->invoke("SpaceCenter", "AutoPilot_TargetPitchAndHeading", _args);
}

inline void SpaceCenter::AutoPilot::wait() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "AutoPilot_Wait", _args);
}

inline std::tuple<double, double, double> SpaceCenter::AutoPilot::attenuation_angle() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_AttenuationAngle", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_attenuation_angle(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_AttenuationAngle", _args);
}

inline bool SpaceCenter::AutoPilot::auto_tune() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_AutoTune", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_auto_tune(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_AutoTune", _args);
}

inline std::tuple<double, double, double> SpaceCenter::AutoPilot::deceleration_time() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_DecelerationTime", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_deceleration_time(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_DecelerationTime", _args);
}

inline float SpaceCenter::AutoPilot::error() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_Error", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::AutoPilot::heading_error() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_HeadingError", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::AutoPilot::overshoot() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_Overshoot", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_overshoot(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_Overshoot", _args);
}

inline float SpaceCenter::AutoPilot::pitch_error() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_PitchError", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::AutoPilot::pitch_pid_gains() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_PitchPIDGains", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_pitch_pid_gains(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_PitchPIDGains", _args);
}

inline SpaceCenter::ReferenceFrame SpaceCenter::AutoPilot::reference_frame() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_ReferenceFrame", _args);
  SpaceCenter::ReferenceFrame _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_reference_frame(SpaceCenter::ReferenceFrame value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_ReferenceFrame", _args);
}

inline float SpaceCenter::AutoPilot::roll_error() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_RollError", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::AutoPilot::roll_pid_gains() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_RollPIDGains", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_roll_pid_gains(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_RollPIDGains", _args);
}

inline double SpaceCenter::AutoPilot::roll_threshold() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_RollThreshold", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_roll_threshold(double value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_RollThreshold", _args);
}

inline bool SpaceCenter::AutoPilot::sas() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_SAS", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_sas(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_SAS", _args);
}

inline SpaceCenter::SASMode SpaceCenter::AutoPilot::sas_mode() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_SASMode", _args);
  SpaceCenter::SASMode _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_sas_mode(SpaceCenter::SASMode value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_SASMode", _args);
}

inline std::tuple<double, double, double> SpaceCenter::AutoPilot::stopping_time() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_StoppingTime", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_stopping_time(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_StoppingTime", _args);
}

inline std::tuple<double, double, double> SpaceCenter::AutoPilot::target_direction() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_TargetDirection", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_target_direction(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_TargetDirection", _args);
}

inline float SpaceCenter::AutoPilot::target_heading() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_TargetHeading", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_target_heading(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_TargetHeading", _args);
}

inline float SpaceCenter::AutoPilot::target_pitch() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_TargetPitch", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_target_pitch(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_TargetPitch", _args);
}

inline float SpaceCenter::AutoPilot::target_roll() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_TargetRoll", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_target_roll(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_TargetRoll", _args);
}

inline std::tuple<double, double, double> SpaceCenter::AutoPilot::time_to_peak() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_TimeToPeak", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_time_to_peak(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_TimeToPeak", _args);
}

inline std::tuple<double, double, double> SpaceCenter::AutoPilot::yaw_pid_gains() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "AutoPilot_get_YawPIDGains", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::AutoPilot::set_yaw_pid_gains(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "AutoPilot_set_YawPIDGains", _args);
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::AutoPilot::attenuation_angle_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_AttenuationAngle", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::AutoPilot::auto_tune_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_AutoTune", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::AutoPilot::deceleration_time_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_DecelerationTime", _args));
}

inline ::krpc::Stream<float> SpaceCenter::AutoPilot::error_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_Error", _args));
}

inline ::krpc::Stream<float> SpaceCenter::AutoPilot::heading_error_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_HeadingError", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::AutoPilot::overshoot_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_Overshoot", _args));
}

inline ::krpc::Stream<float> SpaceCenter::AutoPilot::pitch_error_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_PitchError", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::AutoPilot::pitch_pid_gains_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_PitchPIDGains", _args));
}

inline ::krpc::Stream<SpaceCenter::ReferenceFrame> SpaceCenter::AutoPilot::reference_frame_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ReferenceFrame>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_ReferenceFrame", _args));
}

inline ::krpc::Stream<float> SpaceCenter::AutoPilot::roll_error_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_RollError", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::AutoPilot::roll_pid_gains_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_RollPIDGains", _args));
}

inline ::krpc::Stream<double> SpaceCenter::AutoPilot::roll_threshold_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_RollThreshold", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::AutoPilot::sas_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_SAS", _args));
}

inline ::krpc::Stream<SpaceCenter::SASMode> SpaceCenter::AutoPilot::sas_mode_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::SASMode>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_SASMode", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::AutoPilot::stopping_time_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_StoppingTime", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::AutoPilot::target_direction_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_TargetDirection", _args));
}

inline ::krpc::Stream<float> SpaceCenter::AutoPilot::target_heading_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_TargetHeading", _args));
}

inline ::krpc::Stream<float> SpaceCenter::AutoPilot::target_pitch_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_TargetPitch", _args));
}

inline ::krpc::Stream<float> SpaceCenter::AutoPilot::target_roll_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_TargetRoll", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::AutoPilot::time_to_peak_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_TimeToPeak", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::AutoPilot::yaw_pid_gains_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "AutoPilot_get_YawPIDGains", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::disengage_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_Disengage", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::engage_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_Engage", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::target_pitch_and_heading_call(float pitch, float heading) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(pitch));
  _args.push_back(encoder::encode(heading));
  return this->_client->build_call("SpaceCenter", "AutoPilot_TargetPitchAndHeading", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::wait_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_Wait", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::attenuation_angle_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_AttenuationAngle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_attenuation_angle_call(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_AttenuationAngle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::auto_tune_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_AutoTune", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_auto_tune_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_AutoTune", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::deceleration_time_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_DecelerationTime", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_deceleration_time_call(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_DecelerationTime", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::error_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_Error", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::heading_error_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_HeadingError", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::overshoot_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_Overshoot", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_overshoot_call(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_Overshoot", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::pitch_error_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_PitchError", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::pitch_pid_gains_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_PitchPIDGains", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_pitch_pid_gains_call(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_PitchPIDGains", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::reference_frame_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_ReferenceFrame", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_reference_frame_call(SpaceCenter::ReferenceFrame value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_ReferenceFrame", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::roll_error_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_RollError", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::roll_pid_gains_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_RollPIDGains", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_roll_pid_gains_call(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_RollPIDGains", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::roll_threshold_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_RollThreshold", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_roll_threshold_call(double value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_RollThreshold", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::sas_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_SAS", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_sas_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_SAS", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::sas_mode_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_SASMode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_sas_mode_call(SpaceCenter::SASMode value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_SASMode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::stopping_time_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_StoppingTime", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_stopping_time_call(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_StoppingTime", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::target_direction_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_TargetDirection", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_target_direction_call(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_TargetDirection", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::target_heading_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_TargetHeading", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_target_heading_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_TargetHeading", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::target_pitch_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_TargetPitch", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_target_pitch_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_TargetPitch", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::target_roll_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_TargetRoll", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_target_roll_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_TargetRoll", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::time_to_peak_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_TimeToPeak", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_time_to_peak_call(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_TimeToPeak", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::yaw_pid_gains_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "AutoPilot_get_YawPIDGains", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::AutoPilot::set_yaw_pid_gains_call(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "AutoPilot_set_YawPIDGains", _args);
}

inline SpaceCenter::Camera::Camera(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Camera", id) {}

inline float SpaceCenter::Camera::default_distance() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Camera_get_DefaultDistance", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Camera::distance() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Camera_get_Distance", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Camera::set_distance(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Camera_set_Distance", _args);
}

inline SpaceCenter::CelestialBody SpaceCenter::Camera::focussed_body() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Camera_get_FocussedBody", _args);
  SpaceCenter::CelestialBody _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Camera::set_focussed_body(SpaceCenter::CelestialBody value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Camera_set_FocussedBody", _args);
}

inline SpaceCenter::Node SpaceCenter::Camera::focussed_node() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Camera_get_FocussedNode", _args);
  SpaceCenter::Node _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Camera::set_focussed_node(SpaceCenter::Node value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Camera_set_FocussedNode", _args);
}

inline SpaceCenter::Vessel SpaceCenter::Camera::focussed_vessel() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Camera_get_FocussedVessel", _args);
  SpaceCenter::Vessel _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Camera::set_focussed_vessel(SpaceCenter::Vessel value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Camera_set_FocussedVessel", _args);
}

inline float SpaceCenter::Camera::heading() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Camera_get_Heading", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Camera::set_heading(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Camera_set_Heading", _args);
}

inline float SpaceCenter::Camera::max_distance() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Camera_get_MaxDistance", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Camera::max_pitch() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Camera_get_MaxPitch", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Camera::min_distance() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Camera_get_MinDistance", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Camera::min_pitch() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Camera_get_MinPitch", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::CameraMode SpaceCenter::Camera::mode() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Camera_get_Mode", _args);
  SpaceCenter::CameraMode _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Camera::set_mode(SpaceCenter::CameraMode value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Camera_set_Mode", _args);
}

inline float SpaceCenter::Camera::pitch() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Camera_get_Pitch", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Camera::set_pitch(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Camera_set_Pitch", _args);
}

inline ::krpc::Stream<float> SpaceCenter::Camera::default_distance_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Camera_get_DefaultDistance", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Camera::distance_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Camera_get_Distance", _args));
}

inline ::krpc::Stream<SpaceCenter::CelestialBody> SpaceCenter::Camera::focussed_body_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::CelestialBody>(this->_client, this->_client->build_call("SpaceCenter", "Camera_get_FocussedBody", _args));
}

inline ::krpc::Stream<SpaceCenter::Node> SpaceCenter::Camera::focussed_node_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Node>(this->_client, this->_client->build_call("SpaceCenter", "Camera_get_FocussedNode", _args));
}

inline ::krpc::Stream<SpaceCenter::Vessel> SpaceCenter::Camera::focussed_vessel_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Vessel>(this->_client, this->_client->build_call("SpaceCenter", "Camera_get_FocussedVessel", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Camera::heading_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Camera_get_Heading", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Camera::max_distance_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Camera_get_MaxDistance", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Camera::max_pitch_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Camera_get_MaxPitch", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Camera::min_distance_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Camera_get_MinDistance", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Camera::min_pitch_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Camera_get_MinPitch", _args));
}

inline ::krpc::Stream<SpaceCenter::CameraMode> SpaceCenter::Camera::mode_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::CameraMode>(this->_client, this->_client->build_call("SpaceCenter", "Camera_get_Mode", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Camera::pitch_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Camera_get_Pitch", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::default_distance_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Camera_get_DefaultDistance", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::distance_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Camera_get_Distance", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::set_distance_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Camera_set_Distance", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::focussed_body_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Camera_get_FocussedBody", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::set_focussed_body_call(SpaceCenter::CelestialBody value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Camera_set_FocussedBody", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::focussed_node_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Camera_get_FocussedNode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::set_focussed_node_call(SpaceCenter::Node value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Camera_set_FocussedNode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::focussed_vessel_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Camera_get_FocussedVessel", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::set_focussed_vessel_call(SpaceCenter::Vessel value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Camera_set_FocussedVessel", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::heading_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Camera_get_Heading", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::set_heading_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Camera_set_Heading", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::max_distance_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Camera_get_MaxDistance", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::max_pitch_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Camera_get_MaxPitch", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::min_distance_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Camera_get_MinDistance", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::min_pitch_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Camera_get_MinPitch", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::mode_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Camera_get_Mode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::set_mode_call(SpaceCenter::CameraMode value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Camera_set_Mode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::pitch_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Camera_get_Pitch", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Camera::set_pitch_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Camera_set_Pitch", _args);
}

inline SpaceCenter::CargoBay::CargoBay(Client* client, uint64_t id):
  Object(client, "SpaceCenter::CargoBay", id) {}

inline bool SpaceCenter::CargoBay::open() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CargoBay_get_Open", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::CargoBay::set_open(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "CargoBay_set_Open", _args);
}

inline SpaceCenter::Part SpaceCenter::CargoBay::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CargoBay_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::CargoBayState SpaceCenter::CargoBay::state() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CargoBay_get_State", _args);
  SpaceCenter::CargoBayState _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<bool> SpaceCenter::CargoBay::open_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CargoBay_get_Open", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::CargoBay::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "CargoBay_get_Part", _args));
}

inline ::krpc::Stream<SpaceCenter::CargoBayState> SpaceCenter::CargoBay::state_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::CargoBayState>(this->_client, this->_client->build_call("SpaceCenter", "CargoBay_get_State", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CargoBay::open_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CargoBay_get_Open", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CargoBay::set_open_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "CargoBay_set_Open", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CargoBay::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CargoBay_get_Part", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CargoBay::state_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CargoBay_get_State", _args);
}

inline SpaceCenter::CelestialBody::CelestialBody(Client* client, uint64_t id):
  Object(client, "SpaceCenter::CelestialBody", id) {}

inline double SpaceCenter::CelestialBody::altitude_at_position(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_AltitudeAtPosition", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::CelestialBody::angular_velocity(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_AngularVelocity", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::atmospheric_density_at_position(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_AtmosphericDensityAtPosition", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::bedrock_height(double latitude, double longitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_BedrockHeight", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::CelestialBody::bedrock_position(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_BedrockPosition", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::CelestialBody::biome_at(double latitude, double longitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_BiomeAt", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::density_at(double altitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(altitude));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_DensityAt", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::CelestialBody::direction(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_Direction", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::latitude_at_position(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_LatitudeAtPosition", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::longitude_at_position(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_LongitudeAtPosition", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::CelestialBody::msl_position(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_MSLPosition", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::CelestialBody::position(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_Position", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::CelestialBody::position_at_altitude(double latitude, double longitude, double altitude, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  _args.push_back(encoder::encode(altitude));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_PositionAtAltitude", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::pressure_at(double altitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(altitude));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_PressureAt", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double, double> SpaceCenter::CelestialBody::rotation(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_Rotation", _args);
  std::tuple<double, double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::surface_height(double latitude, double longitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_SurfaceHeight", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::CelestialBody::surface_position(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_SurfacePosition", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::temperature_at(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_TemperatureAt", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::CelestialBody::velocity(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_Velocity", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::atmosphere_depth() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_AtmosphereDepth", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::set<std::string> SpaceCenter::CelestialBody::biomes() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_Biomes", _args);
  std::set<std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::equatorial_radius() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_EquatorialRadius", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::CelestialBody::flying_high_altitude_threshold() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_FlyingHighAltitudeThreshold", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::gravitational_parameter() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_GravitationalParameter", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::CelestialBody::has_atmosphere() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_HasAtmosphere", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::CelestialBody::has_atmospheric_oxygen() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_HasAtmosphericOxygen", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::CelestialBody::has_solid_surface() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_HasSolidSurface", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::initial_rotation() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_InitialRotation", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::CelestialBody::is_star() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_IsStar", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::mass() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_Mass", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::CelestialBody::name() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_Name", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ReferenceFrame SpaceCenter::CelestialBody::non_rotating_reference_frame() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_NonRotatingReferenceFrame", _args);
  SpaceCenter::ReferenceFrame _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Orbit SpaceCenter::CelestialBody::orbit() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_Orbit", _args);
  SpaceCenter::Orbit _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ReferenceFrame SpaceCenter::CelestialBody::orbital_reference_frame() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_OrbitalReferenceFrame", _args);
  SpaceCenter::ReferenceFrame _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ReferenceFrame SpaceCenter::CelestialBody::reference_frame() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_ReferenceFrame", _args);
  SpaceCenter::ReferenceFrame _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::rotation_angle() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_RotationAngle", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::rotational_period() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_RotationalPeriod", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::rotational_speed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_RotationalSpeed", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::CelestialBody> SpaceCenter::CelestialBody::satellites() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_Satellites", _args);
  std::vector<SpaceCenter::CelestialBody> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::CelestialBody::space_high_altitude_threshold() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_SpaceHighAltitudeThreshold", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::sphere_of_influence() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_SphereOfInfluence", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CelestialBody::surface_gravity() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CelestialBody_get_SurfaceGravity", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::altitude_at_position_stream(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_AltitudeAtPosition", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::CelestialBody::angular_velocity_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_AngularVelocity", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::atmospheric_density_at_position_stream(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_AtmosphericDensityAtPosition", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::bedrock_height_stream(double latitude, double longitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_BedrockHeight", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::CelestialBody::bedrock_position_stream(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_BedrockPosition", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::CelestialBody::biome_at_stream(double latitude, double longitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_BiomeAt", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::density_at_stream(double altitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(altitude));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_DensityAt", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::CelestialBody::direction_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_Direction", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::latitude_at_position_stream(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_LatitudeAtPosition", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::longitude_at_position_stream(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_LongitudeAtPosition", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::CelestialBody::msl_position_stream(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_MSLPosition", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::CelestialBody::position_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_Position", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::CelestialBody::position_at_altitude_stream(double latitude, double longitude, double altitude, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  _args.push_back(encoder::encode(altitude));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_PositionAtAltitude", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::pressure_at_stream(double altitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(altitude));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_PressureAt", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double, double>> SpaceCenter::CelestialBody::rotation_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_Rotation", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::surface_height_stream(double latitude, double longitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_SurfaceHeight", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::CelestialBody::surface_position_stream(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_SurfacePosition", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::temperature_at_stream(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_TemperatureAt", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::CelestialBody::velocity_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_Velocity", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::atmosphere_depth_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_AtmosphereDepth", _args));
}

inline ::krpc::Stream<std::set<std::string>> SpaceCenter::CelestialBody::biomes_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::set<std::string>>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_Biomes", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::equatorial_radius_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_EquatorialRadius", _args));
}

inline ::krpc::Stream<float> SpaceCenter::CelestialBody::flying_high_altitude_threshold_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_FlyingHighAltitudeThreshold", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::gravitational_parameter_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_GravitationalParameter", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::CelestialBody::has_atmosphere_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_HasAtmosphere", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::CelestialBody::has_atmospheric_oxygen_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_HasAtmosphericOxygen", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::CelestialBody::has_solid_surface_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_HasSolidSurface", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::initial_rotation_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_InitialRotation", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::CelestialBody::is_star_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_IsStar", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::mass_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_Mass", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::CelestialBody::name_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_Name", _args));
}

inline ::krpc::Stream<SpaceCenter::ReferenceFrame> SpaceCenter::CelestialBody::non_rotating_reference_frame_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ReferenceFrame>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_NonRotatingReferenceFrame", _args));
}

inline ::krpc::Stream<SpaceCenter::Orbit> SpaceCenter::CelestialBody::orbit_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Orbit>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_Orbit", _args));
}

inline ::krpc::Stream<SpaceCenter::ReferenceFrame> SpaceCenter::CelestialBody::orbital_reference_frame_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ReferenceFrame>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_OrbitalReferenceFrame", _args));
}

inline ::krpc::Stream<SpaceCenter::ReferenceFrame> SpaceCenter::CelestialBody::reference_frame_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ReferenceFrame>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_ReferenceFrame", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::rotation_angle_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_RotationAngle", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::rotational_period_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_RotationalPeriod", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::rotational_speed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_RotationalSpeed", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::CelestialBody>> SpaceCenter::CelestialBody::satellites_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::CelestialBody>>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_Satellites", _args));
}

inline ::krpc::Stream<float> SpaceCenter::CelestialBody::space_high_altitude_threshold_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_SpaceHighAltitudeThreshold", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::sphere_of_influence_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_SphereOfInfluence", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CelestialBody::surface_gravity_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CelestialBody_get_SurfaceGravity", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::altitude_at_position_call(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_AltitudeAtPosition", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::angular_velocity_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_AngularVelocity", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::atmospheric_density_at_position_call(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_AtmosphericDensityAtPosition", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::bedrock_height_call(double latitude, double longitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  return this->_client->build_call("SpaceCenter", "CelestialBody_BedrockHeight", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::bedrock_position_call(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_BedrockPosition", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::biome_at_call(double latitude, double longitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  return this->_client->build_call("SpaceCenter", "CelestialBody_BiomeAt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::density_at_call(double altitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(altitude));
  return this->_client->build_call("SpaceCenter", "CelestialBody_DensityAt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::direction_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_Direction", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::latitude_at_position_call(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_LatitudeAtPosition", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::longitude_at_position_call(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_LongitudeAtPosition", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::msl_position_call(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_MSLPosition", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::position_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_Position", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::position_at_altitude_call(double latitude, double longitude, double altitude, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  _args.push_back(encoder::encode(altitude));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_PositionAtAltitude", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::pressure_at_call(double altitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(altitude));
  return this->_client->build_call("SpaceCenter", "CelestialBody_PressureAt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::rotation_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_Rotation", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::surface_height_call(double latitude, double longitude) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  return this->_client->build_call("SpaceCenter", "CelestialBody_SurfaceHeight", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::surface_position_call(double latitude, double longitude, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(latitude));
  _args.push_back(encoder::encode(longitude));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_SurfacePosition", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::temperature_at_call(std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_TemperatureAt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::velocity_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "CelestialBody_Velocity", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::atmosphere_depth_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_AtmosphereDepth", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::biomes_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_Biomes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::equatorial_radius_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_EquatorialRadius", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::flying_high_altitude_threshold_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_FlyingHighAltitudeThreshold", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::gravitational_parameter_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_GravitationalParameter", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::has_atmosphere_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_HasAtmosphere", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::has_atmospheric_oxygen_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_HasAtmosphericOxygen", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::has_solid_surface_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_HasSolidSurface", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::initial_rotation_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_InitialRotation", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::is_star_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_IsStar", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::mass_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_Mass", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::name_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_Name", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::non_rotating_reference_frame_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_NonRotatingReferenceFrame", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::orbit_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_Orbit", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::orbital_reference_frame_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_OrbitalReferenceFrame", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::reference_frame_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_ReferenceFrame", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::rotation_angle_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_RotationAngle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::rotational_period_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_RotationalPeriod", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::rotational_speed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_RotationalSpeed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::satellites_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_Satellites", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::space_high_altitude_threshold_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_SpaceHighAltitudeThreshold", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::sphere_of_influence_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_SphereOfInfluence", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CelestialBody::surface_gravity_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CelestialBody_get_SurfaceGravity", _args);
}

inline SpaceCenter::CommLink::CommLink(Client* client, uint64_t id):
  Object(client, "SpaceCenter::CommLink", id) {}

inline SpaceCenter::CommNode SpaceCenter::CommLink::end() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CommLink_get_End", _args);
  SpaceCenter::CommNode _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::CommLink::signal_strength() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CommLink_get_SignalStrength", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::CommNode SpaceCenter::CommLink::start() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CommLink_get_Start", _args);
  SpaceCenter::CommNode _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::CommLinkType SpaceCenter::CommLink::type() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CommLink_get_Type", _args);
  SpaceCenter::CommLinkType _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<SpaceCenter::CommNode> SpaceCenter::CommLink::end_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::CommNode>(this->_client, this->_client->build_call("SpaceCenter", "CommLink_get_End", _args));
}

inline ::krpc::Stream<double> SpaceCenter::CommLink::signal_strength_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "CommLink_get_SignalStrength", _args));
}

inline ::krpc::Stream<SpaceCenter::CommNode> SpaceCenter::CommLink::start_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::CommNode>(this->_client, this->_client->build_call("SpaceCenter", "CommLink_get_Start", _args));
}

inline ::krpc::Stream<SpaceCenter::CommLinkType> SpaceCenter::CommLink::type_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::CommLinkType>(this->_client, this->_client->build_call("SpaceCenter", "CommLink_get_Type", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CommLink::end_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CommLink_get_End", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CommLink::signal_strength_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CommLink_get_SignalStrength", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CommLink::start_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CommLink_get_Start", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CommLink::type_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CommLink_get_Type", _args);
}

inline SpaceCenter::CommNode::CommNode(Client* client, uint64_t id):
  Object(client, "SpaceCenter::CommNode", id) {}

inline bool SpaceCenter::CommNode::is_control_point() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CommNode_get_IsControlPoint", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::CommNode::is_home() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CommNode_get_IsHome", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::CommNode::is_vessel() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CommNode_get_IsVessel", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::CommNode::name() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CommNode_get_Name", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Vessel SpaceCenter::CommNode::vessel() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CommNode_get_Vessel", _args);
  SpaceCenter::Vessel _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<bool> SpaceCenter::CommNode::is_control_point_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CommNode_get_IsControlPoint", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::CommNode::is_home_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CommNode_get_IsHome", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::CommNode::is_vessel_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CommNode_get_IsVessel", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::CommNode::name_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "CommNode_get_Name", _args));
}

inline ::krpc::Stream<SpaceCenter::Vessel> SpaceCenter::CommNode::vessel_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Vessel>(this->_client, this->_client->build_call("SpaceCenter", "CommNode_get_Vessel", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CommNode::is_control_point_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CommNode_get_IsControlPoint", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CommNode::is_home_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CommNode_get_IsHome", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CommNode::is_vessel_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CommNode_get_IsVessel", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CommNode::name_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CommNode_get_Name", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CommNode::vessel_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CommNode_get_Vessel", _args);
}

inline SpaceCenter::Comms::Comms(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Comms", id) {}

inline bool SpaceCenter::Comms::can_communicate() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Comms_get_CanCommunicate", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Comms::can_transmit_science() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Comms_get_CanTransmitScience", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::CommLink> SpaceCenter::Comms::control_path() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Comms_get_ControlPath", _args);
  std::vector<SpaceCenter::CommLink> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Comms::power() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Comms_get_Power", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Comms::signal_delay() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Comms_get_SignalDelay", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Comms::signal_strength() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Comms_get_SignalStrength", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<bool> SpaceCenter::Comms::can_communicate_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Comms_get_CanCommunicate", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Comms::can_transmit_science_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Comms_get_CanTransmitScience", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::CommLink>> SpaceCenter::Comms::control_path_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::CommLink>>(this->_client, this->_client->build_call("SpaceCenter", "Comms_get_ControlPath", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Comms::power_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Comms_get_Power", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Comms::signal_delay_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Comms_get_SignalDelay", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Comms::signal_strength_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Comms_get_SignalStrength", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Comms::can_communicate_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Comms_get_CanCommunicate", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Comms::can_transmit_science_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Comms_get_CanTransmitScience", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Comms::control_path_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Comms_get_ControlPath", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Comms::power_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Comms_get_Power", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Comms::signal_delay_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Comms_get_SignalDelay", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Comms::signal_strength_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Comms_get_SignalStrength", _args);
}

inline SpaceCenter::Contract::Contract(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Contract", id) {}

inline void SpaceCenter::Contract::accept() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Contract_Accept", _args);
}

inline void SpaceCenter::Contract::cancel() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Contract_Cancel", _args);
}

inline void SpaceCenter::Contract::decline() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Contract_Decline", _args);
}

inline bool SpaceCenter::Contract::active() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_Active", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Contract::can_be_canceled() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_CanBeCanceled", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Contract::can_be_declined() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_CanBeDeclined", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Contract::can_be_failed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_CanBeFailed", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Contract::description() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_Description", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Contract::failed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_Failed", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Contract::funds_advance() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_FundsAdvance", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Contract::funds_completion() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_FundsCompletion", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Contract::funds_failure() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_FundsFailure", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<std::string> SpaceCenter::Contract::keywords() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_Keywords", _args);
  std::vector<std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Contract::notes() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_Notes", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::ContractParameter> SpaceCenter::Contract::parameters() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_Parameters", _args);
  std::vector<SpaceCenter::ContractParameter> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Contract::read() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_Read", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Contract::reputation_completion() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_ReputationCompletion", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Contract::reputation_failure() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_ReputationFailure", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Contract::science_completion() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_ScienceCompletion", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Contract::seen() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_Seen", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ContractState SpaceCenter::Contract::state() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_State", _args);
  SpaceCenter::ContractState _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Contract::synopsis() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_Synopsis", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Contract::title() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_Title", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Contract::type() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Contract_get_Type", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<bool> SpaceCenter::Contract::active_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_Active", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Contract::can_be_canceled_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_CanBeCanceled", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Contract::can_be_declined_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_CanBeDeclined", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Contract::can_be_failed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_CanBeFailed", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Contract::description_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_Description", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Contract::failed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_Failed", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Contract::funds_advance_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_FundsAdvance", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Contract::funds_completion_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_FundsCompletion", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Contract::funds_failure_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_FundsFailure", _args));
}

inline ::krpc::Stream<std::vector<std::string>> SpaceCenter::Contract::keywords_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<std::string>>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_Keywords", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Contract::notes_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_Notes", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::ContractParameter>> SpaceCenter::Contract::parameters_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::ContractParameter>>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_Parameters", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Contract::read_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_Read", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Contract::reputation_completion_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_ReputationCompletion", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Contract::reputation_failure_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_ReputationFailure", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Contract::science_completion_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_ScienceCompletion", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Contract::seen_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_Seen", _args));
}

inline ::krpc::Stream<SpaceCenter::ContractState> SpaceCenter::Contract::state_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ContractState>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_State", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Contract::synopsis_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_Synopsis", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Contract::title_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_Title", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Contract::type_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Contract_get_Type", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::accept_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_Accept", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::cancel_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_Cancel", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::decline_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_Decline", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::active_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_Active", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::can_be_canceled_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_CanBeCanceled", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::can_be_declined_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_CanBeDeclined", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::can_be_failed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_CanBeFailed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::description_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_Description", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::failed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_Failed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::funds_advance_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_FundsAdvance", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::funds_completion_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_FundsCompletion", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::funds_failure_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_FundsFailure", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::keywords_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_Keywords", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::notes_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_Notes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::parameters_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_Parameters", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::read_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_Read", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::reputation_completion_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_ReputationCompletion", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::reputation_failure_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_ReputationFailure", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::science_completion_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_ScienceCompletion", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::seen_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_Seen", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::state_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_State", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::synopsis_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_Synopsis", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::title_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_Title", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Contract::type_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Contract_get_Type", _args);
}

inline SpaceCenter::ContractManager::ContractManager(Client* client, uint64_t id):
  Object(client, "SpaceCenter::ContractManager", id) {}

inline std::vector<SpaceCenter::Contract> SpaceCenter::ContractManager::active_contracts() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractManager_get_ActiveContracts", _args);
  std::vector<SpaceCenter::Contract> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Contract> SpaceCenter::ContractManager::all_contracts() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractManager_get_AllContracts", _args);
  std::vector<SpaceCenter::Contract> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Contract> SpaceCenter::ContractManager::completed_contracts() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractManager_get_CompletedContracts", _args);
  std::vector<SpaceCenter::Contract> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Contract> SpaceCenter::ContractManager::failed_contracts() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractManager_get_FailedContracts", _args);
  std::vector<SpaceCenter::Contract> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Contract> SpaceCenter::ContractManager::offered_contracts() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractManager_get_OfferedContracts", _args);
  std::vector<SpaceCenter::Contract> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::set<std::string> SpaceCenter::ContractManager::types() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractManager_get_Types", _args);
  std::set<std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<std::vector<SpaceCenter::Contract>> SpaceCenter::ContractManager::active_contracts_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Contract>>(this->_client, this->_client->build_call("SpaceCenter", "ContractManager_get_ActiveContracts", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Contract>> SpaceCenter::ContractManager::all_contracts_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Contract>>(this->_client, this->_client->build_call("SpaceCenter", "ContractManager_get_AllContracts", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Contract>> SpaceCenter::ContractManager::completed_contracts_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Contract>>(this->_client, this->_client->build_call("SpaceCenter", "ContractManager_get_CompletedContracts", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Contract>> SpaceCenter::ContractManager::failed_contracts_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Contract>>(this->_client, this->_client->build_call("SpaceCenter", "ContractManager_get_FailedContracts", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Contract>> SpaceCenter::ContractManager::offered_contracts_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Contract>>(this->_client, this->_client->build_call("SpaceCenter", "ContractManager_get_OfferedContracts", _args));
}

inline ::krpc::Stream<std::set<std::string>> SpaceCenter::ContractManager::types_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::set<std::string>>(this->_client, this->_client->build_call("SpaceCenter", "ContractManager_get_Types", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractManager::active_contracts_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractManager_get_ActiveContracts", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractManager::all_contracts_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractManager_get_AllContracts", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractManager::completed_contracts_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractManager_get_CompletedContracts", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractManager::failed_contracts_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractManager_get_FailedContracts", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractManager::offered_contracts_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractManager_get_OfferedContracts", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractManager::types_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractManager_get_Types", _args);
}

inline SpaceCenter::ContractParameter::ContractParameter(Client* client, uint64_t id):
  Object(client, "SpaceCenter::ContractParameter", id) {}

inline std::vector<SpaceCenter::ContractParameter> SpaceCenter::ContractParameter::children() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractParameter_get_Children", _args);
  std::vector<SpaceCenter::ContractParameter> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::ContractParameter::completed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractParameter_get_Completed", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::ContractParameter::failed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractParameter_get_Failed", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::ContractParameter::funds_completion() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractParameter_get_FundsCompletion", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::ContractParameter::funds_failure() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractParameter_get_FundsFailure", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::ContractParameter::notes() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractParameter_get_Notes", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::ContractParameter::optional() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractParameter_get_Optional", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::ContractParameter::reputation_completion() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractParameter_get_ReputationCompletion", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::ContractParameter::reputation_failure() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractParameter_get_ReputationFailure", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::ContractParameter::science_completion() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractParameter_get_ScienceCompletion", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::ContractParameter::title() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ContractParameter_get_Title", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<std::vector<SpaceCenter::ContractParameter>> SpaceCenter::ContractParameter::children_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::ContractParameter>>(this->_client, this->_client->build_call("SpaceCenter", "ContractParameter_get_Children", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::ContractParameter::completed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "ContractParameter_get_Completed", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::ContractParameter::failed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "ContractParameter_get_Failed", _args));
}

inline ::krpc::Stream<double> SpaceCenter::ContractParameter::funds_completion_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "ContractParameter_get_FundsCompletion", _args));
}

inline ::krpc::Stream<double> SpaceCenter::ContractParameter::funds_failure_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "ContractParameter_get_FundsFailure", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::ContractParameter::notes_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "ContractParameter_get_Notes", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::ContractParameter::optional_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "ContractParameter_get_Optional", _args));
}

inline ::krpc::Stream<double> SpaceCenter::ContractParameter::reputation_completion_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "ContractParameter_get_ReputationCompletion", _args));
}

inline ::krpc::Stream<double> SpaceCenter::ContractParameter::reputation_failure_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "ContractParameter_get_ReputationFailure", _args));
}

inline ::krpc::Stream<double> SpaceCenter::ContractParameter::science_completion_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "ContractParameter_get_ScienceCompletion", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::ContractParameter::title_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "ContractParameter_get_Title", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractParameter::children_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractParameter_get_Children", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractParameter::completed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractParameter_get_Completed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractParameter::failed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractParameter_get_Failed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractParameter::funds_completion_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractParameter_get_FundsCompletion", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractParameter::funds_failure_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractParameter_get_FundsFailure", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractParameter::notes_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractParameter_get_Notes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractParameter::optional_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractParameter_get_Optional", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractParameter::reputation_completion_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractParameter_get_ReputationCompletion", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractParameter::reputation_failure_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractParameter_get_ReputationFailure", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractParameter::science_completion_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractParameter_get_ScienceCompletion", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ContractParameter::title_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ContractParameter_get_Title", _args);
}

inline SpaceCenter::Control::Control(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Control", id) {}

inline std::vector<SpaceCenter::Vessel> SpaceCenter::Control::activate_next_stage() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_ActivateNextStage", _args);
  std::vector<SpaceCenter::Vessel> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Node SpaceCenter::Control::add_node(double ut, float prograde = 0.0, float normal = 0.0, float radial = 0.0) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  _args.push_back(encoder::encode(prograde));
  _args.push_back(encoder::encode(normal));
  _args.push_back(encoder::encode(radial));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_AddNode", _args);
  SpaceCenter::Node _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Control::get_action_group(uint32_t group) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(group));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_GetActionGroup", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::remove_nodes() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Control_RemoveNodes", _args);
}

inline void SpaceCenter::Control::set_action_group(uint32_t group, bool state) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(group));
  _args.push_back(encoder::encode(state));
  this->_client->invoke("SpaceCenter", "Control_SetActionGroup", _args);
}

inline void SpaceCenter::Control::toggle_action_group(uint32_t group) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(group));
  this->_client->invoke("SpaceCenter", "Control_ToggleActionGroup", _args);
}

inline bool SpaceCenter::Control::abort() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Abort", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_abort(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Abort", _args);
}

inline bool SpaceCenter::Control::antennas() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Antennas", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_antennas(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Antennas", _args);
}

inline bool SpaceCenter::Control::brakes() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Brakes", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_brakes(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Brakes", _args);
}

inline bool SpaceCenter::Control::cargo_bays() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_CargoBays", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_cargo_bays(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_CargoBays", _args);
}

inline int32_t SpaceCenter::Control::current_stage() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_CurrentStage", _args);
  int32_t _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Control::custom_axis01() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_CustomAxis01", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_custom_axis01(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_CustomAxis01", _args);
}

inline float SpaceCenter::Control::custom_axis02() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_CustomAxis02", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_custom_axis02(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_CustomAxis02", _args);
}

inline float SpaceCenter::Control::custom_axis03() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_CustomAxis03", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_custom_axis03(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_CustomAxis03", _args);
}

inline float SpaceCenter::Control::custom_axis04() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_CustomAxis04", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_custom_axis04(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_CustomAxis04", _args);
}

inline float SpaceCenter::Control::forward() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Forward", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_forward(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Forward", _args);
}

inline bool SpaceCenter::Control::gear() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Gear", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_gear(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Gear", _args);
}

inline SpaceCenter::ControlInputMode SpaceCenter::Control::input_mode() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_InputMode", _args);
  SpaceCenter::ControlInputMode _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_input_mode(SpaceCenter::ControlInputMode value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_InputMode", _args);
}

inline bool SpaceCenter::Control::intakes() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Intakes", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_intakes(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Intakes", _args);
}

inline bool SpaceCenter::Control::legs() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Legs", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_legs(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Legs", _args);
}

inline bool SpaceCenter::Control::lights() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Lights", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_lights(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Lights", _args);
}

inline std::vector<SpaceCenter::Node> SpaceCenter::Control::nodes() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Nodes", _args);
  std::vector<SpaceCenter::Node> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Control::parachutes() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Parachutes", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_parachutes(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Parachutes", _args);
}

inline float SpaceCenter::Control::pitch() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Pitch", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_pitch(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Pitch", _args);
}

inline bool SpaceCenter::Control::radiators() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Radiators", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_radiators(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Radiators", _args);
}

inline bool SpaceCenter::Control::rcs() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_RCS", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_rcs(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_RCS", _args);
}

inline bool SpaceCenter::Control::reaction_wheels() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_ReactionWheels", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_reaction_wheels(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_ReactionWheels", _args);
}

inline bool SpaceCenter::Control::resource_harvesters() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_ResourceHarvesters", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_resource_harvesters(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_ResourceHarvesters", _args);
}

inline bool SpaceCenter::Control::resource_harvesters_active() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_ResourceHarvestersActive", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_resource_harvesters_active(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_ResourceHarvestersActive", _args);
}

inline float SpaceCenter::Control::right() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Right", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_right(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Right", _args);
}

inline float SpaceCenter::Control::roll() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Roll", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_roll(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Roll", _args);
}

inline bool SpaceCenter::Control::sas() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_SAS", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_sas(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_SAS", _args);
}

inline SpaceCenter::SASMode SpaceCenter::Control::sas_mode() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_SASMode", _args);
  SpaceCenter::SASMode _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_sas_mode(SpaceCenter::SASMode value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_SASMode", _args);
}

inline bool SpaceCenter::Control::solar_panels() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_SolarPanels", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_solar_panels(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_SolarPanels", _args);
}

inline SpaceCenter::ControlSource SpaceCenter::Control::source() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Source", _args);
  SpaceCenter::ControlSource _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::SpeedMode SpaceCenter::Control::speed_mode() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_SpeedMode", _args);
  SpaceCenter::SpeedMode _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_speed_mode(SpaceCenter::SpeedMode value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_SpeedMode", _args);
}

inline bool SpaceCenter::Control::stage_lock() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_StageLock", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_stage_lock(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_StageLock", _args);
}

inline SpaceCenter::ControlState SpaceCenter::Control::state() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_State", _args);
  SpaceCenter::ControlState _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Control::throttle() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Throttle", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_throttle(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Throttle", _args);
}

inline float SpaceCenter::Control::up() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Up", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_up(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Up", _args);
}

inline float SpaceCenter::Control::wheel_steering() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_WheelSteering", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_wheel_steering(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_WheelSteering", _args);
}

inline float SpaceCenter::Control::wheel_throttle() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_WheelThrottle", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_wheel_throttle(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_WheelThrottle", _args);
}

inline bool SpaceCenter::Control::wheels() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Wheels", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_wheels(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Wheels", _args);
}

inline float SpaceCenter::Control::yaw() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Control_get_Yaw", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Control::set_yaw(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Control_set_Yaw", _args);
}

inline ::krpc::Stream<std::vector<SpaceCenter::Vessel>> SpaceCenter::Control::activate_next_stage_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Vessel>>(this->_client, this->_client->build_call("SpaceCenter", "Control_ActivateNextStage", _args));
}

inline ::krpc::Stream<SpaceCenter::Node> SpaceCenter::Control::add_node_stream(double ut, float prograde = 0.0, float normal = 0.0, float radial = 0.0) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  _args.push_back(encoder::encode(prograde));
  _args.push_back(encoder::encode(normal));
  _args.push_back(encoder::encode(radial));
  return ::krpc::Stream<SpaceCenter::Node>(this->_client, this->_client->build_call("SpaceCenter", "Control_AddNode", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::get_action_group_stream(uint32_t group) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(group));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_GetActionGroup", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::abort_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Abort", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::antennas_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Antennas", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::brakes_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Brakes", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::cargo_bays_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_CargoBays", _args));
}

inline ::krpc::Stream<int32_t> SpaceCenter::Control::current_stage_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<int32_t>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_CurrentStage", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::custom_axis01_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_CustomAxis01", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::custom_axis02_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_CustomAxis02", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::custom_axis03_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_CustomAxis03", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::custom_axis04_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_CustomAxis04", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::forward_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Forward", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::gear_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Gear", _args));
}

inline ::krpc::Stream<SpaceCenter::ControlInputMode> SpaceCenter::Control::input_mode_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ControlInputMode>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_InputMode", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::intakes_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Intakes", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::legs_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Legs", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::lights_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Lights", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Node>> SpaceCenter::Control::nodes_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Node>>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Nodes", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::parachutes_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Parachutes", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::pitch_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Pitch", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::radiators_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Radiators", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::rcs_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_RCS", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::reaction_wheels_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_ReactionWheels", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::resource_harvesters_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_ResourceHarvesters", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::resource_harvesters_active_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_ResourceHarvestersActive", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::right_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Right", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::roll_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Roll", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::sas_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_SAS", _args));
}

inline ::krpc::Stream<SpaceCenter::SASMode> SpaceCenter::Control::sas_mode_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::SASMode>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_SASMode", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::solar_panels_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_SolarPanels", _args));
}

inline ::krpc::Stream<SpaceCenter::ControlSource> SpaceCenter::Control::source_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ControlSource>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Source", _args));
}

inline ::krpc::Stream<SpaceCenter::SpeedMode> SpaceCenter::Control::speed_mode_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::SpeedMode>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_SpeedMode", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::stage_lock_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_StageLock", _args));
}

inline ::krpc::Stream<SpaceCenter::ControlState> SpaceCenter::Control::state_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ControlState>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_State", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::throttle_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Throttle", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::up_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Up", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::wheel_steering_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_WheelSteering", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::wheel_throttle_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_WheelThrottle", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Control::wheels_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Wheels", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Control::yaw_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Control_get_Yaw", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::activate_next_stage_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_ActivateNextStage", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::add_node_call(double ut, float prograde = 0.0, float normal = 0.0, float radial = 0.0) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  _args.push_back(encoder::encode(prograde));
  _args.push_back(encoder::encode(normal));
  _args.push_back(encoder::encode(radial));
  return this->_client->build_call("SpaceCenter", "Control_AddNode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::get_action_group_call(uint32_t group) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(group));
  return this->_client->build_call("SpaceCenter", "Control_GetActionGroup", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::remove_nodes_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_RemoveNodes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_action_group_call(uint32_t group, bool state) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(group));
  _args.push_back(encoder::encode(state));
  return this->_client->build_call("SpaceCenter", "Control_SetActionGroup", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::toggle_action_group_call(uint32_t group) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(group));
  return this->_client->build_call("SpaceCenter", "Control_ToggleActionGroup", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::abort_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Abort", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_abort_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Abort", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::antennas_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Antennas", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_antennas_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Antennas", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::brakes_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Brakes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_brakes_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Brakes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::cargo_bays_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_CargoBays", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_cargo_bays_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_CargoBays", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::current_stage_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_CurrentStage", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::custom_axis01_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_CustomAxis01", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_custom_axis01_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_CustomAxis01", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::custom_axis02_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_CustomAxis02", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_custom_axis02_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_CustomAxis02", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::custom_axis03_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_CustomAxis03", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_custom_axis03_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_CustomAxis03", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::custom_axis04_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_CustomAxis04", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_custom_axis04_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_CustomAxis04", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::forward_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Forward", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_forward_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Forward", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::gear_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Gear", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_gear_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Gear", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::input_mode_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_InputMode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_input_mode_call(SpaceCenter::ControlInputMode value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_InputMode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::intakes_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Intakes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_intakes_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Intakes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::legs_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Legs", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_legs_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Legs", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::lights_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Lights", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_lights_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Lights", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::nodes_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Nodes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::parachutes_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Parachutes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_parachutes_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Parachutes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::pitch_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Pitch", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_pitch_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Pitch", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::radiators_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Radiators", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_radiators_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Radiators", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::rcs_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_RCS", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_rcs_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_RCS", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::reaction_wheels_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_ReactionWheels", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_reaction_wheels_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_ReactionWheels", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::resource_harvesters_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_ResourceHarvesters", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_resource_harvesters_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_ResourceHarvesters", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::resource_harvesters_active_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_ResourceHarvestersActive", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_resource_harvesters_active_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_ResourceHarvestersActive", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::right_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Right", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_right_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Right", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::roll_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Roll", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_roll_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Roll", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::sas_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_SAS", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_sas_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_SAS", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::sas_mode_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_SASMode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_sas_mode_call(SpaceCenter::SASMode value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_SASMode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::solar_panels_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_SolarPanels", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_solar_panels_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_SolarPanels", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::source_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Source", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::speed_mode_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_SpeedMode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_speed_mode_call(SpaceCenter::SpeedMode value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_SpeedMode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::stage_lock_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_StageLock", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_stage_lock_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_StageLock", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::state_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_State", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::throttle_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Throttle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_throttle_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Throttle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::up_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Up", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_up_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Up", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::wheel_steering_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_WheelSteering", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_wheel_steering_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_WheelSteering", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::wheel_throttle_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_WheelThrottle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_wheel_throttle_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_WheelThrottle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::wheels_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Wheels", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_wheels_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Wheels", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::yaw_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Control_get_Yaw", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Control::set_yaw_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Control_set_Yaw", _args);
}

inline SpaceCenter::ControlSurface::ControlSurface(Client* client, uint64_t id):
  Object(client, "SpaceCenter::ControlSurface", id) {}

inline float SpaceCenter::ControlSurface::authority_limiter() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ControlSurface_get_AuthorityLimiter", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::ControlSurface::set_authority_limiter(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "ControlSurface_set_AuthorityLimiter", _args);
}

inline std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > SpaceCenter::ControlSurface::available_torque() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ControlSurface_get_AvailableTorque", _args);
  std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::ControlSurface::deployed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ControlSurface_get_Deployed", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::ControlSurface::set_deployed(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "ControlSurface_set_Deployed", _args);
}

inline bool SpaceCenter::ControlSurface::inverted() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ControlSurface_get_Inverted", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::ControlSurface::set_inverted(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "ControlSurface_set_Inverted", _args);
}

inline SpaceCenter::Part SpaceCenter::ControlSurface::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ControlSurface_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::ControlSurface::pitch_enabled() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ControlSurface_get_PitchEnabled", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::ControlSurface::set_pitch_enabled(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "ControlSurface_set_PitchEnabled", _args);
}

inline bool SpaceCenter::ControlSurface::roll_enabled() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ControlSurface_get_RollEnabled", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::ControlSurface::set_roll_enabled(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "ControlSurface_set_RollEnabled", _args);
}

inline float SpaceCenter::ControlSurface::surface_area() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ControlSurface_get_SurfaceArea", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::ControlSurface::yaw_enabled() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "ControlSurface_get_YawEnabled", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::ControlSurface::set_yaw_enabled(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "ControlSurface_set_YawEnabled", _args);
}

inline ::krpc::Stream<float> SpaceCenter::ControlSurface::authority_limiter_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "ControlSurface_get_AuthorityLimiter", _args));
}

inline ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> SpaceCenter::ControlSurface::available_torque_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >>(this->_client, this->_client->build_call("SpaceCenter", "ControlSurface_get_AvailableTorque", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::ControlSurface::deployed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "ControlSurface_get_Deployed", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::ControlSurface::inverted_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "ControlSurface_get_Inverted", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::ControlSurface::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "ControlSurface_get_Part", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::ControlSurface::pitch_enabled_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "ControlSurface_get_PitchEnabled", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::ControlSurface::roll_enabled_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "ControlSurface_get_RollEnabled", _args));
}

inline ::krpc::Stream<float> SpaceCenter::ControlSurface::surface_area_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "ControlSurface_get_SurfaceArea", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::ControlSurface::yaw_enabled_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "ControlSurface_get_YawEnabled", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::authority_limiter_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ControlSurface_get_AuthorityLimiter", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::set_authority_limiter_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "ControlSurface_set_AuthorityLimiter", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::available_torque_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ControlSurface_get_AvailableTorque", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::deployed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ControlSurface_get_Deployed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::set_deployed_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "ControlSurface_set_Deployed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::inverted_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ControlSurface_get_Inverted", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::set_inverted_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "ControlSurface_set_Inverted", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ControlSurface_get_Part", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::pitch_enabled_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ControlSurface_get_PitchEnabled", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::set_pitch_enabled_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "ControlSurface_set_PitchEnabled", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::roll_enabled_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ControlSurface_get_RollEnabled", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::set_roll_enabled_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "ControlSurface_set_RollEnabled", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::surface_area_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ControlSurface_get_SurfaceArea", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::yaw_enabled_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "ControlSurface_get_YawEnabled", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::ControlSurface::set_yaw_enabled_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "ControlSurface_set_YawEnabled", _args);
}

inline SpaceCenter::CrewMember::CrewMember(Client* client, uint64_t id):
  Object(client, "SpaceCenter::CrewMember", id) {}

inline bool SpaceCenter::CrewMember::badass() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_Badass", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::CrewMember::set_badass(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "CrewMember_set_Badass", _args);
}

inline std::vector<int32_t> SpaceCenter::CrewMember::career_log_flights() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_CareerLogFlights", _args);
  std::vector<int32_t> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<std::string> SpaceCenter::CrewMember::career_log_targets() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_CareerLogTargets", _args);
  std::vector<std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<std::string> SpaceCenter::CrewMember::career_log_types() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_CareerLogTypes", _args);
  std::vector<std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::CrewMember::courage() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_Courage", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::CrewMember::set_courage(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "CrewMember_set_Courage", _args);
}

inline float SpaceCenter::CrewMember::experience() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_Experience", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::CrewMember::set_experience(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "CrewMember_set_Experience", _args);
}

inline SpaceCenter::CrewMemberGender SpaceCenter::CrewMember::gender() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_Gender", _args);
  SpaceCenter::CrewMemberGender _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::CrewMember::name() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_Name", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::CrewMember::set_name(std::string value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "CrewMember_set_Name", _args);
}

inline bool SpaceCenter::CrewMember::on_mission() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_OnMission", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::RosterStatus SpaceCenter::CrewMember::roster_status() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_RosterStatus", _args);
  SpaceCenter::RosterStatus _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::CrewMember::stupidity() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_Stupidity", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::CrewMember::set_stupidity(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "CrewMember_set_Stupidity", _args);
}

inline SpaceCenter::SuitType SpaceCenter::CrewMember::suit_type() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_SuitType", _args);
  SpaceCenter::SuitType _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::CrewMember::set_suit_type(SpaceCenter::SuitType value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "CrewMember_set_SuitType", _args);
}

inline std::string SpaceCenter::CrewMember::trait() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_Trait", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::CrewMemberType SpaceCenter::CrewMember::type() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_Type", _args);
  SpaceCenter::CrewMemberType _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::CrewMember::veteran() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "CrewMember_get_Veteran", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::CrewMember::set_veteran(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "CrewMember_set_Veteran", _args);
}

inline ::krpc::Stream<bool> SpaceCenter::CrewMember::badass_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_Badass", _args));
}

inline ::krpc::Stream<std::vector<int32_t>> SpaceCenter::CrewMember::career_log_flights_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<int32_t>>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_CareerLogFlights", _args));
}

inline ::krpc::Stream<std::vector<std::string>> SpaceCenter::CrewMember::career_log_targets_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<std::string>>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_CareerLogTargets", _args));
}

inline ::krpc::Stream<std::vector<std::string>> SpaceCenter::CrewMember::career_log_types_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<std::string>>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_CareerLogTypes", _args));
}

inline ::krpc::Stream<float> SpaceCenter::CrewMember::courage_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_Courage", _args));
}

inline ::krpc::Stream<float> SpaceCenter::CrewMember::experience_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_Experience", _args));
}

inline ::krpc::Stream<SpaceCenter::CrewMemberGender> SpaceCenter::CrewMember::gender_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::CrewMemberGender>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_Gender", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::CrewMember::name_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_Name", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::CrewMember::on_mission_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_OnMission", _args));
}

inline ::krpc::Stream<SpaceCenter::RosterStatus> SpaceCenter::CrewMember::roster_status_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::RosterStatus>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_RosterStatus", _args));
}

inline ::krpc::Stream<float> SpaceCenter::CrewMember::stupidity_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_Stupidity", _args));
}

inline ::krpc::Stream<SpaceCenter::SuitType> SpaceCenter::CrewMember::suit_type_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::SuitType>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_SuitType", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::CrewMember::trait_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_Trait", _args));
}

inline ::krpc::Stream<SpaceCenter::CrewMemberType> SpaceCenter::CrewMember::type_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::CrewMemberType>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_Type", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::CrewMember::veteran_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "CrewMember_get_Veteran", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::badass_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_Badass", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::set_badass_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "CrewMember_set_Badass", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::career_log_flights_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_CareerLogFlights", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::career_log_targets_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_CareerLogTargets", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::career_log_types_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_CareerLogTypes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::courage_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_Courage", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::set_courage_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "CrewMember_set_Courage", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::experience_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_Experience", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::set_experience_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "CrewMember_set_Experience", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::gender_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_Gender", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::name_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_Name", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::set_name_call(std::string value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "CrewMember_set_Name", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::on_mission_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_OnMission", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::roster_status_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_RosterStatus", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::stupidity_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_Stupidity", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::set_stupidity_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "CrewMember_set_Stupidity", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::suit_type_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_SuitType", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::set_suit_type_call(SpaceCenter::SuitType value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "CrewMember_set_SuitType", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::trait_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_Trait", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::type_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_Type", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::veteran_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "CrewMember_get_Veteran", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::CrewMember::set_veteran_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "CrewMember_set_Veteran", _args);
}

inline SpaceCenter::Decoupler::Decoupler(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Decoupler", id) {}

inline SpaceCenter::Vessel SpaceCenter::Decoupler::decouple() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Decoupler_Decouple", _args);
  SpaceCenter::Vessel _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::Decoupler::attached_part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Decoupler_get_AttachedPart", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Decoupler::decoupled() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Decoupler_get_Decoupled", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Decoupler::impulse() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Decoupler_get_Impulse", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Decoupler::is_omni_decoupler() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Decoupler_get_IsOmniDecoupler", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::Decoupler::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Decoupler_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Decoupler::staged() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Decoupler_get_Staged", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<SpaceCenter::Vessel> SpaceCenter::Decoupler::decouple_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Vessel>(this->_client, this->_client->build_call("SpaceCenter", "Decoupler_Decouple", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Decoupler::attached_part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Decoupler_get_AttachedPart", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Decoupler::decoupled_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Decoupler_get_Decoupled", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Decoupler::impulse_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Decoupler_get_Impulse", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Decoupler::is_omni_decoupler_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Decoupler_get_IsOmniDecoupler", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Decoupler::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Decoupler_get_Part", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Decoupler::staged_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Decoupler_get_Staged", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Decoupler::decouple_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Decoupler_Decouple", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Decoupler::attached_part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Decoupler_get_AttachedPart", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Decoupler::decoupled_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Decoupler_get_Decoupled", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Decoupler::impulse_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Decoupler_get_Impulse", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Decoupler::is_omni_decoupler_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Decoupler_get_IsOmniDecoupler", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Decoupler::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Decoupler_get_Part", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Decoupler::staged_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Decoupler_get_Staged", _args);
}

inline SpaceCenter::DockingPort::DockingPort(Client* client, uint64_t id):
  Object(client, "SpaceCenter::DockingPort", id) {}

inline std::tuple<double, double, double> SpaceCenter::DockingPort::direction(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_Direction", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::DockingPort::position(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_Position", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double, double> SpaceCenter::DockingPort::rotation(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_Rotation", _args);
  std::tuple<double, double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Vessel SpaceCenter::DockingPort::undock() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_Undock", _args);
  SpaceCenter::Vessel _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::DockingPort::can_rotate() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_get_CanRotate", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::DockingPort::docked_part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_get_DockedPart", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::DockingPort::has_shield() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_get_HasShield", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::DockingPort::maximum_rotation() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_get_MaximumRotation", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::DockingPort::minimum_rotation() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_get_MinimumRotation", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::DockingPort::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::DockingPort::reengage_distance() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_get_ReengageDistance", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ReferenceFrame SpaceCenter::DockingPort::reference_frame() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_get_ReferenceFrame", _args);
  SpaceCenter::ReferenceFrame _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::DockingPort::rotation_locked() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_get_RotationLocked", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::DockingPort::set_rotation_locked(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "DockingPort_set_RotationLocked", _args);
}

inline float SpaceCenter::DockingPort::rotation_target() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_get_RotationTarget", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::DockingPort::set_rotation_target(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "DockingPort_set_RotationTarget", _args);
}

inline bool SpaceCenter::DockingPort::shielded() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_get_Shielded", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::DockingPort::set_shielded(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "DockingPort_set_Shielded", _args);
}

inline SpaceCenter::DockingPortState SpaceCenter::DockingPort::state() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "DockingPort_get_State", _args);
  SpaceCenter::DockingPortState _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::DockingPort::direction_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_Direction", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::DockingPort::position_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_Position", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double, double>> SpaceCenter::DockingPort::rotation_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_Rotation", _args));
}

inline ::krpc::Stream<SpaceCenter::Vessel> SpaceCenter::DockingPort::undock_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Vessel>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_Undock", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::DockingPort::can_rotate_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_get_CanRotate", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::DockingPort::docked_part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_get_DockedPart", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::DockingPort::has_shield_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_get_HasShield", _args));
}

inline ::krpc::Stream<float> SpaceCenter::DockingPort::maximum_rotation_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_get_MaximumRotation", _args));
}

inline ::krpc::Stream<float> SpaceCenter::DockingPort::minimum_rotation_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_get_MinimumRotation", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::DockingPort::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_get_Part", _args));
}

inline ::krpc::Stream<float> SpaceCenter::DockingPort::reengage_distance_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_get_ReengageDistance", _args));
}

inline ::krpc::Stream<SpaceCenter::ReferenceFrame> SpaceCenter::DockingPort::reference_frame_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ReferenceFrame>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_get_ReferenceFrame", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::DockingPort::rotation_locked_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_get_RotationLocked", _args));
}

inline ::krpc::Stream<float> SpaceCenter::DockingPort::rotation_target_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_get_RotationTarget", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::DockingPort::shielded_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_get_Shielded", _args));
}

inline ::krpc::Stream<SpaceCenter::DockingPortState> SpaceCenter::DockingPort::state_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::DockingPortState>(this->_client, this->_client->build_call("SpaceCenter", "DockingPort_get_State", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::direction_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "DockingPort_Direction", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::position_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "DockingPort_Position", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::rotation_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "DockingPort_Rotation", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::undock_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_Undock", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::can_rotate_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_get_CanRotate", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::docked_part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_get_DockedPart", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::has_shield_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_get_HasShield", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::maximum_rotation_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_get_MaximumRotation", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::minimum_rotation_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_get_MinimumRotation", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_get_Part", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::reengage_distance_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_get_ReengageDistance", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::reference_frame_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_get_ReferenceFrame", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::rotation_locked_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_get_RotationLocked", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::set_rotation_locked_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "DockingPort_set_RotationLocked", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::rotation_target_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_get_RotationTarget", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::set_rotation_target_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "DockingPort_set_RotationTarget", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::shielded_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_get_Shielded", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::set_shielded_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "DockingPort_set_Shielded", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::DockingPort::state_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "DockingPort_get_State", _args);
}

inline SpaceCenter::Engine::Engine(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Engine", id) {}

inline float SpaceCenter::Engine::available_thrust_at(double pressure) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(pressure));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_AvailableThrustAt", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Engine::max_thrust_at(double pressure) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(pressure));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_MaxThrustAt", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Engine::specific_impulse_at(double pressure) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(pressure));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_SpecificImpulseAt", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Engine::toggle_mode() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Engine_ToggleMode", _args);
}

inline bool SpaceCenter::Engine::active() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_Active", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Engine::set_active(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Engine_set_Active", _args);
}

inline bool SpaceCenter::Engine::auto_mode_switch() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_AutoModeSwitch", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Engine::set_auto_mode_switch(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Engine_set_AutoModeSwitch", _args);
}

inline float SpaceCenter::Engine::available_thrust() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_AvailableThrust", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > SpaceCenter::Engine::available_torque() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_AvailableTorque", _args);
  std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Engine::can_restart() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_CanRestart", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Engine::can_shutdown() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_CanShutdown", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Engine::gimbal_limit() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_GimbalLimit", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Engine::set_gimbal_limit(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Engine_set_GimbalLimit", _args);
}

inline bool SpaceCenter::Engine::gimbal_locked() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_GimbalLocked", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Engine::set_gimbal_locked(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Engine_set_GimbalLocked", _args);
}

inline float SpaceCenter::Engine::gimbal_range() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_GimbalRange", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Engine::gimballed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_Gimballed", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Engine::has_fuel() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_HasFuel", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Engine::has_modes() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_HasModes", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Engine::independent_throttle() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_IndependentThrottle", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Engine::set_independent_throttle(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Engine_set_IndependentThrottle", _args);
}

inline float SpaceCenter::Engine::kerbin_sea_level_specific_impulse() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_KerbinSeaLevelSpecificImpulse", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Engine::max_thrust() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_MaxThrust", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Engine::max_vacuum_thrust() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_MaxVacuumThrust", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Engine::mode() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_Mode", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Engine::set_mode(std::string value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Engine_set_Mode", _args);
}

inline std::map<std::string, SpaceCenter::Engine> SpaceCenter::Engine::modes() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_Modes", _args);
  std::map<std::string, SpaceCenter::Engine> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::Engine::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<std::string> SpaceCenter::Engine::propellant_names() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_PropellantNames", _args);
  std::vector<std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::map<std::string, float> SpaceCenter::Engine::propellant_ratios() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_PropellantRatios", _args);
  std::map<std::string, float> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Propellant> SpaceCenter::Engine::propellants() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_Propellants", _args);
  std::vector<SpaceCenter::Propellant> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Engine::specific_impulse() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_SpecificImpulse", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Engine::throttle() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_Throttle", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Engine::set_throttle(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Engine_set_Throttle", _args);
}

inline bool SpaceCenter::Engine::throttle_locked() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_ThrottleLocked", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Engine::thrust() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_Thrust", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Engine::thrust_limit() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_ThrustLimit", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Engine::set_thrust_limit(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Engine_set_ThrustLimit", _args);
}

inline std::vector<SpaceCenter::Thruster> SpaceCenter::Engine::thrusters() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_Thrusters", _args);
  std::vector<SpaceCenter::Thruster> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Engine::vacuum_specific_impulse() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Engine_get_VacuumSpecificImpulse", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<float> SpaceCenter::Engine::available_thrust_at_stream(double pressure) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(pressure));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_AvailableThrustAt", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::max_thrust_at_stream(double pressure) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(pressure));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_MaxThrustAt", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::specific_impulse_at_stream(double pressure) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(pressure));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_SpecificImpulseAt", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Engine::active_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_Active", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Engine::auto_mode_switch_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_AutoModeSwitch", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::available_thrust_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_AvailableThrust", _args));
}

inline ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> SpaceCenter::Engine::available_torque_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_AvailableTorque", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Engine::can_restart_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_CanRestart", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Engine::can_shutdown_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_CanShutdown", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::gimbal_limit_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_GimbalLimit", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Engine::gimbal_locked_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_GimbalLocked", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::gimbal_range_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_GimbalRange", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Engine::gimballed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_Gimballed", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Engine::has_fuel_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_HasFuel", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Engine::has_modes_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_HasModes", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Engine::independent_throttle_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_IndependentThrottle", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::kerbin_sea_level_specific_impulse_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_KerbinSeaLevelSpecificImpulse", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::max_thrust_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_MaxThrust", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::max_vacuum_thrust_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_MaxVacuumThrust", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Engine::mode_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_Mode", _args));
}

inline ::krpc::Stream<std::map<std::string, SpaceCenter::Engine>> SpaceCenter::Engine::modes_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::map<std::string, SpaceCenter::Engine>>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_Modes", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Engine::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_Part", _args));
}

inline ::krpc::Stream<std::vector<std::string>> SpaceCenter::Engine::propellant_names_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<std::string>>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_PropellantNames", _args));
}

inline ::krpc::Stream<std::map<std::string, float>> SpaceCenter::Engine::propellant_ratios_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::map<std::string, float>>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_PropellantRatios", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Propellant>> SpaceCenter::Engine::propellants_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Propellant>>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_Propellants", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::specific_impulse_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_SpecificImpulse", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::throttle_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_Throttle", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Engine::throttle_locked_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_ThrottleLocked", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::thrust_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_Thrust", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::thrust_limit_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_ThrustLimit", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Thruster>> SpaceCenter::Engine::thrusters_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Thruster>>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_Thrusters", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Engine::vacuum_specific_impulse_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Engine_get_VacuumSpecificImpulse", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::available_thrust_at_call(double pressure) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(pressure));
  return this->_client->build_call("SpaceCenter", "Engine_AvailableThrustAt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::max_thrust_at_call(double pressure) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(pressure));
  return this->_client->build_call("SpaceCenter", "Engine_MaxThrustAt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::specific_impulse_at_call(double pressure) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(pressure));
  return this->_client->build_call("SpaceCenter", "Engine_SpecificImpulseAt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::toggle_mode_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_ToggleMode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::active_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_Active", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::set_active_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Engine_set_Active", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::auto_mode_switch_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_AutoModeSwitch", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::set_auto_mode_switch_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Engine_set_AutoModeSwitch", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::available_thrust_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_AvailableThrust", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::available_torque_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_AvailableTorque", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::can_restart_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_CanRestart", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::can_shutdown_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_CanShutdown", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::gimbal_limit_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_GimbalLimit", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::set_gimbal_limit_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Engine_set_GimbalLimit", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::gimbal_locked_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_GimbalLocked", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::set_gimbal_locked_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Engine_set_GimbalLocked", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::gimbal_range_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_GimbalRange", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::gimballed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_Gimballed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::has_fuel_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_HasFuel", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::has_modes_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_HasModes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::independent_throttle_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_IndependentThrottle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::set_independent_throttle_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Engine_set_IndependentThrottle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::kerbin_sea_level_specific_impulse_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_KerbinSeaLevelSpecificImpulse", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::max_thrust_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_MaxThrust", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::max_vacuum_thrust_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_MaxVacuumThrust", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::mode_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_Mode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::set_mode_call(std::string value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Engine_set_Mode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::modes_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_Modes", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_Part", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::propellant_names_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_PropellantNames", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::propellant_ratios_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_PropellantRatios", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::propellants_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_Propellants", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::specific_impulse_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_SpecificImpulse", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::throttle_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_Throttle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::set_throttle_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Engine_set_Throttle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::throttle_locked_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_ThrottleLocked", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::thrust_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_Thrust", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::thrust_limit_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_ThrustLimit", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::set_thrust_limit_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Engine_set_ThrustLimit", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::thrusters_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_Thrusters", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Engine::vacuum_specific_impulse_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Engine_get_VacuumSpecificImpulse", _args);
}

inline SpaceCenter::Experiment::Experiment(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Experiment", id) {}

inline void SpaceCenter::Experiment::dump() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Experiment_Dump", _args);
}

inline void SpaceCenter::Experiment::reset() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Experiment_Reset", _args);
}

inline void SpaceCenter::Experiment::run() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Experiment_Run", _args);
}

inline void SpaceCenter::Experiment::transmit() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Experiment_Transmit", _args);
}

inline bool SpaceCenter::Experiment::available() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Experiment_get_Available", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Experiment::biome() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Experiment_get_Biome", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::ScienceData> SpaceCenter::Experiment::data() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Experiment_get_Data", _args);
  std::vector<SpaceCenter::ScienceData> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Experiment::deployed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Experiment_get_Deployed", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Experiment::has_data() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Experiment_get_HasData", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Experiment::inoperable() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Experiment_get_Inoperable", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Experiment::name() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Experiment_get_Name", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::Experiment::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Experiment_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Experiment::rerunnable() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Experiment_get_Rerunnable", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ScienceSubject SpaceCenter::Experiment::science_subject() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Experiment_get_ScienceSubject", _args);
  SpaceCenter::ScienceSubject _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Experiment::title() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Experiment_get_Title", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<bool> SpaceCenter::Experiment::available_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Experiment_get_Available", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Experiment::biome_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Experiment_get_Biome", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::ScienceData>> SpaceCenter::Experiment::data_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::ScienceData>>(this->_client, this->_client->build_call("SpaceCenter", "Experiment_get_Data", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Experiment::deployed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Experiment_get_Deployed", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Experiment::has_data_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Experiment_get_HasData", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Experiment::inoperable_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Experiment_get_Inoperable", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Experiment::name_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Experiment_get_Name", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Experiment::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Experiment_get_Part", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Experiment::rerunnable_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Experiment_get_Rerunnable", _args));
}

inline ::krpc::Stream<SpaceCenter::ScienceSubject> SpaceCenter::Experiment::science_subject_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ScienceSubject>(this->_client, this->_client->build_call("SpaceCenter", "Experiment_get_ScienceSubject", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Experiment::title_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Experiment_get_Title", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::dump_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_Dump", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::reset_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_Reset", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::run_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_Run", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::transmit_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_Transmit", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::available_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_get_Available", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::biome_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_get_Biome", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::data_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_get_Data", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::deployed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_get_Deployed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::has_data_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_get_HasData", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::inoperable_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_get_Inoperable", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::name_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_get_Name", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_get_Part", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::rerunnable_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_get_Rerunnable", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::science_subject_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_get_ScienceSubject", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Experiment::title_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Experiment_get_Title", _args);
}

inline SpaceCenter::Fairing::Fairing(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Fairing", id) {}

inline void SpaceCenter::Fairing::jettison() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Fairing_Jettison", _args);
}

inline bool SpaceCenter::Fairing::jettisoned() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Fairing_get_Jettisoned", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::Fairing::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Fairing_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<bool> SpaceCenter::Fairing::jettisoned_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Fairing_get_Jettisoned", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Fairing::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Fairing_get_Part", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Fairing::jettison_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Fairing_Jettison", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Fairing::jettisoned_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Fairing_get_Jettisoned", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Fairing::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Fairing_get_Part", _args);
}

inline SpaceCenter::Flight::Flight(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Flight", id) {}

inline std::tuple<double, double, double> SpaceCenter::Flight::simulate_aerodynamic_force_at(SpaceCenter::CelestialBody body, std::tuple<double, double, double> position, std::tuple<double, double, double> velocity) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(body));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(velocity));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_SimulateAerodynamicForceAt", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Flight::aerodynamic_force() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_AerodynamicForce", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::angle_of_attack() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_AngleOfAttack", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Flight::anti_normal() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_AntiNormal", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Flight::anti_radial() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_AntiRadial", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::atmosphere_density() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_AtmosphereDensity", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::ballistic_coefficient() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_BallisticCoefficient", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Flight::bedrock_altitude() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_BedrockAltitude", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Flight::center_of_mass() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_CenterOfMass", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Flight::direction() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Direction", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Flight::drag() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Drag", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::drag_coefficient() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_DragCoefficient", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::dynamic_pressure() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_DynamicPressure", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Flight::elevation() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Elevation", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::equivalent_air_speed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_EquivalentAirSpeed", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::g_force() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_GForce", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::heading() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Heading", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Flight::horizontal_speed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_HorizontalSpeed", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Flight::latitude() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Latitude", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Flight::lift() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Lift", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::lift_coefficient() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_LiftCoefficient", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Flight::longitude() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Longitude", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::mach() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Mach", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Flight::mean_altitude() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_MeanAltitude", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Flight::normal() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Normal", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::pitch() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Pitch", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Flight::prograde() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Prograde", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Flight::radial() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Radial", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Flight::retrograde() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Retrograde", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::reynolds_number() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_ReynoldsNumber", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::roll() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Roll", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double, double> SpaceCenter::Flight::rotation() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Rotation", _args);
  std::tuple<double, double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::sideslip_angle() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_SideslipAngle", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Flight::speed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Speed", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::speed_of_sound() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_SpeedOfSound", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::stall_fraction() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_StallFraction", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::static_air_temperature() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_StaticAirTemperature", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::static_pressure() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_StaticPressure", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::static_pressure_at_msl() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_StaticPressureAtMSL", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Flight::surface_altitude() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_SurfaceAltitude", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::terminal_velocity() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_TerminalVelocity", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::thrust_specific_fuel_consumption() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_ThrustSpecificFuelConsumption", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::total_air_temperature() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_TotalAirTemperature", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Flight::true_air_speed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_TrueAirSpeed", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Flight::velocity() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_Velocity", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Flight::vertical_speed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Flight_get_VerticalSpeed", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::simulate_aerodynamic_force_at_stream(SpaceCenter::CelestialBody body, std::tuple<double, double, double> position, std::tuple<double, double, double> velocity) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(body));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(velocity));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_SimulateAerodynamicForceAt", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::aerodynamic_force_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_AerodynamicForce", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::angle_of_attack_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_AngleOfAttack", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::anti_normal_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_AntiNormal", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::anti_radial_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_AntiRadial", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::atmosphere_density_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_AtmosphereDensity", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::ballistic_coefficient_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_BallisticCoefficient", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Flight::bedrock_altitude_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_BedrockAltitude", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::center_of_mass_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_CenterOfMass", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::direction_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Direction", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::drag_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Drag", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::drag_coefficient_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_DragCoefficient", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::dynamic_pressure_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_DynamicPressure", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Flight::elevation_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Elevation", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::equivalent_air_speed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_EquivalentAirSpeed", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::g_force_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_GForce", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::heading_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Heading", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Flight::horizontal_speed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_HorizontalSpeed", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Flight::latitude_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Latitude", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::lift_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Lift", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::lift_coefficient_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_LiftCoefficient", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Flight::longitude_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Longitude", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::mach_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Mach", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Flight::mean_altitude_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_MeanAltitude", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::normal_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Normal", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::pitch_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Pitch", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::prograde_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Prograde", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::radial_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Radial", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::retrograde_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Retrograde", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::reynolds_number_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_ReynoldsNumber", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::roll_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Roll", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double, double>> SpaceCenter::Flight::rotation_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Rotation", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::sideslip_angle_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_SideslipAngle", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Flight::speed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Speed", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::speed_of_sound_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_SpeedOfSound", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::stall_fraction_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_StallFraction", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::static_air_temperature_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_StaticAirTemperature", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::static_pressure_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_StaticPressure", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::static_pressure_at_msl_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_StaticPressureAtMSL", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Flight::surface_altitude_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_SurfaceAltitude", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::terminal_velocity_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_TerminalVelocity", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::thrust_specific_fuel_consumption_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_ThrustSpecificFuelConsumption", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::total_air_temperature_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_TotalAirTemperature", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Flight::true_air_speed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_TrueAirSpeed", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Flight::velocity_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_Velocity", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Flight::vertical_speed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Flight_get_VerticalSpeed", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::simulate_aerodynamic_force_at_call(SpaceCenter::CelestialBody body, std::tuple<double, double, double> position, std::tuple<double, double, double> velocity) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(body));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(velocity));
  return this->_client->build_call("SpaceCenter", "Flight_SimulateAerodynamicForceAt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::aerodynamic_force_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_AerodynamicForce", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::angle_of_attack_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_AngleOfAttack", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::anti_normal_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_AntiNormal", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::anti_radial_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_AntiRadial", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::atmosphere_density_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_AtmosphereDensity", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::ballistic_coefficient_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_BallisticCoefficient", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::bedrock_altitude_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_BedrockAltitude", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::center_of_mass_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_CenterOfMass", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::direction_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Direction", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::drag_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Drag", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::drag_coefficient_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_DragCoefficient", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::dynamic_pressure_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_DynamicPressure", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::elevation_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Elevation", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::equivalent_air_speed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_EquivalentAirSpeed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::g_force_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_GForce", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::heading_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Heading", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::horizontal_speed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_HorizontalSpeed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::latitude_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Latitude", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::lift_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Lift", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::lift_coefficient_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_LiftCoefficient", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::longitude_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Longitude", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::mach_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Mach", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::mean_altitude_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_MeanAltitude", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::normal_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Normal", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::pitch_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Pitch", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::prograde_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Prograde", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::radial_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Radial", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::retrograde_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Retrograde", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::reynolds_number_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_ReynoldsNumber", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::roll_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Roll", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::rotation_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Rotation", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::sideslip_angle_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_SideslipAngle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::speed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Speed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::speed_of_sound_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_SpeedOfSound", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::stall_fraction_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_StallFraction", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::static_air_temperature_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_StaticAirTemperature", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::static_pressure_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_StaticPressure", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::static_pressure_at_msl_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_StaticPressureAtMSL", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::surface_altitude_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_SurfaceAltitude", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::terminal_velocity_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_TerminalVelocity", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::thrust_specific_fuel_consumption_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_ThrustSpecificFuelConsumption", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::total_air_temperature_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_TotalAirTemperature", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::true_air_speed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_TrueAirSpeed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::velocity_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_Velocity", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Flight::vertical_speed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Flight_get_VerticalSpeed", _args);
}

inline SpaceCenter::Force::Force(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Force", id) {}

inline void SpaceCenter::Force::remove() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Force_Remove", _args);
}

inline std::tuple<double, double, double> SpaceCenter::Force::force_vector() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Force_get_ForceVector", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Force::set_force_vector(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Force_set_ForceVector", _args);
}

inline SpaceCenter::Part SpaceCenter::Force::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Force_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Force::position() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Force_get_Position", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Force::set_position(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Force_set_Position", _args);
}

inline SpaceCenter::ReferenceFrame SpaceCenter::Force::reference_frame() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Force_get_ReferenceFrame", _args);
  SpaceCenter::ReferenceFrame _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Force::set_reference_frame(SpaceCenter::ReferenceFrame value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Force_set_ReferenceFrame", _args);
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Force::force_vector_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Force_get_ForceVector", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Force::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Force_get_Part", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Force::position_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Force_get_Position", _args));
}

inline ::krpc::Stream<SpaceCenter::ReferenceFrame> SpaceCenter::Force::reference_frame_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ReferenceFrame>(this->_client, this->_client->build_call("SpaceCenter", "Force_get_ReferenceFrame", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Force::remove_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Force_Remove", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Force::force_vector_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Force_get_ForceVector", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Force::set_force_vector_call(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Force_set_ForceVector", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Force::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Force_get_Part", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Force::position_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Force_get_Position", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Force::set_position_call(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Force_set_Position", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Force::reference_frame_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Force_get_ReferenceFrame", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Force::set_reference_frame_call(SpaceCenter::ReferenceFrame value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Force_set_ReferenceFrame", _args);
}

inline SpaceCenter::Intake::Intake(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Intake", id) {}

inline float SpaceCenter::Intake::area() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Intake_get_Area", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Intake::flow() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Intake_get_Flow", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Intake::open() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Intake_get_Open", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Intake::set_open(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Intake_set_Open", _args);
}

inline SpaceCenter::Part SpaceCenter::Intake::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Intake_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Intake::speed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Intake_get_Speed", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<float> SpaceCenter::Intake::area_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Intake_get_Area", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Intake::flow_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Intake_get_Flow", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Intake::open_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Intake_get_Open", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Intake::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Intake_get_Part", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Intake::speed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Intake_get_Speed", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Intake::area_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Intake_get_Area", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Intake::flow_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Intake_get_Flow", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Intake::open_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Intake_get_Open", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Intake::set_open_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Intake_set_Open", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Intake::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Intake_get_Part", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Intake::speed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Intake_get_Speed", _args);
}

inline SpaceCenter::LaunchClamp::LaunchClamp(Client* client, uint64_t id):
  Object(client, "SpaceCenter::LaunchClamp", id) {}

inline void SpaceCenter::LaunchClamp::release() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "LaunchClamp_Release", _args);
}

inline SpaceCenter::Part SpaceCenter::LaunchClamp::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "LaunchClamp_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::LaunchClamp::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "LaunchClamp_get_Part", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::LaunchClamp::release_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "LaunchClamp_Release", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::LaunchClamp::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "LaunchClamp_get_Part", _args);
}

inline SpaceCenter::LaunchSite::LaunchSite(Client* client, uint64_t id):
  Object(client, "SpaceCenter::LaunchSite", id) {}

inline SpaceCenter::CelestialBody SpaceCenter::LaunchSite::body() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "LaunchSite_get_Body", _args);
  SpaceCenter::CelestialBody _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::EditorFacility SpaceCenter::LaunchSite::editor_facility() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "LaunchSite_get_EditorFacility", _args);
  SpaceCenter::EditorFacility _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::LaunchSite::name() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "LaunchSite_get_Name", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<SpaceCenter::CelestialBody> SpaceCenter::LaunchSite::body_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::CelestialBody>(this->_client, this->_client->build_call("SpaceCenter", "LaunchSite_get_Body", _args));
}

inline ::krpc::Stream<SpaceCenter::EditorFacility> SpaceCenter::LaunchSite::editor_facility_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::EditorFacility>(this->_client, this->_client->build_call("SpaceCenter", "LaunchSite_get_EditorFacility", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::LaunchSite::name_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "LaunchSite_get_Name", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::LaunchSite::body_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "LaunchSite_get_Body", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::LaunchSite::editor_facility_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "LaunchSite_get_EditorFacility", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::LaunchSite::name_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "LaunchSite_get_Name", _args);
}

inline SpaceCenter::Leg::Leg(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Leg", id) {}

inline bool SpaceCenter::Leg::deployable() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Leg_get_Deployable", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Leg::deployed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Leg_get_Deployed", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Leg::set_deployed(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Leg_set_Deployed", _args);
}

inline bool SpaceCenter::Leg::is_grounded() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Leg_get_IsGrounded", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::Leg::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Leg_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::LegState SpaceCenter::Leg::state() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Leg_get_State", _args);
  SpaceCenter::LegState _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<bool> SpaceCenter::Leg::deployable_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Leg_get_Deployable", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Leg::deployed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Leg_get_Deployed", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Leg::is_grounded_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Leg_get_IsGrounded", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Leg::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Leg_get_Part", _args));
}

inline ::krpc::Stream<SpaceCenter::LegState> SpaceCenter::Leg::state_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::LegState>(this->_client, this->_client->build_call("SpaceCenter", "Leg_get_State", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Leg::deployable_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Leg_get_Deployable", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Leg::deployed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Leg_get_Deployed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Leg::set_deployed_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Leg_set_Deployed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Leg::is_grounded_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Leg_get_IsGrounded", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Leg::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Leg_get_Part", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Leg::state_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Leg_get_State", _args);
}

inline SpaceCenter::Light::Light(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Light", id) {}

inline bool SpaceCenter::Light::active() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Light_get_Active", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Light::set_active(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Light_set_Active", _args);
}

inline bool SpaceCenter::Light::blink() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Light_get_Blink", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Light::set_blink(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Light_set_Blink", _args);
}

inline float SpaceCenter::Light::blink_rate() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Light_get_BlinkRate", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Light::set_blink_rate(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Light_set_BlinkRate", _args);
}

inline std::tuple<float, float, float> SpaceCenter::Light::color() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Light_get_Color", _args);
  std::tuple<float, float, float> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Light::set_color(std::tuple<float, float, float> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Light_set_Color", _args);
}

inline SpaceCenter::Part SpaceCenter::Light::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Light_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Light::power_usage() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Light_get_PowerUsage", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<bool> SpaceCenter::Light::active_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Light_get_Active", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Light::blink_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Light_get_Blink", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Light::blink_rate_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Light_get_BlinkRate", _args));
}

inline ::krpc::Stream<std::tuple<float, float, float>> SpaceCenter::Light::color_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<float, float, float>>(this->_client, this->_client->build_call("SpaceCenter", "Light_get_Color", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Light::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Light_get_Part", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Light::power_usage_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Light_get_PowerUsage", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Light::active_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Light_get_Active", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Light::set_active_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Light_set_Active", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Light::blink_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Light_get_Blink", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Light::set_blink_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Light_set_Blink", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Light::blink_rate_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Light_get_BlinkRate", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Light::set_blink_rate_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Light_set_BlinkRate", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Light::color_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Light_get_Color", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Light::set_color_call(std::tuple<float, float, float> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Light_set_Color", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Light::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Light_get_Part", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Light::power_usage_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Light_get_PowerUsage", _args);
}

inline SpaceCenter::Module::Module(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Module", id) {}

inline std::string SpaceCenter::Module::get_field(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_GetField", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Module::get_field_by_id(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_GetFieldById", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Module::has_action(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_HasAction", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Module::has_action_with_id(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_HasActionWithId", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Module::has_event(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_HasEvent", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Module::has_event_with_id(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_HasEventWithId", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Module::has_field(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_HasField", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Module::has_field_with_id(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_HasFieldWithId", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Module::reset_field(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  this->_client->invoke("SpaceCenter", "Module_ResetField", _args);
}

inline void SpaceCenter::Module::reset_field_by_id(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  this->_client->invoke("SpaceCenter", "Module_ResetFieldById", _args);
}

inline void SpaceCenter::Module::set_action(std::string name, bool value = true) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Module_SetAction", _args);
}

inline void SpaceCenter::Module::set_action_by_id(std::string id, bool value = true) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Module_SetActionById", _args);
}

inline void SpaceCenter::Module::set_field_bool(std::string name, bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Module_SetFieldBool", _args);
}

inline void SpaceCenter::Module::set_field_bool_by_id(std::string id, bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Module_SetFieldBoolById", _args);
}

inline void SpaceCenter::Module::set_field_float(std::string name, float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Module_SetFieldFloat", _args);
}

inline void SpaceCenter::Module::set_field_float_by_id(std::string id, float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Module_SetFieldFloatById", _args);
}

inline void SpaceCenter::Module::set_field_int(std::string name, int32_t value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Module_SetFieldInt", _args);
}

inline void SpaceCenter::Module::set_field_int_by_id(std::string id, int32_t value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Module_SetFieldIntById", _args);
}

inline void SpaceCenter::Module::set_field_string(std::string name, std::string value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Module_SetFieldString", _args);
}

inline void SpaceCenter::Module::set_field_string_by_id(std::string id, std::string value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Module_SetFieldStringById", _args);
}

inline void SpaceCenter::Module::trigger_event(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  this->_client->invoke("SpaceCenter", "Module_TriggerEvent", _args);
}

inline void SpaceCenter::Module::trigger_event_by_id(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  this->_client->invoke("SpaceCenter", "Module_TriggerEventById", _args);
}

inline std::vector<std::string> SpaceCenter::Module::actions() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_get_Actions", _args);
  std::vector<std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<std::string> SpaceCenter::Module::actions_by_id() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_get_ActionsById", _args);
  std::vector<std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<std::string> SpaceCenter::Module::events() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_get_Events", _args);
  std::vector<std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<std::string> SpaceCenter::Module::events_by_id() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_get_EventsById", _args);
  std::vector<std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::map<std::string, std::string> SpaceCenter::Module::fields() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_get_Fields", _args);
  std::map<std::string, std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::map<std::string, std::string> SpaceCenter::Module::fields_by_id() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_get_FieldsById", _args);
  std::map<std::string, std::string> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Module::name() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_get_Name", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::Module::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Module_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<std::string> SpaceCenter::Module::get_field_stream(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Module_GetField", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Module::get_field_by_id_stream(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Module_GetFieldById", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Module::has_action_stream(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Module_HasAction", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Module::has_action_with_id_stream(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Module_HasActionWithId", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Module::has_event_stream(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Module_HasEvent", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Module::has_event_with_id_stream(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Module_HasEventWithId", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Module::has_field_stream(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Module_HasField", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Module::has_field_with_id_stream(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Module_HasFieldWithId", _args));
}

inline ::krpc::Stream<std::vector<std::string>> SpaceCenter::Module::actions_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<std::string>>(this->_client, this->_client->build_call("SpaceCenter", "Module_get_Actions", _args));
}

inline ::krpc::Stream<std::vector<std::string>> SpaceCenter::Module::actions_by_id_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<std::string>>(this->_client, this->_client->build_call("SpaceCenter", "Module_get_ActionsById", _args));
}

inline ::krpc::Stream<std::vector<std::string>> SpaceCenter::Module::events_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<std::string>>(this->_client, this->_client->build_call("SpaceCenter", "Module_get_Events", _args));
}

inline ::krpc::Stream<std::vector<std::string>> SpaceCenter::Module::events_by_id_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<std::string>>(this->_client, this->_client->build_call("SpaceCenter", "Module_get_EventsById", _args));
}

inline ::krpc::Stream<std::map<std::string, std::string>> SpaceCenter::Module::fields_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::map<std::string, std::string>>(this->_client, this->_client->build_call("SpaceCenter", "Module_get_Fields", _args));
}

inline ::krpc::Stream<std::map<std::string, std::string>> SpaceCenter::Module::fields_by_id_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::map<std::string, std::string>>(this->_client, this->_client->build_call("SpaceCenter", "Module_get_FieldsById", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Module::name_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Module_get_Name", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Module::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Module_get_Part", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::get_field_call(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  return this->_client->build_call("SpaceCenter", "Module_GetField", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::get_field_by_id_call(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  return this->_client->build_call("SpaceCenter", "Module_GetFieldById", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::has_action_call(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  return this->_client->build_call("SpaceCenter", "Module_HasAction", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::has_action_with_id_call(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  return this->_client->build_call("SpaceCenter", "Module_HasActionWithId", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::has_event_call(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  return this->_client->build_call("SpaceCenter", "Module_HasEvent", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::has_event_with_id_call(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  return this->_client->build_call("SpaceCenter", "Module_HasEventWithId", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::has_field_call(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  return this->_client->build_call("SpaceCenter", "Module_HasField", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::has_field_with_id_call(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  return this->_client->build_call("SpaceCenter", "Module_HasFieldWithId", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::reset_field_call(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  return this->_client->build_call("SpaceCenter", "Module_ResetField", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::reset_field_by_id_call(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  return this->_client->build_call("SpaceCenter", "Module_ResetFieldById", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::set_action_call(std::string name, bool value = true) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Module_SetAction", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::set_action_by_id_call(std::string id, bool value = true) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Module_SetActionById", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::set_field_bool_call(std::string name, bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Module_SetFieldBool", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::set_field_bool_by_id_call(std::string id, bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Module_SetFieldBoolById", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::set_field_float_call(std::string name, float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Module_SetFieldFloat", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::set_field_float_by_id_call(std::string id, float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Module_SetFieldFloatById", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::set_field_int_call(std::string name, int32_t value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Module_SetFieldInt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::set_field_int_by_id_call(std::string id, int32_t value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Module_SetFieldIntById", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::set_field_string_call(std::string name, std::string value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Module_SetFieldString", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::set_field_string_by_id_call(std::string id, std::string value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Module_SetFieldStringById", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::trigger_event_call(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  return this->_client->build_call("SpaceCenter", "Module_TriggerEvent", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::trigger_event_by_id_call(std::string id) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(id));
  return this->_client->build_call("SpaceCenter", "Module_TriggerEventById", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::actions_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Module_get_Actions", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::actions_by_id_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Module_get_ActionsById", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::events_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Module_get_Events", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::events_by_id_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Module_get_EventsById", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::fields_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Module_get_Fields", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::fields_by_id_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Module_get_FieldsById", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::name_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Module_get_Name", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Module::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Module_get_Part", _args);
}

inline SpaceCenter::Node::Node(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Node", id) {}

inline std::tuple<double, double, double> SpaceCenter::Node::burn_vector(SpaceCenter::ReferenceFrame reference_frame = SpaceCenter::ReferenceFrame()) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_BurnVector", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Node::direction(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_Direction", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Node::position(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_Position", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Node::remaining_burn_vector(SpaceCenter::ReferenceFrame reference_frame = SpaceCenter::ReferenceFrame()) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_RemainingBurnVector", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Node::remove() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Node_Remove", _args);
}

inline double SpaceCenter::Node::delta_v() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_get_DeltaV", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Node::set_delta_v(double value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Node_set_DeltaV", _args);
}

inline double SpaceCenter::Node::normal() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_get_Normal", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Node::set_normal(double value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Node_set_Normal", _args);
}

inline SpaceCenter::Orbit SpaceCenter::Node::orbit() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_get_Orbit", _args);
  SpaceCenter::Orbit _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ReferenceFrame SpaceCenter::Node::orbital_reference_frame() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_get_OrbitalReferenceFrame", _args);
  SpaceCenter::ReferenceFrame _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Node::prograde() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_get_Prograde", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Node::set_prograde(double value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Node_set_Prograde", _args);
}

inline double SpaceCenter::Node::radial() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_get_Radial", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Node::set_radial(double value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Node_set_Radial", _args);
}

inline SpaceCenter::ReferenceFrame SpaceCenter::Node::reference_frame() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_get_ReferenceFrame", _args);
  SpaceCenter::ReferenceFrame _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Node::remaining_delta_v() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_get_RemainingDeltaV", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Node::time_to() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_get_TimeTo", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Node::ut() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Node_get_UT", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Node::set_ut(double value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Node_set_UT", _args);
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Node::burn_vector_stream(SpaceCenter::ReferenceFrame reference_frame = SpaceCenter::ReferenceFrame()) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Node_BurnVector", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Node::direction_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Node_Direction", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Node::position_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Node_Position", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Node::remaining_burn_vector_stream(SpaceCenter::ReferenceFrame reference_frame = SpaceCenter::ReferenceFrame()) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Node_RemainingBurnVector", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Node::delta_v_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Node_get_DeltaV", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Node::normal_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Node_get_Normal", _args));
}

inline ::krpc::Stream<SpaceCenter::Orbit> SpaceCenter::Node::orbit_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Orbit>(this->_client, this->_client->build_call("SpaceCenter", "Node_get_Orbit", _args));
}

inline ::krpc::Stream<SpaceCenter::ReferenceFrame> SpaceCenter::Node::orbital_reference_frame_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ReferenceFrame>(this->_client, this->_client->build_call("SpaceCenter", "Node_get_OrbitalReferenceFrame", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Node::prograde_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Node_get_Prograde", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Node::radial_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Node_get_Radial", _args));
}

inline ::krpc::Stream<SpaceCenter::ReferenceFrame> SpaceCenter::Node::reference_frame_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ReferenceFrame>(this->_client, this->_client->build_call("SpaceCenter", "Node_get_ReferenceFrame", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Node::remaining_delta_v_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Node_get_RemainingDeltaV", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Node::time_to_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Node_get_TimeTo", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Node::ut_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Node_get_UT", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::burn_vector_call(SpaceCenter::ReferenceFrame reference_frame = SpaceCenter::ReferenceFrame()) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Node_BurnVector", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::direction_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Node_Direction", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::position_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Node_Position", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::remaining_burn_vector_call(SpaceCenter::ReferenceFrame reference_frame = SpaceCenter::ReferenceFrame()) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Node_RemainingBurnVector", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::remove_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Node_Remove", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::delta_v_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Node_get_DeltaV", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::set_delta_v_call(double value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Node_set_DeltaV", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::normal_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Node_get_Normal", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::set_normal_call(double value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Node_set_Normal", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::orbit_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Node_get_Orbit", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::orbital_reference_frame_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Node_get_OrbitalReferenceFrame", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::prograde_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Node_get_Prograde", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::set_prograde_call(double value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Node_set_Prograde", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::radial_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Node_get_Radial", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::set_radial_call(double value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Node_set_Radial", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::reference_frame_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Node_get_ReferenceFrame", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::remaining_delta_v_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Node_get_RemainingDeltaV", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::time_to_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Node_get_TimeTo", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::ut_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Node_get_UT", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Node::set_ut_call(double value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Node_set_UT", _args);
}

inline SpaceCenter::Orbit::Orbit(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Orbit", id) {}

inline double SpaceCenter::Orbit::distance_at_closest_approach(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_DistanceAtClosestApproach", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::eccentric_anomaly_at_ut(double ut) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_EccentricAnomalyAtUT", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<std::vector<double> > SpaceCenter::Orbit::list_closest_approaches(SpaceCenter::Orbit target, int32_t orbits) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  _args.push_back(encoder::encode(orbits));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_ListClosestApproaches", _args);
  std::vector<std::vector<double> > _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::mean_anomaly_at_ut(double ut) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_MeanAnomalyAtUT", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::orbital_speed_at(double time) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(time));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_OrbitalSpeedAt", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Orbit::position_at(double ut, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_PositionAt", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::radius_at(double ut) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_RadiusAt", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::radius_at_true_anomaly(double true_anomaly) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(true_anomaly));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_RadiusAtTrueAnomaly", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::relative_inclination(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_RelativeInclination", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::time_of_closest_approach(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_TimeOfClosestApproach", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::true_anomaly_at_an(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_TrueAnomalyAtAN", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::true_anomaly_at_dn(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_TrueAnomalyAtDN", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::true_anomaly_at_radius(double radius) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(radius));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_TrueAnomalyAtRadius", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::true_anomaly_at_ut(double ut) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_TrueAnomalyAtUT", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::ut_at_true_anomaly(double true_anomaly) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(true_anomaly));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_UTAtTrueAnomaly", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::apoapsis() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_Apoapsis", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::apoapsis_altitude() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_ApoapsisAltitude", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::argument_of_periapsis() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_ArgumentOfPeriapsis", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::CelestialBody SpaceCenter::Orbit::body() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_Body", _args);
  SpaceCenter::CelestialBody _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::eccentric_anomaly() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_EccentricAnomaly", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::eccentricity() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_Eccentricity", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::epoch() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_Epoch", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::inclination() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_Inclination", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::longitude_of_ascending_node() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_LongitudeOfAscendingNode", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::mean_anomaly() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_MeanAnomaly", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::mean_anomaly_at_epoch() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_MeanAnomalyAtEpoch", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Orbit SpaceCenter::Orbit::next_orbit() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_NextOrbit", _args);
  SpaceCenter::Orbit _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::orbital_speed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_OrbitalSpeed", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::periapsis() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_Periapsis", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::periapsis_altitude() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_PeriapsisAltitude", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::period() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_Period", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::radius() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_Radius", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::semi_major_axis() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_SemiMajorAxis", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::semi_minor_axis() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_SemiMinorAxis", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::speed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_Speed", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::time_to_apoapsis() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_TimeToApoapsis", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::time_to_periapsis() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_TimeToPeriapsis", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::time_to_soi_change() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_TimeToSOIChange", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Orbit::true_anomaly() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Orbit_get_TrueAnomaly", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Orbit::reference_plane_direction(Client& _client, SpaceCenter::ReferenceFrame reference_frame) {
    std::vector<std::string> _args;
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = _client.invoke("SpaceCenter", "Orbit_static_ReferencePlaneDirection", _args);
    std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, &_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Orbit::reference_plane_normal(Client& _client, SpaceCenter::ReferenceFrame reference_frame) {
    std::vector<std::string> _args;
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = _client.invoke("SpaceCenter", "Orbit_static_ReferencePlaneNormal", _args);
    std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, &_client);
  return _result;
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::distance_at_closest_approach_stream(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_DistanceAtClosestApproach", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::eccentric_anomaly_at_ut_stream(double ut) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_EccentricAnomalyAtUT", _args));
}

inline ::krpc::Stream<std::vector<std::vector<double> >> SpaceCenter::Orbit::list_closest_approaches_stream(SpaceCenter::Orbit target, int32_t orbits) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  _args.push_back(encoder::encode(orbits));
  return ::krpc::Stream<std::vector<std::vector<double> >>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_ListClosestApproaches", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::mean_anomaly_at_ut_stream(double ut) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_MeanAnomalyAtUT", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::orbital_speed_at_stream(double time) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(time));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_OrbitalSpeedAt", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Orbit::position_at_stream(double ut, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_PositionAt", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::radius_at_stream(double ut) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_RadiusAt", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::radius_at_true_anomaly_stream(double true_anomaly) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(true_anomaly));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_RadiusAtTrueAnomaly", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::relative_inclination_stream(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_RelativeInclination", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::time_of_closest_approach_stream(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_TimeOfClosestApproach", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::true_anomaly_at_an_stream(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_TrueAnomalyAtAN", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::true_anomaly_at_dn_stream(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_TrueAnomalyAtDN", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::true_anomaly_at_radius_stream(double radius) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(radius));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_TrueAnomalyAtRadius", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::true_anomaly_at_ut_stream(double ut) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_TrueAnomalyAtUT", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::ut_at_true_anomaly_stream(double true_anomaly) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(true_anomaly));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_UTAtTrueAnomaly", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::apoapsis_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_Apoapsis", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::apoapsis_altitude_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_ApoapsisAltitude", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::argument_of_periapsis_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_ArgumentOfPeriapsis", _args));
}

inline ::krpc::Stream<SpaceCenter::CelestialBody> SpaceCenter::Orbit::body_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::CelestialBody>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_Body", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::eccentric_anomaly_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_EccentricAnomaly", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::eccentricity_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_Eccentricity", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::epoch_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_Epoch", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::inclination_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_Inclination", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::longitude_of_ascending_node_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_LongitudeOfAscendingNode", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::mean_anomaly_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_MeanAnomaly", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::mean_anomaly_at_epoch_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_MeanAnomalyAtEpoch", _args));
}

inline ::krpc::Stream<SpaceCenter::Orbit> SpaceCenter::Orbit::next_orbit_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Orbit>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_NextOrbit", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::orbital_speed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_OrbitalSpeed", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::periapsis_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_Periapsis", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::periapsis_altitude_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_PeriapsisAltitude", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::period_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_Period", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::radius_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_Radius", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::semi_major_axis_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_SemiMajorAxis", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::semi_minor_axis_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_SemiMinorAxis", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::speed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_Speed", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::time_to_apoapsis_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_TimeToApoapsis", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::time_to_periapsis_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_TimeToPeriapsis", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::time_to_soi_change_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_TimeToSOIChange", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Orbit::true_anomaly_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Orbit_get_TrueAnomaly", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Orbit::reference_plane_direction_stream(Client& _client, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(&_client, _client.build_call("SpaceCenter", "Orbit_static_ReferencePlaneDirection", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Orbit::reference_plane_normal_stream(Client& _client, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(&_client, _client.build_call("SpaceCenter", "Orbit_static_ReferencePlaneNormal", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::distance_at_closest_approach_call(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  return this->_client->build_call("SpaceCenter", "Orbit_DistanceAtClosestApproach", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::eccentric_anomaly_at_ut_call(double ut) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  return this->_client->build_call("SpaceCenter", "Orbit_EccentricAnomalyAtUT", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::list_closest_approaches_call(SpaceCenter::Orbit target, int32_t orbits) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  _args.push_back(encoder::encode(orbits));
  return this->_client->build_call("SpaceCenter", "Orbit_ListClosestApproaches", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::mean_anomaly_at_ut_call(double ut) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  return this->_client->build_call("SpaceCenter", "Orbit_MeanAnomalyAtUT", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::orbital_speed_at_call(double time) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(time));
  return this->_client->build_call("SpaceCenter", "Orbit_OrbitalSpeedAt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::position_at_call(double ut, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Orbit_PositionAt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::radius_at_call(double ut) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  return this->_client->build_call("SpaceCenter", "Orbit_RadiusAt", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::radius_at_true_anomaly_call(double true_anomaly) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(true_anomaly));
  return this->_client->build_call("SpaceCenter", "Orbit_RadiusAtTrueAnomaly", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::relative_inclination_call(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  return this->_client->build_call("SpaceCenter", "Orbit_RelativeInclination", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::time_of_closest_approach_call(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  return this->_client->build_call("SpaceCenter", "Orbit_TimeOfClosestApproach", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::true_anomaly_at_an_call(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  return this->_client->build_call("SpaceCenter", "Orbit_TrueAnomalyAtAN", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::true_anomaly_at_dn_call(SpaceCenter::Orbit target) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(target));
  return this->_client->build_call("SpaceCenter", "Orbit_TrueAnomalyAtDN", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::true_anomaly_at_radius_call(double radius) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(radius));
  return this->_client->build_call("SpaceCenter", "Orbit_TrueAnomalyAtRadius", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::true_anomaly_at_ut_call(double ut) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(ut));
  return this->_client->build_call("SpaceCenter", "Orbit_TrueAnomalyAtUT", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::ut_at_true_anomaly_call(double true_anomaly) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(true_anomaly));
  return this->_client->build_call("SpaceCenter", "Orbit_UTAtTrueAnomaly", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::apoapsis_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_Apoapsis", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::apoapsis_altitude_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_ApoapsisAltitude", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::argument_of_periapsis_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_ArgumentOfPeriapsis", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::body_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_Body", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::eccentric_anomaly_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_EccentricAnomaly", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::eccentricity_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_Eccentricity", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::epoch_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_Epoch", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::inclination_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_Inclination", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::longitude_of_ascending_node_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_LongitudeOfAscendingNode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::mean_anomaly_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_MeanAnomaly", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::mean_anomaly_at_epoch_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_MeanAnomalyAtEpoch", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::next_orbit_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_NextOrbit", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::orbital_speed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_OrbitalSpeed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::periapsis_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_Periapsis", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::periapsis_altitude_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_PeriapsisAltitude", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::period_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_Period", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::radius_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_Radius", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::semi_major_axis_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_SemiMajorAxis", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::semi_minor_axis_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_SemiMinorAxis", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::speed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_Speed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::time_to_apoapsis_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_TimeToApoapsis", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::time_to_periapsis_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_TimeToPeriapsis", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::time_to_soi_change_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_TimeToSOIChange", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::true_anomaly_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Orbit_get_TrueAnomaly", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::reference_plane_direction_call(Client& _client, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(reference_frame));
  return _client.build_call("SpaceCenter", "Orbit_static_ReferencePlaneDirection", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Orbit::reference_plane_normal_call(Client& _client, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(reference_frame));
  return _client.build_call("SpaceCenter", "Orbit_static_ReferencePlaneNormal", _args);
}

inline SpaceCenter::Parachute::Parachute(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Parachute", id) {}

inline void SpaceCenter::Parachute::arm() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Parachute_Arm", _args);
}

inline void SpaceCenter::Parachute::cut() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Parachute_Cut", _args);
}

inline void SpaceCenter::Parachute::deploy() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  this->_client->invoke("SpaceCenter", "Parachute_Deploy", _args);
}

inline bool SpaceCenter::Parachute::armed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parachute_get_Armed", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Parachute::deploy_altitude() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parachute_get_DeployAltitude", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Parachute::set_deploy_altitude(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Parachute_set_DeployAltitude", _args);
}

inline float SpaceCenter::Parachute::deploy_min_pressure() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parachute_get_DeployMinPressure", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Parachute::set_deploy_min_pressure(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Parachute_set_DeployMinPressure", _args);
}

inline bool SpaceCenter::Parachute::deployed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parachute_get_Deployed", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::Parachute::part() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parachute_get_Part", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ParachuteState SpaceCenter::Parachute::state() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parachute_get_State", _args);
  SpaceCenter::ParachuteState _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<bool> SpaceCenter::Parachute::armed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Parachute_get_Armed", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Parachute::deploy_altitude_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Parachute_get_DeployAltitude", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Parachute::deploy_min_pressure_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Parachute_get_DeployMinPressure", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Parachute::deployed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Parachute_get_Deployed", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Parachute::part_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Parachute_get_Part", _args));
}

inline ::krpc::Stream<SpaceCenter::ParachuteState> SpaceCenter::Parachute::state_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ParachuteState>(this->_client, this->_client->build_call("SpaceCenter", "Parachute_get_State", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parachute::arm_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parachute_Arm", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parachute::cut_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parachute_Cut", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parachute::deploy_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parachute_Deploy", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parachute::armed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parachute_get_Armed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parachute::deploy_altitude_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parachute_get_DeployAltitude", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parachute::set_deploy_altitude_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Parachute_set_DeployAltitude", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parachute::deploy_min_pressure_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parachute_get_DeployMinPressure", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parachute::set_deploy_min_pressure_call(float value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Parachute_set_DeployMinPressure", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parachute::deployed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parachute_get_Deployed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parachute::part_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parachute_get_Part", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parachute::state_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parachute_get_State", _args);
}

inline SpaceCenter::Part::Part(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Part", id) {}

inline SpaceCenter::Force SpaceCenter::Part::add_force(std::tuple<double, double, double> force, std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(force));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_AddForce", _args);
  SpaceCenter::Force _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > SpaceCenter::Part::bounding_box(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_BoundingBox", _args);
  std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> > _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Part::center_of_mass(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_CenterOfMass", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Part::direction(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_Direction", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Part::instantaneous_force(std::tuple<double, double, double> force, std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(force));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  this->_client->invoke("SpaceCenter", "Part_InstantaneousForce", _args);
}

inline std::tuple<double, double, double> SpaceCenter::Part::position(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_Position", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double, double> SpaceCenter::Part::rotation(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_Rotation", _args);
  std::tuple<double, double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Part::velocity(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_Velocity", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Antenna SpaceCenter::Part::antenna() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Antenna", _args);
  SpaceCenter::Antenna _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::AutoStrutMode SpaceCenter::Part::auto_strut_mode() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_AutoStrutMode", _args);
  SpaceCenter::AutoStrutMode _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline uint32_t SpaceCenter::Part::available_seats() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_AvailableSeats", _args);
  uint32_t _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Part::axially_attached() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_AxiallyAttached", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::CargoBay SpaceCenter::Part::cargo_bay() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_CargoBay", _args);
  SpaceCenter::CargoBay _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ReferenceFrame SpaceCenter::Part::center_of_mass_reference_frame() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_CenterOfMassReferenceFrame", _args);
  SpaceCenter::ReferenceFrame _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Part> SpaceCenter::Part::children() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Children", _args);
  std::vector<SpaceCenter::Part> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ControlSurface SpaceCenter::Part::control_surface() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ControlSurface", _args);
  SpaceCenter::ControlSurface _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Part::cost() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Cost", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Part::crossfeed() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Crossfeed", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline int32_t SpaceCenter::Part::decouple_stage() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_DecoupleStage", _args);
  int32_t _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Decoupler SpaceCenter::Part::decoupler() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Decoupler", _args);
  SpaceCenter::Decoupler _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::DockingPort SpaceCenter::Part::docking_port() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_DockingPort", _args);
  SpaceCenter::DockingPort _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Part::dry_mass() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_DryMass", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Part::dynamic_pressure() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_DynamicPressure", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Engine SpaceCenter::Part::engine() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Engine", _args);
  SpaceCenter::Engine _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Experiment SpaceCenter::Part::experiment() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Experiment", _args);
  SpaceCenter::Experiment _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Experiment> SpaceCenter::Part::experiments() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Experiments", _args);
  std::vector<SpaceCenter::Experiment> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Fairing SpaceCenter::Part::fairing() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Fairing", _args);
  SpaceCenter::Fairing _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Part::flag_url() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_FlagURL", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Part::set_flag_url(std::string value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Part_set_FlagURL", _args);
}

inline std::vector<SpaceCenter::Part> SpaceCenter::Part::fuel_lines_from() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_FuelLinesFrom", _args);
  std::vector<SpaceCenter::Part> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Part> SpaceCenter::Part::fuel_lines_to() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_FuelLinesTo", _args);
  std::vector<SpaceCenter::Part> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Part::set_glow(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Part_set_Glow", _args);
}

inline std::tuple<double, double, double> SpaceCenter::Part::highlight_color() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_HighlightColor", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Part::set_highlight_color(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Part_set_HighlightColor", _args);
}

inline bool SpaceCenter::Part::highlighted() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Highlighted", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Part::set_highlighted(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Part_set_Highlighted", _args);
}

inline double SpaceCenter::Part::impact_tolerance() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ImpactTolerance", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<double> SpaceCenter::Part::inertia_tensor() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_InertiaTensor", _args);
  std::vector<double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Intake SpaceCenter::Part::intake() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Intake", _args);
  SpaceCenter::Intake _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Part::is_fuel_line() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_IsFuelLine", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::LaunchClamp SpaceCenter::Part::launch_clamp() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_LaunchClamp", _args);
  SpaceCenter::LaunchClamp _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Leg SpaceCenter::Part::leg() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Leg", _args);
  SpaceCenter::Leg _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Light SpaceCenter::Part::light() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Light", _args);
  SpaceCenter::Light _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Part::mass() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Mass", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Part::massless() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Massless", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Part::max_skin_temperature() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_MaxSkinTemperature", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Part::max_temperature() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_MaxTemperature", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Module> SpaceCenter::Part::modules() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Modules", _args);
  std::vector<SpaceCenter::Module> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::tuple<double, double, double> SpaceCenter::Part::moment_of_inertia() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_MomentOfInertia", _args);
  std::tuple<double, double, double> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Part::name() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Name", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Parachute SpaceCenter::Part::parachute() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Parachute", _args);
  SpaceCenter::Parachute _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::Part::parent() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Parent", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Part::radially_attached() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_RadiallyAttached", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Radiator SpaceCenter::Part::radiator() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Radiator", _args);
  SpaceCenter::Radiator _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::RCS SpaceCenter::Part::rcs() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_RCS", _args);
  SpaceCenter::RCS _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ReactionWheel SpaceCenter::Part::reaction_wheel() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ReactionWheel", _args);
  SpaceCenter::ReactionWheel _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ReferenceFrame SpaceCenter::Part::reference_frame() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ReferenceFrame", _args);
  SpaceCenter::ReferenceFrame _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ResourceConverter SpaceCenter::Part::resource_converter() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ResourceConverter", _args);
  SpaceCenter::ResourceConverter _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ResourceDrain SpaceCenter::Part::resource_drain() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ResourceDrain", _args);
  SpaceCenter::ResourceDrain _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::ResourceHarvester SpaceCenter::Part::resource_harvester() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ResourceHarvester", _args);
  SpaceCenter::ResourceHarvester _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Resources SpaceCenter::Part::resources() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Resources", _args);
  SpaceCenter::Resources _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::RoboticController SpaceCenter::Part::robotic_controller() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_RoboticController", _args);
  SpaceCenter::RoboticController _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::RoboticHinge SpaceCenter::Part::robotic_hinge() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_RoboticHinge", _args);
  SpaceCenter::RoboticHinge _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::RoboticPiston SpaceCenter::Part::robotic_piston() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_RoboticPiston", _args);
  SpaceCenter::RoboticPiston _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::RoboticRotation SpaceCenter::Part::robotic_rotation() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_RoboticRotation", _args);
  SpaceCenter::RoboticRotation _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::RoboticRotor SpaceCenter::Part::robotic_rotor() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_RoboticRotor", _args);
  SpaceCenter::RoboticRotor _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Sensor SpaceCenter::Part::sensor() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Sensor", _args);
  SpaceCenter::Sensor _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline bool SpaceCenter::Part::shielded() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Shielded", _args);
  bool _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline double SpaceCenter::Part::skin_temperature() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_SkinTemperature", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::SolarPanel SpaceCenter::Part::solar_panel() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_SolarPanel", _args);
  SpaceCenter::SolarPanel _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline int32_t SpaceCenter::Part::stage() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Stage", _args);
  int32_t _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Part::tag() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Tag", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Part::set_tag(std::string value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Part_set_Tag", _args);
}

inline double SpaceCenter::Part::temperature() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Temperature", _args);
  double _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Part::thermal_conduction_flux() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ThermalConductionFlux", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Part::thermal_convection_flux() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ThermalConvectionFlux", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Part::thermal_internal_flux() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ThermalInternalFlux", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Part::thermal_mass() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ThermalMass", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Part::thermal_radiation_flux() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ThermalRadiationFlux", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Part::thermal_resource_mass() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ThermalResourceMass", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Part::thermal_skin_mass() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ThermalSkinMass", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline float SpaceCenter::Part::thermal_skin_to_internal_flux() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_ThermalSkinToInternalFlux", _args);
  float _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::string SpaceCenter::Part::title() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Title", _args);
  std::string _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Vessel SpaceCenter::Part::vessel() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Vessel", _args);
  SpaceCenter::Vessel _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Wheel SpaceCenter::Part::wheel() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Part_get_Wheel", _args);
  SpaceCenter::Wheel _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<SpaceCenter::Force> SpaceCenter::Part::add_force_stream(std::tuple<double, double, double> force, std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(force));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<SpaceCenter::Force>(this->_client, this->_client->build_call("SpaceCenter", "Part_AddForce", _args));
}

inline ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >> SpaceCenter::Part::bounding_box_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<std::tuple<double, double, double>, std::tuple<double, double, double> >>(this->_client, this->_client->build_call("SpaceCenter", "Part_BoundingBox", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Part::center_of_mass_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Part_CenterOfMass", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Part::direction_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Part_Direction", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Part::position_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Part_Position", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double, double>> SpaceCenter::Part::rotation_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Part_Rotation", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Part::velocity_stream(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Part_Velocity", _args));
}

inline ::krpc::Stream<SpaceCenter::Antenna> SpaceCenter::Part::antenna_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Antenna>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Antenna", _args));
}

inline ::krpc::Stream<SpaceCenter::AutoStrutMode> SpaceCenter::Part::auto_strut_mode_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::AutoStrutMode>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_AutoStrutMode", _args));
}

inline ::krpc::Stream<uint32_t> SpaceCenter::Part::available_seats_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<uint32_t>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_AvailableSeats", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Part::axially_attached_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_AxiallyAttached", _args));
}

inline ::krpc::Stream<SpaceCenter::CargoBay> SpaceCenter::Part::cargo_bay_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::CargoBay>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_CargoBay", _args));
}

inline ::krpc::Stream<SpaceCenter::ReferenceFrame> SpaceCenter::Part::center_of_mass_reference_frame_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ReferenceFrame>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_CenterOfMassReferenceFrame", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Part>> SpaceCenter::Part::children_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Part>>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Children", _args));
}

inline ::krpc::Stream<SpaceCenter::ControlSurface> SpaceCenter::Part::control_surface_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ControlSurface>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ControlSurface", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Part::cost_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Cost", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Part::crossfeed_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Crossfeed", _args));
}

inline ::krpc::Stream<int32_t> SpaceCenter::Part::decouple_stage_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<int32_t>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_DecoupleStage", _args));
}

inline ::krpc::Stream<SpaceCenter::Decoupler> SpaceCenter::Part::decoupler_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Decoupler>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Decoupler", _args));
}

inline ::krpc::Stream<SpaceCenter::DockingPort> SpaceCenter::Part::docking_port_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::DockingPort>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_DockingPort", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Part::dry_mass_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_DryMass", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Part::dynamic_pressure_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_DynamicPressure", _args));
}

inline ::krpc::Stream<SpaceCenter::Engine> SpaceCenter::Part::engine_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Engine>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Engine", _args));
}

inline ::krpc::Stream<SpaceCenter::Experiment> SpaceCenter::Part::experiment_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Experiment>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Experiment", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Experiment>> SpaceCenter::Part::experiments_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Experiment>>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Experiments", _args));
}

inline ::krpc::Stream<SpaceCenter::Fairing> SpaceCenter::Part::fairing_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Fairing>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Fairing", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Part::flag_url_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_FlagURL", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Part>> SpaceCenter::Part::fuel_lines_from_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Part>>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_FuelLinesFrom", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Part>> SpaceCenter::Part::fuel_lines_to_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Part>>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_FuelLinesTo", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Part::highlight_color_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_HighlightColor", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Part::highlighted_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Highlighted", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Part::impact_tolerance_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ImpactTolerance", _args));
}

inline ::krpc::Stream<std::vector<double>> SpaceCenter::Part::inertia_tensor_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<double>>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_InertiaTensor", _args));
}

inline ::krpc::Stream<SpaceCenter::Intake> SpaceCenter::Part::intake_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Intake>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Intake", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Part::is_fuel_line_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_IsFuelLine", _args));
}

inline ::krpc::Stream<SpaceCenter::LaunchClamp> SpaceCenter::Part::launch_clamp_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::LaunchClamp>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_LaunchClamp", _args));
}

inline ::krpc::Stream<SpaceCenter::Leg> SpaceCenter::Part::leg_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Leg>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Leg", _args));
}

inline ::krpc::Stream<SpaceCenter::Light> SpaceCenter::Part::light_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Light>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Light", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Part::mass_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Mass", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Part::massless_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Massless", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Part::max_skin_temperature_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_MaxSkinTemperature", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Part::max_temperature_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_MaxTemperature", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Module>> SpaceCenter::Part::modules_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Module>>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Modules", _args));
}

inline ::krpc::Stream<std::tuple<double, double, double>> SpaceCenter::Part::moment_of_inertia_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::tuple<double, double, double>>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_MomentOfInertia", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Part::name_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Name", _args));
}

inline ::krpc::Stream<SpaceCenter::Parachute> SpaceCenter::Part::parachute_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Parachute>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Parachute", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Part::parent_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Parent", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Part::radially_attached_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_RadiallyAttached", _args));
}

inline ::krpc::Stream<SpaceCenter::Radiator> SpaceCenter::Part::radiator_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Radiator>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Radiator", _args));
}

inline ::krpc::Stream<SpaceCenter::RCS> SpaceCenter::Part::rcs_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::RCS>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_RCS", _args));
}

inline ::krpc::Stream<SpaceCenter::ReactionWheel> SpaceCenter::Part::reaction_wheel_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ReactionWheel>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ReactionWheel", _args));
}

inline ::krpc::Stream<SpaceCenter::ReferenceFrame> SpaceCenter::Part::reference_frame_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ReferenceFrame>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ReferenceFrame", _args));
}

inline ::krpc::Stream<SpaceCenter::ResourceConverter> SpaceCenter::Part::resource_converter_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ResourceConverter>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ResourceConverter", _args));
}

inline ::krpc::Stream<SpaceCenter::ResourceDrain> SpaceCenter::Part::resource_drain_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ResourceDrain>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ResourceDrain", _args));
}

inline ::krpc::Stream<SpaceCenter::ResourceHarvester> SpaceCenter::Part::resource_harvester_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::ResourceHarvester>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ResourceHarvester", _args));
}

inline ::krpc::Stream<SpaceCenter::Resources> SpaceCenter::Part::resources_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Resources>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Resources", _args));
}

inline ::krpc::Stream<SpaceCenter::RoboticController> SpaceCenter::Part::robotic_controller_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::RoboticController>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_RoboticController", _args));
}

inline ::krpc::Stream<SpaceCenter::RoboticHinge> SpaceCenter::Part::robotic_hinge_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::RoboticHinge>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_RoboticHinge", _args));
}

inline ::krpc::Stream<SpaceCenter::RoboticPiston> SpaceCenter::Part::robotic_piston_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::RoboticPiston>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_RoboticPiston", _args));
}

inline ::krpc::Stream<SpaceCenter::RoboticRotation> SpaceCenter::Part::robotic_rotation_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::RoboticRotation>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_RoboticRotation", _args));
}

inline ::krpc::Stream<SpaceCenter::RoboticRotor> SpaceCenter::Part::robotic_rotor_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::RoboticRotor>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_RoboticRotor", _args));
}

inline ::krpc::Stream<SpaceCenter::Sensor> SpaceCenter::Part::sensor_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Sensor>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Sensor", _args));
}

inline ::krpc::Stream<bool> SpaceCenter::Part::shielded_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<bool>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Shielded", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Part::skin_temperature_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_SkinTemperature", _args));
}

inline ::krpc::Stream<SpaceCenter::SolarPanel> SpaceCenter::Part::solar_panel_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::SolarPanel>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_SolarPanel", _args));
}

inline ::krpc::Stream<int32_t> SpaceCenter::Part::stage_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<int32_t>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Stage", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Part::tag_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Tag", _args));
}

inline ::krpc::Stream<double> SpaceCenter::Part::temperature_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<double>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Temperature", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Part::thermal_conduction_flux_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ThermalConductionFlux", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Part::thermal_convection_flux_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ThermalConvectionFlux", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Part::thermal_internal_flux_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ThermalInternalFlux", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Part::thermal_mass_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ThermalMass", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Part::thermal_radiation_flux_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ThermalRadiationFlux", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Part::thermal_resource_mass_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ThermalResourceMass", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Part::thermal_skin_mass_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ThermalSkinMass", _args));
}

inline ::krpc::Stream<float> SpaceCenter::Part::thermal_skin_to_internal_flux_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<float>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_ThermalSkinToInternalFlux", _args));
}

inline ::krpc::Stream<std::string> SpaceCenter::Part::title_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::string>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Title", _args));
}

inline ::krpc::Stream<SpaceCenter::Vessel> SpaceCenter::Part::vessel_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Vessel>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Vessel", _args));
}

inline ::krpc::Stream<SpaceCenter::Wheel> SpaceCenter::Part::wheel_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Wheel>(this->_client, this->_client->build_call("SpaceCenter", "Part_get_Wheel", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::add_force_call(std::tuple<double, double, double> force, std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(force));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Part_AddForce", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::bounding_box_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Part_BoundingBox", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::center_of_mass_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Part_CenterOfMass", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::direction_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Part_Direction", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::instantaneous_force_call(std::tuple<double, double, double> force, std::tuple<double, double, double> position, SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(force));
  _args.push_back(encoder::encode(position));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Part_InstantaneousForce", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::position_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Part_Position", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::rotation_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Part_Rotation", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::velocity_call(SpaceCenter::ReferenceFrame reference_frame) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(reference_frame));
  return this->_client->build_call("SpaceCenter", "Part_Velocity", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::antenna_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Antenna", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::auto_strut_mode_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_AutoStrutMode", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::available_seats_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_AvailableSeats", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::axially_attached_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_AxiallyAttached", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::cargo_bay_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_CargoBay", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::center_of_mass_reference_frame_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_CenterOfMassReferenceFrame", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::children_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Children", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::control_surface_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ControlSurface", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::cost_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Cost", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::crossfeed_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Crossfeed", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::decouple_stage_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_DecoupleStage", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::decoupler_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Decoupler", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::docking_port_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_DockingPort", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::dry_mass_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_DryMass", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::dynamic_pressure_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_DynamicPressure", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::engine_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Engine", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::experiment_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Experiment", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::experiments_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Experiments", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::fairing_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Fairing", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::flag_url_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_FlagURL", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::set_flag_url_call(std::string value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Part_set_FlagURL", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::fuel_lines_from_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_FuelLinesFrom", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::fuel_lines_to_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_FuelLinesTo", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::set_glow_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Part_set_Glow", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::highlight_color_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_HighlightColor", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::set_highlight_color_call(std::tuple<double, double, double> value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Part_set_HighlightColor", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::highlighted_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Highlighted", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::set_highlighted_call(bool value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Part_set_Highlighted", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::impact_tolerance_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ImpactTolerance", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::inertia_tensor_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_InertiaTensor", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::intake_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Intake", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::is_fuel_line_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_IsFuelLine", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::launch_clamp_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_LaunchClamp", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::leg_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Leg", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::light_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Light", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::mass_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Mass", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::massless_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Massless", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::max_skin_temperature_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_MaxSkinTemperature", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::max_temperature_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_MaxTemperature", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::modules_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Modules", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::moment_of_inertia_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_MomentOfInertia", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::name_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Name", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::parachute_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Parachute", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::parent_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Parent", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::radially_attached_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_RadiallyAttached", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::radiator_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Radiator", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::rcs_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_RCS", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::reaction_wheel_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ReactionWheel", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::reference_frame_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ReferenceFrame", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::resource_converter_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ResourceConverter", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::resource_drain_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ResourceDrain", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::resource_harvester_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ResourceHarvester", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::resources_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Resources", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::robotic_controller_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_RoboticController", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::robotic_hinge_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_RoboticHinge", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::robotic_piston_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_RoboticPiston", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::robotic_rotation_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_RoboticRotation", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::robotic_rotor_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_RoboticRotor", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::sensor_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Sensor", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::shielded_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Shielded", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::skin_temperature_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_SkinTemperature", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::solar_panel_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_SolarPanel", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::stage_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Stage", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::tag_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Tag", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::set_tag_call(std::string value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  return this->_client->build_call("SpaceCenter", "Part_set_Tag", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::temperature_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Temperature", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::thermal_conduction_flux_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ThermalConductionFlux", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::thermal_convection_flux_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ThermalConvectionFlux", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::thermal_internal_flux_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ThermalInternalFlux", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::thermal_mass_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ThermalMass", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::thermal_radiation_flux_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ThermalRadiationFlux", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::thermal_resource_mass_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ThermalResourceMass", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::thermal_skin_mass_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ThermalSkinMass", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::thermal_skin_to_internal_flux_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_ThermalSkinToInternalFlux", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::title_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Title", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::vessel_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Vessel", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Part::wheel_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Part_get_Wheel", _args);
}

inline SpaceCenter::Parts::Parts(Client* client, uint64_t id):
  Object(client, "SpaceCenter::Parts", id) {}

inline std::vector<SpaceCenter::Part> SpaceCenter::Parts::in_decouple_stage(int32_t stage) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(stage));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_InDecoupleStage", _args);
  std::vector<SpaceCenter::Part> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Part> SpaceCenter::Parts::in_stage(int32_t stage) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(stage));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_InStage", _args);
  std::vector<SpaceCenter::Part> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Module> SpaceCenter::Parts::modules_with_name(std::string module_name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(module_name));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_ModulesWithName", _args);
  std::vector<SpaceCenter::Module> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Part> SpaceCenter::Parts::with_module(std::string module_name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(module_name));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_WithModule", _args);
  std::vector<SpaceCenter::Part> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Part> SpaceCenter::Parts::with_name(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_WithName", _args);
  std::vector<SpaceCenter::Part> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Part> SpaceCenter::Parts::with_tag(std::string tag) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(tag));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_WithTag", _args);
  std::vector<SpaceCenter::Part> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Part> SpaceCenter::Parts::with_title(std::string title) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(title));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_WithTitle", _args);
  std::vector<SpaceCenter::Part> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Part> SpaceCenter::Parts::all() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_All", _args);
  std::vector<SpaceCenter::Part> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Antenna> SpaceCenter::Parts::antennas() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Antennas", _args);
  std::vector<SpaceCenter::Antenna> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::CargoBay> SpaceCenter::Parts::cargo_bays() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_CargoBays", _args);
  std::vector<SpaceCenter::CargoBay> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::ControlSurface> SpaceCenter::Parts::control_surfaces() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_ControlSurfaces", _args);
  std::vector<SpaceCenter::ControlSurface> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::Parts::controlling() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Controlling", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline void SpaceCenter::Parts::set_controlling(SpaceCenter::Part value) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(value));
  this->_client->invoke("SpaceCenter", "Parts_set_Controlling", _args);
}

inline std::vector<SpaceCenter::Decoupler> SpaceCenter::Parts::decouplers() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Decouplers", _args);
  std::vector<SpaceCenter::Decoupler> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::DockingPort> SpaceCenter::Parts::docking_ports() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_DockingPorts", _args);
  std::vector<SpaceCenter::DockingPort> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Engine> SpaceCenter::Parts::engines() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Engines", _args);
  std::vector<SpaceCenter::Engine> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Experiment> SpaceCenter::Parts::experiments() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Experiments", _args);
  std::vector<SpaceCenter::Experiment> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Fairing> SpaceCenter::Parts::fairings() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Fairings", _args);
  std::vector<SpaceCenter::Fairing> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Intake> SpaceCenter::Parts::intakes() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Intakes", _args);
  std::vector<SpaceCenter::Intake> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::LaunchClamp> SpaceCenter::Parts::launch_clamps() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_LaunchClamps", _args);
  std::vector<SpaceCenter::LaunchClamp> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Leg> SpaceCenter::Parts::legs() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Legs", _args);
  std::vector<SpaceCenter::Leg> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Light> SpaceCenter::Parts::lights() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Lights", _args);
  std::vector<SpaceCenter::Light> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Parachute> SpaceCenter::Parts::parachutes() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Parachutes", _args);
  std::vector<SpaceCenter::Parachute> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Radiator> SpaceCenter::Parts::radiators() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Radiators", _args);
  std::vector<SpaceCenter::Radiator> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::RCS> SpaceCenter::Parts::rcs() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_RCS", _args);
  std::vector<SpaceCenter::RCS> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::ReactionWheel> SpaceCenter::Parts::reaction_wheels() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_ReactionWheels", _args);
  std::vector<SpaceCenter::ReactionWheel> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::ResourceConverter> SpaceCenter::Parts::resource_converters() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_ResourceConverters", _args);
  std::vector<SpaceCenter::ResourceConverter> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::ResourceDrain> SpaceCenter::Parts::resource_drains() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_ResourceDrains", _args);
  std::vector<SpaceCenter::ResourceDrain> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::ResourceHarvester> SpaceCenter::Parts::resource_harvesters() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_ResourceHarvesters", _args);
  std::vector<SpaceCenter::ResourceHarvester> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::RoboticHinge> SpaceCenter::Parts::robotic_hinges() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_RoboticHinges", _args);
  std::vector<SpaceCenter::RoboticHinge> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::RoboticPiston> SpaceCenter::Parts::robotic_pistons() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_RoboticPistons", _args);
  std::vector<SpaceCenter::RoboticPiston> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::RoboticRotation> SpaceCenter::Parts::robotic_rotations() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_RoboticRotations", _args);
  std::vector<SpaceCenter::RoboticRotation> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::RoboticRotor> SpaceCenter::Parts::robotic_rotors() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_RoboticRotors", _args);
  std::vector<SpaceCenter::RoboticRotor> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline SpaceCenter::Part SpaceCenter::Parts::root() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Root", _args);
  SpaceCenter::Part _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Sensor> SpaceCenter::Parts::sensors() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Sensors", _args);
  std::vector<SpaceCenter::Sensor> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::SolarPanel> SpaceCenter::Parts::solar_panels() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_SolarPanels", _args);
  std::vector<SpaceCenter::SolarPanel> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline std::vector<SpaceCenter::Wheel> SpaceCenter::Parts::wheels() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  std::string _data = this->_client->invoke("SpaceCenter", "Parts_get_Wheels", _args);
  std::vector<SpaceCenter::Wheel> _result;
  decoder::decode(_result, _data, this->_client);
  return _result;
}

inline ::krpc::Stream<std::vector<SpaceCenter::Part>> SpaceCenter::Parts::in_decouple_stage_stream(int32_t stage) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(stage));
  return ::krpc::Stream<std::vector<SpaceCenter::Part>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_InDecoupleStage", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Part>> SpaceCenter::Parts::in_stage_stream(int32_t stage) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(stage));
  return ::krpc::Stream<std::vector<SpaceCenter::Part>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_InStage", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Module>> SpaceCenter::Parts::modules_with_name_stream(std::string module_name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(module_name));
  return ::krpc::Stream<std::vector<SpaceCenter::Module>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_ModulesWithName", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Part>> SpaceCenter::Parts::with_module_stream(std::string module_name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(module_name));
  return ::krpc::Stream<std::vector<SpaceCenter::Part>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_WithModule", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Part>> SpaceCenter::Parts::with_name_stream(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  return ::krpc::Stream<std::vector<SpaceCenter::Part>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_WithName", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Part>> SpaceCenter::Parts::with_tag_stream(std::string tag) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(tag));
  return ::krpc::Stream<std::vector<SpaceCenter::Part>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_WithTag", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Part>> SpaceCenter::Parts::with_title_stream(std::string title) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(title));
  return ::krpc::Stream<std::vector<SpaceCenter::Part>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_WithTitle", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Part>> SpaceCenter::Parts::all_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Part>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_All", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Antenna>> SpaceCenter::Parts::antennas_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Antenna>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Antennas", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::CargoBay>> SpaceCenter::Parts::cargo_bays_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::CargoBay>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_CargoBays", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::ControlSurface>> SpaceCenter::Parts::control_surfaces_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::ControlSurface>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_ControlSurfaces", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Parts::controlling_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Controlling", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Decoupler>> SpaceCenter::Parts::decouplers_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Decoupler>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Decouplers", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::DockingPort>> SpaceCenter::Parts::docking_ports_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::DockingPort>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_DockingPorts", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Engine>> SpaceCenter::Parts::engines_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Engine>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Engines", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Experiment>> SpaceCenter::Parts::experiments_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Experiment>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Experiments", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Fairing>> SpaceCenter::Parts::fairings_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Fairing>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Fairings", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Intake>> SpaceCenter::Parts::intakes_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Intake>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Intakes", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::LaunchClamp>> SpaceCenter::Parts::launch_clamps_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::LaunchClamp>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_LaunchClamps", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Leg>> SpaceCenter::Parts::legs_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Leg>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Legs", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Light>> SpaceCenter::Parts::lights_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Light>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Lights", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Parachute>> SpaceCenter::Parts::parachutes_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Parachute>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Parachutes", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Radiator>> SpaceCenter::Parts::radiators_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Radiator>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Radiators", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::RCS>> SpaceCenter::Parts::rcs_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::RCS>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_RCS", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::ReactionWheel>> SpaceCenter::Parts::reaction_wheels_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::ReactionWheel>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_ReactionWheels", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::ResourceConverter>> SpaceCenter::Parts::resource_converters_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::ResourceConverter>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_ResourceConverters", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::ResourceDrain>> SpaceCenter::Parts::resource_drains_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::ResourceDrain>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_ResourceDrains", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::ResourceHarvester>> SpaceCenter::Parts::resource_harvesters_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::ResourceHarvester>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_ResourceHarvesters", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::RoboticHinge>> SpaceCenter::Parts::robotic_hinges_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::RoboticHinge>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_RoboticHinges", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::RoboticPiston>> SpaceCenter::Parts::robotic_pistons_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::RoboticPiston>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_RoboticPistons", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::RoboticRotation>> SpaceCenter::Parts::robotic_rotations_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::RoboticRotation>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_RoboticRotations", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::RoboticRotor>> SpaceCenter::Parts::robotic_rotors_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::RoboticRotor>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_RoboticRotors", _args));
}

inline ::krpc::Stream<SpaceCenter::Part> SpaceCenter::Parts::root_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<SpaceCenter::Part>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Root", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Sensor>> SpaceCenter::Parts::sensors_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Sensor>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Sensors", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::SolarPanel>> SpaceCenter::Parts::solar_panels_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::SolarPanel>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_SolarPanels", _args));
}

inline ::krpc::Stream<std::vector<SpaceCenter::Wheel>> SpaceCenter::Parts::wheels_stream() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return ::krpc::Stream<std::vector<SpaceCenter::Wheel>>(this->_client, this->_client->build_call("SpaceCenter", "Parts_get_Wheels", _args));
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parts::in_decouple_stage_call(int32_t stage) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(stage));
  return this->_client->build_call("SpaceCenter", "Parts_InDecoupleStage", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parts::in_stage_call(int32_t stage) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(stage));
  return this->_client->build_call("SpaceCenter", "Parts_InStage", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parts::modules_with_name_call(std::string module_name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(module_name));
  return this->_client->build_call("SpaceCenter", "Parts_ModulesWithName", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parts::with_module_call(std::string module_name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(module_name));
  return this->_client->build_call("SpaceCenter", "Parts_WithModule", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parts::with_name_call(std::string name) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(name));
  return this->_client->build_call("SpaceCenter", "Parts_WithName", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parts::with_tag_call(std::string tag) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(tag));
  return this->_client->build_call("SpaceCenter", "Parts_WithTag", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parts::with_title_call(std::string title) {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  _args.push_back(encoder::encode(title));
  return this->_client->build_call("SpaceCenter", "Parts_WithTitle", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parts::all_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parts_get_All", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parts::antennas_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parts_get_Antennas", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parts::cargo_bays_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parts_get_CargoBays", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parts::control_surfaces_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parts_get_ControlSurfaces", _args);
}

inline ::krpc::schema::ProcedureCall SpaceCenter::Parts::controlling_call() {
  std::vector<std::string> _args;
  _args.push_back(encoder::encode(*this));
  return this->_client->build_call("SpaceCenter", "Parts_get_Controlling", _args