///////////////////////////////////////////////////////////////////////////////
//                                                         
// SharedMemory.cc
// ---------------
// Shared memory class implementation
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2016 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: SharedMemory 
// 
// Description: Utility class for shared memory management
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

#include "Exception.h"
#include "SharedMemory.h"

#ifdef HAVE_MINGW
#else
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <iostream>
#endif

SharedMemory::SharedMemory(int key)
{
     _key = key;
     _id = 0;
}

SharedMemory::~SharedMemory()
{
}

void SharedMemory::create(unsigned long size, int shmflag)
{

#ifdef HAVE_MINGW
#else

    if (_id == 0)
    {
	if ( (_id = shmget(_key, size,  shmflag | IPC_CREAT)) == -1)
	{
	    throw Exception(EXLOC, "shmget system error");
	}
    }

    if ( (_addr = shmat(_id, (void*) 0,  shmflag)) == (void*)-1)
    {
	throw Exception(EXLOC, "shmat system error");
    }
#endif     
}

void SharedMemory::attach(int shmflag)
{

#ifdef HAVE_MINGW
#else

    if (_id == 0)
    {
	if ( (_id = shmget(_key, 0,  shmflag)) == -1)
	{
	    throw Exception(EXLOC, "shmget system error");
	}
    }
    if ( (_addr = shmat(_id, (void*) 0,  shmflag)) == (void*)-1)
    {
	throw Exception(EXLOC, "shmat system error");
    }
#endif
}

void SharedMemory::destroy()
{

#ifdef HAVE_MINGW
#else

    if (_id == 0)
    {
	if ( (_id = shmget(_key, 0,  0)) == -1)
	{
	    throw Exception(EXLOC, "shmget system error");
	}
    }
    if ( shmctl(_id, IPC_RMID,(struct shmid_ds*)0) == -1)
    {
	throw Exception(EXLOC, "shmctl system error");
    }
#endif    
}

void SharedMemory::detach()
{

#ifdef HAVE_MINGW
#else

    if ( shmdt((char*)_addr) == -1)
     {
	 throw Exception(EXLOC,  "shmdt system error");
     }
#endif
}

int SharedMemory::getId()
{
     return _id;
}

void* SharedMemory::getAddr()
{
     return _addr;
}

void SharedMemory::printInfo()
{

#ifdef HAVE_MINGW
#else


    struct shmid_ds ds;
    if ( shmctl(_id, IPC_STAT,&ds) == -1)
    {
	throw Exception(EXLOC, "shmctl system error");
    }

    cout << "Segment Size: " << ds.shm_segsz << " bytes"  <<endl;
    cout << "Creator Pid: " << ds.shm_cpid << endl;
    cout << "NumAttach: " << ds.shm_nattch << endl;
#endif  
}
