/*--------------------------------------------------------------------------*/
/* Copyright 2022-2024 NXP                                                  */
/*                                                                          */
/* NXP Proprietary. This software is owned or controlled by NXP and may     */
/* only be used strictly in accordance with the applicable license terms.   */
/* By expressly accepting such terms or by downloading, installing,         */
/* activating and/or otherwise using the software, you are agreeing that    */
/* you have read, and that you agree to comply with and are bound by, such  */
/* license terms.  If you do not agree to be bound by the applicable        */
/* license terms, then you may not retain, install, activate or otherwise   */
/* use the software.                                                        */
/*--------------------------------------------------------------------------*/

#ifndef MCUX_CSSL_ANALYSIS_H_
#define MCUX_CSSL_ANALYSIS_H_

#define MCUX_CSSL_ANALYSIS_STR(a) #a
#define MCUX_CSSL_ANALYSIS_EMPTY()
#define MCUX_CSSL_ANALYSIS_DEFER(id) id MCUX_CSSL_ANALYSIS_EMPTY()
#define MCUX_CSSL_ANALYSIS_EXPAND(...) __VA_ARGS__

#define MCUX_CSSL_ANALYSIS_PRAGMA(x) _Pragma(#x)

/* Compiler defines TODO: decide proper placement for those */
#if defined ( __CC_ARM )
/* Arm Compiler 4/5 */
#define MCUX_CSSL_COMPILER_ARMCC
#define MCUX_CSSL_COMPILER_ARM_COMPILER

#elif defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) && (__ARMCC_VERSION < 6100100)
/* Arm Compiler 6.6 LTM (armclang) */
#define MCUX_CSSL_COMPILER_ARMCLANG_LTM
#define MCUX_CSSL_COMPILER_ARM_COMPILER

#elif defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6100100)
/* Arm Compiler above 6.10.1 (armclang) */
#define MCUX_CSSL_COMPILER_ARMCLANG
#define MCUX_CSSL_COMPILER_ARM_COMPILER

#elif defined (_clang_)
#define MCUX_CSSL_COMPILER_ARM_COMPILER /* i.e. Version 6.01 build 0019  */
#endif // defined ( __CC_ARM )

#if defined(__riscv) && defined(__clang__)
#define MCUX_CSSL_COMPILER_RISCV_LLVM
#endif

/* Example of common patterns, with either just predefined rationale, or a combination of discards. */
#define MCUX_CSSL_ANALYSIS_START_PATTERN_HW_READ() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_DISCARD_VOLATILE("Read from a HW peripheral")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_HW_READ() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DISCARD_VOLATILE()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_HW_WRITE() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_DISCARD_VOLATILE("Write to a HW peripheral")
  /*MCUX_CSSL_ANALYSIS_START_SUPPRESS_CAST_TO_OBJECT("Write to a HW peripheral")*/
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_HW_WRITE() \
  /*MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CAST_TO_OBJECT()*/ \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DISCARD_VOLATILE()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_FALSE_POSITIVE_CAST_TYPES_WITH_SAME_ALIGNMENT() \
  MCUX_CSSL_ANALYSIS_COVERITY_START_FALSE_POSITIVE(CERT_EXP36_C, "Both types have the same alignment (32-bit)")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_FALSE_POSITIVE_CAST_TYPES_WITH_SAME_ALIGNMENT() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(CERT_EXP36_C)

#define MCUX_CSSL_ANALYSIS_START_PATTERN_0U_1U_ARE_UNSIGNED() \
  MCUX_CSSL_ANALYSIS_COVERITY_START_FALSE_POSITIVE(MISRA_C_2012_Rule_10_3, "0u and 1u are unsigned values and not boolean") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_FALSE_POSITIVE(MISRA_C_2012_Rule_10_5, "0u and 1u are unsigned values and not boolean")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_0U_1U_ARE_UNSIGNED() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(MISRA_C_2012_Rule_10_5) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(MISRA_C_2012_Rule_10_3)

#define MCUX_CSSL_ANALYSIS_START_PATTERN_EXTERNAL_API_DECLARATIONS() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_DECLARED_BUT_NEVER_REFERENCED("Consumed by user, it is declared but never referenced. ")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_EXTERNAL_API_DECLARATIONS() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DECLARED_BUT_NEVER_REFERENCED()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_DESCRIPTIVE_IDENTIFIER() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_AMBIGUOUS_IDENTIFIER("Identifiers longer than 31 characters are allowed for more descriptive naming")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_DESCRIPTIVE_IDENTIFIER() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_AMBIGUOUS_IDENTIFIER()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_REINTERPRET_MEMORY_OF_OPAQUE_TYPES() \
  MCUX_CSSL_ANALYSIS_START_PATTERN_CAST_VOID() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_REINTERPRET_MEMORY("explicit pointer casts reinterpreting opaque types of workarea-like buffer objects are allowed.")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_REINTERPRET_MEMORY_OF_OPAQUE_TYPES() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_REINTERPRET_MEMORY() \
  MCUX_CSSL_ANALYSIS_STOP_PATTERN_CAST_VOID()

/* NOTE: this STOP macro doesn't match the START macro in name */
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_REINTERPRET_MEMORY() \
  MCUX_CSSL_ANALYSIS_STOP_PATTERN_CAST_VOID() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_REINTERPRET_MEMORY()

/* Applies to casts between ctx/WA structs
 * e.g. cast from Aead_Context_t to more specific type AeadModes_Context_t
 */
#define MCUX_CSSL_ANALYSIS_START_PATTERN_CAST_TO_MORE_SPECIFIC_TYPE() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_REINTERPRET_MEMORY("Cast to a more specific type is allowed")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_CAST_TO_MORE_SPECIFIC_TYPE() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_REINTERPRET_MEMORY()

/* Applies to casts between ctx structs where the new alignment is seemingly more strict
 * e.g. cast from mcuxClRandom_Context_t to more specific type mcuxClOsccaRandomModes_Context_RNG_t.
 * This macro shall only be used for context structures where the memory in question is aligned on the stricter boundary.
 */
#define MCUX_CSSL_ANALYSIS_START_PATTERN_CAST_TO_MORE_SPECIFIC_ALIGNED_TYPE() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_REINTERPRET_MEMORY("Cast to a more specific type is allowed") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_EXP36_C, "Callers have the guidance to provide a 32-bit aligned context")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_CAST_TO_MORE_SPECIFIC_ALIGNED_TYPE() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_EXP36_C) \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_REINTERPRET_MEMORY()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_EXTERNAL_HEADER() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_C11_EXTENSION("External header outside our control") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_UNDEFINED_VALUE("External header outside our control") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_RESERVED_MACRO_IDENTIFIER("External header outside our control") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_PADDED_TO_ALIGNMENT_BOUNDARY("External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(CERT_INT30_C, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(CERT_INT31_C, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(CERT_INT08_C, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(CERT_INT36_C, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(CERT_INT02_C, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(CERT_STR31_C, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(CERT_ARR30_C, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(OVERRUN, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(INTEGER_OVERFLOW, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_11_4, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_5_4, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_21_2, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_21_1, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_10_4, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_10_3, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_4_10, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_15_6, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_18_1, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_14_4, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_14_3, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_3_1, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_11_6, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_1_1, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_7_2, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_5_5, "External header outside our control") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Directive_4_10, "External header outside our control")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_EXTERNAL_HEADER() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Directive_4_10) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_5_5) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_7_2) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_1_1) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_11_6) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_3_1) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_14_3) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_14_4) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_18_1) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_15_6) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_4_10) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_10_3) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_10_4) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_21_1) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_21_2) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_5_4) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(MISRA_C_2012_Rule_11_4) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(INTEGER_OVERFLOW) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(OVERRUN) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(CERT_ARR30_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(CERT_STR31_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(CERT_INT02_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(CERT_INT36_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(CERT_INT08_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(CERT_INT31_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(CERT_INT30_C) \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_PADDED_TO_ALIGNMENT_BOUNDARY() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_RESERVED_MACRO_IDENTIFIER() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_UNDEFINED_VALUE() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_C11_EXTENSION()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_SFR_ACCESS() \
    MCUX_CSSL_ANALYSIS_START_SUPPRESS_INTEGER_OVERFLOW("Address and offset calculations in a hardware's address range do not wrap/overflow.") \
    MCUX_CSSL_ANALYSIS_START_SUPPRESS_REINTERPRET_MEMORY("SFRs are 32-bit aligned.") \
    MCUX_CSSL_ANALYSIS_START_SUPPRESS_POINTER_CASTING("Necessary for HW access") \
    MCUX_CSSL_ANALYSIS_START_SUPPRESS_TYPECAST_BETWEEN_INTEGER_AND_POINTER("Necessary for HW access")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_SFR_ACCESS() \
    MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_TYPECAST_BETWEEN_INTEGER_AND_POINTER() \
    MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_POINTER_CASTING() \
    MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_REINTERPRET_MEMORY() \
    MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_INTEGER_OVERFLOW()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_HW_ACCESS() \
  MCUX_CSSL_ANALYSIS_START_PATTERN_SFR_ACCESS()
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_HW_ACCESS() \
  MCUX_CSSL_ANALYSIS_STOP_PATTERN_SFR_ACCESS()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_SYMBOL_DEFINED_MORE_THAN_ONCE() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_DEFINED_MORE_THAN_ONCE("This symbol is defined only once.")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_SYMBOL_DEFINED_MORE_THAN_ONCE() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DEFINED_MORE_THAN_ONCE()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_EXTERNAL_MACRO() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT("External macro outside our control, operation is safe on target platform given correct arguments according to an API are provided") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_OPERATIONS_ON_INAPPROPRIATE_TYPE("External macro outside our control, operation is safe on target platform given correct arguments according to an API are provided") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_CONVERSIONS_WITH_INAPPROPRIATE_TYPE("External macro outside our control, operation is safe on target platform given correct arguments according to an API are provided") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_MISSING_EXPLICIT_PARANTHESIS("External macro outside our control, operation is safe on target platform given correct arguments according to an API are provided") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_INVALID_WIDTH_IN_SHIFT_OPERATIONS("External macro outside our control, operation is safe on target platform given correct arguments according to an API are provided")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_EXTERNAL_MACRO() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_INVALID_WIDTH_IN_SHIFT_OPERATIONS() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_MISSING_EXPLICIT_PARANTHESIS() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CONVERSIONS_WITH_INAPPROPRIATE_TYPE() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_OPERATIONS_ON_INAPPROPRIATE_TYPE() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_OBJ_SIZES() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_UNUSED_VARIABLE("Variables used to determine object sizes") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_MISSING_VARIABLE_DECLARATION("Variables used to determine object sizes")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_OBJ_SIZES() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_MISSING_VARIABLE_DECLARATION() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_UNUSED_VARIABLE()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_FUP() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_MISSING_VARIABLE_DECLARATION("External declarations are generated by the FUP processing tool") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_FALSE_POSITIVE(MISRA_C_2012_Rule_5_8, "The FUP processing tool generates a second declaration")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_FUP() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(MISRA_C_2012_Rule_5_8) \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_MISSING_VARIABLE_DECLARATION()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_FUP_PROGRAM() \
  MCUX_CSSL_ANALYSIS_START_PATTERN_EXTERNAL_LINKAGE_FUP() \
  MCUX_CSSL_ANALYSIS_START_PATTERN_DEFINITION_IN_TEMP_FILE_FUP()
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_FUP_PROGRAM() \
  MCUX_CSSL_ANALYSIS_STOP_PATTERN_DEFINITION_IN_TEMP_FILE_FUP() \
  MCUX_CSSL_ANALYSIS_STOP_PATTERN_EXTERNAL_LINKAGE_FUP()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_EXTERNAL_LINKAGE_FUP() \
  MCUX_CSSL_ANALYSIS_START_PATTERN_DESCRIPTIVE_IDENTIFIER() \
  MCUX_CSSL_ANALYSIS_COVERITY_START_FALSE_POSITIVE(MISRA_C_2012_Rule_5_8, "The FUP processing tool generates a second external declaration")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_EXTERNAL_LINKAGE_FUP() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(MISRA_C_2012_Rule_5_8) \
  MCUX_CSSL_ANALYSIS_STOP_PATTERN_DESCRIPTIVE_IDENTIFIER()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_DEFINITION_IN_TEMP_FILE_FUP() \
  MCUX_CSSL_ANALYSIS_COVERITY_START_FALSE_POSITIVE(MISRA_C_2012_Rule_8_6, "The FUP processing tool generates definitions in temporary build files")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_DEFINITION_IN_TEMP_FILE_FUP() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(MISRA_C_2012_Rule_8_6)

#define MCUX_CSSL_ANALYSIS_START_PATTERN_SC_INTEGER_OVERFLOW() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_INTEGER_OVERFLOW("Integer overflows are allowed/expected for security counter variables per design")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_SC_INTEGER_OVERFLOW() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_INTEGER_OVERFLOW()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_SWITCH_STATEMENT_RETURN_TERMINATION() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_SWITCH_STATEMENT_NOT_WELL_FORMED("Return instead of break statement as terminator is allowed")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_SWITCH_STATEMENT_RETURN_TERMINATION() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_SWITCH_STATEMENT_NOT_WELL_FORMED()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_INVARIANT_EXPRESSION_WORKAREA_CALCULATIONS() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT("Invariant expression is allowed in workarea calculation macros")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_INVARIANT_EXPRESSION_WORKAREA_CALCULATIONS() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_ADDRESS_IN_SFR_IS_NOT_REUSED_OUTSIDE() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_ESCAPING_LOCAL_ADDRESS("Address in SFR is for internal use only and does not escape")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_ADDRESS_IN_SFR_IS_NOT_REUSED_OUTSIDE() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_ESCAPING_LOCAL_ADDRESS()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_DI_INTEGER_OVERFLOW() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_INTEGER_OVERFLOW("Integer overflows are allowed/expected for DI variables per design")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_DI_INTEGER_OVERFLOW() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_INTEGER_OVERFLOW()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_DI_CAST_POINTERS() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_TYPECAST_BETWEEN_INTEGER_AND_POINTER("Typecast pointer to integer for DI record/expunge")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_DI_CAST_POINTERS() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_TYPECAST_BETWEEN_INTEGER_AND_POINTER()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_SC_CAST_POINTERS() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_TYPECAST_BETWEEN_INTEGER_AND_POINTER("Typecast pointer (void *) to integer for SC add/sub")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_SC_CAST_POINTERS() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_TYPECAST_BETWEEN_INTEGER_AND_POINTER()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_EXAMPLE_FUNCTION() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_MISSING_FUNCTION_DECLARATION("Declaration is not needed for release package. Cant be static as it is declared and used by testing framweork.") \
  MCUX_CSSL_ANALYSIS_START_PATTERN_DESCRIPTIVE_IDENTIFIER()
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_EXAMPLE_FUNCTION() \
  MCUX_CSSL_ANALYSIS_STOP_PATTERN_DESCRIPTIVE_IDENTIFIER() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_MISSING_FUNCTION_DECLARATION()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_RETURN_IN_SWITCH_STATEMENT() \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_16_1, "Return instead of break in switch statements is allowed") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_16_3, "Return instead of break in switch statements is allowed") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_16_6, "Return instead of break is allowed as conforming switch clause")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_RETURN_IN_SWITCH_STATEMENT() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_16_6) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_16_3) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_16_1)

#define MCUX_CSSL_ANALYSIS_START_PATTERN_CAST_VOID() \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_5, "Typecast pointer (void *) for correct representation to use pointer")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_CAST_VOID() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_5)

#define MCUX_CSSL_ANALYSIS_START_PATTERN_HW_REGISTER_INDEXING() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_OUT_OF_BOUNDS_ACCESS("Apply an index to the base address of a HW peripheral to access the correct SFR-word. The caller is responsible for ensuring that the index is valid.") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(ARRAY_VS_SINGLETON, "Apply an index to the base address of a HW peripheral to access the correct SFR-word. The caller is responsible for ensuring that the index is valid.")
#define MCUX_CSSL_ANALYSIS_START_PATTERN_HW_REGISTER_INDEXING_WITH_OFFSETOF() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_OUT_OF_BOUNDS_ACCESS("Apply an index to the base address of a HW peripheral, where index was computed with 'offsetof' macro to ensure a correct offset.") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(ARRAY_VS_SINGLETON, "Apply an index to the base address of a HW peripheral, where index was computed with 'offsetof' macro to ensure a correct offset.")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_HW_REGISTER_INDEXING() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(ARRAY_VS_SINGLETON) \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_OUT_OF_BOUNDS_ACCESS()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_ERROR_HANDLING_STATE() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_RETURN_CODE_NOT_CHECKED("Already in error handling state")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_ERROR_HANDLING_STATE() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_RETURN_CODE_NOT_CHECKED()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_HYPERLINK_IN_COMMENTS() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_TEXT_IN_COMMENTS("Hyperlinks in comments are allowed for documentation purposes")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_HYPERLINK_IN_COMMENTS() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_TEXT_IN_COMMENTS()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_NULL_POINTER_CONSTANT() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_NULL_POINTER_CONSTANT("NULL is used in code")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_NULL_POINTER_CONSTANT() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_NULL_POINTER_CONSTANT()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_STRING_NOT_MODIFIED() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_MODIFY_STRING_LITERALS("The constant string literal is not being modified")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_STRING_NOT_MODIFIED() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_MODIFY_STRING_LITERALS()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_BOOLEAN_TYPE_FOR_CONDITIONAL_EXPRESSION() \
  MCUX_CSSL_ANALYSIS_COVERITY_START_FALSE_POSITIVE(MISRA_C_2012_Rule_14_4, "Conditional expression does have a boolean type.")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_BOOLEAN_TYPE_FOR_CONDITIONAL_EXPRESSION() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(MISRA_C_2012_Rule_14_4)

#define MCUX_CSSL_ANALYSIS_START_PATTERN_FUNCTIONS_CONSUMED_BY_CUSTOMER() \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_DCL15_C, "can not make function static as it is consumed by customers.") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_DCL19_C, "can not make function static as it is consumed by customers.")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_FUNCTIONS_CONSUMED_BY_CUSTOMER() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_DCL19_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_DCL15_C)

#define MCUX_CSSL_ANALYSIS_START_PATTERN_TYPE_NAME_DOESNT_NEED_TO_BE_WRAPPED_BY_PARANTHESIS() \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_20_7, " Not wrapping a macro parameter in parentheses is allowed, if the parameter represents a type name")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_TYPE_NAME_DOESNT_NEED_TO_BE_WRAPPED_BY_PARANTHESIS() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_20_7)

#define MCUX_CSSL_ANALYSIS_START_PATTERN_IMPLICIT_CAST_FROM_CHAR() \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_10_3, "Implicit cast between char and uint8_t in examples is allowed for readability purposes")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_IMPLICIT_CAST_FROM_CHAR() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_10_3)


#if defined(MCUX_CSSL_COMPILER_ARM_COMPILER)
#define MCUX_CSSL_ANALYSIS_START_PATTERN_SIGNED_TRUNCATION() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_CAST_MAY_RESULT_IN_MISINTERPRETED_DATA("Truncated upper bits are not needed. Operation is implementation defined and documented in arm compiler user guide. If a value of integral type is truncated to a shorter signed integral type, the result is obtained by discarding an appropriate number of most significant bits. If the original number is too large, positive or negative, for the new type, there is no guarantee that the sign of the result is going to be the same as the original. On target ARM architecture two's complement representation is used and a sign will be derived from most significant bit of data remaining after truncation. This operation allows efficient implementation of signed numbers modular arithmetic.")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_SIGNED_TRUNCATION() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CAST_MAY_RESULT_IN_MISINTERPRETED_DATA()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_SIGNED_SHIFT() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_OPERATIONS_ON_INAPPROPRIATE_TYPE("Shift operation on signed numbers is implementation defined and documented in arm compiler user guide. Right shifts on signed quantities are arithmetic (sign extension is performed). Left shifs are logical. This operation allows sign extensions and efficient implementation of signed numbers arithmetic.") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_SIGNED_SHIFT_AMOUNT("Shift operation on signed numbers is implementation defined and documented in arm compiler user guide. Right shifts on signed quantities are arithmetic (sign extension is performed). Left shifs are logical. This operation allows sign extensions and efficient implementation of signed numbers arithmetic.")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_SIGNED_SHIFT() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_SIGNED_SHIFT_AMOUNT() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_OPERATIONS_ON_INAPPROPRIATE_TYPE()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_TWOS_COMPLEMENT_REPRESENTATION() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_CAST_MAY_RESULT_IN_MISINTERPRETED_DATA("Algoritihm works correctly assuming two's complement representation of signed numbers. This is true for target ARM platform.")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_TWOS_COMPLEMENT_REPRESENTATION() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CAST_MAY_RESULT_IN_MISINTERPRETED_DATA()

#elif defined(MCUX_CSSL_COMPILER_RISCV_LLVM)
#define MCUX_CSSL_ANALYSIS_START_PATTERN_SIGNED_TRUNCATION() \
  /* Intentionally empty */
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_SIGNED_TRUNCATION() \
  /* Intentionally empty */

#define MCUX_CSSL_ANALYSIS_START_PATTERN_SIGNED_SHIFT() \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_OPERATIONS_ON_INAPPROPRIATE_TYPE("Shift operation on signed numbers is implementation defined and documented in RISC-V compiler user guide. Right shifts on signed quantities are arithmetic (sign extension is performed).") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_SIGNED_SHIFT_AMOUNT("Shift operation on signed numbers is implementation defined and documented in RISC-V compiler user guide. Right shifts on signed quantities are arithmetic (sign extension is performed).")
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_SIGNED_SHIFT() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_SIGNED_SHIFT_AMOUNT() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_OPERATIONS_ON_INAPPROPRIATE_TYPE()

#define MCUX_CSSL_ANALYSIS_START_PATTERN_TWOS_COMPLEMENT_REPRESENTATION() \
  /* Intentionally empty */
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_TWOS_COMPLEMENT_REPRESENTATION() \
  /* Intentionally empty */

#else //defined(MCUX_CSSL_COMPILER_ARM_COMPILER) || defined(MCUX_CSSL_COMPILER_RISCV_LLVM)

#define MCUX_CSSL_ANALYSIS_START_PATTERN_SIGNED_TRUNCATION() \
  /* Intentionally empty */
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_SIGNED_TRUNCATION() \
  /* Intentionally empty */
#define MCUX_CSSL_ANALYSIS_START_PATTERN_SIGNED_SHIFT() \
  /* Intentionally empty */
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_SIGNED_SHIFT() \
  /* Intentionally empty */
#define MCUX_CSSL_ANALYSIS_START_PATTERN_TWOS_COMPLEMENT_REPRESENTATION() \
  /* Intentionally empty */
#define MCUX_CSSL_ANALYSIS_STOP_PATTERN_TWOS_COMPLEMENT_REPRESENTATION() \
  /* Intentionally empty */
#endif

#define MCUX_CSSL_ANALYSIS_ASSERT_PARAMETER(value, min_value, max_value, return_code) \
  MCUX_CSSL_ANALYSIS_COVERITY_ASSERT(value, min_value, max_value, return_code)

#define MCUX_CSSL_ANALYSIS_ASSERT_PARAMETER_FP_VOID(value, min_value, max_value) \
  MCUX_CSSL_ANALYSIS_COVERITY_ASSERT_FP_VOID(value, min_value, max_value)

#define MCUX_CSSL_ANALYSIS_ASSERT_PARAMETER_VOID(value, min_value, max_value) \
  MCUX_CSSL_ANALYSIS_COVERITY_ASSERT_VOID(value, min_value, max_value)

#define MCUX_CSSL_ANALYSIS_ASSERT_PARAMETER_WITH_DATA_TYPE(value,data_type, min_value, max_value, return_code) \
  MCUX_CSSL_ANALYSIS_COVERITY_ASSERT_WITH_DATA_TYPE(value, data_type, min_value, max_value, return_code)

/* Example of basic violation suppression */
#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_UNREACHABLE_CODE(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_2_1, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_UNREACHABLE_CODE() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_2_1)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_UNUSED_MACRO(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_2_5, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_UNUSED_MACRO() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_2_5)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_TEXT_IN_COMMENTS(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_3_1, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_TEXT_IN_COMMENTS() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_3_1)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_POINTER_CASTING(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_3, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_EXP36_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_EXP39_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_POINTER_CASTING() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_EXP39_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_EXP36_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_3)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_CAST_SIGNED_TO_UNSIGNED_MAY_RESULT_TO_MISINTERPRETED_DATA(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT02_C, rationale)\
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT31_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CAST_SIGNED_TO_UNSIGNED_MAY_RESULT_TO_MISINTERPRETED_DATA() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT31_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT02_C)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_CAST_MAY_RESULT_IN_MISINTERPRETED_DATA(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT02_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT31_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT00_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CAST_MAY_RESULT_IN_MISINTERPRETED_DATA() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT00_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT31_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT02_C) \

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_DISCARD_VOLATILE(rationale) \
  MCUX_CSSL_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wcast-qual, rationale) \
  MCUX_CSSL_ANALYSIS_GHS_START_SUPPRESS_WARNING(1836, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_8, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DISCARD_VOLATILE() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_8) \
  MCUX_CSSL_ANALYSIS_GHS_STOP_SUPPRESS_WARNING(1836) \
  MCUX_CSSL_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wcast-qual)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_CONVERSION_BETWEEN_ENUM_AND_INTEGER_TYPES(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_10_5, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CONVERSION_BETWEEN_ENUM_AND_INTEGER_TYPES() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_10_5)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_MISSING_EXPLICIT_PARANTHESIS(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_12_1, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_MISSING_EXPLICIT_PARANTHESIS() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_12_1)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_OUT_OF_BOUNDS_ACCESS(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_18_1, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_OUT_OF_BOUNDS_ACCESS() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_18_1)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_NOT_ELEMENTS_OF_THE_SAME_ARRAY(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_18_2, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_NOT_ELEMENTS_OF_THE_SAME_ARRAY() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_18_2)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_DISCARD_CONST(rationale) \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_DISCARD_CONST_QUALIFIER(rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DISCARD_CONST() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DISCARD_CONST_QUALIFIER()

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_DISCARD_CONST_QUALIFIER(rationale) \
  MCUX_CSSL_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wcast-qual, rationale) \
  MCUX_CSSL_ANALYSIS_GHS_START_SUPPRESS_WARNING(1836, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_8, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_EXP05_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DISCARD_CONST_QUALIFIER() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_EXP05_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_8) \
  MCUX_CSSL_ANALYSIS_GHS_STOP_SUPPRESS_WARNING(1836) \
  MCUX_CSSL_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wcast-qual)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_NULL_POINTER_CONSTANT(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_9, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_NULL_POINTER_CONSTANT() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_9)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_REINTERPRET_MEMORY_BETWEEN_INAPT_ESSENTIAL_TYPES(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_3, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_EXP39_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_REINTERPRET_MEMORY_BETWEEN_INAPT_ESSENTIAL_TYPES() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_EXP39_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_3)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_REINTERPRET_MEMORY(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_3, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_EXP39_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_REINTERPRET_MEMORY() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_EXP39_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_3)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_ARRAY_OUT_OF_BOUNDS(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_STR31_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_ARR30_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_ARR31_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(OVERRUN, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_ARRAY_OUT_OF_BOUNDS() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(OVERRUN) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_ARR31_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_ARR30_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_STR31_C)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_TYPECAST_BETWEEN_INTEGER_AND_POINTER(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_4, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_6, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT36_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_MSC15_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_TYPECAST_BETWEEN_INTEGER_AND_POINTER() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_MSC15_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT36_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_6) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_4)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_TYPECAST_INTEGER_TO_POINTER(rationale) \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_TYPECAST_BETWEEN_INTEGER_AND_POINTER(rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_TYPECAST_INTEGER_TO_POINTER() \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_TYPECAST_BETWEEN_INTEGER_AND_POINTER()

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_AMBIGUOUS_IDENTIFIER(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_5_1, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_5_4, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_5_5, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_AMBIGUOUS_IDENTIFIER() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_5_5) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_5_4) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_5_1)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_REINTERPRET_STRUCT(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_1, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_REINTERPRET_STRUCT() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_1)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_DECLARED_BUT_NEVER_DEFINED(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_8_6, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DECLARED_BUT_NEVER_DEFINED() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_8_6)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_DEFINED_MORE_THAN_ONCE(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_8_5, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DEFINED_MORE_THAN_ONCE() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_8_5)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_DEAD_CODE(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_2_2, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DEAD_CODE() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_2_2)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_DECLARED_BUT_NEVER_REFERENCED(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_2_2, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DECLARED_BUT_NEVER_REFERENCED() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_2_2)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_FALSE_POSITIVE_INTEGER_CONVERSION_MISINTERPRETS_DATA(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_FALSE_POSITIVE(CERT_INT31_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_FALSE_POSITIVE_INTEGER_CONVERSION_MISINTERPRETS_DATA() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(CERT_INT31_C)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_INTEGER_OVERFLOW(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(INTEGER_OVERFLOW, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT30_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT32_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT08_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_12_4, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_INTEGER_OVERFLOW() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_12_4) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT08_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT32_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT30_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(INTEGER_OVERFLOW)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_INTEGER_WRAP(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT30_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT08_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_INTEGER_WRAP() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT08_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT30_C)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_INTEGER_WRAP_AND_CONVERSION(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT30_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT31_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT08_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_INTEGER_WRAP_AND_CONVERSION() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT08_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT31_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT30_C)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_POINTER_INCOMPATIBLE(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_EXP39_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_POINTER_INCOMPATIBLE() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_EXP39_C)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_USE_UNINITIALIZED_VALUE(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_EXP33_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(UNINIT, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_USE_UNINITIALIZED_VALUE() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(UNINIT) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_EXP33_C)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_DEREFERENCE_NULL_POINTER(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_EXP34_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(FORWARD_NULL, rationale) \
  MCUX_CSSL_ANALYSIS_START_PATTERN_NULL_POINTER_CONSTANT()
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_DEREFERENCE_NULL_POINTER() \
  MCUX_CSSL_ANALYSIS_STOP_PATTERN_NULL_POINTER_CONSTANT() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(FORWARD_NULL) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_EXP34_C)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_SWITCH_STATEMENT_NOT_WELL_FORMED(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_16_1, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_16_3, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_16_6, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_SWITCH_STATEMENT_NOT_WELL_FORMED() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_16_1) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_16_3) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_16_6)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_14_3, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_14_3)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_ESCAPING_LOCAL_ADDRESS(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_18_6, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_DCL30_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_ESCAPING_LOCAL_ADDRESS() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_DCL30_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_18_6)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_C11_EXTENSION(rationale) \
  MCUX_CSSL_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wc11-extensions, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_C11_EXTENSION() \
  MCUX_CSSL_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wc11-extensions)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_UNDEFINED_VALUE(rationale) \
  MCUX_CSSL_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wundef, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_UNDEFINED_VALUE() \
  MCUX_CSSL_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wundef)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_RESERVED_IDENTIFIER(rationale) \
  MCUX_CSSL_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wreserved-identifier, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_RESERVED_IDENTIFIER() \
  MCUX_CSSL_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wreserved-identifier)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_RESERVED_MACRO_IDENTIFIER(rationale) \
  MCUX_CSSL_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wreserved-macro-identifier, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_RESERVED_MACRO_IDENTIFIER() \
  MCUX_CSSL_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wreserved-macro-identifier)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_MISSING_FUNCTION_DECLARATION(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_8_4, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_MISSING_FUNCTION_DECLARATION() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_8_4)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_MISSING_VARIABLE_DECLARATION(rationale) \
  MCUX_CSSL_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wmissing-variable-declarations, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_8_4, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_MISSING_VARIABLE_DECLARATION() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_8_4) \
  MCUX_CSSL_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wmissing-variable-declarations)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_CAST_OF_COMPOSITE_EXPRESSION(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_10_8, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CAST_OF_COMPOSITE_EXPRESSION() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_10_8)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_UNUSED_VARIABLE(rationale) \
  MCUX_CSSL_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wunused-variable, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_UNUSED_VARIABLE() \
  MCUX_CSSL_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wunused-variable)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_PADDED_TO_ALIGNMENT_BOUNDARY(rationale) \
  MCUX_CSSL_ANALYSIS_CLANG_START_SUPPRESS_WARNING(-Wpadded, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_PADDED_TO_ALIGNMENT_BOUNDARY() \
  MCUX_CSSL_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(-Wpadded)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_HARDWARE_ACCESS(rationale) \
  MCUX_CSSL_ANALYSIS_GCC_START_SUPPRESS_WARNING(-Warray-bounds, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_ARR37_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_HARDWARE_ACCESS() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_ARR37_C) \
  MCUX_CSSL_ANALYSIS_GCC_STOP_SUPPRESS_WARNING(-Warray-bounds)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_TAINTED_EXPRESSION(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Directive_4_14, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(TAINTED_SCALAR, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_TAINTED_EXPRESSION() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(TAINTED_SCALAR) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Directive_4_14)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_INVALID_WIDTH_IN_SHIFT_OPERATIONS(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Directive_12_2, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_INVALID_WIDTH_IN_SHIFT_OPERATIONS() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Directive_12_2)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_CONVERSIONS_WITH_INAPPROPRIATE_TYPE(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_10_4, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CONVERSIONS_WITH_INAPPROPRIATE_TYPE() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_10_4)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_OPERATIONS_ON_INAPPROPRIATE_TYPE(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_10_1, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_OPERATIONS_ON_INAPPROPRIATE_TYPE() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_10_1)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_FLEXIBLE_ARRAY(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_18_7, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_FLEXIBLE_ARRAY() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_18_7)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_ASSIGNING_COMPOSITE_EXPRESSION(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_10_6, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_ASSIGNING_COMPOSITE_EXPRESSION() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_10_6)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_STDARG_USAGE(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_17_1, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_STDARG_USAGE() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_17_1)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_VA_ARGS_USAGE(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_10_1, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_10_4, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_20_7, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_VA_ARGS_USAGE() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_20_7) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_10_4) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_10_1)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_TYPECAST_VOIDPTR_TO_FUNCTION(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_5, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_TYPECAST_VOIDPTR_TO_FUNCTION() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_5)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_CONCATENATION_PREPROCESSOR(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_20_10, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CONCATENATION_PREPROCESSOR() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_20_10)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_SIGNED_SHIFT_AMOUNT(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT13_C, rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT14_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_SIGNED_SHIFT_AMOUNT() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT14_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT13_C)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_MODIFY_STRING_LITERALS(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_FALSE_POSITIVE(CERT_STR30_C, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_MODIFY_STRING_LITERALS() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(CERT_STR30_C)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_RETURN_CODE_NOT_CHECKED(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Directive_4_7, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_RETURN_CODE_NOT_CHECKED() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Directive_4_7)

#define MCUX_CSSL_ANALYSIS_START_SUPPRESS_CAST_TO_VOIDPTR(rationale) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_6, rationale)
#define MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CAST_TO_VOIDPTR() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_6)


/* Tool specific handling: Coverity checkers */
#if defined(__SECURITY_COVERITY__) || defined(__QUALITY_COVERITY__)

#define MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(checker_identifier, rationale) \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(coverity compliance block deviate checker_identifier rationale))
#define MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(checker_identifier) \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(coverity compliance end_block checker_identifier))

#define MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(checker_identifier, rationale) \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(coverity compliance block(include) deviate checker_identifier rationale))
#define MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(checker_identifier) \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(coverity compliance end_block(include) checker_identifier))

#define MCUX_CSSL_ANALYSIS_COVERITY_START_FALSE_POSITIVE(checker_identifier, rationale) \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(coverity compliance block fp checker_identifier rationale))
#define MCUX_CSSL_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(checker_identifier) \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(coverity compliance end_block checker_identifier))

/**
* @brief  Invalid function ID for MCUX_CSSL_FP_FUNCTION_EXIT_VOID macro
*/
#define MCUX_CSSL_FP_FUNCID_mcuxCssl_Invalid_Function                     (0xFFFFu)

#else
#define MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(checker_identifier, rationale)
#define MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(checker_identifier)

#define MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE_INCLUDE_BLOCK(checker_identifier, rationale)
#define MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE_INCLUDE_BLOCK(checker_identifier)

#define MCUX_CSSL_ANALYSIS_COVERITY_START_FALSE_POSITIVE(checker_identifier, rationale)
#define MCUX_CSSL_ANALYSIS_COVERITY_STOP_FALSE_POSITIVE(checker_identifier)

#endif

#if defined(__SECURITY_COVERITY__)

/**
* @brief  Coverity assert macro for flow protected void functions
*/
#define MCUX_CSSL_ANALYSIS_COVERITY_ASSERT_FP_VOID(value, min_value, max_value) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(NO_EFFECT, "The minimum or the maximum value may have no effect for the condition.") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT("The minimum or the maximum value may have no effect for the condition.") \
  if(((min_value) <= (value)) && ((value) <= (max_value))) \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(NO_EFFECT) \
  {\
    /* Do Nothing*/ \
  }\
  else \
  {\
    MCUX_CSSL_FP_FUNCTION_EXIT_VOID(mcuxCssl_Invalid_Function); \
  }

/**
* @brief  Coverity assert macro for void functions
*/
#define MCUX_CSSL_ANALYSIS_COVERITY_ASSERT_VOID(value, min_value, max_value) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(NO_EFFECT, "The minimum or the maximum value may have no effect for the condition.") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT("The minimum or the maximum value may have no effect for the condition.") \
  if(((min_value) <= (value)) && ((value) <= (max_value))) \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(NO_EFFECT) \
  {\
    /* Do Nothing*/ \
  }\
  else \
  {\
    return; \
  }

#define MCUX_CSSL_ANALYSIS_COVERITY_ASSERT(value, min_value, max_value, return_code) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(NO_EFFECT, "The minimum or the maximum value may have no effect for the condition.") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT("The minimum or the maximum value may have no effect for the condition.") \
  if(((min_value) <= (value)) && ((value) <= (max_value))) \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(NO_EFFECT) \
  {\
    /* Do Nothing*/ \
  }\
  else \
  {\
    MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_17_4, "For void return, don't need to return a value at the end of function")\
    return return_code;\
    MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_17_4)\
  }

#define MCUX_CSSL_ANALYSIS_COVERITY_ASSERT_WITH_DATA_TYPE(value, data_type, min_value, max_value, return_code) \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(NO_EFFECT, "The minimum or the maximum value may have no effect for the condition.") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(CERT_INT36_C, "The minimum or the maximum value may have no effect for the condition.") \
  MCUX_CSSL_ANALYSIS_START_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT("This is for the case where value is unsigned and min_value is unsigned zero.") \
  MCUX_CSSL_ANALYSIS_COVERITY_START_DEVIATE(MISRA_C_2012_Rule_11_4, "The minimum or the maximum value may have no effect for the condition.") \
  if((((data_type)(min_value)) <= ((data_type)(value))) && (((data_type)(value)) <= ((data_type)(max_value)))) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(MISRA_C_2012_Rule_11_4) \
  MCUX_CSSL_ANALYSIS_STOP_SUPPRESS_CONTROLLING_EXPRESSION_IS_INVARIANT() \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(CERT_INT36_C) \
  MCUX_CSSL_ANALYSIS_COVERITY_STOP_DEVIATE(NO_EFFECT) \
  {\
    /* Do Nothing*/ \
  }\
  else \
  {\
    return return_code;\
  }


#else
#define MCUX_CSSL_ANALYSIS_COVERITY_ASSERT_FP_VOID(value, min_value, max_value)
#define MCUX_CSSL_ANALYSIS_COVERITY_ASSERT_VOID(value, min_value, max_value)

#define MCUX_CSSL_ANALYSIS_COVERITY_ASSERT(value, min_value, max_value, return_code)
#define MCUX_CSSL_ANALYSIS_COVERITY_ASSERT_WITH_DATA_TYPE(value, data_type, min_value, max_value, return_code)
#endif

/* Tool specific handling: Clang warnings */
#if defined(__clang__)
#define MCUX_CSSL_ANALYSIS_CLANG_START_SUPPRESS_WARNING(warning_identifier, rationale) \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(clang diagnostic push)) \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(clang diagnostic ignored MCUX_CSSL_ANALYSIS_STR(warning_identifier)))
#define MCUX_CSSL_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(warning_identifier) \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(clang diagnostic pop))
#else
#define MCUX_CSSL_ANALYSIS_CLANG_START_SUPPRESS_WARNING(warning_identifier, rationale)
#define MCUX_CSSL_ANALYSIS_CLANG_STOP_SUPPRESS_WARNING(warning_identifier)
#endif

/* Tool specific handling: GHS warnings */
#if defined(__ghs__)
#define MCUX_CSSL_ANALYSIS_GHS_START_SUPPRESS_WARNING(warning_identifier, rationale) \
  MCUX_CSSL_ANALYSIS_PRAGMA(ghs nowarning warning_identifier)
#define MCUX_CSSL_ANALYSIS_GHS_STOP_SUPPRESS_WARNING(warning_identifier) \
  MCUX_CSSL_ANALYSIS_PRAGMA(ghs endnowarning warning_identifier)
#else
#define MCUX_CSSL_ANALYSIS_GHS_START_SUPPRESS_WARNING(warning_identifier, rationale)
#define MCUX_CSSL_ANALYSIS_GHS_STOP_SUPPRESS_WARNING(warning_identifier)
#endif

/* Tool specific handling: GCC warnings */
#if defined(__GNUC__)
#define MCUX_CSSL_ANALYSIS_GCC_START_SUPPRESS_WARNING(warning_identifier, rationale)  \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(GCC diagnostic push)) \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(GCC diagnostic ignored MCUX_CSSL_ANALYSIS_STR(warning_identifier)))
#define MCUX_CSSL_ANALYSIS_GCC_STOP_SUPPRESS_WARNING(warning_identifier)  \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(GCC diagnostic pop))
#else
#define MCUX_CSSL_ANALYSIS_GCC_START_SUPPRESS_WARNING(warning_identifier, rationale)
#define MCUX_CSSL_ANALYSIS_GCC_STOP_SUPPRESS_WARNING(warning_identifier)
#endif

/* Arm Compiler 4/5 */
#if defined(MCUX_CSSL_COMPILER_ARMCC)
#define MCUX_CSSL_ANALYSIS_ARMCC_START_SUPPRESS_WARNING(warning_identifier, rationale)  \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(push)) \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(diag_suppress MCUX_CSSL_ANALYSIS_STR(warning_identifier)))
#define MCUX_CSSL_ANALYSIS_ARMCC_STOP_SUPPRESS_WARNING(warning_identifier)  \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(pop))
#else
#define MCUX_CSSL_ANALYSIS_ARMCC_START_SUPPRESS_WARNING(warning_identifier, rationale)
#define MCUX_CSSL_ANALYSIS_ARMCC_STOP_SUPPRESS_WARNING(warning_identifier)
#endif

/* Arm Compiler 6 / Arm Compiler for Embedded 6 */
#if defined(MCUX_CSSL_COMPILER_ARMCLANG) || defined(MCUX_CSSL_COMPILER_ARMCLANG_LTM)
#define MCUX_CSSL_ANALYSIS_ARMCLANG_START_SUPPRESS_WARNING(warning_identifier, rationale)  \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(clang diagnostic push)) \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(clang diagnostic ignored MCUX_CSSL_ANALYSIS_STR(warning_identifier)))
#define MCUX_CSSL_ANALYSIS_ARMCLANG_STOP_SUPPRESS_WARNING(warning_identifier)  \
  MCUX_CSSL_ANALYSIS_EXPAND(MCUX_CSSL_ANALYSIS_DEFER(MCUX_CSSL_ANALYSIS_PRAGMA)(clang diagnostic pop))
#else
#define MCUX_CSSL_ANALYSIS_ARMCLANG_START_SUPPRESS_WARNING(warning_identifier, rationale)
#define MCUX_CSSL_ANALYSIS_ARMCLANG_STOP_SUPPRESS_WARNING(warning_identifier)
#endif

#endif /* MCUX_CSSL_ANALYSIS_H_ */
