/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#ifndef SH4LT_TIME_H_
#define SH4LT_TIME_H_

#include "c/time-info.h"

namespace sh4lt {

class Time {
 public:
  using info_t = struct sh4lt_time_info;
  /**
   * Construct the Time object, internal clock are set at the first call to one of the update
   * methods, not at creation.
   **/
  Time() = default;

  /**
   * Update time informations for a new buffer.
   * Internal clocks are set and the buffer number incremented.
   * Buffer-related timings are set with arguments passed.
   * Upadate is not thread-safe.
   *
   * \param buffer_date The playhead date for the buffer begining, expressed in nanoseconds.
   * \param buffer_duration The buffer duration in nanoseconds, or -1 if no duration.
   *
   * \return the time structure to send with the buffer.
   **/
  auto update(int64_t buffer_date, int64_t buffer_duration) -> info_t;

  /**
   * Get the current time from the system clock, in nanoseconds.
   *
   * \return The current time.
   **/
  static auto system_clock_now() -> int64_t;

  /**
   * Get the current time from the steady clock, in nanoseconds.
   *
   * \return The current time.
   **/
  static auto steady_clock_now() -> int64_t;

 private:
  info_t info_{-1, -1, -1, -1, -1};
};

}  // namespace sh4lt
#endif
