/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#undef NDEBUG  // get assert in release mode

#include <cassert>
#include <iostream>

#include "../sh4lt/shtype/shtype-from-gst-caps.hpp"

using namespace sh4lt;
using namespace shtype;

template <typename T>
void check_value(const ShType& type, const std::string& key, const T& value) {
  auto val = type.get_prop(key);
  assert(!val.empty());
  assert(val.as<T>() == value);
}

void check_value(const ShType& type, const std::string& key, const char* value) {
  auto val = type.get_prop(key);
  assert(!val.empty());
  assert(val.as<std::string>() == value);
}

auto main() -> int {
  { // checking type parsing robustness with a GStreamer simple caps:
    auto test_type_str = std::string(
        "video/x-raw, format=(string)I420, width=(int)320, height=(int)240, "
        "framerate=(fraction)30/1, multiview-mode=(string)mono, pixel-aspect-ratio=(fraction)1/1, "
        "interlace-mode=(string)progressive  , label=(string)\"\\(unexpected\\) \\= chars\\,\"");
    auto type = shtype_from_gst_caps(test_type_str, "a label");
    assert(type.media() == "video/x-raw");
    assert(type.get_properties().size() == 8);
    // std::cout << std::any_cast<std::string>(type.get_prop("label")) << '\n';
    check_value<std::string>(type, "label", R"_(\(unexpected\) \= chars\,)_");
    check_value<std::string>(type, "pixel-aspect-ratio", "1/1");
    check_value<int>(type, "width", 320);
  }

  {  // check building a Type and consistency when serializing/deserializing it
    auto type = shtype_from_gst_caps("video/x-raw", "a label");
    type.set_prop("format", "I420");
    type.set_prop("height", 4096);
    type.set_prop("interlace-mode", "progressive");
    type.set_prop("label", "unexpected \\= comma\\,");
    type.set_prop("multiview-mode", "mono");
    type.set_prop("width", 4096);
    type.set_custom_prop("framerate", "fraction", "30/1");
    type.set_custom_prop("pixel-aspect-ratio", "fraction", "1/1");
    // building an other Type from 'type' serialization and check it has the right name and
    auto type2 = shtype_from_gst_caps(shtype_to_gst_caps(type), "a label");
    assert(type2.media() == "video/x-raw");
    check_value(type2, "format", "I420");
    check_value(type2, "framerate", "30/1");
    check_value(type2, "height", 4096);
    check_value(type2, "interlace-mode", "progressive");
    check_value(type2, "label", "unexpected \\= comma\\,");
    check_value(type2, "multiview-mode", "mono");
    check_value(type2, "pixel-aspect-ratio", "1/1");
    check_value(type2, "width", 4096);
    // check serializations are the same
    assert(shtype_to_gst_caps(type) == shtype_to_gst_caps(type2));
  }
  {  // check int parsing
    auto type = shtype_from_gst_caps(
        "audio/x-raw, channels=2, rate=(int)44100, id=-435, id2=(int)-1, label=-label-, "
        "end-label-char=-, pi=(float)3.14, dur-ns=123456789",
        "a label");
    check_value(type, "channels", 2);
    check_value(type, "rate", 44100);
    check_value(type, "id", -435);
    check_value(type, "id2", -1);
    check_value(type, "label", "-label-");
    check_value(type, "end-label-char", "-");
    check_value(type, "pi", 3.14);
    check_value(type, "dur-ns", 123456789);
  }
  
  return 0;
}

