// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-FileCopyrightText: Copyright (c) 2000 - 2009, Lawrence Livermore National Security, LLC
// SPDX-License-Identifier: BSD-3-Clause

#ifndef vtkTableBasedClipCases_h
#define vtkTableBasedClipCases_h

#include "vtkFiltersGeneralModule.h"
#include "vtkSetGet.h"
#include "vtkCellType.h"

#include <cstdint>
#include <type_traits>

VTK_ABI_NAMESPACE_BEGIN
template <bool TInsideOut>
class VTKFILTERSGENERAL_EXPORT vtkTableBasedClipCases
{
public:
  static constexpr uint8_t DISCARDED_CELL_CASE = TInsideOut * 255;

  // Points of original cell (up to 8, for the hex)
  // Note: we assume P0 is zero in several places.
  // Note: we assume these values are contiguous and monotonic.
  static constexpr uint8_t P0 = 0;
  static constexpr uint8_t P1 = 1;
  static constexpr uint8_t P2 = 2;
  static constexpr uint8_t P3 = 3;
  static constexpr uint8_t P4 = 4;
  static constexpr uint8_t P5 = 5;
  static constexpr uint8_t P6 = 6;
  static constexpr uint8_t P7 = 7;

  // Edges of original cell (up to 12, for the hex)
  // Note: we assume these values are contiguous and monotonic.
  static constexpr uint8_t EA = 8;
  static constexpr uint8_t EB = 9;
  static constexpr uint8_t EC = 10;
  static constexpr uint8_t ED = 11;
  static constexpr uint8_t EE = 12;
  static constexpr uint8_t EF = 13;
  static constexpr uint8_t EG = 14;
  static constexpr uint8_t EH = 15;
  static constexpr uint8_t EI = 16;
  static constexpr uint8_t EJ = 17;
  static constexpr uint8_t EK = 18;
  static constexpr uint8_t EL = 19;

  // New interpolated point (ST_PNT outputs)
  static constexpr uint8_t N0 = 20;

  // Shape types
  static constexpr uint8_t ST_VTX = 32 + VTK_VERTEX;
  static constexpr uint8_t ST_LIN = 32 + VTK_LINE;
  static constexpr uint8_t ST_TRI = 32 + VTK_TRIANGLE;
  static constexpr uint8_t ST_QUA = 32 + VTK_QUAD;
  static constexpr uint8_t ST_TET = 32 + VTK_TETRA;
  static constexpr uint8_t ST_HEX = 32 + VTK_HEXAHEDRON;
  static constexpr uint8_t ST_PYR = 32 + VTK_PYRAMID;
  static constexpr uint8_t ST_WDG = 32 + VTK_WEDGE;
  // This is a special shape type for a centroid point
  static constexpr uint8_t ST_PNT = 32 + VTK_EMPTY_CELL;

  using EDGEIDXS = uint8_t[2];

private:
  static constexpr bool F = false;
  static constexpr bool T = true;
  // Supported Cell Types
  static constexpr bool SupportedCellTypes[VTK_NUMBER_OF_CELL_TYPES] = { F, /*VTK_VERTEX*/ T, F,
    /*VTK_LINE*/ T, F, /*VTK_TRIANGLE*/ T, F, F, /*VTK_PIXEL*/ T, /*VTK_QUAD*/ T, /*VTK_TETRA*/ T,
    /*VTK_VOXEL*/ T, /*VTK_HEXAHEDRON*/ T, /*VTK_WEDGE*/ T, /*VTK_PYRAMID*/ T, F, F, F, F, F, F, F,
    F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F,
    F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F, F };

  // Cell Max Case based on the number of points
  static constexpr uint8_t CellMaxCase[9] = { 0, 1, 3, 7, 15, 31, 63, 127, 255 };

  static constexpr uint8_t NUM_CELL_TYPES = 17;
  static constexpr uint8_t N = 255;
  static constexpr uint8_t MAX_NUM_EDGES = 12; // VTK_HEXAHEDRON

  // edges for each cell
  static constexpr uint8_t CellEdges[NUM_CELL_TYPES][MAX_NUM_EDGES][2] = {
    // VTK_EMPTY_CELL
    { { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_VERTEX
    { { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_POLY_VERTEX
    { { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_LINE
    { { 0, 1 }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_POLY_LINE
    { { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_TRIANGLE
    { { 0, 1 }, { 1, 2 }, { 2, 0 }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_TRIANGLE_STRIP
    { { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_POLYGON
    { { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_PIXEL
    { { 0, 1 }, { 1, 3 }, { 2, 3 }, { 0, 2 }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_QUAD
    { { 0, 1 }, { 1, 2 }, { 2, 3 }, { 3, 0 }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_TETRA
    { { 0, 1 }, { 1, 2 }, { 2, 0 }, { 0, 3 }, { 1, 3 }, { 2, 3 }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_VOXEL
    { { 0, 1 }, { 1, 3 }, { 2, 3 }, { 0, 2 }, { 4, 5 }, { 5, 7 }, { 6, 7 }, { 4, 6 }, { 0, 4 }, { 1, 5 }, { 2, 6 }, { 3, 7 } },
    // VTK_HEXAHEDRON
    { { 0, 1 }, { 1, 2 }, { 2, 3 }, { 3, 0 }, { 4, 5 }, { 5, 6 }, { 6, 7 }, { 7, 4 }, { 0, 4 }, { 1, 5 }, { 3, 7 }, { 2, 6 } },
    // VTK_WEDGE
    { { 0, 1 }, { 1, 2 }, { 2, 0 }, { 3, 4 }, { 4, 5 }, { 5, 3 }, { 0, 3 }, { 1, 4 }, { 2, 5 }, { N, N }, { N, N }, { N, N } },
    // VTK_PYRAMID
    { { 0, 1 }, { 1, 2 }, { 2, 3 }, { 3, 0 }, { 0, 4 }, { 1, 4 }, { 2, 4 }, { 3, 4 }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_PENTAGONAL_PRISM
    { { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
    // VTK_HEXAGONAL_PRISM
    { { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N }, { N, N } },
  };

  // clip table for all cases of each cell
  static constexpr uint8_t CellCases[] = {
    // VTK_VERTEX
    /* case 0 */ 0,
    /* case 1 */ 1,
    ST_VTX, 1, P0,
    // VTK_LINE
    /* case 0 */ 0,
    /* case 1 */ 1,
    ST_LIN, 2, P0, EA,
    /* case 2 */ 1,
    ST_LIN, 2, P1, EA,
    /* case 3 */ 1,
    ST_LIN, 2, P0, P1,
    // VTK_TRIANGLE
    /* case 0 */ 0,
    /* case 1 */ 1,
    ST_TRI, 3, P0, EA, EC,
    /* case 2 */ 1,
    ST_TRI, 3, P1, EB, EA,
    /* case 3 */ 1,
    ST_QUA, 4, P0, P1, EB, EC,
    /* case 4 */ 1,
    ST_TRI, 3, P2, EC, EB,
    /* case 5 */ 1,
    ST_QUA, 4, P2, P0, EA, EB,
    /* case 6 */ 1,
    ST_QUA, 4, P1, P2, EC, EA,
    /* case 7 */ 1,
    ST_TRI, 3, P0, P1, P2,
    // VTK_PIXEL
    /* case 0 */ 0,
    /* case 1 */ 1,
    ST_TRI, 3, P0, EA, ED,
    /* case 2 */ 1,
    ST_TRI, 3, P1, EB, EA,
    /* case 3 */ 1,
    ST_QUA, 4, P0, P1, EB, ED,
    /* case 4 */ 1,
    ST_TRI, 3, P2, ED, EC,
    /* case 5 */ 1,
    ST_QUA, 4, P2, P0, EA, EC,
    /* case 6 */ 2,
    ST_QUA, 4, P2, P1, EB, EC,
    ST_QUA, 4, P1, P2, ED, EA,
    /* case 7 */ 2,
    ST_QUA, 4, P2, P1, EB, EC,
    ST_TRI, 3, P0, P1, P2,
    /* case 8 */ 1,
    ST_TRI, 3, P3, EC, EB,
    /* case 9 */ 2,
    ST_QUA, 4, P0, P3, EC, ED,
    ST_QUA, 4, P3, P0, EA, EB,
    /* case 10 */ 1,
    ST_QUA, 4, P1, P3, EC, EA,
    /* case 11 */ 2,
    ST_QUA, 4, P0, P3, EC, ED,
    ST_TRI, 3, P1, P3, P0,
    /* case 12 */ 1,
    ST_QUA, 4, P3, P2, ED, EB,
    /* case 13 */ 2,
    ST_QUA, 4, P3, P0, EA, EB,
    ST_TRI, 3, P2, P0, P3,
    /* case 14 */ 2,
    ST_QUA, 4, P1, P2, ED, EA,
    ST_TRI, 3, P3, P2, P1,
    /* case 15 */ 1,
    ST_QUA, 4, P0, P1, P3, P2,
    // VTK_QUAD
    /* case 0 */ 0,
    /* case 1 */ 1,
    ST_TRI, 3, P0, EA, ED,
    /* case 2 */ 1,
    ST_TRI, 3, P1, EB, EA,
    /* case 3 */ 1,
    ST_QUA, 4, P0, P1, EB, ED,
    /* case 4 */ 1,
    ST_TRI, 3, P2, EC, EB,
    /* case 5 */ 2,
    ST_QUA, 4, P2, P0, EA, EB,
    ST_QUA, 4, P0, P2, EC, ED,
    /* case 6 */ 1,
    ST_QUA, 4, P1, P2, EC, EA,
    /* case 7 */ 2,
    ST_QUA, 4, P0, P2, EC, ED,
    ST_TRI, 3, P1, P2, P0,
    /* case 8 */ 1,
    ST_TRI, 3, P3, ED, EC,
    /* case 9 */ 1,
    ST_QUA, 4, P3, P0, EA, EC,
    /* case 10 */ 2,
    ST_QUA, 4, P3, P1, EB, EC,
    ST_QUA, 4, P1, P3, ED, EA,
    /* case 11 */ 2,
    ST_QUA, 4, P3, P1, EB, EC,
    ST_TRI, 3, P0, P1, P3,
    /* case 12 */ 1,
    ST_QUA, 4, P2, P3, ED, EB,
    /* case 13 */ 2,
    ST_QUA, 4, P2, P0, EA, EB,
    ST_TRI, 3, P3, P0, P2,
    /* case 14 */ 2,
    ST_QUA, 4, P1, P3, ED, EA,
    ST_TRI, 3, P2, P3, P1,
    /* case 15 */ 1,
    ST_QUA, 4, P0, P1, P2, P3,
    // VTK_TETRA
    /* case 0 */ 0,
    /* case 1 */ 1,
    ST_TET, 4, P0, EA, EC, ED,
    /* case 2 */ 1,
    ST_TET, 4, P1, EB, EA, EE,
    /* case 3 */ 1,
    ST_WDG, 6, P0, ED, EC, P1, EE, EB,
    /* case 4 */ 1,
    ST_TET, 4, P2, EC, EB, EF,
    /* case 5 */ 1,
    ST_WDG, 6, P2, EF, EB, P0, ED, EA,
    /* case 6 */ 1,
    ST_WDG, 6, P1, EE, EA, P2, EF, EC,
    /* case 7 */ 1,
    ST_WDG, 6, ED, EE, EF, P0, P1, P2,
    /* case 8 */ 1,
    ST_TET, 4, P3, EE, ED, EF,
    /* case 9 */ 1,
    ST_WDG, 6, P0, EC, EA, P3, EF, EE,
    /* case 10 */ 1,
    ST_WDG, 6, P3, EF, ED, P1, EB, EA,
    /* case 11 */ 1,
    ST_WDG, 6, P0, P1, P3, EC, EB, EF,
    /* case 12 */ 1,
    ST_WDG, 6, P2, EB, EC, P3, EE, ED,
    /* case 13 */ 1,
    ST_WDG, 6, EA, EB, EE, P0, P2, P3,
    /* case 14 */ 1,
    ST_WDG, 6, P1, P2, P3, EA, EC, ED,
    /* case 15 */ 1,
    ST_TET, 4, P0, P1, P2, P3,
    // VTK_VOXEL
    /* case 0 */ 0,
    /* case 1 */ 1,
    ST_TET, 4, P0, EA, ED, EI,
    /* case 2 */ 1,
    ST_TET, 4, P1, EA, EJ, EB,
    /* case 3 */ 1,
    ST_WDG, 6, P1, EB, EJ, P0, ED, EI,
    /* case 4 */ 1,
    ST_TET, 4, P2, ED, EC, EK,
    /* case 5 */ 1,
    ST_WDG, 6, P2, EK, EC, P0, EI, EA,
    /* case 6 */ 7,
    ST_PNT, 2, EK, EJ,
    ST_PYR, 5, P2, ED, EA, P1, N0,
    ST_PYR, 5, EC, P2, P1, EB, N0,
    ST_TET, 4, EB, P1, EJ, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_TET, 4, EC, P2, N0, EK,
    ST_TET, 4, ED, P2, EK, N0,
    /* case 7 */ 7,
    ST_PNT, 5, EJ, EI, EB, EC, EK,
    ST_PYR, 5, EI, P0, P2, EK, N0,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_TET, 4, EB, P1, EJ, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_PYR, 5, EB, EC, P2, P1, N0,
    ST_TET, 4, P2, EC, EK, N0,
    /* case 8 */ 1,
    ST_TET, 4, P3, EB, EL, EC,
    /* case 9 */ 7,
    ST_PNT, 2, EI, EL,
    ST_PYR, 5, P0, P3, EC, ED, N0,
    ST_PYR, 5, EA, EB, P3, P0, N0,
    ST_TET, 4, EB, EL, P3, N0,
    ST_TET, 4, P3, EL, EC, N0,
    ST_TET, 4, EA, N0, P0, EI,
    ST_TET, 4, ED, EI, P0, N0,
    /* case 10 */ 1,
    ST_WDG, 6, P3, EC, EL, P1, EA, EJ,
    /* case 11 */ 7,
    ST_PNT, 5, EL, EJ, EC, ED, EI,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_PYR, 5, EL, P3, P1, EJ, N0,
    ST_TET, 4, EC, P3, EL, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_PYR, 5, EC, ED, P0, P3, N0,
    ST_TET, 4, P0, ED, EI, N0,
    /* case 12 */ 1,
    ST_WDG, 6, P2, ED, EK, P3, EB, EL,
    /* case 13 */ 7,
    ST_PNT, 5, EL, EK, EB, EA, EI,
    ST_PYR, 5, EK, EI, P0, P2, N0,
    ST_PYR, 5, EL, EK, P2, P3, N0,
    ST_TET, 4, EB, EL, P3, N0,
    ST_TET, 4, P3, P2, P0, N0,
    ST_PYR, 5, EB, P3, P0, EA, N0,
    ST_TET, 4, P0, EI, EA, N0,
    /* case 14 */ 7,
    ST_PNT, 5, EK, EL, ED, EA, EJ,
    ST_PYR, 5, EL, P3, P1, EJ, N0,
    ST_PYR, 5, EK, P2, P3, EL, N0,
    ST_TET, 4, ED, P2, EK, N0,
    ST_TET, 4, P2, P1, P3, N0,
    ST_PYR, 5, ED, EA, P1, P2, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    /* case 15 */ 1,
    ST_HEX, 8, P0, P1, P3, P2, EI, EJ, EL, EK,
    /* case 16 */ 1,
    ST_TET, 4, P4, EE, EI, EH,
    /* case 17 */ 1,
    ST_WDG, 6, P0, ED, EA, P4, EH, EE,
    /* case 18 */ 7,
    ST_PNT, 2, EH, EB,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_PYR, 5, EE, EJ, P1, P4, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_TET, 4, EE, N0, P4, EH,
    ST_TET, 4, EI, EH, P4, N0,
    /* case 19 */ 7,
    ST_PNT, 5, EB, ED, EJ, EE, EH,
    ST_PYR, 5, ED, EH, P4, P0, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_PYR, 5, EJ, P1, P4, EE, N0,
    ST_TET, 4, P4, EH, EE, N0,
    /* case 20 */ 7,
    ST_PNT, 2, EC, EE,
    ST_PYR, 5, P2, P4, EI, ED, N0,
    ST_PYR, 5, EK, EH, P4, P2, N0,
    ST_TET, 4, EH, EE, P4, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_TET, 4, EK, N0, P2, EC,
    ST_TET, 4, ED, EC, P2, N0,
    /* case 21 */ 7,
    ST_PNT, 5, EE, EA, EH, EK, EC,
    ST_PYR, 5, EA, EC, P2, P0, N0,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_TET, 4, EH, EE, P4, N0,
    ST_TET, 4, P4, P0, P2, N0,
    ST_PYR, 5, EH, P4, P2, EK, N0,
    ST_TET, 4, P2, EC, EK, N0,
    /* case 22 */ 5,
    ST_WDG, 6, ED, EA, EI, P2, P1, P4,
    ST_PYR, 5, P2, EK, EH, P4, EC,
    ST_PYR, 5, EE, EJ, P1, P4, EB,
    ST_PYR, 5, EC, P2, P1, EB, P4,
    ST_PYR, 5, EC, EB, EE, EH, P4,
    /* case 23 */ 5,
    ST_TET, 4, P0, P1, P2, P4,
    ST_PYR, 5, EH, P4, P2, EK, EC,
    ST_PYR, 5, EE, EJ, P1, P4, EB,
    ST_PYR, 5, P2, P1, EB, EC, P4,
    ST_PYR, 5, EH, EC, EB, EE, P4,
    /* case 24 */ 2,
    ST_TET, 4, P4, EE, EI, EH,
    ST_TET, 4, P3, EC, EB, EL,
    /* case 25 */ 8,
    ST_PNT, 4, EE, EH, EL, EL,
    ST_PYR, 5, P3, EC, ED, P0, N0,
    ST_PYR, 5, EA, EB, P3, P0, N0,
    ST_TET, 4, P3, EL, EC, N0,
    ST_TET, 4, EB, EL, P3, N0,
    ST_PYR, 5, ED, EH, P4, P0, N0,
    ST_PYR, 5, P0, P4, EE, EA, N0,
    ST_TET, 4, EE, P4, EH, N0,
    /* case 26 */ 8,
    ST_PNT, 4, EL, EC, EH, EH,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_PYR, 5, EJ, P1, P4, EE, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_TET, 4, EE, P4, EH, N0,
    ST_PYR, 5, EA, P1, P3, EC, N0,
    ST_PYR, 5, P1, EJ, EL, P3, N0,
    ST_TET, 4, EL, EC, P3, N0,
    /* case 27 */ 9,
    ST_PNT, 6, ED, EC, EL, EJ, EE, EH,
    ST_TET, 4, P0, P1, P3, N0,
    ST_PYR, 5, ED, P0, P3, EC, N0,
    ST_PYR, 5, ED, EH, P4, P0, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_TET, 4, P4, EH, EE, N0,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_PYR, 5, EJ, EL, P3, P1, N0,
    ST_TET, 4, EL, EC, P3, N0,
    /* case 28 */ 8,
    ST_PNT, 4, EL, EB, EE, EE,
    ST_PYR, 5, P4, EI, ED, P2, N0,
    ST_PYR, 5, EK, EH, P4, P2, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_TET, 4, EH, EE, P4, N0,
    ST_PYR, 5, ED, EB, P3, P2, N0,
    ST_PYR, 5, P2, P3, EL, EK, N0,
    ST_TET, 4, EL, P3, EB, N0,
    /* case 29 */ 9,
    ST_PNT, 6, EA, EB, EL, EK, EH, EE,
    ST_TET, 4, P0, P3, P2, N0,
    ST_PYR, 5, EA, EB, P3, P0, N0,
    ST_PYR, 5, EA, P0, P4, EE, N0,
    ST_TET, 4, P4, P0, P2, N0,
    ST_TET, 4, P4, EH, EE, N0,
    ST_PYR, 5, P4, P2, EK, EH, N0,
    ST_PYR, 5, EK, P2, P3, EL, N0,
    ST_TET, 4, EL, P3, EB, N0,
    /* case 30 */ 9,
    ST_PNT, 5, EL, EJ, EK, EH, EE,
    ST_WDG, 6, ED, EA, EI, P2, P1, P4,
    ST_TET, 4, P1, P2, P4, N0,
    ST_PYR, 5, P2, EK, EH, P4, N0,
    ST_TET, 4, P3, P2, P1, N0,
    ST_PYR, 5, EJ, EL, P3, P1, N0,
    ST_PYR, 5, EL, EK, P2, P3, N0,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_TET, 4, EH, EE, P4, N0,
    /* case 31 */ 9,
    ST_PNT, 5, EJ, EL, EK, EE, EH,
    ST_PYR, 5, P0, P1, P3, P2, N0,
    ST_TET, 4, P2, P4, P0, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_PYR, 5, EJ, EL, P3, P1, N0,
    ST_PYR, 5, EL, EK, P2, P3, N0,
    ST_PYR, 5, EK, EH, P4, P2, N0,
    ST_TET, 4, EE, P4, EH, N0,
    /* case 32 */ 1,
    ST_TET, 4, P5, EF, EJ, EE,
    /* case 33 */ 7,
    ST_PNT, 2, ED, EF,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_TET, 4, EA, P0, N0, ED,
    ST_TET, 4, EI, P0, ED, N0,
    /* case 34 */ 1,
    ST_WDG, 6, P1, EA, EB, P5, EE, EF,
    /* case 35 */ 7,
    ST_PNT, 5, EF, EB, EE, EI, ED,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_PYR, 5, EF, EB, P1, P5, N0,
    ST_TET, 4, EE, EF, P5, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_PYR, 5, EE, P5, P0, EI, N0,
    ST_TET, 4, P0, ED, EI, N0,
    /* case 36 */ 2,
    ST_TET, 4, P2, ED, EC, EK,
    ST_TET, 4, P5, EJ, EE, EF,
    /* case 37 */ 8,
    ST_PNT, 4, EC, EK, EF, EF,
    ST_PYR, 5, P5, P0, EI, EE, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    ST_PYR, 5, EI, P0, P2, EK, N0,
    ST_PYR, 5, P0, EA, EC, P2, N0,
    ST_TET, 4, EC, EK, P2, N0,
    /* case 38 */ 8,
    ST_PNT, 4, EF, EE, EK, EK,
    ST_PYR, 5, P2, ED, EA, P1, N0,
    ST_PYR, 5, EB, EC, P2, P1, N0,
    ST_TET, 4, P2, EK, ED, N0,
    ST_TET, 4, EC, EK, P2, N0,
    ST_PYR, 5, EA, EE, P5, P1, N0,
    ST_PYR, 5, P1, P5, EF, EB, N0,
    ST_TET, 4, EF, P5, EE, N0,
    /* case 39 */ 9,
    ST_PNT, 6, EI, EE, EF, EB, EC, EK,
    ST_TET, 4, P0, P5, P1, N0,
    ST_PYR, 5, EI, EE, P5, P0, N0,
    ST_PYR, 5, EI, P0, P2, EK, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_TET, 4, P2, EC, EK, N0,
    ST_PYR, 5, P2, P1, EB, EC, N0,
    ST_PYR, 5, EB, P1, P5, EF, N0,
    ST_TET, 4, EF, P5, EE, N0,
    /* case 40 */ 7,
    ST_PNT, 2, EC, EE,
    ST_PYR, 5, P3, EB, EJ, P5, N0,
    ST_PYR, 5, EL, P3, P5, EF, N0,
    ST_TET, 4, EF, P5, EE, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_TET, 4, EL, P3, N0, EC,
    ST_TET, 4, EB, P3, EC, N0,
    /* case 41 */ 5,
    ST_WDG, 6, P0, P5, P3, EA, EJ, EB,
    ST_PYR, 5, P0, P3, EC, ED, EI,
    ST_PYR, 5, EL, P3, P5, EF, EE,
    ST_PYR, 5, EI, EE, P5, P0, P3,
    ST_PYR, 5, EI, EC, EL, EE, P3,
    /* case 42 */ 7,
    ST_PNT, 5, EE, EA, EF, EL, EC,
    ST_PYR, 5, EA, P1, P3, EC, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_TET, 4, EF, P5, EE, N0,
    ST_TET, 4, P5, P3, P1, N0,
    ST_PYR, 5, EF, EL, P3, P5, N0,
    ST_TET, 4, P3, EL, EC, N0,
    /* case 43 */ 5,
    ST_TET, 4, P1, P0, P5, P3,
    ST_PYR, 5, EC, ED, P0, P3, EI,
    ST_PYR, 5, EL, P3, P5, EF, EE,
    ST_PYR, 5, P0, EI, EE, P5, P3,
    ST_PYR, 5, EC, EL, EE, EI, P3,
    /* case 44 */ 8,
    ST_PNT, 4, EK, ED, EE, EE,
    ST_PYR, 5, P5, P3, EB, EJ, N0,
    ST_PYR, 5, EL, P3, P5, EF, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_TET, 4, EF, P5, EE, N0,
    ST_PYR, 5, EB, P3, P2, ED, N0,
    ST_PYR, 5, P3, EL, EK, P2, N0,
    ST_TET, 4, EK, ED, P2, N0,
    /* case 45 */ 9,
    ST_PNT, 5, EK, EL, EI, EE, EF,
    ST_WDG, 6, EA, EB, EJ, P0, P3, P5,
    ST_TET, 4, P3, P0, P5, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_TET, 4, P2, P0, P3, N0,
    ST_PYR, 5, EL, EK, P2, P3, N0,
    ST_PYR, 5, EK, EI, P0, P2, N0,
    ST_PYR, 5, P5, EF, EL, P3, N0,
    ST_TET, 4, EE, EF, P5, N0,
    /* case 46 */ 9,
    ST_PNT, 6, EA, ED, EK, EL, EF, EE,
    ST_TET, 4, P1, P3, P2, N0,
    ST_PYR, 5, EA, P1, P2, ED, N0,
    ST_PYR, 5, EA, EE, P5, P1, N0,
    ST_TET, 4, P5, P3, P1, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_PYR, 5, P5, EF, EL, P3, N0,
    ST_PYR, 5, EL, EK, P2, P3, N0,
    ST_TET, 4, EK, ED, P2, N0,
    /* case 47 */ 9,
    ST_PNT, 5, EL, EK, EI, EF, EE,
    ST_PYR, 5, P1, P3, P2, P0, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_TET, 4, P5, P3, P1, N0,
    ST_PYR, 5, P5, EF, EL, P3, N0,
    ST_PYR, 5, EL, EK, P2, P3, N0,
    ST_PYR, 5, EK, EI, P0, P2, N0,
    ST_PYR, 5, EI, EE, P5, P0, N0,
    ST_TET, 4, EF, P5, EE, N0,
    /* case 48 */ 1,
    ST_WDG, 6, P5, EJ, EF, P4, EI, EH,
    /* case 49 */ 7,
    ST_PNT, 5, EF, EH, EJ, EA, ED,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_PYR, 5, EJ, EA, P0, P5, N0,
    ST_TET, 4, P0, EA, ED, N0,
    /* case 50 */ 7,
    ST_PNT, 5, EH, EF, EI, EA, EB,
    ST_PYR, 5, EF, EB, P1, P5, N0,
    ST_PYR, 5, EH, EF, P5, P4, N0,
    ST_TET, 4, EI, EH, P4, N0,
    ST_TET, 4, P4, P5, P1, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    ST_TET, 4, P1, EB, EA, N0,
    /* case 51 */ 1,
    ST_HEX, 8, ED, EB, EF, EH, P0, P1, P5, P4,
    /* case 52 */ 8,
    ST_PNT, 4, EF, EJ, EC, EC,
    ST_PYR, 5, P2, P4, EI, ED, N0,
    ST_PYR, 5, EH, P4, P2, EK, N0,
    ST_TET, 4, P2, ED, EC, N0,
    ST_TET, 4, EK, P2, EC, N0,
    ST_PYR, 5, EI, P4, P5, EJ, N0,
    ST_PYR, 5, P4, EH, EF, P5, N0,
    ST_TET, 4, EF, EJ, P5, N0,
    /* case 53 */ 9,
    ST_PNT, 6, EA, EJ, EF, EH, EK, EC,
    ST_TET, 4, P0, P4, P5, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_PYR, 5, EA, EC, P2, P0, N0,
    ST_TET, 4, P2, P4, P0, N0,
    ST_TET, 4, P2, EC, EK, N0,
    ST_PYR, 5, P2, EK, EH, P4, N0,
    ST_PYR, 5, EH, EF, P5, P4, N0,
    ST_TET, 4, EF, EJ, P5, N0,
    /* case 54 */ 9,
    ST_PNT, 5, EF, EB, EH, EK, EC,
    ST_WDG, 6, P4, P1, P2, EI, EA, ED,
    ST_TET, 4, P1, P2, P4, N0,
    ST_PYR, 5, P4, P2, EK, EH, N0,
    ST_TET, 4, P5, P1, P4, N0,
    ST_PYR, 5, EB, P1, P5, EF, N0,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_PYR, 5, P2, P1, EB, EC, N0,
    ST_TET, 4, EK, P2, EC, N0,
    /* case 55 */ 9,
    ST_PNT, 5, EB, EF, EH, EC, EK,
    ST_PYR, 5, P0, P4, P5, P1, N0,
    ST_TET, 4, P4, P0, P2, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_PYR, 5, P2, P1, EB, EC, N0,
    ST_PYR, 5, EB, P1, P5, EF, N0,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_PYR, 5, EH, P4, P2, EK, N0,
    ST_TET, 4, EC, EK, P2, N0,
    /* case 56 */ 8,
    ST_PNT, 4, EH, EI, EC, EC,
    ST_PYR, 5, P3, EB, EJ, P5, N0,
    ST_PYR, 5, EF, EL, P3, P5, N0,
    ST_TET, 4, P3, EC, EB, N0,
    ST_TET, 4, EL, EC, P3, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_TET, 4, EH, P4, EI, N0,
    /* case 57 */ 9,
    ST_PNT, 5, EH, EF, ED, EC, EL,
    ST_WDG, 6, P0, P5, P3, EA, EJ, EB,
    ST_TET, 4, P5, P3, P0, N0,
    ST_PYR, 5, P0, P3, EC, ED, N0,
    ST_TET, 4, P4, P5, P0, N0,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_PYR, 5, P3, P5, EF, EL, N0,
    ST_TET, 4, EC, P3, EL, N0,
    /* case 58 */ 9,
    ST_PNT, 6, EA, EI, EH, EF, EL, EC,
    ST_TET, 4, P1, P4, P5, N0,
    ST_PYR, 5, EA, EI, P4, P1, N0,
    ST_PYR, 5, EA, P1, P3, EC, N0,
    ST_TET, 4, P3, P1, P5, N0,
    ST_TET, 4, P3, EL, EC, N0,
    ST_PYR, 5, P3, P5, EF, EL, N0,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_TET, 4, EH, P4, EI, N0,
    /* case 59 */ 9,
    ST_PNT, 5, EF, EH, ED, EL, EC,
    ST_PYR, 5, P1, P0, P4, P5, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_TET, 4, P3, P1, P5, N0,
    ST_PYR, 5, P3, P5, EF, EL, N0,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_PYR, 5, ED, P0, P3, EC, N0,
    ST_TET, 4, EL, EC, P3, N0,
    /* case 60 */ 2,
    ST_HEX, 8, P2, P4, P5, P3, EK, EH, EF, EL,
    ST_HEX, 8, ED, EI, EJ, EB, P2, P4, P5, P3,
    /* case 61 */ 11,
    ST_PNT, 6, P0, P3, P2, P4, EF, EH,
    ST_WDG, 6, P0, P5, P3, EA, EJ, EB,
    ST_PYR, 5, EH, EF, P5, P4, N0,
    ST_TET, 4, P4, P5, P0, N0,
    ST_TET, 4, P4, P0, P2, N0,
    ST_PYR, 5, EK, EH, P4, P2, N0,
    ST_PYR, 5, EL, EK, P2, P3, N0,
    ST_TET, 4, P2, P0, P3, N0,
    ST_PYR, 5, EF, EH, EK, EL, N0,
    ST_TET, 4, P3, P0, P5, N0,
    ST_PYR, 5, EF, EL, P3, P5, N0,
    /* case 62 */ 11,
    ST_PNT, 6, P1, P2, P3, P5, EH, EF,
    ST_WDG, 6, EA, EI, ED, P1, P4, P2,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_TET, 4, P5, P1, P4, N0,
    ST_TET, 4, P5, P3, P1, N0,
    ST_PYR, 5, EL, P3, P5, EF, N0,
    ST_PYR, 5, EK, P2, P3, EL, N0,
    ST_TET, 4, P3, P2, P1, N0,
    ST_PYR, 5, EH, EK, EL, EF, N0,
    ST_TET, 4, P2, P4, P1, N0,
    ST_PYR, 5, EH, P4, P2, EK, N0,
    /* case 63 */ 2,
    ST_HEX, 8, P2, P4, P5, P3, EK, EH, EF, EL,
    ST_WDG, 6, P1, P3, P5, P0, P2, P4,
    /* case 64 */ 1,
    ST_TET, 4, P6, EG, EH, EK,
    /* case 65 */ 7,
    ST_PNT, 2, EA, EG,
    ST_PYR, 5, P0, ED, EK, P6, N0,
    ST_PYR, 5, EI, P0, P6, EH, N0,
    ST_TET, 4, EH, P6, EG, N0,
    ST_TET, 4, P6, EK, EG, N0,
    ST_TET, 4, EI, P0, N0, EA,
    ST_TET, 4, ED, P0, EA, N0,
    /* case 66 */ 2,
    ST_TET, 4, P1, EA, EJ, EB,
    ST_TET, 4, P6, EH, EK, EG,
    /* case 67 */ 8,
    ST_PNT, 4, EJ, EB, EG, EG,
    ST_PYR, 5, P6, P0, ED, EK, N0,
    ST_PYR, 5, EI, P0, P6, EH, N0,
    ST_TET, 4, P6, EK, EG, N0,
    ST_TET, 4, EH, P6, EG, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_PYR, 5, P0, EI, EJ, P1, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    /* case 68 */ 1,
    ST_WDG, 6, P6, EH, EG, P2, ED, EC,
    /* case 69 */ 7,
    ST_PNT, 5, EG, EC, EH, EI, EA,
    ST_PYR, 5, EC, P2, P0, EA, N0,
    ST_PYR, 5, EG, P6, P2, EC, N0,
    ST_TET, 4, EH, P6, EG, N0,
    ST_TET, 4, P6, P0, P2, N0,
    ST_PYR, 5, EH, EI, P0, P6, N0,
    ST_TET, 4, P0, EI, EA, N0,
    /* case 70 */ 8,
    ST_PNT, 4, EG, EH, EJ, EJ,
    ST_PYR, 5, P1, P2, ED, EA, N0,
    ST_PYR, 5, EC, P2, P1, EB, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_TET, 4, EB, P1, EJ, N0,
    ST_PYR, 5, ED, P2, P6, EH, N0,
    ST_PYR, 5, P2, EC, EG, P6, N0,
    ST_TET, 4, EG, EH, P6, N0,
    /* case 71 */ 9,
    ST_PNT, 6, EI, EH, EG, EC, EB, EJ,
    ST_TET, 4, P0, P2, P6, N0,
    ST_PYR, 5, EI, P0, P6, EH, N0,
    ST_PYR, 5, EI, EJ, P1, P0, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_PYR, 5, P1, EB, EC, P2, N0,
    ST_PYR, 5, EC, EG, P6, P2, N0,
    ST_TET, 4, EG, EH, P6, N0,
    /* case 72 */ 7,
    ST_PNT, 2, EH, EB,
    ST_PYR, 5, P6, EK, EC, P3, N0,
    ST_PYR, 5, EG, P6, P3, EL, N0,
    ST_TET, 4, EL, P3, EB, N0,
    ST_TET, 4, P3, EC, EB, N0,
    ST_TET, 4, EG, P6, N0, EH,
    ST_TET, 4, EK, P6, EH, N0,
    /* case 73 */ 5,
    ST_WDG, 6, P0, P3, P6, ED, EC, EK,
    ST_PYR, 5, P0, P6, EH, EI, EA,
    ST_PYR, 5, EG, P6, P3, EL, EB,
    ST_PYR, 5, EA, EB, P3, P0, P6,
    ST_PYR, 5, EA, EH, EG, EB, P6,
    /* case 74 */ 8,
    ST_PNT, 4, EJ, EA, EH, EH,
    ST_PYR, 5, P6, EK, EC, P3, N0,
    ST_PYR, 5, EL, EG, P6, P3, N0,
    ST_TET, 4, P6, EH, EK, N0,
    ST_TET, 4, EG, EH, P6, N0,
    ST_PYR, 5, EC, EA, P1, P3, N0,
    ST_PYR, 5, P3, P1, EJ, EL, N0,
    ST_TET, 4, EJ, P1, EA, N0,
    /* case 75 */ 9,
    ST_PNT, 5, EJ, EL, EI, EH, EG,
    ST_WDG, 6, P0, P3, P6, ED, EC, EK,
    ST_TET, 4, P3, P6, P0, N0,
    ST_PYR, 5, P0, P6, EH, EI, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_PYR, 5, EL, P3, P1, EJ, N0,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_PYR, 5, P6, P3, EL, EG, N0,
    ST_TET, 4, EH, P6, EG, N0,
    /* case 76 */ 7,
    ST_PNT, 5, EB, ED, EL, EG, EH,
    ST_PYR, 5, ED, P2, P6, EH, N0,
    ST_PYR, 5, EB, P3, P2, ED, N0,
    ST_TET, 4, EL, P3, EB, N0,
    ST_TET, 4, P3, P6, P2, N0,
    ST_PYR, 5, EL, EG, P6, P3, N0,
    ST_TET, 4, P6, EG, EH, N0,
    /* case 77 */ 5,
    ST_TET, 4, P2, P0, P3, P6,
    ST_PYR, 5, EH, EI, P0, P6, EA,
    ST_PYR, 5, EG, P6, P3, EL, EB,
    ST_PYR, 5, P0, EA, EB, P3, P6,
    ST_PYR, 5, EH, EG, EB, EA, P6,
    /* case 78 */ 9,
    ST_PNT, 6, ED, EA, EJ, EL, EG, EH,
    ST_TET, 4, P2, P1, P3, N0,
    ST_PYR, 5, ED, EA, P1, P2, N0,
    ST_PYR, 5, ED, P2, P6, EH, N0,
    ST_TET, 4, P6, P2, P3, N0,
    ST_TET, 4, P6, EG, EH, N0,
    ST_PYR, 5, P6, P3, EL, EG, N0,
    ST_PYR, 5, EL, P3, P1, EJ, N0,
    ST_TET, 4, EJ, P1, EA, N0,
    /* case 79 */ 9,
    ST_PNT, 5, EL, EJ, EI, EG, EH,
    ST_PYR, 5, P2, P0, P1, P3, N0,
    ST_TET, 4, P0, P2, P6, N0,
    ST_TET, 4, P6, P2, P3, N0,
    ST_PYR, 5, P6, P3, EL, EG, N0,
    ST_PYR, 5, EL, P3, P1, EJ, N0,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_PYR, 5, EI, P0, P6, EH, N0,
    ST_TET, 4, EG, EH, P6, N0,
    /* case 80 */ 1,
    ST_WDG, 6, P4, EI, EE, P6, EK, EG,
    /* case 81 */ 7,
    ST_PNT, 5, EG, EE, EK, ED, EA,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_PYR, 5, EG, EE, P4, P6, N0,
    ST_TET, 4, EK, EG, P6, N0,
    ST_TET, 4, P6, P4, P0, N0,
    ST_PYR, 5, EK, P6, P0, ED, N0,
    ST_TET, 4, P0, EA, ED, N0,
    /* case 82 */ 8,
    ST_PNT, 4, EG, EK, EB, EB,
    ST_PYR, 5, P1, EA, EI, P4, N0,
    ST_PYR, 5, EE, EJ, P1, P4, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    ST_PYR, 5, EI, EK, P6, P4, N0,
    ST_PYR, 5, P4, P6, EG, EE, N0,
    ST_TET, 4, EG, P6, EK, N0,
    /* case 83 */ 9,
    ST_PNT, 6, ED, EK, EG, EE, EJ, EB,
    ST_TET, 4, P0, P6, P4, N0,
    ST_PYR, 5, ED, EK, P6, P0, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_PYR, 5, EE, P4, P6, EG, N0,
    ST_TET, 4, EG, P6, EK, N0,
    /* case 84 */ 7,
    ST_PNT, 5, EC, EG, ED, EI, EE,
    ST_PYR, 5, EG, EE, P4, P6, N0,
    ST_PYR, 5, EC, EG, P6, P2, N0,
    ST_TET, 4, ED, EC, P2, N0,
    ST_TET, 4, P2, P6, P4, N0,
    ST_PYR, 5, ED, P2, P4, EI, N0,
    ST_TET, 4, P4, EE, EI, N0,
    /* case 85 */ 1,
    ST_HEX, 8, EA, EE, EG, EC, P0, P4, P6, P2,
    /* case 86 */ 9,
    ST_PNT, 5, EG, EE, EC, EB, EJ,
    ST_WDG, 6, P2, P4, P1, ED, EI, EA,
    ST_TET, 4, P4, P1, P2, N0,
    ST_PYR, 5, P2, P1, EB, EC, N0,
    ST_TET, 4, P6, P4, P2, N0,
    ST_PYR, 5, EE, P4, P6, EG, N0,
    ST_PYR, 5, EG, P6, P2, EC, N0,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_TET, 4, EB, P1, EJ, N0,
    /* case 87 */ 9,
    ST_PNT, 5, EE, EG, EC, EJ, EB,
    ST_PYR, 5, P0, P2, P6, P4, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_PYR, 5, EE, P4, P6, EG, N0,
    ST_PYR, 5, EG, P6, P2, EC, N0,
    ST_PYR, 5, EC, P2, P1, EB, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    /* case 88 */ 8,
    ST_PNT, 4, EE, EI, EB, EB,
    ST_PYR, 5, P3, P6, EK, EC, N0,
    ST_PYR, 5, EG, P6, P3, EL, N0,
    ST_TET, 4, P3, EC, EB, N0,
    ST_TET, 4, EL, P3, EB, N0,
    ST_PYR, 5, EK, P6, P4, EI, N0,
    ST_PYR, 5, P6, EG, EE, P4, N0,
    ST_TET, 4, EE, EI, P4, N0,
    /* case 89 */ 9,
    ST_PNT, 5, EE, EG, EA, EB, EL,
    ST_WDG, 6, ED, EK, EC, P0, P6, P3,
    ST_TET, 4, P6, P0, P3, N0,
    ST_PYR, 5, P0, EA, EB, P3, N0,
    ST_TET, 4, P4, P0, P6, N0,
    ST_PYR, 5, EG, EE, P4, P6, N0,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_PYR, 5, P3, EL, EG, P6, N0,
    ST_TET, 4, EB, EL, P3, N0,
    /* case 90 */ 2,
    ST_HEX, 8, P4, P1, P3, P6, EE, EJ, EL, EG,
    ST_HEX, 8, EI, EA, EC, EK, P4, P1, P3, P6,
    /* case 91 */ 11,
    ST_PNT, 6, P0, P6, P4, P1, EL, EJ,
    ST_WDG, 6, P0, P3, P6, ED, EC, EK,
    ST_PYR, 5, EJ, EL, P3, P1, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_PYR, 5, EE, EJ, P1, P4, N0,
    ST_PYR, 5, EG, EE, P4, P6, N0,
    ST_TET, 4, P4, P0, P6, N0,
    ST_PYR, 5, EL, EJ, EE, EG, N0,
    ST_TET, 4, P6, P0, P3, N0,
    ST_PYR, 5, EL, EG, P6, P3, N0,
    /* case 92 */ 9,
    ST_PNT, 6, ED, EI, EE, EG, EL, EB,
    ST_TET, 4, P2, P6, P4, N0,
    ST_PYR, 5, ED, P2, P4, EI, N0,
    ST_PYR, 5, ED, EB, P3, P2, N0,
    ST_TET, 4, P3, P6, P2, N0,
    ST_TET, 4, P3, EB, EL, N0,
    ST_PYR, 5, P3, EL, EG, P6, N0,
    ST_PYR, 5, EG, EE, P4, P6, N0,
    ST_TET, 4, EE, EI, P4, N0,
    /* case 93 */ 9,
    ST_PNT, 5, EG, EE, EA, EL, EB,
    ST_PYR, 5, P2, P6, P4, P0, N0,
    ST_TET, 4, P0, P3, P2, N0,
    ST_TET, 4, P3, P6, P2, N0,
    ST_PYR, 5, P3, EL, EG, P6, N0,
    ST_PYR, 5, EG, EE, P4, P6, N0,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_PYR, 5, EA, EB, P3, P0, N0,
    ST_TET, 4, EL, P3, EB, N0,
    /* case 94 */ 11,
    ST_PNT, 6, P2, P4, P6, P3, EJ, EL,
    ST_WDG, 6, ED, EA, EI, P2, P1, P4,
    ST_PYR, 5, EL, P3, P1, EJ, N0,
    ST_TET, 4, P3, P2, P1, N0,
    ST_TET, 4, P3, P6, P2, N0,
    ST_PYR, 5, EG, P6, P3, EL, N0,
    ST_PYR, 5, EE, P4, P6, EG, N0,
    ST_TET, 4, P6, P4, P2, N0,
    ST_PYR, 5, EJ, EE, EG, EL, N0,
    ST_TET, 4, P4, P1, P2, N0,
    ST_PYR, 5, EJ, P1, P4, EE, N0,
    /* case 95 */ 2,
    ST_HEX, 8, P4, P1, P3, P6, EE, EJ, EL, EG,
    ST_WDG, 6, P2, P6, P3, P0, P4, P1,
    /* case 96 */ 7,
    ST_PNT, 2, EK, EJ,
    ST_PYR, 5, P6, P5, EE, EH, N0,
    ST_PYR, 5, EG, EF, P5, P6, N0,
    ST_TET, 4, EF, EJ, P5, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_TET, 4, EG, N0, P6, EK,
    ST_TET, 4, EH, EK, P6, N0,
    /* case 97 */ 5,
    ST_WDG, 6, EI, EE, EH, P0, P5, P6,
    ST_PYR, 5, P0, ED, EK, P6, EA,
    ST_PYR, 5, EG, EF, P5, P6, EJ,
    ST_PYR, 5, EA, P0, P5, EJ, P6,
    ST_PYR, 5, EA, EJ, EG, EK, P6,
    /* case 98 */ 8,
    ST_PNT, 4, EB, EA, EK, EK,
    ST_PYR, 5, P6, P5, EE, EH, N0,
    ST_PYR, 5, EF, P5, P6, EG, N0,
    ST_TET, 4, P6, EH, EK, N0,
    ST_TET, 4, EG, P6, EK, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_PYR, 5, P5, EF, EB, P1, N0,
    ST_TET, 4, EB, EA, P1, N0,
    /* case 99 */ 9,
    ST_PNT, 5, EB, EF, ED, EK, EG,
    ST_WDG, 6, EI, EE, EH, P0, P5, P6,
    ST_TET, 4, P5, P0, P6, N0,
    ST_PYR, 5, P0, ED, EK, P6, N0,
    ST_TET, 4, P1, P0, P5, N0,
    ST_PYR, 5, EF, EB, P1, P5, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_PYR, 5, P6, EG, EF, P5, N0,
    ST_TET, 4, EK, EG, P6, N0,
    /* case 100 */ 8,
    ST_PNT, 4, EC, ED, EJ, EJ,
    ST_PYR, 5, P5, EE, EH, P6, N0,
    ST_PYR, 5, EG, EF, P5, P6, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_TET, 4, EF, EJ, P5, N0,
    ST_PYR, 5, EH, ED, P2, P6, N0,
    ST_PYR, 5, P6, P2, EC, EG, N0,
    ST_TET, 4, EC, P2, ED, N0,
    /* case 101 */ 9,
    ST_PNT, 5, EC, EG, EA, EJ, EF,
    ST_WDG, 6, P0, P6, P5, EI, EH, EE,
    ST_TET, 4, P6, P5, P0, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_TET, 4, P2, P6, P0, N0,
    ST_PYR, 5, EG, P6, P2, EC, N0,
    ST_PYR, 5, EC, P2, P0, EA, N0,
    ST_PYR, 5, P5, P6, EG, EF, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    /* case 102 */ 2,
    ST_HEX, 8, EC, EB, EF, EG, P2, P1, P5, P6,
    ST_HEX, 8, P2, P1, P5, P6, ED, EA, EE, EH,
    /* case 103 */ 11,
    ST_PNT, 6, P0, P6, P2, P1, EF, EB,
    ST_WDG, 6, EI, EE, EH, P0, P5, P6,
    ST_PYR, 5, EB, P1, P5, EF, N0,
    ST_TET, 4, P1, P0, P5, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_PYR, 5, EC, P2, P1, EB, N0,
    ST_PYR, 5, EG, P6, P2, EC, N0,
    ST_TET, 4, P2, P6, P0, N0,
    ST_PYR, 5, EF, EG, EC, EB, N0,
    ST_TET, 4, P6, P5, P0, N0,
    ST_PYR, 5, EF, P5, P6, EG, N0,
    /* case 104 */ 5,
    ST_WDG, 6, P5, P6, P3, EF, EG, EL,
    ST_PYR, 5, P5, P3, EB, EJ, EE,
    ST_PYR, 5, EC, P3, P6, EK, EH,
    ST_PYR, 5, EE, EH, P6, P5, P3,
    ST_PYR, 5, EE, EB, EC, EH, P3,
    /* case 105 */ 5,
    ST_WDG, 6, P3, P0, P5, EB, EA, EJ,
    ST_TET, 4, P3, P0, P5, P6,
    ST_WDG, 6, EG, EF, EL, P6, P5, P3,
    ST_WDG, 6, ED, EK, EC, P0, P6, P3,
    ST_WDG, 6, EE, EH, EI, P5, P6, P0,
    /* case 106 */ 9,
    ST_PNT, 5, EA, EC, EE, EH, EK,
    ST_WDG, 6, EF, EL, EG, P5, P3, P6,
    ST_TET, 4, P3, P5, P6, N0,
    ST_PYR, 5, P5, EE, EH, P6, N0,
    ST_TET, 4, P1, P5, P3, N0,
    ST_PYR, 5, EC, EA, P1, P3, N0,
    ST_PYR, 5, EA, EE, P5, P1, N0,
    ST_PYR, 5, P6, EK, EC, P3, N0,
    ST_TET, 4, EH, EK, P6, N0,
    /* case 107 */ 5,
    ST_WDG, 6, EH, EI, EE, P6, P0, P5,
    ST_WDG, 6, P3, P6, P0, EC, EK, ED,
    ST_WDG, 6, EL, EG, EF, P3, P6, P5,
    ST_TET, 4, P6, P5, P0, P3,
    ST_TET, 4, P3, P1, P5, P0,
    /* case 108 */ 9,
    ST_PNT, 5, ED, EH, EB, EJ, EE,
    ST_WDG, 6, EL, EG, EF, P3, P6, P5,
    ST_TET, 4, P6, P3, P5, N0,
    ST_PYR, 5, P3, EB, EJ, P5, N0,
    ST_TET, 4, P2, P3, P6, N0,
    ST_PYR, 5, EH, ED, P2, P6, N0,
    ST_PYR, 5, ED, EB, P3, P2, N0,
    ST_PYR, 5, P5, EE, EH, P6, N0,
    ST_TET, 4, EJ, EE, P5, N0,
    /* case 109 */ 5,
    ST_WDG, 6, EJ, EA, EB, P5, P0, P3,
    ST_WDG, 6, P6, P5, P0, EH, EE, EI,
    ST_WDG, 6, EG, EF, EL, P6, P5, P3,
    ST_TET, 4, P5, P3, P0, P6,
    ST_TET, 4, P6, P2, P3, P0,
    /* case 110 */ 11,
    ST_PNT, 6, P3, P5, P1, P2, EH, ED,
    ST_WDG, 6, EL, EG, EF, P3, P6, P5,
    ST_PYR, 5, ED, P2, P6, EH, N0,
    ST_TET, 4, P2, P3, P6, N0,
    ST_TET, 4, P2, P1, P3, N0,
    ST_PYR, 5, EA, P1, P2, ED, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_TET, 4, P1, P5, P3, N0,
    ST_PYR, 5, EH, EE, EA, ED, N0,
    ST_TET, 4, P5, P6, P3, N0,
    ST_PYR, 5, EH, P6, P5, EE, N0,
    /* case 111 */ 5,
    ST_WDG, 6, EG, EF, EL, P6, P5, P3,
    ST_WDG, 6, P6, P5, P0, EH, EE, EI,
    ST_PYR, 5, P2, P0, P1, P3, P6,
    ST_TET, 4, P5, P1, P0, P6,
    ST_TET, 4, P5, P3, P1, P6,
    /* case 112 */ 7,
    ST_PNT, 5, EJ, EI, EF, EG, EK,
    ST_PYR, 5, EI, EK, P6, P4, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_TET, 4, EF, EJ, P5, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_PYR, 5, EF, P5, P6, EG, N0,
    ST_TET, 4, P6, EK, EG, N0,
    /* case 113 */ 5,
    ST_TET, 4, P4, P5, P0, P6,
    ST_PYR, 5, EK, P6, P0, ED, EA,
    ST_PYR, 5, EG, EF, P5, P6, EJ,
    ST_PYR, 5, P0, P5, EJ, EA, P6,
    ST_PYR, 5, EK, EA, EJ, EG, P6,
    /* case 114 */ 9,
    ST_PNT, 6, EI, EA, EB, EF, EG, EK,
    ST_TET, 4, P4, P5, P1, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    ST_PYR, 5, EI, EK, P6, P4, N0,
    ST_TET, 4, P6, P5, P4, N0,
    ST_TET, 4, P6, EK, EG, N0,
    ST_PYR, 5, P6, EG, EF, P5, N0,
    ST_PYR, 5, EF, EB, P1, P5, N0,
    ST_TET, 4, EB, EA, P1, N0,
    /* case 115 */ 9,
    ST_PNT, 5, EF, EB, ED, EG, EK,
    ST_PYR, 5, P4, P5, P1, P0, N0,
    ST_TET, 4, P0, P6, P4, N0,
    ST_TET, 4, P6, P5, P4, N0,
    ST_PYR, 5, P6, EG, EF, P5, N0,
    ST_PYR, 5, EF, EB, P1, P5, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_PYR, 5, ED, EK, P6, P0, N0,
    ST_TET, 4, EG, P6, EK, N0,
    /* case 116 */ 9,
    ST_PNT, 6, EI, ED, EC, EG, EF, EJ,
    ST_TET, 4, P4, P2, P6, N0,
    ST_PYR, 5, EI, ED, P2, P4, N0,
    ST_PYR, 5, EI, P4, P5, EJ, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    ST_PYR, 5, P5, P6, EG, EF, N0,
    ST_PYR, 5, EG, P6, P2, EC, N0,
    ST_TET, 4, EC, P2, ED, N0,
    /* case 117 */ 9,
    ST_PNT, 5, EG, EC, EA, EF, EJ,
    ST_PYR, 5, P4, P0, P2, P6, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_PYR, 5, P5, P6, EG, EF, N0,
    ST_PYR, 5, EG, P6, P2, EC, N0,
    ST_PYR, 5, EC, P2, P0, EA, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_TET, 4, EF, EJ, P5, N0,
    /* case 118 */ 11,
    ST_PNT, 6, P4, P2, P6, P5, EB, EF,
    ST_WDG, 6, P4, P1, P2, EI, EA, ED,
    ST_PYR, 5, EF, EB, P1, P5, N0,
    ST_TET, 4, P5, P1, P4, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_PYR, 5, EG, EF, P5, P6, N0,
    ST_PYR, 5, EC, EG, P6, P2, N0,
    ST_TET, 4, P6, P4, P2, N0,
    ST_PYR, 5, EB, EF, EG, EC, N0,
    ST_TET, 4, P2, P4, P1, N0,
    ST_PYR, 5, EB, EC, P2, P1, N0,
    /* case 119 */ 2,
    ST_HEX, 8, EC, EB, EF, EG, P2, P1, P5, P6,
    ST_WDG, 6, P0, P2, P1, P4, P6, P5,
    /* case 120 */ 9,
    ST_PNT, 5, EI, EK, EJ, EB, EC,
    ST_WDG, 6, P5, P6, P3, EF, EG, EL,
    ST_TET, 4, P6, P3, P5, N0,
    ST_PYR, 5, P5, P3, EB, EJ, N0,
    ST_TET, 4, P4, P6, P5, N0,
    ST_PYR, 5, EK, P6, P4, EI, N0,
    ST_PYR, 5, EI, P4, P5, EJ, N0,
    ST_PYR, 5, P3, P6, EK, EC, N0,
    ST_TET, 4, EB, P3, EC, N0,
    /* case 121 */ 5,
    ST_WDG, 6, P3, P0, P5, EB, EA, EJ,
    ST_WDG, 6, EK, EC, ED, P6, P3, P0,
    ST_WDG, 6, P6, P3, P5, EG, EL, EF,
    ST_TET, 4, P3, P0, P5, P6,
    ST_TET, 4, P6, P5, P4, P0,
    /* case 122 */ 11,
    ST_PNT, 6, P5, P3, P1, P4, EK, EI,
    ST_WDG, 6, P5, P6, P3, EF, EG, EL,
    ST_PYR, 5, EI, EK, P6, P4, N0,
    ST_TET, 4, P4, P6, P5, N0,
    ST_TET, 4, P4, P5, P1, N0,
    ST_PYR, 5, EA, EI, P4, P1, N0,
    ST_PYR, 5, EC, EA, P1, P3, N0,
    ST_TET, 4, P1, P5, P3, N0,
    ST_PYR, 5, EK, EI, EA, EC, N0,
    ST_TET, 4, P3, P5, P6, N0,
    ST_PYR, 5, EK, EC, P3, P6, N0,
    /* case 123 */ 5,
    ST_WDG, 6, P6, P3, P5, EG, EL, EF,
    ST_WDG, 6, EK, EC, ED, P6, P3, P0,
    ST_PYR, 5, P4, P5, P1, P0, P6,
    ST_TET, 4, P3, P0, P1, P6,
    ST_TET, 4, P3, P1, P5, P6,
    /* case 124 */ 11,
    ST_PNT, 6, P6, P5, P4, P2, EB, ED,
    ST_WDG, 6, P6, P3, P5, EG, EL, EF,
    ST_PYR, 5, ED, EB, P3, P2, N0,
    ST_TET, 4, P2, P3, P6, N0,
    ST_TET, 4, P2, P6, P4, N0,
    ST_PYR, 5, EI, ED, P2, P4, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_TET, 4, P4, P6, P5, N0,
    ST_PYR, 5, EB, ED, EI, EJ, N0,
    ST_TET, 4, P5, P6, P3, N0,
    ST_PYR, 5, EB, EJ, P5, P3, N0,
    /* case 125 */ 5,
    ST_WDG, 6, P3, P5, P6, EL, EF, EG,
    ST_WDG, 6, EB, EJ, EA, P3, P5, P0,
    ST_PYR, 5, P2, P6, P4, P0, P3,
    ST_TET, 4, P5, P0, P4, P3,
    ST_TET, 4, P5, P4, P6, P3,
    /* case 126 */ 6,
    ST_WDG, 6, ED, EA, EI, P2, P1, P4,
    ST_WDG, 6, EL, EG, EF, P3, P6, P5,
    ST_TET, 4, P5, P4, P6, P3,
    ST_TET, 4, P3, P2, P1, P4,
    ST_TET, 4, P2, P3, P6, P4,
    ST_TET, 4, P3, P1, P5, P4,
    /* case 127 */ 9,
    ST_PNT, 7, P1, P0, P2, P3, P5, P4, P6,
    ST_WDG, 6, P5, P6, P3, EF, EG, EL,
    ST_TET, 4, P5, P6, P3, N0,
    ST_PYR, 5, P4, P0, P2, P6, N0,
    ST_TET, 4, P6, P2, P3, N0,
    ST_PYR, 5, P1, P3, P2, P0, N0,
    ST_TET, 4, P1, P5, P3, N0,
    ST_PYR, 5, P1, P0, P4, P5, N0,
    ST_TET, 4, P5, P4, P6, N0,
    /* case 128 */ 1,
    ST_TET, 4, P7, EF, EG, EL,
    /* case 129 */ 2,
    ST_TET, 4, P0, EI, EA, ED,
    ST_TET, 4, P7, EF, EG, EL,
    /* case 130 */ 7,
    ST_PNT, 2, EA, EG,
    ST_PYR, 5, P1, P7, EL, EB, N0,
    ST_PYR, 5, EJ, EF, P7, P1, N0,
    ST_TET, 4, EF, EG, P7, N0,
    ST_TET, 4, P7, EG, EL, N0,
    ST_TET, 4, EJ, N0, P1, EA,
    ST_TET, 4, EB, EA, P1, N0,
    /* case 131 */ 8,
    ST_PNT, 4, EI, ED, EG, EG,
    ST_PYR, 5, P7, EL, EB, P1, N0,
    ST_PYR, 5, EJ, EF, P7, P1, N0,
    ST_TET, 4, P7, EG, EL, N0,
    ST_TET, 4, EF, EG, P7, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_PYR, 5, P1, P0, EI, EJ, N0,
    ST_TET, 4, EI, P0, ED, N0,
    /* case 132 */ 7,
    ST_PNT, 2, ED, EF,
    ST_PYR, 5, P2, P7, EG, EK, N0,
    ST_PYR, 5, EC, EL, P7, P2, N0,
    ST_TET, 4, EL, EF, P7, N0,
    ST_TET, 4, P7, EF, EG, N0,
    ST_TET, 4, EC, N0, P2, ED,
    ST_TET, 4, EK, ED, P2, N0,
    /* case 133 */ 8,
    ST_PNT, 4, EA, EI, EF, EF,
    ST_PYR, 5, P7, EG, EK, P2, N0,
    ST_PYR, 5, EC, EL, P7, P2, N0,
    ST_TET, 4, P7, EF, EG, N0,
    ST_TET, 4, EL, EF, P7, N0,
    ST_PYR, 5, EK, EI, P0, P2, N0,
    ST_PYR, 5, P2, P0, EA, EC, N0,
    ST_TET, 4, EA, P0, EI, N0,
    /* case 134 */ 5,
    ST_WDG, 6, P1, P7, P2, EB, EL, EC,
    ST_PYR, 5, P1, P2, ED, EA, EJ,
    ST_PYR, 5, EK, P2, P7, EG, EF,
    ST_PYR, 5, EJ, EF, P7, P1, P2,
    ST_PYR, 5, EJ, ED, EK, EF, P2,
    /* case 135 */ 9,
    ST_PNT, 5, EI, EK, EJ, EF, EG,
    ST_WDG, 6, EB, EC, EL, P1, P2, P7,
    ST_TET, 4, P2, P1, P7, N0,
    ST_PYR, 5, P1, EJ, EF, P7, N0,
    ST_TET, 4, P0, P1, P2, N0,
    ST_PYR, 5, EK, EI, P0, P2, N0,
    ST_PYR, 5, EI, EJ, P1, P0, N0,
    ST_PYR, 5, P7, EG, EK, P2, N0,
    ST_TET, 4, EF, EG, P7, N0,
    /* case 136 */ 1,
    ST_WDG, 6, P3, EB, EC, P7, EF, EG,
    /* case 137 */ 8,
    ST_PNT, 4, EG, EF, EI, EI,
    ST_PYR, 5, P0, EA, EB, P3, N0,
    ST_PYR, 5, EC, ED, P0, P3, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_TET, 4, ED, EI, P0, N0,
    ST_PYR, 5, EB, EF, P7, P3, N0,
    ST_PYR, 5, P3, P7, EG, EC, N0,
    ST_TET, 4, EG, P7, EF, N0,
    /* case 138 */ 7,
    ST_PNT, 5, EG, EC, EF, EJ, EA,
    ST_PYR, 5, EC, EA, P1, P3, N0,
    ST_PYR, 5, EG, EC, P3, P7, N0,
    ST_TET, 4, EF, EG, P7, N0,
    ST_TET, 4, P7, P3, P1, N0,
    ST_PYR, 5, EF, P7, P1, EJ, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    /* case 139 */ 9,
    ST_PNT, 6, EJ, EF, EG, EC, ED, EI,
    ST_TET, 4, P1, P7, P3, N0,
    ST_PYR, 5, EJ, EF, P7, P1, N0,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_PYR, 5, P0, P3, EC, ED, N0,
    ST_PYR, 5, EC, P3, P7, EG, N0,
    ST_TET, 4, EG, P7, EF, N0,
    /* case 140 */ 7,
    ST_PNT, 5, EF, EB, EG, EK, ED,
    ST_PYR, 5, EB, P3, P2, ED, N0,
    ST_PYR, 5, EF, P7, P3, EB, N0,
    ST_TET, 4, EG, P7, EF, N0,
    ST_TET, 4, P7, P2, P3, N0,
    ST_PYR, 5, EG, EK, P2, P7, N0,
    ST_TET, 4, P2, EK, ED, N0,
    /* case 141 */ 9,
    ST_PNT, 6, EK, EG, EF, EB, EA, EI,
    ST_TET, 4, P2, P3, P7, N0,
    ST_PYR, 5, EK, P2, P7, EG, N0,
    ST_PYR, 5, EK, EI, P0, P2, N0,
    ST_TET, 4, P0, P3, P2, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_PYR, 5, P0, EA, EB, P3, N0,
    ST_PYR, 5, EB, EF, P7, P3, N0,
    ST_TET, 4, EF, EG, P7, N0,
    /* case 142 */ 5,
    ST_TET, 4, P3, P1, P7, P2,
    ST_PYR, 5, ED, EA, P1, P2, EJ,
    ST_PYR, 5, EK, P2, P7, EG, EF,
    ST_PYR, 5, P1, EJ, EF, P7, P2,
    ST_PYR, 5, ED, EK, EF, EJ, P2,
    /* case 143 */ 9,
    ST_PNT, 5, EK, EI, EJ, EG, EF,
    ST_PYR, 5, P3, P2, P0, P1, N0,
    ST_TET, 4, P1, P7, P3, N0,
    ST_TET, 4, P7, P2, P3, N0,
    ST_PYR, 5, P7, EG, EK, P2, N0,
    ST_PYR, 5, EK, EI, P0, P2, N0,
    ST_PYR, 5, EI, EJ, P1, P0, N0,
    ST_PYR, 5, EJ, EF, P7, P1, N0,
    ST_TET, 4, EG, P7, EF, N0,
    /* case 144 */ 7,
    ST_PNT, 2, EI, EL,
    ST_PYR, 5, P4, EH, EG, P7, N0,
    ST_PYR, 5, EE, P4, P7, EF, N0,
    ST_TET, 4, EF, P7, EL, N0,
    ST_TET, 4, P7, EG, EL, N0,
    ST_TET, 4, EE, P4, N0, EI,
    ST_TET, 4, EH, P4, EI, N0,
    /* case 145 */ 8,
    ST_PNT, 4, EA, ED, EL, EL,
    ST_PYR, 5, P7, P4, EH, EG, N0,
    ST_PYR, 5, EE, P4, P7, EF, N0,
    ST_TET, 4, P7, EG, EL, N0,
    ST_TET, 4, EF, P7, EL, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_PYR, 5, P4, EE, EA, P0, N0,
    ST_TET, 4, EA, ED, P0, N0,
    /* case 146 */ 5,
    ST_WDG, 6, EJ, EF, EE, P1, P7, P4,
    ST_PYR, 5, P1, EA, EI, P4, EB,
    ST_PYR, 5, EH, EG, P7, P4, EL,
    ST_PYR, 5, EB, P1, P7, EL, P4,
    ST_PYR, 5, EB, EL, EH, EI, P4,
    /* case 147 */ 9,
    ST_PNT, 5, ED, EH, EB, EL, EG,
    ST_WDG, 6, P1, P4, P7, EJ, EE, EF,
    ST_TET, 4, P4, P7, P1, N0,
    ST_PYR, 5, P1, P7, EL, EB, N0,
    ST_TET, 4, P0, P4, P1, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_PYR, 5, P7, P4, EH, EG, N0,
    ST_TET, 4, EL, P7, EG, N0,
    /* case 148 */ 5,
    ST_WDG, 6, EH, EG, EK, P4, P7, P2,
    ST_PYR, 5, P4, EI, ED, P2, EE,
    ST_PYR, 5, EC, EL, P7, P2, EF,
    ST_PYR, 5, EE, P4, P7, EF, P2,
    ST_PYR, 5, EE, EF, EC, ED, P2,
    /* case 149 */ 9,
    ST_PNT, 5, EA, EC, EE, EF, EL,
    ST_WDG, 6, P4, P2, P7, EH, EK, EG,
    ST_TET, 4, P2, P7, P4, N0,
    ST_PYR, 5, P4, P7, EF, EE, N0,
    ST_TET, 4, P0, P2, P4, N0,
    ST_PYR, 5, EC, P2, P0, EA, N0,
    ST_PYR, 5, EA, P0, P4, EE, N0,
    ST_PYR, 5, P7, P2, EC, EL, N0,
    ST_TET, 4, EF, P7, EL, N0,
    /* case 150 */ 5,
    ST_WDG, 6, EB, EC, EL, P1, P2, P7,
    ST_TET, 4, P1, P7, P2, P4,
    ST_WDG, 6, P4, P7, P1, EE, EF, EJ,
    ST_WDG, 6, P2, P4, P1, ED, EI, EA,
    ST_WDG, 6, P7, P4, P2, EG, EH, EK,
    /* case 151 */ 5,
    ST_WDG, 6, P7, P1, P4, EF, EJ, EE,
    ST_WDG, 6, EC, EL, EB, P2, P7, P1,
    ST_WDG, 6, P2, P7, P4, EK, EG, EH,
    ST_TET, 4, P7, P1, P4, P2,
    ST_TET, 4, P2, P4, P0, P1,
    /* case 152 */ 8,
    ST_PNT, 4, EC, EB, EI, EI,
    ST_PYR, 5, P4, P7, EF, EE, N0,
    ST_PYR, 5, EG, P7, P4, EH, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_TET, 4, EH, P4, EI, N0,
    ST_PYR, 5, EF, P7, P3, EB, N0,
    ST_PYR, 5, P7, EG, EC, P3, N0,
    ST_TET, 4, EC, EB, P3, N0,
    /* case 153 */ 2,
    ST_HEX, 8, P0, P3, P7, P4, EA, EB, EF, EE,
    ST_HEX, 8, ED, EC, EG, EH, P0, P3, P7, P4,
    /* case 154 */ 9,
    ST_PNT, 5, EC, EG, EA, EI, EH,
    ST_WDG, 6, EJ, EF, EE, P1, P7, P4,
    ST_TET, 4, P7, P1, P4, N0,
    ST_PYR, 5, P1, EA, EI, P4, N0,
    ST_TET, 4, P3, P1, P7, N0,
    ST_PYR, 5, EG, EC, P3, P7, N0,
    ST_PYR, 5, EC, EA, P1, P3, N0,
    ST_PYR, 5, P4, EH, EG, P7, N0,
    ST_TET, 4, EI, EH, P4, N0,
    /* case 155 */ 11,
    ST_PNT, 6, P1, P4, P0, P3, EG, EC,
    ST_WDG, 6, EJ, EF, EE, P1, P7, P4,
    ST_PYR, 5, EC, P3, P7, EG, N0,
    ST_TET, 4, P3, P1, P7, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_PYR, 5, ED, P0, P3, EC, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_TET, 4, P0, P4, P1, N0,
    ST_PYR, 5, EG, EH, ED, EC, N0,
    ST_TET, 4, P4, P7, P1, N0,
    ST_PYR, 5, EG, P7, P4, EH, N0,
    /* case 156 */ 9,
    ST_PNT, 5, EB, EF, ED, EI, EE,
    ST_WDG, 6, P2, P7, P4, EK, EG, EH,
    ST_TET, 4, P7, P4, P2, N0,
    ST_PYR, 5, P2, P4, EI, ED, N0,
    ST_TET, 4, P3, P7, P2, N0,
    ST_PYR, 5, EF, P7, P3, EB, N0,
    ST_PYR, 5, EB, P3, P2, ED, N0,
    ST_PYR, 5, P4, P7, EF, EE, N0,
    ST_TET, 4, EI, P4, EE, N0,
    /* case 157 */ 11,
    ST_PNT, 6, P2, P4, P0, P3, EF, EB,
    ST_WDG, 6, P2, P7, P4, EK, EG, EH,
    ST_PYR, 5, EB, EF, P7, P3, N0,
    ST_TET, 4, P3, P7, P2, N0,
    ST_TET, 4, P3, P2, P0, N0,
    ST_PYR, 5, EA, EB, P3, P0, N0,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_TET, 4, P0, P2, P4, N0,
    ST_PYR, 5, EF, EB, EA, EE, N0,
    ST_TET, 4, P4, P2, P7, N0,
    ST_PYR, 5, EF, EE, P4, P7, N0,
    /* case 158 */ 5,
    ST_WDG, 6, P4, P2, P7, EH, EK, EG,
    ST_WDG, 6, EA, EI, ED, P1, P4, P2,
    ST_WDG, 6, P1, P4, P7, EJ, EE, EF,
    ST_TET, 4, P4, P2, P7, P1,
    ST_TET, 4, P1, P7, P3, P2,
    /* case 159 */ 5,
    ST_WDG, 6, P4, P7, P1, EE, EF, EJ,
    ST_WDG, 6, EH, EG, EK, P4, P7, P2,
    ST_PYR, 5, P0, P1, P3, P2, P4,
    ST_TET, 4, P7, P2, P3, P4,
    ST_TET, 4, P7, P3, P1, P4,
    /* case 160 */ 1,
    ST_WDG, 6, P7, EL, EG, P5, EJ, EE,
    /* case 161 */ 8,
    ST_PNT, 4, EG, EL, ED, ED,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_PYR, 5, EE, P5, P0, EI, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_TET, 4, EI, P0, ED, N0,
    ST_PYR, 5, EJ, P5, P7, EL, N0,
    ST_PYR, 5, P5, EE, EG, P7, N0,
    ST_TET, 4, EG, EL, P7, N0,
    /* case 162 */ 7,
    ST_PNT, 5, EG, EE, EL, EB, EA,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_PYR, 5, EG, P7, P5, EE, N0,
    ST_TET, 4, EL, P7, EG, N0,
    ST_TET, 4, P7, P1, P5, N0,
    ST_PYR, 5, EL, EB, P1, P7, N0,
    ST_TET, 4, P1, EB, EA, N0,
    /* case 163 */ 9,
    ST_PNT, 6, EB, EL, EG, EE, EI, ED,
    ST_TET, 4, P1, P5, P7, N0,
    ST_PYR, 5, EB, P1, P7, EL, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_PYR, 5, EE, EG, P7, P5, N0,
    ST_TET, 4, EG, EL, P7, N0,
    /* case 164 */ 8,
    ST_PNT, 4, EE, EJ, ED, ED,
    ST_PYR, 5, P2, EC, EL, P7, N0,
    ST_PYR, 5, EG, EK, P2, P7, N0,
    ST_TET, 4, P2, ED, EC, N0,
    ST_TET, 4, EK, ED, P2, N0,
    ST_PYR, 5, EL, EJ, P5, P7, N0,
    ST_PYR, 5, P7, P5, EE, EG, N0,
    ST_TET, 4, EE, P5, EJ, N0,
    /* case 165 */ 2,
    ST_HEX, 8, EA, EJ, EL, EC, P0, P5, P7, P2,
    ST_HEX, 8, P0, P5, P7, P2, EI, EE, EG, EK,
    /* case 166 */ 9,
    ST_PNT, 5, EE, EG, EA, ED, EK,
    ST_WDG, 6, P1, P7, P2, EB, EL, EC,
    ST_TET, 4, P7, P2, P1, N0,
    ST_PYR, 5, P1, P2, ED, EA, N0,
    ST_TET, 4, P5, P7, P1, N0,
    ST_PYR, 5, EG, P7, P5, EE, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_PYR, 5, P2, P7, EG, EK, N0,
    ST_TET, 4, ED, P2, EK, N0,
    /* case 167 */ 11,
    ST_PNT, 6, P1, P2, P0, P5, EG, EE,
    ST_WDG, 6, P1, P7, P2, EB, EL, EC,
    ST_PYR, 5, EE, EG, P7, P5, N0,
    ST_TET, 4, P5, P7, P1, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_PYR, 5, EI, EE, P5, P0, N0,
    ST_PYR, 5, EK, EI, P0, P2, N0,
    ST_TET, 4, P0, P1, P2, N0,
    ST_PYR, 5, EG, EE, EI, EK, N0,
    ST_TET, 4, P2, P1, P7, N0,
    ST_PYR, 5, EG, EK, P2, P7, N0,
    /* case 168 */ 7,
    ST_PNT, 5, EC, EG, EB, EJ, EE,
    ST_PYR, 5, EG, P7, P5, EE, N0,
    ST_PYR, 5, EC, P3, P7, EG, N0,
    ST_TET, 4, EB, P3, EC, N0,
    ST_TET, 4, P3, P5, P7, N0,
    ST_PYR, 5, EB, EJ, P5, P3, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    /* case 169 */ 9,
    ST_PNT, 5, EG, EE, EC, ED, EI,
    ST_WDG, 6, EB, EJ, EA, P3, P5, P0,
    ST_TET, 4, P5, P3, P0, N0,
    ST_PYR, 5, P3, EC, ED, P0, N0,
    ST_TET, 4, P7, P3, P5, N0,
    ST_PYR, 5, EE, EG, P7, P5, N0,
    ST_PYR, 5, EG, EC, P3, P7, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_TET, 4, ED, EI, P0, N0,
    /* case 170 */ 1,
    ST_HEX, 8, P1, P5, P7, P3, EA, EE, EG, EC,
    /* case 171 */ 9,
    ST_PNT, 5, EE, EG, EC, EI, ED,
    ST_PYR, 5, P1, P5, P7, P3, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_PYR, 5, EE, EG, P7, P5, N0,
    ST_PYR, 5, EG, EC, P3, P7, N0,
    ST_PYR, 5, EC, ED, P0, P3, N0,
    ST_TET, 4, EI, P0, ED, N0,
    /* case 172 */ 9,
    ST_PNT, 6, EB, EJ, EE, EG, EK, ED,
    ST_TET, 4, P3, P5, P7, N0,
    ST_PYR, 5, EB, EJ, P5, P3, N0,
    ST_PYR, 5, EB, P3, P2, ED, N0,
    ST_TET, 4, P2, P3, P7, N0,
    ST_TET, 4, P2, EK, ED, N0,
    ST_PYR, 5, P2, P7, EG, EK, N0,
    ST_PYR, 5, EG, P7, P5, EE, N0,
    ST_TET, 4, EE, P5, EJ, N0,
    /* case 173 */ 11,
    ST_PNT, 6, P3, P0, P2, P7, EE, EG,
    ST_WDG, 6, EB, EJ, EA, P3, P5, P0,
    ST_PYR, 5, EG, P7, P5, EE, N0,
    ST_TET, 4, P7, P3, P5, N0,
    ST_TET, 4, P7, P2, P3, N0,
    ST_PYR, 5, EK, P2, P7, EG, N0,
    ST_PYR, 5, EI, P0, P2, EK, N0,
    ST_TET, 4, P2, P0, P3, N0,
    ST_PYR, 5, EE, EI, EK, EG, N0,
    ST_TET, 4, P0, P5, P3, N0,
    ST_PYR, 5, EE, P5, P0, EI, N0,
    /* case 174 */ 9,
    ST_PNT, 5, EG, EE, EA, EK, ED,
    ST_PYR, 5, P3, P1, P5, P7, N0,
    ST_TET, 4, P1, P3, P2, N0,
    ST_TET, 4, P2, P3, P7, N0,
    ST_PYR, 5, P2, P7, EG, EK, N0,
    ST_PYR, 5, EG, P7, P5, EE, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_PYR, 5, EA, P1, P2, ED, N0,
    ST_TET, 4, EK, ED, P2, N0,
    /* case 175 */ 2,
    ST_HEX, 8, P0, P5, P7, P2, EI, EE, EG, EK,
    ST_WDG, 6, P3, P2, P7, P1, P0, P5,
    /* case 176 */ 7,
    ST_PNT, 5, EL, EJ, EG, EH, EI,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_PYR, 5, EL, EJ, P5, P7, N0,
    ST_TET, 4, EG, EL, P7, N0,
    ST_TET, 4, P7, P5, P4, N0,
    ST_PYR, 5, EG, P7, P4, EH, N0,
    ST_TET, 4, P4, EI, EH, N0,
    /* case 177 */ 9,
    ST_PNT, 6, EH, EG, EL, EJ, EA, ED,
    ST_TET, 4, P4, P7, P5, N0,
    ST_PYR, 5, EH, EG, P7, P4, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_PYR, 5, EJ, P5, P7, EL, N0,
    ST_TET, 4, EL, P7, EG, N0,
    /* case 178 */ 5,
    ST_TET, 4, P5, P7, P1, P4,
    ST_PYR, 5, EI, P4, P1, EA, EB,
    ST_PYR, 5, EH, EG, P7, P4, EL,
    ST_PYR, 5, P1, P7, EL, EB, P4,
    ST_PYR, 5, EI, EB, EL, EH, P4,
    /* case 179 */ 9,
    ST_PNT, 5, EH, ED, EB, EG, EL,
    ST_PYR, 5, P5, P1, P0, P4, N0,
    ST_TET, 4, P1, P5, P7, N0,
    ST_TET, 4, P7, P5, P4, N0,
    ST_PYR, 5, P7, P4, EH, EG, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_PYR, 5, EB, P1, P7, EL, N0,
    ST_TET, 4, EG, EL, P7, N0,
    /* case 180 */ 9,
    ST_PNT, 5, EJ, EL, EI, ED, EC,
    ST_WDG, 6, EH, EG, EK, P4, P7, P2,
    ST_TET, 4, P7, P4, P2, N0,
    ST_PYR, 5, P4, EI, ED, P2, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_PYR, 5, EL, EJ, P5, P7, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_PYR, 5, P2, EC, EL, P7, N0,
    ST_TET, 4, ED, EC, P2, N0,
    /* case 181 */ 11,
    ST_PNT, 6, P4, P2, P0, P5, EL, EJ,
    ST_WDG, 6, EH, EG, EK, P4, P7, P2,
    ST_PYR, 5, EJ, P5, P7, EL, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_PYR, 5, EC, P2, P0, EA, N0,
    ST_TET, 4, P0, P2, P4, N0,
    ST_PYR, 5, EL, EC, EA, EJ, N0,
    ST_TET, 4, P2, P7, P4, N0,
    ST_PYR, 5, EL, P7, P2, EC, N0,
    /* case 182 */ 5,
    ST_WDG, 6, EK, EH, EG, P2, P4, P7,
    ST_WDG, 6, P1, P2, P4, EA, ED, EI,
    ST_WDG, 6, EB, EC, EL, P1, P2, P7,
    ST_TET, 4, P2, P7, P4, P1,
    ST_TET, 4, P1, P5, P7, P4,
    /* case 183 */ 5,
    ST_WDG, 6, EC, EL, EB, P2, P7, P1,
    ST_WDG, 6, P2, P7, P4, EK, EG, EH,
    ST_PYR, 5, P0, P4, P5, P1, P2,
    ST_TET, 4, P7, P5, P4, P2,
    ST_TET, 4, P7, P1, P5, P2,
    /* case 184 */ 9,
    ST_PNT, 6, EJ, EB, EC, EG, EH, EI,
    ST_TET, 4, P5, P7, P3, N0,
    ST_PYR, 5, EJ, P5, P3, EB, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_TET, 4, P4, P7, P5, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_PYR, 5, P4, EH, EG, P7, N0,
    ST_PYR, 5, EG, EC, P3, P7, N0,
    ST_TET, 4, EC, EB, P3, N0,
    /* case 185 */ 11,
    ST_PNT, 6, P5, P0, P4, P7, EC, EG,
    ST_WDG, 6, P5, P3, P0, EJ, EB, EA,
    ST_PYR, 5, EG, EC, P3, P7, N0,
    ST_TET, 4, P7, P3, P5, N0,
    ST_TET, 4, P7, P5, P4, N0,
    ST_PYR, 5, EH, EG, P7, P4, N0,
    ST_PYR, 5, ED, EH, P4, P0, N0,
    ST_TET, 4, P4, P5, P0, N0,
    ST_PYR, 5, EC, EG, EH, ED, N0,
    ST_TET, 4, P0, P5, P3, N0,
    ST_PYR, 5, EC, ED, P0, P3, N0,
    /* case 186 */ 9,
    ST_PNT, 5, EG, EC, EA, EH, EI,
    ST_PYR, 5, P5, P7, P3, P1, N0,
    ST_TET, 4, P1, P4, P5, N0,
    ST_TET, 4, P4, P7, P5, N0,
    ST_PYR, 5, P4, EH, EG, P7, N0,
    ST_PYR, 5, EG, EC, P3, P7, N0,
    ST_PYR, 5, EC, EA, P1, P3, N0,
    ST_PYR, 5, EA, EI, P4, P1, N0,
    ST_TET, 4, EH, P4, EI, N0,
    /* case 187 */ 2,
    ST_HEX, 8, ED, EC, EG, EH, P0, P3, P7, P4,
    ST_WDG, 6, P1, P0, P3, P5, P4, P7,
    /* case 188 */ 11,
    ST_PNT, 6, P7, P4, P5, P3, ED, EB,
    ST_WDG, 6, EG, EK, EH, P7, P2, P4,
    ST_PYR, 5, EB, P3, P2, ED, N0,
    ST_TET, 4, P3, P7, P2, N0,
    ST_TET, 4, P3, P5, P7, N0,
    ST_PYR, 5, EJ, P5, P3, EB, N0,
    ST_PYR, 5, EI, P4, P5, EJ, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_PYR, 5, ED, EI, EJ, EB, N0,
    ST_TET, 4, P4, P2, P7, N0,
    ST_PYR, 5, ED, P2, P4, EI, N0,
    /* case 189 */ 6,
    ST_WDG, 6, P0, P5, P3, EA, EJ, EB,
    ST_WDG, 6, P4, P2, P7, EH, EK, EG,
    ST_TET, 4, P7, P2, P3, P4,
    ST_TET, 4, P4, P5, P0, P3,
    ST_TET, 4, P0, P2, P4, P3,
    ST_TET, 4, P4, P7, P5, P3,
    /* case 190 */ 5,
    ST_WDG, 6, EK, EH, EG, P2, P4, P7,
    ST_WDG, 6, P2, P4, P1, ED, EI, EA,
    ST_PYR, 5, P3, P1, P5, P7, P2,
    ST_TET, 4, P4, P5, P1, P2,
    ST_TET, 4, P4, P7, P5, P2,
    /* case 191 */ 9,
    ST_PNT, 7, P5, P1, P0, P4, P7, P3, P2,
    ST_WDG, 6, EG, EK, EH, P7, P2, P4,
    ST_TET, 4, P7, P4, P2, N0,
    ST_PYR, 5, P3, P2, P0, P1, N0,
    ST_TET, 4, P2, P4, P0, N0,
    ST_PYR, 5, P5, P1, P0, P4, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_PYR, 5, P5, P7, P3, P1, N0,
    ST_TET, 4, P7, P2, P3, N0,
    /* case 192 */ 1,
    ST_WDG, 6, P7, EF, EL, P6, EH, EK,
    /* case 193 */ 8,
    ST_PNT, 4, EL, EF, EA, EA,
    ST_PYR, 5, P0, P6, EH, EI, N0,
    ST_PYR, 5, EK, P6, P0, ED, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_TET, 4, ED, P0, EA, N0,
    ST_PYR, 5, EH, P6, P7, EF, N0,
    ST_PYR, 5, P6, EK, EL, P7, N0,
    ST_TET, 4, EL, EF, P7, N0,
    /* case 194 */ 8,
    ST_PNT, 4, EK, EH, EA, EA,
    ST_PYR, 5, P1, EJ, EF, P7, N0,
    ST_PYR, 5, EL, EB, P1, P7, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_TET, 4, EB, EA, P1, N0,
    ST_PYR, 5, EF, EH, P6, P7, N0,
    ST_PYR, 5, P7, P6, EK, EL, N0,
    ST_TET, 4, EK, P6, EH, N0,
    /* case 195 */ 2,
    ST_HEX, 8, EI, EH, EF, EJ, P0, P6, P7, P1,
    ST_HEX, 8, P0, P6, P7, P1, ED, EK, EL, EB,
    /* case 196 */ 7,
    ST_PNT, 5, EF, EH, EL, EC, ED,
    ST_PYR, 5, EH, ED, P2, P6, N0,
    ST_PYR, 5, EF, EH, P6, P7, N0,
    ST_TET, 4, EL, EF, P7, N0,
    ST_TET, 4, P7, P6, P2, N0,
    ST_PYR, 5, EL, P7, P2, EC, N0,
    ST_TET, 4, P2, ED, EC, N0,
    /* case 197 */ 9,
    ST_PNT, 6, EC, EL, EF, EH, EI, EA,
    ST_TET, 4, P2, P7, P6, N0,
    ST_PYR, 5, EC, EL, P7, P2, N0,
    ST_PYR, 5, EC, P2, P0, EA, N0,
    ST_TET, 4, P0, P2, P6, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_PYR, 5, P0, P6, EH, EI, N0,
    ST_PYR, 5, EH, P6, P7, EF, N0,
    ST_TET, 4, EF, P7, EL, N0,
    /* case 198 */ 9,
    ST_PNT, 5, EH, EF, ED, EA, EJ,
    ST_WDG, 6, EC, EL, EB, P2, P7, P1,
    ST_TET, 4, P7, P2, P1, N0,
    ST_PYR, 5, P2, ED, EA, P1, N0,
    ST_TET, 4, P6, P2, P7, N0,
    ST_PYR, 5, EF, EH, P6, P7, N0,
    ST_PYR, 5, EH, ED, P2, P6, N0,
    ST_PYR, 5, P1, EJ, EF, P7, N0,
    ST_TET, 4, EA, EJ, P1, N0,
    /* case 199 */ 11,
    ST_PNT, 6, P2, P1, P0, P6, EF, EH,
    ST_WDG, 6, EC, EL, EB, P2, P7, P1,
    ST_PYR, 5, EH, P6, P7, EF, N0,
    ST_TET, 4, P6, P2, P7, N0,
    ST_TET, 4, P6, P0, P2, N0,
    ST_PYR, 5, EI, P0, P6, EH, N0,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_TET, 4, P0, P1, P2, N0,
    ST_PYR, 5, EF, EJ, EI, EH, N0,
    ST_TET, 4, P1, P7, P2, N0,
    ST_PYR, 5, EF, P7, P1, EJ, N0,
    /* case 200 */ 7,
    ST_PNT, 5, EH, EF, EK, EC, EB,
    ST_PYR, 5, EF, P7, P3, EB, N0,
    ST_PYR, 5, EH, P6, P7, EF, N0,
    ST_TET, 4, EK, P6, EH, N0,
    ST_TET, 4, P6, P3, P7, N0,
    ST_PYR, 5, EK, EC, P3, P6, N0,
    ST_TET, 4, P3, EC, EB, N0,
    /* case 201 */ 9,
    ST_PNT, 5, EF, EB, EH, EI, EA,
    ST_WDG, 6, EK, EC, ED, P6, P3, P0,
    ST_TET, 4, P3, P6, P0, N0,
    ST_PYR, 5, P6, EH, EI, P0, N0,
    ST_TET, 4, P7, P6, P3, N0,
    ST_PYR, 5, EB, EF, P7, P3, N0,
    ST_PYR, 5, EF, EH, P6, P7, N0,
    ST_PYR, 5, P0, EA, EB, P3, N0,
    ST_TET, 4, EI, EA, P0, N0,
    /* case 202 */ 9,
    ST_PNT, 6, EC, EK, EH, EF, EJ, EA,
    ST_TET, 4, P3, P7, P6, N0,
    ST_PYR, 5, EC, P3, P6, EK, N0,
    ST_PYR, 5, EC, EA, P1, P3, N0,
    ST_TET, 4, P1, P7, P3, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_PYR, 5, P1, EJ, EF, P7, N0,
    ST_PYR, 5, EF, EH, P6, P7, N0,
    ST_TET, 4, EH, EK, P6, N0,
    /* case 203 */ 11,
    ST_PNT, 6, P3, P0, P1, P7, EH, EF,
    ST_WDG, 6, P3, P6, P0, EC, EK, ED,
    ST_PYR, 5, EF, EH, P6, P7, N0,
    ST_TET, 4, P7, P6, P3, N0,
    ST_TET, 4, P7, P3, P1, N0,
    ST_PYR, 5, EJ, EF, P7, P1, N0,
    ST_PYR, 5, EI, EJ, P1, P0, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_PYR, 5, EH, EF, EJ, EI, N0,
    ST_TET, 4, P0, P3, P6, N0,
    ST_PYR, 5, EH, EI, P0, P6, N0,
    /* case 204 */ 1,
    ST_HEX, 8, P2, P3, P7, P6, ED, EB, EF, EH,
    /* case 205 */ 9,
    ST_PNT, 5, EB, EF, EH, EA, EI,
    ST_PYR, 5, P2, P3, P7, P6, N0,
    ST_TET, 4, P6, P0, P2, N0,
    ST_TET, 4, P0, P3, P2, N0,
    ST_PYR, 5, P0, EA, EB, P3, N0,
    ST_PYR, 5, EB, EF, P7, P3, N0,
    ST_PYR, 5, EF, EH, P6, P7, N0,
    ST_PYR, 5, EH, EI, P0, P6, N0,
    ST_TET, 4, EA, P0, EI, N0,
    /* case 206 */ 9,
    ST_PNT, 5, EF, EH, ED, EJ, EA,
    ST_PYR, 5, P3, P7, P6, P2, N0,
    ST_TET, 4, P2, P1, P3, N0,
    ST_TET, 4, P1, P7, P3, N0,
    ST_PYR, 5, P1, EJ, EF, P7, N0,
    ST_PYR, 5, EF, EH, P6, P7, N0,
    ST_PYR, 5, EH, ED, P2, P6, N0,
    ST_PYR, 5, ED, EA, P1, P2, N0,
    ST_TET, 4, EJ, P1, EA, N0,
    /* case 207 */ 2,
    ST_HEX, 8, EI, EH, EF, EJ, P0, P6, P7, P1,
    ST_WDG, 6, P2, P0, P6, P3, P1, P7,
    /* case 208 */ 7,
    ST_PNT, 5, EL, EK, EF, EE, EI,
    ST_PYR, 5, EK, P6, P4, EI, N0,
    ST_PYR, 5, EL, P7, P6, EK, N0,
    ST_TET, 4, EF, P7, EL, N0,
    ST_TET, 4, P7, P4, P6, N0,
    ST_PYR, 5, EF, EE, P4, P7, N0,
    ST_TET, 4, P4, EE, EI, N0,
    /* case 209 */ 9,
    ST_PNT, 6, EE, EF, EL, EK, ED, EA,
    ST_TET, 4, P4, P6, P7, N0,
    ST_PYR, 5, EE, P4, P7, EF, N0,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_TET, 4, P0, P6, P4, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_PYR, 5, P0, ED, EK, P6, N0,
    ST_PYR, 5, EK, EL, P7, P6, N0,
    ST_TET, 4, EL, EF, P7, N0,
    /* case 210 */ 9,
    ST_PNT, 5, EK, EL, EI, EA, EB,
    ST_WDG, 6, P4, P7, P1, EE, EF, EJ,
    ST_TET, 4, P7, P1, P4, N0,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_TET, 4, P6, P7, P4, N0,
    ST_PYR, 5, EL, P7, P6, EK, N0,
    ST_PYR, 5, EK, P6, P4, EI, N0,
    ST_PYR, 5, P1, P7, EL, EB, N0,
    ST_TET, 4, EA, P1, EB, N0,
    /* case 211 */ 11,
    ST_PNT, 6, P4, P1, P0, P6, EL, EK,
    ST_WDG, 6, P4, P7, P1, EE, EF, EJ,
    ST_PYR, 5, EK, EL, P7, P6, N0,
    ST_TET, 4, P6, P7, P4, N0,
    ST_TET, 4, P6, P4, P0, N0,
    ST_PYR, 5, ED, EK, P6, P0, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_TET, 4, P0, P4, P1, N0,
    ST_PYR, 5, EL, EK, ED, EB, N0,
    ST_TET, 4, P1, P4, P7, N0,
    ST_PYR, 5, EL, EB, P1, P7, N0,
    /* case 212 */ 5,
    ST_TET, 4, P6, P7, P4, P2,
    ST_PYR, 5, ED, P2, P4, EI, EE,
    ST_PYR, 5, EC, EL, P7, P2, EF,
    ST_PYR, 5, P4, P7, EF, EE, P2,
    ST_PYR, 5, ED, EE, EF, EC, P2,
    /* case 213 */ 9,
    ST_PNT, 5, EC, EA, EE, EL, EF,
    ST_PYR, 5, P6, P4, P0, P2, N0,
    ST_TET, 4, P4, P6, P7, N0,
    ST_TET, 4, P7, P6, P2, N0,
    ST_PYR, 5, P7, P2, EC, EL, N0,
    ST_PYR, 5, EC, P2, P0, EA, N0,
    ST_PYR, 5, EA, P0, P4, EE, N0,
    ST_PYR, 5, EE, P4, P7, EF, N0,
    ST_TET, 4, EL, EF, P7, N0,
    /* case 214 */ 5,
    ST_WDG, 6, EB, EC, EL, P1, P2, P7,
    ST_WDG, 6, P4, P1, P2, EI, EA, ED,
    ST_WDG, 6, EE, EJ, EF, P4, P1, P7,
    ST_TET, 4, P1, P7, P2, P4,
    ST_TET, 4, P4, P6, P7, P2,
    /* case 215 */ 5,
    ST_WDG, 6, EJ, EF, EE, P1, P7, P4,
    ST_WDG, 6, P1, P7, P2, EB, EL, EC,
    ST_PYR, 5, P0, P2, P6, P4, P1,
    ST_TET, 4, P7, P6, P2, P1,
    ST_TET, 4, P7, P4, P6, P1,
    /* case 216 */ 9,
    ST_PNT, 6, EK, EC, EB, EF, EE, EI,
    ST_TET, 4, P6, P3, P7, N0,
    ST_PYR, 5, EK, EC, P3, P6, N0,
    ST_PYR, 5, EK, P6, P4, EI, N0,
    ST_TET, 4, P4, P6, P7, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_PYR, 5, P4, P7, EF, EE, N0,
    ST_PYR, 5, EF, P7, P3, EB, N0,
    ST_TET, 4, EB, P3, EC, N0,
    /* case 217 */ 11,
    ST_PNT, 6, P6, P0, P4, P7, EB, EF,
    ST_WDG, 6, EK, EC, ED, P6, P3, P0,
    ST_PYR, 5, EF, P7, P3, EB, N0,
    ST_TET, 4, P7, P6, P3, N0,
    ST_TET, 4, P7, P4, P6, N0,
    ST_PYR, 5, EE, P4, P7, EF, N0,
    ST_PYR, 5, EA, P0, P4, EE, N0,
    ST_TET, 4, P4, P0, P6, N0,
    ST_PYR, 5, EB, EA, EE, EF, N0,
    ST_TET, 4, P0, P3, P6, N0,
    ST_PYR, 5, EB, P3, P0, EA, N0,
    /* case 218 */ 11,
    ST_PNT, 6, P7, P1, P3, P6, EI, EK,
    ST_WDG, 6, EF, EE, EJ, P7, P4, P1,
    ST_PYR, 5, EK, P6, P4, EI, N0,
    ST_TET, 4, P6, P7, P4, N0,
    ST_TET, 4, P6, P3, P7, N0,
    ST_PYR, 5, EC, P3, P6, EK, N0,
    ST_PYR, 5, EA, P1, P3, EC, N0,
    ST_TET, 4, P3, P1, P7, N0,
    ST_PYR, 5, EI, EA, EC, EK, N0,
    ST_TET, 4, P1, P4, P7, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    /* case 219 */ 6,
    ST_WDG, 6, P0, P3, P6, ED, EC, EK,
    ST_WDG, 6, P1, P4, P7, EJ, EE, EF,
    ST_TET, 4, P7, P4, P6, P1,
    ST_TET, 4, P1, P3, P0, P6,
    ST_TET, 4, P0, P4, P1, P6,
    ST_TET, 4, P1, P7, P3, P6,
    /* case 220 */ 9,
    ST_PNT, 5, EF, EB, ED, EE, EI,
    ST_PYR, 5, P6, P2, P3, P7, N0,
    ST_TET, 4, P2, P6, P4, N0,
    ST_TET, 4, P4, P6, P7, N0,
    ST_PYR, 5, P4, P7, EF, EE, N0,
    ST_PYR, 5, EF, P7, P3, EB, N0,
    ST_PYR, 5, EB, P3, P2, ED, N0,
    ST_PYR, 5, ED, P2, P4, EI, N0,
    ST_TET, 4, EE, EI, P4, N0,
    /* case 221 */ 2,
    ST_HEX, 8, P0, P3, P7, P4, EA, EB, EF, EE,
    ST_WDG, 6, P6, P4, P7, P2, P0, P3,
    /* case 222 */ 5,
    ST_WDG, 6, EE, EJ, EF, P4, P1, P7,
    ST_WDG, 6, P4, P1, P2, EI, EA, ED,
    ST_PYR, 5, P6, P2, P3, P7, P4,
    ST_TET, 4, P1, P3, P2, P4,
    ST_TET, 4, P1, P7, P3, P4,
    /* case 223 */ 9,
    ST_PNT, 7, P3, P2, P0, P1, P7, P6, P4,
    ST_WDG, 6, EF, EE, EJ, P7, P4, P1,
    ST_TET, 4, P7, P1, P4, N0,
    ST_PYR, 5, P6, P4, P0, P2, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_PYR, 5, P3, P2, P0, P1, N0,
    ST_TET, 4, P3, P1, P7, N0,
    ST_PYR, 5, P3, P7, P6, P2, N0,
    ST_TET, 4, P7, P4, P6, N0,
    /* case 224 */ 7,
    ST_PNT, 5, EK, EL, EH, EE, EJ,
    ST_PYR, 5, EL, EJ, P5, P7, N0,
    ST_PYR, 5, EK, EL, P7, P6, N0,
    ST_TET, 4, EH, EK, P6, N0,
    ST_TET, 4, P6, P7, P5, N0,
    ST_PYR, 5, EH, P6, P5, EE, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    /* case 225 */ 9,
    ST_PNT, 5, EL, EJ, EK, ED, EA,
    ST_WDG, 6, P6, P5, P0, EH, EE, EI,
    ST_TET, 4, P5, P0, P6, N0,
    ST_PYR, 5, P6, P0, ED, EK, N0,
    ST_TET, 4, P7, P5, P6, N0,
    ST_PYR, 5, EJ, P5, P7, EL, N0,
    ST_PYR, 5, EL, P7, P6, EK, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_TET, 4, ED, P0, EA, N0,
    /* case 226 */ 9,
    ST_PNT, 6, EE, EH, EK, EL, EB, EA,
    ST_TET, 4, P5, P6, P7, N0,
    ST_PYR, 5, EE, EH, P6, P5, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_TET, 4, P1, P5, P7, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_PYR, 5, P1, P7, EL, EB, N0,
    ST_PYR, 5, EL, P7, P6, EK, N0,
    ST_TET, 4, EK, P6, EH, N0,
    /* case 227 */ 11,
    ST_PNT, 6, P5, P0, P1, P7, EK, EL,
    ST_WDG, 6, EE, EH, EI, P5, P6, P0,
    ST_PYR, 5, EL, P7, P6, EK, N0,
    ST_TET, 4, P7, P5, P6, N0,
    ST_TET, 4, P7, P1, P5, N0,
    ST_PYR, 5, EB, P1, P7, EL, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_TET, 4, P1, P0, P5, N0,
    ST_PYR, 5, EK, ED, EB, EL, N0,
    ST_TET, 4, P0, P6, P5, N0,
    ST_PYR, 5, EK, P6, P0, ED, N0,
    /* case 228 */ 9,
    ST_PNT, 6, EH, EE, EJ, EL, EC, ED,
    ST_TET, 4, P6, P7, P5, N0,
    ST_PYR, 5, EH, P6, P5, EE, N0,
    ST_PYR, 5, EH, ED, P2, P6, N0,
    ST_TET, 4, P2, P7, P6, N0,
    ST_TET, 4, P2, ED, EC, N0,
    ST_PYR, 5, P2, EC, EL, P7, N0,
    ST_PYR, 5, EL, EJ, P5, P7, N0,
    ST_TET, 4, EJ, EE, P5, N0,
    /* case 229 */ 11,
    ST_PNT, 6, P6, P0, P2, P7, EJ, EL,
    ST_WDG, 6, P6, P5, P0, EH, EE, EI,
    ST_PYR, 5, EL, EJ, P5, P7, N0,
    ST_TET, 4, P7, P5, P6, N0,
    ST_TET, 4, P7, P6, P2, N0,
    ST_PYR, 5, EC, EL, P7, P2, N0,
    ST_PYR, 5, EA, EC, P2, P0, N0,
    ST_TET, 4, P2, P6, P0, N0,
    ST_PYR, 5, EJ, EL, EC, EA, N0,
    ST_TET, 4, P0, P6, P5, N0,
    ST_PYR, 5, EJ, EA, P0, P5, N0,
    /* case 230 */ 11,
    ST_PNT, 6, P7, P1, P5, P6, ED, EH,
    ST_WDG, 6, P7, P2, P1, EL, EC, EB,
    ST_PYR, 5, EH, ED, P2, P6, N0,
    ST_TET, 4, P6, P2, P7, N0,
    ST_TET, 4, P6, P7, P5, N0,
    ST_PYR, 5, EE, EH, P6, P5, N0,
    ST_PYR, 5, EA, EE, P5, P1, N0,
    ST_TET, 4, P5, P7, P1, N0,
    ST_PYR, 5, ED, EH, EE, EA, N0,
    ST_TET, 4, P1, P7, P2, N0,
    ST_PYR, 5, ED, EA, P1, P2, N0,
    /* case 231 */ 6,
    ST_WDG, 6, EI, EE, EH, P0, P5, P6,
    ST_WDG, 6, EB, EC, EL, P1, P2, P7,
    ST_TET, 4, P7, P6, P2, P1,
    ST_TET, 4, P1, P0, P5, P6,
    ST_TET, 4, P0, P1, P2, P6,
    ST_TET, 4, P1, P5, P7, P6,
    /* case 232 */ 5,
    ST_TET, 4, P7, P5, P6, P3,
    ST_PYR, 5, EB, EJ, P5, P3, EE,
    ST_PYR, 5, EC, P3, P6, EK, EH,
    ST_PYR, 5, P5, EE, EH, P6, P3,
    ST_PYR, 5, EB, EC, EH, EE, P3,
    /* case 233 */ 5,
    ST_WDG, 6, ED, EK, EC, P0, P6, P3,
    ST_WDG, 6, P5, P0, P6, EE, EI, EH,
    ST_WDG, 6, EJ, EA, EB, P5, P0, P3,
    ST_TET, 4, P0, P3, P6, P5,
    ST_TET, 4, P5, P7, P3, P6,
    /* case 234 */ 9,
    ST_PNT, 5, EC, EA, EE, EK, EH,
    ST_PYR, 5, P7, P3, P1, P5, N0,
    ST_TET, 4, P5, P6, P7, N0,
    ST_TET, 4, P6, P3, P7, N0,
    ST_PYR, 5, P6, EK, EC, P3, N0,
    ST_PYR, 5, EC, EA, P1, P3, N0,
    ST_PYR, 5, EA, EE, P5, P1, N0,
    ST_PYR, 5, EE, EH, P6, P5, N0,
    ST_TET, 4, EK, P6, EH, N0,
    /* case 235 */ 5,
    ST_WDG, 6, P0, P6, P5, EI, EH, EE,
    ST_WDG, 6, ED, EK, EC, P0, P6, P3,
    ST_PYR, 5, P1, P5, P7, P3, P0,
    ST_TET, 4, P6, P3, P7, P0,
    ST_TET, 4, P6, P7, P5, P0,
    /* case 236 */ 9,
    ST_PNT, 5, EH, ED, EB, EE, EJ,
    ST_PYR, 5, P7, P6, P2, P3, N0,
    ST_TET, 4, P3, P5, P7, N0,
    ST_TET, 4, P5, P6, P7, N0,
    ST_PYR, 5, P5, EE, EH, P6, N0,
    ST_PYR, 5, EH, ED, P2, P6, N0,
    ST_PYR, 5, ED, EB, P3, P2, N0,
    ST_PYR, 5, EB, EJ, P5, P3, N0,
    ST_TET, 4, EE, P5, EJ, N0,
    /* case 237 */ 5,
    ST_WDG, 6, P0, P5, P3, EA, EJ, EB,
    ST_WDG, 6, EI, EE, EH, P0, P5, P6,
    ST_PYR, 5, P2, P3, P7, P6, P0,
    ST_TET, 4, P5, P6, P7, P0,
    ST_TET, 4, P5, P7, P3, P0,
    /* case 238 */ 2,
    ST_HEX, 8, EA, ED, EH, EE, P1, P2, P6, P5,
    ST_WDG, 6, P3, P1, P2, P7, P5, P6,
    /* case 239 */ 9,
    ST_PNT, 7, P7, P3, P1, P5, P6, P2, P0,
    ST_WDG, 6, EH, EI, EE, P6, P0, P5,
    ST_TET, 4, P6, P5, P0, N0,
    ST_PYR, 5, P2, P0, P1, P3, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_PYR, 5, P7, P3, P1, P5, N0,
    ST_TET, 4, P7, P5, P6, N0,
    ST_PYR, 5, P7, P6, P2, P3, N0,
    ST_TET, 4, P6, P0, P2, N0,
    /* case 240 */ 1,
    ST_HEX, 8, EI, EJ, EL, EK, P4, P5, P7, P6,
    /* case 241 */ 9,
    ST_PNT, 5, EJ, EL, EK, EA, ED,
    ST_PYR, 5, P4, P6, P7, P5, N0,
    ST_TET, 4, P6, P4, P0, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_PYR, 5, EJ, P5, P7, EL, N0,
    ST_PYR, 5, EL, P7, P6, EK, N0,
    ST_PYR, 5, EK, P6, P0, ED, N0,
    ST_TET, 4, EA, ED, P0, N0,
    /* case 242 */ 9,
    ST_PNT, 5, EL, EK, EI, EB, EA,
    ST_PYR, 5, P5, P4, P6, P7, N0,
    ST_TET, 4, P4, P5, P1, N0,
    ST_TET, 4, P1, P5, P7, N0,
    ST_PYR, 5, P1, P7, EL, EB, N0,
    ST_PYR, 5, EL, P7, P6, EK, N0,
    ST_PYR, 5, EK, P6, P4, EI, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    ST_TET, 4, EB, EA, P1, N0,
    /* case 243 */ 2,
    ST_HEX, 8, P0, P6, P7, P1, ED, EK, EL, EB,
    ST_WDG, 6, P5, P1, P7, P4, P0, P6,
    /* case 244 */ 9,
    ST_PNT, 5, EL, EJ, EI, EC, ED,
    ST_PYR, 5, P6, P7, P5, P4, N0,
    ST_TET, 4, P4, P2, P6, N0,
    ST_TET, 4, P2, P7, P6, N0,
    ST_PYR, 5, P2, EC, EL, P7, N0,
    ST_PYR, 5, EL, EJ, P5, P7, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_PYR, 5, EI, ED, P2, P4, N0,
    ST_TET, 4, EC, P2, ED, N0,
    /* case 245 */ 2,
    ST_HEX, 8, EA, EJ, EL, EC, P0, P5, P7, P2,
    ST_WDG, 6, P4, P0, P5, P6, P2, P7,
    /* case 246 */ 5,
    ST_WDG, 6, P2, P1, P7, EC, EB, EL,
    ST_WDG, 6, ED, EA, EI, P2, P1, P4,
    ST_PYR, 5, P6, P7, P5, P4, P2,
    ST_TET, 4, P1, P4, P5, P2,
    ST_TET, 4, P1, P5, P7, P2,
    /* case 247 */ 9,
    ST_PNT, 7, P5, P4, P0, P1, P7, P6, P2,
    ST_WDG, 6, P7, P2, P1, EL, EC, EB,
    ST_TET, 4, P7, P2, P1, N0,
    ST_PYR, 5, P6, P4, P0, P2, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_PYR, 5, P5, P1, P0, P4, N0,
    ST_TET, 4, P5, P7, P1, N0,
    ST_PYR, 5, P5, P4, P6, P7, N0,
    ST_TET, 4, P7, P6, P2, N0,
    /* case 248 */ 9,
    ST_PNT, 5, EK, EI, EJ, EC, EB,
    ST_PYR, 5, P7, P5, P4, P6, N0,
    ST_TET, 4, P5, P7, P3, N0,
    ST_TET, 4, P3, P7, P6, N0,
    ST_PYR, 5, P3, P6, EK, EC, N0,
    ST_PYR, 5, EK, P6, P4, EI, N0,
    ST_PYR, 5, EI, P4, P5, EJ, N0,
    ST_PYR, 5, EJ, P5, P3, EB, N0,
    ST_TET, 4, EC, EB, P3, N0,
    /* case 249 */ 5,
    ST_WDG, 6, EA, EB, EJ, P0, P3, P5,
    ST_WDG, 6, P0, P3, P6, ED, EC, EK,
    ST_PYR, 5, P4, P6, P7, P5, P0,
    ST_TET, 4, P3, P7, P6, P0,
    ST_TET, 4, P3, P5, P7, P0,
    /* case 250 */ 2,
    ST_HEX, 8, P1, P4, P6, P3, EA, EI, EK, EC,
    ST_WDG, 6, P7, P3, P6, P5, P1, P4,
    /* case 251 */ 9,
    ST_PNT, 7, P7, P5, P1, P3, P6, P4, P0,
    ST_WDG, 6, P6, P0, P3, EK, ED, EC,
    ST_TET, 4, P6, P0, P3, N0,
    ST_PYR, 5, P4, P5, P1, P0, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_PYR, 5, P7, P3, P1, P5, N0,
    ST_TET, 4, P7, P6, P3, N0,
    ST_PYR, 5, P7, P5, P4, P6, N0,
    ST_TET, 4, P6, P4, P0, N0,
    /* case 252 */ 2,
    ST_HEX, 8, P4, P2, P3, P5, EI, ED, EB, EJ,
    ST_WDG, 6, P7, P5, P3, P6, P4, P2,
    /* case 253 */ 9,
    ST_PNT, 7, P7, P6, P4, P5, P3, P2, P0,
    ST_WDG, 6, P3, P0, P5, EB, EA, EJ,
    ST_TET, 4, P3, P0, P5, N0,
    ST_PYR, 5, P2, P6, P4, P0, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_PYR, 5, P7, P5, P4, P6, N0,
    ST_TET, 4, P7, P3, P5, N0,
    ST_PYR, 5, P7, P6, P2, P3, N0,
    ST_TET, 4, P3, P2, P0, N0,
    /* case 254 */ 9,
    ST_PNT, 7, P6, P7, P5, P4, P2, P3, P1,
    ST_WDG, 6, ED, EA, EI, P2, P1, P4,
    ST_TET, 4, P2, P4, P1, N0,
    ST_PYR, 5, P3, P1, P5, P7, N0,
    ST_TET, 4, P1, P4, P5, N0,
    ST_PYR, 5, P6, P7, P5, P4, N0,
    ST_TET, 4, P6, P4, P2, N0,
    ST_PYR, 5, P6, P2, P3, P7, N0,
    ST_TET, 4, P2, P1, P3, N0,
    /* case 255 */ 1,
    ST_HEX, 8, P0, P1, P3, P2, P4, P5, P7, P6,
    // VTK_HEXAHEDRON
    /* case 0 */ 0,
    /* case 1 */ 1,
    ST_TET, 4, P0, EA, ED, EI,
    /* case 2 */ 1,
    ST_TET, 4, P1, EA, EJ, EB,
    /* case 3 */ 1,
    ST_WDG, 6, P1, EB, EJ, P0, ED, EI,
    /* case 4 */ 1,
    ST_TET, 4, P2, EB, EL, EC,
    /* case 5 */ 7,
    ST_PNT, 2, EI, EL,
    ST_PYR, 5, P0, P2, EC, ED, N0,
    ST_PYR, 5, EA, EB, P2, P0, N0,
    ST_TET, 4, EB, EL, P2, N0,
    ST_TET, 4, P2, EL, EC, N0,
    ST_TET, 4, EA, N0, P0, EI,
    ST_TET, 4, ED, EI, P0, N0,
    /* case 6 */ 1,
    ST_WDG, 6, P2, EC, EL, P1, EA, EJ,
    /* case 7 */ 7,
    ST_PNT, 5, EI, EJ, ED, EC, EL,
    ST_PYR, 5, EJ, EL, P2, P1, N0,
    ST_PYR, 5, EI, EJ, P1, P0, N0,
    ST_TET, 4, ED, EI, P0, N0,
    ST_TET, 4, P0, P1, P2, N0,
    ST_PYR, 5, ED, P0, P2, EC, N0,
    ST_TET, 4, P2, EL, EC, N0,
    /* case 8 */ 1,
    ST_TET, 4, P3, ED, EC, EK,
    /* case 9 */ 1,
    ST_WDG, 6, P3, EK, EC, P0, EI, EA,
    /* case 10 */ 7,
    ST_PNT, 2, EK, EJ,
    ST_PYR, 5, P3, ED, EA, P1, N0,
    ST_PYR, 5, EC, P3, P1, EB, N0,
    ST_TET, 4, EB, P1, EJ, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_TET, 4, EC, P3, N0, EK,
    ST_TET, 4, ED, P3, EK, N0,
    /* case 11 */ 7,
    ST_PNT, 5, EJ, EI, EB, EC, EK,
    ST_PYR, 5, EI, P0, P3, EK, N0,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_TET, 4, EB, P1, EJ, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_PYR, 5, EB, EC, P3, P1, N0,
    ST_TET, 4, P3, EC, EK, N0,
    /* case 12 */ 1,
    ST_WDG, 6, P3, ED, EK, P2, EB, EL,
    /* case 13 */ 7,
    ST_PNT, 5, EI, EK, EA, EB, EL,
    ST_PYR, 5, EK, P3, P2, EL, N0,
    ST_PYR, 5, EI, P0, P3, EK, N0,
    ST_TET, 4, EA, P0, EI, N0,
    ST_TET, 4, P0, P2, P3, N0,
    ST_PYR, 5, EA, EB, P2, P0, N0,
    ST_TET, 4, P2, EB, EL, N0,
    /* case 14 */ 7,
    ST_PNT, 5, EK, EL, ED, EA, EJ,
    ST_PYR, 5, EL, P2, P1, EJ, N0,
    ST_PYR, 5, EK, P3, P2, EL, N0,
    ST_TET, 4, ED, P3, EK, N0,
    ST_TET, 4, P3, P1, P2, N0,
    ST_PYR, 5, ED, EA, P1, P3, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    /* case 15 */ 1,
    ST_HEX, 8, P0, P1, P2, P3, EI, EJ, EL, EK,
    /* case 16 */ 1,
    ST_TET, 4, P4, EE, EI, EH,
    /* case 17 */ 1,
    ST_WDG, 6, P0, ED, EA, P4, EH, EE,
    /* case 18 */ 7,
    ST_PNT, 2, EH, EB,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_PYR, 5, EE, EJ, P1, P4, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_TET, 4, EE, N0, P4, EH,
    ST_TET, 4, EI, EH, P4, N0,
    /* case 19 */ 7,
    ST_PNT, 5, EB, ED, EJ, EE, EH,
    ST_PYR, 5, ED, EH, P4, P0, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_PYR, 5, EJ, P1, P4, EE, N0,
    ST_TET, 4, P4, EH, EE, N0,
    /* case 20 */ 2,
    ST_TET, 4, P4, EE, EI, EH,
    ST_TET, 4, P2, EC, EB, EL,
    /* case 21 */ 8,
    ST_PNT, 4, EE, EH, EL, EL,
    ST_PYR, 5, P2, EC, ED, P0, N0,
    ST_PYR, 5, EA, EB, P2, P0, N0,
    ST_TET, 4, P2, EL, EC, N0,
    ST_TET, 4, EB, EL, P2, N0,
    ST_PYR, 5, ED, EH, P4, P0, N0,
    ST_PYR, 5, P0, P4, EE, EA, N0,
    ST_TET, 4, EE, P4, EH, N0,
    /* case 22 */ 8,
    ST_PNT, 4, EL, EC, EH, EH,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_PYR, 5, EJ, P1, P4, EE, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_TET, 4, EE, P4, EH, N0,
    ST_PYR, 5, EA, P1, P2, EC, N0,
    ST_PYR, 5, P1, EJ, EL, P2, N0,
    ST_TET, 4, EL, EC, P2, N0,
    /* case 23 */ 9,
    ST_PNT, 6, ED, EC, EL, EJ, EE, EH,
    ST_TET, 4, P0, P1, P2, N0,
    ST_PYR, 5, ED, P0, P2, EC, N0,
    ST_PYR, 5, ED, EH, P4, P0, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_TET, 4, P4, EH, EE, N0,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_PYR, 5, EJ, EL, P2, P1, N0,
    ST_TET, 4, EL, EC, P2, N0,
    /* case 24 */ 7,
    ST_PNT, 2, EC, EE,
    ST_PYR, 5, P3, P4, EI, ED, N0,
    ST_PYR, 5, EK, EH, P4, P3, N0,
    ST_TET, 4, EH, EE, P4, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_TET, 4, EK, N0, P3, EC,
    ST_TET, 4, ED, EC, P3, N0,
    /* case 25 */ 7,
    ST_PNT, 5, EE, EA, EH, EK, EC,
    ST_PYR, 5, EA, EC, P3, P0, N0,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_TET, 4, EH, EE, P4, N0,
    ST_TET, 4, P4, P0, P3, N0,
    ST_PYR, 5, EH, P4, P3, EK, N0,
    ST_TET, 4, P3, EC, EK, N0,
    /* case 26 */ 5,
    ST_WDG, 6, ED, EA, EI, P3, P1, P4,
    ST_PYR, 5, P3, EK, EH, P4, EC,
    ST_PYR, 5, EE, EJ, P1, P4, EB,
    ST_PYR, 5, EC, P3, P1, EB, P4,
    ST_PYR, 5, EC, EB, EE, EH, P4,
    /* case 27 */ 5,
    ST_TET, 4, P0, P1, P3, P4,
    ST_PYR, 5, EH, P4, P3, EK, EC,
    ST_PYR, 5, EE, EJ, P1, P4, EB,
    ST_PYR, 5, P3, P1, EB, EC, P4,
    ST_PYR, 5, EH, EC, EB, EE, P4,
    /* case 28 */ 8,
    ST_PNT, 4, EL, EB, EE, EE,
    ST_PYR, 5, P4, EI, ED, P3, N0,
    ST_PYR, 5, EK, EH, P4, P3, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_TET, 4, EH, EE, P4, N0,
    ST_PYR, 5, ED, EB, P2, P3, N0,
    ST_PYR, 5, P3, P2, EL, EK, N0,
    ST_TET, 4, EL, P2, EB, N0,
    /* case 29 */ 9,
    ST_PNT, 6, EA, EB, EL, EK, EH, EE,
    ST_TET, 4, P0, P2, P3, N0,
    ST_PYR, 5, EA, EB, P2, P0, N0,
    ST_PYR, 5, EA, P0, P4, EE, N0,
    ST_TET, 4, P4, P0, P3, N0,
    ST_TET, 4, P4, EH, EE, N0,
    ST_PYR, 5, P4, P3, EK, EH, N0,
    ST_PYR, 5, EK, P3, P2, EL, N0,
    ST_TET, 4, EL, P2, EB, N0,
    /* case 30 */ 9,
    ST_PNT, 5, EL, EJ, EK, EH, EE,
    ST_WDG, 6, ED, EA, EI, P3, P1, P4,
    ST_TET, 4, P1, P3, P4, N0,
    ST_PYR, 5, P3, EK, EH, P4, N0,
    ST_TET, 4, P2, P3, P1, N0,
    ST_PYR, 5, EJ, EL, P2, P1, N0,
    ST_PYR, 5, EL, EK, P3, P2, N0,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_TET, 4, EH, EE, P4, N0,
    /* case 31 */ 9,
    ST_PNT, 5, EJ, EL, EK, EE, EH,
    ST_PYR, 5, P0, P1, P2, P3, N0,
    ST_TET, 4, P3, P4, P0, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_PYR, 5, EJ, EL, P2, P1, N0,
    ST_PYR, 5, EL, EK, P3, P2, N0,
    ST_PYR, 5, EK, EH, P4, P3, N0,
    ST_TET, 4, EE, P4, EH, N0,
    /* case 32 */ 1,
    ST_TET, 4, P5, EF, EJ, EE,
    /* case 33 */ 7,
    ST_PNT, 2, ED, EF,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_TET, 4, EA, P0, N0, ED,
    ST_TET, 4, EI, P0, ED, N0,
    /* case 34 */ 1,
    ST_WDG, 6, P1, EA, EB, P5, EE, EF,
    /* case 35 */ 7,
    ST_PNT, 5, ED, EB, EI, EE, EF,
    ST_PYR, 5, EB, P1, P5, EF, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_TET, 4, EI, P0, ED, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_PYR, 5, EI, EE, P5, P0, N0,
    ST_TET, 4, P5, EE, EF, N0,
    /* case 36 */ 7,
    ST_PNT, 2, EC, EE,
    ST_PYR, 5, P2, EB, EJ, P5, N0,
    ST_PYR, 5, EL, P2, P5, EF, N0,
    ST_TET, 4, EF, P5, EE, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_TET, 4, EL, P2, N0, EC,
    ST_TET, 4, EB, P2, EC, N0,
    /* case 37 */ 5,
    ST_WDG, 6, P0, P5, P2, EA, EJ, EB,
    ST_PYR, 5, P0, P2, EC, ED, EI,
    ST_PYR, 5, EL, P2, P5, EF, EE,
    ST_PYR, 5, EI, EE, P5, P0, P2,
    ST_PYR, 5, EI, EC, EL, EE, P2,
    /* case 38 */ 7,
    ST_PNT, 5, EE, EA, EF, EL, EC,
    ST_PYR, 5, EA, P1, P2, EC, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_TET, 4, EF, P5, EE, N0,
    ST_TET, 4, P5, P2, P1, N0,
    ST_PYR, 5, EF, EL, P2, P5, N0,
    ST_TET, 4, P2, EL, EC, N0,
    /* case 39 */ 5,
    ST_TET, 4, P1, P0, P5, P2,
    ST_PYR, 5, EC, ED, P0, P2, EI,
    ST_PYR, 5, EL, P2, P5, EF, EE,
    ST_PYR, 5, P0, EI, EE, P5, P2,
    ST_PYR, 5, EC, EL, EE, EI, P2,
    /* case 40 */ 2,
    ST_TET, 4, P3, ED, EC, EK,
    ST_TET, 4, P5, EJ, EE, EF,
    /* case 41 */ 8,
    ST_PNT, 4, EC, EK, EF, EF,
    ST_PYR, 5, P5, P0, EI, EE, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    ST_PYR, 5, EI, P0, P3, EK, N0,
    ST_PYR, 5, P0, EA, EC, P3, N0,
    ST_TET, 4, EC, EK, P3, N0,
    /* case 42 */ 8,
    ST_PNT, 4, EF, EE, EK, EK,
    ST_PYR, 5, P3, ED, EA, P1, N0,
    ST_PYR, 5, EB, EC, P3, P1, N0,
    ST_TET, 4, P3, EK, ED, N0,
    ST_TET, 4, EC, EK, P3, N0,
    ST_PYR, 5, EA, EE, P5, P1, N0,
    ST_PYR, 5, P1, P5, EF, EB, N0,
    ST_TET, 4, EF, P5, EE, N0,
    /* case 43 */ 9,
    ST_PNT, 6, EI, EE, EF, EB, EC, EK,
    ST_TET, 4, P0, P5, P1, N0,
    ST_PYR, 5, EI, EE, P5, P0, N0,
    ST_PYR, 5, EI, P0, P3, EK, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_TET, 4, P3, EC, EK, N0,
    ST_PYR, 5, P3, P1, EB, EC, N0,
    ST_PYR, 5, EB, P1, P5, EF, N0,
    ST_TET, 4, EF, P5, EE, N0,
    /* case 44 */ 8,
    ST_PNT, 4, EK, ED, EE, EE,
    ST_PYR, 5, P5, P2, EB, EJ, N0,
    ST_PYR, 5, EL, P2, P5, EF, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_TET, 4, EF, P5, EE, N0,
    ST_PYR, 5, EB, P2, P3, ED, N0,
    ST_PYR, 5, P2, EL, EK, P3, N0,
    ST_TET, 4, EK, ED, P3, N0,
    /* case 45 */ 9,
    ST_PNT, 5, EK, EL, EI, EE, EF,
    ST_WDG, 6, EA, EB, EJ, P0, P2, P5,
    ST_TET, 4, P2, P0, P5, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_TET, 4, P3, P0, P2, N0,
    ST_PYR, 5, EL, EK, P3, P2, N0,
    ST_PYR, 5, EK, EI, P0, P3, N0,
    ST_PYR, 5, P5, EF, EL, P2, N0,
    ST_TET, 4, EE, EF, P5, N0,
    /* case 46 */ 9,
    ST_PNT, 6, EA, ED, EK, EL, EF, EE,
    ST_TET, 4, P1, P2, P3, N0,
    ST_PYR, 5, EA, P1, P3, ED, N0,
    ST_PYR, 5, EA, EE, P5, P1, N0,
    ST_TET, 4, P5, P2, P1, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_PYR, 5, P5, EF, EL, P2, N0,
    ST_PYR, 5, EL, EK, P3, P2, N0,
    ST_TET, 4, EK, ED, P3, N0,
    /* case 47 */ 9,
    ST_PNT, 5, EL, EK, EI, EF, EE,
    ST_PYR, 5, P1, P2, P3, P0, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_TET, 4, P5, P2, P1, N0,
    ST_PYR, 5, P5, EF, EL, P2, N0,
    ST_PYR, 5, EL, EK, P3, P2, N0,
    ST_PYR, 5, EK, EI, P0, P3, N0,
    ST_PYR, 5, EI, EE, P5, P0, N0,
    ST_TET, 4, EF, P5, EE, N0,
    /* case 48 */ 1,
    ST_WDG, 6, P5, EJ, EF, P4, EI, EH,
    /* case 49 */ 7,
    ST_PNT, 5, ED, EH, EA, EJ, EF,
    ST_PYR, 5, EH, EF, P5, P4, N0,
    ST_PYR, 5, ED, EH, P4, P0, N0,
    ST_TET, 4, EA, ED, P0, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    /* case 50 */ 7,
    ST_PNT, 5, EH, EF, EI, EA, EB,
    ST_PYR, 5, EF, EB, P1, P5, N0,
    ST_PYR, 5, EH, EF, P5, P4, N0,
    ST_TET, 4, EI, EH, P4, N0,
    ST_TET, 4, P4, P5, P1, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    ST_TET, 4, P1, EB, EA, N0,
    /* case 51 */ 1,
    ST_HEX, 8, ED, EB, EF, EH, P0, P1, P5, P4,
    /* case 52 */ 8,
    ST_PNT, 4, EH, EI, EC, EC,
    ST_PYR, 5, P2, EB, EJ, P5, N0,
    ST_PYR, 5, EF, EL, P2, P5, N0,
    ST_TET, 4, P2, EC, EB, N0,
    ST_TET, 4, EL, EC, P2, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_TET, 4, EH, P4, EI, N0,
    /* case 53 */ 9,
    ST_PNT, 5, EH, EF, ED, EC, EL,
    ST_WDG, 6, P0, P5, P2, EA, EJ, EB,
    ST_TET, 4, P5, P2, P0, N0,
    ST_PYR, 5, P0, P2, EC, ED, N0,
    ST_TET, 4, P4, P5, P0, N0,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_PYR, 5, P2, P5, EF, EL, N0,
    ST_TET, 4, EC, P2, EL, N0,
    /* case 54 */ 9,
    ST_PNT, 6, EA, EI, EH, EF, EL, EC,
    ST_TET, 4, P1, P4, P5, N0,
    ST_PYR, 5, EA, EI, P4, P1, N0,
    ST_PYR, 5, EA, P1, P2, EC, N0,
    ST_TET, 4, P2, P1, P5, N0,
    ST_TET, 4, P2, EL, EC, N0,
    ST_PYR, 5, P2, P5, EF, EL, N0,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_TET, 4, EH, P4, EI, N0,
    /* case 55 */ 9,
    ST_PNT, 5, EF, EH, ED, EL, EC,
    ST_PYR, 5, P1, P0, P4, P5, N0,
    ST_TET, 4, P0, P1, P2, N0,
    ST_TET, 4, P2, P1, P5, N0,
    ST_PYR, 5, P2, P5, EF, EL, N0,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_PYR, 5, ED, P0, P2, EC, N0,
    ST_TET, 4, EL, EC, P2, N0,
    /* case 56 */ 8,
    ST_PNT, 4, EF, EJ, EC, EC,
    ST_PYR, 5, P3, P4, EI, ED, N0,
    ST_PYR, 5, EH, P4, P3, EK, N0,
    ST_TET, 4, P3, ED, EC, N0,
    ST_TET, 4, EK, P3, EC, N0,
    ST_PYR, 5, EI, P4, P5, EJ, N0,
    ST_PYR, 5, P4, EH, EF, P5, N0,
    ST_TET, 4, EF, EJ, P5, N0,
    /* case 57 */ 9,
    ST_PNT, 6, EA, EJ, EF, EH, EK, EC,
    ST_TET, 4, P0, P4, P5, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_PYR, 5, EA, EC, P3, P0, N0,
    ST_TET, 4, P3, P4, P0, N0,
    ST_TET, 4, P3, EC, EK, N0,
    ST_PYR, 5, P3, EK, EH, P4, N0,
    ST_PYR, 5, EH, EF, P5, P4, N0,
    ST_TET, 4, EF, EJ, P5, N0,
    /* case 58 */ 9,
    ST_PNT, 5, EF, EB, EH, EK, EC,
    ST_WDG, 6, P4, P1, P3, EI, EA, ED,
    ST_TET, 4, P1, P3, P4, N0,
    ST_PYR, 5, P4, P3, EK, EH, N0,
    ST_TET, 4, P5, P1, P4, N0,
    ST_PYR, 5, EB, P1, P5, EF, N0,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_PYR, 5, P3, P1, EB, EC, N0,
    ST_TET, 4, EK, P3, EC, N0,
    /* case 59 */ 9,
    ST_PNT, 5, EB, EF, EH, EC, EK,
    ST_PYR, 5, P0, P4, P5, P1, N0,
    ST_TET, 4, P4, P0, P3, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_PYR, 5, P3, P1, EB, EC, N0,
    ST_PYR, 5, EB, P1, P5, EF, N0,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_PYR, 5, EH, P4, P3, EK, N0,
    ST_TET, 4, EC, EK, P3, N0,
    /* case 60 */ 2,
    ST_HEX, 8, P3, P4, P5, P2, EK, EH, EF, EL,
    ST_HEX, 8, ED, EI, EJ, EB, P3, P4, P5, P2,
    /* case 61 */ 11,
    ST_PNT, 6, P0, P2, P3, P4, EF, EH,
    ST_WDG, 6, P0, P5, P2, EA, EJ, EB,
    ST_PYR, 5, EH, EF, P5, P4, N0,
    ST_TET, 4, P4, P5, P0, N0,
    ST_TET, 4, P4, P0, P3, N0,
    ST_PYR, 5, EK, EH, P4, P3, N0,
    ST_PYR, 5, EL, EK, P3, P2, N0,
    ST_TET, 4, P3, P0, P2, N0,
    ST_PYR, 5, EF, EH, EK, EL, N0,
    ST_TET, 4, P2, P0, P5, N0,
    ST_PYR, 5, EF, EL, P2, P5, N0,
    /* case 62 */ 11,
    ST_PNT, 6, P1, P3, P2, P5, EH, EF,
    ST_WDG, 6, EA, EI, ED, P1, P4, P3,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_TET, 4, P5, P1, P4, N0,
    ST_TET, 4, P5, P2, P1, N0,
    ST_PYR, 5, EL, P2, P5, EF, N0,
    ST_PYR, 5, EK, P3, P2, EL, N0,
    ST_TET, 4, P2, P3, P1, N0,
    ST_PYR, 5, EH, EK, EL, EF, N0,
    ST_TET, 4, P3, P4, P1, N0,
    ST_PYR, 5, EH, P4, P3, EK, N0,
    /* case 63 */ 2,
    ST_HEX, 8, P3, P4, P5, P2, EK, EH, EF, EL,
    ST_WDG, 6, P1, P2, P5, P0, P3, P4,
    /* case 64 */ 1,
    ST_TET, 4, P6, EF, EG, EL,
    /* case 65 */ 2,
    ST_TET, 4, P0, EI, EA, ED,
    ST_TET, 4, P6, EF, EG, EL,
    /* case 66 */ 7,
    ST_PNT, 2, EA, EG,
    ST_PYR, 5, P1, P6, EL, EB, N0,
    ST_PYR, 5, EJ, EF, P6, P1, N0,
    ST_TET, 4, EF, EG, P6, N0,
    ST_TET, 4, P6, EG, EL, N0,
    ST_TET, 4, EJ, N0, P1, EA,
    ST_TET, 4, EB, EA, P1, N0,
    /* case 67 */ 8,
    ST_PNT, 4, EI, ED, EG, EG,
    ST_PYR, 5, P6, EL, EB, P1, N0,
    ST_PYR, 5, EJ, EF, P6, P1, N0,
    ST_TET, 4, P6, EG, EL, N0,
    ST_TET, 4, EF, EG, P6, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_PYR, 5, P1, P0, EI, EJ, N0,
    ST_TET, 4, EI, P0, ED, N0,
    /* case 68 */ 1,
    ST_WDG, 6, P2, EB, EC, P6, EF, EG,
    /* case 69 */ 8,
    ST_PNT, 4, EG, EF, EI, EI,
    ST_PYR, 5, P0, EA, EB, P2, N0,
    ST_PYR, 5, EC, ED, P0, P2, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_TET, 4, ED, EI, P0, N0,
    ST_PYR, 5, EB, EF, P6, P2, N0,
    ST_PYR, 5, P2, P6, EG, EC, N0,
    ST_TET, 4, EG, P6, EF, N0,
    /* case 70 */ 7,
    ST_PNT, 5, EA, EC, EJ, EF, EG,
    ST_PYR, 5, EC, P2, P6, EG, N0,
    ST_PYR, 5, EA, P1, P2, EC, N0,
    ST_TET, 4, EJ, P1, EA, N0,
    ST_TET, 4, P1, P6, P2, N0,
    ST_PYR, 5, EJ, EF, P6, P1, N0,
    ST_TET, 4, P6, EF, EG, N0,
    /* case 71 */ 9,
    ST_PNT, 6, EJ, EF, EG, EC, ED, EI,
    ST_TET, 4, P1, P6, P2, N0,
    ST_PYR, 5, EJ, EF, P6, P1, N0,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_TET, 4, P0, P1, P2, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_PYR, 5, P0, P2, EC, ED, N0,
    ST_PYR, 5, EC, P2, P6, EG, N0,
    ST_TET, 4, EG, P6, EF, N0,
    /* case 72 */ 7,
    ST_PNT, 2, ED, EF,
    ST_PYR, 5, P3, P6, EG, EK, N0,
    ST_PYR, 5, EC, EL, P6, P3, N0,
    ST_TET, 4, EL, EF, P6, N0,
    ST_TET, 4, P6, EF, EG, N0,
    ST_TET, 4, EC, N0, P3, ED,
    ST_TET, 4, EK, ED, P3, N0,
    /* case 73 */ 8,
    ST_PNT, 4, EA, EI, EF, EF,
    ST_PYR, 5, P6, EG, EK, P3, N0,
    ST_PYR, 5, EC, EL, P6, P3, N0,
    ST_TET, 4, P6, EF, EG, N0,
    ST_TET, 4, EL, EF, P6, N0,
    ST_PYR, 5, EK, EI, P0, P3, N0,
    ST_PYR, 5, P3, P0, EA, EC, N0,
    ST_TET, 4, EA, P0, EI, N0,
    /* case 74 */ 5,
    ST_WDG, 6, P1, P6, P3, EB, EL, EC,
    ST_PYR, 5, P1, P3, ED, EA, EJ,
    ST_PYR, 5, EK, P3, P6, EG, EF,
    ST_PYR, 5, EJ, EF, P6, P1, P3,
    ST_PYR, 5, EJ, ED, EK, EF, P3,
    /* case 75 */ 9,
    ST_PNT, 5, EI, EK, EJ, EF, EG,
    ST_WDG, 6, EB, EC, EL, P1, P3, P6,
    ST_TET, 4, P3, P1, P6, N0,
    ST_PYR, 5, P1, EJ, EF, P6, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_PYR, 5, EK, EI, P0, P3, N0,
    ST_PYR, 5, EI, EJ, P1, P0, N0,
    ST_PYR, 5, P6, EG, EK, P3, N0,
    ST_TET, 4, EF, EG, P6, N0,
    /* case 76 */ 7,
    ST_PNT, 5, ED, EB, EK, EG, EF,
    ST_PYR, 5, EB, EF, P6, P2, N0,
    ST_PYR, 5, ED, EB, P2, P3, N0,
    ST_TET, 4, EK, ED, P3, N0,
    ST_TET, 4, P3, P2, P6, N0,
    ST_PYR, 5, EK, P3, P6, EG, N0,
    ST_TET, 4, P6, EF, EG, N0,
    /* case 77 */ 9,
    ST_PNT, 6, EK, EG, EF, EB, EA, EI,
    ST_TET, 4, P3, P2, P6, N0,
    ST_PYR, 5, EK, P3, P6, EG, N0,
    ST_PYR, 5, EK, EI, P0, P3, N0,
    ST_TET, 4, P0, P2, P3, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_PYR, 5, P0, EA, EB, P2, N0,
    ST_PYR, 5, EB, EF, P6, P2, N0,
    ST_TET, 4, EF, EG, P6, N0,
    /* case 78 */ 5,
    ST_TET, 4, P2, P1, P6, P3,
    ST_PYR, 5, ED, EA, P1, P3, EJ,
    ST_PYR, 5, EK, P3, P6, EG, EF,
    ST_PYR, 5, P1, EJ, EF, P6, P3,
    ST_PYR, 5, ED, EK, EF, EJ, P3,
    /* case 79 */ 9,
    ST_PNT, 5, EK, EI, EJ, EG, EF,
    ST_PYR, 5, P2, P3, P0, P1, N0,
    ST_TET, 4, P1, P6, P2, N0,
    ST_TET, 4, P6, P3, P2, N0,
    ST_PYR, 5, P6, EG, EK, P3, N0,
    ST_PYR, 5, EK, EI, P0, P3, N0,
    ST_PYR, 5, EI, EJ, P1, P0, N0,
    ST_PYR, 5, EJ, EF, P6, P1, N0,
    ST_TET, 4, EG, P6, EF, N0,
    /* case 80 */ 7,
    ST_PNT, 2, EI, EL,
    ST_PYR, 5, P4, EH, EG, P6, N0,
    ST_PYR, 5, EE, P4, P6, EF, N0,
    ST_TET, 4, EF, P6, EL, N0,
    ST_TET, 4, P6, EG, EL, N0,
    ST_TET, 4, EE, P4, N0, EI,
    ST_TET, 4, EH, P4, EI, N0,
    /* case 81 */ 8,
    ST_PNT, 4, EA, ED, EL, EL,
    ST_PYR, 5, P6, P4, EH, EG, N0,
    ST_PYR, 5, EE, P4, P6, EF, N0,
    ST_TET, 4, P6, EG, EL, N0,
    ST_TET, 4, EF, P6, EL, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_PYR, 5, P4, EE, EA, P0, N0,
    ST_TET, 4, EA, ED, P0, N0,
    /* case 82 */ 5,
    ST_WDG, 6, EJ, EF, EE, P1, P6, P4,
    ST_PYR, 5, P1, EA, EI, P4, EB,
    ST_PYR, 5, EH, EG, P6, P4, EL,
    ST_PYR, 5, EB, P1, P6, EL, P4,
    ST_PYR, 5, EB, EL, EH, EI, P4,
    /* case 83 */ 9,
    ST_PNT, 5, ED, EH, EB, EL, EG,
    ST_WDG, 6, P1, P4, P6, EJ, EE, EF,
    ST_TET, 4, P4, P6, P1, N0,
    ST_PYR, 5, P1, P6, EL, EB, N0,
    ST_TET, 4, P0, P4, P1, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_PYR, 5, P6, P4, EH, EG, N0,
    ST_TET, 4, EL, P6, EG, N0,
    /* case 84 */ 8,
    ST_PNT, 4, EC, EB, EI, EI,
    ST_PYR, 5, P4, P6, EF, EE, N0,
    ST_PYR, 5, EG, P6, P4, EH, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_TET, 4, EH, P4, EI, N0,
    ST_PYR, 5, EF, P6, P2, EB, N0,
    ST_PYR, 5, P6, EG, EC, P2, N0,
    ST_TET, 4, EC, EB, P2, N0,
    /* case 85 */ 2,
    ST_HEX, 8, P0, P2, P6, P4, EA, EB, EF, EE,
    ST_HEX, 8, ED, EC, EG, EH, P0, P2, P6, P4,
    /* case 86 */ 9,
    ST_PNT, 5, EC, EG, EA, EI, EH,
    ST_WDG, 6, EJ, EF, EE, P1, P6, P4,
    ST_TET, 4, P6, P1, P4, N0,
    ST_PYR, 5, P1, EA, EI, P4, N0,
    ST_TET, 4, P2, P1, P6, N0,
    ST_PYR, 5, EG, EC, P2, P6, N0,
    ST_PYR, 5, EC, EA, P1, P2, N0,
    ST_PYR, 5, P4, EH, EG, P6, N0,
    ST_TET, 4, EI, EH, P4, N0,
    /* case 87 */ 11,
    ST_PNT, 6, P1, P4, P0, P2, EG, EC,
    ST_WDG, 6, EJ, EF, EE, P1, P6, P4,
    ST_PYR, 5, EC, P2, P6, EG, N0,
    ST_TET, 4, P2, P1, P6, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_PYR, 5, ED, P0, P2, EC, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_TET, 4, P0, P4, P1, N0,
    ST_PYR, 5, EG, EH, ED, EC, N0,
    ST_TET, 4, P4, P6, P1, N0,
    ST_PYR, 5, EG, P6, P4, EH, N0,
    /* case 88 */ 5,
    ST_WDG, 6, EH, EG, EK, P4, P6, P3,
    ST_PYR, 5, P4, EI, ED, P3, EE,
    ST_PYR, 5, EC, EL, P6, P3, EF,
    ST_PYR, 5, EE, P4, P6, EF, P3,
    ST_PYR, 5, EE, EF, EC, ED, P3,
    /* case 89 */ 9,
    ST_PNT, 5, EA, EC, EE, EF, EL,
    ST_WDG, 6, P4, P3, P6, EH, EK, EG,
    ST_TET, 4, P3, P6, P4, N0,
    ST_PYR, 5, P4, P6, EF, EE, N0,
    ST_TET, 4, P0, P3, P4, N0,
    ST_PYR, 5, EC, P3, P0, EA, N0,
    ST_PYR, 5, EA, P0, P4, EE, N0,
    ST_PYR, 5, P6, P3, EC, EL, N0,
    ST_TET, 4, EF, P6, EL, N0,
    /* case 90 */ 5,
    ST_WDG, 6, EB, EC, EL, P1, P3, P6,
    ST_TET, 4, P1, P6, P3, P4,
    ST_WDG, 6, P4, P6, P1, EE, EF, EJ,
    ST_WDG, 6, P3, P4, P1, ED, EI, EA,
    ST_WDG, 6, P6, P4, P3, EG, EH, EK,
    /* case 91 */ 5,
    ST_WDG, 6, P6, P3, P1, EL, EC, EB,
    ST_WDG, 6, EH, EG, EK, P4, P6, P3,
    ST_WDG, 6, P4, P6, P1, EE, EF, EJ,
    ST_TET, 4, P6, P3, P1, P4,
    ST_TET, 4, P4, P1, P0, P3,
    /* case 92 */ 9,
    ST_PNT, 5, EB, EF, ED, EI, EE,
    ST_WDG, 6, P3, P6, P4, EK, EG, EH,
    ST_TET, 4, P6, P4, P3, N0,
    ST_PYR, 5, P3, P4, EI, ED, N0,
    ST_TET, 4, P2, P6, P3, N0,
    ST_PYR, 5, EF, P6, P2, EB, N0,
    ST_PYR, 5, EB, P2, P3, ED, N0,
    ST_PYR, 5, P4, P6, EF, EE, N0,
    ST_TET, 4, EI, P4, EE, N0,
    /* case 93 */ 11,
    ST_PNT, 6, P3, P4, P0, P2, EF, EB,
    ST_WDG, 6, P3, P6, P4, EK, EG, EH,
    ST_PYR, 5, EB, EF, P6, P2, N0,
    ST_TET, 4, P2, P6, P3, N0,
    ST_TET, 4, P2, P3, P0, N0,
    ST_PYR, 5, EA, EB, P2, P0, N0,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_TET, 4, P0, P3, P4, N0,
    ST_PYR, 5, EF, EB, EA, EE, N0,
    ST_TET, 4, P4, P3, P6, N0,
    ST_PYR, 5, EF, EE, P4, P6, N0,
    /* case 94 */ 5,
    ST_WDG, 6, EE, EJ, EF, P4, P1, P6,
    ST_WDG, 6, P3, P4, P1, ED, EI, EA,
    ST_WDG, 6, EK, EH, EG, P3, P4, P6,
    ST_TET, 4, P4, P6, P1, P3,
    ST_TET, 4, P3, P2, P6, P1,
    /* case 95 */ 5,
    ST_WDG, 6, P4, P6, P1, EE, EF, EJ,
    ST_WDG, 6, EH, EG, EK, P4, P6, P3,
    ST_PYR, 5, P0, P1, P2, P3, P4,
    ST_TET, 4, P6, P3, P2, P4,
    ST_TET, 4, P6, P2, P1, P4,
    /* case 96 */ 1,
    ST_WDG, 6, P6, EL, EG, P5, EJ, EE,
    /* case 97 */ 8,
    ST_PNT, 4, EG, EL, ED, ED,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_PYR, 5, EE, P5, P0, EI, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_TET, 4, EI, P0, ED, N0,
    ST_PYR, 5, EJ, P5, P6, EL, N0,
    ST_PYR, 5, P5, EE, EG, P6, N0,
    ST_TET, 4, EG, EL, P6, N0,
    /* case 98 */ 7,
    ST_PNT, 5, EA, EE, EB, EL, EG,
    ST_PYR, 5, EE, EG, P6, P5, N0,
    ST_PYR, 5, EA, EE, P5, P1, N0,
    ST_TET, 4, EB, EA, P1, N0,
    ST_TET, 4, P1, P5, P6, N0,
    ST_PYR, 5, EB, P1, P6, EL, N0,
    ST_TET, 4, P6, EG, EL, N0,
    /* case 99 */ 9,
    ST_PNT, 6, EB, EL, EG, EE, EI, ED,
    ST_TET, 4, P1, P5, P6, N0,
    ST_PYR, 5, EB, P1, P6, EL, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_PYR, 5, EE, EG, P6, P5, N0,
    ST_TET, 4, EG, EL, P6, N0,
    /* case 100 */ 7,
    ST_PNT, 5, EC, EG, EB, EJ, EE,
    ST_PYR, 5, EG, P6, P5, EE, N0,
    ST_PYR, 5, EC, P2, P6, EG, N0,
    ST_TET, 4, EB, P2, EC, N0,
    ST_TET, 4, P2, P5, P6, N0,
    ST_PYR, 5, EB, EJ, P5, P2, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    /* case 101 */ 9,
    ST_PNT, 5, EG, EE, EC, ED, EI,
    ST_WDG, 6, EB, EJ, EA, P2, P5, P0,
    ST_TET, 4, P5, P2, P0, N0,
    ST_PYR, 5, P2, EC, ED, P0, N0,
    ST_TET, 4, P6, P2, P5, N0,
    ST_PYR, 5, EE, EG, P6, P5, N0,
    ST_PYR, 5, EG, EC, P2, P6, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_TET, 4, ED, EI, P0, N0,
    /* case 102 */ 1,
    ST_HEX, 8, P1, P5, P6, P2, EA, EE, EG, EC,
    /* case 103 */ 9,
    ST_PNT, 5, EE, EG, EC, EI, ED,
    ST_PYR, 5, P1, P5, P6, P2, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_PYR, 5, EE, EG, P6, P5, N0,
    ST_PYR, 5, EG, EC, P2, P6, N0,
    ST_PYR, 5, EC, ED, P0, P2, N0,
    ST_TET, 4, EI, P0, ED, N0,
    /* case 104 */ 8,
    ST_PNT, 4, EE, EJ, ED, ED,
    ST_PYR, 5, P3, EC, EL, P6, N0,
    ST_PYR, 5, EG, EK, P3, P6, N0,
    ST_TET, 4, P3, ED, EC, N0,
    ST_TET, 4, EK, ED, P3, N0,
    ST_PYR, 5, EL, EJ, P5, P6, N0,
    ST_PYR, 5, P6, P5, EE, EG, N0,
    ST_TET, 4, EE, P5, EJ, N0,
    /* case 105 */ 2,
    ST_HEX, 8, EA, EJ, EL, EC, P0, P5, P6, P3,
    ST_HEX, 8, P0, P5, P6, P3, EI, EE, EG, EK,
    /* case 106 */ 9,
    ST_PNT, 5, EE, EG, EA, ED, EK,
    ST_WDG, 6, P1, P6, P3, EB, EL, EC,
    ST_TET, 4, P6, P3, P1, N0,
    ST_PYR, 5, P1, P3, ED, EA, N0,
    ST_TET, 4, P5, P6, P1, N0,
    ST_PYR, 5, EG, P6, P5, EE, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_PYR, 5, P3, P6, EG, EK, N0,
    ST_TET, 4, ED, P3, EK, N0,
    /* case 107 */ 11,
    ST_PNT, 6, P1, P3, P0, P5, EG, EE,
    ST_WDG, 6, P1, P6, P3, EB, EL, EC,
    ST_PYR, 5, EE, EG, P6, P5, N0,
    ST_TET, 4, P5, P6, P1, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_PYR, 5, EI, EE, P5, P0, N0,
    ST_PYR, 5, EK, EI, P0, P3, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_PYR, 5, EG, EE, EI, EK, N0,
    ST_TET, 4, P3, P1, P6, N0,
    ST_PYR, 5, EG, EK, P3, P6, N0,
    /* case 108 */ 9,
    ST_PNT, 6, EB, EJ, EE, EG, EK, ED,
    ST_TET, 4, P2, P5, P6, N0,
    ST_PYR, 5, EB, EJ, P5, P2, N0,
    ST_PYR, 5, EB, P2, P3, ED, N0,
    ST_TET, 4, P3, P2, P6, N0,
    ST_TET, 4, P3, EK, ED, N0,
    ST_PYR, 5, P3, P6, EG, EK, N0,
    ST_PYR, 5, EG, P6, P5, EE, N0,
    ST_TET, 4, EE, P5, EJ, N0,
    /* case 109 */ 11,
    ST_PNT, 6, P2, P0, P3, P6, EE, EG,
    ST_WDG, 6, EB, EJ, EA, P2, P5, P0,
    ST_PYR, 5, EG, P6, P5, EE, N0,
    ST_TET, 4, P6, P2, P5, N0,
    ST_TET, 4, P6, P3, P2, N0,
    ST_PYR, 5, EK, P3, P6, EG, N0,
    ST_PYR, 5, EI, P0, P3, EK, N0,
    ST_TET, 4, P3, P0, P2, N0,
    ST_PYR, 5, EE, EI, EK, EG, N0,
    ST_TET, 4, P0, P5, P2, N0,
    ST_PYR, 5, EE, P5, P0, EI, N0,
    /* case 110 */ 9,
    ST_PNT, 5, EG, EE, EA, EK, ED,
    ST_PYR, 5, P2, P1, P5, P6, N0,
    ST_TET, 4, P1, P2, P3, N0,
    ST_TET, 4, P3, P2, P6, N0,
    ST_PYR, 5, P3, P6, EG, EK, N0,
    ST_PYR, 5, EG, P6, P5, EE, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_PYR, 5, EA, P1, P3, ED, N0,
    ST_TET, 4, EK, ED, P3, N0,
    /* case 111 */ 2,
    ST_HEX, 8, P0, P5, P6, P3, EI, EE, EG, EK,
    ST_WDG, 6, P2, P3, P6, P1, P0, P5,
    /* case 112 */ 7,
    ST_PNT, 5, EI, EJ, EH, EG, EL,
    ST_PYR, 5, EJ, P5, P6, EL, N0,
    ST_PYR, 5, EI, P4, P5, EJ, N0,
    ST_TET, 4, EH, P4, EI, N0,
    ST_TET, 4, P4, P6, P5, N0,
    ST_PYR, 5, EH, EG, P6, P4, N0,
    ST_TET, 4, P6, EG, EL, N0,
    /* case 113 */ 9,
    ST_PNT, 6, EH, EG, EL, EJ, EA, ED,
    ST_TET, 4, P4, P6, P5, N0,
    ST_PYR, 5, EH, EG, P6, P4, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_PYR, 5, EJ, P5, P6, EL, N0,
    ST_TET, 4, EL, P6, EG, N0,
    /* case 114 */ 5,
    ST_TET, 4, P5, P6, P1, P4,
    ST_PYR, 5, EI, P4, P1, EA, EB,
    ST_PYR, 5, EH, EG, P6, P4, EL,
    ST_PYR, 5, P1, P6, EL, EB, P4,
    ST_PYR, 5, EI, EB, EL, EH, P4,
    /* case 115 */ 9,
    ST_PNT, 5, EH, ED, EB, EG, EL,
    ST_PYR, 5, P5, P1, P0, P4, N0,
    ST_TET, 4, P1, P5, P6, N0,
    ST_TET, 4, P6, P5, P4, N0,
    ST_PYR, 5, P6, P4, EH, EG, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_PYR, 5, EB, P1, P6, EL, N0,
    ST_TET, 4, EG, EL, P6, N0,
    /* case 116 */ 9,
    ST_PNT, 6, EJ, EB, EC, EG, EH, EI,
    ST_TET, 4, P5, P6, P2, N0,
    ST_PYR, 5, EJ, P5, P2, EB, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_TET, 4, P4, P6, P5, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_PYR, 5, P4, EH, EG, P6, N0,
    ST_PYR, 5, EG, EC, P2, P6, N0,
    ST_TET, 4, EC, EB, P2, N0,
    /* case 117 */ 11,
    ST_PNT, 6, P5, P0, P4, P6, EC, EG,
    ST_WDG, 6, P5, P2, P0, EJ, EB, EA,
    ST_PYR, 5, EG, EC, P2, P6, N0,
    ST_TET, 4, P6, P2, P5, N0,
    ST_TET, 4, P6, P5, P4, N0,
    ST_PYR, 5, EH, EG, P6, P4, N0,
    ST_PYR, 5, ED, EH, P4, P0, N0,
    ST_TET, 4, P4, P5, P0, N0,
    ST_PYR, 5, EC, EG, EH, ED, N0,
    ST_TET, 4, P0, P5, P2, N0,
    ST_PYR, 5, EC, ED, P0, P2, N0,
    /* case 118 */ 9,
    ST_PNT, 5, EG, EC, EA, EH, EI,
    ST_PYR, 5, P5, P6, P2, P1, N0,
    ST_TET, 4, P1, P4, P5, N0,
    ST_TET, 4, P4, P6, P5, N0,
    ST_PYR, 5, P4, EH, EG, P6, N0,
    ST_PYR, 5, EG, EC, P2, P6, N0,
    ST_PYR, 5, EC, EA, P1, P2, N0,
    ST_PYR, 5, EA, EI, P4, P1, N0,
    ST_TET, 4, EH, P4, EI, N0,
    /* case 119 */ 2,
    ST_HEX, 8, ED, EC, EG, EH, P0, P2, P6, P4,
    ST_WDG, 6, P1, P0, P2, P5, P4, P6,
    /* case 120 */ 9,
    ST_PNT, 5, EJ, EL, EI, ED, EC,
    ST_WDG, 6, EH, EG, EK, P4, P6, P3,
    ST_TET, 4, P6, P4, P3, N0,
    ST_PYR, 5, P4, EI, ED, P3, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_PYR, 5, EL, EJ, P5, P6, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_PYR, 5, P3, EC, EL, P6, N0,
    ST_TET, 4, ED, EC, P3, N0,
    /* case 121 */ 11,
    ST_PNT, 6, P4, P3, P0, P5, EL, EJ,
    ST_WDG, 6, EH, EG, EK, P4, P6, P3,
    ST_PYR, 5, EJ, P5, P6, EL, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_PYR, 5, EC, P3, P0, EA, N0,
    ST_TET, 4, P0, P3, P4, N0,
    ST_PYR, 5, EL, EC, EA, EJ, N0,
    ST_TET, 4, P3, P6, P4, N0,
    ST_PYR, 5, EL, P6, P3, EC, N0,
    /* case 122 */ 5,
    ST_WDG, 6, P3, P1, P6, EC, EB, EL,
    ST_WDG, 6, EI, ED, EA, P4, P3, P1,
    ST_WDG, 6, P4, P3, P6, EH, EK, EG,
    ST_TET, 4, P3, P1, P6, P4,
    ST_TET, 4, P4, P6, P5, P1,
    /* case 123 */ 5,
    ST_WDG, 6, EC, EL, EB, P3, P6, P1,
    ST_WDG, 6, P3, P6, P4, EK, EG, EH,
    ST_PYR, 5, P0, P4, P5, P1, P3,
    ST_TET, 4, P6, P5, P4, P3,
    ST_TET, 4, P6, P1, P5, P3,
    /* case 124 */ 11,
    ST_PNT, 6, P6, P4, P5, P2, ED, EB,
    ST_WDG, 6, EG, EK, EH, P6, P3, P4,
    ST_PYR, 5, EB, P2, P3, ED, N0,
    ST_TET, 4, P2, P6, P3, N0,
    ST_TET, 4, P2, P5, P6, N0,
    ST_PYR, 5, EJ, P5, P2, EB, N0,
    ST_PYR, 5, EI, P4, P5, EJ, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_PYR, 5, ED, EI, EJ, EB, N0,
    ST_TET, 4, P4, P3, P6, N0,
    ST_PYR, 5, ED, P3, P4, EI, N0,
    /* case 125 */ 6,
    ST_WDG, 6, EH, EG, EK, P4, P6, P3,
    ST_WDG, 6, EJ, EA, EB, P5, P0, P2,
    ST_TET, 4, P2, P3, P0, P5,
    ST_TET, 4, P5, P4, P6, P3,
    ST_TET, 4, P4, P5, P0, P3,
    ST_TET, 4, P5, P6, P2, P3,
    /* case 126 */ 5,
    ST_WDG, 6, EK, EH, EG, P3, P4, P6,
    ST_WDG, 6, P3, P4, P1, ED, EI, EA,
    ST_PYR, 5, P2, P1, P5, P6, P3,
    ST_TET, 4, P4, P5, P1, P3,
    ST_TET, 4, P4, P6, P5, P3,
    /* case 127 */ 9,
    ST_PNT, 7, P0, P1, P2, P3, P4, P5, P6,
    ST_WDG, 6, EH, EG, EK, P4, P6, P3,
    ST_TET, 4, P4, P3, P6, N0,
    ST_PYR, 5, P5, P6, P2, P1, N0,
    ST_TET, 4, P6, P3, P2, N0,
    ST_PYR, 5, P0, P1, P2, P3, N0,
    ST_TET, 4, P0, P3, P4, N0,
    ST_PYR, 5, P0, P4, P5, P1, N0,
    ST_TET, 4, P4, P6, P5, N0,
    /* case 128 */ 1,
    ST_TET, 4, P7, EG, EH, EK,
    /* case 129 */ 7,
    ST_PNT, 2, EA, EG,
    ST_PYR, 5, P0, ED, EK, P7, N0,
    ST_PYR, 5, EI, P0, P7, EH, N0,
    ST_TET, 4, EH, P7, EG, N0,
    ST_TET, 4, P7, EK, EG, N0,
    ST_TET, 4, EI, P0, N0, EA,
    ST_TET, 4, ED, P0, EA, N0,
    /* case 130 */ 2,
    ST_TET, 4, P1, EA, EJ, EB,
    ST_TET, 4, P7, EH, EK, EG,
    /* case 131 */ 8,
    ST_PNT, 4, EJ, EB, EG, EG,
    ST_PYR, 5, P7, P0, ED, EK, N0,
    ST_PYR, 5, EI, P0, P7, EH, N0,
    ST_TET, 4, P7, EK, EG, N0,
    ST_TET, 4, EH, P7, EG, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_PYR, 5, P0, EI, EJ, P1, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    /* case 132 */ 7,
    ST_PNT, 2, EH, EB,
    ST_PYR, 5, P7, EK, EC, P2, N0,
    ST_PYR, 5, EG, P7, P2, EL, N0,
    ST_TET, 4, EL, P2, EB, N0,
    ST_TET, 4, P2, EC, EB, N0,
    ST_TET, 4, EG, P7, N0, EH,
    ST_TET, 4, EK, P7, EH, N0,
    /* case 133 */ 5,
    ST_WDG, 6, P0, P2, P7, ED, EC, EK,
    ST_PYR, 5, P0, P7, EH, EI, EA,
    ST_PYR, 5, EG, P7, P2, EL, EB,
    ST_PYR, 5, EA, EB, P2, P0, P7,
    ST_PYR, 5, EA, EH, EG, EB, P7,
    /* case 134 */ 8,
    ST_PNT, 4, EJ, EA, EH, EH,
    ST_PYR, 5, P7, EK, EC, P2, N0,
    ST_PYR, 5, EL, EG, P7, P2, N0,
    ST_TET, 4, P7, EH, EK, N0,
    ST_TET, 4, EG, EH, P7, N0,
    ST_PYR, 5, EC, EA, P1, P2, N0,
    ST_PYR, 5, P2, P1, EJ, EL, N0,
    ST_TET, 4, EJ, P1, EA, N0,
    /* case 135 */ 9,
    ST_PNT, 5, EJ, EL, EI, EH, EG,
    ST_WDG, 6, P0, P2, P7, ED, EC, EK,
    ST_TET, 4, P2, P7, P0, N0,
    ST_PYR, 5, P0, P7, EH, EI, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_PYR, 5, EL, P2, P1, EJ, N0,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_PYR, 5, P7, P2, EL, EG, N0,
    ST_TET, 4, EH, P7, EG, N0,
    /* case 136 */ 1,
    ST_WDG, 6, P7, EH, EG, P3, ED, EC,
    /* case 137 */ 7,
    ST_PNT, 5, EA, EC, EI, EH, EG,
    ST_PYR, 5, EC, EG, P7, P3, N0,
    ST_PYR, 5, EA, EC, P3, P0, N0,
    ST_TET, 4, EI, EA, P0, N0,
    ST_TET, 4, P0, P3, P7, N0,
    ST_PYR, 5, EI, P0, P7, EH, N0,
    ST_TET, 4, P7, EG, EH, N0,
    /* case 138 */ 8,
    ST_PNT, 4, EG, EH, EJ, EJ,
    ST_PYR, 5, P1, P3, ED, EA, N0,
    ST_PYR, 5, EC, P3, P1, EB, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_TET, 4, EB, P1, EJ, N0,
    ST_PYR, 5, ED, P3, P7, EH, N0,
    ST_PYR, 5, P3, EC, EG, P7, N0,
    ST_TET, 4, EG, EH, P7, N0,
    /* case 139 */ 9,
    ST_PNT, 6, EI, EH, EG, EC, EB, EJ,
    ST_TET, 4, P0, P3, P7, N0,
    ST_PYR, 5, EI, P0, P7, EH, N0,
    ST_PYR, 5, EI, EJ, P1, P0, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_PYR, 5, P1, EB, EC, P3, N0,
    ST_PYR, 5, EC, EG, P7, P3, N0,
    ST_TET, 4, EG, EH, P7, N0,
    /* case 140 */ 7,
    ST_PNT, 5, EB, ED, EL, EG, EH,
    ST_PYR, 5, ED, P3, P7, EH, N0,
    ST_PYR, 5, EB, P2, P3, ED, N0,
    ST_TET, 4, EL, P2, EB, N0,
    ST_TET, 4, P2, P7, P3, N0,
    ST_PYR, 5, EL, EG, P7, P2, N0,
    ST_TET, 4, P7, EG, EH, N0,
    /* case 141 */ 5,
    ST_TET, 4, P3, P0, P2, P7,
    ST_PYR, 5, EH, EI, P0, P7, EA,
    ST_PYR, 5, EG, P7, P2, EL, EB,
    ST_PYR, 5, P0, EA, EB, P2, P7,
    ST_PYR, 5, EH, EG, EB, EA, P7,
    /* case 142 */ 9,
    ST_PNT, 6, ED, EA, EJ, EL, EG, EH,
    ST_TET, 4, P3, P1, P2, N0,
    ST_PYR, 5, ED, EA, P1, P3, N0,
    ST_PYR, 5, ED, P3, P7, EH, N0,
    ST_TET, 4, P7, P3, P2, N0,
    ST_TET, 4, P7, EG, EH, N0,
    ST_PYR, 5, P7, P2, EL, EG, N0,
    ST_PYR, 5, EL, P2, P1, EJ, N0,
    ST_TET, 4, EJ, P1, EA, N0,
    /* case 143 */ 9,
    ST_PNT, 5, EL, EJ, EI, EG, EH,
    ST_PYR, 5, P3, P0, P1, P2, N0,
    ST_TET, 4, P0, P3, P7, N0,
    ST_TET, 4, P7, P3, P2, N0,
    ST_PYR, 5, P7, P2, EL, EG, N0,
    ST_PYR, 5, EL, P2, P1, EJ, N0,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_PYR, 5, EI, P0, P7, EH, N0,
    ST_TET, 4, EG, EH, P7, N0,
    /* case 144 */ 1,
    ST_WDG, 6, P4, EI, EE, P7, EK, EG,
    /* case 145 */ 7,
    ST_PNT, 5, EA, EE, ED, EK, EG,
    ST_PYR, 5, EE, P4, P7, EG, N0,
    ST_PYR, 5, EA, P0, P4, EE, N0,
    ST_TET, 4, ED, P0, EA, N0,
    ST_TET, 4, P0, P7, P4, N0,
    ST_PYR, 5, ED, EK, P7, P0, N0,
    ST_TET, 4, P7, EK, EG, N0,
    /* case 146 */ 8,
    ST_PNT, 4, EG, EK, EB, EB,
    ST_PYR, 5, P1, EA, EI, P4, N0,
    ST_PYR, 5, EE, EJ, P1, P4, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    ST_PYR, 5, EI, EK, P7, P4, N0,
    ST_PYR, 5, P4, P7, EG, EE, N0,
    ST_TET, 4, EG, P7, EK, N0,
    /* case 147 */ 9,
    ST_PNT, 6, ED, EK, EG, EE, EJ, EB,
    ST_TET, 4, P0, P7, P4, N0,
    ST_PYR, 5, ED, EK, P7, P0, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_PYR, 5, EE, P4, P7, EG, N0,
    ST_TET, 4, EG, P7, EK, N0,
    /* case 148 */ 8,
    ST_PNT, 4, EE, EI, EB, EB,
    ST_PYR, 5, P2, P7, EK, EC, N0,
    ST_PYR, 5, EG, P7, P2, EL, N0,
    ST_TET, 4, P2, EC, EB, N0,
    ST_TET, 4, EL, P2, EB, N0,
    ST_PYR, 5, EK, P7, P4, EI, N0,
    ST_PYR, 5, P7, EG, EE, P4, N0,
    ST_TET, 4, EE, EI, P4, N0,
    /* case 149 */ 9,
    ST_PNT, 5, EE, EG, EA, EB, EL,
    ST_WDG, 6, ED, EK, EC, P0, P7, P2,
    ST_TET, 4, P7, P0, P2, N0,
    ST_PYR, 5, P0, EA, EB, P2, N0,
    ST_TET, 4, P4, P0, P7, N0,
    ST_PYR, 5, EG, EE, P4, P7, N0,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_PYR, 5, P2, EL, EG, P7, N0,
    ST_TET, 4, EB, EL, P2, N0,
    /* case 150 */ 2,
    ST_HEX, 8, P4, P1, P2, P7, EE, EJ, EL, EG,
    ST_HEX, 8, EI, EA, EC, EK, P4, P1, P2, P7,
    /* case 151 */ 11,
    ST_PNT, 6, P0, P7, P4, P1, EL, EJ,
    ST_WDG, 6, P0, P2, P7, ED, EC, EK,
    ST_PYR, 5, EJ, EL, P2, P1, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_PYR, 5, EE, EJ, P1, P4, N0,
    ST_PYR, 5, EG, EE, P4, P7, N0,
    ST_TET, 4, P4, P0, P7, N0,
    ST_PYR, 5, EL, EJ, EE, EG, N0,
    ST_TET, 4, P7, P0, P2, N0,
    ST_PYR, 5, EL, EG, P7, P2, N0,
    /* case 152 */ 7,
    ST_PNT, 5, EC, EG, ED, EI, EE,
    ST_PYR, 5, EG, EE, P4, P7, N0,
    ST_PYR, 5, EC, EG, P7, P3, N0,
    ST_TET, 4, ED, EC, P3, N0,
    ST_TET, 4, P3, P7, P4, N0,
    ST_PYR, 5, ED, P3, P4, EI, N0,
    ST_TET, 4, P4, EE, EI, N0,
    /* case 153 */ 1,
    ST_HEX, 8, EA, EE, EG, EC, P0, P4, P7, P3,
    /* case 154 */ 9,
    ST_PNT, 5, EG, EE, EC, EB, EJ,
    ST_WDG, 6, P3, P4, P1, ED, EI, EA,
    ST_TET, 4, P4, P1, P3, N0,
    ST_PYR, 5, P3, P1, EB, EC, N0,
    ST_TET, 4, P7, P4, P3, N0,
    ST_PYR, 5, EE, P4, P7, EG, N0,
    ST_PYR, 5, EG, P7, P3, EC, N0,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_TET, 4, EB, P1, EJ, N0,
    /* case 155 */ 9,
    ST_PNT, 5, EE, EG, EC, EJ, EB,
    ST_PYR, 5, P0, P3, P7, P4, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_PYR, 5, EE, P4, P7, EG, N0,
    ST_PYR, 5, EG, P7, P3, EC, N0,
    ST_PYR, 5, EC, P3, P1, EB, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    /* case 156 */ 9,
    ST_PNT, 6, ED, EI, EE, EG, EL, EB,
    ST_TET, 4, P3, P7, P4, N0,
    ST_PYR, 5, ED, P3, P4, EI, N0,
    ST_PYR, 5, ED, EB, P2, P3, N0,
    ST_TET, 4, P2, P7, P3, N0,
    ST_TET, 4, P2, EB, EL, N0,
    ST_PYR, 5, P2, EL, EG, P7, N0,
    ST_PYR, 5, EG, EE, P4, P7, N0,
    ST_TET, 4, EE, EI, P4, N0,
    /* case 157 */ 9,
    ST_PNT, 5, EG, EE, EA, EL, EB,
    ST_PYR, 5, P3, P7, P4, P0, N0,
    ST_TET, 4, P0, P2, P3, N0,
    ST_TET, 4, P2, P7, P3, N0,
    ST_PYR, 5, P2, EL, EG, P7, N0,
    ST_PYR, 5, EG, EE, P4, P7, N0,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_PYR, 5, EA, EB, P2, P0, N0,
    ST_TET, 4, EL, P2, EB, N0,
    /* case 158 */ 11,
    ST_PNT, 6, P3, P4, P7, P2, EJ, EL,
    ST_WDG, 6, ED, EA, EI, P3, P1, P4,
    ST_PYR, 5, EL, P2, P1, EJ, N0,
    ST_TET, 4, P2, P3, P1, N0,
    ST_TET, 4, P2, P7, P3, N0,
    ST_PYR, 5, EG, P7, P2, EL, N0,
    ST_PYR, 5, EE, P4, P7, EG, N0,
    ST_TET, 4, P7, P4, P3, N0,
    ST_PYR, 5, EJ, EE, EG, EL, N0,
    ST_TET, 4, P4, P1, P3, N0,
    ST_PYR, 5, EJ, P1, P4, EE, N0,
    /* case 159 */ 2,
    ST_HEX, 8, P4, P1, P2, P7, EE, EJ, EL, EG,
    ST_WDG, 6, P3, P7, P2, P0, P4, P1,
    /* case 160 */ 7,
    ST_PNT, 2, EK, EJ,
    ST_PYR, 5, P7, P5, EE, EH, N0,
    ST_PYR, 5, EG, EF, P5, P7, N0,
    ST_TET, 4, EF, EJ, P5, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_TET, 4, EG, N0, P7, EK,
    ST_TET, 4, EH, EK, P7, N0,
    /* case 161 */ 5,
    ST_WDG, 6, EI, EE, EH, P0, P5, P7,
    ST_PYR, 5, P0, ED, EK, P7, EA,
    ST_PYR, 5, EG, EF, P5, P7, EJ,
    ST_PYR, 5, EA, P0, P5, EJ, P7,
    ST_PYR, 5, EA, EJ, EG, EK, P7,
    /* case 162 */ 8,
    ST_PNT, 4, EB, EA, EK, EK,
    ST_PYR, 5, P7, P5, EE, EH, N0,
    ST_PYR, 5, EF, P5, P7, EG, N0,
    ST_TET, 4, P7, EH, EK, N0,
    ST_TET, 4, EG, P7, EK, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_PYR, 5, P5, EF, EB, P1, N0,
    ST_TET, 4, EB, EA, P1, N0,
    /* case 163 */ 9,
    ST_PNT, 5, EB, EF, ED, EK, EG,
    ST_WDG, 6, EI, EE, EH, P0, P5, P7,
    ST_TET, 4, P5, P0, P7, N0,
    ST_PYR, 5, P0, ED, EK, P7, N0,
    ST_TET, 4, P1, P0, P5, N0,
    ST_PYR, 5, EF, EB, P1, P5, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_PYR, 5, P7, EG, EF, P5, N0,
    ST_TET, 4, EK, EG, P7, N0,
    /* case 164 */ 5,
    ST_WDG, 6, P5, P7, P2, EF, EG, EL,
    ST_PYR, 5, P5, P2, EB, EJ, EE,
    ST_PYR, 5, EC, P2, P7, EK, EH,
    ST_PYR, 5, EE, EH, P7, P5, P2,
    ST_PYR, 5, EE, EB, EC, EH, P2,
    /* case 165 */ 5,
    ST_WDG, 6, P2, P0, P5, EB, EA, EJ,
    ST_TET, 4, P2, P0, P5, P7,
    ST_WDG, 6, EG, EF, EL, P7, P5, P2,
    ST_WDG, 6, ED, EK, EC, P0, P7, P2,
    ST_WDG, 6, EE, EH, EI, P5, P7, P0,
    /* case 166 */ 9,
    ST_PNT, 5, EA, EC, EE, EH, EK,
    ST_WDG, 6, EF, EL, EG, P5, P2, P7,
    ST_TET, 4, P2, P5, P7, N0,
    ST_PYR, 5, P5, EE, EH, P7, N0,
    ST_TET, 4, P1, P5, P2, N0,
    ST_PYR, 5, EC, EA, P1, P2, N0,
    ST_PYR, 5, EA, EE, P5, P1, N0,
    ST_PYR, 5, P7, EK, EC, P2, N0,
    ST_TET, 4, EH, EK, P7, N0,
    /* case 167 */ 5,
    ST_WDG, 6, EH, EI, EE, P7, P0, P5,
    ST_WDG, 6, P2, P7, P0, EC, EK, ED,
    ST_WDG, 6, EL, EG, EF, P2, P7, P5,
    ST_TET, 4, P7, P5, P0, P2,
    ST_TET, 4, P2, P1, P5, P0,
    /* case 168 */ 8,
    ST_PNT, 4, EC, ED, EJ, EJ,
    ST_PYR, 5, P5, EE, EH, P7, N0,
    ST_PYR, 5, EG, EF, P5, P7, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_TET, 4, EF, EJ, P5, N0,
    ST_PYR, 5, EH, ED, P3, P7, N0,
    ST_PYR, 5, P7, P3, EC, EG, N0,
    ST_TET, 4, EC, P3, ED, N0,
    /* case 169 */ 9,
    ST_PNT, 5, EC, EG, EA, EJ, EF,
    ST_WDG, 6, P0, P7, P5, EI, EH, EE,
    ST_TET, 4, P7, P5, P0, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_TET, 4, P3, P7, P0, N0,
    ST_PYR, 5, EG, P7, P3, EC, N0,
    ST_PYR, 5, EC, P3, P0, EA, N0,
    ST_PYR, 5, P5, P7, EG, EF, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    /* case 170 */ 2,
    ST_HEX, 8, EC, EB, EF, EG, P3, P1, P5, P7,
    ST_HEX, 8, P3, P1, P5, P7, ED, EA, EE, EH,
    /* case 171 */ 11,
    ST_PNT, 6, P0, P7, P3, P1, EF, EB,
    ST_WDG, 6, EI, EE, EH, P0, P5, P7,
    ST_PYR, 5, EB, P1, P5, EF, N0,
    ST_TET, 4, P1, P0, P5, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_PYR, 5, EC, P3, P1, EB, N0,
    ST_PYR, 5, EG, P7, P3, EC, N0,
    ST_TET, 4, P3, P7, P0, N0,
    ST_PYR, 5, EF, EG, EC, EB, N0,
    ST_TET, 4, P7, P5, P0, N0,
    ST_PYR, 5, EF, P5, P7, EG, N0,
    /* case 172 */ 9,
    ST_PNT, 5, ED, EH, EB, EJ, EE,
    ST_WDG, 6, EL, EG, EF, P2, P7, P5,
    ST_TET, 4, P7, P2, P5, N0,
    ST_PYR, 5, P2, EB, EJ, P5, N0,
    ST_TET, 4, P3, P2, P7, N0,
    ST_PYR, 5, EH, ED, P3, P7, N0,
    ST_PYR, 5, ED, EB, P2, P3, N0,
    ST_PYR, 5, P5, EE, EH, P7, N0,
    ST_TET, 4, EJ, EE, P5, N0,
    /* case 173 */ 5,
    ST_WDG, 6, EJ, EA, EB, P5, P0, P2,
    ST_WDG, 6, P7, P5, P0, EH, EE, EI,
    ST_WDG, 6, EG, EF, EL, P7, P5, P2,
    ST_TET, 4, P5, P2, P0, P7,
    ST_TET, 4, P7, P3, P2, P0,
    /* case 174 */ 11,
    ST_PNT, 6, P2, P5, P1, P3, EH, ED,
    ST_WDG, 6, EL, EG, EF, P2, P7, P5,
    ST_PYR, 5, ED, P3, P7, EH, N0,
    ST_TET, 4, P3, P2, P7, N0,
    ST_TET, 4, P3, P1, P2, N0,
    ST_PYR, 5, EA, P1, P3, ED, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_TET, 4, P1, P5, P2, N0,
    ST_PYR, 5, EH, EE, EA, ED, N0,
    ST_TET, 4, P5, P7, P2, N0,
    ST_PYR, 5, EH, P7, P5, EE, N0,
    /* case 175 */ 5,
    ST_WDG, 6, EG, EF, EL, P7, P5, P2,
    ST_WDG, 6, P7, P5, P0, EH, EE, EI,
    ST_PYR, 5, P3, P0, P1, P2, P7,
    ST_TET, 4, P5, P1, P0, P7,
    ST_TET, 4, P5, P2, P1, P7,
    /* case 176 */ 7,
    ST_PNT, 5, EJ, EI, EF, EG, EK,
    ST_PYR, 5, EI, EK, P7, P4, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_TET, 4, EF, EJ, P5, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_PYR, 5, EF, P5, P7, EG, N0,
    ST_TET, 4, P7, EK, EG, N0,
    /* case 177 */ 5,
    ST_TET, 4, P4, P5, P0, P7,
    ST_PYR, 5, EK, P7, P0, ED, EA,
    ST_PYR, 5, EG, EF, P5, P7, EJ,
    ST_PYR, 5, P0, P5, EJ, EA, P7,
    ST_PYR, 5, EK, EA, EJ, EG, P7,
    /* case 178 */ 9,
    ST_PNT, 6, EI, EA, EB, EF, EG, EK,
    ST_TET, 4, P4, P5, P1, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    ST_PYR, 5, EI, EK, P7, P4, N0,
    ST_TET, 4, P7, P5, P4, N0,
    ST_TET, 4, P7, EK, EG, N0,
    ST_PYR, 5, P7, EG, EF, P5, N0,
    ST_PYR, 5, EF, EB, P1, P5, N0,
    ST_TET, 4, EB, EA, P1, N0,
    /* case 179 */ 9,
    ST_PNT, 5, EF, EB, ED, EG, EK,
    ST_PYR, 5, P4, P5, P1, P0, N0,
    ST_TET, 4, P0, P7, P4, N0,
    ST_TET, 4, P7, P5, P4, N0,
    ST_PYR, 5, P7, EG, EF, P5, N0,
    ST_PYR, 5, EF, EB, P1, P5, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_PYR, 5, ED, EK, P7, P0, N0,
    ST_TET, 4, EG, P7, EK, N0,
    /* case 180 */ 9,
    ST_PNT, 5, EI, EK, EJ, EB, EC,
    ST_WDG, 6, P5, P7, P2, EF, EG, EL,
    ST_TET, 4, P7, P2, P5, N0,
    ST_PYR, 5, P5, P2, EB, EJ, N0,
    ST_TET, 4, P4, P7, P5, N0,
    ST_PYR, 5, EK, P7, P4, EI, N0,
    ST_PYR, 5, EI, P4, P5, EJ, N0,
    ST_PYR, 5, P2, P7, EK, EC, N0,
    ST_TET, 4, EB, P2, EC, N0,
    /* case 181 */ 5,
    ST_WDG, 6, P2, P0, P5, EB, EA, EJ,
    ST_WDG, 6, EK, EC, ED, P7, P2, P0,
    ST_WDG, 6, P7, P2, P5, EG, EL, EF,
    ST_TET, 4, P2, P0, P5, P7,
    ST_TET, 4, P7, P5, P4, P0,
    /* case 182 */ 11,
    ST_PNT, 6, P5, P2, P1, P4, EK, EI,
    ST_WDG, 6, P5, P7, P2, EF, EG, EL,
    ST_PYR, 5, EI, EK, P7, P4, N0,
    ST_TET, 4, P4, P7, P5, N0,
    ST_TET, 4, P4, P5, P1, N0,
    ST_PYR, 5, EA, EI, P4, P1, N0,
    ST_PYR, 5, EC, EA, P1, P2, N0,
    ST_TET, 4, P1, P5, P2, N0,
    ST_PYR, 5, EK, EI, EA, EC, N0,
    ST_TET, 4, P2, P5, P7, N0,
    ST_PYR, 5, EK, EC, P2, P7, N0,
    /* case 183 */ 5,
    ST_WDG, 6, P7, P2, P5, EG, EL, EF,
    ST_WDG, 6, EK, EC, ED, P7, P2, P0,
    ST_PYR, 5, P4, P5, P1, P0, P7,
    ST_TET, 4, P2, P0, P1, P7,
    ST_TET, 4, P2, P1, P5, P7,
    /* case 184 */ 9,
    ST_PNT, 6, EI, ED, EC, EG, EF, EJ,
    ST_TET, 4, P4, P3, P7, N0,
    ST_PYR, 5, EI, ED, P3, P4, N0,
    ST_PYR, 5, EI, P4, P5, EJ, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    ST_PYR, 5, P5, P7, EG, EF, N0,
    ST_PYR, 5, EG, P7, P3, EC, N0,
    ST_TET, 4, EC, P3, ED, N0,
    /* case 185 */ 9,
    ST_PNT, 5, EG, EC, EA, EF, EJ,
    ST_PYR, 5, P4, P0, P3, P7, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_PYR, 5, P5, P7, EG, EF, N0,
    ST_PYR, 5, EG, P7, P3, EC, N0,
    ST_PYR, 5, EC, P3, P0, EA, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_TET, 4, EF, EJ, P5, N0,
    /* case 186 */ 11,
    ST_PNT, 6, P4, P3, P7, P5, EB, EF,
    ST_WDG, 6, P4, P1, P3, EI, EA, ED,
    ST_PYR, 5, EF, EB, P1, P5, N0,
    ST_TET, 4, P5, P1, P4, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_PYR, 5, EG, EF, P5, P7, N0,
    ST_PYR, 5, EC, EG, P7, P3, N0,
    ST_TET, 4, P7, P4, P3, N0,
    ST_PYR, 5, EB, EF, EG, EC, N0,
    ST_TET, 4, P3, P4, P1, N0,
    ST_PYR, 5, EB, EC, P3, P1, N0,
    /* case 187 */ 2,
    ST_HEX, 8, EC, EB, EF, EG, P3, P1, P5, P7,
    ST_WDG, 6, P0, P3, P1, P4, P7, P5,
    /* case 188 */ 11,
    ST_PNT, 6, P7, P5, P4, P3, EB, ED,
    ST_WDG, 6, P7, P2, P5, EG, EL, EF,
    ST_PYR, 5, ED, EB, P2, P3, N0,
    ST_TET, 4, P3, P2, P7, N0,
    ST_TET, 4, P3, P7, P4, N0,
    ST_PYR, 5, EI, ED, P3, P4, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_TET, 4, P4, P7, P5, N0,
    ST_PYR, 5, EB, ED, EI, EJ, N0,
    ST_TET, 4, P5, P7, P2, N0,
    ST_PYR, 5, EB, EJ, P5, P2, N0,
    /* case 189 */ 5,
    ST_WDG, 6, P2, P5, P7, EL, EF, EG,
    ST_WDG, 6, EB, EJ, EA, P2, P5, P0,
    ST_PYR, 5, P3, P7, P4, P0, P2,
    ST_TET, 4, P5, P0, P4, P2,
    ST_TET, 4, P5, P4, P7, P2,
    /* case 190 */ 6,
    ST_WDG, 6, ED, EA, EI, P3, P1, P4,
    ST_WDG, 6, EL, EG, EF, P2, P7, P5,
    ST_TET, 4, P5, P4, P7, P2,
    ST_TET, 4, P2, P3, P1, P4,
    ST_TET, 4, P3, P2, P7, P4,
    ST_TET, 4, P2, P1, P5, P4,
    /* case 191 */ 9,
    ST_PNT, 7, P1, P0, P3, P2, P5, P4, P7,
    ST_WDG, 6, P5, P7, P2, EF, EG, EL,
    ST_TET, 4, P5, P7, P2, N0,
    ST_PYR, 5, P4, P0, P3, P7, N0,
    ST_TET, 4, P7, P3, P2, N0,
    ST_PYR, 5, P1, P2, P3, P0, N0,
    ST_TET, 4, P1, P5, P2, N0,
    ST_PYR, 5, P1, P0, P4, P5, N0,
    ST_TET, 4, P5, P4, P7, N0,
    /* case 192 */ 1,
    ST_WDG, 6, P6, EF, EL, P7, EH, EK,
    /* case 193 */ 8,
    ST_PNT, 4, EL, EF, EA, EA,
    ST_PYR, 5, P0, P7, EH, EI, N0,
    ST_PYR, 5, EK, P7, P0, ED, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_TET, 4, ED, P0, EA, N0,
    ST_PYR, 5, EH, P7, P6, EF, N0,
    ST_PYR, 5, P7, EK, EL, P6, N0,
    ST_TET, 4, EL, EF, P6, N0,
    /* case 194 */ 8,
    ST_PNT, 4, EK, EH, EA, EA,
    ST_PYR, 5, P1, EJ, EF, P6, N0,
    ST_PYR, 5, EL, EB, P1, P6, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_TET, 4, EB, EA, P1, N0,
    ST_PYR, 5, EF, EH, P7, P6, N0,
    ST_PYR, 5, P6, P7, EK, EL, N0,
    ST_TET, 4, EK, P7, EH, N0,
    /* case 195 */ 2,
    ST_HEX, 8, EI, EH, EF, EJ, P0, P7, P6, P1,
    ST_HEX, 8, P0, P7, P6, P1, ED, EK, EL, EB,
    /* case 196 */ 7,
    ST_PNT, 5, EH, EF, EK, EC, EB,
    ST_PYR, 5, EF, P6, P2, EB, N0,
    ST_PYR, 5, EH, P7, P6, EF, N0,
    ST_TET, 4, EK, P7, EH, N0,
    ST_TET, 4, P7, P2, P6, N0,
    ST_PYR, 5, EK, EC, P2, P7, N0,
    ST_TET, 4, P2, EC, EB, N0,
    /* case 197 */ 9,
    ST_PNT, 5, EF, EB, EH, EI, EA,
    ST_WDG, 6, EK, EC, ED, P7, P2, P0,
    ST_TET, 4, P2, P7, P0, N0,
    ST_PYR, 5, P7, EH, EI, P0, N0,
    ST_TET, 4, P6, P7, P2, N0,
    ST_PYR, 5, EB, EF, P6, P2, N0,
    ST_PYR, 5, EF, EH, P7, P6, N0,
    ST_PYR, 5, P0, EA, EB, P2, N0,
    ST_TET, 4, EI, EA, P0, N0,
    /* case 198 */ 9,
    ST_PNT, 6, EC, EK, EH, EF, EJ, EA,
    ST_TET, 4, P2, P6, P7, N0,
    ST_PYR, 5, EC, P2, P7, EK, N0,
    ST_PYR, 5, EC, EA, P1, P2, N0,
    ST_TET, 4, P1, P6, P2, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_PYR, 5, P1, EJ, EF, P6, N0,
    ST_PYR, 5, EF, EH, P7, P6, N0,
    ST_TET, 4, EH, EK, P7, N0,
    /* case 199 */ 11,
    ST_PNT, 6, P2, P0, P1, P6, EH, EF,
    ST_WDG, 6, P2, P7, P0, EC, EK, ED,
    ST_PYR, 5, EF, EH, P7, P6, N0,
    ST_TET, 4, P6, P7, P2, N0,
    ST_TET, 4, P6, P2, P1, N0,
    ST_PYR, 5, EJ, EF, P6, P1, N0,
    ST_PYR, 5, EI, EJ, P1, P0, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_PYR, 5, EH, EF, EJ, EI, N0,
    ST_TET, 4, P0, P2, P7, N0,
    ST_PYR, 5, EH, EI, P0, P7, N0,
    /* case 200 */ 7,
    ST_PNT, 5, ED, EH, EC, EL, EF,
    ST_PYR, 5, EH, P7, P6, EF, N0,
    ST_PYR, 5, ED, P3, P7, EH, N0,
    ST_TET, 4, EC, P3, ED, N0,
    ST_TET, 4, P3, P6, P7, N0,
    ST_PYR, 5, EC, EL, P6, P3, N0,
    ST_TET, 4, P6, EL, EF, N0,
    /* case 201 */ 9,
    ST_PNT, 6, EC, EL, EF, EH, EI, EA,
    ST_TET, 4, P3, P6, P7, N0,
    ST_PYR, 5, EC, EL, P6, P3, N0,
    ST_PYR, 5, EC, P3, P0, EA, N0,
    ST_TET, 4, P0, P3, P7, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_PYR, 5, P0, P7, EH, EI, N0,
    ST_PYR, 5, EH, P7, P6, EF, N0,
    ST_TET, 4, EF, P6, EL, N0,
    /* case 202 */ 9,
    ST_PNT, 5, EH, EF, ED, EA, EJ,
    ST_WDG, 6, EC, EL, EB, P3, P6, P1,
    ST_TET, 4, P6, P3, P1, N0,
    ST_PYR, 5, P3, ED, EA, P1, N0,
    ST_TET, 4, P7, P3, P6, N0,
    ST_PYR, 5, EF, EH, P7, P6, N0,
    ST_PYR, 5, EH, ED, P3, P7, N0,
    ST_PYR, 5, P1, EJ, EF, P6, N0,
    ST_TET, 4, EA, EJ, P1, N0,
    /* case 203 */ 11,
    ST_PNT, 6, P3, P1, P0, P7, EF, EH,
    ST_WDG, 6, EC, EL, EB, P3, P6, P1,
    ST_PYR, 5, EH, P7, P6, EF, N0,
    ST_TET, 4, P7, P3, P6, N0,
    ST_TET, 4, P7, P0, P3, N0,
    ST_PYR, 5, EI, P0, P7, EH, N0,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_PYR, 5, EF, EJ, EI, EH, N0,
    ST_TET, 4, P1, P6, P3, N0,
    ST_PYR, 5, EF, P6, P1, EJ, N0,
    /* case 204 */ 1,
    ST_HEX, 8, P3, P2, P6, P7, ED, EB, EF, EH,
    /* case 205 */ 9,
    ST_PNT, 5, EB, EF, EH, EA, EI,
    ST_PYR, 5, P3, P2, P6, P7, N0,
    ST_TET, 4, P7, P0, P3, N0,
    ST_TET, 4, P0, P2, P3, N0,
    ST_PYR, 5, P0, EA, EB, P2, N0,
    ST_PYR, 5, EB, EF, P6, P2, N0,
    ST_PYR, 5, EF, EH, P7, P6, N0,
    ST_PYR, 5, EH, EI, P0, P7, N0,
    ST_TET, 4, EA, P0, EI, N0,
    /* case 206 */ 9,
    ST_PNT, 5, EF, EH, ED, EJ, EA,
    ST_PYR, 5, P2, P6, P7, P3, N0,
    ST_TET, 4, P3, P1, P2, N0,
    ST_TET, 4, P1, P6, P2, N0,
    ST_PYR, 5, P1, EJ, EF, P6, N0,
    ST_PYR, 5, EF, EH, P7, P6, N0,
    ST_PYR, 5, EH, ED, P3, P7, N0,
    ST_PYR, 5, ED, EA, P1, P3, N0,
    ST_TET, 4, EJ, P1, EA, N0,
    /* case 207 */ 2,
    ST_HEX, 8, EI, EH, EF, EJ, P0, P7, P6, P1,
    ST_WDG, 6, P3, P0, P7, P2, P1, P6,
    /* case 208 */ 7,
    ST_PNT, 5, EI, EK, EE, EF, EL,
    ST_PYR, 5, EK, EL, P6, P7, N0,
    ST_PYR, 5, EI, EK, P7, P4, N0,
    ST_TET, 4, EE, EI, P4, N0,
    ST_TET, 4, P4, P7, P6, N0,
    ST_PYR, 5, EE, P4, P6, EF, N0,
    ST_TET, 4, P6, EL, EF, N0,
    /* case 209 */ 9,
    ST_PNT, 6, EE, EF, EL, EK, ED, EA,
    ST_TET, 4, P4, P7, P6, N0,
    ST_PYR, 5, EE, P4, P6, EF, N0,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_TET, 4, P0, P7, P4, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_PYR, 5, P0, ED, EK, P7, N0,
    ST_PYR, 5, EK, EL, P6, P7, N0,
    ST_TET, 4, EL, EF, P6, N0,
    /* case 210 */ 9,
    ST_PNT, 5, EK, EL, EI, EA, EB,
    ST_WDG, 6, P4, P6, P1, EE, EF, EJ,
    ST_TET, 4, P6, P1, P4, N0,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_TET, 4, P7, P6, P4, N0,
    ST_PYR, 5, EL, P6, P7, EK, N0,
    ST_PYR, 5, EK, P7, P4, EI, N0,
    ST_PYR, 5, P1, P6, EL, EB, N0,
    ST_TET, 4, EA, P1, EB, N0,
    /* case 211 */ 11,
    ST_PNT, 6, P4, P1, P0, P7, EL, EK,
    ST_WDG, 6, P4, P6, P1, EE, EF, EJ,
    ST_PYR, 5, EK, EL, P6, P7, N0,
    ST_TET, 4, P7, P6, P4, N0,
    ST_TET, 4, P7, P4, P0, N0,
    ST_PYR, 5, ED, EK, P7, P0, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_TET, 4, P0, P4, P1, N0,
    ST_PYR, 5, EL, EK, ED, EB, N0,
    ST_TET, 4, P1, P4, P6, N0,
    ST_PYR, 5, EL, EB, P1, P6, N0,
    /* case 212 */ 9,
    ST_PNT, 6, EK, EC, EB, EF, EE, EI,
    ST_TET, 4, P7, P2, P6, N0,
    ST_PYR, 5, EK, EC, P2, P7, N0,
    ST_PYR, 5, EK, P7, P4, EI, N0,
    ST_TET, 4, P4, P7, P6, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_PYR, 5, P4, P6, EF, EE, N0,
    ST_PYR, 5, EF, P6, P2, EB, N0,
    ST_TET, 4, EB, P2, EC, N0,
    /* case 213 */ 11,
    ST_PNT, 6, P7, P0, P4, P6, EB, EF,
    ST_WDG, 6, EK, EC, ED, P7, P2, P0,
    ST_PYR, 5, EF, P6, P2, EB, N0,
    ST_TET, 4, P6, P7, P2, N0,
    ST_TET, 4, P6, P4, P7, N0,
    ST_PYR, 5, EE, P4, P6, EF, N0,
    ST_PYR, 5, EA, P0, P4, EE, N0,
    ST_TET, 4, P4, P0, P7, N0,
    ST_PYR, 5, EB, EA, EE, EF, N0,
    ST_TET, 4, P0, P2, P7, N0,
    ST_PYR, 5, EB, P2, P0, EA, N0,
    /* case 214 */ 11,
    ST_PNT, 6, P6, P1, P2, P7, EI, EK,
    ST_WDG, 6, EF, EE, EJ, P6, P4, P1,
    ST_PYR, 5, EK, P7, P4, EI, N0,
    ST_TET, 4, P7, P6, P4, N0,
    ST_TET, 4, P7, P2, P6, N0,
    ST_PYR, 5, EC, P2, P7, EK, N0,
    ST_PYR, 5, EA, P1, P2, EC, N0,
    ST_TET, 4, P2, P1, P6, N0,
    ST_PYR, 5, EI, EA, EC, EK, N0,
    ST_TET, 4, P1, P4, P6, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    /* case 215 */ 6,
    ST_WDG, 6, EJ, EF, EE, P1, P6, P4,
    ST_WDG, 6, EC, ED, EK, P2, P0, P7,
    ST_TET, 4, P7, P4, P0, P2,
    ST_TET, 4, P2, P1, P6, P4,
    ST_TET, 4, P1, P2, P0, P4,
    ST_TET, 4, P2, P6, P7, P4,
    /* case 216 */ 5,
    ST_TET, 4, P7, P6, P4, P3,
    ST_PYR, 5, ED, P3, P4, EI, EE,
    ST_PYR, 5, EC, EL, P6, P3, EF,
    ST_PYR, 5, P4, P6, EF, EE, P3,
    ST_PYR, 5, ED, EE, EF, EC, P3,
    /* case 217 */ 9,
    ST_PNT, 5, EC, EA, EE, EL, EF,
    ST_PYR, 5, P7, P4, P0, P3, N0,
    ST_TET, 4, P4, P7, P6, N0,
    ST_TET, 4, P6, P7, P3, N0,
    ST_PYR, 5, P6, P3, EC, EL, N0,
    ST_PYR, 5, EC, P3, P0, EA, N0,
    ST_PYR, 5, EA, P0, P4, EE, N0,
    ST_PYR, 5, EE, P4, P6, EF, N0,
    ST_TET, 4, EL, EF, P6, N0,
    /* case 218 */ 5,
    ST_WDG, 6, P1, P4, P6, EJ, EE, EF,
    ST_WDG, 6, ED, EA, EI, P3, P1, P4,
    ST_WDG, 6, P3, P1, P6, EC, EB, EL,
    ST_TET, 4, P1, P4, P6, P3,
    ST_TET, 4, P3, P6, P7, P4,
    /* case 219 */ 5,
    ST_WDG, 6, EJ, EF, EE, P1, P6, P4,
    ST_WDG, 6, P1, P6, P3, EB, EL, EC,
    ST_PYR, 5, P0, P3, P7, P4, P1,
    ST_TET, 4, P6, P7, P3, P1,
    ST_TET, 4, P6, P4, P7, P1,
    /* case 220 */ 9,
    ST_PNT, 5, EF, EB, ED, EE, EI,
    ST_PYR, 5, P7, P3, P2, P6, N0,
    ST_TET, 4, P3, P7, P4, N0,
    ST_TET, 4, P4, P7, P6, N0,
    ST_PYR, 5, P4, P6, EF, EE, N0,
    ST_PYR, 5, EF, P6, P2, EB, N0,
    ST_PYR, 5, EB, P2, P3, ED, N0,
    ST_PYR, 5, ED, P3, P4, EI, N0,
    ST_TET, 4, EE, EI, P4, N0,
    /* case 221 */ 2,
    ST_HEX, 8, P0, P2, P6, P4, EA, EB, EF, EE,
    ST_WDG, 6, P7, P4, P6, P3, P0, P2,
    /* case 222 */ 5,
    ST_WDG, 6, EE, EJ, EF, P4, P1, P6,
    ST_WDG, 6, P4, P1, P3, EI, EA, ED,
    ST_PYR, 5, P7, P3, P2, P6, P4,
    ST_TET, 4, P1, P2, P3, P4,
    ST_TET, 4, P1, P6, P2, P4,
    /* case 223 */ 9,
    ST_PNT, 7, P0, P3, P7, P4, P1, P2, P6,
    ST_WDG, 6, EJ, EF, EE, P1, P6, P4,
    ST_TET, 4, P1, P4, P6, N0,
    ST_PYR, 5, P2, P6, P7, P3, N0,
    ST_TET, 4, P6, P4, P7, N0,
    ST_PYR, 5, P0, P3, P7, P4, N0,
    ST_TET, 4, P0, P4, P1, N0,
    ST_PYR, 5, P0, P1, P2, P3, N0,
    ST_TET, 4, P1, P6, P2, N0,
    /* case 224 */ 7,
    ST_PNT, 5, EK, EL, EH, EE, EJ,
    ST_PYR, 5, EL, EJ, P5, P6, N0,
    ST_PYR, 5, EK, EL, P6, P7, N0,
    ST_TET, 4, EH, EK, P7, N0,
    ST_TET, 4, P7, P6, P5, N0,
    ST_PYR, 5, EH, P7, P5, EE, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    /* case 225 */ 9,
    ST_PNT, 5, EL, EJ, EK, ED, EA,
    ST_WDG, 6, P7, P5, P0, EH, EE, EI,
    ST_TET, 4, P5, P0, P7, N0,
    ST_PYR, 5, P7, P0, ED, EK, N0,
    ST_TET, 4, P6, P5, P7, N0,
    ST_PYR, 5, EJ, P5, P6, EL, N0,
    ST_PYR, 5, EL, P6, P7, EK, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_TET, 4, ED, P0, EA, N0,
    /* case 226 */ 9,
    ST_PNT, 6, EE, EH, EK, EL, EB, EA,
    ST_TET, 4, P5, P7, P6, N0,
    ST_PYR, 5, EE, EH, P7, P5, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_TET, 4, P1, P5, P6, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_PYR, 5, P1, P6, EL, EB, N0,
    ST_PYR, 5, EL, P6, P7, EK, N0,
    ST_TET, 4, EK, P7, EH, N0,
    /* case 227 */ 11,
    ST_PNT, 6, P5, P0, P1, P6, EK, EL,
    ST_WDG, 6, EE, EH, EI, P5, P7, P0,
    ST_PYR, 5, EL, P6, P7, EK, N0,
    ST_TET, 4, P6, P5, P7, N0,
    ST_TET, 4, P6, P1, P5, N0,
    ST_PYR, 5, EB, P1, P6, EL, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_TET, 4, P1, P0, P5, N0,
    ST_PYR, 5, EK, ED, EB, EL, N0,
    ST_TET, 4, P0, P7, P5, N0,
    ST_PYR, 5, EK, P7, P0, ED, N0,
    /* case 228 */ 5,
    ST_TET, 4, P6, P5, P7, P2,
    ST_PYR, 5, EB, EJ, P5, P2, EE,
    ST_PYR, 5, EC, P2, P7, EK, EH,
    ST_PYR, 5, P5, EE, EH, P7, P2,
    ST_PYR, 5, EB, EC, EH, EE, P2,
    /* case 229 */ 5,
    ST_WDG, 6, EI, EE, EH, P0, P5, P7,
    ST_WDG, 6, P2, P0, P5, EB, EA, EJ,
    ST_WDG, 6, EC, ED, EK, P2, P0, P7,
    ST_TET, 4, P0, P7, P5, P2,
    ST_TET, 4, P2, P6, P7, P5,
    /* case 230 */ 9,
    ST_PNT, 5, EC, EA, EE, EK, EH,
    ST_PYR, 5, P6, P2, P1, P5, N0,
    ST_TET, 4, P5, P7, P6, N0,
    ST_TET, 4, P7, P2, P6, N0,
    ST_PYR, 5, P7, EK, EC, P2, N0,
    ST_PYR, 5, EC, EA, P1, P2, N0,
    ST_PYR, 5, EA, EE, P5, P1, N0,
    ST_PYR, 5, EE, EH, P7, P5, N0,
    ST_TET, 4, EK, P7, EH, N0,
    /* case 231 */ 5,
    ST_WDG, 6, P0, P7, P5, EI, EH, EE,
    ST_WDG, 6, ED, EK, EC, P0, P7, P2,
    ST_PYR, 5, P1, P5, P6, P2, P0,
    ST_TET, 4, P7, P2, P6, P0,
    ST_TET, 4, P7, P6, P5, P0,
    /* case 232 */ 9,
    ST_PNT, 6, EH, EE, EJ, EL, EC, ED,
    ST_TET, 4, P7, P6, P5, N0,
    ST_PYR, 5, EH, P7, P5, EE, N0,
    ST_PYR, 5, EH, ED, P3, P7, N0,
    ST_TET, 4, P3, P6, P7, N0,
    ST_TET, 4, P3, ED, EC, N0,
    ST_PYR, 5, P3, EC, EL, P6, N0,
    ST_PYR, 5, EL, EJ, P5, P6, N0,
    ST_TET, 4, EJ, EE, P5, N0,
    /* case 233 */ 11,
    ST_PNT, 6, P7, P0, P3, P6, EJ, EL,
    ST_WDG, 6, P7, P5, P0, EH, EE, EI,
    ST_PYR, 5, EL, EJ, P5, P6, N0,
    ST_TET, 4, P6, P5, P7, N0,
    ST_TET, 4, P6, P7, P3, N0,
    ST_PYR, 5, EC, EL, P6, P3, N0,
    ST_PYR, 5, EA, EC, P3, P0, N0,
    ST_TET, 4, P3, P7, P0, N0,
    ST_PYR, 5, EJ, EL, EC, EA, N0,
    ST_TET, 4, P0, P7, P5, N0,
    ST_PYR, 5, EJ, EA, P0, P5, N0,
    /* case 234 */ 11,
    ST_PNT, 6, P6, P1, P5, P7, ED, EH,
    ST_WDG, 6, P6, P3, P1, EL, EC, EB,
    ST_PYR, 5, EH, ED, P3, P7, N0,
    ST_TET, 4, P7, P3, P6, N0,
    ST_TET, 4, P7, P6, P5, N0,
    ST_PYR, 5, EE, EH, P7, P5, N0,
    ST_PYR, 5, EA, EE, P5, P1, N0,
    ST_TET, 4, P5, P6, P1, N0,
    ST_PYR, 5, ED, EH, EE, EA, N0,
    ST_TET, 4, P1, P6, P3, N0,
    ST_PYR, 5, ED, EA, P1, P3, N0,
    /* case 235 */ 6,
    ST_WDG, 6, P1, P6, P3, EB, EL, EC,
    ST_WDG, 6, P5, P0, P7, EE, EI, EH,
    ST_TET, 4, P7, P0, P3, P5,
    ST_TET, 4, P5, P6, P1, P3,
    ST_TET, 4, P1, P0, P5, P3,
    ST_TET, 4, P5, P7, P6, P3,
    /* case 236 */ 9,
    ST_PNT, 5, EH, ED, EB, EE, EJ,
    ST_PYR, 5, P6, P7, P3, P2, N0,
    ST_TET, 4, P2, P5, P6, N0,
    ST_TET, 4, P5, P7, P6, N0,
    ST_PYR, 5, P5, EE, EH, P7, N0,
    ST_PYR, 5, EH, ED, P3, P7, N0,
    ST_PYR, 5, ED, EB, P2, P3, N0,
    ST_PYR, 5, EB, EJ, P5, P2, N0,
    ST_TET, 4, EE, P5, EJ, N0,
    /* case 237 */ 5,
    ST_WDG, 6, P0, P5, P2, EA, EJ, EB,
    ST_WDG, 6, EI, EE, EH, P0, P5, P7,
    ST_PYR, 5, P3, P2, P6, P7, P0,
    ST_TET, 4, P5, P7, P6, P0,
    ST_TET, 4, P5, P6, P2, P0,
    /* case 238 */ 2,
    ST_HEX, 8, EA, ED, EH, EE, P1, P3, P7, P5,
    ST_WDG, 6, P2, P1, P3, P6, P5, P7,
    /* case 239 */ 9,
    ST_PNT, 7, P3, P2, P1, P0, P7, P6, P5,
    ST_WDG, 6, P7, P5, P0, EH, EE, EI,
    ST_TET, 4, P7, P5, P0, N0,
    ST_PYR, 5, P6, P2, P1, P5, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_PYR, 5, P3, P0, P1, P2, N0,
    ST_TET, 4, P3, P7, P0, N0,
    ST_PYR, 5, P3, P2, P6, P7, N0,
    ST_TET, 4, P7, P6, P5, N0,
    /* case 240 */ 1,
    ST_HEX, 8, EI, EJ, EL, EK, P4, P5, P6, P7,
    /* case 241 */ 9,
    ST_PNT, 5, EJ, EL, EK, EA, ED,
    ST_PYR, 5, P4, P7, P6, P5, N0,
    ST_TET, 4, P7, P4, P0, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_PYR, 5, EJ, P5, P6, EL, N0,
    ST_PYR, 5, EL, P6, P7, EK, N0,
    ST_PYR, 5, EK, P7, P0, ED, N0,
    ST_TET, 4, EA, ED, P0, N0,
    /* case 242 */ 9,
    ST_PNT, 5, EL, EK, EI, EB, EA,
    ST_PYR, 5, P5, P4, P7, P6, N0,
    ST_TET, 4, P4, P5, P1, N0,
    ST_TET, 4, P1, P5, P6, N0,
    ST_PYR, 5, P1, P6, EL, EB, N0,
    ST_PYR, 5, EL, P6, P7, EK, N0,
    ST_PYR, 5, EK, P7, P4, EI, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    ST_TET, 4, EB, EA, P1, N0,
    /* case 243 */ 2,
    ST_HEX, 8, P0, P7, P6, P1, ED, EK, EL, EB,
    ST_WDG, 6, P5, P1, P6, P4, P0, P7,
    /* case 244 */ 9,
    ST_PNT, 5, EK, EI, EJ, EC, EB,
    ST_PYR, 5, P6, P5, P4, P7, N0,
    ST_TET, 4, P5, P6, P2, N0,
    ST_TET, 4, P2, P6, P7, N0,
    ST_PYR, 5, P2, P7, EK, EC, N0,
    ST_PYR, 5, EK, P7, P4, EI, N0,
    ST_PYR, 5, EI, P4, P5, EJ, N0,
    ST_PYR, 5, EJ, P5, P2, EB, N0,
    ST_TET, 4, EC, EB, P2, N0,
    /* case 245 */ 5,
    ST_WDG, 6, EA, EB, EJ, P0, P2, P5,
    ST_WDG, 6, P0, P2, P7, ED, EC, EK,
    ST_PYR, 5, P4, P7, P6, P5, P0,
    ST_TET, 4, P2, P6, P7, P0,
    ST_TET, 4, P2, P5, P6, P0,
    /* case 246 */ 2,
    ST_HEX, 8, P1, P4, P7, P2, EA, EI, EK, EC,
    ST_WDG, 6, P6, P2, P7, P5, P1, P4,
    /* case 247 */ 9,
    ST_PNT, 7, P4, P5, P1, P0, P7, P6, P2,
    ST_WDG, 6, EK, EC, ED, P7, P2, P0,
    ST_TET, 4, P7, P0, P2, N0,
    ST_PYR, 5, P6, P2, P1, P5, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_PYR, 5, P4, P5, P1, P0, N0,
    ST_TET, 4, P4, P0, P7, N0,
    ST_PYR, 5, P4, P7, P6, P5, N0,
    ST_TET, 4, P7, P2, P6, N0,
    /* case 248 */ 9,
    ST_PNT, 5, EL, EJ, EI, EC, ED,
    ST_PYR, 5, P7, P6, P5, P4, N0,
    ST_TET, 4, P4, P3, P7, N0,
    ST_TET, 4, P3, P6, P7, N0,
    ST_PYR, 5, P3, EC, EL, P6, N0,
    ST_PYR, 5, EL, EJ, P5, P6, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_PYR, 5, EI, ED, P3, P4, N0,
    ST_TET, 4, EC, P3, ED, N0,
    /* case 249 */ 2,
    ST_HEX, 8, EA, EJ, EL, EC, P0, P5, P6, P3,
    ST_WDG, 6, P4, P0, P5, P7, P3, P6,
    /* case 250 */ 5,
    ST_WDG, 6, P3, P1, P6, EC, EB, EL,
    ST_WDG, 6, ED, EA, EI, P3, P1, P4,
    ST_PYR, 5, P7, P6, P5, P4, P3,
    ST_TET, 4, P1, P4, P5, P3,
    ST_TET, 4, P1, P5, P6, P3,
    /* case 251 */ 9,
    ST_PNT, 7, P0, P4, P7, P3, P1, P5, P6,
    ST_WDG, 6, P1, P6, P3, EB, EL, EC,
    ST_TET, 4, P1, P6, P3, N0,
    ST_PYR, 5, P5, P4, P7, P6, N0,
    ST_TET, 4, P6, P7, P3, N0,
    ST_PYR, 5, P0, P3, P7, P4, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_PYR, 5, P0, P4, P5, P1, N0,
    ST_TET, 4, P1, P5, P6, N0,
    /* case 252 */ 2,
    ST_HEX, 8, P4, P3, P2, P5, EI, ED, EB, EJ,
    ST_WDG, 6, P6, P5, P2, P7, P4, P3,
    /* case 253 */ 9,
    ST_PNT, 7, P3, P7, P4, P0, P2, P6, P5,
    ST_WDG, 6, EB, EJ, EA, P2, P5, P0,
    ST_TET, 4, P2, P0, P5, N0,
    ST_PYR, 5, P6, P5, P4, P7, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_PYR, 5, P3, P7, P4, P0, N0,
    ST_TET, 4, P3, P0, P2, N0,
    ST_PYR, 5, P3, P2, P6, P7, N0,
    ST_TET, 4, P2, P5, P6, N0,
    /* case 254 */ 9,
    ST_PNT, 7, P7, P6, P5, P4, P3, P2, P1,
    ST_WDG, 6, ED, EA, EI, P3, P1, P4,
    ST_TET, 4, P3, P4, P1, N0,
    ST_PYR, 5, P2, P1, P5, P6, N0,
    ST_TET, 4, P1, P4, P5, N0,
    ST_PYR, 5, P7, P6, P5, P4, N0,
    ST_TET, 4, P7, P4, P3, N0,
    ST_PYR, 5, P7, P3, P2, P6, N0,
    ST_TET, 4, P3, P1, P2, N0,
    /* case 255 */ 1,
    ST_HEX, 8, P0, P1, P2, P3, P4, P5, P6, P7,
    // VTK_WEDGE
    /* case 0 */ 0,
    /* case 1 */ 1,
    ST_TET, 4, EG, EA, EC, P0,
    /* case 2 */ 1,
    ST_TET, 4, EH, EB, EA, P1,
    /* case 3 */ 1,
    ST_WDG, 6, EC, EG, P0, EB, EH, P1,
    /* case 4 */ 1,
    ST_TET, 4, EI, EC, EB, P2,
    /* case 5 */ 1,
    ST_WDG, 6, EB, EI, P2, EA, EG, P0,
    /* case 6 */ 1,
    ST_WDG, 6, EA, EH, P1, EC, EI, P2,
    /* case 7 */ 1,
    ST_WDG, 6, P0, P1, P2, EG, EH, EI,
    /* case 8 */ 1,
    ST_TET, 4, EG, EF, ED, P3,
    /* case 9 */ 1,
    ST_WDG, 6, P0, EA, EC, P3, ED, EF,
    /* case 10 */ 7,
    ST_PNT, 6, EA, EB, EH, ED, EF, EG,
    ST_PYR, 5, EG, EA, P1, P3, N0,
    ST_PYR, 5, P3, P1, EH, ED, N0,
    ST_TET, 4, P3, ED, EF, N0,
    ST_TET, 4, EF, EG, P3, N0,
    ST_TET, 4, P1, EB, EH, N0,
    ST_TET, 4, P1, EA, EB, N0,
    /* case 11 */ 7,
    ST_PNT, 5, EB, EC, EF, ED, EH,
    ST_TET, 4, P0, P1, P3, N0,
    ST_PYR, 5, EC, EB, P1, P0, N0,
    ST_PYR, 5, EC, P0, P3, EF, N0,
    ST_TET, 4, EF, P3, ED, N0,
    ST_PYR, 5, P3, P1, EH, ED, N0,
    ST_TET, 4, P1, EB, EH, N0,
    /* case 12 */ 7,
    ST_PNT, 6, EF, ED, EG, EC, EB, EI,
    ST_PYR, 5, EI, P2, P3, EF, N0,
    ST_PYR, 5, P2, EC, EG, P3, N0,
    ST_TET, 4, P2, EB, EC, N0,
    ST_TET, 4, EB, P2, EI, N0,
    ST_TET, 4, P3, EG, ED, N0,
    ST_TET, 4, P3, ED, EF, N0,
    /* case 13 */ 7,
    ST_PNT, 5, EB, EA, ED, EF, EI,
    ST_TET, 4, P0, P3, P2, N0,
    ST_PYR, 5, EA, P0, P2, EB, N0,
    ST_PYR, 5, EA, ED, P3, P0, N0,
    ST_TET, 4, ED, EF, P3, N0,
    ST_PYR, 5, P3, EF, EI, P2, N0,
    ST_TET, 4, P2, EI, EB, N0,
    /* case 14 */ 8,
    ST_PNT, 7, ED, EF, EI, EH, P3, P2, P1,
    ST_WDG, 6, P2, P1, P3, EC, EA, EG,
    ST_PYR, 5, EF, ED, EH, EI, N0,
    ST_PYR, 5, EH, P1, P2, EI, N0,
    ST_TET, 4, P3, P2, P1, N0,
    ST_TET, 4, P3, ED, EF, N0,
    ST_PYR, 5, ED, P3, P1, EH, N0,
    ST_PYR, 5, EI, P2, P3, EF, N0,
    /* case 15 */ 8,
    ST_PNT, 7, P1, P2, P3, EF, ED, EH, EI,
    ST_TET, 4, P0, P2, P1, P3,
    ST_PYR, 5, EF, ED, EH, EI, N0,
    ST_PYR, 5, EI, EH, P1, P2, N0,
    ST_TET, 4, P3, P2, P1, N0,
    ST_TET, 4, P3, ED, EF, N0,
    ST_PYR, 5, P3, P1, EH, ED, N0,
    ST_PYR, 5, P2, P3, EF, EI, N0,
    /* case 16 */ 1,
    ST_TET, 4, EH, ED, EE, P4,
    /* case 17 */ 7,
    ST_PNT, 6, ED, EE, EH, EA, EC, EG,
    ST_PYR, 5, EG, P0, P4, ED, N0,
    ST_PYR, 5, P0, EA, EH, P4, N0,
    ST_TET, 4, P0, EC, EA, N0,
    ST_TET, 4, EC, P0, EG, N0,
    ST_TET, 4, P4, EH, EE, N0,
    ST_TET, 4, P4, EE, ED, N0,
    /* case 18 */ 1,
    ST_WDG, 6, P1, EB, EA, P4, EE, ED,
    /* case 19 */ 7,
    ST_PNT, 5, EC, EB, EE, ED, EG,
    ST_TET, 4, P1, P4, P0, N0,
    ST_PYR, 5, EB, P1, P0, EC, N0,
    ST_PYR, 5, EB, EE, P4, P1, N0,
    ST_TET, 4, EE, ED, P4, N0,
    ST_PYR, 5, P4, ED, EG, P0, N0,
    ST_TET, 4, P0, EG, EC, N0,
    /* case 20 */ 7,
    ST_PNT, 6, EB, EC, EI, EE, ED, EH,
    ST_PYR, 5, EH, EB, P2, P4, N0,
    ST_PYR, 5, P4, P2, EI, EE, N0,
    ST_TET, 4, P4, EE, ED, N0,
    ST_TET, 4, ED, EH, P4, N0,
    ST_TET, 4, P2, EC, EI, N0,
    ST_TET, 4, P2, EB, EC, N0,
    /* case 21 */ 8,
    ST_PNT, 7, EE, ED, EG, EI, P4, P0, P2,
    ST_WDG, 6, P0, P2, P4, EA, EB, EH,
    ST_PYR, 5, ED, EE, EI, EG, N0,
    ST_PYR, 5, EI, P2, P0, EG, N0,
    ST_TET, 4, P4, P0, P2, N0,
    ST_TET, 4, P4, EE, ED, N0,
    ST_PYR, 5, EE, P4, P2, EI, N0,
    ST_PYR, 5, EG, P0, P4, ED, N0,
    /* case 22 */ 7,
    ST_PNT, 5, EC, EA, ED, EE, EI,
    ST_TET, 4, P1, P2, P4, N0,
    ST_PYR, 5, EA, EC, P2, P1, N0,
    ST_PYR, 5, EA, P1, P4, ED, N0,
    ST_TET, 4, ED, P4, EE, N0,
    ST_PYR, 5, P4, P2, EI, EE, N0,
    ST_TET, 4, P2, EC, EI, N0,
    /* case 23 */ 8,
    ST_PNT, 7, P2, P0, P4, ED, EE, EI, EG,
    ST_TET, 4, P1, P0, P2, P4,
    ST_PYR, 5, ED, EE, EI, EG, N0,
    ST_PYR, 5, EG, EI, P2, P0, N0,
    ST_TET, 4, P4, P0, P2, N0,
    ST_TET, 4, P4, EE, ED, N0,
    ST_PYR, 5, P4, P2, EI, EE, N0,
    ST_PYR, 5, P0, P4, ED, EG, N0,
    /* case 24 */ 1,
    ST_WDG, 6, EE, EH, P4, EF, EG, P3,
    /* case 25 */ 7,
    ST_PNT, 5, EE, EF, EC, EA, EH,
    ST_TET, 4, P3, P0, P4, N0,
    ST_PYR, 5, EF, P3, P4, EE, N0,
    ST_PYR, 5, EF, EC, P0, P3, N0,
    ST_TET, 4, EC, EA, P0, N0,
    ST_PYR, 5, P0, EA, EH, P4, N0,
    ST_TET, 4, P4, EH, EE, N0,
    /* case 26 */ 7,
    ST_PNT, 5, EF, EE, EB, EA, EG,
    ST_TET, 4, P4, P3, P1, N0,
    ST_PYR, 5, EE, EF, P3, P4, N0,
    ST_PYR, 5, EE, P4, P1, EB, N0,
    ST_TET, 4, EB, P1, EA, N0,
    ST_PYR, 5, P1, P3, EG, EA, N0,
    ST_TET, 4, P3, EF, EG, N0,
    /* case 27 */ 1,
    ST_HEX, 8, P3, P4, EE, EF, P0, P1, EB, EC,
    /* case 28 */ 8,
    ST_PNT, 7, EC, EB, EH, EG, P2, P4, P3,
    ST_WDG, 6, EE, EF, EI, P4, P3, P2,
    ST_PYR, 5, EB, EH, EG, EC, N0,
    ST_PYR, 5, EG, EH, P4, P3, N0,
    ST_TET, 4, P2, P3, P4, N0,
    ST_TET, 4, P2, EB, EC, N0,
    ST_PYR, 5, EC, EG, P3, P2, N0,
    ST_PYR, 5, EH, EB, P2, P4, N0,
    /* case 29 */ 3,
    ST_WDG, 6, P2, P3, P4, EI, EF, EE,
    ST_TET, 4, P2, P3, P4, P0,
    ST_WDG, 6, P2, P4, P0, EB, EH, EA,
    /* case 30 */ 3,
    ST_WDG, 6, EG, EA, EC, P3, P1, P2,
    ST_TET, 4, P3, P2, P1, P4,
    ST_WDG, 6, EF, EI, EE, P3, P2, P4,
    /* case 31 */ 2,
    ST_WDG, 6, EI, EE, EF, P2, P4, P3,
    ST_PYR, 5, P0, P1, P4, P3, P2,
    /* case 32 */ 1,
    ST_TET, 4, EI, EE, EF, P5,
    /* case 33 */ 7,
    ST_PNT, 6, EC, EA, EG, EF, EE, EI,
    ST_PYR, 5, EI, EC, P0, P5, N0,
    ST_PYR, 5, P5, P0, EG, EF, N0,
    ST_TET, 4, P5, EF, EE, N0,
    ST_TET, 4, EE, EI, P5, N0,
    ST_TET, 4, P0, EA, EG, N0,
    ST_TET, 4, P0, EC, EA, N0,
    /* case 34 */ 7,
    ST_PNT, 6, EE, EF, EI, EB, EA, EH,
    ST_PYR, 5, EH, P1, P5, EE, N0,
    ST_PYR, 5, P1, EB, EI, P5, N0,
    ST_TET, 4, P1, EA, EB, N0,
    ST_TET, 4, EA, P1, EH, N0,
    ST_TET, 4, P5, EI, EF, N0,
    ST_TET, 4, P5, EF, EE, N0,
    /* case 35 */ 8,
    ST_PNT, 7, EF, EE, EH, EG, P5, P1, P0,
    ST_WDG, 6, P1, P0, P5, EB, EC, EI,
    ST_PYR, 5, EE, EF, EG, EH, N0,
    ST_PYR, 5, EG, P0, P1, EH, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_TET, 4, P5, EF, EE, N0,
    ST_PYR, 5, EF, P5, P0, EG, N0,
    ST_PYR, 5, EH, P1, P5, EE, N0,
    /* case 36 */ 1,
    ST_WDG, 6, P2, EC, EB, P5, EF, EE,
    /* case 37 */ 7,
    ST_PNT, 5, EA, EB, EE, EF, EG,
    ST_TET, 4, P2, P0, P5, N0,
    ST_PYR, 5, EB, EA, P0, P2, N0,
    ST_PYR, 5, EB, P2, P5, EE, N0,
    ST_TET, 4, EE, P5, EF, N0,
    ST_PYR, 5, P5, P0, EG, EF, N0,
    ST_TET, 4, P0, EA, EG, N0,
    /* case 38 */ 7,
    ST_PNT, 5, EA, EC, EF, EE, EH,
    ST_TET, 4, P2, P5, P1, N0,
    ST_PYR, 5, EC, P2, P1, EA, N0,
    ST_PYR, 5, EC, EF, P5, P2, N0,
    ST_TET, 4, EF, EE, P5, N0,
    ST_PYR, 5, P5, EE, EH, P1, N0,
    ST_TET, 4, P1, EH, EA, N0,
    /* case 39 */ 8,
    ST_PNT, 7, P0, P1, P5, EE, EF, EG, EH,
    ST_TET, 4, P2, P1, P0, P5,
    ST_PYR, 5, EE, EF, EG, EH, N0,
    ST_PYR, 5, EH, EG, P0, P1, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_TET, 4, P5, EF, EE, N0,
    ST_PYR, 5, P5, P0, EG, EF, N0,
    ST_PYR, 5, P1, P5, EE, EH, N0,
    /* case 40 */ 1,
    ST_WDG, 6, ED, EG, P3, EE, EI, P5,
    /* case 41 */ 7,
    ST_PNT, 5, EE, ED, EA, EC, EI,
    ST_TET, 4, P3, P5, P0, N0,
    ST_PYR, 5, ED, EE, P5, P3, N0,
    ST_PYR, 5, ED, P3, P0, EA, N0,
    ST_TET, 4, EA, P0, EC, N0,
    ST_PYR, 5, P0, P5, EI, EC, N0,
    ST_TET, 4, P5, EE, EI, N0,
    /* case 42 */ 8,
    ST_PNT, 7, EB, EA, EG, EI, P1, P3, P5,
    ST_WDG, 6, ED, EE, EH, P3, P5, P1,
    ST_PYR, 5, EA, EG, EI, EB, N0,
    ST_PYR, 5, EI, EG, P3, P5, N0,
    ST_TET, 4, P1, P5, P3, N0,
    ST_TET, 4, P1, EA, EB, N0,
    ST_PYR, 5, EB, EI, P5, P1, N0,
    ST_PYR, 5, EG, EA, P1, P3, N0,
    /* case 43 */ 3,
    ST_WDG, 6, EI, EC, EB, P5, P0, P1,
    ST_TET, 4, P5, P1, P0, P3,
    ST_WDG, 6, EE, EH, ED, P5, P1, P3,
    /* case 44 */ 7,
    ST_PNT, 5, ED, EE, EB, EC, EG,
    ST_TET, 4, P5, P2, P3, N0,
    ST_PYR, 5, EE, P5, P3, ED, N0,
    ST_PYR, 5, EE, EB, P2, P5, N0,
    ST_TET, 4, EB, EC, P2, N0,
    ST_PYR, 5, P2, EC, EG, P3, N0,
    ST_TET, 4, P3, EG, ED, N0,
    /* case 45 */ 1,
    ST_HEX, 8, P5, P3, ED, EE, P2, P0, EA, EB,
    /* case 46 */ 3,
    ST_WDG, 6, P1, P5, P3, EH, EE, ED,
    ST_TET, 4, P1, P5, P3, P2,
    ST_WDG, 6, P1, P3, P2, EA, EG, EC,
    /* case 47 */ 2,
    ST_WDG, 6, EH, ED, EE, P1, P3, P5,
    ST_PYR, 5, P2, P0, P3, P5, P1,
    /* case 48 */ 1,
    ST_WDG, 6, EF, EI, P5, ED, EH, P4,
    /* case 49 */ 8,
    ST_PNT, 7, EA, EC, EI, EH, P0, P5, P4,
    ST_WDG, 6, EF, ED, EG, P5, P4, P0,
    ST_PYR, 5, EC, EI, EH, EA, N0,
    ST_PYR, 5, EH, EI, P5, P4, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_TET, 4, P0, EC, EA, N0,
    ST_PYR, 5, EA, EH, P4, P0, N0,
    ST_PYR, 5, EI, EC, P0, P5, N0,
    /* case 50 */ 7,
    ST_PNT, 5, EF, ED, EA, EB, EI,
    ST_TET, 4, P4, P1, P5, N0,
    ST_PYR, 5, ED, P4, P5, EF, N0,
    ST_PYR, 5, ED, EA, P1, P4, N0,
    ST_TET, 4, EA, EB, P1, N0,
    ST_PYR, 5, P1, EB, EI, P5, N0,
    ST_TET, 4, P5, EI, EF, N0,
    /* case 51 */ 3,
    ST_WDG, 6, P0, P4, P5, EG, ED, EF,
    ST_TET, 4, P0, P4, P5, P1,
    ST_WDG, 6, P0, P5, P1, EC, EI, EB,
    /* case 52 */ 7,
    ST_PNT, 5, ED, EF, EC, EB, EH,
    ST_TET, 4, P5, P4, P2, N0,
    ST_PYR, 5, EF, ED, P4, P5, N0,
    ST_PYR, 5, EF, P5, P2, EC, N0,
    ST_TET, 4, EC, P2, EB, N0,
    ST_PYR, 5, P2, P4, EH, EB, N0,
    ST_TET, 4, P4, ED, EH, N0,
    /* case 53 */ 3,
    ST_WDG, 6, EH, EB, EA, P4, P2, P0,
    ST_TET, 4, P4, P0, P2, P5,
    ST_WDG, 6, ED, EG, EF, P4, P0, P5,
    /* case 54 */ 1,
    ST_HEX, 8, P4, P5, EF, ED, P1, P2, EC, EA,
    /* case 55 */ 2,
    ST_WDG, 6, EG, EF, ED, P0, P5, P4,
    ST_PYR, 5, P1, P2, P5, P4, P0,
    /* case 56 */ 1,
    ST_WDG, 6, EG, EH, EI, P3, P4, P5,
    /* case 57 */ 8,
    ST_PNT, 7, P4, P5, P0, EC, EA, EH, EI,
    ST_TET, 4, P3, P4, P5, P0,
    ST_PYR, 5, EC, EI, EH, EA, N0,
    ST_PYR, 5, EI, P5, P4, EH, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_TET, 4, P0, EC, EA, N0,
    ST_PYR, 5, P0, EA, EH, P4, N0,
    ST_PYR, 5, P5, EI, EC, P0, N0,
    /* case 58 */ 8,
    ST_PNT, 7, P5, P3, P1, EA, EB, EI, EG,
    ST_TET, 4, P4, P5, P3, P1,
    ST_PYR, 5, EA, EG, EI, EB, N0,
    ST_PYR, 5, EG, P3, P5, EI, N0,
    ST_TET, 4, P1, P5, P3, N0,
    ST_TET, 4, P1, EA, EB, N0,
    ST_PYR, 5, P1, EB, EI, P5, N0,
    ST_PYR, 5, P3, EG, EA, P1, N0,
    /* case 59 */ 2,
    ST_WDG, 6, P5, P1, P0, EI, EB, EC,
    ST_PYR, 5, P3, P0, P1, P4, P5,
    /* case 60 */ 8,
    ST_PNT, 7, P3, P4, P2, EB, EC, EG, EH,
    ST_TET, 4, P5, P3, P4, P2,
    ST_PYR, 5, EB, EH, EG, EC, N0,
    ST_PYR, 5, EH, P4, P3, EG, N0,
    ST_TET, 4, P2, P3, P4, N0,
    ST_TET, 4, P2, EB, EC, N0,
    ST_PYR, 5, P2, EC, EG, P3, N0,
    ST_PYR, 5, P4, EH, EB, P2, N0,
    /* case 61 */ 2,
    ST_WDG, 6, P4, P0, P2, EH, EA, EB,
    ST_PYR, 5, P5, P2, P0, P3, P4,
    /* case 62 */ 2,
    ST_WDG, 6, P3, P2, P1, EG, EC, EA,
    ST_PYR, 5, P4, P1, P2, P5, P3,
    /* case 63 */ 1,
    ST_WDG, 6, P0, P1, P2, P3, P4, P5,
    // VTK_PYRAMID
    /* case 0 */ 0,
    /* case 1 */ 1,
    ST_TET, 4, P0, EA, ED, EE,
    /* case 2 */ 1,
    ST_TET, 4, P1, EB, EA, EF,
    /* case 3 */ 1,
    ST_WDG, 6, EB, EF, P1, ED, EE, P0,
    /* case 4 */ 1,
    ST_TET, 4, P2, EC, EB, EG,
    /* case 5 */ 2,
    ST_WDG, 6, P0, EA, EE, P2, EB, EG,
    ST_WDG, 6, ED, P0, EE, EC, P2, EG,
    /* case 6 */ 1,
    ST_WDG, 6, EC, EG, P2, EA, EF, P1,
    /* case 7 */ 2,
    ST_WDG, 6, EE, EF, EG, P0, P1, P2,
    ST_WDG, 6, P2, EC, EG, P0, ED, EE,
    /* case 8 */ 1,
    ST_TET, 4, P3, ED, EC, EH,
    /* case 9 */ 1,
    ST_WDG, 6, EA, EE, P0, EC, EH, P3,
    /* case 10 */ 2,
    ST_WDG, 6, P3, ED, EH, P1, EA, EF,
    ST_WDG, 6, EC, P3, EH, EB, P1, EF,
    /* case 11 */ 2,
    ST_WDG, 6, EH, EE, EF, P3, P0, P1,
    ST_WDG, 6, P1, EB, EF, P3, EC, EH,
    /* case 12 */ 1,
    ST_WDG, 6, ED, EH, P3, EB, EG, P2,
    /* case 13 */ 2,
    ST_WDG, 6, EG, EH, EE, P2, P3, P0,
    ST_WDG, 6, P0, EA, EE, P2, EB, EG,
    /* case 14 */ 2,
    ST_WDG, 6, EF, EG, EH, P1, P2, P3,
    ST_WDG, 6, P3, ED, EH, P1, EA, EF,
    /* case 15 */ 1,
    ST_HEX, 8, P0, P1, P2, P3, EE, EF, EG, EH,
    /* case 16 */ 1,
    ST_PYR, 5, EE, EF, EG, EH, P4,
    /* case 17 */ 2,
    ST_WDG, 6, P4, EF, EH, P0, EA, ED,
    ST_TET, 4, EF, EG, EH, P4,
    /* case 18 */ 2,
    ST_WDG, 6, P4, EG, EE, P1, EB, EA,
    ST_TET, 4, EG, EH, EE, P4,
    /* case 19 */ 7,
    ST_PNT, 7, EH, EG, ED, EB, P0, P1, P4,
    ST_PYR, 5, EG, EH, ED, EB, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_TET, 4, P0, P4, P1, N0,
    ST_TET, 4, EH, EG, P4, N0,
    ST_PYR, 5, EH, P4, P0, ED, N0,
    ST_PYR, 5, P4, EG, EB, P1, N0,
    /* case 20 */ 2,
    ST_WDG, 6, P4, EH, EF, P2, EC, EB,
    ST_TET, 4, EH, EE, EF, P4,
    /* case 21 */ 2,
    ST_WDG, 6, EA, EB, EF, P0, P2, P4,
    ST_WDG, 6, EC, ED, EH, P2, P0, P4,
    /* case 22 */ 7,
    ST_PNT, 7, EE, EH, EA, EC, P1, P2, P4,
    ST_PYR, 5, EH, EE, EA, EC, N0,
    ST_PYR, 5, EA, P1, P2, EC, N0,
    ST_TET, 4, P1, P4, P2, N0,
    ST_TET, 4, EE, EH, P4, N0,
    ST_PYR, 5, EE, P4, P1, EA, N0,
    ST_PYR, 5, P4, EH, EC, P2, N0,
    /* case 23 */ 2,
    ST_WDG, 6, P0, P2, P4, ED, EC, EH,
    ST_TET, 4, P0, P1, P2, P4,
    /* case 24 */ 2,
    ST_WDG, 6, P4, EE, EG, P3, ED, EC,
    ST_TET, 4, EE, EF, EG, P4,
    /* case 25 */ 7,
    ST_PNT, 7, EG, EF, EC, EA, P3, P0, P4,
    ST_PYR, 5, EF, EG, EC, EA, N0,
    ST_PYR, 5, EC, P3, P0, EA, N0,
    ST_TET, 4, P3, P4, P0, N0,
    ST_TET, 4, EG, EF, P4, N0,
    ST_PYR, 5, EG, P4, P3, EC, N0,
    ST_PYR, 5, P4, EF, EA, P0, N0,
    /* case 26 */ 2,
    ST_WDG, 6, ED, EA, EE, P3, P1, P4,
    ST_WDG, 6, EB, EC, EG, P1, P3, P4,
    /* case 27 */ 2,
    ST_WDG, 6, P3, P1, P4, EC, EB, EG,
    ST_TET, 4, P3, P0, P1, P4,
    /* case 28 */ 7,
    ST_PNT, 7, EF, EE, EB, ED, P2, P3, P4,
    ST_PYR, 5, EE, EF, EB, ED, N0,
    ST_PYR, 5, EB, P2, P3, ED, N0,
    ST_TET, 4, P2, P4, P3, N0,
    ST_TET, 4, EF, EE, P4, N0,
    ST_PYR, 5, EF, P4, P2, EB, N0,
    ST_PYR, 5, P4, EE, ED, P3, N0,
    /* case 29 */ 2,
    ST_WDG, 6, P2, P0, P4, EB, EA, EF,
    ST_TET, 4, P2, P3, P0, P4,
    /* case 30 */ 2,
    ST_WDG, 6, P1, P3, P4, EA, ED, EE,
    ST_TET, 4, P1, P2, P3, P4,
    /* case 31 */ 1,
    ST_PYR, 5, P0, P1, P2, P3, P4,
  };

  // offset into CellCases for each cell
  static constexpr uint16_t StartCellCases[] = {
    // VTK_VERTEX
    0, 1, // case 0 - 1
    // VTK_LINE
    5, 6, 11, 16, // case 0 - 3
    // VTK_TRIANGLE
    21, 22, 28, 34, 41, 47, 54, 61, // case 0 - 7
    // VTK_PIXEL
    67, 68, 74, 80, 87, 93, 100, 113, // case 0 - 7
    125, 131, 144, 151, 163, 170, 182, 194, // case 8 - 15
    // VTK_QUAD
    201, 202, 208, 214, 221, 227, 240, 247, // case 0 - 7
    259, 265, 272, 285, 297, 304, 316, 328, // case 8 - 15
    // VTK_TETRA
    335, 336, 343, 350, 359, 366, 375, 384, // case 0 - 7
    393, 400, 409, 418, 427, 436, 445, 454, // case 8 - 15
    // VTK_VOXEL
    461, 462, 469, 476, 485, 492, 501, 544, // case 0 - 7
    591, 598, 641, 650, 697, 706, 753, 800, // case 8 - 15
    811, 818, 827, 870, 917, 960, 1007, 1044, // case 16 - 23
    1079, 1092, 1145, 1198, 1259, 1312, 1373, 1435, // case 24 - 31
    1496, 1503, 1546, 1555, 1602, 1615, 1668, 1721, // case 32 - 39
    1782, 1825, 1862, 1909, 1944, 1997, 2059, 2120, // case 40 - 47
    2181, 2190, 2237, 2284, 2295, 2348, 2409, 2471, // case 48 - 55
    2532, 2585, 2647, 2708, 2769, 2790, 2866, 2942, // case 56 - 63
    2961, 2968, 3011, 3024, 3077, 3086, 3133, 3186, // case 64 - 71
    3247, 3290, 3327, 3380, 3442, 3489, 3524, 3585, // case 72 - 79
    3646, 3655, 3702, 3755, 3816, 3863, 3874, 3936, // case 80 - 87
    3997, 4050, 4112, 4133, 4209, 4270, 4331, 4407, // case 88 - 95
    4426, 4469, 4506, 4559, 4621, 4674, 4736, 4757, // case 96 - 103
    4833, 4870, 4909, 4971, 5008, 5070, 5107, 5183, // case 104 - 111
    5219, 5266, 5301, 5362, 5423, 5484, 5545, 5621, // case 112 - 119
    5640, 5702, 5739, 5815, 5851, 5927, 5963, 6004, // case 120 - 127
    6067, 6074, 6087, 6130, 6183, 6226, 6279, 6316, // case 128 - 135
    6378, 6387, 6440, 6487, 6548, 6595, 6656, 6691, // case 136 - 143
    6752, 6795, 6848, 6885, 6947, 6984, 7046, 7085, // case 144 - 151
    7122, 7175, 7196, 7258, 7334, 7396, 7472, 7509, // case 152 - 159
    7545, 7554, 7607, 7654, 7715, 7768, 7789, 7851, // case 160 - 167
    7927, 7974, 8036, 8047, 8108, 8169, 8245, 8306, // case 168 - 175
    8325, 8372, 8433, 8468, 8529, 8591, 8667, 8704, // case 176 - 183
    8740, 8801, 8877, 8938, 8957, 9033, 9074, 9110, // case 184 - 191
    9173, 9182, 9235, 9288, 9309, 9356, 9417, 9479, // case 192 - 199
    9555, 9602, 9664, 9725, 9801, 9812, 9873, 9934, // case 200 - 207
    9953, 10000, 10061, 10123, 10199, 10234, 10295, 10332, // case 208 - 215
    10368, 10429, 10505, 10581, 10622, 10683, 10702, 10738, // case 216 - 223
    10801, 10848, 10910, 10971, 11047, 11108, 11184, 11260, // case 224 - 231
    11301, 11336, 11373, 11434, 11470, 11531, 11567, 11586, // case 232 - 239
    11649, 11660, 11721, 11782, 11801, 11862, 11881, 11917, // case 240 - 247
    11980, 12041, 12077, 12096, 12159, 12178, 12241, 12304, // case 248 - 255
    // VTK_HEXAHEDRON
    12315, 12316, 12323, 12330, 12339, 12346, 12389, 12398, // case 0 - 7
    12445, 12452, 12461, 12504, 12551, 12560, 12607, 12654, // case 8 - 15
    12665, 12672, 12681, 12724, 12771, 12784, 12837, 12890, // case 16 - 23
    12951, 12994, 13041, 13078, 13113, 13166, 13227, 13289, // case 24 - 31
    13350, 13357, 13400, 13409, 13456, 13499, 13536, 13583, // case 32 - 39
    13618, 13631, 13684, 13737, 13798, 13851, 13913, 13974, // case 40 - 47
    14035, 14044, 14091, 14138, 14149, 14202, 14264, 14325, // case 48 - 55
    14386, 14439, 14500, 14562, 14623, 14644, 14720, 14796, // case 56 - 63
    14815, 14822, 14835, 14878, 14931, 14940, 14993, 15040, // case 64 - 71
    15101, 15144, 15197, 15234, 15296, 15343, 15404, 15439, // case 72 - 79
    15500, 15543, 15596, 15633, 15695, 15748, 15769, 15831, // case 80 - 87
    15907, 15944, 16006, 16045, 16082, 16144, 16220, 16257, // case 88 - 95
    16293, 16302, 16355, 16402, 16463, 16510, 16572, 16583, // case 96 - 103
    16644, 16697, 16718, 16780, 16856, 16917, 16993, 17054, // case 104 - 111
    17073, 17120, 17181, 17216, 17277, 17338, 17414, 17475, // case 112 - 119
    17494, 17556, 17632, 17669, 17705, 17781, 17822, 17858, // case 120 - 127
    17921, 17928, 17971, 17984, 18037, 18080, 18117, 18170, // case 128 - 135
    18232, 18241, 18288, 18341, 18402, 18449, 18484, 18545, // case 136 - 143
    18606, 18615, 18662, 18715, 18776, 18829, 18891, 18912, // case 144 - 151
    18988, 19035, 19046, 19108, 19169, 19230, 19291, 19367, // case 152 - 159
    19386, 19429, 19466, 19519, 19581, 19618, 19657, 19719, // case 160 - 167
    19756, 19809, 19871, 19892, 19968, 20030, 20067, 20143, // case 168 - 175
    20179, 20226, 20261, 20322, 20383, 20445, 20482, 20558, // case 176 - 183
    20594, 20655, 20716, 20792, 20811, 20887, 20923, 20964, // case 184 - 191
    21027, 21036, 21089, 21142, 21163, 21210, 21272, 21333, // case 192 - 199
    21409, 21456, 21517, 21579, 21655, 21666, 21727, 21788, // case 200 - 207
    21807, 21854, 21915, 21977, 22053, 22114, 22190, 22266, // case 208 - 215
    22307, 22342, 22403, 22440, 22476, 22537, 22556, 22592, // case 216 - 223
    22655, 22702, 22764, 22825, 22901, 22936, 22973, 23034, // case 224 - 231
    23070, 23131, 23207, 23283, 23324, 23385, 23421, 23440, // case 232 - 239
    23503, 23514, 23575, 23636, 23655, 23716, 23752, 23771, // case 240 - 247
    23834, 23895, 23914, 23950, 24013, 24032, 24095, 24158, // case 248 - 255
    // VTK_WEDGE
    24169, 24170, 24177, 24184, 24193, 24200, 24209, 24218, // case 0 - 7
    24227, 24234, 24243, 24290, 24337, 24384, 24431, 24489, // case 8 - 15
    24545, 24552, 24599, 24608, 24655, 24702, 24760, 24807, // case 16 - 23
    24863, 24872, 24919, 24966, 24977, 25035, 25058, 25081, // case 24 - 31
    25097, 25104, 25151, 25198, 25256, 25265, 25312, 25359, // case 32 - 39
    25415, 25424, 25471, 25529, 25552, 25599, 25610, 25633, // case 40 - 47
    25649, 25658, 25716, 25763, 25786, 25833, 25856, 25867, // case 48 - 55
    25883, 25892, 25948, 26004, 26020, 26076, 26092, 26108, // case 56 - 63
    // VTK_PYRAMID
    26117, 26118, 26125, 26132, 26141, 26148, 26165, 26174, // case 0 - 7
    26191, 26198, 26207, 26224, 26241, 26250, 26267, 26284, // case 8 - 15
    26295, 26303, 26318, 26333, 26383, 26398, 26415, 26465, // case 16 - 23
    26480, 26495, 26545, 26562, 26577, 26627, 26642, 26657, // case 24 - 31
  };

  //  clip table for all cases of each cell
  static constexpr uint8_t CellCasesInsideOut[] = {
    // VTK_VERTEX
    /* case 0 */ 1,
    ST_VTX, 1, P0,
    /* case 1 */ 0,
    // VTK_LINE
    /* case 0 */ 1,
    ST_LIN, 2, P0, P1,
    /* case 1 */ 1,
    ST_LIN, 2, EA, P1,
    /* case 2 */ 1,
    ST_LIN, 2, EA, P0,
    /* case 3 */ 0,
    // VTK_TRIANGLE
    /* case 0 */ 1,
    ST_TRI, 3, P0, P1, P2,
    /* case 1 */ 1,
    ST_QUA, 4, P1, P2, EC, EA,
    /* case 2 */ 1,
    ST_QUA, 4, P2, P0, EA, EB,
    /* case 3 */ 1,
    ST_TRI, 3, EC, EB, P2,
    /* case 4 */ 1,
    ST_QUA, 4, P0, P1, EB, EC,
    /* case 5 */ 1,
    ST_TRI, 3, EB, EA, P1,
    /* case 6 */ 1,
    ST_TRI, 3, EA, EC, P0,
    /* case 7 */ 0,
    // VTK_PIXEL
    /* case 0 */ 1,
    ST_QUA, 4, P0, P1, P3, P2,
    /* case 1 */ 2,
    ST_QUA, 4, ED, EA, P1, P2,
    ST_TRI, 3, P2, P1, P3,
    /* case 2 */ 2,
    ST_QUA, 4, EA, EB, P3, P0,
    ST_TRI, 3, P0, P3, P2,
    /* case 3 */ 1,
    ST_QUA, 4, ED, EB, P3, P2,
    /* case 4 */ 2,
    ST_QUA, 4, EC, ED, P0, P3,
    ST_TRI, 3, P3, P0, P1,
    /* case 5 */ 1,
    ST_QUA, 4, EC, EA, P1, P3,
    /* case 6 */ 2,
    ST_TRI, 3, EA, ED, P0,
    ST_TRI, 3, EC, EB, P3,
    /* case 7 */ 1,
    ST_TRI, 3, EC, EB, P3,
    /* case 8 */ 2,
    ST_QUA, 4, EB, EC, P2, P1,
    ST_TRI, 3, P1, P2, P0,
    /* case 9 */ 2,
    ST_TRI, 3, EB, EA, P1,
    ST_TRI, 3, ED, EC, P2,
    /* case 10 */ 1,
    ST_QUA, 4, EA, EC, P2, P0,
    /* case 11 */ 1,
    ST_TRI, 3, ED, EC, P2,
    /* case 12 */ 1,
    ST_QUA, 4, EB, ED, P0, P1,
    /* case 13 */ 1,
    ST_TRI, 3, EB, EA, P1,
    /* case 14 */ 1,
    ST_TRI, 3, EA, ED, P0,
    /* case 15 */ 0,
    // VTK_QUAD
    /* case 0 */ 1,
    ST_QUA, 4, P0, P1, P2, P3,
    /* case 1 */ 2,
    ST_QUA, 4, ED, EA, P1, P3,
    ST_TRI, 3, P3, P1, P2,
    /* case 2 */ 2,
    ST_QUA, 4, EA, EB, P2, P0,
    ST_TRI, 3, P0, P2, P3,
    /* case 3 */ 1,
    ST_QUA, 4, ED, EB, P2, P3,
    /* case 4 */ 2,
    ST_QUA, 4, EB, EC, P3, P1,
    ST_TRI, 3, P1, P3, P0,
    /* case 5 */ 2,
    ST_TRI, 3, ED, EC, P3,
    ST_TRI, 3, EB, EA, P1,
    /* case 6 */ 1,
    ST_QUA, 4, EA, EC, P3, P0,
    /* case 7 */ 1,
    ST_TRI, 3, ED, EC, P3,
    /* case 8 */ 2,
    ST_QUA, 4, EC, ED, P0, P2,
    ST_TRI, 3, P2, P0, P1,
    /* case 9 */ 1,
    ST_QUA, 4, EC, EA, P1, P2,
    /* case 10 */ 2,
    ST_TRI, 3, EA, ED, P0,
    ST_TRI, 3, EC, EB, P2,
    /* case 11 */ 1,
    ST_TRI, 3, EC, EB, P2,
    /* case 12 */ 1,
    ST_QUA, 4, EB, ED, P0, P1,
    /* case 13 */ 1,
    ST_TRI, 3, EB, EA, P1,
    /* case 14 */ 1,
    ST_TRI, 3, EA, ED, P0,
    /* case 15 */ 0,
    // VTK_TETRA
    /* case 0 */ 1,
    ST_TET, 4, P0, P1, P2, P3,
    /* case 1 */ 1,
    ST_WDG, 6, EA, ED, EC, P1, P3, P2,
    /* case 2 */ 1,
    ST_WDG, 6, P0, P3, P2, EA, EE, EB,
    /* case 3 */ 1,
    ST_WDG, 6, P3, ED, EE, P2, EC, EB,
    /* case 4 */ 1,
    ST_WDG, 6, EC, EF, EB, P0, P3, P1,
    /* case 5 */ 1,
    ST_WDG, 6, P1, EA, EB, P3, ED, EF,
    /* case 6 */ 1,
    ST_WDG, 6, P3, EE, EF, P0, EA, EC,
    /* case 7 */ 1,
    ST_TET, 4, ED, EE, EF, P3,
    /* case 8 */ 1,
    ST_WDG, 6, P0, P2, P1, ED, EF, EE,
    /* case 9 */ 1,
    ST_WDG, 6, P2, EC, EF, P1, EA, EE,
    /* case 10 */ 1,
    ST_WDG, 6, P0, EA, ED, P2, EB, EF,
    /* case 11 */ 1,
    ST_TET, 4, EC, EF, EB, P2,
    /* case 12 */ 1,
    ST_WDG, 6, P1, EB, EE, P0, EC, ED,
    /* case 13 */ 1,
    ST_TET, 4, EA, EB, EE, P1,
    /* case 14 */ 1,
    ST_TET, 4, EA, ED, EC, P0,
    /* case 15 */ 0,
    // VTK_VOXEL
    /* case 0 */ 1,
    ST_HEX, 8, P0, P1, P3, P2, P4, P5, P7, P6,
    /* case 1 */ 9,
    ST_PNT, 7, P1, P3, P2, P4, P5, P7, P6,
    ST_WDG, 6, P1, P2, P4, EA, ED, EI,
    ST_TET, 4, P1, P2, P4, N0,
    ST_TET, 4, P1, P3, P2, N0,
    ST_PYR, 5, P7, P6, P2, P3, N0,
    ST_PYR, 5, P5, P7, P3, P1, N0,
    ST_PYR, 5, P4, P6, P7, P5, N0,
    ST_TET, 4, P2, P6, P4, N0,
    ST_TET, 4, P4, P5, P1, N0,
    /* case 2 */ 9,
    ST_PNT, 7, P5, P4, P0, P3, P7, P6, P2,
    ST_WDG, 6, EJ, EA, EB, P5, P0, P3,
    ST_TET, 4, P5, P3, P0, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_PYR, 5, P6, P4, P0, P2, N0,
    ST_PYR, 5, P7, P5, P4, P6, N0,
    ST_PYR, 5, P3, P7, P6, P2, N0,
    ST_TET, 4, P0, P3, P2, N0,
    ST_TET, 4, P3, P5, P7, N0,
    /* case 3 */ 2,
    ST_HEX, 8, EB, P3, P2, ED, EJ, P5, P4, EI,
    ST_WDG, 6, P3, P7, P5, P2, P6, P4,
    /* case 4 */ 9,
    ST_PNT, 7, P3, P1, P0, P6, P7, P5, P4,
    ST_WDG, 6, EC, ED, EK, P3, P0, P6,
    ST_TET, 4, P3, P6, P0, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_PYR, 5, P5, P1, P0, P4, N0,
    ST_PYR, 5, P7, P3, P1, P5, N0,
    ST_PYR, 5, P6, P7, P5, P4, N0,
    ST_TET, 4, P0, P6, P4, N0,
    ST_TET, 4, P6, P3, P7, N0,
    /* case 5 */ 2,
    ST_HEX, 8, EK, P6, P4, EI, EC, P3, P1, EA,
    ST_WDG, 6, P6, P7, P3, P4, P5, P1,
    /* case 6 */ 12,
    ST_PNT, 2, EK, EJ,
    ST_PYR, 5, P6, P7, P5, P4, N0,
    ST_TET, 4, P7, P3, P5, N0,
    ST_TET, 4, P6, P3, P7, N0,
    ST_TET, 4, P0, P6, P4, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_PYR, 5, P5, EJ, EA, P0, N0,
    ST_PYR, 5, P3, EB, EJ, P5, N0,
    ST_TET, 4, P3, EC, EB, N0,
    ST_PYR, 5, P6, EK, EC, P3, N0,
    ST_PYR, 5, P6, P0, ED, EK, N0,
    ST_TET, 4, P0, EA, ED, N0,
    /* case 7 */ 9,
    ST_PNT, 5, EJ, EI, EB, EC, EK,
    ST_PYR, 5, P5, P4, P6, P7, N0,
    ST_TET, 4, P6, N0, P3, P7,
    ST_PYR, 5, P4, EI, EK, P6, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_TET, 4, P3, P5, P7, N0,
    ST_PYR, 5, P3, EB, EJ, P5, N0,
    ST_TET, 4, P3, EC, EB, N0,
    ST_PYR, 5, EK, EC, P3, P6, N0,
    /* case 8 */ 9,
    ST_PNT, 7, P7, P5, P1, P2, P6, P4, P0,
    ST_WDG, 6, EL, EB, EC, P7, P1, P2,
    ST_TET, 4, P7, P2, P1, N0,
    ST_TET, 4, P7, P1, P5, N0,
    ST_PYR, 5, P4, P5, P1, P0, N0,
    ST_PYR, 5, P6, P7, P5, P4, N0,
    ST_PYR, 5, P2, P6, P4, P0, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_TET, 4, P2, P7, P6, N0,
    /* case 9 */ 12,
    ST_PNT, 2, EI, EL,
    ST_PYR, 5, P4, P6, P7, P5, N0,
    ST_TET, 4, P5, P7, P1, N0,
    ST_TET, 4, P4, P5, P1, N0,
    ST_TET, 4, P2, P6, P4, N0,
    ST_TET, 4, P7, P6, P2, N0,
    ST_PYR, 5, P7, P2, EC, EL, N0,
    ST_PYR, 5, P1, P7, EL, EB, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_PYR, 5, P4, EI, ED, P2, N0,
    ST_TET, 4, P2, ED, EC, N0,
    /* case 10 */ 2,
    ST_HEX, 8, EC, P2, P0, EA, EL, P7, P5, EJ,
    ST_WDG, 6, P2, P6, P7, P0, P4, P5,
    /* case 11 */ 9,
    ST_PNT, 5, EL, EJ, EC, ED, EI,
    ST_PYR, 5, P7, P5, P4, P6, N0,
    ST_TET, 4, P4, N0, P2, P6,
    ST_PYR, 5, P5, EJ, EI, P4, N0,
    ST_PYR, 5, EL, EJ, P5, P7, N0,
    ST_TET, 4, P2, P7, P6, N0,
    ST_PYR, 5, P2, EC, EL, P7, N0,
    ST_TET, 4, P2, ED, EC, N0,
    ST_PYR, 5, EI, ED, P2, P4, N0,
    /* case 12 */ 2,
    ST_HEX, 8, EL, P7, P6, EK, EB, P1, P0, ED,
    ST_WDG, 6, P0, P4, P6, P1, P5, P7,
    /* case 13 */ 9,
    ST_PNT, 5, EL, EK, EB, EA, EI,
    ST_PYR, 5, P7, P5, P4, P6, N0,
    ST_TET, 4, P4, P1, N0, P5,
    ST_PYR, 5, P6, P4, EI, EK, N0,
    ST_PYR, 5, EL, P7, P6, EK, N0,
    ST_TET, 4, P1, P5, P7, N0,
    ST_PYR, 5, P1, P7, EL, EB, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    /* case 14 */ 9,
    ST_PNT, 5, EK, EL, ED, EA, EJ,
    ST_PYR, 5, P6, P7, P5, P4, N0,
    ST_TET, 4, P5, N0, P0, P4,
    ST_PYR, 5, P7, EL, EJ, P5, N0,
    ST_PYR, 5, EK, EL, P7, P6, N0,
    ST_TET, 4, P0, P6, P4, N0,
    ST_PYR, 5, P0, ED, EK, P6, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_PYR, 5, EJ, EA, P0, P5, N0,
    /* case 15 */ 1,
    ST_HEX, 8, EI, EJ, EL, EK, P4, P5, P7, P6,
    /* case 16 */ 9,
    ST_PNT, 7, P5, P1, P0, P6, P7, P3, P2,
    ST_WDG, 6, P5, P0, P6, EE, EI, EH,
    ST_TET, 4, P5, P0, P6, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_PYR, 5, P3, P2, P0, P1, N0,
    ST_PYR, 5, P7, P3, P1, P5, N0,
    ST_PYR, 5, P6, P2, P3, P7, N0,
    ST_TET, 4, P0, P2, P6, N0,
    ST_TET, 4, P6, P7, P5, N0,
    /* case 17 */ 2,
    ST_HEX, 8, EE, P5, P1, EA, EH, P6, P2, ED,
    ST_WDG, 6, P2, P3, P1, P6, P7, P5,
    /* case 18 */ 12,
    ST_PNT, 2, EH, EB,
    ST_PYR, 5, P6, P2, P3, P7, N0,
    ST_TET, 4, P7, P3, P5, N0,
    ST_TET, 4, P6, P7, P5, N0,
    ST_TET, 4, P0, P2, P6, N0,
    ST_TET, 4, P3, P2, P0, N0,
    ST_PYR, 5, P3, P0, EA, EB, N0,
    ST_PYR, 5, P5, P3, EB, EJ, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_PYR, 5, P6, P5, EE, EH, N0,
    ST_PYR, 5, P6, EH, EI, P0, N0,
    ST_TET, 4, P0, EI, EA, N0,
    /* case 19 */ 9,
    ST_PNT, 5, EB, ED, EJ, EE, EH,
    ST_PYR, 5, P3, P7, P6, P2, N0,
    ST_TET, 4, P6, P5, N0, P7,
    ST_PYR, 5, P2, P6, EH, ED, N0,
    ST_PYR, 5, EB, P3, P2, ED, N0,
    ST_TET, 4, P5, P7, P3, N0,
    ST_PYR, 5, P5, P3, EB, EJ, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_PYR, 5, EH, P6, P5, EE, N0,
    /* case 20 */ 12,
    ST_PNT, 2, EC, EE,
    ST_PYR, 5, P3, P1, P5, P7, N0,
    ST_TET, 4, P7, P5, P6, N0,
    ST_TET, 4, P3, P7, P6, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_PYR, 5, P5, P0, EI, EE, N0,
    ST_PYR, 5, P6, P5, EE, EH, N0,
    ST_TET, 4, P6, EH, EK, N0,
    ST_PYR, 5, P3, P6, EK, EC, N0,
    ST_PYR, 5, P3, EC, ED, P0, N0,
    ST_TET, 4, P0, ED, EI, N0,
    /* case 21 */ 9,
    ST_PNT, 5, EE, EA, EH, EK, EC,
    ST_PYR, 5, P5, P7, P3, P1, N0,
    ST_TET, 4, P3, P6, N0, P7,
    ST_PYR, 5, P1, P3, EC, EA, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_TET, 4, P6, P7, P5, N0,
    ST_PYR, 5, P6, P5, EE, EH, N0,
    ST_TET, 4, P6, EH, EK, N0,
    ST_PYR, 5, EC, P3, P6, EK, N0,
    /* case 22 */ 6,
    ST_TET, 4, P0, EA, ED, EI,
    ST_TET, 4, P5, P6, P7, P3,
    ST_PYR, 5, EC, P3, P6, EK, EH,
    ST_PYR, 5, EB, EJ, P5, P3, EE,
    ST_PYR, 5, P6, P5, EE, EH, P3,
    ST_PYR, 5, EH, EE, EB, EC, P3,
    /* case 23 */ 5,
    ST_TET, 4, P5, P6, P7, P3,
    ST_PYR, 5, EC, P3, P6, EK, EH,
    ST_PYR, 5, EB, EJ, P5, P3, EE,
    ST_PYR, 5, P6, P5, EE, EH, P3,
    ST_PYR, 5, EH, EE, EB, EC, P3,
    /* case 24 */ 6,
    ST_WDG, 6, EB, EC, EL, P1, P2, P7,
    ST_WDG, 6, P0, P6, P5, EI, EH, EE,
    ST_TET, 4, P2, P1, P7, P6,
    ST_TET, 4, P5, P6, P7, P1,
    ST_TET, 4, P0, P5, P1, P6,
    ST_TET, 4, P2, P6, P0, P1,
    /* case 25 */ 11,
    ST_PNT, 4, EE, EH, EL, EL,
    ST_PYR, 5, P7, P2, EC, EL, N0,
    ST_TET, 4, EC, P2, ED, N0,
    ST_PYR, 5, P6, EH, ED, P2, N0,
    ST_TET, 4, P7, P6, P2, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_TET, 4, P5, P7, P1, N0,
    ST_PYR, 5, P1, P7, EL, EB, N0,
    ST_TET, 4, P5, P6, P7, N0,
    ST_PYR, 5, P5, EE, EH, P6, N0,
    ST_PYR, 5, P5, P1, EA, EE, N0,
    /* case 26 */ 11,
    ST_PNT, 4, EL, EC, EH, EH,
    ST_PYR, 5, P6, EH, EI, P0, N0,
    ST_TET, 4, EI, EA, P0, N0,
    ST_PYR, 5, P2, P0, EA, EC, N0,
    ST_TET, 4, P6, P0, P2, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_TET, 4, P7, P5, P6, N0,
    ST_PYR, 5, P5, EE, EH, P6, N0,
    ST_TET, 4, P7, P6, P2, N0,
    ST_PYR, 5, P7, P2, EC, EL, N0,
    ST_PYR, 5, P7, EL, EJ, P5, N0,
    /* case 27 */ 9,
    ST_PNT, 6, ED, EC, EL, EJ, EE, EH,
    ST_TET, 4, P7, P5, P6, N0,
    ST_PYR, 5, P7, EL, EJ, P5, N0,
    ST_TET, 4, P2, P7, P6, N0,
    ST_PYR, 5, P2, EC, EL, P7, N0,
    ST_TET, 4, ED, EC, P2, N0,
    ST_PYR, 5, P2, P6, EH, ED, N0,
    ST_PYR, 5, EH, P6, P5, EE, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    /* case 28 */ 11,
    ST_PNT, 4, EL, EB, EE, EE,
    ST_PYR, 5, P5, P0, EI, EE, N0,
    ST_TET, 4, EI, P0, ED, N0,
    ST_PYR, 5, P1, EB, ED, P0, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_TET, 4, P6, EH, EK, N0,
    ST_TET, 4, P7, P5, P6, N0,
    ST_PYR, 5, P6, P5, EE, EH, N0,
    ST_TET, 4, P7, P1, P5, N0,
    ST_PYR, 5, P7, EL, EB, P1, N0,
    ST_PYR, 5, P7, P6, EK, EL, N0,
    /* case 29 */ 9,
    ST_PNT, 6, EA, EB, EL, EK, EH, EE,
    ST_TET, 4, P7, P5, P6, N0,
    ST_PYR, 5, P7, P6, EK, EL, N0,
    ST_TET, 4, P1, P5, P7, N0,
    ST_PYR, 5, P1, P7, EL, EB, N0,
    ST_TET, 4, EA, P1, EB, N0,
    ST_PYR, 5, P1, EA, EE, P5, N0,
    ST_PYR, 5, EE, EH, P6, P5, N0,
    ST_TET, 4, P6, EH, EK, N0,
    /* case 30 */ 8,
    ST_PNT, 5, EL, EJ, EK, EH, EE,
    ST_TET, 4, P0, EA, ED, EI,
    ST_PYR, 5, P5, P7, EL, EJ, N0,
    ST_PYR, 5, P7, P6, EK, EL, N0,
    ST_TET, 4, P6, EH, EK, N0,
    ST_TET, 4, P7, P5, P6, N0,
    ST_PYR, 5, P6, P5, EE, EH, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    /* case 31 */ 7,
    ST_PNT, 5, EJ, EL, EK, EE, EH,
    ST_PYR, 5, P7, P6, EK, EL, N0,
    ST_TET, 4, P6, EH, EK, N0,
    ST_PYR, 5, P5, P7, EL, EJ, N0,
    ST_TET, 4, EE, P5, EJ, N0,
    ST_PYR, 5, EH, P6, P5, EE, N0,
    ST_TET, 4, P6, P7, P5, N0,
    /* case 32 */ 9,
    ST_PNT, 7, P7, P3, P1, P4, P6, P2, P0,
    ST_WDG, 6, P7, P1, P4, EF, EJ, EE,
    ST_TET, 4, P7, P1, P4, N0,
    ST_TET, 4, P7, P3, P1, N0,
    ST_PYR, 5, P2, P0, P1, P3, N0,
    ST_PYR, 5, P6, P2, P3, P7, N0,
    ST_PYR, 5, P4, P0, P2, P6, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_TET, 4, P4, P6, P7, N0,
    /* case 33 */ 12,
    ST_PNT, 2, ED, EF,
    ST_PYR, 5, P2, P3, P7, P6, N0,
    ST_TET, 4, P3, P1, P7, N0,
    ST_TET, 4, P2, P1, P3, N0,
    ST_TET, 4, P4, P2, P6, N0,
    ST_TET, 4, P7, P4, P6, N0,
    ST_PYR, 5, P7, EF, EE, P4, N0,
    ST_PYR, 5, P1, EJ, EF, P7, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_PYR, 5, P2, ED, EA, P1, N0,
    ST_PYR, 5, P2, P4, EI, ED, N0,
    ST_TET, 4, P4, EE, EI, N0,
    /* case 34 */ 2,
    ST_HEX, 8, EF, P7, P3, EB, EE, P4, P0, EA,
    ST_WDG, 6, P0, P2, P3, P4, P6, P7,
    /* case 35 */ 9,
    ST_PNT, 5, EF, EB, EE, EI, ED,
    ST_PYR, 5, P7, P6, P2, P3, N0,
    ST_TET, 4, P2, P4, N0, P6,
    ST_PYR, 5, P3, P2, ED, EB, N0,
    ST_PYR, 5, EF, P7, P3, EB, N0,
    ST_TET, 4, P4, P6, P7, N0,
    ST_PYR, 5, P4, P7, EF, EE, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_PYR, 5, ED, P2, P4, EI, N0,
    /* case 36 */ 6,
    ST_WDG, 6, P1, P4, P7, EJ, EE, EF,
    ST_WDG, 6, ED, EK, EC, P0, P6, P3,
    ST_TET, 4, P4, P7, P1, P6,
    ST_TET, 4, P3, P7, P6, P1,
    ST_TET, 4, P0, P1, P3, P6,
    ST_TET, 4, P4, P0, P6, P1,
    /* case 37 */ 11,
    ST_PNT, 4, EC, EK, EF, EF,
    ST_PYR, 5, P7, EF, EE, P4, N0,
    ST_TET, 4, EE, EI, P4, N0,
    ST_PYR, 5, P6, P4, EI, EK, N0,
    ST_TET, 4, P7, P4, P6, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_TET, 4, P3, P1, P7, N0,
    ST_PYR, 5, P1, EJ, EF, P7, N0,
    ST_TET, 4, P3, P7, P6, N0,
    ST_PYR, 5, P3, P6, EK, EC, N0,
    ST_PYR, 5, P3, EC, EA, P1, N0,
    /* case 38 */ 11,
    ST_PNT, 4, EF, EE, EK, EK,
    ST_PYR, 5, P6, P0, ED, EK, N0,
    ST_TET, 4, ED, P0, EA, N0,
    ST_PYR, 5, P4, EE, EA, P0, N0,
    ST_TET, 4, P6, P4, P0, N0,
    ST_TET, 4, P3, EC, EB, N0,
    ST_TET, 4, P7, P6, P3, N0,
    ST_PYR, 5, P3, P6, EK, EC, N0,
    ST_TET, 4, P7, P4, P6, N0,
    ST_PYR, 5, P7, EF, EE, P4, N0,
    ST_PYR, 5, P7, P3, EB, EF, N0,
    /* case 39 */ 9,
    ST_PNT, 6, EI, EE, EF, EB, EC, EK,
    ST_TET, 4, P7, P6, P3, N0,
    ST_PYR, 5, P7, P3, EB, EF, N0,
    ST_TET, 4, P4, P6, P7, N0,
    ST_PYR, 5, P4, P7, EF, EE, N0,
    ST_TET, 4, EI, P4, EE, N0,
    ST_PYR, 5, P4, EI, EK, P6, N0,
    ST_PYR, 5, EK, EC, P3, P6, N0,
    ST_TET, 4, P3, EC, EB, N0,
    /* case 40 */ 12,
    ST_PNT, 2, EC, EE,
    ST_PYR, 5, P2, P6, P4, P0, N0,
    ST_TET, 4, P6, P7, P4, N0,
    ST_TET, 4, P2, P7, P6, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_PYR, 5, P7, EF, EE, P4, N0,
    ST_TET, 4, P7, EL, EF, N0,
    ST_PYR, 5, P2, EC, EL, P7, N0,
    ST_PYR, 5, P2, P1, EB, EC, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    /* case 41 */ 6,
    ST_TET, 4, P1, EA, EJ, EB,
    ST_TET, 4, P7, P6, P2, P4,
    ST_PYR, 5, EI, ED, P2, P4, EC,
    ST_PYR, 5, EE, P4, P7, EF, EL,
    ST_PYR, 5, P2, EC, EL, P7, P4,
    ST_PYR, 5, EC, EI, EE, EL, P4,
    /* case 42 */ 9,
    ST_PNT, 5, EE, EA, EF, EL, EC,
    ST_PYR, 5, P4, P0, P2, P6, N0,
    ST_TET, 4, P2, N0, P7, P6,
    ST_PYR, 5, P0, EA, EC, P2, N0,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_TET, 4, P7, P4, P6, N0,
    ST_PYR, 5, P7, EF, EE, P4, N0,
    ST_TET, 4, P7, EL, EF, N0,
    ST_PYR, 5, EC, EL, P7, P2, N0,
    /* case 43 */ 5,
    ST_TET, 4, P7, P6, P2, P4,
    ST_PYR, 5, EI, ED, P2, P4, EC,
    ST_PYR, 5, EE, P4, P7, EF, EL,
    ST_PYR, 5, P2, EC, EL, P7, P4,
    ST_PYR, 5, EC, EI, EE, EL, P4,
    /* case 44 */ 11,
    ST_PNT, 4, EK, ED, EE, EE,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    ST_PYR, 5, P0, P1, EB, ED, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_TET, 4, P7, EL, EF, N0,
    ST_TET, 4, P6, P7, P4, N0,
    ST_PYR, 5, P7, EF, EE, P4, N0,
    ST_TET, 4, P6, P4, P0, N0,
    ST_PYR, 5, P6, P0, ED, EK, N0,
    ST_PYR, 5, P6, EK, EL, P7, N0,
    /* case 45 */ 8,
    ST_PNT, 5, EK, EL, EI, EE, EF,
    ST_TET, 4, P1, EB, EA, EJ,
    ST_PYR, 5, P7, P6, EK, EL, N0,
    ST_PYR, 5, P6, P4, EI, EK, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_TET, 4, P6, P7, P4, N0,
    ST_PYR, 5, P4, P7, EF, EE, N0,
    ST_TET, 4, P7, EL, EF, N0,
    /* case 46 */ 9,
    ST_PNT, 6, EA, ED, EK, EL, EF, EE,
    ST_TET, 4, P6, P7, P4, N0,
    ST_PYR, 5, P6, EK, EL, P7, N0,
    ST_TET, 4, P0, P6, P4, N0,
    ST_PYR, 5, P0, ED, EK, P6, N0,
    ST_TET, 4, EA, ED, P0, N0,
    ST_PYR, 5, P0, P4, EE, EA, N0,
    ST_PYR, 5, EE, P4, P7, EF, N0,
    ST_TET, 4, P7, EL, EF, N0,
    /* case 47 */ 7,
    ST_PNT, 5, EL, EK, EI, EF, EE,
    ST_PYR, 5, P6, P4, EI, EK, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_PYR, 5, P7, P6, EK, EL, N0,
    ST_TET, 4, EF, P7, EL, N0,
    ST_PYR, 5, EE, P4, P7, EF, N0,
    ST_TET, 4, P4, P6, P7, N0,
    /* case 48 */ 2,
    ST_HEX, 8, EJ, P1, P0, EI, EF, P7, P6, EH,
    ST_WDG, 6, P1, P3, P7, P0, P2, P6,
    /* case 49 */ 9,
    ST_PNT, 5, EF, EH, EJ, EA, ED,
    ST_PYR, 5, P7, P6, P2, P3, N0,
    ST_TET, 4, P2, N0, P1, P3,
    ST_PYR, 5, P6, EH, ED, P2, N0,
    ST_PYR, 5, EF, EH, P6, P7, N0,
    ST_TET, 4, P1, P7, P3, N0,
    ST_PYR, 5, P1, EJ, EF, P7, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_PYR, 5, ED, EA, P1, P2, N0,
    /* case 50 */ 9,
    ST_PNT, 5, EH, EF, EI, EA, EB,
    ST_PYR, 5, P6, P2, P3, P7, N0,
    ST_TET, 4, P3, P0, N0, P2,
    ST_PYR, 5, P7, P3, EB, EF, N0,
    ST_PYR, 5, EH, P6, P7, EF, N0,
    ST_TET, 4, P0, P2, P6, N0,
    ST_PYR, 5, P0, P6, EH, EI, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_PYR, 5, EB, P3, P0, EA, N0,
    /* case 51 */ 1,
    ST_HEX, 8, P2, P3, P7, P6, ED, EB, EF, EH,
    /* case 52 */ 11,
    ST_PNT, 4, EF, EJ, EC, EC,
    ST_PYR, 5, P3, EC, ED, P0, N0,
    ST_TET, 4, ED, EI, P0, N0,
    ST_PYR, 5, P1, P0, EI, EJ, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_TET, 4, P6, EH, EK, N0,
    ST_TET, 4, P7, P6, P3, N0,
    ST_PYR, 5, P6, EK, EC, P3, N0,
    ST_TET, 4, P7, P3, P1, N0,
    ST_PYR, 5, P7, P1, EJ, EF, N0,
    ST_PYR, 5, P7, EF, EH, P6, N0,
    /* case 53 */ 9,
    ST_PNT, 6, EA, EJ, EF, EH, EK, EC,
    ST_TET, 4, P7, P6, P3, N0,
    ST_PYR, 5, P7, EF, EH, P6, N0,
    ST_TET, 4, P1, P7, P3, N0,
    ST_PYR, 5, P1, EJ, EF, P7, N0,
    ST_TET, 4, EA, EJ, P1, N0,
    ST_PYR, 5, P1, P3, EC, EA, N0,
    ST_PYR, 5, EC, P3, P6, EK, N0,
    ST_TET, 4, P6, EH, EK, N0,
    /* case 54 */ 8,
    ST_PNT, 5, EF, EB, EH, EK, EC,
    ST_TET, 4, P0, EI, EA, ED,
    ST_PYR, 5, P3, EB, EF, P7, N0,
    ST_PYR, 5, P7, EF, EH, P6, N0,
    ST_TET, 4, P6, EH, EK, N0,
    ST_TET, 4, P7, P6, P3, N0,
    ST_PYR, 5, P6, EK, EC, P3, N0,
    ST_TET, 4, P3, EC, EB, N0,
    /* case 55 */ 7,
    ST_PNT, 5, EB, EF, EH, EC, EK,
    ST_PYR, 5, P7, EF, EH, P6, N0,
    ST_TET, 4, P6, EH, EK, N0,
    ST_PYR, 5, P3, EB, EF, P7, N0,
    ST_TET, 4, EC, EB, P3, N0,
    ST_PYR, 5, EK, EC, P3, P6, N0,
    ST_TET, 4, P6, P3, P7, N0,
    /* case 56 */ 11,
    ST_PNT, 4, EH, EI, EC, EC,
    ST_PYR, 5, P2, P1, EB, EC, N0,
    ST_TET, 4, EB, P1, EJ, N0,
    ST_PYR, 5, P0, EI, EJ, P1, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_TET, 4, P7, EL, EF, N0,
    ST_TET, 4, P6, P2, P7, N0,
    ST_PYR, 5, P7, P2, EC, EL, N0,
    ST_TET, 4, P6, P0, P2, N0,
    ST_PYR, 5, P6, EH, EI, P0, N0,
    ST_PYR, 5, P6, P7, EF, EH, N0,
    /* case 57 */ 8,
    ST_PNT, 5, EH, EF, ED, EC, EL,
    ST_TET, 4, P1, EA, EJ, EB,
    ST_PYR, 5, P7, EF, EH, P6, N0,
    ST_PYR, 5, P6, EH, ED, P2, N0,
    ST_TET, 4, P2, ED, EC, N0,
    ST_TET, 4, P6, P2, P7, N0,
    ST_PYR, 5, P2, EC, EL, P7, N0,
    ST_TET, 4, P7, EL, EF, N0,
    /* case 58 */ 9,
    ST_PNT, 6, EA, EI, EH, EF, EL, EC,
    ST_TET, 4, P6, P2, P7, N0,
    ST_PYR, 5, P6, P7, EF, EH, N0,
    ST_TET, 4, P0, P2, P6, N0,
    ST_PYR, 5, P0, P6, EH, EI, N0,
    ST_TET, 4, EA, P0, EI, N0,
    ST_PYR, 5, P0, EA, EC, P2, N0,
    ST_PYR, 5, EC, EL, P7, P2, N0,
    ST_TET, 4, P7, EL, EF, N0,
    /* case 59 */ 7,
    ST_PNT, 5, EF, EH, ED, EL, EC,
    ST_PYR, 5, P6, EH, ED, P2, N0,
    ST_TET, 4, P2, ED, EC, N0,
    ST_PYR, 5, P7, EF, EH, P6, N0,
    ST_TET, 4, EL, EF, P7, N0,
    ST_PYR, 5, EC, EL, P7, P2, N0,
    ST_TET, 4, P2, P7, P6, N0,
    /* case 60 */ 2,
    ST_WDG, 6, P1, EB, EJ, P0, ED, EI,
    ST_WDG, 6, P7, EF, EL, P6, EH, EK,
    /* case 61 */ 2,
    ST_WDG, 6, EH, P6, EK, EF, P7, EL,
    ST_TET, 4, EA, P1, EB, EJ,
    /* case 62 */ 2,
    ST_WDG, 6, EH, P6, EK, EF, P7, EL,
    ST_TET, 4, EA, ED, P0, EI,
    /* case 63 */ 1,
    ST_WDG, 6, P6, EK, EH, P7, EL, EF,
    /* case 64 */ 9,
    ST_PNT, 7, P7, P5, P4, P2, P3, P1, P0,
    ST_WDG, 6, P7, P4, P2, EG, EH, EK,
    ST_TET, 4, P7, P4, P2, N0,
    ST_TET, 4, P7, P5, P4, N0,
    ST_PYR, 5, P1, P0, P4, P5, N0,
    ST_PYR, 5, P3, P1, P5, P7, N0,
    ST_PYR, 5, P2, P0, P1, P3, N0,
    ST_TET, 4, P4, P0, P2, N0,
    ST_TET, 4, P2, P3, P7, N0,
    /* case 65 */ 12,
    ST_PNT, 2, EA, EG,
    ST_PYR, 5, P1, P5, P7, P3, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_TET, 4, P1, P4, P5, N0,
    ST_TET, 4, P2, P1, P3, N0,
    ST_TET, 4, P7, P2, P3, N0,
    ST_PYR, 5, P7, EG, EK, P2, N0,
    ST_PYR, 5, P4, EH, EG, P7, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_PYR, 5, P1, EA, EI, P4, N0,
    ST_PYR, 5, P1, P2, ED, EA, N0,
    ST_TET, 4, P2, EK, ED, N0,
    /* case 66 */ 6,
    ST_WDG, 6, P4, P2, P7, EH, EK, EG,
    ST_WDG, 6, EA, EB, EJ, P0, P3, P5,
    ST_TET, 4, P2, P7, P4, P3,
    ST_TET, 4, P5, P7, P3, P4,
    ST_TET, 4, P0, P4, P5, P3,
    ST_TET, 4, P2, P0, P3, P4,
    /* case 67 */ 11,
    ST_PNT, 4, EJ, EB, EG, EG,
    ST_PYR, 5, P7, EG, EK, P2, N0,
    ST_TET, 4, EK, ED, P2, N0,
    ST_PYR, 5, P3, P2, ED, EB, N0,
    ST_TET, 4, P7, P2, P3, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_PYR, 5, P4, EH, EG, P7, N0,
    ST_TET, 4, P5, P7, P3, N0,
    ST_PYR, 5, P5, P3, EB, EJ, N0,
    ST_PYR, 5, P5, EJ, EI, P4, N0,
    /* case 68 */ 2,
    ST_HEX, 8, EH, P4, P0, ED, EG, P7, P3, EC,
    ST_WDG, 6, P4, P5, P7, P0, P1, P3,
    /* case 69 */ 9,
    ST_PNT, 5, EG, EC, EH, EI, EA,
    ST_PYR, 5, P7, P3, P1, P5, N0,
    ST_TET, 4, P1, N0, P4, P5,
    ST_PYR, 5, P3, EC, EA, P1, N0,
    ST_PYR, 5, EG, EC, P3, P7, N0,
    ST_TET, 4, P4, P7, P5, N0,
    ST_PYR, 5, P4, EH, EG, P7, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_PYR, 5, EA, EI, P4, P1, N0,
    /* case 70 */ 11,
    ST_PNT, 4, EG, EH, EJ, EJ,
    ST_PYR, 5, P5, EJ, EA, P0, N0,
    ST_TET, 4, EA, ED, P0, N0,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_TET, 4, P3, EC, EB, N0,
    ST_TET, 4, P7, P3, P5, N0,
    ST_PYR, 5, P3, EB, EJ, P5, N0,
    ST_TET, 4, P7, P5, P4, N0,
    ST_PYR, 5, P7, P4, EH, EG, N0,
    ST_PYR, 5, P7, EG, EC, P3, N0,
    /* case 71 */ 9,
    ST_PNT, 6, EI, EH, EG, EC, EB, EJ,
    ST_TET, 4, P7, P3, P5, N0,
    ST_PYR, 5, P7, EG, EC, P3, N0,
    ST_TET, 4, P4, P7, P5, N0,
    ST_PYR, 5, P4, EH, EG, P7, N0,
    ST_TET, 4, EI, EH, P4, N0,
    ST_PYR, 5, P4, P5, EJ, EI, N0,
    ST_PYR, 5, EJ, P5, P3, EB, N0,
    ST_TET, 4, P3, EC, EB, N0,
    /* case 72 */ 12,
    ST_PNT, 2, EH, EB,
    ST_PYR, 5, P4, P5, P1, P0, N0,
    ST_TET, 4, P5, P7, P1, N0,
    ST_TET, 4, P4, P7, P5, N0,
    ST_TET, 4, P2, P4, P0, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_PYR, 5, P1, EB, EC, P2, N0,
    ST_PYR, 5, P7, EL, EB, P1, N0,
    ST_TET, 4, P7, EG, EL, N0,
    ST_PYR, 5, P4, EH, EG, P7, N0,
    ST_PYR, 5, P4, P2, EK, EH, N0,
    ST_TET, 4, P2, EC, EK, N0,
    /* case 73 */ 6,
    ST_TET, 4, P2, ED, EC, EK,
    ST_TET, 4, P7, P5, P4, P1,
    ST_PYR, 5, EA, EI, P4, P1, EH,
    ST_PYR, 5, EB, P1, P7, EL, EG,
    ST_PYR, 5, P4, EH, EG, P7, P1,
    ST_PYR, 5, EH, EA, EB, EG, P1,
    /* case 74 */ 11,
    ST_PNT, 4, EJ, EA, EH, EH,
    ST_PYR, 5, P4, P2, EK, EH, N0,
    ST_TET, 4, EK, P2, EC, N0,
    ST_PYR, 5, P0, EA, EC, P2, N0,
    ST_TET, 4, P4, P0, P2, N0,
    ST_TET, 4, P7, EG, EL, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_PYR, 5, P7, P4, EH, EG, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_PYR, 5, P5, EJ, EA, P0, N0,
    ST_PYR, 5, P5, P7, EL, EJ, N0,
    /* case 75 */ 8,
    ST_PNT, 5, EJ, EL, EI, EH, EG,
    ST_TET, 4, P2, ED, EC, EK,
    ST_PYR, 5, P7, EL, EJ, P5, N0,
    ST_PYR, 5, P5, EJ, EI, P4, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_PYR, 5, P4, EH, EG, P7, N0,
    ST_TET, 4, P7, EG, EL, N0,
    /* case 76 */ 9,
    ST_PNT, 5, EB, ED, EL, EG, EH,
    ST_PYR, 5, P1, P0, P4, P5, N0,
    ST_TET, 4, P4, N0, P7, P5,
    ST_PYR, 5, P0, ED, EH, P4, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_TET, 4, P7, P1, P5, N0,
    ST_PYR, 5, P7, EL, EB, P1, N0,
    ST_TET, 4, P7, EG, EL, N0,
    ST_PYR, 5, EH, EG, P7, P4, N0,
    /* case 77 */ 5,
    ST_TET, 4, P7, P5, P4, P1,
    ST_PYR, 5, EA, EI, P4, P1, EH,
    ST_PYR, 5, EB, P1, P7, EL, EG,
    ST_PYR, 5, P4, EH, EG, P7, P1,
    ST_PYR, 5, EH, EA, EB, EG, P1,
    /* case 78 */ 9,
    ST_PNT, 6, ED, EA, EJ, EL, EG, EH,
    ST_TET, 4, P5, P4, P7, N0,
    ST_PYR, 5, P5, P7, EL, EJ, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_TET, 4, ED, P0, EA, N0,
    ST_PYR, 5, P0, ED, EH, P4, N0,
    ST_PYR, 5, EH, EG, P7, P4, N0,
    ST_TET, 4, P7, EG, EL, N0,
    /* case 79 */ 7,
    ST_PNT, 5, EL, EJ, EI, EG, EH,
    ST_PYR, 5, P5, EJ, EI, P4, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_PYR, 5, P7, EL, EJ, P5, N0,
    ST_TET, 4, EG, EL, P7, N0,
    ST_PYR, 5, EH, EG, P7, P4, N0,
    ST_TET, 4, P4, P7, P5, N0,
    /* case 80 */ 2,
    ST_HEX, 8, EG, P7, P5, EE, EK, P2, P0, EI,
    ST_WDG, 6, P0, P1, P5, P2, P3, P7,
    /* case 81 */ 9,
    ST_PNT, 5, EG, EE, EK, ED, EA,
    ST_PYR, 5, P7, P3, P1, P5, N0,
    ST_TET, 4, P1, P2, N0, P3,
    ST_PYR, 5, P5, P1, EA, EE, N0,
    ST_PYR, 5, EG, P7, P5, EE, N0,
    ST_TET, 4, P2, P3, P7, N0,
    ST_PYR, 5, P2, P7, EG, EK, N0,
    ST_TET, 4, P2, EK, ED, N0,
    ST_PYR, 5, EA, P1, P2, ED, N0,
    /* case 82 */ 11,
    ST_PNT, 4, EG, EK, EB, EB,
    ST_PYR, 5, P3, P0, EA, EB, N0,
    ST_TET, 4, EA, P0, EI, N0,
    ST_PYR, 5, P2, EK, EI, P0, N0,
    ST_TET, 4, P3, P2, P0, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_TET, 4, P7, P3, P5, N0,
    ST_PYR, 5, P5, P3, EB, EJ, N0,
    ST_TET, 4, P7, P2, P3, N0,
    ST_PYR, 5, P7, EG, EK, P2, N0,
    ST_PYR, 5, P7, P5, EE, EG, N0,
    /* case 83 */ 9,
    ST_PNT, 6, ED, EK, EG, EE, EJ, EB,
    ST_TET, 4, P7, P3, P5, N0,
    ST_PYR, 5, P7, P5, EE, EG, N0,
    ST_TET, 4, P2, P3, P7, N0,
    ST_PYR, 5, P2, P7, EG, EK, N0,
    ST_TET, 4, ED, P2, EK, N0,
    ST_PYR, 5, P2, ED, EB, P3, N0,
    ST_PYR, 5, EB, EJ, P5, P3, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    /* case 84 */ 9,
    ST_PNT, 5, EC, EG, ED, EI, EE,
    ST_PYR, 5, P3, P1, P5, P7, N0,
    ST_TET, 4, P5, P0, N0, P1,
    ST_PYR, 5, P7, P5, EE, EG, N0,
    ST_PYR, 5, EC, P3, P7, EG, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_PYR, 5, P0, P3, EC, ED, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_PYR, 5, EE, P5, P0, EI, N0,
    /* case 85 */ 1,
    ST_HEX, 8, P1, P5, P7, P3, EA, EE, EG, EC,
    /* case 86 */ 8,
    ST_PNT, 5, EG, EE, EC, EB, EJ,
    ST_TET, 4, P0, ED, EI, EA,
    ST_PYR, 5, P5, EE, EG, P7, N0,
    ST_PYR, 5, P7, EG, EC, P3, N0,
    ST_TET, 4, P3, EC, EB, N0,
    ST_TET, 4, P7, P3, P5, N0,
    ST_PYR, 5, P3, EB, EJ, P5, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    /* case 87 */ 7,
    ST_PNT, 5, EE, EG, EC, EJ, EB,
    ST_PYR, 5, P7, EG, EC, P3, N0,
    ST_TET, 4, P3, EC, EB, N0,
    ST_PYR, 5, P5, EE, EG, P7, N0,
    ST_TET, 4, EJ, EE, P5, N0,
    ST_PYR, 5, EB, EJ, P5, P3, N0,
    ST_TET, 4, P3, P5, P7, N0,
    /* case 88 */ 11,
    ST_PNT, 4, EE, EI, EB, EB,
    ST_PYR, 5, P1, EB, EC, P2, N0,
    ST_TET, 4, EC, EK, P2, N0,
    ST_PYR, 5, P0, P2, EK, EI, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_TET, 4, P7, EG, EL, N0,
    ST_TET, 4, P5, P7, P1, N0,
    ST_PYR, 5, P7, EL, EB, P1, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_PYR, 5, P5, P0, EI, EE, N0,
    ST_PYR, 5, P5, EE, EG, P7, N0,
    /* case 89 */ 8,
    ST_PNT, 5, EE, EG, EA, EB, EL,
    ST_TET, 4, P2, EK, ED, EC,
    ST_PYR, 5, P7, P5, EE, EG, N0,
    ST_PYR, 5, P5, P1, EA, EE, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_TET, 4, P5, P7, P1, N0,
    ST_PYR, 5, P1, P7, EL, EB, N0,
    ST_TET, 4, P7, EG, EL, N0,
    /* case 90 */ 2,
    ST_WDG, 6, P2, EK, EC, P0, EI, EA,
    ST_WDG, 6, P7, EL, EG, P5, EJ, EE,
    /* case 91 */ 2,
    ST_WDG, 6, EJ, P5, EE, EL, P7, EG,
    ST_TET, 4, ED, P2, EK, EC,
    /* case 92 */ 9,
    ST_PNT, 6, ED, EI, EE, EG, EL, EB,
    ST_TET, 4, P5, P7, P1, N0,
    ST_PYR, 5, P5, EE, EG, P7, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_TET, 4, ED, EI, P0, N0,
    ST_PYR, 5, P0, P1, EB, ED, N0,
    ST_PYR, 5, EB, P1, P7, EL, N0,
    ST_TET, 4, P7, EG, EL, N0,
    /* case 93 */ 7,
    ST_PNT, 5, EG, EE, EA, EL, EB,
    ST_PYR, 5, P5, P1, EA, EE, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_PYR, 5, P7, P5, EE, EG, N0,
    ST_TET, 4, EL, P7, EG, N0,
    ST_PYR, 5, EB, P1, P7, EL, N0,
    ST_TET, 4, P1, P5, P7, N0,
    /* case 94 */ 2,
    ST_WDG, 6, EJ, P5, EE, EL, P7, EG,
    ST_TET, 4, ED, EI, P0, EA,
    /* case 95 */ 1,
    ST_WDG, 6, P5, EE, EJ, P7, EG, EL,
    /* case 96 */ 12,
    ST_PNT, 2, EK, EJ,
    ST_PYR, 5, P2, P0, P1, P3, N0,
    ST_TET, 4, P3, P1, P7, N0,
    ST_TET, 4, P2, P3, P7, N0,
    ST_TET, 4, P4, P0, P2, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_PYR, 5, P7, P1, EJ, EF, N0,
    ST_TET, 4, P7, EF, EG, N0,
    ST_PYR, 5, P2, P7, EG, EK, N0,
    ST_PYR, 5, P2, EK, EH, P4, N0,
    ST_TET, 4, P4, EH, EE, N0,
    /* case 97 */ 6,
    ST_TET, 4, P4, EE, EI, EH,
    ST_TET, 4, P7, P2, P3, P1,
    ST_PYR, 5, EA, P1, P2, ED, EK,
    ST_PYR, 5, EJ, EF, P7, P1, EG,
    ST_PYR, 5, P2, P7, EG, EK, P1,
    ST_PYR, 5, EK, EG, EJ, EA, P1,
    /* case 98 */ 11,
    ST_PNT, 4, EB, EA, EK, EK,
    ST_PYR, 5, P2, EK, EH, P4, N0,
    ST_TET, 4, EH, EE, P4, N0,
    ST_PYR, 5, P0, P4, EE, EA, N0,
    ST_TET, 4, P2, P4, P0, N0,
    ST_TET, 4, P7, EF, EG, N0,
    ST_TET, 4, P3, P7, P2, N0,
    ST_PYR, 5, P7, EG, EK, P2, N0,
    ST_TET, 4, P3, P2, P0, N0,
    ST_PYR, 5, P3, P0, EA, EB, N0,
    ST_PYR, 5, P3, EB, EF, P7, N0,
    /* case 99 */ 8,
    ST_PNT, 5, EB, EF, ED, EK, EG,
    ST_TET, 4, P4, EE, EI, EH,
    ST_PYR, 5, P7, P3, EB, EF, N0,
    ST_PYR, 5, P3, P2, ED, EB, N0,
    ST_TET, 4, P2, EK, ED, N0,
    ST_TET, 4, P3, P7, P2, N0,
    ST_PYR, 5, P2, P7, EG, EK, N0,
    ST_TET, 4, P7, EF, EG, N0,
    /* case 100 */ 11,
    ST_PNT, 4, EC, ED, EJ, EJ,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_TET, 4, EE, P4, EH, N0,
    ST_PYR, 5, P0, ED, EH, P4, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_TET, 4, P7, EF, EG, N0,
    ST_TET, 4, P3, P1, P7, N0,
    ST_PYR, 5, P7, P1, EJ, EF, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_PYR, 5, P3, EC, ED, P0, N0,
    ST_PYR, 5, P3, P7, EG, EC, N0,
    /* case 101 */ 8,
    ST_PNT, 5, EC, EG, EA, EJ, EF,
    ST_TET, 4, P4, EI, EH, EE,
    ST_PYR, 5, P7, EG, EC, P3, N0,
    ST_PYR, 5, P3, EC, EA, P1, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_TET, 4, P3, P1, P7, N0,
    ST_PYR, 5, P1, EJ, EF, P7, N0,
    ST_TET, 4, P7, EF, EG, N0,
    /* case 102 */ 2,
    ST_WDG, 6, P0, ED, EA, P4, EH, EE,
    ST_WDG, 6, P3, EB, EC, P7, EF, EG,
    /* case 103 */ 2,
    ST_WDG, 6, EF, P7, EG, EB, P3, EC,
    ST_TET, 4, EI, EH, P4, EE,
    /* case 104 */ 6,
    ST_TET, 4, P7, EF, EG, EL,
    ST_TET, 4, P2, P0, P1, P4,
    ST_PYR, 5, EE, EJ, P1, P4, EB,
    ST_PYR, 5, EH, P4, P2, EK, EC,
    ST_PYR, 5, P1, EB, EC, P2, P4,
    ST_PYR, 5, EB, EE, EH, EC, P4,
    /* case 105 */ 4,
    ST_TET, 4, EH, EI, EE, P4,
    ST_TET, 4, EK, EC, ED, P2,
    ST_TET, 4, EG, EF, EL, P7,
    ST_TET, 4, EB, EJ, EA, P1,
    /* case 106 */ 8,
    ST_PNT, 5, EA, EC, EE, EH, EK,
    ST_TET, 4, P7, EL, EF, EG,
    ST_PYR, 5, P2, P0, EA, EC, N0,
    ST_PYR, 5, P0, P4, EE, EA, N0,
    ST_TET, 4, P4, EH, EE, N0,
    ST_TET, 4, P0, P2, P4, N0,
    ST_PYR, 5, P4, P2, EK, EH, N0,
    ST_TET, 4, P2, EC, EK, N0,
    /* case 107 */ 3,
    ST_TET, 4, EL, EG, EF, P7,
    ST_TET, 4, EC, P2, ED, EK,
    ST_TET, 4, EE, EI, P4, EH,
    /* case 108 */ 8,
    ST_PNT, 5, ED, EH, EB, EJ, EE,
    ST_TET, 4, P7, EG, EL, EF,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_PYR, 5, P0, P1, EB, ED, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_TET, 4, P0, P4, P1, N0,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_TET, 4, P4, EH, EE, N0,
    /* case 109 */ 3,
    ST_TET, 4, EG, EF, EL, P7,
    ST_TET, 4, EH, P4, EI, EE,
    ST_TET, 4, EB, EA, P1, EJ,
    /* case 110 */ 2,
    ST_WDG, 6, EH, P4, EE, ED, P0, EA,
    ST_TET, 4, EL, EF, P7, EG,
    /* case 111 */ 2,
    ST_TET, 4, EE, EH, EI, P4,
    ST_TET, 4, EF, EL, EG, P7,
    /* case 112 */ 9,
    ST_PNT, 5, EJ, EI, EF, EG, EK,
    ST_PYR, 5, P1, P3, P2, P0, N0,
    ST_TET, 4, P2, P7, N0, P3,
    ST_PYR, 5, P0, P2, EK, EI, N0,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_TET, 4, P7, P3, P1, N0,
    ST_PYR, 5, P7, P1, EJ, EF, N0,
    ST_TET, 4, P7, EF, EG, N0,
    ST_PYR, 5, EK, P2, P7, EG, N0,
    /* case 113 */ 5,
    ST_TET, 4, P7, P2, P3, P1,
    ST_PYR, 5, EA, P1, P2, ED, EK,
    ST_PYR, 5, EJ, EF, P7, P1, EG,
    ST_PYR, 5, P2, P7, EG, EK, P1,
    ST_PYR, 5, EK, EG, EJ, EA, P1,
    /* case 114 */ 9,
    ST_PNT, 6, EI, EA, EB, EF, EG, EK,
    ST_TET, 4, P3, P7, P2, N0,
    ST_PYR, 5, P3, EB, EF, P7, N0,
    ST_TET, 4, P0, P3, P2, N0,
    ST_PYR, 5, P0, EA, EB, P3, N0,
    ST_TET, 4, EI, EA, P0, N0,
    ST_PYR, 5, P0, P2, EK, EI, N0,
    ST_PYR, 5, EK, P2, P7, EG, N0,
    ST_TET, 4, P7, EF, EG, N0,
    /* case 115 */ 7,
    ST_PNT, 5, EF, EB, ED, EG, EK,
    ST_PYR, 5, P3, P2, ED, EB, N0,
    ST_TET, 4, P2, EK, ED, N0,
    ST_PYR, 5, P7, P3, EB, EF, N0,
    ST_TET, 4, EG, P7, EF, N0,
    ST_PYR, 5, EK, P2, P7, EG, N0,
    ST_TET, 4, P2, P3, P7, N0,
    /* case 116 */ 9,
    ST_PNT, 6, EI, ED, EC, EG, EF, EJ,
    ST_TET, 4, P3, P1, P7, N0,
    ST_PYR, 5, P3, P7, EG, EC, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_PYR, 5, P0, P3, EC, ED, N0,
    ST_TET, 4, EI, P0, ED, N0,
    ST_PYR, 5, P0, EI, EJ, P1, N0,
    ST_PYR, 5, EJ, EF, P7, P1, N0,
    ST_TET, 4, P7, EF, EG, N0,
    /* case 117 */ 7,
    ST_PNT, 5, EG, EC, EA, EF, EJ,
    ST_PYR, 5, P3, EC, EA, P1, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_PYR, 5, P7, EG, EC, P3, N0,
    ST_TET, 4, EF, EG, P7, N0,
    ST_PYR, 5, EJ, EF, P7, P1, N0,
    ST_TET, 4, P1, P7, P3, N0,
    /* case 118 */ 2,
    ST_WDG, 6, EF, P7, EG, EB, P3, EC,
    ST_TET, 4, EI, P0, ED, EA,
    /* case 119 */ 1,
    ST_WDG, 6, P7, EG, EF, P3, EC, EB,
    /* case 120 */ 8,
    ST_PNT, 5, EI, EK, EJ, EB, EC,
    ST_TET, 4, P7, EF, EG, EL,
    ST_PYR, 5, P2, EK, EI, P0, N0,
    ST_PYR, 5, P0, EI, EJ, P1, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_TET, 4, P0, P1, P2, N0,
    ST_PYR, 5, P1, EB, EC, P2, N0,
    ST_TET, 4, P2, EC, EK, N0,
    /* case 121 */ 3,
    ST_TET, 4, EG, EF, EL, P7,
    ST_TET, 4, EK, ED, P2, EC,
    ST_TET, 4, EJ, P1, EA, EB,
    /* case 122 */ 2,
    ST_WDG, 6, EI, P0, EA, EK, P2, EC,
    ST_TET, 4, EF, P7, EL, EG,
    /* case 123 */ 2,
    ST_TET, 4, EC, ED, EK, P2,
    ST_TET, 4, EL, EG, EF, P7,
    /* case 124 */ 2,
    ST_WDG, 6, ED, P0, EI, EB, P1, EJ,
    ST_TET, 4, EG, P7, EF, EL,
    /* case 125 */ 2,
    ST_TET, 4, EJ, EA, EB, P1,
    ST_TET, 4, EF, EL, EG, P7,
    /* case 126 */ 2,
    ST_TET, 4, EA, EI, ED, P0,
    ST_TET, 4, EL, EG, EF, P7,
    /* case 127 */ 1,
    ST_TET, 4, EF, EL, EG, P7,
    /* case 128 */ 9,
    ST_PNT, 7, P6, P4, P5, P3, P2, P0, P1,
    ST_WDG, 6, EG, EF, EL, P6, P5, P3,
    ST_TET, 4, P6, P3, P5, N0,
    ST_TET, 4, P6, P5, P4, N0,
    ST_PYR, 5, P0, P4, P5, P1, N0,
    ST_PYR, 5, P2, P6, P4, P0, N0,
    ST_PYR, 5, P3, P2, P0, P1, N0,
    ST_TET, 4, P5, P3, P1, N0,
    ST_TET, 4, P3, P6, P2, N0,
    /* case 129 */ 6,
    ST_WDG, 6, P5, P6, P3, EF, EG, EL,
    ST_WDG, 6, EI, ED, EA, P4, P2, P1,
    ST_TET, 4, P6, P3, P5, P2,
    ST_TET, 4, P1, P3, P2, P5,
    ST_TET, 4, P4, P5, P1, P2,
    ST_TET, 4, P6, P4, P2, P5,
    /* case 130 */ 12,
    ST_PNT, 2, EA, EG,
    ST_PYR, 5, P0, P2, P6, P4, N0,
    ST_TET, 4, P4, P6, P5, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_TET, 4, P3, P2, P0, N0,
    ST_TET, 4, P6, P2, P3, N0,
    ST_PYR, 5, P6, P3, EL, EG, N0,
    ST_PYR, 5, P5, P6, EG, EF, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_PYR, 5, P0, EA, EB, P3, N0,
    ST_TET, 4, P3, EB, EL, N0,
    /* case 131 */ 11,
    ST_PNT, 4, EI, ED, EG, EG,
    ST_PYR, 5, P6, P3, EL, EG, N0,
    ST_TET, 4, EL, P3, EB, N0,
    ST_PYR, 5, P2, ED, EB, P3, N0,
    ST_TET, 4, P6, P2, P3, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    ST_TET, 4, P4, P6, P5, N0,
    ST_PYR, 5, P5, P6, EG, EF, N0,
    ST_TET, 4, P4, P2, P6, N0,
    ST_PYR, 5, P4, EI, ED, P2, N0,
    ST_PYR, 5, P4, P5, EJ, EI, N0,
    /* case 132 */ 12,
    ST_PNT, 2, ED, EF,
    ST_PYR, 5, P0, P4, P5, P1, N0,
    ST_TET, 4, P1, P5, P3, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_TET, 4, P6, P4, P0, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_PYR, 5, P5, P6, EG, EF, N0,
    ST_PYR, 5, P3, P5, EF, EL, N0,
    ST_TET, 4, P3, EL, EC, N0,
    ST_PYR, 5, P0, P3, EC, ED, N0,
    ST_PYR, 5, P0, ED, EK, P6, N0,
    ST_TET, 4, P6, EK, EG, N0,
    /* case 133 */ 11,
    ST_PNT, 4, EA, EI, EF, EF,
    ST_PYR, 5, P5, P6, EG, EF, N0,
    ST_TET, 4, EG, P6, EK, N0,
    ST_PYR, 5, P4, EI, EK, P6, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_TET, 4, P3, EL, EC, N0,
    ST_TET, 4, P1, P5, P3, N0,
    ST_PYR, 5, P3, P5, EF, EL, N0,
    ST_TET, 4, P1, P4, P5, N0,
    ST_PYR, 5, P1, EA, EI, P4, N0,
    ST_PYR, 5, P1, P3, EC, EA, N0,
    /* case 134 */ 6,
    ST_TET, 4, P3, EB, EL, EC,
    ST_TET, 4, P6, P4, P0, P5,
    ST_PYR, 5, EJ, EA, P0, P5, ED,
    ST_PYR, 5, EF, P5, P6, EG, EK,
    ST_PYR, 5, P0, ED, EK, P6, P5,
    ST_PYR, 5, ED, EJ, EF, EK, P5,
    /* case 135 */ 8,
    ST_PNT, 5, EI, EK, EJ, EF, EG,
    ST_TET, 4, P3, EC, EB, EL,
    ST_PYR, 5, P6, P4, EI, EK, N0,
    ST_PYR, 5, P4, P5, EJ, EI, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    ST_TET, 4, P4, P6, P5, N0,
    ST_PYR, 5, P5, P6, EG, EF, N0,
    ST_TET, 4, P6, EK, EG, N0,
    /* case 136 */ 2,
    ST_HEX, 8, EG, P6, P2, EC, EF, P5, P1, EB,
    ST_WDG, 6, P1, P0, P2, P5, P4, P6,
    /* case 137 */ 11,
    ST_PNT, 4, EG, EF, EI, EI,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_TET, 4, EA, P1, EB, N0,
    ST_PYR, 5, P5, EF, EB, P1, N0,
    ST_TET, 4, P4, P5, P1, N0,
    ST_TET, 4, P2, ED, EC, N0,
    ST_TET, 4, P6, P4, P2, N0,
    ST_PYR, 5, P2, P4, EI, ED, N0,
    ST_TET, 4, P6, P5, P4, N0,
    ST_PYR, 5, P6, EG, EF, P5, N0,
    ST_PYR, 5, P6, P2, EC, EG, N0,
    /* case 138 */ 9,
    ST_PNT, 5, EG, EC, EF, EJ, EA,
    ST_PYR, 5, P6, P4, P0, P2, N0,
    ST_TET, 4, P0, P5, N0, P4,
    ST_PYR, 5, P2, P0, EA, EC, N0,
    ST_PYR, 5, EG, P6, P2, EC, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_PYR, 5, P5, P6, EG, EF, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    /* case 139 */ 9,
    ST_PNT, 6, EJ, EF, EG, EC, ED, EI,
    ST_TET, 4, P6, P4, P2, N0,
    ST_PYR, 5, P6, P2, EC, EG, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_PYR, 5, P5, P6, EG, EF, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    ST_PYR, 5, P5, EJ, EI, P4, N0,
    ST_PYR, 5, EI, ED, P2, P4, N0,
    ST_TET, 4, P2, ED, EC, N0,
    /* case 140 */ 9,
    ST_PNT, 5, EF, EB, EG, EK, ED,
    ST_PYR, 5, P5, P1, P0, P4, N0,
    ST_TET, 4, P0, N0, P6, P4,
    ST_PYR, 5, P1, EB, ED, P0, N0,
    ST_PYR, 5, EF, EB, P1, P5, N0,
    ST_TET, 4, P6, P5, P4, N0,
    ST_PYR, 5, P6, EG, EF, P5, N0,
    ST_TET, 4, P6, EK, EG, N0,
    ST_PYR, 5, ED, EK, P6, P0, N0,
    /* case 141 */ 9,
    ST_PNT, 6, EK, EG, EF, EB, EA, EI,
    ST_TET, 4, P5, P1, P4, N0,
    ST_PYR, 5, P5, EF, EB, P1, N0,
    ST_TET, 4, P6, P5, P4, N0,
    ST_PYR, 5, P6, EG, EF, P5, N0,
    ST_TET, 4, EK, EG, P6, N0,
    ST_PYR, 5, P6, P4, EI, EK, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    ST_TET, 4, P1, EB, EA, N0,
    /* case 142 */ 5,
    ST_TET, 4, P6, P4, P0, P5,
    ST_PYR, 5, EJ, EA, P0, P5, ED,
    ST_PYR, 5, EF, P5, P6, EG, EK,
    ST_PYR, 5, P0, ED, EK, P6, P5,
    ST_PYR, 5, ED, EJ, EF, EK, P5,
    /* case 143 */ 7,
    ST_PNT, 5, EK, EI, EJ, EG, EF,
    ST_PYR, 5, P4, P5, EJ, EI, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    ST_PYR, 5, P6, P4, EI, EK, N0,
    ST_TET, 4, EG, P6, EK, N0,
    ST_PYR, 5, EF, P5, P6, EG, N0,
    ST_TET, 4, P5, P4, P6, N0,
    /* case 144 */ 12,
    ST_PNT, 2, EI, EL,
    ST_PYR, 5, P0, P1, P3, P2, N0,
    ST_TET, 4, P1, P5, P3, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_TET, 4, P6, P0, P2, N0,
    ST_TET, 4, P3, P6, P2, N0,
    ST_PYR, 5, P3, EL, EG, P6, N0,
    ST_PYR, 5, P5, EF, EL, P3, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_PYR, 5, P0, P6, EH, EI, N0,
    ST_TET, 4, P6, EG, EH, N0,
    /* case 145 */ 11,
    ST_PNT, 4, EA, ED, EL, EL,
    ST_PYR, 5, P3, EL, EG, P6, N0,
    ST_TET, 4, EG, EH, P6, N0,
    ST_PYR, 5, P2, P6, EH, ED, N0,
    ST_TET, 4, P3, P6, P2, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_TET, 4, P1, P5, P3, N0,
    ST_PYR, 5, P5, EF, EL, P3, N0,
    ST_TET, 4, P1, P3, P2, N0,
    ST_PYR, 5, P1, P2, ED, EA, N0,
    ST_PYR, 5, P1, EA, EE, P5, N0,
    /* case 146 */ 6,
    ST_TET, 4, P5, EF, EJ, EE,
    ST_TET, 4, P6, P0, P2, P3,
    ST_PYR, 5, EB, P3, P0, EA, EI,
    ST_PYR, 5, EL, EG, P6, P3, EH,
    ST_PYR, 5, P0, P6, EH, EI, P3,
    ST_PYR, 5, EI, EH, EL, EB, P3,
    /* case 147 */ 8,
    ST_PNT, 5, ED, EH, EB, EL, EG,
    ST_TET, 4, P5, EJ, EE, EF,
    ST_PYR, 5, P6, EH, ED, P2, N0,
    ST_PYR, 5, P2, ED, EB, P3, N0,
    ST_TET, 4, P3, EB, EL, N0,
    ST_TET, 4, P2, P3, P6, N0,
    ST_PYR, 5, P3, EL, EG, P6, N0,
    ST_TET, 4, P6, EG, EH, N0,
    /* case 148 */ 6,
    ST_TET, 4, P6, EG, EH, EK,
    ST_TET, 4, P3, P0, P1, P5,
    ST_PYR, 5, EE, P5, P0, EI, ED,
    ST_PYR, 5, EF, EL, P3, P5, EC,
    ST_PYR, 5, P0, P3, EC, ED, P5,
    ST_PYR, 5, ED, EC, EF, EE, P5,
    /* case 149 */ 8,
    ST_PNT, 5, EA, EC, EE, EF, EL,
    ST_TET, 4, P6, EH, EK, EG,
    ST_PYR, 5, P3, EC, EA, P1, N0,
    ST_PYR, 5, P1, EA, EE, P5, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_TET, 4, P1, P5, P3, N0,
    ST_PYR, 5, P5, EF, EL, P3, N0,
    ST_TET, 4, P3, EL, EC, N0,
    /* case 150 */ 4,
    ST_TET, 4, EH, EG, EK, P6,
    ST_TET, 4, EI, ED, EA, P0,
    ST_TET, 4, EE, EJ, EF, P5,
    ST_TET, 4, EB, EC, EL, P3,
    /* case 151 */ 3,
    ST_TET, 4, EK, EH, EG, P6,
    ST_TET, 4, EC, EB, P3, EL,
    ST_TET, 4, EE, P5, EJ, EF,
    /* case 152 */ 11,
    ST_PNT, 4, EC, EB, EI, EI,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_TET, 4, EE, EF, P5, N0,
    ST_PYR, 5, P1, P5, EF, EB, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_TET, 4, P6, EG, EH, N0,
    ST_TET, 4, P2, P6, P0, N0,
    ST_PYR, 5, P6, EH, EI, P0, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_PYR, 5, P2, P1, EB, EC, N0,
    ST_PYR, 5, P2, EC, EG, P6, N0,
    /* case 153 */ 2,
    ST_WDG, 6, P6, EH, EG, P2, ED, EC,
    ST_WDG, 6, P5, EF, EE, P1, EB, EA,
    /* case 154 */ 8,
    ST_PNT, 5, EC, EG, EA, EI, EH,
    ST_TET, 4, P5, EF, EJ, EE,
    ST_PYR, 5, P6, P2, EC, EG, N0,
    ST_PYR, 5, P2, P0, EA, EC, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_TET, 4, P2, P6, P0, N0,
    ST_PYR, 5, P0, P6, EH, EI, N0,
    ST_TET, 4, P6, EG, EH, N0,
    /* case 155 */ 2,
    ST_WDG, 6, EG, P6, EH, EC, P2, ED,
    ST_TET, 4, EJ, EE, P5, EF,
    /* case 156 */ 8,
    ST_PNT, 5, EB, EF, ED, EI, EE,
    ST_TET, 4, P6, EK, EG, EH,
    ST_PYR, 5, P5, EF, EB, P1, N0,
    ST_PYR, 5, P1, EB, ED, P0, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_TET, 4, P1, P0, P5, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_TET, 4, P5, EE, EF, N0,
    /* case 157 */ 2,
    ST_WDG, 6, EB, P1, EA, EF, P5, EE,
    ST_TET, 4, EK, P6, EH, EG,
    /* case 158 */ 3,
    ST_TET, 4, EJ, EF, EE, P5,
    ST_TET, 4, EA, ED, P0, EI,
    ST_TET, 4, EG, P6, EK, EH,
    /* case 159 */ 2,
    ST_TET, 4, EG, EK, EH, P6,
    ST_TET, 4, EF, EE, EJ, P5,
    /* case 160 */ 2,
    ST_HEX, 8, EL, P3, P1, EJ, EG, P6, P4, EE,
    ST_WDG, 6, P3, P2, P6, P1, P0, P4,
    /* case 161 */ 11,
    ST_PNT, 4, EG, EL, ED, ED,
    ST_PYR, 5, P2, ED, EA, P1, N0,
    ST_TET, 4, EA, EJ, P1, N0,
    ST_PYR, 5, P3, P1, EJ, EL, N0,
    ST_TET, 4, P2, P1, P3, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_TET, 4, P6, P4, P2, N0,
    ST_PYR, 5, P4, EI, ED, P2, N0,
    ST_TET, 4, P6, P2, P3, N0,
    ST_PYR, 5, P6, P3, EL, EG, N0,
    ST_PYR, 5, P6, EG, EE, P4, N0,
    /* case 162 */ 9,
    ST_PNT, 5, EG, EE, EL, EB, EA,
    ST_PYR, 5, P6, P4, P0, P2, N0,
    ST_TET, 4, P0, N0, P3, P2,
    ST_PYR, 5, P4, EE, EA, P0, N0,
    ST_PYR, 5, EG, EE, P4, P6, N0,
    ST_TET, 4, P3, P6, P2, N0,
    ST_PYR, 5, P3, EL, EG, P6, N0,
    ST_TET, 4, P3, EB, EL, N0,
    ST_PYR, 5, EA, EB, P3, P0, N0,
    /* case 163 */ 9,
    ST_PNT, 6, EB, EL, EG, EE, EI, ED,
    ST_TET, 4, P6, P4, P2, N0,
    ST_PYR, 5, P6, EG, EE, P4, N0,
    ST_TET, 4, P3, P6, P2, N0,
    ST_PYR, 5, P3, EL, EG, P6, N0,
    ST_TET, 4, EB, EL, P3, N0,
    ST_PYR, 5, P3, P2, ED, EB, N0,
    ST_PYR, 5, ED, P2, P4, EI, N0,
    ST_TET, 4, P4, EE, EI, N0,
    /* case 164 */ 11,
    ST_PNT, 4, EE, EJ, ED, ED,
    ST_PYR, 5, P0, P3, EC, ED, N0,
    ST_TET, 4, EC, P3, EL, N0,
    ST_PYR, 5, P1, EJ, EL, P3, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_TET, 4, P6, EK, EG, N0,
    ST_TET, 4, P4, P0, P6, N0,
    ST_PYR, 5, P6, P0, ED, EK, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_PYR, 5, P4, P6, EG, EE, N0,
    /* case 165 */ 2,
    ST_WDG, 6, P4, EI, EE, P6, EK, EG,
    ST_WDG, 6, P1, EJ, EA, P3, EL, EC,
    /* case 166 */ 8,
    ST_PNT, 5, EE, EG, EA, ED, EK,
    ST_TET, 4, P3, EB, EL, EC,
    ST_PYR, 5, P6, EG, EE, P4, N0,
    ST_PYR, 5, P4, EE, EA, P0, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_TET, 4, P4, P0, P6, N0,
    ST_PYR, 5, P0, ED, EK, P6, N0,
    ST_TET, 4, P6, EK, EG, N0,
    /* case 167 */ 2,
    ST_WDG, 6, EE, P4, EI, EG, P6, EK,
    ST_TET, 4, EB, P3, EC, EL,
    /* case 168 */ 9,
    ST_PNT, 5, EC, EG, EB, EJ, EE,
    ST_PYR, 5, P2, P6, P4, P0, N0,
    ST_TET, 4, P4, N0, P1, P0,
    ST_PYR, 5, P6, EG, EE, P4, N0,
    ST_PYR, 5, EC, EG, P6, P2, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_PYR, 5, P1, EB, EC, P2, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_PYR, 5, EE, EJ, P1, P4, N0,
    /* case 169 */ 8,
    ST_PNT, 5, EG, EE, EC, ED, EI,
    ST_TET, 4, P1, EJ, EB, EA,
    ST_PYR, 5, P4, P6, EG, EE, N0,
    ST_PYR, 5, P6, P2, EC, EG, N0,
    ST_TET, 4, P2, ED, EC, N0,
    ST_TET, 4, P6, P4, P2, N0,
    ST_PYR, 5, P2, P4, EI, ED, N0,
    ST_TET, 4, P4, EE, EI, N0,
    /* case 170 */ 1,
    ST_HEX, 8, EA, EE, EG, EC, P0, P4, P6, P2,
    /* case 171 */ 7,
    ST_PNT, 5, EE, EG, EC, EI, ED,
    ST_PYR, 5, P6, P2, EC, EG, N0,
    ST_TET, 4, P2, ED, EC, N0,
    ST_PYR, 5, P4, P6, EG, EE, N0,
    ST_TET, 4, EI, P4, EE, N0,
    ST_PYR, 5, ED, P2, P4, EI, N0,
    ST_TET, 4, P2, P6, P4, N0,
    /* case 172 */ 9,
    ST_PNT, 6, EB, EJ, EE, EG, EK, ED,
    ST_TET, 4, P4, P0, P6, N0,
    ST_PYR, 5, P4, P6, EG, EE, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_TET, 4, EB, P1, EJ, N0,
    ST_PYR, 5, P1, EB, ED, P0, N0,
    ST_PYR, 5, ED, EK, P6, P0, N0,
    ST_TET, 4, P6, EK, EG, N0,
    /* case 173 */ 2,
    ST_WDG, 6, EE, P4, EI, EG, P6, EK,
    ST_TET, 4, EB, EA, P1, EJ,
    /* case 174 */ 7,
    ST_PNT, 5, EG, EE, EA, EK, ED,
    ST_PYR, 5, P4, EE, EA, P0, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_PYR, 5, P6, EG, EE, P4, N0,
    ST_TET, 4, EK, EG, P6, N0,
    ST_PYR, 5, ED, EK, P6, P0, N0,
    ST_TET, 4, P0, P6, P4, N0,
    /* case 175 */ 1,
    ST_WDG, 6, P4, EI, EE, P6, EK, EG,
    /* case 176 */ 9,
    ST_PNT, 5, EL, EJ, EG, EH, EI,
    ST_PYR, 5, P3, P2, P0, P1, N0,
    ST_TET, 4, P0, P6, N0, P2,
    ST_PYR, 5, P1, P0, EI, EJ, N0,
    ST_PYR, 5, EL, P3, P1, EJ, N0,
    ST_TET, 4, P6, P2, P3, N0,
    ST_PYR, 5, P6, P3, EL, EG, N0,
    ST_TET, 4, P6, EG, EH, N0,
    ST_PYR, 5, EI, P0, P6, EH, N0,
    /* case 177 */ 9,
    ST_PNT, 6, EH, EG, EL, EJ, EA, ED,
    ST_TET, 4, P3, P2, P1, N0,
    ST_PYR, 5, P3, P1, EJ, EL, N0,
    ST_TET, 4, P6, P2, P3, N0,
    ST_PYR, 5, P6, P3, EL, EG, N0,
    ST_TET, 4, EH, P6, EG, N0,
    ST_PYR, 5, P6, EH, ED, P2, N0,
    ST_PYR, 5, ED, EA, P1, P2, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    /* case 178 */ 5,
    ST_TET, 4, P6, P0, P2, P3,
    ST_PYR, 5, EB, P3, P0, EA, EI,
    ST_PYR, 5, EL, EG, P6, P3, EH,
    ST_PYR, 5, P0, P6, EH, EI, P3,
    ST_PYR, 5, EI, EH, EL, EB, P3,
    /* case 179 */ 7,
    ST_PNT, 5, EH, ED, EB, EG, EL,
    ST_PYR, 5, P2, ED, EB, P3, N0,
    ST_TET, 4, P3, EB, EL, N0,
    ST_PYR, 5, P6, EH, ED, P2, N0,
    ST_TET, 4, EG, EH, P6, N0,
    ST_PYR, 5, EL, EG, P6, P3, N0,
    ST_TET, 4, P3, P6, P2, N0,
    /* case 180 */ 8,
    ST_PNT, 5, EJ, EL, EI, ED, EC,
    ST_TET, 4, P6, EG, EH, EK,
    ST_PYR, 5, P3, P1, EJ, EL, N0,
    ST_PYR, 5, P1, P0, EI, EJ, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_PYR, 5, P0, P3, EC, ED, N0,
    ST_TET, 4, P3, EL, EC, N0,
    /* case 181 */ 2,
    ST_WDG, 6, EL, P3, EC, EJ, P1, EA,
    ST_TET, 4, EH, EK, P6, EG,
    /* case 182 */ 3,
    ST_TET, 4, EB, EC, EL, P3,
    ST_TET, 4, EA, P0, EI, ED,
    ST_TET, 4, EG, EH, P6, EK,
    /* case 183 */ 2,
    ST_TET, 4, EG, EK, EH, P6,
    ST_TET, 4, EL, EB, EC, P3,
    /* case 184 */ 9,
    ST_PNT, 6, EJ, EB, EC, EG, EH, EI,
    ST_TET, 4, P2, P6, P0, N0,
    ST_PYR, 5, P2, EC, EG, P6, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_PYR, 5, P1, EB, EC, P2, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    ST_PYR, 5, P1, P0, EI, EJ, N0,
    ST_PYR, 5, EI, P0, P6, EH, N0,
    ST_TET, 4, P6, EG, EH, N0,
    /* case 185 */ 2,
    ST_WDG, 6, EG, P6, EH, EC, P2, ED,
    ST_TET, 4, EJ, P1, EA, EB,
    /* case 186 */ 7,
    ST_PNT, 5, EG, EC, EA, EH, EI,
    ST_PYR, 5, P2, P0, EA, EC, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_PYR, 5, P6, P2, EC, EG, N0,
    ST_TET, 4, EH, P6, EG, N0,
    ST_PYR, 5, EI, P0, P6, EH, N0,
    ST_TET, 4, P0, P2, P6, N0,
    /* case 187 */ 1,
    ST_WDG, 6, P6, EH, EG, P2, ED, EC,
    /* case 188 */ 2,
    ST_WDG, 6, ED, P0, EI, EB, P1, EJ,
    ST_TET, 4, EG, EH, P6, EK,
    /* case 189 */ 2,
    ST_TET, 4, EJ, EA, EB, P1,
    ST_TET, 4, EH, EG, EK, P6,
    /* case 190 */ 2,
    ST_TET, 4, EI, ED, EA, P0,
    ST_TET, 4, EH, EG, EK, P6,
    /* case 191 */ 1,
    ST_TET, 4, EG, EK, EH, P6,
    /* case 192 */ 2,
    ST_HEX, 8, EF, P5, P4, EH, EL, P3, P2, EK,
    ST_WDG, 6, P5, P1, P3, P4, P0, P2,
    /* case 193 */ 11,
    ST_PNT, 4, EL, EF, EA, EA,
    ST_PYR, 5, P1, EA, EI, P4, N0,
    ST_TET, 4, EI, EH, P4, N0,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_TET, 4, P1, P4, P5, N0,
    ST_TET, 4, P2, EK, ED, N0,
    ST_TET, 4, P3, P2, P1, N0,
    ST_PYR, 5, P2, ED, EA, P1, N0,
    ST_TET, 4, P3, P1, P5, N0,
    ST_PYR, 5, P3, P5, EF, EL, N0,
    ST_PYR, 5, P3, EL, EK, P2, N0,
    /* case 194 */ 11,
    ST_PNT, 4, EK, EH, EA, EA,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    ST_PYR, 5, P4, EH, EF, P5, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_TET, 4, P3, EB, EL, N0,
    ST_TET, 4, P2, P0, P3, N0,
    ST_PYR, 5, P3, P0, EA, EB, N0,
    ST_TET, 4, P2, P4, P0, N0,
    ST_PYR, 5, P2, EK, EH, P4, N0,
    ST_PYR, 5, P2, P3, EL, EK, N0,
    /* case 195 */ 2,
    ST_WDG, 6, P2, ED, EK, P3, EB, EL,
    ST_WDG, 6, P4, EH, EI, P5, EF, EJ,
    /* case 196 */ 9,
    ST_PNT, 5, EF, EH, EL, EC, ED,
    ST_PYR, 5, P5, P1, P0, P4, N0,
    ST_TET, 4, P0, P3, N0, P1,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_PYR, 5, EF, P5, P4, EH, N0,
    ST_TET, 4, P3, P1, P5, N0,
    ST_PYR, 5, P3, P5, EF, EL, N0,
    ST_TET, 4, P3, EL, EC, N0,
    ST_PYR, 5, ED, P0, P3, EC, N0,
    /* case 197 */ 9,
    ST_PNT, 6, EC, EL, EF, EH, EI, EA,
    ST_TET, 4, P5, P1, P4, N0,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_TET, 4, P3, P1, P5, N0,
    ST_PYR, 5, P3, P5, EF, EL, N0,
    ST_TET, 4, EC, P3, EL, N0,
    ST_PYR, 5, P3, EC, EA, P1, N0,
    ST_PYR, 5, EA, EI, P4, P1, N0,
    ST_TET, 4, P4, EI, EH, N0,
    /* case 198 */ 8,
    ST_PNT, 5, EH, EF, ED, EA, EJ,
    ST_TET, 4, P3, EL, EC, EB,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_TET, 4, P4, P5, P0, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    /* case 199 */ 2,
    ST_WDG, 6, EF, P5, EJ, EH, P4, EI,
    ST_TET, 4, EC, EB, P3, EL,
    /* case 200 */ 9,
    ST_PNT, 5, EH, EF, EK, EC, EB,
    ST_PYR, 5, P4, P5, P1, P0, N0,
    ST_TET, 4, P1, N0, P2, P0,
    ST_PYR, 5, P5, EF, EB, P1, N0,
    ST_PYR, 5, EH, EF, P5, P4, N0,
    ST_TET, 4, P2, P4, P0, N0,
    ST_PYR, 5, P2, EK, EH, P4, N0,
    ST_TET, 4, P2, EC, EK, N0,
    ST_PYR, 5, EB, EC, P2, P1, N0,
    /* case 201 */ 8,
    ST_PNT, 5, EF, EB, EH, EI, EA,
    ST_TET, 4, P2, EC, EK, ED,
    ST_PYR, 5, P1, P5, EF, EB, N0,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_TET, 4, P5, P1, P4, N0,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_TET, 4, P1, EB, EA, N0,
    /* case 202 */ 9,
    ST_PNT, 6, EC, EK, EH, EF, EJ, EA,
    ST_TET, 4, P4, P5, P0, N0,
    ST_PYR, 5, P4, EH, EF, P5, N0,
    ST_TET, 4, P2, P4, P0, N0,
    ST_PYR, 5, P2, EK, EH, P4, N0,
    ST_TET, 4, EC, EK, P2, N0,
    ST_PYR, 5, P2, P0, EA, EC, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    /* case 203 */ 2,
    ST_WDG, 6, EF, P5, EJ, EH, P4, EI,
    ST_TET, 4, EC, P2, ED, EK,
    /* case 204 */ 1,
    ST_HEX, 8, ED, EB, EF, EH, P0, P1, P5, P4,
    /* case 205 */ 7,
    ST_PNT, 5, EB, EF, EH, EA, EI,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_PYR, 5, P1, P5, EF, EB, N0,
    ST_TET, 4, EA, P1, EB, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    ST_TET, 4, P4, P5, P1, N0,
    /* case 206 */ 7,
    ST_PNT, 5, EF, EH, ED, EJ, EA,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_TET, 4, P0, P4, P5, N0,
    /* case 207 */ 1,
    ST_WDG, 6, P5, EJ, EF, P4, EI, EH,
    /* case 208 */ 9,
    ST_PNT, 5, EL, EK, EF, EE, EI,
    ST_PYR, 5, P3, P2, P0, P1, N0,
    ST_TET, 4, P0, N0, P5, P1,
    ST_PYR, 5, P2, EK, EI, P0, N0,
    ST_PYR, 5, EL, EK, P2, P3, N0,
    ST_TET, 4, P5, P3, P1, N0,
    ST_PYR, 5, P5, EF, EL, P3, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_PYR, 5, EI, EE, P5, P0, N0,
    /* case 209 */ 9,
    ST_PNT, 6, EE, EF, EL, EK, ED, EA,
    ST_TET, 4, P3, P2, P1, N0,
    ST_PYR, 5, P3, EL, EK, P2, N0,
    ST_TET, 4, P5, P3, P1, N0,
    ST_PYR, 5, P5, EF, EL, P3, N0,
    ST_TET, 4, EE, EF, P5, N0,
    ST_PYR, 5, P5, P1, EA, EE, N0,
    ST_PYR, 5, EA, P1, P2, ED, N0,
    ST_TET, 4, P2, EK, ED, N0,
    /* case 210 */ 8,
    ST_PNT, 5, EK, EL, EI, EA, EB,
    ST_TET, 4, P5, EE, EF, EJ,
    ST_PYR, 5, P3, EL, EK, P2, N0,
    ST_PYR, 5, P2, EK, EI, P0, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_TET, 4, P2, P0, P3, N0,
    ST_PYR, 5, P0, EA, EB, P3, N0,
    ST_TET, 4, P3, EB, EL, N0,
    /* case 211 */ 2,
    ST_WDG, 6, EK, P2, ED, EL, P3, EB,
    ST_TET, 4, EE, P5, EJ, EF,
    /* case 212 */ 5,
    ST_TET, 4, P3, P0, P1, P5,
    ST_PYR, 5, EE, P5, P0, EI, ED,
    ST_PYR, 5, EF, EL, P3, P5, EC,
    ST_PYR, 5, P0, P3, EC, ED, P5,
    ST_PYR, 5, ED, EC, EF, EE, P5,
    /* case 213 */ 7,
    ST_PNT, 5, EC, EA, EE, EL, EF,
    ST_PYR, 5, P1, EA, EE, P5, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_PYR, 5, P3, EC, EA, P1, N0,
    ST_TET, 4, EL, EC, P3, N0,
    ST_PYR, 5, EF, EL, P3, P5, N0,
    ST_TET, 4, P5, P3, P1, N0,
    /* case 214 */ 3,
    ST_TET, 4, EE, EJ, EF, P5,
    ST_TET, 4, EI, P0, ED, EA,
    ST_TET, 4, EL, EC, P3, EB,
    /* case 215 */ 2,
    ST_TET, 4, EL, EB, EC, P3,
    ST_TET, 4, EF, EE, EJ, P5,
    /* case 216 */ 9,
    ST_PNT, 6, EK, EC, EB, EF, EE, EI,
    ST_TET, 4, P1, P0, P5, N0,
    ST_PYR, 5, P1, P5, EF, EB, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_PYR, 5, P2, P1, EB, EC, N0,
    ST_TET, 4, EK, P2, EC, N0,
    ST_PYR, 5, P2, EK, EI, P0, N0,
    ST_PYR, 5, EI, EE, P5, P0, N0,
    ST_TET, 4, P5, EE, EF, N0,
    /* case 217 */ 2,
    ST_WDG, 6, EB, P1, EA, EF, P5, EE,
    ST_TET, 4, EK, ED, P2, EC,
    /* case 218 */ 2,
    ST_WDG, 6, EI, P0, EA, EK, P2, EC,
    ST_TET, 4, EF, EJ, P5, EE,
    /* case 219 */ 2,
    ST_TET, 4, EC, ED, EK, P2,
    ST_TET, 4, EJ, EF, EE, P5,
    /* case 220 */ 7,
    ST_PNT, 5, EF, EB, ED, EE, EI,
    ST_PYR, 5, P1, EB, ED, P0, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_PYR, 5, P5, EF, EB, P1, N0,
    ST_TET, 4, EE, EF, P5, N0,
    ST_PYR, 5, EI, EE, P5, P0, N0,
    ST_TET, 4, P0, P5, P1, N0,
    /* case 221 */ 1,
    ST_WDG, 6, P1, EA, EB, P5, EE, EF,
    /* case 222 */ 2,
    ST_TET, 4, EA, EI, ED, P0,
    ST_TET, 4, EJ, EF, EE, P5,
    /* case 223 */ 1,
    ST_TET, 4, EF, EE, EJ, P5,
    /* case 224 */ 9,
    ST_PNT, 5, EK, EL, EH, EE, EJ,
    ST_PYR, 5, P2, P0, P1, P3, N0,
    ST_TET, 4, P1, P4, N0, P0,
    ST_PYR, 5, P3, P1, EJ, EL, N0,
    ST_PYR, 5, EK, P2, P3, EL, N0,
    ST_TET, 4, P4, P0, P2, N0,
    ST_PYR, 5, P4, P2, EK, EH, N0,
    ST_TET, 4, P4, EH, EE, N0,
    ST_PYR, 5, EJ, P1, P4, EE, N0,
    /* case 225 */ 8,
    ST_PNT, 5, EL, EJ, EK, ED, EA,
    ST_TET, 4, P4, EH, EE, EI,
    ST_PYR, 5, P1, EJ, EL, P3, N0,
    ST_PYR, 5, P3, EL, EK, P2, N0,
    ST_TET, 4, P2, EK, ED, N0,
    ST_TET, 4, P3, P2, P1, N0,
    ST_PYR, 5, P2, ED, EA, P1, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    /* case 226 */ 9,
    ST_PNT, 6, EE, EH, EK, EL, EB, EA,
    ST_TET, 4, P2, P0, P3, N0,
    ST_PYR, 5, P2, P3, EL, EK, N0,
    ST_TET, 4, P4, P0, P2, N0,
    ST_PYR, 5, P4, P2, EK, EH, N0,
    ST_TET, 4, EE, P4, EH, N0,
    ST_PYR, 5, P4, EE, EA, P0, N0,
    ST_PYR, 5, EA, EB, P3, P0, N0,
    ST_TET, 4, P3, EB, EL, N0,
    /* case 227 */ 2,
    ST_WDG, 6, EK, P2, ED, EL, P3, EB,
    ST_TET, 4, EE, EI, P4, EH,
    /* case 228 */ 9,
    ST_PNT, 6, EH, EE, EJ, EL, EC, ED,
    ST_TET, 4, P1, P3, P0, N0,
    ST_PYR, 5, P1, EJ, EL, P3, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_TET, 4, EH, EE, P4, N0,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_PYR, 5, ED, P0, P3, EC, N0,
    ST_TET, 4, P3, EL, EC, N0,
    /* case 229 */ 2,
    ST_WDG, 6, EL, P3, EC, EJ, P1, EA,
    ST_TET, 4, EH, P4, EI, EE,
    /* case 230 */ 2,
    ST_WDG, 6, EH, P4, EE, ED, P0, EA,
    ST_TET, 4, EL, P3, EB, EC,
    /* case 231 */ 2,
    ST_TET, 4, EE, EH, EI, P4,
    ST_TET, 4, EB, EC, EL, P3,
    /* case 232 */ 5,
    ST_TET, 4, P2, P0, P1, P4,
    ST_PYR, 5, EE, EJ, P1, P4, EB,
    ST_PYR, 5, EH, P4, P2, EK, EC,
    ST_PYR, 5, P1, EB, EC, P2, P4,
    ST_PYR, 5, EB, EE, EH, EC, P4,
    /* case 233 */ 3,
    ST_TET, 4, EJ, EA, EB, P1,
    ST_TET, 4, EE, P4, EH, EI,
    ST_TET, 4, EC, EK, P2, ED,
    /* case 234 */ 7,
    ST_PNT, 5, EC, EA, EE, EK, EH,
    ST_PYR, 5, P0, P4, EE, EA, N0,
    ST_TET, 4, P4, EH, EE, N0,
    ST_PYR, 5, P2, P0, EA, EC, N0,
    ST_TET, 4, EK, P2, EC, N0,
    ST_PYR, 5, EH, P4, P2, EK, N0,
    ST_TET, 4, P4, P0, P2, N0,
    /* case 235 */ 2,
    ST_TET, 4, EK, EC, ED, P2,
    ST_TET, 4, EH, EI, EE, P4,
    /* case 236 */ 7,
    ST_PNT, 5, EH, ED, EB, EE, EJ,
    ST_PYR, 5, P0, P1, EB, ED, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_TET, 4, EE, P4, EH, N0,
    ST_PYR, 5, EJ, P1, P4, EE, N0,
    ST_TET, 4, P1, P0, P4, N0,
    /* case 237 */ 2,
    ST_TET, 4, EE, EH, EI, P4,
    ST_TET, 4, EJ, EA, EB, P1,
    /* case 238 */ 1,
    ST_WDG, 6, P4, EE, EH, P0, EA, ED,
    /* case 239 */ 1,
    ST_TET, 4, EH, EI, EE, P4,
    /* case 240 */ 1,
    ST_HEX, 8, P0, P1, P3, P2, EI, EJ, EL, EK,
    /* case 241 */ 7,
    ST_PNT, 5, EJ, EL, EK, EA, ED,
    ST_PYR, 5, P3, EL, EK, P2, N0,
    ST_TET, 4, P2, EK, ED, N0,
    ST_PYR, 5, P1, EJ, EL, P3, N0,
    ST_TET, 4, EA, EJ, P1, N0,
    ST_PYR, 5, ED, EA, P1, P2, N0,
    ST_TET, 4, P2, P1, P3, N0,
    /* case 242 */ 7,
    ST_PNT, 5, EL, EK, EI, EB, EA,
    ST_PYR, 5, P2, EK, EI, P0, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_PYR, 5, P3, EL, EK, P2, N0,
    ST_TET, 4, EB, EL, P3, N0,
    ST_PYR, 5, EA, EB, P3, P0, N0,
    ST_TET, 4, P0, P3, P2, N0,
    /* case 243 */ 1,
    ST_WDG, 6, P2, ED, EK, P3, EB, EL,
    /* case 244 */ 7,
    ST_PNT, 5, EL, EJ, EI, EC, ED,
    ST_PYR, 5, P1, P0, EI, EJ, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_PYR, 5, P3, P1, EJ, EL, N0,
    ST_TET, 4, EC, P3, EL, N0,
    ST_PYR, 5, ED, P0, P3, EC, N0,
    ST_TET, 4, P0, P1, P3, N0,
    /* case 245 */ 1,
    ST_WDG, 6, P3, EC, EL, P1, EA, EJ,
    /* case 246 */ 2,
    ST_TET, 4, EA, EI, ED, P0,
    ST_TET, 4, EB, EC, EL, P3,
    /* case 247 */ 1,
    ST_TET, 4, EL, EB, EC, P3,
    /* case 248 */ 7,
    ST_PNT, 5, EK, EI, EJ, EC, EB,
    ST_PYR, 5, P0, EI, EJ, P1, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_PYR, 5, P2, EK, EI, P0, N0,
    ST_TET, 4, EC, EK, P2, N0,
    ST_PYR, 5, EB, EC, P2, P1, N0,
    ST_TET, 4, P1, P2, P0, N0,
    /* case 249 */ 2,
    ST_TET, 4, EC, ED, EK, P2,
    ST_TET, 4, EB, EJ, EA, P1,
    /* case 250 */ 1,
    ST_WDG, 6, P0, EA, EI, P2, EC, EK,
    /* case 251 */ 1,
    ST_TET, 4, EK, EC, ED, P2,
    /* case 252 */ 1,
    ST_WDG, 6, P0, EI, ED, P1, EJ, EB,
    /* case 253 */ 1,
    ST_TET, 4, EB, EJ, EA, P1,
    /* case 254 */ 1,
    ST_TET, 4, ED, EA, EI, P0,
    /* case 255 */ 0,
    // VTK_HEXAHEDRON
    /* case 0 */ 1,
    ST_HEX, 8, P0, P1, P2, P3, P4, P5, P6, P7,
    /* case 1 */ 9,
    ST_PNT, 7, P1, P2, P3, P4, P5, P6, P7,
    ST_WDG, 6, P1, P3, P4, EA, ED, EI,
    ST_TET, 4, P1, P3, P4, N0,
    ST_TET, 4, P1, P2, P3, N0,
    ST_PYR, 5, P6, P7, P3, P2, N0,
    ST_PYR, 5, P5, P6, P2, P1, N0,
    ST_PYR, 5, P4, P7, P6, P5, N0,
    ST_TET, 4, P3, P7, P4, N0,
    ST_TET, 4, P4, P5, P1, N0,
    /* case 2 */ 9,
    ST_PNT, 7, P5, P4, P0, P2, P6, P7, P3,
    ST_WDG, 6, EJ, EA, EB, P5, P0, P2,
    ST_TET, 4, P5, P2, P0, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_PYR, 5, P7, P4, P0, P3, N0,
    ST_PYR, 5, P6, P5, P4, P7, N0,
    ST_PYR, 5, P2, P6, P7, P3, N0,
    ST_TET, 4, P0, P2, P3, N0,
    ST_TET, 4, P2, P5, P6, N0,
    /* case 3 */ 2,
    ST_HEX, 8, EB, P2, P3, ED, EJ, P5, P4, EI,
    ST_WDG, 6, P2, P6, P5, P3, P7, P4,
    /* case 4 */ 9,
    ST_PNT, 7, P6, P5, P1, P3, P7, P4, P0,
    ST_WDG, 6, EL, EB, EC, P6, P1, P3,
    ST_TET, 4, P6, P3, P1, N0,
    ST_TET, 4, P6, P1, P5, N0,
    ST_PYR, 5, P4, P5, P1, P0, N0,
    ST_PYR, 5, P7, P6, P5, P4, N0,
    ST_PYR, 5, P3, P7, P4, P0, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_TET, 4, P3, P6, P7, N0,
    /* case 5 */ 12,
    ST_PNT, 2, EI, EL,
    ST_PYR, 5, P4, P7, P6, P5, N0,
    ST_TET, 4, P5, P6, P1, N0,
    ST_TET, 4, P4, P5, P1, N0,
    ST_TET, 4, P3, P7, P4, N0,
    ST_TET, 4, P6, P7, P3, N0,
    ST_PYR, 5, P6, P3, EC, EL, N0,
    ST_PYR, 5, P1, P6, EL, EB, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_PYR, 5, P4, EI, ED, P3, N0,
    ST_TET, 4, P3, ED, EC, N0,
    /* case 6 */ 2,
    ST_HEX, 8, EC, P3, P0, EA, EL, P6, P5, EJ,
    ST_WDG, 6, P3, P7, P6, P0, P4, P5,
    /* case 7 */ 9,
    ST_PNT, 5, EI, EJ, ED, EC, EL,
    ST_PYR, 5, P4, P7, P6, P5, N0,
    ST_TET, 4, P6, P3, N0, P7,
    ST_PYR, 5, P5, P6, EL, EJ, N0,
    ST_PYR, 5, EI, P4, P5, EJ, N0,
    ST_TET, 4, P3, P7, P4, N0,
    ST_PYR, 5, P3, P4, EI, ED, N0,
    ST_TET, 4, P3, ED, EC, N0,
    ST_PYR, 5, EL, P6, P3, EC, N0,
    /* case 8 */ 9,
    ST_PNT, 7, P2, P1, P0, P7, P6, P5, P4,
    ST_WDG, 6, EC, ED, EK, P2, P0, P7,
    ST_TET, 4, P2, P7, P0, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_PYR, 5, P5, P1, P0, P4, N0,
    ST_PYR, 5, P6, P2, P1, P5, N0,
    ST_PYR, 5, P7, P6, P5, P4, N0,
    ST_TET, 4, P0, P7, P4, N0,
    ST_TET, 4, P7, P2, P6, N0,
    /* case 9 */ 2,
    ST_HEX, 8, EK, P7, P4, EI, EC, P2, P1, EA,
    ST_WDG, 6, P7, P6, P2, P4, P5, P1,
    /* case 10 */ 12,
    ST_PNT, 2, EK, EJ,
    ST_PYR, 5, P7, P6, P5, P4, N0,
    ST_TET, 4, P6, P2, P5, N0,
    ST_TET, 4, P7, P2, P6, N0,
    ST_TET, 4, P0, P7, P4, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_PYR, 5, P5, EJ, EA, P0, N0,
    ST_PYR, 5, P2, EB, EJ, P5, N0,
    ST_TET, 4, P2, EC, EB, N0,
    ST_PYR, 5, P7, EK, EC, P2, N0,
    ST_PYR, 5, P7, P0, ED, EK, N0,
    ST_TET, 4, P0, EA, ED, N0,
    /* case 11 */ 9,
    ST_PNT, 5, EJ, EI, EB, EC, EK,
    ST_PYR, 5, P5, P4, P7, P6, N0,
    ST_TET, 4, P7, N0, P2, P6,
    ST_PYR, 5, P4, EI, EK, P7, N0,
    ST_PYR, 5, EJ, EI, P4, P5, N0,
    ST_TET, 4, P2, P5, P6, N0,
    ST_PYR, 5, P2, EB, EJ, P5, N0,
    ST_TET, 4, P2, EC, EB, N0,
    ST_PYR, 5, EK, EC, P2, P7, N0,
    /* case 12 */ 2,
    ST_HEX, 8, EL, P6, P7, EK, EB, P1, P0, ED,
    ST_WDG, 6, P0, P4, P7, P1, P5, P6,
    /* case 13 */ 9,
    ST_PNT, 5, EI, EK, EA, EB, EL,
    ST_PYR, 5, P4, P7, P6, P5, N0,
    ST_TET, 4, P6, N0, P1, P5,
    ST_PYR, 5, P7, EK, EL, P6, N0,
    ST_PYR, 5, EI, EK, P7, P4, N0,
    ST_TET, 4, P1, P4, P5, N0,
    ST_PYR, 5, P1, EA, EI, P4, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_PYR, 5, EL, EB, P1, P6, N0,
    /* case 14 */ 9,
    ST_PNT, 5, EK, EL, ED, EA, EJ,
    ST_PYR, 5, P7, P6, P5, P4, N0,
    ST_TET, 4, P5, N0, P0, P4,
    ST_PYR, 5, P6, EL, EJ, P5, N0,
    ST_PYR, 5, EK, EL, P6, P7, N0,
    ST_TET, 4, P0, P7, P4, N0,
    ST_PYR, 5, P0, ED, EK, P7, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_PYR, 5, EJ, EA, P0, P5, N0,
    /* case 15 */ 1,
    ST_HEX, 8, EI, EJ, EL, EK, P4, P5, P6, P7,
    /* case 16 */ 9,
    ST_PNT, 7, P5, P1, P0, P7, P6, P2, P3,
    ST_WDG, 6, P5, P0, P7, EE, EI, EH,
    ST_TET, 4, P5, P0, P7, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_PYR, 5, P2, P3, P0, P1, N0,
    ST_PYR, 5, P6, P2, P1, P5, N0,
    ST_PYR, 5, P7, P3, P2, P6, N0,
    ST_TET, 4, P0, P3, P7, N0,
    ST_TET, 4, P7, P6, P5, N0,
    /* case 17 */ 2,
    ST_HEX, 8, EE, P5, P1, EA, EH, P7, P3, ED,
    ST_WDG, 6, P3, P2, P1, P7, P6, P5,
    /* case 18 */ 12,
    ST_PNT, 2, EH, EB,
    ST_PYR, 5, P7, P3, P2, P6, N0,
    ST_TET, 4, P6, P2, P5, N0,
    ST_TET, 4, P7, P6, P5, N0,
    ST_TET, 4, P0, P3, P7, N0,
    ST_TET, 4, P2, P3, P0, N0,
    ST_PYR, 5, P2, P0, EA, EB, N0,
    ST_PYR, 5, P5, P2, EB, EJ, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_PYR, 5, P7, P5, EE, EH, N0,
    ST_PYR, 5, P7, EH, EI, P0, N0,
    ST_TET, 4, P0, EI, EA, N0,
    /* case 19 */ 9,
    ST_PNT, 5, EB, ED, EJ, EE, EH,
    ST_PYR, 5, P2, P6, P7, P3, N0,
    ST_TET, 4, P7, P5, N0, P6,
    ST_PYR, 5, P3, P7, EH, ED, N0,
    ST_PYR, 5, EB, P2, P3, ED, N0,
    ST_TET, 4, P5, P6, P2, N0,
    ST_PYR, 5, P5, P2, EB, EJ, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_PYR, 5, EH, P7, P5, EE, N0,
    /* case 20 */ 6,
    ST_WDG, 6, EB, EC, EL, P1, P3, P6,
    ST_WDG, 6, P0, P7, P5, EI, EH, EE,
    ST_TET, 4, P3, P1, P6, P7,
    ST_TET, 4, P5, P7, P6, P1,
    ST_TET, 4, P0, P5, P1, P7,
    ST_TET, 4, P3, P7, P0, P1,
    /* case 21 */ 11,
    ST_PNT, 4, EE, EH, EL, EL,
    ST_PYR, 5, P6, P3, EC, EL, N0,
    ST_TET, 4, EC, P3, ED, N0,
    ST_PYR, 5, P7, EH, ED, P3, N0,
    ST_TET, 4, P6, P7, P3, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_TET, 4, P5, P6, P1, N0,
    ST_PYR, 5, P1, P6, EL, EB, N0,
    ST_TET, 4, P5, P7, P6, N0,
    ST_PYR, 5, P5, EE, EH, P7, N0,
    ST_PYR, 5, P5, P1, EA, EE, N0,
    /* case 22 */ 11,
    ST_PNT, 4, EL, EC, EH, EH,
    ST_PYR, 5, P7, EH, EI, P0, N0,
    ST_TET, 4, EI, EA, P0, N0,
    ST_PYR, 5, P3, P0, EA, EC, N0,
    ST_TET, 4, P7, P0, P3, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_TET, 4, P6, P5, P7, N0,
    ST_PYR, 5, P5, EE, EH, P7, N0,
    ST_TET, 4, P6, P7, P3, N0,
    ST_PYR, 5, P6, P3, EC, EL, N0,
    ST_PYR, 5, P6, EL, EJ, P5, N0,
    /* case 23 */ 9,
    ST_PNT, 6, ED, EC, EL, EJ, EE, EH,
    ST_TET, 4, P6, P5, P7, N0,
    ST_PYR, 5, P6, EL, EJ, P5, N0,
    ST_TET, 4, P3, P6, P7, N0,
    ST_PYR, 5, P3, EC, EL, P6, N0,
    ST_TET, 4, ED, EC, P3, N0,
    ST_PYR, 5, P3, P7, EH, ED, N0,
    ST_PYR, 5, EH, P7, P5, EE, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    /* case 24 */ 12,
    ST_PNT, 2, EC, EE,
    ST_PYR, 5, P2, P1, P5, P6, N0,
    ST_TET, 4, P6, P5, P7, N0,
    ST_TET, 4, P2, P6, P7, N0,
    ST_TET, 4, P0, P1, P2, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_PYR, 5, P5, P0, EI, EE, N0,
    ST_PYR, 5, P7, P5, EE, EH, N0,
    ST_TET, 4, P7, EH, EK, N0,
    ST_PYR, 5, P2, P7, EK, EC, N0,
    ST_PYR, 5, P2, EC, ED, P0, N0,
    ST_TET, 4, P0, ED, EI, N0,
    /* case 25 */ 9,
    ST_PNT, 5, EE, EA, EH, EK, EC,
    ST_PYR, 5, P5, P6, P2, P1, N0,
    ST_TET, 4, P2, P7, N0, P6,
    ST_PYR, 5, P1, P2, EC, EA, N0,
    ST_PYR, 5, EE, P5, P1, EA, N0,
    ST_TET, 4, P7, P6, P5, N0,
    ST_PYR, 5, P7, P5, EE, EH, N0,
    ST_TET, 4, P7, EH, EK, N0,
    ST_PYR, 5, EC, P2, P7, EK, N0,
    /* case 26 */ 6,
    ST_TET, 4, P0, EA, ED, EI,
    ST_TET, 4, P5, P7, P6, P2,
    ST_PYR, 5, EC, P2, P7, EK, EH,
    ST_PYR, 5, EB, EJ, P5, P2, EE,
    ST_PYR, 5, P7, P5, EE, EH, P2,
    ST_PYR, 5, EH, EE, EB, EC, P2,
    /* case 27 */ 5,
    ST_TET, 4, P5, P7, P6, P2,
    ST_PYR, 5, EC, P2, P7, EK, EH,
    ST_PYR, 5, EB, EJ, P5, P2, EE,
    ST_PYR, 5, P7, P5, EE, EH, P2,
    ST_PYR, 5, EH, EE, EB, EC, P2,
    /* case 28 */ 11,
    ST_PNT, 4, EL, EB, EE, EE,
    ST_PYR, 5, P5, P0, EI, EE, N0,
    ST_TET, 4, EI, P0, ED, N0,
    ST_PYR, 5, P1, EB, ED, P0, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_TET, 4, P7, EH, EK, N0,
    ST_TET, 4, P6, P5, P7, N0,
    ST_PYR, 5, P7, P5, EE, EH, N0,
    ST_TET, 4, P6, P1, P5, N0,
    ST_PYR, 5, P6, EL, EB, P1, N0,
    ST_PYR, 5, P6, P7, EK, EL, N0,
    /* case 29 */ 9,
    ST_PNT, 6, EA, EB, EL, EK, EH, EE,
    ST_TET, 4, P6, P5, P7, N0,
    ST_PYR, 5, P6, P7, EK, EL, N0,
    ST_TET, 4, P1, P5, P6, N0,
    ST_PYR, 5, P1, P6, EL, EB, N0,
    ST_TET, 4, EA, P1, EB, N0,
    ST_PYR, 5, P1, EA, EE, P5, N0,
    ST_PYR, 5, EE, EH, P7, P5, N0,
    ST_TET, 4, P7, EH, EK, N0,
    /* case 30 */ 8,
    ST_PNT, 5, EL, EJ, EK, EH, EE,
    ST_TET, 4, P0, EA, ED, EI,
    ST_PYR, 5, P5, P6, EL, EJ, N0,
    ST_PYR, 5, P6, P7, EK, EL, N0,
    ST_TET, 4, P7, EH, EK, N0,
    ST_TET, 4, P6, P5, P7, N0,
    ST_PYR, 5, P7, P5, EE, EH, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    /* case 31 */ 7,
    ST_PNT, 5, EJ, EL, EK, EE, EH,
    ST_PYR, 5, P6, P7, EK, EL, N0,
    ST_TET, 4, P7, EH, EK, N0,
    ST_PYR, 5, P5, P6, EL, EJ, N0,
    ST_TET, 4, EE, P5, EJ, N0,
    ST_PYR, 5, EH, P7, P5, EE, N0,
    ST_TET, 4, P7, P6, P5, N0,
    /* case 32 */ 9,
    ST_PNT, 7, P6, P2, P1, P4, P7, P3, P0,
    ST_WDG, 6, P6, P1, P4, EF, EJ, EE,
    ST_TET, 4, P6, P1, P4, N0,
    ST_TET, 4, P6, P2, P1, N0,
    ST_PYR, 5, P3, P0, P1, P2, N0,
    ST_PYR, 5, P7, P3, P2, P6, N0,
    ST_PYR, 5, P4, P0, P3, P7, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_TET, 4, P4, P7, P6, N0,
    /* case 33 */ 12,
    ST_PNT, 2, ED, EF,
    ST_PYR, 5, P3, P2, P6, P7, N0,
    ST_TET, 4, P2, P1, P6, N0,
    ST_TET, 4, P3, P1, P2, N0,
    ST_TET, 4, P4, P3, P7, N0,
    ST_TET, 4, P6, P4, P7, N0,
    ST_PYR, 5, P6, EF, EE, P4, N0,
    ST_PYR, 5, P1, EJ, EF, P6, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_PYR, 5, P3, ED, EA, P1, N0,
    ST_PYR, 5, P3, P4, EI, ED, N0,
    ST_TET, 4, P4, EE, EI, N0,
    /* case 34 */ 2,
    ST_HEX, 8, EF, P6, P2, EB, EE, P4, P0, EA,
    ST_WDG, 6, P0, P3, P2, P4, P7, P6,
    /* case 35 */ 9,
    ST_PNT, 5, ED, EB, EI, EE, EF,
    ST_PYR, 5, P3, P2, P6, P7, N0,
    ST_TET, 4, P6, N0, P4, P7,
    ST_PYR, 5, P2, EB, EF, P6, N0,
    ST_PYR, 5, ED, EB, P2, P3, N0,
    ST_TET, 4, P4, P3, P7, N0,
    ST_PYR, 5, P4, EI, ED, P3, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_PYR, 5, EF, EE, P4, P6, N0,
    /* case 36 */ 12,
    ST_PNT, 2, EC, EE,
    ST_PYR, 5, P3, P7, P4, P0, N0,
    ST_TET, 4, P7, P6, P4, N0,
    ST_TET, 4, P3, P6, P7, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_PYR, 5, P6, EF, EE, P4, N0,
    ST_TET, 4, P6, EL, EF, N0,
    ST_PYR, 5, P3, EC, EL, P6, N0,
    ST_PYR, 5, P3, P1, EB, EC, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    /* case 37 */ 6,
    ST_TET, 4, P1, EA, EJ, EB,
    ST_TET, 4, P6, P7, P3, P4,
    ST_PYR, 5, EI, ED, P3, P4, EC,
    ST_PYR, 5, EE, P4, P6, EF, EL,
    ST_PYR, 5, P3, EC, EL, P6, P4,
    ST_PYR, 5, EC, EI, EE, EL, P4,
    /* case 38 */ 9,
    ST_PNT, 5, EE, EA, EF, EL, EC,
    ST_PYR, 5, P4, P0, P3, P7, N0,
    ST_TET, 4, P3, N0, P6, P7,
    ST_PYR, 5, P0, EA, EC, P3, N0,
    ST_PYR, 5, EE, EA, P0, P4, N0,
    ST_TET, 4, P6, P4, P7, N0,
    ST_PYR, 5, P6, EF, EE, P4, N0,
    ST_TET, 4, P6, EL, EF, N0,
    ST_PYR, 5, EC, EL, P6, P3, N0,
    /* case 39 */ 5,
    ST_TET, 4, P6, P7, P3, P4,
    ST_PYR, 5, EI, ED, P3, P4, EC,
    ST_PYR, 5, EE, P4, P6, EF, EL,
    ST_PYR, 5, P3, EC, EL, P6, P4,
    ST_PYR, 5, EC, EI, EE, EL, P4,
    /* case 40 */ 6,
    ST_WDG, 6, P1, P4, P6, EJ, EE, EF,
    ST_WDG, 6, ED, EK, EC, P0, P7, P2,
    ST_TET, 4, P4, P6, P1, P7,
    ST_TET, 4, P2, P6, P7, P1,
    ST_TET, 4, P0, P1, P2, P7,
    ST_TET, 4, P4, P0, P7, P1,
    /* case 41 */ 11,
    ST_PNT, 4, EC, EK, EF, EF,
    ST_PYR, 5, P6, EF, EE, P4, N0,
    ST_TET, 4, EE, EI, P4, N0,
    ST_PYR, 5, P7, P4, EI, EK, N0,
    ST_TET, 4, P6, P4, P7, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_TET, 4, P2, P1, P6, N0,
    ST_PYR, 5, P1, EJ, EF, P6, N0,
    ST_TET, 4, P2, P6, P7, N0,
    ST_PYR, 5, P2, P7, EK, EC, N0,
    ST_PYR, 5, P2, EC, EA, P1, N0,
    /* case 42 */ 11,
    ST_PNT, 4, EF, EE, EK, EK,
    ST_PYR, 5, P7, P0, ED, EK, N0,
    ST_TET, 4, ED, P0, EA, N0,
    ST_PYR, 5, P4, EE, EA, P0, N0,
    ST_TET, 4, P7, P4, P0, N0,
    ST_TET, 4, P2, EC, EB, N0,
    ST_TET, 4, P6, P7, P2, N0,
    ST_PYR, 5, P2, P7, EK, EC, N0,
    ST_TET, 4, P6, P4, P7, N0,
    ST_PYR, 5, P6, EF, EE, P4, N0,
    ST_PYR, 5, P6, P2, EB, EF, N0,
    /* case 43 */ 9,
    ST_PNT, 6, EI, EE, EF, EB, EC, EK,
    ST_TET, 4, P6, P7, P2, N0,
    ST_PYR, 5, P6, P2, EB, EF, N0,
    ST_TET, 4, P4, P7, P6, N0,
    ST_PYR, 5, P4, P6, EF, EE, N0,
    ST_TET, 4, EI, P4, EE, N0,
    ST_PYR, 5, P4, EI, EK, P7, N0,
    ST_PYR, 5, EK, EC, P2, P7, N0,
    ST_TET, 4, P2, EC, EB, N0,
    /* case 44 */ 11,
    ST_PNT, 4, EK, ED, EE, EE,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    ST_PYR, 5, P0, P1, EB, ED, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_TET, 4, P6, EL, EF, N0,
    ST_TET, 4, P7, P6, P4, N0,
    ST_PYR, 5, P6, EF, EE, P4, N0,
    ST_TET, 4, P7, P4, P0, N0,
    ST_PYR, 5, P7, P0, ED, EK, N0,
    ST_PYR, 5, P7, EK, EL, P6, N0,
    /* case 45 */ 8,
    ST_PNT, 5, EK, EL, EI, EE, EF,
    ST_TET, 4, P1, EB, EA, EJ,
    ST_PYR, 5, P6, P7, EK, EL, N0,
    ST_PYR, 5, P7, P4, EI, EK, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_TET, 4, P7, P6, P4, N0,
    ST_PYR, 5, P4, P6, EF, EE, N0,
    ST_TET, 4, P6, EL, EF, N0,
    /* case 46 */ 9,
    ST_PNT, 6, EA, ED, EK, EL, EF, EE,
    ST_TET, 4, P7, P6, P4, N0,
    ST_PYR, 5, P7, EK, EL, P6, N0,
    ST_TET, 4, P0, P7, P4, N0,
    ST_PYR, 5, P0, ED, EK, P7, N0,
    ST_TET, 4, EA, ED, P0, N0,
    ST_PYR, 5, P0, P4, EE, EA, N0,
    ST_PYR, 5, EE, P4, P6, EF, N0,
    ST_TET, 4, P6, EL, EF, N0,
    /* case 47 */ 7,
    ST_PNT, 5, EL, EK, EI, EF, EE,
    ST_PYR, 5, P7, P4, EI, EK, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_PYR, 5, P6, P7, EK, EL, N0,
    ST_TET, 4, EF, P6, EL, N0,
    ST_PYR, 5, EE, P4, P6, EF, N0,
    ST_TET, 4, P4, P7, P6, N0,
    /* case 48 */ 2,
    ST_HEX, 8, EJ, P1, P0, EI, EF, P6, P7, EH,
    ST_WDG, 6, P1, P2, P6, P0, P3, P7,
    /* case 49 */ 9,
    ST_PNT, 5, ED, EH, EA, EJ, EF,
    ST_PYR, 5, P3, P2, P6, P7, N0,
    ST_TET, 4, P6, P1, N0, P2,
    ST_PYR, 5, P7, P6, EF, EH, N0,
    ST_PYR, 5, ED, P3, P7, EH, N0,
    ST_TET, 4, P1, P2, P3, N0,
    ST_PYR, 5, P1, P3, ED, EA, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_PYR, 5, EF, P6, P1, EJ, N0,
    /* case 50 */ 9,
    ST_PNT, 5, EH, EF, EI, EA, EB,
    ST_PYR, 5, P7, P3, P2, P6, N0,
    ST_TET, 4, P2, P0, N0, P3,
    ST_PYR, 5, P6, P2, EB, EF, N0,
    ST_PYR, 5, EH, P7, P6, EF, N0,
    ST_TET, 4, P0, P3, P7, N0,
    ST_PYR, 5, P0, P7, EH, EI, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_PYR, 5, EB, P2, P0, EA, N0,
    /* case 51 */ 1,
    ST_HEX, 8, P3, P2, P6, P7, ED, EB, EF, EH,
    /* case 52 */ 11,
    ST_PNT, 4, EH, EI, EC, EC,
    ST_PYR, 5, P3, P1, EB, EC, N0,
    ST_TET, 4, EB, P1, EJ, N0,
    ST_PYR, 5, P0, EI, EJ, P1, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_TET, 4, P6, EL, EF, N0,
    ST_TET, 4, P7, P3, P6, N0,
    ST_PYR, 5, P6, P3, EC, EL, N0,
    ST_TET, 4, P7, P0, P3, N0,
    ST_PYR, 5, P7, EH, EI, P0, N0,
    ST_PYR, 5, P7, P6, EF, EH, N0,
    /* case 53 */ 8,
    ST_PNT, 5, EH, EF, ED, EC, EL,
    ST_TET, 4, P1, EA, EJ, EB,
    ST_PYR, 5, P6, EF, EH, P7, N0,
    ST_PYR, 5, P7, EH, ED, P3, N0,
    ST_TET, 4, P3, ED, EC, N0,
    ST_TET, 4, P7, P3, P6, N0,
    ST_PYR, 5, P3, EC, EL, P6, N0,
    ST_TET, 4, P6, EL, EF, N0,
    /* case 54 */ 9,
    ST_PNT, 6, EA, EI, EH, EF, EL, EC,
    ST_TET, 4, P7, P3, P6, N0,
    ST_PYR, 5, P7, P6, EF, EH, N0,
    ST_TET, 4, P0, P3, P7, N0,
    ST_PYR, 5, P0, P7, EH, EI, N0,
    ST_TET, 4, EA, P0, EI, N0,
    ST_PYR, 5, P0, EA, EC, P3, N0,
    ST_PYR, 5, EC, EL, P6, P3, N0,
    ST_TET, 4, P6, EL, EF, N0,
    /* case 55 */ 7,
    ST_PNT, 5, EF, EH, ED, EL, EC,
    ST_PYR, 5, P7, EH, ED, P3, N0,
    ST_TET, 4, P3, ED, EC, N0,
    ST_PYR, 5, P6, EF, EH, P7, N0,
    ST_TET, 4, EL, EF, P6, N0,
    ST_PYR, 5, EC, EL, P6, P3, N0,
    ST_TET, 4, P3, P6, P7, N0,
    /* case 56 */ 11,
    ST_PNT, 4, EF, EJ, EC, EC,
    ST_PYR, 5, P2, EC, ED, P0, N0,
    ST_TET, 4, ED, EI, P0, N0,
    ST_PYR, 5, P1, P0, EI, EJ, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_TET, 4, P7, EH, EK, N0,
    ST_TET, 4, P6, P7, P2, N0,
    ST_PYR, 5, P7, EK, EC, P2, N0,
    ST_TET, 4, P6, P2, P1, N0,
    ST_PYR, 5, P6, P1, EJ, EF, N0,
    ST_PYR, 5, P6, EF, EH, P7, N0,
    /* case 57 */ 9,
    ST_PNT, 6, EA, EJ, EF, EH, EK, EC,
    ST_TET, 4, P6, P7, P2, N0,
    ST_PYR, 5, P6, EF, EH, P7, N0,
    ST_TET, 4, P1, P6, P2, N0,
    ST_PYR, 5, P1, EJ, EF, P6, N0,
    ST_TET, 4, EA, EJ, P1, N0,
    ST_PYR, 5, P1, P2, EC, EA, N0,
    ST_PYR, 5, EC, P2, P7, EK, N0,
    ST_TET, 4, P7, EH, EK, N0,
    /* case 58 */ 8,
    ST_PNT, 5, EF, EB, EH, EK, EC,
    ST_TET, 4, P0, EI, EA, ED,
    ST_PYR, 5, P2, EB, EF, P6, N0,
    ST_PYR, 5, P6, EF, EH, P7, N0,
    ST_TET, 4, P7, EH, EK, N0,
    ST_TET, 4, P6, P7, P2, N0,
    ST_PYR, 5, P7, EK, EC, P2, N0,
    ST_TET, 4, P2, EC, EB, N0,
    /* case 59 */ 7,
    ST_PNT, 5, EB, EF, EH, EC, EK,
    ST_PYR, 5, P6, EF, EH, P7, N0,
    ST_TET, 4, P7, EH, EK, N0,
    ST_PYR, 5, P2, EB, EF, P6, N0,
    ST_TET, 4, EC, EB, P2, N0,
    ST_PYR, 5, EK, EC, P2, P7, N0,
    ST_TET, 4, P7, P2, P6, N0,
    /* case 60 */ 2,
    ST_WDG, 6, P1, EB, EJ, P0, ED, EI,
    ST_WDG, 6, P6, EF, EL, P7, EH, EK,
    /* case 61 */ 2,
    ST_WDG, 6, EH, P7, EK, EF, P6, EL,
    ST_TET, 4, EA, P1, EB, EJ,
    /* case 62 */ 2,
    ST_WDG, 6, EH, P7, EK, EF, P6, EL,
    ST_TET, 4, EA, ED, P0, EI,
    /* case 63 */ 1,
    ST_WDG, 6, P7, EK, EH, P6, EL, EF,
    /* case 64 */ 9,
    ST_PNT, 7, P7, P4, P5, P2, P3, P0, P1,
    ST_WDG, 6, EG, EF, EL, P7, P5, P2,
    ST_TET, 4, P7, P2, P5, N0,
    ST_TET, 4, P7, P5, P4, N0,
    ST_PYR, 5, P0, P4, P5, P1, N0,
    ST_PYR, 5, P3, P7, P4, P0, N0,
    ST_PYR, 5, P2, P3, P0, P1, N0,
    ST_TET, 4, P5, P2, P1, N0,
    ST_TET, 4, P2, P7, P3, N0,
    /* case 65 */ 6,
    ST_WDG, 6, P5, P7, P2, EF, EG, EL,
    ST_WDG, 6, EI, ED, EA, P4, P3, P1,
    ST_TET, 4, P7, P2, P5, P3,
    ST_TET, 4, P1, P2, P3, P5,
    ST_TET, 4, P4, P5, P1, P3,
    ST_TET, 4, P7, P4, P3, P5,
    /* case 66 */ 12,
    ST_PNT, 2, EA, EG,
    ST_PYR, 5, P0, P3, P7, P4, N0,
    ST_TET, 4, P4, P7, P5, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_TET, 4, P2, P3, P0, N0,
    ST_TET, 4, P7, P3, P2, N0,
    ST_PYR, 5, P7, P2, EL, EG, N0,
    ST_PYR, 5, P5, P7, EG, EF, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_PYR, 5, P0, EA, EB, P2, N0,
    ST_TET, 4, P2, EB, EL, N0,
    /* case 67 */ 11,
    ST_PNT, 4, EI, ED, EG, EG,
    ST_PYR, 5, P7, P2, EL, EG, N0,
    ST_TET, 4, EL, P2, EB, N0,
    ST_PYR, 5, P3, ED, EB, P2, N0,
    ST_TET, 4, P7, P3, P2, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    ST_TET, 4, P4, P7, P5, N0,
    ST_PYR, 5, P5, P7, EG, EF, N0,
    ST_TET, 4, P4, P3, P7, N0,
    ST_PYR, 5, P4, EI, ED, P3, N0,
    ST_PYR, 5, P4, P5, EJ, EI, N0,
    /* case 68 */ 2,
    ST_HEX, 8, EG, P7, P3, EC, EF, P5, P1, EB,
    ST_WDG, 6, P1, P0, P3, P5, P4, P7,
    /* case 69 */ 11,
    ST_PNT, 4, EG, EF, EI, EI,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_TET, 4, EA, P1, EB, N0,
    ST_PYR, 5, P5, EF, EB, P1, N0,
    ST_TET, 4, P4, P5, P1, N0,
    ST_TET, 4, P3, ED, EC, N0,
    ST_TET, 4, P7, P4, P3, N0,
    ST_PYR, 5, P3, P4, EI, ED, N0,
    ST_TET, 4, P7, P5, P4, N0,
    ST_PYR, 5, P7, EG, EF, P5, N0,
    ST_PYR, 5, P7, P3, EC, EG, N0,
    /* case 70 */ 9,
    ST_PNT, 5, EA, EC, EJ, EF, EG,
    ST_PYR, 5, P0, P3, P7, P4, N0,
    ST_TET, 4, P7, N0, P5, P4,
    ST_PYR, 5, P3, EC, EG, P7, N0,
    ST_PYR, 5, EA, EC, P3, P0, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_PYR, 5, P5, EJ, EA, P0, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    ST_PYR, 5, EG, EF, P5, P7, N0,
    /* case 71 */ 9,
    ST_PNT, 6, EJ, EF, EG, EC, ED, EI,
    ST_TET, 4, P7, P4, P3, N0,
    ST_PYR, 5, P7, P3, EC, EG, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_PYR, 5, P5, P7, EG, EF, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    ST_PYR, 5, P5, EJ, EI, P4, N0,
    ST_PYR, 5, EI, ED, P3, P4, N0,
    ST_TET, 4, P3, ED, EC, N0,
    /* case 72 */ 12,
    ST_PNT, 2, ED, EF,
    ST_PYR, 5, P0, P4, P5, P1, N0,
    ST_TET, 4, P1, P5, P2, N0,
    ST_TET, 4, P0, P1, P2, N0,
    ST_TET, 4, P7, P4, P0, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_PYR, 5, P5, P7, EG, EF, N0,
    ST_PYR, 5, P2, P5, EF, EL, N0,
    ST_TET, 4, P2, EL, EC, N0,
    ST_PYR, 5, P0, P2, EC, ED, N0,
    ST_PYR, 5, P0, ED, EK, P7, N0,
    ST_TET, 4, P7, EK, EG, N0,
    /* case 73 */ 11,
    ST_PNT, 4, EA, EI, EF, EF,
    ST_PYR, 5, P5, P7, EG, EF, N0,
    ST_TET, 4, EG, P7, EK, N0,
    ST_PYR, 5, P4, EI, EK, P7, N0,
    ST_TET, 4, P5, P4, P7, N0,
    ST_TET, 4, P2, EL, EC, N0,
    ST_TET, 4, P1, P5, P2, N0,
    ST_PYR, 5, P2, P5, EF, EL, N0,
    ST_TET, 4, P1, P4, P5, N0,
    ST_PYR, 5, P1, EA, EI, P4, N0,
    ST_PYR, 5, P1, P2, EC, EA, N0,
    /* case 74 */ 6,
    ST_TET, 4, P2, EB, EL, EC,
    ST_TET, 4, P7, P4, P0, P5,
    ST_PYR, 5, EJ, EA, P0, P5, ED,
    ST_PYR, 5, EF, P5, P7, EG, EK,
    ST_PYR, 5, P0, ED, EK, P7, P5,
    ST_PYR, 5, ED, EJ, EF, EK, P5,
    /* case 75 */ 8,
    ST_PNT, 5, EI, EK, EJ, EF, EG,
    ST_TET, 4, P2, EC, EB, EL,
    ST_PYR, 5, P7, P4, EI, EK, N0,
    ST_PYR, 5, P4, P5, EJ, EI, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    ST_TET, 4, P4, P7, P5, N0,
    ST_PYR, 5, P5, P7, EG, EF, N0,
    ST_TET, 4, P7, EK, EG, N0,
    /* case 76 */ 9,
    ST_PNT, 5, ED, EB, EK, EG, EF,
    ST_PYR, 5, P0, P4, P5, P1, N0,
    ST_TET, 4, P5, P7, N0, P4,
    ST_PYR, 5, P1, P5, EF, EB, N0,
    ST_PYR, 5, ED, P0, P1, EB, N0,
    ST_TET, 4, P7, P4, P0, N0,
    ST_PYR, 5, P7, P0, ED, EK, N0,
    ST_TET, 4, P7, EK, EG, N0,
    ST_PYR, 5, EF, P5, P7, EG, N0,
    /* case 77 */ 9,
    ST_PNT, 6, EK, EG, EF, EB, EA, EI,
    ST_TET, 4, P5, P1, P4, N0,
    ST_PYR, 5, P5, EF, EB, P1, N0,
    ST_TET, 4, P7, P5, P4, N0,
    ST_PYR, 5, P7, EG, EF, P5, N0,
    ST_TET, 4, EK, EG, P7, N0,
    ST_PYR, 5, P7, P4, EI, EK, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    ST_TET, 4, P1, EB, EA, N0,
    /* case 78 */ 5,
    ST_TET, 4, P7, P4, P0, P5,
    ST_PYR, 5, EJ, EA, P0, P5, ED,
    ST_PYR, 5, EF, P5, P7, EG, EK,
    ST_PYR, 5, P0, ED, EK, P7, P5,
    ST_PYR, 5, ED, EJ, EF, EK, P5,
    /* case 79 */ 7,
    ST_PNT, 5, EK, EI, EJ, EG, EF,
    ST_PYR, 5, P4, P5, EJ, EI, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    ST_PYR, 5, P7, P4, EI, EK, N0,
    ST_TET, 4, EG, P7, EK, N0,
    ST_PYR, 5, EF, P5, P7, EG, N0,
    ST_TET, 4, P5, P4, P7, N0,
    /* case 80 */ 12,
    ST_PNT, 2, EI, EL,
    ST_PYR, 5, P0, P1, P2, P3, N0,
    ST_TET, 4, P1, P5, P2, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_TET, 4, P7, P0, P3, N0,
    ST_TET, 4, P2, P7, P3, N0,
    ST_PYR, 5, P2, EL, EG, P7, N0,
    ST_PYR, 5, P5, EF, EL, P2, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_PYR, 5, P0, P7, EH, EI, N0,
    ST_TET, 4, P7, EG, EH, N0,
    /* case 81 */ 11,
    ST_PNT, 4, EA, ED, EL, EL,
    ST_PYR, 5, P2, EL, EG, P7, N0,
    ST_TET, 4, EG, EH, P7, N0,
    ST_PYR, 5, P3, P7, EH, ED, N0,
    ST_TET, 4, P2, P7, P3, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_TET, 4, P1, P5, P2, N0,
    ST_PYR, 5, P5, EF, EL, P2, N0,
    ST_TET, 4, P1, P2, P3, N0,
    ST_PYR, 5, P1, P3, ED, EA, N0,
    ST_PYR, 5, P1, EA, EE, P5, N0,
    /* case 82 */ 6,
    ST_TET, 4, P5, EF, EJ, EE,
    ST_TET, 4, P7, P0, P3, P2,
    ST_PYR, 5, EB, P2, P0, EA, EI,
    ST_PYR, 5, EL, EG, P7, P2, EH,
    ST_PYR, 5, P0, P7, EH, EI, P2,
    ST_PYR, 5, EI, EH, EL, EB, P2,
    /* case 83 */ 8,
    ST_PNT, 5, ED, EH, EB, EL, EG,
    ST_TET, 4, P5, EJ, EE, EF,
    ST_PYR, 5, P7, EH, ED, P3, N0,
    ST_PYR, 5, P3, ED, EB, P2, N0,
    ST_TET, 4, P2, EB, EL, N0,
    ST_TET, 4, P3, P2, P7, N0,
    ST_PYR, 5, P2, EL, EG, P7, N0,
    ST_TET, 4, P7, EG, EH, N0,
    /* case 84 */ 11,
    ST_PNT, 4, EC, EB, EI, EI,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_TET, 4, EE, EF, P5, N0,
    ST_PYR, 5, P1, P5, EF, EB, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_TET, 4, P7, EG, EH, N0,
    ST_TET, 4, P3, P7, P0, N0,
    ST_PYR, 5, P7, EH, EI, P0, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_PYR, 5, P3, P1, EB, EC, N0,
    ST_PYR, 5, P3, EC, EG, P7, N0,
    /* case 85 */ 2,
    ST_WDG, 6, P7, EH, EG, P3, ED, EC,
    ST_WDG, 6, P5, EF, EE, P1, EB, EA,
    /* case 86 */ 8,
    ST_PNT, 5, EC, EG, EA, EI, EH,
    ST_TET, 4, P5, EF, EJ, EE,
    ST_PYR, 5, P7, P3, EC, EG, N0,
    ST_PYR, 5, P3, P0, EA, EC, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_TET, 4, P3, P7, P0, N0,
    ST_PYR, 5, P0, P7, EH, EI, N0,
    ST_TET, 4, P7, EG, EH, N0,
    /* case 87 */ 2,
    ST_WDG, 6, EG, P7, EH, EC, P3, ED,
    ST_TET, 4, EJ, EE, P5, EF,
    /* case 88 */ 6,
    ST_TET, 4, P7, EG, EH, EK,
    ST_TET, 4, P2, P0, P1, P5,
    ST_PYR, 5, EE, P5, P0, EI, ED,
    ST_PYR, 5, EF, EL, P2, P5, EC,
    ST_PYR, 5, P0, P2, EC, ED, P5,
    ST_PYR, 5, ED, EC, EF, EE, P5,
    /* case 89 */ 8,
    ST_PNT, 5, EA, EC, EE, EF, EL,
    ST_TET, 4, P7, EH, EK, EG,
    ST_PYR, 5, P2, EC, EA, P1, N0,
    ST_PYR, 5, P1, EA, EE, P5, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_TET, 4, P1, P5, P2, N0,
    ST_PYR, 5, P5, EF, EL, P2, N0,
    ST_TET, 4, P2, EL, EC, N0,
    /* case 90 */ 4,
    ST_TET, 4, EH, EG, EK, P7,
    ST_TET, 4, EI, ED, EA, P0,
    ST_TET, 4, EE, EJ, EF, P5,
    ST_TET, 4, EB, EC, EL, P2,
    /* case 91 */ 3,
    ST_TET, 4, EE, EJ, EF, P5,
    ST_TET, 4, EH, EK, P7, EG,
    ST_TET, 4, EB, P2, EC, EL,
    /* case 92 */ 8,
    ST_PNT, 5, EB, EF, ED, EI, EE,
    ST_TET, 4, P7, EK, EG, EH,
    ST_PYR, 5, P5, EF, EB, P1, N0,
    ST_PYR, 5, P1, EB, ED, P0, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_TET, 4, P1, P0, P5, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_TET, 4, P5, EE, EF, N0,
    /* case 93 */ 2,
    ST_WDG, 6, EB, P1, EA, EF, P5, EE,
    ST_TET, 4, EK, P7, EH, EG,
    /* case 94 */ 3,
    ST_TET, 4, EK, EH, EG, P7,
    ST_TET, 4, ED, P0, EA, EI,
    ST_TET, 4, EF, EJ, P5, EE,
    /* case 95 */ 2,
    ST_TET, 4, EG, EK, EH, P7,
    ST_TET, 4, EF, EE, EJ, P5,
    /* case 96 */ 2,
    ST_HEX, 8, EL, P2, P1, EJ, EG, P7, P4, EE,
    ST_WDG, 6, P2, P3, P7, P1, P0, P4,
    /* case 97 */ 11,
    ST_PNT, 4, EG, EL, ED, ED,
    ST_PYR, 5, P3, ED, EA, P1, N0,
    ST_TET, 4, EA, EJ, P1, N0,
    ST_PYR, 5, P2, P1, EJ, EL, N0,
    ST_TET, 4, P3, P1, P2, N0,
    ST_TET, 4, P4, EE, EI, N0,
    ST_TET, 4, P7, P4, P3, N0,
    ST_PYR, 5, P4, EI, ED, P3, N0,
    ST_TET, 4, P7, P3, P2, N0,
    ST_PYR, 5, P7, P2, EL, EG, N0,
    ST_PYR, 5, P7, EG, EE, P4, N0,
    /* case 98 */ 9,
    ST_PNT, 5, EA, EE, EB, EL, EG,
    ST_PYR, 5, P0, P3, P7, P4, N0,
    ST_TET, 4, P7, P2, N0, P3,
    ST_PYR, 5, P4, P7, EG, EE, N0,
    ST_PYR, 5, EA, P0, P4, EE, N0,
    ST_TET, 4, P2, P3, P0, N0,
    ST_PYR, 5, P2, P0, EA, EB, N0,
    ST_TET, 4, P2, EB, EL, N0,
    ST_PYR, 5, EG, P7, P2, EL, N0,
    /* case 99 */ 9,
    ST_PNT, 6, EB, EL, EG, EE, EI, ED,
    ST_TET, 4, P7, P4, P3, N0,
    ST_PYR, 5, P7, EG, EE, P4, N0,
    ST_TET, 4, P2, P7, P3, N0,
    ST_PYR, 5, P2, EL, EG, P7, N0,
    ST_TET, 4, EB, EL, P2, N0,
    ST_PYR, 5, P2, P3, ED, EB, N0,
    ST_PYR, 5, ED, P3, P4, EI, N0,
    ST_TET, 4, P4, EE, EI, N0,
    /* case 100 */ 9,
    ST_PNT, 5, EC, EG, EB, EJ, EE,
    ST_PYR, 5, P3, P7, P4, P0, N0,
    ST_TET, 4, P4, N0, P1, P0,
    ST_PYR, 5, P7, EG, EE, P4, N0,
    ST_PYR, 5, EC, EG, P7, P3, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_PYR, 5, P1, EB, EC, P3, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_PYR, 5, EE, EJ, P1, P4, N0,
    /* case 101 */ 8,
    ST_PNT, 5, EG, EE, EC, ED, EI,
    ST_TET, 4, P1, EJ, EB, EA,
    ST_PYR, 5, P4, P7, EG, EE, N0,
    ST_PYR, 5, P7, P3, EC, EG, N0,
    ST_TET, 4, P3, ED, EC, N0,
    ST_TET, 4, P7, P4, P3, N0,
    ST_PYR, 5, P3, P4, EI, ED, N0,
    ST_TET, 4, P4, EE, EI, N0,
    /* case 102 */ 1,
    ST_HEX, 8, EA, EE, EG, EC, P0, P4, P7, P3,
    /* case 103 */ 7,
    ST_PNT, 5, EE, EG, EC, EI, ED,
    ST_PYR, 5, P7, P3, EC, EG, N0,
    ST_TET, 4, P3, ED, EC, N0,
    ST_PYR, 5, P4, P7, EG, EE, N0,
    ST_TET, 4, EI, P4, EE, N0,
    ST_PYR, 5, ED, P3, P4, EI, N0,
    ST_TET, 4, P3, P7, P4, N0,
    /* case 104 */ 11,
    ST_PNT, 4, EE, EJ, ED, ED,
    ST_PYR, 5, P0, P2, EC, ED, N0,
    ST_TET, 4, EC, P2, EL, N0,
    ST_PYR, 5, P1, EJ, EL, P2, N0,
    ST_TET, 4, P0, P1, P2, N0,
    ST_TET, 4, P7, EK, EG, N0,
    ST_TET, 4, P4, P0, P7, N0,
    ST_PYR, 5, P7, P0, ED, EK, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_PYR, 5, P4, P7, EG, EE, N0,
    /* case 105 */ 2,
    ST_WDG, 6, P4, EI, EE, P7, EK, EG,
    ST_WDG, 6, P1, EJ, EA, P2, EL, EC,
    /* case 106 */ 8,
    ST_PNT, 5, EE, EG, EA, ED, EK,
    ST_TET, 4, P2, EB, EL, EC,
    ST_PYR, 5, P7, EG, EE, P4, N0,
    ST_PYR, 5, P4, EE, EA, P0, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_TET, 4, P4, P0, P7, N0,
    ST_PYR, 5, P0, ED, EK, P7, N0,
    ST_TET, 4, P7, EK, EG, N0,
    /* case 107 */ 2,
    ST_WDG, 6, EE, P4, EI, EG, P7, EK,
    ST_TET, 4, EB, P2, EC, EL,
    /* case 108 */ 9,
    ST_PNT, 6, EB, EJ, EE, EG, EK, ED,
    ST_TET, 4, P4, P0, P7, N0,
    ST_PYR, 5, P4, P7, EG, EE, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_TET, 4, EB, P1, EJ, N0,
    ST_PYR, 5, P1, EB, ED, P0, N0,
    ST_PYR, 5, ED, EK, P7, P0, N0,
    ST_TET, 4, P7, EK, EG, N0,
    /* case 109 */ 2,
    ST_WDG, 6, EE, P4, EI, EG, P7, EK,
    ST_TET, 4, EB, EA, P1, EJ,
    /* case 110 */ 7,
    ST_PNT, 5, EG, EE, EA, EK, ED,
    ST_PYR, 5, P4, EE, EA, P0, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_PYR, 5, P7, EG, EE, P4, N0,
    ST_TET, 4, EK, EG, P7, N0,
    ST_PYR, 5, ED, EK, P7, P0, N0,
    ST_TET, 4, P0, P7, P4, N0,
    /* case 111 */ 1,
    ST_WDG, 6, P4, EI, EE, P7, EK, EG,
    /* case 112 */ 9,
    ST_PNT, 5, EI, EJ, EH, EG, EL,
    ST_PYR, 5, P0, P1, P2, P3, N0,
    ST_TET, 4, P2, N0, P7, P3,
    ST_PYR, 5, P1, EJ, EL, P2, N0,
    ST_PYR, 5, EI, EJ, P1, P0, N0,
    ST_TET, 4, P7, P0, P3, N0,
    ST_PYR, 5, P7, EH, EI, P0, N0,
    ST_TET, 4, P7, EG, EH, N0,
    ST_PYR, 5, EL, EG, P7, P2, N0,
    /* case 113 */ 9,
    ST_PNT, 6, EH, EG, EL, EJ, EA, ED,
    ST_TET, 4, P2, P3, P1, N0,
    ST_PYR, 5, P2, P1, EJ, EL, N0,
    ST_TET, 4, P7, P3, P2, N0,
    ST_PYR, 5, P7, P2, EL, EG, N0,
    ST_TET, 4, EH, P7, EG, N0,
    ST_PYR, 5, P7, EH, ED, P3, N0,
    ST_PYR, 5, ED, EA, P1, P3, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    /* case 114 */ 5,
    ST_TET, 4, P7, P0, P3, P2,
    ST_PYR, 5, EB, P2, P0, EA, EI,
    ST_PYR, 5, EL, EG, P7, P2, EH,
    ST_PYR, 5, P0, P7, EH, EI, P2,
    ST_PYR, 5, EI, EH, EL, EB, P2,
    /* case 115 */ 7,
    ST_PNT, 5, EH, ED, EB, EG, EL,
    ST_PYR, 5, P3, ED, EB, P2, N0,
    ST_TET, 4, P2, EB, EL, N0,
    ST_PYR, 5, P7, EH, ED, P3, N0,
    ST_TET, 4, EG, EH, P7, N0,
    ST_PYR, 5, EL, EG, P7, P2, N0,
    ST_TET, 4, P2, P7, P3, N0,
    /* case 116 */ 9,
    ST_PNT, 6, EJ, EB, EC, EG, EH, EI,
    ST_TET, 4, P3, P7, P0, N0,
    ST_PYR, 5, P3, EC, EG, P7, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_PYR, 5, P1, EB, EC, P3, N0,
    ST_TET, 4, EJ, EB, P1, N0,
    ST_PYR, 5, P1, P0, EI, EJ, N0,
    ST_PYR, 5, EI, P0, P7, EH, N0,
    ST_TET, 4, P7, EG, EH, N0,
    /* case 117 */ 2,
    ST_WDG, 6, EG, P7, EH, EC, P3, ED,
    ST_TET, 4, EJ, P1, EA, EB,
    /* case 118 */ 7,
    ST_PNT, 5, EG, EC, EA, EH, EI,
    ST_PYR, 5, P3, P0, EA, EC, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_PYR, 5, P7, P3, EC, EG, N0,
    ST_TET, 4, EH, P7, EG, N0,
    ST_PYR, 5, EI, P0, P7, EH, N0,
    ST_TET, 4, P0, P3, P7, N0,
    /* case 119 */ 1,
    ST_WDG, 6, P7, EH, EG, P3, ED, EC,
    /* case 120 */ 8,
    ST_PNT, 5, EJ, EL, EI, ED, EC,
    ST_TET, 4, P7, EG, EH, EK,
    ST_PYR, 5, P2, P1, EJ, EL, N0,
    ST_PYR, 5, P1, P0, EI, EJ, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_TET, 4, P1, P2, P0, N0,
    ST_PYR, 5, P0, P2, EC, ED, N0,
    ST_TET, 4, P2, EL, EC, N0,
    /* case 121 */ 2,
    ST_WDG, 6, EL, P2, EC, EJ, P1, EA,
    ST_TET, 4, EH, EK, P7, EG,
    /* case 122 */ 3,
    ST_TET, 4, EH, EG, EK, P7,
    ST_TET, 4, EI, EA, P0, ED,
    ST_TET, 4, EL, P2, EB, EC,
    /* case 123 */ 2,
    ST_TET, 4, EG, EK, EH, P7,
    ST_TET, 4, EL, EB, EC, P2,
    /* case 124 */ 2,
    ST_WDG, 6, ED, P0, EI, EB, P1, EJ,
    ST_TET, 4, EG, EH, P7, EK,
    /* case 125 */ 2,
    ST_TET, 4, EG, EK, EH, P7,
    ST_TET, 4, EJ, EA, EB, P1,
    /* case 126 */ 2,
    ST_TET, 4, EI, ED, EA, P0,
    ST_TET, 4, EH, EG, EK, P7,
    /* case 127 */ 1,
    ST_TET, 4, EH, EG, EK, P7,
    /* case 128 */ 9,
    ST_PNT, 7, P6, P5, P4, P3, P2, P1, P0,
    ST_WDG, 6, P6, P4, P3, EG, EH, EK,
    ST_TET, 4, P6, P4, P3, N0,
    ST_TET, 4, P6, P5, P4, N0,
    ST_PYR, 5, P1, P0, P4, P5, N0,
    ST_PYR, 5, P2, P1, P5, P6, N0,
    ST_PYR, 5, P3, P0, P1, P2, N0,
    ST_TET, 4, P4, P0, P3, N0,
    ST_TET, 4, P3, P2, P6, N0,
    /* case 129 */ 12,
    ST_PNT, 2, EA, EG,
    ST_PYR, 5, P1, P5, P6, P2, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_TET, 4, P1, P4, P5, N0,
    ST_TET, 4, P3, P1, P2, N0,
    ST_TET, 4, P6, P3, P2, N0,
    ST_PYR, 5, P6, EG, EK, P3, N0,
    ST_PYR, 5, P4, EH, EG, P6, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_PYR, 5, P1, EA, EI, P4, N0,
    ST_PYR, 5, P1, P3, ED, EA, N0,
    ST_TET, 4, P3, EK, ED, N0,
    /* case 130 */ 6,
    ST_WDG, 6, P4, P3, P6, EH, EK, EG,
    ST_WDG, 6, EA, EB, EJ, P0, P2, P5,
    ST_TET, 4, P3, P6, P4, P2,
    ST_TET, 4, P5, P6, P2, P4,
    ST_TET, 4, P0, P4, P5, P2,
    ST_TET, 4, P3, P0, P2, P4,
    /* case 131 */ 11,
    ST_PNT, 4, EJ, EB, EG, EG,
    ST_PYR, 5, P6, EG, EK, P3, N0,
    ST_TET, 4, EK, ED, P3, N0,
    ST_PYR, 5, P2, P3, ED, EB, N0,
    ST_TET, 4, P6, P3, P2, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_PYR, 5, P4, EH, EG, P6, N0,
    ST_TET, 4, P5, P6, P2, N0,
    ST_PYR, 5, P5, P2, EB, EJ, N0,
    ST_PYR, 5, P5, EJ, EI, P4, N0,
    /* case 132 */ 12,
    ST_PNT, 2, EH, EB,
    ST_PYR, 5, P4, P5, P1, P0, N0,
    ST_TET, 4, P5, P6, P1, N0,
    ST_TET, 4, P4, P6, P5, N0,
    ST_TET, 4, P3, P4, P0, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_PYR, 5, P1, EB, EC, P3, N0,
    ST_PYR, 5, P6, EL, EB, P1, N0,
    ST_TET, 4, P6, EG, EL, N0,
    ST_PYR, 5, P4, EH, EG, P6, N0,
    ST_PYR, 5, P4, P3, EK, EH, N0,
    ST_TET, 4, P3, EC, EK, N0,
    /* case 133 */ 6,
    ST_TET, 4, P3, ED, EC, EK,
    ST_TET, 4, P6, P5, P4, P1,
    ST_PYR, 5, EA, EI, P4, P1, EH,
    ST_PYR, 5, EB, P1, P6, EL, EG,
    ST_PYR, 5, P4, EH, EG, P6, P1,
    ST_PYR, 5, EH, EA, EB, EG, P1,
    /* case 134 */ 11,
    ST_PNT, 4, EJ, EA, EH, EH,
    ST_PYR, 5, P4, P3, EK, EH, N0,
    ST_TET, 4, EK, P3, EC, N0,
    ST_PYR, 5, P0, EA, EC, P3, N0,
    ST_TET, 4, P4, P0, P3, N0,
    ST_TET, 4, P6, EG, EL, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_PYR, 5, P6, P4, EH, EG, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_PYR, 5, P5, EJ, EA, P0, N0,
    ST_PYR, 5, P5, P6, EL, EJ, N0,
    /* case 135 */ 8,
    ST_PNT, 5, EJ, EL, EI, EH, EG,
    ST_TET, 4, P3, ED, EC, EK,
    ST_PYR, 5, P6, EL, EJ, P5, N0,
    ST_PYR, 5, P5, EJ, EI, P4, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_TET, 4, P5, P4, P6, N0,
    ST_PYR, 5, P4, EH, EG, P6, N0,
    ST_TET, 4, P6, EG, EL, N0,
    /* case 136 */ 2,
    ST_HEX, 8, EH, P4, P0, ED, EG, P6, P2, EC,
    ST_WDG, 6, P4, P5, P6, P0, P1, P2,
    /* case 137 */ 9,
    ST_PNT, 5, EA, EC, EI, EH, EG,
    ST_PYR, 5, P1, P5, P6, P2, N0,
    ST_TET, 4, P6, P4, N0, P5,
    ST_PYR, 5, P2, P6, EG, EC, N0,
    ST_PYR, 5, EA, P1, P2, EC, N0,
    ST_TET, 4, P4, P5, P1, N0,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_PYR, 5, EG, P6, P4, EH, N0,
    /* case 138 */ 11,
    ST_PNT, 4, EG, EH, EJ, EJ,
    ST_PYR, 5, P5, EJ, EA, P0, N0,
    ST_TET, 4, EA, ED, P0, N0,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_TET, 4, P5, P0, P4, N0,
    ST_TET, 4, P2, EC, EB, N0,
    ST_TET, 4, P6, P2, P5, N0,
    ST_PYR, 5, P2, EB, EJ, P5, N0,
    ST_TET, 4, P6, P5, P4, N0,
    ST_PYR, 5, P6, P4, EH, EG, N0,
    ST_PYR, 5, P6, EG, EC, P2, N0,
    /* case 139 */ 9,
    ST_PNT, 6, EI, EH, EG, EC, EB, EJ,
    ST_TET, 4, P6, P2, P5, N0,
    ST_PYR, 5, P6, EG, EC, P2, N0,
    ST_TET, 4, P4, P6, P5, N0,
    ST_PYR, 5, P4, EH, EG, P6, N0,
    ST_TET, 4, EI, EH, P4, N0,
    ST_PYR, 5, P4, P5, EJ, EI, N0,
    ST_PYR, 5, EJ, P5, P2, EB, N0,
    ST_TET, 4, P2, EC, EB, N0,
    /* case 140 */ 9,
    ST_PNT, 5, EB, ED, EL, EG, EH,
    ST_PYR, 5, P1, P0, P4, P5, N0,
    ST_TET, 4, P4, N0, P6, P5,
    ST_PYR, 5, P0, ED, EH, P4, N0,
    ST_PYR, 5, EB, ED, P0, P1, N0,
    ST_TET, 4, P6, P1, P5, N0,
    ST_PYR, 5, P6, EL, EB, P1, N0,
    ST_TET, 4, P6, EG, EL, N0,
    ST_PYR, 5, EH, EG, P6, P4, N0,
    /* case 141 */ 5,
    ST_TET, 4, P6, P5, P4, P1,
    ST_PYR, 5, EA, EI, P4, P1, EH,
    ST_PYR, 5, EB, P1, P6, EL, EG,
    ST_PYR, 5, P4, EH, EG, P6, P1,
    ST_PYR, 5, EH, EA, EB, EG, P1,
    /* case 142 */ 9,
    ST_PNT, 6, ED, EA, EJ, EL, EG, EH,
    ST_TET, 4, P5, P4, P6, N0,
    ST_PYR, 5, P5, P6, EL, EJ, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_TET, 4, ED, P0, EA, N0,
    ST_PYR, 5, P0, ED, EH, P4, N0,
    ST_PYR, 5, EH, EG, P6, P4, N0,
    ST_TET, 4, P6, EG, EL, N0,
    /* case 143 */ 7,
    ST_PNT, 5, EL, EJ, EI, EG, EH,
    ST_PYR, 5, P5, EJ, EI, P4, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_PYR, 5, P6, EL, EJ, P5, N0,
    ST_TET, 4, EG, EL, P6, N0,
    ST_PYR, 5, EH, EG, P6, P4, N0,
    ST_TET, 4, P4, P6, P5, N0,
    /* case 144 */ 2,
    ST_HEX, 8, EG, P6, P5, EE, EK, P3, P0, EI,
    ST_WDG, 6, P0, P1, P5, P3, P2, P6,
    /* case 145 */ 9,
    ST_PNT, 5, EA, EE, ED, EK, EG,
    ST_PYR, 5, P1, P5, P6, P2, N0,
    ST_TET, 4, P6, N0, P3, P2,
    ST_PYR, 5, P5, EE, EG, P6, N0,
    ST_PYR, 5, EA, EE, P5, P1, N0,
    ST_TET, 4, P3, P1, P2, N0,
    ST_PYR, 5, P3, ED, EA, P1, N0,
    ST_TET, 4, P3, EK, ED, N0,
    ST_PYR, 5, EG, EK, P3, P6, N0,
    /* case 146 */ 11,
    ST_PNT, 4, EG, EK, EB, EB,
    ST_PYR, 5, P2, P0, EA, EB, N0,
    ST_TET, 4, EA, P0, EI, N0,
    ST_PYR, 5, P3, EK, EI, P0, N0,
    ST_TET, 4, P2, P3, P0, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    ST_TET, 4, P6, P2, P5, N0,
    ST_PYR, 5, P5, P2, EB, EJ, N0,
    ST_TET, 4, P6, P3, P2, N0,
    ST_PYR, 5, P6, EG, EK, P3, N0,
    ST_PYR, 5, P6, P5, EE, EG, N0,
    /* case 147 */ 9,
    ST_PNT, 6, ED, EK, EG, EE, EJ, EB,
    ST_TET, 4, P6, P2, P5, N0,
    ST_PYR, 5, P6, P5, EE, EG, N0,
    ST_TET, 4, P3, P2, P6, N0,
    ST_PYR, 5, P3, P6, EG, EK, N0,
    ST_TET, 4, ED, P3, EK, N0,
    ST_PYR, 5, P3, ED, EB, P2, N0,
    ST_PYR, 5, EB, EJ, P5, P2, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    /* case 148 */ 11,
    ST_PNT, 4, EE, EI, EB, EB,
    ST_PYR, 5, P1, EB, EC, P3, N0,
    ST_TET, 4, EC, EK, P3, N0,
    ST_PYR, 5, P0, P3, EK, EI, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_TET, 4, P6, EG, EL, N0,
    ST_TET, 4, P5, P6, P1, N0,
    ST_PYR, 5, P6, EL, EB, P1, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_PYR, 5, P5, P0, EI, EE, N0,
    ST_PYR, 5, P5, EE, EG, P6, N0,
    /* case 149 */ 8,
    ST_PNT, 5, EE, EG, EA, EB, EL,
    ST_TET, 4, P3, EK, ED, EC,
    ST_PYR, 5, P6, P5, EE, EG, N0,
    ST_PYR, 5, P5, P1, EA, EE, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_TET, 4, P5, P6, P1, N0,
    ST_PYR, 5, P1, P6, EL, EB, N0,
    ST_TET, 4, P6, EG, EL, N0,
    /* case 150 */ 2,
    ST_WDG, 6, P3, EK, EC, P0, EI, EA,
    ST_WDG, 6, P6, EL, EG, P5, EJ, EE,
    /* case 151 */ 2,
    ST_WDG, 6, EJ, P5, EE, EL, P6, EG,
    ST_TET, 4, ED, P3, EK, EC,
    /* case 152 */ 9,
    ST_PNT, 5, EC, EG, ED, EI, EE,
    ST_PYR, 5, P2, P1, P5, P6, N0,
    ST_TET, 4, P5, P0, N0, P1,
    ST_PYR, 5, P6, P5, EE, EG, N0,
    ST_PYR, 5, EC, P2, P6, EG, N0,
    ST_TET, 4, P0, P1, P2, N0,
    ST_PYR, 5, P0, P2, EC, ED, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_PYR, 5, EE, P5, P0, EI, N0,
    /* case 153 */ 1,
    ST_HEX, 8, P1, P5, P6, P2, EA, EE, EG, EC,
    /* case 154 */ 8,
    ST_PNT, 5, EG, EE, EC, EB, EJ,
    ST_TET, 4, P0, ED, EI, EA,
    ST_PYR, 5, P5, EE, EG, P6, N0,
    ST_PYR, 5, P6, EG, EC, P2, N0,
    ST_TET, 4, P2, EC, EB, N0,
    ST_TET, 4, P6, P2, P5, N0,
    ST_PYR, 5, P2, EB, EJ, P5, N0,
    ST_TET, 4, P5, EJ, EE, N0,
    /* case 155 */ 7,
    ST_PNT, 5, EE, EG, EC, EJ, EB,
    ST_PYR, 5, P6, EG, EC, P2, N0,
    ST_TET, 4, P2, EC, EB, N0,
    ST_PYR, 5, P5, EE, EG, P6, N0,
    ST_TET, 4, EJ, EE, P5, N0,
    ST_PYR, 5, EB, EJ, P5, P2, N0,
    ST_TET, 4, P2, P5, P6, N0,
    /* case 156 */ 9,
    ST_PNT, 6, ED, EI, EE, EG, EL, EB,
    ST_TET, 4, P5, P6, P1, N0,
    ST_PYR, 5, P5, EE, EG, P6, N0,
    ST_TET, 4, P0, P5, P1, N0,
    ST_PYR, 5, P0, EI, EE, P5, N0,
    ST_TET, 4, ED, EI, P0, N0,
    ST_PYR, 5, P0, P1, EB, ED, N0,
    ST_PYR, 5, EB, P1, P6, EL, N0,
    ST_TET, 4, P6, EG, EL, N0,
    /* case 157 */ 7,
    ST_PNT, 5, EG, EE, EA, EL, EB,
    ST_PYR, 5, P5, P1, EA, EE, N0,
    ST_TET, 4, P1, EB, EA, N0,
    ST_PYR, 5, P6, P5, EE, EG, N0,
    ST_TET, 4, EL, P6, EG, N0,
    ST_PYR, 5, EB, P1, P6, EL, N0,
    ST_TET, 4, P1, P5, P6, N0,
    /* case 158 */ 2,
    ST_WDG, 6, EJ, P5, EE, EL, P6, EG,
    ST_TET, 4, ED, EI, P0, EA,
    /* case 159 */ 1,
    ST_WDG, 6, P5, EE, EJ, P6, EG, EL,
    /* case 160 */ 12,
    ST_PNT, 2, EK, EJ,
    ST_PYR, 5, P3, P0, P1, P2, N0,
    ST_TET, 4, P2, P1, P6, N0,
    ST_TET, 4, P3, P2, P6, N0,
    ST_TET, 4, P4, P0, P3, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_PYR, 5, P6, P1, EJ, EF, N0,
    ST_TET, 4, P6, EF, EG, N0,
    ST_PYR, 5, P3, P6, EG, EK, N0,
    ST_PYR, 5, P3, EK, EH, P4, N0,
    ST_TET, 4, P4, EH, EE, N0,
    /* case 161 */ 6,
    ST_TET, 4, P4, EE, EI, EH,
    ST_TET, 4, P6, P3, P2, P1,
    ST_PYR, 5, EA, P1, P3, ED, EK,
    ST_PYR, 5, EJ, EF, P6, P1, EG,
    ST_PYR, 5, P3, P6, EG, EK, P1,
    ST_PYR, 5, EK, EG, EJ, EA, P1,
    /* case 162 */ 11,
    ST_PNT, 4, EB, EA, EK, EK,
    ST_PYR, 5, P3, EK, EH, P4, N0,
    ST_TET, 4, EH, EE, P4, N0,
    ST_PYR, 5, P0, P4, EE, EA, N0,
    ST_TET, 4, P3, P4, P0, N0,
    ST_TET, 4, P6, EF, EG, N0,
    ST_TET, 4, P2, P6, P3, N0,
    ST_PYR, 5, P6, EG, EK, P3, N0,
    ST_TET, 4, P2, P3, P0, N0,
    ST_PYR, 5, P2, P0, EA, EB, N0,
    ST_PYR, 5, P2, EB, EF, P6, N0,
    /* case 163 */ 8,
    ST_PNT, 5, EB, EF, ED, EK, EG,
    ST_TET, 4, P4, EE, EI, EH,
    ST_PYR, 5, P6, P2, EB, EF, N0,
    ST_PYR, 5, P2, P3, ED, EB, N0,
    ST_TET, 4, P3, EK, ED, N0,
    ST_TET, 4, P2, P6, P3, N0,
    ST_PYR, 5, P3, P6, EG, EK, N0,
    ST_TET, 4, P6, EF, EG, N0,
    /* case 164 */ 6,
    ST_TET, 4, P6, EF, EG, EL,
    ST_TET, 4, P3, P0, P1, P4,
    ST_PYR, 5, EE, EJ, P1, P4, EB,
    ST_PYR, 5, EH, P4, P3, EK, EC,
    ST_PYR, 5, P1, EB, EC, P3, P4,
    ST_PYR, 5, EB, EE, EH, EC, P4,
    /* case 165 */ 4,
    ST_TET, 4, EH, EI, EE, P4,
    ST_TET, 4, EK, EC, ED, P3,
    ST_TET, 4, EG, EF, EL, P6,
    ST_TET, 4, EB, EJ, EA, P1,
    /* case 166 */ 8,
    ST_PNT, 5, EA, EC, EE, EH, EK,
    ST_TET, 4, P6, EL, EF, EG,
    ST_PYR, 5, P3, P0, EA, EC, N0,
    ST_PYR, 5, P0, P4, EE, EA, N0,
    ST_TET, 4, P4, EH, EE, N0,
    ST_TET, 4, P0, P3, P4, N0,
    ST_PYR, 5, P4, P3, EK, EH, N0,
    ST_TET, 4, P3, EC, EK, N0,
    /* case 167 */ 3,
    ST_TET, 4, EL, EG, EF, P6,
    ST_TET, 4, EC, P3, ED, EK,
    ST_TET, 4, EE, EI, P4, EH,
    /* case 168 */ 11,
    ST_PNT, 4, EC, ED, EJ, EJ,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_TET, 4, EE, P4, EH, N0,
    ST_PYR, 5, P0, ED, EH, P4, N0,
    ST_TET, 4, P1, P0, P4, N0,
    ST_TET, 4, P6, EF, EG, N0,
    ST_TET, 4, P2, P1, P6, N0,
    ST_PYR, 5, P6, P1, EJ, EF, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_PYR, 5, P2, EC, ED, P0, N0,
    ST_PYR, 5, P2, P6, EG, EC, N0,
    /* case 169 */ 8,
    ST_PNT, 5, EC, EG, EA, EJ, EF,
    ST_TET, 4, P4, EI, EH, EE,
    ST_PYR, 5, P6, EG, EC, P2, N0,
    ST_PYR, 5, P2, EC, EA, P1, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_TET, 4, P2, P1, P6, N0,
    ST_PYR, 5, P1, EJ, EF, P6, N0,
    ST_TET, 4, P6, EF, EG, N0,
    /* case 170 */ 2,
    ST_WDG, 6, P0, ED, EA, P4, EH, EE,
    ST_WDG, 6, P2, EB, EC, P6, EF, EG,
    /* case 171 */ 2,
    ST_WDG, 6, EF, P6, EG, EB, P2, EC,
    ST_TET, 4, EI, EH, P4, EE,
    /* case 172 */ 8,
    ST_PNT, 5, ED, EH, EB, EJ, EE,
    ST_TET, 4, P6, EG, EL, EF,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_PYR, 5, P0, P1, EB, ED, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_TET, 4, P0, P4, P1, N0,
    ST_PYR, 5, P1, P4, EE, EJ, N0,
    ST_TET, 4, P4, EH, EE, N0,
    /* case 173 */ 3,
    ST_TET, 4, EG, EF, EL, P6,
    ST_TET, 4, EH, P4, EI, EE,
    ST_TET, 4, EB, EA, P1, EJ,
    /* case 174 */ 2,
    ST_WDG, 6, EH, P4, EE, ED, P0, EA,
    ST_TET, 4, EL, EF, P6, EG,
    /* case 175 */ 2,
    ST_TET, 4, EE, EH, EI, P4,
    ST_TET, 4, EF, EL, EG, P6,
    /* case 176 */ 9,
    ST_PNT, 5, EJ, EI, EF, EG, EK,
    ST_PYR, 5, P1, P2, P3, P0, N0,
    ST_TET, 4, P3, P6, N0, P2,
    ST_PYR, 5, P0, P3, EK, EI, N0,
    ST_PYR, 5, EJ, P1, P0, EI, N0,
    ST_TET, 4, P6, P2, P1, N0,
    ST_PYR, 5, P6, P1, EJ, EF, N0,
    ST_TET, 4, P6, EF, EG, N0,
    ST_PYR, 5, EK, P3, P6, EG, N0,
    /* case 177 */ 5,
    ST_TET, 4, P6, P3, P2, P1,
    ST_PYR, 5, EA, P1, P3, ED, EK,
    ST_PYR, 5, EJ, EF, P6, P1, EG,
    ST_PYR, 5, P3, P6, EG, EK, P1,
    ST_PYR, 5, EK, EG, EJ, EA, P1,
    /* case 178 */ 9,
    ST_PNT, 6, EI, EA, EB, EF, EG, EK,
    ST_TET, 4, P2, P6, P3, N0,
    ST_PYR, 5, P2, EB, EF, P6, N0,
    ST_TET, 4, P0, P2, P3, N0,
    ST_PYR, 5, P0, EA, EB, P2, N0,
    ST_TET, 4, EI, EA, P0, N0,
    ST_PYR, 5, P0, P3, EK, EI, N0,
    ST_PYR, 5, EK, P3, P6, EG, N0,
    ST_TET, 4, P6, EF, EG, N0,
    /* case 179 */ 7,
    ST_PNT, 5, EF, EB, ED, EG, EK,
    ST_PYR, 5, P2, P3, ED, EB, N0,
    ST_TET, 4, P3, EK, ED, N0,
    ST_PYR, 5, P6, P2, EB, EF, N0,
    ST_TET, 4, EG, P6, EF, N0,
    ST_PYR, 5, EK, P3, P6, EG, N0,
    ST_TET, 4, P3, P2, P6, N0,
    /* case 180 */ 8,
    ST_PNT, 5, EI, EK, EJ, EB, EC,
    ST_TET, 4, P6, EF, EG, EL,
    ST_PYR, 5, P3, EK, EI, P0, N0,
    ST_PYR, 5, P0, EI, EJ, P1, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_TET, 4, P0, P1, P3, N0,
    ST_PYR, 5, P1, EB, EC, P3, N0,
    ST_TET, 4, P3, EC, EK, N0,
    /* case 181 */ 3,
    ST_TET, 4, EG, EF, EL, P6,
    ST_TET, 4, EK, ED, P3, EC,
    ST_TET, 4, EJ, P1, EA, EB,
    /* case 182 */ 2,
    ST_WDG, 6, EI, P0, EA, EK, P3, EC,
    ST_TET, 4, EF, P6, EL, EG,
    /* case 183 */ 2,
    ST_TET, 4, EC, ED, EK, P3,
    ST_TET, 4, EL, EG, EF, P6,
    /* case 184 */ 9,
    ST_PNT, 6, EI, ED, EC, EG, EF, EJ,
    ST_TET, 4, P2, P1, P6, N0,
    ST_PYR, 5, P2, P6, EG, EC, N0,
    ST_TET, 4, P0, P1, P2, N0,
    ST_PYR, 5, P0, P2, EC, ED, N0,
    ST_TET, 4, EI, P0, ED, N0,
    ST_PYR, 5, P0, EI, EJ, P1, N0,
    ST_PYR, 5, EJ, EF, P6, P1, N0,
    ST_TET, 4, P6, EF, EG, N0,
    /* case 185 */ 7,
    ST_PNT, 5, EG, EC, EA, EF, EJ,
    ST_PYR, 5, P2, EC, EA, P1, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    ST_PYR, 5, P6, EG, EC, P2, N0,
    ST_TET, 4, EF, EG, P6, N0,
    ST_PYR, 5, EJ, EF, P6, P1, N0,
    ST_TET, 4, P1, P6, P2, N0,
    /* case 186 */ 2,
    ST_WDG, 6, EF, P6, EG, EB, P2, EC,
    ST_TET, 4, EI, P0, ED, EA,
    /* case 187 */ 1,
    ST_WDG, 6, P6, EG, EF, P2, EC, EB,
    /* case 188 */ 2,
    ST_WDG, 6, ED, P0, EI, EB, P1, EJ,
    ST_TET, 4, EG, P6, EF, EL,
    /* case 189 */ 2,
    ST_TET, 4, EJ, EA, EB, P1,
    ST_TET, 4, EF, EL, EG, P6,
    /* case 190 */ 2,
    ST_TET, 4, EA, EI, ED, P0,
    ST_TET, 4, EL, EG, EF, P6,
    /* case 191 */ 1,
    ST_TET, 4, EF, EL, EG, P6,
    /* case 192 */ 2,
    ST_HEX, 8, EF, P5, P4, EH, EL, P2, P3, EK,
    ST_WDG, 6, P5, P1, P2, P4, P0, P3,
    /* case 193 */ 11,
    ST_PNT, 4, EL, EF, EA, EA,
    ST_PYR, 5, P1, EA, EI, P4, N0,
    ST_TET, 4, EI, EH, P4, N0,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_TET, 4, P1, P4, P5, N0,
    ST_TET, 4, P3, EK, ED, N0,
    ST_TET, 4, P2, P3, P1, N0,
    ST_PYR, 5, P3, ED, EA, P1, N0,
    ST_TET, 4, P2, P1, P5, N0,
    ST_PYR, 5, P2, P5, EF, EL, N0,
    ST_PYR, 5, P2, EL, EK, P3, N0,
    /* case 194 */ 11,
    ST_PNT, 4, EK, EH, EA, EA,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    ST_PYR, 5, P4, EH, EF, P5, N0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_TET, 4, P2, EB, EL, N0,
    ST_TET, 4, P3, P0, P2, N0,
    ST_PYR, 5, P2, P0, EA, EB, N0,
    ST_TET, 4, P3, P4, P0, N0,
    ST_PYR, 5, P3, EK, EH, P4, N0,
    ST_PYR, 5, P3, P2, EL, EK, N0,
    /* case 195 */ 2,
    ST_WDG, 6, P3, ED, EK, P2, EB, EL,
    ST_WDG, 6, P4, EH, EI, P5, EF, EJ,
    /* case 196 */ 9,
    ST_PNT, 5, EH, EF, EK, EC, EB,
    ST_PYR, 5, P4, P5, P1, P0, N0,
    ST_TET, 4, P1, N0, P3, P0,
    ST_PYR, 5, P5, EF, EB, P1, N0,
    ST_PYR, 5, EH, EF, P5, P4, N0,
    ST_TET, 4, P3, P4, P0, N0,
    ST_PYR, 5, P3, EK, EH, P4, N0,
    ST_TET, 4, P3, EC, EK, N0,
    ST_PYR, 5, EB, EC, P3, P1, N0,
    /* case 197 */ 8,
    ST_PNT, 5, EF, EB, EH, EI, EA,
    ST_TET, 4, P3, EC, EK, ED,
    ST_PYR, 5, P1, P5, EF, EB, N0,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_TET, 4, P5, P1, P4, N0,
    ST_PYR, 5, P4, P1, EA, EI, N0,
    ST_TET, 4, P1, EB, EA, N0,
    /* case 198 */ 9,
    ST_PNT, 6, EC, EK, EH, EF, EJ, EA,
    ST_TET, 4, P4, P5, P0, N0,
    ST_PYR, 5, P4, EH, EF, P5, N0,
    ST_TET, 4, P3, P4, P0, N0,
    ST_PYR, 5, P3, EK, EH, P4, N0,
    ST_TET, 4, EC, EK, P3, N0,
    ST_PYR, 5, P3, P0, EA, EC, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    /* case 199 */ 2,
    ST_WDG, 6, EF, P5, EJ, EH, P4, EI,
    ST_TET, 4, EC, P3, ED, EK,
    /* case 200 */ 9,
    ST_PNT, 5, ED, EH, EC, EL, EF,
    ST_PYR, 5, P0, P4, P5, P1, N0,
    ST_TET, 4, P5, N0, P2, P1,
    ST_PYR, 5, P4, EH, EF, P5, N0,
    ST_PYR, 5, ED, EH, P4, P0, N0,
    ST_TET, 4, P2, P0, P1, N0,
    ST_PYR, 5, P2, EC, ED, P0, N0,
    ST_TET, 4, P2, EL, EC, N0,
    ST_PYR, 5, EF, EL, P2, P5, N0,
    /* case 201 */ 9,
    ST_PNT, 6, EC, EL, EF, EH, EI, EA,
    ST_TET, 4, P5, P1, P4, N0,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_TET, 4, P2, P1, P5, N0,
    ST_PYR, 5, P2, P5, EF, EL, N0,
    ST_TET, 4, EC, P2, EL, N0,
    ST_PYR, 5, P2, EC, EA, P1, N0,
    ST_PYR, 5, EA, EI, P4, P1, N0,
    ST_TET, 4, P4, EI, EH, N0,
    /* case 202 */ 8,
    ST_PNT, 5, EH, EF, ED, EA, EJ,
    ST_TET, 4, P2, EL, EC, EB,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_TET, 4, P4, P5, P0, N0,
    ST_PYR, 5, P0, P5, EJ, EA, N0,
    ST_TET, 4, P5, EF, EJ, N0,
    /* case 203 */ 2,
    ST_WDG, 6, EF, P5, EJ, EH, P4, EI,
    ST_TET, 4, EC, EB, P2, EL,
    /* case 204 */ 1,
    ST_HEX, 8, ED, EB, EF, EH, P0, P1, P5, P4,
    /* case 205 */ 7,
    ST_PNT, 5, EB, EF, EH, EA, EI,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_TET, 4, P4, EI, EH, N0,
    ST_PYR, 5, P1, P5, EF, EB, N0,
    ST_TET, 4, EA, P1, EB, N0,
    ST_PYR, 5, EI, P4, P1, EA, N0,
    ST_TET, 4, P4, P5, P1, N0,
    /* case 206 */ 7,
    ST_PNT, 5, EF, EH, ED, EJ, EA,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_TET, 4, P0, EA, ED, N0,
    ST_PYR, 5, P5, P4, EH, EF, N0,
    ST_TET, 4, EJ, P5, EF, N0,
    ST_PYR, 5, EA, P0, P5, EJ, N0,
    ST_TET, 4, P0, P4, P5, N0,
    /* case 207 */ 1,
    ST_WDG, 6, P5, EJ, EF, P4, EI, EH,
    /* case 208 */ 9,
    ST_PNT, 5, EI, EK, EE, EF, EL,
    ST_PYR, 5, P0, P1, P2, P3, N0,
    ST_TET, 4, P2, P5, N0, P1,
    ST_PYR, 5, P3, P2, EL, EK, N0,
    ST_PYR, 5, EI, P0, P3, EK, N0,
    ST_TET, 4, P5, P1, P0, N0,
    ST_PYR, 5, P5, P0, EI, EE, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_PYR, 5, EL, P2, P5, EF, N0,
    /* case 209 */ 9,
    ST_PNT, 6, EE, EF, EL, EK, ED, EA,
    ST_TET, 4, P2, P3, P1, N0,
    ST_PYR, 5, P2, EL, EK, P3, N0,
    ST_TET, 4, P5, P2, P1, N0,
    ST_PYR, 5, P5, EF, EL, P2, N0,
    ST_TET, 4, EE, EF, P5, N0,
    ST_PYR, 5, P5, P1, EA, EE, N0,
    ST_PYR, 5, EA, P1, P3, ED, N0,
    ST_TET, 4, P3, EK, ED, N0,
    /* case 210 */ 8,
    ST_PNT, 5, EK, EL, EI, EA, EB,
    ST_TET, 4, P5, EE, EF, EJ,
    ST_PYR, 5, P2, EL, EK, P3, N0,
    ST_PYR, 5, P3, EK, EI, P0, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_TET, 4, P3, P0, P2, N0,
    ST_PYR, 5, P0, EA, EB, P2, N0,
    ST_TET, 4, P2, EB, EL, N0,
    /* case 211 */ 2,
    ST_WDG, 6, EK, P3, ED, EL, P2, EB,
    ST_TET, 4, EE, P5, EJ, EF,
    /* case 212 */ 9,
    ST_PNT, 6, EK, EC, EB, EF, EE, EI,
    ST_TET, 4, P1, P0, P5, N0,
    ST_PYR, 5, P1, P5, EF, EB, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_PYR, 5, P3, P1, EB, EC, N0,
    ST_TET, 4, EK, P3, EC, N0,
    ST_PYR, 5, P3, EK, EI, P0, N0,
    ST_PYR, 5, EI, EE, P5, P0, N0,
    ST_TET, 4, P5, EE, EF, N0,
    /* case 213 */ 2,
    ST_WDG, 6, EB, P1, EA, EF, P5, EE,
    ST_TET, 4, EK, ED, P3, EC,
    /* case 214 */ 2,
    ST_WDG, 6, EI, P0, EA, EK, P3, EC,
    ST_TET, 4, EF, EJ, P5, EE,
    /* case 215 */ 2,
    ST_TET, 4, EF, EE, EJ, P5,
    ST_TET, 4, EC, ED, EK, P3,
    /* case 216 */ 5,
    ST_TET, 4, P2, P0, P1, P5,
    ST_PYR, 5, EE, P5, P0, EI, ED,
    ST_PYR, 5, EF, EL, P2, P5, EC,
    ST_PYR, 5, P0, P2, EC, ED, P5,
    ST_PYR, 5, ED, EC, EF, EE, P5,
    /* case 217 */ 7,
    ST_PNT, 5, EC, EA, EE, EL, EF,
    ST_PYR, 5, P1, EA, EE, P5, N0,
    ST_TET, 4, P5, EE, EF, N0,
    ST_PYR, 5, P2, EC, EA, P1, N0,
    ST_TET, 4, EL, EC, P2, N0,
    ST_PYR, 5, EF, EL, P2, P5, N0,
    ST_TET, 4, P5, P2, P1, N0,
    /* case 218 */ 3,
    ST_TET, 4, EC, EL, EB, P2,
    ST_TET, 4, ED, EI, P0, EA,
    ST_TET, 4, EF, P5, EE, EJ,
    /* case 219 */ 2,
    ST_TET, 4, EL, EB, EC, P2,
    ST_TET, 4, EF, EE, EJ, P5,
    /* case 220 */ 7,
    ST_PNT, 5, EF, EB, ED, EE, EI,
    ST_PYR, 5, P1, EB, ED, P0, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_PYR, 5, P5, EF, EB, P1, N0,
    ST_TET, 4, EE, EF, P5, N0,
    ST_PYR, 5, EI, EE, P5, P0, N0,
    ST_TET, 4, P0, P5, P1, N0,
    /* case 221 */ 1,
    ST_WDG, 6, P1, EA, EB, P5, EE, EF,
    /* case 222 */ 2,
    ST_TET, 4, EA, EI, ED, P0,
    ST_TET, 4, EJ, EF, EE, P5,
    /* case 223 */ 1,
    ST_TET, 4, EJ, EF, EE, P5,
    /* case 224 */ 9,
    ST_PNT, 5, EK, EL, EH, EE, EJ,
    ST_PYR, 5, P3, P0, P1, P2, N0,
    ST_TET, 4, P1, P4, N0, P0,
    ST_PYR, 5, P2, P1, EJ, EL, N0,
    ST_PYR, 5, EK, P3, P2, EL, N0,
    ST_TET, 4, P4, P0, P3, N0,
    ST_PYR, 5, P4, P3, EK, EH, N0,
    ST_TET, 4, P4, EH, EE, N0,
    ST_PYR, 5, EJ, P1, P4, EE, N0,
    /* case 225 */ 8,
    ST_PNT, 5, EL, EJ, EK, ED, EA,
    ST_TET, 4, P4, EH, EE, EI,
    ST_PYR, 5, P1, EJ, EL, P2, N0,
    ST_PYR, 5, P2, EL, EK, P3, N0,
    ST_TET, 4, P3, EK, ED, N0,
    ST_TET, 4, P2, P3, P1, N0,
    ST_PYR, 5, P3, ED, EA, P1, N0,
    ST_TET, 4, P1, EA, EJ, N0,
    /* case 226 */ 9,
    ST_PNT, 6, EE, EH, EK, EL, EB, EA,
    ST_TET, 4, P3, P0, P2, N0,
    ST_PYR, 5, P3, P2, EL, EK, N0,
    ST_TET, 4, P4, P0, P3, N0,
    ST_PYR, 5, P4, P3, EK, EH, N0,
    ST_TET, 4, EE, P4, EH, N0,
    ST_PYR, 5, P4, EE, EA, P0, N0,
    ST_PYR, 5, EA, EB, P2, P0, N0,
    ST_TET, 4, P2, EB, EL, N0,
    /* case 227 */ 2,
    ST_WDG, 6, EK, P3, ED, EL, P2, EB,
    ST_TET, 4, EE, EI, P4, EH,
    /* case 228 */ 5,
    ST_TET, 4, P3, P0, P1, P4,
    ST_PYR, 5, EE, EJ, P1, P4, EB,
    ST_PYR, 5, EH, P4, P3, EK, EC,
    ST_PYR, 5, P1, EB, EC, P3, P4,
    ST_PYR, 5, EB, EE, EH, EC, P4,
    /* case 229 */ 3,
    ST_TET, 4, EC, ED, EK, P3,
    ST_TET, 4, EB, P1, EJ, EA,
    ST_TET, 4, EH, EE, P4, EI,
    /* case 230 */ 7,
    ST_PNT, 5, EC, EA, EE, EK, EH,
    ST_PYR, 5, P0, P4, EE, EA, N0,
    ST_TET, 4, P4, EH, EE, N0,
    ST_PYR, 5, P3, P0, EA, EC, N0,
    ST_TET, 4, EK, P3, EC, N0,
    ST_PYR, 5, EH, P4, P3, EK, N0,
    ST_TET, 4, P4, P0, P3, N0,
    /* case 231 */ 2,
    ST_TET, 4, EK, EC, ED, P3,
    ST_TET, 4, EH, EI, EE, P4,
    /* case 232 */ 9,
    ST_PNT, 6, EH, EE, EJ, EL, EC, ED,
    ST_TET, 4, P1, P2, P0, N0,
    ST_PYR, 5, P1, EJ, EL, P2, N0,
    ST_TET, 4, P4, P1, P0, N0,
    ST_PYR, 5, P4, EE, EJ, P1, N0,
    ST_TET, 4, EH, EE, P4, N0,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_PYR, 5, ED, P0, P2, EC, N0,
    ST_TET, 4, P2, EL, EC, N0,
    /* case 233 */ 2,
    ST_WDG, 6, EL, P2, EC, EJ, P1, EA,
    ST_TET, 4, EH, P4, EI, EE,
    /* case 234 */ 2,
    ST_WDG, 6, EH, P4, EE, ED, P0, EA,
    ST_TET, 4, EL, P2, EB, EC,
    /* case 235 */ 2,
    ST_TET, 4, EL, EB, EC, P2,
    ST_TET, 4, EE, EH, EI, P4,
    /* case 236 */ 7,
    ST_PNT, 5, EH, ED, EB, EE, EJ,
    ST_PYR, 5, P0, P1, EB, ED, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_PYR, 5, P4, P0, ED, EH, N0,
    ST_TET, 4, EE, P4, EH, N0,
    ST_PYR, 5, EJ, P1, P4, EE, N0,
    ST_TET, 4, P1, P0, P4, N0,
    /* case 237 */ 2,
    ST_TET, 4, EE, EH, EI, P4,
    ST_TET, 4, EJ, EA, EB, P1,
    /* case 238 */ 1,
    ST_WDG, 6, P4, EE, EH, P0, EA, ED,
    /* case 239 */ 1,
    ST_TET, 4, EH, EI, EE, P4,
    /* case 240 */ 1,
    ST_HEX, 8, P0, P1, P2, P3, EI, EJ, EL, EK,
    /* case 241 */ 7,
    ST_PNT, 5, EJ, EL, EK, EA, ED,
    ST_PYR, 5, P2, EL, EK, P3, N0,
    ST_TET, 4, P3, EK, ED, N0,
    ST_PYR, 5, P1, EJ, EL, P2, N0,
    ST_TET, 4, EA, EJ, P1, N0,
    ST_PYR, 5, ED, EA, P1, P3, N0,
    ST_TET, 4, P3, P1, P2, N0,
    /* case 242 */ 7,
    ST_PNT, 5, EL, EK, EI, EB, EA,
    ST_PYR, 5, P3, EK, EI, P0, N0,
    ST_TET, 4, P0, EI, EA, N0,
    ST_PYR, 5, P2, EL, EK, P3, N0,
    ST_TET, 4, EB, EL, P2, N0,
    ST_PYR, 5, EA, EB, P2, P0, N0,
    ST_TET, 4, P0, P2, P3, N0,
    /* case 243 */ 1,
    ST_WDG, 6, P3, ED, EK, P2, EB, EL,
    /* case 244 */ 7,
    ST_PNT, 5, EK, EI, EJ, EC, EB,
    ST_PYR, 5, P0, EI, EJ, P1, N0,
    ST_TET, 4, P1, EJ, EB, N0,
    ST_PYR, 5, P3, EK, EI, P0, N0,
    ST_TET, 4, EC, EK, P3, N0,
    ST_PYR, 5, EB, EC, P3, P1, N0,
    ST_TET, 4, P1, P3, P0, N0,
    /* case 245 */ 2,
    ST_TET, 4, EC, ED, EK, P3,
    ST_TET, 4, EB, EJ, EA, P1,
    /* case 246 */ 1,
    ST_WDG, 6, P0, EA, EI, P3, EC, EK,
    /* case 247 */ 1,
    ST_TET, 4, EK, EC, ED, P3,
    /* case 248 */ 7,
    ST_PNT, 5, EL, EJ, EI, EC, ED,
    ST_PYR, 5, P1, P0, EI, EJ, N0,
    ST_TET, 4, P0, ED, EI, N0,
    ST_PYR, 5, P2, P1, EJ, EL, N0,
    ST_TET, 4, EC, P2, EL, N0,
    ST_PYR, 5, ED, P0, P2, EC, N0,
    ST_TET, 4, P0, P1, P2, N0,
    /* case 249 */ 1,
    ST_WDG, 6, P2, EC, EL, P1, EA, EJ,
    /* case 250 */ 2,
    ST_TET, 4, EA, EI, ED, P0,
    ST_TET, 4, EB, EC, EL, P2,
    /* case 251 */ 1,
    ST_TET, 4, EB, EC, EL, P2,
    /* case 252 */ 1,
    ST_WDG, 6, P0, EI, ED, P1, EJ, EB,
    /* case 253 */ 1,
    ST_TET, 4, EB, EJ, EA, P1,
    /* case 254 */ 1,
    ST_TET, 4, ED, EA, EI, P0,
    /* case 255 */ 0,
    // VTK_WEDGE
    /* case 0 */ 1,
    ST_WDG, 6, P0, P1, P2, P3, P4, P5,
    /* case 1 */ 2,
    ST_WDG, 6, EA, EC, EG, P1, P2, P3,
    ST_PYR, 5, P1, P2, P5, P4, P3,
    /* case 2 */ 2,
    ST_WDG, 6, EB, EA, EH, P2, P0, P4,
    ST_PYR, 5, P2, P0, P3, P5, P4,
    /* case 3 */ 8,
    ST_PNT, 7, P2, P3, P4, EB, EC, EG, EH,
    ST_TET, 4, P4, P5, P3, P2,
    ST_TET, 4, P2, P3, P4, N0,
    ST_PYR, 5, EG, EH, P4, P3, N0,
    ST_PYR, 5, EB, EH, EG, EC, N0,
    ST_TET, 4, P2, EB, EC, N0,
    ST_PYR, 5, P2, EC, EG, P3, N0,
    ST_PYR, 5, EH, EB, P2, P4, N0,
    /* case 4 */ 2,
    ST_WDG, 6, EC, EB, EI, P0, P1, P5,
    ST_PYR, 5, P0, P1, P4, P3, P5,
    /* case 5 */ 8,
    ST_PNT, 7, P1, P5, P3, EA, EB, EI, EG,
    ST_TET, 4, P3, P4, P5, P1,
    ST_TET, 4, P1, P5, P3, N0,
    ST_PYR, 5, EI, EG, P3, P5, N0,
    ST_PYR, 5, EA, EG, EI, EB, N0,
    ST_TET, 4, P1, EA, EB, N0,
    ST_PYR, 5, P1, EB, EI, P5, N0,
    ST_PYR, 5, EG, EA, P1, P3, N0,
    /* case 6 */ 8,
    ST_PNT, 7, P0, P4, P5, EC, EA, EH, EI,
    ST_TET, 4, P5, P3, P4, P0,
    ST_TET, 4, P0, P4, P5, N0,
    ST_PYR, 5, EH, EI, P5, P4, N0,
    ST_PYR, 5, EC, EI, EH, EA, N0,
    ST_TET, 4, P0, EC, EA, N0,
    ST_PYR, 5, P0, EA, EH, P4, N0,
    ST_PYR, 5, EI, EC, P0, P5, N0,
    /* case 7 */ 1,
    ST_WDG, 6, EG, EH, EI, P3, P4, P5,
    /* case 8 */ 2,
    ST_WDG, 6, P4, P5, P0, ED, EF, EG,
    ST_PYR, 5, P4, P1, P2, P5, P0,
    /* case 9 */ 1,
    ST_HEX, 8, P1, P2, P5, P4, EA, EC, EF, ED,
    /* case 10 */ 9,
    ST_PNT, 6, EA, EB, EH, ED, EF, EG,
    ST_PYR, 5, P5, P0, EG, EF, N0,
    ST_TET, 4, P0, EA, EG, N0,
    ST_PYR, 5, P0, P2, EB, EA, N0,
    ST_TET, 4, P5, P2, P0, N0,
    ST_PYR, 5, P4, EH, EB, P2, N0,
    ST_TET, 4, P5, P4, P2, N0,
    ST_PYR, 5, EF, ED, P4, P5, N0,
    ST_TET, 4, ED, EH, P4, N0,
    /* case 11 */ 7,
    ST_PNT, 5, EB, EC, EF, ED, EH,
    ST_PYR, 5, P4, P5, EF, ED, N0,
    ST_TET, 4, ED, EH, P4, N0,
    ST_PYR, 5, EC, EF, P5, P2, N0,
    ST_PYR, 5, EB, P2, P4, EH, N0,
    ST_TET, 4, P4, P2, P5, N0,
    ST_TET, 4, P2, EB, EC, N0,
    /* case 12 */ 9,
    ST_PNT, 6, EF, ED, EG, EC, EB, EI,
    ST_PYR, 5, P1, EB, EI, P5, N0,
    ST_TET, 4, P5, EI, EF, N0,
    ST_PYR, 5, P5, EF, ED, P4, N0,
    ST_TET, 4, P1, P5, P4, N0,
    ST_PYR, 5, P0, P4, ED, EG, N0,
    ST_TET, 4, P1, P4, P0, N0,
    ST_PYR, 5, EB, P1, P0, EC, N0,
    ST_TET, 4, EC, P0, EG, N0,
    /* case 13 */ 7,
    ST_PNT, 5, EB, EA, ED, EF, EI,
    ST_PYR, 5, P5, EF, ED, P4, N0,
    ST_TET, 4, EF, P5, EI, N0,
    ST_PYR, 5, EA, P1, P4, ED, N0,
    ST_PYR, 5, EB, EI, P5, P1, N0,
    ST_TET, 4, P5, P4, P1, N0,
    ST_TET, 4, P1, EA, EB, N0,
    /* case 14 */ 2,
    ST_TET, 4, P0, EC, EA, EG,
    ST_WDG, 6, EF, EI, P5, ED, EH, P4,
    /* case 15 */ 1,
    ST_WDG, 6, ED, P4, EH, EF, P5, EI,
    /* case 16 */ 2,
    ST_WDG, 6, P5, P3, P1, EE, ED, EH,
    ST_PYR, 5, P5, P2, P0, P3, P1,
    /* case 17 */ 9,
    ST_PNT, 6, ED, EE, EH, EA, EC, EG,
    ST_PYR, 5, P2, EC, EG, P3, N0,
    ST_TET, 4, P3, EG, ED, N0,
    ST_PYR, 5, P3, ED, EE, P5, N0,
    ST_TET, 4, P2, P3, P5, N0,
    ST_PYR, 5, P1, P5, EE, EH, N0,
    ST_TET, 4, P2, P5, P1, N0,
    ST_PYR, 5, EC, P2, P1, EA, N0,
    ST_TET, 4, EA, P1, EH, N0,
    /* case 18 */ 1,
    ST_HEX, 8, P2, P0, P3, P5, EB, EA, ED, EE,
    /* case 19 */ 7,
    ST_PNT, 5, EC, EB, EE, ED, EG,
    ST_PYR, 5, P3, ED, EE, P5, N0,
    ST_TET, 4, ED, P3, EG, N0,
    ST_PYR, 5, EB, P2, P5, EE, N0,
    ST_PYR, 5, EC, EG, P3, P2, N0,
    ST_TET, 4, P3, P5, P2, N0,
    ST_TET, 4, P2, EB, EC, N0,
    /* case 20 */ 9,
    ST_PNT, 6, EB, EC, EI, EE, ED, EH,
    ST_PYR, 5, P3, P1, EH, ED, N0,
    ST_TET, 4, P1, EB, EH, N0,
    ST_PYR, 5, P1, P0, EC, EB, N0,
    ST_TET, 4, P3, P0, P1, N0,
    ST_PYR, 5, P5, EI, EC, P0, N0,
    ST_TET, 4, P3, P5, P0, N0,
    ST_PYR, 5, ED, EE, P5, P3, N0,
    ST_TET, 4, EE, EI, P5, N0,
    /* case 21 */ 2,
    ST_TET, 4, P1, EA, EB, EH,
    ST_WDG, 6, ED, EG, P3, EE, EI, P5,
    /* case 22 */ 7,
    ST_PNT, 5, EC, EA, ED, EE, EI,
    ST_PYR, 5, P5, P3, ED, EE, N0,
    ST_TET, 4, EE, EI, P5, N0,
    ST_PYR, 5, EA, ED, P3, P0, N0,
    ST_PYR, 5, EC, P0, P5, EI, N0,
    ST_TET, 4, P5, P0, P3, N0,
    ST_TET, 4, P0, EC, EA, N0,
    /* case 23 */ 1,
    ST_WDG, 6, EE, P5, EI, ED, P3, EG,
    /* case 24 */ 8,
    ST_PNT, 7, P5, P0, P1, EE, EF, EG, EH,
    ST_TET, 4, P1, P0, P2, P5,
    ST_TET, 4, P5, P1, P0, N0,
    ST_PYR, 5, EG, P0, P1, EH, N0,
    ST_PYR, 5, EE, EF, EG, EH, N0,
    ST_TET, 4, P5, EF, EE, N0,
    ST_PYR, 5, P5, P0, EG, EF, N0,
    ST_PYR, 5, EH, P1, P5, EE, N0,
    /* case 25 */ 7,
    ST_PNT, 5, EE, EF, EC, EA, EH,
    ST_PYR, 5, P1, EA, EC, P2, N0,
    ST_TET, 4, EA, P1, EH, N0,
    ST_PYR, 5, EF, P5, P2, EC, N0,
    ST_PYR, 5, EE, EH, P1, P5, N0,
    ST_TET, 4, P1, P2, P5, N0,
    ST_TET, 4, P5, EF, EE, N0,
    /* case 26 */ 7,
    ST_PNT, 5, EF, EE, EB, EA, EG,
    ST_PYR, 5, P0, P2, EB, EA, N0,
    ST_TET, 4, EA, EG, P0, N0,
    ST_PYR, 5, EE, EB, P2, P5, N0,
    ST_PYR, 5, EF, P5, P0, EG, N0,
    ST_TET, 4, P0, P5, P2, N0,
    ST_TET, 4, P5, EF, EE, N0,
    /* case 27 */ 1,
    ST_WDG, 6, EF, P5, EE, EC, P2, EB,
    /* case 28 */ 2,
    ST_TET, 4, P5, EF, EE, EI,
    ST_WDG, 6, EC, EG, P0, EB, EH, P1,
    /* case 29 */ 2,
    ST_TET, 4, P1, EA, EB, EH,
    ST_TET, 4, EF, EE, P5, EI,
    /* case 30 */ 2,
    ST_TET, 4, P5, EF, EE, EI,
    ST_TET, 4, EA, P0, EC, EG,
    /* case 31 */ 1,
    ST_TET, 4, EF, EI, EE, P5,
    /* case 32 */ 2,
    ST_WDG, 6, P3, P4, P2, EF, EE, EI,
    ST_PYR, 5, P3, P0, P1, P4, P2,
    /* case 33 */ 9,
    ST_PNT, 6, EC, EA, EG, EF, EE, EI,
    ST_PYR, 5, P4, P2, EI, EE, N0,
    ST_TET, 4, P2, EC, EI, N0,
    ST_PYR, 5, P2, P1, EA, EC, N0,
    ST_TET, 4, P4, P1, P2, N0,
    ST_PYR, 5, P3, EG, EA, P1, N0,
    ST_TET, 4, P4, P3, P1, N0,
    ST_PYR, 5, EE, EF, P3, P4, N0,
    ST_TET, 4, EF, EG, P3, N0,
    /* case 34 */ 9,
    ST_PNT, 6, EE, EF, EI, EB, EA, EH,
    ST_PYR, 5, P0, EA, EH, P4, N0,
    ST_TET, 4, P4, EH, EE, N0,
    ST_PYR, 5, P4, EE, EF, P3, N0,
    ST_TET, 4, P0, P4, P3, N0,
    ST_PYR, 5, P2, P3, EF, EI, N0,
    ST_TET, 4, P0, P3, P2, N0,
    ST_PYR, 5, EA, P0, P2, EB, N0,
    ST_TET, 4, EB, P2, EI, N0,
    /* case 35 */ 2,
    ST_TET, 4, P2, EB, EC, EI,
    ST_WDG, 6, EE, EH, P4, EF, EG, P3,
    /* case 36 */ 1,
    ST_HEX, 8, P0, P1, P4, P3, EC, EB, EE, EF,
    /* case 37 */ 7,
    ST_PNT, 5, EA, EB, EE, EF, EG,
    ST_PYR, 5, P3, P4, EE, EF, N0,
    ST_TET, 4, EF, EG, P3, N0,
    ST_PYR, 5, EB, EE, P4, P1, N0,
    ST_PYR, 5, EA, P1, P3, EG, N0,
    ST_TET, 4, P3, P1, P4, N0,
    ST_TET, 4, P1, EA, EB, N0,
    /* case 38 */ 7,
    ST_PNT, 5, EA, EC, EF, EE, EH,
    ST_PYR, 5, P4, EE, EF, P3, N0,
    ST_TET, 4, EE, P4, EH, N0,
    ST_PYR, 5, EC, P0, P3, EF, N0,
    ST_PYR, 5, EA, EH, P4, P0, N0,
    ST_TET, 4, P4, P3, P0, N0,
    ST_TET, 4, P0, EC, EA, N0,
    /* case 39 */ 1,
    ST_WDG, 6, EF, P3, EG, EE, P4, EH,
    /* case 40 */ 8,
    ST_PNT, 7, P4, P2, P0, ED, EE, EI, EG,
    ST_TET, 4, P0, P2, P1, P4,
    ST_TET, 4, P4, P0, P2, N0,
    ST_PYR, 5, EI, P2, P0, EG, N0,
    ST_PYR, 5, ED, EE, EI, EG, N0,
    ST_TET, 4, P4, EE, ED, N0,
    ST_PYR, 5, P4, P2, EI, EE, N0,
    ST_PYR, 5, EG, P0, P4, ED, N0,
    /* case 41 */ 7,
    ST_PNT, 5, EE, ED, EA, EC, EI,
    ST_PYR, 5, P2, P1, EA, EC, N0,
    ST_TET, 4, EC, EI, P2, N0,
    ST_PYR, 5, ED, EA, P1, P4, N0,
    ST_PYR, 5, EE, P4, P2, EI, N0,
    ST_TET, 4, P2, P4, P1, N0,
    ST_TET, 4, P4, EE, ED, N0,
    /* case 42 */ 2,
    ST_TET, 4, P4, EE, ED, EH,
    ST_WDG, 6, EB, EI, P2, EA, EG, P0,
    /* case 43 */ 2,
    ST_TET, 4, P4, EE, ED, EH,
    ST_TET, 4, EC, P2, EB, EI,
    /* case 44 */ 7,
    ST_PNT, 5, ED, EE, EB, EC, EG,
    ST_PYR, 5, P0, EC, EB, P1, N0,
    ST_TET, 4, EC, P0, EG, N0,
    ST_PYR, 5, EE, P4, P1, EB, N0,
    ST_PYR, 5, ED, EG, P0, P4, N0,
    ST_TET, 4, P0, P1, P4, N0,
    ST_TET, 4, P4, EE, ED, N0,
    /* case 45 */ 1,
    ST_WDG, 6, EE, P4, ED, EB, P1, EA,
    /* case 46 */ 2,
    ST_TET, 4, P0, EC, EA, EG,
    ST_TET, 4, EE, ED, P4, EH,
    /* case 47 */ 1,
    ST_TET, 4, EE, EH, ED, P4,
    /* case 48 */ 8,
    ST_PNT, 7, P3, P1, P2, EF, ED, EH, EI,
    ST_TET, 4, P2, P1, P0, P3,
    ST_TET, 4, P3, P2, P1, N0,
    ST_PYR, 5, EH, P1, P2, EI, N0,
    ST_PYR, 5, EF, ED, EH, EI, N0,
    ST_TET, 4, P3, ED, EF, N0,
    ST_PYR, 5, P3, P1, EH, ED, N0,
    ST_PYR, 5, EI, P2, P3, EF, N0,
    /* case 49 */ 2,
    ST_TET, 4, P3, ED, EF, EG,
    ST_WDG, 6, EA, EH, P1, EC, EI, P2,
    /* case 50 */ 7,
    ST_PNT, 5, EF, ED, EA, EB, EI,
    ST_PYR, 5, P2, EB, EA, P0, N0,
    ST_TET, 4, EB, P2, EI, N0,
    ST_PYR, 5, ED, P3, P0, EA, N0,
    ST_PYR, 5, EF, EI, P2, P3, N0,
    ST_TET, 4, P2, P0, P3, N0,
    ST_TET, 4, P3, ED, EF, N0,
    /* case 51 */ 2,
    ST_TET, 4, P2, EB, EC, EI,
    ST_TET, 4, ED, EF, P3, EG,
    /* case 52 */ 7,
    ST_PNT, 5, ED, EF, EC, EB, EH,
    ST_PYR, 5, P1, P0, EC, EB, N0,
    ST_TET, 4, EB, EH, P1, N0,
    ST_PYR, 5, EF, EC, P0, P3, N0,
    ST_PYR, 5, ED, P3, P1, EH, N0,
    ST_TET, 4, P1, P3, P0, N0,
    ST_TET, 4, P3, ED, EF, N0,
    /* case 53 */ 2,
    ST_TET, 4, P3, ED, EF, EG,
    ST_TET, 4, EB, P1, EA, EH,
    /* case 54 */ 1,
    ST_WDG, 6, ED, P3, EF, EA, P0, EC,
    /* case 55 */ 1,
    ST_TET, 4, ED, EG, EF, P3,
    /* case 56 */ 1,
    ST_WDG, 6, P0, P1, P2, EG, EH, EI,
    /* case 57 */ 1,
    ST_WDG, 6, EC, P2, EI, EA, P1, EH,
    /* case 58 */ 1,
    ST_WDG, 6, EA, P0, EG, EB, P2, EI,
    /* case 59 */ 1,
    ST_TET, 4, EC, EB, EI, P2,
    /* case 60 */ 1,
    ST_WDG, 6, EB, P1, EH, EC, P0, EG,
    /* case 61 */ 1,
    ST_TET, 4, EB, EA, EH, P1,
    /* case 62 */ 1,
    ST_TET, 4, EA, EC, EG, P0,
    /* case 63 */ 0,
    // VTK_PYRAMID
    /* case 0 */ 1,
    ST_PYR, 5, P0, P1, P2, P3, P4,
    /* case 1 */ 2,
    ST_WDG, 6, EA, EE, ED, P1, P4, P3,
    ST_TET, 4, P1, P2, P3, P4,
    /* case 2 */ 2,
    ST_WDG, 6, EB, EF, EA, P2, P4, P0,
    ST_TET, 4, P2, P3, P0, P4,
    /* case 3 */ 7,
    ST_PNT, 7, P4, EF, EE, EB, ED, P2, P3,
    ST_TET, 4, EE, P4, EF, N0,
    ST_PYR, 5, EB, ED, EE, EF, N0,
    ST_PYR, 5, EB, EF, P4, P2, N0,
    ST_TET, 4, P2, P4, P3, N0,
    ST_PYR, 5, P3, P4, EE, ED, N0,
    ST_PYR, 5, P2, P3, ED, EB, N0,
    /* case 4 */ 2,
    ST_WDG, 6, EC, EG, EB, P3, P4, P1,
    ST_TET, 4, P3, P0, P1, P4,
    /* case 5 */ 2,
    ST_WDG, 6, EE, P4, EG, EA, P1, EB,
    ST_WDG, 6, P4, EE, EG, P3, ED, EC,
    /* case 6 */ 7,
    ST_PNT, 7, P4, EG, EF, EC, EA, P3, P0,
    ST_TET, 4, EF, P4, EG, N0,
    ST_PYR, 5, EC, EA, EF, EG, N0,
    ST_PYR, 5, EC, EG, P4, P3, N0,
    ST_TET, 4, P3, P4, P0, N0,
    ST_PYR, 5, P0, P4, EF, EA, N0,
    ST_PYR, 5, P3, P0, EA, EC, N0,
    /* case 7 */ 2,
    ST_TET, 4, EE, EF, EG, P4,
    ST_WDG, 6, EC, ED, P3, EG, EE, P4,
    /* case 8 */ 2,
    ST_WDG, 6, ED, EH, EC, P0, P4, P2,
    ST_TET, 4, P0, P1, P2, P4,
    /* case 9 */ 7,
    ST_PNT, 7, P4, EE, EH, EA, EC, P1, P2,
    ST_TET, 4, EH, P4, EE, N0,
    ST_PYR, 5, EA, EC, EH, EE, N0,
    ST_PYR, 5, EA, EE, P4, P1, N0,
    ST_TET, 4, P1, P4, P2, N0,
    ST_PYR, 5, P2, P4, EH, EC, N0,
    ST_PYR, 5, P1, P2, EC, EA, N0,
    /* case 10 */ 2,
    ST_WDG, 6, EH, P4, EF, ED, P0, EA,
    ST_WDG, 6, P4, EH, EF, P2, EC, EB,
    /* case 11 */ 2,
    ST_TET, 4, EH, EE, EF, P4,
    ST_WDG, 6, EB, EC, P2, EF, EH, P4,
    /* case 12 */ 7,
    ST_PNT, 7, P4, EH, EG, ED, EB, P0, P1,
    ST_TET, 4, EG, P4, EH, N0,
    ST_PYR, 5, ED, EB, EG, EH, N0,
    ST_PYR, 5, ED, EH, P4, P0, N0,
    ST_TET, 4, P0, P4, P1, N0,
    ST_PYR, 5, P1, P4, EG, EB, N0,
    ST_PYR, 5, P0, P1, EB, ED, N0,
    /* case 13 */ 2,
    ST_TET, 4, EG, EH, EE, P4,
    ST_WDG, 6, EA, EB, P1, EE, EG, P4,
    /* case 14 */ 2,
    ST_TET, 4, EF, EG, EH, P4,
    ST_WDG, 6, ED, EA, P0, EH, EF, P4,
    /* case 15 */ 1,
    ST_PYR, 5, EE, EF, EG, EH, P4,
    /* case 16 */ 1,
    ST_HEX, 8, P0, P1, P2, P3, EE, EF, EG, EH,
    /* case 17 */ 2,
    ST_WDG, 6, ED, EH, P3, EA, EF, P1,
    ST_WDG, 6, EF, EG, EH, P1, P2, P3,
    /* case 18 */ 2,
    ST_WDG, 6, EA, EE, P0, EB, EG, P2,
    ST_WDG, 6, EG, EH, EE, P2, P3, P0,
    /* case 19 */ 1,
    ST_WDG, 6, ED, EH, P3, EB, EG, P2,
    /* case 20 */ 2,
    ST_WDG, 6, EB, EF, P1, EC, EH, P3,
    ST_WDG, 6, EH, EE, EF, P3, P0, P1,
    /* case 21 */ 2,
    ST_TET, 4, EA, P1, EB, EF,
    ST_TET, 4, P3, ED, EC, EH,
    /* case 22 */ 1,
    ST_WDG, 6, EA, EE, P0, EC, EH, P3,
    /* case 23 */ 1,
    ST_TET, 4, P3, ED, EC, EH,
    /* case 24 */ 2,
    ST_WDG, 6, EC, EG, P2, ED, EE, P0,
    ST_WDG, 6, EE, EF, EG, P0, P1, P2,
    /* case 25 */ 1,
    ST_WDG, 6, EC, EG, P2, EA, EF, P1,
    /* case 26 */ 2,
    ST_TET, 4, ED, P0, EA, EE,
    ST_TET, 4, P2, EC, EB, EG,
    /* case 27 */ 1,
    ST_TET, 4, P2, EC, EB, EG,
    /* case 28 */ 1,
    ST_WDG, 6, EB, EF, P1, ED, EE, P0,
    /* case 29 */ 1,
    ST_TET, 4, P1, EB, EA, EF,
    /* case 30 */ 1,
    ST_TET, 4, P0, EA, ED, EE,
    /* case 31 */ 0,
  };

  // offset into CellCases for each cell
  static constexpr uint16_t StartCellCasesInsideOut[] = {
    // VTK_VERTEX
    0, 4, // case 0 - 1
    // VTK_LINE
    5, 10, 15, 20, // case 0 - 3
    // VTK_TRIANGLE
    21, 27, 34, 41, 47, 54, 60, 66, // case 0 - 7
    // VTK_PIXEL
    67, 74, 86, 98, 105, 117, 124, 135, // case 0 - 7
    141, 153, 164, 171, 177, 184, 190, 196, // case 8 - 15
    // VTK_QUAD
    197, 204, 216, 228, 235, 247, 258, 265, // case 0 - 7
    271, 283, 290, 301, 307, 314, 320, 326, // case 8 - 15
    // VTK_TETRA
    327, 334, 343, 352, 361, 370, 379, 388, // case 0 - 7
    395, 404, 413, 422, 429, 438, 445, 452, // case 8 - 15
    // VTK_VOXEL
    453, 464, 527, 590, 609, 672, 691, 767, // case 0 - 7
    828, 891, 967, 986, 1047, 1066, 1127, 1188, // case 8 - 15
    1199, 1262, 1281, 1357, 1418, 1494, 1555, 1596, // case 16 - 23
    1631, 1672, 1744, 1816, 1877, 1949, 2010, 2063, // case 24 - 31
    2110, 2173, 2249, 2268, 2329, 2370, 2442, 2514, // case 32 - 39
    2575, 2651, 2692, 2753, 2788, 2860, 2913, 2974, // case 40 - 47
    3021, 3040, 3101, 3162, 3173, 3245, 3306, 3359, // case 48 - 55
    3406, 3478, 3531, 3592, 3639, 3656, 3671, 3686, // case 56 - 63
    3695, 3758, 3834, 3875, 3947, 3966, 4027, 4099, // case 64 - 71
    4160, 4236, 4277, 4349, 4402, 4463, 4498, 4559, // case 72 - 79
    4606, 4625, 4686, 4758, 4819, 4880, 4891, 4944, // case 80 - 87
    4991, 5063, 5116, 5133, 5148, 5209, 5256, 5271, // case 88 - 95
    5280, 5356, 5397, 5469, 5522, 5594, 5647, 5664, // case 96 - 103
    5679, 5720, 5745, 5798, 5817, 5870, 5889, 5904, // case 104 - 111
    5917, 5978, 6013, 6074, 6121, 6182, 6229, 6244, // case 112 - 119
    6253, 6306, 6325, 6340, 6353, 6368, 6381, 6394, // case 120 - 127
    6401, 6464, 6505, 6581, 6653, 6729, 6801, 6842, // case 128 - 135
    6895, 6914, 6986, 7047, 7108, 7169, 7230, 7265, // case 136 - 143
    7312, 7388, 7460, 7501, 7554, 7595, 7648, 7673, // case 144 - 151
    7692, 7764, 7781, 7834, 7849, 7902, 7917, 7936, // case 152 - 159
    7949, 7968, 8040, 8101, 8162, 8234, 8251, 8304, // case 160 - 167
    8319, 8380, 8433, 8444, 8491, 8552, 8567, 8614, // case 168 - 175
    8623, 8684, 8745, 8780, 8827, 8880, 8895, 8914, // case 176 - 183
    8927, 8988, 9003, 9050, 9059, 9074, 9087, 9100, // case 184 - 191
    9107, 9126, 9198, 9270, 9287, 9348, 9409, 9462, // case 192 - 199
    9477, 9538, 9591, 9652, 9667, 9678, 9725, 9772, // case 200 - 207
    9781, 9842, 9903, 9956, 9971, 10006, 10053, 10072, // case 208 - 215
    10085, 10146, 10161, 10176, 10189, 10236, 10245, 10258, // case 216 - 223
    10265, 10326, 10379, 10440, 10455, 10516, 10531, 10546, // case 224 - 231
    10559, 10594, 10613, 10660, 10673, 10720, 10733, 10742, // case 232 - 239
    10749, 10760, 10807, 10854, 10863, 10910, 10919, 10932, // case 240 - 247
    10939, 10986, 10999, 11008, 11015, 11024, 11031, 11038, // case 248 - 255
    // VTK_HEXAHEDRON
    11039, 11050, 11113, 11176, 11195, 11258, 11334, 11353, // case 0 - 7
    11414, 11477, 11496, 11572, 11633, 11652, 11713, 11774, // case 8 - 15
    11785, 11848, 11867, 11943, 12004, 12045, 12117, 12189, // case 16 - 23
    12250, 12326, 12387, 12428, 12463, 12535, 12596, 12649, // case 24 - 31
    12696, 12759, 12835, 12854, 12915, 12991, 13032, 13093, // case 32 - 39
    13128, 13169, 13241, 13313, 13374, 13446, 13499, 13560, // case 40 - 47
    13607, 13626, 13687, 13748, 13759, 13831, 13884, 13945, // case 48 - 55
    13992, 14064, 14125, 14178, 14225, 14242, 14257, 14272, // case 56 - 63
    14281, 14344, 14385, 14461, 14533, 14552, 14624, 14685, // case 64 - 71
    14746, 14822, 14894, 14935, 14988, 15049, 15110, 15145, // case 72 - 79
    15192, 15268, 15340, 15381, 15434, 15506, 15523, 15576, // case 80 - 87
    15591, 15632, 15685, 15710, 15729, 15782, 15797, 15816, // case 88 - 95
    15829, 15848, 15920, 15981, 16042, 16103, 16156, 16167, // case 96 - 103
    16214, 16286, 16303, 16356, 16371, 16432, 16447, 16494, // case 104 - 111
    16503, 16564, 16625, 16660, 16707, 16768, 16783, 16830, // case 112 - 119
    16839, 16892, 16907, 16926, 16939, 16954, 16967, 16980, // case 120 - 127
    16987, 17050, 17126, 17167, 17239, 17315, 17356, 17428, // case 128 - 135
    17481, 17500, 17561, 17633, 17694, 17755, 17790, 17851, // case 136 - 143
    17898, 17917, 17978, 18050, 18111, 18183, 18236, 18253, // case 144 - 151
    18268, 18329, 18340, 18393, 18440, 18501, 18548, 18563, // case 152 - 159
    18572, 18648, 18689, 18761, 18814, 18855, 18880, 18933, // case 160 - 167
    18952, 19024, 19077, 19094, 19109, 19162, 19181, 19196, // case 168 - 175
    19209, 19270, 19305, 19366, 19413, 19466, 19485, 19500, // case 176 - 183
    19513, 19574, 19621, 19636, 19645, 19660, 19673, 19686, // case 184 - 191
    19693, 19712, 19784, 19856, 19873, 19934, 19987, 20048, // case 192 - 199
    20063, 20124, 20185, 20238, 20253, 20264, 20311, 20358, // case 200 - 207
    20367, 20428, 20489, 20542, 20557, 20618, 20633, 20648, // case 208 - 215
    20661, 20696, 20743, 20762, 20775, 20822, 20831, 20844, // case 216 - 223
    20851, 20912, 20965, 21026, 21041, 21076, 21095, 21142, // case 224 - 231
    21155, 21216, 21231, 21246, 21259, 21306, 21319, 21328, // case 232 - 239
    21335, 21346, 21393, 21440, 21449, 21496, 21509, 21518, // case 240 - 247
    21525, 21572, 21581, 21594, 21601, 21610, 21617, 21624, // case 248 - 255
    // VTK_WEDGE
    21625, 21634, 21650, 21666, 21722, 21738, 21794, 21850, // case 0 - 7
    21859, 21875, 21886, 21947, 21994, 22055, 22102, 22117, // case 8 - 15
    22126, 22142, 22203, 22214, 22261, 22322, 22337, 22384, // case 16 - 23
    22393, 22449, 22496, 22543, 22552, 22567, 22580, 22593, // case 24 - 31
    22600, 22616, 22677, 22738, 22753, 22764, 22811, 22858, // case 32 - 39
    22867, 22923, 22970, 22985, 22998, 23045, 23054, 23067, // case 40 - 47
    23074, 23130, 23145, 23192, 23205, 23252, 23265, 23274, // case 48 - 55
    23281, 23290, 23299, 23308, 23315, 23324, 23331, 23338, // case 56 - 63
    // VTK_PYRAMID
    23339, 23347, 23362, 23377, 23427, 23442, 23459, 23509, // case 0 - 7
    23524, 23539, 23589, 23606, 23621, 23671, 23686, 23701, // case 8 - 15
    23709, 23720, 23737, 23754, 23763, 23780, 23793, 23802, // case 16 - 23
    23809, 23826, 23835, 23848, 23855, 23864, 23871, 23878, // case 24 - 31
  };

#if defined(VTK_COMPILER_GCC) && VTK_COMPILER_GCC_VERSION <= 40805
//XXX(gcc-4.8.5)
//GCC 4.8.5 has the following bugs
//internal compiler error: unexpected expression 'static_cast<int16_t>(-1)' of kind static_cast_expr
//internal compiler error: unexpected expression '(int16_t)((-1))' of kind cast_expr
#define VTK_CLIP_CAST(type, value) value
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wnarrowing"
#else
#define VTK_CLIP_CAST(type, value) static_cast<type>(value)
#endif

  // index into StartCellCases for each cell
  static constexpr int16_t CellCasesStartIndexLookUp[NUM_CELL_TYPES] = {
    VTK_CLIP_CAST(int16_t, -1),   // 0 = VTK_EMPTY_CELL
    VTK_CLIP_CAST(int16_t, 0),    // 1 = VTK_VERTEX
    VTK_CLIP_CAST(int16_t, -1),   // 2 = VTK_POLY_VERTEX
    VTK_CLIP_CAST(int16_t, 2),    // 3 = VTK_LINE
    VTK_CLIP_CAST(int16_t, -1),   // 4 = VTK_POLY_LINE
    VTK_CLIP_CAST(int16_t, 6),    // 5 = VTK_TRIANGLE
    VTK_CLIP_CAST(int16_t, -1),   // 6 = VTK_TRIANGLE_STRIP
    VTK_CLIP_CAST(int16_t, -1),   // 7 = VTK_POLYGON
    VTK_CLIP_CAST(int16_t, 14),   // 8 = VTK_PIXEL
    VTK_CLIP_CAST(int16_t, 30),   // 9 = VTK_QUAD
    VTK_CLIP_CAST(int16_t, 46),   // 10 = VTK_TETRA
    VTK_CLIP_CAST(int16_t, 62),   // 11 = VTK_VOXEL
    VTK_CLIP_CAST(int16_t, 318),  // 12 = VTK_HEXAHEDRON
    VTK_CLIP_CAST(int16_t, 574),  // 13 = VTK_WEDGE
    VTK_CLIP_CAST(int16_t, 638),  // 14 = VTK_PYRAMID
    VTK_CLIP_CAST(int16_t, -1),   // 15 = VTK_PENTAGONAL_PRISM
    VTK_CLIP_CAST(int16_t, -1),   // 16 = VTK_HEXAGONAL_PRISM
  };

#if defined(VTK_COMPILER_GCC) && VTK_COMPILER_GCC_VERSION <= 40805
#pragma GCC diagnostic pop
#endif

public:
  /**
   * Given a cell type return if it's supported by this class.
   */
  VTK_ALWAYS_INLINE static constexpr bool IsCellTypeSupported(int cellType)
  {
    return SupportedCellTypes[cellType];
  }

  ///@{
  /**
   * Given the number of points and a case index, return if the cell is kept.
   */
  template <bool InOut = TInsideOut>
  typename std::enable_if<InOut, bool>::type VTK_ALWAYS_INLINE static constexpr IsCellKept(
    vtkIdType vtkNotUsed(numberOfPoints), uint8_t caseIndex)
  {
    return caseIndex == 0;
  }
  template <bool InOut = TInsideOut>
  typename std::enable_if<!InOut, bool>::type VTK_ALWAYS_INLINE static constexpr IsCellKept(
    vtkIdType numberOfPoints, uint8_t caseIndex)
  {
    return caseIndex == CellMaxCase[numberOfPoints];
  }
  ///@}

  ///@{
  /**
   * Given the number of points and a case index, return if the cell is discarded.
   */
  template <bool InOut = TInsideOut>
  typename std::enable_if<InOut, bool>::type VTK_ALWAYS_INLINE static constexpr IsCellDiscarded(
    vtkIdType numberOfPoints, uint8_t caseIndex)
  {
    return caseIndex == CellMaxCase[numberOfPoints];
  }
  template <bool InOut = TInsideOut>
  typename std::enable_if<!InOut, bool>::type VTK_ALWAYS_INLINE static constexpr IsCellDiscarded(
    vtkIdType vtkNotUsed(numberOfPoints), uint8_t caseIndex)
  {
    return caseIndex == 0;
  }
  ///@}

  ///@{
  /**
   * Given a cell type and a case index, return the case in the form of
   * number of output cells,
   * 1. shape type, number of points, and point ids p0, p1, p2, ...
   * 2. shape type, number of points, and point ids p0, p1, p2, ...
   * ...
   */
  template <bool InOut = TInsideOut>
  typename std::enable_if<InOut, uint8_t*>::type VTK_ALWAYS_INLINE static GetCellCase(
    int cellType, uint8_t caseIndex)
  {
    return const_cast<uint8_t*>(
      &CellCasesInsideOut[StartCellCasesInsideOut[CellCasesStartIndexLookUp[cellType] +
        caseIndex]]);
  }
  template <bool InOut = TInsideOut>
  typename std::enable_if<!InOut, uint8_t*>::type VTK_ALWAYS_INLINE static GetCellCase(
    int cellType, uint8_t caseIndex)
  {
    return const_cast<uint8_t*>(
      &CellCases[StartCellCases[CellCasesStartIndexLookUp[cellType] + caseIndex]]);
  }
  ///@}

  /**
   * Given a cell type, return the edges of the cell.
   *
   * This is used in correspondence with GetShapeCase.
   */
  VTK_ALWAYS_INLINE static constexpr EDGEIDXS* GetCellEdges(int cellType)
  {
    return const_cast<EDGEIDXS*>(CellEdges[cellType]);
  }

  /**
   * Given a shape type, return the VTK cell type.
   */
  VTK_ALWAYS_INLINE static constexpr uint8_t GetCellType(uint8_t shapeType)
  {
    return shapeType & 0x07;
  }
};

VTK_ABI_NAMESPACE_END

#endif // vtkTableBasedClipCases_h
