#![allow(clippy::allow_attributes, reason = "clippy is weird with macros")]
#![allow(clippy::empty_structs_with_brackets, reason = "generated by macro")]

pub mod change;
pub mod input;

mod util_types;
use input::{SourceRoot, SourceRootId};
use line_index::LineIndex;
use syntax::Parse;
use triomphe::Arc;
pub use util_types::*;
pub use vfs::FileId;
use vfs::{AnchoredPath, VfsPath};

pub trait FileLoader {
    fn resolve_path(
        &self,
        path: AnchoredPath<'_>,
    ) -> Option<FileId>;
}

#[salsa::query_group(SourceDatabaseStorage)]
pub trait SourceDatabase: FileLoader {
    #[salsa::input]
    fn file_text(
        &self,
        file_id: FileId,
    ) -> Arc<String>;

    #[salsa::input]
    fn file_path(
        &self,
        file_id: FileId,
    ) -> VfsPath;

    #[salsa::input]
    fn file_id(
        &self,
        path: VfsPath,
    ) -> FileId;

    /// Path to a file, relative to the root of its source root.
    /// Source root of the file.
    #[salsa::input]
    fn file_source_root(
        &self,
        file_id: FileId,
    ) -> SourceRootId;
    /// Contents of the source root.
    #[salsa::input]
    fn source_root(
        &self,
        id: SourceRootId,
    ) -> Arc<SourceRoot>;

    #[salsa::invoke(parse_query)]
    fn parse(
        &self,
        key: FileId,
    ) -> Parse;

    fn line_index(
        &self,
        key: FileId,
    ) -> Arc<LineIndex>;
}

fn line_index(
    database: &dyn SourceDatabase,
    file_id: FileId,
) -> Arc<LineIndex> {
    let text = database.file_text(file_id);
    Arc::new(LineIndex::new(&text))
}

fn parse_query(
    database: &dyn SourceDatabase,
    file_id: FileId,
) -> Parse {
    let source = database.file_text(file_id);
    syntax::parse(&source)
}

/// Silly workaround for cyclic deps between the traits
pub struct FileLoaderDelegate<T>(pub T);

impl<T: SourceDatabase> FileLoader for FileLoaderDelegate<&'_ T> {
    fn resolve_path(
        &self,
        path: AnchoredPath<'_>,
    ) -> Option<FileId> {
        // FIXME: this *somehow* should be platform agnostic...
        let source_root = self.0.file_source_root(path.anchor);
        let source_root = self.0.source_root(source_root);
        source_root.resolve_path(path)
    }
}
