use std::{fmt, str::FromStr};

use crate::install::{ClientOptions, ServerOptions};

#[derive(Debug, Clone)]
pub enum PgoTrainingCrate {
    // Use WA's own sources for PGO training
    WgslAnalyzer,
    // Download a WGSL/WESL package from `https://github.com/{0}` and use it for PGO training.
    GitHub(String),
}

impl FromStr for PgoTrainingCrate {
    type Err = String;

    fn from_str(shell: &str) -> Result<Self, Self::Err> {
        match shell {
            "wgsl-analyzer" => Ok(Self::WgslAnalyzer),
            url => Ok(Self::GitHub(url.to_owned())),
        }
    }
}

xflags::xflags! {
    src "./src/flags.rs"

    /// Run custom build command.
    cmd xtask {
        /// Install wgsl-analyzer server or editor plugin.
        cmd install {
            /// Install only VS Code plugin.
            optional --client
            /// One of `code`, `code-exploration`, `code-insiders`, `codium`, or `code-oss`.
            optional --code-binary name: String

            /// Install only the language server.
            optional --server
            /// Use mimalloc allocator for server.
            optional --mimalloc
            /// Use jemalloc allocator for server.
            optional --jemalloc

            /// build in release with debug info set to 2.
            optional --dev-rel

            /// Apply PGO optimizations
            optional --pgo pgo: PgoTrainingCrate
        }

        cmd fuzz-tests {}

        cmd release {
            optional --dry-run
        }

        cmd dist {
            /// Use mimalloc allocator for server
            optional --mimalloc
            /// Use jemalloc allocator for server
            optional --jemalloc
            optional --client-patch-version version: String
            /// Use cargo-zigbuild
            optional --zig
            /// Apply PGO optimizations
            optional --pgo pgo: PgoTrainingCrate
        }

        /// Read a changelog AsciiDoc file and update the GitHub Releases entry in Markdown.
        cmd publish-release-notes {
            /// Only run conversion and show the result.
            optional --dry-run
            /// Target changelog file.
            required changelog: String
        }

        /// Builds a benchmark version of wgsl-analyzer and puts it into `./target`.
        cmd bb {
            required suffix: String
        }

        cmd codegen {
            optional rtype: CodegenType
            optional --check
        }

        cmd tidy {}
    }
}

// generated start
// The following code is generated by `xflags` macro.
// Run `env UPDATE_XFLAGS=1 cargo build` to regenerate.
#[derive(Debug)]
pub struct Xtask {
    pub subcommand: XtaskCmd,
}

#[derive(Debug)]
pub enum XtaskCmd {
    Install(Install),
    FuzzTests(FuzzTests),
    Release(Release),
    Dist(Dist),
    PublishReleaseNotes(PublishReleaseNotes),
    Bb(Bb),
    Codegen(Codegen),
    Tidy(Tidy),
}

#[derive(Debug)]
pub struct Install {
    pub client: bool,
    pub code_binary: Option<String>,
    pub server: bool,
    pub mimalloc: bool,
    pub jemalloc: bool,
    pub dev_rel: bool,
    pub pgo: Option<PgoTrainingCrate>,
}

#[derive(Debug)]
pub struct FuzzTests;

#[derive(Debug)]
pub struct Release {
    pub dry_run: bool,
}

#[derive(Debug)]
pub struct Dist {
    pub mimalloc: bool,
    pub jemalloc: bool,
    pub client_patch_version: Option<String>,
    pub zig: bool,
    pub pgo: Option<PgoTrainingCrate>,
}

#[derive(Debug)]
pub struct PublishReleaseNotes {
    pub changelog: String,

    pub dry_run: bool,
}

#[derive(Debug)]
pub struct Bb {
    pub suffix: String,
}

#[derive(Debug)]
pub struct Codegen {
    pub rtype: Option<CodegenType>,
    pub check: bool,
}

#[derive(Debug)]
pub struct Tidy;

impl Xtask {
    pub fn from_env_or_exit() -> Self {
        Self::from_env_or_exit_()
    }

    pub fn from_env() -> xflags::Result<Self> {
        Self::from_env_()
    }

    pub fn from_vec(arguments: Vec<std::ffi::OsString>) -> xflags::Result<Self> {
        Self::from_vec_(arguments)
    }
}
// generated end

#[derive(Clone, Copy, Debug, Default)]
pub enum CodegenType {
    #[default]
    All,
    AssistsDocTests,
    DiagnosticsDocs,
    LintDefinitions,
    ParserTests,
    FeatureDocs,
}

impl fmt::Display for CodegenType {
    fn fmt(
        &self,
        formatter: &mut fmt::Formatter<'_>,
    ) -> fmt::Result {
        match self {
            Self::All => write!(formatter, "all"),
            Self::AssistsDocTests => write!(formatter, "assists-doc-tests"),
            Self::DiagnosticsDocs => write!(formatter, "diagnostics-docs"),
            Self::LintDefinitions => write!(formatter, "lint-definitions"),
            Self::ParserTests => write!(formatter, "parser-tests"),
            Self::FeatureDocs => write!(formatter, "feature-docs"),
        }
    }
}

impl FromStr for CodegenType {
    type Err = String;

    fn from_str(string: &str) -> Result<Self, Self::Err> {
        match string {
            "all" => Ok(Self::All),
            "assists-doc-tests" => Ok(Self::AssistsDocTests),
            "diagnostics-docs" => Ok(Self::DiagnosticsDocs),
            "lint-definitions" => Ok(Self::LintDefinitions),
            "parser-tests" => Ok(Self::ParserTests),
            "feature-docs" => Ok(Self::FeatureDocs),
            _ => Err("Invalid option".to_owned()),
        }
    }
}

#[derive(Debug)]
pub enum MeasurementType {
    WeslRsTests,
    AnalyzeBevy,
}

impl FromStr for MeasurementType {
    type Err = String;

    fn from_str(string: &str) -> Result<Self, Self::Err> {
        match string {
            "wesl-rs_tests" => Ok(Self::WeslRsTests),
            "bevy-16.0.0" => Ok(Self::AnalyzeBevy),
            _ => Err("Invalid option".to_owned()),
        }
    }
}

impl AsRef<str> for MeasurementType {
    fn as_ref(&self) -> &str {
        match self {
            Self::WeslRsTests => "wesl-rs_tests",
            Self::AnalyzeBevy => "bevy-16.0.0",
        }
    }
}

#[derive(Clone, Copy, Debug)]
pub(crate) enum Malloc {
    System,
    Mimalloc,
    Jemalloc,
}

impl Malloc {
    pub(crate) fn to_features(self) -> &'static [&'static str] {
        match self {
            Self::System => &[][..],
            Self::Mimalloc => &["--features", "mimalloc"],
            Self::Jemalloc => &["--features", "jemalloc"],
        }
    }
}

impl Install {
    pub(crate) fn server(&self) -> Option<ServerOptions> {
        if self.client && !self.server {
            return None;
        }
        let malloc = if self.mimalloc {
            Malloc::Mimalloc
        } else if self.jemalloc {
            Malloc::Jemalloc
        } else {
            Malloc::System
        };
        Some(ServerOptions {
            malloc,
            dev_rel: self.dev_rel,
            pgo: self.pgo.clone(),
        })
    }

    pub(crate) fn client(&self) -> Option<ClientOptions> {
        if self.server && !self.client {
            return None;
        }
        Some(ClientOptions {
            code_binary: self.code_binary.clone(),
        })
    }
}

impl Dist {
    pub(crate) const fn allocator(&self) -> Malloc {
        if self.mimalloc {
            Malloc::Mimalloc
        } else if self.jemalloc {
            Malloc::Jemalloc
        } else {
            Malloc::System
        }
    }
}
