%define OT_ProcessSample_doc
"Collection of fields.

Available constructors:
    ProcessSample(*mesh, K, d*)

    ProcessSample(*mesh, coll*)

    ProcessSample(*K, field*)

Parameters
----------
mesh : :class:`~openturns.Mesh`
    The mesh shared by all the fields in the collection.
K : int
    Number of fields in the collection.
d : int
    Dimension  of the values of the field.
field : :class:`~openturns.Field`
    One field.
coll : sequence of :class:`~openturns.Sample`
    List of field values.

Notes
-----
A :class:`~openturns.ProcessSample` stores a sample of fields.
We note `K` the number of fields contained in the process sample and `d` the dimension of the values associated to each vertex of the common mesh :math:`\\cM \\in \\Rset^n`.

These fields can be generated by a stochastic process.

- In the first usage, we fix the common mesh with `mesh`, the number of fields contained in the sample with `K` and the dimension of the values with `d`. The values of the fields are by default fixed to zero.

- In the second usage, the  collection of fields is filled with `K` copies of the given field `field`.

We note :math:`\\vect{x}_i^k \\in \\Rset^d` the value of the field `k` at the vertex `i`. We note `N` the number of vertices of :math:`\\cM`, with :math:`0 \\leq i \\leq N-1` and :math:`1 \\leq k \\leq K`.

Examples
--------
Create a bi dimensional mesh as a box:

>>> import openturns as ot
>>> myIndices = [10, 5]
>>> myMesher = ot.IntervalMesher(myIndices)
>>> lowerBound = [0.0, 0.0]
>>> upperBound = [2.0, 1.0]
>>> myInterval = ot.Interval(lowerBound, upperBound)
>>> myMesh = myMesher.build(myInterval)

Create a second order normal porcess of dimension 3:

>>> amplitude = [5]
>>> scale = [3, 3]
>>> model = ot.ExponentialModel(scale, amplitude)
>>> myProcess = ot.GaussianProcess(model, myMesh)

Generate a sample of different fields:

>>> n = 10
>>> mySampleFields = myProcess.getSample(n)

Duplicate the same field:

>>> myField = myProcess.getRealization()
>>> n = 10
>>> mySampleFields2 = ot.ProcessSample(n, myField)
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation
OT_ProcessSample_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_add_doc
"Add a field to the collection.

Parameters
----------
field : :class:`~openturns.Field`
    A new field to add. 
    This field shares the same mesh and the same dimension as the other
    fields of the collection.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::add
OT_ProcessSample_add_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_getField_doc
"Field accessor.

Parameters
----------
i : int
    Field index

Returns
-------
field : :class:`~openturns.Field`
    i-th field in the sample.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::getField
OT_ProcessSample_getField_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_setField_doc
"Field accessor.

Parameters
----------
i : int
    Field index
field : :class:`~openturns.Field`
    i-th field in the sample.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::setField
OT_ProcessSample_setField_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeMean_doc
"Compute the mean field of the collection of fields.

Returns
-------
mean : :class:`~openturns.Field`
    The mean field has the same dimension `d` and the same mesh as the fields
    contained in the collection. At each vertex of the mesh, we compute
    the mean of the values at this vertex of the `K` fields contained
    in the process sample:

    .. math::

        \\forall i \\in [0,N-1], \\quad \\dfrac{1}{K} \\sum_{k=1}^K \\vect{x}_i^k
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeMean
OT_ProcessSample_computeMean_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeSpatialMean_doc
"Compute the spatial mean of the values of the fields.

Returns
-------
spatialMean : :class:`~openturns.Sample`
    Its size is the number `K` of fields in the collection.
    Its dimension is `d`. The `k` numerical point is the spatial mean of the field `k`:

    .. math::

        \\forall k \\in [1,K], \\quad \\dfrac{1}{N}\\sum_{i=0}^{N-1} \\vect{x}_i^k
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeSpatialMean
OT_ProcessSample_computeSpatialMean_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeTemporalMean_doc
"Compute the temporal mean of the values of the fields.

Returns
-------
spatialMean : :class:`~openturns.Sample`
    Its size is the number `K` of fields in the collection.
    Its dimension is `d`.
    The `k` numerical point is the temporal mean of the field `k`:

    .. math::

        \\forall k \\in [1,K], \\quad \\dfrac{1}{N}\\sum_{i=0}^{N-1} \\vect{x}_i^k

This method can be used only when the mesh can be interpreted as a regular grid.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeTemporalMean
OT_ProcessSample_computeTemporalMean_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeStandardDeviation_doc
"Compute the standard deviation field of the collection of fields.

Returns
-------
stddev : :class:`~openturns.Field`
    The standard deviation field has the same dimension `d` and the same mesh
    as the fields contained in the collection.
    At each vertex of the mesh, we compute the standard deviation of the
    values at this vertex of the `K` fields contained in the process sample:

    .. math::

        \\forall i \\in [0,N-1], \\quad \\sqrt{\\dfrac{1}{K-1} \\sum_{k=1}^K (\\vect{x}_i^k - \\hat{\\mu}_i)^2}"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeStandardDeviation
OT_ProcessSample_computeStandardDeviation_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeQuantilePerComponent_doc
"Compute the quantile field of the collection of fields.

Parameters
----------
p : float, :math:`0 \\leq p \\leq 1`, or sequence of float
    Order of the quantile.

Returns
-------
quantile : :class:`~openturns.Field` or :class:`~openturns.ProcessSample`
    This field has the same size and the same dimension as the fields
    of the collection. At each vertex of the mesh, we compute the
    component-wise empirical quantile of order `p`.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeQuantilePerComponent
OT_ProcessSample_computeQuantilePerComponent_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeRange_doc
"Compute the range field of the collection of fields.

Returns
-------
range : :class:`~openturns.Field`
    This field has the same size and the same dimension as the fields
    of the collection. At each vertex of the mesh, we compute the
    component-wise range, i.e. the difference between the largest and
    the lowest value in the field collection.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeRange
OT_ProcessSample_computeRange_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeMedian_doc
"Compute the median field of the collection of fields.

Returns
-------
median : :class:`~openturns.Field`
    This field has the same size and the same dimension as the fields
    of the collection. At each vertex of the mesh, we compute the
    component-wise empirical median. It is the same as
    `computeQuantilePerComponent(0.5)`.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeMedian
OT_ProcessSample_computeMedian_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeVariance_doc
"Compute the variance field of the collection of fields.

Returns
-------
variance : :class:`~openturns.Field`
    The variance field has the same dimension `d` and the same mesh
    as the fields contained in the collection.
    At each vertex of the mesh, we compute the variance of the
    values at this vertex of the `K` fields contained in the process sample.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeVariance
OT_ProcessSample_computeVariance_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeSkewness_doc
"Compute the skewness field of the collection of fields.

Returns
-------
skewness : :class:`~openturns.Field`
    The squewness field has the same dimension `d` and the same mesh
    as the fields contained in the collection.
    At each vertex of the mesh, we compute the skewness of the
    values at this vertex of the `K` fields contained in the process sample.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeSkewness
OT_ProcessSample_computeSkewness_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeKurtosis_doc
"Compute the kurtosis field of the collection of fields.

Returns
-------
kurtosis : :class:`~openturns.Field`
    The squewness field has the same dimension `d` and the same mesh
    as the fields contained in the collection.
    At each vertex of the mesh, we compute the kurtosis of the
    values at this vertex of the `K` fields contained in the process sample.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeKurtosis
OT_ProcessSample_computeKurtosis_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_getMin_doc
"Compute the min field of the collection of fields.

Returns
-------
min : :class:`~openturns.Field`
    This field has the same size and the same dimension as the fields
    of the collection. At each vertex of the mesh, we compute the
    component-wise minimum.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::getMin
OT_ProcessSample_getMin_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_getMax_doc
"Compute the max field of the collection of fields.

Returns
-------
max : :class:`~openturns.Field`
    This field has the same size and the same dimension as the fields
    of the collection. At each vertex of the mesh, we compute the
    component-wise maximum.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::getMax
OT_ProcessSample_getMax_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeCentralMoment_doc
"Compute the central moment field of the collection of fields.

Parameters
----------
k : int
    Order of the moment.

Returns
-------
moment : :class:`~openturns.Field`
    This field has the same size and the same dimension as the fields
    of the collection. At each vertex of the mesh, we compute the
    component-wise central moment of order `n`.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeCentralMoment
OT_ProcessSample_computeCentralMoment_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeRawMoment_doc
"Compute the raw moment field of the collection of fields.

Parameters
----------
k : int
    Order of the moment.

Returns
-------
moment : :class:`~openturns.Field`
    This field has the same size and the same dimension as the fields
    of the collection. At each vertex of the mesh, we compute the
    component-wise raw moment of order `n`.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeRawMoment
OT_ProcessSample_computeRawMoment_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_computeEmpiricalCDF_doc
"Compute the empirical CDF field of the collection of fields.

Parameters
----------
point : sequence of float
    The point at which all the marginal CDF are computed. Its dimension
    must match the process sample dimension.
tail : bool (optional, default=False)
    Flag to if it is the CDF or the complementary CDF which is computed.

Returns
-------
cdf : :class:`~openturns.Field`
    This field is scalar and has the same size as the process sample.
    At each vertex of the mesh, we compute the component-wise empirical
    CDF at the given point.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::computeEmpiricalCDF
OT_ProcessSample_computeEmpiricalCDF_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_getSize_doc
"Get the size of the collection of fields.

Returns
-------
K : int
    Number of fields in the collection.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::getSize
OT_ProcessSample_getSize_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_getMesh_doc
"Get the mesh of the fields.

Returns
-------
mesh : :class:`~openturns.Mesh`
    The mesh shared by all the fields of the collection.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::getMesh
OT_ProcessSample_getMesh_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_getTimeGrid_doc
"Get the time grid of the fields.

Returns
-------
mesh : :class:`~openturns.RegularGrid`
    The time grid shared by all the fields of the collection.
    Can be used only if the mesh can be interpreted as a regular time grid.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::getTimeGrid
OT_ProcessSample_getTimeGrid_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_getDimension_doc
"Get the dimension of the values of fields.

Returns
-------
d : int
    Dimension of the values of the fields.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::getDimension
OT_ProcessSample_getDimension_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_getMarginal_doc
"Marginal sample accessor.

Parameters
----------
indice : int or sequence of int
    Index of the marginals.

Returns
-------
sample : :class:`~openturns.ProcessSample`
    The marginal sample.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::getMarginal
OT_ProcessSample_getMarginal_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_drawMarginal_doc
"Draw the selected field.

Parameters
----------
indice : int
    Index of the field that is drawn in the graph.
interpolate : bool (optional, default=True)
    Whether to draw interpolated values.

Returns
-------
graph : :class:`~openturns.Graph`
    The graph of the selected field using the `interpolate` method.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::drawMarginal
OT_ProcessSample_drawMarginal_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_draw_doc
"Draw the whole field.

Parameters
----------
interpolate : bool (optional, default=True)
    Whether to draw interpolated values.

Returns
-------
graph : :class:`~openturns.GridLayout`
    The graph of the whole field using the `interpolate` method.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::draw
OT_ProcessSample_draw_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_drawMarginalCorrelation_doc
"Draw correlation between 2 marginals.

This is only available on a 1-d mesh.

Parameters
----------
i, j : int, :math:`i, j < d`
    Marginal indices.

Returns
-------
graph : :class:`~openturns.Graph`
    The graph of the correlation between 2 marginals.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::drawMarginalCorrelation
OT_ProcessSample_drawMarginalCorrelation_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_drawCorrelation_doc
"Draw correlation between all marginals.

This is only available on a 1-d mesh.

Returns
-------
graph : :class:`~openturns.GridLayout`
    The graph of the correlation between all marginals.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::drawCorrelation
OT_ProcessSample_drawCorrelation_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_getSampleAtVertex_doc
"Extract sample at specific vertex of the mesh.

Parameters
----------
index : int, :math:`index < N`
    Node index in the mesh.

Returns
-------
sample_at_i : :class:`Sample`
    The sample of all values of the trajectories at the desired node.
"
%enddef
%feature("docstring") OT::ProcessSampleImplementation::getSampleAtVertex
OT_ProcessSample_getSampleAtVertex_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_erase_doc
"Erase point(s) at or between index(es) (in-place).

Parameters
----------
f : int, :math:`0 \\leq f < m`
    The index of the first point to erase.
l : int, :math:`f < l \\leq m`, optional
    The index after the last point to erase.
    Default uses `l = f + 1` and only removes `sample[f]`."
%enddef
%feature("docstring") OT::ProcessSampleImplementation::erase
OT_ProcessSample_erase_doc

// ---------------------------------------------------------------------

%define OT_ProcessSample_clear_doc
"Erase all field values."
%enddef
%feature("docstring") OT::ProcessSampleImplementation::clear
OT_ProcessSample_clear_doc
