%define OT_EnumerateFunction_doc
"Enumerate function.

Notes
-----
*EnumerateFunction* is a bijection :math:`\\vect{\\alpha}`  from :math:`\\Nset` to
:math:`\\Nset^{n_X}`. This bijection is based on a particular enumeration rule
of the set of multi-indices.
For any integer :math:`i \\in \\Nset`, the value of the enumerate function is:

.. math::

    \\vect{\\alpha}(i) = (\\alpha_1(i),\\dots, \\alpha_{n_X}(i)) \\in {\\Nset}^{n_X}.

The first multi-index is:

.. math::

    \\vect{\\alpha}(0) = (0, 0, \\dots, 0).

Let :math:`i, j \\in \\Nset` be any pair of indices. If :math:`|i - j|\\leq 1` then:

.. math::

    \\left|\\sum_{k=1}^{n_X} \\left[ \\alpha_k(i) - \\alpha_k(j) \\right]\\right| \\leq 1

where :math:`\\alpha_k(i)` is the k-th component of the multi-index :math:`\\vect{\\alpha}(i)`.
This provides a necessary but insufficient condition 
for the construction of the bijection: 
a supplementary hypothesis must be made. 

For example, consider the dimension :math:`\\textrm{dim}=2`. 
The following mapping is an enumerate function:

.. math::

    \\vect{\\alpha}(0) &= (0, \\; 0) \\\\
    \\vect{\\alpha}(1) &= (1, \\; 0) \\\\
    \\vect{\\alpha}(2) &= (0, \\; 1) \\\\
    \\vect{\\alpha}(3) &= (2, \\; 0) \\\\
    \\vect{\\alpha}(4) &= (1, \\; 1) \\\\
    \\vect{\\alpha}(5) &= (0, \\; 2) \\\\
    \\vect{\\alpha}(6) &= (3, \\; 0)

For the functional expansion (respectively polynomial chaos expansion), the
multi-index :math:`\\vect{\\alpha}` is the collection of marginal degrees of the
selected orthogonal functions (respectively orthogonal polynomials). 
More precisely, after the selection of the type of orthogonal functions (respectively
orthogonal polynomials) for the construction of the orthogonal basis, the
*EnumerateFunction* defines the
marginal degrees of the univariate functions (respectively univariate polynomials).
The multivariate tensor product function
is based on the product of marginal univariate functions:

.. math::

    \\psi_{\\vect{\\alpha}}(\\vect{x}) = \\prod_{i = 1}^{n_X} \\pi_{\\alpha_i}^{(i)}(x_i)

for any point :math:`\\vect{x}` where :math:`\\pi_{\\alpha_i}^{(i)}` is the
polynomial of degree :math:`\\alpha_i` for the i-th marginal.
The total degree of the polynomial is equal to the sum of the integers 
in the multi-index:

.. math::

    \\operatorname{degree} \\left(\\psi_{\\vect{\\alpha}} \\right) 
    = \\sum_{i = 1}^{n_X} \\alpha_i = \\|\\vect{\\alpha}\\|_1.

For example, consider the multi-index :math:`\\vect{\\alpha} = (2, 0, 3)`. 
This multi-index defines a multivariate polynomial which has a input
vector of dimension 3.
The marginal degree of the polynomial of the first dimension is equal to 2, 
the marginal degree of the polynomial of the second dimension is equal to 0 
and the marginal degree of the polynomial of the third dimension is equal to 3.

Examples
--------
>>> import openturns as ot
>>> dim = 2
>>> enum_func = ot.LinearEnumerateFunction(dim)
>>> for i in range(6):
...     print(enum_func(i))
[0,0]
[1,0]
[0,1]
[2,0]
[1,1]
[0,2]

See Also
--------
LinearEnumerateFunction, HyperbolicAnisotropicEnumerateFunction, NormInfEnumerateFunction"
%enddef
%feature("docstring") OT::EnumerateFunctionImplementation
OT_EnumerateFunction_doc

// ---------------------------------------------------------------------

%define OT_EnumerateFunction_getDimension_doc
"Return the dimension of the *EnumerateFunction*.

Returns
-------
dim : int, :math:`dim \\geq 0`
    Dimension of the *EnumerateFunction*."
%enddef
%feature("docstring") OT::EnumerateFunctionImplementation::getDimension
OT_EnumerateFunction_getDimension_doc

// ---------------------------------------------------------------------

%define OT_EnumerateFunction_getMaximumDegreeCardinal_doc
"Get the number of multi-indices of total degree lower or equal to a threshold.

Parameters
----------
max_deg : int
    Maximum total degree.

Returns
-------
cardinal : int
    Number of multi-indices in the basis of total degree :math:`\\leq \\max_{deg}`.

Notes
-----
In the specific context of a linear enumeration (:class:`LinearEnumerateFunction`)
this is also the cumulated cardinal of stratas of index :math:`\\leq \\max_{deg}`.

Examples
--------
>>> import openturns as ot
>>> dim = 2
>>> enum_func = ot.LinearEnumerateFunction(dim)
>>> enum_func.getMaximumDegreeCardinal(2)
6"
%enddef
%feature("docstring") OT::EnumerateFunctionImplementation::getMaximumDegreeCardinal
OT_EnumerateFunction_getMaximumDegreeCardinal_doc

// ---------------------------------------------------------------------

%define OT_EnumerateFunction_getMaximumDegreeStrataIndex_doc
"Get the largest index of the strata containing multi-indices lower or equal to the given maximum degree.

Parameters
----------
max_deg : int
    Maximum total degree.

Returns
-------
index : int
    Index of the last strata that contains multi-indices of total degree :math:`\\leq \\max_{deg}`.

Notes
-----
In the specific context of a linear enumeration (:class:`LinearEnumerateFunction`)
this is the strata of index *max_deg*.

Examples
--------
>>> import openturns as ot
>>> dim = 2
>>> enum_func = ot.LinearEnumerateFunction(dim)
>>> enum_func.getMaximumDegreeStrataIndex(2)
2"
%enddef
%feature("docstring") OT::EnumerateFunctionImplementation::getMaximumDegreeStrataIndex
OT_EnumerateFunction_getMaximumDegreeStrataIndex_doc

// ---------------------------------------------------------------------

%define OT_EnumerateFunction_getStrataCardinal_doc
"Get the number of multi-indices in the basis inside a given strata.

Parameters
----------
strataIndex : int
    Index of the strata of the tensorized basis.

Returns
-------
cardinal : int
    Number of multi-indices in the basis inside the strata *strataIndex*.

Notes
-----
In the specific context of a linear enumeration (:class:`LinearEnumerateFunction`)
the strata *strataIndex* consists of a hyperplane of all the multi-indices of total degree
*strataIndex*, and its cardinal is *strataIndex* + 1.

Examples
--------
>>> import openturns as ot
>>> dim = 2
>>> enum_func = ot.LinearEnumerateFunction(dim)
>>> enum_func.getStrataCardinal(2)
3"
%enddef
%feature("docstring") OT::EnumerateFunctionImplementation::getStrataCardinal
OT_EnumerateFunction_getStrataCardinal_doc

// ---------------------------------------------------------------------

%define OT_EnumerateFunction_getStrataCumulatedCardinal_doc
"Get the number of multi-indices in the basis inside a range of stratas.

Parameters
----------
strataIndex : int
    Index of the strata of the tensorized basis.

Returns
-------
cardinal : int
    Number of multi-indices in the basis inside the stratas of index lower or equal to *strataIndex*.

Notes
-----
The number of multi-indices is the total of multi-indices inside the stratas.
In the specific context of a linear enumeration (:class:`LinearEnumerateFunction`)
this returns the number of multi-indices of maximal total degree *strataIndex*.

Examples
--------
>>> import openturns as ot
>>> dim = 2
>>> enum_func = ot.LinearEnumerateFunction(dim)
>>> enum_func.getStrataCumulatedCardinal(2)
6
>>> sum([enum_func.getStrataCardinal(i) for i in range(3)])
6"

%enddef
%feature("docstring") OT::EnumerateFunctionImplementation::getStrataCumulatedCardinal
OT_EnumerateFunction_getStrataCumulatedCardinal_doc

// ---------------------------------------------------------------------

%define OT_EnumerateFunction_inverse_doc
"Get the antecedent of a indices list in the *EnumerateFunction*.

Parameters
----------
multiIndex : sequence of int
    List of indices.

Returns
-------
antecedent : int
    Represents the antecedent of the *multiIndex* in the *EnumerateFunction*.

Examples
--------
>>> import openturns as ot
>>> dim = 2
>>> enum_func = ot.LinearEnumerateFunction(dim)
>>> for i in range(6):
...     print(str(i)+' '+str(enum_func(i)))
0 [0,0]
1 [1,0]
2 [0,1]
3 [2,0]
4 [1,1]
5 [0,2]
>>> print(enum_func.inverse([1,1]))
4"
%enddef
%feature("docstring") OT::EnumerateFunctionImplementation::inverse
OT_EnumerateFunction_inverse_doc

// ---------------------------------------------------------------------

%define OT_EnumerateFunction_setDimension_doc
"Set the dimension of the *EnumerateFunction*.

Parameters
----------
dim : int, :math:`dim \\geq 0`
    Dimension of the *EnumerateFunction*."
%enddef
%feature("docstring") OT::EnumerateFunctionImplementation::setDimension
OT_EnumerateFunction_setDimension_doc

// ---------------------------------------------------------------------

%define OT_EnumerateFunction_getBasisSizeFromTotalDegree_doc
"Get the basis size corresponding to a total degree.

Parameters
----------
max_deg : int
    Maximum total degree.

Returns
-------
size : int
    Number of multi-indices in the basis of total degree :math:`\\leq \\max_{deg}`.

Notes
-----
In the specific context of a linear enumeration (:class:`LinearEnumerateFunction`)
this is also the cumulated cardinal of stratas up to *max_deg*.
    
Examples
--------
>>> import openturns as ot
>>> dim = 2
>>> enum_func = ot.LinearEnumerateFunction(dim)
>>> enum_func.getBasisSizeFromTotalDegree(3)
10
>>> enum_func.getStrataCumulatedCardinal(3)
10"
%enddef
%feature("docstring") OT::EnumerateFunctionImplementation::getBasisSizeFromTotalDegree
OT_EnumerateFunction_getBasisSizeFromTotalDegree_doc


// ---------------------------------------------------------------------

%define OT_EnumerateFunction_setUpperBound_doc
"Accessor to the upper bound.

Parameters
----------
ub : sequence of int
    Upper bound of the indices (inclusive)."
%enddef
%feature("docstring") OT::EnumerateFunctionImplementation::setUpperBound
OT_EnumerateFunction_setUpperBound_doc

// ---------------------------------------------------------------------

%define OT_EnumerateFunction_getUpperBound_doc
"Accessor to the upper bound.

Returns
-------
ub : sequence of int
    Upper bound of the indices (inclusive)."
%enddef
%feature("docstring") OT::EnumerateFunctionImplementation::getUpperBound
OT_EnumerateFunction_getUpperBound_doc

// ---------------------------------------------------------------------

%define OT_EnumerateFunction_getMarginal_doc
"Get the marginal enumerate function.

Parameters
----------
indices : int or sequence of int, :math:`0 \\leq i < n`
    List of marginal indices.

Returns
-------
enumerateFunction : :class:`~openturns.EnumerateFunction`
    The marginal enumerate function."
%enddef
%feature("docstring") OT::EnumerateFunctionImplementation::getMarginal
OT_EnumerateFunction_getMarginal_doc
