/**
 * ServiceType.java
 *
 * (C) Copyright IBM Corp. 2005, 2007
 *
 * THIS FILE IS PROVIDED UNDER THE TERMS OF THE COMMON PUBLIC LICENSE 
 * ("AGREEMENT"). ANY USE, REPRODUCTION OR DISTRIBUTION OF THIS FILE 
 * CONSTITUTES RECIPIENTS ACCEPTANCE OF THE AGREEMENT.
 *
 * You can obtain a current copy of the Common Public License from
 * http://www.opensource.org/licenses/cpl1.0.php
 *
 * @author: Roberto Pineiro, IBM, roberto.pineiro@us.ibm.com  
 * @author: Chung-hao Tan, IBM ,chungtan@us.ibm.com
 * 
 * Change History
 * Flag       Date        Prog         Description
 *------------------------------------------------------------------------------- 
 * 1516246    2006-07-22  lupusalex    Integrate SLP client code
 * 1804402    2007-09-28  ebak         IPv6 ready SLP
 */


package org.sblim.slp;


import java.util.Locale;
import java.util.Vector;


/**
 * The Advertiser is the SA interface, allowing clients to register new service
 * instances with SLP, to change the attributes of existing services, and to
 * deregister service instances. New registrations and modifications of
 * attributes are made in the language locale with which the Advertiser was
 * created, deregistrations of service instances are made for all locales.
 */
public interface Advertiser {

	/**
	 * Return the language locale with which this object was created.
	 * 
	 * @return The locale
	 */
	public abstract Locale getLocale();

	/**
	 * Register a new service with SLP having the given attributes. The API
	 * library is required to perform the operation in all scopes obtained
	 * through configuration.
	 * 
	 * @param pURL
	 *            The URL for the service.
	 * @param pAttributes
	 *            A vector of ServiceLocationAttribute objects describing the
	 *            service.
	 * @throws ServiceLocationException
	 */
	public abstract void register(ServiceURL pURL, Vector pAttributes)
			throws ServiceLocationException;

	/**
	 * Deregister a service from the SLP framework. This has the effect of
	 * deregistering the service from every language locale. The API library is
	 * required to perform the operation in all scopes obtained through
	 * configuration.
	 * 
	 * @param pURL
	 *            The URL for the service.
	 * @throws ServiceLocationException
	 */
	public abstract void deregister(ServiceURL pURL) throws ServiceLocationException;

	/**
	 * Update the registration by adding the given attributes. The API library
	 * is required to perform the operation in all scopes obtained through
	 * configuration.
	 * 
	 * @param pURL
	 *            The URL for the service.
	 * @param pAttributes
	 *            A Vector of ServiceLocationAttribute objects to add to the
	 *            existing registration. Use an empty vector to update the URL
	 *            alone. May not be null.
	 * @throws ServiceLocationException
	 */
	public abstract void addAttributes(ServiceURL pURL, Vector pAttributes)
			throws ServiceLocationException;

	/**
	 * Delete the attributes from a URL for the locale with which the Advertiser
	 * was created. The API library is required to perform the operation in all
	 * scopes obtained through configuration.
	 * 
	 * @param pURL
	 *            The URL for the service.
	 * @param pAttributeIds
	 *            A vector of Strings indicating the ids of the attributes to
	 *            remove. The strings may be attribute ids or they may be
	 *            wildcard patterns to match ids. See [7] for the syntax of
	 *            wildcard patterns. The strings may include SLP reserved
	 *            characters, they will be escaped by the API before
	 *            transmission. May not be the empty vector or null.
	 * @throws ServiceLocationException
	 */
	public abstract void deleteAttributes(ServiceURL pURL, Vector pAttributeIds)
			throws ServiceLocationException;
}
