/*
 * Copyright (C) 2021-2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include "ArgumentCoders.h"
#include "Connection.h"
#include "MessageNames.h"
#include <WebCore/NotificationResources.h>
#include <optional>
#include <wtf/Forward.h>
#include <wtf/RefCounted.h>
#include <wtf/ThreadSafeRefCounted.h>
#include <wtf/UUID.h>
#include <wtf/Vector.h>

namespace WTF {
class UUID;
}

namespace WebCore {
class SecurityOriginData;
enum class PushPermissionState : uint8_t;
struct NotificationData;
}

namespace Messages {
namespace NotificationManagerMessageHandler {

static inline IPC::ReceiverName messageReceiverName()
{
    return IPC::ReceiverName::NotificationManagerMessageHandler;
}

class ShowNotification {
public:
    using Arguments = std::tuple<WebCore::NotificationData, RefPtr<WebCore::NotificationResources>>;

    static IPC::MessageName name() { return IPC::MessageName::NotificationManagerMessageHandler_ShowNotification; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NotificationManagerMessageHandler_ShowNotificationReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    using Promise = WTF::NativePromise<void, IPC::Error>;
    ShowNotification(const WebCore::NotificationData& notificationData, const RefPtr<WebCore::NotificationResources>& notificationResources)
        : m_arguments(notificationData, notificationResources)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::NotificationData&, const RefPtr<WebCore::NotificationResources>&> m_arguments;
};

class CancelNotification {
public:
    using Arguments = std::tuple<WTF::UUID>;

    static IPC::MessageName name() { return IPC::MessageName::NotificationManagerMessageHandler_CancelNotification; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit CancelNotification(const WTF::UUID& notificationID)
        : m_arguments(notificationID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WTF::UUID&> m_arguments;
};

class ClearNotifications {
public:
    using Arguments = std::tuple<Vector<WTF::UUID>>;

    static IPC::MessageName name() { return IPC::MessageName::NotificationManagerMessageHandler_ClearNotifications; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit ClearNotifications(const Vector<WTF::UUID>& notificationIDs)
        : m_arguments(notificationIDs)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const Vector<WTF::UUID>&> m_arguments;
};

class DidDestroyNotification {
public:
    using Arguments = std::tuple<WTF::UUID>;

    static IPC::MessageName name() { return IPC::MessageName::NotificationManagerMessageHandler_DidDestroyNotification; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    explicit DidDestroyNotification(const WTF::UUID& notificationID)
        : m_arguments(notificationID)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WTF::UUID&> m_arguments;
};

class PageWasNotifiedOfNotificationPermission {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::NotificationManagerMessageHandler_PageWasNotifiedOfNotificationPermission; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<> m_arguments;
};

class RequestPermission {
public:
    using Arguments = std::tuple<WebCore::SecurityOriginData>;

    static IPC::MessageName name() { return IPC::MessageName::NotificationManagerMessageHandler_RequestPermission; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NotificationManagerMessageHandler_RequestPermissionReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    using Promise = WTF::NativePromise<bool, IPC::Error>;
    explicit RequestPermission(const WebCore::SecurityOriginData& origin)
        : m_arguments(origin)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::SecurityOriginData&> m_arguments;
};

class SetAppBadge {
public:
    using Arguments = std::tuple<WebCore::SecurityOriginData, std::optional<uint64_t>>;

    static IPC::MessageName name() { return IPC::MessageName::NotificationManagerMessageHandler_SetAppBadge; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    SetAppBadge(const WebCore::SecurityOriginData& origin, const std::optional<uint64_t>& badge)
        : m_arguments(origin, badge)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::SecurityOriginData&, const std::optional<uint64_t>&> m_arguments;
};

class GetPermissionState {
public:
    using Arguments = std::tuple<WebCore::SecurityOriginData>;

    static IPC::MessageName name() { return IPC::MessageName::NotificationManagerMessageHandler_GetPermissionState; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::NotificationManagerMessageHandler_GetPermissionStateReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<WebCore::PushPermissionState>;
    using Reply = CompletionHandler<void(WebCore::PushPermissionState)>;
    using Promise = WTF::NativePromise<WebCore::PushPermissionState, IPC::Error>;
    explicit GetPermissionState(const WebCore::SecurityOriginData& origin)
        : m_arguments(origin)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::SecurityOriginData&> m_arguments;
};

class GetPermissionStateSync {
public:
    using Arguments = std::tuple<WebCore::SecurityOriginData>;

    static IPC::MessageName name() { return IPC::MessageName::NotificationManagerMessageHandler_GetPermissionStateSync; }
    static constexpr bool isSync = true;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;

    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<WebCore::PushPermissionState>;
    using Reply = CompletionHandler<void(WebCore::PushPermissionState)>;
    explicit GetPermissionStateSync(const WebCore::SecurityOriginData& origin)
        : m_arguments(origin)
    {
    }

    auto&& arguments()
    {
        return WTFMove(m_arguments);
    }

private:
    std::tuple<const WebCore::SecurityOriginData&> m_arguments;
};

} // namespace NotificationManagerMessageHandler
} // namespace Messages
