/* vim: set sw=8: -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * gog-gradient.c :
 *
 * Copyright (C) 2003 Jody Goldberg (jody@gnome.org)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gnumeric-config.h>
#include <goffice/utils/go-gradient.h>
#include <goffice/utils/go-color.h>

#include <src/gnumeric-i18n.h>
#include <widgets/widget-pixmap-combo.h>
#include <gdk-pixbuf/gdk-pixdata.h>
#include <string.h>


static struct {
	GOGradientDirection dir;
	char const *name;
} grad_dir_names[] = {
	{ GO_GRADIENT_N_TO_S,            "n-s" },
	{ GO_GRADIENT_S_TO_N,            "s-n" },
	{ GO_GRADIENT_N_TO_S_MIRRORED,   "n-s-mirrored" },
	{ GO_GRADIENT_S_TO_N_MIRRORED,   "s-n-mirrored" },
	{ GO_GRADIENT_W_TO_E,            "w-e" },
	{ GO_GRADIENT_E_TO_W,            "e-w" },
	{ GO_GRADIENT_W_TO_E_MIRRORED,   "w-e-mirrored" },
	{ GO_GRADIENT_E_TO_W_MIRRORED,   "e-w-mirrored" },
	{ GO_GRADIENT_NW_TO_SE,          "nw-se" },
	{ GO_GRADIENT_SE_TO_NW,          "se-nw" },
	{ GO_GRADIENT_NW_TO_SE_MIRRORED, "nw-se-mirrored" },
	{ GO_GRADIENT_SE_TO_NW_MIRRORED, "se-nw-mirrored" },
	{ GO_GRADIENT_NE_TO_SW,          "ne-sw" },
	{ GO_GRADIENT_SW_TO_NE,          "sw-ne" },
	{ GO_GRADIENT_SW_TO_NE_MIRRORED, "sw-ne-mirrored" },
	{ GO_GRADIENT_NE_TO_SW_MIRRORED, "ne-sw-mirrored" },
};

GOGradientDirection
go_gradient_dir_from_str (char const *name)
{
	unsigned i;
	GOGradientDirection ret = GO_GRADIENT_N_TO_S;

	for (i = 0; 
	     i < sizeof grad_dir_names / sizeof grad_dir_names[0]; i++) {
		if (strcmp (grad_dir_names[i].name, name) == 0) {
			ret = grad_dir_names[i].dir;
			break;
		}
	}
	return ret;
}

char const *
go_gradient_dir_as_str (GOGradientDirection dir)
{
	unsigned i;
	char const *ret = "pattern";

	for (i = 0; i < sizeof grad_dir_names / sizeof grad_dir_names[0]; i++) {
		if (grad_dir_names[i].dir == dir) {
			ret = grad_dir_names[i].name;
			break;
		}
	}
	return ret;
}

GtkWidget *
go_gradient_selector (GOColor start, GOColor end)
{
#warning Add names to these after release
	static PixmapComboElement elements[] = {
		{ NULL, NULL, GO_GRADIENT_N_TO_S },
		{ NULL, NULL, GO_GRADIENT_S_TO_N },
		{ NULL, NULL, GO_GRADIENT_N_TO_S_MIRRORED },
		{ NULL, NULL, GO_GRADIENT_S_TO_N_MIRRORED },
		{ NULL, NULL, GO_GRADIENT_W_TO_E },
		{ NULL, NULL, GO_GRADIENT_E_TO_W },
		{ NULL, NULL, GO_GRADIENT_W_TO_E_MIRRORED },
		{ NULL, NULL, GO_GRADIENT_E_TO_W_MIRRORED },
		{ NULL, NULL, GO_GRADIENT_NW_TO_SE },
		{ NULL, NULL, GO_GRADIENT_SE_TO_NW },
		{ NULL, NULL, GO_GRADIENT_NW_TO_SE_MIRRORED },
		{ NULL, NULL, GO_GRADIENT_SE_TO_NW_MIRRORED },
		{ NULL, NULL, GO_GRADIENT_NE_TO_SW },
		{ NULL, NULL, GO_GRADIENT_SW_TO_NE },
		{ NULL, NULL, GO_GRADIENT_SW_TO_NE_MIRRORED },
		{ NULL, NULL, GO_GRADIENT_NE_TO_SW_MIRRORED }	
	};
	guint i, length;
	GdkPixdata  pixdata;
	GtkWidget  *w;
	gpointer    data;
	ArtRender *render;
	ArtGradientLinear gradient;
	ArtGradientStop stops[2];
	int const sizex = 20, sizey = 20;
	GdkPixbuf *pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, sizex, sizey);
	int rowstride = gdk_pixbuf_get_rowstride (pixbuf);
	void *pixels = gdk_pixbuf_get_pixels (pixbuf);
	int n_channels = gdk_pixbuf_get_n_channels (pixbuf);

	for (i = 0; i < G_N_ELEMENTS (elements); i++) {
		memset (pixels, 0, rowstride * sizey);
		render = art_render_new (0, 0, sizex, sizey,
					 pixels,
					 rowstride,
					 n_channels - 1,
					 8, ART_ALPHA_SEPARATE, NULL);
		if (elements[i].inline_gdkpixbuf != NULL)
			g_free ((gpointer) elements[i].inline_gdkpixbuf);

		go_gradient_setup (&gradient,
				   i, start, end,
				   0, 0, sizex, sizey,
				   stops);

		art_render_gradient_linear (render, &gradient,
					    ART_FILTER_NEAREST);
		art_render_invoke (render);

		data = gdk_pixdata_from_pixbuf (&pixdata, pixbuf, FALSE);
		elements[i].inline_gdkpixbuf = gdk_pixdata_serialize (&pixdata, &length);
		g_free (data);
	}
	g_object_unref (pixbuf);
	w = pixmap_combo_new (elements, 4, 4, FALSE);
	gtk_combo_box_set_tearable (GTK_COMBO_BOX (w), FALSE);
	return w;
}

void
go_gradient_setup (ArtGradientLinear *gradient,
		   GOGradientDirection dir, GOColor col0, GOColor col1,
		   double x0, double y0, double x1, double y1,
		   ArtGradientStop *stops)
{
	double dx = x1 - x0;
	double dy = y1 - y0;

	if (dir < 4) {
		gradient->a = 0.;
		gradient->b = 1. / (dy ? dy : 1);
		gradient->c = - 1.e-10 - (gradient->a * x0 + gradient->b * y0);
	} else if (dir < 8) {
		gradient->a = 1. / (dx ? dx : 1);
		gradient->b = 0.;
		gradient->c = -(gradient->a * x0 + gradient->b * y0);
	} else if (dir < 12) {
		double d = dx * dx + dy * dy;
		if (!d) d = 1;
		gradient->a = dx / d;
		gradient->b = dy / d;
		gradient->c = -(gradient->a * x0 + gradient->b * y0);
	} else {
		double d = dx * dx + dy * dy;
		if (!d) d = 1;
		gradient->a = -dx / d;
		gradient->b = dy / d;
		/* Note: this gradient is anchored at (x1,y0).  */
		gradient->c = -(gradient->a * x1 + gradient->b * y0);
	}

	gradient->stops = stops;
	gradient->n_stops = 2;
	stops[0].offset = 0;
	stops[1].offset = 1;

	switch (dir % 4) {
	case 0:
		gradient->spread = ART_GRADIENT_PAD;
		go_color_to_artpix (stops[0].color, col0);
		go_color_to_artpix (stops[1].color, col1);
		break;
	case 1:
		gradient->spread = ART_GRADIENT_PAD;
		go_color_to_artpix (stops[0].color, col1);
		go_color_to_artpix (stops[1].color, col0);
		break;
	case 2:
		gradient->spread = ART_GRADIENT_REFLECT;
		go_color_to_artpix (stops[0].color, col0);
		go_color_to_artpix (stops[1].color, col1);
		gradient->a *= 2;
		gradient->b *= 2;
		gradient->c *= 2;
		break;
	case 3:
		gradient->spread = ART_GRADIENT_REFLECT;
		go_color_to_artpix (stops[0].color, col1);
		go_color_to_artpix (stops[1].color, col0);
		gradient->a *= 2;
		gradient->b *= 2;
		gradient->c *= 2;
		break;
	}
}
