/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.common.medias;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import org.videolan.vlma.common.programs.IVlProgram;
import org.videolan.vlma.common.VlServer;

/**
 * This class represents a channel where files are diffused.
 * 
 * @author MagSoft
 */
public class VlFilesChannel extends VlMedia {

    public static final String SAP_GROUP = "Files";

    private static final Logger logger = Logger.getLogger(VlFilesChannel.class);

    private String fullname;
    
    /**
     *  The list of files of this channel
     */
    private List<String> files;
    
    /**
     * The server which has to diffuse this channel
     */
    private VlServer server;

    /**
     * The constructor of the class
     */
    public VlFilesChannel() {
        files = new ArrayList<String>();
    }
    
    /**
     * Sets the media program.
     * 
     * @param program the program to set
     */
    public void setProgram(IVlProgram program)
    {
        super.setProgram(program);
        updateSapGroup();
    }
    
    /**
     * Updates the SAP group.
     * 
     */
    protected void updateSapGroup() {
        if (program != null)
            program.setSapGroup(SAP_GROUP);
    }


    /**
     * Gives the full name of the channel (used in the SAP announce).
     * 
     * @return the channel name
     */
    public String getFullname() {
        return fullname;
    }

    /**
     * Sets the channel fullname.
     * 
     * @param fullname
     *            the name to set
     */
    public void setFullname(String fullname) {
        this.fullname = fullname;
    }
    
    public List<String> getFiles() {
        return this.files;
    }
    
    public void setFiles(List<String> filesList) {
        this.files = filesList;
    }

    public VlServer getServer() {
        return this.server;
    }
    
    public void setServer(VlServer server_given) {
        this.server = server_given;
    }
    
    
    /**
     * Compares two VLFilesChannels.
     * 
     * @return true if objects are the same, false otherwise
     */
    public boolean equals(Object o) {
        if (o.getClass().equals(VlFilesChannel.class)) {
            VlFilesChannel c = (VlFilesChannel) o;
            logger.log(Level.DEBUG, "Comparaison de " + getName() + " et "
                    + c.getName());
            return (c.name.equals(this.name));
        }
        return false;
    }

    public boolean sameGroup(IVlMedia media) {
        if (media.getClass() == VlFilesChannel.class) {
            VlFilesChannel c = (VlFilesChannel) media;
            /* Two fileChannels belong to the same group if they are
             * must be diffused by the same server. */
            return c.getServer().equals(this.getServer());
        }
        return false;
    }

    public boolean belongsToGroup(VlMediaGroup group) {
        if (!group.medias.isEmpty()) {
            return sameGroup(group.medias.get(0));
        }
        return false;
    }

    /**
     * Gives the channel hashcode.
     * 
     * @return hashCode the channel hashcode
     */
    public int hashCode() {
        return name.hashCode();
    }

    public int compareTo(IVlMedia other) {
        if (other.getClass().equals(VlFilesChannel.class)) {
            VlFilesChannel channel = (VlFilesChannel) other;
            return name.compareTo(channel.getName());
        }
        return -1;
    }
    
    public int getId() {
        return hashCode();
    }

}
