
/****************************************************************************
 *  (c) Copyright 2007 Wi-Fi Alliance.  All Rights Reserved
 *
 *
 *  LICENSE
 *
 *  License is granted only to Wi-Fi Alliance members and designated
 *  contractors ($B!H(BAuthorized Licensees$B!I(B)..AN  Authorized Licensees are granted
 *  the non-exclusive, worldwide, limited right to use, copy, import, export
 *  and distribute this software:
 *  (i) solely for noncommercial applications and solely for testing Wi-Fi
 *  equipment; and
 *  (ii) solely for the purpose of embedding the software into Authorized
 *  Licensee$B!G(Bs proprietary equipment and software products for distribution to
 *  its customers under a license with at least the same restrictions as
 *  contained in this License, including, without limitation, the disclaimer of
 *  warranty and limitation of liability, below..AN  The distribution rights
 *  granted in clause
 *  (ii), above, include distribution to third party companies who will
 *  redistribute the Authorized Licensee$B!G(Bs product to their customers with or
 *  without such third party$B!G(Bs private label. Other than expressly granted
 *  herein, this License is not transferable or sublicensable, and it does not
 *  extend to and may not be used with non-Wi-Fi applications..AN  Wi-Fi Alliance
 *  reserves all rights not expressly granted herein..AN 
 *.AN 
 *  Except as specifically set forth above, commercial derivative works of
 *  this software or applications that use the Wi-Fi scripts generated by this
 *  software are NOT AUTHORIZED without specific prior written permission from
 *  Wi-Fi Alliance.
 *.AN 
 *  Non-Commercial derivative works of this software for internal use are
 *  authorized and are limited by the same restrictions; provided, however,
 *  that the Authorized Licensee shall provide Wi-Fi Alliance with a copy of
 *  such derivative works under a perpetual, payment-free license to use,
 *  modify, and distribute such derivative works for purposes of testing Wi-Fi
 *  equipment.
 *.AN 
 *  Neither the name of the author nor "Wi-Fi Alliance" may be used to endorse
 *  or promote products that are derived from or that use this software without
 *  specific prior written permission from Wi-Fi Alliance.
 *
 *  THIS SOFTWARE IS PROVIDED BY WI-FI ALLIANCE "AS IS" AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS FOR A.AN PARTICULAR PURPOSE,
 *  ARE DISCLAIMED. IN NO EVENT SHALL WI-FI ALLIANCE BE LIABLE FOR ANY DIRECT,
 *  INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, THE COST OF PROCUREMENT OF SUBSTITUTE
 *  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 *  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE) ARISING IN ANY WAY OUT OF
 *  THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. ******************************************************************************
 */
/*
 *    File: wfa_tg.c 
 *    Library functions for traffic generator.
 *    They are shared with both TC and DUT agent.
 * 
 *    Revision History:
 *       2006/03/10 -- initially created by qhu 
 *       2006/06/01 -- BETA release by qhu 
 *       2006/06/13 -- 00.02 release by qhu
 *       2006/06/30 -- 00.10 Release by qhu
 *       2006/07/10 -- 01.00 Release by qhu
 *       2006/08/01 -- 01.03 fix ping result unreachable caused by random output
 *                           timing. Add 2 seconds wait-time. 
 *                           change priority level to 40 max while traffic
 *                           sending/receiving and 60 min for normal state.
 *       2006/09/01 -- 01.05 Release by qhu
 *       2006/10/26 -- 01.06 released by qhu
 *       2006/10.20 -- bug fixes report by n. ojannen
 *       2006/12/30 -- bug fix. a potential race condition for transaction test, *                     reported by c.benson.
 *       2006/01/11 -- 01.10 released by qhu
 *       2007/02/15  -- WMM Extension Beta released by qhu, mkaroshi
 *       2007/03/15  -- develop a better rate control method for sendLong
 *                      a major bug fix for File receive that opens socket with
 *                      source port. It should be destination port. This was
 *                      an early mistake because of interpretation.
 *       2007/03/30 -- 01.40 WPA2 and Official WMM Beta Release by qhu
 *       2007/04/20 -- 02.00 WPA2 and Official WMM Release by qhu
 *       2007/08/15 --  02.10 WMM-Power Save release by qhu
 *       2007/10/10 -- 02.20 Voice SOHO beta -- qhu
 *       2007/11/07 -- 02.30 Voice HSO -- qhu
 *           -- revise the code for traffic throttled control
 *           -- take care of EAGAIN error return by socket sendto() -- suggested
 *              by tterhaar
 *       2007/12/10 -- 02.32 Further modify the throttled mechanism for traffic
 *                     test.
 *       2008/01/03 -- 02.34 Support the result upload command
 *       2008/02/07 -- 02.40 Support Updated WMM-PS method
 *                     Comment out the process priority settings, user could
 *                     decide to use them
 *       2008/03/14 -- 02.41 Bug #12, reset a socket before closing it, change 
 *                           to close the socket id first then reset it to -1 
 *                           in function tmout_stop_send()
 *
 *                           Bug #14. Not A Bug, impove the rate control by
 *                           change 50 ms period to 20 ms period and lower
 *                           max rate from 3000 to 2500 in function 
 *                           wfaTxSleepTime().
 */
#include <sys/time.h>
#include <time.h>
#include <signal.h>
#include <fcntl.h>
#include <netdb.h>
#include <pthread.h>
#include <math.h>

#include "wfa_portall.h"
#include "wfa_stdincs.h"
#include "wfa_debug.h"
#include "wfa_ver.h"
#include "wfa_main.h"
#include "wfa_tlv.h"
#include "wfa_tg.h"
#include "wfa_cmds.h"
#include "wfa_sock.h"
#include "wfa_rsp.h"
#include "wfa_wmmps.h"
#include "wfa_miscs.h"

extern tgStream_t gStreams[];
extern BOOL gtgRecv;
extern BOOL gtgSend;
extern BOOL gtgTransac;
extern int gtimeOut;
extern int gRegSec;
extern BOOL gtgCaliRTD;

int btSockfd = -1;
int adj_latency;

extern tgStream_t *findStreamProfile(int);
extern int wfaTrafficSendTo(int, char *, int, struct sockaddr *);
extern int wfaTrafficRecv(int, char *, struct sockaddr *);
extern void wfaSendPing(tgPingStart_t *staPing, float *interval, int streamid);
extern int wfaStopPing(dutCmdResponse_t *stpResp, int streamid);
extern unsigned short wfa_defined_debug;
extern int tgSockfds[];

extern tgWMM_t wmm_thr[];

extern double min_rttime;
extern double gtgPktRTDelay;
extern void int2BuffBigEndian(int val, char *buf);
extern int bigEndianBuff2Int(char *buff);

#ifdef WFA_WMM_PS_EXT
extern int gtgWmmPS;
extern wfaWmmPS_t wmmps_info;
extern int psSockfd;
extern unsigned int psTxMsg[];
extern unsigned int psRxMsg[];
extern int gtgPsPktRecvd;

extern void wfaSetDUTPwrMgmt(int mode);
void wmmps_wait_state_proc();

#endif



static int streamId = 0;
static int totalTranPkts = 0, sentTranPkts = 0;
int slotCnt = 0;

extern int usedThread;
extern int runLoop;
extern int sendThrId;

char e2eResults[124];
#if 0  /* for test purpose only */
char *e2eResults = "/tmp/e2e1198798626.txt";
#endif


extern dutCmdResponse_t gGenericResp;

/* Some devices may only support UDP ECHO and do not have ICMP level ping */
// #define WFA_PING_UDP_ECHO_ONLY     1

/*
 * findStreamProfile(): search existing stream profile by stream id
 * input: id - stream id;
 * return: matched stream profile
 */
tgStream_t *findStreamProfile(int id)
{
    int i;
    tgStream_t *myStream = gStreams;

    for(i = 0; i< WFA_MAX_TRAFFIC_STREAMS; i++)
    {
       if(myStream->id == id)
          return myStream;
       
       myStream++;
    }

    return NULL;
}

/*
 * wfaTGSendPing(): Instruct Traffic Generator to send ping packets
 *
 */
int wfaTGSendPing(int len, BYTE *caCmdBuf, int *respLen, BYTE *respBuf)
{
    int streamid = ++streamId;
    float interval;      /* it could be subseconds/100s minisecond */
    tgPingStart_t *staPing = (tgPingStart_t *)caCmdBuf;
    dutCmdResponse_t *spresp = &gGenericResp;

#ifdef WFA_PING_UDP_ECHO_ONLY
    tgStream_t *myStream = NULL;
#endif

    DPRINT_INFO(WFA_OUT, "Entering wfaTSendPing ...\n");
    if(staPing->frameSize == 0)
         staPing->frameSize = 100;

    printf("fr %i\n", staPing->frameRate);
    if(staPing->frameRate == 0)
         staPing->frameRate = 1;

    interval = (float) 1/staPing->frameRate;
    printf("TG: interval %f\n", interval);

    if(staPing->duration == 0)
         staPing->duration = 30;

    switch(staPing->type)
    {
    case WFA_PING_ICMP_ECHO:
#ifndef WFA_PING_UDP_ECHO_ONLY
        printf("The steam ID is:%d \n",streamId);

        if(cmd_traffic_send_ping(sigma_dut_ptr(), staPing, spresp) < 0){        
               spresp->status = STATUS_ERROR;  
               *respLen = 0;
                return FALSE;
        }
        //wfaSendPing(staPing, &interval, streamId);

        spresp->status = STATUS_COMPLETE;
        spresp->streamId = streamid;
#else
        printf("Only support UDP ECHO\n");
#endif
        break;

    case WFA_PING_UDP_ECHO:
    {
#ifdef WFA_PING_UDP_ECHO_ONLY
        /*
         * Make this like a transaction testing
         * Then make it a profile and run it
         */
        myStream = &gStreams[slotCnt++];
        memset(myStream, 0, sizeof(tgStream_t));
        memcpy(&myStream->profile, caCmdBuf, len);
        myStream->id = streamid; /* the id start from 1 */
        myStream->tblidx = slotCnt-1;

        btSockfd = wfaCreateUDPSock("127.0.0.1", WFA_UDP_ECHO_PORT);
        if((btSockfd = wfaConnectUDPPeer(btSockfd, staPing->dipaddr, WFA_UDP_ECHO_PORT)) > 0)
        {
            gtgTransac = streamid;
            gtgSend = streamid;
            totalTranPkts = 512;
            sentTranPkts = 0;

            /*
             * the framerate here is used to derive the timeout
             * value for waiting transaction echo responses.
             */
            gtimeOut = MINISECONDS/staPing->frameRate;  /* in msec */

            /* set to longest time */
            if(staPing->duration == 0)
                staPing->duration = 3600;
        }
#else
        printf("Doesn't support UDP Echo\n");
#endif
        break;
    }
    default:
    {
        spresp->status = STATUS_INVALID;
        spresp->streamId = streamid;
    }
    }


    wfaEncodeTLV(WFA_TRAFFIC_SEND_PING_RESP_TLV, sizeof(dutCmdResponse_t), (BYTE *)spresp, respBuf);
    *respLen = WFA_TLV_HDR_LEN + sizeof(dutCmdResponse_t);

    return TRUE;
}

/*
 * tgStopPing(): Instruct Traffic Generator to stop ping packets
 *
 */
int wfaTGStopPing(int len, BYTE *caCmdBuf, int *respLen, BYTE *respBuf)
{
    int streamid = *(int *)(caCmdBuf);
    dutCmdResponse_t *stpResp = &gGenericResp;
    tgStream_t *myStream;
    int i;

    stpResp->status = STATUS_COMPLETE;

    printf("CS: The length %d\n and the command buff is \n",len);

    for (i=0;i<8;i++)
       printf(" %x ",caCmdBuf[i]);

    printf("\nthe stream id is %d",streamid);

    if( gtgTransac == streamid&&gtgSend == streamid)
    {
        gtgTransac =0;
        gtgSend = 0;
//        gtimeOut = 0;
        gtgRecv = 0;
        alarm(0);

        myStream = findStreamProfile(streamid);
        if(myStream == NULL)
        {
            stpResp->status = STATUS_INVALID;
        }

        stpResp->cmdru.pingStp.sendCnt = myStream->stats.txFrames;
        stpResp->cmdru.pingStp.repliedCnt = myStream->stats.rxFrames;
    }
    else
    {
#if 0
        if(wfaStopPing(stpResp, streamid)== FALSE)
        {
            stpResp->status = STATUS_COMPLETE;

            wfaEncodeTLV(WFA_TRAFFIC_STOP_PING_RESP_TLV, sizeof(dutCmdResponse_t), (BYTE *)stpResp, respBuf);
            *respLen = WFA_TLV_HDR_LEN + sizeof(dutCmdResponse_t);
        }
#endif

        if(cmd_traffic_stop_ping(sigma_dut_ptr(), streamid, stpResp) < 0){        
                       stpResp->status = STATUS_ERROR;  
                       *respLen = 0;
                        return FALSE;
        }        
        //wfaStopPing(stpResp, streamid);
     }

     wfaEncodeTLV(WFA_TRAFFIC_STOP_PING_RESP_TLV, sizeof(dutCmdResponse_t), (BYTE *)stpResp, respBuf);
     *respLen = WFA_TLV_HDR_LEN + sizeof(dutCmdResponse_t);

    return TRUE;
}

/*
 * wfaTGConfig: store the traffic profile setting that will be used to
 *           instruct traffic generation.
 * input: cmd -- not used
 * response: send success back to controller
 * return: success or fail
 * Note: the profile storage is a global space.
 */
int wfaTGConfig(int len, BYTE *caCmdBuf, int *respLen, BYTE *respBuf)
{
    int ret = FALSE;
    tgStream_t *myStream = NULL;
    dutCmdResponse_t *confResp = &gGenericResp;
    
    /* if the stream table over maximum, reset it */ 
    if(slotCnt == WFA_MAX_TRAFFIC_STREAMS)
       slotCnt = 0;

    if(slotCnt == 0)
    {
       printf("resetting stream table\n");
       wMEMSET(gStreams, 0, WFA_MAX_TRAFFIC_STREAMS*sizeof(tgStream_t));
    }
   
    DPRINT_INFO(WFA_OUT, "entering tcConfig ...\n");
    myStream = &gStreams[slotCnt++];
    wMEMSET(myStream, 0, sizeof(tgStream_t));
    wMEMCPY(&myStream->profile, caCmdBuf, len);
    myStream->id = ++streamId; /* the id start from 1 */ 
    myStream->tblidx = slotCnt-1;

#if 0
    DPRINT_INFO(WFA_OUT, "profile %i direction %i dest ip %s dport %i source %s sport %i rate %i duration %i size %i class %i delay %i\n", myStream->profile.profile, myStream->profile.direction, myStream->profile.dipaddr, myStream->profile.dport, myStream->profile.sipaddr, myStream->profile.sport, myStream->profile.rate, myStream->profile.duration, myStream->profile.pksize, myStream->profile.trafficClass, myStream->profile.startdelay);
#endif
 
    confResp->status = STATUS_COMPLETE; 
    confResp->streamId = myStream->id;
    wfaEncodeTLV(WFA_TRAFFIC_AGENT_CONFIG_RESP_TLV, sizeof(dutCmdResponse_t), (BYTE *)confResp, respBuf);
    *respLen = WFA_TLV_HDR_LEN + sizeof(dutCmdResponse_t); 


    return ret;
}

/* RecvStart: instruct traffic generator to start receiving 
 *                 based on a profile
 * input:      cmd -- not used
 * response:   inform controller for "running"
 * return:     success or failed 
 */
int wfaTGRecvStart(int len, BYTE *parms, int *respLen, BYTE *respBuf)
{
    int status = STATUS_COMPLETE, i;
    int numStreams = len/4;
    int streamid;
    tgProfile_t *theProfile;
    tgStream_t *myStream;

    DPRINT_INFO(WFA_OUT, "entering tgRecvStart\n");

    /*
     * The function wfaSetProcPriority called here is to enhance the real-time
     * performance for packet receiving. It is only for tuning and optional
     * to implement
     */
    //wfaSetProcPriority(60);

    for(i=0; i<numStreams; i++)
    {
        wMEMCPY(&streamid, parms+(4*i), 4); /* changed from 2 to 4, bug reported by n.ojanen */
        myStream = findStreamProfile(streamid); 
        if(myStream == NULL)
        {
            status = STATUS_INVALID;
            return status;    
        }

        theProfile = &myStream->profile;
        if(theProfile == NULL)
        {
           status = STATUS_INVALID;
           return status;
        }

        /* calculate the frame interval which is used to derive its jitter */
        if(theProfile->rate != 0 && theProfile->rate < 5000)
           myStream->fmInterval = 1000000/theProfile->rate; /* in ms */
        else
           myStream->fmInterval = 0;

        if(theProfile->direction != DIRECT_RECV)
        {
           status = STATUS_INVALID;
           return status;
        }

        wMEMSET(&myStream->stats, 0, sizeof(tgStats_t));

        // mark the stream active
        myStream->state = WFA_STREAM_ACTIVE;
       
        switch(theProfile->profile)
        {
#ifdef WFA_WPA2_SINGLE_THREAD  
           case PROF_MCAST:
           case PROF_FILE_TX:
//           case PROF_IPTV:
               btSockfd = wfaCreateUDPSock(theProfile->dipaddr, theProfile->dport);
               gtgRecv = streamid;

               if(btSockfd < 0)
                  status = STATUS_ERROR;
               else
               {
                  /* get current flags setting */
                  int ioflags = wFCNTL(btSockfd, F_GETFL, 0);

                  /* set only BLOCKING flag to non-blocking */
                  wFCNTL(btSockfd, F_SETFL, ioflags | O_NONBLOCK);
               }
           break;
#else

           case PROF_TRANSC:
	   case PROF_CALI_RTD:  /* Calibrate roundtrip delay */
              gtgTransac = streamid;
           case PROF_MCAST:
           case PROF_FILE_TX:
           case PROF_IPTV:
              gtgRecv = streamid;
              wmm_thr[usedThread].thr_flag = streamid;
              wPT_MUTEX_LOCK(&wmm_thr[usedThread].thr_flag_mutex);
              wPT_COND_SIGNAL(&wmm_thr[usedThread].thr_flag_cond);
              wPT_MUTEX_UNLOCK(&wmm_thr[usedThread].thr_flag_mutex);
              printf("Recv Start in thread %i for streamid %i\n", usedThread, streamid);
	      usedThread++;
           break;
#endif       
           case PROF_UAPSD:
#ifdef WFA_WMM_PS_EXT
           status = STATUS_COMPLETE;
           psSockfd = wfaCreateUDPSock(theProfile->dipaddr, WFA_WMMPS_UDP_PORT);

           wmmps_info.sta_state = 0;
           wmmps_info.wait_state = WFA_WAIT_STAUT_00;

           wMEMSET(&wmmps_info.psToAddr, 0, sizeof(wmmps_info.psToAddr));
           wmmps_info.psToAddr.sin_family = AF_INET;
           wmmps_info.psToAddr.sin_addr.s_addr = inet_addr(theProfile->sipaddr);
           wmmps_info.psToAddr.sin_port = htons(theProfile->sport);
	   wmmps_info.reset = 0;

           wmm_thr[usedThread].thr_flag = streamid;
           wmmps_info.streamid = streamid;
           wPT_MUTEX_LOCK(&wmm_thr[usedThread].thr_flag_mutex);
           wPT_COND_SIGNAL(&wmm_thr[usedThread].thr_flag_cond);
           gtgWmmPS = streamid;;
           wPT_MUTEX_UNLOCK(&wmm_thr[usedThread].thr_flag_mutex);
           usedThread++;
#endif   /* WFA_WMM_PS_EXT */
           break;
       }
    }

    /* encode a TLV for response for "complete/error ..." */
    wfaEncodeTLV(WFA_TRAFFIC_AGENT_RECV_START_RESP_TLV, sizeof(int), 
                 (BYTE *)&status, respBuf);
    *respLen = WFA_TLV_HDR_LEN + sizeof(int); 

    return TRUE;
}

/*
 * tgRecvStop: instruct traffic generator to stop receiving based on a profile
 * input:      cmd -- not used
 * response:   inform controller for "complete"
 * return:     success or failed 
 */
int wfaTGRecvStop(int len, BYTE *parms, int *respLen, BYTE *respBuf)
{
    int status = STATUS_COMPLETE, i;
    int numStreams = len/4;
    unsigned int streamid;
    tgProfile_t *theProfile;
    tgStream_t *myStream=NULL;
    dutCmdResponse_t statResp;
    BYTE dutRspBuf[WFA_RESP_BUF_SZ];
    int id_cnt = 0;

    DPRINT_INFO(WFA_OUT, "entering tgRecvStop with length %d\n",len);

    /* in case that send-stream not done yet, an optional delay */
    while(sendThrId != 0)
      sleep(1);

    /*
     * After finishing the receiving command, it should lower itself back to 
     * normal level. It is optional implementation if it is not called 
     * while it starts receiving for raising priority level.
     */
    //wfaSetProcPriority(30);
    wMEMSET(dutRspBuf, 0, WFA_RESP_BUF_SZ);
    for(i=0; i<numStreams; i++)
    {
        wMEMCPY(&streamid, parms+(4*i), 4);
        printf(" stop stream id %i\n", streamid);
        myStream = findStreamProfile(streamid); 
        if(myStream == NULL)
        {
            status = STATUS_INVALID;
            wfaEncodeTLV(WFA_TRAFFIC_AGENT_RECV_STOP_RESP_TLV, 4, (BYTE *)&status, respBuf);
            *respLen = WFA_TLV_HDR_LEN + 4;
            printf("stream table empty\n");
            continue;    
        }

        theProfile = &myStream->profile;
        if(theProfile == NULL)
        {
           status = STATUS_INVALID;
           wfaEncodeTLV(WFA_TRAFFIC_AGENT_RECV_STOP_RESP_TLV, 4, (BYTE *)&status, respBuf);
           *respLen = WFA_TLV_HDR_LEN + 4;

           return TRUE;
        }

        if(theProfile->direction != DIRECT_RECV)
        {
           status = STATUS_INVALID;
           wfaEncodeTLV(WFA_TRAFFIC_AGENT_RECV_STOP_RESP_TLV, 4, (BYTE *)&status, respBuf);
           *respLen = WFA_TLV_HDR_LEN + 4;

           return TRUE;
        }

        /* reset its flags , close sockets */
        switch(theProfile->profile)
        {
         case PROF_TRANSC:
	 case PROF_CALI_RTD:
	     gtgTransac = 0;
         case PROF_MCAST:
         case PROF_FILE_TX:
         case PROF_IPTV:
             gtgRecv = 0;
             if(tgSockfds[myStream->tblidx] != -1)
             {
                wCLOSE(tgSockfds[myStream->tblidx]);
                tgSockfds[myStream->tblidx] = -1;
             }
         break;

	 case PROF_UAPSD:
#ifdef WFA_WMM_PS_EXT
         gtgWmmPS = 0;
         gtgPsPktRecvd = 0;

         if(psSockfd != -1)
         {
             wCLOSE(psSockfd);
             psSockfd = -1;
         }

         wMEMSET(&wmmps_info, 0, sizeof(wfaWmmPS_t));

         wfaSetDUTPwrMgmt(PS_OFF);
#endif /* WFA_WMM_PS_EXT */
         break;

         }
 
    	/* encode a TLV for response for "complete/error ..." */
    	statResp.status = STATUS_COMPLETE; 
    	statResp.streamId = streamid; 

#if 1
    	DPRINT_INFO(WFA_OUT, "stream Id %u rx %u total %llu\n", streamid, myStream->stats.rxFrames, myStream->stats.rxPayloadBytes);
#endif
    	wMEMCPY(&statResp.cmdru.stats, &myStream->stats, sizeof(tgStats_t));
     	wMEMCPY((dutRspBuf + i * sizeof(dutCmdResponse_t)), (BYTE *)&statResp, sizeof(dutCmdResponse_t));
	id_cnt++;

        // Not empty it but require to reset the entire table before test starts.
        //wMEMSET(myStream, 0, sizeof(tgStream_t));
    }

    // mark the stream inactive
    myStream->state = WFA_STREAM_INACTIVE;

    printf("Sending back the statistics at recvstop\n");
    wfaEncodeTLV(WFA_TRAFFIC_AGENT_RECV_STOP_RESP_TLV, id_cnt * sizeof(dutCmdResponse_t), dutRspBuf, respBuf);

    /* done here */
    *respLen = WFA_TLV_HDR_LEN + numStreams * sizeof(dutCmdResponse_t); 

    return TRUE;
}

/*
 * wfaTGSendStart: instruct traffic generator to start sending based on a profile
 * input:      cmd -- not used
 * response:   inform controller for "running"
 * return:     success or failed 
 */
int wfaTGSendStart(int len, BYTE *parms, int *respLen, BYTE *respBuf)
{
    int i=0, streamid=0;
    int numStreams = len/4;

    tgProfile_t *theProfile;
    tgStream_t *myStream = NULL;

    dutCmdResponse_t staSendResp;

    DPRINT_INFO(WFA_OUT, "Entering tgSendStart for %i streams ...\n", numStreams);
    for(i=0; i<numStreams; i++)
    {
        wMEMCPY(&streamid, parms+(4*i), 4);
        myStream = findStreamProfile(streamid); 
        if(myStream == NULL)
        {
           staSendResp.status = STATUS_INVALID;
           wfaEncodeTLV(WFA_TRAFFIC_AGENT_SEND_RESP_TLV, 4, (BYTE *)&staSendResp, respBuf);
           *respLen = WFA_TLV_HDR_LEN + 4;
           return TRUE;
        }

        theProfile = &myStream->profile;
        if(theProfile == NULL)
        {
           staSendResp.status = STATUS_INVALID;
           wfaEncodeTLV(WFA_TRAFFIC_AGENT_SEND_RESP_TLV, 4, (BYTE *)&staSendResp, respBuf);
           *respLen = WFA_TLV_HDR_LEN + 4;

           return TRUE;
        }
               
        if(theProfile->direction != DIRECT_SEND)
        {
           staSendResp.status = STATUS_INVALID;
           wfaEncodeTLV(WFA_TRAFFIC_AGENT_SEND_RESP_TLV, 4, (BYTE *)&staSendResp, respBuf);
           *respLen = WFA_TLV_HDR_LEN + 4;

           return TRUE;
        }

        /*
         * need to reset the stats
         */
        wMEMSET(&myStream->stats, 0, sizeof(tgStats_t));

        // mark the stream active;
        myStream->state = WFA_STREAM_ACTIVE;

        switch(theProfile->profile)
        {
        case PROF_FILE_TX:
        case PROF_MCAST:
        case PROF_TRANSC:
        gtgTransac = streamid;
        gtgSend = streamid;
        case PROF_CALI_RTD:
        gtgCaliRTD = streamid;
        case PROF_IPTV:
        gtgSend = streamid;
        /*
         * singal the thread to Sending WMM traffic 
         */
         
        //if(usedThread < 
        wmm_thr[usedThread].thr_flag = streamid;
        wPT_MUTEX_LOCK(&wmm_thr[usedThread].thr_flag_mutex);
        wPT_COND_SIGNAL(&wmm_thr[usedThread].thr_flag_cond);
        wPT_MUTEX_UNLOCK(&wmm_thr[usedThread].thr_flag_mutex);
        usedThread++;
        //wfaSetProcPriority(90);

        break;
        } 
    }

    *respLen = 0;
    return TRUE;
}

int wfaTGReset(int len, BYTE *parms, int *respLen, BYTE *respBuf)
{
    dutCmdResponse_t *resetResp = &gGenericResp;
    int i;

    /* need to reset all traffic socket fds */
    if(btSockfd != -1)
    {
       wCLOSE(btSockfd);
       btSockfd = -1;
    }

    for(i = 0; i<WFA_MAX_TRAFFIC_STREAMS; i++)
    {
        if(tgSockfds[i] != -1)
        {
            wCLOSE(tgSockfds[i]);
            tgSockfds[i] = -1;
        }
    }

    /* reset the timer alarm if it was armed */
    wALARM(0);

    /* just reset the flags for the command */
    gtgRecv = 0;
    gtgSend = 0;
    gtgTransac = 0;
#ifdef WFA_VOICE_EXT
    gtgCaliRTD = 0;
    min_rttime = 0xFFFFFFFF;
    gtgPktRTDelay = 0xFFFFFFFF;
#endif

    totalTranPkts = 0;
    //gtimeOut = 0;

    runLoop = 0;

    usedThread = 0;
#ifdef WFA_WMM_PS_EXT
    gtgWmmPS = 0;
    gtgPsPktRecvd = 0;

    if(psSockfd != -1)
    {
       wCLOSE(psSockfd);
       psSockfd = -1;
    }

    wMEMSET(&wmmps_info, 0, sizeof(wfaWmmPS_t));

//         wfaSetDUTPwrMgmt(PS_OFF);
#endif

    e2eResults[0] = '\0';

    /* Also need to clean up WMM streams NOT DONE YET!*/
    slotCnt = 0;             /* reset stream profile container */
    wMEMSET(gStreams, 0, WFA_MAX_TRAFFIC_STREAMS); 

    /*
     * After be asked to reset, it should lower itself back to 
     * normal level. It is optional implementation if it is not called 
     * while it starts sending/receiving for raising priority level.
     */
    //wfaSetProcPriority(20);

    /* encode a TLV for response for "complete ..." */
    resetResp->status = STATUS_COMPLETE; 
    wfaEncodeTLV(WFA_TRAFFIC_AGENT_RESET_RESP_TLV, 4, 
                 (BYTE *)resetResp, respBuf);
    *respLen = WFA_TLV_HDR_LEN + 4; 

    return TRUE;
}

/*
 * calculate the sleep time for different frame rate
 * It should be done according the device
 * This is just one way to adjust the packet delivery speed. If you find 
 * you device does not meet the test requirements, you MUST re-adjust 
 * the method.
 */

/* The HZ value could be found in the build header file */
/* 100 -> 10ms, 1000 -> 1ms , etc                       */
#define WFA_KERNEL_MIN_TIMER_RES   100   /* HZ Value for 10 ms */

void wfaTxSleepTime(int profile, int rate, int *sleepTime, int *throttledRate)
{
    *sleepTime=0;     /* in microseconds */
    /* calculate the sleep time based on frame rate */

    /*
     * Framerate is still required for Multicast traffic
     * Sleep and hold for a timeout.
     *
     * For WMM traffic, the framerate must also need for VO and VI.
     * the framerate 500, OS may not handle it precisely.
     */
    switch(profile)
    {
        /*
         * make it a fix rate
         * according to test plan, it requires ~80kbps which is around 50 frames/s
         * For other cases which may want to run experiments for very high rate, 
         * the change should accommodate the requirement.
         */
        case PROF_MCAST:
        if(rate < 500 && rate >= 50)
        {
            *sleepTime = 100000;  /* sleep for 100 ms */
            *throttledRate = WFA_MCAST_FRATE;
        }
        else
        {
            *sleepTime = 100000;
            *throttledRate = rate;
        }
#if 0
        *throttledRate = WFA_MCAST_FRATE;
#endif
        break;

        /*
         * Vendor must find ways to better adjust the speed for their own device
         */
        case PROF_IPTV:
        case PROF_FILE_TX:
        if(rate >=50 || rate == 0)
        {
	    /* 
	     * this sleepTime indeed is now being used for time period  
	     * to send packets in the throttled Rate.  
	     * The idea here is that in each fixed 20 minisecond period,
	     * The device will send rate/50 (rate = packets / second),
	     * then go sleep for rest of time.
	     */
            *sleepTime = 20000; /* fixed 20 miniseconds */
            *throttledRate = (rate?rate:25000)/50; 
	    printf("Sleep time %i, throttledRate %i\n", *sleepTime, *throttledRate);
        }
        else if (rate > 0 && rate <= 50) /* typically for voice */
        {
            *throttledRate = 1;
            *sleepTime = 1000*1000/rate;
        }
        break;
        default:
          DPRINT_ERR(WFA_ERR, "Incorrect profile\n");
    } 
}

#define WFA_TIME_DIFF(before, after, rtime, dtime) \
             dtime = rtime + (after.tv_sec*1000000 + after.tv_usec) - (before.tv_sec*1000000 + before.tv_usec);

void buzz_time(int delay)
{
    struct timeval now, stop;
    int diff;
    int remain_time = 0;

    wGETTIMEOFDAY(&stop, 0);

    stop.tv_usec += delay;
    if(stop.tv_usec > 1000000)
    {
        stop.tv_usec -=1000000;
	stop.tv_sec +=1;
    }

    do
    {
        wGETTIMEOFDAY(&now, 0);
	WFA_TIME_DIFF(now, stop, remain_time, diff);
    } while(diff>0);
}

/**************************************************/
/* the actually functions to send/receive packets */
/**************************************************/

/* This is going to be a blocking SEND till it finishes */
int wfaSendLongFile(int mySockfd, int streamid, BYTE *aRespBuf, int *aRespLen)
{
    tgProfile_t           *theProf = NULL;
    tgStream_t            *myStream = NULL;
    struct sockaddr_in    toAddr;
    char                  *packBuf; 
    int  packLen;
    int  bytesSent;
    dutCmdResponse_t sendResp;
    int sleepTime = 0;
    int throttledRate = 0;
    struct timeval before, after,af; 
    int difftime = 0, counter = 0;
    struct timeval stime;
    int throttled_est_cost;
    int act_sleep_time;
    gettimeofday(&af,0);
   
    DPRINT_INFO(WFA_OUT, "Entering sendLongFile %i\n", streamid);

    /* find the profile */
    myStream = findStreamProfile(streamid);
    if(myStream == NULL)
    {
        return FALSE;    
    }

    theProf = &myStream->profile;

    if(theProf == NULL)
    {
        return FALSE;
    }

    packLen = theProf->pksize;

    /* allocate a buf */
    packBuf = (char *)malloc(packLen);
    wMEMSET(packBuf, 1, packLen);

    /* fill in the header */
    wSTRNCPY(packBuf, "1345678", sizeof(tgHeader_t));

    /* initialize the destination address */
    wMEMSET(&toAddr, 0, sizeof(toAddr));
    toAddr.sin_family = AF_INET;
    toAddr.sin_addr.s_addr = inet_addr(theProf->dipaddr);
    toAddr.sin_port = htons(theProf->dport); 

    /* if a frame rate and duration are defined, then we know
     * interval for each packet and how many packets it needs to
     * send.
     */
    if(theProf->duration != 0)
    {
        printf("duration %i\n", theProf->duration);
        
        /* 
         *  use this to decide periodical interval sleep time and frames to send
         *  int the each interval.
         *  Each device should adopt a own algorithm for better performance
         */
        wfaTxSleepTime(theProf->profile, theProf->rate, &sleepTime, &throttledRate);
        /* 
         * alright, we need to raise the priority level of the process
         * to improve the real-time performance of packet sending.
         * Since this is for tuning purpose, it is optional implementation.
         */
        //wfaSetProcPriority(60);
	
	//interval = 1*1000000/theProf->rate ; // in usec;

	// Here assumes it takes 20 usec to send a packet
	throttled_est_cost = throttledRate * 20;  // MUST estimate the cost per ppk
	act_sleep_time = sleepTime - adj_latency;
	if (act_sleep_time <= 0)
	    act_sleep_time = sleepTime;  

        printf("sleep time %i act_sleep_time %i\n", sleepTime, act_sleep_time);

        runLoop=1;
        while(runLoop)
        {
	   counter++;
           /* fill in the counter */
           int2BuffBigEndian(counter, &((tgHeader_t *)packBuf)->hdr[8]);

           /*
            * the following code is only used to slow down
            * over fast traffic flooding the buffer and cause
            * packet drop or the other end not able to receive due to
            * some limitations, purely for experiment purpose.
            * each implementation needs some fine tune to it.
            */
	   if(counter ==1)
	   {
               wGETTIMEOFDAY(&before, NULL);

               before.tv_usec += sleepTime;
               if(before.tv_usec > 1000000)
               {
                   before.tv_usec -= 1000000;
                   before.tv_sec +=1;
               }
           }

           if(throttledRate != 0)
           {
              if(counter%throttledRate == 0)
              {
                 wGETTIMEOFDAY(&after, NULL);
	         difftime = wfa_itime_diff(&after, &before);

                 if(difftime > adj_latency)
                 {
                    // too much time left, go sleep
                    wUSLEEP(difftime-adj_latency);

                    wGETTIMEOFDAY(&after, NULL);
	            difftime = wfa_itime_diff(&after, &before);
                 }

		 // burn the rest to absort latency
	         if(difftime >0)
	             buzz_time(difftime);

	         before.tv_usec += sleepTime;
	         if(before.tv_usec > 1000000)
	         {
	             before.tv_usec -= 1000000;
		     before.tv_sec +=1;
	         }
              }
           } // otherwise, it floods 

	   /*
	    * Fill the timestamp to the header.
	    */
	   wGETTIMEOFDAY(&stime, NULL);

	   int2BuffBigEndian(stime.tv_sec, &((tgHeader_t *)packBuf)->hdr[12]);
	   int2BuffBigEndian(stime.tv_usec, &((tgHeader_t *)packBuf)->hdr[16]);

           bytesSent = wfaTrafficSendTo(mySockfd, packBuf, packLen, 
                            (struct sockaddr *)&toAddr);

           if(bytesSent != -1)
           {
              myStream->stats.txPayloadBytes += bytesSent; 
              myStream->stats.txFrames++ ;
           }
           else
           {
               int errsv = errno;
               switch(errsv)
               {
	           case EAGAIN:
                   case ENOBUFS:
                        DPRINT_ERR(WFA_ERR, "send error\n");
                        wUSLEEP(1000);             /* hold for 1 ms */
                        counter-- ;
                        myStream->stats.txFrames--;
                   break;
                   case ECONNRESET:
                        runLoop = 0;
                   break;
                   case EPIPE:
                        runLoop = 0;
                   break;
                   default:
                      perror("sendto: ");
                      DPRINT_ERR(WFA_ERR, "Packet sent error\n");
               }
           }

        }


        /*
         * lower back to an original level if the process is raised previously
         * It is optional.
         */
        //wfaSetProcPriority(30); 
    }
    else /* invalid parameters */
    {
        /* encode a TLV for response for "invalid ..." */
        sendResp.status = STATUS_INVALID;
        wfaEncodeTLV(WFA_TRAFFIC_AGENT_SEND_RESP_TLV, 4, 
                 (BYTE *)&sendResp, (BYTE *)aRespBuf);

        /* done here */
        *aRespLen = WFA_TLV_HDR_LEN + 4; 

        return DONE;
    }

    gtgSend = 0;

    /* free the buffer */
    wFREE(packBuf);

    //printf("done sending long\n");
    /* return statistics */
    sendResp.status = STATUS_COMPLETE;
    sendResp.streamId = myStream->id;
    wMEMCPY(&sendResp.cmdru.stats, &myStream->stats, sizeof(tgStats_t)); 

#if 0
    DPRINT_INFO(WFA_OUT, "stream Id %u tx %u total %llu\n", myStream->id, myStream->stats.txFrames, myStream->stats.txPayloadBytes);
#endif

    wfaEncodeTLV(WFA_TRAFFIC_AGENT_SEND_RESP_TLV, sizeof(dutCmdResponse_t), 
                 (BYTE *)&sendResp, (BYTE *)aRespBuf);

    *aRespLen = WFA_TLV_HDR_LEN + sizeof(dutCmdResponse_t);

    return DONE;
}

/* this only sends one packet a time */
int wfaSendShortFile(int mySockfd, int streamid, BYTE *sendBuf, int pksize, BYTE *aRespBuf, int *aRespLen)
{
   BYTE *packBuf = sendBuf; 
   struct sockaddr_in toAddr;
   tgProfile_t *theProf;
   tgStream_t *myStream;
   int packLen, bytesSent=-1;
   dutCmdResponse_t sendResp;

   if(mySockfd == -1)
   {
      /* stop */ 
      gtgTransac = 0;
      //gtimeOut = 0; 
      gtgRecv = 0;
      gtgSend = 0;
      printf("stop short traffic\n");

      myStream = findStreamProfile(streamid);
      if(myStream != NULL)
      {
          sendResp.status = STATUS_COMPLETE;
          sendResp.streamId = streamid;
          wMEMCPY(&sendResp.cmdru.stats, &myStream->stats, sizeof(tgStats_t)); 
    
          wfaEncodeTLV(WFA_TRAFFIC_AGENT_SEND_RESP_TLV, sizeof(dutCmdResponse_t), (BYTE *)&sendResp, aRespBuf);

          *aRespLen = WFA_TLV_HDR_LEN + sizeof(dutCmdResponse_t);
      }

      return DONE;
   }

   /* find the profile */
   myStream = findStreamProfile(streamid);

   if(myStream == NULL)
   {
        return FALSE;
   }

   theProf = &myStream->profile;
   if(theProf == NULL)
   {
        return FALSE;
   }

   if(pksize == 0)
      packLen = theProf->pksize;
   else
      packLen = pksize;

   wMEMSET(&toAddr, 0, sizeof(toAddr));
   toAddr.sin_family = AF_INET;
   toAddr.sin_addr.s_addr = inet_addr(theProf->sipaddr);
   toAddr.sin_port = htons(theProf->sport); 

   if(gtgRecv && gtgTransac)
   {
//      printf("mySock %i sipaddr %s sport %i\n", mySockfd, theProf->sipaddr, theProf->sport); 
      toAddr.sin_addr.s_addr = inet_addr(theProf->sipaddr);
      toAddr.sin_port = htons(theProf->sport); 
   }
   else if(gtgSend && gtgTransac)
   {
//      printf("mySock %i dipaddr %s dport %i\n", mySockfd, theProf->dipaddr, theProf->dport); 
      toAddr.sin_addr.s_addr = inet_addr(theProf->dipaddr);
      toAddr.sin_port = htons(theProf->dport); 
   }

   int2BuffBigEndian(myStream->stats.txFrames, &((tgHeader_t *)packBuf)->hdr[8]);

   if(mySockfd != -1)
      bytesSent = wfaTrafficSendTo(mySockfd, (char *)packBuf, packLen, (struct sockaddr *)&toAddr);

   if(bytesSent != -1)
   {
      myStream->stats.txFrames++;
      myStream->stats.txPayloadBytes += bytesSent;
   }
   else
   {
      int errsv = errno;
      switch(errsv)
      {
          case EAGAIN:
          case ENOBUFS:
             DPRINT_ERR(WFA_ERR, "send error\n");
             wUSLEEP(1000);             /* hold for 1 ms */
             myStream->stats.txFrames--;
	  break;
	  default:
              ;;
             //perror("sendto: ");
      }
   }

   sentTranPkts++;

   return TRUE;
}

/* always receive from a specified IP address and Port */
int wfaRecvFile(int mySockfd, int streamid, char *recvBuf)
{
    /* how many packets are received */  
    char *packBuf = recvBuf; 
    struct sockaddr_in fromAddr;
    tgProfile_t *theProf;
    tgStream_t *myStream;
    unsigned int bytesRecvd;
    int lostPkts;

    /* find the profile */
    myStream = findStreamProfile(streamid);
    if(myStream == NULL)
    {
        return FALSE;
    }

    theProf = &myStream->profile;
    if(theProf == NULL)
    {
        return FALSE;
    }

    wMEMSET(packBuf, 0, MAX_UDP_LEN);

    wMEMSET(&fromAddr, 0, sizeof(fromAddr));
    fromAddr.sin_family = AF_INET;
    fromAddr.sin_addr.s_addr = inet_addr(theProf->dipaddr);
    fromAddr.sin_port = htons(theProf->dport); 

    if(gtgRecv && gtgTransac)
    {
       fromAddr.sin_addr.s_addr = inet_addr(theProf->sipaddr);
       fromAddr.sin_port = htons(theProf->sport); 
    }
    else if(gtgSend && gtgTransac)
    {
       fromAddr.sin_addr.s_addr = inet_addr(theProf->dipaddr);
       fromAddr.sin_port = htons(theProf->dport); 
    }

    /* it is always to receive at least one packet, in case more in the
       queue, just pick them up.
     */
    bytesRecvd = wfaTrafficRecv(mySockfd, packBuf, (struct sockaddr *)&fromAddr);
    if(bytesRecvd != -1)
    {
       myStream->stats.rxFrames++;    
       myStream->stats.rxPayloadBytes +=bytesRecvd;

       /*
        *  Get the lost packet count
        */
       lostPkts = bigEndianBuff2Int(&((tgHeader_t *)packBuf)->hdr[8]) - 1 - myStream->lastPktSN; 
       myStream->stats.lostPkts += lostPkts; 
       myStream->lastPktSN = bigEndianBuff2Int(&((tgHeader_t *)packBuf)->hdr[8]); 
    }
    else
    {
#if 0
       DPRINT_ERR(WFA_ERR, "Packet received error\n");
       perror("receive error");
#endif
    }
    return (bytesRecvd);
}
