/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.common.orders;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.Socket;
import java.util.Iterator;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.videolan.vlma.common.Configuration;
import org.videolan.vlma.common.VlServer;
import org.videolan.vlma.common.adapters.IVlAdapter;
import org.videolan.vlma.common.medias.IVlMedia;
import org.videolan.vlma.common.medias.VlMediaGroup;
import org.videolan.vlma.common.medias.VlSatChannel;
import org.videolan.vlma.common.medias.VlTNTChannel;
import org.videolan.vlma.common.medias.VlFilesChannel;

/**
 * This class represents an order given to a VLC server.
 * It is associated with a media group.
 * 
 * @author SylV
 */
public class VlOrder {
    private static final Logger logger = Logger.getLogger(VlOrder.class);

    private VlServer server;

    private IVlAdapter adapter;

    private VlMediaGroup medias;

    /**
     * Time to wait for during telnet connections.
     */
    private static final int TELNET_DEFAULT_WAIT = 100;

    /**
     * Time to wait for the response.
     */
    private static final int TELNET_DEFAULT_DELAY = 1000;

    /**
     * Default TTL for the diffusion.
     */
    private static final int VLC_TTL = 12;

    /**
     * The default bandwidth.
     */
    private static final int VLC_DVB_BANDWIDTH = 8;

    private Socket telnetSocket;

    private PrintWriter telnetOut;

    private BufferedReader telnetIn;

    /**
     * Gets the adapter associated to the order.
     * 
     * @return the adapter
     */
    public IVlAdapter getAdapter() {
        return adapter;
    }

    /**
     * Sets the adapter associated with the order.
     * 
     * @param adapter
     *            the adapter to set
     */
    public void setAdapter(IVlAdapter adapter) {
        this.adapter = adapter;
    }

    /**
     * Gets the server associated with the order.
     * 
     * @return the server
     */
    public VlServer getServer() {
        return server;
    }

    /**
     * Sets the server associated with the order.
     * 
     * @param server
     *            the server
     */
    public void setServer(VlServer server) {
        this.server = server;
    }

    /**
     * Gets the media group to stream.
     * 
     * @return the media group
     */
    public VlMediaGroup getMedias() {
        return medias;
    }

    /**
     * Sets the media group to stream.
     * 
     * @param medias
     *            the media group
     */
    public void setMedias(VlMediaGroup medias) {
        this.medias = medias;
    }

    /**
     * Gets the VLM command name associated with this order.
     * 
     * @return the command name
     */
    private String getVLMCommandName() {
        return "flux-" + adapter.getName();
    }

    /**
     * Connects to the VLC server using telnet.
     * 
     * @throws IOException
     *             Connection aborted.
     */
    private void telnetConnect() throws IOException {
        telnetSocket = new Socket(server.getIp(), Configuration.getInstance().getVlcTelnetPort());
        telnetOut = new PrintWriter(telnetSocket.getOutputStream(), true);
        telnetIn = new BufferedReader(new InputStreamReader(telnetSocket
                .getInputStream()));
        telnetOut.println(Configuration.getInstance().getVlcTelnetPassword());
    }

    /**
     * Closes the telnet connection.
     * 
     * @throws IOException
     *             Operation aborted.
     */
    private void telnetClose() throws IOException {
        telnetOut.close();
        telnetIn.close();
        telnetSocket.close();
    }

    /**
     * Sends a command to to VLC server.
     * 
     * @param command
     *            the command to send
     * @throws IOException
     *             Operation aborted.
     */
    private void telnetCommand(String command) throws IOException {
        logger.log(Level.DEBUG, "Envoi de la commande " + command + " à "
                + server.getName());
        telnetOut.println(command);
        int d = TELNET_DEFAULT_DELAY;
        while (!telnetIn.ready() && (d -= TELNET_DEFAULT_WAIT) > 0) {
            try {
                Thread.sleep(TELNET_DEFAULT_WAIT);
            } catch (InterruptedException e) {
            }
        }
        String r = "";
        while (telnetIn.ready()) {
            r += (char) telnetIn.read();
        }
        logger.log(Level.DEBUG, "Command result: " + r);
    }

    /**
     * Orders the media group diffusion by the server.
     * 
     * @throws IOException
     *             Operation aborted.
     */
    public void start() throws IOException {
        // Let's connect to the server using telnet
        telnetConnect();
        // Remove the stream if there is one
        telnetCommand("setup " + getVLMCommandName() + " disabled");
        telnetCommand("control " + getVLMCommandName() + " stop");
        telnetCommand("del " + getVLMCommandName());
        // Create a multicast diffusion program
        telnetCommand("new " + getVLMCommandName() + " broadcast");
        // Verbose mode using colors
        telnetCommand("setup " + getVLMCommandName() + " option vvv");
        telnetCommand("setup " + getVLMCommandName() + " option color");
        telnetCommand("setup " + getVLMCommandName() + " option ttl="
                + Integer.toString(VLC_TTL));

        // The media to stream is a satellite channel
        if (medias.getMediaClass() == VlSatChannel.class) {
            if (!medias.medias.isEmpty()) {
                VlSatChannel ch = (VlSatChannel) medias.medias.get(0);
                telnetCommand("setup " + getVLMCommandName() + " input dvb://");
                telnetCommand("setup " + getVLMCommandName()
                        + " option dvb-frequency="
                        + Integer.toString(ch.getFrequency() * 1000));
                telnetCommand("setup " + getVLMCommandName()
                        + " option dvb-adapter=" + getAdapter().getName());
                telnetCommand("setup " + getVLMCommandName()
                        + " option dvb-bandwidth="
                        + Integer.toString(VLC_DVB_BANDWIDTH));
                if (ch.getPolarisation() == 'H') {
                    telnetCommand("setup " + getVLMCommandName()
                            + " option dvb-voltage=18");
                } else if (ch.getPolarisation() == 'V') {
                    telnetCommand("setup " + getVLMCommandName()
                            + " option dvb-voltage=13");
                }
                telnetCommand("setup " + getVLMCommandName()
                        + " option dvb-srate=" + ch.getSymbolRate());

                String ids = "";
                String command = "";
                Iterator<IVlMedia> k = medias.medias.iterator();
                while (k.hasNext()) {
                    ch = (VlSatChannel) k.next();
                    if (ids != "") {
                        ids += ",";
                    }
                    ids += Integer.toString(ch.getSid());
                    if (command != "") {
                        command += ",";
                    }
                    command += String
                            .format(
                                    "dst=standard{mux=ts,access=udp,dst=%s,sap,name=\"%s\",group=\"%s\"},select=\"program=%d\"",
                                    ch.getProgram().getIp().getHostAddress(),
                                    ch.getProgram().getSapName(),
                                    ch.getProgram().getSapGroup(),
                                    ch.getSid());
                }
                // Orders to stream the channels
                telnetCommand("setup " + getVLMCommandName()
                        + " option programs=" + ids);
                command = "setup " + getVLMCommandName()
                        + " output #duplicate{" + command + "}";
                telnetCommand(command);
                telnetCommand("setup " + getVLMCommandName() + " enabled");
                telnetCommand("control " + getVLMCommandName() + " play");
            }
        }
        // The media to stream is a TNT channel
        else if (medias.getMediaClass() == VlTNTChannel.class) {
            if (!medias.medias.isEmpty()) {
                VlTNTChannel ch = (VlTNTChannel) medias.medias.get(0);
                telnetCommand("setup " + getVLMCommandName() + " input dvb://");
                telnetCommand("setup " + getVLMCommandName()
                        + " option dvb-frequency="
                        + Integer.toString(ch.getFrequency()));
                telnetCommand("setup " + getVLMCommandName()
                        + " option dvb-adapter=" + getAdapter().getName());
                telnetCommand("setup " + getVLMCommandName()
                        + " option dvb-bandwidth="
                        + Integer.toString(VLC_DVB_BANDWIDTH));

                String ids = "";
                String command = "";
                Iterator<IVlMedia> k = medias.medias.iterator();
                while (k.hasNext()) {
                    ch = (VlTNTChannel) k.next();
                    if (ids != "") {
                        ids += ",";
                    }
                    ids += Integer.toString(ch.getSid());
                    if (command != "") {
                        command += ",";
                    }
                    command += String
                            .format(
                                    "dst=standard{mux=ts,access=udp,dst=%s,sap,name=\"%s\",group=\"%s\"},select=\"program=%d\"",
                                    ch.getProgram().getIp().getHostAddress(),
                                    ch.getProgram().getSapName(), ch
                                            .getProgram().getSapGroup(), ch
                                            .getSid());
                }
                // Orders to stream the channels
                telnetCommand("setup " + getVLMCommandName()
                        + " option programs=" + ids);
                command = "setup " + getVLMCommandName()
                        + " output #duplicate{" + command + "}";
                telnetCommand(command);
                telnetCommand("setup " + getVLMCommandName() + " enabled");
                telnetCommand("control " + getVLMCommandName() + " play");
            }
        } 
        // The media to stream is a file
        else if (medias.getMediaClass() == VlFilesChannel.class && !medias.medias.isEmpty()) {
            VlFilesChannel ch = (VlFilesChannel) medias.medias.get(0);

            telnetCommand("setup " + getVLMCommandName()
                    + " loop");
            telnetCommand("setup " + getVLMCommandName()
                    + " option sout-keep");
                
            // Set the input with the file list
            for (String file_path : ch.getFiles()) {
                telnetCommand("setup " + getVLMCommandName() + " input \"" + file_path + "\"");
            }
            
            String command = String
            .format(
                    "#standard{mux=ts,access=udp,dst=%s,sap,name=\"%s\",group=\"%s\"}",
                    ch.getProgram().getIp().getHostAddress(),
                    ch.getProgram().getSapName(),
                    ch.getProgram().getSapGroup());
            
            // Orders to stream the channels
            command = "setup " + getVLMCommandName()
                    + " output " + command;
            
            telnetCommand(command);
            telnetCommand("setup " + getVLMCommandName() + " enabled");
            telnetCommand("control " + getVLMCommandName() + " play");
        }
        // Close connection
        telnetClose();
    }

    /**
     * Orders a server to stop streaming.
     * 
     * @throws IOException
     */
    public void stop() throws IOException {
        telnetConnect();
        telnetCommand("control " + getVLMCommandName() + " disabled");
        telnetCommand("control " + getVLMCommandName() + " stop");
        telnetCommand("del " + getVLMCommandName());
        telnetClose();
    }

}
