/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/testing/aws_test_harness.h>

#include <aws/io/io.h>

/* Initialize this library and its dependencies.
 * This will fail if:
 * - the error info list is out of sync with the error enums.
 * - there is a memory leak */
static int s_test_io_library_init(struct aws_allocator *allocator, void *ctx) {
    (void)allocator;
    (void)ctx;

    aws_io_library_init(allocator);
    aws_io_library_clean_up();
    return AWS_OP_SUCCESS;
}
AWS_TEST_CASE(io_library_init, s_test_io_library_init)

/* Ensure the library can go through the init/cleanup cycle multiple times */
static int s_test_io_library_init_cleanup_init_cleanup(struct aws_allocator *allocator, void *ctx) {
    (void)allocator;
    (void)ctx;

    aws_io_library_init(allocator);
    aws_io_library_clean_up();

    aws_io_library_init(allocator);
    aws_io_library_clean_up();

    return AWS_OP_SUCCESS;
}
AWS_TEST_CASE(io_library_init_cleanup_init_cleanup, s_test_io_library_init_cleanup_init_cleanup)

/* Make sure that new error codes are being added to the end instead of in the middle. */
static int s_test_io_library_error_order(struct aws_allocator *allocator, void *ctx) {
    (void)allocator;
    (void)ctx;

    aws_io_library_init(allocator);
    // Checking against expected int values of the error codes.
    // The checked error codes are the first, somewhere in the middle, and last as of this commit.
    int error_code_initial = 1024;
    int error_code_mid = 1067;
    int error_code_end = 1194;

    ASSERT_TRUE(error_code_initial == AWS_IO_CHANNEL_ERROR_ERROR_CANT_ACCEPT_INPUT);
    ASSERT_TRUE(error_code_mid == AWS_IO_TLS_NEGOTIATION_TIMEOUT);
    ASSERT_TRUE(error_code_end == AWS_IO_TLS_HOST_NAME_MISMATCH);
    aws_io_library_clean_up();
    return AWS_OP_SUCCESS;
}

AWS_TEST_CASE(io_library_error_order, s_test_io_library_error_order)

static int s_test_io_error_code_is_retryable(struct aws_allocator *allocator, void *ctx) {
    (void)allocator;
    (void)ctx;

    int error_code = 0;

    ASSERT_FALSE(aws_io_error_code_is_retryable(error_code));
    {
        error_code = AWS_IO_SOCKET_CLOSED;
        ASSERT_TRUE(aws_io_error_code_is_retryable(error_code));
    }
    {
        error_code = AWS_IO_SOCKET_CONNECT_ABORTED;
        ASSERT_TRUE(aws_io_error_code_is_retryable(error_code));
    }
    {
        error_code = AWS_IO_SOCKET_CONNECTION_REFUSED;
        ASSERT_TRUE(aws_io_error_code_is_retryable(error_code));
    }
    {
        error_code = AWS_IO_SOCKET_NETWORK_DOWN;
        ASSERT_TRUE(aws_io_error_code_is_retryable(error_code));
    }
    {
        error_code = AWS_IO_DNS_QUERY_AGAIN;
        ASSERT_TRUE(aws_io_error_code_is_retryable(error_code));
    }
    {
        error_code = AWS_IO_DNS_NO_ADDRESS_FOR_HOST;
        ASSERT_TRUE(aws_io_error_code_is_retryable(error_code));
    }
    {
        error_code = AWS_IO_SOCKET_TIMEOUT;
        ASSERT_TRUE(aws_io_error_code_is_retryable(error_code));
    }
    {
        error_code = AWS_IO_TLS_NEGOTIATION_TIMEOUT;
        ASSERT_TRUE(aws_io_error_code_is_retryable(error_code));
    }

    {
        error_code = AWS_IO_DNS_QUERY_FAILED;
        ASSERT_FALSE(aws_io_error_code_is_retryable(error_code));
    }

    return AWS_OP_SUCCESS;
}

AWS_TEST_CASE(io_error_code_is_retryable, s_test_io_error_code_is_retryable)
