package main

import (
	"fmt"
	"strings"
)

func (bc *generousContext) generateGoCode() error {
	bc.task("Generating go code")

	doc := bc.newGenerousRelativeDoc("../messages/messages.go")

	doc.line("// Code generated by generous; DO NOT EDIT.")
	doc.line("")
	doc.line("package messages")
	doc.line("")
	doc.line("import (")
	doc.line("	%q", "encoding/json")
	doc.line("	%q", "github.com/pkg/errors")
	doc.line("")
	doc.line("	%q", "github.com/itchio/butler/butlerd")
	doc.line("	%q", "github.com/itchio/butler/butlerd/jsonrpc2")
	doc.line(")")
	doc.line("")

	doc.line("type router interface {")
	doc.line("  Register(method string, hf butlerd.RequestHandler)")
	doc.line("  RegisterNotification(method string, nf butlerd.NotificationHandler)")
	doc.line("}")
	doc.line("")

	scope := newScope(bc)
	must(scope.assimilate("github.com/itchio/butler/butlerd", "types.go"))

	var clientRequests []string

	for _, category := range scope.categoryList {
		cat := scope.categories[category]
		doc.line("")
		doc.line("//==============================")
		doc.line("// %s", category)
		doc.line("//==============================")
		doc.line("")

		for _, entry := range cat.entries {
			switch entry.kind {
			case entryKindParams:
				ts := asType(entry.gd)
				varName := strings.TrimSuffix(ts.Name.Name, "Params")
				typeName := varName + "Type"
				paramsTypeName := fmt.Sprintf("butlerd.%s", ts.Name.Name)
				resultTypeName := fmt.Sprintf("butlerd.%sResult", strings.TrimSuffix(ts.Name.Name, "Params"))
				method := entry.name
				if entry.caller == callerClient {
					clientRequests = append(clientRequests, method)
				}

				doc.line("// %s (Request)", method)
				doc.line("")
				doc.line("type %s struct {}", typeName)
				doc.line("")
				doc.line("var _ RequestMessage = (*%s)(nil)", typeName)
				doc.line("")
				doc.line("func (r *%s) Method() string {", typeName)
				doc.line("  return %#v", method)
				doc.line("}")

				{
					helperPrefix := ""
					if entry.caller == callerServer {
						helperPrefix = "Test"
					}
					doc.line("")
					doc.line("func (r *%s) %sRegister(router router, f func(*butlerd.RequestContext, %s) (*%s, error)) {", typeName, helperPrefix, paramsTypeName, resultTypeName)
					doc.line("  router.Register(%#v, func (rc *butlerd.RequestContext) (interface{}, error) {", method)
					doc.line("    var params %s", paramsTypeName)
					doc.line("    err := json.Unmarshal(*rc.Params, &params)")
					doc.line("    if err != nil {")
					doc.line("    	return nil, &butlerd.RpcError{Code: jsonrpc2.CodeParseError, Message: err.Error()}")
					doc.line("    }")
					doc.line("    err = params.Validate()")
					doc.line("    if err != nil {")
					doc.line("    	return nil, err")
					doc.line("    }")
					doc.line("    res, err := f(rc, params)")
					doc.line("    if err != nil {")
					doc.line("    	return nil, err")
					doc.line("    }")
					doc.line("    if res == nil {")
					doc.line("    	return nil, errors.New(%#v)", fmt.Sprintf("internal error: nil result for %s", method))
					doc.line("    }")
					doc.line("    return res, nil")
					doc.line("  })")
					doc.line("}")
				}

				{
					helperPrefix := ""
					if entry.caller == callerClient {
						helperPrefix = "Test"
					}
					doc.line("")
					doc.line("func (r *%s) %sCall(rc *butlerd.RequestContext, params %s) (*%s, error) {", typeName, helperPrefix, paramsTypeName, resultTypeName)
					doc.line("  var result %s", resultTypeName)
					doc.line("  err := rc.Call(%#v, params, &result)", method)
					doc.line("  return &result, err")
					doc.line("}")
				}

				doc.line("")
				doc.line("var %s *%s", varName, typeName)
				doc.line("")

			case entryKindNotification:
				ts := asType(entry.gd)
				varName := strings.TrimSuffix(ts.Name.Name, "Notification")
				typeName := varName + "Type"
				paramsTypeName := fmt.Sprintf("butlerd.%s", ts.Name.Name)
				method := entry.name

				doc.line("// %s (Notification)", method)
				doc.line("")
				doc.line("type %s struct {}", typeName)
				doc.line("")
				doc.line("var _ NotificationMessage = (*%s)(nil)", typeName)
				doc.line("")
				doc.line("func (r *%s) Method() string {", typeName)
				doc.line("  return %#v", method)
				doc.line("}")
				doc.line("")
				doc.line("func (r *%s) Notify(rc *butlerd.RequestContext, params %s) (error) {", typeName, paramsTypeName)
				doc.line("  return rc.Notify(%#v, params)", method)
				doc.line("}")
				doc.line("")
				doc.line("func (r *%s) Register(router router, f func(%s)) {", typeName, paramsTypeName)
				doc.line("  router.RegisterNotification(%#v, func (notif jsonrpc2.Notification) {", method)
				doc.line("    var params %s", paramsTypeName)
				doc.line("    if notif.Params != nil {")
				doc.line("      err := json.Unmarshal(*notif.Params, &params)")
				doc.line("      if err != nil {")
				doc.line("        return")
				doc.line("      }")
				doc.line("    }")
				doc.line("    f(params)")
				doc.line("  })")
				doc.line("}")
				doc.line("")
				doc.line("var %s *%s", varName, typeName)
				doc.line("")
			}
		}
	}

	doc.line("")
	doc.line("func EnsureAllRequests(router *butlerd.Router) {")
	for _, method := range clientRequests {
		doc.line("  if _, ok := router.Handlers[%#v]; !ok { panic(%#v) }", method, fmt.Sprintf("missing request handler for (%s)", method))
	}
	doc.line("}")
	doc.line("")

	doc.commit("")
	doc.write()

	return nil
}
