/*
 *
 *  Copyright (C) 2022, OFFIS e.V.
 *  All rights reserved.  See COPYRIGHT file for details.
 *
 *  This software and supporting documentation were developed by
 *
 *    OFFIS e.V.
 *    R&D Division Health
 *    Escherweg 2
 *    D-26121 Oldenburg, Germany
 *
 *
 *  Module:  oficonv
 *
 *  Author:  Marco Eichelberg
 *
 *  Purpose: GBK related tables for oficonv unit tests
 *
 */

#ifndef TABLES_GBK_H
#define TABLES_GBK_H

// GBK (Chinese) is a true superset of GB2312.
// Every character is either a single byte < 128, or a sequence of two bytes
// where the first byte has a value >= 0x81, and the second byte has a value
// >= 0x40. In this case the first byte defines the
// row and the second byte the entry within that row.

// Most rows in the GB2312 table use all byte positions from 40 to 7E and 80 to FE for the second byte
unsigned char gbk_full_row[] = {
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e,
  0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe
};

// GB2312 row 08 has been extended with block graphics characters
unsigned char gbk_row_0xa8[] = {
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e,
  0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
  0x90, 0x91, 0x92, 0x93, 0x94, 0x95,
        0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba,
                                0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9
};

// GB2312 row 09 has been extended
unsigned char gbk_row_0xa9[] = {
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,       0x59, 0x5a,       0x5c,
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e,
  0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88,
                          0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef
};

// Some rows do not use the character positions > 0xa0
unsigned char gbk_row_0xaa_to_0xaf[] = {
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e,
  0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
  0xa0
};

// Row 0x7d (end of level 1 character set) does not use the last five bytes
unsigned char gbk_row_0xd7[] = {
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
  0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e,
  0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
  0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
  0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
  0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
  0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
  0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
  0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
  0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9
};

// Row 0xfe (end of level 2 character set) does not use all byte positions
unsigned char gbk_row_0xfe[] = {
  0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
  0x50,                   0x54, 0x55, 0x56, 0x57, 0x58,       0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
  0x60,       0x62, 0x63, 0x64, 0x65,             0x68, 0x69, 0x6a, 0x6b,             0x6e, 0x6f,
  0x70, 0x71, 0x72, 0x73, 0x74, 0x75,       0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d,      
  0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
              0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f
};

unsigned char gbk_row_0x81_converted_to_utf8[] = {
  0xe4, 0xb8, 0x82, 0xe4, 0xb8, 0x84, 0xe4, 0xb8, 0x85, 0xe4, 0xb8, 0x86,
  0xe4, 0xb8, 0x8f, 0xe4, 0xb8, 0x92, 0xe4, 0xb8, 0x97, 0xe4, 0xb8, 0x9f,
  0xe4, 0xb8, 0xa0, 0xe4, 0xb8, 0xa1, 0xe4, 0xb8, 0xa3, 0xe4, 0xb8, 0xa6,
  0xe4, 0xb8, 0xa9, 0xe4, 0xb8, 0xae, 0xe4, 0xb8, 0xaf, 0xe4, 0xb8, 0xb1,
  0xe4, 0xb8, 0xb3, 0xe4, 0xb8, 0xb5, 0xe4, 0xb8, 0xb7, 0xe4, 0xb8, 0xbc,
  0xe4, 0xb9, 0x80, 0xe4, 0xb9, 0x81, 0xe4, 0xb9, 0x82, 0xe4, 0xb9, 0x84,
  0xe4, 0xb9, 0x86, 0xe4, 0xb9, 0x8a, 0xe4, 0xb9, 0x91, 0xe4, 0xb9, 0x95,
  0xe4, 0xb9, 0x97, 0xe4, 0xb9, 0x9a, 0xe4, 0xb9, 0x9b, 0xe4, 0xb9, 0xa2,
  0xe4, 0xb9, 0xa3, 0xe4, 0xb9, 0xa4, 0xe4, 0xb9, 0xa5, 0xe4, 0xb9, 0xa7,
  0xe4, 0xb9, 0xa8, 0xe4, 0xb9, 0xaa, 0xe4, 0xb9, 0xab, 0xe4, 0xb9, 0xac,
  0xe4, 0xb9, 0xad, 0xe4, 0xb9, 0xae, 0xe4, 0xb9, 0xaf, 0xe4, 0xb9, 0xb2,
  0xe4, 0xb9, 0xb4, 0xe4, 0xb9, 0xb5, 0xe4, 0xb9, 0xb6, 0xe4, 0xb9, 0xb7,
  0xe4, 0xb9, 0xb8, 0xe4, 0xb9, 0xb9, 0xe4, 0xb9, 0xba, 0xe4, 0xb9, 0xbb,
  0xe4, 0xb9, 0xbc, 0xe4, 0xb9, 0xbd, 0xe4, 0xb9, 0xbf, 0xe4, 0xba, 0x80,
  0xe4, 0xba, 0x81, 0xe4, 0xba, 0x82, 0xe4, 0xba, 0x83, 0xe4, 0xba, 0x84,
  0xe4, 0xba, 0x85, 0xe4, 0xba, 0x87, 0xe4, 0xba, 0x8a, 0xe4, 0xba, 0x90,
  0xe4, 0xba, 0x96, 0xe4, 0xba, 0x97, 0xe4, 0xba, 0x99, 0xe4, 0xba, 0x9c,
  0xe4, 0xba, 0x9d, 0xe4, 0xba, 0x9e, 0xe4, 0xba, 0xa3, 0xe4, 0xba, 0xaa,
  0xe4, 0xba, 0xaf, 0xe4, 0xba, 0xb0, 0xe4, 0xba, 0xb1, 0xe4, 0xba, 0xb4,
  0xe4, 0xba, 0xb6, 0xe4, 0xba, 0xb7, 0xe4, 0xba, 0xb8, 0xe4, 0xba, 0xb9,
  0xe4, 0xba, 0xbc, 0xe4, 0xba, 0xbd, 0xe4, 0xba, 0xbe, 0xe4, 0xbb, 0x88,
  0xe4, 0xbb, 0x8c, 0xe4, 0xbb, 0x8f, 0xe4, 0xbb, 0x90, 0xe4, 0xbb, 0x92,
  0xe4, 0xbb, 0x9a, 0xe4, 0xbb, 0x9b, 0xe4, 0xbb, 0x9c, 0xe4, 0xbb, 0xa0,
  0xe4, 0xbb, 0xa2, 0xe4, 0xbb, 0xa6, 0xe4, 0xbb, 0xa7, 0xe4, 0xbb, 0xa9,
  0xe4, 0xbb, 0xad, 0xe4, 0xbb, 0xae, 0xe4, 0xbb, 0xaf, 0xe4, 0xbb, 0xb1,
  0xe4, 0xbb, 0xb4, 0xe4, 0xbb, 0xb8, 0xe4, 0xbb, 0xb9, 0xe4, 0xbb, 0xba,
  0xe4, 0xbb, 0xbc, 0xe4, 0xbb, 0xbe, 0xe4, 0xbc, 0x80, 0xe4, 0xbc, 0x82,
  0xe4, 0xbc, 0x83, 0xe4, 0xbc, 0x84, 0xe4, 0xbc, 0x85, 0xe4, 0xbc, 0x86,
  0xe4, 0xbc, 0x87, 0xe4, 0xbc, 0x88, 0xe4, 0xbc, 0x8b, 0xe4, 0xbc, 0x8c,
  0xe4, 0xbc, 0x92, 0xe4, 0xbc, 0x93, 0xe4, 0xbc, 0x94, 0xe4, 0xbc, 0x95,
  0xe4, 0xbc, 0x96, 0xe4, 0xbc, 0x9c, 0xe4, 0xbc, 0x9d, 0xe4, 0xbc, 0xa1,
  0xe4, 0xbc, 0xa3, 0xe4, 0xbc, 0xa8, 0xe4, 0xbc, 0xa9, 0xe4, 0xbc, 0xac,
  0xe4, 0xbc, 0xad, 0xe4, 0xbc, 0xae, 0xe4, 0xbc, 0xb1, 0xe4, 0xbc, 0xb3,
  0xe4, 0xbc, 0xb5, 0xe4, 0xbc, 0xb7, 0xe4, 0xbc, 0xb9, 0xe4, 0xbc, 0xbb,
  0xe4, 0xbc, 0xbe, 0xe4, 0xbc, 0xbf, 0xe4, 0xbd, 0x80, 0xe4, 0xbd, 0x81,
  0xe4, 0xbd, 0x82, 0xe4, 0xbd, 0x84, 0xe4, 0xbd, 0x85, 0xe4, 0xbd, 0x87,
  0xe4, 0xbd, 0x88, 0xe4, 0xbd, 0x89, 0xe4, 0xbd, 0x8a, 0xe4, 0xbd, 0x8b,
  0xe4, 0xbd, 0x8c, 0xe4, 0xbd, 0x92, 0xe4, 0xbd, 0x94, 0xe4, 0xbd, 0x96,
  0xe4, 0xbd, 0xa1, 0xe4, 0xbd, 0xa2, 0xe4, 0xbd, 0xa6, 0xe4, 0xbd, 0xa8,
  0xe4, 0xbd, 0xaa, 0xe4, 0xbd, 0xab, 0xe4, 0xbd, 0xad, 0xe4, 0xbd, 0xae,
  0xe4, 0xbd, 0xb1, 0xe4, 0xbd, 0xb2, 0xe4, 0xbd, 0xb5, 0xe4, 0xbd, 0xb7,
  0xe4, 0xbd, 0xb8, 0xe4, 0xbd, 0xb9, 0xe4, 0xbd, 0xba, 0xe4, 0xbd, 0xbd,
  0xe4, 0xbe, 0x80, 0xe4, 0xbe, 0x81, 0xe4, 0xbe, 0x82, 0xe4, 0xbe, 0x85,
  0xe4, 0xbe, 0x86, 0xe4, 0xbe, 0x87, 0xe4, 0xbe, 0x8a, 0xe4, 0xbe, 0x8c,
  0xe4, 0xbe, 0x8e, 0xe4, 0xbe, 0x90, 0xe4, 0xbe, 0x92, 0xe4, 0xbe, 0x93,
  0xe4, 0xbe, 0x95, 0xe4, 0xbe, 0x96, 0xe4, 0xbe, 0x98, 0xe4, 0xbe, 0x99,
  0xe4, 0xbe, 0x9a, 0xe4, 0xbe, 0x9c, 0xe4, 0xbe, 0x9e, 0xe4, 0xbe, 0x9f,
  0xe4, 0xbe, 0xa1, 0xe4, 0xbe, 0xa2
};

unsigned char gbk_row_0x82_converted_to_utf8[] = {
  0xe4, 0xbe, 0xa4, 0xe4, 0xbe,
  0xab, 0xe4, 0xbe, 0xad, 0xe4, 0xbe, 0xb0, 0xe4, 0xbe, 0xb1, 0xe4, 0xbe,
  0xb2, 0xe4, 0xbe, 0xb3, 0xe4, 0xbe, 0xb4, 0xe4, 0xbe, 0xb6, 0xe4, 0xbe,
  0xb7, 0xe4, 0xbe, 0xb8, 0xe4, 0xbe, 0xb9, 0xe4, 0xbe, 0xba, 0xe4, 0xbe,
  0xbb, 0xe4, 0xbe, 0xbc, 0xe4, 0xbe, 0xbd, 0xe4, 0xbe, 0xbe, 0xe4, 0xbf,
  0x80, 0xe4, 0xbf, 0x81, 0xe4, 0xbf, 0x82, 0xe4, 0xbf, 0x86, 0xe4, 0xbf,
  0x87, 0xe4, 0xbf, 0x88, 0xe4, 0xbf, 0x89, 0xe4, 0xbf, 0x8b, 0xe4, 0xbf,
  0x8c, 0xe4, 0xbf, 0x8d, 0xe4, 0xbf, 0x92, 0xe4, 0xbf, 0x93, 0xe4, 0xbf,
  0x94, 0xe4, 0xbf, 0x95, 0xe4, 0xbf, 0x96, 0xe4, 0xbf, 0x99, 0xe4, 0xbf,
  0x9b, 0xe4, 0xbf, 0xa0, 0xe4, 0xbf, 0xa2, 0xe4, 0xbf, 0xa4, 0xe4, 0xbf,
  0xa5, 0xe4, 0xbf, 0xa7, 0xe4, 0xbf, 0xab, 0xe4, 0xbf, 0xac, 0xe4, 0xbf,
  0xb0, 0xe4, 0xbf, 0xb2, 0xe4, 0xbf, 0xb4, 0xe4, 0xbf, 0xb5, 0xe4, 0xbf,
  0xb6, 0xe4, 0xbf, 0xb7, 0xe4, 0xbf, 0xb9, 0xe4, 0xbf, 0xbb, 0xe4, 0xbf,
  0xbc, 0xe4, 0xbf, 0xbd, 0xe4, 0xbf, 0xbf, 0xe5, 0x80, 0x80, 0xe5, 0x80,
  0x81, 0xe5, 0x80, 0x82, 0xe5, 0x80, 0x83, 0xe5, 0x80, 0x84, 0xe5, 0x80,
  0x85, 0xe5, 0x80, 0x86, 0xe5, 0x80, 0x87, 0xe5, 0x80, 0x88, 0xe5, 0x80,
  0x89, 0xe5, 0x80, 0x8a, 0xe5, 0x80, 0x8b, 0xe5, 0x80, 0x8e, 0xe5, 0x80,
  0x90, 0xe5, 0x80, 0x91, 0xe5, 0x80, 0x93, 0xe5, 0x80, 0x95, 0xe5, 0x80,
  0x96, 0xe5, 0x80, 0x97, 0xe5, 0x80, 0x9b, 0xe5, 0x80, 0x9d, 0xe5, 0x80,
  0x9e, 0xe5, 0x80, 0xa0, 0xe5, 0x80, 0xa2, 0xe5, 0x80, 0xa3, 0xe5, 0x80,
  0xa4, 0xe5, 0x80, 0xa7, 0xe5, 0x80, 0xab, 0xe5, 0x80, 0xaf, 0xe5, 0x80,
  0xb0, 0xe5, 0x80, 0xb1, 0xe5, 0x80, 0xb2, 0xe5, 0x80, 0xb3, 0xe5, 0x80,
  0xb4, 0xe5, 0x80, 0xb5, 0xe5, 0x80, 0xb6, 0xe5, 0x80, 0xb7, 0xe5, 0x80,
  0xb8, 0xe5, 0x80, 0xb9, 0xe5, 0x80, 0xbb, 0xe5, 0x80, 0xbd, 0xe5, 0x80,
  0xbf, 0xe5, 0x81, 0x80, 0xe5, 0x81, 0x81, 0xe5, 0x81, 0x82, 0xe5, 0x81,
  0x84, 0xe5, 0x81, 0x85, 0xe5, 0x81, 0x86, 0xe5, 0x81, 0x89, 0xe5, 0x81,
  0x8a, 0xe5, 0x81, 0x8b, 0xe5, 0x81, 0x8d, 0xe5, 0x81, 0x90, 0xe5, 0x81,
  0x91, 0xe5, 0x81, 0x92, 0xe5, 0x81, 0x93, 0xe5, 0x81, 0x94, 0xe5, 0x81,
  0x96, 0xe5, 0x81, 0x97, 0xe5, 0x81, 0x98, 0xe5, 0x81, 0x99, 0xe5, 0x81,
  0x9b, 0xe5, 0x81, 0x9d, 0xe5, 0x81, 0x9e, 0xe5, 0x81, 0x9f, 0xe5, 0x81,
  0xa0, 0xe5, 0x81, 0xa1, 0xe5, 0x81, 0xa2, 0xe5, 0x81, 0xa3, 0xe5, 0x81,
  0xa4, 0xe5, 0x81, 0xa6, 0xe5, 0x81, 0xa7, 0xe5, 0x81, 0xa8, 0xe5, 0x81,
  0xa9, 0xe5, 0x81, 0xaa, 0xe5, 0x81, 0xab, 0xe5, 0x81, 0xad, 0xe5, 0x81,
  0xae, 0xe5, 0x81, 0xaf, 0xe5, 0x81, 0xb0, 0xe5, 0x81, 0xb1, 0xe5, 0x81,
  0xb2, 0xe5, 0x81, 0xb3, 0xe5, 0x81, 0xb4, 0xe5, 0x81, 0xb5, 0xe5, 0x81,
  0xb8, 0xe5, 0x81, 0xb9, 0xe5, 0x81, 0xba, 0xe5, 0x81, 0xbc, 0xe5, 0x81,
  0xbd, 0xe5, 0x82, 0x81, 0xe5, 0x82, 0x82, 0xe5, 0x82, 0x83, 0xe5, 0x82,
  0x84, 0xe5, 0x82, 0x86, 0xe5, 0x82, 0x87, 0xe5, 0x82, 0x89, 0xe5, 0x82,
  0x8a, 0xe5, 0x82, 0x8b, 0xe5, 0x82, 0x8c, 0xe5, 0x82, 0x8e, 0xe5, 0x82,
  0x8f, 0xe5, 0x82, 0x90, 0xe5, 0x82, 0x91, 0xe5, 0x82, 0x92, 0xe5, 0x82,
  0x93, 0xe5, 0x82, 0x94, 0xe5, 0x82, 0x95, 0xe5, 0x82, 0x96, 0xe5, 0x82,
  0x97, 0xe5, 0x82, 0x98, 0xe5, 0x82, 0x99, 0xe5, 0x82, 0x9a, 0xe5, 0x82,
  0x9b, 0xe5, 0x82, 0x9c, 0xe5, 0x82, 0x9d, 0xe5, 0x82, 0x9e, 0xe5, 0x82,
  0x9f, 0xe5, 0x82, 0xa0, 0xe5, 0x82, 0xa1, 0xe5, 0x82, 0xa2, 0xe5, 0x82,
  0xa4, 0xe5, 0x82, 0xa6, 0xe5, 0x82, 0xaa, 0xe5, 0x82, 0xab, 0xe5, 0x82,
  0xad, 0xe5, 0x82, 0xae, 0xe5, 0x82, 0xaf, 0xe5, 0x82, 0xb0, 0xe5, 0x82,
  0xb1, 0xe5, 0x82, 0xb3, 0xe5, 0x82, 0xb4, 0xe5, 0x82, 0xb5, 0xe5, 0x82,
  0xb6, 0xe5, 0x82, 0xb7, 0xe5, 0x82, 0xb8, 0xe5, 0x82, 0xb9, 0xe5, 0x82,
  0xbc
};

unsigned char gbk_row_0x83_converted_to_utf8[] = {
  0xe5, 0x82, 0xbd, 0xe5, 0x82, 0xbe, 0xe5, 0x82, 0xbf, 0xe5,
  0x83, 0x80, 0xe5, 0x83, 0x81, 0xe5, 0x83, 0x82, 0xe5, 0x83, 0x83, 0xe5,
  0x83, 0x84, 0xe5, 0x83, 0x85, 0xe5, 0x83, 0x86, 0xe5, 0x83, 0x87, 0xe5,
  0x83, 0x88, 0xe5, 0x83, 0x89, 0xe5, 0x83, 0x8a, 0xe5, 0x83, 0x8b, 0xe5,
  0x83, 0x8c, 0xe5, 0x83, 0x8d, 0xe5, 0x83, 0x8e, 0xe5, 0x83, 0x90, 0xe5,
  0x83, 0x91, 0xe5, 0x83, 0x92, 0xe5, 0x83, 0x93, 0xe5, 0x83, 0x94, 0xe5,
  0x83, 0x95, 0xe5, 0x83, 0x97, 0xe5, 0x83, 0x98, 0xe5, 0x83, 0x99, 0xe5,
  0x83, 0x9b, 0xe5, 0x83, 0x9c, 0xe5, 0x83, 0x9d, 0xe5, 0x83, 0x9e, 0xe5,
  0x83, 0x9f, 0xe5, 0x83, 0xa0, 0xe5, 0x83, 0xa1, 0xe5, 0x83, 0xa2, 0xe5,
  0x83, 0xa3, 0xe5, 0x83, 0xa4, 0xe5, 0x83, 0xa5, 0xe5, 0x83, 0xa8, 0xe5,
  0x83, 0xa9, 0xe5, 0x83, 0xaa, 0xe5, 0x83, 0xab, 0xe5, 0x83, 0xaf, 0xe5,
  0x83, 0xb0, 0xe5, 0x83, 0xb1, 0xe5, 0x83, 0xb2, 0xe5, 0x83, 0xb4, 0xe5,
  0x83, 0xb6, 0xe5, 0x83, 0xb7, 0xe5, 0x83, 0xb8, 0xe5, 0x83, 0xb9, 0xe5,
  0x83, 0xba, 0xe5, 0x83, 0xbc, 0xe5, 0x83, 0xbd, 0xe5, 0x83, 0xbe, 0xe5,
  0x83, 0xbf, 0xe5, 0x84, 0x80, 0xe5, 0x84, 0x81, 0xe5, 0x84, 0x82, 0xe5,
  0x84, 0x83, 0xe5, 0x84, 0x84, 0xe5, 0x84, 0x85, 0xe5, 0x84, 0x88, 0xe5,
  0x84, 0x89, 0xe5, 0x84, 0x8a, 0xe5, 0x84, 0x8c, 0xe5, 0x84, 0x8d, 0xe5,
  0x84, 0x8e, 0xe5, 0x84, 0x8f, 0xe5, 0x84, 0x90, 0xe5, 0x84, 0x91, 0xe5,
  0x84, 0x93, 0xe5, 0x84, 0x94, 0xe5, 0x84, 0x95, 0xe5, 0x84, 0x96, 0xe5,
  0x84, 0x97, 0xe5, 0x84, 0x98, 0xe5, 0x84, 0x99, 0xe5, 0x84, 0x9a, 0xe5,
  0x84, 0x9b, 0xe5, 0x84, 0x9c, 0xe5, 0x84, 0x9d, 0xe5, 0x84, 0x9e, 0xe5,
  0x84, 0x9f, 0xe5, 0x84, 0xa0, 0xe5, 0x84, 0xa2, 0xe5, 0x84, 0xa3, 0xe5,
  0x84, 0xa4, 0xe5, 0x84, 0xa5, 0xe5, 0x84, 0xa6, 0xe5, 0x84, 0xa7, 0xe5,
  0x84, 0xa8, 0xe5, 0x84, 0xa9, 0xe5, 0x84, 0xaa, 0xe5, 0x84, 0xab, 0xe5,
  0x84, 0xac, 0xe5, 0x84, 0xad, 0xe5, 0x84, 0xae, 0xe5, 0x84, 0xaf, 0xe5,
  0x84, 0xb0, 0xe5, 0x84, 0xb1, 0xe5, 0x84, 0xb2, 0xe5, 0x84, 0xb3, 0xe5,
  0x84, 0xb4, 0xe5, 0x84, 0xb5, 0xe5, 0x84, 0xb6, 0xe5, 0x84, 0xb7, 0xe5,
  0x84, 0xb8, 0xe5, 0x84, 0xb9, 0xe5, 0x84, 0xba, 0xe5, 0x84, 0xbb, 0xe5,
  0x84, 0xbc, 0xe5, 0x84, 0xbd, 0xe5, 0x84, 0xbe, 0xe5, 0x85, 0x82, 0xe5,
  0x85, 0x87, 0xe5, 0x85, 0x8a, 0xe5, 0x85, 0x8c, 0xe5, 0x85, 0x8e, 0xe5,
  0x85, 0x8f, 0xe5, 0x85, 0x90, 0xe5, 0x85, 0x92, 0xe5, 0x85, 0x93, 0xe5,
  0x85, 0x97, 0xe5, 0x85, 0x98, 0xe5, 0x85, 0x99, 0xe5, 0x85, 0x9b, 0xe5,
  0x85, 0x9d, 0xe5, 0x85, 0x9e, 0xe5, 0x85, 0x9f, 0xe5, 0x85, 0xa0, 0xe5,
  0x85, 0xa1, 0xe5, 0x85, 0xa3, 0xe5, 0x85, 0xa4, 0xe5, 0x85, 0xa6, 0xe5,
  0x85, 0xa7, 0xe5, 0x85, 0xa9, 0xe5, 0x85, 0xaa, 0xe5, 0x85, 0xaf, 0xe5,
  0x85, 0xb2, 0xe5, 0x85, 0xba, 0xe5, 0x85, 0xbe, 0xe5, 0x85, 0xbf, 0xe5,
  0x86, 0x83, 0xe5, 0x86, 0x84, 0xe5, 0x86, 0x86, 0xe5, 0x86, 0x87, 0xe5,
  0x86, 0x8a, 0xe5, 0x86, 0x8b, 0xe5, 0x86, 0x8e, 0xe5, 0x86, 0x8f, 0xe5,
  0x86, 0x90, 0xe5, 0x86, 0x91, 0xe5, 0x86, 0x93, 0xe5, 0x86, 0x94, 0xe5,
  0x86, 0x98, 0xe5, 0x86, 0x9a, 0xe5, 0x86, 0x9d, 0xe5, 0x86, 0x9e, 0xe5,
  0x86, 0x9f, 0xe5, 0x86, 0xa1, 0xe5, 0x86, 0xa3, 0xe5, 0x86, 0xa6, 0xe5,
  0x86, 0xa7, 0xe5, 0x86, 0xa8, 0xe5, 0x86, 0xa9, 0xe5, 0x86, 0xaa, 0xe5,
  0x86, 0xad, 0xe5, 0x86, 0xae, 0xe5, 0x86, 0xb4, 0xe5, 0x86, 0xb8, 0xe5,
  0x86, 0xb9, 0xe5, 0x86, 0xba, 0xe5, 0x86, 0xbe, 0xe5, 0x86, 0xbf, 0xe5,
  0x87, 0x81, 0xe5, 0x87, 0x82, 0xe5, 0x87, 0x83, 0xe5, 0x87, 0x85, 0xe5,
  0x87, 0x88, 0xe5, 0x87, 0x8a, 0xe5, 0x87, 0x8d, 0xe5, 0x87, 0x8e, 0xe5,
  0x87, 0x90, 0xe5, 0x87, 0x92, 0xe5, 0x87, 0x93, 0xe5, 0x87, 0x94, 0xe5,
  0x87, 0x95, 0xe5, 0x87, 0x96, 0xe5, 0x87, 0x97
};

unsigned char gbk_row_0x84_converted_to_utf8[] = {
  0xe5, 0x87, 0x98,
  0xe5, 0x87, 0x99, 0xe5, 0x87, 0x9a, 0xe5, 0x87, 0x9c, 0xe5, 0x87, 0x9e,
  0xe5, 0x87, 0x9f, 0xe5, 0x87, 0xa2, 0xe5, 0x87, 0xa3, 0xe5, 0x87, 0xa5,
  0xe5, 0x87, 0xa6, 0xe5, 0x87, 0xa7, 0xe5, 0x87, 0xa8, 0xe5, 0x87, 0xa9,
  0xe5, 0x87, 0xaa, 0xe5, 0x87, 0xac, 0xe5, 0x87, 0xae, 0xe5, 0x87, 0xb1,
  0xe5, 0x87, 0xb2, 0xe5, 0x87, 0xb4, 0xe5, 0x87, 0xb7, 0xe5, 0x87, 0xbe,
  0xe5, 0x88, 0x84, 0xe5, 0x88, 0x85, 0xe5, 0x88, 0x89, 0xe5, 0x88, 0x8b,
  0xe5, 0x88, 0x8c, 0xe5, 0x88, 0x8f, 0xe5, 0x88, 0x90, 0xe5, 0x88, 0x93,
  0xe5, 0x88, 0x94, 0xe5, 0x88, 0x95, 0xe5, 0x88, 0x9c, 0xe5, 0x88, 0x9e,
  0xe5, 0x88, 0x9f, 0xe5, 0x88, 0xa1, 0xe5, 0x88, 0xa2, 0xe5, 0x88, 0xa3,
  0xe5, 0x88, 0xa5, 0xe5, 0x88, 0xa6, 0xe5, 0x88, 0xa7, 0xe5, 0x88, 0xaa,
  0xe5, 0x88, 0xac, 0xe5, 0x88, 0xaf, 0xe5, 0x88, 0xb1, 0xe5, 0x88, 0xb2,
  0xe5, 0x88, 0xb4, 0xe5, 0x88, 0xb5, 0xe5, 0x88, 0xbc, 0xe5, 0x88, 0xbe,
  0xe5, 0x89, 0x84, 0xe5, 0x89, 0x85, 0xe5, 0x89, 0x86, 0xe5, 0x89, 0x87,
  0xe5, 0x89, 0x88, 0xe5, 0x89, 0x89, 0xe5, 0x89, 0x8b, 0xe5, 0x89, 0x8e,
  0xe5, 0x89, 0x8f, 0xe5, 0x89, 0x92, 0xe5, 0x89, 0x93, 0xe5, 0x89, 0x95,
  0xe5, 0x89, 0x97, 0xe5, 0x89, 0x98, 0xe5, 0x89, 0x99, 0xe5, 0x89, 0x9a,
  0xe5, 0x89, 0x9b, 0xe5, 0x89, 0x9d, 0xe5, 0x89, 0x9f, 0xe5, 0x89, 0xa0,
  0xe5, 0x89, 0xa2, 0xe5, 0x89, 0xa3, 0xe5, 0x89, 0xa4, 0xe5, 0x89, 0xa6,
  0xe5, 0x89, 0xa8, 0xe5, 0x89, 0xab, 0xe5, 0x89, 0xac, 0xe5, 0x89, 0xad,
  0xe5, 0x89, 0xae, 0xe5, 0x89, 0xb0, 0xe5, 0x89, 0xb1, 0xe5, 0x89, 0xb3,
  0xe5, 0x89, 0xb4, 0xe5, 0x89, 0xb5, 0xe5, 0x89, 0xb6, 0xe5, 0x89, 0xb7,
  0xe5, 0x89, 0xb8, 0xe5, 0x89, 0xb9, 0xe5, 0x89, 0xba, 0xe5, 0x89, 0xbb,
  0xe5, 0x89, 0xbc, 0xe5, 0x89, 0xbe, 0xe5, 0x8a, 0x80, 0xe5, 0x8a, 0x83,
  0xe5, 0x8a, 0x84, 0xe5, 0x8a, 0x85, 0xe5, 0x8a, 0x86, 0xe5, 0x8a, 0x87,
  0xe5, 0x8a, 0x89, 0xe5, 0x8a, 0x8a, 0xe5, 0x8a, 0x8b, 0xe5, 0x8a, 0x8c,
  0xe5, 0x8a, 0x8d, 0xe5, 0x8a, 0x8e, 0xe5, 0x8a, 0x8f, 0xe5, 0x8a, 0x91,
  0xe5, 0x8a, 0x92, 0xe5, 0x8a, 0x94, 0xe5, 0x8a, 0x95, 0xe5, 0x8a, 0x96,
  0xe5, 0x8a, 0x97, 0xe5, 0x8a, 0x98, 0xe5, 0x8a, 0x99, 0xe5, 0x8a, 0x9a,
  0xe5, 0x8a, 0x9c, 0xe5, 0x8a, 0xa4, 0xe5, 0x8a, 0xa5, 0xe5, 0x8a, 0xa6,
  0xe5, 0x8a, 0xa7, 0xe5, 0x8a, 0xae, 0xe5, 0x8a, 0xaf, 0xe5, 0x8a, 0xb0,
  0xe5, 0x8a, 0xb4, 0xe5, 0x8a, 0xb5, 0xe5, 0x8a, 0xb6, 0xe5, 0x8a, 0xb7,
  0xe5, 0x8a, 0xb8, 0xe5, 0x8a, 0xb9, 0xe5, 0x8a, 0xba, 0xe5, 0x8a, 0xbb,
  0xe5, 0x8a, 0xbc, 0xe5, 0x8a, 0xbd, 0xe5, 0x8b, 0x80, 0xe5, 0x8b, 0x81,
  0xe5, 0x8b, 0x82, 0xe5, 0x8b, 0x84, 0xe5, 0x8b, 0x85, 0xe5, 0x8b, 0x86,
  0xe5, 0x8b, 0x88, 0xe5, 0x8b, 0x8a, 0xe5, 0x8b, 0x8c, 0xe5, 0x8b, 0x8d,
  0xe5, 0x8b, 0x8e, 0xe5, 0x8b, 0x8f, 0xe5, 0x8b, 0x91, 0xe5, 0x8b, 0x93,
  0xe5, 0x8b, 0x94, 0xe5, 0x8b, 0x95, 0xe5, 0x8b, 0x97, 0xe5, 0x8b, 0x99,
  0xe5, 0x8b, 0x9a, 0xe5, 0x8b, 0x9b, 0xe5, 0x8b, 0x9c, 0xe5, 0x8b, 0x9d,
  0xe5, 0x8b, 0x9e, 0xe5, 0x8b, 0xa0, 0xe5, 0x8b, 0xa1, 0xe5, 0x8b, 0xa2,
  0xe5, 0x8b, 0xa3, 0xe5, 0x8b, 0xa5, 0xe5, 0x8b, 0xa6, 0xe5, 0x8b, 0xa7,
  0xe5, 0x8b, 0xa8, 0xe5, 0x8b, 0xa9, 0xe5, 0x8b, 0xaa, 0xe5, 0x8b, 0xab,
  0xe5, 0x8b, 0xac, 0xe5, 0x8b, 0xad, 0xe5, 0x8b, 0xae, 0xe5, 0x8b, 0xaf,
  0xe5, 0x8b, 0xb1, 0xe5, 0x8b, 0xb2, 0xe5, 0x8b, 0xb3, 0xe5, 0x8b, 0xb4,
  0xe5, 0x8b, 0xb5, 0xe5, 0x8b, 0xb6, 0xe5, 0x8b, 0xb7, 0xe5, 0x8b, 0xb8,
  0xe5, 0x8b, 0xbb, 0xe5, 0x8b, 0xbc, 0xe5, 0x8b, 0xbd, 0xe5, 0x8c, 0x81,
  0xe5, 0x8c, 0x82, 0xe5, 0x8c, 0x83, 0xe5, 0x8c, 0x84, 0xe5, 0x8c, 0x87,
  0xe5, 0x8c, 0x89, 0xe5, 0x8c, 0x8a, 0xe5, 0x8c, 0x8b, 0xe5, 0x8c, 0x8c,
  0xe5, 0x8c, 0x8e
};

unsigned char gbk_row_0x85_converted_to_utf8[] = {
  0xe5, 0x8c, 0x91, 0xe5, 0x8c, 0x92, 0xe5, 0x8c,
  0x93, 0xe5, 0x8c, 0x94, 0xe5, 0x8c, 0x98, 0xe5, 0x8c, 0x9b, 0xe5, 0x8c,
  0x9c, 0xe5, 0x8c, 0x9e, 0xe5, 0x8c, 0x9f, 0xe5, 0x8c, 0xa2, 0xe5, 0x8c,
  0xa4, 0xe5, 0x8c, 0xa5, 0xe5, 0x8c, 0xa7, 0xe5, 0x8c, 0xa8, 0xe5, 0x8c,
  0xa9, 0xe5, 0x8c, 0xab, 0xe5, 0x8c, 0xac, 0xe5, 0x8c, 0xad, 0xe5, 0x8c,
  0xaf, 0xe5, 0x8c, 0xb0, 0xe5, 0x8c, 0xb1, 0xe5, 0x8c, 0xb2, 0xe5, 0x8c,
  0xb3, 0xe5, 0x8c, 0xb4, 0xe5, 0x8c, 0xb5, 0xe5, 0x8c, 0xb6, 0xe5, 0x8c,
  0xb7, 0xe5, 0x8c, 0xb8, 0xe5, 0x8c, 0xbc, 0xe5, 0x8c, 0xbd, 0xe5, 0x8d,
  0x80, 0xe5, 0x8d, 0x82, 0xe5, 0x8d, 0x84, 0xe5, 0x8d, 0x86, 0xe5, 0x8d,
  0x8b, 0xe5, 0x8d, 0x8c, 0xe5, 0x8d, 0x8d, 0xe5, 0x8d, 0x90, 0xe5, 0x8d,
  0x94, 0xe5, 0x8d, 0x98, 0xe5, 0x8d, 0x99, 0xe5, 0x8d, 0x9b, 0xe5, 0x8d,
  0x9d, 0xe5, 0x8d, 0xa5, 0xe5, 0x8d, 0xa8, 0xe5, 0x8d, 0xaa, 0xe5, 0x8d,
  0xac, 0xe5, 0x8d, 0xad, 0xe5, 0x8d, 0xb2, 0xe5, 0x8d, 0xb6, 0xe5, 0x8d,
  0xb9, 0xe5, 0x8d, 0xbb, 0xe5, 0x8d, 0xbc, 0xe5, 0x8d, 0xbd, 0xe5, 0x8d,
  0xbe, 0xe5, 0x8e, 0x80, 0xe5, 0x8e, 0x81, 0xe5, 0x8e, 0x83, 0xe5, 0x8e,
  0x87, 0xe5, 0x8e, 0x88, 0xe5, 0x8e, 0x8a, 0xe5, 0x8e, 0x8e, 0xe5, 0x8e,
  0x8f, 0xe5, 0x8e, 0x90, 0xe5, 0x8e, 0x91, 0xe5, 0x8e, 0x92, 0xe5, 0x8e,
  0x93, 0xe5, 0x8e, 0x94, 0xe5, 0x8e, 0x96, 0xe5, 0x8e, 0x97, 0xe5, 0x8e,
  0x99, 0xe5, 0x8e, 0x9b, 0xe5, 0x8e, 0x9c, 0xe5, 0x8e, 0x9e, 0xe5, 0x8e,
  0xa0, 0xe5, 0x8e, 0xa1, 0xe5, 0x8e, 0xa4, 0xe5, 0x8e, 0xa7, 0xe5, 0x8e,
  0xaa, 0xe5, 0x8e, 0xab, 0xe5, 0x8e, 0xac, 0xe5, 0x8e, 0xad, 0xe5, 0x8e,
  0xaf, 0xe5, 0x8e, 0xb0, 0xe5, 0x8e, 0xb1, 0xe5, 0x8e, 0xb2, 0xe5, 0x8e,
  0xb3, 0xe5, 0x8e, 0xb4, 0xe5, 0x8e, 0xb5, 0xe5, 0x8e, 0xb7, 0xe5, 0x8e,
  0xb8, 0xe5, 0x8e, 0xb9, 0xe5, 0x8e, 0xba, 0xe5, 0x8e, 0xbc, 0xe5, 0x8e,
  0xbd, 0xe5, 0x8e, 0xbe, 0xe5, 0x8f, 0x80, 0xe5, 0x8f, 0x83, 0xe5, 0x8f,
  0x84, 0xe5, 0x8f, 0x85, 0xe5, 0x8f, 0x86, 0xe5, 0x8f, 0x87, 0xe5, 0x8f,
  0x8e, 0xe5, 0x8f, 0x8f, 0xe5, 0x8f, 0x90, 0xe5, 0x8f, 0x92, 0xe5, 0x8f,
  0x93, 0xe5, 0x8f, 0x95, 0xe5, 0x8f, 0x9a, 0xe5, 0x8f, 0x9c, 0xe5, 0x8f,
  0x9d, 0xe5, 0x8f, 0x9e, 0xe5, 0x8f, 0xa1, 0xe5, 0x8f, 0xa2, 0xe5, 0x8f,
  0xa7, 0xe5, 0x8f, 0xb4, 0xe5, 0x8f, 0xba, 0xe5, 0x8f, 0xbe, 0xe5, 0x8f,
  0xbf, 0xe5, 0x90, 0x80, 0xe5, 0x90, 0x82, 0xe5, 0x90, 0x85, 0xe5, 0x90,
  0x87, 0xe5, 0x90, 0x8b, 0xe5, 0x90, 0x94, 0xe5, 0x90, 0x98, 0xe5, 0x90,
  0x99, 0xe5, 0x90, 0x9a, 0xe5, 0x90, 0x9c, 0xe5, 0x90, 0xa2, 0xe5, 0x90,
  0xa4, 0xe5, 0x90, 0xa5, 0xe5, 0x90, 0xaa, 0xe5, 0x90, 0xb0, 0xe5, 0x90,
  0xb3, 0xe5, 0x90, 0xb6, 0xe5, 0x90, 0xb7, 0xe5, 0x90, 0xba, 0xe5, 0x90,
  0xbd, 0xe5, 0x90, 0xbf, 0xe5, 0x91, 0x81, 0xe5, 0x91, 0x82, 0xe5, 0x91,
  0x84, 0xe5, 0x91, 0x85, 0xe5, 0x91, 0x87, 0xe5, 0x91, 0x89, 0xe5, 0x91,
  0x8c, 0xe5, 0x91, 0x8d, 0xe5, 0x91, 0x8e, 0xe5, 0x91, 0x8f, 0xe5, 0x91,
  0x91, 0xe5, 0x91, 0x9a, 0xe5, 0x91, 0x9d, 0xe5, 0x91, 0x9e, 0xe5, 0x91,
  0x9f, 0xe5, 0x91, 0xa0, 0xe5, 0x91, 0xa1, 0xe5, 0x91, 0xa3, 0xe5, 0x91,
  0xa5, 0xe5, 0x91, 0xa7, 0xe5, 0x91, 0xa9, 0xe5, 0x91, 0xaa, 0xe5, 0x91,
  0xab, 0xe5, 0x91, 0xac, 0xe5, 0x91, 0xad, 0xe5, 0x91, 0xae, 0xe5, 0x91,
  0xaf, 0xe5, 0x91, 0xb0, 0xe5, 0x91, 0xb4, 0xe5, 0x91, 0xb9, 0xe5, 0x91,
  0xba, 0xe5, 0x91, 0xbe, 0xe5, 0x91, 0xbf, 0xe5, 0x92, 0x81, 0xe5, 0x92,
  0x83, 0xe5, 0x92, 0x85, 0xe5, 0x92, 0x87, 0xe5, 0x92, 0x88, 0xe5, 0x92,
  0x89, 0xe5, 0x92, 0x8a, 0xe5, 0x92, 0x8d, 0xe5, 0x92, 0x91, 0xe5, 0x92,
  0x93, 0xe5, 0x92, 0x97, 0xe5, 0x92, 0x98, 0xe5, 0x92, 0x9c, 0xe5, 0x92,
  0x9e, 0xe5, 0x92, 0x9f, 0xe5, 0x92, 0xa0, 0xe5, 0x92, 0xa1

};

unsigned char gbk_row_0x86_converted_to_utf8[] = {
  0xe5,
  0x92, 0xa2, 0xe5, 0x92, 0xa5, 0xe5, 0x92, 0xae, 0xe5, 0x92, 0xb0, 0xe5,
  0x92, 0xb2, 0xe5, 0x92, 0xb5, 0xe5, 0x92, 0xb6, 0xe5, 0x92, 0xb7, 0xe5,
  0x92, 0xb9, 0xe5, 0x92, 0xba, 0xe5, 0x92, 0xbc, 0xe5, 0x92, 0xbe, 0xe5,
  0x93, 0x83, 0xe5, 0x93, 0x85, 0xe5, 0x93, 0x8a, 0xe5, 0x93, 0x8b, 0xe5,
  0x93, 0x96, 0xe5, 0x93, 0x98, 0xe5, 0x93, 0x9b, 0xe5, 0x93, 0xa0, 0xe5,
  0x93, 0xa1, 0xe5, 0x93, 0xa2, 0xe5, 0x93, 0xa3, 0xe5, 0x93, 0xa4, 0xe5,
  0x93, 0xab, 0xe5, 0x93, 0xac, 0xe5, 0x93, 0xaf, 0xe5, 0x93, 0xb0, 0xe5,
  0x93, 0xb1, 0xe5, 0x93, 0xb4, 0xe5, 0x93, 0xb5, 0xe5, 0x93, 0xb6, 0xe5,
  0x93, 0xb7, 0xe5, 0x93, 0xb8, 0xe5, 0x93, 0xb9, 0xe5, 0x93, 0xbb, 0xe5,
  0x93, 0xbe, 0xe5, 0x94, 0x80, 0xe5, 0x94, 0x82, 0xe5, 0x94, 0x83, 0xe5,
  0x94, 0x84, 0xe5, 0x94, 0x85, 0xe5, 0x94, 0x88, 0xe5, 0x94, 0x8a, 0xe5,
  0x94, 0x8b, 0xe5, 0x94, 0x8c, 0xe5, 0x94, 0x8d, 0xe5, 0x94, 0x8e, 0xe5,
  0x94, 0x92, 0xe5, 0x94, 0x93, 0xe5, 0x94, 0x95, 0xe5, 0x94, 0x96, 0xe5,
  0x94, 0x97, 0xe5, 0x94, 0x98, 0xe5, 0x94, 0x99, 0xe5, 0x94, 0x9a, 0xe5,
  0x94, 0x9c, 0xe5, 0x94, 0x9d, 0xe5, 0x94, 0x9e, 0xe5, 0x94, 0x9f, 0xe5,
  0x94, 0xa1, 0xe5, 0x94, 0xa5, 0xe5, 0x94, 0xa6, 0xe5, 0x94, 0xa8, 0xe5,
  0x94, 0xa9, 0xe5, 0x94, 0xab, 0xe5, 0x94, 0xad, 0xe5, 0x94, 0xb2, 0xe5,
  0x94, 0xb4, 0xe5, 0x94, 0xb5, 0xe5, 0x94, 0xb6, 0xe5, 0x94, 0xb8, 0xe5,
  0x94, 0xb9, 0xe5, 0x94, 0xba, 0xe5, 0x94, 0xbb, 0xe5, 0x94, 0xbd, 0xe5,
  0x95, 0x80, 0xe5, 0x95, 0x82, 0xe5, 0x95, 0x85, 0xe5, 0x95, 0x87, 0xe5,
  0x95, 0x88, 0xe5, 0x95, 0x8b, 0xe5, 0x95, 0x8c, 0xe5, 0x95, 0x8d, 0xe5,
  0x95, 0x8e, 0xe5, 0x95, 0x8f, 0xe5, 0x95, 0x91, 0xe5, 0x95, 0x92, 0xe5,
  0x95, 0x93, 0xe5, 0x95, 0x94, 0xe5, 0x95, 0x97, 0xe5, 0x95, 0x98, 0xe5,
  0x95, 0x99, 0xe5, 0x95, 0x9a, 0xe5, 0x95, 0x9b, 0xe5, 0x95, 0x9d, 0xe5,
  0x95, 0x9e, 0xe5, 0x95, 0x9f, 0xe5, 0x95, 0xa0, 0xe5, 0x95, 0xa2, 0xe5,
  0x95, 0xa3, 0xe5, 0x95, 0xa8, 0xe5, 0x95, 0xa9, 0xe5, 0x95, 0xab, 0xe5,
  0x95, 0xaf, 0xe5, 0x95, 0xb0, 0xe5, 0x95, 0xb1, 0xe5, 0x95, 0xb2, 0xe5,
  0x95, 0xb3, 0xe5, 0x95, 0xb4, 0xe5, 0x95, 0xb9, 0xe5, 0x95, 0xba, 0xe5,
  0x95, 0xbd, 0xe5, 0x95, 0xbf, 0xe5, 0x96, 0x85, 0xe5, 0x96, 0x86, 0xe5,
  0x96, 0x8c, 0xe5, 0x96, 0x8d, 0xe5, 0x96, 0x8e, 0xe5, 0x96, 0x90, 0xe5,
  0x96, 0x92, 0xe5, 0x96, 0x93, 0xe5, 0x96, 0x95, 0xe5, 0x96, 0x96, 0xe5,
  0x96, 0x97, 0xe5, 0x96, 0x9a, 0xe5, 0x96, 0x9b, 0xe5, 0x96, 0x9e, 0xe5,
  0x96, 0xa0, 0xe5, 0x96, 0xa1, 0xe5, 0x96, 0xa2, 0xe5, 0x96, 0xa3, 0xe5,
  0x96, 0xa4, 0xe5, 0x96, 0xa5, 0xe5, 0x96, 0xa6, 0xe5, 0x96, 0xa8, 0xe5,
  0x96, 0xa9, 0xe5, 0x96, 0xaa, 0xe5, 0x96, 0xab, 0xe5, 0x96, 0xac, 0xe5,
  0x96, 0xad, 0xe5, 0x96, 0xae, 0xe5, 0x96, 0xaf, 0xe5, 0x96, 0xb0, 0xe5,
  0x96, 0xb2, 0xe5, 0x96, 0xb4, 0xe5, 0x96, 0xb6, 0xe5, 0x96, 0xb8, 0xe5,
  0x96, 0xba, 0xe5, 0x96, 0xbc, 0xe5, 0x96, 0xbf, 0xe5, 0x97, 0x80, 0xe5,
  0x97, 0x81, 0xe5, 0x97, 0x82, 0xe5, 0x97, 0x83, 0xe5, 0x97, 0x86, 0xe5,
  0x97, 0x87, 0xe5, 0x97, 0x88, 0xe5, 0x97, 0x8a, 0xe5, 0x97, 0x8b, 0xe5,
  0x97, 0x8e, 0xe5, 0x97, 0x8f, 0xe5, 0x97, 0x90, 0xe5, 0x97, 0x95, 0xe5,
  0x97, 0x97, 0xe5, 0x97, 0x98, 0xe5, 0x97, 0x99, 0xe5, 0x97, 0x9a, 0xe5,
  0x97, 0x9b, 0xe5, 0x97, 0x9e, 0xe5, 0x97, 0xa0, 0xe5, 0x97, 0xa2, 0xe5,
  0x97, 0xa7, 0xe5, 0x97, 0xa9, 0xe5, 0x97, 0xad, 0xe5, 0x97, 0xae, 0xe5,
  0x97, 0xb0, 0xe5, 0x97, 0xb1, 0xe5, 0x97, 0xb4, 0xe5, 0x97, 0xb6, 0xe5,
  0x97, 0xb8, 0xe5, 0x97, 0xb9, 0xe5, 0x97, 0xba, 0xe5, 0x97, 0xbb, 0xe5,
  0x97, 0xbc, 0xe5, 0x97, 0xbf, 0xe5, 0x98, 0x82, 0xe5, 0x98, 0x83, 0xe5,
  0x98, 0x84, 0xe5, 0x98, 0x85
};

unsigned char gbk_row_0x87_converted_to_utf8[] = {
  0xe5, 0x98, 0x86, 0xe5, 0x98, 0x87,
  0xe5, 0x98, 0x8a, 0xe5, 0x98, 0x8b, 0xe5, 0x98, 0x8d, 0xe5, 0x98, 0x90,
  0xe5, 0x98, 0x91, 0xe5, 0x98, 0x92, 0xe5, 0x98, 0x93, 0xe5, 0x98, 0x94,
  0xe5, 0x98, 0x95, 0xe5, 0x98, 0x96, 0xe5, 0x98, 0x97, 0xe5, 0x98, 0x99,
  0xe5, 0x98, 0x9a, 0xe5, 0x98, 0x9c, 0xe5, 0x98, 0x9d, 0xe5, 0x98, 0xa0,
  0xe5, 0x98, 0xa1, 0xe5, 0x98, 0xa2, 0xe5, 0x98, 0xa5, 0xe5, 0x98, 0xa6,
  0xe5, 0x98, 0xa8, 0xe5, 0x98, 0xa9, 0xe5, 0x98, 0xaa, 0xe5, 0x98, 0xab,
  0xe5, 0x98, 0xae, 0xe5, 0x98, 0xaf, 0xe5, 0x98, 0xb0, 0xe5, 0x98, 0xb3,
  0xe5, 0x98, 0xb5, 0xe5, 0x98, 0xb7, 0xe5, 0x98, 0xb8, 0xe5, 0x98, 0xba,
  0xe5, 0x98, 0xbc, 0xe5, 0x98, 0xbd, 0xe5, 0x98, 0xbe, 0xe5, 0x99, 0x80,
  0xe5, 0x99, 0x81, 0xe5, 0x99, 0x82, 0xe5, 0x99, 0x83, 0xe5, 0x99, 0x84,
  0xe5, 0x99, 0x85, 0xe5, 0x99, 0x86, 0xe5, 0x99, 0x87, 0xe5, 0x99, 0x88,
  0xe5, 0x99, 0x89, 0xe5, 0x99, 0x8a, 0xe5, 0x99, 0x8b, 0xe5, 0x99, 0x8f,
  0xe5, 0x99, 0x90, 0xe5, 0x99, 0x91, 0xe5, 0x99, 0x92, 0xe5, 0x99, 0x93,
  0xe5, 0x99, 0x95, 0xe5, 0x99, 0x96, 0xe5, 0x99, 0x9a, 0xe5, 0x99, 0x9b,
  0xe5, 0x99, 0x9d, 0xe5, 0x99, 0x9e, 0xe5, 0x99, 0x9f, 0xe5, 0x99, 0xa0,
  0xe5, 0x99, 0xa1, 0xe5, 0x99, 0xa3, 0xe5, 0x99, 0xa5, 0xe5, 0x99, 0xa6,
  0xe5, 0x99, 0xa7, 0xe5, 0x99, 0xad, 0xe5, 0x99, 0xae, 0xe5, 0x99, 0xaf,
  0xe5, 0x99, 0xb0, 0xe5, 0x99, 0xb2, 0xe5, 0x99, 0xb3, 0xe5, 0x99, 0xb4,
  0xe5, 0x99, 0xb5, 0xe5, 0x99, 0xb7, 0xe5, 0x99, 0xb8, 0xe5, 0x99, 0xb9,
  0xe5, 0x99, 0xba, 0xe5, 0x99, 0xbd, 0xe5, 0x99, 0xbe, 0xe5, 0x99, 0xbf,
  0xe5, 0x9a, 0x80, 0xe5, 0x9a, 0x81, 0xe5, 0x9a, 0x82, 0xe5, 0x9a, 0x83,
  0xe5, 0x9a, 0x84, 0xe5, 0x9a, 0x87, 0xe5, 0x9a, 0x88, 0xe5, 0x9a, 0x89,
  0xe5, 0x9a, 0x8a, 0xe5, 0x9a, 0x8b, 0xe5, 0x9a, 0x8c, 0xe5, 0x9a, 0x8d,
  0xe5, 0x9a, 0x90, 0xe5, 0x9a, 0x91, 0xe5, 0x9a, 0x92, 0xe5, 0x9a, 0x94,
  0xe5, 0x9a, 0x95, 0xe5, 0x9a, 0x96, 0xe5, 0x9a, 0x97, 0xe5, 0x9a, 0x98,
  0xe5, 0x9a, 0x99, 0xe5, 0x9a, 0x9a, 0xe5, 0x9a, 0x9b, 0xe5, 0x9a, 0x9c,
  0xe5, 0x9a, 0x9d, 0xe5, 0x9a, 0x9e, 0xe5, 0x9a, 0x9f, 0xe5, 0x9a, 0xa0,
  0xe5, 0x9a, 0xa1, 0xe5, 0x9a, 0xa2, 0xe5, 0x9a, 0xa4, 0xe5, 0x9a, 0xa5,
  0xe5, 0x9a, 0xa6, 0xe5, 0x9a, 0xa7, 0xe5, 0x9a, 0xa8, 0xe5, 0x9a, 0xa9,
  0xe5, 0x9a, 0xaa, 0xe5, 0x9a, 0xab, 0xe5, 0x9a, 0xac, 0xe5, 0x9a, 0xad,
  0xe5, 0x9a, 0xae, 0xe5, 0x9a, 0xb0, 0xe5, 0x9a, 0xb1, 0xe5, 0x9a, 0xb2,
  0xe5, 0x9a, 0xb3, 0xe5, 0x9a, 0xb4, 0xe5, 0x9a, 0xb5, 0xe5, 0x9a, 0xb6,
  0xe5, 0x9a, 0xb8, 0xe5, 0x9a, 0xb9, 0xe5, 0x9a, 0xba, 0xe5, 0x9a, 0xbb,
  0xe5, 0x9a, 0xbd, 0xe5, 0x9a, 0xbe, 0xe5, 0x9a, 0xbf, 0xe5, 0x9b, 0x80,
  0xe5, 0x9b, 0x81, 0xe5, 0x9b, 0x82, 0xe5, 0x9b, 0x83, 0xe5, 0x9b, 0x84,
  0xe5, 0x9b, 0x85, 0xe5, 0x9b, 0x86, 0xe5, 0x9b, 0x87, 0xe5, 0x9b, 0x88,
  0xe5, 0x9b, 0x89, 0xe5, 0x9b, 0x8b, 0xe5, 0x9b, 0x8c, 0xe5, 0x9b, 0x8d,
  0xe5, 0x9b, 0x8e, 0xe5, 0x9b, 0x8f, 0xe5, 0x9b, 0x90, 0xe5, 0x9b, 0x91,
  0xe5, 0x9b, 0x92, 0xe5, 0x9b, 0x93, 0xe5, 0x9b, 0x95, 0xe5, 0x9b, 0x96,
  0xe5, 0x9b, 0x98, 0xe5, 0x9b, 0x99, 0xe5, 0x9b, 0x9c, 0xe5, 0x9b, 0xa3,
  0xe5, 0x9b, 0xa5, 0xe5, 0x9b, 0xa6, 0xe5, 0x9b, 0xa7, 0xe5, 0x9b, 0xa8,
  0xe5, 0x9b, 0xa9, 0xe5, 0x9b, 0xaa, 0xe5, 0x9b, 0xac, 0xe5, 0x9b, 0xae,
  0xe5, 0x9b, 0xaf, 0xe5, 0x9b, 0xb2, 0xe5, 0x9b, 0xb3, 0xe5, 0x9b, 0xb6,
  0xe5, 0x9b, 0xb7, 0xe5, 0x9b, 0xb8, 0xe5, 0x9b, 0xbb, 0xe5, 0x9b, 0xbc,
  0xe5, 0x9c, 0x80, 0xe5, 0x9c, 0x81, 0xe5, 0x9c, 0x82, 0xe5, 0x9c, 0x85,
  0xe5, 0x9c, 0x87, 0xe5, 0x9c, 0x8b, 0xe5, 0x9c, 0x8c, 0xe5, 0x9c, 0x8d,
  0xe5, 0x9c, 0x8e, 0xe5, 0x9c, 0x8f, 0xe5, 0x9c, 0x90, 0xe5, 0x9c, 0x91,
  };

unsigned char gbk_row_0x88_converted_to_utf8[] = {
  0xe5, 0x9c, 0x92, 0xe5, 0x9c, 0x93, 0xe5, 0x9c, 0x94, 0xe5, 0x9c,
  0x95, 0xe5, 0x9c, 0x96, 0xe5, 0x9c, 0x97, 0xe5, 0x9c, 0x98, 0xe5, 0x9c,
  0x99, 0xe5, 0x9c, 0x9a, 0xe5, 0x9c, 0x9b, 0xe5, 0x9c, 0x9d, 0xe5, 0x9c,
  0x9e, 0xe5, 0x9c, 0xa0, 0xe5, 0x9c, 0xa1, 0xe5, 0x9c, 0xa2, 0xe5, 0x9c,
  0xa4, 0xe5, 0x9c, 0xa5, 0xe5, 0x9c, 0xa6, 0xe5, 0x9c, 0xa7, 0xe5, 0x9c,
  0xab, 0xe5, 0x9c, 0xb1, 0xe5, 0x9c, 0xb2, 0xe5, 0x9c, 0xb4, 0xe5, 0x9c,
  0xb5, 0xe5, 0x9c, 0xb6, 0xe5, 0x9c, 0xb7, 0xe5, 0x9c, 0xb8, 0xe5, 0x9c,
  0xbc, 0xe5, 0x9c, 0xbd, 0xe5, 0x9c, 0xbf, 0xe5, 0x9d, 0x81, 0xe5, 0x9d,
  0x83, 0xe5, 0x9d, 0x84, 0xe5, 0x9d, 0x85, 0xe5, 0x9d, 0x86, 0xe5, 0x9d,
  0x88, 0xe5, 0x9d, 0x89, 0xe5, 0x9d, 0x8b, 0xe5, 0x9d, 0x92, 0xe5, 0x9d,
  0x93, 0xe5, 0x9d, 0x94, 0xe5, 0x9d, 0x95, 0xe5, 0x9d, 0x96, 0xe5, 0x9d,
  0x98, 0xe5, 0x9d, 0x99, 0xe5, 0x9d, 0xa2, 0xe5, 0x9d, 0xa3, 0xe5, 0x9d,
  0xa5, 0xe5, 0x9d, 0xa7, 0xe5, 0x9d, 0xac, 0xe5, 0x9d, 0xae, 0xe5, 0x9d,
  0xb0, 0xe5, 0x9d, 0xb1, 0xe5, 0x9d, 0xb2, 0xe5, 0x9d, 0xb4, 0xe5, 0x9d,
  0xb5, 0xe5, 0x9d, 0xb8, 0xe5, 0x9d, 0xb9, 0xe5, 0x9d, 0xba, 0xe5, 0x9d,
  0xbd, 0xe5, 0x9d, 0xbe, 0xe5, 0x9d, 0xbf, 0xe5, 0x9e, 0x80, 0xe5, 0x9e,
  0x81, 0xe5, 0x9e, 0x87, 0xe5, 0x9e, 0x88, 0xe5, 0x9e, 0x89, 0xe5, 0x9e,
  0x8a, 0xe5, 0x9e, 0x8d, 0xe5, 0x9e, 0x8e, 0xe5, 0x9e, 0x8f, 0xe5, 0x9e,
  0x90, 0xe5, 0x9e, 0x91, 0xe5, 0x9e, 0x94, 0xe5, 0x9e, 0x95, 0xe5, 0x9e,
  0x96, 0xe5, 0x9e, 0x97, 0xe5, 0x9e, 0x98, 0xe5, 0x9e, 0x99, 0xe5, 0x9e,
  0x9a, 0xe5, 0x9e, 0x9c, 0xe5, 0x9e, 0x9d, 0xe5, 0x9e, 0x9e, 0xe5, 0x9e,
  0x9f, 0xe5, 0x9e, 0xa5, 0xe5, 0x9e, 0xa8, 0xe5, 0x9e, 0xaa, 0xe5, 0x9e,
  0xac, 0xe5, 0x9e, 0xaf, 0xe5, 0x9e, 0xb0, 0xe5, 0x9e, 0xb1, 0xe5, 0x9e,
  0xb3, 0xe5, 0x9e, 0xb5, 0xe5, 0x9e, 0xb6, 0xe5, 0x9e, 0xb7, 0xe5, 0x9e,
  0xb9, 0xe5, 0x9e, 0xba, 0xe5, 0x9e, 0xbb, 0xe5, 0x9e, 0xbc, 0xe5, 0x9e,
  0xbd, 0xe5, 0x9e, 0xbe, 0xe5, 0x9e, 0xbf, 0xe5, 0x9f, 0x80, 0xe5, 0x9f,
  0x81, 0xe5, 0x9f, 0x84, 0xe5, 0x9f, 0x85, 0xe5, 0x9f, 0x86, 0xe5, 0x9f,
  0x87, 0xe5, 0x9f, 0x88, 0xe5, 0x9f, 0x89, 0xe5, 0x9f, 0x8a, 0xe5, 0x9f,
  0x8c, 0xe5, 0x9f, 0x8d, 0xe5, 0x9f, 0x90, 0xe5, 0x9f, 0x91, 0xe5, 0x9f,
  0x93, 0xe5, 0x9f, 0x96, 0xe5, 0x9f, 0x97, 0xe5, 0x9f, 0x9b, 0xe5, 0x9f,
  0x9c, 0xe5, 0x9f, 0x9e, 0xe5, 0x9f, 0xa1, 0xe5, 0x9f, 0xa2, 0xe5, 0x9f,
  0xa3, 0xe5, 0x9f, 0xa5, 0xe5, 0x9f, 0xa6, 0xe5, 0x9f, 0xa7, 0xe5, 0x9f,
  0xa8, 0xe5, 0x9f, 0xa9, 0xe5, 0x9f, 0xaa, 0xe5, 0x9f, 0xab, 0xe5, 0x9f,
  0xac, 0xe5, 0x9f, 0xae, 0xe5, 0x9f, 0xb0, 0xe5, 0x9f, 0xb1, 0xe5, 0x9f,
  0xb2, 0xe5, 0x9f, 0xb3, 0xe5, 0x9f, 0xb5, 0xe5, 0x9f, 0xb6, 0xe5, 0x9f,
  0xb7, 0xe5, 0x9f, 0xbb, 0xe5, 0x9f, 0xbc, 0xe5, 0x9f, 0xbe, 0xe5, 0x9f,
  0xbf, 0xe5, 0xa0, 0x81, 0xe5, 0xa0, 0x83, 0xe5, 0xa0, 0x84, 0xe5, 0xa0,
  0x85, 0xe5, 0xa0, 0x88, 0xe5, 0xa0, 0x89, 0xe5, 0xa0, 0x8a, 0xe5, 0xa0,
  0x8c, 0xe5, 0xa0, 0x8e, 0xe5, 0xa0, 0x8f, 0xe5, 0xa0, 0x90, 0xe5, 0xa0,
  0x92, 0xe5, 0xa0, 0x93, 0xe5, 0xa0, 0x94, 0xe5, 0xa0, 0x96, 0xe5, 0xa0,
  0x97, 0xe5, 0xa0, 0x98, 0xe5, 0xa0, 0x9a, 0xe5, 0xa0, 0x9b, 0xe5, 0xa0,
  0x9c, 0xe5, 0xa0, 0x9d, 0xe5, 0xa0, 0x9f, 0xe5, 0xa0, 0xa2, 0xe5, 0xa0,
  0xa3, 0xe5, 0xa0, 0xa5, 0xe5, 0xa0, 0xa6, 0xe5, 0xa0, 0xa7, 0xe5, 0xa0,
  0xa8, 0xe5, 0xa0, 0xa9, 0xe5, 0xa0, 0xab, 0xe5, 0xa0, 0xac, 0xe5, 0xa0,
  0xad, 0xe5, 0xa0, 0xae, 0xe5, 0xa0, 0xaf, 0xe5, 0xa0, 0xb1, 0xe5, 0xa0,
  0xb2, 0xe5, 0xa0, 0xb3, 0xe5, 0xa0, 0xb4, 0xe5, 0xa0, 0xb6, 0xe5, 0xa0,
  0xb7, 0xe5, 0xa0, 0xb8, 0xe5, 0xa0, 0xb9, 0xe5, 0xa0, 0xba, 0xe5, 0xa0,
  0xbb, 0xe5, 0xa0, 0xbc, 0xe5, 0xa0, 0xbd
};

unsigned char gbk_row_0x89_converted_to_utf8[] = {
  0xe5, 0xa0, 0xbe, 0xe5,
  0xa0, 0xbf, 0xe5, 0xa1, 0x80, 0xe5, 0xa1, 0x81, 0xe5, 0xa1, 0x82, 0xe5,
  0xa1, 0x83, 0xe5, 0xa1, 0x85, 0xe5, 0xa1, 0x86, 0xe5, 0xa1, 0x87, 0xe5,
  0xa1, 0x88, 0xe5, 0xa1, 0x89, 0xe5, 0xa1, 0x8a, 0xe5, 0xa1, 0x8b, 0xe5,
  0xa1, 0x8e, 0xe5, 0xa1, 0x8f, 0xe5, 0xa1, 0x90, 0xe5, 0xa1, 0x92, 0xe5,
  0xa1, 0x93, 0xe5, 0xa1, 0x95, 0xe5, 0xa1, 0x96, 0xe5, 0xa1, 0x97, 0xe5,
  0xa1, 0x99, 0xe5, 0xa1, 0x9a, 0xe5, 0xa1, 0x9b, 0xe5, 0xa1, 0x9c, 0xe5,
  0xa1, 0x9d, 0xe5, 0xa1, 0x9f, 0xe5, 0xa1, 0xa0, 0xe5, 0xa1, 0xa1, 0xe5,
  0xa1, 0xa2, 0xe5, 0xa1, 0xa3, 0xe5, 0xa1, 0xa4, 0xe5, 0xa1, 0xa6, 0xe5,
  0xa1, 0xa7, 0xe5, 0xa1, 0xa8, 0xe5, 0xa1, 0xa9, 0xe5, 0xa1, 0xaa, 0xe5,
  0xa1, 0xad, 0xe5, 0xa1, 0xae, 0xe5, 0xa1, 0xaf, 0xe5, 0xa1, 0xb0, 0xe5,
  0xa1, 0xb1, 0xe5, 0xa1, 0xb2, 0xe5, 0xa1, 0xb3, 0xe5, 0xa1, 0xb4, 0xe5,
  0xa1, 0xb5, 0xe5, 0xa1, 0xb6, 0xe5, 0xa1, 0xb7, 0xe5, 0xa1, 0xb8, 0xe5,
  0xa1, 0xb9, 0xe5, 0xa1, 0xba, 0xe5, 0xa1, 0xbb, 0xe5, 0xa1, 0xbc, 0xe5,
  0xa1, 0xbd, 0xe5, 0xa1, 0xbf, 0xe5, 0xa2, 0x82, 0xe5, 0xa2, 0x84, 0xe5,
  0xa2, 0x86, 0xe5, 0xa2, 0x87, 0xe5, 0xa2, 0x88, 0xe5, 0xa2, 0x8a, 0xe5,
  0xa2, 0x8b, 0xe5, 0xa2, 0x8c, 0xe5, 0xa2, 0x8d, 0xe5, 0xa2, 0x8e, 0xe5,
  0xa2, 0x8f, 0xe5, 0xa2, 0x90, 0xe5, 0xa2, 0x91, 0xe5, 0xa2, 0x94, 0xe5,
  0xa2, 0x95, 0xe5, 0xa2, 0x96, 0xe5, 0xa2, 0x97, 0xe5, 0xa2, 0x98, 0xe5,
  0xa2, 0x9b, 0xe5, 0xa2, 0x9c, 0xe5, 0xa2, 0x9d, 0xe5, 0xa2, 0xa0, 0xe5,
  0xa2, 0xa1, 0xe5, 0xa2, 0xa2, 0xe5, 0xa2, 0xa3, 0xe5, 0xa2, 0xa4, 0xe5,
  0xa2, 0xa5, 0xe5, 0xa2, 0xa6, 0xe5, 0xa2, 0xa7, 0xe5, 0xa2, 0xaa, 0xe5,
  0xa2, 0xab, 0xe5, 0xa2, 0xac, 0xe5, 0xa2, 0xad, 0xe5, 0xa2, 0xae, 0xe5,
  0xa2, 0xaf, 0xe5, 0xa2, 0xb0, 0xe5, 0xa2, 0xb1, 0xe5, 0xa2, 0xb2, 0xe5,
  0xa2, 0xb3, 0xe5, 0xa2, 0xb4, 0xe5, 0xa2, 0xb5, 0xe5, 0xa2, 0xb6, 0xe5,
  0xa2, 0xb7, 0xe5, 0xa2, 0xb8, 0xe5, 0xa2, 0xb9, 0xe5, 0xa2, 0xba, 0xe5,
  0xa2, 0xbb, 0xe5, 0xa2, 0xbd, 0xe5, 0xa2, 0xbe, 0xe5, 0xa2, 0xbf, 0xe5,
  0xa3, 0x80, 0xe5, 0xa3, 0x82, 0xe5, 0xa3, 0x83, 0xe5, 0xa3, 0x84, 0xe5,
  0xa3, 0x86, 0xe5, 0xa3, 0x87, 0xe5, 0xa3, 0x88, 0xe5, 0xa3, 0x89, 0xe5,
  0xa3, 0x8a, 0xe5, 0xa3, 0x8b, 0xe5, 0xa3, 0x8c, 0xe5, 0xa3, 0x8d, 0xe5,
  0xa3, 0x8e, 0xe5, 0xa3, 0x8f, 0xe5, 0xa3, 0x90, 0xe5, 0xa3, 0x92, 0xe5,
  0xa3, 0x93, 0xe5, 0xa3, 0x94, 0xe5, 0xa3, 0x96, 0xe5, 0xa3, 0x97, 0xe5,
  0xa3, 0x98, 0xe5, 0xa3, 0x99, 0xe5, 0xa3, 0x9a, 0xe5, 0xa3, 0x9b, 0xe5,
  0xa3, 0x9c, 0xe5, 0xa3, 0x9d, 0xe5, 0xa3, 0x9e, 0xe5, 0xa3, 0x9f, 0xe5,
  0xa3, 0xa0, 0xe5, 0xa3, 0xa1, 0xe5, 0xa3, 0xa2, 0xe5, 0xa3, 0xa3, 0xe5,
  0xa3, 0xa5, 0xe5, 0xa3, 0xa6, 0xe5, 0xa3, 0xa7, 0xe5, 0xa3, 0xa8, 0xe5,
  0xa3, 0xa9, 0xe5, 0xa3, 0xaa, 0xe5, 0xa3, 0xad, 0xe5, 0xa3, 0xaf, 0xe5,
  0xa3, 0xb1, 0xe5, 0xa3, 0xb2, 0xe5, 0xa3, 0xb4, 0xe5, 0xa3, 0xb5, 0xe5,
  0xa3, 0xb7, 0xe5, 0xa3, 0xb8, 0xe5, 0xa3, 0xba, 0xe5, 0xa3, 0xbb, 0xe5,
  0xa3, 0xbc, 0xe5, 0xa3, 0xbd, 0xe5, 0xa3, 0xbe, 0xe5, 0xa3, 0xbf, 0xe5,
  0xa4, 0x80, 0xe5, 0xa4, 0x81, 0xe5, 0xa4, 0x83, 0xe5, 0xa4, 0x85, 0xe5,
  0xa4, 0x86, 0xe5, 0xa4, 0x88, 0xe5, 0xa4, 0x89, 0xe5, 0xa4, 0x8a, 0xe5,
  0xa4, 0x8b, 0xe5, 0xa4, 0x8c, 0xe5, 0xa4, 0x8e, 0xe5, 0xa4, 0x90, 0xe5,
  0xa4, 0x91, 0xe5, 0xa4, 0x92, 0xe5, 0xa4, 0x93, 0xe5, 0xa4, 0x97, 0xe5,
  0xa4, 0x98, 0xe5, 0xa4, 0x9b, 0xe5, 0xa4, 0x9d, 0xe5, 0xa4, 0x9e, 0xe5,
  0xa4, 0xa0, 0xe5, 0xa4, 0xa1, 0xe5, 0xa4, 0xa2, 0xe5, 0xa4, 0xa3, 0xe5,
  0xa4, 0xa6, 0xe5, 0xa4, 0xa8, 0xe5, 0xa4, 0xac, 0xe5, 0xa4, 0xb0, 0xe5,
  0xa4, 0xb2, 0xe5, 0xa4, 0xb3, 0xe5, 0xa4, 0xb5, 0xe5, 0xa4, 0xb6, 0xe5,
  0xa4, 0xbb
};

unsigned char gbk_row_0x8a_converted_to_utf8[] = {
  0xe5, 0xa4, 0xbd, 0xe5, 0xa4, 0xbe, 0xe5, 0xa4, 0xbf,
  0xe5, 0xa5, 0x80, 0xe5, 0xa5, 0x83, 0xe5, 0xa5, 0x85, 0xe5, 0xa5, 0x86,
  0xe5, 0xa5, 0x8a, 0xe5, 0xa5, 0x8c, 0xe5, 0xa5, 0x8d, 0xe5, 0xa5, 0x90,
  0xe5, 0xa5, 0x92, 0xe5, 0xa5, 0x93, 0xe5, 0xa5, 0x99, 0xe5, 0xa5, 0x9b,
  0xe5, 0xa5, 0x9c, 0xe5, 0xa5, 0x9d, 0xe5, 0xa5, 0x9e, 0xe5, 0xa5, 0x9f,
  0xe5, 0xa5, 0xa1, 0xe5, 0xa5, 0xa3, 0xe5, 0xa5, 0xa4, 0xe5, 0xa5, 0xa6,
  0xe5, 0xa5, 0xa7, 0xe5, 0xa5, 0xa8, 0xe5, 0xa5, 0xa9, 0xe5, 0xa5, 0xaa,
  0xe5, 0xa5, 0xab, 0xe5, 0xa5, 0xac, 0xe5, 0xa5, 0xad, 0xe5, 0xa5, 0xae,
  0xe5, 0xa5, 0xaf, 0xe5, 0xa5, 0xb0, 0xe5, 0xa5, 0xb1, 0xe5, 0xa5, 0xb2,
  0xe5, 0xa5, 0xb5, 0xe5, 0xa5, 0xb7, 0xe5, 0xa5, 0xba, 0xe5, 0xa5, 0xbb,
  0xe5, 0xa5, 0xbc, 0xe5, 0xa5, 0xbe, 0xe5, 0xa5, 0xbf, 0xe5, 0xa6, 0x80,
  0xe5, 0xa6, 0x85, 0xe5, 0xa6, 0x89, 0xe5, 0xa6, 0x8b, 0xe5, 0xa6, 0x8c,
  0xe5, 0xa6, 0x8e, 0xe5, 0xa6, 0x8f, 0xe5, 0xa6, 0x90, 0xe5, 0xa6, 0x91,
  0xe5, 0xa6, 0x94, 0xe5, 0xa6, 0x95, 0xe5, 0xa6, 0x98, 0xe5, 0xa6, 0x9a,
  0xe5, 0xa6, 0x9b, 0xe5, 0xa6, 0x9c, 0xe5, 0xa6, 0x9d, 0xe5, 0xa6, 0x9f,
  0xe5, 0xa6, 0xa0, 0xe5, 0xa6, 0xa1, 0xe5, 0xa6, 0xa2, 0xe5, 0xa6, 0xa6,
  0xe5, 0xa6, 0xa7, 0xe5, 0xa6, 0xac, 0xe5, 0xa6, 0xad, 0xe5, 0xa6, 0xb0,
  0xe5, 0xa6, 0xb1, 0xe5, 0xa6, 0xb3, 0xe5, 0xa6, 0xb4, 0xe5, 0xa6, 0xb5,
  0xe5, 0xa6, 0xb6, 0xe5, 0xa6, 0xb7, 0xe5, 0xa6, 0xb8, 0xe5, 0xa6, 0xba,
  0xe5, 0xa6, 0xbc, 0xe5, 0xa6, 0xbd, 0xe5, 0xa6, 0xbf, 0xe5, 0xa7, 0x80,
  0xe5, 0xa7, 0x81, 0xe5, 0xa7, 0x82, 0xe5, 0xa7, 0x83, 0xe5, 0xa7, 0x84,
  0xe5, 0xa7, 0x85, 0xe5, 0xa7, 0x87, 0xe5, 0xa7, 0x88, 0xe5, 0xa7, 0x89,
  0xe5, 0xa7, 0x8c, 0xe5, 0xa7, 0x8d, 0xe5, 0xa7, 0x8e, 0xe5, 0xa7, 0x8f,
  0xe5, 0xa7, 0x95, 0xe5, 0xa7, 0x96, 0xe5, 0xa7, 0x99, 0xe5, 0xa7, 0x9b,
  0xe5, 0xa7, 0x9e, 0xe5, 0xa7, 0x9f, 0xe5, 0xa7, 0xa0, 0xe5, 0xa7, 0xa1,
  0xe5, 0xa7, 0xa2, 0xe5, 0xa7, 0xa4, 0xe5, 0xa7, 0xa6, 0xe5, 0xa7, 0xa7,
  0xe5, 0xa7, 0xa9, 0xe5, 0xa7, 0xaa, 0xe5, 0xa7, 0xab, 0xe5, 0xa7, 0xad,
  0xe5, 0xa7, 0xae, 0xe5, 0xa7, 0xaf, 0xe5, 0xa7, 0xb0, 0xe5, 0xa7, 0xb1,
  0xe5, 0xa7, 0xb2, 0xe5, 0xa7, 0xb3, 0xe5, 0xa7, 0xb4, 0xe5, 0xa7, 0xb5,
  0xe5, 0xa7, 0xb6, 0xe5, 0xa7, 0xb7, 0xe5, 0xa7, 0xb8, 0xe5, 0xa7, 0xba,
  0xe5, 0xa7, 0xbc, 0xe5, 0xa7, 0xbd, 0xe5, 0xa7, 0xbe, 0xe5, 0xa8, 0x80,
  0xe5, 0xa8, 0x82, 0xe5, 0xa8, 0x8a, 0xe5, 0xa8, 0x8b, 0xe5, 0xa8, 0x8d,
  0xe5, 0xa8, 0x8e, 0xe5, 0xa8, 0x8f, 0xe5, 0xa8, 0x90, 0xe5, 0xa8, 0x92,
  0xe5, 0xa8, 0x94, 0xe5, 0xa8, 0x95, 0xe5, 0xa8, 0x96, 0xe5, 0xa8, 0x97,
  0xe5, 0xa8, 0x99, 0xe5, 0xa8, 0x9a, 0xe5, 0xa8, 0x9b, 0xe5, 0xa8, 0x9d,
  0xe5, 0xa8, 0x9e, 0xe5, 0xa8, 0xa1, 0xe5, 0xa8, 0xa2, 0xe5, 0xa8, 0xa4,
  0xe5, 0xa8, 0xa6, 0xe5, 0xa8, 0xa7, 0xe5, 0xa8, 0xa8, 0xe5, 0xa8, 0xaa,
  0xe5, 0xa8, 0xab, 0xe5, 0xa8, 0xac, 0xe5, 0xa8, 0xad, 0xe5, 0xa8, 0xae,
  0xe5, 0xa8, 0xaf, 0xe5, 0xa8, 0xb0, 0xe5, 0xa8, 0xb3, 0xe5, 0xa8, 0xb5,
  0xe5, 0xa8, 0xb7, 0xe5, 0xa8, 0xb8, 0xe5, 0xa8, 0xb9, 0xe5, 0xa8, 0xba,
  0xe5, 0xa8, 0xbb, 0xe5, 0xa8, 0xbd, 0xe5, 0xa8, 0xbe, 0xe5, 0xa8, 0xbf,
  0xe5, 0xa9, 0x81, 0xe5, 0xa9, 0x82, 0xe5, 0xa9, 0x83, 0xe5, 0xa9, 0x84,
  0xe5, 0xa9, 0x85, 0xe5, 0xa9, 0x87, 0xe5, 0xa9, 0x88, 0xe5, 0xa9, 0x8b,
  0xe5, 0xa9, 0x8c, 0xe5, 0xa9, 0x8d, 0xe5, 0xa9, 0x8e, 0xe5, 0xa9, 0x8f,
  0xe5, 0xa9, 0x90, 0xe5, 0xa9, 0x91, 0xe5, 0xa9, 0x92, 0xe5, 0xa9, 0x93,
  0xe5, 0xa9, 0x94, 0xe5, 0xa9, 0x96, 0xe5, 0xa9, 0x97, 0xe5, 0xa9, 0x98,
  0xe5, 0xa9, 0x99, 0xe5, 0xa9, 0x9b, 0xe5, 0xa9, 0x9c, 0xe5, 0xa9, 0x9d,
  0xe5, 0xa9, 0x9e, 0xe5, 0xa9, 0x9f, 0xe5, 0xa9, 0xa0
};

unsigned char gbk_row_0x8b_converted_to_utf8[] = {
  0xe5, 0xa9,
  0xa1, 0xe5, 0xa9, 0xa3, 0xe5, 0xa9, 0xa4, 0xe5, 0xa9, 0xa5, 0xe5, 0xa9,
  0xa6, 0xe5, 0xa9, 0xa8, 0xe5, 0xa9, 0xa9, 0xe5, 0xa9, 0xab, 0xe5, 0xa9,
  0xac, 0xe5, 0xa9, 0xad, 0xe5, 0xa9, 0xae, 0xe5, 0xa9, 0xaf, 0xe5, 0xa9,
  0xb0, 0xe5, 0xa9, 0xb1, 0xe5, 0xa9, 0xb2, 0xe5, 0xa9, 0xb3, 0xe5, 0xa9,
  0xb8, 0xe5, 0xa9, 0xb9, 0xe5, 0xa9, 0xbb, 0xe5, 0xa9, 0xbc, 0xe5, 0xa9,
  0xbd, 0xe5, 0xa9, 0xbe, 0xe5, 0xaa, 0x80, 0xe5, 0xaa, 0x81, 0xe5, 0xaa,
  0x82, 0xe5, 0xaa, 0x83, 0xe5, 0xaa, 0x84, 0xe5, 0xaa, 0x85, 0xe5, 0xaa,
  0x86, 0xe5, 0xaa, 0x87, 0xe5, 0xaa, 0x88, 0xe5, 0xaa, 0x89, 0xe5, 0xaa,
  0x8a, 0xe5, 0xaa, 0x8b, 0xe5, 0xaa, 0x8c, 0xe5, 0xaa, 0x8d, 0xe5, 0xaa,
  0x8e, 0xe5, 0xaa, 0x8f, 0xe5, 0xaa, 0x90, 0xe5, 0xaa, 0x91, 0xe5, 0xaa,
  0x93, 0xe5, 0xaa, 0x94, 0xe5, 0xaa, 0x95, 0xe5, 0xaa, 0x96, 0xe5, 0xaa,
  0x97, 0xe5, 0xaa, 0x98, 0xe5, 0xaa, 0x99, 0xe5, 0xaa, 0x9c, 0xe5, 0xaa,
  0x9d, 0xe5, 0xaa, 0x9e, 0xe5, 0xaa, 0x9f, 0xe5, 0xaa, 0xa0, 0xe5, 0xaa,
  0xa1, 0xe5, 0xaa, 0xa2, 0xe5, 0xaa, 0xa3, 0xe5, 0xaa, 0xa4, 0xe5, 0xaa,
  0xa5, 0xe5, 0xaa, 0xa6, 0xe5, 0xaa, 0xa7, 0xe5, 0xaa, 0xa8, 0xe5, 0xaa,
  0xa9, 0xe5, 0xaa, 0xab, 0xe5, 0xaa, 0xac, 0xe5, 0xaa, 0xad, 0xe5, 0xaa,
  0xae, 0xe5, 0xaa, 0xaf, 0xe5, 0xaa, 0xb0, 0xe5, 0xaa, 0xb1, 0xe5, 0xaa,
  0xb4, 0xe5, 0xaa, 0xb6, 0xe5, 0xaa, 0xb7, 0xe5, 0xaa, 0xb9, 0xe5, 0xaa,
  0xba, 0xe5, 0xaa, 0xbb, 0xe5, 0xaa, 0xbc, 0xe5, 0xaa, 0xbd, 0xe5, 0xaa,
  0xbf, 0xe5, 0xab, 0x80, 0xe5, 0xab, 0x83, 0xe5, 0xab, 0x84, 0xe5, 0xab,
  0x85, 0xe5, 0xab, 0x86, 0xe5, 0xab, 0x87, 0xe5, 0xab, 0x88, 0xe5, 0xab,
  0x8a, 0xe5, 0xab, 0x8b, 0xe5, 0xab, 0x8d, 0xe5, 0xab, 0x8e, 0xe5, 0xab,
  0x8f, 0xe5, 0xab, 0x90, 0xe5, 0xab, 0x91, 0xe5, 0xab, 0x93, 0xe5, 0xab,
  0x95, 0xe5, 0xab, 0x97, 0xe5, 0xab, 0x99, 0xe5, 0xab, 0x9a, 0xe5, 0xab,
  0x9b, 0xe5, 0xab, 0x9d, 0xe5, 0xab, 0x9e, 0xe5, 0xab, 0x9f, 0xe5, 0xab,
  0xa2, 0xe5, 0xab, 0xa4, 0xe5, 0xab, 0xa5, 0xe5, 0xab, 0xa7, 0xe5, 0xab,
  0xa8, 0xe5, 0xab, 0xaa, 0xe5, 0xab, 0xac, 0xe5, 0xab, 0xad, 0xe5, 0xab,
  0xae, 0xe5, 0xab, 0xaf, 0xe5, 0xab, 0xb0, 0xe5, 0xab, 0xb2, 0xe5, 0xab,
  0xb3, 0xe5, 0xab, 0xb4, 0xe5, 0xab, 0xb5, 0xe5, 0xab, 0xb6, 0xe5, 0xab,
  0xb7, 0xe5, 0xab, 0xb8, 0xe5, 0xab, 0xb9, 0xe5, 0xab, 0xba, 0xe5, 0xab,
  0xbb, 0xe5, 0xab, 0xbc, 0xe5, 0xab, 0xbd, 0xe5, 0xab, 0xbe, 0xe5, 0xab,
  0xbf, 0xe5, 0xac, 0x80, 0xe5, 0xac, 0x81, 0xe5, 0xac, 0x82, 0xe5, 0xac,
  0x83, 0xe5, 0xac, 0x84, 0xe5, 0xac, 0x85, 0xe5, 0xac, 0x86, 0xe5, 0xac,
  0x87, 0xe5, 0xac, 0x88, 0xe5, 0xac, 0x8a, 0xe5, 0xac, 0x8b, 0xe5, 0xac,
  0x8c, 0xe5, 0xac, 0x8d, 0xe5, 0xac, 0x8e, 0xe5, 0xac, 0x8f, 0xe5, 0xac,
  0x90, 0xe5, 0xac, 0x91, 0xe5, 0xac, 0x92, 0xe5, 0xac, 0x93, 0xe5, 0xac,
  0x94, 0xe5, 0xac, 0x95, 0xe5, 0xac, 0x98, 0xe5, 0xac, 0x99, 0xe5, 0xac,
  0x9a, 0xe5, 0xac, 0x9b, 0xe5, 0xac, 0x9c, 0xe5, 0xac, 0x9d, 0xe5, 0xac,
  0x9e, 0xe5, 0xac, 0x9f, 0xe5, 0xac, 0xa0, 0xe5, 0xac, 0xa1, 0xe5, 0xac,
  0xa2, 0xe5, 0xac, 0xa3, 0xe5, 0xac, 0xa4, 0xe5, 0xac, 0xa5, 0xe5, 0xac,
  0xa6, 0xe5, 0xac, 0xa7, 0xe5, 0xac, 0xa8, 0xe5, 0xac, 0xa9, 0xe5, 0xac,
  0xaa, 0xe5, 0xac, 0xab, 0xe5, 0xac, 0xac, 0xe5, 0xac, 0xad, 0xe5, 0xac,
  0xae, 0xe5, 0xac, 0xaf, 0xe5, 0xac, 0xb0, 0xe5, 0xac, 0xb1, 0xe5, 0xac,
  0xb3, 0xe5, 0xac, 0xb5, 0xe5, 0xac, 0xb6, 0xe5, 0xac, 0xb8, 0xe5, 0xac,
  0xb9, 0xe5, 0xac, 0xba, 0xe5, 0xac, 0xbb, 0xe5, 0xac, 0xbc, 0xe5, 0xac,
  0xbd, 0xe5, 0xac, 0xbe, 0xe5, 0xac, 0xbf, 0xe5, 0xad, 0x81, 0xe5, 0xad,
  0x82, 0xe5, 0xad, 0x83, 0xe5, 0xad, 0x84, 0xe5, 0xad, 0x85, 0xe5, 0xad,
  0x86, 0xe5, 0xad, 0x87
};

unsigned char gbk_row_0x8c_converted_to_utf8[] = {
  0xe5, 0xad, 0x88, 0xe5, 0xad, 0x89, 0xe5,
  0xad, 0x8a, 0xe5, 0xad, 0x8b, 0xe5, 0xad, 0x8c, 0xe5, 0xad, 0x8d, 0xe5,
  0xad, 0x8e, 0xe5, 0xad, 0x8f, 0xe5, 0xad, 0x92, 0xe5, 0xad, 0x96, 0xe5,
  0xad, 0x9e, 0xe5, 0xad, 0xa0, 0xe5, 0xad, 0xa1, 0xe5, 0xad, 0xa7, 0xe5,
  0xad, 0xa8, 0xe5, 0xad, 0xab, 0xe5, 0xad, 0xad, 0xe5, 0xad, 0xae, 0xe5,
  0xad, 0xaf, 0xe5, 0xad, 0xb2, 0xe5, 0xad, 0xb4, 0xe5, 0xad, 0xb6, 0xe5,
  0xad, 0xb7, 0xe5, 0xad, 0xb8, 0xe5, 0xad, 0xb9, 0xe5, 0xad, 0xbb, 0xe5,
  0xad, 0xbc, 0xe5, 0xad, 0xbe, 0xe5, 0xad, 0xbf, 0xe5, 0xae, 0x82, 0xe5,
  0xae, 0x86, 0xe5, 0xae, 0x8a, 0xe5, 0xae, 0x8d, 0xe5, 0xae, 0x8e, 0xe5,
  0xae, 0x90, 0xe5, 0xae, 0x91, 0xe5, 0xae, 0x92, 0xe5, 0xae, 0x94, 0xe5,
  0xae, 0x96, 0xe5, 0xae, 0x9f, 0xe5, 0xae, 0xa7, 0xe5, 0xae, 0xa8, 0xe5,
  0xae, 0xa9, 0xe5, 0xae, 0xac, 0xe5, 0xae, 0xad, 0xe5, 0xae, 0xae, 0xe5,
  0xae, 0xaf, 0xe5, 0xae, 0xb1, 0xe5, 0xae, 0xb2, 0xe5, 0xae, 0xb7, 0xe5,
  0xae, 0xba, 0xe5, 0xae, 0xbb, 0xe5, 0xae, 0xbc, 0xe5, 0xaf, 0x80, 0xe5,
  0xaf, 0x81, 0xe5, 0xaf, 0x83, 0xe5, 0xaf, 0x88, 0xe5, 0xaf, 0x89, 0xe5,
  0xaf, 0x8a, 0xe5, 0xaf, 0x8b, 0xe5, 0xaf, 0x8d, 0xe5, 0xaf, 0x8e, 0xe5,
  0xaf, 0x8f, 0xe5, 0xaf, 0x91, 0xe5, 0xaf, 0x94, 0xe5, 0xaf, 0x95, 0xe5,
  0xaf, 0x96, 0xe5, 0xaf, 0x97, 0xe5, 0xaf, 0x98, 0xe5, 0xaf, 0x99, 0xe5,
  0xaf, 0x9a, 0xe5, 0xaf, 0x9b, 0xe5, 0xaf, 0x9c, 0xe5, 0xaf, 0xa0, 0xe5,
  0xaf, 0xa2, 0xe5, 0xaf, 0xa3, 0xe5, 0xaf, 0xa6, 0xe5, 0xaf, 0xa7, 0xe5,
  0xaf, 0xa9, 0xe5, 0xaf, 0xaa, 0xe5, 0xaf, 0xab, 0xe5, 0xaf, 0xac, 0xe5,
  0xaf, 0xad, 0xe5, 0xaf, 0xaf, 0xe5, 0xaf, 0xb1, 0xe5, 0xaf, 0xb2, 0xe5,
  0xaf, 0xb3, 0xe5, 0xaf, 0xb4, 0xe5, 0xaf, 0xb5, 0xe5, 0xaf, 0xb6, 0xe5,
  0xaf, 0xb7, 0xe5, 0xaf, 0xbd, 0xe5, 0xaf, 0xbe, 0xe5, 0xb0, 0x80, 0xe5,
  0xb0, 0x82, 0xe5, 0xb0, 0x83, 0xe5, 0xb0, 0x85, 0xe5, 0xb0, 0x87, 0xe5,
  0xb0, 0x88, 0xe5, 0xb0, 0x8b, 0xe5, 0xb0, 0x8c, 0xe5, 0xb0, 0x8d, 0xe5,
  0xb0, 0x8e, 0xe5, 0xb0, 0x90, 0xe5, 0xb0, 0x92, 0xe5, 0xb0, 0x93, 0xe5,
  0xb0, 0x97, 0xe5, 0xb0, 0x99, 0xe5, 0xb0, 0x9b, 0xe5, 0xb0, 0x9e, 0xe5,
  0xb0, 0x9f, 0xe5, 0xb0, 0xa0, 0xe5, 0xb0, 0xa1, 0xe5, 0xb0, 0xa3, 0xe5,
  0xb0, 0xa6, 0xe5, 0xb0, 0xa8, 0xe5, 0xb0, 0xa9, 0xe5, 0xb0, 0xaa, 0xe5,
  0xb0, 0xab, 0xe5, 0xb0, 0xad, 0xe5, 0xb0, 0xae, 0xe5, 0xb0, 0xaf, 0xe5,
  0xb0, 0xb0, 0xe5, 0xb0, 0xb2, 0xe5, 0xb0, 0xb3, 0xe5, 0xb0, 0xb5, 0xe5,
  0xb0, 0xb6, 0xe5, 0xb0, 0xb7, 0xe5, 0xb1, 0x83, 0xe5, 0xb1, 0x84, 0xe5,
  0xb1, 0x86, 0xe5, 0xb1, 0x87, 0xe5, 0xb1, 0x8c, 0xe5, 0xb1, 0x8d, 0xe5,
  0xb1, 0x92, 0xe5, 0xb1, 0x93, 0xe5, 0xb1, 0x94, 0xe5, 0xb1, 0x96, 0xe5,
  0xb1, 0x97, 0xe5, 0xb1, 0x98, 0xe5, 0xb1, 0x9a, 0xe5, 0xb1, 0x9b, 0xe5,
  0xb1, 0x9c, 0xe5, 0xb1, 0x9d, 0xe5, 0xb1, 0x9f, 0xe5, 0xb1, 0xa2, 0xe5,
  0xb1, 0xa4, 0xe5, 0xb1, 0xa7, 0xe5, 0xb1, 0xa8, 0xe5, 0xb1, 0xa9, 0xe5,
  0xb1, 0xaa, 0xe5, 0xb1, 0xab, 0xe5, 0xb1, 0xac, 0xe5, 0xb1, 0xad, 0xe5,
  0xb1, 0xb0, 0xe5, 0xb1, 0xb2, 0xe5, 0xb1, 0xb3, 0xe5, 0xb1, 0xb4, 0xe5,
  0xb1, 0xb5, 0xe5, 0xb1, 0xb6, 0xe5, 0xb1, 0xb7, 0xe5, 0xb1, 0xb8, 0xe5,
  0xb1, 0xbb, 0xe5, 0xb1, 0xbc, 0xe5, 0xb1, 0xbd, 0xe5, 0xb1, 0xbe, 0xe5,
  0xb2, 0x80, 0xe5, 0xb2, 0x83, 0xe5, 0xb2, 0x84, 0xe5, 0xb2, 0x85, 0xe5,
  0xb2, 0x86, 0xe5, 0xb2, 0x87, 0xe5, 0xb2, 0x89, 0xe5, 0xb2, 0x8a, 0xe5,
  0xb2, 0x8b, 0xe5, 0xb2, 0x8e, 0xe5, 0xb2, 0x8f, 0xe5, 0xb2, 0x92, 0xe5,
  0xb2, 0x93, 0xe5, 0xb2, 0x95, 0xe5, 0xb2, 0x9d, 0xe5, 0xb2, 0x9e, 0xe5,
  0xb2, 0x9f, 0xe5, 0xb2, 0xa0, 0xe5, 0xb2, 0xa1, 0xe5, 0xb2, 0xa4, 0xe5,
  0xb2, 0xa5, 0xe5, 0xb2, 0xa6, 0xe5, 0xb2, 0xa7, 0xe5, 0xb2, 0xa8
};

unsigned char gbk_row_0x8d_converted_to_utf8[] = {
  0xe5, 0xb2, 0xaa, 0xe5, 0xb2, 0xae, 0xe5, 0xb2, 0xaf, 0xe5, 0xb2, 0xb0,
  0xe5, 0xb2, 0xb2, 0xe5, 0xb2, 0xb4, 0xe5, 0xb2, 0xb6, 0xe5, 0xb2, 0xb9,
  0xe5, 0xb2, 0xba, 0xe5, 0xb2, 0xbb, 0xe5, 0xb2, 0xbc, 0xe5, 0xb2, 0xbe,
  0xe5, 0xb3, 0x80, 0xe5, 0xb3, 0x82, 0xe5, 0xb3, 0x83, 0xe5, 0xb3, 0x85,
  0xe5, 0xb3, 0x86, 0xe5, 0xb3, 0x87, 0xe5, 0xb3, 0x88, 0xe5, 0xb3, 0x89,
  0xe5, 0xb3, 0x8a, 0xe5, 0xb3, 0x8c, 0xe5, 0xb3, 0x8d, 0xe5, 0xb3, 0x8e,
  0xe5, 0xb3, 0x8f, 0xe5, 0xb3, 0x90, 0xe5, 0xb3, 0x91, 0xe5, 0xb3, 0x93,
  0xe5, 0xb3, 0x94, 0xe5, 0xb3, 0x95, 0xe5, 0xb3, 0x96, 0xe5, 0xb3, 0x97,
  0xe5, 0xb3, 0x98, 0xe5, 0xb3, 0x9a, 0xe5, 0xb3, 0x9b, 0xe5, 0xb3, 0x9c,
  0xe5, 0xb3, 0x9d, 0xe5, 0xb3, 0x9e, 0xe5, 0xb3, 0x9f, 0xe5, 0xb3, 0xa0,
  0xe5, 0xb3, 0xa2, 0xe5, 0xb3, 0xa3, 0xe5, 0xb3, 0xa7, 0xe5, 0xb3, 0xa9,
  0xe5, 0xb3, 0xab, 0xe5, 0xb3, 0xac, 0xe5, 0xb3, 0xae, 0xe5, 0xb3, 0xaf,
  0xe5, 0xb3, 0xb1, 0xe5, 0xb3, 0xb2, 0xe5, 0xb3, 0xb3, 0xe5, 0xb3, 0xb4,
  0xe5, 0xb3, 0xb5, 0xe5, 0xb3, 0xb6, 0xe5, 0xb3, 0xb7, 0xe5, 0xb3, 0xb8,
  0xe5, 0xb3, 0xb9, 0xe5, 0xb3, 0xba, 0xe5, 0xb3, 0xbc, 0xe5, 0xb3, 0xbd,
  0xe5, 0xb3, 0xbe, 0xe5, 0xb3, 0xbf, 0xe5, 0xb4, 0x80, 0xe5, 0xb4, 0x81,
  0xe5, 0xb4, 0x84, 0xe5, 0xb4, 0x85, 0xe5, 0xb4, 0x88, 0xe5, 0xb4, 0x89,
  0xe5, 0xb4, 0x8a, 0xe5, 0xb4, 0x8b, 0xe5, 0xb4, 0x8c, 0xe5, 0xb4, 0x8d,
  0xe5, 0xb4, 0x8f, 0xe5, 0xb4, 0x90, 0xe5, 0xb4, 0x91, 0xe5, 0xb4, 0x92,
  0xe5, 0xb4, 0x93, 0xe5, 0xb4, 0x95, 0xe5, 0xb4, 0x97, 0xe5, 0xb4, 0x98,
  0xe5, 0xb4, 0x99, 0xe5, 0xb4, 0x9a, 0xe5, 0xb4, 0x9c, 0xe5, 0xb4, 0x9d,
  0xe5, 0xb4, 0x9f, 0xe5, 0xb4, 0xa0, 0xe5, 0xb4, 0xa1, 0xe5, 0xb4, 0xa2,
  0xe5, 0xb4, 0xa3, 0xe5, 0xb4, 0xa5, 0xe5, 0xb4, 0xa8, 0xe5, 0xb4, 0xaa,
  0xe5, 0xb4, 0xab, 0xe5, 0xb4, 0xac, 0xe5, 0xb4, 0xaf, 0xe5, 0xb4, 0xb0,
  0xe5, 0xb4, 0xb1, 0xe5, 0xb4, 0xb2, 0xe5, 0xb4, 0xb3, 0xe5, 0xb4, 0xb5,
  0xe5, 0xb4, 0xb6, 0xe5, 0xb4, 0xb7, 0xe5, 0xb4, 0xb8, 0xe5, 0xb4, 0xb9,
  0xe5, 0xb4, 0xba, 0xe5, 0xb4, 0xbb, 0xe5, 0xb4, 0xbc, 0xe5, 0xb4, 0xbf,
  0xe5, 0xb5, 0x80, 0xe5, 0xb5, 0x81, 0xe5, 0xb5, 0x82, 0xe5, 0xb5, 0x83,
  0xe5, 0xb5, 0x84, 0xe5, 0xb5, 0x85, 0xe5, 0xb5, 0x86, 0xe5, 0xb5, 0x88,
  0xe5, 0xb5, 0x89, 0xe5, 0xb5, 0x8d, 0xe5, 0xb5, 0x8e, 0xe5, 0xb5, 0x8f,
  0xe5, 0xb5, 0x90, 0xe5, 0xb5, 0x91, 0xe5, 0xb5, 0x92, 0xe5, 0xb5, 0x93,
  0xe5, 0xb5, 0x94, 0xe5, 0xb5, 0x95, 0xe5, 0xb5, 0x96, 0xe5, 0xb5, 0x97,
  0xe5, 0xb5, 0x99, 0xe5, 0xb5, 0x9a, 0xe5, 0xb5, 0x9c, 0xe5, 0xb5, 0x9e,
  0xe5, 0xb5, 0x9f, 0xe5, 0xb5, 0xa0, 0xe5, 0xb5, 0xa1, 0xe5, 0xb5, 0xa2,
  0xe5, 0xb5, 0xa3, 0xe5, 0xb5, 0xa4, 0xe5, 0xb5, 0xa5, 0xe5, 0xb5, 0xa6,
  0xe5, 0xb5, 0xa7, 0xe5, 0xb5, 0xa8, 0xe5, 0xb5, 0xaa, 0xe5, 0xb5, 0xad,
  0xe5, 0xb5, 0xae, 0xe5, 0xb5, 0xb0, 0xe5, 0xb5, 0xb1, 0xe5, 0xb5, 0xb2,
  0xe5, 0xb5, 0xb3, 0xe5, 0xb5, 0xb5, 0xe5, 0xb5, 0xb6, 0xe5, 0xb5, 0xb7,
  0xe5, 0xb5, 0xb8, 0xe5, 0xb5, 0xb9, 0xe5, 0xb5, 0xba, 0xe5, 0xb5, 0xbb,
  0xe5, 0xb5, 0xbc, 0xe5, 0xb5, 0xbd, 0xe5, 0xb5, 0xbe, 0xe5, 0xb5, 0xbf,
  0xe5, 0xb6, 0x80, 0xe5, 0xb6, 0x81, 0xe5, 0xb6, 0x83, 0xe5, 0xb6, 0x84,
  0xe5, 0xb6, 0x85, 0xe5, 0xb6, 0x86, 0xe5, 0xb6, 0x87, 0xe5, 0xb6, 0x88,
  0xe5, 0xb6, 0x89, 0xe5, 0xb6, 0x8a, 0xe5, 0xb6, 0x8b, 0xe5, 0xb6, 0x8c,
  0xe5, 0xb6, 0x8d, 0xe5, 0xb6, 0x8e, 0xe5, 0xb6, 0x8f, 0xe5, 0xb6, 0x90,
  0xe5, 0xb6, 0x91, 0xe5, 0xb6, 0x92, 0xe5, 0xb6, 0x93, 0xe5, 0xb6, 0x94,
  0xe5, 0xb6, 0x95, 0xe5, 0xb6, 0x96, 0xe5, 0xb6, 0x97, 0xe5, 0xb6, 0x98,
  0xe5, 0xb6, 0x9a, 0xe5, 0xb6, 0x9b, 0xe5, 0xb6, 0x9c, 0xe5, 0xb6, 0x9e,
  0xe5, 0xb6, 0x9f, 0xe5, 0xb6, 0xa0
};

unsigned char gbk_row_0x8e_converted_to_utf8[] = {
  0xe5, 0xb6, 0xa1, 0xe5, 0xb6,
  0xa2, 0xe5, 0xb6, 0xa3, 0xe5, 0xb6, 0xa4, 0xe5, 0xb6, 0xa5, 0xe5, 0xb6,
  0xa6, 0xe5, 0xb6, 0xa7, 0xe5, 0xb6, 0xa8, 0xe5, 0xb6, 0xa9, 0xe5, 0xb6,
  0xaa, 0xe5, 0xb6, 0xab, 0xe5, 0xb6, 0xac, 0xe5, 0xb6, 0xad, 0xe5, 0xb6,
  0xae, 0xe5, 0xb6, 0xaf, 0xe5, 0xb6, 0xb0, 0xe5, 0xb6, 0xb1, 0xe5, 0xb6,
  0xb2, 0xe5, 0xb6, 0xb3, 0xe5, 0xb6, 0xb4, 0xe5, 0xb6, 0xb5, 0xe5, 0xb6,
  0xb6, 0xe5, 0xb6, 0xb8, 0xe5, 0xb6, 0xb9, 0xe5, 0xb6, 0xba, 0xe5, 0xb6,
  0xbb, 0xe5, 0xb6, 0xbc, 0xe5, 0xb6, 0xbd, 0xe5, 0xb6, 0xbe, 0xe5, 0xb6,
  0xbf, 0xe5, 0xb7, 0x80, 0xe5, 0xb7, 0x81, 0xe5, 0xb7, 0x82, 0xe5, 0xb7,
  0x83, 0xe5, 0xb7, 0x84, 0xe5, 0xb7, 0x86, 0xe5, 0xb7, 0x87, 0xe5, 0xb7,
  0x88, 0xe5, 0xb7, 0x89, 0xe5, 0xb7, 0x8a, 0xe5, 0xb7, 0x8b, 0xe5, 0xb7,
  0x8c, 0xe5, 0xb7, 0x8e, 0xe5, 0xb7, 0x8f, 0xe5, 0xb7, 0x90, 0xe5, 0xb7,
  0x91, 0xe5, 0xb7, 0x92, 0xe5, 0xb7, 0x93, 0xe5, 0xb7, 0x94, 0xe5, 0xb7,
  0x95, 0xe5, 0xb7, 0x96, 0xe5, 0xb7, 0x97, 0xe5, 0xb7, 0x98, 0xe5, 0xb7,
  0x99, 0xe5, 0xb7, 0x9a, 0xe5, 0xb7, 0x9c, 0xe5, 0xb7, 0x9f, 0xe5, 0xb7,
  0xa0, 0xe5, 0xb7, 0xa3, 0xe5, 0xb7, 0xa4, 0xe5, 0xb7, 0xaa, 0xe5, 0xb7,
  0xac, 0xe5, 0xb7, 0xad, 0xe5, 0xb7, 0xb0, 0xe5, 0xb7, 0xb5, 0xe5, 0xb7,
  0xb6, 0xe5, 0xb7, 0xb8, 0xe5, 0xb7, 0xb9, 0xe5, 0xb7, 0xba, 0xe5, 0xb7,
  0xbb, 0xe5, 0xb7, 0xbc, 0xe5, 0xb7, 0xbf, 0xe5, 0xb8, 0x80, 0xe5, 0xb8,
  0x84, 0xe5, 0xb8, 0x87, 0xe5, 0xb8, 0x89, 0xe5, 0xb8, 0x8a, 0xe5, 0xb8,
  0x8b, 0xe5, 0xb8, 0x8d, 0xe5, 0xb8, 0x8e, 0xe5, 0xb8, 0x92, 0xe5, 0xb8,
  0x93, 0xe5, 0xb8, 0x97, 0xe5, 0xb8, 0x9e, 0xe5, 0xb8, 0x9f, 0xe5, 0xb8,
  0xa0, 0xe5, 0xb8, 0xa1, 0xe5, 0xb8, 0xa2, 0xe5, 0xb8, 0xa3, 0xe5, 0xb8,
  0xa4, 0xe5, 0xb8, 0xa5, 0xe5, 0xb8, 0xa8, 0xe5, 0xb8, 0xa9, 0xe5, 0xb8,
  0xaa, 0xe5, 0xb8, 0xab, 0xe5, 0xb8, 0xac, 0xe5, 0xb8, 0xaf, 0xe5, 0xb8,
  0xb0, 0xe5, 0xb8, 0xb2, 0xe5, 0xb8, 0xb3, 0xe5, 0xb8, 0xb4, 0xe5, 0xb8,
  0xb5, 0xe5, 0xb8, 0xb6, 0xe5, 0xb8, 0xb9, 0xe5, 0xb8, 0xba, 0xe5, 0xb8,
  0xbe, 0xe5, 0xb8, 0xbf, 0xe5, 0xb9, 0x80, 0xe5, 0xb9, 0x81, 0xe5, 0xb9,
  0x83, 0xe5, 0xb9, 0x86, 0xe5, 0xb9, 0x87, 0xe5, 0xb9, 0x88, 0xe5, 0xb9,
  0x89, 0xe5, 0xb9, 0x8a, 0xe5, 0xb9, 0x8b, 0xe5, 0xb9, 0x8d, 0xe5, 0xb9,
  0x8e, 0xe5, 0xb9, 0x8f, 0xe5, 0xb9, 0x90, 0xe5, 0xb9, 0x91, 0xe5, 0xb9,
  0x92, 0xe5, 0xb9, 0x93, 0xe5, 0xb9, 0x96, 0xe5, 0xb9, 0x97, 0xe5, 0xb9,
  0x98, 0xe5, 0xb9, 0x99, 0xe5, 0xb9, 0x9a, 0xe5, 0xb9, 0x9c, 0xe5, 0xb9,
  0x9d, 0xe5, 0xb9, 0x9f, 0xe5, 0xb9, 0xa0, 0xe5, 0xb9, 0xa3, 0xe5, 0xb9,
  0xa4, 0xe5, 0xb9, 0xa5, 0xe5, 0xb9, 0xa6, 0xe5, 0xb9, 0xa7, 0xe5, 0xb9,
  0xa8, 0xe5, 0xb9, 0xa9, 0xe5, 0xb9, 0xaa, 0xe5, 0xb9, 0xab, 0xe5, 0xb9,
  0xac, 0xe5, 0xb9, 0xad, 0xe5, 0xb9, 0xae, 0xe5, 0xb9, 0xaf, 0xe5, 0xb9,
  0xb0, 0xe5, 0xb9, 0xb1, 0xe5, 0xb9, 0xb5, 0xe5, 0xb9, 0xb7, 0xe5, 0xb9,
  0xb9, 0xe5, 0xb9, 0xbe, 0xe5, 0xba, 0x81, 0xe5, 0xba, 0x82, 0xe5, 0xba,
  0x83, 0xe5, 0xba, 0x85, 0xe5, 0xba, 0x88, 0xe5, 0xba, 0x89, 0xe5, 0xba,
  0x8c, 0xe5, 0xba, 0x8d, 0xe5, 0xba, 0x8e, 0xe5, 0xba, 0x92, 0xe5, 0xba,
  0x98, 0xe5, 0xba, 0x9b, 0xe5, 0xba, 0x9d, 0xe5, 0xba, 0xa1, 0xe5, 0xba,
  0xa2, 0xe5, 0xba, 0xa3, 0xe5, 0xba, 0xa4, 0xe5, 0xba, 0xa8, 0xe5, 0xba,
  0xa9, 0xe5, 0xba, 0xaa, 0xe5, 0xba, 0xab, 0xe5, 0xba, 0xac, 0xe5, 0xba,
  0xae, 0xe5, 0xba, 0xaf, 0xe5, 0xba, 0xb0, 0xe5, 0xba, 0xb1, 0xe5, 0xba,
  0xb2, 0xe5, 0xba, 0xb4, 0xe5, 0xba, 0xba, 0xe5, 0xba, 0xbb, 0xe5, 0xba,
  0xbc, 0xe5, 0xba, 0xbd, 0xe5, 0xba, 0xbf, 0xe5, 0xbb, 0x80, 0xe5, 0xbb,
  0x81, 0xe5, 0xbb, 0x82, 0xe5, 0xbb, 0x83, 0xe5, 0xbb, 0x84, 0xe5, 0xbb,
  0x85
};

unsigned char gbk_row_0x8f_converted_to_utf8[] = {
  0xe5, 0xbb, 0x86, 0xe5, 0xbb, 0x87, 0xe5, 0xbb, 0x88, 0xe5,
  0xbb, 0x8b, 0xe5, 0xbb, 0x8c, 0xe5, 0xbb, 0x8d, 0xe5, 0xbb, 0x8e, 0xe5,
  0xbb, 0x8f, 0xe5, 0xbb, 0x90, 0xe5, 0xbb, 0x94, 0xe5, 0xbb, 0x95, 0xe5,
  0xbb, 0x97, 0xe5, 0xbb, 0x98, 0xe5, 0xbb, 0x99, 0xe5, 0xbb, 0x9a, 0xe5,
  0xbb, 0x9c, 0xe5, 0xbb, 0x9d, 0xe5, 0xbb, 0x9e, 0xe5, 0xbb, 0x9f, 0xe5,
  0xbb, 0xa0, 0xe5, 0xbb, 0xa1, 0xe5, 0xbb, 0xa2, 0xe5, 0xbb, 0xa3, 0xe5,
  0xbb, 0xa4, 0xe5, 0xbb, 0xa5, 0xe5, 0xbb, 0xa6, 0xe5, 0xbb, 0xa7, 0xe5,
  0xbb, 0xa9, 0xe5, 0xbb, 0xab, 0xe5, 0xbb, 0xac, 0xe5, 0xbb, 0xad, 0xe5,
  0xbb, 0xae, 0xe5, 0xbb, 0xaf, 0xe5, 0xbb, 0xb0, 0xe5, 0xbb, 0xb1, 0xe5,
  0xbb, 0xb2, 0xe5, 0xbb, 0xb3, 0xe5, 0xbb, 0xb5, 0xe5, 0xbb, 0xb8, 0xe5,
  0xbb, 0xb9, 0xe5, 0xbb, 0xbb, 0xe5, 0xbb, 0xbc, 0xe5, 0xbb, 0xbd, 0xe5,
  0xbc, 0x85, 0xe5, 0xbc, 0x86, 0xe5, 0xbc, 0x87, 0xe5, 0xbc, 0x89, 0xe5,
  0xbc, 0x8c, 0xe5, 0xbc, 0x8d, 0xe5, 0xbc, 0x8e, 0xe5, 0xbc, 0x90, 0xe5,
  0xbc, 0x92, 0xe5, 0xbc, 0x94, 0xe5, 0xbc, 0x96, 0xe5, 0xbc, 0x99, 0xe5,
  0xbc, 0x9a, 0xe5, 0xbc, 0x9c, 0xe5, 0xbc, 0x9d, 0xe5, 0xbc, 0x9e, 0xe5,
  0xbc, 0xa1, 0xe5, 0xbc, 0xa2, 0xe5, 0xbc, 0xa3, 0xe5, 0xbc, 0xa4, 0xe5,
  0xbc, 0xa8, 0xe5, 0xbc, 0xab, 0xe5, 0xbc, 0xac, 0xe5, 0xbc, 0xae, 0xe5,
  0xbc, 0xb0, 0xe5, 0xbc, 0xb2, 0xe5, 0xbc, 0xb3, 0xe5, 0xbc, 0xb4, 0xe5,
  0xbc, 0xb5, 0xe5, 0xbc, 0xb6, 0xe5, 0xbc, 0xb7, 0xe5, 0xbc, 0xb8, 0xe5,
  0xbc, 0xbb, 0xe5, 0xbc, 0xbd, 0xe5, 0xbc, 0xbe, 0xe5, 0xbc, 0xbf, 0xe5,
  0xbd, 0x81, 0xe5, 0xbd, 0x82, 0xe5, 0xbd, 0x83, 0xe5, 0xbd, 0x84, 0xe5,
  0xbd, 0x85, 0xe5, 0xbd, 0x86, 0xe5, 0xbd, 0x87, 0xe5, 0xbd, 0x88, 0xe5,
  0xbd, 0x89, 0xe5, 0xbd, 0x8a, 0xe5, 0xbd, 0x8b, 0xe5, 0xbd, 0x8c, 0xe5,
  0xbd, 0x8d, 0xe5, 0xbd, 0x8e, 0xe5, 0xbd, 0x8f, 0xe5, 0xbd, 0x91, 0xe5,
  0xbd, 0x94, 0xe5, 0xbd, 0x99, 0xe5, 0xbd, 0x9a, 0xe5, 0xbd, 0x9b, 0xe5,
  0xbd, 0x9c, 0xe5, 0xbd, 0x9e, 0xe5, 0xbd, 0x9f, 0xe5, 0xbd, 0xa0, 0xe5,
  0xbd, 0xa3, 0xe5, 0xbd, 0xa5, 0xe5, 0xbd, 0xa7, 0xe5, 0xbd, 0xa8, 0xe5,
  0xbd, 0xab, 0xe5, 0xbd, 0xae, 0xe5, 0xbd, 0xaf, 0xe5, 0xbd, 0xb2, 0xe5,
  0xbd, 0xb4, 0xe5, 0xbd, 0xb5, 0xe5, 0xbd, 0xb6, 0xe5, 0xbd, 0xb8, 0xe5,
  0xbd, 0xba, 0xe5, 0xbd, 0xbd, 0xe5, 0xbd, 0xbe, 0xe5, 0xbd, 0xbf, 0xe5,
  0xbe, 0x83, 0xe5, 0xbe, 0x86, 0xe5, 0xbe, 0x8d, 0xe5, 0xbe, 0x8e, 0xe5,
  0xbe, 0x8f, 0xe5, 0xbe, 0x91, 0xe5, 0xbe, 0x93, 0xe5, 0xbe, 0x94, 0xe5,
  0xbe, 0x96, 0xe5, 0xbe, 0x9a, 0xe5, 0xbe, 0x9b, 0xe5, 0xbe, 0x9d, 0xe5,
  0xbe, 0x9e, 0xe5, 0xbe, 0x9f, 0xe5, 0xbe, 0xa0, 0xe5, 0xbe, 0xa2, 0xe5,
  0xbe, 0xa3, 0xe5, 0xbe, 0xa4, 0xe5, 0xbe, 0xa5, 0xe5, 0xbe, 0xa6, 0xe5,
  0xbe, 0xa7, 0xe5, 0xbe, 0xa9, 0xe5, 0xbe, 0xab, 0xe5, 0xbe, 0xac, 0xe5,
  0xbe, 0xaf, 0xe5, 0xbe, 0xb0, 0xe5, 0xbe, 0xb1, 0xe5, 0xbe, 0xb2, 0xe5,
  0xbe, 0xb3, 0xe5, 0xbe, 0xb4, 0xe5, 0xbe, 0xb6, 0xe5, 0xbe, 0xb8, 0xe5,
  0xbe, 0xb9, 0xe5, 0xbe, 0xba, 0xe5, 0xbe, 0xbb, 0xe5, 0xbe, 0xbe, 0xe5,
  0xbe, 0xbf, 0xe5, 0xbf, 0x80, 0xe5, 0xbf, 0x81, 0xe5, 0xbf, 0x82, 0xe5,
  0xbf, 0x87, 0xe5, 0xbf, 0x88, 0xe5, 0xbf, 0x8a, 0xe5, 0xbf, 0x8b, 0xe5,
  0xbf, 0x8e, 0xe5, 0xbf, 0x93, 0xe5, 0xbf, 0x94, 0xe5, 0xbf, 0x95, 0xe5,
  0xbf, 0x9a, 0xe5, 0xbf, 0x9b, 0xe5, 0xbf, 0x9c, 0xe5, 0xbf, 0x9e, 0xe5,
  0xbf, 0x9f, 0xe5, 0xbf, 0xa2, 0xe5, 0xbf, 0xa3, 0xe5, 0xbf, 0xa5, 0xe5,
  0xbf, 0xa6, 0xe5, 0xbf, 0xa8, 0xe5, 0xbf, 0xa9, 0xe5, 0xbf, 0xac, 0xe5,
  0xbf, 0xaf, 0xe5, 0xbf, 0xb0, 0xe5, 0xbf, 0xb2, 0xe5, 0xbf, 0xb3, 0xe5,
  0xbf, 0xb4, 0xe5, 0xbf, 0xb6, 0xe5, 0xbf, 0xb7, 0xe5, 0xbf, 0xb9, 0xe5,
  0xbf, 0xba, 0xe5, 0xbf, 0xbc, 0xe6, 0x80, 0x87
};

unsigned char gbk_row_0x90_converted_to_utf8[] = {
  0xe6, 0x80, 0x88,
  0xe6, 0x80, 0x89, 0xe6, 0x80, 0x8b, 0xe6, 0x80, 0x8c, 0xe6, 0x80, 0x90,
  0xe6, 0x80, 0x91, 0xe6, 0x80, 0x93, 0xe6, 0x80, 0x97, 0xe6, 0x80, 0x98,
  0xe6, 0x80, 0x9a, 0xe6, 0x80, 0x9e, 0xe6, 0x80, 0x9f, 0xe6, 0x80, 0xa2,
  0xe6, 0x80, 0xa3, 0xe6, 0x80, 0xa4, 0xe6, 0x80, 0xac, 0xe6, 0x80, 0xad,
  0xe6, 0x80, 0xae, 0xe6, 0x80, 0xb0, 0xe6, 0x80, 0xb1, 0xe6, 0x80, 0xb2,
  0xe6, 0x80, 0xb3, 0xe6, 0x80, 0xb4, 0xe6, 0x80, 0xb6, 0xe6, 0x80, 0xb7,
  0xe6, 0x80, 0xb8, 0xe6, 0x80, 0xb9, 0xe6, 0x80, 0xba, 0xe6, 0x80, 0xbd,
  0xe6, 0x80, 0xbe, 0xe6, 0x81, 0x80, 0xe6, 0x81, 0x84, 0xe6, 0x81, 0x85,
  0xe6, 0x81, 0x86, 0xe6, 0x81, 0x87, 0xe6, 0x81, 0x88, 0xe6, 0x81, 0x89,
  0xe6, 0x81, 0x8a, 0xe6, 0x81, 0x8c, 0xe6, 0x81, 0x8e, 0xe6, 0x81, 0x8f,
  0xe6, 0x81, 0x91, 0xe6, 0x81, 0x93, 0xe6, 0x81, 0x94, 0xe6, 0x81, 0x96,
  0xe6, 0x81, 0x97, 0xe6, 0x81, 0x98, 0xe6, 0x81, 0x9b, 0xe6, 0x81, 0x9c,
  0xe6, 0x81, 0x9e, 0xe6, 0x81, 0x9f, 0xe6, 0x81, 0xa0, 0xe6, 0x81, 0xa1,
  0xe6, 0x81, 0xa5, 0xe6, 0x81, 0xa6, 0xe6, 0x81, 0xae, 0xe6, 0x81, 0xb1,
  0xe6, 0x81, 0xb2, 0xe6, 0x81, 0xb4, 0xe6, 0x81, 0xb5, 0xe6, 0x81, 0xb7,
  0xe6, 0x81, 0xbe, 0xe6, 0x82, 0x80, 0xe6, 0x82, 0x81, 0xe6, 0x82, 0x82,
  0xe6, 0x82, 0x85, 0xe6, 0x82, 0x86, 0xe6, 0x82, 0x87, 0xe6, 0x82, 0x88,
  0xe6, 0x82, 0x8a, 0xe6, 0x82, 0x8b, 0xe6, 0x82, 0x8e, 0xe6, 0x82, 0x8f,
  0xe6, 0x82, 0x90, 0xe6, 0x82, 0x91, 0xe6, 0x82, 0x93, 0xe6, 0x82, 0x95,
  0xe6, 0x82, 0x97, 0xe6, 0x82, 0x98, 0xe6, 0x82, 0x99, 0xe6, 0x82, 0x9c,
  0xe6, 0x82, 0x9e, 0xe6, 0x82, 0xa1, 0xe6, 0x82, 0xa2, 0xe6, 0x82, 0xa4,
  0xe6, 0x82, 0xa5, 0xe6, 0x82, 0xa7, 0xe6, 0x82, 0xa9, 0xe6, 0x82, 0xaa,
  0xe6, 0x82, 0xae, 0xe6, 0x82, 0xb0, 0xe6, 0x82, 0xb3, 0xe6, 0x82, 0xb5,
  0xe6, 0x82, 0xb6, 0xe6, 0x82, 0xb7, 0xe6, 0x82, 0xb9, 0xe6, 0x82, 0xba,
  0xe6, 0x82, 0xbd, 0xe6, 0x82, 0xbe, 0xe6, 0x82, 0xbf, 0xe6, 0x83, 0x80,
  0xe6, 0x83, 0x81, 0xe6, 0x83, 0x82, 0xe6, 0x83, 0x83, 0xe6, 0x83, 0x84,
  0xe6, 0x83, 0x87, 0xe6, 0x83, 0x88, 0xe6, 0x83, 0x89, 0xe6, 0x83, 0x8c,
  0xe6, 0x83, 0x8d, 0xe6, 0x83, 0x8e, 0xe6, 0x83, 0x8f, 0xe6, 0x83, 0x90,
  0xe6, 0x83, 0x92, 0xe6, 0x83, 0x93, 0xe6, 0x83, 0x94, 0xe6, 0x83, 0x96,
  0xe6, 0x83, 0x97, 0xe6, 0x83, 0x99, 0xe6, 0x83, 0x9b, 0xe6, 0x83, 0x9e,
  0xe6, 0x83, 0xa1, 0xe6, 0x83, 0xa2, 0xe6, 0x83, 0xa3, 0xe6, 0x83, 0xa4,
  0xe6, 0x83, 0xa5, 0xe6, 0x83, 0xaa, 0xe6, 0x83, 0xb1, 0xe6, 0x83, 0xb2,
  0xe6, 0x83, 0xb5, 0xe6, 0x83, 0xb7, 0xe6, 0x83, 0xb8, 0xe6, 0x83, 0xbb,
  0xe6, 0x83, 0xbc, 0xe6, 0x83, 0xbd, 0xe6, 0x83, 0xbe, 0xe6, 0x83, 0xbf,
  0xe6, 0x84, 0x82, 0xe6, 0x84, 0x83, 0xe6, 0x84, 0x84, 0xe6, 0x84, 0x85,
  0xe6, 0x84, 0x87, 0xe6, 0x84, 0x8a, 0xe6, 0x84, 0x8b, 0xe6, 0x84, 0x8c,
  0xe6, 0x84, 0x90, 0xe6, 0x84, 0x91, 0xe6, 0x84, 0x92, 0xe6, 0x84, 0x93,
  0xe6, 0x84, 0x94, 0xe6, 0x84, 0x96, 0xe6, 0x84, 0x97, 0xe6, 0x84, 0x98,
  0xe6, 0x84, 0x99, 0xe6, 0x84, 0x9b, 0xe6, 0x84, 0x9c, 0xe6, 0x84, 0x9d,
  0xe6, 0x84, 0x9e, 0xe6, 0x84, 0xa1, 0xe6, 0x84, 0xa2, 0xe6, 0x84, 0xa5,
  0xe6, 0x84, 0xa8, 0xe6, 0x84, 0xa9, 0xe6, 0x84, 0xaa, 0xe6, 0x84, 0xac,
  0xe6, 0x84, 0xad, 0xe6, 0x84, 0xae, 0xe6, 0x84, 0xaf, 0xe6, 0x84, 0xb0,
  0xe6, 0x84, 0xb1, 0xe6, 0x84, 0xb2, 0xe6, 0x84, 0xb3, 0xe6, 0x84, 0xb4,
  0xe6, 0x84, 0xb5, 0xe6, 0x84, 0xb6, 0xe6, 0x84, 0xb7, 0xe6, 0x84, 0xb8,
  0xe6, 0x84, 0xb9, 0xe6, 0x84, 0xba, 0xe6, 0x84, 0xbb, 0xe6, 0x84, 0xbc,
  0xe6, 0x84, 0xbd, 0xe6, 0x84, 0xbe, 0xe6, 0x85, 0x80, 0xe6, 0x85, 0x81,
  0xe6, 0x85, 0x82, 0xe6, 0x85, 0x83, 0xe6, 0x85, 0x84, 0xe6, 0x85, 0x85,
  0xe6, 0x85, 0x86
};

unsigned char gbk_row_0x91_converted_to_utf8[] = {
  0xe6, 0x85, 0x87, 0xe6, 0x85, 0x89, 0xe6, 0x85,
  0x8b, 0xe6, 0x85, 0x8d, 0xe6, 0x85, 0x8f, 0xe6, 0x85, 0x90, 0xe6, 0x85,
  0x92, 0xe6, 0x85, 0x93, 0xe6, 0x85, 0x94, 0xe6, 0x85, 0x96, 0xe6, 0x85,
  0x97, 0xe6, 0x85, 0x98, 0xe6, 0x85, 0x99, 0xe6, 0x85, 0x9a, 0xe6, 0x85,
  0x9b, 0xe6, 0x85, 0x9c, 0xe6, 0x85, 0x9e, 0xe6, 0x85, 0x9f, 0xe6, 0x85,
  0xa0, 0xe6, 0x85, 0xa1, 0xe6, 0x85, 0xa3, 0xe6, 0x85, 0xa4, 0xe6, 0x85,
  0xa5, 0xe6, 0x85, 0xa6, 0xe6, 0x85, 0xa9, 0xe6, 0x85, 0xaa, 0xe6, 0x85,
  0xab, 0xe6, 0x85, 0xac, 0xe6, 0x85, 0xad, 0xe6, 0x85, 0xae, 0xe6, 0x85,
  0xaf, 0xe6, 0x85, 0xb1, 0xe6, 0x85, 0xb2, 0xe6, 0x85, 0xb3, 0xe6, 0x85,
  0xb4, 0xe6, 0x85, 0xb6, 0xe6, 0x85, 0xb8, 0xe6, 0x85, 0xb9, 0xe6, 0x85,
  0xba, 0xe6, 0x85, 0xbb, 0xe6, 0x85, 0xbc, 0xe6, 0x85, 0xbd, 0xe6, 0x85,
  0xbe, 0xe6, 0x85, 0xbf, 0xe6, 0x86, 0x80, 0xe6, 0x86, 0x81, 0xe6, 0x86,
  0x82, 0xe6, 0x86, 0x83, 0xe6, 0x86, 0x84, 0xe6, 0x86, 0x85, 0xe6, 0x86,
  0x86, 0xe6, 0x86, 0x87, 0xe6, 0x86, 0x88, 0xe6, 0x86, 0x89, 0xe6, 0x86,
  0x8a, 0xe6, 0x86, 0x8c, 0xe6, 0x86, 0x8d, 0xe6, 0x86, 0x8f, 0xe6, 0x86,
  0x90, 0xe6, 0x86, 0x91, 0xe6, 0x86, 0x92, 0xe6, 0x86, 0x93, 0xe6, 0x86,
  0x95, 0xe6, 0x86, 0x96, 0xe6, 0x86, 0x97, 0xe6, 0x86, 0x98, 0xe6, 0x86,
  0x99, 0xe6, 0x86, 0x9a, 0xe6, 0x86, 0x9b, 0xe6, 0x86, 0x9c, 0xe6, 0x86,
  0x9e, 0xe6, 0x86, 0x9f, 0xe6, 0x86, 0xa0, 0xe6, 0x86, 0xa1, 0xe6, 0x86,
  0xa2, 0xe6, 0x86, 0xa3, 0xe6, 0x86, 0xa4, 0xe6, 0x86, 0xa5, 0xe6, 0x86,
  0xa6, 0xe6, 0x86, 0xaa, 0xe6, 0x86, 0xab, 0xe6, 0x86, 0xad, 0xe6, 0x86,
  0xae, 0xe6, 0x86, 0xaf, 0xe6, 0x86, 0xb0, 0xe6, 0x86, 0xb1, 0xe6, 0x86,
  0xb2, 0xe6, 0x86, 0xb3, 0xe6, 0x86, 0xb4, 0xe6, 0x86, 0xb5, 0xe6, 0x86,
  0xb6, 0xe6, 0x86, 0xb8, 0xe6, 0x86, 0xb9, 0xe6, 0x86, 0xba, 0xe6, 0x86,
  0xbb, 0xe6, 0x86, 0xbc, 0xe6, 0x86, 0xbd, 0xe6, 0x86, 0xbf, 0xe6, 0x87,
  0x80, 0xe6, 0x87, 0x81, 0xe6, 0x87, 0x83, 0xe6, 0x87, 0x84, 0xe6, 0x87,
  0x85, 0xe6, 0x87, 0x86, 0xe6, 0x87, 0x87, 0xe6, 0x87, 0x89, 0xe6, 0x87,
  0x8c, 0xe6, 0x87, 0x8d, 0xe6, 0x87, 0x8e, 0xe6, 0x87, 0x8f, 0xe6, 0x87,
  0x90, 0xe6, 0x87, 0x93, 0xe6, 0x87, 0x95, 0xe6, 0x87, 0x96, 0xe6, 0x87,
  0x97, 0xe6, 0x87, 0x98, 0xe6, 0x87, 0x99, 0xe6, 0x87, 0x9a, 0xe6, 0x87,
  0x9b, 0xe6, 0x87, 0x9c, 0xe6, 0x87, 0x9d, 0xe6, 0x87, 0x9e, 0xe6, 0x87,
  0x9f, 0xe6, 0x87, 0xa0, 0xe6, 0x87, 0xa1, 0xe6, 0x87, 0xa2, 0xe6, 0x87,
  0xa3, 0xe6, 0x87, 0xa4, 0xe6, 0x87, 0xa5, 0xe6, 0x87, 0xa7, 0xe6, 0x87,
  0xa8, 0xe6, 0x87, 0xa9, 0xe6, 0x87, 0xaa, 0xe6, 0x87, 0xab, 0xe6, 0x87,
  0xac, 0xe6, 0x87, 0xad, 0xe6, 0x87, 0xae, 0xe6, 0x87, 0xaf, 0xe6, 0x87,
  0xb0, 0xe6, 0x87, 0xb1, 0xe6, 0x87, 0xb2, 0xe6, 0x87, 0xb3, 0xe6, 0x87,
  0xb4, 0xe6, 0x87, 0xb6, 0xe6, 0x87, 0xb7, 0xe6, 0x87, 0xb8, 0xe6, 0x87,
  0xb9, 0xe6, 0x87, 0xba, 0xe6, 0x87, 0xbb, 0xe6, 0x87, 0xbc, 0xe6, 0x87,
  0xbd, 0xe6, 0x87, 0xbe, 0xe6, 0x88, 0x80, 0xe6, 0x88, 0x81, 0xe6, 0x88,
  0x82, 0xe6, 0x88, 0x83, 0xe6, 0x88, 0x84, 0xe6, 0x88, 0x85, 0xe6, 0x88,
  0x87, 0xe6, 0x88, 0x89, 0xe6, 0x88, 0x93, 0xe6, 0x88, 0x94, 0xe6, 0x88,
  0x99, 0xe6, 0x88, 0x9c, 0xe6, 0x88, 0x9d, 0xe6, 0x88, 0x9e, 0xe6, 0x88,
  0xa0, 0xe6, 0x88, 0xa3, 0xe6, 0x88, 0xa6, 0xe6, 0x88, 0xa7, 0xe6, 0x88,
  0xa8, 0xe6, 0x88, 0xa9, 0xe6, 0x88, 0xab, 0xe6, 0x88, 0xad, 0xe6, 0x88,
  0xaf, 0xe6, 0x88, 0xb0, 0xe6, 0x88, 0xb1, 0xe6, 0x88, 0xb2, 0xe6, 0x88,
  0xb5, 0xe6, 0x88, 0xb6, 0xe6, 0x88, 0xb8, 0xe6, 0x88, 0xb9, 0xe6, 0x88,
  0xba, 0xe6, 0x88, 0xbb, 0xe6, 0x88, 0xbc, 0xe6, 0x89, 0x82, 0xe6, 0x89,
  0x84, 0xe6, 0x89, 0x85, 0xe6, 0x89, 0x86, 0xe6, 0x89, 0x8a
};

unsigned char gbk_row_0x92_converted_to_utf8[] = {
  0xe6,
  0x89, 0x8f, 0xe6, 0x89, 0x90, 0xe6, 0x89, 0x95, 0xe6, 0x89, 0x96, 0xe6,
  0x89, 0x97, 0xe6, 0x89, 0x99, 0xe6, 0x89, 0x9a, 0xe6, 0x89, 0x9c, 0xe6,
  0x89, 0x9d, 0xe6, 0x89, 0x9e, 0xe6, 0x89, 0x9f, 0xe6, 0x89, 0xa0, 0xe6,
  0x89, 0xa1, 0xe6, 0x89, 0xa2, 0xe6, 0x89, 0xa4, 0xe6, 0x89, 0xa5, 0xe6,
  0x89, 0xa8, 0xe6, 0x89, 0xb1, 0xe6, 0x89, 0xb2, 0xe6, 0x89, 0xb4, 0xe6,
  0x89, 0xb5, 0xe6, 0x89, 0xb7, 0xe6, 0x89, 0xb8, 0xe6, 0x89, 0xba, 0xe6,
  0x89, 0xbb, 0xe6, 0x89, 0xbd, 0xe6, 0x8a, 0x81, 0xe6, 0x8a, 0x82, 0xe6,
  0x8a, 0x83, 0xe6, 0x8a, 0x85, 0xe6, 0x8a, 0x86, 0xe6, 0x8a, 0x87, 0xe6,
  0x8a, 0x88, 0xe6, 0x8a, 0x8b, 0xe6, 0x8a, 0x8c, 0xe6, 0x8a, 0x8d, 0xe6,
  0x8a, 0x8e, 0xe6, 0x8a, 0x8f, 0xe6, 0x8a, 0x90, 0xe6, 0x8a, 0x94, 0xe6,
  0x8a, 0x99, 0xe6, 0x8a, 0x9c, 0xe6, 0x8a, 0x9d, 0xe6, 0x8a, 0x9e, 0xe6,
  0x8a, 0xa3, 0xe6, 0x8a, 0xa6, 0xe6, 0x8a, 0xa7, 0xe6, 0x8a, 0xa9, 0xe6,
  0x8a, 0xaa, 0xe6, 0x8a, 0xad, 0xe6, 0x8a, 0xae, 0xe6, 0x8a, 0xaf, 0xe6,
  0x8a, 0xb0, 0xe6, 0x8a, 0xb2, 0xe6, 0x8a, 0xb3, 0xe6, 0x8a, 0xb4, 0xe6,
  0x8a, 0xb6, 0xe6, 0x8a, 0xb7, 0xe6, 0x8a, 0xb8, 0xe6, 0x8a, 0xba, 0xe6,
  0x8a, 0xbe, 0xe6, 0x8b, 0x80, 0xe6, 0x8b, 0x81, 0xe6, 0x8b, 0x83, 0xe6,
  0x8b, 0x8b, 0xe6, 0x8b, 0x8f, 0xe6, 0x8b, 0x91, 0xe6, 0x8b, 0x95, 0xe6,
  0x8b, 0x9d, 0xe6, 0x8b, 0x9e, 0xe6, 0x8b, 0xa0, 0xe6, 0x8b, 0xa1, 0xe6,
  0x8b, 0xa4, 0xe6, 0x8b, 0xaa, 0xe6, 0x8b, 0xab, 0xe6, 0x8b, 0xb0, 0xe6,
  0x8b, 0xb2, 0xe6, 0x8b, 0xb5, 0xe6, 0x8b, 0xb8, 0xe6, 0x8b, 0xb9, 0xe6,
  0x8b, 0xba, 0xe6, 0x8b, 0xbb, 0xe6, 0x8c, 0x80, 0xe6, 0x8c, 0x83, 0xe6,
  0x8c, 0x84, 0xe6, 0x8c, 0x85, 0xe6, 0x8c, 0x86, 0xe6, 0x8c, 0x8a, 0xe6,
  0x8c, 0x8b, 0xe6, 0x8c, 0x8c, 0xe6, 0x8c, 0x8d, 0xe6, 0x8c, 0x8f, 0xe6,
  0x8c, 0x90, 0xe6, 0x8c, 0x92, 0xe6, 0x8c, 0x93, 0xe6, 0x8c, 0x94, 0xe6,
  0x8c, 0x95, 0xe6, 0x8c, 0x97, 0xe6, 0x8c, 0x98, 0xe6, 0x8c, 0x99, 0xe6,
  0x8c, 0x9c, 0xe6, 0x8c, 0xa6, 0xe6, 0x8c, 0xa7, 0xe6, 0x8c, 0xa9, 0xe6,
  0x8c, 0xac, 0xe6, 0x8c, 0xad, 0xe6, 0x8c, 0xae, 0xe6, 0x8c, 0xb0, 0xe6,
  0x8c, 0xb1, 0xe6, 0x8c, 0xb3, 0xe6, 0x8c, 0xb4, 0xe6, 0x8c, 0xb5, 0xe6,
  0x8c, 0xb6, 0xe6, 0x8c, 0xb7, 0xe6, 0x8c, 0xb8, 0xe6, 0x8c, 0xbb, 0xe6,
  0x8c, 0xbc, 0xe6, 0x8c, 0xbe, 0xe6, 0x8c, 0xbf, 0xe6, 0x8d, 0x80, 0xe6,
  0x8d, 0x81, 0xe6, 0x8d, 0x84, 0xe6, 0x8d, 0x87, 0xe6, 0x8d, 0x88, 0xe6,
  0x8d, 0x8a, 0xe6, 0x8d, 0x91, 0xe6, 0x8d, 0x92, 0xe6, 0x8d, 0x93, 0xe6,
  0x8d, 0x94, 0xe6, 0x8d, 0x96, 0xe6, 0x8d, 0x97, 0xe6, 0x8d, 0x98, 0xe6,
  0x8d, 0x99, 0xe6, 0x8d, 0x9a, 0xe6, 0x8d, 0x9b, 0xe6, 0x8d, 0x9c, 0xe6,
  0x8d, 0x9d, 0xe6, 0x8d, 0xa0, 0xe6, 0x8d, 0xa4, 0xe6, 0x8d, 0xa5, 0xe6,
  0x8d, 0xa6, 0xe6, 0x8d, 0xa8, 0xe6, 0x8d, 0xaa, 0xe6, 0x8d, 0xab, 0xe6,
  0x8d, 0xac, 0xe6, 0x8d, 0xaf, 0xe6, 0x8d, 0xb0, 0xe6, 0x8d, 0xb2, 0xe6,
  0x8d, 0xb3, 0xe6, 0x8d, 0xb4, 0xe6, 0x8d, 0xb5, 0xe6, 0x8d, 0xb8, 0xe6,
  0x8d, 0xb9, 0xe6, 0x8d, 0xbc, 0xe6, 0x8d, 0xbd, 0xe6, 0x8d, 0xbe, 0xe6,
  0x8d, 0xbf, 0xe6, 0x8e, 0x81, 0xe6, 0x8e, 0x83, 0xe6, 0x8e, 0x84, 0xe6,
  0x8e, 0x85, 0xe6, 0x8e, 0x86, 0xe6, 0x8e, 0x8b, 0xe6, 0x8e, 0x8d, 0xe6,
  0x8e, 0x91, 0xe6, 0x8e, 0x93, 0xe6, 0x8e, 0x94, 0xe6, 0x8e, 0x95, 0xe6,
  0x8e, 0x97, 0xe6, 0x8e, 0x99, 0xe6, 0x8e, 0x9a, 0xe6, 0x8e, 0x9b, 0xe6,
  0x8e, 0x9c, 0xe6, 0x8e, 0x9d, 0xe6, 0x8e, 0x9e, 0xe6, 0x8e, 0x9f, 0xe6,
  0x8e, 0xa1, 0xe6, 0x8e, 0xa4, 0xe6, 0x8e, 0xa6, 0xe6, 0x8e, 0xab, 0xe6,
  0x8e, 0xaf, 0xe6, 0x8e, 0xb1, 0xe6, 0x8e, 0xb2, 0xe6, 0x8e, 0xb5, 0xe6,
  0x8e, 0xb6, 0xe6, 0x8e, 0xb9, 0xe6, 0x8e, 0xbb, 0xe6, 0x8e, 0xbd, 0xe6,
  0x8e, 0xbf, 0xe6, 0x8f, 0x80
};

unsigned char gbk_row_0x93_converted_to_utf8[] = {
  0xe6, 0x8f, 0x81, 0xe6, 0x8f, 0x82,
  0xe6, 0x8f, 0x83, 0xe6, 0x8f, 0x85, 0xe6, 0x8f, 0x87, 0xe6, 0x8f, 0x88,
  0xe6, 0x8f, 0x8a, 0xe6, 0x8f, 0x8b, 0xe6, 0x8f, 0x8c, 0xe6, 0x8f, 0x91,
  0xe6, 0x8f, 0x93, 0xe6, 0x8f, 0x94, 0xe6, 0x8f, 0x95, 0xe6, 0x8f, 0x97,
  0xe6, 0x8f, 0x98, 0xe6, 0x8f, 0x99, 0xe6, 0x8f, 0x9a, 0xe6, 0x8f, 0x9b,
  0xe6, 0x8f, 0x9c, 0xe6, 0x8f, 0x9d, 0xe6, 0x8f, 0x9f, 0xe6, 0x8f, 0xa2,
  0xe6, 0x8f, 0xa4, 0xe6, 0x8f, 0xa5, 0xe6, 0x8f, 0xa6, 0xe6, 0x8f, 0xa7,
  0xe6, 0x8f, 0xa8, 0xe6, 0x8f, 0xab, 0xe6, 0x8f, 0xac, 0xe6, 0x8f, 0xae,
  0xe6, 0x8f, 0xaf, 0xe6, 0x8f, 0xb0, 0xe6, 0x8f, 0xb1, 0xe6, 0x8f, 0xb3,
  0xe6, 0x8f, 0xb5, 0xe6, 0x8f, 0xb7, 0xe6, 0x8f, 0xb9, 0xe6, 0x8f, 0xba,
  0xe6, 0x8f, 0xbb, 0xe6, 0x8f, 0xbc, 0xe6, 0x8f, 0xbe, 0xe6, 0x90, 0x83,
  0xe6, 0x90, 0x84, 0xe6, 0x90, 0x86, 0xe6, 0x90, 0x87, 0xe6, 0x90, 0x88,
  0xe6, 0x90, 0x89, 0xe6, 0x90, 0x8a, 0xe6, 0x90, 0x8d, 0xe6, 0x90, 0x8e,
  0xe6, 0x90, 0x91, 0xe6, 0x90, 0x92, 0xe6, 0x90, 0x95, 0xe6, 0x90, 0x96,
  0xe6, 0x90, 0x97, 0xe6, 0x90, 0x98, 0xe6, 0x90, 0x99, 0xe6, 0x90, 0x9a,
  0xe6, 0x90, 0x9d, 0xe6, 0x90, 0x9f, 0xe6, 0x90, 0xa2, 0xe6, 0x90, 0xa3,
  0xe6, 0x90, 0xa4, 0xe6, 0x90, 0xa5, 0xe6, 0x90, 0xa7, 0xe6, 0x90, 0xa8,
  0xe6, 0x90, 0xa9, 0xe6, 0x90, 0xab, 0xe6, 0x90, 0xae, 0xe6, 0x90, 0xaf,
  0xe6, 0x90, 0xb0, 0xe6, 0x90, 0xb1, 0xe6, 0x90, 0xb2, 0xe6, 0x90, 0xb3,
  0xe6, 0x90, 0xb5, 0xe6, 0x90, 0xb6, 0xe6, 0x90, 0xb7, 0xe6, 0x90, 0xb8,
  0xe6, 0x90, 0xb9, 0xe6, 0x90, 0xbb, 0xe6, 0x90, 0xbc, 0xe6, 0x90, 0xbe,
  0xe6, 0x91, 0x80, 0xe6, 0x91, 0x82, 0xe6, 0x91, 0x83, 0xe6, 0x91, 0x89,
  0xe6, 0x91, 0x8b, 0xe6, 0x91, 0x8c, 0xe6, 0x91, 0x8d, 0xe6, 0x91, 0x8e,
  0xe6, 0x91, 0x8f, 0xe6, 0x91, 0x90, 0xe6, 0x91, 0x91, 0xe6, 0x91, 0x93,
  0xe6, 0x91, 0x95, 0xe6, 0x91, 0x96, 0xe6, 0x91, 0x97, 0xe6, 0x91, 0x99,
  0xe6, 0x91, 0x9a, 0xe6, 0x91, 0x9b, 0xe6, 0x91, 0x9c, 0xe6, 0x91, 0x9d,
  0xe6, 0x91, 0x9f, 0xe6, 0x91, 0xa0, 0xe6, 0x91, 0xa1, 0xe6, 0x91, 0xa2,
  0xe6, 0x91, 0xa3, 0xe6, 0x91, 0xa4, 0xe6, 0x91, 0xa5, 0xe6, 0x91, 0xa6,
  0xe6, 0x91, 0xa8, 0xe6, 0x91, 0xaa, 0xe6, 0x91, 0xab, 0xe6, 0x91, 0xac,
  0xe6, 0x91, 0xae, 0xe6, 0x91, 0xaf, 0xe6, 0x91, 0xb0, 0xe6, 0x91, 0xb1,
  0xe6, 0x91, 0xb2, 0xe6, 0x91, 0xb3, 0xe6, 0x91, 0xb4, 0xe6, 0x91, 0xb5,
  0xe6, 0x91, 0xb6, 0xe6, 0x91, 0xb7, 0xe6, 0x91, 0xbb, 0xe6, 0x91, 0xbc,
  0xe6, 0x91, 0xbd, 0xe6, 0x91, 0xbe, 0xe6, 0x91, 0xbf, 0xe6, 0x92, 0x80,
  0xe6, 0x92, 0x81, 0xe6, 0x92, 0x83, 0xe6, 0x92, 0x86, 0xe6, 0x92, 0x88,
  0xe6, 0x92, 0x89, 0xe6, 0x92, 0x8a, 0xe6, 0x92, 0x8b, 0xe6, 0x92, 0x8c,
  0xe6, 0x92, 0x8d, 0xe6, 0x92, 0x8e, 0xe6, 0x92, 0x8f, 0xe6, 0x92, 0x90,
  0xe6, 0x92, 0x93, 0xe6, 0x92, 0x94, 0xe6, 0x92, 0x97, 0xe6, 0x92, 0x98,
  0xe6, 0x92, 0x9a, 0xe6, 0x92, 0x9b, 0xe6, 0x92, 0x9c, 0xe6, 0x92, 0x9d,
  0xe6, 0x92, 0x9f, 0xe6, 0x92, 0xa0, 0xe6, 0x92, 0xa1, 0xe6, 0x92, 0xa2,
  0xe6, 0x92, 0xa3, 0xe6, 0x92, 0xa5, 0xe6, 0x92, 0xa6, 0xe6, 0x92, 0xa7,
  0xe6, 0x92, 0xa8, 0xe6, 0x92, 0xaa, 0xe6, 0x92, 0xab, 0xe6, 0x92, 0xaf,
  0xe6, 0x92, 0xb1, 0xe6, 0x92, 0xb2, 0xe6, 0x92, 0xb3, 0xe6, 0x92, 0xb4,
  0xe6, 0x92, 0xb6, 0xe6, 0x92, 0xb9, 0xe6, 0x92, 0xbb, 0xe6, 0x92, 0xbd,
  0xe6, 0x92, 0xbe, 0xe6, 0x92, 0xbf, 0xe6, 0x93, 0x81, 0xe6, 0x93, 0x83,
  0xe6, 0x93, 0x84, 0xe6, 0x93, 0x86, 0xe6, 0x93, 0x87, 0xe6, 0x93, 0x88,
  0xe6, 0x93, 0x89, 0xe6, 0x93, 0x8a, 0xe6, 0x93, 0x8b, 0xe6, 0x93, 0x8c,
  0xe6, 0x93, 0x8f, 0xe6, 0x93, 0x91, 0xe6, 0x93, 0x93, 0xe6, 0x93, 0x94,
  0xe6, 0x93, 0x95, 0xe6, 0x93, 0x96, 0xe6, 0x93, 0x99, 0xe6, 0x93, 0x9a,
  };

unsigned char gbk_row_0x94_converted_to_utf8[] = {
  0xe6, 0x93, 0x9b, 0xe6, 0x93, 0x9c, 0xe6, 0x93, 0x9d, 0xe6, 0x93,
  0x9f, 0xe6, 0x93, 0xa0, 0xe6, 0x93, 0xa1, 0xe6, 0x93, 0xa3, 0xe6, 0x93,
  0xa5, 0xe6, 0x93, 0xa7, 0xe6, 0x93, 0xa8, 0xe6, 0x93, 0xa9, 0xe6, 0x93,
  0xaa, 0xe6, 0x93, 0xab, 0xe6, 0x93, 0xac, 0xe6, 0x93, 0xad, 0xe6, 0x93,
  0xae, 0xe6, 0x93, 0xaf, 0xe6, 0x93, 0xb0, 0xe6, 0x93, 0xb1, 0xe6, 0x93,
  0xb2, 0xe6, 0x93, 0xb3, 0xe6, 0x93, 0xb4, 0xe6, 0x93, 0xb5, 0xe6, 0x93,
  0xb6, 0xe6, 0x93, 0xb7, 0xe6, 0x93, 0xb8, 0xe6, 0x93, 0xb9, 0xe6, 0x93,
  0xba, 0xe6, 0x93, 0xbb, 0xe6, 0x93, 0xbc, 0xe6, 0x93, 0xbd, 0xe6, 0x93,
  0xbe, 0xe6, 0x93, 0xbf, 0xe6, 0x94, 0x81, 0xe6, 0x94, 0x82, 0xe6, 0x94,
  0x83, 0xe6, 0x94, 0x84, 0xe6, 0x94, 0x85, 0xe6, 0x94, 0x86, 0xe6, 0x94,
  0x87, 0xe6, 0x94, 0x88, 0xe6, 0x94, 0x8a, 0xe6, 0x94, 0x8b, 0xe6, 0x94,
  0x8c, 0xe6, 0x94, 0x8d, 0xe6, 0x94, 0x8e, 0xe6, 0x94, 0x8f, 0xe6, 0x94,
  0x90, 0xe6, 0x94, 0x91, 0xe6, 0x94, 0x93, 0xe6, 0x94, 0x94, 0xe6, 0x94,
  0x95, 0xe6, 0x94, 0x96, 0xe6, 0x94, 0x97, 0xe6, 0x94, 0x99, 0xe6, 0x94,
  0x9a, 0xe6, 0x94, 0x9b, 0xe6, 0x94, 0x9c, 0xe6, 0x94, 0x9d, 0xe6, 0x94,
  0x9e, 0xe6, 0x94, 0x9f, 0xe6, 0x94, 0xa0, 0xe6, 0x94, 0xa1, 0xe6, 0x94,
  0xa2, 0xe6, 0x94, 0xa3, 0xe6, 0x94, 0xa4, 0xe6, 0x94, 0xa6, 0xe6, 0x94,
  0xa7, 0xe6, 0x94, 0xa8, 0xe6, 0x94, 0xa9, 0xe6, 0x94, 0xaa, 0xe6, 0x94,
  0xac, 0xe6, 0x94, 0xad, 0xe6, 0x94, 0xb0, 0xe6, 0x94, 0xb1, 0xe6, 0x94,
  0xb2, 0xe6, 0x94, 0xb3, 0xe6, 0x94, 0xb7, 0xe6, 0x94, 0xba, 0xe6, 0x94,
  0xbc, 0xe6, 0x94, 0xbd, 0xe6, 0x95, 0x80, 0xe6, 0x95, 0x81, 0xe6, 0x95,
  0x82, 0xe6, 0x95, 0x83, 0xe6, 0x95, 0x84, 0xe6, 0x95, 0x86, 0xe6, 0x95,
  0x87, 0xe6, 0x95, 0x8a, 0xe6, 0x95, 0x8b, 0xe6, 0x95, 0x8d, 0xe6, 0x95,
  0x8e, 0xe6, 0x95, 0x90, 0xe6, 0x95, 0x92, 0xe6, 0x95, 0x93, 0xe6, 0x95,
  0x94, 0xe6, 0x95, 0x97, 0xe6, 0x95, 0x98, 0xe6, 0x95, 0x9a, 0xe6, 0x95,
  0x9c, 0xe6, 0x95, 0x9f, 0xe6, 0x95, 0xa0, 0xe6, 0x95, 0xa1, 0xe6, 0x95,
  0xa4, 0xe6, 0x95, 0xa5, 0xe6, 0x95, 0xa7, 0xe6, 0x95, 0xa8, 0xe6, 0x95,
  0xa9, 0xe6, 0x95, 0xaa, 0xe6, 0x95, 0xad, 0xe6, 0x95, 0xae, 0xe6, 0x95,
  0xaf, 0xe6, 0x95, 0xb1, 0xe6, 0x95, 0xb3, 0xe6, 0x95, 0xb5, 0xe6, 0x95,
  0xb6, 0xe6, 0x95, 0xb8, 0xe6, 0x95, 0xb9, 0xe6, 0x95, 0xba, 0xe6, 0x95,
  0xbb, 0xe6, 0x95, 0xbc, 0xe6, 0x95, 0xbd, 0xe6, 0x95, 0xbe, 0xe6, 0x95,
  0xbf, 0xe6, 0x96, 0x80, 0xe6, 0x96, 0x81, 0xe6, 0x96, 0x82, 0xe6, 0x96,
  0x83, 0xe6, 0x96, 0x84, 0xe6, 0x96, 0x85, 0xe6, 0x96, 0x86, 0xe6, 0x96,
  0x88, 0xe6, 0x96, 0x89, 0xe6, 0x96, 0x8a, 0xe6, 0x96, 0x8d, 0xe6, 0x96,
  0x8e, 0xe6, 0x96, 0x8f, 0xe6, 0x96, 0x92, 0xe6, 0x96, 0x94, 0xe6, 0x96,
  0x95, 0xe6, 0x96, 0x96, 0xe6, 0x96, 0x98, 0xe6, 0x96, 0x9a, 0xe6, 0x96,
  0x9d, 0xe6, 0x96, 0x9e, 0xe6, 0x96, 0xa0, 0xe6, 0x96, 0xa2, 0xe6, 0x96,
  0xa3, 0xe6, 0x96, 0xa6, 0xe6, 0x96, 0xa8, 0xe6, 0x96, 0xaa, 0xe6, 0x96,
  0xac, 0xe6, 0x96, 0xae, 0xe6, 0x96, 0xb1, 0xe6, 0x96, 0xb2, 0xe6, 0x96,
  0xb3, 0xe6, 0x96, 0xb4, 0xe6, 0x96, 0xb5, 0xe6, 0x96, 0xb6, 0xe6, 0x96,
  0xb7, 0xe6, 0x96, 0xb8, 0xe6, 0x96, 0xba, 0xe6, 0x96, 0xbb, 0xe6, 0x96,
  0xbe, 0xe6, 0x96, 0xbf, 0xe6, 0x97, 0x80, 0xe6, 0x97, 0x82, 0xe6, 0x97,
  0x87, 0xe6, 0x97, 0x88, 0xe6, 0x97, 0x89, 0xe6, 0x97, 0x8a, 0xe6, 0x97,
  0x8d, 0xe6, 0x97, 0x90, 0xe6, 0x97, 0x91, 0xe6, 0x97, 0x93, 0xe6, 0x97,
  0x94, 0xe6, 0x97, 0x95, 0xe6, 0x97, 0x98, 0xe6, 0x97, 0x99, 0xe6, 0x97,
  0x9a, 0xe6, 0x97, 0x9b, 0xe6, 0x97, 0x9c, 0xe6, 0x97, 0x9d, 0xe6, 0x97,
  0x9e, 0xe6, 0x97, 0x9f, 0xe6, 0x97, 0xa1, 0xe6, 0x97, 0xa3, 0xe6, 0x97,
  0xa4, 0xe6, 0x97, 0xaa, 0xe6, 0x97, 0xab
};

unsigned char gbk_row_0x95_converted_to_utf8[] = {
  0xe6, 0x97, 0xb2, 0xe6,
  0x97, 0xb3, 0xe6, 0x97, 0xb4, 0xe6, 0x97, 0xb5, 0xe6, 0x97, 0xb8, 0xe6,
  0x97, 0xb9, 0xe6, 0x97, 0xbb, 0xe6, 0x97, 0xbc, 0xe6, 0x97, 0xbd, 0xe6,
  0x97, 0xbe, 0xe6, 0x97, 0xbf, 0xe6, 0x98, 0x81, 0xe6, 0x98, 0x84, 0xe6,
  0x98, 0x85, 0xe6, 0x98, 0x87, 0xe6, 0x98, 0x88, 0xe6, 0x98, 0x89, 0xe6,
  0x98, 0x8b, 0xe6, 0x98, 0x8d, 0xe6, 0x98, 0x90, 0xe6, 0x98, 0x91, 0xe6,
  0x98, 0x92, 0xe6, 0x98, 0x96, 0xe6, 0x98, 0x97, 0xe6, 0x98, 0x98, 0xe6,
  0x98, 0x9a, 0xe6, 0x98, 0x9b, 0xe6, 0x98, 0x9c, 0xe6, 0x98, 0x9e, 0xe6,
  0x98, 0xa1, 0xe6, 0x98, 0xa2, 0xe6, 0x98, 0xa3, 0xe6, 0x98, 0xa4, 0xe6,
  0x98, 0xa6, 0xe6, 0x98, 0xa9, 0xe6, 0x98, 0xaa, 0xe6, 0x98, 0xab, 0xe6,
  0x98, 0xac, 0xe6, 0x98, 0xae, 0xe6, 0x98, 0xb0, 0xe6, 0x98, 0xb2, 0xe6,
  0x98, 0xb3, 0xe6, 0x98, 0xb7, 0xe6, 0x98, 0xb8, 0xe6, 0x98, 0xb9, 0xe6,
  0x98, 0xba, 0xe6, 0x98, 0xbb, 0xe6, 0x98, 0xbd, 0xe6, 0x98, 0xbf, 0xe6,
  0x99, 0x80, 0xe6, 0x99, 0x82, 0xe6, 0x99, 0x84, 0xe6, 0x99, 0x85, 0xe6,
  0x99, 0x86, 0xe6, 0x99, 0x87, 0xe6, 0x99, 0x88, 0xe6, 0x99, 0x89, 0xe6,
  0x99, 0x8a, 0xe6, 0x99, 0x8d, 0xe6, 0x99, 0x8e, 0xe6, 0x99, 0x90, 0xe6,
  0x99, 0x91, 0xe6, 0x99, 0x98, 0xe6, 0x99, 0x99, 0xe6, 0x99, 0x9b, 0xe6,
  0x99, 0x9c, 0xe6, 0x99, 0x9d, 0xe6, 0x99, 0x9e, 0xe6, 0x99, 0xa0, 0xe6,
  0x99, 0xa2, 0xe6, 0x99, 0xa3, 0xe6, 0x99, 0xa5, 0xe6, 0x99, 0xa7, 0xe6,
  0x99, 0xa9, 0xe6, 0x99, 0xaa, 0xe6, 0x99, 0xab, 0xe6, 0x99, 0xac, 0xe6,
  0x99, 0xad, 0xe6, 0x99, 0xb1, 0xe6, 0x99, 0xb2, 0xe6, 0x99, 0xb3, 0xe6,
  0x99, 0xb5, 0xe6, 0x99, 0xb8, 0xe6, 0x99, 0xb9, 0xe6, 0x99, 0xbb, 0xe6,
  0x99, 0xbc, 0xe6, 0x99, 0xbd, 0xe6, 0x99, 0xbf, 0xe6, 0x9a, 0x80, 0xe6,
  0x9a, 0x81, 0xe6, 0x9a, 0x83, 0xe6, 0x9a, 0x85, 0xe6, 0x9a, 0x86, 0xe6,
  0x9a, 0x88, 0xe6, 0x9a, 0x89, 0xe6, 0x9a, 0x8a, 0xe6, 0x9a, 0x8b, 0xe6,
  0x9a, 0x8d, 0xe6, 0x9a, 0x8e, 0xe6, 0x9a, 0x8f, 0xe6, 0x9a, 0x90, 0xe6,
  0x9a, 0x92, 0xe6, 0x9a, 0x93, 0xe6, 0x9a, 0x94, 0xe6, 0x9a, 0x95, 0xe6,
  0x9a, 0x98, 0xe6, 0x9a, 0x99, 0xe6, 0x9a, 0x9a, 0xe6, 0x9a, 0x9b, 0xe6,
  0x9a, 0x9c, 0xe6, 0x9a, 0x9e, 0xe6, 0x9a, 0x9f, 0xe6, 0x9a, 0xa0, 0xe6,
  0x9a, 0xa1, 0xe6, 0x9a, 0xa2, 0xe6, 0x9a, 0xa3, 0xe6, 0x9a, 0xa4, 0xe6,
  0x9a, 0xa5, 0xe6, 0x9a, 0xa6, 0xe6, 0x9a, 0xa9, 0xe6, 0x9a, 0xaa, 0xe6,
  0x9a, 0xab, 0xe6, 0x9a, 0xac, 0xe6, 0x9a, 0xad, 0xe6, 0x9a, 0xaf, 0xe6,
  0x9a, 0xb0, 0xe6, 0x9a, 0xb1, 0xe6, 0x9a, 0xb2, 0xe6, 0x9a, 0xb3, 0xe6,
  0x9a, 0xb5, 0xe6, 0x9a, 0xb6, 0xe6, 0x9a, 0xb7, 0xe6, 0x9a, 0xb8, 0xe6,
  0x9a, 0xba, 0xe6, 0x9a, 0xbb, 0xe6, 0x9a, 0xbc, 0xe6, 0x9a, 0xbd, 0xe6,
  0x9a, 0xbf, 0xe6, 0x9b, 0x80, 0xe6, 0x9b, 0x81, 0xe6, 0x9b, 0x82, 0xe6,
  0x9b, 0x83, 0xe6, 0x9b, 0x84, 0xe6, 0x9b, 0x85, 0xe6, 0x9b, 0x86, 0xe6,
  0x9b, 0x87, 0xe6, 0x9b, 0x88, 0xe6, 0x9b, 0x89, 0xe6, 0x9b, 0x8a, 0xe6,
  0x9b, 0x8b, 0xe6, 0x9b, 0x8c, 0xe6, 0x9b, 0x8d, 0xe6, 0x9b, 0x8e, 0xe6,
  0x9b, 0x8f, 0xe6, 0x9b, 0x90, 0xe6, 0x9b, 0x91, 0xe6, 0x9b, 0x92, 0xe6,
  0x9b, 0x93, 0xe6, 0x9b, 0x94, 0xe6, 0x9b, 0x95, 0xe6, 0x9b, 0x96, 0xe6,
  0x9b, 0x97, 0xe6, 0x9b, 0x98, 0xe6, 0x9b, 0x9a, 0xe6, 0x9b, 0x9e, 0xe6,
  0x9b, 0x9f, 0xe6, 0x9b, 0xa0, 0xe6, 0x9b, 0xa1, 0xe6, 0x9b, 0xa2, 0xe6,
  0x9b, 0xa3, 0xe6, 0x9b, 0xa4, 0xe6, 0x9b, 0xa5, 0xe6, 0x9b, 0xa7, 0xe6,
  0x9b, 0xa8, 0xe6, 0x9b, 0xaa, 0xe6, 0x9b, 0xab, 0xe6, 0x9b, 0xac, 0xe6,
  0x9b, 0xad, 0xe6, 0x9b, 0xae, 0xe6, 0x9b, 0xaf, 0xe6, 0x9b, 0xb1, 0xe6,
  0x9b, 0xb5, 0xe6, 0x9b, 0xb6, 0xe6, 0x9b, 0xb8, 0xe6, 0x9b, 0xba, 0xe6,
  0x9b, 0xbb, 0xe6, 0x9b, 0xbd, 0xe6, 0x9c, 0x81, 0xe6, 0x9c, 0x82, 0xe6,
  0x9c, 0x83
};

unsigned char gbk_row_0x96_converted_to_utf8[] = {
  0xe6, 0x9c, 0x84, 0xe6, 0x9c, 0x85, 0xe6, 0x9c, 0x86,
  0xe6, 0x9c, 0x87, 0xe6, 0x9c, 0x8c, 0xe6, 0x9c, 0x8e, 0xe6, 0x9c, 0x8f,
  0xe6, 0x9c, 0x91, 0xe6, 0x9c, 0x92, 0xe6, 0x9c, 0x93, 0xe6, 0x9c, 0x96,
  0xe6, 0x9c, 0x98, 0xe6, 0x9c, 0x99, 0xe6, 0x9c, 0x9a, 0xe6, 0x9c, 0x9c,
  0xe6, 0x9c, 0x9e, 0xe6, 0x9c, 0xa0, 0xe6, 0x9c, 0xa1, 0xe6, 0x9c, 0xa2,
  0xe6, 0x9c, 0xa3, 0xe6, 0x9c, 0xa4, 0xe6, 0x9c, 0xa5, 0xe6, 0x9c, 0xa7,
  0xe6, 0x9c, 0xa9, 0xe6, 0x9c, 0xae, 0xe6, 0x9c, 0xb0, 0xe6, 0x9c, 0xb2,
  0xe6, 0x9c, 0xb3, 0xe6, 0x9c, 0xb6, 0xe6, 0x9c, 0xb7, 0xe6, 0x9c, 0xb8,
  0xe6, 0x9c, 0xb9, 0xe6, 0x9c, 0xbb, 0xe6, 0x9c, 0xbc, 0xe6, 0x9c, 0xbe,
  0xe6, 0x9c, 0xbf, 0xe6, 0x9d, 0x81, 0xe6, 0x9d, 0x84, 0xe6, 0x9d, 0x85,
  0xe6, 0x9d, 0x87, 0xe6, 0x9d, 0x8a, 0xe6, 0x9d, 0x8b, 0xe6, 0x9d, 0x8d,
  0xe6, 0x9d, 0x92, 0xe6, 0x9d, 0x94, 0xe6, 0x9d, 0x95, 0xe6, 0x9d, 0x97,
  0xe6, 0x9d, 0x98, 0xe6, 0x9d, 0x99, 0xe6, 0x9d, 0x9a, 0xe6, 0x9d, 0x9b,
  0xe6, 0x9d, 0x9d, 0xe6, 0x9d, 0xa2, 0xe6, 0x9d, 0xa3, 0xe6, 0x9d, 0xa4,
  0xe6, 0x9d, 0xa6, 0xe6, 0x9d, 0xa7, 0xe6, 0x9d, 0xab, 0xe6, 0x9d, 0xac,
  0xe6, 0x9d, 0xae, 0xe6, 0x9d, 0xb1, 0xe6, 0x9d, 0xb4, 0xe6, 0x9d, 0xb6,
  0xe6, 0x9d, 0xb8, 0xe6, 0x9d, 0xb9, 0xe6, 0x9d, 0xba, 0xe6, 0x9d, 0xbb,
  0xe6, 0x9d, 0xbd, 0xe6, 0x9e, 0x80, 0xe6, 0x9e, 0x82, 0xe6, 0x9e, 0x83,
  0xe6, 0x9e, 0x85, 0xe6, 0x9e, 0x86, 0xe6, 0x9e, 0x88, 0xe6, 0x9e, 0x8a,
  0xe6, 0x9e, 0x8c, 0xe6, 0x9e, 0x8d, 0xe6, 0x9e, 0x8e, 0xe6, 0x9e, 0x8f,
  0xe6, 0x9e, 0x91, 0xe6, 0x9e, 0x92, 0xe6, 0x9e, 0x93, 0xe6, 0x9e, 0x94,
  0xe6, 0x9e, 0x96, 0xe6, 0x9e, 0x99, 0xe6, 0x9e, 0x9b, 0xe6, 0x9e, 0x9f,
  0xe6, 0x9e, 0xa0, 0xe6, 0x9e, 0xa1, 0xe6, 0x9e, 0xa4, 0xe6, 0x9e, 0xa6,
  0xe6, 0x9e, 0xa9, 0xe6, 0x9e, 0xac, 0xe6, 0x9e, 0xae, 0xe6, 0x9e, 0xb1,
  0xe6, 0x9e, 0xb2, 0xe6, 0x9e, 0xb4, 0xe6, 0x9e, 0xb9, 0xe6, 0x9e, 0xba,
  0xe6, 0x9e, 0xbb, 0xe6, 0x9e, 0xbc, 0xe6, 0x9e, 0xbd, 0xe6, 0x9e, 0xbe,
  0xe6, 0x9e, 0xbf, 0xe6, 0x9f, 0x80, 0xe6, 0x9f, 0x82, 0xe6, 0x9f, 0x85,
  0xe6, 0x9f, 0x86, 0xe6, 0x9f, 0x87, 0xe6, 0x9f, 0x88, 0xe6, 0x9f, 0x89,
  0xe6, 0x9f, 0x8a, 0xe6, 0x9f, 0x8b, 0xe6, 0x9f, 0x8c, 0xe6, 0x9f, 0x8d,
  0xe6, 0x9f, 0x8e, 0xe6, 0x9f, 0x95, 0xe6, 0x9f, 0x96, 0xe6, 0x9f, 0x97,
  0xe6, 0x9f, 0x9b, 0xe6, 0x9f, 0x9f, 0xe6, 0x9f, 0xa1, 0xe6, 0x9f, 0xa3,
  0xe6, 0x9f, 0xa4, 0xe6, 0x9f, 0xa6, 0xe6, 0x9f, 0xa7, 0xe6, 0x9f, 0xa8,
  0xe6, 0x9f, 0xaa, 0xe6, 0x9f, 0xab, 0xe6, 0x9f, 0xad, 0xe6, 0x9f, 0xae,
  0xe6, 0x9f, 0xb2, 0xe6, 0x9f, 0xb5, 0xe6, 0x9f, 0xb6, 0xe6, 0x9f, 0xb7,
  0xe6, 0x9f, 0xb8, 0xe6, 0x9f, 0xb9, 0xe6, 0x9f, 0xba, 0xe6, 0x9f, 0xbb,
  0xe6, 0x9f, 0xbc, 0xe6, 0x9f, 0xbe, 0xe6, 0xa0, 0x81, 0xe6, 0xa0, 0x82,
  0xe6, 0xa0, 0x83, 0xe6, 0xa0, 0x84, 0xe6, 0xa0, 0x86, 0xe6, 0xa0, 0x8d,
  0xe6, 0xa0, 0x90, 0xe6, 0xa0, 0x92, 0xe6, 0xa0, 0x94, 0xe6, 0xa0, 0x95,
  0xe6, 0xa0, 0x98, 0xe6, 0xa0, 0x99, 0xe6, 0xa0, 0x9a, 0xe6, 0xa0, 0x9b,
  0xe6, 0xa0, 0x9c, 0xe6, 0xa0, 0x9e, 0xe6, 0xa0, 0x9f, 0xe6, 0xa0, 0xa0,
  0xe6, 0xa0, 0xa2, 0xe6, 0xa0, 0xa3, 0xe6, 0xa0, 0xa4, 0xe6, 0xa0, 0xa5,
  0xe6, 0xa0, 0xa6, 0xe6, 0xa0, 0xa7, 0xe6, 0xa0, 0xa8, 0xe6, 0xa0, 0xab,
  0xe6, 0xa0, 0xac, 0xe6, 0xa0, 0xad, 0xe6, 0xa0, 0xae, 0xe6, 0xa0, 0xaf,
  0xe6, 0xa0, 0xb0, 0xe6, 0xa0, 0xb1, 0xe6, 0xa0, 0xb4, 0xe6, 0xa0, 0xb5,
  0xe6, 0xa0, 0xb6, 0xe6, 0xa0, 0xba, 0xe6, 0xa0, 0xbb, 0xe6, 0xa0, 0xbf,
  0xe6, 0xa1, 0x87, 0xe6, 0xa1, 0x8b, 0xe6, 0xa1, 0x8d, 0xe6, 0xa1, 0x8f,
  0xe6, 0xa1, 0x92, 0xe6, 0xa1, 0x96, 0xe6, 0xa1, 0x97, 0xe6, 0xa1, 0x98,
  0xe6, 0xa1, 0x99, 0xe6, 0xa1, 0x9a, 0xe6, 0xa1, 0x9b
};

unsigned char gbk_row_0x97_converted_to_utf8[] = {
  0xe6, 0xa1,
  0x9c, 0xe6, 0xa1, 0x9d, 0xe6, 0xa1, 0x9e, 0xe6, 0xa1, 0x9f, 0xe6, 0xa1,
  0xaa, 0xe6, 0xa1, 0xac, 0xe6, 0xa1, 0xad, 0xe6, 0xa1, 0xae, 0xe6, 0xa1,
  0xaf, 0xe6, 0xa1, 0xb0, 0xe6, 0xa1, 0xb1, 0xe6, 0xa1, 0xb2, 0xe6, 0xa1,
  0xb3, 0xe6, 0xa1, 0xb5, 0xe6, 0xa1, 0xb8, 0xe6, 0xa1, 0xb9, 0xe6, 0xa1,
  0xba, 0xe6, 0xa1, 0xbb, 0xe6, 0xa1, 0xbc, 0xe6, 0xa1, 0xbd, 0xe6, 0xa1,
  0xbe, 0xe6, 0xa1, 0xbf, 0xe6, 0xa2, 0x80, 0xe6, 0xa2, 0x82, 0xe6, 0xa2,
  0x84, 0xe6, 0xa2, 0x87, 0xe6, 0xa2, 0x88, 0xe6, 0xa2, 0x89, 0xe6, 0xa2,
  0x8a, 0xe6, 0xa2, 0x8b, 0xe6, 0xa2, 0x8c, 0xe6, 0xa2, 0x8d, 0xe6, 0xa2,
  0x8e, 0xe6, 0xa2, 0x90, 0xe6, 0xa2, 0x91, 0xe6, 0xa2, 0x92, 0xe6, 0xa2,
  0x94, 0xe6, 0xa2, 0x95, 0xe6, 0xa2, 0x96, 0xe6, 0xa2, 0x98, 0xe6, 0xa2,
  0x99, 0xe6, 0xa2, 0x9a, 0xe6, 0xa2, 0x9b, 0xe6, 0xa2, 0x9c, 0xe6, 0xa2,
  0x9d, 0xe6, 0xa2, 0x9e, 0xe6, 0xa2, 0x9f, 0xe6, 0xa2, 0xa0, 0xe6, 0xa2,
  0xa1, 0xe6, 0xa2, 0xa3, 0xe6, 0xa2, 0xa4, 0xe6, 0xa2, 0xa5, 0xe6, 0xa2,
  0xa9, 0xe6, 0xa2, 0xaa, 0xe6, 0xa2, 0xab, 0xe6, 0xa2, 0xac, 0xe6, 0xa2,
  0xae, 0xe6, 0xa2, 0xb1, 0xe6, 0xa2, 0xb2, 0xe6, 0xa2, 0xb4, 0xe6, 0xa2,
  0xb6, 0xe6, 0xa2, 0xb7, 0xe6, 0xa2, 0xb8, 0xe6, 0xa2, 0xb9, 0xe6, 0xa2,
  0xba, 0xe6, 0xa2, 0xbb, 0xe6, 0xa2, 0xbc, 0xe6, 0xa2, 0xbd, 0xe6, 0xa2,
  0xbe, 0xe6, 0xa2, 0xbf, 0xe6, 0xa3, 0x81, 0xe6, 0xa3, 0x83, 0xe6, 0xa3,
  0x84, 0xe6, 0xa3, 0x85, 0xe6, 0xa3, 0x86, 0xe6, 0xa3, 0x87, 0xe6, 0xa3,
  0x88, 0xe6, 0xa3, 0x8a, 0xe6, 0xa3, 0x8c, 0xe6, 0xa3, 0x8e, 0xe6, 0xa3,
  0x8f, 0xe6, 0xa3, 0x90, 0xe6, 0xa3, 0x91, 0xe6, 0xa3, 0x93, 0xe6, 0xa3,
  0x94, 0xe6, 0xa3, 0x96, 0xe6, 0xa3, 0x97, 0xe6, 0xa3, 0x99, 0xe6, 0xa3,
  0x9b, 0xe6, 0xa3, 0x9c, 0xe6, 0xa3, 0x9d, 0xe6, 0xa3, 0x9e, 0xe6, 0xa3,
  0x9f, 0xe6, 0xa3, 0xa1, 0xe6, 0xa3, 0xa2, 0xe6, 0xa3, 0xa4, 0xe6, 0xa3,
  0xa5, 0xe6, 0xa3, 0xa6, 0xe6, 0xa3, 0xa7, 0xe6, 0xa3, 0xa8, 0xe6, 0xa3,
  0xa9, 0xe6, 0xa3, 0xaa, 0xe6, 0xa3, 0xab, 0xe6, 0xa3, 0xac, 0xe6, 0xa3,
  0xad, 0xe6, 0xa3, 0xaf, 0xe6, 0xa3, 0xb2, 0xe6, 0xa3, 0xb3, 0xe6, 0xa3,
  0xb4, 0xe6, 0xa3, 0xb6, 0xe6, 0xa3, 0xb7, 0xe6, 0xa3, 0xb8, 0xe6, 0xa3,
  0xbb, 0xe6, 0xa3, 0xbd, 0xe6, 0xa3, 0xbe, 0xe6, 0xa3, 0xbf, 0xe6, 0xa4,
  0x80, 0xe6, 0xa4, 0x82, 0xe6, 0xa4, 0x83, 0xe6, 0xa4, 0x84, 0xe6, 0xa4,
  0x86, 0xe6, 0xa4, 0x87, 0xe6, 0xa4, 0x88, 0xe6, 0xa4, 0x89, 0xe6, 0xa4,
  0x8a, 0xe6, 0xa4, 0x8c, 0xe6, 0xa4, 0x8f, 0xe6, 0xa4, 0x91, 0xe6, 0xa4,
  0x93, 0xe6, 0xa4, 0x94, 0xe6, 0xa4, 0x95, 0xe6, 0xa4, 0x96, 0xe6, 0xa4,
  0x97, 0xe6, 0xa4, 0x98, 0xe6, 0xa4, 0x99, 0xe6, 0xa4, 0x9a, 0xe6, 0xa4,
  0x9b, 0xe6, 0xa4, 0x9c, 0xe6, 0xa4, 0x9d, 0xe6, 0xa4, 0x9e, 0xe6, 0xa4,
  0xa1, 0xe6, 0xa4, 0xa2, 0xe6, 0xa4, 0xa3, 0xe6, 0xa4, 0xa5, 0xe6, 0xa4,
  0xa6, 0xe6, 0xa4, 0xa7, 0xe6, 0xa4, 0xa8, 0xe6, 0xa4, 0xa9, 0xe6, 0xa4,
  0xaa, 0xe6, 0xa4, 0xab, 0xe6, 0xa4, 0xac, 0xe6, 0xa4, 0xae, 0xe6, 0xa4,
  0xaf, 0xe6, 0xa4, 0xb1, 0xe6, 0xa4, 0xb2, 0xe6, 0xa4, 0xb3, 0xe6, 0xa4,
  0xb5, 0xe6, 0xa4, 0xb6, 0xe6, 0xa4, 0xb7, 0xe6, 0xa4, 0xb8, 0xe6, 0xa4,
  0xba, 0xe6, 0xa4, 0xbb, 0xe6, 0xa4, 0xbc, 0xe6, 0xa4, 0xbe, 0xe6, 0xa5,
  0x80, 0xe6, 0xa5, 0x81, 0xe6, 0xa5, 0x83, 0xe6, 0xa5, 0x84, 0xe6, 0xa5,
  0x85, 0xe6, 0xa5, 0x86, 0xe6, 0xa5, 0x87, 0xe6, 0xa5, 0x88, 0xe6, 0xa5,
  0x89, 0xe6, 0xa5, 0x8a, 0xe6, 0xa5, 0x8b, 0xe6, 0xa5, 0x8c, 0xe6, 0xa5,
  0x8d, 0xe6, 0xa5, 0x8e, 0xe6, 0xa5, 0x8f, 0xe6, 0xa5, 0x90, 0xe6, 0xa5,
  0x91, 0xe6, 0xa5, 0x92, 0xe6, 0xa5, 0x93, 0xe6, 0xa5, 0x95, 0xe6, 0xa5,
  0x96, 0xe6, 0xa5, 0x98, 0xe6, 0xa5, 0x99, 0xe6, 0xa5, 0x9b, 0xe6, 0xa5,
  0x9c, 0xe6, 0xa5, 0x9f
};

unsigned char gbk_row_0x98_converted_to_utf8[] = {
  0xe6, 0xa5, 0xa1, 0xe6, 0xa5, 0xa2, 0xe6,
  0xa5, 0xa4, 0xe6, 0xa5, 0xa5, 0xe6, 0xa5, 0xa7, 0xe6, 0xa5, 0xa8, 0xe6,
  0xa5, 0xa9, 0xe6, 0xa5, 0xaa, 0xe6, 0xa5, 0xac, 0xe6, 0xa5, 0xad, 0xe6,
  0xa5, 0xaf, 0xe6, 0xa5, 0xb0, 0xe6, 0xa5, 0xb2, 0xe6, 0xa5, 0xb3, 0xe6,
  0xa5, 0xb4, 0xe6, 0xa5, 0xb5, 0xe6, 0xa5, 0xb6, 0xe6, 0xa5, 0xba, 0xe6,
  0xa5, 0xbb, 0xe6, 0xa5, 0xbd, 0xe6, 0xa5, 0xbe, 0xe6, 0xa5, 0xbf, 0xe6,
  0xa6, 0x81, 0xe6, 0xa6, 0x83, 0xe6, 0xa6, 0x85, 0xe6, 0xa6, 0x8a, 0xe6,
  0xa6, 0x8b, 0xe6, 0xa6, 0x8c, 0xe6, 0xa6, 0x8e, 0xe6, 0xa6, 0x8f, 0xe6,
  0xa6, 0x90, 0xe6, 0xa6, 0x91, 0xe6, 0xa6, 0x92, 0xe6, 0xa6, 0x93, 0xe6,
  0xa6, 0x96, 0xe6, 0xa6, 0x97, 0xe6, 0xa6, 0x99, 0xe6, 0xa6, 0x9a, 0xe6,
  0xa6, 0x9d, 0xe6, 0xa6, 0x9e, 0xe6, 0xa6, 0x9f, 0xe6, 0xa6, 0xa0, 0xe6,
  0xa6, 0xa1, 0xe6, 0xa6, 0xa2, 0xe6, 0xa6, 0xa3, 0xe6, 0xa6, 0xa4, 0xe6,
  0xa6, 0xa5, 0xe6, 0xa6, 0xa6, 0xe6, 0xa6, 0xa9, 0xe6, 0xa6, 0xaa, 0xe6,
  0xa6, 0xac, 0xe6, 0xa6, 0xae, 0xe6, 0xa6, 0xaf, 0xe6, 0xa6, 0xb0, 0xe6,
  0xa6, 0xb2, 0xe6, 0xa6, 0xb3, 0xe6, 0xa6, 0xb5, 0xe6, 0xa6, 0xb6, 0xe6,
  0xa6, 0xb8, 0xe6, 0xa6, 0xb9, 0xe6, 0xa6, 0xba, 0xe6, 0xa6, 0xbc, 0xe6,
  0xa6, 0xbd, 0xe6, 0xa6, 0xbe, 0xe6, 0xa6, 0xbf, 0xe6, 0xa7, 0x80, 0xe6,
  0xa7, 0x82, 0xe6, 0xa7, 0x83, 0xe6, 0xa7, 0x84, 0xe6, 0xa7, 0x85, 0xe6,
  0xa7, 0x86, 0xe6, 0xa7, 0x87, 0xe6, 0xa7, 0x88, 0xe6, 0xa7, 0x89, 0xe6,
  0xa7, 0x8b, 0xe6, 0xa7, 0x8d, 0xe6, 0xa7, 0x8f, 0xe6, 0xa7, 0x91, 0xe6,
  0xa7, 0x92, 0xe6, 0xa7, 0x93, 0xe6, 0xa7, 0x95, 0xe6, 0xa7, 0x96, 0xe6,
  0xa7, 0x97, 0xe6, 0xa7, 0x98, 0xe6, 0xa7, 0x99, 0xe6, 0xa7, 0x9a, 0xe6,
  0xa7, 0x9c, 0xe6, 0xa7, 0x9d, 0xe6, 0xa7, 0x9e, 0xe6, 0xa7, 0xa1, 0xe6,
  0xa7, 0xa2, 0xe6, 0xa7, 0xa3, 0xe6, 0xa7, 0xa4, 0xe6, 0xa7, 0xa5, 0xe6,
  0xa7, 0xa6, 0xe6, 0xa7, 0xa7, 0xe6, 0xa7, 0xa8, 0xe6, 0xa7, 0xa9, 0xe6,
  0xa7, 0xaa, 0xe6, 0xa7, 0xab, 0xe6, 0xa7, 0xac, 0xe6, 0xa7, 0xae, 0xe6,
  0xa7, 0xaf, 0xe6, 0xa7, 0xb0, 0xe6, 0xa7, 0xb1, 0xe6, 0xa7, 0xb3, 0xe6,
  0xa7, 0xb4, 0xe6, 0xa7, 0xb5, 0xe6, 0xa7, 0xb6, 0xe6, 0xa7, 0xb7, 0xe6,
  0xa7, 0xb8, 0xe6, 0xa7, 0xb9, 0xe6, 0xa7, 0xba, 0xe6, 0xa7, 0xbb, 0xe6,
  0xa7, 0xbc, 0xe6, 0xa7, 0xbe, 0xe6, 0xa8, 0x80, 0xe6, 0xa8, 0x81, 0xe6,
  0xa8, 0x82, 0xe6, 0xa8, 0x83, 0xe6, 0xa8, 0x84, 0xe6, 0xa8, 0x85, 0xe6,
  0xa8, 0x86, 0xe6, 0xa8, 0x87, 0xe6, 0xa8, 0x88, 0xe6, 0xa8, 0x89, 0xe6,
  0xa8, 0x8b, 0xe6, 0xa8, 0x8c, 0xe6, 0xa8, 0x8d, 0xe6, 0xa8, 0x8e, 0xe6,
  0xa8, 0x8f, 0xe6, 0xa8, 0x90, 0xe6, 0xa8, 0x91, 0xe6, 0xa8, 0x92, 0xe6,
  0xa8, 0x93, 0xe6, 0xa8, 0x94, 0xe6, 0xa8, 0x95, 0xe6, 0xa8, 0x96, 0xe6,
  0xa8, 0x99, 0xe6, 0xa8, 0x9a, 0xe6, 0xa8, 0x9b, 0xe6, 0xa8, 0x9c, 0xe6,
  0xa8, 0x9d, 0xe6, 0xa8, 0x9e, 0xe6, 0xa8, 0xa0, 0xe6, 0xa8, 0xa2, 0xe6,
  0xa8, 0xa3, 0xe6, 0xa8, 0xa4, 0xe6, 0xa8, 0xa5, 0xe6, 0xa8, 0xa6, 0xe6,
  0xa8, 0xa7, 0xe6, 0xa8, 0xa9, 0xe6, 0xa8, 0xab, 0xe6, 0xa8, 0xac, 0xe6,
  0xa8, 0xad, 0xe6, 0xa8, 0xae, 0xe6, 0xa8, 0xb0, 0xe6, 0xa8, 0xb2, 0xe6,
  0xa8, 0xb3, 0xe6, 0xa8, 0xb4, 0xe6, 0xa8, 0xb6, 0xe6, 0xa8, 0xb7, 0xe6,
  0xa8, 0xb8, 0xe6, 0xa8, 0xb9, 0xe6, 0xa8, 0xba, 0xe6, 0xa8, 0xbb, 0xe6,
  0xa8, 0xbc, 0xe6, 0xa8, 0xbf, 0xe6, 0xa9, 0x80, 0xe6, 0xa9, 0x81, 0xe6,
  0xa9, 0x82, 0xe6, 0xa9, 0x83, 0xe6, 0xa9, 0x85, 0xe6, 0xa9, 0x86, 0xe6,
  0xa9, 0x88, 0xe6, 0xa9, 0x89, 0xe6, 0xa9, 0x8a, 0xe6, 0xa9, 0x8b, 0xe6,
  0xa9, 0x8c, 0xe6, 0xa9, 0x8d, 0xe6, 0xa9, 0x8e, 0xe6, 0xa9, 0x8f, 0xe6,
  0xa9, 0x91, 0xe6, 0xa9, 0x92, 0xe6, 0xa9, 0x93, 0xe6, 0xa9, 0x94, 0xe6,
  0xa9, 0x95, 0xe6, 0xa9, 0x96, 0xe6, 0xa9, 0x97, 0xe6, 0xa9, 0x9a
};

unsigned char gbk_row_0x99_converted_to_utf8[] = {
  0xe6, 0xa9, 0x9c, 0xe6, 0xa9, 0x9d, 0xe6, 0xa9, 0x9e, 0xe6, 0xa9, 0x9f,
  0xe6, 0xa9, 0xa0, 0xe6, 0xa9, 0xa2, 0xe6, 0xa9, 0xa3, 0xe6, 0xa9, 0xa4,
  0xe6, 0xa9, 0xa6, 0xe6, 0xa9, 0xa7, 0xe6, 0xa9, 0xa8, 0xe6, 0xa9, 0xa9,
  0xe6, 0xa9, 0xaa, 0xe6, 0xa9, 0xab, 0xe6, 0xa9, 0xac, 0xe6, 0xa9, 0xad,
  0xe6, 0xa9, 0xae, 0xe6, 0xa9, 0xaf, 0xe6, 0xa9, 0xb0, 0xe6, 0xa9, 0xb2,
  0xe6, 0xa9, 0xb3, 0xe6, 0xa9, 0xb4, 0xe6, 0xa9, 0xb5, 0xe6, 0xa9, 0xb6,
  0xe6, 0xa9, 0xb7, 0xe6, 0xa9, 0xb8, 0xe6, 0xa9, 0xba, 0xe6, 0xa9, 0xbb,
  0xe6, 0xa9, 0xbd, 0xe6, 0xa9, 0xbe, 0xe6, 0xa9, 0xbf, 0xe6, 0xaa, 0x81,
  0xe6, 0xaa, 0x82, 0xe6, 0xaa, 0x83, 0xe6, 0xaa, 0x85, 0xe6, 0xaa, 0x86,
  0xe6, 0xaa, 0x87, 0xe6, 0xaa, 0x88, 0xe6, 0xaa, 0x89, 0xe6, 0xaa, 0x8a,
  0xe6, 0xaa, 0x8b, 0xe6, 0xaa, 0x8c, 0xe6, 0xaa, 0x8d, 0xe6, 0xaa, 0x8f,
  0xe6, 0xaa, 0x92, 0xe6, 0xaa, 0x93, 0xe6, 0xaa, 0x94, 0xe6, 0xaa, 0x95,
  0xe6, 0xaa, 0x96, 0xe6, 0xaa, 0x98, 0xe6, 0xaa, 0x99, 0xe6, 0xaa, 0x9a,
  0xe6, 0xaa, 0x9b, 0xe6, 0xaa, 0x9c, 0xe6, 0xaa, 0x9d, 0xe6, 0xaa, 0x9e,
  0xe6, 0xaa, 0x9f, 0xe6, 0xaa, 0xa1, 0xe6, 0xaa, 0xa2, 0xe6, 0xaa, 0xa3,
  0xe6, 0xaa, 0xa4, 0xe6, 0xaa, 0xa5, 0xe6, 0xaa, 0xa6, 0xe6, 0xaa, 0xa7,
  0xe6, 0xaa, 0xa8, 0xe6, 0xaa, 0xaa, 0xe6, 0xaa, 0xad, 0xe6, 0xaa, 0xae,
  0xe6, 0xaa, 0xaf, 0xe6, 0xaa, 0xb0, 0xe6, 0xaa, 0xb1, 0xe6, 0xaa, 0xb2,
  0xe6, 0xaa, 0xb3, 0xe6, 0xaa, 0xb4, 0xe6, 0xaa, 0xb5, 0xe6, 0xaa, 0xb6,
  0xe6, 0xaa, 0xb7, 0xe6, 0xaa, 0xb8, 0xe6, 0xaa, 0xb9, 0xe6, 0xaa, 0xba,
  0xe6, 0xaa, 0xbb, 0xe6, 0xaa, 0xbc, 0xe6, 0xaa, 0xbd, 0xe6, 0xaa, 0xbe,
  0xe6, 0xaa, 0xbf, 0xe6, 0xab, 0x80, 0xe6, 0xab, 0x81, 0xe6, 0xab, 0x82,
  0xe6, 0xab, 0x83, 0xe6, 0xab, 0x84, 0xe6, 0xab, 0x85, 0xe6, 0xab, 0x86,
  0xe6, 0xab, 0x87, 0xe6, 0xab, 0x88, 0xe6, 0xab, 0x89, 0xe6, 0xab, 0x8a,
  0xe6, 0xab, 0x8b, 0xe6, 0xab, 0x8c, 0xe6, 0xab, 0x8d, 0xe6, 0xab, 0x8e,
  0xe6, 0xab, 0x8f, 0xe6, 0xab, 0x90, 0xe6, 0xab, 0x91, 0xe6, 0xab, 0x92,
  0xe6, 0xab, 0x93, 0xe6, 0xab, 0x94, 0xe6, 0xab, 0x95, 0xe6, 0xab, 0x96,
  0xe6, 0xab, 0x97, 0xe6, 0xab, 0x98, 0xe6, 0xab, 0x99, 0xe6, 0xab, 0x9a,
  0xe6, 0xab, 0x9b, 0xe6, 0xab, 0x9c, 0xe6, 0xab, 0x9d, 0xe6, 0xab, 0x9e,
  0xe6, 0xab, 0x9f, 0xe6, 0xab, 0xa0, 0xe6, 0xab, 0xa1, 0xe6, 0xab, 0xa2,
  0xe6, 0xab, 0xa3, 0xe6, 0xab, 0xa4, 0xe6, 0xab, 0xa5, 0xe6, 0xab, 0xa6,
  0xe6, 0xab, 0xa7, 0xe6, 0xab, 0xa8, 0xe6, 0xab, 0xa9, 0xe6, 0xab, 0xaa,
  0xe6, 0xab, 0xab, 0xe6, 0xab, 0xac, 0xe6, 0xab, 0xad, 0xe6, 0xab, 0xae,
  0xe6, 0xab, 0xaf, 0xe6, 0xab, 0xb0, 0xe6, 0xab, 0xb1, 0xe6, 0xab, 0xb2,
  0xe6, 0xab, 0xb3, 0xe6, 0xab, 0xb4, 0xe6, 0xab, 0xb5, 0xe6, 0xab, 0xb6,
  0xe6, 0xab, 0xb7, 0xe6, 0xab, 0xb8, 0xe6, 0xab, 0xb9, 0xe6, 0xab, 0xba,
  0xe6, 0xab, 0xbb, 0xe6, 0xab, 0xbc, 0xe6, 0xab, 0xbd, 0xe6, 0xab, 0xbe,
  0xe6, 0xab, 0xbf, 0xe6, 0xac, 0x80, 0xe6, 0xac, 0x81, 0xe6, 0xac, 0x82,
  0xe6, 0xac, 0x83, 0xe6, 0xac, 0x84, 0xe6, 0xac, 0x85, 0xe6, 0xac, 0x86,
  0xe6, 0xac, 0x87, 0xe6, 0xac, 0x88, 0xe6, 0xac, 0x89, 0xe6, 0xac, 0x8a,
  0xe6, 0xac, 0x8b, 0xe6, 0xac, 0x8c, 0xe6, 0xac, 0x8d, 0xe6, 0xac, 0x8e,
  0xe6, 0xac, 0x8f, 0xe6, 0xac, 0x90, 0xe6, 0xac, 0x91, 0xe6, 0xac, 0x92,
  0xe6, 0xac, 0x93, 0xe6, 0xac, 0x94, 0xe6, 0xac, 0x95, 0xe6, 0xac, 0x96,
  0xe6, 0xac, 0x97, 0xe6, 0xac, 0x98, 0xe6, 0xac, 0x99, 0xe6, 0xac, 0x9a,
  0xe6, 0xac, 0x9b, 0xe6, 0xac, 0x9c, 0xe6, 0xac, 0x9d, 0xe6, 0xac, 0x9e,
  0xe6, 0xac, 0x9f, 0xe6, 0xac, 0xa5, 0xe6, 0xac, 0xa6, 0xe6, 0xac, 0xa8,
  0xe6, 0xac, 0xa9, 0xe6, 0xac, 0xaa, 0xe6, 0xac, 0xab, 0xe6, 0xac, 0xac,
  0xe6, 0xac, 0xad, 0xe6, 0xac, 0xae
};

unsigned char gbk_row_0x9a_converted_to_utf8[] = {
  0xe6, 0xac, 0xaf, 0xe6, 0xac,
  0xb0, 0xe6, 0xac, 0xb1, 0xe6, 0xac, 0xb3, 0xe6, 0xac, 0xb4, 0xe6, 0xac,
  0xb5, 0xe6, 0xac, 0xb6, 0xe6, 0xac, 0xb8, 0xe6, 0xac, 0xbb, 0xe6, 0xac,
  0xbc, 0xe6, 0xac, 0xbd, 0xe6, 0xac, 0xbf, 0xe6, 0xad, 0x80, 0xe6, 0xad,
  0x81, 0xe6, 0xad, 0x82, 0xe6, 0xad, 0x84, 0xe6, 0xad, 0x85, 0xe6, 0xad,
  0x88, 0xe6, 0xad, 0x8a, 0xe6, 0xad, 0x8b, 0xe6, 0xad, 0x8d, 0xe6, 0xad,
  0x8e, 0xe6, 0xad, 0x8f, 0xe6, 0xad, 0x90, 0xe6, 0xad, 0x91, 0xe6, 0xad,
  0x92, 0xe6, 0xad, 0x93, 0xe6, 0xad, 0x94, 0xe6, 0xad, 0x95, 0xe6, 0xad,
  0x96, 0xe6, 0xad, 0x97, 0xe6, 0xad, 0x98, 0xe6, 0xad, 0x9a, 0xe6, 0xad,
  0x9b, 0xe6, 0xad, 0x9c, 0xe6, 0xad, 0x9d, 0xe6, 0xad, 0x9e, 0xe6, 0xad,
  0x9f, 0xe6, 0xad, 0xa0, 0xe6, 0xad, 0xa1, 0xe6, 0xad, 0xa8, 0xe6, 0xad,
  0xa9, 0xe6, 0xad, 0xab, 0xe6, 0xad, 0xac, 0xe6, 0xad, 0xad, 0xe6, 0xad,
  0xae, 0xe6, 0xad, 0xaf, 0xe6, 0xad, 0xb0, 0xe6, 0xad, 0xb1, 0xe6, 0xad,
  0xb2, 0xe6, 0xad, 0xb3, 0xe6, 0xad, 0xb4, 0xe6, 0xad, 0xb5, 0xe6, 0xad,
  0xb6, 0xe6, 0xad, 0xb7, 0xe6, 0xad, 0xb8, 0xe6, 0xad, 0xba, 0xe6, 0xad,
  0xbd, 0xe6, 0xad, 0xbe, 0xe6, 0xad, 0xbf, 0xe6, 0xae, 0x80, 0xe6, 0xae,
  0x85, 0xe6, 0xae, 0x88, 0xe6, 0xae, 0x8c, 0xe6, 0xae, 0x8e, 0xe6, 0xae,
  0x8f, 0xe6, 0xae, 0x90, 0xe6, 0xae, 0x91, 0xe6, 0xae, 0x94, 0xe6, 0xae,
  0x95, 0xe6, 0xae, 0x97, 0xe6, 0xae, 0x98, 0xe6, 0xae, 0x99, 0xe6, 0xae,
  0x9c, 0xe6, 0xae, 0x9d, 0xe6, 0xae, 0x9e, 0xe6, 0xae, 0x9f, 0xe6, 0xae,
  0xa0, 0xe6, 0xae, 0xa2, 0xe6, 0xae, 0xa3, 0xe6, 0xae, 0xa4, 0xe6, 0xae,
  0xa5, 0xe6, 0xae, 0xa6, 0xe6, 0xae, 0xa7, 0xe6, 0xae, 0xa8, 0xe6, 0xae,
  0xa9, 0xe6, 0xae, 0xab, 0xe6, 0xae, 0xac, 0xe6, 0xae, 0xad, 0xe6, 0xae,
  0xae, 0xe6, 0xae, 0xaf, 0xe6, 0xae, 0xb0, 0xe6, 0xae, 0xb1, 0xe6, 0xae,
  0xb2, 0xe6, 0xae, 0xb6, 0xe6, 0xae, 0xb8, 0xe6, 0xae, 0xb9, 0xe6, 0xae,
  0xba, 0xe6, 0xae, 0xbb, 0xe6, 0xae, 0xbc, 0xe6, 0xae, 0xbd, 0xe6, 0xae,
  0xbe, 0xe6, 0xaf, 0x80, 0xe6, 0xaf, 0x83, 0xe6, 0xaf, 0x84, 0xe6, 0xaf,
  0x86, 0xe6, 0xaf, 0x87, 0xe6, 0xaf, 0x88, 0xe6, 0xaf, 0x89, 0xe6, 0xaf,
  0x8a, 0xe6, 0xaf, 0x8c, 0xe6, 0xaf, 0x8e, 0xe6, 0xaf, 0x90, 0xe6, 0xaf,
  0x91, 0xe6, 0xaf, 0x98, 0xe6, 0xaf, 0x9a, 0xe6, 0xaf, 0x9c, 0xe6, 0xaf,
  0x9d, 0xe6, 0xaf, 0x9e, 0xe6, 0xaf, 0x9f, 0xe6, 0xaf, 0xa0, 0xe6, 0xaf,
  0xa2, 0xe6, 0xaf, 0xa3, 0xe6, 0xaf, 0xa4, 0xe6, 0xaf, 0xa5, 0xe6, 0xaf,
  0xa6, 0xe6, 0xaf, 0xa7, 0xe6, 0xaf, 0xa8, 0xe6, 0xaf, 0xa9, 0xe6, 0xaf,
  0xac, 0xe6, 0xaf, 0xad, 0xe6, 0xaf, 0xae, 0xe6, 0xaf, 0xb0, 0xe6, 0xaf,
  0xb1, 0xe6, 0xaf, 0xb2, 0xe6, 0xaf, 0xb4, 0xe6, 0xaf, 0xb6, 0xe6, 0xaf,
  0xb7, 0xe6, 0xaf, 0xb8, 0xe6, 0xaf, 0xba, 0xe6, 0xaf, 0xbb, 0xe6, 0xaf,
  0xbc, 0xe6, 0xaf, 0xbe, 0xe6, 0xaf, 0xbf, 0xe6, 0xb0, 0x80, 0xe6, 0xb0,
  0x81, 0xe6, 0xb0, 0x82, 0xe6, 0xb0, 0x83, 0xe6, 0xb0, 0x84, 0xe6, 0xb0,
  0x88, 0xe6, 0xb0, 0x89, 0xe6, 0xb0, 0x8a, 0xe6, 0xb0, 0x8b, 0xe6, 0xb0,
  0x8c, 0xe6, 0xb0, 0x8e, 0xe6, 0xb0, 0x92, 0xe6, 0xb0, 0x97, 0xe6, 0xb0,
  0x9c, 0xe6, 0xb0, 0x9d, 0xe6, 0xb0, 0x9e, 0xe6, 0xb0, 0xa0, 0xe6, 0xb0,
  0xa3, 0xe6, 0xb0, 0xa5, 0xe6, 0xb0, 0xab, 0xe6, 0xb0, 0xac, 0xe6, 0xb0,
  0xad, 0xe6, 0xb0, 0xb1, 0xe6, 0xb0, 0xb3, 0xe6, 0xb0, 0xb6, 0xe6, 0xb0,
  0xb7, 0xe6, 0xb0, 0xb9, 0xe6, 0xb0, 0xba, 0xe6, 0xb0, 0xbb, 0xe6, 0xb0,
  0xbc, 0xe6, 0xb0, 0xbe, 0xe6, 0xb0, 0xbf, 0xe6, 0xb1, 0x83, 0xe6, 0xb1,
  0x84, 0xe6, 0xb1, 0x85, 0xe6, 0xb1, 0x88, 0xe6, 0xb1, 0x8b, 0xe6, 0xb1,
  0x8c, 0xe6, 0xb1, 0x8d, 0xe6, 0xb1, 0x8e, 0xe6, 0xb1, 0x8f, 0xe6, 0xb1,
  0x91, 0xe6, 0xb1, 0x92, 0xe6, 0xb1, 0x93, 0xe6, 0xb1, 0x96, 0xe6, 0xb1,
  0x98
};

unsigned char gbk_row_0x9b_converted_to_utf8[] = {
  0xe6, 0xb1, 0x99, 0xe6, 0xb1, 0x9a, 0xe6, 0xb1, 0xa2, 0xe6,
  0xb1, 0xa3, 0xe6, 0xb1, 0xa5, 0xe6, 0xb1, 0xa6, 0xe6, 0xb1, 0xa7, 0xe6,
  0xb1, 0xab, 0xe6, 0xb1, 0xac, 0xe6, 0xb1, 0xad, 0xe6, 0xb1, 0xae, 0xe6,
  0xb1, 0xaf, 0xe6, 0xb1, 0xb1, 0xe6, 0xb1, 0xb3, 0xe6, 0xb1, 0xb5, 0xe6,
  0xb1, 0xb7, 0xe6, 0xb1, 0xb8, 0xe6, 0xb1, 0xba, 0xe6, 0xb1, 0xbb, 0xe6,
  0xb1, 0xbc, 0xe6, 0xb1, 0xbf, 0xe6, 0xb2, 0x80, 0xe6, 0xb2, 0x84, 0xe6,
  0xb2, 0x87, 0xe6, 0xb2, 0x8a, 0xe6, 0xb2, 0x8b, 0xe6, 0xb2, 0x8d, 0xe6,
  0xb2, 0x8e, 0xe6, 0xb2, 0x91, 0xe6, 0xb2, 0x92, 0xe6, 0xb2, 0x95, 0xe6,
  0xb2, 0x96, 0xe6, 0xb2, 0x97, 0xe6, 0xb2, 0x98, 0xe6, 0xb2, 0x9a, 0xe6,
  0xb2, 0x9c, 0xe6, 0xb2, 0x9d, 0xe6, 0xb2, 0x9e, 0xe6, 0xb2, 0xa0, 0xe6,
  0xb2, 0xa2, 0xe6, 0xb2, 0xa8, 0xe6, 0xb2, 0xac, 0xe6, 0xb2, 0xaf, 0xe6,
  0xb2, 0xb0, 0xe6, 0xb2, 0xb4, 0xe6, 0xb2, 0xb5, 0xe6, 0xb2, 0xb6, 0xe6,
  0xb2, 0xb7, 0xe6, 0xb2, 0xba, 0xe6, 0xb3, 0x80, 0xe6, 0xb3, 0x81, 0xe6,
  0xb3, 0x82, 0xe6, 0xb3, 0x83, 0xe6, 0xb3, 0x86, 0xe6, 0xb3, 0x87, 0xe6,
  0xb3, 0x88, 0xe6, 0xb3, 0x8b, 0xe6, 0xb3, 0x8d, 0xe6, 0xb3, 0x8e, 0xe6,
  0xb3, 0x8f, 0xe6, 0xb3, 0x91, 0xe6, 0xb3, 0x92, 0xe6, 0xb3, 0x98, 0xe6,
  0xb3, 0x99, 0xe6, 0xb3, 0x9a, 0xe6, 0xb3, 0x9c, 0xe6, 0xb3, 0x9d, 0xe6,
  0xb3, 0x9f, 0xe6, 0xb3, 0xa4, 0xe6, 0xb3, 0xa6, 0xe6, 0xb3, 0xa7, 0xe6,
  0xb3, 0xa9, 0xe6, 0xb3, 0xac, 0xe6, 0xb3, 0xad, 0xe6, 0xb3, 0xb2, 0xe6,
  0xb3, 0xb4, 0xe6, 0xb3, 0xb9, 0xe6, 0xb3, 0xbf, 0xe6, 0xb4, 0x80, 0xe6,
  0xb4, 0x82, 0xe6, 0xb4, 0x83, 0xe6, 0xb4, 0x85, 0xe6, 0xb4, 0x86, 0xe6,
  0xb4, 0x88, 0xe6, 0xb4, 0x89, 0xe6, 0xb4, 0x8a, 0xe6, 0xb4, 0x8d, 0xe6,
  0xb4, 0x8f, 0xe6, 0xb4, 0x90, 0xe6, 0xb4, 0x91, 0xe6, 0xb4, 0x93, 0xe6,
  0xb4, 0x94, 0xe6, 0xb4, 0x95, 0xe6, 0xb4, 0x96, 0xe6, 0xb4, 0x98, 0xe6,
  0xb4, 0x9c, 0xe6, 0xb4, 0x9d, 0xe6, 0xb4, 0x9f, 0xe6, 0xb4, 0xa0, 0xe6,
  0xb4, 0xa1, 0xe6, 0xb4, 0xa2, 0xe6, 0xb4, 0xa3, 0xe6, 0xb4, 0xa4, 0xe6,
  0xb4, 0xa6, 0xe6, 0xb4, 0xa8, 0xe6, 0xb4, 0xa9, 0xe6, 0xb4, 0xac, 0xe6,
  0xb4, 0xad, 0xe6, 0xb4, 0xaf, 0xe6, 0xb4, 0xb0, 0xe6, 0xb4, 0xb4, 0xe6,
  0xb4, 0xb6, 0xe6, 0xb4, 0xb7, 0xe6, 0xb4, 0xb8, 0xe6, 0xb4, 0xba, 0xe6,
  0xb4, 0xbf, 0xe6, 0xb5, 0x80, 0xe6, 0xb5, 0x82, 0xe6, 0xb5, 0x84, 0xe6,
  0xb5, 0x89, 0xe6, 0xb5, 0x8c, 0xe6, 0xb5, 0x90, 0xe6, 0xb5, 0x95, 0xe6,
  0xb5, 0x96, 0xe6, 0xb5, 0x97, 0xe6, 0xb5, 0x98, 0xe6, 0xb5, 0x9b, 0xe6,
  0xb5, 0x9d, 0xe6, 0xb5, 0x9f, 0xe6, 0xb5, 0xa1, 0xe6, 0xb5, 0xa2, 0xe6,
  0xb5, 0xa4, 0xe6, 0xb5, 0xa5, 0xe6, 0xb5, 0xa7, 0xe6, 0xb5, 0xa8, 0xe6,
  0xb5, 0xab, 0xe6, 0xb5, 0xac, 0xe6, 0xb5, 0xad, 0xe6, 0xb5, 0xb0, 0xe6,
  0xb5, 0xb1, 0xe6, 0xb5, 0xb2, 0xe6, 0xb5, 0xb3, 0xe6, 0xb5, 0xb5, 0xe6,
  0xb5, 0xb6, 0xe6, 0xb5, 0xb9, 0xe6, 0xb5, 0xba, 0xe6, 0xb5, 0xbb, 0xe6,
  0xb5, 0xbd, 0xe6, 0xb5, 0xbe, 0xe6, 0xb5, 0xbf, 0xe6, 0xb6, 0x80, 0xe6,
  0xb6, 0x81, 0xe6, 0xb6, 0x83, 0xe6, 0xb6, 0x84, 0xe6, 0xb6, 0x86, 0xe6,
  0xb6, 0x87, 0xe6, 0xb6, 0x8a, 0xe6, 0xb6, 0x8b, 0xe6, 0xb6, 0x8d, 0xe6,
  0xb6, 0x8f, 0xe6, 0xb6, 0x90, 0xe6, 0xb6, 0x92, 0xe6, 0xb6, 0x96, 0xe6,
  0xb6, 0x97, 0xe6, 0xb6, 0x98, 0xe6, 0xb6, 0x99, 0xe6, 0xb6, 0x9a, 0xe6,
  0xb6, 0x9c, 0xe6, 0xb6, 0xa2, 0xe6, 0xb6, 0xa5, 0xe6, 0xb6, 0xac, 0xe6,
  0xb6, 0xad, 0xe6, 0xb6, 0xb0, 0xe6, 0xb6, 0xb1, 0xe6, 0xb6, 0xb3, 0xe6,
  0xb6, 0xb4, 0xe6, 0xb6, 0xb6, 0xe6, 0xb6, 0xb7, 0xe6, 0xb6, 0xb9, 0xe6,
  0xb6, 0xba, 0xe6, 0xb6, 0xbb, 0xe6, 0xb6, 0xbc, 0xe6, 0xb6, 0xbd, 0xe6,
  0xb6, 0xbe, 0xe6, 0xb7, 0x81, 0xe6, 0xb7, 0x82, 0xe6, 0xb7, 0x83, 0xe6,
  0xb7, 0x88, 0xe6, 0xb7, 0x89, 0xe6, 0xb7, 0x8a
};

unsigned char gbk_row_0x9c_converted_to_utf8[] = {
  0xe6, 0xb7, 0x8d,
  0xe6, 0xb7, 0x8e, 0xe6, 0xb7, 0x8f, 0xe6, 0xb7, 0x90, 0xe6, 0xb7, 0x92,
  0xe6, 0xb7, 0x93, 0xe6, 0xb7, 0x94, 0xe6, 0xb7, 0x95, 0xe6, 0xb7, 0x97,
  0xe6, 0xb7, 0x9a, 0xe6, 0xb7, 0x9b, 0xe6, 0xb7, 0x9c, 0xe6, 0xb7, 0x9f,
  0xe6, 0xb7, 0xa2, 0xe6, 0xb7, 0xa3, 0xe6, 0xb7, 0xa5, 0xe6, 0xb7, 0xa7,
  0xe6, 0xb7, 0xa8, 0xe6, 0xb7, 0xa9, 0xe6, 0xb7, 0xaa, 0xe6, 0xb7, 0xad,
  0xe6, 0xb7, 0xaf, 0xe6, 0xb7, 0xb0, 0xe6, 0xb7, 0xb2, 0xe6, 0xb7, 0xb4,
  0xe6, 0xb7, 0xb5, 0xe6, 0xb7, 0xb6, 0xe6, 0xb7, 0xb8, 0xe6, 0xb7, 0xba,
  0xe6, 0xb7, 0xbd, 0xe6, 0xb7, 0xbe, 0xe6, 0xb7, 0xbf, 0xe6, 0xb8, 0x80,
  0xe6, 0xb8, 0x81, 0xe6, 0xb8, 0x82, 0xe6, 0xb8, 0x83, 0xe6, 0xb8, 0x84,
  0xe6, 0xb8, 0x86, 0xe6, 0xb8, 0x87, 0xe6, 0xb8, 0x88, 0xe6, 0xb8, 0x89,
  0xe6, 0xb8, 0x8b, 0xe6, 0xb8, 0x8f, 0xe6, 0xb8, 0x92, 0xe6, 0xb8, 0x93,
  0xe6, 0xb8, 0x95, 0xe6, 0xb8, 0x98, 0xe6, 0xb8, 0x99, 0xe6, 0xb8, 0x9b,
  0xe6, 0xb8, 0x9c, 0xe6, 0xb8, 0x9e, 0xe6, 0xb8, 0x9f, 0xe6, 0xb8, 0xa2,
  0xe6, 0xb8, 0xa6, 0xe6, 0xb8, 0xa7, 0xe6, 0xb8, 0xa8, 0xe6, 0xb8, 0xaa,
  0xe6, 0xb8, 0xac, 0xe6, 0xb8, 0xae, 0xe6, 0xb8, 0xb0, 0xe6, 0xb8, 0xb1,
  0xe6, 0xb8, 0xb3, 0xe6, 0xb8, 0xb5, 0xe6, 0xb8, 0xb6, 0xe6, 0xb8, 0xb7,
  0xe6, 0xb8, 0xb9, 0xe6, 0xb8, 0xbb, 0xe6, 0xb8, 0xbc, 0xe6, 0xb8, 0xbd,
  0xe6, 0xb8, 0xbe, 0xe6, 0xb8, 0xbf, 0xe6, 0xb9, 0x80, 0xe6, 0xb9, 0x81,
  0xe6, 0xb9, 0x82, 0xe6, 0xb9, 0x85, 0xe6, 0xb9, 0x86, 0xe6, 0xb9, 0x87,
  0xe6, 0xb9, 0x88, 0xe6, 0xb9, 0x89, 0xe6, 0xb9, 0x8a, 0xe6, 0xb9, 0x8b,
  0xe6, 0xb9, 0x8c, 0xe6, 0xb9, 0x8f, 0xe6, 0xb9, 0x90, 0xe6, 0xb9, 0x91,
  0xe6, 0xb9, 0x92, 0xe6, 0xb9, 0x95, 0xe6, 0xb9, 0x97, 0xe6, 0xb9, 0x99,
  0xe6, 0xb9, 0x9a, 0xe6, 0xb9, 0x9c, 0xe6, 0xb9, 0x9d, 0xe6, 0xb9, 0x9e,
  0xe6, 0xb9, 0xa0, 0xe6, 0xb9, 0xa1, 0xe6, 0xb9, 0xa2, 0xe6, 0xb9, 0xa3,
  0xe6, 0xb9, 0xa4, 0xe6, 0xb9, 0xa5, 0xe6, 0xb9, 0xa6, 0xe6, 0xb9, 0xa7,
  0xe6, 0xb9, 0xa8, 0xe6, 0xb9, 0xa9, 0xe6, 0xb9, 0xaa, 0xe6, 0xb9, 0xac,
  0xe6, 0xb9, 0xad, 0xe6, 0xb9, 0xaf, 0xe6, 0xb9, 0xb0, 0xe6, 0xb9, 0xb1,
  0xe6, 0xb9, 0xb2, 0xe6, 0xb9, 0xb3, 0xe6, 0xb9, 0xb4, 0xe6, 0xb9, 0xb5,
  0xe6, 0xb9, 0xb6, 0xe6, 0xb9, 0xb7, 0xe6, 0xb9, 0xb8, 0xe6, 0xb9, 0xb9,
  0xe6, 0xb9, 0xba, 0xe6, 0xb9, 0xbb, 0xe6, 0xb9, 0xbc, 0xe6, 0xb9, 0xbd,
  0xe6, 0xba, 0x80, 0xe6, 0xba, 0x81, 0xe6, 0xba, 0x82, 0xe6, 0xba, 0x84,
  0xe6, 0xba, 0x87, 0xe6, 0xba, 0x88, 0xe6, 0xba, 0x8a, 0xe6, 0xba, 0x8b,
  0xe6, 0xba, 0x8c, 0xe6, 0xba, 0x8d, 0xe6, 0xba, 0x8e, 0xe6, 0xba, 0x91,
  0xe6, 0xba, 0x92, 0xe6, 0xba, 0x93, 0xe6, 0xba, 0x94, 0xe6, 0xba, 0x95,
  0xe6, 0xba, 0x96, 0xe6, 0xba, 0x97, 0xe6, 0xba, 0x99, 0xe6, 0xba, 0x9a,
  0xe6, 0xba, 0x9b, 0xe6, 0xba, 0x9d, 0xe6, 0xba, 0x9e, 0xe6, 0xba, 0xa0,
  0xe6, 0xba, 0xa1, 0xe6, 0xba, 0xa3, 0xe6, 0xba, 0xa4, 0xe6, 0xba, 0xa6,
  0xe6, 0xba, 0xa8, 0xe6, 0xba, 0xa9, 0xe6, 0xba, 0xab, 0xe6, 0xba, 0xac,
  0xe6, 0xba, 0xad, 0xe6, 0xba, 0xae, 0xe6, 0xba, 0xb0, 0xe6, 0xba, 0xb3,
  0xe6, 0xba, 0xb5, 0xe6, 0xba, 0xb8, 0xe6, 0xba, 0xb9, 0xe6, 0xba, 0xbc,
  0xe6, 0xba, 0xbe, 0xe6, 0xba, 0xbf, 0xe6, 0xbb, 0x80, 0xe6, 0xbb, 0x83,
  0xe6, 0xbb, 0x84, 0xe6, 0xbb, 0x85, 0xe6, 0xbb, 0x86, 0xe6, 0xbb, 0x88,
  0xe6, 0xbb, 0x89, 0xe6, 0xbb, 0x8a, 0xe6, 0xbb, 0x8c, 0xe6, 0xbb, 0x8d,
  0xe6, 0xbb, 0x8e, 0xe6, 0xbb, 0x90, 0xe6, 0xbb, 0x92, 0xe6, 0xbb, 0x96,
  0xe6, 0xbb, 0x98, 0xe6, 0xbb, 0x99, 0xe6, 0xbb, 0x9b, 0xe6, 0xbb, 0x9c,
  0xe6, 0xbb, 0x9d, 0xe6, 0xbb, 0xa3, 0xe6, 0xbb, 0xa7, 0xe6, 0xbb, 0xaa,
  0xe6, 0xbb, 0xab, 0xe6, 0xbb, 0xac, 0xe6, 0xbb, 0xad, 0xe6, 0xbb, 0xae,
  0xe6, 0xbb, 0xaf
};

unsigned char gbk_row_0x9d_converted_to_utf8[] = {
  0xe6, 0xbb, 0xb0, 0xe6, 0xbb, 0xb1, 0xe6, 0xbb,
  0xb2, 0xe6, 0xbb, 0xb3, 0xe6, 0xbb, 0xb5, 0xe6, 0xbb, 0xb6, 0xe6, 0xbb,
  0xb7, 0xe6, 0xbb, 0xb8, 0xe6, 0xbb, 0xba, 0xe6, 0xbb, 0xbb, 0xe6, 0xbb,
  0xbc, 0xe6, 0xbb, 0xbd, 0xe6, 0xbb, 0xbe, 0xe6, 0xbb, 0xbf, 0xe6, 0xbc,
  0x80, 0xe6, 0xbc, 0x81, 0xe6, 0xbc, 0x83, 0xe6, 0xbc, 0x84, 0xe6, 0xbc,
  0x85, 0xe6, 0xbc, 0x87, 0xe6, 0xbc, 0x88, 0xe6, 0xbc, 0x8a, 0xe6, 0xbc,
  0x8b, 0xe6, 0xbc, 0x8c, 0xe6, 0xbc, 0x8d, 0xe6, 0xbc, 0x8e, 0xe6, 0xbc,
  0x90, 0xe6, 0xbc, 0x91, 0xe6, 0xbc, 0x92, 0xe6, 0xbc, 0x96, 0xe6, 0xbc,
  0x97, 0xe6, 0xbc, 0x98, 0xe6, 0xbc, 0x99, 0xe6, 0xbc, 0x9a, 0xe6, 0xbc,
  0x9b, 0xe6, 0xbc, 0x9c, 0xe6, 0xbc, 0x9d, 0xe6, 0xbc, 0x9e, 0xe6, 0xbc,
  0x9f, 0xe6, 0xbc, 0xa1, 0xe6, 0xbc, 0xa2, 0xe6, 0xbc, 0xa3, 0xe6, 0xbc,
  0xa5, 0xe6, 0xbc, 0xa6, 0xe6, 0xbc, 0xa7, 0xe6, 0xbc, 0xa8, 0xe6, 0xbc,
  0xac, 0xe6, 0xbc, 0xae, 0xe6, 0xbc, 0xb0, 0xe6, 0xbc, 0xb2, 0xe6, 0xbc,
  0xb4, 0xe6, 0xbc, 0xb5, 0xe6, 0xbc, 0xb7, 0xe6, 0xbc, 0xb8, 0xe6, 0xbc,
  0xb9, 0xe6, 0xbc, 0xba, 0xe6, 0xbc, 0xbb, 0xe6, 0xbc, 0xbc, 0xe6, 0xbc,
  0xbd, 0xe6, 0xbc, 0xbf, 0xe6, 0xbd, 0x80, 0xe6, 0xbd, 0x81, 0xe6, 0xbd,
  0x82, 0xe6, 0xbd, 0x83, 0xe6, 0xbd, 0x84, 0xe6, 0xbd, 0x85, 0xe6, 0xbd,
  0x88, 0xe6, 0xbd, 0x89, 0xe6, 0xbd, 0x8a, 0xe6, 0xbd, 0x8c, 0xe6, 0xbd,
  0x8e, 0xe6, 0xbd, 0x8f, 0xe6, 0xbd, 0x90, 0xe6, 0xbd, 0x91, 0xe6, 0xbd,
  0x92, 0xe6, 0xbd, 0x93, 0xe6, 0xbd, 0x94, 0xe6, 0xbd, 0x95, 0xe6, 0xbd,
  0x96, 0xe6, 0xbd, 0x97, 0xe6, 0xbd, 0x99, 0xe6, 0xbd, 0x9a, 0xe6, 0xbd,
  0x9b, 0xe6, 0xbd, 0x9d, 0xe6, 0xbd, 0x9f, 0xe6, 0xbd, 0xa0, 0xe6, 0xbd,
  0xa1, 0xe6, 0xbd, 0xa3, 0xe6, 0xbd, 0xa4, 0xe6, 0xbd, 0xa5, 0xe6, 0xbd,
  0xa7, 0xe6, 0xbd, 0xa8, 0xe6, 0xbd, 0xa9, 0xe6, 0xbd, 0xaa, 0xe6, 0xbd,
  0xab, 0xe6, 0xbd, 0xac, 0xe6, 0xbd, 0xaf, 0xe6, 0xbd, 0xb0, 0xe6, 0xbd,
  0xb1, 0xe6, 0xbd, 0xb3, 0xe6, 0xbd, 0xb5, 0xe6, 0xbd, 0xb6, 0xe6, 0xbd,
  0xb7, 0xe6, 0xbd, 0xb9, 0xe6, 0xbd, 0xbb, 0xe6, 0xbd, 0xbd, 0xe6, 0xbd,
  0xbe, 0xe6, 0xbd, 0xbf, 0xe6, 0xbe, 0x80, 0xe6, 0xbe, 0x81, 0xe6, 0xbe,
  0x82, 0xe6, 0xbe, 0x83, 0xe6, 0xbe, 0x85, 0xe6, 0xbe, 0x86, 0xe6, 0xbe,
  0x87, 0xe6, 0xbe, 0x8a, 0xe6, 0xbe, 0x8b, 0xe6, 0xbe, 0x8f, 0xe6, 0xbe,
  0x90, 0xe6, 0xbe, 0x91, 0xe6, 0xbe, 0x92, 0xe6, 0xbe, 0x93, 0xe6, 0xbe,
  0x94, 0xe6, 0xbe, 0x95, 0xe6, 0xbe, 0x96, 0xe6, 0xbe, 0x97, 0xe6, 0xbe,
  0x98, 0xe6, 0xbe, 0x99, 0xe6, 0xbe, 0x9a, 0xe6, 0xbe, 0x9b, 0xe6, 0xbe,
  0x9d, 0xe6, 0xbe, 0x9e, 0xe6, 0xbe, 0x9f, 0xe6, 0xbe, 0xa0, 0xe6, 0xbe,
  0xa2, 0xe6, 0xbe, 0xa3, 0xe6, 0xbe, 0xa4, 0xe6, 0xbe, 0xa5, 0xe6, 0xbe,
  0xa6, 0xe6, 0xbe, 0xa8, 0xe6, 0xbe, 0xa9, 0xe6, 0xbe, 0xaa, 0xe6, 0xbe,
  0xab, 0xe6, 0xbe, 0xac, 0xe6, 0xbe, 0xad, 0xe6, 0xbe, 0xae, 0xe6, 0xbe,
  0xaf, 0xe6, 0xbe, 0xb0, 0xe6, 0xbe, 0xb1, 0xe6, 0xbe, 0xb2, 0xe6, 0xbe,
  0xb4, 0xe6, 0xbe, 0xb5, 0xe6, 0xbe, 0xb7, 0xe6, 0xbe, 0xb8, 0xe6, 0xbe,
  0xba, 0xe6, 0xbe, 0xbb, 0xe6, 0xbe, 0xbc, 0xe6, 0xbe, 0xbd, 0xe6, 0xbe,
  0xbe, 0xe6, 0xbe, 0xbf, 0xe6, 0xbf, 0x81, 0xe6, 0xbf, 0x83, 0xe6, 0xbf,
  0x84, 0xe6, 0xbf, 0x85, 0xe6, 0xbf, 0x86, 0xe6, 0xbf, 0x87, 0xe6, 0xbf,
  0x88, 0xe6, 0xbf, 0x8a, 0xe6, 0xbf, 0x8b, 0xe6, 0xbf, 0x8c, 0xe6, 0xbf,
  0x8d, 0xe6, 0xbf, 0x8e, 0xe6, 0xbf, 0x8f, 0xe6, 0xbf, 0x90, 0xe6, 0xbf,
  0x93, 0xe6, 0xbf, 0x94, 0xe6, 0xbf, 0x95, 0xe6, 0xbf, 0x96, 0xe6, 0xbf,
  0x97, 0xe6, 0xbf, 0x98, 0xe6, 0xbf, 0x99, 0xe6, 0xbf, 0x9a, 0xe6, 0xbf,
  0x9b, 0xe6, 0xbf, 0x9c, 0xe6, 0xbf, 0x9d, 0xe6, 0xbf, 0x9f, 0xe6, 0xbf,
  0xa2, 0xe6, 0xbf, 0xa3, 0xe6, 0xbf, 0xa4, 0xe6, 0xbf, 0xa5
};

unsigned char gbk_row_0x9e_converted_to_utf8[] = {
  0xe6,
  0xbf, 0xa6, 0xe6, 0xbf, 0xa7, 0xe6, 0xbf, 0xa8, 0xe6, 0xbf, 0xa9, 0xe6,
  0xbf, 0xaa, 0xe6, 0xbf, 0xab, 0xe6, 0xbf, 0xac, 0xe6, 0xbf, 0xad, 0xe6,
  0xbf, 0xb0, 0xe6, 0xbf, 0xb1, 0xe6, 0xbf, 0xb2, 0xe6, 0xbf, 0xb3, 0xe6,
  0xbf, 0xb4, 0xe6, 0xbf, 0xb5, 0xe6, 0xbf, 0xb6, 0xe6, 0xbf, 0xb7, 0xe6,
  0xbf, 0xb8, 0xe6, 0xbf, 0xb9, 0xe6, 0xbf, 0xba, 0xe6, 0xbf, 0xbb, 0xe6,
  0xbf, 0xbc, 0xe6, 0xbf, 0xbd, 0xe6, 0xbf, 0xbe, 0xe6, 0xbf, 0xbf, 0xe7,
  0x80, 0x80, 0xe7, 0x80, 0x81, 0xe7, 0x80, 0x82, 0xe7, 0x80, 0x83, 0xe7,
  0x80, 0x84, 0xe7, 0x80, 0x85, 0xe7, 0x80, 0x86, 0xe7, 0x80, 0x87, 0xe7,
  0x80, 0x88, 0xe7, 0x80, 0x89, 0xe7, 0x80, 0x8a, 0xe7, 0x80, 0x8b, 0xe7,
  0x80, 0x8c, 0xe7, 0x80, 0x8d, 0xe7, 0x80, 0x8e, 0xe7, 0x80, 0x8f, 0xe7,
  0x80, 0x90, 0xe7, 0x80, 0x92, 0xe7, 0x80, 0x93, 0xe7, 0x80, 0x94, 0xe7,
  0x80, 0x95, 0xe7, 0x80, 0x96, 0xe7, 0x80, 0x97, 0xe7, 0x80, 0x98, 0xe7,
  0x80, 0x99, 0xe7, 0x80, 0x9c, 0xe7, 0x80, 0x9d, 0xe7, 0x80, 0x9e, 0xe7,
  0x80, 0x9f, 0xe7, 0x80, 0xa0, 0xe7, 0x80, 0xa1, 0xe7, 0x80, 0xa2, 0xe7,
  0x80, 0xa4, 0xe7, 0x80, 0xa5, 0xe7, 0x80, 0xa6, 0xe7, 0x80, 0xa7, 0xe7,
  0x80, 0xa8, 0xe7, 0x80, 0xa9, 0xe7, 0x80, 0xaa, 0xe7, 0x80, 0xab, 0xe7,
  0x80, 0xac, 0xe7, 0x80, 0xad, 0xe7, 0x80, 0xae, 0xe7, 0x80, 0xaf, 0xe7,
  0x80, 0xb0, 0xe7, 0x80, 0xb1, 0xe7, 0x80, 0xb2, 0xe7, 0x80, 0xb3, 0xe7,
  0x80, 0xb4, 0xe7, 0x80, 0xb6, 0xe7, 0x80, 0xb7, 0xe7, 0x80, 0xb8, 0xe7,
  0x80, 0xba, 0xe7, 0x80, 0xbb, 0xe7, 0x80, 0xbc, 0xe7, 0x80, 0xbd, 0xe7,
  0x80, 0xbe, 0xe7, 0x80, 0xbf, 0xe7, 0x81, 0x80, 0xe7, 0x81, 0x81, 0xe7,
  0x81, 0x82, 0xe7, 0x81, 0x83, 0xe7, 0x81, 0x84, 0xe7, 0x81, 0x85, 0xe7,
  0x81, 0x86, 0xe7, 0x81, 0x87, 0xe7, 0x81, 0x88, 0xe7, 0x81, 0x89, 0xe7,
  0x81, 0x8a, 0xe7, 0x81, 0x8b, 0xe7, 0x81, 0x8d, 0xe7, 0x81, 0x8e, 0xe7,
  0x81, 0x90, 0xe7, 0x81, 0x91, 0xe7, 0x81, 0x92, 0xe7, 0x81, 0x93, 0xe7,
  0x81, 0x94, 0xe7, 0x81, 0x95, 0xe7, 0x81, 0x96, 0xe7, 0x81, 0x97, 0xe7,
  0x81, 0x98, 0xe7, 0x81, 0x99, 0xe7, 0x81, 0x9a, 0xe7, 0x81, 0x9b, 0xe7,
  0x81, 0x9c, 0xe7, 0x81, 0x9d, 0xe7, 0x81, 0x9f, 0xe7, 0x81, 0xa0, 0xe7,
  0x81, 0xa1, 0xe7, 0x81, 0xa2, 0xe7, 0x81, 0xa3, 0xe7, 0x81, 0xa4, 0xe7,
  0x81, 0xa5, 0xe7, 0x81, 0xa6, 0xe7, 0x81, 0xa7, 0xe7, 0x81, 0xa8, 0xe7,
  0x81, 0xa9, 0xe7, 0x81, 0xaa, 0xe7, 0x81, 0xae, 0xe7, 0x81, 0xb1, 0xe7,
  0x81, 0xb2, 0xe7, 0x81, 0xb3, 0xe7, 0x81, 0xb4, 0xe7, 0x81, 0xb7, 0xe7,
  0x81, 0xb9, 0xe7, 0x81, 0xba, 0xe7, 0x81, 0xbb, 0xe7, 0x81, 0xbd, 0xe7,
  0x82, 0x81, 0xe7, 0x82, 0x82, 0xe7, 0x82, 0x83, 0xe7, 0x82, 0x84, 0xe7,
  0x82, 0x86, 0xe7, 0x82, 0x87, 0xe7, 0x82, 0x88, 0xe7, 0x82, 0x8b, 0xe7,
  0x82, 0x8c, 0xe7, 0x82, 0x8d, 0xe7, 0x82, 0x8f, 0xe7, 0x82, 0x90, 0xe7,
  0x82, 0x91, 0xe7, 0x82, 0x93, 0xe7, 0x82, 0x97, 0xe7, 0x82, 0x98, 0xe7,
  0x82, 0x9a, 0xe7, 0x82, 0x9b, 0xe7, 0x82, 0x9e, 0xe7, 0x82, 0x9f, 0xe7,
  0x82, 0xa0, 0xe7, 0x82, 0xa1, 0xe7, 0x82, 0xa2, 0xe7, 0x82, 0xa3, 0xe7,
  0x82, 0xa4, 0xe7, 0x82, 0xa5, 0xe7, 0x82, 0xa6, 0xe7, 0x82, 0xa7, 0xe7,
  0x82, 0xa8, 0xe7, 0x82, 0xa9, 0xe7, 0x82, 0xaa, 0xe7, 0x82, 0xb0, 0xe7,
  0x82, 0xb2, 0xe7, 0x82, 0xb4, 0xe7, 0x82, 0xb5, 0xe7, 0x82, 0xb6, 0xe7,
  0x82, 0xba, 0xe7, 0x82, 0xbe, 0xe7, 0x82, 0xbf, 0xe7, 0x83, 0x84, 0xe7,
  0x83, 0x85, 0xe7, 0x83, 0x86, 0xe7, 0x83, 0x87, 0xe7, 0x83, 0x89, 0xe7,
  0x83, 0x8b, 0xe7, 0x83, 0x8c, 0xe7, 0x83, 0x8d, 0xe7, 0x83, 0x8e, 0xe7,
  0x83, 0x8f, 0xe7, 0x83, 0x90, 0xe7, 0x83, 0x91, 0xe7, 0x83, 0x92, 0xe7,
  0x83, 0x93, 0xe7, 0x83, 0x94, 0xe7, 0x83, 0x95, 0xe7, 0x83, 0x96, 0xe7,
  0x83, 0x97, 0xe7, 0x83, 0x9a
};

unsigned char gbk_row_0x9f_converted_to_utf8[] = {
  0xe7, 0x83, 0x9c, 0xe7, 0x83, 0x9d,
  0xe7, 0x83, 0x9e, 0xe7, 0x83, 0xa0, 0xe7, 0x83, 0xa1, 0xe7, 0x83, 0xa2,
  0xe7, 0x83, 0xa3, 0xe7, 0x83, 0xa5, 0xe7, 0x83, 0xaa, 0xe7, 0x83, 0xae,
  0xe7, 0x83, 0xb0, 0xe7, 0x83, 0xb1, 0xe7, 0x83, 0xb2, 0xe7, 0x83, 0xb3,
  0xe7, 0x83, 0xb4, 0xe7, 0x83, 0xb5, 0xe7, 0x83, 0xb6, 0xe7, 0x83, 0xb8,
  0xe7, 0x83, 0xba, 0xe7, 0x83, 0xbb, 0xe7, 0x83, 0xbc, 0xe7, 0x83, 0xbe,
  0xe7, 0x83, 0xbf, 0xe7, 0x84, 0x80, 0xe7, 0x84, 0x81, 0xe7, 0x84, 0x82,
  0xe7, 0x84, 0x83, 0xe7, 0x84, 0x84, 0xe7, 0x84, 0x85, 0xe7, 0x84, 0x86,
  0xe7, 0x84, 0x87, 0xe7, 0x84, 0x88, 0xe7, 0x84, 0x8b, 0xe7, 0x84, 0x8c,
  0xe7, 0x84, 0x8d, 0xe7, 0x84, 0x8e, 0xe7, 0x84, 0x8f, 0xe7, 0x84, 0x91,
  0xe7, 0x84, 0x92, 0xe7, 0x84, 0x94, 0xe7, 0x84, 0x97, 0xe7, 0x84, 0x9b,
  0xe7, 0x84, 0x9c, 0xe7, 0x84, 0x9d, 0xe7, 0x84, 0x9e, 0xe7, 0x84, 0x9f,
  0xe7, 0x84, 0xa0, 0xe7, 0x84, 0xa1, 0xe7, 0x84, 0xa2, 0xe7, 0x84, 0xa3,
  0xe7, 0x84, 0xa4, 0xe7, 0x84, 0xa5, 0xe7, 0x84, 0xa7, 0xe7, 0x84, 0xa8,
  0xe7, 0x84, 0xa9, 0xe7, 0x84, 0xaa, 0xe7, 0x84, 0xab, 0xe7, 0x84, 0xac,
  0xe7, 0x84, 0xad, 0xe7, 0x84, 0xae, 0xe7, 0x84, 0xb2, 0xe7, 0x84, 0xb3,
  0xe7, 0x84, 0xb4, 0xe7, 0x84, 0xb5, 0xe7, 0x84, 0xb7, 0xe7, 0x84, 0xb8,
  0xe7, 0x84, 0xb9, 0xe7, 0x84, 0xba, 0xe7, 0x84, 0xbb, 0xe7, 0x84, 0xbc,
  0xe7, 0x84, 0xbd, 0xe7, 0x84, 0xbe, 0xe7, 0x84, 0xbf, 0xe7, 0x85, 0x80,
  0xe7, 0x85, 0x81, 0xe7, 0x85, 0x82, 0xe7, 0x85, 0x83, 0xe7, 0x85, 0x84,
  0xe7, 0x85, 0x86, 0xe7, 0x85, 0x87, 0xe7, 0x85, 0x88, 0xe7, 0x85, 0x89,
  0xe7, 0x85, 0x8b, 0xe7, 0x85, 0x8d, 0xe7, 0x85, 0x8f, 0xe7, 0x85, 0x90,
  0xe7, 0x85, 0x91, 0xe7, 0x85, 0x92, 0xe7, 0x85, 0x93, 0xe7, 0x85, 0x94,
  0xe7, 0x85, 0x95, 0xe7, 0x85, 0x96, 0xe7, 0x85, 0x97, 0xe7, 0x85, 0x98,
  0xe7, 0x85, 0x99, 0xe7, 0x85, 0x9a, 0xe7, 0x85, 0x9b, 0xe7, 0x85, 0x9d,
  0xe7, 0x85, 0x9f, 0xe7, 0x85, 0xa0, 0xe7, 0x85, 0xa1, 0xe7, 0x85, 0xa2,
  0xe7, 0x85, 0xa3, 0xe7, 0x85, 0xa5, 0xe7, 0x85, 0xa9, 0xe7, 0x85, 0xaa,
  0xe7, 0x85, 0xab, 0xe7, 0x85, 0xac, 0xe7, 0x85, 0xad, 0xe7, 0x85, 0xaf,
  0xe7, 0x85, 0xb0, 0xe7, 0x85, 0xb1, 0xe7, 0x85, 0xb4, 0xe7, 0x85, 0xb5,
  0xe7, 0x85, 0xb6, 0xe7, 0x85, 0xb7, 0xe7, 0x85, 0xb9, 0xe7, 0x85, 0xbb,
  0xe7, 0x85, 0xbc, 0xe7, 0x85, 0xbe, 0xe7, 0x85, 0xbf, 0xe7, 0x86, 0x80,
  0xe7, 0x86, 0x81, 0xe7, 0x86, 0x82, 0xe7, 0x86, 0x83, 0xe7, 0x86, 0x85,
  0xe7, 0x86, 0x86, 0xe7, 0x86, 0x87, 0xe7, 0x86, 0x88, 0xe7, 0x86, 0x89,
  0xe7, 0x86, 0x8b, 0xe7, 0x86, 0x8c, 0xe7, 0x86, 0x8d, 0xe7, 0x86, 0x8e,
  0xe7, 0x86, 0x90, 0xe7, 0x86, 0x91, 0xe7, 0x86, 0x92, 0xe7, 0x86, 0x93,
  0xe7, 0x86, 0x95, 0xe7, 0x86, 0x96, 0xe7, 0x86, 0x97, 0xe7, 0x86, 0x9a,
  0xe7, 0x86, 0x9b, 0xe7, 0x86, 0x9c, 0xe7, 0x86, 0x9d, 0xe7, 0x86, 0x9e,
  0xe7, 0x86, 0xa1, 0xe7, 0x86, 0xa2, 0xe7, 0x86, 0xa3, 0xe7, 0x86, 0xa4,
  0xe7, 0x86, 0xa5, 0xe7, 0x86, 0xa6, 0xe7, 0x86, 0xa7, 0xe7, 0x86, 0xa9,
  0xe7, 0x86, 0xaa, 0xe7, 0x86, 0xab, 0xe7, 0x86, 0xad, 0xe7, 0x86, 0xae,
  0xe7, 0x86, 0xaf, 0xe7, 0x86, 0xb0, 0xe7, 0x86, 0xb1, 0xe7, 0x86, 0xb2,
  0xe7, 0x86, 0xb4, 0xe7, 0x86, 0xb6, 0xe7, 0x86, 0xb7, 0xe7, 0x86, 0xb8,
  0xe7, 0x86, 0xba, 0xe7, 0x86, 0xbb, 0xe7, 0x86, 0xbc, 0xe7, 0x86, 0xbd,
  0xe7, 0x86, 0xbe, 0xe7, 0x86, 0xbf, 0xe7, 0x87, 0x80, 0xe7, 0x87, 0x81,
  0xe7, 0x87, 0x82, 0xe7, 0x87, 0x84, 0xe7, 0x87, 0x85, 0xe7, 0x87, 0x86,
  0xe7, 0x87, 0x87, 0xe7, 0x87, 0x88, 0xe7, 0x87, 0x89, 0xe7, 0x87, 0x8a,
  0xe7, 0x87, 0x8b, 0xe7, 0x87, 0x8c, 0xe7, 0x87, 0x8d, 0xe7, 0x87, 0x8f,
  0xe7, 0x87, 0x90, 0xe7, 0x87, 0x91, 0xe7, 0x87, 0x92, 0xe7, 0x87, 0x93
};

static unsigned char gbk_row_0xa0_converted_to_utf8[] = {
  0xe7, 0x87, 0x96, 0xe7, 0x87, 0x97, 0xe7, 0x87, 0x98, 0xe7, 0x87, 0x99,
  0xe7, 0x87, 0x9a, 0xe7, 0x87, 0x9b, 0xe7, 0x87, 0x9c, 0xe7, 0x87, 0x9d,
  0xe7, 0x87, 0x9e, 0xe7, 0x87, 0x9f, 0xe7, 0x87, 0xa1, 0xe7, 0x87, 0xa2,
  0xe7, 0x87, 0xa3, 0xe7, 0x87, 0xa4, 0xe7, 0x87, 0xa6, 0xe7, 0x87, 0xa8,
  0xe7, 0x87, 0xa9, 0xe7, 0x87, 0xaa, 0xe7, 0x87, 0xab, 0xe7, 0x87, 0xac,
  0xe7, 0x87, 0xad, 0xe7, 0x87, 0xaf, 0xe7, 0x87, 0xb0, 0xe7, 0x87, 0xb1,
  0xe7, 0x87, 0xb2, 0xe7, 0x87, 0xb3, 0xe7, 0x87, 0xb4, 0xe7, 0x87, 0xb5,
  0xe7, 0x87, 0xb6, 0xe7, 0x87, 0xb7, 0xe7, 0x87, 0xb8, 0xe7, 0x87, 0xba,
  0xe7, 0x87, 0xbb, 0xe7, 0x87, 0xbc, 0xe7, 0x87, 0xbd, 0xe7, 0x87, 0xbe,
  0xe7, 0x87, 0xbf, 0xe7, 0x88, 0x80, 0xe7, 0x88, 0x81, 0xe7, 0x88, 0x82,
  0xe7, 0x88, 0x83, 0xe7, 0x88, 0x84, 0xe7, 0x88, 0x85, 0xe7, 0x88, 0x87,
  0xe7, 0x88, 0x88, 0xe7, 0x88, 0x89, 0xe7, 0x88, 0x8a, 0xe7, 0x88, 0x8b,
  0xe7, 0x88, 0x8c, 0xe7, 0x88, 0x8d, 0xe7, 0x88, 0x8e, 0xe7, 0x88, 0x8f,
  0xe7, 0x88, 0x90, 0xe7, 0x88, 0x91, 0xe7, 0x88, 0x92, 0xe7, 0x88, 0x93,
  0xe7, 0x88, 0x94, 0xe7, 0x88, 0x95, 0xe7, 0x88, 0x96, 0xe7, 0x88, 0x97,
  0xe7, 0x88, 0x98, 0xe7, 0x88, 0x99, 0xe7, 0x88, 0x9a, 0xe7, 0x88, 0x9b,
  0xe7, 0x88, 0x9c, 0xe7, 0x88, 0x9e, 0xe7, 0x88, 0x9f, 0xe7, 0x88, 0xa0,
  0xe7, 0x88, 0xa1, 0xe7, 0x88, 0xa2, 0xe7, 0x88, 0xa3, 0xe7, 0x88, 0xa4,
  0xe7, 0x88, 0xa5, 0xe7, 0x88, 0xa6, 0xe7, 0x88, 0xa7, 0xe7, 0x88, 0xa9,
  0xe7, 0x88, 0xab, 0xe7, 0x88, 0xad, 0xe7, 0x88, 0xae, 0xe7, 0x88, 0xaf,
  0xe7, 0x88, 0xb2, 0xe7, 0x88, 0xb3, 0xe7, 0x88, 0xb4, 0xe7, 0x88, 0xba,
  0xe7, 0x88, 0xbc, 0xe7, 0x88, 0xbe, 0xe7, 0x89, 0x80, 0xe7, 0x89, 0x81,
  0xe7, 0x89, 0x82, 0xe7, 0x89, 0x83, 0xe7, 0x89, 0x84, 0xe7, 0x89, 0x85,
  0xe7, 0x89, 0x86, 0xe7, 0x89, 0x89, 0xe7, 0x89, 0x8a, 0xe7, 0x89, 0x8b,
  0xe7, 0x89, 0x8e, 0xe7, 0x89, 0x8f, 0xe7, 0x89, 0x90, 0xe7, 0x89, 0x91,
  0xe7, 0x89, 0x93, 0xe7, 0x89, 0x94, 0xe7, 0x89, 0x95, 0xe7, 0x89, 0x97,
  0xe7, 0x89, 0x98, 0xe7, 0x89, 0x9a, 0xe7, 0x89, 0x9c, 0xe7, 0x89, 0x9e,
  0xe7, 0x89, 0xa0, 0xe7, 0x89, 0xa3, 0xe7, 0x89, 0xa4, 0xe7, 0x89, 0xa5,
  0xe7, 0x89, 0xa8, 0xe7, 0x89, 0xaa, 0xe7, 0x89, 0xab, 0xe7, 0x89, 0xac,
  0xe7, 0x89, 0xad, 0xe7, 0x89, 0xb0, 0xe7, 0x89, 0xb1, 0xe7, 0x89, 0xb3,
  0xe7, 0x89, 0xb4, 0xe7, 0x89, 0xb6, 0xe7, 0x89, 0xb7, 0xe7, 0x89, 0xb8,
  0xe7, 0x89, 0xbb, 0xe7, 0x89, 0xbc, 0xe7, 0x89, 0xbd, 0xe7, 0x8a, 0x82,
  0xe7, 0x8a, 0x83, 0xe7, 0x8a, 0x85, 0xe7, 0x8a, 0x86, 0xe7, 0x8a, 0x87,
  0xe7, 0x8a, 0x88, 0xe7, 0x8a, 0x89, 0xe7, 0x8a, 0x8c, 0xe7, 0x8a, 0x8e,
  0xe7, 0x8a, 0x90, 0xe7, 0x8a, 0x91, 0xe7, 0x8a, 0x93, 0xe7, 0x8a, 0x94,
  0xe7, 0x8a, 0x95, 0xe7, 0x8a, 0x96, 0xe7, 0x8a, 0x97, 0xe7, 0x8a, 0x98,
  0xe7, 0x8a, 0x99, 0xe7, 0x8a, 0x9a, 0xe7, 0x8a, 0x9b, 0xe7, 0x8a, 0x9c,
  0xe7, 0x8a, 0x9d, 0xe7, 0x8a, 0x9e, 0xe7, 0x8a, 0xa0, 0xe7, 0x8a, 0xa1,
  0xe7, 0x8a, 0xa2, 0xe7, 0x8a, 0xa3, 0xe7, 0x8a, 0xa4, 0xe7, 0x8a, 0xa5,
  0xe7, 0x8a, 0xa6, 0xe7, 0x8a, 0xa7, 0xe7, 0x8a, 0xa8, 0xe7, 0x8a, 0xa9,
  0xe7, 0x8a, 0xaa, 0xe7, 0x8a, 0xab, 0xe7, 0x8a, 0xae, 0xe7, 0x8a, 0xb1,
  0xe7, 0x8a, 0xb2, 0xe7, 0x8a, 0xb3, 0xe7, 0x8a, 0xb5, 0xe7, 0x8a, 0xba,
  0xe7, 0x8a, 0xbb, 0xe7, 0x8a, 0xbc, 0xe7, 0x8a, 0xbd, 0xe7, 0x8a, 0xbe,
  0xe7, 0x8a, 0xbf, 0xe7, 0x8b, 0x80, 0xe7, 0x8b, 0x85, 0xe7, 0x8b, 0x86,
  0xe7, 0x8b, 0x87, 0xe7, 0x8b, 0x89, 0xe7, 0x8b, 0x8a, 0xe7, 0x8b, 0x8b,
  0xe7, 0x8b, 0x8c, 0xe7, 0x8b, 0x8f, 0xe7, 0x8b, 0x91, 0xe7, 0x8b, 0x93,
  0xe7, 0x8b, 0x94, 0xe7, 0x8b, 0x95, 0xe7, 0x8b, 0x96, 0xe7, 0x8b, 0x98,
  0xe7, 0x8b, 0x9a, 0xe7, 0x8b, 0x9b
};

// Rows 0xa1 to 0xa7 are identical to GB2312 and not repeated here.

static unsigned char gbk_row_0xa8_converted_to_utf8[] = {
  0xcb, 0x8a, 0xcb, 0x8b, 0xcb, 0x99, 0xe2, 0x80, 0x93, 0xe2, 0x80, 0x95,
  0xe2, 0x80, 0xa5, 0xe2, 0x80, 0xb5, 0xe2, 0x84, 0x85, 0xe2, 0x84, 0x89,
  0xe2, 0x86, 0x96, 0xe2, 0x86, 0x97, 0xe2, 0x86, 0x98, 0xe2, 0x86, 0x99,
  0xe2, 0x88, 0x95, 0xe2, 0x88, 0x9f, 0xe2, 0x88, 0xa3, 0xe2, 0x89, 0x92,
  0xe2, 0x89, 0xa6, 0xe2, 0x89, 0xa7, 0xe2, 0x8a, 0xbf, 0xe2, 0x95, 0x90,
  0xe2, 0x95, 0x91, 0xe2, 0x95, 0x92, 0xe2, 0x95, 0x93, 0xe2, 0x95, 0x94,
  0xe2, 0x95, 0x95, 0xe2, 0x95, 0x96, 0xe2, 0x95, 0x97, 0xe2, 0x95, 0x98,
  0xe2, 0x95, 0x99, 0xe2, 0x95, 0x9a, 0xe2, 0x95, 0x9b, 0xe2, 0x95, 0x9c,
  0xe2, 0x95, 0x9d, 0xe2, 0x95, 0x9e, 0xe2, 0x95, 0x9f, 0xe2, 0x95, 0xa0,
  0xe2, 0x95, 0xa1, 0xe2, 0x95, 0xa2, 0xe2, 0x95, 0xa3, 0xe2, 0x95, 0xa4,
  0xe2, 0x95, 0xa5, 0xe2, 0x95, 0xa6, 0xe2, 0x95, 0xa7, 0xe2, 0x95, 0xa8,
  0xe2, 0x95, 0xa9, 0xe2, 0x95, 0xaa, 0xe2, 0x95, 0xab, 0xe2, 0x95, 0xac,
  0xe2, 0x95, 0xad, 0xe2, 0x95, 0xae, 0xe2, 0x95, 0xaf, 0xe2, 0x95, 0xb0,
  0xe2, 0x95, 0xb1, 0xe2, 0x95, 0xb2, 0xe2, 0x95, 0xb3, 0xe2, 0x96, 0x81,
  0xe2, 0x96, 0x82, 0xe2, 0x96, 0x83, 0xe2, 0x96, 0x84, 0xe2, 0x96, 0x85,
  0xe2, 0x96, 0x86, 0xe2, 0x96, 0x87, 0xe2, 0x96, 0x88, 0xe2, 0x96, 0x89,
  0xe2, 0x96, 0x8a, 0xe2, 0x96, 0x8b, 0xe2, 0x96, 0x8c, 0xe2, 0x96, 0x8d,
  0xe2, 0x96, 0x8e, 0xe2, 0x96, 0x8f, 0xe2, 0x96, 0x93, 0xe2, 0x96, 0x94,
  0xe2, 0x96, 0x95, 0xe2, 0x96, 0xbc, 0xe2, 0x96, 0xbd, 0xe2, 0x97, 0xa2,
  0xe2, 0x97, 0xa3, 0xe2, 0x97, 0xa4, 0xe2, 0x97, 0xa5, 0xe2, 0x98, 0x89,
  0xe2, 0x8a, 0x95, 0xe3, 0x80, 0x92, 0xe3, 0x80, 0x9d, 0xe3, 0x80, 0x9e,
  0xc4, 0x81, 0xc3, 0xa1, 0xc7, 0x8e, 0xc3, 0xa0, 0xc4, 0x93, 0xc3, 0xa9,
  0xc4, 0x9b, 0xc3, 0xa8, 0xc4, 0xab, 0xc3, 0xad, 0xc7, 0x90, 0xc3, 0xac,
  0xc5, 0x8d, 0xc3, 0xb3, 0xc7, 0x92, 0xc3, 0xb2, 0xc5, 0xab, 0xc3, 0xba,
  0xc7, 0x94, 0xc3, 0xb9, 0xc7, 0x96, 0xc7, 0x98, 0xc7, 0x9a, 0xc7, 0x9c,
  0xc3, 0xbc, 0xc3, 0xaa, 0xe3, 0x84, 0x85, 0xe3, 0x84, 0x86, 0xe3, 0x84,
  0x87, 0xe3, 0x84, 0x88, 0xe3, 0x84, 0x89, 0xe3, 0x84, 0x8a, 0xe3, 0x84,
  0x8b, 0xe3, 0x84, 0x8c, 0xe3, 0x84, 0x8d, 0xe3, 0x84, 0x8e, 0xe3, 0x84,
  0x8f, 0xe3, 0x84, 0x90, 0xe3, 0x84, 0x91, 0xe3, 0x84, 0x92, 0xe3, 0x84,
  0x93, 0xe3, 0x84, 0x94, 0xe3, 0x84, 0x95, 0xe3, 0x84, 0x96, 0xe3, 0x84,
  0x97, 0xe3, 0x84, 0x98, 0xe3, 0x84, 0x99, 0xe3, 0x84, 0x9a, 0xe3, 0x84,
  0x9b, 0xe3, 0x84, 0x9c, 0xe3, 0x84, 0x9d, 0xe3, 0x84, 0x9e, 0xe3, 0x84,
  0x9f, 0xe3, 0x84, 0xa0, 0xe3, 0x84, 0xa1, 0xe3, 0x84, 0xa2, 0xe3, 0x84,
  0xa3, 0xe3, 0x84, 0xa4, 0xe3, 0x84, 0xa5, 0xe3, 0x84, 0xa6, 0xe3, 0x84,
  0xa7, 0xe3, 0x84, 0xa8, 0xe3, 0x84, 0xa9
};

unsigned char gbk_row_0xa9_converted_to_utf8[] = {
  0xe3, 0x80, 0xa1, 0xe3, 0x80, 0xa2, 0xe3, 0x80, 0xa3, 0xe3, 0x80, 0xa4,
  0xe3, 0x80, 0xa5, 0xe3, 0x80, 0xa6, 0xe3, 0x80, 0xa7, 0xe3, 0x80, 0xa8,
  0xe3, 0x80, 0xa9, 0xe3, 0x8a, 0xa3, 0xe3, 0x8e, 0x8e, 0xe3, 0x8e, 0x8f,
  0xe3, 0x8e, 0x9c, 0xe3, 0x8e, 0x9d, 0xe3, 0x8e, 0x9e, 0xe3, 0x8e, 0xa1,
  0xe3, 0x8f, 0x84, 0xe3, 0x8f, 0x8e, 0xe3, 0x8f, 0x91, 0xe3, 0x8f, 0x92,
  0xe3, 0x8f, 0x95, 0xef, 0xb8, 0xb0, 0xef, 0xbf, 0xa2, 0xef, 0xbf, 0xa4,
  0xe2, 0x84, 0xa1, 0xe3, 0x88, 0xb1, 0xe2, 0x80, 0x90, 0xe3, 0x83, 0xbc,
  0xe3, 0x82, 0x9b, 0xe3, 0x82, 0x9c, 0xe3, 0x83, 0xbd, 0xe3, 0x83, 0xbe,
  0xe3, 0x80, 0x86, 0xe3, 0x82, 0x9d, 0xe3, 0x82, 0x9e, 0xef, 0xb9, 0x89,
  0xef, 0xb9, 0x8a, 0xef, 0xb9, 0x8b, 0xef, 0xb9, 0x8c, 0xef, 0xb9, 0x8d,
  0xef, 0xb9, 0x8e, 0xef, 0xb9, 0x8f, 0xef, 0xb9, 0x90, 0xef, 0xb9, 0x91,
  0xef, 0xb9, 0x92, 0xef, 0xb9, 0x94, 0xef, 0xb9, 0x95, 0xef, 0xb9, 0x96,
  0xef, 0xb9, 0x97, 0xef, 0xb9, 0x99, 0xef, 0xb9, 0x9a, 0xef, 0xb9, 0x9b,
  0xef, 0xb9, 0x9c, 0xef, 0xb9, 0x9d, 0xef, 0xb9, 0x9e, 0xef, 0xb9, 0x9f,
  0xef, 0xb9, 0xa0, 0xef, 0xb9, 0xa1, 0xef, 0xb9, 0xa2, 0xef, 0xb9, 0xa3,
  0xef, 0xb9, 0xa4, 0xef, 0xb9, 0xa5, 0xef, 0xb9, 0xa6, 0xef, 0xb9, 0xa8,
  0xef, 0xb9, 0xa9, 0xef, 0xb9, 0xaa, 0xef, 0xb9, 0xab, 0xe2, 0x94, 0x80,
  0xe2, 0x94, 0x81, 0xe2, 0x94, 0x82, 0xe2, 0x94, 0x83, 0xe2, 0x94, 0x84,
  0xe2, 0x94, 0x85, 0xe2, 0x94, 0x86, 0xe2, 0x94, 0x87, 0xe2, 0x94, 0x88,
  0xe2, 0x94, 0x89, 0xe2, 0x94, 0x8a, 0xe2, 0x94, 0x8b, 0xe2, 0x94, 0x8c,
  0xe2, 0x94, 0x8d, 0xe2, 0x94, 0x8e, 0xe2, 0x94, 0x8f, 0xe2, 0x94, 0x90,
  0xe2, 0x94, 0x91, 0xe2, 0x94, 0x92, 0xe2, 0x94, 0x93, 0xe2, 0x94, 0x94,
  0xe2, 0x94, 0x95, 0xe2, 0x94, 0x96, 0xe2, 0x94, 0x97, 0xe2, 0x94, 0x98,
  0xe2, 0x94, 0x99, 0xe2, 0x94, 0x9a, 0xe2, 0x94, 0x9b, 0xe2, 0x94, 0x9c,
  0xe2, 0x94, 0x9d, 0xe2, 0x94, 0x9e, 0xe2, 0x94, 0x9f, 0xe2, 0x94, 0xa0,
  0xe2, 0x94, 0xa1, 0xe2, 0x94, 0xa2, 0xe2, 0x94, 0xa3, 0xe2, 0x94, 0xa4,
  0xe2, 0x94, 0xa5, 0xe2, 0x94, 0xa6, 0xe2, 0x94, 0xa7, 0xe2, 0x94, 0xa8,
  0xe2, 0x94, 0xa9, 0xe2, 0x94, 0xaa, 0xe2, 0x94, 0xab, 0xe2, 0x94, 0xac,
  0xe2, 0x94, 0xad, 0xe2, 0x94, 0xae, 0xe2, 0x94, 0xaf, 0xe2, 0x94, 0xb0,
  0xe2, 0x94, 0xb1, 0xe2, 0x94, 0xb2, 0xe2, 0x94, 0xb3, 0xe2, 0x94, 0xb4,
  0xe2, 0x94, 0xb5, 0xe2, 0x94, 0xb6, 0xe2, 0x94, 0xb7, 0xe2, 0x94, 0xb8,
  0xe2, 0x94, 0xb9, 0xe2, 0x94, 0xba, 0xe2, 0x94, 0xbb, 0xe2, 0x94, 0xbc,
  0xe2, 0x94, 0xbd, 0xe2, 0x94, 0xbe, 0xe2, 0x94, 0xbf, 0xe2, 0x95, 0x80,
  0xe2, 0x95, 0x81, 0xe2, 0x95, 0x82, 0xe2, 0x95, 0x83, 0xe2, 0x95, 0x84,
  0xe2, 0x95, 0x85, 0xe2, 0x95, 0x86, 0xe2, 0x95, 0x87, 0xe2, 0x95, 0x88,
  0xe2, 0x95, 0x89, 0xe2, 0x95, 0x8a, 0xe2, 0x95, 0x8b
};

unsigned char gbk_row_0xaa_converted_to_utf8[] = {
  0xe7, 0x8b, 0x9c, 0xe7, 0x8b, 0x9d, 0xe7, 0x8b, 0x9f, 0xe7, 0x8b, 0xa2,
  0xe7, 0x8b, 0xa3, 0xe7, 0x8b, 0xa4, 0xe7, 0x8b, 0xa5, 0xe7, 0x8b, 0xa6,
  0xe7, 0x8b, 0xa7, 0xe7, 0x8b, 0xaa, 0xe7, 0x8b, 0xab, 0xe7, 0x8b, 0xb5,
  0xe7, 0x8b, 0xb6, 0xe7, 0x8b, 0xb9, 0xe7, 0x8b, 0xbd, 0xe7, 0x8b, 0xbe,
  0xe7, 0x8b, 0xbf, 0xe7, 0x8c, 0x80, 0xe7, 0x8c, 0x82, 0xe7, 0x8c, 0x84,
  0xe7, 0x8c, 0x85, 0xe7, 0x8c, 0x86, 0xe7, 0x8c, 0x87, 0xe7, 0x8c, 0x88,
  0xe7, 0x8c, 0x89, 0xe7, 0x8c, 0x8b, 0xe7, 0x8c, 0x8c, 0xe7, 0x8c, 0x8d,
  0xe7, 0x8c, 0x8f, 0xe7, 0x8c, 0x90, 0xe7, 0x8c, 0x91, 0xe7, 0x8c, 0x92,
  0xe7, 0x8c, 0x94, 0xe7, 0x8c, 0x98, 0xe7, 0x8c, 0x99, 0xe7, 0x8c, 0x9a,
  0xe7, 0x8c, 0x9f, 0xe7, 0x8c, 0xa0, 0xe7, 0x8c, 0xa3, 0xe7, 0x8c, 0xa4,
  0xe7, 0x8c, 0xa6, 0xe7, 0x8c, 0xa7, 0xe7, 0x8c, 0xa8, 0xe7, 0x8c, 0xad,
  0xe7, 0x8c, 0xaf, 0xe7, 0x8c, 0xb0, 0xe7, 0x8c, 0xb2, 0xe7, 0x8c, 0xb3,
  0xe7, 0x8c, 0xb5, 0xe7, 0x8c, 0xb6, 0xe7, 0x8c, 0xba, 0xe7, 0x8c, 0xbb,
  0xe7, 0x8c, 0xbc, 0xe7, 0x8c, 0xbd, 0xe7, 0x8d, 0x80, 0xe7, 0x8d, 0x81,
  0xe7, 0x8d, 0x82, 0xe7, 0x8d, 0x83, 0xe7, 0x8d, 0x84, 0xe7, 0x8d, 0x85,
  0xe7, 0x8d, 0x86, 0xe7, 0x8d, 0x87, 0xe7, 0x8d, 0x88, 0xe7, 0x8d, 0x89,
  0xe7, 0x8d, 0x8a, 0xe7, 0x8d, 0x8b, 0xe7, 0x8d, 0x8c, 0xe7, 0x8d, 0x8e,
  0xe7, 0x8d, 0x8f, 0xe7, 0x8d, 0x91, 0xe7, 0x8d, 0x93, 0xe7, 0x8d, 0x94,
  0xe7, 0x8d, 0x95, 0xe7, 0x8d, 0x96, 0xe7, 0x8d, 0x98, 0xe7, 0x8d, 0x99,
  0xe7, 0x8d, 0x9a, 0xe7, 0x8d, 0x9b, 0xe7, 0x8d, 0x9c, 0xe7, 0x8d, 0x9d,
  0xe7, 0x8d, 0x9e, 0xe7, 0x8d, 0x9f, 0xe7, 0x8d, 0xa1, 0xe7, 0x8d, 0xa2,
  0xe7, 0x8d, 0xa3, 0xe7, 0x8d, 0xa4, 0xe7, 0x8d, 0xa5, 0xe7, 0x8d, 0xa6,
  0xe7, 0x8d, 0xa7, 0xe7, 0x8d, 0xa8, 0xe7, 0x8d, 0xa9, 0xe7, 0x8d, 0xaa,
  0xe7, 0x8d, 0xab, 0xe7, 0x8d, 0xae, 0xe7, 0x8d, 0xb0, 0xe7, 0x8d, 0xb1,
};

unsigned char gbk_row_0xab_converted_to_utf8[] = {
  0xe7, 0x8d, 0xb2, 0xe7, 0x8d, 0xb3, 0xe7, 0x8d, 0xb4, 0xe7, 0x8d,
  0xb5, 0xe7, 0x8d, 0xb6, 0xe7, 0x8d, 0xb7, 0xe7, 0x8d, 0xb8, 0xe7, 0x8d,
  0xb9, 0xe7, 0x8d, 0xba, 0xe7, 0x8d, 0xbb, 0xe7, 0x8d, 0xbc, 0xe7, 0x8d,
  0xbd, 0xe7, 0x8d, 0xbf, 0xe7, 0x8e, 0x80, 0xe7, 0x8e, 0x81, 0xe7, 0x8e,
  0x82, 0xe7, 0x8e, 0x83, 0xe7, 0x8e, 0x85, 0xe7, 0x8e, 0x86, 0xe7, 0x8e,
  0x88, 0xe7, 0x8e, 0x8a, 0xe7, 0x8e, 0x8c, 0xe7, 0x8e, 0x8d, 0xe7, 0x8e,
  0x8f, 0xe7, 0x8e, 0x90, 0xe7, 0x8e, 0x92, 0xe7, 0x8e, 0x93, 0xe7, 0x8e,
  0x94, 0xe7, 0x8e, 0x95, 0xe7, 0x8e, 0x97, 0xe7, 0x8e, 0x98, 0xe7, 0x8e,
  0x99, 0xe7, 0x8e, 0x9a, 0xe7, 0x8e, 0x9c, 0xe7, 0x8e, 0x9d, 0xe7, 0x8e,
  0x9e, 0xe7, 0x8e, 0xa0, 0xe7, 0x8e, 0xa1, 0xe7, 0x8e, 0xa3, 0xe7, 0x8e,
  0xa4, 0xe7, 0x8e, 0xa5, 0xe7, 0x8e, 0xa6, 0xe7, 0x8e, 0xa7, 0xe7, 0x8e,
  0xa8, 0xe7, 0x8e, 0xaa, 0xe7, 0x8e, 0xac, 0xe7, 0x8e, 0xad, 0xe7, 0x8e,
  0xb1, 0xe7, 0x8e, 0xb4, 0xe7, 0x8e, 0xb5, 0xe7, 0x8e, 0xb6, 0xe7, 0x8e,
  0xb8, 0xe7, 0x8e, 0xb9, 0xe7, 0x8e, 0xbc, 0xe7, 0x8e, 0xbd, 0xe7, 0x8e,
  0xbe, 0xe7, 0x8e, 0xbf, 0xe7, 0x8f, 0x81, 0xe7, 0x8f, 0x83, 0xe7, 0x8f,
  0x84, 0xe7, 0x8f, 0x85, 0xe7, 0x8f, 0x86, 0xe7, 0x8f, 0x87, 0xe7, 0x8f,
  0x8b, 0xe7, 0x8f, 0x8c, 0xe7, 0x8f, 0x8e, 0xe7, 0x8f, 0x92, 0xe7, 0x8f,
  0x93, 0xe7, 0x8f, 0x94, 0xe7, 0x8f, 0x95, 0xe7, 0x8f, 0x96, 0xe7, 0x8f,
  0x97, 0xe7, 0x8f, 0x98, 0xe7, 0x8f, 0x9a, 0xe7, 0x8f, 0x9b, 0xe7, 0x8f,
  0x9c, 0xe7, 0x8f, 0x9d, 0xe7, 0x8f, 0x9f, 0xe7, 0x8f, 0xa1, 0xe7, 0x8f,
  0xa2, 0xe7, 0x8f, 0xa3, 0xe7, 0x8f, 0xa4, 0xe7, 0x8f, 0xa6, 0xe7, 0x8f,
  0xa8, 0xe7, 0x8f, 0xaa, 0xe7, 0x8f, 0xab, 0xe7, 0x8f, 0xac, 0xe7, 0x8f,
  0xae, 0xe7, 0x8f, 0xaf, 0xe7, 0x8f, 0xb0, 0xe7, 0x8f, 0xb1, 0xe7, 0x8f,
  0xb3, 0xe7, 0x8f, 0xb4, 0xe7, 0x8f, 0xb5, 0xe7, 0x8f, 0xb6, 0xe7, 0x8f,
  0xb7
};

unsigned char gbk_row_0xac_converted_to_utf8[] = {
  0xe7, 0x8f, 0xb8, 0xe7, 0x8f, 0xb9, 0xe7, 0x8f, 0xba, 0xe7,
  0x8f, 0xbb, 0xe7, 0x8f, 0xbc, 0xe7, 0x8f, 0xbd, 0xe7, 0x8f, 0xbe, 0xe7,
  0x8f, 0xbf, 0xe7, 0x90, 0x80, 0xe7, 0x90, 0x81, 0xe7, 0x90, 0x82, 0xe7,
  0x90, 0x84, 0xe7, 0x90, 0x87, 0xe7, 0x90, 0x88, 0xe7, 0x90, 0x8b, 0xe7,
  0x90, 0x8c, 0xe7, 0x90, 0x8d, 0xe7, 0x90, 0x8e, 0xe7, 0x90, 0x91, 0xe7,
  0x90, 0x92, 0xe7, 0x90, 0x93, 0xe7, 0x90, 0x94, 0xe7, 0x90, 0x95, 0xe7,
  0x90, 0x96, 0xe7, 0x90, 0x97, 0xe7, 0x90, 0x98, 0xe7, 0x90, 0x99, 0xe7,
  0x90, 0x9c, 0xe7, 0x90, 0x9d, 0xe7, 0x90, 0x9e, 0xe7, 0x90, 0x9f, 0xe7,
  0x90, 0xa0, 0xe7, 0x90, 0xa1, 0xe7, 0x90, 0xa3, 0xe7, 0x90, 0xa4, 0xe7,
  0x90, 0xa7, 0xe7, 0x90, 0xa9, 0xe7, 0x90, 0xab, 0xe7, 0x90, 0xad, 0xe7,
  0x90, 0xaf, 0xe7, 0x90, 0xb1, 0xe7, 0x90, 0xb2, 0xe7, 0x90, 0xb7, 0xe7,
  0x90, 0xb8, 0xe7, 0x90, 0xb9, 0xe7, 0x90, 0xba, 0xe7, 0x90, 0xbb, 0xe7,
  0x90, 0xbd, 0xe7, 0x90, 0xbe, 0xe7, 0x90, 0xbf, 0xe7, 0x91, 0x80, 0xe7,
  0x91, 0x82, 0xe7, 0x91, 0x83, 0xe7, 0x91, 0x84, 0xe7, 0x91, 0x85, 0xe7,
  0x91, 0x86, 0xe7, 0x91, 0x87, 0xe7, 0x91, 0x88, 0xe7, 0x91, 0x89, 0xe7,
  0x91, 0x8a, 0xe7, 0x91, 0x8b, 0xe7, 0x91, 0x8c, 0xe7, 0x91, 0x8d, 0xe7,
  0x91, 0x8e, 0xe7, 0x91, 0x8f, 0xe7, 0x91, 0x90, 0xe7, 0x91, 0x91, 0xe7,
  0x91, 0x92, 0xe7, 0x91, 0x93, 0xe7, 0x91, 0x94, 0xe7, 0x91, 0x96, 0xe7,
  0x91, 0x98, 0xe7, 0x91, 0x9d, 0xe7, 0x91, 0xa0, 0xe7, 0x91, 0xa1, 0xe7,
  0x91, 0xa2, 0xe7, 0x91, 0xa3, 0xe7, 0x91, 0xa4, 0xe7, 0x91, 0xa5, 0xe7,
  0x91, 0xa6, 0xe7, 0x91, 0xa7, 0xe7, 0x91, 0xa8, 0xe7, 0x91, 0xa9, 0xe7,
  0x91, 0xaa, 0xe7, 0x91, 0xab, 0xe7, 0x91, 0xac, 0xe7, 0x91, 0xae, 0xe7,
  0x91, 0xaf, 0xe7, 0x91, 0xb1, 0xe7, 0x91, 0xb2, 0xe7, 0x91, 0xb3, 0xe7,
  0x91, 0xb4, 0xe7, 0x91, 0xb5, 0xe7, 0x91, 0xb8, 0xe7, 0x91, 0xb9, 0xe7,
  0x91, 0xba
};

unsigned char gbk_row_0xad_converted_to_utf8[] = {
  0xe7, 0x91, 0xbb, 0xe7, 0x91, 0xbc, 0xe7, 0x91, 0xbd,
  0xe7, 0x91, 0xbf, 0xe7, 0x92, 0x82, 0xe7, 0x92, 0x84, 0xe7, 0x92, 0x85,
  0xe7, 0x92, 0x86, 0xe7, 0x92, 0x88, 0xe7, 0x92, 0x89, 0xe7, 0x92, 0x8a,
  0xe7, 0x92, 0x8c, 0xe7, 0x92, 0x8d, 0xe7, 0x92, 0x8f, 0xe7, 0x92, 0x91,
  0xe7, 0x92, 0x92, 0xe7, 0x92, 0x93, 0xe7, 0x92, 0x94, 0xe7, 0x92, 0x95,
  0xe7, 0x92, 0x96, 0xe7, 0x92, 0x97, 0xe7, 0x92, 0x98, 0xe7, 0x92, 0x99,
  0xe7, 0x92, 0x9a, 0xe7, 0x92, 0x9b, 0xe7, 0x92, 0x9d, 0xe7, 0x92, 0x9f,
  0xe7, 0x92, 0xa0, 0xe7, 0x92, 0xa1, 0xe7, 0x92, 0xa2, 0xe7, 0x92, 0xa3,
  0xe7, 0x92, 0xa4, 0xe7, 0x92, 0xa5, 0xe7, 0x92, 0xa6, 0xe7, 0x92, 0xaa,
  0xe7, 0x92, 0xab, 0xe7, 0x92, 0xac, 0xe7, 0x92, 0xad, 0xe7, 0x92, 0xae,
  0xe7, 0x92, 0xaf, 0xe7, 0x92, 0xb0, 0xe7, 0x92, 0xb1, 0xe7, 0x92, 0xb2,
  0xe7, 0x92, 0xb3, 0xe7, 0x92, 0xb4, 0xe7, 0x92, 0xb5, 0xe7, 0x92, 0xb6,
  0xe7, 0x92, 0xb7, 0xe7, 0x92, 0xb8, 0xe7, 0x92, 0xb9, 0xe7, 0x92, 0xbb,
  0xe7, 0x92, 0xbc, 0xe7, 0x92, 0xbd, 0xe7, 0x92, 0xbe, 0xe7, 0x92, 0xbf,
  0xe7, 0x93, 0x80, 0xe7, 0x93, 0x81, 0xe7, 0x93, 0x82, 0xe7, 0x93, 0x83,
  0xe7, 0x93, 0x84, 0xe7, 0x93, 0x85, 0xe7, 0x93, 0x86, 0xe7, 0x93, 0x87,
  0xe7, 0x93, 0x88, 0xe7, 0x93, 0x89, 0xe7, 0x93, 0x8a, 0xe7, 0x93, 0x8b,
  0xe7, 0x93, 0x8c, 0xe7, 0x93, 0x8d, 0xe7, 0x93, 0x8e, 0xe7, 0x93, 0x8f,
  0xe7, 0x93, 0x90, 0xe7, 0x93, 0x91, 0xe7, 0x93, 0x93, 0xe7, 0x93, 0x94,
  0xe7, 0x93, 0x95, 0xe7, 0x93, 0x96, 0xe7, 0x93, 0x97, 0xe7, 0x93, 0x98,
  0xe7, 0x93, 0x99, 0xe7, 0x93, 0x9a, 0xe7, 0x93, 0x9b, 0xe7, 0x93, 0x9d,
  0xe7, 0x93, 0x9f, 0xe7, 0x93, 0xa1, 0xe7, 0x93, 0xa5, 0xe7, 0x93, 0xa7,
  0xe7, 0x93, 0xa8, 0xe7, 0x93, 0xa9, 0xe7, 0x93, 0xaa, 0xe7, 0x93, 0xab,
  0xe7, 0x93, 0xac, 0xe7, 0x93, 0xad, 0xe7, 0x93, 0xb0, 0xe7, 0x93, 0xb1,
  0xe7, 0x93, 0xb2
};

unsigned char gbk_row_0xae_converted_to_utf8[] = {
  0xe7, 0x93, 0xb3, 0xe7, 0x93, 0xb5, 0xe7, 0x93,
  0xb8, 0xe7, 0x93, 0xb9, 0xe7, 0x93, 0xba, 0xe7, 0x93, 0xbb, 0xe7, 0x93,
  0xbc, 0xe7, 0x93, 0xbd, 0xe7, 0x93, 0xbe, 0xe7, 0x94, 0x80, 0xe7, 0x94,
  0x81, 0xe7, 0x94, 0x82, 0xe7, 0x94, 0x83, 0xe7, 0x94, 0x85, 0xe7, 0x94,
  0x86, 0xe7, 0x94, 0x87, 0xe7, 0x94, 0x88, 0xe7, 0x94, 0x89, 0xe7, 0x94,
  0x8a, 0xe7, 0x94, 0x8b, 0xe7, 0x94, 0x8c, 0xe7, 0x94, 0x8e, 0xe7, 0x94,
  0x90, 0xe7, 0x94, 0x92, 0xe7, 0x94, 0x94, 0xe7, 0x94, 0x95, 0xe7, 0x94,
  0x96, 0xe7, 0x94, 0x97, 0xe7, 0x94, 0x9b, 0xe7, 0x94, 0x9d, 0xe7, 0x94,
  0x9e, 0xe7, 0x94, 0xa0, 0xe7, 0x94, 0xa1, 0xe7, 0x94, 0xa2, 0xe7, 0x94,
  0xa3, 0xe7, 0x94, 0xa4, 0xe7, 0x94, 0xa6, 0xe7, 0x94, 0xa7, 0xe7, 0x94,
  0xaa, 0xe7, 0x94, 0xae, 0xe7, 0x94, 0xb4, 0xe7, 0x94, 0xb6, 0xe7, 0x94,
  0xb9, 0xe7, 0x94, 0xbc, 0xe7, 0x94, 0xbd, 0xe7, 0x94, 0xbf, 0xe7, 0x95,
  0x81, 0xe7, 0x95, 0x82, 0xe7, 0x95, 0x83, 0xe7, 0x95, 0x84, 0xe7, 0x95,
  0x86, 0xe7, 0x95, 0x87, 0xe7, 0x95, 0x89, 0xe7, 0x95, 0x8a, 0xe7, 0x95,
  0x8d, 0xe7, 0x95, 0x90, 0xe7, 0x95, 0x91, 0xe7, 0x95, 0x92, 0xe7, 0x95,
  0x93, 0xe7, 0x95, 0x95, 0xe7, 0x95, 0x96, 0xe7, 0x95, 0x97, 0xe7, 0x95,
  0x98, 0xe7, 0x95, 0x9d, 0xe7, 0x95, 0x9e, 0xe7, 0x95, 0x9f, 0xe7, 0x95,
  0xa0, 0xe7, 0x95, 0xa1, 0xe7, 0x95, 0xa2, 0xe7, 0x95, 0xa3, 0xe7, 0x95,
  0xa4, 0xe7, 0x95, 0xa7, 0xe7, 0x95, 0xa8, 0xe7, 0x95, 0xa9, 0xe7, 0x95,
  0xab, 0xe7, 0x95, 0xac, 0xe7, 0x95, 0xad, 0xe7, 0x95, 0xae, 0xe7, 0x95,
  0xaf, 0xe7, 0x95, 0xb0, 0xe7, 0x95, 0xb1, 0xe7, 0x95, 0xb3, 0xe7, 0x95,
  0xb5, 0xe7, 0x95, 0xb6, 0xe7, 0x95, 0xb7, 0xe7, 0x95, 0xba, 0xe7, 0x95,
  0xbb, 0xe7, 0x95, 0xbc, 0xe7, 0x95, 0xbd, 0xe7, 0x95, 0xbe, 0xe7, 0x96,
  0x80, 0xe7, 0x96, 0x81, 0xe7, 0x96, 0x82, 0xe7, 0x96, 0x84, 0xe7, 0x96,
  0x85, 0xe7, 0x96, 0x87
};

unsigned char gbk_row_0xaf_converted_to_utf8[] = {
  0xe7, 0x96, 0x88, 0xe7, 0x96, 0x89, 0xe7,
  0x96, 0x8a, 0xe7, 0x96, 0x8c, 0xe7, 0x96, 0x8d, 0xe7, 0x96, 0x8e, 0xe7,
  0x96, 0x90, 0xe7, 0x96, 0x93, 0xe7, 0x96, 0x95, 0xe7, 0x96, 0x98, 0xe7,
  0x96, 0x9b, 0xe7, 0x96, 0x9c, 0xe7, 0x96, 0x9e, 0xe7, 0x96, 0xa2, 0xe7,
  0x96, 0xa6, 0xe7, 0x96, 0xa7, 0xe7, 0x96, 0xa8, 0xe7, 0x96, 0xa9, 0xe7,
  0x96, 0xaa, 0xe7, 0x96, 0xad, 0xe7, 0x96, 0xb6, 0xe7, 0x96, 0xb7, 0xe7,
  0x96, 0xba, 0xe7, 0x96, 0xbb, 0xe7, 0x96, 0xbf, 0xe7, 0x97, 0x80, 0xe7,
  0x97, 0x81, 0xe7, 0x97, 0x86, 0xe7, 0x97, 0x8b, 0xe7, 0x97, 0x8c, 0xe7,
  0x97, 0x8e, 0xe7, 0x97, 0x8f, 0xe7, 0x97, 0x90, 0xe7, 0x97, 0x91, 0xe7,
  0x97, 0x93, 0xe7, 0x97, 0x97, 0xe7, 0x97, 0x99, 0xe7, 0x97, 0x9a, 0xe7,
  0x97, 0x9c, 0xe7, 0x97, 0x9d, 0xe7, 0x97, 0x9f, 0xe7, 0x97, 0xa0, 0xe7,
  0x97, 0xa1, 0xe7, 0x97, 0xa5, 0xe7, 0x97, 0xa9, 0xe7, 0x97, 0xac, 0xe7,
  0x97, 0xad, 0xe7, 0x97, 0xae, 0xe7, 0x97, 0xaf, 0xe7, 0x97, 0xb2, 0xe7,
  0x97, 0xb3, 0xe7, 0x97, 0xb5, 0xe7, 0x97, 0xb6, 0xe7, 0x97, 0xb7, 0xe7,
  0x97, 0xb8, 0xe7, 0x97, 0xba, 0xe7, 0x97, 0xbb, 0xe7, 0x97, 0xbd, 0xe7,
  0x97, 0xbe, 0xe7, 0x98, 0x82, 0xe7, 0x98, 0x84, 0xe7, 0x98, 0x86, 0xe7,
  0x98, 0x87, 0xe7, 0x98, 0x88, 0xe7, 0x98, 0x89, 0xe7, 0x98, 0x8b, 0xe7,
  0x98, 0x8d, 0xe7, 0x98, 0x8e, 0xe7, 0x98, 0x8f, 0xe7, 0x98, 0x91, 0xe7,
  0x98, 0x92, 0xe7, 0x98, 0x93, 0xe7, 0x98, 0x94, 0xe7, 0x98, 0x96, 0xe7,
  0x98, 0x9a, 0xe7, 0x98, 0x9c, 0xe7, 0x98, 0x9d, 0xe7, 0x98, 0x9e, 0xe7,
  0x98, 0xa1, 0xe7, 0x98, 0xa3, 0xe7, 0x98, 0xa7, 0xe7, 0x98, 0xa8, 0xe7,
  0x98, 0xac, 0xe7, 0x98, 0xae, 0xe7, 0x98, 0xaf, 0xe7, 0x98, 0xb1, 0xe7,
  0x98, 0xb2, 0xe7, 0x98, 0xb6, 0xe7, 0x98, 0xb7, 0xe7, 0x98, 0xb9, 0xe7,
  0x98, 0xba, 0xe7, 0x98, 0xbb, 0xe7, 0x98, 0xbd, 0xe7, 0x99, 0x81, 0xe7,
  0x99, 0x82, 0xe7, 0x99, 0x84
};

unsigned char gbk_row_0xb0_converted_to_utf8[] = {
  0xe7, 0x99, 0x85, 0xe7, 0x99, 0x86, 0xe7, 0x99, 0x87, 0xe7, 0x99, 0x88,
  0xe7, 0x99, 0x89, 0xe7, 0x99, 0x8a, 0xe7, 0x99, 0x8b, 0xe7, 0x99, 0x8e,
  0xe7, 0x99, 0x8f, 0xe7, 0x99, 0x90, 0xe7, 0x99, 0x91, 0xe7, 0x99, 0x92,
  0xe7, 0x99, 0x93, 0xe7, 0x99, 0x95, 0xe7, 0x99, 0x97, 0xe7, 0x99, 0x98,
  0xe7, 0x99, 0x99, 0xe7, 0x99, 0x9a, 0xe7, 0x99, 0x9b, 0xe7, 0x99, 0x9d,
  0xe7, 0x99, 0x9f, 0xe7, 0x99, 0xa0, 0xe7, 0x99, 0xa1, 0xe7, 0x99, 0xa2,
  0xe7, 0x99, 0xa4, 0xe7, 0x99, 0xa5, 0xe7, 0x99, 0xa6, 0xe7, 0x99, 0xa7,
  0xe7, 0x99, 0xa8, 0xe7, 0x99, 0xa9, 0xe7, 0x99, 0xaa, 0xe7, 0x99, 0xac,
  0xe7, 0x99, 0xad, 0xe7, 0x99, 0xae, 0xe7, 0x99, 0xb0, 0xe7, 0x99, 0xb1,
  0xe7, 0x99, 0xb2, 0xe7, 0x99, 0xb3, 0xe7, 0x99, 0xb4, 0xe7, 0x99, 0xb5,
  0xe7, 0x99, 0xb6, 0xe7, 0x99, 0xb7, 0xe7, 0x99, 0xb9, 0xe7, 0x99, 0xba,
  0xe7, 0x99, 0xbc, 0xe7, 0x99, 0xbf, 0xe7, 0x9a, 0x80, 0xe7, 0x9a, 0x81,
  0xe7, 0x9a, 0x83, 0xe7, 0x9a, 0x85, 0xe7, 0x9a, 0x89, 0xe7, 0x9a, 0x8a,
  0xe7, 0x9a, 0x8c, 0xe7, 0x9a, 0x8d, 0xe7, 0x9a, 0x8f, 0xe7, 0x9a, 0x90,
  0xe7, 0x9a, 0x92, 0xe7, 0x9a, 0x94, 0xe7, 0x9a, 0x95, 0xe7, 0x9a, 0x97,
  0xe7, 0x9a, 0x98, 0xe7, 0x9a, 0x9a, 0xe7, 0x9a, 0x9b, 0xe7, 0x9a, 0x9c,
  0xe7, 0x9a, 0x9d, 0xe7, 0x9a, 0x9e, 0xe7, 0x9a, 0x9f, 0xe7, 0x9a, 0xa0,
  0xe7, 0x9a, 0xa1, 0xe7, 0x9a, 0xa2, 0xe7, 0x9a, 0xa3, 0xe7, 0x9a, 0xa5,
  0xe7, 0x9a, 0xa6, 0xe7, 0x9a, 0xa7, 0xe7, 0x9a, 0xa8, 0xe7, 0x9a, 0xa9,
  0xe7, 0x9a, 0xaa, 0xe7, 0x9a, 0xab, 0xe7, 0x9a, 0xac, 0xe7, 0x9a, 0xad,
  0xe7, 0x9a, 0xaf, 0xe7, 0x9a, 0xb0, 0xe7, 0x9a, 0xb3, 0xe7, 0x9a, 0xb5,
  0xe7, 0x9a, 0xb6, 0xe7, 0x9a, 0xb7, 0xe7, 0x9a, 0xb8, 0xe7, 0x9a, 0xb9,
  0xe7, 0x9a, 0xba, 0xe7, 0x9a, 0xbb, 0xe7, 0x9a, 0xbc, 0xe7, 0x9a, 0xbd,
  0xe7, 0x9a, 0xbe, 0xe7, 0x9b, 0x80, 0xe7, 0x9b, 0x81, 0xe7, 0x9b, 0x83,
  0xe5, 0x95, 0x8a, 0xe9, 0x98, 0xbf, 0xe5, 0x9f, 0x83, 0xe6, 0x8c, 0xa8,
  0xe5, 0x93, 0x8e, 0xe5, 0x94, 0x89, 0xe5, 0x93, 0x80, 0xe7, 0x9a, 0x91,
  0xe7, 0x99, 0x8c, 0xe8, 0x94, 0xbc, 0xe7, 0x9f, 0xae, 0xe8, 0x89, 0xbe,
  0xe7, 0xa2, 0x8d, 0xe7, 0x88, 0xb1, 0xe9, 0x9a, 0x98, 0xe9, 0x9e, 0x8d,
  0xe6, 0xb0, 0xa8, 0xe5, 0xae, 0x89, 0xe4, 0xbf, 0xba, 0xe6, 0x8c, 0x89,
  0xe6, 0x9a, 0x97, 0xe5, 0xb2, 0xb8, 0xe8, 0x83, 0xba, 0xe6, 0xa1, 0x88,
  0xe8, 0x82, 0xae, 0xe6, 0x98, 0x82, 0xe7, 0x9b, 0x8e, 0xe5, 0x87, 0xb9,
  0xe6, 0x95, 0x96, 0xe7, 0x86, 0xac, 0xe7, 0xbf, 0xb1, 0xe8, 0xa2, 0x84,
  0xe5, 0x82, 0xb2, 0xe5, 0xa5, 0xa5, 0xe6, 0x87, 0x8a, 0xe6, 0xbe, 0xb3,
  0xe8, 0x8a, 0xad, 0xe6, 0x8d, 0x8c, 0xe6, 0x89, 0x92, 0xe5, 0x8f, 0xad,
  0xe5, 0x90, 0xa7, 0xe7, 0xac, 0x86, 0xe5, 0x85, 0xab, 0xe7, 0x96, 0xa4,
  0xe5, 0xb7, 0xb4, 0xe6, 0x8b, 0x94, 0xe8, 0xb7, 0x8b, 0xe9, 0x9d, 0xb6,
  0xe6, 0x8a, 0x8a, 0xe8, 0x80, 0x99, 0xe5, 0x9d, 0x9d, 0xe9, 0x9c, 0xb8,
  0xe7, 0xbd, 0xa2, 0xe7, 0x88, 0xb8, 0xe7, 0x99, 0xbd, 0xe6, 0x9f, 0x8f,
  0xe7, 0x99, 0xbe, 0xe6, 0x91, 0x86, 0xe4, 0xbd, 0xb0, 0xe8, 0xb4, 0xa5,
  0xe6, 0x8b, 0x9c, 0xe7, 0xa8, 0x97, 0xe6, 0x96, 0x91, 0xe7, 0x8f, 0xad,
  0xe6, 0x90, 0xac, 0xe6, 0x89, 0xb3, 0xe8, 0x88, 0xac, 0xe9, 0xa2, 0x81,
  0xe6, 0x9d, 0xbf, 0xe7, 0x89, 0x88, 0xe6, 0x89, 0xae, 0xe6, 0x8b, 0x8c,
  0xe4, 0xbc, 0xb4, 0xe7, 0x93, 0xa3, 0xe5, 0x8d, 0x8a, 0xe5, 0x8a, 0x9e,
  0xe7, 0xbb, 0x8a, 0xe9, 0x82, 0xa6, 0xe5, 0xb8, 0xae, 0xe6, 0xa2, 0x86,
  0xe6, 0xa6, 0x9c, 0xe8, 0x86, 0x80, 0xe7, 0xbb, 0x91, 0xe6, 0xa3, 0x92,
  0xe7, 0xa3, 0x85, 0xe8, 0x9a, 0x8c, 0xe9, 0x95, 0x91, 0xe5, 0x82, 0x8d,
  0xe8, 0xb0, 0xa4, 0xe8, 0x8b, 0x9e, 0xe8, 0x83, 0x9e, 0xe5, 0x8c, 0x85,
  0xe8, 0xa4, 0x92, 0xe5, 0x89, 0xa5
};

unsigned char gbk_row_0xb1_converted_to_utf8[] = {
  0xe7, 0x9b, 0x84, 0xe7, 0x9b,
  0x87, 0xe7, 0x9b, 0x89, 0xe7, 0x9b, 0x8b, 0xe7, 0x9b, 0x8c, 0xe7, 0x9b,
  0x93, 0xe7, 0x9b, 0x95, 0xe7, 0x9b, 0x99, 0xe7, 0x9b, 0x9a, 0xe7, 0x9b,
  0x9c, 0xe7, 0x9b, 0x9d, 0xe7, 0x9b, 0x9e, 0xe7, 0x9b, 0xa0, 0xe7, 0x9b,
  0xa1, 0xe7, 0x9b, 0xa2, 0xe7, 0x9b, 0xa3, 0xe7, 0x9b, 0xa4, 0xe7, 0x9b,
  0xa6, 0xe7, 0x9b, 0xa7, 0xe7, 0x9b, 0xa8, 0xe7, 0x9b, 0xa9, 0xe7, 0x9b,
  0xaa, 0xe7, 0x9b, 0xab, 0xe7, 0x9b, 0xac, 0xe7, 0x9b, 0xad, 0xe7, 0x9b,
  0xb0, 0xe7, 0x9b, 0xb3, 0xe7, 0x9b, 0xb5, 0xe7, 0x9b, 0xb6, 0xe7, 0x9b,
  0xb7, 0xe7, 0x9b, 0xba, 0xe7, 0x9b, 0xbb, 0xe7, 0x9b, 0xbd, 0xe7, 0x9b,
  0xbf, 0xe7, 0x9c, 0x80, 0xe7, 0x9c, 0x82, 0xe7, 0x9c, 0x83, 0xe7, 0x9c,
  0x85, 0xe7, 0x9c, 0x86, 0xe7, 0x9c, 0x8a, 0xe7, 0x9c, 0x8c, 0xe7, 0x9c,
  0x8e, 0xe7, 0x9c, 0x8f, 0xe7, 0x9c, 0x90, 0xe7, 0x9c, 0x91, 0xe7, 0x9c,
  0x92, 0xe7, 0x9c, 0x93, 0xe7, 0x9c, 0x94, 0xe7, 0x9c, 0x95, 0xe7, 0x9c,
  0x96, 0xe7, 0x9c, 0x97, 0xe7, 0x9c, 0x98, 0xe7, 0x9c, 0x9b, 0xe7, 0x9c,
  0x9c, 0xe7, 0x9c, 0x9d, 0xe7, 0x9c, 0x9e, 0xe7, 0x9c, 0xa1, 0xe7, 0x9c,
  0xa3, 0xe7, 0x9c, 0xa4, 0xe7, 0x9c, 0xa5, 0xe7, 0x9c, 0xa7, 0xe7, 0x9c,
  0xaa, 0xe7, 0x9c, 0xab, 0xe7, 0x9c, 0xac, 0xe7, 0x9c, 0xae, 0xe7, 0x9c,
  0xb0, 0xe7, 0x9c, 0xb1, 0xe7, 0x9c, 0xb2, 0xe7, 0x9c, 0xb3, 0xe7, 0x9c,
  0xb4, 0xe7, 0x9c, 0xb9, 0xe7, 0x9c, 0xbb, 0xe7, 0x9c, 0xbd, 0xe7, 0x9c,
  0xbe, 0xe7, 0x9c, 0xbf, 0xe7, 0x9d, 0x82, 0xe7, 0x9d, 0x84, 0xe7, 0x9d,
  0x85, 0xe7, 0x9d, 0x86, 0xe7, 0x9d, 0x88, 0xe7, 0x9d, 0x89, 0xe7, 0x9d,
  0x8a, 0xe7, 0x9d, 0x8b, 0xe7, 0x9d, 0x8c, 0xe7, 0x9d, 0x8d, 0xe7, 0x9d,
  0x8e, 0xe7, 0x9d, 0x8f, 0xe7, 0x9d, 0x92, 0xe7, 0x9d, 0x93, 0xe7, 0x9d,
  0x94, 0xe7, 0x9d, 0x95, 0xe7, 0x9d, 0x96, 0xe7, 0x9d, 0x97, 0xe7, 0x9d,
  0x98, 0xe7, 0x9d, 0x99, 0xe7, 0x9d, 0x9c, 0xe8, 0x96, 0x84, 0xe9, 0x9b,
  0xb9, 0xe4, 0xbf, 0x9d, 0xe5, 0xa0, 0xa1, 0xe9, 0xa5, 0xb1, 0xe5, 0xae,
  0x9d, 0xe6, 0x8a, 0xb1, 0xe6, 0x8a, 0xa5, 0xe6, 0x9a, 0xb4, 0xe8, 0xb1,
  0xb9, 0xe9, 0xb2, 0x8d, 0xe7, 0x88, 0x86, 0xe6, 0x9d, 0xaf, 0xe7, 0xa2,
  0x91, 0xe6, 0x82, 0xb2, 0xe5, 0x8d, 0x91, 0xe5, 0x8c, 0x97, 0xe8, 0xbe,
  0x88, 0xe8, 0x83, 0x8c, 0xe8, 0xb4, 0x9d, 0xe9, 0x92, 0xa1, 0xe5, 0x80,
  0x8d, 0xe7, 0x8b, 0x88, 0xe5, 0xa4, 0x87, 0xe6, 0x83, 0xab, 0xe7, 0x84,
  0x99, 0xe8, 0xa2, 0xab, 0xe5, 0xa5, 0x94, 0xe8, 0x8b, 0xaf, 0xe6, 0x9c,
  0xac, 0xe7, 0xac, 0xa8, 0xe5, 0xb4, 0xa9, 0xe7, 0xbb, 0xb7, 0xe7, 0x94,
  0xad, 0xe6, 0xb3, 0xb5, 0xe8, 0xb9, 0xa6, 0xe8, 0xbf, 0xb8, 0xe9, 0x80,
  0xbc, 0xe9, 0xbc, 0xbb, 0xe6, 0xaf, 0x94, 0xe9, 0x84, 0x99, 0xe7, 0xac,
  0x94, 0xe5, 0xbd, 0xbc, 0xe7, 0xa2, 0xa7, 0xe8, 0x93, 0x96, 0xe8, 0x94,
  0xbd, 0xe6, 0xaf, 0x95, 0xe6, 0xaf, 0x99, 0xe6, 0xaf, 0x96, 0xe5, 0xb8,
  0x81, 0xe5, 0xba, 0x87, 0xe7, 0x97, 0xb9, 0xe9, 0x97, 0xad, 0xe6, 0x95,
  0x9d, 0xe5, 0xbc, 0x8a, 0xe5, 0xbf, 0x85, 0xe8, 0xbe, 0x9f, 0xe5, 0xa3,
  0x81, 0xe8, 0x87, 0x82, 0xe9, 0x81, 0xbf, 0xe9, 0x99, 0x9b, 0xe9, 0x9e,
  0xad, 0xe8, 0xbe, 0xb9, 0xe7, 0xbc, 0x96, 0xe8, 0xb4, 0xac, 0xe6, 0x89,
  0x81, 0xe4, 0xbe, 0xbf, 0xe5, 0x8f, 0x98, 0xe5, 0x8d, 0x9e, 0xe8, 0xbe,
  0xa8, 0xe8, 0xbe, 0xa9, 0xe8, 0xbe, 0xab, 0xe9, 0x81, 0x8d, 0xe6, 0xa0,
  0x87, 0xe5, 0xbd, 0xaa, 0xe8, 0x86, 0x98, 0xe8, 0xa1, 0xa8, 0xe9, 0xb3,
  0x96, 0xe6, 0x86, 0x8b, 0xe5, 0x88, 0xab, 0xe7, 0x98, 0xaa, 0xe5, 0xbd,
  0xac, 0xe6, 0x96, 0x8c, 0xe6, 0xbf, 0x92, 0xe6, 0xbb, 0xa8, 0xe5, 0xae,
  0xbe, 0xe6, 0x91, 0x88, 0xe5, 0x85, 0xb5, 0xe5, 0x86, 0xb0, 0xe6, 0x9f,
  0x84, 0xe4, 0xb8, 0x99, 0xe7, 0xa7, 0x89, 0xe9, 0xa5, 0xbc, 0xe7, 0x82,
  0xb3
};

unsigned char gbk_row_0xb2_converted_to_utf8[] = {
  0xe7, 0x9d, 0x9d, 0xe7, 0x9d, 0x9e, 0xe7, 0x9d, 0x9f, 0xe7,
  0x9d, 0xa0, 0xe7, 0x9d, 0xa4, 0xe7, 0x9d, 0xa7, 0xe7, 0x9d, 0xa9, 0xe7,
  0x9d, 0xaa, 0xe7, 0x9d, 0xad, 0xe7, 0x9d, 0xae, 0xe7, 0x9d, 0xaf, 0xe7,
  0x9d, 0xb0, 0xe7, 0x9d, 0xb1, 0xe7, 0x9d, 0xb2, 0xe7, 0x9d, 0xb3, 0xe7,
  0x9d, 0xb4, 0xe7, 0x9d, 0xb5, 0xe7, 0x9d, 0xb6, 0xe7, 0x9d, 0xb7, 0xe7,
  0x9d, 0xb8, 0xe7, 0x9d, 0xba, 0xe7, 0x9d, 0xbb, 0xe7, 0x9d, 0xbc, 0xe7,
  0x9e, 0x81, 0xe7, 0x9e, 0x82, 0xe7, 0x9e, 0x83, 0xe7, 0x9e, 0x86, 0xe7,
  0x9e, 0x87, 0xe7, 0x9e, 0x88, 0xe7, 0x9e, 0x89, 0xe7, 0x9e, 0x8a, 0xe7,
  0x9e, 0x8b, 0xe7, 0x9e, 0x8f, 0xe7, 0x9e, 0x90, 0xe7, 0x9e, 0x93, 0xe7,
  0x9e, 0x94, 0xe7, 0x9e, 0x95, 0xe7, 0x9e, 0x96, 0xe7, 0x9e, 0x97, 0xe7,
  0x9e, 0x98, 0xe7, 0x9e, 0x99, 0xe7, 0x9e, 0x9a, 0xe7, 0x9e, 0x9b, 0xe7,
  0x9e, 0x9c, 0xe7, 0x9e, 0x9d, 0xe7, 0x9e, 0x9e, 0xe7, 0x9e, 0xa1, 0xe7,
  0x9e, 0xa3, 0xe7, 0x9e, 0xa4, 0xe7, 0x9e, 0xa6, 0xe7, 0x9e, 0xa8, 0xe7,
  0x9e, 0xab, 0xe7, 0x9e, 0xad, 0xe7, 0x9e, 0xae, 0xe7, 0x9e, 0xaf, 0xe7,
  0x9e, 0xb1, 0xe7, 0x9e, 0xb2, 0xe7, 0x9e, 0xb4, 0xe7, 0x9e, 0xb6, 0xe7,
  0x9e, 0xb7, 0xe7, 0x9e, 0xb8, 0xe7, 0x9e, 0xb9, 0xe7, 0x9e, 0xba, 0xe7,
  0x9e, 0xbc, 0xe7, 0x9e, 0xbe, 0xe7, 0x9f, 0x80, 0xe7, 0x9f, 0x81, 0xe7,
  0x9f, 0x82, 0xe7, 0x9f, 0x83, 0xe7, 0x9f, 0x84, 0xe7, 0x9f, 0x85, 0xe7,
  0x9f, 0x86, 0xe7, 0x9f, 0x87, 0xe7, 0x9f, 0x88, 0xe7, 0x9f, 0x89, 0xe7,
  0x9f, 0x8a, 0xe7, 0x9f, 0x8b, 0xe7, 0x9f, 0x8c, 0xe7, 0x9f, 0x8e, 0xe7,
  0x9f, 0x8f, 0xe7, 0x9f, 0x90, 0xe7, 0x9f, 0x91, 0xe7, 0x9f, 0x92, 0xe7,
  0x9f, 0x93, 0xe7, 0x9f, 0x94, 0xe7, 0x9f, 0x95, 0xe7, 0x9f, 0x96, 0xe7,
  0x9f, 0x98, 0xe7, 0x9f, 0x99, 0xe7, 0x9f, 0x9a, 0xe7, 0x9f, 0x9d, 0xe7,
  0x9f, 0x9e, 0xe7, 0x9f, 0x9f, 0xe7, 0x9f, 0xa0, 0xe7, 0x9f, 0xa1, 0xe7,
  0x9f, 0xa4, 0xe7, 0x97, 0x85, 0xe5, 0xb9, 0xb6, 0xe7, 0x8e, 0xbb, 0xe8,
  0x8f, 0xa0, 0xe6, 0x92, 0xad, 0xe6, 0x8b, 0xa8, 0xe9, 0x92, 0xb5, 0xe6,
  0xb3, 0xa2, 0xe5, 0x8d, 0x9a, 0xe5, 0x8b, 0x83, 0xe6, 0x90, 0x8f, 0xe9,
  0x93, 0x82, 0xe7, 0xae, 0x94, 0xe4, 0xbc, 0xaf, 0xe5, 0xb8, 0x9b, 0xe8,
  0x88, 0xb6, 0xe8, 0x84, 0x96, 0xe8, 0x86, 0x8a, 0xe6, 0xb8, 0xa4, 0xe6,
  0xb3, 0x8a, 0xe9, 0xa9, 0xb3, 0xe6, 0x8d, 0x95, 0xe5, 0x8d, 0x9c, 0xe5,
  0x93, 0xba, 0xe8, 0xa1, 0xa5, 0xe5, 0x9f, 0xa0, 0xe4, 0xb8, 0x8d, 0xe5,
  0xb8, 0x83, 0xe6, 0xad, 0xa5, 0xe7, 0xb0, 0xbf, 0xe9, 0x83, 0xa8, 0xe6,
  0x80, 0x96, 0xe6, 0x93, 0xa6, 0xe7, 0x8c, 0x9c, 0xe8, 0xa3, 0x81, 0xe6,
  0x9d, 0x90, 0xe6, 0x89, 0x8d, 0xe8, 0xb4, 0xa2, 0xe7, 0x9d, 0xac, 0xe8,
  0xb8, 0xa9, 0xe9, 0x87, 0x87, 0xe5, 0xbd, 0xa9, 0xe8, 0x8f, 0x9c, 0xe8,
  0x94, 0xa1, 0xe9, 0xa4, 0x90, 0xe5, 0x8f, 0x82, 0xe8, 0x9a, 0x95, 0xe6,
  0xae, 0x8b, 0xe6, 0x83, 0xad, 0xe6, 0x83, 0xa8, 0xe7, 0x81, 0xbf, 0xe8,
  0x8b, 0x8d, 0xe8, 0x88, 0xb1, 0xe4, 0xbb, 0x93, 0xe6, 0xb2, 0xa7, 0xe8,
  0x97, 0x8f, 0xe6, 0x93, 0x8d, 0xe7, 0xb3, 0x99, 0xe6, 0xa7, 0xbd, 0xe6,
  0x9b, 0xb9, 0xe8, 0x8d, 0x89, 0xe5, 0x8e, 0x95, 0xe7, 0xad, 0x96, 0xe4,
  0xbe, 0xa7, 0xe5, 0x86, 0x8c, 0xe6, 0xb5, 0x8b, 0xe5, 0xb1, 0x82, 0xe8,
  0xb9, 0xad, 0xe6, 0x8f, 0x92, 0xe5, 0x8f, 0x89, 0xe8, 0x8c, 0xac, 0xe8,
  0x8c, 0xb6, 0xe6, 0x9f, 0xa5, 0xe7, 0xa2, 0xb4, 0xe6, 0x90, 0xbd, 0xe5,
  0xaf, 0x9f, 0xe5, 0xb2, 0x94, 0xe5, 0xb7, 0xae, 0xe8, 0xaf, 0xa7, 0xe6,
  0x8b, 0x86, 0xe6, 0x9f, 0xb4, 0xe8, 0xb1, 0xba, 0xe6, 0x90, 0x80, 0xe6,
  0x8e, 0xba, 0xe8, 0x9d, 0x89, 0xe9, 0xa6, 0x8b, 0xe8, 0xb0, 0x97, 0xe7,
  0xbc, 0xa0, 0xe9, 0x93, 0xb2, 0xe4, 0xba, 0xa7, 0xe9, 0x98, 0x90, 0xe9,
  0xa2, 0xa4, 0xe6, 0x98, 0x8c, 0xe7, 0x8c, 0x96
};

unsigned char gbk_row_0xb3_converted_to_utf8[] = {
  0xe7, 0x9f, 0xa6,
  0xe7, 0x9f, 0xa8, 0xe7, 0x9f, 0xaa, 0xe7, 0x9f, 0xaf, 0xe7, 0x9f, 0xb0,
  0xe7, 0x9f, 0xb1, 0xe7, 0x9f, 0xb2, 0xe7, 0x9f, 0xb4, 0xe7, 0x9f, 0xb5,
  0xe7, 0x9f, 0xb7, 0xe7, 0x9f, 0xb9, 0xe7, 0x9f, 0xba, 0xe7, 0x9f, 0xbb,
  0xe7, 0x9f, 0xbc, 0xe7, 0xa0, 0x83, 0xe7, 0xa0, 0x84, 0xe7, 0xa0, 0x85,
  0xe7, 0xa0, 0x86, 0xe7, 0xa0, 0x87, 0xe7, 0xa0, 0x88, 0xe7, 0xa0, 0x8a,
  0xe7, 0xa0, 0x8b, 0xe7, 0xa0, 0x8e, 0xe7, 0xa0, 0x8f, 0xe7, 0xa0, 0x90,
  0xe7, 0xa0, 0x93, 0xe7, 0xa0, 0x95, 0xe7, 0xa0, 0x99, 0xe7, 0xa0, 0x9b,
  0xe7, 0xa0, 0x9e, 0xe7, 0xa0, 0xa0, 0xe7, 0xa0, 0xa1, 0xe7, 0xa0, 0xa2,
  0xe7, 0xa0, 0xa4, 0xe7, 0xa0, 0xa8, 0xe7, 0xa0, 0xaa, 0xe7, 0xa0, 0xab,
  0xe7, 0xa0, 0xae, 0xe7, 0xa0, 0xaf, 0xe7, 0xa0, 0xb1, 0xe7, 0xa0, 0xb2,
  0xe7, 0xa0, 0xb3, 0xe7, 0xa0, 0xb5, 0xe7, 0xa0, 0xb6, 0xe7, 0xa0, 0xbd,
  0xe7, 0xa0, 0xbf, 0xe7, 0xa1, 0x81, 0xe7, 0xa1, 0x82, 0xe7, 0xa1, 0x83,
  0xe7, 0xa1, 0x84, 0xe7, 0xa1, 0x86, 0xe7, 0xa1, 0x88, 0xe7, 0xa1, 0x89,
  0xe7, 0xa1, 0x8a, 0xe7, 0xa1, 0x8b, 0xe7, 0xa1, 0x8d, 0xe7, 0xa1, 0x8f,
  0xe7, 0xa1, 0x91, 0xe7, 0xa1, 0x93, 0xe7, 0xa1, 0x94, 0xe7, 0xa1, 0x98,
  0xe7, 0xa1, 0x99, 0xe7, 0xa1, 0x9a, 0xe7, 0xa1, 0x9b, 0xe7, 0xa1, 0x9c,
  0xe7, 0xa1, 0x9e, 0xe7, 0xa1, 0x9f, 0xe7, 0xa1, 0xa0, 0xe7, 0xa1, 0xa1,
  0xe7, 0xa1, 0xa2, 0xe7, 0xa1, 0xa3, 0xe7, 0xa1, 0xa4, 0xe7, 0xa1, 0xa5,
  0xe7, 0xa1, 0xa6, 0xe7, 0xa1, 0xa7, 0xe7, 0xa1, 0xa8, 0xe7, 0xa1, 0xa9,
  0xe7, 0xa1, 0xaf, 0xe7, 0xa1, 0xb0, 0xe7, 0xa1, 0xb1, 0xe7, 0xa1, 0xb2,
  0xe7, 0xa1, 0xb3, 0xe7, 0xa1, 0xb4, 0xe7, 0xa1, 0xb5, 0xe7, 0xa1, 0xb6,
  0xe7, 0xa1, 0xb8, 0xe7, 0xa1, 0xb9, 0xe7, 0xa1, 0xba, 0xe7, 0xa1, 0xbb,
  0xe7, 0xa1, 0xbd, 0xe7, 0xa1, 0xbe, 0xe7, 0xa1, 0xbf, 0xe7, 0xa2, 0x80,
  0xe7, 0xa2, 0x81, 0xe7, 0xa2, 0x82, 0xe7, 0xa2, 0x83, 0xe5, 0x9c, 0xba,
  0xe5, 0xb0, 0x9d, 0xe5, 0xb8, 0xb8, 0xe9, 0x95, 0xbf, 0xe5, 0x81, 0xbf,
  0xe8, 0x82, 0xa0, 0xe5, 0x8e, 0x82, 0xe6, 0x95, 0x9e, 0xe7, 0x95, 0x85,
  0xe5, 0x94, 0xb1, 0xe5, 0x80, 0xa1, 0xe8, 0xb6, 0x85, 0xe6, 0x8a, 0x84,
  0xe9, 0x92, 0x9e, 0xe6, 0x9c, 0x9d, 0xe5, 0x98, 0xb2, 0xe6, 0xbd, 0xae,
  0xe5, 0xb7, 0xa2, 0xe5, 0x90, 0xb5, 0xe7, 0x82, 0x92, 0xe8, 0xbd, 0xa6,
  0xe6, 0x89, 0xaf, 0xe6, 0x92, 0xa4, 0xe6, 0x8e, 0xa3, 0xe5, 0xbd, 0xbb,
  0xe6, 0xbe, 0x88, 0xe9, 0x83, 0xb4, 0xe8, 0x87, 0xa3, 0xe8, 0xbe, 0xb0,
  0xe5, 0xb0, 0x98, 0xe6, 0x99, 0xa8, 0xe5, 0xbf, 0xb1, 0xe6, 0xb2, 0x89,
  0xe9, 0x99, 0x88, 0xe8, 0xb6, 0x81, 0xe8, 0xa1, 0xac, 0xe6, 0x92, 0x91,
  0xe7, 0xa7, 0xb0, 0xe5, 0x9f, 0x8e, 0xe6, 0xa9, 0x99, 0xe6, 0x88, 0x90,
  0xe5, 0x91, 0x88, 0xe4, 0xb9, 0x98, 0xe7, 0xa8, 0x8b, 0xe6, 0x83, 0xa9,
  0xe6, 0xbe, 0x84, 0xe8, 0xaf, 0x9a, 0xe6, 0x89, 0xbf, 0xe9, 0x80, 0x9e,
  0xe9, 0xaa, 0x8b, 0xe7, 0xa7, 0xa4, 0xe5, 0x90, 0x83, 0xe7, 0x97, 0xb4,
  0xe6, 0x8c, 0x81, 0xe5, 0x8c, 0x99, 0xe6, 0xb1, 0xa0, 0xe8, 0xbf, 0x9f,
  0xe5, 0xbc, 0x9b, 0xe9, 0xa9, 0xb0, 0xe8, 0x80, 0xbb, 0xe9, 0xbd, 0xbf,
  0xe4, 0xbe, 0x88, 0xe5, 0xb0, 0xba, 0xe8, 0xb5, 0xa4, 0xe7, 0xbf, 0x85,
  0xe6, 0x96, 0xa5, 0xe7, 0x82, 0xbd, 0xe5, 0x85, 0x85, 0xe5, 0x86, 0xb2,
  0xe8, 0x99, 0xab, 0xe5, 0xb4, 0x87, 0xe5, 0xae, 0xa0, 0xe6, 0x8a, 0xbd,
  0xe9, 0x85, 0xac, 0xe7, 0x95, 0xb4, 0xe8, 0xb8, 0x8c, 0xe7, 0xa8, 0xa0,
  0xe6, 0x84, 0x81, 0xe7, 0xad, 0xb9, 0xe4, 0xbb, 0x87, 0xe7, 0xbb, 0xb8,
  0xe7, 0x9e, 0x85, 0xe4, 0xb8, 0x91, 0xe8, 0x87, 0xad, 0xe5, 0x88, 0x9d,
  0xe5, 0x87, 0xba, 0xe6, 0xa9, 0xb1, 0xe5, 0x8e, 0xa8, 0xe8, 0xba, 0x87,
  0xe9, 0x94, 0x84, 0xe9, 0x9b, 0x8f, 0xe6, 0xbb, 0x81, 0xe9, 0x99, 0xa4,
  0xe6, 0xa5, 0x9a
};

unsigned char gbk_row_0xb4_converted_to_utf8[] = {
  0xe7, 0xa2, 0x84, 0xe7, 0xa2, 0x85, 0xe7, 0xa2,
  0x86, 0xe7, 0xa2, 0x88, 0xe7, 0xa2, 0x8a, 0xe7, 0xa2, 0x8b, 0xe7, 0xa2,
  0x8f, 0xe7, 0xa2, 0x90, 0xe7, 0xa2, 0x92, 0xe7, 0xa2, 0x94, 0xe7, 0xa2,
  0x95, 0xe7, 0xa2, 0x96, 0xe7, 0xa2, 0x99, 0xe7, 0xa2, 0x9d, 0xe7, 0xa2,
  0x9e, 0xe7, 0xa2, 0xa0, 0xe7, 0xa2, 0xa2, 0xe7, 0xa2, 0xa4, 0xe7, 0xa2,
  0xa6, 0xe7, 0xa2, 0xa8, 0xe7, 0xa2, 0xa9, 0xe7, 0xa2, 0xaa, 0xe7, 0xa2,
  0xab, 0xe7, 0xa2, 0xac, 0xe7, 0xa2, 0xad, 0xe7, 0xa2, 0xae, 0xe7, 0xa2,
  0xaf, 0xe7, 0xa2, 0xb5, 0xe7, 0xa2, 0xb6, 0xe7, 0xa2, 0xb7, 0xe7, 0xa2,
  0xb8, 0xe7, 0xa2, 0xba, 0xe7, 0xa2, 0xbb, 0xe7, 0xa2, 0xbc, 0xe7, 0xa2,
  0xbd, 0xe7, 0xa2, 0xbf, 0xe7, 0xa3, 0x80, 0xe7, 0xa3, 0x82, 0xe7, 0xa3,
  0x83, 0xe7, 0xa3, 0x84, 0xe7, 0xa3, 0x86, 0xe7, 0xa3, 0x87, 0xe7, 0xa3,
  0x88, 0xe7, 0xa3, 0x8c, 0xe7, 0xa3, 0x8d, 0xe7, 0xa3, 0x8e, 0xe7, 0xa3,
  0x8f, 0xe7, 0xa3, 0x91, 0xe7, 0xa3, 0x92, 0xe7, 0xa3, 0x93, 0xe7, 0xa3,
  0x96, 0xe7, 0xa3, 0x97, 0xe7, 0xa3, 0x98, 0xe7, 0xa3, 0x9a, 0xe7, 0xa3,
  0x9b, 0xe7, 0xa3, 0x9c, 0xe7, 0xa3, 0x9d, 0xe7, 0xa3, 0x9e, 0xe7, 0xa3,
  0x9f, 0xe7, 0xa3, 0xa0, 0xe7, 0xa3, 0xa1, 0xe7, 0xa3, 0xa2, 0xe7, 0xa3,
  0xa3, 0xe7, 0xa3, 0xa4, 0xe7, 0xa3, 0xa5, 0xe7, 0xa3, 0xa6, 0xe7, 0xa3,
  0xa7, 0xe7, 0xa3, 0xa9, 0xe7, 0xa3, 0xaa, 0xe7, 0xa3, 0xab, 0xe7, 0xa3,
  0xad, 0xe7, 0xa3, 0xae, 0xe7, 0xa3, 0xaf, 0xe7, 0xa3, 0xb0, 0xe7, 0xa3,
  0xb1, 0xe7, 0xa3, 0xb3, 0xe7, 0xa3, 0xb5, 0xe7, 0xa3, 0xb6, 0xe7, 0xa3,
  0xb8, 0xe7, 0xa3, 0xb9, 0xe7, 0xa3, 0xbb, 0xe7, 0xa3, 0xbc, 0xe7, 0xa3,
  0xbd, 0xe7, 0xa3, 0xbe, 0xe7, 0xa3, 0xbf, 0xe7, 0xa4, 0x80, 0xe7, 0xa4,
  0x82, 0xe7, 0xa4, 0x83, 0xe7, 0xa4, 0x84, 0xe7, 0xa4, 0x86, 0xe7, 0xa4,
  0x87, 0xe7, 0xa4, 0x88, 0xe7, 0xa4, 0x89, 0xe7, 0xa4, 0x8a, 0xe7, 0xa4,
  0x8b, 0xe7, 0xa4, 0x8c, 0xe7, 0xa1, 0x80, 0xe5, 0x82, 0xa8, 0xe7, 0x9f,
  0x97, 0xe6, 0x90, 0x90, 0xe8, 0xa7, 0xa6, 0xe5, 0xa4, 0x84, 0xe6, 0x8f,
  0xa3, 0xe5, 0xb7, 0x9d, 0xe7, 0xa9, 0xbf, 0xe6, 0xa4, 0xbd, 0xe4, 0xbc,
  0xa0, 0xe8, 0x88, 0xb9, 0xe5, 0x96, 0x98, 0xe4, 0xb8, 0xb2, 0xe7, 0x96,
  0xae, 0xe7, 0xaa, 0x97, 0xe5, 0xb9, 0xa2, 0xe5, 0xba, 0x8a, 0xe9, 0x97,
  0xaf, 0xe5, 0x88, 0x9b, 0xe5, 0x90, 0xb9, 0xe7, 0x82, 0x8a, 0xe6, 0x8d,
  0xb6, 0xe9, 0x94, 0xa4, 0xe5, 0x9e, 0x82, 0xe6, 0x98, 0xa5, 0xe6, 0xa4,
  0xbf, 0xe9, 0x86, 0x87, 0xe5, 0x94, 0x87, 0xe6, 0xb7, 0xb3, 0xe7, 0xba,
  0xaf, 0xe8, 0xa0, 0xa2, 0xe6, 0x88, 0xb3, 0xe7, 0xbb, 0xb0, 0xe7, 0x96,
  0xb5, 0xe8, 0x8c, 0xa8, 0xe7, 0xa3, 0x81, 0xe9, 0x9b, 0x8c, 0xe8, 0xbe,
  0x9e, 0xe6, 0x85, 0x88, 0xe7, 0x93, 0xb7, 0xe8, 0xaf, 0x8d, 0xe6, 0xad,
  0xa4, 0xe5, 0x88, 0xba, 0xe8, 0xb5, 0x90, 0xe6, 0xac, 0xa1, 0xe8, 0x81,
  0xaa, 0xe8, 0x91, 0xb1, 0xe5, 0x9b, 0xb1, 0xe5, 0x8c, 0x86, 0xe4, 0xbb,
  0x8e, 0xe4, 0xb8, 0x9b, 0xe5, 0x87, 0x91, 0xe7, 0xb2, 0x97, 0xe9, 0x86,
  0x8b, 0xe7, 0xb0, 0x87, 0xe4, 0xbf, 0x83, 0xe8, 0xb9, 0xbf, 0xe7, 0xaf,
  0xa1, 0xe7, 0xaa, 0x9c, 0xe6, 0x91, 0xa7, 0xe5, 0xb4, 0x94, 0xe5, 0x82,
  0xac, 0xe8, 0x84, 0x86, 0xe7, 0x98, 0x81, 0xe7, 0xb2, 0xb9, 0xe6, 0xb7,
  0xac, 0xe7, 0xbf, 0xa0, 0xe6, 0x9d, 0x91, 0xe5, 0xad, 0x98, 0xe5, 0xaf,
  0xb8, 0xe7, 0xa3, 0x8b, 0xe6, 0x92, 0xae, 0xe6, 0x90, 0x93, 0xe6, 0x8e,
  0xaa, 0xe6, 0x8c, 0xab, 0xe9, 0x94, 0x99, 0xe6, 0x90, 0xad, 0xe8, 0xbe,
  0xbe, 0xe7, 0xad, 0x94, 0xe7, 0x98, 0xa9, 0xe6, 0x89, 0x93, 0xe5, 0xa4,
  0xa7, 0xe5, 0x91, 0x86, 0xe6, 0xad, 0xb9, 0xe5, 0x82, 0xa3, 0xe6, 0x88,
  0xb4, 0xe5, 0xb8, 0xa6, 0xe6, 0xae, 0x86, 0xe4, 0xbb, 0xa3, 0xe8, 0xb4,
  0xb7, 0xe8, 0xa2, 0x8b, 0xe5, 0xbe, 0x85, 0xe9, 0x80, 0xae
};

unsigned char gbk_row_0xb5_converted_to_utf8[] = {
  0xe7,
  0xa4, 0x8d, 0xe7, 0xa4, 0x8e, 0xe7, 0xa4, 0x8f, 0xe7, 0xa4, 0x90, 0xe7,
  0xa4, 0x91, 0xe7, 0xa4, 0x92, 0xe7, 0xa4, 0x94, 0xe7, 0xa4, 0x95, 0xe7,
  0xa4, 0x96, 0xe7, 0xa4, 0x97, 0xe7, 0xa4, 0x98, 0xe7, 0xa4, 0x99, 0xe7,
  0xa4, 0x9a, 0xe7, 0xa4, 0x9b, 0xe7, 0xa4, 0x9c, 0xe7, 0xa4, 0x9d, 0xe7,
  0xa4, 0x9f, 0xe7, 0xa4, 0xa0, 0xe7, 0xa4, 0xa1, 0xe7, 0xa4, 0xa2, 0xe7,
  0xa4, 0xa3, 0xe7, 0xa4, 0xa5, 0xe7, 0xa4, 0xa6, 0xe7, 0xa4, 0xa7, 0xe7,
  0xa4, 0xa8, 0xe7, 0xa4, 0xa9, 0xe7, 0xa4, 0xaa, 0xe7, 0xa4, 0xab, 0xe7,
  0xa4, 0xac, 0xe7, 0xa4, 0xad, 0xe7, 0xa4, 0xae, 0xe7, 0xa4, 0xaf, 0xe7,
  0xa4, 0xb0, 0xe7, 0xa4, 0xb1, 0xe7, 0xa4, 0xb2, 0xe7, 0xa4, 0xb3, 0xe7,
  0xa4, 0xb5, 0xe7, 0xa4, 0xb6, 0xe7, 0xa4, 0xb7, 0xe7, 0xa4, 0xb8, 0xe7,
  0xa4, 0xb9, 0xe7, 0xa4, 0xbd, 0xe7, 0xa4, 0xbf, 0xe7, 0xa5, 0x82, 0xe7,
  0xa5, 0x83, 0xe7, 0xa5, 0x84, 0xe7, 0xa5, 0x85, 0xe7, 0xa5, 0x87, 0xe7,
  0xa5, 0x8a, 0xe7, 0xa5, 0x8b, 0xe7, 0xa5, 0x8c, 0xe7, 0xa5, 0x8d, 0xe7,
  0xa5, 0x8e, 0xe7, 0xa5, 0x8f, 0xe7, 0xa5, 0x90, 0xe7, 0xa5, 0x91, 0xe7,
  0xa5, 0x92, 0xe7, 0xa5, 0x94, 0xe7, 0xa5, 0x95, 0xe7, 0xa5, 0x98, 0xe7,
  0xa5, 0x99, 0xe7, 0xa5, 0xa1, 0xe7, 0xa5, 0xa3, 0xe7, 0xa5, 0xa4, 0xe7,
  0xa5, 0xa6, 0xe7, 0xa5, 0xa9, 0xe7, 0xa5, 0xaa, 0xe7, 0xa5, 0xab, 0xe7,
  0xa5, 0xac, 0xe7, 0xa5, 0xae, 0xe7, 0xa5, 0xb0, 0xe7, 0xa5, 0xb1, 0xe7,
  0xa5, 0xb2, 0xe7, 0xa5, 0xb3, 0xe7, 0xa5, 0xb4, 0xe7, 0xa5, 0xb5, 0xe7,
  0xa5, 0xb6, 0xe7, 0xa5, 0xb9, 0xe7, 0xa5, 0xbb, 0xe7, 0xa5, 0xbc, 0xe7,
  0xa5, 0xbd, 0xe7, 0xa5, 0xbe, 0xe7, 0xa5, 0xbf, 0xe7, 0xa6, 0x82, 0xe7,
  0xa6, 0x83, 0xe7, 0xa6, 0x86, 0xe7, 0xa6, 0x87, 0xe7, 0xa6, 0x88, 0xe7,
  0xa6, 0x89, 0xe7, 0xa6, 0x8b, 0xe7, 0xa6, 0x8c, 0xe7, 0xa6, 0x8d, 0xe7,
  0xa6, 0x8e, 0xe7, 0xa6, 0x90, 0xe7, 0xa6, 0x91, 0xe7, 0xa6, 0x92, 0xe6,
  0x80, 0xa0, 0xe8, 0x80, 0xbd, 0xe6, 0x8b, 0x85, 0xe4, 0xb8, 0xb9, 0xe5,
  0x8d, 0x95, 0xe9, 0x83, 0xb8, 0xe6, 0x8e, 0xb8, 0xe8, 0x83, 0x86, 0xe6,
  0x97, 0xa6, 0xe6, 0xb0, 0xae, 0xe4, 0xbd, 0x86, 0xe6, 0x83, 0xae, 0xe6,
  0xb7, 0xa1, 0xe8, 0xaf, 0x9e, 0xe5, 0xbc, 0xb9, 0xe8, 0x9b, 0x8b, 0xe5,
  0xbd, 0x93, 0xe6, 0x8c, 0xa1, 0xe5, 0x85, 0x9a, 0xe8, 0x8d, 0xa1, 0xe6,
  0xa1, 0xa3, 0xe5, 0x88, 0x80, 0xe6, 0x8d, 0xa3, 0xe8, 0xb9, 0x88, 0xe5,
  0x80, 0x92, 0xe5, 0xb2, 0x9b, 0xe7, 0xa5, 0xb7, 0xe5, 0xaf, 0xbc, 0xe5,
  0x88, 0xb0, 0xe7, 0xa8, 0xbb, 0xe6, 0x82, 0xbc, 0xe9, 0x81, 0x93, 0xe7,
  0x9b, 0x97, 0xe5, 0xbe, 0xb7, 0xe5, 0xbe, 0x97, 0xe7, 0x9a, 0x84, 0xe8,
  0xb9, 0xac, 0xe7, 0x81, 0xaf, 0xe7, 0x99, 0xbb, 0xe7, 0xad, 0x89, 0xe7,
  0x9e, 0xaa, 0xe5, 0x87, 0xb3, 0xe9, 0x82, 0x93, 0xe5, 0xa0, 0xa4, 0xe4,
  0xbd, 0x8e, 0xe6, 0xbb, 0xb4, 0xe8, 0xbf, 0xaa, 0xe6, 0x95, 0x8c, 0xe7,
  0xac, 0x9b, 0xe7, 0x8b, 0x84, 0xe6, 0xb6, 0xa4, 0xe7, 0xbf, 0x9f, 0xe5,
  0xab, 0xa1, 0xe6, 0x8a, 0xb5, 0xe5, 0xba, 0x95, 0xe5, 0x9c, 0xb0, 0xe8,
  0x92, 0x82, 0xe7, 0xac, 0xac, 0xe5, 0xb8, 0x9d, 0xe5, 0xbc, 0x9f, 0xe9,
  0x80, 0x92, 0xe7, 0xbc, 0x94, 0xe9, 0xa2, 0xa0, 0xe6, 0x8e, 0x82, 0xe6,
  0xbb, 0x87, 0xe7, 0xa2, 0x98, 0xe7, 0x82, 0xb9, 0xe5, 0x85, 0xb8, 0xe9,
  0x9d, 0x9b, 0xe5, 0x9e, 0xab, 0xe7, 0x94, 0xb5, 0xe4, 0xbd, 0x83, 0xe7,
  0x94, 0xb8, 0xe5, 0xba, 0x97, 0xe6, 0x83, 0xa6, 0xe5, 0xa5, 0xa0, 0xe6,
  0xb7, 0x80, 0xe6, 0xae, 0xbf, 0xe7, 0xa2, 0x89, 0xe5, 0x8f, 0xbc, 0xe9,
  0x9b, 0x95, 0xe5, 0x87, 0x8b, 0xe5, 0x88, 0x81, 0xe6, 0x8e, 0x89, 0xe5,
  0x90, 0x8a, 0xe9, 0x92, 0x93, 0xe8, 0xb0, 0x83, 0xe8, 0xb7, 0x8c, 0xe7,
  0x88, 0xb9, 0xe7, 0xa2, 0x9f, 0xe8, 0x9d, 0xb6, 0xe8, 0xbf, 0xad, 0xe8,
  0xb0, 0x8d, 0xe5, 0x8f, 0xa0
};

unsigned char gbk_row_0xb6_converted_to_utf8[] = {
  0xe7, 0xa6, 0x93, 0xe7, 0xa6, 0x94,
  0xe7, 0xa6, 0x95, 0xe7, 0xa6, 0x96, 0xe7, 0xa6, 0x97, 0xe7, 0xa6, 0x98,
  0xe7, 0xa6, 0x99, 0xe7, 0xa6, 0x9b, 0xe7, 0xa6, 0x9c, 0xe7, 0xa6, 0x9d,
  0xe7, 0xa6, 0x9e, 0xe7, 0xa6, 0x9f, 0xe7, 0xa6, 0xa0, 0xe7, 0xa6, 0xa1,
  0xe7, 0xa6, 0xa2, 0xe7, 0xa6, 0xa3, 0xe7, 0xa6, 0xa4, 0xe7, 0xa6, 0xa5,
  0xe7, 0xa6, 0xa6, 0xe7, 0xa6, 0xa8, 0xe7, 0xa6, 0xa9, 0xe7, 0xa6, 0xaa,
  0xe7, 0xa6, 0xab, 0xe7, 0xa6, 0xac, 0xe7, 0xa6, 0xad, 0xe7, 0xa6, 0xae,
  0xe7, 0xa6, 0xaf, 0xe7, 0xa6, 0xb0, 0xe7, 0xa6, 0xb1, 0xe7, 0xa6, 0xb2,
  0xe7, 0xa6, 0xb4, 0xe7, 0xa6, 0xb5, 0xe7, 0xa6, 0xb6, 0xe7, 0xa6, 0xb7,
  0xe7, 0xa6, 0xb8, 0xe7, 0xa6, 0xbc, 0xe7, 0xa6, 0xbf, 0xe7, 0xa7, 0x82,
  0xe7, 0xa7, 0x84, 0xe7, 0xa7, 0x85, 0xe7, 0xa7, 0x87, 0xe7, 0xa7, 0x88,
  0xe7, 0xa7, 0x8a, 0xe7, 0xa7, 0x8c, 0xe7, 0xa7, 0x8e, 0xe7, 0xa7, 0x8f,
  0xe7, 0xa7, 0x90, 0xe7, 0xa7, 0x93, 0xe7, 0xa7, 0x94, 0xe7, 0xa7, 0x96,
  0xe7, 0xa7, 0x97, 0xe7, 0xa7, 0x99, 0xe7, 0xa7, 0x9a, 0xe7, 0xa7, 0x9b,
  0xe7, 0xa7, 0x9c, 0xe7, 0xa7, 0x9d, 0xe7, 0xa7, 0x9e, 0xe7, 0xa7, 0xa0,
  0xe7, 0xa7, 0xa1, 0xe7, 0xa7, 0xa2, 0xe7, 0xa7, 0xa5, 0xe7, 0xa7, 0xa8,
  0xe7, 0xa7, 0xaa, 0xe7, 0xa7, 0xac, 0xe7, 0xa7, 0xae, 0xe7, 0xa7, 0xb1,
  0xe7, 0xa7, 0xb2, 0xe7, 0xa7, 0xb3, 0xe7, 0xa7, 0xb4, 0xe7, 0xa7, 0xb5,
  0xe7, 0xa7, 0xb6, 0xe7, 0xa7, 0xb7, 0xe7, 0xa7, 0xb9, 0xe7, 0xa7, 0xba,
  0xe7, 0xa7, 0xbc, 0xe7, 0xa7, 0xbe, 0xe7, 0xa7, 0xbf, 0xe7, 0xa8, 0x81,
  0xe7, 0xa8, 0x84, 0xe7, 0xa8, 0x85, 0xe7, 0xa8, 0x87, 0xe7, 0xa8, 0x88,
  0xe7, 0xa8, 0x89, 0xe7, 0xa8, 0x8a, 0xe7, 0xa8, 0x8c, 0xe7, 0xa8, 0x8f,
  0xe7, 0xa8, 0x90, 0xe7, 0xa8, 0x91, 0xe7, 0xa8, 0x92, 0xe7, 0xa8, 0x93,
  0xe7, 0xa8, 0x95, 0xe7, 0xa8, 0x96, 0xe7, 0xa8, 0x98, 0xe7, 0xa8, 0x99,
  0xe7, 0xa8, 0x9b, 0xe7, 0xa8, 0x9c, 0xe4, 0xb8, 0x81, 0xe7, 0x9b, 0xaf,
  0xe5, 0x8f, 0xae, 0xe9, 0x92, 0x89, 0xe9, 0xa1, 0xb6, 0xe9, 0xbc, 0x8e,
  0xe9, 0x94, 0xad, 0xe5, 0xae, 0x9a, 0xe8, 0xae, 0xa2, 0xe4, 0xb8, 0xa2,
  0xe4, 0xb8, 0x9c, 0xe5, 0x86, 0xac, 0xe8, 0x91, 0xa3, 0xe6, 0x87, 0x82,
  0xe5, 0x8a, 0xa8, 0xe6, 0xa0, 0x8b, 0xe4, 0xbe, 0x97, 0xe6, 0x81, 0xab,
  0xe5, 0x86, 0xbb, 0xe6, 0xb4, 0x9e, 0xe5, 0x85, 0x9c, 0xe6, 0x8a, 0x96,
  0xe6, 0x96, 0x97, 0xe9, 0x99, 0xa1, 0xe8, 0xb1, 0x86, 0xe9, 0x80, 0x97,
  0xe7, 0x97, 0x98, 0xe9, 0x83, 0xbd, 0xe7, 0x9d, 0xa3, 0xe6, 0xaf, 0x92,
  0xe7, 0x8a, 0x8a, 0xe7, 0x8b, 0xac, 0xe8, 0xaf, 0xbb, 0xe5, 0xa0, 0xb5,
  0xe7, 0x9d, 0xb9, 0xe8, 0xb5, 0x8c, 0xe6, 0x9d, 0x9c, 0xe9, 0x95, 0x80,
  0xe8, 0x82, 0x9a, 0xe5, 0xba, 0xa6, 0xe6, 0xb8, 0xa1, 0xe5, 0xa6, 0x92,
  0xe7, 0xab, 0xaf, 0xe7, 0x9f, 0xad, 0xe9, 0x94, 0xbb, 0xe6, 0xae, 0xb5,
  0xe6, 0x96, 0xad, 0xe7, 0xbc, 0x8e, 0xe5, 0xa0, 0x86, 0xe5, 0x85, 0x91,
  0xe9, 0x98, 0x9f, 0xe5, 0xaf, 0xb9, 0xe5, 0xa2, 0xa9, 0xe5, 0x90, 0xa8,
  0xe8, 0xb9, 0xb2, 0xe6, 0x95, 0xa6, 0xe9, 0xa1, 0xbf, 0xe5, 0x9b, 0xa4,
  0xe9, 0x92, 0x9d, 0xe7, 0x9b, 0xbe, 0xe9, 0x81, 0x81, 0xe6, 0x8e, 0x87,
  0xe5, 0x93, 0x86, 0xe5, 0xa4, 0x9a, 0xe5, 0xa4, 0xba, 0xe5, 0x9e, 0x9b,
  0xe8, 0xba, 0xb2, 0xe6, 0x9c, 0xb5, 0xe8, 0xb7, 0xba, 0xe8, 0x88, 0xb5,
  0xe5, 0x89, 0x81, 0xe6, 0x83, 0xb0, 0xe5, 0xa0, 0x95, 0xe8, 0x9b, 0xbe,
  0xe5, 0xb3, 0xa8, 0xe9, 0xb9, 0x85, 0xe4, 0xbf, 0x84, 0xe9, 0xa2, 0x9d,
  0xe8, 0xae, 0xb9, 0xe5, 0xa8, 0xa5, 0xe6, 0x81, 0xb6, 0xe5, 0x8e, 0x84,
  0xe6, 0x89, 0xbc, 0xe9, 0x81, 0x8f, 0xe9, 0x84, 0x82, 0xe9, 0xa5, 0xbf,
  0xe6, 0x81, 0xa9, 0xe8, 0x80, 0x8c, 0xe5, 0x84, 0xbf, 0xe8, 0x80, 0xb3,
  0xe5, 0xb0, 0x94, 0xe9, 0xa5, 0xb5, 0xe6, 0xb4, 0xb1, 0xe4, 0xba, 0x8c,
};
unsigned char gbk_row_0xb7_converted_to_utf8[] = {
        0xe7, 0xa8, 0x9d, 0xe7, 0xa8, 0x9f, 0xe7, 0xa8, 0xa1, 0xe7, 0xa8,
  0xa2, 0xe7, 0xa8, 0xa4, 0xe7, 0xa8, 0xa5, 0xe7, 0xa8, 0xa6, 0xe7, 0xa8,
  0xa7, 0xe7, 0xa8, 0xa8, 0xe7, 0xa8, 0xa9, 0xe7, 0xa8, 0xaa, 0xe7, 0xa8,
  0xab, 0xe7, 0xa8, 0xac, 0xe7, 0xa8, 0xad, 0xe7, 0xa8, 0xae, 0xe7, 0xa8,
  0xaf, 0xe7, 0xa8, 0xb0, 0xe7, 0xa8, 0xb1, 0xe7, 0xa8, 0xb2, 0xe7, 0xa8,
  0xb4, 0xe7, 0xa8, 0xb5, 0xe7, 0xa8, 0xb6, 0xe7, 0xa8, 0xb8, 0xe7, 0xa8,
  0xba, 0xe7, 0xa8, 0xbe, 0xe7, 0xa9, 0x80, 0xe7, 0xa9, 0x81, 0xe7, 0xa9,
  0x82, 0xe7, 0xa9, 0x83, 0xe7, 0xa9, 0x84, 0xe7, 0xa9, 0x85, 0xe7, 0xa9,
  0x87, 0xe7, 0xa9, 0x88, 0xe7, 0xa9, 0x89, 0xe7, 0xa9, 0x8a, 0xe7, 0xa9,
  0x8b, 0xe7, 0xa9, 0x8c, 0xe7, 0xa9, 0x8d, 0xe7, 0xa9, 0x8e, 0xe7, 0xa9,
  0x8f, 0xe7, 0xa9, 0x90, 0xe7, 0xa9, 0x92, 0xe7, 0xa9, 0x93, 0xe7, 0xa9,
  0x94, 0xe7, 0xa9, 0x95, 0xe7, 0xa9, 0x96, 0xe7, 0xa9, 0x98, 0xe7, 0xa9,
  0x99, 0xe7, 0xa9, 0x9a, 0xe7, 0xa9, 0x9b, 0xe7, 0xa9, 0x9c, 0xe7, 0xa9,
  0x9d, 0xe7, 0xa9, 0x9e, 0xe7, 0xa9, 0x9f, 0xe7, 0xa9, 0xa0, 0xe7, 0xa9,
  0xa1, 0xe7, 0xa9, 0xa2, 0xe7, 0xa9, 0xa3, 0xe7, 0xa9, 0xa4, 0xe7, 0xa9,
  0xa5, 0xe7, 0xa9, 0xa6, 0xe7, 0xa9, 0xa7, 0xe7, 0xa9, 0xa8, 0xe7, 0xa9,
  0xa9, 0xe7, 0xa9, 0xaa, 0xe7, 0xa9, 0xab, 0xe7, 0xa9, 0xac, 0xe7, 0xa9,
  0xad, 0xe7, 0xa9, 0xae, 0xe7, 0xa9, 0xaf, 0xe7, 0xa9, 0xb1, 0xe7, 0xa9,
  0xb2, 0xe7, 0xa9, 0xb3, 0xe7, 0xa9, 0xb5, 0xe7, 0xa9, 0xbb, 0xe7, 0xa9,
  0xbc, 0xe7, 0xa9, 0xbd, 0xe7, 0xa9, 0xbe, 0xe7, 0xaa, 0x82, 0xe7, 0xaa,
  0x85, 0xe7, 0xaa, 0x87, 0xe7, 0xaa, 0x89, 0xe7, 0xaa, 0x8a, 0xe7, 0xaa,
  0x8b, 0xe7, 0xaa, 0x8c, 0xe7, 0xaa, 0x8e, 0xe7, 0xaa, 0x8f, 0xe7, 0xaa,
  0x90, 0xe7, 0xaa, 0x93, 0xe7, 0xaa, 0x94, 0xe7, 0xaa, 0x99, 0xe7, 0xaa,
  0x9a, 0xe7, 0xaa, 0x9b, 0xe7, 0xaa, 0x9e, 0xe7, 0xaa, 0xa1, 0xe7, 0xaa,
  0xa2, 0xe8, 0xb4, 0xb0, 0xe5, 0x8f, 0x91, 0xe7, 0xbd, 0x9a, 0xe7, 0xad,
  0x8f, 0xe4, 0xbc, 0x90, 0xe4, 0xb9, 0x8f, 0xe9, 0x98, 0x80, 0xe6, 0xb3,
  0x95, 0xe7, 0x8f, 0x90, 0xe8, 0x97, 0xa9, 0xe5, 0xb8, 0x86, 0xe7, 0x95,
  0xaa, 0xe7, 0xbf, 0xbb, 0xe6, 0xa8, 0x8a, 0xe7, 0x9f, 0xbe, 0xe9, 0x92,
  0x92, 0xe7, 0xb9, 0x81, 0xe5, 0x87, 0xa1, 0xe7, 0x83, 0xa6, 0xe5, 0x8f,
  0x8d, 0xe8, 0xbf, 0x94, 0xe8, 0x8c, 0x83, 0xe8, 0xb4, 0xa9, 0xe7, 0x8a,
  0xaf, 0xe9, 0xa5, 0xad, 0xe6, 0xb3, 0x9b, 0xe5, 0x9d, 0x8a, 0xe8, 0x8a,
  0xb3, 0xe6, 0x96, 0xb9, 0xe8, 0x82, 0xaa, 0xe6, 0x88, 0xbf, 0xe9, 0x98,
  0xb2, 0xe5, 0xa6, 0xa8, 0xe4, 0xbb, 0xbf, 0xe8, 0xae, 0xbf, 0xe7, 0xba,
  0xba, 0xe6, 0x94, 0xbe, 0xe8, 0x8f, 0xb2, 0xe9, 0x9d, 0x9e, 0xe5, 0x95,
  0xa1, 0xe9, 0xa3, 0x9e, 0xe8, 0x82, 0xa5, 0xe5, 0x8c, 0xaa, 0xe8, 0xaf,
  0xbd, 0xe5, 0x90, 0xa0, 0xe8, 0x82, 0xba, 0xe5, 0xba, 0x9f, 0xe6, 0xb2,
  0xb8, 0xe8, 0xb4, 0xb9, 0xe8, 0x8a, 0xac, 0xe9, 0x85, 0x9a, 0xe5, 0x90,
  0xa9, 0xe6, 0xb0, 0x9b, 0xe5, 0x88, 0x86, 0xe7, 0xba, 0xb7, 0xe5, 0x9d,
  0x9f, 0xe7, 0x84, 0x9a, 0xe6, 0xb1, 0xbe, 0xe7, 0xb2, 0x89, 0xe5, 0xa5,
  0x8b, 0xe4, 0xbb, 0xbd, 0xe5, 0xbf, 0xbf, 0xe6, 0x84, 0xa4, 0xe7, 0xb2,
  0xaa, 0xe4, 0xb8, 0xb0, 0xe5, 0xb0, 0x81, 0xe6, 0x9e, 0xab, 0xe8, 0x9c,
  0x82, 0xe5, 0xb3, 0xb0, 0xe9, 0x94, 0x8b, 0xe9, 0xa3, 0x8e, 0xe7, 0x96,
  0xaf, 0xe7, 0x83, 0xbd, 0xe9, 0x80, 0xa2, 0xe5, 0x86, 0xaf, 0xe7, 0xbc,
  0x9d, 0xe8, 0xae, 0xbd, 0xe5, 0xa5, 0x89, 0xe5, 0x87, 0xa4, 0xe4, 0xbd,
  0x9b, 0xe5, 0x90, 0xa6, 0xe5, 0xa4, 0xab, 0xe6, 0x95, 0xb7, 0xe8, 0x82,
  0xa4, 0xe5, 0xad, 0xb5, 0xe6, 0x89, 0xb6, 0xe6, 0x8b, 0x82, 0xe8, 0xbe,
  0x90, 0xe5, 0xb9, 0x85, 0xe6, 0xb0, 0x9f, 0xe7, 0xac, 0xa6, 0xe4, 0xbc,
  0x8f, 0xe4, 0xbf, 0x98, 0xe6, 0x9c, 0x8d
};
unsigned char gbk_row_0xb8_converted_to_utf8[] = {
  0xe7, 0xaa, 0xa3, 0xe7,
  0xaa, 0xa4, 0xe7, 0xaa, 0xa7, 0xe7, 0xaa, 0xa9, 0xe7, 0xaa, 0xaa, 0xe7,
  0xaa, 0xab, 0xe7, 0xaa, 0xae, 0xe7, 0xaa, 0xaf, 0xe7, 0xaa, 0xb0, 0xe7,
  0xaa, 0xb1, 0xe7, 0xaa, 0xb2, 0xe7, 0xaa, 0xb4, 0xe7, 0xaa, 0xb5, 0xe7,
  0xaa, 0xb6, 0xe7, 0xaa, 0xb7, 0xe7, 0xaa, 0xb8, 0xe7, 0xaa, 0xb9, 0xe7,
  0xaa, 0xba, 0xe7, 0xaa, 0xbb, 0xe7, 0xaa, 0xbc, 0xe7, 0xaa, 0xbd, 0xe7,
  0xaa, 0xbe, 0xe7, 0xab, 0x80, 0xe7, 0xab, 0x81, 0xe7, 0xab, 0x82, 0xe7,
  0xab, 0x83, 0xe7, 0xab, 0x84, 0xe7, 0xab, 0x85, 0xe7, 0xab, 0x86, 0xe7,
  0xab, 0x87, 0xe7, 0xab, 0x88, 0xe7, 0xab, 0x89, 0xe7, 0xab, 0x8a, 0xe7,
  0xab, 0x8c, 0xe7, 0xab, 0x8d, 0xe7, 0xab, 0x8e, 0xe7, 0xab, 0x8f, 0xe7,
  0xab, 0x90, 0xe7, 0xab, 0x91, 0xe7, 0xab, 0x92, 0xe7, 0xab, 0x93, 0xe7,
  0xab, 0x94, 0xe7, 0xab, 0x95, 0xe7, 0xab, 0x97, 0xe7, 0xab, 0x98, 0xe7,
  0xab, 0x9a, 0xe7, 0xab, 0x9b, 0xe7, 0xab, 0x9c, 0xe7, 0xab, 0x9d, 0xe7,
  0xab, 0xa1, 0xe7, 0xab, 0xa2, 0xe7, 0xab, 0xa4, 0xe7, 0xab, 0xa7, 0xe7,
  0xab, 0xa8, 0xe7, 0xab, 0xa9, 0xe7, 0xab, 0xaa, 0xe7, 0xab, 0xab, 0xe7,
  0xab, 0xac, 0xe7, 0xab, 0xae, 0xe7, 0xab, 0xb0, 0xe7, 0xab, 0xb1, 0xe7,
  0xab, 0xb2, 0xe7, 0xab, 0xb3, 0xe7, 0xab, 0xb4, 0xe7, 0xab, 0xb5, 0xe7,
  0xab, 0xb6, 0xe7, 0xab, 0xb7, 0xe7, 0xab, 0xb8, 0xe7, 0xab, 0xbb, 0xe7,
  0xab, 0xbc, 0xe7, 0xab, 0xbe, 0xe7, 0xac, 0x80, 0xe7, 0xac, 0x81, 0xe7,
  0xac, 0x82, 0xe7, 0xac, 0x85, 0xe7, 0xac, 0x87, 0xe7, 0xac, 0x89, 0xe7,
  0xac, 0x8c, 0xe7, 0xac, 0x8d, 0xe7, 0xac, 0x8e, 0xe7, 0xac, 0x90, 0xe7,
  0xac, 0x92, 0xe7, 0xac, 0x93, 0xe7, 0xac, 0x96, 0xe7, 0xac, 0x97, 0xe7,
  0xac, 0x98, 0xe7, 0xac, 0x9a, 0xe7, 0xac, 0x9c, 0xe7, 0xac, 0x9d, 0xe7,
  0xac, 0x9f, 0xe7, 0xac, 0xa1, 0xe7, 0xac, 0xa2, 0xe7, 0xac, 0xa3, 0xe7,
  0xac, 0xa7, 0xe7, 0xac, 0xa9, 0xe7, 0xac, 0xad, 0xe6, 0xb5, 0xae, 0xe6,
  0xb6, 0xaa, 0xe7, 0xa6, 0x8f, 0xe8, 0xa2, 0xb1, 0xe5, 0xbc, 0x97, 0xe7,
  0x94, 0xab, 0xe6, 0x8a, 0x9a, 0xe8, 0xbe, 0x85, 0xe4, 0xbf, 0xaf, 0xe9,
  0x87, 0x9c, 0xe6, 0x96, 0xa7, 0xe8, 0x84, 0xaf, 0xe8, 0x85, 0x91, 0xe5,
  0xba, 0x9c, 0xe8, 0x85, 0x90, 0xe8, 0xb5, 0xb4, 0xe5, 0x89, 0xaf, 0xe8,
  0xa6, 0x86, 0xe8, 0xb5, 0x8b, 0xe5, 0xa4, 0x8d, 0xe5, 0x82, 0x85, 0xe4,
  0xbb, 0x98, 0xe9, 0x98, 0x9c, 0xe7, 0x88, 0xb6, 0xe8, 0x85, 0xb9, 0xe8,
  0xb4, 0x9f, 0xe5, 0xaf, 0x8c, 0xe8, 0xae, 0xa3, 0xe9, 0x99, 0x84, 0xe5,
  0xa6, 0x87, 0xe7, 0xbc, 0x9a, 0xe5, 0x92, 0x90, 0xe5, 0x99, 0xb6, 0xe5,
  0x98, 0x8e, 0xe8, 0xaf, 0xa5, 0xe6, 0x94, 0xb9, 0xe6, 0xa6, 0x82, 0xe9,
  0x92, 0x99, 0xe7, 0x9b, 0x96, 0xe6, 0xba, 0x89, 0xe5, 0xb9, 0xb2, 0xe7,
  0x94, 0x98, 0xe6, 0x9d, 0x86, 0xe6, 0x9f, 0x91, 0xe7, 0xab, 0xbf, 0xe8,
  0x82, 0x9d, 0xe8, 0xb5, 0xb6, 0xe6, 0x84, 0x9f, 0xe7, 0xa7, 0x86, 0xe6,
  0x95, 0xa2, 0xe8, 0xb5, 0xa3, 0xe5, 0x86, 0x88, 0xe5, 0x88, 0x9a, 0xe9,
  0x92, 0xa2, 0xe7, 0xbc, 0xb8, 0xe8, 0x82, 0x9b, 0xe7, 0xba, 0xb2, 0xe5,
  0xb2, 0x97, 0xe6, 0xb8, 0xaf, 0xe6, 0x9d, 0xa0, 0xe7, 0xaf, 0x99, 0xe7,
  0x9a, 0x8b, 0xe9, 0xab, 0x98, 0xe8, 0x86, 0x8f, 0xe7, 0xbe, 0x94, 0xe7,
  0xb3, 0x95, 0xe6, 0x90, 0x9e, 0xe9, 0x95, 0x90, 0xe7, 0xa8, 0xbf, 0xe5,
  0x91, 0x8a, 0xe5, 0x93, 0xa5, 0xe6, 0xad, 0x8c, 0xe6, 0x90, 0x81, 0xe6,
  0x88, 0x88, 0xe9, 0xb8, 0xbd, 0xe8, 0x83, 0xb3, 0xe7, 0x96, 0x99, 0xe5,
  0x89, 0xb2, 0xe9, 0x9d, 0xa9, 0xe8, 0x91, 0x9b, 0xe6, 0xa0, 0xbc, 0xe8,
  0x9b, 0xa4, 0xe9, 0x98, 0x81, 0xe9, 0x9a, 0x94, 0xe9, 0x93, 0xac, 0xe4,
  0xb8, 0xaa, 0xe5, 0x90, 0x84, 0xe7, 0xbb, 0x99, 0xe6, 0xa0, 0xb9, 0xe8,
  0xb7, 0x9f, 0xe8, 0x80, 0x95, 0xe6, 0x9b, 0xb4, 0xe5, 0xba, 0x9a, 0xe7,
  0xbe, 0xb9
};

unsigned char gbk_row_0xb9_converted_to_utf8[] = {
  0xe7, 0xac, 0xaf, 0xe7, 0xac, 0xb0, 0xe7, 0xac, 0xb2,
  0xe7, 0xac, 0xb4, 0xe7, 0xac, 0xb5, 0xe7, 0xac, 0xb6, 0xe7, 0xac, 0xb7,
  0xe7, 0xac, 0xb9, 0xe7, 0xac, 0xbb, 0xe7, 0xac, 0xbd, 0xe7, 0xac, 0xbf,
  0xe7, 0xad, 0x80, 0xe7, 0xad, 0x81, 0xe7, 0xad, 0x82, 0xe7, 0xad, 0x83,
  0xe7, 0xad, 0x84, 0xe7, 0xad, 0x86, 0xe7, 0xad, 0x88, 0xe7, 0xad, 0x8a,
  0xe7, 0xad, 0x8d, 0xe7, 0xad, 0x8e, 0xe7, 0xad, 0x93, 0xe7, 0xad, 0x95,
  0xe7, 0xad, 0x97, 0xe7, 0xad, 0x99, 0xe7, 0xad, 0x9c, 0xe7, 0xad, 0x9e,
  0xe7, 0xad, 0x9f, 0xe7, 0xad, 0xa1, 0xe7, 0xad, 0xa3, 0xe7, 0xad, 0xa4,
  0xe7, 0xad, 0xa5, 0xe7, 0xad, 0xa6, 0xe7, 0xad, 0xa7, 0xe7, 0xad, 0xa8,
  0xe7, 0xad, 0xa9, 0xe7, 0xad, 0xaa, 0xe7, 0xad, 0xab, 0xe7, 0xad, 0xac,
  0xe7, 0xad, 0xad, 0xe7, 0xad, 0xaf, 0xe7, 0xad, 0xb0, 0xe7, 0xad, 0xb3,
  0xe7, 0xad, 0xb4, 0xe7, 0xad, 0xb6, 0xe7, 0xad, 0xb8, 0xe7, 0xad, 0xba,
  0xe7, 0xad, 0xbc, 0xe7, 0xad, 0xbd, 0xe7, 0xad, 0xbf, 0xe7, 0xae, 0x81,
  0xe7, 0xae, 0x82, 0xe7, 0xae, 0x83, 0xe7, 0xae, 0x84, 0xe7, 0xae, 0x86,
  0xe7, 0xae, 0x87, 0xe7, 0xae, 0x88, 0xe7, 0xae, 0x89, 0xe7, 0xae, 0x8a,
  0xe7, 0xae, 0x8b, 0xe7, 0xae, 0x8c, 0xe7, 0xae, 0x8e, 0xe7, 0xae, 0x8f,
  0xe7, 0xae, 0x91, 0xe7, 0xae, 0x92, 0xe7, 0xae, 0x93, 0xe7, 0xae, 0x96,
  0xe7, 0xae, 0x98, 0xe7, 0xae, 0x99, 0xe7, 0xae, 0x9a, 0xe7, 0xae, 0x9b,
  0xe7, 0xae, 0x9e, 0xe7, 0xae, 0x9f, 0xe7, 0xae, 0xa0, 0xe7, 0xae, 0xa3,
  0xe7, 0xae, 0xa4, 0xe7, 0xae, 0xa5, 0xe7, 0xae, 0xae, 0xe7, 0xae, 0xaf,
  0xe7, 0xae, 0xb0, 0xe7, 0xae, 0xb2, 0xe7, 0xae, 0xb3, 0xe7, 0xae, 0xb5,
  0xe7, 0xae, 0xb6, 0xe7, 0xae, 0xb7, 0xe7, 0xae, 0xb9, 0xe7, 0xae, 0xba,
  0xe7, 0xae, 0xbb, 0xe7, 0xae, 0xbc, 0xe7, 0xae, 0xbd, 0xe7, 0xae, 0xbe,
  0xe7, 0xae, 0xbf, 0xe7, 0xaf, 0x80, 0xe7, 0xaf, 0x82, 0xe7, 0xaf, 0x83,
  0xe7, 0xaf, 0x84, 0xe5, 0x9f, 0x82, 0xe8, 0x80, 0xbf, 0xe6, 0xa2, 0x97,
  0xe5, 0xb7, 0xa5, 0xe6, 0x94, 0xbb, 0xe5, 0x8a, 0x9f, 0xe6, 0x81, 0xad,
  0xe9, 0xbe, 0x9a, 0xe4, 0xbe, 0x9b, 0xe8, 0xba, 0xac, 0xe5, 0x85, 0xac,
  0xe5, 0xae, 0xab, 0xe5, 0xbc, 0x93, 0xe5, 0xb7, 0xa9, 0xe6, 0xb1, 0x9e,
  0xe6, 0x8b, 0xb1, 0xe8, 0xb4, 0xa1, 0xe5, 0x85, 0xb1, 0xe9, 0x92, 0xa9,
  0xe5, 0x8b, 0xbe, 0xe6, 0xb2, 0x9f, 0xe8, 0x8b, 0x9f, 0xe7, 0x8b, 0x97,
  0xe5, 0x9e, 0xa2, 0xe6, 0x9e, 0x84, 0xe8, 0xb4, 0xad, 0xe5, 0xa4, 0x9f,
  0xe8, 0xbe, 0x9c, 0xe8, 0x8f, 0x87, 0xe5, 0x92, 0x95, 0xe7, 0xae, 0x8d,
  0xe4, 0xbc, 0xb0, 0xe6, 0xb2, 0xbd, 0xe5, 0xad, 0xa4, 0xe5, 0xa7, 0x91,
  0xe9, 0xbc, 0x93, 0xe5, 0x8f, 0xa4, 0xe8, 0x9b, 0x8a, 0xe9, 0xaa, 0xa8,
  0xe8, 0xb0, 0xb7, 0xe8, 0x82, 0xa1, 0xe6, 0x95, 0x85, 0xe9, 0xa1, 0xbe,
  0xe5, 0x9b, 0xba, 0xe9, 0x9b, 0x87, 0xe5, 0x88, 0xae, 0xe7, 0x93, 0x9c,
  0xe5, 0x89, 0x90, 0xe5, 0xaf, 0xa1, 0xe6, 0x8c, 0x82, 0xe8, 0xa4, 0x82,
  0xe4, 0xb9, 0x96, 0xe6, 0x8b, 0x90, 0xe6, 0x80, 0xaa, 0xe6, 0xa3, 0xba,
  0xe5, 0x85, 0xb3, 0xe5, 0xae, 0x98, 0xe5, 0x86, 0xa0, 0xe8, 0xa7, 0x82,
  0xe7, 0xae, 0xa1, 0xe9, 0xa6, 0x86, 0xe7, 0xbd, 0x90, 0xe6, 0x83, 0xaf,
  0xe7, 0x81, 0x8c, 0xe8, 0xb4, 0xaf, 0xe5, 0x85, 0x89, 0xe5, 0xb9, 0xbf,
  0xe9, 0x80, 0x9b, 0xe7, 0x91, 0xb0, 0xe8, 0xa7, 0x84, 0xe5, 0x9c, 0xad,
  0xe7, 0xa1, 0x85, 0xe5, 0xbd, 0x92, 0xe9, 0xbe, 0x9f, 0xe9, 0x97, 0xba,
  0xe8, 0xbd, 0xa8, 0xe9, 0xac, 0xbc, 0xe8, 0xaf, 0xa1, 0xe7, 0x99, 0xb8,
  0xe6, 0xa1, 0x82, 0xe6, 0x9f, 0x9c, 0xe8, 0xb7, 0xaa, 0xe8, 0xb4, 0xb5,
  0xe5, 0x88, 0xbd, 0xe8, 0xbe, 0x8a, 0xe6, 0xbb, 0x9a, 0xe6, 0xa3, 0x8d,
  0xe9, 0x94, 0x85, 0xe9, 0x83, 0xad, 0xe5, 0x9b, 0xbd, 0xe6, 0x9e, 0x9c,
  0xe8, 0xa3, 0xb9, 0xe8, 0xbf, 0x87, 0xe5, 0x93, 0x88
};

unsigned char gbk_row_0xba_converted_to_utf8[] = {
  0xe7, 0xaf,
  0x85, 0xe7, 0xaf, 0x88, 0xe7, 0xaf, 0x89, 0xe7, 0xaf, 0x8a, 0xe7, 0xaf,
  0x8b, 0xe7, 0xaf, 0x8d, 0xe7, 0xaf, 0x8e, 0xe7, 0xaf, 0x8f, 0xe7, 0xaf,
  0x90, 0xe7, 0xaf, 0x92, 0xe7, 0xaf, 0x94, 0xe7, 0xaf, 0x95, 0xe7, 0xaf,
  0x96, 0xe7, 0xaf, 0x97, 0xe7, 0xaf, 0x98, 0xe7, 0xaf, 0x9b, 0xe7, 0xaf,
  0x9c, 0xe7, 0xaf, 0x9e, 0xe7, 0xaf, 0x9f, 0xe7, 0xaf, 0xa0, 0xe7, 0xaf,
  0xa2, 0xe7, 0xaf, 0xa3, 0xe7, 0xaf, 0xa4, 0xe7, 0xaf, 0xa7, 0xe7, 0xaf,
  0xa8, 0xe7, 0xaf, 0xa9, 0xe7, 0xaf, 0xab, 0xe7, 0xaf, 0xac, 0xe7, 0xaf,
  0xad, 0xe7, 0xaf, 0xaf, 0xe7, 0xaf, 0xb0, 0xe7, 0xaf, 0xb2, 0xe7, 0xaf,
  0xb3, 0xe7, 0xaf, 0xb4, 0xe7, 0xaf, 0xb5, 0xe7, 0xaf, 0xb6, 0xe7, 0xaf,
  0xb8, 0xe7, 0xaf, 0xb9, 0xe7, 0xaf, 0xba, 0xe7, 0xaf, 0xbb, 0xe7, 0xaf,
  0xbd, 0xe7, 0xaf, 0xbf, 0xe7, 0xb0, 0x80, 0xe7, 0xb0, 0x81, 0xe7, 0xb0,
  0x82, 0xe7, 0xb0, 0x83, 0xe7, 0xb0, 0x84, 0xe7, 0xb0, 0x85, 0xe7, 0xb0,
  0x86, 0xe7, 0xb0, 0x88, 0xe7, 0xb0, 0x89, 0xe7, 0xb0, 0x8a, 0xe7, 0xb0,
  0x8d, 0xe7, 0xb0, 0x8e, 0xe7, 0xb0, 0x90, 0xe7, 0xb0, 0x91, 0xe7, 0xb0,
  0x92, 0xe7, 0xb0, 0x93, 0xe7, 0xb0, 0x94, 0xe7, 0xb0, 0x95, 0xe7, 0xb0,
  0x97, 0xe7, 0xb0, 0x98, 0xe7, 0xb0, 0x99, 0xe7, 0xb0, 0x9a, 0xe7, 0xb0,
  0x9b, 0xe7, 0xb0, 0x9c, 0xe7, 0xb0, 0x9d, 0xe7, 0xb0, 0x9e, 0xe7, 0xb0,
  0xa0, 0xe7, 0xb0, 0xa1, 0xe7, 0xb0, 0xa2, 0xe7, 0xb0, 0xa3, 0xe7, 0xb0,
  0xa4, 0xe7, 0xb0, 0xa5, 0xe7, 0xb0, 0xa8, 0xe7, 0xb0, 0xa9, 0xe7, 0xb0,
  0xab, 0xe7, 0xb0, 0xac, 0xe7, 0xb0, 0xad, 0xe7, 0xb0, 0xae, 0xe7, 0xb0,
  0xaf, 0xe7, 0xb0, 0xb0, 0xe7, 0xb0, 0xb1, 0xe7, 0xb0, 0xb2, 0xe7, 0xb0,
  0xb3, 0xe7, 0xb0, 0xb4, 0xe7, 0xb0, 0xb5, 0xe7, 0xb0, 0xb6, 0xe7, 0xb0,
  0xb7, 0xe7, 0xb0, 0xb9, 0xe7, 0xb0, 0xba, 0xe7, 0xb0, 0xbb, 0xe7, 0xb0,
  0xbc, 0xe7, 0xb0, 0xbd, 0xe7, 0xb0, 0xbe, 0xe7, 0xb1, 0x82, 0xe9, 0xaa,
  0xb8, 0xe5, 0xad, 0xa9, 0xe6, 0xb5, 0xb7, 0xe6, 0xb0, 0xa6, 0xe4, 0xba,
  0xa5, 0xe5, 0xae, 0xb3, 0xe9, 0xaa, 0x87, 0xe9, 0x85, 0xa3, 0xe6, 0x86,
  0xa8, 0xe9, 0x82, 0xaf, 0xe9, 0x9f, 0xa9, 0xe5, 0x90, 0xab, 0xe6, 0xb6,
  0xb5, 0xe5, 0xaf, 0x92, 0xe5, 0x87, 0xbd, 0xe5, 0x96, 0x8a, 0xe7, 0xbd,
  0x95, 0xe7, 0xbf, 0xb0, 0xe6, 0x92, 0xbc, 0xe6, 0x8d, 0x8d, 0xe6, 0x97,
  0xb1, 0xe6, 0x86, 0xbe, 0xe6, 0x82, 0x8d, 0xe7, 0x84, 0x8a, 0xe6, 0xb1,
  0x97, 0xe6, 0xb1, 0x89, 0xe5, 0xa4, 0xaf, 0xe6, 0x9d, 0xad, 0xe8, 0x88,
  0xaa, 0xe5, 0xa3, 0x95, 0xe5, 0x9a, 0x8e, 0xe8, 0xb1, 0xaa, 0xe6, 0xaf,
  0xab, 0xe9, 0x83, 0x9d, 0xe5, 0xa5, 0xbd, 0xe8, 0x80, 0x97, 0xe5, 0x8f,
  0xb7, 0xe6, 0xb5, 0xa9, 0xe5, 0x91, 0xb5, 0xe5, 0x96, 0x9d, 0xe8, 0x8d,
  0xb7, 0xe8, 0x8f, 0x8f, 0xe6, 0xa0, 0xb8, 0xe7, 0xa6, 0xbe, 0xe5, 0x92,
  0x8c, 0xe4, 0xbd, 0x95, 0xe5, 0x90, 0x88, 0xe7, 0x9b, 0x92, 0xe8, 0xb2,
  0x89, 0xe9, 0x98, 0x82, 0xe6, 0xb2, 0xb3, 0xe6, 0xb6, 0xb8, 0xe8, 0xb5,
  0xab, 0xe8, 0xa4, 0x90, 0xe9, 0xb9, 0xa4, 0xe8, 0xb4, 0xba, 0xe5, 0x98,
  0xbf, 0xe9, 0xbb, 0x91, 0xe7, 0x97, 0x95, 0xe5, 0xbe, 0x88, 0xe7, 0x8b,
  0xa0, 0xe6, 0x81, 0xa8, 0xe5, 0x93, 0xbc, 0xe4, 0xba, 0xa8, 0xe6, 0xa8,
  0xaa, 0xe8, 0xa1, 0xa1, 0xe6, 0x81, 0x92, 0xe8, 0xbd, 0xb0, 0xe5, 0x93,
  0x84, 0xe7, 0x83, 0x98, 0xe8, 0x99, 0xb9, 0xe9, 0xb8, 0xbf, 0xe6, 0xb4,
  0xaa, 0xe5, 0xae, 0x8f, 0xe5, 0xbc, 0x98, 0xe7, 0xba, 0xa2, 0xe5, 0x96,
  0x89, 0xe4, 0xbe, 0xaf, 0xe7, 0x8c, 0xb4, 0xe5, 0x90, 0xbc, 0xe5, 0x8e,
  0x9a, 0xe5, 0x80, 0x99, 0xe5, 0x90, 0x8e, 0xe5, 0x91, 0xbc, 0xe4, 0xb9,
  0x8e, 0xe5, 0xbf, 0xbd, 0xe7, 0x91, 0x9a, 0xe5, 0xa3, 0xb6, 0xe8, 0x91,
  0xab, 0xe8, 0x83, 0xa1, 0xe8, 0x9d, 0xb4, 0xe7, 0x8b, 0x90, 0xe7, 0xb3,
  0x8a, 0xe6, 0xb9, 0x96
};

unsigned char gbk_row_0xbb_converted_to_utf8[] = {
  0xe7, 0xb1, 0x83, 0xe7, 0xb1, 0x84, 0xe7,
  0xb1, 0x85, 0xe7, 0xb1, 0x86, 0xe7, 0xb1, 0x87, 0xe7, 0xb1, 0x88, 0xe7,
  0xb1, 0x89, 0xe7, 0xb1, 0x8a, 0xe7, 0xb1, 0x8b, 0xe7, 0xb1, 0x8c, 0xe7,
  0xb1, 0x8e, 0xe7, 0xb1, 0x8f, 0xe7, 0xb1, 0x90, 0xe7, 0xb1, 0x91, 0xe7,
  0xb1, 0x92, 0xe7, 0xb1, 0x93, 0xe7, 0xb1, 0x94, 0xe7, 0xb1, 0x95, 0xe7,
  0xb1, 0x96, 0xe7, 0xb1, 0x97, 0xe7, 0xb1, 0x98, 0xe7, 0xb1, 0x99, 0xe7,
  0xb1, 0x9a, 0xe7, 0xb1, 0x9b, 0xe7, 0xb1, 0x9c, 0xe7, 0xb1, 0x9d, 0xe7,
  0xb1, 0x9e, 0xe7, 0xb1, 0x9f, 0xe7, 0xb1, 0xa0, 0xe7, 0xb1, 0xa1, 0xe7,
  0xb1, 0xa2, 0xe7, 0xb1, 0xa3, 0xe7, 0xb1, 0xa4, 0xe7, 0xb1, 0xa5, 0xe7,
  0xb1, 0xa6, 0xe7, 0xb1, 0xa7, 0xe7, 0xb1, 0xa8, 0xe7, 0xb1, 0xa9, 0xe7,
  0xb1, 0xaa, 0xe7, 0xb1, 0xab, 0xe7, 0xb1, 0xac, 0xe7, 0xb1, 0xad, 0xe7,
  0xb1, 0xae, 0xe7, 0xb1, 0xaf, 0xe7, 0xb1, 0xb0, 0xe7, 0xb1, 0xb1, 0xe7,
  0xb1, 0xb2, 0xe7, 0xb1, 0xb5, 0xe7, 0xb1, 0xb6, 0xe7, 0xb1, 0xb7, 0xe7,
  0xb1, 0xb8, 0xe7, 0xb1, 0xb9, 0xe7, 0xb1, 0xba, 0xe7, 0xb1, 0xbe, 0xe7,
  0xb1, 0xbf, 0xe7, 0xb2, 0x80, 0xe7, 0xb2, 0x81, 0xe7, 0xb2, 0x82, 0xe7,
  0xb2, 0x83, 0xe7, 0xb2, 0x84, 0xe7, 0xb2, 0x85, 0xe7, 0xb2, 0x86, 0xe7,
  0xb2, 0x87, 0xe7, 0xb2, 0x88, 0xe7, 0xb2, 0x8a, 0xe7, 0xb2, 0x8b, 0xe7,
  0xb2, 0x8c, 0xe7, 0xb2, 0x8d, 0xe7, 0xb2, 0x8e, 0xe7, 0xb2, 0x8f, 0xe7,
  0xb2, 0x90, 0xe7, 0xb2, 0x93, 0xe7, 0xb2, 0x94, 0xe7, 0xb2, 0x96, 0xe7,
  0xb2, 0x99, 0xe7, 0xb2, 0x9a, 0xe7, 0xb2, 0x9b, 0xe7, 0xb2, 0xa0, 0xe7,
  0xb2, 0xa1, 0xe7, 0xb2, 0xa3, 0xe7, 0xb2, 0xa6, 0xe7, 0xb2, 0xa7, 0xe7,
  0xb2, 0xa8, 0xe7, 0xb2, 0xa9, 0xe7, 0xb2, 0xab, 0xe7, 0xb2, 0xac, 0xe7,
  0xb2, 0xad, 0xe7, 0xb2, 0xaf, 0xe7, 0xb2, 0xb0, 0xe7, 0xb2, 0xb4, 0xe7,
  0xb2, 0xb5, 0xe7, 0xb2, 0xb6, 0xe7, 0xb2, 0xb7, 0xe7, 0xb2, 0xb8, 0xe7,
  0xb2, 0xba, 0xe7, 0xb2, 0xbb, 0xe5, 0xbc, 0xa7, 0xe8, 0x99, 0x8e, 0xe5,
  0x94, 0xac, 0xe6, 0x8a, 0xa4, 0xe4, 0xba, 0x92, 0xe6, 0xb2, 0xaa, 0xe6,
  0x88, 0xb7, 0xe8, 0x8a, 0xb1, 0xe5, 0x93, 0x97, 0xe5, 0x8d, 0x8e, 0xe7,
  0x8c, 0xbe, 0xe6, 0xbb, 0x91, 0xe7, 0x94, 0xbb, 0xe5, 0x88, 0x92, 0xe5,
  0x8c, 0x96, 0xe8, 0xaf, 0x9d, 0xe6, 0xa7, 0x90, 0xe5, 0xbe, 0x8a, 0xe6,
  0x80, 0x80, 0xe6, 0xb7, 0xae, 0xe5, 0x9d, 0x8f, 0xe6, 0xac, 0xa2, 0xe7,
  0x8e, 0xaf, 0xe6, 0xa1, 0x93, 0xe8, 0xbf, 0x98, 0xe7, 0xbc, 0x93, 0xe6,
  0x8d, 0xa2, 0xe6, 0x82, 0xa3, 0xe5, 0x94, 0xa4, 0xe7, 0x97, 0xaa, 0xe8,
  0xb1, 0xa2, 0xe7, 0x84, 0x95, 0xe6, 0xb6, 0xa3, 0xe5, 0xae, 0xa6, 0xe5,
  0xb9, 0xbb, 0xe8, 0x8d, 0x92, 0xe6, 0x85, 0x8c, 0xe9, 0xbb, 0x84, 0xe7,
  0xa3, 0xba, 0xe8, 0x9d, 0x97, 0xe7, 0xb0, 0xa7, 0xe7, 0x9a, 0x87, 0xe5,
  0x87, 0xb0, 0xe6, 0x83, 0xb6, 0xe7, 0x85, 0x8c, 0xe6, 0x99, 0x83, 0xe5,
  0xb9, 0x8c, 0xe6, 0x81, 0x8d, 0xe8, 0xb0, 0x8e, 0xe7, 0x81, 0xb0, 0xe6,
  0x8c, 0xa5, 0xe8, 0xbe, 0x89, 0xe5, 0xbe, 0xbd, 0xe6, 0x81, 0xa2, 0xe8,
  0x9b, 0x94, 0xe5, 0x9b, 0x9e, 0xe6, 0xaf, 0x81, 0xe6, 0x82, 0x94, 0xe6,
  0x85, 0xa7, 0xe5, 0x8d, 0x89, 0xe6, 0x83, 0xa0, 0xe6, 0x99, 0xa6, 0xe8,
  0xb4, 0xbf, 0xe7, 0xa7, 0xbd, 0xe4, 0xbc, 0x9a, 0xe7, 0x83, 0xa9, 0xe6,
  0xb1, 0x87, 0xe8, 0xae, 0xb3, 0xe8, 0xaf, 0xb2, 0xe7, 0xbb, 0x98, 0xe8,
  0x8d, 0xa4, 0xe6, 0x98, 0x8f, 0xe5, 0xa9, 0x9a, 0xe9, 0xad, 0x82, 0xe6,
  0xb5, 0x91, 0xe6, 0xb7, 0xb7, 0xe8, 0xb1, 0x81, 0xe6, 0xb4, 0xbb, 0xe4,
  0xbc, 0x99, 0xe7, 0x81, 0xab, 0xe8, 0x8e, 0xb7, 0xe6, 0x88, 0x96, 0xe6,
  0x83, 0x91, 0xe9, 0x9c, 0x8d, 0xe8, 0xb4, 0xa7, 0xe7, 0xa5, 0xb8, 0xe5,
  0x87, 0xbb, 0xe5, 0x9c, 0xbe, 0xe5, 0x9f, 0xba, 0xe6, 0x9c, 0xba, 0xe7,
  0x95, 0xb8, 0xe7, 0xa8, 0xbd, 0xe7, 0xa7, 0xaf, 0xe7, 0xae, 0x95
};

unsigned char gbk_row_0xbc_converted_to_utf8[] = {
  0xe7, 0xb2, 0xbf, 0xe7, 0xb3, 0x80, 0xe7, 0xb3, 0x82, 0xe7, 0xb3, 0x83,
  0xe7, 0xb3, 0x84, 0xe7, 0xb3, 0x86, 0xe7, 0xb3, 0x89, 0xe7, 0xb3, 0x8b,
  0xe7, 0xb3, 0x8e, 0xe7, 0xb3, 0x8f, 0xe7, 0xb3, 0x90, 0xe7, 0xb3, 0x91,
  0xe7, 0xb3, 0x92, 0xe7, 0xb3, 0x93, 0xe7, 0xb3, 0x94, 0xe7, 0xb3, 0x98,
  0xe7, 0xb3, 0x9a, 0xe7, 0xb3, 0x9b, 0xe7, 0xb3, 0x9d, 0xe7, 0xb3, 0x9e,
  0xe7, 0xb3, 0xa1, 0xe7, 0xb3, 0xa2, 0xe7, 0xb3, 0xa3, 0xe7, 0xb3, 0xa4,
  0xe7, 0xb3, 0xa5, 0xe7, 0xb3, 0xa6, 0xe7, 0xb3, 0xa7, 0xe7, 0xb3, 0xa9,
  0xe7, 0xb3, 0xaa, 0xe7, 0xb3, 0xab, 0xe7, 0xb3, 0xac, 0xe7, 0xb3, 0xad,
  0xe7, 0xb3, 0xae, 0xe7, 0xb3, 0xb0, 0xe7, 0xb3, 0xb1, 0xe7, 0xb3, 0xb2,
  0xe7, 0xb3, 0xb3, 0xe7, 0xb3, 0xb4, 0xe7, 0xb3, 0xb5, 0xe7, 0xb3, 0xb6,
  0xe7, 0xb3, 0xb7, 0xe7, 0xb3, 0xb9, 0xe7, 0xb3, 0xba, 0xe7, 0xb3, 0xbc,
  0xe7, 0xb3, 0xbd, 0xe7, 0xb3, 0xbe, 0xe7, 0xb3, 0xbf, 0xe7, 0xb4, 0x80,
  0xe7, 0xb4, 0x81, 0xe7, 0xb4, 0x82, 0xe7, 0xb4, 0x83, 0xe7, 0xb4, 0x84,
  0xe7, 0xb4, 0x85, 0xe7, 0xb4, 0x86, 0xe7, 0xb4, 0x87, 0xe7, 0xb4, 0x88,
  0xe7, 0xb4, 0x89, 0xe7, 0xb4, 0x8b, 0xe7, 0xb4, 0x8c, 0xe7, 0xb4, 0x8d,
  0xe7, 0xb4, 0x8e, 0xe7, 0xb4, 0x8f, 0xe7, 0xb4, 0x90, 0xe7, 0xb4, 0x91,
  0xe7, 0xb4, 0x92, 0xe7, 0xb4, 0x93, 0xe7, 0xb4, 0x94, 0xe7, 0xb4, 0x95,
  0xe7, 0xb4, 0x96, 0xe7, 0xb4, 0x97, 0xe7, 0xb4, 0x98, 0xe7, 0xb4, 0x99,
  0xe7, 0xb4, 0x9a, 0xe7, 0xb4, 0x9b, 0xe7, 0xb4, 0x9c, 0xe7, 0xb4, 0x9d,
  0xe7, 0xb4, 0x9e, 0xe7, 0xb4, 0x9f, 0xe7, 0xb4, 0xa1, 0xe7, 0xb4, 0xa3,
  0xe7, 0xb4, 0xa4, 0xe7, 0xb4, 0xa5, 0xe7, 0xb4, 0xa6, 0xe7, 0xb4, 0xa8,
  0xe7, 0xb4, 0xa9, 0xe7, 0xb4, 0xaa, 0xe7, 0xb4, 0xac, 0xe7, 0xb4, 0xad,
  0xe7, 0xb4, 0xae, 0xe7, 0xb4, 0xb0, 0xe7, 0xb4, 0xb1, 0xe7, 0xb4, 0xb2,
  0xe7, 0xb4, 0xb3, 0xe7, 0xb4, 0xb4, 0xe7, 0xb4, 0xb5, 0xe7, 0xb4, 0xb6,
  0xe8, 0x82, 0x8c, 0xe9, 0xa5, 0xa5, 0xe8, 0xbf, 0xb9, 0xe6, 0xbf, 0x80,
  0xe8, 0xae, 0xa5, 0xe9, 0xb8, 0xa1, 0xe5, 0xa7, 0xac, 0xe7, 0xbb, 0xa9,
  0xe7, 0xbc, 0x89, 0xe5, 0x90, 0x89, 0xe6, 0x9e, 0x81, 0xe6, 0xa3, 0x98,
  0xe8, 0xbe, 0x91, 0xe7, 0xb1, 0x8d, 0xe9, 0x9b, 0x86, 0xe5, 0x8f, 0x8a,
  0xe6, 0x80, 0xa5, 0xe7, 0x96, 0xbe, 0xe6, 0xb1, 0xb2, 0xe5, 0x8d, 0xb3,
  0xe5, 0xab, 0x89, 0xe7, 0xba, 0xa7, 0xe6, 0x8c, 0xa4, 0xe5, 0x87, 0xa0,
  0xe8, 0x84, 0x8a, 0xe5, 0xb7, 0xb1, 0xe8, 0x93, 0x9f, 0xe6, 0x8a, 0x80,
  0xe5, 0x86, 0x80, 0xe5, 0xad, 0xa3, 0xe4, 0xbc, 0x8e, 0xe7, 0xa5, 0xad,
  0xe5, 0x89, 0x82, 0xe6, 0x82, 0xb8, 0xe6, 0xb5, 0x8e, 0xe5, 0xaf, 0x84,
  0xe5, 0xaf, 0x82, 0xe8, 0xae, 0xa1, 0xe8, 0xae, 0xb0, 0xe6, 0x97, 0xa2,
  0xe5, 0xbf, 0x8c, 0xe9, 0x99, 0x85, 0xe5, 0xa6, 0x93, 0xe7, 0xbb, 0xa7,
  0xe7, 0xba, 0xaa, 0xe5, 0x98, 0x89, 0xe6, 0x9e, 0xb7, 0xe5, 0xa4, 0xb9,
  0xe4, 0xbd, 0xb3, 0xe5, 0xae, 0xb6, 0xe5, 0x8a, 0xa0, 0xe8, 0x8d, 0x9a,
  0xe9, 0xa2, 0x8a, 0xe8, 0xb4, 0xbe, 0xe7, 0x94, 0xb2, 0xe9, 0x92, 0xbe,
  0xe5, 0x81, 0x87, 0xe7, 0xa8, 0xbc, 0xe4, 0xbb, 0xb7, 0xe6, 0x9e, 0xb6,
  0xe9, 0xa9, 0xbe, 0xe5, 0xab, 0x81, 0xe6, 0xad, 0xbc, 0xe7, 0x9b, 0x91,
  0xe5, 0x9d, 0x9a, 0xe5, 0xb0, 0x96, 0xe7, 0xac, 0xba, 0xe9, 0x97, 0xb4,
  0xe7, 0x85, 0x8e, 0xe5, 0x85, 0xbc, 0xe8, 0x82, 0xa9, 0xe8, 0x89, 0xb0,
  0xe5, 0xa5, 0xb8, 0xe7, 0xbc, 0x84, 0xe8, 0x8c, 0xa7, 0xe6, 0xa3, 0x80,
  0xe6, 0x9f, 0xac, 0xe7, 0xa2, 0xb1, 0xe7, 0xa1, 0xb7, 0xe6, 0x8b, 0xa3,
  0xe6, 0x8d, 0xa1, 0xe7, 0xae, 0x80, 0xe4, 0xbf, 0xad, 0xe5, 0x89, 0xaa,
  0xe5, 0x87, 0x8f, 0xe8, 0x8d, 0x90, 0xe6, 0xa7, 0x9b, 0xe9, 0x89, 0xb4,
  0xe8, 0xb7, 0xb5, 0xe8, 0xb4, 0xb1, 0xe8, 0xa7, 0x81, 0xe9, 0x94, 0xae,
  0xe7, 0xae, 0xad, 0xe4, 0xbb, 0xb6
};

unsigned char gbk_row_0xbd_converted_to_utf8[] = {
  0xe7, 0xb4, 0xb7, 0xe7, 0xb4,
  0xb8, 0xe7, 0xb4, 0xb9, 0xe7, 0xb4, 0xba, 0xe7, 0xb4, 0xbb, 0xe7, 0xb4,
  0xbc, 0xe7, 0xb4, 0xbd, 0xe7, 0xb4, 0xbe, 0xe7, 0xb4, 0xbf, 0xe7, 0xb5,
  0x80, 0xe7, 0xb5, 0x81, 0xe7, 0xb5, 0x82, 0xe7, 0xb5, 0x83, 0xe7, 0xb5,
  0x84, 0xe7, 0xb5, 0x85, 0xe7, 0xb5, 0x86, 0xe7, 0xb5, 0x87, 0xe7, 0xb5,
  0x88, 0xe7, 0xb5, 0x89, 0xe7, 0xb5, 0x8a, 0xe7, 0xb5, 0x8b, 0xe7, 0xb5,
  0x8c, 0xe7, 0xb5, 0x8d, 0xe7, 0xb5, 0x8e, 0xe7, 0xb5, 0x8f, 0xe7, 0xb5,
  0x90, 0xe7, 0xb5, 0x91, 0xe7, 0xb5, 0x92, 0xe7, 0xb5, 0x93, 0xe7, 0xb5,
  0x94, 0xe7, 0xb5, 0x95, 0xe7, 0xb5, 0x96, 0xe7, 0xb5, 0x97, 0xe7, 0xb5,
  0x98, 0xe7, 0xb5, 0x99, 0xe7, 0xb5, 0x9a, 0xe7, 0xb5, 0x9b, 0xe7, 0xb5,
  0x9c, 0xe7, 0xb5, 0x9d, 0xe7, 0xb5, 0x9e, 0xe7, 0xb5, 0x9f, 0xe7, 0xb5,
  0xa0, 0xe7, 0xb5, 0xa1, 0xe7, 0xb5, 0xa2, 0xe7, 0xb5, 0xa3, 0xe7, 0xb5,
  0xa4, 0xe7, 0xb5, 0xa5, 0xe7, 0xb5, 0xa6, 0xe7, 0xb5, 0xa7, 0xe7, 0xb5,
  0xa8, 0xe7, 0xb5, 0xa9, 0xe7, 0xb5, 0xaa, 0xe7, 0xb5, 0xab, 0xe7, 0xb5,
  0xac, 0xe7, 0xb5, 0xad, 0xe7, 0xb5, 0xaf, 0xe7, 0xb5, 0xb0, 0xe7, 0xb5,
  0xb1, 0xe7, 0xb5, 0xb2, 0xe7, 0xb5, 0xb3, 0xe7, 0xb5, 0xb4, 0xe7, 0xb5,
  0xb5, 0xe7, 0xb5, 0xb6, 0xe7, 0xb5, 0xb8, 0xe7, 0xb5, 0xb9, 0xe7, 0xb5,
  0xba, 0xe7, 0xb5, 0xbb, 0xe7, 0xb5, 0xbc, 0xe7, 0xb5, 0xbd, 0xe7, 0xb5,
  0xbe, 0xe7, 0xb5, 0xbf, 0xe7, 0xb6, 0x80, 0xe7, 0xb6, 0x81, 0xe7, 0xb6,
  0x82, 0xe7, 0xb6, 0x83, 0xe7, 0xb6, 0x84, 0xe7, 0xb6, 0x85, 0xe7, 0xb6,
  0x86, 0xe7, 0xb6, 0x87, 0xe7, 0xb6, 0x88, 0xe7, 0xb6, 0x89, 0xe7, 0xb6,
  0x8a, 0xe7, 0xb6, 0x8b, 0xe7, 0xb6, 0x8c, 0xe7, 0xb6, 0x8d, 0xe7, 0xb6,
  0x8e, 0xe7, 0xb6, 0x8f, 0xe7, 0xb6, 0x90, 0xe7, 0xb6, 0x91, 0xe7, 0xb6,
  0x92, 0xe7, 0xb6, 0x93, 0xe7, 0xb6, 0x94, 0xe7, 0xb6, 0x95, 0xe7, 0xb6,
  0x96, 0xe7, 0xb6, 0x97, 0xe7, 0xb6, 0x98, 0xe5, 0x81, 0xa5, 0xe8, 0x88,
  0xb0, 0xe5, 0x89, 0x91, 0xe9, 0xa5, 0xaf, 0xe6, 0xb8, 0x90, 0xe6, 0xba,
  0x85, 0xe6, 0xb6, 0xa7, 0xe5, 0xbb, 0xba, 0xe5, 0x83, 0xb5, 0xe5, 0xa7,
  0x9c, 0xe5, 0xb0, 0x86, 0xe6, 0xb5, 0x86, 0xe6, 0xb1, 0x9f, 0xe7, 0x96,
  0x86, 0xe8, 0x92, 0x8b, 0xe6, 0xa1, 0xa8, 0xe5, 0xa5, 0x96, 0xe8, 0xae,
  0xb2, 0xe5, 0x8c, 0xa0, 0xe9, 0x85, 0xb1, 0xe9, 0x99, 0x8d, 0xe8, 0x95,
  0x89, 0xe6, 0xa4, 0x92, 0xe7, 0xa4, 0x81, 0xe7, 0x84, 0xa6, 0xe8, 0x83,
  0xb6, 0xe4, 0xba, 0xa4, 0xe9, 0x83, 0x8a, 0xe6, 0xb5, 0x87, 0xe9, 0xaa,
  0x84, 0xe5, 0xa8, 0x87, 0xe5, 0x9a, 0xbc, 0xe6, 0x90, 0x85, 0xe9, 0x93,
  0xb0, 0xe7, 0x9f, 0xab, 0xe4, 0xbe, 0xa5, 0xe8, 0x84, 0x9a, 0xe7, 0x8b,
  0xa1, 0xe8, 0xa7, 0x92, 0xe9, 0xa5, 0xba, 0xe7, 0xbc, 0xb4, 0xe7, 0xbb,
  0x9e, 0xe5, 0x89, 0xbf, 0xe6, 0x95, 0x99, 0xe9, 0x85, 0xb5, 0xe8, 0xbd,
  0xbf, 0xe8, 0xbe, 0x83, 0xe5, 0x8f, 0xab, 0xe7, 0xaa, 0x96, 0xe6, 0x8f,
  0xad, 0xe6, 0x8e, 0xa5, 0xe7, 0x9a, 0x86, 0xe7, 0xa7, 0xb8, 0xe8, 0xa1,
  0x97, 0xe9, 0x98, 0xb6, 0xe6, 0x88, 0xaa, 0xe5, 0x8a, 0xab, 0xe8, 0x8a,
  0x82, 0xe6, 0xa1, 0x94, 0xe6, 0x9d, 0xb0, 0xe6, 0x8d, 0xb7, 0xe7, 0x9d,
  0xab, 0xe7, 0xab, 0xad, 0xe6, 0xb4, 0x81, 0xe7, 0xbb, 0x93, 0xe8, 0xa7,
  0xa3, 0xe5, 0xa7, 0x90, 0xe6, 0x88, 0x92, 0xe8, 0x97, 0x89, 0xe8, 0x8a,
  0xa5, 0xe7, 0x95, 0x8c, 0xe5, 0x80, 0x9f, 0xe4, 0xbb, 0x8b, 0xe7, 0x96,
  0xa5, 0xe8, 0xaf, 0xab, 0xe5, 0xb1, 0x8a, 0xe5, 0xb7, 0xbe, 0xe7, 0xad,
  0x8b, 0xe6, 0x96, 0xa4, 0xe9, 0x87, 0x91, 0xe4, 0xbb, 0x8a, 0xe6, 0xb4,
  0xa5, 0xe8, 0xa5, 0x9f, 0xe7, 0xb4, 0xa7, 0xe9, 0x94, 0xa6, 0xe4, 0xbb,
  0x85, 0xe8, 0xb0, 0xa8, 0xe8, 0xbf, 0x9b, 0xe9, 0x9d, 0xb3, 0xe6, 0x99,
  0x8b, 0xe7, 0xa6, 0x81, 0xe8, 0xbf, 0x91, 0xe7, 0x83, 0xac, 0xe6, 0xb5,
  0xb8
};

unsigned char gbk_row_0xbe_converted_to_utf8[] = {
  0xe7, 0xb6, 0x99, 0xe7, 0xb6, 0x9a, 0xe7, 0xb6, 0x9b, 0xe7,
  0xb6, 0x9c, 0xe7, 0xb6, 0x9d, 0xe7, 0xb6, 0x9e, 0xe7, 0xb6, 0x9f, 0xe7,
  0xb6, 0xa0, 0xe7, 0xb6, 0xa1, 0xe7, 0xb6, 0xa2, 0xe7, 0xb6, 0xa3, 0xe7,
  0xb6, 0xa4, 0xe7, 0xb6, 0xa5, 0xe7, 0xb6, 0xa7, 0xe7, 0xb6, 0xa8, 0xe7,
  0xb6, 0xa9, 0xe7, 0xb6, 0xaa, 0xe7, 0xb6, 0xab, 0xe7, 0xb6, 0xac, 0xe7,
  0xb6, 0xad, 0xe7, 0xb6, 0xaf, 0xe7, 0xb6, 0xb0, 0xe7, 0xb6, 0xb1, 0xe7,
  0xb6, 0xb2, 0xe7, 0xb6, 0xb3, 0xe7, 0xb6, 0xb4, 0xe7, 0xb6, 0xb5, 0xe7,
  0xb6, 0xb6, 0xe7, 0xb6, 0xb7, 0xe7, 0xb6, 0xb8, 0xe7, 0xb6, 0xb9, 0xe7,
  0xb6, 0xba, 0xe7, 0xb6, 0xbb, 0xe7, 0xb6, 0xbc, 0xe7, 0xb6, 0xbd, 0xe7,
  0xb6, 0xbe, 0xe7, 0xb6, 0xbf, 0xe7, 0xb7, 0x80, 0xe7, 0xb7, 0x81, 0xe7,
  0xb7, 0x82, 0xe7, 0xb7, 0x83, 0xe7, 0xb7, 0x84, 0xe7, 0xb7, 0x85, 0xe7,
  0xb7, 0x86, 0xe7, 0xb7, 0x87, 0xe7, 0xb7, 0x88, 0xe7, 0xb7, 0x89, 0xe7,
  0xb7, 0x8a, 0xe7, 0xb7, 0x8b, 0xe7, 0xb7, 0x8c, 0xe7, 0xb7, 0x8d, 0xe7,
  0xb7, 0x8e, 0xe7, 0xb7, 0x8f, 0xe7, 0xb7, 0x90, 0xe7, 0xb7, 0x91, 0xe7,
  0xb7, 0x92, 0xe7, 0xb7, 0x93, 0xe7, 0xb7, 0x94, 0xe7, 0xb7, 0x95, 0xe7,
  0xb7, 0x96, 0xe7, 0xb7, 0x97, 0xe7, 0xb7, 0x98, 0xe7, 0xb7, 0x99, 0xe7,
  0xb7, 0x9a, 0xe7, 0xb7, 0x9b, 0xe7, 0xb7, 0x9c, 0xe7, 0xb7, 0x9d, 0xe7,
  0xb7, 0x9e, 0xe7, 0xb7, 0x9f, 0xe7, 0xb7, 0xa0, 0xe7, 0xb7, 0xa1, 0xe7,
  0xb7, 0xa2, 0xe7, 0xb7, 0xa3, 0xe7, 0xb7, 0xa4, 0xe7, 0xb7, 0xa5, 0xe7,
  0xb7, 0xa6, 0xe7, 0xb7, 0xa7, 0xe7, 0xb7, 0xa8, 0xe7, 0xb7, 0xa9, 0xe7,
  0xb7, 0xaa, 0xe7, 0xb7, 0xab, 0xe7, 0xb7, 0xac, 0xe7, 0xb7, 0xad, 0xe7,
  0xb7, 0xae, 0xe7, 0xb7, 0xaf, 0xe7, 0xb7, 0xb0, 0xe7, 0xb7, 0xb1, 0xe7,
  0xb7, 0xb2, 0xe7, 0xb7, 0xb3, 0xe7, 0xb7, 0xb4, 0xe7, 0xb7, 0xb5, 0xe7,
  0xb7, 0xb6, 0xe7, 0xb7, 0xb7, 0xe7, 0xb7, 0xb8, 0xe7, 0xb7, 0xb9, 0xe7,
  0xb7, 0xba, 0xe5, 0xb0, 0xbd, 0xe5, 0x8a, 0xb2, 0xe8, 0x8d, 0x86, 0xe5,
  0x85, 0xa2, 0xe8, 0x8c, 0x8e, 0xe7, 0x9d, 0x9b, 0xe6, 0x99, 0xb6, 0xe9,
  0xb2, 0xb8, 0xe4, 0xba, 0xac, 0xe6, 0x83, 0x8a, 0xe7, 0xb2, 0xbe, 0xe7,
  0xb2, 0xb3, 0xe7, 0xbb, 0x8f, 0xe4, 0xba, 0x95, 0xe8, 0xad, 0xa6, 0xe6,
  0x99, 0xaf, 0xe9, 0xa2, 0x88, 0xe9, 0x9d, 0x99, 0xe5, 0xa2, 0x83, 0xe6,
  0x95, 0xac, 0xe9, 0x95, 0x9c, 0xe5, 0xbe, 0x84, 0xe7, 0x97, 0x89, 0xe9,
  0x9d, 0x96, 0xe7, 0xab, 0x9f, 0xe7, 0xab, 0x9e, 0xe5, 0x87, 0x80, 0xe7,
  0x82, 0xaf, 0xe7, 0xaa, 0x98, 0xe6, 0x8f, 0xaa, 0xe7, 0xa9, 0xb6, 0xe7,
  0xba, 0xa0, 0xe7, 0x8e, 0x96, 0xe9, 0x9f, 0xad, 0xe4, 0xb9, 0x85, 0xe7,
  0x81, 0xb8, 0xe4, 0xb9, 0x9d, 0xe9, 0x85, 0x92, 0xe5, 0x8e, 0xa9, 0xe6,
  0x95, 0x91, 0xe6, 0x97, 0xa7, 0xe8, 0x87, 0xbc, 0xe8, 0x88, 0x85, 0xe5,
  0x92, 0x8e, 0xe5, 0xb0, 0xb1, 0xe7, 0x96, 0x9a, 0xe9, 0x9e, 0xa0, 0xe6,
  0x8b, 0x98, 0xe7, 0x8b, 0x99, 0xe7, 0x96, 0xbd, 0xe5, 0xb1, 0x85, 0xe9,
  0xa9, 0xb9, 0xe8, 0x8f, 0x8a, 0xe5, 0xb1, 0x80, 0xe5, 0x92, 0x80, 0xe7,
  0x9f, 0xa9, 0xe4, 0xb8, 0xbe, 0xe6, 0xb2, 0xae, 0xe8, 0x81, 0x9a, 0xe6,
  0x8b, 0x92, 0xe6, 0x8d, 0xae, 0xe5, 0xb7, 0xa8, 0xe5, 0x85, 0xb7, 0xe8,
  0xb7, 0x9d, 0xe8, 0xb8, 0x9e, 0xe9, 0x94, 0xaf, 0xe4, 0xbf, 0xb1, 0xe5,
  0x8f, 0xa5, 0xe6, 0x83, 0xa7, 0xe7, 0x82, 0xac, 0xe5, 0x89, 0xa7, 0xe6,
  0x8d, 0x90, 0xe9, 0xb9, 0x83, 0xe5, 0xa8, 0x9f, 0xe5, 0x80, 0xa6, 0xe7,
  0x9c, 0xb7, 0xe5, 0x8d, 0xb7, 0xe7, 0xbb, 0xa2, 0xe6, 0x92, 0x85, 0xe6,
  0x94, 0xab, 0xe6, 0x8a, 0x89, 0xe6, 0x8e, 0x98, 0xe5, 0x80, 0x94, 0xe7,
  0x88, 0xb5, 0xe8, 0xa7, 0x89, 0xe5, 0x86, 0xb3, 0xe8, 0xaf, 0x80, 0xe7,
  0xbb, 0x9d, 0xe5, 0x9d, 0x87, 0xe8, 0x8f, 0x8c, 0xe9, 0x92, 0xa7, 0xe5,
  0x86, 0x9b, 0xe5, 0x90, 0x9b, 0xe5, 0xb3, 0xbb
};

unsigned char gbk_row_0xbf_converted_to_utf8[] = {
  0xe7, 0xb7, 0xbb,
  0xe7, 0xb7, 0xbc, 0xe7, 0xb7, 0xbd, 0xe7, 0xb7, 0xbe, 0xe7, 0xb7, 0xbf,
  0xe7, 0xb8, 0x80, 0xe7, 0xb8, 0x81, 0xe7, 0xb8, 0x82, 0xe7, 0xb8, 0x83,
  0xe7, 0xb8, 0x84, 0xe7, 0xb8, 0x85, 0xe7, 0xb8, 0x86, 0xe7, 0xb8, 0x87,
  0xe7, 0xb8, 0x88, 0xe7, 0xb8, 0x89, 0xe7, 0xb8, 0x8a, 0xe7, 0xb8, 0x8b,
  0xe7, 0xb8, 0x8c, 0xe7, 0xb8, 0x8d, 0xe7, 0xb8, 0x8e, 0xe7, 0xb8, 0x8f,
  0xe7, 0xb8, 0x90, 0xe7, 0xb8, 0x91, 0xe7, 0xb8, 0x92, 0xe7, 0xb8, 0x93,
  0xe7, 0xb8, 0x94, 0xe7, 0xb8, 0x95, 0xe7, 0xb8, 0x96, 0xe7, 0xb8, 0x97,
  0xe7, 0xb8, 0x98, 0xe7, 0xb8, 0x99, 0xe7, 0xb8, 0x9a, 0xe7, 0xb8, 0x9b,
  0xe7, 0xb8, 0x9c, 0xe7, 0xb8, 0x9d, 0xe7, 0xb8, 0x9e, 0xe7, 0xb8, 0x9f,
  0xe7, 0xb8, 0xa0, 0xe7, 0xb8, 0xa1, 0xe7, 0xb8, 0xa2, 0xe7, 0xb8, 0xa3,
  0xe7, 0xb8, 0xa4, 0xe7, 0xb8, 0xa5, 0xe7, 0xb8, 0xa6, 0xe7, 0xb8, 0xa7,
  0xe7, 0xb8, 0xa8, 0xe7, 0xb8, 0xa9, 0xe7, 0xb8, 0xaa, 0xe7, 0xb8, 0xab,
  0xe7, 0xb8, 0xac, 0xe7, 0xb8, 0xad, 0xe7, 0xb8, 0xae, 0xe7, 0xb8, 0xaf,
  0xe7, 0xb8, 0xb0, 0xe7, 0xb8, 0xb1, 0xe7, 0xb8, 0xb2, 0xe7, 0xb8, 0xb3,
  0xe7, 0xb8, 0xb4, 0xe7, 0xb8, 0xb5, 0xe7, 0xb8, 0xb6, 0xe7, 0xb8, 0xb7,
  0xe7, 0xb8, 0xb8, 0xe7, 0xb8, 0xb9, 0xe7, 0xb8, 0xba, 0xe7, 0xb8, 0xbc,
  0xe7, 0xb8, 0xbd, 0xe7, 0xb8, 0xbe, 0xe7, 0xb8, 0xbf, 0xe7, 0xb9, 0x80,
  0xe7, 0xb9, 0x82, 0xe7, 0xb9, 0x83, 0xe7, 0xb9, 0x84, 0xe7, 0xb9, 0x85,
  0xe7, 0xb9, 0x86, 0xe7, 0xb9, 0x88, 0xe7, 0xb9, 0x89, 0xe7, 0xb9, 0x8a,
  0xe7, 0xb9, 0x8b, 0xe7, 0xb9, 0x8c, 0xe7, 0xb9, 0x8d, 0xe7, 0xb9, 0x8e,
  0xe7, 0xb9, 0x8f, 0xe7, 0xb9, 0x90, 0xe7, 0xb9, 0x91, 0xe7, 0xb9, 0x92,
  0xe7, 0xb9, 0x93, 0xe7, 0xb9, 0x94, 0xe7, 0xb9, 0x95, 0xe7, 0xb9, 0x96,
  0xe7, 0xb9, 0x97, 0xe7, 0xb9, 0x98, 0xe7, 0xb9, 0x99, 0xe7, 0xb9, 0x9a,
  0xe7, 0xb9, 0x9b, 0xe7, 0xb9, 0x9c, 0xe7, 0xb9, 0x9d, 0xe4, 0xbf, 0x8a,
  0xe7, 0xab, 0xa3, 0xe6, 0xb5, 0x9a, 0xe9, 0x83, 0xa1, 0xe9, 0xaa, 0x8f,
  0xe5, 0x96, 0x80, 0xe5, 0x92, 0x96, 0xe5, 0x8d, 0xa1, 0xe5, 0x92, 0xaf,
  0xe5, 0xbc, 0x80, 0xe6, 0x8f, 0xa9, 0xe6, 0xa5, 0xb7, 0xe5, 0x87, 0xaf,
  0xe6, 0x85, 0xa8, 0xe5, 0x88, 0x8a, 0xe5, 0xa0, 0xaa, 0xe5, 0x8b, 0x98,
  0xe5, 0x9d, 0x8e, 0xe7, 0xa0, 0x8d, 0xe7, 0x9c, 0x8b, 0xe5, 0xba, 0xb7,
  0xe6, 0x85, 0xb7, 0xe7, 0xb3, 0xa0, 0xe6, 0x89, 0x9b, 0xe6, 0x8a, 0x97,
  0xe4, 0xba, 0xa2, 0xe7, 0x82, 0x95, 0xe8, 0x80, 0x83, 0xe6, 0x8b, 0xb7,
  0xe7, 0x83, 0xa4, 0xe9, 0x9d, 0xa0, 0xe5, 0x9d, 0xb7, 0xe8, 0x8b, 0x9b,
  0xe6, 0x9f, 0xaf, 0xe6, 0xa3, 0xb5, 0xe7, 0xa3, 0x95, 0xe9, 0xa2, 0x97,
  0xe7, 0xa7, 0x91, 0xe5, 0xa3, 0xb3, 0xe5, 0x92, 0xb3, 0xe5, 0x8f, 0xaf,
  0xe6, 0xb8, 0xb4, 0xe5, 0x85, 0x8b, 0xe5, 0x88, 0xbb, 0xe5, 0xae, 0xa2,
  0xe8, 0xaf, 0xbe, 0xe8, 0x82, 0xaf, 0xe5, 0x95, 0x83, 0xe5, 0x9e, 0xa6,
  0xe6, 0x81, 0xb3, 0xe5, 0x9d, 0x91, 0xe5, 0x90, 0xad, 0xe7, 0xa9, 0xba,
  0xe6, 0x81, 0x90, 0xe5, 0xad, 0x94, 0xe6, 0x8e, 0xa7, 0xe6, 0x8a, 0xa0,
  0xe5, 0x8f, 0xa3, 0xe6, 0x89, 0xa3, 0xe5, 0xaf, 0x87, 0xe6, 0x9e, 0xaf,
  0xe5, 0x93, 0xad, 0xe7, 0xaa, 0x9f, 0xe8, 0x8b, 0xa6, 0xe9, 0x85, 0xb7,
  0xe5, 0xba, 0x93, 0xe8, 0xa3, 0xa4, 0xe5, 0xa4, 0xb8, 0xe5, 0x9e, 0xae,
  0xe6, 0x8c, 0x8e, 0xe8, 0xb7, 0xa8, 0xe8, 0x83, 0xaf, 0xe5, 0x9d, 0x97,
  0xe7, 0xad, 0xb7, 0xe4, 0xbe, 0xa9, 0xe5, 0xbf, 0xab, 0xe5, 0xae, 0xbd,
  0xe6, 0xac, 0xbe, 0xe5, 0x8c, 0xa1, 0xe7, 0xad, 0x90, 0xe7, 0x8b, 0x82,
  0xe6, 0xa1, 0x86, 0xe7, 0x9f, 0xbf, 0xe7, 0x9c, 0xb6, 0xe6, 0x97, 0xb7,
  0xe5, 0x86, 0xb5, 0xe4, 0xba, 0x8f, 0xe7, 0x9b, 0x94, 0xe5, 0xb2, 0xbf,
  0xe7, 0xaa, 0xa5, 0xe8, 0x91, 0xb5, 0xe5, 0xa5, 0x8e, 0xe9, 0xad, 0x81,
  0xe5, 0x82, 0x80
};

unsigned char gbk_row_0xc0_converted_to_utf8[] = {
  0xe7, 0xb9, 0x9e, 0xe7, 0xb9, 0x9f, 0xe7, 0xb9,
  0xa0, 0xe7, 0xb9, 0xa1, 0xe7, 0xb9, 0xa2, 0xe7, 0xb9, 0xa3, 0xe7, 0xb9,
  0xa4, 0xe7, 0xb9, 0xa5, 0xe7, 0xb9, 0xa6, 0xe7, 0xb9, 0xa7, 0xe7, 0xb9,
  0xa8, 0xe7, 0xb9, 0xa9, 0xe7, 0xb9, 0xaa, 0xe7, 0xb9, 0xab, 0xe7, 0xb9,
  0xac, 0xe7, 0xb9, 0xad, 0xe7, 0xb9, 0xae, 0xe7, 0xb9, 0xaf, 0xe7, 0xb9,
  0xb0, 0xe7, 0xb9, 0xb1, 0xe7, 0xb9, 0xb2, 0xe7, 0xb9, 0xb3, 0xe7, 0xb9,
  0xb4, 0xe7, 0xb9, 0xb5, 0xe7, 0xb9, 0xb6, 0xe7, 0xb9, 0xb7, 0xe7, 0xb9,
  0xb8, 0xe7, 0xb9, 0xb9, 0xe7, 0xb9, 0xba, 0xe7, 0xb9, 0xbb, 0xe7, 0xb9,
  0xbc, 0xe7, 0xb9, 0xbd, 0xe7, 0xb9, 0xbe, 0xe7, 0xb9, 0xbf, 0xe7, 0xba,
  0x80, 0xe7, 0xba, 0x81, 0xe7, 0xba, 0x83, 0xe7, 0xba, 0x84, 0xe7, 0xba,
  0x85, 0xe7, 0xba, 0x86, 0xe7, 0xba, 0x87, 0xe7, 0xba, 0x88, 0xe7, 0xba,
  0x89, 0xe7, 0xba, 0x8a, 0xe7, 0xba, 0x8b, 0xe7, 0xba, 0x8c, 0xe7, 0xba,
  0x8d, 0xe7, 0xba, 0x8e, 0xe7, 0xba, 0x8f, 0xe7, 0xba, 0x90, 0xe7, 0xba,
  0x91, 0xe7, 0xba, 0x92, 0xe7, 0xba, 0x93, 0xe7, 0xba, 0x94, 0xe7, 0xba,
  0x95, 0xe7, 0xba, 0x96, 0xe7, 0xba, 0x97, 0xe7, 0xba, 0x98, 0xe7, 0xba,
  0x99, 0xe7, 0xba, 0x9a, 0xe7, 0xba, 0x9c, 0xe7, 0xba, 0x9d, 0xe7, 0xba,
  0x9e, 0xe7, 0xba, 0xae, 0xe7, 0xba, 0xb4, 0xe7, 0xba, 0xbb, 0xe7, 0xba,
  0xbc, 0xe7, 0xbb, 0x96, 0xe7, 0xbb, 0xa4, 0xe7, 0xbb, 0xac, 0xe7, 0xbb,
  0xb9, 0xe7, 0xbc, 0x8a, 0xe7, 0xbc, 0x90, 0xe7, 0xbc, 0x9e, 0xe7, 0xbc,
  0xb7, 0xe7, 0xbc, 0xb9, 0xe7, 0xbc, 0xbb, 0xe7, 0xbc, 0xbc, 0xe7, 0xbc,
  0xbd, 0xe7, 0xbc, 0xbe, 0xe7, 0xbc, 0xbf, 0xe7, 0xbd, 0x80, 0xe7, 0xbd,
  0x81, 0xe7, 0xbd, 0x83, 0xe7, 0xbd, 0x86, 0xe7, 0xbd, 0x87, 0xe7, 0xbd,
  0x88, 0xe7, 0xbd, 0x89, 0xe7, 0xbd, 0x8a, 0xe7, 0xbd, 0x8b, 0xe7, 0xbd,
  0x8c, 0xe7, 0xbd, 0x8d, 0xe7, 0xbd, 0x8e, 0xe7, 0xbd, 0x8f, 0xe7, 0xbd,
  0x92, 0xe7, 0xbd, 0x93, 0xe9, 0xa6, 0x88, 0xe6, 0x84, 0xa7, 0xe6, 0xba,
  0x83, 0xe5, 0x9d, 0xa4, 0xe6, 0x98, 0x86, 0xe6, 0x8d, 0x86, 0xe5, 0x9b,
  0xb0, 0xe6, 0x8b, 0xac, 0xe6, 0x89, 0xa9, 0xe5, 0xbb, 0x93, 0xe9, 0x98,
  0x94, 0xe5, 0x9e, 0x83, 0xe6, 0x8b, 0x89, 0xe5, 0x96, 0x87, 0xe8, 0x9c,
  0xa1, 0xe8, 0x85, 0x8a, 0xe8, 0xbe, 0xa3, 0xe5, 0x95, 0xa6, 0xe8, 0x8e,
  0xb1, 0xe6, 0x9d, 0xa5, 0xe8, 0xb5, 0x96, 0xe8, 0x93, 0x9d, 0xe5, 0xa9,
  0xaa, 0xe6, 0xa0, 0x8f, 0xe6, 0x8b, 0xa6, 0xe7, 0xaf, 0xae, 0xe9, 0x98,
  0x91, 0xe5, 0x85, 0xb0, 0xe6, 0xbe, 0x9c, 0xe8, 0xb0, 0xb0, 0xe6, 0x8f,
  0xbd, 0xe8, 0xa7, 0x88, 0xe6, 0x87, 0x92, 0xe7, 0xbc, 0x86, 0xe7, 0x83,
  0x82, 0xe6, 0xbb, 0xa5, 0xe7, 0x90, 0x85, 0xe6, 0xa6, 0x94, 0xe7, 0x8b,
  0xbc, 0xe5, 0xbb, 0x8a, 0xe9, 0x83, 0x8e, 0xe6, 0x9c, 0x97, 0xe6, 0xb5,
  0xaa, 0xe6, 0x8d, 0x9e, 0xe5, 0x8a, 0xb3, 0xe7, 0x89, 0xa2, 0xe8, 0x80,
  0x81, 0xe4, 0xbd, 0xac, 0xe5, 0xa7, 0xa5, 0xe9, 0x85, 0xaa, 0xe7, 0x83,
  0x99, 0xe6, 0xb6, 0x9d, 0xe5, 0x8b, 0x92, 0xe4, 0xb9, 0x90, 0xe9, 0x9b,
  0xb7, 0xe9, 0x95, 0xad, 0xe8, 0x95, 0xbe, 0xe7, 0xa3, 0x8a, 0xe7, 0xb4,
  0xaf, 0xe5, 0x84, 0xa1, 0xe5, 0x9e, 0x92, 0xe6, 0x93, 0x82, 0xe8, 0x82,
  0x8b, 0xe7, 0xb1, 0xbb, 0xe6, 0xb3, 0xaa, 0xe6, 0xa3, 0xb1, 0xe6, 0xa5,
  0x9e, 0xe5, 0x86, 0xb7, 0xe5, 0x8e, 0x98, 0xe6, 0xa2, 0xa8, 0xe7, 0x8a,
  0x81, 0xe9, 0xbb, 0x8e, 0xe7, 0xaf, 0xb1, 0xe7, 0x8b, 0xb8, 0xe7, 0xa6,
  0xbb, 0xe6, 0xbc, 0x93, 0xe7, 0x90, 0x86, 0xe6, 0x9d, 0x8e, 0xe9, 0x87,
  0x8c, 0xe9, 0xb2, 0xa4, 0xe7, 0xa4, 0xbc, 0xe8, 0x8e, 0x89, 0xe8, 0x8d,
  0x94, 0xe5, 0x90, 0x8f, 0xe6, 0xa0, 0x97, 0xe4, 0xb8, 0xbd, 0xe5, 0x8e,
  0x89, 0xe5, 0x8a, 0xb1, 0xe7, 0xa0, 0xbe, 0xe5, 0x8e, 0x86, 0xe5, 0x88,
  0xa9, 0xe5, 0x82, 0x88, 0xe4, 0xbe, 0x8b, 0xe4, 0xbf, 0x90
};

unsigned char gbk_row_0xc1_converted_to_utf8[] = {
  0xe7,
  0xbd, 0x96, 0xe7, 0xbd, 0x99, 0xe7, 0xbd, 0x9b, 0xe7, 0xbd, 0x9c, 0xe7,
  0xbd, 0x9d, 0xe7, 0xbd, 0x9e, 0xe7, 0xbd, 0xa0, 0xe7, 0xbd, 0xa3, 0xe7,
  0xbd, 0xa4, 0xe7, 0xbd, 0xa5, 0xe7, 0xbd, 0xa6, 0xe7, 0xbd, 0xa7, 0xe7,
  0xbd, 0xab, 0xe7, 0xbd, 0xac, 0xe7, 0xbd, 0xad, 0xe7, 0xbd, 0xaf, 0xe7,
  0xbd, 0xb0, 0xe7, 0xbd, 0xb3, 0xe7, 0xbd, 0xb5, 0xe7, 0xbd, 0xb6, 0xe7,
  0xbd, 0xb7, 0xe7, 0xbd, 0xb8, 0xe7, 0xbd, 0xba, 0xe7, 0xbd, 0xbb, 0xe7,
  0xbd, 0xbc, 0xe7, 0xbd, 0xbd, 0xe7, 0xbd, 0xbf, 0xe7, 0xbe, 0x80, 0xe7,
  0xbe, 0x82, 0xe7, 0xbe, 0x83, 0xe7, 0xbe, 0x84, 0xe7, 0xbe, 0x85, 0xe7,
  0xbe, 0x86, 0xe7, 0xbe, 0x87, 0xe7, 0xbe, 0x88, 0xe7, 0xbe, 0x89, 0xe7,
  0xbe, 0x8b, 0xe7, 0xbe, 0x8d, 0xe7, 0xbe, 0x8f, 0xe7, 0xbe, 0x90, 0xe7,
  0xbe, 0x91, 0xe7, 0xbe, 0x92, 0xe7, 0xbe, 0x93, 0xe7, 0xbe, 0x95, 0xe7,
  0xbe, 0x96, 0xe7, 0xbe, 0x97, 0xe7, 0xbe, 0x98, 0xe7, 0xbe, 0x99, 0xe7,
  0xbe, 0x9b, 0xe7, 0xbe, 0x9c, 0xe7, 0xbe, 0xa0, 0xe7, 0xbe, 0xa2, 0xe7,
  0xbe, 0xa3, 0xe7, 0xbe, 0xa5, 0xe7, 0xbe, 0xa6, 0xe7, 0xbe, 0xa8, 0xe7,
  0xbe, 0xa9, 0xe7, 0xbe, 0xaa, 0xe7, 0xbe, 0xab, 0xe7, 0xbe, 0xac, 0xe7,
  0xbe, 0xad, 0xe7, 0xbe, 0xae, 0xe7, 0xbe, 0xb1, 0xe7, 0xbe, 0xb3, 0xe7,
  0xbe, 0xb4, 0xe7, 0xbe, 0xb5, 0xe7, 0xbe, 0xb6, 0xe7, 0xbe, 0xb7, 0xe7,
  0xbe, 0xba, 0xe7, 0xbe, 0xbb, 0xe7, 0xbe, 0xbe, 0xe7, 0xbf, 0x80, 0xe7,
  0xbf, 0x82, 0xe7, 0xbf, 0x83, 0xe7, 0xbf, 0x84, 0xe7, 0xbf, 0x86, 0xe7,
  0xbf, 0x87, 0xe7, 0xbf, 0x88, 0xe7, 0xbf, 0x89, 0xe7, 0xbf, 0x8b, 0xe7,
  0xbf, 0x8d, 0xe7, 0xbf, 0x8f, 0xe7, 0xbf, 0x90, 0xe7, 0xbf, 0x91, 0xe7,
  0xbf, 0x92, 0xe7, 0xbf, 0x93, 0xe7, 0xbf, 0x96, 0xe7, 0xbf, 0x97, 0xe7,
  0xbf, 0x99, 0xe7, 0xbf, 0x9a, 0xe7, 0xbf, 0x9b, 0xe7, 0xbf, 0x9c, 0xe7,
  0xbf, 0x9d, 0xe7, 0xbf, 0x9e, 0xe7, 0xbf, 0xa2, 0xe7, 0xbf, 0xa3, 0xe7,
  0x97, 0xa2, 0xe7, 0xab, 0x8b, 0xe7, 0xb2, 0x92, 0xe6, 0xb2, 0xa5, 0xe9,
  0x9a, 0xb6, 0xe5, 0x8a, 0x9b, 0xe7, 0x92, 0x83, 0xe5, 0x93, 0xa9, 0xe4,
  0xbf, 0xa9, 0xe8, 0x81, 0x94, 0xe8, 0x8e, 0xb2, 0xe8, 0xbf, 0x9e, 0xe9,
  0x95, 0xb0, 0xe5, 0xbb, 0x89, 0xe6, 0x80, 0x9c, 0xe6, 0xb6, 0x9f, 0xe5,
  0xb8, 0x98, 0xe6, 0x95, 0x9b, 0xe8, 0x84, 0xb8, 0xe9, 0x93, 0xbe, 0xe6,
  0x81, 0x8b, 0xe7, 0x82, 0xbc, 0xe7, 0xbb, 0x83, 0xe7, 0xb2, 0xae, 0xe5,
  0x87, 0x89, 0xe6, 0xa2, 0x81, 0xe7, 0xb2, 0xb1, 0xe8, 0x89, 0xaf, 0xe4,
  0xb8, 0xa4, 0xe8, 0xbe, 0x86, 0xe9, 0x87, 0x8f, 0xe6, 0x99, 0xbe, 0xe4,
  0xba, 0xae, 0xe8, 0xb0, 0x85, 0xe6, 0x92, 0xa9, 0xe8, 0x81, 0x8a, 0xe5,
  0x83, 0x9a, 0xe7, 0x96, 0x97, 0xe7, 0x87, 0x8e, 0xe5, 0xaf, 0xa5, 0xe8,
  0xbe, 0xbd, 0xe6, 0xbd, 0xa6, 0xe4, 0xba, 0x86, 0xe6, 0x92, 0x82, 0xe9,
  0x95, 0xa3, 0xe5, 0xbb, 0x96, 0xe6, 0x96, 0x99, 0xe5, 0x88, 0x97, 0xe8,
  0xa3, 0x82, 0xe7, 0x83, 0x88, 0xe5, 0x8a, 0xa3, 0xe7, 0x8c, 0x8e, 0xe7,
  0x90, 0xb3, 0xe6, 0x9e, 0x97, 0xe7, 0xa3, 0xb7, 0xe9, 0x9c, 0x96, 0xe4,
  0xb8, 0xb4, 0xe9, 0x82, 0xbb, 0xe9, 0xb3, 0x9e, 0xe6, 0xb7, 0x8b, 0xe5,
  0x87, 0x9b, 0xe8, 0xb5, 0x81, 0xe5, 0x90, 0x9d, 0xe6, 0x8b, 0x8e, 0xe7,
  0x8e, 0xb2, 0xe8, 0x8f, 0xb1, 0xe9, 0x9b, 0xb6, 0xe9, 0xbe, 0x84, 0xe9,
  0x93, 0x83, 0xe4, 0xbc, 0xb6, 0xe7, 0xbe, 0x9a, 0xe5, 0x87, 0x8c, 0xe7,
  0x81, 0xb5, 0xe9, 0x99, 0xb5, 0xe5, 0xb2, 0xad, 0xe9, 0xa2, 0x86, 0xe5,
  0x8f, 0xa6, 0xe4, 0xbb, 0xa4, 0xe6, 0xba, 0x9c, 0xe7, 0x90, 0x89, 0xe6,
  0xa6, 0xb4, 0xe7, 0xa1, 0xab, 0xe9, 0xa6, 0x8f, 0xe7, 0x95, 0x99, 0xe5,
  0x88, 0x98, 0xe7, 0x98, 0xa4, 0xe6, 0xb5, 0x81, 0xe6, 0x9f, 0xb3, 0xe5,
  0x85, 0xad, 0xe9, 0xbe, 0x99, 0xe8, 0x81, 0x8b, 0xe5, 0x92, 0x99, 0xe7,
  0xac, 0xbc, 0xe7, 0xaa, 0xbf
};

unsigned char gbk_row_0xc2_converted_to_utf8[] = {
  0xe7, 0xbf, 0xa4, 0xe7, 0xbf, 0xa7,
  0xe7, 0xbf, 0xa8, 0xe7, 0xbf, 0xaa, 0xe7, 0xbf, 0xab, 0xe7, 0xbf, 0xac,
  0xe7, 0xbf, 0xad, 0xe7, 0xbf, 0xaf, 0xe7, 0xbf, 0xb2, 0xe7, 0xbf, 0xb4,
  0xe7, 0xbf, 0xb5, 0xe7, 0xbf, 0xb6, 0xe7, 0xbf, 0xb7, 0xe7, 0xbf, 0xb8,
  0xe7, 0xbf, 0xb9, 0xe7, 0xbf, 0xba, 0xe7, 0xbf, 0xbd, 0xe7, 0xbf, 0xbe,
  0xe7, 0xbf, 0xbf, 0xe8, 0x80, 0x82, 0xe8, 0x80, 0x87, 0xe8, 0x80, 0x88,
  0xe8, 0x80, 0x89, 0xe8, 0x80, 0x8a, 0xe8, 0x80, 0x8e, 0xe8, 0x80, 0x8f,
  0xe8, 0x80, 0x91, 0xe8, 0x80, 0x93, 0xe8, 0x80, 0x9a, 0xe8, 0x80, 0x9b,
  0xe8, 0x80, 0x9d, 0xe8, 0x80, 0x9e, 0xe8, 0x80, 0x9f, 0xe8, 0x80, 0xa1,
  0xe8, 0x80, 0xa3, 0xe8, 0x80, 0xa4, 0xe8, 0x80, 0xab, 0xe8, 0x80, 0xac,
  0xe8, 0x80, 0xad, 0xe8, 0x80, 0xae, 0xe8, 0x80, 0xaf, 0xe8, 0x80, 0xb0,
  0xe8, 0x80, 0xb2, 0xe8, 0x80, 0xb4, 0xe8, 0x80, 0xb9, 0xe8, 0x80, 0xba,
  0xe8, 0x80, 0xbc, 0xe8, 0x80, 0xbe, 0xe8, 0x81, 0x80, 0xe8, 0x81, 0x81,
  0xe8, 0x81, 0x84, 0xe8, 0x81, 0x85, 0xe8, 0x81, 0x87, 0xe8, 0x81, 0x88,
  0xe8, 0x81, 0x89, 0xe8, 0x81, 0x8e, 0xe8, 0x81, 0x8f, 0xe8, 0x81, 0x90,
  0xe8, 0x81, 0x91, 0xe8, 0x81, 0x93, 0xe8, 0x81, 0x95, 0xe8, 0x81, 0x96,
  0xe8, 0x81, 0x97, 0xe8, 0x81, 0x99, 0xe8, 0x81, 0x9b, 0xe8, 0x81, 0x9c,
  0xe8, 0x81, 0x9d, 0xe8, 0x81, 0x9e, 0xe8, 0x81, 0x9f, 0xe8, 0x81, 0xa0,
  0xe8, 0x81, 0xa1, 0xe8, 0x81, 0xa2, 0xe8, 0x81, 0xa3, 0xe8, 0x81, 0xa4,
  0xe8, 0x81, 0xa5, 0xe8, 0x81, 0xa6, 0xe8, 0x81, 0xa7, 0xe8, 0x81, 0xa8,
  0xe8, 0x81, 0xab, 0xe8, 0x81, 0xac, 0xe8, 0x81, 0xad, 0xe8, 0x81, 0xae,
  0xe8, 0x81, 0xaf, 0xe8, 0x81, 0xb0, 0xe8, 0x81, 0xb2, 0xe8, 0x81, 0xb3,
  0xe8, 0x81, 0xb4, 0xe8, 0x81, 0xb5, 0xe8, 0x81, 0xb6, 0xe8, 0x81, 0xb7,
  0xe8, 0x81, 0xb8, 0xe8, 0x81, 0xb9, 0xe8, 0x81, 0xba, 0xe8, 0x81, 0xbb,
  0xe8, 0x81, 0xbc, 0xe8, 0x81, 0xbd, 0xe9, 0x9a, 0x86, 0xe5, 0x9e, 0x84,
  0xe6, 0x8b, 0xa2, 0xe9, 0x99, 0x87, 0xe6, 0xa5, 0xbc, 0xe5, 0xa8, 0x84,
  0xe6, 0x90, 0x82, 0xe7, 0xaf, 0x93, 0xe6, 0xbc, 0x8f, 0xe9, 0x99, 0x8b,
  0xe8, 0x8a, 0xa6, 0xe5, 0x8d, 0xa2, 0xe9, 0xa2, 0x85, 0xe5, 0xba, 0x90,
  0xe7, 0x82, 0x89, 0xe6, 0x8e, 0xb3, 0xe5, 0x8d, 0xa4, 0xe8, 0x99, 0x8f,
  0xe9, 0xb2, 0x81, 0xe9, 0xba, 0x93, 0xe7, 0xa2, 0x8c, 0xe9, 0x9c, 0xb2,
  0xe8, 0xb7, 0xaf, 0xe8, 0xb5, 0x82, 0xe9, 0xb9, 0xbf, 0xe6, 0xbd, 0x9e,
  0xe7, 0xa6, 0x84, 0xe5, 0xbd, 0x95, 0xe9, 0x99, 0x86, 0xe6, 0x88, 0xae,
  0xe9, 0xa9, 0xb4, 0xe5, 0x90, 0x95, 0xe9, 0x93, 0x9d, 0xe4, 0xbe, 0xa3,
  0xe6, 0x97, 0x85, 0xe5, 0xb1, 0xa5, 0xe5, 0xb1, 0xa1, 0xe7, 0xbc, 0x95,
  0xe8, 0x99, 0x91, 0xe6, 0xb0, 0xaf, 0xe5, 0xbe, 0x8b, 0xe7, 0x8e, 0x87,
  0xe6, 0xbb, 0xa4, 0xe7, 0xbb, 0xbf, 0xe5, 0xb3, 0xa6, 0xe6, 0x8c, 0x9b,
  0xe5, 0xad, 0xaa, 0xe6, 0xbb, 0xa6, 0xe5, 0x8d, 0xb5, 0xe4, 0xb9, 0xb1,
  0xe6, 0x8e, 0xa0, 0xe7, 0x95, 0xa5, 0xe6, 0x8a, 0xa1, 0xe8, 0xbd, 0xae,
  0xe4, 0xbc, 0xa6, 0xe4, 0xbb, 0x91, 0xe6, 0xb2, 0xa6, 0xe7, 0xba, 0xb6,
  0xe8, 0xae, 0xba, 0xe8, 0x90, 0x9d, 0xe8, 0x9e, 0xba, 0xe7, 0xbd, 0x97,
  0xe9, 0x80, 0xbb, 0xe9, 0x94, 0xa3, 0xe7, 0xae, 0xa9, 0xe9, 0xaa, 0xa1,
  0xe8, 0xa3, 0xb8, 0xe8, 0x90, 0xbd, 0xe6, 0xb4, 0x9b, 0xe9, 0xaa, 0x86,
  0xe7, 0xbb, 0x9c, 0xe5, 0xa6, 0x88, 0xe9, 0xba, 0xbb, 0xe7, 0x8e, 0x9b,
  0xe7, 0xa0, 0x81, 0xe8, 0x9a, 0x82, 0xe9, 0xa9, 0xac, 0xe9, 0xaa, 0x82,
  0xe5, 0x98, 0x9b, 0xe5, 0x90, 0x97, 0xe5, 0x9f, 0x8b, 0xe4, 0xb9, 0xb0,
  0xe9, 0xba, 0xa6, 0xe5, 0x8d, 0x96, 0xe8, 0xbf, 0x88, 0xe8, 0x84, 0x89,
  0xe7, 0x9e, 0x92, 0xe9, 0xa6, 0x92, 0xe8, 0x9b, 0xae, 0xe6, 0xbb, 0xa1,
  0xe8, 0x94, 0x93, 0xe6, 0x9b, 0xbc, 0xe6, 0x85, 0xa2, 0xe6, 0xbc, 0xab,
};

unsigned char gbk_row_0xc3_converted_to_utf8[] = {
        0xe8, 0x81, 0xbe, 0xe8, 0x82, 0x81, 0xe8, 0x82, 0x82, 0xe8, 0x82,
  0x85, 0xe8, 0x82, 0x88, 0xe8, 0x82, 0x8a, 0xe8, 0x82, 0x8d, 0xe8, 0x82,
  0x8e, 0xe8, 0x82, 0x8f, 0xe8, 0x82, 0x90, 0xe8, 0x82, 0x91, 0xe8, 0x82,
  0x92, 0xe8, 0x82, 0x94, 0xe8, 0x82, 0x95, 0xe8, 0x82, 0x97, 0xe8, 0x82,
  0x99, 0xe8, 0x82, 0x9e, 0xe8, 0x82, 0xa3, 0xe8, 0x82, 0xa6, 0xe8, 0x82,
  0xa7, 0xe8, 0x82, 0xa8, 0xe8, 0x82, 0xac, 0xe8, 0x82, 0xb0, 0xe8, 0x82,
  0xb3, 0xe8, 0x82, 0xb5, 0xe8, 0x82, 0xb6, 0xe8, 0x82, 0xb8, 0xe8, 0x82,
  0xb9, 0xe8, 0x82, 0xbb, 0xe8, 0x83, 0x85, 0xe8, 0x83, 0x87, 0xe8, 0x83,
  0x88, 0xe8, 0x83, 0x89, 0xe8, 0x83, 0x8a, 0xe8, 0x83, 0x8b, 0xe8, 0x83,
  0x8f, 0xe8, 0x83, 0x90, 0xe8, 0x83, 0x91, 0xe8, 0x83, 0x92, 0xe8, 0x83,
  0x93, 0xe8, 0x83, 0x94, 0xe8, 0x83, 0x95, 0xe8, 0x83, 0x98, 0xe8, 0x83,
  0x9f, 0xe8, 0x83, 0xa0, 0xe8, 0x83, 0xa2, 0xe8, 0x83, 0xa3, 0xe8, 0x83,
  0xa6, 0xe8, 0x83, 0xae, 0xe8, 0x83, 0xb5, 0xe8, 0x83, 0xb7, 0xe8, 0x83,
  0xb9, 0xe8, 0x83, 0xbb, 0xe8, 0x83, 0xbe, 0xe8, 0x83, 0xbf, 0xe8, 0x84,
  0x80, 0xe8, 0x84, 0x81, 0xe8, 0x84, 0x83, 0xe8, 0x84, 0x84, 0xe8, 0x84,
  0x85, 0xe8, 0x84, 0x87, 0xe8, 0x84, 0x88, 0xe8, 0x84, 0x8b, 0xe8, 0x84,
  0x8c, 0xe8, 0x84, 0x95, 0xe8, 0x84, 0x97, 0xe8, 0x84, 0x99, 0xe8, 0x84,
  0x9b, 0xe8, 0x84, 0x9c, 0xe8, 0x84, 0x9d, 0xe8, 0x84, 0x9f, 0xe8, 0x84,
  0xa0, 0xe8, 0x84, 0xa1, 0xe8, 0x84, 0xa2, 0xe8, 0x84, 0xa3, 0xe8, 0x84,
  0xa4, 0xe8, 0x84, 0xa5, 0xe8, 0x84, 0xa6, 0xe8, 0x84, 0xa7, 0xe8, 0x84,
  0xa8, 0xe8, 0x84, 0xa9, 0xe8, 0x84, 0xaa, 0xe8, 0x84, 0xab, 0xe8, 0x84,
  0xad, 0xe8, 0x84, 0xae, 0xe8, 0x84, 0xb0, 0xe8, 0x84, 0xb3, 0xe8, 0x84,
  0xb4, 0xe8, 0x84, 0xb5, 0xe8, 0x84, 0xb7, 0xe8, 0x84, 0xb9, 0xe8, 0x84,
  0xba, 0xe8, 0x84, 0xbb, 0xe8, 0x84, 0xbc, 0xe8, 0x84, 0xbd, 0xe8, 0x84,
  0xbf, 0xe8, 0xb0, 0xa9, 0xe8, 0x8a, 0x92, 0xe8, 0x8c, 0xab, 0xe7, 0x9b,
  0xb2, 0xe6, 0xb0, 0x93, 0xe5, 0xbf, 0x99, 0xe8, 0x8e, 0xbd, 0xe7, 0x8c,
  0xab, 0xe8, 0x8c, 0x85, 0xe9, 0x94, 0x9a, 0xe6, 0xaf, 0x9b, 0xe7, 0x9f,
  0x9b, 0xe9, 0x93, 0x86, 0xe5, 0x8d, 0xaf, 0xe8, 0x8c, 0x82, 0xe5, 0x86,
  0x92, 0xe5, 0xb8, 0xbd, 0xe8, 0xb2, 0x8c, 0xe8, 0xb4, 0xb8, 0xe4, 0xb9,
  0x88, 0xe7, 0x8e, 0xab, 0xe6, 0x9e, 0x9a, 0xe6, 0xa2, 0x85, 0xe9, 0x85,
  0xb6, 0xe9, 0x9c, 0x89, 0xe7, 0x85, 0xa4, 0xe6, 0xb2, 0xa1, 0xe7, 0x9c,
  0x89, 0xe5, 0xaa, 0x92, 0xe9, 0x95, 0x81, 0xe6, 0xaf, 0x8f, 0xe7, 0xbe,
  0x8e, 0xe6, 0x98, 0xa7, 0xe5, 0xaf, 0x90, 0xe5, 0xa6, 0xb9, 0xe5, 0xaa,
  0x9a, 0xe9, 0x97, 0xa8, 0xe9, 0x97, 0xb7, 0xe4, 0xbb, 0xac, 0xe8, 0x90,
  0x8c, 0xe8, 0x92, 0x99, 0xe6, 0xaa, 0xac, 0xe7, 0x9b, 0x9f, 0xe9, 0x94,
  0xb0, 0xe7, 0x8c, 0x9b, 0xe6, 0xa2, 0xa6, 0xe5, 0xad, 0x9f, 0xe7, 0x9c,
  0xaf, 0xe9, 0x86, 0x9a, 0xe9, 0x9d, 0xa1, 0xe7, 0xb3, 0x9c, 0xe8, 0xbf,
  0xb7, 0xe8, 0xb0, 0x9c, 0xe5, 0xbc, 0xa5, 0xe7, 0xb1, 0xb3, 0xe7, 0xa7,
  0x98, 0xe8, 0xa7, 0x85, 0xe6, 0xb3, 0x8c, 0xe8, 0x9c, 0x9c, 0xe5, 0xaf,
  0x86, 0xe5, 0xb9, 0x82, 0xe6, 0xa3, 0x89, 0xe7, 0x9c, 0xa0, 0xe7, 0xbb,
  0xb5, 0xe5, 0x86, 0x95, 0xe5, 0x85, 0x8d, 0xe5, 0x8b, 0x89, 0xe5, 0xa8,
  0xa9, 0xe7, 0xbc, 0x85, 0xe9, 0x9d, 0xa2, 0xe8, 0x8b, 0x97, 0xe6, 0x8f,
  0x8f, 0xe7, 0x9e, 0x84, 0xe8, 0x97, 0x90, 0xe7, 0xa7, 0x92, 0xe6, 0xb8,
  0xba, 0xe5, 0xba, 0x99, 0xe5, 0xa6, 0x99, 0xe8, 0x94, 0x91, 0xe7, 0x81,
  0xad, 0xe6, 0xb0, 0x91, 0xe6, 0x8a, 0xbf, 0xe7, 0x9a, 0xbf, 0xe6, 0x95,
  0x8f, 0xe6, 0x82, 0xaf, 0xe9, 0x97, 0xbd, 0xe6, 0x98, 0x8e, 0xe8, 0x9e,
  0x9f, 0xe9, 0xb8, 0xa3, 0xe9, 0x93, 0xad, 0xe5, 0x90, 0x8d, 0xe5, 0x91,
  0xbd, 0xe8, 0xb0, 0xac, 0xe6, 0x91, 0xb8
};

unsigned char gbk_row_0xc4_converted_to_utf8[] = {
  0xe8, 0x85, 0x80, 0xe8,
  0x85, 0x81, 0xe8, 0x85, 0x82, 0xe8, 0x85, 0x83, 0xe8, 0x85, 0x84, 0xe8,
  0x85, 0x85, 0xe8, 0x85, 0x87, 0xe8, 0x85, 0x89, 0xe8, 0x85, 0x8d, 0xe8,
  0x85, 0x8e, 0xe8, 0x85, 0x8f, 0xe8, 0x85, 0x92, 0xe8, 0x85, 0x96, 0xe8,
  0x85, 0x97, 0xe8, 0x85, 0x98, 0xe8, 0x85, 0x9b, 0xe8, 0x85, 0x9c, 0xe8,
  0x85, 0x9d, 0xe8, 0x85, 0x9e, 0xe8, 0x85, 0x9f, 0xe8, 0x85, 0xa1, 0xe8,
  0x85, 0xa2, 0xe8, 0x85, 0xa3, 0xe8, 0x85, 0xa4, 0xe8, 0x85, 0xa6, 0xe8,
  0x85, 0xa8, 0xe8, 0x85, 0xaa, 0xe8, 0x85, 0xab, 0xe8, 0x85, 0xac, 0xe8,
  0x85, 0xaf, 0xe8, 0x85, 0xb2, 0xe8, 0x85, 0xb3, 0xe8, 0x85, 0xb5, 0xe8,
  0x85, 0xb6, 0xe8, 0x85, 0xb7, 0xe8, 0x85, 0xb8, 0xe8, 0x86, 0x81, 0xe8,
  0x86, 0x83, 0xe8, 0x86, 0x84, 0xe8, 0x86, 0x85, 0xe8, 0x86, 0x86, 0xe8,
  0x86, 0x87, 0xe8, 0x86, 0x89, 0xe8, 0x86, 0x8b, 0xe8, 0x86, 0x8c, 0xe8,
  0x86, 0x8d, 0xe8, 0x86, 0x8e, 0xe8, 0x86, 0x90, 0xe8, 0x86, 0x92, 0xe8,
  0x86, 0x93, 0xe8, 0x86, 0x94, 0xe8, 0x86, 0x95, 0xe8, 0x86, 0x96, 0xe8,
  0x86, 0x97, 0xe8, 0x86, 0x99, 0xe8, 0x86, 0x9a, 0xe8, 0x86, 0x9e, 0xe8,
  0x86, 0x9f, 0xe8, 0x86, 0xa0, 0xe8, 0x86, 0xa1, 0xe8, 0x86, 0xa2, 0xe8,
  0x86, 0xa4, 0xe8, 0x86, 0xa5, 0xe8, 0x86, 0xa7, 0xe8, 0x86, 0xa9, 0xe8,
  0x86, 0xab, 0xe8, 0x86, 0xac, 0xe8, 0x86, 0xad, 0xe8, 0x86, 0xae, 0xe8,
  0x86, 0xaf, 0xe8, 0x86, 0xb0, 0xe8, 0x86, 0xb1, 0xe8, 0x86, 0xb2, 0xe8,
  0x86, 0xb4, 0xe8, 0x86, 0xb5, 0xe8, 0x86, 0xb6, 0xe8, 0x86, 0xb7, 0xe8,
  0x86, 0xb8, 0xe8, 0x86, 0xb9, 0xe8, 0x86, 0xbc, 0xe8, 0x86, 0xbd, 0xe8,
  0x86, 0xbe, 0xe8, 0x86, 0xbf, 0xe8, 0x87, 0x84, 0xe8, 0x87, 0x85, 0xe8,
  0x87, 0x87, 0xe8, 0x87, 0x88, 0xe8, 0x87, 0x89, 0xe8, 0x87, 0x8b, 0xe8,
  0x87, 0x8d, 0xe8, 0x87, 0x8e, 0xe8, 0x87, 0x8f, 0xe8, 0x87, 0x90, 0xe8,
  0x87, 0x91, 0xe8, 0x87, 0x92, 0xe8, 0x87, 0x93, 0xe6, 0x91, 0xb9, 0xe8,
  0x98, 0x91, 0xe6, 0xa8, 0xa1, 0xe8, 0x86, 0x9c, 0xe7, 0xa3, 0xa8, 0xe6,
  0x91, 0xa9, 0xe9, 0xad, 0x94, 0xe6, 0x8a, 0xb9, 0xe6, 0x9c, 0xab, 0xe8,
  0x8e, 0xab, 0xe5, 0xa2, 0xa8, 0xe9, 0xbb, 0x98, 0xe6, 0xb2, 0xab, 0xe6,
  0xbc, 0xa0, 0xe5, 0xaf, 0x9e, 0xe9, 0x99, 0x8c, 0xe8, 0xb0, 0x8b, 0xe7,
  0x89, 0x9f, 0xe6, 0x9f, 0x90, 0xe6, 0x8b, 0x87, 0xe7, 0x89, 0xa1, 0xe4,
  0xba, 0xa9, 0xe5, 0xa7, 0x86, 0xe6, 0xaf, 0x8d, 0xe5, 0xa2, 0x93, 0xe6,
  0x9a, 0xae, 0xe5, 0xb9, 0x95, 0xe5, 0x8b, 0x9f, 0xe6, 0x85, 0x95, 0xe6,
  0x9c, 0xa8, 0xe7, 0x9b, 0xae, 0xe7, 0x9d, 0xa6, 0xe7, 0x89, 0xa7, 0xe7,
  0xa9, 0x86, 0xe6, 0x8b, 0xbf, 0xe5, 0x93, 0xaa, 0xe5, 0x91, 0x90, 0xe9,
  0x92, 0xa0, 0xe9, 0x82, 0xa3, 0xe5, 0xa8, 0x9c, 0xe7, 0xba, 0xb3, 0xe6,
  0xb0, 0x96, 0xe4, 0xb9, 0x83, 0xe5, 0xa5, 0xb6, 0xe8, 0x80, 0x90, 0xe5,
  0xa5, 0x88, 0xe5, 0x8d, 0x97, 0xe7, 0x94, 0xb7, 0xe9, 0x9a, 0xbe, 0xe5,
  0x9b, 0x8a, 0xe6, 0x8c, 0xa0, 0xe8, 0x84, 0x91, 0xe6, 0x81, 0xbc, 0xe9,
  0x97, 0xb9, 0xe6, 0xb7, 0x96, 0xe5, 0x91, 0xa2, 0xe9, 0xa6, 0x81, 0xe5,
  0x86, 0x85, 0xe5, 0xab, 0xa9, 0xe8, 0x83, 0xbd, 0xe5, 0xa6, 0xae, 0xe9,
  0x9c, 0x93, 0xe5, 0x80, 0xaa, 0xe6, 0xb3, 0xa5, 0xe5, 0xb0, 0xbc, 0xe6,
  0x8b, 0x9f, 0xe4, 0xbd, 0xa0, 0xe5, 0x8c, 0xbf, 0xe8, 0x85, 0xbb, 0xe9,
  0x80, 0x86, 0xe6, 0xba, 0xba, 0xe8, 0x94, 0xab, 0xe6, 0x8b, 0x88, 0xe5,
  0xb9, 0xb4, 0xe7, 0xa2, 0xbe, 0xe6, 0x92, 0xb5, 0xe6, 0x8d, 0xbb, 0xe5,
  0xbf, 0xb5, 0xe5, 0xa8, 0x98, 0xe9, 0x85, 0xbf, 0xe9, 0xb8, 0x9f, 0xe5,
  0xb0, 0xbf, 0xe6, 0x8d, 0x8f, 0xe8, 0x81, 0x82, 0xe5, 0xad, 0xbd, 0xe5,
  0x95, 0xae, 0xe9, 0x95, 0x8a, 0xe9, 0x95, 0x8d, 0xe6, 0xb6, 0x85, 0xe6,
  0x82, 0xa8, 0xe6, 0x9f, 0xa0, 0xe7, 0x8b, 0x9e, 0xe5, 0x87, 0x9d, 0xe5,
  0xae, 0x81
};

unsigned char gbk_row_0xc5_converted_to_utf8[] = {
  0xe8, 0x87, 0x94, 0xe8, 0x87, 0x95, 0xe8, 0x87, 0x96,
  0xe8, 0x87, 0x97, 0xe8, 0x87, 0x98, 0xe8, 0x87, 0x99, 0xe8, 0x87, 0x9a,
  0xe8, 0x87, 0x9b, 0xe8, 0x87, 0x9c, 0xe8, 0x87, 0x9d, 0xe8, 0x87, 0x9e,
  0xe8, 0x87, 0x9f, 0xe8, 0x87, 0xa0, 0xe8, 0x87, 0xa1, 0xe8, 0x87, 0xa2,
  0xe8, 0x87, 0xa4, 0xe8, 0x87, 0xa5, 0xe8, 0x87, 0xa6, 0xe8, 0x87, 0xa8,
  0xe8, 0x87, 0xa9, 0xe8, 0x87, 0xab, 0xe8, 0x87, 0xae, 0xe8, 0x87, 0xaf,
  0xe8, 0x87, 0xb0, 0xe8, 0x87, 0xb1, 0xe8, 0x87, 0xb2, 0xe8, 0x87, 0xb5,
  0xe8, 0x87, 0xb6, 0xe8, 0x87, 0xb7, 0xe8, 0x87, 0xb8, 0xe8, 0x87, 0xb9,
  0xe8, 0x87, 0xba, 0xe8, 0x87, 0xbd, 0xe8, 0x87, 0xbf, 0xe8, 0x88, 0x83,
  0xe8, 0x88, 0x87, 0xe8, 0x88, 0x88, 0xe8, 0x88, 0x89, 0xe8, 0x88, 0x8a,
  0xe8, 0x88, 0x8b, 0xe8, 0x88, 0x8e, 0xe8, 0x88, 0x8f, 0xe8, 0x88, 0x91,
  0xe8, 0x88, 0x93, 0xe8, 0x88, 0x95, 0xe8, 0x88, 0x96, 0xe8, 0x88, 0x97,
  0xe8, 0x88, 0x98, 0xe8, 0x88, 0x99, 0xe8, 0x88, 0x9a, 0xe8, 0x88, 0x9d,
  0xe8, 0x88, 0xa0, 0xe8, 0x88, 0xa4, 0xe8, 0x88, 0xa5, 0xe8, 0x88, 0xa6,
  0xe8, 0x88, 0xa7, 0xe8, 0x88, 0xa9, 0xe8, 0x88, 0xae, 0xe8, 0x88, 0xb2,
  0xe8, 0x88, 0xba, 0xe8, 0x88, 0xbc, 0xe8, 0x88, 0xbd, 0xe8, 0x88, 0xbf,
  0xe8, 0x89, 0x80, 0xe8, 0x89, 0x81, 0xe8, 0x89, 0x82, 0xe8, 0x89, 0x83,
  0xe8, 0x89, 0x85, 0xe8, 0x89, 0x86, 0xe8, 0x89, 0x88, 0xe8, 0x89, 0x8a,
  0xe8, 0x89, 0x8c, 0xe8, 0x89, 0x8d, 0xe8, 0x89, 0x8e, 0xe8, 0x89, 0x90,
  0xe8, 0x89, 0x91, 0xe8, 0x89, 0x92, 0xe8, 0x89, 0x93, 0xe8, 0x89, 0x94,
  0xe8, 0x89, 0x95, 0xe8, 0x89, 0x96, 0xe8, 0x89, 0x97, 0xe8, 0x89, 0x99,
  0xe8, 0x89, 0x9b, 0xe8, 0x89, 0x9c, 0xe8, 0x89, 0x9d, 0xe8, 0x89, 0x9e,
  0xe8, 0x89, 0xa0, 0xe8, 0x89, 0xa1, 0xe8, 0x89, 0xa2, 0xe8, 0x89, 0xa3,
  0xe8, 0x89, 0xa4, 0xe8, 0x89, 0xa5, 0xe8, 0x89, 0xa6, 0xe8, 0x89, 0xa7,
  0xe8, 0x89, 0xa9, 0xe6, 0x8b, 0xa7, 0xe6, 0xb3, 0x9e, 0xe7, 0x89, 0x9b,
  0xe6, 0x89, 0xad, 0xe9, 0x92, 0xae, 0xe7, 0xba, 0xbd, 0xe8, 0x84, 0x93,
  0xe6, 0xb5, 0x93, 0xe5, 0x86, 0x9c, 0xe5, 0xbc, 0x84, 0xe5, 0xa5, 0xb4,
  0xe5, 0x8a, 0xaa, 0xe6, 0x80, 0x92, 0xe5, 0xa5, 0xb3, 0xe6, 0x9a, 0x96,
  0xe8, 0x99, 0x90, 0xe7, 0x96, 0x9f, 0xe6, 0x8c, 0xaa, 0xe6, 0x87, 0xa6,
  0xe7, 0xb3, 0xaf, 0xe8, 0xaf, 0xba, 0xe5, 0x93, 0xa6, 0xe6, 0xac, 0xa7,
  0xe9, 0xb8, 0xa5, 0xe6, 0xae, 0xb4, 0xe8, 0x97, 0x95, 0xe5, 0x91, 0x95,
  0xe5, 0x81, 0xb6, 0xe6, 0xb2, 0xa4, 0xe5, 0x95, 0xaa, 0xe8, 0xb6, 0xb4,
  0xe7, 0x88, 0xac, 0xe5, 0xb8, 0x95, 0xe6, 0x80, 0x95, 0xe7, 0x90, 0xb6,
  0xe6, 0x8b, 0x8d, 0xe6, 0x8e, 0x92, 0xe7, 0x89, 0x8c, 0xe5, 0xbe, 0x98,
  0xe6, 0xb9, 0x83, 0xe6, 0xb4, 0xbe, 0xe6, 0x94, 0x80, 0xe6, 0xbd, 0x98,
  0xe7, 0x9b, 0x98, 0xe7, 0xa3, 0x90, 0xe7, 0x9b, 0xbc, 0xe7, 0x95, 0x94,
  0xe5, 0x88, 0xa4, 0xe5, 0x8f, 0x9b, 0xe4, 0xb9, 0x93, 0xe5, 0xba, 0x9e,
  0xe6, 0x97, 0x81, 0xe8, 0x80, 0xaa, 0xe8, 0x83, 0x96, 0xe6, 0x8a, 0x9b,
  0xe5, 0x92, 0x86, 0xe5, 0x88, 0xa8, 0xe7, 0x82, 0xae, 0xe8, 0xa2, 0x8d,
  0xe8, 0xb7, 0x91, 0xe6, 0xb3, 0xa1, 0xe5, 0x91, 0xb8, 0xe8, 0x83, 0x9a,
  0xe5, 0x9f, 0xb9, 0xe8, 0xa3, 0xb4, 0xe8, 0xb5, 0x94, 0xe9, 0x99, 0xaa,
  0xe9, 0x85, 0x8d, 0xe4, 0xbd, 0xa9, 0xe6, 0xb2, 0x9b, 0xe5, 0x96, 0xb7,
  0xe7, 0x9b, 0x86, 0xe7, 0xa0, 0xb0, 0xe6, 0x8a, 0xa8, 0xe7, 0x83, 0xb9,
  0xe6, 0xbe, 0x8e, 0xe5, 0xbd, 0xad, 0xe8, 0x93, 0xac, 0xe6, 0xa3, 0x9a,
  0xe7, 0xa1, 0xbc, 0xe7, 0xaf, 0xb7, 0xe8, 0x86, 0xa8, 0xe6, 0x9c, 0x8b,
  0xe9, 0xb9, 0x8f, 0xe6, 0x8d, 0xa7, 0xe7, 0xa2, 0xb0, 0xe5, 0x9d, 0xaf,
  0xe7, 0xa0, 0x92, 0xe9, 0x9c, 0xb9, 0xe6, 0x89, 0xb9, 0xe6, 0x8a, 0xab,
  0xe5, 0x8a, 0x88, 0xe7, 0x90, 0xb5, 0xe6, 0xaf, 0x97
};

unsigned char gbk_row_0xc6_converted_to_utf8[] = {
  0xe8, 0x89,
  0xaa, 0xe8, 0x89, 0xab, 0xe8, 0x89, 0xac, 0xe8, 0x89, 0xad, 0xe8, 0x89,
  0xb1, 0xe8, 0x89, 0xb5, 0xe8, 0x89, 0xb6, 0xe8, 0x89, 0xb7, 0xe8, 0x89,
  0xb8, 0xe8, 0x89, 0xbb, 0xe8, 0x89, 0xbc, 0xe8, 0x8a, 0x80, 0xe8, 0x8a,
  0x81, 0xe8, 0x8a, 0x83, 0xe8, 0x8a, 0x85, 0xe8, 0x8a, 0x86, 0xe8, 0x8a,
  0x87, 0xe8, 0x8a, 0x89, 0xe8, 0x8a, 0x8c, 0xe8, 0x8a, 0x90, 0xe8, 0x8a,
  0x93, 0xe8, 0x8a, 0x94, 0xe8, 0x8a, 0x95, 0xe8, 0x8a, 0x96, 0xe8, 0x8a,
  0x9a, 0xe8, 0x8a, 0x9b, 0xe8, 0x8a, 0x9e, 0xe8, 0x8a, 0xa0, 0xe8, 0x8a,
  0xa2, 0xe8, 0x8a, 0xa3, 0xe8, 0x8a, 0xa7, 0xe8, 0x8a, 0xb2, 0xe8, 0x8a,
  0xb5, 0xe8, 0x8a, 0xb6, 0xe8, 0x8a, 0xba, 0xe8, 0x8a, 0xbb, 0xe8, 0x8a,
  0xbc, 0xe8, 0x8a, 0xbf, 0xe8, 0x8b, 0x80, 0xe8, 0x8b, 0x82, 0xe8, 0x8b,
  0x83, 0xe8, 0x8b, 0x85, 0xe8, 0x8b, 0x86, 0xe8, 0x8b, 0x89, 0xe8, 0x8b,
  0x90, 0xe8, 0x8b, 0x96, 0xe8, 0x8b, 0x99, 0xe8, 0x8b, 0x9a, 0xe8, 0x8b,
  0x9d, 0xe8, 0x8b, 0xa2, 0xe8, 0x8b, 0xa7, 0xe8, 0x8b, 0xa8, 0xe8, 0x8b,
  0xa9, 0xe8, 0x8b, 0xaa, 0xe8, 0x8b, 0xac, 0xe8, 0x8b, 0xad, 0xe8, 0x8b,
  0xae, 0xe8, 0x8b, 0xb0, 0xe8, 0x8b, 0xb2, 0xe8, 0x8b, 0xb3, 0xe8, 0x8b,
  0xb5, 0xe8, 0x8b, 0xb6, 0xe8, 0x8b, 0xb8, 0xe8, 0x8b, 0xba, 0xe8, 0x8b,
  0xbc, 0xe8, 0x8b, 0xbd, 0xe8, 0x8b, 0xbe, 0xe8, 0x8b, 0xbf, 0xe8, 0x8c,
  0x80, 0xe8, 0x8c, 0x8a, 0xe8, 0x8c, 0x8b, 0xe8, 0x8c, 0x8d, 0xe8, 0x8c,
  0x90, 0xe8, 0x8c, 0x92, 0xe8, 0x8c, 0x93, 0xe8, 0x8c, 0x96, 0xe8, 0x8c,
  0x98, 0xe8, 0x8c, 0x99, 0xe8, 0x8c, 0x9d, 0xe8, 0x8c, 0x9e, 0xe8, 0x8c,
  0x9f, 0xe8, 0x8c, 0xa0, 0xe8, 0x8c, 0xa1, 0xe8, 0x8c, 0xa2, 0xe8, 0x8c,
  0xa3, 0xe8, 0x8c, 0xa4, 0xe8, 0x8c, 0xa5, 0xe8, 0x8c, 0xa6, 0xe8, 0x8c,
  0xa9, 0xe8, 0x8c, 0xaa, 0xe8, 0x8c, 0xae, 0xe8, 0x8c, 0xb0, 0xe8, 0x8c,
  0xb2, 0xe8, 0x8c, 0xb7, 0xe8, 0x8c, 0xbb, 0xe8, 0x8c, 0xbd, 0xe5, 0x95,
  0xa4, 0xe8, 0x84, 0xbe, 0xe7, 0x96, 0xb2, 0xe7, 0x9a, 0xae, 0xe5, 0x8c,
  0xb9, 0xe7, 0x97, 0x9e, 0xe5, 0x83, 0xbb, 0xe5, 0xb1, 0x81, 0xe8, 0xad,
  0xac, 0xe7, 0xaf, 0x87, 0xe5, 0x81, 0x8f, 0xe7, 0x89, 0x87, 0xe9, 0xaa,
  0x97, 0xe9, 0xa3, 0x98, 0xe6, 0xbc, 0x82, 0xe7, 0x93, 0xa2, 0xe7, 0xa5,
  0xa8, 0xe6, 0x92, 0x87, 0xe7, 0x9e, 0xa5, 0xe6, 0x8b, 0xbc, 0xe9, 0xa2,
  0x91, 0xe8, 0xb4, 0xab, 0xe5, 0x93, 0x81, 0xe8, 0x81, 0x98, 0xe4, 0xb9,
  0x92, 0xe5, 0x9d, 0xaa, 0xe8, 0x8b, 0xb9, 0xe8, 0x90, 0x8d, 0xe5, 0xb9,
  0xb3, 0xe5, 0x87, 0xad, 0xe7, 0x93, 0xb6, 0xe8, 0xaf, 0x84, 0xe5, 0xb1,
  0x8f, 0xe5, 0x9d, 0xa1, 0xe6, 0xb3, 0xbc, 0xe9, 0xa2, 0x87, 0xe5, 0xa9,
  0x86, 0xe7, 0xa0, 0xb4, 0xe9, 0xad, 0x84, 0xe8, 0xbf, 0xab, 0xe7, 0xb2,
  0x95, 0xe5, 0x89, 0x96, 0xe6, 0x89, 0x91, 0xe9, 0x93, 0xba, 0xe4, 0xbb,
  0x86, 0xe8, 0x8e, 0x86, 0xe8, 0x91, 0xa1, 0xe8, 0x8f, 0xa9, 0xe8, 0x92,
  0xb2, 0xe5, 0x9f, 0x94, 0xe6, 0x9c, 0xb4, 0xe5, 0x9c, 0x83, 0xe6, 0x99,
  0xae, 0xe6, 0xb5, 0xa6, 0xe8, 0xb0, 0xb1, 0xe6, 0x9b, 0x9d, 0xe7, 0x80,
  0x91, 0xe6, 0x9c, 0x9f, 0xe6, 0xac, 0xba, 0xe6, 0xa0, 0x96, 0xe6, 0x88,
  0x9a, 0xe5, 0xa6, 0xbb, 0xe4, 0xb8, 0x83, 0xe5, 0x87, 0x84, 0xe6, 0xbc,
  0x86, 0xe6, 0x9f, 0x92, 0xe6, 0xb2, 0x8f, 0xe5, 0x85, 0xb6, 0xe6, 0xa3,
  0x8b, 0xe5, 0xa5, 0x87, 0xe6, 0xad, 0xa7, 0xe7, 0x95, 0xa6, 0xe5, 0xb4,
  0x8e, 0xe8, 0x84, 0x90, 0xe9, 0xbd, 0x90, 0xe6, 0x97, 0x97, 0xe7, 0xa5,
  0x88, 0xe7, 0xa5, 0x81, 0xe9, 0xaa, 0x91, 0xe8, 0xb5, 0xb7, 0xe5, 0xb2,
  0x82, 0xe4, 0xb9, 0x9e, 0xe4, 0xbc, 0x81, 0xe5, 0x90, 0xaf, 0xe5, 0xa5,
  0x91, 0xe7, 0xa0, 0x8c, 0xe5, 0x99, 0xa8, 0xe6, 0xb0, 0x94, 0xe8, 0xbf,
  0x84, 0xe5, 0xbc, 0x83, 0xe6, 0xb1, 0xbd, 0xe6, 0xb3, 0xa3, 0xe8, 0xae,
  0xab, 0xe6, 0x8e, 0x90
};

unsigned char gbk_row_0xc7_converted_to_utf8[] = {
  0xe8, 0x8c, 0xbe, 0xe8, 0x8c, 0xbf, 0xe8,
  0x8d, 0x81, 0xe8, 0x8d, 0x82, 0xe8, 0x8d, 0x84, 0xe8, 0x8d, 0x85, 0xe8,
  0x8d, 0x88, 0xe8, 0x8d, 0x8a, 0xe8, 0x8d, 0x8b, 0xe8, 0x8d, 0x8c, 0xe8,
  0x8d, 0x8d, 0xe8, 0x8d, 0x8e, 0xe8, 0x8d, 0x93, 0xe8, 0x8d, 0x95, 0xe8,
  0x8d, 0x96, 0xe8, 0x8d, 0x97, 0xe8, 0x8d, 0x98, 0xe8, 0x8d, 0x99, 0xe8,
  0x8d, 0x9d, 0xe8, 0x8d, 0xa2, 0xe8, 0x8d, 0xb0, 0xe8, 0x8d, 0xb1, 0xe8,
  0x8d, 0xb2, 0xe8, 0x8d, 0xb3, 0xe8, 0x8d, 0xb4, 0xe8, 0x8d, 0xb5, 0xe8,
  0x8d, 0xb6, 0xe8, 0x8d, 0xb9, 0xe8, 0x8d, 0xba, 0xe8, 0x8d, 0xbe, 0xe8,
  0x8d, 0xbf, 0xe8, 0x8e, 0x80, 0xe8, 0x8e, 0x81, 0xe8, 0x8e, 0x82, 0xe8,
  0x8e, 0x83, 0xe8, 0x8e, 0x84, 0xe8, 0x8e, 0x87, 0xe8, 0x8e, 0x88, 0xe8,
  0x8e, 0x8a, 0xe8, 0x8e, 0x8b, 0xe8, 0x8e, 0x8c, 0xe8, 0x8e, 0x8d, 0xe8,
  0x8e, 0x8f, 0xe8, 0x8e, 0x90, 0xe8, 0x8e, 0x91, 0xe8, 0x8e, 0x94, 0xe8,
  0x8e, 0x95, 0xe8, 0x8e, 0x96, 0xe8, 0x8e, 0x97, 0xe8, 0x8e, 0x99, 0xe8,
  0x8e, 0x9a, 0xe8, 0x8e, 0x9d, 0xe8, 0x8e, 0x9f, 0xe8, 0x8e, 0xa1, 0xe8,
  0x8e, 0xa2, 0xe8, 0x8e, 0xa3, 0xe8, 0x8e, 0xa4, 0xe8, 0x8e, 0xa5, 0xe8,
  0x8e, 0xa6, 0xe8, 0x8e, 0xa7, 0xe8, 0x8e, 0xac, 0xe8, 0x8e, 0xad, 0xe8,
  0x8e, 0xae, 0xe8, 0x8e, 0xaf, 0xe8, 0x8e, 0xb5, 0xe8, 0x8e, 0xbb, 0xe8,
  0x8e, 0xbe, 0xe8, 0x8e, 0xbf, 0xe8, 0x8f, 0x82, 0xe8, 0x8f, 0x83, 0xe8,
  0x8f, 0x84, 0xe8, 0x8f, 0x86, 0xe8, 0x8f, 0x88, 0xe8, 0x8f, 0x89, 0xe8,
  0x8f, 0x8b, 0xe8, 0x8f, 0x8d, 0xe8, 0x8f, 0x8e, 0xe8, 0x8f, 0x90, 0xe8,
  0x8f, 0x91, 0xe8, 0x8f, 0x92, 0xe8, 0x8f, 0x93, 0xe8, 0x8f, 0x95, 0xe8,
  0x8f, 0x97, 0xe8, 0x8f, 0x99, 0xe8, 0x8f, 0x9a, 0xe8, 0x8f, 0x9b, 0xe8,
  0x8f, 0x9e, 0xe8, 0x8f, 0xa2, 0xe8, 0x8f, 0xa3, 0xe8, 0x8f, 0xa4, 0xe8,
  0x8f, 0xa6, 0xe8, 0x8f, 0xa7, 0xe8, 0x8f, 0xa8, 0xe8, 0x8f, 0xab, 0xe8,
  0x8f, 0xac, 0xe8, 0x8f, 0xad, 0xe6, 0x81, 0xb0, 0xe6, 0xb4, 0xbd, 0xe7,
  0x89, 0xb5, 0xe6, 0x89, 0xa6, 0xe9, 0x92, 0x8e, 0xe9, 0x93, 0x85, 0xe5,
  0x8d, 0x83, 0xe8, 0xbf, 0x81, 0xe7, 0xad, 0xbe, 0xe4, 0xbb, 0x9f, 0xe8,
  0xb0, 0xa6, 0xe4, 0xb9, 0xbe, 0xe9, 0xbb, 0x94, 0xe9, 0x92, 0xb1, 0xe9,
  0x92, 0xb3, 0xe5, 0x89, 0x8d, 0xe6, 0xbd, 0x9c, 0xe9, 0x81, 0xa3, 0xe6,
  0xb5, 0x85, 0xe8, 0xb0, 0xb4, 0xe5, 0xa0, 0x91, 0xe5, 0xb5, 0x8c, 0xe6,
  0xac, 0xa0, 0xe6, 0xad, 0x89, 0xe6, 0x9e, 0xaa, 0xe5, 0x91, 0x9b, 0xe8,
  0x85, 0x94, 0xe7, 0xbe, 0x8c, 0xe5, 0xa2, 0x99, 0xe8, 0x94, 0xb7, 0xe5,
  0xbc, 0xba, 0xe6, 0x8a, 0xa2, 0xe6, 0xa9, 0x87, 0xe9, 0x94, 0xb9, 0xe6,
  0x95, 0xb2, 0xe6, 0x82, 0x84, 0xe6, 0xa1, 0xa5, 0xe7, 0x9e, 0xa7, 0xe4,
  0xb9, 0x94, 0xe4, 0xbe, 0xa8, 0xe5, 0xb7, 0xa7, 0xe9, 0x9e, 0x98, 0xe6,
  0x92, 0xac, 0xe7, 0xbf, 0x98, 0xe5, 0xb3, 0xad, 0xe4, 0xbf, 0x8f, 0xe7,
  0xaa, 0x8d, 0xe5, 0x88, 0x87, 0xe8, 0x8c, 0x84, 0xe4, 0xb8, 0x94, 0xe6,
  0x80, 0xaf, 0xe7, 0xaa, 0x83, 0xe9, 0x92, 0xa6, 0xe4, 0xbe, 0xb5, 0xe4,
  0xba, 0xb2, 0xe7, 0xa7, 0xa6, 0xe7, 0x90, 0xb4, 0xe5, 0x8b, 0xa4, 0xe8,
  0x8a, 0xb9, 0xe6, 0x93, 0x92, 0xe7, 0xa6, 0xbd, 0xe5, 0xaf, 0x9d, 0xe6,
  0xb2, 0x81, 0xe9, 0x9d, 0x92, 0xe8, 0xbd, 0xbb, 0xe6, 0xb0, 0xa2, 0xe5,
  0x80, 0xbe, 0xe5, 0x8d, 0xbf, 0xe6, 0xb8, 0x85, 0xe6, 0x93, 0x8e, 0xe6,
  0x99, 0xb4, 0xe6, 0xb0, 0xb0, 0xe6, 0x83, 0x85, 0xe9, 0xa1, 0xb7, 0xe8,
  0xaf, 0xb7, 0xe5, 0xba, 0x86, 0xe7, 0x90, 0xbc, 0xe7, 0xa9, 0xb7, 0xe7,
  0xa7, 0x8b, 0xe4, 0xb8, 0x98, 0xe9, 0x82, 0xb1, 0xe7, 0x90, 0x83, 0xe6,
  0xb1, 0x82, 0xe5, 0x9b, 0x9a, 0xe9, 0x85, 0x8b, 0xe6, 0xb3, 0x85, 0xe8,
  0xb6, 0x8b, 0xe5, 0x8c, 0xba, 0xe8, 0x9b, 0x86, 0xe6, 0x9b, 0xb2, 0xe8,
  0xba, 0xaf, 0xe5, 0xb1, 0x88, 0xe9, 0xa9, 0xb1, 0xe6, 0xb8, 0xa0
};

unsigned char gbk_row_0xc8_converted_to_utf8[] = {
  0xe8, 0x8f, 0xae, 0xe8, 0x8f, 0xaf, 0xe8, 0x8f, 0xb3, 0xe8, 0x8f, 0xb4,
  0xe8, 0x8f, 0xb5, 0xe8, 0x8f, 0xb6, 0xe8, 0x8f, 0xb7, 0xe8, 0x8f, 0xba,
  0xe8, 0x8f, 0xbb, 0xe8, 0x8f, 0xbc, 0xe8, 0x8f, 0xbe, 0xe8, 0x8f, 0xbf,
  0xe8, 0x90, 0x80, 0xe8, 0x90, 0x82, 0xe8, 0x90, 0x85, 0xe8, 0x90, 0x87,
  0xe8, 0x90, 0x88, 0xe8, 0x90, 0x89, 0xe8, 0x90, 0x8a, 0xe8, 0x90, 0x90,
  0xe8, 0x90, 0x92, 0xe8, 0x90, 0x93, 0xe8, 0x90, 0x94, 0xe8, 0x90, 0x95,
  0xe8, 0x90, 0x96, 0xe8, 0x90, 0x97, 0xe8, 0x90, 0x99, 0xe8, 0x90, 0x9a,
  0xe8, 0x90, 0x9b, 0xe8, 0x90, 0x9e, 0xe8, 0x90, 0x9f, 0xe8, 0x90, 0xa0,
  0xe8, 0x90, 0xa1, 0xe8, 0x90, 0xa2, 0xe8, 0x90, 0xa3, 0xe8, 0x90, 0xa9,
  0xe8, 0x90, 0xaa, 0xe8, 0x90, 0xab, 0xe8, 0x90, 0xac, 0xe8, 0x90, 0xad,
  0xe8, 0x90, 0xae, 0xe8, 0x90, 0xaf, 0xe8, 0x90, 0xb0, 0xe8, 0x90, 0xb2,
  0xe8, 0x90, 0xb3, 0xe8, 0x90, 0xb4, 0xe8, 0x90, 0xb5, 0xe8, 0x90, 0xb6,
  0xe8, 0x90, 0xb7, 0xe8, 0x90, 0xb9, 0xe8, 0x90, 0xba, 0xe8, 0x90, 0xbb,
  0xe8, 0x90, 0xbe, 0xe8, 0x90, 0xbf, 0xe8, 0x91, 0x80, 0xe8, 0x91, 0x81,
  0xe8, 0x91, 0x82, 0xe8, 0x91, 0x83, 0xe8, 0x91, 0x84, 0xe8, 0x91, 0x85,
  0xe8, 0x91, 0x87, 0xe8, 0x91, 0x88, 0xe8, 0x91, 0x89, 0xe8, 0x91, 0x8a,
  0xe8, 0x91, 0x8b, 0xe8, 0x91, 0x8c, 0xe8, 0x91, 0x8d, 0xe8, 0x91, 0x8e,
  0xe8, 0x91, 0x8f, 0xe8, 0x91, 0x90, 0xe8, 0x91, 0x92, 0xe8, 0x91, 0x93,
  0xe8, 0x91, 0x94, 0xe8, 0x91, 0x95, 0xe8, 0x91, 0x96, 0xe8, 0x91, 0x98,
  0xe8, 0x91, 0x9d, 0xe8, 0x91, 0x9e, 0xe8, 0x91, 0x9f, 0xe8, 0x91, 0xa0,
  0xe8, 0x91, 0xa2, 0xe8, 0x91, 0xa4, 0xe8, 0x91, 0xa5, 0xe8, 0x91, 0xa6,
  0xe8, 0x91, 0xa7, 0xe8, 0x91, 0xa8, 0xe8, 0x91, 0xaa, 0xe8, 0x91, 0xae,
  0xe8, 0x91, 0xaf, 0xe8, 0x91, 0xb0, 0xe8, 0x91, 0xb2, 0xe8, 0x91, 0xb4,
  0xe8, 0x91, 0xb7, 0xe8, 0x91, 0xb9, 0xe8, 0x91, 0xbb, 0xe8, 0x91, 0xbc,
  0xe5, 0x8f, 0x96, 0xe5, 0xa8, 0xb6, 0xe9, 0xbe, 0x8b, 0xe8, 0xb6, 0xa3,
  0xe5, 0x8e, 0xbb, 0xe5, 0x9c, 0x88, 0xe9, 0xa2, 0xa7, 0xe6, 0x9d, 0x83,
  0xe9, 0x86, 0x9b, 0xe6, 0xb3, 0x89, 0xe5, 0x85, 0xa8, 0xe7, 0x97, 0x8a,
  0xe6, 0x8b, 0xb3, 0xe7, 0x8a, 0xac, 0xe5, 0x88, 0xb8, 0xe5, 0x8a, 0x9d,
  0xe7, 0xbc, 0xba, 0xe7, 0x82, 0x94, 0xe7, 0x98, 0xb8, 0xe5, 0x8d, 0xb4,
  0xe9, 0xb9, 0x8a, 0xe6, 0xa6, 0xb7, 0xe7, 0xa1, 0xae, 0xe9, 0x9b, 0x80,
  0xe8, 0xa3, 0x99, 0xe7, 0xbe, 0xa4, 0xe7, 0x84, 0xb6, 0xe7, 0x87, 0x83,
  0xe5, 0x86, 0x89, 0xe6, 0x9f, 0x93, 0xe7, 0x93, 0xa4, 0xe5, 0xa3, 0xa4,
  0xe6, 0x94, 0x98, 0xe5, 0x9a, 0xb7, 0xe8, 0xae, 0xa9, 0xe9, 0xa5, 0xb6,
  0xe6, 0x89, 0xb0, 0xe7, 0xbb, 0x95, 0xe6, 0x83, 0xb9, 0xe7, 0x83, 0xad,
  0xe5, 0xa3, 0xac, 0xe4, 0xbb, 0x81, 0xe4, 0xba, 0xba, 0xe5, 0xbf, 0x8d,
  0xe9, 0x9f, 0xa7, 0xe4, 0xbb, 0xbb, 0xe8, 0xae, 0xa4, 0xe5, 0x88, 0x83,
  0xe5, 0xa6, 0x8a, 0xe7, 0xba, 0xab, 0xe6, 0x89, 0x94, 0xe4, 0xbb, 0x8d,
  0xe6, 0x97, 0xa5, 0xe6, 0x88, 0x8e, 0xe8, 0x8c, 0xb8, 0xe8, 0x93, 0x89,
  0xe8, 0x8d, 0xa3, 0xe8, 0x9e, 0x8d, 0xe7, 0x86, 0x94, 0xe6, 0xba, 0xb6,
  0xe5, 0xae, 0xb9, 0xe7, 0xbb, 0x92, 0xe5, 0x86, 0x97, 0xe6, 0x8f, 0x89,
  0xe6, 0x9f, 0x94, 0xe8, 0x82, 0x89, 0xe8, 0x8c, 0xb9, 0xe8, 0xa0, 0x95,
  0xe5, 0x84, 0x92, 0xe5, 0xad, 0xba, 0xe5, 0xa6, 0x82, 0xe8, 0xbe, 0xb1,
  0xe4, 0xb9, 0xb3, 0xe6, 0xb1, 0x9d, 0xe5, 0x85, 0xa5, 0xe8, 0xa4, 0xa5,
  0xe8, 0xbd, 0xaf, 0xe9, 0x98, 0xae, 0xe8, 0x95, 0x8a, 0xe7, 0x91, 0x9e,
  0xe9, 0x94, 0x90, 0xe9, 0x97, 0xb0, 0xe6, 0xb6, 0xa6, 0xe8, 0x8b, 0xa5,
  0xe5, 0xbc, 0xb1, 0xe6, 0x92, 0x92, 0xe6, 0xb4, 0x92, 0xe8, 0x90, 0xa8,
  0xe8, 0x85, 0xae, 0xe9, 0xb3, 0x83, 0xe5, 0xa1, 0x9e, 0xe8, 0xb5, 0x9b,
  0xe4, 0xb8, 0x89, 0xe5, 0x8f, 0x81
};

unsigned char gbk_row_0xc9_converted_to_utf8[] = {
  0xe8, 0x91, 0xbd, 0xe8, 0x91,
  0xbe, 0xe8, 0x91, 0xbf, 0xe8, 0x92, 0x80, 0xe8, 0x92, 0x81, 0xe8, 0x92,
  0x83, 0xe8, 0x92, 0x84, 0xe8, 0x92, 0x85, 0xe8, 0x92, 0x86, 0xe8, 0x92,
  0x8a, 0xe8, 0x92, 0x8d, 0xe8, 0x92, 0x8f, 0xe8, 0x92, 0x90, 0xe8, 0x92,
  0x91, 0xe8, 0x92, 0x92, 0xe8, 0x92, 0x93, 0xe8, 0x92, 0x94, 0xe8, 0x92,
  0x95, 0xe8, 0x92, 0x96, 0xe8, 0x92, 0x98, 0xe8, 0x92, 0x9a, 0xe8, 0x92,
  0x9b, 0xe8, 0x92, 0x9d, 0xe8, 0x92, 0x9e, 0xe8, 0x92, 0x9f, 0xe8, 0x92,
  0xa0, 0xe8, 0x92, 0xa2, 0xe8, 0x92, 0xa3, 0xe8, 0x92, 0xa4, 0xe8, 0x92,
  0xa5, 0xe8, 0x92, 0xa6, 0xe8, 0x92, 0xa7, 0xe8, 0x92, 0xa8, 0xe8, 0x92,
  0xa9, 0xe8, 0x92, 0xaa, 0xe8, 0x92, 0xab, 0xe8, 0x92, 0xac, 0xe8, 0x92,
  0xad, 0xe8, 0x92, 0xae, 0xe8, 0x92, 0xb0, 0xe8, 0x92, 0xb1, 0xe8, 0x92,
  0xb3, 0xe8, 0x92, 0xb5, 0xe8, 0x92, 0xb6, 0xe8, 0x92, 0xb7, 0xe8, 0x92,
  0xbb, 0xe8, 0x92, 0xbc, 0xe8, 0x92, 0xbe, 0xe8, 0x93, 0x80, 0xe8, 0x93,
  0x82, 0xe8, 0x93, 0x83, 0xe8, 0x93, 0x85, 0xe8, 0x93, 0x86, 0xe8, 0x93,
  0x87, 0xe8, 0x93, 0x88, 0xe8, 0x93, 0x8b, 0xe8, 0x93, 0x8c, 0xe8, 0x93,
  0x8e, 0xe8, 0x93, 0x8f, 0xe8, 0x93, 0x92, 0xe8, 0x93, 0x94, 0xe8, 0x93,
  0x95, 0xe8, 0x93, 0x97, 0xe8, 0x93, 0x98, 0xe8, 0x93, 0x99, 0xe8, 0x93,
  0x9a, 0xe8, 0x93, 0x9b, 0xe8, 0x93, 0x9c, 0xe8, 0x93, 0x9e, 0xe8, 0x93,
  0xa1, 0xe8, 0x93, 0xa2, 0xe8, 0x93, 0xa4, 0xe8, 0x93, 0xa7, 0xe8, 0x93,
  0xa8, 0xe8, 0x93, 0xa9, 0xe8, 0x93, 0xaa, 0xe8, 0x93, 0xab, 0xe8, 0x93,
  0xad, 0xe8, 0x93, 0xae, 0xe8, 0x93, 0xaf, 0xe8, 0x93, 0xb1, 0xe8, 0x93,
  0xb2, 0xe8, 0x93, 0xb3, 0xe8, 0x93, 0xb4, 0xe8, 0x93, 0xb5, 0xe8, 0x93,
  0xb6, 0xe8, 0x93, 0xb7, 0xe8, 0x93, 0xb8, 0xe8, 0x93, 0xb9, 0xe8, 0x93,
  0xba, 0xe8, 0x93, 0xbb, 0xe8, 0x93, 0xbd, 0xe8, 0x93, 0xbe, 0xe8, 0x94,
  0x80, 0xe8, 0x94, 0x81, 0xe8, 0x94, 0x82, 0xe4, 0xbc, 0x9e, 0xe6, 0x95,
  0xa3, 0xe6, 0xa1, 0x91, 0xe5, 0x97, 0x93, 0xe4, 0xb8, 0xa7, 0xe6, 0x90,
  0x94, 0xe9, 0xaa, 0x9a, 0xe6, 0x89, 0xab, 0xe5, 0xab, 0x82, 0xe7, 0x91,
  0x9f, 0xe8, 0x89, 0xb2, 0xe6, 0xb6, 0xa9, 0xe6, 0xa3, 0xae, 0xe5, 0x83,
  0xa7, 0xe8, 0x8e, 0x8e, 0xe7, 0xa0, 0x82, 0xe6, 0x9d, 0x80, 0xe5, 0x88,
  0xb9, 0xe6, 0xb2, 0x99, 0xe7, 0xba, 0xb1, 0xe5, 0x82, 0xbb, 0xe5, 0x95,
  0xa5, 0xe7, 0x85, 0x9e, 0xe7, 0xad, 0x9b, 0xe6, 0x99, 0x92, 0xe7, 0x8f,
  0x8a, 0xe8, 0x8b, 0xab, 0xe6, 0x9d, 0x89, 0xe5, 0xb1, 0xb1, 0xe5, 0x88,
  0xa0, 0xe7, 0x85, 0xbd, 0xe8, 0xa1, 0xab, 0xe9, 0x97, 0xaa, 0xe9, 0x99,
  0x95, 0xe6, 0x93, 0x85, 0xe8, 0xb5, 0xa1, 0xe8, 0x86, 0xb3, 0xe5, 0x96,
  0x84, 0xe6, 0xb1, 0x95, 0xe6, 0x89, 0x87, 0xe7, 0xbc, 0xae, 0xe5, 0xa2,
  0x92, 0xe4, 0xbc, 0xa4, 0xe5, 0x95, 0x86, 0xe8, 0xb5, 0x8f, 0xe6, 0x99,
  0x8c, 0xe4, 0xb8, 0x8a, 0xe5, 0xb0, 0x9a, 0xe8, 0xa3, 0xb3, 0xe6, 0xa2,
  0xa2, 0xe6, 0x8d, 0x8e, 0xe7, 0xa8, 0x8d, 0xe7, 0x83, 0xa7, 0xe8, 0x8a,
  0x8d, 0xe5, 0x8b, 0xba, 0xe9, 0x9f, 0xb6, 0xe5, 0xb0, 0x91, 0xe5, 0x93,
  0xa8, 0xe9, 0x82, 0xb5, 0xe7, 0xbb, 0x8d, 0xe5, 0xa5, 0xa2, 0xe8, 0xb5,
  0x8a, 0xe8, 0x9b, 0x87, 0xe8, 0x88, 0x8c, 0xe8, 0x88, 0x8d, 0xe8, 0xb5,
  0xa6, 0xe6, 0x91, 0x84, 0xe5, 0xb0, 0x84, 0xe6, 0x85, 0x91, 0xe6, 0xb6,
  0x89, 0xe7, 0xa4, 0xbe, 0xe8, 0xae, 0xbe, 0xe7, 0xa0, 0xb7, 0xe7, 0x94,
  0xb3, 0xe5, 0x91, 0xbb, 0xe4, 0xbc, 0xb8, 0xe8, 0xba, 0xab, 0xe6, 0xb7,
  0xb1, 0xe5, 0xa8, 0xa0, 0xe7, 0xbb, 0x85, 0xe7, 0xa5, 0x9e, 0xe6, 0xb2,
  0x88, 0xe5, 0xae, 0xa1, 0xe5, 0xa9, 0xb6, 0xe7, 0x94, 0x9a, 0xe8, 0x82,
  0xbe, 0xe6, 0x85, 0x8e, 0xe6, 0xb8, 0x97, 0xe5, 0xa3, 0xb0, 0xe7, 0x94,
  0x9f, 0xe7, 0x94, 0xa5, 0xe7, 0x89, 0xb2, 0xe5, 0x8d, 0x87, 0xe7, 0xbb,
  0xb3
};

unsigned char gbk_row_0xca_converted_to_utf8[] = {
  0xe8, 0x94, 0x83, 0xe8, 0x94, 0x84, 0xe8, 0x94, 0x85, 0xe8,
  0x94, 0x86, 0xe8, 0x94, 0x87, 0xe8, 0x94, 0x88, 0xe8, 0x94, 0x89, 0xe8,
  0x94, 0x8a, 0xe8, 0x94, 0x8b, 0xe8, 0x94, 0x8d, 0xe8, 0x94, 0x8e, 0xe8,
  0x94, 0x8f, 0xe8, 0x94, 0x90, 0xe8, 0x94, 0x92, 0xe8, 0x94, 0x94, 0xe8,
  0x94, 0x95, 0xe8, 0x94, 0x96, 0xe8, 0x94, 0x98, 0xe8, 0x94, 0x99, 0xe8,
  0x94, 0x9b, 0xe8, 0x94, 0x9c, 0xe8, 0x94, 0x9d, 0xe8, 0x94, 0x9e, 0xe8,
  0x94, 0xa0, 0xe8, 0x94, 0xa2, 0xe8, 0x94, 0xa3, 0xe8, 0x94, 0xa4, 0xe8,
  0x94, 0xa5, 0xe8, 0x94, 0xa6, 0xe8, 0x94, 0xa7, 0xe8, 0x94, 0xa8, 0xe8,
  0x94, 0xa9, 0xe8, 0x94, 0xaa, 0xe8, 0x94, 0xad, 0xe8, 0x94, 0xae, 0xe8,
  0x94, 0xaf, 0xe8, 0x94, 0xb0, 0xe8, 0x94, 0xb1, 0xe8, 0x94, 0xb2, 0xe8,
  0x94, 0xb3, 0xe8, 0x94, 0xb4, 0xe8, 0x94, 0xb5, 0xe8, 0x94, 0xb6, 0xe8,
  0x94, 0xbe, 0xe8, 0x94, 0xbf, 0xe8, 0x95, 0x80, 0xe8, 0x95, 0x81, 0xe8,
  0x95, 0x82, 0xe8, 0x95, 0x84, 0xe8, 0x95, 0x85, 0xe8, 0x95, 0x86, 0xe8,
  0x95, 0x87, 0xe8, 0x95, 0x8b, 0xe8, 0x95, 0x8c, 0xe8, 0x95, 0x8d, 0xe8,
  0x95, 0x8e, 0xe8, 0x95, 0x8f, 0xe8, 0x95, 0x90, 0xe8, 0x95, 0x91, 0xe8,
  0x95, 0x92, 0xe8, 0x95, 0x93, 0xe8, 0x95, 0x94, 0xe8, 0x95, 0x95, 0xe8,
  0x95, 0x97, 0xe8, 0x95, 0x98, 0xe8, 0x95, 0x9a, 0xe8, 0x95, 0x9b, 0xe8,
  0x95, 0x9c, 0xe8, 0x95, 0x9d, 0xe8, 0x95, 0x9f, 0xe8, 0x95, 0xa0, 0xe8,
  0x95, 0xa1, 0xe8, 0x95, 0xa2, 0xe8, 0x95, 0xa3, 0xe8, 0x95, 0xa5, 0xe8,
  0x95, 0xa6, 0xe8, 0x95, 0xa7, 0xe8, 0x95, 0xa9, 0xe8, 0x95, 0xaa, 0xe8,
  0x95, 0xab, 0xe8, 0x95, 0xac, 0xe8, 0x95, 0xad, 0xe8, 0x95, 0xae, 0xe8,
  0x95, 0xaf, 0xe8, 0x95, 0xb0, 0xe8, 0x95, 0xb1, 0xe8, 0x95, 0xb3, 0xe8,
  0x95, 0xb5, 0xe8, 0x95, 0xb6, 0xe8, 0x95, 0xb7, 0xe8, 0x95, 0xb8, 0xe8,
  0x95, 0xbc, 0xe8, 0x95, 0xbd, 0xe8, 0x95, 0xbf, 0xe8, 0x96, 0x80, 0xe8,
  0x96, 0x81, 0xe7, 0x9c, 0x81, 0xe7, 0x9b, 0x9b, 0xe5, 0x89, 0xa9, 0xe8,
  0x83, 0x9c, 0xe5, 0x9c, 0xa3, 0xe5, 0xb8, 0x88, 0xe5, 0xa4, 0xb1, 0xe7,
  0x8b, 0xae, 0xe6, 0x96, 0xbd, 0xe6, 0xb9, 0xbf, 0xe8, 0xaf, 0x97, 0xe5,
  0xb0, 0xb8, 0xe8, 0x99, 0xb1, 0xe5, 0x8d, 0x81, 0xe7, 0x9f, 0xb3, 0xe6,
  0x8b, 0xbe, 0xe6, 0x97, 0xb6, 0xe4, 0xbb, 0x80, 0xe9, 0xa3, 0x9f, 0xe8,
  0x9a, 0x80, 0xe5, 0xae, 0x9e, 0xe8, 0xaf, 0x86, 0xe5, 0x8f, 0xb2, 0xe7,
  0x9f, 0xa2, 0xe4, 0xbd, 0xbf, 0xe5, 0xb1, 0x8e, 0xe9, 0xa9, 0xb6, 0xe5,
  0xa7, 0x8b, 0xe5, 0xbc, 0x8f, 0xe7, 0xa4, 0xba, 0xe5, 0xa3, 0xab, 0xe4,
  0xb8, 0x96, 0xe6, 0x9f, 0xbf, 0xe4, 0xba, 0x8b, 0xe6, 0x8b, 0xad, 0xe8,
  0xaa, 0x93, 0xe9, 0x80, 0x9d, 0xe5, 0x8a, 0xbf, 0xe6, 0x98, 0xaf, 0xe5,
  0x97, 0x9c, 0xe5, 0x99, 0xac, 0xe9, 0x80, 0x82, 0xe4, 0xbb, 0x95, 0xe4,
  0xbe, 0x8d, 0xe9, 0x87, 0x8a, 0xe9, 0xa5, 0xb0, 0xe6, 0xb0, 0x8f, 0xe5,
  0xb8, 0x82, 0xe6, 0x81, 0x83, 0xe5, 0xae, 0xa4, 0xe8, 0xa7, 0x86, 0xe8,
  0xaf, 0x95, 0xe6, 0x94, 0xb6, 0xe6, 0x89, 0x8b, 0xe9, 0xa6, 0x96, 0xe5,
  0xae, 0x88, 0xe5, 0xaf, 0xbf, 0xe6, 0x8e, 0x88, 0xe5, 0x94, 0xae, 0xe5,
  0x8f, 0x97, 0xe7, 0x98, 0xa6, 0xe5, 0x85, 0xbd, 0xe8, 0x94, 0xac, 0xe6,
  0x9e, 0xa2, 0xe6, 0xa2, 0xb3, 0xe6, 0xae, 0x8a, 0xe6, 0x8a, 0x92, 0xe8,
  0xbe, 0x93, 0xe5, 0x8f, 0x94, 0xe8, 0x88, 0x92, 0xe6, 0xb7, 0x91, 0xe7,
  0x96, 0x8f, 0xe4, 0xb9, 0xa6, 0xe8, 0xb5, 0x8e, 0xe5, 0xad, 0xb0, 0xe7,
  0x86, 0x9f, 0xe8, 0x96, 0xaf, 0xe6, 0x9a, 0x91, 0xe6, 0x9b, 0x99, 0xe7,
  0xbd, 0xb2, 0xe8, 0x9c, 0x80, 0xe9, 0xbb, 0x8d, 0xe9, 0xbc, 0xa0, 0xe5,
  0xb1, 0x9e, 0xe6, 0x9c, 0xaf, 0xe8, 0xbf, 0xb0, 0xe6, 0xa0, 0x91, 0xe6,
  0x9d, 0x9f, 0xe6, 0x88, 0x8d, 0xe7, 0xab, 0x96, 0xe5, 0xa2, 0x85, 0xe5,
  0xba, 0xb6, 0xe6, 0x95, 0xb0, 0xe6, 0xbc, 0xb1
};

unsigned char gbk_row_0xcb_converted_to_utf8[] = {
  0xe8, 0x96, 0x82,
  0xe8, 0x96, 0x83, 0xe8, 0x96, 0x86, 0xe8, 0x96, 0x88, 0xe8, 0x96, 0x89,
  0xe8, 0x96, 0x8a, 0xe8, 0x96, 0x8b, 0xe8, 0x96, 0x8c, 0xe8, 0x96, 0x8d,
  0xe8, 0x96, 0x8e, 0xe8, 0x96, 0x90, 0xe8, 0x96, 0x91, 0xe8, 0x96, 0x92,
  0xe8, 0x96, 0x93, 0xe8, 0x96, 0x94, 0xe8, 0x96, 0x95, 0xe8, 0x96, 0x96,
  0xe8, 0x96, 0x97, 0xe8, 0x96, 0x98, 0xe8, 0x96, 0x99, 0xe8, 0x96, 0x9a,
  0xe8, 0x96, 0x9d, 0xe8, 0x96, 0x9e, 0xe8, 0x96, 0x9f, 0xe8, 0x96, 0xa0,
  0xe8, 0x96, 0xa1, 0xe8, 0x96, 0xa2, 0xe8, 0x96, 0xa3, 0xe8, 0x96, 0xa5,
  0xe8, 0x96, 0xa6, 0xe8, 0x96, 0xa7, 0xe8, 0x96, 0xa9, 0xe8, 0x96, 0xab,
  0xe8, 0x96, 0xac, 0xe8, 0x96, 0xad, 0xe8, 0x96, 0xb1, 0xe8, 0x96, 0xb2,
  0xe8, 0x96, 0xb3, 0xe8, 0x96, 0xb4, 0xe8, 0x96, 0xb5, 0xe8, 0x96, 0xb6,
  0xe8, 0x96, 0xb8, 0xe8, 0x96, 0xba, 0xe8, 0x96, 0xbb, 0xe8, 0x96, 0xbc,
  0xe8, 0x96, 0xbd, 0xe8, 0x96, 0xbe, 0xe8, 0x96, 0xbf, 0xe8, 0x97, 0x80,
  0xe8, 0x97, 0x82, 0xe8, 0x97, 0x83, 0xe8, 0x97, 0x84, 0xe8, 0x97, 0x85,
  0xe8, 0x97, 0x86, 0xe8, 0x97, 0x87, 0xe8, 0x97, 0x88, 0xe8, 0x97, 0x8a,
  0xe8, 0x97, 0x8b, 0xe8, 0x97, 0x8c, 0xe8, 0x97, 0x8d, 0xe8, 0x97, 0x8e,
  0xe8, 0x97, 0x91, 0xe8, 0x97, 0x92, 0xe8, 0x97, 0x94, 0xe8, 0x97, 0x96,
  0xe8, 0x97, 0x97, 0xe8, 0x97, 0x98, 0xe8, 0x97, 0x99, 0xe8, 0x97, 0x9a,
  0xe8, 0x97, 0x9b, 0xe8, 0x97, 0x9d, 0xe8, 0x97, 0x9e, 0xe8, 0x97, 0x9f,
  0xe8, 0x97, 0xa0, 0xe8, 0x97, 0xa1, 0xe8, 0x97, 0xa2, 0xe8, 0x97, 0xa3,
  0xe8, 0x97, 0xa5, 0xe8, 0x97, 0xa6, 0xe8, 0x97, 0xa7, 0xe8, 0x97, 0xa8,
  0xe8, 0x97, 0xaa, 0xe8, 0x97, 0xab, 0xe8, 0x97, 0xac, 0xe8, 0x97, 0xad,
  0xe8, 0x97, 0xae, 0xe8, 0x97, 0xaf, 0xe8, 0x97, 0xb0, 0xe8, 0x97, 0xb1,
  0xe8, 0x97, 0xb2, 0xe8, 0x97, 0xb3, 0xe8, 0x97, 0xb4, 0xe8, 0x97, 0xb5,
  0xe8, 0x97, 0xb6, 0xe8, 0x97, 0xb7, 0xe8, 0x97, 0xb8, 0xe6, 0x81, 0x95,
  0xe5, 0x88, 0xb7, 0xe8, 0x80, 0x8d, 0xe6, 0x91, 0x94, 0xe8, 0xa1, 0xb0,
  0xe7, 0x94, 0xa9, 0xe5, 0xb8, 0x85, 0xe6, 0xa0, 0x93, 0xe6, 0x8b, 0xb4,
  0xe9, 0x9c, 0x9c, 0xe5, 0x8f, 0x8c, 0xe7, 0x88, 0xbd, 0xe8, 0xb0, 0x81,
  0xe6, 0xb0, 0xb4, 0xe7, 0x9d, 0xa1, 0xe7, 0xa8, 0x8e, 0xe5, 0x90, 0xae,
  0xe7, 0x9e, 0xac, 0xe9, 0xa1, 0xba, 0xe8, 0x88, 0x9c, 0xe8, 0xaf, 0xb4,
  0xe7, 0xa1, 0x95, 0xe6, 0x9c, 0x94, 0xe7, 0x83, 0x81, 0xe6, 0x96, 0xaf,
  0xe6, 0x92, 0x95, 0xe5, 0x98, 0xb6, 0xe6, 0x80, 0x9d, 0xe7, 0xa7, 0x81,
  0xe5, 0x8f, 0xb8, 0xe4, 0xb8, 0x9d, 0xe6, 0xad, 0xbb, 0xe8, 0x82, 0x86,
  0xe5, 0xaf, 0xba, 0xe5, 0x97, 0xa3, 0xe5, 0x9b, 0x9b, 0xe4, 0xbc, 0xba,
  0xe4, 0xbc, 0xbc, 0xe9, 0xa5, 0xb2, 0xe5, 0xb7, 0xb3, 0xe6, 0x9d, 0xbe,
  0xe8, 0x80, 0xb8, 0xe6, 0x80, 0x82, 0xe9, 0xa2, 0x82, 0xe9, 0x80, 0x81,
  0xe5, 0xae, 0x8b, 0xe8, 0xae, 0xbc, 0xe8, 0xaf, 0xb5, 0xe6, 0x90, 0x9c,
  0xe8, 0x89, 0x98, 0xe6, 0x93, 0x9e, 0xe5, 0x97, 0xbd, 0xe8, 0x8b, 0x8f,
  0xe9, 0x85, 0xa5, 0xe4, 0xbf, 0x97, 0xe7, 0xb4, 0xa0, 0xe9, 0x80, 0x9f,
  0xe7, 0xb2, 0x9f, 0xe5, 0x83, 0xb3, 0xe5, 0xa1, 0x91, 0xe6, 0xba, 0xaf,
  0xe5, 0xae, 0xbf, 0xe8, 0xaf, 0x89, 0xe8, 0x82, 0x83, 0xe9, 0x85, 0xb8,
  0xe8, 0x92, 0x9c, 0xe7, 0xae, 0x97, 0xe8, 0x99, 0xbd, 0xe9, 0x9a, 0x8b,
  0xe9, 0x9a, 0x8f, 0xe7, 0xbb, 0xa5, 0xe9, 0xab, 0x93, 0xe7, 0xa2, 0x8e,
  0xe5, 0xb2, 0x81, 0xe7, 0xa9, 0x97, 0xe9, 0x81, 0x82, 0xe9, 0x9a, 0xa7,
  0xe7, 0xa5, 0x9f, 0xe5, 0xad, 0x99, 0xe6, 0x8d, 0x9f, 0xe7, 0xac, 0x8b,
  0xe8, 0x93, 0x91, 0xe6, 0xa2, 0xad, 0xe5, 0x94, 0x86, 0xe7, 0xbc, 0xa9,
  0xe7, 0x90, 0x90, 0xe7, 0xb4, 0xa2, 0xe9, 0x94, 0x81, 0xe6, 0x89, 0x80,
  0xe5, 0xa1, 0x8c, 0xe4, 0xbb, 0x96, 0xe5, 0xae, 0x83, 0xe5, 0xa5, 0xb9,
  0xe5, 0xa1, 0x94
};

unsigned char gbk_row_0xcc_converted_to_utf8[] = {
  0xe8, 0x97, 0xb9, 0xe8, 0x97, 0xba, 0xe8, 0x97,
  0xbc, 0xe8, 0x97, 0xbd, 0xe8, 0x97, 0xbe, 0xe8, 0x98, 0x80, 0xe8, 0x98,
  0x81, 0xe8, 0x98, 0x82, 0xe8, 0x98, 0x83, 0xe8, 0x98, 0x84, 0xe8, 0x98,
  0x86, 0xe8, 0x98, 0x87, 0xe8, 0x98, 0x88, 0xe8, 0x98, 0x89, 0xe8, 0x98,
  0x8a, 0xe8, 0x98, 0x8b, 0xe8, 0x98, 0x8c, 0xe8, 0x98, 0x8d, 0xe8, 0x98,
  0x8e, 0xe8, 0x98, 0x8f, 0xe8, 0x98, 0x90, 0xe8, 0x98, 0x92, 0xe8, 0x98,
  0x93, 0xe8, 0x98, 0x94, 0xe8, 0x98, 0x95, 0xe8, 0x98, 0x97, 0xe8, 0x98,
  0x98, 0xe8, 0x98, 0x99, 0xe8, 0x98, 0x9a, 0xe8, 0x98, 0x9b, 0xe8, 0x98,
  0x9c, 0xe8, 0x98, 0x9d, 0xe8, 0x98, 0x9e, 0xe8, 0x98, 0x9f, 0xe8, 0x98,
  0xa0, 0xe8, 0x98, 0xa1, 0xe8, 0x98, 0xa2, 0xe8, 0x98, 0xa3, 0xe8, 0x98,
  0xa4, 0xe8, 0x98, 0xa5, 0xe8, 0x98, 0xa6, 0xe8, 0x98, 0xa8, 0xe8, 0x98,
  0xaa, 0xe8, 0x98, 0xab, 0xe8, 0x98, 0xac, 0xe8, 0x98, 0xad, 0xe8, 0x98,
  0xae, 0xe8, 0x98, 0xaf, 0xe8, 0x98, 0xb0, 0xe8, 0x98, 0xb1, 0xe8, 0x98,
  0xb2, 0xe8, 0x98, 0xb3, 0xe8, 0x98, 0xb4, 0xe8, 0x98, 0xb5, 0xe8, 0x98,
  0xb6, 0xe8, 0x98, 0xb7, 0xe8, 0x98, 0xb9, 0xe8, 0x98, 0xba, 0xe8, 0x98,
  0xbb, 0xe8, 0x98, 0xbd, 0xe8, 0x98, 0xbe, 0xe8, 0x98, 0xbf, 0xe8, 0x99,
  0x80, 0xe8, 0x99, 0x81, 0xe8, 0x99, 0x82, 0xe8, 0x99, 0x83, 0xe8, 0x99,
  0x84, 0xe8, 0x99, 0x85, 0xe8, 0x99, 0x86, 0xe8, 0x99, 0x87, 0xe8, 0x99,
  0x88, 0xe8, 0x99, 0x89, 0xe8, 0x99, 0x8a, 0xe8, 0x99, 0x8b, 0xe8, 0x99,
  0x8c, 0xe8, 0x99, 0x92, 0xe8, 0x99, 0x93, 0xe8, 0x99, 0x95, 0xe8, 0x99,
  0x96, 0xe8, 0x99, 0x97, 0xe8, 0x99, 0x98, 0xe8, 0x99, 0x99, 0xe8, 0x99,
  0x9b, 0xe8, 0x99, 0x9c, 0xe8, 0x99, 0x9d, 0xe8, 0x99, 0x9f, 0xe8, 0x99,
  0xa0, 0xe8, 0x99, 0xa1, 0xe8, 0x99, 0xa3, 0xe8, 0x99, 0xa4, 0xe8, 0x99,
  0xa5, 0xe8, 0x99, 0xa6, 0xe8, 0x99, 0xa7, 0xe8, 0x99, 0xa8, 0xe8, 0x99,
  0xa9, 0xe8, 0x99, 0xaa, 0xe7, 0x8d, 0xad, 0xe6, 0x8c, 0x9e, 0xe8, 0xb9,
  0x8b, 0xe8, 0xb8, 0x8f, 0xe8, 0x83, 0x8e, 0xe8, 0x8b, 0x94, 0xe6, 0x8a,
  0xac, 0xe5, 0x8f, 0xb0, 0xe6, 0xb3, 0xb0, 0xe9, 0x85, 0x9e, 0xe5, 0xa4,
  0xaa, 0xe6, 0x80, 0x81, 0xe6, 0xb1, 0xb0, 0xe5, 0x9d, 0x8d, 0xe6, 0x91,
  0x8a, 0xe8, 0xb4, 0xaa, 0xe7, 0x98, 0xab, 0xe6, 0xbb, 0xa9, 0xe5, 0x9d,
  0x9b, 0xe6, 0xaa, 0x80, 0xe7, 0x97, 0xb0, 0xe6, 0xbd, 0xad, 0xe8, 0xb0,
  0xad, 0xe8, 0xb0, 0x88, 0xe5, 0x9d, 0xa6, 0xe6, 0xaf, 0xaf, 0xe8, 0xa2,
  0x92, 0xe7, 0xa2, 0xb3, 0xe6, 0x8e, 0xa2, 0xe5, 0x8f, 0xb9, 0xe7, 0x82,
  0xad, 0xe6, 0xb1, 0xa4, 0xe5, 0xa1, 0x98, 0xe6, 0x90, 0xaa, 0xe5, 0xa0,
  0x82, 0xe6, 0xa3, 0xa0, 0xe8, 0x86, 0x9b, 0xe5, 0x94, 0x90, 0xe7, 0xb3,
  0x96, 0xe5, 0x80, 0x98, 0xe8, 0xba, 0xba, 0xe6, 0xb7, 0x8c, 0xe8, 0xb6,
  0x9f, 0xe7, 0x83, 0xab, 0xe6, 0x8e, 0x8f, 0xe6, 0xb6, 0x9b, 0xe6, 0xbb,
  0x94, 0xe7, 0xbb, 0xa6, 0xe8, 0x90, 0x84, 0xe6, 0xa1, 0x83, 0xe9, 0x80,
  0x83, 0xe6, 0xb7, 0x98, 0xe9, 0x99, 0xb6, 0xe8, 0xae, 0xa8, 0xe5, 0xa5,
  0x97, 0xe7, 0x89, 0xb9, 0xe8, 0x97, 0xa4, 0xe8, 0x85, 0xbe, 0xe7, 0x96,
  0xbc, 0xe8, 0xaa, 0x8a, 0xe6, 0xa2, 0xaf, 0xe5, 0x89, 0x94, 0xe8, 0xb8,
  0xa2, 0xe9, 0x94, 0x91, 0xe6, 0x8f, 0x90, 0xe9, 0xa2, 0x98, 0xe8, 0xb9,
  0x84, 0xe5, 0x95, 0xbc, 0xe4, 0xbd, 0x93, 0xe6, 0x9b, 0xbf, 0xe5, 0x9a,
  0x8f, 0xe6, 0x83, 0x95, 0xe6, 0xb6, 0x95, 0xe5, 0x89, 0x83, 0xe5, 0xb1,
  0x89, 0xe5, 0xa4, 0xa9, 0xe6, 0xb7, 0xbb, 0xe5, 0xa1, 0xab, 0xe7, 0x94,
  0xb0, 0xe7, 0x94, 0x9c, 0xe6, 0x81, 0xac, 0xe8, 0x88, 0x94, 0xe8, 0x85,
  0x86, 0xe6, 0x8c, 0x91, 0xe6, 0x9d, 0xa1, 0xe8, 0xbf, 0xa2, 0xe7, 0x9c,
  0xba, 0xe8, 0xb7, 0xb3, 0xe8, 0xb4, 0xb4, 0xe9, 0x93, 0x81, 0xe5, 0xb8,
  0x96, 0xe5, 0x8e, 0x85, 0xe5, 0x90, 0xac, 0xe7, 0x83, 0x83
};

unsigned char gbk_row_0xcd_converted_to_utf8[] = {
  0xe8,
  0x99, 0xad, 0xe8, 0x99, 0xaf, 0xe8, 0x99, 0xb0, 0xe8, 0x99, 0xb2, 0xe8,
  0x99, 0xb3, 0xe8, 0x99, 0xb4, 0xe8, 0x99, 0xb5, 0xe8, 0x99, 0xb6, 0xe8,
  0x99, 0xb7, 0xe8, 0x99, 0xb8, 0xe8, 0x9a, 0x83, 0xe8, 0x9a, 0x84, 0xe8,
  0x9a, 0x85, 0xe8, 0x9a, 0x86, 0xe8, 0x9a, 0x87, 0xe8, 0x9a, 0x88, 0xe8,
  0x9a, 0x89, 0xe8, 0x9a, 0x8e, 0xe8, 0x9a, 0x8f, 0xe8, 0x9a, 0x90, 0xe8,
  0x9a, 0x91, 0xe8, 0x9a, 0x92, 0xe8, 0x9a, 0x94, 0xe8, 0x9a, 0x96, 0xe8,
  0x9a, 0x97, 0xe8, 0x9a, 0x98, 0xe8, 0x9a, 0x99, 0xe8, 0x9a, 0x9a, 0xe8,
  0x9a, 0x9b, 0xe8, 0x9a, 0x9e, 0xe8, 0x9a, 0x9f, 0xe8, 0x9a, 0xa0, 0xe8,
  0x9a, 0xa1, 0xe8, 0x9a, 0xa2, 0xe8, 0x9a, 0xa5, 0xe8, 0x9a, 0xa6, 0xe8,
  0x9a, 0xab, 0xe8, 0x9a, 0xad, 0xe8, 0x9a, 0xae, 0xe8, 0x9a, 0xb2, 0xe8,
  0x9a, 0xb3, 0xe8, 0x9a, 0xb7, 0xe8, 0x9a, 0xb8, 0xe8, 0x9a, 0xb9, 0xe8,
  0x9a, 0xbb, 0xe8, 0x9a, 0xbc, 0xe8, 0x9a, 0xbd, 0xe8, 0x9a, 0xbe, 0xe8,
  0x9a, 0xbf, 0xe8, 0x9b, 0x81, 0xe8, 0x9b, 0x82, 0xe8, 0x9b, 0x83, 0xe8,
  0x9b, 0x85, 0xe8, 0x9b, 0x88, 0xe8, 0x9b, 0x8c, 0xe8, 0x9b, 0x8d, 0xe8,
  0x9b, 0x92, 0xe8, 0x9b, 0x93, 0xe8, 0x9b, 0x95, 0xe8, 0x9b, 0x96, 0xe8,
  0x9b, 0x97, 0xe8, 0x9b, 0x9a, 0xe8, 0x9b, 0x9c, 0xe8, 0x9b, 0x9d, 0xe8,
  0x9b, 0xa0, 0xe8, 0x9b, 0xa1, 0xe8, 0x9b, 0xa2, 0xe8, 0x9b, 0xa3, 0xe8,
  0x9b, 0xa5, 0xe8, 0x9b, 0xa6, 0xe8, 0x9b, 0xa7, 0xe8, 0x9b, 0xa8, 0xe8,
  0x9b, 0xaa, 0xe8, 0x9b, 0xab, 0xe8, 0x9b, 0xac, 0xe8, 0x9b, 0xaf, 0xe8,
  0x9b, 0xb5, 0xe8, 0x9b, 0xb6, 0xe8, 0x9b, 0xb7, 0xe8, 0x9b, 0xba, 0xe8,
  0x9b, 0xbb, 0xe8, 0x9b, 0xbc, 0xe8, 0x9b, 0xbd, 0xe8, 0x9b, 0xbf, 0xe8,
  0x9c, 0x81, 0xe8, 0x9c, 0x84, 0xe8, 0x9c, 0x85, 0xe8, 0x9c, 0x86, 0xe8,
  0x9c, 0x8b, 0xe8, 0x9c, 0x8c, 0xe8, 0x9c, 0x8e, 0xe8, 0x9c, 0x8f, 0xe8,
  0x9c, 0x90, 0xe8, 0x9c, 0x91, 0xe8, 0x9c, 0x94, 0xe8, 0x9c, 0x96, 0xe6,
  0xb1, 0x80, 0xe5, 0xbb, 0xb7, 0xe5, 0x81, 0x9c, 0xe4, 0xba, 0xad, 0xe5,
  0xba, 0xad, 0xe6, 0x8c, 0xba, 0xe8, 0x89, 0x87, 0xe9, 0x80, 0x9a, 0xe6,
  0xa1, 0x90, 0xe9, 0x85, 0xae, 0xe7, 0x9e, 0xb3, 0xe5, 0x90, 0x8c, 0xe9,
  0x93, 0x9c, 0xe5, 0xbd, 0xa4, 0xe7, 0xab, 0xa5, 0xe6, 0xa1, 0xb6, 0xe6,
  0x8d, 0x85, 0xe7, 0xad, 0x92, 0xe7, 0xbb, 0x9f, 0xe7, 0x97, 0x9b, 0xe5,
  0x81, 0xb7, 0xe6, 0x8a, 0x95, 0xe5, 0xa4, 0xb4, 0xe9, 0x80, 0x8f, 0xe5,
  0x87, 0xb8, 0xe7, 0xa7, 0x83, 0xe7, 0xaa, 0x81, 0xe5, 0x9b, 0xbe, 0xe5,
  0xbe, 0x92, 0xe9, 0x80, 0x94, 0xe6, 0xb6, 0x82, 0xe5, 0xb1, 0xa0, 0xe5,
  0x9c, 0x9f, 0xe5, 0x90, 0x90, 0xe5, 0x85, 0x94, 0xe6, 0xb9, 0x8d, 0xe5,
  0x9b, 0xa2, 0xe6, 0x8e, 0xa8, 0xe9, 0xa2, 0x93, 0xe8, 0x85, 0xbf, 0xe8,
  0x9c, 0x95, 0xe8, 0xa4, 0xaa, 0xe9, 0x80, 0x80, 0xe5, 0x90, 0x9e, 0xe5,
  0xb1, 0xaf, 0xe8, 0x87, 0x80, 0xe6, 0x8b, 0x96, 0xe6, 0x89, 0x98, 0xe8,
  0x84, 0xb1, 0xe9, 0xb8, 0xb5, 0xe9, 0x99, 0x80, 0xe9, 0xa9, 0xae, 0xe9,
  0xa9, 0xbc, 0xe6, 0xa4, 0xad, 0xe5, 0xa6, 0xa5, 0xe6, 0x8b, 0x93, 0xe5,
  0x94, 0xbe, 0xe6, 0x8c, 0x96, 0xe5, 0x93, 0x87, 0xe8, 0x9b, 0x99, 0xe6,
  0xb4, 0xbc, 0xe5, 0xa8, 0x83, 0xe7, 0x93, 0xa6, 0xe8, 0xa2, 0x9c, 0xe6,
  0xad, 0xaa, 0xe5, 0xa4, 0x96, 0xe8, 0xb1, 0x8c, 0xe5, 0xbc, 0xaf, 0xe6,
  0xb9, 0xbe, 0xe7, 0x8e, 0xa9, 0xe9, 0xa1, 0xbd, 0xe4, 0xb8, 0xb8, 0xe7,
  0x83, 0xb7, 0xe5, 0xae, 0x8c, 0xe7, 0xa2, 0x97, 0xe6, 0x8c, 0xbd, 0xe6,
  0x99, 0x9a, 0xe7, 0x9a, 0x96, 0xe6, 0x83, 0x8b, 0xe5, 0xae, 0x9b, 0xe5,
  0xa9, 0x89, 0xe4, 0xb8, 0x87, 0xe8, 0x85, 0x95, 0xe6, 0xb1, 0xaa, 0xe7,
  0x8e, 0x8b, 0xe4, 0xba, 0xa1, 0xe6, 0x9e, 0x89, 0xe7, 0xbd, 0x91, 0xe5,
  0xbe, 0x80, 0xe6, 0x97, 0xba, 0xe6, 0x9c, 0x9b, 0xe5, 0xbf, 0x98, 0xe5,
  0xa6, 0x84, 0xe5, 0xa8, 0x81
};

unsigned char gbk_row_0xce_converted_to_utf8[] = {
  0xe8, 0x9c, 0x99, 0xe8, 0x9c, 0x9b,
  0xe8, 0x9c, 0x9d, 0xe8, 0x9c, 0x9f, 0xe8, 0x9c, 0xa0, 0xe8, 0x9c, 0xa4,
  0xe8, 0x9c, 0xa6, 0xe8, 0x9c, 0xa7, 0xe8, 0x9c, 0xa8, 0xe8, 0x9c, 0xaa,
  0xe8, 0x9c, 0xab, 0xe8, 0x9c, 0xac, 0xe8, 0x9c, 0xad, 0xe8, 0x9c, 0xaf,
  0xe8, 0x9c, 0xb0, 0xe8, 0x9c, 0xb2, 0xe8, 0x9c, 0xb3, 0xe8, 0x9c, 0xb5,
  0xe8, 0x9c, 0xb6, 0xe8, 0x9c, 0xb8, 0xe8, 0x9c, 0xb9, 0xe8, 0x9c, 0xba,
  0xe8, 0x9c, 0xbc, 0xe8, 0x9c, 0xbd, 0xe8, 0x9d, 0x80, 0xe8, 0x9d, 0x81,
  0xe8, 0x9d, 0x82, 0xe8, 0x9d, 0x83, 0xe8, 0x9d, 0x84, 0xe8, 0x9d, 0x85,
  0xe8, 0x9d, 0x86, 0xe8, 0x9d, 0x8a, 0xe8, 0x9d, 0x8b, 0xe8, 0x9d, 0x8d,
  0xe8, 0x9d, 0x8f, 0xe8, 0x9d, 0x90, 0xe8, 0x9d, 0x91, 0xe8, 0x9d, 0x92,
  0xe8, 0x9d, 0x94, 0xe8, 0x9d, 0x95, 0xe8, 0x9d, 0x96, 0xe8, 0x9d, 0x98,
  0xe8, 0x9d, 0x9a, 0xe8, 0x9d, 0x9b, 0xe8, 0x9d, 0x9c, 0xe8, 0x9d, 0x9d,
  0xe8, 0x9d, 0x9e, 0xe8, 0x9d, 0x9f, 0xe8, 0x9d, 0xa1, 0xe8, 0x9d, 0xa2,
  0xe8, 0x9d, 0xa6, 0xe8, 0x9d, 0xa7, 0xe8, 0x9d, 0xa8, 0xe8, 0x9d, 0xa9,
  0xe8, 0x9d, 0xaa, 0xe8, 0x9d, 0xab, 0xe8, 0x9d, 0xac, 0xe8, 0x9d, 0xad,
  0xe8, 0x9d, 0xaf, 0xe8, 0x9d, 0xb1, 0xe8, 0x9d, 0xb2, 0xe8, 0x9d, 0xb3,
  0xe8, 0x9d, 0xb5, 0xe8, 0x9d, 0xb7, 0xe8, 0x9d, 0xb8, 0xe8, 0x9d, 0xb9,
  0xe8, 0x9d, 0xba, 0xe8, 0x9d, 0xbf, 0xe8, 0x9e, 0x80, 0xe8, 0x9e, 0x81,
  0xe8, 0x9e, 0x84, 0xe8, 0x9e, 0x86, 0xe8, 0x9e, 0x87, 0xe8, 0x9e, 0x89,
  0xe8, 0x9e, 0x8a, 0xe8, 0x9e, 0x8c, 0xe8, 0x9e, 0x8e, 0xe8, 0x9e, 0x8f,
  0xe8, 0x9e, 0x90, 0xe8, 0x9e, 0x91, 0xe8, 0x9e, 0x92, 0xe8, 0x9e, 0x94,
  0xe8, 0x9e, 0x95, 0xe8, 0x9e, 0x96, 0xe8, 0x9e, 0x98, 0xe8, 0x9e, 0x99,
  0xe8, 0x9e, 0x9a, 0xe8, 0x9e, 0x9b, 0xe8, 0x9e, 0x9c, 0xe8, 0x9e, 0x9d,
  0xe8, 0x9e, 0x9e, 0xe8, 0x9e, 0xa0, 0xe8, 0x9e, 0xa1, 0xe8, 0x9e, 0xa2,
  0xe8, 0x9e, 0xa3, 0xe8, 0x9e, 0xa4, 0xe5, 0xb7, 0x8d, 0xe5, 0xbe, 0xae,
  0xe5, 0x8d, 0xb1, 0xe9, 0x9f, 0xa6, 0xe8, 0xbf, 0x9d, 0xe6, 0xa1, 0x85,
  0xe5, 0x9b, 0xb4, 0xe5, 0x94, 0xaf, 0xe6, 0x83, 0x9f, 0xe4, 0xb8, 0xba,
  0xe6, 0xbd, 0x8d, 0xe7, 0xbb, 0xb4, 0xe8, 0x8b, 0x87, 0xe8, 0x90, 0x8e,
  0xe5, 0xa7, 0x94, 0xe4, 0xbc, 0x9f, 0xe4, 0xbc, 0xaa, 0xe5, 0xb0, 0xbe,
  0xe7, 0xba, 0xac, 0xe6, 0x9c, 0xaa, 0xe8, 0x94, 0x9a, 0xe5, 0x91, 0xb3,
  0xe7, 0x95, 0x8f, 0xe8, 0x83, 0x83, 0xe5, 0x96, 0x82, 0xe9, 0xad, 0x8f,
  0xe4, 0xbd, 0x8d, 0xe6, 0xb8, 0xad, 0xe8, 0xb0, 0x93, 0xe5, 0xb0, 0x89,
  0xe6, 0x85, 0xb0, 0xe5, 0x8d, 0xab, 0xe7, 0x98, 0x9f, 0xe6, 0xb8, 0xa9,
  0xe8, 0x9a, 0x8a, 0xe6, 0x96, 0x87, 0xe9, 0x97, 0xbb, 0xe7, 0xba, 0xb9,
  0xe5, 0x90, 0xbb, 0xe7, 0xa8, 0xb3, 0xe7, 0xb4, 0x8a, 0xe9, 0x97, 0xae,
  0xe5, 0x97, 0xa1, 0xe7, 0xbf, 0x81, 0xe7, 0x93, 0xae, 0xe6, 0x8c, 0x9d,
  0xe8, 0x9c, 0x97, 0xe6, 0xb6, 0xa1, 0xe7, 0xaa, 0x9d, 0xe6, 0x88, 0x91,
  0xe6, 0x96, 0xa1, 0xe5, 0x8d, 0xa7, 0xe6, 0x8f, 0xa1, 0xe6, 0xb2, 0x83,
  0xe5, 0xb7, 0xab, 0xe5, 0x91, 0x9c, 0xe9, 0x92, 0xa8, 0xe4, 0xb9, 0x8c,
  0xe6, 0xb1, 0xa1, 0xe8, 0xaf, 0xac, 0xe5, 0xb1, 0x8b, 0xe6, 0x97, 0xa0,
  0xe8, 0x8a, 0x9c, 0xe6, 0xa2, 0xa7, 0xe5, 0x90, 0xbe, 0xe5, 0x90, 0xb4,
  0xe6, 0xaf, 0x8b, 0xe6, 0xad, 0xa6, 0xe4, 0xba, 0x94, 0xe6, 0x8d, 0x82,
  0xe5, 0x8d, 0x88, 0xe8, 0x88, 0x9e, 0xe4, 0xbc, 0x8d, 0xe4, 0xbe, 0xae,
  0xe5, 0x9d, 0x9e, 0xe6, 0x88, 0x8a, 0xe9, 0x9b, 0xbe, 0xe6, 0x99, 0xa4,
  0xe7, 0x89, 0xa9, 0xe5, 0x8b, 0xbf, 0xe5, 0x8a, 0xa1, 0xe6, 0x82, 0x9f,
  0xe8, 0xaf, 0xaf, 0xe6, 0x98, 0x94, 0xe7, 0x86, 0x99, 0xe6, 0x9e, 0x90,
  0xe8, 0xa5, 0xbf, 0xe7, 0xa1, 0x92, 0xe7, 0x9f, 0xbd, 0xe6, 0x99, 0xb0,
  0xe5, 0x98, 0xbb, 0xe5, 0x90, 0xb8, 0xe9, 0x94, 0xa1, 0xe7, 0x89, 0xba,
};

unsigned char gbk_row_0xcf_converted_to_utf8[] = {
        0xe8, 0x9e, 0xa5, 0xe8, 0x9e, 0xa6, 0xe8, 0x9e, 0xa7, 0xe8, 0x9e,
  0xa9, 0xe8, 0x9e, 0xaa, 0xe8, 0x9e, 0xae, 0xe8, 0x9e, 0xb0, 0xe8, 0x9e,
  0xb1, 0xe8, 0x9e, 0xb2, 0xe8, 0x9e, 0xb4, 0xe8, 0x9e, 0xb6, 0xe8, 0x9e,
  0xb7, 0xe8, 0x9e, 0xb8, 0xe8, 0x9e, 0xb9, 0xe8, 0x9e, 0xbb, 0xe8, 0x9e,
  0xbc, 0xe8, 0x9e, 0xbe, 0xe8, 0x9e, 0xbf, 0xe8, 0x9f, 0x81, 0xe8, 0x9f,
  0x82, 0xe8, 0x9f, 0x83, 0xe8, 0x9f, 0x84, 0xe8, 0x9f, 0x85, 0xe8, 0x9f,
  0x87, 0xe8, 0x9f, 0x88, 0xe8, 0x9f, 0x89, 0xe8, 0x9f, 0x8c, 0xe8, 0x9f,
  0x8d, 0xe8, 0x9f, 0x8e, 0xe8, 0x9f, 0x8f, 0xe8, 0x9f, 0x90, 0xe8, 0x9f,
  0x94, 0xe8, 0x9f, 0x95, 0xe8, 0x9f, 0x96, 0xe8, 0x9f, 0x97, 0xe8, 0x9f,
  0x98, 0xe8, 0x9f, 0x99, 0xe8, 0x9f, 0x9a, 0xe8, 0x9f, 0x9c, 0xe8, 0x9f,
  0x9d, 0xe8, 0x9f, 0x9e, 0xe8, 0x9f, 0x9f, 0xe8, 0x9f, 0xa1, 0xe8, 0x9f,
  0xa2, 0xe8, 0x9f, 0xa3, 0xe8, 0x9f, 0xa4, 0xe8, 0x9f, 0xa6, 0xe8, 0x9f,
  0xa7, 0xe8, 0x9f, 0xa8, 0xe8, 0x9f, 0xa9, 0xe8, 0x9f, 0xab, 0xe8, 0x9f,
  0xac, 0xe8, 0x9f, 0xad, 0xe8, 0x9f, 0xaf, 0xe8, 0x9f, 0xb0, 0xe8, 0x9f,
  0xb1, 0xe8, 0x9f, 0xb2, 0xe8, 0x9f, 0xb3, 0xe8, 0x9f, 0xb4, 0xe8, 0x9f,
  0xb5, 0xe8, 0x9f, 0xb6, 0xe8, 0x9f, 0xb7, 0xe8, 0x9f, 0xb8, 0xe8, 0x9f,
  0xba, 0xe8, 0x9f, 0xbb, 0xe8, 0x9f, 0xbc, 0xe8, 0x9f, 0xbd, 0xe8, 0x9f,
  0xbf, 0xe8, 0xa0, 0x80, 0xe8, 0xa0, 0x81, 0xe8, 0xa0, 0x82, 0xe8, 0xa0,
  0x84, 0xe8, 0xa0, 0x85, 0xe8, 0xa0, 0x86, 0xe8, 0xa0, 0x87, 0xe8, 0xa0,
  0x88, 0xe8, 0xa0, 0x89, 0xe8, 0xa0, 0x8b, 0xe8, 0xa0, 0x8c, 0xe8, 0xa0,
  0x8d, 0xe8, 0xa0, 0x8e, 0xe8, 0xa0, 0x8f, 0xe8, 0xa0, 0x90, 0xe8, 0xa0,
  0x91, 0xe8, 0xa0, 0x92, 0xe8, 0xa0, 0x94, 0xe8, 0xa0, 0x97, 0xe8, 0xa0,
  0x98, 0xe8, 0xa0, 0x99, 0xe8, 0xa0, 0x9a, 0xe8, 0xa0, 0x9c, 0xe8, 0xa0,
  0x9d, 0xe8, 0xa0, 0x9e, 0xe8, 0xa0, 0x9f, 0xe8, 0xa0, 0xa0, 0xe8, 0xa0,
  0xa3, 0xe7, 0xa8, 0x80, 0xe6, 0x81, 0xaf, 0xe5, 0xb8, 0x8c, 0xe6, 0x82,
  0x89, 0xe8, 0x86, 0x9d, 0xe5, 0xa4, 0x95, 0xe6, 0x83, 0x9c, 0xe7, 0x86,
  0x84, 0xe7, 0x83, 0xaf, 0xe6, 0xba, 0xaa, 0xe6, 0xb1, 0x90, 0xe7, 0x8a,
  0x80, 0xe6, 0xaa, 0x84, 0xe8, 0xa2, 0xad, 0xe5, 0xb8, 0xad, 0xe4, 0xb9,
  0xa0, 0xe5, 0xaa, 0xb3, 0xe5, 0x96, 0x9c, 0xe9, 0x93, 0xa3, 0xe6, 0xb4,
  0x97, 0xe7, 0xb3, 0xbb, 0xe9, 0x9a, 0x99, 0xe6, 0x88, 0x8f, 0xe7, 0xbb,
  0x86, 0xe7, 0x9e, 0x8e, 0xe8, 0x99, 0xbe, 0xe5, 0x8c, 0xa3, 0xe9, 0x9c,
  0x9e, 0xe8, 0xbe, 0x96, 0xe6, 0x9a, 0x87, 0xe5, 0xb3, 0xa1, 0xe4, 0xbe,
  0xa0, 0xe7, 0x8b, 0xad, 0xe4, 0xb8, 0x8b, 0xe5, 0x8e, 0xa6, 0xe5, 0xa4,
  0x8f, 0xe5, 0x90, 0x93, 0xe6, 0x8e, 0x80, 0xe9, 0x94, 0xa8, 0xe5, 0x85,
  0x88, 0xe4, 0xbb, 0x99, 0xe9, 0xb2, 0x9c, 0xe7, 0xba, 0xa4, 0xe5, 0x92,
  0xb8, 0xe8, 0xb4, 0xa4, 0xe8, 0xa1, 0x94, 0xe8, 0x88, 0xb7, 0xe9, 0x97,
  0xb2, 0xe6, 0xb6, 0x8e, 0xe5, 0xbc, 0xa6, 0xe5, 0xab, 0x8c, 0xe6, 0x98,
  0xbe, 0xe9, 0x99, 0xa9, 0xe7, 0x8e, 0xb0, 0xe7, 0x8c, 0xae, 0xe5, 0x8e,
  0xbf, 0xe8, 0x85, 0xba, 0xe9, 0xa6, 0x85, 0xe7, 0xbe, 0xa1, 0xe5, 0xae,
  0xaa, 0xe9, 0x99, 0xb7, 0xe9, 0x99, 0x90, 0xe7, 0xba, 0xbf, 0xe7, 0x9b,
  0xb8, 0xe5, 0x8e, 0xa2, 0xe9, 0x95, 0xb6, 0xe9, 0xa6, 0x99, 0xe7, 0xae,
  0xb1, 0xe8, 0xa5, 0x84, 0xe6, 0xb9, 0x98, 0xe4, 0xb9, 0xa1, 0xe7, 0xbf,
  0x94, 0xe7, 0xa5, 0xa5, 0xe8, 0xaf, 0xa6, 0xe6, 0x83, 0xb3, 0xe5, 0x93,
  0x8d, 0xe4, 0xba, 0xab, 0xe9, 0xa1, 0xb9, 0xe5, 0xb7, 0xb7, 0xe6, 0xa9,
  0xa1, 0xe5, 0x83, 0x8f, 0xe5, 0x90, 0x91, 0xe8, 0xb1, 0xa1, 0xe8, 0x90,
  0xa7, 0xe7, 0xa1, 0x9d, 0xe9, 0x9c, 0x84, 0xe5, 0x89, 0x8a, 0xe5, 0x93,
  0xae, 0xe5, 0x9a, 0xa3, 0xe9, 0x94, 0x80, 0xe6, 0xb6, 0x88, 0xe5, 0xae,
  0xb5, 0xe6, 0xb7, 0x86, 0xe6, 0x99, 0x93
};

unsigned char gbk_row_0xd0_converted_to_utf8[] = {
  0xe8, 0xa0, 0xa4, 0xe8,
  0xa0, 0xa5, 0xe8, 0xa0, 0xa6, 0xe8, 0xa0, 0xa7, 0xe8, 0xa0, 0xa8, 0xe8,
  0xa0, 0xa9, 0xe8, 0xa0, 0xaa, 0xe8, 0xa0, 0xab, 0xe8, 0xa0, 0xac, 0xe8,
  0xa0, 0xad, 0xe8, 0xa0, 0xae, 0xe8, 0xa0, 0xaf, 0xe8, 0xa0, 0xb0, 0xe8,
  0xa0, 0xb1, 0xe8, 0xa0, 0xb3, 0xe8, 0xa0, 0xb4, 0xe8, 0xa0, 0xb5, 0xe8,
  0xa0, 0xb6, 0xe8, 0xa0, 0xb7, 0xe8, 0xa0, 0xb8, 0xe8, 0xa0, 0xba, 0xe8,
  0xa0, 0xbb, 0xe8, 0xa0, 0xbd, 0xe8, 0xa0, 0xbe, 0xe8, 0xa0, 0xbf, 0xe8,
  0xa1, 0x81, 0xe8, 0xa1, 0x82, 0xe8, 0xa1, 0x83, 0xe8, 0xa1, 0x86, 0xe8,
  0xa1, 0x87, 0xe8, 0xa1, 0x88, 0xe8, 0xa1, 0x89, 0xe8, 0xa1, 0x8a, 0xe8,
  0xa1, 0x8b, 0xe8, 0xa1, 0x8e, 0xe8, 0xa1, 0x8f, 0xe8, 0xa1, 0x90, 0xe8,
  0xa1, 0x91, 0xe8, 0xa1, 0x92, 0xe8, 0xa1, 0x93, 0xe8, 0xa1, 0x95, 0xe8,
  0xa1, 0x96, 0xe8, 0xa1, 0x98, 0xe8, 0xa1, 0x9a, 0xe8, 0xa1, 0x9b, 0xe8,
  0xa1, 0x9c, 0xe8, 0xa1, 0x9d, 0xe8, 0xa1, 0x9e, 0xe8, 0xa1, 0x9f, 0xe8,
  0xa1, 0xa0, 0xe8, 0xa1, 0xa6, 0xe8, 0xa1, 0xa7, 0xe8, 0xa1, 0xaa, 0xe8,
  0xa1, 0xad, 0xe8, 0xa1, 0xaf, 0xe8, 0xa1, 0xb1, 0xe8, 0xa1, 0xb3, 0xe8,
  0xa1, 0xb4, 0xe8, 0xa1, 0xb5, 0xe8, 0xa1, 0xb6, 0xe8, 0xa1, 0xb8, 0xe8,
  0xa1, 0xb9, 0xe8, 0xa1, 0xba, 0xe8, 0xa1, 0xbb, 0xe8, 0xa1, 0xbc, 0xe8,
  0xa2, 0x80, 0xe8, 0xa2, 0x83, 0xe8, 0xa2, 0x86, 0xe8, 0xa2, 0x87, 0xe8,
  0xa2, 0x89, 0xe8, 0xa2, 0x8a, 0xe8, 0xa2, 0x8c, 0xe8, 0xa2, 0x8e, 0xe8,
  0xa2, 0x8f, 0xe8, 0xa2, 0x90, 0xe8, 0xa2, 0x91, 0xe8, 0xa2, 0x93, 0xe8,
  0xa2, 0x94, 0xe8, 0xa2, 0x95, 0xe8, 0xa2, 0x97, 0xe8, 0xa2, 0x98, 0xe8,
  0xa2, 0x99, 0xe8, 0xa2, 0x9a, 0xe8, 0xa2, 0x9b, 0xe8, 0xa2, 0x9d, 0xe8,
  0xa2, 0x9e, 0xe8, 0xa2, 0x9f, 0xe8, 0xa2, 0xa0, 0xe8, 0xa2, 0xa1, 0xe8,
  0xa2, 0xa3, 0xe8, 0xa2, 0xa5, 0xe8, 0xa2, 0xa6, 0xe8, 0xa2, 0xa7, 0xe8,
  0xa2, 0xa8, 0xe8, 0xa2, 0xa9, 0xe8, 0xa2, 0xaa, 0xe5, 0xb0, 0x8f, 0xe5,
  0xad, 0x9d, 0xe6, 0xa0, 0xa1, 0xe8, 0x82, 0x96, 0xe5, 0x95, 0xb8, 0xe7,
  0xac, 0x91, 0xe6, 0x95, 0x88, 0xe6, 0xa5, 0x94, 0xe4, 0xba, 0x9b, 0xe6,
  0xad, 0x87, 0xe8, 0x9d, 0x8e, 0xe9, 0x9e, 0x8b, 0xe5, 0x8d, 0x8f, 0xe6,
  0x8c, 0x9f, 0xe6, 0x90, 0xba, 0xe9, 0x82, 0xaa, 0xe6, 0x96, 0x9c, 0xe8,
  0x83, 0x81, 0xe8, 0xb0, 0x90, 0xe5, 0x86, 0x99, 0xe6, 0xa2, 0xb0, 0xe5,
  0x8d, 0xb8, 0xe8, 0x9f, 0xb9, 0xe6, 0x87, 0x88, 0xe6, 0xb3, 0x84, 0xe6,
  0xb3, 0xbb, 0xe8, 0xb0, 0xa2, 0xe5, 0xb1, 0x91, 0xe8, 0x96, 0xaa, 0xe8,
  0x8a, 0xaf, 0xe9, 0x94, 0x8c, 0xe6, 0xac, 0xa3, 0xe8, 0xbe, 0x9b, 0xe6,
  0x96, 0xb0, 0xe5, 0xbf, 0xbb, 0xe5, 0xbf, 0x83, 0xe4, 0xbf, 0xa1, 0xe8,
  0xa1, 0x85, 0xe6, 0x98, 0x9f, 0xe8, 0x85, 0xa5, 0xe7, 0x8c, 0xa9, 0xe6,
  0x83, 0xba, 0xe5, 0x85, 0xb4, 0xe5, 0x88, 0x91, 0xe5, 0x9e, 0x8b, 0xe5,
  0xbd, 0xa2, 0xe9, 0x82, 0xa2, 0xe8, 0xa1, 0x8c, 0xe9, 0x86, 0x92, 0xe5,
  0xb9, 0xb8, 0xe6, 0x9d, 0x8f, 0xe6, 0x80, 0xa7, 0xe5, 0xa7, 0x93, 0xe5,
  0x85, 0x84, 0xe5, 0x87, 0xb6, 0xe8, 0x83, 0xb8, 0xe5, 0x8c, 0x88, 0xe6,
  0xb1, 0xb9, 0xe9, 0x9b, 0x84, 0xe7, 0x86, 0x8a, 0xe4, 0xbc, 0x91, 0xe4,
  0xbf, 0xae, 0xe7, 0xbe, 0x9e, 0xe6, 0x9c, 0xbd, 0xe5, 0x97, 0x85, 0xe9,
  0x94, 0x88, 0xe7, 0xa7, 0x80, 0xe8, 0xa2, 0x96, 0xe7, 0xbb, 0xa3, 0xe5,
  0xa2, 0x9f, 0xe6, 0x88, 0x8c, 0xe9, 0x9c, 0x80, 0xe8, 0x99, 0x9a, 0xe5,
  0x98, 0x98, 0xe9, 0xa1, 0xbb, 0xe5, 0xbe, 0x90, 0xe8, 0xae, 0xb8, 0xe8,
  0x93, 0x84, 0xe9, 0x85, 0x97, 0xe5, 0x8f, 0x99, 0xe6, 0x97, 0xad, 0xe5,
  0xba, 0x8f, 0xe7, 0x95, 0x9c, 0xe6, 0x81, 0xa4, 0xe7, 0xb5, 0xae, 0xe5,
  0xa9, 0xbf, 0xe7, 0xbb, 0xaa, 0xe7, 0xbb, 0xad, 0xe8, 0xbd, 0xa9, 0xe5,
  0x96, 0xa7, 0xe5, 0xae, 0xa3, 0xe6, 0x82, 0xac, 0xe6, 0x97, 0x8b, 0xe7,
  0x8e, 0x84
};

unsigned char gbk_row_0xd1_converted_to_utf8[] = {
  0xe8, 0xa2, 0xac, 0xe8, 0xa2, 0xae, 0xe8, 0xa2, 0xaf,
  0xe8, 0xa2, 0xb0, 0xe8, 0xa2, 0xb2, 0xe8, 0xa2, 0xb3, 0xe8, 0xa2, 0xb4,
  0xe8, 0xa2, 0xb5, 0xe8, 0xa2, 0xb6, 0xe8, 0xa2, 0xb8, 0xe8, 0xa2, 0xb9,
  0xe8, 0xa2, 0xba, 0xe8, 0xa2, 0xbb, 0xe8, 0xa2, 0xbd, 0xe8, 0xa2, 0xbe,
  0xe8, 0xa2, 0xbf, 0xe8, 0xa3, 0x80, 0xe8, 0xa3, 0x83, 0xe8, 0xa3, 0x84,
  0xe8, 0xa3, 0x87, 0xe8, 0xa3, 0x88, 0xe8, 0xa3, 0x8a, 0xe8, 0xa3, 0x8b,
  0xe8, 0xa3, 0x8c, 0xe8, 0xa3, 0x8d, 0xe8, 0xa3, 0x8f, 0xe8, 0xa3, 0x90,
  0xe8, 0xa3, 0x91, 0xe8, 0xa3, 0x93, 0xe8, 0xa3, 0x96, 0xe8, 0xa3, 0x97,
  0xe8, 0xa3, 0x9a, 0xe8, 0xa3, 0x9b, 0xe8, 0xa3, 0x9c, 0xe8, 0xa3, 0x9d,
  0xe8, 0xa3, 0x9e, 0xe8, 0xa3, 0xa0, 0xe8, 0xa3, 0xa1, 0xe8, 0xa3, 0xa6,
  0xe8, 0xa3, 0xa7, 0xe8, 0xa3, 0xa9, 0xe8, 0xa3, 0xaa, 0xe8, 0xa3, 0xab,
  0xe8, 0xa3, 0xac, 0xe8, 0xa3, 0xad, 0xe8, 0xa3, 0xae, 0xe8, 0xa3, 0xaf,
  0xe8, 0xa3, 0xb2, 0xe8, 0xa3, 0xb5, 0xe8, 0xa3, 0xb6, 0xe8, 0xa3, 0xb7,
  0xe8, 0xa3, 0xba, 0xe8, 0xa3, 0xbb, 0xe8, 0xa3, 0xbd, 0xe8, 0xa3, 0xbf,
  0xe8, 0xa4, 0x80, 0xe8, 0xa4, 0x81, 0xe8, 0xa4, 0x83, 0xe8, 0xa4, 0x84,
  0xe8, 0xa4, 0x85, 0xe8, 0xa4, 0x86, 0xe8, 0xa4, 0x87, 0xe8, 0xa4, 0x88,
  0xe8, 0xa4, 0x89, 0xe8, 0xa4, 0x8b, 0xe8, 0xa4, 0x8c, 0xe8, 0xa4, 0x8d,
  0xe8, 0xa4, 0x8e, 0xe8, 0xa4, 0x8f, 0xe8, 0xa4, 0x91, 0xe8, 0xa4, 0x94,
  0xe8, 0xa4, 0x95, 0xe8, 0xa4, 0x96, 0xe8, 0xa4, 0x97, 0xe8, 0xa4, 0x98,
  0xe8, 0xa4, 0x9c, 0xe8, 0xa4, 0x9d, 0xe8, 0xa4, 0x9e, 0xe8, 0xa4, 0x9f,
  0xe8, 0xa4, 0xa0, 0xe8, 0xa4, 0xa2, 0xe8, 0xa4, 0xa3, 0xe8, 0xa4, 0xa4,
  0xe8, 0xa4, 0xa6, 0xe8, 0xa4, 0xa7, 0xe8, 0xa4, 0xa8, 0xe8, 0xa4, 0xa9,
  0xe8, 0xa4, 0xac, 0xe8, 0xa4, 0xad, 0xe8, 0xa4, 0xae, 0xe8, 0xa4, 0xaf,
  0xe8, 0xa4, 0xb1, 0xe8, 0xa4, 0xb2, 0xe8, 0xa4, 0xb3, 0xe8, 0xa4, 0xb5,
  0xe8, 0xa4, 0xb7, 0xe9, 0x80, 0x89, 0xe7, 0x99, 0xa3, 0xe7, 0x9c, 0xa9,
  0xe7, 0xbb, 0x9a, 0xe9, 0x9d, 0xb4, 0xe8, 0x96, 0x9b, 0xe5, 0xad, 0xa6,
  0xe7, 0xa9, 0xb4, 0xe9, 0x9b, 0xaa, 0xe8, 0xa1, 0x80, 0xe5, 0x8b, 0x8b,
  0xe7, 0x86, 0x8f, 0xe5, 0xbe, 0xaa, 0xe6, 0x97, 0xac, 0xe8, 0xaf, 0xa2,
  0xe5, 0xaf, 0xbb, 0xe9, 0xa9, 0xaf, 0xe5, 0xb7, 0xa1, 0xe6, 0xae, 0x89,
  0xe6, 0xb1, 0x9b, 0xe8, 0xae, 0xad, 0xe8, 0xae, 0xaf, 0xe9, 0x80, 0x8a,
  0xe8, 0xbf, 0x85, 0xe5, 0x8e, 0x8b, 0xe6, 0x8a, 0xbc, 0xe9, 0xb8, 0xa6,
  0xe9, 0xb8, 0xad, 0xe5, 0x91, 0x80, 0xe4, 0xb8, 0xab, 0xe8, 0x8a, 0xbd,
  0xe7, 0x89, 0x99, 0xe8, 0x9a, 0x9c, 0xe5, 0xb4, 0x96, 0xe8, 0xa1, 0x99,
  0xe6, 0xb6, 0xaf, 0xe9, 0x9b, 0x85, 0xe5, 0x93, 0x91, 0xe4, 0xba, 0x9a,
  0xe8, 0xae, 0xb6, 0xe7, 0x84, 0x89, 0xe5, 0x92, 0xbd, 0xe9, 0x98, 0x89,
  0xe7, 0x83, 0x9f, 0xe6, 0xb7, 0xb9, 0xe7, 0x9b, 0x90, 0xe4, 0xb8, 0xa5,
  0xe7, 0xa0, 0x94, 0xe8, 0x9c, 0x92, 0xe5, 0xb2, 0xa9, 0xe5, 0xbb, 0xb6,
  0xe8, 0xa8, 0x80, 0xe9, 0xa2, 0x9c, 0xe9, 0x98, 0x8e, 0xe7, 0x82, 0x8e,
  0xe6, 0xb2, 0xbf, 0xe5, 0xa5, 0x84, 0xe6, 0x8e, 0xa9, 0xe7, 0x9c, 0xbc,
  0xe8, 0xa1, 0x8d, 0xe6, 0xbc, 0x94, 0xe8, 0x89, 0xb3, 0xe5, 0xa0, 0xb0,
  0xe7, 0x87, 0x95, 0xe5, 0x8e, 0x8c, 0xe7, 0xa0, 0x9a, 0xe9, 0x9b, 0x81,
  0xe5, 0x94, 0x81, 0xe5, 0xbd, 0xa6, 0xe7, 0x84, 0xb0, 0xe5, 0xae, 0xb4,
  0xe8, 0xb0, 0x9a, 0xe9, 0xaa, 0x8c, 0xe6, 0xae, 0x83, 0xe5, 0xa4, 0xae,
  0xe9, 0xb8, 0xaf, 0xe7, 0xa7, 0xa7, 0xe6, 0x9d, 0xa8, 0xe6, 0x89, 0xac,
  0xe4, 0xbd, 0xaf, 0xe7, 0x96, 0xa1, 0xe7, 0xbe, 0x8a, 0xe6, 0xb4, 0x8b,
  0xe9, 0x98, 0xb3, 0xe6, 0xb0, 0xa7, 0xe4, 0xbb, 0xb0, 0xe7, 0x97, 0x92,
  0xe5, 0x85, 0xbb, 0xe6, 0xa0, 0xb7, 0xe6, 0xbc, 0xbe, 0xe9, 0x82, 0x80,
  0xe8, 0x85, 0xb0, 0xe5, 0xa6, 0x96, 0xe7, 0x91, 0xb6
};

unsigned char gbk_row_0xd2_converted_to_utf8[] = {
  0xe8, 0xa4,
  0xb8, 0xe8, 0xa4, 0xb9, 0xe8, 0xa4, 0xba, 0xe8, 0xa4, 0xbb, 0xe8, 0xa4,
  0xbc, 0xe8, 0xa4, 0xbd, 0xe8, 0xa4, 0xbe, 0xe8, 0xa4, 0xbf, 0xe8, 0xa5,
  0x80, 0xe8, 0xa5, 0x82, 0xe8, 0xa5, 0x83, 0xe8, 0xa5, 0x85, 0xe8, 0xa5,
  0x86, 0xe8, 0xa5, 0x87, 0xe8, 0xa5, 0x88, 0xe8, 0xa5, 0x89, 0xe8, 0xa5,
  0x8a, 0xe8, 0xa5, 0x8b, 0xe8, 0xa5, 0x8c, 0xe8, 0xa5, 0x8d, 0xe8, 0xa5,
  0x8e, 0xe8, 0xa5, 0x8f, 0xe8, 0xa5, 0x90, 0xe8, 0xa5, 0x91, 0xe8, 0xa5,
  0x92, 0xe8, 0xa5, 0x93, 0xe8, 0xa5, 0x94, 0xe8, 0xa5, 0x95, 0xe8, 0xa5,
  0x96, 0xe8, 0xa5, 0x97, 0xe8, 0xa5, 0x98, 0xe8, 0xa5, 0x99, 0xe8, 0xa5,
  0x9a, 0xe8, 0xa5, 0x9b, 0xe8, 0xa5, 0x9c, 0xe8, 0xa5, 0x9d, 0xe8, 0xa5,
  0xa0, 0xe8, 0xa5, 0xa1, 0xe8, 0xa5, 0xa2, 0xe8, 0xa5, 0xa3, 0xe8, 0xa5,
  0xa4, 0xe8, 0xa5, 0xa5, 0xe8, 0xa5, 0xa7, 0xe8, 0xa5, 0xa8, 0xe8, 0xa5,
  0xa9, 0xe8, 0xa5, 0xaa, 0xe8, 0xa5, 0xab, 0xe8, 0xa5, 0xac, 0xe8, 0xa5,
  0xad, 0xe8, 0xa5, 0xae, 0xe8, 0xa5, 0xaf, 0xe8, 0xa5, 0xb0, 0xe8, 0xa5,
  0xb1, 0xe8, 0xa5, 0xb2, 0xe8, 0xa5, 0xb3, 0xe8, 0xa5, 0xb4, 0xe8, 0xa5,
  0xb5, 0xe8, 0xa5, 0xb6, 0xe8, 0xa5, 0xb7, 0xe8, 0xa5, 0xb8, 0xe8, 0xa5,
  0xb9, 0xe8, 0xa5, 0xba, 0xe8, 0xa5, 0xbc, 0xe8, 0xa5, 0xbd, 0xe8, 0xa5,
  0xbe, 0xe8, 0xa6, 0x80, 0xe8, 0xa6, 0x82, 0xe8, 0xa6, 0x84, 0xe8, 0xa6,
  0x85, 0xe8, 0xa6, 0x87, 0xe8, 0xa6, 0x88, 0xe8, 0xa6, 0x89, 0xe8, 0xa6,
  0x8a, 0xe8, 0xa6, 0x8b, 0xe8, 0xa6, 0x8c, 0xe8, 0xa6, 0x8d, 0xe8, 0xa6,
  0x8e, 0xe8, 0xa6, 0x8f, 0xe8, 0xa6, 0x90, 0xe8, 0xa6, 0x91, 0xe8, 0xa6,
  0x92, 0xe8, 0xa6, 0x93, 0xe8, 0xa6, 0x94, 0xe8, 0xa6, 0x95, 0xe8, 0xa6,
  0x96, 0xe8, 0xa6, 0x97, 0xe8, 0xa6, 0x98, 0xe8, 0xa6, 0x99, 0xe8, 0xa6,
  0x9a, 0xe8, 0xa6, 0x9b, 0xe8, 0xa6, 0x9c, 0xe8, 0xa6, 0x9d, 0xe8, 0xa6,
  0x9e, 0xe8, 0xa6, 0x9f, 0xe8, 0xa6, 0xa0, 0xe8, 0xa6, 0xa1, 0xe6, 0x91,
  0x87, 0xe5, 0xb0, 0xa7, 0xe9, 0x81, 0xa5, 0xe7, 0xaa, 0x91, 0xe8, 0xb0,
  0xa3, 0xe5, 0xa7, 0x9a, 0xe5, 0x92, 0xac, 0xe8, 0x88, 0x80, 0xe8, 0x8d,
  0xaf, 0xe8, 0xa6, 0x81, 0xe8, 0x80, 0x80, 0xe6, 0xa4, 0xb0, 0xe5, 0x99,
  0x8e, 0xe8, 0x80, 0xb6, 0xe7, 0x88, 0xb7, 0xe9, 0x87, 0x8e, 0xe5, 0x86,
  0xb6, 0xe4, 0xb9, 0x9f, 0xe9, 0xa1, 0xb5, 0xe6, 0x8e, 0x96, 0xe4, 0xb8,
  0x9a, 0xe5, 0x8f, 0xb6, 0xe6, 0x9b, 0xb3, 0xe8, 0x85, 0x8b, 0xe5, 0xa4,
  0x9c, 0xe6, 0xb6, 0xb2, 0xe4, 0xb8, 0x80, 0xe5, 0xa3, 0xb9, 0xe5, 0x8c,
  0xbb, 0xe6, 0x8f, 0x96, 0xe9, 0x93, 0xb1, 0xe4, 0xbe, 0x9d, 0xe4, 0xbc,
  0x8a, 0xe8, 0xa1, 0xa3, 0xe9, 0xa2, 0x90, 0xe5, 0xa4, 0xb7, 0xe9, 0x81,
  0x97, 0xe7, 0xa7, 0xbb, 0xe4, 0xbb, 0xaa, 0xe8, 0x83, 0xb0, 0xe7, 0x96,
  0x91, 0xe6, 0xb2, 0x82, 0xe5, 0xae, 0x9c, 0xe5, 0xa7, 0xa8, 0xe5, 0xbd,
  0x9d, 0xe6, 0xa4, 0x85, 0xe8, 0x9a, 0x81, 0xe5, 0x80, 0x9a, 0xe5, 0xb7,
  0xb2, 0xe4, 0xb9, 0x99, 0xe7, 0x9f, 0xa3, 0xe4, 0xbb, 0xa5, 0xe8, 0x89,
  0xba, 0xe6, 0x8a, 0x91, 0xe6, 0x98, 0x93, 0xe9, 0x82, 0x91, 0xe5, 0xb1,
  0xb9, 0xe4, 0xba, 0xbf, 0xe5, 0xbd, 0xb9, 0xe8, 0x87, 0x86, 0xe9, 0x80,
  0xb8, 0xe8, 0x82, 0x84, 0xe7, 0x96, 0xab, 0xe4, 0xba, 0xa6, 0xe8, 0xa3,
  0x94, 0xe6, 0x84, 0x8f, 0xe6, 0xaf, 0x85, 0xe5, 0xbf, 0x86, 0xe4, 0xb9,
  0x89, 0xe7, 0x9b, 0x8a, 0xe6, 0xba, 0xa2, 0xe8, 0xaf, 0xa3, 0xe8, 0xae,
  0xae, 0xe8, 0xb0, 0x8a, 0xe8, 0xaf, 0x91, 0xe5, 0xbc, 0x82, 0xe7, 0xbf,
  0xbc, 0xe7, 0xbf, 0x8c, 0xe7, 0xbb, 0x8e, 0xe8, 0x8c, 0xb5, 0xe8, 0x8d,
  0xab, 0xe5, 0x9b, 0xa0, 0xe6, 0xae, 0xb7, 0xe9, 0x9f, 0xb3, 0xe9, 0x98,
  0xb4, 0xe5, 0xa7, 0xbb, 0xe5, 0x90, 0x9f, 0xe9, 0x93, 0xb6, 0xe6, 0xb7,
  0xab, 0xe5, 0xaf, 0x85, 0xe9, 0xa5, 0xae, 0xe5, 0xb0, 0xb9, 0xe5, 0xbc,
  0x95, 0xe9, 0x9a, 0x90
};

unsigned char gbk_row_0xd3_converted_to_utf8[] = {
  0xe8, 0xa6, 0xa2, 0xe8, 0xa6, 0xa3, 0xe8,
  0xa6, 0xa4, 0xe8, 0xa6, 0xa5, 0xe8, 0xa6, 0xa6, 0xe8, 0xa6, 0xa7, 0xe8,
  0xa6, 0xa8, 0xe8, 0xa6, 0xa9, 0xe8, 0xa6, 0xaa, 0xe8, 0xa6, 0xab, 0xe8,
  0xa6, 0xac, 0xe8, 0xa6, 0xad, 0xe8, 0xa6, 0xae, 0xe8, 0xa6, 0xaf, 0xe8,
  0xa6, 0xb0, 0xe8, 0xa6, 0xb1, 0xe8, 0xa6, 0xb2, 0xe8, 0xa6, 0xb3, 0xe8,
  0xa6, 0xb4, 0xe8, 0xa6, 0xb5, 0xe8, 0xa6, 0xb6, 0xe8, 0xa6, 0xb7, 0xe8,
  0xa6, 0xb8, 0xe8, 0xa6, 0xb9, 0xe8, 0xa6, 0xba, 0xe8, 0xa6, 0xbb, 0xe8,
  0xa6, 0xbc, 0xe8, 0xa6, 0xbd, 0xe8, 0xa6, 0xbe, 0xe8, 0xa6, 0xbf, 0xe8,
  0xa7, 0x80, 0xe8, 0xa7, 0x83, 0xe8, 0xa7, 0x8d, 0xe8, 0xa7, 0x93, 0xe8,
  0xa7, 0x94, 0xe8, 0xa7, 0x95, 0xe8, 0xa7, 0x97, 0xe8, 0xa7, 0x98, 0xe8,
  0xa7, 0x99, 0xe8, 0xa7, 0x9b, 0xe8, 0xa7, 0x9d, 0xe8, 0xa7, 0x9f, 0xe8,
  0xa7, 0xa0, 0xe8, 0xa7, 0xa1, 0xe8, 0xa7, 0xa2, 0xe8, 0xa7, 0xa4, 0xe8,
  0xa7, 0xa7, 0xe8, 0xa7, 0xa8, 0xe8, 0xa7, 0xa9, 0xe8, 0xa7, 0xaa, 0xe8,
  0xa7, 0xac, 0xe8, 0xa7, 0xad, 0xe8, 0xa7, 0xae, 0xe8, 0xa7, 0xb0, 0xe8,
  0xa7, 0xb1, 0xe8, 0xa7, 0xb2, 0xe8, 0xa7, 0xb4, 0xe8, 0xa7, 0xb5, 0xe8,
  0xa7, 0xb6, 0xe8, 0xa7, 0xb7, 0xe8, 0xa7, 0xb8, 0xe8, 0xa7, 0xb9, 0xe8,
  0xa7, 0xba, 0xe8, 0xa7, 0xbb, 0xe8, 0xa7, 0xbc, 0xe8, 0xa7, 0xbd, 0xe8,
  0xa7, 0xbe, 0xe8, 0xa7, 0xbf, 0xe8, 0xa8, 0x81, 0xe8, 0xa8, 0x82, 0xe8,
  0xa8, 0x83, 0xe8, 0xa8, 0x84, 0xe8, 0xa8, 0x85, 0xe8, 0xa8, 0x86, 0xe8,
  0xa8, 0x88, 0xe8, 0xa8, 0x89, 0xe8, 0xa8, 0x8a, 0xe8, 0xa8, 0x8b, 0xe8,
  0xa8, 0x8c, 0xe8, 0xa8, 0x8d, 0xe8, 0xa8, 0x8e, 0xe8, 0xa8, 0x8f, 0xe8,
  0xa8, 0x90, 0xe8, 0xa8, 0x91, 0xe8, 0xa8, 0x92, 0xe8, 0xa8, 0x93, 0xe8,
  0xa8, 0x94, 0xe8, 0xa8, 0x95, 0xe8, 0xa8, 0x96, 0xe8, 0xa8, 0x97, 0xe8,
  0xa8, 0x98, 0xe8, 0xa8, 0x99, 0xe8, 0xa8, 0x9a, 0xe8, 0xa8, 0x9b, 0xe8,
  0xa8, 0x9c, 0xe8, 0xa8, 0x9d, 0xe5, 0x8d, 0xb0, 0xe8, 0x8b, 0xb1, 0xe6,
  0xa8, 0xb1, 0xe5, 0xa9, 0xb4, 0xe9, 0xb9, 0xb0, 0xe5, 0xba, 0x94, 0xe7,
  0xbc, 0xa8, 0xe8, 0x8e, 0xb9, 0xe8, 0x90, 0xa4, 0xe8, 0x90, 0xa5, 0xe8,
  0x8d, 0xa7, 0xe8, 0x9d, 0x87, 0xe8, 0xbf, 0x8e, 0xe8, 0xb5, 0xa2, 0xe7,
  0x9b, 0x88, 0xe5, 0xbd, 0xb1, 0xe9, 0xa2, 0x96, 0xe7, 0xa1, 0xac, 0xe6,
  0x98, 0xa0, 0xe5, 0x93, 0x9f, 0xe6, 0x8b, 0xa5, 0xe4, 0xbd, 0xa3, 0xe8,
  0x87, 0x83, 0xe7, 0x97, 0x88, 0xe5, 0xba, 0xb8, 0xe9, 0x9b, 0x8d, 0xe8,
  0xb8, 0x8a, 0xe8, 0x9b, 0xb9, 0xe5, 0x92, 0x8f, 0xe6, 0xb3, 0xb3, 0xe6,
  0xb6, 0x8c, 0xe6, 0xb0, 0xb8, 0xe6, 0x81, 0xbf, 0xe5, 0x8b, 0x87, 0xe7,
  0x94, 0xa8, 0xe5, 0xb9, 0xbd, 0xe4, 0xbc, 0x98, 0xe6, 0x82, 0xa0, 0xe5,
  0xbf, 0xa7, 0xe5, 0xb0, 0xa4, 0xe7, 0x94, 0xb1, 0xe9, 0x82, 0xae, 0xe9,
  0x93, 0x80, 0xe7, 0x8a, 0xb9, 0xe6, 0xb2, 0xb9, 0xe6, 0xb8, 0xb8, 0xe9,
  0x85, 0x89, 0xe6, 0x9c, 0x89, 0xe5, 0x8f, 0x8b, 0xe5, 0x8f, 0xb3, 0xe4,
  0xbd, 0x91, 0xe9, 0x87, 0x89, 0xe8, 0xaf, 0xb1, 0xe5, 0x8f, 0x88, 0xe5,
  0xb9, 0xbc, 0xe8, 0xbf, 0x82, 0xe6, 0xb7, 0xa4, 0xe4, 0xba, 0x8e, 0xe7,
  0x9b, 0x82, 0xe6, 0xa6, 0x86, 0xe8, 0x99, 0x9e, 0xe6, 0x84, 0x9a, 0xe8,
  0x88, 0x86, 0xe4, 0xbd, 0x99, 0xe4, 0xbf, 0x9e, 0xe9, 0x80, 0xbe, 0xe9,
  0xb1, 0xbc, 0xe6, 0x84, 0x89, 0xe6, 0xb8, 0x9d, 0xe6, 0xb8, 0x94, 0xe9,
  0x9a, 0x85, 0xe4, 0xba, 0x88, 0xe5, 0xa8, 0xb1, 0xe9, 0x9b, 0xa8, 0xe4,
  0xb8, 0x8e, 0xe5, 0xb1, 0xbf, 0xe7, 0xa6, 0xb9, 0xe5, 0xae, 0x87, 0xe8,
  0xaf, 0xad, 0xe7, 0xbe, 0xbd, 0xe7, 0x8e, 0x89, 0xe5, 0x9f, 0x9f, 0xe8,
  0x8a, 0x8b, 0xe9, 0x83, 0x81, 0xe5, 0x90, 0x81, 0xe9, 0x81, 0x87, 0xe5,
  0x96, 0xbb, 0xe5, 0xb3, 0xaa, 0xe5, 0xbe, 0xa1, 0xe6, 0x84, 0x88, 0xe6,
  0xac, 0xb2, 0xe7, 0x8b, 0xb1, 0xe8, 0x82, 0xb2, 0xe8, 0xaa, 0x89
};

unsigned char gbk_row_0xd4_converted_to_utf8[] = {
  0xe8, 0xa8, 0x9e, 0xe8, 0xa8, 0x9f, 0xe8, 0xa8, 0xa0, 0xe8, 0xa8, 0xa1,
  0xe8, 0xa8, 0xa2, 0xe8, 0xa8, 0xa3, 0xe8, 0xa8, 0xa4, 0xe8, 0xa8, 0xa5,
  0xe8, 0xa8, 0xa6, 0xe8, 0xa8, 0xa7, 0xe8, 0xa8, 0xa8, 0xe8, 0xa8, 0xa9,
  0xe8, 0xa8, 0xaa, 0xe8, 0xa8, 0xab, 0xe8, 0xa8, 0xac, 0xe8, 0xa8, 0xad,
  0xe8, 0xa8, 0xae, 0xe8, 0xa8, 0xaf, 0xe8, 0xa8, 0xb0, 0xe8, 0xa8, 0xb1,
  0xe8, 0xa8, 0xb2, 0xe8, 0xa8, 0xb3, 0xe8, 0xa8, 0xb4, 0xe8, 0xa8, 0xb5,
  0xe8, 0xa8, 0xb6, 0xe8, 0xa8, 0xb7, 0xe8, 0xa8, 0xb8, 0xe8, 0xa8, 0xb9,
  0xe8, 0xa8, 0xba, 0xe8, 0xa8, 0xbb, 0xe8, 0xa8, 0xbc, 0xe8, 0xa8, 0xbd,
  0xe8, 0xa8, 0xbf, 0xe8, 0xa9, 0x80, 0xe8, 0xa9, 0x81, 0xe8, 0xa9, 0x82,
  0xe8, 0xa9, 0x83, 0xe8, 0xa9, 0x84, 0xe8, 0xa9, 0x85, 0xe8, 0xa9, 0x86,
  0xe8, 0xa9, 0x87, 0xe8, 0xa9, 0x89, 0xe8, 0xa9, 0x8a, 0xe8, 0xa9, 0x8b,
  0xe8, 0xa9, 0x8c, 0xe8, 0xa9, 0x8d, 0xe8, 0xa9, 0x8e, 0xe8, 0xa9, 0x8f,
  0xe8, 0xa9, 0x90, 0xe8, 0xa9, 0x91, 0xe8, 0xa9, 0x92, 0xe8, 0xa9, 0x93,
  0xe8, 0xa9, 0x94, 0xe8, 0xa9, 0x95, 0xe8, 0xa9, 0x96, 0xe8, 0xa9, 0x97,
  0xe8, 0xa9, 0x98, 0xe8, 0xa9, 0x99, 0xe8, 0xa9, 0x9a, 0xe8, 0xa9, 0x9b,
  0xe8, 0xa9, 0x9c, 0xe8, 0xa9, 0x9d, 0xe8, 0xa9, 0x9e, 0xe8, 0xa9, 0x9f,
  0xe8, 0xa9, 0xa0, 0xe8, 0xa9, 0xa1, 0xe8, 0xa9, 0xa2, 0xe8, 0xa9, 0xa3,
  0xe8, 0xa9, 0xa4, 0xe8, 0xa9, 0xa5, 0xe8, 0xa9, 0xa6, 0xe8, 0xa9, 0xa7,
  0xe8, 0xa9, 0xa8, 0xe8, 0xa9, 0xa9, 0xe8, 0xa9, 0xaa, 0xe8, 0xa9, 0xab,
  0xe8, 0xa9, 0xac, 0xe8, 0xa9, 0xad, 0xe8, 0xa9, 0xae, 0xe8, 0xa9, 0xaf,
  0xe8, 0xa9, 0xb0, 0xe8, 0xa9, 0xb1, 0xe8, 0xa9, 0xb2, 0xe8, 0xa9, 0xb3,
  0xe8, 0xa9, 0xb4, 0xe8, 0xa9, 0xb5, 0xe8, 0xa9, 0xb6, 0xe8, 0xa9, 0xb7,
  0xe8, 0xa9, 0xb8, 0xe8, 0xa9, 0xba, 0xe8, 0xa9, 0xbb, 0xe8, 0xa9, 0xbc,
  0xe8, 0xa9, 0xbd, 0xe8, 0xa9, 0xbe, 0xe8, 0xa9, 0xbf, 0xe8, 0xaa, 0x80,
  0xe6, 0xb5, 0xb4, 0xe5, 0xaf, 0x93, 0xe8, 0xa3, 0x95, 0xe9, 0xa2, 0x84,
  0xe8, 0xb1, 0xab, 0xe9, 0xa9, 0xad, 0xe9, 0xb8, 0xb3, 0xe6, 0xb8, 0x8a,
  0xe5, 0x86, 0xa4, 0xe5, 0x85, 0x83, 0xe5, 0x9e, 0xa3, 0xe8, 0xa2, 0x81,
  0xe5, 0x8e, 0x9f, 0xe6, 0x8f, 0xb4, 0xe8, 0xbe, 0x95, 0xe5, 0x9b, 0xad,
  0xe5, 0x91, 0x98, 0xe5, 0x9c, 0x86, 0xe7, 0x8c, 0xbf, 0xe6, 0xba, 0x90,
  0xe7, 0xbc, 0x98, 0xe8, 0xbf, 0x9c, 0xe8, 0x8b, 0x91, 0xe6, 0x84, 0xbf,
  0xe6, 0x80, 0xa8, 0xe9, 0x99, 0xa2, 0xe6, 0x9b, 0xb0, 0xe7, 0xba, 0xa6,
  0xe8, 0xb6, 0x8a, 0xe8, 0xb7, 0x83, 0xe9, 0x92, 0xa5, 0xe5, 0xb2, 0xb3,
  0xe7, 0xb2, 0xa4, 0xe6, 0x9c, 0x88, 0xe6, 0x82, 0xa6, 0xe9, 0x98, 0x85,
  0xe8, 0x80, 0x98, 0xe4, 0xba, 0x91, 0xe9, 0x83, 0xa7, 0xe5, 0x8c, 0x80,
  0xe9, 0x99, 0xa8, 0xe5, 0x85, 0x81, 0xe8, 0xbf, 0x90, 0xe8, 0x95, 0xb4,
  0xe9, 0x85, 0x9d, 0xe6, 0x99, 0x95, 0xe9, 0x9f, 0xb5, 0xe5, 0xad, 0x95,
  0xe5, 0x8c, 0x9d, 0xe7, 0xa0, 0xb8, 0xe6, 0x9d, 0x82, 0xe6, 0xa0, 0xbd,
  0xe5, 0x93, 0x89, 0xe7, 0x81, 0xbe, 0xe5, 0xae, 0xb0, 0xe8, 0xbd, 0xbd,
  0xe5, 0x86, 0x8d, 0xe5, 0x9c, 0xa8, 0xe5, 0x92, 0xb1, 0xe6, 0x94, 0x92,
  0xe6, 0x9a, 0x82, 0xe8, 0xb5, 0x9e, 0xe8, 0xb5, 0x83, 0xe8, 0x84, 0x8f,
  0xe8, 0x91, 0xac, 0xe9, 0x81, 0xad, 0xe7, 0xb3, 0x9f, 0xe5, 0x87, 0xbf,
  0xe8, 0x97, 0xbb, 0xe6, 0x9e, 0xa3, 0xe6, 0x97, 0xa9, 0xe6, 0xbe, 0xa1,
  0xe8, 0x9a, 0xa4, 0xe8, 0xba, 0x81, 0xe5, 0x99, 0xaa, 0xe9, 0x80, 0xa0,
  0xe7, 0x9a, 0x82, 0xe7, 0x81, 0xb6, 0xe7, 0x87, 0xa5, 0xe8, 0xb4, 0xa3,
  0xe6, 0x8b, 0xa9, 0xe5, 0x88, 0x99, 0xe6, 0xb3, 0xbd, 0xe8, 0xb4, 0xbc,
  0xe6, 0x80, 0x8e, 0xe5, 0xa2, 0x9e, 0xe6, 0x86, 0x8e, 0xe6, 0x9b, 0xbe,
  0xe8, 0xb5, 0xa0, 0xe6, 0x89, 0x8e, 0xe5, 0x96, 0xb3, 0xe6, 0xb8, 0xa3,
  0xe6, 0x9c, 0xad, 0xe8, 0xbd, 0xa7
};

unsigned char gbk_row_0xd5_converted_to_utf8[] = {
  0xe8, 0xaa, 0x81, 0xe8, 0xaa,
  0x82, 0xe8, 0xaa, 0x83, 0xe8, 0xaa, 0x84, 0xe8, 0xaa, 0x85, 0xe8, 0xaa,
  0x86, 0xe8, 0xaa, 0x87, 0xe8, 0xaa, 0x88, 0xe8, 0xaa, 0x8b, 0xe8, 0xaa,
  0x8c, 0xe8, 0xaa, 0x8d, 0xe8, 0xaa, 0x8e, 0xe8, 0xaa, 0x8f, 0xe8, 0xaa,
  0x90, 0xe8, 0xaa, 0x91, 0xe8, 0xaa, 0x92, 0xe8, 0xaa, 0x94, 0xe8, 0xaa,
  0x95, 0xe8, 0xaa, 0x96, 0xe8, 0xaa, 0x97, 0xe8, 0xaa, 0x98, 0xe8, 0xaa,
  0x99, 0xe8, 0xaa, 0x9a, 0xe8, 0xaa, 0x9b, 0xe8, 0xaa, 0x9c, 0xe8, 0xaa,
  0x9d, 0xe8, 0xaa, 0x9e, 0xe8, 0xaa, 0x9f, 0xe8, 0xaa, 0xa0, 0xe8, 0xaa,
  0xa1, 0xe8, 0xaa, 0xa2, 0xe8, 0xaa, 0xa3, 0xe8, 0xaa, 0xa4, 0xe8, 0xaa,
  0xa5, 0xe8, 0xaa, 0xa6, 0xe8, 0xaa, 0xa7, 0xe8, 0xaa, 0xa8, 0xe8, 0xaa,
  0xa9, 0xe8, 0xaa, 0xaa, 0xe8, 0xaa, 0xab, 0xe8, 0xaa, 0xac, 0xe8, 0xaa,
  0xad, 0xe8, 0xaa, 0xae, 0xe8, 0xaa, 0xaf, 0xe8, 0xaa, 0xb0, 0xe8, 0xaa,
  0xb1, 0xe8, 0xaa, 0xb2, 0xe8, 0xaa, 0xb3, 0xe8, 0xaa, 0xb4, 0xe8, 0xaa,
  0xb5, 0xe8, 0xaa, 0xb6, 0xe8, 0xaa, 0xb7, 0xe8, 0xaa, 0xb8, 0xe8, 0xaa,
  0xb9, 0xe8, 0xaa, 0xba, 0xe8, 0xaa, 0xbb, 0xe8, 0xaa, 0xbc, 0xe8, 0xaa,
  0xbd, 0xe8, 0xaa, 0xbe, 0xe8, 0xaa, 0xbf, 0xe8, 0xab, 0x80, 0xe8, 0xab,
  0x81, 0xe8, 0xab, 0x82, 0xe8, 0xab, 0x83, 0xe8, 0xab, 0x84, 0xe8, 0xab,
  0x85, 0xe8, 0xab, 0x86, 0xe8, 0xab, 0x87, 0xe8, 0xab, 0x88, 0xe8, 0xab,
  0x89, 0xe8, 0xab, 0x8a, 0xe8, 0xab, 0x8b, 0xe8, 0xab, 0x8c, 0xe8, 0xab,
  0x8d, 0xe8, 0xab, 0x8e, 0xe8, 0xab, 0x8f, 0xe8, 0xab, 0x90, 0xe8, 0xab,
  0x91, 0xe8, 0xab, 0x92, 0xe8, 0xab, 0x93, 0xe8, 0xab, 0x94, 0xe8, 0xab,
  0x95, 0xe8, 0xab, 0x96, 0xe8, 0xab, 0x97, 0xe8, 0xab, 0x98, 0xe8, 0xab,
  0x99, 0xe8, 0xab, 0x9a, 0xe8, 0xab, 0x9b, 0xe8, 0xab, 0x9c, 0xe8, 0xab,
  0x9d, 0xe8, 0xab, 0x9e, 0xe8, 0xab, 0x9f, 0xe8, 0xab, 0xa0, 0xe8, 0xab,
  0xa1, 0xe8, 0xab, 0xa2, 0xe8, 0xab, 0xa3, 0xe9, 0x93, 0xa1, 0xe9, 0x97,
  0xb8, 0xe7, 0x9c, 0xa8, 0xe6, 0xa0, 0x85, 0xe6, 0xa6, 0xa8, 0xe5, 0x92,
  0x8b, 0xe4, 0xb9, 0x8d, 0xe7, 0x82, 0xb8, 0xe8, 0xaf, 0x88, 0xe6, 0x91,
  0x98, 0xe6, 0x96, 0x8b, 0xe5, 0xae, 0x85, 0xe7, 0xaa, 0x84, 0xe5, 0x80,
  0xba, 0xe5, 0xaf, 0xa8, 0xe7, 0x9e, 0xbb, 0xe6, 0xaf, 0xa1, 0xe8, 0xa9,
  0xb9, 0xe7, 0xb2, 0x98, 0xe6, 0xb2, 0xbe, 0xe7, 0x9b, 0x8f, 0xe6, 0x96,
  0xa9, 0xe8, 0xbe, 0x97, 0xe5, 0xb4, 0xad, 0xe5, 0xb1, 0x95, 0xe8, 0x98,
  0xb8, 0xe6, 0xa0, 0x88, 0xe5, 0x8d, 0xa0, 0xe6, 0x88, 0x98, 0xe7, 0xab,
  0x99, 0xe6, 0xb9, 0x9b, 0xe7, 0xbb, 0xbd, 0xe6, 0xa8, 0x9f, 0xe7, 0xab,
  0xa0, 0xe5, 0xbd, 0xb0, 0xe6, 0xbc, 0xb3, 0xe5, 0xbc, 0xa0, 0xe6, 0x8e,
  0x8c, 0xe6, 0xb6, 0xa8, 0xe6, 0x9d, 0x96, 0xe4, 0xb8, 0x88, 0xe5, 0xb8,
  0x90, 0xe8, 0xb4, 0xa6, 0xe4, 0xbb, 0x97, 0xe8, 0x83, 0x80, 0xe7, 0x98,
  0xb4, 0xe9, 0x9a, 0x9c, 0xe6, 0x8b, 0x9b, 0xe6, 0x98, 0xad, 0xe6, 0x89,
  0xbe, 0xe6, 0xb2, 0xbc, 0xe8, 0xb5, 0xb5, 0xe7, 0x85, 0xa7, 0xe7, 0xbd,
  0xa9, 0xe5, 0x85, 0x86, 0xe8, 0x82, 0x87, 0xe5, 0x8f, 0xac, 0xe9, 0x81,
  0xae, 0xe6, 0x8a, 0x98, 0xe5, 0x93, 0xb2, 0xe8, 0x9b, 0xb0, 0xe8, 0xbe,
  0x99, 0xe8, 0x80, 0x85, 0xe9, 0x94, 0x97, 0xe8, 0x94, 0x97, 0xe8, 0xbf,
  0x99, 0xe6, 0xb5, 0x99, 0xe7, 0x8f, 0x8d, 0xe6, 0x96, 0x9f, 0xe7, 0x9c,
  0x9f, 0xe7, 0x94, 0x84, 0xe7, 0xa0, 0xa7, 0xe8, 0x87, 0xbb, 0xe8, 0xb4,
  0x9e, 0xe9, 0x92, 0x88, 0xe4, 0xbe, 0xa6, 0xe6, 0x9e, 0x95, 0xe7, 0x96,
  0xb9, 0xe8, 0xaf, 0x8a, 0xe9, 0x9c, 0x87, 0xe6, 0x8c, 0xaf, 0xe9, 0x95,
  0x87, 0xe9, 0x98, 0xb5, 0xe8, 0x92, 0xb8, 0xe6, 0x8c, 0xa3, 0xe7, 0x9d,
  0x81, 0xe5, 0xbe, 0x81, 0xe7, 0x8b, 0xb0, 0xe4, 0xba, 0x89, 0xe6, 0x80,
  0x94, 0xe6, 0x95, 0xb4, 0xe6, 0x8b, 0xaf, 0xe6, 0xad, 0xa3, 0xe6, 0x94,
  0xbf
};

unsigned char gbk_row_0xd6_converted_to_utf8[] = {
  0xe8, 0xab, 0xa4, 0xe8, 0xab, 0xa5, 0xe8, 0xab, 0xa6, 0xe8,
  0xab, 0xa7, 0xe8, 0xab, 0xa8, 0xe8, 0xab, 0xa9, 0xe8, 0xab, 0xaa, 0xe8,
  0xab, 0xab, 0xe8, 0xab, 0xac, 0xe8, 0xab, 0xad, 0xe8, 0xab, 0xae, 0xe8,
  0xab, 0xaf, 0xe8, 0xab, 0xb0, 0xe8, 0xab, 0xb1, 0xe8, 0xab, 0xb2, 0xe8,
  0xab, 0xb3, 0xe8, 0xab, 0xb4, 0xe8, 0xab, 0xb5, 0xe8, 0xab, 0xb6, 0xe8,
  0xab, 0xb7, 0xe8, 0xab, 0xb8, 0xe8, 0xab, 0xb9, 0xe8, 0xab, 0xba, 0xe8,
  0xab, 0xbb, 0xe8, 0xab, 0xbc, 0xe8, 0xab, 0xbd, 0xe8, 0xab, 0xbe, 0xe8,
  0xab, 0xbf, 0xe8, 0xac, 0x80, 0xe8, 0xac, 0x81, 0xe8, 0xac, 0x82, 0xe8,
  0xac, 0x83, 0xe8, 0xac, 0x84, 0xe8, 0xac, 0x85, 0xe8, 0xac, 0x86, 0xe8,
  0xac, 0x88, 0xe8, 0xac, 0x89, 0xe8, 0xac, 0x8a, 0xe8, 0xac, 0x8b, 0xe8,
  0xac, 0x8c, 0xe8, 0xac, 0x8d, 0xe8, 0xac, 0x8e, 0xe8, 0xac, 0x8f, 0xe8,
  0xac, 0x90, 0xe8, 0xac, 0x91, 0xe8, 0xac, 0x92, 0xe8, 0xac, 0x93, 0xe8,
  0xac, 0x94, 0xe8, 0xac, 0x95, 0xe8, 0xac, 0x96, 0xe8, 0xac, 0x97, 0xe8,
  0xac, 0x98, 0xe8, 0xac, 0x99, 0xe8, 0xac, 0x9a, 0xe8, 0xac, 0x9b, 0xe8,
  0xac, 0x9c, 0xe8, 0xac, 0x9d, 0xe8, 0xac, 0x9e, 0xe8, 0xac, 0x9f, 0xe8,
  0xac, 0xa0, 0xe8, 0xac, 0xa1, 0xe8, 0xac, 0xa2, 0xe8, 0xac, 0xa3, 0xe8,
  0xac, 0xa4, 0xe8, 0xac, 0xa5, 0xe8, 0xac, 0xa7, 0xe8, 0xac, 0xa8, 0xe8,
  0xac, 0xa9, 0xe8, 0xac, 0xaa, 0xe8, 0xac, 0xab, 0xe8, 0xac, 0xac, 0xe8,
  0xac, 0xad, 0xe8, 0xac, 0xae, 0xe8, 0xac, 0xaf, 0xe8, 0xac, 0xb0, 0xe8,
  0xac, 0xb1, 0xe8, 0xac, 0xb2, 0xe8, 0xac, 0xb3, 0xe8, 0xac, 0xb4, 0xe8,
  0xac, 0xb5, 0xe8, 0xac, 0xb6, 0xe8, 0xac, 0xb7, 0xe8, 0xac, 0xb8, 0xe8,
  0xac, 0xb9, 0xe8, 0xac, 0xba, 0xe8, 0xac, 0xbb, 0xe8, 0xac, 0xbc, 0xe8,
  0xac, 0xbd, 0xe8, 0xac, 0xbe, 0xe8, 0xac, 0xbf, 0xe8, 0xad, 0x80, 0xe8,
  0xad, 0x81, 0xe8, 0xad, 0x82, 0xe8, 0xad, 0x83, 0xe8, 0xad, 0x84, 0xe8,
  0xad, 0x85, 0xe5, 0xb8, 0xa7, 0xe7, 0x97, 0x87, 0xe9, 0x83, 0x91, 0xe8,
  0xaf, 0x81, 0xe8, 0x8a, 0x9d, 0xe6, 0x9e, 0x9d, 0xe6, 0x94, 0xaf, 0xe5,
  0x90, 0xb1, 0xe8, 0x9c, 0x98, 0xe7, 0x9f, 0xa5, 0xe8, 0x82, 0xa2, 0xe8,
  0x84, 0x82, 0xe6, 0xb1, 0x81, 0xe4, 0xb9, 0x8b, 0xe7, 0xbb, 0x87, 0xe8,
  0x81, 0x8c, 0xe7, 0x9b, 0xb4, 0xe6, 0xa4, 0x8d, 0xe6, 0xae, 0x96, 0xe6,
  0x89, 0xa7, 0xe5, 0x80, 0xbc, 0xe4, 0xbe, 0x84, 0xe5, 0x9d, 0x80, 0xe6,
  0x8c, 0x87, 0xe6, 0xad, 0xa2, 0xe8, 0xb6, 0xbe, 0xe5, 0x8f, 0xaa, 0xe6,
  0x97, 0xa8, 0xe7, 0xba, 0xb8, 0xe5, 0xbf, 0x97, 0xe6, 0x8c, 0x9a, 0xe6,
  0x8e, 0xb7, 0xe8, 0x87, 0xb3, 0xe8, 0x87, 0xb4, 0xe7, 0xbd, 0xae, 0xe5,
  0xb8, 0x9c, 0xe5, 0xb3, 0x99, 0xe5, 0x88, 0xb6, 0xe6, 0x99, 0xba, 0xe7,
  0xa7, 0xa9, 0xe7, 0xa8, 0x9a, 0xe8, 0xb4, 0xa8, 0xe7, 0x82, 0x99, 0xe7,
  0x97, 0x94, 0xe6, 0xbb, 0x9e, 0xe6, 0xb2, 0xbb, 0xe7, 0xaa, 0x92, 0xe4,
  0xb8, 0xad, 0xe7, 0x9b, 0x85, 0xe5, 0xbf, 0xa0, 0xe9, 0x92, 0x9f, 0xe8,
  0xa1, 0xb7, 0xe7, 0xbb, 0x88, 0xe7, 0xa7, 0x8d, 0xe8, 0x82, 0xbf, 0xe9,
  0x87, 0x8d, 0xe4, 0xbb, 0xb2, 0xe4, 0xbc, 0x97, 0xe8, 0x88, 0x9f, 0xe5,
  0x91, 0xa8, 0xe5, 0xb7, 0x9e, 0xe6, 0xb4, 0xb2, 0xe8, 0xaf, 0x8c, 0xe7,
  0xb2, 0xa5, 0xe8, 0xbd, 0xb4, 0xe8, 0x82, 0x98, 0xe5, 0xb8, 0x9a, 0xe5,
  0x92, 0x92, 0xe7, 0x9a, 0xb1, 0xe5, 0xae, 0x99, 0xe6, 0x98, 0xbc, 0xe9,
  0xaa, 0xa4, 0xe7, 0x8f, 0xa0, 0xe6, 0xa0, 0xaa, 0xe8, 0x9b, 0x9b, 0xe6,
  0x9c, 0xb1, 0xe7, 0x8c, 0xaa, 0xe8, 0xaf, 0xb8, 0xe8, 0xaf, 0x9b, 0xe9,
  0x80, 0x90, 0xe7, 0xab, 0xb9, 0xe7, 0x83, 0x9b, 0xe7, 0x85, 0xae, 0xe6,
  0x8b, 0x84, 0xe7, 0x9e, 0xa9, 0xe5, 0x98, 0xb1, 0xe4, 0xb8, 0xbb, 0xe8,
  0x91, 0x97, 0xe6, 0x9f, 0xb1, 0xe5, 0x8a, 0xa9, 0xe8, 0x9b, 0x80, 0xe8,
  0xb4, 0xae, 0xe9, 0x93, 0xb8, 0xe7, 0xad, 0x91
};

unsigned char gbk_row_0xd7_converted_to_utf8[] = {
  0xe8, 0xad, 0x86,
  0xe8, 0xad, 0x87, 0xe8, 0xad, 0x88, 0xe8, 0xad, 0x89, 0xe8, 0xad, 0x8a,
  0xe8, 0xad, 0x8b, 0xe8, 0xad, 0x8c, 0xe8, 0xad, 0x8d, 0xe8, 0xad, 0x8e,
  0xe8, 0xad, 0x8f, 0xe8, 0xad, 0x90, 0xe8, 0xad, 0x91, 0xe8, 0xad, 0x92,
  0xe8, 0xad, 0x93, 0xe8, 0xad, 0x94, 0xe8, 0xad, 0x95, 0xe8, 0xad, 0x96,
  0xe8, 0xad, 0x97, 0xe8, 0xad, 0x98, 0xe8, 0xad, 0x99, 0xe8, 0xad, 0x9a,
  0xe8, 0xad, 0x9b, 0xe8, 0xad, 0x9c, 0xe8, 0xad, 0x9d, 0xe8, 0xad, 0x9e,
  0xe8, 0xad, 0x9f, 0xe8, 0xad, 0xa0, 0xe8, 0xad, 0xa1, 0xe8, 0xad, 0xa2,
  0xe8, 0xad, 0xa3, 0xe8, 0xad, 0xa4, 0xe8, 0xad, 0xa5, 0xe8, 0xad, 0xa7,
  0xe8, 0xad, 0xa8, 0xe8, 0xad, 0xa9, 0xe8, 0xad, 0xaa, 0xe8, 0xad, 0xab,
  0xe8, 0xad, 0xad, 0xe8, 0xad, 0xae, 0xe8, 0xad, 0xaf, 0xe8, 0xad, 0xb0,
  0xe8, 0xad, 0xb1, 0xe8, 0xad, 0xb2, 0xe8, 0xad, 0xb3, 0xe8, 0xad, 0xb4,
  0xe8, 0xad, 0xb5, 0xe8, 0xad, 0xb6, 0xe8, 0xad, 0xb7, 0xe8, 0xad, 0xb8,
  0xe8, 0xad, 0xb9, 0xe8, 0xad, 0xba, 0xe8, 0xad, 0xbb, 0xe8, 0xad, 0xbc,
  0xe8, 0xad, 0xbd, 0xe8, 0xad, 0xbe, 0xe8, 0xad, 0xbf, 0xe8, 0xae, 0x80,
  0xe8, 0xae, 0x81, 0xe8, 0xae, 0x82, 0xe8, 0xae, 0x83, 0xe8, 0xae, 0x84,
  0xe8, 0xae, 0x85, 0xe8, 0xae, 0x86, 0xe8, 0xae, 0x87, 0xe8, 0xae, 0x88,
  0xe8, 0xae, 0x89, 0xe8, 0xae, 0x8a, 0xe8, 0xae, 0x8b, 0xe8, 0xae, 0x8c,
  0xe8, 0xae, 0x8d, 0xe8, 0xae, 0x8e, 0xe8, 0xae, 0x8f, 0xe8, 0xae, 0x90,
  0xe8, 0xae, 0x91, 0xe8, 0xae, 0x92, 0xe8, 0xae, 0x93, 0xe8, 0xae, 0x94,
  0xe8, 0xae, 0x95, 0xe8, 0xae, 0x96, 0xe8, 0xae, 0x97, 0xe8, 0xae, 0x98,
  0xe8, 0xae, 0x99, 0xe8, 0xae, 0x9a, 0xe8, 0xae, 0x9b, 0xe8, 0xae, 0x9c,
  0xe8, 0xae, 0x9d, 0xe8, 0xae, 0x9e, 0xe8, 0xae, 0x9f, 0xe8, 0xae, 0xac,
  0xe8, 0xae, 0xb1, 0xe8, 0xae, 0xbb, 0xe8, 0xaf, 0x87, 0xe8, 0xaf, 0x90,
  0xe8, 0xaf, 0xaa, 0xe8, 0xb0, 0x89, 0xe8, 0xb0, 0x9e, 0xe4, 0xbd, 0x8f,
  0xe6, 0xb3, 0xa8, 0xe7, 0xa5, 0x9d, 0xe9, 0xa9, 0xbb, 0xe6, 0x8a, 0x93,
  0xe7, 0x88, 0xaa, 0xe6, 0x8b, 0xbd, 0xe4, 0xb8, 0x93, 0xe7, 0xa0, 0x96,
  0xe8, 0xbd, 0xac, 0xe6, 0x92, 0xb0, 0xe8, 0xb5, 0x9a, 0xe7, 0xaf, 0x86,
  0xe6, 0xa1, 0xa9, 0xe5, 0xba, 0x84, 0xe8, 0xa3, 0x85, 0xe5, 0xa6, 0x86,
  0xe6, 0x92, 0x9e, 0xe5, 0xa3, 0xae, 0xe7, 0x8a, 0xb6, 0xe6, 0xa4, 0x8e,
  0xe9, 0x94, 0xa5, 0xe8, 0xbf, 0xbd, 0xe8, 0xb5, 0x98, 0xe5, 0x9d, 0xa0,
  0xe7, 0xbc, 0x80, 0xe8, 0xb0, 0x86, 0xe5, 0x87, 0x86, 0xe6, 0x8d, 0x89,
  0xe6, 0x8b, 0x99, 0xe5, 0x8d, 0x93, 0xe6, 0xa1, 0x8c, 0xe7, 0x90, 0xa2,
  0xe8, 0x8c, 0x81, 0xe9, 0x85, 0x8c, 0xe5, 0x95, 0x84, 0xe7, 0x9d, 0x80,
  0xe7, 0x81, 0xbc, 0xe6, 0xb5, 0x8a, 0xe5, 0x85, 0xb9, 0xe5, 0x92, 0xa8,
  0xe8, 0xb5, 0x84, 0xe5, 0xa7, 0xbf, 0xe6, 0xbb, 0x8b, 0xe6, 0xb7, 0x84,
  0xe5, 0xad, 0x9c, 0xe7, 0xb4, 0xab, 0xe4, 0xbb, 0x94, 0xe7, 0xb1, 0xbd,
  0xe6, 0xbb, 0x93, 0xe5, 0xad, 0x90, 0xe8, 0x87, 0xaa, 0xe6, 0xb8, 0x8d,
  0xe5, 0xad, 0x97, 0xe9, 0xac, 0x83, 0xe6, 0xa3, 0x95, 0xe8, 0xb8, 0xaa,
  0xe5, 0xae, 0x97, 0xe7, 0xbb, 0xbc, 0xe6, 0x80, 0xbb, 0xe7, 0xba, 0xb5,
  0xe9, 0x82, 0xb9, 0xe8, 0xb5, 0xb0, 0xe5, 0xa5, 0x8f, 0xe6, 0x8f, 0x8d,
  0xe7, 0xa7, 0x9f, 0xe8, 0xb6, 0xb3, 0xe5, 0x8d, 0x92, 0xe6, 0x97, 0x8f,
  0xe7, 0xa5, 0x96, 0xe8, 0xaf, 0x85, 0xe9, 0x98, 0xbb, 0xe7, 0xbb, 0x84,
  0xe9, 0x92, 0xbb, 0xe7, 0xba, 0x82, 0xe5, 0x98, 0xb4, 0xe9, 0x86, 0x89,
  0xe6, 0x9c, 0x80, 0xe7, 0xbd, 0xaa, 0xe5, 0xb0, 0x8a, 0xe9, 0x81, 0xb5,
  0xe6, 0x98, 0xa8, 0xe5, 0xb7, 0xa6, 0xe4, 0xbd, 0x90, 0xe6, 0x9f, 0x9e,
  0xe5, 0x81, 0x9a, 0xe4, 0xbd, 0x9c, 0xe5, 0x9d, 0x90, 0xe5, 0xba, 0xa7
};

unsigned char gbk_row_0xd8_converted_to_utf8[] = {
  0xe8, 0xb0, 0xb8, 0xe8, 0xb0, 0xb9, 0xe8, 0xb0, 0xba, 0xe8, 0xb0, 0xbb,
  0xe8, 0xb0, 0xbc, 0xe8, 0xb0, 0xbd, 0xe8, 0xb0, 0xbe, 0xe8, 0xb0, 0xbf,
  0xe8, 0xb1, 0x80, 0xe8, 0xb1, 0x82, 0xe8, 0xb1, 0x83, 0xe8, 0xb1, 0x84,
  0xe8, 0xb1, 0x85, 0xe8, 0xb1, 0x88, 0xe8, 0xb1, 0x8a, 0xe8, 0xb1, 0x8b,
  0xe8, 0xb1, 0x8d, 0xe8, 0xb1, 0x8e, 0xe8, 0xb1, 0x8f, 0xe8, 0xb1, 0x90,
  0xe8, 0xb1, 0x91, 0xe8, 0xb1, 0x92, 0xe8, 0xb1, 0x93, 0xe8, 0xb1, 0x94,
  0xe8, 0xb1, 0x96, 0xe8, 0xb1, 0x97, 0xe8, 0xb1, 0x98, 0xe8, 0xb1, 0x99,
  0xe8, 0xb1, 0x9b, 0xe8, 0xb1, 0x9c, 0xe8, 0xb1, 0x9d, 0xe8, 0xb1, 0x9e,
  0xe8, 0xb1, 0x9f, 0xe8, 0xb1, 0xa0, 0xe8, 0xb1, 0xa3, 0xe8, 0xb1, 0xa4,
  0xe8, 0xb1, 0xa5, 0xe8, 0xb1, 0xa6, 0xe8, 0xb1, 0xa7, 0xe8, 0xb1, 0xa8,
  0xe8, 0xb1, 0xa9, 0xe8, 0xb1, 0xac, 0xe8, 0xb1, 0xad, 0xe8, 0xb1, 0xae,
  0xe8, 0xb1, 0xaf, 0xe8, 0xb1, 0xb0, 0xe8, 0xb1, 0xb1, 0xe8, 0xb1, 0xb2,
  0xe8, 0xb1, 0xb4, 0xe8, 0xb1, 0xb5, 0xe8, 0xb1, 0xb6, 0xe8, 0xb1, 0xb7,
  0xe8, 0xb1, 0xbb, 0xe8, 0xb1, 0xbc, 0xe8, 0xb1, 0xbd, 0xe8, 0xb1, 0xbe,
  0xe8, 0xb1, 0xbf, 0xe8, 0xb2, 0x80, 0xe8, 0xb2, 0x81, 0xe8, 0xb2, 0x83,
  0xe8, 0xb2, 0x84, 0xe8, 0xb2, 0x86, 0xe8, 0xb2, 0x87, 0xe8, 0xb2, 0x88,
  0xe8, 0xb2, 0x8b, 0xe8, 0xb2, 0x8d, 0xe8, 0xb2, 0x8e, 0xe8, 0xb2, 0x8f,
  0xe8, 0xb2, 0x90, 0xe8, 0xb2, 0x91, 0xe8, 0xb2, 0x92, 0xe8, 0xb2, 0x93,
  0xe8, 0xb2, 0x95, 0xe8, 0xb2, 0x96, 0xe8, 0xb2, 0x97, 0xe8, 0xb2, 0x99,
  0xe8, 0xb2, 0x9a, 0xe8, 0xb2, 0x9b, 0xe8, 0xb2, 0x9c, 0xe8, 0xb2, 0x9d,
  0xe8, 0xb2, 0x9e, 0xe8, 0xb2, 0x9f, 0xe8, 0xb2, 0xa0, 0xe8, 0xb2, 0xa1,
  0xe8, 0xb2, 0xa2, 0xe8, 0xb2, 0xa3, 0xe8, 0xb2, 0xa4, 0xe8, 0xb2, 0xa5,
  0xe8, 0xb2, 0xa6, 0xe8, 0xb2, 0xa7, 0xe8, 0xb2, 0xa8, 0xe8, 0xb2, 0xa9,
  0xe8, 0xb2, 0xaa, 0xe8, 0xb2, 0xab, 0xe8, 0xb2, 0xac, 0xe8, 0xb2, 0xad,
  0xe4, 0xba, 0x8d, 0xe4, 0xb8, 0x8c, 0xe5, 0x85, 0x80, 0xe4, 0xb8, 0x90,
  0xe5, 0xbb, 0xbf, 0xe5, 0x8d, 0x85, 0xe4, 0xb8, 0x95, 0xe4, 0xba, 0x98,
  0xe4, 0xb8, 0x9e, 0xe9, 0xac, 0xb2, 0xe5, 0xad, 0xac, 0xe5, 0x99, 0xa9,
  0xe4, 0xb8, 0xa8, 0xe7, 0xa6, 0xba, 0xe4, 0xb8, 0xbf, 0xe5, 0x8c, 0x95,
  0xe4, 0xb9, 0x87, 0xe5, 0xa4, 0xad, 0xe7, 0x88, 0xbb, 0xe5, 0x8d, 0xae,
  0xe6, 0xb0, 0x90, 0xe5, 0x9b, 0x9f, 0xe8, 0x83, 0xa4, 0xe9, 0xa6, 0x97,
  0xe6, 0xaf, 0x93, 0xe7, 0x9d, 0xbe, 0xe9, 0xbc, 0x97, 0xe4, 0xb8, 0xb6,
  0xe4, 0xba, 0x9f, 0xe9, 0xbc, 0x90, 0xe4, 0xb9, 0x9c, 0xe4, 0xb9, 0xa9,
  0xe4, 0xba, 0x93, 0xe8, 0x8a, 0x88, 0xe5, 0xad, 0x9b, 0xe5, 0x95, 0xac,
  0xe5, 0x98, 0x8f, 0xe4, 0xbb, 0x84, 0xe5, 0x8e, 0x8d, 0xe5, 0x8e, 0x9d,
  0xe5, 0x8e, 0xa3, 0xe5, 0x8e, 0xa5, 0xe5, 0x8e, 0xae, 0xe9, 0x9d, 0xa5,
  0xe8, 0xb5, 0x9d, 0xe5, 0x8c, 0x9a, 0xe5, 0x8f, 0xb5, 0xe5, 0x8c, 0xa6,
  0xe5, 0x8c, 0xae, 0xe5, 0x8c, 0xbe, 0xe8, 0xb5, 0x9c, 0xe5, 0x8d, 0xa6,
  0xe5, 0x8d, 0xa3, 0xe5, 0x88, 0x82, 0xe5, 0x88, 0x88, 0xe5, 0x88, 0x8e,
  0xe5, 0x88, 0xad, 0xe5, 0x88, 0xb3, 0xe5, 0x88, 0xbf, 0xe5, 0x89, 0x80,
  0xe5, 0x89, 0x8c, 0xe5, 0x89, 0x9e, 0xe5, 0x89, 0xa1, 0xe5, 0x89, 0x9c,
  0xe8, 0x92, 0xaf, 0xe5, 0x89, 0xbd, 0xe5, 0x8a, 0x82, 0xe5, 0x8a, 0x81,
  0xe5, 0x8a, 0x90, 0xe5, 0x8a, 0x93, 0xe5, 0x86, 0x82, 0xe7, 0xbd, 0x94,
  0xe4, 0xba, 0xbb, 0xe4, 0xbb, 0x83, 0xe4, 0xbb, 0x89, 0xe4, 0xbb, 0x82,
  0xe4, 0xbb, 0xa8, 0xe4, 0xbb, 0xa1, 0xe4, 0xbb, 0xab, 0xe4, 0xbb, 0x9e,
  0xe4, 0xbc, 0x9b, 0xe4, 0xbb, 0xb3, 0xe4, 0xbc, 0xa2, 0xe4, 0xbd, 0xa4,
  0xe4, 0xbb, 0xb5, 0xe4, 0xbc, 0xa5, 0xe4, 0xbc, 0xa7, 0xe4, 0xbc, 0x89,
  0xe4, 0xbc, 0xab, 0xe4, 0xbd, 0x9e, 0xe4, 0xbd, 0xa7, 0xe6, 0x94, 0xb8,
  0xe4, 0xbd, 0x9a, 0xe4, 0xbd, 0x9d
};

unsigned char gbk_row_0xd9_converted_to_utf8[] = {
  0xe8, 0xb2, 0xae, 0xe8, 0xb2,
  0xaf, 0xe8, 0xb2, 0xb0, 0xe8, 0xb2, 0xb1, 0xe8, 0xb2, 0xb2, 0xe8, 0xb2,
  0xb3, 0xe8, 0xb2, 0xb4, 0xe8, 0xb2, 0xb5, 0xe8, 0xb2, 0xb6, 0xe8, 0xb2,
  0xb7, 0xe8, 0xb2, 0xb8, 0xe8, 0xb2, 0xb9, 0xe8, 0xb2, 0xba, 0xe8, 0xb2,
  0xbb, 0xe8, 0xb2, 0xbc, 0xe8, 0xb2, 0xbd, 0xe8, 0xb2, 0xbe, 0xe8, 0xb2,
  0xbf, 0xe8, 0xb3, 0x80, 0xe8, 0xb3, 0x81, 0xe8, 0xb3, 0x82, 0xe8, 0xb3,
  0x83, 0xe8, 0xb3, 0x84, 0xe8, 0xb3, 0x85, 0xe8, 0xb3, 0x86, 0xe8, 0xb3,
  0x87, 0xe8, 0xb3, 0x88, 0xe8, 0xb3, 0x89, 0xe8, 0xb3, 0x8a, 0xe8, 0xb3,
  0x8b, 0xe8, 0xb3, 0x8c, 0xe8, 0xb3, 0x8d, 0xe8, 0xb3, 0x8e, 0xe8, 0xb3,
  0x8f, 0xe8, 0xb3, 0x90, 0xe8, 0xb3, 0x91, 0xe8, 0xb3, 0x92, 0xe8, 0xb3,
  0x93, 0xe8, 0xb3, 0x94, 0xe8, 0xb3, 0x95, 0xe8, 0xb3, 0x96, 0xe8, 0xb3,
  0x97, 0xe8, 0xb3, 0x98, 0xe8, 0xb3, 0x99, 0xe8, 0xb3, 0x9a, 0xe8, 0xb3,
  0x9b, 0xe8, 0xb3, 0x9c, 0xe8, 0xb3, 0x9d, 0xe8, 0xb3, 0x9e, 0xe8, 0xb3,
  0x9f, 0xe8, 0xb3, 0xa0, 0xe8, 0xb3, 0xa1, 0xe8, 0xb3, 0xa2, 0xe8, 0xb3,
  0xa3, 0xe8, 0xb3, 0xa4, 0xe8, 0xb3, 0xa5, 0xe8, 0xb3, 0xa6, 0xe8, 0xb3,
  0xa7, 0xe8, 0xb3, 0xa8, 0xe8, 0xb3, 0xa9, 0xe8, 0xb3, 0xaa, 0xe8, 0xb3,
  0xab, 0xe8, 0xb3, 0xac, 0xe8, 0xb3, 0xad, 0xe8, 0xb3, 0xae, 0xe8, 0xb3,
  0xaf, 0xe8, 0xb3, 0xb0, 0xe8, 0xb3, 0xb1, 0xe8, 0xb3, 0xb2, 0xe8, 0xb3,
  0xb3, 0xe8, 0xb3, 0xb4, 0xe8, 0xb3, 0xb5, 0xe8, 0xb3, 0xb6, 0xe8, 0xb3,
  0xb7, 0xe8, 0xb3, 0xb8, 0xe8, 0xb3, 0xb9, 0xe8, 0xb3, 0xba, 0xe8, 0xb3,
  0xbb, 0xe8, 0xb3, 0xbc, 0xe8, 0xb3, 0xbd, 0xe8, 0xb3, 0xbe, 0xe8, 0xb3,
  0xbf, 0xe8, 0xb4, 0x80, 0xe8, 0xb4, 0x81, 0xe8, 0xb4, 0x82, 0xe8, 0xb4,
  0x83, 0xe8, 0xb4, 0x84, 0xe8, 0xb4, 0x85, 0xe8, 0xb4, 0x86, 0xe8, 0xb4,
  0x87, 0xe8, 0xb4, 0x88, 0xe8, 0xb4, 0x89, 0xe8, 0xb4, 0x8a, 0xe8, 0xb4,
  0x8b, 0xe8, 0xb4, 0x8c, 0xe8, 0xb4, 0x8d, 0xe4, 0xbd, 0x9f, 0xe4, 0xbd,
  0x97, 0xe4, 0xbc, 0xb2, 0xe4, 0xbc, 0xbd, 0xe4, 0xbd, 0xb6, 0xe4, 0xbd,
  0xb4, 0xe4, 0xbe, 0x91, 0xe4, 0xbe, 0x89, 0xe4, 0xbe, 0x83, 0xe4, 0xbe,
  0x8f, 0xe4, 0xbd, 0xbe, 0xe4, 0xbd, 0xbb, 0xe4, 0xbe, 0xaa, 0xe4, 0xbd,
  0xbc, 0xe4, 0xbe, 0xac, 0xe4, 0xbe, 0x94, 0xe4, 0xbf, 0xa6, 0xe4, 0xbf,
  0xa8, 0xe4, 0xbf, 0xaa, 0xe4, 0xbf, 0x85, 0xe4, 0xbf, 0x9a, 0xe4, 0xbf,
  0xa3, 0xe4, 0xbf, 0x9c, 0xe4, 0xbf, 0x91, 0xe4, 0xbf, 0x9f, 0xe4, 0xbf,
  0xb8, 0xe5, 0x80, 0xa9, 0xe5, 0x81, 0x8c, 0xe4, 0xbf, 0xb3, 0xe5, 0x80,
  0xac, 0xe5, 0x80, 0x8f, 0xe5, 0x80, 0xae, 0xe5, 0x80, 0xad, 0xe4, 0xbf,
  0xbe, 0xe5, 0x80, 0x9c, 0xe5, 0x80, 0x8c, 0xe5, 0x80, 0xa5, 0xe5, 0x80,
  0xa8, 0xe5, 0x81, 0xbe, 0xe5, 0x81, 0x83, 0xe5, 0x81, 0x95, 0xe5, 0x81,
  0x88, 0xe5, 0x81, 0x8e, 0xe5, 0x81, 0xac, 0xe5, 0x81, 0xbb, 0xe5, 0x82,
  0xa5, 0xe5, 0x82, 0xa7, 0xe5, 0x82, 0xa9, 0xe5, 0x82, 0xba, 0xe5, 0x83,
  0x96, 0xe5, 0x84, 0x86, 0xe5, 0x83, 0xad, 0xe5, 0x83, 0xac, 0xe5, 0x83,
  0xa6, 0xe5, 0x83, 0xae, 0xe5, 0x84, 0x87, 0xe5, 0x84, 0x8b, 0xe4, 0xbb,
  0x9d, 0xe6, 0xb0, 0xbd, 0xe4, 0xbd, 0x98, 0xe4, 0xbd, 0xa5, 0xe4, 0xbf,
  0x8e, 0xe9, 0xbe, 0xa0, 0xe6, 0xb1, 0x86, 0xe7, 0xb1, 0xb4, 0xe5, 0x85,
  0xae, 0xe5, 0xb7, 0xbd, 0xe9, 0xbb, 0x89, 0xe9, 0xa6, 0x98, 0xe5, 0x86,
  0x81, 0xe5, 0xa4, 0x94, 0xe5, 0x8b, 0xb9, 0xe5, 0x8c, 0x8d, 0xe8, 0xa8,
  0x87, 0xe5, 0x8c, 0x90, 0xe5, 0x87, 0xab, 0xe5, 0xa4, 0x99, 0xe5, 0x85,
  0x95, 0xe4, 0xba, 0xa0, 0xe5, 0x85, 0x96, 0xe4, 0xba, 0xb3, 0xe8, 0xa1,
  0xae, 0xe8, 0xa2, 0xa4, 0xe4, 0xba, 0xb5, 0xe8, 0x84, 0x94, 0xe8, 0xa3,
  0x92, 0xe7, 0xa6, 0x80, 0xe5, 0xac, 0xb4, 0xe8, 0xa0, 0x83, 0xe7, 0xbe,
  0xb8, 0xe5, 0x86, 0xab, 0xe5, 0x86, 0xb1, 0xe5, 0x86, 0xbd, 0xe5, 0x86,
  0xbc
};

unsigned char gbk_row_0xda_converted_to_utf8[] = {
  0xe8, 0xb4, 0x8e, 0xe8, 0xb4, 0x8f, 0xe8, 0xb4, 0x90, 0xe8,
  0xb4, 0x91, 0xe8, 0xb4, 0x92, 0xe8, 0xb4, 0x93, 0xe8, 0xb4, 0x94, 0xe8,
  0xb4, 0x95, 0xe8, 0xb4, 0x96, 0xe8, 0xb4, 0x97, 0xe8, 0xb4, 0x98, 0xe8,
  0xb4, 0x99, 0xe8, 0xb4, 0x9a, 0xe8, 0xb4, 0x9b, 0xe8, 0xb4, 0x9c, 0xe8,
  0xb4, 0xa0, 0xe8, 0xb5, 0x91, 0xe8, 0xb5, 0x92, 0xe8, 0xb5, 0x97, 0xe8,
  0xb5, 0x9f, 0xe8, 0xb5, 0xa5, 0xe8, 0xb5, 0xa8, 0xe8, 0xb5, 0xa9, 0xe8,
  0xb5, 0xaa, 0xe8, 0xb5, 0xac, 0xe8, 0xb5, 0xae, 0xe8, 0xb5, 0xaf, 0xe8,
  0xb5, 0xb1, 0xe8, 0xb5, 0xb2, 0xe8, 0xb5, 0xb8, 0xe8, 0xb5, 0xb9, 0xe8,
  0xb5, 0xba, 0xe8, 0xb5, 0xbb, 0xe8, 0xb5, 0xbc, 0xe8, 0xb5, 0xbd, 0xe8,
  0xb5, 0xbe, 0xe8, 0xb5, 0xbf, 0xe8, 0xb6, 0x80, 0xe8, 0xb6, 0x82, 0xe8,
  0xb6, 0x83, 0xe8, 0xb6, 0x86, 0xe8, 0xb6, 0x87, 0xe8, 0xb6, 0x88, 0xe8,
  0xb6, 0x89, 0xe8, 0xb6, 0x8c, 0xe8, 0xb6, 0x8d, 0xe8, 0xb6, 0x8e, 0xe8,
  0xb6, 0x8f, 0xe8, 0xb6, 0x90, 0xe8, 0xb6, 0x92, 0xe8, 0xb6, 0x93, 0xe8,
  0xb6, 0x95, 0xe8, 0xb6, 0x96, 0xe8, 0xb6, 0x97, 0xe8, 0xb6, 0x98, 0xe8,
  0xb6, 0x99, 0xe8, 0xb6, 0x9a, 0xe8, 0xb6, 0x9b, 0xe8, 0xb6, 0x9c, 0xe8,
  0xb6, 0x9d, 0xe8, 0xb6, 0x9e, 0xe8, 0xb6, 0xa0, 0xe8, 0xb6, 0xa1, 0xe8,
  0xb6, 0xa2, 0xe8, 0xb6, 0xa4, 0xe8, 0xb6, 0xa5, 0xe8, 0xb6, 0xa6, 0xe8,
  0xb6, 0xa7, 0xe8, 0xb6, 0xa8, 0xe8, 0xb6, 0xa9, 0xe8, 0xb6, 0xaa, 0xe8,
  0xb6, 0xab, 0xe8, 0xb6, 0xac, 0xe8, 0xb6, 0xad, 0xe8, 0xb6, 0xae, 0xe8,
  0xb6, 0xaf, 0xe8, 0xb6, 0xb0, 0xe8, 0xb6, 0xb2, 0xe8, 0xb6, 0xb6, 0xe8,
  0xb6, 0xb7, 0xe8, 0xb6, 0xb9, 0xe8, 0xb6, 0xbb, 0xe8, 0xb6, 0xbd, 0xe8,
  0xb7, 0x80, 0xe8, 0xb7, 0x81, 0xe8, 0xb7, 0x82, 0xe8, 0xb7, 0x85, 0xe8,
  0xb7, 0x87, 0xe8, 0xb7, 0x88, 0xe8, 0xb7, 0x89, 0xe8, 0xb7, 0x8a, 0xe8,
  0xb7, 0x8d, 0xe8, 0xb7, 0x90, 0xe8, 0xb7, 0x92, 0xe8, 0xb7, 0x93, 0xe8,
  0xb7, 0x94, 0xe5, 0x87, 0x87, 0xe5, 0x86, 0x96, 0xe5, 0x86, 0xa2, 0xe5,
  0x86, 0xa5, 0xe8, 0xae, 0xa0, 0xe8, 0xae, 0xa6, 0xe8, 0xae, 0xa7, 0xe8,
  0xae, 0xaa, 0xe8, 0xae, 0xb4, 0xe8, 0xae, 0xb5, 0xe8, 0xae, 0xb7, 0xe8,
  0xaf, 0x82, 0xe8, 0xaf, 0x83, 0xe8, 0xaf, 0x8b, 0xe8, 0xaf, 0x8f, 0xe8,
  0xaf, 0x8e, 0xe8, 0xaf, 0x92, 0xe8, 0xaf, 0x93, 0xe8, 0xaf, 0x94, 0xe8,
  0xaf, 0x96, 0xe8, 0xaf, 0x98, 0xe8, 0xaf, 0x99, 0xe8, 0xaf, 0x9c, 0xe8,
  0xaf, 0x9f, 0xe8, 0xaf, 0xa0, 0xe8, 0xaf, 0xa4, 0xe8, 0xaf, 0xa8, 0xe8,
  0xaf, 0xa9, 0xe8, 0xaf, 0xae, 0xe8, 0xaf, 0xb0, 0xe8, 0xaf, 0xb3, 0xe8,
  0xaf, 0xb6, 0xe8, 0xaf, 0xb9, 0xe8, 0xaf, 0xbc, 0xe8, 0xaf, 0xbf, 0xe8,
  0xb0, 0x80, 0xe8, 0xb0, 0x82, 0xe8, 0xb0, 0x84, 0xe8, 0xb0, 0x87, 0xe8,
  0xb0, 0x8c, 0xe8, 0xb0, 0x8f, 0xe8, 0xb0, 0x91, 0xe8, 0xb0, 0x92, 0xe8,
  0xb0, 0x94, 0xe8, 0xb0, 0x95, 0xe8, 0xb0, 0x96, 0xe8, 0xb0, 0x99, 0xe8,
  0xb0, 0x9b, 0xe8, 0xb0, 0x98, 0xe8, 0xb0, 0x9d, 0xe8, 0xb0, 0x9f, 0xe8,
  0xb0, 0xa0, 0xe8, 0xb0, 0xa1, 0xe8, 0xb0, 0xa5, 0xe8, 0xb0, 0xa7, 0xe8,
  0xb0, 0xaa, 0xe8, 0xb0, 0xab, 0xe8, 0xb0, 0xae, 0xe8, 0xb0, 0xaf, 0xe8,
  0xb0, 0xb2, 0xe8, 0xb0, 0xb3, 0xe8, 0xb0, 0xb5, 0xe8, 0xb0, 0xb6, 0xe5,
  0x8d, 0xa9, 0xe5, 0x8d, 0xba, 0xe9, 0x98, 0x9d, 0xe9, 0x98, 0xa2, 0xe9,
  0x98, 0xa1, 0xe9, 0x98, 0xb1, 0xe9, 0x98, 0xaa, 0xe9, 0x98, 0xbd, 0xe9,
  0x98, 0xbc, 0xe9, 0x99, 0x82, 0xe9, 0x99, 0x89, 0xe9, 0x99, 0x94, 0xe9,
  0x99, 0x9f, 0xe9, 0x99, 0xa7, 0xe9, 0x99, 0xac, 0xe9, 0x99, 0xb2, 0xe9,
  0x99, 0xb4, 0xe9, 0x9a, 0x88, 0xe9, 0x9a, 0x8d, 0xe9, 0x9a, 0x97, 0xe9,
  0x9a, 0xb0, 0xe9, 0x82, 0x97, 0xe9, 0x82, 0x9b, 0xe9, 0x82, 0x9d, 0xe9,
  0x82, 0x99, 0xe9, 0x82, 0xac, 0xe9, 0x82, 0xa1, 0xe9, 0x82, 0xb4, 0xe9,
  0x82, 0xb3, 0xe9, 0x82, 0xb6, 0xe9, 0x82, 0xba
};

unsigned char gbk_row_0xdb_converted_to_utf8[] = {
  0xe8, 0xb7, 0x95,
  0xe8, 0xb7, 0x98, 0xe8, 0xb7, 0x99, 0xe8, 0xb7, 0x9c, 0xe8, 0xb7, 0xa0,
  0xe8, 0xb7, 0xa1, 0xe8, 0xb7, 0xa2, 0xe8, 0xb7, 0xa5, 0xe8, 0xb7, 0xa6,
  0xe8, 0xb7, 0xa7, 0xe8, 0xb7, 0xa9, 0xe8, 0xb7, 0xad, 0xe8, 0xb7, 0xae,
  0xe8, 0xb7, 0xb0, 0xe8, 0xb7, 0xb1, 0xe8, 0xb7, 0xb2, 0xe8, 0xb7, 0xb4,
  0xe8, 0xb7, 0xb6, 0xe8, 0xb7, 0xbc, 0xe8, 0xb7, 0xbe, 0xe8, 0xb7, 0xbf,
  0xe8, 0xb8, 0x80, 0xe8, 0xb8, 0x81, 0xe8, 0xb8, 0x82, 0xe8, 0xb8, 0x83,
  0xe8, 0xb8, 0x84, 0xe8, 0xb8, 0x86, 0xe8, 0xb8, 0x87, 0xe8, 0xb8, 0x88,
  0xe8, 0xb8, 0x8b, 0xe8, 0xb8, 0x8d, 0xe8, 0xb8, 0x8e, 0xe8, 0xb8, 0x90,
  0xe8, 0xb8, 0x91, 0xe8, 0xb8, 0x92, 0xe8, 0xb8, 0x93, 0xe8, 0xb8, 0x95,
  0xe8, 0xb8, 0x96, 0xe8, 0xb8, 0x97, 0xe8, 0xb8, 0x98, 0xe8, 0xb8, 0x99,
  0xe8, 0xb8, 0x9a, 0xe8, 0xb8, 0x9b, 0xe8, 0xb8, 0x9c, 0xe8, 0xb8, 0xa0,
  0xe8, 0xb8, 0xa1, 0xe8, 0xb8, 0xa4, 0xe8, 0xb8, 0xa5, 0xe8, 0xb8, 0xa6,
  0xe8, 0xb8, 0xa7, 0xe8, 0xb8, 0xa8, 0xe8, 0xb8, 0xab, 0xe8, 0xb8, 0xad,
  0xe8, 0xb8, 0xb0, 0xe8, 0xb8, 0xb2, 0xe8, 0xb8, 0xb3, 0xe8, 0xb8, 0xb4,
  0xe8, 0xb8, 0xb6, 0xe8, 0xb8, 0xb7, 0xe8, 0xb8, 0xb8, 0xe8, 0xb8, 0xbb,
  0xe8, 0xb8, 0xbc, 0xe8, 0xb8, 0xbe, 0xe8, 0xb8, 0xbf, 0xe8, 0xb9, 0x83,
  0xe8, 0xb9, 0x85, 0xe8, 0xb9, 0x86, 0xe8, 0xb9, 0x8c, 0xe8, 0xb9, 0x8d,
  0xe8, 0xb9, 0x8e, 0xe8, 0xb9, 0x8f, 0xe8, 0xb9, 0x90, 0xe8, 0xb9, 0x93,
  0xe8, 0xb9, 0x94, 0xe8, 0xb9, 0x95, 0xe8, 0xb9, 0x96, 0xe8, 0xb9, 0x97,
  0xe8, 0xb9, 0x98, 0xe8, 0xb9, 0x9a, 0xe8, 0xb9, 0x9b, 0xe8, 0xb9, 0x9c,
  0xe8, 0xb9, 0x9d, 0xe8, 0xb9, 0x9e, 0xe8, 0xb9, 0x9f, 0xe8, 0xb9, 0xa0,
  0xe8, 0xb9, 0xa1, 0xe8, 0xb9, 0xa2, 0xe8, 0xb9, 0xa3, 0xe8, 0xb9, 0xa4,
  0xe8, 0xb9, 0xa5, 0xe8, 0xb9, 0xa7, 0xe8, 0xb9, 0xa8, 0xe8, 0xb9, 0xaa,
  0xe8, 0xb9, 0xab, 0xe8, 0xb9, 0xae, 0xe8, 0xb9, 0xb1, 0xe9, 0x82, 0xb8,
  0xe9, 0x82, 0xb0, 0xe9, 0x83, 0x8f, 0xe9, 0x83, 0x85, 0xe9, 0x82, 0xbe,
  0xe9, 0x83, 0x90, 0xe9, 0x83, 0x84, 0xe9, 0x83, 0x87, 0xe9, 0x83, 0x93,
  0xe9, 0x83, 0xa6, 0xe9, 0x83, 0xa2, 0xe9, 0x83, 0x9c, 0xe9, 0x83, 0x97,
  0xe9, 0x83, 0x9b, 0xe9, 0x83, 0xab, 0xe9, 0x83, 0xaf, 0xe9, 0x83, 0xbe,
  0xe9, 0x84, 0x84, 0xe9, 0x84, 0xa2, 0xe9, 0x84, 0x9e, 0xe9, 0x84, 0xa3,
  0xe9, 0x84, 0xb1, 0xe9, 0x84, 0xaf, 0xe9, 0x84, 0xb9, 0xe9, 0x85, 0x83,
  0xe9, 0x85, 0x86, 0xe5, 0x88, 0x8d, 0xe5, 0xa5, 0x82, 0xe5, 0x8a, 0xa2,
  0xe5, 0x8a, 0xac, 0xe5, 0x8a, 0xad, 0xe5, 0x8a, 0xbe, 0xe5, 0x93, 0xbf,
  0xe5, 0x8b, 0x90, 0xe5, 0x8b, 0x96, 0xe5, 0x8b, 0xb0, 0xe5, 0x8f, 0x9f,
  0xe7, 0x87, 0xae, 0xe7, 0x9f, 0x8d, 0xe5, 0xbb, 0xb4, 0xe5, 0x87, 0xb5,
  0xe5, 0x87, 0xbc, 0xe9, 0xac, 0xaf, 0xe5, 0x8e, 0xb6, 0xe5, 0xbc, 0x81,
  0xe7, 0x95, 0x9a, 0xe5, 0xb7, 0xaf, 0xe5, 0x9d, 0x8c, 0xe5, 0x9e, 0xa9,
  0xe5, 0x9e, 0xa1, 0xe5, 0xa1, 0xbe, 0xe5, 0xa2, 0xbc, 0xe5, 0xa3, 0x85,
  0xe5, 0xa3, 0x91, 0xe5, 0x9c, 0xa9, 0xe5, 0x9c, 0xac, 0xe5, 0x9c, 0xaa,
  0xe5, 0x9c, 0xb3, 0xe5, 0x9c, 0xb9, 0xe5, 0x9c, 0xae, 0xe5, 0x9c, 0xaf,
  0xe5, 0x9d, 0x9c, 0xe5, 0x9c, 0xbb, 0xe5, 0x9d, 0x82, 0xe5, 0x9d, 0xa9,
  0xe5, 0x9e, 0x85, 0xe5, 0x9d, 0xab, 0xe5, 0x9e, 0x86, 0xe5, 0x9d, 0xbc,
  0xe5, 0x9d, 0xbb, 0xe5, 0x9d, 0xa8, 0xe5, 0x9d, 0xad, 0xe5, 0x9d, 0xb6,
  0xe5, 0x9d, 0xb3, 0xe5, 0x9e, 0xad, 0xe5, 0x9e, 0xa4, 0xe5, 0x9e, 0x8c,
  0xe5, 0x9e, 0xb2, 0xe5, 0x9f, 0x8f, 0xe5, 0x9e, 0xa7, 0xe5, 0x9e, 0xb4,
  0xe5, 0x9e, 0x93, 0xe5, 0x9e, 0xa0, 0xe5, 0x9f, 0x95, 0xe5, 0x9f, 0x98,
  0xe5, 0x9f, 0x9a, 0xe5, 0x9f, 0x99, 0xe5, 0x9f, 0x92, 0xe5, 0x9e, 0xb8,
  0xe5, 0x9f, 0xb4, 0xe5, 0x9f, 0xaf, 0xe5, 0x9f, 0xb8, 0xe5, 0x9f, 0xa4,
  0xe5, 0x9f, 0x9d
};

unsigned char gbk_row_0xdc_converted_to_utf8[] = {
  0xe8, 0xb9, 0xb3, 0xe8, 0xb9, 0xb5, 0xe8, 0xb9,
  0xb7, 0xe8, 0xb9, 0xb8, 0xe8, 0xb9, 0xb9, 0xe8, 0xb9, 0xba, 0xe8, 0xb9,
  0xbb, 0xe8, 0xb9, 0xbd, 0xe8, 0xb9, 0xbe, 0xe8, 0xba, 0x80, 0xe8, 0xba,
  0x82, 0xe8, 0xba, 0x83, 0xe8, 0xba, 0x84, 0xe8, 0xba, 0x86, 0xe8, 0xba,
  0x88, 0xe8, 0xba, 0x89, 0xe8, 0xba, 0x8a, 0xe8, 0xba, 0x8b, 0xe8, 0xba,
  0x8c, 0xe8, 0xba, 0x8d, 0xe8, 0xba, 0x8e, 0xe8, 0xba, 0x91, 0xe8, 0xba,
  0x92, 0xe8, 0xba, 0x93, 0xe8, 0xba, 0x95, 0xe8, 0xba, 0x96, 0xe8, 0xba,
  0x97, 0xe8, 0xba, 0x98, 0xe8, 0xba, 0x99, 0xe8, 0xba, 0x9a, 0xe8, 0xba,
  0x9b, 0xe8, 0xba, 0x9d, 0xe8, 0xba, 0x9f, 0xe8, 0xba, 0xa0, 0xe8, 0xba,
  0xa1, 0xe8, 0xba, 0xa2, 0xe8, 0xba, 0xa3, 0xe8, 0xba, 0xa4, 0xe8, 0xba,
  0xa5, 0xe8, 0xba, 0xa6, 0xe8, 0xba, 0xa7, 0xe8, 0xba, 0xa8, 0xe8, 0xba,
  0xa9, 0xe8, 0xba, 0xaa, 0xe8, 0xba, 0xad, 0xe8, 0xba, 0xae, 0xe8, 0xba,
  0xb0, 0xe8, 0xba, 0xb1, 0xe8, 0xba, 0xb3, 0xe8, 0xba, 0xb4, 0xe8, 0xba,
  0xb5, 0xe8, 0xba, 0xb6, 0xe8, 0xba, 0xb7, 0xe8, 0xba, 0xb8, 0xe8, 0xba,
  0xb9, 0xe8, 0xba, 0xbb, 0xe8, 0xba, 0xbc, 0xe8, 0xba, 0xbd, 0xe8, 0xba,
  0xbe, 0xe8, 0xba, 0xbf, 0xe8, 0xbb, 0x80, 0xe8, 0xbb, 0x81, 0xe8, 0xbb,
  0x82, 0xe8, 0xbb, 0x83, 0xe8, 0xbb, 0x84, 0xe8, 0xbb, 0x85, 0xe8, 0xbb,
  0x86, 0xe8, 0xbb, 0x87, 0xe8, 0xbb, 0x88, 0xe8, 0xbb, 0x89, 0xe8, 0xbb,
  0x8a, 0xe8, 0xbb, 0x8b, 0xe8, 0xbb, 0x8c, 0xe8, 0xbb, 0x8d, 0xe8, 0xbb,
  0x8f, 0xe8, 0xbb, 0x90, 0xe8, 0xbb, 0x91, 0xe8, 0xbb, 0x92, 0xe8, 0xbb,
  0x93, 0xe8, 0xbb, 0x94, 0xe8, 0xbb, 0x95, 0xe8, 0xbb, 0x96, 0xe8, 0xbb,
  0x97, 0xe8, 0xbb, 0x98, 0xe8, 0xbb, 0x99, 0xe8, 0xbb, 0x9a, 0xe8, 0xbb,
  0x9b, 0xe8, 0xbb, 0x9c, 0xe8, 0xbb, 0x9d, 0xe8, 0xbb, 0x9e, 0xe8, 0xbb,
  0x9f, 0xe8, 0xbb, 0xa0, 0xe8, 0xbb, 0xa1, 0xe8, 0xbb, 0xa2, 0xe8, 0xbb,
  0xa3, 0xe8, 0xbb, 0xa4, 0xe5, 0xa0, 0x8b, 0xe5, 0xa0, 0x8d, 0xe5, 0x9f,
  0xbd, 0xe5, 0x9f, 0xad, 0xe5, 0xa0, 0x80, 0xe5, 0xa0, 0x9e, 0xe5, 0xa0,
  0x99, 0xe5, 0xa1, 0x84, 0xe5, 0xa0, 0xa0, 0xe5, 0xa1, 0xa5, 0xe5, 0xa1,
  0xac, 0xe5, 0xa2, 0x81, 0xe5, 0xa2, 0x89, 0xe5, 0xa2, 0x9a, 0xe5, 0xa2,
  0x80, 0xe9, 0xa6, 0xa8, 0xe9, 0xbc, 0x99, 0xe6, 0x87, 0xbf, 0xe8, 0x89,
  0xb9, 0xe8, 0x89, 0xbd, 0xe8, 0x89, 0xbf, 0xe8, 0x8a, 0x8f, 0xe8, 0x8a,
  0x8a, 0xe8, 0x8a, 0xa8, 0xe8, 0x8a, 0x84, 0xe8, 0x8a, 0x8e, 0xe8, 0x8a,
  0x91, 0xe8, 0x8a, 0x97, 0xe8, 0x8a, 0x99, 0xe8, 0x8a, 0xab, 0xe8, 0x8a,
  0xb8, 0xe8, 0x8a, 0xbe, 0xe8, 0x8a, 0xb0, 0xe8, 0x8b, 0x88, 0xe8, 0x8b,
  0x8a, 0xe8, 0x8b, 0xa3, 0xe8, 0x8a, 0x98, 0xe8, 0x8a, 0xb7, 0xe8, 0x8a,
  0xae, 0xe8, 0x8b, 0x8b, 0xe8, 0x8b, 0x8c, 0xe8, 0x8b, 0x81, 0xe8, 0x8a,
  0xa9, 0xe8, 0x8a, 0xb4, 0xe8, 0x8a, 0xa1, 0xe8, 0x8a, 0xaa, 0xe8, 0x8a,
  0x9f, 0xe8, 0x8b, 0x84, 0xe8, 0x8b, 0x8e, 0xe8, 0x8a, 0xa4, 0xe8, 0x8b,
  0xa1, 0xe8, 0x8c, 0x89, 0xe8, 0x8b, 0xb7, 0xe8, 0x8b, 0xa4, 0xe8, 0x8c,
  0x8f, 0xe8, 0x8c, 0x87, 0xe8, 0x8b, 0x9c, 0xe8, 0x8b, 0xb4, 0xe8, 0x8b,
  0x92, 0xe8, 0x8b, 0x98, 0xe8, 0x8c, 0x8c, 0xe8, 0x8b, 0xbb, 0xe8, 0x8b,
  0x93, 0xe8, 0x8c, 0x91, 0xe8, 0x8c, 0x9a, 0xe8, 0x8c, 0x86, 0xe8, 0x8c,
  0x94, 0xe8, 0x8c, 0x95, 0xe8, 0x8b, 0xa0, 0xe8, 0x8b, 0x95, 0xe8, 0x8c,
  0x9c, 0xe8, 0x8d, 0x91, 0xe8, 0x8d, 0x9b, 0xe8, 0x8d, 0x9c, 0xe8, 0x8c,
  0x88, 0xe8, 0x8e, 0x92, 0xe8, 0x8c, 0xbc, 0xe8, 0x8c, 0xb4, 0xe8, 0x8c,
  0xb1, 0xe8, 0x8e, 0x9b, 0xe8, 0x8d, 0x9e, 0xe8, 0x8c, 0xaf, 0xe8, 0x8d,
  0x8f, 0xe8, 0x8d, 0x87, 0xe8, 0x8d, 0x83, 0xe8, 0x8d, 0x9f, 0xe8, 0x8d,
  0x80, 0xe8, 0x8c, 0x97, 0xe8, 0x8d, 0xa0, 0xe8, 0x8c, 0xad, 0xe8, 0x8c,
  0xba, 0xe8, 0x8c, 0xb3, 0xe8, 0x8d, 0xa6, 0xe8, 0x8d, 0xa5
};

unsigned char gbk_row_0xdd_converted_to_utf8[] = {
  0xe8,
  0xbb, 0xa5, 0xe8, 0xbb, 0xa6, 0xe8, 0xbb, 0xa7, 0xe8, 0xbb, 0xa8, 0xe8,
  0xbb, 0xa9, 0xe8, 0xbb, 0xaa, 0xe8, 0xbb, 0xab, 0xe8, 0xbb, 0xac, 0xe8,
  0xbb, 0xad, 0xe8, 0xbb, 0xae, 0xe8, 0xbb, 0xaf, 0xe8, 0xbb, 0xb0, 0xe8,
  0xbb, 0xb1, 0xe8, 0xbb, 0xb2, 0xe8, 0xbb, 0xb3, 0xe8, 0xbb, 0xb4, 0xe8,
  0xbb, 0xb5, 0xe8, 0xbb, 0xb6, 0xe8, 0xbb, 0xb7, 0xe8, 0xbb, 0xb8, 0xe8,
  0xbb, 0xb9, 0xe8, 0xbb, 0xba, 0xe8, 0xbb, 0xbb, 0xe8, 0xbb, 0xbc, 0xe8,
  0xbb, 0xbd, 0xe8, 0xbb, 0xbe, 0xe8, 0xbb, 0xbf, 0xe8, 0xbc, 0x80, 0xe8,
  0xbc, 0x81, 0xe8, 0xbc, 0x82, 0xe8, 0xbc, 0x83, 0xe8, 0xbc, 0x84, 0xe8,
  0xbc, 0x85, 0xe8, 0xbc, 0x86, 0xe8, 0xbc, 0x87, 0xe8, 0xbc, 0x88, 0xe8,
  0xbc, 0x89, 0xe8, 0xbc, 0x8a, 0xe8, 0xbc, 0x8b, 0xe8, 0xbc, 0x8c, 0xe8,
  0xbc, 0x8d, 0xe8, 0xbc, 0x8e, 0xe8, 0xbc, 0x8f, 0xe8, 0xbc, 0x90, 0xe8,
  0xbc, 0x91, 0xe8, 0xbc, 0x92, 0xe8, 0xbc, 0x93, 0xe8, 0xbc, 0x94, 0xe8,
  0xbc, 0x95, 0xe8, 0xbc, 0x96, 0xe8, 0xbc, 0x97, 0xe8, 0xbc, 0x98, 0xe8,
  0xbc, 0x99, 0xe8, 0xbc, 0x9a, 0xe8, 0xbc, 0x9b, 0xe8, 0xbc, 0x9c, 0xe8,
  0xbc, 0x9d, 0xe8, 0xbc, 0x9e, 0xe8, 0xbc, 0x9f, 0xe8, 0xbc, 0xa0, 0xe8,
  0xbc, 0xa1, 0xe8, 0xbc, 0xa2, 0xe8, 0xbc, 0xa3, 0xe8, 0xbc, 0xa4, 0xe8,
  0xbc, 0xa5, 0xe8, 0xbc, 0xa6, 0xe8, 0xbc, 0xa7, 0xe8, 0xbc, 0xa8, 0xe8,
  0xbc, 0xa9, 0xe8, 0xbc, 0xaa, 0xe8, 0xbc, 0xab, 0xe8, 0xbc, 0xac, 0xe8,
  0xbc, 0xad, 0xe8, 0xbc, 0xae, 0xe8, 0xbc, 0xaf, 0xe8, 0xbc, 0xb0, 0xe8,
  0xbc, 0xb1, 0xe8, 0xbc, 0xb2, 0xe8, 0xbc, 0xb3, 0xe8, 0xbc, 0xb4, 0xe8,
  0xbc, 0xb5, 0xe8, 0xbc, 0xb6, 0xe8, 0xbc, 0xb7, 0xe8, 0xbc, 0xb8, 0xe8,
  0xbc, 0xb9, 0xe8, 0xbc, 0xba, 0xe8, 0xbc, 0xbb, 0xe8, 0xbc, 0xbc, 0xe8,
  0xbc, 0xbd, 0xe8, 0xbc, 0xbe, 0xe8, 0xbc, 0xbf, 0xe8, 0xbd, 0x80, 0xe8,
  0xbd, 0x81, 0xe8, 0xbd, 0x82, 0xe8, 0xbd, 0x83, 0xe8, 0xbd, 0x84, 0xe8,
  0x8d, 0xa8, 0xe8, 0x8c, 0x9b, 0xe8, 0x8d, 0xa9, 0xe8, 0x8d, 0xac, 0xe8,
  0x8d, 0xaa, 0xe8, 0x8d, 0xad, 0xe8, 0x8d, 0xae, 0xe8, 0x8e, 0xb0, 0xe8,
  0x8d, 0xb8, 0xe8, 0x8e, 0xb3, 0xe8, 0x8e, 0xb4, 0xe8, 0x8e, 0xa0, 0xe8,
  0x8e, 0xaa, 0xe8, 0x8e, 0x93, 0xe8, 0x8e, 0x9c, 0xe8, 0x8e, 0x85, 0xe8,
  0x8d, 0xbc, 0xe8, 0x8e, 0xb6, 0xe8, 0x8e, 0xa9, 0xe8, 0x8d, 0xbd, 0xe8,
  0x8e, 0xb8, 0xe8, 0x8d, 0xbb, 0xe8, 0x8e, 0x98, 0xe8, 0x8e, 0x9e, 0xe8,
  0x8e, 0xa8, 0xe8, 0x8e, 0xba, 0xe8, 0x8e, 0xbc, 0xe8, 0x8f, 0x81, 0xe8,
  0x90, 0x81, 0xe8, 0x8f, 0xa5, 0xe8, 0x8f, 0x98, 0xe5, 0xa0, 0x87, 0xe8,
  0x90, 0x98, 0xe8, 0x90, 0x8b, 0xe8, 0x8f, 0x9d, 0xe8, 0x8f, 0xbd, 0xe8,
  0x8f, 0x96, 0xe8, 0x90, 0x9c, 0xe8, 0x90, 0xb8, 0xe8, 0x90, 0x91, 0xe8,
  0x90, 0x86, 0xe8, 0x8f, 0x94, 0xe8, 0x8f, 0x9f, 0xe8, 0x90, 0x8f, 0xe8,
  0x90, 0x83, 0xe8, 0x8f, 0xb8, 0xe8, 0x8f, 0xb9, 0xe8, 0x8f, 0xaa, 0xe8,
  0x8f, 0x85, 0xe8, 0x8f, 0x80, 0xe8, 0x90, 0xa6, 0xe8, 0x8f, 0xb0, 0xe8,
  0x8f, 0xa1, 0xe8, 0x91, 0x9c, 0xe8, 0x91, 0x91, 0xe8, 0x91, 0x9a, 0xe8,
  0x91, 0x99, 0xe8, 0x91, 0xb3, 0xe8, 0x92, 0x87, 0xe8, 0x92, 0x88, 0xe8,
  0x91, 0xba, 0xe8, 0x92, 0x89, 0xe8, 0x91, 0xb8, 0xe8, 0x90, 0xbc, 0xe8,
  0x91, 0x86, 0xe8, 0x91, 0xa9, 0xe8, 0x91, 0xb6, 0xe8, 0x92, 0x8c, 0xe8,
  0x92, 0x8e, 0xe8, 0x90, 0xb1, 0xe8, 0x91, 0xad, 0xe8, 0x93, 0x81, 0xe8,
  0x93, 0x8d, 0xe8, 0x93, 0x90, 0xe8, 0x93, 0xa6, 0xe8, 0x92, 0xbd, 0xe8,
  0x93, 0x93, 0xe8, 0x93, 0x8a, 0xe8, 0x92, 0xbf, 0xe8, 0x92, 0xba, 0xe8,
  0x93, 0xa0, 0xe8, 0x92, 0xa1, 0xe8, 0x92, 0xb9, 0xe8, 0x92, 0xb4, 0xe8,
  0x92, 0x97, 0xe8, 0x93, 0xa5, 0xe8, 0x93, 0xa3, 0xe8, 0x94, 0x8c, 0xe7,
  0x94, 0x8d, 0xe8, 0x94, 0xb8, 0xe8, 0x93, 0xb0, 0xe8, 0x94, 0xb9, 0xe8,
  0x94, 0x9f, 0xe8, 0x94, 0xba
};

unsigned char gbk_row_0xde_converted_to_utf8[] = {
  0xe8, 0xbd, 0x85, 0xe8, 0xbd, 0x86,
  0xe8, 0xbd, 0x87, 0xe8, 0xbd, 0x88, 0xe8, 0xbd, 0x89, 0xe8, 0xbd, 0x8a,
  0xe8, 0xbd, 0x8b, 0xe8, 0xbd, 0x8c, 0xe8, 0xbd, 0x8d, 0xe8, 0xbd, 0x8e,
  0xe8, 0xbd, 0x8f, 0xe8, 0xbd, 0x90, 0xe8, 0xbd, 0x91, 0xe8, 0xbd, 0x92,
  0xe8, 0xbd, 0x93, 0xe8, 0xbd, 0x94, 0xe8, 0xbd, 0x95, 0xe8, 0xbd, 0x96,
  0xe8, 0xbd, 0x97, 0xe8, 0xbd, 0x98, 0xe8, 0xbd, 0x99, 0xe8, 0xbd, 0x9a,
  0xe8, 0xbd, 0x9b, 0xe8, 0xbd, 0x9c, 0xe8, 0xbd, 0x9d, 0xe8, 0xbd, 0x9e,
  0xe8, 0xbd, 0x9f, 0xe8, 0xbd, 0xa0, 0xe8, 0xbd, 0xa1, 0xe8, 0xbd, 0xa2,
  0xe8, 0xbd, 0xa3, 0xe8, 0xbd, 0xa4, 0xe8, 0xbd, 0xa5, 0xe8, 0xbd, 0xaa,
  0xe8, 0xbe, 0x80, 0xe8, 0xbe, 0x8c, 0xe8, 0xbe, 0x92, 0xe8, 0xbe, 0x9d,
  0xe8, 0xbe, 0xa0, 0xe8, 0xbe, 0xa1, 0xe8, 0xbe, 0xa2, 0xe8, 0xbe, 0xa4,
  0xe8, 0xbe, 0xa5, 0xe8, 0xbe, 0xa6, 0xe8, 0xbe, 0xa7, 0xe8, 0xbe, 0xaa,
  0xe8, 0xbe, 0xac, 0xe8, 0xbe, 0xad, 0xe8, 0xbe, 0xae, 0xe8, 0xbe, 0xaf,
  0xe8, 0xbe, 0xb2, 0xe8, 0xbe, 0xb3, 0xe8, 0xbe, 0xb4, 0xe8, 0xbe, 0xb5,
  0xe8, 0xbe, 0xb7, 0xe8, 0xbe, 0xb8, 0xe8, 0xbe, 0xba, 0xe8, 0xbe, 0xbb,
  0xe8, 0xbe, 0xbc, 0xe8, 0xbe, 0xbf, 0xe8, 0xbf, 0x80, 0xe8, 0xbf, 0x83,
  0xe8, 0xbf, 0x86, 0xe8, 0xbf, 0x89, 0xe8, 0xbf, 0x8a, 0xe8, 0xbf, 0x8b,
  0xe8, 0xbf, 0x8c, 0xe8, 0xbf, 0x8d, 0xe8, 0xbf, 0x8f, 0xe8, 0xbf, 0x92,
  0xe8, 0xbf, 0x96, 0xe8, 0xbf, 0x97, 0xe8, 0xbf, 0x9a, 0xe8, 0xbf, 0xa0,
  0xe8, 0xbf, 0xa1, 0xe8, 0xbf, 0xa3, 0xe8, 0xbf, 0xa7, 0xe8, 0xbf, 0xac,
  0xe8, 0xbf, 0xaf, 0xe8, 0xbf, 0xb1, 0xe8, 0xbf, 0xb2, 0xe8, 0xbf, 0xb4,
  0xe8, 0xbf, 0xb5, 0xe8, 0xbf, 0xb6, 0xe8, 0xbf, 0xba, 0xe8, 0xbf, 0xbb,
  0xe8, 0xbf, 0xbc, 0xe8, 0xbf, 0xbe, 0xe8, 0xbf, 0xbf, 0xe9, 0x80, 0x87,
  0xe9, 0x80, 0x88, 0xe9, 0x80, 0x8c, 0xe9, 0x80, 0x8e, 0xe9, 0x80, 0x93,
  0xe9, 0x80, 0x95, 0xe9, 0x80, 0x98, 0xe8, 0x95, 0x96, 0xe8, 0x94, 0xbb,
  0xe8, 0x93, 0xbf, 0xe8, 0x93, 0xbc, 0xe8, 0x95, 0x99, 0xe8, 0x95, 0x88,
  0xe8, 0x95, 0xa8, 0xe8, 0x95, 0xa4, 0xe8, 0x95, 0x9e, 0xe8, 0x95, 0xba,
  0xe7, 0x9e, 0xa2, 0xe8, 0x95, 0x83, 0xe8, 0x95, 0xb2, 0xe8, 0x95, 0xbb,
  0xe8, 0x96, 0xa4, 0xe8, 0x96, 0xa8, 0xe8, 0x96, 0x87, 0xe8, 0x96, 0x8f,
  0xe8, 0x95, 0xb9, 0xe8, 0x96, 0xae, 0xe8, 0x96, 0x9c, 0xe8, 0x96, 0x85,
  0xe8, 0x96, 0xb9, 0xe8, 0x96, 0xb7, 0xe8, 0x96, 0xb0, 0xe8, 0x97, 0x93,
  0xe8, 0x97, 0x81, 0xe8, 0x97, 0x9c, 0xe8, 0x97, 0xbf, 0xe8, 0x98, 0xa7,
  0xe8, 0x98, 0x85, 0xe8, 0x98, 0xa9, 0xe8, 0x98, 0x96, 0xe8, 0x98, 0xbc,
  0xe5, 0xbb, 0xbe, 0xe5, 0xbc, 0x88, 0xe5, 0xa4, 0xbc, 0xe5, 0xa5, 0x81,
  0xe8, 0x80, 0xb7, 0xe5, 0xa5, 0x95, 0xe5, 0xa5, 0x9a, 0xe5, 0xa5, 0x98,
  0xe5, 0x8c, 0x8f, 0xe5, 0xb0, 0xa2, 0xe5, 0xb0, 0xa5, 0xe5, 0xb0, 0xac,
  0xe5, 0xb0, 0xb4, 0xe6, 0x89, 0x8c, 0xe6, 0x89, 0xaa, 0xe6, 0x8a, 0x9f,
  0xe6, 0x8a, 0xbb, 0xe6, 0x8b, 0x8a, 0xe6, 0x8b, 0x9a, 0xe6, 0x8b, 0x97,
  0xe6, 0x8b, 0xae, 0xe6, 0x8c, 0xa2, 0xe6, 0x8b, 0xb6, 0xe6, 0x8c, 0xb9,
  0xe6, 0x8d, 0x8b, 0xe6, 0x8d, 0x83, 0xe6, 0x8e, 0xad, 0xe6, 0x8f, 0xb6,
  0xe6, 0x8d, 0xb1, 0xe6, 0x8d, 0xba, 0xe6, 0x8e, 0x8e, 0xe6, 0x8e, 0xb4,
  0xe6, 0x8d, 0xad, 0xe6, 0x8e, 0xac, 0xe6, 0x8e, 0x8a, 0xe6, 0x8d, 0xa9,
  0xe6, 0x8e, 0xae, 0xe6, 0x8e, 0xbc, 0xe6, 0x8f, 0xb2, 0xe6, 0x8f, 0xb8,
  0xe6, 0x8f, 0xa0, 0xe6, 0x8f, 0xbf, 0xe6, 0x8f, 0x84, 0xe6, 0x8f, 0x9e,
  0xe6, 0x8f, 0x8e, 0xe6, 0x91, 0x92, 0xe6, 0x8f, 0x86, 0xe6, 0x8e, 0xbe,
  0xe6, 0x91, 0x85, 0xe6, 0x91, 0x81, 0xe6, 0x90, 0x8b, 0xe6, 0x90, 0x9b,
  0xe6, 0x90, 0xa0, 0xe6, 0x90, 0x8c, 0xe6, 0x90, 0xa6, 0xe6, 0x90, 0xa1,
  0xe6, 0x91, 0x9e, 0xe6, 0x92, 0x84, 0xe6, 0x91, 0xad, 0xe6, 0x92, 0x96,
};

unsigned char gbk_row_0xdf_converted_to_utf8[] = {
        0xe9, 0x80, 0x99, 0xe9, 0x80, 0x9c, 0xe9, 0x80, 0xa3, 0xe9, 0x80,
  0xa4, 0xe9, 0x80, 0xa5, 0xe9, 0x80, 0xa7, 0xe9, 0x80, 0xa8, 0xe9, 0x80,
  0xa9, 0xe9, 0x80, 0xaa, 0xe9, 0x80, 0xab, 0xe9, 0x80, 0xac, 0xe9, 0x80,
  0xb0, 0xe9, 0x80, 0xb1, 0xe9, 0x80, 0xb2, 0xe9, 0x80, 0xb3, 0xe9, 0x80,
  0xb4, 0xe9, 0x80, 0xb7, 0xe9, 0x80, 0xb9, 0xe9, 0x80, 0xba, 0xe9, 0x80,
  0xbd, 0xe9, 0x80, 0xbf, 0xe9, 0x81, 0x80, 0xe9, 0x81, 0x83, 0xe9, 0x81,
  0x85, 0xe9, 0x81, 0x86, 0xe9, 0x81, 0x88, 0xe9, 0x81, 0x89, 0xe9, 0x81,
  0x8a, 0xe9, 0x81, 0x8b, 0xe9, 0x81, 0x8c, 0xe9, 0x81, 0x8e, 0xe9, 0x81,
  0x94, 0xe9, 0x81, 0x95, 0xe9, 0x81, 0x96, 0xe9, 0x81, 0x99, 0xe9, 0x81,
  0x9a, 0xe9, 0x81, 0x9c, 0xe9, 0x81, 0x9d, 0xe9, 0x81, 0x9e, 0xe9, 0x81,
  0x9f, 0xe9, 0x81, 0xa0, 0xe9, 0x81, 0xa1, 0xe9, 0x81, 0xa4, 0xe9, 0x81,
  0xa6, 0xe9, 0x81, 0xa7, 0xe9, 0x81, 0xa9, 0xe9, 0x81, 0xaa, 0xe9, 0x81,
  0xab, 0xe9, 0x81, 0xac, 0xe9, 0x81, 0xaf, 0xe9, 0x81, 0xb0, 0xe9, 0x81,
  0xb1, 0xe9, 0x81, 0xb2, 0xe9, 0x81, 0xb3, 0xe9, 0x81, 0xb6, 0xe9, 0x81,
  0xb7, 0xe9, 0x81, 0xb8, 0xe9, 0x81, 0xb9, 0xe9, 0x81, 0xba, 0xe9, 0x81,
  0xbb, 0xe9, 0x81, 0xbc, 0xe9, 0x81, 0xbe, 0xe9, 0x82, 0x81, 0xe9, 0x82,
  0x84, 0xe9, 0x82, 0x85, 0xe9, 0x82, 0x86, 0xe9, 0x82, 0x87, 0xe9, 0x82,
  0x89, 0xe9, 0x82, 0x8a, 0xe9, 0x82, 0x8c, 0xe9, 0x82, 0x8d, 0xe9, 0x82,
  0x8e, 0xe9, 0x82, 0x8f, 0xe9, 0x82, 0x90, 0xe9, 0x82, 0x92, 0xe9, 0x82,
  0x94, 0xe9, 0x82, 0x96, 0xe9, 0x82, 0x98, 0xe9, 0x82, 0x9a, 0xe9, 0x82,
  0x9c, 0xe9, 0x82, 0x9e, 0xe9, 0x82, 0x9f, 0xe9, 0x82, 0xa0, 0xe9, 0x82,
  0xa4, 0xe9, 0x82, 0xa5, 0xe9, 0x82, 0xa7, 0xe9, 0x82, 0xa8, 0xe9, 0x82,
  0xa9, 0xe9, 0x82, 0xab, 0xe9, 0x82, 0xad, 0xe9, 0x82, 0xb2, 0xe9, 0x82,
  0xb7, 0xe9, 0x82, 0xbc, 0xe9, 0x82, 0xbd, 0xe9, 0x82, 0xbf, 0xe9, 0x83,
  0x80, 0xe6, 0x91, 0xba, 0xe6, 0x92, 0xb7, 0xe6, 0x92, 0xb8, 0xe6, 0x92,
  0x99, 0xe6, 0x92, 0xba, 0xe6, 0x93, 0x80, 0xe6, 0x93, 0x90, 0xe6, 0x93,
  0x97, 0xe6, 0x93, 0xa4, 0xe6, 0x93, 0xa2, 0xe6, 0x94, 0x89, 0xe6, 0x94,
  0xa5, 0xe6, 0x94, 0xae, 0xe5, 0xbc, 0x8b, 0xe5, 0xbf, 0x92, 0xe7, 0x94,
  0x99, 0xe5, 0xbc, 0x91, 0xe5, 0x8d, 0x9f, 0xe5, 0x8f, 0xb1, 0xe5, 0x8f,
  0xbd, 0xe5, 0x8f, 0xa9, 0xe5, 0x8f, 0xa8, 0xe5, 0x8f, 0xbb, 0xe5, 0x90,
  0x92, 0xe5, 0x90, 0x96, 0xe5, 0x90, 0x86, 0xe5, 0x91, 0x8b, 0xe5, 0x91,
  0x92, 0xe5, 0x91, 0x93, 0xe5, 0x91, 0x94, 0xe5, 0x91, 0x96, 0xe5, 0x91,
  0x83, 0xe5, 0x90, 0xa1, 0xe5, 0x91, 0x97, 0xe5, 0x91, 0x99, 0xe5, 0x90,
  0xa3, 0xe5, 0x90, 0xb2, 0xe5, 0x92, 0x82, 0xe5, 0x92, 0x94, 0xe5, 0x91,
  0xb7, 0xe5, 0x91, 0xb1, 0xe5, 0x91, 0xa4, 0xe5, 0x92, 0x9a, 0xe5, 0x92,
  0x9b, 0xe5, 0x92, 0x84, 0xe5, 0x91, 0xb6, 0xe5, 0x91, 0xa6, 0xe5, 0x92,
  0x9d, 0xe5, 0x93, 0x90, 0xe5, 0x92, 0xad, 0xe5, 0x93, 0x82, 0xe5, 0x92,
  0xb4, 0xe5, 0x93, 0x92, 0xe5, 0x92, 0xa7, 0xe5, 0x92, 0xa6, 0xe5, 0x93,
  0x93, 0xe5, 0x93, 0x94, 0xe5, 0x91, 0xb2, 0xe5, 0x92, 0xa3, 0xe5, 0x93,
  0x95, 0xe5, 0x92, 0xbb, 0xe5, 0x92, 0xbf, 0xe5, 0x93, 0x8c, 0xe5, 0x93,
  0x99, 0xe5, 0x93, 0x9a, 0xe5, 0x93, 0x9c, 0xe5, 0x92, 0xa9, 0xe5, 0x92,
  0xaa, 0xe5, 0x92, 0xa4, 0xe5, 0x93, 0x9d, 0xe5, 0x93, 0x8f, 0xe5, 0x93,
  0x9e, 0xe5, 0x94, 0x9b, 0xe5, 0x93, 0xa7, 0xe5, 0x94, 0xa0, 0xe5, 0x93,
  0xbd, 0xe5, 0x94, 0x94, 0xe5, 0x93, 0xb3, 0xe5, 0x94, 0xa2, 0xe5, 0x94,
  0xa3, 0xe5, 0x94, 0x8f, 0xe5, 0x94, 0x91, 0xe5, 0x94, 0xa7, 0xe5, 0x94,
  0xaa, 0xe5, 0x95, 0xa7, 0xe5, 0x96, 0x8f, 0xe5, 0x96, 0xb5, 0xe5, 0x95,
  0x89, 0xe5, 0x95, 0xad, 0xe5, 0x95, 0x81, 0xe5, 0x95, 0x95, 0xe5, 0x94,
  0xbf, 0xe5, 0x95, 0x90, 0xe5, 0x94, 0xbc
};

unsigned char gbk_row_0xe0_converted_to_utf8[] = {
  0xe9, 0x83, 0x82, 0xe9,
  0x83, 0x83, 0xe9, 0x83, 0x86, 0xe9, 0x83, 0x88, 0xe9, 0x83, 0x89, 0xe9,
  0x83, 0x8b, 0xe9, 0x83, 0x8c, 0xe9, 0x83, 0x8d, 0xe9, 0x83, 0x92, 0xe9,
  0x83, 0x94, 0xe9, 0x83, 0x95, 0xe9, 0x83, 0x96, 0xe9, 0x83, 0x98, 0xe9,
  0x83, 0x99, 0xe9, 0x83, 0x9a, 0xe9, 0x83, 0x9e, 0xe9, 0x83, 0x9f, 0xe9,
  0x83, 0xa0, 0xe9, 0x83, 0xa3, 0xe9, 0x83, 0xa4, 0xe9, 0x83, 0xa5, 0xe9,
  0x83, 0xa9, 0xe9, 0x83, 0xaa, 0xe9, 0x83, 0xac, 0xe9, 0x83, 0xae, 0xe9,
  0x83, 0xb0, 0xe9, 0x83, 0xb1, 0xe9, 0x83, 0xb2, 0xe9, 0x83, 0xb3, 0xe9,
  0x83, 0xb5, 0xe9, 0x83, 0xb6, 0xe9, 0x83, 0xb7, 0xe9, 0x83, 0xb9, 0xe9,
  0x83, 0xba, 0xe9, 0x83, 0xbb, 0xe9, 0x83, 0xbc, 0xe9, 0x83, 0xbf, 0xe9,
  0x84, 0x80, 0xe9, 0x84, 0x81, 0xe9, 0x84, 0x83, 0xe9, 0x84, 0x85, 0xe9,
  0x84, 0x86, 0xe9, 0x84, 0x87, 0xe9, 0x84, 0x88, 0xe9, 0x84, 0x89, 0xe9,
  0x84, 0x8a, 0xe9, 0x84, 0x8b, 0xe9, 0x84, 0x8c, 0xe9, 0x84, 0x8d, 0xe9,
  0x84, 0x8e, 0xe9, 0x84, 0x8f, 0xe9, 0x84, 0x90, 0xe9, 0x84, 0x91, 0xe9,
  0x84, 0x92, 0xe9, 0x84, 0x93, 0xe9, 0x84, 0x94, 0xe9, 0x84, 0x95, 0xe9,
  0x84, 0x96, 0xe9, 0x84, 0x97, 0xe9, 0x84, 0x98, 0xe9, 0x84, 0x9a, 0xe9,
  0x84, 0x9b, 0xe9, 0x84, 0x9c, 0xe9, 0x84, 0x9d, 0xe9, 0x84, 0x9f, 0xe9,
  0x84, 0xa0, 0xe9, 0x84, 0xa1, 0xe9, 0x84, 0xa4, 0xe9, 0x84, 0xa5, 0xe9,
  0x84, 0xa6, 0xe9, 0x84, 0xa7, 0xe9, 0x84, 0xa8, 0xe9, 0x84, 0xa9, 0xe9,
  0x84, 0xaa, 0xe9, 0x84, 0xab, 0xe9, 0x84, 0xac, 0xe9, 0x84, 0xad, 0xe9,
  0x84, 0xae, 0xe9, 0x84, 0xb0, 0xe9, 0x84, 0xb2, 0xe9, 0x84, 0xb3, 0xe9,
  0x84, 0xb4, 0xe9, 0x84, 0xb5, 0xe9, 0x84, 0xb6, 0xe9, 0x84, 0xb7, 0xe9,
  0x84, 0xb8, 0xe9, 0x84, 0xba, 0xe9, 0x84, 0xbb, 0xe9, 0x84, 0xbc, 0xe9,
  0x84, 0xbd, 0xe9, 0x84, 0xbe, 0xe9, 0x84, 0xbf, 0xe9, 0x85, 0x80, 0xe9,
  0x85, 0x81, 0xe9, 0x85, 0x82, 0xe9, 0x85, 0x84, 0xe5, 0x94, 0xb7, 0xe5,
  0x95, 0x96, 0xe5, 0x95, 0xb5, 0xe5, 0x95, 0xb6, 0xe5, 0x95, 0xb7, 0xe5,
  0x94, 0xb3, 0xe5, 0x94, 0xb0, 0xe5, 0x95, 0x9c, 0xe5, 0x96, 0x8b, 0xe5,
  0x97, 0x92, 0xe5, 0x96, 0x83, 0xe5, 0x96, 0xb1, 0xe5, 0x96, 0xb9, 0xe5,
  0x96, 0x88, 0xe5, 0x96, 0x81, 0xe5, 0x96, 0x9f, 0xe5, 0x95, 0xbe, 0xe5,
  0x97, 0x96, 0xe5, 0x96, 0x91, 0xe5, 0x95, 0xbb, 0xe5, 0x97, 0x9f, 0xe5,
  0x96, 0xbd, 0xe5, 0x96, 0xbe, 0xe5, 0x96, 0x94, 0xe5, 0x96, 0x99, 0xe5,
  0x97, 0xaa, 0xe5, 0x97, 0xb7, 0xe5, 0x97, 0x89, 0xe5, 0x98, 0x9f, 0xe5,
  0x97, 0x91, 0xe5, 0x97, 0xab, 0xe5, 0x97, 0xac, 0xe5, 0x97, 0x94, 0xe5,
  0x97, 0xa6, 0xe5, 0x97, 0x9d, 0xe5, 0x97, 0x84, 0xe5, 0x97, 0xaf, 0xe5,
  0x97, 0xa5, 0xe5, 0x97, 0xb2, 0xe5, 0x97, 0xb3, 0xe5, 0x97, 0x8c, 0xe5,
  0x97, 0x8d, 0xe5, 0x97, 0xa8, 0xe5, 0x97, 0xb5, 0xe5, 0x97, 0xa4, 0xe8,
  0xbe, 0x94, 0xe5, 0x98, 0x9e, 0xe5, 0x98, 0x88, 0xe5, 0x98, 0x8c, 0xe5,
  0x98, 0x81, 0xe5, 0x98, 0xa4, 0xe5, 0x98, 0xa3, 0xe5, 0x97, 0xbe, 0xe5,
  0x98, 0x80, 0xe5, 0x98, 0xa7, 0xe5, 0x98, 0xad, 0xe5, 0x99, 0x98, 0xe5,
  0x98, 0xb9, 0xe5, 0x99, 0x97, 0xe5, 0x98, 0xac, 0xe5, 0x99, 0x8d, 0xe5,
  0x99, 0xa2, 0xe5, 0x99, 0x99, 0xe5, 0x99, 0x9c, 0xe5, 0x99, 0x8c, 0xe5,
  0x99, 0x94, 0xe5, 0x9a, 0x86, 0xe5, 0x99, 0xa4, 0xe5, 0x99, 0xb1, 0xe5,
  0x99, 0xab, 0xe5, 0x99, 0xbb, 0xe5, 0x99, 0xbc, 0xe5, 0x9a, 0x85, 0xe5,
  0x9a, 0x93, 0xe5, 0x9a, 0xaf, 0xe5, 0x9b, 0x94, 0xe5, 0x9b, 0x97, 0xe5,
  0x9b, 0x9d, 0xe5, 0x9b, 0xa1, 0xe5, 0x9b, 0xb5, 0xe5, 0x9b, 0xab, 0xe5,
  0x9b, 0xb9, 0xe5, 0x9b, 0xbf, 0xe5, 0x9c, 0x84, 0xe5, 0x9c, 0x8a, 0xe5,
  0x9c, 0x89, 0xe5, 0x9c, 0x9c, 0xe5, 0xb8, 0x8f, 0xe5, 0xb8, 0x99, 0xe5,
  0xb8, 0x94, 0xe5, 0xb8, 0x91, 0xe5, 0xb8, 0xb1, 0xe5, 0xb8, 0xbb, 0xe5,
  0xb8, 0xbc
};

unsigned char gbk_row_0xe1_converted_to_utf8[] = {
  0xe9, 0x85, 0x85, 0xe9, 0x85, 0x87, 0xe9, 0x85, 0x88,
  0xe9, 0x85, 0x91, 0xe9, 0x85, 0x93, 0xe9, 0x85, 0x94, 0xe9, 0x85, 0x95,
  0xe9, 0x85, 0x96, 0xe9, 0x85, 0x98, 0xe9, 0x85, 0x99, 0xe9, 0x85, 0x9b,
  0xe9, 0x85, 0x9c, 0xe9, 0x85, 0x9f, 0xe9, 0x85, 0xa0, 0xe9, 0x85, 0xa6,
  0xe9, 0x85, 0xa7, 0xe9, 0x85, 0xa8, 0xe9, 0x85, 0xab, 0xe9, 0x85, 0xad,
  0xe9, 0x85, 0xb3, 0xe9, 0x85, 0xba, 0xe9, 0x85, 0xbb, 0xe9, 0x85, 0xbc,
  0xe9, 0x86, 0x80, 0xe9, 0x86, 0x81, 0xe9, 0x86, 0x82, 0xe9, 0x86, 0x83,
  0xe9, 0x86, 0x84, 0xe9, 0x86, 0x86, 0xe9, 0x86, 0x88, 0xe9, 0x86, 0x8a,
  0xe9, 0x86, 0x8e, 0xe9, 0x86, 0x8f, 0xe9, 0x86, 0x93, 0xe9, 0x86, 0x94,
  0xe9, 0x86, 0x95, 0xe9, 0x86, 0x96, 0xe9, 0x86, 0x97, 0xe9, 0x86, 0x98,
  0xe9, 0x86, 0x99, 0xe9, 0x86, 0x9c, 0xe9, 0x86, 0x9d, 0xe9, 0x86, 0x9e,
  0xe9, 0x86, 0x9f, 0xe9, 0x86, 0xa0, 0xe9, 0x86, 0xa1, 0xe9, 0x86, 0xa4,
  0xe9, 0x86, 0xa5, 0xe9, 0x86, 0xa6, 0xe9, 0x86, 0xa7, 0xe9, 0x86, 0xa8,
  0xe9, 0x86, 0xa9, 0xe9, 0x86, 0xab, 0xe9, 0x86, 0xac, 0xe9, 0x86, 0xb0,
  0xe9, 0x86, 0xb1, 0xe9, 0x86, 0xb2, 0xe9, 0x86, 0xb3, 0xe9, 0x86, 0xb6,
  0xe9, 0x86, 0xb7, 0xe9, 0x86, 0xb8, 0xe9, 0x86, 0xb9, 0xe9, 0x86, 0xbb,
  0xe9, 0x86, 0xbc, 0xe9, 0x86, 0xbd, 0xe9, 0x86, 0xbe, 0xe9, 0x86, 0xbf,
  0xe9, 0x87, 0x80, 0xe9, 0x87, 0x81, 0xe9, 0x87, 0x82, 0xe9, 0x87, 0x83,
  0xe9, 0x87, 0x84, 0xe9, 0x87, 0x85, 0xe9, 0x87, 0x86, 0xe9, 0x87, 0x88,
  0xe9, 0x87, 0x8b, 0xe9, 0x87, 0x90, 0xe9, 0x87, 0x92, 0xe9, 0x87, 0x93,
  0xe9, 0x87, 0x94, 0xe9, 0x87, 0x95, 0xe9, 0x87, 0x96, 0xe9, 0x87, 0x97,
  0xe9, 0x87, 0x98, 0xe9, 0x87, 0x99, 0xe9, 0x87, 0x9a, 0xe9, 0x87, 0x9b,
  0xe9, 0x87, 0x9d, 0xe9, 0x87, 0x9e, 0xe9, 0x87, 0x9f, 0xe9, 0x87, 0xa0,
  0xe9, 0x87, 0xa1, 0xe9, 0x87, 0xa2, 0xe9, 0x87, 0xa3, 0xe9, 0x87, 0xa4,
  0xe9, 0x87, 0xa5, 0xe5, 0xb8, 0xb7, 0xe5, 0xb9, 0x84, 0xe5, 0xb9, 0x94,
  0xe5, 0xb9, 0x9b, 0xe5, 0xb9, 0x9e, 0xe5, 0xb9, 0xa1, 0xe5, 0xb2, 0x8c,
  0xe5, 0xb1, 0xba, 0xe5, 0xb2, 0x8d, 0xe5, 0xb2, 0x90, 0xe5, 0xb2, 0x96,
  0xe5, 0xb2, 0x88, 0xe5, 0xb2, 0x98, 0xe5, 0xb2, 0x99, 0xe5, 0xb2, 0x91,
  0xe5, 0xb2, 0x9a, 0xe5, 0xb2, 0x9c, 0xe5, 0xb2, 0xb5, 0xe5, 0xb2, 0xa2,
  0xe5, 0xb2, 0xbd, 0xe5, 0xb2, 0xac, 0xe5, 0xb2, 0xab, 0xe5, 0xb2, 0xb1,
  0xe5, 0xb2, 0xa3, 0xe5, 0xb3, 0x81, 0xe5, 0xb2, 0xb7, 0xe5, 0xb3, 0x84,
  0xe5, 0xb3, 0x92, 0xe5, 0xb3, 0xa4, 0xe5, 0xb3, 0x8b, 0xe5, 0xb3, 0xa5,
  0xe5, 0xb4, 0x82, 0xe5, 0xb4, 0x83, 0xe5, 0xb4, 0xa7, 0xe5, 0xb4, 0xa6,
  0xe5, 0xb4, 0xae, 0xe5, 0xb4, 0xa4, 0xe5, 0xb4, 0x9e, 0xe5, 0xb4, 0x86,
  0xe5, 0xb4, 0x9b, 0xe5, 0xb5, 0x98, 0xe5, 0xb4, 0xbe, 0xe5, 0xb4, 0xb4,
  0xe5, 0xb4, 0xbd, 0xe5, 0xb5, 0xac, 0xe5, 0xb5, 0x9b, 0xe5, 0xb5, 0xaf,
  0xe5, 0xb5, 0x9d, 0xe5, 0xb5, 0xab, 0xe5, 0xb5, 0x8b, 0xe5, 0xb5, 0x8a,
  0xe5, 0xb5, 0xa9, 0xe5, 0xb5, 0xb4, 0xe5, 0xb6, 0x82, 0xe5, 0xb6, 0x99,
  0xe5, 0xb6, 0x9d, 0xe8, 0xb1, 0xb3, 0xe5, 0xb6, 0xb7, 0xe5, 0xb7, 0x85,
  0xe5, 0xbd, 0xb3, 0xe5, 0xbd, 0xb7, 0xe5, 0xbe, 0x82, 0xe5, 0xbe, 0x87,
  0xe5, 0xbe, 0x89, 0xe5, 0xbe, 0x8c, 0xe5, 0xbe, 0x95, 0xe5, 0xbe, 0x99,
  0xe5, 0xbe, 0x9c, 0xe5, 0xbe, 0xa8, 0xe5, 0xbe, 0xad, 0xe5, 0xbe, 0xb5,
  0xe5, 0xbe, 0xbc, 0xe8, 0xa1, 0xa2, 0xe5, 0xbd, 0xa1, 0xe7, 0x8a, 0xad,
  0xe7, 0x8a, 0xb0, 0xe7, 0x8a, 0xb4, 0xe7, 0x8a, 0xb7, 0xe7, 0x8a, 0xb8,
  0xe7, 0x8b, 0x83, 0xe7, 0x8b, 0x81, 0xe7, 0x8b, 0x8e, 0xe7, 0x8b, 0x8d,
  0xe7, 0x8b, 0x92, 0xe7, 0x8b, 0xa8, 0xe7, 0x8b, 0xaf, 0xe7, 0x8b, 0xa9,
  0xe7, 0x8b, 0xb2, 0xe7, 0x8b, 0xb4, 0xe7, 0x8b, 0xb7, 0xe7, 0x8c, 0x81,
  0xe7, 0x8b, 0xb3, 0xe7, 0x8c, 0x83, 0xe7, 0x8b, 0xba
};

unsigned char gbk_row_0xe2_converted_to_utf8[] = {
  0xe9, 0x87,
  0xa6, 0xe9, 0x87, 0xa7, 0xe9, 0x87, 0xa8, 0xe9, 0x87, 0xa9, 0xe9, 0x87,
  0xaa, 0xe9, 0x87, 0xab, 0xe9, 0x87, 0xac, 0xe9, 0x87, 0xad, 0xe9, 0x87,
  0xae, 0xe9, 0x87, 0xaf, 0xe9, 0x87, 0xb0, 0xe9, 0x87, 0xb1, 0xe9, 0x87,
  0xb2, 0xe9, 0x87, 0xb3, 0xe9, 0x87, 0xb4, 0xe9, 0x87, 0xb5, 0xe9, 0x87,
  0xb6, 0xe9, 0x87, 0xb7, 0xe9, 0x87, 0xb8, 0xe9, 0x87, 0xb9, 0xe9, 0x87,
  0xba, 0xe9, 0x87, 0xbb, 0xe9, 0x87, 0xbc, 0xe9, 0x87, 0xbd, 0xe9, 0x87,
  0xbe, 0xe9, 0x87, 0xbf, 0xe9, 0x88, 0x80, 0xe9, 0x88, 0x81, 0xe9, 0x88,
  0x82, 0xe9, 0x88, 0x83, 0xe9, 0x88, 0x84, 0xe9, 0x88, 0x85, 0xe9, 0x88,
  0x86, 0xe9, 0x88, 0x87, 0xe9, 0x88, 0x88, 0xe9, 0x88, 0x89, 0xe9, 0x88,
  0x8a, 0xe9, 0x88, 0x8b, 0xe9, 0x88, 0x8c, 0xe9, 0x88, 0x8d, 0xe9, 0x88,
  0x8e, 0xe9, 0x88, 0x8f, 0xe9, 0x88, 0x90, 0xe9, 0x88, 0x91, 0xe9, 0x88,
  0x92, 0xe9, 0x88, 0x93, 0xe9, 0x88, 0x94, 0xe9, 0x88, 0x95, 0xe9, 0x88,
  0x96, 0xe9, 0x88, 0x97, 0xe9, 0x88, 0x98, 0xe9, 0x88, 0x99, 0xe9, 0x88,
  0x9a, 0xe9, 0x88, 0x9b, 0xe9, 0x88, 0x9c, 0xe9, 0x88, 0x9d, 0xe9, 0x88,
  0x9e, 0xe9, 0x88, 0x9f, 0xe9, 0x88, 0xa0, 0xe9, 0x88, 0xa1, 0xe9, 0x88,
  0xa2, 0xe9, 0x88, 0xa3, 0xe9, 0x88, 0xa4, 0xe9, 0x88, 0xa5, 0xe9, 0x88,
  0xa6, 0xe9, 0x88, 0xa7, 0xe9, 0x88, 0xa8, 0xe9, 0x88, 0xa9, 0xe9, 0x88,
  0xaa, 0xe9, 0x88, 0xab, 0xe9, 0x88, 0xac, 0xe9, 0x88, 0xad, 0xe9, 0x88,
  0xae, 0xe9, 0x88, 0xaf, 0xe9, 0x88, 0xb0, 0xe9, 0x88, 0xb1, 0xe9, 0x88,
  0xb2, 0xe9, 0x88, 0xb3, 0xe9, 0x88, 0xb4, 0xe9, 0x88, 0xb5, 0xe9, 0x88,
  0xb6, 0xe9, 0x88, 0xb7, 0xe9, 0x88, 0xb8, 0xe9, 0x88, 0xb9, 0xe9, 0x88,
  0xba, 0xe9, 0x88, 0xbb, 0xe9, 0x88, 0xbc, 0xe9, 0x88, 0xbd, 0xe9, 0x88,
  0xbe, 0xe9, 0x88, 0xbf, 0xe9, 0x89, 0x80, 0xe9, 0x89, 0x81, 0xe9, 0x89,
  0x82, 0xe9, 0x89, 0x83, 0xe9, 0x89, 0x84, 0xe9, 0x89, 0x85, 0xe7, 0x8b,
  0xbb, 0xe7, 0x8c, 0x97, 0xe7, 0x8c, 0x93, 0xe7, 0x8c, 0xa1, 0xe7, 0x8c,
  0x8a, 0xe7, 0x8c, 0x9e, 0xe7, 0x8c, 0x9d, 0xe7, 0x8c, 0x95, 0xe7, 0x8c,
  0xa2, 0xe7, 0x8c, 0xb9, 0xe7, 0x8c, 0xa5, 0xe7, 0x8c, 0xac, 0xe7, 0x8c,
  0xb8, 0xe7, 0x8c, 0xb1, 0xe7, 0x8d, 0x90, 0xe7, 0x8d, 0x8d, 0xe7, 0x8d,
  0x97, 0xe7, 0x8d, 0xa0, 0xe7, 0x8d, 0xac, 0xe7, 0x8d, 0xaf, 0xe7, 0x8d,
  0xbe, 0xe8, 0x88, 0x9b, 0xe5, 0xa4, 0xa5, 0xe9, 0xa3, 0xa7, 0xe5, 0xa4,
  0xa4, 0xe5, 0xa4, 0x82, 0xe9, 0xa5, 0xa3, 0xe9, 0xa5, 0xa7, 0xe9, 0xa5,
  0xa8, 0xe9, 0xa5, 0xa9, 0xe9, 0xa5, 0xaa, 0xe9, 0xa5, 0xab, 0xe9, 0xa5,
  0xac, 0xe9, 0xa5, 0xb4, 0xe9, 0xa5, 0xb7, 0xe9, 0xa5, 0xbd, 0xe9, 0xa6,
  0x80, 0xe9, 0xa6, 0x84, 0xe9, 0xa6, 0x87, 0xe9, 0xa6, 0x8a, 0xe9, 0xa6,
  0x8d, 0xe9, 0xa6, 0x90, 0xe9, 0xa6, 0x91, 0xe9, 0xa6, 0x93, 0xe9, 0xa6,
  0x94, 0xe9, 0xa6, 0x95, 0xe5, 0xba, 0x80, 0xe5, 0xba, 0x91, 0xe5, 0xba,
  0x8b, 0xe5, 0xba, 0x96, 0xe5, 0xba, 0xa5, 0xe5, 0xba, 0xa0, 0xe5, 0xba,
  0xb9, 0xe5, 0xba, 0xb5, 0xe5, 0xba, 0xbe, 0xe5, 0xba, 0xb3, 0xe8, 0xb5,
  0x93, 0xe5, 0xbb, 0x92, 0xe5, 0xbb, 0x91, 0xe5, 0xbb, 0x9b, 0xe5, 0xbb,
  0xa8, 0xe5, 0xbb, 0xaa, 0xe8, 0x86, 0xba, 0xe5, 0xbf, 0x84, 0xe5, 0xbf,
  0x89, 0xe5, 0xbf, 0x96, 0xe5, 0xbf, 0x8f, 0xe6, 0x80, 0x83, 0xe5, 0xbf,
  0xae, 0xe6, 0x80, 0x84, 0xe5, 0xbf, 0xa1, 0xe5, 0xbf, 0xa4, 0xe5, 0xbf,
  0xbe, 0xe6, 0x80, 0x85, 0xe6, 0x80, 0x86, 0xe5, 0xbf, 0xaa, 0xe5, 0xbf,
  0xad, 0xe5, 0xbf, 0xb8, 0xe6, 0x80, 0x99, 0xe6, 0x80, 0xb5, 0xe6, 0x80,
  0xa6, 0xe6, 0x80, 0x9b, 0xe6, 0x80, 0x8f, 0xe6, 0x80, 0x8d, 0xe6, 0x80,
  0xa9, 0xe6, 0x80, 0xab, 0xe6, 0x80, 0x8a, 0xe6, 0x80, 0xbf, 0xe6, 0x80,
  0xa1, 0xe6, 0x81, 0xb8, 0xe6, 0x81, 0xb9, 0xe6, 0x81, 0xbb, 0xe6, 0x81,
  0xba, 0xe6, 0x81, 0x82
};

unsigned char gbk_row_0xe3_converted_to_utf8[] = {
  0xe9, 0x89, 0x86, 0xe9, 0x89, 0x87, 0xe9,
  0x89, 0x88, 0xe9, 0x89, 0x89, 0xe9, 0x89, 0x8a, 0xe9, 0x89, 0x8b, 0xe9,
  0x89, 0x8c, 0xe9, 0x89, 0x8d, 0xe9, 0x89, 0x8e, 0xe9, 0x89, 0x8f, 0xe9,
  0x89, 0x90, 0xe9, 0x89, 0x91, 0xe9, 0x89, 0x92, 0xe9, 0x89, 0x93, 0xe9,
  0x89, 0x94, 0xe9, 0x89, 0x95, 0xe9, 0x89, 0x96, 0xe9, 0x89, 0x97, 0xe9,
  0x89, 0x98, 0xe9, 0x89, 0x99, 0xe9, 0x89, 0x9a, 0xe9, 0x89, 0x9b, 0xe9,
  0x89, 0x9c, 0xe9, 0x89, 0x9d, 0xe9, 0x89, 0x9e, 0xe9, 0x89, 0x9f, 0xe9,
  0x89, 0xa0, 0xe9, 0x89, 0xa1, 0xe9, 0x89, 0xa2, 0xe9, 0x89, 0xa3, 0xe9,
  0x89, 0xa4, 0xe9, 0x89, 0xa5, 0xe9, 0x89, 0xa6, 0xe9, 0x89, 0xa7, 0xe9,
  0x89, 0xa8, 0xe9, 0x89, 0xa9, 0xe9, 0x89, 0xaa, 0xe9, 0x89, 0xab, 0xe9,
  0x89, 0xac, 0xe9, 0x89, 0xad, 0xe9, 0x89, 0xae, 0xe9, 0x89, 0xaf, 0xe9,
  0x89, 0xb0, 0xe9, 0x89, 0xb1, 0xe9, 0x89, 0xb2, 0xe9, 0x89, 0xb3, 0xe9,
  0x89, 0xb5, 0xe9, 0x89, 0xb6, 0xe9, 0x89, 0xb7, 0xe9, 0x89, 0xb8, 0xe9,
  0x89, 0xb9, 0xe9, 0x89, 0xba, 0xe9, 0x89, 0xbb, 0xe9, 0x89, 0xbc, 0xe9,
  0x89, 0xbd, 0xe9, 0x89, 0xbe, 0xe9, 0x89, 0xbf, 0xe9, 0x8a, 0x80, 0xe9,
  0x8a, 0x81, 0xe9, 0x8a, 0x82, 0xe9, 0x8a, 0x83, 0xe9, 0x8a, 0x84, 0xe9,
  0x8a, 0x85, 0xe9, 0x8a, 0x86, 0xe9, 0x8a, 0x87, 0xe9, 0x8a, 0x88, 0xe9,
  0x8a, 0x89, 0xe9, 0x8a, 0x8a, 0xe9, 0x8a, 0x8b, 0xe9, 0x8a, 0x8c, 0xe9,
  0x8a, 0x8d, 0xe9, 0x8a, 0x8f, 0xe9, 0x8a, 0x90, 0xe9, 0x8a, 0x91, 0xe9,
  0x8a, 0x92, 0xe9, 0x8a, 0x93, 0xe9, 0x8a, 0x94, 0xe9, 0x8a, 0x95, 0xe9,
  0x8a, 0x96, 0xe9, 0x8a, 0x97, 0xe9, 0x8a, 0x98, 0xe9, 0x8a, 0x99, 0xe9,
  0x8a, 0x9a, 0xe9, 0x8a, 0x9b, 0xe9, 0x8a, 0x9c, 0xe9, 0x8a, 0x9d, 0xe9,
  0x8a, 0x9e, 0xe9, 0x8a, 0x9f, 0xe9, 0x8a, 0xa0, 0xe9, 0x8a, 0xa1, 0xe9,
  0x8a, 0xa2, 0xe9, 0x8a, 0xa3, 0xe9, 0x8a, 0xa4, 0xe9, 0x8a, 0xa5, 0xe9,
  0x8a, 0xa6, 0xe9, 0x8a, 0xa7, 0xe6, 0x81, 0xaa, 0xe6, 0x81, 0xbd, 0xe6,
  0x82, 0x96, 0xe6, 0x82, 0x9a, 0xe6, 0x82, 0xad, 0xe6, 0x82, 0x9d, 0xe6,
  0x82, 0x83, 0xe6, 0x82, 0x92, 0xe6, 0x82, 0x8c, 0xe6, 0x82, 0x9b, 0xe6,
  0x83, 0xac, 0xe6, 0x82, 0xbb, 0xe6, 0x82, 0xb1, 0xe6, 0x83, 0x9d, 0xe6,
  0x83, 0x98, 0xe6, 0x83, 0x86, 0xe6, 0x83, 0x9a, 0xe6, 0x82, 0xb4, 0xe6,
  0x84, 0xa0, 0xe6, 0x84, 0xa6, 0xe6, 0x84, 0x95, 0xe6, 0x84, 0xa3, 0xe6,
  0x83, 0xb4, 0xe6, 0x84, 0x80, 0xe6, 0x84, 0x8e, 0xe6, 0x84, 0xab, 0xe6,
  0x85, 0x8a, 0xe6, 0x85, 0xb5, 0xe6, 0x86, 0xac, 0xe6, 0x86, 0x94, 0xe6,
  0x86, 0xa7, 0xe6, 0x86, 0xb7, 0xe6, 0x87, 0x94, 0xe6, 0x87, 0xb5, 0xe5,
  0xbf, 0x9d, 0xe9, 0x9a, 0xb3, 0xe9, 0x97, 0xa9, 0xe9, 0x97, 0xab, 0xe9,
  0x97, 0xb1, 0xe9, 0x97, 0xb3, 0xe9, 0x97, 0xb5, 0xe9, 0x97, 0xb6, 0xe9,
  0x97, 0xbc, 0xe9, 0x97, 0xbe, 0xe9, 0x98, 0x83, 0xe9, 0x98, 0x84, 0xe9,
  0x98, 0x86, 0xe9, 0x98, 0x88, 0xe9, 0x98, 0x8a, 0xe9, 0x98, 0x8b, 0xe9,
  0x98, 0x8c, 0xe9, 0x98, 0x8d, 0xe9, 0x98, 0x8f, 0xe9, 0x98, 0x92, 0xe9,
  0x98, 0x95, 0xe9, 0x98, 0x96, 0xe9, 0x98, 0x97, 0xe9, 0x98, 0x99, 0xe9,
  0x98, 0x9a, 0xe4, 0xb8, 0xac, 0xe7, 0x88, 0xbf, 0xe6, 0x88, 0x95, 0xe6,
  0xb0, 0xb5, 0xe6, 0xb1, 0x94, 0xe6, 0xb1, 0x9c, 0xe6, 0xb1, 0x8a, 0xe6,
  0xb2, 0xa3, 0xe6, 0xb2, 0x85, 0xe6, 0xb2, 0x90, 0xe6, 0xb2, 0x94, 0xe6,
  0xb2, 0x8c, 0xe6, 0xb1, 0xa8, 0xe6, 0xb1, 0xa9, 0xe6, 0xb1, 0xb4, 0xe6,
  0xb1, 0xb6, 0xe6, 0xb2, 0x86, 0xe6, 0xb2, 0xa9, 0xe6, 0xb3, 0x90, 0xe6,
  0xb3, 0x94, 0xe6, 0xb2, 0xad, 0xe6, 0xb3, 0xb7, 0xe6, 0xb3, 0xb8, 0xe6,
  0xb3, 0xb1, 0xe6, 0xb3, 0x97, 0xe6, 0xb2, 0xb2, 0xe6, 0xb3, 0xa0, 0xe6,
  0xb3, 0x96, 0xe6, 0xb3, 0xba, 0xe6, 0xb3, 0xab, 0xe6, 0xb3, 0xae, 0xe6,
  0xb2, 0xb1, 0xe6, 0xb3, 0x93, 0xe6, 0xb3, 0xaf, 0xe6, 0xb3, 0xbe
};

unsigned char gbk_row_0xe4_converted_to_utf8[] = {
  0xe9, 0x8a, 0xa8, 0xe9, 0x8a, 0xa9, 0xe9, 0x8a, 0xaa, 0xe9, 0x8a, 0xab,
  0xe9, 0x8a, 0xac, 0xe9, 0x8a, 0xad, 0xe9, 0x8a, 0xaf, 0xe9, 0x8a, 0xb0,
  0xe9, 0x8a, 0xb1, 0xe9, 0x8a, 0xb2, 0xe9, 0x8a, 0xb3, 0xe9, 0x8a, 0xb4,
  0xe9, 0x8a, 0xb5, 0xe9, 0x8a, 0xb6, 0xe9, 0x8a, 0xb7, 0xe9, 0x8a, 0xb8,
  0xe9, 0x8a, 0xb9, 0xe9, 0x8a, 0xba, 0xe9, 0x8a, 0xbb, 0xe9, 0x8a, 0xbc,
  0xe9, 0x8a, 0xbd, 0xe9, 0x8a, 0xbe, 0xe9, 0x8a, 0xbf, 0xe9, 0x8b, 0x80,
  0xe9, 0x8b, 0x81, 0xe9, 0x8b, 0x82, 0xe9, 0x8b, 0x83, 0xe9, 0x8b, 0x84,
  0xe9, 0x8b, 0x85, 0xe9, 0x8b, 0x86, 0xe9, 0x8b, 0x87, 0xe9, 0x8b, 0x89,
  0xe9, 0x8b, 0x8a, 0xe9, 0x8b, 0x8b, 0xe9, 0x8b, 0x8c, 0xe9, 0x8b, 0x8d,
  0xe9, 0x8b, 0x8e, 0xe9, 0x8b, 0x8f, 0xe9, 0x8b, 0x90, 0xe9, 0x8b, 0x91,
  0xe9, 0x8b, 0x92, 0xe9, 0x8b, 0x93, 0xe9, 0x8b, 0x94, 0xe9, 0x8b, 0x95,
  0xe9, 0x8b, 0x96, 0xe9, 0x8b, 0x97, 0xe9, 0x8b, 0x98, 0xe9, 0x8b, 0x99,
  0xe9, 0x8b, 0x9a, 0xe9, 0x8b, 0x9b, 0xe9, 0x8b, 0x9c, 0xe9, 0x8b, 0x9d,
  0xe9, 0x8b, 0x9e, 0xe9, 0x8b, 0x9f, 0xe9, 0x8b, 0xa0, 0xe9, 0x8b, 0xa1,
  0xe9, 0x8b, 0xa2, 0xe9, 0x8b, 0xa3, 0xe9, 0x8b, 0xa4, 0xe9, 0x8b, 0xa5,
  0xe9, 0x8b, 0xa6, 0xe9, 0x8b, 0xa7, 0xe9, 0x8b, 0xa8, 0xe9, 0x8b, 0xa9,
  0xe9, 0x8b, 0xaa, 0xe9, 0x8b, 0xab, 0xe9, 0x8b, 0xac, 0xe9, 0x8b, 0xad,
  0xe9, 0x8b, 0xae, 0xe9, 0x8b, 0xaf, 0xe9, 0x8b, 0xb0, 0xe9, 0x8b, 0xb1,
  0xe9, 0x8b, 0xb2, 0xe9, 0x8b, 0xb3, 0xe9, 0x8b, 0xb4, 0xe9, 0x8b, 0xb5,
  0xe9, 0x8b, 0xb6, 0xe9, 0x8b, 0xb7, 0xe9, 0x8b, 0xb8, 0xe9, 0x8b, 0xb9,
  0xe9, 0x8b, 0xba, 0xe9, 0x8b, 0xbb, 0xe9, 0x8b, 0xbc, 0xe9, 0x8b, 0xbd,
  0xe9, 0x8b, 0xbe, 0xe9, 0x8b, 0xbf, 0xe9, 0x8c, 0x80, 0xe9, 0x8c, 0x81,
  0xe9, 0x8c, 0x82, 0xe9, 0x8c, 0x83, 0xe9, 0x8c, 0x84, 0xe9, 0x8c, 0x85,
  0xe9, 0x8c, 0x86, 0xe9, 0x8c, 0x87, 0xe9, 0x8c, 0x88, 0xe9, 0x8c, 0x89,
  0xe6, 0xb4, 0xb9, 0xe6, 0xb4, 0xa7, 0xe6, 0xb4, 0x8c, 0xe6, 0xb5, 0x83,
  0xe6, 0xb5, 0x88, 0xe6, 0xb4, 0x87, 0xe6, 0xb4, 0x84, 0xe6, 0xb4, 0x99,
  0xe6, 0xb4, 0x8e, 0xe6, 0xb4, 0xab, 0xe6, 0xb5, 0x8d, 0xe6, 0xb4, 0xae,
  0xe6, 0xb4, 0xb5, 0xe6, 0xb4, 0x9a, 0xe6, 0xb5, 0x8f, 0xe6, 0xb5, 0x92,
  0xe6, 0xb5, 0x94, 0xe6, 0xb4, 0xb3, 0xe6, 0xb6, 0x91, 0xe6, 0xb5, 0xaf,
  0xe6, 0xb6, 0x9e, 0xe6, 0xb6, 0xa0, 0xe6, 0xb5, 0x9e, 0xe6, 0xb6, 0x93,
  0xe6, 0xb6, 0x94, 0xe6, 0xb5, 0x9c, 0xe6, 0xb5, 0xa0, 0xe6, 0xb5, 0xbc,
  0xe6, 0xb5, 0xa3, 0xe6, 0xb8, 0x9a, 0xe6, 0xb7, 0x87, 0xe6, 0xb7, 0x85,
  0xe6, 0xb7, 0x9e, 0xe6, 0xb8, 0x8e, 0xe6, 0xb6, 0xbf, 0xe6, 0xb7, 0xa0,
  0xe6, 0xb8, 0x91, 0xe6, 0xb7, 0xa6, 0xe6, 0xb7, 0x9d, 0xe6, 0xb7, 0x99,
  0xe6, 0xb8, 0x96, 0xe6, 0xb6, 0xab, 0xe6, 0xb8, 0x8c, 0xe6, 0xb6, 0xae,
  0xe6, 0xb8, 0xab, 0xe6, 0xb9, 0xae, 0xe6, 0xb9, 0x8e, 0xe6, 0xb9, 0xab,
  0xe6, 0xba, 0xb2, 0xe6, 0xb9, 0x9f, 0xe6, 0xba, 0x86, 0xe6, 0xb9, 0x93,
  0xe6, 0xb9, 0x94, 0xe6, 0xb8, 0xb2, 0xe6, 0xb8, 0xa5, 0xe6, 0xb9, 0x84,
  0xe6, 0xbb, 0x9f, 0xe6, 0xba, 0xb1, 0xe6, 0xba, 0x98, 0xe6, 0xbb, 0xa0,
  0xe6, 0xbc, 0xad, 0xe6, 0xbb, 0xa2, 0xe6, 0xba, 0xa5, 0xe6, 0xba, 0xa7,
  0xe6, 0xba, 0xbd, 0xe6, 0xba, 0xbb, 0xe6, 0xba, 0xb7, 0xe6, 0xbb, 0x97,
  0xe6, 0xba, 0xb4, 0xe6, 0xbb, 0x8f, 0xe6, 0xba, 0x8f, 0xe6, 0xbb, 0x82,
  0xe6, 0xba, 0x9f, 0xe6, 0xbd, 0xa2, 0xe6, 0xbd, 0x86, 0xe6, 0xbd, 0x87,
  0xe6, 0xbc, 0xa4, 0xe6, 0xbc, 0x95, 0xe6, 0xbb, 0xb9, 0xe6, 0xbc, 0xaf,
  0xe6, 0xbc, 0xb6, 0xe6, 0xbd, 0x8b, 0xe6, 0xbd, 0xb4, 0xe6, 0xbc, 0xaa,
  0xe6, 0xbc, 0x89, 0xe6, 0xbc, 0xa9, 0xe6, 0xbe, 0x89, 0xe6, 0xbe, 0x8d,
  0xe6, 0xbe, 0x8c, 0xe6, 0xbd, 0xb8, 0xe6, 0xbd, 0xb2, 0xe6, 0xbd, 0xbc,
  0xe6, 0xbd, 0xba, 0xe6, 0xbf, 0x91
};

unsigned char gbk_row_0xe5_converted_to_utf8[] = {
  0xe9, 0x8c, 0x8a, 0xe9, 0x8c,
  0x8b, 0xe9, 0x8c, 0x8c, 0xe9, 0x8c, 0x8d, 0xe9, 0x8c, 0x8e, 0xe9, 0x8c,
  0x8f, 0xe9, 0x8c, 0x90, 0xe9, 0x8c, 0x91, 0xe9, 0x8c, 0x92, 0xe9, 0x8c,
  0x93, 0xe9, 0x8c, 0x94, 0xe9, 0x8c, 0x95, 0xe9, 0x8c, 0x96, 0xe9, 0x8c,
  0x97, 0xe9, 0x8c, 0x98, 0xe9, 0x8c, 0x99, 0xe9, 0x8c, 0x9a, 0xe9, 0x8c,
  0x9b, 0xe9, 0x8c, 0x9c, 0xe9, 0x8c, 0x9d, 0xe9, 0x8c, 0x9e, 0xe9, 0x8c,
  0x9f, 0xe9, 0x8c, 0xa0, 0xe9, 0x8c, 0xa1, 0xe9, 0x8c, 0xa2, 0xe9, 0x8c,
  0xa3, 0xe9, 0x8c, 0xa4, 0xe9, 0x8c, 0xa5, 0xe9, 0x8c, 0xa6, 0xe9, 0x8c,
  0xa7, 0xe9, 0x8c, 0xa8, 0xe9, 0x8c, 0xa9, 0xe9, 0x8c, 0xaa, 0xe9, 0x8c,
  0xab, 0xe9, 0x8c, 0xac, 0xe9, 0x8c, 0xad, 0xe9, 0x8c, 0xae, 0xe9, 0x8c,
  0xaf, 0xe9, 0x8c, 0xb0, 0xe9, 0x8c, 0xb1, 0xe9, 0x8c, 0xb2, 0xe9, 0x8c,
  0xb3, 0xe9, 0x8c, 0xb4, 0xe9, 0x8c, 0xb5, 0xe9, 0x8c, 0xb6, 0xe9, 0x8c,
  0xb7, 0xe9, 0x8c, 0xb8, 0xe9, 0x8c, 0xb9, 0xe9, 0x8c, 0xba, 0xe9, 0x8c,
  0xbb, 0xe9, 0x8c, 0xbc, 0xe9, 0x8c, 0xbd, 0xe9, 0x8c, 0xbf, 0xe9, 0x8d,
  0x80, 0xe9, 0x8d, 0x81, 0xe9, 0x8d, 0x82, 0xe9, 0x8d, 0x83, 0xe9, 0x8d,
  0x84, 0xe9, 0x8d, 0x85, 0xe9, 0x8d, 0x86, 0xe9, 0x8d, 0x87, 0xe9, 0x8d,
  0x88, 0xe9, 0x8d, 0x89, 0xe9, 0x8d, 0x8a, 0xe9, 0x8d, 0x8b, 0xe9, 0x8d,
  0x8c, 0xe9, 0x8d, 0x8d, 0xe9, 0x8d, 0x8e, 0xe9, 0x8d, 0x8f, 0xe9, 0x8d,
  0x90, 0xe9, 0x8d, 0x91, 0xe9, 0x8d, 0x92, 0xe9, 0x8d, 0x93, 0xe9, 0x8d,
  0x94, 0xe9, 0x8d, 0x95, 0xe9, 0x8d, 0x96, 0xe9, 0x8d, 0x97, 0xe9, 0x8d,
  0x98, 0xe9, 0x8d, 0x99, 0xe9, 0x8d, 0x9a, 0xe9, 0x8d, 0x9b, 0xe9, 0x8d,
  0x9c, 0xe9, 0x8d, 0x9d, 0xe9, 0x8d, 0x9e, 0xe9, 0x8d, 0x9f, 0xe9, 0x8d,
  0xa0, 0xe9, 0x8d, 0xa1, 0xe9, 0x8d, 0xa2, 0xe9, 0x8d, 0xa3, 0xe9, 0x8d,
  0xa4, 0xe9, 0x8d, 0xa5, 0xe9, 0x8d, 0xa6, 0xe9, 0x8d, 0xa7, 0xe9, 0x8d,
  0xa8, 0xe9, 0x8d, 0xa9, 0xe9, 0x8d, 0xab, 0xe6, 0xbf, 0x89, 0xe6, 0xbe,
  0xa7, 0xe6, 0xbe, 0xb9, 0xe6, 0xbe, 0xb6, 0xe6, 0xbf, 0x82, 0xe6, 0xbf,
  0xa1, 0xe6, 0xbf, 0xae, 0xe6, 0xbf, 0x9e, 0xe6, 0xbf, 0xa0, 0xe6, 0xbf,
  0xaf, 0xe7, 0x80, 0x9a, 0xe7, 0x80, 0xa3, 0xe7, 0x80, 0x9b, 0xe7, 0x80,
  0xb9, 0xe7, 0x80, 0xb5, 0xe7, 0x81, 0x8f, 0xe7, 0x81, 0x9e, 0xe5, 0xae,
  0x80, 0xe5, 0xae, 0x84, 0xe5, 0xae, 0x95, 0xe5, 0xae, 0x93, 0xe5, 0xae,
  0xa5, 0xe5, 0xae, 0xb8, 0xe7, 0x94, 0xaf, 0xe9, 0xaa, 0x9e, 0xe6, 0x90,
  0xb4, 0xe5, 0xaf, 0xa4, 0xe5, 0xaf, 0xae, 0xe8, 0xa4, 0xb0, 0xe5, 0xaf,
  0xb0, 0xe8, 0xb9, 0x87, 0xe8, 0xac, 0x87, 0xe8, 0xbe, 0xb6, 0xe8, 0xbf,
  0x93, 0xe8, 0xbf, 0x95, 0xe8, 0xbf, 0xa5, 0xe8, 0xbf, 0xae, 0xe8, 0xbf,
  0xa4, 0xe8, 0xbf, 0xa9, 0xe8, 0xbf, 0xa6, 0xe8, 0xbf, 0xb3, 0xe8, 0xbf,
  0xa8, 0xe9, 0x80, 0x85, 0xe9, 0x80, 0x84, 0xe9, 0x80, 0x8b, 0xe9, 0x80,
  0xa6, 0xe9, 0x80, 0x91, 0xe9, 0x80, 0x8d, 0xe9, 0x80, 0x96, 0xe9, 0x80,
  0xa1, 0xe9, 0x80, 0xb5, 0xe9, 0x80, 0xb6, 0xe9, 0x80, 0xad, 0xe9, 0x80,
  0xaf, 0xe9, 0x81, 0x84, 0xe9, 0x81, 0x91, 0xe9, 0x81, 0x92, 0xe9, 0x81,
  0x90, 0xe9, 0x81, 0xa8, 0xe9, 0x81, 0x98, 0xe9, 0x81, 0xa2, 0xe9, 0x81,
  0x9b, 0xe6, 0x9a, 0xb9, 0xe9, 0x81, 0xb4, 0xe9, 0x81, 0xbd, 0xe9, 0x82,
  0x82, 0xe9, 0x82, 0x88, 0xe9, 0x82, 0x83, 0xe9, 0x82, 0x8b, 0xe5, 0xbd,
  0x90, 0xe5, 0xbd, 0x97, 0xe5, 0xbd, 0x96, 0xe5, 0xbd, 0x98, 0xe5, 0xb0,
  0xbb, 0xe5, 0x92, 0xab, 0xe5, 0xb1, 0x90, 0xe5, 0xb1, 0x99, 0xe5, 0xad,
  0xb1, 0xe5, 0xb1, 0xa3, 0xe5, 0xb1, 0xa6, 0xe7, 0xbe, 0xbc, 0xe5, 0xbc,
  0xaa, 0xe5, 0xbc, 0xa9, 0xe5, 0xbc, 0xad, 0xe8, 0x89, 0xb4, 0xe5, 0xbc,
  0xbc, 0xe9, 0xac, 0xbb, 0xe5, 0xb1, 0xae, 0xe5, 0xa6, 0x81, 0xe5, 0xa6,
  0x83, 0xe5, 0xa6, 0x8d, 0xe5, 0xa6, 0xa9, 0xe5, 0xa6, 0xaa, 0xe5, 0xa6,
  0xa3
};

unsigned char gbk_row_0xe6_converted_to_utf8[] = {
  0xe9, 0x8d, 0xac, 0xe9, 0x8d, 0xad, 0xe9, 0x8d, 0xae, 0xe9,
  0x8d, 0xaf, 0xe9, 0x8d, 0xb0, 0xe9, 0x8d, 0xb1, 0xe9, 0x8d, 0xb2, 0xe9,
  0x8d, 0xb3, 0xe9, 0x8d, 0xb4, 0xe9, 0x8d, 0xb5, 0xe9, 0x8d, 0xb6, 0xe9,
  0x8d, 0xb7, 0xe9, 0x8d, 0xb8, 0xe9, 0x8d, 0xb9, 0xe9, 0x8d, 0xba, 0xe9,
  0x8d, 0xbb, 0xe9, 0x8d, 0xbc, 0xe9, 0x8d, 0xbd, 0xe9, 0x8d, 0xbe, 0xe9,
  0x8d, 0xbf, 0xe9, 0x8e, 0x80, 0xe9, 0x8e, 0x81, 0xe9, 0x8e, 0x82, 0xe9,
  0x8e, 0x83, 0xe9, 0x8e, 0x84, 0xe9, 0x8e, 0x85, 0xe9, 0x8e, 0x86, 0xe9,
  0x8e, 0x87, 0xe9, 0x8e, 0x88, 0xe9, 0x8e, 0x89, 0xe9, 0x8e, 0x8a, 0xe9,
  0x8e, 0x8b, 0xe9, 0x8e, 0x8c, 0xe9, 0x8e, 0x8d, 0xe9, 0x8e, 0x8e, 0xe9,
  0x8e, 0x90, 0xe9, 0x8e, 0x91, 0xe9, 0x8e, 0x92, 0xe9, 0x8e, 0x93, 0xe9,
  0x8e, 0x94, 0xe9, 0x8e, 0x95, 0xe9, 0x8e, 0x96, 0xe9, 0x8e, 0x97, 0xe9,
  0x8e, 0x98, 0xe9, 0x8e, 0x99, 0xe9, 0x8e, 0x9a, 0xe9, 0x8e, 0x9b, 0xe9,
  0x8e, 0x9c, 0xe9, 0x8e, 0x9d, 0xe9, 0x8e, 0x9e, 0xe9, 0x8e, 0x9f, 0xe9,
  0x8e, 0xa0, 0xe9, 0x8e, 0xa1, 0xe9, 0x8e, 0xa2, 0xe9, 0x8e, 0xa3, 0xe9,
  0x8e, 0xa4, 0xe9, 0x8e, 0xa5, 0xe9, 0x8e, 0xa6, 0xe9, 0x8e, 0xa7, 0xe9,
  0x8e, 0xa8, 0xe9, 0x8e, 0xa9, 0xe9, 0x8e, 0xaa, 0xe9, 0x8e, 0xab, 0xe9,
  0x8e, 0xac, 0xe9, 0x8e, 0xad, 0xe9, 0x8e, 0xae, 0xe9, 0x8e, 0xaf, 0xe9,
  0x8e, 0xb0, 0xe9, 0x8e, 0xb1, 0xe9, 0x8e, 0xb2, 0xe9, 0x8e, 0xb3, 0xe9,
  0x8e, 0xb4, 0xe9, 0x8e, 0xb5, 0xe9, 0x8e, 0xb6, 0xe9, 0x8e, 0xb7, 0xe9,
  0x8e, 0xb8, 0xe9, 0x8e, 0xb9, 0xe9, 0x8e, 0xba, 0xe9, 0x8e, 0xbb, 0xe9,
  0x8e, 0xbc, 0xe9, 0x8e, 0xbd, 0xe9, 0x8e, 0xbe, 0xe9, 0x8e, 0xbf, 0xe9,
  0x8f, 0x80, 0xe9, 0x8f, 0x81, 0xe9, 0x8f, 0x82, 0xe9, 0x8f, 0x83, 0xe9,
  0x8f, 0x84, 0xe9, 0x8f, 0x85, 0xe9, 0x8f, 0x86, 0xe9, 0x8f, 0x87, 0xe9,
  0x8f, 0x88, 0xe9, 0x8f, 0x89, 0xe9, 0x8f, 0x8b, 0xe9, 0x8f, 0x8c, 0xe9,
  0x8f, 0x8d, 0xe5, 0xa6, 0x97, 0xe5, 0xa7, 0x8a, 0xe5, 0xa6, 0xab, 0xe5,
  0xa6, 0x9e, 0xe5, 0xa6, 0xa4, 0xe5, 0xa7, 0x92, 0xe5, 0xa6, 0xb2, 0xe5,
  0xa6, 0xaf, 0xe5, 0xa7, 0x97, 0xe5, 0xa6, 0xbe, 0xe5, 0xa8, 0x85, 0xe5,
  0xa8, 0x86, 0xe5, 0xa7, 0x9d, 0xe5, 0xa8, 0x88, 0xe5, 0xa7, 0xa3, 0xe5,
  0xa7, 0x98, 0xe5, 0xa7, 0xb9, 0xe5, 0xa8, 0x8c, 0xe5, 0xa8, 0x89, 0xe5,
  0xa8, 0xb2, 0xe5, 0xa8, 0xb4, 0xe5, 0xa8, 0x91, 0xe5, 0xa8, 0xa3, 0xe5,
  0xa8, 0x93, 0xe5, 0xa9, 0x80, 0xe5, 0xa9, 0xa7, 0xe5, 0xa9, 0x8a, 0xe5,
  0xa9, 0x95, 0xe5, 0xa8, 0xbc, 0xe5, 0xa9, 0xa2, 0xe5, 0xa9, 0xb5, 0xe8,
  0x83, 0xac, 0xe5, 0xaa, 0xaa, 0xe5, 0xaa, 0x9b, 0xe5, 0xa9, 0xb7, 0xe5,
  0xa9, 0xba, 0xe5, 0xaa, 0xbe, 0xe5, 0xab, 0xab, 0xe5, 0xaa, 0xb2, 0xe5,
  0xab, 0x92, 0xe5, 0xab, 0x94, 0xe5, 0xaa, 0xb8, 0xe5, 0xab, 0xa0, 0xe5,
  0xab, 0xa3, 0xe5, 0xab, 0xb1, 0xe5, 0xab, 0x96, 0xe5, 0xab, 0xa6, 0xe5,
  0xab, 0x98, 0xe5, 0xab, 0x9c, 0xe5, 0xac, 0x89, 0xe5, 0xac, 0x97, 0xe5,
  0xac, 0x96, 0xe5, 0xac, 0xb2, 0xe5, 0xac, 0xb7, 0xe5, 0xad, 0x80, 0xe5,
  0xb0, 0x95, 0xe5, 0xb0, 0x9c, 0xe5, 0xad, 0x9a, 0xe5, 0xad, 0xa5, 0xe5,
  0xad, 0xb3, 0xe5, 0xad, 0x91, 0xe5, 0xad, 0x93, 0xe5, 0xad, 0xa2, 0xe9,
  0xa9, 0xb5, 0xe9, 0xa9, 0xb7, 0xe9, 0xa9, 0xb8, 0xe9, 0xa9, 0xba, 0xe9,
  0xa9, 0xbf, 0xe9, 0xa9, 0xbd, 0xe9, 0xaa, 0x80, 0xe9, 0xaa, 0x81, 0xe9,
  0xaa, 0x85, 0xe9, 0xaa, 0x88, 0xe9, 0xaa, 0x8a, 0xe9, 0xaa, 0x90, 0xe9,
  0xaa, 0x92, 0xe9, 0xaa, 0x93, 0xe9, 0xaa, 0x96, 0xe9, 0xaa, 0x98, 0xe9,
  0xaa, 0x9b, 0xe9, 0xaa, 0x9c, 0xe9, 0xaa, 0x9d, 0xe9, 0xaa, 0x9f, 0xe9,
  0xaa, 0xa0, 0xe9, 0xaa, 0xa2, 0xe9, 0xaa, 0xa3, 0xe9, 0xaa, 0xa5, 0xe9,
  0xaa, 0xa7, 0xe7, 0xba, 0x9f, 0xe7, 0xba, 0xa1, 0xe7, 0xba, 0xa3, 0xe7,
  0xba, 0xa5, 0xe7, 0xba, 0xa8, 0xe7, 0xba, 0xa9
};

unsigned char gbk_row_0xe7_converted_to_utf8[] = {
  0xe9, 0x8f, 0x8e,
  0xe9, 0x8f, 0x8f, 0xe9, 0x8f, 0x90, 0xe9, 0x8f, 0x91, 0xe9, 0x8f, 0x92,
  0xe9, 0x8f, 0x93, 0xe9, 0x8f, 0x94, 0xe9, 0x8f, 0x95, 0xe9, 0x8f, 0x97,
  0xe9, 0x8f, 0x98, 0xe9, 0x8f, 0x99, 0xe9, 0x8f, 0x9a, 0xe9, 0x8f, 0x9b,
  0xe9, 0x8f, 0x9c, 0xe9, 0x8f, 0x9d, 0xe9, 0x8f, 0x9e, 0xe9, 0x8f, 0x9f,
  0xe9, 0x8f, 0xa0, 0xe9, 0x8f, 0xa1, 0xe9, 0x8f, 0xa2, 0xe9, 0x8f, 0xa3,
  0xe9, 0x8f, 0xa4, 0xe9, 0x8f, 0xa5, 0xe9, 0x8f, 0xa6, 0xe9, 0x8f, 0xa7,
  0xe9, 0x8f, 0xa8, 0xe9, 0x8f, 0xa9, 0xe9, 0x8f, 0xaa, 0xe9, 0x8f, 0xab,
  0xe9, 0x8f, 0xac, 0xe9, 0x8f, 0xad, 0xe9, 0x8f, 0xae, 0xe9, 0x8f, 0xaf,
  0xe9, 0x8f, 0xb0, 0xe9, 0x8f, 0xb1, 0xe9, 0x8f, 0xb2, 0xe9, 0x8f, 0xb3,
  0xe9, 0x8f, 0xb4, 0xe9, 0x8f, 0xb5, 0xe9, 0x8f, 0xb6, 0xe9, 0x8f, 0xb7,
  0xe9, 0x8f, 0xb8, 0xe9, 0x8f, 0xb9, 0xe9, 0x8f, 0xba, 0xe9, 0x8f, 0xbb,
  0xe9, 0x8f, 0xbc, 0xe9, 0x8f, 0xbd, 0xe9, 0x8f, 0xbe, 0xe9, 0x8f, 0xbf,
  0xe9, 0x90, 0x80, 0xe9, 0x90, 0x81, 0xe9, 0x90, 0x82, 0xe9, 0x90, 0x83,
  0xe9, 0x90, 0x84, 0xe9, 0x90, 0x85, 0xe9, 0x90, 0x86, 0xe9, 0x90, 0x87,
  0xe9, 0x90, 0x88, 0xe9, 0x90, 0x89, 0xe9, 0x90, 0x8a, 0xe9, 0x90, 0x8b,
  0xe9, 0x90, 0x8c, 0xe9, 0x90, 0x8d, 0xe9, 0x90, 0x8e, 0xe9, 0x90, 0x8f,
  0xe9, 0x90, 0x90, 0xe9, 0x90, 0x91, 0xe9, 0x90, 0x92, 0xe9, 0x90, 0x93,
  0xe9, 0x90, 0x94, 0xe9, 0x90, 0x95, 0xe9, 0x90, 0x96, 0xe9, 0x90, 0x97,
  0xe9, 0x90, 0x98, 0xe9, 0x90, 0x99, 0xe9, 0x90, 0x9a, 0xe9, 0x90, 0x9b,
  0xe9, 0x90, 0x9c, 0xe9, 0x90, 0x9d, 0xe9, 0x90, 0x9e, 0xe9, 0x90, 0x9f,
  0xe9, 0x90, 0xa0, 0xe9, 0x90, 0xa1, 0xe9, 0x90, 0xa2, 0xe9, 0x90, 0xa3,
  0xe9, 0x90, 0xa4, 0xe9, 0x90, 0xa5, 0xe9, 0x90, 0xa6, 0xe9, 0x90, 0xa7,
  0xe9, 0x90, 0xa8, 0xe9, 0x90, 0xa9, 0xe9, 0x90, 0xaa, 0xe9, 0x90, 0xab,
  0xe9, 0x90, 0xac, 0xe9, 0x90, 0xad, 0xe9, 0x90, 0xae, 0xe7, 0xba, 0xad,
  0xe7, 0xba, 0xb0, 0xe7, 0xba, 0xbe, 0xe7, 0xbb, 0x80, 0xe7, 0xbb, 0x81,
  0xe7, 0xbb, 0x82, 0xe7, 0xbb, 0x89, 0xe7, 0xbb, 0x8b, 0xe7, 0xbb, 0x8c,
  0xe7, 0xbb, 0x90, 0xe7, 0xbb, 0x94, 0xe7, 0xbb, 0x97, 0xe7, 0xbb, 0x9b,
  0xe7, 0xbb, 0xa0, 0xe7, 0xbb, 0xa1, 0xe7, 0xbb, 0xa8, 0xe7, 0xbb, 0xab,
  0xe7, 0xbb, 0xae, 0xe7, 0xbb, 0xaf, 0xe7, 0xbb, 0xb1, 0xe7, 0xbb, 0xb2,
  0xe7, 0xbc, 0x8d, 0xe7, 0xbb, 0xb6, 0xe7, 0xbb, 0xba, 0xe7, 0xbb, 0xbb,
  0xe7, 0xbb, 0xbe, 0xe7, 0xbc, 0x81, 0xe7, 0xbc, 0x82, 0xe7, 0xbc, 0x83,
  0xe7, 0xbc, 0x87, 0xe7, 0xbc, 0x88, 0xe7, 0xbc, 0x8b, 0xe7, 0xbc, 0x8c,
  0xe7, 0xbc, 0x8f, 0xe7, 0xbc, 0x91, 0xe7, 0xbc, 0x92, 0xe7, 0xbc, 0x97,
  0xe7, 0xbc, 0x99, 0xe7, 0xbc, 0x9c, 0xe7, 0xbc, 0x9b, 0xe7, 0xbc, 0x9f,
  0xe7, 0xbc, 0xa1, 0xe7, 0xbc, 0xa2, 0xe7, 0xbc, 0xa3, 0xe7, 0xbc, 0xa4,
  0xe7, 0xbc, 0xa5, 0xe7, 0xbc, 0xa6, 0xe7, 0xbc, 0xa7, 0xe7, 0xbc, 0xaa,
  0xe7, 0xbc, 0xab, 0xe7, 0xbc, 0xac, 0xe7, 0xbc, 0xad, 0xe7, 0xbc, 0xaf,
  0xe7, 0xbc, 0xb0, 0xe7, 0xbc, 0xb1, 0xe7, 0xbc, 0xb2, 0xe7, 0xbc, 0xb3,
  0xe7, 0xbc, 0xb5, 0xe5, 0xb9, 0xba, 0xe7, 0x95, 0xbf, 0xe5, 0xb7, 0x9b,
  0xe7, 0x94, 0xbe, 0xe9, 0x82, 0x95, 0xe7, 0x8e, 0x8e, 0xe7, 0x8e, 0x91,
  0xe7, 0x8e, 0xae, 0xe7, 0x8e, 0xa2, 0xe7, 0x8e, 0x9f, 0xe7, 0x8f, 0x8f,
  0xe7, 0x8f, 0x82, 0xe7, 0x8f, 0x91, 0xe7, 0x8e, 0xb7, 0xe7, 0x8e, 0xb3,
  0xe7, 0x8f, 0x80, 0xe7, 0x8f, 0x89, 0xe7, 0x8f, 0x88, 0xe7, 0x8f, 0xa5,
  0xe7, 0x8f, 0x99, 0xe9, 0xa1, 0xbc, 0xe7, 0x90, 0x8a, 0xe7, 0x8f, 0xa9,
  0xe7, 0x8f, 0xa7, 0xe7, 0x8f, 0x9e, 0xe7, 0x8e, 0xba, 0xe7, 0x8f, 0xb2,
  0xe7, 0x90, 0x8f, 0xe7, 0x90, 0xaa, 0xe7, 0x91, 0x9b, 0xe7, 0x90, 0xa6,
  0xe7, 0x90, 0xa5, 0xe7, 0x90, 0xa8, 0xe7, 0x90, 0xb0, 0xe7, 0x90, 0xae,
  0xe7, 0x90, 0xac
};

unsigned char gbk_row_0xe8_converted_to_utf8[] = {
  0xe9, 0x90, 0xaf, 0xe9, 0x90, 0xb0, 0xe9, 0x90,
  0xb1, 0xe9, 0x90, 0xb2, 0xe9, 0x90, 0xb3, 0xe9, 0x90, 0xb4, 0xe9, 0x90,
  0xb5, 0xe9, 0x90, 0xb6, 0xe9, 0x90, 0xb7, 0xe9, 0x90, 0xb8, 0xe9, 0x90,
  0xb9, 0xe9, 0x90, 0xba, 0xe9, 0x90, 0xbb, 0xe9, 0x90, 0xbc, 0xe9, 0x90,
  0xbd, 0xe9, 0x90, 0xbf, 0xe9, 0x91, 0x80, 0xe9, 0x91, 0x81, 0xe9, 0x91,
  0x82, 0xe9, 0x91, 0x83, 0xe9, 0x91, 0x84, 0xe9, 0x91, 0x85, 0xe9, 0x91,
  0x86, 0xe9, 0x91, 0x87, 0xe9, 0x91, 0x88, 0xe9, 0x91, 0x89, 0xe9, 0x91,
  0x8a, 0xe9, 0x91, 0x8b, 0xe9, 0x91, 0x8c, 0xe9, 0x91, 0x8d, 0xe9, 0x91,
  0x8e, 0xe9, 0x91, 0x8f, 0xe9, 0x91, 0x90, 0xe9, 0x91, 0x91, 0xe9, 0x91,
  0x92, 0xe9, 0x91, 0x93, 0xe9, 0x91, 0x94, 0xe9, 0x91, 0x95, 0xe9, 0x91,
  0x96, 0xe9, 0x91, 0x97, 0xe9, 0x91, 0x98, 0xe9, 0x91, 0x99, 0xe9, 0x91,
  0x9a, 0xe9, 0x91, 0x9b, 0xe9, 0x91, 0x9c, 0xe9, 0x91, 0x9d, 0xe9, 0x91,
  0x9e, 0xe9, 0x91, 0x9f, 0xe9, 0x91, 0xa0, 0xe9, 0x91, 0xa1, 0xe9, 0x91,
  0xa2, 0xe9, 0x91, 0xa3, 0xe9, 0x91, 0xa4, 0xe9, 0x91, 0xa5, 0xe9, 0x91,
  0xa6, 0xe9, 0x91, 0xa7, 0xe9, 0x91, 0xa8, 0xe9, 0x91, 0xa9, 0xe9, 0x91,
  0xaa, 0xe9, 0x91, 0xac, 0xe9, 0x91, 0xad, 0xe9, 0x91, 0xae, 0xe9, 0x91,
  0xaf, 0xe9, 0x91, 0xb0, 0xe9, 0x91, 0xb1, 0xe9, 0x91, 0xb2, 0xe9, 0x91,
  0xb3, 0xe9, 0x91, 0xb4, 0xe9, 0x91, 0xb5, 0xe9, 0x91, 0xb6, 0xe9, 0x91,
  0xb7, 0xe9, 0x91, 0xb8, 0xe9, 0x91, 0xb9, 0xe9, 0x91, 0xba, 0xe9, 0x91,
  0xbb, 0xe9, 0x91, 0xbc, 0xe9, 0x91, 0xbd, 0xe9, 0x91, 0xbe, 0xe9, 0x91,
  0xbf, 0xe9, 0x92, 0x80, 0xe9, 0x92, 0x81, 0xe9, 0x92, 0x82, 0xe9, 0x92,
  0x83, 0xe9, 0x92, 0x84, 0xe9, 0x92, 0x91, 0xe9, 0x92, 0x96, 0xe9, 0x92,
  0x98, 0xe9, 0x93, 0x87, 0xe9, 0x93, 0x8f, 0xe9, 0x93, 0x93, 0xe9, 0x93,
  0x94, 0xe9, 0x93, 0x9a, 0xe9, 0x93, 0xa6, 0xe9, 0x93, 0xbb, 0xe9, 0x94,
  0x9c, 0xe9, 0x94, 0xa0, 0xe7, 0x90, 0x9b, 0xe7, 0x90, 0x9a, 0xe7, 0x91,
  0x81, 0xe7, 0x91, 0x9c, 0xe7, 0x91, 0x97, 0xe7, 0x91, 0x95, 0xe7, 0x91,
  0x99, 0xe7, 0x91, 0xb7, 0xe7, 0x91, 0xad, 0xe7, 0x91, 0xbe, 0xe7, 0x92,
  0x9c, 0xe7, 0x92, 0x8e, 0xe7, 0x92, 0x80, 0xe7, 0x92, 0x81, 0xe7, 0x92,
  0x87, 0xe7, 0x92, 0x8b, 0xe7, 0x92, 0x9e, 0xe7, 0x92, 0xa8, 0xe7, 0x92,
  0xa9, 0xe7, 0x92, 0x90, 0xe7, 0x92, 0xa7, 0xe7, 0x93, 0x92, 0xe7, 0x92,
  0xba, 0xe9, 0x9f, 0xaa, 0xe9, 0x9f, 0xab, 0xe9, 0x9f, 0xac, 0xe6, 0x9d,
  0x8c, 0xe6, 0x9d, 0x93, 0xe6, 0x9d, 0x9e, 0xe6, 0x9d, 0x88, 0xe6, 0x9d,
  0xa9, 0xe6, 0x9e, 0xa5, 0xe6, 0x9e, 0x87, 0xe6, 0x9d, 0xaa, 0xe6, 0x9d,
  0xb3, 0xe6, 0x9e, 0x98, 0xe6, 0x9e, 0xa7, 0xe6, 0x9d, 0xb5, 0xe6, 0x9e,
  0xa8, 0xe6, 0x9e, 0x9e, 0xe6, 0x9e, 0xad, 0xe6, 0x9e, 0x8b, 0xe6, 0x9d,
  0xb7, 0xe6, 0x9d, 0xbc, 0xe6, 0x9f, 0xb0, 0xe6, 0xa0, 0x89, 0xe6, 0x9f,
  0x98, 0xe6, 0xa0, 0x8a, 0xe6, 0x9f, 0xa9, 0xe6, 0x9e, 0xb0, 0xe6, 0xa0,
  0x8c, 0xe6, 0x9f, 0x99, 0xe6, 0x9e, 0xb5, 0xe6, 0x9f, 0x9a, 0xe6, 0x9e,
  0xb3, 0xe6, 0x9f, 0x9d, 0xe6, 0xa0, 0x80, 0xe6, 0x9f, 0x83, 0xe6, 0x9e,
  0xb8, 0xe6, 0x9f, 0xa2, 0xe6, 0xa0, 0x8e, 0xe6, 0x9f, 0x81, 0xe6, 0x9f,
  0xbd, 0xe6, 0xa0, 0xb2, 0xe6, 0xa0, 0xb3, 0xe6, 0xa1, 0xa0, 0xe6, 0xa1,
  0xa1, 0xe6, 0xa1, 0x8e, 0xe6, 0xa1, 0xa2, 0xe6, 0xa1, 0x84, 0xe6, 0xa1,
  0xa4, 0xe6, 0xa2, 0x83, 0xe6, 0xa0, 0x9d, 0xe6, 0xa1, 0x95, 0xe6, 0xa1,
  0xa6, 0xe6, 0xa1, 0x81, 0xe6, 0xa1, 0xa7, 0xe6, 0xa1, 0x80, 0xe6, 0xa0,
  0xbe, 0xe6, 0xa1, 0x8a, 0xe6, 0xa1, 0x89, 0xe6, 0xa0, 0xa9, 0xe6, 0xa2,
  0xb5, 0xe6, 0xa2, 0x8f, 0xe6, 0xa1, 0xb4, 0xe6, 0xa1, 0xb7, 0xe6, 0xa2,
  0x93, 0xe6, 0xa1, 0xab, 0xe6, 0xa3, 0x82, 0xe6, 0xa5, 0xae, 0xe6, 0xa3,
  0xbc, 0xe6, 0xa4, 0x9f, 0xe6, 0xa4, 0xa0, 0xe6, 0xa3, 0xb9
};

unsigned char gbk_row_0xe9_converted_to_utf8[] = {
  0xe9,
  0x94, 0xa7, 0xe9, 0x94, 0xb3, 0xe9, 0x94, 0xbd, 0xe9, 0x95, 0x83, 0xe9,
  0x95, 0x88, 0xe9, 0x95, 0x8b, 0xe9, 0x95, 0x95, 0xe9, 0x95, 0x9a, 0xe9,
  0x95, 0xa0, 0xe9, 0x95, 0xae, 0xe9, 0x95, 0xb4, 0xe9, 0x95, 0xb5, 0xe9,
  0x95, 0xb7, 0xe9, 0x95, 0xb8, 0xe9, 0x95, 0xb9, 0xe9, 0x95, 0xba, 0xe9,
  0x95, 0xbb, 0xe9, 0x95, 0xbc, 0xe9, 0x95, 0xbd, 0xe9, 0x95, 0xbe, 0xe9,
  0x96, 0x80, 0xe9, 0x96, 0x81, 0xe9, 0x96, 0x82, 0xe9, 0x96, 0x83, 0xe9,
  0x96, 0x84, 0xe9, 0x96, 0x85, 0xe9, 0x96, 0x86, 0xe9, 0x96, 0x87, 0xe9,
  0x96, 0x88, 0xe9, 0x96, 0x89, 0xe9, 0x96, 0x8a, 0xe9, 0x96, 0x8b, 0xe9,
  0x96, 0x8c, 0xe9, 0x96, 0x8d, 0xe9, 0x96, 0x8e, 0xe9, 0x96, 0x8f, 0xe9,
  0x96, 0x90, 0xe9, 0x96, 0x91, 0xe9, 0x96, 0x92, 0xe9, 0x96, 0x93, 0xe9,
  0x96, 0x94, 0xe9, 0x96, 0x95, 0xe9, 0x96, 0x96, 0xe9, 0x96, 0x97, 0xe9,
  0x96, 0x98, 0xe9, 0x96, 0x99, 0xe9, 0x96, 0x9a, 0xe9, 0x96, 0x9b, 0xe9,
  0x96, 0x9c, 0xe9, 0x96, 0x9d, 0xe9, 0x96, 0x9e, 0xe9, 0x96, 0x9f, 0xe9,
  0x96, 0xa0, 0xe9, 0x96, 0xa1, 0xe9, 0x96, 0xa2, 0xe9, 0x96, 0xa3, 0xe9,
  0x96, 0xa4, 0xe9, 0x96, 0xa5, 0xe9, 0x96, 0xa6, 0xe9, 0x96, 0xa7, 0xe9,
  0x96, 0xa8, 0xe9, 0x96, 0xa9, 0xe9, 0x96, 0xaa, 0xe9, 0x96, 0xab, 0xe9,
  0x96, 0xac, 0xe9, 0x96, 0xad, 0xe9, 0x96, 0xae, 0xe9, 0x96, 0xaf, 0xe9,
  0x96, 0xb0, 0xe9, 0x96, 0xb1, 0xe9, 0x96, 0xb2, 0xe9, 0x96, 0xb3, 0xe9,
  0x96, 0xb4, 0xe9, 0x96, 0xb5, 0xe9, 0x96, 0xb6, 0xe9, 0x96, 0xb7, 0xe9,
  0x96, 0xb8, 0xe9, 0x96, 0xb9, 0xe9, 0x96, 0xba, 0xe9, 0x96, 0xbb, 0xe9,
  0x96, 0xbc, 0xe9, 0x96, 0xbd, 0xe9, 0x96, 0xbe, 0xe9, 0x96, 0xbf, 0xe9,
  0x97, 0x80, 0xe9, 0x97, 0x81, 0xe9, 0x97, 0x82, 0xe9, 0x97, 0x83, 0xe9,
  0x97, 0x84, 0xe9, 0x97, 0x85, 0xe9, 0x97, 0x86, 0xe9, 0x97, 0x87, 0xe9,
  0x97, 0x88, 0xe9, 0x97, 0x89, 0xe9, 0x97, 0x8a, 0xe9, 0x97, 0x8b, 0xe6,
  0xa4, 0xa4, 0xe6, 0xa3, 0xb0, 0xe6, 0xa4, 0x8b, 0xe6, 0xa4, 0x81, 0xe6,
  0xa5, 0x97, 0xe6, 0xa3, 0xa3, 0xe6, 0xa4, 0x90, 0xe6, 0xa5, 0xb1, 0xe6,
  0xa4, 0xb9, 0xe6, 0xa5, 0xa0, 0xe6, 0xa5, 0x82, 0xe6, 0xa5, 0x9d, 0xe6,
  0xa6, 0x84, 0xe6, 0xa5, 0xab, 0xe6, 0xa6, 0x80, 0xe6, 0xa6, 0x98, 0xe6,
  0xa5, 0xb8, 0xe6, 0xa4, 0xb4, 0xe6, 0xa7, 0x8c, 0xe6, 0xa6, 0x87, 0xe6,
  0xa6, 0x88, 0xe6, 0xa7, 0x8e, 0xe6, 0xa6, 0x89, 0xe6, 0xa5, 0xa6, 0xe6,
  0xa5, 0xa3, 0xe6, 0xa5, 0xb9, 0xe6, 0xa6, 0x9b, 0xe6, 0xa6, 0xa7, 0xe6,
  0xa6, 0xbb, 0xe6, 0xa6, 0xab, 0xe6, 0xa6, 0xad, 0xe6, 0xa7, 0x94, 0xe6,
  0xa6, 0xb1, 0xe6, 0xa7, 0x81, 0xe6, 0xa7, 0x8a, 0xe6, 0xa7, 0x9f, 0xe6,
  0xa6, 0x95, 0xe6, 0xa7, 0xa0, 0xe6, 0xa6, 0x8d, 0xe6, 0xa7, 0xbf, 0xe6,
  0xa8, 0xaf, 0xe6, 0xa7, 0xad, 0xe6, 0xa8, 0x97, 0xe6, 0xa8, 0x98, 0xe6,
  0xa9, 0xa5, 0xe6, 0xa7, 0xb2, 0xe6, 0xa9, 0x84, 0xe6, 0xa8, 0xbe, 0xe6,
  0xaa, 0xa0, 0xe6, 0xa9, 0x90, 0xe6, 0xa9, 0x9b, 0xe6, 0xa8, 0xb5, 0xe6,
  0xaa, 0x8e, 0xe6, 0xa9, 0xb9, 0xe6, 0xa8, 0xbd, 0xe6, 0xa8, 0xa8, 0xe6,
  0xa9, 0x98, 0xe6, 0xa9, 0xbc, 0xe6, 0xaa, 0x91, 0xe6, 0xaa, 0x90, 0xe6,
  0xaa, 0xa9, 0xe6, 0xaa, 0x97, 0xe6, 0xaa, 0xab, 0xe7, 0x8c, 0xb7, 0xe7,
  0x8d, 0x92, 0xe6, 0xae, 0x81, 0xe6, 0xae, 0x82, 0xe6, 0xae, 0x87, 0xe6,
  0xae, 0x84, 0xe6, 0xae, 0x92, 0xe6, 0xae, 0x93, 0xe6, 0xae, 0x8d, 0xe6,
  0xae, 0x9a, 0xe6, 0xae, 0x9b, 0xe6, 0xae, 0xa1, 0xe6, 0xae, 0xaa, 0xe8,
  0xbd, 0xab, 0xe8, 0xbd, 0xad, 0xe8, 0xbd, 0xb1, 0xe8, 0xbd, 0xb2, 0xe8,
  0xbd, 0xb3, 0xe8, 0xbd, 0xb5, 0xe8, 0xbd, 0xb6, 0xe8, 0xbd, 0xb8, 0xe8,
  0xbd, 0xb7, 0xe8, 0xbd, 0xb9, 0xe8, 0xbd, 0xba, 0xe8, 0xbd, 0xbc, 0xe8,
  0xbd, 0xbe, 0xe8, 0xbe, 0x81, 0xe8, 0xbe, 0x82, 0xe8, 0xbe, 0x84, 0xe8,
  0xbe, 0x87, 0xe8, 0xbe, 0x8b
};

unsigned char gbk_row_0xea_converted_to_utf8[] = {
  0xe9, 0x97, 0x8c, 0xe9, 0x97, 0x8d,
  0xe9, 0x97, 0x8e, 0xe9, 0x97, 0x8f, 0xe9, 0x97, 0x90, 0xe9, 0x97, 0x91,
  0xe9, 0x97, 0x92, 0xe9, 0x97, 0x93, 0xe9, 0x97, 0x94, 0xe9, 0x97, 0x95,
  0xe9, 0x97, 0x96, 0xe9, 0x97, 0x97, 0xe9, 0x97, 0x98, 0xe9, 0x97, 0x99,
  0xe9, 0x97, 0x9a, 0xe9, 0x97, 0x9b, 0xe9, 0x97, 0x9c, 0xe9, 0x97, 0x9d,
  0xe9, 0x97, 0x9e, 0xe9, 0x97, 0x9f, 0xe9, 0x97, 0xa0, 0xe9, 0x97, 0xa1,
  0xe9, 0x97, 0xa2, 0xe9, 0x97, 0xa3, 0xe9, 0x97, 0xa4, 0xe9, 0x97, 0xa5,
  0xe9, 0x97, 0xa6, 0xe9, 0x97, 0xa7, 0xe9, 0x97, 0xac, 0xe9, 0x97, 0xbf,
  0xe9, 0x98, 0x87, 0xe9, 0x98, 0x93, 0xe9, 0x98, 0x98, 0xe9, 0x98, 0x9b,
  0xe9, 0x98, 0x9e, 0xe9, 0x98, 0xa0, 0xe9, 0x98, 0xa3, 0xe9, 0x98, 0xa4,
  0xe9, 0x98, 0xa5, 0xe9, 0x98, 0xa6, 0xe9, 0x98, 0xa7, 0xe9, 0x98, 0xa8,
  0xe9, 0x98, 0xa9, 0xe9, 0x98, 0xab, 0xe9, 0x98, 0xac, 0xe9, 0x98, 0xad,
  0xe9, 0x98, 0xaf, 0xe9, 0x98, 0xb0, 0xe9, 0x98, 0xb7, 0xe9, 0x98, 0xb8,
  0xe9, 0x98, 0xb9, 0xe9, 0x98, 0xba, 0xe9, 0x98, 0xbe, 0xe9, 0x99, 0x81,
  0xe9, 0x99, 0x83, 0xe9, 0x99, 0x8a, 0xe9, 0x99, 0x8e, 0xe9, 0x99, 0x8f,
  0xe9, 0x99, 0x91, 0xe9, 0x99, 0x92, 0xe9, 0x99, 0x93, 0xe9, 0x99, 0x96,
  0xe9, 0x99, 0x97, 0xe9, 0x99, 0x98, 0xe9, 0x99, 0x99, 0xe9, 0x99, 0x9a,
  0xe9, 0x99, 0x9c, 0xe9, 0x99, 0x9d, 0xe9, 0x99, 0x9e, 0xe9, 0x99, 0xa0,
  0xe9, 0x99, 0xa3, 0xe9, 0x99, 0xa5, 0xe9, 0x99, 0xa6, 0xe9, 0x99, 0xab,
  0xe9, 0x99, 0xad, 0xe9, 0x99, 0xae, 0xe9, 0x99, 0xaf, 0xe9, 0x99, 0xb0,
  0xe9, 0x99, 0xb1, 0xe9, 0x99, 0xb3, 0xe9, 0x99, 0xb8, 0xe9, 0x99, 0xb9,
  0xe9, 0x99, 0xba, 0xe9, 0x99, 0xbb, 0xe9, 0x99, 0xbc, 0xe9, 0x99, 0xbd,
  0xe9, 0x99, 0xbe, 0xe9, 0x99, 0xbf, 0xe9, 0x9a, 0x80, 0xe9, 0x9a, 0x81,
  0xe9, 0x9a, 0x82, 0xe9, 0x9a, 0x83, 0xe9, 0x9a, 0x84, 0xe9, 0x9a, 0x87,
  0xe9, 0x9a, 0x89, 0xe9, 0x9a, 0x8a, 0xe8, 0xbe, 0x8d, 0xe8, 0xbe, 0x8e,
  0xe8, 0xbe, 0x8f, 0xe8, 0xbe, 0x98, 0xe8, 0xbe, 0x9a, 0xe8, 0xbb, 0x8e,
  0xe6, 0x88, 0x8b, 0xe6, 0x88, 0x97, 0xe6, 0x88, 0x9b, 0xe6, 0x88, 0x9f,
  0xe6, 0x88, 0xa2, 0xe6, 0x88, 0xa1, 0xe6, 0x88, 0xa5, 0xe6, 0x88, 0xa4,
  0xe6, 0x88, 0xac, 0xe8, 0x87, 0xa7, 0xe7, 0x93, 0xaf, 0xe7, 0x93, 0xb4,
  0xe7, 0x93, 0xbf, 0xe7, 0x94, 0x8f, 0xe7, 0x94, 0x91, 0xe7, 0x94, 0x93,
  0xe6, 0x94, 0xb4, 0xe6, 0x97, 0xae, 0xe6, 0x97, 0xaf, 0xe6, 0x97, 0xb0,
  0xe6, 0x98, 0x8a, 0xe6, 0x98, 0x99, 0xe6, 0x9d, 0xb2, 0xe6, 0x98, 0x83,
  0xe6, 0x98, 0x95, 0xe6, 0x98, 0x80, 0xe7, 0x82, 0x85, 0xe6, 0x9b, 0xb7,
  0xe6, 0x98, 0x9d, 0xe6, 0x98, 0xb4, 0xe6, 0x98, 0xb1, 0xe6, 0x98, 0xb6,
  0xe6, 0x98, 0xb5, 0xe8, 0x80, 0x86, 0xe6, 0x99, 0x9f, 0xe6, 0x99, 0x94,
  0xe6, 0x99, 0x81, 0xe6, 0x99, 0x8f, 0xe6, 0x99, 0x96, 0xe6, 0x99, 0xa1,
  0xe6, 0x99, 0x97, 0xe6, 0x99, 0xb7, 0xe6, 0x9a, 0x84, 0xe6, 0x9a, 0x8c,
  0xe6, 0x9a, 0xa7, 0xe6, 0x9a, 0x9d, 0xe6, 0x9a, 0xbe, 0xe6, 0x9b, 0x9b,
  0xe6, 0x9b, 0x9c, 0xe6, 0x9b, 0xa6, 0xe6, 0x9b, 0xa9, 0xe8, 0xb4, 0xb2,
  0xe8, 0xb4, 0xb3, 0xe8, 0xb4, 0xb6, 0xe8, 0xb4, 0xbb, 0xe8, 0xb4, 0xbd,
  0xe8, 0xb5, 0x80, 0xe8, 0xb5, 0x85, 0xe8, 0xb5, 0x86, 0xe8, 0xb5, 0x88,
  0xe8, 0xb5, 0x89, 0xe8, 0xb5, 0x87, 0xe8, 0xb5, 0x8d, 0xe8, 0xb5, 0x95,
  0xe8, 0xb5, 0x99, 0xe8, 0xa7, 0x87, 0xe8, 0xa7, 0x8a, 0xe8, 0xa7, 0x8b,
  0xe8, 0xa7, 0x8c, 0xe8, 0xa7, 0x8e, 0xe8, 0xa7, 0x8f, 0xe8, 0xa7, 0x90,
  0xe8, 0xa7, 0x91, 0xe7, 0x89, 0xae, 0xe7, 0x8a, 0x9f, 0xe7, 0x89, 0x9d,
  0xe7, 0x89, 0xa6, 0xe7, 0x89, 0xaf, 0xe7, 0x89, 0xbe, 0xe7, 0x89, 0xbf,
  0xe7, 0x8a, 0x84, 0xe7, 0x8a, 0x8b, 0xe7, 0x8a, 0x8d, 0xe7, 0x8a, 0x8f,
  0xe7, 0x8a, 0x92, 0xe6, 0x8c, 0x88, 0xe6, 0x8c, 0xb2, 0xe6, 0x8e, 0xb0,
};

unsigned char gbk_row_0xeb_converted_to_utf8[] = {
        0xe9, 0x9a, 0x8c, 0xe9, 0x9a, 0x8e, 0xe9, 0x9a, 0x91, 0xe9, 0x9a,
  0x92, 0xe9, 0x9a, 0x93, 0xe9, 0x9a, 0x95, 0xe9, 0x9a, 0x96, 0xe9, 0x9a,
  0x9a, 0xe9, 0x9a, 0x9b, 0xe9, 0x9a, 0x9d, 0xe9, 0x9a, 0x9e, 0xe9, 0x9a,
  0x9f, 0xe9, 0x9a, 0xa0, 0xe9, 0x9a, 0xa1, 0xe9, 0x9a, 0xa2, 0xe9, 0x9a,
  0xa3, 0xe9, 0x9a, 0xa4, 0xe9, 0x9a, 0xa5, 0xe9, 0x9a, 0xa6, 0xe9, 0x9a,
  0xa8, 0xe9, 0x9a, 0xa9, 0xe9, 0x9a, 0xaa, 0xe9, 0x9a, 0xab, 0xe9, 0x9a,
  0xac, 0xe9, 0x9a, 0xad, 0xe9, 0x9a, 0xae, 0xe9, 0x9a, 0xaf, 0xe9, 0x9a,
  0xb1, 0xe9, 0x9a, 0xb2, 0xe9, 0x9a, 0xb4, 0xe9, 0x9a, 0xb5, 0xe9, 0x9a,
  0xb7, 0xe9, 0x9a, 0xb8, 0xe9, 0x9a, 0xba, 0xe9, 0x9a, 0xbb, 0xe9, 0x9a,
  0xbf, 0xe9, 0x9b, 0x82, 0xe9, 0x9b, 0x83, 0xe9, 0x9b, 0x88, 0xe9, 0x9b,
  0x8a, 0xe9, 0x9b, 0x8b, 0xe9, 0x9b, 0x90, 0xe9, 0x9b, 0x91, 0xe9, 0x9b,
  0x93, 0xe9, 0x9b, 0x94, 0xe9, 0x9b, 0x96, 0xe9, 0x9b, 0x97, 0xe9, 0x9b,
  0x98, 0xe9, 0x9b, 0x99, 0xe9, 0x9b, 0x9a, 0xe9, 0x9b, 0x9b, 0xe9, 0x9b,
  0x9c, 0xe9, 0x9b, 0x9d, 0xe9, 0x9b, 0x9e, 0xe9, 0x9b, 0x9f, 0xe9, 0x9b,
  0xa1, 0xe9, 0x9b, 0xa2, 0xe9, 0x9b, 0xa3, 0xe9, 0x9b, 0xa4, 0xe9, 0x9b,
  0xa5, 0xe9, 0x9b, 0xa6, 0xe9, 0x9b, 0xa7, 0xe9, 0x9b, 0xab, 0xe9, 0x9b,
  0xac, 0xe9, 0x9b, 0xad, 0xe9, 0x9b, 0xae, 0xe9, 0x9b, 0xb0, 0xe9, 0x9b,
  0xb1, 0xe9, 0x9b, 0xb2, 0xe9, 0x9b, 0xb4, 0xe9, 0x9b, 0xb5, 0xe9, 0x9b,
  0xb8, 0xe9, 0x9b, 0xba, 0xe9, 0x9b, 0xbb, 0xe9, 0x9b, 0xbc, 0xe9, 0x9b,
  0xbd, 0xe9, 0x9b, 0xbf, 0xe9, 0x9c, 0x82, 0xe9, 0x9c, 0x83, 0xe9, 0x9c,
  0x85, 0xe9, 0x9c, 0x8a, 0xe9, 0x9c, 0x8b, 0xe9, 0x9c, 0x8c, 0xe9, 0x9c,
  0x90, 0xe9, 0x9c, 0x91, 0xe9, 0x9c, 0x92, 0xe9, 0x9c, 0x94, 0xe9, 0x9c,
  0x95, 0xe9, 0x9c, 0x97, 0xe9, 0x9c, 0x98, 0xe9, 0x9c, 0x99, 0xe9, 0x9c,
  0x9a, 0xe9, 0x9c, 0x9b, 0xe9, 0x9c, 0x9d, 0xe9, 0x9c, 0x9f, 0xe9, 0x9c,
  0xa0, 0xe6, 0x90, 0xbf, 0xe6, 0x93, 0x98, 0xe8, 0x80, 0x84, 0xe6, 0xaf,
  0xaa, 0xe6, 0xaf, 0xb3, 0xe6, 0xaf, 0xbd, 0xe6, 0xaf, 0xb5, 0xe6, 0xaf,
  0xb9, 0xe6, 0xb0, 0x85, 0xe6, 0xb0, 0x87, 0xe6, 0xb0, 0x86, 0xe6, 0xb0,
  0x8d, 0xe6, 0xb0, 0x95, 0xe6, 0xb0, 0x98, 0xe6, 0xb0, 0x99, 0xe6, 0xb0,
  0x9a, 0xe6, 0xb0, 0xa1, 0xe6, 0xb0, 0xa9, 0xe6, 0xb0, 0xa4, 0xe6, 0xb0,
  0xaa, 0xe6, 0xb0, 0xb2, 0xe6, 0x94, 0xb5, 0xe6, 0x95, 0x95, 0xe6, 0x95,
  0xab, 0xe7, 0x89, 0x8d, 0xe7, 0x89, 0x92, 0xe7, 0x89, 0x96, 0xe7, 0x88,
  0xb0, 0xe8, 0x99, 0xa2, 0xe5, 0x88, 0x96, 0xe8, 0x82, 0x9f, 0xe8, 0x82,
  0x9c, 0xe8, 0x82, 0x93, 0xe8, 0x82, 0xbc, 0xe6, 0x9c, 0x8a, 0xe8, 0x82,
  0xbd, 0xe8, 0x82, 0xb1, 0xe8, 0x82, 0xab, 0xe8, 0x82, 0xad, 0xe8, 0x82,
  0xb4, 0xe8, 0x82, 0xb7, 0xe8, 0x83, 0xa7, 0xe8, 0x83, 0xa8, 0xe8, 0x83,
  0xa9, 0xe8, 0x83, 0xaa, 0xe8, 0x83, 0x9b, 0xe8, 0x83, 0x82, 0xe8, 0x83,
  0x84, 0xe8, 0x83, 0x99, 0xe8, 0x83, 0x8d, 0xe8, 0x83, 0x97, 0xe6, 0x9c,
  0x90, 0xe8, 0x83, 0x9d, 0xe8, 0x83, 0xab, 0xe8, 0x83, 0xb1, 0xe8, 0x83,
  0xb4, 0xe8, 0x83, 0xad, 0xe8, 0x84, 0x8d, 0xe8, 0x84, 0x8e, 0xe8, 0x83,
  0xb2, 0xe8, 0x83, 0xbc, 0xe6, 0x9c, 0x95, 0xe8, 0x84, 0x92, 0xe8, 0xb1,
  0x9a, 0xe8, 0x84, 0xb6, 0xe8, 0x84, 0x9e, 0xe8, 0x84, 0xac, 0xe8, 0x84,
  0x98, 0xe8, 0x84, 0xb2, 0xe8, 0x85, 0x88, 0xe8, 0x85, 0x8c, 0xe8, 0x85,
  0x93, 0xe8, 0x85, 0xb4, 0xe8, 0x85, 0x99, 0xe8, 0x85, 0x9a, 0xe8, 0x85,
  0xb1, 0xe8, 0x85, 0xa0, 0xe8, 0x85, 0xa9, 0xe8, 0x85, 0xbc, 0xe8, 0x85,
  0xbd, 0xe8, 0x85, 0xad, 0xe8, 0x85, 0xa7, 0xe5, 0xa1, 0x8d, 0xe5, 0xaa,
  0xb5, 0xe8, 0x86, 0x88, 0xe8, 0x86, 0x82, 0xe8, 0x86, 0x91, 0xe6, 0xbb,
  0x95, 0xe8, 0x86, 0xa3, 0xe8, 0x86, 0xaa, 0xe8, 0x87, 0x8c, 0xe6, 0x9c,
  0xa6, 0xe8, 0x87, 0x8a, 0xe8, 0x86, 0xbb
};

unsigned char gbk_row_0xec_converted_to_utf8[] = {
  0xe9, 0x9c, 0xa1, 0xe9,
  0x9c, 0xa2, 0xe9, 0x9c, 0xa3, 0xe9, 0x9c, 0xa4, 0xe9, 0x9c, 0xa5, 0xe9,
  0x9c, 0xa6, 0xe9, 0x9c, 0xa7, 0xe9, 0x9c, 0xa8, 0xe9, 0x9c, 0xa9, 0xe9,
  0x9c, 0xab, 0xe9, 0x9c, 0xac, 0xe9, 0x9c, 0xae, 0xe9, 0x9c, 0xaf, 0xe9,
  0x9c, 0xb1, 0xe9, 0x9c, 0xb3, 0xe9, 0x9c, 0xb4, 0xe9, 0x9c, 0xb5, 0xe9,
  0x9c, 0xb6, 0xe9, 0x9c, 0xb7, 0xe9, 0x9c, 0xba, 0xe9, 0x9c, 0xbb, 0xe9,
  0x9c, 0xbc, 0xe9, 0x9c, 0xbd, 0xe9, 0x9c, 0xbf, 0xe9, 0x9d, 0x80, 0xe9,
  0x9d, 0x81, 0xe9, 0x9d, 0x82, 0xe9, 0x9d, 0x83, 0xe9, 0x9d, 0x84, 0xe9,
  0x9d, 0x85, 0xe9, 0x9d, 0x86, 0xe9, 0x9d, 0x87, 0xe9, 0x9d, 0x88, 0xe9,
  0x9d, 0x89, 0xe9, 0x9d, 0x8a, 0xe9, 0x9d, 0x8b, 0xe9, 0x9d, 0x8c, 0xe9,
  0x9d, 0x8d, 0xe9, 0x9d, 0x8e, 0xe9, 0x9d, 0x8f, 0xe9, 0x9d, 0x90, 0xe9,
  0x9d, 0x91, 0xe9, 0x9d, 0x94, 0xe9, 0x9d, 0x95, 0xe9, 0x9d, 0x97, 0xe9,
  0x9d, 0x98, 0xe9, 0x9d, 0x9a, 0xe9, 0x9d, 0x9c, 0xe9, 0x9d, 0x9d, 0xe9,
  0x9d, 0x9f, 0xe9, 0x9d, 0xa3, 0xe9, 0x9d, 0xa4, 0xe9, 0x9d, 0xa6, 0xe9,
  0x9d, 0xa7, 0xe9, 0x9d, 0xa8, 0xe9, 0x9d, 0xaa, 0xe9, 0x9d, 0xab, 0xe9,
  0x9d, 0xac, 0xe9, 0x9d, 0xad, 0xe9, 0x9d, 0xae, 0xe9, 0x9d, 0xaf, 0xe9,
  0x9d, 0xb0, 0xe9, 0x9d, 0xb1, 0xe9, 0x9d, 0xb2, 0xe9, 0x9d, 0xb5, 0xe9,
  0x9d, 0xb7, 0xe9, 0x9d, 0xb8, 0xe9, 0x9d, 0xb9, 0xe9, 0x9d, 0xba, 0xe9,
  0x9d, 0xbb, 0xe9, 0x9d, 0xbd, 0xe9, 0x9d, 0xbe, 0xe9, 0x9d, 0xbf, 0xe9,
  0x9e, 0x80, 0xe9, 0x9e, 0x81, 0xe9, 0x9e, 0x82, 0xe9, 0x9e, 0x83, 0xe9,
  0x9e, 0x84, 0xe9, 0x9e, 0x86, 0xe9, 0x9e, 0x87, 0xe9, 0x9e, 0x88, 0xe9,
  0x9e, 0x89, 0xe9, 0x9e, 0x8a, 0xe9, 0x9e, 0x8c, 0xe9, 0x9e, 0x8e, 0xe9,
  0x9e, 0x8f, 0xe9, 0x9e, 0x90, 0xe9, 0x9e, 0x93, 0xe9, 0x9e, 0x95, 0xe9,
  0x9e, 0x96, 0xe9, 0x9e, 0x97, 0xe9, 0x9e, 0x99, 0xe9, 0x9e, 0x9a, 0xe9,
  0x9e, 0x9b, 0xe9, 0x9e, 0x9c, 0xe9, 0x9e, 0x9d, 0xe8, 0x87, 0x81, 0xe8,
  0x86, 0xa6, 0xe6, 0xac, 0xa4, 0xe6, 0xac, 0xb7, 0xe6, 0xac, 0xb9, 0xe6,
  0xad, 0x83, 0xe6, 0xad, 0x86, 0xe6, 0xad, 0x99, 0xe9, 0xa3, 0x91, 0xe9,
  0xa3, 0x92, 0xe9, 0xa3, 0x93, 0xe9, 0xa3, 0x95, 0xe9, 0xa3, 0x99, 0xe9,
  0xa3, 0x9a, 0xe6, 0xae, 0xb3, 0xe5, 0xbd, 0x80, 0xe6, 0xaf, 0x82, 0xe8,
  0xa7, 0xb3, 0xe6, 0x96, 0x90, 0xe9, 0xbd, 0x91, 0xe6, 0x96, 0x93, 0xe6,
  0x96, 0xbc, 0xe6, 0x97, 0x86, 0xe6, 0x97, 0x84, 0xe6, 0x97, 0x83, 0xe6,
  0x97, 0x8c, 0xe6, 0x97, 0x8e, 0xe6, 0x97, 0x92, 0xe6, 0x97, 0x96, 0xe7,
  0x82, 0x80, 0xe7, 0x82, 0x9c, 0xe7, 0x82, 0x96, 0xe7, 0x82, 0x9d, 0xe7,
  0x82, 0xbb, 0xe7, 0x83, 0x80, 0xe7, 0x82, 0xb7, 0xe7, 0x82, 0xab, 0xe7,
  0x82, 0xb1, 0xe7, 0x83, 0xa8, 0xe7, 0x83, 0x8a, 0xe7, 0x84, 0x90, 0xe7,
  0x84, 0x93, 0xe7, 0x84, 0x96, 0xe7, 0x84, 0xaf, 0xe7, 0x84, 0xb1, 0xe7,
  0x85, 0xb3, 0xe7, 0x85, 0x9c, 0xe7, 0x85, 0xa8, 0xe7, 0x85, 0x85, 0xe7,
  0x85, 0xb2, 0xe7, 0x85, 0x8a, 0xe7, 0x85, 0xb8, 0xe7, 0x85, 0xba, 0xe7,
  0x86, 0x98, 0xe7, 0x86, 0xb3, 0xe7, 0x86, 0xb5, 0xe7, 0x86, 0xa8, 0xe7,
  0x86, 0xa0, 0xe7, 0x87, 0xa0, 0xe7, 0x87, 0x94, 0xe7, 0x87, 0xa7, 0xe7,
  0x87, 0xb9, 0xe7, 0x88, 0x9d, 0xe7, 0x88, 0xa8, 0xe7, 0x81, 0xac, 0xe7,
  0x84, 0x98, 0xe7, 0x85, 0xa6, 0xe7, 0x86, 0xb9, 0xe6, 0x88, 0xbe, 0xe6,
  0x88, 0xbd, 0xe6, 0x89, 0x83, 0xe6, 0x89, 0x88, 0xe6, 0x89, 0x89, 0xe7,
  0xa4, 0xbb, 0xe7, 0xa5, 0x80, 0xe7, 0xa5, 0x86, 0xe7, 0xa5, 0x89, 0xe7,
  0xa5, 0x9b, 0xe7, 0xa5, 0x9c, 0xe7, 0xa5, 0x93, 0xe7, 0xa5, 0x9a, 0xe7,
  0xa5, 0xa2, 0xe7, 0xa5, 0x97, 0xe7, 0xa5, 0xa0, 0xe7, 0xa5, 0xaf, 0xe7,
  0xa5, 0xa7, 0xe7, 0xa5, 0xba, 0xe7, 0xa6, 0x85, 0xe7, 0xa6, 0x8a, 0xe7,
  0xa6, 0x9a, 0xe7, 0xa6, 0xa7, 0xe7, 0xa6, 0xb3, 0xe5, 0xbf, 0x91, 0xe5,
  0xbf, 0x90
};

unsigned char gbk_row_0xed_converted_to_utf8[] = {
  0xe9, 0x9e, 0x9e, 0xe9, 0x9e, 0x9f, 0xe9, 0x9e, 0xa1,
  0xe9, 0x9e, 0xa2, 0xe9, 0x9e, 0xa4, 0xe9, 0x9e, 0xa5, 0xe9, 0x9e, 0xa6,
  0xe9, 0x9e, 0xa7, 0xe9, 0x9e, 0xa8, 0xe9, 0x9e, 0xa9, 0xe9, 0x9e, 0xaa,
  0xe9, 0x9e, 0xac, 0xe9, 0x9e, 0xae, 0xe9, 0x9e, 0xb0, 0xe9, 0x9e, 0xb1,
  0xe9, 0x9e, 0xb3, 0xe9, 0x9e, 0xb5, 0xe9, 0x9e, 0xb6, 0xe9, 0x9e, 0xb7,
  0xe9, 0x9e, 0xb8, 0xe9, 0x9e, 0xb9, 0xe9, 0x9e, 0xba, 0xe9, 0x9e, 0xbb,
  0xe9, 0x9e, 0xbc, 0xe9, 0x9e, 0xbd, 0xe9, 0x9e, 0xbe, 0xe9, 0x9e, 0xbf,
  0xe9, 0x9f, 0x80, 0xe9, 0x9f, 0x81, 0xe9, 0x9f, 0x82, 0xe9, 0x9f, 0x83,
  0xe9, 0x9f, 0x84, 0xe9, 0x9f, 0x85, 0xe9, 0x9f, 0x86, 0xe9, 0x9f, 0x87,
  0xe9, 0x9f, 0x88, 0xe9, 0x9f, 0x89, 0xe9, 0x9f, 0x8a, 0xe9, 0x9f, 0x8b,
  0xe9, 0x9f, 0x8c, 0xe9, 0x9f, 0x8d, 0xe9, 0x9f, 0x8e, 0xe9, 0x9f, 0x8f,
  0xe9, 0x9f, 0x90, 0xe9, 0x9f, 0x91, 0xe9, 0x9f, 0x92, 0xe9, 0x9f, 0x93,
  0xe9, 0x9f, 0x94, 0xe9, 0x9f, 0x95, 0xe9, 0x9f, 0x96, 0xe9, 0x9f, 0x97,
  0xe9, 0x9f, 0x98, 0xe9, 0x9f, 0x99, 0xe9, 0x9f, 0x9a, 0xe9, 0x9f, 0x9b,
  0xe9, 0x9f, 0x9c, 0xe9, 0x9f, 0x9d, 0xe9, 0x9f, 0x9e, 0xe9, 0x9f, 0x9f,
  0xe9, 0x9f, 0xa0, 0xe9, 0x9f, 0xa1, 0xe9, 0x9f, 0xa2, 0xe9, 0x9f, 0xa3,
  0xe9, 0x9f, 0xa4, 0xe9, 0x9f, 0xa5, 0xe9, 0x9f, 0xa8, 0xe9, 0x9f, 0xae,
  0xe9, 0x9f, 0xaf, 0xe9, 0x9f, 0xb0, 0xe9, 0x9f, 0xb1, 0xe9, 0x9f, 0xb2,
  0xe9, 0x9f, 0xb4, 0xe9, 0x9f, 0xb7, 0xe9, 0x9f, 0xb8, 0xe9, 0x9f, 0xb9,
  0xe9, 0x9f, 0xba, 0xe9, 0x9f, 0xbb, 0xe9, 0x9f, 0xbc, 0xe9, 0x9f, 0xbd,
  0xe9, 0x9f, 0xbe, 0xe9, 0x9f, 0xbf, 0xe9, 0xa0, 0x80, 0xe9, 0xa0, 0x81,
  0xe9, 0xa0, 0x82, 0xe9, 0xa0, 0x83, 0xe9, 0xa0, 0x84, 0xe9, 0xa0, 0x85,
  0xe9, 0xa0, 0x86, 0xe9, 0xa0, 0x87, 0xe9, 0xa0, 0x88, 0xe9, 0xa0, 0x89,
  0xe9, 0xa0, 0x8a, 0xe9, 0xa0, 0x8b, 0xe9, 0xa0, 0x8c, 0xe9, 0xa0, 0x8d,
  0xe9, 0xa0, 0x8e, 0xe6, 0x80, 0xbc, 0xe6, 0x81, 0x9d, 0xe6, 0x81, 0x9a,
  0xe6, 0x81, 0xa7, 0xe6, 0x81, 0x81, 0xe6, 0x81, 0x99, 0xe6, 0x81, 0xa3,
  0xe6, 0x82, 0xab, 0xe6, 0x84, 0x86, 0xe6, 0x84, 0x8d, 0xe6, 0x85, 0x9d,
  0xe6, 0x86, 0xa9, 0xe6, 0x86, 0x9d, 0xe6, 0x87, 0x8b, 0xe6, 0x87, 0x91,
  0xe6, 0x88, 0x86, 0xe8, 0x82, 0x80, 0xe8, 0x81, 0xbf, 0xe6, 0xb2, 0x93,
  0xe6, 0xb3, 0xb6, 0xe6, 0xb7, 0xbc, 0xe7, 0x9f, 0xb6, 0xe7, 0x9f, 0xb8,
  0xe7, 0xa0, 0x80, 0xe7, 0xa0, 0x89, 0xe7, 0xa0, 0x97, 0xe7, 0xa0, 0x98,
  0xe7, 0xa0, 0x91, 0xe6, 0x96, 0xab, 0xe7, 0xa0, 0xad, 0xe7, 0xa0, 0x9c,
  0xe7, 0xa0, 0x9d, 0xe7, 0xa0, 0xb9, 0xe7, 0xa0, 0xba, 0xe7, 0xa0, 0xbb,
  0xe7, 0xa0, 0x9f, 0xe7, 0xa0, 0xbc, 0xe7, 0xa0, 0xa5, 0xe7, 0xa0, 0xac,
  0xe7, 0xa0, 0xa3, 0xe7, 0xa0, 0xa9, 0xe7, 0xa1, 0x8e, 0xe7, 0xa1, 0xad,
  0xe7, 0xa1, 0x96, 0xe7, 0xa1, 0x97, 0xe7, 0xa0, 0xa6, 0xe7, 0xa1, 0x90,
  0xe7, 0xa1, 0x87, 0xe7, 0xa1, 0x8c, 0xe7, 0xa1, 0xaa, 0xe7, 0xa2, 0x9b,
  0xe7, 0xa2, 0x93, 0xe7, 0xa2, 0x9a, 0xe7, 0xa2, 0x87, 0xe7, 0xa2, 0x9c,
  0xe7, 0xa2, 0xa1, 0xe7, 0xa2, 0xa3, 0xe7, 0xa2, 0xb2, 0xe7, 0xa2, 0xb9,
  0xe7, 0xa2, 0xa5, 0xe7, 0xa3, 0x94, 0xe7, 0xa3, 0x99, 0xe7, 0xa3, 0x89,
  0xe7, 0xa3, 0xac, 0xe7, 0xa3, 0xb2, 0xe7, 0xa4, 0x85, 0xe7, 0xa3, 0xb4,
  0xe7, 0xa4, 0x93, 0xe7, 0xa4, 0xa4, 0xe7, 0xa4, 0x9e, 0xe7, 0xa4, 0xb4,
  0xe9, 0xbe, 0x9b, 0xe9, 0xbb, 0xb9, 0xe9, 0xbb, 0xbb, 0xe9, 0xbb, 0xbc,
  0xe7, 0x9b, 0xb1, 0xe7, 0x9c, 0x84, 0xe7, 0x9c, 0x8d, 0xe7, 0x9b, 0xb9,
  0xe7, 0x9c, 0x87, 0xe7, 0x9c, 0x88, 0xe7, 0x9c, 0x9a, 0xe7, 0x9c, 0xa2,
  0xe7, 0x9c, 0x99, 0xe7, 0x9c, 0xad, 0xe7, 0x9c, 0xa6, 0xe7, 0x9c, 0xb5,
  0xe7, 0x9c, 0xb8, 0xe7, 0x9d, 0x90, 0xe7, 0x9d, 0x91, 0xe7, 0x9d, 0x87,
  0xe7, 0x9d, 0x83, 0xe7, 0x9d, 0x9a, 0xe7, 0x9d, 0xa8
};

unsigned char gbk_row_0xee_converted_to_utf8[] = {
  0xe9, 0xa0,
  0x8f, 0xe9, 0xa0, 0x90, 0xe9, 0xa0, 0x91, 0xe9, 0xa0, 0x92, 0xe9, 0xa0,
  0x93, 0xe9, 0xa0, 0x94, 0xe9, 0xa0, 0x95, 0xe9, 0xa0, 0x96, 0xe9, 0xa0,
  0x97, 0xe9, 0xa0, 0x98, 0xe9, 0xa0, 0x99, 0xe9, 0xa0, 0x9a, 0xe9, 0xa0,
  0x9b, 0xe9, 0xa0, 0x9c, 0xe9, 0xa0, 0x9d, 0xe9, 0xa0, 0x9e, 0xe9, 0xa0,
  0x9f, 0xe9, 0xa0, 0xa0, 0xe9, 0xa0, 0xa1, 0xe9, 0xa0, 0xa2, 0xe9, 0xa0,
  0xa3, 0xe9, 0xa0, 0xa4, 0xe9, 0xa0, 0xa5, 0xe9, 0xa0, 0xa6, 0xe9, 0xa0,
  0xa7, 0xe9, 0xa0, 0xa8, 0xe9, 0xa0, 0xa9, 0xe9, 0xa0, 0xaa, 0xe9, 0xa0,
  0xab, 0xe9, 0xa0, 0xac, 0xe9, 0xa0, 0xad, 0xe9, 0xa0, 0xae, 0xe9, 0xa0,
  0xaf, 0xe9, 0xa0, 0xb0, 0xe9, 0xa0, 0xb1, 0xe9, 0xa0, 0xb2, 0xe9, 0xa0,
  0xb3, 0xe9, 0xa0, 0xb4, 0xe9, 0xa0, 0xb5, 0xe9, 0xa0, 0xb6, 0xe9, 0xa0,
  0xb7, 0xe9, 0xa0, 0xb8, 0xe9, 0xa0, 0xb9, 0xe9, 0xa0, 0xba, 0xe9, 0xa0,
  0xbb, 0xe9, 0xa0, 0xbc, 0xe9, 0xa0, 0xbd, 0xe9, 0xa0, 0xbe, 0xe9, 0xa0,
  0xbf, 0xe9, 0xa1, 0x80, 0xe9, 0xa1, 0x81, 0xe9, 0xa1, 0x82, 0xe9, 0xa1,
  0x83, 0xe9, 0xa1, 0x84, 0xe9, 0xa1, 0x85, 0xe9, 0xa1, 0x86, 0xe9, 0xa1,
  0x87, 0xe9, 0xa1, 0x88, 0xe9, 0xa1, 0x89, 0xe9, 0xa1, 0x8a, 0xe9, 0xa1,
  0x8b, 0xe9, 0xa1, 0x8c, 0xe9, 0xa1, 0x8d, 0xe9, 0xa1, 0x8e, 0xe9, 0xa1,
  0x8f, 0xe9, 0xa1, 0x90, 0xe9, 0xa1, 0x91, 0xe9, 0xa1, 0x92, 0xe9, 0xa1,
  0x93, 0xe9, 0xa1, 0x94, 0xe9, 0xa1, 0x95, 0xe9, 0xa1, 0x96, 0xe9, 0xa1,
  0x97, 0xe9, 0xa1, 0x98, 0xe9, 0xa1, 0x99, 0xe9, 0xa1, 0x9a, 0xe9, 0xa1,
  0x9b, 0xe9, 0xa1, 0x9c, 0xe9, 0xa1, 0x9d, 0xe9, 0xa1, 0x9e, 0xe9, 0xa1,
  0x9f, 0xe9, 0xa1, 0xa0, 0xe9, 0xa1, 0xa1, 0xe9, 0xa1, 0xa2, 0xe9, 0xa1,
  0xa3, 0xe9, 0xa1, 0xa4, 0xe9, 0xa1, 0xa5, 0xe9, 0xa1, 0xa6, 0xe9, 0xa1,
  0xa7, 0xe9, 0xa1, 0xa8, 0xe9, 0xa1, 0xa9, 0xe9, 0xa1, 0xaa, 0xe9, 0xa1,
  0xab, 0xe9, 0xa1, 0xac, 0xe9, 0xa1, 0xad, 0xe9, 0xa1, 0xae, 0xe7, 0x9d,
  0xa2, 0xe7, 0x9d, 0xa5, 0xe7, 0x9d, 0xbf, 0xe7, 0x9e, 0x8d, 0xe7, 0x9d,
  0xbd, 0xe7, 0x9e, 0x80, 0xe7, 0x9e, 0x8c, 0xe7, 0x9e, 0x91, 0xe7, 0x9e,
  0x9f, 0xe7, 0x9e, 0xa0, 0xe7, 0x9e, 0xb0, 0xe7, 0x9e, 0xb5, 0xe7, 0x9e,
  0xbd, 0xe7, 0x94, 0xba, 0xe7, 0x95, 0x80, 0xe7, 0x95, 0x8e, 0xe7, 0x95,
  0x8b, 0xe7, 0x95, 0x88, 0xe7, 0x95, 0x9b, 0xe7, 0x95, 0xb2, 0xe7, 0x95,
  0xb9, 0xe7, 0x96, 0x83, 0xe7, 0xbd, 0x98, 0xe7, 0xbd, 0xa1, 0xe7, 0xbd,
  0x9f, 0xe8, 0xa9, 0x88, 0xe7, 0xbd, 0xa8, 0xe7, 0xbd, 0xb4, 0xe7, 0xbd,
  0xb1, 0xe7, 0xbd, 0xb9, 0xe7, 0xbe, 0x81, 0xe7, 0xbd, 0xbe, 0xe7, 0x9b,
  0x8d, 0xe7, 0x9b, 0xa5, 0xe8, 0xa0, 0xb2, 0xe9, 0x92, 0x85, 0xe9, 0x92,
  0x86, 0xe9, 0x92, 0x87, 0xe9, 0x92, 0x8b, 0xe9, 0x92, 0x8a, 0xe9, 0x92,
  0x8c, 0xe9, 0x92, 0x8d, 0xe9, 0x92, 0x8f, 0xe9, 0x92, 0x90, 0xe9, 0x92,
  0x94, 0xe9, 0x92, 0x97, 0xe9, 0x92, 0x95, 0xe9, 0x92, 0x9a, 0xe9, 0x92,
  0x9b, 0xe9, 0x92, 0x9c, 0xe9, 0x92, 0xa3, 0xe9, 0x92, 0xa4, 0xe9, 0x92,
  0xab, 0xe9, 0x92, 0xaa, 0xe9, 0x92, 0xad, 0xe9, 0x92, 0xac, 0xe9, 0x92,
  0xaf, 0xe9, 0x92, 0xb0, 0xe9, 0x92, 0xb2, 0xe9, 0x92, 0xb4, 0xe9, 0x92,
  0xb6, 0xe9, 0x92, 0xb7, 0xe9, 0x92, 0xb8, 0xe9, 0x92, 0xb9, 0xe9, 0x92,
  0xba, 0xe9, 0x92, 0xbc, 0xe9, 0x92, 0xbd, 0xe9, 0x92, 0xbf, 0xe9, 0x93,
  0x84, 0xe9, 0x93, 0x88, 0xe9, 0x93, 0x89, 0xe9, 0x93, 0x8a, 0xe9, 0x93,
  0x8b, 0xe9, 0x93, 0x8c, 0xe9, 0x93, 0x8d, 0xe9, 0x93, 0x8e, 0xe9, 0x93,
  0x90, 0xe9, 0x93, 0x91, 0xe9, 0x93, 0x92, 0xe9, 0x93, 0x95, 0xe9, 0x93,
  0x96, 0xe9, 0x93, 0x97, 0xe9, 0x93, 0x99, 0xe9, 0x93, 0x98, 0xe9, 0x93,
  0x9b, 0xe9, 0x93, 0x9e, 0xe9, 0x93, 0x9f, 0xe9, 0x93, 0xa0, 0xe9, 0x93,
  0xa2, 0xe9, 0x93, 0xa4, 0xe9, 0x93, 0xa5, 0xe9, 0x93, 0xa7, 0xe9, 0x93,
  0xa8, 0xe9, 0x93, 0xaa
};

unsigned char gbk_row_0xef_converted_to_utf8[] = {
  0xe9, 0xa1, 0xaf, 0xe9, 0xa1, 0xb0, 0xe9,
  0xa1, 0xb1, 0xe9, 0xa1, 0xb2, 0xe9, 0xa1, 0xb3, 0xe9, 0xa1, 0xb4, 0xe9,
  0xa2, 0x8b, 0xe9, 0xa2, 0x8e, 0xe9, 0xa2, 0x92, 0xe9, 0xa2, 0x95, 0xe9,
  0xa2, 0x99, 0xe9, 0xa2, 0xa3, 0xe9, 0xa2, 0xa8, 0xe9, 0xa2, 0xa9, 0xe9,
  0xa2, 0xaa, 0xe9, 0xa2, 0xab, 0xe9, 0xa2, 0xac, 0xe9, 0xa2, 0xad, 0xe9,
  0xa2, 0xae, 0xe9, 0xa2, 0xaf, 0xe9, 0xa2, 0xb0, 0xe9, 0xa2, 0xb1, 0xe9,
  0xa2, 0xb2, 0xe9, 0xa2, 0xb3, 0xe9, 0xa2, 0xb4, 0xe9, 0xa2, 0xb5, 0xe9,
  0xa2, 0xb6, 0xe9, 0xa2, 0xb7, 0xe9, 0xa2, 0xb8, 0xe9, 0xa2, 0xb9, 0xe9,
  0xa2, 0xba, 0xe9, 0xa2, 0xbb, 0xe9, 0xa2, 0xbc, 0xe9, 0xa2, 0xbd, 0xe9,
  0xa2, 0xbe, 0xe9, 0xa2, 0xbf, 0xe9, 0xa3, 0x80, 0xe9, 0xa3, 0x81, 0xe9,
  0xa3, 0x82, 0xe9, 0xa3, 0x83, 0xe9, 0xa3, 0x84, 0xe9, 0xa3, 0x85, 0xe9,
  0xa3, 0x86, 0xe9, 0xa3, 0x87, 0xe9, 0xa3, 0x88, 0xe9, 0xa3, 0x89, 0xe9,
  0xa3, 0x8a, 0xe9, 0xa3, 0x8b, 0xe9, 0xa3, 0x8c, 0xe9, 0xa3, 0x8d, 0xe9,
  0xa3, 0x8f, 0xe9, 0xa3, 0x90, 0xe9, 0xa3, 0x94, 0xe9, 0xa3, 0x96, 0xe9,
  0xa3, 0x97, 0xe9, 0xa3, 0x9b, 0xe9, 0xa3, 0x9c, 0xe9, 0xa3, 0x9d, 0xe9,
  0xa3, 0xa0, 0xe9, 0xa3, 0xa1, 0xe9, 0xa3, 0xa2, 0xe9, 0xa3, 0xa3, 0xe9,
  0xa3, 0xa4, 0xe9, 0xa3, 0xa5, 0xe9, 0xa3, 0xa6, 0xe9, 0xa3, 0xa9, 0xe9,
  0xa3, 0xaa, 0xe9, 0xa3, 0xab, 0xe9, 0xa3, 0xac, 0xe9, 0xa3, 0xad, 0xe9,
  0xa3, 0xae, 0xe9, 0xa3, 0xaf, 0xe9, 0xa3, 0xb0, 0xe9, 0xa3, 0xb1, 0xe9,
  0xa3, 0xb2, 0xe9, 0xa3, 0xb3, 0xe9, 0xa3, 0xb4, 0xe9, 0xa3, 0xb5, 0xe9,
  0xa3, 0xb6, 0xe9, 0xa3, 0xb7, 0xe9, 0xa3, 0xb8, 0xe9, 0xa3, 0xb9, 0xe9,
  0xa3, 0xba, 0xe9, 0xa3, 0xbb, 0xe9, 0xa3, 0xbc, 0xe9, 0xa3, 0xbd, 0xe9,
  0xa3, 0xbe, 0xe9, 0xa3, 0xbf, 0xe9, 0xa4, 0x80, 0xe9, 0xa4, 0x81, 0xe9,
  0xa4, 0x82, 0xe9, 0xa4, 0x83, 0xe9, 0xa4, 0x84, 0xe9, 0xa4, 0x85, 0xe9,
  0xa4, 0x86, 0xe9, 0xa4, 0x87, 0xe9, 0x93, 0xa9, 0xe9, 0x93, 0xab, 0xe9,
  0x93, 0xae, 0xe9, 0x93, 0xaf, 0xe9, 0x93, 0xb3, 0xe9, 0x93, 0xb4, 0xe9,
  0x93, 0xb5, 0xe9, 0x93, 0xb7, 0xe9, 0x93, 0xb9, 0xe9, 0x93, 0xbc, 0xe9,
  0x93, 0xbd, 0xe9, 0x93, 0xbf, 0xe9, 0x94, 0x83, 0xe9, 0x94, 0x82, 0xe9,
  0x94, 0x86, 0xe9, 0x94, 0x87, 0xe9, 0x94, 0x89, 0xe9, 0x94, 0x8a, 0xe9,
  0x94, 0x8d, 0xe9, 0x94, 0x8e, 0xe9, 0x94, 0x8f, 0xe9, 0x94, 0x92, 0xe9,
  0x94, 0x93, 0xe9, 0x94, 0x94, 0xe9, 0x94, 0x95, 0xe9, 0x94, 0x96, 0xe9,
  0x94, 0x98, 0xe9, 0x94, 0x9b, 0xe9, 0x94, 0x9d, 0xe9, 0x94, 0x9e, 0xe9,
  0x94, 0x9f, 0xe9, 0x94, 0xa2, 0xe9, 0x94, 0xaa, 0xe9, 0x94, 0xab, 0xe9,
  0x94, 0xa9, 0xe9, 0x94, 0xac, 0xe9, 0x94, 0xb1, 0xe9, 0x94, 0xb2, 0xe9,
  0x94, 0xb4, 0xe9, 0x94, 0xb6, 0xe9, 0x94, 0xb7, 0xe9, 0x94, 0xb8, 0xe9,
  0x94, 0xbc, 0xe9, 0x94, 0xbe, 0xe9, 0x94, 0xbf, 0xe9, 0x95, 0x82, 0xe9,
  0x94, 0xb5, 0xe9, 0x95, 0x84, 0xe9, 0x95, 0x85, 0xe9, 0x95, 0x86, 0xe9,
  0x95, 0x89, 0xe9, 0x95, 0x8c, 0xe9, 0x95, 0x8e, 0xe9, 0x95, 0x8f, 0xe9,
  0x95, 0x92, 0xe9, 0x95, 0x93, 0xe9, 0x95, 0x94, 0xe9, 0x95, 0x96, 0xe9,
  0x95, 0x97, 0xe9, 0x95, 0x98, 0xe9, 0x95, 0x99, 0xe9, 0x95, 0x9b, 0xe9,
  0x95, 0x9e, 0xe9, 0x95, 0x9f, 0xe9, 0x95, 0x9d, 0xe9, 0x95, 0xa1, 0xe9,
  0x95, 0xa2, 0xe9, 0x95, 0xa4, 0xe9, 0x95, 0xa5, 0xe9, 0x95, 0xa6, 0xe9,
  0x95, 0xa7, 0xe9, 0x95, 0xa8, 0xe9, 0x95, 0xa9, 0xe9, 0x95, 0xaa, 0xe9,
  0x95, 0xab, 0xe9, 0x95, 0xac, 0xe9, 0x95, 0xaf, 0xe9, 0x95, 0xb1, 0xe9,
  0x95, 0xb2, 0xe9, 0x95, 0xb3, 0xe9, 0x94, 0xba, 0xe7, 0x9f, 0xa7, 0xe7,
  0x9f, 0xac, 0xe9, 0x9b, 0x89, 0xe7, 0xa7, 0x95, 0xe7, 0xa7, 0xad, 0xe7,
  0xa7, 0xa3, 0xe7, 0xa7, 0xab, 0xe7, 0xa8, 0x86, 0xe5, 0xb5, 0x87, 0xe7,
  0xa8, 0x83, 0xe7, 0xa8, 0x82, 0xe7, 0xa8, 0x9e, 0xe7, 0xa8, 0x94
};

unsigned char gbk_row_0xf0_converted_to_utf8[] = {
  0xe9, 0xa4, 0x88, 0xe9, 0xa4, 0x89, 0xe9, 0xa4, 0x8a, 0xe9, 0xa4, 0x8b,
  0xe9, 0xa4, 0x8c, 0xe9, 0xa4, 0x8e, 0xe9, 0xa4, 0x8f, 0xe9, 0xa4, 0x91,
  0xe9, 0xa4, 0x92, 0xe9, 0xa4, 0x93, 0xe9, 0xa4, 0x94, 0xe9, 0xa4, 0x95,
  0xe9, 0xa4, 0x96, 0xe9, 0xa4, 0x97, 0xe9, 0xa4, 0x98, 0xe9, 0xa4, 0x99,
  0xe9, 0xa4, 0x9a, 0xe9, 0xa4, 0x9b, 0xe9, 0xa4, 0x9c, 0xe9, 0xa4, 0x9d,
  0xe9, 0xa4, 0x9e, 0xe9, 0xa4, 0x9f, 0xe9, 0xa4, 0xa0, 0xe9, 0xa4, 0xa1,
  0xe9, 0xa4, 0xa2, 0xe9, 0xa4, 0xa3, 0xe9, 0xa4, 0xa4, 0xe9, 0xa4, 0xa5,
  0xe9, 0xa4, 0xa6, 0xe9, 0xa4, 0xa7, 0xe9, 0xa4, 0xa8, 0xe9, 0xa4, 0xa9,
  0xe9, 0xa4, 0xaa, 0xe9, 0xa4, 0xab, 0xe9, 0xa4, 0xac, 0xe9, 0xa4, 0xad,
  0xe9, 0xa4, 0xaf, 0xe9, 0xa4, 0xb0, 0xe9, 0xa4, 0xb1, 0xe9, 0xa4, 0xb2,
  0xe9, 0xa4, 0xb3, 0xe9, 0xa4, 0xb4, 0xe9, 0xa4, 0xb5, 0xe9, 0xa4, 0xb6,
  0xe9, 0xa4, 0xb7, 0xe9, 0xa4, 0xb8, 0xe9, 0xa4, 0xb9, 0xe9, 0xa4, 0xba,
  0xe9, 0xa4, 0xbb, 0xe9, 0xa4, 0xbc, 0xe9, 0xa4, 0xbd, 0xe9, 0xa4, 0xbe,
  0xe9, 0xa4, 0xbf, 0xe9, 0xa5, 0x80, 0xe9, 0xa5, 0x81, 0xe9, 0xa5, 0x82,
  0xe9, 0xa5, 0x83, 0xe9, 0xa5, 0x84, 0xe9, 0xa5, 0x85, 0xe9, 0xa5, 0x86,
  0xe9, 0xa5, 0x87, 0xe9, 0xa5, 0x88, 0xe9, 0xa5, 0x89, 0xe9, 0xa5, 0x8a,
  0xe9, 0xa5, 0x8b, 0xe9, 0xa5, 0x8c, 0xe9, 0xa5, 0x8d, 0xe9, 0xa5, 0x8e,
  0xe9, 0xa5, 0x8f, 0xe9, 0xa5, 0x90, 0xe9, 0xa5, 0x91, 0xe9, 0xa5, 0x92,
  0xe9, 0xa5, 0x93, 0xe9, 0xa5, 0x96, 0xe9, 0xa5, 0x97, 0xe9, 0xa5, 0x98,
  0xe9, 0xa5, 0x99, 0xe9, 0xa5, 0x9a, 0xe9, 0xa5, 0x9b, 0xe9, 0xa5, 0x9c,
  0xe9, 0xa5, 0x9d, 0xe9, 0xa5, 0x9e, 0xe9, 0xa5, 0x9f, 0xe9, 0xa5, 0xa0,
  0xe9, 0xa5, 0xa1, 0xe9, 0xa5, 0xa2, 0xe9, 0xa5, 0xa4, 0xe9, 0xa5, 0xa6,
  0xe9, 0xa5, 0xb3, 0xe9, 0xa5, 0xb8, 0xe9, 0xa5, 0xb9, 0xe9, 0xa5, 0xbb,
  0xe9, 0xa5, 0xbe, 0xe9, 0xa6, 0x82, 0xe9, 0xa6, 0x83, 0xe9, 0xa6, 0x89,
  0xe7, 0xa8, 0xb9, 0xe7, 0xa8, 0xb7, 0xe7, 0xa9, 0x91, 0xe9, 0xbb, 0x8f,
  0xe9, 0xa6, 0xa5, 0xe7, 0xa9, 0xb0, 0xe7, 0x9a, 0x88, 0xe7, 0x9a, 0x8e,
  0xe7, 0x9a, 0x93, 0xe7, 0x9a, 0x99, 0xe7, 0x9a, 0xa4, 0xe7, 0x93, 0x9e,
  0xe7, 0x93, 0xa0, 0xe7, 0x94, 0xac, 0xe9, 0xb8, 0xa0, 0xe9, 0xb8, 0xa2,
  0xe9, 0xb8, 0xa8, 0xe9, 0xb8, 0xa9, 0xe9, 0xb8, 0xaa, 0xe9, 0xb8, 0xab,
  0xe9, 0xb8, 0xac, 0xe9, 0xb8, 0xb2, 0xe9, 0xb8, 0xb1, 0xe9, 0xb8, 0xb6,
  0xe9, 0xb8, 0xb8, 0xe9, 0xb8, 0xb7, 0xe9, 0xb8, 0xb9, 0xe9, 0xb8, 0xba,
  0xe9, 0xb8, 0xbe, 0xe9, 0xb9, 0x81, 0xe9, 0xb9, 0x82, 0xe9, 0xb9, 0x84,
  0xe9, 0xb9, 0x86, 0xe9, 0xb9, 0x87, 0xe9, 0xb9, 0x88, 0xe9, 0xb9, 0x89,
  0xe9, 0xb9, 0x8b, 0xe9, 0xb9, 0x8c, 0xe9, 0xb9, 0x8e, 0xe9, 0xb9, 0x91,
  0xe9, 0xb9, 0x95, 0xe9, 0xb9, 0x97, 0xe9, 0xb9, 0x9a, 0xe9, 0xb9, 0x9b,
  0xe9, 0xb9, 0x9c, 0xe9, 0xb9, 0x9e, 0xe9, 0xb9, 0xa3, 0xe9, 0xb9, 0xa6,
  0xe9, 0xb9, 0xa7, 0xe9, 0xb9, 0xa8, 0xe9, 0xb9, 0xa9, 0xe9, 0xb9, 0xaa,
  0xe9, 0xb9, 0xab, 0xe9, 0xb9, 0xac, 0xe9, 0xb9, 0xb1, 0xe9, 0xb9, 0xad,
  0xe9, 0xb9, 0xb3, 0xe7, 0x96, 0x92, 0xe7, 0x96, 0x94, 0xe7, 0x96, 0x96,
  0xe7, 0x96, 0xa0, 0xe7, 0x96, 0x9d, 0xe7, 0x96, 0xac, 0xe7, 0x96, 0xa3,
  0xe7, 0x96, 0xb3, 0xe7, 0x96, 0xb4, 0xe7, 0x96, 0xb8, 0xe7, 0x97, 0x84,
  0xe7, 0x96, 0xb1, 0xe7, 0x96, 0xb0, 0xe7, 0x97, 0x83, 0xe7, 0x97, 0x82,
  0xe7, 0x97, 0x96, 0xe7, 0x97, 0x8d, 0xe7, 0x97, 0xa3, 0xe7, 0x97, 0xa8,
  0xe7, 0x97, 0xa6, 0xe7, 0x97, 0xa4, 0xe7, 0x97, 0xab, 0xe7, 0x97, 0xa7,
  0xe7, 0x98, 0x83, 0xe7, 0x97, 0xb1, 0xe7, 0x97, 0xbc, 0xe7, 0x97, 0xbf,
  0xe7, 0x98, 0x90, 0xe7, 0x98, 0x80, 0xe7, 0x98, 0x85, 0xe7, 0x98, 0x8c,
  0xe7, 0x98, 0x97, 0xe7, 0x98, 0x8a, 0xe7, 0x98, 0xa5, 0xe7, 0x98, 0x98,
  0xe7, 0x98, 0x95, 0xe7, 0x98, 0x99
};

unsigned char gbk_row_0xf1_converted_to_utf8[] = {
  0xe9, 0xa6, 0x8c, 0xe9, 0xa6,
  0x8e, 0xe9, 0xa6, 0x9a, 0xe9, 0xa6, 0x9b, 0xe9, 0xa6, 0x9c, 0xe9, 0xa6,
  0x9d, 0xe9, 0xa6, 0x9e, 0xe9, 0xa6, 0x9f, 0xe9, 0xa6, 0xa0, 0xe9, 0xa6,
  0xa1, 0xe9, 0xa6, 0xa2, 0xe9, 0xa6, 0xa3, 0xe9, 0xa6, 0xa4, 0xe9, 0xa6,
  0xa6, 0xe9, 0xa6, 0xa7, 0xe9, 0xa6, 0xa9, 0xe9, 0xa6, 0xaa, 0xe9, 0xa6,
  0xab, 0xe9, 0xa6, 0xac, 0xe9, 0xa6, 0xad, 0xe9, 0xa6, 0xae, 0xe9, 0xa6,
  0xaf, 0xe9, 0xa6, 0xb0, 0xe9, 0xa6, 0xb1, 0xe9, 0xa6, 0xb2, 0xe9, 0xa6,
  0xb3, 0xe9, 0xa6, 0xb4, 0xe9, 0xa6, 0xb5, 0xe9, 0xa6, 0xb6, 0xe9, 0xa6,
  0xb7, 0xe9, 0xa6, 0xb8, 0xe9, 0xa6, 0xb9, 0xe9, 0xa6, 0xba, 0xe9, 0xa6,
  0xbb, 0xe9, 0xa6, 0xbc, 0xe9, 0xa6, 0xbd, 0xe9, 0xa6, 0xbe, 0xe9, 0xa6,
  0xbf, 0xe9, 0xa7, 0x80, 0xe9, 0xa7, 0x81, 0xe9, 0xa7, 0x82, 0xe9, 0xa7,
  0x83, 0xe9, 0xa7, 0x84, 0xe9, 0xa7, 0x85, 0xe9, 0xa7, 0x86, 0xe9, 0xa7,
  0x87, 0xe9, 0xa7, 0x88, 0xe9, 0xa7, 0x89, 0xe9, 0xa7, 0x8a, 0xe9, 0xa7,
  0x8b, 0xe9, 0xa7, 0x8c, 0xe9, 0xa7, 0x8d, 0xe9, 0xa7, 0x8e, 0xe9, 0xa7,
  0x8f, 0xe9, 0xa7, 0x90, 0xe9, 0xa7, 0x91, 0xe9, 0xa7, 0x92, 0xe9, 0xa7,
  0x93, 0xe9, 0xa7, 0x94, 0xe9, 0xa7, 0x95, 0xe9, 0xa7, 0x96, 0xe9, 0xa7,
  0x97, 0xe9, 0xa7, 0x98, 0xe9, 0xa7, 0x99, 0xe9, 0xa7, 0x9a, 0xe9, 0xa7,
  0x9b, 0xe9, 0xa7, 0x9c, 0xe9, 0xa7, 0x9d, 0xe9, 0xa7, 0x9e, 0xe9, 0xa7,
  0x9f, 0xe9, 0xa7, 0xa0, 0xe9, 0xa7, 0xa1, 0xe9, 0xa7, 0xa2, 0xe9, 0xa7,
  0xa3, 0xe9, 0xa7, 0xa4, 0xe9, 0xa7, 0xa5, 0xe9, 0xa7, 0xa6, 0xe9, 0xa7,
  0xa7, 0xe9, 0xa7, 0xa8, 0xe9, 0xa7, 0xa9, 0xe9, 0xa7, 0xaa, 0xe9, 0xa7,
  0xab, 0xe9, 0xa7, 0xac, 0xe9, 0xa7, 0xad, 0xe9, 0xa7, 0xae, 0xe9, 0xa7,
  0xaf, 0xe9, 0xa7, 0xb0, 0xe9, 0xa7, 0xb1, 0xe9, 0xa7, 0xb2, 0xe9, 0xa7,
  0xb3, 0xe9, 0xa7, 0xb4, 0xe9, 0xa7, 0xb5, 0xe9, 0xa7, 0xb6, 0xe9, 0xa7,
  0xb7, 0xe9, 0xa7, 0xb8, 0xe9, 0xa7, 0xb9, 0xe7, 0x98, 0x9b, 0xe7, 0x98,
  0xbc, 0xe7, 0x98, 0xa2, 0xe7, 0x98, 0xa0, 0xe7, 0x99, 0x80, 0xe7, 0x98,
  0xad, 0xe7, 0x98, 0xb0, 0xe7, 0x98, 0xbf, 0xe7, 0x98, 0xb5, 0xe7, 0x99,
  0x83, 0xe7, 0x98, 0xbe, 0xe7, 0x98, 0xb3, 0xe7, 0x99, 0x8d, 0xe7, 0x99,
  0x9e, 0xe7, 0x99, 0x94, 0xe7, 0x99, 0x9c, 0xe7, 0x99, 0x96, 0xe7, 0x99,
  0xab, 0xe7, 0x99, 0xaf, 0xe7, 0xbf, 0x8a, 0xe7, 0xab, 0xa6, 0xe7, 0xa9,
  0xb8, 0xe7, 0xa9, 0xb9, 0xe7, 0xaa, 0x80, 0xe7, 0xaa, 0x86, 0xe7, 0xaa,
  0x88, 0xe7, 0xaa, 0x95, 0xe7, 0xaa, 0xa6, 0xe7, 0xaa, 0xa0, 0xe7, 0xaa,
  0xac, 0xe7, 0xaa, 0xa8, 0xe7, 0xaa, 0xad, 0xe7, 0xaa, 0xb3, 0xe8, 0xa1,
  0xa4, 0xe8, 0xa1, 0xa9, 0xe8, 0xa1, 0xb2, 0xe8, 0xa1, 0xbd, 0xe8, 0xa1,
  0xbf, 0xe8, 0xa2, 0x82, 0xe8, 0xa2, 0xa2, 0xe8, 0xa3, 0x86, 0xe8, 0xa2,
  0xb7, 0xe8, 0xa2, 0xbc, 0xe8, 0xa3, 0x89, 0xe8, 0xa3, 0xa2, 0xe8, 0xa3,
  0x8e, 0xe8, 0xa3, 0xa3, 0xe8, 0xa3, 0xa5, 0xe8, 0xa3, 0xb1, 0xe8, 0xa4,
  0x9a, 0xe8, 0xa3, 0xbc, 0xe8, 0xa3, 0xa8, 0xe8, 0xa3, 0xbe, 0xe8, 0xa3,
  0xb0, 0xe8, 0xa4, 0xa1, 0xe8, 0xa4, 0x99, 0xe8, 0xa4, 0x93, 0xe8, 0xa4,
  0x9b, 0xe8, 0xa4, 0x8a, 0xe8, 0xa4, 0xb4, 0xe8, 0xa4, 0xab, 0xe8, 0xa4,
  0xb6, 0xe8, 0xa5, 0x81, 0xe8, 0xa5, 0xa6, 0xe8, 0xa5, 0xbb, 0xe7, 0x96,
  0x8b, 0xe8, 0x83, 0xa5, 0xe7, 0x9a, 0xb2, 0xe7, 0x9a, 0xb4, 0xe7, 0x9f,
  0x9c, 0xe8, 0x80, 0x92, 0xe8, 0x80, 0x94, 0xe8, 0x80, 0x96, 0xe8, 0x80,
  0x9c, 0xe8, 0x80, 0xa0, 0xe8, 0x80, 0xa2, 0xe8, 0x80, 0xa5, 0xe8, 0x80,
  0xa6, 0xe8, 0x80, 0xa7, 0xe8, 0x80, 0xa9, 0xe8, 0x80, 0xa8, 0xe8, 0x80,
  0xb1, 0xe8, 0x80, 0x8b, 0xe8, 0x80, 0xb5, 0xe8, 0x81, 0x83, 0xe8, 0x81,
  0x86, 0xe8, 0x81, 0x8d, 0xe8, 0x81, 0x92, 0xe8, 0x81, 0xa9, 0xe8, 0x81,
  0xb1, 0xe8, 0xa6, 0x83, 0xe9, 0xa1, 0xb8, 0xe9, 0xa2, 0x80, 0xe9, 0xa2,
  0x83
};

unsigned char gbk_row_0xf2_converted_to_utf8[] = {
  0xe9, 0xa7, 0xba, 0xe9, 0xa7, 0xbb, 0xe9, 0xa7, 0xbc, 0xe9,
  0xa7, 0xbd, 0xe9, 0xa7, 0xbe, 0xe9, 0xa7, 0xbf, 0xe9, 0xa8, 0x80, 0xe9,
  0xa8, 0x81, 0xe9, 0xa8, 0x82, 0xe9, 0xa8, 0x83, 0xe9, 0xa8, 0x84, 0xe9,
  0xa8, 0x85, 0xe9, 0xa8, 0x86, 0xe9, 0xa8, 0x87, 0xe9, 0xa8, 0x88, 0xe9,
  0xa8, 0x89, 0xe9, 0xa8, 0x8a, 0xe9, 0xa8, 0x8b, 0xe9, 0xa8, 0x8c, 0xe9,
  0xa8, 0x8d, 0xe9, 0xa8, 0x8e, 0xe9, 0xa8, 0x8f, 0xe9, 0xa8, 0x90, 0xe9,
  0xa8, 0x91, 0xe9, 0xa8, 0x92, 0xe9, 0xa8, 0x93, 0xe9, 0xa8, 0x94, 0xe9,
  0xa8, 0x95, 0xe9, 0xa8, 0x96, 0xe9, 0xa8, 0x97, 0xe9, 0xa8, 0x98, 0xe9,
  0xa8, 0x99, 0xe9, 0xa8, 0x9a, 0xe9, 0xa8, 0x9b, 0xe9, 0xa8, 0x9c, 0xe9,
  0xa8, 0x9d, 0xe9, 0xa8, 0x9e, 0xe9, 0xa8, 0x9f, 0xe9, 0xa8, 0xa0, 0xe9,
  0xa8, 0xa1, 0xe9, 0xa8, 0xa2, 0xe9, 0xa8, 0xa3, 0xe9, 0xa8, 0xa4, 0xe9,
  0xa8, 0xa5, 0xe9, 0xa8, 0xa6, 0xe9, 0xa8, 0xa7, 0xe9, 0xa8, 0xa8, 0xe9,
  0xa8, 0xa9, 0xe9, 0xa8, 0xaa, 0xe9, 0xa8, 0xab, 0xe9, 0xa8, 0xac, 0xe9,
  0xa8, 0xad, 0xe9, 0xa8, 0xae, 0xe9, 0xa8, 0xaf, 0xe9, 0xa8, 0xb0, 0xe9,
  0xa8, 0xb1, 0xe9, 0xa8, 0xb2, 0xe9, 0xa8, 0xb3, 0xe9, 0xa8, 0xb4, 0xe9,
  0xa8, 0xb5, 0xe9, 0xa8, 0xb6, 0xe9, 0xa8, 0xb7, 0xe9, 0xa8, 0xb8, 0xe9,
  0xa8, 0xb9, 0xe9, 0xa8, 0xba, 0xe9, 0xa8, 0xbb, 0xe9, 0xa8, 0xbc, 0xe9,
  0xa8, 0xbd, 0xe9, 0xa8, 0xbe, 0xe9, 0xa8, 0xbf, 0xe9, 0xa9, 0x80, 0xe9,
  0xa9, 0x81, 0xe9, 0xa9, 0x82, 0xe9, 0xa9, 0x83, 0xe9, 0xa9, 0x84, 0xe9,
  0xa9, 0x85, 0xe9, 0xa9, 0x86, 0xe9, 0xa9, 0x87, 0xe9, 0xa9, 0x88, 0xe9,
  0xa9, 0x89, 0xe9, 0xa9, 0x8a, 0xe9, 0xa9, 0x8b, 0xe9, 0xa9, 0x8c, 0xe9,
  0xa9, 0x8d, 0xe9, 0xa9, 0x8e, 0xe9, 0xa9, 0x8f, 0xe9, 0xa9, 0x90, 0xe9,
  0xa9, 0x91, 0xe9, 0xa9, 0x92, 0xe9, 0xa9, 0x93, 0xe9, 0xa9, 0x94, 0xe9,
  0xa9, 0x95, 0xe9, 0xa9, 0x96, 0xe9, 0xa9, 0x97, 0xe9, 0xa9, 0x98, 0xe9,
  0xa9, 0x99, 0xe9, 0xa2, 0x89, 0xe9, 0xa2, 0x8c, 0xe9, 0xa2, 0x8d, 0xe9,
  0xa2, 0x8f, 0xe9, 0xa2, 0x94, 0xe9, 0xa2, 0x9a, 0xe9, 0xa2, 0x9b, 0xe9,
  0xa2, 0x9e, 0xe9, 0xa2, 0x9f, 0xe9, 0xa2, 0xa1, 0xe9, 0xa2, 0xa2, 0xe9,
  0xa2, 0xa5, 0xe9, 0xa2, 0xa6, 0xe8, 0x99, 0x8d, 0xe8, 0x99, 0x94, 0xe8,
  0x99, 0xac, 0xe8, 0x99, 0xae, 0xe8, 0x99, 0xbf, 0xe8, 0x99, 0xba, 0xe8,
  0x99, 0xbc, 0xe8, 0x99, 0xbb, 0xe8, 0x9a, 0xa8, 0xe8, 0x9a, 0x8d, 0xe8,
  0x9a, 0x8b, 0xe8, 0x9a, 0xac, 0xe8, 0x9a, 0x9d, 0xe8, 0x9a, 0xa7, 0xe8,
  0x9a, 0xa3, 0xe8, 0x9a, 0xaa, 0xe8, 0x9a, 0x93, 0xe8, 0x9a, 0xa9, 0xe8,
  0x9a, 0xb6, 0xe8, 0x9b, 0x84, 0xe8, 0x9a, 0xb5, 0xe8, 0x9b, 0x8e, 0xe8,
  0x9a, 0xb0, 0xe8, 0x9a, 0xba, 0xe8, 0x9a, 0xb1, 0xe8, 0x9a, 0xaf, 0xe8,
  0x9b, 0x89, 0xe8, 0x9b, 0x8f, 0xe8, 0x9a, 0xb4, 0xe8, 0x9b, 0xa9, 0xe8,
  0x9b, 0xb1, 0xe8, 0x9b, 0xb2, 0xe8, 0x9b, 0xad, 0xe8, 0x9b, 0xb3, 0xe8,
  0x9b, 0x90, 0xe8, 0x9c, 0x93, 0xe8, 0x9b, 0x9e, 0xe8, 0x9b, 0xb4, 0xe8,
  0x9b, 0x9f, 0xe8, 0x9b, 0x98, 0xe8, 0x9b, 0x91, 0xe8, 0x9c, 0x83, 0xe8,
  0x9c, 0x87, 0xe8, 0x9b, 0xb8, 0xe8, 0x9c, 0x88, 0xe8, 0x9c, 0x8a, 0xe8,
  0x9c, 0x8d, 0xe8, 0x9c, 0x89, 0xe8, 0x9c, 0xa3, 0xe8, 0x9c, 0xbb, 0xe8,
  0x9c, 0x9e, 0xe8, 0x9c, 0xa5, 0xe8, 0x9c, 0xae, 0xe8, 0x9c, 0x9a, 0xe8,
  0x9c, 0xbe, 0xe8, 0x9d, 0x88, 0xe8, 0x9c, 0xb4, 0xe8, 0x9c, 0xb1, 0xe8,
  0x9c, 0xa9, 0xe8, 0x9c, 0xb7, 0xe8, 0x9c, 0xbf, 0xe8, 0x9e, 0x82, 0xe8,
  0x9c, 0xa2, 0xe8, 0x9d, 0xbd, 0xe8, 0x9d, 0xbe, 0xe8, 0x9d, 0xbb, 0xe8,
  0x9d, 0xa0, 0xe8, 0x9d, 0xb0, 0xe8, 0x9d, 0x8c, 0xe8, 0x9d, 0xae, 0xe8,
  0x9e, 0x8b, 0xe8, 0x9d, 0x93, 0xe8, 0x9d, 0xa3, 0xe8, 0x9d, 0xbc, 0xe8,
  0x9d, 0xa4, 0xe8, 0x9d, 0x99, 0xe8, 0x9d, 0xa5, 0xe8, 0x9e, 0x93, 0xe8,
  0x9e, 0xaf, 0xe8, 0x9e, 0xa8, 0xe8, 0x9f, 0x92
};

unsigned char gbk_row_0xf3_converted_to_utf8[] = {
  0xe9, 0xa9, 0x9a,
  0xe9, 0xa9, 0x9b, 0xe9, 0xa9, 0x9c, 0xe9, 0xa9, 0x9d, 0xe9, 0xa9, 0x9e,
  0xe9, 0xa9, 0x9f, 0xe9, 0xa9, 0xa0, 0xe9, 0xa9, 0xa1, 0xe9, 0xa9, 0xa2,
  0xe9, 0xa9, 0xa3, 0xe9, 0xa9, 0xa4, 0xe9, 0xa9, 0xa5, 0xe9, 0xa9, 0xa6,
  0xe9, 0xa9, 0xa7, 0xe9, 0xa9, 0xa8, 0xe9, 0xa9, 0xa9, 0xe9, 0xa9, 0xaa,
  0xe9, 0xa9, 0xab, 0xe9, 0xa9, 0xb2, 0xe9, 0xaa, 0x83, 0xe9, 0xaa, 0x89,
  0xe9, 0xaa, 0x8d, 0xe9, 0xaa, 0x8e, 0xe9, 0xaa, 0x94, 0xe9, 0xaa, 0x95,
  0xe9, 0xaa, 0x99, 0xe9, 0xaa, 0xa6, 0xe9, 0xaa, 0xa9, 0xe9, 0xaa, 0xaa,
  0xe9, 0xaa, 0xab, 0xe9, 0xaa, 0xac, 0xe9, 0xaa, 0xad, 0xe9, 0xaa, 0xae,
  0xe9, 0xaa, 0xaf, 0xe9, 0xaa, 0xb2, 0xe9, 0xaa, 0xb3, 0xe9, 0xaa, 0xb4,
  0xe9, 0xaa, 0xb5, 0xe9, 0xaa, 0xb9, 0xe9, 0xaa, 0xbb, 0xe9, 0xaa, 0xbd,
  0xe9, 0xaa, 0xbe, 0xe9, 0xaa, 0xbf, 0xe9, 0xab, 0x83, 0xe9, 0xab, 0x84,
  0xe9, 0xab, 0x86, 0xe9, 0xab, 0x87, 0xe9, 0xab, 0x88, 0xe9, 0xab, 0x89,
  0xe9, 0xab, 0x8a, 0xe9, 0xab, 0x8d, 0xe9, 0xab, 0x8e, 0xe9, 0xab, 0x8f,
  0xe9, 0xab, 0x90, 0xe9, 0xab, 0x92, 0xe9, 0xab, 0x94, 0xe9, 0xab, 0x95,
  0xe9, 0xab, 0x96, 0xe9, 0xab, 0x97, 0xe9, 0xab, 0x99, 0xe9, 0xab, 0x9a,
  0xe9, 0xab, 0x9b, 0xe9, 0xab, 0x9c, 0xe9, 0xab, 0x9d, 0xe9, 0xab, 0x9e,
  0xe9, 0xab, 0xa0, 0xe9, 0xab, 0xa2, 0xe9, 0xab, 0xa3, 0xe9, 0xab, 0xa4,
  0xe9, 0xab, 0xa5, 0xe9, 0xab, 0xa7, 0xe9, 0xab, 0xa8, 0xe9, 0xab, 0xa9,
  0xe9, 0xab, 0xaa, 0xe9, 0xab, 0xac, 0xe9, 0xab, 0xae, 0xe9, 0xab, 0xb0,
  0xe9, 0xab, 0xb1, 0xe9, 0xab, 0xb2, 0xe9, 0xab, 0xb3, 0xe9, 0xab, 0xb4,
  0xe9, 0xab, 0xb5, 0xe9, 0xab, 0xb6, 0xe9, 0xab, 0xb7, 0xe9, 0xab, 0xb8,
  0xe9, 0xab, 0xba, 0xe9, 0xab, 0xbc, 0xe9, 0xab, 0xbd, 0xe9, 0xab, 0xbe,
  0xe9, 0xab, 0xbf, 0xe9, 0xac, 0x80, 0xe9, 0xac, 0x81, 0xe9, 0xac, 0x82,
  0xe9, 0xac, 0x84, 0xe9, 0xac, 0x85, 0xe9, 0xac, 0x86, 0xe8, 0x9f, 0x86,
  0xe8, 0x9e, 0x88, 0xe8, 0x9e, 0x85, 0xe8, 0x9e, 0xad, 0xe8, 0x9e, 0x97,
  0xe8, 0x9e, 0x83, 0xe8, 0x9e, 0xab, 0xe8, 0x9f, 0xa5, 0xe8, 0x9e, 0xac,
  0xe8, 0x9e, 0xb5, 0xe8, 0x9e, 0xb3, 0xe8, 0x9f, 0x8b, 0xe8, 0x9f, 0x93,
  0xe8, 0x9e, 0xbd, 0xe8, 0x9f, 0x91, 0xe8, 0x9f, 0x80, 0xe8, 0x9f, 0x8a,
  0xe8, 0x9f, 0x9b, 0xe8, 0x9f, 0xaa, 0xe8, 0x9f, 0xa0, 0xe8, 0x9f, 0xae,
  0xe8, 0xa0, 0x96, 0xe8, 0xa0, 0x93, 0xe8, 0x9f, 0xbe, 0xe8, 0xa0, 0x8a,
  0xe8, 0xa0, 0x9b, 0xe8, 0xa0, 0xa1, 0xe8, 0xa0, 0xb9, 0xe8, 0xa0, 0xbc,
  0xe7, 0xbc, 0xb6, 0xe7, 0xbd, 0x82, 0xe7, 0xbd, 0x84, 0xe7, 0xbd, 0x85,
  0xe8, 0x88, 0x90, 0xe7, 0xab, 0xba, 0xe7, 0xab, 0xbd, 0xe7, 0xac, 0x88,
  0xe7, 0xac, 0x83, 0xe7, 0xac, 0x84, 0xe7, 0xac, 0x95, 0xe7, 0xac, 0x8a,
  0xe7, 0xac, 0xab, 0xe7, 0xac, 0x8f, 0xe7, 0xad, 0x87, 0xe7, 0xac, 0xb8,
  0xe7, 0xac, 0xaa, 0xe7, 0xac, 0x99, 0xe7, 0xac, 0xae, 0xe7, 0xac, 0xb1,
  0xe7, 0xac, 0xa0, 0xe7, 0xac, 0xa5, 0xe7, 0xac, 0xa4, 0xe7, 0xac, 0xb3,
  0xe7, 0xac, 0xbe, 0xe7, 0xac, 0x9e, 0xe7, 0xad, 0x98, 0xe7, 0xad, 0x9a,
  0xe7, 0xad, 0x85, 0xe7, 0xad, 0xb5, 0xe7, 0xad, 0x8c, 0xe7, 0xad, 0x9d,
  0xe7, 0xad, 0xa0, 0xe7, 0xad, 0xae, 0xe7, 0xad, 0xbb, 0xe7, 0xad, 0xa2,
  0xe7, 0xad, 0xb2, 0xe7, 0xad, 0xb1, 0xe7, 0xae, 0x90, 0xe7, 0xae, 0xa6,
  0xe7, 0xae, 0xa7, 0xe7, 0xae, 0xb8, 0xe7, 0xae, 0xac, 0xe7, 0xae, 0x9d,
  0xe7, 0xae, 0xa8, 0xe7, 0xae, 0x85, 0xe7, 0xae, 0xaa, 0xe7, 0xae, 0x9c,
  0xe7, 0xae, 0xa2, 0xe7, 0xae, 0xab, 0xe7, 0xae, 0xb4, 0xe7, 0xaf, 0x91,
  0xe7, 0xaf, 0x81, 0xe7, 0xaf, 0x8c, 0xe7, 0xaf, 0x9d, 0xe7, 0xaf, 0x9a,
  0xe7, 0xaf, 0xa5, 0xe7, 0xaf, 0xa6, 0xe7, 0xaf, 0xaa, 0xe7, 0xb0, 0x8c,
  0xe7, 0xaf, 0xbe, 0xe7, 0xaf, 0xbc, 0xe7, 0xb0, 0x8f, 0xe7, 0xb0, 0x96,
  0xe7, 0xb0, 0x8b
};

unsigned char gbk_row_0xf4_converted_to_utf8[] = {
  0xe9, 0xac, 0x87, 0xe9, 0xac, 0x89, 0xe9, 0xac,
  0x8a, 0xe9, 0xac, 0x8b, 0xe9, 0xac, 0x8c, 0xe9, 0xac, 0x8d, 0xe9, 0xac,
  0x8e, 0xe9, 0xac, 0x90, 0xe9, 0xac, 0x91, 0xe9, 0xac, 0x92, 0xe9, 0xac,
  0x94, 0xe9, 0xac, 0x95, 0xe9, 0xac, 0x96, 0xe9, 0xac, 0x97, 0xe9, 0xac,
  0x98, 0xe9, 0xac, 0x99, 0xe9, 0xac, 0x9a, 0xe9, 0xac, 0x9b, 0xe9, 0xac,
  0x9c, 0xe9, 0xac, 0x9d, 0xe9, 0xac, 0x9e, 0xe9, 0xac, 0xa0, 0xe9, 0xac,
  0xa1, 0xe9, 0xac, 0xa2, 0xe9, 0xac, 0xa4, 0xe9, 0xac, 0xa5, 0xe9, 0xac,
  0xa6, 0xe9, 0xac, 0xa7, 0xe9, 0xac, 0xa8, 0xe9, 0xac, 0xa9, 0xe9, 0xac,
  0xaa, 0xe9, 0xac, 0xab, 0xe9, 0xac, 0xac, 0xe9, 0xac, 0xad, 0xe9, 0xac,
  0xae, 0xe9, 0xac, 0xb0, 0xe9, 0xac, 0xb1, 0xe9, 0xac, 0xb3, 0xe9, 0xac,
  0xb4, 0xe9, 0xac, 0xb5, 0xe9, 0xac, 0xb6, 0xe9, 0xac, 0xb7, 0xe9, 0xac,
  0xb8, 0xe9, 0xac, 0xb9, 0xe9, 0xac, 0xba, 0xe9, 0xac, 0xbd, 0xe9, 0xac,
  0xbe, 0xe9, 0xac, 0xbf, 0xe9, 0xad, 0x80, 0xe9, 0xad, 0x86, 0xe9, 0xad,
  0x8a, 0xe9, 0xad, 0x8b, 0xe9, 0xad, 0x8c, 0xe9, 0xad, 0x8e, 0xe9, 0xad,
  0x90, 0xe9, 0xad, 0x92, 0xe9, 0xad, 0x93, 0xe9, 0xad, 0x95, 0xe9, 0xad,
  0x96, 0xe9, 0xad, 0x97, 0xe9, 0xad, 0x98, 0xe9, 0xad, 0x99, 0xe9, 0xad,
  0x9a, 0xe9, 0xad, 0x9b, 0xe9, 0xad, 0x9c, 0xe9, 0xad, 0x9d, 0xe9, 0xad,
  0x9e, 0xe9, 0xad, 0x9f, 0xe9, 0xad, 0xa0, 0xe9, 0xad, 0xa1, 0xe9, 0xad,
  0xa2, 0xe9, 0xad, 0xa3, 0xe9, 0xad, 0xa4, 0xe9, 0xad, 0xa5, 0xe9, 0xad,
  0xa6, 0xe9, 0xad, 0xa7, 0xe9, 0xad, 0xa8, 0xe9, 0xad, 0xa9, 0xe9, 0xad,
  0xaa, 0xe9, 0xad, 0xab, 0xe9, 0xad, 0xac, 0xe9, 0xad, 0xad, 0xe9, 0xad,
  0xae, 0xe9, 0xad, 0xaf, 0xe9, 0xad, 0xb0, 0xe9, 0xad, 0xb1, 0xe9, 0xad,
  0xb2, 0xe9, 0xad, 0xb3, 0xe9, 0xad, 0xb4, 0xe9, 0xad, 0xb5, 0xe9, 0xad,
  0xb6, 0xe9, 0xad, 0xb7, 0xe9, 0xad, 0xb8, 0xe9, 0xad, 0xb9, 0xe9, 0xad,
  0xba, 0xe9, 0xad, 0xbb, 0xe7, 0xb0, 0x9f, 0xe7, 0xb0, 0xaa, 0xe7, 0xb0,
  0xa6, 0xe7, 0xb0, 0xb8, 0xe7, 0xb1, 0x81, 0xe7, 0xb1, 0x80, 0xe8, 0x87,
  0xbe, 0xe8, 0x88, 0x81, 0xe8, 0x88, 0x82, 0xe8, 0x88, 0x84, 0xe8, 0x87,
  0xac, 0xe8, 0xa1, 0x84, 0xe8, 0x88, 0xa1, 0xe8, 0x88, 0xa2, 0xe8, 0x88,
  0xa3, 0xe8, 0x88, 0xad, 0xe8, 0x88, 0xaf, 0xe8, 0x88, 0xa8, 0xe8, 0x88,
  0xab, 0xe8, 0x88, 0xb8, 0xe8, 0x88, 0xbb, 0xe8, 0x88, 0xb3, 0xe8, 0x88,
  0xb4, 0xe8, 0x88, 0xbe, 0xe8, 0x89, 0x84, 0xe8, 0x89, 0x89, 0xe8, 0x89,
  0x8b, 0xe8, 0x89, 0x8f, 0xe8, 0x89, 0x9a, 0xe8, 0x89, 0x9f, 0xe8, 0x89,
  0xa8, 0xe8, 0xa1, 0xbe, 0xe8, 0xa2, 0x85, 0xe8, 0xa2, 0x88, 0xe8, 0xa3,
  0x98, 0xe8, 0xa3, 0x9f, 0xe8, 0xa5, 0x9e, 0xe7, 0xbe, 0x9d, 0xe7, 0xbe,
  0x9f, 0xe7, 0xbe, 0xa7, 0xe7, 0xbe, 0xaf, 0xe7, 0xbe, 0xb0, 0xe7, 0xbe,
  0xb2, 0xe7, 0xb1, 0xbc, 0xe6, 0x95, 0x89, 0xe7, 0xb2, 0x91, 0xe7, 0xb2,
  0x9d, 0xe7, 0xb2, 0x9c, 0xe7, 0xb2, 0x9e, 0xe7, 0xb2, 0xa2, 0xe7, 0xb2,
  0xb2, 0xe7, 0xb2, 0xbc, 0xe7, 0xb2, 0xbd, 0xe7, 0xb3, 0x81, 0xe7, 0xb3,
  0x87, 0xe7, 0xb3, 0x8c, 0xe7, 0xb3, 0x8d, 0xe7, 0xb3, 0x88, 0xe7, 0xb3,
  0x85, 0xe7, 0xb3, 0x97, 0xe7, 0xb3, 0xa8, 0xe8, 0x89, 0xae, 0xe6, 0x9a,
  0xa8, 0xe7, 0xbe, 0xbf, 0xe7, 0xbf, 0x8e, 0xe7, 0xbf, 0x95, 0xe7, 0xbf,
  0xa5, 0xe7, 0xbf, 0xa1, 0xe7, 0xbf, 0xa6, 0xe7, 0xbf, 0xa9, 0xe7, 0xbf,
  0xae, 0xe7, 0xbf, 0xb3, 0xe7, 0xb3, 0xb8, 0xe7, 0xb5, 0xb7, 0xe7, 0xb6,
  0xa6, 0xe7, 0xb6, 0xae, 0xe7, 0xb9, 0x87, 0xe7, 0xba, 0x9b, 0xe9, 0xba,
  0xb8, 0xe9, 0xba, 0xb4, 0xe8, 0xb5, 0xb3, 0xe8, 0xb6, 0x84, 0xe8, 0xb6,
  0x94, 0xe8, 0xb6, 0x91, 0xe8, 0xb6, 0xb1, 0xe8, 0xb5, 0xa7, 0xe8, 0xb5,
  0xad, 0xe8, 0xb1, 0x87, 0xe8, 0xb1, 0x89, 0xe9, 0x85, 0x8a, 0xe9, 0x85,
  0x90, 0xe9, 0x85, 0x8e, 0xe9, 0x85, 0x8f, 0xe9, 0x85, 0xa4
};

unsigned char gbk_row_0xf5_converted_to_utf8[] = {
  0xe9,
  0xad, 0xbc, 0xe9, 0xad, 0xbd, 0xe9, 0xad, 0xbe, 0xe9, 0xad, 0xbf, 0xe9,
  0xae, 0x80, 0xe9, 0xae, 0x81, 0xe9, 0xae, 0x82, 0xe9, 0xae, 0x83, 0xe9,
  0xae, 0x84, 0xe9, 0xae, 0x85, 0xe9, 0xae, 0x86, 0xe9, 0xae, 0x87, 0xe9,
  0xae, 0x88, 0xe9, 0xae, 0x89, 0xe9, 0xae, 0x8a, 0xe9, 0xae, 0x8b, 0xe9,
  0xae, 0x8c, 0xe9, 0xae, 0x8d, 0xe9, 0xae, 0x8e, 0xe9, 0xae, 0x8f, 0xe9,
  0xae, 0x90, 0xe9, 0xae, 0x91, 0xe9, 0xae, 0x92, 0xe9, 0xae, 0x93, 0xe9,
  0xae, 0x94, 0xe9, 0xae, 0x95, 0xe9, 0xae, 0x96, 0xe9, 0xae, 0x97, 0xe9,
  0xae, 0x98, 0xe9, 0xae, 0x99, 0xe9, 0xae, 0x9a, 0xe9, 0xae, 0x9b, 0xe9,
  0xae, 0x9c, 0xe9, 0xae, 0x9d, 0xe9, 0xae, 0x9e, 0xe9, 0xae, 0x9f, 0xe9,
  0xae, 0xa0, 0xe9, 0xae, 0xa1, 0xe9, 0xae, 0xa2, 0xe9, 0xae, 0xa3, 0xe9,
  0xae, 0xa4, 0xe9, 0xae, 0xa5, 0xe9, 0xae, 0xa6, 0xe9, 0xae, 0xa7, 0xe9,
  0xae, 0xa8, 0xe9, 0xae, 0xa9, 0xe9, 0xae, 0xaa, 0xe9, 0xae, 0xab, 0xe9,
  0xae, 0xac, 0xe9, 0xae, 0xad, 0xe9, 0xae, 0xae, 0xe9, 0xae, 0xaf, 0xe9,
  0xae, 0xb0, 0xe9, 0xae, 0xb1, 0xe9, 0xae, 0xb2, 0xe9, 0xae, 0xb3, 0xe9,
  0xae, 0xb4, 0xe9, 0xae, 0xb5, 0xe9, 0xae, 0xb6, 0xe9, 0xae, 0xb7, 0xe9,
  0xae, 0xb8, 0xe9, 0xae, 0xb9, 0xe9, 0xae, 0xba, 0xe9, 0xae, 0xbb, 0xe9,
  0xae, 0xbc, 0xe9, 0xae, 0xbd, 0xe9, 0xae, 0xbe, 0xe9, 0xae, 0xbf, 0xe9,
  0xaf, 0x80, 0xe9, 0xaf, 0x81, 0xe9, 0xaf, 0x82, 0xe9, 0xaf, 0x83, 0xe9,
  0xaf, 0x84, 0xe9, 0xaf, 0x85, 0xe9, 0xaf, 0x86, 0xe9, 0xaf, 0x87, 0xe9,
  0xaf, 0x88, 0xe9, 0xaf, 0x89, 0xe9, 0xaf, 0x8a, 0xe9, 0xaf, 0x8b, 0xe9,
  0xaf, 0x8c, 0xe9, 0xaf, 0x8d, 0xe9, 0xaf, 0x8e, 0xe9, 0xaf, 0x8f, 0xe9,
  0xaf, 0x90, 0xe9, 0xaf, 0x91, 0xe9, 0xaf, 0x92, 0xe9, 0xaf, 0x93, 0xe9,
  0xaf, 0x94, 0xe9, 0xaf, 0x95, 0xe9, 0xaf, 0x96, 0xe9, 0xaf, 0x97, 0xe9,
  0xaf, 0x98, 0xe9, 0xaf, 0x99, 0xe9, 0xaf, 0x9a, 0xe9, 0xaf, 0x9b, 0xe9,
  0x85, 0xa2, 0xe9, 0x85, 0xa1, 0xe9, 0x85, 0xb0, 0xe9, 0x85, 0xa9, 0xe9,
  0x85, 0xaf, 0xe9, 0x85, 0xbd, 0xe9, 0x85, 0xbe, 0xe9, 0x85, 0xb2, 0xe9,
  0x85, 0xb4, 0xe9, 0x85, 0xb9, 0xe9, 0x86, 0x8c, 0xe9, 0x86, 0x85, 0xe9,
  0x86, 0x90, 0xe9, 0x86, 0x8d, 0xe9, 0x86, 0x91, 0xe9, 0x86, 0xa2, 0xe9,
  0x86, 0xa3, 0xe9, 0x86, 0xaa, 0xe9, 0x86, 0xad, 0xe9, 0x86, 0xae, 0xe9,
  0x86, 0xaf, 0xe9, 0x86, 0xb5, 0xe9, 0x86, 0xb4, 0xe9, 0x86, 0xba, 0xe8,
  0xb1, 0x95, 0xe9, 0xb9, 0xbe, 0xe8, 0xb6, 0xb8, 0xe8, 0xb7, 0xab, 0xe8,
  0xb8, 0x85, 0xe8, 0xb9, 0x99, 0xe8, 0xb9, 0xa9, 0xe8, 0xb6, 0xb5, 0xe8,
  0xb6, 0xbf, 0xe8, 0xb6, 0xbc, 0xe8, 0xb6, 0xba, 0xe8, 0xb7, 0x84, 0xe8,
  0xb7, 0x96, 0xe8, 0xb7, 0x97, 0xe8, 0xb7, 0x9a, 0xe8, 0xb7, 0x9e, 0xe8,
  0xb7, 0x8e, 0xe8, 0xb7, 0x8f, 0xe8, 0xb7, 0x9b, 0xe8, 0xb7, 0x86, 0xe8,
  0xb7, 0xac, 0xe8, 0xb7, 0xb7, 0xe8, 0xb7, 0xb8, 0xe8, 0xb7, 0xa3, 0xe8,
  0xb7, 0xb9, 0xe8, 0xb7, 0xbb, 0xe8, 0xb7, 0xa4, 0xe8, 0xb8, 0x89, 0xe8,
  0xb7, 0xbd, 0xe8, 0xb8, 0x94, 0xe8, 0xb8, 0x9d, 0xe8, 0xb8, 0x9f, 0xe8,
  0xb8, 0xac, 0xe8, 0xb8, 0xae, 0xe8, 0xb8, 0xa3, 0xe8, 0xb8, 0xaf, 0xe8,
  0xb8, 0xba, 0xe8, 0xb9, 0x80, 0xe8, 0xb8, 0xb9, 0xe8, 0xb8, 0xb5, 0xe8,
  0xb8, 0xbd, 0xe8, 0xb8, 0xb1, 0xe8, 0xb9, 0x89, 0xe8, 0xb9, 0x81, 0xe8,
  0xb9, 0x82, 0xe8, 0xb9, 0x91, 0xe8, 0xb9, 0x92, 0xe8, 0xb9, 0x8a, 0xe8,
  0xb9, 0xb0, 0xe8, 0xb9, 0xb6, 0xe8, 0xb9, 0xbc, 0xe8, 0xb9, 0xaf, 0xe8,
  0xb9, 0xb4, 0xe8, 0xba, 0x85, 0xe8, 0xba, 0x8f, 0xe8, 0xba, 0x94, 0xe8,
  0xba, 0x90, 0xe8, 0xba, 0x9c, 0xe8, 0xba, 0x9e, 0xe8, 0xb1, 0xb8, 0xe8,
  0xb2, 0x82, 0xe8, 0xb2, 0x8a, 0xe8, 0xb2, 0x85, 0xe8, 0xb2, 0x98, 0xe8,
  0xb2, 0x94, 0xe6, 0x96, 0x9b, 0xe8, 0xa7, 0x96, 0xe8, 0xa7, 0x9e, 0xe8,
  0xa7, 0x9a, 0xe8, 0xa7, 0x9c
};

unsigned char gbk_row_0xf6_converted_to_utf8[] = {
  0xe9, 0xaf, 0x9c, 0xe9, 0xaf, 0x9d,
  0xe9, 0xaf, 0x9e, 0xe9, 0xaf, 0x9f, 0xe9, 0xaf, 0xa0, 0xe9, 0xaf, 0xa1,
  0xe9, 0xaf, 0xa2, 0xe9, 0xaf, 0xa3, 0xe9, 0xaf, 0xa4, 0xe9, 0xaf, 0xa5,
  0xe9, 0xaf, 0xa6, 0xe9, 0xaf, 0xa7, 0xe9, 0xaf, 0xa8, 0xe9, 0xaf, 0xa9,
  0xe9, 0xaf, 0xaa, 0xe9, 0xaf, 0xab, 0xe9, 0xaf, 0xac, 0xe9, 0xaf, 0xad,
  0xe9, 0xaf, 0xae, 0xe9, 0xaf, 0xaf, 0xe9, 0xaf, 0xb0, 0xe9, 0xaf, 0xb1,
  0xe9, 0xaf, 0xb2, 0xe9, 0xaf, 0xb3, 0xe9, 0xaf, 0xb4, 0xe9, 0xaf, 0xb5,
  0xe9, 0xaf, 0xb6, 0xe9, 0xaf, 0xb7, 0xe9, 0xaf, 0xb8, 0xe9, 0xaf, 0xb9,
  0xe9, 0xaf, 0xba, 0xe9, 0xaf, 0xbb, 0xe9, 0xaf, 0xbc, 0xe9, 0xaf, 0xbd,
  0xe9, 0xaf, 0xbe, 0xe9, 0xaf, 0xbf, 0xe9, 0xb0, 0x80, 0xe9, 0xb0, 0x81,
  0xe9, 0xb0, 0x82, 0xe9, 0xb0, 0x83, 0xe9, 0xb0, 0x84, 0xe9, 0xb0, 0x85,
  0xe9, 0xb0, 0x86, 0xe9, 0xb0, 0x87, 0xe9, 0xb0, 0x88, 0xe9, 0xb0, 0x89,
  0xe9, 0xb0, 0x8a, 0xe9, 0xb0, 0x8b, 0xe9, 0xb0, 0x8c, 0xe9, 0xb0, 0x8d,
  0xe9, 0xb0, 0x8e, 0xe9, 0xb0, 0x8f, 0xe9, 0xb0, 0x90, 0xe9, 0xb0, 0x91,
  0xe9, 0xb0, 0x92, 0xe9, 0xb0, 0x93, 0xe9, 0xb0, 0x94, 0xe9, 0xb0, 0x95,
  0xe9, 0xb0, 0x96, 0xe9, 0xb0, 0x97, 0xe9, 0xb0, 0x98, 0xe9, 0xb0, 0x99,
  0xe9, 0xb0, 0x9a, 0xe9, 0xb0, 0x9b, 0xe9, 0xb0, 0x9c, 0xe9, 0xb0, 0x9d,
  0xe9, 0xb0, 0x9e, 0xe9, 0xb0, 0x9f, 0xe9, 0xb0, 0xa0, 0xe9, 0xb0, 0xa1,
  0xe9, 0xb0, 0xa2, 0xe9, 0xb0, 0xa3, 0xe9, 0xb0, 0xa4, 0xe9, 0xb0, 0xa5,
  0xe9, 0xb0, 0xa6, 0xe9, 0xb0, 0xa7, 0xe9, 0xb0, 0xa8, 0xe9, 0xb0, 0xa9,
  0xe9, 0xb0, 0xaa, 0xe9, 0xb0, 0xab, 0xe9, 0xb0, 0xac, 0xe9, 0xb0, 0xad,
  0xe9, 0xb0, 0xae, 0xe9, 0xb0, 0xaf, 0xe9, 0xb0, 0xb0, 0xe9, 0xb0, 0xb1,
  0xe9, 0xb0, 0xb2, 0xe9, 0xb0, 0xb3, 0xe9, 0xb0, 0xb4, 0xe9, 0xb0, 0xb5,
  0xe9, 0xb0, 0xb6, 0xe9, 0xb0, 0xb7, 0xe9, 0xb0, 0xb8, 0xe9, 0xb0, 0xb9,
  0xe9, 0xb0, 0xba, 0xe9, 0xb0, 0xbb, 0xe8, 0xa7, 0xa5, 0xe8, 0xa7, 0xab,
  0xe8, 0xa7, 0xaf, 0xe8, 0xa8, 0xbe, 0xe8, 0xac, 0xa6, 0xe9, 0x9d, 0x93,
  0xe9, 0x9b, 0xa9, 0xe9, 0x9b, 0xb3, 0xe9, 0x9b, 0xaf, 0xe9, 0x9c, 0x86,
  0xe9, 0x9c, 0x81, 0xe9, 0x9c, 0x88, 0xe9, 0x9c, 0x8f, 0xe9, 0x9c, 0x8e,
  0xe9, 0x9c, 0xaa, 0xe9, 0x9c, 0xad, 0xe9, 0x9c, 0xb0, 0xe9, 0x9c, 0xbe,
  0xe9, 0xbe, 0x80, 0xe9, 0xbe, 0x83, 0xe9, 0xbe, 0x85, 0xe9, 0xbe, 0x86,
  0xe9, 0xbe, 0x87, 0xe9, 0xbe, 0x88, 0xe9, 0xbe, 0x89, 0xe9, 0xbe, 0x8a,
  0xe9, 0xbe, 0x8c, 0xe9, 0xbb, 0xbe, 0xe9, 0xbc, 0x8b, 0xe9, 0xbc, 0x8d,
  0xe9, 0x9a, 0xb9, 0xe9, 0x9a, 0xbc, 0xe9, 0x9a, 0xbd, 0xe9, 0x9b, 0x8e,
  0xe9, 0x9b, 0x92, 0xe7, 0x9e, 0xbf, 0xe9, 0x9b, 0xa0, 0xe9, 0x8a, 0x8e,
  0xe9, 0x8a, 0xae, 0xe9, 0x8b, 0x88, 0xe9, 0x8c, 0xbe, 0xe9, 0x8d, 0xaa,
  0xe9, 0x8f, 0x8a, 0xe9, 0x8e, 0x8f, 0xe9, 0x90, 0xbe, 0xe9, 0x91, 0xab,
  0xe9, 0xb1, 0xbf, 0xe9, 0xb2, 0x82, 0xe9, 0xb2, 0x85, 0xe9, 0xb2, 0x86,
  0xe9, 0xb2, 0x87, 0xe9, 0xb2, 0x88, 0xe7, 0xa8, 0xa3, 0xe9, 0xb2, 0x8b,
  0xe9, 0xb2, 0x8e, 0xe9, 0xb2, 0x90, 0xe9, 0xb2, 0x91, 0xe9, 0xb2, 0x92,
  0xe9, 0xb2, 0x94, 0xe9, 0xb2, 0x95, 0xe9, 0xb2, 0x9a, 0xe9, 0xb2, 0x9b,
  0xe9, 0xb2, 0x9e, 0xe9, 0xb2, 0x9f, 0xe9, 0xb2, 0xa0, 0xe9, 0xb2, 0xa1,
  0xe9, 0xb2, 0xa2, 0xe9, 0xb2, 0xa3, 0xe9, 0xb2, 0xa5, 0xe9, 0xb2, 0xa6,
  0xe9, 0xb2, 0xa7, 0xe9, 0xb2, 0xa8, 0xe9, 0xb2, 0xa9, 0xe9, 0xb2, 0xab,
  0xe9, 0xb2, 0xad, 0xe9, 0xb2, 0xae, 0xe9, 0xb2, 0xb0, 0xe9, 0xb2, 0xb1,
  0xe9, 0xb2, 0xb2, 0xe9, 0xb2, 0xb3, 0xe9, 0xb2, 0xb4, 0xe9, 0xb2, 0xb5,
  0xe9, 0xb2, 0xb6, 0xe9, 0xb2, 0xb7, 0xe9, 0xb2, 0xba, 0xe9, 0xb2, 0xbb,
  0xe9, 0xb2, 0xbc, 0xe9, 0xb2, 0xbd, 0xe9, 0xb3, 0x84, 0xe9, 0xb3, 0x85,
  0xe9, 0xb3, 0x86, 0xe9, 0xb3, 0x87, 0xe9, 0xb3, 0x8a, 0xe9, 0xb3, 0x8b,
};

unsigned char gbk_row_0xf7_converted_to_utf8[] = {
        0xe9, 0xb0, 0xbc, 0xe9, 0xb0, 0xbd, 0xe9, 0xb0, 0xbe, 0xe9, 0xb0,
  0xbf, 0xe9, 0xb1, 0x80, 0xe9, 0xb1, 0x81, 0xe9, 0xb1, 0x82, 0xe9, 0xb1,
  0x83, 0xe9, 0xb1, 0x84, 0xe9, 0xb1, 0x85, 0xe9, 0xb1, 0x86, 0xe9, 0xb1,
  0x87, 0xe9, 0xb1, 0x88, 0xe9, 0xb1, 0x89, 0xe9, 0xb1, 0x8a, 0xe9, 0xb1,
  0x8b, 0xe9, 0xb1, 0x8c, 0xe9, 0xb1, 0x8d, 0xe9, 0xb1, 0x8e, 0xe9, 0xb1,
  0x8f, 0xe9, 0xb1, 0x90, 0xe9, 0xb1, 0x91, 0xe9, 0xb1, 0x92, 0xe9, 0xb1,
  0x93, 0xe9, 0xb1, 0x94, 0xe9, 0xb1, 0x95, 0xe9, 0xb1, 0x96, 0xe9, 0xb1,
  0x97, 0xe9, 0xb1, 0x98, 0xe9, 0xb1, 0x99, 0xe9, 0xb1, 0x9a, 0xe9, 0xb1,
  0x9b, 0xe9, 0xb1, 0x9c, 0xe9, 0xb1, 0x9d, 0xe9, 0xb1, 0x9e, 0xe9, 0xb1,
  0x9f, 0xe9, 0xb1, 0xa0, 0xe9, 0xb1, 0xa1, 0xe9, 0xb1, 0xa2, 0xe9, 0xb1,
  0xa3, 0xe9, 0xb1, 0xa4, 0xe9, 0xb1, 0xa5, 0xe9, 0xb1, 0xa6, 0xe9, 0xb1,
  0xa7, 0xe9, 0xb1, 0xa8, 0xe9, 0xb1, 0xa9, 0xe9, 0xb1, 0xaa, 0xe9, 0xb1,
  0xab, 0xe9, 0xb1, 0xac, 0xe9, 0xb1, 0xad, 0xe9, 0xb1, 0xae, 0xe9, 0xb1,
  0xaf, 0xe9, 0xb1, 0xb0, 0xe9, 0xb1, 0xb1, 0xe9, 0xb1, 0xb2, 0xe9, 0xb1,
  0xb3, 0xe9, 0xb1, 0xb4, 0xe9, 0xb1, 0xb5, 0xe9, 0xb1, 0xb6, 0xe9, 0xb1,
  0xb7, 0xe9, 0xb1, 0xb8, 0xe9, 0xb1, 0xb9, 0xe9, 0xb1, 0xba, 0xe9, 0xb1,
  0xbb, 0xe9, 0xb1, 0xbd, 0xe9, 0xb1, 0xbe, 0xe9, 0xb2, 0x80, 0xe9, 0xb2,
  0x83, 0xe9, 0xb2, 0x84, 0xe9, 0xb2, 0x89, 0xe9, 0xb2, 0x8a, 0xe9, 0xb2,
  0x8c, 0xe9, 0xb2, 0x8f, 0xe9, 0xb2, 0x93, 0xe9, 0xb2, 0x96, 0xe9, 0xb2,
  0x97, 0xe9, 0xb2, 0x98, 0xe9, 0xb2, 0x99, 0xe9, 0xb2, 0x9d, 0xe9, 0xb2,
  0xaa, 0xe9, 0xb2, 0xac, 0xe9, 0xb2, 0xaf, 0xe9, 0xb2, 0xb9, 0xe9, 0xb2,
  0xbe, 0xe9, 0xb2, 0xbf, 0xe9, 0xb3, 0x80, 0xe9, 0xb3, 0x81, 0xe9, 0xb3,
  0x82, 0xe9, 0xb3, 0x88, 0xe9, 0xb3, 0x89, 0xe9, 0xb3, 0x91, 0xe9, 0xb3,
  0x92, 0xe9, 0xb3, 0x9a, 0xe9, 0xb3, 0x9b, 0xe9, 0xb3, 0xa0, 0xe9, 0xb3,
  0xa1, 0xe9, 0xb3, 0x8c, 0xe9, 0xb3, 0x8d, 0xe9, 0xb3, 0x8e, 0xe9, 0xb3,
  0x8f, 0xe9, 0xb3, 0x90, 0xe9, 0xb3, 0x93, 0xe9, 0xb3, 0x94, 0xe9, 0xb3,
  0x95, 0xe9, 0xb3, 0x97, 0xe9, 0xb3, 0x98, 0xe9, 0xb3, 0x99, 0xe9, 0xb3,
  0x9c, 0xe9, 0xb3, 0x9d, 0xe9, 0xb3, 0x9f, 0xe9, 0xb3, 0xa2, 0xe9, 0x9d,
  0xbc, 0xe9, 0x9e, 0x85, 0xe9, 0x9e, 0x91, 0xe9, 0x9e, 0x92, 0xe9, 0x9e,
  0x94, 0xe9, 0x9e, 0xaf, 0xe9, 0x9e, 0xab, 0xe9, 0x9e, 0xa3, 0xe9, 0x9e,
  0xb2, 0xe9, 0x9e, 0xb4, 0xe9, 0xaa, 0xb1, 0xe9, 0xaa, 0xb0, 0xe9, 0xaa,
  0xb7, 0xe9, 0xb9, 0x98, 0xe9, 0xaa, 0xb6, 0xe9, 0xaa, 0xba, 0xe9, 0xaa,
  0xbc, 0xe9, 0xab, 0x81, 0xe9, 0xab, 0x80, 0xe9, 0xab, 0x85, 0xe9, 0xab,
  0x82, 0xe9, 0xab, 0x8b, 0xe9, 0xab, 0x8c, 0xe9, 0xab, 0x91, 0xe9, 0xad,
  0x85, 0xe9, 0xad, 0x83, 0xe9, 0xad, 0x87, 0xe9, 0xad, 0x89, 0xe9, 0xad,
  0x88, 0xe9, 0xad, 0x8d, 0xe9, 0xad, 0x91, 0xe9, 0xa3, 0xa8, 0xe9, 0xa4,
  0x8d, 0xe9, 0xa4, 0xae, 0xe9, 0xa5, 0x95, 0xe9, 0xa5, 0x94, 0xe9, 0xab,
  0x9f, 0xe9, 0xab, 0xa1, 0xe9, 0xab, 0xa6, 0xe9, 0xab, 0xaf, 0xe9, 0xab,
  0xab, 0xe9, 0xab, 0xbb, 0xe9, 0xab, 0xad, 0xe9, 0xab, 0xb9, 0xe9, 0xac,
  0x88, 0xe9, 0xac, 0x8f, 0xe9, 0xac, 0x93, 0xe9, 0xac, 0x9f, 0xe9, 0xac,
  0xa3, 0xe9, 0xba, 0xbd, 0xe9, 0xba, 0xbe, 0xe7, 0xb8, 0xbb, 0xe9, 0xba,
  0x82, 0xe9, 0xba, 0x87, 0xe9, 0xba, 0x88, 0xe9, 0xba, 0x8b, 0xe9, 0xba,
  0x92, 0xe9, 0x8f, 0x96, 0xe9, 0xba, 0x9d, 0xe9, 0xba, 0x9f, 0xe9, 0xbb,
  0x9b, 0xe9, 0xbb, 0x9c, 0xe9, 0xbb, 0x9d, 0xe9, 0xbb, 0xa0, 0xe9, 0xbb,
  0x9f, 0xe9, 0xbb, 0xa2, 0xe9, 0xbb, 0xa9, 0xe9, 0xbb, 0xa7, 0xe9, 0xbb,
  0xa5, 0xe9, 0xbb, 0xaa, 0xe9, 0xbb, 0xaf, 0xe9, 0xbc, 0xa2, 0xe9, 0xbc,
  0xac, 0xe9, 0xbc, 0xaf, 0xe9, 0xbc, 0xb9, 0xe9, 0xbc, 0xb7, 0xe9, 0xbc,
  0xbd, 0xe9, 0xbc, 0xbe, 0xe9, 0xbd, 0x84
};

unsigned char gbk_row_0xf8_converted_to_utf8[] = {
  0xe9, 0xb3, 0xa3, 0xe9, 0xb3, 0xa4, 0xe9, 0xb3, 0xa5, 0xe9, 0xb3, 0xa6,
  0xe9, 0xb3, 0xa7, 0xe9, 0xb3, 0xa8, 0xe9, 0xb3, 0xa9, 0xe9, 0xb3, 0xaa,
  0xe9, 0xb3, 0xab, 0xe9, 0xb3, 0xac, 0xe9, 0xb3, 0xad, 0xe9, 0xb3, 0xae,
  0xe9, 0xb3, 0xaf, 0xe9, 0xb3, 0xb0, 0xe9, 0xb3, 0xb1, 0xe9, 0xb3, 0xb2,
  0xe9, 0xb3, 0xb3, 0xe9, 0xb3, 0xb4, 0xe9, 0xb3, 0xb5, 0xe9, 0xb3, 0xb6,
  0xe9, 0xb3, 0xb7, 0xe9, 0xb3, 0xb8, 0xe9, 0xb3, 0xb9, 0xe9, 0xb3, 0xba,
  0xe9, 0xb3, 0xbb, 0xe9, 0xb3, 0xbc, 0xe9, 0xb3, 0xbd, 0xe9, 0xb3, 0xbe,
  0xe9, 0xb3, 0xbf, 0xe9, 0xb4, 0x80, 0xe9, 0xb4, 0x81, 0xe9, 0xb4, 0x82,
  0xe9, 0xb4, 0x83, 0xe9, 0xb4, 0x84, 0xe9, 0xb4, 0x85, 0xe9, 0xb4, 0x86,
  0xe9, 0xb4, 0x87, 0xe9, 0xb4, 0x88, 0xe9, 0xb4, 0x89, 0xe9, 0xb4, 0x8a,
  0xe9, 0xb4, 0x8b, 0xe9, 0xb4, 0x8c, 0xe9, 0xb4, 0x8d, 0xe9, 0xb4, 0x8e,
  0xe9, 0xb4, 0x8f, 0xe9, 0xb4, 0x90, 0xe9, 0xb4, 0x91, 0xe9, 0xb4, 0x92,
  0xe9, 0xb4, 0x93, 0xe9, 0xb4, 0x94, 0xe9, 0xb4, 0x95, 0xe9, 0xb4, 0x96,
  0xe9, 0xb4, 0x97, 0xe9, 0xb4, 0x98, 0xe9, 0xb4, 0x99, 0xe9, 0xb4, 0x9a,
  0xe9, 0xb4, 0x9b, 0xe9, 0xb4, 0x9c, 0xe9, 0xb4, 0x9d, 0xe9, 0xb4, 0x9e,
  0xe9, 0xb4, 0x9f, 0xe9, 0xb4, 0xa0, 0xe9, 0xb4, 0xa1, 0xe9, 0xb4, 0xa2,
  0xe9, 0xb4, 0xa3, 0xe9, 0xb4, 0xa4, 0xe9, 0xb4, 0xa5, 0xe9, 0xb4, 0xa6,
  0xe9, 0xb4, 0xa7, 0xe9, 0xb4, 0xa8, 0xe9, 0xb4, 0xa9, 0xe9, 0xb4, 0xaa,
  0xe9, 0xb4, 0xab, 0xe9, 0xb4, 0xac, 0xe9, 0xb4, 0xad, 0xe9, 0xb4, 0xae,
  0xe9, 0xb4, 0xaf, 0xe9, 0xb4, 0xb0, 0xe9, 0xb4, 0xb1, 0xe9, 0xb4, 0xb2,
  0xe9, 0xb4, 0xb3, 0xe9, 0xb4, 0xb4, 0xe9, 0xb4, 0xb5, 0xe9, 0xb4, 0xb6,
  0xe9, 0xb4, 0xb7, 0xe9, 0xb4, 0xb8, 0xe9, 0xb4, 0xb9, 0xe9, 0xb4, 0xba,
  0xe9, 0xb4, 0xbb, 0xe9, 0xb4, 0xbc, 0xe9, 0xb4, 0xbd, 0xe9, 0xb4, 0xbe,
  0xe9, 0xb4, 0xbf, 0xe9, 0xb5, 0x80, 0xe9, 0xb5, 0x81, 0xe9, 0xb5, 0x82,
};

unsigned char gbk_row_0xf9_converted_to_utf8[] = {
        0xe9, 0xb5, 0x83, 0xe9, 0xb5, 0x84, 0xe9, 0xb5, 0x85, 0xe9, 0xb5,
  0x86, 0xe9, 0xb5, 0x87, 0xe9, 0xb5, 0x88, 0xe9, 0xb5, 0x89, 0xe9, 0xb5,
  0x8a, 0xe9, 0xb5, 0x8b, 0xe9, 0xb5, 0x8c, 0xe9, 0xb5, 0x8d, 0xe9, 0xb5,
  0x8e, 0xe9, 0xb5, 0x8f, 0xe9, 0xb5, 0x90, 0xe9, 0xb5, 0x91, 0xe9, 0xb5,
  0x92, 0xe9, 0xb5, 0x93, 0xe9, 0xb5, 0x94, 0xe9, 0xb5, 0x95, 0xe9, 0xb5,
  0x96, 0xe9, 0xb5, 0x97, 0xe9, 0xb5, 0x98, 0xe9, 0xb5, 0x99, 0xe9, 0xb5,
  0x9a, 0xe9, 0xb5, 0x9b, 0xe9, 0xb5, 0x9c, 0xe9, 0xb5, 0x9d, 0xe9, 0xb5,
  0x9e, 0xe9, 0xb5, 0x9f, 0xe9, 0xb5, 0xa0, 0xe9, 0xb5, 0xa1, 0xe9, 0xb5,
  0xa2, 0xe9, 0xb5, 0xa3, 0xe9, 0xb5, 0xa4, 0xe9, 0xb5, 0xa5, 0xe9, 0xb5,
  0xa6, 0xe9, 0xb5, 0xa7, 0xe9, 0xb5, 0xa8, 0xe9, 0xb5, 0xa9, 0xe9, 0xb5,
  0xaa, 0xe9, 0xb5, 0xab, 0xe9, 0xb5, 0xac, 0xe9, 0xb5, 0xad, 0xe9, 0xb5,
  0xae, 0xe9, 0xb5, 0xaf, 0xe9, 0xb5, 0xb0, 0xe9, 0xb5, 0xb1, 0xe9, 0xb5,
  0xb2, 0xe9, 0xb5, 0xb3, 0xe9, 0xb5, 0xb4, 0xe9, 0xb5, 0xb5, 0xe9, 0xb5,
  0xb6, 0xe9, 0xb5, 0xb7, 0xe9, 0xb5, 0xb8, 0xe9, 0xb5, 0xb9, 0xe9, 0xb5,
  0xba, 0xe9, 0xb5, 0xbb, 0xe9, 0xb5, 0xbc, 0xe9, 0xb5, 0xbd, 0xe9, 0xb5,
  0xbe, 0xe9, 0xb5, 0xbf, 0xe9, 0xb6, 0x80, 0xe9, 0xb6, 0x81, 0xe9, 0xb6,
  0x82, 0xe9, 0xb6, 0x83, 0xe9, 0xb6, 0x84, 0xe9, 0xb6, 0x85, 0xe9, 0xb6,
  0x86, 0xe9, 0xb6, 0x87, 0xe9, 0xb6, 0x88, 0xe9, 0xb6, 0x89, 0xe9, 0xb6,
  0x8a, 0xe9, 0xb6, 0x8b, 0xe9, 0xb6, 0x8c, 0xe9, 0xb6, 0x8d, 0xe9, 0xb6,
  0x8e, 0xe9, 0xb6, 0x8f, 0xe9, 0xb6, 0x90, 0xe9, 0xb6, 0x91, 0xe9, 0xb6,
  0x92, 0xe9, 0xb6, 0x93, 0xe9, 0xb6, 0x94, 0xe9, 0xb6, 0x95, 0xe9, 0xb6,
  0x96, 0xe9, 0xb6, 0x97, 0xe9, 0xb6, 0x98, 0xe9, 0xb6, 0x99, 0xe9, 0xb6,
  0x9a, 0xe9, 0xb6, 0x9b, 0xe9, 0xb6, 0x9c, 0xe9, 0xb6, 0x9d, 0xe9, 0xb6,
  0x9e, 0xe9, 0xb6, 0x9f, 0xe9, 0xb6, 0xa0, 0xe9, 0xb6, 0xa1, 0xe9, 0xb6,
  0xa2
};

unsigned char gbk_row_0xfa_converted_to_utf8[] = {
  0xe9, 0xb6, 0xa3, 0xe9, 0xb6, 0xa4, 0xe9, 0xb6, 0xa5, 0xe9,
  0xb6, 0xa6, 0xe9, 0xb6, 0xa7, 0xe9, 0xb6, 0xa8, 0xe9, 0xb6, 0xa9, 0xe9,
  0xb6, 0xaa, 0xe9, 0xb6, 0xab, 0xe9, 0xb6, 0xac, 0xe9, 0xb6, 0xad, 0xe9,
  0xb6, 0xae, 0xe9, 0xb6, 0xaf, 0xe9, 0xb6, 0xb0, 0xe9, 0xb6, 0xb1, 0xe9,
  0xb6, 0xb2, 0xe9, 0xb6, 0xb3, 0xe9, 0xb6, 0xb4, 0xe9, 0xb6, 0xb5, 0xe9,
  0xb6, 0xb6, 0xe9, 0xb6, 0xb7, 0xe9, 0xb6, 0xb8, 0xe9, 0xb6, 0xb9, 0xe9,
  0xb6, 0xba, 0xe9, 0xb6, 0xbb, 0xe9, 0xb6, 0xbc, 0xe9, 0xb6, 0xbd, 0xe9,
  0xb6, 0xbe, 0xe9, 0xb6, 0xbf, 0xe9, 0xb7, 0x80, 0xe9, 0xb7, 0x81, 0xe9,
  0xb7, 0x82, 0xe9, 0xb7, 0x83, 0xe9, 0xb7, 0x84, 0xe9, 0xb7, 0x85, 0xe9,
  0xb7, 0x86, 0xe9, 0xb7, 0x87, 0xe9, 0xb7, 0x88, 0xe9, 0xb7, 0x89, 0xe9,
  0xb7, 0x8a, 0xe9, 0xb7, 0x8b, 0xe9, 0xb7, 0x8c, 0xe9, 0xb7, 0x8d, 0xe9,
  0xb7, 0x8e, 0xe9, 0xb7, 0x8f, 0xe9, 0xb7, 0x90, 0xe9, 0xb7, 0x91, 0xe9,
  0xb7, 0x92, 0xe9, 0xb7, 0x93, 0xe9, 0xb7, 0x94, 0xe9, 0xb7, 0x95, 0xe9,
  0xb7, 0x96, 0xe9, 0xb7, 0x97, 0xe9, 0xb7, 0x98, 0xe9, 0xb7, 0x99, 0xe9,
  0xb7, 0x9a, 0xe9, 0xb7, 0x9b, 0xe9, 0xb7, 0x9c, 0xe9, 0xb7, 0x9d, 0xe9,
  0xb7, 0x9e, 0xe9, 0xb7, 0x9f, 0xe9, 0xb7, 0xa0, 0xe9, 0xb7, 0xa1, 0xe9,
  0xb7, 0xa2, 0xe9, 0xb7, 0xa3, 0xe9, 0xb7, 0xa4, 0xe9, 0xb7, 0xa5, 0xe9,
  0xb7, 0xa6, 0xe9, 0xb7, 0xa7, 0xe9, 0xb7, 0xa8, 0xe9, 0xb7, 0xa9, 0xe9,
  0xb7, 0xaa, 0xe9, 0xb7, 0xab, 0xe9, 0xb7, 0xac, 0xe9, 0xb7, 0xad, 0xe9,
  0xb7, 0xae, 0xe9, 0xb7, 0xaf, 0xe9, 0xb7, 0xb0, 0xe9, 0xb7, 0xb1, 0xe9,
  0xb7, 0xb2, 0xe9, 0xb7, 0xb3, 0xe9, 0xb7, 0xb4, 0xe9, 0xb7, 0xb5, 0xe9,
  0xb7, 0xb6, 0xe9, 0xb7, 0xb7, 0xe9, 0xb7, 0xb8, 0xe9, 0xb7, 0xb9, 0xe9,
  0xb7, 0xba, 0xe9, 0xb7, 0xbb, 0xe9, 0xb7, 0xbc, 0xe9, 0xb7, 0xbd, 0xe9,
  0xb7, 0xbe, 0xe9, 0xb7, 0xbf, 0xe9, 0xb8, 0x80, 0xe9, 0xb8, 0x81, 0xe9,
  0xb8, 0x82
};

unsigned char gbk_row_0xfb_converted_to_utf8[] = {
  0xe9, 0xb8, 0x83, 0xe9, 0xb8, 0x84, 0xe9, 0xb8, 0x85,
  0xe9, 0xb8, 0x86, 0xe9, 0xb8, 0x87, 0xe9, 0xb8, 0x88, 0xe9, 0xb8, 0x89,
  0xe9, 0xb8, 0x8a, 0xe9, 0xb8, 0x8b, 0xe9, 0xb8, 0x8c, 0xe9, 0xb8, 0x8d,
  0xe9, 0xb8, 0x8e, 0xe9, 0xb8, 0x8f, 0xe9, 0xb8, 0x90, 0xe9, 0xb8, 0x91,
  0xe9, 0xb8, 0x92, 0xe9, 0xb8, 0x93, 0xe9, 0xb8, 0x94, 0xe9, 0xb8, 0x95,
  0xe9, 0xb8, 0x96, 0xe9, 0xb8, 0x97, 0xe9, 0xb8, 0x98, 0xe9, 0xb8, 0x99,
  0xe9, 0xb8, 0x9a, 0xe9, 0xb8, 0x9b, 0xe9, 0xb8, 0x9c, 0xe9, 0xb8, 0x9d,
  0xe9, 0xb8, 0x9e, 0xe9, 0xb8, 0xa4, 0xe9, 0xb8, 0xa7, 0xe9, 0xb8, 0xae,
  0xe9, 0xb8, 0xb0, 0xe9, 0xb8, 0xb4, 0xe9, 0xb8, 0xbb, 0xe9, 0xb8, 0xbc,
  0xe9, 0xb9, 0x80, 0xe9, 0xb9, 0x8d, 0xe9, 0xb9, 0x90, 0xe9, 0xb9, 0x92,
  0xe9, 0xb9, 0x93, 0xe9, 0xb9, 0x94, 0xe9, 0xb9, 0x96, 0xe9, 0xb9, 0x99,
  0xe9, 0xb9, 0x9d, 0xe9, 0xb9, 0x9f, 0xe9, 0xb9, 0xa0, 0xe9, 0xb9, 0xa1,
  0xe9, 0xb9, 0xa2, 0xe9, 0xb9, 0xa5, 0xe9, 0xb9, 0xae, 0xe9, 0xb9, 0xaf,
  0xe9, 0xb9, 0xb2, 0xe9, 0xb9, 0xb4, 0xe9, 0xb9, 0xb5, 0xe9, 0xb9, 0xb6,
  0xe9, 0xb9, 0xb7, 0xe9, 0xb9, 0xb8, 0xe9, 0xb9, 0xb9, 0xe9, 0xb9, 0xba,
  0xe9, 0xb9, 0xbb, 0xe9, 0xb9, 0xbc, 0xe9, 0xb9, 0xbd, 0xe9, 0xba, 0x80,
  0xe9, 0xba, 0x81, 0xe9, 0xba, 0x83, 0xe9, 0xba, 0x84, 0xe9, 0xba, 0x85,
  0xe9, 0xba, 0x86, 0xe9, 0xba, 0x89, 0xe9, 0xba, 0x8a, 0xe9, 0xba, 0x8c,
  0xe9, 0xba, 0x8d, 0xe9, 0xba, 0x8e, 0xe9, 0xba, 0x8f, 0xe9, 0xba, 0x90,
  0xe9, 0xba, 0x91, 0xe9, 0xba, 0x94, 0xe9, 0xba, 0x95, 0xe9, 0xba, 0x96,
  0xe9, 0xba, 0x97, 0xe9, 0xba, 0x98, 0xe9, 0xba, 0x99, 0xe9, 0xba, 0x9a,
  0xe9, 0xba, 0x9b, 0xe9, 0xba, 0x9c, 0xe9, 0xba, 0x9e, 0xe9, 0xba, 0xa0,
  0xe9, 0xba, 0xa1, 0xe9, 0xba, 0xa2, 0xe9, 0xba, 0xa3, 0xe9, 0xba, 0xa4,
  0xe9, 0xba, 0xa5, 0xe9, 0xba, 0xa7, 0xe9, 0xba, 0xa8, 0xe9, 0xba, 0xa9,
  0xe9, 0xba, 0xaa
};

unsigned char gbk_row_0xfc_converted_to_utf8[] = {
  0xe9, 0xba, 0xab, 0xe9, 0xba, 0xac, 0xe9, 0xba,
  0xad, 0xe9, 0xba, 0xae, 0xe9, 0xba, 0xaf, 0xe9, 0xba, 0xb0, 0xe9, 0xba,
  0xb1, 0xe9, 0xba, 0xb2, 0xe9, 0xba, 0xb3, 0xe9, 0xba, 0xb5, 0xe9, 0xba,
  0xb6, 0xe9, 0xba, 0xb7, 0xe9, 0xba, 0xb9, 0xe9, 0xba, 0xba, 0xe9, 0xba,
  0xbc, 0xe9, 0xba, 0xbf, 0xe9, 0xbb, 0x80, 0xe9, 0xbb, 0x81, 0xe9, 0xbb,
  0x82, 0xe9, 0xbb, 0x83, 0xe9, 0xbb, 0x85, 0xe9, 0xbb, 0x86, 0xe9, 0xbb,
  0x87, 0xe9, 0xbb, 0x88, 0xe9, 0xbb, 0x8a, 0xe9, 0xbb, 0x8b, 0xe9, 0xbb,
  0x8c, 0xe9, 0xbb, 0x90, 0xe9, 0xbb, 0x92, 0xe9, 0xbb, 0x93, 0xe9, 0xbb,
  0x95, 0xe9, 0xbb, 0x96, 0xe9, 0xbb, 0x97, 0xe9, 0xbb, 0x99, 0xe9, 0xbb,
  0x9a, 0xe9, 0xbb, 0x9e, 0xe9, 0xbb, 0xa1, 0xe9, 0xbb, 0xa3, 0xe9, 0xbb,
  0xa4, 0xe9, 0xbb, 0xa6, 0xe9, 0xbb, 0xa8, 0xe9, 0xbb, 0xab, 0xe9, 0xbb,
  0xac, 0xe9, 0xbb, 0xad, 0xe9, 0xbb, 0xae, 0xe9, 0xbb, 0xb0, 0xe9, 0xbb,
  0xb1, 0xe9, 0xbb, 0xb2, 0xe9, 0xbb, 0xb3, 0xe9, 0xbb, 0xb4, 0xe9, 0xbb,
  0xb5, 0xe9, 0xbb, 0xb6, 0xe9, 0xbb, 0xb7, 0xe9, 0xbb, 0xb8, 0xe9, 0xbb,
  0xba, 0xe9, 0xbb, 0xbd, 0xe9, 0xbb, 0xbf, 0xe9, 0xbc, 0x80, 0xe9, 0xbc,
  0x81, 0xe9, 0xbc, 0x82, 0xe9, 0xbc, 0x83, 0xe9, 0xbc, 0x84, 0xe9, 0xbc,
  0x85, 0xe9, 0xbc, 0x86, 0xe9, 0xbc, 0x87, 0xe9, 0xbc, 0x88, 0xe9, 0xbc,
  0x89, 0xe9, 0xbc, 0x8a, 0xe9, 0xbc, 0x8c, 0xe9, 0xbc, 0x8f, 0xe9, 0xbc,
  0x91, 0xe9, 0xbc, 0x92, 0xe9, 0xbc, 0x94, 0xe9, 0xbc, 0x95, 0xe9, 0xbc,
  0x96, 0xe9, 0xbc, 0x98, 0xe9, 0xbc, 0x9a, 0xe9, 0xbc, 0x9b, 0xe9, 0xbc,
  0x9c, 0xe9, 0xbc, 0x9d, 0xe9, 0xbc, 0x9e, 0xe9, 0xbc, 0x9f, 0xe9, 0xbc,
  0xa1, 0xe9, 0xbc, 0xa3, 0xe9, 0xbc, 0xa4, 0xe9, 0xbc, 0xa5, 0xe9, 0xbc,
  0xa6, 0xe9, 0xbc, 0xa7, 0xe9, 0xbc, 0xa8, 0xe9, 0xbc, 0xa9, 0xe9, 0xbc,
  0xaa, 0xe9, 0xbc, 0xab, 0xe9, 0xbc, 0xad, 0xe9, 0xbc, 0xae, 0xe9, 0xbc,
  0xb0, 0xe9, 0xbc, 0xb1
};

unsigned char gbk_row_0xfd_converted_to_utf8[] = {
  0xe9, 0xbc, 0xb2, 0xe9, 0xbc, 0xb3, 0xe9,
  0xbc, 0xb4, 0xe9, 0xbc, 0xb5, 0xe9, 0xbc, 0xb6, 0xe9, 0xbc, 0xb8, 0xe9,
  0xbc, 0xba, 0xe9, 0xbc, 0xbc, 0xe9, 0xbc, 0xbf, 0xe9, 0xbd, 0x80, 0xe9,
  0xbd, 0x81, 0xe9, 0xbd, 0x82, 0xe9, 0xbd, 0x83, 0xe9, 0xbd, 0x85, 0xe9,
  0xbd, 0x86, 0xe9, 0xbd, 0x87, 0xe9, 0xbd, 0x88, 0xe9, 0xbd, 0x89, 0xe9,
  0xbd, 0x8a, 0xe9, 0xbd, 0x8b, 0xe9, 0xbd, 0x8c, 0xe9, 0xbd, 0x8d, 0xe9,
  0xbd, 0x8e, 0xe9, 0xbd, 0x8f, 0xe9, 0xbd, 0x92, 0xe9, 0xbd, 0x93, 0xe9,
  0xbd, 0x94, 0xe9, 0xbd, 0x95, 0xe9, 0xbd, 0x96, 0xe9, 0xbd, 0x97, 0xe9,
  0xbd, 0x98, 0xe9, 0xbd, 0x99, 0xe9, 0xbd, 0x9a, 0xe9, 0xbd, 0x9b, 0xe9,
  0xbd, 0x9c, 0xe9, 0xbd, 0x9d, 0xe9, 0xbd, 0x9e, 0xe9, 0xbd, 0x9f, 0xe9,
  0xbd, 0xa0, 0xe9, 0xbd, 0xa1, 0xe9, 0xbd, 0xa2, 0xe9, 0xbd, 0xa3, 0xe9,
  0xbd, 0xa4, 0xe9, 0xbd, 0xa5, 0xe9, 0xbd, 0xa6, 0xe9, 0xbd, 0xa7, 0xe9,
  0xbd, 0xa8, 0xe9, 0xbd, 0xa9, 0xe9, 0xbd, 0xaa, 0xe9, 0xbd, 0xab, 0xe9,
  0xbd, 0xac, 0xe9, 0xbd, 0xad, 0xe9, 0xbd, 0xae, 0xe9, 0xbd, 0xaf, 0xe9,
  0xbd, 0xb0, 0xe9, 0xbd, 0xb1, 0xe9, 0xbd, 0xb2, 0xe9, 0xbd, 0xb3, 0xe9,
  0xbd, 0xb4, 0xe9, 0xbd, 0xb5, 0xe9, 0xbd, 0xb6, 0xe9, 0xbd, 0xb7, 0xe9,
  0xbd, 0xb8, 0xe9, 0xbd, 0xb9, 0xe9, 0xbd, 0xba, 0xe9, 0xbd, 0xbb, 0xe9,
  0xbd, 0xbc, 0xe9, 0xbd, 0xbd, 0xe9, 0xbd, 0xbe, 0xe9, 0xbe, 0x81, 0xe9,
  0xbe, 0x82, 0xe9, 0xbe, 0x8d, 0xe9, 0xbe, 0x8e, 0xe9, 0xbe, 0x8f, 0xe9,
  0xbe, 0x90, 0xe9, 0xbe, 0x91, 0xe9, 0xbe, 0x92, 0xe9, 0xbe, 0x93, 0xe9,
  0xbe, 0x94, 0xe9, 0xbe, 0x95, 0xe9, 0xbe, 0x96, 0xe9, 0xbe, 0x97, 0xe9,
  0xbe, 0x98, 0xe9, 0xbe, 0x9c, 0xe9, 0xbe, 0x9d, 0xe9, 0xbe, 0x9e, 0xe9,
  0xbe, 0xa1, 0xe9, 0xbe, 0xa2, 0xe9, 0xbe, 0xa3, 0xe9, 0xbe, 0xa4, 0xe9,
  0xbe, 0xa5, 0xef, 0xa4, 0xac, 0xef, 0xa5, 0xb9, 0xef, 0xa6, 0x95, 0xef,
  0xa7, 0xa7, 0xef, 0xa7, 0xb1
};

unsigned char gbk_row_0xfe_converted_to_utf8[] = {
  0xef, 0xa8, 0x8c, 0xef, 0xa8, 0x8d, 0xef, 0xa8, 0x8e, 0xef, 0xa8, 0x8f,
  0xef, 0xa8, 0x91, 0xef, 0xa8, 0x93, 0xef, 0xa8, 0x94, 0xef, 0xa8, 0x98,
  0xef, 0xa8, 0x9f, 0xef, 0xa8, 0xa0, 0xef, 0xa8, 0xa1, 0xef, 0xa8, 0xa3,
  0xef, 0xa8, 0xa4, 0xef, 0xa8, 0xa7, 0xef, 0xa8, 0xa8, 0xef, 0xa8, 0xa9,
  0xe2, 0xba, 0x81, 0xe2, 0xba, 0x84, 0xe3, 0x91, 0xb3, 0xe3, 0x91, 0x87,
  0xe2, 0xba, 0x88, 0xe2, 0xba, 0x8b, 0xe3, 0x96, 0x9e, 0xe3, 0x98, 0x9a,
  0xe3, 0x98, 0x8e, 0xe2, 0xba, 0x8c, 0xe2, 0xba, 0x97, 0xe3, 0xa5, 0xae,
  0xe3, 0xa4, 0x98, 0xe3, 0xa7, 0x8f, 0xe3, 0xa7, 0x9f, 0xe3, 0xa9, 0xb3,
  0xe3, 0xa7, 0x90, 0xe3, 0xad, 0x8e, 0xe3, 0xb1, 0xae, 0xe3, 0xb3, 0xa0,
  0xe2, 0xba, 0xa7, 0xe2, 0xba, 0xaa, 0xe4, 0x81, 0x96, 0xe4, 0x85, 0x9f,
  0xe2, 0xba, 0xae, 0xe4, 0x8c, 0xb7, 0xe2, 0xba, 0xb3, 0xe2, 0xba, 0xb6,
  0xe2, 0xba, 0xb7, 0xe4, 0x8e, 0xb1, 0xe4, 0x8e, 0xac, 0xe2, 0xba, 0xbb,
  0xe4, 0x8f, 0x9d, 0xe4, 0x93, 0x96, 0xe4, 0x99, 0xa1, 0xe4, 0x99, 0x8c,
  0xe4, 0x9c, 0xa3, 0xe4, 0x9c, 0xa9, 0xe4, 0x9d, 0xbc, 0xe4, 0x9e, 0x8d,
  0xe2, 0xbb, 0x8a, 0xe4, 0xa5, 0x87, 0xe4, 0xa5, 0xba, 0xe4, 0xa5, 0xbd,
  0xe4, 0xa6, 0x82, 0xe4, 0xa6, 0x83, 0xe4, 0xa6, 0x85, 0xe4, 0xa6, 0x86,
  0xe4, 0xa6, 0x9f, 0xe4, 0xa6, 0x9b, 0xe4, 0xa6, 0xb7, 0xe4, 0xa6, 0xb6,
  0xe4, 0xb2, 0xa3, 0xe4, 0xb2, 0x9f, 0xe4, 0xb2, 0xa0, 0xe4, 0xb2, 0xa1,
  0xe4, 0xb1, 0xb7, 0xe4, 0xb2, 0xa2, 0xe4, 0xb4, 0x93, 0xe4, 0xb4, 0x94,
  0xe4, 0xb4, 0x95, 0xe4, 0xb4, 0x96, 0xe4, 0xb4, 0x97, 0xe4, 0xb4, 0x98,
  0xe4, 0xb4, 0x99, 0xe4, 0xb6, 0xae
};

unsigned char gbk_row_0xa8_alternative[] = {
  0xa8, 0x40, 0xa8, 0x41, 0xa8, 0x42, 0xa8, 0x43, 0xa1, 0xaa, 0xa8, 0x45,
  0xa8, 0x46, 0xa8, 0x47, 0xa8, 0x48, 0xa8, 0x49, 0xa8, 0x4a, 0xa8, 0x4b,
  0xa8, 0x4c, 0xa8, 0x4d, 0xa8, 0x4e, 0xa8, 0x4f, 0xa8, 0x50, 0xa8, 0x51,
  0xa8, 0x52, 0xa8, 0x53, 0xa8, 0x54, 0xa8, 0x55, 0xa8, 0x56, 0xa8, 0x57,
  0xa8, 0x58, 0xa8, 0x59, 0xa8, 0x5a, 0xa8, 0x5b, 0xa8, 0x5c, 0xa8, 0x5d,
  0xa8, 0x5e, 0xa8, 0x5f, 0xa8, 0x60, 0xa8, 0x61, 0xa8, 0x62, 0xa8, 0x63,
  0xa8, 0x64, 0xa8, 0x65, 0xa8, 0x66, 0xa8, 0x67, 0xa8, 0x68, 0xa8, 0x69,
  0xa8, 0x6a, 0xa8, 0x6b, 0xa8, 0x6c, 0xa8, 0x6d, 0xa8, 0x6e, 0xa8, 0x6f,
  0xa8, 0x70, 0xa8, 0x71, 0xa8, 0x72, 0xa8, 0x73, 0xa8, 0x74, 0xa8, 0x75,
  0xa8, 0x76, 0xa8, 0x77, 0xa8, 0x78, 0xa8, 0x79, 0xa8, 0x7a, 0xa8, 0x7b,
  0xa8, 0x7c, 0xa8, 0x7d, 0xa8, 0x7e, 0xa8, 0x80, 0xa8, 0x81, 0xa8, 0x82,
  0xa8, 0x83, 0xa8, 0x84, 0xa8, 0x85, 0xa8, 0x86, 0xa8, 0x87, 0xa8, 0x88,
  0xa8, 0x89, 0xa8, 0x8a, 0xa8, 0x8b, 0xa8, 0x8c, 0xa8, 0x8d, 0xa8, 0x8e,
  0xa8, 0x8f, 0xa8, 0x90, 0xa8, 0x91, 0xa8, 0x92, 0xa8, 0x93, 0xa8, 0x94,
  0xa8, 0x95, 0xa8, 0xa1, 0xa8, 0xa2, 0xa8, 0xa3, 0xa8, 0xa4, 0xa8, 0xa5,
  0xa8, 0xa6, 0xa8, 0xa7, 0xa8, 0xa8, 0xa8, 0xa9, 0xa8, 0xaa, 0xa8, 0xab,
  0xa8, 0xac, 0xa8, 0xad, 0xa8, 0xae, 0xa8, 0xaf, 0xa8, 0xb0, 0xa8, 0xb1,
  0xa8, 0xb2, 0xa8, 0xb3, 0xa8, 0xb4, 0xa8, 0xb5, 0xa8, 0xb6, 0xa8, 0xb7,
  0xa8, 0xb8, 0xa8, 0xb9, 0xa8, 0xba, 0xa8, 0xc5, 0xa8, 0xc6, 0xa8, 0xc7,
  0xa8, 0xc8, 0xa8, 0xc9, 0xa8, 0xca, 0xa8, 0xcb, 0xa8, 0xcc, 0xa8, 0xcd,
  0xa8, 0xce, 0xa8, 0xcf, 0xa8, 0xd0, 0xa8, 0xd1, 0xa8, 0xd2, 0xa8, 0xd3,
  0xa8, 0xd4, 0xa8, 0xd5, 0xa8, 0xd6, 0xa8, 0xd7, 0xa8, 0xd8, 0xa8, 0xd9,
  0xa8, 0xda, 0xa8, 0xdb, 0xa8, 0xdc, 0xa8, 0xdd, 0xa8, 0xde, 0xa8, 0xdf,
  0xa8, 0xe0, 0xa8, 0xe1, 0xa8, 0xe2, 0xa8, 0xe3, 0xa8, 0xe4, 0xa8, 0xe5,
  0xa8, 0xe6, 0xa8, 0xe7, 0xa8, 0xe8, 0xa8, 0xe9
};

unsigned char gbk_row_0xa9_alternative[] = {
  0xa9, 0x40, 0xa9, 0x41, 0xa9, 0x42, 0xa9, 0x43, 0xa9, 0x44, 0xa9, 0x45,
  0xa9, 0x46, 0xa9, 0x47, 0xa9, 0x48, 0xa9, 0x49, 0xa9, 0x4a, 0xa9, 0x4b,
  0xa9, 0x4c, 0xa9, 0x4d, 0xa9, 0x4e, 0xa9, 0x4f, 0xa9, 0x50, 0xa9, 0x51,
  0xa9, 0x52, 0xa9, 0x53, 0xa9, 0x54, 0xa9, 0x55, 0xa9, 0xd6, 0xa9, 0xd7,
  0xa9, 0x59, 0xa9, 0x5a, 0xa9, 0x5c, 0xa9, 0x60, 0xa9, 0x61, 0xa9, 0x62,
  0xa9, 0x63, 0xa9, 0x64, 0xa9, 0x65, 0xa9, 0x66, 0xa9, 0x67, 0xa9, 0x68,
  0xa9, 0x69, 0xa9, 0x6a, 0xa9, 0x6b, 0xa9, 0x6c, 0xa9, 0x6d, 0xa9, 0x6e,
  0xa9, 0x6f, 0xa9, 0x70, 0xa9, 0x71, 0xa9, 0x72, 0xa9, 0x73, 0xa9, 0x74,
  0xa9, 0x75, 0xa9, 0x76, 0xa9, 0x77, 0xa9, 0x78, 0xa9, 0x79, 0xa9, 0x7a,
  0xa9, 0x7b, 0xa9, 0x7c, 0xa9, 0x7d, 0xa9, 0x7e, 0xa9, 0x80, 0xa9, 0x81,
  0xa9, 0x82, 0xa9, 0x83, 0xa9, 0x84, 0xa9, 0x85, 0xa9, 0x86, 0xa9, 0x87,
  0xa9, 0x88, 0xa9, 0xa4, 0xa9, 0xa5, 0xa9, 0xa6, 0xa9, 0xa7, 0xa9, 0xa8,
  0xa9, 0xa9, 0xa9, 0xaa, 0xa9, 0xab, 0xa9, 0xac, 0xa9, 0xad, 0xa9, 0xae,
  0xa9, 0xaf, 0xa9, 0xb0, 0xa9, 0xb1, 0xa9, 0xb2, 0xa9, 0xb3, 0xa9, 0xb4,
  0xa9, 0xb5, 0xa9, 0xb6, 0xa9, 0xb7, 0xa9, 0xb8, 0xa9, 0xb9, 0xa9, 0xba,
  0xa9, 0xbb, 0xa9, 0xbc, 0xa9, 0xbd, 0xa9, 0xbe, 0xa9, 0xbf, 0xa9, 0xc0,
  0xa9, 0xc1, 0xa9, 0xc2, 0xa9, 0xc3, 0xa9, 0xc4, 0xa9, 0xc5, 0xa9, 0xc6,
  0xa9, 0xc7, 0xa9, 0xc8, 0xa9, 0xc9, 0xa9, 0xca, 0xa9, 0xcb, 0xa9, 0xcc,
  0xa9, 0xcd, 0xa9, 0xce, 0xa9, 0xcf, 0xa9, 0xd0, 0xa9, 0xd1, 0xa9, 0xd2,
  0xa9, 0xd3, 0xa9, 0xd4, 0xa9, 0xd5, 0xa9, 0xd6, 0xa9, 0xd7, 0xa9, 0xd8,
  0xa9, 0xd9, 0xa9, 0xda, 0xa9, 0xdb, 0xa9, 0xdc, 0xa9, 0xdd, 0xa9, 0xde,
  0xa9, 0xdf, 0xa9, 0xe0, 0xa9, 0xe1, 0xa9, 0xe2, 0xa9, 0xe3, 0xa9, 0xe4,
  0xa9, 0xe5, 0xa9, 0xe6, 0xa9, 0xe7, 0xa9, 0xe8, 0xa9, 0xe9, 0xa9, 0xea,
  0xa9, 0xeb, 0xa9, 0xec, 0xa9, 0xed, 0xa9, 0xee, 0xa9, 0xef
};

unsigned char gbk_row_0xfe_alternative[] = {
  0xfe, 0x40, 0xfe, 0x41, 0xfe, 0x42, 0xfe, 0x43, 0xfe, 0x44, 0xfe, 0x45,
  0xfe, 0x46, 0xfe, 0x47, 0xfe, 0x48, 0xfe, 0x49, 0xfe, 0x4a, 0xfe, 0x4b,
  0xfe, 0x4c, 0xfe, 0x4d, 0xfe, 0x4e, 0xfe, 0x4f, 0xfe, 0xd0, 0xfe, 0xd4,
  0xfe, 0xd5, 0xfe, 0xd6, 0xfe, 0xd7, 0xfe, 0xd8, 0xfe, 0xda, 0xfe, 0xdb,
  0xfe, 0xdc, 0xfe, 0xdd, 0xfe, 0xde, 0xfe, 0xdf, 0xfe, 0xe0, 0xfe, 0xe2,
  0xfe, 0xe3, 0xfe, 0xe4, 0xfe, 0xe5, 0xfe, 0xe8, 0xfe, 0xe9, 0xfe, 0xea,
  0xfe, 0xeb, 0xfe, 0xee, 0xfe, 0xef, 0xfe, 0xf0, 0xfe, 0xf1, 0xfe, 0xf2,
  0xfe, 0xf3, 0xfe, 0xf4, 0xfe, 0xf5, 0xfe, 0xf7, 0xfe, 0xf8, 0xfe, 0xf9,
  0xfe, 0xfa, 0xfe, 0xfb, 0xfe, 0xfc, 0xfe, 0xfd, 0xfe, 0x80, 0xfe, 0x81,
  0xfe, 0x82, 0xfe, 0x83, 0xfe, 0x84, 0xfe, 0x85, 0xfe, 0x86, 0xfe, 0x87,
  0xfe, 0x88, 0xfe, 0x89, 0xfe, 0x8a, 0xfe, 0x8b, 0xfe, 0x8c, 0xfe, 0x8d,
  0xfe, 0x8e, 0xfe, 0x8f, 0xfe, 0x92, 0xfe, 0x93, 0xfe, 0x94, 0xfe, 0x95,
  0xfe, 0x96, 0xfe, 0x97, 0xfe, 0x98, 0xfe, 0x99, 0xfe, 0x9a, 0xfe, 0x9b,
  0xfe, 0x9c, 0xfe, 0x9d, 0xfe, 0x9e, 0xfe, 0x9f
};

#endif
