/*
 * This file is part of sh4lt.
 *
 * sh4lt is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * sh4lt is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with sh4lt.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "./key-val-serializer.hpp"
#include <iostream>
#include <iterator>
#include <string>

namespace sh4lt::infotree::keyval {

using BasicSerializerData = struct basic_serializer_data_t {
  std::vector<std::string> path_{};
  std::string result_{};
};

auto path_to_string(const std::vector<std::string>& path) -> std::string {
  std::stringstream result;
  std::copy(path.begin(), path.end(), std::ostream_iterator<std::string>(result, "."));
  return result.str();
}

void on_visiting_node(const std::string& key, const InfoTree::ptrc node, bool, BasicSerializerData* data) {
  data->path_.push_back(key);
  const auto& value = node->read_data();
  if (!value.empty())
    data->result_.append("." + infotree::keyval::path_to_string(data->path_) + " " +
                         Any::to_string(value) + "\n");
}

void on_node_visited(const std::string&, const InfoTree::ptrc, bool, BasicSerializerData* data) {
  data->path_.pop_back();
}

auto serialize(InfoTree::ptrc tree) -> std::string {
  BasicSerializerData data;
  InfoTree::preorder_tree_walk(tree,
       [&data](
          const std::string& key, InfoTree::ptrc node, bool is_array_element) {
        on_visiting_node(key, node, is_array_element, &data);
        return true; // always true because we want preorder_tree_walk to continue visit siblings
      },
      [&data](const std::string& key, InfoTree::ptrc node, bool is_array_element) {
        on_node_visited(key, node, is_array_element, &data);
        return true; // always true because we want preorder_tree_walk to continue visit siblings
      });
  return data.result_;
}

auto deserialize(const std::string& serialized) -> InfoTree::ptr {
  InfoTree::ptr tree = InfoTree::make();
  std::istringstream ss(serialized);
  std::string line ;
  while (std::getline(ss, line)) {
    std::istringstream line_ss(line);
    std::string absolute_key;
    while (std::getline(line_ss, absolute_key, ' ') && absolute_key.empty()) {
    }
    std::string value;
    while (std::getline(line_ss, value, ' ') && value.empty()) {
    }
    // getting the rest of the value, if it contains space in it
    std::string val_cont;
    std::getline(line_ss, val_cont);
    if (!val_cont.empty()) value += ' ' + val_cont;
    if (!absolute_key.empty() && !value.empty()) tree->graft(absolute_key, InfoTree::make(value));
  }
  return tree;
}

}  // namespace sh4lt
