/*
 * This file is part of libsh4lt.
 *
 * libsh4lt is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef __SH4LT_ANY_DEFS_H__
#define __SH4LT_ANY_DEFS_H__

#include "./serialize-string.hpp"

namespace sh4lt {
template <class T>
using StorageType = typename std::decay<T>::type;

enum AnyCategory { OTHER, BOOLEAN, INTEGRAL, FLOATING_POINT, NONE };

enum AnyArithmeticType {
    NOT_DEFINED,
    INT,
    SHORT,
    LONG,
    LONG_LONG,
    UNSIGNED_INT,
    UNSIGNED_SHORT,
    UNSIGNED_LONG,
    UNSIGNED_LONG_LONG,
    DOUBLE,
    FLOAT,
    LONG_DOUBLE
};

struct AnyValueBase {
    virtual ~AnyValueBase() {}
    virtual AnyValueBase* clone() const = 0;
    virtual std::string to_string() const = 0;
    AnyCategory category_{AnyCategory::NONE};
    AnyArithmeticType arithmetic_type_{AnyArithmeticType::NOT_DEFINED};
};

template <typename T>
struct AnyValueDerived : AnyValueBase {
    template <typename U>
    AnyValueDerived(U&& value,
                    AnyCategory category = AnyCategory::NONE,
                    AnyArithmeticType arithmetic_type = AnyArithmeticType::NOT_DEFINED)
            : value_(std::forward<U>(value)) {
      category_ = category;
      arithmetic_type_ = arithmetic_type;
    }

    T value_;

    AnyValueBase* clone() const {
      return new AnyValueDerived<T>(value_, category_, arithmetic_type_);
    }

    std::string to_string() const { return serialize::apply<T>(value_); }
};

template <>
struct AnyValueDerived<std::nullptr_t> : AnyValueBase {
    template <typename U>
    AnyValueDerived(U&& value) : value_(std::forward<U>(value)) {}
    std::nullptr_t value_;
    AnyValueBase* clone() const { return new AnyValueDerived<std::nullptr_t>(value_); }
    std::string to_string() const { return std::string("null"); }
};

// this is for Any with complex value, where default serialization will
// be implemented as follow
template <typename T>
struct DefaultSerializable {
    virtual ~DefaultSerializable() {}
    template <typename U>
    friend std::ostream& operator<<(std::ostream& os, const DefaultSerializable<U>&) {
      os << "not serializable";
      return os;
    }
};

} // namespace sh4lt
#endif
