/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#include "./writer.hpp"
#include "./ipcs/reader.hpp"
#include <cstring> // memcpy
#include <functional> // std::hash

namespace sh4lt {

Writer::Writer(const ShType& data_descr,
               size_t memsize,
               logger::Logger::ptr log,
               UnixSocketProtocol::ServerSide::onClientConnect on_client_connect,
               UnixSocketProtocol::ServerSide::onClientDisconnect on_client_disconnect,
               mode_t unix_permission)
    : path_(ShType::get_path(data_descr.label(), data_descr.group())),
      connect_data_(memsize, ShType::serialize(data_descr)),
      proto_(on_client_connect, on_client_disconnect, [this]() { return this->connect_data_; }),
      srv_(std::make_unique<UnixSocketServer>(
          path_,
          &proto_,
          log.get(),
          [&](int) { sem_->cancel_commited_reader(); },
          unix_permission)),
      shm_(std::make_unique<sysVShm>(ftok(path_.c_str(), 'n'),
                                     memsize,
                                     log.get(),
                                     /*owner = */ true,
                                     unix_permission)),
      sem_(std::make_unique<sysVSem>(
          ftok(path_.c_str(), 'm'), log.get(), /*owner = */ true, unix_permission)),
      log_(log),
      alloc_size_(memsize) {
  if (!data_descr) {
    log_->error("Shtype is not valid");
    for (const std::string& it: data_descr.msgs()) {
      log_->error(it.c_str());
    }
    is_valid_ = false;
    return;
  }
  if(path_.empty()) {
    log_->error(ShType::get_log_for_path_issue().c_str());
    return;
  }
  if (!(*srv_.get()) || !(*shm_.get()) || !(*sem_.get())) {
    sem_.reset();
    shm_.reset();
    srv_.reset();
    // checking if a server is responding with at this sh4lt path
    bool can_read = false;
    {
      log_->debug("checking if a sh4lt having the same path is already active");
      Reader inspector(path_, nullptr, nullptr, nullptr, log_.get());
      can_read = static_cast<bool>(inspector);
    }
    if (!can_read) {
      log_->debug("writer detected a dead Sh4lt, will clean and retry");
      force_sockserv_cleaning(path_, log.get());
      srv_ = std::make_unique<UnixSocketServer>(
          path_, &proto_, log.get(), [&](int) { sem_->cancel_commited_reader(); }, unix_permission);
      force_shm_cleaning(ftok(path_.c_str(), 'n'), log.get());
      shm_ = std::make_unique<sysVShm>(ftok(path_.c_str(), 'n'),
                                       memsize,
                                       log.get(),
                                       /* owner = */ true,
                                       unix_permission);
      force_semaphore_cleaning(ftok(path_.c_str(), 'm'), log.get());
      sem_ = std::make_unique<sysVSem>(
          ftok(path_.c_str(), 'm'), log.get(), /* owner = */ true, unix_permission);
      is_valid_ = (*srv_.get()) && (*shm_.get()) && (*sem_.get());
    } else {
      log_->error("an other writer is using the same path");
      is_valid_ = false;
    }
  }
  if (!is_valid_) {
    log_->warning("writer failled initialization");
    return;
  }
  srv_->start_serving();
  log_->debug("writer initialized");
}

bool Writer::copy_to_shm(const void* data, size_t size, int64_t buffer_date, int64_t buffer_duration) {
  auto time = time_.update(buffer_date, buffer_duration);
  bool res = true;
  if (nullptr == sem_) {
    log_->warning("semaphore is not initialized");
    return false;
  }

  if (!(*sem_)) {
    log_->warning("semaphore was not correctly initialized");
    return false;
  }
  {
    WriteLock wlock(sem_.get());
    if (size > connect_data_.shm_size_) {
      log_->debug("resizing sh4lt (%) from % bytes to % bytes",
                  path_,
                  std::to_string(connect_data_.shm_size_),
                  std::to_string(size));
      shm_.reset();
      shm_ =
          std::make_unique<sysVShm>(ftok(path_.c_str(), 'n'), size, log_.get(), /*owner = */ true);
      connect_data_.shm_size_ = size;
      alloc_size_ = size;
      if (!shm_) {
        log_->error("resizing shared memory failed");
        return false;
      }
   }
   auto num_readers = srv_->notify_update(size, &time);
   if (0 < num_readers) {
     wlock.commit_readers(num_readers);
    }
    auto dest = shm_->get_mem();
    if (dest != std::memcpy(dest, data, size)) res = false;
  }  // release wlock & lock
  return res;
}

auto Writer::get_one_write_access() -> std::unique_ptr<OneWriteAccess> {
  return std::unique_ptr<OneWriteAccess>(
      new OneWriteAccess(this, sem_.get(), shm_->get_mem(), srv_.get(), log_));
}

auto Writer::get_one_write_access_ptr() -> OneWriteAccess* {
  return new OneWriteAccess(this, sem_.get(), shm_->get_mem(), srv_.get(), log_);
}

auto Writer::get_one_write_access_resize(size_t new_size) -> std::unique_ptr<OneWriteAccess> {
  auto res = std::unique_ptr<OneWriteAccess>(
      new OneWriteAccess(this, sem_.get(), nullptr, srv_.get(), log_));
  if (shm_->get_size() != new_size) {
    log_->debug("resizing sh4lt (%) from % bytes to % bytes",
                path_,
                std::to_string(connect_data_.shm_size_),
                std::to_string(new_size));
    shm_.reset();
    shm_ = std::make_unique<sysVShm>(
        ftok(path_.c_str(), 'n'), new_size, log_.get(), /*owner = */ true);
  }
  res->mem_ = shm_->get_mem();
  connect_data_.shm_size_ = new_size;
  alloc_size_ = new_size;
  return res;
}

auto Writer::get_one_write_access_ptr_resize(size_t new_size) -> OneWriteAccess* {
  auto res = new OneWriteAccess(this, sem_.get(), nullptr, srv_.get(), log_);
  log_->debug("resizing sh4lt (%) from % bytes to % bytes",
              path_,
              std::to_string(connect_data_.shm_size_),
              std::to_string(new_size));
  shm_.reset();
  shm_ =
      std::make_unique<sysVShm>(ftok(path_.c_str(), 'n'), new_size, log_.get(), /*owner = */ true);
  res->mem_ = shm_->get_mem();
  connect_data_.shm_size_ = new_size;
  alloc_size_ = new_size;
  return res;
}

auto Writer::alloc_size() const -> size_t { return alloc_size_; }

OneWriteAccess::OneWriteAccess(
    Writer* writer, sysVSem* sem, void* mem, UnixSocketServer* srv, logger::Logger::ptr log)
    : writer_(writer), wlock_(sem), mem_(mem), srv_(srv), log_(log) {}

auto OneWriteAccess::shm_resize(size_t new_size) -> size_t {
  writer_->shm_.reset();
  writer_->shm_ = std::make_unique<sysVShm>(
      ftok(writer_->path_.c_str(), 'n'), new_size, log_.get(), /*owner = */ true);
  if (!writer_->shm_) return 0;
  mem_ = writer_->shm_->get_mem();
  writer_->connect_data_.shm_size_ = new_size;
  writer_->alloc_size_ = new_size;
  return new_size;
}

auto OneWriteAccess::notify_clients(size_t size, int64_t buffer_date, int64_t buffer_duration)
    -> short {
  if (has_notified_) {
    log_->warning(
        "one notification only is expected per OneWriteAccess instance, "
        "ignoring current invocation");
    return 0;
  }
  has_notified_ = true;
  auto time_info = writer_->time_.update(buffer_date, buffer_duration);
  short num_readers = srv_->notify_update(size, &time_info);
  // log->debug("one write access for % readers", std::to_string(num_readers));
  if (0 < num_readers) {
    wlock_.commit_readers(num_readers);
  }
  return num_readers;
}

}  // namespace sh4lt
