/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { ApprovalMode, Config, DEFAULT_GEMINI_EMBEDDING_MODEL, DEFAULT_GEMINI_MODEL, DEFAULT_MEMORY_FILE_FILTERING_OPTIONS, EditTool, FileDiscoveryService, getCurrentGeminiMdFilename, loadServerHierarchicalMemory, setGeminiMdFilename as setServerGeminiMdFilename, ShellTool, WriteFileTool, } from '@qwen-code/qwen-code-core';
import * as fs from 'node:fs';
import { homedir } from 'node:os';
import * as path from 'node:path';
import process from 'node:process';
import { hideBin } from 'yargs/helpers';
import yargs from 'yargs/yargs';
import { extensionsCommand } from '../commands/extensions.js';
import { mcpCommand } from '../commands/mcp.js';
import { resolvePath } from '../utils/resolvePath.js';
import { getCliVersion } from '../utils/version.js';
import { annotateActiveExtensions } from './extension.js';
import { loadSandboxConfig } from './sandboxConfig.js';
import { isWorkspaceTrusted } from './trustedFolders.js';
// Simple console logger for now - replace with actual logger if available
const logger = {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    debug: (...args) => console.debug('[DEBUG]', ...args),
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    warn: (...args) => console.warn('[WARN]', ...args),
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    error: (...args) => console.error('[ERROR]', ...args),
};
const VALID_APPROVAL_MODE_VALUES = [
    'plan',
    'default',
    'auto-edit',
    'yolo',
];
function formatApprovalModeError(value) {
    return new Error(`Invalid approval mode: ${value}. Valid values are: ${VALID_APPROVAL_MODE_VALUES.join(', ')}`);
}
function parseApprovalModeValue(value) {
    const normalized = value.trim().toLowerCase();
    switch (normalized) {
        case 'plan':
            return ApprovalMode.PLAN;
        case 'default':
            return ApprovalMode.DEFAULT;
        case 'yolo':
            return ApprovalMode.YOLO;
        case 'auto_edit':
        case 'autoedit':
        case 'auto-edit':
            return ApprovalMode.AUTO_EDIT;
        default:
            throw formatApprovalModeError(value);
    }
}
export async function parseArguments(settings) {
    const yargsInstance = yargs(hideBin(process.argv))
        // Set locale to English for consistent output, especially in tests
        .locale('en')
        .scriptName('qwen')
        .usage('Usage: qwen [options] [command]\n\nQwen Code - Launch an interactive CLI, use -p/--prompt for non-interactive mode')
        .command('$0', 'Launch Qwen Code', (yargsInstance) => yargsInstance
        .option('model', {
        alias: 'm',
        type: 'string',
        description: `Model`,
        default: process.env['GEMINI_MODEL'],
    })
        .option('prompt', {
        alias: 'p',
        type: 'string',
        description: 'Prompt. Appended to input on stdin (if any).',
    })
        .option('prompt-interactive', {
        alias: 'i',
        type: 'string',
        description: 'Execute the provided prompt and continue in interactive mode',
    })
        .option('sandbox', {
        alias: 's',
        type: 'boolean',
        description: 'Run in sandbox?',
    })
        .option('sandbox-image', {
        type: 'string',
        description: 'Sandbox image URI.',
    })
        .option('debug', {
        alias: 'd',
        type: 'boolean',
        description: 'Run in debug mode?',
        default: false,
    })
        .option('all-files', {
        alias: ['a'],
        type: 'boolean',
        description: 'Include ALL files in context?',
        default: false,
    })
        .option('show-memory-usage', {
        type: 'boolean',
        description: 'Show memory usage in status bar',
        default: false,
    })
        .option('yolo', {
        alias: 'y',
        type: 'boolean',
        description: 'Automatically accept all actions (aka YOLO mode, see https://www.youtube.com/watch?v=xvFZjo5PgG0 for more details)?',
        default: false,
    })
        .option('approval-mode', {
        type: 'string',
        choices: ['plan', 'default', 'auto-edit', 'yolo'],
        description: 'Set the approval mode: plan (plan only), default (prompt for approval), auto-edit (auto-approve edit tools), yolo (auto-approve all tools)',
    })
        .option('telemetry', {
        type: 'boolean',
        description: 'Enable telemetry? This flag specifically controls if telemetry is sent. Other --telemetry-* flags set specific values but do not enable telemetry on their own.',
    })
        .option('telemetry-target', {
        type: 'string',
        choices: ['local', 'gcp'],
        description: 'Set the telemetry target (local or gcp). Overrides settings files.',
    })
        .option('telemetry-otlp-endpoint', {
        type: 'string',
        description: 'Set the OTLP endpoint for telemetry. Overrides environment variables and settings files.',
    })
        .option('telemetry-otlp-protocol', {
        type: 'string',
        choices: ['grpc', 'http'],
        description: 'Set the OTLP protocol for telemetry (grpc or http). Overrides settings files.',
    })
        .option('telemetry-log-prompts', {
        type: 'boolean',
        description: 'Enable or disable logging of user prompts for telemetry. Overrides settings files.',
    })
        .option('telemetry-outfile', {
        type: 'string',
        description: 'Redirect all telemetry output to the specified file.',
    })
        .option('checkpointing', {
        alias: 'c',
        type: 'boolean',
        description: 'Enables checkpointing of file edits',
        default: false,
    })
        .option('experimental-acp', {
        type: 'boolean',
        description: 'Starts the agent in ACP mode',
    })
        .option('allowed-mcp-server-names', {
        type: 'array',
        string: true,
        description: 'Allowed MCP server names',
    })
        .option('allowed-tools', {
        type: 'array',
        string: true,
        description: 'Tools that are allowed to run without confirmation',
    })
        .option('extensions', {
        alias: 'e',
        type: 'array',
        string: true,
        description: 'A list of extensions to use. If not provided, all extensions are used.',
    })
        .option('list-extensions', {
        alias: 'l',
        type: 'boolean',
        description: 'List all available extensions and exit.',
    })
        .option('proxy', {
        type: 'string',
        description: 'Proxy for qwen client, like schema://user:password@host:port',
    })
        .option('include-directories', {
        type: 'array',
        string: true,
        description: 'Additional directories to include in the workspace (comma-separated or multiple --include-directories)',
        coerce: (dirs) => 
        // Handle comma-separated values
        dirs.flatMap((dir) => dir.split(',').map((d) => d.trim())),
    })
        .option('openai-logging', {
        type: 'boolean',
        description: 'Enable logging of OpenAI API calls for debugging and analysis',
    })
        .option('openai-api-key', {
        type: 'string',
        description: 'OpenAI API key to use for authentication',
    })
        .option('openai-base-url', {
        type: 'string',
        description: 'OpenAI base URL (for custom endpoints)',
    })
        .option('tavily-api-key', {
        type: 'string',
        description: 'Tavily API key for web search functionality',
    })
        .option('screen-reader', {
        type: 'boolean',
        description: 'Enable screen reader mode for accessibility.',
        default: false,
    })
        .option('vlm-switch-mode', {
        type: 'string',
        choices: ['once', 'session', 'persist'],
        description: 'Default behavior when images are detected in input. Values: once (one-time switch), session (switch for entire session), persist (continue with current model). Overrides settings files.',
        default: process.env['VLM_SWITCH_MODE'],
    })
        .check((argv) => {
        if (argv.prompt && argv['promptInteractive']) {
            throw new Error('Cannot use both --prompt (-p) and --prompt-interactive (-i) together');
        }
        if (argv.yolo && argv['approvalMode']) {
            throw new Error('Cannot use both --yolo (-y) and --approval-mode together. Use --approval-mode=yolo instead.');
        }
        return true;
    }))
        // Register MCP subcommands
        .command(mcpCommand);
    if (settings?.experimental?.extensionManagement ?? false) {
        yargsInstance.command(extensionsCommand);
    }
    yargsInstance
        .version(await getCliVersion()) // This will enable the --version flag based on package.json
        .alias('v', 'version')
        .help()
        .alias('h', 'help')
        .strict()
        .demandCommand(0, 0); // Allow base command to run with no subcommands
    yargsInstance.wrap(yargsInstance.terminalWidth());
    const result = await yargsInstance.parse();
    // Handle case where MCP subcommands are executed - they should exit the process
    // and not return to main CLI logic
    if (result._.length > 0 &&
        (result._[0] === 'mcp' || result._[0] === 'extensions')) {
        // MCP commands handle their own execution and process exit
        process.exit(0);
    }
    // The import format is now only controlled by settings.memoryImportFormat
    // We no longer accept it as a CLI argument
    return result;
}
// This function is now a thin wrapper around the server's implementation.
// It's kept in the CLI for now as App.tsx directly calls it for memory refresh.
// TODO: Consider if App.tsx should get memory via a server call or if Config should refresh itself.
export async function loadHierarchicalGeminiMemory(currentWorkingDirectory, includeDirectoriesToReadGemini = [], debugMode, fileService, settings, extensionContextFilePaths = [], memoryImportFormat = 'tree', fileFilteringOptions) {
    // FIX: Use real, canonical paths for a reliable comparison to handle symlinks.
    const realCwd = fs.realpathSync(path.resolve(currentWorkingDirectory));
    const realHome = fs.realpathSync(path.resolve(homedir()));
    const isHomeDirectory = realCwd === realHome;
    // If it is the home directory, pass an empty string to the core memory
    // function to signal that it should skip the workspace search.
    const effectiveCwd = isHomeDirectory ? '' : currentWorkingDirectory;
    if (debugMode) {
        logger.debug(`CLI: Delegating hierarchical memory load to server for CWD: ${currentWorkingDirectory} (memoryImportFormat: ${memoryImportFormat})`);
    }
    // Directly call the server function with the corrected path.
    return loadServerHierarchicalMemory(effectiveCwd, includeDirectoriesToReadGemini, debugMode, fileService, extensionContextFilePaths, memoryImportFormat, fileFilteringOptions, settings.context?.discoveryMaxDirs);
}
export async function loadCliConfig(settings, extensions, sessionId, argv, cwd = process.cwd()) {
    const debugMode = argv.debug ||
        [process.env['DEBUG'], process.env['DEBUG_MODE']].some((v) => v === 'true' || v === '1') ||
        false;
    const memoryImportFormat = settings.context?.importFormat || 'tree';
    const ideMode = settings.ide?.enabled ?? false;
    const folderTrustFeature = settings.security?.folderTrust?.featureEnabled ?? false;
    const folderTrustSetting = settings.security?.folderTrust?.enabled ?? true;
    const folderTrust = folderTrustFeature && folderTrustSetting;
    const trustedFolder = isWorkspaceTrusted(settings);
    const allExtensions = annotateActiveExtensions(extensions, argv.extensions || [], cwd);
    const activeExtensions = extensions.filter((_, i) => allExtensions[i].isActive);
    // Handle OpenAI API key from command line
    if (argv.openaiApiKey) {
        process.env['OPENAI_API_KEY'] = argv.openaiApiKey;
    }
    // Handle OpenAI base URL from command line
    if (argv.openaiBaseUrl) {
        process.env['OPENAI_BASE_URL'] = argv.openaiBaseUrl;
    }
    // Handle Tavily API key from command line
    if (argv.tavilyApiKey) {
        process.env['TAVILY_API_KEY'] = argv.tavilyApiKey;
    }
    // Set the context filename in the server's memoryTool module BEFORE loading memory
    // TODO(b/343434939): This is a bit of a hack. The contextFileName should ideally be passed
    // directly to the Config constructor in core, and have core handle setGeminiMdFilename.
    // However, loadHierarchicalGeminiMemory is called *before* createServerConfig.
    if (settings.context?.fileName) {
        setServerGeminiMdFilename(settings.context.fileName);
    }
    else {
        // Reset to default if not provided in settings.
        setServerGeminiMdFilename(getCurrentGeminiMdFilename());
    }
    const extensionContextFilePaths = activeExtensions.flatMap((e) => e.contextFiles);
    const fileService = new FileDiscoveryService(cwd);
    const fileFiltering = {
        ...DEFAULT_MEMORY_FILE_FILTERING_OPTIONS,
        ...settings.context?.fileFiltering,
    };
    const includeDirectories = (settings.context?.includeDirectories || [])
        .map(resolvePath)
        .concat((argv.includeDirectories || []).map(resolvePath));
    // Call the (now wrapper) loadHierarchicalGeminiMemory which calls the server's version
    const { memoryContent, fileCount } = await loadHierarchicalGeminiMemory(cwd, settings.context?.loadMemoryFromIncludeDirectories
        ? includeDirectories
        : [], debugMode, fileService, settings, extensionContextFilePaths, memoryImportFormat, fileFiltering);
    let mcpServers = mergeMcpServers(settings, activeExtensions);
    const question = argv.promptInteractive || argv.prompt || '';
    // Determine approval mode with backward compatibility
    let approvalMode;
    if (argv.approvalMode) {
        approvalMode = parseApprovalModeValue(argv.approvalMode);
    }
    else if (argv.yolo) {
        approvalMode = ApprovalMode.YOLO;
    }
    else if (settings.approvalMode) {
        approvalMode = parseApprovalModeValue(settings.approvalMode);
    }
    else {
        approvalMode = ApprovalMode.DEFAULT;
    }
    // Force approval mode to default if the folder is not trusted.
    if (!trustedFolder &&
        approvalMode !== ApprovalMode.DEFAULT &&
        approvalMode !== ApprovalMode.PLAN) {
        logger.warn(`Approval mode overridden to "default" because the current folder is not trusted.`);
        approvalMode = ApprovalMode.DEFAULT;
    }
    const interactive = !!argv.promptInteractive || (process.stdin.isTTY && question.length === 0);
    // In non-interactive mode, exclude tools that require a prompt.
    const extraExcludes = [];
    if (!interactive && !argv.experimentalAcp) {
        switch (approvalMode) {
            case ApprovalMode.PLAN:
            case ApprovalMode.DEFAULT:
                // In default non-interactive mode, all tools that require approval are excluded.
                extraExcludes.push(ShellTool.Name, EditTool.Name, WriteFileTool.Name);
                break;
            case ApprovalMode.AUTO_EDIT:
                // In auto-edit non-interactive mode, only tools that still require a prompt are excluded.
                extraExcludes.push(ShellTool.Name);
                break;
            case ApprovalMode.YOLO:
                // No extra excludes for YOLO mode.
                break;
            default:
                // This should never happen due to validation earlier, but satisfies the linter
                break;
        }
    }
    const excludeTools = mergeExcludeTools(settings, activeExtensions, extraExcludes.length > 0 ? extraExcludes : undefined);
    const blockedMcpServers = [];
    if (!argv.allowedMcpServerNames) {
        if (settings.mcp?.allowed) {
            mcpServers = allowedMcpServers(mcpServers, settings.mcp.allowed, blockedMcpServers);
        }
        if (settings.mcp?.excluded) {
            const excludedNames = new Set(settings.mcp.excluded.filter(Boolean));
            if (excludedNames.size > 0) {
                mcpServers = Object.fromEntries(Object.entries(mcpServers).filter(([key]) => !excludedNames.has(key)));
            }
        }
    }
    if (argv.allowedMcpServerNames) {
        mcpServers = allowedMcpServers(mcpServers, argv.allowedMcpServerNames, blockedMcpServers);
    }
    const sandboxConfig = await loadSandboxConfig(settings, argv);
    const cliVersion = await getCliVersion();
    const screenReader = argv.screenReader !== undefined
        ? argv.screenReader
        : (settings.ui?.accessibility?.screenReader ?? false);
    const vlmSwitchMode = argv.vlmSwitchMode || settings.experimental?.vlmSwitchMode;
    return new Config({
        sessionId,
        embeddingModel: DEFAULT_GEMINI_EMBEDDING_MODEL,
        sandbox: sandboxConfig,
        targetDir: cwd,
        includeDirectories,
        loadMemoryFromIncludeDirectories: settings.context?.loadMemoryFromIncludeDirectories || false,
        debugMode,
        question,
        fullContext: argv.allFiles || false,
        coreTools: settings.tools?.core || undefined,
        allowedTools: argv.allowedTools || settings.tools?.allowed || undefined,
        excludeTools,
        toolDiscoveryCommand: settings.tools?.discoveryCommand,
        toolCallCommand: settings.tools?.callCommand,
        mcpServerCommand: settings.mcp?.serverCommand,
        mcpServers,
        userMemory: memoryContent,
        geminiMdFileCount: fileCount,
        approvalMode,
        showMemoryUsage: argv.showMemoryUsage || settings.ui?.showMemoryUsage || false,
        accessibility: {
            ...settings.ui?.accessibility,
            screenReader,
        },
        telemetry: {
            enabled: argv.telemetry ?? settings.telemetry?.enabled,
            target: (argv.telemetryTarget ??
                settings.telemetry?.target),
            otlpEndpoint: argv.telemetryOtlpEndpoint ??
                process.env['OTEL_EXPORTER_OTLP_ENDPOINT'] ??
                settings.telemetry?.otlpEndpoint,
            otlpProtocol: ['grpc', 'http'].find((p) => p ===
                (argv.telemetryOtlpProtocol ?? settings.telemetry?.otlpProtocol)),
            logPrompts: argv.telemetryLogPrompts ?? settings.telemetry?.logPrompts,
            outfile: argv.telemetryOutfile ?? settings.telemetry?.outfile,
        },
        usageStatisticsEnabled: settings.privacy?.usageStatisticsEnabled ?? true,
        // Git-aware file filtering settings
        fileFiltering: {
            respectGitIgnore: settings.context?.fileFiltering?.respectGitIgnore,
            respectGeminiIgnore: settings.context?.fileFiltering?.respectGeminiIgnore,
            enableRecursiveFileSearch: settings.context?.fileFiltering?.enableRecursiveFileSearch,
            disableFuzzySearch: settings.context?.fileFiltering?.disableFuzzySearch,
        },
        checkpointing: argv.checkpointing || settings.general?.checkpointing?.enabled,
        proxy: argv.proxy ||
            process.env['HTTPS_PROXY'] ||
            process.env['https_proxy'] ||
            process.env['HTTP_PROXY'] ||
            process.env['http_proxy'],
        cwd,
        fileDiscoveryService: fileService,
        bugCommand: settings.advanced?.bugCommand,
        model: argv.model || settings.model?.name || DEFAULT_GEMINI_MODEL,
        extensionContextFilePaths,
        sessionTokenLimit: settings.sessionTokenLimit ?? -1,
        maxSessionTurns: settings.model?.maxSessionTurns ?? -1,
        experimentalZedIntegration: argv.experimentalAcp || false,
        listExtensions: argv.listExtensions || false,
        extensions: allExtensions,
        blockedMcpServers,
        noBrowser: !!process.env['NO_BROWSER'],
        enableOpenAILogging: (typeof argv.openaiLogging === 'undefined'
            ? settings.enableOpenAILogging
            : argv.openaiLogging) ?? false,
        systemPromptMappings: (settings.systemPromptMappings ?? [
            {
                baseUrls: [
                    'https://dashscope.aliyuncs.com/compatible-mode/v1/',
                    'https://dashscope-intl.aliyuncs.com/compatible-mode/v1/',
                ],
                modelNames: ['qwen3-coder-plus'],
                template: 'SYSTEM_TEMPLATE:{"name":"qwen3_coder","params":{"is_git_repository":{RUNTIME_VARS_IS_GIT_REPO},"sandbox":"{RUNTIME_VARS_SANDBOX}"}}',
            },
        ]),
        authType: settings.security?.auth?.selectedType,
        contentGenerator: settings.contentGenerator,
        cliVersion,
        tavilyApiKey: argv.tavilyApiKey ||
            settings.tavilyApiKey ||
            process.env['TAVILY_API_KEY'],
        summarizeToolOutput: settings.model?.summarizeToolOutput,
        ideMode,
        chatCompression: settings.model?.chatCompression,
        folderTrustFeature,
        folderTrust,
        interactive,
        trustedFolder,
        useRipgrep: settings.tools?.useRipgrep,
        shouldUseNodePtyShell: settings.tools?.usePty,
        skipNextSpeakerCheck: settings.model?.skipNextSpeakerCheck,
        enablePromptCompletion: settings.general?.enablePromptCompletion ?? false,
        skipLoopDetection: settings.skipLoopDetection ?? false,
        vlmSwitchMode,
    });
}
function allowedMcpServers(mcpServers, allowMCPServers, blockedMcpServers) {
    const allowedNames = new Set(allowMCPServers.filter(Boolean));
    if (allowedNames.size > 0) {
        mcpServers = Object.fromEntries(Object.entries(mcpServers).filter(([key, server]) => {
            const isAllowed = allowedNames.has(key);
            if (!isAllowed) {
                blockedMcpServers.push({
                    name: key,
                    extensionName: server.extensionName || '',
                });
            }
            return isAllowed;
        }));
    }
    else {
        blockedMcpServers.push(...Object.entries(mcpServers).map(([key, server]) => ({
            name: key,
            extensionName: server.extensionName || '',
        })));
        mcpServers = {};
    }
    return mcpServers;
}
function mergeMcpServers(settings, extensions) {
    const mcpServers = { ...(settings.mcpServers || {}) };
    for (const extension of extensions) {
        Object.entries(extension.config.mcpServers || {}).forEach(([key, server]) => {
            if (mcpServers[key]) {
                logger.warn(`Skipping extension MCP config for server with key "${key}" as it already exists.`);
                return;
            }
            mcpServers[key] = {
                ...server,
                extensionName: extension.config.name,
            };
        });
    }
    return mcpServers;
}
function mergeExcludeTools(settings, extensions, extraExcludes) {
    const allExcludeTools = new Set([
        ...(settings.tools?.exclude || []),
        ...(extraExcludes || []),
    ]);
    for (const extension of extensions) {
        for (const tool of extension.config.excludeTools || []) {
            allExcludeTools.add(tool);
        }
    }
    return [...allExcludeTools];
}
//# sourceMappingURL=config.js.map