/*  reedsol_logs.h - Log and antilog tables for Reed-Solomon */
/*
    libzint - the open source barcode library
    Copyright (C) 2020-2022 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/* SPDX-License-Identifier: BSD-3-Clause */

#ifndef Z_REEDSOL_LOGS_H
#define Z_REEDSOL_LOGS_H

/* Static log/antilog tables for prime polys of up to degree 8 (> 8 too large so generated at runtime instead).
 * Antilog tables doubled to avoid mod. */

/* Paste output of "backend/tests/test_reedsol -f generate -g" here */
static const unsigned char logt_0x13[16] = {
    0x00, 0x00, 0x01, 0x04, 0x02, 0x08, 0x05, 0x0A, 0x03, 0x0E, 0x09, 0x07, 0x06, 0x0D, 0x0B, 0x0C,
};
static const unsigned char alog_0x13[30] = {
    0x01, 0x02, 0x04, 0x08, 0x03, 0x06, 0x0C, 0x0B, 0x05, 0x0A, 0x07, 0x0E, 0x0F, 0x0D, 0x09,
    0x01, 0x02, 0x04, 0x08, 0x03, 0x06, 0x0C, 0x0B, 0x05, 0x0A, 0x07, 0x0E, 0x0F, 0x0D, 0x09,
};

static const unsigned char logt_0x25[32] = {
    0x00, 0x00, 0x01, 0x12, 0x02, 0x05, 0x13, 0x0B, 0x03, 0x1D, 0x06, 0x1B, 0x14, 0x08, 0x0C, 0x17,
    0x04, 0x0A, 0x1E, 0x11, 0x07, 0x16, 0x1C, 0x1A, 0x15, 0x19, 0x09, 0x10, 0x0D, 0x0E, 0x18, 0x0F,
};
static const unsigned char alog_0x25[62] = {
    0x01, 0x02, 0x04, 0x08, 0x10, 0x05, 0x0A, 0x14, 0x0D, 0x1A, 0x11, 0x07, 0x0E, 0x1C, 0x1D, 0x1F,
    0x1B, 0x13, 0x03, 0x06, 0x0C, 0x18, 0x15, 0x0F, 0x1E, 0x19, 0x17, 0x0B, 0x16, 0x09, 0x12,
    0x01, 0x02, 0x04, 0x08, 0x10, 0x05, 0x0A, 0x14, 0x0D, 0x1A, 0x11, 0x07, 0x0E, 0x1C, 0x1D, 0x1F,
    0x1B, 0x13, 0x03, 0x06, 0x0C, 0x18, 0x15, 0x0F, 0x1E, 0x19, 0x17, 0x0B, 0x16, 0x09, 0x12,
};

static const unsigned char logt_0x43[64] = {
    0x00, 0x00, 0x01, 0x06, 0x02, 0x0C, 0x07, 0x1A, 0x03, 0x20, 0x0D, 0x23, 0x08, 0x30, 0x1B, 0x12,
    0x04, 0x18, 0x21, 0x10, 0x0E, 0x34, 0x24, 0x36, 0x09, 0x2D, 0x31, 0x26, 0x1C, 0x29, 0x13, 0x38,
    0x05, 0x3E, 0x19, 0x0B, 0x22, 0x1F, 0x11, 0x2F, 0x0F, 0x17, 0x35, 0x33, 0x25, 0x2C, 0x37, 0x28,
    0x0A, 0x3D, 0x2E, 0x1E, 0x32, 0x16, 0x27, 0x2B, 0x1D, 0x3C, 0x2A, 0x15, 0x14, 0x3B, 0x39, 0x3A,
};
static const unsigned char alog_0x43[126] = {
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x03, 0x06, 0x0C, 0x18, 0x30, 0x23, 0x05, 0x0A, 0x14, 0x28,
    0x13, 0x26, 0x0F, 0x1E, 0x3C, 0x3B, 0x35, 0x29, 0x11, 0x22, 0x07, 0x0E, 0x1C, 0x38, 0x33, 0x25,
    0x09, 0x12, 0x24, 0x0B, 0x16, 0x2C, 0x1B, 0x36, 0x2F, 0x1D, 0x3A, 0x37, 0x2D, 0x19, 0x32, 0x27,
    0x0D, 0x1A, 0x34, 0x2B, 0x15, 0x2A, 0x17, 0x2E, 0x1F, 0x3E, 0x3F, 0x3D, 0x39, 0x31, 0x21,
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x03, 0x06, 0x0C, 0x18, 0x30, 0x23, 0x05, 0x0A, 0x14, 0x28,
    0x13, 0x26, 0x0F, 0x1E, 0x3C, 0x3B, 0x35, 0x29, 0x11, 0x22, 0x07, 0x0E, 0x1C, 0x38, 0x33, 0x25,
    0x09, 0x12, 0x24, 0x0B, 0x16, 0x2C, 0x1B, 0x36, 0x2F, 0x1D, 0x3A, 0x37, 0x2D, 0x19, 0x32, 0x27,
    0x0D, 0x1A, 0x34, 0x2B, 0x15, 0x2A, 0x17, 0x2E, 0x1F, 0x3E, 0x3F, 0x3D, 0x39, 0x31, 0x21,
};

static const unsigned char logt_0x89[128] = {
    0x00, 0x00, 0x01, 0x1F, 0x02, 0x3E, 0x20, 0x67, 0x03, 0x07, 0x3F, 0x0F, 0x21, 0x54, 0x68, 0x5D,
    0x04, 0x7C, 0x08, 0x79, 0x40, 0x4F, 0x10, 0x73, 0x22, 0x0B, 0x55, 0x26, 0x69, 0x2E, 0x5E, 0x33,
    0x05, 0x52, 0x7D, 0x3C, 0x09, 0x2C, 0x7A, 0x4D, 0x41, 0x43, 0x50, 0x2A, 0x11, 0x45, 0x74, 0x17,
    0x23, 0x76, 0x0C, 0x1C, 0x56, 0x19, 0x27, 0x39, 0x6A, 0x13, 0x2F, 0x59, 0x5F, 0x47, 0x34, 0x6E,
    0x06, 0x0E, 0x53, 0x5C, 0x7E, 0x1E, 0x3D, 0x66, 0x0A, 0x25, 0x2D, 0x32, 0x7B, 0x78, 0x4E, 0x72,
    0x42, 0x29, 0x44, 0x16, 0x51, 0x3B, 0x2B, 0x4C, 0x12, 0x58, 0x46, 0x6D, 0x75, 0x1B, 0x18, 0x38,
    0x24, 0x31, 0x77, 0x71, 0x0D, 0x5B, 0x1D, 0x65, 0x57, 0x6C, 0x1A, 0x37, 0x28, 0x15, 0x3A, 0x4B,
    0x6B, 0x36, 0x14, 0x4A, 0x30, 0x70, 0x5A, 0x64, 0x60, 0x61, 0x48, 0x62, 0x35, 0x49, 0x6F, 0x63,
};
static const unsigned char alog_0x89[254] = {
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x09, 0x12, 0x24, 0x48, 0x19, 0x32, 0x64, 0x41, 0x0B,
    0x16, 0x2C, 0x58, 0x39, 0x72, 0x6D, 0x53, 0x2F, 0x5E, 0x35, 0x6A, 0x5D, 0x33, 0x66, 0x45, 0x03,
    0x06, 0x0C, 0x18, 0x30, 0x60, 0x49, 0x1B, 0x36, 0x6C, 0x51, 0x2B, 0x56, 0x25, 0x4A, 0x1D, 0x3A,
    0x74, 0x61, 0x4B, 0x1F, 0x3E, 0x7C, 0x71, 0x6B, 0x5F, 0x37, 0x6E, 0x55, 0x23, 0x46, 0x05, 0x0A,
    0x14, 0x28, 0x50, 0x29, 0x52, 0x2D, 0x5A, 0x3D, 0x7A, 0x7D, 0x73, 0x6F, 0x57, 0x27, 0x4E, 0x15,
    0x2A, 0x54, 0x21, 0x42, 0x0D, 0x1A, 0x34, 0x68, 0x59, 0x3B, 0x76, 0x65, 0x43, 0x0F, 0x1E, 0x3C,
    0x78, 0x79, 0x7B, 0x7F, 0x77, 0x67, 0x47, 0x07, 0x0E, 0x1C, 0x38, 0x70, 0x69, 0x5B, 0x3F, 0x7E,
    0x75, 0x63, 0x4F, 0x17, 0x2E, 0x5C, 0x31, 0x62, 0x4D, 0x13, 0x26, 0x4C, 0x11, 0x22, 0x44,
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x09, 0x12, 0x24, 0x48, 0x19, 0x32, 0x64, 0x41, 0x0B,
    0x16, 0x2C, 0x58, 0x39, 0x72, 0x6D, 0x53, 0x2F, 0x5E, 0x35, 0x6A, 0x5D, 0x33, 0x66, 0x45, 0x03,
    0x06, 0x0C, 0x18, 0x30, 0x60, 0x49, 0x1B, 0x36, 0x6C, 0x51, 0x2B, 0x56, 0x25, 0x4A, 0x1D, 0x3A,
    0x74, 0x61, 0x4B, 0x1F, 0x3E, 0x7C, 0x71, 0x6B, 0x5F, 0x37, 0x6E, 0x55, 0x23, 0x46, 0x05, 0x0A,
    0x14, 0x28, 0x50, 0x29, 0x52, 0x2D, 0x5A, 0x3D, 0x7A, 0x7D, 0x73, 0x6F, 0x57, 0x27, 0x4E, 0x15,
    0x2A, 0x54, 0x21, 0x42, 0x0D, 0x1A, 0x34, 0x68, 0x59, 0x3B, 0x76, 0x65, 0x43, 0x0F, 0x1E, 0x3C,
    0x78, 0x79, 0x7B, 0x7F, 0x77, 0x67, 0x47, 0x07, 0x0E, 0x1C, 0x38, 0x70, 0x69, 0x5B, 0x3F, 0x7E,
    0x75, 0x63, 0x4F, 0x17, 0x2E, 0x5C, 0x31, 0x62, 0x4D, 0x13, 0x26, 0x4C, 0x11, 0x22, 0x44,
};

static const unsigned char logt_0x11d[256] = {
    0x00, 0x00, 0x01, 0x19, 0x02, 0x32, 0x1A, 0xC6, 0x03, 0xDF, 0x33, 0xEE, 0x1B, 0x68, 0xC7, 0x4B,
    0x04, 0x64, 0xE0, 0x0E, 0x34, 0x8D, 0xEF, 0x81, 0x1C, 0xC1, 0x69, 0xF8, 0xC8, 0x08, 0x4C, 0x71,
    0x05, 0x8A, 0x65, 0x2F, 0xE1, 0x24, 0x0F, 0x21, 0x35, 0x93, 0x8E, 0xDA, 0xF0, 0x12, 0x82, 0x45,
    0x1D, 0xB5, 0xC2, 0x7D, 0x6A, 0x27, 0xF9, 0xB9, 0xC9, 0x9A, 0x09, 0x78, 0x4D, 0xE4, 0x72, 0xA6,
    0x06, 0xBF, 0x8B, 0x62, 0x66, 0xDD, 0x30, 0xFD, 0xE2, 0x98, 0x25, 0xB3, 0x10, 0x91, 0x22, 0x88,
    0x36, 0xD0, 0x94, 0xCE, 0x8F, 0x96, 0xDB, 0xBD, 0xF1, 0xD2, 0x13, 0x5C, 0x83, 0x38, 0x46, 0x40,
    0x1E, 0x42, 0xB6, 0xA3, 0xC3, 0x48, 0x7E, 0x6E, 0x6B, 0x3A, 0x28, 0x54, 0xFA, 0x85, 0xBA, 0x3D,
    0xCA, 0x5E, 0x9B, 0x9F, 0x0A, 0x15, 0x79, 0x2B, 0x4E, 0xD4, 0xE5, 0xAC, 0x73, 0xF3, 0xA7, 0x57,
    0x07, 0x70, 0xC0, 0xF7, 0x8C, 0x80, 0x63, 0x0D, 0x67, 0x4A, 0xDE, 0xED, 0x31, 0xC5, 0xFE, 0x18,
    0xE3, 0xA5, 0x99, 0x77, 0x26, 0xB8, 0xB4, 0x7C, 0x11, 0x44, 0x92, 0xD9, 0x23, 0x20, 0x89, 0x2E,
    0x37, 0x3F, 0xD1, 0x5B, 0x95, 0xBC, 0xCF, 0xCD, 0x90, 0x87, 0x97, 0xB2, 0xDC, 0xFC, 0xBE, 0x61,
    0xF2, 0x56, 0xD3, 0xAB, 0x14, 0x2A, 0x5D, 0x9E, 0x84, 0x3C, 0x39, 0x53, 0x47, 0x6D, 0x41, 0xA2,
    0x1F, 0x2D, 0x43, 0xD8, 0xB7, 0x7B, 0xA4, 0x76, 0xC4, 0x17, 0x49, 0xEC, 0x7F, 0x0C, 0x6F, 0xF6,
    0x6C, 0xA1, 0x3B, 0x52, 0x29, 0x9D, 0x55, 0xAA, 0xFB, 0x60, 0x86, 0xB1, 0xBB, 0xCC, 0x3E, 0x5A,
    0xCB, 0x59, 0x5F, 0xB0, 0x9C, 0xA9, 0xA0, 0x51, 0x0B, 0xF5, 0x16, 0xEB, 0x7A, 0x75, 0x2C, 0xD7,
    0x4F, 0xAE, 0xD5, 0xE9, 0xE6, 0xE7, 0xAD, 0xE8, 0x74, 0xD6, 0xF4, 0xEA, 0xA8, 0x50, 0x58, 0xAF,
};
static const unsigned char alog_0x11d[510] = {
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1D, 0x3A, 0x74, 0xE8, 0xCD, 0x87, 0x13, 0x26,
    0x4C, 0x98, 0x2D, 0x5A, 0xB4, 0x75, 0xEA, 0xC9, 0x8F, 0x03, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0,
    0x9D, 0x27, 0x4E, 0x9C, 0x25, 0x4A, 0x94, 0x35, 0x6A, 0xD4, 0xB5, 0x77, 0xEE, 0xC1, 0x9F, 0x23,
    0x46, 0x8C, 0x05, 0x0A, 0x14, 0x28, 0x50, 0xA0, 0x5D, 0xBA, 0x69, 0xD2, 0xB9, 0x6F, 0xDE, 0xA1,
    0x5F, 0xBE, 0x61, 0xC2, 0x99, 0x2F, 0x5E, 0xBC, 0x65, 0xCA, 0x89, 0x0F, 0x1E, 0x3C, 0x78, 0xF0,
    0xFD, 0xE7, 0xD3, 0xBB, 0x6B, 0xD6, 0xB1, 0x7F, 0xFE, 0xE1, 0xDF, 0xA3, 0x5B, 0xB6, 0x71, 0xE2,
    0xD9, 0xAF, 0x43, 0x86, 0x11, 0x22, 0x44, 0x88, 0x0D, 0x1A, 0x34, 0x68, 0xD0, 0xBD, 0x67, 0xCE,
    0x81, 0x1F, 0x3E, 0x7C, 0xF8, 0xED, 0xC7, 0x93, 0x3B, 0x76, 0xEC, 0xC5, 0x97, 0x33, 0x66, 0xCC,
    0x85, 0x17, 0x2E, 0x5C, 0xB8, 0x6D, 0xDA, 0xA9, 0x4F, 0x9E, 0x21, 0x42, 0x84, 0x15, 0x2A, 0x54,
    0xA8, 0x4D, 0x9A, 0x29, 0x52, 0xA4, 0x55, 0xAA, 0x49, 0x92, 0x39, 0x72, 0xE4, 0xD5, 0xB7, 0x73,
    0xE6, 0xD1, 0xBF, 0x63, 0xC6, 0x91, 0x3F, 0x7E, 0xFC, 0xE5, 0xD7, 0xB3, 0x7B, 0xF6, 0xF1, 0xFF,
    0xE3, 0xDB, 0xAB, 0x4B, 0x96, 0x31, 0x62, 0xC4, 0x95, 0x37, 0x6E, 0xDC, 0xA5, 0x57, 0xAE, 0x41,
    0x82, 0x19, 0x32, 0x64, 0xC8, 0x8D, 0x07, 0x0E, 0x1C, 0x38, 0x70, 0xE0, 0xDD, 0xA7, 0x53, 0xA6,
    0x51, 0xA2, 0x59, 0xB2, 0x79, 0xF2, 0xF9, 0xEF, 0xC3, 0x9B, 0x2B, 0x56, 0xAC, 0x45, 0x8A, 0x09,
    0x12, 0x24, 0x48, 0x90, 0x3D, 0x7A, 0xF4, 0xF5, 0xF7, 0xF3, 0xFB, 0xEB, 0xCB, 0x8B, 0x0B, 0x16,
    0x2C, 0x58, 0xB0, 0x7D, 0xFA, 0xE9, 0xCF, 0x83, 0x1B, 0x36, 0x6C, 0xD8, 0xAD, 0x47, 0x8E,
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1D, 0x3A, 0x74, 0xE8, 0xCD, 0x87, 0x13, 0x26,
    0x4C, 0x98, 0x2D, 0x5A, 0xB4, 0x75, 0xEA, 0xC9, 0x8F, 0x03, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0,
    0x9D, 0x27, 0x4E, 0x9C, 0x25, 0x4A, 0x94, 0x35, 0x6A, 0xD4, 0xB5, 0x77, 0xEE, 0xC1, 0x9F, 0x23,
    0x46, 0x8C, 0x05, 0x0A, 0x14, 0x28, 0x50, 0xA0, 0x5D, 0xBA, 0x69, 0xD2, 0xB9, 0x6F, 0xDE, 0xA1,
    0x5F, 0xBE, 0x61, 0xC2, 0x99, 0x2F, 0x5E, 0xBC, 0x65, 0xCA, 0x89, 0x0F, 0x1E, 0x3C, 0x78, 0xF0,
    0xFD, 0xE7, 0xD3, 0xBB, 0x6B, 0xD6, 0xB1, 0x7F, 0xFE, 0xE1, 0xDF, 0xA3, 0x5B, 0xB6, 0x71, 0xE2,
    0xD9, 0xAF, 0x43, 0x86, 0x11, 0x22, 0x44, 0x88, 0x0D, 0x1A, 0x34, 0x68, 0xD0, 0xBD, 0x67, 0xCE,
    0x81, 0x1F, 0x3E, 0x7C, 0xF8, 0xED, 0xC7, 0x93, 0x3B, 0x76, 0xEC, 0xC5, 0x97, 0x33, 0x66, 0xCC,
    0x85, 0x17, 0x2E, 0x5C, 0xB8, 0x6D, 0xDA, 0xA9, 0x4F, 0x9E, 0x21, 0x42, 0x84, 0x15, 0x2A, 0x54,
    0xA8, 0x4D, 0x9A, 0x29, 0x52, 0xA4, 0x55, 0xAA, 0x49, 0x92, 0x39, 0x72, 0xE4, 0xD5, 0xB7, 0x73,
    0xE6, 0xD1, 0xBF, 0x63, 0xC6, 0x91, 0x3F, 0x7E, 0xFC, 0xE5, 0xD7, 0xB3, 0x7B, 0xF6, 0xF1, 0xFF,
    0xE3, 0xDB, 0xAB, 0x4B, 0x96, 0x31, 0x62, 0xC4, 0x95, 0x37, 0x6E, 0xDC, 0xA5, 0x57, 0xAE, 0x41,
    0x82, 0x19, 0x32, 0x64, 0xC8, 0x8D, 0x07, 0x0E, 0x1C, 0x38, 0x70, 0xE0, 0xDD, 0xA7, 0x53, 0xA6,
    0x51, 0xA2, 0x59, 0xB2, 0x79, 0xF2, 0xF9, 0xEF, 0xC3, 0x9B, 0x2B, 0x56, 0xAC, 0x45, 0x8A, 0x09,
    0x12, 0x24, 0x48, 0x90, 0x3D, 0x7A, 0xF4, 0xF5, 0xF7, 0xF3, 0xFB, 0xEB, 0xCB, 0x8B, 0x0B, 0x16,
    0x2C, 0x58, 0xB0, 0x7D, 0xFA, 0xE9, 0xCF, 0x83, 0x1B, 0x36, 0x6C, 0xD8, 0xAD, 0x47, 0x8E,
};

static const unsigned char logt_0x12d[256] = {
    0x00, 0x00, 0x01, 0xF0, 0x02, 0xE1, 0xF1, 0x35, 0x03, 0x26, 0xE2, 0x85, 0xF2, 0x2B, 0x36, 0xD2,
    0x04, 0xC3, 0x27, 0x72, 0xE3, 0x6A, 0x86, 0x1C, 0xF3, 0x8C, 0x2C, 0x17, 0x37, 0x76, 0xD3, 0xEA,
    0x05, 0xDB, 0xC4, 0x60, 0x28, 0xDE, 0x73, 0x67, 0xE4, 0x4E, 0x6B, 0x7D, 0x87, 0x08, 0x1D, 0xA2,
    0xF4, 0xBA, 0x8D, 0xB4, 0x2D, 0x63, 0x18, 0x31, 0x38, 0x0D, 0x77, 0x99, 0xD4, 0xC7, 0xEB, 0x5B,
    0x06, 0x4C, 0xDC, 0xD9, 0xC5, 0x0B, 0x61, 0xB8, 0x29, 0x24, 0xDF, 0xFD, 0x74, 0x8A, 0x68, 0xC1,
    0xE5, 0x56, 0x4F, 0xAB, 0x6C, 0xA5, 0x7E, 0x91, 0x88, 0x22, 0x09, 0x4A, 0x1E, 0x20, 0xA3, 0x54,
    0xF5, 0xAD, 0xBB, 0xCC, 0x8E, 0x51, 0xB5, 0xBE, 0x2E, 0x58, 0x64, 0x9F, 0x19, 0xE7, 0x32, 0xCF,
    0x39, 0x93, 0x0E, 0x43, 0x78, 0x80, 0x9A, 0xF8, 0xD5, 0xA7, 0xC8, 0x3F, 0xEC, 0x6E, 0x5C, 0xB0,
    0x07, 0xA1, 0x4D, 0x7C, 0xDD, 0x66, 0xDA, 0x5F, 0xC6, 0x5A, 0x0C, 0x98, 0x62, 0x30, 0xB9, 0xB3,
    0x2A, 0xD1, 0x25, 0x84, 0xE0, 0x34, 0xFE, 0xEF, 0x75, 0xE9, 0x8B, 0x16, 0x69, 0x1B, 0xC2, 0x71,
    0xE6, 0xCE, 0x57, 0x9E, 0x50, 0xBD, 0xAC, 0xCB, 0x6D, 0xAF, 0xA6, 0x3E, 0x7F, 0xF7, 0x92, 0x42,
    0x89, 0xC0, 0x23, 0xFC, 0x0A, 0xB7, 0x4B, 0xD8, 0x1F, 0x53, 0x21, 0x49, 0xA4, 0x90, 0x55, 0xAA,
    0xF6, 0x41, 0xAE, 0x3D, 0xBC, 0xCA, 0xCD, 0x9D, 0x8F, 0xA9, 0x52, 0x48, 0xB6, 0xD7, 0xBF, 0xFB,
    0x2F, 0xB2, 0x59, 0x97, 0x65, 0x5E, 0xA0, 0x7B, 0x1A, 0x70, 0xE8, 0x15, 0x33, 0xEE, 0xD0, 0x83,
    0x3A, 0x45, 0x94, 0x12, 0x0F, 0x10, 0x44, 0x11, 0x79, 0x95, 0x81, 0x13, 0x9B, 0x3B, 0xF9, 0x46,
    0xD6, 0xFA, 0xA8, 0x47, 0xC9, 0x9C, 0x40, 0x3C, 0xED, 0x82, 0x6F, 0x14, 0x5D, 0x7A, 0xB1, 0x96,
};
static const unsigned char alog_0x12d[510] = {
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x2D, 0x5A, 0xB4, 0x45, 0x8A, 0x39, 0x72, 0xE4,
    0xE5, 0xE7, 0xE3, 0xEB, 0xFB, 0xDB, 0x9B, 0x1B, 0x36, 0x6C, 0xD8, 0x9D, 0x17, 0x2E, 0x5C, 0xB8,
    0x5D, 0xBA, 0x59, 0xB2, 0x49, 0x92, 0x09, 0x12, 0x24, 0x48, 0x90, 0x0D, 0x1A, 0x34, 0x68, 0xD0,
    0x8D, 0x37, 0x6E, 0xDC, 0x95, 0x07, 0x0E, 0x1C, 0x38, 0x70, 0xE0, 0xED, 0xF7, 0xC3, 0xAB, 0x7B,
    0xF6, 0xC1, 0xAF, 0x73, 0xE6, 0xE1, 0xEF, 0xF3, 0xCB, 0xBB, 0x5B, 0xB6, 0x41, 0x82, 0x29, 0x52,
    0xA4, 0x65, 0xCA, 0xB9, 0x5F, 0xBE, 0x51, 0xA2, 0x69, 0xD2, 0x89, 0x3F, 0x7E, 0xFC, 0xD5, 0x87,
    0x23, 0x46, 0x8C, 0x35, 0x6A, 0xD4, 0x85, 0x27, 0x4E, 0x9C, 0x15, 0x2A, 0x54, 0xA8, 0x7D, 0xFA,
    0xD9, 0x9F, 0x13, 0x26, 0x4C, 0x98, 0x1D, 0x3A, 0x74, 0xE8, 0xFD, 0xD7, 0x83, 0x2B, 0x56, 0xAC,
    0x75, 0xEA, 0xF9, 0xDF, 0x93, 0x0B, 0x16, 0x2C, 0x58, 0xB0, 0x4D, 0x9A, 0x19, 0x32, 0x64, 0xC8,
    0xBD, 0x57, 0xAE, 0x71, 0xE2, 0xE9, 0xFF, 0xD3, 0x8B, 0x3B, 0x76, 0xEC, 0xF5, 0xC7, 0xA3, 0x6B,
    0xD6, 0x81, 0x2F, 0x5E, 0xBC, 0x55, 0xAA, 0x79, 0xF2, 0xC9, 0xBF, 0x53, 0xA6, 0x61, 0xC2, 0xA9,
    0x7F, 0xFE, 0xD1, 0x8F, 0x33, 0x66, 0xCC, 0xB5, 0x47, 0x8E, 0x31, 0x62, 0xC4, 0xA5, 0x67, 0xCE,
    0xB1, 0x4F, 0x9E, 0x11, 0x22, 0x44, 0x88, 0x3D, 0x7A, 0xF4, 0xC5, 0xA7, 0x63, 0xC6, 0xA1, 0x6F,
    0xDE, 0x91, 0x0F, 0x1E, 0x3C, 0x78, 0xF0, 0xCD, 0xB7, 0x43, 0x86, 0x21, 0x42, 0x84, 0x25, 0x4A,
    0x94, 0x05, 0x0A, 0x14, 0x28, 0x50, 0xA0, 0x6D, 0xDA, 0x99, 0x1F, 0x3E, 0x7C, 0xF8, 0xDD, 0x97,
    0x03, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0xAD, 0x77, 0xEE, 0xF1, 0xCF, 0xB3, 0x4B, 0x96,
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x2D, 0x5A, 0xB4, 0x45, 0x8A, 0x39, 0x72, 0xE4,
    0xE5, 0xE7, 0xE3, 0xEB, 0xFB, 0xDB, 0x9B, 0x1B, 0x36, 0x6C, 0xD8, 0x9D, 0x17, 0x2E, 0x5C, 0xB8,
    0x5D, 0xBA, 0x59, 0xB2, 0x49, 0x92, 0x09, 0x12, 0x24, 0x48, 0x90, 0x0D, 0x1A, 0x34, 0x68, 0xD0,
    0x8D, 0x37, 0x6E, 0xDC, 0x95, 0x07, 0x0E, 0x1C, 0x38, 0x70, 0xE0, 0xED, 0xF7, 0xC3, 0xAB, 0x7B,
    0xF6, 0xC1, 0xAF, 0x73, 0xE6, 0xE1, 0xEF, 0xF3, 0xCB, 0xBB, 0x5B, 0xB6, 0x41, 0x82, 0x29, 0x52,
    0xA4, 0x65, 0xCA, 0xB9, 0x5F, 0xBE, 0x51, 0xA2, 0x69, 0xD2, 0x89, 0x3F, 0x7E, 0xFC, 0xD5, 0x87,
    0x23, 0x46, 0x8C, 0x35, 0x6A, 0xD4, 0x85, 0x27, 0x4E, 0x9C, 0x15, 0x2A, 0x54, 0xA8, 0x7D, 0xFA,
    0xD9, 0x9F, 0x13, 0x26, 0x4C, 0x98, 0x1D, 0x3A, 0x74, 0xE8, 0xFD, 0xD7, 0x83, 0x2B, 0x56, 0xAC,
    0x75, 0xEA, 0xF9, 0xDF, 0x93, 0x0B, 0x16, 0x2C, 0x58, 0xB0, 0x4D, 0x9A, 0x19, 0x32, 0x64, 0xC8,
    0xBD, 0x57, 0xAE, 0x71, 0xE2, 0xE9, 0xFF, 0xD3, 0x8B, 0x3B, 0x76, 0xEC, 0xF5, 0xC7, 0xA3, 0x6B,
    0xD6, 0x81, 0x2F, 0x5E, 0xBC, 0x55, 0xAA, 0x79, 0xF2, 0xC9, 0xBF, 0x53, 0xA6, 0x61, 0xC2, 0xA9,
    0x7F, 0xFE, 0xD1, 0x8F, 0x33, 0x66, 0xCC, 0xB5, 0x47, 0x8E, 0x31, 0x62, 0xC4, 0xA5, 0x67, 0xCE,
    0xB1, 0x4F, 0x9E, 0x11, 0x22, 0x44, 0x88, 0x3D, 0x7A, 0xF4, 0xC5, 0xA7, 0x63, 0xC6, 0xA1, 0x6F,
    0xDE, 0x91, 0x0F, 0x1E, 0x3C, 0x78, 0xF0, 0xCD, 0xB7, 0x43, 0x86, 0x21, 0x42, 0x84, 0x25, 0x4A,
    0x94, 0x05, 0x0A, 0x14, 0x28, 0x50, 0xA0, 0x6D, 0xDA, 0x99, 0x1F, 0x3E, 0x7C, 0xF8, 0xDD, 0x97,
    0x03, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0xAD, 0x77, 0xEE, 0xF1, 0xCF, 0xB3, 0x4B, 0x96,
};

static const unsigned char logt_0x163[256] = {
    0x00, 0x00, 0x01, 0xC5, 0x02, 0x8B, 0xC6, 0x6C, 0x03, 0x32, 0x8C, 0xC0, 0xC7, 0x25, 0x6D, 0x51,
    0x04, 0x17, 0x33, 0xEE, 0x8D, 0xD8, 0xC1, 0xEA, 0xC8, 0x0C, 0x26, 0xF7, 0x6E, 0x86, 0x52, 0x7C,
    0x05, 0x42, 0x18, 0x91, 0x34, 0x73, 0xEF, 0x4C, 0x8E, 0xCE, 0xD9, 0xD1, 0xC2, 0xBD, 0xEB, 0xF4,
    0xC9, 0x2D, 0x0D, 0xDC, 0x27, 0xB4, 0xF8, 0xA4, 0x6F, 0xB0, 0x87, 0xD4, 0x53, 0x1E, 0x7D, 0x9E,
    0x06, 0x64, 0x43, 0x37, 0x19, 0x81, 0x92, 0xE3, 0x35, 0xE1, 0x74, 0x76, 0xF0, 0x9A, 0x4D, 0x78,
    0x8F, 0x4A, 0xCF, 0xF2, 0xDA, 0xA2, 0xD2, 0x9C, 0xC3, 0x6A, 0xBE, 0x4F, 0xEC, 0xE8, 0xF5, 0x7A,
    0xCA, 0xAC, 0x2E, 0x08, 0x0E, 0x57, 0xDD, 0x66, 0x28, 0x12, 0xB5, 0x45, 0xF9, 0x5E, 0xA5, 0x39,
    0x70, 0xBA, 0xB1, 0x1B, 0x88, 0x22, 0xD5, 0x83, 0x54, 0x5B, 0x1F, 0x94, 0x7E, 0x97, 0x9F, 0xE5,
    0x07, 0xAB, 0x65, 0x56, 0x44, 0x11, 0x38, 0x5D, 0x1A, 0xB9, 0x82, 0x21, 0x93, 0x5A, 0xE4, 0x96,
    0x36, 0x63, 0xE2, 0x80, 0x75, 0xE0, 0x77, 0x99, 0xF1, 0x49, 0x9B, 0xA1, 0x4E, 0x69, 0x79, 0xE7,
    0x90, 0x41, 0x4B, 0x72, 0xD0, 0xCD, 0xF3, 0xBC, 0xDB, 0x2C, 0xA3, 0xB3, 0xD3, 0xAF, 0x9D, 0x1D,
    0xC4, 0xFE, 0x6B, 0x8A, 0xBF, 0x31, 0x50, 0x24, 0xED, 0x16, 0xE9, 0xD7, 0xF6, 0x0B, 0x7B, 0x85,
    0xCB, 0x3F, 0xAD, 0x2A, 0x2F, 0xFC, 0x09, 0x14, 0x0F, 0xA9, 0x58, 0xB7, 0xDE, 0x61, 0x67, 0x47,
    0x29, 0x3E, 0x13, 0xFB, 0xB6, 0xA8, 0x46, 0x60, 0xFA, 0x3D, 0x5F, 0xA7, 0xA6, 0x3C, 0x3A, 0x3B,
    0x71, 0x40, 0xBB, 0xCC, 0xB2, 0x2B, 0x1C, 0xAE, 0x89, 0xFD, 0x23, 0x30, 0xD6, 0x15, 0x84, 0x0A,
    0x55, 0xAA, 0x5C, 0x10, 0x20, 0xB8, 0x95, 0x59, 0x7F, 0x62, 0x98, 0xDF, 0xA0, 0x48, 0xE6, 0x68,
};
static const unsigned char alog_0x163[510] = {
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x63, 0xC6, 0xEF, 0xBD, 0x19, 0x32, 0x64, 0xC8,
    0xF3, 0x85, 0x69, 0xD2, 0xC7, 0xED, 0xB9, 0x11, 0x22, 0x44, 0x88, 0x73, 0xE6, 0xAF, 0x3D, 0x7A,
    0xF4, 0x8B, 0x75, 0xEA, 0xB7, 0x0D, 0x1A, 0x34, 0x68, 0xD0, 0xC3, 0xE5, 0xA9, 0x31, 0x62, 0xC4,
    0xEB, 0xB5, 0x09, 0x12, 0x24, 0x48, 0x90, 0x43, 0x86, 0x6F, 0xDE, 0xDF, 0xDD, 0xD9, 0xD1, 0xC1,
    0xE1, 0xA1, 0x21, 0x42, 0x84, 0x6B, 0xD6, 0xCF, 0xFD, 0x99, 0x51, 0xA2, 0x27, 0x4E, 0x9C, 0x5B,
    0xB6, 0x0F, 0x1E, 0x3C, 0x78, 0xF0, 0x83, 0x65, 0xCA, 0xF7, 0x8D, 0x79, 0xF2, 0x87, 0x6D, 0xDA,
    0xD7, 0xCD, 0xF9, 0x91, 0x41, 0x82, 0x67, 0xCE, 0xFF, 0x9D, 0x59, 0xB2, 0x07, 0x0E, 0x1C, 0x38,
    0x70, 0xE0, 0xA3, 0x25, 0x4A, 0x94, 0x4B, 0x96, 0x4F, 0x9E, 0x5F, 0xBE, 0x1F, 0x3E, 0x7C, 0xF8,
    0x93, 0x45, 0x8A, 0x77, 0xEE, 0xBF, 0x1D, 0x3A, 0x74, 0xE8, 0xB3, 0x05, 0x0A, 0x14, 0x28, 0x50,
    0xA0, 0x23, 0x46, 0x8C, 0x7B, 0xF6, 0x8F, 0x7D, 0xFA, 0x97, 0x4D, 0x9A, 0x57, 0xAE, 0x3F, 0x7E,
    0xFC, 0x9B, 0x55, 0xAA, 0x37, 0x6E, 0xDC, 0xDB, 0xD5, 0xC9, 0xF1, 0x81, 0x61, 0xC2, 0xE7, 0xAD,
    0x39, 0x72, 0xE4, 0xAB, 0x35, 0x6A, 0xD4, 0xCB, 0xF5, 0x89, 0x71, 0xE2, 0xA7, 0x2D, 0x5A, 0xB4,
    0x0B, 0x16, 0x2C, 0x58, 0xB0, 0x03, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0xE3, 0xA5, 0x29, 0x52,
    0xA4, 0x2B, 0x56, 0xAC, 0x3B, 0x76, 0xEC, 0xBB, 0x15, 0x2A, 0x54, 0xA8, 0x33, 0x66, 0xCC, 0xFB,
    0x95, 0x49, 0x92, 0x47, 0x8E, 0x7F, 0xFE, 0x9F, 0x5D, 0xBA, 0x17, 0x2E, 0x5C, 0xB8, 0x13, 0x26,
    0x4C, 0x98, 0x53, 0xA6, 0x2F, 0x5E, 0xBC, 0x1B, 0x36, 0x6C, 0xD8, 0xD3, 0xC5, 0xE9, 0xB1,
    0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x63, 0xC6, 0xEF, 0xBD, 0x19, 0x32, 0x64, 0xC8,
    0xF3, 0x85, 0x69, 0xD2, 0xC7, 0xED, 0xB9, 0x11, 0x22, 0x44, 0x88, 0x73, 0xE6, 0xAF, 0x3D, 0x7A,
    0xF4, 0x8B, 0x75, 0xEA, 0xB7, 0x0D, 0x1A, 0x34, 0x68, 0xD0, 0xC3, 0xE5, 0xA9, 0x31, 0x62, 0xC4,
    0xEB, 0xB5, 0x09, 0x12, 0x24, 0x48, 0x90, 0x43, 0x86, 0x6F, 0xDE, 0xDF, 0xDD, 0xD9, 0xD1, 0xC1,
    0xE1, 0xA1, 0x21, 0x42, 0x84, 0x6B, 0xD6, 0xCF, 0xFD, 0x99, 0x51, 0xA2, 0x27, 0x4E, 0x9C, 0x5B,
    0xB6, 0x0F, 0x1E, 0x3C, 0x78, 0xF0, 0x83, 0x65, 0xCA, 0xF7, 0x8D, 0x79, 0xF2, 0x87, 0x6D, 0xDA,
    0xD7, 0xCD, 0xF9, 0x91, 0x41, 0x82, 0x67, 0xCE, 0xFF, 0x9D, 0x59, 0xB2, 0x07, 0x0E, 0x1C, 0x38,
    0x70, 0xE0, 0xA3, 0x25, 0x4A, 0x94, 0x4B, 0x96, 0x4F, 0x9E, 0x5F, 0xBE, 0x1F, 0x3E, 0x7C, 0xF8,
    0x93, 0x45, 0x8A, 0x77, 0xEE, 0xBF, 0x1D, 0x3A, 0x74, 0xE8, 0xB3, 0x05, 0x0A, 0x14, 0x28, 0x50,
    0xA0, 0x23, 0x46, 0x8C, 0x7B, 0xF6, 0x8F, 0x7D, 0xFA, 0x97, 0x4D, 0x9A, 0x57, 0xAE, 0x3F, 0x7E,
    0xFC, 0x9B, 0x55, 0xAA, 0x37, 0x6E, 0xDC, 0xDB, 0xD5, 0xC9, 0xF1, 0x81, 0x61, 0xC2, 0xE7, 0xAD,
    0x39, 0x72, 0xE4, 0xAB, 0x35, 0x6A, 0xD4, 0xCB, 0xF5, 0x89, 0x71, 0xE2, 0xA7, 0x2D, 0x5A, 0xB4,
    0x0B, 0x16, 0x2C, 0x58, 0xB0, 0x03, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0xE3, 0xA5, 0x29, 0x52,
    0xA4, 0x2B, 0x56, 0xAC, 0x3B, 0x76, 0xEC, 0xBB, 0x15, 0x2A, 0x54, 0xA8, 0x33, 0x66, 0xCC, 0xFB,
    0x95, 0x49, 0x92, 0x47, 0x8E, 0x7F, 0xFE, 0x9F, 0x5D, 0xBA, 0x17, 0x2E, 0x5C, 0xB8, 0x13, 0x26,
    0x4C, 0x98, 0x53, 0xA6, 0x2F, 0x5E, 0xBC, 0x1B, 0x36, 0x6C, 0xD8, 0xD3, 0xC5, 0xE9, 0xB1,
};

/* vim: set ts=4 sw=4 et : */
#endif /* Z_REEDSOL_LOGS_H */
