/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { describe, it, expect, beforeEach, afterEach, vi } from 'vitest';
import { GrepTool } from './grep.js';
import path from 'path';
import fs from 'fs/promises';
import os from 'os';
import { createMockWorkspaceContext } from '../test-utils/mockWorkspaceContext.js';
// Mock the child_process module to control grep/git grep behavior
vi.mock('child_process', () => ({
    spawn: vi.fn(() => ({
        on: (event, cb) => {
            if (event === 'error' || event === 'close') {
                // Simulate command not found or error for git grep and system grep
                // to force it to fall back to JS implementation.
                setTimeout(() => cb(1), 0); // cb(1) for error/close
            }
        },
        stdout: { on: vi.fn() },
        stderr: { on: vi.fn() },
    })),
}));
describe('GrepTool', () => {
    let tempRootDir;
    let grepTool;
    const abortSignal = new AbortController().signal;
    const mockConfig = {
        getTargetDir: () => tempRootDir,
        getWorkspaceContext: () => createMockWorkspaceContext(tempRootDir),
    };
    beforeEach(async () => {
        tempRootDir = await fs.mkdtemp(path.join(os.tmpdir(), 'grep-tool-root-'));
        grepTool = new GrepTool(mockConfig);
        // Create some test files and directories
        await fs.writeFile(path.join(tempRootDir, 'fileA.txt'), 'hello world\nsecond line with world');
        await fs.writeFile(path.join(tempRootDir, 'fileB.js'), 'const foo = "bar";\nfunction baz() { return "hello"; }');
        await fs.mkdir(path.join(tempRootDir, 'sub'));
        await fs.writeFile(path.join(tempRootDir, 'sub', 'fileC.txt'), 'another world in sub dir');
        await fs.writeFile(path.join(tempRootDir, 'sub', 'fileD.md'), '# Markdown file\nThis is a test.');
    });
    afterEach(async () => {
        await fs.rm(tempRootDir, { recursive: true, force: true });
    });
    describe('validateToolParams', () => {
        it('should return null for valid params (pattern only)', () => {
            const params = { pattern: 'hello' };
            expect(grepTool.validateToolParams(params)).toBeNull();
        });
        it('should return null for valid params (pattern and path)', () => {
            const params = { pattern: 'hello', path: '.' };
            expect(grepTool.validateToolParams(params)).toBeNull();
        });
        it('should return null for valid params (pattern, path, and include)', () => {
            const params = {
                pattern: 'hello',
                path: '.',
                include: '*.txt',
            };
            expect(grepTool.validateToolParams(params)).toBeNull();
        });
        it('should return error if pattern is missing', () => {
            const params = { path: '.' };
            expect(grepTool.validateToolParams(params)).toBe(`params must have required property 'pattern'`);
        });
        it('should return error for invalid regex pattern', () => {
            const params = { pattern: '[[' };
            expect(grepTool.validateToolParams(params)).toContain('Invalid regular expression pattern');
        });
        it('should return error if path does not exist', () => {
            const params = { pattern: 'hello', path: 'nonexistent' };
            // Check for the core error message, as the full path might vary
            expect(grepTool.validateToolParams(params)).toContain('Failed to access path stats for');
            expect(grepTool.validateToolParams(params)).toContain('nonexistent');
        });
        it('should return error if path is a file, not a directory', async () => {
            const filePath = path.join(tempRootDir, 'fileA.txt');
            const params = { pattern: 'hello', path: filePath };
            expect(grepTool.validateToolParams(params)).toContain(`Path is not a directory: ${filePath}`);
        });
    });
    describe('execute', () => {
        it('should find matches for a simple pattern in all files', async () => {
            const params = { pattern: 'world' };
            const invocation = grepTool.build(params);
            const result = await invocation.execute(abortSignal);
            expect(result.llmContent).toContain('Found 3 matches for pattern "world" in the workspace directory');
            expect(result.llmContent).toContain('File: fileA.txt');
            expect(result.llmContent).toContain('L1: hello world');
            expect(result.llmContent).toContain('L2: second line with world');
            expect(result.llmContent).toContain(`File: ${path.join('sub', 'fileC.txt')}`);
            expect(result.llmContent).toContain('L1: another world in sub dir');
            expect(result.returnDisplay).toBe('Found 3 matches');
        });
        it('should find matches in a specific path', async () => {
            const params = { pattern: 'world', path: 'sub' };
            const invocation = grepTool.build(params);
            const result = await invocation.execute(abortSignal);
            expect(result.llmContent).toContain('Found 1 match for pattern "world" in path "sub"');
            expect(result.llmContent).toContain('File: fileC.txt'); // Path relative to 'sub'
            expect(result.llmContent).toContain('L1: another world in sub dir');
            expect(result.returnDisplay).toBe('Found 1 match');
        });
        it('should find matches with an include glob', async () => {
            const params = { pattern: 'hello', include: '*.js' };
            const invocation = grepTool.build(params);
            const result = await invocation.execute(abortSignal);
            expect(result.llmContent).toContain('Found 1 match for pattern "hello" in the workspace directory (filter: "*.js"):');
            expect(result.llmContent).toContain('File: fileB.js');
            expect(result.llmContent).toContain('L2: function baz() { return "hello"; }');
            expect(result.returnDisplay).toBe('Found 1 match');
        });
        it('should find matches with an include glob and path', async () => {
            await fs.writeFile(path.join(tempRootDir, 'sub', 'another.js'), 'const greeting = "hello";');
            const params = {
                pattern: 'hello',
                path: 'sub',
                include: '*.js',
            };
            const invocation = grepTool.build(params);
            const result = await invocation.execute(abortSignal);
            expect(result.llmContent).toContain('Found 1 match for pattern "hello" in path "sub" (filter: "*.js")');
            expect(result.llmContent).toContain('File: another.js');
            expect(result.llmContent).toContain('L1: const greeting = "hello";');
            expect(result.returnDisplay).toBe('Found 1 match');
        });
        it('should return "No matches found" when pattern does not exist', async () => {
            const params = { pattern: 'nonexistentpattern' };
            const invocation = grepTool.build(params);
            const result = await invocation.execute(abortSignal);
            expect(result.llmContent).toContain('No matches found for pattern "nonexistentpattern" in the workspace directory.');
            expect(result.returnDisplay).toBe('No matches found');
        });
        it('should handle regex special characters correctly', async () => {
            const params = { pattern: 'foo.*bar' }; // Matches 'const foo = "bar";'
            const invocation = grepTool.build(params);
            const result = await invocation.execute(abortSignal);
            expect(result.llmContent).toContain('Found 1 match for pattern "foo.*bar" in the workspace directory:');
            expect(result.llmContent).toContain('File: fileB.js');
            expect(result.llmContent).toContain('L1: const foo = "bar";');
        });
        it('should be case-insensitive by default (JS fallback)', async () => {
            const params = { pattern: 'HELLO' };
            const invocation = grepTool.build(params);
            const result = await invocation.execute(abortSignal);
            expect(result.llmContent).toContain('Found 2 matches for pattern "HELLO" in the workspace directory:');
            expect(result.llmContent).toContain('File: fileA.txt');
            expect(result.llmContent).toContain('L1: hello world');
            expect(result.llmContent).toContain('File: fileB.js');
            expect(result.llmContent).toContain('L2: function baz() { return "hello"; }');
        });
        it('should throw an error if params are invalid', async () => {
            const params = { path: '.' }; // Invalid: pattern missing
            expect(() => grepTool.build(params)).toThrow(/params must have required property 'pattern'/);
        });
    });
    describe('multi-directory workspace', () => {
        it('should search across all workspace directories when no path is specified', async () => {
            // Create additional directory with test files
            const secondDir = await fs.mkdtemp(path.join(os.tmpdir(), 'grep-tool-second-'));
            await fs.writeFile(path.join(secondDir, 'other.txt'), 'hello from second directory\nworld in second');
            await fs.writeFile(path.join(secondDir, 'another.js'), 'function world() { return "test"; }');
            // Create a mock config with multiple directories
            const multiDirConfig = {
                getTargetDir: () => tempRootDir,
                getWorkspaceContext: () => createMockWorkspaceContext(tempRootDir, [secondDir]),
            };
            const multiDirGrepTool = new GrepTool(multiDirConfig);
            const params = { pattern: 'world' };
            const invocation = multiDirGrepTool.build(params);
            const result = await invocation.execute(abortSignal);
            // Should find matches in both directories
            expect(result.llmContent).toContain('Found 5 matches for pattern "world"');
            // Matches from first directory
            expect(result.llmContent).toContain('fileA.txt');
            expect(result.llmContent).toContain('L1: hello world');
            expect(result.llmContent).toContain('L2: second line with world');
            expect(result.llmContent).toContain('fileC.txt');
            expect(result.llmContent).toContain('L1: another world in sub dir');
            // Matches from second directory (with directory name prefix)
            const secondDirName = path.basename(secondDir);
            expect(result.llmContent).toContain(`File: ${path.join(secondDirName, 'other.txt')}`);
            expect(result.llmContent).toContain('L2: world in second');
            expect(result.llmContent).toContain(`File: ${path.join(secondDirName, 'another.js')}`);
            expect(result.llmContent).toContain('L1: function world()');
            // Clean up
            await fs.rm(secondDir, { recursive: true, force: true });
        });
        it('should search only specified path within workspace directories', async () => {
            // Create additional directory
            const secondDir = await fs.mkdtemp(path.join(os.tmpdir(), 'grep-tool-second-'));
            await fs.mkdir(path.join(secondDir, 'sub'));
            await fs.writeFile(path.join(secondDir, 'sub', 'test.txt'), 'hello from second sub directory');
            // Create a mock config with multiple directories
            const multiDirConfig = {
                getTargetDir: () => tempRootDir,
                getWorkspaceContext: () => createMockWorkspaceContext(tempRootDir, [secondDir]),
            };
            const multiDirGrepTool = new GrepTool(multiDirConfig);
            // Search only in the 'sub' directory of the first workspace
            const params = { pattern: 'world', path: 'sub' };
            const invocation = multiDirGrepTool.build(params);
            const result = await invocation.execute(abortSignal);
            // Should only find matches in the specified sub directory
            expect(result.llmContent).toContain('Found 1 match for pattern "world" in path "sub"');
            expect(result.llmContent).toContain('File: fileC.txt');
            expect(result.llmContent).toContain('L1: another world in sub dir');
            // Should not contain matches from second directory
            expect(result.llmContent).not.toContain('test.txt');
            // Clean up
            await fs.rm(secondDir, { recursive: true, force: true });
        });
    });
    describe('getDescription', () => {
        it('should generate correct description with pattern only', () => {
            const params = { pattern: 'testPattern' };
            const invocation = grepTool.build(params);
            expect(invocation.getDescription()).toBe("'testPattern'");
        });
        it('should generate correct description with pattern and include', () => {
            const params = {
                pattern: 'testPattern',
                include: '*.ts',
            };
            const invocation = grepTool.build(params);
            expect(invocation.getDescription()).toBe("'testPattern' in *.ts");
        });
        it('should generate correct description with pattern and path', async () => {
            const dirPath = path.join(tempRootDir, 'src', 'app');
            await fs.mkdir(dirPath, { recursive: true });
            const params = {
                pattern: 'testPattern',
                path: path.join('src', 'app'),
            };
            const invocation = grepTool.build(params);
            // The path will be relative to the tempRootDir, so we check for containment.
            expect(invocation.getDescription()).toContain("'testPattern' within");
            expect(invocation.getDescription()).toContain(path.join('src', 'app'));
        });
        it('should indicate searching across all workspace directories when no path specified', () => {
            // Create a mock config with multiple directories
            const multiDirConfig = {
                getTargetDir: () => tempRootDir,
                getWorkspaceContext: () => createMockWorkspaceContext(tempRootDir, ['/another/dir']),
            };
            const multiDirGrepTool = new GrepTool(multiDirConfig);
            const params = { pattern: 'testPattern' };
            const invocation = multiDirGrepTool.build(params);
            expect(invocation.getDescription()).toBe("'testPattern' across all workspace directories");
        });
        it('should generate correct description with pattern, include, and path', async () => {
            const dirPath = path.join(tempRootDir, 'src', 'app');
            await fs.mkdir(dirPath, { recursive: true });
            const params = {
                pattern: 'testPattern',
                include: '*.ts',
                path: path.join('src', 'app'),
            };
            const invocation = grepTool.build(params);
            expect(invocation.getDescription()).toContain("'testPattern' in *.ts within");
            expect(invocation.getDescription()).toContain(path.join('src', 'app'));
        });
        it('should use ./ for root path in description', () => {
            const params = { pattern: 'testPattern', path: '.' };
            const invocation = grepTool.build(params);
            expect(invocation.getDescription()).toBe("'testPattern' within ./");
        });
    });
});
//# sourceMappingURL=grep.test.js.map