#-------------------------------------------------------------------------------
# SuiteSparse/GraphBLAS/cmake_modules/GraphBLASConfig.cmake
#-------------------------------------------------------------------------------

# The following copyright and license applies to just this file only, not to
# the library itself:
# GraphBLASConfig.cmake, Copyright (c) 2017-2025, Timothy A. Davis.  All Rights
# Reserved.
# SPDX-License-Identifier: BSD-3-clause

#-------------------------------------------------------------------------------

# Finds the GraphBLAS include file and compiled library.
# The following targets are defined:
#   SuiteSparse::GraphBLAS           - for the shared library (if available)
#   SuiteSparse::GraphBLAS_static    - for the static library (if available)

# For backward compatibility the following variables are set:

# GRAPHBLAS_INCLUDE_DIR - where to find GraphBLAS.h, etc.
# GRAPHBLAS_LIBRARY     - dynamic GraphBLAS library
# GRAPHBLAS_STATIC      - static GraphBLAS library
# GRAPHBLAS_LIBRARIES   - libraries when using GraphBLAS
# GRAPHBLAS_FOUND       - true if GraphBLAS found

# Set ``CMAKE_MODULE_PATH`` to the parent folder where this module file is
# installed.

#-------------------------------------------------------------------------------


####### Expanded from @PACKAGE_INIT@ by configure_package_config_file() #######
####### Any changes to this file will be overwritten by the next CMake run ####
####### The input file was GraphBLASConfig.cmake.in                            ########

get_filename_component(PACKAGE_PREFIX_DIR "${CMAKE_CURRENT_LIST_DIR}/../../../" ABSOLUTE)

macro(set_and_check _var _file)
  set(${_var} "${_file}")
  if(NOT EXISTS "${_file}")
    message(FATAL_ERROR "File or directory ${_file} referenced by variable ${_var} does not exist !")
  endif()
endmacro()

macro(check_required_components _NAME)
  foreach(comp ${${_NAME}_FIND_COMPONENTS})
    if(NOT ${_NAME}_${comp}_FOUND)
      if(${_NAME}_FIND_REQUIRED_${comp})
        set(${_NAME}_FOUND FALSE)
      endif()
    endif()
  endforeach()
endmacro()

####################################################################################

set ( GRAPHBLAS_DATE "Mar 3, 2025" )
set ( GRAPHBLAS_VERSION_MAJOR 10 )
set ( GRAPHBLAS_VERSION_MINOR 0 )
set ( GRAPHBLAS_VERSION_PATCH 0 )
set ( GRAPHBLAS_VERSION "10.0.0" )

# Check for dependent targets
include ( CMakeFindDependencyMacro )
set ( _dependencies_found ON )

if ( OFF )
    # Look for imported targets of additional dependency if GraphBLAS was built with CUDA

    if ( NOT GraphBLAS_CUDA_FOUND )
        if ( OFF )
            # First check in a common build tree
            find_dependency ( GraphBLAS_CUDA 10.0.0
                PATHS ${CMAKE_SOURCE_DIR}/../GraphBLAS/build/CUDA NO_DEFAULT_PATH )
            # Then, check in the currently active CMAKE_MODULE_PATH
            if ( NOT GraphBLAS_CUDA_FOUND )
                find_dependency ( GraphBLAS_CUDA 10.0.0 )
            endif ( )
        else ( )
            find_dependency ( GraphBLAS_CUDA 10.0.0 )
        endif ( )
    endif ( )
    if ( NOT GraphBLAS_CUDA_FOUND )
        set ( _dependencies_found OFF )
    endif ( )
endif ( )

# Look for OpenMP
if ( ON AND NOT OpenMP_C_FOUND )
    find_dependency ( OpenMP COMPONENTS C )
    if ( NOT OpenMP_C_FOUND )
        set ( _dependencies_found OFF )
    endif ( )
endif ( )

if ( NOT _dependencies_found )
    set ( GraphBLAS_FOUND OFF )
    return ( )
endif ( )

# Import target
include ( ${CMAKE_CURRENT_LIST_DIR}/GraphBLASTargets.cmake )

if ( ON )
    if ( TARGET SuiteSparse::GraphBLAS )
        get_property ( _graphblas_aliased TARGET SuiteSparse::GraphBLAS
            PROPERTY ALIASED_TARGET )
        if ( "${_graphblas_aliased}" STREQUAL "" )
            target_include_directories ( SuiteSparse::GraphBLAS SYSTEM AFTER INTERFACE
                "$<TARGET_PROPERTY:OpenMP::OpenMP_C,INTERFACE_INCLUDE_DIRECTORIES>" )
        else ( )
            target_include_directories ( ${_graphblas_aliased} SYSTEM AFTER INTERFACE
                "$<TARGET_PROPERTY:OpenMP::OpenMP_C,INTERFACE_INCLUDE_DIRECTORIES>" )
        endif ( )
    endif ( )
    if ( TARGET SuiteSparse::GraphBLAS_static )
        get_property ( _graphblas_aliased TARGET SuiteSparse::GraphBLAS_static
            PROPERTY ALIASED_TARGET )
        if ( "${_graphblas_aliased}" STREQUAL "" )
            target_include_directories ( SuiteSparse::GraphBLAS_static SYSTEM AFTER INTERFACE
                "$<TARGET_PROPERTY:OpenMP::OpenMP_C,INTERFACE_INCLUDE_DIRECTORIES>" )
        else ( )
            target_include_directories ( ${_graphblas_aliased} SYSTEM AFTER INTERFACE
                "$<TARGET_PROPERTY:OpenMP::OpenMP_C,INTERFACE_INCLUDE_DIRECTORIES>" )
        endif ( )
    endif ( )
endif ( )

# The following is only for backward compatibility with FindGraphBLAS.

set ( _target_shared SuiteSparse::GraphBLAS )
set ( _target_static SuiteSparse::GraphBLAS_static )
set ( _var_prefix "GRAPHBLAS" )

if ( NOT ON AND NOT TARGET ${_target_shared} )
    # make sure there is always an import target without suffix )
    add_library ( ${_target_shared} ALIAS ${_target_static} )
    set ( _shared_is_aliased ON )
endif ( )

get_target_property ( ${_var_prefix}_INCLUDE_DIR ${_target_shared} INTERFACE_INCLUDE_DIRECTORIES )
if ( ${_var_prefix}_INCLUDE_DIR )
    # First item in SuiteSparse targets contains the "main" header directory.
    list ( GET ${_var_prefix}_INCLUDE_DIR 0 ${_var_prefix}_INCLUDE_DIR )
endif ( )
get_target_property ( ${_var_prefix}_LIBRARY ${_target_shared} IMPORTED_IMPLIB )
if ( NOT ${_var_prefix}_LIBRARY )
    get_target_property ( _library_chk ${_target_shared} IMPORTED_LOCATION )
    if ( EXISTS ${_library_chk} )
        set ( ${_var_prefix}_LIBRARY ${_library_chk} )
    endif ( )
endif ( )
if ( TARGET ${_target_static} )
    get_target_property ( ${_var_prefix}_STATIC ${_target_static} IMPORTED_LOCATION )
endif ( )

# Check for most common build types
set ( _config_types "Debug" "Release" "RelWithDebInfo" "MinSizeRel" "None" )

get_property ( _isMultiConfig GLOBAL PROPERTY GENERATOR_IS_MULTI_CONFIG )
if ( _isMultiConfig )
    # For multi-configuration generators (e.g., Visual Studio), prefer those
    # configurations.
    list ( PREPEND _config_types ${CMAKE_CONFIGURATION_TYPES} )
else ( )
    # For single-configuration generators, prefer the current configuration.
    list ( PREPEND _config_types ${CMAKE_BUILD_TYPE} )
endif ( )

list ( REMOVE_DUPLICATES _config_types )

foreach ( _config ${_config_types} )
    string ( TOUPPER ${_config} _uc_config )
    if ( NOT ${_var_prefix}_LIBRARY )
        get_target_property ( _library_chk ${_target_shared}
            IMPORTED_IMPLIB_${_uc_config} )
        if ( EXISTS ${_library_chk} )
            set ( ${_var_prefix}_LIBRARY ${_library_chk} )
        endif ( )
    endif ( )
    if ( NOT ${_var_prefix}_LIBRARY )
        get_target_property ( _library_chk ${_target_shared}
            IMPORTED_LOCATION_${_uc_config} )
        if ( EXISTS ${_library_chk} )
            set ( ${_var_prefix}_LIBRARY ${_library_chk} )
        endif ( )
    endif ( )
    if ( TARGET ${_target_static} AND NOT ${_var_prefix}_STATIC )
        get_target_property ( _library_chk ${_target_static}
            IMPORTED_LOCATION_${_uc_config} )
        if ( EXISTS ${_library_chk} )
            set ( ${_var_prefix}_STATIC ${_library_chk} )
        endif ( )
    endif ( )
endforeach ( )

set ( GRAPHBLAS_LIBRARIES ${GRAPHBLAS_LIBRARY} )

macro ( suitesparse_check_exist _var _files )
  # ignore generator expressions
  string ( GENEX_STRIP "${_files}" _files2 )

  foreach ( _file ${_files2} )
    if ( NOT EXISTS "${_file}" )
      message ( FATAL_ERROR "File or directory ${_file} referenced by variable ${_var} does not exist!" )
    endif ( )
  endforeach ()
endmacro ( )

suitesparse_check_exist ( GRAPHBLAS_INCLUDE_DIR ${GRAPHBLAS_INCLUDE_DIR} )
suitesparse_check_exist ( GRAPHBLAS_LIBRARY ${GRAPHBLAS_LIBRARY} )

message ( STATUS "GraphBLAS version: ${GRAPHBLAS_VERSION}" )
message ( STATUS "GraphBLAS include: ${GRAPHBLAS_INCLUDE_DIR}" )
message ( STATUS "GraphBLAS library: ${GRAPHBLAS_LIBRARY}" )
message ( STATUS "GraphBLAS static:  ${GRAPHBLAS_STATIC}" )

if ( NOT _shared_is_aliased AND GRAPHBLAS_LIBRARY )
    # Get library name from filename of library
    # This might be something like:
    #   /usr/lib/libgraphblas.so or /usr/lib/libgraphblas.a or graphblas64
    # convert to library name that can be used with -l flags for pkg-config
    set ( GRAPHBLAS_LIBRARY_TMP ${GRAPHBLAS_LIBRARY} )
    string ( FIND ${GRAPHBLAS_LIBRARY} "." _pos REVERSE )
    if ( ${_pos} EQUAL "-1" )
        set ( _graphblas_library_name ${GRAPHBLAS_LIBRARY} )
    else ( )
      set ( _kinds "SHARED" "STATIC" )
      if ( WIN32 )
          list ( PREPEND _kinds "IMPORT" )
      endif ( )
      foreach ( _kind IN LISTS _kinds )
          set ( _regex ".*\\/(lib)?([^\\.]*)(${CMAKE_${_kind}_LIBRARY_SUFFIX})" )
          if ( ${GRAPHBLAS_LIBRARY} MATCHES ${_regex} )
              string ( REGEX REPLACE ${_regex} "\\2" _libname ${GRAPHBLAS_LIBRARY} )
              if ( NOT "${_libname}" STREQUAL "" )
                  set ( _graphblas_library_name ${_libname} )
                  break ()
              endif ( )
          endif ( )
      endforeach ( )
    endif ( )

    set_target_properties ( SuiteSparse::GraphBLAS PROPERTIES
        OUTPUT_NAME ${_graphblas_library_name} )
endif ( )

if ( GRAPHBLAS_STATIC )
    # Get library name from filename of library
    # This might be something like:
    #   /usr/lib/libgraphblas.so or /usr/lib/libgraphblas.a or graphblas64
    # convert to library name that can be used with -l flags for pkg-config
    set ( GRAPHBLAS_LIBRARY_TMP ${GRAPHBLAS_STATIC} )
    string ( FIND ${GRAPHBLAS_STATIC} "." _pos REVERSE )
    if ( ${_pos} EQUAL "-1" )
        set ( _graphblas_library_name ${GRAPHBLAS_STATIC} )
    else ( )
      set ( _kinds "SHARED" "STATIC" )
      if ( WIN32 )
          list ( PREPEND _kinds "IMPORT" )
      endif ( )
      foreach ( _kind IN LISTS _kinds )
          set ( _regex ".*\\/(lib)?([^\\.]*)(${CMAKE_${_kind}_LIBRARY_SUFFIX})" )
          if ( ${GRAPHBLAS_STATIC} MATCHES ${_regex} )
              string ( REGEX REPLACE ${_regex} "\\2" _libname ${GRAPHBLAS_STATIC} )
              if ( NOT "${_libname}" STREQUAL "" )
                  set ( _graphblas_library_name ${_libname} )
                  break ()
              endif ( )
          endif ( )
      endforeach ( )
    endif ( )

    set_target_properties ( SuiteSparse::GraphBLAS_static PROPERTIES
        OUTPUT_NAME ${_graphblas_library_name} )
endif ( )
